package Net::Cisco::ACS::Host;
use strict;
use Moose;
use Data::Dumper;

BEGIN {
    use Exporter ();
    use vars qw($VERSION @ISA @EXPORT @EXPORT_OK %EXPORT_TAGS %actions);
    $VERSION     = '0.03';
    @ISA         = qw(Exporter);
    @EXPORT      = qw();
    @EXPORT_OK   = qw();
    %EXPORT_TAGS = ();
};

    %actions = (	"query" => "/Rest/Identity/Host",
					"create" => "/Rest/Identity/Host",
               		"update" => "/Rest/Identity/Host",
                	"getByName" => "/Rest/Identity/Host/name/",
                	"getById" => "/Rest/Identity/Host/id/",
           ); 

# MOOSE!		   
	   
has 'description' => (
      is  => 'rw',
      isa => 'Any',
  );

has 'id' => (
      is  => 'rw',
      isa => 'Str',
  );

has 'identityGroupName' => (
      is  => 'rw',
      isa => 'Str',
  );
  
has 'created' => (
      is  => 'rw',
      isa => 'Str',
  );

has 'enabled' => (
      is  => 'rw',
      isa => 'Str',
  );
  
has 'lastEnabled' => (
      is  => 'rw',
      isa => 'Str',
  );

has 'lastModified' => (
      is  => 'rw',
      isa => 'Str',
  );

has 'macAddress' => ( # Not sure if query for Host can be done on MAC Address or on 'name' getByName vs getByMacAddress??!?
      is  => 'rw',
      isa => 'Str',
  );
  
# No Moose	

sub toXML
{ my $self = shift;
  my $result = "";
  my $id = $self->id;
  my $description = $self->description || "";
  my $identitygroupname = $self->identityGroupName || "All Groups";
  my $enabled = $self->enabled || "true";
  my $lastenabled = $self->lastEnabled || "";
  my $lastmodified = $self->lastModified || "";  
  my $macaddress = $self->macAddress || "";    
  if ($id) { $result = "   <id>$id</id>\n"; }
  $result = <<XML;
	<description>$description</description>
	<identityGroupName>$identitygroupname</identityGroupName>
	<enabled>$enabled</enabled>
	<macAddress>$macaddress</macAddress>
XML

  return $result;
}

sub header
{ my $self = shift;
  my $hosts = shift;
  return qq(<?xml version="1.0" encoding="UTF-8" standalone="yes"?><ns2:hosts xmlns:ns2="identity.rest.mgmt.acs.nm.cisco.com">$hosts</ns1:hosts>);
}
	
=head1 NAME

Net::Cisco::ACS::Host - Access Cisco ACS functionality through REST API - Host fields

=head1 SYNOPSIS

	use Net::Cisco::ACS;
	use Net::Cisco::ACS::Host;

	my $acs = Net::Cisco::ACS->new(hostname => '10.0.0.1', username => 'acsadmin', password => 'testPassword');
	my $host = Net::Cisco::ACS::Host->new(macAddress=>"00-00-00-00-00-00", "description"=>"R2D2","identityGroupName"=>"All Groups:Droids");
	
	my %hosts = $acs->hosts;
	# Retrieve all hosts from ACS
	# Returns hash with host macAddress / Net::Cisco::ACS::Host pairs

	print $acs->hosts->{"00-00-00-00-00-00"}->toXML;
	# Dump in XML format (used by ACS for API calls)
	
	my $host = $acs->hosts("macAddress","00-00-00-00-00-00");
	# Faster call to request specific host information by MAC Address. No name field exists.

	my $host = $acs->hosts("id","250");
	# Faster call to request specific hosts information by ID (assigned by ACS, present in Net::Cisco::ACS::Host)

	$host->id(0); # Required for new host!
	my $id = $acs->create($host);
	# Create new host based on Net::Cisco::ACS::Host instance
	# Return value is ID generated by ACS
	print "Record ID is $id" if $id;
	print $Net::Cisco::ACS::ERROR unless $id;
	# $Net::Cisco::ACS::ERROR contains details about failure

	my $id = $acs->update($host);
	# Update existing device based on Net::Cisco::ACS::Host instance
	# Return value is ID generated by ACS
	print "Record ID is $id" if $id;
	print $Net::Cisco::ACS::ERROR unless $id;
	# $Net::Cisco::ACS::ERROR contains details about failure

	$acs->delete($host);
	# Delete existing host based on Net::Cisco::ACS::Host instance
  
=head1 DESCRIPTION

The Net::Cisco::ACS::Host class holds all the Internal Host relevant information from Cisco ACS 5.x. Hosts (in this case) are not to be confused with L<Net::Cisco::ACS::Device>. Hosts are used as an identity item, in case of e.g dot1x configurations.

=head1 USAGE

The Net::Cisco::ACS::Host class holds all the user relevant information from Cisco ACS 5.x

=head1 USAGE

All calls are typically handled through an instance of the L<Net::Cisco::ACS> class. L<Net::Cisco::ACS::Host> acts as a container for host identity related information.

=over 3

=item new

Class constructor. Returns object of L<Net::Cisco::ACS::Host> on succes. The following fields can be set / retrieved:

=over 5

=item description

=item enabled

=item identityGroupName

=item macAddress

=item id

Formatting rules may be in place & enforced by Cisco ACS.

=back

Read-only values:

=over 5

=item created

=item lastModified

=item lastEnabled

=back

=over 3

=item description 

The host description, typically used for host name. This is the only text-field for this record type.

=item enabled

Boolean flag to indicate record status.

=item macAddress

MAC Address for host. Used for actual identification.

=item id

Cisco ACS generates a unique ID for each Host record. This field cannot be updated within ACS but is used for reference. Set to 0 when creating a new record or when duplicating an existing host.

=item toXML

Dump the record in ACS accept XML formatting (without header).

=item header

Generate the correct XML header. Takes output of C<toXML> as argument.

=back

=back

=head1 BUGS

None yet

=head1 SUPPORT

None yet :)

=head1 AUTHOR

    Hendrik Van Belleghem
    CPAN ID: BEATNIK
    hendrik.vanbelleghem@gmail.com

=head1 COPYRIGHT

This program is free software licensed under the...

	The General Public License (GPL)
	Version 2, June 1991

The full text of the license can be found in the
LICENSE file included with this module.


=head1 SEE ALSO

perl(1).

=cut

#################### main pod documentation end ###################
__PACKAGE__->meta->make_immutable();

1;
# The preceding line will help the module return a true value

