#!/usr/bin/env python3

import importlib.util
import os

from fenrirscreenreader.core.i18n import _

# Load base configuration class
_base_path = os.path.join(os.path.dirname(__file__), "..", "config_base.py")
_spec = importlib.util.spec_from_file_location("config_base", _base_path)
_module = importlib.util.module_from_spec(_spec)
_spec.loader.exec_module(_module)
config_command = _module.config_command


class command(config_command):
    def __init__(self):
        super().__init__()

    def get_description(self):
        return "Set punctuation verbosity level"

    def run(self):
        current_level = self.get_setting("general", "punctuationLevel", "some")

        # Present current level
        level_descriptions = {
            "none": "None - no punctuation spoken",
            "some": "Some - basic punctuation only",
            "most": "Most - detailed punctuation",
            "all": "All - every punctuation mark",
        }

        current_description = level_descriptions.get(current_level, "Unknown")
        self.present_text(f"Current punctuation level: {current_description}")

        # Cycle through the four levels
        levels = ["none", "some", "most", "all"]
        try:
            current_index = levels.index(current_level)
            next_index = (current_index + 1) % len(levels)
            new_level = levels[next_index]
        except ValueError:
            new_level = "some"  # Default to some

        success = self.set_setting("general", "punctuationLevel", new_level)

        if success:
            new_description = level_descriptions[new_level]
            self.present_text(f"Punctuation level set to: {new_description}")
            self.play_sound("Accept")
        else:
            self.present_text("Failed to change punctuation level")
            self.play_sound("Error")
