mod completion_edit;
mod completion_hint;
mod completion_kind;

use std::{ops::Deref, path::Path};

pub use completion_edit::{CompletionEdit, CompletionTextEdit, InsertReplaceEdit};
pub use completion_hint::{AddLeadingComma, AddTrailingComma, CommaHint, CompletionHint};
pub use completion_kind::CompletionKind;
use tombi_document_tree::dig_accessors;
use tombi_schema_store::{Accessor, SchemaUri, get_schema_name};
use tombi_text::{FromLsp, IntoLsp};

use crate::get_tombi_github_uri;

#[derive(Debug, Clone, PartialEq, Eq, PartialOrd, Ord)]
#[repr(u8)]
pub enum CompletionContentPriority {
    Custom(String),
    Default,
    Const,
    Enum,
    Example,
    Key,
    OptionalKey,
    AdditionalKey,
    TypeHint,
    TypeHintKey,
    TypeHintTrue,
    TypeHintFalse,
}

impl CompletionContentPriority {
    pub fn as_prefix(&self) -> String {
        match self {
            CompletionContentPriority::Custom(value) => value.to_string(),
            // NOTE: Completion candidates provided by extensions are assigned priority numbers starting from 50,
            //       allowing them to be placed above basic features.
            CompletionContentPriority::Default => "50".to_string(),
            CompletionContentPriority::Const => "51".to_string(),
            CompletionContentPriority::Enum => "52".to_string(),
            CompletionContentPriority::Example => "53".to_string(),
            CompletionContentPriority::Key => "54".to_string(),
            CompletionContentPriority::OptionalKey => "55".to_string(),
            CompletionContentPriority::AdditionalKey => "56".to_string(),
            CompletionContentPriority::TypeHint => "57".to_string(),
            CompletionContentPriority::TypeHintKey => "58".to_string(),
            CompletionContentPriority::TypeHintTrue => "59".to_string(),
            CompletionContentPriority::TypeHintFalse => "60".to_string(),
        }
    }
}

#[derive(Debug, Clone, PartialEq, Eq)]
pub enum CommentContext {
    DocumentDirective(tombi_ast::Comment),
    ValueDirective(tombi_ast::Comment),
    Normal(tombi_ast::Comment),
}

#[derive(Debug, Clone, PartialEq, Eq)]
pub struct CompletionContent {
    pub label: String,
    pub kind: CompletionKind,
    pub emoji_icon: Option<char>,
    pub priority: CompletionContentPriority,
    pub detail: Option<String>,
    pub documentation: Option<String>,
    pub filter_text: Option<String>,
    pub schema_uri: Option<SchemaUri>,
    pub deprecated: Option<bool>,
    pub edit: Option<CompletionEdit>,
    pub preselect: Option<bool>,
    pub in_comment: bool,
}

impl CompletionContent {
    pub fn new_const_value(
        label: String,
        detail: Option<String>,
        documentation: Option<String>,
        edit: Option<CompletionEdit>,
        schema_uri: Option<&SchemaUri>,
        deprecated: Option<bool>,
    ) -> Self {
        Self {
            label: label.clone(),
            kind: CompletionKind::Enum,
            emoji_icon: None,
            priority: CompletionContentPriority::Const,
            detail,
            documentation,
            filter_text: None,
            schema_uri: schema_uri.cloned(),
            edit,
            deprecated,
            preselect: None,
            in_comment: false,
        }
    }

    pub fn new_enum_value(
        label: String,
        detail: Option<String>,
        documentation: Option<String>,
        edit: Option<CompletionEdit>,
        schema_uri: Option<&SchemaUri>,
        deprecated: Option<bool>,
    ) -> Self {
        Self {
            label: label.clone(),
            kind: CompletionKind::Enum,
            emoji_icon: None,
            priority: CompletionContentPriority::Enum,
            detail,
            documentation,
            filter_text: None,
            schema_uri: schema_uri.cloned(),
            edit,
            deprecated,
            preselect: None,
            in_comment: false,
        }
    }

    pub fn new_default_value(
        label: String,
        detail: Option<String>,
        documentation: Option<String>,
        edit: Option<CompletionEdit>,
        schema_uri: Option<&SchemaUri>,
        deprecated: Option<bool>,
    ) -> Self {
        Self {
            label,
            kind: CompletionKind::Enum,
            emoji_icon: None,
            priority: CompletionContentPriority::Default,
            detail,
            documentation,
            filter_text: None,
            schema_uri: schema_uri.cloned(),
            edit,
            deprecated,
            preselect: Some(true),
            in_comment: false,
        }
    }

    pub fn new_example_value(
        label: String,
        detail: Option<String>,
        documentation: Option<String>,
        edit: Option<CompletionEdit>,
        schema_uri: Option<&SchemaUri>,
        deprecated: Option<bool>,
    ) -> Self {
        Self {
            label: label.clone(),
            kind: CompletionKind::Enum,
            emoji_icon: None,
            priority: CompletionContentPriority::Example,
            detail,
            documentation,
            filter_text: None,
            schema_uri: schema_uri.cloned(),
            edit,
            deprecated,
            preselect: None,
            in_comment: false,
        }
    }

    pub fn new_type_hint_value(
        kind: CompletionKind,
        label: impl Into<String>,
        detail: impl Into<String>,
        edit: Option<CompletionEdit>,
        schema_uri: Option<&SchemaUri>,
    ) -> Self {
        Self {
            label: label.into(),
            kind,
            emoji_icon: None,
            priority: CompletionContentPriority::TypeHint,
            detail: Some(detail.into()),
            documentation: None,
            filter_text: None,
            schema_uri: schema_uri.cloned(),
            edit,
            deprecated: None,
            preselect: None,
            in_comment: false,
        }
    }

    pub fn new_type_hint_boolean(
        value: bool,
        edit: Option<CompletionEdit>,
        schema_uri: Option<&SchemaUri>,
    ) -> Self {
        Self {
            label: value.to_string(),
            kind: CompletionKind::Boolean,
            emoji_icon: None,
            priority: if value {
                CompletionContentPriority::TypeHintTrue
            } else {
                CompletionContentPriority::TypeHintFalse
            },
            detail: Some("Boolean".to_string()),
            documentation: None,
            filter_text: None,
            schema_uri: schema_uri.cloned(),
            edit,
            deprecated: None,
            preselect: None,
            in_comment: false,
        }
    }

    pub fn new_type_hint_string(
        kind: CompletionKind,
        quote: char,
        detail: impl Into<String>,
        edit: Option<CompletionEdit>,
        schema_uri: Option<&SchemaUri>,
    ) -> Self {
        Self {
            label: format!("{quote}{quote}"),
            kind,
            emoji_icon: None,
            priority: CompletionContentPriority::TypeHint,
            detail: Some(detail.into()),
            documentation: None,
            filter_text: None,
            schema_uri: schema_uri.cloned(),
            edit,
            deprecated: None,
            preselect: None,
            in_comment: false,
        }
    }

    pub fn new_type_hint_inline_table(
        position: tombi_text::Position,
        schema_uri: Option<&SchemaUri>,
        completion_hint: Option<CompletionHint>,
    ) -> Self {
        Self {
            label: "{}".to_string(),
            kind: CompletionKind::Table,
            emoji_icon: None,
            priority: CompletionContentPriority::TypeHint,
            detail: Some("InlineTable".to_string()),
            documentation: None,
            filter_text: None,
            schema_uri: schema_uri.cloned(),
            edit: CompletionEdit::new_inline_table(position, completion_hint),
            deprecated: None,
            preselect: None,
            in_comment: false,
        }
    }

    pub fn new_type_hint_key(
        key_name: &str,
        key_range: tombi_text::Range,
        schema_uri: Option<&SchemaUri>,
        completion_hint: Option<CompletionHint>,
    ) -> Self {
        let edit = CompletionEdit::new_key(key_name, key_range, completion_hint);

        Self {
            label: "$key".to_string(),
            kind: CompletionKind::Table,
            emoji_icon: None,
            priority: CompletionContentPriority::TypeHintKey,
            detail: Some("Key".to_string()),
            documentation: None,
            filter_text: Some(key_name.to_string()),
            schema_uri: schema_uri.cloned(),
            edit,
            deprecated: None,
            preselect: None,
            in_comment: false,
        }
    }

    pub fn new_type_hint_empty_key(
        position: tombi_text::Position,
        schema_uri: Option<&SchemaUri>,
        completion_hint: Option<CompletionHint>,
    ) -> Self {
        Self {
            label: "$key".to_string(),
            kind: CompletionKind::Key,
            emoji_icon: None,
            priority: CompletionContentPriority::TypeHintKey,
            detail: Some("Key".to_string()),
            documentation: None,
            filter_text: None,
            edit: CompletionEdit::new_additional_key(
                "key",
                tombi_text::Range::at(position),
                completion_hint,
            ),
            schema_uri: schema_uri.cloned(),
            deprecated: None,
            preselect: None,
            in_comment: false,
        }
    }

    pub fn new_key(
        key_name: &str,
        position: tombi_text::Position,
        detail: Option<String>,
        documentation: Option<String>,
        required_keys: Option<&Vec<String>>,
        schema_uri: Option<&SchemaUri>,
        deprecated: Option<bool>,
        completion_hint: Option<CompletionHint>,
    ) -> Self {
        let label = key_name.to_string();
        let required = required_keys
            .map(|required_keys| required_keys.contains(&label))
            .unwrap_or_default();

        let key_range = match completion_hint {
            Some(
                CompletionHint::DotTrigger { range } | CompletionHint::EqualTrigger { range, .. },
            ) => tombi_text::Range::new(range.end, position),
            _ => tombi_text::Range::at(position),
        };

        Self {
            label,
            kind: CompletionKind::Key,
            emoji_icon: None,
            priority: if required {
                CompletionContentPriority::Key
            } else {
                CompletionContentPriority::OptionalKey
            },
            detail,
            documentation,
            filter_text: None,
            edit: CompletionEdit::new_key(key_name, key_range, completion_hint),
            schema_uri: schema_uri.cloned(),
            deprecated,
            preselect: None,
            in_comment: false,
        }
    }

    pub fn new_pattern_key(
        key_label: Option<&str>,
        patterns: &[String],
        position: tombi_text::Position,
        schema_uri: Option<&SchemaUri>,
        completion_hint: Option<CompletionHint>,
    ) -> Self {
        let key_label = key_label.unwrap_or("key");
        Self {
            label: format!("${key_label}"),
            kind: CompletionKind::Key,
            emoji_icon: None,
            priority: CompletionContentPriority::AdditionalKey,
            detail: Some("Pattern Key".to_string()),
            documentation: if !patterns.is_empty() {
                let mut documentation = "Allowed Patterns:\n\n".to_string();
                for pattern in patterns {
                    documentation.push_str(&format!("- `{pattern}`\n"));
                }
                Some(documentation)
            } else {
                None
            },
            filter_text: None,
            edit: CompletionEdit::new_additional_key(
                key_label,
                tombi_text::Range::at(position),
                completion_hint,
            ),
            schema_uri: schema_uri.cloned(),
            deprecated: None,
            preselect: None,
            in_comment: false,
        }
    }

    pub fn new_additional_key(
        key_label: Option<&str>,
        position: tombi_text::Position,
        schema_uri: Option<&SchemaUri>,
        deprecated: Option<bool>,
        completion_hint: Option<CompletionHint>,
    ) -> Self {
        let key_label = key_label.unwrap_or("key");
        Self {
            label: format!("${key_label}"),
            kind: CompletionKind::Key,
            emoji_icon: None,
            priority: CompletionContentPriority::AdditionalKey,
            detail: Some("Additional Key".to_string()),
            documentation: None,
            filter_text: None,
            edit: CompletionEdit::new_additional_key(
                key_label,
                tombi_text::Range::at(position),
                completion_hint,
            ),
            schema_uri: schema_uri.cloned(),
            deprecated,
            preselect: None,
            in_comment: false,
        }
    }

    pub fn new_magic_triggers(
        key: &str,
        position: tombi_text::Position,
        schema_uri: Option<&SchemaUri>,
    ) -> Vec<Self> {
        [(".", "Dot Trigger"), ("=", "Equal Trigger")]
            .into_iter()
            .map(|(trigger, detail)| Self {
                label: trigger.to_string(),
                kind: CompletionKind::MagicTrigger,
                emoji_icon: None,
                priority: CompletionContentPriority::TypeHint,
                detail: Some(detail.to_string()),
                documentation: None,
                filter_text: Some(format!("{key}{trigger}")),
                edit: CompletionEdit::new_magic_trigger(trigger, position),
                schema_uri: schema_uri.cloned(),
                deprecated: None,
                preselect: None,
                in_comment: false,
            })
            .collect()
    }

    /// Creates a new schema comment directive completion content.
    ///
    /// NOTE: schema directive is formatted to follow Taplo's format.
    ///       If Taplo didn't exist, it would be formatted as `# schema: ${1:url}`.
    ///
    ///       See: https://taplo.tamasfe.dev/configuration/directives.html#the-schema-directive
    ///
    /// ```toml
    /// #:schema https://...
    /// ```
    pub fn new_comment_directive(
        directive_name: &str,
        detail: impl Into<String>,
        documentation: impl Into<String>,
        edit: Option<CompletionEdit>,
    ) -> Self {
        Self {
            label: directive_name.to_string(),
            kind: CompletionKind::CommentDirective,
            emoji_icon: None,
            priority: CompletionContentPriority::Key,
            detail: Some(detail.into()),
            documentation: Some(documentation.into()),
            filter_text: None,
            edit,
            schema_uri: None,
            deprecated: None,
            preselect: None,
            in_comment: false,
        }
    }

    pub fn with_position(mut self, position: tombi_text::Position) -> Self {
        self.edit = self.edit.map(|edit| edit.with_position(position));
        self
    }
}

impl FromLsp<CompletionContent> for tower_lsp::lsp_types::CompletionItem {
    fn from_lsp(
        source: CompletionContent,
        line_index: &tombi_text::LineIndex,
    ) -> tower_lsp::lsp_types::CompletionItem {
        const SECTION_SEPARATOR: &str = "-----";

        let sorted_text = format!("{}_{}", source.priority.as_prefix(), &source.label);

        let mut schema_text = None;
        if let Some(schema_uri) = &source.schema_uri {
            let schema_uri = match get_tombi_github_uri(schema_uri) {
                Some(schema_uri) => schema_uri,
                None => schema_uri.deref().clone(),
            };
            if let Some(schema_filename) = get_schema_name(&schema_uri) {
                schema_text = Some(format!("Schema: [{schema_filename}]({schema_uri})\n"));
            }
        }
        let documentation = match source.documentation {
            Some(documentation) => {
                let mut documentation = documentation;
                if let Some(schema_text) = schema_text {
                    documentation.push_str(&format!("\n\n{SECTION_SEPARATOR}\n\n"));
                    documentation.push_str(&schema_text);
                }
                Some(documentation)
            }
            None => schema_text,
        };

        let (insert_text_format, text_edit, additional_text_edits) = match source.edit {
            Some(edit) => (
                edit.insert_text_format,
                Some(edit.text_edit.into_lsp(line_index)),
                edit.additional_text_edits.map(|edits| {
                    edits
                        .into_iter()
                        .map(|edit| edit.into_lsp(line_index))
                        .collect()
                }),
            ),
            None => (None, None, None),
        };

        let label_details = match source.priority {
            CompletionContentPriority::Custom(_) => {
                Some(tower_lsp::lsp_types::CompletionItemLabelDetails {
                    detail: None,
                    description: source.detail.clone(),
                })
            }
            CompletionContentPriority::Default => {
                Some(tower_lsp::lsp_types::CompletionItemLabelDetails {
                    detail: None,
                    description: Some(match &source.detail {
                        Some(detail) => format!("[Default] {detail}"),
                        None => "Default".to_string(),
                    }),
                })
            }
            CompletionContentPriority::Const => {
                Some(tower_lsp::lsp_types::CompletionItemLabelDetails {
                    detail: None,
                    description: Some(match &source.detail {
                        Some(detail) => detail.to_string(),
                        None => "Const".to_string(),
                    }),
                })
            }
            CompletionContentPriority::Enum => {
                Some(tower_lsp::lsp_types::CompletionItemLabelDetails {
                    detail: None,
                    description: Some(match &source.detail {
                        Some(detail) => detail.to_string(),
                        None => "Enum".to_string(),
                    }),
                })
            }
            CompletionContentPriority::Example => {
                Some(tower_lsp::lsp_types::CompletionItemLabelDetails {
                    detail: None,
                    description: Some(match &source.detail {
                        Some(detail) => detail.to_string(),
                        None => "Example".to_string(),
                    }),
                })
            }
            CompletionContentPriority::Key => {
                Some(tower_lsp::lsp_types::CompletionItemLabelDetails {
                    detail: None,
                    description: source.detail.clone(),
                })
            }
            CompletionContentPriority::OptionalKey | CompletionContentPriority::AdditionalKey => {
                Some(tower_lsp::lsp_types::CompletionItemLabelDetails {
                    detail: Some("?".to_string()),
                    description: source.detail.clone(),
                })
            }
            CompletionContentPriority::TypeHint
            | CompletionContentPriority::TypeHintKey
            | CompletionContentPriority::TypeHintTrue
            | CompletionContentPriority::TypeHintFalse => {
                Some(tower_lsp::lsp_types::CompletionItemLabelDetails {
                    detail: None,
                    description: Some(match &source.detail {
                        Some(detail) if !detail.trim().is_empty() => detail.clone(),
                        _ => "Type Hint".to_string(),
                    }),
                })
            }
        }
        .map(|mut details| {
            if let Some(emoji_icon) = source.emoji_icon {
                details.description = Some(format!(
                    "{} {}",
                    emoji_icon,
                    details.description.unwrap_or_default()
                ));
            }
            details
        });

        tower_lsp::lsp_types::CompletionItem {
            label: source.label,
            label_details,
            kind: Some(source.kind.into()),
            detail: source.detail.map(|detail| {
                if let Some(emoji_icon) = source.emoji_icon {
                    format!("{emoji_icon} {detail}")
                } else {
                    detail
                }
            }),
            documentation: documentation.map(|documentation| {
                tower_lsp::lsp_types::Documentation::MarkupContent(
                    tower_lsp::lsp_types::MarkupContent {
                        kind: tower_lsp::lsp_types::MarkupKind::Markdown,
                        value: documentation,
                    },
                )
            }),
            sort_text: Some(sorted_text),
            filter_text: source.filter_text,
            insert_text_format,
            text_edit,
            insert_text_mode: Some(tower_lsp::lsp_types::InsertTextMode::ADJUST_INDENTATION),
            additional_text_edits,
            preselect: source.preselect,
            deprecated: source.deprecated,
            ..Default::default()
        }
    }
}

/// Creates a new file path completion content.
///
/// If `allowed_extensions` is `None`, all file extensions are allowed.
/// If `allowed_extensions` is `Some`, only the specified file extensions are allowed.
///
/// # Examples
///
/// ```toml
/// [package.build]
/// path = "src/█"
/// ```
pub fn completion_file_path(
    text_document_uri: &tombi_uri::Uri,
    document_tree: &tombi_document_tree::DocumentTree,
    position: tombi_text::Position,
    accessors: &[Accessor],
    allowed_extensions: Option<&[&str]>,
) -> Option<Vec<CompletionContent>> {
    let Ok(source_path) = text_document_uri.to_file_path() else {
        return None;
    };
    let Some(base_dir) = source_path.parent() else {
        return None;
    };

    let Some((_, tombi_document_tree::Value::String(string))) =
        dig_accessors(document_tree, accessors)
    else {
        return None;
    };

    if !string.range().contains(position) {
        return None;
    }

    let completions = get_file_path_completions(
        base_dir,
        string.value(),
        string.unquoted_range(),
        allowed_extensions,
    );

    if completions.is_empty() {
        None
    } else {
        Some(completions)
    }
}

/// Creates completion content for directory paths only.
///
/// Use this when the schema expects a directory path (e.g. workspace members).
///
/// # Examples
///
/// ```toml
/// [workspace]
/// members = ["crates/█"]
/// ```
pub fn completion_directory_path(
    text_document_uri: &tombi_uri::Uri,
    document_tree: &tombi_document_tree::DocumentTree,
    position: tombi_text::Position,
    accessors: &[Accessor],
) -> Option<Vec<CompletionContent>> {
    completion_file_path(
        text_document_uri,
        document_tree,
        position,
        accessors,
        Some(&[]),
    )
}

pub fn get_file_path_completions(
    base_dir: &Path,
    path_text: &str,
    path_range: tombi_text::Range,
    allowed_extensions: Option<&[&str]>,
) -> Vec<CompletionContent> {
    let mut completions = Vec::new();

    let (search_dir, prefix, path_prefix) = if path_text.is_empty() {
        (base_dir.to_path_buf(), String::new(), String::new())
    } else {
        let path = Path::new(path_text);
        if path_text.ends_with('/') {
            (base_dir.join(path), String::new(), path_text.to_string())
        } else {
            let parent = path.parent().unwrap_or(Path::new(""));
            let file_name = path.file_name().and_then(|s| s.to_str()).unwrap_or("");
            let prefix_str = if parent == Path::new("") {
                String::new()
            } else {
                format!("{}/", parent.display())
            };
            (base_dir.join(parent), file_name.to_string(), prefix_str)
        }
    };

    let Ok(entries) = std::fs::read_dir(&search_dir) else {
        return completions;
    };

    for entry in entries.flatten() {
        let Ok(file_name) = entry.file_name().into_string() else {
            continue;
        };

        if file_name.starts_with('.') {
            continue;
        }

        if !prefix.is_empty() && !file_name.starts_with(&prefix) {
            continue;
        }

        let Ok(metadata) = entry.metadata() else {
            continue;
        };

        let is_dir = metadata.is_dir();
        let is_allowed_file = is_allowed_extension(&file_name, allowed_extensions);

        if !is_dir && !is_allowed_file {
            continue;
        }

        let relative_path = match (path_prefix.is_empty(), is_dir) {
            (true, true) => format!("{}/", file_name),
            (true, false) => file_name.clone(),
            (false, true) => format!("{}{}/", path_prefix, file_name),
            (false, false) => format!("{}{}", path_prefix, file_name),
        };

        let detail = if is_dir {
            "Directory".to_string()
        } else if is_json_only_extensions(allowed_extensions) {
            "JSON file".to_string()
        } else {
            "File".to_string()
        };

        let completion_edit =
            CompletionEdit::new_string_literal_while_editing(&relative_path, path_range);

        if let Some(edit) = completion_edit {
            completions.push(CompletionContent {
                label: relative_path,
                kind: CompletionKind::File,
                emoji_icon: if is_dir { Some('📁') } else { Some('📄') },
                priority: CompletionContentPriority::Custom("50".to_string()),
                detail: Some(detail),
                documentation: None,
                filter_text: None,
                schema_uri: None,
                deprecated: None,
                edit: Some(edit),
                preselect: None,
                in_comment: false,
            });
        }
    }

    completions
}

fn is_allowed_extension(file_name: &str, allowed_extensions: Option<&[&str]>) -> bool {
    let Some(allowed_extensions) = allowed_extensions else {
        return false;
    };
    if allowed_extensions.is_empty() {
        return true;
    }

    let extension = Path::new(file_name)
        .extension()
        .and_then(|ext| ext.to_str());
    extension.is_some_and(|ext| {
        allowed_extensions
            .iter()
            .any(|allowed| ext.eq_ignore_ascii_case(allowed))
    })
}

fn is_json_only_extensions(allowed_extensions: Option<&[&str]>) -> bool {
    let Some(allowed_extensions) = allowed_extensions else {
        return false;
    };
    allowed_extensions.len() == 1 && allowed_extensions[0].eq_ignore_ascii_case("json")
}
