//
// ********************************************************************
// * License and Disclaimer                                           *
// *                                                                  *
// * The  Geant4 software  is  copyright of the Copyright Holders  of *
// * the Geant4 Collaboration.  It is provided  under  the terms  and *
// * conditions of the Geant4 Software License,  included in the file *
// * LICENSE and available at  http://cern.ch/geant4/license .  These *
// * include a list of copyright holders.                             *
// *                                                                  *
// * Neither the authors of this software system, nor their employing *
// * institutes,nor the agencies providing financial support for this *
// * work  make  any representation or  warranty, express or implied, *
// * regarding  this  software system or assume any liability for its *
// * use.  Please see the license in the file  LICENSE  and URL above *
// * for the full disclaimer and the limitation of liability.         *
// *                                                                  *
// * This  code  implementation is the result of  the  scientific and *
// * technical work of the GEANT4 collaboration.                      *
// * By using,  copying,  modifying or  distributing the software (or *
// * any work based  on the software)  you  agree  to acknowledge its *
// * use  in  resulting  scientific  publications,  and indicate your *
// * acceptance of all terms of the Geant4 Software license.          *
// ********************************************************************
//
/// \file DetectorMessenger.cc
/// \brief Implementation of the DetectorMessenger class

#include "DetectorMessenger.hh"

#include "DetectorConstruction.hh"

#include "G4UIcmdWithADoubleAndUnit.hh"
#include "G4UIcmdWithAnInteger.hh"
#include "G4UIcmdWithoutParameter.hh"
#include "G4UIcommand.hh"
#include "G4UIdirectory.hh"
#include "G4UIparameter.hh"

#include <sstream>

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

DetectorMessenger::DetectorMessenger(DetectorConstruction* Det) : fDetector(Det)
{
  fTestemDir = new G4UIdirectory("/testem/");
  fTestemDir->SetGuidance(" detector control.");

  fDetDir = new G4UIdirectory("/testem/det/");
  fDetDir->SetGuidance("detector construction commands");

  fNbAbsorCmd = new G4UIcmdWithAnInteger("/testem/det/setNbOfAbsor", this);
  fNbAbsorCmd->SetGuidance("Set number of Absorbers.");
  fNbAbsorCmd->SetParameterName("NbAbsor", false);
  fNbAbsorCmd->SetRange("NbAbsor>0");
  fNbAbsorCmd->AvailableForStates(G4State_PreInit, G4State_Idle);
  fNbAbsorCmd->SetToBeBroadcasted(false);

  fAbsorCmd = new G4UIcommand("/testem/det/setAbsor", this);
  fAbsorCmd->SetGuidance("Set the absor nb, the material, the thickness.");
  fAbsorCmd->SetGuidance("  absor number : from 1 to NbOfAbsor");
  fAbsorCmd->SetGuidance("  material name");
  fAbsorCmd->SetGuidance("  thickness (with unit) : t>0.");
  //
  G4UIparameter* AbsNbPrm = new G4UIparameter("AbsorNb", 'i', false);
  AbsNbPrm->SetGuidance("absor number : from 1 to NbOfAbsor");
  AbsNbPrm->SetParameterRange("AbsorNb>0");
  fAbsorCmd->SetParameter(AbsNbPrm);
  //
  G4UIparameter* MatPrm = new G4UIparameter("material", 's', false);
  MatPrm->SetGuidance("material name");
  fAbsorCmd->SetParameter(MatPrm);
  //
  G4UIparameter* ThickPrm = new G4UIparameter("thickness", 'd', false);
  ThickPrm->SetGuidance("thickness of absorber");
  ThickPrm->SetParameterRange("thickness>0.");
  fAbsorCmd->SetParameter(ThickPrm);
  //
  G4UIparameter* unitPrm = new G4UIparameter("unit", 's', false);
  unitPrm->SetGuidance("unit of thickness");
  G4String unitList = G4UIcommand::UnitsList(G4UIcommand::CategoryOf("mm"));
  unitPrm->SetParameterCandidates(unitList);
  fAbsorCmd->SetParameter(unitPrm);
  //
  fAbsorCmd->AvailableForStates(G4State_PreInit, G4State_Idle);
  fAbsorCmd->SetToBeBroadcasted(false);

  fNdivCmd = new G4UIcommand("/testem/det/nDivAbsor", this);
  fNdivCmd->SetGuidance("Divide the absor nb : number of divisions");
  fNdivCmd->SetGuidance("  absor number : from 1 to NbOfAbsor");
  fNdivCmd->SetGuidance("  number of divisions >= 0");
  //
  G4UIparameter* AbsNbPar = new G4UIparameter("AbsorNb", 'i', false);
  AbsNbPar->SetGuidance("absor number : from 1 to NbOfAbsor");
  AbsNbPar->SetParameterRange("AbsorNb>0");
  fNdivCmd->SetParameter(AbsNbPar);
  //
  G4UIparameter* NdivPrm = new G4UIparameter("NdivNb", 'i', false);
  NdivPrm->SetGuidance("nb of divisions > 0");
  NdivPrm->SetParameterRange("NdivNb>0");
  fNdivCmd->SetParameter(NdivPrm);
  //
  fNdivCmd->AvailableForStates(G4State_PreInit, G4State_Idle);
  fNdivCmd->SetToBeBroadcasted(false);

  fSizeYZCmd = new G4UIcmdWithADoubleAndUnit("/testem/det/setSizeYZ", this);
  fSizeYZCmd->SetGuidance("Set sizeYZ of the absorber");
  fSizeYZCmd->SetParameterName("SizeYZ", false);
  fSizeYZCmd->SetRange("SizeYZ>0.");
  fSizeYZCmd->SetUnitCategory("Length");
  fSizeYZCmd->AvailableForStates(G4State_PreInit, G4State_Idle);
  fSizeYZCmd->SetToBeBroadcasted(false);
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

DetectorMessenger::~DetectorMessenger()
{
  delete fNbAbsorCmd;
  delete fAbsorCmd;
  delete fNdivCmd;
  delete fSizeYZCmd;
  delete fDetDir;
  delete fTestemDir;
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......

void DetectorMessenger::SetNewValue(G4UIcommand* command, G4String newValue)
{
  if (command == fNbAbsorCmd) {
    fDetector->SetNbOfAbsor(fNbAbsorCmd->GetNewIntValue(newValue));
  }

  if (command == fAbsorCmd) {
    G4int num;
    G4double tick;
    G4String unt, mat;
    std::istringstream is(newValue);
    is >> num >> mat >> tick >> unt;
    G4String material = mat;
    tick *= G4UIcommand::ValueOf(unt);
    fDetector->SetAbsorMaterial(num, material);
    fDetector->SetAbsorThickness(num, tick);
  }

  if (command == fNdivCmd) {
    G4int num, ndiv;
    std::istringstream is(newValue);
    is >> num >> ndiv;
    fDetector->SetNbOfDivisions(num, ndiv);
  }

  if (command == fSizeYZCmd) {
    fDetector->SetAbsorSizeYZ(fSizeYZCmd->GetNewDoubleValue(newValue));
  }
}

//....oooOO0OOooo........oooOO0OOooo........oooOO0OOooo........oooOO0OOooo......
