/*************************************************************************
 * Copyright (C) 1995-2025, Rene Brun and Fons Rademakers.               *
 * All rights reserved.                                                  *
 *                                                                       *
 * For the licensing terms see $ROOTSYS/LICENSE.                         *
 * For the list of contributors see $ROOTSYS/README/CREDITS.             *
 *************************************************************************/

#include <ROOT/Browsable/RElement.hxx>
#include <ROOT/Browsable/RProvider.hxx>
#include <ROOT/Browsable/RLevelIter.hxx>
#include <ROOT/Browsable/RItem.hxx>
#include <ROOT/Browsable/RNTupleItem.hxx>

#include <ROOT/RNTupleReader.hxx>
#include <ROOT/RNTupleBrowseUtils.hxx>
#include <ROOT/RNTupleDescriptor.hxx>

#include "TClass.h"
#include "RFieldHolder.hxx"

#include "RVisualizationHolder.hxx"

namespace ROOT::Experimental {
class TObjectDrawable;
}
using namespace std::string_literals;
using namespace ROOT::Browsable;

// ==============================================================================================
/** \class RTreeMapElement
\ingroup rbrowser
\brief Browsing element representing TreeMap visualization for RNTuple
\author Patryk Pilichowski
\date 2025
\warning This is part of the ROOT 7 prototype! It will change without notice. It might trigger earthquakes. Feedback is
welcome!
*/

class RTreeMapElement : public RElement {
protected:
   std::shared_ptr<ROOT::RNTupleReader> fNtplReader;
   std::string fFileName;

public:
   RTreeMapElement(std::shared_ptr<ROOT::RNTupleReader> ntplReader, const std::string &fileName = "")
      : RElement(), fNtplReader(ntplReader), fFileName(fileName)
   {
   }

   ~RTreeMapElement() override = default;

   /** Name of TreeMap visualization */
   std::string GetName() const override { return "TreeMap"; }

   /** Title of TreeMap visualization */
   std::string GetTitle() const override { return "TreeMap visualization of RNTuple structure and disk usage"; }

   /** No children for TreeMap visualization */
   std::unique_ptr<RLevelIter> GetChildsIter() override { return nullptr; }

   /** Return class of RNTuple for consistency */
   const TClass *GetClass() const { return TClass::GetClass<ROOT::RNTuple>(); }

   /** Return holder with visualization data */
   std::unique_ptr<RHolder> GetObject() override
   {
      return std::make_unique<RVisualizationHolder>(fNtplReader, fFileName, fNtplReader->GetDescriptor().GetName());
   }

   /** Default action is to draw the treemap */
   EActionKind GetDefaultAction() const override { return kActDraw7; }

   /** Check if visualization is capable of being drawn */
   bool IsCapable(EActionKind kind) const override { return kind == kActDraw6 || kind == kActDraw7; }

   /** Create item with TreeMap icon */
   std::unique_ptr<RItem> CreateItem() const override
   {
      auto item = std::make_unique<RNTupleItem>(GetName(), 0, "sap-icon://Chart-Tree-Map");
      item->SetTitle(GetTitle());
      return item;
   }

   /** Not expandable by default */
   bool IsFolder() const override { return false; }

   /** Set filename for treemap creation */
   void SetFileName(const std::string &fileName) { fFileName = fileName; }
};

// ==============================================================================================
/** \class RVisualizationElement
\ingroup rbrowser
\brief Browsing element representing visualization folder for RNTuple
\author Patryk Pilichowski
\date 2025
\warning This is part of the ROOT 7 prototype! It will change without notice. It might trigger earthquakes. Feedback is
welcome!
*/

class RVisualizationElement : public RElement {
protected:
   std::shared_ptr<ROOT::RNTupleReader> fNtplReader;
   std::string fFileName;

public:
   RVisualizationElement(std::shared_ptr<ROOT::RNTupleReader> ntplReader, const std::string &fileName = "")
      : RElement(), fNtplReader(ntplReader), fFileName(fileName)
   {
   }

   ~RVisualizationElement() override = default;

   /** Name of visualization folder */
   std::string GetName() const override { return "Visualization"; }

   /** Title of visualization folder */
   std::string GetTitle() const override { return "Visualization tools and options for RNTuple data"; }

   /** Create iterator for visualization children */
   std::unique_ptr<RLevelIter> GetChildsIter() override;

   /** Return class of RNTuple for consistency */
   const TClass *GetClass() const { return TClass::GetClass<ROOT::RNTuple>(); }

   /** No direct object for folder */
   std::unique_ptr<RHolder> GetObject() override { return nullptr; }

   /** Default action is none for folder */
   EActionKind GetDefaultAction() const override { return kActNone; }

   /** Create item with visualization folder icon */
   std::unique_ptr<RItem> CreateItem() const override
   {
      auto item =
         std::make_unique<RNTupleItem>(GetName(), 1, "sap-icon://show", RNTupleItem::ECategory::kVisualization);
      item->SetTitle(GetTitle());
      return item;
   }

   /** This is a folder */
   bool IsFolder() const override { return true; }

   /** Set filename for child elements */
   void SetFileName(const std::string &fileName) { fFileName = fileName; }
};

// ==============================================================================================
/** \class RVisualizationIterator
\ingroup rbrowser
\brief Iterator over visualization options
\author Patryk Pilichowski
\date 2025
\warning This is part of the ROOT 7 prototype! It will change without notice. It might trigger earthquakes. Feedback is
welcome!
*/

class RVisualizationIterator : public RLevelIter {
   std::shared_ptr<ROOT::RNTupleReader> fNtplReader;
   std::string fFileName;
   int fCounter{-1};
   int fTotalItems{1};

public:
   RVisualizationIterator(std::shared_ptr<ROOT::RNTupleReader> ntplReader, const std::string &fileName = "")
      : fNtplReader(ntplReader), fFileName(fileName)
   {
   }

   ~RVisualizationIterator() override = default;

   bool Next() override { return ++fCounter < fTotalItems; }

   std::string GetItemName() const override
   {
      if (fCounter == 0) {
         return "TreeMap";
      }
      return "";
   }

   bool CanItemHaveChilds() const override { return false; }

   /** Create item for the browser */
   std::unique_ptr<RItem> CreateItem() override
   {
      if (fCounter == 0) {
         auto item = std::make_unique<RNTupleItem>("TreeMap", 0, "sap-icon://Chart-Tree-Map");
         item->SetTitle("TreeMap visualization of RNTuple structure and disk usage");
         return item;
      }
      return nullptr;
   }

   std::shared_ptr<RElement> GetElement() override
   {
      if (fCounter == 0) {
         return std::make_shared<RTreeMapElement>(fNtplReader, fFileName);
      }
      return nullptr;
   }
};

std::unique_ptr<RLevelIter> RVisualizationElement::GetChildsIter()
{
   return std::make_unique<RVisualizationIterator>(fNtplReader, fFileName);
}

// ==============================================================================================

class RFieldElement : public RElement {
protected:
   std::shared_ptr<ROOT::RNTupleReader> fNtplReader;
   std::string fParentName;
   ROOT::DescriptorId_t fFieldId;
   std::string fDisplayName;

public:
   RFieldElement(std::shared_ptr<ROOT::RNTupleReader> ntplReader, const std::string &parent_name,
                 const ROOT::DescriptorId_t id, const std::string &displayName)
      : RElement(), fNtplReader(ntplReader), fParentName(parent_name), fFieldId(id), fDisplayName(displayName)
   {
   }

   ~RFieldElement() override = default;

   /** Name of RField */
   std::string GetName() const override
   {
      return fNtplReader->GetDescriptor().GetFieldDescriptor(fFieldId).GetFieldName();
   }

   /** Title of RField */
   std::string GetTitle() const override
   {
      auto &fld = fNtplReader->GetDescriptor().GetFieldDescriptor(fFieldId);
      return "RField name "s + fld.GetFieldName() + " type "s + fld.GetTypeName();
   }

   std::unique_ptr<RLevelIter> GetChildsIter() override;

   /** Return class of field  - for a moment using RNTuple class as dummy */
   const TClass *GetClass() const { return TClass::GetClass<ROOT::RNTuple>(); }

   std::unique_ptr<RHolder> GetObject() override
   {
      return std::make_unique<RFieldHolder>(fNtplReader, fParentName, fFieldId, fDisplayName);
   }

   EActionKind GetDefaultAction() const override
   {
      auto range = fNtplReader->GetDescriptor().GetFieldIterable(fFieldId);
      if (range.begin() != range.end())
         return kActNone;
      return kActDraw7;
   }

   bool IsCapable(EActionKind kind) const override
   {
      if ((kind == kActDraw6) || (kind == kActDraw7))
         return GetDefaultAction() == kActDraw7;

      return false;
   }
};

// ==============================================================================================

/** \class RNTupleElement
\ingroup rbrowser
\brief Browsing element representing of RNTuple
\author Sergey Linev <S.Linev@gsi.de>
\date 2021-03-08
\warning This is part of the ROOT 7 prototype! It will change without notice. It might trigger earthquakes. Feedback is
welcome!
*/

class RNTupleElement : public RElement {
protected:
   std::shared_ptr<ROOT::RNTupleReader> fNtplReader;
   std::string fFileName;

public:
   RNTupleElement(const std::string &ntplName, const std::string &filename) : fFileName(filename)
   {
      fNtplReader = ROOT::RNTupleReader::Open(ntplName, filename);
   }

   ~RNTupleElement() override = default;

   /** Returns true if no ntuple found */
   bool IsNull() const { return !fNtplReader; }

   /** Name of NTuple */
   std::string GetName() const override { return fNtplReader->GetDescriptor().GetName(); }

   /** Title of NTuple */
   std::string GetTitle() const override { return "RNTuple title"s; }

   /** Get the filename */
   const std::string &GetFileName() const { return fFileName; }

   /** Create iterator for childs elements if any */
   std::unique_ptr<RLevelIter> GetChildsIter() override;

   const TClass *GetClass() const { return TClass::GetClass<ROOT::RNTuple>(); }

   std::unique_ptr<RItem> CreateItem() const override
   {
      auto item = std::make_unique<RNTupleItem>(GetName(), -1, "sap-icon://table-chart");
      item->SetTitle(GetTitle());
      return item;
   }
};

// ==============================================================================================

/** \class RNTupleIterator
\ingroup rbrowser
\brief Iterator over RNTuple fields & visualization entry
\author Sergey Linev <S.Linev@gsi.de>
\date 2021-03-08
\warning This is part of the ROOT 7 prototype! It will change without notice. It might trigger earthquakes. Feedback is
welcome!
*/

class RNTupleIterator : public RLevelIter {
   std::shared_ptr<ROOT::RNTupleReader> fNtplReader;
   std::vector<ROOT::DescriptorId_t> fProvidedFieldIds;
   std::vector<ROOT::DescriptorId_t> fActualFieldIds;
   std::string fParentName;
   std::string fFileName;
   int fCounter{-1};
   bool fHasVisualization{false};
   int fTotalItems{0};

public:
   RNTupleIterator(std::shared_ptr<ROOT::RNTupleReader> ntplReader, std::vector<ROOT::DescriptorId_t> &&ids,
                   const std::string &parent_name = "", bool includeVisualization = false,
                   const std::string &fileName = "")
      : fNtplReader(ntplReader),
        fProvidedFieldIds(ids),
        fParentName(parent_name),
        fFileName(fileName),
        fHasVisualization(includeVisualization)
   {
      const auto &desc = fNtplReader->GetDescriptor();
      fActualFieldIds.reserve(fProvidedFieldIds.size());
      for (auto fid : fProvidedFieldIds) {
         fActualFieldIds.emplace_back(ROOT::Internal::GetNextBrowsableField(fid, desc));
      }

      fTotalItems = fProvidedFieldIds.size();
      if (fHasVisualization) {
         fTotalItems++;
      }
   }

   ~RNTupleIterator() override = default;

   bool Next() override { return ++fCounter < fTotalItems; }

   std::string GetItemName() const override
   {
      if (fHasVisualization && fCounter == 0) {
         return "Visualization";
      }
      int fieldIndex = fHasVisualization ? fCounter - 1 : fCounter;
      return fNtplReader->GetDescriptor().GetFieldDescriptor(fProvidedFieldIds[fieldIndex]).GetFieldName();
   }

   bool CanItemHaveChilds() const override
   {
      if (fHasVisualization && fCounter == 0) {
         return true;
      }
      int fieldIndex = fHasVisualization ? fCounter - 1 : fCounter;
      auto subrange = fNtplReader->GetDescriptor().GetFieldIterable(fActualFieldIds[fieldIndex]);
      return subrange.begin() != subrange.end();
   }

   /** Create element for the browser */
   std::unique_ptr<RItem> CreateItem() override
   {
      if (fHasVisualization && fCounter == 0) {
         auto item = std::make_unique<RNTupleItem>("Visualization", 1, "sap-icon://show",
                                                   RNTupleItem::ECategory::kVisualization);
         item->SetTitle("Visualization tools and options for RNTuple data");
         return item;
      }

      int fieldIndex = fHasVisualization ? fCounter - 1 : fCounter;
      int nchilds = 0;
      for (auto &sub : fNtplReader->GetDescriptor().GetFieldIterable(fActualFieldIds[fieldIndex])) {
         (void)sub;
         nchilds++;
      }

      const auto &field = fNtplReader->GetDescriptor().GetFieldDescriptor(fProvidedFieldIds[fieldIndex]);

      auto item = std::make_unique<RNTupleItem>(field.GetFieldName(), nchilds,
                                                nchilds > 0 ? "sap-icon://split" : "sap-icon://e-care");

      item->SetTitle("RField name "s + field.GetFieldName() + " type "s + field.GetTypeName());
      return item;
   }

   std::shared_ptr<RElement> GetElement() override
   {
      if (fHasVisualization && fCounter == 0) {
         return std::make_shared<RVisualizationElement>(fNtplReader, fFileName);
      }

      int fieldIndex = fHasVisualization ? fCounter - 1 : fCounter;
      const auto name = fNtplReader->GetDescriptor().GetFieldDescriptor(fProvidedFieldIds[fieldIndex]).GetFieldName();
      return std::make_shared<RFieldElement>(fNtplReader, fParentName, fActualFieldIds[fieldIndex], name);
   }
};

std::unique_ptr<RLevelIter> RFieldElement::GetChildsIter()
{
   std::vector<ROOT::DescriptorId_t> ids;
   std::string prefix;

   const auto &desc = fNtplReader->GetDescriptor();
   for (auto &f : fNtplReader->GetDescriptor().GetFieldIterable(ROOT::Internal::GetNextBrowsableField(fFieldId, desc)))
      ids.emplace_back(f.GetId());

   if (ids.size() == 0)
      return nullptr;

   prefix = fParentName;
   const auto &fld = desc.GetFieldDescriptor(fFieldId);
   prefix.append(fld.GetFieldName());
   prefix.append(".");

   return std::make_unique<RNTupleIterator>(fNtplReader, std::move(ids), prefix);
}

std::unique_ptr<RLevelIter> RNTupleElement::GetChildsIter()
{
   std::vector<ROOT::DescriptorId_t> ids;

   for (auto &f : fNtplReader->GetDescriptor().GetTopLevelFields())
      ids.emplace_back(f.GetId());

   if (ids.size() == 0)
      return nullptr;

   return std::make_unique<RNTupleIterator>(fNtplReader, std::move(ids), "", true, fFileName);
}

// ==============================================================================================

/** \class RNTupleBrowseProvider
\ingroup rbrowser
\brief Provider for browsing RNTuple classes
\author Sergey Linev <S.Linev@gsi.de>
\date 2021-03-08
\warning This is part of the ROOT 7 prototype! It will change without notice. It might trigger earthquakes. Feedback is
welcome!
*/

class RNTupleBrowseProvider : public RProvider {

public:
   RNTupleBrowseProvider()
   {
      RegisterNTupleFunc([](const std::string &tuple_name, const std::string &filename) -> std::shared_ptr<RElement> {
         auto elem = std::make_shared<RNTupleElement>(tuple_name, filename);
         return elem->IsNull() ? nullptr : elem;
      });
   }

   ~RNTupleBrowseProvider() override { RegisterNTupleFunc(nullptr); }

} newRNTupleBrowseProvider;
