// Copyright 2020 The Gitea Authors. All rights reserved.
// SPDX-License-Identifier: MIT

package pulls

import (
	stdctx "context"

	"code.gitea.io/sdk/gitea"
	"code.gitea.io/tea/cmd/flags"
	"code.gitea.io/tea/modules/context"
	"code.gitea.io/tea/modules/interact"
	"code.gitea.io/tea/modules/task"
	"github.com/urfave/cli/v3"
)

// CmdPullsCreate creates a pull request
var CmdPullsCreate = cli.Command{
	Name:        "create",
	Aliases:     []string{"c"},
	Usage:       "Create a pull-request",
	Description: "Create a pull-request in the current repo",
	Action:      runPullsCreate,
	Flags: append([]cli.Flag{
		&cli.StringFlag{
			Name:  "head",
			Usage: "Branch name of the PR source (default is current one). To specify a different head repo, use <user>:<branch>",
		},
		&cli.StringFlag{
			Name:    "base",
			Aliases: []string{"b"},
			Usage:   "Branch name of the PR target (default is repos default branch)",
		},
		&cli.BoolFlag{
			Name:    "allow-maintainer-edits",
			Aliases: []string{"edits"},
			Usage:   "Enable maintainers to push to the base branch of created pull",
			Value:   true,
		},
		&cli.BoolFlag{
			Name:  "agit",
			Usage: "Create an agit flow pull request",
		},
		&cli.StringFlag{
			Name:  "topic",
			Usage: "Topic name for agit flow pull request",
		},
	}, flags.IssuePRCreateFlags...),
}

func runPullsCreate(_ stdctx.Context, cmd *cli.Command) error {
	ctx := context.InitCommand(cmd)
	ctx.Ensure(context.CtxRequirement{
		LocalRepo:  true,
		RemoteRepo: true,
	})

	// no args -> interactive mode
	if ctx.IsInteractiveMode() {
		if err := interact.CreatePull(ctx); err != nil && !interact.IsQuitting(err) {
			return err
		}
		return nil
	}

	// else use args to create PR
	opts, err := flags.GetIssuePRCreateFlags(ctx)
	if err != nil {
		return err
	}

	if ctx.Bool("agit") {
		return task.CreateAgitFlowPull(
			ctx,
			ctx.String("remote"),
			ctx.String("head"),
			ctx.String("base"),
			ctx.String("topic"),
			opts,
			interact.PromptPassword,
		)
	}

	var allowMaintainerEdits *bool
	if ctx.IsSet("allow-maintainer-edits") {
		allowMaintainerEdits = gitea.OptionalBool(ctx.Bool("allow-maintainer-edits"))
	}

	return task.CreatePull(
		ctx,
		ctx.String("base"),
		ctx.String("head"),
		allowMaintainerEdits,
		opts,
	)
}
