/*
 * This file is part of the KubeVirt project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Copyright The KubeVirt Authors.
 */

package apply_test

import (
	. "github.com/onsi/ginkgo/v2"
	. "github.com/onsi/gomega"

	k8sfield "k8s.io/apimachinery/pkg/util/validation/field"

	virtv1 "kubevirt.io/api/core/v1"
	v1beta1 "kubevirt.io/api/instancetype/v1beta1"

	"kubevirt.io/kubevirt/pkg/instancetype/apply"
	"kubevirt.io/kubevirt/pkg/libvmi"
	"kubevirt.io/kubevirt/pkg/pointer"
)

var _ = Describe("Preference.Architecture", func() {
	var (
		vmi              *virtv1.VirtualMachineInstance
		instancetypeSpec *v1beta1.VirtualMachineInstancetypeSpec
		preferenceSpec   *v1beta1.VirtualMachinePreferenceSpec

		field      = k8sfield.NewPath("spec", "template", "spec")
		vmiApplier = apply.NewVMIApplier()
	)

	BeforeEach(func() {
		vmi = libvmi.New()
	})

	It("should apply preferred architecture to VMI when architecture is not set", func() {
		preferenceSpec = &v1beta1.VirtualMachinePreferenceSpec{
			PreferredArchitecture: pointer.P("arm64"),
		}

		Expect(vmiApplier.ApplyToVMI(field, instancetypeSpec, preferenceSpec, &vmi.Spec, &vmi.ObjectMeta)).To(Succeed())
		Expect(vmi.Spec.Architecture).To(Equal("arm64"))
	})

	It("should not override existing architecture in VMI", func() {
		vmi.Spec.Architecture = "amd64"
		preferenceSpec = &v1beta1.VirtualMachinePreferenceSpec{
			PreferredArchitecture: pointer.P("arm64"),
		}

		Expect(vmiApplier.ApplyToVMI(field, instancetypeSpec, preferenceSpec, &vmi.Spec, &vmi.ObjectMeta)).To(Succeed())
		Expect(vmi.Spec.Architecture).To(Equal("amd64"))
	})

	DescribeTable("should not apply when", func(preferenceSpec *v1beta1.VirtualMachinePreferenceSpec) {
		Expect(vmiApplier.ApplyToVMI(field, instancetypeSpec, preferenceSpec, &vmi.Spec, &vmi.ObjectMeta)).To(Succeed())
		Expect(vmi.Spec.Architecture).To(BeEmpty())
	},
		Entry("PreferenceSpec is nil", nil),
		Entry("PreferredArchitecture is nil", &v1beta1.VirtualMachinePreferenceSpec{}),
		Entry("PreferredArchitecture is an empty string", &v1beta1.VirtualMachinePreferenceSpec{
			PreferredArchitecture: pointer.P(""),
		}),
	)
})
