// Copyright 2026 The Fuchsia Authors
//
// Licensed under a BSD-style license <LICENSE-BSD>, Apache License, Version 2.0
// <LICENSE-APACHE or https://www.apache.org/licenses/LICENSE-2.0>, or the MIT
// license <LICENSE-MIT or https://opensource.org/licenses/MIT>, at your option.
// This file may not be copied, modified, or distributed except according to
// those terms.

// These tests cause errors which are generated by a later compilation pass than
// the other errors we generate, and so if they're compiled in the same file,
// the compiler will never get to that pass, and so we won't get the errors.

include!("../include.rs");

use util::AU16;
use zerocopy::{transmute, transmute_mut, try_transmute};

// Although this is not a soundness requirement, we currently require that the
// size of the destination type is not smaller than the size of the source type.
const TRANSMUTE_DECREASE_SIZE: u8 = transmute!(AU16(0));
//~[msrv, stable, nightly]^ ERROR: cannot transmute between types of different sizes, or dependently-sized types
//~[stable, nightly]^^ ERROR: transmuting from 2-byte type to 1-byte type: `AU16` -> `u8`

// `transmute!` does not support transmuting from a smaller type to a larger
// one.
const TRANSMUTE_INCREASE_SIZE: AU16 = transmute!(0u8);
//~[msrv, stable, nightly]^ ERROR: cannot transmute between types of different sizes, or dependently-sized types
//~[stable, nightly]^^ ERROR: transmuting from 1-byte type to 2-byte type: `u8` -> `AU16`

// `transmute!` does not support transmuting from a smaller type to a larger
// one.
const TRANSMUTE_INCREASE_SIZE_ALLOW_SHRINK: AU16 = transmute!(#![allow(shrink)] 0u8);
//~[msrv, stable, nightly]^ ERROR: cannot transmute between types of different sizes, or dependently-sized types
//~[stable, nightly]^^ ERROR: transmuting from 1-byte type to 2-byte type: `u8` -> `Transmute<u8, AU16>`

const ARRAY_OF_U8S: [u8; 2] = [0u8; 2];

// `transmute_mut!` cannot, generally speaking, be used in const contexts.
const TRANSMUTE_MUT_CONST_CONTEXT: &mut [u8; 2] = transmute_mut!(&mut ARRAY_OF_U8S);
//~[msrv]^ ERROR: mutable references are not allowed in constants
//~[msrv]^^ ERROR: calls in constants are limited to constant functions, tuple structs and tuple variants
//~[msrv]^^^ ERROR: calls in constants are limited to constant functions, tuple structs and tuple variants
//~[msrv]^^^^ ERROR: temporary value dropped while borrowed
//~[stable]^^^^^ ERROR: cannot call non-const method `Wrap::<&mut [u8; 2], &mut [u8; 2]>::transmute_mut_inference_helper` in constants
//~[stable]^^^^^^ ERROR: cannot call non-const method `Wrap::<&mut [u8; 2], &mut [u8; 2]>::transmute_mut` in constants
//~[nightly]^^^^^^^ ERROR: cannot call non-const method `zerocopy::util::macro_util::Wrap::<&mut [u8; 2], &mut [u8; 2]>::transmute_mut_inference_helper` in constants
//~[nightly]^^^^^^^^ ERROR: cannot call non-const method `zerocopy::util::macro_util::Wrap::<&mut [u8; 2], &mut [u8; 2]>::transmute_mut` in constants

// Should fail because the file is 4 bytes long, not 8.
const INCLUDE_VALUE_WRONG_SIZE: u64 = zerocopy::include_value!("../../testdata/include_value/data");
//~[msrv, stable, nightly]^ ERROR: cannot transmute between types of different sizes, or dependently-sized types
//~[stable, nightly]^^ ERROR: transmuting from 4-byte type to 8-byte type: `[u8; 4]` -> `u64`

fn main() {
    // Although this is not a soundness requirement, we currently require that
    // the size of the destination type is not smaller than the size of the
    // source type.
    let _decrease_size: Result<u8, _> = try_transmute!(AU16(0));
    //~[msrv, stable, nightly]^ ERROR: cannot transmute between types of different sizes, or dependently-sized types

    // `try_transmute!` does not support transmuting from a smaller type to a larger
    // one.
    let _increase_size: Result<AU16, _> = try_transmute!(0u8);
    //~[msrv, stable, nightly]^ ERROR: cannot transmute between types of different sizes, or dependently-sized types

    fn _transmute_mut_increase_lifetime() {
        let mut x = 0u64;
        // It is illegal to increase the lifetime scope.
        let _: &'static mut u64 = zerocopy::transmute_mut!(&mut x);
        //~[msrv, stable, nightly]^ ERROR: `x` does not live long enough
    }

    fn _transmute_ref_increase_lifetime() {
        let x = 0u64;
        // It is illegal to increase the lifetime scope.
        let _: &'static u64 = zerocopy::transmute_ref!(&x);
        //~[msrv, stable, nightly]^ ERROR: `x` does not live long enough
    }
}
