/*
 * Copyright (c) 2024
 *      Tim Woodall. All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * SPDX short identifier: BSD-2-Clause
 */

#include "config.h" // IWYU pragma: keep

#include "readtape.h"

#include <fcntl.h>
#include <unistd.h>
#include <sys/mtio.h>
#include "compat/include/compaterr.h" // IWYU pragma: keep
#include <errno.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include "compat/include/protocols/dumprestore.h"


#include "restore.h"
#include "extern.h"

#include <rmt/rmt.h>
#include "common/dumprmt.h"

static int	mt = -1;
#ifdef RRESTORE
static int	is_remote = 0;
#endif
static int	magtapein = 0;	/* input is from magtape */
static int	eot = 0;
static off_t	tapepos = 0;
static char*	readbuffer = 0;	// Data we're read from the tape but not delivered yet.
static size_t	datalen = 0;	// Number of bytes of data in readbuffer
static int	compressed = 0;	// Tape is compressed
static int	have_blocklen = 0;	// Old tapes before compression did not have blocksize recorded
static int	useCalculatedPos = 0;	// file returns the blocksize we ask for, not the blocksize that was written.
static int	ignoreCompressionFlag = 0;	// Whether to ignore the compression flag to read buggy 0.4b45 dumps

void set_ignoreCompressionFlag(void) {
	ignoreCompressionFlag = 1;
}

int open_tape(int open_remote, const char* tape)
{
#ifdef RRESTORE
	is_remote = open_remote;
#else
	(void)open_remote;
#endif
	magtapein = 0;
#ifdef RRESTORE
	if (is_remote) {
		mt = rmtopen(tape, O_RDONLY);
		if (mt != -1)
			magtapein = rmtseek(0, LSEEK_IS_MAGTAPE);
	} else
#endif
	if (strcmp(tape, "-") == 0)
		mt = 0;
	else {
		mt = open(tape, O_RDONLY, 0);
		if (mt != -1) {
			struct mtget mt_stat;
			magtapein = ioctl(mt, MTIOCGET, (char *)&mt_stat) == 0;
		}
	}
	tapepos = 0;
	datalen = 0;
	eot = 0;
	compressed = 0;

	return mt;
}

void close_tape(void)
{
	if (mt < 0)
		return;
#ifdef RRESTORE
	if (is_remote)
		rmtclose();
	else
#endif
	close(mt);
	mt = -1;
#ifdef RRESTORE
	is_remote = 0;
#endif
	magtapein = 0;
	tapepos = 0;
}

int ioctl_tape(int cmd, int count) {
#ifdef RRESTORE
	if (is_remote)
		return rmtioctl(cmd, count);
#endif
	struct mtop tcom;
	tcom.mt_op = cmd;
	tcom.mt_count = count;
	return ioctl(mt, MTIOCTOP, &tcom);
}

int magtapein_tape(void) {
	return magtapein;
}

#ifdef USE_QFA

int getpos_tape(off_t *pos) {
	int err = 0;

	if (useCalculatedPos) {
		*pos = tapepos;
#ifdef RRESTORE
	} else if (is_remote) {
		*pos = rmtseek(0, LSEEK_GET_TAPEPOS);
		err = *pos < 0;
#endif
	} else if (magtapein) {
		struct mtpos mtpos = {0};
		*pos = 0;

		struct mtop buf;
		buf.mt_op = MTSETDRVBUFFER;
		buf.mt_count = MT_ST_BOOLEANS | MT_ST_SCSI2LOGICAL;
		err = (ioctl(mt, MTIOCTOP, &buf) < 0);

		if (!err) {
			err = (ioctl(mt, MTIOCPOS, &mtpos) < 0);
			*pos = mtpos.mt_blkno;
		} else {
			warn("MTSETDRVBUFFER trying MTIOCGET instead\n");
			struct mtget mt_stat;
			err = (ioctl(mt, MTIOCGET, (char *)&mt_stat) < 0);
			if (!err)
				*pos = mt_stat.mt_blkno;
		}
	} else {
		*pos = lseek(mt, 0, SEEK_CUR);
		err = (*pos < 0);
	}
	if (err) {
		err = errno;
		fprintf(stderr, "[%jd] error: %d (getting tapepos: %jd)\n",
			(intmax_t)getpid(), err, (intmax_t)*pos);
	}
	return err;
}

int setpos_tape(off_t pos) {
	/* This will cause "odd problems" if it is called before the first block has been read so don't do that! */
	int err = 0;

	/* Drop any cached data */
	datalen = 0;
#ifdef RRESTORE
	if (is_remote)
		err = rmtseek(pos, LSEEK_GO2_TAPEPOS) < 0;
	else
#endif
	if (magtapein) {
		struct mtop buf;
		buf.mt_op = MTSETDRVBUFFER;
		buf.mt_count = MT_ST_BOOLEANS | MT_ST_SCSI2LOGICAL;
		err = (ioctl(mt, MTIOCTOP, &buf) < 0);

		if(!err) {
			buf.mt_op = MTSEEK;
			buf.mt_count = (int) pos;
			err = (ioctl(mt, MTIOCTOP, &buf) < 0);
		}
	} else {
		pos = lseek(mt, pos, SEEK_SET);
		err = (pos < 0);
	}
	if (err) {
		err = errno;
		fprintf(stdout, "[%jd] error: %d (setting tapepos: %jd)\n",
			(intmax_t)getpid(), err, (intmax_t)pos);
		return err;
	}
	return err;
}
#endif

#define PREFIXSIZE	sizeof(struct tapebuf)

static ssize_t read_some_data(void) {
	// Uncompressed tapes were written with ntrec*TP_BSIZE sized blocks.
	// Compressed tapes are written with blocks from "small" to
	// ntrec*TP_BSIZE+PREFIXSIZE except when ntrec*TP_BSIZE is the maximum
	// size for the tape when two blocks are written for incompressible
	// data.
	// Tapes that have been written from a file might have an arbitrary
	// block size.
	static size_t maxReadSize = -1;
	static size_t size = 0;
	if (size == 0)
		size = ntrec*TP_BSIZE+PREFIXSIZE;

	if (eot)
		return 0;
	ssize_t len;
	do {
		if (size > maxReadSize)
			size = maxReadSize;
#ifdef RRESTORE
		if (is_remote)
			len = rmtread(readbuffer+datalen, size);
		else
#endif
		len = read(mt, readbuffer+datalen, size);
		if(len == -1) {
			if(tapepos == 0 && maxReadSize == (size_t)-1)
				/* This might be the maximum size the tape can support */
				maxReadSize = ntrec*TP_BSIZE;
			else
				break;
		}
	} while(size > maxReadSize);

	if (len == 0)
		eot = 1;
	if (len > 0)
		datalen += len;
	return len;
}

static int tapebuf_getcompressed_mangled(const struct tapebuf* tb)
{
	return *(const uint32_t*)tb & (1U<<31)?1:0;
}
static int tapebuf_getflags_mangled(const struct tapebuf* tb)
{
	return (*(const uint32_t*)tb&0x70000000) >> 28;
}
static int tapebuf_getlength_mangled(const struct tapebuf* tb)
{
	return *(const uint32_t*)tb&0x0fffffff;
}


/* Returns 0 on eot, -1 on error, bytes read on successful read */
int readblock_tape(char* dst, size_t maxsize, size_t* sizeread) {
	/* This is "utterly horrible"
	 * The original uncompressed tapes expected to consist of tape blocks
	 * of ntrec sized blocks and nothing else
	 * Then compression came along and the tape blocks could be arbitrarily
	 * sized.
	 * But worst of all, the decoder, the resynching, everything, depended
	 * on the tape drive maintaining the block boundaries.
	 * This is why, for instance, you couldn't write a (compressed) dump to a
	 * file and then later write it to tape.
	 */

	if (!readbuffer) {
		// This should be twice the maximum size we can be asked to read.
		// Compressed tapes write blocks this size when they cannot be compressed.
		readbuffer = malloc((ntrec * TP_BSIZE + PREFIXSIZE) * 2);
		if (!readbuffer)
			errx(1, "Can't allocate memory for readbuffer");
	}

	size_t tapesize = 0;
	if (datalen < sizeof(struct tapebuf)) {
		read_some_data();
	}
	if (datalen < sizeof(struct tapebuf)) {
		return eot&&!datalen?0:-1;
	}
	if (tapepos == 0) {
		// First block on tape is uncompressed and does not have a header.
		// Note that there were some differences in tapes written from CVS before the 0.4b22 release.
		// struct tapebuf added in 4d5ed2f38cb1dce0244a05f94c8aec7f725a1f33 Wed Feb 21 16:13:04 2001 +0000
		// Modified to a single uint32_t in 2b7475327b6a1a580f76eca13a18f68a2943a5b1 Sun Mar 18 15:35:43 2001 +0000
		// Stopped the first block having a header in 86fab47e75fc141db053271654a2570fddcd0f46 Sat May 12 11:36:12 2001 +0000
		// all applied before the 0.4b22 release

		struct s_spcl spclpt;
		cvtflag = 0;
		memcpy(&spclpt, readbuffer, TP_BSIZE);
		if (converthead(&spclpt) == GOOD)
			goto goodheader;
		cvtflag++;
		if (converthead(&spclpt) == GOOD) {
			fprintf(stderr, "Converting to new file system format.\n");
			goto goodheader;
		}
		// Test if this dump was written with an initial header
		tapesize = sizeof(struct tapebuf);
		memcpy(&spclpt, readbuffer+tapesize, TP_BSIZE);
		cvtflag = 0;
		memcpy(&spclpt, readbuffer, TP_BSIZE);
		if (converthead(&spclpt) == GOOD)
			goto goodheader;
		errx(1, "Tape is not a dump tape");
goodheader:
		if (spclpt.c_type == TS_TAPE) {
			if (spclpt.c_flags & DR_COMPRESSED && !ignoreCompressionFlag)
				compressed = 1;
		} else
			errx(1, "First block is not TS_TAPE\n");

		if (spclpt.c_ntrec == 0)	// This wasn't added until compression was added.
			tapesize += ntrec * TP_BSIZE;
		else {
			have_blocklen = 1;
			tapesize += spclpt.c_ntrec * TP_BSIZE;
		}
		if (spclpt.c_ntrec > ntrec)
			errx(1, "Tape blocksize is too large, use "
					"\'-b %d\' ", spclpt.c_ntrec);
		if (tapesize < datalen)
			useCalculatedPos = 1;
		if (magtapein && tapesize != datalen) {
#ifdef USE_QFA
			/*
			 * We could enhance the QFA file to have tapepos=<logical-block>.<offset> for tapepos.
			 * Then, after a seek we'd have to discard offset bytes
			 */
			if (tapeposflag || createtapeposflag)
				errx(1, "Dump appears to have been written with a blocksize setting of -b %zu but we read %f blocks from the tape. QFA will not work",
					tapesize/TP_BSIZE, (double)datalen/TP_BSIZE);
			else
#endif
			warnx("Dump appears to have been written with a blocksize setting of -b %zu but we read %f blocks from the tape. QFA will not work",
					tapesize/TP_BSIZE, (double)datalen/TP_BSIZE);
		}
	} else if (compressed) {
		struct tapebuf* hdr = (struct tapebuf*)readbuffer;
		if (Bcvt)
			swabst("i", (u_char*)hdr);
		// Up to dump 0.4b48 tapebuffer used bitfields rather than shift and masks.
		// Tapes written on a system where the bitfields were assigned from MSB first will not work.
		// We apply a heuristic to try and determine which way it should be.
		static int mangle_bitfields = -1;
		if (mangle_bitfields == -1) {
			/* Note that dump currently restricts max block size to 16383KB
			 * (2^24-1) so the top 4 bits of the length MUST be zero.
			 * 1MB requires 21 bits to represent (as per diagrams below)
			 * 64-127MB requires 27 bits to represent - this is the largest blocksize for which the heuristics below still work.
			 * 128-255MB requires 28 bits to represent
			 * A compressed tape with a 256MB block size and an
			 * incompressible block would overflow and could not be
			 * restored without some work to encode this length in
			 * one of the unused values.
			 */
			//                                     3      2         1
			//                                     1      4         4         4        read mangled                  read normal
			// case 1a. buffer was not compressed. 0000000LLLLLLLLLLL0000000000fff0 -> mc=0 ml=length*16+flags*2     nc=0 nl=length
			//      1b. (bitfield reversed)        0fff0000000LLLLLLLLLLL0000000000 -> mc=0 ml=length                nc=0 nl=length/16+flags<<28
			// case 2a. buffer was compressed.     0000000LLLLLLLLLLLLLLLLLLLLLfff1 -> mc=0 ml=length*16+flags*2+1   nc=1 nl=length
			//      2b. (bitfield reversed)        1fff000000LLLLLLLLLLLLLLLLLLLLLL -> mc=1 ml=length                nc=? nl=length/16+flags<<28+1<<31

			if (tapebuf_getcompressed_mangled(hdr) == 1)
				// Case 2b. Only way mc can be set.
				mangle_bitfields = 1;
			/* blocks written without compression must be exactly ntrec*TP_BSIZE in size */
			else if (tapebuf_getlength_mangled(hdr) != ntrec*TP_BSIZE)
				// This covers case 2a and 1a. 2a because length will be odd when read mangled, and 1a because length will be 16x too big.
				mangle_bitfields = 0;
			else
				// Only case 1b remains.
				mangle_bitfields = 1;
			if (mangle_bitfields)
				warnx("Tape appears to have been written with different bitfield ordering\n");
		}
		if (mangle_bitfields) {
			/* rewrite the header in the expected order */
			int l = tapebuf_getlength_mangled(hdr);
			int f = tapebuf_getflags_mangled(hdr);
			int c = tapebuf_getcompressed_mangled(hdr);
			tapebuf_setlength(hdr, l);
			tapebuf_setflags(hdr, f);
			tapebuf_setcompressed(hdr, c);
		}
		tapesize = tapebuf_getlength(hdr) + sizeof(*hdr);
	} else {
		struct tapebuf* hdr = (struct tapebuf*)dst;
		tapesize = ntrec * TP_BSIZE;
		tapebuf_setlength(hdr, tapesize);
		tapebuf_setflags(hdr, 0);
		tapebuf_setcompressed(hdr, 0);
		dst += sizeof(struct tapebuf);
		maxsize -= sizeof(struct tapebuf);
	}
	if (tapesize > ntrec * TP_BSIZE + PREFIXSIZE)
		errx(1, "Tape block size error. This should not happen. Try using a larger -b setting\n");

	// At this point tapesize is the size that was originally written to tape
	// This might be bigger than the data available in readbuffer - for
	// example when reading from a file we're not following the original
	// blocking so we might have a partial block already read.
	while (datalen < tapesize &&
		read_some_data() > 0) {
	}
	if (datalen < tapesize) {
		if (have_blocklen) {
			/* Archive files weren't padded to the correct size until v0.4b50 */
			tapesize = datalen;
			warnx("Insufficient data on tape. datalen=%zu tapelen=%zu This should not happen.\n"
				"If this is not a pre v0.4b50 archive file then try using a different -b setting\n",
				datalen, tapesize);
		}
		else { // We don't know what the original blocksize was.
			tapesize = datalen;
			warnx("Attempting to handle short block read on old tape. Reading blocksize probably doesn't match writing blocksize");
		}
	}
	size_t len = tapesize;
	if (len > maxsize) {
		len = maxsize;
		warnx("Size requested (%zu) is smaller than data on tape (%zu). Some data is lost. This should not happen\n", maxsize, tapesize);
	}
	memcpy(dst, readbuffer, len);
	memmove(readbuffer, readbuffer+tapesize, datalen - tapesize);
	if (tapepos>0 && !compressed) {
		len += sizeof (struct tapebuf);
	}
	datalen -= tapesize;
	tapepos += tapesize;
	if (sizeread)
		*sizeread = len;
	return len;
}
