/*
 * Copyright (c) 2024
 *      Tim Woodall. All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * SPDX short identifier: BSD-2-Clause
 */

/*
 * mangle-tape - tool to mangle the bitfields of a tape to test the different
 * encodings that might have been created over the years.
 */

#include "config.h" // IWYU pragma: keep
#include "faketape-lib.h"

#include "bswap_header.h"

#include <inttypes.h>
#include <iostream>
#include <vector>
#include <cstring>
#include <fcntl.h>
#include <err.h>
#include <unistd.h>
#include <getopt.h>
#include <zlib.h>
#include <bzlib.h>
#include <lzo/lzo1x.h>
#include <optional>
#include <lzo/lzoconf.h>
#include <stdlib.h>
#include <zconf.h>
#include <memory>
#include <string>


// Messy. Ideally we'd use readblock_tape from restore/readtape.c but there are
// too many issues with global variables that we don't use but need to link in.

// Manually copied from protocols/dumprestore.h

#define COMPRESS_ZLIB	0
#define COMPRESS_BZLIB	1
#define COMPRESS_LZO	2

std::vector<uint8_t> databuffer;

[[noreturn]] void usage(const char* prog) {
	std::cerr << "Usage " << prog << " in-tape out-tape" << std::endl;
	exit(1);
}

[[noreturn]] void fatal(const char* msg, const char* prog) {
	std::cerr << "FATAL: " << msg << std::endl << std::endl;
	usage(prog);
}

int main(int argc, char* argv[]) {

	FakeTape	tapein;
	FakeTape	tapeout;

	const char* prog = argv[0];

	const char* opts = "dhv";
	struct option lopts[] = {
		{ .name = "version", .has_arg = 0, .flag = nullptr, .val = 'v' },
		{ .name = "debug", .has_arg = 0, .flag = nullptr, .val = 'd' },
		{ .name = "mangle-bitfields", .has_arg = 0, .flag = nullptr, .val = 1 },
		{ .name = "byteswap-output", .has_arg = 0, .flag = nullptr, .val = 4 },
		{ .name = "zlib-compress", .has_arg = 1, .flag = nullptr, .val = 5 },
		{ .name = "bzlib-compress", .has_arg = 1, .flag = nullptr, .val = 6 },
		{ .name = "lzo-compress", .has_arg = 0, .flag = nullptr, .val = 7 },
		{},
	};
	std::string tapedev;

	int ch;

	bool mangle_bitfields = false;
	bool bswap_output = false;
	bool zlib_compress = false;
	bool bzlib_compress = false;
	bool lzo_compress = false;
	int compress_level = 0;
	while (( ch = getopt_long(argc, argv, opts, lopts, nullptr)) != -1)
		switch(ch) {
			case 'd':
				tapein.debug = std::make_unique<std::ostream>(std::cerr.rdbuf());
				tapein.debug->copyfmt(std::cerr);
				tapein.debug->clear(std::cerr.rdstate());
				tapeout.debug = std::make_unique<std::ostream>(std::cerr.rdbuf());
				tapeout.debug->copyfmt(std::cerr);
				tapeout.debug->clear(std::cerr.rdstate());
				break;
			case 'v':
				std::cerr << prog << " version 1.0" << std::endl;
				break;
			case 'h':
				usage(prog);
			case 1:
				mangle_bitfields = true;
				break;
			case 4:
				bswap_output = true;
				break;
			case 5:
				if (zlib_compress || bzlib_compress || lzo_compress)
					fatal("Only one compression method can be specified", prog);
				zlib_compress = true;
				compress_level = atoi(optarg);
				break;
			case 6:
				if (zlib_compress || bzlib_compress || lzo_compress)
					fatal("Only one compression method can be specified", prog);
				bzlib_compress = true;
				compress_level = atoi(optarg);
				break;
			case 7:
				if (zlib_compress || bzlib_compress || lzo_compress)
					fatal("Only one compression method can be specified", prog);
				lzo_compress = true;
				break;
			default:
				usage(prog);
		}
	argc -= optind;
	argv += optind;

	if (argc != 2)
		usage(prog);

	/* Now open the dump image */
	warnx("Starting %s", prog);
	if(tapein.open(argv[0], O_RDONLY) != FakeTape::Response::OK) {
		err(1, "%s does not exist.",  argv[0]);
	}
	if(tapeout.open(argv[1], O_WRONLY|O_CREAT) != FakeTape::Response::OK) {
		err(1, "%s can't be opened.",  argv[1]);
	}

	std::vector<uint8_t> buffer(FakeTape::getMaxBlockSize());

	bswap_context inctx;
	bswap_context outctx;
	outctx.from_cpu = bswap_output;

	while (true) {
		bswap_header(nullptr, 0, &inctx);
		bswap_header(nullptr, 0, &outctx);
		/* Read the first block */
		size_t readlen;
		if(tapein.readData(buffer.data(), buffer.size(), &readlen) != FakeTape::Response::OK)
			err(1, "Failed to read first block from dump image");
		if (readlen == 0)
			break;

		header* hdr = reinterpret_cast<header*>(buffer.data());

		if (!checksum(hdr))
			errx(1, "This doesn't appear to be a dump tape, or was written on a system with byteswapped values");

		if (hdr->c_magic != NFS_MAGIC) {
			inctx.to_cpu = true;
			warnx("Input tape appears to be byteswapped");
		}
		bswap_header(buffer.data(), readlen, &inctx);

		if (hdr->c_magic != NFS_MAGIC)
			errx(1, "magic failure");

		bool compressed = hdr->c_flags & DR_COMPRESSED;
		uint32_t ntrec = hdr->c_ntrec;

		if (zlib_compress || bzlib_compress || lzo_compress)
			hdr->c_flags |= DR_COMPRESSED;
		else
			hdr->c_flags &= ~DR_COMPRESSED;
		setchecksum(hdr);

		std::cerr << "Tape has ntrec of " << ntrec << std::endl;

		bswap_header(buffer.data(), readlen, &outctx);

		if(tapeout.writeData(buffer.data(), readlen) != FakeTape::Response::OK)
			err(1, "Failed to write first block to dump image");

		std::optional<bool> demangle_bitfields;
		while(true) {

			if(tapein.readData(buffer.data(), buffer.size(), &readlen) != FakeTape::Response::OK)
				err(1, "Failed to read block from dump image");
			if (readlen == 0)
				break;
			uint32_t bsize = readlen;
			uint32_t ctype = 0;
			uint32_t cflag = 0;

			if (compressed) {
				if (inctx.to_cpu) {
					auto data = buffer.data();
					swap4(1, data);
				}
				uint32_t clen = *reinterpret_cast<uint32_t*>(buffer.data());
				if (!demangle_bitfields.has_value()) {
					if(clen>>31)
						// Case 2b. Only way mc can be set.
						demangle_bitfields = true;
					/* blocks written without compression must be exactly ntrec*TP_BSIZE in size */
					else if ((clen & 0xfffffff) != ntrec * TP_BSIZE)
						// This covers case 2a and 1a. 2a because length will be odd when read mangled, and 1a because length will be 16x too big.
						demangle_bitfields = false;
					else
						// Only case 1b remains.
						demangle_bitfields = true;
					if (*demangle_bitfields)
						std::cerr << "Warning: Tape appears to have been written with different bitfield ordering" << std::endl;
				}
				if (*demangle_bitfields) {
					bsize = clen & 0xfffffff;
					ctype = (clen&0x70000000)>>28;
					cflag = clen>>31;
				} else {
					bsize = clen >> 4;
					ctype = (clen&0xe)>>1;
					cflag = clen&1;
				}
				if (bsize == readlen) {
					size_t extra;
					if(tapein.readData(buffer.data() + readlen, 4, &extra) != FakeTape::Response::OK)
						err(1, "Failed to read continuation block from dump image");
					readlen += extra;
					bswap_header(buffer.data()+4, readlen-4, &inctx);
				} else if (bsize != readlen - 4) {
					err(1, "Unexpected size of compressed block");
				}
				/* TODO check ctype and use correct decompressor */
				if (cflag) {
					switch(ctype) {
						case COMPRESS_ZLIB: {
							std::vector<uint8_t> cmprbuf(FakeTape::getMaxBlockSize());
							std::swap(cmprbuf, buffer);
							uLongf decmprlen = buffer.size();
							int status = uncompress(buffer.data()+4, &decmprlen, cmprbuf.data()+4, bsize);
							switch (status) {
								case Z_OK: break;
								case Z_MEM_ERROR: errx(1, "Memory error while decompressing");
								case Z_BUF_ERROR: errx(1, "BUF error while decompressing");;
								case Z_DATA_ERROR: errx(1, "Corrupt compressed data while decompressing");
								default: errx(1, "Unexpected return from uncompress");
							}
							bsize = decmprlen;
							break;
						}
						case COMPRESS_BZLIB: {
							std::vector<uint8_t> cmprbuf(FakeTape::getMaxBlockSize());
							std::swap(cmprbuf, buffer);
							unsigned int decmprlen = buffer.size();
							int status = BZ2_bzBuffToBuffDecompress((char*)buffer.data()+4, &decmprlen, (char*)cmprbuf.data()+4, bsize, 0, 0);

							switch (status) {
								case BZ_OK: break;
								case BZ_MEM_ERROR: errx(1, "Memory error while decompressing");
								case BZ_OUTBUFF_FULL: errx(1, "BUF error while decompressing");;
								case BZ_DATA_ERROR:
								case BZ_DATA_ERROR_MAGIC:
								case BZ_UNEXPECTED_EOF:
									errx(1, "Corrupt compressed data while decompressing");
								default: errx(1, "Unexpected return from uncompress");
							}
							bsize = decmprlen;
							break;
						}
						case COMPRESS_LZO: {
							std::vector<uint8_t> cmprbuf(FakeTape::getMaxBlockSize());
							std::swap(cmprbuf, buffer);
							lzo_uint decmprlen = buffer.size();
							int status = lzo1x_decompress(cmprbuf.data()+4, bsize, buffer.data()+4, &decmprlen, nullptr);

							switch (status) {
								case LZO_E_OK: break;
								case LZO_E_ERROR: errx(1, "LZO error while decompressing");
								case LZO_E_EOF_NOT_FOUND: errx(1, "LZO No EOF error while decompressing");;
								default: errx(1, "Unexpected return from LZO uncompress");
							}
							bsize = decmprlen;
							break;
						}
					}
				}
				bswap_header(buffer.data()+4, bsize, &inctx);
				readlen = bsize + 4;
			} else {
				bswap_header(buffer.data(), readlen, &inctx);
				std::memmove(buffer.data()+4, buffer.data(), readlen);
				readlen += 4;
			}
			/* At this point we have uncompressed data with a compression header (header contains garbage) */
			cflag = 0;
			ctype = 0;
			bswap_header(buffer.data()+4, readlen-4, &outctx);
			if (zlib_compress) {
				std::vector<uint8_t> cmprbuf(FakeTape::getMaxBlockSize());
				std::swap(cmprbuf, buffer);
				uLongf cmprlen = buffer.size();
				int status = compress2(buffer.data()+4, &cmprlen, cmprbuf.data()+4, bsize, compress_level);
				switch (status) {
					case Z_OK: break;
					case Z_MEM_ERROR: errx(1, "Memory error while compressing");
					case Z_BUF_ERROR: errx(1, "BUF error while compressing");;
					case Z_STREAM_ERROR: errx(1, "Stream error while compressing");
					default: errx(1, "Unexpected return from compress");
				}
				if (cmprlen < bsize) {
					bsize = cmprlen;
					cflag = 1;
					readlen = bsize + 4;
				} else
					std::swap(cmprbuf, buffer);
			}
			uint32_t clen;
			if (mangle_bitfields)
				clen = (cflag<<31) | (ctype<<28) | bsize;
			else
				clen = (bsize<<4) | (ctype<<1) | cflag;
			*reinterpret_cast<uint32_t*>(buffer.data()) = clen;
			if (outctx.from_cpu) {
				auto data = buffer.data();
				swap4(1, data);
			}
			if (zlib_compress || bzlib_compress || lzo_compress) {
				if(tapeout.writeData(buffer.data(), readlen) != FakeTape::Response::OK)
					err(1, "Write compressed block failed");
			} else {
				if(tapeout.writeData(buffer.data()+4, readlen-4) != FakeTape::Response::OK)
					err(1, "Write uncompressed block failed");
			}
		}
		if(tapeout.writeData(buffer.data(), 0) != FakeTape::Response::OK)
			err(1, "Write EOF failed");
	}

	warnx("Finished");
}

/* vim: set sw=8 sts=8 ts=8 noexpandtab: */
