﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System;
using System.Collections.Generic;
using Cake.Core;
using Cake.Core.Annotations;
using Cake.Core.IO;

namespace Cake.Common.Tools.MSBuild
{
    /// <summary>
    /// <para>Contains functionality related to <see href="https://msdn.microsoft.com/en-us/library/dd393574.aspx">MSBuild</see>.</para>
    /// <para>
    /// In order to use the commands for this alias, MSBuild will already have to be installed on the machine the Cake Script
    /// is being executed.
    /// </para>
    /// </summary>
    [CakeAliasCategory("MSBuild")]
    public static class MSBuildAliases
    {
        /// <summary>
        /// Builds the specified solution or MsBuild project file using MSBuild.
        /// </summary>
        /// <param name="context">The context.</param>
        /// <param name="solution">The solution or MsBuild project file to build.</param>
        /// <example>
        /// <code>
        /// MSBuild("./src/Cake.sln");
        /// </code>
        /// </example>
        [CakeMethodAlias]
        public static void MSBuild(this ICakeContext context, FilePath solution)
        {
            MSBuild(context, solution, (MSBuildSettings settings) => { });
        }

        /// <summary>
        /// Builds the specified solution or MsBuild project file using MSBuild.
        /// </summary>
        /// <param name="context">The context.</param>
        /// <param name="solution">The solution or MsBuild project file to build.</param>
        /// <param name="standardOutputAction">The action to invoke with the standard output.</param>
        /// <example>
        /// <code>
        /// MSBuild("./src/Cake.sln",
        ///     output => foreach (var line in output) outputBuilder.AppendLine(line));
        /// </code>
        /// </example>
        [CakeMethodAlias]
        public static void MSBuild(this ICakeContext context, FilePath solution, Action<IEnumerable<string>> standardOutputAction)
        {
            MSBuild(context, solution, settings => { }, standardOutputAction);
        }

        /// <summary>
        /// Builds the specified solution or MsBuild project file using MSBuild.
        /// </summary>
        /// <param name="context">The context.</param>
        /// <param name="solution">The solution or MsBuild project file to build.</param>
        /// <param name="configurator">The settings configurator.</param>
        /// <example>
        /// <code>
        /// MSBuild("./src/Cake.sln", configurator =>
        ///     configurator.SetConfiguration("Debug")
        ///         .SetVerbosity(Verbosity.Minimal)
        ///         .UseToolVersion(MSBuildToolVersion.VS2015)
        ///         .SetMSBuildPlatform(MSBuildPlatform.x86)
        ///         .SetPlatformTarget(PlatformTarget.MSIL));
        /// </code>
        /// </example>
        [CakeMethodAlias]
        public static void MSBuild(this ICakeContext context, FilePath solution, Action<MSBuildSettings> configurator)
        {
            ArgumentNullException.ThrowIfNull(context);
            ArgumentNullException.ThrowIfNull(configurator);

            var settings = new MSBuildSettings();
            configurator(settings);

            // Perform the build.
            MSBuild(context, solution, settings);
        }

        /// <summary>
        /// Builds the specified solution or MsBuild project file using MSBuild.
        /// </summary>
        /// <param name="context">The context.</param>
        /// <param name="solution">The solution or MsBuild project file to build.</param>
        /// <param name="configurator">The settings configurator.</param>
        /// <param name="standardOutputAction">The action to invoke with the standard output.</param>
        /// <example>
        /// <code>
        /// var outputBuilder = new StringBuilder();
        /// MSBuild("./src/Cake.sln", configurator =>
        ///     configurator.SetConfiguration("Debug")
        ///         .SetVerbosity(Verbosity.Minimal)
        ///         .UseToolVersion(MSBuildToolVersion.VS2015)
        ///         .SetMSBuildPlatform(MSBuildPlatform.x86)
        ///         .SetPlatformTarget(PlatformTarget.MSIL),
        ///     output => foreach (var line in output) outputBuilder.AppendLine(line));
        /// </code>
        /// </example>
        [CakeMethodAlias]
        public static void MSBuild(this ICakeContext context, FilePath solution, Action<MSBuildSettings> configurator, Action<IEnumerable<string>> standardOutputAction)
        {
            ArgumentNullException.ThrowIfNull(context);
            ArgumentNullException.ThrowIfNull(configurator);
            ArgumentNullException.ThrowIfNull(standardOutputAction);

            var settings = new MSBuildSettings();
            configurator(settings);

            // Perform the build.
            MSBuild(context, solution, settings, standardOutputAction);
        }

        /// <summary>
        /// Builds the specified solution using MSBuild.
        /// </summary>
        /// <param name="context">The context.</param>
        /// <param name="solution">The solution or MsBuild project file to build.</param>
        /// <param name="settings">The settings.</param>
        /// <example>
        /// <code>
        /// MSBuild("./src/Cake.sln", new MSBuildSettings {
        ///     Verbosity = Verbosity.Minimal,
        ///     ToolVersion = MSBuildToolVersion.VS2015,
        ///     Configuration = "Release",
        ///     PlatformTarget = PlatformTarget.MSIL
        ///     });
        /// </code>
        /// </example>
        [CakeMethodAlias]
        public static void MSBuild(this ICakeContext context, FilePath solution, MSBuildSettings settings)
        {
            ArgumentNullException.ThrowIfNull(context);
            ArgumentNullException.ThrowIfNull(settings);

            var runner = new MSBuildRunner(context.FileSystem, context.Environment, context.ProcessRunner, context.Tools);
            runner.Run(solution, settings, null);
        }

        /// <summary>
        /// Builds the specified solution or MsBuild project file using MSBuild.
        /// </summary>
        /// <param name="context">The context.</param>
        /// <param name="solution">The solution to build.</param>
        /// <param name="settings">The settings.</param>
        /// <param name="standardOutputAction">The action to invoke with the standard output.</param>
        /// <example>
        /// <code>
        /// var outputBuilder = new StringBuilder();
        /// MSBuild("./src/Cake.sln", new MSBuildSettings {
        ///     Verbosity = Verbosity.Minimal,
        ///     ToolVersion = MSBuildToolVersion.VS2015,
        ///     Configuration = "Release",
        ///     PlatformTarget = PlatformTarget.MSIL
        ///     },
        ///     output => foreach (var line in output) outputBuilder.AppendLine(line));
        /// </code>
        /// </example>
        [CakeMethodAlias]
        public static void MSBuild(this ICakeContext context, FilePath solution, MSBuildSettings settings, Action<IEnumerable<string>> standardOutputAction)
        {
            ArgumentNullException.ThrowIfNull(context);
            ArgumentNullException.ThrowIfNull(settings);
            ArgumentNullException.ThrowIfNull(standardOutputAction);

            var runner = new MSBuildRunner(context.FileSystem, context.Environment, context.ProcessRunner, context.Tools);
            runner.Run(solution, settings, standardOutputAction);
        }
    }
}