package node

import (
	"encoding/json"
	"fmt"
	"reflect"

	apierrors "k8s.io/apimachinery/pkg/api/errors"
	kyaml "k8s.io/apimachinery/pkg/util/yaml"
	listersv1 "k8s.io/client-go/listers/core/v1"

	configv1 "github.com/openshift/api/config/v1"
	configlistersv1 "github.com/openshift/client-go/config/listers/config/v1"
	"github.com/openshift/library-go/pkg/operator/configobserver"
	"github.com/openshift/library-go/pkg/operator/resource/resourcemerge"
	"github.com/openshift/library-go/pkg/operator/v1helpers"
)

const (
	// static pod revision config maps used by installer controller to track
	// configs across different revisions
	revisionConfigMapName = "config"
	revisionConfigMapKey  = "config.yaml"
)

type revisionDiffProfileSuppressor struct {
	operatorClient  v1helpers.StaticPodOperatorClient
	configMapLister listersv1.ConfigMapNamespaceLister
	usedConfigPaths [][]string
}

// NewSuppressConfigUpdateUntilSameProfileFunc is used to create a conditional func (ShouldSuppressConfigUpdatesFunc)
// that can be used by the latency profile config observer to determine if a new update to observedConfig should be rolled out or not.
// It uses a config map lister and status obtained from a static pod operator client to check if all active revisions on the cluster have
// common values for the required latency config paths or not. Config maps generated by installer controller
// are read in order to determine the current active static pod revision and compare observedConfig values from it.
func NewSuppressConfigUpdateUntilSameProfileFunc(
	operatorClient v1helpers.StaticPodOperatorClient,
	configMapLister listersv1.ConfigMapNamespaceLister,
	latencyConfigs []LatencyConfigProfileTuple,
) ShouldSuppressConfigUpdatesFunc {
	ret := revisionDiffProfileSuppressor{
		operatorClient:  operatorClient,
		configMapLister: configMapLister,
		usedConfigPaths: GetUsedLatencyConfigPaths(latencyConfigs),
	}

	// creates an actual conditional fn that can be invoked each time config observer
	// intends to update observed config
	return ret.shouldSuppressConfigUpdates
}

func (s *revisionDiffProfileSuppressor) shouldSuppressConfigUpdates() (suppress bool, reason string, err error) {
	operatorSpec, operatorStatus, _, err := s.operatorClient.GetStaticPodOperatorState()
	if err != nil {
		return false, "", err
	}

	// support initial profile with any value
	if isDayZeroFromStatus(operatorStatus) {
		return false, "", nil
	}

	mergedConfigRaw, err := resourcemerge.MergeProcessConfig(
		nil,
		operatorSpec.OperatorSpec.ObservedConfig.Raw,
		operatorSpec.OperatorSpec.UnsupportedConfigOverrides.Raw,
	)
	if err != nil {
		return false, "", err
	}

	var mergedConfig map[string]interface{}
	err = json.Unmarshal(mergedConfigRaw, &mergedConfig)
	if err != nil {
		return false, "", err
	}
	mergedConfigPruned := configobserver.Pruned(mergedConfig, s.usedConfigPaths...)

	// Collect the unique set of revisions for all nodes in cluster
	uniqueRevisionMap := make(map[int32]bool)
	for _, nodeStatus := range operatorStatus.NodeStatuses {
		revision := nodeStatus.CurrentRevision
		// in order to avoid revision 0 config map not found error
		if revision == 0 {
			continue
		}
		uniqueRevisionMap[revision] = true
	}

	for revision := range uniqueRevisionMap {
		configMapNameWithRevision := fmt.Sprintf("%s-%d", revisionConfigMapName, revision)
		configMap, err := s.configMapLister.Get(configMapNameWithRevision)
		if err != nil {
			return false, "", err
		}

		// read observed config from current config map
		configData, ok := configMap.Data[revisionConfigMapKey]
		if !ok {
			return false, "", fmt.Errorf("could not find %s in %s config map from %s namespace", revisionConfigMapKey, configMap.Name, configMap.Namespace)
		}

		var currentConfig map[string]interface{}
		if err := json.Unmarshal([]byte(configData), &currentConfig); err != nil {
			return false, "", err
		}

		// prune the current config with only paths that is supposed to be monitored
		// and keep comparing with the current config, in case of mis match, suppress
		currentConfigPruned := configobserver.Pruned(currentConfig, s.usedConfigPaths...)
		if !reflect.DeepEqual(mergedConfigPruned, currentConfigPruned) {
			// suppress=true: when config values don't match
			return true, "not all static pod revisions match the observedConfig for latency profile set on the cluster", nil
		}
	}
	// suppress=false, when all config values are identical
	return false, "", nil
}

type rejectExtremeProfilesSupressor struct {
	operatorClient      v1helpers.StaticPodOperatorClient
	configNodeLister    configlistersv1.NodeLister
	usedConfigPaths     [][]string
	knownProfileConfigs map[configv1.WorkerLatencyProfileType]map[string]interface{}
	rejectionScenarios  []LatencyProfileRejectionScenario
}

// NewSuppressConfigUpdateForExtremeProfilesFunc is used to create a conditional func (ShouldSuppressConfigUpdatesFunc)
// that can be used by the latency profile config observer to determine if a new update to observedConfig should be rolled out or not.
// It checks the current value of observedConfig from operator's state (static pod operator resource), tries to determine the latency profile
// that was applied by the config observer and compares the same against the desired profile set on the cluster. In case, the update from
// the active profile to the desired profile is an extreme scenario i.e. one of the scenarios specified in rejectionScenarios,
// the suppressor suppresses the update.
func NewSuppressConfigUpdateForExtremeProfilesFunc(
	operatorClient v1helpers.StaticPodOperatorClient,
	configNodeLister configlistersv1.NodeLister,
	latencyConfigs []LatencyConfigProfileTuple,
	rejectionScenarios []LatencyProfileRejectionScenario,
) (ShouldSuppressConfigUpdatesFunc, error) {
	knownProfileConfigs, err := GenerateConfigsForEachLatencyProfile(latencyConfigs)
	if err != nil {
		return nil, err
	}

	ret := rejectExtremeProfilesSupressor{
		operatorClient:      operatorClient,
		configNodeLister:    configNodeLister,
		usedConfigPaths:     GetUsedLatencyConfigPaths(latencyConfigs),
		knownProfileConfigs: knownProfileConfigs,
		rejectionScenarios:  rejectionScenarios,
	}
	return ret.shouldSuppressConfigUpdates, nil
}

func (s *rejectExtremeProfilesSupressor) shouldSuppressConfigUpdates() (suppress bool, reason string, err error) {
	operatorSpec, operatorStatus, _, err := s.operatorClient.GetStaticPodOperatorState()
	if err != nil {
		return false, "", err
	}

	// support initial profile with any value
	if isDayZeroFromStatus(operatorStatus) {
		return false, "", nil
	}

	// raw bytes of observed config in JSON format
	observedConfigRaw, err := kyaml.ToJSON(operatorSpec.OperatorSpec.ObservedConfig.Raw)
	if err != nil {
		return false, "", err
	}

	var observedConfig map[string]interface{}
	err = json.Unmarshal(observedConfigRaw, &observedConfig)
	if err != nil {
		return false, "", err
	}
	observedConfigPruned := configobserver.Pruned(observedConfig, s.usedConfigPaths...)

	// get desired profile from config node object
	configNode, err := s.configNodeLister.Get("cluster")
	if err != nil {
		if !apierrors.IsNotFound(err) {
			return false, "", err
		}
		// suppress=False
		return false, "", nil
	}

	// desired profile (can be "" also)
	toProfile := configNode.Spec.WorkerLatencyProfile

	isRejected, fromProfile := ShouldRejectProfileTransition(observedConfigPruned, toProfile, s.knownProfileConfigs, s.rejectionScenarios)
	if isRejected {
		// suppress=True
		return true, fmt.Sprintf("rejected update from %q to %q latency profile as extreme profile transition is unsupported", fromProfile, toProfile), nil
	}
	return false, "", nil
}
