/* ************************************************************************
 * Copyright (C) 2016-2025 Advanced Micro Devices, Inc. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * ************************************************************************ */
#define ROCBLAS_NO_DEPRECATED_WARNINGS
#include "hipblas.h"
#include "exceptions.hpp"
#include "limits.h"
#include "rocblas/rocblas.h"
#ifdef __HIP_PLATFORM_SOLVER__
#include "rocsolver/rocsolver.h"
#endif
#include <algorithm>
#include <functional>
#include <hip/library_types.h>
#include <math.h>

extern "C" hipblasStatus_t hipblasConvertStatus(rocblas_status_ error);

// Attempt a rocBLAS call; if it gets an allocation error, query the
// size needed and attempt to allocate it, retrying the operation
static hipblasStatus_t hipblasDemandAlloc(rocblas_handle                   handle,
                                          std::function<hipblasStatus_t()> func)
{
    hipblasStatus_t status = func();
    if(status == HIPBLAS_STATUS_ALLOC_FAILED)
    {
        rocblas_status blas_status = rocblas_start_device_memory_size_query(handle);
        if(blas_status != rocblas_status_success)
            status = hipblasConvertStatus(blas_status);
        else
        {
            status = func();
            if(status == HIPBLAS_STATUS_SUCCESS)
            {
                size_t size;
                blas_status = rocblas_stop_device_memory_size_query(handle, &size);
                if(blas_status != rocblas_status_success)
                    status = hipblasConvertStatus(blas_status);
                else
                {
                    status = func();
                }
            }
        }
    }
    return status;
}

#define HIPBLAS_DEMAND_ALLOC(status__) \
    hipblasDemandAlloc(rocblas_handle(handle), [&]() -> hipblasStatus_t { return status__; })

extern "C" {

rocblas_operation_ hipblasConvertOperation(hipblasOperation_t op)
{
    switch(op)
    {
    case HIPBLAS_OP_N:
        return rocblas_operation_none;
    case HIPBLAS_OP_T:
        return rocblas_operation_transpose;
    case HIPBLAS_OP_C:
        return rocblas_operation_conjugate_transpose;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

rocblas_fill_ hipblasConvertFill(hipblasFillMode_t fill)
{
    switch(fill)
    {
    case HIPBLAS_FILL_MODE_UPPER:
        return rocblas_fill_upper;
    case HIPBLAS_FILL_MODE_LOWER:
        return rocblas_fill_lower;
    case HIPBLAS_FILL_MODE_FULL:
        return rocblas_fill_full;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

rocblas_diagonal_ hipblasConvertDiag(hipblasDiagType_t diagonal)
{
    switch(diagonal)
    {
    case HIPBLAS_DIAG_NON_UNIT:
        return rocblas_diagonal_non_unit;
    case HIPBLAS_DIAG_UNIT:
        return rocblas_diagonal_unit;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

rocblas_side_ hipblasConvertSide(hipblasSideMode_t side)
{
    switch(side)
    {
    case HIPBLAS_SIDE_LEFT:
        return rocblas_side_left;
    case HIPBLAS_SIDE_RIGHT:
        return rocblas_side_right;
    case HIPBLAS_SIDE_BOTH:
        return rocblas_side_both;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

rocblas_pointer_mode hipblasGetRocblasPointerMode(hipblasPointerMode_t mode)
{
    switch(mode)
    {
    case HIPBLAS_POINTER_MODE_HOST:
        return rocblas_pointer_mode_host;

    case HIPBLAS_POINTER_MODE_DEVICE:
        return rocblas_pointer_mode_device;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

hipblasPointerMode_t hipblasConvertPointerMode(rocblas_pointer_mode mode)
{
    switch(mode)
    {
    case rocblas_pointer_mode_host:
        return HIPBLAS_POINTER_MODE_HOST;

    case rocblas_pointer_mode_device:
        return HIPBLAS_POINTER_MODE_DEVICE;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

rocblas_datatype hipblasConvertDatatype(hipDataType type)
{
    switch(type)
    {
    case HIP_R_16F:
        return rocblas_datatype_f16_r;

    case HIP_R_32F:
        return rocblas_datatype_f32_r;

    case HIP_R_64F:
        return rocblas_datatype_f64_r;

    case HIP_C_16F:
        return rocblas_datatype_f16_c;

    case HIP_C_32F:
        return rocblas_datatype_f32_c;

    case HIP_C_64F:
        return rocblas_datatype_f64_c;

    case HIP_R_8I:
        return rocblas_datatype_i8_r;

    case HIP_R_8U:
        return rocblas_datatype_u8_r;

    case HIP_R_32I:
        return rocblas_datatype_i32_r;

    case HIP_R_32U:
        return rocblas_datatype_u32_r;

    case HIP_C_8I:
        return rocblas_datatype_i8_c;

    case HIP_C_8U:
        return rocblas_datatype_u8_c;

    case HIP_C_32I:
        return rocblas_datatype_i32_c;

    case HIP_C_32U:
        return rocblas_datatype_u32_c;

    case HIP_R_16BF:
        return rocblas_datatype_bf16_r;

    case HIP_C_16BF:
        return rocblas_datatype_bf16_c;

    default:
        throw HIPBLAS_STATUS_INVALID_ENUM;
    }
}

rocblas_gemm_algo hipblasConvertGemmAlgo(hipblasGemmAlgo_t algo)
{
    switch(algo)
    {
    case HIPBLAS_GEMM_DEFAULT:
        return rocblas_gemm_algo_standard;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

rocblas_gemm_flags hipblasConvertGemmFlags(hipblasGemmFlags_t flags)
{
    switch(flags)
    {
    case HIPBLAS_GEMM_FLAGS_NONE:
        return rocblas_gemm_flags_none;
    case HIPBLAS_GEMM_FLAGS_USE_CU_EFFICIENCY:
        return rocblas_gemm_flags_use_cu_efficiency;
    case HIPBLAS_GEMM_FLAGS_FP16_ALT_IMPL:
        return rocblas_gemm_flags_fp16_alt_impl;
    case HIPBLAS_GEMM_FLAGS_CHECK_SOLUTION_INDEX:
        return rocblas_gemm_flags_check_solution_index;
    case HIPBLAS_GEMM_FLAGS_FP16_ALT_IMPL_RNZ:
        return rocblas_gemm_flags_fp16_alt_impl_rnz;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

rocblas_atomics_mode hipblasConvertAtomicsMode(hipblasAtomicsMode_t mode)
{
    switch(mode)
    {
    case HIPBLAS_ATOMICS_NOT_ALLOWED:
        return rocblas_atomics_not_allowed;
    case HIPBLAS_ATOMICS_ALLOWED:
        return rocblas_atomics_allowed;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

rocblas_math_mode hipblasGetRocblasMathMode(hipblasMath_t mode)
{
    switch(mode)
    {
    case HIPBLAS_DEFAULT_MATH:
        return rocblas_default_math;
    case HIPBLAS_XF32_XDL_MATH:
        return rocblas_xf32_xdl_math_op;
    case HIPBLAS_PEDANTIC_MATH:
    case HIPBLAS_TF32_TENSOR_OP_MATH:
    case HIPBLAS_MATH_DISALLOW_REDUCED_PRECISION_REDUCTION:
    case HIPBLAS_TENSOR_OP_MATH:
        break;
    }
    throw HIPBLAS_STATUS_NOT_SUPPORTED;
}

hipblasMath_t hipblasConvertMathMode(rocblas_math_mode mode)
{
    switch(mode)
    {
    case rocblas_default_math:
        return HIPBLAS_DEFAULT_MATH;
    case rocblas_xf32_xdl_math_op:
        return HIPBLAS_XF32_XDL_MATH;
    }
    throw HIPBLAS_STATUS_INVALID_ENUM;
}

hipblasStatus_t hipblasConvertStatus(rocblas_status_ error)
{
    switch(error)
    {
    case rocblas_status_size_unchanged:
    case rocblas_status_size_increased:
    case rocblas_status_success:
        return HIPBLAS_STATUS_SUCCESS;
    case rocblas_status_invalid_handle:
        return HIPBLAS_STATUS_NOT_INITIALIZED;
    case rocblas_status_not_implemented:
        return HIPBLAS_STATUS_NOT_SUPPORTED;
    case rocblas_status_invalid_pointer:
    case rocblas_status_invalid_size:
    case rocblas_status_invalid_value:
        return HIPBLAS_STATUS_INVALID_VALUE;
    case rocblas_status_memory_error:
        return HIPBLAS_STATUS_ALLOC_FAILED;
    case rocblas_status_internal_error:
        return HIPBLAS_STATUS_INTERNAL_ERROR;
    default:
        return HIPBLAS_STATUS_UNKNOWN;
    }
}

hipblasStatus_t hipblasCreate(hipblasHandle_t* handle)
try
{
    if(!handle)
        return HIPBLAS_STATUS_HANDLE_IS_NULLPTR;

    // Create the rocBLAS handle
    return hipblasConvertStatus(rocblas_create_handle((rocblas_handle*)handle));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDestroy(hipblasHandle_t handle)
try
{
    return hipblasConvertStatus(rocblas_destroy_handle((rocblas_handle)handle));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSetStream(hipblasHandle_t handle, hipStream_t streamId)
try
{
    if(handle == nullptr)
    {
        return HIPBLAS_STATUS_NOT_INITIALIZED;
    }
    return hipblasConvertStatus(rocblas_set_stream((rocblas_handle)handle, streamId));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasGetStream(hipblasHandle_t handle, hipStream_t* streamId)
try
{
    if(handle == nullptr)
    {
        return HIPBLAS_STATUS_NOT_INITIALIZED;
    }
    return hipblasConvertStatus(rocblas_get_stream((rocblas_handle)handle, streamId));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSetPointerMode(hipblasHandle_t handle, hipblasPointerMode_t mode)
try
{
    return hipblasConvertStatus(
        rocblas_set_pointer_mode((rocblas_handle)handle, hipblasGetRocblasPointerMode(mode)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasGetPointerMode(hipblasHandle_t handle, hipblasPointerMode_t* mode)
try
{
    rocblas_pointer_mode rocblas_mode;
    rocblas_status       status = rocblas_get_pointer_mode((rocblas_handle)handle, &rocblas_mode);
    *mode                       = hipblasConvertPointerMode(rocblas_mode);
    return hipblasConvertStatus(status);
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSetMathMode(hipblasHandle_t handle, hipblasMath_t mode)
try
{
    return hipblasConvertStatus(
        rocblas_set_math_mode((rocblas_handle)handle, hipblasGetRocblasMathMode(mode)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasGetMathMode(hipblasHandle_t handle, hipblasMath_t* mode)
try
{
    rocblas_math_mode rocblas_mode;
    rocblas_status    status = rocblas_get_math_mode((rocblas_handle)handle, &rocblas_mode);
    *mode                    = hipblasConvertMathMode(rocblas_mode);
    return hipblasConvertStatus(status);
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasSetWorkspace(hipblasHandle_t handle, void* workspace, size_t workspaceSizeInBytes)
try
{
    return hipblasConvertStatus(
        rocblas_set_workspace((rocblas_handle)handle, workspace, workspaceSizeInBytes));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSetVector(int n, int elemSize, const void* x, int incx, void* y, int incy)
try
{
    return hipblasConvertStatus(rocblas_set_vector(n, elemSize, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasGetVector(int n, int elemSize, const void* x, int incx, void* y, int incy)
try
{
    return hipblasConvertStatus(rocblas_get_vector(n, elemSize, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasSetMatrix(int rows, int cols, int elemSize, const void* A, int lda, void* B, int ldb)
try
{
    return hipblasConvertStatus(rocblas_set_matrix(rows, cols, elemSize, A, lda, B, ldb));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasGetMatrix(int rows, int cols, int elemSize, const void* A, int lda, void* B, int ldb)
try
{
    return hipblasConvertStatus(rocblas_get_matrix(rows, cols, elemSize, A, lda, B, ldb));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSetVectorAsync(
    int n, int elemSize, const void* x, int incx, void* y, int incy, hipStream_t stream)
try
{
    return hipblasConvertStatus(rocblas_set_vector_async(n, elemSize, x, incx, y, incy, stream));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasGetVectorAsync(
    int n, int elemSize, const void* x, int incx, void* y, int incy, hipStream_t stream)
try
{
    return hipblasConvertStatus(rocblas_get_vector_async(n, elemSize, x, incx, y, incy, stream));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSetMatrixAsync(
    int rows, int cols, int elemSize, const void* A, int lda, void* B, int ldb, hipStream_t stream)
try
{
    return hipblasConvertStatus(
        rocblas_set_matrix_async(rows, cols, elemSize, A, lda, B, ldb, stream));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasGetMatrixAsync(
    int rows, int cols, int elemSize, const void* A, int lda, void* B, int ldb, hipStream_t stream)
try
{
    return hipblasConvertStatus(
        rocblas_get_matrix_async(rows, cols, elemSize, A, lda, B, ldb, stream));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// atomics mode
hipblasStatus_t hipblasSetAtomicsMode(hipblasHandle_t handle, hipblasAtomicsMode_t atomics_mode)
try
{
    return hipblasConvertStatus(
        rocblas_set_atomics_mode((rocblas_handle)handle, hipblasConvertAtomicsMode(atomics_mode)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasGetAtomicsMode(hipblasHandle_t handle, hipblasAtomicsMode_t* atomics_mode)
try
{
    return hipblasConvertStatus(
        rocblas_get_atomics_mode((rocblas_handle)handle, (rocblas_atomics_mode*)atomics_mode));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amax
hipblasStatus_t hipblasIsamax(hipblasHandle_t handle, int n, const float* x, int incx, int* result)
try
{
    return hipblasConvertStatus(rocblas_isamax((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIdamax(hipblasHandle_t handle, int n, const double* x, int incx, int* result)
try
{
    return hipblasConvertStatus(rocblas_idamax((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasIcamax(hipblasHandle_t handle, int n, const hipComplex* x, int incx, int* result)
try
{
    return hipblasConvertStatus(
        rocblas_icamax((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasIzamax(hipblasHandle_t handle, int n, const hipDoubleComplex* x, int incx, int* result)
try
{
    return hipblasConvertStatus(
        rocblas_izamax((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amax_64
hipblasStatus_t hipblasIsamax_64(
    hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, int64_t* result)
try
{
    return hipblasConvertStatus(rocblas_isamax_64((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIdamax_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, int64_t* result)
try
{
    return hipblasConvertStatus(rocblas_idamax_64((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIcamax_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* x, int64_t incx, int64_t* result)
try
{
    return hipblasConvertStatus(
        rocblas_icamax_64((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzamax_64(
    hipblasHandle_t handle, int64_t n, const hipDoubleComplex* x, int64_t incx, int64_t* result)
try
{
    return hipblasConvertStatus(
        rocblas_izamax_64((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amax_batched
hipblasStatus_t hipblasIsamaxBatched(
    hipblasHandle_t handle, int n, const float* const x[], int incx, int batchCount, int* result)
try
{
    return hipblasConvertStatus(
        rocblas_isamax_batched((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIdamaxBatched(
    hipblasHandle_t handle, int n, const double* const x[], int incx, int batchCount, int* result)
try
{
    return hipblasConvertStatus(
        rocblas_idamax_batched((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIcamaxBatched(hipblasHandle_t         handle,
                                     int                     n,
                                     const hipComplex* const x[],
                                     int                     incx,
                                     int                     batchCount,
                                     int*                    result)
try
{
    return hipblasConvertStatus(rocblas_icamax_batched((rocblas_handle)handle,
                                                       n,
                                                       (const rocblas_float_complex* const*)x,
                                                       incx,
                                                       batchCount,
                                                       result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzamaxBatched(hipblasHandle_t               handle,
                                     int                           n,
                                     const hipDoubleComplex* const x[],
                                     int                           incx,
                                     int                           batchCount,
                                     int*                          result)
try
{
    return hipblasConvertStatus(rocblas_izamax_batched((rocblas_handle)handle,
                                                       n,
                                                       (const rocblas_double_complex* const*)x,
                                                       incx,
                                                       batchCount,
                                                       result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amax_batched_64
hipblasStatus_t hipblasIsamaxBatched_64(hipblasHandle_t    handle,
                                        int64_t            n,
                                        const float* const x[],
                                        int64_t            incx,
                                        int64_t            batchCount,
                                        int64_t*           result)
try
{
    return hipblasConvertStatus(
        rocblas_isamax_batched_64((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIdamaxBatched_64(hipblasHandle_t     handle,
                                        int64_t             n,
                                        const double* const x[],
                                        int64_t             incx,
                                        int64_t             batchCount,
                                        int64_t*            result)
try
{
    return hipblasConvertStatus(
        rocblas_idamax_batched_64((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIcamaxBatched_64(hipblasHandle_t         handle,
                                        int64_t                 n,
                                        const hipComplex* const x[],
                                        int64_t                 incx,
                                        int64_t                 batchCount,
                                        int64_t*                result)
try
{
    return hipblasConvertStatus(rocblas_icamax_batched_64((rocblas_handle)handle,
                                                          n,
                                                          (const rocblas_float_complex* const*)x,
                                                          incx,
                                                          batchCount,
                                                          result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzamaxBatched_64(hipblasHandle_t               handle,
                                        int64_t                       n,
                                        const hipDoubleComplex* const x[],
                                        int64_t                       incx,
                                        int64_t                       batchCount,
                                        int64_t*                      result)
try
{
    return hipblasConvertStatus(rocblas_izamax_batched_64((rocblas_handle)handle,
                                                          n,
                                                          (const rocblas_double_complex* const*)x,
                                                          incx,
                                                          batchCount,
                                                          result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amax_strided_batched
hipblasStatus_t hipblasIsamaxStridedBatched(hipblasHandle_t handle,
                                            int             n,
                                            const float*    x,
                                            int             incx,
                                            hipblasStride   stridex,
                                            int             batchCount,
                                            int*            result)
try
{
    return hipblasConvertStatus(rocblas_isamax_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIdamaxStridedBatched(hipblasHandle_t handle,
                                            int             n,
                                            const double*   x,
                                            int             incx,
                                            hipblasStride   stridex,
                                            int             batchCount,
                                            int*            result)
try
{
    return hipblasConvertStatus(rocblas_idamax_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIcamaxStridedBatched(hipblasHandle_t   handle,
                                            int               n,
                                            const hipComplex* x,
                                            int               incx,
                                            hipblasStride     stridex,
                                            int               batchCount,
                                            int*              result)
try
{
    return hipblasConvertStatus(rocblas_icamax_strided_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzamaxStridedBatched(hipblasHandle_t         handle,
                                            int                     n,
                                            const hipDoubleComplex* x,
                                            int                     incx,
                                            hipblasStride           stridex,
                                            int                     batchCount,
                                            int*                    result)
try
{
    return hipblasConvertStatus(rocblas_izamax_strided_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amax_strided_batched_64
hipblasStatus_t hipblasIsamaxStridedBatched_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const float*    x,
                                               int64_t         incx,
                                               hipblasStride   stridex,
                                               int64_t         batchCount,
                                               int64_t*        result)
try
{
    return hipblasConvertStatus(rocblas_isamax_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIdamaxStridedBatched_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const double*   x,
                                               int64_t         incx,
                                               hipblasStride   stridex,
                                               int64_t         batchCount,
                                               int64_t*        result)
try
{
    return hipblasConvertStatus(rocblas_idamax_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIcamaxStridedBatched_64(hipblasHandle_t   handle,
                                               int64_t           n,
                                               const hipComplex* x,
                                               int64_t           incx,
                                               hipblasStride     stridex,
                                               int64_t           batchCount,
                                               int64_t*          result)
try
{
    return hipblasConvertStatus(rocblas_icamax_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzamaxStridedBatched_64(hipblasHandle_t         handle,
                                               int64_t                 n,
                                               const hipDoubleComplex* x,
                                               int64_t                 incx,
                                               hipblasStride           stridex,
                                               int64_t                 batchCount,
                                               int64_t*                result)
try
{
    return hipblasConvertStatus(rocblas_izamax_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amin
hipblasStatus_t hipblasIsamin(hipblasHandle_t handle, int n, const float* x, int incx, int* result)
try
{
    return hipblasConvertStatus(rocblas_isamin((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIdamin(hipblasHandle_t handle, int n, const double* x, int incx, int* result)
try
{
    return hipblasConvertStatus(rocblas_idamin((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasIcamin(hipblasHandle_t handle, int n, const hipComplex* x, int incx, int* result)
try
{
    return hipblasConvertStatus(
        rocblas_icamin((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasIzamin(hipblasHandle_t handle, int n, const hipDoubleComplex* x, int incx, int* result)
try
{
    return hipblasConvertStatus(
        rocblas_izamin((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amin_64
hipblasStatus_t hipblasIsamin_64(
    hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, int64_t* result)
try
{
    return hipblasConvertStatus(rocblas_isamin_64((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIdamin_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, int64_t* result)
try
{
    return hipblasConvertStatus(rocblas_idamin_64((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIcamin_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* x, int64_t incx, int64_t* result)
try
{
    return hipblasConvertStatus(
        rocblas_icamin_64((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzamin_64(
    hipblasHandle_t handle, int64_t n, const hipDoubleComplex* x, int64_t incx, int64_t* result)
try
{
    return hipblasConvertStatus(
        rocblas_izamin_64((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amin_batched
hipblasStatus_t hipblasIsaminBatched(
    hipblasHandle_t handle, int n, const float* const x[], int incx, int batchCount, int* result)
try
{
    return hipblasConvertStatus(
        rocblas_isamin_batched((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIdaminBatched(
    hipblasHandle_t handle, int n, const double* const x[], int incx, int batchCount, int* result)
try
{
    return hipblasConvertStatus(
        rocblas_idamin_batched((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIcaminBatched(hipblasHandle_t         handle,
                                     int                     n,
                                     const hipComplex* const x[],
                                     int                     incx,
                                     int                     batchCount,
                                     int*                    result)
try
{
    return hipblasConvertStatus(rocblas_icamin_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzaminBatched(hipblasHandle_t               handle,
                                     int                           n,
                                     const hipDoubleComplex* const x[],
                                     int                           incx,
                                     int                           batchCount,
                                     int*                          result)
try
{
    return hipblasConvertStatus(rocblas_izamin_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amin_batched_64
hipblasStatus_t hipblasIsaminBatched_64(hipblasHandle_t    handle,
                                        int64_t            n,
                                        const float* const x[],
                                        int64_t            incx,
                                        int64_t            batchCount,
                                        int64_t*           result)
try
{
    return hipblasConvertStatus(
        rocblas_isamin_batched_64((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIdaminBatched_64(hipblasHandle_t     handle,
                                        int64_t             n,
                                        const double* const x[],
                                        int64_t             incx,
                                        int64_t             batchCount,
                                        int64_t*            result)
try
{
    return hipblasConvertStatus(
        rocblas_idamin_batched_64((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIcaminBatched_64(hipblasHandle_t         handle,
                                        int64_t                 n,
                                        const hipComplex* const x[],
                                        int64_t                 incx,
                                        int64_t                 batchCount,
                                        int64_t*                result)
try
{
    return hipblasConvertStatus(rocblas_icamin_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzaminBatched_64(hipblasHandle_t               handle,
                                        int64_t                       n,
                                        const hipDoubleComplex* const x[],
                                        int64_t                       incx,
                                        int64_t                       batchCount,
                                        int64_t*                      result)
try
{
    return hipblasConvertStatus(rocblas_izamin_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amin_strided_batched
hipblasStatus_t hipblasIsaminStridedBatched(hipblasHandle_t handle,
                                            int             n,
                                            const float*    x,
                                            int             incx,
                                            hipblasStride   stridex,
                                            int             batchCount,
                                            int*            result)
try
{
    return hipblasConvertStatus(rocblas_isamin_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIdaminStridedBatched(hipblasHandle_t handle,
                                            int             n,
                                            const double*   x,
                                            int             incx,
                                            hipblasStride   stridex,
                                            int             batchCount,
                                            int*            result)
try
{
    return hipblasConvertStatus(rocblas_idamin_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIcaminStridedBatched(hipblasHandle_t   handle,
                                            int               n,
                                            const hipComplex* x,
                                            int               incx,
                                            hipblasStride     stridex,
                                            int               batchCount,
                                            int*              result)
try
{
    return hipblasConvertStatus(rocblas_icamin_strided_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzaminStridedBatched(hipblasHandle_t         handle,
                                            int                     n,
                                            const hipDoubleComplex* x,
                                            int                     incx,
                                            hipblasStride           stridex,
                                            int                     batchCount,
                                            int*                    result)
try
{
    return hipblasConvertStatus(rocblas_izamin_strided_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// amin_strided_batched_64
hipblasStatus_t hipblasIsaminStridedBatched_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const float*    x,
                                               int64_t         incx,
                                               hipblasStride   stridex,
                                               int64_t         batchCount,
                                               int64_t*        result)
try
{
    return hipblasConvertStatus(rocblas_isamin_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIdaminStridedBatched_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const double*   x,
                                               int64_t         incx,
                                               hipblasStride   stridex,
                                               int64_t         batchCount,
                                               int64_t*        result)
try
{
    return hipblasConvertStatus(rocblas_idamin_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIcaminStridedBatched_64(hipblasHandle_t   handle,
                                               int64_t           n,
                                               const hipComplex* x,
                                               int64_t           incx,
                                               hipblasStride     stridex,
                                               int64_t           batchCount,
                                               int64_t*          result)
try
{
    return hipblasConvertStatus(rocblas_icamin_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasIzaminStridedBatched_64(hipblasHandle_t         handle,
                                               int64_t                 n,
                                               const hipDoubleComplex* x,
                                               int64_t                 incx,
                                               hipblasStride           stridex,
                                               int64_t                 batchCount,
                                               int64_t*                result)
try
{
    return hipblasConvertStatus(rocblas_izamin_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// asum
hipblasStatus_t hipblasSasum(hipblasHandle_t handle, int n, const float* x, int incx, float* result)
try
{
    return hipblasConvertStatus(rocblas_sasum((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasDasum(hipblasHandle_t handle, int n, const double* x, int incx, double* result)
try
{
    return hipblasConvertStatus(rocblas_dasum((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasScasum(hipblasHandle_t handle, int n, const hipComplex* x, int incx, float* result)
try
{
    return hipblasConvertStatus(
        rocblas_scasum((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDzasum(
    hipblasHandle_t handle, int n, const hipDoubleComplex* x, int incx, double* result)
try
{
    return hipblasConvertStatus(
        rocblas_dzasum((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// asum_64
hipblasStatus_t
    hipblasSasum_64(hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, float* result)
try
{
    return hipblasConvertStatus(rocblas_sasum_64((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDasum_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, double* result)
try
{
    return hipblasConvertStatus(rocblas_dasum_64((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasScasum_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* x, int64_t incx, float* result)
try
{
    return hipblasConvertStatus(
        rocblas_scasum_64((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDzasum_64(
    hipblasHandle_t handle, int64_t n, const hipDoubleComplex* x, int64_t incx, double* result)
try
{
    return hipblasConvertStatus(
        rocblas_dzasum_64((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// asum_batched
hipblasStatus_t hipblasSasumBatched(
    hipblasHandle_t handle, int n, const float* const x[], int incx, int batchCount, float* result)
try
{
    return hipblasConvertStatus(
        rocblas_sasum_batched((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDasumBatched(hipblasHandle_t     handle,
                                    int                 n,
                                    const double* const x[],
                                    int                 incx,
                                    int                 batchCount,
                                    double*             result)
try
{
    return hipblasConvertStatus(
        rocblas_dasum_batched((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasScasumBatched(hipblasHandle_t         handle,
                                     int                     n,
                                     const hipComplex* const x[],
                                     int                     incx,
                                     int                     batchCount,
                                     float*                  result)
try
{
    return hipblasConvertStatus(rocblas_scasum_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDzasumBatched(hipblasHandle_t               handle,
                                     int                           n,
                                     const hipDoubleComplex* const x[],
                                     int                           incx,
                                     int                           batchCount,
                                     double*                       result)
try
{
    return hipblasConvertStatus(rocblas_dzasum_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// asum_batched_64
hipblasStatus_t hipblasSasumBatched_64(hipblasHandle_t    handle,
                                       int64_t            n,
                                       const float* const x[],
                                       int64_t            incx,
                                       int64_t            batchCount,
                                       float*             result)
try
{
    return hipblasConvertStatus(
        rocblas_sasum_batched_64((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDasumBatched_64(hipblasHandle_t     handle,
                                       int64_t             n,
                                       const double* const x[],
                                       int64_t             incx,
                                       int64_t             batchCount,
                                       double*             result)
try
{
    return hipblasConvertStatus(
        rocblas_dasum_batched_64((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasScasumBatched_64(hipblasHandle_t         handle,
                                        int64_t                 n,
                                        const hipComplex* const x[],
                                        int64_t                 incx,
                                        int64_t                 batchCount,
                                        float*                  result)
try
{
    return hipblasConvertStatus(rocblas_scasum_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDzasumBatched_64(hipblasHandle_t               handle,
                                        int64_t                       n,
                                        const hipDoubleComplex* const x[],
                                        int64_t                       incx,
                                        int64_t                       batchCount,
                                        double*                       result)
try
{
    return hipblasConvertStatus(rocblas_dzasum_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// asum_strided_batched
hipblasStatus_t hipblasSasumStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const float*    x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount,
                                           float*          result)
try
{
    return hipblasConvertStatus(rocblas_sasum_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDasumStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const double*   x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount,
                                           double*         result)
try
{
    return hipblasConvertStatus(rocblas_dasum_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasScasumStridedBatched(hipblasHandle_t   handle,
                                            int               n,
                                            const hipComplex* x,
                                            int               incx,
                                            hipblasStride     stridex,
                                            int               batchCount,
                                            float*            result)
try
{
    return hipblasConvertStatus(rocblas_scasum_strided_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDzasumStridedBatched(hipblasHandle_t         handle,
                                            int                     n,
                                            const hipDoubleComplex* x,
                                            int                     incx,
                                            hipblasStride           stridex,
                                            int                     batchCount,
                                            double*                 result)
try
{
    return hipblasConvertStatus(rocblas_dzasum_strided_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// asum_strided_batched_64
hipblasStatus_t hipblasSasumStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const float*    x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount,
                                              float*          result)
try
{
    return hipblasConvertStatus(rocblas_sasum_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDasumStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const double*   x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount,
                                              double*         result)
try
{
    return hipblasConvertStatus(rocblas_dasum_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasScasumStridedBatched_64(hipblasHandle_t   handle,
                                               int64_t           n,
                                               const hipComplex* x,
                                               int64_t           incx,
                                               hipblasStride     stridex,
                                               int64_t           batchCount,
                                               float*            result)
try
{
    return hipblasConvertStatus(rocblas_scasum_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDzasumStridedBatched_64(hipblasHandle_t         handle,
                                               int64_t                 n,
                                               const hipDoubleComplex* x,
                                               int64_t                 incx,
                                               hipblasStride           stridex,
                                               int64_t                 batchCount,
                                               double*                 result)
try
{
    return hipblasConvertStatus(rocblas_dzasum_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// axpy
hipblasStatus_t hipblasHaxpy(hipblasHandle_t    handle,
                             int                n,
                             const hipblasHalf* alpha,
                             const hipblasHalf* x,
                             int                incx,
                             hipblasHalf*       y,
                             int                incy)
try
{
    return hipblasConvertStatus(rocblas_haxpy((rocblas_handle)handle,
                                              n,
                                              (rocblas_half*)alpha,
                                              (rocblas_half*)x,
                                              incx,
                                              (rocblas_half*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSaxpy(
    hipblasHandle_t handle, int n, const float* alpha, const float* x, int incx, float* y, int incy)
try
{
    return hipblasConvertStatus(rocblas_saxpy((rocblas_handle)handle, n, alpha, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDaxpy(hipblasHandle_t handle,
                             int             n,
                             const double*   alpha,
                             const double*   x,
                             int             incx,
                             double*         y,
                             int             incy)
try
{
    return hipblasConvertStatus(rocblas_daxpy((rocblas_handle)handle, n, alpha, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCaxpy(hipblasHandle_t   handle,
                             int               n,
                             const hipComplex* alpha,
                             const hipComplex* x,
                             int               incx,
                             hipComplex*       y,
                             int               incy)
try
{
    return hipblasConvertStatus(rocblas_caxpy((rocblas_handle)handle,
                                              n,
                                              (rocblas_float_complex*)alpha,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZaxpy(hipblasHandle_t         handle,
                             int                     n,
                             const hipDoubleComplex* alpha,
                             const hipDoubleComplex* x,
                             int                     incx,
                             hipDoubleComplex*       y,
                             int                     incy)
try
{
    return hipblasConvertStatus(rocblas_zaxpy((rocblas_handle)handle,
                                              n,
                                              (rocblas_double_complex*)alpha,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// 64-bit interface
hipblasStatus_t hipblasHaxpy_64(hipblasHandle_t    handle,
                                int64_t            n,
                                const hipblasHalf* alpha,
                                const hipblasHalf* x,
                                int64_t            incx,
                                hipblasHalf*       y,
                                int64_t            incy)
try
{
    return hipblasConvertStatus(rocblas_haxpy_64((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_half*)alpha,
                                                 (rocblas_half*)x,
                                                 incx,
                                                 (rocblas_half*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSaxpy_64(hipblasHandle_t handle,
                                int64_t         n,
                                const float*    alpha,
                                const float*    x,
                                int64_t         incx,
                                float*          y,
                                int64_t         incy)
try
{
    return hipblasConvertStatus(
        rocblas_saxpy_64((rocblas_handle)handle, n, alpha, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDaxpy_64(hipblasHandle_t handle,
                                int64_t         n,
                                const double*   alpha,
                                const double*   x,
                                int64_t         incx,
                                double*         y,
                                int64_t         incy)
try
{
    return hipblasConvertStatus(
        rocblas_daxpy_64((rocblas_handle)handle, n, alpha, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCaxpy_64(hipblasHandle_t   handle,
                                int64_t           n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int64_t           incx,
                                hipComplex*       y,
                                int64_t           incy)
try
{
    return hipblasConvertStatus(rocblas_caxpy_64((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_float_complex*)alpha,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZaxpy_64(hipblasHandle_t         handle,
                                int64_t                 n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int64_t                 incx,
                                hipDoubleComplex*       y,
                                int64_t                 incy)
try
{
    return hipblasConvertStatus(rocblas_zaxpy_64((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// axpy_batched
hipblasStatus_t hipblasHaxpyBatched(hipblasHandle_t          handle,
                                    int                      n,
                                    const hipblasHalf*       alpha,
                                    const hipblasHalf* const x[],
                                    int                      incx,
                                    hipblasHalf* const       y[],
                                    int                      incy,
                                    int                      batchCount)
try
{
    return hipblasConvertStatus(rocblas_haxpy_batched((rocblas_handle)handle,
                                                      n,
                                                      (rocblas_half*)alpha,
                                                      (rocblas_half* const*)x,
                                                      incx,
                                                      (rocblas_half* const*)y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSaxpyBatched(hipblasHandle_t    handle,
                                    int                n,
                                    const float*       alpha,
                                    const float* const x[],
                                    int                incx,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_saxpy_batched((rocblas_handle)handle, n, alpha, x, incx, y, incy, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDaxpyBatched(hipblasHandle_t     handle,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const x[],
                                    int                 incx,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_daxpy_batched((rocblas_handle)handle, n, alpha, x, incx, y, incy, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCaxpyBatched(hipblasHandle_t         handle,
                                    int                     n,
                                    const hipComplex*       alpha,
                                    const hipComplex* const x[],
                                    int                     incx,
                                    hipComplex* const       y[],
                                    int                     incy,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_caxpy_batched((rocblas_handle)handle,
                                                      n,
                                                      (rocblas_float_complex*)alpha,
                                                      (rocblas_float_complex* const*)x,
                                                      incx,
                                                      (rocblas_float_complex* const*)y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZaxpyBatched(hipblasHandle_t               handle,
                                    int                           n,
                                    const hipDoubleComplex*       alpha,
                                    const hipDoubleComplex* const x[],
                                    int                           incx,
                                    hipDoubleComplex* const       y[],
                                    int                           incy,
                                    int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zaxpy_batched((rocblas_handle)handle,
                                                      n,
                                                      (rocblas_double_complex*)alpha,
                                                      (rocblas_double_complex* const*)x,
                                                      incx,
                                                      (rocblas_double_complex* const*)y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// 64-bit interface
hipblasStatus_t hipblasHaxpyBatched_64(hipblasHandle_t          handle,
                                       int64_t                  n,
                                       const hipblasHalf*       alpha,
                                       const hipblasHalf* const x[],
                                       int64_t                  incx,
                                       hipblasHalf* const       y[],
                                       int64_t                  incy,
                                       int64_t                  batchCount)
try
{
    return hipblasConvertStatus(rocblas_haxpy_batched_64((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_half*)alpha,
                                                         (rocblas_half* const*)x,
                                                         incx,
                                                         (rocblas_half* const*)y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSaxpyBatched_64(hipblasHandle_t    handle,
                                       int64_t            n,
                                       const float*       alpha,
                                       const float* const x[],
                                       int64_t            incx,
                                       float* const       y[],
                                       int64_t            incy,
                                       int64_t            batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_saxpy_batched_64((rocblas_handle)handle, n, alpha, x, incx, y, incy, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDaxpyBatched_64(hipblasHandle_t     handle,
                                       int64_t             n,
                                       const double*       alpha,
                                       const double* const x[],
                                       int64_t             incx,
                                       double* const       y[],
                                       int64_t             incy,
                                       int64_t             batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_daxpy_batched_64((rocblas_handle)handle, n, alpha, x, incx, y, incy, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCaxpyBatched_64(hipblasHandle_t         handle,
                                       int64_t                 n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int64_t                 incx,
                                       hipComplex* const       y[],
                                       int64_t                 incy,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_caxpy_batched_64((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex* const*)x,
                                                         incx,
                                                         (rocblas_float_complex* const*)y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZaxpyBatched_64(hipblasHandle_t               handle,
                                       int64_t                       n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int64_t                       incx,
                                       hipDoubleComplex* const       y[],
                                       int64_t                       incy,
                                       int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_zaxpy_batched_64((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex* const*)x,
                                                         incx,
                                                         (rocblas_double_complex* const*)y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// axpy_strided_batched
hipblasStatus_t hipblasHaxpyStridedBatched(hipblasHandle_t    handle,
                                           int                n,
                                           const hipblasHalf* alpha,
                                           const hipblasHalf* x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           hipblasHalf*       y,
                                           int                incy,
                                           hipblasStride      stridey,
                                           int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_haxpy_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              (rocblas_half*)alpha,
                                                              (rocblas_half*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_half*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSaxpyStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const float*    alpha,
                                           const float*    x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           float*          y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_saxpy_strided_batched(
        (rocblas_handle)handle, n, alpha, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDaxpyStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const double*   alpha,
                                           const double*   x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           double*         y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_daxpy_strided_batched(
        (rocblas_handle)handle, n, alpha, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCaxpyStridedBatched(hipblasHandle_t   handle,
                                           int               n,
                                           const hipComplex* alpha,
                                           const hipComplex* x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           hipComplex*       y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_caxpy_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              (rocblas_float_complex*)alpha,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZaxpyStridedBatched(hipblasHandle_t         handle,
                                           int                     n,
                                           const hipDoubleComplex* alpha,
                                           const hipDoubleComplex* x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           hipDoubleComplex*       y,
                                           int                     incy,
                                           hipblasStride           stridey,
                                           int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zaxpy_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              (rocblas_double_complex*)alpha,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// 64-bit interface
hipblasStatus_t hipblasHaxpyStridedBatched_64(hipblasHandle_t    handle,
                                              int64_t            n,
                                              const hipblasHalf* alpha,
                                              const hipblasHalf* x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              hipblasHalf*       y,
                                              int64_t            incy,
                                              hipblasStride      stridey,
                                              int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_haxpy_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_half*)alpha,
                                                                 (rocblas_half*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_half*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSaxpyStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const float*    alpha,
                                              const float*    x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              float*          y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_saxpy_strided_batched_64(
        (rocblas_handle)handle, n, alpha, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDaxpyStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const double*   alpha,
                                              const double*   x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              double*         y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_daxpy_strided_batched_64(
        (rocblas_handle)handle, n, alpha, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCaxpyStridedBatched_64(hipblasHandle_t   handle,
                                              int64_t           n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              hipComplex*       y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_caxpy_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZaxpyStridedBatched_64(hipblasHandle_t         handle,
                                              int64_t                 n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              hipDoubleComplex*       y,
                                              int64_t                 incy,
                                              hipblasStride           stridey,
                                              int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zaxpy_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// copy
hipblasStatus_t
    hipblasScopy(hipblasHandle_t handle, int n, const float* x, int incx, float* y, int incy)
try
{
    return hipblasConvertStatus(rocblas_scopy((rocblas_handle)handle, n, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasDcopy(hipblasHandle_t handle, int n, const double* x, int incx, double* y, int incy)
try
{
    return hipblasConvertStatus(rocblas_dcopy((rocblas_handle)handle, n, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCcopy(
    hipblasHandle_t handle, int n, const hipComplex* x, int incx, hipComplex* y, int incy)
try
{
    return hipblasConvertStatus(rocblas_ccopy((rocblas_handle)handle,
                                              n,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZcopy(hipblasHandle_t         handle,
                             int                     n,
                             const hipDoubleComplex* x,
                             int                     incx,
                             hipDoubleComplex*       y,
                             int                     incy)
try
{
    return hipblasConvertStatus(rocblas_zcopy((rocblas_handle)handle,
                                              n,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// 64-bit interface
hipblasStatus_t hipblasScopy_64(
    hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, float* y, int64_t incy)
try
{
    return hipblasConvertStatus(rocblas_scopy_64((rocblas_handle)handle, n, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDcopy_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, double* y, int64_t incy)
try
{
    return hipblasConvertStatus(rocblas_dcopy_64((rocblas_handle)handle, n, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCcopy_64(hipblasHandle_t   handle,
                                int64_t           n,
                                const hipComplex* x,
                                int64_t           incx,
                                hipComplex*       y,
                                int64_t           incy)
try
{
    return hipblasConvertStatus(rocblas_ccopy_64((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZcopy_64(hipblasHandle_t         handle,
                                int64_t                 n,
                                const hipDoubleComplex* x,
                                int64_t                 incx,
                                hipDoubleComplex*       y,
                                int64_t                 incy)
try
{
    return hipblasConvertStatus(rocblas_zcopy_64((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// copy_batched
hipblasStatus_t hipblasScopyBatched(hipblasHandle_t    handle,
                                    int                n,
                                    const float* const x[],
                                    int                incx,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_scopy_batched((rocblas_handle)handle, n, x, incx, y, incy, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDcopyBatched(hipblasHandle_t     handle,
                                    int                 n,
                                    const double* const x[],
                                    int                 incx,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_dcopy_batched((rocblas_handle)handle, n, x, incx, y, incy, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCcopyBatched(hipblasHandle_t         handle,
                                    int                     n,
                                    const hipComplex* const x[],
                                    int                     incx,
                                    hipComplex* const       y[],
                                    int                     incy,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_ccopy_batched((rocblas_handle)handle,
                                                      n,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      (rocblas_float_complex**)y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZcopyBatched(hipblasHandle_t               handle,
                                    int                           n,
                                    const hipDoubleComplex* const x[],
                                    int                           incx,
                                    hipDoubleComplex* const       y[],
                                    int                           incy,
                                    int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zcopy_batched((rocblas_handle)handle,
                                                      n,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      (rocblas_double_complex**)y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// 64-bit interface
hipblasStatus_t hipblasScopyBatched_64(hipblasHandle_t    handle,
                                       int64_t            n,
                                       const float* const x[],
                                       int64_t            incx,
                                       float* const       y[],
                                       int64_t            incy,
                                       int64_t            batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_scopy_batched_64((rocblas_handle)handle, n, x, incx, y, incy, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDcopyBatched_64(hipblasHandle_t     handle,
                                       int64_t             n,
                                       const double* const x[],
                                       int64_t             incx,
                                       double* const       y[],
                                       int64_t             incy,
                                       int64_t             batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_dcopy_batched_64((rocblas_handle)handle, n, x, incx, y, incy, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCcopyBatched_64(hipblasHandle_t         handle,
                                       int64_t                 n,
                                       const hipComplex* const x[],
                                       int64_t                 incx,
                                       hipComplex* const       y[],
                                       int64_t                 incy,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_ccopy_batched_64((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZcopyBatched_64(hipblasHandle_t               handle,
                                       int64_t                       n,
                                       const hipDoubleComplex* const x[],
                                       int64_t                       incx,
                                       hipDoubleComplex* const       y[],
                                       int64_t                       incy,
                                       int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_zcopy_batched_64((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// copy_strided_batched
hipblasStatus_t hipblasScopyStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const float*    x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           float*          y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_scopy_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDcopyStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const double*   x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           double*         y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dcopy_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCcopyStridedBatched(hipblasHandle_t   handle,
                                           int               n,
                                           const hipComplex* x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           hipComplex*       y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_ccopy_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZcopyStridedBatched(hipblasHandle_t         handle,
                                           int                     n,
                                           const hipDoubleComplex* x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           hipDoubleComplex*       y,
                                           int                     incy,
                                           hipblasStride           stridey,
                                           int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zcopy_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// 64-bit interface
hipblasStatus_t hipblasScopyStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const float*    x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              float*          y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_scopy_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDcopyStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const double*   x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              double*         y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_dcopy_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCcopyStridedBatched_64(hipblasHandle_t   handle,
                                              int64_t           n,
                                              const hipComplex* x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              hipComplex*       y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_ccopy_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZcopyStridedBatched_64(hipblasHandle_t         handle,
                                              int64_t                 n,
                                              const hipDoubleComplex* x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              hipDoubleComplex*       y,
                                              int64_t                 incy,
                                              hipblasStride           stridey,
                                              int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zcopy_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// dot
hipblasStatus_t hipblasHdot(hipblasHandle_t    handle,
                            int                n,
                            const hipblasHalf* x,
                            int                incx,
                            const hipblasHalf* y,
                            int                incy,
                            hipblasHalf*       result)
try
{
    return hipblasConvertStatus(rocblas_hdot((rocblas_handle)handle,
                                             n,
                                             (rocblas_half*)x,
                                             incx,
                                             (rocblas_half*)y,
                                             incy,
                                             (rocblas_half*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasBfdot(hipblasHandle_t        handle,
                             int                    n,
                             const hipblasBfloat16* x,
                             int                    incx,
                             const hipblasBfloat16* y,
                             int                    incy,
                             hipblasBfloat16*       result)
try
{
    return hipblasConvertStatus(rocblas_bfdot((rocblas_handle)handle,
                                              n,
                                              (rocblas_bfloat16*)x,
                                              incx,
                                              (rocblas_bfloat16*)y,
                                              incy,
                                              (rocblas_bfloat16*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSdot(hipblasHandle_t handle,
                            int             n,
                            const float*    x,
                            int             incx,
                            const float*    y,
                            int             incy,
                            float*          result)
try
{
    return hipblasConvertStatus(rocblas_sdot((rocblas_handle)handle, n, x, incx, y, incy, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDdot(hipblasHandle_t handle,
                            int             n,
                            const double*   x,
                            int             incx,
                            const double*   y,
                            int             incy,
                            double*         result)
try
{
    return hipblasConvertStatus(rocblas_ddot((rocblas_handle)handle, n, x, incx, y, incy, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotc(hipblasHandle_t   handle,
                             int               n,
                             const hipComplex* x,
                             int               incx,
                             const hipComplex* y,
                             int               incy,
                             hipComplex*       result)
try
{
    return hipblasConvertStatus(rocblas_cdotc((rocblas_handle)handle,
                                              n,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)y,
                                              incy,
                                              (rocblas_float_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotu(hipblasHandle_t   handle,
                             int               n,
                             const hipComplex* x,
                             int               incx,
                             const hipComplex* y,
                             int               incy,
                             hipComplex*       result)
try
{
    return hipblasConvertStatus(rocblas_cdotu((rocblas_handle)handle,
                                              n,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)y,
                                              incy,
                                              (rocblas_float_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotc(hipblasHandle_t         handle,
                             int                     n,
                             const hipDoubleComplex* x,
                             int                     incx,
                             const hipDoubleComplex* y,
                             int                     incy,
                             hipDoubleComplex*       result)
try
{
    return hipblasConvertStatus(rocblas_zdotc((rocblas_handle)handle,
                                              n,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)y,
                                              incy,
                                              (rocblas_double_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotu(hipblasHandle_t         handle,
                             int                     n,
                             const hipDoubleComplex* x,
                             int                     incx,
                             const hipDoubleComplex* y,
                             int                     incy,
                             hipDoubleComplex*       result)
try
{
    return hipblasConvertStatus(rocblas_zdotu((rocblas_handle)handle,
                                              n,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)y,
                                              incy,
                                              (rocblas_double_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// dot_64
hipblasStatus_t hipblasHdot_64(hipblasHandle_t    handle,
                               int64_t            n,
                               const hipblasHalf* x,
                               int64_t            incx,
                               const hipblasHalf* y,
                               int64_t            incy,
                               hipblasHalf*       result)
try
{
    return hipblasConvertStatus(rocblas_hdot_64((rocblas_handle)handle,
                                                n,
                                                (rocblas_half*)x,
                                                incx,
                                                (rocblas_half*)y,
                                                incy,
                                                (rocblas_half*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasBfdot_64(hipblasHandle_t        handle,
                                int64_t                n,
                                const hipblasBfloat16* x,
                                int64_t                incx,
                                const hipblasBfloat16* y,
                                int64_t                incy,
                                hipblasBfloat16*       result)
try
{
    return hipblasConvertStatus(rocblas_bfdot_64((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_bfloat16*)x,
                                                 incx,
                                                 (rocblas_bfloat16*)y,
                                                 incy,
                                                 (rocblas_bfloat16*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSdot_64(hipblasHandle_t handle,
                               int64_t         n,
                               const float*    x,
                               int64_t         incx,
                               const float*    y,
                               int64_t         incy,
                               float*          result)
try
{
    return hipblasConvertStatus(
        rocblas_sdot_64((rocblas_handle)handle, n, x, incx, y, incy, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDdot_64(hipblasHandle_t handle,
                               int64_t         n,
                               const double*   x,
                               int64_t         incx,
                               const double*   y,
                               int64_t         incy,
                               double*         result)
try
{
    return hipblasConvertStatus(
        rocblas_ddot_64((rocblas_handle)handle, n, x, incx, y, incy, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotc_64(hipblasHandle_t   handle,
                                int64_t           n,
                                const hipComplex* x,
                                int64_t           incx,
                                const hipComplex* y,
                                int64_t           incy,
                                hipComplex*       result)
try
{
    return hipblasConvertStatus(rocblas_cdotc_64((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)y,
                                                 incy,
                                                 (rocblas_float_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotu_64(hipblasHandle_t   handle,
                                int64_t           n,
                                const hipComplex* x,
                                int64_t           incx,
                                const hipComplex* y,
                                int64_t           incy,
                                hipComplex*       result)
try
{
    return hipblasConvertStatus(rocblas_cdotu_64((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)y,
                                                 incy,
                                                 (rocblas_float_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotc_64(hipblasHandle_t         handle,
                                int64_t                 n,
                                const hipDoubleComplex* x,
                                int64_t                 incx,
                                const hipDoubleComplex* y,
                                int64_t                 incy,
                                hipDoubleComplex*       result)
try
{
    return hipblasConvertStatus(rocblas_zdotc_64((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)y,
                                                 incy,
                                                 (rocblas_double_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotu_64(hipblasHandle_t         handle,
                                int64_t                 n,
                                const hipDoubleComplex* x,
                                int64_t                 incx,
                                const hipDoubleComplex* y,
                                int64_t                 incy,
                                hipDoubleComplex*       result)
try
{
    return hipblasConvertStatus(rocblas_zdotu_64((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)y,
                                                 incy,
                                                 (rocblas_double_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// dot_batched
hipblasStatus_t hipblasHdotBatched(hipblasHandle_t          handle,
                                   int                      n,
                                   const hipblasHalf* const x[],
                                   int                      incx,
                                   const hipblasHalf* const y[],
                                   int                      incy,
                                   int                      batchCount,
                                   hipblasHalf*             result)
try
{
    return hipblasConvertStatus(rocblas_hdot_batched((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_half* const*)x,
                                                     incx,
                                                     (rocblas_half* const*)y,
                                                     incy,
                                                     batchCount,
                                                     (rocblas_half*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasBfdotBatched(hipblasHandle_t              handle,
                                    int                          n,
                                    const hipblasBfloat16* const x[],
                                    int                          incx,
                                    const hipblasBfloat16* const y[],
                                    int                          incy,
                                    int                          batchCount,
                                    hipblasBfloat16*             result)
try
{
    return hipblasConvertStatus(rocblas_bfdot_batched((rocblas_handle)handle,
                                                      n,
                                                      (rocblas_bfloat16* const*)x,
                                                      incx,
                                                      (rocblas_bfloat16* const*)y,
                                                      incy,
                                                      batchCount,
                                                      (rocblas_bfloat16*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSdotBatched(hipblasHandle_t    handle,
                                   int                n,
                                   const float* const x[],
                                   int                incx,
                                   const float* const y[],
                                   int                incy,
                                   int                batchCount,
                                   float*             result)
try
{
    return hipblasConvertStatus(
        rocblas_sdot_batched((rocblas_handle)handle, n, x, incx, y, incy, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDdotBatched(hipblasHandle_t     handle,
                                   int                 n,
                                   const double* const x[],
                                   int                 incx,
                                   const double* const y[],
                                   int                 incy,
                                   int                 batchCount,
                                   double*             result)
try
{
    return hipblasConvertStatus(
        rocblas_ddot_batched((rocblas_handle)handle, n, x, incx, y, incy, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotcBatched(hipblasHandle_t         handle,
                                    int                     n,
                                    const hipComplex* const x[],
                                    int                     incx,
                                    const hipComplex* const y[],
                                    int                     incy,
                                    int                     batchCount,
                                    hipComplex*             result)
try
{
    return hipblasConvertStatus(rocblas_cdotc_batched((rocblas_handle)handle,
                                                      n,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      (rocblas_float_complex**)y,
                                                      incy,
                                                      batchCount,
                                                      (rocblas_float_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotuBatched(hipblasHandle_t         handle,
                                    int                     n,
                                    const hipComplex* const x[],
                                    int                     incx,
                                    const hipComplex* const y[],
                                    int                     incy,
                                    int                     batchCount,
                                    hipComplex*             result)
try
{
    return hipblasConvertStatus(rocblas_cdotu_batched((rocblas_handle)handle,
                                                      n,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      (rocblas_float_complex**)y,
                                                      incy,
                                                      batchCount,
                                                      (rocblas_float_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotcBatched(hipblasHandle_t               handle,
                                    int                           n,
                                    const hipDoubleComplex* const x[],
                                    int                           incx,
                                    const hipDoubleComplex* const y[],
                                    int                           incy,
                                    int                           batchCount,
                                    hipDoubleComplex*             result)
try
{
    return hipblasConvertStatus(rocblas_zdotc_batched((rocblas_handle)handle,
                                                      n,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      (rocblas_double_complex**)y,
                                                      incy,
                                                      batchCount,
                                                      (rocblas_double_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotuBatched(hipblasHandle_t               handle,
                                    int                           n,
                                    const hipDoubleComplex* const x[],
                                    int                           incx,
                                    const hipDoubleComplex* const y[],
                                    int                           incy,
                                    int                           batchCount,
                                    hipDoubleComplex*             result)
try
{
    return hipblasConvertStatus(rocblas_zdotu_batched((rocblas_handle)handle,
                                                      n,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      (rocblas_double_complex**)y,
                                                      incy,
                                                      batchCount,
                                                      (rocblas_double_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// dot_batched_64
hipblasStatus_t hipblasHdotBatched_64(hipblasHandle_t          handle,
                                      int64_t                  n,
                                      const hipblasHalf* const x[],
                                      int64_t                  incx,
                                      const hipblasHalf* const y[],
                                      int64_t                  incy,
                                      int64_t                  batchCount,
                                      hipblasHalf*             result)
try
{
    return hipblasConvertStatus(rocblas_hdot_batched_64((rocblas_handle)handle,
                                                        n,
                                                        (rocblas_half* const*)x,
                                                        incx,
                                                        (rocblas_half* const*)y,
                                                        incy,
                                                        batchCount,
                                                        (rocblas_half*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasBfdotBatched_64(hipblasHandle_t              handle,
                                       int64_t                      n,
                                       const hipblasBfloat16* const x[],
                                       int64_t                      incx,
                                       const hipblasBfloat16* const y[],
                                       int64_t                      incy,
                                       int64_t                      batchCount,
                                       hipblasBfloat16*             result)
try
{
    return hipblasConvertStatus(rocblas_bfdot_batched_64((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_bfloat16* const*)x,
                                                         incx,
                                                         (rocblas_bfloat16* const*)y,
                                                         incy,
                                                         batchCount,
                                                         (rocblas_bfloat16*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSdotBatched_64(hipblasHandle_t    handle,
                                      int64_t            n,
                                      const float* const x[],
                                      int64_t            incx,
                                      const float* const y[],
                                      int64_t            incy,
                                      int64_t            batchCount,
                                      float*             result)
try
{
    return hipblasConvertStatus(
        rocblas_sdot_batched_64((rocblas_handle)handle, n, x, incx, y, incy, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDdotBatched_64(hipblasHandle_t     handle,
                                      int64_t             n,
                                      const double* const x[],
                                      int64_t             incx,
                                      const double* const y[],
                                      int64_t             incy,
                                      int64_t             batchCount,
                                      double*             result)
try
{
    return hipblasConvertStatus(
        rocblas_ddot_batched_64((rocblas_handle)handle, n, x, incx, y, incy, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotcBatched_64(hipblasHandle_t         handle,
                                       int64_t                 n,
                                       const hipComplex* const x[],
                                       int64_t                 incx,
                                       const hipComplex* const y[],
                                       int64_t                 incy,
                                       int64_t                 batchCount,
                                       hipComplex*             result)
try
{
    return hipblasConvertStatus(rocblas_cdotc_batched_64((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         batchCount,
                                                         (rocblas_float_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotuBatched_64(hipblasHandle_t         handle,
                                       int64_t                 n,
                                       const hipComplex* const x[],
                                       int64_t                 incx,
                                       const hipComplex* const y[],
                                       int64_t                 incy,
                                       int64_t                 batchCount,
                                       hipComplex*             result)
try
{
    return hipblasConvertStatus(rocblas_cdotu_batched_64((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         batchCount,
                                                         (rocblas_float_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotcBatched_64(hipblasHandle_t               handle,
                                       int64_t                       n,
                                       const hipDoubleComplex* const x[],
                                       int64_t                       incx,
                                       const hipDoubleComplex* const y[],
                                       int64_t                       incy,
                                       int64_t                       batchCount,
                                       hipDoubleComplex*             result)
try
{
    return hipblasConvertStatus(rocblas_zdotc_batched_64((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         batchCount,
                                                         (rocblas_double_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotuBatched_64(hipblasHandle_t               handle,
                                       int64_t                       n,
                                       const hipDoubleComplex* const x[],
                                       int64_t                       incx,
                                       const hipDoubleComplex* const y[],
                                       int64_t                       incy,
                                       int64_t                       batchCount,
                                       hipDoubleComplex*             result)
try
{
    return hipblasConvertStatus(rocblas_zdotu_batched_64((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         batchCount,
                                                         (rocblas_double_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// dot_strided_batched
hipblasStatus_t hipblasHdotStridedBatched(hipblasHandle_t    handle,
                                          int                n,
                                          const hipblasHalf* x,
                                          int                incx,
                                          hipblasStride      stridex,
                                          const hipblasHalf* y,
                                          int                incy,
                                          hipblasStride      stridey,
                                          int                batchCount,
                                          hipblasHalf*       result)
try
{
    return hipblasConvertStatus(rocblas_hdot_strided_batched((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_half*)x,
                                                             incx,
                                                             stridex,
                                                             (rocblas_half*)y,
                                                             incy,
                                                             stridey,
                                                             batchCount,
                                                             (rocblas_half*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasBfdotStridedBatched(hipblasHandle_t        handle,
                                           int                    n,
                                           const hipblasBfloat16* x,
                                           int                    incx,
                                           hipblasStride          stridex,
                                           const hipblasBfloat16* y,
                                           int                    incy,
                                           hipblasStride          stridey,
                                           int                    batchCount,
                                           hipblasBfloat16*       result)
try
{
    return hipblasConvertStatus(rocblas_bfdot_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              (rocblas_bfloat16*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_bfloat16*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount,
                                                              (rocblas_bfloat16*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSdotStridedBatched(hipblasHandle_t handle,
                                          int             n,
                                          const float*    x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          const float*    y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          int             batchCount,
                                          float*          result)
try
{
    return hipblasConvertStatus(rocblas_sdot_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDdotStridedBatched(hipblasHandle_t handle,
                                          int             n,
                                          const double*   x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          const double*   y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          int             batchCount,
                                          double*         result)
try
{
    return hipblasConvertStatus(rocblas_ddot_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotcStridedBatched(hipblasHandle_t   handle,
                                           int               n,
                                           const hipComplex* x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const hipComplex* y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount,
                                           hipComplex*       result)
try
{
    return hipblasConvertStatus(rocblas_cdotc_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount,
                                                              (rocblas_float_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotuStridedBatched(hipblasHandle_t   handle,
                                           int               n,
                                           const hipComplex* x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const hipComplex* y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount,
                                           hipComplex*       result)
try
{
    return hipblasConvertStatus(rocblas_cdotu_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount,
                                                              (rocblas_float_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotcStridedBatched(hipblasHandle_t         handle,
                                           int                     n,
                                           const hipDoubleComplex* x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           const hipDoubleComplex* y,
                                           int                     incy,
                                           hipblasStride           stridey,
                                           int                     batchCount,
                                           hipDoubleComplex*       result)
try
{
    return hipblasConvertStatus(rocblas_zdotc_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount,
                                                              (rocblas_double_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotuStridedBatched(hipblasHandle_t         handle,
                                           int                     n,
                                           const hipDoubleComplex* x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           const hipDoubleComplex* y,
                                           int                     incy,
                                           hipblasStride           stridey,
                                           int                     batchCount,
                                           hipDoubleComplex*       result)
try
{
    return hipblasConvertStatus(rocblas_zdotu_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount,
                                                              (rocblas_double_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// dot_strided_batched_64
hipblasStatus_t hipblasHdotStridedBatched_64(hipblasHandle_t    handle,
                                             int64_t            n,
                                             const hipblasHalf* x,
                                             int64_t            incx,
                                             hipblasStride      stridex,
                                             const hipblasHalf* y,
                                             int64_t            incy,
                                             hipblasStride      stridey,
                                             int64_t            batchCount,
                                             hipblasHalf*       result)
try
{
    return hipblasConvertStatus(rocblas_hdot_strided_batched_64((rocblas_handle)handle,
                                                                n,
                                                                (rocblas_half*)x,
                                                                incx,
                                                                stridex,
                                                                (rocblas_half*)y,
                                                                incy,
                                                                stridey,
                                                                batchCount,
                                                                (rocblas_half*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasBfdotStridedBatched_64(hipblasHandle_t        handle,
                                              int64_t                n,
                                              const hipblasBfloat16* x,
                                              int64_t                incx,
                                              hipblasStride          stridex,
                                              const hipblasBfloat16* y,
                                              int64_t                incy,
                                              hipblasStride          stridey,
                                              int64_t                batchCount,
                                              hipblasBfloat16*       result)
try
{
    return hipblasConvertStatus(rocblas_bfdot_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_bfloat16*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_bfloat16*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount,
                                                                 (rocblas_bfloat16*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasSdotStridedBatched_64(hipblasHandle_t handle,
                                             int64_t         n,
                                             const float*    x,
                                             int64_t         incx,
                                             hipblasStride   stridex,
                                             const float*    y,
                                             int64_t         incy,
                                             hipblasStride   stridey,
                                             int64_t         batchCount,
                                             float*          result)
try
{
    return hipblasConvertStatus(rocblas_sdot_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDdotStridedBatched_64(hipblasHandle_t handle,
                                             int64_t         n,
                                             const double*   x,
                                             int64_t         incx,
                                             hipblasStride   stridex,
                                             const double*   y,
                                             int64_t         incy,
                                             hipblasStride   stridey,
                                             int64_t         batchCount,
                                             double*         result)
try
{
    return hipblasConvertStatus(rocblas_ddot_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotcStridedBatched_64(hipblasHandle_t   handle,
                                              int64_t           n,
                                              const hipComplex* x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount,
                                              hipComplex*       result)
try
{
    return hipblasConvertStatus(rocblas_cdotc_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount,
                                                                 (rocblas_float_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCdotuStridedBatched_64(hipblasHandle_t   handle,
                                              int64_t           n,
                                              const hipComplex* x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount,
                                              hipComplex*       result)
try
{
    return hipblasConvertStatus(rocblas_cdotu_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount,
                                                                 (rocblas_float_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotcStridedBatched_64(hipblasHandle_t         handle,
                                              int64_t                 n,
                                              const hipDoubleComplex* x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int64_t                 incy,
                                              hipblasStride           stridey,
                                              int64_t                 batchCount,
                                              hipDoubleComplex*       result)
try
{
    return hipblasConvertStatus(rocblas_zdotc_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount,
                                                                 (rocblas_double_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdotuStridedBatched_64(hipblasHandle_t         handle,
                                              int64_t                 n,
                                              const hipDoubleComplex* x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int64_t                 incy,
                                              hipblasStride           stridey,
                                              int64_t                 batchCount,
                                              hipDoubleComplex*       result)
try
{
    return hipblasConvertStatus(rocblas_zdotu_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount,
                                                                 (rocblas_double_complex*)result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// nrm2
hipblasStatus_t hipblasSnrm2(hipblasHandle_t handle, int n, const float* x, int incx, float* result)
try
{
    return hipblasConvertStatus(rocblas_snrm2((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasDnrm2(hipblasHandle_t handle, int n, const double* x, int incx, double* result)
try
{
    return hipblasConvertStatus(rocblas_dnrm2((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasScnrm2(hipblasHandle_t handle, int n, const hipComplex* x, int incx, float* result)
try
{
    return hipblasConvertStatus(
        rocblas_scnrm2((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDznrm2(
    hipblasHandle_t handle, int n, const hipDoubleComplex* x, int incx, double* result)
try
{
    return hipblasConvertStatus(
        rocblas_dznrm2((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// nrm2_64
hipblasStatus_t
    hipblasSnrm2_64(hipblasHandle_t handle, int64_t n, const float* x, int64_t incx, float* result)
try
{
    return hipblasConvertStatus(rocblas_snrm2_64((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDnrm2_64(
    hipblasHandle_t handle, int64_t n, const double* x, int64_t incx, double* result)
try
{
    return hipblasConvertStatus(rocblas_dnrm2_64((rocblas_handle)handle, n, x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasScnrm2_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* x, int64_t incx, float* result)
try
{
    return hipblasConvertStatus(
        rocblas_scnrm2_64((rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDznrm2_64(
    hipblasHandle_t handle, int64_t n, const hipDoubleComplex* x, int64_t incx, double* result)
try
{
    return hipblasConvertStatus(
        rocblas_dznrm2_64((rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// nrm2_batched
hipblasStatus_t hipblasSnrm2Batched(
    hipblasHandle_t handle, int n, const float* const x[], int incx, int batchCount, float* result)
try
{
    return hipblasConvertStatus(
        rocblas_snrm2_batched((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDnrm2Batched(hipblasHandle_t     handle,
                                    int                 n,
                                    const double* const x[],
                                    int                 incx,
                                    int                 batchCount,
                                    double*             result)
try
{
    return hipblasConvertStatus(
        rocblas_dnrm2_batched((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasScnrm2Batched(hipblasHandle_t         handle,
                                     int                     n,
                                     const hipComplex* const x[],
                                     int                     incx,
                                     int                     batchCount,
                                     float*                  result)
try
{
    return hipblasConvertStatus(rocblas_scnrm2_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDznrm2Batched(hipblasHandle_t               handle,
                                     int                           n,
                                     const hipDoubleComplex* const x[],
                                     int                           incx,
                                     int                           batchCount,
                                     double*                       result)
try
{
    return hipblasConvertStatus(rocblas_dznrm2_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// nrm2_batched_64
hipblasStatus_t hipblasSnrm2Batched_64(hipblasHandle_t    handle,
                                       int64_t            n,
                                       const float* const x[],
                                       int64_t            incx,
                                       int64_t            batchCount,
                                       float*             result)
try
{
    return hipblasConvertStatus(
        rocblas_snrm2_batched_64((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDnrm2Batched_64(hipblasHandle_t     handle,
                                       int64_t             n,
                                       const double* const x[],
                                       int64_t             incx,
                                       int64_t             batchCount,
                                       double*             result)
try
{
    return hipblasConvertStatus(
        rocblas_dnrm2_batched_64((rocblas_handle)handle, n, x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasScnrm2Batched_64(hipblasHandle_t         handle,
                                        int64_t                 n,
                                        const hipComplex* const x[],
                                        int64_t                 incx,
                                        int64_t                 batchCount,
                                        float*                  result)
try
{
    return hipblasConvertStatus(rocblas_scnrm2_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDznrm2Batched_64(hipblasHandle_t               handle,
                                        int64_t                       n,
                                        const hipDoubleComplex* const x[],
                                        int64_t                       incx,
                                        int64_t                       batchCount,
                                        double*                       result)
try
{
    return hipblasConvertStatus(rocblas_dznrm2_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex* const*)x, incx, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// nrm2_strided_batched
hipblasStatus_t hipblasSnrm2StridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const float*    x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount,
                                           float*          result)
try
{
    return hipblasConvertStatus(rocblas_snrm2_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDnrm2StridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const double*   x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount,
                                           double*         result)
try
{
    return hipblasConvertStatus(rocblas_dnrm2_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasScnrm2StridedBatched(hipblasHandle_t   handle,
                                            int               n,
                                            const hipComplex* x,
                                            int               incx,
                                            hipblasStride     stridex,
                                            int               batchCount,
                                            float*            result)
try
{
    return hipblasConvertStatus(rocblas_scnrm2_strided_batched(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDznrm2StridedBatched(hipblasHandle_t         handle,
                                            int                     n,
                                            const hipDoubleComplex* x,
                                            int                     incx,
                                            hipblasStride           stridex,
                                            int                     batchCount,
                                            double*                 result)
try
{
    return hipblasConvertStatus(rocblas_dznrm2_strided_batched(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// nrm2_strided_batched_64
hipblasStatus_t hipblasSnrm2StridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const float*    x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount,
                                              float*          result)
try
{
    return hipblasConvertStatus(rocblas_snrm2_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDnrm2StridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const double*   x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount,
                                              double*         result)
try
{
    return hipblasConvertStatus(rocblas_dnrm2_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasScnrm2StridedBatched_64(hipblasHandle_t   handle,
                                               int64_t           n,
                                               const hipComplex* x,
                                               int64_t           incx,
                                               hipblasStride     stridex,
                                               int64_t           batchCount,
                                               float*            result)
try
{
    return hipblasConvertStatus(rocblas_scnrm2_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_float_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDznrm2StridedBatched_64(hipblasHandle_t         handle,
                                               int64_t                 n,
                                               const hipDoubleComplex* x,
                                               int64_t                 incx,
                                               hipblasStride           stridex,
                                               int64_t                 batchCount,
                                               double*                 result)
try
{
    return hipblasConvertStatus(rocblas_dznrm2_strided_batched_64(
        (rocblas_handle)handle, n, (rocblas_double_complex*)x, incx, stridex, batchCount, result));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rot
hipblasStatus_t hipblasSrot(hipblasHandle_t handle,
                            int             n,
                            float*          x,
                            int             incx,
                            float*          y,
                            int             incy,
                            const float*    c,
                            const float*    s)
try
{
    return hipblasConvertStatus(rocblas_srot((rocblas_handle)handle, n, x, incx, y, incy, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrot(hipblasHandle_t handle,
                            int             n,
                            double*         x,
                            int             incx,
                            double*         y,
                            int             incy,
                            const double*   c,
                            const double*   s)
try
{
    return hipblasConvertStatus(rocblas_drot((rocblas_handle)handle, n, x, incx, y, incy, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrot(hipblasHandle_t   handle,
                            int               n,
                            hipComplex*       x,
                            int               incx,
                            hipComplex*       y,
                            int               incy,
                            const float*      c,
                            const hipComplex* s)
try
{
    return hipblasConvertStatus(rocblas_crot((rocblas_handle)handle,
                                             n,
                                             (rocblas_float_complex*)x,
                                             incx,
                                             (rocblas_float_complex*)y,
                                             incy,
                                             c,
                                             (rocblas_float_complex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsrot(hipblasHandle_t handle,
                             int             n,
                             hipComplex*     x,
                             int             incx,
                             hipComplex*     y,
                             int             incy,
                             const float*    c,
                             const float*    s)
try
{
    return hipblasConvertStatus(rocblas_csrot((rocblas_handle)handle,
                                              n,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)y,
                                              incy,
                                              c,
                                              s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrot(hipblasHandle_t         handle,
                            int                     n,
                            hipDoubleComplex*       x,
                            int                     incx,
                            hipDoubleComplex*       y,
                            int                     incy,
                            const double*           c,
                            const hipDoubleComplex* s)
try
{
    return hipblasConvertStatus(rocblas_zrot((rocblas_handle)handle,
                                             n,
                                             (rocblas_double_complex*)x,
                                             incx,
                                             (rocblas_double_complex*)y,
                                             incy,
                                             c,
                                             (rocblas_double_complex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdrot(hipblasHandle_t   handle,
                             int               n,
                             hipDoubleComplex* x,
                             int               incx,
                             hipDoubleComplex* y,
                             int               incy,
                             const double*     c,
                             const double*     s)
try
{
    return hipblasConvertStatus(rocblas_zdrot((rocblas_handle)handle,
                                              n,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)y,
                                              incy,
                                              c,
                                              s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rot_64
hipblasStatus_t hipblasSrot_64(hipblasHandle_t handle,
                               int64_t         n,
                               float*          x,
                               int64_t         incx,
                               float*          y,
                               int64_t         incy,
                               const float*    c,
                               const float*    s)
try
{
    return hipblasConvertStatus(rocblas_srot_64((rocblas_handle)handle, n, x, incx, y, incy, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrot_64(hipblasHandle_t handle,
                               int64_t         n,
                               double*         x,
                               int64_t         incx,
                               double*         y,
                               int64_t         incy,
                               const double*   c,
                               const double*   s)
try
{
    return hipblasConvertStatus(rocblas_drot_64((rocblas_handle)handle, n, x, incx, y, incy, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrot_64(hipblasHandle_t   handle,
                               int64_t           n,
                               hipComplex*       x,
                               int64_t           incx,
                               hipComplex*       y,
                               int64_t           incy,
                               const float*      c,
                               const hipComplex* s)
try
{
    return hipblasConvertStatus(rocblas_crot_64((rocblas_handle)handle,
                                                n,
                                                (rocblas_float_complex*)x,
                                                incx,
                                                (rocblas_float_complex*)y,
                                                incy,
                                                c,
                                                (rocblas_float_complex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsrot_64(hipblasHandle_t handle,
                                int64_t         n,
                                hipComplex*     x,
                                int64_t         incx,
                                hipComplex*     y,
                                int64_t         incy,
                                const float*    c,
                                const float*    s)
try
{
    return hipblasConvertStatus(rocblas_csrot_64((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)y,
                                                 incy,
                                                 c,
                                                 s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrot_64(hipblasHandle_t         handle,
                               int64_t                 n,
                               hipDoubleComplex*       x,
                               int64_t                 incx,
                               hipDoubleComplex*       y,
                               int64_t                 incy,
                               const double*           c,
                               const hipDoubleComplex* s)
try
{
    return hipblasConvertStatus(rocblas_zrot_64((rocblas_handle)handle,
                                                n,
                                                (rocblas_double_complex*)x,
                                                incx,
                                                (rocblas_double_complex*)y,
                                                incy,
                                                c,
                                                (rocblas_double_complex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdrot_64(hipblasHandle_t   handle,
                                int64_t           n,
                                hipDoubleComplex* x,
                                int64_t           incx,
                                hipDoubleComplex* y,
                                int64_t           incy,
                                const double*     c,
                                const double*     s)
try
{
    return hipblasConvertStatus(rocblas_zdrot_64((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)y,
                                                 incy,
                                                 c,
                                                 s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rot_batched
hipblasStatus_t hipblasSrotBatched(hipblasHandle_t handle,
                                   int             n,
                                   float* const    x[],
                                   int             incx,
                                   float* const    y[],
                                   int             incy,
                                   const float*    c,
                                   const float*    s,
                                   int             batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_srot_batched((rocblas_handle)handle, n, x, incx, y, incy, c, s, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotBatched(hipblasHandle_t handle,
                                   int             n,
                                   double* const   x[],
                                   int             incx,
                                   double* const   y[],
                                   int             incy,
                                   const double*   c,
                                   const double*   s,
                                   int             batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_drot_batched((rocblas_handle)handle, n, x, incx, y, incy, c, s, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrotBatched(hipblasHandle_t   handle,
                                   int               n,
                                   hipComplex* const x[],
                                   int               incx,
                                   hipComplex* const y[],
                                   int               incy,
                                   const float*      c,
                                   const hipComplex* s,
                                   int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_crot_batched((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_float_complex**)x,
                                                     incx,
                                                     (rocblas_float_complex**)y,
                                                     incy,
                                                     c,
                                                     (rocblas_float_complex*)s,
                                                     batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsrotBatched(hipblasHandle_t   handle,
                                    int               n,
                                    hipComplex* const x[],
                                    int               incx,
                                    hipComplex* const y[],
                                    int               incy,
                                    const float*      c,
                                    const float*      s,
                                    int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_csrot_batched((rocblas_handle)handle,
                                                      n,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      (rocblas_float_complex**)y,
                                                      incy,
                                                      c,
                                                      s,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrotBatched(hipblasHandle_t         handle,
                                   int                     n,
                                   hipDoubleComplex* const x[],
                                   int                     incx,
                                   hipDoubleComplex* const y[],
                                   int                     incy,
                                   const double*           c,
                                   const hipDoubleComplex* s,
                                   int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zrot_batched((rocblas_handle)handle,
                                                     n,
                                                     (rocblas_double_complex**)x,
                                                     incx,
                                                     (rocblas_double_complex**)y,
                                                     incy,
                                                     c,
                                                     (rocblas_double_complex*)s,
                                                     batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdrotBatched(hipblasHandle_t         handle,
                                    int                     n,
                                    hipDoubleComplex* const x[],
                                    int                     incx,
                                    hipDoubleComplex* const y[],
                                    int                     incy,
                                    const double*           c,
                                    const double*           s,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zdrot_batched((rocblas_handle)handle,
                                                      n,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      (rocblas_double_complex**)y,
                                                      incy,
                                                      c,
                                                      s,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rot_batched_64
hipblasStatus_t hipblasSrotBatched_64(hipblasHandle_t handle,
                                      int64_t         n,
                                      float* const    x[],
                                      int64_t         incx,
                                      float* const    y[],
                                      int64_t         incy,
                                      const float*    c,
                                      const float*    s,
                                      int64_t         batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_srot_batched_64((rocblas_handle)handle, n, x, incx, y, incy, c, s, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotBatched_64(hipblasHandle_t handle,
                                      int64_t         n,
                                      double* const   x[],
                                      int64_t         incx,
                                      double* const   y[],
                                      int64_t         incy,
                                      const double*   c,
                                      const double*   s,
                                      int64_t         batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_drot_batched_64((rocblas_handle)handle, n, x, incx, y, incy, c, s, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrotBatched_64(hipblasHandle_t   handle,
                                      int64_t           n,
                                      hipComplex* const x[],
                                      int64_t           incx,
                                      hipComplex* const y[],
                                      int64_t           incy,
                                      const float*      c,
                                      const hipComplex* s,
                                      int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_crot_batched_64((rocblas_handle)handle,
                                                        n,
                                                        (rocblas_float_complex**)x,
                                                        incx,
                                                        (rocblas_float_complex**)y,
                                                        incy,
                                                        c,
                                                        (rocblas_float_complex*)s,
                                                        batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsrotBatched_64(hipblasHandle_t   handle,
                                       int64_t           n,
                                       hipComplex* const x[],
                                       int64_t           incx,
                                       hipComplex* const y[],
                                       int64_t           incy,
                                       const float*      c,
                                       const float*      s,
                                       int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_csrot_batched_64((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         c,
                                                         s,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrotBatched_64(hipblasHandle_t         handle,
                                      int64_t                 n,
                                      hipDoubleComplex* const x[],
                                      int64_t                 incx,
                                      hipDoubleComplex* const y[],
                                      int64_t                 incy,
                                      const double*           c,
                                      const hipDoubleComplex* s,
                                      int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zrot_batched_64((rocblas_handle)handle,
                                                        n,
                                                        (rocblas_double_complex**)x,
                                                        incx,
                                                        (rocblas_double_complex**)y,
                                                        incy,
                                                        c,
                                                        (rocblas_double_complex*)s,
                                                        batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdrotBatched_64(hipblasHandle_t         handle,
                                       int64_t                 n,
                                       hipDoubleComplex* const x[],
                                       int64_t                 incx,
                                       hipDoubleComplex* const y[],
                                       int64_t                 incy,
                                       const double*           c,
                                       const double*           s,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zdrot_batched_64((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         c,
                                                         s,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rot_strided_batched
hipblasStatus_t hipblasSrotStridedBatched(hipblasHandle_t handle,
                                          int             n,
                                          float*          x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          float*          y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          const float*    c,
                                          const float*    s,
                                          int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_srot_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, c, s, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotStridedBatched(hipblasHandle_t handle,
                                          int             n,
                                          double*         x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          double*         y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          const double*   c,
                                          const double*   s,
                                          int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_drot_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, c, s, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrotStridedBatched(hipblasHandle_t   handle,
                                          int               n,
                                          hipComplex*       x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          hipComplex*       y,
                                          int               incy,
                                          hipblasStride     stridey,
                                          const float*      c,
                                          const hipComplex* s,
                                          int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_crot_strided_batched((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_float_complex*)x,
                                                             incx,
                                                             stridex,
                                                             (rocblas_float_complex*)y,
                                                             incy,
                                                             stridey,
                                                             c,
                                                             (rocblas_float_complex*)s,
                                                             batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsrotStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           hipComplex*     x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           hipComplex*     y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           const float*    c,
                                           const float*    s,
                                           int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_csrot_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stridey,
                                                              c,
                                                              s,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrotStridedBatched(hipblasHandle_t         handle,
                                          int                     n,
                                          hipDoubleComplex*       x,
                                          int                     incx,
                                          hipblasStride           stridex,
                                          hipDoubleComplex*       y,
                                          int                     incy,
                                          hipblasStride           stridey,
                                          const double*           c,
                                          const hipDoubleComplex* s,
                                          int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zrot_strided_batched((rocblas_handle)handle,
                                                             n,
                                                             (rocblas_double_complex*)x,
                                                             incx,
                                                             stridex,
                                                             (rocblas_double_complex*)y,
                                                             incy,
                                                             stridey,
                                                             c,
                                                             (rocblas_double_complex*)s,
                                                             batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdrotStridedBatched(hipblasHandle_t   handle,
                                           int               n,
                                           hipDoubleComplex* x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           hipDoubleComplex* y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           const double*     c,
                                           const double*     s,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_zdrot_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stridey,
                                                              c,
                                                              s,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rot_strided_batched_64
hipblasStatus_t hipblasSrotStridedBatched_64(hipblasHandle_t handle,
                                             int64_t         n,
                                             float*          x,
                                             int64_t         incx,
                                             hipblasStride   stridex,
                                             float*          y,
                                             int64_t         incy,
                                             hipblasStride   stridey,
                                             const float*    c,
                                             const float*    s,
                                             int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_srot_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, c, s, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotStridedBatched_64(hipblasHandle_t handle,
                                             int64_t         n,
                                             double*         x,
                                             int64_t         incx,
                                             hipblasStride   stridex,
                                             double*         y,
                                             int64_t         incy,
                                             hipblasStride   stridey,
                                             const double*   c,
                                             const double*   s,
                                             int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_drot_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, c, s, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrotStridedBatched_64(hipblasHandle_t   handle,
                                             int64_t           n,
                                             hipComplex*       x,
                                             int64_t           incx,
                                             hipblasStride     stridex,
                                             hipComplex*       y,
                                             int64_t           incy,
                                             hipblasStride     stridey,
                                             const float*      c,
                                             const hipComplex* s,
                                             int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_crot_strided_batched_64((rocblas_handle)handle,
                                                                n,
                                                                (rocblas_float_complex*)x,
                                                                incx,
                                                                stridex,
                                                                (rocblas_float_complex*)y,
                                                                incy,
                                                                stridey,
                                                                c,
                                                                (rocblas_float_complex*)s,
                                                                batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsrotStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              hipComplex*     x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              hipComplex*     y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              const float*    c,
                                              const float*    s,
                                              int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_csrot_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 c,
                                                                 s,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrotStridedBatched_64(hipblasHandle_t         handle,
                                             int64_t                 n,
                                             hipDoubleComplex*       x,
                                             int64_t                 incx,
                                             hipblasStride           stridex,
                                             hipDoubleComplex*       y,
                                             int64_t                 incy,
                                             hipblasStride           stridey,
                                             const double*           c,
                                             const hipDoubleComplex* s,
                                             int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zrot_strided_batched_64((rocblas_handle)handle,
                                                                n,
                                                                (rocblas_double_complex*)x,
                                                                incx,
                                                                stridex,
                                                                (rocblas_double_complex*)y,
                                                                incy,
                                                                stridey,
                                                                c,
                                                                (rocblas_double_complex*)s,
                                                                batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdrotStridedBatched_64(hipblasHandle_t   handle,
                                              int64_t           n,
                                              hipDoubleComplex* x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              hipDoubleComplex* y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              const double*     c,
                                              const double*     s,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zdrot_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 c,
                                                                 s,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotg
hipblasStatus_t hipblasSrotg(hipblasHandle_t handle, float* a, float* b, float* c, float* s)
try
{
    return hipblasConvertStatus(rocblas_srotg((rocblas_handle)handle, a, b, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotg(hipblasHandle_t handle, double* a, double* b, double* c, double* s)
try
{
    return hipblasConvertStatus(rocblas_drotg((rocblas_handle)handle, a, b, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasCrotg(hipblasHandle_t handle, hipComplex* a, hipComplex* b, float* c, hipComplex* s)
try
{
    return hipblasConvertStatus(rocblas_crotg((rocblas_handle)handle,
                                              (rocblas_float_complex*)a,
                                              (rocblas_float_complex*)b,
                                              c,
                                              (rocblas_float_complex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrotg(hipblasHandle_t   handle,
                             hipDoubleComplex* a,
                             hipDoubleComplex* b,
                             double*           c,
                             hipDoubleComplex* s)
try
{
    return hipblasConvertStatus(rocblas_zrotg((rocblas_handle)handle,
                                              (rocblas_double_complex*)a,
                                              (rocblas_double_complex*)b,
                                              c,
                                              (rocblas_double_complex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotg_64
hipblasStatus_t hipblasSrotg_64(hipblasHandle_t handle, float* a, float* b, float* c, float* s)
try
{
    return hipblasConvertStatus(rocblas_srotg_64((rocblas_handle)handle, a, b, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotg_64(hipblasHandle_t handle, double* a, double* b, double* c, double* s)
try
{
    return hipblasConvertStatus(rocblas_drotg_64((rocblas_handle)handle, a, b, c, s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasCrotg_64(hipblasHandle_t handle, hipComplex* a, hipComplex* b, float* c, hipComplex* s)
try
{
    return hipblasConvertStatus(rocblas_crotg_64((rocblas_handle)handle,
                                                 (rocblas_float_complex*)a,
                                                 (rocblas_float_complex*)b,
                                                 c,
                                                 (rocblas_float_complex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrotg_64(hipblasHandle_t   handle,
                                hipDoubleComplex* a,
                                hipDoubleComplex* b,
                                double*           c,
                                hipDoubleComplex* s)
try
{
    return hipblasConvertStatus(rocblas_zrotg_64((rocblas_handle)handle,
                                                 (rocblas_double_complex*)a,
                                                 (rocblas_double_complex*)b,
                                                 c,
                                                 (rocblas_double_complex*)s));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotg_batched
hipblasStatus_t hipblasSrotgBatched(hipblasHandle_t handle,
                                    float* const    a[],
                                    float* const    b[],
                                    float* const    c[],
                                    float* const    s[],
                                    int             batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_srotg_batched((rocblas_handle)handle, a, b, c, s, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotgBatched(hipblasHandle_t handle,
                                    double* const   a[],
                                    double* const   b[],
                                    double* const   c[],
                                    double* const   s[],
                                    int             batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_drotg_batched((rocblas_handle)handle, a, b, c, s, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrotgBatched(hipblasHandle_t   handle,
                                    hipComplex* const a[],
                                    hipComplex* const b[],
                                    float* const      c[],
                                    hipComplex* const s[],
                                    int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_crotg_batched((rocblas_handle)handle,
                                                      (rocblas_float_complex**)a,
                                                      (rocblas_float_complex**)b,
                                                      c,
                                                      (rocblas_float_complex**)s,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrotgBatched(hipblasHandle_t         handle,
                                    hipDoubleComplex* const a[],
                                    hipDoubleComplex* const b[],
                                    double* const           c[],
                                    hipDoubleComplex* const s[],
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zrotg_batched((rocblas_handle)handle,
                                                      (rocblas_double_complex**)a,
                                                      (rocblas_double_complex**)b,
                                                      c,
                                                      (rocblas_double_complex**)s,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotg_batched_64
hipblasStatus_t hipblasSrotgBatched_64(hipblasHandle_t handle,
                                       float* const    a[],
                                       float* const    b[],
                                       float* const    c[],
                                       float* const    s[],
                                       int64_t         batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_srotg_batched_64((rocblas_handle)handle, a, b, c, s, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotgBatched_64(hipblasHandle_t handle,
                                       double* const   a[],
                                       double* const   b[],
                                       double* const   c[],
                                       double* const   s[],
                                       int64_t         batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_drotg_batched_64((rocblas_handle)handle, a, b, c, s, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrotgBatched_64(hipblasHandle_t   handle,
                                       hipComplex* const a[],
                                       hipComplex* const b[],
                                       float* const      c[],
                                       hipComplex* const s[],
                                       int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_crotg_batched_64((rocblas_handle)handle,
                                                         (rocblas_float_complex**)a,
                                                         (rocblas_float_complex**)b,
                                                         c,
                                                         (rocblas_float_complex**)s,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrotgBatched_64(hipblasHandle_t         handle,
                                       hipDoubleComplex* const a[],
                                       hipDoubleComplex* const b[],
                                       double* const           c[],
                                       hipDoubleComplex* const s[],
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zrotg_batched_64((rocblas_handle)handle,
                                                         (rocblas_double_complex**)a,
                                                         (rocblas_double_complex**)b,
                                                         c,
                                                         (rocblas_double_complex**)s,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotg_strided_batched
hipblasStatus_t hipblasSrotgStridedBatched(hipblasHandle_t handle,
                                           float*          a,
                                           hipblasStride   stride_a,
                                           float*          b,
                                           hipblasStride   stride_b,
                                           float*          c,
                                           hipblasStride   stride_c,
                                           float*          s,
                                           hipblasStride   stride_s,
                                           int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_srotg_strided_batched(
        (rocblas_handle)handle, a, stride_a, b, stride_b, c, stride_c, s, stride_s, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotgStridedBatched(hipblasHandle_t handle,
                                           double*         a,
                                           hipblasStride   stride_a,
                                           double*         b,
                                           hipblasStride   stride_b,
                                           double*         c,
                                           hipblasStride   stride_c,
                                           double*         s,
                                           hipblasStride   stride_s,
                                           int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_drotg_strided_batched(
        (rocblas_handle)handle, a, stride_a, b, stride_b, c, stride_c, s, stride_s, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrotgStridedBatched(hipblasHandle_t handle,
                                           hipComplex*     a,
                                           hipblasStride   stride_a,
                                           hipComplex*     b,
                                           hipblasStride   stride_b,
                                           float*          c,
                                           hipblasStride   stride_c,
                                           hipComplex*     s,
                                           hipblasStride   stride_s,
                                           int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_crotg_strided_batched((rocblas_handle)handle,
                                                              (rocblas_float_complex*)a,
                                                              stride_a,
                                                              (rocblas_float_complex*)b,
                                                              stride_b,
                                                              c,
                                                              stride_c,
                                                              (rocblas_float_complex*)s,
                                                              stride_s,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrotgStridedBatched(hipblasHandle_t   handle,
                                           hipDoubleComplex* a,
                                           hipblasStride     stride_a,
                                           hipDoubleComplex* b,
                                           hipblasStride     stride_b,
                                           double*           c,
                                           hipblasStride     stride_c,
                                           hipDoubleComplex* s,
                                           hipblasStride     stride_s,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_zrotg_strided_batched((rocblas_handle)handle,
                                                              (rocblas_double_complex*)a,
                                                              stride_a,
                                                              (rocblas_double_complex*)b,
                                                              stride_b,
                                                              c,
                                                              stride_c,
                                                              (rocblas_double_complex*)s,
                                                              stride_s,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotg_strided_batched_64
hipblasStatus_t hipblasSrotgStridedBatched_64(hipblasHandle_t handle,
                                              float*          a,
                                              hipblasStride   stride_a,
                                              float*          b,
                                              hipblasStride   stride_b,
                                              float*          c,
                                              hipblasStride   stride_c,
                                              float*          s,
                                              hipblasStride   stride_s,
                                              int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_srotg_strided_batched_64(
        (rocblas_handle)handle, a, stride_a, b, stride_b, c, stride_c, s, stride_s, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotgStridedBatched_64(hipblasHandle_t handle,
                                              double*         a,
                                              hipblasStride   stride_a,
                                              double*         b,
                                              hipblasStride   stride_b,
                                              double*         c,
                                              hipblasStride   stride_c,
                                              double*         s,
                                              hipblasStride   stride_s,
                                              int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_drotg_strided_batched_64(
        (rocblas_handle)handle, a, stride_a, b, stride_b, c, stride_c, s, stride_s, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCrotgStridedBatched_64(hipblasHandle_t handle,
                                              hipComplex*     a,
                                              hipblasStride   stride_a,
                                              hipComplex*     b,
                                              hipblasStride   stride_b,
                                              float*          c,
                                              hipblasStride   stride_c,
                                              hipComplex*     s,
                                              hipblasStride   stride_s,
                                              int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_crotg_strided_batched_64((rocblas_handle)handle,
                                                                 (rocblas_float_complex*)a,
                                                                 stride_a,
                                                                 (rocblas_float_complex*)b,
                                                                 stride_b,
                                                                 c,
                                                                 stride_c,
                                                                 (rocblas_float_complex*)s,
                                                                 stride_s,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZrotgStridedBatched_64(hipblasHandle_t   handle,
                                              hipDoubleComplex* a,
                                              hipblasStride     stride_a,
                                              hipDoubleComplex* b,
                                              hipblasStride     stride_b,
                                              double*           c,
                                              hipblasStride     stride_c,
                                              hipDoubleComplex* s,
                                              hipblasStride     stride_s,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zrotg_strided_batched_64((rocblas_handle)handle,
                                                                 (rocblas_double_complex*)a,
                                                                 stride_a,
                                                                 (rocblas_double_complex*)b,
                                                                 stride_b,
                                                                 c,
                                                                 stride_c,
                                                                 (rocblas_double_complex*)s,
                                                                 stride_s,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotm
hipblasStatus_t hipblasSrotm(
    hipblasHandle_t handle, int n, float* x, int incx, float* y, int incy, const float* param)
try
{
    return hipblasConvertStatus(rocblas_srotm((rocblas_handle)handle, n, x, incx, y, incy, param));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotm(
    hipblasHandle_t handle, int n, double* x, int incx, double* y, int incy, const double* param)
try
{
    return hipblasConvertStatus(rocblas_drotm((rocblas_handle)handle, n, x, incx, y, incy, param));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotm_64
hipblasStatus_t hipblasSrotm_64(hipblasHandle_t handle,
                                int64_t         n,
                                float*          x,
                                int64_t         incx,
                                float*          y,
                                int64_t         incy,
                                const float*    param)
try
{
    return hipblasConvertStatus(
        rocblas_srotm_64((rocblas_handle)handle, n, x, incx, y, incy, param));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotm_64(hipblasHandle_t handle,
                                int64_t         n,
                                double*         x,
                                int64_t         incx,
                                double*         y,
                                int64_t         incy,
                                const double*   param)
try
{
    return hipblasConvertStatus(
        rocblas_drotm_64((rocblas_handle)handle, n, x, incx, y, incy, param));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotm_batched
hipblasStatus_t hipblasSrotmBatched(hipblasHandle_t    handle,
                                    int                n,
                                    float* const       x[],
                                    int                incx,
                                    float* const       y[],
                                    int                incy,
                                    const float* const param[],
                                    int                batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_srotm_batched((rocblas_handle)handle, n, x, incx, y, incy, param, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotmBatched(hipblasHandle_t     handle,
                                    int                 n,
                                    double* const       x[],
                                    int                 incx,
                                    double* const       y[],
                                    int                 incy,
                                    const double* const param[],
                                    int                 batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_drotm_batched((rocblas_handle)handle, n, x, incx, y, incy, param, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotm_batched_64
hipblasStatus_t hipblasSrotmBatched_64(hipblasHandle_t    handle,
                                       int64_t            n,
                                       float* const       x[],
                                       int64_t            incx,
                                       float* const       y[],
                                       int64_t            incy,
                                       const float* const param[],
                                       int64_t            batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_srotm_batched_64((rocblas_handle)handle, n, x, incx, y, incy, param, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotmBatched_64(hipblasHandle_t     handle,
                                       int64_t             n,
                                       double* const       x[],
                                       int64_t             incx,
                                       double* const       y[],
                                       int64_t             incy,
                                       const double* const param[],
                                       int64_t             batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_drotm_batched_64((rocblas_handle)handle, n, x, incx, y, incy, param, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotm_strided_batched
hipblasStatus_t hipblasSrotmStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           float*          x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           float*          y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           const float*    param,
                                           hipblasStride   strideParam,
                                           int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_srotm_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              y,
                                                              incy,
                                                              stridey,
                                                              param,
                                                              strideParam,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotmStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           double*         x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           double*         y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           const double*   param,
                                           hipblasStride   strideParam,
                                           int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_drotm_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              y,
                                                              incy,
                                                              stridey,
                                                              param,
                                                              strideParam,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotm_strided_batched_64
hipblasStatus_t hipblasSrotmStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              float*          x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              float*          y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              const float*    param,
                                              hipblasStride   strideParam,
                                              int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_srotm_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 y,
                                                                 incy,
                                                                 stridey,
                                                                 param,
                                                                 strideParam,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotmStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              double*         x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              double*         y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              const double*   param,
                                              hipblasStride   strideParam,
                                              int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_drotm_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 y,
                                                                 incy,
                                                                 stridey,
                                                                 param,
                                                                 strideParam,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotmg
hipblasStatus_t hipblasSrotmg(
    hipblasHandle_t handle, float* d1, float* d2, float* x1, const float* y1, float* param)
try
{
    return hipblasConvertStatus(rocblas_srotmg((rocblas_handle)handle, d1, d2, x1, y1, param));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotmg(
    hipblasHandle_t handle, double* d1, double* d2, double* x1, const double* y1, double* param)
try
{
    return hipblasConvertStatus(rocblas_drotmg((rocblas_handle)handle, d1, d2, x1, y1, param));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotmg_64
hipblasStatus_t hipblasSrotmg_64(
    hipblasHandle_t handle, float* d1, float* d2, float* x1, const float* y1, float* param)
try
{
    return hipblasConvertStatus(rocblas_srotmg_64((rocblas_handle)handle, d1, d2, x1, y1, param));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotmg_64(
    hipblasHandle_t handle, double* d1, double* d2, double* x1, const double* y1, double* param)
try
{
    return hipblasConvertStatus(rocblas_drotmg_64((rocblas_handle)handle, d1, d2, x1, y1, param));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotmg_batched
hipblasStatus_t hipblasSrotmgBatched(hipblasHandle_t    handle,
                                     float* const       d1[],
                                     float* const       d2[],
                                     float* const       x1[],
                                     const float* const y1[],
                                     float* const       param[],
                                     int                batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_srotmg_batched((rocblas_handle)handle, d1, d2, x1, y1, param, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotmgBatched(hipblasHandle_t     handle,
                                     double* const       d1[],
                                     double* const       d2[],
                                     double* const       x1[],
                                     const double* const y1[],
                                     double* const       param[],
                                     int                 batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_drotmg_batched((rocblas_handle)handle, d1, d2, x1, y1, param, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotmg_batched_64
hipblasStatus_t hipblasSrotmgBatched_64(hipblasHandle_t    handle,
                                        float* const       d1[],
                                        float* const       d2[],
                                        float* const       x1[],
                                        const float* const y1[],
                                        float* const       param[],
                                        int64_t            batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_srotmg_batched_64((rocblas_handle)handle, d1, d2, x1, y1, param, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotmgBatched_64(hipblasHandle_t     handle,
                                        double* const       d1[],
                                        double* const       d2[],
                                        double* const       x1[],
                                        const double* const y1[],
                                        double* const       param[],
                                        int64_t             batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_drotmg_batched_64((rocblas_handle)handle, d1, d2, x1, y1, param, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotmg_strided_batched
hipblasStatus_t hipblasSrotmgStridedBatched(hipblasHandle_t handle,
                                            float*          d1,
                                            hipblasStride   stride_d1,
                                            float*          d2,
                                            hipblasStride   stride_d2,
                                            float*          x1,
                                            hipblasStride   stride_x1,
                                            const float*    y1,
                                            hipblasStride   stride_y1,
                                            float*          param,
                                            hipblasStride   strideParam,
                                            int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_srotmg_strided_batched((rocblas_handle)handle,
                                                               d1,
                                                               stride_d1,
                                                               d2,
                                                               stride_d2,
                                                               x1,
                                                               stride_x1,
                                                               y1,
                                                               stride_y1,
                                                               param,
                                                               strideParam,
                                                               batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotmgStridedBatched(hipblasHandle_t handle,
                                            double*         d1,
                                            hipblasStride   stride_d1,
                                            double*         d2,
                                            hipblasStride   stride_d2,
                                            double*         x1,
                                            hipblasStride   stride_x1,
                                            const double*   y1,
                                            hipblasStride   stride_y1,
                                            double*         param,
                                            hipblasStride   strideParam,
                                            int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_drotmg_strided_batched((rocblas_handle)handle,
                                                               d1,
                                                               stride_d1,
                                                               d2,
                                                               stride_d2,
                                                               x1,
                                                               stride_x1,
                                                               y1,
                                                               stride_y1,
                                                               param,
                                                               strideParam,
                                                               batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// rotmg_strided_batched_64
hipblasStatus_t hipblasSrotmgStridedBatched_64(hipblasHandle_t handle,
                                               float*          d1,
                                               hipblasStride   stride_d1,
                                               float*          d2,
                                               hipblasStride   stride_d2,
                                               float*          x1,
                                               hipblasStride   stride_x1,
                                               const float*    y1,
                                               hipblasStride   stride_y1,
                                               float*          param,
                                               hipblasStride   strideParam,
                                               int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_srotmg_strided_batched_64((rocblas_handle)handle,
                                                                  d1,
                                                                  stride_d1,
                                                                  d2,
                                                                  stride_d2,
                                                                  x1,
                                                                  stride_x1,
                                                                  y1,
                                                                  stride_y1,
                                                                  param,
                                                                  strideParam,
                                                                  batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDrotmgStridedBatched_64(hipblasHandle_t handle,
                                               double*         d1,
                                               hipblasStride   stride_d1,
                                               double*         d2,
                                               hipblasStride   stride_d2,
                                               double*         x1,
                                               hipblasStride   stride_x1,
                                               const double*   y1,
                                               hipblasStride   stride_y1,
                                               double*         param,
                                               hipblasStride   strideParam,
                                               int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_drotmg_strided_batched_64((rocblas_handle)handle,
                                                                  d1,
                                                                  stride_d1,
                                                                  d2,
                                                                  stride_d2,
                                                                  x1,
                                                                  stride_x1,
                                                                  y1,
                                                                  stride_y1,
                                                                  param,
                                                                  strideParam,
                                                                  batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// scal
hipblasStatus_t hipblasSscal(hipblasHandle_t handle, int n, const float* alpha, float* x, int incx)
try
{
    return hipblasConvertStatus(rocblas_sscal((rocblas_handle)handle, n, alpha, x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasDscal(hipblasHandle_t handle, int n, const double* alpha, double* x, int incx)
try
{
    return hipblasConvertStatus(rocblas_dscal((rocblas_handle)handle, n, alpha, x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasCscal(hipblasHandle_t handle, int n, const hipComplex* alpha, hipComplex* x, int incx)
try
{
    return hipblasConvertStatus(rocblas_cscal(
        (rocblas_handle)handle, n, (rocblas_float_complex*)alpha, (rocblas_float_complex*)x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasCsscal(hipblasHandle_t handle, int n, const float* alpha, hipComplex* x, int incx)
try
{
    return hipblasConvertStatus(
        rocblas_csscal((rocblas_handle)handle, n, alpha, (rocblas_float_complex*)x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZscal(
    hipblasHandle_t handle, int n, const hipDoubleComplex* alpha, hipDoubleComplex* x, int incx)
try
{
    return hipblasConvertStatus(rocblas_zscal((rocblas_handle)handle,
                                              n,
                                              (rocblas_double_complex*)alpha,
                                              (rocblas_double_complex*)x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasZdscal(hipblasHandle_t handle, int n, const double* alpha, hipDoubleComplex* x, int incx)
try
{
    return hipblasConvertStatus(
        rocblas_zdscal((rocblas_handle)handle, n, alpha, (rocblas_double_complex*)x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// scal_64
hipblasStatus_t
    hipblasSscal_64(hipblasHandle_t handle, int64_t n, const float* alpha, float* x, int64_t incx)
try
{
    return hipblasConvertStatus(rocblas_sscal_64((rocblas_handle)handle, n, alpha, x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasDscal_64(hipblasHandle_t handle, int64_t n, const double* alpha, double* x, int64_t incx)
try
{
    return hipblasConvertStatus(rocblas_dscal_64((rocblas_handle)handle, n, alpha, x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCscal_64(
    hipblasHandle_t handle, int64_t n, const hipComplex* alpha, hipComplex* x, int64_t incx)
try
{
    return hipblasConvertStatus(rocblas_cscal_64(
        (rocblas_handle)handle, n, (rocblas_float_complex*)alpha, (rocblas_float_complex*)x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsscal_64(
    hipblasHandle_t handle, int64_t n, const float* alpha, hipComplex* x, int64_t incx)
try
{
    return hipblasConvertStatus(
        rocblas_csscal_64((rocblas_handle)handle, n, alpha, (rocblas_float_complex*)x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZscal_64(hipblasHandle_t         handle,
                                int64_t                 n,
                                const hipDoubleComplex* alpha,
                                hipDoubleComplex*       x,
                                int64_t                 incx)
try
{
    return hipblasConvertStatus(rocblas_zscal_64((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdscal_64(
    hipblasHandle_t handle, int64_t n, const double* alpha, hipDoubleComplex* x, int64_t incx)
try
{
    return hipblasConvertStatus(
        rocblas_zdscal_64((rocblas_handle)handle, n, alpha, (rocblas_double_complex*)x, incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// scal_batched
hipblasStatus_t hipblasSscalBatched(
    hipblasHandle_t handle, int n, const float* alpha, float* const x[], int incx, int batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_sscal_batched((rocblas_handle)handle, n, alpha, x, incx, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDscalBatched(
    hipblasHandle_t handle, int n, const double* alpha, double* const x[], int incx, int batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_dscal_batched((rocblas_handle)handle, n, alpha, x, incx, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCscalBatched(hipblasHandle_t   handle,
                                    int               n,
                                    const hipComplex* alpha,
                                    hipComplex* const x[],
                                    int               incx,
                                    int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_cscal_batched((rocblas_handle)handle,
                                                      n,
                                                      (rocblas_float_complex*)alpha,
                                                      (rocblas_float_complex* const*)x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZscalBatched(hipblasHandle_t         handle,
                                    int                     n,
                                    const hipDoubleComplex* alpha,
                                    hipDoubleComplex* const x[],
                                    int                     incx,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zscal_batched((rocblas_handle)handle,
                                                      n,
                                                      (rocblas_double_complex*)alpha,
                                                      (rocblas_double_complex* const*)x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsscalBatched(hipblasHandle_t   handle,
                                     int               n,
                                     const float*      alpha,
                                     hipComplex* const x[],
                                     int               incx,
                                     int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_csscal_batched(
        (rocblas_handle)handle, n, alpha, (rocblas_float_complex* const*)x, incx, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdscalBatched(hipblasHandle_t         handle,
                                     int                     n,
                                     const double*           alpha,
                                     hipDoubleComplex* const x[],
                                     int                     incx,
                                     int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zdscal_batched(
        (rocblas_handle)handle, n, alpha, (rocblas_double_complex* const*)x, incx, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// scal_batched_64
hipblasStatus_t hipblasSscalBatched_64(hipblasHandle_t handle,
                                       int64_t         n,
                                       const float*    alpha,
                                       float* const    x[],
                                       int64_t         incx,
                                       int64_t         batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_sscal_batched_64((rocblas_handle)handle, n, alpha, x, incx, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDscalBatched_64(hipblasHandle_t handle,
                                       int64_t         n,
                                       const double*   alpha,
                                       double* const   x[],
                                       int64_t         incx,
                                       int64_t         batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_dscal_batched_64((rocblas_handle)handle, n, alpha, x, incx, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCscalBatched_64(hipblasHandle_t   handle,
                                       int64_t           n,
                                       const hipComplex* alpha,
                                       hipComplex* const x[],
                                       int64_t           incx,
                                       int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_cscal_batched_64((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex* const*)x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZscalBatched_64(hipblasHandle_t         handle,
                                       int64_t                 n,
                                       const hipDoubleComplex* alpha,
                                       hipDoubleComplex* const x[],
                                       int64_t                 incx,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zscal_batched_64((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex* const*)x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsscalBatched_64(hipblasHandle_t   handle,
                                        int64_t           n,
                                        const float*      alpha,
                                        hipComplex* const x[],
                                        int64_t           incx,
                                        int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_csscal_batched_64(
        (rocblas_handle)handle, n, alpha, (rocblas_float_complex* const*)x, incx, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdscalBatched_64(hipblasHandle_t         handle,
                                        int64_t                 n,
                                        const double*           alpha,
                                        hipDoubleComplex* const x[],
                                        int64_t                 incx,
                                        int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zdscal_batched_64(
        (rocblas_handle)handle, n, alpha, (rocblas_double_complex* const*)x, incx, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// scal_strided_batched
hipblasStatus_t hipblasSscalStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const float*    alpha,
                                           float*          x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_sscal_strided_batched(
        (rocblas_handle)handle, n, alpha, x, incx, stridex, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDscalStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           const double*   alpha,
                                           double*         x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dscal_strided_batched(
        (rocblas_handle)handle, n, alpha, x, incx, stridex, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCscalStridedBatched(hipblasHandle_t   handle,
                                           int               n,
                                           const hipComplex* alpha,
                                           hipComplex*       x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_cscal_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              (rocblas_float_complex*)alpha,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZscalStridedBatched(hipblasHandle_t         handle,
                                           int                     n,
                                           const hipDoubleComplex* alpha,
                                           hipDoubleComplex*       x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zscal_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              (rocblas_double_complex*)alpha,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsscalStridedBatched(hipblasHandle_t handle,
                                            int             n,
                                            const float*    alpha,
                                            hipComplex*     x,
                                            int             incx,
                                            hipblasStride   stridex,
                                            int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_csscal_strided_batched(
        (rocblas_handle)handle, n, alpha, (rocblas_float_complex*)x, incx, stridex, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdscalStridedBatched(hipblasHandle_t   handle,
                                            int               n,
                                            const double*     alpha,
                                            hipDoubleComplex* x,
                                            int               incx,
                                            hipblasStride     stridex,
                                            int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_zdscal_strided_batched(
        (rocblas_handle)handle, n, alpha, (rocblas_double_complex*)x, incx, stridex, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// scal_strided_batched_64
hipblasStatus_t hipblasSscalStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const float*    alpha,
                                              float*          x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_sscal_strided_batched_64(
        (rocblas_handle)handle, n, alpha, x, incx, stridex, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDscalStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              const double*   alpha,
                                              double*         x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_dscal_strided_batched_64(
        (rocblas_handle)handle, n, alpha, x, incx, stridex, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCscalStridedBatched_64(hipblasHandle_t   handle,
                                              int64_t           n,
                                              const hipComplex* alpha,
                                              hipComplex*       x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_cscal_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZscalStridedBatched_64(hipblasHandle_t         handle,
                                              int64_t                 n,
                                              const hipDoubleComplex* alpha,
                                              hipDoubleComplex*       x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zscal_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsscalStridedBatched_64(hipblasHandle_t handle,
                                               int64_t         n,
                                               const float*    alpha,
                                               hipComplex*     x,
                                               int64_t         incx,
                                               hipblasStride   stridex,
                                               int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_csscal_strided_batched_64(
        (rocblas_handle)handle, n, alpha, (rocblas_float_complex*)x, incx, stridex, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZdscalStridedBatched_64(hipblasHandle_t   handle,
                                               int64_t           n,
                                               const double*     alpha,
                                               hipDoubleComplex* x,
                                               int64_t           incx,
                                               hipblasStride     stridex,
                                               int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zdscal_strided_batched_64(
        (rocblas_handle)handle, n, alpha, (rocblas_double_complex*)x, incx, stridex, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// swap
hipblasStatus_t hipblasSswap(hipblasHandle_t handle, int n, float* x, int incx, float* y, int incy)
try
{
    return hipblasConvertStatus(rocblas_sswap((rocblas_handle)handle, n, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasDswap(hipblasHandle_t handle, int n, double* x, int incx, double* y, int incy)
try
{
    return hipblasConvertStatus(rocblas_dswap((rocblas_handle)handle, n, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t
    hipblasCswap(hipblasHandle_t handle, int n, hipComplex* x, int incx, hipComplex* y, int incy)
try
{
    return hipblasConvertStatus(rocblas_cswap((rocblas_handle)handle,
                                              n,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZswap(
    hipblasHandle_t handle, int n, hipDoubleComplex* x, int incx, hipDoubleComplex* y, int incy)
try
{
    return hipblasConvertStatus(rocblas_zswap((rocblas_handle)handle,
                                              n,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// swap_64
hipblasStatus_t hipblasSswap_64(
    hipblasHandle_t handle, int64_t n, float* x, int64_t incx, float* y, int64_t incy)
try
{
    return hipblasConvertStatus(rocblas_sswap_64((rocblas_handle)handle, n, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDswap_64(
    hipblasHandle_t handle, int64_t n, double* x, int64_t incx, double* y, int64_t incy)
try
{
    return hipblasConvertStatus(rocblas_dswap_64((rocblas_handle)handle, n, x, incx, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCswap_64(
    hipblasHandle_t handle, int64_t n, hipComplex* x, int64_t incx, hipComplex* y, int64_t incy)
try
{
    return hipblasConvertStatus(rocblas_cswap_64((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZswap_64(hipblasHandle_t   handle,
                                int64_t           n,
                                hipDoubleComplex* x,
                                int64_t           incx,
                                hipDoubleComplex* y,
                                int64_t           incy)
try
{
    return hipblasConvertStatus(rocblas_zswap_64((rocblas_handle)handle,
                                                 n,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// swap_batched
hipblasStatus_t hipblasSswapBatched(hipblasHandle_t handle,
                                    int             n,
                                    float* const    x[],
                                    int             incx,
                                    float* const    y[],
                                    int             incy,
                                    int             batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_sswap_batched((rocblas_handle)handle, n, x, incx, y, incy, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDswapBatched(hipblasHandle_t handle,
                                    int             n,
                                    double* const   x[],
                                    int             incx,
                                    double* const   y[],
                                    int             incy,
                                    int             batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_dswap_batched((rocblas_handle)handle, n, x, incx, y, incy, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCswapBatched(hipblasHandle_t   handle,
                                    int               n,
                                    hipComplex* const x[],
                                    int               incx,
                                    hipComplex* const y[],
                                    int               incy,
                                    int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_cswap_batched((rocblas_handle)handle,
                                                      n,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      (rocblas_float_complex**)y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZswapBatched(hipblasHandle_t         handle,
                                    int                     n,
                                    hipDoubleComplex* const x[],
                                    int                     incx,
                                    hipDoubleComplex* const y[],
                                    int                     incy,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zswap_batched((rocblas_handle)handle,
                                                      n,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      (rocblas_double_complex**)y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// swap_batched_64
hipblasStatus_t hipblasSswapBatched_64(hipblasHandle_t handle,
                                       int64_t         n,
                                       float* const    x[],
                                       int64_t         incx,
                                       float* const    y[],
                                       int64_t         incy,
                                       int64_t         batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_sswap_batched_64((rocblas_handle)handle, n, x, incx, y, incy, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDswapBatched_64(hipblasHandle_t handle,
                                       int64_t         n,
                                       double* const   x[],
                                       int64_t         incx,
                                       double* const   y[],
                                       int64_t         incy,
                                       int64_t         batchCount)
try
{
    return hipblasConvertStatus(
        rocblas_dswap_batched_64((rocblas_handle)handle, n, x, incx, y, incy, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCswapBatched_64(hipblasHandle_t   handle,
                                       int64_t           n,
                                       hipComplex* const x[],
                                       int64_t           incx,
                                       hipComplex* const y[],
                                       int64_t           incy,
                                       int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_cswap_batched_64((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZswapBatched_64(hipblasHandle_t         handle,
                                       int64_t                 n,
                                       hipDoubleComplex* const x[],
                                       int64_t                 incx,
                                       hipDoubleComplex* const y[],
                                       int64_t                 incy,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zswap_batched_64((rocblas_handle)handle,
                                                         n,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// swap_strided_batched
hipblasStatus_t hipblasSswapStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           float*          x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           float*          y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_sswap_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDswapStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           double*         x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           double*         y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dswap_strided_batched(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCswapStridedBatched(hipblasHandle_t handle,
                                           int             n,
                                           hipComplex*     x,
                                           int             incx,
                                           hipblasStride   stridex,
                                           hipComplex*     y,
                                           int             incy,
                                           hipblasStride   stridey,
                                           int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_cswap_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZswapStridedBatched(hipblasHandle_t   handle,
                                           int               n,
                                           hipDoubleComplex* x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           hipDoubleComplex* y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_zswap_strided_batched((rocblas_handle)handle,
                                                              n,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// swap_strided_batched_64
hipblasStatus_t hipblasSswapStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              float*          x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              float*          y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_sswap_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDswapStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              double*         x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              double*         y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_dswap_strided_batched_64(
        (rocblas_handle)handle, n, x, incx, stridex, y, incy, stridey, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCswapStridedBatched_64(hipblasHandle_t handle,
                                              int64_t         n,
                                              hipComplex*     x,
                                              int64_t         incx,
                                              hipblasStride   stridex,
                                              hipComplex*     y,
                                              int64_t         incy,
                                              hipblasStride   stridey,
                                              int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_cswap_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZswapStridedBatched_64(hipblasHandle_t   handle,
                                              int64_t           n,
                                              hipDoubleComplex* x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              hipDoubleComplex* y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zswap_strided_batched_64((rocblas_handle)handle,
                                                                 n,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gbmv
hipblasStatus_t hipblasSgbmv(hipblasHandle_t    handle,
                             hipblasOperation_t trans,
                             int                m,
                             int                n,
                             int                kl,
                             int                ku,
                             const float*       alpha,
                             const float*       A,
                             int                lda,
                             const float*       x,
                             int                incx,
                             const float*       beta,
                             float*             y,
                             int                incy)
try
{
    return hipblasConvertStatus(rocblas_sgbmv((rocblas_handle)handle,
                                              hipblasConvertOperation(trans),
                                              m,
                                              n,
                                              kl,
                                              ku,
                                              alpha,
                                              A,
                                              lda,
                                              x,
                                              incx,
                                              beta,
                                              y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgbmv(hipblasHandle_t    handle,
                             hipblasOperation_t trans,
                             int                m,
                             int                n,
                             int                kl,
                             int                ku,
                             const double*      alpha,
                             const double*      A,
                             int                lda,
                             const double*      x,
                             int                incx,
                             const double*      beta,
                             double*            y,
                             int                incy)
try
{
    return hipblasConvertStatus(rocblas_dgbmv((rocblas_handle)handle,
                                              hipblasConvertOperation(trans),
                                              m,
                                              n,
                                              kl,
                                              ku,
                                              alpha,
                                              A,
                                              lda,
                                              x,
                                              incx,
                                              beta,
                                              y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgbmv(hipblasHandle_t    handle,
                             hipblasOperation_t trans,
                             int                m,
                             int                n,
                             int                kl,
                             int                ku,
                             const hipComplex*  alpha,
                             const hipComplex*  A,
                             int                lda,
                             const hipComplex*  x,
                             int                incx,
                             const hipComplex*  beta,
                             hipComplex*        y,
                             int                incy)
try
{
    return hipblasConvertStatus(rocblas_cgbmv((rocblas_handle)handle,
                                              hipblasConvertOperation(trans),
                                              m,
                                              n,
                                              kl,
                                              ku,
                                              (rocblas_float_complex*)alpha,
                                              (rocblas_float_complex*)A,
                                              lda,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)beta,
                                              (rocblas_float_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgbmv(hipblasHandle_t         handle,
                             hipblasOperation_t      trans,
                             int                     m,
                             int                     n,
                             int                     kl,
                             int                     ku,
                             const hipDoubleComplex* alpha,
                             const hipDoubleComplex* A,
                             int                     lda,
                             const hipDoubleComplex* x,
                             int                     incx,
                             const hipDoubleComplex* beta,
                             hipDoubleComplex*       y,
                             int                     incy)
try
{
    return hipblasConvertStatus(rocblas_zgbmv((rocblas_handle)handle,
                                              hipblasConvertOperation(trans),
                                              m,
                                              n,
                                              kl,
                                              ku,
                                              (rocblas_double_complex*)alpha,
                                              (rocblas_double_complex*)A,
                                              lda,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)beta,
                                              (rocblas_double_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gbmv_64
hipblasStatus_t hipblasSgbmv_64(hipblasHandle_t    handle,
                                hipblasOperation_t trans,
                                int64_t            m,
                                int64_t            n,
                                int64_t            kl,
                                int64_t            ku,
                                const float*       alpha,
                                const float*       A,
                                int64_t            lda,
                                const float*       x,
                                int64_t            incx,
                                const float*       beta,
                                float*             y,
                                int64_t            incy)
try
{
    return hipblasConvertStatus(rocblas_sgbmv_64((rocblas_handle)handle,
                                                 hipblasConvertOperation(trans),
                                                 m,
                                                 n,
                                                 kl,
                                                 ku,
                                                 alpha,
                                                 A,
                                                 lda,
                                                 x,
                                                 incx,
                                                 beta,
                                                 y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgbmv_64(hipblasHandle_t    handle,
                                hipblasOperation_t trans,
                                int64_t            m,
                                int64_t            n,
                                int64_t            kl,
                                int64_t            ku,
                                const double*      alpha,
                                const double*      A,
                                int64_t            lda,
                                const double*      x,
                                int64_t            incx,
                                const double*      beta,
                                double*            y,
                                int64_t            incy)
try
{
    return hipblasConvertStatus(rocblas_dgbmv_64((rocblas_handle)handle,
                                                 hipblasConvertOperation(trans),
                                                 m,
                                                 n,
                                                 kl,
                                                 ku,
                                                 alpha,
                                                 A,
                                                 lda,
                                                 x,
                                                 incx,
                                                 beta,
                                                 y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgbmv_64(hipblasHandle_t    handle,
                                hipblasOperation_t trans,
                                int64_t            m,
                                int64_t            n,
                                int64_t            kl,
                                int64_t            ku,
                                const hipComplex*  alpha,
                                const hipComplex*  A,
                                int64_t            lda,
                                const hipComplex*  x,
                                int64_t            incx,
                                const hipComplex*  beta,
                                hipComplex*        y,
                                int64_t            incy)
try
{
    return hipblasConvertStatus(rocblas_cgbmv_64((rocblas_handle)handle,
                                                 hipblasConvertOperation(trans),
                                                 m,
                                                 n,
                                                 kl,
                                                 ku,
                                                 (rocblas_float_complex*)alpha,
                                                 (rocblas_float_complex*)A,
                                                 lda,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)beta,
                                                 (rocblas_float_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgbmv_64(hipblasHandle_t         handle,
                                hipblasOperation_t      trans,
                                int64_t                 m,
                                int64_t                 n,
                                int64_t                 kl,
                                int64_t                 ku,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* A,
                                int64_t                 lda,
                                const hipDoubleComplex* x,
                                int64_t                 incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int64_t                 incy)
try
{
    return hipblasConvertStatus(rocblas_zgbmv_64((rocblas_handle)handle,
                                                 hipblasConvertOperation(trans),
                                                 m,
                                                 n,
                                                 kl,
                                                 ku,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)A,
                                                 lda,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)beta,
                                                 (rocblas_double_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gbmv_batched
hipblasStatus_t hipblasSgbmvBatched(hipblasHandle_t    handle,
                                    hipblasOperation_t trans,
                                    int                m,
                                    int                n,
                                    int                kl,
                                    int                ku,
                                    const float*       alpha,
                                    const float* const A[],
                                    int                lda,
                                    const float* const x[],
                                    int                incx,
                                    const float*       beta,
                                    float* const       y[],
                                    int                incy,
                                    int                batch_count)
try
{
    return hipblasConvertStatus(rocblas_sgbmv_batched((rocblas_handle)handle,
                                                      hipblasConvertOperation(trans),
                                                      m,
                                                      n,
                                                      kl,
                                                      ku,
                                                      alpha,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      beta,
                                                      y,
                                                      incy,
                                                      batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgbmvBatched(hipblasHandle_t     handle,
                                    hipblasOperation_t  trans,
                                    int                 m,
                                    int                 n,
                                    int                 kl,
                                    int                 ku,
                                    const double*       alpha,
                                    const double* const A[],
                                    int                 lda,
                                    const double* const x[],
                                    int                 incx,
                                    const double*       beta,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batch_count)
try
{
    return hipblasConvertStatus(rocblas_dgbmv_batched((rocblas_handle)handle,
                                                      hipblasConvertOperation(trans),
                                                      m,
                                                      n,
                                                      kl,
                                                      ku,
                                                      alpha,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      beta,
                                                      y,
                                                      incy,
                                                      batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgbmvBatched(hipblasHandle_t         handle,
                                    hipblasOperation_t      trans,
                                    int                     m,
                                    int                     n,
                                    int                     kl,
                                    int                     ku,
                                    const hipComplex*       alpha,
                                    const hipComplex* const A[],
                                    int                     lda,
                                    const hipComplex* const x[],
                                    int                     incx,
                                    const hipComplex*       beta,
                                    hipComplex* const       y[],
                                    int                     incy,
                                    int                     batch_count)
try
{
    return hipblasConvertStatus(rocblas_cgbmv_batched((rocblas_handle)handle,
                                                      hipblasConvertOperation(trans),
                                                      m,
                                                      n,
                                                      kl,
                                                      ku,
                                                      (rocblas_float_complex*)alpha,
                                                      (rocblas_float_complex**)A,
                                                      lda,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      (rocblas_float_complex*)beta,
                                                      (rocblas_float_complex**)y,
                                                      incy,
                                                      batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgbmvBatched(hipblasHandle_t               handle,
                                    hipblasOperation_t            trans,
                                    int                           m,
                                    int                           n,
                                    int                           kl,
                                    int                           ku,
                                    const hipDoubleComplex*       alpha,
                                    const hipDoubleComplex* const A[],
                                    int                           lda,
                                    const hipDoubleComplex* const x[],
                                    int                           incx,
                                    const hipDoubleComplex*       beta,
                                    hipDoubleComplex* const       y[],
                                    int                           incy,
                                    int                           batch_count)
try
{
    return hipblasConvertStatus(rocblas_zgbmv_batched((rocblas_handle)handle,
                                                      hipblasConvertOperation(trans),
                                                      m,
                                                      n,
                                                      kl,
                                                      ku,
                                                      (rocblas_double_complex*)alpha,
                                                      (rocblas_double_complex**)A,
                                                      lda,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      (rocblas_double_complex*)beta,
                                                      (rocblas_double_complex**)y,
                                                      incy,
                                                      batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gbmv_batched_64
hipblasStatus_t hipblasSgbmvBatched_64(hipblasHandle_t    handle,
                                       hipblasOperation_t trans,
                                       int64_t            m,
                                       int64_t            n,
                                       int64_t            kl,
                                       int64_t            ku,
                                       const float*       alpha,
                                       const float* const A[],
                                       int64_t            lda,
                                       const float* const x[],
                                       int64_t            incx,
                                       const float*       beta,
                                       float* const       y[],
                                       int64_t            incy,
                                       int64_t            batch_count)
try
{
    return hipblasConvertStatus(rocblas_sgbmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertOperation(trans),
                                                         m,
                                                         n,
                                                         kl,
                                                         ku,
                                                         alpha,
                                                         A,
                                                         lda,
                                                         x,
                                                         incx,
                                                         beta,
                                                         y,
                                                         incy,
                                                         batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgbmvBatched_64(hipblasHandle_t     handle,
                                       hipblasOperation_t  trans,
                                       int64_t             m,
                                       int64_t             n,
                                       int64_t             kl,
                                       int64_t             ku,
                                       const double*       alpha,
                                       const double* const A[],
                                       int64_t             lda,
                                       const double* const x[],
                                       int64_t             incx,
                                       const double*       beta,
                                       double* const       y[],
                                       int64_t             incy,
                                       int64_t             batch_count)
try
{
    return hipblasConvertStatus(rocblas_dgbmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertOperation(trans),
                                                         m,
                                                         n,
                                                         kl,
                                                         ku,
                                                         alpha,
                                                         A,
                                                         lda,
                                                         x,
                                                         incx,
                                                         beta,
                                                         y,
                                                         incy,
                                                         batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgbmvBatched_64(hipblasHandle_t         handle,
                                       hipblasOperation_t      trans,
                                       int64_t                 m,
                                       int64_t                 n,
                                       int64_t                 kl,
                                       int64_t                 ku,
                                       const hipComplex*       alpha,
                                       const hipComplex* const A[],
                                       int64_t                 lda,
                                       const hipComplex* const x[],
                                       int64_t                 incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int64_t                 incy,
                                       int64_t                 batch_count)
try
{
    return hipblasConvertStatus(rocblas_cgbmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertOperation(trans),
                                                         m,
                                                         n,
                                                         kl,
                                                         ku,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex**)A,
                                                         lda,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex*)beta,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgbmvBatched_64(hipblasHandle_t               handle,
                                       hipblasOperation_t            trans,
                                       int64_t                       m,
                                       int64_t                       n,
                                       int64_t                       kl,
                                       int64_t                       ku,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const A[],
                                       int64_t                       lda,
                                       const hipDoubleComplex* const x[],
                                       int64_t                       incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int64_t                       incy,
                                       int64_t                       batch_count)
try
{
    return hipblasConvertStatus(rocblas_zgbmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertOperation(trans),
                                                         m,
                                                         n,
                                                         kl,
                                                         ku,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex**)A,
                                                         lda,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex*)beta,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gbmv_strided_batched
hipblasStatus_t hipblasSgbmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasOperation_t trans,
                                           int                m,
                                           int                n,
                                           int                kl,
                                           int                ku,
                                           const float*       alpha,
                                           const float*       A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           const float*       x,
                                           int                incx,
                                           hipblasStride      stride_x,
                                           const float*       beta,
                                           float*             y,
                                           int                incy,
                                           hipblasStride      stride_y,
                                           int                batch_count)
try
{
    return hipblasConvertStatus(rocblas_sgbmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertOperation(trans),
                                                              m,
                                                              n,
                                                              kl,
                                                              ku,
                                                              alpha,
                                                              A,
                                                              lda,
                                                              stride_a,
                                                              x,
                                                              incx,
                                                              stride_x,
                                                              beta,
                                                              y,
                                                              incy,
                                                              stride_y,
                                                              batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgbmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasOperation_t trans,
                                           int                m,
                                           int                n,
                                           int                kl,
                                           int                ku,
                                           const double*      alpha,
                                           const double*      A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           const double*      x,
                                           int                incx,
                                           hipblasStride      stride_x,
                                           const double*      beta,
                                           double*            y,
                                           int                incy,
                                           hipblasStride      stride_y,
                                           int                batch_count)
try
{
    return hipblasConvertStatus(rocblas_dgbmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertOperation(trans),
                                                              m,
                                                              n,
                                                              kl,
                                                              ku,
                                                              alpha,
                                                              A,
                                                              lda,
                                                              stride_a,
                                                              x,
                                                              incx,
                                                              stride_x,
                                                              beta,
                                                              y,
                                                              incy,
                                                              stride_y,
                                                              batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgbmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasOperation_t trans,
                                           int                m,
                                           int                n,
                                           int                kl,
                                           int                ku,
                                           const hipComplex*  alpha,
                                           const hipComplex*  A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           const hipComplex*  x,
                                           int                incx,
                                           hipblasStride      stride_x,
                                           const hipComplex*  beta,
                                           hipComplex*        y,
                                           int                incy,
                                           hipblasStride      stride_y,
                                           int                batch_count)
try
{
    return hipblasConvertStatus(rocblas_cgbmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertOperation(trans),
                                                              m,
                                                              n,
                                                              kl,
                                                              ku,
                                                              (rocblas_float_complex*)alpha,
                                                              (rocblas_float_complex*)A,
                                                              lda,
                                                              stride_a,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stride_x,
                                                              (rocblas_float_complex*)beta,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stride_y,
                                                              batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgbmvStridedBatched(hipblasHandle_t         handle,
                                           hipblasOperation_t      trans,
                                           int                     m,
                                           int                     n,
                                           int                     kl,
                                           int                     ku,
                                           const hipDoubleComplex* alpha,
                                           const hipDoubleComplex* A,
                                           int                     lda,
                                           hipblasStride           stride_a,
                                           const hipDoubleComplex* x,
                                           int                     incx,
                                           hipblasStride           stride_x,
                                           const hipDoubleComplex* beta,
                                           hipDoubleComplex*       y,
                                           int                     incy,
                                           hipblasStride           stride_y,
                                           int                     batch_count)
try
{
    return hipblasConvertStatus(rocblas_zgbmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertOperation(trans),
                                                              m,
                                                              n,
                                                              kl,
                                                              ku,
                                                              (rocblas_double_complex*)alpha,
                                                              (rocblas_double_complex*)A,
                                                              lda,
                                                              stride_a,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stride_x,
                                                              (rocblas_double_complex*)beta,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stride_y,
                                                              batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gbmv_strided_batched_64
hipblasStatus_t hipblasSgbmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasOperation_t trans,
                                              int64_t            m,
                                              int64_t            n,
                                              int64_t            kl,
                                              int64_t            ku,
                                              const float*       alpha,
                                              const float*       A,
                                              int64_t            lda,
                                              hipblasStride      stride_a,
                                              const float*       x,
                                              int64_t            incx,
                                              hipblasStride      stride_x,
                                              const float*       beta,
                                              float*             y,
                                              int64_t            incy,
                                              hipblasStride      stride_y,
                                              int64_t            batch_count)
try
{
    return hipblasConvertStatus(rocblas_sgbmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertOperation(trans),
                                                                 m,
                                                                 n,
                                                                 kl,
                                                                 ku,
                                                                 alpha,
                                                                 A,
                                                                 lda,
                                                                 stride_a,
                                                                 x,
                                                                 incx,
                                                                 stride_x,
                                                                 beta,
                                                                 y,
                                                                 incy,
                                                                 stride_y,
                                                                 batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgbmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasOperation_t trans,
                                              int64_t            m,
                                              int64_t            n,
                                              int64_t            kl,
                                              int64_t            ku,
                                              const double*      alpha,
                                              const double*      A,
                                              int64_t            lda,
                                              hipblasStride      stride_a,
                                              const double*      x,
                                              int64_t            incx,
                                              hipblasStride      stride_x,
                                              const double*      beta,
                                              double*            y,
                                              int64_t            incy,
                                              hipblasStride      stride_y,
                                              int64_t            batch_count)
try
{
    return hipblasConvertStatus(rocblas_dgbmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertOperation(trans),
                                                                 m,
                                                                 n,
                                                                 kl,
                                                                 ku,
                                                                 alpha,
                                                                 A,
                                                                 lda,
                                                                 stride_a,
                                                                 x,
                                                                 incx,
                                                                 stride_x,
                                                                 beta,
                                                                 y,
                                                                 incy,
                                                                 stride_y,
                                                                 batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgbmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasOperation_t trans,
                                              int64_t            m,
                                              int64_t            n,
                                              int64_t            kl,
                                              int64_t            ku,
                                              const hipComplex*  alpha,
                                              const hipComplex*  A,
                                              int64_t            lda,
                                              hipblasStride      stride_a,
                                              const hipComplex*  x,
                                              int64_t            incx,
                                              hipblasStride      stride_x,
                                              const hipComplex*  beta,
                                              hipComplex*        y,
                                              int64_t            incy,
                                              hipblasStride      stride_y,
                                              int64_t            batch_count)
try
{
    return hipblasConvertStatus(rocblas_cgbmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertOperation(trans),
                                                                 m,
                                                                 n,
                                                                 kl,
                                                                 ku,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)A,
                                                                 lda,
                                                                 stride_a,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stride_x,
                                                                 (rocblas_float_complex*)beta,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stride_y,
                                                                 batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgbmvStridedBatched_64(hipblasHandle_t         handle,
                                              hipblasOperation_t      trans,
                                              int64_t                 m,
                                              int64_t                 n,
                                              int64_t                 kl,
                                              int64_t                 ku,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* A,
                                              int64_t                 lda,
                                              hipblasStride           stride_a,
                                              const hipDoubleComplex* x,
                                              int64_t                 incx,
                                              hipblasStride           stride_x,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int64_t                 incy,
                                              hipblasStride           stride_y,
                                              int64_t                 batch_count)
try
{
    return hipblasConvertStatus(rocblas_zgbmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertOperation(trans),
                                                                 m,
                                                                 n,
                                                                 kl,
                                                                 ku,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)A,
                                                                 lda,
                                                                 stride_a,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stride_x,
                                                                 (rocblas_double_complex*)beta,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stride_y,
                                                                 batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gemv
hipblasStatus_t hipblasSgemv(hipblasHandle_t    handle,
                             hipblasOperation_t trans,
                             int                m,
                             int                n,
                             const float*       alpha,
                             const float*       A,
                             int                lda,
                             const float*       x,
                             int                incx,
                             const float*       beta,
                             float*             y,
                             int                incy)
try
{
    return hipblasConvertStatus(rocblas_sgemv((rocblas_handle)handle,
                                              hipblasConvertOperation(trans),
                                              m,
                                              n,
                                              alpha,
                                              A,
                                              lda,
                                              x,
                                              incx,
                                              beta,
                                              y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgemv(hipblasHandle_t    handle,
                             hipblasOperation_t trans,
                             int                m,
                             int                n,
                             const double*      alpha,
                             const double*      A,
                             int                lda,
                             const double*      x,
                             int                incx,
                             const double*      beta,
                             double*            y,
                             int                incy)
try
{
    return hipblasConvertStatus(rocblas_dgemv((rocblas_handle)handle,
                                              hipblasConvertOperation(trans),
                                              m,
                                              n,
                                              alpha,
                                              A,
                                              lda,
                                              x,
                                              incx,
                                              beta,
                                              y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemv(hipblasHandle_t    handle,
                             hipblasOperation_t trans,
                             int                m,
                             int                n,
                             const hipComplex*  alpha,
                             const hipComplex*  A,
                             int                lda,
                             const hipComplex*  x,
                             int                incx,
                             const hipComplex*  beta,
                             hipComplex*        y,
                             int                incy)
try
{
    return hipblasConvertStatus(rocblas_cgemv((rocblas_handle)handle,
                                              hipblasConvertOperation(trans),
                                              m,
                                              n,
                                              (rocblas_float_complex*)alpha,
                                              (rocblas_float_complex*)A,
                                              lda,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)beta,
                                              (rocblas_float_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemv(hipblasHandle_t         handle,
                             hipblasOperation_t      trans,
                             int                     m,
                             int                     n,
                             const hipDoubleComplex* alpha,
                             const hipDoubleComplex* A,
                             int                     lda,
                             const hipDoubleComplex* x,
                             int                     incx,
                             const hipDoubleComplex* beta,
                             hipDoubleComplex*       y,
                             int                     incy)
try
{
    return hipblasConvertStatus(rocblas_zgemv((rocblas_handle)handle,
                                              hipblasConvertOperation(trans),
                                              m,
                                              n,
                                              (rocblas_double_complex*)alpha,
                                              (rocblas_double_complex*)A,
                                              lda,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)beta,
                                              (rocblas_double_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gemv_64
hipblasStatus_t hipblasSgemv_64(hipblasHandle_t    handle,
                                hipblasOperation_t trans,
                                int64_t            m,
                                int64_t            n,
                                const float*       alpha,
                                const float*       A,
                                int64_t            lda,
                                const float*       x,
                                int64_t            incx,
                                const float*       beta,
                                float*             y,
                                int64_t            incy)
try
{
    return hipblasConvertStatus(rocblas_sgemv_64((rocblas_handle)handle,
                                                 hipblasConvertOperation(trans),
                                                 m,
                                                 n,
                                                 alpha,
                                                 A,
                                                 lda,
                                                 x,
                                                 incx,
                                                 beta,
                                                 y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgemv_64(hipblasHandle_t    handle,
                                hipblasOperation_t trans,
                                int64_t            m,
                                int64_t            n,
                                const double*      alpha,
                                const double*      A,
                                int64_t            lda,
                                const double*      x,
                                int64_t            incx,
                                const double*      beta,
                                double*            y,
                                int64_t            incy)
try
{
    return hipblasConvertStatus(rocblas_dgemv_64((rocblas_handle)handle,
                                                 hipblasConvertOperation(trans),
                                                 m,
                                                 n,
                                                 alpha,
                                                 A,
                                                 lda,
                                                 x,
                                                 incx,
                                                 beta,
                                                 y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemv_64(hipblasHandle_t    handle,
                                hipblasOperation_t trans,
                                int64_t            m,
                                int64_t            n,
                                const hipComplex*  alpha,
                                const hipComplex*  A,
                                int64_t            lda,
                                const hipComplex*  x,
                                int64_t            incx,
                                const hipComplex*  beta,
                                hipComplex*        y,
                                int64_t            incy)
try
{
    return hipblasConvertStatus(rocblas_cgemv_64((rocblas_handle)handle,
                                                 hipblasConvertOperation(trans),
                                                 m,
                                                 n,
                                                 (rocblas_float_complex*)alpha,
                                                 (rocblas_float_complex*)A,
                                                 lda,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)beta,
                                                 (rocblas_float_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemv_64(hipblasHandle_t         handle,
                                hipblasOperation_t      trans,
                                int64_t                 m,
                                int64_t                 n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* A,
                                int64_t                 lda,
                                const hipDoubleComplex* x,
                                int64_t                 incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int64_t                 incy)
try
{
    return hipblasConvertStatus(rocblas_zgemv_64((rocblas_handle)handle,
                                                 hipblasConvertOperation(trans),
                                                 m,
                                                 n,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)A,
                                                 lda,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)beta,
                                                 (rocblas_double_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gemv_batched
hipblasStatus_t hipblasSgemvBatched(hipblasHandle_t    handle,
                                    hipblasOperation_t trans,
                                    int                m,
                                    int                n,
                                    const float*       alpha,
                                    const float* const A[],
                                    int                lda,
                                    const float* const x[],
                                    int                incx,
                                    const float*       beta,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_sgemv_batched((rocblas_handle)handle,
                                                      hipblasConvertOperation(trans),
                                                      m,
                                                      n,
                                                      alpha,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      beta,
                                                      y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgemvBatched(hipblasHandle_t     handle,
                                    hipblasOperation_t  trans,
                                    int                 m,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const A[],
                                    int                 lda,
                                    const double* const x[],
                                    int                 incx,
                                    const double*       beta,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_dgemv_batched((rocblas_handle)handle,
                                                      hipblasConvertOperation(trans),
                                                      m,
                                                      n,
                                                      alpha,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      beta,
                                                      y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemvBatched(hipblasHandle_t         handle,
                                    hipblasOperation_t      trans,
                                    int                     m,
                                    int                     n,
                                    const hipComplex*       alpha,
                                    const hipComplex* const A[],
                                    int                     lda,
                                    const hipComplex* const x[],
                                    int                     incx,
                                    const hipComplex*       beta,
                                    hipComplex* const       y[],
                                    int                     incy,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_cgemv_batched((rocblas_handle)handle,
                                                      hipblasConvertOperation(trans),
                                                      m,
                                                      n,
                                                      (rocblas_float_complex*)alpha,
                                                      (rocblas_float_complex**)A,
                                                      lda,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      (rocblas_float_complex*)beta,
                                                      (rocblas_float_complex**)y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemvBatched(hipblasHandle_t               handle,
                                    hipblasOperation_t            trans,
                                    int                           m,
                                    int                           n,
                                    const hipDoubleComplex*       alpha,
                                    const hipDoubleComplex* const A[],
                                    int                           lda,
                                    const hipDoubleComplex* const x[],
                                    int                           incx,
                                    const hipDoubleComplex*       beta,
                                    hipDoubleComplex* const       y[],
                                    int                           incy,
                                    int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zgemv_batched((rocblas_handle)handle,
                                                      hipblasConvertOperation(trans),
                                                      m,
                                                      n,
                                                      (rocblas_double_complex*)alpha,
                                                      (rocblas_double_complex**)A,
                                                      lda,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      (rocblas_double_complex*)beta,
                                                      (rocblas_double_complex**)y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gemv_batched_64
hipblasStatus_t hipblasSgemvBatched_64(hipblasHandle_t    handle,
                                       hipblasOperation_t trans,
                                       int64_t            m,
                                       int64_t            n,
                                       const float*       alpha,
                                       const float* const A[],
                                       int64_t            lda,
                                       const float* const x[],
                                       int64_t            incx,
                                       const float*       beta,
                                       float* const       y[],
                                       int64_t            incy,
                                       int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_sgemv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertOperation(trans),
                                                         m,
                                                         n,
                                                         alpha,
                                                         A,
                                                         lda,
                                                         x,
                                                         incx,
                                                         beta,
                                                         y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgemvBatched_64(hipblasHandle_t     handle,
                                       hipblasOperation_t  trans,
                                       int64_t             m,
                                       int64_t             n,
                                       const double*       alpha,
                                       const double* const A[],
                                       int64_t             lda,
                                       const double* const x[],
                                       int64_t             incx,
                                       const double*       beta,
                                       double* const       y[],
                                       int64_t             incy,
                                       int64_t             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dgemv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertOperation(trans),
                                                         m,
                                                         n,
                                                         alpha,
                                                         A,
                                                         lda,
                                                         x,
                                                         incx,
                                                         beta,
                                                         y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemvBatched_64(hipblasHandle_t         handle,
                                       hipblasOperation_t      trans,
                                       int64_t                 m,
                                       int64_t                 n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const A[],
                                       int64_t                 lda,
                                       const hipComplex* const x[],
                                       int64_t                 incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int64_t                 incy,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_cgemv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertOperation(trans),
                                                         m,
                                                         n,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex**)A,
                                                         lda,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex*)beta,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemvBatched_64(hipblasHandle_t               handle,
                                       hipblasOperation_t            trans,
                                       int64_t                       m,
                                       int64_t                       n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const A[],
                                       int64_t                       lda,
                                       const hipDoubleComplex* const x[],
                                       int64_t                       incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int64_t                       incy,
                                       int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_zgemv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertOperation(trans),
                                                         m,
                                                         n,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex**)A,
                                                         lda,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex*)beta,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gemv_strided_batched
hipblasStatus_t hipblasSgemvStridedBatched(hipblasHandle_t    handle,
                                           hipblasOperation_t trans,
                                           int                m,
                                           int                n,
                                           const float*       alpha,
                                           const float*       A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           const float*       x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           const float*       beta,
                                           float*             y,
                                           int                incy,
                                           hipblasStride      stridey,
                                           int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_sgemv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertOperation(trans),
                                                              m,
                                                              n,
                                                              alpha,
                                                              A,
                                                              lda,
                                                              strideA,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              beta,
                                                              y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgemvStridedBatched(hipblasHandle_t    handle,
                                           hipblasOperation_t trans,
                                           int                m,
                                           int                n,
                                           const double*      alpha,
                                           const double*      A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           const double*      x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           const double*      beta,
                                           double*            y,
                                           int                incy,
                                           hipblasStride      stridey,
                                           int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_dgemv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertOperation(trans),
                                                              m,
                                                              n,
                                                              alpha,
                                                              A,
                                                              lda,
                                                              strideA,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              beta,
                                                              y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemvStridedBatched(hipblasHandle_t    handle,
                                           hipblasOperation_t trans,
                                           int                m,
                                           int                n,
                                           const hipComplex*  alpha,
                                           const hipComplex*  A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           const hipComplex*  x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           const hipComplex*  beta,
                                           hipComplex*        y,
                                           int                incy,
                                           hipblasStride      stridey,
                                           int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_cgemv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertOperation(trans),
                                                              m,
                                                              n,
                                                              (rocblas_float_complex*)alpha,
                                                              (rocblas_float_complex*)A,
                                                              lda,
                                                              strideA,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_float_complex*)beta,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemvStridedBatched(hipblasHandle_t         handle,
                                           hipblasOperation_t      trans,
                                           int                     m,
                                           int                     n,
                                           const hipDoubleComplex* alpha,
                                           const hipDoubleComplex* A,
                                           int                     lda,
                                           hipblasStride           strideA,
                                           const hipDoubleComplex* x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           const hipDoubleComplex* beta,
                                           hipDoubleComplex*       y,
                                           int                     incy,
                                           hipblasStride           stridey,
                                           int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zgemv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertOperation(trans),
                                                              m,
                                                              n,
                                                              (rocblas_double_complex*)alpha,
                                                              (rocblas_double_complex*)A,
                                                              lda,
                                                              strideA,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_double_complex*)beta,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// gemv_strided_batched_64
hipblasStatus_t hipblasSgemvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasOperation_t trans,
                                              int64_t            m,
                                              int64_t            n,
                                              const float*       alpha,
                                              const float*       A,
                                              int64_t            lda,
                                              hipblasStride      strideA,
                                              const float*       x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              const float*       beta,
                                              float*             y,
                                              int64_t            incy,
                                              hipblasStride      stridey,
                                              int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_sgemv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertOperation(trans),
                                                                 m,
                                                                 n,
                                                                 alpha,
                                                                 A,
                                                                 lda,
                                                                 strideA,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 beta,
                                                                 y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgemvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasOperation_t trans,
                                              int64_t            m,
                                              int64_t            n,
                                              const double*      alpha,
                                              const double*      A,
                                              int64_t            lda,
                                              hipblasStride      strideA,
                                              const double*      x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              const double*      beta,
                                              double*            y,
                                              int64_t            incy,
                                              hipblasStride      stridey,
                                              int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_dgemv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertOperation(trans),
                                                                 m,
                                                                 n,
                                                                 alpha,
                                                                 A,
                                                                 lda,
                                                                 strideA,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 beta,
                                                                 y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgemvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasOperation_t trans,
                                              int64_t            m,
                                              int64_t            n,
                                              const hipComplex*  alpha,
                                              const hipComplex*  A,
                                              int64_t            lda,
                                              hipblasStride      strideA,
                                              const hipComplex*  x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              const hipComplex*  beta,
                                              hipComplex*        y,
                                              int64_t            incy,
                                              hipblasStride      stridey,
                                              int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_cgemv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertOperation(trans),
                                                                 m,
                                                                 n,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)beta,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgemvStridedBatched_64(hipblasHandle_t         handle,
                                              hipblasOperation_t      trans,
                                              int64_t                 m,
                                              int64_t                 n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* A,
                                              int64_t                 lda,
                                              hipblasStride           strideA,
                                              const hipDoubleComplex* x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int64_t                 incy,
                                              hipblasStride           stridey,
                                              int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zgemv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertOperation(trans),
                                                                 m,
                                                                 n,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)beta,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// ger
hipblasStatus_t hipblasSger(hipblasHandle_t handle,
                            int             m,
                            int             n,
                            const float*    alpha,
                            const float*    x,
                            int             incx,
                            const float*    y,
                            int             incy,
                            float*          A,
                            int             lda)
try
{
    return hipblasConvertStatus(
        rocblas_sger((rocblas_handle)handle, m, n, alpha, x, incx, y, incy, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDger(hipblasHandle_t handle,
                            int             m,
                            int             n,
                            const double*   alpha,
                            const double*   x,
                            int             incx,
                            const double*   y,
                            int             incy,
                            double*         A,
                            int             lda)
try
{
    return hipblasConvertStatus(
        rocblas_dger((rocblas_handle)handle, m, n, alpha, x, incx, y, incy, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgeru(hipblasHandle_t   handle,
                             int               m,
                             int               n,
                             const hipComplex* alpha,
                             const hipComplex* x,
                             int               incx,
                             const hipComplex* y,
                             int               incy,
                             hipComplex*       A,
                             int               lda)
try
{
    return hipblasConvertStatus(rocblas_cgeru((rocblas_handle)handle,
                                              m,
                                              n,
                                              (rocblas_float_complex*)alpha,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)y,
                                              incy,
                                              (rocblas_float_complex*)A,
                                              lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgerc(hipblasHandle_t   handle,
                             int               m,
                             int               n,
                             const hipComplex* alpha,
                             const hipComplex* x,
                             int               incx,
                             const hipComplex* y,
                             int               incy,
                             hipComplex*       A,
                             int               lda)
try
{
    return hipblasConvertStatus(rocblas_cgerc((rocblas_handle)handle,
                                              m,
                                              n,
                                              (rocblas_float_complex*)alpha,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)y,
                                              incy,
                                              (rocblas_float_complex*)A,
                                              lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgeru(hipblasHandle_t         handle,
                             int                     m,
                             int                     n,
                             const hipDoubleComplex* alpha,
                             const hipDoubleComplex* x,
                             int                     incx,
                             const hipDoubleComplex* y,
                             int                     incy,
                             hipDoubleComplex*       A,
                             int                     lda)
try
{
    return hipblasConvertStatus(rocblas_zgeru((rocblas_handle)handle,
                                              m,
                                              n,
                                              (rocblas_double_complex*)alpha,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)y,
                                              incy,
                                              (rocblas_double_complex*)A,
                                              lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgerc(hipblasHandle_t         handle,
                             int                     m,
                             int                     n,
                             const hipDoubleComplex* alpha,
                             const hipDoubleComplex* x,
                             int                     incx,
                             const hipDoubleComplex* y,
                             int                     incy,
                             hipDoubleComplex*       A,
                             int                     lda)
try
{
    return hipblasConvertStatus(rocblas_zgerc((rocblas_handle)handle,
                                              m,
                                              n,
                                              (rocblas_double_complex*)alpha,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)y,
                                              incy,
                                              (rocblas_double_complex*)A,
                                              lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// ger_64
hipblasStatus_t hipblasSger_64(hipblasHandle_t handle,
                               int64_t         m,
                               int64_t         n,
                               const float*    alpha,
                               const float*    x,
                               int64_t         incx,
                               const float*    y,
                               int64_t         incy,
                               float*          A,
                               int64_t         lda)
try
{
    return hipblasConvertStatus(
        rocblas_sger_64((rocblas_handle)handle, m, n, alpha, x, incx, y, incy, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDger_64(hipblasHandle_t handle,
                               int64_t         m,
                               int64_t         n,
                               const double*   alpha,
                               const double*   x,
                               int64_t         incx,
                               const double*   y,
                               int64_t         incy,
                               double*         A,
                               int64_t         lda)
try
{
    return hipblasConvertStatus(
        rocblas_dger_64((rocblas_handle)handle, m, n, alpha, x, incx, y, incy, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgeru_64(hipblasHandle_t   handle,
                                int64_t           m,
                                int64_t           n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int64_t           incx,
                                const hipComplex* y,
                                int64_t           incy,
                                hipComplex*       A,
                                int64_t           lda)
try
{
    return hipblasConvertStatus(rocblas_cgeru_64((rocblas_handle)handle,
                                                 m,
                                                 n,
                                                 (rocblas_float_complex*)alpha,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)y,
                                                 incy,
                                                 (rocblas_float_complex*)A,
                                                 lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgerc_64(hipblasHandle_t   handle,
                                int64_t           m,
                                int64_t           n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int64_t           incx,
                                const hipComplex* y,
                                int64_t           incy,
                                hipComplex*       A,
                                int64_t           lda)
try
{
    return hipblasConvertStatus(rocblas_cgerc_64((rocblas_handle)handle,
                                                 m,
                                                 n,
                                                 (rocblas_float_complex*)alpha,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)y,
                                                 incy,
                                                 (rocblas_float_complex*)A,
                                                 lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgeru_64(hipblasHandle_t         handle,
                                int64_t                 m,
                                int64_t                 n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int64_t                 incx,
                                const hipDoubleComplex* y,
                                int64_t                 incy,
                                hipDoubleComplex*       A,
                                int64_t                 lda)
try
{
    return hipblasConvertStatus(rocblas_zgeru_64((rocblas_handle)handle,
                                                 m,
                                                 n,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)y,
                                                 incy,
                                                 (rocblas_double_complex*)A,
                                                 lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgerc_64(hipblasHandle_t         handle,
                                int64_t                 m,
                                int64_t                 n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int64_t                 incx,
                                const hipDoubleComplex* y,
                                int64_t                 incy,
                                hipDoubleComplex*       A,
                                int64_t                 lda)
try
{
    return hipblasConvertStatus(rocblas_zgerc_64((rocblas_handle)handle,
                                                 m,
                                                 n,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)y,
                                                 incy,
                                                 (rocblas_double_complex*)A,
                                                 lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// ger_batched
hipblasStatus_t hipblasSgerBatched(hipblasHandle_t    handle,
                                   int                m,
                                   int                n,
                                   const float*       alpha,
                                   const float* const x[],
                                   int                incx,
                                   const float* const y[],
                                   int                incy,
                                   float* const       A[],
                                   int                lda,
                                   int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_sger_batched(
        (rocblas_handle)handle, m, n, alpha, x, incx, y, incy, A, lda, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgerBatched(hipblasHandle_t     handle,
                                   int                 m,
                                   int                 n,
                                   const double*       alpha,
                                   const double* const x[],
                                   int                 incx,
                                   const double* const y[],
                                   int                 incy,
                                   double* const       A[],
                                   int                 lda,
                                   int                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_dger_batched(
        (rocblas_handle)handle, m, n, alpha, x, incx, y, incy, A, lda, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgeruBatched(hipblasHandle_t         handle,
                                    int                     m,
                                    int                     n,
                                    const hipComplex*       alpha,
                                    const hipComplex* const x[],
                                    int                     incx,
                                    const hipComplex* const y[],
                                    int                     incy,
                                    hipComplex* const       A[],
                                    int                     lda,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_cgeru_batched((rocblas_handle)handle,
                                                      m,
                                                      n,
                                                      (rocblas_float_complex*)alpha,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      (rocblas_float_complex**)y,
                                                      incy,
                                                      (rocblas_float_complex**)A,
                                                      lda,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgercBatched(hipblasHandle_t         handle,
                                    int                     m,
                                    int                     n,
                                    const hipComplex*       alpha,
                                    const hipComplex* const x[],
                                    int                     incx,
                                    const hipComplex* const y[],
                                    int                     incy,
                                    hipComplex* const       A[],
                                    int                     lda,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_cgerc_batched((rocblas_handle)handle,
                                                      m,
                                                      n,
                                                      (rocblas_float_complex*)alpha,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      (rocblas_float_complex**)y,
                                                      incy,
                                                      (rocblas_float_complex**)A,
                                                      lda,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgeruBatched(hipblasHandle_t               handle,
                                    int                           m,
                                    int                           n,
                                    const hipDoubleComplex*       alpha,
                                    const hipDoubleComplex* const x[],
                                    int                           incx,
                                    const hipDoubleComplex* const y[],
                                    int                           incy,
                                    hipDoubleComplex* const       A[],
                                    int                           lda,
                                    int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zgeru_batched((rocblas_handle)handle,
                                                      m,
                                                      n,
                                                      (rocblas_double_complex*)alpha,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      (rocblas_double_complex**)y,
                                                      incy,
                                                      (rocblas_double_complex**)A,
                                                      lda,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgercBatched(hipblasHandle_t               handle,
                                    int                           m,
                                    int                           n,
                                    const hipDoubleComplex*       alpha,
                                    const hipDoubleComplex* const x[],
                                    int                           incx,
                                    const hipDoubleComplex* const y[],
                                    int                           incy,
                                    hipDoubleComplex* const       A[],
                                    int                           lda,
                                    int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zgerc_batched((rocblas_handle)handle,
                                                      m,
                                                      n,
                                                      (rocblas_double_complex*)alpha,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      (rocblas_double_complex**)y,
                                                      incy,
                                                      (rocblas_double_complex**)A,
                                                      lda,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// ger_batched_64
hipblasStatus_t hipblasSgerBatched_64(hipblasHandle_t    handle,
                                      int64_t            m,
                                      int64_t            n,
                                      const float*       alpha,
                                      const float* const x[],
                                      int64_t            incx,
                                      const float* const y[],
                                      int64_t            incy,
                                      float* const       A[],
                                      int64_t            lda,
                                      int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_sger_batched_64(
        (rocblas_handle)handle, m, n, alpha, x, incx, y, incy, A, lda, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgerBatched_64(hipblasHandle_t     handle,
                                      int64_t             m,
                                      int64_t             n,
                                      const double*       alpha,
                                      const double* const x[],
                                      int64_t             incx,
                                      const double* const y[],
                                      int64_t             incy,
                                      double* const       A[],
                                      int64_t             lda,
                                      int64_t             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dger_batched_64(
        (rocblas_handle)handle, m, n, alpha, x, incx, y, incy, A, lda, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgeruBatched_64(hipblasHandle_t         handle,
                                       int64_t                 m,
                                       int64_t                 n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int64_t                 incx,
                                       const hipComplex* const y[],
                                       int64_t                 incy,
                                       hipComplex* const       A[],
                                       int64_t                 lda,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_cgeru_batched_64((rocblas_handle)handle,
                                                         m,
                                                         n,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         (rocblas_float_complex**)A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgercBatched_64(hipblasHandle_t         handle,
                                       int64_t                 m,
                                       int64_t                 n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int64_t                 incx,
                                       const hipComplex* const y[],
                                       int64_t                 incy,
                                       hipComplex* const       A[],
                                       int64_t                 lda,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_cgerc_batched_64((rocblas_handle)handle,
                                                         m,
                                                         n,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         (rocblas_float_complex**)A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgeruBatched_64(hipblasHandle_t               handle,
                                       int64_t                       m,
                                       int64_t                       n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int64_t                       incx,
                                       const hipDoubleComplex* const y[],
                                       int64_t                       incy,
                                       hipDoubleComplex* const       A[],
                                       int64_t                       lda,
                                       int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_zgeru_batched_64((rocblas_handle)handle,
                                                         m,
                                                         n,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         (rocblas_double_complex**)A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgercBatched_64(hipblasHandle_t               handle,
                                       int64_t                       m,
                                       int64_t                       n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int64_t                       incx,
                                       const hipDoubleComplex* const y[],
                                       int64_t                       incy,
                                       hipDoubleComplex* const       A[],
                                       int64_t                       lda,
                                       int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_zgerc_batched_64((rocblas_handle)handle,
                                                         m,
                                                         n,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         (rocblas_double_complex**)A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// ger_strided_batched
hipblasStatus_t hipblasSgerStridedBatched(hipblasHandle_t handle,
                                          int             m,
                                          int             n,
                                          const float*    alpha,
                                          const float*    x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          const float*    y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          float*          A,
                                          int             lda,
                                          hipblasStride   strideA,
                                          int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_sger_strided_batched((rocblas_handle)handle,
                                                             m,
                                                             n,
                                                             alpha,
                                                             x,
                                                             incx,
                                                             stridex,
                                                             y,
                                                             incy,
                                                             stridey,
                                                             A,
                                                             lda,
                                                             strideA,
                                                             batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgerStridedBatched(hipblasHandle_t handle,
                                          int             m,
                                          int             n,
                                          const double*   alpha,
                                          const double*   x,
                                          int             incx,
                                          hipblasStride   stridex,
                                          const double*   y,
                                          int             incy,
                                          hipblasStride   stridey,
                                          double*         A,
                                          int             lda,
                                          hipblasStride   strideA,
                                          int             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dger_strided_batched((rocblas_handle)handle,
                                                             m,
                                                             n,
                                                             alpha,
                                                             x,
                                                             incx,
                                                             stridex,
                                                             y,
                                                             incy,
                                                             stridey,
                                                             A,
                                                             lda,
                                                             strideA,
                                                             batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgeruStridedBatched(hipblasHandle_t   handle,
                                           int               m,
                                           int               n,
                                           const hipComplex* alpha,
                                           const hipComplex* x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const hipComplex* y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           hipComplex*       A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_cgeru_strided_batched((rocblas_handle)handle,
                                                              m,
                                                              n,
                                                              (rocblas_float_complex*)alpha,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stridey,
                                                              (rocblas_float_complex*)A,
                                                              lda,
                                                              strideA,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgercStridedBatched(hipblasHandle_t   handle,
                                           int               m,
                                           int               n,
                                           const hipComplex* alpha,
                                           const hipComplex* x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const hipComplex* y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           hipComplex*       A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_cgerc_strided_batched((rocblas_handle)handle,
                                                              m,
                                                              n,
                                                              (rocblas_float_complex*)alpha,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stridey,
                                                              (rocblas_float_complex*)A,
                                                              lda,
                                                              strideA,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgeruStridedBatched(hipblasHandle_t         handle,
                                           int                     m,
                                           int                     n,
                                           const hipDoubleComplex* alpha,
                                           const hipDoubleComplex* x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           const hipDoubleComplex* y,
                                           int                     incy,
                                           hipblasStride           stridey,
                                           hipDoubleComplex*       A,
                                           int                     lda,
                                           hipblasStride           strideA,
                                           int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zgeru_strided_batched((rocblas_handle)handle,
                                                              m,
                                                              n,
                                                              (rocblas_double_complex*)alpha,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stridey,
                                                              (rocblas_double_complex*)A,
                                                              lda,
                                                              strideA,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgercStridedBatched(hipblasHandle_t         handle,
                                           int                     m,
                                           int                     n,
                                           const hipDoubleComplex* alpha,
                                           const hipDoubleComplex* x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           const hipDoubleComplex* y,
                                           int                     incy,
                                           hipblasStride           stridey,
                                           hipDoubleComplex*       A,
                                           int                     lda,
                                           hipblasStride           strideA,
                                           int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zgerc_strided_batched((rocblas_handle)handle,
                                                              m,
                                                              n,
                                                              (rocblas_double_complex*)alpha,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stridey,
                                                              (rocblas_double_complex*)A,
                                                              lda,
                                                              strideA,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// ger_strided_batched_64
hipblasStatus_t hipblasSgerStridedBatched_64(hipblasHandle_t handle,
                                             int64_t         m,
                                             int64_t         n,
                                             const float*    alpha,
                                             const float*    x,
                                             int64_t         incx,
                                             hipblasStride   stridex,
                                             const float*    y,
                                             int64_t         incy,
                                             hipblasStride   stridey,
                                             float*          A,
                                             int64_t         lda,
                                             hipblasStride   strideA,
                                             int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_sger_strided_batched_64((rocblas_handle)handle,
                                                                m,
                                                                n,
                                                                alpha,
                                                                x,
                                                                incx,
                                                                stridex,
                                                                y,
                                                                incy,
                                                                stridey,
                                                                A,
                                                                lda,
                                                                strideA,
                                                                batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDgerStridedBatched_64(hipblasHandle_t handle,
                                             int64_t         m,
                                             int64_t         n,
                                             const double*   alpha,
                                             const double*   x,
                                             int64_t         incx,
                                             hipblasStride   stridex,
                                             const double*   y,
                                             int64_t         incy,
                                             hipblasStride   stridey,
                                             double*         A,
                                             int64_t         lda,
                                             hipblasStride   strideA,
                                             int64_t         batchCount)
try
{
    return hipblasConvertStatus(rocblas_dger_strided_batched_64((rocblas_handle)handle,
                                                                m,
                                                                n,
                                                                alpha,
                                                                x,
                                                                incx,
                                                                stridex,
                                                                y,
                                                                incy,
                                                                stridey,
                                                                A,
                                                                lda,
                                                                strideA,
                                                                batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgeruStridedBatched_64(hipblasHandle_t   handle,
                                              int64_t           m,
                                              int64_t           n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              hipComplex*       A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_cgeru_strided_batched_64((rocblas_handle)handle,
                                                                 m,
                                                                 n,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 (rocblas_float_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCgercStridedBatched_64(hipblasHandle_t   handle,
                                              int64_t           m,
                                              int64_t           n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              hipComplex*       A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_cgerc_strided_batched_64((rocblas_handle)handle,
                                                                 m,
                                                                 n,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 (rocblas_float_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgeruStridedBatched_64(hipblasHandle_t         handle,
                                              int64_t                 m,
                                              int64_t                 n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int64_t                 incy,
                                              hipblasStride           stridey,
                                              hipDoubleComplex*       A,
                                              int64_t                 lda,
                                              hipblasStride           strideA,
                                              int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zgeru_strided_batched_64((rocblas_handle)handle,
                                                                 m,
                                                                 n,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 (rocblas_double_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZgercStridedBatched_64(hipblasHandle_t         handle,
                                              int64_t                 m,
                                              int64_t                 n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int64_t                 incy,
                                              hipblasStride           stridey,
                                              hipDoubleComplex*       A,
                                              int64_t                 lda,
                                              hipblasStride           strideA,
                                              int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zgerc_strided_batched_64((rocblas_handle)handle,
                                                                 m,
                                                                 n,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 (rocblas_double_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hbmv
hipblasStatus_t hipblasChbmv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             int               k,
                             const hipComplex* alpha,
                             const hipComplex* A,
                             int               lda,
                             const hipComplex* x,
                             int               incx,
                             const hipComplex* beta,
                             hipComplex*       y,
                             int               incy)
try
{
    return hipblasConvertStatus(rocblas_chbmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              k,
                                              (rocblas_float_complex*)alpha,
                                              (rocblas_float_complex*)A,
                                              lda,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)beta,
                                              (rocblas_float_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhbmv(hipblasHandle_t         handle,
                             hipblasFillMode_t       uplo,
                             int                     n,
                             int                     k,
                             const hipDoubleComplex* alpha,
                             const hipDoubleComplex* A,
                             int                     lda,
                             const hipDoubleComplex* x,
                             int                     incx,
                             const hipDoubleComplex* beta,
                             hipDoubleComplex*       y,
                             int                     incy)
try
{
    return hipblasConvertStatus(rocblas_zhbmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              k,
                                              (rocblas_double_complex*)alpha,
                                              (rocblas_double_complex*)A,
                                              lda,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)beta,
                                              (rocblas_double_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hbmv_64
hipblasStatus_t hipblasChbmv_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                int64_t           k,
                                const hipComplex* alpha,
                                const hipComplex* A,
                                int64_t           lda,
                                const hipComplex* x,
                                int64_t           incx,
                                const hipComplex* beta,
                                hipComplex*       y,
                                int64_t           incy)
try
{
    return hipblasConvertStatus(rocblas_chbmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 k,
                                                 (rocblas_float_complex*)alpha,
                                                 (rocblas_float_complex*)A,
                                                 lda,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)beta,
                                                 (rocblas_float_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhbmv_64(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int64_t                 n,
                                int64_t                 k,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* A,
                                int64_t                 lda,
                                const hipDoubleComplex* x,
                                int64_t                 incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int64_t                 incy)
try
{
    return hipblasConvertStatus(rocblas_zhbmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 k,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)A,
                                                 lda,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)beta,
                                                 (rocblas_double_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hbmv_batched
hipblasStatus_t hipblasChbmvBatched(hipblasHandle_t         handle,
                                    hipblasFillMode_t       uplo,
                                    int                     n,
                                    int                     k,
                                    const hipComplex*       alpha,
                                    const hipComplex* const A[],
                                    int                     lda,
                                    const hipComplex* const x[],
                                    int                     incx,
                                    const hipComplex*       beta,
                                    hipComplex* const       y[],
                                    int                     incy,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_chbmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      k,
                                                      (rocblas_float_complex*)alpha,
                                                      (rocblas_float_complex**)A,
                                                      lda,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      (rocblas_float_complex*)beta,
                                                      (rocblas_float_complex**)y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhbmvBatched(hipblasHandle_t               handle,
                                    hipblasFillMode_t             uplo,
                                    int                           n,
                                    int                           k,
                                    const hipDoubleComplex*       alpha,
                                    const hipDoubleComplex* const A[],
                                    int                           lda,
                                    const hipDoubleComplex* const x[],
                                    int                           incx,
                                    const hipDoubleComplex*       beta,
                                    hipDoubleComplex* const       y[],
                                    int                           incy,
                                    int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zhbmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      k,
                                                      (rocblas_double_complex*)alpha,
                                                      (rocblas_double_complex**)A,
                                                      lda,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      (rocblas_double_complex*)beta,
                                                      (rocblas_double_complex**)y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hbmv_batched_64
hipblasStatus_t hipblasChbmvBatched_64(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int64_t                 n,
                                       int64_t                 k,
                                       const hipComplex*       alpha,
                                       const hipComplex* const A[],
                                       int64_t                 lda,
                                       const hipComplex* const x[],
                                       int64_t                 incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int64_t                 incy,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_chbmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         k,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex**)A,
                                                         lda,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex*)beta,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhbmvBatched_64(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int64_t                       n,
                                       int64_t                       k,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const A[],
                                       int64_t                       lda,
                                       const hipDoubleComplex* const x[],
                                       int64_t                       incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int64_t                       incy,
                                       int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_zhbmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         k,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex**)A,
                                                         lda,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex*)beta,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hbmv_strided_batched
hipblasStatus_t hipblasChbmvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           int               k,
                                           const hipComplex* alpha,
                                           const hipComplex* A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           const hipComplex* x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const hipComplex* beta,
                                           hipComplex*       y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_chbmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              k,
                                                              (rocblas_float_complex*)alpha,
                                                              (rocblas_float_complex*)A,
                                                              lda,
                                                              strideA,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_float_complex*)beta,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhbmvStridedBatched(hipblasHandle_t         handle,
                                           hipblasFillMode_t       uplo,
                                           int                     n,
                                           int                     k,
                                           const hipDoubleComplex* alpha,
                                           const hipDoubleComplex* A,
                                           int                     lda,
                                           hipblasStride           strideA,
                                           const hipDoubleComplex* x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           const hipDoubleComplex* beta,
                                           hipDoubleComplex*       y,
                                           int                     incy,
                                           hipblasStride           stridey,
                                           int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zhbmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              k,
                                                              (rocblas_double_complex*)alpha,
                                                              (rocblas_double_complex*)A,
                                                              lda,
                                                              strideA,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_double_complex*)beta,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hbmv_strided_batched_64
hipblasStatus_t hipblasChbmvStridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              int64_t           k,
                                              const hipComplex* alpha,
                                              const hipComplex* A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              const hipComplex* x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const hipComplex* beta,
                                              hipComplex*       y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_chbmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 k,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)beta,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhbmvStridedBatched_64(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int64_t                 n,
                                              int64_t                 k,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* A,
                                              int64_t                 lda,
                                              hipblasStride           strideA,
                                              const hipDoubleComplex* x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int64_t                 incy,
                                              hipblasStride           stridey,
                                              int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zhbmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 k,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)beta,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hemv
hipblasStatus_t hipblasChemv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const hipComplex* alpha,
                             const hipComplex* A,
                             int               lda,
                             const hipComplex* x,
                             int               incx,
                             const hipComplex* beta,
                             hipComplex*       y,
                             int               incy)
try
{
    return hipblasConvertStatus(rocblas_chemv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              (rocblas_float_complex*)alpha,
                                              (rocblas_float_complex*)A,
                                              lda,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)beta,
                                              (rocblas_float_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhemv(hipblasHandle_t         handle,
                             hipblasFillMode_t       uplo,
                             int                     n,
                             const hipDoubleComplex* alpha,
                             const hipDoubleComplex* A,
                             int                     lda,
                             const hipDoubleComplex* x,
                             int                     incx,
                             const hipDoubleComplex* beta,
                             hipDoubleComplex*       y,
                             int                     incy)
try
{
    return hipblasConvertStatus(rocblas_zhemv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              (rocblas_double_complex*)alpha,
                                              (rocblas_double_complex*)A,
                                              lda,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)beta,
                                              (rocblas_double_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hemv_64
hipblasStatus_t hipblasChemv_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const hipComplex* alpha,
                                const hipComplex* A,
                                int64_t           lda,
                                const hipComplex* x,
                                int64_t           incx,
                                const hipComplex* beta,
                                hipComplex*       y,
                                int64_t           incy)
try
{
    return hipblasConvertStatus(rocblas_chemv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 (rocblas_float_complex*)alpha,
                                                 (rocblas_float_complex*)A,
                                                 lda,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)beta,
                                                 (rocblas_float_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhemv_64(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int64_t                 n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* A,
                                int64_t                 lda,
                                const hipDoubleComplex* x,
                                int64_t                 incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int64_t                 incy)
try
{
    return hipblasConvertStatus(rocblas_zhemv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)A,
                                                 lda,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)beta,
                                                 (rocblas_double_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hemv_batched
hipblasStatus_t hipblasChemvBatched(hipblasHandle_t         handle,
                                    hipblasFillMode_t       uplo,
                                    int                     n,
                                    const hipComplex*       alpha,
                                    const hipComplex* const A[],
                                    int                     lda,
                                    const hipComplex* const x[],
                                    int                     incx,
                                    const hipComplex*       beta,
                                    hipComplex* const       y[],
                                    int                     incy,
                                    int                     batch_count)
try
{
    return hipblasConvertStatus(rocblas_chemv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      (rocblas_float_complex*)alpha,
                                                      (rocblas_float_complex**)A,
                                                      lda,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      (rocblas_float_complex*)beta,
                                                      (rocblas_float_complex**)y,
                                                      incy,
                                                      batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhemvBatched(hipblasHandle_t               handle,
                                    hipblasFillMode_t             uplo,
                                    int                           n,
                                    const hipDoubleComplex*       alpha,
                                    const hipDoubleComplex* const A[],
                                    int                           lda,
                                    const hipDoubleComplex* const x[],
                                    int                           incx,
                                    const hipDoubleComplex*       beta,
                                    hipDoubleComplex* const       y[],
                                    int                           incy,
                                    int                           batch_count)
try
{
    return hipblasConvertStatus(rocblas_zhemv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      (rocblas_double_complex*)alpha,
                                                      (rocblas_double_complex**)A,
                                                      lda,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      (rocblas_double_complex*)beta,
                                                      (rocblas_double_complex**)y,
                                                      incy,
                                                      batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hemv_batched_64
hipblasStatus_t hipblasChemvBatched_64(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int64_t                 n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const A[],
                                       int64_t                 lda,
                                       const hipComplex* const x[],
                                       int64_t                 incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int64_t                 incy,
                                       int64_t                 batch_count)
try
{
    return hipblasConvertStatus(rocblas_chemv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex**)A,
                                                         lda,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex*)beta,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhemvBatched_64(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int64_t                       n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const A[],
                                       int64_t                       lda,
                                       const hipDoubleComplex* const x[],
                                       int64_t                       incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int64_t                       incy,
                                       int64_t                       batch_count)
try
{
    return hipblasConvertStatus(rocblas_zhemv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex**)A,
                                                         lda,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex*)beta,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hemv_strided_batched
hipblasStatus_t hipblasChemvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const hipComplex* alpha,
                                           const hipComplex* A,
                                           int               lda,
                                           hipblasStride     stride_a,
                                           const hipComplex* x,
                                           int               incx,
                                           hipblasStride     stride_x,
                                           const hipComplex* beta,
                                           hipComplex*       y,
                                           int               incy,
                                           hipblasStride     stride_y,
                                           int               batch_count)
try
{
    return hipblasConvertStatus(rocblas_chemv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              (rocblas_float_complex*)alpha,
                                                              (rocblas_float_complex*)A,
                                                              lda,
                                                              stride_a,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stride_x,
                                                              (rocblas_float_complex*)beta,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stride_y,
                                                              batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhemvStridedBatched(hipblasHandle_t         handle,
                                           hipblasFillMode_t       uplo,
                                           int                     n,
                                           const hipDoubleComplex* alpha,
                                           const hipDoubleComplex* A,
                                           int                     lda,
                                           hipblasStride           stride_a,
                                           const hipDoubleComplex* x,
                                           int                     incx,
                                           hipblasStride           stride_x,
                                           const hipDoubleComplex* beta,
                                           hipDoubleComplex*       y,
                                           int                     incy,
                                           hipblasStride           stride_y,
                                           int                     batch_count)
try
{
    return hipblasConvertStatus(rocblas_zhemv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              (rocblas_double_complex*)alpha,
                                                              (rocblas_double_complex*)A,
                                                              lda,
                                                              stride_a,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stride_x,
                                                              (rocblas_double_complex*)beta,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stride_y,
                                                              batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hemv_strided_batched_64
hipblasStatus_t hipblasChemvStridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const hipComplex* alpha,
                                              const hipComplex* A,
                                              int64_t           lda,
                                              hipblasStride     stride_a,
                                              const hipComplex* x,
                                              int64_t           incx,
                                              hipblasStride     stride_x,
                                              const hipComplex* beta,
                                              hipComplex*       y,
                                              int64_t           incy,
                                              hipblasStride     stride_y,
                                              int64_t           batch_count)
try
{
    return hipblasConvertStatus(rocblas_chemv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)A,
                                                                 lda,
                                                                 stride_a,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stride_x,
                                                                 (rocblas_float_complex*)beta,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stride_y,
                                                                 batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhemvStridedBatched_64(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int64_t                 n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* A,
                                              int64_t                 lda,
                                              hipblasStride           stride_a,
                                              const hipDoubleComplex* x,
                                              int64_t                 incx,
                                              hipblasStride           stride_x,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int64_t                 incy,
                                              hipblasStride           stride_y,
                                              int64_t                 batch_count)
try
{
    return hipblasConvertStatus(rocblas_zhemv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)A,
                                                                 lda,
                                                                 stride_a,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stride_x,
                                                                 (rocblas_double_complex*)beta,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stride_y,
                                                                 batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// her
hipblasStatus_t hipblasCher(hipblasHandle_t   handle,
                            hipblasFillMode_t uplo,
                            int               n,
                            const float*      alpha,
                            const hipComplex* x,
                            int               incx,
                            hipComplex*       A,
                            int               lda)
try
{
    return hipblasConvertStatus(rocblas_cher((rocblas_handle)handle,
                                             hipblasConvertFill(uplo),
                                             n,
                                             alpha,
                                             (rocblas_float_complex*)x,
                                             incx,
                                             (rocblas_float_complex*)A,
                                             lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZher(hipblasHandle_t         handle,
                            hipblasFillMode_t       uplo,
                            int                     n,
                            const double*           alpha,
                            const hipDoubleComplex* x,
                            int                     incx,
                            hipDoubleComplex*       A,
                            int                     lda)
try
{
    return hipblasConvertStatus(rocblas_zher((rocblas_handle)handle,
                                             hipblasConvertFill(uplo),
                                             n,
                                             alpha,
                                             (rocblas_double_complex*)x,
                                             incx,
                                             (rocblas_double_complex*)A,
                                             lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// her_64
hipblasStatus_t hipblasCher_64(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int64_t           n,
                               const float*      alpha,
                               const hipComplex* x,
                               int64_t           incx,
                               hipComplex*       A,
                               int64_t           lda)
try
{
    return hipblasConvertStatus(rocblas_cher_64((rocblas_handle)handle,
                                                hipblasConvertFill(uplo),
                                                n,
                                                alpha,
                                                (rocblas_float_complex*)x,
                                                incx,
                                                (rocblas_float_complex*)A,
                                                lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZher_64(hipblasHandle_t         handle,
                               hipblasFillMode_t       uplo,
                               int64_t                 n,
                               const double*           alpha,
                               const hipDoubleComplex* x,
                               int64_t                 incx,
                               hipDoubleComplex*       A,
                               int64_t                 lda)
try
{
    return hipblasConvertStatus(rocblas_zher_64((rocblas_handle)handle,
                                                hipblasConvertFill(uplo),
                                                n,
                                                alpha,
                                                (rocblas_double_complex*)x,
                                                incx,
                                                (rocblas_double_complex*)A,
                                                lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// her_batched
hipblasStatus_t hipblasCherBatched(hipblasHandle_t         handle,
                                   hipblasFillMode_t       uplo,
                                   int                     n,
                                   const float*            alpha,
                                   const hipComplex* const x[],
                                   int                     incx,
                                   hipComplex* const       A[],
                                   int                     lda,
                                   int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_cher_batched((rocblas_handle)handle,
                                                     hipblasConvertFill(uplo),
                                                     n,
                                                     alpha,
                                                     (rocblas_float_complex**)x,
                                                     incx,
                                                     (rocblas_float_complex**)A,
                                                     lda,
                                                     batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZherBatched(hipblasHandle_t               handle,
                                   hipblasFillMode_t             uplo,
                                   int                           n,
                                   const double*                 alpha,
                                   const hipDoubleComplex* const x[],
                                   int                           incx,
                                   hipDoubleComplex* const       A[],
                                   int                           lda,
                                   int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zher_batched((rocblas_handle)handle,
                                                     hipblasConvertFill(uplo),
                                                     n,
                                                     alpha,
                                                     (rocblas_double_complex**)x,
                                                     incx,
                                                     (rocblas_double_complex**)A,
                                                     lda,
                                                     batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// her_batched_64
hipblasStatus_t hipblasCherBatched_64(hipblasHandle_t         handle,
                                      hipblasFillMode_t       uplo,
                                      int64_t                 n,
                                      const float*            alpha,
                                      const hipComplex* const x[],
                                      int64_t                 incx,
                                      hipComplex* const       A[],
                                      int64_t                 lda,
                                      int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_cher_batched_64((rocblas_handle)handle,
                                                        hipblasConvertFill(uplo),
                                                        n,
                                                        alpha,
                                                        (rocblas_float_complex**)x,
                                                        incx,
                                                        (rocblas_float_complex**)A,
                                                        lda,
                                                        batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZherBatched_64(hipblasHandle_t               handle,
                                      hipblasFillMode_t             uplo,
                                      int64_t                       n,
                                      const double*                 alpha,
                                      const hipDoubleComplex* const x[],
                                      int64_t                       incx,
                                      hipDoubleComplex* const       A[],
                                      int64_t                       lda,
                                      int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_zher_batched_64((rocblas_handle)handle,
                                                        hipblasConvertFill(uplo),
                                                        n,
                                                        alpha,
                                                        (rocblas_double_complex**)x,
                                                        incx,
                                                        (rocblas_double_complex**)A,
                                                        lda,
                                                        batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// her_strided_batched
hipblasStatus_t hipblasCherStridedBatched(hipblasHandle_t   handle,
                                          hipblasFillMode_t uplo,
                                          int               n,
                                          const float*      alpha,
                                          const hipComplex* x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          hipComplex*       A,
                                          int               lda,
                                          hipblasStride     strideA,
                                          int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_cher_strided_batched((rocblas_handle)handle,
                                                             hipblasConvertFill(uplo),
                                                             n,
                                                             alpha,
                                                             (rocblas_float_complex*)x,
                                                             incx,
                                                             stridex,
                                                             (rocblas_float_complex*)A,
                                                             lda,
                                                             strideA,
                                                             batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZherStridedBatched(hipblasHandle_t         handle,
                                          hipblasFillMode_t       uplo,
                                          int                     n,
                                          const double*           alpha,
                                          const hipDoubleComplex* x,
                                          int                     incx,
                                          hipblasStride           stridex,
                                          hipDoubleComplex*       A,
                                          int                     lda,
                                          hipblasStride           strideA,
                                          int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zher_strided_batched((rocblas_handle)handle,
                                                             hipblasConvertFill(uplo),
                                                             n,
                                                             alpha,
                                                             (rocblas_double_complex*)x,
                                                             incx,
                                                             stridex,
                                                             (rocblas_double_complex*)A,
                                                             lda,
                                                             strideA,
                                                             batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// her_strided_batched_64
hipblasStatus_t hipblasCherStridedBatched_64(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int64_t           n,
                                             const float*      alpha,
                                             const hipComplex* x,
                                             int64_t           incx,
                                             hipblasStride     stridex,
                                             hipComplex*       A,
                                             int64_t           lda,
                                             hipblasStride     strideA,
                                             int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_cher_strided_batched_64((rocblas_handle)handle,
                                                                hipblasConvertFill(uplo),
                                                                n,
                                                                alpha,
                                                                (rocblas_float_complex*)x,
                                                                incx,
                                                                stridex,
                                                                (rocblas_float_complex*)A,
                                                                lda,
                                                                strideA,
                                                                batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZherStridedBatched_64(hipblasHandle_t         handle,
                                             hipblasFillMode_t       uplo,
                                             int64_t                 n,
                                             const double*           alpha,
                                             const hipDoubleComplex* x,
                                             int64_t                 incx,
                                             hipblasStride           stridex,
                                             hipDoubleComplex*       A,
                                             int64_t                 lda,
                                             hipblasStride           strideA,
                                             int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zher_strided_batched_64((rocblas_handle)handle,
                                                                hipblasConvertFill(uplo),
                                                                n,
                                                                alpha,
                                                                (rocblas_double_complex*)x,
                                                                incx,
                                                                stridex,
                                                                (rocblas_double_complex*)A,
                                                                lda,
                                                                strideA,
                                                                batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// her2
hipblasStatus_t hipblasCher2(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const hipComplex* alpha,
                             const hipComplex* x,
                             int               incx,
                             const hipComplex* y,
                             int               incy,
                             hipComplex*       A,
                             int               lda)
try
{
    return hipblasConvertStatus(rocblas_cher2((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              (rocblas_float_complex*)alpha,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)y,
                                              incy,
                                              (rocblas_float_complex*)A,
                                              lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZher2(hipblasHandle_t         handle,
                             hipblasFillMode_t       uplo,
                             int                     n,
                             const hipDoubleComplex* alpha,
                             const hipDoubleComplex* x,
                             int                     incx,
                             const hipDoubleComplex* y,
                             int                     incy,
                             hipDoubleComplex*       A,
                             int                     lda)
try
{
    return hipblasConvertStatus(rocblas_zher2((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              (rocblas_double_complex*)alpha,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)y,
                                              incy,
                                              (rocblas_double_complex*)A,
                                              lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// her2_64
hipblasStatus_t hipblasCher2_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int64_t           incx,
                                const hipComplex* y,
                                int64_t           incy,
                                hipComplex*       A,
                                int64_t           lda)
try
{
    return hipblasConvertStatus(rocblas_cher2_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 (rocblas_float_complex*)alpha,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)y,
                                                 incy,
                                                 (rocblas_float_complex*)A,
                                                 lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZher2_64(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int64_t                 n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int64_t                 incx,
                                const hipDoubleComplex* y,
                                int64_t                 incy,
                                hipDoubleComplex*       A,
                                int64_t                 lda)
try
{
    return hipblasConvertStatus(rocblas_zher2_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)y,
                                                 incy,
                                                 (rocblas_double_complex*)A,
                                                 lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// her2_batched
hipblasStatus_t hipblasCher2Batched(hipblasHandle_t         handle,
                                    hipblasFillMode_t       uplo,
                                    int                     n,
                                    const hipComplex*       alpha,
                                    const hipComplex* const x[],
                                    int                     incx,
                                    const hipComplex* const y[],
                                    int                     incy,
                                    hipComplex* const       A[],
                                    int                     lda,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_cher2_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      (rocblas_float_complex*)alpha,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      (rocblas_float_complex**)y,
                                                      incy,
                                                      (rocblas_float_complex**)A,
                                                      lda,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZher2Batched(hipblasHandle_t               handle,
                                    hipblasFillMode_t             uplo,
                                    int                           n,
                                    const hipDoubleComplex*       alpha,
                                    const hipDoubleComplex* const x[],
                                    int                           incx,
                                    const hipDoubleComplex* const y[],
                                    int                           incy,
                                    hipDoubleComplex* const       A[],
                                    int                           lda,
                                    int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zher2_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      (rocblas_double_complex*)alpha,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      (rocblas_double_complex**)y,
                                                      incy,
                                                      (rocblas_double_complex**)A,
                                                      lda,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// her2_batched_64
hipblasStatus_t hipblasCher2Batched_64(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int64_t                 n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int64_t                 incx,
                                       const hipComplex* const y[],
                                       int64_t                 incy,
                                       hipComplex* const       A[],
                                       int64_t                 lda,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_cher2_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         (rocblas_float_complex**)A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZher2Batched_64(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int64_t                       n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int64_t                       incx,
                                       const hipDoubleComplex* const y[],
                                       int64_t                       incy,
                                       hipDoubleComplex* const       A[],
                                       int64_t                       lda,
                                       int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_zher2_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         (rocblas_double_complex**)A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// her2_strided_batched
hipblasStatus_t hipblasCher2StridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const hipComplex* alpha,
                                           const hipComplex* x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const hipComplex* y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           hipComplex*       A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_cher2_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              (rocblas_float_complex*)alpha,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stridey,
                                                              (rocblas_float_complex*)A,
                                                              lda,
                                                              strideA,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZher2StridedBatched(hipblasHandle_t         handle,
                                           hipblasFillMode_t       uplo,
                                           int                     n,
                                           const hipDoubleComplex* alpha,
                                           const hipDoubleComplex* x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           const hipDoubleComplex* y,
                                           int                     incy,
                                           hipblasStride           stridey,
                                           hipDoubleComplex*       A,
                                           int                     lda,
                                           hipblasStride           strideA,
                                           int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zher2_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              (rocblas_double_complex*)alpha,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stridey,
                                                              (rocblas_double_complex*)A,
                                                              lda,
                                                              strideA,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// her2_strided_batched_64
hipblasStatus_t hipblasCher2StridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              hipComplex*       A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_cher2_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 (rocblas_float_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZher2StridedBatched_64(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int64_t                 n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int64_t                 incy,
                                              hipblasStride           stridey,
                                              hipDoubleComplex*       A,
                                              int64_t                 lda,
                                              hipblasStride           strideA,
                                              int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zher2_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 (rocblas_double_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpmv
hipblasStatus_t hipblasChpmv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const hipComplex* alpha,
                             const hipComplex* AP,
                             const hipComplex* x,
                             int               incx,
                             const hipComplex* beta,
                             hipComplex*       y,
                             int               incy)
try
{
    return hipblasConvertStatus(rocblas_chpmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              (rocblas_float_complex*)alpha,
                                              (rocblas_float_complex*)AP,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)beta,
                                              (rocblas_float_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpmv(hipblasHandle_t         handle,
                             hipblasFillMode_t       uplo,
                             int                     n,
                             const hipDoubleComplex* alpha,
                             const hipDoubleComplex* AP,
                             const hipDoubleComplex* x,
                             int                     incx,
                             const hipDoubleComplex* beta,
                             hipDoubleComplex*       y,
                             int                     incy)
try
{
    return hipblasConvertStatus(rocblas_zhpmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              (rocblas_double_complex*)alpha,
                                              (rocblas_double_complex*)AP,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)beta,
                                              (rocblas_double_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpmv_64
hipblasStatus_t hipblasChpmv_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const hipComplex* alpha,
                                const hipComplex* AP,
                                const hipComplex* x,
                                int64_t           incx,
                                const hipComplex* beta,
                                hipComplex*       y,
                                int64_t           incy)
try
{
    return hipblasConvertStatus(rocblas_chpmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 (rocblas_float_complex*)alpha,
                                                 (rocblas_float_complex*)AP,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)beta,
                                                 (rocblas_float_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpmv_64(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int64_t                 n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* AP,
                                const hipDoubleComplex* x,
                                int64_t                 incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int64_t                 incy)
try
{
    return hipblasConvertStatus(rocblas_zhpmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)AP,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)beta,
                                                 (rocblas_double_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpmv_batched
hipblasStatus_t hipblasChpmvBatched(hipblasHandle_t         handle,
                                    hipblasFillMode_t       uplo,
                                    int                     n,
                                    const hipComplex*       alpha,
                                    const hipComplex* const AP[],
                                    const hipComplex* const x[],
                                    int                     incx,
                                    const hipComplex*       beta,
                                    hipComplex* const       y[],
                                    int                     incy,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_chpmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      (rocblas_float_complex*)alpha,
                                                      (rocblas_float_complex**)AP,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      (rocblas_float_complex*)beta,
                                                      (rocblas_float_complex**)y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpmvBatched(hipblasHandle_t               handle,
                                    hipblasFillMode_t             uplo,
                                    int                           n,
                                    const hipDoubleComplex*       alpha,
                                    const hipDoubleComplex* const AP[],
                                    const hipDoubleComplex* const x[],
                                    int                           incx,
                                    const hipDoubleComplex*       beta,
                                    hipDoubleComplex* const       y[],
                                    int                           incy,
                                    int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zhpmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      (rocblas_double_complex*)alpha,
                                                      (rocblas_double_complex**)AP,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      (rocblas_double_complex*)beta,
                                                      (rocblas_double_complex**)y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpmv_batched_64
hipblasStatus_t hipblasChpmvBatched_64(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int64_t                 n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const AP[],
                                       const hipComplex* const x[],
                                       int64_t                 incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int64_t                 incy,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_chpmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex**)AP,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex*)beta,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpmvBatched_64(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int64_t                       n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const AP[],
                                       const hipDoubleComplex* const x[],
                                       int64_t                       incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int64_t                       incy,
                                       int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_zhpmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex**)AP,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex*)beta,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpmv_strided_batched
hipblasStatus_t hipblasChpmvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const hipComplex* alpha,
                                           const hipComplex* AP,
                                           hipblasStride     strideAP,
                                           const hipComplex* x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const hipComplex* beta,
                                           hipComplex*       y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_chpmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              (rocblas_float_complex*)alpha,
                                                              (rocblas_float_complex*)AP,
                                                              strideAP,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_float_complex*)beta,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpmvStridedBatched(hipblasHandle_t         handle,
                                           hipblasFillMode_t       uplo,
                                           int                     n,
                                           const hipDoubleComplex* alpha,
                                           const hipDoubleComplex* AP,
                                           hipblasStride           strideAP,
                                           const hipDoubleComplex* x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           const hipDoubleComplex* beta,
                                           hipDoubleComplex*       y,
                                           int                     incy,
                                           hipblasStride           stridey,
                                           int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zhpmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              (rocblas_double_complex*)alpha,
                                                              (rocblas_double_complex*)AP,
                                                              strideAP,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_double_complex*)beta,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpmv_strided_batched_64
hipblasStatus_t hipblasChpmvStridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const hipComplex* alpha,
                                              const hipComplex* AP,
                                              hipblasStride     strideAP,
                                              const hipComplex* x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const hipComplex* beta,
                                              hipComplex*       y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_chpmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)AP,
                                                                 strideAP,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)beta,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpmvStridedBatched_64(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int64_t                 n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* AP,
                                              hipblasStride           strideAP,
                                              const hipDoubleComplex* x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int64_t                 incy,
                                              hipblasStride           stridey,
                                              int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zhpmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)AP,
                                                                 strideAP,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)beta,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpr
hipblasStatus_t hipblasChpr(hipblasHandle_t   handle,
                            hipblasFillMode_t uplo,
                            int               n,
                            const float*      alpha,
                            const hipComplex* x,
                            int               incx,
                            hipComplex*       AP)
try
{
    return hipblasConvertStatus(rocblas_chpr((rocblas_handle)handle,
                                             hipblasConvertFill(uplo),
                                             n,
                                             alpha,
                                             (rocblas_float_complex*)x,
                                             incx,
                                             (rocblas_float_complex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpr(hipblasHandle_t         handle,
                            hipblasFillMode_t       uplo,
                            int                     n,
                            const double*           alpha,
                            const hipDoubleComplex* x,
                            int                     incx,
                            hipDoubleComplex*       AP)
try
{
    return hipblasConvertStatus(rocblas_zhpr((rocblas_handle)handle,
                                             hipblasConvertFill(uplo),
                                             n,
                                             alpha,
                                             (rocblas_double_complex*)x,
                                             incx,
                                             (rocblas_double_complex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpr_64
hipblasStatus_t hipblasChpr_64(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int64_t           n,
                               const float*      alpha,
                               const hipComplex* x,
                               int64_t           incx,
                               hipComplex*       AP)
try
{
    return hipblasConvertStatus(rocblas_chpr_64((rocblas_handle)handle,
                                                hipblasConvertFill(uplo),
                                                n,
                                                alpha,
                                                (rocblas_float_complex*)x,
                                                incx,
                                                (rocblas_float_complex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpr_64(hipblasHandle_t         handle,
                               hipblasFillMode_t       uplo,
                               int64_t                 n,
                               const double*           alpha,
                               const hipDoubleComplex* x,
                               int64_t                 incx,
                               hipDoubleComplex*       AP)
try
{
    return hipblasConvertStatus(rocblas_zhpr_64((rocblas_handle)handle,
                                                hipblasConvertFill(uplo),
                                                n,
                                                alpha,
                                                (rocblas_double_complex*)x,
                                                incx,
                                                (rocblas_double_complex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpr_batched
hipblasStatus_t hipblasChprBatched(hipblasHandle_t         handle,
                                   hipblasFillMode_t       uplo,
                                   int                     n,
                                   const float*            alpha,
                                   const hipComplex* const x[],
                                   int                     incx,
                                   hipComplex* const       AP[],
                                   int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_chpr_batched((rocblas_handle)handle,
                                                     hipblasConvertFill(uplo),
                                                     n,
                                                     alpha,
                                                     (rocblas_float_complex**)x,
                                                     incx,
                                                     (rocblas_float_complex**)AP,
                                                     batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhprBatched(hipblasHandle_t               handle,
                                   hipblasFillMode_t             uplo,
                                   int                           n,
                                   const double*                 alpha,
                                   const hipDoubleComplex* const x[],
                                   int                           incx,
                                   hipDoubleComplex* const       AP[],
                                   int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zhpr_batched((rocblas_handle)handle,
                                                     hipblasConvertFill(uplo),
                                                     n,
                                                     alpha,
                                                     (rocblas_double_complex**)x,
                                                     incx,
                                                     (rocblas_double_complex**)AP,
                                                     batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpr_batched_64
hipblasStatus_t hipblasChprBatched_64(hipblasHandle_t         handle,
                                      hipblasFillMode_t       uplo,
                                      int64_t                 n,
                                      const float*            alpha,
                                      const hipComplex* const x[],
                                      int64_t                 incx,
                                      hipComplex* const       AP[],
                                      int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_chpr_batched_64((rocblas_handle)handle,
                                                        hipblasConvertFill(uplo),
                                                        n,
                                                        alpha,
                                                        (rocblas_float_complex**)x,
                                                        incx,
                                                        (rocblas_float_complex**)AP,
                                                        batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhprBatched_64(hipblasHandle_t               handle,
                                      hipblasFillMode_t             uplo,
                                      int64_t                       n,
                                      const double*                 alpha,
                                      const hipDoubleComplex* const x[],
                                      int64_t                       incx,
                                      hipDoubleComplex* const       AP[],
                                      int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_zhpr_batched_64((rocblas_handle)handle,
                                                        hipblasConvertFill(uplo),
                                                        n,
                                                        alpha,
                                                        (rocblas_double_complex**)x,
                                                        incx,
                                                        (rocblas_double_complex**)AP,
                                                        batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpr_strided_batched
hipblasStatus_t hipblasChprStridedBatched(hipblasHandle_t   handle,
                                          hipblasFillMode_t uplo,
                                          int               n,
                                          const float*      alpha,
                                          const hipComplex* x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          hipComplex*       AP,
                                          hipblasStride     strideAP,
                                          int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_chpr_strided_batched((rocblas_handle)handle,
                                                             hipblasConvertFill(uplo),
                                                             n,
                                                             alpha,
                                                             (rocblas_float_complex*)x,
                                                             incx,
                                                             stridex,
                                                             (rocblas_float_complex*)AP,
                                                             strideAP,
                                                             batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhprStridedBatched(hipblasHandle_t         handle,
                                          hipblasFillMode_t       uplo,
                                          int                     n,
                                          const double*           alpha,
                                          const hipDoubleComplex* x,
                                          int                     incx,
                                          hipblasStride           stridex,
                                          hipDoubleComplex*       AP,
                                          hipblasStride           strideAP,
                                          int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zhpr_strided_batched((rocblas_handle)handle,
                                                             hipblasConvertFill(uplo),
                                                             n,
                                                             alpha,
                                                             (rocblas_double_complex*)x,
                                                             incx,
                                                             stridex,
                                                             (rocblas_double_complex*)AP,
                                                             strideAP,
                                                             batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpr_strided_batched_64
hipblasStatus_t hipblasChprStridedBatched_64(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int64_t           n,
                                             const float*      alpha,
                                             const hipComplex* x,
                                             int64_t           incx,
                                             hipblasStride     stridex,
                                             hipComplex*       AP,
                                             hipblasStride     strideAP,
                                             int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_chpr_strided_batched_64((rocblas_handle)handle,
                                                                hipblasConvertFill(uplo),
                                                                n,
                                                                alpha,
                                                                (rocblas_float_complex*)x,
                                                                incx,
                                                                stridex,
                                                                (rocblas_float_complex*)AP,
                                                                strideAP,
                                                                batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhprStridedBatched_64(hipblasHandle_t         handle,
                                             hipblasFillMode_t       uplo,
                                             int64_t                 n,
                                             const double*           alpha,
                                             const hipDoubleComplex* x,
                                             int64_t                 incx,
                                             hipblasStride           stridex,
                                             hipDoubleComplex*       AP,
                                             hipblasStride           strideAP,
                                             int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zhpr_strided_batched_64((rocblas_handle)handle,
                                                                hipblasConvertFill(uplo),
                                                                n,
                                                                alpha,
                                                                (rocblas_double_complex*)x,
                                                                incx,
                                                                stridex,
                                                                (rocblas_double_complex*)AP,
                                                                strideAP,
                                                                batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpr2
hipblasStatus_t hipblasChpr2(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const hipComplex* alpha,
                             const hipComplex* x,
                             int               incx,
                             const hipComplex* y,
                             int               incy,
                             hipComplex*       AP)
try
{
    return hipblasConvertStatus(rocblas_chpr2((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              (rocblas_float_complex*)alpha,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)y,
                                              incy,
                                              (rocblas_float_complex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpr2(hipblasHandle_t         handle,
                             hipblasFillMode_t       uplo,
                             int                     n,
                             const hipDoubleComplex* alpha,
                             const hipDoubleComplex* x,
                             int                     incx,
                             const hipDoubleComplex* y,
                             int                     incy,
                             hipDoubleComplex*       AP)
try
{
    return hipblasConvertStatus(rocblas_zhpr2((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              (rocblas_double_complex*)alpha,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)y,
                                              incy,
                                              (rocblas_double_complex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpr2_64
hipblasStatus_t hipblasChpr2_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int64_t           incx,
                                const hipComplex* y,
                                int64_t           incy,
                                hipComplex*       AP)
try
{
    return hipblasConvertStatus(rocblas_chpr2_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 (rocblas_float_complex*)alpha,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)y,
                                                 incy,
                                                 (rocblas_float_complex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpr2_64(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int64_t                 n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int64_t                 incx,
                                const hipDoubleComplex* y,
                                int64_t                 incy,
                                hipDoubleComplex*       AP)
try
{
    return hipblasConvertStatus(rocblas_zhpr2_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)y,
                                                 incy,
                                                 (rocblas_double_complex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpr2_batched
hipblasStatus_t hipblasChpr2Batched(hipblasHandle_t         handle,
                                    hipblasFillMode_t       uplo,
                                    int                     n,
                                    const hipComplex*       alpha,
                                    const hipComplex* const x[],
                                    int                     incx,
                                    const hipComplex* const y[],
                                    int                     incy,
                                    hipComplex* const       AP[],
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_chpr2_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      (rocblas_float_complex*)alpha,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      (rocblas_float_complex**)y,
                                                      incy,
                                                      (rocblas_float_complex**)AP,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpr2Batched(hipblasHandle_t               handle,
                                    hipblasFillMode_t             uplo,
                                    int                           n,
                                    const hipDoubleComplex*       alpha,
                                    const hipDoubleComplex* const x[],
                                    int                           incx,
                                    const hipDoubleComplex* const y[],
                                    int                           incy,
                                    hipDoubleComplex* const       AP[],
                                    int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zhpr2_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      (rocblas_double_complex*)alpha,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      (rocblas_double_complex**)y,
                                                      incy,
                                                      (rocblas_double_complex**)AP,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpr2_batched
hipblasStatus_t hipblasChpr2Batched_64(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int64_t                 n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int64_t                 incx,
                                       const hipComplex* const y[],
                                       int64_t                 incy,
                                       hipComplex* const       AP[],
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_chpr2_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         (rocblas_float_complex**)AP,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpr2Batched_64(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int64_t                       n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int64_t                       incx,
                                       const hipDoubleComplex* const y[],
                                       int64_t                       incy,
                                       hipDoubleComplex* const       AP[],
                                       int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_zhpr2_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         (rocblas_double_complex**)AP,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpr2_strided_batched
hipblasStatus_t hipblasChpr2StridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const hipComplex* alpha,
                                           const hipComplex* x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const hipComplex* y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           hipComplex*       AP,
                                           hipblasStride     strideAP,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_chpr2_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              (rocblas_float_complex*)alpha,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stridey,
                                                              (rocblas_float_complex*)AP,
                                                              strideAP,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpr2StridedBatched(hipblasHandle_t         handle,
                                           hipblasFillMode_t       uplo,
                                           int                     n,
                                           const hipDoubleComplex* alpha,
                                           const hipDoubleComplex* x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           const hipDoubleComplex* y,
                                           int                     incy,
                                           hipblasStride           stridey,
                                           hipDoubleComplex*       AP,
                                           hipblasStride           strideAP,
                                           int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zhpr2_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              (rocblas_double_complex*)alpha,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stridey,
                                                              (rocblas_double_complex*)AP,
                                                              strideAP,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// hpr2_strided_batched_64
hipblasStatus_t hipblasChpr2StridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              hipComplex*       AP,
                                              hipblasStride     strideAP,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_chpr2_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 (rocblas_float_complex*)AP,
                                                                 strideAP,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZhpr2StridedBatched_64(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int64_t                 n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int64_t                 incy,
                                              hipblasStride           stridey,
                                              hipDoubleComplex*       AP,
                                              hipblasStride           strideAP,
                                              int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zhpr2_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 (rocblas_double_complex*)AP,
                                                                 strideAP,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// sbmv
hipblasStatus_t hipblasSsbmv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             int               k,
                             const float*      alpha,
                             const float*      A,
                             int               lda,
                             const float*      x,
                             int               incx,
                             const float*      beta,
                             float*            y,
                             int               incy)
try
{
    return hipblasConvertStatus(rocblas_ssbmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              k,
                                              alpha,
                                              A,
                                              lda,
                                              x,
                                              incx,
                                              beta,
                                              y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsbmv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             int               k,
                             const double*     alpha,
                             const double*     A,
                             int               lda,
                             const double*     x,
                             int               incx,
                             const double*     beta,
                             double*           y,
                             int               incy)
try
{
    return hipblasConvertStatus(rocblas_dsbmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              k,
                                              alpha,
                                              A,
                                              lda,
                                              x,
                                              incx,
                                              beta,
                                              y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// sbmv_64
hipblasStatus_t hipblasSsbmv_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                int64_t           k,
                                const float*      alpha,
                                const float*      A,
                                int64_t           lda,
                                const float*      x,
                                int64_t           incx,
                                const float*      beta,
                                float*            y,
                                int64_t           incy)
try
{
    return hipblasConvertStatus(rocblas_ssbmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 k,
                                                 alpha,
                                                 A,
                                                 lda,
                                                 x,
                                                 incx,
                                                 beta,
                                                 y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsbmv_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                int64_t           k,
                                const double*     alpha,
                                const double*     A,
                                int64_t           lda,
                                const double*     x,
                                int64_t           incx,
                                const double*     beta,
                                double*           y,
                                int64_t           incy)
try
{
    return hipblasConvertStatus(rocblas_dsbmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 k,
                                                 alpha,
                                                 A,
                                                 lda,
                                                 x,
                                                 incx,
                                                 beta,
                                                 y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// sbmv_batched
hipblasStatus_t hipblasSsbmvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    int                n,
                                    int                k,
                                    const float*       alpha,
                                    const float* const A[],
                                    int                lda,
                                    const float* const x[],
                                    int                incx,
                                    const float*       beta,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_ssbmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      k,
                                                      alpha,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      beta,
                                                      y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsbmvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    int                 n,
                                    int                 k,
                                    const double*       alpha,
                                    const double* const A[],
                                    int                 lda,
                                    const double* const x[],
                                    int                 incx,
                                    const double*       beta,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_dsbmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      k,
                                                      alpha,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      beta,
                                                      y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// sbmv_batched_64
hipblasStatus_t hipblasSsbmvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       int64_t            n,
                                       int64_t            k,
                                       const float*       alpha,
                                       const float* const A[],
                                       int64_t            lda,
                                       const float* const x[],
                                       int64_t            incx,
                                       const float*       beta,
                                       float* const       y[],
                                       int64_t            incy,
                                       int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_ssbmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         k,
                                                         alpha,
                                                         A,
                                                         lda,
                                                         x,
                                                         incx,
                                                         beta,
                                                         y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsbmvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       int64_t             n,
                                       int64_t             k,
                                       const double*       alpha,
                                       const double* const A[],
                                       int64_t             lda,
                                       const double* const x[],
                                       int64_t             incx,
                                       const double*       beta,
                                       double* const       y[],
                                       int64_t             incy,
                                       int64_t             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dsbmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         k,
                                                         alpha,
                                                         A,
                                                         lda,
                                                         x,
                                                         incx,
                                                         beta,
                                                         y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// sbmv_strided_batched
hipblasStatus_t hipblasSsbmvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           int               k,
                                           const float*      alpha,
                                           const float*      A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           const float*      x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const float*      beta,
                                           float*            y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_ssbmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              k,
                                                              alpha,
                                                              A,
                                                              lda,
                                                              strideA,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              beta,
                                                              y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsbmvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           int               k,
                                           const double*     alpha,
                                           const double*     A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           const double*     x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const double*     beta,
                                           double*           y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_dsbmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              k,
                                                              alpha,
                                                              A,
                                                              lda,
                                                              strideA,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              beta,
                                                              y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// sbmv_strided_batched_64
hipblasStatus_t hipblasSsbmvStridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              int64_t           k,
                                              const float*      alpha,
                                              const float*      A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              const float*      x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const float*      beta,
                                              float*            y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_ssbmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 k,
                                                                 alpha,
                                                                 A,
                                                                 lda,
                                                                 strideA,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 beta,
                                                                 y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsbmvStridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              int64_t           k,
                                              const double*     alpha,
                                              const double*     A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              const double*     x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const double*     beta,
                                              double*           y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_dsbmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 k,
                                                                 alpha,
                                                                 A,
                                                                 lda,
                                                                 strideA,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 beta,
                                                                 y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spmv
hipblasStatus_t hipblasSspmv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const float*      alpha,
                             const float*      AP,
                             const float*      x,
                             int               incx,
                             const float*      beta,
                             float*            y,
                             int               incy)
try
{
    return hipblasConvertStatus(rocblas_sspmv(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, AP, x, incx, beta, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspmv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const double*     alpha,
                             const double*     AP,
                             const double*     x,
                             int               incx,
                             const double*     beta,
                             double*           y,
                             int               incy)
try
{
    return hipblasConvertStatus(rocblas_dspmv(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, AP, x, incx, beta, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spmv_64
hipblasStatus_t hipblasSspmv_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const float*      alpha,
                                const float*      AP,
                                const float*      x,
                                int64_t           incx,
                                const float*      beta,
                                float*            y,
                                int64_t           incy)
try
{
    return hipblasConvertStatus(rocblas_sspmv_64(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, AP, x, incx, beta, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspmv_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const double*     alpha,
                                const double*     AP,
                                const double*     x,
                                int64_t           incx,
                                const double*     beta,
                                double*           y,
                                int64_t           incy)
try
{
    return hipblasConvertStatus(rocblas_dspmv_64(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, AP, x, incx, beta, y, incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spmv_batched
hipblasStatus_t hipblasSspmvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    int                n,
                                    const float*       alpha,
                                    const float* const AP[],
                                    const float* const x[],
                                    int                incx,
                                    const float*       beta,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_sspmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      alpha,
                                                      AP,
                                                      x,
                                                      incx,
                                                      beta,
                                                      y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspmvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const AP[],
                                    const double* const x[],
                                    int                 incx,
                                    const double*       beta,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_dspmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      alpha,
                                                      AP,
                                                      x,
                                                      incx,
                                                      beta,
                                                      y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spmv_batched_64
hipblasStatus_t hipblasSspmvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       int64_t            n,
                                       const float*       alpha,
                                       const float* const AP[],
                                       const float* const x[],
                                       int64_t            incx,
                                       const float*       beta,
                                       float* const       y[],
                                       int64_t            incy,
                                       int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_sspmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         alpha,
                                                         AP,
                                                         x,
                                                         incx,
                                                         beta,
                                                         y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspmvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       int64_t             n,
                                       const double*       alpha,
                                       const double* const AP[],
                                       const double* const x[],
                                       int64_t             incx,
                                       const double*       beta,
                                       double* const       y[],
                                       int64_t             incy,
                                       int64_t             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dspmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         alpha,
                                                         AP,
                                                         x,
                                                         incx,
                                                         beta,
                                                         y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spmv_strided_batched
hipblasStatus_t hipblasSspmvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const float*      alpha,
                                           const float*      AP,
                                           hipblasStride     strideAP,
                                           const float*      x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const float*      beta,
                                           float*            y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_sspmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              alpha,
                                                              AP,
                                                              strideAP,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              beta,
                                                              y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspmvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const double*     alpha,
                                           const double*     AP,
                                           hipblasStride     strideAP,
                                           const double*     x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const double*     beta,
                                           double*           y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_dspmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              alpha,
                                                              AP,
                                                              strideAP,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              beta,
                                                              y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spmv_strided_batched_64
hipblasStatus_t hipblasSspmvStridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const float*      alpha,
                                              const float*      AP,
                                              hipblasStride     strideAP,
                                              const float*      x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const float*      beta,
                                              float*            y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_sspmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 AP,
                                                                 strideAP,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 beta,
                                                                 y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspmvStridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const double*     alpha,
                                              const double*     AP,
                                              hipblasStride     strideAP,
                                              const double*     x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const double*     beta,
                                              double*           y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_dspmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 AP,
                                                                 strideAP,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 beta,
                                                                 y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spr
hipblasStatus_t hipblasSspr(hipblasHandle_t   handle,
                            hipblasFillMode_t uplo,
                            int               n,
                            const float*      alpha,
                            const float*      x,
                            int               incx,
                            float*            AP)
try
{
    return hipblasConvertStatus(
        rocblas_sspr((rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspr(hipblasHandle_t   handle,
                            hipblasFillMode_t uplo,
                            int               n,
                            const double*     alpha,
                            const double*     x,
                            int               incx,
                            double*           AP)
try
{
    return hipblasConvertStatus(
        rocblas_dspr((rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCspr(hipblasHandle_t   handle,
                            hipblasFillMode_t uplo,
                            int               n,
                            const hipComplex* alpha,
                            const hipComplex* x,
                            int               incx,
                            hipComplex*       AP)
try
{
    return hipblasConvertStatus(rocblas_cspr((rocblas_handle)handle,
                                             hipblasConvertFill(uplo),
                                             n,
                                             (rocblas_float_complex*)alpha,
                                             (rocblas_float_complex*)x,
                                             incx,
                                             (rocblas_float_complex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZspr(hipblasHandle_t         handle,
                            hipblasFillMode_t       uplo,
                            int                     n,
                            const hipDoubleComplex* alpha,
                            const hipDoubleComplex* x,
                            int                     incx,
                            hipDoubleComplex*       AP)
try
{
    return hipblasConvertStatus(rocblas_zspr((rocblas_handle)handle,
                                             hipblasConvertFill(uplo),
                                             n,
                                             (rocblas_double_complex*)alpha,
                                             (rocblas_double_complex*)x,
                                             incx,
                                             (rocblas_double_complex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spr_64
hipblasStatus_t hipblasSspr_64(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int64_t           n,
                               const float*      alpha,
                               const float*      x,
                               int64_t           incx,
                               float*            AP)
try
{
    return hipblasConvertStatus(
        rocblas_sspr_64((rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspr_64(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int64_t           n,
                               const double*     alpha,
                               const double*     x,
                               int64_t           incx,
                               double*           AP)
try
{
    return hipblasConvertStatus(
        rocblas_dspr_64((rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCspr_64(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int64_t           n,
                               const hipComplex* alpha,
                               const hipComplex* x,
                               int64_t           incx,
                               hipComplex*       AP)
try
{
    return hipblasConvertStatus(rocblas_cspr_64((rocblas_handle)handle,
                                                hipblasConvertFill(uplo),
                                                n,
                                                (rocblas_float_complex*)alpha,
                                                (rocblas_float_complex*)x,
                                                incx,
                                                (rocblas_float_complex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZspr_64(hipblasHandle_t         handle,
                               hipblasFillMode_t       uplo,
                               int64_t                 n,
                               const hipDoubleComplex* alpha,
                               const hipDoubleComplex* x,
                               int64_t                 incx,
                               hipDoubleComplex*       AP)
try
{
    return hipblasConvertStatus(rocblas_zspr_64((rocblas_handle)handle,
                                                hipblasConvertFill(uplo),
                                                n,
                                                (rocblas_double_complex*)alpha,
                                                (rocblas_double_complex*)x,
                                                incx,
                                                (rocblas_double_complex*)AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spr_batched
hipblasStatus_t hipblasSsprBatched(hipblasHandle_t    handle,
                                   hipblasFillMode_t  uplo,
                                   int                n,
                                   const float*       alpha,
                                   const float* const x[],
                                   int                incx,
                                   float* const       AP[],
                                   int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_sspr_batched(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, AP, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsprBatched(hipblasHandle_t     handle,
                                   hipblasFillMode_t   uplo,
                                   int                 n,
                                   const double*       alpha,
                                   const double* const x[],
                                   int                 incx,
                                   double* const       AP[],
                                   int                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_dspr_batched(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, AP, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsprBatched(hipblasHandle_t         handle,
                                   hipblasFillMode_t       uplo,
                                   int                     n,
                                   const hipComplex*       alpha,
                                   const hipComplex* const x[],
                                   int                     incx,
                                   hipComplex* const       AP[],
                                   int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_cspr_batched((rocblas_handle)handle,
                                                     hipblasConvertFill(uplo),
                                                     n,
                                                     (rocblas_float_complex*)alpha,
                                                     (rocblas_float_complex**)x,
                                                     incx,
                                                     (rocblas_float_complex**)AP,
                                                     batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsprBatched(hipblasHandle_t               handle,
                                   hipblasFillMode_t             uplo,
                                   int                           n,
                                   const hipDoubleComplex*       alpha,
                                   const hipDoubleComplex* const x[],
                                   int                           incx,
                                   hipDoubleComplex* const       AP[],
                                   int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zspr_batched((rocblas_handle)handle,
                                                     hipblasConvertFill(uplo),
                                                     n,
                                                     (rocblas_double_complex*)alpha,
                                                     (rocblas_double_complex**)x,
                                                     incx,
                                                     (rocblas_double_complex**)AP,
                                                     batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spr_batched_64
hipblasStatus_t hipblasSsprBatched_64(hipblasHandle_t    handle,
                                      hipblasFillMode_t  uplo,
                                      int64_t            n,
                                      const float*       alpha,
                                      const float* const x[],
                                      int64_t            incx,
                                      float* const       AP[],
                                      int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_sspr_batched_64(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, AP, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsprBatched_64(hipblasHandle_t     handle,
                                      hipblasFillMode_t   uplo,
                                      int64_t             n,
                                      const double*       alpha,
                                      const double* const x[],
                                      int64_t             incx,
                                      double* const       AP[],
                                      int64_t             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dspr_batched_64(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, AP, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsprBatched_64(hipblasHandle_t         handle,
                                      hipblasFillMode_t       uplo,
                                      int64_t                 n,
                                      const hipComplex*       alpha,
                                      const hipComplex* const x[],
                                      int64_t                 incx,
                                      hipComplex* const       AP[],
                                      int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_cspr_batched_64((rocblas_handle)handle,
                                                        hipblasConvertFill(uplo),
                                                        n,
                                                        (rocblas_float_complex*)alpha,
                                                        (rocblas_float_complex**)x,
                                                        incx,
                                                        (rocblas_float_complex**)AP,
                                                        batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsprBatched_64(hipblasHandle_t               handle,
                                      hipblasFillMode_t             uplo,
                                      int64_t                       n,
                                      const hipDoubleComplex*       alpha,
                                      const hipDoubleComplex* const x[],
                                      int64_t                       incx,
                                      hipDoubleComplex* const       AP[],
                                      int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_zspr_batched_64((rocblas_handle)handle,
                                                        hipblasConvertFill(uplo),
                                                        n,
                                                        (rocblas_double_complex*)alpha,
                                                        (rocblas_double_complex**)x,
                                                        incx,
                                                        (rocblas_double_complex**)AP,
                                                        batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spr_strided_batched
hipblasStatus_t hipblasSsprStridedBatched(hipblasHandle_t   handle,
                                          hipblasFillMode_t uplo,
                                          int               n,
                                          const float*      alpha,
                                          const float*      x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          float*            AP,
                                          hipblasStride     strideAP,
                                          int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_sspr_strided_batched((rocblas_handle)handle,
                                                             hipblasConvertFill(uplo),
                                                             n,
                                                             alpha,
                                                             x,
                                                             incx,
                                                             stridex,
                                                             AP,
                                                             strideAP,
                                                             batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsprStridedBatched(hipblasHandle_t   handle,
                                          hipblasFillMode_t uplo,
                                          int               n,
                                          const double*     alpha,
                                          const double*     x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          double*           AP,
                                          hipblasStride     strideAP,
                                          int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_dspr_strided_batched((rocblas_handle)handle,
                                                             hipblasConvertFill(uplo),
                                                             n,
                                                             alpha,
                                                             x,
                                                             incx,
                                                             stridex,
                                                             AP,
                                                             strideAP,
                                                             batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsprStridedBatched(hipblasHandle_t   handle,
                                          hipblasFillMode_t uplo,
                                          int               n,
                                          const hipComplex* alpha,
                                          const hipComplex* x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          hipComplex*       AP,
                                          hipblasStride     strideAP,
                                          int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_cspr_strided_batched((rocblas_handle)handle,
                                                             hipblasConvertFill(uplo),
                                                             n,
                                                             (rocblas_float_complex*)alpha,
                                                             (rocblas_float_complex*)x,
                                                             incx,
                                                             stridex,
                                                             (rocblas_float_complex*)AP,
                                                             strideAP,
                                                             batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsprStridedBatched(hipblasHandle_t         handle,
                                          hipblasFillMode_t       uplo,
                                          int                     n,
                                          const hipDoubleComplex* alpha,
                                          const hipDoubleComplex* x,
                                          int                     incx,
                                          hipblasStride           stridex,
                                          hipDoubleComplex*       AP,
                                          hipblasStride           strideAP,
                                          int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zspr_strided_batched((rocblas_handle)handle,
                                                             hipblasConvertFill(uplo),
                                                             n,
                                                             (rocblas_double_complex*)alpha,
                                                             (rocblas_double_complex*)x,
                                                             incx,
                                                             stridex,
                                                             (rocblas_double_complex*)AP,
                                                             strideAP,
                                                             batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spr_strided_batched_64
hipblasStatus_t hipblasSsprStridedBatched_64(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int64_t           n,
                                             const float*      alpha,
                                             const float*      x,
                                             int64_t           incx,
                                             hipblasStride     stridex,
                                             float*            AP,
                                             hipblasStride     strideAP,
                                             int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_sspr_strided_batched_64((rocblas_handle)handle,
                                                                hipblasConvertFill(uplo),
                                                                n,
                                                                alpha,
                                                                x,
                                                                incx,
                                                                stridex,
                                                                AP,
                                                                strideAP,
                                                                batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsprStridedBatched_64(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int64_t           n,
                                             const double*     alpha,
                                             const double*     x,
                                             int64_t           incx,
                                             hipblasStride     stridex,
                                             double*           AP,
                                             hipblasStride     strideAP,
                                             int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_dspr_strided_batched_64((rocblas_handle)handle,
                                                                hipblasConvertFill(uplo),
                                                                n,
                                                                alpha,
                                                                x,
                                                                incx,
                                                                stridex,
                                                                AP,
                                                                strideAP,
                                                                batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsprStridedBatched_64(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int64_t           n,
                                             const hipComplex* alpha,
                                             const hipComplex* x,
                                             int64_t           incx,
                                             hipblasStride     stridex,
                                             hipComplex*       AP,
                                             hipblasStride     strideAP,
                                             int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_cspr_strided_batched_64((rocblas_handle)handle,
                                                                hipblasConvertFill(uplo),
                                                                n,
                                                                (rocblas_float_complex*)alpha,
                                                                (rocblas_float_complex*)x,
                                                                incx,
                                                                stridex,
                                                                (rocblas_float_complex*)AP,
                                                                strideAP,
                                                                batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsprStridedBatched_64(hipblasHandle_t         handle,
                                             hipblasFillMode_t       uplo,
                                             int64_t                 n,
                                             const hipDoubleComplex* alpha,
                                             const hipDoubleComplex* x,
                                             int64_t                 incx,
                                             hipblasStride           stridex,
                                             hipDoubleComplex*       AP,
                                             hipblasStride           strideAP,
                                             int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zspr_strided_batched_64((rocblas_handle)handle,
                                                                hipblasConvertFill(uplo),
                                                                n,
                                                                (rocblas_double_complex*)alpha,
                                                                (rocblas_double_complex*)x,
                                                                incx,
                                                                stridex,
                                                                (rocblas_double_complex*)AP,
                                                                strideAP,
                                                                batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spr2
hipblasStatus_t hipblasSspr2(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const float*      alpha,
                             const float*      x,
                             int               incx,
                             const float*      y,
                             int               incy,
                             float*            AP)
try
{
    return hipblasConvertStatus(rocblas_sspr2(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, y, incy, AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspr2(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const double*     alpha,
                             const double*     x,
                             int               incx,
                             const double*     y,
                             int               incy,
                             double*           AP)
try
{
    return hipblasConvertStatus(rocblas_dspr2(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, y, incy, AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spr2_64
hipblasStatus_t hipblasSspr2_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const float*      alpha,
                                const float*      x,
                                int64_t           incx,
                                const float*      y,
                                int64_t           incy,
                                float*            AP)
try
{
    return hipblasConvertStatus(rocblas_sspr2_64(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, y, incy, AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspr2_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const double*     alpha,
                                const double*     x,
                                int64_t           incx,
                                const double*     y,
                                int64_t           incy,
                                double*           AP)
try
{
    return hipblasConvertStatus(rocblas_dspr2_64(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, y, incy, AP));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spr2_batched
hipblasStatus_t hipblasSspr2Batched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    int                n,
                                    const float*       alpha,
                                    const float* const x[],
                                    int                incx,
                                    const float* const y[],
                                    int                incy,
                                    float* const       AP[],
                                    int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_sspr2_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      alpha,
                                                      x,
                                                      incx,
                                                      y,
                                                      incy,
                                                      AP,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspr2Batched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const x[],
                                    int                 incx,
                                    const double* const y[],
                                    int                 incy,
                                    double* const       AP[],
                                    int                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_dspr2_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      alpha,
                                                      x,
                                                      incx,
                                                      y,
                                                      incy,
                                                      AP,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spr2_batched_64
hipblasStatus_t hipblasSspr2Batched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       int64_t            n,
                                       const float*       alpha,
                                       const float* const x[],
                                       int64_t            incx,
                                       const float* const y[],
                                       int64_t            incy,
                                       float* const       AP[],
                                       int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_sspr2_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         alpha,
                                                         x,
                                                         incx,
                                                         y,
                                                         incy,
                                                         AP,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspr2Batched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       int64_t             n,
                                       const double*       alpha,
                                       const double* const x[],
                                       int64_t             incx,
                                       const double* const y[],
                                       int64_t             incy,
                                       double* const       AP[],
                                       int64_t             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dspr2_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         alpha,
                                                         x,
                                                         incx,
                                                         y,
                                                         incy,
                                                         AP,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spr2_strided_batched
hipblasStatus_t hipblasSspr2StridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const float*      alpha,
                                           const float*      x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const float*      y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           float*            AP,
                                           hipblasStride     strideAP,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_sspr2_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              alpha,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              y,
                                                              incy,
                                                              stridey,
                                                              AP,
                                                              strideAP,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspr2StridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const double*     alpha,
                                           const double*     x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const double*     y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           double*           AP,
                                           hipblasStride     strideAP,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_dspr2_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              alpha,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              y,
                                                              incy,
                                                              stridey,
                                                              AP,
                                                              strideAP,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// spr2_strided_batched_64
hipblasStatus_t hipblasSspr2StridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const float*      alpha,
                                              const float*      x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const float*      y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              float*            AP,
                                              hipblasStride     strideAP,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_sspr2_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 y,
                                                                 incy,
                                                                 stridey,
                                                                 AP,
                                                                 strideAP,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDspr2StridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const double*     alpha,
                                              const double*     x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const double*     y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              double*           AP,
                                              hipblasStride     strideAP,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_dspr2_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 y,
                                                                 incy,
                                                                 stridey,
                                                                 AP,
                                                                 strideAP,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// symv
hipblasStatus_t hipblasSsymv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const float*      alpha,
                             const float*      A,
                             int               lda,
                             const float*      x,
                             int               incx,
                             const float*      beta,
                             float*            y,
                             int               incy)
try
{
    return hipblasConvertStatus(rocblas_ssymv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              alpha,
                                              A,
                                              lda,
                                              x,
                                              incx,
                                              beta,
                                              y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsymv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const double*     alpha,
                             const double*     A,
                             int               lda,
                             const double*     x,
                             int               incx,
                             const double*     beta,
                             double*           y,
                             int               incy)
try
{
    return hipblasConvertStatus(rocblas_dsymv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              alpha,
                                              A,
                                              lda,
                                              x,
                                              incx,
                                              beta,
                                              y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsymv(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const hipComplex* alpha,
                             const hipComplex* A,
                             int               lda,
                             const hipComplex* x,
                             int               incx,
                             const hipComplex* beta,
                             hipComplex*       y,
                             int               incy)
try
{
    return hipblasConvertStatus(rocblas_csymv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              (rocblas_float_complex*)alpha,
                                              (rocblas_float_complex*)A,
                                              lda,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)beta,
                                              (rocblas_float_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsymv(hipblasHandle_t         handle,
                             hipblasFillMode_t       uplo,
                             int                     n,
                             const hipDoubleComplex* alpha,
                             const hipDoubleComplex* A,
                             int                     lda,
                             const hipDoubleComplex* x,
                             int                     incx,
                             const hipDoubleComplex* beta,
                             hipDoubleComplex*       y,
                             int                     incy)
try
{
    return hipblasConvertStatus(rocblas_zsymv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              (rocblas_double_complex*)alpha,
                                              (rocblas_double_complex*)A,
                                              lda,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)beta,
                                              (rocblas_double_complex*)y,
                                              incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// symv_64
hipblasStatus_t hipblasSsymv_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const float*      alpha,
                                const float*      A,
                                int64_t           lda,
                                const float*      x,
                                int64_t           incx,
                                const float*      beta,
                                float*            y,
                                int64_t           incy)
try
{
    return hipblasConvertStatus(rocblas_ssymv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 alpha,
                                                 A,
                                                 lda,
                                                 x,
                                                 incx,
                                                 beta,
                                                 y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsymv_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const double*     alpha,
                                const double*     A,
                                int64_t           lda,
                                const double*     x,
                                int64_t           incx,
                                const double*     beta,
                                double*           y,
                                int64_t           incy)
try
{
    return hipblasConvertStatus(rocblas_dsymv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 alpha,
                                                 A,
                                                 lda,
                                                 x,
                                                 incx,
                                                 beta,
                                                 y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsymv_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const hipComplex* alpha,
                                const hipComplex* A,
                                int64_t           lda,
                                const hipComplex* x,
                                int64_t           incx,
                                const hipComplex* beta,
                                hipComplex*       y,
                                int64_t           incy)
try
{
    return hipblasConvertStatus(rocblas_csymv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 (rocblas_float_complex*)alpha,
                                                 (rocblas_float_complex*)A,
                                                 lda,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)beta,
                                                 (rocblas_float_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsymv_64(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int64_t                 n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* A,
                                int64_t                 lda,
                                const hipDoubleComplex* x,
                                int64_t                 incx,
                                const hipDoubleComplex* beta,
                                hipDoubleComplex*       y,
                                int64_t                 incy)
try
{
    return hipblasConvertStatus(rocblas_zsymv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)A,
                                                 lda,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)beta,
                                                 (rocblas_double_complex*)y,
                                                 incy));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// symv_batched
hipblasStatus_t hipblasSsymvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    int                n,
                                    const float*       alpha,
                                    const float* const A[],
                                    int                lda,
                                    const float* const x[],
                                    int                incx,
                                    const float*       beta,
                                    float* const       y[],
                                    int                incy,
                                    int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_ssymv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      alpha,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      beta,
                                                      y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsymvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const A[],
                                    int                 lda,
                                    const double* const x[],
                                    int                 incx,
                                    const double*       beta,
                                    double* const       y[],
                                    int                 incy,
                                    int                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_dsymv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      alpha,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      beta,
                                                      y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsymvBatched(hipblasHandle_t         handle,
                                    hipblasFillMode_t       uplo,
                                    int                     n,
                                    const hipComplex*       alpha,
                                    const hipComplex* const A[],
                                    int                     lda,
                                    const hipComplex* const x[],
                                    int                     incx,
                                    const hipComplex*       beta,
                                    hipComplex* const       y[],
                                    int                     incy,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_csymv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      (rocblas_float_complex*)alpha,
                                                      (rocblas_float_complex**)A,
                                                      lda,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      (rocblas_float_complex*)beta,
                                                      (rocblas_float_complex**)y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsymvBatched(hipblasHandle_t               handle,
                                    hipblasFillMode_t             uplo,
                                    int                           n,
                                    const hipDoubleComplex*       alpha,
                                    const hipDoubleComplex* const A[],
                                    int                           lda,
                                    const hipDoubleComplex* const x[],
                                    int                           incx,
                                    const hipDoubleComplex*       beta,
                                    hipDoubleComplex* const       y[],
                                    int                           incy,
                                    int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zsymv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      (rocblas_double_complex*)alpha,
                                                      (rocblas_double_complex**)A,
                                                      lda,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      (rocblas_double_complex*)beta,
                                                      (rocblas_double_complex**)y,
                                                      incy,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// symv_batched_64
hipblasStatus_t hipblasSsymvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       int64_t            n,
                                       const float*       alpha,
                                       const float* const A[],
                                       int64_t            lda,
                                       const float* const x[],
                                       int64_t            incx,
                                       const float*       beta,
                                       float* const       y[],
                                       int64_t            incy,
                                       int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_ssymv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         alpha,
                                                         A,
                                                         lda,
                                                         x,
                                                         incx,
                                                         beta,
                                                         y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsymvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       int64_t             n,
                                       const double*       alpha,
                                       const double* const A[],
                                       int64_t             lda,
                                       const double* const x[],
                                       int64_t             incx,
                                       const double*       beta,
                                       double* const       y[],
                                       int64_t             incy,
                                       int64_t             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dsymv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         alpha,
                                                         A,
                                                         lda,
                                                         x,
                                                         incx,
                                                         beta,
                                                         y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsymvBatched_64(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int64_t                 n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const A[],
                                       int64_t                 lda,
                                       const hipComplex* const x[],
                                       int64_t                 incx,
                                       const hipComplex*       beta,
                                       hipComplex* const       y[],
                                       int64_t                 incy,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_csymv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex**)A,
                                                         lda,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex*)beta,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsymvBatched_64(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int64_t                       n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const A[],
                                       int64_t                       lda,
                                       const hipDoubleComplex* const x[],
                                       int64_t                       incx,
                                       const hipDoubleComplex*       beta,
                                       hipDoubleComplex* const       y[],
                                       int64_t                       incy,
                                       int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_zsymv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex**)A,
                                                         lda,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex*)beta,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// symv_strided_batched
hipblasStatus_t hipblasSsymvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const float*      alpha,
                                           const float*      A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           const float*      x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const float*      beta,
                                           float*            y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_ssymv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              alpha,
                                                              A,
                                                              lda,
                                                              strideA,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              beta,
                                                              y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsymvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const double*     alpha,
                                           const double*     A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           const double*     x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const double*     beta,
                                           double*           y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_dsymv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              alpha,
                                                              A,
                                                              lda,
                                                              strideA,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              beta,
                                                              y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsymvStridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const hipComplex* alpha,
                                           const hipComplex* A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           const hipComplex* x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const hipComplex* beta,
                                           hipComplex*       y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_csymv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              (rocblas_float_complex*)alpha,
                                                              (rocblas_float_complex*)A,
                                                              lda,
                                                              strideA,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_float_complex*)beta,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsymvStridedBatched(hipblasHandle_t         handle,
                                           hipblasFillMode_t       uplo,
                                           int                     n,
                                           const hipDoubleComplex* alpha,
                                           const hipDoubleComplex* A,
                                           int                     lda,
                                           hipblasStride           strideA,
                                           const hipDoubleComplex* x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           const hipDoubleComplex* beta,
                                           hipDoubleComplex*       y,
                                           int                     incy,
                                           hipblasStride           stridey,
                                           int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zsymv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              (rocblas_double_complex*)alpha,
                                                              (rocblas_double_complex*)A,
                                                              lda,
                                                              strideA,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_double_complex*)beta,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stridey,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// symv_strided_batched_64
hipblasStatus_t hipblasSsymvStridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const float*      alpha,
                                              const float*      A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              const float*      x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const float*      beta,
                                              float*            y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_ssymv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 A,
                                                                 lda,
                                                                 strideA,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 beta,
                                                                 y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsymvStridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const double*     alpha,
                                              const double*     A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              const double*     x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const double*     beta,
                                              double*           y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_dsymv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 A,
                                                                 lda,
                                                                 strideA,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 beta,
                                                                 y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsymvStridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const hipComplex* alpha,
                                              const hipComplex* A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              const hipComplex* x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const hipComplex* beta,
                                              hipComplex*       y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_csymv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)beta,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsymvStridedBatched_64(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int64_t                 n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* A,
                                              int64_t                 lda,
                                              hipblasStride           strideA,
                                              const hipDoubleComplex* x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* beta,
                                              hipDoubleComplex*       y,
                                              int64_t                 incy,
                                              hipblasStride           stridey,
                                              int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zsymv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)beta,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// syr
hipblasStatus_t hipblasSsyr(hipblasHandle_t   handle,
                            hipblasFillMode_t uplo,
                            int               n,
                            const float*      alpha,
                            const float*      x,
                            int               incx,
                            float*            A,
                            int               lda)
try
{
    return hipblasConvertStatus(
        rocblas_ssyr((rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsyr(hipblasHandle_t   handle,
                            hipblasFillMode_t uplo,
                            int               n,
                            const double*     alpha,
                            const double*     x,
                            int               incx,
                            double*           A,
                            int               lda)
try
{
    return hipblasConvertStatus(
        rocblas_dsyr((rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyr(hipblasHandle_t   handle,
                            hipblasFillMode_t uplo,
                            int               n,
                            const hipComplex* alpha,
                            const hipComplex* x,
                            int               incx,
                            hipComplex*       A,
                            int               lda)
try
{
    return hipblasConvertStatus(rocblas_csyr((rocblas_handle)handle,
                                             hipblasConvertFill(uplo),
                                             n,
                                             (rocblas_float_complex*)alpha,
                                             (rocblas_float_complex*)x,
                                             incx,
                                             (rocblas_float_complex*)A,
                                             lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyr(hipblasHandle_t         handle,
                            hipblasFillMode_t       uplo,
                            int                     n,
                            const hipDoubleComplex* alpha,
                            const hipDoubleComplex* x,
                            int                     incx,
                            hipDoubleComplex*       A,
                            int                     lda)
try
{
    return hipblasConvertStatus(rocblas_zsyr((rocblas_handle)handle,
                                             hipblasConvertFill(uplo),
                                             n,
                                             (rocblas_double_complex*)alpha,
                                             (rocblas_double_complex*)x,
                                             incx,
                                             (rocblas_double_complex*)A,
                                             lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// syr_64
hipblasStatus_t hipblasSsyr_64(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int64_t           n,
                               const float*      alpha,
                               const float*      x,
                               int64_t           incx,
                               float*            A,
                               int64_t           lda)
try
{
    return hipblasConvertStatus(rocblas_ssyr_64(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsyr_64(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int64_t           n,
                               const double*     alpha,
                               const double*     x,
                               int64_t           incx,
                               double*           A,
                               int64_t           lda)
try
{
    return hipblasConvertStatus(rocblas_dsyr_64(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyr_64(hipblasHandle_t   handle,
                               hipblasFillMode_t uplo,
                               int64_t           n,
                               const hipComplex* alpha,
                               const hipComplex* x,
                               int64_t           incx,
                               hipComplex*       A,
                               int64_t           lda)
try
{
    return hipblasConvertStatus(rocblas_csyr_64((rocblas_handle)handle,
                                                hipblasConvertFill(uplo),
                                                n,
                                                (rocblas_float_complex*)alpha,
                                                (rocblas_float_complex*)x,
                                                incx,
                                                (rocblas_float_complex*)A,
                                                lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyr_64(hipblasHandle_t         handle,
                               hipblasFillMode_t       uplo,
                               int64_t                 n,
                               const hipDoubleComplex* alpha,
                               const hipDoubleComplex* x,
                               int64_t                 incx,
                               hipDoubleComplex*       A,
                               int64_t                 lda)
try
{
    return hipblasConvertStatus(rocblas_zsyr_64((rocblas_handle)handle,
                                                hipblasConvertFill(uplo),
                                                n,
                                                (rocblas_double_complex*)alpha,
                                                (rocblas_double_complex*)x,
                                                incx,
                                                (rocblas_double_complex*)A,
                                                lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// syr_batched
hipblasStatus_t hipblasSsyrBatched(hipblasHandle_t    handle,
                                   hipblasFillMode_t  uplo,
                                   int                n,
                                   const float*       alpha,
                                   const float* const x[],
                                   int                incx,
                                   float* const       A[],
                                   int                lda,
                                   int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_ssyr_batched(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, A, lda, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsyrBatched(hipblasHandle_t     handle,
                                   hipblasFillMode_t   uplo,
                                   int                 n,
                                   const double*       alpha,
                                   const double* const x[],
                                   int                 incx,
                                   double* const       A[],
                                   int                 lda,
                                   int                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_dsyr_batched(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, A, lda, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyrBatched(hipblasHandle_t         handle,
                                   hipblasFillMode_t       uplo,
                                   int                     n,
                                   const hipComplex*       alpha,
                                   const hipComplex* const x[],
                                   int                     incx,
                                   hipComplex* const       A[],
                                   int                     lda,
                                   int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_csyr_batched((rocblas_handle)handle,
                                                     hipblasConvertFill(uplo),
                                                     n,
                                                     (rocblas_float_complex*)alpha,
                                                     (rocblas_float_complex**)x,
                                                     incx,
                                                     (rocblas_float_complex**)A,
                                                     lda,
                                                     batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyrBatched(hipblasHandle_t               handle,
                                   hipblasFillMode_t             uplo,
                                   int                           n,
                                   const hipDoubleComplex*       alpha,
                                   const hipDoubleComplex* const x[],
                                   int                           incx,
                                   hipDoubleComplex* const       A[],
                                   int                           lda,
                                   int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zsyr_batched((rocblas_handle)handle,
                                                     hipblasConvertFill(uplo),
                                                     n,
                                                     (rocblas_double_complex*)alpha,
                                                     (rocblas_double_complex**)x,
                                                     incx,
                                                     (rocblas_double_complex**)A,
                                                     lda,
                                                     batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// syr_batched_64
hipblasStatus_t hipblasSsyrBatched_64(hipblasHandle_t    handle,
                                      hipblasFillMode_t  uplo,
                                      int64_t            n,
                                      const float*       alpha,
                                      const float* const x[],
                                      int64_t            incx,
                                      float* const       A[],
                                      int64_t            lda,
                                      int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_ssyr_batched_64(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, A, lda, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsyrBatched_64(hipblasHandle_t     handle,
                                      hipblasFillMode_t   uplo,
                                      int64_t             n,
                                      const double*       alpha,
                                      const double* const x[],
                                      int64_t             incx,
                                      double* const       A[],
                                      int64_t             lda,
                                      int64_t             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dsyr_batched_64(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, A, lda, batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyrBatched_64(hipblasHandle_t         handle,
                                      hipblasFillMode_t       uplo,
                                      int64_t                 n,
                                      const hipComplex*       alpha,
                                      const hipComplex* const x[],
                                      int64_t                 incx,
                                      hipComplex* const       A[],
                                      int64_t                 lda,
                                      int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_csyr_batched_64((rocblas_handle)handle,
                                                        hipblasConvertFill(uplo),
                                                        n,
                                                        (rocblas_float_complex*)alpha,
                                                        (rocblas_float_complex**)x,
                                                        incx,
                                                        (rocblas_float_complex**)A,
                                                        lda,
                                                        batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyrBatched_64(hipblasHandle_t               handle,
                                      hipblasFillMode_t             uplo,
                                      int64_t                       n,
                                      const hipDoubleComplex*       alpha,
                                      const hipDoubleComplex* const x[],
                                      int64_t                       incx,
                                      hipDoubleComplex* const       A[],
                                      int64_t                       lda,
                                      int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_zsyr_batched_64((rocblas_handle)handle,
                                                        hipblasConvertFill(uplo),
                                                        n,
                                                        (rocblas_double_complex*)alpha,
                                                        (rocblas_double_complex**)x,
                                                        incx,
                                                        (rocblas_double_complex**)A,
                                                        lda,
                                                        batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// syr_strided_batched
hipblasStatus_t hipblasSsyrStridedBatched(hipblasHandle_t   handle,
                                          hipblasFillMode_t uplo,
                                          int               n,
                                          const float*      alpha,
                                          const float*      x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          float*            A,
                                          int               lda,
                                          hipblasStride     strideA,
                                          int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_ssyr_strided_batched((rocblas_handle)handle,
                                                             hipblasConvertFill(uplo),
                                                             n,
                                                             alpha,
                                                             x,
                                                             incx,
                                                             stridex,
                                                             A,
                                                             lda,
                                                             strideA,
                                                             batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsyrStridedBatched(hipblasHandle_t   handle,
                                          hipblasFillMode_t uplo,
                                          int               n,
                                          const double*     alpha,
                                          const double*     x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          double*           A,
                                          int               lda,
                                          hipblasStride     strideA,
                                          int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_dsyr_strided_batched((rocblas_handle)handle,
                                                             hipblasConvertFill(uplo),
                                                             n,
                                                             alpha,
                                                             x,
                                                             incx,
                                                             stridex,
                                                             A,
                                                             lda,
                                                             strideA,
                                                             batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyrStridedBatched(hipblasHandle_t   handle,
                                          hipblasFillMode_t uplo,
                                          int               n,
                                          const hipComplex* alpha,
                                          const hipComplex* x,
                                          int               incx,
                                          hipblasStride     stridex,
                                          hipComplex*       A,
                                          int               lda,
                                          hipblasStride     strideA,
                                          int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_csyr_strided_batched((rocblas_handle)handle,
                                                             hipblasConvertFill(uplo),
                                                             n,
                                                             (rocblas_float_complex*)alpha,
                                                             (rocblas_float_complex*)x,
                                                             incx,
                                                             stridex,
                                                             (rocblas_float_complex*)A,
                                                             lda,
                                                             strideA,
                                                             batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyrStridedBatched(hipblasHandle_t         handle,
                                          hipblasFillMode_t       uplo,
                                          int                     n,
                                          const hipDoubleComplex* alpha,
                                          const hipDoubleComplex* x,
                                          int                     incx,
                                          hipblasStride           stridex,
                                          hipDoubleComplex*       A,
                                          int                     lda,
                                          hipblasStride           strideA,
                                          int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zsyr_strided_batched((rocblas_handle)handle,
                                                             hipblasConvertFill(uplo),
                                                             n,
                                                             (rocblas_double_complex*)alpha,
                                                             (rocblas_double_complex*)x,
                                                             incx,
                                                             stridex,
                                                             (rocblas_double_complex*)A,
                                                             lda,
                                                             strideA,
                                                             batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// syr_strided_batched_64
hipblasStatus_t hipblasSsyrStridedBatched_64(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int64_t           n,
                                             const float*      alpha,
                                             const float*      x,
                                             int64_t           incx,
                                             hipblasStride     stridex,
                                             float*            A,
                                             int64_t           lda,
                                             hipblasStride     strideA,
                                             int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_ssyr_strided_batched_64((rocblas_handle)handle,
                                                                hipblasConvertFill(uplo),
                                                                n,
                                                                alpha,
                                                                x,
                                                                incx,
                                                                stridex,
                                                                A,
                                                                lda,
                                                                strideA,
                                                                batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsyrStridedBatched_64(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int64_t           n,
                                             const double*     alpha,
                                             const double*     x,
                                             int64_t           incx,
                                             hipblasStride     stridex,
                                             double*           A,
                                             int64_t           lda,
                                             hipblasStride     strideA,
                                             int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_dsyr_strided_batched_64((rocblas_handle)handle,
                                                                hipblasConvertFill(uplo),
                                                                n,
                                                                alpha,
                                                                x,
                                                                incx,
                                                                stridex,
                                                                A,
                                                                lda,
                                                                strideA,
                                                                batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyrStridedBatched_64(hipblasHandle_t   handle,
                                             hipblasFillMode_t uplo,
                                             int64_t           n,
                                             const hipComplex* alpha,
                                             const hipComplex* x,
                                             int64_t           incx,
                                             hipblasStride     stridex,
                                             hipComplex*       A,
                                             int64_t           lda,
                                             hipblasStride     strideA,
                                             int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_csyr_strided_batched_64((rocblas_handle)handle,
                                                                hipblasConvertFill(uplo),
                                                                n,
                                                                (rocblas_float_complex*)alpha,
                                                                (rocblas_float_complex*)x,
                                                                incx,
                                                                stridex,
                                                                (rocblas_float_complex*)A,
                                                                lda,
                                                                strideA,
                                                                batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyrStridedBatched_64(hipblasHandle_t         handle,
                                             hipblasFillMode_t       uplo,
                                             int64_t                 n,
                                             const hipDoubleComplex* alpha,
                                             const hipDoubleComplex* x,
                                             int64_t                 incx,
                                             hipblasStride           stridex,
                                             hipDoubleComplex*       A,
                                             int64_t                 lda,
                                             hipblasStride           strideA,
                                             int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zsyr_strided_batched_64((rocblas_handle)handle,
                                                                hipblasConvertFill(uplo),
                                                                n,
                                                                (rocblas_double_complex*)alpha,
                                                                (rocblas_double_complex*)x,
                                                                incx,
                                                                stridex,
                                                                (rocblas_double_complex*)A,
                                                                lda,
                                                                strideA,
                                                                batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// syr2
hipblasStatus_t hipblasSsyr2(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const float*      alpha,
                             const float*      x,
                             int               incx,
                             const float*      y,
                             int               incy,
                             float*            A,
                             int               lda)
try
{
    return hipblasConvertStatus(rocblas_ssyr2(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, y, incy, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsyr2(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const double*     alpha,
                             const double*     x,
                             int               incx,
                             const double*     y,
                             int               incy,
                             double*           A,
                             int               lda)
try
{
    return hipblasConvertStatus(rocblas_dsyr2(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, y, incy, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyr2(hipblasHandle_t   handle,
                             hipblasFillMode_t uplo,
                             int               n,
                             const hipComplex* alpha,
                             const hipComplex* x,
                             int               incx,
                             const hipComplex* y,
                             int               incy,
                             hipComplex*       A,
                             int               lda)
try
{
    return hipblasConvertStatus(rocblas_csyr2((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              (rocblas_float_complex*)alpha,
                                              (rocblas_float_complex*)x,
                                              incx,
                                              (rocblas_float_complex*)y,
                                              incy,
                                              (rocblas_float_complex*)A,
                                              lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyr2(hipblasHandle_t         handle,
                             hipblasFillMode_t       uplo,
                             int                     n,
                             const hipDoubleComplex* alpha,
                             const hipDoubleComplex* x,
                             int                     incx,
                             const hipDoubleComplex* y,
                             int                     incy,
                             hipDoubleComplex*       A,
                             int                     lda)
try
{
    return hipblasConvertStatus(rocblas_zsyr2((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              n,
                                              (rocblas_double_complex*)alpha,
                                              (rocblas_double_complex*)x,
                                              incx,
                                              (rocblas_double_complex*)y,
                                              incy,
                                              (rocblas_double_complex*)A,
                                              lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// syr2_64
hipblasStatus_t hipblasSsyr2_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const float*      alpha,
                                const float*      x,
                                int64_t           incx,
                                const float*      y,
                                int64_t           incy,
                                float*            A,
                                int64_t           lda)
try
{
    return hipblasConvertStatus(rocblas_ssyr2_64(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, y, incy, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsyr2_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const double*     alpha,
                                const double*     x,
                                int64_t           incx,
                                const double*     y,
                                int64_t           incy,
                                double*           A,
                                int64_t           lda)
try
{
    return hipblasConvertStatus(rocblas_dsyr2_64(
        (rocblas_handle)handle, hipblasConvertFill(uplo), n, alpha, x, incx, y, incy, A, lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyr2_64(hipblasHandle_t   handle,
                                hipblasFillMode_t uplo,
                                int64_t           n,
                                const hipComplex* alpha,
                                const hipComplex* x,
                                int64_t           incx,
                                const hipComplex* y,
                                int64_t           incy,
                                hipComplex*       A,
                                int64_t           lda)
try
{
    return hipblasConvertStatus(rocblas_csyr2_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 (rocblas_float_complex*)alpha,
                                                 (rocblas_float_complex*)x,
                                                 incx,
                                                 (rocblas_float_complex*)y,
                                                 incy,
                                                 (rocblas_float_complex*)A,
                                                 lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyr2_64(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                int64_t                 n,
                                const hipDoubleComplex* alpha,
                                const hipDoubleComplex* x,
                                int64_t                 incx,
                                const hipDoubleComplex* y,
                                int64_t                 incy,
                                hipDoubleComplex*       A,
                                int64_t                 lda)
try
{
    return hipblasConvertStatus(rocblas_zsyr2_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 n,
                                                 (rocblas_double_complex*)alpha,
                                                 (rocblas_double_complex*)x,
                                                 incx,
                                                 (rocblas_double_complex*)y,
                                                 incy,
                                                 (rocblas_double_complex*)A,
                                                 lda));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// syr2_batched
hipblasStatus_t hipblasSsyr2Batched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    int                n,
                                    const float*       alpha,
                                    const float* const x[],
                                    int                incx,
                                    const float* const y[],
                                    int                incy,
                                    float* const       A[],
                                    int                lda,
                                    int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_ssyr2_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      alpha,
                                                      x,
                                                      incx,
                                                      y,
                                                      incy,
                                                      A,
                                                      lda,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsyr2Batched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    int                 n,
                                    const double*       alpha,
                                    const double* const x[],
                                    int                 incx,
                                    const double* const y[],
                                    int                 incy,
                                    double* const       A[],
                                    int                 lda,
                                    int                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_dsyr2_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      alpha,
                                                      x,
                                                      incx,
                                                      y,
                                                      incy,
                                                      A,
                                                      lda,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyr2Batched(hipblasHandle_t         handle,
                                    hipblasFillMode_t       uplo,
                                    int                     n,
                                    const hipComplex*       alpha,
                                    const hipComplex* const x[],
                                    int                     incx,
                                    const hipComplex* const y[],
                                    int                     incy,
                                    hipComplex* const       A[],
                                    int                     lda,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_csyr2_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      (rocblas_float_complex*)alpha,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      (rocblas_float_complex**)y,
                                                      incy,
                                                      (rocblas_float_complex**)A,
                                                      lda,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyr2Batched(hipblasHandle_t               handle,
                                    hipblasFillMode_t             uplo,
                                    int                           n,
                                    const hipDoubleComplex*       alpha,
                                    const hipDoubleComplex* const x[],
                                    int                           incx,
                                    const hipDoubleComplex* const y[],
                                    int                           incy,
                                    hipDoubleComplex* const       A[],
                                    int                           lda,
                                    int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zsyr2_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      n,
                                                      (rocblas_double_complex*)alpha,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      (rocblas_double_complex**)y,
                                                      incy,
                                                      (rocblas_double_complex**)A,
                                                      lda,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// syr2_batched_64
hipblasStatus_t hipblasSsyr2Batched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       int64_t            n,
                                       const float*       alpha,
                                       const float* const x[],
                                       int64_t            incx,
                                       const float* const y[],
                                       int64_t            incy,
                                       float* const       A[],
                                       int64_t            lda,
                                       int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_ssyr2_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         alpha,
                                                         x,
                                                         incx,
                                                         y,
                                                         incy,
                                                         A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsyr2Batched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       int64_t             n,
                                       const double*       alpha,
                                       const double* const x[],
                                       int64_t             incx,
                                       const double* const y[],
                                       int64_t             incy,
                                       double* const       A[],
                                       int64_t             lda,
                                       int64_t             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dsyr2_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         alpha,
                                                         x,
                                                         incx,
                                                         y,
                                                         incy,
                                                         A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyr2Batched_64(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       int64_t                 n,
                                       const hipComplex*       alpha,
                                       const hipComplex* const x[],
                                       int64_t                 incx,
                                       const hipComplex* const y[],
                                       int64_t                 incy,
                                       hipComplex* const       A[],
                                       int64_t                 lda,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_csyr2_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         (rocblas_float_complex*)alpha,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         (rocblas_float_complex**)y,
                                                         incy,
                                                         (rocblas_float_complex**)A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyr2Batched_64(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       int64_t                       n,
                                       const hipDoubleComplex*       alpha,
                                       const hipDoubleComplex* const x[],
                                       int64_t                       incx,
                                       const hipDoubleComplex* const y[],
                                       int64_t                       incy,
                                       hipDoubleComplex* const       A[],
                                       int64_t                       lda,
                                       int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_zsyr2_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         n,
                                                         (rocblas_double_complex*)alpha,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         (rocblas_double_complex**)y,
                                                         incy,
                                                         (rocblas_double_complex**)A,
                                                         lda,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// syr2_strided_batched
hipblasStatus_t hipblasSsyr2StridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const float*      alpha,
                                           const float*      x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const float*      y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           float*            A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_ssyr2_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              alpha,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              y,
                                                              incy,
                                                              stridey,
                                                              A,
                                                              lda,
                                                              strideA,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsyr2StridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const double*     alpha,
                                           const double*     x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const double*     y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           double*           A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_dsyr2_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              alpha,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              y,
                                                              incy,
                                                              stridey,
                                                              A,
                                                              lda,
                                                              strideA,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyr2StridedBatched(hipblasHandle_t   handle,
                                           hipblasFillMode_t uplo,
                                           int               n,
                                           const hipComplex* alpha,
                                           const hipComplex* x,
                                           int               incx,
                                           hipblasStride     stridex,
                                           const hipComplex* y,
                                           int               incy,
                                           hipblasStride     stridey,
                                           hipComplex*       A,
                                           int               lda,
                                           hipblasStride     strideA,
                                           int               batchCount)
try
{
    return hipblasConvertStatus(rocblas_csyr2_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              (rocblas_float_complex*)alpha,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_float_complex*)y,
                                                              incy,
                                                              stridey,
                                                              (rocblas_float_complex*)A,
                                                              lda,
                                                              strideA,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyr2StridedBatched(hipblasHandle_t         handle,
                                           hipblasFillMode_t       uplo,
                                           int                     n,
                                           const hipDoubleComplex* alpha,
                                           const hipDoubleComplex* x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           const hipDoubleComplex* y,
                                           int                     incy,
                                           hipblasStride           stridey,
                                           hipDoubleComplex*       A,
                                           int                     lda,
                                           hipblasStride           strideA,
                                           int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_zsyr2_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              n,
                                                              (rocblas_double_complex*)alpha,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              (rocblas_double_complex*)y,
                                                              incy,
                                                              stridey,
                                                              (rocblas_double_complex*)A,
                                                              lda,
                                                              strideA,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// syr2_strided_batched_64
hipblasStatus_t hipblasSsyr2StridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const float*      alpha,
                                              const float*      x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const float*      y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              float*            A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_ssyr2_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 y,
                                                                 incy,
                                                                 stridey,
                                                                 A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDsyr2StridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const double*     alpha,
                                              const double*     x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const double*     y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              double*           A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_dsyr2_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 alpha,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 y,
                                                                 incy,
                                                                 stridey,
                                                                 A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCsyr2StridedBatched_64(hipblasHandle_t   handle,
                                              hipblasFillMode_t uplo,
                                              int64_t           n,
                                              const hipComplex* alpha,
                                              const hipComplex* x,
                                              int64_t           incx,
                                              hipblasStride     stridex,
                                              const hipComplex* y,
                                              int64_t           incy,
                                              hipblasStride     stridey,
                                              hipComplex*       A,
                                              int64_t           lda,
                                              hipblasStride     strideA,
                                              int64_t           batchCount)
try
{
    return hipblasConvertStatus(rocblas_csyr2_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 (rocblas_float_complex*)alpha,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_float_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 (rocblas_float_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZsyr2StridedBatched_64(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              int64_t                 n,
                                              const hipDoubleComplex* alpha,
                                              const hipDoubleComplex* x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              const hipDoubleComplex* y,
                                              int64_t                 incy,
                                              hipblasStride           stridey,
                                              hipDoubleComplex*       A,
                                              int64_t                 lda,
                                              hipblasStride           strideA,
                                              int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_zsyr2_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 n,
                                                                 (rocblas_double_complex*)alpha,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 (rocblas_double_complex*)y,
                                                                 incy,
                                                                 stridey,
                                                                 (rocblas_double_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tbmv
hipblasStatus_t hipblasStbmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             int                k,
                             const float*       A,
                             int                lda,
                             float*             x,
                             int                incx)
try
{
    return hipblasConvertStatus(rocblas_stbmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              k,
                                              A,
                                              lda,
                                              x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtbmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             int                k,
                             const double*      A,
                             int                lda,
                             double*            x,
                             int                incx)
try
{
    return hipblasConvertStatus(rocblas_dtbmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              k,
                                              A,
                                              lda,
                                              x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             int                k,
                             const hipComplex*  A,
                             int                lda,
                             hipComplex*        x,
                             int                incx)
try
{
    return hipblasConvertStatus(rocblas_ctbmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              k,
                                              (rocblas_float_complex*)A,
                                              lda,
                                              (rocblas_float_complex*)x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbmv(hipblasHandle_t         handle,
                             hipblasFillMode_t       uplo,
                             hipblasOperation_t      transA,
                             hipblasDiagType_t       diag,
                             int                     n,
                             int                     k,
                             const hipDoubleComplex* A,
                             int                     lda,
                             hipDoubleComplex*       x,
                             int                     incx)
try
{
    return hipblasConvertStatus(rocblas_ztbmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              k,
                                              (rocblas_double_complex*)A,
                                              lda,
                                              (rocblas_double_complex*)x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tbmv_64
hipblasStatus_t hipblasStbmv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                int64_t            k,
                                const float*       A,
                                int64_t            lda,
                                float*             x,
                                int64_t            incx)
try
{
    return hipblasConvertStatus(rocblas_stbmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 k,
                                                 A,
                                                 lda,
                                                 x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtbmv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                int64_t            k,
                                const double*      A,
                                int64_t            lda,
                                double*            x,
                                int64_t            incx)
try
{
    return hipblasConvertStatus(rocblas_dtbmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 k,
                                                 A,
                                                 lda,
                                                 x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbmv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                int64_t            k,
                                const hipComplex*  A,
                                int64_t            lda,
                                hipComplex*        x,
                                int64_t            incx)
try
{
    return hipblasConvertStatus(rocblas_ctbmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 k,
                                                 (rocblas_float_complex*)A,
                                                 lda,
                                                 (rocblas_float_complex*)x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbmv_64(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                hipblasDiagType_t       diag,
                                int64_t                 n,
                                int64_t                 k,
                                const hipDoubleComplex* A,
                                int64_t                 lda,
                                hipDoubleComplex*       x,
                                int64_t                 incx)
try
{
    return hipblasConvertStatus(rocblas_ztbmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 k,
                                                 (rocblas_double_complex*)A,
                                                 lda,
                                                 (rocblas_double_complex*)x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tbmv_batched
hipblasStatus_t hipblasStbmvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    hipblasOperation_t transA,
                                    hipblasDiagType_t  diag,
                                    int                n,
                                    int                k,
                                    const float* const A[],
                                    int                lda,
                                    float* const       x[],
                                    int                incx,
                                    int                batch_count)
try
{
    return hipblasConvertStatus(rocblas_stbmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      k,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtbmvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    hipblasOperation_t  transA,
                                    hipblasDiagType_t   diag,
                                    int                 n,
                                    int                 k,
                                    const double* const A[],
                                    int                 lda,
                                    double* const       x[],
                                    int                 incx,
                                    int                 batch_count)
try
{
    return hipblasConvertStatus(rocblas_dtbmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      k,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbmvBatched(hipblasHandle_t         handle,
                                    hipblasFillMode_t       uplo,
                                    hipblasOperation_t      transA,
                                    hipblasDiagType_t       diag,
                                    int                     n,
                                    int                     k,
                                    const hipComplex* const A[],
                                    int                     lda,
                                    hipComplex* const       x[],
                                    int                     incx,
                                    int                     batch_count)
try
{
    return hipblasConvertStatus(rocblas_ctbmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      k,
                                                      (rocblas_float_complex**)A,
                                                      lda,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbmvBatched(hipblasHandle_t               handle,
                                    hipblasFillMode_t             uplo,
                                    hipblasOperation_t            transA,
                                    hipblasDiagType_t             diag,
                                    int                           n,
                                    int                           k,
                                    const hipDoubleComplex* const A[],
                                    int                           lda,
                                    hipDoubleComplex* const       x[],
                                    int                           incx,
                                    int                           batch_count)
try
{
    return hipblasConvertStatus(rocblas_ztbmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      k,
                                                      (rocblas_double_complex**)A,
                                                      lda,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tbmv_batched_64
hipblasStatus_t hipblasStbmvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       hipblasOperation_t transA,
                                       hipblasDiagType_t  diag,
                                       int64_t            n,
                                       int64_t            k,
                                       const float* const A[],
                                       int64_t            lda,
                                       float* const       x[],
                                       int64_t            incx,
                                       int64_t            batch_count)
try
{
    return hipblasConvertStatus(rocblas_stbmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         k,
                                                         A,
                                                         lda,
                                                         x,
                                                         incx,
                                                         batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtbmvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       hipblasOperation_t  transA,
                                       hipblasDiagType_t   diag,
                                       int64_t             n,
                                       int64_t             k,
                                       const double* const A[],
                                       int64_t             lda,
                                       double* const       x[],
                                       int64_t             incx,
                                       int64_t             batch_count)
try
{
    return hipblasConvertStatus(rocblas_dtbmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         k,
                                                         A,
                                                         lda,
                                                         x,
                                                         incx,
                                                         batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbmvBatched_64(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       hipblasDiagType_t       diag,
                                       int64_t                 n,
                                       int64_t                 k,
                                       const hipComplex* const A[],
                                       int64_t                 lda,
                                       hipComplex* const       x[],
                                       int64_t                 incx,
                                       int64_t                 batch_count)
try
{
    return hipblasConvertStatus(rocblas_ctbmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         k,
                                                         (rocblas_float_complex**)A,
                                                         lda,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbmvBatched_64(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       hipblasOperation_t            transA,
                                       hipblasDiagType_t             diag,
                                       int64_t                       n,
                                       int64_t                       k,
                                       const hipDoubleComplex* const A[],
                                       int64_t                       lda,
                                       hipDoubleComplex* const       x[],
                                       int64_t                       incx,
                                       int64_t                       batch_count)
try
{
    return hipblasConvertStatus(rocblas_ztbmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         k,
                                                         (rocblas_double_complex**)A,
                                                         lda,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tbmv_strided_batched
hipblasStatus_t hipblasStbmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           int                k,
                                           const float*       A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           float*             x,
                                           int                incx,
                                           hipblasStride      stride_x,
                                           int                batch_count)
try
{
    return hipblasConvertStatus(rocblas_stbmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              k,
                                                              A,
                                                              lda,
                                                              stride_a,
                                                              x,
                                                              incx,
                                                              stride_x,
                                                              batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtbmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           int                k,
                                           const double*      A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           double*            x,
                                           int                incx,
                                           hipblasStride      stride_x,
                                           int                batch_count)
try
{
    return hipblasConvertStatus(rocblas_dtbmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              k,
                                                              A,
                                                              lda,
                                                              stride_a,
                                                              x,
                                                              incx,
                                                              stride_x,
                                                              batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           int                k,
                                           const hipComplex*  A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           hipComplex*        x,
                                           int                incx,
                                           hipblasStride      stride_x,
                                           int                batch_count)
try
{
    return hipblasConvertStatus(rocblas_ctbmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              k,
                                                              (rocblas_float_complex*)A,
                                                              lda,
                                                              stride_a,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stride_x,
                                                              batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbmvStridedBatched(hipblasHandle_t         handle,
                                           hipblasFillMode_t       uplo,
                                           hipblasOperation_t      transA,
                                           hipblasDiagType_t       diag,
                                           int                     n,
                                           int                     k,
                                           const hipDoubleComplex* A,
                                           int                     lda,
                                           hipblasStride           stride_a,
                                           hipDoubleComplex*       x,
                                           int                     incx,
                                           hipblasStride           stride_x,
                                           int                     batch_count)
try
{
    return hipblasConvertStatus(rocblas_ztbmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              k,
                                                              (rocblas_double_complex*)A,
                                                              lda,
                                                              stride_a,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stride_x,
                                                              batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tbmv_strided_batched_64
hipblasStatus_t hipblasStbmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              int64_t            k,
                                              const float*       A,
                                              int64_t            lda,
                                              hipblasStride      stride_a,
                                              float*             x,
                                              int64_t            incx,
                                              hipblasStride      stride_x,
                                              int64_t            batch_count)
try
{
    return hipblasConvertStatus(rocblas_stbmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 k,
                                                                 A,
                                                                 lda,
                                                                 stride_a,
                                                                 x,
                                                                 incx,
                                                                 stride_x,
                                                                 batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtbmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              int64_t            k,
                                              const double*      A,
                                              int64_t            lda,
                                              hipblasStride      stride_a,
                                              double*            x,
                                              int64_t            incx,
                                              hipblasStride      stride_x,
                                              int64_t            batch_count)
try
{
    return hipblasConvertStatus(rocblas_dtbmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 k,
                                                                 A,
                                                                 lda,
                                                                 stride_a,
                                                                 x,
                                                                 incx,
                                                                 stride_x,
                                                                 batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              int64_t            k,
                                              const hipComplex*  A,
                                              int64_t            lda,
                                              hipblasStride      stride_a,
                                              hipComplex*        x,
                                              int64_t            incx,
                                              hipblasStride      stride_x,
                                              int64_t            batch_count)
try
{
    return hipblasConvertStatus(rocblas_ctbmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 k,
                                                                 (rocblas_float_complex*)A,
                                                                 lda,
                                                                 stride_a,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stride_x,
                                                                 batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbmvStridedBatched_64(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              hipblasOperation_t      transA,
                                              hipblasDiagType_t       diag,
                                              int64_t                 n,
                                              int64_t                 k,
                                              const hipDoubleComplex* A,
                                              int64_t                 lda,
                                              hipblasStride           stride_a,
                                              hipDoubleComplex*       x,
                                              int64_t                 incx,
                                              hipblasStride           stride_x,
                                              int64_t                 batch_count)
try
{
    return hipblasConvertStatus(rocblas_ztbmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 k,
                                                                 (rocblas_double_complex*)A,
                                                                 lda,
                                                                 stride_a,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stride_x,
                                                                 batch_count));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tbsv
hipblasStatus_t hipblasStbsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             int                k,
                             const float*       A,
                             int                lda,
                             float*             x,
                             int                incx)
try
{
    return hipblasConvertStatus(rocblas_stbsv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              k,
                                              A,
                                              lda,
                                              x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtbsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             int                k,
                             const double*      A,
                             int                lda,
                             double*            x,
                             int                incx)
try
{
    return hipblasConvertStatus(rocblas_dtbsv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              k,
                                              A,
                                              lda,
                                              x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             int                k,
                             const hipComplex*  A,
                             int                lda,
                             hipComplex*        x,
                             int                incx)
try
{
    return hipblasConvertStatus(rocblas_ctbsv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              k,
                                              (rocblas_float_complex*)A,
                                              lda,
                                              (rocblas_float_complex*)x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbsv(hipblasHandle_t         handle,
                             hipblasFillMode_t       uplo,
                             hipblasOperation_t      transA,
                             hipblasDiagType_t       diag,
                             int                     n,
                             int                     k,
                             const hipDoubleComplex* A,
                             int                     lda,
                             hipDoubleComplex*       x,
                             int                     incx)
try
{
    return hipblasConvertStatus(rocblas_ztbsv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              k,
                                              (rocblas_double_complex*)A,
                                              lda,
                                              (rocblas_double_complex*)x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tbsv_64
hipblasStatus_t hipblasStbsv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                int64_t            k,
                                const float*       A,
                                int64_t            lda,
                                float*             x,
                                int64_t            incx)
try
{
    return hipblasConvertStatus(rocblas_stbsv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 k,
                                                 A,
                                                 lda,
                                                 x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtbsv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                int64_t            k,
                                const double*      A,
                                int64_t            lda,
                                double*            x,
                                int64_t            incx)
try
{
    return hipblasConvertStatus(rocblas_dtbsv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 k,
                                                 A,
                                                 lda,
                                                 x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbsv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                int64_t            k,
                                const hipComplex*  A,
                                int64_t            lda,
                                hipComplex*        x,
                                int64_t            incx)
try
{
    return hipblasConvertStatus(rocblas_ctbsv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 k,
                                                 (rocblas_float_complex*)A,
                                                 lda,
                                                 (rocblas_float_complex*)x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbsv_64(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                hipblasDiagType_t       diag,
                                int64_t                 n,
                                int64_t                 k,
                                const hipDoubleComplex* A,
                                int64_t                 lda,
                                hipDoubleComplex*       x,
                                int64_t                 incx)
try
{
    return hipblasConvertStatus(rocblas_ztbsv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 k,
                                                 (rocblas_double_complex*)A,
                                                 lda,
                                                 (rocblas_double_complex*)x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tbsv_batched
hipblasStatus_t hipblasStbsvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    hipblasOperation_t transA,
                                    hipblasDiagType_t  diag,
                                    int                n,
                                    int                k,
                                    const float* const A[],
                                    int                lda,
                                    float* const       x[],
                                    int                incx,
                                    int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_stbsv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      k,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtbsvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    hipblasOperation_t  transA,
                                    hipblasDiagType_t   diag,
                                    int                 n,
                                    int                 k,
                                    const double* const A[],
                                    int                 lda,
                                    double* const       x[],
                                    int                 incx,
                                    int                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_dtbsv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      k,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbsvBatched(hipblasHandle_t         handle,
                                    hipblasFillMode_t       uplo,
                                    hipblasOperation_t      transA,
                                    hipblasDiagType_t       diag,
                                    int                     n,
                                    int                     k,
                                    const hipComplex* const A[],
                                    int                     lda,
                                    hipComplex* const       x[],
                                    int                     incx,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_ctbsv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      k,
                                                      (rocblas_float_complex**)A,
                                                      lda,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbsvBatched(hipblasHandle_t               handle,
                                    hipblasFillMode_t             uplo,
                                    hipblasOperation_t            transA,
                                    hipblasDiagType_t             diag,
                                    int                           n,
                                    int                           k,
                                    const hipDoubleComplex* const A[],
                                    int                           lda,
                                    hipDoubleComplex* const       x[],
                                    int                           incx,
                                    int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_ztbsv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      k,
                                                      (rocblas_double_complex**)A,
                                                      lda,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tbsv_batched_64
hipblasStatus_t hipblasStbsvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       hipblasOperation_t transA,
                                       hipblasDiagType_t  diag,
                                       int64_t            n,
                                       int64_t            k,
                                       const float* const A[],
                                       int64_t            lda,
                                       float* const       x[],
                                       int64_t            incx,
                                       int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_stbsv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         k,
                                                         A,
                                                         lda,
                                                         x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtbsvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       hipblasOperation_t  transA,
                                       hipblasDiagType_t   diag,
                                       int64_t             n,
                                       int64_t             k,
                                       const double* const A[],
                                       int64_t             lda,
                                       double* const       x[],
                                       int64_t             incx,
                                       int64_t             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dtbsv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         k,
                                                         A,
                                                         lda,
                                                         x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbsvBatched_64(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       hipblasDiagType_t       diag,
                                       int64_t                 n,
                                       int64_t                 k,
                                       const hipComplex* const A[],
                                       int64_t                 lda,
                                       hipComplex* const       x[],
                                       int64_t                 incx,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_ctbsv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         k,
                                                         (rocblas_float_complex**)A,
                                                         lda,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbsvBatched_64(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       hipblasOperation_t            transA,
                                       hipblasDiagType_t             diag,
                                       int64_t                       n,
                                       int64_t                       k,
                                       const hipDoubleComplex* const A[],
                                       int64_t                       lda,
                                       hipDoubleComplex* const       x[],
                                       int64_t                       incx,
                                       int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_ztbsv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         k,
                                                         (rocblas_double_complex**)A,
                                                         lda,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tbsv_strided_batched
hipblasStatus_t hipblasStbsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           int                k,
                                           const float*       A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           float*             x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_stbsv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              k,
                                                              A,
                                                              lda,
                                                              strideA,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtbsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           int                k,
                                           const double*      A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           double*            x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_dtbsv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              k,
                                                              A,
                                                              lda,
                                                              strideA,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           int                k,
                                           const hipComplex*  A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           hipComplex*        x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_ctbsv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              k,
                                                              (rocblas_float_complex*)A,
                                                              lda,
                                                              strideA,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbsvStridedBatched(hipblasHandle_t         handle,
                                           hipblasFillMode_t       uplo,
                                           hipblasOperation_t      transA,
                                           hipblasDiagType_t       diag,
                                           int                     n,
                                           int                     k,
                                           const hipDoubleComplex* A,
                                           int                     lda,
                                           hipblasStride           strideA,
                                           hipDoubleComplex*       x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_ztbsv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              k,
                                                              (rocblas_double_complex*)A,
                                                              lda,
                                                              strideA,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tbsv_strided_batched_64
hipblasStatus_t hipblasStbsvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              int64_t            k,
                                              const float*       A,
                                              int64_t            lda,
                                              hipblasStride      strideA,
                                              float*             x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_stbsv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 k,
                                                                 A,
                                                                 lda,
                                                                 strideA,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtbsvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              int64_t            k,
                                              const double*      A,
                                              int64_t            lda,
                                              hipblasStride      strideA,
                                              double*            x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_dtbsv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 k,
                                                                 A,
                                                                 lda,
                                                                 strideA,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtbsvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              int64_t            k,
                                              const hipComplex*  A,
                                              int64_t            lda,
                                              hipblasStride      strideA,
                                              hipComplex*        x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_ctbsv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 k,
                                                                 (rocblas_float_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtbsvStridedBatched_64(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              hipblasOperation_t      transA,
                                              hipblasDiagType_t       diag,
                                              int64_t                 n,
                                              int64_t                 k,
                                              const hipDoubleComplex* A,
                                              int64_t                 lda,
                                              hipblasStride           strideA,
                                              hipDoubleComplex*       x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_ztbsv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 k,
                                                                 (rocblas_double_complex*)A,
                                                                 lda,
                                                                 strideA,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tpmv
hipblasStatus_t hipblasStpmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const float*       AP,
                             float*             x,
                             int                incx)
try
{
    return hipblasConvertStatus(rocblas_stpmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              AP,
                                              x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtpmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const double*      AP,
                             double*            x,
                             int                incx)
try
{
    return hipblasConvertStatus(rocblas_dtpmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              AP,
                                              x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const hipComplex*  AP,
                             hipComplex*        x,
                             int                incx)
try
{
    return hipblasConvertStatus(rocblas_ctpmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              (rocblas_float_complex*)AP,
                                              (rocblas_float_complex*)x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpmv(hipblasHandle_t         handle,
                             hipblasFillMode_t       uplo,
                             hipblasOperation_t      transA,
                             hipblasDiagType_t       diag,
                             int                     n,
                             const hipDoubleComplex* AP,
                             hipDoubleComplex*       x,
                             int                     incx)
try
{
    return hipblasConvertStatus(rocblas_ztpmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              (rocblas_double_complex*)AP,
                                              (rocblas_double_complex*)x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tpmv_64
hipblasStatus_t hipblasStpmv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const float*       AP,
                                float*             x,
                                int64_t            incx)
try
{
    return hipblasConvertStatus(rocblas_stpmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 AP,
                                                 x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtpmv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const double*      AP,
                                double*            x,
                                int64_t            incx)
try
{
    return hipblasConvertStatus(rocblas_dtpmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 AP,
                                                 x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpmv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const hipComplex*  AP,
                                hipComplex*        x,
                                int64_t            incx)
try
{
    return hipblasConvertStatus(rocblas_ctpmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 (rocblas_float_complex*)AP,
                                                 (rocblas_float_complex*)x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpmv_64(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                hipblasDiagType_t       diag,
                                int64_t                 n,
                                const hipDoubleComplex* AP,
                                hipDoubleComplex*       x,
                                int64_t                 incx)
try
{
    return hipblasConvertStatus(rocblas_ztpmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 (rocblas_double_complex*)AP,
                                                 (rocblas_double_complex*)x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tpmv_batched
hipblasStatus_t hipblasStpmvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    hipblasOperation_t transA,
                                    hipblasDiagType_t  diag,
                                    int                n,
                                    const float* const AP[],
                                    float* const       x[],
                                    int                incx,
                                    int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_stpmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      AP,
                                                      x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtpmvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    hipblasOperation_t  transA,
                                    hipblasDiagType_t   diag,
                                    int                 n,
                                    const double* const AP[],
                                    double* const       x[],
                                    int                 incx,
                                    int                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_dtpmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      AP,
                                                      x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpmvBatched(hipblasHandle_t         handle,
                                    hipblasFillMode_t       uplo,
                                    hipblasOperation_t      transA,
                                    hipblasDiagType_t       diag,
                                    int                     n,
                                    const hipComplex* const AP[],
                                    hipComplex* const       x[],
                                    int                     incx,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_ctpmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      (rocblas_float_complex**)AP,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpmvBatched(hipblasHandle_t               handle,
                                    hipblasFillMode_t             uplo,
                                    hipblasOperation_t            transA,
                                    hipblasDiagType_t             diag,
                                    int                           n,
                                    const hipDoubleComplex* const AP[],
                                    hipDoubleComplex* const       x[],
                                    int                           incx,
                                    int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_ztpmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      (rocblas_double_complex**)AP,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tpmv_batched_64
hipblasStatus_t hipblasStpmvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       hipblasOperation_t transA,
                                       hipblasDiagType_t  diag,
                                       int64_t            n,
                                       const float* const AP[],
                                       float* const       x[],
                                       int64_t            incx,
                                       int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_stpmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         AP,
                                                         x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtpmvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       hipblasOperation_t  transA,
                                       hipblasDiagType_t   diag,
                                       int64_t             n,
                                       const double* const AP[],
                                       double* const       x[],
                                       int64_t             incx,
                                       int64_t             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dtpmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         AP,
                                                         x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpmvBatched_64(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       hipblasDiagType_t       diag,
                                       int64_t                 n,
                                       const hipComplex* const AP[],
                                       hipComplex* const       x[],
                                       int64_t                 incx,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_ctpmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         (rocblas_float_complex**)AP,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpmvBatched_64(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       hipblasOperation_t            transA,
                                       hipblasDiagType_t             diag,
                                       int64_t                       n,
                                       const hipDoubleComplex* const AP[],
                                       hipDoubleComplex* const       x[],
                                       int64_t                       incx,
                                       int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_ztpmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         (rocblas_double_complex**)AP,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tpmv_strided_batched
hipblasStatus_t hipblasStpmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const float*       AP,
                                           hipblasStride      strideAP,
                                           float*             x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_stpmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              AP,
                                                              strideAP,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtpmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const double*      AP,
                                           hipblasStride      strideAP,
                                           double*            x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_dtpmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              AP,
                                                              strideAP,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const hipComplex*  AP,
                                           hipblasStride      strideAP,
                                           hipComplex*        x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_ctpmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              (rocblas_float_complex*)AP,
                                                              strideAP,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpmvStridedBatched(hipblasHandle_t         handle,
                                           hipblasFillMode_t       uplo,
                                           hipblasOperation_t      transA,
                                           hipblasDiagType_t       diag,
                                           int                     n,
                                           const hipDoubleComplex* AP,
                                           hipblasStride           strideAP,
                                           hipDoubleComplex*       x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_ztpmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              (rocblas_double_complex*)AP,
                                                              strideAP,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tpmv_strided_batched_64
hipblasStatus_t hipblasStpmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const float*       AP,
                                              hipblasStride      strideAP,
                                              float*             x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_stpmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 AP,
                                                                 strideAP,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtpmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const double*      AP,
                                              hipblasStride      strideAP,
                                              double*            x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_dtpmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 AP,
                                                                 strideAP,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const hipComplex*  AP,
                                              hipblasStride      strideAP,
                                              hipComplex*        x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_ctpmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 (rocblas_float_complex*)AP,
                                                                 strideAP,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpmvStridedBatched_64(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              hipblasOperation_t      transA,
                                              hipblasDiagType_t       diag,
                                              int64_t                 n,
                                              const hipDoubleComplex* AP,
                                              hipblasStride           strideAP,
                                              hipDoubleComplex*       x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_ztpmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 (rocblas_double_complex*)AP,
                                                                 strideAP,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tpsv
hipblasStatus_t hipblasStpsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const float*       AP,
                             float*             x,
                             int                incx)
try
{
    return hipblasConvertStatus(rocblas_stpsv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              AP,
                                              x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtpsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const double*      AP,
                             double*            x,
                             int                incx)
try
{
    return hipblasConvertStatus(rocblas_dtpsv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              AP,
                                              x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const hipComplex*  AP,
                             hipComplex*        x,
                             int                incx)
try
{
    return hipblasConvertStatus(rocblas_ctpsv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              (rocblas_float_complex*)AP,
                                              (rocblas_float_complex*)x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpsv(hipblasHandle_t         handle,
                             hipblasFillMode_t       uplo,
                             hipblasOperation_t      transA,
                             hipblasDiagType_t       diag,
                             int                     n,
                             const hipDoubleComplex* AP,
                             hipDoubleComplex*       x,
                             int                     incx)
try
{
    return hipblasConvertStatus(rocblas_ztpsv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              (rocblas_double_complex*)AP,
                                              (rocblas_double_complex*)x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tpsv_64
hipblasStatus_t hipblasStpsv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const float*       AP,
                                float*             x,
                                int64_t            incx)
try
{
    return hipblasConvertStatus(rocblas_stpsv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 AP,
                                                 x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtpsv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const double*      AP,
                                double*            x,
                                int64_t            incx)
try
{
    return hipblasConvertStatus(rocblas_dtpsv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 AP,
                                                 x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpsv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const hipComplex*  AP,
                                hipComplex*        x,
                                int64_t            incx)
try
{
    return hipblasConvertStatus(rocblas_ctpsv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 (rocblas_float_complex*)AP,
                                                 (rocblas_float_complex*)x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpsv_64(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                hipblasDiagType_t       diag,
                                int64_t                 n,
                                const hipDoubleComplex* AP,
                                hipDoubleComplex*       x,
                                int64_t                 incx)
try
{
    return hipblasConvertStatus(rocblas_ztpsv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 (rocblas_double_complex*)AP,
                                                 (rocblas_double_complex*)x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tpsv_batched
hipblasStatus_t hipblasStpsvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    hipblasOperation_t transA,
                                    hipblasDiagType_t  diag,
                                    int                n,
                                    const float* const AP[],
                                    float* const       x[],
                                    int                incx,
                                    int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_stpsv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      AP,
                                                      x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtpsvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    hipblasOperation_t  transA,
                                    hipblasDiagType_t   diag,
                                    int                 n,
                                    const double* const AP[],
                                    double* const       x[],
                                    int                 incx,
                                    int                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_dtpsv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      AP,
                                                      x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpsvBatched(hipblasHandle_t         handle,
                                    hipblasFillMode_t       uplo,
                                    hipblasOperation_t      transA,
                                    hipblasDiagType_t       diag,
                                    int                     n,
                                    const hipComplex* const AP[],
                                    hipComplex* const       x[],
                                    int                     incx,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_ctpsv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      (rocblas_float_complex**)AP,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpsvBatched(hipblasHandle_t               handle,
                                    hipblasFillMode_t             uplo,
                                    hipblasOperation_t            transA,
                                    hipblasDiagType_t             diag,
                                    int                           n,
                                    const hipDoubleComplex* const AP[],
                                    hipDoubleComplex* const       x[],
                                    int                           incx,
                                    int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_ztpsv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      (rocblas_double_complex**)AP,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tpsv_batched_64
hipblasStatus_t hipblasStpsvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       hipblasOperation_t transA,
                                       hipblasDiagType_t  diag,
                                       int64_t            n,
                                       const float* const AP[],
                                       float* const       x[],
                                       int64_t            incx,
                                       int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_stpsv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         AP,
                                                         x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtpsvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       hipblasOperation_t  transA,
                                       hipblasDiagType_t   diag,
                                       int64_t             n,
                                       const double* const AP[],
                                       double* const       x[],
                                       int64_t             incx,
                                       int64_t             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dtpsv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         AP,
                                                         x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpsvBatched_64(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       hipblasDiagType_t       diag,
                                       int64_t                 n,
                                       const hipComplex* const AP[],
                                       hipComplex* const       x[],
                                       int64_t                 incx,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_ctpsv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         (rocblas_float_complex**)AP,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpsvBatched_64(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       hipblasOperation_t            transA,
                                       hipblasDiagType_t             diag,
                                       int64_t                       n,
                                       const hipDoubleComplex* const AP[],
                                       hipDoubleComplex* const       x[],
                                       int64_t                       incx,
                                       int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_ztpsv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         (rocblas_double_complex**)AP,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tpsv_strided_batched
hipblasStatus_t hipblasStpsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const float*       AP,
                                           hipblasStride      strideAP,
                                           float*             x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_stpsv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              AP,
                                                              strideAP,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtpsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const double*      AP,
                                           hipblasStride      strideAP,
                                           double*            x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_dtpsv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              AP,
                                                              strideAP,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const hipComplex*  AP,
                                           hipblasStride      strideAP,
                                           hipComplex*        x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_ctpsv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              (rocblas_float_complex*)AP,
                                                              strideAP,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpsvStridedBatched(hipblasHandle_t         handle,
                                           hipblasFillMode_t       uplo,
                                           hipblasOperation_t      transA,
                                           hipblasDiagType_t       diag,
                                           int                     n,
                                           const hipDoubleComplex* AP,
                                           hipblasStride           strideAP,
                                           hipDoubleComplex*       x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_ztpsv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              (rocblas_double_complex*)AP,
                                                              strideAP,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// tpsv_strided_batched_64
hipblasStatus_t hipblasStpsvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const float*       AP,
                                              hipblasStride      strideAP,
                                              float*             x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_stpsv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 AP,
                                                                 strideAP,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtpsvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const double*      AP,
                                              hipblasStride      strideAP,
                                              double*            x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_dtpsv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 AP,
                                                                 strideAP,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtpsvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const hipComplex*  AP,
                                              hipblasStride      strideAP,
                                              hipComplex*        x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_ctpsv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 (rocblas_float_complex*)AP,
                                                                 strideAP,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtpsvStridedBatched_64(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              hipblasOperation_t      transA,
                                              hipblasDiagType_t       diag,
                                              int64_t                 n,
                                              const hipDoubleComplex* AP,
                                              hipblasStride           strideAP,
                                              hipDoubleComplex*       x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_ztpsv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 (rocblas_double_complex*)AP,
                                                                 strideAP,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// trmv
hipblasStatus_t hipblasStrmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const float*       A,
                             int                lda,
                             float*             x,
                             int                incx)
try
{
    return hipblasConvertStatus(rocblas_strmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              A,
                                              lda,
                                              x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtrmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const double*      A,
                             int                lda,
                             double*            x,
                             int                incx)
try
{
    return hipblasConvertStatus(rocblas_dtrmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              A,
                                              lda,
                                              x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtrmv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const hipComplex*  A,
                             int                lda,
                             hipComplex*        x,
                             int                incx)
try
{
    return hipblasConvertStatus(rocblas_ctrmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              (rocblas_float_complex*)A,
                                              lda,
                                              (rocblas_float_complex*)x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtrmv(hipblasHandle_t         handle,
                             hipblasFillMode_t       uplo,
                             hipblasOperation_t      transA,
                             hipblasDiagType_t       diag,
                             int                     n,
                             const hipDoubleComplex* A,
                             int                     lda,
                             hipDoubleComplex*       x,
                             int                     incx)
try
{
    return hipblasConvertStatus(rocblas_ztrmv((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              (rocblas_double_complex*)A,
                                              lda,
                                              (rocblas_double_complex*)x,
                                              incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// trmv
hipblasStatus_t hipblasStrmv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const float*       A,
                                int64_t            lda,
                                float*             x,
                                int64_t            incx)
try
{
    return hipblasConvertStatus(rocblas_strmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 A,
                                                 lda,
                                                 x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtrmv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const double*      A,
                                int64_t            lda,
                                double*            x,
                                int64_t            incx)
try
{
    return hipblasConvertStatus(rocblas_dtrmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 A,
                                                 lda,
                                                 x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtrmv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const hipComplex*  A,
                                int64_t            lda,
                                hipComplex*        x,
                                int64_t            incx)
try
{
    return hipblasConvertStatus(rocblas_ctrmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 (rocblas_float_complex*)A,
                                                 lda,
                                                 (rocblas_float_complex*)x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtrmv_64(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                hipblasDiagType_t       diag,
                                int64_t                 n,
                                const hipDoubleComplex* A,
                                int64_t                 lda,
                                hipDoubleComplex*       x,
                                int64_t                 incx)
try
{
    return hipblasConvertStatus(rocblas_ztrmv_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 hipblasConvertDiag(diag),
                                                 n,
                                                 (rocblas_double_complex*)A,
                                                 lda,
                                                 (rocblas_double_complex*)x,
                                                 incx));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// trmv_batched
hipblasStatus_t hipblasStrmvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    hipblasOperation_t transA,
                                    hipblasDiagType_t  diag,
                                    int                n,
                                    const float* const A[],
                                    int                lda,
                                    float* const       x[],
                                    int                incx,
                                    int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_strmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtrmvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    hipblasOperation_t  transA,
                                    hipblasDiagType_t   diag,
                                    int                 n,
                                    const double* const A[],
                                    int                 lda,
                                    double* const       x[],
                                    int                 incx,
                                    int                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_dtrmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtrmvBatched(hipblasHandle_t         handle,
                                    hipblasFillMode_t       uplo,
                                    hipblasOperation_t      transA,
                                    hipblasDiagType_t       diag,
                                    int                     n,
                                    const hipComplex* const A[],
                                    int                     lda,
                                    hipComplex* const       x[],
                                    int                     incx,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_ctrmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      (rocblas_float_complex**)A,
                                                      lda,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtrmvBatched(hipblasHandle_t               handle,
                                    hipblasFillMode_t             uplo,
                                    hipblasOperation_t            transA,
                                    hipblasDiagType_t             diag,
                                    int                           n,
                                    const hipDoubleComplex* const A[],
                                    int                           lda,
                                    hipDoubleComplex* const       x[],
                                    int                           incx,
                                    int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_ztrmv_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      (rocblas_double_complex**)A,
                                                      lda,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// trmv_batched_64
hipblasStatus_t hipblasStrmvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       hipblasOperation_t transA,
                                       hipblasDiagType_t  diag,
                                       int64_t            n,
                                       const float* const A[],
                                       int64_t            lda,
                                       float* const       x[],
                                       int64_t            incx,
                                       int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_strmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         A,
                                                         lda,
                                                         x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtrmvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       hipblasOperation_t  transA,
                                       hipblasDiagType_t   diag,
                                       int64_t             n,
                                       const double* const A[],
                                       int64_t             lda,
                                       double* const       x[],
                                       int64_t             incx,
                                       int64_t             batchCount)
try
{
    return hipblasConvertStatus(rocblas_dtrmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         A,
                                                         lda,
                                                         x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtrmvBatched_64(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       hipblasDiagType_t       diag,
                                       int64_t                 n,
                                       const hipComplex* const A[],
                                       int64_t                 lda,
                                       hipComplex* const       x[],
                                       int64_t                 incx,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_ctrmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         (rocblas_float_complex**)A,
                                                         lda,
                                                         (rocblas_float_complex**)x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtrmvBatched_64(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       hipblasOperation_t            transA,
                                       hipblasDiagType_t             diag,
                                       int64_t                       n,
                                       const hipDoubleComplex* const A[],
                                       int64_t                       lda,
                                       hipDoubleComplex* const       x[],
                                       int64_t                       incx,
                                       int64_t                       batchCount)
try
{
    return hipblasConvertStatus(rocblas_ztrmv_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         hipblasConvertDiag(diag),
                                                         n,
                                                         (rocblas_double_complex**)A,
                                                         lda,
                                                         (rocblas_double_complex**)x,
                                                         incx,
                                                         batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// trmv_strided_batched
hipblasStatus_t hipblasStrmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const float*       A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           float*             x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_strmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              A,
                                                              lda,
                                                              stride_a,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtrmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const double*      A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           double*            x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_dtrmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              A,
                                                              lda,
                                                              stride_a,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtrmvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const hipComplex*  A,
                                           int                lda,
                                           hipblasStride      stride_a,
                                           hipComplex*        x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batchCount)
try
{
    return hipblasConvertStatus(rocblas_ctrmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              (rocblas_float_complex*)A,
                                                              lda,
                                                              stride_a,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtrmvStridedBatched(hipblasHandle_t         handle,
                                           hipblasFillMode_t       uplo,
                                           hipblasOperation_t      transA,
                                           hipblasDiagType_t       diag,
                                           int                     n,
                                           const hipDoubleComplex* A,
                                           int                     lda,
                                           hipblasStride           stride_a,
                                           hipDoubleComplex*       x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_ztrmv_strided_batched((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              (rocblas_double_complex*)A,
                                                              lda,
                                                              stride_a,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// trmv_strided_batched_64
hipblasStatus_t hipblasStrmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const float*       A,
                                              int64_t            lda,
                                              hipblasStride      stride_a,
                                              float*             x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_strmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 A,
                                                                 lda,
                                                                 stride_a,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtrmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const double*      A,
                                              int64_t            lda,
                                              hipblasStride      stride_a,
                                              double*            x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_dtrmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 A,
                                                                 lda,
                                                                 stride_a,
                                                                 x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtrmvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const hipComplex*  A,
                                              int64_t            lda,
                                              hipblasStride      stride_a,
                                              hipComplex*        x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batchCount)
try
{
    return hipblasConvertStatus(rocblas_ctrmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 (rocblas_float_complex*)A,
                                                                 lda,
                                                                 stride_a,
                                                                 (rocblas_float_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtrmvStridedBatched_64(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              hipblasOperation_t      transA,
                                              hipblasDiagType_t       diag,
                                              int64_t                 n,
                                              const hipDoubleComplex* A,
                                              int64_t                 lda,
                                              hipblasStride           stride_a,
                                              hipDoubleComplex*       x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_ztrmv_strided_batched_64((rocblas_handle)handle,
                                                                 hipblasConvertFill(uplo),
                                                                 hipblasConvertOperation(transA),
                                                                 hipblasConvertDiag(diag),
                                                                 n,
                                                                 (rocblas_double_complex*)A,
                                                                 lda,
                                                                 stride_a,
                                                                 (rocblas_double_complex*)x,
                                                                 incx,
                                                                 stridex,
                                                                 batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// trsv
hipblasStatus_t hipblasStrsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const float*       A,
                             int                lda,
                             float*             x,
                             int                incx)
try
{
    return HIPBLAS_DEMAND_ALLOC(hipblasConvertStatus(rocblas_strsv((rocblas_handle)handle,
                                                                   hipblasConvertFill(uplo),
                                                                   hipblasConvertOperation(transA),
                                                                   hipblasConvertDiag(diag),
                                                                   n,
                                                                   A,
                                                                   lda,
                                                                   x,
                                                                   incx)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtrsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const double*      A,
                             int                lda,
                             double*            x,
                             int                incx)
try
{
    return HIPBLAS_DEMAND_ALLOC(hipblasConvertStatus(rocblas_dtrsv((rocblas_handle)handle,
                                                                   hipblasConvertFill(uplo),
                                                                   hipblasConvertOperation(transA),
                                                                   hipblasConvertDiag(diag),
                                                                   n,
                                                                   A,
                                                                   lda,
                                                                   x,
                                                                   incx)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtrsv(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             hipblasDiagType_t  diag,
                             int                n,
                             const hipComplex*  A,
                             int                lda,
                             hipComplex*        x,
                             int                incx)
try
{
    return HIPBLAS_DEMAND_ALLOC(hipblasConvertStatus(rocblas_ctrsv((rocblas_handle)handle,
                                                                   hipblasConvertFill(uplo),
                                                                   hipblasConvertOperation(transA),
                                                                   hipblasConvertDiag(diag),
                                                                   n,
                                                                   (rocblas_float_complex*)A,
                                                                   lda,
                                                                   (rocblas_float_complex*)x,
                                                                   incx)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtrsv(hipblasHandle_t         handle,
                             hipblasFillMode_t       uplo,
                             hipblasOperation_t      transA,
                             hipblasDiagType_t       diag,
                             int                     n,
                             const hipDoubleComplex* A,
                             int                     lda,
                             hipDoubleComplex*       x,
                             int                     incx)
try
{
    return HIPBLAS_DEMAND_ALLOC(hipblasConvertStatus(rocblas_ztrsv((rocblas_handle)handle,
                                                                   hipblasConvertFill(uplo),
                                                                   hipblasConvertOperation(transA),
                                                                   hipblasConvertDiag(diag),
                                                                   n,
                                                                   (rocblas_double_complex*)A,
                                                                   lda,
                                                                   (rocblas_double_complex*)x,
                                                                   incx)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// trsv_64
hipblasStatus_t hipblasStrsv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const float*       A,
                                int64_t            lda,
                                float*             x,
                                int64_t            incx)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_strsv_64((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              A,
                                              lda,
                                              x,
                                              incx)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtrsv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const double*      A,
                                int64_t            lda,
                                double*            x,
                                int64_t            incx)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_dtrsv_64((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              A,
                                              lda,
                                              x,
                                              incx)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtrsv_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                hipblasDiagType_t  diag,
                                int64_t            n,
                                const hipComplex*  A,
                                int64_t            lda,
                                hipComplex*        x,
                                int64_t            incx)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_ctrsv_64((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              (rocblas_float_complex*)A,
                                              lda,
                                              (rocblas_float_complex*)x,
                                              incx)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtrsv_64(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                hipblasDiagType_t       diag,
                                int64_t                 n,
                                const hipDoubleComplex* A,
                                int64_t                 lda,
                                hipDoubleComplex*       x,
                                int64_t                 incx)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_ztrsv_64((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              hipblasConvertDiag(diag),
                                              n,
                                              (rocblas_double_complex*)A,
                                              lda,
                                              (rocblas_double_complex*)x,
                                              incx)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// trsv_batched
hipblasStatus_t hipblasStrsvBatched(hipblasHandle_t    handle,
                                    hipblasFillMode_t  uplo,
                                    hipblasOperation_t transA,
                                    hipblasDiagType_t  diag,
                                    int                n,
                                    const float* const A[],
                                    int                lda,
                                    float* const       x[],
                                    int                incx,
                                    int                batch_count)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_strsv_batched((rocblas_handle)handle,
                                                   hipblasConvertFill(uplo),
                                                   hipblasConvertOperation(transA),
                                                   hipblasConvertDiag(diag),
                                                   n,
                                                   A,
                                                   lda,
                                                   x,
                                                   incx,
                                                   batch_count)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtrsvBatched(hipblasHandle_t     handle,
                                    hipblasFillMode_t   uplo,
                                    hipblasOperation_t  transA,
                                    hipblasDiagType_t   diag,
                                    int                 n,
                                    const double* const A[],
                                    int                 lda,
                                    double* const       x[],
                                    int                 incx,
                                    int                 batch_count)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_dtrsv_batched((rocblas_handle)handle,
                                                   hipblasConvertFill(uplo),
                                                   hipblasConvertOperation(transA),
                                                   hipblasConvertDiag(diag),
                                                   n,
                                                   A,
                                                   lda,
                                                   x,
                                                   incx,
                                                   batch_count)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtrsvBatched(hipblasHandle_t         handle,
                                    hipblasFillMode_t       uplo,
                                    hipblasOperation_t      transA,
                                    hipblasDiagType_t       diag,
                                    int                     n,
                                    const hipComplex* const A[],
                                    int                     lda,
                                    hipComplex* const       x[],
                                    int                     incx,
                                    int                     batch_count)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_ctrsv_batched((rocblas_handle)handle,
                                                   hipblasConvertFill(uplo),
                                                   hipblasConvertOperation(transA),
                                                   hipblasConvertDiag(diag),
                                                   n,
                                                   (rocblas_float_complex**)A,
                                                   lda,
                                                   (rocblas_float_complex**)x,
                                                   incx,
                                                   batch_count)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtrsvBatched(hipblasHandle_t               handle,
                                    hipblasFillMode_t             uplo,
                                    hipblasOperation_t            transA,
                                    hipblasDiagType_t             diag,
                                    int                           n,
                                    const hipDoubleComplex* const A[],
                                    int                           lda,
                                    hipDoubleComplex* const       x[],
                                    int                           incx,
                                    int                           batch_count)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_ztrsv_batched((rocblas_handle)handle,
                                                   hipblasConvertFill(uplo),
                                                   hipblasConvertOperation(transA),
                                                   hipblasConvertDiag(diag),
                                                   n,
                                                   (rocblas_double_complex**)A,
                                                   lda,
                                                   (rocblas_double_complex**)x,
                                                   incx,
                                                   batch_count)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// trsv_batched_64
hipblasStatus_t hipblasStrsvBatched_64(hipblasHandle_t    handle,
                                       hipblasFillMode_t  uplo,
                                       hipblasOperation_t transA,
                                       hipblasDiagType_t  diag,
                                       int64_t            n,
                                       const float* const A[],
                                       int64_t            lda,
                                       float* const       x[],
                                       int64_t            incx,
                                       int64_t            batch_count)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_strsv_batched_64((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      batch_count)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtrsvBatched_64(hipblasHandle_t     handle,
                                       hipblasFillMode_t   uplo,
                                       hipblasOperation_t  transA,
                                       hipblasDiagType_t   diag,
                                       int64_t             n,
                                       const double* const A[],
                                       int64_t             lda,
                                       double* const       x[],
                                       int64_t             incx,
                                       int64_t             batch_count)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_dtrsv_batched_64((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      A,
                                                      lda,
                                                      x,
                                                      incx,
                                                      batch_count)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtrsvBatched_64(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       hipblasDiagType_t       diag,
                                       int64_t                 n,
                                       const hipComplex* const A[],
                                       int64_t                 lda,
                                       hipComplex* const       x[],
                                       int64_t                 incx,
                                       int64_t                 batch_count)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_ctrsv_batched_64((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      (rocblas_float_complex**)A,
                                                      lda,
                                                      (rocblas_float_complex**)x,
                                                      incx,
                                                      batch_count)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtrsvBatched_64(hipblasHandle_t               handle,
                                       hipblasFillMode_t             uplo,
                                       hipblasOperation_t            transA,
                                       hipblasDiagType_t             diag,
                                       int64_t                       n,
                                       const hipDoubleComplex* const A[],
                                       int64_t                       lda,
                                       hipDoubleComplex* const       x[],
                                       int64_t                       incx,
                                       int64_t                       batch_count)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_ztrsv_batched_64((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      hipblasConvertDiag(diag),
                                                      n,
                                                      (rocblas_double_complex**)A,
                                                      lda,
                                                      (rocblas_double_complex**)x,
                                                      incx,
                                                      batch_count)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// trsv_strided_batched
hipblasStatus_t hipblasStrsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const float*       A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           float*             x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batch_count)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_strsv_strided_batched((rocblas_handle)handle,
                                                           hipblasConvertFill(uplo),
                                                           hipblasConvertOperation(transA),
                                                           hipblasConvertDiag(diag),
                                                           n,
                                                           A,
                                                           lda,
                                                           strideA,
                                                           x,
                                                           incx,
                                                           stridex,
                                                           batch_count)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtrsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const double*      A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           double*            x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batch_count)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_dtrsv_strided_batched((rocblas_handle)handle,
                                                           hipblasConvertFill(uplo),
                                                           hipblasConvertOperation(transA),
                                                           hipblasConvertDiag(diag),
                                                           n,
                                                           A,
                                                           lda,
                                                           strideA,
                                                           x,
                                                           incx,
                                                           stridex,
                                                           batch_count)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtrsvStridedBatched(hipblasHandle_t    handle,
                                           hipblasFillMode_t  uplo,
                                           hipblasOperation_t transA,
                                           hipblasDiagType_t  diag,
                                           int                n,
                                           const hipComplex*  A,
                                           int                lda,
                                           hipblasStride      strideA,
                                           hipComplex*        x,
                                           int                incx,
                                           hipblasStride      stridex,
                                           int                batch_count)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_ctrsv_strided_batched((rocblas_handle)handle,
                                                           hipblasConvertFill(uplo),
                                                           hipblasConvertOperation(transA),
                                                           hipblasConvertDiag(diag),
                                                           n,
                                                           (rocblas_float_complex*)A,
                                                           lda,
                                                           strideA,
                                                           (rocblas_float_complex*)x,
                                                           incx,
                                                           stridex,
                                                           batch_count)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtrsvStridedBatched(hipblasHandle_t         handle,
                                           hipblasFillMode_t       uplo,
                                           hipblasOperation_t      transA,
                                           hipblasDiagType_t       diag,
                                           int                     n,
                                           const hipDoubleComplex* A,
                                           int                     lda,
                                           hipblasStride           strideA,
                                           hipDoubleComplex*       x,
                                           int                     incx,
                                           hipblasStride           stridex,
                                           int                     batch_count)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_ztrsv_strided_batched((rocblas_handle)handle,
                                                           hipblasConvertFill(uplo),
                                                           hipblasConvertOperation(transA),
                                                           hipblasConvertDiag(diag),
                                                           n,
                                                           (rocblas_double_complex*)A,
                                                           lda,
                                                           strideA,
                                                           (rocblas_double_complex*)x,
                                                           incx,
                                                           stridex,
                                                           batch_count)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// trsv_strided_batched_64
hipblasStatus_t hipblasStrsvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const float*       A,
                                              int64_t            lda,
                                              hipblasStride      strideA,
                                              float*             x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batch_count)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_strsv_strided_batched_64((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              A,
                                                              lda,
                                                              strideA,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              batch_count)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasDtrsvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const double*      A,
                                              int64_t            lda,
                                              hipblasStride      strideA,
                                              double*            x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batch_count)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_dtrsv_strided_batched_64((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              A,
                                                              lda,
                                                              strideA,
                                                              x,
                                                              incx,
                                                              stridex,
                                                              batch_count)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasCtrsvStridedBatched_64(hipblasHandle_t    handle,
                                              hipblasFillMode_t  uplo,
                                              hipblasOperation_t transA,
                                              hipblasDiagType_t  diag,
                                              int64_t            n,
                                              const hipComplex*  A,
                                              int64_t            lda,
                                              hipblasStride      strideA,
                                              hipComplex*        x,
                                              int64_t            incx,
                                              hipblasStride      stridex,
                                              int64_t            batch_count)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_ctrsv_strided_batched_64((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              (rocblas_float_complex*)A,
                                                              lda,
                                                              strideA,
                                                              (rocblas_float_complex*)x,
                                                              incx,
                                                              stridex,
                                                              batch_count)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZtrsvStridedBatched_64(hipblasHandle_t         handle,
                                              hipblasFillMode_t       uplo,
                                              hipblasOperation_t      transA,
                                              hipblasDiagType_t       diag,
                                              int64_t                 n,
                                              const hipDoubleComplex* A,
                                              int64_t                 lda,
                                              hipblasStride           strideA,
                                              hipDoubleComplex*       x,
                                              int64_t                 incx,
                                              hipblasStride           stridex,
                                              int64_t                 batch_count)
try
{
    return HIPBLAS_DEMAND_ALLOC(
        hipblasConvertStatus(rocblas_ztrsv_strided_batched_64((rocblas_handle)handle,
                                                              hipblasConvertFill(uplo),
                                                              hipblasConvertOperation(transA),
                                                              hipblasConvertDiag(diag),
                                                              n,
                                                              (rocblas_double_complex*)A,
                                                              lda,
                                                              strideA,
                                                              (rocblas_double_complex*)x,
                                                              incx,
                                                              stridex,
                                                              batch_count)));
}
catch(...)
{
    return hipblas_exception_to_status();
}

//------------------------------------------------------------------------------------------------------------

// herk
hipblasStatus_t hipblasCherk(hipblasHandle_t    handle,
                             hipblasFillMode_t  uplo,
                             hipblasOperation_t transA,
                             int                n,
                             int                k,
                             const float*       alpha,
                             const hipComplex*  A,
                             int                lda,
                             const float*       beta,
                             hipComplex*        C,
                             int                ldc)
try
{
    return hipblasConvertStatus(rocblas_cherk((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              n,
                                              k,
                                              alpha,
                                              (rocblas_float_complex*)A,
                                              lda,
                                              beta,
                                              (rocblas_float_complex*)C,
                                              ldc));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZherk(hipblasHandle_t         handle,
                             hipblasFillMode_t       uplo,
                             hipblasOperation_t      transA,
                             int                     n,
                             int                     k,
                             const double*           alpha,
                             const hipDoubleComplex* A,
                             int                     lda,
                             const double*           beta,
                             hipDoubleComplex*       C,
                             int                     ldc)
try
{
    return hipblasConvertStatus(rocblas_zherk((rocblas_handle)handle,
                                              hipblasConvertFill(uplo),
                                              hipblasConvertOperation(transA),
                                              n,
                                              k,
                                              alpha,
                                              (rocblas_double_complex*)A,
                                              lda,
                                              beta,
                                              (rocblas_double_complex*)C,
                                              ldc));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// herk_64
hipblasStatus_t hipblasCherk_64(hipblasHandle_t    handle,
                                hipblasFillMode_t  uplo,
                                hipblasOperation_t transA,
                                int64_t            n,
                                int64_t            k,
                                const float*       alpha,
                                const hipComplex*  A,
                                int64_t            lda,
                                const float*       beta,
                                hipComplex*        C,
                                int64_t            ldc)
try
{
    return hipblasConvertStatus(rocblas_cherk_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 n,
                                                 k,
                                                 alpha,
                                                 (rocblas_float_complex*)A,
                                                 lda,
                                                 beta,
                                                 (rocblas_float_complex*)C,
                                                 ldc));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZherk_64(hipblasHandle_t         handle,
                                hipblasFillMode_t       uplo,
                                hipblasOperation_t      transA,
                                int64_t                 n,
                                int64_t                 k,
                                const double*           alpha,
                                const hipDoubleComplex* A,
                                int64_t                 lda,
                                const double*           beta,
                                hipDoubleComplex*       C,
                                int64_t                 ldc)
try
{
    return hipblasConvertStatus(rocblas_zherk_64((rocblas_handle)handle,
                                                 hipblasConvertFill(uplo),
                                                 hipblasConvertOperation(transA),
                                                 n,
                                                 k,
                                                 alpha,
                                                 (rocblas_double_complex*)A,
                                                 lda,
                                                 beta,
                                                 (rocblas_double_complex*)C,
                                                 ldc));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// herk_batched
hipblasStatus_t hipblasCherkBatched(hipblasHandle_t         handle,
                                    hipblasFillMode_t       uplo,
                                    hipblasOperation_t      transA,
                                    int                     n,
                                    int                     k,
                                    const float*            alpha,
                                    const hipComplex* const A[],
                                    int                     lda,
                                    const float*            beta,
                                    hipComplex* const       C[],
                                    int                     ldc,
                                    int                     batchCount)
try
{
    return hipblasConvertStatus(rocblas_cherk_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      n,
                                                      k,
                                                      alpha,
                                                      (rocblas_float_complex**)A,
                                                      lda,
                                                      beta,
                                                      (rocblas_float_complex**)C,
                                                      ldc,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

hipblasStatus_t hipblasZherkBatched(hipblasHandle_t               handle,
                                    hipblasFillMode_t             uplo,
                                    hipblasOperation_t            transA,
                                    int                           n,
                                    int                           k,
                                    const double*                 alpha,
                                    const hipDoubleComplex* const A[],
                                    int                           lda,
                                    const double*                 beta,
                                    hipDoubleComplex* const       C[],
                                    int                           ldc,
                                    int                           batchCount)
try
{
    return hipblasConvertStatus(rocblas_zherk_batched((rocblas_handle)handle,
                                                      hipblasConvertFill(uplo),
                                                      hipblasConvertOperation(transA),
                                                      n,
                                                      k,
                                                      alpha,
                                                      (rocblas_double_complex**)A,
                                                      lda,
                                                      beta,
                                                      (rocblas_double_complex**)C,
                                                      ldc,
                                                      batchCount));
}
catch(...)
{
    return hipblas_exception_to_status();
}

// herk_batched_64
hipblasStatus_t hipblasCherkBatched_64(hipblasHandle_t         handle,
                                       hipblasFillMode_t       uplo,
                                       hipblasOperation_t      transA,
                                       int64_t                 n,
                                       int64_t                 k,
                                       const float*            alpha,
                                       const hipComplex* const A[],
                                       int64_t                 lda,
                                       const float*            beta,
                                       hipComplex* const       C[],
                                       int64_t                 ldc,
                                       int64_t                 batchCount)
try
{
    return hipblasConvertStatus(rocblas_cherk_batched_64((rocblas_handle)handle,
                                                         hipblasConvertFill(uplo),
                                                         hipblasConvertOperation(transA),
                                                         n,
                                                         k,
                                                         alpha,
                     