use tombi_syntax::{SyntaxKind::*, T};

use super::Parse;
use crate::{
    ErrorKind::*,
    parse::{
        TS_LINE_END, begin_dangling_comments, end_dangling_comments, invalid_line,
        leading_comments, peek_leading_comments, trailing_comment,
    },
    parser::Parser,
    token_set::TS_NEXT_SECTION,
};

impl Parse for tombi_ast::ArrayOfTable {
    fn parse(p: &mut Parser<'_>) {
        let m = p.start();

        leading_comments(p);

        debug_assert!(p.at(T!("[[")));

        p.eat(T!("[["));

        tombi_ast::Keys::parse(p);

        if !p.eat(T!("]]")) {
            invalid_line(p, ExpectedDoubleBracketEnd);
        }

        trailing_comment(p);

        if !p.at_ts(TS_LINE_END) {
            invalid_line(p, ExpectedLineBreak);
        }
        p.eat(LINE_BREAK);

        begin_dangling_comments(p);

        loop {
            while p.eat(LINE_BREAK) {}
            let n = peek_leading_comments(p);

            if p.nth_at_ts(n, TS_NEXT_SECTION) {
                break;
            }

            tombi_ast::KeyValue::parse(p);

            if !p.at_ts(TS_LINE_END) {
                invalid_line(p, ExpectedLineBreak);
            }
        }

        end_dangling_comments(p, false);

        m.complete(p, ARRAY_OF_TABLE);
    }
}

#[cfg(test)]
mod test {
    use crate::{ErrorKind::*, test_parser};

    test_parser! {
        #[test]
        fn invalid_array_of_table1(
            r#"
            [[]]
            key1 = 1
            key2 = 2
            "#
        ) -> Err([SyntaxError(ExpectedKey, 0:2..0:3)])
    }

    test_parser! {
        #[test]
        fn invalid_array_of_table2(
            r#"
            [[aaa.]]
            key1 = 1
            key2 = 2
            "#
        ) -> Err([SyntaxError(ForbiddenKeysLastPeriod, 0:6..0:7)])
    }

    test_parser! {
        #[test]
        fn invalid_array_of_table3(
            r#"
            [[aaa.bbb
            key1 = 1
            key2 = 2
            "#
        ) -> Err([SyntaxError(ExpectedDoubleBracketEnd, 0:9..1:0)])
    }

    test_parser! {
        #[test]
        fn invalid_array_of_table4(
            r#"
            [[aaa.bbb]
            key1 = 1
            key2 = 2
            "#
        ) -> Err([SyntaxError(ExpectedDoubleBracketEnd, 0:9..0:10)])
    }

    test_parser! {
        #[test]
        fn invalid_array_of_table5(
            r#"
            [[aaa.bbb]]
            key1 = 1 INVALID COMMENT
            key2 = 2
            "#
        ) -> Err([SyntaxError(ExpectedLineBreak, 1:9..1:16)])
    }

    test_parser! {
        #[test]
        fn hex_like_array_of_table_key(
            r#"
            [[0x96f]]
            name = "hex-like"
            "#
        ) -> Ok(_)
    }

    test_parser! {
        #[test]
        fn octal_like_array_of_table_key(
            r#"
            [[0o755]]
            mode = "permissions"
            "#
        ) -> Ok(_)
    }

    test_parser! {
        #[test]
        fn binary_like_array_of_table_key(
            r#"
            [[0b1010]]
            flags = true
            "#
        ) -> Ok(_)
    }
}
