#!/bin/bash
# A script to configure the module's debug log level.
#
# Note: Please do not modify this script directly,
# as modifying this script will invalidate this script.
set -e

# 设计说明：
# DDE 应用默认不输出调试日志。
# - 开启调试时：创建配置文件，设置 DDE_DEBUG_LEVEL 环境变量
# - 关闭调试时：删除配置文件，恢复默认行为
#
# 本脚本配置两个层级：
# 1. /etc/X11/Xsession.d/ - 用户图形会话中的 DDE 应用（包括用户级服务）
# 2. /etc/systemd/system.conf.d/ - 系统级 systemd 服务
#
# 支持的日志级别：debug, info, warning, error, fatal, disable

readonly pkg_name="dde"
readonly xsession_file="/etc/X11/Xsession.d/98-deepin-debug-config-dde"
readonly systemd_system_dir="/etc/systemd/system.conf.d"
readonly systemd_system_file="$systemd_system_dir/98-deepin-debug-dde.conf"

debug="no"

is_root() {
    local current_uid
    current_uid=$(id -u)

    if [ "$current_uid" -ne 0 ]; then
        echo "You need to have root privileges to run this script."
        return 1
    fi
    return 0
}

can_exec() {
    if ! is_root; then
        return 1
    fi
    if [ $# -ne 1 ]; then
        echo "Invalid argument"
        return 1
    fi
    for x in "$@"; do
        case $x in
            debug=*)
                debug=${x#debug=}
                ;;
        esac
    done
    case "${debug}" in
        "on" | "off" | "debug" | "info" | "warning" | "error" | "fatal" | "disable")
            return 0
            ;;
        *)
            echo "Not support ${debug} parameter: $*"
            return 1
            ;;
    esac
}

update_xsession_config() {
    local level=$1
    local config_file=$2

    if [[ "$level" == "off" ]]; then
        if [[ -f "$config_file" ]]; then
            rm -f "$config_file"
        fi
        return
    fi

    if [[ "$level" == "on" ]]; then
        level="debug"
    fi

    echo "export DDE_DEBUG_LEVEL=$level" > "$config_file"
}

update_systemd_config() {
    local level=$1
    local config_file=$2

    if [[ "$level" == "off" ]]; then
        if [[ -f "$config_file" ]]; then
            rm -f "$config_file"
        fi
        return
    fi

    if [[ "$level" == "on" ]]; then
        level="debug"
    fi

    echo "[Manager]" > "$config_file"
    echo "DefaultEnvironment=\"DDE_DEBUG_LEVEL=$level\"" >> "$config_file"
}

main() {
    if ! can_exec "$@"; then
        exit 1
    fi

    # 创建必要的目录
    mkdir -p "$(dirname "$xsession_file")"
    mkdir -p "$systemd_system_dir"

    # 更新 X11 会话配置（用户图形会话和用户级服务）
    update_xsession_config "$debug" "$xsession_file"

    # 更新 systemd 系统级配置（系统服务）
    update_systemd_config "$debug" "$systemd_system_file"

    echo "Debug logs are set to '${debug}' for ${pkg_name}."
}

main "$@"
