#include "util.h"
#include "module_configure.h"
#include <string.h>
#include <errno.h>
#include <getopt.h>
#include <locale.h>
#include <unistd.h>

/**
 * Configuration structure to hold parsed command-line arguments.
 */
typedef struct arg_cfg {
    // Actions
    bool set;              ///< --set: configure debug mode
    bool get;              ///< --get: query debug info
    bool install_dbg;      ///< --install-dbg: install debug package
    bool reload;           ///< --reload: reload configurations

    // Arguments
    char *module_names;             ///< -m: module name(s)
    char *module_types;             ///< -t: module type(s) [not used in current code]
    char *level;                    ///< -l: log level
    char *coredump_arg;             ///< -c: coredump on/off
    char *dbg_pkg_name;             ///< -i: debug package name

    // Sub-commands (derived from combinations)
    bool show_debug_level_of_type;  ///< -l with -g: show level for a module
    bool get_coredump_state;        ///< -c with -g: get coredump status
} arg_cfg;

/**
 * Display usage information.
 * @param cmd The program name.
 */
void showUsage(const char *cmd) {
    printf(N_("Usage: %s [options]\n"), cmd);
    printf(N_("Options:\n"));
    printf(N_("\t-h, --help\t\tno arg, show help\n"));
    printf(N_("\t-s, --set\t\tno arg, configure debug mode, used with --coredump or --level\n"));
    printf(N_("\t-g, --get\t\tno arg, get debug info, used with --coredump or --level\n"));
    printf(N_("\t-c, --coredump\t\toptional arg, switch/get coredump state\n"));
    printf(N_("\t-i, --install-dbg\t\trequire one arg, install debug package, e.g. -i systemd\n"));
    printf(N_("\t-l, --level\t\toptional arg, set/get log level\n"));
    printf(N_("\t-m, --module\t\trequire one arg, specify module to configure, e.g. -m systemd\n"));
    printf(N_("\t-r, --reload\t\tno arg, reload configuration and apply changes\n"));
    printf("\n");
}

/**
 * Cleanup function for _cleanup_ attribute to free arg_cfg and its members.
 * @param g_cfg Pointer to pointer of arg_cfg.
 */
void arg_cfg_unrefp(arg_cfg **g_cfg) {
    if (!g_cfg || !*g_cfg) return;
    arg_cfg *cfg = *g_cfg;

    free(cfg->module_names);
    free(cfg->module_types);
    free(cfg->level);
    free(cfg->coredump_arg);
    free(cfg->dbg_pkg_name);
    free(cfg);
}

/**
 * Validate the parsed configuration for logical consistency.
 * @param cfg The parsed argument configuration.
 * @return true if valid, false otherwise.
 */
bool check_g_cfg_is_valid(const arg_cfg *cfg) {
    if (!cfg) return false;

    int actions = cfg->set + cfg->get + cfg->install_dbg + cfg->reload;
    if (actions != 1) {
        fprintf(stderr, N_("Error: Exactly one action (-s, -g, -i, -r) must be specified.\n"));
        return false;
    }

    if (cfg->set) {
        if (cfg->coredump_arg) {
            // Setting coredump state
            if (cfg->module_names || cfg->module_types || cfg->level) {
                fprintf(stderr, N_("Error: -c cannot be used with -m or -l in --set mode.\n"));
                return false;
            }
            if (strcmp(cfg->coredump_arg, "on") != 0 && strcmp(cfg->coredump_arg, "off") != 0) {
                fprintf(stderr, N_("Error: -c argument must be 'on' or 'off'.\n"));
                return false;
            }
        } else {
            // Setting module log level
            if (!cfg->level) {
                fprintf(stderr, N_("Error: --set requires -l <level> when not using -c.\n"));
                return false;
            }
            if (!cfg->module_names && !cfg->module_types) {
                fprintf(stderr, N_("Error: --set requires -m <name> or equivalent when not using -c.\n"));
                return false;
            }
        }
    }

    if (cfg->get) {
        if (cfg->get_coredump_state) {
            if (cfg->module_names || cfg->module_types || cfg->level) {
                fprintf(stderr, N_("Error: -c cannot be combined with -m or -l in --get mode.\n"));
                return false;
            }
        } else if (cfg->show_debug_level_of_type) {
            if (!cfg->module_names) {
                fprintf(stderr, N_("Error: -l requires -m <name> in --get mode.\n"));
                return false;
            }
        } else {
            if (cfg->module_names || cfg->module_types || cfg->level) {
                fprintf(stderr, N_("Error: -g without -l or -c should have no other parameters.\n"));
                return false;
            }
        }
    }

    if (cfg->install_dbg) {
        if (!cfg->dbg_pkg_name) {
            fprintf(stderr, N_("Error: --install-dbg requires a package name.\n"));
            return false;
        }
    }

    if (cfg->reload) {
        if (cfg->module_names || cfg->module_types || cfg->level ||
            cfg->coredump_arg || cfg->dbg_pkg_name) {
            fprintf(stderr, N_("Error: --reload cannot be used with other parameters.\n"));
            return false;
        }
    }

    return true;
}

/**
 * Parse command-line arguments into the configuration structure.
 * @param argc Argument count.
 * @param argv Argument vector.
 * @param cfg Output configuration structure.
 * @return true on success, false on error.
 */
static bool parse_arguments(int argc, char *argv[], arg_cfg *cfg) {
    const struct option longopts[] = {
        { "module",       required_argument, NULL, 'm' },
        { "level",        no_argument,       NULL, 'l' },
        { "coredump",     no_argument,       NULL, 'c' },
        { "install-dbg",  required_argument, NULL, 'i' },
        { "help",         no_argument,       NULL, 'h' },
        { "set",          no_argument,       NULL, 's' },
        { "get",          no_argument,       NULL, 'g' },
        { "reload",       no_argument,       NULL, 'r' },
        { NULL,           0,                 NULL, 0 }
    };

    int c;
    while ((c = getopt_long(argc, argv, "m:lci:hgsr", longopts, NULL)) != -1) {
        switch (c) {
            case 's':
                cfg->set = true;
                break;
            case 'g':
                cfg->get = true;
                break;
            case 'm':
                cfg->module_names = strdup(optarg);
                if (!cfg->module_names) {
                    fprintf(stderr, N_("Error: Out of memory allocating memory for -m.\n"));
                    return false;
                }
                break;
            case 'l':
                if (optind < argc && argv[optind][0]!='-') {
                    cfg->level = strdup(argv[optind++]);
                    if (!cfg->level) {
                        fprintf(stderr, N_("Error: Out of memory allocating memory for -l.\n"));
                        return false;
                    }
                }
                if (cfg->get) {
                    cfg->show_debug_level_of_type = true;
                }
                break;
            case 'c':
                if (optind < argc && argv[optind][0]!='-') {
                    cfg->coredump_arg = strdup(argv[optind++]);
                    if (!cfg->coredump_arg) {
                        fprintf(stderr, N_("Error: Out of memory allocating memory for -c.\n"));
                        return false;
                    }
                }
                if (cfg->get) {
                    cfg->get_coredump_state = true;
                }
                break;
            case 'i':
                cfg->dbg_pkg_name = strdup(optarg);
                if (!cfg->dbg_pkg_name) {
                    fprintf(stderr, N_("Error: Out of memory allocating memory for -i.\n"));
                    return false;
                }
                cfg->install_dbg = true;
                break;
            case 'r':
                cfg->reload = true;
                break;
            case 'h':
                showUsage(argv[0]);
                exit(OK);
            default:
                fprintf(stderr, N_("Try '%s --help' for more information.\n"), PACKAGE);
                return false;
        }
    }

    if (optind < argc) {
        fprintf(stderr, N_("Unexpected argument: %s\n"), argv[optind]);
        return false;
    }

    return true;
}

/**
 * Execute the command based on parsed configuration.
 * @param cfg The configuration to act upon.
 * @return OK on success, ERROR on failure.
 */
static int execute_command(const arg_cfg *cfg) {
    int r = OK;

    if (cfg->get) {
        if (cfg->show_debug_level_of_type) {
            char *debug_level = NULL;
            r = config_module_get_debug_level_by_types(cfg->module_names, &debug_level);
            if (r < 0) {
                return r;
            }
            fprintf(stdout, "%s\n", debug_level);
            free(debug_level);
        } else if (cfg->get_coredump_state) {
            char *coredump_state = NULL;
            r = config_module_get_debug_level_by_type("coredump", &coredump_state);
            if (r < 0) {
                fprintf(stderr, N_("Error: Failed to get coredump state.\n"));
                return r;
            }
            fprintf(stdout, "%s\n", coredump_state);
            free(coredump_state);
        } else {
            char **names = get_module_names();
            if (names) {
                printf("Support module names:\n");
                for (int i = 0; names[i] != NULL; i++) {
                    printf("\t%s\n", names[i]);
                }
                free(names);
            }
        }
        return OK;
    }

    // Operations requiring root privileges
    if (getuid() != 0) {
        fprintf(stderr, N_("Error: %s: Permission denied.\n"), PACKAGE);
        return ERROR;
    }

    if (cfg->set) {
        if (cfg->module_types) {
            r = config_modules_set_debug_level_by_types(cfg->module_types, cfg->level);
        } else if (cfg->module_names) {
            r = config_module_set_debug_level_by_module_names(cfg->module_names, cfg->level);
        } else if (cfg->coredump_arg) {
            bool open_coredump = (strcmp(cfg->coredump_arg, "on") == 0);
            r = config_system_coredump(open_coredump);
        }
        if (r >= 0) {
            config_module_check_log();  // Only call if previous op succeeded
        }
        if (r < 0) return r;
    } else if (cfg->install_dbg) {
        r = config_modules_install_dbgpkgs(cfg->dbg_pkg_name);
        if (r < 0) return r;
    } else if (cfg->reload) {
        r = config_module_reload_debug_level();
        if (r >= 0) {
            config_module_check_log();
        }
    }

    return OK;
}

/**
 * Main entry point.
 */
int main(int argc, char *argv[]) {
    setlocale(LC_ALL, "");
    bindtextdomain(PACKAGE, LOCALEDIR);
    textdomain(PACKAGE);

    if (argc == 1) {
        showUsage(PACKAGE);
        return OK;
    }

    _cleanup_(arg_cfg_unrefp) arg_cfg *g_cfg = calloc(1, sizeof(arg_cfg));
    if (!g_cfg) {
        fprintf(stderr, N_("Error: Failed to allocate configuration structure: %m\n"));
        goto fail;
    }

    if (!parse_arguments(argc, argv, g_cfg))             goto fail;
    if (!check_g_cfg_is_valid(g_cfg))                    goto fail;
    if (init_module_cfgs(MODULES_DEBUG_CONFIG_PATH)!=OK) goto fail;
    if (execute_command(g_cfg) != OK)                    goto fail;

    fprintf(stdout, N_("Done.\n"));
    return OK;

fail:
    fprintf(stderr, N_("Failed.\n"));
    deinit_module_cfgs();
    return ERROR;
}
