//
// Created by 张雪明 <zhangxueming@uniontech.com> on 2023/11/7.
//
#include "DataBackupWidget.h"
#include <DFontSizeManager>
#include <DPushButton>
#include <DSuggestButton>
#include <QHeaderView>
#include <QDir>
#include <QStandardPaths>
#include <DFileDialog>
#include <DDialog>
#include <QUuid>
#include "common/BaseItemDelegate.h"
#include "utils/Utils.h"

const int TABLE_COL_OPT_ID = 0;
const int TABLE_COL_DIR = 1;
const int TABLE_COL_ACTION = 2;

BackupPathDeleteWidget::BackupPathDeleteWidget(QWidget *parent) : QWidget(parent)
{
    QHBoxLayout *layout = new QHBoxLayout;
    layout->setAlignment(Qt::AlignLeft);
    setLayout(layout);
    m_deleteBtn = new DPushButton;
    m_deleteBtn->setStyleSheet("QPushButton {"
                               "color: #0081FF;"
                               "border: none;"
                               "background-color: transparent;"
                               "image:url(:/resources/icons/delete.svg);"
                               "}");
    //m_deleteBtn->setIcon(QIcon::fromTheme(":/resources/icons/delete.svg"));
    layout->addWidget(m_deleteBtn);
    m_deleteBtn->setAccessibleName("BackupPathDeleteWidget_deleteBtn");
    m_spinner = new DSpinner;
    m_spinner->setFixedSize(24, 24);
    m_spinner->setVisible(false);
    m_spinner->setAccessibleName("BackupPathDeleteWidget_DSpinner");
    layout->addWidget(m_spinner);
    connect(m_deleteBtn, &QPushButton::clicked, this, &BackupPathDeleteWidget::onDelete);
}

void BackupPathDeleteWidget::setBackupPathInfo(const QString &opID, const QString &path)
{
    m_opIDPath.first = opID;
    m_opIDPath.second = path;
}

void BackupPathDeleteWidget::onDelete()
{
//    bool authed = false;
//    if (m_backupInfo.operateType == OperateType::SystemBackup) {
//        authed = Utils::authorization();
//    } else if (m_backupInfo.operateType == OperateType::UserDataBackup) {
//        authed = Utils::checkCommonUserAuthentication();
//    }
//
//    if (!authed) {
//        return;
//    }
//
//    if (m_isRemoving) {
//        return;
//    }
    //m_isRemoving = true;

    // 先修改状态，后发送信号，避免删除失败导致状态无法更正
    m_deleteBtn->setVisible(false);
    m_spinner->setVisible(true);
    m_spinner->start();

    Q_EMIT deleteBackupPathClicked(m_opIDPath.first);
}

void BackupPathDeleteWidget::setDeleteBtnVisible(bool visible)
{
    if (m_deleteBtn != nullptr) {
        m_deleteBtn->setVisible(visible);
    }
}

void BackupPathDeleteWidget::stopSpinner()
{
    if (m_spinner != nullptr) {
        m_spinner->stop();
    }
}

void BackupPathDeleteWidget::setSpinnerVisible(bool visible)
{
    if (m_spinner != nullptr) {
        m_spinner->setVisible(visible);
    }
}

DataBackupWidget::DataBackupWidget(DWidget *parent)
{
    initUI();
    initLayout();
    QString fileName = "/etc/fstab";
    m_fstabInfos = FSTab::getFSTabFromFile(fileName);
    FSTab::getFstabBindDir(m_fstabInfos, m_bindDirMap);
    m_isImmutable = Utils::isImmutableSystem();
}

QString DataBackupWidget::getDirBindPath(const QString &path)
{
    return m_bindDirMap[path];
}

void DataBackupWidget::initUI()
{
    this->setAccessibleName("uosDataBackup_DataBackupWidget");
    m_title = new DLabel(this);
    m_title->setText(tr("Data Backup"));
    m_title->setAccessibleName("uos_DataBackupWidget_Title");
    m_title->setAlignment(Qt::AlignCenter);
    DFontSizeManager::instance()->bind(m_title, DFontSizeManager::T3);
    QFont font = m_title->font();
    font.setWeight(QFont::DemiBold);
    m_title->setFont(font);
    QPalette titlePalette;
    titlePalette.setColor(QPalette::BrightText, QColor(qRgb(0,26,46)));
    m_title->setPalette(titlePalette);

    m_backupDirTitle = new DLabel(this);
    m_backupDirTitle->setText(tr("Select the data to back up"));
    m_backupDirTitle->setAccessibleName("uos_DataBackupWidget_BackupDirTitle");
    m_backupDirTitle->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
    QFont backupDirTitleFont = m_backupDirTitle->font();
    backupDirTitleFont.setWeight(QFont::DemiBold);
    m_backupDirTitle->setFont(backupDirTitleFont);
    QPalette backupDirTitlePalette;
    backupDirTitlePalette.setColor(QPalette::BrightText, QColor(qRgba(0,0,0,0.85)));
    m_backupDirTitle->setPalette(backupDirTitlePalette);

    m_addIconBtn = new DIconButton(this);
    m_addIconBtn->setAccessibleName("uos_DataBackupWidget_addIconButton");
    m_addIconBtn->setIcon(DStyle::SP_IncreaseElement);
    m_addIconBtn->setFixedSize(QSize(36,36));

//    m_deleteIconBtn = new DIconButton(this);
//    m_deleteIconBtn->setAccessibleName("uos_DataBackupWidget_deleteIconButton");
//    m_deleteIconBtn->setIcon(DStyle::SP_DecreaseElement);
//    m_deleteIconBtn->setFixedSize(QSize(36,36));

    m_tableView = new BaseTableView(this);
    m_tableView->setAccessibleName("uos_DataBackupWidget_tableView");
    m_model = new QStandardItemModel;
    m_tableView->setModel(m_model);
    m_tableView->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    auto itemDelegate = new BaseItemDelegate(this);
    m_tableView->setItemDelegate(itemDelegate);

    m_backupDir = new DLabel(this);
    m_backupDir->setText(tr("Backup directory"));
    m_backupDir->setAccessibleName("uos_DataBackupWidget_BackupDirectory");
    m_backupDir->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
    QPalette backupDirPalette;
    backupDirPalette.setColor(QPalette::BrightText, QColor(qRgb(65,77,104)));
    m_backupDir->setPalette(backupDirPalette);
    connect(qApp, &QApplication::fontChanged, this, &DataBackupWidget::onBackDirFontChanged);
    int comboWidth = getComboWidth(m_backupDir->font());

    m_comboBox = new DComboBox(this);
    m_comboBox->setAccessibleName("uos_DataBackupWidget_ComboBox");
    m_comboBox->setAccessibleDescription("uos_DataBackupWidget_ComboBoxDes");
    m_comboBox->setFixedWidth(comboWidth);

    m_noteLabel = new DLabel(this);
    m_noteLabel->setText(tr("Note"));
    m_noteLabel->setAccessibleName("uos_DataBackupWidget_NoteLabel");
    m_noteLabel->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
    QPalette noteLabelPalette;
    noteLabelPalette.setColor(QPalette::BrightText, QColor(qRgb(65,77,104)));
    m_noteLabel->setPalette(noteLabelPalette);

    m_noteEdit = new DLineEdit(this);
    m_noteEdit->setFixedWidth(m_comboBox->width());
#if DTK_VERSION >= DTK_VERSION_CHECK(5, 2, 2, 13)
    m_noteEdit->setPlaceholderText(tr("Optional"));
#endif
    m_noteEdit->setAccessibleName("uos_DataBackupWidget_NoteContents");
    m_noteEdit->setFocusPolicy(Qt::ClickFocus);
    connect(m_noteEdit, &DLineEdit::textChanged, this, &DataBackupWidget::onNoteTextChanged);

    m_spinner = new DSpinner(this);
    m_spinner->setAccessibleName("uos_DataBackupWidget_DSpinner");
    m_spinner->setFocusPolicy(Qt::NoFocus);
    m_spinner->setFixedSize(16,16);
    m_spinner->hide();

    m_tips= new DLabel(this);
    m_tips->setAccessibleName("uos_DataBackupWidget_Tips");
    m_tips->setAlignment(Qt::AlignCenter);
    m_tips->setFocusPolicy(Qt::NoFocus);
    m_tips->setStyleSheet("QLabel {"
                          "color: #FF5736;"
                          "}");

    m_cancelBtn = new DPushButton(this);
    m_cancelBtn->setText(tr("Cancel", "button"));
    m_cancelBtn->setAccessibleName("uos_DataBackupWidget_Cancel");
    m_cancelBtn->setFixedSize(140, 36);

    m_startBtn = new DSuggestButton(this);
    m_startBtn->setText(tr("Start Backup"));
    m_startBtn->setAccessibleName("uos_DataBackupWidget_StartBackup");
    m_startBtn->setFixedSize(140, 36);
    m_startBtn->setEnabled(false);
}

int DataBackupWidget::getComboWidth(const QFont &font)
{
    int backDirLen = m_backupDir->text().length();
    QFontMetrics fontMetrics(m_backupDir->font());
    QRect backDirTextRect = fontMetrics.boundingRect(m_backupDir->text());
    int backDirWidth = backDirTextRect.width();
    int fontHeight = fontMetrics.height();
    int pointSize = font.pointSize();
    int comboWidth = 650 - backDirWidth + backDirLen + pointSize;
    if (fontHeight < 21) {
        comboWidth += 8;
    } else if (fontHeight <= 23) {
        comboWidth += 6;
    } else if (fontHeight < 25) {
        comboWidth += 10;
    } else if (fontHeight >= 27) {
        comboWidth += 3;
    }
    // qWarning()<<"comboWidth: "<<comboWidth<<", fontHeight: "<<fontHeight<<", backDirWidth: "<<backDirWidth<<", pointSize: "<<pointSize;
    return comboWidth;
}

void DataBackupWidget::onBackDirFontChanged(const QFont &font)
{
    int comboWidth = getComboWidth(m_backupDir->font());
    m_comboBox->setFixedWidth(comboWidth);
    m_noteEdit->setFixedWidth(comboWidth);
}

void DataBackupWidget::initLayout()
{
    m_mainVLayout = new QVBoxLayout(this);
    m_mainVLayout->setContentsMargins(62, 20, 62, 10);
    m_mainVLayout->setAlignment(Qt::AlignHCenter);

    m_mainVLayout->addWidget(m_title);

    QHBoxLayout *backupDirTitleIconBtnHLayout = new QHBoxLayout;
    backupDirTitleIconBtnHLayout->addWidget(m_backupDirTitle);
    backupDirTitleIconBtnHLayout->addSpacing(120);
    //backupDirTitleIconBtnHLayout->addWidget(m_deleteIconBtn);
    backupDirTitleIconBtnHLayout->addWidget(m_addIconBtn);
    m_mainVLayout->addLayout(backupDirTitleIconBtnHLayout);
    m_mainVLayout->addSpacing(5);

    initTableView();
    m_mainVLayout->addWidget(m_tableView);

    QHBoxLayout *backupDirComboBoxHLayout = new QHBoxLayout;
    backupDirComboBoxHLayout->addWidget(m_backupDir);
    backupDirComboBoxHLayout->addSpacing(8);
    backupDirComboBoxHLayout->addWidget(m_comboBox);
    m_mainVLayout->addLayout(backupDirComboBoxHLayout);
    m_mainVLayout->addSpacing(10);

    QHBoxLayout *noteHLayout = new QHBoxLayout;
    noteHLayout->addWidget(m_noteLabel);
    noteHLayout->addSpacing(8);
    noteHLayout->addWidget(m_noteEdit);
    m_mainVLayout->addLayout(noteHLayout);
    m_mainVLayout->addSpacing(10);

    QHBoxLayout *loadingHLayout = new QHBoxLayout;
    loadingHLayout->addStretch();
    loadingHLayout->addWidget(m_spinner);
    loadingHLayout->addSpacing(10);
    loadingHLayout->addWidget(m_tips);
    loadingHLayout->addStretch();
    m_mainVLayout->addLayout(loadingHLayout);
    m_mainVLayout->addSpacing(10);

    QHBoxLayout *cancelStartBtnHBoxLayout = new QHBoxLayout;
    cancelStartBtnHBoxLayout->setAlignment(Qt::AlignHCenter);
    cancelStartBtnHBoxLayout->addWidget(m_cancelBtn);
    cancelStartBtnHBoxLayout->addWidget(m_startBtn);
    m_mainVLayout->addLayout(cancelStartBtnHBoxLayout);

    connect(m_addIconBtn, &DIconButton::clicked, this, &DataBackupWidget::onAddBtnClicked);
    connect(m_cancelBtn, &QPushButton::clicked, this, &DataBackupWidget::cancel);
    connect(m_startBtn, &QPushButton::clicked, this, &DataBackupWidget::onStartBackup);
}

void DataBackupWidget::initTableView()
{
    //m_tableView->setFixedHeight(300);
    m_tableView->setEditTriggers(QAbstractItemView::NoEditTriggers);
    m_tableView->setSortingEnabled(false);
    m_tableView->setFocusPolicy(Qt::NoFocus);
    m_tableView->setRowBackgroundDelta(10);

    auto optIdItem = new QStandardItem(tr("ID."));
    optIdItem->setTextAlignment(Qt::AlignVCenter);

    auto directoryItem = new QStandardItem(tr("Directory"));
    directoryItem->setTextAlignment(Qt::AlignVCenter);

    auto actionItem = new QStandardItem(tr("Action"));
    actionItem->setTextAlignment(Qt::AlignVCenter);


    m_model->setHorizontalHeaderItem(TABLE_COL_OPT_ID, optIdItem);
    m_model->setHorizontalHeaderItem(TABLE_COL_DIR, directoryItem);
    m_model->setHorizontalHeaderItem(TABLE_COL_ACTION, actionItem);

//    m_tableView->header()->setSectionResizeMode(QHeaderView::Stretch);
    m_tableView->setColumnWidth(TABLE_COL_OPT_ID, 20);
    m_tableView->setColumnWidth(TABLE_COL_DIR, 610);
    m_tableView->setColumnWidth(TABLE_COL_ACTION, 30);

    m_tableView->hideColumn(TABLE_COL_OPT_ID);

    // 禁止拖动列宽
    QHeaderView *headerView = m_tableView->header();
    headerView->setSectionResizeMode(TABLE_COL_OPT_ID, QHeaderView::ResizeMode::Fixed);
    headerView->setSectionResizeMode(TABLE_COL_DIR, QHeaderView::ResizeMode::Fixed);
    headerView->setSectionResizeMode(TABLE_COL_ACTION, QHeaderView::ResizeMode::Fixed);
}

void DataBackupWidget::setDestDevice(const QJsonObject &jsonObject)
{
    m_comboBox->clear();
    m_partitionList.clear();
    QJsonArray disks = jsonObject.value("disks").toArray();
    for (auto a : disks) {
        QString diskName = a.toObject().value("disk").toString();
        QJsonArray partitions = a.toObject().value("partitions").toArray();
        for (auto p : partitions) {
            Partition partition;
            partition.unmarshal(p.toObject());
            m_partitionList.append(partition);
            m_comboBox->addItem(QString("%1(%2),  %3,  (%4/%5)").arg(diskName).arg(partition.fsType)
                                        .arg(partition.name).arg(Utils::byte2DisplaySize(partition.used)).arg(Utils::byte2DisplaySize(partition.size)));
        }
    }

    if (m_partitionList.isEmpty()) {
        m_startBtn->setEnabled(false);
        this->setTips(tr("Please insert a removable disk, supported format: ext4, btrfs, xfs"));
        this->setTipsStyleSheet("QLabel {"
                                "color: #FF5736;"
                                "}");
        this->setAddIconBtnEnable(false);
    } else {
        this->setTips("");
        this->setAddIconBtnEnable(true);
    }
}

QString DataBackupWidget::getDestDeviceUUID()
{
    if (m_comboBox->count() == m_partitionList.size()) {
        Partition selectPartition = m_partitionList.at(m_comboBox->currentIndex());
        if (!selectPartition.fsTypeSupportedDataBackup) {
            DDialog dlg(this);
            dlg.setMessage(::QObject::tr("The file system of the device is unsupported. Please select one in ext4, btrfs, xfs format."));
            dlg.setIcon(QIcon::fromTheme("dialog-warning"));
            dlg.setAccessibleName("DataBackupWidget_deviceUnsupportedDDialog");
            QRect rect = geometry();
            dlg.move(rect.center());
            dlg.moveToCenter();
            dlg.exec();
            return "";
        }
        return selectPartition.uuid;
    }

    return "";
}

void DataBackupWidget::onAddBtnClicked(bool clicked)
{
    QString userPath = QStandardPaths::writableLocation(QStandardPaths::HomeLocation);
    DFileDialog dialog(this);
    dialog.setFileMode(DFileDialog::ExistingFiles);
    dialog.setDirectory(userPath);
    //dialog.setNameFilter(filter);
    dialog.setOption(QFileDialog::HideNameFilterDetails);
    //dialog.setWindowTitle(tr("Import Photos and Videos"));
    dialog.setAllowMixedSelection(true);
    dialog.setAccessibleName("DataBackupWidget_AddBtnClickedDFileDialog");
    const int mode = dialog.exec();
    if (mode != QDialog::Accepted) {
        return;
    }
    const QStringList &fileList = dialog.selectedFiles();
    if (fileList.isEmpty()) {
        return;
    }

    bool enableStartBtn = false;
    static QString curUserName = Utils::getUserName();
    QString homeBindPath = m_bindDirMap["/home"];
    QString userSupportPath = userPath;
    if (!homeBindPath.isEmpty()) {
        if (homeBindPath.endsWith("/")) {
            userSupportPath = homeBindPath + curUserName;
        } else {
            userSupportPath = homeBindPath + "/" + curUserName;
        }
    }
    QDir homeDir("/home");
    QString realHomeDir = homeDir.canonicalPath();
    if (m_isImmutable) {
        if ("/home" != realHomeDir) {
            userSupportPath = realHomeDir + "/" + curUserName;
        }
    }

    QStringList supportPathList;
    QString homeDirErrMsg;
    QString pathExistErrMsg;
    int existPathNum = 0;
    for (const QString &path : fileList) {
        if (!path.startsWith(userPath) && !path.startsWith(userSupportPath)) {
            homeDirErrMsg = tr("Please select data in the current user's home directory");
        } else {
            auto items = m_model->findItems(path, Qt::MatchExactly, TABLE_COL_DIR);
            if (!items.isEmpty()) {
                ++existPathNum;
                if (pathExistErrMsg.isEmpty()) {
                    pathExistErrMsg = tr("The selected directory is in included in %1").arg(path);
                }
            } else {
                supportPathList.append(path);
            }
        }
    }

    if (existPathNum > 1) {
        pathExistErrMsg = QString("%1 ...").arg(pathExistErrMsg);
    }

    if (!homeDirErrMsg.isEmpty()) {
        DDialog dialog(this);
        dialog.setMessage(homeDirErrMsg);
        dialog.setIcon(QIcon::fromTheme("dialog-warning"));
        dialog.setAccessibleName("DataBackupWidget_AddPathHomeWarningDialog");
        QRect rect = geometry();
        dialog.move(rect.center());
        dialog.moveToCenter();
        dialog.exec();
    }

    if (!pathExistErrMsg.isEmpty()) {
        DDialog dialog(this);
        dialog.setMessage(pathExistErrMsg);
        dialog.setIcon(QIcon::fromTheme("dialog-warning"));
        dialog.setAccessibleName("DataBackupWidget_pathExistErrMsgDialog");
        QRect rect = geometry();
        dialog.move(rect.center());
        dialog.moveToCenter();
        dialog.exec();
    }

    for (const QString &path : supportPathList) {
        int currentRow = m_model->rowCount();
        QString opID = QUuid::createUuid().toString();
        m_model->setItem(currentRow, TABLE_COL_OPT_ID, new QStandardItem(opID));
        auto pathItem = new QStandardItem(path);
        m_model->setItem(currentRow, TABLE_COL_DIR, pathItem);

        auto actionItem = new QStandardItem();
        actionItem->setAccessibleText(opID + "_deleteWidget");
        m_model->setItem(currentRow, TABLE_COL_ACTION, actionItem);
        auto deleteWidget = new BackupPathDeleteWidget;
        deleteWidget->setBackupPathInfo(opID, path);
        m_tableView->setIndexWidget(m_model->index(currentRow, TABLE_COL_ACTION), deleteWidget);
        connect(deleteWidget, &BackupPathDeleteWidget::deleteBackupPathClicked,
                this, &DataBackupWidget::onDeleteBackupPathClicked);
        enableStartBtn = true;
    }

    int n = m_model->rowCount();
    if (n <= 0) {
        m_startBtn->setEnabled(false);
    } else {
        m_startBtn->setEnabled(true);
    }
}

void DataBackupWidget::onDeleteBackupPathClicked(const QString &opID)
{
    auto items = m_model->findItems(opID, Qt::MatchExactly, TABLE_COL_OPT_ID);
    for (auto &item : items) {
        m_model->removeRow(m_model->indexFromItem(item).row());
    }

    int n = m_model->rowCount();
    if (n <= 0) {
        m_startBtn->setEnabled(false);
    }
}

void DataBackupWidget::onStartBackup()
{
    Q_EMIT start(m_noteEdit->text());
}

QStringList DataBackupWidget::getBackupFiles()
{
    QStringList backupFiles;
    auto root = m_model->invisibleRootItem();
    if (nullptr == root) {
        return backupFiles;
    }

    QDir homeDir("/home");
    QString realHomeDir = homeDir.canonicalPath();
    QString homePre;
    if (realHomeDir.startsWith("/var/home")) {
        homePre = "/var";
    }
    int row = 0;
    while (root->child(row) != nullptr) {
        auto item = root->child(row, TABLE_COL_DIR);
        if (item == nullptr) {
            continue;
        }

        if (m_isImmutable && !homePre.isEmpty()) {
            backupFiles <<homePre + item->text();
        } else {
            backupFiles << item->text();
        }
#ifdef QT_DEBUG
        qInfo() << item->text();
#endif
        ++row;
    }

    return backupFiles;
}

void DataBackupWidget::setTips(const QString &tips)
{
    if (nullptr != m_tips) {
        m_tips->setText(tips);
    }
}

void DataBackupWidget::setTipsStyleSheet(const QString &styleSheet)
{
    if (nullptr != m_tips) {
        m_tips->setStyleSheet(styleSheet);
    }
}

void DataBackupWidget::startSpinner()
{
    if (nullptr != m_spinner) {
        m_spinner->show();
        m_spinner->start();
    }
}

void DataBackupWidget::stopSpinner()
{
    if (nullptr != m_spinner) {
        m_spinner->stop();
        m_spinner->hide();
    }
}

void DataBackupWidget::setButtonEnable(bool enable)
{
    if (nullptr != m_startBtn) {
        m_startBtn->setEnabled(enable);
    }

    if (nullptr != m_cancelBtn) {
        m_cancelBtn->setEnabled(enable);
    }

    if (nullptr != m_addIconBtn) {
        m_addIconBtn->setEnabled(enable);
    }

    if (nullptr != m_comboBox) {
        m_comboBox->setEnabled(enable);
    }

    if (nullptr != m_noteEdit) {
        m_noteEdit->setEnabled(enable);
    }
}

QString DataBackupWidget::getNote()
{
    if (nullptr != m_noteEdit) {
        return m_noteEdit->text();
    }

    return "";
}

void DataBackupWidget::resetWidget()
{
    if (nullptr != m_startBtn) {
        m_startBtn->setEnabled(false);
    }

    if (nullptr != m_cancelBtn) {
        m_cancelBtn->setEnabled(true);
    }

    if (nullptr != m_comboBox) {
        m_comboBox->setEnabled(true);
    }

    if (nullptr != m_addIconBtn) {
        m_addIconBtn->setEnabled(true);
    }

    if (nullptr != m_noteEdit) {
        m_noteEdit->clear();
        m_noteEdit->setEnabled(true);
    }

    if (nullptr != m_model) {
        m_model->removeRows(0, m_model->rowCount());
    }

    if (nullptr != m_tips) {
        m_tips->setText("");
    }

    if (nullptr != m_spinner) {
        m_spinner->hide();
        m_spinner->stop();
    }
}

void DataBackupWidget::setAddIconBtnEnable(bool enable)
{
    if (nullptr != m_addIconBtn) {
        m_addIconBtn->setEnabled(enable);
    }
}

void DataBackupWidget::onNoteTextChanged()
{
    const int maxNotesLen = 40;
    QString notes = m_noteEdit->text();
    int len = notes.length();
    if (len > maxNotesLen) {
        QLineEdit * lineEdit = m_noteEdit->lineEdit();
        int position = lineEdit->cursorPosition();
        int diff = len - maxNotesLen;
        notes.remove(position - diff, diff);
        m_noteEdit->setText(notes);
        lineEdit->setCursorPosition(position - diff);
    }
}
