package Sisimai::Reason::MailerError;
use v5.26;
use strict;
use warnings;

sub text  { 'mailererror' }
sub description { 'Email returned due to a mailer program has not exited successfully' }
sub match {
    # Try to match that the given text and regular expressions
    # @param    [String] argv1  String to be matched with regular expressions
    # @return   [Integer]       0: Did not match
    #                           1: Matched
    # @since v4.0.0
    my $class = shift;
    my $argv1 = shift // return 0;

    state $index = [
        ' || exit ',
        'procmail: ',
        'bin/procmail',
        'bin/maidrop',
        'command failed: ',
        'command died with status ',
        'command output:',
        'mailer error',
        'pipe to |/',
        'x-unix; ',
    ];
    return 1 if grep { rindex($argv1, $_) > -1 } @$index;
    return 0;
}

sub true {
    # The bounce reason is mailer error or not
    # @param    [Sisimai::Fact] argvs   Object to be detected the reason
    # @return   [Integer]               1: is mailer error
    #                                   0: is not mailer error
    # @see http://www.ietf.org/rfc/rfc2822.txt
    return 0;
}

1;
__END__

=encoding utf-8

=head1 NAME

Sisimai::Reason::MailerError - Bounce reason is C<mailererror> or not.

=head1 SYNOPSIS

    use Sisimai::Reason::MailerError;
    print Sisimai::Reason::MailerError->match('X-Unix; 255');   # 1

=head1 DESCRIPTION

C<Sisimai::Reason::MailerError> checks the bounce reason is C<mailererror> or not. This class is
called only C<Sisimai::Reason> class.

This is the error that the mailer program has not exited successfully or exited unexpectedly on the
destination mail server.

    X-Actual-Recipient: X-Unix; |/home/kijitora/mail/catch.php
    Diagnostic-Code: X-Unix; 255

=head1 CLASS METHODS

=head2 C<B<text()>>

C<text()> method returns the fixed string C<mailererror>.

    print Sisimai::Reason::MailerError->text;  # mailererror

=head2 C<B<match(I<string>)>>

C<match()> method returns C<1> if the argument matched with patterns defined in this class.

    print Sisimai::Reason::MailerError->match('X-Unix; 255');   # 1

=head2 C<B<true(I<Sisimai::Fact>)>>

C<true()> method returns C<1> if the bounce reason is C<mailererror>. The argument must be C<Sisimai::Fact>
object and this method is called only from C<Sisimai::Reason> class.

=head1 AUTHOR

azumakuniyuki

=head1 COPYRIGHT

Copyright (C) 2014-2017,2020,2021,2023-2025 azumakuniyuki, All rights reserved.

=head1 LICENSE

This software is distributed under The BSD 2-Clause License.

=cut

