#############################################################################
##  targetsQuotient.gi
#############################################################################
##
##  This file is part of the LINS package.
##
##  This file's authors include Friedrich Rober.
##
##  Please refer to the COPYRIGHT file for details.
##
##  SPDX-License-Identifier: GPL-2.0-or-later
##
#############################################################################


#############################################################################
##  LINS_TargetsQuotient
#############################################################################
##  Usage:
##
##  The main function `LowIndexNormalSubgroupsSearch` calls the function
##
##                      `LINS_FindTQuotients`
##
##  with the argument `targets` set to this.
#############################################################################
##  Description:
##
##  This is a pregenerated list following the specifications
##  for the argument `targets` in the function `LINS_FindTQuotients`.
##
##  The list was computed by the code in `addGroup.gi`.
##
##  Let $T$ be a non-abelian simple group.
##  Then the pregenerated list `LINS_TargetsQuotient` contains
##  information on any subgroup $Q$ of $Aut(T x T x ... x T)$,
##  such that $(T x T x ... x T)$ is a subgroup of $Q$
##  and $Q$ acts transitively on the set of copies of $T$,
##  with group order $|Q|$ up to the maximum index bound `LINS_MaxIndex`.
##
##  Let $Q$ be such a group of interest,
##  then the information about $Q$ consists of the following:
##
##  - 1 : the group order $|Q|$
##  - 2 : an index of some non-trivial subgroup $S < Q$,
##        that has trivial core in $Q$
##  - 3 : name of the group $T ^ d$
##
##  The list `LINS_TargetsQuotient` is sorted by information $1$.
#############################################################################

BindGlobal("LINS_TargetsQuotient_Index", 10000000);

BindGlobal("LINS_TargetsQuotient",
List(
[ [        60, "[ (1,2,3,4,5), (3,4,5) ]", "A5^1" ],
  [       120, "[ (4,5), (1,2,3,4,5) ]", "A5^1" ],
  [       168, "[ (3,7,5)(4,8,6), (1,2,6)(3,4,8) ]", "PSL(2,7)^1" ],
  [       336, "[ (3,8,7,6,5,4), (1,2,6)(3,4,8) ]", "PSL(2,7)^1" ],
  [       360, "[ (1,2,3,4,5)(8,10,11,9,12), (4,5,6)(7,12,10)(8,11,9) ]", "A6^1" ],
  [       504, "[ (3,8,6,4,9,7,5), (1,2,3)(4,7,5)(6,9,8) ]", "PSL(2,8)^1" ],
  [       660, "[ (3,11,9,7,5)(4,12,10,8,6), (1,2,8)(3,7,9)(4,10,5)(6,12,11) ]", "PSL(2,11)^1" ],
  [       720, "[ (1,7,2,8,3,9,4,10)(5,11,6,12), (1,2,3,4,5)(8,10,11,9,12) ]", "A6^1" ],
  [       720, "[ (1,9,4,8,3,7,2,10)(5,12,6,11), (1,2,3,4,5)(8,10,11,9,12) ]", "A6^1" ],
  [       720, "[ (5,6)(7,9)(8,10)(11,12), (1,2,3,4,5)(8,10,11,9,12) ]", "A6^1" ],
  [      1092, "[ (3,13,11,9,7,5)(4,14,12,10,8,6), (1,2,9)(3,8,10)(4,5,12)(6,13,14) ]", "PSL(2,13)^1" ],
  [      1320, "[ (3,12,11,10,9,8,7,6,5,4), (1,2,8)(3,7,9)(4,10,5)(6,12,11) ]", "PSL(2,11)^1" ],
  [      1440, "[ (1,7,2,8,3,9,4,10)(5,11,6,12), (5,6)(7,9)(8,10)(11,12), (1,2,3,4,5)(8,10,11,9,12) ]", "A6^1" ],
  [      1512, "[ (4,5,7)(6,9,8), (3,8,6,4,9,7,5), (1,2,3)(4,7,5)(6,9,8) ]", "PSL(2,8)^1" ],
  [      2184, "[ (3,14,13,12,11,10,9,8,7,6,5,4), (1,2,9)(3,8,10)(4,5,12)(6,13,14) ]", "PSL(2,13)^1" ],
  [      2448, "[ (3,17,15,13,11,9,7,5)(4,18,16,14,12,10,8,6), (1,2,11)(3,13,9)(4,15,6)(5,8,12)(7,18,16)(10,14,17) ]", "PSL(2,17)^1" ],
  [      2520, "[ (1,2,3,4,5,6,7), (5,6,7) ]", "A7^1" ],
  [      3420, "[ (3,19,17,15,13,11,9,7,5)(4,20,18,16,14,12,10,8,6), (1,2,12)(3,11,13)(4,17,6)(5,14,8)(7,20,18)(10,19,16) ]", "PSL(2,19)^1" ],
  [      4080, "[ (3,16,14,13,11,4,8,6,17,15,5,12,10,9,7), (1,2,3)(6,14,8)(7,11,10)(9,17,15)(12,16,13) ]", "PSL(2,16)^1" ],
  [      4896, "[ (3,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,11)(3,13,9)(4,15,6)(5,8,12)(7,18,16)(10,14,17) ]", "PSL(2,17)^1" ],
  [      5040, "[ (6,7), (1,2,3,4,5,6,7) ]", "A7^1" ],
  [      5616, "[ (5,8,11)(6,9,12)(7,10,13)(15,24,21)(16,26,22)(17,25,23), (1,2,5)(3,8,7)(4,11,6)(9,10,13)(14,15,18)(16,21,20)(17,24,19)(22,23,26) ]", "PSL(3,3)^1" ],
  [      6048, "[ (3,4,6,10,12,18,19,23)(5,8,13,20,17,11,16,7)(9,14,21,15,22,24,26,28)(25,27), (1,2,3,5,9,15,16,18)(4,7,12,19,24,27,26,23)(6,11,17,13,8,10,14,21)(20,25) ]", "PSU(3,3)^1" ],
  [      6072, "[ (3,23,21,19,17,15,13,11,9,7,5)(4,24,22,20,18,16,14,12,10,8,6), (1,2,14)(3,12,16)(4,18,9)(5,20,6)(7,13,11)(8,23,22)(10,24,19)(15,21,17) ]", "PSL(2,23)^1" ],
  [      6840, "[ (3,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,12)(3,11,13)(4,17,6)(5,14,8)(7,20,18)(10,19,16) ]", "PSL(2,19)^1" ],
  [      7200, "[ (1,6,5,7,4,8,3,9,2,10), (8,9,10) ]", "A5^2" ],
  [      7800, "[ (3,25,23,6,20,18,5,15,13,4,10,8)(7,26,24,22,21,19,17,16,14,12,11,9), (1,2,5)(3,4,6)(7,18,14)(8,22,9)(10,17,12)(11,25,24)(15,26,19)(16,23,21) ]", "PSL(2,25)^1" ],
  [      7920, "[ (1,2,3,4,5,6,7,8,9,10,11), (3,7,11,8)(4,10,5,6) ]", "M11^1" ],
  [      8160, "[ (6,9)(7,12)(8,15)(10,13)(11,16)(14,17), (3,16,14,13,11,4,8,6,17,15,5,12,10,9,7), (1,2,3)(6,14,8)(7,11,10)(9,17,15)(12,16,13) ]", "PSL(2,16)^1" ],
  [      9828, "[ (3,27,25,23,21,19,17,16,14,12,10,8,6)(4,15,13,11,9,7,5,28,26,24,22,20,18), (1,2,4)(5,8,24)(6,21,10)(7,16,15)(9,25,28)(11,13,14)(12,27,23)(17,26,18)(19,20,22) ]", "PSL(2,27)^1" ],
  [     11232, "[ (1,14)(2,15)(3,16)(4,17)(5,18)(6,19)(7,20)(8,21)(9,22)(10,23)(11,24)(12,25)(13,26), (5,8,11)(6,9,12)(7,10,13)(15,24,21)(16,26,22)(17,25,23), (1,2,5)(3,8,7)(4,11,6)(9,10,13)(14,15,18)(16,21,20)(17,24,19)(22,23,26) ]", "PSL(3,3)^1" ],
  [     12096, "[ (3,28,12,15)(4,21,18,26)(5,20,17,7)(6,24,19,14)(8,16,11,13)(9,23,22,10), (3,4,6,10,12,18,19,23)(5,8,13,20,17,11,16,7)(9,14,21,15,22,24,26,28)(25,27), (1,2,3,5,9,15,16,18)(4,7,12,19,24,27,26,23)(6,11,17,13,8,10,14,21)(20,25) ]", "PSU(3,3)^1" ],
  [     12144, "[ (3,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,14)(3,12,16)(4,18,9)(5,20,6)(7,13,11)(8,23,22)(10,24,19)(15,21,17) ]", "PSL(2,23)^1" ],
  [     12180, "[ (3,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,17)(3,16,18)(4,12,21)(5,23,9)(6,7,24)(8,15,14)(10,27,28)(11,29,25)(13,22,30)(19,26,20) ]", "PSL(2,29)^1" ],
  [     14400, "[ (1,6,5,7,4,8,3,10)(2,9), (4,5)(9,10), (8,9,10) ]", "A5^2" ],
  [     14400, "[ (1,6,5,7,4,8,3,9,2,10), (4,5)(9,10), (1,5,4,3,2) ]", "A5^2" ],
  [     14880, "[ (3,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,18)(3,24,12)(4,30,5)(6,32,31)(7,10,22)(8,15,16)(9,11,19)(14,26,29)(17,25,27)(20,21,28) ]", "PSL(2,31)^1" ],
  [     15600, "[ (3,26,25,24,23,22,6,21,20,19,18,17,5,16,15,14,13,12,4,11,10,9,8,7), (1,2,5)(3,4,6)(7,18,14)(8,22,9)(10,17,12)(11,25,24)(15,26,19)(16,23,21) ]", "PSL(2,25)^1" ],
  [     15600, "[ (3,26,6,21,5,16,4,11)(7,10,22,25,17,20,12,15)(8,14,23,9,18,24,13,19), (3,25,23,6,20,18,5,15,13,4,10,8)(7,26,24,22,21,19,17,16,14,12,11,9), (1,2,5)(3,4,6)(7,18,14)(8,22,9)(10,17,12)(11,25,24)(15,26,19)(16,23,21) ]", "PSL(2,25)^1" ],
  [     15600, "[ (7,11)(8,15)(9,19)(10,23)(12,16)(13,20)(14,24)(17,21)(18,25)(22,26), (3,25,23,6,20,18,5,15,13,4,10,8)(7,26,24,22,21,19,17,16,14,12,11,9), (1,2,5)(3,4,6)(7,18,14)(8,22,9)(10,17,12)(11,25,24)(15,26,19)(16,23,21) ]", "PSL(2,25)^1" ],
  [     16320, "[ (4,5)(6,7,9,12)(8,10,15,13)(11,17,16,14), (6,9)(7,12)(8,15)(10,13)(11,16)(14,17), (3,16,14,13,11,4,8,6,17,15,5,12,10,9,7), (1,2,3)(6,14,8)(7,11,10)(9,17,15)(12,16,13) ]", "PSL(2,16)^1" ],
  [     19656, "[ (3,28,27,26,25,24,23,22,21,20,19,18,17,4,16,15,14,13,12,11,10,9,8,7,6,5), (1,2,4)(5,8,24)(6,21,10)(7,16,15)(9,25,28)(11,13,14)(12,27,23)(17,26,18)(19,20,22) ]", "PSL(2,27)^1" ],
  [     20160, "[ (1,2,3,4,5,6,7), (6,7,8) ]", "A8^1" ],
  [     20160, "[ (3,4,5)(7,9,8)(10,14,18)(11,17,20)(12,15,21)(13,16,19)(24,26,25)(28,29,30)(31,39,35)(32,41,38)(33,42,36)(34,40,37), (1,2,6,7,11,3,10)(4,14,8,15,16,20,13)(5,18,9,19,21,17,12)(22,23,28,24,32,31,27)(25,36,41,42,34,39,30)(26,40,38,37,33,35,29) ]", "PSL(3,4)^1" ],
  [     24360, "[ (3,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,17)(3,16,18)(4,12,21)(5,23,9)(6,7,24)(8,15,14)(10,27,28)(11,29,25)(13,22,30)(19,26,20) ]", "PSL(2,29)^1" ],
  [     25308, "[ (3,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,21)(3,20,22)(4,31,10)(5,34,6)(7,14,24)(8,37,36)(9,23,13)(11,38,32)(12,16,17)(18,28,35)(19,33,29)(25,26,30) ]", "PSL(2,37)^1" ],
  [     25920, "[ (3,4)(6,7)(9,10)(12,13)(17,20)(18,21)(19,22)(23,32)(24,33)(25,34)(26,38)(27,39)(28,40)(29,35)(30,36)(31,37), (1,5,14,17,27,22,19,36,3)(2,6,32)(4,7,23,20,37,13,16,26,40)(8,24,29,30,39,10,33,11,34)(9,15,35)(12,25,38)(21,28,31) ]", "PSp(4,3)^1" ],
  [     28800, "[ (1,6,5,7,4,8,3,9,2,10), (4,5), (9,10), (8,9,10) ]", "A5^2" ],
  [     29120, "[ (1,28,10,44)(3,50,11,42)(4,43,53,64)(5,9,39,52)(6,36,63,13)(7,51,60,57)(8,33,37,16)(12,24,55,29)(14,30,48,47)(15,19,61,54)(17,59,22,62)(18,23,34,31)(20,38,49,25)(21,26,45,58)(27,32,41,65)(35,46,40,56), (1,2)(3,10)(4,42)(5,18)(6,50)(7,26)(8,58)(9,34)(12,28)(13,45)(14,44)(15,23)(16,31)(17,21)(19,39)(20,38)(22,25)(24,61)(27,60)(29,65)(30,55)(32,33)(35,52)(36,49)(37,59)(40,54)(41,62)(43,53)(46,48)(47,56)(51,63)(57,64) ]", "Sz(8)^1" ],
  [     29484, "[ (5,7,13)(6,10,21)(8,16,14)(9,18,22)(11,24,15)(12,27,23)(17,19,25)(20,28,26), (3,27,25,23,21,19,17,16,14,12,10,8,6)(4,15,13,11,9,7,5,28,26,24,22,20,18), (1,2,4)(5,8,24)(6,21,10)(7,16,15)(9,25,28)(11,13,14)(12,27,23)(17,26,18)(19,20,22) ]", "PSL(2,27)^1" ],
  [     29760, "[ (3,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,18)(3,24,12)(4,30,5)(6,32,31)(7,10,22)(8,15,16)(9,11,19)(14,26,29)(17,25,27)(20,21,28) ]", "PSL(2,31)^1" ],
  [     31200, "[ (7,11)(8,15)(9,19)(10,23)(12,16)(13,20)(14,24)(17,21)(18,25)(22,26), (3,26,25,24,23,22,6,21,20,19,18,17,5,16,15,14,13,12,4,11,10,9,8,7), (1,2,5)(3,4,6)(7,18,14)(8,22,9)(10,17,12)(11,25,24)(15,26,19)(16,23,21) ]", "PSL(2,25)^1" ],
  [     32736, "[ (3,32,30,28,26,24,22,20,18,16,14,12,10,8,6,4,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5), (1,2,3)(4,16,20)(5,29,6)(7,24,9)(8,32,31)(10,12,18)(11,14,15)(13,30,28)(17,21,33)(19,25,27)(22,23,26) ]", "PSL(2,32)^1" ],
  [     34440, "[ (3,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,23)(3,37,9)(4,24,21)(5,31,13)(6,36,7)(8,12,29)(10,40,39)(11,20,18)(14,16,19)(15,41,33)(17,34,38)(22,42,25)(26,35,28)(27,30,32) ]", "PSL(2,41)^1" ],
  [     39732, "[ (3,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,24)(3,39,9)(4,12,35)(5,14,32)(6,19,26)(7,33,11)(8,25,18)(10,20,21)(13,36,44)(15,41,37)(16,34,43)(22,29,42)(23,40,30)(27,28,38) ]", "PSL(2,43)^1" ],
  [     40320, "[ (1,22)(2,23)(3,24)(4,25)(5,26)(6,27)(7,28)(8,29)(9,30)(10,31)(11,32)(12,33)(13,34)(14,35)(15,36)(16,37)(17,38)(18,39)(19,40)(20,41)(21,42), (3,4,5)(7,9,8)(10,14,18)(11,17,20)(12,15,21)(13,16,19)(24,26,25)(28,29,30)(31,39,35)(32,41,38)(33,42,36)(34,40,37), (1,2,6,7,11,3,10)(4,14,8,15,16,20,13)(5,18,9,19,21,17,12)(22,23,28,24,32,31,27)(25,36,41,42,34,39,30)(26,40,38,37,33,35,29) ]", "PSL(3,4)^1" ],
  [     40320, "[ (1,22)(2,23)(3,24)(4,26)(5,25)(6,27)(7,28)(8,30)(9,29)(10,31)(11,32)(12,34)(13,33)(14,39)(15,40)(16,42)(17,41)(18,35)(19,36)(20,38)(21,37), (3,4,5)(7,9,8)(10,14,18)(11,17,20)(12,15,21)(13,16,19)(24,26,25)(28,29,30)(31,39,35)(32,41,38)(33,42,36)(34,40,37), (1,2,6,7,11,3,10)(4,14,8,15,16,20,13)(5,18,9,19,21,17,12)(22,23,28,24,32,31,27)(25,36,41,42,34,39,30)(26,40,38,37,33,35,29) ]", "PSL(3,4)^1" ],
  [     40320, "[ (4,5)(8,9)(12,13)(14,18)(15,19)(16,21)(17,20)(25,26)(29,30)(33,34)(35,39)(36,40)(37,42)(38,41), (3,4,5)(7,9,8)(10,14,18)(11,17,20)(12,15,21)(13,16,19)(24,26,25)(28,29,30)(31,39,35)(32,41,38)(33,42,36)(34,40,37), (1,2,6,7,11,3,10)(4,14,8,15,16,20,13)(5,18,9,19,21,17,12)(22,23,28,24,32,31,27)(25,36,41,42,34,39,30)(26,40,38,37,33,35,29) ]", "PSL(3,4)^1" ],
  [     40320, "[ (7,8), (1,2,3,4,5,6,7) ]", "A8^1" ],
  [     50616, "[ (3,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,21)(3,20,22)(4,31,10)(5,34,6)(7,14,24)(8,37,36)(9,23,13)(11,38,32)(12,16,17)(18,28,35)(19,33,29)(25,26,30) ]", "PSL(2,37)^1" ],
  [     51840, "[ (6,7)(8,11)(9,13)(10,12)(15,16)(17,20)(18,22)(19,21)(24,25)(26,29)(27,31)(28,30)(33,34)(35,38)(36,40)(37,39), (3,4)(6,7)(9,10)(12,13)(17,20)(18,21)(19,22)(23,32)(24,33)(25,34)(26,38)(27,39)(28,40)(29,35)(30,36)(31,37), (1,5,14,17,27,22,19,36,3)(2,6,32)(4,7,23,20,37,13,16,26,40)(8,24,29,30,39,10,33,11,34)(9,15,35)(12,25,38)(21,28,31) ]", "PSp(4,3)^1" ],
  [     51888, "[ (3,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,26)(3,8,44)(4,34,17)(5,43,7)(6,28,21)(9,47,45)(10,16,29)(11,32,12)(13,15,27)(14,22,19)(18,48,35)(20,41,40)(23,36,42)(24,46,31)(25,37,39)(30,38,33) ]", "PSL(2,47)^1" ],
  [     56448, "[ (1,5)(2,8)(3,12)(4,14)(6,16)(7,19)(9,21)(10,23)(11,25)(13,27)(15,29)(17,31)(18,33)(20,34)(22,36)(24,37)(26,38)(28,39)(30,40)(32,41)(35,42), (1,20,32)(2,13,24)(3,9,4)(6,11,22)(7,28,35)(10,18,26)(15,17,30), (1,32,15)(2,22,3)(4,35,30)(6,24,7)(9,20,18)(10,17,11)(13,26,28) ]", "PSL(2,7)^2" ],
  [     58800, "[ (3,49,47,45,8,42,40,38,7,35,33,31,6,28,26,24,5,21,19,17,4,14,12,10)(9,50,48,46,44,43,41,39,37,36,34,32,30,29,27,25,23,22,20,18,16,15,13,11), (1,2,6)(3,4,8)(9,25,33)(10,20,37)(11,32,24)(12,15,41)(13,38,16)(14,30,23)(17,31,19)(18,44,47)(21,46,43)(22,39,49)(26,34,50)(27,48,35)(28,42,40)(29,45,36) ]", "PSL(2,49)^1" ],
  [     58968, "[ (5,7,13)(6,10,21)(8,16,14)(9,18,22)(11,24,15)(12,27,23)(17,19,25)(20,28,26), (3,28,27,26,25,24,23,22,21,20,19,18,17,4,16,15,14,13,12,11,10,9,8,7,6,5), (1,2,4)(5,8,24)(6,21,10)(7,16,15)(9,25,28)(11,13,14)(12,27,23)(17,26,18)(19,20,22) ]", "PSL(2,27)^1" ],
  [     60480, "[ (7,9,8)(10,18,14)(11,21,16)(12,19,17)(13,20,15)(28,29,30)(31,35,39)(32,37,42)(33,38,40)(34,36,41), (3,4,5)(7,9,8)(10,14,18)(11,17,20)(12,15,21)(13,16,19)(24,26,25)(28,29,30)(31,39,35)(32,41,38)(33,42,36)(34,40,37), (1,2,6,7,11,3,10)(4,14,8,15,16,20,13)(5,18,9,19,21,17,12)(22,23,28,24,32,31,27)(25,36,41,42,34,39,30)(26,40,38,37,33,35,29) ]", "PSL(3,4)^1" ],
  [     62400, "[ (3,4,6,10,18,31,49,56,63,5,8,14,25,42,58)(7,12,22,38,41,32,50,48,37,55,11,20,34,13,24)(9,16,28,45,61,64,59,17,29,23,40,54,62,65,15)(19,33,51,52,35,53,27,44,60,26,30,47,57,46,39)(21,36,43), (1,2,3,5,9,17,30,48,55,20,35,54,45,58,63)(4,7,13,18,32,38,56,60,61,42,59,62,8,15,27)(6,11,21,37,49,47,50,51,29,46,33,40,57,34,52)(10,19,12,23,22,39,31,28,36,53,25,14,26,43,16)(24,41,44) ]", "PSU(3,4)^1" ],
  [     68880, "[ (3,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,23)(3,37,9)(4,24,21)(5,31,13)(6,36,7)(8,12,29)(10,40,39)(11,20,18)(14,16,19)(15,41,33)(17,34,38)(22,42,25)(26,35,28)(27,30,32) ]", "PSL(2,41)^1" ],
  [     74412, "[ (3,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,29)(3,28,30)(4,12,45)(5,34,22)(6,47,8)(7,19,35)(9,10,42)(11,52,50)(13,46,54)(14,20,27)(15,25,21)(16,48,49)(17,26,18)(23,39,51)(24,53,36)(31,38,44)(32,41,40)(33,43,37) ]", "PSL(2,53)^1" ],
  [     79464, "[ (3,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,24)(3,39,9)(4,12,35)(5,14,32)(6,19,26)(7,33,11)(8,25,18)(10,20,21)(13,36,44)(15,41,37)(16,34,43)(22,29,42)(23,40,30)(27,28,38) ]", "PSL(2,43)^1" ],
  [     80640, "[ (4,5)(8,9)(12,13)(14,18)(15,19)(16,21)(17,20)(25,26)(29,30)(33,34)(35,39)(36,40)(37,42)(38,41), (1,22)(2,23)(3,24)(4,25)(5,26)(6,27)(7,28)(8,29)(9,30)(10,31)(11,32)(12,33)(13,34)(14,35)(15,36)(16,37)(17,38)(18,39)(19,40)(20,41)(21,42), (3,4,5)(7,9,8)(10,14,18)(11,17,20)(12,15,21)(13,16,19)(24,26,25)(28,29,30)(31,39,35)(32,41,38)(33,42,36)(34,40,37), (1,2,6,7,11,3,10)(4,14,8,15,16,20,13)(5,18,9,19,21,17,12)(22,23,28,24,32,31,27)(25,36,41,42,34,39,30)(26,40,38,37,33,35,29) ]", "PSL(3,4)^1" ],
  [     87360, "[ (4,7,5)(6,8,9)(12,15,13)(14,16,17)(18,42,26)(19,43,27)(20,47,29)(21,44,31)(22,48,33)(23,45,28)(24,49,30)(25,46,32)(34,50,58)(35,51,59)(36,55,61)(37,52,63)(38,56,65)(39,53,60)(40,57,62)(41,54,64), (1,28,10,44)(3,50,11,42)(4,43,53,64)(5,9,39,52)(6,36,63,13)(7,51,60,57)(8,33,37,16)(12,24,55,29)(14,30,48,47)(15,19,61,54)(17,59,22,62)(18,23,34,31)(20,38,49,25)(21,26,45,58)(27,32,41,65)(35,46,40,56), (1,2)(3,10)(4,42)(5,18)(6,50)(7,26)(8,58)(9,34)(12,28)(13,45)(14,44)(15,23)(16,31)(17,21)(19,39)(20,38)(22,25)(24,61)(27,60)(29,65)(30,55)(32,33)(35,52)(36,49)(37,59)(40,54)(41,62)(43,53)(46,48)(47,56)(51,63)(57,64) ]", "Sz(8)^1" ],
  [     95040, "[ (3,7,11,8)(4,10,5,6)(13,19)(14,23)(15,17,20,21)(16,22,24,18), (1,2,3,4,5,6,7,8,9,10,11)(14,18,22,17,24,16,20,23,19,15,21), (1,12)(2,11)(3,6)(4,8)(5,9)(7,10)(13,19)(14,24)(15,23)(16,18)(17,20)(21,22) ]", "M12^1" ],
  [    102660, "[ (3,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,32)(3,31,33)(4,40,23)(5,26,36)(6,48,13)(7,50,10)(8,15,44)(9,39,19)(11,22,34)(12,18,37)(14,57,54)(16,58,51)(17,21,29)(20,49,56)(24,60,41)(25,55,45)(27,46,52)(28,38,59)(30,42,53)(35,43,47) ]", "PSL(2,59)^1" ],
  [    103776, "[ (3,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,26)(3,8,44)(4,34,17)(5,43,7)(6,28,21)(9,47,45)(10,16,29)(11,32,12)(13,15,27)(14,22,19)(18,48,35)(20,41,40)(23,36,42)(24,46,31)(25,37,39)(30,38,33) ]", "PSL(2,47)^1" ],
  [    112896, "[ (1,5)(2,8)(3,12)(4,14)(6,16)(7,19)(9,21)(10,23)(11,25)(13,27)(15,29)(17,31)(18,33)(20,34)(22,36)(24,37)(26,38)(28,39)(30,40)(32,41)(35,42), (2,24,6,28)(3,4,30,18,10,11,13,7)(8,37)(9,35,22,26,17,20,32,15)(12,23)(14,33)(16,39)(19,25)(21,38)(29,31)(34,41)(36,42), (1,20,32)(2,13,24)(3,9,4)(6,11,22)(7,28,35)(10,18,26)(15,17,30) ]", "PSL(2,7)^2" ],
  [    112896, "[ (1,5)(2,8,24,37,6,16,28,39)(3,12,4,14,30,40,18,33,10,23,11,25,13,27,7,19)(9,21,35,42,22,36,26,38,17,31,20,34,32,41,15,29), (2,24,6,28)(3,4,30,18,10,11,13,7)(8,37)(9,35,22,26,17,20,32,15)(12,23)(14,33)(16,39)(19,25)(21,38)(29,31)(34,41)(36,42), (5,12,23)(8,19,33)(14,25,37)(16,29,38)(21,31,39)(27,36,42)(34,41,40) ]", "PSL(2,7)^2" ],
  [    113460, "[ (3,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,33)(3,32,34)(4,27,38)(5,11,53)(6,21,42)(7,46,16)(8,12,49)(9,31,29)(10,44,15)(13,55,61)(14,19,36)(17,54,58)(18,25,26)(20,59,50)(22,56,51)(24,45,60)(28,39,62)(30,47,52)(35,57,37)(40,41,48) ]", "PSL(2,61)^1" ],
  [    117600, "[ (3,50,49,48,47,46,45,44,8,43,42,41,40,39,38,37,7,36,35,34,33,32,31,30,6,29,28,27,26,25,24,23,5,22,21,20,19,18,17,16,4,15,14,13,12,11,10,9), (1,2,6)(3,4,8)(9,25,33)(10,20,37)(11,32,24)(12,15,41)(13,38,16)(14,30,23)(17,31,19)(18,44,47)(21,46,43)(22,39,49)(26,34,50)(27,48,35)(28,42,40)(29,45,36) ]", "PSL(2,49)^1" ],
  [    117600, "[ (3,50,8,43,7,36,6,29,5,22,4,15)(9,14,44,49,37,42,30,35,23,28,16,21)(10,20,45,13,38,48,31,41,24,34,17,27)(11,26,46,19,39,12,32,47,25,40,18,33), (3,49,47,45,8,42,40,38,7,35,33,31,6,28,26,24,5,21,19,17,4,14,12,10)(9,50,48,46,44,43,41,39,37,36,34,32,30,29,27,25,23,22,20,18,16,15,13,11), (1,2,6)(3,4,8)(9,25,33)(10,20,37)(11,32,24)(12,15,41)(13,38,16)(14,30,23)(17,31,19)(18,44,47)(21,46,43)(22,39,49)(26,34,50)(27,48,35)(28,42,40)(29,45,36) ]", "PSL(2,49)^1" ],
  [    117600, "[ (9,15)(10,21)(11,27)(12,33)(13,39)(14,45)(16,22)(17,28)(18,34)(19,40)(20,46)(23,29)(24,35)(25,41)(26,47)(30,36)(31,42)(32,48)(37,43)(38,49)(44,50), (3,49,47,45,8,42,40,38,7,35,33,31,6,28,26,24,5,21,19,17,4,14,12,10)(9,50,48,46,44,43,41,39,37,36,34,32,30,29,27,25,23,22,20,18,16,15,13,11), (1,2,6)(3,4,8)(9,25,33)(10,20,37)(11,32,24)(12,15,41)(13,38,16)(14,30,23)(17,31,19)(18,44,47)(21,46,43)(22,39,49)(26,34,50)(27,48,35)(28,42,40)(29,45,36) ]", "PSL(2,49)^1" ],
  [    120960, "[ (1,22)(2,23)(3,24)(4,25)(5,26)(6,27)(7,28)(8,29)(9,30)(10,31)(11,32)(12,33)(13,34)(14,35)(15,36)(16,37)(17,38)(18,39)(19,40)(20,41)(21,42), (7,8,9)(10,14,18)(11,16,21)(12,17,19)(13,15,20)(28,30,29)(31,39,35)(32,42,37)(33,40,38)(34,41,36), (3,4,5)(7,9,8)(10,14,18)(11,17,20)(12,15,21)(13,16,19)(24,26,25)(28,29,30)(31,39,35)(32,41,38)(33,42,36)(34,40,37), (1,2,6,7,11,3,10)(4,14,8,15,16,20,13)(5,18,9,19,21,17,12)(22,23,28,24,32,31,27)(25,36,41,42,34,39,30)(26,40,38,37,33,35,29) ]", "PSL(3,4)^1" ],
  [    120960, "[ (4,5)(8,9)(12,13)(14,18)(15,19)(16,21)(17,20)(25,26)(29,30)(33,34)(35,39)(36,40)(37,42)(38,41), (7,8,9)(10,14,18)(11,16,21)(12,17,19)(13,15,20)(28,30,29)(31,39,35)(32,42,37)(33,40,38)(34,41,36), (3,4,5)(7,9,8)(10,14,18)(11,17,20)(12,15,21)(13,16,19)(24,26,25)(28,29,30)(31,39,35)(32,41,38)(33,42,36)(34,40,37), (1,2,6,7,11,3,10)(4,14,8,15,16,20,13)(5,18,9,19,21,17,12)(22,23,28,24,32,31,27)(25,36,41,42,34,39,30)(26,40,38,37,33,35,29) ]", "PSL(3,4)^1" ],
  [    120960, "[ (7,9,8)(10,18,14)(11,21,16)(12,19,17)(13,20,15)(28,29,30)(31,35,39)(32,37,42)(33,38,40)(34,36,41), (1,22)(2,23)(3,24)(4,26)(5,25)(6,27)(7,28)(8,30)(9,29)(10,31)(11,32)(12,34)(13,33)(14,39)(15,40)(16,42)(17,41)(18,35)(19,36)(20,38)(21,37), (3,4,5)(7,9,8)(10,14,18)(11,17,20)(12,15,21)(13,16,19)(24,26,25)(28,29,30)(31,39,35)(32,41,38)(33,42,36)(34,40,37), (1,2,6,7,11,3,10)(4,14,8,15,16,20,13)(5,18,9,19,21,17,12)(22,23,28,24,32,31,27)(25,36,41,42,34,39,30)(26,40,38,37,33,35,29) ]", "PSL(3,4)^1" ],
  [    124800, "[ (3,55,31,24,8,41)(4,13,49,38,14,37)(5,7,58,32,18,11)(6,22,56,48,25,34)(9,51,64,44,40,57)(10,50,63,20,42,12)(15,46,61,52,23,60)(16,27,59,47,54,33)(17,19,62,53,28,30)(21,36,43)(26,45,39,29,35,65), (3,4,6,10,18,31,49,56,63,5,8,14,25,42,58)(7,12,22,38,41,32,50,48,37,55,11,20,34,13,24)(9,16,28,45,61,64,59,17,29,23,40,54,62,65,15)(19,33,51,52,35,53,27,44,60,26,30,47,57,46,39)(21,36,43), (1,2,3,5,9,17,30,48,55,20,35,54,45,58,63)(4,7,13,18,32,38,56,60,61,42,59,62,8,15,27)(6,11,21,37,49,47,50,51,29,46,33,40,57,34,52)(10,19,12,23,22,39,31,28,36,53,25,14,26,43,16)(24,41,44) ]", "PSU(3,4)^1" ],
  [    126000, "[ (3,4,6,10,17,31,33,51)(5,8,14,25,45,13,23,42)(7,12,21,38,55,86,77,92)(9,16,29,52,81,83,111,73)(11,19,34,58,88,22,40,67)(15,27,44,71,46,72,70,101)(18,32,35,59,89,100,116,125)(20,36,61,91,114,126,119,105)(24,39,66,98)(26,47,74,103,80,108,122,57)(28,50,48,76,79,30,54,84)(37,63,94,118,68,99,95,115)(41,64,96,75,104,43,69,85)(49,78,106,62,93,117,65,90)(53,82,109,102,121,123,60,56)(87,113,110,124,112,97,120,107), (1,2,3,5,9)(4,7,13,24,44)(6,11,20,37,64)(8,15,28,51,81)(10,18,33,57,87)(12,22,41,68,100)(14,26,48,77,105)(16,30,55,42,17)(19,35,60,58,34)(21,39,25,46,73)(23,43,70,76,93)(27,49,79,107,122)(29,53,83,112,117)(31,56,86,108,123)(32,52,59,90,101)(36,62,94,89,66)(38,65,97,50,80)(45,69,63,95,119)(47,75,98,121,99)(54,85,109,72,96)(61,92,116,126,84)(67,71,102,118,114)(74,103,78,106,120)(82,110,125,113,124)(88,111,91,115,104) ]", "PSU(3,5)^1" ],
  [    148824, "[ (3,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,29)(3,28,30)(4,12,45)(5,34,22)(6,47,8)(7,19,35)(9,10,42)(11,52,50)(13,46,54)(14,20,27)(15,25,21)(16,48,49)(17,26,18)(23,39,51)(24,53,36)(31,38,44)(32,41,40)(33,43,37) ]", "PSL(2,53)^1" ],
  [    150348, "[ (3,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,36)(3,35,37)(4,63,8)(5,21,49)(6,24,45)(7,38,30)(9,68,64)(10,54,11)(12,41,22)(13,19,43)(14,44,17)(15,20,40)(16,26,33)(18,62,61)(23,51,67)(27,48,66)(28,58,55)(29,53,59)(31,60,50)(32,52,57)(34,65,42)(39,46,56) ]", "PSL(2,67)^1" ],
  [    163680, "[ (4,5,7,11,19)(6,9,15,27,20)(8,13,23,12,21)(10,17,31,28,22)(14,25,16,29,24)(18,33,32,30,26), (3,32,30,28,26,24,22,20,18,16,14,12,10,8,6,4,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5), (1,2,3)(4,16,20)(5,29,6)(7,24,9)(8,32,31)(10,12,18)(11,14,15)(13,30,28)(17,21,33)(19,25,27)(22,23,26) ]", "PSL(2,32)^1" ],
  [    175560, "[ (1,17)(2,67)(3,135)(4,83)(5,115)(6,8)(7,210)(9,114)(10,214)(11,149)(12,59)(13,53)(14,119)(15,179)(16,139)(18,169)(19,50)(20,232)(21,109)(22,128)(23,202)(24,82)(25,56)(26,134)(27,196)(28,234)(29,133)(30,102)(31,256)(32,204)(33,123)(34,148)(35,248)(36,164)(37,185)(38,137)(39,177)(40,165)(41,230)(42,113)(44,93)(45,127)(46,198)(47,180)(48,197)(49,239)(51,112)(52,265)(54,88)(55,66)(57,263)(58,62)(60,89)(61,238)(63,258)(64,97)(65,219)(68,186)(69,120)(70,124)(71,242)(72,211)(73,246)(74,175)(75,92)(76,192)(77,153)(78,215)(79,162)(80,183)(81,244)(84,85)(86,132)(87,264)(90,152)(91,103)(94,220)(95,201)(96,108)(98,171)(99,228)(100,178)(101,187)(104,253)(105,141)(106,147)(107,174)(110,247)(111,184)(116,146)(117,154)(118,158)(121,226)(122,157)(125,161)(126,212)(129,155)(130,159)(131,249)(136,235)(138,218)(140,205)(142,176)(143,227)(145,189)(156,181)(160,236)(163,172)(166,217)(167,251)(170,190)(173,237)(182,262)(188,209)(191,243)(193,259)(194,240)(195,224)(199,229)(200,208)(203,255)(213,225)(216,252)(221,257)(223,254)(231,260)(233,245)(250,261), (1,154,190)(2,78,249)(3,139,165)(4,265,119)(5,201,258)(6,187,237)(7,99,124)(8,246,240)(9,96,180)(10,75,36)(11,141,16)(12,174,215)(13,164,58)(14,242,191)(15,140,26)(17,37,118)(18,59,170)(19,65,189)(20,162,172)(22,223,104)(23,117,45)(24,62,256)(25,152,232)(27,72,128)(28,54,196)(29,40,126)(30,238,255)(31,87,251)(32,142,106)(33,202,261)(34,193,205)(35,171,183)(38,57,254)(39,70,143)(41,123,55)(42,145,110)(43,64,134)(44,195,231)(46,56,248)(47,51,184)(48,66,79)(49,50,207)(52,100,83)(53,158,121)(60,211,88)(61,113,132)(63,98,218)(67,136,81)(68,227,213)(69,192,138)(71,175,93)(73,122,160)(74,137,148)(76,263,125)(77,108,101)(80,264,97)(82,178,163)(84,103,167)(85,262,252)(86,216,166)(89,243,244)(91,241,120)(92,235,214)(94,206,102)(95,144,173)(105,159,157)(107,188,197)(109,156,199)(111,135,253)(112,245,150)(114,247,220)(115,198,194)(116,239,177)(130,230,260)(131,161,200)(133,146,176)(147,204,168)(149,222,181)(151,236,212)(153,225,233)(155,217,226)(169,210,219)(179,182,203)(185,209,224)(186,257,234)(208,221,259)(228,229,266) ]", "J_1^1" ],
  [    178920, "[ (3,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,38)(3,32,44)(4,20,55)(5,46,28)(6,42,31)(7,35,37)(8,57,14)(9,12,58)(10,54,15)(11,25,43)(13,17,49)(16,40,23)(18,64,67)(19,68,62)(21,56,72)(22,66,61)(24,29,26)(27,59,63)(30,71,48)(33,51,65)(34,70,45)(36,60,53)(39,41,69)(47,52,50) ]", "PSL(2,71)^1" ],
  [    181440, "[ (1,2,3,4,5,6,7,8,9), (7,8,9) ]", "A9^1" ],
  [    190080, "[ (1,13,5,17)(2,14,3,15,12,24,9,21,6,18,4,16)(7,19,8,20,10,22)(11,23), (3,7,11,8)(4,10,5,6)(13,19)(14,23)(15,17,20,21)(16,22,24,18) ]", "M12^1" ],
  [    194472, "[ (3,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,39)(3,31,47)(4,25,52)(5,44,32)(6,58,17)(7,64,10)(8,16,57)(9,23,49)(11,33,37)(12,56,13)(14,71,68)(15,30,36)(18,28,35)(19,38,24)(20,72,61)(21,62,70)(22,66,65)(26,53,74)(29,55,69)(34,73,46)(40,59,54)(41,45,67)(42,48,63)(43,50,60) ]", "PSL(2,73)^1" ],
  [    205320, "[ (3,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,32)(3,31,33)(4,40,23)(5,26,36)(6,48,13)(7,50,10)(8,15,44)(9,39,19)(11,22,34)(12,18,37)(14,57,54)(16,58,51)(17,21,29)(20,49,56)(24,60,41)(25,55,45)(27,46,52)(28,38,59)(30,42,53)(35,43,47) ]", "PSL(2,59)^1" ],
  [    225792, "[ (1,5)(2,8)(3,12)(4,14)(6,16)(7,19)(9,21)(10,23)(11,25)(13,27)(15,29)(17,31)(18,33)(20,34)(22,36)(24,37)(26,38)(28,39)(30,40)(32,41)(35,42), (8,37)(12,23)(14,33)(16,39)(19,25)(21,38)(29,31)(34,41)(36,42), (2,24,6,28)(3,4,30,18,10,11,13,7)(9,35,22,26,17,20,32,15), (5,41,34)(8,37,27)(12,14,21)(16,36,25)(19,42,39)(23,38,33)(29,40,31) ]", "PSL(2,7)^2" ],
  [    226920, "[ (3,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,33)(3,32,34)(4,27,38)(5,11,53)(6,21,42)(7,46,16)(8,12,49)(9,31,29)(10,44,15)(13,55,61)(14,19,36)(17,54,58)(18,25,26)(20,59,50)(22,56,51)(24,45,60)(28,39,62)(30,47,52)(35,57,37)(40,41,48) ]", "PSL(2,61)^1" ],
  [    235200, "[ (9,15)(10,21)(11,27)(12,33)(13,39)(14,45)(16,22)(17,28)(18,34)(19,40)(20,46)(23,29)(24,35)(25,41)(26,47)(30,36)(31,42)(32,48)(37,43)(38,49)(44,50), (3,50,49,48,47,46,45,44,8,43,42,41,40,39,38,37,7,36,35,34,33,32,31,30,6,29,28,27,26,25,24,23,5,22,21,20,19,18,17,16,4,15,14,13,12,11,10,9), (1,2,6)(3,4,8)(9,25,33)(10,20,37)(11,32,24)(12,15,41)(13,38,16)(14,30,23)(17,31,19)(18,44,47)(21,46,43)(22,39,49)(26,34,50)(27,48,35)(28,42,40)(29,45,36) ]", "PSL(2,49)^1" ],
  [    241920, "[ (4,5)(8,9)(12,13)(14,18)(15,19)(16,21)(17,20)(25,26)(29,30)(33,34)(35,39)(36,40)(37,42)(38,41), (1,22)(2,23)(3,24)(4,25)(5,26)(6,27)(7,28)(8,29)(9,30)(10,31)(11,32)(12,33)(13,34)(14,35)(15,36)(16,37)(17,38)(18,39)(19,40)(20,41)(21,42), (7,8,9)(10,14,18)(11,16,21)(12,17,19)(13,15,20)(28,30,29)(31,39,35)(32,42,37)(33,40,38)(34,41,36), (3,4,5)(7,9,8)(10,14,18)(11,17,20)(12,15,21)(13,16,19)(24,26,25)(28,29,30)(31,39,35)(32,41,38)(33,42,36)(34,40,37), (1,2,6,7,11,3,10)(4,14,8,15,16,20,13)(5,18,9,19,21,17,12)(22,23,28,24,32,31,27)(25,36,41,42,34,39,30)(26,40,38,37,33,35,29) ]", "PSL(3,4)^1" ],
  [    246480, "[ (3,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,42)(3,38,46)(4,34,49)(5,54,28)(6,59,22)(7,41,39)(8,67,12)(9,10,68)(11,58,18)(13,23,51)(14,53,20)(15,40,32)(16,74,75)(17,76,72)(19,21,47)(24,36,27)(25,78,62)(26,73,66)(30,79,56)(31,70,64)(33,61,71)(35,50,80)(37,63,65)(43,77,45)(44,69,52)(48,60,57) ]", "PSL(2,79)^1" ],
  [    249600, "[ (3,29,24,26)(4,40,38,51)(5,54,32,27)(6,62,48,30)(7,47,18,16)(8,65,55,39)(9,13,44,14)(10,15,20,52)(11,33,58,59)(12,46,63,23)(17,34,53,56)(19,25,28,22)(21,36)(31,45,41,35)(37,57,49,64)(42,61,50,60), (3,55,31,24,8,41)(4,13,49,38,14,37)(5,7,58,32,18,11)(6,22,56,48,25,34)(9,51,64,44,40,57)(10,50,63,20,42,12)(15,46,61,52,23,60)(16,27,59,47,54,33)(17,19,62,53,28,30)(21,36,43)(26,45,39,29,35,65), (3,4,6,10,18,31,49,56,63,5,8,14,25,42,58)(7,12,22,38,41,32,50,48,37,55,11,20,34,13,24)(9,16,28,45,61,64,59,17,29,23,40,54,62,65,15)(19,33,51,52,35,53,27,44,60,26,30,47,57,46,39)(21,36,43), (1,2,3,5,9,17,30,48,55,20,35,54,45,58,63)(4,7,13,18,32,38,56,60,61,42,59,62,8,15,27)(6,11,21,37,49,47,50,51,29,46,33,40,57,34,52)(10,19,12,23,22,39,31,28,36,53,25,14,26,43,16)(24,41,44) ]", "PSU(3,4)^1" ],
  [    252000, "[ (3,95)(4,118)(5,103)(6,37)(7,79)(8,26)(9,27)(10,99)(11,43)(12,50)(13,80)(14,108)(15,52)(16,70)(17,94)(18,36)(19,96)(20,59)(21,54)(22,69)(23,47)(24,66)(25,74)(28,55)(29,71)(30,86)(31,115)(32,119)(33,68)(34,85)(35,91)(38,76)(39,98)(40,75)(41,67)(42,122)(44,83)(45,57)(46,73)(48,77)(49,93)(51,63)(53,110)(56,97)(58,104)(60,87)(61,100)(64,88)(65,106)(72,81)(82,107)(84,92)(89,126)(101,111)(102,113)(105,116)(109,112)(114,125)(120,121)(123,124), (3,4,6,10,17,31,33,51)(5,8,14,25,45,13,23,42)(7,12,21,38,55,86,77,92)(9,16,29,52,81,83,111,73)(11,19,34,58,88,22,40,67)(15,27,44,71,46,72,70,101)(18,32,35,59,89,100,116,125)(20,36,61,91,114,126,119,105)(24,39,66,98)(26,47,74,103,80,108,122,57)(28,50,48,76,79,30,54,84)(37,63,94,118,68,99,95,115)(41,64,96,75,104,43,69,85)(49,78,106,62,93,117,65,90)(53,82,109,102,121,123,60,56)(87,113,110,124,112,97,120,107), (1,2,3,5,9)(4,7,13,24,44)(6,11,20,37,64)(8,15,28,51,81)(10,18,33,57,87)(12,22,41,68,100)(14,26,48,77,105)(16,30,55,42,17)(19,35,60,58,34)(21,39,25,46,73)(23,43,70,76,93)(27,49,79,107,122)(29,53,83,112,117)(31,56,86,108,123)(32,52,59,90,101)(36,62,94,89,66)(38,65,97,50,80)(45,69,63,95,119)(47,75,98,121,99)(54,85,109,72,96)(61,92,116,126,84)(67,71,102,118,114)(74,103,78,106,120)(82,110,125,113,124)(88,111,91,115,104) ]", "PSU(3,5)^1" ],
  [    259200, "[ (1,19,3,20,5,24,4,22,6,21)(2,23)(7,18,11,16,10,13,9,15,8,14)(12,17), (16,17,18)(19,24,22)(20,23,21) ]", "A6^2" ],
  [    262080, "[ (3,64,62,60,58,57,55,53,51,10,48,47,45,43,42,40,38,36,8,33,31,4,28,27,25,23,21,6,18,16,14,12,65,63,61,59,11,56,54,52,50,49,5,46,44,9,41,39,37,35,34,32,30,29,7,26,24,22,20,19,17,15,13), (1,2,3)(6,9,7)(8,11,10)(12,18,58)(13,24,51)(14,54,20)(15,35,38)(16,40,33)(17,43,29)(19,59,65)(21,46,22)(23,63,57)(25,36,28)(26,53,64)(27,32,30)(31,56,55)(34,60,48)(37,61,44)(39,42,62)(41,52,49)(45,50,47) ]", "PSL(2,64)^1" ],
  [    265680, "[ (3,81,79,77,75,10,72,70,68,66,9,63,61,59,57,8,54,52,50,48,4,45,43,41,39,7,36,34,32,30,6,27,25,23,21,5,18,16,14,12)(11,82,80,78,76,74,73,71,69,67,65,64,62,60,58,56,55,53,51,49,47,46,44,42,40,38,37,35,33,31,29,28,26,24,22,20,19,17,15,13), (1,2,4)(5,6,8)(7,9,10)(11,21,71)(12,66,25)(13,43,47)(14,55,35)(15,37,52)(16,34,54)(17,63,24)(18,20,65)(19,36,49)(22,72,82)(23,51,29)(26,33,45)(27,81,68)(28,73,75)(30,76,69)(31,32,40)(38,79,58)(39,59,77)(41,56,78)(42,70,64)(44,57,74)(46,50,80)(48,60,67)(53,61,62) ]", "PSL(2,81)^1" ],
  [    285852, "[ (3,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,44)(3,43,45)(4,54,33)(5,17,69)(6,64,21)(7,70,14)(8,30,53)(9,22,60)(10,65,16)(11,41,39)(12,27,52)(13,32,46)(15,26,50)(18,81,74)(19,71,83)(20,40,31)(23,78,72)(24,82,67)(25,29,37)(28,66,79)(34,84,55)(35,58,80)(36,61,76)(38,62,73)(42,56,75)(47,77,49)(48,68,57)(51,59,63) ]", "PSL(2,83)^1" ],
  [    300696, "[ (3,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,36)(3,35,37)(4,63,8)(5,21,49)(6,24,45)(7,38,30)(9,68,64)(10,54,11)(12,41,22)(13,19,43)(14,44,17)(15,20,40)(16,26,33)(18,62,61)(23,51,67)(27,48,66)(28,58,55)(29,53,59)(31,60,50)(32,52,57)(34,65,42)(39,46,56) ]", "PSL(2,67)^1" ],
  [    352440, "[ (3,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,47)(3,31,63)(4,15,78)(5,49,43)(6,22,69)(7,10,80)(8,34,55)(9,29,59)(11,42,44)(12,64,21)(13,18,66)(14,84,87)(16,79,90)(17,53,27)(19,46,32)(20,54,23)(24,37,36)(25,72,88)(26,38,33)(28,76,81)(30,82,73)(35,65,85)(39,60,86)(40,74,71)(41,77,67)(45,89,51)(48,75,62)(50,52,83)(56,68,61)(57,70,58) ]", "PSL(2,89)^1" ],
  [    357840, "[ (3,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,38)(3,32,44)(4,20,55)(5,46,28)(6,42,31)(7,35,37)(8,57,14)(9,12,58)(10,54,15)(11,25,43)(13,17,49)(16,40,23)(18,64,67)(19,68,62)(21,56,72)(22,66,61)(24,29,26)(27,59,63)(30,71,48)(33,51,65)(34,70,45)(36,60,53)(39,41,69)(47,52,50) ]", "PSL(2,71)^1" ],
  [    362880, "[ (8,9), (1,2,3,4,5,6,7,8,9) ]", "A9^1" ],
  [    372000, "[ (3,4,5,6)(8,11,10,9)(12,22)(13,26,15,24)(14,23,16,25)(17,27)(18,31,20,29)(19,28,21,30)(34,37,36,35)(39,40,41,42)(43,53)(44,55,46,57)(45,56,47,54)(48,58)(49,60,51,62)(50,61,52,59), (1,2,7,10,25,3,12,9,20,31,18,21,16,24,4,17,11,30,13,29,28,23,5,22)(6,27,8,15,19,26)(32,33,39,36,54,55,60,56,45,48,42,37,59,51,52,57,50,47,58,41,34,44,43,38)(35,49,62,46,53,40) ]", "PSL(3,5)^1" ],
  [    378000, "[ (3,9,50,31,83,54,6,29,76,51,73,28,17,81,30,4,16,48,33,111,84,10,52,79)(5,74,90,13,57,93,14,80,78,42,47,65,45,122,62,8,103,49,23,26,117,25,108,106)(7,44,63,86,101,95,21,46,118,92,27,37,55,70,99,12,71,94,77,15,115,38,72,68)(11,82,18,22,60,100,34,102,35,67,53,125,88,123,89,19,109,32,40,56,116,58,121,59)(20,87,41,126,97,43,61,110,96,105,107,85,114,112,104,36,113,64,119,120,69,91,124,75)(24,98,66,39), (1,2,3,5,9)(4,7,13,24,44)(6,11,20,37,64)(8,15,28,51,81)(10,18,33,57,87)(12,22,41,68,100)(14,26,48,77,105)(16,30,55,42,17)(19,35,60,58,34)(21,39,25,46,73)(23,43,70,76,93)(27,49,79,107,122)(29,53,83,112,117)(31,56,86,108,123)(32,52,59,90,101)(36,62,94,89,66)(38,65,97,50,80)(45,69,63,95,119)(47,75,98,121,99)(54,85,109,72,96)(61,92,116,126,84)(67,71,102,118,114)(74,103,78,106,120)(82,110,125,113,124)(88,111,91,115,104) ]", "PSU(3,5)^1" ],
  [    388944, "[ (3,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,39)(3,31,47)(4,25,52)(5,44,32)(6,58,17)(7,64,10)(8,16,57)(9,23,49)(11,33,37)(12,56,13)(14,71,68)(15,30,36)(18,28,35)(19,38,24)(20,72,61)(21,62,70)(22,66,65)(26,53,74)(29,55,69)(34,73,46)(40,59,54)(41,45,67)(42,48,63)(43,50,60) ]", "PSL(2,73)^1" ],
  [    443520, "[ (1,2,3,4,5,6,7,8,9,10,11)(12,13,14,15,16,17,18,19,20,21,22), (1,4,5,9,3)(2,8,10,7,6)(12,15,16,20,14)(13,19,21,18,17), (1,21)(2,10,8,6)(3,13,4,17)(5,19,9,18)(11,22)(12,14,16,20) ]", "M22^1" ],
  [    456288, "[ (3,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,51)(3,17,85)(4,43,58)(5,88,12)(6,38,61)(7,89,9)(8,27,70)(10,62,33)(11,20,74)(13,95,93)(14,97,90)(15,25,65)(16,42,47)(18,63,24)(19,56,30)(21,48,36)(22,52,31)(23,29,53)(26,34,45)(28,82,91)(32,75,94)(37,77,87)(39,84,78)(40,92,69)(41,64,96)(44,59,98)(46,83,72)(49,73,79)(50,80,71)(54,81,66)(55,60,86)(57,68,76) ]", "PSL(2,97)^1" ],
  [    492960, "[ (3,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,42)(3,38,46)(4,34,49)(5,54,28)(6,59,22)(7,41,39)(8,67,12)(9,10,68)(11,58,18)(13,23,51)(14,53,20)(15,40,32)(16,74,75)(17,76,72)(19,21,47)(24,36,27)(25,78,62)(26,73,66)(30,79,56)(31,70,64)(33,61,71)(35,50,80)(37,63,65)(43,77,45)(44,69,52)(48,60,57) ]", "PSL(2,79)^1" ],
  [    508032, "[ (1,29)(2,42,20,41)(3,55)(4,48,9,49)(5,31,17,30)(6,37,27,44)(7,45,14,53)(8,35,18,34)(10,40,25,56)(11,43)(12,50,19,54)(13,52,26,38)(15,36,16,47)(21,51,28,39)(22,33,23,32)(24,46), (29,50,45,36,46,33,51)(30,53,37,32,38,48,39)(31,55,42,43,44,35,34)(40,41,56,49,54,47,52) ]", "PSL(2,8)^2" ],
  [    515100, "[ (3,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,53)(3,52,54)(4,84,21)(5,29,75)(6,15,88)(7,23,79)(8,71,30)(9,63,37)(10,42,57)(11,16,82)(12,50,47)(13,60,36)(14,62,33)(17,67,25)(18,91,100)(19,32,58)(20,55,34)(22,102,85)(24,90,95)(26,45,38)(27,83,99)(28,40,41)(31,77,101)(35,98,76)(39,89,81)(43,97,69)(44,92,73)(46,93,70)(48,74,87)(49,64,96)(51,86,72)(56,94,59)(61,80,68)(65,66,78) ]", "PSL(2,101)^1" ],
  [    518400, "[ (1,13,9,19,3,16,10,21)(2,17,12,22,6,14,7,20,5,18,11,24,4,15,8,23), (1,7,3,10,5,11,2,9)(4,8,6,12)(13,19,15,22,17,23,14,21)(16,20,18,24), (1,6,4,3,2)(7,12,8,10,11) ]", "A6^2" ],
  [    518400, "[ (1,13,9,21)(2,18,11,23)(3,16,10,19)(4,15,8,24)(5,17,12,20)(6,14,7,22), (1,9)(2,7,3,8,5,12,4,10,6,11)(13,21)(14,19,15,20,17,24,16,22,18,23), (13,14,15,16,17)(20,22,23,21,24) ]", "A6^2" ],
  [    518400, "[ (1,19,3,20,5,24,4,22,6,21)(2,23)(7,18,11,16,10,13,9,15,8,14)(12,17), (1,7,3,10,5,11,2,9)(4,8,6,12)(13,19,15,22,17,23,14,21)(16,20,18,24), (2,5,3)(7,8,10)(9,11,12) ]", "A6^2" ],
  [    518400, "[ (1,19,3,20,5,24,4,22,6,21)(2,23)(7,18,11,16,10,13,9,15,8,14)(12,17), (1,9)(2,7,3,8,5,12,4,10,6,11)(13,21)(14,19,15,20,17,24,16,22,18,23), (13,14,15,16,17)(20,22,23,21,24) ]", "A6^2" ],
  [    518400, "[ (1,19,3,20,5,24,4,22,6,21)(2,23)(7,18,11,16,10,13,9,15,8,14)(12,17), (5,6)(7,9)(8,10)(11,12)(17,18)(19,21)(20,22)(23,24), (2,5,3)(7,8,10)(9,11,12) ]", "A6^2" ],
  [    518400, "[ (1,21)(2,24,4,20,5,23)(3,22,6,19)(7,17,12,18,11,16,10,13,9,15,8,14), (5,6)(7,9)(8,10)(11,12)(17,18)(19,21)(20,22)(23,24), (16,17,18)(19,24,22)(20,23,21) ]", "A6^2" ],
  [    524160, "[ (4,5)(12,19)(13,26)(14,32)(15,39)(16,46)(17,52)(18,59)(20,27)(21,33)(22,40)(23,47)(24,53)(25,60)(28,34)(29,41)(30,54)(31,61)(35,42)(36,48)(37,55)(38,62)(43,49)(44,56)(45,63)(50,57)(51,64)(58,65), (3,64,62,60,58,57,55,53,51,10,48,47,45,43,42,40,38,36,8,33,31,4,28,27,25,23,21,6,18,16,14,12,65,63,61,59,11,56,54,52,50,49,5,46,44,9,41,39,37,35,34,32,30,29,7,26,24,22,20,19,17,15,13), (1,2,3)(6,9,7)(8,11,10)(12,18,58)(13,24,51)(14,54,20)(15,35,38)(16,40,33)(17,43,29)(19,59,65)(21,46,22)(23,63,57)(25,36,28)(26,53,64)(27,32,30)(31,56,55)(34,60,48)(37,61,44)(39,42,62)(41,52,49)(45,50,47) ]", "PSL(2,64)^1" ],
  [    531360, "[ (11,19)(12,27)(13,35)(14,43)(15,51)(16,59)(17,67)(18,75)(20,28)(21,36)(22,44)(23,52)(24,60)(25,68)(26,76)(29,37)(30,45)(31,53)(32,61)(33,69)(34,77)(38,46)(39,54)(40,62)(41,70)(42,78)(47,55)(48,63)(49,71)(50,79)(56,64)(57,72)(58,80)(65,73)(66,81)(74,82), (3,81,79,77,75,10,72,70,68,66,9,63,61,59,57,8,54,52,50,48,4,45,43,41,39,7,36,34,32,30,6,27,25,23,21,5,18,16,14,12)(11,82,80,78,76,74,73,71,69,67,65,64,62,60,58,56,55,53,51,49,47,46,44,42,40,38,37,35,33,31,29,28,26,24,22,20,19,17,15,13), (1,2,4)(5,6,8)(7,9,10)(11,21,71)(12,66,25)(13,43,47)(14,55,35)(15,37,52)(16,34,54)(17,63,24)(18,20,65)(19,36,49)(22,72,82)(23,51,29)(26,33,45)(27,81,68)(28,73,75)(30,76,69)(31,32,40)(38,79,58)(39,59,77)(41,56,78)(42,70,64)(44,57,74)(46,50,80)(48,60,67)(53,61,62) ]", "PSL(2,81)^1" ],
  [    531360, "[ (3,82,10,73,9,64,8,55,4,46,7,37,6,28,5,19)(11,18,74,81,65,72,56,63,47,54,38,45,29,36,20,27)(12,26,75,17,66,80,57,71,48,62,39,53,30,44,21,35)(13,34,76,25,67,16,58,79,49,70,40,61,31,52,22,43)(14,42,77,33,68,24,59,15,50,78,41,69,32,60,23,51), (3,81,79,77,75,10,72,70,68,66,9,63,61,59,57,8,54,52,50,48,4,45,43,41,39,7,36,34,32,30,6,27,25,23,21,5,18,16,14,12)(11,82,80,78,76,74,73,71,69,67,65,64,62,60,58,56,55,53,51,49,47,46,44,42,40,38,37,35,33,31,29,28,26,24,22,20,19,17,15,13), (1,2,4)(5,6,8)(7,9,10)(11,21,71)(12,66,25)(13,43,47)(14,55,35)(15,37,52)(16,34,54)(17,63,24)(18,20,65)(19,36,49)(22,72,82)(23,51,29)(26,33,45)(27,81,68)(28,73,75)(30,76,69)(31,32,40)(38,79,58)(39,59,77)(41,56,78)(42,70,64)(44,57,74)(46,50,80)(48,60,67)(53,61,62) ]", "PSL(2,81)^1" ],
  [    531360, "[ (3,82,81,80,79,78,77,76,75,74,10,73,72,71,70,69,68,67,66,65,9,64,63,62,61,60,59,58,57,56,8,55,54,53,52,51,50,49,48,47,4,46,45,44,43,42,41,40,39,38,7,37,36,35,34,33,32,31,30,29,6,28,27,26,25,24,23,22,21,20,5,19,18,17,16,15,14,13,12,11), (1,2,4)(5,6,8)(7,9,10)(11,21,71)(12,66,25)(13,43,47)(14,55,35)(15,37,52)(16,34,54)(17,63,24)(18,20,65)(19,36,49)(22,72,82)(23,51,29)(26,33,45)(27,81,68)(28,73,75)(30,76,69)(31,32,40)(38,79,58)(39,59,77)(41,56,78)(42,70,64)(44,57,74)(46,50,80)(48,60,67)(53,61,62) ]", "PSL(2,81)^1" ],
  [    546312, "[ (3,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,54)(3,10,98)(4,73,34)(5,40,66)(6,30,75)(7,24,80)(8,92,11)(9,21,81)(12,23,76)(13,60,38)(14,46,51)(15,49,47)(16,100,97)(17,39,55)(18,64,29)(19,56,36)(20,26,65)(22,31,58)(25,45,41)(27,87,99)(28,84,101)(32,85,96)(33,78,102)(35,104,74)(42,68,103)(43,82,88)(44,90,79)(48,95,70)(50,77,86)(52,89,72)(53,69,91)(57,62,94)(59,93,61)(63,83,67) ]", "PSL(2,103)^1" ],
  [    571704, "[ (3,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,44)(3,43,45)(4,54,33)(5,17,69)(6,64,21)(7,70,14)(8,30,53)(9,22,60)(10,65,16)(11,41,39)(12,27,52)(13,32,46)(15,26,50)(18,81,74)(19,71,83)(20,40,31)(23,78,72)(24,82,67)(25,29,37)(28,66,79)(34,84,55)(35,58,80)(36,61,76)(38,62,73)(42,56,75)(47,77,49)(48,68,57)(51,59,63) ]", "PSL(2,83)^1" ],
  [    604800, "[ (1,58,96,14,36,4,25,71)(2,20,85,42,50,92,64,31)(3,51,5,17,89,34,99,11)(6,70)(7,90,12,19,82,86,54,23)(8,43,37,57,77,75,30,73)(9,52,53,95)(10,59,91,28,97,32,68,40)(13,56,74,78,80,46,38,21)(15,100,63,41,27,67,22,35)(16,69,83,24)(18,33,45,81,55,29,98,60)(26,72,47,62)(39,76)(44,79,94,65,88,61,66,49)(48,84), (1,91,65,44,12,82,54,92,70,63)(2,78,80,85,81,5,73,35,17,24)(3,86,22,40,94,99,8,66,41,20)(4,62,74,57,27,58,97,34,50,55)(6,56,38,19,29,7,59,28,96,95)(9,13,47,10,32,33,11,37,98,67)(14,77,71,23,36,84,49,93,76,79)(15,31,18,45,16,26,72,30,42,60)(21,52,90,53,43,46,69,68,83,64)(25,75,87,89,39,88,51,48,100,61) ]", "J_2^1" ],
  [    612468, "[ (3,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,56)(3,55,57)(4,92,19)(5,9,101)(6,22,87)(7,27,81)(8,70,37)(10,33,72)(11,103,107)(12,86,17)(13,28,74)(14,52,49)(15,29,71)(16,46,53)(18,58,39)(20,108,93)(21,62,32)(23,47,45)(24,43,48)(25,90,106)(26,100,95)(30,51,34)(31,85,105)(35,44,36)(38,84,99)(40,79,102)(41,83,97)(42,104,75)(50,91,80)(54,94,73)(59,66,96)(60,98,63)(61,82,78)(64,69,88)(65,89,67)(68,77,76) ]", "PSL(2,107)^1" ],
  [    647460, "[ (3,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,57)(3,108,6)(4,17,96)(5,52,60)(7,28,82)(8,33,76)(9,89,19)(10,13,94)(11,55,51)(12,71,34)(14,26,77)(15,72,30)(16,53,48)(18,97,110)(20,101,104)(21,22,74)(23,45,49)(24,58,35)(25,105,95)(27,46,44)(29,47,41)(31,36,50)(32,86,107)(37,88,100)(38,81,106)(40,92,93)(42,99,84)(43,102,80)(54,62,109)(56,90,79)(59,103,63)(61,98,66)(64,78,83)(65,69,91)(67,85,73)(68,87,70) ]", "PSL(2,109)^1" ],
  [    648000, "[ (1,15,6)(2,14,7)(3,11,8)(4,12,9)(5,13,10), (11,12,13,14,15), (1,5,4,3,2) ]", "A5^3" ],
  [    704880, "[ (3,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,47)(3,31,63)(4,15,78)(5,49,43)(6,22,69)(7,10,80)(8,34,55)(9,29,59)(11,42,44)(12,64,21)(13,18,66)(14,84,87)(16,79,90)(17,53,27)(19,46,32)(20,54,23)(24,37,36)(25,72,88)(26,38,33)(28,76,81)(30,82,73)(35,65,85)(39,60,86)(40,74,71)(41,77,67)(45,89,51)(48,75,62)(50,52,83)(56,68,61)(57,70,58) ]", "PSL(2,89)^1" ],
  [    721392, "[ (3,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,59)(3,47,71)(4,35,82)(5,76,40)(6,27,88)(7,65,49)(8,45,68)(9,13,99)(10,77,34)(11,23,87)(12,85,24)(14,89,18)(15,58,48)(16,51,54)(17,72,32)(19,105,109)(20,80,21)(22,55,44)(25,39,57)(26,52,43)(28,37,56)(29,104,100)(30,91,112)(31,95,107)(33,106,94)(36,83,114)(38,98,97)(41,108,84)(42,113,78)(46,101,86)(50,73,110)(53,111,69)(60,103,70)(61,79,93)(62,81,90)(63,96,74)(64,67,102)(66,92,75) ]", "PSL(2,113)^1" ],
  [    744000, "[ (1,32)(2,33)(3,34)(4,35)(5,36)(6,37)(7,38)(8,39)(9,40)(10,41)(11,42)(12,43)(13,44)(14,45)(15,46)(16,47)(17,48)(18,49)(19,50)(20,51)(21,52)(22,53)(23,54)(24,55)(25,56)(26,57)(27,58)(28,59)(29,60)(30,61)(31,62), (3,4,5,6)(8,11,10,9)(12,22)(13,26,15,24)(14,23,16,25)(17,27)(18,31,20,29)(19,28,21,30)(34,37,36,35)(39,40,41,42)(43,53)(44,55,46,57)(45,56,47,54)(48,58)(49,60,51,62)(50,61,52,59), (1,2,7,10,25,3,12,9,20,31,18,21,16,24,4,17,11,30,13,29,28,23,5,22)(6,27,8,15,19,26)(32,33,39,36,54,55,60,56,45,48,42,37,59,51,52,57,50,47,58,41,34,44,43,38)(35,49,62,46,53,40) ]", "PSL(3,5)^1" ],
  [    756000, "[ (3,95)(4,118)(5,103)(6,37)(7,79)(8,26)(9,27)(10,99)(11,43)(12,50)(13,80)(14,108)(15,52)(16,70)(17,94)(18,36)(19,96)(20,59)(21,54)(22,69)(23,47)(24,66)(25,74)(28,55)(29,71)(30,86)(31,115)(32,119)(33,68)(34,85)(35,91)(38,76)(39,98)(40,75)(41,67)(42,122)(44,83)(45,57)(46,73)(48,77)(49,93)(51,63)(53,110)(56,97)(58,104)(60,87)(61,100)(64,88)(65,106)(72,81)(82,107)(84,92)(89,126)(101,111)(102,113)(105,116)(109,112)(114,125)(120,121)(123,124), (3,9,50,31,83,54,6,29,76,51,73,28,17,81,30,4,16,48,33,111,84,10,52,79)(5,74,90,13,57,93,14,80,78,42,47,65,45,122,62,8,103,49,23,26,117,25,108,106)(7,44,63,86,101,95,21,46,118,92,27,37,55,70,99,12,71,94,77,15,115,38,72,68)(11,82,18,22,60,100,34,102,35,67,53,125,88,123,89,19,109,32,40,56,116,58,121,59)(20,87,41,126,97,43,61,110,96,105,107,85,114,112,104,36,113,64,119,120,69,91,124,75)(24,98,66,39), (1,2,3,5,9)(4,7,13,24,44)(6,11,20,37,64)(8,15,28,51,81)(10,18,33,57,87)(12,22,41,68,100)(14,26,48,77,105)(16,30,55,42,17)(19,35,60,58,34)(21,39,25,46,73)(23,43,70,76,93)(27,49,79,107,122)(29,53,83,112,117)(31,56,86,108,123)(32,52,59,90,101)(36,62,94,89,66)(38,65,97,50,80)(45,69,63,95,119)(47,75,98,121,99)(54,85,109,72,96)(61,92,116,126,84)(67,71,102,118,114)(74,103,78,106,120)(82,110,125,113,124)(88,111,91,115,104) ]", "PSU(3,5)^1" ],
  [    786240, "[ (6,7,9)(8,11,10)(12,26,15)(13,39,19)(14,52,22)(16,27,29)(17,40,32)(18,53,35)(20,41,46)(21,54,49)(23,28,56)(24,42,59)(25,55,63)(30,43,33)(31,57,36)(34,44,47)(37,45,60)(38,58,64)(48,61,50)(51,62,65), (3,64,62,60,58,57,55,53,51,10,48,47,45,43,42,40,38,36,8,33,31,4,28,27,25,23,21,6,18,16,14,12,65,63,61,59,11,56,54,52,50,49,5,46,44,9,41,39,37,35,34,32,30,29,7,26,24,22,20,19,17,15,13), (1,2,3)(6,9,7)(8,11,10)(12,18,58)(13,24,51)(14,54,20)(15,35,38)(16,40,33)(17,43,29)(19,59,65)(21,46,22)(23,63,57)(25,36,28)(26,53,64)(27,32,30)(31,56,55)(34,60,48)(37,61,44)(39,42,62)(41,52,49)(45,50,47) ]", "PSL(2,64)^1" ],
  [    871200, "[ (1,13,6,22,7,23,4,14,9,19,12,20)(2,15,8,16,5,17,10,21,3,24,11,18), (2,7,6,5,3)(8,11,9,12,10) ]", "PSL(2,11)^2" ],
  [    885720, "[ (3,121,119,117,115,113,12,110,108,106,104,102,11,99,97,95,93,91,10,88,86,84,82,80,9,77,75,73,71,69,8,66,64,62,60,58,7,55,53,51,49,47,6,44,42,40,38,36,5,33,31,29,27,25,4,22,20,18,16,14)(13,122,120,118,116,114,112,111,109,107,105,103,101,100,98,96,94,92,90,89,87,85,83,81,79,78,76,74,72,70,68,67,65,63,61,59,57,56,54,52,50,48,46,45,43,41,39,37,35,34,32,30,28,26,24,23,21,19,17,15), (1,2,8)(3,7,9)(4,10,5)(6,12,11)(13,43,91)(14,96,37)(15,110,23)(16,90,41)(17,57,73)(18,63,67)(19,20,109)(21,32,95)(22,46,79)(24,59,64)(25,120,112)(26,115,116)(27,34,87)(28,58,61)(29,38,80)(30,35,82)(31,69,47)(33,50,65)(36,60,51)(39,121,98)(40,103,114)(42,52,54)(44,92,122)(45,119,94)(48,101,108)(53,100,105)(55,97,106)(56,89,113)(62,78,118)(66,104,88)(68,72,117)(70,85,102)(71,76,111)(74,77,107)(75,99,84)(81,83,93) ]", "PSL(2,121)^1" ],
  [    887040, "[ (6,12)(7,20)(8,11)(9,22)(13,14)(15,17)(16,19), (1,21)(2,10,8,6)(3,13,4,17)(5,19,9,18)(11,22)(12,14,16,20), (1,2,3,4,5,6,7,8,9,10,11)(12,13,14,15,16,17,18,19,20,21,22) ]", "M22^1" ],
  [    912576, "[ (3,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,51)(3,17,85)(4,43,58)(5,88,12)(6,38,61)(7,89,9)(8,27,70)(10,62,33)(11,20,74)(13,95,93)(14,97,90)(15,25,65)(16,42,47)(18,63,24)(19,56,30)(21,48,36)(22,52,31)(23,29,53)(26,34,45)(28,82,91)(32,75,94)(37,77,87)(39,84,78)(40,92,69)(41,64,96)(44,59,98)(46,83,72)(49,73,79)(50,80,71)(54,81,66)(55,60,86)(57,68,76) ]", "PSL(2,97)^1" ],
  [    976500, "[ (3,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,5,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8)(4,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,6,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38), (1,2,5)(3,4,6)(7,94,38)(8,118,13)(9,57,73)(10,103,26)(11,82,46)(12,52,75)(14,91,35)(15,125,120)(16,83,40)(17,68,54)(18,100,21)(19,97,23)(20,34,86)(22,43,74)(24,85,31)(25,62,53)(27,45,67)(28,63,49)(29,56,55)(30,123,107)(32,64,44)(33,115,112)(36,114,110)(37,41,61)(39,126,95)(42,119,98)(47,99,113)(48,109,102)(50,117,93)(51,122,87)(58,81,121)(59,90,111)(60,76,124)(65,116,79)(66,88,106)(69,101,89)(70,105,84)(71,108,80)(72,92,96)(77,104,78) ]", "PSL(2,125)^1" ],
  [    979200, "[ (7,8,9)(10,14,18)(11,16,21)(12,17,19)(13,15,20)(23,24,25)(26,30,34)(27,32,37)(28,33,35)(29,31,36)(39,40,41)(42,46,50)(43,48,53)(44,49,51)(45,47,52)(55,56,57)(58,62,66)(59,64,69)(60,65,67)(61,63,68)(71,72,73)(74,78,82)(75,80,85)(76,81,83)(77,79,84)(86,115,110)(88,111,122)(89,107,118)(90,119,114)(91,97,150)(92,160,130)(93,142,95)(94,137,170)(96,164,126)(98,133,158)(99,105,146)(100,156,138)(101,154,103)(102,129,166)(104,168,134)(106,125,162)(108,116,112)(109,117,121)(123,136,145)(124,141,135)(127,132,153)(128,149,131)(140,159,157)(144,167,165)(148,163,169)(152,155,161), (1,6,22,42,3)(2,7,38,10,39)(4,8,54,62,49)(5,9,70,82,52)(11,23,26,27,43)(12,71,34,61,48)(13,55,30,76,53)(14,56,46,17,40)(15,72,66,79,51)(16,24,74,37,45)(18,73,50,20,41)(19,57,78,67,47)(21,25,58,32,44)(28,59,64,65,33)(29,75,85,84,36)(31,60,80,83,68)(35,77,69,63,81)(86,166,94,96,138)(87,157,121,112,128)(88,90,167,154,123)(89,160,136,165,133)(91,124,169,95,110)(92,111,119,98,146)(93,153,152,97,103)(99,139,137,164,147)(100,102,120,151,113)(101,129,170,126,156)(104,148,122,125,159)(105,117,135,163,116)(106,131,168,114,141)(107,134,155,127,162)(108,118,109,150,142)(115,144,132,161,145)(130,158,140,143,149) ]", "PSp(4,4)^1" ],
  [   1024128, "[ (3,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,66)(3,120,12)(4,48,83)(5,33,97)(6,59,70)(7,40,88)(8,96,31)(9,21,105)(10,79,46)(11,67,57)(13,42,80)(14,91,30)(15,22,98)(16,19,100)(17,71,47)(18,23,94)(20,78,37)(24,55,56)(25,50,60)(26,58,51)(27,111,123)(28,68,39)(29,43,63)(32,113,116)(34,110,117)(35,99,127)(36,124,101)(38,109,114)(41,118,102)(44,92,125)(49,84,128)(52,90,119)(53,122,86)(54,112,95)(61,115,85)(62,73,126)(64,104,93)(65,121,75)(69,89,103)(72,82,107)(74,106,81)(76,77,108) ]", "PSL(2,127)^1" ],
  [   1030200, "[ (3,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,53)(3,52,54)(4,84,21)(5,29,75)(6,15,88)(7,23,79)(8,71,30)(9,63,37)(10,42,57)(11,16,82)(12,50,47)(13,60,36)(14,62,33)(17,67,25)(18,91,100)(19,32,58)(20,55,34)(22,102,85)(24,90,95)(26,45,38)(27,83,99)(28,40,41)(31,77,101)(35,98,76)(39,89,81)(43,97,69)(44,92,73)(46,93,70)(48,74,87)(49,64,96)(51,86,72)(56,94,59)(61,80,68)(65,66,78) ]", "PSL(2,101)^1" ],
  [   1036800, "[ (1,13,9,19,3,16,10,21)(2,17,12,22,6,14,7,20,5,18,11,24,4,15,8,23), (1,7,3,10,5,11,2,9)(4,8,6,12)(13,19,15,22,17,23,14,21)(16,20,18,24), (5,6)(7,9)(8,10)(11,12)(17,18)(19,21)(20,22)(23,24), (13,14,15,16,17)(20,22,23,21,24) ]", "A6^2" ],
  [   1036800, "[ (1,13,9,19,3,16,10,21)(2,17,12,22,6,14,7,20,5,18,11,24,4,15,8,23), (13,19,15,22,17,23,14,21)(16,20,18,24), (1,7,3,10,5,11,2,9)(4,8,6,12)(13,19,15,22,17,23,14,21)(16,20,18,24), (16,17,18)(19,24,22)(20,23,21) ]", "A6^2" ],
  [   1036800, "[ (1,13,9,19,3,16,10,21)(2,17,12,22,6,14,7,20,5,18,11,24,4,15,8,23), (5,6)(7,9)(8,10)(11,12)(13,21)(14,19,15,20,17,24,16,22,18,23), (1,7,3,10,5,11,2,9)(4,8,6,12)(13,19,15,22,17,23,14,21)(16,20,18,24), (13,14,15,16,17)(20,22,23,21,24) ]", "A6^2" ],
  [   1036800, "[ (1,13,9,21)(2,18,11,23)(3,16,10,19)(4,15,8,24)(5,17,12,20)(6,14,7,22), (1,7,3,10,5,11,2,9)(4,8,6,12)(13,19,15,22,17,23,14,21)(16,20,18,24), (5,6)(7,9)(8,10)(11,12)(17,18)(19,21)(20,22)(23,24), (16,17,18)(19,24,22)(20,23,21) ]", "A6^2" ],
  [   1036800, "[ (1,13,9,21)(2,18,11,23)(3,16,10,19)(4,15,8,24)(5,17,12,20)(6,14,7,22), (13,21)(14,19,15,20,17,24,16,22,18,23), (1,9)(2,7,3,8,5,12,4,10,6,11)(13,21)(14,19,15,20,17,24,16,22,18,23), (2,5,3)(7,8,10)(9,11,12) ]", "A6^2" ],
  [   1036800, "[ (1,13,9,21)(2,18,11,23)(3,16,10,19)(4,15,8,24)(5,17,12,20)(6,14,7,22), (5,6)(7,9)(8,10)(11,12)(13,19,15,22,17,23,14,21)(16,20,18,24), (1,9)(2,7,3,8,5,12,4,10,6,11)(13,21)(14,19,15,20,17,24,16,22,18,23), (1,6,4,3,2)(7,12,8,10,11) ]", "A6^2" ],
  [   1036800, "[ (1,19,3,20,5,24,4,22,6,21)(2,23)(7,18,11,16,10,13,9,15,8,14)(12,17), (5,6)(7,9)(8,10)(11,12)(17,18)(19,21)(20,22)(23,24), (1,7,3,10,5,11,2,9)(4,8,6,12)(13,19,15,22,17,23,14,21)(16,20,18,24), (16,17,18)(19,24,22)(20,23,21) ]", "A6^2" ],
  [   1036800, "[ (1,21)(2,24,4,20,5,23)(3,22,6,19)(7,17,12,18,11,16,10,13,9,15,8,14), (1,7,3,10,5,11,2,9)(4,8,6,12)(13,19,15,22,17,23,14,21)(16,20,18,24), (5,6)(7,9)(8,10)(11,12)(17,18)(19,21)(20,22)(23,24), (13,14,15,16,17)(20,22,23,21,24) ]", "A6^2" ],
  [   1036800, "[ (1,21)(2,24,4,20,5,23)(3,22,6,19)(7,17,12,18,11,16,10,13,9,15,8,14), (1,7,3,10,5,11,2,9)(4,8,6,12)(13,21)(14,19,15,20,17,24,16,22,18,23), (5,6)(7,9)(8,10)(11,12)(17,18)(19,21)(20,22)(23,24), (2,5,3)(7,8,10)(9,11,12) ]", "A6^2" ],
  [   1036800, "[ (1,21)(2,24,4,20,5,23)(3,22,6,19)(7,17,12,18,11,16,10,13,9,15,8,14), (17,18)(19,21)(20,22)(23,24), (5,6)(7,9)(8,10)(11,12)(17,18)(19,21)(20,22)(23,24), (2,5,3)(7,8,10)(9,11,12) ]", "A6^2" ],
  [   1062720, "[ (3,82,79,71,4,46,43,35)(5,37,16,26,8,73,52,62)(6,64,25,53,9,28,61,17)(7,19,34,80,10,55,70,44)(11,12,15,23,47,48,51,59)(13,18,31,72,49,54,67,36)(14,20,39,24,50,56,75,60)(21,42,32,74,57,78,68,38)(22,45,40,27,58,81,76,63)(29,66,33,77,65,30,69,41), (11,19)(12,27)(13,35)(14,43)(15,51)(16,59)(17,67)(18,75)(20,28)(21,36)(22,44)(23,52)(24,60)(25,68)(26,76)(29,37)(30,45)(31,53)(32,61)(33,69)(34,77)(38,46)(39,54)(40,62)(41,70)(42,78)(47,55)(48,63)(49,71)(50,79)(56,64)(57,72)(58,80)(65,73)(66,81)(74,82), (3,81,79,77,75,10,72,70,68,66,9,63,61,59,57,8,54,52,50,48,4,45,43,41,39,7,36,34,32,30,6,27,25,23,21,5,18,16,14,12)(11,82,80,78,76,74,73,71,69,67,65,64,62,60,58,56,55,53,51,49,47,46,44,42,40,38,37,35,33,31,29,28,26,24,22,20,19,17,15,13), (1,2,4)(5,6,8)(7,9,10)(11,21,71)(12,66,25)(13,43,47)(14,55,35)(15,37,52)(16,34,54)(17,63,24)(18,20,65)(19,36,49)(22,72,82)(23,51,29)(26,33,45)(27,81,68)(28,73,75)(30,76,69)(31,32,40)(38,79,58)(39,59,77)(41,56,78)(42,70,64)(44,57,74)(46,50,80)(48,60,67)(53,61,62) ]", "PSL(2,81)^1" ],
  [   1062720, "[ (3,82,81,80,79,78,77,76,75,74,10,73,72,71,70,69,68,67,66,65,9,64,63,62,61,60,59,58,57,56,8,55,54,53,52,51,50,49,48,47,4,46,45,44,43,42,41,40,39,38,7,37,36,35,34,33,32,31,30,29,6,28,27,26,25,24,23,22,21,20,5,19,18,17,16,15,14,13,12,11), (11,19)(12,27)(13,35)(14,43)(15,51)(16,59)(17,67)(18,75)(20,28)(21,36)(22,44)(23,52)(24,60)(25,68)(26,76)(29,37)(30,45)(31,53)(32,61)(33,69)(34,77)(38,46)(39,54)(40,62)(41,70)(42,78)(47,55)(48,63)(49,71)(50,79)(56,64)(57,72)(58,80)(65,73)(66,81)(74,82), (1,2,4)(5,6,8)(7,9,10)(11,21,71)(12,66,25)(13,43,47)(14,55,35)(15,37,52)(16,34,54)(17,63,24)(18,20,65)(19,36,49)(22,72,82)(23,51,29)(26,33,45)(27,81,68)(28,73,75)(30,76,69)(31,32,40)(38,79,58)(39,59,77)(41,56,78)(42,70,64)(44,57,74)(46,50,80)(48,60,67)(53,61,62) ]", "PSL(2,81)^1" ],
  [   1062720, "[ (5,7)(6,9)(8,10)(11,13,19,35)(12,16,27,59)(14,21,43,36)(15,24,51,60)(17,29,67,37)(18,32,75,61)(20,40,28,62)(22,46,44,38)(23,48,52,63)(25,54,68,39)(26,56,76,64)(30,70,45,41)(31,73,53,65)(33,78,69,42)(34,81,77,66)(47,49,55,71)(50,57,79,72)(58,82,80,74), (11,19)(12,27)(13,35)(14,43)(15,51)(16,59)(17,67)(18,75)(20,28)(21,36)(22,44)(23,52)(24,60)(25,68)(26,76)(29,37)(30,45)(31,53)(32,61)(33,69)(34,77)(38,46)(39,54)(40,62)(41,70)(42,78)(47,55)(48,63)(49,71)(50,79)(56,64)(57,72)(58,80)(65,73)(66,81)(74,82), (3,81,79,77,75,10,72,70,68,66,9,63,61,59,57,8,54,52,50,48,4,45,43,41,39,7,36,34,32,30,6,27,25,23,21,5,18,16,14,12)(11,82,80,78,76,74,73,71,69,67,65,64,62,60,58,56,55,53,51,49,47,46,44,42,40,38,37,35,33,31,29,28,26,24,22,20,19,17,15,13), (1,2,4)(5,6,8)(7,9,10)(11,21,71)(12,66,25)(13,43,47)(14,55,35)(15,37,52)(16,34,54)(17,63,24)(18,20,65)(19,36,49)(22,72,82)(23,51,29)(26,33,45)(27,81,68)(28,73,75)(30,76,69)(31,32,40)(38,79,58)(39,59,77)(41,56,78)(42,70,64)(44,57,74)(46,50,80)(48,60,67)(53,61,62) ]", "PSL(2,81)^1" ],
  [   1092624, "[ (3,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,54)(3,10,98)(4,73,34)(5,40,66)(6,30,75)(7,24,80)(8,92,11)(9,21,81)(12,23,76)(13,60,38)(14,46,51)(15,49,47)(16,100,97)(17,39,55)(18,64,29)(19,56,36)(20,26,65)(22,31,58)(25,45,41)(27,87,99)(28,84,101)(32,85,96)(33,78,102)(35,104,74)(42,68,103)(43,82,88)(44,90,79)(48,95,70)(50,77,86)(52,89,72)(53,69,91)(57,62,94)(59,93,61)(63,83,67) ]", "PSL(2,103)^1" ],
  [   1123980, "[ (3,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,68)(3,67,69)(4,126,9)(5,22,112)(6,54,79)(7,25,107)(8,70,61)(10,132,127)(11,87,41)(12,77,50)(13,110,16)(14,28,97)(15,52,72)(17,21,101)(18,33,88)(19,27,93)(20,81,38)(23,42,74)(24,114,131)(26,123,120)(29,111,129)(30,58,51)(31,71,37)(32,63,44)(34,60,45)(35,115,119)(36,47,56)(39,108,122)(40,46,53)(43,109,117)(48,103,118)(49,125,95)(55,116,98)(57,82,130)(59,124,86)(62,94,113)(64,84,121)(65,105,99)(66,128,75)(73,104,92)(76,102,91)(78,106,85)(80,89,100)(83,90,96) ]", "PSL(2,131)^1" ],
  [   1209600, "[ (5,72)(7,37)(9,64)(10,65)(11,30)(12,34)(13,24)(14,46)(15,48)(16,35)(17,75)(18,56)(20,98)(21,52)(22,89)(23,39)(26,50)(27,85)(28,43)(29,92)(32,96)(33,71)(36,44)(38,93)(40,90)(41,79)(42,51)(45,55)(49,84)(53,97)(57,78)(58,88)(59,81)(61,83)(62,76)(63,100)(66,74)(68,70)(69,77)(73,87)(82,86)(91,99)(94,95), (1,58,96,14,36,4,25,71)(2,20,85,42,50,92,64,31)(3,51,5,17,89,34,99,11)(6,70)(7,90,12,19,82,86,54,23)(8,43,37,57,77,75,30,73)(9,52,53,95)(10,59,91,28,97,32,68,40)(13,56,74,78,80,46,38,21)(15,100,63,41,27,67,22,35)(16,69,83,24)(18,33,45,81,55,29,98,60)(26,72,47,62)(39,76)(44,79,94,65,88,61,66,49)(48,84) ]", "J_2^1" ],
  [   1224936, "[ (3,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,56)(3,55,57)(4,92,19)(5,9,101)(6,22,87)(7,27,81)(8,70,37)(10,33,72)(11,103,107)(12,86,17)(13,28,74)(14,52,49)(15,29,71)(16,46,53)(18,58,39)(20,108,93)(21,62,32)(23,47,45)(24,43,48)(25,90,106)(26,100,95)(30,51,34)(31,85,105)(35,44,36)(38,84,99)(40,79,102)(41,83,97)(42,104,75)(50,91,80)(54,94,73)(59,66,96)(60,98,63)(61,82,78)(64,69,88)(65,89,67)(68,77,76) ]", "PSL(2,107)^1" ],
  [   1285608, "[ (3,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,71)(3,61,81)(4,51,90)(5,122,18)(6,9,130)(7,78,60)(8,53,84)(10,119,16)(11,87,47)(12,133,136)(13,70,62)(14,29,102)(15,25,105)(17,36,92)(19,98,28)(20,137,124)(21,48,76)(22,30,93)(23,132,126)(24,46,75)(26,77,42)(27,32,86)(31,35,79)(33,69,43)(34,57,54)(37,117,127)(38,68,39)(40,113,128)(41,59,45)(44,123,114)(49,112,120)(50,106,125)(52,91,138)(55,131,95)(56,110,115)(58,89,134)(63,107,111)(64,135,82)(65,116,100)(66,94,121)(67,96,118)(72,129,80)(73,109,99)(74,104,103)(83,101,97)(85,108,88) ]", "PSL(2,137)^1" ],
  [   1294920, "[ (3,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,57)(3,108,6)(4,17,96)(5,52,60)(7,28,82)(8,33,76)(9,89,19)(10,13,94)(11,55,51)(12,71,34)(14,26,77)(15,72,30)(16,53,48)(18,97,110)(20,101,104)(21,22,74)(23,45,49)(24,58,35)(25,105,95)(27,46,44)(29,47,41)(31,36,50)(32,86,107)(37,88,100)(38,81,106)(40,92,93)(42,99,84)(43,102,80)(54,62,109)(56,90,79)(59,103,63)(61,98,66)(64,78,83)(65,69,91)(67,85,73)(68,87,70) ]", "PSL(2,109)^1" ],
  [   1296000, "[ (1,15,6)(2,14,7)(3,11,8)(4,12,9)(5,13,10), (4,5)(9,10)(14,15), (13,14,15), (1,5,4,3,2) ]", "A5^3" ],
  [   1296000, "[ (4,5)(6,12,10,13,9,15)(7,14)(8,11), (1,15,6)(2,14,7)(3,11,8)(4,12,9)(5,13,10), (11,12,13,14,15) ]", "A5^3" ],
  [   1296000, "[ (6,12,9,14,7,15)(8,11)(10,13), (1,15,6)(2,14,7)(3,11,8)(4,12,9)(5,13,10), (1,5,4,3,2) ]", "A5^3" ],
  [   1342740, "[ (3,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,72)(3,71,73)(4,31,112)(5,124,18)(6,128,13)(7,103,37)(8,93,46)(9,60,78)(10,54,83)(11,50,86)(12,26,109)(14,57,76)(15,92,40)(16,138,131)(17,34,96)(19,101,27)(20,139,126)(21,59,67)(22,80,45)(23,36,88)(24,81,42)(25,53,69)(28,89,30)(29,39,79)(32,113,140)(33,44,70)(35,118,132)(38,62,47)(41,137,107)(43,125,117)(48,110,127)(51,136,98)(52,129,104)(55,116,114)(56,108,121)(58,94,133)(61,90,134)(63,120,102)(64,122,99)(65,105,115)(66,84,135)(68,87,130)(74,100,111)(75,91,119)(77,85,123)(82,106,97) ]", "PSL(2,139)^1" ],
  [   1442784, "[ (3,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,59)(3,47,71)(4,35,82)(5,76,40)(6,27,88)(7,65,49)(8,45,68)(9,13,99)(10,77,34)(11,23,87)(12,85,24)(14,89,18)(15,58,48)(16,51,54)(17,72,32)(19,105,109)(20,80,21)(22,55,44)(25,39,57)(26,52,43)(28,37,56)(29,104,100)(30,91,112)(31,95,107)(33,106,94)(36,83,114)(38,98,97)(41,108,84)(42,113,78)(46,101,86)(50,73,110)(53,111,69)(60,103,70)(61,79,93)(62,81,90)(63,96,74)(64,67,102)(66,92,75) ]", "PSL(2,113)^1" ],
  [   1451520, "[ (4,5)(6,7)(12,13)(14,15)(20,21)(22,23)(28,29)(30,31)(32,41)(33,40)(34,43)(35,42)(36,44)(37,45)(38,46)(39,47)(48,57)(49,56)(50,59)(51,58)(52,60)(53,61)(54,62)(55,63), (1,8,16,32,4,2)(3,9,24,48,36,6)(5,10,17,40,20,34)(7,11,25,56,52,38)(12,18,33)(13,26,49,44,22,35)(14,19,41,28,50,37)(15,27,57,60,54,39)(21,42)(23,43,29,58,53,46)(30,51,45)(31,59,61,62,55,47) ]", "PSp(6,2)^1" ],
  [   1524096, "[ (1,29)(2,42,20,41)(3,55)(4,48,9,49)(5,31,17,30)(6,37,27,44)(7,45,14,53)(8,35,18,34)(10,40,25,56)(11,43)(12,50,19,54)(13,52,26,38)(15,36,16,47)(21,51,28,39)(22,33,23,32)(24,46), (2,27,9,8,22,24,28,26,4)(3,13,17,12,23,15,19,6,25)(5,21,14,10,18,7,16,20,11)(30,39,45,56,35,53,36,42,43)(31,54,33,47,50,44,40,55,38)(32,46,51,52,49,41,37,48,34), (30,36,56)(31,40,47)(32,37,52)(33,38,44)(34,41,51)(35,39,42)(43,53,45)(46,48,49)(50,54,55), (1,7,28,27,23,20,11)(2,14,18,15,17,16,10)(3,9,6,26,24,19,12)(4,22,8,5,25,21,13) ]", "PSL(2,8)^2" ],
  [   1524096, "[ (1,29)(2,42,20,41)(3,55)(4,48,9,49)(5,31,17,30)(6,37,27,44)(7,45,14,53)(8,35,18,34)(10,40,25,56)(11,43)(12,50,19,54)(13,52,26,38)(15,36,16,47)(21,51,28,39)(22,33,23,32)(24,46), (2,27,9,8,22,24,28,26,4)(3,13,17,12,23,15,19,6,25)(5,21,14,10,18,7,16,20,11)(30,53,39,36,45,42,56,43,35)(31,44,54,40,33,55,47,38,50)(32,41,46,37,51,48,52,34,49), (1,15,9)(3,10,21)(4,14,26)(5,16,23)(6,17,20)(7,18,8)(11,22,25)(12,24,13)(19,27,28) ]", "PSL(2,8)^2" ],
  [   1572480, "[ (6,7,9)(8,11,10)(12,26,15)(13,39,19)(14,52,22)(16,27,29)(17,40,32)(18,53,35)(20,41,46)(21,54,49)(23,28,56)(24,42,59)(25,55,63)(30,43,33)(31,57,36)(34,44,47)(37,45,60)(38,58,64)(48,61,50)(51,62,65), (4,5)(12,19)(13,26)(14,32)(15,39)(16,46)(17,52)(18,59)(20,27)(21,33)(22,40)(23,47)(24,53)(25,60)(28,34)(29,41)(30,54)(31,61)(35,42)(36,48)(37,55)(38,62)(43,49)(44,56)(45,63)(50,57)(51,64)(58,65), (3,64,62,60,58,57,55,53,51,10,48,47,45,43,42,40,38,36,8,33,31,4,28,27,25,23,21,6,18,16,14,12,65,63,61,59,11,56,54,52,50,49,5,46,44,9,41,39,37,35,34,32,30,29,7,26,24,22,20,19,17,15,13), (1,2,3)(6,9,7)(8,11,10)(12,18,58)(13,24,51)(14,54,20)(15,35,38)(16,40,33)(17,43,29)(19,59,65)(21,46,22)(23,63,57)(25,36,28)(26,53,64)(27,32,30)(31,56,55)(34,60,48)(37,61,44)(39,42,62)(41,52,49)(45,50,47) ]", "PSL(2,64)^1" ],
  [   1653900, "[ (3,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,77)(3,76,78)(4,138,15)(5,121,31)(6,51,100)(7,101,49)(8,29,120)(9,68,80)(10,45,102)(11,110,36)(12,28,117)(13,124,20)(14,79,64)(16,150,139)(17,47,93)(18,42,97)(19,56,82)(21,111,25)(22,69,66)(23,71,63)(24,41,92)(26,104,27)(30,141,134)(32,65,60)(33,149,123)(34,125,146)(35,67,55)(37,126,142)(38,46,73)(39,70,48)(40,58,59)(43,133,129)(44,143,118)(50,128,127)(52,109,144)(53,147,105)(54,103,148)(57,112,136)(61,107,137)(62,113,130)(72,98,135)(74,86,145)(75,140,90)(81,108,116)(83,131,91)(84,115,106)(85,132,88)(87,119,99)(89,122,94)(95,96,114) ]", "PSL(2,149)^1" ],
  [   1721400, "[ (3,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,78)(3,8,148)(4,11,144)(5,62,92)(6,133,20)(7,55,97)(9,83,67)(10,95,54)(12,145,152)(13,106,40)(14,36,109)(15,46,98)(16,22,121)(17,113,29)(18,71,70)(19,68,72)(21,42,96)(23,150,136)(24,44,91)(25,27,107)(26,34,99)(28,100,31)(30,63,66)(32,76,51)(33,45,81)(35,134,140)(37,48,74)(38,52,69)(39,79,41)(43,139,127)(47,120,142)(49,129,131)(50,143,116)(56,128,125)(57,122,130)(58,110,141)(59,101,149)(60,114,135)(61,146,102)(64,94,151)(65,112,132)(73,147,89)(75,111,123)(77,117,115)(80,124,105)(82,108,119)(84,88,137)(85,138,86)(87,104,118)(90,93,126) ]", "PSL(2,151)^1" ],
  [   1742400, "[ (1,13,6,20)(2,18)(3,16,5,21)(4,14,9,23)(7,19,12,22)(8,24,11,15)(10,17), (3,6)(4,12)(5,9)(7,11)(8,10)(15,18)(16,24)(17,21)(19,23)(20,22), (1,4,2)(3,8,6)(5,9,7)(10,12,11) ]", "PSL(2,11)^2" ],
  [   1742400, "[ (1,13,6,22,7,23,4,14,9,19,12,20)(2,15,8,16,5,17,10,21,3,24,11,18), (3,6)(4,12)(5,9)(7,11)(8,10)(15,18)(16,24)(17,21)(19,23)(20,22), (13,15,14)(16,24,22)(17,23,19)(18,20,21) ]", "PSL(2,11)^2" ],
  [   1771440, "[ (13,23)(14,33)(15,43)(16,53)(17,63)(18,73)(19,83)(20,93)(21,103)(22,113)(24,34)(25,44)(26,54)(27,64)(28,74)(29,84)(30,94)(31,104)(32,114)(35,45)(36,55)(37,65)(38,75)(39,85)(40,95)(41,105)(42,115)(46,56)(47,66)(48,76)(49,86)(50,96)(51,106)(52,116)(57,67)(58,77)(59,87)(60,97)(61,107)(62,117)(68,78)(69,88)(70,98)(71,108)(72,118)(79,89)(80,99)(81,109)(82,119)(90,100)(91,110)(92,120)(101,111)(102,121)(112,122), (3,121,119,117,115,113,12,110,108,106,104,102,11,99,97,95,93,91,10,88,86,84,82,80,9,77,75,73,71,69,8,66,64,62,60,58,7,55,53,51,49,47,6,44,42,40,38,36,5,33,31,29,27,25,4,22,20,18,16,14)(13,122,120,118,116,114,112,111,109,107,105,103,101,100,98,96,94,92,90,89,87,85,83,81,79,78,76,74,72,70,68,67,65,63,61,59,57,56,54,52,50,48,46,45,43,41,39,37,35,34,32,30,28,26,24,23,21,19,17,15), (1,2,8)(3,7,9)(4,10,5)(6,12,11)(13,43,91)(14,96,37)(15,110,23)(16,90,41)(17,57,73)(18,63,67)(19,20,109)(21,32,95)(22,46,79)(24,59,64)(25,120,112)(26,115,116)(27,34,87)(28,58,61)(29,38,80)(30,35,82)(31,69,47)(33,50,65)(36,60,51)(39,121,98)(40,103,114)(42,52,54)(44,92,122)(45,119,94)(48,101,108)(53,100,105)(55,97,106)(56,89,113)(62,78,118)(66,104,88)(68,72,117)(70,85,102)(71,76,111)(74,77,107)(75,99,84)(81,83,93) ]", "PSL(2,121)^1" ],
  [   1771440, "[ (3,122,12,111,11,100,10,89,9,78,8,67,7,56,6,45,5,34,4,23)(13,22,112,121,101,110,90,99,79,88,68,77,57,66,46,55,35,44,24,33)(14,32,113,21,102,120,91,109,80,98,69,87,58,76,47,65,36,54,25,43)(15,42,114,31,103,20,92,119,81,108,70,97,59,86,48,75,37,64,26,53)(16,52,115,41,104,30,93,19,82,118,71,107,60,96,49,85,38,74,27,63)(17,62,116,51,105,40,94,29,83,18,72,117,61,106,50,95,39,84,28,73), (3,121,119,117,115,113,12,110,108,106,104,102,11,99,97,95,93,91,10,88,86,84,82,80,9,77,75,73,71,69,8,66,64,62,60,58,7,55,53,51,49,47,6,44,42,40,38,36,5,33,31,29,27,25,4,22,20,18,16,14)(13,122,120,118,116,114,112,111,109,107,105,103,101,100,98,96,94,92,90,89,87,85,83,81,79,78,76,74,72,70,68,67,65,63,61,59,57,56,54,52,50,48,46,45,43,41,39,37,35,34,32,30,28,26,24,23,21,19,17,15), (1,2,8)(3,7,9)(4,10,5)(6,12,11)(13,43,91)(14,96,37)(15,110,23)(16,90,41)(17,57,73)(18,63,67)(19,20,109)(21,32,95)(22,46,79)(24,59,64)(25,120,112)(26,115,116)(27,34,87)(28,58,61)(29,38,80)(30,35,82)(31,69,47)(33,50,65)(36,60,51)(39,121,98)(40,103,114)(42,52,54)(44,92,122)(45,119,94)(48,101,108)(53,100,105)(55,97,106)(56,89,113)(62,78,118)(66,104,88)(68,72,117)(70,85,102)(71,76,111)(74,77,107)(75,99,84)(81,83,93) ]", "PSL(2,121)^1" ],
  [   1771440, "[ (3,122,121,120,119,118,117,116,115,114,113,112,12,111,110,109,108,107,106,105,104,103,102,101,11,100,99,98,97,96,95,94,93,92,91,90,10,89,88,87,86,85,84,83,82,81,80,79,9,78,77,76,75,74,73,72,71,70,69,68,8,67,66,65,64,63,62,61,60,59,58,57,7,56,55,54,53,52,51,50,49,48,47,46,6,45,44,43,42,41,40,39,38,37,36,35,5,34,33,32,31,30,29,28,27,26,25,24,4,23,22,21,20,19,18,17,16,15,14,13), (1,2,8)(3,7,9)(4,10,5)(6,12,11)(13,43,91)(14,96,37)(15,110,23)(16,90,41)(17,57,73)(18,63,67)(19,20,109)(21,32,95)(22,46,79)(24,59,64)(25,120,112)(26,115,116)(27,34,87)(28,58,61)(29,38,80)(30,35,82)(31,69,47)(33,50,65)(36,60,51)(39,121,98)(40,103,114)(42,52,54)(44,92,122)(45,119,94)(48,101,108)(53,100,105)(55,97,106)(56,89,113)(62,78,118)(66,104,88)(68,72,117)(70,85,102)(71,76,111)(74,77,107)(75,99,84)(81,83,93) ]", "PSL(2,121)^1" ],
  [   1814400, "[ (1,2,3,4,5,6,7,8,9), (8,9,10) ]", "A10^1" ],
  [   1876896, "[ (3,4,5,6,7,8)(10,15,14,13,12,11)(16,44,30)(17,50,35,20,47,32)(18,45,36,21,48,33)(19,46,31,22,49,34)(23,51,37)(24,57,42,27,54,39)(25,52,43,28,55,40)(26,53,38,29,56,41)(60,65,64,63,62,61)(67,68,69,70,71,72)(73,87,101)(74,89,104,77,92,107)(75,90,105,78,93,102)(76,91,106,79,88,103)(80,94,108)(81,96,111,84,99,114)(82,97,112,85,100,109)(83,98,113,86,95,110), (1,2,9,13,41,3,16,11,27,57,31,54,52,38,6,37)(4,23,15,55,17,46,35,40)(5,30,12,34,33,26,50,21,32,19,18,53,45,28,22,39)(7,44,14,48,49,56,24,36,47,42,8,51,10,20,25,43)(58,59,67,63,95,97,111,100,90,78,101,70,60,74,73,66)(61,81,110,93,99,83,82,75,80,68,62,88,106,77,94,69)(64,102,91,85,96,104,105,112,107,84,89,113,114,79,108,72)(65,109,86,103,98,76,87,71) ]", "PSL(3,7)^1" ],
  [   1934868, "[ (3,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,81)(3,96,66)(4,14,147)(5,70,90)(6,97,62)(7,18,140)(8,27,130)(9,58,98)(10,106,49)(11,95,59)(12,117,36)(13,20,132)(15,148,158)(16,102,47)(17,39,109)(19,75,71)(21,76,68)(22,144,155)(23,118,24)(25,99,41)(26,38,101)(28,89,48)(29,105,31)(30,142,149)(32,135,154)(33,50,82)(34,43,88)(35,78,52)(37,51,77)(40,79,46)(42,69,54)(44,139,138)(45,150,126)(53,123,145)(56,152,113)(57,133,131)(60,146,115)(61,124,136)(63,137,121)(64,104,153)(65,156,100)(67,151,103)(72,92,157)(73,134,114)(74,119,128)(80,112,129)(83,122,116)(84,127,110)(85,111,125)(86,141,94)(87,143,91)(93,120,108) ]", "PSL(2,157)^1" ],
  [   1953000, "[ (3,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,6,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,5,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,4,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7), (1,2,5)(3,4,6)(7,94,38)(8,118,13)(9,57,73)(10,103,26)(11,82,46)(12,52,75)(14,91,35)(15,125,120)(16,83,40)(17,68,54)(18,100,21)(19,97,23)(20,34,86)(22,43,74)(24,85,31)(25,62,53)(27,45,67)(28,63,49)(29,56,55)(30,123,107)(32,64,44)(33,115,112)(36,114,110)(37,41,61)(39,126,95)(42,119,98)(47,99,113)(48,109,102)(50,117,93)(51,122,87)(58,81,121)(59,90,111)(60,76,124)(65,116,79)(66,88,106)(69,101,89)(70,105,84)(71,108,80)(72,92,96)(77,104,78) ]", "PSL(2,125)^1" ],
  [   1958400, "[ (4,5)(8,9)(12,13)(14,18)(15,19)(16,21)(17,20)(24,25)(28,29)(30,34)(31,35)(32,37)(33,36)(40,41)(44,45)(46,50)(47,51)(48,53)(49,52)(54,70)(55,71)(56,73)(57,72)(58,74)(59,75)(60,77)(61,76)(62,82)(63,83)(64,85)(65,84)(66,78)(67,79)(68,81)(69,80)(86,101)(87,143)(88,124)(89,118)(90,169)(91,123)(92,117)(93,144)(94,170)(95,167)(96,126)(97,145)(98,116)(103,115)(104,168)(105,146)(106,125)(108,133)(109,160)(110,154)(111,135)(112,158)(114,148)(119,163)(121,130)(122,141)(128,140)(129,166)(131,159)(132,153)(136,150)(138,156)(142,165)(149,157)(152,161), (7,8,9)(10,14,18)(11,16,21)(12,17,19)(13,15,20)(23,24,25)(26,30,34)(27,32,37)(28,33,35)(29,31,36)(39,40,41)(42,46,50)(43,48,53)(44,49,51)(45,47,52)(55,56,57)(58,62,66)(59,64,69)(60,65,67)(61,63,68)(71,72,73)(74,78,82)(75,80,85)(76,81,83)(77,79,84)(86,115,110)(88,111,122)(89,107,118)(90,119,114)(91,97,150)(92,160,130)(93,142,95)(94,137,170)(96,164,126)(98,133,158)(99,105,146)(100,156,138)(101,154,103)(102,129,166)(104,168,134)(106,125,162)(108,116,112)(109,117,121)(123,136,145)(124,141,135)(127,132,153)(128,149,131)(140,159,157)(144,167,165)(148,163,169)(152,155,161), (1,6,22,42,3)(2,7,38,10,39)(4,8,54,62,49)(5,9,70,82,52)(11,23,26,27,43)(12,71,34,61,48)(13,55,30,76,53)(14,56,46,17,40)(15,72,66,79,51)(16,24,74,37,45)(18,73,50,20,41)(19,57,78,67,47)(21,25,58,32,44)(28,59,64,65,33)(29,75,85,84,36)(31,60,80,83,68)(35,77,69,63,81)(86,166,94,96,138)(87,157,121,112,128)(88,90,167,154,123)(89,160,136,165,133)(91,124,169,95,110)(92,111,119,98,146)(93,153,152,97,103)(99,139,137,164,147)(100,102,120,151,113)(101,129,170,126,156)(104,148,122,125,159)(105,117,135,163,116)(106,131,168,114,141)(107,134,155,127,162)(108,118,109,150,142)(115,144,132,161,145)(130,158,140,143,149) ]", "PSp(4,4)^1" ],
  [   2048256, "[ (3,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,66)(3,120,12)(4,48,83)(5,33,97)(6,59,70)(7,40,88)(8,96,31)(9,21,105)(10,79,46)(11,67,57)(13,42,80)(14,91,30)(15,22,98)(16,19,100)(17,71,47)(18,23,94)(20,78,37)(24,55,56)(25,50,60)(26,58,51)(27,111,123)(28,68,39)(29,43,63)(32,113,116)(34,110,117)(35,99,127)(36,124,101)(38,109,114)(41,118,102)(44,92,125)(49,84,128)(52,90,119)(53,122,86)(54,112,95)(61,115,85)(62,73,126)(64,104,93)(65,121,75)(69,89,103)(72,82,107)(74,106,81)(76,77,108) ]", "PSL(2,127)^1" ],
  [   2073600, "[ (1,13,9,19,3,16,10,21)(2,17,12,22,6,14,7,20,5,18,11,24,4,15,8,23), (13,19,15,22,17,23,14,21)(16,20,18,24), (1,7,3,10,5,11,2,9)(4,8,6,12)(13,19,15,22,17,23,14,21)(16,20,18,24), (5,6)(7,9)(8,10)(11,12)(17,18)(19,21)(20,22)(23,24), (2,5,3)(7,8,10)(9,11,12) ]", "A6^2" ],
  [   2073600, "[ (1,13,9,19,3,16,10,21)(2,17,12,22,6,14,7,20,5,18,11,24,4,15,8,23), (13,21)(14,19,15,20,17,24,16,22,18,23), (1,7,3,10,5,11,2,9)(4,8,6,12)(13,19,15,22,17,23,14,21)(16,20,18,24), (5,6)(7,9)(8,10)(11,12)(17,18)(19,21)(20,22)(23,24), (2,5,3)(7,8,10)(9,11,12) ]", "A6^2" ],
  [   2073600, "[ (1,13,9,19,3,16,10,21)(2,17,12,22,6,14,7,20,5,18,11,24,4,15,8,23), (17,18)(19,21)(20,22)(23,24), (1,7,3,10,5,11,2,9)(4,8,6,12)(13,19,15,22,17,23,14,21)(16,20,18,24), (5,6)(7,9)(8,10)(11,12)(17,18)(19,21)(20,22)(23,24), (2,5,3)(7,8,10)(9,11,12) ]", "A6^2" ],
  [   2073600, "[ (1,13,9,21)(2,18,11,23)(3,16,10,19)(4,15,8,24)(5,17,12,20)(6,14,7,22), (13,19,15,22,17,23,14,21)(16,20,18,24), (1,7,3,10,5,11,2,9)(4,8,6,12)(13,19,15,22,17,23,14,21)(16,20,18,24), (5,6)(7,9)(8,10)(11,12)(17,18)(19,21)(20,22)(23,24), (1,6,4,3,2)(7,12,8,10,11) ]", "A6^2" ],
  [   2073600, "[ (1,13,9,21)(2,18,11,23)(3,16,10,19)(4,15,8,24)(5,17,12,20)(6,14,7,22), (13,21)(14,19,15,20,17,24,16,22,18,23), (1,7,3,10,5,11,2,9)(4,8,6,12)(13,19,15,22,17,23,14,21)(16,20,18,24), (5,6)(7,9)(8,10)(11,12)(17,18)(19,21)(20,22)(23,24), (2,5,3)(7,8,10)(9,11,12) ]", "A6^2" ],
  [   2073600, "[ (1,21)(2,24,4,20,5,23)(3,22,6,19)(7,17,12,18,11,16,10,13,9,15,8,14), (17,18)(19,21)(20,22)(23,24), (1,7,3,10,5,11,2,9)(4,8,6,12)(13,19,15,22,17,23,14,21)(16,20,18,24), (5,6)(7,9)(8,10)(11,12)(17,18)(19,21)(20,22)(23,24), (16,17,18)(19,24,22)(20,23,21) ]", "A6^2" ],
  [   2097024, "[ (3,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6,4,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5), (1,2,3)(4,123,9)(5,116,15)(6,67,63)(7,102,27)(8,76,52)(10,129,124)(11,74,51)(12,40,84)(13,22,101)(14,43,79)(16,75,45)(17,128,118)(18,99,19)(20,87,29)(21,77,38)(23,41,72)(24,73,39)(25,83,28)(26,48,62)(30,64,42)(31,126,106)(32,111,120)(33,68,35)(34,115,114)(36,53,47)(37,44,55)(46,113,104)(49,93,121)(50,108,105)(54,90,119)(56,112,95)(57,125,81)(58,117,88)(59,122,82)(60,109,94)(61,92,110)(65,100,98)(66,127,70)(69,103,91)(71,85,107)(78,89,96)(80,97,86) ]", "PSL(2,128)^1" ],
  [   2125440, "[ (5,7)(6,9)(8,10)(11,13,19,35)(12,16,27,59)(14,21,43,36)(15,24,51,60)(17,29,67,37)(18,32,75,61)(20,40,28,62)(22,46,44,38)(23,48,52,63)(25,54,68,39)(26,56,76,64)(30,70,45,41)(31,73,53,65)(33,78,69,42)(34,81,77,66)(47,49,55,71)(50,57,79,72)(58,82,80,74), (11,19)(12,27)(13,35)(14,43)(15,51)(16,59)(17,67)(18,75)(20,28)(21,36)(22,44)(23,52)(24,60)(25,68)(26,76)(29,37)(30,45)(31,53)(32,61)(33,69)(34,77)(38,46)(39,54)(40,62)(41,70)(42,78)(47,55)(48,63)(49,71)(50,79)(56,64)(57,72)(58,80)(65,73)(66,81)(74,82), (3,82,81,80,79,78,77,76,75,74,10,73,72,71,70,69,68,67,66,65,9,64,63,62,61,60,59,58,57,56,8,55,54,53,52,51,50,49,48,47,4,46,45,44,43,42,41,40,39,38,7,37,36,35,34,33,32,31,30,29,6,28,27,26,25,24,23,22,21,20,5,19,18,17,16,15,14,13,12,11), (1,2,4)(5,6,8)(7,9,10)(11,21,71)(12,66,25)(13,43,47)(14,55,35)(15,37,52)(16,34,54)(17,63,24)(18,20,65)(19,36,49)(22,72,82)(23,51,29)(26,33,45)(27,81,68)(28,73,75)(30,76,69)(31,32,40)(38,79,58)(39,59,77)(41,56,78)(42,70,64)(44,57,74)(46,50,80)(48,60,67)(53,61,62) ]", "PSL(2,81)^1" ],
  [   2165292, "[ (3,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,84)(3,83,85)(4,145,22)(5,69,97)(6,44,121)(7,27,137)(8,98,65)(9,18,144)(10,107,54)(11,55,105)(12,142,17)(13,56,102)(14,76,81)(15,36,120)(16,46,109)(19,37,115)(20,72,79)(21,86,64)(23,164,146)(24,150,159)(25,118,28)(26,156,151)(29,90,52)(30,101,40)(31,141,161)(32,106,33)(34,88,49)(35,95,41)(38,91,42)(39,58,74)(43,68,60)(45,51,75)(47,124,162)(48,132,153)(50,143,140)(53,131,149)(59,122,152)(61,158,114)(62,136,135)(63,113,157)(66,112,155)(67,138,128)(70,160,103)(71,99,163)(73,133,127)(77,130,126)(78,139,116)(80,134,119)(82,147,104)(87,92,154)(89,96,148)(93,117,123)(94,110,129)(100,125,108) ]", "PSL(2,163)^1" ],
  [   2247960, "[ (3,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,68)(3,67,69)(4,126,9)(5,22,112)(6,54,79)(7,25,107)(8,70,61)(10,132,127)(11,87,41)(12,77,50)(13,110,16)(14,28,97)(15,52,72)(17,21,101)(18,33,88)(19,27,93)(20,81,38)(23,42,74)(24,114,131)(26,123,120)(29,111,129)(30,58,51)(31,71,37)(32,63,44)(34,60,45)(35,115,119)(36,47,56)(39,108,122)(40,46,53)(43,109,117)(48,103,118)(49,125,95)(55,116,98)(57,82,130)(59,124,86)(62,94,113)(64,84,121)(65,105,99)(66,128,75)(73,104,92)(76,102,91)(78,106,85)(80,89,100)(83,90,96) ]", "PSL(2,131)^1" ],
  [   2328648, "[ (3,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,86)(3,46,126)(4,118,53)(5,109,61)(6,72,97)(7,83,85)(8,37,130)(9,27,139)(10,34,131)(11,149,15)(12,20,143)(13,47,115)(14,129,32)(16,68,91)(17,48,110)(18,108,49)(19,101,55)(21,80,74)(22,50,103)(23,161,157)(24,95,56)(25,45,105)(26,113,36)(28,82,65)(29,152,160)(30,93,52)(31,78,66)(33,145,163)(35,102,38)(39,76,60)(40,51,84)(41,138,162)(42,135,164)(43,158,140)(44,58,73)(54,168,119)(57,125,159)(59,146,136)(62,124,155)(63,167,111)(64,154,123)(67,127,147)(69,122,150)(70,137,134)(71,153,117)(75,100,166)(77,148,116)(79,142,120)(81,104,156)(87,89,165)(88,121,132)(90,144,107)(92,151,98)(94,141,106)(96,133,112)(99,114,128) ]", "PSL(2,167)^1" ],
  [   2384928, "[ (1,15,14,20,11,25,4,16,8,23,5,26,2,17)(3,18,6,27,9,24,10,22,12,21,7,19,13,28), (15,19,16)(17,25,26)(18,24,27)(20,21,23) ]", "PSL(2,13)^2" ],
  [   2413320, "[ (3,169,167,165,163,161,159,14,156,154,152,150,148,146,13,143,141,139,137,135,133,12,130,128,126,124,122,120,11,117,115,113,111,109,107,10,104,102,100,98,96,94,9,91,89,87,85,83,81,8,78,76,74,72,70,68,7,65,63,61,59,57,55,6,52,50,48,46,44,42,5,39,37,35,33,31,29,4,26,24,22,20,18,16)(15,170,168,166,164,162,160,158,157,155,153,151,149,147,145,144,142,140,138,136,134,132,131,129,127,125,123,121,119,118,116,114,112,110,108,106,105,103,101,99,97,95,93,92,90,88,86,84,82,80,79,77,75,73,71,69,67,66,64,62,60,58,56,54,53,51,49,47,45,43,41,40,38,36,34,32,30,28,27,25,23,21,19,17), (1,2,9)(3,8,10)(4,5,12)(6,13,14)(15,146,38)(16,158,25)(17,68,114)(18,28,153)(19,151,29)(20,130,50)(21,76,103)(22,118,60)(23,115,62)(24,78,98)(26,93,80)(27,169,160)(30,128,41)(31,69,99)(32,157,167)(33,36,131)(34,166,156)(35,104,61)(37,120,42)(39,170,147)(40,97,63)(43,84,72)(44,59,96)(45,108,46)(47,79,74)(48,56,95)(49,85,66)(51,91,58)(52,75,73)(53,64,83)(54,149,152)(55,165,135)(57,155,144)(65,148,143)(67,163,125)(70,162,123)(71,117,168)(77,140,139)(81,150,124)(82,109,164)(86,116,154)(87,107,161)(88,145,122)(89,126,141)(90,129,137)(92,159,105)(94,134,127)(100,136,119)(101,142,113)(102,121,132)(106,138,111)(110,133,112) ]", "PSL(2,169)^1" ],
  [   2571216, "[ (3,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,71)(3,61,81)(4,51,90)(5,122,18)(6,9,130)(7,78,60)(8,53,84)(10,119,16)(11,87,47)(12,133,136)(13,70,62)(14,29,102)(15,25,105)(17,36,92)(19,98,28)(20,137,124)(21,48,76)(22,30,93)(23,132,126)(24,46,75)(26,77,42)(27,32,86)(31,35,79)(33,69,43)(34,57,54)(37,117,127)(38,68,39)(40,113,128)(41,59,45)(44,123,114)(49,112,120)(50,106,125)(52,91,138)(55,131,95)(56,110,115)(58,89,134)(63,107,111)(64,135,82)(65,116,100)(66,94,121)(67,96,118)(72,129,80)(73,109,99)(74,104,103)(83,101,97)(85,108,88) ]", "PSL(2,137)^1" ],
  [   2588772, "[ (3,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,89)(3,88,90)(4,62,115)(5,50,126)(6,35,140)(7,16,158)(8,39,134)(9,92,80)(10,150,21)(11,137,33)(12,147,22)(13,45,123)(14,94,73)(15,113,53)(17,96,68)(18,32,131)(19,108,54)(20,162,171)(23,59,99)(24,25,132)(26,60,95)(27,69,85)(28,168,157)(29,51,101)(30,87,64)(31,166,156)(34,71,76)(36,48,97)(37,104,40)(38,143,172)(41,167,145)(42,61,78)(43,66,72)(44,139,170)(46,153,154)(47,146,160)(49,75,57)(52,128,173)(55,133,165)(56,67,58)(63,116,174)(65,163,125)(70,159,124)(74,141,138)(77,127,149)(79,119,155)(81,130,142)(82,161,110)(83,118,152)(84,164,105)(86,169,98)(91,148,114)(93,109,151)(100,117,136)(102,107,144)(103,129,121)(106,112,135)(111,122,120) ]", "PSL(2,173)^1" ],
  [   2592000, "[ (1,15,6)(2,14,7)(3,11,8)(4,12,9)(5,13,10), (4,5)(9,10), (9,10)(14,15), (6,10,9,8,7) ]", "A5^3" ],
  [   2592000, "[ (6,12,9,14,7,15)(8,11)(10,13), (1,15,6)(2,14,7)(3,11,8)(4,12,9)(5,13,10), (4,5)(9,10)(14,15), (2,4,3) ]", "A5^3" ],
  [   2685480, "[ (3,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,72)(3,71,73)(4,31,112)(5,124,18)(6,128,13)(7,103,37)(8,93,46)(9,60,78)(10,54,83)(11,50,86)(12,26,109)(14,57,76)(15,92,40)(16,138,131)(17,34,96)(19,101,27)(20,139,126)(21,59,67)(22,80,45)(23,36,88)(24,81,42)(25,53,69)(28,89,30)(29,39,79)(32,113,140)(33,44,70)(35,118,132)(38,62,47)(41,137,107)(43,125,117)(48,110,127)(51,136,98)(52,129,104)(55,116,114)(56,108,121)(58,94,133)(61,90,134)(63,120,102)(64,122,99)(65,105,115)(66,84,135)(68,87,130)(74,100,111)(75,91,119)(77,85,123)(82,106,97) ]", "PSL(2,139)^1" ],
  [   2867580, "[ (3,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,92)(3,91,93)(4,162,21)(5,132,50)(6,54,127)(7,104,76)(8,147,32)(9,18,160)(10,82,95)(11,47,129)(12,70,105)(13,17,157)(14,128,45)(15,41,131)(16,26,145)(19,135,33)(20,94,73)(22,180,163)(23,43,121)(24,166,175)(25,46,116)(27,167,171)(28,71,88)(29,83,75)(30,85,72)(31,36,120)(34,86,67)(35,65,87)(37,176,152)(38,107,42)(39,158,168)(40,78,69)(44,59,84)(48,81,58)(49,165,151)(51,74,62)(52,179,134)(53,143,169)(55,137,173)(56,170,139)(57,130,178)(60,66,61)(63,141,161)(64,148,153)(68,138,159)(77,146,142)(79,114,172)(80,177,108)(89,102,174)(90,164,111)(96,113,156)(97,119,149)(98,150,117)(99,154,112)(100,125,140)(101,155,109)(103,136,126)(106,144,115)(110,133,122)(118,124,123) ]", "PSL(2,179)^1" ],
  [   2929500, "[ (7,11,31)(8,16,55)(9,21,79)(10,26,103)(12,36,32)(13,40,56)(14,45,80)(15,50,104)(17,60,33)(18,65,57)(19,69,81)(20,74,105)(22,84,34)(23,89,58)(24,94,82)(25,98,106)(27,108,35)(28,113,59)(29,118,83)(30,123,107)(37,41,61)(38,46,85)(39,51,109)(42,66,62)(43,70,86)(44,75,110)(47,90,63)(48,95,87)(49,99,111)(52,114,64)(53,119,88)(54,124,112)(67,71,91)(68,76,115)(72,96,92)(73,100,116)(77,120,93)(78,125,117)(97,101,121)(102,126,122), (3,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,5,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8)(4,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,6,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38), (1,2,5)(3,4,6)(7,94,38)(8,118,13)(9,57,73)(10,103,26)(11,82,46)(12,52,75)(14,91,35)(15,125,120)(16,83,40)(17,68,54)(18,100,21)(19,97,23)(20,34,86)(22,43,74)(24,85,31)(25,62,53)(27,45,67)(28,63,49)(29,56,55)(30,123,107)(32,64,44)(33,115,112)(36,114,110)(37,41,61)(39,126,95)(42,119,98)(47,99,113)(48,109,102)(50,117,93)(51,122,87)(58,81,121)(59,90,111)(60,76,124)(65,116,79)(66,88,106)(69,101,89)(70,105,84)(71,108,80)(72,92,96)(77,104,78) ]", "PSL(2,125)^1" ],
  [   2964780, "[ (3,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,93)(3,92,94)(4,37,148)(5,117,67)(6,161,22)(7,98,84)(8,155,26)(9,133,47)(10,17,162)(11,136,42)(12,150,27)(13,58,118)(14,44,131)(15,64,110)(16,33,140)(18,90,81)(19,61,109)(20,45,124)(21,82,86)(23,29,137)(24,169,176)(25,180,164)(28,54,107)(30,85,74)(31,178,160)(32,43,114)(34,99,56)(35,89,65)(36,174,159)(38,149,182)(39,59,91)(40,83,66)(41,75,73)(46,153,170)(48,70,71)(49,157,163)(50,175,144)(51,60,78)(52,57,80)(53,177,139)(55,142,172)(62,141,166)(68,128,173)(69,181,119)(72,143,154)(76,122,171)(77,125,167)(79,132,158)(87,152,130)(88,179,102)(95,127,147)(96,168,105)(97,151,121)(100,104,165)(101,156,112)(103,146,120)(106,129,134)(108,126,135)(111,145,113)(115,116,138) ]", "PSL(2,181)^1" ],
  [   3265920, "[ (4,5,7,11)(6,9,15,26)(8,13,22,39)(10,17,18,31)(12,20,35,61)(14,24,43,74)(16,28,50,87)(19,33,23,41)(21,37,65,112)(25,45,78,133)(27,48,83,137)(29,52,91,111)(30,54,94,154)(32,56,97,157)(34,59,101,161)(36,63,108,170)(38,67)(40,70,119,98)(42,72,122,185)(44,76,129,194)(46,80,126,53)(47,81,138,155)(49,85,143,130)(51,89,150,211)(55,95,93,96)(57,58,99,69)(60,103,165,226)(62,106,168,229)(64,110,172,233)(66,84,77,131)(68,116,175,132)(71,120,182,128)(73,124,187,243)(75,127,192,92)(79,135,158,218)(82,140,203,254)(86,145)(88,148,190,201)(90,152,213,178)(100,118,180,209)(102,163,104,141)(105,167,228,219)(107,151,117,179)(109,171,232,252)(113,174,146,208)(114,176,237,195)(115,142,204,251)(121,184,217,249)(123,186,242,255)(125,189,246,273)(134,198,193,177)(136,199,250,196)(139,202,231,147)(144,206,256,236)(149,210,259,265)(153,214,207,166)(156,191,247,260)(159,220,266,253)(160,181,238,258)(162,224,257,244)(164,215,262,205)(169,230,271,279)(173,235,269,200)(183,240,197,248)(188,245,241,223)(212,261,275,264)(216,263,276,234)(221,267,277,274)(222,268,278,239)(225,227,270,272), (1,2,3,4,6,10,18,32,57)(5,8,14,25,46,81,139,180,238)(7,12,21,38,68,117,99,159,221)(9,16,29,53,93,28,51,90,153)(11,19,34,60,104,87,147,209,258)(13,23,42,73,125,190,91,120,183)(15,27,49,86,146,145,143,204,97)(17,30,54)(20,36,64,111,135,37,66,114,177)(22,40,71,121,148,203,243,214,41)(24,44,77,132,196,249,233,89,151)(26,47,82,141,94,155,216,250,240)(31,55,96)(33,58,100,160,222,232,262,165,227)(35,62,107)(39,69,118,181,239,210,260,78,134)(43,75,128,194,131,176,182,127,193)(45,79,136)(48,84,142,205,255,271,279,152,163)(50,88,149)(52,92,129,174,206,257,273,263,108)(56,98,158,219,265,154,215,246,259)(59,102,164,225,70,76,130,195,248)(61,105,110,140,186,112,173,236,270)(63,109,95,156,217,171,157,72,123)(65,113,175)(67,115,83)(74,126,191,198,185,241,85,144,207)(80,137,200,251,247,218,264,275,199)(101,162,124,188,235,256,187,244,272)(103,166,122)(106,169,167,202,253,274,268,150,212)(116,178,189,172,234,179,161,223,269)(119,133,197)(138,201,252)(170,231,220,267,278,276,230,229,261)(184,211,228,254,224,245,208,237,192)(213,226,242)(266,277,280) ]", "PSU(4,3)^1" ],
  [   3307800, "[ (3,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,77)(3,76,78)(4,138,15)(5,121,31)(6,51,100)(7,101,49)(8,29,120)(9,68,80)(10,45,102)(11,110,36)(12,28,117)(13,124,20)(14,79,64)(16,150,139)(17,47,93)(18,42,97)(19,56,82)(21,111,25)(22,69,66)(23,71,63)(24,41,92)(26,104,27)(30,141,134)(32,65,60)(33,149,123)(34,125,146)(35,67,55)(37,126,142)(38,46,73)(39,70,48)(40,58,59)(43,133,129)(44,143,118)(50,128,127)(52,109,144)(53,147,105)(54,103,148)(57,112,136)(61,107,137)(62,113,130)(72,98,135)(74,86,145)(75,140,90)(81,108,116)(83,131,91)(84,115,106)(85,132,88)(87,119,99)(89,122,94)(95,96,114) ]", "PSL(2,149)^1" ],
  [   3442800, "[ (3,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,78)(3,8,148)(4,11,144)(5,62,92)(6,133,20)(7,55,97)(9,83,67)(10,95,54)(12,145,152)(13,106,40)(14,36,109)(15,46,98)(16,22,121)(17,113,29)(18,71,70)(19,68,72)(21,42,96)(23,150,136)(24,44,91)(25,27,107)(26,34,99)(28,100,31)(30,63,66)(32,76,51)(33,45,81)(35,134,140)(37,48,74)(38,52,69)(39,79,41)(43,139,127)(47,120,142)(49,129,131)(50,143,116)(56,128,125)(57,122,130)(58,110,141)(59,101,149)(60,114,135)(61,146,102)(64,94,151)(65,112,132)(73,147,89)(75,111,123)(77,117,115)(80,124,105)(82,108,119)(84,88,137)(85,138,86)(87,104,118)(90,93,126) ]", "PSL(2,151)^1" ],
  [   3483840, "[ (3,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,98)(3,54,142)(4,150,45)(5,55,139)(6,17,176)(7,34,158)(8,135,56)(9,161,29)(10,70,119)(11,115,73)(12,86,101)(13,89,97)(14,146,39)(15,153,31)(16,114,69)(18,53,128)(19,129,51)(20,179,190)(21,58,120)(22,137,40)(23,113,63)(24,149,26)(25,80,94)(27,88,84)(28,122,49)(30,125,44)(32,75,92)(33,87,79)(35,187,167)(36,78,85)(37,96,66)(38,162,189)(41,106,52)(42,64,93)(43,181,165)(46,192,151)(47,172,170)(48,60,91)(50,182,157)(57,141,191)(59,174,156)(61,188,140)(62,72,65)(67,177,145)(68,143,178)(71,166,152)(74,168,147)(76,138,175)(77,126,186)(81,185,123)(82,180,127)(83,173,133)(90,155,144)(95,110,184)(99,107,183)(100,159,130)(102,116,171)(103,132,154)(104,121,164)(105,136,148)(108,169,112)(109,163,117)(111,118,160)(124,134,131) ]", "PSL(2,191)^1" ],
  [   3484800, "[ (1,13,6,22,7,23,4,14,9,19,12,20)(2,15,8,16,5,17,10,21,3,24,11,18), (3,6)(4,12)(5,9)(7,11)(8,10), (15,18)(16,24)(17,21)(19,23)(20,22), (2,7,6,5,3)(8,11,9,12,10) ]", "PSL(2,11)^2" ],
  [   3542880, "[ (13,23)(14,33)(15,43)(16,53)(17,63)(18,73)(19,83)(20,93)(21,103)(22,113)(24,34)(25,44)(26,54)(27,64)(28,74)(29,84)(30,94)(31,104)(32,114)(35,45)(36,55)(37,65)(38,75)(39,85)(40,95)(41,105)(42,115)(46,56)(47,66)(48,76)(49,86)(50,96)(51,106)(52,116)(57,67)(58,77)(59,87)(60,97)(61,107)(62,117)(68,78)(69,88)(70,98)(71,108)(72,118)(79,89)(80,99)(81,109)(82,119)(90,100)(91,110)(92,120)(101,111)(102,121)(112,122), (3,122,121,120,119,118,117,116,115,114,113,112,12,111,110,109,108,107,106,105,104,103,102,101,11,100,99,98,97,96,95,94,93,92,91,90,10,89,88,87,86,85,84,83,82,81,80,79,9,78,77,76,75,74,73,72,71,70,69,68,8,67,66,65,64,63,62,61,60,59,58,57,7,56,55,54,53,52,51,50,49,48,47,46,6,45,44,43,42,41,40,39,38,37,36,35,5,34,33,32,31,30,29,28,27,26,25,24,4,23,22,21,20,19,18,17,16,15,14,13), (1,2,8)(3,7,9)(4,10,5)(6,12,11)(13,43,91)(14,96,37)(15,110,23)(16,90,41)(17,57,73)(18,63,67)(19,20,109)(21,32,95)(22,46,79)(24,59,64)(25,120,112)(26,115,116)(27,34,87)(28,58,61)(29,38,80)(30,35,82)(31,69,47)(33,50,65)(36,60,51)(39,121,98)(40,103,114)(42,52,54)(44,92,122)(45,119,94)(48,101,108)(53,100,105)(55,97,106)(56,89,113)(62,78,118)(66,104,88)(68,72,117)(70,85,102)(71,76,111)(74,77,107)(75,99,84)(81,83,93) ]", "PSL(2,121)^1" ],
  [   3594432, "[ (3,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,99)(3,65,133)(4,173,24)(5,116,80)(6,177,18)(7,70,124)(8,112,81)(9,91,101)(10,162,29)(11,77,113)(12,130,59)(13,146,42)(14,46,141)(15,149,37)(16,53,132)(17,62,122)(19,63,119)(20,22,159)(21,192,180)(23,148,30)(25,194,174)(26,72,103)(27,40,134)(28,129,44)(31,83,87)(32,71,98)(33,75,93)(34,89,78)(35,41,125)(36,188,169)(38,108,55)(39,176,178)(43,110,48)(45,54,102)(47,94,60)(49,183,161)(50,175,168)(51,58,92)(52,185,156)(56,61,84)(57,152,184)(64,158,171)(66,145,182)(68,186,139)(69,170,154)(73,157,163)(74,128,191)(76,136,181)(79,135,179)(82,193,118)(85,121,187)(86,190,117)(88,155,150)(90,160,143)(95,126,172)(96,144,153)(97,107,189)(100,127,166)(104,151,138)(105,123,165)(106,140,147)(109,164,120)(111,115,167)(114,137,142) ]", "PSL(2,193)^1" ],
  [   3628800, "[ (9,10), (1,2,3,4,5,6,7,8,9) ]", "A10^1" ],
  [   3753792, "[ (1,58)(2,59)(3,60)(4,61)(5,62)(6,63)(7,64)(8,65)(9,66)(10,67)(11,68)(12,69)(13,70)(14,71)(15,72)(16,73)(17,74)(18,75)(19,76)(20,77)(21,78)(22,79)(23,80)(24,81)(25,82)(26,83)(27,84)(28,85)(29,86)(30,87)(31,88)(32,89)(33,90)(34,91)(35,92)(36,93)(37,94)(38,95)(39,96)(40,97)(41,98)(42,99)(43,100)(44,101)(45,102)(46,103)(47,104)(48,105)(49,106)(50,107)(51,108)(52,109)(53,110)(54,111)(55,112)(56,113)(57,114), (3,4,5,6,7,8)(10,15,14,13,12,11)(16,44,30)(17,50,35,20,47,32)(18,45,36,21,48,33)(19,46,31,22,49,34)(23,51,37)(24,57,42,27,54,39)(25,52,43,28,55,40)(26,53,38,29,56,41)(60,65,64,63,62,61)(67,68,69,70,71,72)(73,87,101)(74,89,104,77,92,107)(75,90,105,78,93,102)(76,91,106,79,88,103)(80,94,108)(81,96,111,84,99,114)(82,97,112,85,100,109)(83,98,113,86,95,110), (1,2,9,13,41,3,16,11,27,57,31,54,52,38,6,37)(4,23,15,55,17,46,35,40)(5,30,12,34,33,26,50,21,32,19,18,53,45,28,22,39)(7,44,14,48,49,56,24,36,47,42,8,51,10,20,25,43)(58,59,67,63,95,97,111,100,90,78,101,70,60,74,73,66)(61,81,110,93,99,83,82,75,80,68,62,88,106,77,94,69)(64,102,91,85,96,104,105,112,107,84,89,113,114,79,108,72)(65,109,86,103,98,76,87,71) ]", "PSL(3,7)^1" ],
  [   3822588, "[ (3,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,198,196,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,101)(3,100,102)(4,116,85)(5,12,188)(6,131,68)(7,138,60)(8,87,110)(9,183,13)(10,38,157)(11,25,169)(14,190,197)(15,107,83)(16,63,126)(17,137,51)(18,84,103)(19,193,189)(20,136,49)(21,57,127)(22,128,55)(23,47,135)(24,69,112)(26,140,39)(27,28,150)(29,146,30)(31,77,97)(32,114,59)(33,177,191)(34,42,129)(35,46,124)(36,108,61)(37,89,79)(40,93,72)(41,106,58)(43,82,80)(44,50,111)(45,164,192)(48,104,53)(52,174,175)(54,81,70)(56,173,172)(62,176,163)(64,195,142)(65,185,151)(66,182,153)(67,155,179)(71,196,134)(73,160,168)(74,180,147)(75,145,181)(76,139,186)(78,156,167)(86,198,117)(88,170,143)(90,133,178)(91,152,158)(92,115,194)(94,166,141)(95,187,119)(96,161,144)(98,154,149)(99,118,184)(105,125,171)(109,162,130)(113,165,123)(120,159,122)(121,148,132) ]", "PSL(2,197)^1" ],
  [   3869736, "[ (3,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,81)(3,96,66)(4,14,147)(5,70,90)(6,97,62)(7,18,140)(8,27,130)(9,58,98)(10,106,49)(11,95,59)(12,117,36)(13,20,132)(15,148,158)(16,102,47)(17,39,109)(19,75,71)(21,76,68)(22,144,155)(23,118,24)(25,99,41)(26,38,101)(28,89,48)(29,105,31)(30,142,149)(32,135,154)(33,50,82)(34,43,88)(35,78,52)(37,51,77)(40,79,46)(42,69,54)(44,139,138)(45,150,126)(53,123,145)(56,152,113)(57,133,131)(60,146,115)(61,124,136)(63,137,121)(64,104,153)(65,156,100)(67,151,103)(72,92,157)(73,134,114)(74,119,128)(80,112,129)(83,122,116)(84,127,110)(85,111,125)(86,141,94)(87,143,91)(93,120,108) ]", "PSL(2,157)^1" ],
  [   3916800, "[ (1,86,81,166,40,125,65,150,22,107,31,116,84,169,55,140)(2,87,53,138,77,162,4,89,18,103,35,120,67,152,50,135)(3,88,25,110,62,147,60,145,58,143,23,108,9,94,32,117)(5,90,61,146,21,106,54,139,78,163,27,112,34,119,12,97)(6,91,8,93,46,131,45,130,10,95,82,167,16,101,39,124)(7,92,63,148,51,136,85,170,42,127,41,126,33,118,71,156)(11,96,72,157,30,115,74,159,70,155,64,149,19,104,17,102)(13,98,75,160,83,168,36,121,38,123,49,134,57,142,80,165)(14,99,68,153,24,109,48,133)(15,100,26,111,56,141,79,164,66,151,20,105,44,129,29,114)(28,113,43,128,69,154,59,144,52,137,73,158,47,132,37,122)(76,161), (4,5)(8,9)(12,13)(14,18)(15,19)(16,21)(17,20)(24,25)(28,29)(30,34)(31,35)(32,37)(33,36)(40,41)(44,45)(46,50)(47,51)(48,53)(49,52)(54,70)(55,71)(56,73)(57,72)(58,74)(59,75)(60,77)(61,76)(62,82)(63,83)(64,85)(65,84)(66,78)(67,79)(68,81)(69,80)(86,101)(87,143)(88,124)(89,118)(90,169)(91,123)(92,117)(93,144)(94,170)(95,167)(96,126)(97,145)(98,116)(103,115)(104,168)(105,146)(106,125)(108,133)(109,160)(110,154)(111,135)(112,158)(114,148)(119,163)(121,130)(122,141)(128,140)(129,166)(131,159)(132,153)(136,150)(138,156)(142,165)(149,157)(152,161) ]", "PSp(4,4)^1" ],
  [   3940200, "[ (3,199,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,200,198,196,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,102)(3,194,10)(4,88,115)(5,56,146)(6,144,57)(7,27,173)(8,160,39)(9,103,95)(11,63,133)(12,78,117)(13,43,151)(14,139,54)(15,130,62)(16,159,32)(17,162,28)(18,128,61)(19,91,97)(20,153,34)(21,50,136)(22,104,81)(23,64,120)(24,59,124)(25,83,99)(26,152,29)(30,48,129)(31,177,197)(33,92,82)(35,66,106)(36,85,86)(37,132,38)(40,90,77)(41,96,70)(42,187,176)(44,196,165)(45,188,172)(46,94,67)(47,49,111)(51,184,170)(52,178,175)(53,161,191)(55,79,73)(58,148,199)(60,198,147)(65,190,150)(68,154,183)(71,141,193)(72,167,166)(74,189,142)(75,156,174)(76,186,143)(80,145,180)(84,140,181)(87,126,192)(89,116,200)(93,155,157)(98,138,169)(100,182,123)(101,195,109)(105,121,179)(107,113,185)(108,163,134)(110,158,137)(112,171,122)(114,164,127)(118,119,168)(125,149,131) ]", "PSL(2,199)^1" ],
  [   4147200, "[ (1,13,9,19,3,16,10,21)(2,17,12,22,6,14,7,20,5,18,11,24,4,15,8,23), (1,9)(2,7,3,8,5,12,4,10,6,11), (13,21)(14,19,15,20,17,24,16,22,18,23), (5,6)(7,9)(8,10)(11,12), (17,18)(19,21)(20,22)(23,24), (13,14,15,16,17)(20,22,23,21,24) ]", "A6^2" ],
  [   4245696, "[ (1,183,136,258,64,38,160,151,190,176,303,300)(2,234,177,245,268,26,51,150,138,24,129,142)(3,312,292,246,200,228,86,152,193,278,304,83)(4,16,56,295,156,174,263,92,269,209,270,277)(5,334,12,272,207,194,65,238,198,265,333,253)(6,206,280,22,159,154,89,148,232,330,267,240)(7,336,87,237,116,239,315,227,55,109,78,199)(8,102,34,338,192,31,37,217,242,344,313,133)(9,119,208,314,255,141,310,18,113,172,282,266)(10,33,121,181,281,147,61,345,66,179,81,241)(11,59,331,124)(13,214,49,140,254,123,257,210,324,178,339,307)(14,73,251,294,27,168,68,297,162,126,221,104)(15,105,259,191,249,48,74,84,215,351,75,279)(17,163,289,262,350,248,39,166,46,82,88,225)(19,349,216,311,261,171,220,305,189,54,145,205)(20,318,41,323,42,233,302,98,60,80,218,155)(21,167,226,90,247,197,196,212,157,131,69,285)(23,36,284,114,231,132)(25,44,319,213,322,309,180,342,58,290,57,71)(28,182,153,201,299,187,343,101,40,185,77,308)(29,340,115,244,195,72,286,47,202,184,96,85)(30,118,271,346,320,223,164,106,326,243,236,287)(32,135,301,250,76,158)(35,173,144,276,211,306,325,62,288,117,222,45)(43,169,230,79,204,252,188,146,229,137,93,107)(50,329,293,63)(52,53,291,70,111,321,112,283,122,97,274,149)(67,100,139,275,260,327,161,95,341,332,347,235)(91,256,298,186,127,99,328,110,273,128,348,143)(94,103,130,108)(120,203,316,337)(125,175,264,219)(134,335,165,224)(352,507,489,414,428,462,481,700,429,415,361,540)(353,555,373,693,448,498,403,625,623,362,435,691)(354,518,461,473,676,661,469,454,532,605,365,598)(355,452,590,471,519,385,632,694,667,687,464,479)(356,502,426,537,526,631,684,412,408,371,654,393)(357,368,379,523,395,524,360,701,591,382,485,380)(358,495,410,400,366,594,463,626,422,567,653,560)(359,556,520,503,664,434,436,639,677,645,583,494)(363,643,407,627,647,663,501,418,499,369,652,535)(364,496,678,601,404,378,666,480,391,551,630,613)(367,534,533,565,681,472,509,513,651,574,573,386)(370,635,572,611,672,578,539,478,614,628,542,401)(372,620,671,466,607,584,531,659,488,484,606,558)(374,505,655,608,668,529,699,389,465,402,588,638)(375,696,417,592,568,514,596,440,685,455,447,538)(376,562,545,662,460,433,510,536,522,609,521,432)(377,612,604,595,575,482,549,438,683,670,427,445)(381,493,477,398,633,688,603,431,409,559,610,487)(383,517,669,486,650,597,686,557,506,474,602,618)(384,476,437,579,563,491,420,441,512,600,554,692)(387,657,570,421,392,587,443,553,636,702,423,622)(388,660,451,673,682,419,424,475,621,450,641,589)(390,439,582,679,467,642)(394,571,640,528,624,690)(396,581,530,593,698,680,665,619,566,425,646,548)(397,458,515)(399,459,453,444,430,552,634,541,470,675,497,616)(405,508,689,615,449,585,504,543,637,511,416,674)(406,644,564,442,629,490,580,550,516,649,547,561)(411,483,586,617,525,577,648,658,527,413,697,569)(446,695,457,468,599,456,576,492,544,546,656,500), (1,157,72,268,63,159,237)(2,341,202,248,329,187,266)(3,42,62,128,164,179,313)(4,336,38,100,73,278,337)(5,282,146,227,251,182,35)(6,349,258,89,348,55,181)(7,221,155,264,214,8,259)(9,223,203,163,286,143,299)(10,64,106,328,347,167,217)(11,40,22,212,277,123,195)(12,74,126,156,307,76,124)(13,69,148,196,20,198,306)(14,249,211,325,270,243,289)(15,91,280,133,60,116,315)(16,138,232,52,114,84,218)(17,108,98,206,252,36,85)(18,318,246,90,267,208,326)(19,61,272,242,324,92,331)(21,295,284,323,65,263,344)(23,130,350,229,172,101,170)(24,178,194,225,107,327,144)(25,129,228,189,222,135,220)(26,231,185,302,173,271,117)(27,186,120,312,28,342,56)(29,319,49,234,165,88,279)(30,200,255,253,314,102,175)(31,346,300,86,254,136,121)(32,82,46,176,122,150,309)(33,68,317,204,219,180,171)(34,192,235,132,233,322,66)(37,41,151,226,93,99,250)(39,238,273,184,162,239,43)(44,152,304,53,51,134,141)(45,245,87,110,261,351,303)(47,115,109,50,183,83,340)(48,292,158,201,333,236,298)(54,338,71,95,112,147,290)(57,58,113,80,199,216,301)(59,154,166,320,265,283,168)(67,288,161,281,345,293,94)(70,207,343,142,145,332,276)(75,81,244,103,96,140,257)(77,274,310,296,125,335,287)(78,210,111,262,224,247,104)(79,339,97,240,297,190,316)(105,294,308,330,256,275,149)(118,119,241,260,193,131,174)(127,160,139,230,291,209,191)(137,311,213,321,197,153,169)(177,269,215,334,205,188,305)(352,699,592,571,618,407,517)(353,372,417,479,533,391,453)(354,694,601,502,408,426,648)(355,585,693,439,672,506,454)(356,691,632,637,395,522,560)(357,496,653,581,565,594,476)(358,640,438,650,550,495,466)(359,673,365,629,620,591,401)(360,428,539,491,448,678,473)(361,573,436,397,597,524,469)(362,444,475,540,551,546,405)(363,545,687,449,558,461,627)(364,371,681,586,701,389,529)(366,579,631,583,612,655,654)(367,414,657,421,607,373,656)(368,470,645,437,566,458,563)(369,516,483,547,638,625,513)(370,582,485,652,537,578,532)(374,525,662,478,623,643,455)(375,690,499,549,636,697,400)(376,602,639,574,472,520,671)(377,666,676,696,465,593,450)(378,651,494,695,642,481,433)(379,413,700,624,492,609,381)(380,416,633,474,390,512,383)(382,564,403,424,543,515,667)(384,501,500,548,488,504,394)(385,675,621,664,634,398,489)(386,660,459,440,393,452,464)(387,599,572,677,447,605,510)(388,462,523,497,498,505,526)(392,580,423,432,477,514,668)(396,644,460,688,614,508,615)(399,649,409,445,484,415,686)(402,456,646,554,588,622,451)(404,570,598,412,527,561,457)(406,467,528,427,562,692,596)(410,630,559,658,619,576,534)(411,493,418,557,584,431,471)(419,530,552,541,443,626,463)(420,490,702,682,555,553,680)(422,611,659,503,568,519,577)(425,665,679,518,604,635,442)(429,482,606,441,608,538,595)(430,670,434,590,683,641,684)(435,663,569,509,685,468,480)(446,698,542,487,617,628,669)(486,511,616,689,610,661,647)(507,536,587,674,556,575,567)(521,613,589,544,531,603,535) ]", "G(2, 3)^1" ],
  [   4330584, "[ (3,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,84)(3,83,85)(4,145,22)(5,69,97)(6,44,121)(7,27,137)(8,98,65)(9,18,144)(10,107,54)(11,55,105)(12,142,17)(13,56,102)(14,76,81)(15,36,120)(16,46,109)(19,37,115)(20,72,79)(21,86,64)(23,164,146)(24,150,159)(25,118,28)(26,156,151)(29,90,52)(30,101,40)(31,141,161)(32,106,33)(34,88,49)(35,95,41)(38,91,42)(39,58,74)(43,68,60)(45,51,75)(47,124,162)(48,132,153)(50,143,140)(53,131,149)(59,122,152)(61,158,114)(62,136,135)(63,113,157)(66,112,155)(67,138,128)(70,160,103)(71,99,163)(73,133,127)(77,130,126)(78,139,116)(80,134,119)(82,147,104)(87,92,154)(89,96,148)(93,117,123)(94,110,129)(100,125,108) ]", "PSL(2,163)^1" ],
  [   4572288, "[ (1,29)(2,45,16,55,23,49,8,43,10,54,6,46)(3,33,25,42,14,30,18,52,9,34,26,47)(4,51,27,31,19,44,17,35,11,56,20,32)(5,50,13,48,28,53)(7,36,21,37,24,41,22,40,12,38,15,39), (2,24,27,28,9,26,8,4,22)(3,15,13,19,17,6,12,25,23)(5,7,21,16,14,20,10,11,18), (30,53,39,36,45,42,56,43,35)(31,44,54,40,33,55,47,38,50)(32,41,46,37,51,48,52,34,49), (30,36,56)(31,40,47)(32,37,52)(33,38,44)(34,41,51)(35,39,42)(43,53,45)(46,48,49)(50,54,55), (1,7,28,27,23,20,11)(2,14,18,15,17,16,10)(3,9,6,26,24,19,12)(4,22,8,5,25,21,13) ]", "PSL(2,8)^2" ],
  [   4657296, "[ (3,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,86)(3,46,126)(4,118,53)(5,109,61)(6,72,97)(7,83,85)(8,37,130)(9,27,139)(10,34,131)(11,149,15)(12,20,143)(13,47,115)(14,129,32)(16,68,91)(17,48,110)(18,108,49)(19,101,55)(21,80,74)(22,50,103)(23,161,157)(24,95,56)(25,45,105)(26,113,36)(28,82,65)(29,152,160)(30,93,52)(31,78,66)(33,145,163)(35,102,38)(39,76,60)(40,51,84)(41,138,162)(42,135,164)(43,158,140)(44,58,73)(54,168,119)(57,125,159)(59,146,136)(62,124,155)(63,167,111)(64,154,123)(67,127,147)(69,122,150)(70,137,134)(71,153,117)(75,100,166)(77,148,116)(79,142,120)(81,104,156)(87,89,165)(88,121,132)(90,144,107)(92,151,98)(94,141,106)(96,133,112)(99,114,128) ]", "PSL(2,167)^1" ],
  [   4680000, "[ (3,6,5,4)(8,11,10,9)(13,16,15,14)(18,21,20,19)(23,26,25,24)(28,31,30,29)(33,35)(34,36)(37,52,47,42)(38,55,48,45)(39,56,49,46)(40,53,50,43)(41,54,51,44)(57,132,107,82)(58,135,108,85)(59,136,109,86)(60,133,110,83)(61,134,111,84)(62,152,122,92)(63,155,123,95)(64,156,124,96)(65,153,125,93)(66,154,126,94)(67,137,127,97)(68,140,128,100)(69,141,129,101)(70,138,130,98)(71,139,131,99)(72,142,112,102)(73,145,113,105)(74,146,114,106)(75,143,115,103)(76,144,116,104)(77,147,117,87)(78,150,118,90)(79,151,119,91)(80,148,120,88)(81,149,121,89), (1,7,32,37,63,56,41,138,50,40,88,44,39,113,3)(2,8,107)(4,10,57,52,141,100,46,114,25,35,62,156,49,90,119)(5,11,82,42,89,144,51,65,81,55,116,19,34,137,125)(6,9,132,47,115,31,36,87,94,45,139,75,54,66,131)(12,58,102,68,153,149,148,98,71,103,118,14,108,22,110)(13,33,112)(15,83,142,151,123,21,109,17,84,67,78,80,105,143,126)(16,133,72,104,93,95,70,128,30,111,27,136,147,73,129)(18,59,127)(20,134,97,146,76,154,124,29,61,77,130,24,85,92,120)(23,135,122)(26,60,152,99,121)(28,86,117)(43,64,106)(48,140,150)(53,91,69)(74,79,155)(96,145,101) ]", "PSp(4,5)^1" ],
  [   4696860, "[ (3,211,209,207,205,203,201,199,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,212,210,208,206,204,202,200,198,196,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,108)(3,107,109)(4,65,150)(5,186,28)(6,22,191)(7,119,93)(8,113,98)(9,42,168)(10,195,14)(11,86,122)(12,35,172)(13,37,169)(15,153,51)(16,125,78)(17,43,159)(18,102,99)(19,29,171)(20,135,64)(21,206,202)(23,55,141)(24,62,133)(25,194,210)(26,160,33)(27,68,124)(30,211,188)(31,127,61)(32,59,128)(34,111,74)(36,82,101)(38,69,112)(39,84,96)(40,129,50)(41,52,126)(44,181,204)(45,187,197)(46,106,67)(47,179,203)(48,174,207)(49,54,116)(53,95,71)(56,190,183)(57,173,199)(58,76,85)(60,80,79)(63,201,165)(66,151,212)(70,77,72)(75,161,193)(81,196,152)(83,154,192)(87,176,166)(88,157,184)(89,185,155)(90,164,175)(91,200,138)(92,148,189)(94,130,205)(97,209,123)(100,162,167)(103,208,118)(104,147,178)(105,182,142)(110,170,149)(114,198,117)(115,134,180)(120,132,177)(121,163,145)(131,140,158)(136,156,137)(139,146,144) ]", "PSL(2,211)^1" ],
  [   4769856, "[ (1,15,14,20,11,25,4,16,8,23,5,26,2,17)(3,18,6,27,9,24,10,22,12,21,7,19,13,28), (3,14,4,5,12,8,13,10,9,6,7,11)(17,28,18,19,26,22,27,24,23,20,21,25), (1,4,2)(5,11,7)(6,10,9)(8,14,13) ]", "PSL(2,13)^2" ],
  [   4769856, "[ (1,15,14,21,7,26,2,28,3,19,13,18,6,24,10,27,9,23,5,22,12,25,4,16,8,20,11,17), (3,14,4,5,12,8,13,10,9,6,7,11)(17,28,18,19,26,22,27,24,23,20,21,25), (15,19,16)(17,25,26)(18,24,27)(20,21,23) ]", "PSL(2,13)^2" ],
  [   4826640, "[ (15,27)(16,39)(17,51)(18,63)(19,75)(20,87)(21,99)(22,111)(23,123)(24,135)(25,147)(26,159)(28,40)(29,52)(30,64)(31,76)(32,88)(33,100)(34,112)(35,124)(36,136)(37,148)(38,160)(41,53)(42,65)(43,77)(44,89)(45,101)(46,113)(47,125)(48,137)(49,149)(50,161)(54,66)(55,78)(56,90)(57,102)(58,114)(59,126)(60,138)(61,150)(62,162)(67,79)(68,91)(69,103)(70,115)(71,127)(72,139)(73,151)(74,163)(80,92)(81,104)(82,116)(83,128)(84,140)(85,152)(86,164)(93,105)(94,117)(95,129)(96,141)(97,153)(98,165)(106,118)(107,130)(108,142)(109,154)(110,166)(119,131)(120,143)(121,155)(122,167)(132,144)(133,156)(134,168)(145,157)(146,169)(158,170), (3,169,167,165,163,161,159,14,156,154,152,150,148,146,13,143,141,139,137,135,133,12,130,128,126,124,122,120,11,117,115,113,111,109,107,10,104,102,100,98,96,94,9,91,89,87,85,83,81,8,78,76,74,72,70,68,7,65,63,61,59,57,55,6,52,50,48,46,44,42,5,39,37,35,33,31,29,4,26,24,22,20,18,16)(15,170,168,166,164,162,160,158,157,155,153,151,149,147,145,144,142,140,138,136,134,132,131,129,127,125,123,121,119,118,116,114,112,110,108,106,105,103,101,99,97,95,93,92,90,88,86,84,82,80,79,77,75,73,71,69,67,66,64,62,60,58,56,54,53,51,49,47,45,43,41,40,38,36,34,32,30,28,27,25,23,21,19,17), (1,2,9)(3,8,10)(4,5,12)(6,13,14)(15,146,38)(16,158,25)(17,68,114)(18,28,153)(19,151,29)(20,130,50)(21,76,103)(22,118,60)(23,115,62)(24,78,98)(26,93,80)(27,169,160)(30,128,41)(31,69,99)(32,157,167)(33,36,131)(34,166,156)(35,104,61)(37,120,42)(39,170,147)(40,97,63)(43,84,72)(44,59,96)(45,108,46)(47,79,74)(48,56,95)(49,85,66)(51,91,58)(52,75,73)(53,64,83)(54,149,152)(55,165,135)(57,155,144)(65,148,143)(67,163,125)(70,162,123)(71,117,168)(77,140,139)(81,150,124)(82,109,164)(86,116,154)(87,107,161)(88,145,122)(89,126,141)(90,129,137)(92,159,105)(94,134,127)(100,136,119)(101,142,113)(102,121,132)(106,138,111)(110,133,112) ]", "PSL(2,169)^1" ],
  [   4826640, "[ (3,170,14,157,13,144,12,131,11,118,10,105,9,92,8,79,7,66,6,53,5,40,4,27)(15,26,158,169,145,156,132,143,119,130,106,117,93,104,80,91,67,78,54,65,41,52,28,39)(16,38,159,25,146,168,133,155,120,142,107,129,94,116,81,103,68,90,55,77,42,64,29,51)(17,50,160,37,147,24,134,167,121,154,108,141,95,128,82,115,69,102,56,89,43,76,30,63)(18,62,161,49,148,36,135,23,122,166,109,153,96,140,83,127,70,114,57,101,44,88,31,75)(19,74,162,61,149,48,136,35,123,22,110,165,97,152,84,139,71,126,58,113,45,100,32,87)(20,86,163,73,150,60,137,47,124,34,111,21,98,164,85,151,72,138,59,125,46,112,33,99), (3,169,167,165,163,161,159,14,156,154,152,150,148,146,13,143,141,139,137,135,133,12,130,128,126,124,122,120,11,117,115,113,111,109,107,10,104,102,100,98,96,94,9,91,89,87,85,83,81,8,78,76,74,72,70,68,7,65,63,61,59,57,55,6,52,50,48,46,44,42,5,39,37,35,33,31,29,4,26,24,22,20,18,16)(15,170,168,166,164,162,160,158,157,155,153,151,149,147,145,144,142,140,138,136,134,132,131,129,127,125,123,121,119,118,116,114,112,110,108,106,105,103,101,99,97,95,93,92,90,88,86,84,82,80,79,77,75,73,71,69,67,66,64,62,60,58,56,54,53,51,49,47,45,43,41,40,38,36,34,32,30,28,27,25,23,21,19,17), (1,2,9)(3,8,10)(4,5,12)(6,13,14)(15,146,38)(16,158,25)(17,68,114)(18,28,153)(19,151,29)(20,130,50)(21,76,103)(22,118,60)(23,115,62)(24,78,98)(26,93,80)(27,169,160)(30,128,41)(31,69,99)(32,157,167)(33,36,131)(34,166,156)(35,104,61)(37,120,42)(39,170,147)(40,97,63)(43,84,72)(44,59,96)(45,108,46)(47,79,74)(48,56,95)(49,85,66)(51,91,58)(52,75,73)(53,64,83)(54,149,152)(55,165,135)(57,155,144)(65,148,143)(67,163,125)(70,162,123)(71,117,168)(77,140,139)(81,150,124)(82,109,164)(86,116,154)(87,107,161)(88,145,122)(89,126,141)(90,129,137)(92,159,105)(94,134,127)(100,136,119)(101,142,113)(102,121,132)(106,138,111)(110,133,112) ]", "PSL(2,169)^1" ],
  [   4826640, "[ (3,170,169,168,167,166,165,164,163,162,161,160,159,158,14,157,156,155,154,153,152,151,150,149,148,147,146,145,13,144,143,142,141,140,139,138,137,136,135,134,133,132,12,131,130,129,128,127,126,125,124,123,122,121,120,119,11,118,117,116,115,114,113,112,111,110,109,108,107,106,10,105,104,103,102,101,100,99,98,97,96,95,94,93,9,92,91,90,89,88,87,86,85,84,83,82,81,80,8,79,78,77,76,75,74,73,72,71,70,69,68,67,7,66,65,64,63,62,61,60,59,58,57,56,55,54,6,53,52,51,50,49,48,47,46,45,44,43,42,41,5,40,39,38,37,36,35,34,33,32,31,30,29,28,4,27,26,25,24,23,22,21,20,19,18,17,16,15), (1,2,9)(3,8,10)(4,5,12)(6,13,14)(15,146,38)(16,158,25)(17,68,114)(18,28,153)(19,151,29)(20,130,50)(21,76,103)(22,118,60)(23,115,62)(24,78,98)(26,93,80)(27,169,160)(30,128,41)(31,69,99)(32,157,167)(33,36,131)(34,166,156)(35,104,61)(37,120,42)(39,170,147)(40,97,63)(43,84,72)(44,59,96)(45,108,46)(47,79,74)(48,56,95)(49,85,66)(51,91,58)(52,75,73)(53,64,83)(54,149,152)(55,165,135)(57,155,144)(65,148,143)(67,163,125)(70,162,123)(71,117,168)(77,140,139)(81,150,124)(82,109,164)(86,116,154)(87,107,161)(88,145,122)(89,126,141)(90,129,137)(92,159,105)(94,134,127)(100,136,119)(101,142,113)(102,121,132)(106,138,111)(110,133,112) ]", "PSL(2,169)^1" ],
  [   5177544, "[ (3,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,89)(3,88,90)(4,62,115)(5,50,126)(6,35,140)(7,16,158)(8,39,134)(9,92,80)(10,150,21)(11,137,33)(12,147,22)(13,45,123)(14,94,73)(15,113,53)(17,96,68)(18,32,131)(19,108,54)(20,162,171)(23,59,99)(24,25,132)(26,60,95)(27,69,85)(28,168,157)(29,51,101)(30,87,64)(31,166,156)(34,71,76)(36,48,97)(37,104,40)(38,143,172)(41,167,145)(42,61,78)(43,66,72)(44,139,170)(46,153,154)(47,146,160)(49,75,57)(52,128,173)(55,133,165)(56,67,58)(63,116,174)(65,163,125)(70,159,124)(74,141,138)(77,127,149)(79,119,155)(81,130,142)(82,161,110)(83,118,152)(84,164,105)(86,169,98)(91,148,114)(93,109,151)(100,117,136)(102,107,144)(103,129,121)(106,112,135)(111,122,120) ]", "PSL(2,173)^1" ],
  [   5184000, "[ (4,5)(6,12,10,13,9,15)(7,14)(8,11), (1,15,6)(2,14,7)(3,11,8)(4,12,9)(5,13,10), (4,5)(9,10), (9,10)(14,15), (11,12,13,14,15) ]", "A5^3" ],
  [   5184000, "[ (4,5)(9,10)(14,15), (1,15,6)(2,14,7)(3,11,8)(4,12,9)(5,13,10), (4,5)(9,10), (9,10)(14,15), (13,14,15), (1,5,4,3,2) ]", "A5^3" ],
  [   5184000, "[ (6,12,9,14,7,15)(8,11)(10,13), (1,15,6)(2,14,7)(3,11,8)(4,12,9)(5,13,10), (4,5)(9,10), (9,10)(14,15), (11,12,13,14,15) ]", "A5^3" ],
  [   5515776, "[ (3,4,6,10,18,33,61,47,58,87,155,263,82,119,85,152,258,384,485,510,321)(5,8,14,26,48,89,159,104,109,192,309,199,320,228,182,297,66,122,213,337,453)(7,12,22,40,75,140,210,146,249,38,71,132,106,100,178,98,175,288,410,463,512)(9,16,30,32,59,110,193,311,34,63,116,133,131,226,94,168,280,236,362,144,212)(11,20,37,69,128,19,35,65,120,209,332,296,60,83,148,251,376,227,353,368,292)(13,24,44,29,54,101,180)(15,28,52,97,173,286,408,493,501,476,467,335,451,509,25,46,86,154,261,246,372)(17,31,57,107,188,305,425,480,127,220,343,459,323,441,215,339,457,483,513,505,96)(21,39,73,136,234,359,422,158,267,393,430,443,43,81,62,114,45,84,150,255,365)(23,42,79,145,247,313,432,240,166,231,355,361,411,456,135,232,357,470,392,468,497)(27,50,93,167,278,401,496,358,471,162,271,80,55,103,183,299,419,495,491,327,448)(36,67,124,216,200,252,378,435,137,235,176,245,371,76,142,243,369,477,481,256,257)(41,77,143,244,273,325,444,445,370,478,396,189,306,379,383,484,382,156,184,301,421)(49,91,163,272,356,469,64,118,206,328,117,204,294,407,284,386,486,266,391,488,461)(51,95,170,283,405,423,502,302,315,434,494,503,304,424,275,254,381,259,385,74,138)(53,99,70,130,224,349,466,482,351,197,317,436,262,319,112,196,316,113,198,248,374)(56,105,186,303,102,181,264,388,354,415,402,346,420,260,364,147,237,363,250,160,269)(68,126,218,342,458,490,499,438,151,179,293,417,187,219,169,281,403,498,318,437,504)(72,134,230,295,233,344,298,324,177,291,414,465,447,506,500,174,108,190,308,427,221)(78,121,211,334,141,241,129,222,345,460,115,201,229,92,165,276,399,277,387,377,330)(88,157,265,390,397,492,90,161,270,395,473,426,153,194,312,310,429,225,341,282,404)(111,195,314,268,394,489,416,333,139,238,123,214,338,455,479,164,274,398,242,307,373)(125,217,340,289,412,171,185,149,253,380,202,322,203,290,413,389,474,433,442,400,440)(172,285,406,428,472,375,223,347,462,300,348,464,511,454,207,329,287,409,336,452,418)(191,208,331,450,508,350,279,360,205,326,446,239,366,475,431,487,439,367,352,449,507), (1,2,3,5,9,17,32,60,112,197,318,438,498,198,319,439,505,16,31,58,109)(4,7,13,25,47,88,158,268,329,432,380,395,491,451,291,415,218,138,237,364,474)(6,11,21,39,74,139,239,343,224,350,75,141,242,241,368,476,332,220,344,459,93)(8,15,29,55,104,185,302,423,105,187,304,345,52,98,176,290,357,285,407,150,256)(10,19,36,68,127,132,228,305,426,73,137,236,258,50,94,169,282,331,89,160,170)(12,23,43,82,147,175,289,230,354,417,462,342,424,276,400,495,502,122,81,42,80)(14,27,51,96,172,211,335,403,374,427,475,452,57,108,191,154,262,387,450,485,317)(18,34,64,119,208,190,306,271,366,308,406,238,365,311,178,292,416,441,457,40,76)(20,38,72,135,233,274,33,62,115,202,323,442,301,422,501,143,245,328,449,429,503)(22,41,78,35,66,123,215,48,90,162,193,63,117,205,327,339,359,118,207,330,376)(24,45,85,153,260,386,371,480,455,103,184,247,373,481,477,46,87,156,264,389,487)(26,49,92,166,277,167,279,402,216,209,333,270,396,408,255,383,412,116,203,324,443)(28,53,100,179,294,418,500,303,363,281,173,99,177,280,148,252,379,470,378,483,469)(30,56,106,61,113,199,128,221,196,261,65,121,212,336,251,377,482,437,97,174,287)(37,70,131,227,206,186,136,234,360,159,71,133,229,226,352,286,326,447,488,506,471)(44,83,149,254,382,453,372,265,157,266,392,484,512,272,168,161,195,315,435,297,393)(54,102,182,298,84,151,257,353,181,296,232,358,472,398,494,414,213,249,375,253,165)(59,111,114,200,321,338,456,399,362,458,444,293,188,134,231,356,192,310,430,486,513)(67,125,79,146,250,355,419,142,204,325,445,489,492,288,411,283,401,497,217,341,101)(69,129,223,348,465,508,434,410,496,507,163,273,397,493,243,370,479,120,210,299,420)(77,144,246,337,454,510,309,428,155,86,107,189,307,145,248,124,130,225,351,468,391)(91,164,275,369,322,390,388)(95,171,284,340,384,478,235,361,473,244,320,440,314,433,269,347,463,126,219,183,300)(110,194,313,421,312,431,425,460,349,467,504,405,385,334,409,394,490,448,466,509,214)(140,240,367,381,446,404,499,267,201,152,259,180,295,413,464,278,263,222,346,461,511) ]", "PSU(3,8)^1" ],
  [   5544672, "[ (3,223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,224,222,220,218,216,214,212,210,208,206,204,202,200,198,196,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,114)(3,156,72)(4,198,29)(5,67,159)(6,210,15)(7,174,50)(8,125,98)(9,41,181)(10,56,165)(11,75,145)(12,108,111)(13,134,84)(14,38,179)(16,55,160)(17,48,166)(18,222,213)(19,121,91)(20,99,112)(21,149,61)(22,73,136)(23,182,26)(24,36,171)(25,65,141)(27,176,28)(30,224,199)(31,118,82)(32,148,51)(33,70,128)(34,131,66)(35,58,138)(37,89,105)(39,86,106)(40,95,96)(42,101,88)(43,124,64)(44,85,102)(45,71,115)(46,205,202)(47,187,219)(49,190,214)(52,201,200)(53,59,119)(54,221,178)(57,192,204)(60,78,93)(62,180,211)(63,172,218)(68,173,212)(69,161,223)(74,81,76)(79,207,167)(80,196,177)(83,153,217)(87,163,203)(90,170,193)(92,155,206)(94,215,144)(97,194,162)(100,158,195)(103,220,130)(104,185,164)(107,209,137)(109,169,175)(110,197,146)(113,157,183)(116,129,208)(117,120,216)(122,142,189)(123,139,191)(126,143,184)(127,186,140)(132,133,188)(135,150,168)(147,154,152) ]", "PSL(2,223)^1" ],
  [   5630688, "[ (10,14,12)(11,15,13)(16,44,30)(17,49,33)(18,50,34)(19,45,35)(20,46,36)(21,47,31)(22,48,32)(23,51,37)(24,56,40)(25,57,41)(26,52,42)(27,53,43)(28,54,38)(29,55,39)(67,69,71)(68,70,72)(73,87,101)(74,90,106)(75,91,107)(76,92,102)(77,93,103)(78,88,104)(79,89,105)(80,94,108)(81,97,113)(82,98,114)(83,99,109)(84,100,110)(85,95,111)(86,96,112), (3,4,5,6,7,8)(10,15,14,13,12,11)(16,44,30)(17,50,35,20,47,32)(18,45,36,21,48,33)(19,46,31,22,49,34)(23,51,37)(24,57,42,27,54,39)(25,52,43,28,55,40)(26,53,38,29,56,41)(60,65,64,63,62,61)(67,68,69,70,71,72)(73,87,101)(74,89,104,77,92,107)(75,90,105,78,93,102)(76,91,106,79,88,103)(80,94,108)(81,96,111,84,99,114)(82,97,112,85,100,109)(83,98,113,86,95,110), (1,2,9,13,41,3,16,11,27,57,31,54,52,38,6,37)(4,23,15,55,17,46,35,40)(5,30,12,34,33,26,50,21,32,19,18,53,45,28,22,39)(7,44,14,48,49,56,24,36,47,42,8,51,10,20,25,43)(58,59,67,63,95,97,111,100,90,78,101,70,60,74,73,66)(61,81,110,93,99,83,82,75,80,68,62,88,106,77,94,69)(64,102,91,85,96,104,105,112,107,84,89,113,114,79,108,72)(65,109,86,103,98,76,87,71) ]", "PSL(3,7)^1" ],
  [   5663616, "[ (3,4,6,10,18,33,60,103,74,123,138,211,237,310,339,231,66,19,35,63,107,172,250,301,336,338,46,80,134,207,283,90,152,188,192,168,246,320,140,213,128,202,180,32,58,100,163,241)(5,8,14,25,47,77,129,203,270,104,170,248,321,325,96,21,39,71,118,160,158,236,309,317,343,342,220,280,251,286,124,197,271,326,278,330,335,340,171,249,240,131,94,157,235,196,204,279)(7,12,22,41,17,31,57,98,164,242,314,147,215,263,322,92,155,177,256,324,332,36,65,109,176,101,11,20,37,67,55,95,159,238,311,26,49,85,143,34,61,105,84,141,161,68,112,130)(9,16,29,54,93,156,233,306,214,288,217,290,153,230,302,277,329,344,341,106,86,145,219,173,252,282,135,208,284,99,144,40,72,120,191,108,174,254,292,334,303,337,24,45,78,117,187,265)(13,23,43,75,125,198,273,244,316,327,165,210,272,224,296,195,269,305,28,52,91,154,232,264,59,102,167,245,318,247,307,126,200,178,258,234,294,308,185,229,295,121,64,82,137,76,127,201)(15,27,51,89,150,227,298,243,193,122,194,175,212,287,199,275,328,239,183,261,300,323,331,44,30,56,97,162,50,87,146,221,293,70,116,151,228,268,253,88,148,223,111,179,260,169,53,62)(38,69,114,184,262,48,83,139,149,225,297,313,315,79,132,205,142,216,289,333,257,319,267,312,276,166,42,73,119,190,222,81,136,209,285,115,186,133,206,281,259,189,266,226,255,304,181,110)(113,182,218,291,299,274), (1,2,3,5,9,17,32,59,14,26,50,88,149,226,293,179,31,47,82,138,49,86,129,33)(4,7,13,24,46,81,109,177,257,80,135,52,61,18,34,62,106,171,154,127,39,16,30,12)(6,11,21,40,54,94,158,237,287,97,163,107,173,253,184,263,219,273,221,294,269,320,324,246)(8,15,28,53,92,123,196,241,207,233,307,170,182,209,286,281,328,159,239,313,258,262,256,105)(10,19,36,66,75,126,148,224,187,20,38,70,117,188,103,169,194,202,278,321,344,254,249,322)(22,42,74,124,197,272,327,334,311,85,144,218,292,41,57,99,165,244,317,343,100,166,143,217)(23,44,77,130,65,110,178,259,298,164,243,315,342,289,291,235,308,302,336,60,104,58,101,56)(25,48,84,142,114,185,35,64,72,121,193,150,191,261,275,167,174,102,168,247,69,115,112,181)(27,45,79,133,91,89,151,156,234,248,245,319,67,111,180,73,122,195,270,325,252,176,236,208)(29,55,96,161,240,314,341,284,306,339,330,242,260,137,205,280,186,264,162,95,160,213,230,303)(37,68,113,183,134,83,140,214,250,323,216,190,267,285,51,90,153,231,304,338,227,299,141,215)(43,76,128,136,210,203,145,220,152,229,301,266,116,63,108,175,255,118,189,71,119,98,146,93)(78,131,204,125,199,276,211,87,147,222,295,157,200,277,228,300,201,132,206,282,331,337,326,155)(120,192,268,225,283,318,172,251,265,279,316,333,310,232,305,329,223,238,312,340,297,335,139,212)(198,274,296)(288,332,290) ]", "PSU(3,7)^1" ],
  [   5735160, "[ (3,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,92)(3,91,93)(4,162,21)(5,132,50)(6,54,127)(7,104,76)(8,147,32)(9,18,160)(10,82,95)(11,47,129)(12,70,105)(13,17,157)(14,128,45)(15,41,131)(16,26,145)(19,135,33)(20,94,73)(22,180,163)(23,43,121)(24,166,175)(25,46,116)(27,167,171)(28,71,88)(29,83,75)(30,85,72)(31,36,120)(34,86,67)(35,65,87)(37,176,152)(38,107,42)(39,158,168)(40,78,69)(44,59,84)(48,81,58)(49,165,151)(51,74,62)(52,179,134)(53,143,169)(55,137,173)(56,170,139)(57,130,178)(60,66,61)(63,141,161)(64,148,153)(68,138,159)(77,146,142)(79,114,172)(80,177,108)(89,102,174)(90,164,111)(96,113,156)(97,119,149)(98,150,117)(99,154,112)(100,125,140)(101,155,109)(103,136,126)(106,144,115)(110,133,122)(118,124,123) ]", "PSL(2,179)^1" ],
  [   5848428, "[ (3,227,225,223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,228,226,224,222,220,218,216,214,212,210,208,206,204,202,200,198,196,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,116)(3,115,117)(4,70,161)(5,105,125)(6,24,205)(7,17,211)(8,42,185)(9,44,182)(10,146,79)(11,58,166)(12,26,197)(13,189,33)(14,69,152)(15,88,132)(16,142,77)(18,158,59)(19,154,62)(20,123,92)(21,215,225)(22,45,168)(23,110,102)(25,54,156)(27,208,226)(28,167,40)(29,61,145)(30,56,149)(31,68,136)(32,82,121)(34,148,53)(35,206,220)(36,95,104)(37,67,131)(38,124,73)(39,99,97)(41,119,75)(43,219,199)(46,91,98)(47,190,224)(48,106,81)(49,114,72)(50,188,223)(51,55,129)(52,94,89)(57,85,93)(60,112,63)(64,187,210)(65,204,192)(66,174,221)(71,162,228)(74,214,173)(76,178,207)(78,213,170)(80,163,218)(83,176,202)(84,198,179)(86,222,153)(87,171,203)(90,216,155)(96,164,201)(100,144,217)(101,165,195)(103,177,181)(107,127,227)(108,194,159)(109,212,140)(111,150,200)(113,191,157)(118,183,160)(120,172,169)(122,209,130)(126,184,151)(128,137,196)(133,193,135)(134,141,186)(138,180,143)(139,147,175) ]", "PSL(2,227)^1" ],
  [   5859000, "[ (7,11,31)(8,16,55)(9,21,79)(10,26,103)(12,36,32)(13,40,56)(14,45,80)(15,50,104)(17,60,33)(18,65,57)(19,69,81)(20,74,105)(22,84,34)(23,89,58)(24,94,82)(25,98,106)(27,108,35)(28,113,59)(29,118,83)(30,123,107)(37,41,61)(38,46,85)(39,51,109)(42,66,62)(43,70,86)(44,75,110)(47,90,63)(48,95,87)(49,99,111)(52,114,64)(53,119,88)(54,124,112)(67,71,91)(68,76,115)(72,96,92)(73,100,116)(77,120,93)(78,125,117)(97,101,121)(102,126,122), (3,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,6,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,5,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,4,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7), (1,2,5)(3,4,6)(7,94,38)(8,118,13)(9,57,73)(10,103,26)(11,82,46)(12,52,75)(14,91,35)(15,125,120)(16,83,40)(17,68,54)(18,100,21)(19,97,23)(20,34,86)(22,43,74)(24,85,31)(25,62,53)(27,45,67)(28,63,49)(29,56,55)(30,123,107)(32,64,44)(33,115,112)(36,114,110)(37,41,61)(39,126,95)(42,119,98)(47,99,113)(48,109,102)(50,117,93)(51,122,87)(58,81,121)(59,90,111)(60,76,124)(65,116,79)(66,88,106)(69,101,89)(70,105,84)(71,108,80)(72,92,96)(77,104,78) ]", "PSL(2,125)^1" ],
  [   5929560, "[ (3,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,93)(3,92,94)(4,37,148)(5,117,67)(6,161,22)(7,98,84)(8,155,26)(9,133,47)(10,17,162)(11,136,42)(12,150,27)(13,58,118)(14,44,131)(15,64,110)(16,33,140)(18,90,81)(19,61,109)(20,45,124)(21,82,86)(23,29,137)(24,169,176)(25,180,164)(28,54,107)(30,85,74)(31,178,160)(32,43,114)(34,99,56)(35,89,65)(36,174,159)(38,149,182)(39,59,91)(40,83,66)(41,75,73)(46,153,170)(48,70,71)(49,157,163)(50,175,144)(51,60,78)(52,57,80)(53,177,139)(55,142,172)(62,141,166)(68,128,173)(69,181,119)(72,143,154)(76,122,171)(77,125,167)(79,132,158)(87,152,130)(88,179,102)(95,127,147)(96,168,105)(97,151,121)(100,104,165)(101,156,112)(103,146,120)(106,129,134)(108,126,135)(111,145,113)(115,116,138) ]", "PSL(2,181)^1" ],
  [   6004380, "[ (3,229,227,225,223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,230,228,226,224,222,220,218,216,214,212,210,208,206,204,202,200,198,196,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,117)(3,96,138)(4,10,223)(5,125,107)(6,209,22)(7,190,40)(8,43,186)(9,166,62)(11,224,230)(12,193,32)(13,84,140)(14,66,157)(15,164,58)(16,169,52)(17,114,106)(18,71,148)(19,173,45)(20,101,116)(21,72,144)(23,55,159)(24,137,76)(25,228,212)(26,92,119)(27,35,175)(28,104,105)(29,139,69)(30,64,143)(31,126,80)(33,123,81)(34,82,121)(36,56,145)(37,49,151)(38,63,136)(39,67,131)(41,222,202)(42,46,149)(44,227,194)(47,102,88)(48,191,226)(50,134,53)(51,112,74)(54,73,110)(57,93,87)(59,199,207)(60,78,99)(61,215,189)(65,218,182)(68,225,172)(70,219,176)(75,179,211)(77,168,220)(83,185,197)(85,188,192)(86,163,216)(89,178,198)(90,162,213)(91,170,204)(94,150,221)(95,205,165)(97,210,158)(98,171,196)(100,184,181)(103,167,195)(108,203,154)(109,229,127)(111,201,153)(113,152,200)(115,142,208)(118,133,214)(120,217,128)(122,183,160)(124,161,180)(129,130,206)(132,187,146)(135,156,174)(141,177,147) ]", "PSL(2,229)^1" ],
  [   6065280, "[ (14,23,32)(15,24,33)(16,25,34)(17,26,35)(18,27,36)(19,28,37)(20,29,38)(21,30,39)(22,31,40)(45,72,63)(46,74,64)(47,73,65)(48,78,66)(49,80,67)(50,79,68)(51,75,69)(52,77,70)(53,76,71), (1,2,5,14)(3,8,23,16)(4,11,32,15)(6,17,7,20)(9,26,25,22)(10,29,34,18)(12,35,33,21)(13,38,24,19)(27,28,31,40)(30,37,36,39)(41,42,45,54)(43,48,63,56)(44,51,72,55)(46,57,47,60)(49,66,65,62)(50,69,74,58)(52,75,73,61)(53,78,64,59)(67,68,71,80)(70,77,76,79) ]", "PSL(4,3)^1" ],
  [   6324552, "[ (3,233,231,229,227,225,223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,234,232,230,228,226,224,222,220,218,216,214,212,210,208,206,204,202,200,198,196,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,119)(3,47,191)(4,207,30)(5,114,122)(6,217,18)(7,194,40)(8,154,79)(9,169,63)(10,203,28)(11,143,87)(12,73,156)(13,125,103)(14,104,123)(15,53,173)(16,166,59)(17,160,64)(19,101,121)(20,138,83)(21,232,220)(22,180,39)(23,161,57)(24,29,188)(25,127,89)(26,51,164)(27,168,46)(31,234,208)(32,38,171)(33,80,128)(34,55,152)(35,228,210)(36,76,129)(37,142,62)(41,108,92)(42,60,139)(43,93,105)(44,231,198)(45,90,106)(48,75,118)(49,85,107)(50,209,214)(52,91,98)(54,126,61)(56,97,88)(58,216,199)(65,185,223)(66,94,81)(67,200,206)(68,102,71)(69,229,175)(70,211,192)(72,222,179)(74,187,212)(77,215,181)(78,221,174)(82,165,226)(84,230,159)(86,183,204)(95,227,151)(96,201,176)(99,178,196)(100,218,155)(109,162,202)(110,158,205)(111,213,149)(112,184,177)(113,225,135)(115,134,224)(116,124,233)(117,137,219)(120,163,190)(130,197,146)(131,153,189)(132,148,193)(133,145,195)(136,170,167)(140,147,186)(141,182,150)(144,172,157) ]", "PSL(2,233)^1" ],
  [   6531840, "[ (4,278,7,222)(5,268,11,239)(6,52,15,111)(8,110,22,233)(9,29,26,91)(10,78,18,25)(12,82,35,203)(13,64,39,172)(14,171,43,252)(16,193,50,134)(17,45,31,133)(19,88,23,190)(20,254,61,140)(21,245,65,223)(24,109,74,232)(27,194,83,76)(28,198,87,177)(30,127,94,92)(32,184,97,249)(33,201,41,148)(34,51,101,150)(36,225,108,270)(37,188,112,241)(40,265,119,210)(42,263,122,234)(44,48,129,137)(46,247,126,156)(47,240,138,248)(49,269,143,173)(53,260,80,191)(54,75,154,192)(55,71,93,182)(56,121,157,217)(57,202,99,147)(58,139,69,231)(59,211,161,89)(60,106,165,229)(62,226,168,103)(63,272,170,227)(66,174,77,208)(68,206,175,236)(70,259,98,149)(72,216,185,276)(73,212,187,275)(79,164,158,262)(81,183,155,197)(84,113,131,146)(85,235,130,200)(86,145)(90,255,213,186)(95,128,96,120)(100,220,180,253)(102,250,104,136)(105,214,228,166)(107,125,117,246)(114,142,237,251)(115,195,204,176)(116,144,132,256)(118,159,209,266)(123,152,242,178)(124,264,243,261)(135,205,218,215)(141,196,163,199)(151,273,179,189)(153,219,207,167)(160,277,238,221)(162,271,257,169)(181,267,258,274)(224,230,244,279), (4,5,7,11)(6,9,15,26)(8,13,22,39)(10,17,18,31)(12,20,35,61)(14,24,43,74)(16,28,50,87)(19,33,23,41)(21,37,65,112)(25,45,78,133)(27,48,83,137)(29,52,91,111)(30,54,94,154)(32,56,97,157)(34,59,101,161)(36,63,108,170)(38,67)(40,70,119,98)(42,72,122,185)(44,76,129,194)(46,80,126,53)(47,81,138,155)(49,85,143,130)(51,89,150,211)(55,95,93,96)(57,58,99,69)(60,103,165,226)(62,106,168,229)(64,110,172,233)(66,84,77,131)(68,116,175,132)(71,120,182,128)(73,124,187,243)(75,127,192,92)(79,135,158,218)(82,140,203,254)(86,145)(88,148,190,201)(90,152,213,178)(100,118,180,209)(102,163,104,141)(105,167,228,219)(107,151,117,179)(109,171,232,252)(113,174,146,208)(114,176,237,195)(115,142,204,251)(121,184,217,249)(123,186,242,255)(125,189,246,273)(134,198,193,177)(136,199,250,196)(139,202,231,147)(144,206,256,236)(149,210,259,265)(153,214,207,166)(156,191,247,260)(159,220,266,253)(160,181,238,258)(162,224,257,244)(164,215,262,205)(169,230,271,279)(173,235,269,200)(183,240,197,248)(188,245,241,223)(212,261,275,264)(216,263,276,234)(221,267,277,274)(222,268,278,239)(225,227,270,272), (1,2,3,4,6,10,18,32,57)(5,8,14,25,46,81,139,180,238)(7,12,21,38,68,117,99,159,221)(9,16,29,53,93,28,51,90,153)(11,19,34,60,104,87,147,209,258)(13,23,42,73,125,190,91,120,183)(15,27,49,86,146,145,143,204,97)(17,30,54)(20,36,64,111,135,37,66,114,177)(22,40,71,121,148,203,243,214,41)(24,44,77,132,196,249,233,89,151)(26,47,82,141,94,155,216,250,240)(31,55,96)(33,58,100,160,222,232,262,165,227)(35,62,107)(39,69,118,181,239,210,260,78,134)(43,75,128,194,131,176,182,127,193)(45,79,136)(48,84,142,205,255,271,279,152,163)(50,88,149)(52,92,129,174,206,257,273,263,108)(56,98,158,219,265,154,215,246,259)(59,102,164,225,70,76,130,195,248)(61,105,110,140,186,112,173,236,270)(63,109,95,156,217,171,157,72,123)(65,113,175)(67,115,83)(74,126,191,198,185,241,85,144,207)(80,137,200,251,247,218,264,275,199)(101,162,124,188,235,256,187,244,272)(103,166,122)(106,169,167,202,253,274,268,150,212)(116,178,189,172,234,179,161,223,269)(119,133,197)(138,201,252)(170,231,220,267,278,276,230,229,261)(184,211,228,254,224,245,208,237,192)(213,226,242)(266,277,280) ]", "PSU(4,3)^1" ],
  [   6531840, "[ (5,11)(8,19)(9,26)(13,41)(14,34)(16,47)(17,31)(20,61)(22,23)(24,161)(25,60)(28,155)(29,82)(30,55)(33,39)(36,105)(37,112)(40,42)(43,101)(44,223)(45,226)(46,104)(48,137)(50,138)(51,216)(52,254)(53,141)(54,96)(56,157)(58,69)(59,74)(63,219)(64,110)(66,173)(70,185)(71,73)(72,98)(75,162)(76,241)(77,269)(78,165)(79,242)(80,163)(81,87)(84,200)(85,130)(88,201)(89,234)(90,250)(91,203)(92,224)(93,94)(95,154)(100,118)(102,126)(103,133)(106,229)(108,228)(109,265)(111,140)(114,236)(116,132)(119,122)(120,243)(121,125)(123,158)(124,128)(127,244)(129,245)(131,235)(134,227)(135,186)(136,213)(139,147)(142,251)(144,195)(148,190)(149,252)(150,276)(151,179)(152,199)(153,240)(156,215)(160,181)(164,191)(166,197)(167,170)(169,261)(171,259)(172,233)(174,208)(176,256)(177,270)(178,196)(180,209)(182,187)(183,214)(184,273)(188,194)(189,249)(192,257)(193,272)(198,225)(202,231)(205,247)(206,237)(207,248)(210,232)(211,263)(212,230)(217,246)(218,255)(220,253)(222,239)(238,258)(260,262)(264,271)(267,274)(268,278)(275,279), (4,5,7,11)(6,9,15,26)(8,13,22,39)(10,17,18,31)(12,20,35,61)(14,24,43,74)(16,28,50,87)(19,33,23,41)(21,37,65,112)(25,45,78,133)(27,48,83,137)(29,52,91,111)(30,54,94,154)(32,56,97,157)(34,59,101,161)(36,63,108,170)(38,67)(40,70,119,98)(42,72,122,185)(44,76,129,194)(46,80,126,53)(47,81,138,155)(49,85,143,130)(51,89,150,211)(55,95,93,96)(57,58,99,69)(60,103,165,226)(62,106,168,229)(64,110,172,233)(66,84,77,131)(68,116,175,132)(71,120,182,128)(73,124,187,243)(75,127,192,92)(79,135,158,218)(82,140,203,254)(86,145)(88,148,190,201)(90,152,213,178)(100,118,180,209)(102,163,104,141)(105,167,228,219)(107,151,117,179)(109,171,232,252)(113,174,146,208)(114,176,237,195)(115,142,204,251)(121,184,217,249)(123,186,242,255)(125,189,246,273)(134,198,193,177)(136,199,250,196)(139,202,231,147)(144,206,256,236)(149,210,259,265)(153,214,207,166)(156,191,247,260)(159,220,266,253)(160,181,238,258)(162,224,257,244)(164,215,262,205)(169,230,271,279)(173,235,269,200)(183,240,197,248)(188,245,241,223)(212,261,275,264)(216,263,276,234)(221,267,277,274)(222,268,278,239)(225,227,270,272), (1,2,3,4,6,10,18,32,57)(5,8,14,25,46,81,139,180,238)(7,12,21,38,68,117,99,159,221)(9,16,29,53,93,28,51,90,153)(11,19,34,60,104,87,147,209,258)(13,23,42,73,125,190,91,120,183)(15,27,49,86,146,145,143,204,97)(17,30,54)(20,36,64,111,135,37,66,114,177)(22,40,71,121,148,203,243,214,41)(24,44,77,132,196,249,233,89,151)(26,47,82,141,94,155,216,250,240)(31,55,96)(33,58,100,160,222,232,262,165,227)(35,62,107)(39,69,118,181,239,210,260,78,134)(43,75,128,194,131,176,182,127,193)(45,79,136)(48,84,142,205,255,271,279,152,163)(50,88,149)(52,92,129,174,206,257,273,263,108)(56,98,158,219,265,154,215,246,259)(59,102,164,225,70,76,130,195,248)(61,105,110,140,186,112,173,236,270)(63,109,95,156,217,171,157,72,123)(65,113,175)(67,115,83)(74,126,191,198,185,241,85,144,207)(80,137,200,251,247,218,264,275,199)(101,162,124,188,235,256,187,244,272)(103,166,122)(106,169,167,202,253,274,268,150,212)(116,178,189,172,234,179,161,223,269)(119,133,197)(138,201,252)(170,231,220,267,278,276,230,229,261)(184,211,228,254,224,245,208,237,192)(213,226,242)(266,277,280) ]", "PSU(4,3)^1" ],
  [   6531840, "[ (6,12)(8,19)(9,20)(10,62)(13,33)(14,42)(15,35)(16,105)(17,106)(18,168)(21,27)(22,23)(24,72)(25,103)(26,61)(28,167)(29,254)(30,261)(31,229)(32,107)(34,40)(36,47)(37,48)(38,67)(39,41)(43,122)(44,241)(45,165)(46,242)(49,113)(50,228)(51,265)(52,82)(53,186)(54,275)(55,169)(56,151)(57,99)(58,69)(59,70)(60,133)(63,81)(64,201)(65,83)(66,200)(68,115)(71,162)(73,75)(74,185)(76,223)(77,235)(78,226)(79,104)(80,255)(84,173)(85,174)(86,145)(87,219)(88,110)(89,149)(90,260)(91,140)(92,243)(93,271)(94,264)(95,230)(96,279)(97,117)(98,161)(101,119)(102,158)(108,138)(109,216)(111,203)(112,137)(114,256)(116,142)(120,224)(121,273)(123,126)(124,127)(125,184)(128,244)(129,188)(130,208)(131,269)(132,251)(134,166)(135,141)(136,164)(139,231)(143,146)(144,237)(147,202)(148,172)(150,210)(152,156)(153,198)(154,212)(155,170)(157,179)(160,238)(163,218)(171,263)(175,204)(176,236)(177,207)(178,247)(181,258)(182,257)(183,272)(187,192)(189,217)(190,233)(191,213)(193,214)(194,245)(195,206)(196,205)(197,227)(199,215)(211,259)(221,277)(225,240)(232,276)(234,252)(246,249)(248,270)(250,262)(267,274), (4,5,7,11)(6,9,15,26)(8,13,22,39)(10,17,18,31)(12,20,35,61)(14,24,43,74)(16,28,50,87)(19,33,23,41)(21,37,65,112)(25,45,78,133)(27,48,83,137)(29,52,91,111)(30,54,94,154)(32,56,97,157)(34,59,101,161)(36,63,108,170)(38,67)(40,70,119,98)(42,72,122,185)(44,76,129,194)(46,80,126,53)(47,81,138,155)(49,85,143,130)(51,89,150,211)(55,95,93,96)(57,58,99,69)(60,103,165,226)(62,106,168,229)(64,110,172,233)(66,84,77,131)(68,116,175,132)(71,120,182,128)(73,124,187,243)(75,127,192,92)(79,135,158,218)(82,140,203,254)(86,145)(88,148,190,201)(90,152,213,178)(100,118,180,209)(102,163,104,141)(105,167,228,219)(107,151,117,179)(109,171,232,252)(113,174,146,208)(114,176,237,195)(115,142,204,251)(121,184,217,249)(123,186,242,255)(125,189,246,273)(134,198,193,177)(136,199,250,196)(139,202,231,147)(144,206,256,236)(149,210,259,265)(153,214,207,166)(156,191,247,260)(159,220,266,253)(160,181,238,258)(162,224,257,244)(164,215,262,205)(169,230,271,279)(173,235,269,200)(183,240,197,248)(188,245,241,223)(212,261,275,264)(216,263,276,234)(221,267,277,274)(222,268,278,239)(225,227,270,272), (1,2,3,4,6,10,18,32,57)(5,8,14,25,46,81,139,180,238)(7,12,21,38,68,117,99,159,221)(9,16,29,53,93,28,51,90,153)(11,19,34,60,104,87,147,209,258)(13,23,42,73,125,190,91,120,183)(15,27,49,86,146,145,143,204,97)(17,30,54)(20,36,64,111,135,37,66,114,177)(22,40,71,121,148,203,243,214,41)(24,44,77,132,196,249,233,89,151)(26,47,82,141,94,155,216,250,240)(31,55,96)(33,58,100,160,222,232,262,165,227)(35,62,107)(39,69,118,181,239,210,260,78,134)(43,75,128,194,131,176,182,127,193)(45,79,136)(48,84,142,205,255,271,279,152,163)(50,88,149)(52,92,129,174,206,257,273,263,108)(56,98,158,219,265,154,215,246,259)(59,102,164,225,70,76,130,195,248)(61,105,110,140,186,112,173,236,270)(63,109,95,156,217,171,157,72,123)(65,113,175)(67,115,83)(74,126,191,198,185,241,85,144,207)(80,137,200,251,247,218,264,275,199)(101,162,124,188,235,256,187,244,272)(103,166,122)(106,169,167,202,253,274,268,150,212)(116,178,189,172,234,179,161,223,269)(119,133,197)(138,201,252)(170,231,220,267,278,276,230,229,261)(184,211,228,254,224,245,208,237,192)(213,226,242)(266,277,280) ]", "PSU(4,3)^1" ],
  [   6825840, "[ (3,239,237,235,233,231,229,227,225,223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,240,238,236,234,232,230,228,226,224,222,220,218,216,214,212,210,208,206,204,202,200,198,196,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,122)(3,56,188)(4,162,81)(5,18,224)(6,147,94)(7,154,86)(8,177,62)(9,106,132)(10,137,100)(11,50,186)(12,125,110)(13,193,41)(14,76,157)(15,49,183)(16,167,64)(17,19,211)(20,226,239)(21,57,169)(22,109,116)(23,123,101)(24,103,120)(25,142,80)(26,191,30)(27,181,39)(28,60,159)(29,70,148)(31,176,40)(32,43,172)(33,225,227)(34,139,74)(35,47,165)(36,173,38)(37,84,126)(42,92,113)(44,108,95)(45,91,111)(46,55,146)(48,130,69)(51,231,203)(52,117,78)(53,218,214)(54,89,104)(58,194,233)(59,115,73)(61,195,229)(63,217,205)(65,83,99)(66,88,93)(67,236,182)(68,213,204)(71,208,206)(72,201,212)(75,187,223)(77,228,180)(79,197,209)(82,240,163)(85,184,216)(87,168,230)(90,237,158)(96,174,215)(97,238,150)(98,189,198)(102,219,164)(105,210,170)(107,234,144)(112,138,235)(114,196,175)(118,160,207)(119,232,134)(121,221,143)(124,141,220)(127,192,166)(128,135,222)(129,185,171)(131,152,202)(133,153,199)(136,200,149)(140,155,190)(145,161,179)(151,156,178) ]", "PSL(2,239)^1" ],
  [   6967680, "[ (3,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,98)(3,54,142)(4,150,45)(5,55,139)(6,17,176)(7,34,158)(8,135,56)(9,161,29)(10,70,119)(11,115,73)(12,86,101)(13,89,97)(14,146,39)(15,153,31)(16,114,69)(18,53,128)(19,129,51)(20,179,190)(21,58,120)(22,137,40)(23,113,63)(24,149,26)(25,80,94)(27,88,84)(28,122,49)(30,125,44)(32,75,92)(33,87,79)(35,187,167)(36,78,85)(37,96,66)(38,162,189)(41,106,52)(42,64,93)(43,181,165)(46,192,151)(47,172,170)(48,60,91)(50,182,157)(57,141,191)(59,174,156)(61,188,140)(62,72,65)(67,177,145)(68,143,178)(71,166,152)(74,168,147)(76,138,175)(77,126,186)(81,185,123)(82,180,127)(83,173,133)(90,155,144)(95,110,184)(99,107,183)(100,159,130)(102,116,171)(103,132,154)(104,121,164)(105,136,148)(108,169,112)(109,163,117)(111,118,160)(124,134,131) ]", "PSL(2,191)^1" ],
  [   6998640, "[ (3,241,239,237,235,233,231,229,227,225,223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,242,240,238,236,234,232,230,228,226,224,222,220,218,216,214,212,210,208,206,204,202,200,198,196,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,123)(3,173,73)(4,33,212)(5,137,107)(6,54,189)(7,153,89)(8,142,99)(9,230,10)(11,167,71)(12,194,43)(13,146,90)(14,124,111)(15,105,129)(16,237,236)(17,56,176)(18,82,149)(19,200,30)(20,174,55)(21,103,125)(22,182,45)(23,61,165)(24,177,48)(25,116,108)(26,145,78)(27,50,172)(28,41,180)(29,184,36)(31,47,171)(32,51,166)(34,213,242)(35,87,127)(37,102,110)(38,134,77)(39,126,84)(40,58,151)(42,67,140)(44,113,92)(46,227,216)(49,86,114)(52,234,203)(53,131,65)(57,192,240)(59,94,96)(60,91,98)(62,217,210)(63,118,68)(64,224,201)(66,205,218)(69,222,198)(70,190,229)(72,226,191)(74,196,219)(75,199,215)(76,88,85)(79,235,175)(80,195,214)(81,185,223)(93,239,157)(95,188,206)(97,164,228)(100,233,156)(101,220,168)(104,238,147)(106,179,204)(109,241,139)(112,208,169)(115,193,181)(117,141,231)(119,159,211)(120,207,162)(121,143,225)(122,232,135)(128,183,178)(130,221,138)(132,160,197)(133,202,154)(136,144,209)(148,155,186)(150,152,187)(158,170,161) ]", "PSL(2,241)^1" ],
  [   7174332, "[ (3,243,241,239,237,235,233,231,229,227,225,223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6)(4,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5,244,242,240,238,236,234,232,230,228,226,224,222,220,218,216,214,212,210,208,206,204,202,200,198,196,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126), (1,2,4)(5,56,192)(6,79,168)(7,159,87)(8,177,68)(9,124,121)(10,228,15)(11,13,229)(12,26,215)(14,157,82)(16,42,195)(17,138,98)(18,77,158)(19,122,113)(20,236,238)(21,239,234)(22,194,37)(23,27,203)(24,99,130)(25,31,197)(28,70,155)(29,59,165)(30,88,135)(32,48,173)(33,140,80)(34,223,237)(35,69,149)(36,63,154)(38,112,104)(39,75,139)(40,118,96)(41,105,108)(43,166,44)(45,60,148)(46,222,226)(47,142,64)(49,116,89)(50,51,152)(52,218,224)(53,78,123)(54,207,233)(55,227,212)(57,193,244)(58,93,103)(61,73,120)(62,106,86)(65,74,115)(66,71,117)(67,85,102)(72,241,181)(76,201,217)(81,170,243)(83,206,205)(84,190,220)(90,242,162)(91,172,231)(92,235,167)(94,179,221)(95,186,213)(97,198,199)(100,180,214)(101,189,204)(107,202,185)(109,216,169)(110,174,210)(111,232,151)(114,161,219)(119,150,225)(125,240,128)(126,171,196)(127,230,136)(129,176,188)(131,209,153)(132,178,183)(133,200,160)(134,175,184)(137,211,145)(141,144,208)(143,187,163)(146,156,191)(147,164,182) ]", "PSL(2,243)^1" ],
  [   7188864, "[ (3,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,99)(3,65,133)(4,173,24)(5,116,80)(6,177,18)(7,70,124)(8,112,81)(9,91,101)(10,162,29)(11,77,113)(12,130,59)(13,146,42)(14,46,141)(15,149,37)(16,53,132)(17,62,122)(19,63,119)(20,22,159)(21,192,180)(23,148,30)(25,194,174)(26,72,103)(27,40,134)(28,129,44)(31,83,87)(32,71,98)(33,75,93)(34,89,78)(35,41,125)(36,188,169)(38,108,55)(39,176,178)(43,110,48)(45,54,102)(47,94,60)(49,183,161)(50,175,168)(51,58,92)(52,185,156)(56,61,84)(57,152,184)(64,158,171)(66,145,182)(68,186,139)(69,170,154)(73,157,163)(74,128,191)(76,136,181)(79,135,179)(82,193,118)(85,121,187)(86,190,117)(88,155,150)(90,160,143)(95,126,172)(96,144,153)(97,107,189)(100,127,166)(104,151,138)(105,123,165)(106,140,147)(109,164,120)(111,115,167)(114,137,142) ]", "PSL(2,193)^1" ],
  [   7645176, "[ (3,198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,101)(3,100,102)(4,116,85)(5,12,188)(6,131,68)(7,138,60)(8,87,110)(9,183,13)(10,38,157)(11,25,169)(14,190,197)(15,107,83)(16,63,126)(17,137,51)(18,84,103)(19,193,189)(20,136,49)(21,57,127)(22,128,55)(23,47,135)(24,69,112)(26,140,39)(27,28,150)(29,146,30)(31,77,97)(32,114,59)(33,177,191)(34,42,129)(35,46,124)(36,108,61)(37,89,79)(40,93,72)(41,106,58)(43,82,80)(44,50,111)(45,164,192)(48,104,53)(52,174,175)(54,81,70)(56,173,172)(62,176,163)(64,195,142)(65,185,151)(66,182,153)(67,155,179)(71,196,134)(73,160,168)(74,180,147)(75,145,181)(76,139,186)(78,156,167)(86,198,117)(88,170,143)(90,133,178)(91,152,158)(92,115,194)(94,166,141)(95,187,119)(96,161,144)(98,154,149)(99,118,184)(105,125,171)(109,162,130)(113,165,123)(120,159,122)(121,148,132) ]", "PSL(2,197)^1" ],
  [   7880400, "[ (3,200,199,198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,102)(3,194,10)(4,88,115)(5,56,146)(6,144,57)(7,27,173)(8,160,39)(9,103,95)(11,63,133)(12,78,117)(13,43,151)(14,139,54)(15,130,62)(16,159,32)(17,162,28)(18,128,61)(19,91,97)(20,153,34)(21,50,136)(22,104,81)(23,64,120)(24,59,124)(25,83,99)(26,152,29)(30,48,129)(31,177,197)(33,92,82)(35,66,106)(36,85,86)(37,132,38)(40,90,77)(41,96,70)(42,187,176)(44,196,165)(45,188,172)(46,94,67)(47,49,111)(51,184,170)(52,178,175)(53,161,191)(55,79,73)(58,148,199)(60,198,147)(65,190,150)(68,154,183)(71,141,193)(72,167,166)(74,189,142)(75,156,174)(76,186,143)(80,145,180)(84,140,181)(87,126,192)(89,116,200)(93,155,157)(98,138,169)(100,182,123)(101,195,109)(105,121,179)(107,113,185)(108,163,134)(110,158,137)(112,171,122)(114,164,127)(118,119,168)(125,149,131) ]", "PSL(2,199)^1" ],
  [   7906500, "[ (3,251,249,247,245,243,241,239,237,235,233,231,229,227,225,223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,252,250,248,246,244,242,240,238,236,234,232,230,228,226,224,222,220,218,216,214,212,210,208,206,204,202,200,198,196,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,128)(3,143,113)(4,130,125)(5,171,83)(6,194,59)(7,129,123)(8,33,218)(9,170,80)(10,141,108)(11,227,21)(12,193,54)(13,211,35)(14,23,222)(15,201,43)(16,81,162)(17,182,60)(18,97,144)(19,158,82)(20,215,24)(22,116,121)(25,79,155)(26,46,187)(27,157,75)(28,65,166)(29,245,235)(30,163,66)(31,137,91)(32,89,138)(34,233,242)(36,37,186)(38,223,248)(39,56,164)(40,72,147)(41,236,232)(42,103,114)(44,154,61)(45,243,221)(47,88,124)(48,51,160)(49,100,110)(50,152,57)(52,87,120)(53,67,139)(55,241,213)(58,106,95)(62,250,197)(63,244,202)(64,122,73)(68,107,84)(69,210,230)(70,219,220)(71,77,111)(74,239,196)(76,105,78)(85,251,173)(86,247,176)(90,191,228)(92,200,217)(93,226,190)(94,175,240)(96,205,208)(98,237,174)(99,229,181)(101,177,231)(102,212,195)(104,206,199)(109,184,216)(112,159,238)(115,246,148)(117,189,203)(118,167,224)(119,225,165)(126,252,131)(127,249,133)(132,168,209)(134,192,183)(135,140,234)(136,169,204)(142,153,214)(145,179,185)(146,156,207)(149,188,172)(150,198,161)(151,180,178) ]", "PSL(2,251)^1" ],
  [   8487168, "[ (3,257,255,253,251,249,247,245,243,241,239,237,235,233,231,229,227,225,223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,258,256,254,252,250,248,246,244,242,240,238,236,234,232,230,228,226,224,222,220,218,216,214,212,210,208,206,204,202,200,198,196,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,131)(3,83,179)(4,35,226)(5,28,232)(6,206,53)(7,64,194)(8,153,104)(9,240,16)(10,94,161)(11,59,195)(12,118,135)(13,107,145)(14,20,231)(15,91,159)(17,133,115)(18,205,42)(19,174,72)(21,96,148)(22,253,246)(23,49,193)(24,65,176)(25,140,100)(26,134,105)(27,141,97)(29,149,87)(30,234,257)(31,242,248)(32,199,34)(33,89,143)(36,227,258)(37,44,184)(38,71,156)(39,74,152)(40,177,48)(41,151,73)(43,62,160)(45,66,154)(46,167,52)(47,120,98)(50,77,138)(51,130,84)(54,136,75)(55,150,60)(56,256,209)(57,244,220)(58,82,125)(61,123,81)(63,230,228)(67,203,251)(68,198,255)(69,213,239)(70,116,79)(76,90,99)(78,218,225)(80,93,92)(85,222,214)(86,197,238)(88,243,190)(95,216,210)(101,168,252)(102,200,219)(103,171,247)(106,191,224)(108,196,217)(109,254,158)(110,188,223)(111,221,189)(112,207,202)(113,233,175)(114,166,241)(117,155,249)(119,173,229)(121,235,165)(122,237,162)(124,185,212)(126,208,187)(127,144,250)(128,236,157)(129,245,147)(132,211,178)(137,180,204)(139,201,181)(142,215,164)(146,192,183)(163,172,186)(169,182,170) ]", "PSL(2,257)^1" ],
  [   8491392, "[ (1,352,314,665,48,399)(2,353,73,424,218,569)(3,354,316,667,37,388)(4,355,205,556,276,627)(5,356,162,513,103,454)(6,357,287,638,123,474)(7,358,124,475,196,547)(8,359,90,441,265,616)(9,360,28,379,288,639)(10,361,39,390,94,445)(11,362,138,489,114,465)(12,363,178,529,30,381)(13,364,225,576,237,588)(14,365,34,385,325,676)(15,366,142,493,87,438)(16,367,105,456,147,498)(17,368,198,549,31,382)(18,369,248,599,339,690)(19,370,295,646,330,681)(20,371,240,591,266,617)(21,372,238,589,157,508)(22,373,128,479,93,444)(23,374,305,656,171,522)(24,375,293,644,315,666)(25,376,256,607,64,415)(26,377,313,664,181,532)(27,378,319,670,161,512)(29,380,49,400,231,582)(32,383,88,439,273,624)(33,384,202,553,38,389)(35,386,232,583,344,695)(36,387,77,428,170,521)(40,391,214,565,68,419)(41,392,219,570,130,481)(42,393,58,409,284,635)(43,394,110,461,245,596)(44,395)(45,396,230,581,341,692)(46,397,335,686,65,416)(47,398,63,414,199,550)(50,401,153,504,268,619)(51,402,303,654,167,518)(52,403,177,528,249,600)(53,404,321,672,272,623)(54,405,290,641,99,450)(55,406,137,488,186,537)(56,407,168,519,343,694)(57,408,192,543,337,688)(59,410,259,610,81,432)(60,411,309,660,282,633)(61,412,78,429,324,675)(62,413)(66,417,292,643,96,447)(67,418,155,506,190,541)(69,420,334,685,298,649)(70,421,158,509,71,422)(72,423,206,557,97,448)(74,425,220,571,193,544)(75,426,211,562,348,699)(76,427,215,566,228,579)(79,430,333,684,216,567)(80,431,318,669,253,604)(82,433)(83,434,187,538,95,446)(84,435,117,468,242,593)(85,436,133,484,255,606)(86,437,308,659,129,480)(89,440,101,452,149,500)(91,442,328,679,125,476)(92,443,104,455,154,505)(98,449)(100,451,182,533,119,470)(102,453,152,503,283,634)(106,457,166,517,169,520)(107,458,140,491,197,548)(108,459,229,580,135,486)(109,460,236,587,261,612)(111,462,258,609,212,563)(112,463,269,620,351,702)(113,464,304,655,278,629)(115,466,207,558,143,494)(116,467,227,578,160,511)(118,469,189,540,184,535)(120,471,338,689,326,677)(121,472,204,555,235,586)(122,473,185,536,280,631)(126,477,296,647,175,526)(127,478,347,698,141,492)(131,482,144,495,267,618)(132,483,302,653,263,614)(134,485,279,630,262,613)(136,487,312,663,251,602)(139,490,250,601,217,568)(145,496,151,502,191,542)(146,497,203,554,327,678)(148,499)(150,501,224,575,210,561)(156,507,183,534,323,674)(159,510,275,626,306,657)(163,514,311,662,179,530)(164,515,336,687,294,645)(165,516,285,636,286,637)(172,523,317,668,252,603)(173,524,301,652,299,650)(174,525,257,608,310,661)(176,527,188,539,208,559)(180,531,221,572,234,585)(194,545,320,671,307,658)(195,546)(200,551,226,577,201,552)(209,560)(213,564,300,651,264,615)(222,573,346,697,246,597)(223,574,233,584,239,590)(241,592,289,640,332,683)(243,594,340,691,297,648)(244,595,342,693,345,696)(247,598,277,628,271,622)(254,605,350,701,281,632)(260,611,291,642,349,700)(270,621,329,680,331,682)(274,625)(322,673), (1,183,136,258,64,38,160,151,190,176,303,300)(2,234,177,245,268,26,51,150,138,24,129,142)(3,312,292,246,200,228,86,152,193,278,304,83)(4,16,56,295,156,174,263,92,269,209,270,277)(5,334,12,272,207,194,65,238,198,265,333,253)(6,206,280,22,159,154,89,148,232,330,267,240)(7,336,87,237,116,239,315,227,55,109,78,199)(8,102,34,338,192,31,37,217,242,344,313,133)(9,119,208,314,255,141,310,18,113,172,282,266)(10,33,121,181,281,147,61,345,66,179,81,241)(11,59,331,124)(13,214,49,140,254,123,257,210,324,178,339,307)(14,73,251,294,27,168,68,297,162,126,221,104)(15,105,259,191,249,48,74,84,215,351,75,279)(17,163,289,262,350,248,39,166,46,82,88,225)(19,349,216,311,261,171,220,305,189,54,145,205)(20,318,41,323,42,233,302,98,60,80,218,155)(21,167,226,90,247,197,196,212,157,131,69,285)(23,36,284,114,231,132)(25,44,319,213,322,309,180,342,58,290,57,71)(28,182,153,201,299,187,343,101,40,185,77,308)(29,340,115,244,195,72,286,47,202,184,96,85)(30,118,271,346,320,223,164,106,326,243,236,287)(32,135,301,250,76,158)(35,173,144,276,211,306,325,62,288,117,222,45)(43,169,230,79,204,252,188,146,229,137,93,107)(50,329,293,63)(52,53,291,70,111,321,112,283,122,97,274,149)(67,100,139,275,260,327,161,95,341,332,347,235)(91,256,298,186,127,99,328,110,273,128,348,143)(94,103,130,108)(120,203,316,337)(125,175,264,219)(134,335,165,224)(352,507,489,414,428,462,481,700,429,415,361,540)(353,555,373,693,448,498,403,625,623,362,435,691)(354,518,461,473,676,661,469,454,532,605,365,598)(355,452,590,471,519,385,632,694,667,687,464,479)(356,502,426,537,526,631,684,412,408,371,654,393)(357,368,379,523,395,524,360,701,591,382,485,380)(358,495,410,400,366,594,463,626,422,567,653,560)(359,556,520,503,664,434,436,639,677,645,583,494)(363,643,407,627,647,663,501,418,499,369,652,535)(364,496,678,601,404,378,666,480,391,551,630,613)(367,534,533,565,681,472,509,513,651,574,573,386)(370,635,572,611,672,578,539,478,614,628,542,401)(372,620,671,466,607,584,531,659,488,484,606,558)(374,505,655,608,668,529,699,389,465,402,588,638)(375,696,417,592,568,514,596,440,685,455,447,538)(376,562,545,662,460,433,510,536,522,609,521,432)(377,612,604,595,575,482,549,438,683,670,427,445)(381,493,477,398,633,688,603,431,409,559,610,487)(383,517,669,486,650,597,686,557,506,474,602,618)(384,476,437,579,563,491,420,441,512,600,554,692)(387,657,570,421,392,587,443,553,636,702,423,622)(388,660,451,673,682,419,424,475,621,450,641,589)(390,439,582,679,467,642)(394,571,640,528,624,690)(396,581,530,593,698,680,665,619,566,425,646,548)(397,458,515)(399,459,453,444,430,552,634,541,470,675,497,616)(405,508,689,615,449,585,504,543,637,511,416,674)(406,644,564,442,629,490,580,550,516,649,547,561)(411,483,586,617,525,577,648,658,527,413,697,569)(446,695,457,468,599,456,576,492,544,546,656,500) ]", "G(2, 3)^1" ],
  [   9095592, "[ (3,263,261,259,257,255,253,251,249,247,245,243,241,239,237,235,233,231,229,227,225,223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,264,262,260,258,256,254,252,250,248,246,244,242,240,238,236,234,232,230,228,226,224,222,220,218,216,214,212,210,208,206,204,202,200,198,196,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,134)(3,206,62)(4,156,111)(5,144,122)(6,27,238)(7,14,250)(8,24,239)(9,92,170)(10,83,178)(11,123,137)(12,184,75)(13,236,22)(15,210,46)(16,202,53)(17,121,133)(18,254,261)(19,51,201)(20,77,174)(21,172,78)(23,192,56)(25,33,213)(26,208,37)(28,171,72)(29,244,260)(30,241,262)(31,91,149)(32,255,246)(34,188,49)(35,116,120)(36,129,106)(38,74,159)(39,187,45)(40,99,132)(41,102,128)(42,47,182)(43,63,165)(44,114,113)(48,73,150)(50,117,104)(52,89,130)(54,160,57)(55,235,243)(58,253,222)(59,70,142)(60,242,231)(61,85,125)(64,107,100)(65,127,79)(66,252,215)(67,217,249)(68,110,93)(69,115,87)(71,105,95)(76,245,212)(80,234,219)(81,229,223)(82,88,101)(84,256,193)(86,221,226)(90,185,258)(94,191,248)(96,247,190)(97,240,196)(98,176,259)(103,205,225)(108,214,211)(109,194,230)(112,264,157)(118,195,220)(119,177,237)(124,263,146)(126,198,209)(131,145,257)(135,147,251)(136,169,228)(138,179,216)(139,232,162)(140,166,227)(141,203,189)(143,183,207)(148,152,233)(151,218,164)(153,199,181)(154,224,155)(158,200,175)(161,204,168)(163,197,173)(167,180,186) ]", "PSL(2,263)^1" ],
  [   9360000, "[ (8,11,10,9)(12,27,22,17)(13,31,25,19)(14,28,26,20)(15,29,23,21)(16,30,24,18)(32,33,35,36)(37,53,50,46)(38,55,51,42)(39,54,49,44)(40,56,47,43)(41,52,48,45)(57,58,60,61)(62,78,75,71)(63,80,76,67)(64,79,74,69)(65,81,72,68)(66,77,73,70)(82,83,85,86)(87,103,100,96)(88,105,101,92)(89,104,99,94)(90,106,97,93)(91,102,98,95)(107,108,110,111)(112,128,125,121)(113,130,126,117)(114,129,124,119)(115,131,122,118)(116,127,123,120)(132,133,135,136)(137,153,150,146)(138,155,151,142)(139,154,149,144)(140,156,147,143)(141,152,148,145), (3,6,5,4)(8,11,10,9)(13,16,15,14)(18,21,20,19)(23,26,25,24)(28,31,30,29)(33,35)(34,36)(37,52,47,42)(38,55,48,45)(39,56,49,46)(40,53,50,43)(41,54,51,44)(57,132,107,82)(58,135,108,85)(59,136,109,86)(60,133,110,83)(61,134,111,84)(62,152,122,92)(63,155,123,95)(64,156,124,96)(65,153,125,93)(66,154,126,94)(67,137,127,97)(68,140,128,100)(69,141,129,101)(70,138,130,98)(71,139,131,99)(72,142,112,102)(73,145,113,105)(74,146,114,106)(75,143,115,103)(76,144,116,104)(77,147,117,87)(78,150,118,90)(79,151,119,91)(80,148,120,88)(81,149,121,89), (1,7,32,37,63,56,41,138,50,40,88,44,39,113,3)(2,8,107)(4,10,57,52,141,100,46,114,25,35,62,156,49,90,119)(5,11,82,42,89,144,51,65,81,55,116,19,34,137,125)(6,9,132,47,115,31,36,87,94,45,139,75,54,66,131)(12,58,102,68,153,149,148,98,71,103,118,14,108,22,110)(13,33,112)(15,83,142,151,123,21,109,17,84,67,78,80,105,143,126)(16,133,72,104,93,95,70,128,30,111,27,136,147,73,129)(18,59,127)(20,134,97,146,76,154,124,29,61,77,130,24,85,92,120)(23,135,122)(26,60,152,99,121)(28,86,117)(43,64,106)(48,140,150)(53,91,69)(74,79,155)(96,145,101) ]", "PSp(4,5)^1" ],
  [   9393720, "[ (3,212,211,210,209,208,207,206,205,204,203,202,201,200,199,198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,108)(3,107,109)(4,65,150)(5,186,28)(6,22,191)(7,119,93)(8,113,98)(9,42,168)(10,195,14)(11,86,122)(12,35,172)(13,37,169)(15,153,51)(16,125,78)(17,43,159)(18,102,99)(19,29,171)(20,135,64)(21,206,202)(23,55,141)(24,62,133)(25,194,210)(26,160,33)(27,68,124)(30,211,188)(31,127,61)(32,59,128)(34,111,74)(36,82,101)(38,69,112)(39,84,96)(40,129,50)(41,52,126)(44,181,204)(45,187,197)(46,106,67)(47,179,203)(48,174,207)(49,54,116)(53,95,71)(56,190,183)(57,173,199)(58,76,85)(60,80,79)(63,201,165)(66,151,212)(70,77,72)(75,161,193)(81,196,152)(83,154,192)(87,176,166)(88,157,184)(89,185,155)(90,164,175)(91,200,138)(92,148,189)(94,130,205)(97,209,123)(100,162,167)(103,208,118)(104,147,178)(105,182,142)(110,170,149)(114,198,117)(115,134,180)(120,132,177)(121,163,145)(131,140,158)(136,156,137)(139,146,144) ]", "PSL(2,211)^1" ],
  [   9539712, "[ (1,15,14,20,11,25,4,16,8,23,5,26,2,17)(3,18,6,27,9,24,10,22,12,21,7,19,13,28), (3,14,4,5,12,8,13,10,9,6,7,11), (17,28,18,19,26,22,27,24,23,20,21,25), (2,12,8,6,5,3)(7,13,11,14,9,10) ]", "PSL(2,13)^2" ],
  [   9653280, "[ (15,27)(16,39)(17,51)(18,63)(19,75)(20,87)(21,99)(22,111)(23,123)(24,135)(25,147)(26,159)(28,40)(29,52)(30,64)(31,76)(32,88)(33,100)(34,112)(35,124)(36,136)(37,148)(38,160)(41,53)(42,65)(43,77)(44,89)(45,101)(46,113)(47,125)(48,137)(49,149)(50,161)(54,66)(55,78)(56,90)(57,102)(58,114)(59,126)(60,138)(61,150)(62,162)(67,79)(68,91)(69,103)(70,115)(71,127)(72,139)(73,151)(74,163)(80,92)(81,104)(82,116)(83,128)(84,140)(85,152)(86,164)(93,105)(94,117)(95,129)(96,141)(97,153)(98,165)(106,118)(107,130)(108,142)(109,154)(110,166)(119,131)(120,143)(121,155)(122,167)(132,144)(133,156)(134,168)(145,157)(146,169)(158,170), (3,170,169,168,167,166,165,164,163,162,161,160,159,158,14,157,156,155,154,153,152,151,150,149,148,147,146,145,13,144,143,142,141,140,139,138,137,136,135,134,133,132,12,131,130,129,128,127,126,125,124,123,122,121,120,119,11,118,117,116,115,114,113,112,111,110,109,108,107,106,10,105,104,103,102,101,100,99,98,97,96,95,94,93,9,92,91,90,89,88,87,86,85,84,83,82,81,80,8,79,78,77,76,75,74,73,72,71,70,69,68,67,7,66,65,64,63,62,61,60,59,58,57,56,55,54,6,53,52,51,50,49,48,47,46,45,44,43,42,41,5,40,39,38,37,36,35,34,33,32,31,30,29,28,4,27,26,25,24,23,22,21,20,19,18,17,16,15), (1,2,9)(3,8,10)(4,5,12)(6,13,14)(15,146,38)(16,158,25)(17,68,114)(18,28,153)(19,151,29)(20,130,50)(21,76,103)(22,118,60)(23,115,62)(24,78,98)(26,93,80)(27,169,160)(30,128,41)(31,69,99)(32,157,167)(33,36,131)(34,166,156)(35,104,61)(37,120,42)(39,170,147)(40,97,63)(43,84,72)(44,59,96)(45,108,46)(47,79,74)(48,56,95)(49,85,66)(51,91,58)(52,75,73)(53,64,83)(54,149,152)(55,165,135)(57,155,144)(65,148,143)(67,163,125)(70,162,123)(71,117,168)(77,140,139)(81,150,124)(82,109,164)(86,116,154)(87,107,161)(88,145,122)(89,126,141)(90,129,137)(92,159,105)(94,134,127)(100,136,119)(101,142,113)(102,121,132)(106,138,111)(110,133,112) ]", "PSL(2,169)^1" ],
  [   9732420, "[ (3,269,267,265,263,261,259,257,255,253,251,249,247,245,243,241,239,237,235,233,231,229,227,225,223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,270,268,266,264,262,260,258,256,254,252,250,248,246,244,242,240,238,236,234,232,230,228,226,224,222,220,218,216,214,212,210,208,206,204,202,200,198,196,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,137)(3,136,138)(4,28,245)(5,197,75)(6,187,84)(7,32,238)(8,66,203)(9,55,213)(10,42,225)(11,160,106)(12,123,142)(13,57,207)(14,169,94)(15,145,117)(16,96,165)(17,159,101)(18,233,26)(19,91,167)(20,177,80)(21,188,68)(22,134,121)(23,88,166)(24,218,35)(25,73,179)(27,39,211)(29,246,270)(30,112,135)(31,172,74)(33,193,51)(34,100,143)(36,242,267)(37,59,181)(38,76,163)(40,192,45)(41,256,248)(43,50,184)(44,144,89)(46,113,118)(47,110,120)(48,150,79)(49,232,264)(52,133,92)(53,146,78)(54,125,98)(56,250,239)(58,155,64)(60,148,69)(61,219,265)(62,116,99)(63,235,247)(65,127,85)(67,217,261)(70,103,104)(71,208,266)(72,83,122)(77,269,199)(81,241,223)(82,234,229)(86,253,206)(87,268,190)(90,224,231)(93,215,237)(95,201,249)(97,254,194)(102,243,200)(105,260,180)(107,183,255)(108,186,251)(109,178,258)(111,198,236)(114,263,168)(115,257,173)(119,216,210)(124,226,195)(126,214,205)(128,221,196)(129,259,157)(130,230,185)(131,174,240)(132,151,262)(139,162,244)(140,252,153)(141,222,182)(147,209,189)(149,220,176)(152,191,202)(154,164,227)(156,161,228)(158,212,175)(170,171,204) ]", "PSL(2,269)^1" ],
  [   9951120, "[ (3,271,269,267,265,263,261,259,257,255,253,251,249,247,245,243,241,239,237,235,233,231,229,227,225,223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5)(4,272,270,268,266,264,262,260,258,256,254,252,250,248,246,244,242,240,238,236,234,232,230,228,226,224,222,220,218,216,214,212,210,208,206,204,202,200,198,196,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6), (1,2,138)(3,254,22)(4,266,9)(5,260,14)(6,38,235)(7,164,108)(8,185,86)(10,272,267)(11,69,199)(12,200,67)(13,223,43)(15,46,218)(16,271,262)(17,184,78)(18,99,162)(19,150,110)(20,62,197)(21,101,157)(23,35,221)(24,146,109)(25,143,111)(26,72,181)(27,186,66)(28,201,50)(29,121,129)(30,44,205)(31,125,123)(32,132,115)(33,118,128)(34,163,82)(36,182,61)(37,139,103)(39,48,192)(40,85,154)(41,238,270)(42,68,169)(45,136,98)(47,152,80)(49,51,179)(52,57,170)(53,263,233)(54,142,83)(55,241,253)(56,63,160)(58,230,261)(59,120,100)(60,131,88)(64,141,74)(65,87,127)(70,105,104)(71,232,246)(73,89,117)(75,248,226)(76,264,209)(77,207,265)(79,214,256)(81,96,102)(84,228,237)(90,249,210)(91,268,190)(92,259,198)(94,240,215)(95,204,250)(97,225,227)(106,219,224)(107,208,234)(112,269,168)(113,242,194)(114,177,258)(116,213,220)(119,175,255)(122,191,236)(124,229,196)(126,257,166)(130,252,167)(133,251,165)(134,222,193)(135,212,202)(137,239,173)(140,231,178)(144,244,161)(145,216,188)(147,155,247)(148,158,243)(149,189,211)(151,245,153)(156,217,176)(159,187,203)(171,206,172)(174,180,195) ]", "PSL(2,271)^1" ],
  [   9999360, "[ (16,24)(17,25)(18,26)(19,27)(20,28)(21,29)(22,30)(23,31)(39,55)(40,56)(41,57)(42,58)(43,59)(44,60)(45,61)(46,62), (1,2,4,8,16)(3,6,12,24,17)(5,10,20,9,18)(7,14,28,25,19)(11,22,13,26,21)(15,30,29,27,23)(32,33,35,39,47)(34,37,43,55,48)(36,41,51,40,49)(38,45,59,56,50)(42,53,44,57,52)(46,61,60,58,54) ]", "PSL(5,2)^1" ],
  [  10368000, "[ (6,12,9,14,7,15)(8,11)(10,13), (1,15,6)(2,14,7)(3,11,8)(4,12,9)(5,13,10), (4,5), (14,15), (9,10), (1,5,4,3,2) ]", "A5^3" ],
  [  11031552, "[ (3,371,152,378,87,36,10,243,485,235,82,216,61,481)(4,257,258,142,155,137,18,124,510,477,119,245,47,252)(5,161,297,157,192,429,26,473,213,397,320,282,159,194)(6,435,384,67,263,369,33,176,321,200,85,256,58,76)(7,141,98,377,38,92,40,222,410,121,106,399,210,115)(8,310,66,395,309,390,48,341,337,153,228,90,104,88)(9,284,168,206,63,163,32,266,362,204,131,469,193,461)(11,237,251,402,209,102,69,160,353,260,60,388,35,105)(12,229,175,129,71,78,75,276,463,460,100,334,146,387)(13,54,24,101,44,180,29)(14,265,122,225,199,426,89,492,453,404,182,312,109,270)(15,365,46,62,476,267,97,73,261,84,451,443,408,359)(16,91,280,486,116,117,59,356,144,488,226,407,311,118)(17,274,339,123,220,394,107,307,513,455,323,333,425,195)(19,364,292,354,148,186,120,363,37,346,227,181,296,269)(20,303,376,250,332,420,128,264,368,56,83,147,65,415)(21,52,114,154,393,335,136,286,150,372,43,25,422,501)(22,330,288,165,132,345,140,211,512,277,178,201,249,241)(23,379,232,445,231,77,145,382,392,396,411,273,432,301)(27,230,299,190,162,465,167,344,491,221,55,500,496,177)(28,158,86,39,467,45,173,430,246,234,509,255,493,81)(30,328,236,272,133,391,110,294,212,64,94,49,34,386)(31,268,457,242,343,338,188,416,505,111,441,164,480,238)(41,357,484,355,478,247,244,468,184,456,306,240,444,42)(50,414,419,233,271,427,278,506,327,324,103,134,358,108)(51,439,254,446,434,508,283,449,385,475,304,360,502,208)(53,287,196,348,197,472,130,452,198,454,262,347,466,285)(57,214,483,489,459,373,305,479,96,139,215,314,127,398)(68,253,281,340,179,433,342,322,318,171,187,440,499,413)(70,464,113,375,436,418,349,207,374,462,112,406,351,409)(72,448,174,183,291,471,295,93,308,495,447,80,298,401)(74,191,424,487,302,326,95,450,381,352,494,366,405,279)(79,370,470,143,361,156,313,189,497,325,135,421,166,383)(99,428,316,336,317,511,224,223,248,172,319,329,482,300)(125,169,389,151,380,218,289,498,442,417,203,490,185,504)(126,474,403,202,293,412,458,400,437,290,219,149,438,217)(138,205,275,331,315,367,170,239,259,350,503,507,423,431), (3,4,6,10,18,33,61,47,58,87,155,263,82,119,85,152,258,384,485,510,321)(5,8,14,26,48,89,159,104,109,192,309,199,320,228,182,297,66,122,213,337,453)(7,12,22,40,75,140,210,146,249,38,71,132,106,100,178,98,175,288,410,463,512)(9,16,30,32,59,110,193,311,34,63,116,133,131,226,94,168,280,236,362,144,212)(11,20,37,69,128,19,35,65,120,209,332,296,60,83,148,251,376,227,353,368,292)(13,24,44,29,54,101,180)(15,28,52,97,173,286,408,493,501,476,467,335,451,509,25,46,86,154,261,246,372)(17,31,57,107,188,305,425,480,127,220,343,459,323,441,215,339,457,483,513,505,96)(21,39,73,136,234,359,422,158,267,393,430,443,43,81,62,114,45,84,150,255,365)(23,42,79,145,247,313,432,240,166,231,355,361,411,456,135,232,357,470,392,468,497)(27,50,93,167,278,401,496,358,471,162,271,80,55,103,183,299,419,495,491,327,448)(36,67,124,216,200,252,378,435,137,235,176,245,371,76,142,243,369,477,481,256,257)(41,77,143,244,273,325,444,445,370,478,396,189,306,379,383,484,382,156,184,301,421)(49,91,163,272,356,469,64,118,206,328,117,204,294,407,284,386,486,266,391,488,461)(51,95,170,283,405,423,502,302,315,434,494,503,304,424,275,254,381,259,385,74,138)(53,99,70,130,224,349,466,482,351,197,317,436,262,319,112,196,316,113,198,248,374)(56,105,186,303,102,181,264,388,354,415,402,346,420,260,364,147,237,363,250,160,269)(68,126,218,342,458,490,499,438,151,179,293,417,187,219,169,281,403,498,318,437,504)(72,134,230,295,233,344,298,324,177,291,414,465,447,506,500,174,108,190,308,427,221)(78,121,211,334,141,241,129,222,345,460,115,201,229,92,165,276,399,277,387,377,330)(88,157,265,390,397,492,90,161,270,395,473,426,153,194,312,310,429,225,341,282,404)(111,195,314,268,394,489,416,333,139,238,123,214,338,455,479,164,274,398,242,307,373)(125,217,340,289,412,171,185,149,253,380,202,322,203,290,413,389,474,433,442,400,440)(172,285,406,428,472,375,223,347,462,300,348,464,511,454,207,329,287,409,336,452,418)(191,208,331,450,508,350,279,360,205,326,446,239,366,475,431,487,439,367,352,449,507), (1,2,3,5,9,17,32,60,112,197,318,438,498,198,319,439,505,16,31,58,109)(4,7,13,25,47,88,158,268,329,432,380,395,491,451,291,415,218,138,237,364,474)(6,11,21,39,74,139,239,343,224,350,75,141,242,241,368,476,332,220,344,459,93)(8,15,29,55,104,185,302,423,105,187,304,345,52,98,176,290,357,285,407,150,256)(10,19,36,68,127,132,228,305,426,73,137,236,258,50,94,169,282,331,89,160,170)(12,23,43,82,147,175,289,230,354,417,462,342,424,276,400,495,502,122,81,42,80)(14,27,51,96,172,211,335,403,374,427,475,452,57,108,191,154,262,387,450,485,317)(18,34,64,119,208,190,306,271,366,308,406,238,365,311,178,292,416,441,457,40,76)(20,38,72,135,233,274,33,62,115,202,323,442,301,422,501,143,245,328,449,429,503)(22,41,78,35,66,123,215,48,90,162,193,63,117,205,327,339,359,118,207,330,376)(24,45,85,153,260,386,371,480,455,103,184,247,373,481,477,46,87,156,264,389,487)(26,49,92,166,277,167,279,402,216,209,333,270,396,408,255,383,412,116,203,324,443)(28,53,100,179,294,418,500,303,363,281,173,99,177,280,148,252,379,470,378,483,469)(30,56,106,61,113,199,128,221,196,261,65,121,212,336,251,377,482,437,97,174,287)(37,70,131,227,206,186,136,234,360,159,71,133,229,226,352,286,326,447,488,506,471)(44,83,149,254,382,453,372,265,157,266,392,484,512,272,168,161,195,315,435,297,393)(54,102,182,298,84,151,257,353,181,296,232,358,472,398,494,414,213,249,375,253,165)(59,111,114,200,321,338,456,399,362,458,444,293,188,134,231,356,192,310,430,486,513)(67,125,79,146,250,355,419,142,204,325,445,489,492,288,411,283,401,497,217,341,101)(69,129,223,348,465,508,434,410,496,507,163,273,397,493,243,370,479,120,210,299,420)(77,144,246,337,454,510,309,428,155,86,107,189,307,145,248,124,130,225,351,468,391)(91,164,275,369,322,390,388)(95,171,284,340,384,478,235,361,473,244,320,440,314,433,269,347,463,126,219,183,300)(110,194,313,421,312,431,425,460,349,467,504,405,385,334,409,394,490,448,466,509,214)(140,240,367,381,446,404,499,267,201,152,259,180,295,413,464,278,263,222,346,461,511) ]", "PSU(3,8)^1" ],
  [  11089344, "[ (3,224,223,222,221,220,219,218,217,216,215,214,213,212,211,210,209,208,207,206,205,204,203,202,201,200,199,198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,114)(3,156,72)(4,198,29)(5,67,159)(6,210,15)(7,174,50)(8,125,98)(9,41,181)(10,56,165)(11,75,145)(12,108,111)(13,134,84)(14,38,179)(16,55,160)(17,48,166)(18,222,213)(19,121,91)(20,99,112)(21,149,61)(22,73,136)(23,182,26)(24,36,171)(25,65,141)(27,176,28)(30,224,199)(31,118,82)(32,148,51)(33,70,128)(34,131,66)(35,58,138)(37,89,105)(39,86,106)(40,95,96)(42,101,88)(43,124,64)(44,85,102)(45,71,115)(46,205,202)(47,187,219)(49,190,214)(52,201,200)(53,59,119)(54,221,178)(57,192,204)(60,78,93)(62,180,211)(63,172,218)(68,173,212)(69,161,223)(74,81,76)(79,207,167)(80,196,177)(83,153,217)(87,163,203)(90,170,193)(92,155,206)(94,215,144)(97,194,162)(100,158,195)(103,220,130)(104,185,164)(107,209,137)(109,169,175)(110,197,146)(113,157,183)(116,129,208)(117,120,216)(122,142,189)(123,139,191)(126,143,184)(127,186,140)(132,133,188)(135,150,168)(147,154,152) ]", "PSL(2,223)^1" ],
  [  11261376, "[ (1,58)(2,59)(3,60)(4,61)(5,62)(6,63)(7,64)(8,65)(9,66)(10,67)(11,68)(12,69)(13,70)(14,71)(15,72)(16,73)(17,74)(18,75)(19,76)(20,77)(21,78)(22,79)(23,80)(24,81)(25,82)(26,83)(27,84)(28,85)(29,86)(30,87)(31,88)(32,89)(33,90)(34,91)(35,92)(36,93)(37,94)(38,95)(39,96)(40,97)(41,98)(42,99)(43,100)(44,101)(45,102)(46,103)(47,104)(48,105)(49,106)(50,107)(51,108)(52,109)(53,110)(54,111)(55,112)(56,113)(57,114), (10,14,12)(11,15,13)(16,44,30)(17,49,33)(18,50,34)(19,45,35)(20,46,36)(21,47,31)(22,48,32)(23,51,37)(24,56,40)(25,57,41)(26,52,42)(27,53,43)(28,54,38)(29,55,39)(67,69,71)(68,70,72)(73,87,101)(74,90,106)(75,91,107)(76,92,102)(77,93,103)(78,88,104)(79,89,105)(80,94,108)(81,97,113)(82,98,114)(83,99,109)(84,100,110)(85,95,111)(86,96,112), (3,4,5,6,7,8)(10,15,14,13,12,11)(16,44,30)(17,50,35,20,47,32)(18,45,36,21,48,33)(19,46,31,22,49,34)(23,51,37)(24,57,42,27,54,39)(25,52,43,28,55,40)(26,53,38,29,56,41)(60,65,64,63,62,61)(67,68,69,70,71,72)(73,87,101)(74,89,104,77,92,107)(75,90,105,78,93,102)(76,91,106,79,88,103)(80,94,108)(81,96,111,84,99,114)(82,97,112,85,100,109)(83,98,113,86,95,110), (1,2,9,13,41,3,16,11,27,57,31,54,52,38,6,37)(4,23,15,55,17,46,35,40)(5,30,12,34,33,26,50,21,32,19,18,53,45,28,22,39)(7,44,14,48,49,56,24,36,47,42,8,51,10,20,25,43)(58,59,67,63,95,97,111,100,90,78,101,70,60,74,73,66)(61,81,110,93,99,83,82,75,80,68,62,88,106,77,94,69)(64,102,91,85,96,104,105,112,107,84,89,113,114,79,108,72)(65,109,86,103,98,76,87,71) ]", "PSL(3,7)^1" ],
  [  11327232, "[ (3,62,128,88,152,221,336,44,66,275,74,243)(4,298,202,53,188,253,338,146,19,331,123,199)(5,17,240,161,271,49,343,37,39,332,270,215)(6,287,180,227,192,169,46,268,35,87,138,323)(7,321,61,47,159,235,176,335,155,251,164,158)(8,147,131,41,326,141,342,26,71,20,104,324)(9,337,303,120,72,282,252,344,329,288,214,16)(10,300,32,212,168,150,80,260,63,228,211,50)(11,14,256,94,314,278,22,220,84,118,311,170)(12,160,105,248,238,25,101,157,177,330,242,280)(13,315,295,262,200,255,59,186,269,119,316,257)(15,241,148,213,293,90,30,301,328,231,193,103)(18,162,58,261,246,175,134,89,107,179,237,151)(21,263,203,31,279,68,249,85,197,67,317,36)(23,333,121,313,178,184,102,226,305,115,327,73)(24,265,191,334,135,40,341,173,217,277,29,306)(27,60,223,163,70,140,56,283,239,250,122,339)(28,266,165,285,43,42,64,289,258,297,167,114)(33,116,100,97,320,183,207,194,172,51,310,111)(34,196,95,340,109,286,92,236,98,325,130,77)(38,125,259,137,136,294,276,318,142,91,149,272)(45,233,108,187,208,292,106,144,290,219,54,302)(48,229,304,126,133,264,190,195,319,244,79,201)(52,69,210,189,75,209,82,166,234,216,245,225)(55,124,65,309,322,96,57,129,112,204,143,171)(76,281,308,81,247,312,154,205,224,139,198,110)(78,230,174,156,284,117,86,254,153,99,93,145)(83,127,181,185,206,307,222,232,267,296,132,273)(113,218,299)(182,291,274), (3,4,6,10,18,33,60,103,74,123,138,211,237,310,339,231,66,19,35,63,107,172,250,301,336,338,46,80,134,207,283,90,152,188,192,168,246,320,140,213,128,202,180,32,58,100,163,241)(5,8,14,25,47,77,129,203,270,104,170,248,321,325,96,21,39,71,118,160,158,236,309,317,343,342,220,280,251,286,124,197,271,326,278,330,335,340,171,249,240,131,94,157,235,196,204,279)(7,12,22,41,17,31,57,98,164,242,314,147,215,263,322,92,155,177,256,324,332,36,65,109,176,101,11,20,37,67,55,95,159,238,311,26,49,85,143,34,61,105,84,141,161,68,112,130)(9,16,29,54,93,156,233,306,214,288,217,290,153,230,302,277,329,344,341,106,86,145,219,173,252,282,135,208,284,99,144,40,72,120,191,108,174,254,292,334,303,337,24,45,78,117,187,265)(13,23,43,75,125,198,273,244,316,327,165,210,272,224,296,195,269,305,28,52,91,154,232,264,59,102,167,245,318,247,307,126,200,178,258,234,294,308,185,229,295,121,64,82,137,76,127,201)(15,27,51,89,150,227,298,243,193,122,194,175,212,287,199,275,328,239,183,261,300,323,331,44,30,56,97,162,50,87,146,221,293,70,116,151,228,268,253,88,148,223,111,179,260,169,53,62)(38,69,114,184,262,48,83,139,149,225,297,313,315,79,132,205,142,216,289,333,257,319,267,312,276,166,42,73,119,190,222,81,136,209,285,115,186,133,206,281,259,189,266,226,255,304,181,110)(113,182,218,291,299,274), (1,2,3,5,9,17,32,59,14,26,50,88,149,226,293,179,31,47,82,138,49,86,129,33)(4,7,13,24,46,81,109,177,257,80,135,52,61,18,34,62,106,171,154,127,39,16,30,12)(6,11,21,40,54,94,158,237,287,97,163,107,173,253,184,263,219,273,221,294,269,320,324,246)(8,15,28,53,92,123,196,241,207,233,307,170,182,209,286,281,328,159,239,313,258,262,256,105)(10,19,36,66,75,126,148,224,187,20,38,70,117,188,103,169,194,202,278,321,344,254,249,322)(22,42,74,124,197,272,327,334,311,85,144,218,292,41,57,99,165,244,317,343,100,166,143,217)(23,44,77,130,65,110,178,259,298,164,243,315,342,289,291,235,308,302,336,60,104,58,101,56)(25,48,84,142,114,185,35,64,72,121,193,150,191,261,275,167,174,102,168,247,69,115,112,181)(27,45,79,133,91,89,151,156,234,248,245,319,67,111,180,73,122,195,270,325,252,176,236,208)(29,55,96,161,240,314,341,284,306,339,330,242,260,137,205,280,186,264,162,95,160,213,230,303)(37,68,113,183,134,83,140,214,250,323,216,190,267,285,51,90,153,231,304,338,227,299,141,215)(43,76,128,136,210,203,145,220,152,229,301,266,116,63,108,175,255,118,189,71,119,98,146,93)(78,131,204,125,199,276,211,87,147,222,295,157,200,277,228,300,201,132,206,282,331,337,326,155)(120,192,268,225,283,318,172,251,265,279,316,333,310,232,305,329,223,238,312,340,297,335,139,212)(198,274,296)(288,332,290) ]", "PSU(3,7)^1" ],
  [  11696856, "[ (3,228,227,226,225,224,223,222,221,220,219,218,217,216,215,214,213,212,211,210,209,208,207,206,205,204,203,202,201,200,199,198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,116)(3,115,117)(4,70,161)(5,105,125)(6,24,205)(7,17,211)(8,42,185)(9,44,182)(10,146,79)(11,58,166)(12,26,197)(13,189,33)(14,69,152)(15,88,132)(16,142,77)(18,158,59)(19,154,62)(20,123,92)(21,215,225)(22,45,168)(23,110,102)(25,54,156)(27,208,226)(28,167,40)(29,61,145)(30,56,149)(31,68,136)(32,82,121)(34,148,53)(35,206,220)(36,95,104)(37,67,131)(38,124,73)(39,99,97)(41,119,75)(43,219,199)(46,91,98)(47,190,224)(48,106,81)(49,114,72)(50,188,223)(51,55,129)(52,94,89)(57,85,93)(60,112,63)(64,187,210)(65,204,192)(66,174,221)(71,162,228)(74,214,173)(76,178,207)(78,213,170)(80,163,218)(83,176,202)(84,198,179)(86,222,153)(87,171,203)(90,216,155)(96,164,201)(100,144,217)(101,165,195)(103,177,181)(107,127,227)(108,194,159)(109,212,140)(111,150,200)(113,191,157)(118,183,160)(120,172,169)(122,209,130)(126,184,151)(128,137,196)(133,193,135)(134,141,186)(138,180,143)(139,147,175) ]", "PSL(2,227)^1" ],
  [  11985408, "[ (1,19,10,35,8,29,3,27)(2,25,13,34,17,24,9,36)(4,20)(5,23,7,30,15,32,16,22)(6,31,12,26,11,33,14,21)(18,28), (1,4,2)(3,10,6)(5,12,8)(7,16,11)(9,17,14)(13,18,15) ]", "PSL(2,17)^2" ],
  [  12008760, "[ (3,230,229,228,227,226,225,224,223,222,221,220,219,218,217,216,215,214,213,212,211,210,209,208,207,206,205,204,203,202,201,200,199,198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,117)(3,96,138)(4,10,223)(5,125,107)(6,209,22)(7,190,40)(8,43,186)(9,166,62)(11,224,230)(12,193,32)(13,84,140)(14,66,157)(15,164,58)(16,169,52)(17,114,106)(18,71,148)(19,173,45)(20,101,116)(21,72,144)(23,55,159)(24,137,76)(25,228,212)(26,92,119)(27,35,175)(28,104,105)(29,139,69)(30,64,143)(31,126,80)(33,123,81)(34,82,121)(36,56,145)(37,49,151)(38,63,136)(39,67,131)(41,222,202)(42,46,149)(44,227,194)(47,102,88)(48,191,226)(50,134,53)(51,112,74)(54,73,110)(57,93,87)(59,199,207)(60,78,99)(61,215,189)(65,218,182)(68,225,172)(70,219,176)(75,179,211)(77,168,220)(83,185,197)(85,188,192)(86,163,216)(89,178,198)(90,162,213)(91,170,204)(94,150,221)(95,205,165)(97,210,158)(98,171,196)(100,184,181)(103,167,195)(108,203,154)(109,229,127)(111,201,153)(113,152,200)(115,142,208)(118,133,214)(120,217,128)(122,183,160)(124,161,180)(129,130,206)(132,187,146)(135,156,174)(141,177,147) ]", "PSL(2,229)^1" ],
  [  12130560, "[ (1,41)(2,42)(3,43)(4,44)(5,45)(6,46)(7,47)(8,48)(9,49)(10,50)(11,51)(12,52)(13,53)(14,54)(15,55)(16,56)(17,57)(18,58)(19,59)(20,60)(21,61)(22,62)(23,63)(24,64)(25,65)(26,66)(27,67)(28,68)(29,69)(30,70)(31,71)(32,72)(33,73)(34,74)(35,75)(36,76)(37,77)(38,78)(39,79)(40,80), (14,23,32)(15,24,33)(16,25,34)(17,26,35)(18,27,36)(19,28,37)(20,29,38)(21,30,39)(22,31,40)(45,72,63)(46,74,64)(47,73,65)(48,78,66)(49,80,67)(50,79,68)(51,75,69)(52,77,70)(53,76,71), (1,2,5,14)(3,8,23,16)(4,11,32,15)(6,17,7,20)(9,26,25,22)(10,29,34,18)(12,35,33,21)(13,38,24,19)(27,28,31,40)(30,37,36,39)(41,42,45,54)(43,48,63,56)(44,51,72,55)(46,57,47,60)(49,66,65,62)(50,69,74,58)(52,75,73,61)(53,78,64,59)(67,68,71,80)(70,77,76,79) ]", "PSL(4,3)^1" ],
  [  12130560, "[ (1,41)(2,42)(3,43)(4,44)(5,45)(6,46)(7,47)(8,48)(9,49)(10,50)(11,51)(12,52)(13,53)(14,54)(15,56)(16,55)(17,60)(18,62)(19,61)(20,57)(21,59)(22,58)(23,72)(24,74)(25,73)(26,78)(27,80)(28,79)(29,75)(30,77)(31,76)(32,63)(33,65)(34,64)(35,69)(36,71)(37,70)(38,66)(39,68)(40,67), (14,23,32)(15,24,33)(16,25,34)(17,26,35)(18,27,36)(19,28,37)(20,29,38)(21,30,39)(22,31,40)(45,72,63)(46,74,64)(47,73,65)(48,78,66)(49,80,67)(50,79,68)(51,75,69)(52,77,70)(53,76,71), (1,2,5,14)(3,8,23,16)(4,11,32,15)(6,17,7,20)(9,26,25,22)(10,29,34,18)(12,35,33,21)(13,38,24,19)(27,28,31,40)(30,37,36,39)(41,42,45,54)(43,48,63,56)(44,51,72,55)(46,57,47,60)(49,66,65,62)(50,69,74,58)(52,75,73,61)(53,78,64,59)(67,68,71,80)(70,77,76,79) ]", "PSL(4,3)^1" ],
  [  12130560, "[ (15,16)(17,20)(18,22)(19,21)(23,32)(24,34)(25,33)(26,38)(27,40)(28,39)(29,35)(30,37)(31,36)(55,56)(57,60)(58,62)(59,61)(63,72)(64,74)(65,73)(66,78)(67,80)(68,79)(69,75)(70,77)(71,76), (14,23,32)(15,24,33)(16,25,34)(17,26,35)(18,27,36)(19,28,37)(20,29,38)(21,30,39)(22,31,40)(45,72,63)(46,74,64)(47,73,65)(48,78,66)(49,80,67)(50,79,68)(51,75,69)(52,77,70)(53,76,71), (1,2,5,14)(3,8,23,16)(4,11,32,15)(6,17,7,20)(9,26,25,22)(10,29,34,18)(12,35,33,21)(13,38,24,19)(27,28,31,40)(30,37,36,39)(41,42,45,54)(43,48,63,56)(44,51,72,55)(46,57,47,60)(49,66,65,62)(50,69,74,58)(52,75,73,61)(53,78,64,59)(67,68,71,80)(70,77,76,79) ]", "PSL(4,3)^1" ],
  [  12649104, "[ (3,234,233,232,231,230,229,228,227,226,225,224,223,222,221,220,219,218,217,216,215,214,213,212,211,210,209,208,207,206,205,204,203,202,201,200,199,198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,119)(3,47,191)(4,207,30)(5,114,122)(6,217,18)(7,194,40)(8,154,79)(9,169,63)(10,203,28)(11,143,87)(12,73,156)(13,125,103)(14,104,123)(15,53,173)(16,166,59)(17,160,64)(19,101,121)(20,138,83)(21,232,220)(22,180,39)(23,161,57)(24,29,188)(25,127,89)(26,51,164)(27,168,46)(31,234,208)(32,38,171)(33,80,128)(34,55,152)(35,228,210)(36,76,129)(37,142,62)(41,108,92)(42,60,139)(43,93,105)(44,231,198)(45,90,106)(48,75,118)(49,85,107)(50,209,214)(52,91,98)(54,126,61)(56,97,88)(58,216,199)(65,185,223)(66,94,81)(67,200,206)(68,102,71)(69,229,175)(70,211,192)(72,222,179)(74,187,212)(77,215,181)(78,221,174)(82,165,226)(84,230,159)(86,183,204)(95,227,151)(96,201,176)(99,178,196)(100,218,155)(109,162,202)(110,158,205)(111,213,149)(112,184,177)(113,225,135)(115,134,224)(116,124,233)(117,137,219)(120,163,190)(130,197,146)(131,153,189)(132,148,193)(133,145,195)(136,170,167)(140,147,186)(141,182,150)(144,172,157) ]", "PSL(2,233)^1" ],
  [  12700800, "[ (1,8,7,9,6,10,5,11,4,12,3,13,2,14), (12,13,14) ]", "A7^2" ],
  [  13063680, "[ (4,268,5,278,7,239,11,222)(6,254,20,52,15,140,61,111)(8,64,33,88,22,172,41,190)(9,82,35,91,26,203,12,29)(10,165,106,78,18,60,229,25)(13,110,23,148,39,233,19,201)(14,259,72,51,43,149,185,150)(16,272,167,240,50,227,219,248)(17,226,168,133,31,103,62,45)(21,129,48,245,65,44,137,223)(24,265,122,89,74,210,42,211)(27,188,37,194,83,241,112,76)(28,225,228,197,87,270,105,183)(30,244,275,71,94,224,212,182)(32,273,151,184,97,189,179,249)(34,216,70,171,101,276,98,252)(36,198,81,214,108,177,155,166)(40,109,59,263,119,232,161,234)(46,205,255,136,126,215,186,250)(47,153,63,193,138,207,170,134)(49,77,174,269,143,66,208,173)(53,262,242,196,80,164,123,199)(54,162,264,120,154,257,261,128)(55,73,230,127,93,187,279,92)(56,125,117,217,157,246,107,121)(57,231,69,202,99,139,58,147)(68,237,142,206,175,114,251,236)(75,95,124,271,192,96,243,169)(79,191,141,178,158,260,163,152)(84,113,235,85,131,146,200,130)(86,145)(90,218,156,104,213,135,247,102)(100,253,118,159,180,220,209,266)(115,144,116,195,204,256,132,176)(160,277,258,267,238,221,181,274), (6,12)(8,19)(9,20)(10,62)(13,33)(14,42)(15,35)(16,105)(17,106)(18,168)(21,27)(22,23)(24,72)(25,103)(26,61)(28,167)(29,254)(30,261)(31,229)(32,107)(34,40)(36,47)(37,48)(38,67)(39,41)(43,122)(44,241)(45,165)(46,242)(49,113)(50,228)(51,265)(52,82)(53,186)(54,275)(55,169)(56,151)(57,99)(58,69)(59,70)(60,133)(63,81)(64,201)(65,83)(66,200)(68,115)(71,162)(73,75)(74,185)(76,223)(77,235)(78,226)(79,104)(80,255)(84,173)(85,174)(86,145)(87,219)(88,110)(89,149)(90,260)(91,140)(92,243)(93,271)(94,264)(95,230)(96,279)(97,117)(98,161)(101,119)(102,158)(108,138)(109,216)(111,203)(112,137)(114,256)(116,142)(120,224)(121,273)(123,126)(124,127)(125,184)(128,244)(129,188)(130,208)(131,269)(132,251)(134,166)(135,141)(136,164)(139,231)(143,146)(144,237)(147,202)(148,172)(150,210)(152,156)(153,198)(154,212)(155,170)(157,179)(160,238)(163,218)(171,263)(175,204)(176,236)(177,207)(178,247)(181,258)(182,257)(183,272)(187,192)(189,217)(190,233)(191,213)(193,214)(194,245)(195,206)(196,205)(197,227)(199,215)(211,259)(221,277)(225,240)(232,276)(234,252)(246,249)(248,270)(250,262)(267,274), (1,2,3,4,6,10,18,32,57)(5,8,14,25,46,81,139,180,238)(7,12,21,38,68,117,99,159,221)(9,16,29,53,93,28,51,90,153)(11,19,34,60,104,87,147,209,258)(13,23,42,73,125,190,91,120,183)(15,27,49,86,146,145,143,204,97)(17,30,54)(20,36,64,111,135,37,66,114,177)(22,40,71,121,148,203,243,214,41)(24,44,77,132,196,249,233,89,151)(26,47,82,141,94,155,216,250,240)(31,55,96)(33,58,100,160,222,232,262,165,227)(35,62,107)(39,69,118,181,239,210,260,78,134)(43,75,128,194,131,176,182,127,193)(45,79,136)(48,84,142,205,255,271,279,152,163)(50,88,149)(52,92,129,174,206,257,273,263,108)(56,98,158,219,265,154,215,246,259)(59,102,164,225,70,76,130,195,248)(61,105,110,140,186,112,173,236,270)(63,109,95,156,217,171,157,72,123)(65,113,175)(67,115,83)(74,126,191,198,185,241,85,144,207)(80,137,200,251,247,218,264,275,199)(101,162,124,188,235,256,187,244,272)(103,166,122)(106,169,167,202,253,274,268,150,212)(116,178,189,172,234,179,161,223,269)(119,133,197)(138,201,252)(170,231,220,267,278,276,230,229,261)(184,211,228,254,224,245,208,237,192)(213,226,242)(266,277,280) ]", "PSU(4,3)^1" ],
  [  13063680, "[ (4,278,7,222)(5,268,11,239)(6,52,15,111)(8,110,22,233)(9,29,26,91)(10,78,18,25)(12,82,35,203)(13,64,39,172)(14,171,43,252)(16,193,50,134)(17,45,31,133)(19,88,23,190)(20,254,61,140)(21,245,65,223)(24,109,74,232)(27,194,83,76)(28,198,87,177)(30,127,94,92)(32,184,97,249)(33,201,41,148)(34,51,101,150)(36,225,108,270)(37,188,112,241)(40,265,119,210)(42,263,122,234)(44,48,129,137)(46,247,126,156)(47,240,138,248)(49,269,143,173)(53,260,80,191)(54,75,154,192)(55,71,93,182)(56,121,157,217)(57,202,99,147)(58,139,69,231)(59,211,161,89)(60,106,165,229)(62,226,168,103)(63,272,170,227)(66,174,77,208)(68,206,175,236)(70,259,98,149)(72,216,185,276)(73,212,187,275)(79,164,158,262)(81,183,155,197)(84,113,131,146)(85,235,130,200)(86,145)(90,255,213,186)(95,128,96,120)(100,220,180,253)(102,250,104,136)(105,214,228,166)(107,125,117,246)(114,142,237,251)(115,195,204,176)(116,144,132,256)(118,159,209,266)(123,152,242,178)(124,264,243,261)(135,205,218,215)(141,196,163,199)(151,273,179,189)(153,219,207,167)(160,277,238,221)(162,271,257,169)(181,267,258,274)(224,230,244,279), (6,12)(8,19)(9,20)(10,62)(13,33)(14,42)(15,35)(16,105)(17,106)(18,168)(21,27)(22,23)(24,72)(25,103)(26,61)(28,167)(29,254)(30,261)(31,229)(32,107)(34,40)(36,47)(37,48)(38,67)(39,41)(43,122)(44,241)(45,165)(46,242)(49,113)(50,228)(51,265)(52,82)(53,186)(54,275)(55,169)(56,151)(57,99)(58,69)(59,70)(60,133)(63,81)(64,201)(65,83)(66,200)(68,115)(71,162)(73,75)(74,185)(76,223)(77,235)(78,226)(79,104)(80,255)(84,173)(85,174)(86,145)(87,219)(88,110)(89,149)(90,260)(91,140)(92,243)(93,271)(94,264)(95,230)(96,279)(97,117)(98,161)(101,119)(102,158)(108,138)(109,216)(111,203)(112,137)(114,256)(116,142)(120,224)(121,273)(123,126)(124,127)(125,184)(128,244)(129,188)(130,208)(131,269)(132,251)(134,166)(135,141)(136,164)(139,231)(143,146)(144,237)(147,202)(148,172)(150,210)(152,156)(153,198)(154,212)(155,170)(157,179)(160,238)(163,218)(171,263)(175,204)(176,236)(177,207)(178,247)(181,258)(182,257)(183,272)(187,192)(189,217)(190,233)(191,213)(193,214)(194,245)(195,206)(196,205)(197,227)(199,215)(211,259)(221,277)(225,240)(232,276)(234,252)(246,249)(248,270)(250,262)(267,274), (4,5,7,11)(6,9,15,26)(8,13,22,39)(10,17,18,31)(12,20,35,61)(14,24,43,74)(16,28,50,87)(19,33,23,41)(21,37,65,112)(25,45,78,133)(27,48,83,137)(29,52,91,111)(30,54,94,154)(32,56,97,157)(34,59,101,161)(36,63,108,170)(38,67)(40,70,119,98)(42,72,122,185)(44,76,129,194)(46,80,126,53)(47,81,138,155)(49,85,143,130)(51,89,150,211)(55,95,93,96)(57,58,99,69)(60,103,165,226)(62,106,168,229)(64,110,172,233)(66,84,77,131)(68,116,175,132)(71,120,182,128)(73,124,187,243)(75,127,192,92)(79,135,158,218)(82,140,203,254)(86,145)(88,148,190,201)(90,152,213,178)(100,118,180,209)(102,163,104,141)(105,167,228,219)(107,151,117,179)(109,171,232,252)(113,174,146,208)(114,176,237,195)(115,142,204,251)(121,184,217,249)(123,186,242,255)(125,189,246,273)(134,198,193,177)(136,199,250,196)(139,202,231,147)(144,206,256,236)(149,210,259,265)(153,214,207,166)(156,191,247,260)(159,220,266,253)(160,181,238,258)(162,224,257,244)(164,215,262,205)(169,230,271,279)(173,235,269,200)(183,240,197,248)(188,245,241,223)(212,261,275,264)(216,263,276,234)(221,267,277,274)(222,268,278,239)(225,227,270,272), (1,2,3,4,6,10,18,32,57)(5,8,14,25,46,81,139,180,238)(7,12,21,38,68,117,99,159,221)(9,16,29,53,93,28,51,90,153)(11,19,34,60,104,87,147,209,258)(13,23,42,73,125,190,91,120,183)(15,27,49,86,146,145,143,204,97)(17,30,54)(20,36,64,111,135,37,66,114,177)(22,40,71,121,148,203,243,214,41)(24,44,77,132,196,249,233,89,151)(26,47,82,141,94,155,216,250,240)(31,55,96)(33,58,100,160,222,232,262,165,227)(35,62,107)(39,69,118,181,239,210,260,78,134)(43,75,128,194,131,176,182,127,193)(45,79,136)(48,84,142,205,255,271,279,152,163)(50,88,149)(52,92,129,174,206,257,273,263,108)(56,98,158,219,265,154,215,246,259)(59,102,164,225,70,76,130,195,248)(61,105,110,140,186,112,173,236,270)(63,109,95,156,217,171,157,72,123)(65,113,175)(67,115,83)(74,126,191,198,185,241,85,144,207)(80,137,200,251,247,218,264,275,199)(101,162,124,188,235,256,187,244,272)(103,166,122)(106,169,167,202,253,274,268,150,212)(116,178,189,172,234,179,161,223,269)(119,133,197)(138,201,252)(170,231,220,267,278,276,230,229,261)(184,211,228,254,224,245,208,237,192)(213,226,242)(266,277,280) ]", "PSU(4,3)^1" ],
  [  13063680, "[ (5,11)(8,19)(9,26)(13,41)(14,34)(16,47)(17,31)(20,61)(22,23)(24,161)(25,60)(28,155)(29,82)(30,55)(33,39)(36,105)(37,112)(40,42)(43,101)(44,223)(45,226)(46,104)(48,137)(50,138)(51,216)(52,254)(53,141)(54,96)(56,157)(58,69)(59,74)(63,219)(64,110)(66,173)(70,185)(71,73)(72,98)(75,162)(76,241)(77,269)(78,165)(79,242)(80,163)(81,87)(84,200)(85,130)(88,201)(89,234)(90,250)(91,203)(92,224)(93,94)(95,154)(100,118)(102,126)(103,133)(106,229)(108,228)(109,265)(111,140)(114,236)(116,132)(119,122)(120,243)(121,125)(123,158)(124,128)(127,244)(129,245)(131,235)(134,227)(135,186)(136,213)(139,147)(142,251)(144,195)(148,190)(149,252)(150,276)(151,179)(152,199)(153,240)(156,215)(160,181)(164,191)(166,197)(167,170)(169,261)(171,259)(172,233)(174,208)(176,256)(177,270)(178,196)(180,209)(182,187)(183,214)(184,273)(188,194)(189,249)(192,257)(193,272)(198,225)(202,231)(205,247)(206,237)(207,248)(210,232)(211,263)(212,230)(217,246)(218,255)(220,253)(222,239)(238,258)(260,262)(264,271)(267,274)(268,278)(275,279), (6,12)(8,19)(9,20)(10,62)(13,33)(14,42)(15,35)(16,105)(17,106)(18,168)(21,27)(22,23)(24,72)(25,103)(26,61)(28,167)(29,254)(30,261)(31,229)(32,107)(34,40)(36,47)(37,48)(38,67)(39,41)(43,122)(44,241)(45,165)(46,242)(49,113)(50,228)(51,265)(52,82)(53,186)(54,275)(55,169)(56,151)(57,99)(58,69)(59,70)(60,133)(63,81)(64,201)(65,83)(66,200)(68,115)(71,162)(73,75)(74,185)(76,223)(77,235)(78,226)(79,104)(80,255)(84,173)(85,174)(86,145)(87,219)(88,110)(89,149)(90,260)(91,140)(92,243)(93,271)(94,264)(95,230)(96,279)(97,117)(98,161)(101,119)(102,158)(108,138)(109,216)(111,203)(112,137)(114,256)(116,142)(120,224)(121,273)(123,126)(124,127)(125,184)(128,244)(129,188)(130,208)(131,269)(132,251)(134,166)(135,141)(136,164)(139,231)(143,146)(144,237)(147,202)(148,172)(150,210)(152,156)(153,198)(154,212)(155,170)(157,179)(160,238)(163,218)(171,263)(175,204)(176,236)(177,207)(178,247)(181,258)(182,257)(183,272)(187,192)(189,217)(190,233)(191,213)(193,214)(194,245)(195,206)(196,205)(197,227)(199,215)(211,259)(221,277)(225,240)(232,276)(234,252)(246,249)(248,270)(250,262)(267,274), (4,5,7,11)(6,9,15,26)(8,13,22,39)(10,17,18,31)(12,20,35,61)(14,24,43,74)(16,28,50,87)(19,33,23,41)(21,37,65,112)(25,45,78,133)(27,48,83,137)(29,52,91,111)(30,54,94,154)(32,56,97,157)(34,59,101,161)(36,63,108,170)(38,67)(40,70,119,98)(42,72,122,185)(44,76,129,194)(46,80,126,53)(47,81,138,155)(49,85,143,130)(51,89,150,211)(55,95,93,96)(57,58,99,69)(60,103,165,226)(62,106,168,229)(64,110,172,233)(66,84,77,131)(68,116,175,132)(71,120,182,128)(73,124,187,243)(75,127,192,92)(79,135,158,218)(82,140,203,254)(86,145)(88,148,190,201)(90,152,213,178)(100,118,180,209)(102,163,104,141)(105,167,228,219)(107,151,117,179)(109,171,232,252)(113,174,146,208)(114,176,237,195)(115,142,204,251)(121,184,217,249)(123,186,242,255)(125,189,246,273)(134,198,193,177)(136,199,250,196)(139,202,231,147)(144,206,256,236)(149,210,259,265)(153,214,207,166)(156,191,247,260)(159,220,266,253)(160,181,238,258)(162,224,257,244)(164,215,262,205)(169,230,271,279)(173,235,269,200)(183,240,197,248)(188,245,241,223)(212,261,275,264)(216,263,276,234)(221,267,277,274)(222,268,278,239)(225,227,270,272), (1,2,3,4,6,10,18,32,57)(5,8,14,25,46,81,139,180,238)(7,12,21,38,68,117,99,159,221)(9,16,29,53,93,28,51,90,153)(11,19,34,60,104,87,147,209,258)(13,23,42,73,125,190,91,120,183)(15,27,49,86,146,145,143,204,97)(17,30,54)(20,36,64,111,135,37,66,114,177)(22,40,71,121,148,203,243,214,41)(24,44,77,132,196,249,233,89,151)(26,47,82,141,94,155,216,250,240)(31,55,96)(33,58,100,160,222,232,262,165,227)(35,62,107)(39,69,118,181,239,210,260,78,134)(43,75,128,194,131,176,182,127,193)(45,79,136)(48,84,142,205,255,271,279,152,163)(50,88,149)(52,92,129,174,206,257,273,263,108)(56,98,158,219,265,154,215,246,259)(59,102,164,225,70,76,130,195,248)(61,105,110,140,186,112,173,236,270)(63,109,95,156,217,171,157,72,123)(65,113,175)(67,115,83)(74,126,191,198,185,241,85,144,207)(80,137,200,251,247,218,264,275,199)(101,162,124,188,235,256,187,244,272)(103,166,122)(106,169,167,202,253,274,268,150,212)(116,178,189,172,234,179,161,223,269)(119,133,197)(138,201,252)(170,231,220,267,278,276,230,229,261)(184,211,228,254,224,245,208,237,192)(213,226,242)(266,277,280) ]", "PSU(4,3)^1" ],
  [  13651680, "[ (3,240,239,238,237,236,235,234,233,232,231,230,229,228,227,226,225,224,223,222,221,220,219,218,217,216,215,214,213,212,211,210,209,208,207,206,205,204,203,202,201,200,199,198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,122)(3,56,188)(4,162,81)(5,18,224)(6,147,94)(7,154,86)(8,177,62)(9,106,132)(10,137,100)(11,50,186)(12,125,110)(13,193,41)(14,76,157)(15,49,183)(16,167,64)(17,19,211)(20,226,239)(21,57,169)(22,109,116)(23,123,101)(24,103,120)(25,142,80)(26,191,30)(27,181,39)(28,60,159)(29,70,148)(31,176,40)(32,43,172)(33,225,227)(34,139,74)(35,47,165)(36,173,38)(37,84,126)(42,92,113)(44,108,95)(45,91,111)(46,55,146)(48,130,69)(51,231,203)(52,117,78)(53,218,214)(54,89,104)(58,194,233)(59,115,73)(61,195,229)(63,217,205)(65,83,99)(66,88,93)(67,236,182)(68,213,204)(71,208,206)(72,201,212)(75,187,223)(77,228,180)(79,197,209)(82,240,163)(85,184,216)(87,168,230)(90,237,158)(96,174,215)(97,238,150)(98,189,198)(102,219,164)(105,210,170)(107,234,144)(112,138,235)(114,196,175)(118,160,207)(119,232,134)(121,221,143)(124,141,220)(127,192,166)(128,135,222)(129,185,171)(131,152,202)(133,153,199)(136,200,149)(140,155,190)(145,161,179)(151,156,178) ]", "PSL(2,239)^1" ],
  [  13997280, "[ (3,242,241,240,239,238,237,236,235,234,233,232,231,230,229,228,227,226,225,224,223,222,221,220,219,218,217,216,215,214,213,212,211,210,209,208,207,206,205,204,203,202,201,200,199,198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,123)(3,173,73)(4,33,212)(5,137,107)(6,54,189)(7,153,89)(8,142,99)(9,230,10)(11,167,71)(12,194,43)(13,146,90)(14,124,111)(15,105,129)(16,237,236)(17,56,176)(18,82,149)(19,200,30)(20,174,55)(21,103,125)(22,182,45)(23,61,165)(24,177,48)(25,116,108)(26,145,78)(27,50,172)(28,41,180)(29,184,36)(31,47,171)(32,51,166)(34,213,242)(35,87,127)(37,102,110)(38,134,77)(39,126,84)(40,58,151)(42,67,140)(44,113,92)(46,227,216)(49,86,114)(52,234,203)(53,131,65)(57,192,240)(59,94,96)(60,91,98)(62,217,210)(63,118,68)(64,224,201)(66,205,218)(69,222,198)(70,190,229)(72,226,191)(74,196,219)(75,199,215)(76,88,85)(79,235,175)(80,195,214)(81,185,223)(93,239,157)(95,188,206)(97,164,228)(100,233,156)(101,220,168)(104,238,147)(106,179,204)(109,241,139)(112,208,169)(115,193,181)(117,141,231)(119,159,211)(120,207,162)(121,143,225)(122,232,135)(128,183,178)(130,221,138)(132,160,197)(133,202,154)(136,144,209)(148,155,186)(150,152,187)(158,170,161) ]", "PSL(2,241)^1" ],
  [  14224896, "[ (1,5,15)(2,8,21)(3,12,28)(4,14,31)(6,17,34)(7,20,39)(9,23,41)(10,25,44)(11,27,46)(13,30,48)(16,33,50)(18,36,52)(19,38,54)(22,40,55)(24,43,57)(26,45,58)(29,47,59)(32,49,60)(35,51,61)(37,53,62)(42,56,63), (5,40,53)(8,30,45)(12,23,14)(17,27,43)(20,49,56)(25,38,47)(33,36,51), (1,37,16)(2,24,3)(4,42,35)(6,26,7)(9,22,19)(10,18,11)(13,29,32) ]", "PSL(2,7)^3" ],
  [  14348664, "[ (3,244,243,242,241,240,239,238,237,236,235,234,233,232,231,230,229,228,227,226,225,224,223,222,221,220,219,218,217,216,215,214,213,212,211,210,209,208,207,206,205,204,203,202,201,200,199,198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,4,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5), (1,2,4)(5,56,192)(6,79,168)(7,159,87)(8,177,68)(9,124,121)(10,228,15)(11,13,229)(12,26,215)(14,157,82)(16,42,195)(17,138,98)(18,77,158)(19,122,113)(20,236,238)(21,239,234)(22,194,37)(23,27,203)(24,99,130)(25,31,197)(28,70,155)(29,59,165)(30,88,135)(32,48,173)(33,140,80)(34,223,237)(35,69,149)(36,63,154)(38,112,104)(39,75,139)(40,118,96)(41,105,108)(43,166,44)(45,60,148)(46,222,226)(47,142,64)(49,116,89)(50,51,152)(52,218,224)(53,78,123)(54,207,233)(55,227,212)(57,193,244)(58,93,103)(61,73,120)(62,106,86)(65,74,115)(66,71,117)(67,85,102)(72,241,181)(76,201,217)(81,170,243)(83,206,205)(84,190,220)(90,242,162)(91,172,231)(92,235,167)(94,179,221)(95,186,213)(97,198,199)(100,180,214)(101,189,204)(107,202,185)(109,216,169)(110,174,210)(111,232,151)(114,161,219)(119,150,225)(125,240,128)(126,171,196)(127,230,136)(129,176,188)(131,209,153)(132,178,183)(133,200,160)(134,175,184)(137,211,145)(141,144,208)(143,187,163)(146,156,191)(147,164,182) ]", "PSL(2,243)^1" ],
  [  14679168, "[ (4,5,7,11,19,35,67)(6,9,15,27,51,99,68)(8,13,23,43,83,36,69)(10,17,31,59,115,100,70)(12,21,39,75,20,37,71)(14,25,47,91,52,101,72)(16,29,55,107,84,38,73)(18,33,63,123,116,102,74)(22,41,79,28,53,103,76)(24,45,87,44,85,40,77)(26,49,95,60,117,104,78)(30,57,111,92,54,105,80)(32,61,119,108,86,42,81)(34,65,127,124,118,106,82)(46,89,48,93,56,109,88)(50,97,64,125,120,110,90)(58,113,96,62,121,112,94)(66,129,128,126,122,114,98), (3,128,126,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6,4,129,127,125,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5), (1,2,3)(4,123,9)(5,116,15)(6,67,63)(7,102,27)(8,76,52)(10,129,124)(11,74,51)(12,40,84)(13,22,101)(14,43,79)(16,75,45)(17,128,118)(18,99,19)(20,87,29)(21,77,38)(23,41,72)(24,73,39)(25,83,28)(26,48,62)(30,64,42)(31,126,106)(32,111,120)(33,68,35)(34,115,114)(36,53,47)(37,44,55)(46,113,104)(49,93,121)(50,108,105)(54,90,119)(56,112,95)(57,125,81)(58,117,88)(59,122,82)(60,109,94)(61,92,110)(65,100,98)(66,127,70)(69,103,91)(71,85,107)(78,89,96)(80,97,86) ]", "PSL(2,128)^1" ],
  [  15813000, "[ (3,252,251,250,249,248,247,246,245,244,243,242,241,240,239,238,237,236,235,234,233,232,231,230,229,228,227,226,225,224,223,222,221,220,219,218,217,216,215,214,213,212,211,210,209,208,207,206,205,204,203,202,201,200,199,198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,128)(3,143,113)(4,130,125)(5,171,83)(6,194,59)(7,129,123)(8,33,218)(9,170,80)(10,141,108)(11,227,21)(12,193,54)(13,211,35)(14,23,222)(15,201,43)(16,81,162)(17,182,60)(18,97,144)(19,158,82)(20,215,24)(22,116,121)(25,79,155)(26,46,187)(27,157,75)(28,65,166)(29,245,235)(30,163,66)(31,137,91)(32,89,138)(34,233,242)(36,37,186)(38,223,248)(39,56,164)(40,72,147)(41,236,232)(42,103,114)(44,154,61)(45,243,221)(47,88,124)(48,51,160)(49,100,110)(50,152,57)(52,87,120)(53,67,139)(55,241,213)(58,106,95)(62,250,197)(63,244,202)(64,122,73)(68,107,84)(69,210,230)(70,219,220)(71,77,111)(74,239,196)(76,105,78)(85,251,173)(86,247,176)(90,191,228)(92,200,217)(93,226,190)(94,175,240)(96,205,208)(98,237,174)(99,229,181)(101,177,231)(102,212,195)(104,206,199)(109,184,216)(112,159,238)(115,246,148)(117,189,203)(118,167,224)(119,225,165)(126,252,131)(127,249,133)(132,168,209)(134,192,183)(135,140,234)(136,169,204)(142,153,214)(145,179,185)(146,156,207)(149,188,172)(150,198,161)(151,180,178) ]", "PSL(2,251)^1" ],
  [  16547328, "[ (3,322,109,510,380,159,384,149,48,152,171,14,119,289,5,263,217,337,87,440,122,47,442,297,33,474,228,10,413,199,4,203,192,321,202,104,485,253,89,258,185,26,85,412,8,82,340,453,155,125,213,58,400,66,61,433,182,18,389,320,6,290,309)(7,80,262,463,162,317,288,358,351,98,401,466,100,167,224,132,50,70,38,448,53,146,491,248,140,419,113,40,183,196,12,55,319,512,271,436,410,471,197,175,496,482,178,278,349,106,93,130,71,27,99,249,327,374,210,495,198,75,299,316,22,103,112)(9,467,480,144,501,305,236,408,107,168,173,31,226,52,96,133,15,513,63,246,457,311,154,215,110,46,323,32,509,343,16,335,127,212,476,425,362,493,188,280,286,57,94,97,17,131,28,505,116,372,483,34,261,339,193,86,441,59,25,459,30,451,220)(11,331,379,368,191,189,227,449,478,251,367,445,83,487,325,296,475,244,209,239,77,65,326,421,19,360,184,69,350,382,20,450,383,292,208,306,353,507,396,376,352,370,148,439,444,60,431,273,332,366,143,120,446,41,35,205,301,128,279,156,37,508,484)(13,54,24,101,44,180,29)(21,407,242,255,204,274,84,328,479,114,118,338,81,469,123,443,272,139,393,91,416,158,461,394,359,391,314,136,486,111,39,284,307,365,294,398,150,117,164,45,206,455,62,64,214,43,356,238,430,163,333,267,49,489,422,488,268,234,266,195,73,386,373)(23,315,402,468,502,354,470,405,264,232,170,102,456,51,186,361,74,56,231,259,160,240,254,363,313,424,147,145,503,260,42,434,346,497,302,415,392,423,388,357,283,181,135,95,303,411,138,105,355,385,269,166,381,250,432,275,237,247,304,364,79,494,420)(36,90,293,256,397,151,477,265,499,243,88,458,76,282,218,245,225,68,235,310,437,435,194,498,252,426,281,216,395,219,67,161,417,257,492,179,481,390,438,369,157,490,142,404,342,371,341,126,176,429,504,137,312,318,378,153,403,200,473,169,124,270,187)(72,92,375,427,201,428,190,460,285,174,222,418,506,241,336,465,334,287,291,121,207,324,330,511,344,387,348,295,399,462,134,165,223,221,229,472,308,115,406,108,345,172,500,129,452,447,141,409,414,211,329,177,78,454,298,377,464,233,277,300,230,276,347), (1,2,3,5,9,17,32,60,112,197,318,438,498,198,319,439,505,16,31,58,109)(4,7,13,25,47,88,158,268,329,432,380,395,491,451,291,415,218,138,237,364,474)(6,11,21,39,74,139,239,343,224,350,75,141,242,241,368,476,332,220,344,459,93)(8,15,29,55,104,185,302,423,105,187,304,345,52,98,176,290,357,285,407,150,256)(10,19,36,68,127,132,228,305,426,73,137,236,258,50,94,169,282,331,89,160,170)(12,23,43,82,147,175,289,230,354,417,462,342,424,276,400,495,502,122,81,42,80)(14,27,51,96,172,211,335,403,374,427,475,452,57,108,191,154,262,387,450,485,317)(18,34,64,119,208,190,306,271,366,308,406,238,365,311,178,292,416,441,457,40,76)(20,38,72,135,233,274,33,62,115,202,323,442,301,422,501,143,245,328,449,429,503)(22,41,78,35,66,123,215,48,90,162,193,63,117,205,327,339,359,118,207,330,376)(24,45,85,153,260,386,371,480,455,103,184,247,373,481,477,46,87,156,264,389,487)(26,49,92,166,277,167,279,402,216,209,333,270,396,408,255,383,412,116,203,324,443)(28,53,100,179,294,418,500,303,363,281,173,99,177,280,148,252,379,470,378,483,469)(30,56,106,61,113,199,128,221,196,261,65,121,212,336,251,377,482,437,97,174,287)(37,70,131,227,206,186,136,234,360,159,71,133,229,226,352,286,326,447,488,506,471)(44,83,149,254,382,453,372,265,157,266,392,484,512,272,168,161,195,315,435,297,393)(54,102,182,298,84,151,257,353,181,296,232,358,472,398,494,414,213,249,375,253,165)(59,111,114,200,321,338,456,399,362,458,444,293,188,134,231,356,192,310,430,486,513)(67,125,79,146,250,355,419,142,204,325,445,489,492,288,411,283,401,497,217,341,101)(69,129,223,348,465,508,434,410,496,507,163,273,397,493,243,370,479,120,210,299,420)(77,144,246,337,454,510,309,428,155,86,107,189,307,145,248,124,130,225,351,468,391)(91,164,275,369,322,390,388)(95,171,284,340,384,478,235,361,473,244,320,440,314,433,269,347,463,126,219,183,300)(110,194,313,421,312,431,425,460,349,467,504,405,385,334,409,394,490,448,466,509,214)(140,240,367,381,446,404,499,267,201,152,259,180,295,413,464,278,263,222,346,461,511) ]", "PSU(3,8)^1" ],
  [  16547328, "[ (3,361,249,47,392,98,85,247,12)(4,232,40,58,42,71,152,166,288)(5,264,248,104,260,349,182,269,262)(6,468,463,87,313,140,258,411,106)(7,384,357,146,10,79,178,155,231)(8,402,112,109,363,53,297,303,482)(9,246,127,311,286,339,94,451,31)(11,141,126,65,201,151,148,387,281)(13,24,101)(14,364,197,192,56,316,66,388,70)(15,457,362,493,57,59,25,220,133)(16,52,107,34,476,343,168,86,483)(17,236,97,480,32,467,215,116,154)(18,432,38,263,456,175,485,497,22)(19,334,318,60,115,458,368,277,417)(20,345,498,120,276,342,251,121,293)(21,469,242,158,294,394,62,488,214)(23,210,321,240,100,61,135,512,119)(26,250,224,309,102,436,122,346,198)(27,289,502,358,202,424,183,433,138)(28,96,226,501,425,212,46,441,193)(29,180,54)(30,408,505,63,509,305,280,372,323)(33,355,75,82,470,132,510,145,410)(35,222,219,83,165,504,292,78,499)(36,326,285,435,439,462,142,331,329)(37,229,187,209,377,437,376,241,490)(39,328,455,267,486,373,114,163,416)(41,190,282,445,295,492,383,414,153)(43,204,111,255,391,333,359,356,479)(45,64,195,73,407,139,393,461,164)(48,105,374,199,354,466,213,147,319)(49,307,443,118,489,150,284,338,234)(50,149,95,471,413,315,299,125,254)(51,278,400,302,80,171,275,491,203)(55,380,434,327,474,381,401,340,170)(67,475,409,137,507,428,243,279,348)(68,69,399,438,332,211,169,227,460)(72,312,370,324,88,484,500,161,77)(74,495,253,423,167,389,304,271,217)(76,449,418,257,350,223,378,366,454)(81,386,274,365,91,314,422,206,238)(84,117,398,136,266,268,430,272,123)(89,181,196,320,420,99,337,160,351)(90,184,344,194,273,447,404,189,427)(92,403,128,330,218,296,129,179,353)(93,322,259,162,442,283,419,412,494)(103,290,405,448,440,503,496,185,385)(108,310,306,230,157,301,291,270,325)(110,261,188,131,173,459,144,335,513)(113,453,186,130,159,415,317,228,237)(124,367,511,235,450,452,369,446,375)(134,341,244,177,397,396,174,426,156)(143,233,395,478,465,429,382,308,265)(172,216,191,347,176,360,207,477,431)(200,508,406,245,239,300,481,352,287)(205,336,371,487,472,256,208,464,252)(221,473,379,298,225,421,506,390,444), (3,4,6,10,18,33,61,47,58,87,155,263,82,119,85,152,258,384,485,510,321)(5,8,14,26,48,89,159,104,109,192,309,199,320,228,182,297,66,122,213,337,453)(7,12,22,40,75,140,210,146,249,38,71,132,106,100,178,98,175,288,410,463,512)(9,16,30,32,59,110,193,311,34,63,116,133,131,226,94,168,280,236,362,144,212)(11,20,37,69,128,19,35,65,120,209,332,296,60,83,148,251,376,227,353,368,292)(13,24,44,29,54,101,180)(15,28,52,97,173,286,408,493,501,476,467,335,451,509,25,46,86,154,261,246,372)(17,31,57,107,188,305,425,480,127,220,343,459,323,441,215,339,457,483,513,505,96)(21,39,73,136,234,359,422,158,267,393,430,443,43,81,62,114,45,84,150,255,365)(23,42,79,145,247,313,432,240,166,231,355,361,411,456,135,232,357,470,392,468,497)(27,50,93,167,278,401,496,358,471,162,271,80,55,103,183,299,419,495,491,327,448)(36,67,124,216,200,252,378,435,137,235,176,245,371,76,142,243,369,477,481,256,257)(41,77,143,244,273,325,444,445,370,478,396,189,306,379,383,484,382,156,184,301,421)(49,91,163,272,356,469,64,118,206,328,117,204,294,407,284,386,486,266,391,488,461)(51,95,170,283,405,423,502,302,315,434,494,503,304,424,275,254,381,259,385,74,138)(53,99,70,130,224,349,466,482,351,197,317,436,262,319,112,196,316,113,198,248,374)(56,105,186,303,102,181,264,388,354,415,402,346,420,260,364,147,237,363,250,160,269)(68,126,218,342,458,490,499,438,151,179,293,417,187,219,169,281,403,498,318,437,504)(72,134,230,295,233,344,298,324,177,291,414,465,447,506,500,174,108,190,308,427,221)(78,121,211,334,141,241,129,222,345,460,115,201,229,92,165,276,399,277,387,377,330)(88,157,265,390,397,492,90,161,270,395,473,426,153,194,312,310,429,225,341,282,404)(111,195,314,268,394,489,416,333,139,238,123,214,338,455,479,164,274,398,242,307,373)(125,217,340,289,412,171,185,149,253,380,202,322,203,290,413,389,474,433,442,400,440)(172,285,406,428,472,375,223,347,462,300,348,464,511,454,207,329,287,409,336,452,418)(191,208,331,450,508,350,279,360,205,326,446,239,366,475,431,487,439,367,352,449,507), (1,2,3,5,9,17,32,60,112,197,318,438,498,198,319,439,505,16,31,58,109)(4,7,13,25,47,88,158,268,329,432,380,395,491,451,291,415,218,138,237,364,474)(6,11,21,39,74,139,239,343,224,350,75,141,242,241,368,476,332,220,344,459,93)(8,15,29,55,104,185,302,423,105,187,304,345,52,98,176,290,357,285,407,150,256)(10,19,36,68,127,132,228,305,426,73,137,236,258,50,94,169,282,331,89,160,170)(12,23,43,82,147,175,289,230,354,417,462,342,424,276,400,495,502,122,81,42,80)(14,27,51,96,172,211,335,403,374,427,475,452,57,108,191,154,262,387,450,485,317)(18,34,64,119,208,190,306,271,366,308,406,238,365,311,178,292,416,441,457,40,76)(20,38,72,135,233,274,33,62,115,202,323,442,301,422,501,143,245,328,449,429,503)(22,41,78,35,66,123,215,48,90,162,193,63,117,205,327,339,359,118,207,330,376)(24,45,85,153,260,386,371,480,455,103,184,247,373,481,477,46,87,156,264,389,487)(26,49,92,166,277,167,279,402,216,209,333,270,396,408,255,383,412,116,203,324,443)(28,53,100,179,294,418,500,303,363,281,173,99,177,280,148,252,379,470,378,483,469)(30,56,106,61,113,199,128,221,196,261,65,121,212,336,251,377,482,437,97,174,287)(37,70,131,227,206,186,136,234,360,159,71,133,229,226,352,286,326,447,488,506,471)(44,83,149,254,382,453,372,265,157,266,392,484,512,272,168,161,195,315,435,297,393)(54,102,182,298,84,151,257,353,181,296,232,358,472,398,494,414,213,249,375,253,165)(59,111,114,200,321,338,456,399,362,458,444,293,188,134,231,356,192,310,430,486,513)(67,125,79,146,250,355,419,142,204,325,445,489,492,288,411,283,401,497,217,341,101)(69,129,223,348,465,508,434,410,496,507,163,273,397,493,243,370,479,120,210,299,420)(77,144,246,337,454,510,309,428,155,86,107,189,307,145,248,124,130,225,351,468,391)(91,164,275,369,322,390,388)(95,171,284,340,384,478,235,361,473,244,320,440,314,433,269,347,463,126,219,183,300)(110,194,313,421,312,431,425,460,349,467,504,405,385,334,409,394,490,448,466,509,214)(140,240,367,381,446,404,499,267,201,152,259,180,295,413,464,278,263,222,346,461,511) ]", "PSU(3,8)^1" ],
  [  16547328, "[ (3,55,102)(4,358,354)(5,463,502)(6,93,420)(7,259,109)(8,178,494)(9,226,133)(10,491,237)(11,216,500)(12,304,320)(13,44,180)(14,38,275)(16,34,168)(17,215,480)(18,103,269)(19,257,506)(20,256,308)(21,43,114)(22,302,66)(23,289,99)(24,54,29)(26,75,385)(27,105,152)(28,86,173)(30,32,144)(31,220,459)(33,471,303)(35,243,190)(36,177,209)(37,142,134)(39,158,255)(40,170,453)(41,403,375)(42,400,349)(46,476,261)(47,327,346)(48,512,95)(49,266,486)(50,363,510)(51,192,175)(52,335,501)(53,313,440)(56,321,495)(57,188,339)(58,183,147)(59,94,193)(60,378,221)(61,167,388)(62,359,267)(63,116,110)(64,206,488)(65,176,72)(67,295,83)(68,285,421)(69,235,344)(70,357,149)(71,434,159)(74,337,249)(76,298,368)(77,417,300)(78,208,397)(79,413,197)(80,415,384)(81,430,234)(82,299,402)(84,443,365)(85,401,250)(87,162,160)(88,330,279)(89,98,423)(90,201,326)(91,294,461)(92,507,312)(96,505,107)(97,408,451)(100,254,213)(104,140,424)(106,138,182)(108,296,245)(111,274,373)(112,468,290)(113,247,412)(115,431,265)(117,328,284)(118,272,163)(119,271,364)(120,252,233)(121,367,270)(122,410,503)(123,139,242)(124,427,227)(125,316,231)(126,409,244)(127,457,513)(128,200,291)(129,475,161)(130,361,322)(131,280,236)(132,405,309)(135,433,196)(136,150,422)(137,414,292)(141,331,404)(143,499,454)(145,380,319)(146,283,297)(148,477,324)(151,428,396)(153,211,366)(154,372,286)(155,278,186)(156,281,348)(157,276,446)(164,394,398)(165,487,341)(166,171,262)(169,329,379)(172,273,498)(174,376,435)(179,452,383)(181,258,419)(184,293,207)(185,482,411)(187,472,325)(189,342,347)(191,395,277)(194,387,439)(195,214,268)(198,497,253)(199,210,315)(202,351,240)(203,248,232)(204,356,391)(205,426,222)(212,362,311)(217,436,456)(218,511,445)(219,418,478)(223,382,504)(224,432,389)(225,377,239)(228,288,381)(229,508,473)(230,353,481)(238,455,479)(241,352,390)(246,467,509)(251,371,465)(260,485,496)(263,448,264)(282,460,449)(287,370,318)(301,490,336)(305,343,425)(306,437,464)(307,489,338)(310,345,450)(314,416,333)(317,392,442)(323,483,441)(332,369,447)(334,360,429)(340,466,470)(350,492,399)(355,474,374)(386,469,407)(406,484,458)(438,462,444), (3,4,6,10,18,33,61,47,58,87,155,263,82,119,85,152,258,384,485,510,321)(5,8,14,26,48,89,159,104,109,192,309,199,320,228,182,297,66,122,213,337,453)(7,12,22,40,75,140,210,146,249,38,71,132,106,100,178,98,175,288,410,463,512)(9,16,30,32,59,110,193,311,34,63,116,133,131,226,94,168,280,236,362,144,212)(11,20,37,69,128,19,35,65,120,209,332,296,60,83,148,251,376,227,353,368,292)(13,24,44,29,54,101,180)(15,28,52,97,173,286,408,493,501,476,467,335,451,509,25,46,86,154,261,246,372)(17,31,57,107,188,305,425,480,127,220,343,459,323,441,215,339,457,483,513,505,96)(21,39,73,136,234,359,422,158,267,393,430,443,43,81,62,114,45,84,150,255,365)(23,42,79,145,247,313,432,240,166,231,355,361,411,456,135,232,357,470,392,468,497)(27,50,93,167,278,401,496,358,471,162,271,80,55,103,183,299,419,495,491,327,448)(36,67,124,216,200,252,378,435,137,235,176,245,371,76,142,243,369,477,481,256,257)(41,77,143,244,273,325,444,445,370,478,396,189,306,379,383,484,382,156,184,301,421)(49,91,163,272,356,469,64,118,206,328,117,204,294,407,284,386,486,266,391,488,461)(51,95,170,283,405,423,502,302,315,434,494,503,304,424,275,254,381,259,385,74,138)(53,99,70,130,224,349,466,482,351,197,317,436,262,319,112,196,316,113,198,248,374)(56,105,186,303,102,181,264,388,354,415,402,346,420,260,364,147,237,363,250,160,269)(68,126,218,342,458,490,499,438,151,179,293,417,187,219,169,281,403,498,318,437,504)(72,134,230,295,233,344,298,324,177,291,414,465,447,506,500,174,108,190,308,427,221)(78,121,211,334,141,241,129,222,345,460,115,201,229,92,165,276,399,277,387,377,330)(88,157,265,390,397,492,90,161,270,395,473,426,153,194,312,310,429,225,341,282,404)(111,195,314,268,394,489,416,333,139,238,123,214,338,455,479,164,274,398,242,307,373)(125,217,340,289,412,171,185,149,253,380,202,322,203,290,413,389,474,433,442,400,440)(172,285,406,428,472,375,223,347,462,300,348,464,511,454,207,329,287,409,336,452,418)(191,208,331,450,508,350,279,360,205,326,446,239,366,475,431,487,439,367,352,449,507), (1,2,3,5,9,17,32,60,112,197,318,438,498,198,319,439,505,16,31,58,109)(4,7,13,25,47,88,158,268,329,432,380,395,491,451,291,415,218,138,237,364,474)(6,11,21,39,74,139,239,343,224,350,75,141,242,241,368,476,332,220,344,459,93)(8,15,29,55,104,185,302,423,105,187,304,345,52,98,176,290,357,285,407,150,256)(10,19,36,68,127,132,228,305,426,73,137,236,258,50,94,169,282,331,89,160,170)(12,23,43,82,147,175,289,230,354,417,462,342,424,276,400,495,502,122,81,42,80)(14,27,51,96,172,211,335,403,374,427,475,452,57,108,191,154,262,387,450,485,317)(18,34,64,119,208,190,306,271,366,308,406,238,365,311,178,292,416,441,457,40,76)(20,38,72,135,233,274,33,62,115,202,323,442,301,422,501,143,245,328,449,429,503)(22,41,78,35,66,123,215,48,90,162,193,63,117,205,327,339,359,118,207,330,376)(24,45,85,153,260,386,371,480,455,103,184,247,373,481,477,46,87,156,264,389,487)(26,49,92,166,277,167,279,402,216,209,333,270,396,408,255,383,412,116,203,324,443)(28,53,100,179,294,418,500,303,363,281,173,99,177,280,148,252,379,470,378,483,469)(30,56,106,61,113,199,128,221,196,261,65,121,212,336,251,377,482,437,97,174,287)(37,70,131,227,206,186,136,234,360,159,71,133,229,226,352,286,326,447,488,506,471)(44,83,149,254,382,453,372,265,157,266,392,484,512,272,168,161,195,315,435,297,393)(54,102,182,298,84,151,257,353,181,296,232,358,472,398,494,414,213,249,375,253,165)(59,111,114,200,321,338,456,399,362,458,444,293,188,134,231,356,192,310,430,486,513)(67,125,79,146,250,355,419,142,204,325,445,489,492,288,411,283,401,497,217,341,101)(69,129,223,348,465,508,434,410,496,507,163,273,397,493,243,370,479,120,210,299,420)(77,144,246,337,454,510,309,428,155,86,107,189,307,145,248,124,130,225,351,468,391)(91,164,275,369,322,390,388)(95,171,284,340,384,478,235,361,473,244,320,440,314,433,269,347,463,126,219,183,300)(110,194,313,421,312,431,425,460,349,467,504,405,385,334,409,394,490,448,466,509,214)(140,240,367,381,446,404,499,267,201,152,259,180,295,413,464,278,263,222,346,461,511) ]", "PSU(3,8)^1" ],
  [  16974336, "[ (3,258,257,256,255,254,253,252,251,250,249,248,247,246,245,244,243,242,241,240,239,238,237,236,235,234,233,232,231,230,229,228,227,226,225,224,223,222,221,220,219,218,217,216,215,214,213,212,211,210,209,208,207,206,205,204,203,202,201,200,199,198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,131)(3,83,179)(4,35,226)(5,28,232)(6,206,53)(7,64,194)(8,153,104)(9,240,16)(10,94,161)(11,59,195)(12,118,135)(13,107,145)(14,20,231)(15,91,159)(17,133,115)(18,205,42)(19,174,72)(21,96,148)(22,253,246)(23,49,193)(24,65,176)(25,140,100)(26,134,105)(27,141,97)(29,149,87)(30,234,257)(31,242,248)(32,199,34)(33,89,143)(36,227,258)(37,44,184)(38,71,156)(39,74,152)(40,177,48)(41,151,73)(43,62,160)(45,66,154)(46,167,52)(47,120,98)(50,77,138)(51,130,84)(54,136,75)(55,150,60)(56,256,209)(57,244,220)(58,82,125)(61,123,81)(63,230,228)(67,203,251)(68,198,255)(69,213,239)(70,116,79)(76,90,99)(78,218,225)(80,93,92)(85,222,214)(86,197,238)(88,243,190)(95,216,210)(101,168,252)(102,200,219)(103,171,247)(106,191,224)(108,196,217)(109,254,158)(110,188,223)(111,221,189)(112,207,202)(113,233,175)(114,166,241)(117,155,249)(119,173,229)(121,235,165)(122,237,162)(124,185,212)(126,208,187)(127,144,250)(128,236,157)(129,245,147)(132,211,178)(137,180,204)(139,201,181)(142,215,164)(146,192,183)(163,172,186)(169,182,170) ]", "PSL(2,257)^1" ],
  [  18191184, "[ (3,264,263,262,261,260,259,258,257,256,255,254,253,252,251,250,249,248,247,246,245,244,243,242,241,240,239,238,237,236,235,234,233,232,231,230,229,228,227,226,225,224,223,222,221,220,219,218,217,216,215,214,213,212,211,210,209,208,207,206,205,204,203,202,201,200,199,198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,134)(3,206,62)(4,156,111)(5,144,122)(6,27,238)(7,14,250)(8,24,239)(9,92,170)(10,83,178)(11,123,137)(12,184,75)(13,236,22)(15,210,46)(16,202,53)(17,121,133)(18,254,261)(19,51,201)(20,77,174)(21,172,78)(23,192,56)(25,33,213)(26,208,37)(28,171,72)(29,244,260)(30,241,262)(31,91,149)(32,255,246)(34,188,49)(35,116,120)(36,129,106)(38,74,159)(39,187,45)(40,99,132)(41,102,128)(42,47,182)(43,63,165)(44,114,113)(48,73,150)(50,117,104)(52,89,130)(54,160,57)(55,235,243)(58,253,222)(59,70,142)(60,242,231)(61,85,125)(64,107,100)(65,127,79)(66,252,215)(67,217,249)(68,110,93)(69,115,87)(71,105,95)(76,245,212)(80,234,219)(81,229,223)(82,88,101)(84,256,193)(86,221,226)(90,185,258)(94,191,248)(96,247,190)(97,240,196)(98,176,259)(103,205,225)(108,214,211)(109,194,230)(112,264,157)(118,195,220)(119,177,237)(124,263,146)(126,198,209)(131,145,257)(135,147,251)(136,169,228)(138,179,216)(139,232,162)(140,166,227)(141,203,189)(143,183,207)(148,152,233)(151,218,164)(153,199,181)(154,224,155)(158,200,175)(161,204,168)(163,197,173)(167,180,186) ]", "PSL(2,263)^1" ],
  [  19464840, "[ (3,270,269,268,267,266,265,264,263,262,261,260,259,258,257,256,255,254,253,252,251,250,249,248,247,246,245,244,243,242,241,240,239,238,237,236,235,234,233,232,231,230,229,228,227,226,225,224,223,222,221,220,219,218,217,216,215,214,213,212,211,210,209,208,207,206,205,204,203,202,201,200,199,198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,137)(3,136,138)(4,28,245)(5,197,75)(6,187,84)(7,32,238)(8,66,203)(9,55,213)(10,42,225)(11,160,106)(12,123,142)(13,57,207)(14,169,94)(15,145,117)(16,96,165)(17,159,101)(18,233,26)(19,91,167)(20,177,80)(21,188,68)(22,134,121)(23,88,166)(24,218,35)(25,73,179)(27,39,211)(29,246,270)(30,112,135)(31,172,74)(33,193,51)(34,100,143)(36,242,267)(37,59,181)(38,76,163)(40,192,45)(41,256,248)(43,50,184)(44,144,89)(46,113,118)(47,110,120)(48,150,79)(49,232,264)(52,133,92)(53,146,78)(54,125,98)(56,250,239)(58,155,64)(60,148,69)(61,219,265)(62,116,99)(63,235,247)(65,127,85)(67,217,261)(70,103,104)(71,208,266)(72,83,122)(77,269,199)(81,241,223)(82,234,229)(86,253,206)(87,268,190)(90,224,231)(93,215,237)(95,201,249)(97,254,194)(102,243,200)(105,260,180)(107,183,255)(108,186,251)(109,178,258)(111,198,236)(114,263,168)(115,257,173)(119,216,210)(124,226,195)(126,214,205)(128,221,196)(129,259,157)(130,230,185)(131,174,240)(132,151,262)(139,162,244)(140,252,153)(141,222,182)(147,209,189)(149,220,176)(152,191,202)(154,164,227)(156,161,228)(158,212,175)(170,171,204) ]", "PSL(2,269)^1" ],
  [  19902240, "[ (3,272,271,270,269,268,267,266,265,264,263,262,261,260,259,258,257,256,255,254,253,252,251,250,249,248,247,246,245,244,243,242,241,240,239,238,237,236,235,234,233,232,231,230,229,228,227,226,225,224,223,222,221,220,219,218,217,216,215,214,213,212,211,210,209,208,207,206,205,204,203,202,201,200,199,198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5,4), (1,2,138)(3,254,22)(4,266,9)(5,260,14)(6,38,235)(7,164,108)(8,185,86)(10,272,267)(11,69,199)(12,200,67)(13,223,43)(15,46,218)(16,271,262)(17,184,78)(18,99,162)(19,150,110)(20,62,197)(21,101,157)(23,35,221)(24,146,109)(25,143,111)(26,72,181)(27,186,66)(28,201,50)(29,121,129)(30,44,205)(31,125,123)(32,132,115)(33,118,128)(34,163,82)(36,182,61)(37,139,103)(39,48,192)(40,85,154)(41,238,270)(42,68,169)(45,136,98)(47,152,80)(49,51,179)(52,57,170)(53,263,233)(54,142,83)(55,241,253)(56,63,160)(58,230,261)(59,120,100)(60,131,88)(64,141,74)(65,87,127)(70,105,104)(71,232,246)(73,89,117)(75,248,226)(76,264,209)(77,207,265)(79,214,256)(81,96,102)(84,228,237)(90,249,210)(91,268,190)(92,259,198)(94,240,215)(95,204,250)(97,225,227)(106,219,224)(107,208,234)(112,269,168)(113,242,194)(114,177,258)(116,213,220)(119,175,255)(122,191,236)(124,229,196)(126,257,166)(130,252,167)(133,251,165)(134,222,193)(135,212,202)(137,239,173)(140,231,178)(144,244,161)(145,216,188)(147,155,247)(148,158,243)(149,189,211)(151,245,153)(156,217,176)(159,187,203)(171,206,172)(174,180,195) ]", "PSL(2,271)^1" ],
  [  19998720, "[ (1,32)(2,33)(3,34)(4,35)(5,36)(6,37)(7,38)(8,39)(9,40)(10,41)(11,42)(12,43)(13,44)(14,45)(15,46)(16,47)(17,48)(18,49)(19,50)(20,51)(21,52)(22,53)(23,54)(24,55)(25,56)(26,57)(27,58)(28,59)(29,60)(30,61)(31,62), (16,24)(17,25)(18,26)(19,27)(20,28)(21,29)(22,30)(23,31)(39,55)(40,56)(41,57)(42,58)(43,59)(44,60)(45,61)(46,62), (1,2,4,8,16)(3,6,12,24,17)(5,10,20,9,18)(7,14,28,25,19)(11,22,13,26,21)(15,30,29,27,23)(32,33,35,39,47)(34,37,43,55,48)(36,41,51,40,49)(38,45,59,56,50)(42,53,44,57,52)(46,61,60,58,54) ]", "PSL(5,2)^1" ],
  [  23970816, "[ (1,19,10,22,5,32,16,25,13,24,9,26,11,23,7,28,18,31,12,34,17,35,8,36,2,21,6,29,3,33,14,27)(4,20)(15,30), (3,5,10,18,6,7,15,12,11,16,4,9,14,13,8,17)(21,25,22,35,24,34,26,36,29,31,28,30,32,23,33,27), (2,10,13,9,6,7,5,3)(8,18,15,11,14,16,17,12) ]", "PSL(2,17)^2" ],
  [  23970816, "[ (1,19,10,35,8,29,3,27)(2,25,13,34,17,24,9,36)(4,20)(5,23,7,30,15,32,16,22)(6,31,12,26,11,33,14,21)(18,28), (3,5,10,18,6,7,15,12,11,16,4,9,14,13,8,17)(21,25,22,35,24,34,26,36,29,31,28,30,32,23,33,27), (20,33,30,27,22,23,36,29)(24,31,25,35,28,26,32,34) ]", "PSL(2,17)^2" ],
  [  24261120, "[ (15,16)(17,20)(18,22)(19,21)(23,32)(24,34)(25,33)(26,38)(27,40)(28,39)(29,35)(30,37)(31,36)(55,56)(57,60)(58,62)(59,61)(63,72)(64,74)(65,73)(66,78)(67,80)(68,79)(69,75)(70,77)(71,76), (1,41)(2,42)(3,43)(4,44)(5,45)(6,46)(7,47)(8,48)(9,49)(10,50)(11,51)(12,52)(13,53)(14,54)(15,55)(16,56)(17,57)(18,58)(19,59)(20,60)(21,61)(22,62)(23,63)(24,64)(25,65)(26,66)(27,67)(28,68)(29,69)(30,70)(31,71)(32,72)(33,73)(34,74)(35,75)(36,76)(37,77)(38,78)(39,79)(40,80), (14,23,32)(15,24,33)(16,25,34)(17,26,35)(18,27,36)(19,28,37)(20,29,38)(21,30,39)(22,31,40)(45,72,63)(46,74,64)(47,73,65)(48,78,66)(49,80,67)(50,79,68)(51,75,69)(52,77,70)(53,76,71), (1,2,5,14)(3,8,23,16)(4,11,32,15)(6,17,7,20)(9,26,25,22)(10,29,34,18)(12,35,33,21)(13,38,24,19)(27,28,31,40)(30,37,36,39)(41,42,45,54)(43,48,63,56)(44,51,72,55)(46,57,47,60)(49,66,65,62)(50,69,74,58)(52,75,73,61)(53,78,64,59)(67,68,71,80)(70,77,76,79) ]", "PSL(4,3)^1" ],
  [  25401600, "[ (1,8,7,9,6,10,5,11,4,12,3,13,2,14), (6,7)(13,14), (8,9,10,11,12,13,14) ]", "A7^2" ],
  [  25401600, "[ (1,8,7,9,6,10,5,11,4,12,3,14)(2,13), (6,7)(13,14), (2,4,3) ]", "A7^2" ],
  [  26127360, "[ (4,268,5,278,7,239,11,222)(6,254,20,52,15,140,61,111)(8,64,33,88,22,172,41,190)(9,82,35,91,26,203,12,29)(10,165,106,78,18,60,229,25)(13,110,23,148,39,233,19,201)(14,259,72,51,43,149,185,150)(16,272,167,240,50,227,219,248)(17,226,168,133,31,103,62,45)(21,129,48,245,65,44,137,223)(24,265,122,89,74,210,42,211)(27,188,37,194,83,241,112,76)(28,225,228,197,87,270,105,183)(30,244,275,71,94,224,212,182)(32,273,151,184,97,189,179,249)(34,216,70,171,101,276,98,252)(36,198,81,214,108,177,155,166)(40,109,59,263,119,232,161,234)(46,205,255,136,126,215,186,250)(47,153,63,193,138,207,170,134)(49,77,174,269,143,66,208,173)(53,262,242,196,80,164,123,199)(54,162,264,120,154,257,261,128)(55,73,230,127,93,187,279,92)(56,125,117,217,157,246,107,121)(57,231,69,202,99,139,58,147)(68,237,142,206,175,114,251,236)(75,95,124,271,192,96,243,169)(79,191,141,178,158,260,163,152)(84,113,235,85,131,146,200,130)(86,145)(90,218,156,104,213,135,247,102)(100,253,118,159,180,220,209,266)(115,144,116,195,204,256,132,176)(160,277,258,267,238,221,181,274), (5,11)(8,19)(9,26)(13,41)(14,34)(16,47)(17,31)(20,61)(22,23)(24,161)(25,60)(28,155)(29,82)(30,55)(33,39)(36,105)(37,112)(40,42)(43,101)(44,223)(45,226)(46,104)(48,137)(50,138)(51,216)(52,254)(53,141)(54,96)(56,157)(58,69)(59,74)(63,219)(64,110)(66,173)(70,185)(71,73)(72,98)(75,162)(76,241)(77,269)(78,165)(79,242)(80,163)(81,87)(84,200)(85,130)(88,201)(89,234)(90,250)(91,203)(92,224)(93,94)(95,154)(100,118)(102,126)(103,133)(106,229)(108,228)(109,265)(111,140)(114,236)(116,132)(119,122)(120,243)(121,125)(123,158)(124,128)(127,244)(129,245)(131,235)(134,227)(135,186)(136,213)(139,147)(142,251)(144,195)(148,190)(149,252)(150,276)(151,179)(152,199)(153,240)(156,215)(160,181)(164,191)(166,197)(167,170)(169,261)(171,259)(172,233)(174,208)(176,256)(177,270)(178,196)(180,209)(182,187)(183,214)(184,273)(188,194)(189,249)(192,257)(193,272)(198,225)(202,231)(205,247)(206,237)(207,248)(210,232)(211,263)(212,230)(217,246)(218,255)(220,253)(222,239)(238,258)(260,262)(264,271)(267,274)(268,278)(275,279), (6,12)(8,19)(9,20)(10,62)(13,33)(14,42)(15,35)(16,105)(17,106)(18,168)(21,27)(22,23)(24,72)(25,103)(26,61)(28,167)(29,254)(30,261)(31,229)(32,107)(34,40)(36,47)(37,48)(38,67)(39,41)(43,122)(44,241)(45,165)(46,242)(49,113)(50,228)(51,265)(52,82)(53,186)(54,275)(55,169)(56,151)(57,99)(58,69)(59,70)(60,133)(63,81)(64,201)(65,83)(66,200)(68,115)(71,162)(73,75)(74,185)(76,223)(77,235)(78,226)(79,104)(80,255)(84,173)(85,174)(86,145)(87,219)(88,110)(89,149)(90,260)(91,140)(92,243)(93,271)(94,264)(95,230)(96,279)(97,117)(98,161)(101,119)(102,158)(108,138)(109,216)(111,203)(112,137)(114,256)(116,142)(120,224)(121,273)(123,126)(124,127)(125,184)(128,244)(129,188)(130,208)(131,269)(132,251)(134,166)(135,141)(136,164)(139,231)(143,146)(144,237)(147,202)(148,172)(150,210)(152,156)(153,198)(154,212)(155,170)(157,179)(160,238)(163,218)(171,263)(175,204)(176,236)(177,207)(178,247)(181,258)(182,257)(183,272)(187,192)(189,217)(190,233)(191,213)(193,214)(194,245)(195,206)(196,205)(197,227)(199,215)(211,259)(221,277)(225,240)(232,276)(234,252)(246,249)(248,270)(250,262)(267,274), (1,2,3,4,6,10,18,32,57)(5,8,14,25,46,81,139,180,238)(7,12,21,38,68,117,99,159,221)(9,16,29,53,93,28,51,90,153)(11,19,34,60,104,87,147,209,258)(13,23,42,73,125,190,91,120,183)(15,27,49,86,146,145,143,204,97)(17,30,54)(20,36,64,111,135,37,66,114,177)(22,40,71,121,148,203,243,214,41)(24,44,77,132,196,249,233,89,151)(26,47,82,141,94,155,216,250,240)(31,55,96)(33,58,100,160,222,232,262,165,227)(35,62,107)(39,69,118,181,239,210,260,78,134)(43,75,128,194,131,176,182,127,193)(45,79,136)(48,84,142,205,255,271,279,152,163)(50,88,149)(52,92,129,174,206,257,273,263,108)(56,98,158,219,265,154,215,246,259)(59,102,164,225,70,76,130,195,248)(61,105,110,140,186,112,173,236,270)(63,109,95,156,217,171,157,72,123)(65,113,175)(67,115,83)(74,126,191,198,185,241,85,144,207)(80,137,200,251,247,218,264,275,199)(101,162,124,188,235,256,187,244,272)(103,166,122)(106,169,167,202,253,274,268,150,212)(116,178,189,172,234,179,161,223,269)(119,133,197)(138,201,252)(170,231,220,267,278,276,230,229,261)(184,211,228,254,224,245,208,237,192)(213,226,242)(266,277,280) ]", "PSU(4,3)^1" ],
  [  28449792, "[ (1,5,15)(2,8,21)(3,12,28)(4,14,31)(6,17,34)(7,20,39)(9,23,41)(10,25,44)(11,27,46)(13,30,48)(16,33,50)(18,36,52)(19,38,54)(22,40,55)(24,43,57)(26,45,58)(29,47,59)(32,49,60)(35,51,61)(37,53,62)(42,56,63), (2,26,6,32)(3,4,35,19,10,11,13,7)(8,45)(9,42,24,29,18,22,37,16)(12,25)(14,38)(17,49)(20,27)(21,58)(23,47)(28,44)(31,54)(33,36)(34,60)(39,46)(40,53)(41,59)(43,56)(50,52)(55,62)(57,63), (1,37,16)(2,24,3)(4,42,35)(6,26,7)(9,22,19)(10,18,11)(13,29,32) ]", "PSL(2,7)^3" ],
  [  28449792, "[ (2,26,6,32)(3,4,35,19,10,11,13,7)(5,15)(8,58)(9,42,24,29,18,22,37,16)(12,44)(14,54)(17,60)(20,46)(21,45)(23,59)(25,28)(27,39)(30,48)(31,38)(33,52)(34,49)(36,50)(40,62)(41,47)(43,63)(51,61)(53,55)(56,57), (1,5,15)(2,8,21)(3,12,28)(4,14,31)(6,17,34)(7,20,39)(9,23,41)(10,25,44)(11,27,46)(13,30,48)(16,33,50)(18,36,52)(19,38,54)(22,40,55)(24,43,57)(26,45,58)(29,47,59)(32,49,60)(35,51,61)(37,53,62)(42,56,63), (5,40,53)(8,30,45)(12,23,14)(17,27,43)(20,49,56)(25,38,47)(33,36,51) ]", "PSL(2,7)^3" ],
  [  28449792, "[ (5,15)(8,21)(12,28)(14,31)(17,34)(20,39)(23,41)(25,44)(27,46)(30,48)(33,50)(36,52)(38,54)(40,55)(43,57)(45,58)(47,59)(49,60)(51,61)(53,62)(56,63), (1,5,15)(2,8,21)(3,12,28)(4,14,31)(6,17,34)(7,20,39)(9,23,41)(10,25,44)(11,27,46)(13,30,48)(16,33,50)(18,36,52)(19,38,54)(22,40,55)(24,43,57)(26,45,58)(29,47,59)(32,49,60)(35,51,61)(37,53,62)(42,56,63), (5,40,53)(8,30,45)(12,23,14)(17,27,43)(20,49,56)(25,38,47)(33,36,51), (1,37,16)(2,24,3)(4,42,35)(6,26,7)(9,22,19)(10,18,11)(13,29,32) ]", "PSL(2,7)^3" ],
  [  33094656, "[ (3,371,152,378,87,36,10,243,485,235,82,216,61,481)(4,257,258,142,155,137,18,124,510,477,119,245,47,252)(5,161,297,157,192,429,26,473,213,397,320,282,159,194)(6,435,384,67,263,369,33,176,321,200,85,256,58,76)(7,141,98,377,38,92,40,222,410,121,106,399,210,115)(8,310,66,395,309,390,48,341,337,153,228,90,104,88)(9,284,168,206,63,163,32,266,362,204,131,469,193,461)(11,237,251,402,209,102,69,160,353,260,60,388,35,105)(12,229,175,129,71,78,75,276,463,460,100,334,146,387)(13,54,24,101,44,180,29)(14,265,122,225,199,426,89,492,453,404,182,312,109,270)(15,365,46,62,476,267,97,73,261,84,451,443,408,359)(16,91,280,486,116,117,59,356,144,488,226,407,311,118)(17,274,339,123,220,394,107,307,513,455,323,333,425,195)(19,364,292,354,148,186,120,363,37,346,227,181,296,269)(20,303,376,250,332,420,128,264,368,56,83,147,65,415)(21,52,114,154,393,335,136,286,150,372,43,25,422,501)(22,330,288,165,132,345,140,211,512,277,178,201,249,241)(23,379,232,445,231,77,145,382,392,396,411,273,432,301)(27,230,299,190,162,465,167,344,491,221,55,500,496,177)(28,158,86,39,467,45,173,430,246,234,509,255,493,81)(30,328,236,272,133,391,110,294,212,64,94,49,34,386)(31,268,457,242,343,338,188,416,505,111,441,164,480,238)(41,357,484,355,478,247,244,468,184,456,306,240,444,42)(50,414,419,233,271,427,278,506,327,324,103,134,358,108)(51,439,254,446,434,508,283,449,385,475,304,360,502,208)(53,287,196,348,197,472,130,452,198,454,262,347,466,285)(57,214,483,489,459,373,305,479,96,139,215,314,127,398)(68,253,281,340,179,433,342,322,318,171,187,440,499,413)(70,464,113,375,436,418,349,207,374,462,112,406,351,409)(72,448,174,183,291,471,295,93,308,495,447,80,298,401)(74,191,424,487,302,326,95,450,381,352,494,366,405,279)(79,370,470,143,361,156,313,189,497,325,135,421,166,383)(99,428,316,336,317,511,224,223,248,172,319,329,482,300)(125,169,389,151,380,218,289,498,442,417,203,490,185,504)(126,474,403,202,293,412,458,400,437,290,219,149,438,217)(138,205,275,331,315,367,170,239,259,350,503,507,423,431), (3,322,109,510,380,159,384,149,48,152,171,14,119,289,5,263,217,337,87,440,122,47,442,297,33,474,228,10,413,199,4,203,192,321,202,104,485,253,89,258,185,26,85,412,8,82,340,453,155,125,213,58,400,66,61,433,182,18,389,320,6,290,309)(7,80,262,463,162,317,288,358,351,98,401,466,100,167,224,132,50,70,38,448,53,146,491,248,140,419,113,40,183,196,12,55,319,512,271,436,410,471,197,175,496,482,178,278,349,106,93,130,71,27,99,249,327,374,210,495,198,75,299,316,22,103,112)(9,467,480,144,501,305,236,408,107,168,173,31,226,52,96,133,15,513,63,246,457,311,154,215,110,46,323,32,509,343,16,335,127,212,476,425,362,493,188,280,286,57,94,97,17,131,28,505,116,372,483,34,261,339,193,86,441,59,25,459,30,451,220)(11,331,379,368,191,189,227,449,478,251,367,445,83,487,325,296,475,244,209,239,77,65,326,421,19,360,184,69,350,382,20,450,383,292,208,306,353,507,396,376,352,370,148,439,444,60,431,273,332,366,143,120,446,41,35,205,301,128,279,156,37,508,484)(13,54,24,101,44,180,29)(21,407,242,255,204,274,84,328,479,114,118,338,81,469,123,443,272,139,393,91,416,158,461,394,359,391,314,136,486,111,39,284,307,365,294,398,150,117,164,45,206,455,62,64,214,43,356,238,430,163,333,267,49,489,422,488,268,234,266,195,73,386,373)(23,315,402,468,502,354,470,405,264,232,170,102,456,51,186,361,74,56,231,259,160,240,254,363,313,424,147,145,503,260,42,434,346,497,302,415,392,423,388,357,283,181,135,95,303,411,138,105,355,385,269,166,381,250,432,275,237,247,304,364,79,494,420)(36,90,293,256,397,151,477,265,499,243,88,458,76,282,218,245,225,68,235,310,437,435,194,498,252,426,281,216,395,219,67,161,417,257,492,179,481,390,438,369,157,490,142,404,342,371,341,126,176,429,504,137,312,318,378,153,403,200,473,169,124,270,187)(72,92,375,427,201,428,190,460,285,174,222,418,506,241,336,465,334,287,291,121,207,324,330,511,344,387,348,295,399,462,134,165,223,221,229,472,308,115,406,108,345,172,500,129,452,447,141,409,414,211,329,177,78,454,298,377,464,233,277,300,230,276,347), (1,2,3,5,9,17,32,60,112,197,318,438,498,198,319,439,505,16,31,58,109)(4,7,13,25,47,88,158,268,329,432,380,395,491,451,291,415,218,138,237,364,474)(6,11,21,39,74,139,239,343,224,350,75,141,242,241,368,476,332,220,344,459,93)(8,15,29,55,104,185,302,423,105,187,304,345,52,98,176,290,357,285,407,150,256)(10,19,36,68,127,132,228,305,426,73,137,236,258,50,94,169,282,331,89,160,170)(12,23,43,82,147,175,289,230,354,417,462,342,424,276,400,495,502,122,81,42,80)(14,27,51,96,172,211,335,403,374,427,475,452,57,108,191,154,262,387,450,485,317)(18,34,64,119,208,190,306,271,366,308,406,238,365,311,178,292,416,441,457,40,76)(20,38,72,135,233,274,33,62,115,202,323,442,301,422,501,143,245,328,449,429,503)(22,41,78,35,66,123,215,48,90,162,193,63,117,205,327,339,359,118,207,330,376)(24,45,85,153,260,386,371,480,455,103,184,247,373,481,477,46,87,156,264,389,487)(26,49,92,166,277,167,279,402,216,209,333,270,396,408,255,383,412,116,203,324,443)(28,53,100,179,294,418,500,303,363,281,173,99,177,280,148,252,379,470,378,483,469)(30,56,106,61,113,199,128,221,196,261,65,121,212,336,251,377,482,437,97,174,287)(37,70,131,227,206,186,136,234,360,159,71,133,229,226,352,286,326,447,488,506,471)(44,83,149,254,382,453,372,265,157,266,392,484,512,272,168,161,195,315,435,297,393)(54,102,182,298,84,151,257,353,181,296,232,358,472,398,494,414,213,249,375,253,165)(59,111,114,200,321,338,456,399,362,458,444,293,188,134,231,356,192,310,430,486,513)(67,125,79,146,250,355,419,142,204,325,445,489,492,288,411,283,401,497,217,341,101)(69,129,223,348,465,508,434,410,496,507,163,273,397,493,243,370,479,120,210,299,420)(77,144,246,337,454,510,309,428,155,86,107,189,307,145,248,124,130,225,351,468,391)(91,164,275,369,322,390,388)(95,171,284,340,384,478,235,361,473,244,320,440,314,433,269,347,463,126,219,183,300)(110,194,313,421,312,431,425,460,349,467,504,405,385,334,409,394,490,448,466,509,214)(140,240,367,381,446,404,499,267,201,152,259,180,295,413,464,278,263,222,346,461,511) ]", "PSU(3,8)^1" ],
  [  33094656, "[ (3,371,152,378,87,36,10,243,485,235,82,216,61,481)(4,257,258,142,155,137,18,124,510,477,119,245,47,252)(5,161,297,157,192,429,26,473,213,397,320,282,159,194)(6,435,384,67,263,369,33,176,321,200,85,256,58,76)(7,141,98,377,38,92,40,222,410,121,106,399,210,115)(8,310,66,395,309,390,48,341,337,153,228,90,104,88)(9,284,168,206,63,163,32,266,362,204,131,469,193,461)(11,237,251,402,209,102,69,160,353,260,60,388,35,105)(12,229,175,129,71,78,75,276,463,460,100,334,146,387)(13,54,24,101,44,180,29)(14,265,122,225,199,426,89,492,453,404,182,312,109,270)(15,365,46,62,476,267,97,73,261,84,451,443,408,359)(16,91,280,486,116,117,59,356,144,488,226,407,311,118)(17,274,339,123,220,394,107,307,513,455,323,333,425,195)(19,364,292,354,148,186,120,363,37,346,227,181,296,269)(20,303,376,250,332,420,128,264,368,56,83,147,65,415)(21,52,114,154,393,335,136,286,150,372,43,25,422,501)(22,330,288,165,132,345,140,211,512,277,178,201,249,241)(23,379,232,445,231,77,145,382,392,396,411,273,432,301)(27,230,299,190,162,465,167,344,491,221,55,500,496,177)(28,158,86,39,467,45,173,430,246,234,509,255,493,81)(30,328,236,272,133,391,110,294,212,64,94,49,34,386)(31,268,457,242,343,338,188,416,505,111,441,164,480,238)(41,357,484,355,478,247,244,468,184,456,306,240,444,42)(50,414,419,233,271,427,278,506,327,324,103,134,358,108)(51,439,254,446,434,508,283,449,385,475,304,360,502,208)(53,287,196,348,197,472,130,452,198,454,262,347,466,285)(57,214,483,489,459,373,305,479,96,139,215,314,127,398)(68,253,281,340,179,433,342,322,318,171,187,440,499,413)(70,464,113,375,436,418,349,207,374,462,112,406,351,409)(72,448,174,183,291,471,295,93,308,495,447,80,298,401)(74,191,424,487,302,326,95,450,381,352,494,366,405,279)(79,370,470,143,361,156,313,189,497,325,135,421,166,383)(99,428,316,336,317,511,224,223,248,172,319,329,482,300)(125,169,389,151,380,218,289,498,442,417,203,490,185,504)(126,474,403,202,293,412,458,400,437,290,219,149,438,217)(138,205,275,331,315,367,170,239,259,350,503,507,423,431), (3,55,102)(4,358,354)(5,463,502)(6,93,420)(7,259,109)(8,178,494)(9,226,133)(10,491,237)(11,216,500)(12,304,320)(13,44,180)(14,38,275)(16,34,168)(17,215,480)(18,103,269)(19,257,506)(20,256,308)(21,43,114)(22,302,66)(23,289,99)(24,54,29)(26,75,385)(27,105,152)(28,86,173)(30,32,144)(31,220,459)(33,471,303)(35,243,190)(36,177,209)(37,142,134)(39,158,255)(40,170,453)(41,403,375)(42,400,349)(46,476,261)(47,327,346)(48,512,95)(49,266,486)(50,363,510)(51,192,175)(52,335,501)(53,313,440)(56,321,495)(57,188,339)(58,183,147)(59,94,193)(60,378,221)(61,167,388)(62,359,267)(63,116,110)(64,206,488)(65,176,72)(67,295,83)(68,285,421)(69,235,344)(70,357,149)(71,434,159)(74,337,249)(76,298,368)(77,417,300)(78,208,397)(79,413,197)(80,415,384)(81,430,234)(82,299,402)(84,443,365)(85,401,250)(87,162,160)(88,330,279)(89,98,423)(90,201,326)(91,294,461)(92,507,312)(96,505,107)(97,408,451)(100,254,213)(104,140,424)(106,138,182)(108,296,245)(111,274,373)(112,468,290)(113,247,412)(115,431,265)(117,328,284)(118,272,163)(119,271,364)(120,252,233)(121,367,270)(122,410,503)(123,139,242)(124,427,227)(125,316,231)(126,409,244)(127,457,513)(128,200,291)(129,475,161)(130,361,322)(131,280,236)(132,405,309)(135,433,196)(136,150,422)(137,414,292)(141,331,404)(143,499,454)(145,380,319)(146,283,297)(148,477,324)(151,428,396)(153,211,366)(154,372,286)(155,278,186)(156,281,348)(157,276,446)(164,394,398)(165,487,341)(166,171,262)(169,329,379)(172,273,498)(174,376,435)(179,452,383)(181,258,419)(184,293,207)(185,482,411)(187,472,325)(189,342,347)(191,395,277)(194,387,439)(195,214,268)(198,497,253)(199,210,315)(202,351,240)(203,248,232)(204,356,391)(205,426,222)(212,362,311)(217,436,456)(218,511,445)(219,418,478)(223,382,504)(224,432,389)(225,377,239)(228,288,381)(229,508,473)(230,353,481)(238,455,479)(241,352,390)(246,467,509)(251,371,465)(260,485,496)(263,448,264)(282,460,449)(287,370,318)(301,490,336)(305,343,425)(306,437,464)(307,489,338)(310,345,450)(314,416,333)(317,392,442)(323,483,441)(332,369,447)(334,360,429)(340,466,470)(350,492,399)(355,474,374)(386,469,407)(406,484,458)(438,462,444), (3,4,6,10,18,33,61,47,58,87,155,263,82,119,85,152,258,384,485,510,321)(5,8,14,26,48,89,159,104,109,192,309,199,320,228,182,297,66,122,213,337,453)(7,12,22,40,75,140,210,146,249,38,71,132,106,100,178,98,175,288,410,463,512)(9,16,30,32,59,110,193,311,34,63,116,133,131,226,94,168,280,236,362,144,212)(11,20,37,69,128,19,35,65,120,209,332,296,60,83,148,251,376,227,353,368,292)(13,24,44,29,54,101,180)(15,28,52,97,173,286,408,493,501,476,467,335,451,509,25,46,86,154,261,246,372)(17,31,57,107,188,305,425,480,127,220,343,459,323,441,215,339,457,483,513,505,96)(21,39,73,136,234,359,422,158,267,393,430,443,43,81,62,114,45,84,150,255,365)(23,42,79,145,247,313,432,240,166,231,355,361,411,456,135,232,357,470,392,468,497)(27,50,93,167,278,401,496,358,471,162,271,80,55,103,183,299,419,495,491,327,448)(36,67,124,216,200,252,378,435,137,235,176,245,371,76,142,243,369,477,481,256,257)(41,77,143,244,273,325,444,445,370,478,396,189,306,379,383,484,382,156,184,301,421)(49,91,163,272,356,469,64,118,206,328,117,204,294,407,284,386,486,266,391,488,461)(51,95,170,283,405,423,502,302,315,434,494,503,304,424,275,254,381,259,385,74,138)(53,99,70,130,224,349,466,482,351,197,317,436,262,319,112,196,316,113,198,248,374)(56,105,186,303,102,181,264,388,354,415,402,346,420,260,364,147,237,363,250,160,269)(68,126,218,342,458,490,499,438,151,179,293,417,187,219,169,281,403,498,318,437,504)(72,134,230,295,233,344,298,324,177,291,414,465,447,506,500,174,108,190,308,427,221)(78,121,211,334,141,241,129,222,345,460,115,201,229,92,165,276,399,277,387,377,330)(88,157,265,390,397,492,90,161,270,395,473,426,153,194,312,310,429,225,341,282,404)(111,195,314,268,394,489,416,333,139,238,123,214,338,455,479,164,274,398,242,307,373)(125,217,340,289,412,171,185,149,253,380,202,322,203,290,413,389,474,433,442,400,440)(172,285,406,428,472,375,223,347,462,300,348,464,511,454,207,329,287,409,336,452,418)(191,208,331,450,508,350,279,360,205,326,446,239,366,475,431,487,439,367,352,449,507), (1,2,3,5,9,17,32,60,112,197,318,438,498,198,319,439,505,16,31,58,109)(4,7,13,25,47,88,158,268,329,432,380,395,491,451,291,415,218,138,237,364,474)(6,11,21,39,74,139,239,343,224,350,75,141,242,241,368,476,332,220,344,459,93)(8,15,29,55,104,185,302,423,105,187,304,345,52,98,176,290,357,285,407,150,256)(10,19,36,68,127,132,228,305,426,73,137,236,258,50,94,169,282,331,89,160,170)(12,23,43,82,147,175,289,230,354,417,462,342,424,276,400,495,502,122,81,42,80)(14,27,51,96,172,211,335,403,374,427,475,452,57,108,191,154,262,387,450,485,317)(18,34,64,119,208,190,306,271,366,308,406,238,365,311,178,292,416,441,457,40,76)(20,38,72,135,233,274,33,62,115,202,323,442,301,422,501,143,245,328,449,429,503)(22,41,78,35,66,123,215,48,90,162,193,63,117,205,327,339,359,118,207,330,376)(24,45,85,153,260,386,371,480,455,103,184,247,373,481,477,46,87,156,264,389,487)(26,49,92,166,277,167,279,402,216,209,333,270,396,408,255,383,412,116,203,324,443)(28,53,100,179,294,418,500,303,363,281,173,99,177,280,148,252,379,470,378,483,469)(30,56,106,61,113,199,128,221,196,261,65,121,212,336,251,377,482,437,97,174,287)(37,70,131,227,206,186,136,234,360,159,71,133,229,226,352,286,326,447,488,506,471)(44,83,149,254,382,453,372,265,157,266,392,484,512,272,168,161,195,315,435,297,393)(54,102,182,298,84,151,257,353,181,296,232,358,472,398,494,414,213,249,375,253,165)(59,111,114,200,321,338,456,399,362,458,444,293,188,134,231,356,192,310,430,486,513)(67,125,79,146,250,355,419,142,204,325,445,489,492,288,411,283,401,497,217,341,101)(69,129,223,348,465,508,434,410,496,507,163,273,397,493,243,370,479,120,210,299,420)(77,144,246,337,454,510,309,428,155,86,107,189,307,145,248,124,130,225,351,468,391)(91,164,275,369,322,390,388)(95,171,284,340,384,478,235,361,473,244,320,440,314,433,269,347,463,126,219,183,300)(110,194,313,421,312,431,425,460,349,467,504,405,385,334,409,394,490,448,466,509,214)(140,240,367,381,446,404,499,267,201,152,259,180,295,413,464,278,263,222,346,461,511) ]", "PSU(3,8)^1" ],
  [  35871660, "[ (5,7,13,31,85)(6,10,22,58,165)(8,16,40,112,86)(9,19,49,138,166)(11,25,67,192,87)(12,28,76,219,167)(14,34,94,32,88)(15,37,103,59,168)(17,43,121,113,89)(18,46,129,139,169)(20,52,147,193,90)(21,55,156,220,170)(23,61,174,33,91)(24,64,183,60,171)(26,70,201,114,92)(27,73,210,140,172)(29,79,228,194,93)(30,82,237,221,173)(35,97,41,115,95)(36,100,50,141,175)(38,106,68,195,96)(39,109,77,222,176)(42,118,104,62,177)(44,124,122,116,98)(45,126,130,142,178)(47,132,148,196,99)(48,135,157,223,179)(51,144,184,63,180)(53,150,202,117,101)(54,153,211,143,181)(56,159,229,197,102)(57,162,238,224,182)(65,186,69,198,105)(66,189,78,225,185)(71,204,123,119,107)(72,207,131,145,187)(74,213,149,199,108)(75,216,158,226,188)(80,231,203,120,110)(81,234,212,146,190)(83,240,230,200,111)(84,243,239,227,191)(125,127,133,151,205)(128,136,160,232,206)(134,154,214,152,208)(137,163,241,233,209)(155,217,161,235,215)(164,244,242,236,218), (3,243,241,239,237,235,233,231,229,227,225,223,221,219,217,215,213,211,209,207,205,203,201,199,197,195,193,191,189,187,185,183,181,179,177,175,173,171,169,167,165,163,161,159,157,155,153,151,149,147,145,143,141,139,137,135,133,131,129,127,125,124,122,120,118,116,114,112,110,108,106,104,102,100,98,96,94,92,90,88,86,84,82,80,78,76,74,72,70,68,66,64,62,60,58,56,54,52,50,48,46,44,42,40,38,36,34,32,30,28,26,24,22,20,18,16,14,12,10,8,6)(4,123,121,119,117,115,113,111,109,107,105,103,101,99,97,95,93,91,89,87,85,83,81,79,77,75,73,71,69,67,65,63,61,59,57,55,53,51,49,47,45,43,41,39,37,35,33,31,29,27,25,23,21,19,17,15,13,11,9,7,5,244,242,240,238,236,234,232,230,228,226,224,222,220,218,216,214,212,210,208,206,204,202,200,198,196,194,192,190,188,186,184,182,180,178,176,174,172,170,168,166,164,162,160,158,156,154,152,150,148,146,144,142,140,138,136,134,132,130,128,126), (1,2,4)(5,56,192)(6,79,168)(7,159,87)(8,177,68)(9,124,121)(10,228,15)(11,13,229)(12,26,215)(14,157,82)(16,42,195)(17,138,98)(18,77,158)(19,122,113)(20,236,238)(21,239,234)(22,194,37)(23,27,203)(24,99,130)(25,31,197)(28,70,155)(29,59,165)(30,88,135)(32,48,173)(33,140,80)(34,223,237)(35,69,149)(36,63,154)(38,112,104)(39,75,139)(40,118,96)(41,105,108)(43,166,44)(45,60,148)(46,222,226)(47,142,64)(49,116,89)(50,51,152)(52,218,224)(53,78,123)(54,207,233)(55,227,212)(57,193,244)(58,93,103)(61,73,120)(62,106,86)(65,74,115)(66,71,117)(67,85,102)(72,241,181)(76,201,217)(81,170,243)(83,206,205)(84,190,220)(90,242,162)(91,172,231)(92,235,167)(94,179,221)(95,186,213)(97,198,199)(100,180,214)(101,189,204)(107,202,185)(109,216,169)(110,174,210)(111,232,151)(114,161,219)(119,150,225)(125,240,128)(126,171,196)(127,230,136)(129,176,188)(131,209,153)(132,178,183)(133,200,160)(134,175,184)(137,211,145)(141,144,208)(143,187,163)(146,156,191)(147,164,182) ]", "PSL(2,243)^1" ],
  [  47941632, "[ (1,19,10,22,5,32,16,25,13,24,9,26,11,23,7,28,18,31,12,34,17,35,8,36,2,21,6,29,3,33,14,27)(4,20)(15,30), (3,5,10,18,6,7,15,12,11,16,4,9,14,13,8,17), (21,25,22,35,24,34,26,36,29,31,28,30,32,23,33,27), (2,10,13,9,6,7,5,3)(8,18,15,11,14,16,17,12) ]", "PSL(2,17)^2" ],
  [  49641984, "[ (3,55,102)(4,358,354)(5,463,502)(6,93,420)(7,259,109)(8,178,494)(9,226,133)(10,491,237)(11,216,500)(12,304,320)(13,44,180)(14,38,275)(16,34,168)(17,215,480)(18,103,269)(19,257,506)(20,256,308)(21,43,114)(22,302,66)(23,289,99)(24,54,29)(26,75,385)(27,105,152)(28,86,173)(30,32,144)(31,220,459)(33,471,303)(35,243,190)(36,177,209)(37,142,134)(39,158,255)(40,170,453)(41,403,375)(42,400,349)(46,476,261)(47,327,346)(48,512,95)(49,266,486)(50,363,510)(51,192,175)(52,335,501)(53,313,440)(56,321,495)(57,188,339)(58,183,147)(59,94,193)(60,378,221)(61,167,388)(62,359,267)(63,116,110)(64,206,488)(65,176,72)(67,295,83)(68,285,421)(69,235,344)(70,357,149)(71,434,159)(74,337,249)(76,298,368)(77,417,300)(78,208,397)(79,413,197)(80,415,384)(81,430,234)(82,299,402)(84,443,365)(85,401,250)(87,162,160)(88,330,279)(89,98,423)(90,201,326)(91,294,461)(92,507,312)(96,505,107)(97,408,451)(100,254,213)(104,140,424)(106,138,182)(108,296,245)(111,274,373)(112,468,290)(113,247,412)(115,431,265)(117,328,284)(118,272,163)(119,271,364)(120,252,233)(121,367,270)(122,410,503)(123,139,242)(124,427,227)(125,316,231)(126,409,244)(127,457,513)(128,200,291)(129,475,161)(130,361,322)(131,280,236)(132,405,309)(135,433,196)(136,150,422)(137,414,292)(141,331,404)(143,499,454)(145,380,319)(146,283,297)(148,477,324)(151,428,396)(153,211,366)(154,372,286)(155,278,186)(156,281,348)(157,276,446)(164,394,398)(165,487,341)(166,171,262)(169,329,379)(172,273,498)(174,376,435)(179,452,383)(181,258,419)(184,293,207)(185,482,411)(187,472,325)(189,342,347)(191,395,277)(194,387,439)(195,214,268)(198,497,253)(199,210,315)(202,351,240)(203,248,232)(204,356,391)(205,426,222)(212,362,311)(217,436,456)(218,511,445)(219,418,478)(223,382,504)(224,432,389)(225,377,239)(228,288,381)(229,508,473)(230,353,481)(238,455,479)(241,352,390)(246,467,509)(251,371,465)(260,485,496)(263,448,264)(282,460,449)(287,370,318)(301,490,336)(305,343,425)(306,437,464)(307,489,338)(310,345,450)(314,416,333)(317,392,442)(323,483,441)(332,369,447)(334,360,429)(340,466,470)(350,492,399)(355,474,374)(386,469,407)(406,484,458)(438,462,444), (3,322,109,510,380,159,384,149,48,152,171,14,119,289,5,263,217,337,87,440,122,47,442,297,33,474,228,10,413,199,4,203,192,321,202,104,485,253,89,258,185,26,85,412,8,82,340,453,155,125,213,58,400,66,61,433,182,18,389,320,6,290,309)(7,80,262,463,162,317,288,358,351,98,401,466,100,167,224,132,50,70,38,448,53,146,491,248,140,419,113,40,183,196,12,55,319,512,271,436,410,471,197,175,496,482,178,278,349,106,93,130,71,27,99,249,327,374,210,495,198,75,299,316,22,103,112)(9,467,480,144,501,305,236,408,107,168,173,31,226,52,96,133,15,513,63,246,457,311,154,215,110,46,323,32,509,343,16,335,127,212,476,425,362,493,188,280,286,57,94,97,17,131,28,505,116,372,483,34,261,339,193,86,441,59,25,459,30,451,220)(11,331,379,368,191,189,227,449,478,251,367,445,83,487,325,296,475,244,209,239,77,65,326,421,19,360,184,69,350,382,20,450,383,292,208,306,353,507,396,376,352,370,148,439,444,60,431,273,332,366,143,120,446,41,35,205,301,128,279,156,37,508,484)(13,54,24,101,44,180,29)(21,407,242,255,204,274,84,328,479,114,118,338,81,469,123,443,272,139,393,91,416,158,461,394,359,391,314,136,486,111,39,284,307,365,294,398,150,117,164,45,206,455,62,64,214,43,356,238,430,163,333,267,49,489,422,488,268,234,266,195,73,386,373)(23,315,402,468,502,354,470,405,264,232,170,102,456,51,186,361,74,56,231,259,160,240,254,363,313,424,147,145,503,260,42,434,346,497,302,415,392,423,388,357,283,181,135,95,303,411,138,105,355,385,269,166,381,250,432,275,237,247,304,364,79,494,420)(36,90,293,256,397,151,477,265,499,243,88,458,76,282,218,245,225,68,235,310,437,435,194,498,252,426,281,216,395,219,67,161,417,257,492,179,481,390,438,369,157,490,142,404,342,371,341,126,176,429,504,137,312,318,378,153,403,200,473,169,124,270,187)(72,92,375,427,201,428,190,460,285,174,222,418,506,241,336,465,334,287,291,121,207,324,330,511,344,387,348,295,399,462,134,165,223,221,229,472,308,115,406,108,345,172,500,129,452,447,141,409,414,211,329,177,78,454,298,377,464,233,277,300,230,276,347), (1,2,3,5,9,17,32,60,112,197,318,438,498,198,319,439,505,16,31,58,109)(4,7,13,25,47,88,158,268,329,432,380,395,491,451,291,415,218,138,237,364,474)(6,11,21,39,74,139,239,343,224,350,75,141,242,241,368,476,332,220,344,459,93)(8,15,29,55,104,185,302,423,105,187,304,345,52,98,176,290,357,285,407,150,256)(10,19,36,68,127,132,228,305,426,73,137,236,258,50,94,169,282,331,89,160,170)(12,23,43,82,147,175,289,230,354,417,462,342,424,276,400,495,502,122,81,42,80)(14,27,51,96,172,211,335,403,374,427,475,452,57,108,191,154,262,387,450,485,317)(18,34,64,119,208,190,306,271,366,308,406,238,365,311,178,292,416,441,457,40,76)(20,38,72,135,233,274,33,62,115,202,323,442,301,422,501,143,245,328,449,429,503)(22,41,78,35,66,123,215,48,90,162,193,63,117,205,327,339,359,118,207,330,376)(24,45,85,153,260,386,371,480,455,103,184,247,373,481,477,46,87,156,264,389,487)(26,49,92,166,277,167,279,402,216,209,333,270,396,408,255,383,412,116,203,324,443)(28,53,100,179,294,418,500,303,363,281,173,99,177,280,148,252,379,470,378,483,469)(30,56,106,61,113,199,128,221,196,261,65,121,212,336,251,377,482,437,97,174,287)(37,70,131,227,206,186,136,234,360,159,71,133,229,226,352,286,326,447,488,506,471)(44,83,149,254,382,453,372,265,157,266,392,484,512,272,168,161,195,315,435,297,393)(54,102,182,298,84,151,257,353,181,296,232,358,472,398,494,414,213,249,375,253,165)(59,111,114,200,321,338,456,399,362,458,444,293,188,134,231,356,192,310,430,486,513)(67,125,79,146,250,355,419,142,204,325,445,489,492,288,411,283,401,497,217,341,101)(69,129,223,348,465,508,434,410,496,507,163,273,397,493,243,370,479,120,210,299,420)(77,144,246,337,454,510,309,428,155,86,107,189,307,145,248,124,130,225,351,468,391)(91,164,275,369,322,390,388)(95,171,284,340,384,478,235,361,473,244,320,440,314,433,269,347,463,126,219,183,300)(110,194,313,421,312,431,425,460,349,467,504,405,385,334,409,394,490,448,466,509,214)(140,240,367,381,446,404,499,267,201,152,259,180,295,413,464,278,263,222,346,461,511) ]", "PSU(3,8)^1" ],
  [  50803200, "[ (1,8,7,9,6,10,5,11,4,12,3,13,2,14), (6,7), (13,14), (2,4,3) ]", "A7^2" ],
  [  56899584, "[ (1,5,15)(2,8,21)(3,12,28)(4,14,31)(6,17,34)(7,20,39)(9,23,41)(10,25,44)(11,27,46)(13,30,48)(16,33,50)(18,36,52)(19,38,54)(22,40,55)(24,43,57)(26,45,58)(29,47,59)(32,49,60)(35,51,61)(37,53,62)(42,56,63), (8,45)(12,25)(14,38)(17,49)(20,27)(21,58)(23,47)(28,44)(31,54)(33,36)(34,60)(39,46)(40,53)(41,59)(43,56)(50,52)(55,62)(57,63), (2,26,6,32)(3,4,35,19,10,11,13,7)(8,45)(9,42,24,29,18,22,37,16)(12,25)(14,38)(17,49)(20,27)(23,47)(33,36)(40,53)(43,56), (5,40,53)(8,30,45)(12,23,14)(17,27,43)(20,49,56)(25,38,47)(33,36,51) ]", "PSL(2,7)^3" ],
  [  56899584, "[ (5,15)(8,21)(12,28)(14,31)(17,34)(20,39)(23,41)(25,44)(27,46)(30,48)(33,50)(36,52)(38,54)(40,55)(43,57)(45,58)(47,59)(49,60)(51,61)(53,62)(56,63), (1,5,15)(2,8,21)(3,12,28)(4,14,31)(6,17,34)(7,20,39)(9,23,41)(10,25,44)(11,27,46)(13,30,48)(16,33,50)(18,36,52)(19,38,54)(22,40,55)(24,43,57)(26,45,58)(29,47,59)(32,49,60)(35,51,61)(37,53,62)(42,56,63), (2,26,6,32)(3,4,35,19,10,11,13,7)(8,45)(9,42,24,29,18,22,37,16)(12,25)(14,38)(17,49)(20,27)(21,58)(23,47)(28,44)(31,54)(33,36)(34,60)(39,46)(40,53)(41,59)(43,56)(50,52)(55,62)(57,63), (1,37,16)(2,24,3)(4,42,35)(6,26,7)(9,22,19)(10,18,11)(13,29,32) ]", "PSL(2,7)^3" ],
  [  71743320, "[ (5,7,13,31,85)(6,10,22,58,165)(8,16,40,112,86)(9,19,49,138,166)(11,25,67,192,87)(12,28,76,219,167)(14,34,94,32,88)(15,37,103,59,168)(17,43,121,113,89)(18,46,129,139,169)(20,52,147,193,90)(21,55,156,220,170)(23,61,174,33,91)(24,64,183,60,171)(26,70,201,114,92)(27,73,210,140,172)(29,79,228,194,93)(30,82,237,221,173)(35,97,41,115,95)(36,100,50,141,175)(38,106,68,195,96)(39,109,77,222,176)(42,118,104,62,177)(44,124,122,116,98)(45,126,130,142,178)(47,132,148,196,99)(48,135,157,223,179)(51,144,184,63,180)(53,150,202,117,101)(54,153,211,143,181)(56,159,229,197,102)(57,162,238,224,182)(65,186,69,198,105)(66,189,78,225,185)(71,204,123,119,107)(72,207,131,145,187)(74,213,149,199,108)(75,216,158,226,188)(80,231,203,120,110)(81,234,212,146,190)(83,240,230,200,111)(84,243,239,227,191)(125,127,133,151,205)(128,136,160,232,206)(134,154,214,152,208)(137,163,241,233,209)(155,217,161,235,215)(164,244,242,236,218), (3,244,243,242,241,240,239,238,237,236,235,234,233,232,231,230,229,228,227,226,225,224,223,222,221,220,219,218,217,216,215,214,213,212,211,210,209,208,207,206,205,204,203,202,201,200,199,198,197,196,195,194,193,192,191,190,189,188,187,186,185,184,183,182,181,180,179,178,177,176,175,174,173,172,171,170,169,168,167,166,165,164,163,162,161,160,159,158,157,156,155,154,153,152,151,150,149,148,147,146,145,144,143,142,141,140,139,138,137,136,135,134,133,132,131,130,129,128,127,126,125,4,124,123,122,121,120,119,118,117,116,115,114,113,112,111,110,109,108,107,106,105,104,103,102,101,100,99,98,97,96,95,94,93,92,91,90,89,88,87,86,85,84,83,82,81,80,79,78,77,76,75,74,73,72,71,70,69,68,67,66,65,64,63,62,61,60,59,58,57,56,55,54,53,52,51,50,49,48,47,46,45,44,43,42,41,40,39,38,37,36,35,34,33,32,31,30,29,28,27,26,25,24,23,22,21,20,19,18,17,16,15,14,13,12,11,10,9,8,7,6,5), (1,2,4)(5,56,192)(6,79,168)(7,159,87)(8,177,68)(9,124,121)(10,228,15)(11,13,229)(12,26,215)(14,157,82)(16,42,195)(17,138,98)(18,77,158)(19,122,113)(20,236,238)(21,239,234)(22,194,37)(23,27,203)(24,99,130)(25,31,197)(28,70,155)(29,59,165)(30,88,135)(32,48,173)(33,140,80)(34,223,237)(35,69,149)(36,63,154)(38,112,104)(39,75,139)(40,118,96)(41,105,108)(43,166,44)(45,60,148)(46,222,226)(47,142,64)(49,116,89)(50,51,152)(52,218,224)(53,78,123)(54,207,233)(55,227,212)(57,193,244)(58,93,103)(61,73,120)(62,106,86)(65,74,115)(66,71,117)(67,85,102)(72,241,181)(76,201,217)(81,170,243)(83,206,205)(84,190,220)(90,242,162)(91,172,231)(92,235,167)(94,179,221)(95,186,213)(97,198,199)(100,180,214)(101,189,204)(107,202,185)(109,216,169)(110,174,210)(111,232,151)(114,161,219)(119,150,225)(125,240,128)(126,171,196)(127,230,136)(129,176,188)(131,209,153)(132,178,183)(133,200,160)(134,175,184)(137,211,145)(141,144,208)(143,187,163)(146,156,191)(147,164,182) ]", "PSL(2,243)^1" ],
  [  99283968, "[ (3,251,299,243,260,221)(4,19,271,124,363,134)(5,475,12,473,254,276)(6,376,401,176,56,295)(7,404,138,222,122,239)(8,450,288,341,74,277)(9,328,280,266,212,488)(10,35,27,481,102,344)(11,491,378,160,230,82)(13,24,101)(14,431,106,492,170,377)(15,443,97,73,476,359)(16,461,133,356,32,49)(17,111,483,307,457,139)(18,227,419,252,354,324)(20,471,67,264,298,85)(21,501,114,286,136,372)(22,88,494,211,48,487)(23,329,185,382,317,218)(25,150,154,393,52,43)(26,508,146,194,502,78)(28,39,246,430,509,158)(29,180,54)(30,117,193,294,311,163)(31,214,323,416,96,123)(33,65,80,76,420,291)(34,407,362,391,116,284)(36,237,465,61,353,496)(37,327,477,364,414,258)(38,312,259,399,199,205)(40,426,423,115,109,367)(41,70,437,468,207,474)(42,375,217,244,112,458)(45,86,255,493,234,467)(46,267,451,84,408,62)(47,120,50,137,269,506)(51,387,320,449,75,157)(53,318,166,452,253,156)(55,216,388,190,87,209)(57,195,480,479,107,338)(58,368,495,369,303,174)(59,204,236,118,168,64)(60,167,235,402,177,152)(63,272,226,469,94,486)(66,191,140,153,381,241)(68,497,472,322,370,466)(69,162,371,105,500,485)(71,397,304,334,297,446)(72,263,332,93,256,147)(77,319,169,432,336,442)(79,287,433,189,198,499)(81,173)(83,183,200,250,447,384)(89,350,410,270,275,141)(90,424,165,309,352,178)(91,131,386,144,206,110)(92,453,331,210,265,503)(95,345,104,279,249,390)(98,225,315,121,182,507)(99,151,145,223,203,301)(100,429,385,129,159,439)(103,142,186,233,119,296)(108,510,148,278,257,181)(113,403,240,462,290,478)(125,379,248,498,392,428)(126,355,418,400,306,351)(127,394,505,373,425,268)(128,448,435,415,308,321)(130,179,135,285,440,143)(132,310,302,201,337,366)(149,444,349,293,247,409)(155,292,358,245,346,427)(161,283,460,213,208,175)(164,215,333,343,489,220)(171,325,196,281,470,454)(172,380,445,316,490,411)(184,436,219,357,406,202)(187,313,347,340,383,197)(188,314,513,238,459,274)(192,360,463,282,434,229)(224,504,231,300,289,273)(228,326,512,395,405,330)(232,511,389,396,482,417)(242,305,455,441,398,339)(261,365)(262,342,361,348,413,421)(335,422)(374,438,456,464,412,484), (3,361,249,47,392,98,85,247,12)(4,232,40,58,42,71,152,166,288)(5,264,248,104,260,349,182,269,262)(6,468,463,87,313,140,258,411,106)(7,384,357,146,10,79,178,155,231)(8,402,112,109,363,53,297,303,482)(9,246,127,311,286,339,94,451,31)(11,141,126,65,201,151,148,387,281)(13,24,101)(14,364,197,192,56,316,66,388,70)(15,457,362,493,57,59,25,220,133)(16,52,107,34,476,343,168,86,483)(17,236,97,480,32,467,215,116,154)(18,432,38,263,456,175,485,497,22)(19,334,318,60,115,458,368,277,417)(20,345,498,120,276,342,251,121,293)(21,469,242,158,294,394,62,488,214)(23,210,321,240,100,61,135,512,119)(26,250,224,309,102,436,122,346,198)(27,289,502,358,202,424,183,433,138)(28,96,226,501,425,212,46,441,193)(29,180,54)(30,408,505,63,509,305,280,372,323)(33,355,75,82,470,132,510,145,410)(35,222,219,83,165,504,292,78,499)(36,326,285,435,439,462,142,331,329)(37,229,187,209,377,437,376,241,490)(39,328,455,267,486,373,114,163,416)(41,190,282,445,295,492,383,414,153)(43,204,111,255,391,333,359,356,479)(45,64,195,73,407,139,393,461,164)(48,105,374,199,354,466,213,147,319)(49,307,443,118,489,150,284,338,234)(50,149,95,471,413,315,299,125,254)(51,278,400,302,80,171,275,491,203)(55,380,434,327,474,381,401,340,170)(67,475,409,137,507,428,243,279,348)(68,69,399,438,332,211,169,227,460)(72,312,370,324,88,484,500,161,77)(74,495,253,423,167,389,304,271,217)(76,449,418,257,350,223,378,366,454)(81,386,274,365,91,314,422,206,238)(84,117,398,136,266,268,430,272,123)(89,181,196,320,420,99,337,160,351)(90,184,344,194,273,447,404,189,427)(92,403,128,330,218,296,129,179,353)(93,322,259,162,442,283,419,412,494)(103,290,405,448,440,503,496,185,385)(108,310,306,230,157,301,291,270,325)(110,261,188,131,173,459,144,335,513)(113,453,186,130,159,415,317,228,237)(124,367,511,235,450,452,369,446,375)(134,341,244,177,397,396,174,426,156)(143,233,395,478,465,429,382,308,265)(172,216,191,347,176,360,207,477,431)(200,508,406,245,239,300,481,352,287)(205,336,371,487,472,256,208,464,252)(221,473,379,298,225,421,506,390,444), (3,55,102)(4,358,354)(5,463,502)(6,93,420)(7,259,109)(8,178,494)(9,226,133)(10,491,237)(11,216,500)(12,304,320)(13,44,180)(14,38,275)(16,34,168)(17,215,480)(18,103,269)(19,257,506)(20,256,308)(21,43,114)(22,302,66)(23,289,99)(24,54,29)(26,75,385)(27,105,152)(28,86,173)(30,32,144)(31,220,459)(33,471,303)(35,243,190)(36,177,209)(37,142,134)(39,158,255)(40,170,453)(41,403,375)(42,400,349)(46,476,261)(47,327,346)(48,512,95)(49,266,486)(50,363,510)(51,192,175)(52,335,501)(53,313,440)(56,321,495)(57,188,339)(58,183,147)(59,94,193)(60,378,221)(61,167,388)(62,359,267)(63,116,110)(64,206,488)(65,176,72)(67,295,83)(68,285,421)(69,235,344)(70,357,149)(71,434,159)(74,337,249)(76,298,368)(77,417,300)(78,208,397)(79,413,197)(80,415,384)(81,430,234)(82,299,402)(84,443,365)(85,401,250)(87,162,160)(88,330,279)(89,98,423)(90,201,326)(91,294,461)(92,507,312)(96,505,107)(97,408,451)(100,254,213)(104,140,424)(106,138,182)(108,296,245)(111,274,373)(112,468,290)(113,247,412)(115,431,265)(117,328,284)(118,272,163)(119,271,364)(120,252,233)(121,367,270)(122,410,503)(123,139,242)(124,427,227)(125,316,231)(126,409,244)(127,457,513)(128,200,291)(129,475,161)(130,361,322)(131,280,236)(132,405,309)(135,433,196)(136,150,422)(137,414,292)(141,331,404)(143,499,454)(145,380,319)(146,283,297)(148,477,324)(151,428,396)(153,211,366)(154,372,286)(155,278,186)(156,281,348)(157,276,446)(164,394,398)(165,487,341)(166,171,262)(169,329,379)(172,273,498)(174,376,435)(179,452,383)(181,258,419)(184,293,207)(185,482,411)(187,472,325)(189,342,347)(191,395,277)(194,387,439)(195,214,268)(198,497,253)(199,210,315)(202,351,240)(203,248,232)(204,356,391)(205,426,222)(212,362,311)(217,436,456)(218,511,445)(219,418,478)(223,382,504)(224,432,389)(225,377,239)(228,288,381)(229,508,473)(230,353,481)(238,455,479)(241,352,390)(246,467,509)(251,371,465)(260,485,496)(263,448,264)(282,460,449)(287,370,318)(301,490,336)(305,343,425)(306,437,464)(307,489,338)(310,345,450)(314,416,333)(317,392,442)(323,483,441)(332,369,447)(334,360,429)(340,466,470)(350,492,399)(355,474,374)(386,469,407)(406,484,458)(438,462,444), (1,2,3,5,9,17,32,60,112,197,318,438,498,198,319,439,505,16,31,58,109)(4,7,13,25,47,88,158,268,329,432,380,395,491,451,291,415,218,138,237,364,474)(6,11,21,39,74,139,239,343,224,350,75,141,242,241,368,476,332,220,344,459,93)(8,15,29,55,104,185,302,423,105,187,304,345,52,98,176,290,357,285,407,150,256)(10,19,36,68,127,132,228,305,426,73,137,236,258,50,94,169,282,331,89,160,170)(12,23,43,82,147,175,289,230,354,417,462,342,424,276,400,495,502,122,81,42,80)(14,27,51,96,172,211,335,403,374,427,475,452,57,108,191,154,262,387,450,485,317)(18,34,64,119,208,190,306,271,366,308,406,238,365,311,178,292,416,441,457,40,76)(20,38,72,135,233,274,33,62,115,202,323,442,301,422,501,143,245,328,449,429,503)(22,41,78,35,66,123,215,48,90,162,193,63,117,205,327,339,359,118,207,330,376)(24,45,85,153,260,386,371,480,455,103,184,247,373,481,477,46,87,156,264,389,487)(26,49,92,166,277,167,279,402,216,209,333,270,396,408,255,383,412,116,203,324,443)(28,53,100,179,294,418,500,303,363,281,173,99,177,280,148,252,379,470,378,483,469)(30,56,106,61,113,199,128,221,196,261,65,121,212,336,251,377,482,437,97,174,287)(37,70,131,227,206,186,136,234,360,159,71,133,229,226,352,286,326,447,488,506,471)(44,83,149,254,382,453,372,265,157,266,392,484,512,272,168,161,195,315,435,297,393)(54,102,182,298,84,151,257,353,181,296,232,358,472,398,494,414,213,249,375,253,165)(59,111,114,200,321,338,456,399,362,458,444,293,188,134,231,356,192,310,430,486,513)(67,125,79,146,250,355,419,142,204,325,445,489,492,288,411,283,401,497,217,341,101)(69,129,223,348,465,508,434,410,496,507,163,273,397,493,243,370,479,120,210,299,420)(77,144,246,337,454,510,309,428,155,86,107,189,307,145,248,124,130,225,351,468,391)(91,164,275,369,322,390,388)(95,171,284,340,384,478,235,361,473,244,320,440,314,433,269,347,463,126,219,183,300)(110,194,313,421,312,431,425,460,349,467,504,405,385,334,409,394,490,448,466,509,214)(140,240,367,381,446,404,499,267,201,152,259,180,295,413,464,278,263,222,346,461,511) ]", "PSU(3,8)^1" ],
  [ 113799168, "[ (2,26,6,32)(3,4,35,19,10,11,13,7)(5,15)(8,58)(9,42,24,29,18,22,37,16)(12,44)(14,54)(17,60)(20,46)(21,45)(23,59)(25,28)(27,39)(30,48)(31,38)(33,52)(34,49)(36,50)(40,62)(41,47)(43,63)(51,61)(53,55)(56,57), (1,5,15)(2,8,21)(3,12,28)(4,14,31)(6,17,34)(7,20,39)(9,23,41)(10,25,44)(11,27,46)(13,30,48)(16,33,50)(18,36,52)(19,38,54)(22,40,55)(24,43,57)(26,45,58)(29,47,59)(32,49,60)(35,51,61)(37,53,62)(42,56,63), (8,45)(12,25)(14,38)(17,49)(20,27)(21,58)(23,47)(28,44)(31,54)(33,36)(34,60)(39,46)(40,53)(41,59)(43,56)(50,52)(55,62)(57,63), (2,26,6,32)(3,4,35,19,10,11,13,7)(8,45)(9,42,24,29,18,22,37,16)(12,25)(14,38)(17,49)(20,27)(23,47)(33,36)(40,53)(43,56), (5,53,33)(8,43,12)(14,56,51)(17,45,20)(23,40,38)(25,36,27)(30,47,49) ]", "PSL(2,7)^3" ],
  [ 113799168, "[ (2,26,6,32)(3,4,35,19,10,11,13,7)(8,45)(9,42,24,29,18,22,37,16)(12,25)(14,38)(17,49)(20,27)(21,58)(23,47)(28,44)(31,54)(33,36)(34,60)(39,46)(40,53)(41,59)(43,56)(50,52)(55,62)(57,63), (1,5,15)(2,8,21)(3,12,28)(4,14,31)(6,17,34)(7,20,39)(9,23,41)(10,25,44)(11,27,46)(13,30,48)(16,33,50)(18,36,52)(19,38,54)(22,40,55)(24,43,57)(26,45,58)(29,47,59)(32,49,60)(35,51,61)(37,53,62)(42,56,63), (8,45)(12,25)(14,38)(17,49)(20,27)(21,58)(23,47)(28,44)(31,54)(33,36)(34,60)(39,46)(40,53)(41,59)(43,56)(50,52)(55,62)(57,63), (2,26,6,32)(3,4,35,19,10,11,13,7)(8,45)(9,42,24,29,18,22,37,16)(12,25)(14,38)(17,49)(20,27)(23,47)(33,36)(40,53)(43,56), (1,37,16)(2,24,3)(4,42,35)(6,26,7)(9,22,19)(10,18,11)(13,29,32) ]", "PSL(2,7)^3" ],
  [ 113799168, "[ (5,15)(8,21)(12,28)(14,31)(17,34)(20,39)(23,41)(25,44)(27,46)(30,48)(33,50)(36,52)(38,54)(40,55)(43,57)(45,58)(47,59)(49,60)(51,61)(53,62)(56,63), (1,5,15)(2,8,21)(3,12,28)(4,14,31)(6,17,34)(7,20,39)(9,23,41)(10,25,44)(11,27,46)(13,30,48)(16,33,50)(18,36,52)(19,38,54)(22,40,55)(24,43,57)(26,45,58)(29,47,59)(32,49,60)(35,51,61)(37,53,62)(42,56,63), (8,45)(12,25)(14,38)(17,49)(20,27)(21,58)(23,47)(28,44)(31,54)(33,36)(34,60)(39,46)(40,53)(41,59)(43,56)(50,52)(55,62)(57,63), (2,26,6,32)(3,4,35,19,10,11,13,7)(8,45)(9,42,24,29,18,22,37,16)(12,25)(14,38)(17,49)(20,27)(23,47)(33,36)(40,53)(43,56), (5,53,33)(8,43,12)(14,56,51)(17,45,20)(23,40,38)(25,36,27)(30,47,49) ]", "PSL(2,7)^3" ],
  [ 227598336, "[ (5,15)(8,21)(12,28)(14,31)(17,34)(20,39)(23,41)(25,44)(27,46)(30,48)(33,50)(36,52)(38,54)(40,55)(43,57)(45,58)(47,59)(49,60)(51,61)(53,62)(56,63), (1,5,15)(2,8,21)(3,12,28)(4,14,31)(6,17,34)(7,20,39)(9,23,41)(10,25,44)(11,27,46)(13,30,48)(16,33,50)(18,36,52)(19,38,54)(22,40,55)(24,43,57)(26,45,58)(29,47,59)(32,49,60)(35,51,61)(37,53,62)(42,56,63), (21,58)(28,44)(31,54)(34,60)(39,46)(41,59)(50,52)(55,62)(57,63), (8,45)(12,25)(14,38)(17,49)(20,27)(23,47)(33,36)(40,53)(43,56), (2,26,6,32)(3,4,35,19,10,11,13,7)(9,42,24,29,18,22,37,16), (1,22,37)(2,13,26)(3,9,4)(6,11,24)(7,32,42)(10,19,29)(16,18,35) ]", "PSL(2,7)^3" ] ],
LINS_TransformTargets
));
