/*
    Copyright (C) 2012  Dan Vratil <dvratil@redhat.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public
    License as published by the Free Software Foundation; either
    version 2.1 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with this library; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef QMLSCREEN_H
#define QMLSCREEN_H

#include <QQuickItem>

#include <KF5/KScreen/kscreen/output.h>
#include "qmloutput.h"

class QQmlEngine;

namespace KScreen {
class Output;
class Config;
}

class QMLScreen : public QQuickItem
{
    Q_OBJECT

    Q_PROPERTY(QSize maxScreenSize
               READ maxScreenSize
               CONSTANT)

    Q_PROPERTY(int connectedOutputsCount
               READ connectedOutputsCount
               NOTIFY connectedOutputsCountChanged)

    Q_PROPERTY(int enabledOutputsCount
               READ enabledOutputsCount
               NOTIFY enabledOutputsCountChanged)

    Q_PROPERTY(float outputScale
               READ outputScale
               NOTIFY outputScaleChanged)

public:
    explicit QMLScreen(QQuickItem *parent = nullptr);
    ~QMLScreen() override;

    int connectedOutputsCount() const;
    int enabledOutputsCount() const;

    QMLOutput *primaryOutput() const;
    QList<QMLOutput*> outputs() const;

    QSize maxScreenSize() const;

    float outputScale() const;

    KScreen::ConfigPtr config() const;
    void setConfig(const KScreen::ConfigPtr &config);

    void updateOutputsPlacement();

    void setActiveOutput(QMLOutput *output);

    void setScreenPos(QMLOutput *output, bool isReleased);

    void setScreenCenterPos();
public Q_SLOTS:
    void setActiveOutput()
    {
        setActiveOutput(qobject_cast<QMLOutput *>(sender()));
    }

    void setActiveOutputByCombox(int screenId);
    void setScreenPos(bool isReleased)
    {
        setScreenPos(qobject_cast<QMLOutput*>(sender()), isReleased);
    }

Q_SIGNALS:
    void connectedOutputsCountChanged();
    void enabledOutputsCountChanged();

    void outputScaleChanged();

    void focusedOutputChanged(QMLOutput *output);

    void released();

private Q_SLOTS:
    void addOutput(const KScreen::OutputPtr &output);
    void removeOutput(int outputId);

    void outputConnectedChanged();
    void outputEnabledChanged();
    void outputPositionChanged();

    void viewSizeChanged();

private:
    void qmlOutputMoved(QMLOutput *qmlOutput);
    void updateCornerOutputs();
    void setOutputScale(float scale);

    void setX100GPU();

    bool mIsX100;

    KScreen::ConfigPtr m_config;
    QHash<KScreen::OutputPtr, QMLOutput *> m_outputMap;
    QVector<QMLOutput*> m_manuallyMovedOutputs;
    int m_connectedOutputsCount = 0;
    int m_enabledOutputsCount = 0;

    float m_outputScale = 1.0 / 14.0;// 缩放比例

    QMLOutput *m_leftmost = nullptr;
    QMLOutput *m_topmost = nullptr;
    QMLOutput *m_rightmost = nullptr;
    QMLOutput *m_bottommost = nullptr;
};

#endif // QMLSCREEN_H
