/**
 * Copyright 2013-2023 Software Radio Systems Limited
 *
 * This file is part of srsRAN.
 *
 * srsRAN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of
 * the License, or (at your option) any later version.
 *
 * srsRAN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * A copy of the GNU Affero General Public License can be found in
 * the LICENSE file in the top-level directory of this distribution
 * and at http://www.gnu.org/licenses/.
 *
 */

#include "srsran/asn1/ngap.h"
#include <sstream>

using namespace asn1;
using namespace asn1::ngap;

/*******************************************************************************
 *                                Struct Methods
 ******************************************************************************/

// CPTransportLayerInformation ::= CHOICE
void cp_transport_layer_info_c::destroy_()
{
  switch (type_) {
    case types::endpoint_ip_address:
      c.destroy<bounded_bitstring<1, 160, true, true> >();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<cp_transport_layer_info_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void cp_transport_layer_info_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::endpoint_ip_address:
      c.init<bounded_bitstring<1, 160, true, true> >();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<cp_transport_layer_info_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cp_transport_layer_info_c");
  }
}
cp_transport_layer_info_c::cp_transport_layer_info_c(const cp_transport_layer_info_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::endpoint_ip_address:
      c.init(other.c.get<bounded_bitstring<1, 160, true, true> >());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<cp_transport_layer_info_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cp_transport_layer_info_c");
  }
}
cp_transport_layer_info_c& cp_transport_layer_info_c::operator=(const cp_transport_layer_info_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::endpoint_ip_address:
      c.set(other.c.get<bounded_bitstring<1, 160, true, true> >());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<cp_transport_layer_info_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cp_transport_layer_info_c");
  }

  return *this;
}
bounded_bitstring<1, 160, true, true>& cp_transport_layer_info_c::set_endpoint_ip_address()
{
  set(types::endpoint_ip_address);
  return c.get<bounded_bitstring<1, 160, true, true> >();
}
protocol_ie_single_container_s<cp_transport_layer_info_ext_ies_o>& cp_transport_layer_info_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<cp_transport_layer_info_ext_ies_o> >();
}
void cp_transport_layer_info_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::endpoint_ip_address:
      j.write_str("endpointIPAddress", c.get<bounded_bitstring<1, 160, true, true> >().to_string());
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<cp_transport_layer_info_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "cp_transport_layer_info_c");
  }
  j.end_obj();
}
SRSASN_CODE cp_transport_layer_info_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::endpoint_ip_address:
      HANDLE_CODE((c.get<bounded_bitstring<1, 160, true, true> >().pack(bref)));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<cp_transport_layer_info_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cp_transport_layer_info_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cp_transport_layer_info_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::endpoint_ip_address:
      HANDLE_CODE((c.get<bounded_bitstring<1, 160, true, true> >().unpack(bref)));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<cp_transport_layer_info_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cp_transport_layer_info_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* cp_transport_layer_info_c::types_opts::to_string() const
{
  static const char* options[] = {"endpointIPAddress", "choice-Extensions"};
  return convert_enum_idx(options, 2, value, "cp_transport_layer_info_c::types");
}

// AMF-TNLAssociationSetupItem ::= SEQUENCE
SRSASN_CODE amf_tnlassoc_setup_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(amf_tnlassoc_address.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE amf_tnlassoc_setup_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(amf_tnlassoc_address.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void amf_tnlassoc_setup_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("aMF-TNLAssociationAddress");
  amf_tnlassoc_address.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TNLAssociationUsage ::= ENUMERATED
const char* tnlassoc_usage_opts::to_string() const
{
  static const char* options[] = {"ue", "non-ue", "both"};
  return convert_enum_idx(options, 3, value, "tnlassoc_usage_e");
}

// AMF-TNLAssociationToAddItem ::= SEQUENCE
SRSASN_CODE amf_tnlassoc_to_add_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(tnlassoc_usage_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(amf_tnlassoc_address.pack(bref));
  if (tnlassoc_usage_present) {
    HANDLE_CODE(tnlassoc_usage.pack(bref));
  }
  HANDLE_CODE(pack_integer(bref, tnl_address_weight_factor, (uint16_t)0u, (uint16_t)255u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE amf_tnlassoc_to_add_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(tnlassoc_usage_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(amf_tnlassoc_address.unpack(bref));
  if (tnlassoc_usage_present) {
    HANDLE_CODE(tnlassoc_usage.unpack(bref));
  }
  HANDLE_CODE(unpack_integer(tnl_address_weight_factor, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void amf_tnlassoc_to_add_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("aMF-TNLAssociationAddress");
  amf_tnlassoc_address.to_json(j);
  if (tnlassoc_usage_present) {
    j.write_str("tNLAssociationUsage", tnlassoc_usage.to_string());
  }
  j.write_int("tNLAddressWeightFactor", tnl_address_weight_factor);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AMF-TNLAssociationToRemoveItem ::= SEQUENCE
SRSASN_CODE amf_tnlassoc_to_rem_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(amf_tnlassoc_address.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE amf_tnlassoc_to_rem_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(amf_tnlassoc_address.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void amf_tnlassoc_to_rem_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("aMF-TNLAssociationAddress");
  amf_tnlassoc_address.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AMF-TNLAssociationToUpdateItem ::= SEQUENCE
SRSASN_CODE amf_tnlassoc_to_upd_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(tnlassoc_usage_present, 1));
  HANDLE_CODE(bref.pack(tnl_address_weight_factor_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(amf_tnlassoc_address.pack(bref));
  if (tnlassoc_usage_present) {
    HANDLE_CODE(tnlassoc_usage.pack(bref));
  }
  if (tnl_address_weight_factor_present) {
    HANDLE_CODE(pack_integer(bref, tnl_address_weight_factor, (uint16_t)0u, (uint16_t)255u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE amf_tnlassoc_to_upd_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(tnlassoc_usage_present, 1));
  HANDLE_CODE(bref.unpack(tnl_address_weight_factor_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(amf_tnlassoc_address.unpack(bref));
  if (tnlassoc_usage_present) {
    HANDLE_CODE(tnlassoc_usage.unpack(bref));
  }
  if (tnl_address_weight_factor_present) {
    HANDLE_CODE(unpack_integer(tnl_address_weight_factor, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void amf_tnlassoc_to_upd_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("aMF-TNLAssociationAddress");
  amf_tnlassoc_address.to_json(j);
  if (tnlassoc_usage_present) {
    j.write_str("tNLAssociationUsage", tnlassoc_usage.to_string());
  }
  if (tnl_address_weight_factor_present) {
    j.write_int("tNLAddressWeightFactor", tnl_address_weight_factor);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// S-NSSAI ::= SEQUENCE
SRSASN_CODE s_nssai_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(sd_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(sst.pack(bref));
  if (sd_present) {
    HANDLE_CODE(sd.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE s_nssai_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(sd_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(sst.unpack(bref));
  if (sd_present) {
    HANDLE_CODE(sd.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void s_nssai_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("sST", sst.to_string());
  if (sd_present) {
    j.write_str("sD", sd.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// SliceSupportItem ::= SEQUENCE
SRSASN_CODE slice_support_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(s_nssai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE slice_support_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(s_nssai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void slice_support_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("s-NSSAI");
  s_nssai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// GUAMI ::= SEQUENCE
SRSASN_CODE guami_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(amf_region_id.pack(bref));
  HANDLE_CODE(amf_set_id.pack(bref));
  HANDLE_CODE(amf_pointer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE guami_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(amf_region_id.unpack(bref));
  HANDLE_CODE(amf_set_id.unpack(bref));
  HANDLE_CODE(amf_pointer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void guami_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNIdentity", plmn_id.to_string());
  j.write_str("aMFRegionID", amf_region_id.to_string());
  j.write_str("aMFSetID", amf_set_id.to_string());
  j.write_str("aMFPointer", amf_pointer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PLMNSupportItem ::= SEQUENCE
SRSASN_CODE plmn_support_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, slice_support_list, 1, 1024, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE plmn_support_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(slice_support_list, bref, 1, 1024, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void plmn_support_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNIdentity", plmn_id.to_string());
  j.start_array("sliceSupportList");
  for (const auto& e1 : slice_support_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ServedGUAMIItem ::= SEQUENCE
SRSASN_CODE served_guami_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(backup_amf_name_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(guami.pack(bref));
  if (backup_amf_name_present) {
    HANDLE_CODE(backup_amf_name.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE served_guami_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(backup_amf_name_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(guami.unpack(bref));
  if (backup_amf_name_present) {
    HANDLE_CODE(backup_amf_name.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void served_guami_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("gUAMI");
  guami.to_json(j);
  if (backup_amf_name_present) {
    j.write_str("backupAMFName", backup_amf_name.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AMFConfigurationUpdateIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t amf_cfg_upd_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {1, 96, 86, 80, 6, 7, 8};
  return map_enum_number(options, 7, idx, "id");
}
bool amf_cfg_upd_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {1, 96, 86, 80, 6, 7, 8};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e amf_cfg_upd_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 1:
      return crit_e::reject;
    case 96:
      return crit_e::reject;
    case 86:
      return crit_e::ignore;
    case 80:
      return crit_e::reject;
    case 6:
      return crit_e::ignore;
    case 7:
      return crit_e::ignore;
    case 8:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
amf_cfg_upd_ies_o::value_c amf_cfg_upd_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 1:
      ret.set(value_c::types::amf_name);
      break;
    case 96:
      ret.set(value_c::types::served_guami_list);
      break;
    case 86:
      ret.set(value_c::types::relative_amf_capacity);
      break;
    case 80:
      ret.set(value_c::types::plmn_support_list);
      break;
    case 6:
      ret.set(value_c::types::amf_tnlassoc_to_add_list);
      break;
    case 7:
      ret.set(value_c::types::amf_tnlassoc_to_rem_list);
      break;
    case 8:
      ret.set(value_c::types::amf_tnlassoc_to_upd_list);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e amf_cfg_upd_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 1:
      return presence_e::optional;
    case 96:
      return presence_e::optional;
    case 86:
      return presence_e::optional;
    case 80:
      return presence_e::optional;
    case 6:
      return presence_e::optional;
    case 7:
      return presence_e::optional;
    case 8:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void amf_cfg_upd_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::amf_name:
      c.destroy<printable_string<1, 150, true, true> >();
      break;
    case types::served_guami_list:
      c.destroy<served_guami_list_l>();
      break;
    case types::plmn_support_list:
      c.destroy<plmn_support_list_l>();
      break;
    case types::amf_tnlassoc_to_add_list:
      c.destroy<amf_tnlassoc_to_add_list_l>();
      break;
    case types::amf_tnlassoc_to_rem_list:
      c.destroy<amf_tnlassoc_to_rem_list_l>();
      break;
    case types::amf_tnlassoc_to_upd_list:
      c.destroy<amf_tnlassoc_to_upd_list_l>();
      break;
    default:
      break;
  }
}
void amf_cfg_upd_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_name:
      c.init<printable_string<1, 150, true, true> >();
      break;
    case types::served_guami_list:
      c.init<served_guami_list_l>();
      break;
    case types::relative_amf_capacity:
      break;
    case types::plmn_support_list:
      c.init<plmn_support_list_l>();
      break;
    case types::amf_tnlassoc_to_add_list:
      c.init<amf_tnlassoc_to_add_list_l>();
      break;
    case types::amf_tnlassoc_to_rem_list:
      c.init<amf_tnlassoc_to_rem_list_l>();
      break;
    case types::amf_tnlassoc_to_upd_list:
      c.init<amf_tnlassoc_to_upd_list_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_ies_o::value_c");
  }
}
amf_cfg_upd_ies_o::value_c::value_c(const amf_cfg_upd_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_name:
      c.init(other.c.get<printable_string<1, 150, true, true> >());
      break;
    case types::served_guami_list:
      c.init(other.c.get<served_guami_list_l>());
      break;
    case types::relative_amf_capacity:
      c.init(other.c.get<uint16_t>());
      break;
    case types::plmn_support_list:
      c.init(other.c.get<plmn_support_list_l>());
      break;
    case types::amf_tnlassoc_to_add_list:
      c.init(other.c.get<amf_tnlassoc_to_add_list_l>());
      break;
    case types::amf_tnlassoc_to_rem_list:
      c.init(other.c.get<amf_tnlassoc_to_rem_list_l>());
      break;
    case types::amf_tnlassoc_to_upd_list:
      c.init(other.c.get<amf_tnlassoc_to_upd_list_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_ies_o::value_c");
  }
}
amf_cfg_upd_ies_o::value_c& amf_cfg_upd_ies_o::value_c::operator=(const amf_cfg_upd_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_name:
      c.set(other.c.get<printable_string<1, 150, true, true> >());
      break;
    case types::served_guami_list:
      c.set(other.c.get<served_guami_list_l>());
      break;
    case types::relative_amf_capacity:
      c.set(other.c.get<uint16_t>());
      break;
    case types::plmn_support_list:
      c.set(other.c.get<plmn_support_list_l>());
      break;
    case types::amf_tnlassoc_to_add_list:
      c.set(other.c.get<amf_tnlassoc_to_add_list_l>());
      break;
    case types::amf_tnlassoc_to_rem_list:
      c.set(other.c.get<amf_tnlassoc_to_rem_list_l>());
      break;
    case types::amf_tnlassoc_to_upd_list:
      c.set(other.c.get<amf_tnlassoc_to_upd_list_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_ies_o::value_c");
  }

  return *this;
}
printable_string<1, 150, true, true>& amf_cfg_upd_ies_o::value_c::amf_name()
{
  assert_choice_type(types::amf_name, type_, "Value");
  return c.get<printable_string<1, 150, true, true> >();
}
served_guami_list_l& amf_cfg_upd_ies_o::value_c::served_guami_list()
{
  assert_choice_type(types::served_guami_list, type_, "Value");
  return c.get<served_guami_list_l>();
}
uint16_t& amf_cfg_upd_ies_o::value_c::relative_amf_capacity()
{
  assert_choice_type(types::relative_amf_capacity, type_, "Value");
  return c.get<uint16_t>();
}
plmn_support_list_l& amf_cfg_upd_ies_o::value_c::plmn_support_list()
{
  assert_choice_type(types::plmn_support_list, type_, "Value");
  return c.get<plmn_support_list_l>();
}
amf_tnlassoc_to_add_list_l& amf_cfg_upd_ies_o::value_c::amf_tnlassoc_to_add_list()
{
  assert_choice_type(types::amf_tnlassoc_to_add_list, type_, "Value");
  return c.get<amf_tnlassoc_to_add_list_l>();
}
amf_tnlassoc_to_rem_list_l& amf_cfg_upd_ies_o::value_c::amf_tnlassoc_to_rem_list()
{
  assert_choice_type(types::amf_tnlassoc_to_rem_list, type_, "Value");
  return c.get<amf_tnlassoc_to_rem_list_l>();
}
amf_tnlassoc_to_upd_list_l& amf_cfg_upd_ies_o::value_c::amf_tnlassoc_to_upd_list()
{
  assert_choice_type(types::amf_tnlassoc_to_upd_list, type_, "Value");
  return c.get<amf_tnlassoc_to_upd_list_l>();
}
const printable_string<1, 150, true, true>& amf_cfg_upd_ies_o::value_c::amf_name() const
{
  assert_choice_type(types::amf_name, type_, "Value");
  return c.get<printable_string<1, 150, true, true> >();
}
const served_guami_list_l& amf_cfg_upd_ies_o::value_c::served_guami_list() const
{
  assert_choice_type(types::served_guami_list, type_, "Value");
  return c.get<served_guami_list_l>();
}
const uint16_t& amf_cfg_upd_ies_o::value_c::relative_amf_capacity() const
{
  assert_choice_type(types::relative_amf_capacity, type_, "Value");
  return c.get<uint16_t>();
}
const plmn_support_list_l& amf_cfg_upd_ies_o::value_c::plmn_support_list() const
{
  assert_choice_type(types::plmn_support_list, type_, "Value");
  return c.get<plmn_support_list_l>();
}
const amf_tnlassoc_to_add_list_l& amf_cfg_upd_ies_o::value_c::amf_tnlassoc_to_add_list() const
{
  assert_choice_type(types::amf_tnlassoc_to_add_list, type_, "Value");
  return c.get<amf_tnlassoc_to_add_list_l>();
}
const amf_tnlassoc_to_rem_list_l& amf_cfg_upd_ies_o::value_c::amf_tnlassoc_to_rem_list() const
{
  assert_choice_type(types::amf_tnlassoc_to_rem_list, type_, "Value");
  return c.get<amf_tnlassoc_to_rem_list_l>();
}
const amf_tnlassoc_to_upd_list_l& amf_cfg_upd_ies_o::value_c::amf_tnlassoc_to_upd_list() const
{
  assert_choice_type(types::amf_tnlassoc_to_upd_list, type_, "Value");
  return c.get<amf_tnlassoc_to_upd_list_l>();
}
void amf_cfg_upd_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_name:
      j.write_str("PrintableString", c.get<printable_string<1, 150, true, true> >().to_string());
      break;
    case types::served_guami_list:
      j.start_array("ServedGUAMIList");
      for (const auto& e1 : c.get<served_guami_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::relative_amf_capacity:
      j.write_int("INTEGER (0..255)", c.get<uint16_t>());
      break;
    case types::plmn_support_list:
      j.start_array("PLMNSupportList");
      for (const auto& e1 : c.get<plmn_support_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::amf_tnlassoc_to_add_list:
      j.start_array("AMF-TNLAssociationToAddList");
      for (const auto& e1 : c.get<amf_tnlassoc_to_add_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::amf_tnlassoc_to_rem_list:
      j.start_array("AMF-TNLAssociationToRemoveList");
      for (const auto& e1 : c.get<amf_tnlassoc_to_rem_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::amf_tnlassoc_to_upd_list:
      j.start_array("AMF-TNLAssociationToUpdateList");
      for (const auto& e1 : c.get<amf_tnlassoc_to_upd_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE amf_cfg_upd_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_name:
      HANDLE_CODE((c.get<printable_string<1, 150, true, true> >().pack(bref)));
      break;
    case types::served_guami_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<served_guami_list_l>(), 1, 256, true));
      break;
    case types::relative_amf_capacity:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)255u, false, true));
      break;
    case types::plmn_support_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<plmn_support_list_l>(), 1, 12, true));
      break;
    case types::amf_tnlassoc_to_add_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<amf_tnlassoc_to_add_list_l>(), 1, 32, true));
      break;
    case types::amf_tnlassoc_to_rem_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<amf_tnlassoc_to_rem_list_l>(), 1, 32, true));
      break;
    case types::amf_tnlassoc_to_upd_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<amf_tnlassoc_to_upd_list_l>(), 1, 32, true));
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE amf_cfg_upd_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_name:
      HANDLE_CODE((c.get<printable_string<1, 150, true, true> >().unpack(bref)));
      break;
    case types::served_guami_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<served_guami_list_l>(), bref, 1, 256, true));
      break;
    case types::relative_amf_capacity:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)255u, false, true));
      break;
    case types::plmn_support_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<plmn_support_list_l>(), bref, 1, 12, true));
      break;
    case types::amf_tnlassoc_to_add_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<amf_tnlassoc_to_add_list_l>(), bref, 1, 32, true));
      break;
    case types::amf_tnlassoc_to_rem_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<amf_tnlassoc_to_rem_list_l>(), bref, 1, 32, true));
      break;
    case types::amf_tnlassoc_to_upd_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<amf_tnlassoc_to_upd_list_l>(), bref, 1, 32, true));
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* amf_cfg_upd_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"PrintableString",
                                  "ServedGUAMIList",
                                  "INTEGER (0..255)",
                                  "PLMNSupportList",
                                  "AMF-TNLAssociationToAddList",
                                  "AMF-TNLAssociationToRemoveList",
                                  "AMF-TNLAssociationToUpdateList"};
  return convert_enum_idx(options, 7, value, "amf_cfg_upd_ies_o::value_c::types");
}
uint8_t amf_cfg_upd_ies_o::value_c::types_opts::to_number() const
{
  if (value == relative_amf_capacity) {
    return 0;
  }
  invalid_enum_number(value, "amf_cfg_upd_ies_o::value_c::types");
  return 0;
}

template struct asn1::protocol_ie_field_s<amf_cfg_upd_ies_o>;

amf_cfg_upd_ies_container::amf_cfg_upd_ies_container() :
  amf_name(1, crit_e::reject),
  served_guami_list(96, crit_e::reject),
  relative_amf_capacity(86, crit_e::ignore),
  plmn_support_list(80, crit_e::reject),
  amf_tnlassoc_to_add_list(6, crit_e::ignore),
  amf_tnlassoc_to_rem_list(7, crit_e::ignore),
  amf_tnlassoc_to_upd_list(8, crit_e::ignore)
{}
SRSASN_CODE amf_cfg_upd_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += amf_name_present ? 1 : 0;
  nof_ies += served_guami_list_present ? 1 : 0;
  nof_ies += relative_amf_capacity_present ? 1 : 0;
  nof_ies += plmn_support_list_present ? 1 : 0;
  nof_ies += amf_tnlassoc_to_add_list_present ? 1 : 0;
  nof_ies += amf_tnlassoc_to_rem_list_present ? 1 : 0;
  nof_ies += amf_tnlassoc_to_upd_list_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  if (amf_name_present) {
    HANDLE_CODE(amf_name.pack(bref));
  }
  if (served_guami_list_present) {
    HANDLE_CODE(served_guami_list.pack(bref));
  }
  if (relative_amf_capacity_present) {
    HANDLE_CODE(relative_amf_capacity.pack(bref));
  }
  if (plmn_support_list_present) {
    HANDLE_CODE(plmn_support_list.pack(bref));
  }
  if (amf_tnlassoc_to_add_list_present) {
    HANDLE_CODE(amf_tnlassoc_to_add_list.pack(bref));
  }
  if (amf_tnlassoc_to_rem_list_present) {
    HANDLE_CODE(amf_tnlassoc_to_rem_list.pack(bref));
  }
  if (amf_tnlassoc_to_upd_list_present) {
    HANDLE_CODE(amf_tnlassoc_to_upd_list.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE amf_cfg_upd_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 1: {
        amf_name_present = true;
        amf_name.id      = id;
        HANDLE_CODE(amf_name.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_name.value.unpack(bref));
        break;
      }
      case 96: {
        served_guami_list_present = true;
        served_guami_list.id      = id;
        HANDLE_CODE(served_guami_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(served_guami_list.value.unpack(bref));
        break;
      }
      case 86: {
        relative_amf_capacity_present = true;
        relative_amf_capacity.id      = id;
        HANDLE_CODE(relative_amf_capacity.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(relative_amf_capacity.value.unpack(bref));
        break;
      }
      case 80: {
        plmn_support_list_present = true;
        plmn_support_list.id      = id;
        HANDLE_CODE(plmn_support_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(plmn_support_list.value.unpack(bref));
        break;
      }
      case 6: {
        amf_tnlassoc_to_add_list_present = true;
        amf_tnlassoc_to_add_list.id      = id;
        HANDLE_CODE(amf_tnlassoc_to_add_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_tnlassoc_to_add_list.value.unpack(bref));
        break;
      }
      case 7: {
        amf_tnlassoc_to_rem_list_present = true;
        amf_tnlassoc_to_rem_list.id      = id;
        HANDLE_CODE(amf_tnlassoc_to_rem_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_tnlassoc_to_rem_list.value.unpack(bref));
        break;
      }
      case 8: {
        amf_tnlassoc_to_upd_list_present = true;
        amf_tnlassoc_to_upd_list.id      = id;
        HANDLE_CODE(amf_tnlassoc_to_upd_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_tnlassoc_to_upd_list.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void amf_cfg_upd_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (amf_name_present) {
    j.write_fieldname("");
    amf_name.to_json(j);
  }
  if (served_guami_list_present) {
    j.write_fieldname("");
    served_guami_list.to_json(j);
  }
  if (relative_amf_capacity_present) {
    j.write_fieldname("");
    relative_amf_capacity.to_json(j);
  }
  if (plmn_support_list_present) {
    j.write_fieldname("");
    plmn_support_list.to_json(j);
  }
  if (amf_tnlassoc_to_add_list_present) {
    j.write_fieldname("");
    amf_tnlassoc_to_add_list.to_json(j);
  }
  if (amf_tnlassoc_to_rem_list_present) {
    j.write_fieldname("");
    amf_tnlassoc_to_rem_list.to_json(j);
  }
  if (amf_tnlassoc_to_upd_list_present) {
    j.write_fieldname("");
    amf_tnlassoc_to_upd_list.to_json(j);
  }
  j.end_obj();
}

// CauseMisc ::= ENUMERATED
const char* cause_misc_opts::to_string() const
{
  static const char* options[] = {"control-processing-overload",
                                  "not-enough-user-plane-processing-resources",
                                  "hardware-failure",
                                  "om-intervention",
                                  "unknown-PLMN",
                                  "unspecified"};
  return convert_enum_idx(options, 6, value, "cause_misc_e");
}

// CauseNas ::= ENUMERATED
const char* cause_nas_opts::to_string() const
{
  static const char* options[] = {"normal-release", "authentication-failure", "deregister", "unspecified"};
  return convert_enum_idx(options, 4, value, "cause_nas_e");
}

// CauseProtocol ::= ENUMERATED
const char* cause_protocol_opts::to_string() const
{
  static const char* options[] = {"transfer-syntax-error",
                                  "abstract-syntax-error-reject",
                                  "abstract-syntax-error-ignore-and-notify",
                                  "message-not-compatible-with-receiver-state",
                                  "semantic-error",
                                  "abstract-syntax-error-falsely-constructed-message",
                                  "unspecified"};
  return convert_enum_idx(options, 7, value, "cause_protocol_e");
}

// CauseRadioNetwork ::= ENUMERATED
const char* cause_radio_network_opts::to_string() const
{
  static const char* options[] = {"unspecified",
                                  "txnrelocoverall-expiry",
                                  "successful-handover",
                                  "release-due-to-ngran-generated-reason",
                                  "release-due-to-5gc-generated-reason",
                                  "handover-cancelled",
                                  "partial-handover",
                                  "ho-failure-in-target-5GC-ngran-node-or-target-system",
                                  "ho-target-not-allowed",
                                  "tngrelocoverall-expiry",
                                  "tngrelocprep-expiry",
                                  "cell-not-available",
                                  "unknown-targetID",
                                  "no-radio-resources-available-in-target-cell",
                                  "unknown-local-UE-NGAP-ID",
                                  "inconsistent-remote-UE-NGAP-ID",
                                  "handover-desirable-for-radio-reason",
                                  "time-critical-handover",
                                  "resource-optimisation-handover",
                                  "reduce-load-in-serving-cell",
                                  "user-inactivity",
                                  "radio-connection-with-ue-lost",
                                  "radio-resources-not-available",
                                  "invalid-qos-combination",
                                  "failure-in-radio-interface-procedure",
                                  "interaction-with-other-procedure",
                                  "unknown-PDU-session-ID",
                                  "unkown-qos-flow-ID",
                                  "multiple-PDU-session-ID-instances",
                                  "multiple-qos-flow-ID-instances",
                                  "encryption-and-or-integrity-protection-algorithms-not-supported",
                                  "ng-intra-system-handover-triggered",
                                  "ng-inter-system-handover-triggered",
                                  "xn-handover-triggered",
                                  "not-supported-5QI-value",
                                  "ue-context-transfer",
                                  "ims-voice-eps-fallback-or-rat-fallback-triggered",
                                  "up-integrity-protection-not-possible",
                                  "up-confidentiality-protection-not-possible",
                                  "slice-not-supported",
                                  "ue-in-rrc-inactive-state-not-reachable",
                                  "redirection",
                                  "resources-not-available-for-the-slice",
                                  "ue-max-integrity-protected-data-rate-reason",
                                  "release-due-to-cn-detected-mobility",
                                  "n26-interface-not-available",
                                  "release-due-to-pre-emption"};
  return convert_enum_idx(options, 47, value, "cause_radio_network_e");
}

// CauseTransport ::= ENUMERATED
const char* cause_transport_opts::to_string() const
{
  static const char* options[] = {"transport-resource-unavailable", "unspecified"};
  return convert_enum_idx(options, 2, value, "cause_transport_e");
}

// TypeOfError ::= ENUMERATED
const char* type_of_error_opts::to_string() const
{
  static const char* options[] = {"not-understood", "missing"};
  return convert_enum_idx(options, 2, value, "type_of_error_e");
}

// Cause ::= CHOICE
void cause_c::destroy_()
{
  switch (type_) {
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<cause_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void cause_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::radio_network:
      break;
    case types::transport:
      break;
    case types::nas:
      break;
    case types::protocol:
      break;
    case types::misc:
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<cause_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cause_c");
  }
}
cause_c::cause_c(const cause_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::radio_network:
      c.init(other.c.get<cause_radio_network_e>());
      break;
    case types::transport:
      c.init(other.c.get<cause_transport_e>());
      break;
    case types::nas:
      c.init(other.c.get<cause_nas_e>());
      break;
    case types::protocol:
      c.init(other.c.get<cause_protocol_e>());
      break;
    case types::misc:
      c.init(other.c.get<cause_misc_e>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<cause_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cause_c");
  }
}
cause_c& cause_c::operator=(const cause_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::radio_network:
      c.set(other.c.get<cause_radio_network_e>());
      break;
    case types::transport:
      c.set(other.c.get<cause_transport_e>());
      break;
    case types::nas:
      c.set(other.c.get<cause_nas_e>());
      break;
    case types::protocol:
      c.set(other.c.get<cause_protocol_e>());
      break;
    case types::misc:
      c.set(other.c.get<cause_misc_e>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<cause_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cause_c");
  }

  return *this;
}
cause_radio_network_e& cause_c::set_radio_network()
{
  set(types::radio_network);
  return c.get<cause_radio_network_e>();
}
cause_transport_e& cause_c::set_transport()
{
  set(types::transport);
  return c.get<cause_transport_e>();
}
cause_nas_e& cause_c::set_nas()
{
  set(types::nas);
  return c.get<cause_nas_e>();
}
cause_protocol_e& cause_c::set_protocol()
{
  set(types::protocol);
  return c.get<cause_protocol_e>();
}
cause_misc_e& cause_c::set_misc()
{
  set(types::misc);
  return c.get<cause_misc_e>();
}
protocol_ie_single_container_s<cause_ext_ies_o>& cause_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<cause_ext_ies_o> >();
}
void cause_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::radio_network:
      j.write_str("radioNetwork", c.get<cause_radio_network_e>().to_string());
      break;
    case types::transport:
      j.write_str("transport", c.get<cause_transport_e>().to_string());
      break;
    case types::nas:
      j.write_str("nas", c.get<cause_nas_e>().to_string());
      break;
    case types::protocol:
      j.write_str("protocol", c.get<cause_protocol_e>().to_string());
      break;
    case types::misc:
      j.write_str("misc", c.get<cause_misc_e>().to_string());
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<cause_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "cause_c");
  }
  j.end_obj();
}
SRSASN_CODE cause_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::radio_network:
      HANDLE_CODE(c.get<cause_radio_network_e>().pack(bref));
      break;
    case types::transport:
      HANDLE_CODE(c.get<cause_transport_e>().pack(bref));
      break;
    case types::nas:
      HANDLE_CODE(c.get<cause_nas_e>().pack(bref));
      break;
    case types::protocol:
      HANDLE_CODE(c.get<cause_protocol_e>().pack(bref));
      break;
    case types::misc:
      HANDLE_CODE(c.get<cause_misc_e>().pack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<cause_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cause_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cause_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::radio_network:
      HANDLE_CODE(c.get<cause_radio_network_e>().unpack(bref));
      break;
    case types::transport:
      HANDLE_CODE(c.get<cause_transport_e>().unpack(bref));
      break;
    case types::nas:
      HANDLE_CODE(c.get<cause_nas_e>().unpack(bref));
      break;
    case types::protocol:
      HANDLE_CODE(c.get<cause_protocol_e>().unpack(bref));
      break;
    case types::misc:
      HANDLE_CODE(c.get<cause_misc_e>().unpack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<cause_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cause_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* cause_c::types_opts::to_string() const
{
  static const char* options[] = {"radioNetwork", "transport", "nas", "protocol", "misc", "choice-Extensions"};
  return convert_enum_idx(options, 6, value, "cause_c::types");
}
uint8_t cause_c::types_opts::to_number() const
{
  static const uint8_t options[] = {2};
  return map_enum_number(options, 1, value, "cause_c::types");
}

// CriticalityDiagnostics-IE-Item ::= SEQUENCE
SRSASN_CODE crit_diagnostics_ie_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(iecrit.pack(bref));
  HANDLE_CODE(pack_integer(bref, ie_id, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(type_of_error.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE crit_diagnostics_ie_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(iecrit.unpack(bref));
  HANDLE_CODE(unpack_integer(ie_id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  HANDLE_CODE(type_of_error.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void crit_diagnostics_ie_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("iECriticality", iecrit.to_string());
  j.write_int("iE-ID", ie_id);
  j.write_str("typeOfError", type_of_error.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TNLAssociationItem ::= SEQUENCE
SRSASN_CODE tnlassoc_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(tnlassoc_address.pack(bref));
  HANDLE_CODE(cause.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tnlassoc_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(tnlassoc_address.unpack(bref));
  HANDLE_CODE(cause.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tnlassoc_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("tNLAssociationAddress");
  tnlassoc_address.to_json(j);
  j.write_fieldname("cause");
  cause.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TriggeringMessage ::= ENUMERATED
const char* trigger_msg_opts::to_string() const
{
  static const char* options[] = {"initiating-message", "successful-outcome", "unsuccessfull-outcome"};
  return convert_enum_idx(options, 3, value, "trigger_msg_e");
}

// CriticalityDiagnostics ::= SEQUENCE
SRSASN_CODE crit_diagnostics_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(proc_code_present, 1));
  HANDLE_CODE(bref.pack(trigger_msg_present, 1));
  HANDLE_CODE(bref.pack(proc_crit_present, 1));
  HANDLE_CODE(bref.pack(ies_crit_diagnostics.size() > 0, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (proc_code_present) {
    HANDLE_CODE(pack_integer(bref, proc_code, (uint16_t)0u, (uint16_t)255u, false, true));
  }
  if (trigger_msg_present) {
    HANDLE_CODE(trigger_msg.pack(bref));
  }
  if (proc_crit_present) {
    HANDLE_CODE(proc_crit.pack(bref));
  }
  if (ies_crit_diagnostics.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ies_crit_diagnostics, 1, 256, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE crit_diagnostics_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(proc_code_present, 1));
  HANDLE_CODE(bref.unpack(trigger_msg_present, 1));
  HANDLE_CODE(bref.unpack(proc_crit_present, 1));
  bool ies_crit_diagnostics_present;
  HANDLE_CODE(bref.unpack(ies_crit_diagnostics_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (proc_code_present) {
    HANDLE_CODE(unpack_integer(proc_code, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  }
  if (trigger_msg_present) {
    HANDLE_CODE(trigger_msg.unpack(bref));
  }
  if (proc_crit_present) {
    HANDLE_CODE(proc_crit.unpack(bref));
  }
  if (ies_crit_diagnostics_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ies_crit_diagnostics, bref, 1, 256, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void crit_diagnostics_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (proc_code_present) {
    j.write_int("procedureCode", proc_code);
  }
  if (trigger_msg_present) {
    j.write_str("triggeringMessage", trigger_msg.to_string());
  }
  if (proc_crit_present) {
    j.write_str("procedureCriticality", proc_crit.to_string());
  }
  if (ies_crit_diagnostics.size() > 0) {
    j.start_array("iEsCriticalityDiagnostics");
    for (const auto& e1 : ies_crit_diagnostics) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AMFConfigurationUpdateAcknowledgeIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t amf_cfg_upd_ack_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {5, 4, 19};
  return map_enum_number(options, 3, idx, "id");
}
bool amf_cfg_upd_ack_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {5, 4, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e amf_cfg_upd_ack_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 5:
      return crit_e::ignore;
    case 4:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
amf_cfg_upd_ack_ies_o::value_c amf_cfg_upd_ack_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 5:
      ret.set(value_c::types::amf_tnlassoc_setup_list);
      break;
    case 4:
      ret.set(value_c::types::amf_tnlassoc_failed_to_setup_list);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e amf_cfg_upd_ack_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 5:
      return presence_e::optional;
    case 4:
      return presence_e::optional;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void amf_cfg_upd_ack_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::amf_tnlassoc_setup_list:
      c.destroy<amf_tnlassoc_setup_list_l>();
      break;
    case types::amf_tnlassoc_failed_to_setup_list:
      c.destroy<tnlassoc_list_l>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void amf_cfg_upd_ack_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_tnlassoc_setup_list:
      c.init<amf_tnlassoc_setup_list_l>();
      break;
    case types::amf_tnlassoc_failed_to_setup_list:
      c.init<tnlassoc_list_l>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_ack_ies_o::value_c");
  }
}
amf_cfg_upd_ack_ies_o::value_c::value_c(const amf_cfg_upd_ack_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_tnlassoc_setup_list:
      c.init(other.c.get<amf_tnlassoc_setup_list_l>());
      break;
    case types::amf_tnlassoc_failed_to_setup_list:
      c.init(other.c.get<tnlassoc_list_l>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_ack_ies_o::value_c");
  }
}
amf_cfg_upd_ack_ies_o::value_c& amf_cfg_upd_ack_ies_o::value_c::operator=(const amf_cfg_upd_ack_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_tnlassoc_setup_list:
      c.set(other.c.get<amf_tnlassoc_setup_list_l>());
      break;
    case types::amf_tnlassoc_failed_to_setup_list:
      c.set(other.c.get<tnlassoc_list_l>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_ack_ies_o::value_c");
  }

  return *this;
}
amf_tnlassoc_setup_list_l& amf_cfg_upd_ack_ies_o::value_c::amf_tnlassoc_setup_list()
{
  assert_choice_type(types::amf_tnlassoc_setup_list, type_, "Value");
  return c.get<amf_tnlassoc_setup_list_l>();
}
tnlassoc_list_l& amf_cfg_upd_ack_ies_o::value_c::amf_tnlassoc_failed_to_setup_list()
{
  assert_choice_type(types::amf_tnlassoc_failed_to_setup_list, type_, "Value");
  return c.get<tnlassoc_list_l>();
}
crit_diagnostics_s& amf_cfg_upd_ack_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const amf_tnlassoc_setup_list_l& amf_cfg_upd_ack_ies_o::value_c::amf_tnlassoc_setup_list() const
{
  assert_choice_type(types::amf_tnlassoc_setup_list, type_, "Value");
  return c.get<amf_tnlassoc_setup_list_l>();
}
const tnlassoc_list_l& amf_cfg_upd_ack_ies_o::value_c::amf_tnlassoc_failed_to_setup_list() const
{
  assert_choice_type(types::amf_tnlassoc_failed_to_setup_list, type_, "Value");
  return c.get<tnlassoc_list_l>();
}
const crit_diagnostics_s& amf_cfg_upd_ack_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void amf_cfg_upd_ack_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_tnlassoc_setup_list:
      j.start_array("AMF-TNLAssociationSetupList");
      for (const auto& e1 : c.get<amf_tnlassoc_setup_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::amf_tnlassoc_failed_to_setup_list:
      j.start_array("TNLAssociationList");
      for (const auto& e1 : c.get<tnlassoc_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_ack_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE amf_cfg_upd_ack_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_tnlassoc_setup_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<amf_tnlassoc_setup_list_l>(), 1, 32, true));
      break;
    case types::amf_tnlassoc_failed_to_setup_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<tnlassoc_list_l>(), 1, 32, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_ack_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE amf_cfg_upd_ack_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_tnlassoc_setup_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<amf_tnlassoc_setup_list_l>(), bref, 1, 32, true));
      break;
    case types::amf_tnlassoc_failed_to_setup_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<tnlassoc_list_l>(), bref, 1, 32, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_ack_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* amf_cfg_upd_ack_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"AMF-TNLAssociationSetupList", "TNLAssociationList", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 3, value, "amf_cfg_upd_ack_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<amf_cfg_upd_ack_ies_o>;

amf_cfg_upd_ack_ies_container::amf_cfg_upd_ack_ies_container() :
  amf_tnlassoc_setup_list(5, crit_e::ignore),
  amf_tnlassoc_failed_to_setup_list(4, crit_e::ignore),
  crit_diagnostics(19, crit_e::ignore)
{}
SRSASN_CODE amf_cfg_upd_ack_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += amf_tnlassoc_setup_list_present ? 1 : 0;
  nof_ies += amf_tnlassoc_failed_to_setup_list_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  if (amf_tnlassoc_setup_list_present) {
    HANDLE_CODE(amf_tnlassoc_setup_list.pack(bref));
  }
  if (amf_tnlassoc_failed_to_setup_list_present) {
    HANDLE_CODE(amf_tnlassoc_failed_to_setup_list.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE amf_cfg_upd_ack_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 5: {
        amf_tnlassoc_setup_list_present = true;
        amf_tnlassoc_setup_list.id      = id;
        HANDLE_CODE(amf_tnlassoc_setup_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_tnlassoc_setup_list.value.unpack(bref));
        break;
      }
      case 4: {
        amf_tnlassoc_failed_to_setup_list_present = true;
        amf_tnlassoc_failed_to_setup_list.id      = id;
        HANDLE_CODE(amf_tnlassoc_failed_to_setup_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_tnlassoc_failed_to_setup_list.value.unpack(bref));
        break;
      }
      case 19: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void amf_cfg_upd_ack_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (amf_tnlassoc_setup_list_present) {
    j.write_fieldname("");
    amf_tnlassoc_setup_list.to_json(j);
  }
  if (amf_tnlassoc_failed_to_setup_list_present) {
    j.write_fieldname("");
    amf_tnlassoc_failed_to_setup_list.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// TimeToWait ::= ENUMERATED
const char* time_to_wait_opts::to_string() const
{
  static const char* options[] = {"v1s", "v2s", "v5s", "v10s", "v20s", "v60s"};
  return convert_enum_idx(options, 6, value, "time_to_wait_e");
}
uint8_t time_to_wait_opts::to_number() const
{
  static const uint8_t options[] = {1, 2, 5, 10, 20, 60};
  return map_enum_number(options, 6, value, "time_to_wait_e");
}

// AMFConfigurationUpdateFailureIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t amf_cfg_upd_fail_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {15, 107, 19};
  return map_enum_number(options, 3, idx, "id");
}
bool amf_cfg_upd_fail_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {15, 107, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e amf_cfg_upd_fail_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 15:
      return crit_e::ignore;
    case 107:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
amf_cfg_upd_fail_ies_o::value_c amf_cfg_upd_fail_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 15:
      ret.set(value_c::types::cause);
      break;
    case 107:
      ret.set(value_c::types::time_to_wait);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e amf_cfg_upd_fail_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 15:
      return presence_e::mandatory;
    case 107:
      return presence_e::optional;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void amf_cfg_upd_fail_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void amf_cfg_upd_fail_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cause:
      c.init<cause_c>();
      break;
    case types::time_to_wait:
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_fail_ies_o::value_c");
  }
}
amf_cfg_upd_fail_ies_o::value_c::value_c(const amf_cfg_upd_fail_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::time_to_wait:
      c.init(other.c.get<time_to_wait_e>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_fail_ies_o::value_c");
  }
}
amf_cfg_upd_fail_ies_o::value_c&
amf_cfg_upd_fail_ies_o::value_c::operator=(const amf_cfg_upd_fail_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::time_to_wait:
      c.set(other.c.get<time_to_wait_e>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_fail_ies_o::value_c");
  }

  return *this;
}
cause_c& amf_cfg_upd_fail_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
time_to_wait_e& amf_cfg_upd_fail_ies_o::value_c::time_to_wait()
{
  assert_choice_type(types::time_to_wait, type_, "Value");
  return c.get<time_to_wait_e>();
}
crit_diagnostics_s& amf_cfg_upd_fail_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const cause_c& amf_cfg_upd_fail_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const time_to_wait_e& amf_cfg_upd_fail_ies_o::value_c::time_to_wait() const
{
  assert_choice_type(types::time_to_wait, type_, "Value");
  return c.get<time_to_wait_e>();
}
const crit_diagnostics_s& amf_cfg_upd_fail_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void amf_cfg_upd_fail_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::time_to_wait:
      j.write_str("TimeToWait", c.get<time_to_wait_e>().to_string());
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_fail_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE amf_cfg_upd_fail_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::time_to_wait:
      HANDLE_CODE(c.get<time_to_wait_e>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_fail_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE amf_cfg_upd_fail_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::time_to_wait:
      HANDLE_CODE(c.get<time_to_wait_e>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "amf_cfg_upd_fail_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* amf_cfg_upd_fail_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"Cause", "TimeToWait", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 3, value, "amf_cfg_upd_fail_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<amf_cfg_upd_fail_ies_o>;

amf_cfg_upd_fail_ies_container::amf_cfg_upd_fail_ies_container() :
  cause(15, crit_e::ignore), time_to_wait(107, crit_e::ignore), crit_diagnostics(19, crit_e::ignore)
{}
SRSASN_CODE amf_cfg_upd_fail_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 1;
  nof_ies += time_to_wait_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(cause.pack(bref));
  if (time_to_wait_present) {
    HANDLE_CODE(time_to_wait.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE amf_cfg_upd_fail_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 1;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 15: {
        nof_mandatory_ies--;
        cause.id = id;
        HANDLE_CODE(cause.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cause.value.unpack(bref));
        break;
      }
      case 107: {
        time_to_wait_present = true;
        time_to_wait.id      = id;
        HANDLE_CODE(time_to_wait.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(time_to_wait.value.unpack(bref));
        break;
      }
      case 19: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void amf_cfg_upd_fail_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  cause.to_json(j);
  if (time_to_wait_present) {
    j.write_fieldname("");
    time_to_wait.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// GNB-ID ::= CHOICE
void gnb_id_c::destroy_()
{
  switch (type_) {
    case types::gnb_id:
      c.destroy<bounded_bitstring<22, 32, false, true> >();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<gnb_id_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void gnb_id_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::gnb_id:
      c.init<bounded_bitstring<22, 32, false, true> >();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<gnb_id_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "gnb_id_c");
  }
}
gnb_id_c::gnb_id_c(const gnb_id_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::gnb_id:
      c.init(other.c.get<bounded_bitstring<22, 32, false, true> >());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<gnb_id_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "gnb_id_c");
  }
}
gnb_id_c& gnb_id_c::operator=(const gnb_id_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::gnb_id:
      c.set(other.c.get<bounded_bitstring<22, 32, false, true> >());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<gnb_id_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "gnb_id_c");
  }

  return *this;
}
bounded_bitstring<22, 32, false, true>& gnb_id_c::set_gnb_id()
{
  set(types::gnb_id);
  return c.get<bounded_bitstring<22, 32, false, true> >();
}
protocol_ie_single_container_s<gnb_id_ext_ies_o>& gnb_id_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<gnb_id_ext_ies_o> >();
}
void gnb_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::gnb_id:
      j.write_str("gNB-ID", c.get<bounded_bitstring<22, 32, false, true> >().to_string());
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<gnb_id_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "gnb_id_c");
  }
  j.end_obj();
}
SRSASN_CODE gnb_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::gnb_id:
      HANDLE_CODE((c.get<bounded_bitstring<22, 32, false, true> >().pack(bref)));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<gnb_id_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "gnb_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE gnb_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::gnb_id:
      HANDLE_CODE((c.get<bounded_bitstring<22, 32, false, true> >().unpack(bref)));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<gnb_id_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "gnb_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* gnb_id_c::types_opts::to_string() const
{
  static const char* options[] = {"gNB-ID", "choice-Extensions"};
  return convert_enum_idx(options, 2, value, "gnb_id_c::types");
}

// N3IWF-ID ::= CHOICE
void n3_iwf_id_c::destroy_()
{
  switch (type_) {
    case types::n3_iwf_id:
      c.destroy<fixed_bitstring<16, false, true> >();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<n3_iwf_id_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void n3_iwf_id_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::n3_iwf_id:
      c.init<fixed_bitstring<16, false, true> >();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<n3_iwf_id_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "n3_iwf_id_c");
  }
}
n3_iwf_id_c::n3_iwf_id_c(const n3_iwf_id_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::n3_iwf_id:
      c.init(other.c.get<fixed_bitstring<16, false, true> >());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<n3_iwf_id_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "n3_iwf_id_c");
  }
}
n3_iwf_id_c& n3_iwf_id_c::operator=(const n3_iwf_id_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::n3_iwf_id:
      c.set(other.c.get<fixed_bitstring<16, false, true> >());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<n3_iwf_id_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "n3_iwf_id_c");
  }

  return *this;
}
fixed_bitstring<16, false, true>& n3_iwf_id_c::set_n3_iwf_id()
{
  set(types::n3_iwf_id);
  return c.get<fixed_bitstring<16, false, true> >();
}
protocol_ie_single_container_s<n3_iwf_id_ext_ies_o>& n3_iwf_id_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<n3_iwf_id_ext_ies_o> >();
}
void n3_iwf_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::n3_iwf_id:
      j.write_str("n3IWF-ID", c.get<fixed_bitstring<16, false, true> >().to_string());
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<n3_iwf_id_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "n3_iwf_id_c");
  }
  j.end_obj();
}
SRSASN_CODE n3_iwf_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::n3_iwf_id:
      HANDLE_CODE((c.get<fixed_bitstring<16, false, true> >().pack(bref)));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<n3_iwf_id_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "n3_iwf_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE n3_iwf_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::n3_iwf_id:
      HANDLE_CODE((c.get<fixed_bitstring<16, false, true> >().unpack(bref)));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<n3_iwf_id_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "n3_iwf_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* n3_iwf_id_c::types_opts::to_string() const
{
  static const char* options[] = {"n3IWF-ID", "choice-Extensions"};
  return convert_enum_idx(options, 2, value, "n3_iwf_id_c::types");
}
uint8_t n3_iwf_id_c::types_opts::to_number() const
{
  static const uint8_t options[] = {3};
  return map_enum_number(options, 1, value, "n3_iwf_id_c::types");
}

// NgENB-ID ::= CHOICE
void ng_enb_id_c::destroy_()
{
  switch (type_) {
    case types::macro_ng_enb_id:
      c.destroy<fixed_bitstring<20, false, true> >();
      break;
    case types::short_macro_ng_enb_id:
      c.destroy<fixed_bitstring<18, false, true> >();
      break;
    case types::long_macro_ng_enb_id:
      c.destroy<fixed_bitstring<21, false, true> >();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<ng_enb_id_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void ng_enb_id_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::macro_ng_enb_id:
      c.init<fixed_bitstring<20, false, true> >();
      break;
    case types::short_macro_ng_enb_id:
      c.init<fixed_bitstring<18, false, true> >();
      break;
    case types::long_macro_ng_enb_id:
      c.init<fixed_bitstring<21, false, true> >();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<ng_enb_id_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_enb_id_c");
  }
}
ng_enb_id_c::ng_enb_id_c(const ng_enb_id_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::macro_ng_enb_id:
      c.init(other.c.get<fixed_bitstring<20, false, true> >());
      break;
    case types::short_macro_ng_enb_id:
      c.init(other.c.get<fixed_bitstring<18, false, true> >());
      break;
    case types::long_macro_ng_enb_id:
      c.init(other.c.get<fixed_bitstring<21, false, true> >());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<ng_enb_id_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_enb_id_c");
  }
}
ng_enb_id_c& ng_enb_id_c::operator=(const ng_enb_id_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::macro_ng_enb_id:
      c.set(other.c.get<fixed_bitstring<20, false, true> >());
      break;
    case types::short_macro_ng_enb_id:
      c.set(other.c.get<fixed_bitstring<18, false, true> >());
      break;
    case types::long_macro_ng_enb_id:
      c.set(other.c.get<fixed_bitstring<21, false, true> >());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<ng_enb_id_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_enb_id_c");
  }

  return *this;
}
fixed_bitstring<20, false, true>& ng_enb_id_c::set_macro_ng_enb_id()
{
  set(types::macro_ng_enb_id);
  return c.get<fixed_bitstring<20, false, true> >();
}
fixed_bitstring<18, false, true>& ng_enb_id_c::set_short_macro_ng_enb_id()
{
  set(types::short_macro_ng_enb_id);
  return c.get<fixed_bitstring<18, false, true> >();
}
fixed_bitstring<21, false, true>& ng_enb_id_c::set_long_macro_ng_enb_id()
{
  set(types::long_macro_ng_enb_id);
  return c.get<fixed_bitstring<21, false, true> >();
}
protocol_ie_single_container_s<ng_enb_id_ext_ies_o>& ng_enb_id_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<ng_enb_id_ext_ies_o> >();
}
void ng_enb_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::macro_ng_enb_id:
      j.write_str("macroNgENB-ID", c.get<fixed_bitstring<20, false, true> >().to_string());
      break;
    case types::short_macro_ng_enb_id:
      j.write_str("shortMacroNgENB-ID", c.get<fixed_bitstring<18, false, true> >().to_string());
      break;
    case types::long_macro_ng_enb_id:
      j.write_str("longMacroNgENB-ID", c.get<fixed_bitstring<21, false, true> >().to_string());
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<ng_enb_id_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ng_enb_id_c");
  }
  j.end_obj();
}
SRSASN_CODE ng_enb_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::macro_ng_enb_id:
      HANDLE_CODE((c.get<fixed_bitstring<20, false, true> >().pack(bref)));
      break;
    case types::short_macro_ng_enb_id:
      HANDLE_CODE((c.get<fixed_bitstring<18, false, true> >().pack(bref)));
      break;
    case types::long_macro_ng_enb_id:
      HANDLE_CODE((c.get<fixed_bitstring<21, false, true> >().pack(bref)));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<ng_enb_id_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ng_enb_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ng_enb_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::macro_ng_enb_id:
      HANDLE_CODE((c.get<fixed_bitstring<20, false, true> >().unpack(bref)));
      break;
    case types::short_macro_ng_enb_id:
      HANDLE_CODE((c.get<fixed_bitstring<18, false, true> >().unpack(bref)));
      break;
    case types::long_macro_ng_enb_id:
      HANDLE_CODE((c.get<fixed_bitstring<21, false, true> >().unpack(bref)));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<ng_enb_id_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ng_enb_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ng_enb_id_c::types_opts::to_string() const
{
  static const char* options[] = {"macroNgENB-ID", "shortMacroNgENB-ID", "longMacroNgENB-ID", "choice-Extensions"};
  return convert_enum_idx(options, 4, value, "ng_enb_id_c::types");
}

// GlobalGNB-ID ::= SEQUENCE
SRSASN_CODE global_gnb_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(gnb_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE global_gnb_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(gnb_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void global_gnb_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNIdentity", plmn_id.to_string());
  j.write_fieldname("gNB-ID");
  gnb_id.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// GlobalN3IWF-ID ::= SEQUENCE
SRSASN_CODE global_n3_iwf_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(n3_iwf_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE global_n3_iwf_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(n3_iwf_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void global_n3_iwf_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNIdentity", plmn_id.to_string());
  j.write_fieldname("n3IWF-ID");
  n3_iwf_id.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// GlobalNgENB-ID ::= SEQUENCE
SRSASN_CODE global_ng_enb_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(ng_enb_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE global_ng_enb_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(ng_enb_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void global_ng_enb_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNIdentity", plmn_id.to_string());
  j.write_fieldname("ngENB-ID");
  ng_enb_id.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// GlobalRANNodeID ::= CHOICE
void global_ran_node_id_c::destroy_()
{
  switch (type_) {
    case types::global_gnb_id:
      c.destroy<global_gnb_id_s>();
      break;
    case types::global_ng_enb_id:
      c.destroy<global_ng_enb_id_s>();
      break;
    case types::global_n3_iwf_id:
      c.destroy<global_n3_iwf_id_s>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<global_ran_node_id_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void global_ran_node_id_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::global_gnb_id:
      c.init<global_gnb_id_s>();
      break;
    case types::global_ng_enb_id:
      c.init<global_ng_enb_id_s>();
      break;
    case types::global_n3_iwf_id:
      c.init<global_n3_iwf_id_s>();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<global_ran_node_id_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "global_ran_node_id_c");
  }
}
global_ran_node_id_c::global_ran_node_id_c(const global_ran_node_id_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::global_gnb_id:
      c.init(other.c.get<global_gnb_id_s>());
      break;
    case types::global_ng_enb_id:
      c.init(other.c.get<global_ng_enb_id_s>());
      break;
    case types::global_n3_iwf_id:
      c.init(other.c.get<global_n3_iwf_id_s>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<global_ran_node_id_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "global_ran_node_id_c");
  }
}
global_ran_node_id_c& global_ran_node_id_c::operator=(const global_ran_node_id_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::global_gnb_id:
      c.set(other.c.get<global_gnb_id_s>());
      break;
    case types::global_ng_enb_id:
      c.set(other.c.get<global_ng_enb_id_s>());
      break;
    case types::global_n3_iwf_id:
      c.set(other.c.get<global_n3_iwf_id_s>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<global_ran_node_id_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "global_ran_node_id_c");
  }

  return *this;
}
global_gnb_id_s& global_ran_node_id_c::set_global_gnb_id()
{
  set(types::global_gnb_id);
  return c.get<global_gnb_id_s>();
}
global_ng_enb_id_s& global_ran_node_id_c::set_global_ng_enb_id()
{
  set(types::global_ng_enb_id);
  return c.get<global_ng_enb_id_s>();
}
global_n3_iwf_id_s& global_ran_node_id_c::set_global_n3_iwf_id()
{
  set(types::global_n3_iwf_id);
  return c.get<global_n3_iwf_id_s>();
}
protocol_ie_single_container_s<global_ran_node_id_ext_ies_o>& global_ran_node_id_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<global_ran_node_id_ext_ies_o> >();
}
void global_ran_node_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::global_gnb_id:
      j.write_fieldname("globalGNB-ID");
      c.get<global_gnb_id_s>().to_json(j);
      break;
    case types::global_ng_enb_id:
      j.write_fieldname("globalNgENB-ID");
      c.get<global_ng_enb_id_s>().to_json(j);
      break;
    case types::global_n3_iwf_id:
      j.write_fieldname("globalN3IWF-ID");
      c.get<global_n3_iwf_id_s>().to_json(j);
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<global_ran_node_id_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "global_ran_node_id_c");
  }
  j.end_obj();
}
SRSASN_CODE global_ran_node_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::global_gnb_id:
      HANDLE_CODE(c.get<global_gnb_id_s>().pack(bref));
      break;
    case types::global_ng_enb_id:
      HANDLE_CODE(c.get<global_ng_enb_id_s>().pack(bref));
      break;
    case types::global_n3_iwf_id:
      HANDLE_CODE(c.get<global_n3_iwf_id_s>().pack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<global_ran_node_id_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "global_ran_node_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE global_ran_node_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::global_gnb_id:
      HANDLE_CODE(c.get<global_gnb_id_s>().unpack(bref));
      break;
    case types::global_ng_enb_id:
      HANDLE_CODE(c.get<global_ng_enb_id_s>().unpack(bref));
      break;
    case types::global_n3_iwf_id:
      HANDLE_CODE(c.get<global_n3_iwf_id_s>().unpack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<global_ran_node_id_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "global_ran_node_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* global_ran_node_id_c::types_opts::to_string() const
{
  static const char* options[] = {"globalGNB-ID", "globalNgENB-ID", "globalN3IWF-ID", "choice-Extensions"};
  return convert_enum_idx(options, 4, value, "global_ran_node_id_c::types");
}
uint8_t global_ran_node_id_c::types_opts::to_number() const
{
  if (value == global_n3_iwf_id) {
    return 3;
  }
  invalid_enum_number(value, "global_ran_node_id_c::types");
  return 0;
}

// TAI ::= SEQUENCE
SRSASN_CODE tai_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(tac.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(tac.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNIdentity", plmn_id.to_string());
  j.write_str("tAC", tac.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AMFPagingTarget ::= CHOICE
void amf_paging_target_c::destroy_()
{
  switch (type_) {
    case types::global_ran_node_id:
      c.destroy<global_ran_node_id_c>();
      break;
    case types::tai:
      c.destroy<tai_s>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<amf_paging_target_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void amf_paging_target_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::global_ran_node_id:
      c.init<global_ran_node_id_c>();
      break;
    case types::tai:
      c.init<tai_s>();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<amf_paging_target_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "amf_paging_target_c");
  }
}
amf_paging_target_c::amf_paging_target_c(const amf_paging_target_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::global_ran_node_id:
      c.init(other.c.get<global_ran_node_id_c>());
      break;
    case types::tai:
      c.init(other.c.get<tai_s>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<amf_paging_target_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "amf_paging_target_c");
  }
}
amf_paging_target_c& amf_paging_target_c::operator=(const amf_paging_target_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::global_ran_node_id:
      c.set(other.c.get<global_ran_node_id_c>());
      break;
    case types::tai:
      c.set(other.c.get<tai_s>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<amf_paging_target_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "amf_paging_target_c");
  }

  return *this;
}
global_ran_node_id_c& amf_paging_target_c::set_global_ran_node_id()
{
  set(types::global_ran_node_id);
  return c.get<global_ran_node_id_c>();
}
tai_s& amf_paging_target_c::set_tai()
{
  set(types::tai);
  return c.get<tai_s>();
}
protocol_ie_single_container_s<amf_paging_target_ext_ies_o>& amf_paging_target_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<amf_paging_target_ext_ies_o> >();
}
void amf_paging_target_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::global_ran_node_id:
      j.write_fieldname("globalRANNodeID");
      c.get<global_ran_node_id_c>().to_json(j);
      break;
    case types::tai:
      j.write_fieldname("tAI");
      c.get<tai_s>().to_json(j);
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<amf_paging_target_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "amf_paging_target_c");
  }
  j.end_obj();
}
SRSASN_CODE amf_paging_target_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::global_ran_node_id:
      HANDLE_CODE(c.get<global_ran_node_id_c>().pack(bref));
      break;
    case types::tai:
      HANDLE_CODE(c.get<tai_s>().pack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<amf_paging_target_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "amf_paging_target_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE amf_paging_target_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::global_ran_node_id:
      HANDLE_CODE(c.get<global_ran_node_id_c>().unpack(bref));
      break;
    case types::tai:
      HANDLE_CODE(c.get<tai_s>().unpack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<amf_paging_target_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "amf_paging_target_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* amf_paging_target_c::types_opts::to_string() const
{
  static const char* options[] = {"globalRANNodeID", "tAI", "choice-Extensions"};
  return convert_enum_idx(options, 3, value, "amf_paging_target_c::types");
}

// TimerApproachForGUAMIRemoval ::= ENUMERATED
const char* timer_approach_for_guami_removal_opts::to_string() const
{
  static const char* options[] = {"apply-timer"};
  return convert_enum_idx(options, 1, value, "timer_approach_for_guami_removal_e");
}

// UnavailableGUAMIItem ::= SEQUENCE
SRSASN_CODE unavailable_guami_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(timer_approach_for_guami_removal_present, 1));
  HANDLE_CODE(bref.pack(backup_amf_name_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(guami.pack(bref));
  if (timer_approach_for_guami_removal_present) {
    HANDLE_CODE(timer_approach_for_guami_removal.pack(bref));
  }
  if (backup_amf_name_present) {
    HANDLE_CODE(backup_amf_name.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE unavailable_guami_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(timer_approach_for_guami_removal_present, 1));
  HANDLE_CODE(bref.unpack(backup_amf_name_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(guami.unpack(bref));
  if (timer_approach_for_guami_removal_present) {
    HANDLE_CODE(timer_approach_for_guami_removal.unpack(bref));
  }
  if (backup_amf_name_present) {
    HANDLE_CODE(backup_amf_name.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void unavailable_guami_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("gUAMI");
  guami.to_json(j);
  if (timer_approach_for_guami_removal_present) {
    j.write_str("timerApproachForGUAMIRemoval", "apply-timer");
  }
  if (backup_amf_name_present) {
    j.write_str("backupAMFName", backup_amf_name.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AMFStatusIndicationIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t amf_status_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {120};
  return map_enum_number(options, 1, idx, "id");
}
bool amf_status_ind_ies_o::is_id_valid(const uint32_t& id)
{
  return 120 == id;
}
crit_e amf_status_ind_ies_o::get_crit(const uint32_t& id)
{
  if (id == 120) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
amf_status_ind_ies_o::value_c amf_status_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 120) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e amf_status_ind_ies_o::get_presence(const uint32_t& id)
{
  if (id == 120) {
    return presence_e::mandatory;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void amf_status_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("UnavailableGUAMIList");
  for (const auto& e1 : c) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}
SRSASN_CODE amf_status_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(pack_dyn_seq_of(bref, c, 1, 256, true));
  return SRSASN_SUCCESS;
}
SRSASN_CODE amf_status_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(unpack_dyn_seq_of(c, bref, 1, 256, true));
  return SRSASN_SUCCESS;
}

const char* amf_status_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"UnavailableGUAMIList"};
  return convert_enum_idx(options, 1, value, "amf_status_ind_ies_o::value_c::types");
}

// DataForwardingAccepted ::= ENUMERATED
const char* data_forwarding_accepted_opts::to_string() const
{
  static const char* options[] = {"data-forwarding-accepted"};
  return convert_enum_idx(options, 1, value, "data_forwarding_accepted_e");
}

// GTPTunnel ::= SEQUENCE
SRSASN_CODE gtp_tunnel_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(transport_layer_address.pack(bref));
  HANDLE_CODE(gtp_teid.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE gtp_tunnel_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(transport_layer_address.unpack(bref));
  HANDLE_CODE(gtp_teid.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void gtp_tunnel_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("transportLayerAddress", transport_layer_address.to_string());
  j.write_str("gTP-TEID", gtp_teid.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// QosFlowItemWithDataForwarding ::= SEQUENCE
SRSASN_CODE qos_flow_item_with_data_forwarding_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(data_forwarding_accepted_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, qos_flow_id, (uint8_t)0u, (uint8_t)63u, true, true));
  if (data_forwarding_accepted_present) {
    HANDLE_CODE(data_forwarding_accepted.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE qos_flow_item_with_data_forwarding_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(data_forwarding_accepted_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(qos_flow_id, bref, (uint8_t)0u, (uint8_t)63u, true, true));
  if (data_forwarding_accepted_present) {
    HANDLE_CODE(data_forwarding_accepted.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void qos_flow_item_with_data_forwarding_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("qosFlowIdentifier", qos_flow_id);
  if (data_forwarding_accepted_present) {
    j.write_str("dataForwardingAccepted", "data-forwarding-accepted");
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UPTransportLayerInformation ::= CHOICE
void up_transport_layer_info_c::destroy_()
{
  switch (type_) {
    case types::gtp_tunnel:
      c.destroy<gtp_tunnel_s>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<up_transport_layer_info_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void up_transport_layer_info_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::gtp_tunnel:
      c.init<gtp_tunnel_s>();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<up_transport_layer_info_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "up_transport_layer_info_c");
  }
}
up_transport_layer_info_c::up_transport_layer_info_c(const up_transport_layer_info_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::gtp_tunnel:
      c.init(other.c.get<gtp_tunnel_s>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<up_transport_layer_info_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "up_transport_layer_info_c");
  }
}
up_transport_layer_info_c& up_transport_layer_info_c::operator=(const up_transport_layer_info_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::gtp_tunnel:
      c.set(other.c.get<gtp_tunnel_s>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<up_transport_layer_info_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "up_transport_layer_info_c");
  }

  return *this;
}
gtp_tunnel_s& up_transport_layer_info_c::set_gtp_tunnel()
{
  set(types::gtp_tunnel);
  return c.get<gtp_tunnel_s>();
}
protocol_ie_single_container_s<up_transport_layer_info_ext_ies_o>& up_transport_layer_info_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<up_transport_layer_info_ext_ies_o> >();
}
void up_transport_layer_info_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::gtp_tunnel:
      j.write_fieldname("gTPTunnel");
      c.get<gtp_tunnel_s>().to_json(j);
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<up_transport_layer_info_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "up_transport_layer_info_c");
  }
  j.end_obj();
}
SRSASN_CODE up_transport_layer_info_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::gtp_tunnel:
      HANDLE_CODE(c.get<gtp_tunnel_s>().pack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<up_transport_layer_info_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "up_transport_layer_info_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE up_transport_layer_info_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::gtp_tunnel:
      HANDLE_CODE(c.get<gtp_tunnel_s>().unpack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<up_transport_layer_info_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "up_transport_layer_info_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* up_transport_layer_info_c::types_opts::to_string() const
{
  static const char* options[] = {"gTPTunnel", "choice-Extensions"};
  return convert_enum_idx(options, 2, value, "up_transport_layer_info_c::types");
}

// AdditionalDLUPTNLInformationForHOItem ::= SEQUENCE
SRSASN_CODE add_dluptnl_info_for_ho_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(add_dl_forwarding_uptnl_info_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(add_dl_ngu_up_tnl_info.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, add_qos_flow_setup_resp_list, 1, 64, true));
  if (add_dl_forwarding_uptnl_info_present) {
    HANDLE_CODE(add_dl_forwarding_uptnl_info.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE add_dluptnl_info_for_ho_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(add_dl_forwarding_uptnl_info_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(add_dl_ngu_up_tnl_info.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(add_qos_flow_setup_resp_list, bref, 1, 64, true));
  if (add_dl_forwarding_uptnl_info_present) {
    HANDLE_CODE(add_dl_forwarding_uptnl_info.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void add_dluptnl_info_for_ho_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("additionalDL-NGU-UP-TNLInformation");
  add_dl_ngu_up_tnl_info.to_json(j);
  j.start_array("additionalQosFlowSetupResponseList");
  for (const auto& e1 : add_qos_flow_setup_resp_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (add_dl_forwarding_uptnl_info_present) {
    j.write_fieldname("additionalDLForwardingUPTNLInformation");
    add_dl_forwarding_uptnl_info.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Pre-emptionCapability ::= ENUMERATED
const char* pre_emption_cap_opts::to_string() const
{
  static const char* options[] = {"shall-not-trigger-pre-emption", "may-trigger-pre-emption"};
  return convert_enum_idx(options, 2, value, "pre_emption_cap_e");
}

// Pre-emptionVulnerability ::= ENUMERATED
const char* pre_emption_vulnerability_opts::to_string() const
{
  static const char* options[] = {"not-pre-emptable", "pre-emptable"};
  return convert_enum_idx(options, 2, value, "pre_emption_vulnerability_e");
}

// AllocationAndRetentionPriority ::= SEQUENCE
SRSASN_CODE alloc_and_retention_prio_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, prio_level_arp, (uint8_t)1u, (uint8_t)15u, false, true));
  HANDLE_CODE(pre_emption_cap.pack(bref));
  HANDLE_CODE(pre_emption_vulnerability.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE alloc_and_retention_prio_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(prio_level_arp, bref, (uint8_t)1u, (uint8_t)15u, false, true));
  HANDLE_CODE(pre_emption_cap.unpack(bref));
  HANDLE_CODE(pre_emption_vulnerability.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void alloc_and_retention_prio_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("priorityLevelARP", prio_level_arp);
  j.write_str("pre-emptionCapability", pre_emption_cap.to_string());
  j.write_str("pre-emptionVulnerability", pre_emption_vulnerability.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AllowedNSSAI-Item ::= SEQUENCE
SRSASN_CODE allowed_nssai_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(s_nssai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE allowed_nssai_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(s_nssai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void allowed_nssai_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("s-NSSAI");
  s_nssai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EUTRA-CGI ::= SEQUENCE
SRSASN_CODE eutra_cgi_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(eutra_cell_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE eutra_cgi_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(eutra_cell_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void eutra_cgi_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNIdentity", plmn_id.to_string());
  j.write_str("eUTRACellIdentity", eutra_cell_id.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// NR-CGI ::= SEQUENCE
SRSASN_CODE nr_cgi_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(nrcell_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE nr_cgi_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(nrcell_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void nr_cgi_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNIdentity", plmn_id.to_string());
  j.write_str("nRCellIdentity", nrcell_id.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// NGRAN-CGI ::= CHOICE
void ngran_cgi_c::destroy_()
{
  switch (type_) {
    case types::nr_cgi:
      c.destroy<nr_cgi_s>();
      break;
    case types::eutra_cgi:
      c.destroy<eutra_cgi_s>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<ngran_cgi_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void ngran_cgi_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::nr_cgi:
      c.init<nr_cgi_s>();
      break;
    case types::eutra_cgi:
      c.init<eutra_cgi_s>();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<ngran_cgi_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ngran_cgi_c");
  }
}
ngran_cgi_c::ngran_cgi_c(const ngran_cgi_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::nr_cgi:
      c.init(other.c.get<nr_cgi_s>());
      break;
    case types::eutra_cgi:
      c.init(other.c.get<eutra_cgi_s>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<ngran_cgi_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ngran_cgi_c");
  }
}
ngran_cgi_c& ngran_cgi_c::operator=(const ngran_cgi_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::nr_cgi:
      c.set(other.c.get<nr_cgi_s>());
      break;
    case types::eutra_cgi:
      c.set(other.c.get<eutra_cgi_s>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<ngran_cgi_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ngran_cgi_c");
  }

  return *this;
}
nr_cgi_s& ngran_cgi_c::set_nr_cgi()
{
  set(types::nr_cgi);
  return c.get<nr_cgi_s>();
}
eutra_cgi_s& ngran_cgi_c::set_eutra_cgi()
{
  set(types::eutra_cgi);
  return c.get<eutra_cgi_s>();
}
protocol_ie_single_container_s<ngran_cgi_ext_ies_o>& ngran_cgi_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<ngran_cgi_ext_ies_o> >();
}
void ngran_cgi_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::nr_cgi:
      j.write_fieldname("nR-CGI");
      c.get<nr_cgi_s>().to_json(j);
      break;
    case types::eutra_cgi:
      j.write_fieldname("eUTRA-CGI");
      c.get<eutra_cgi_s>().to_json(j);
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<ngran_cgi_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ngran_cgi_c");
  }
  j.end_obj();
}
SRSASN_CODE ngran_cgi_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::nr_cgi:
      HANDLE_CODE(c.get<nr_cgi_s>().pack(bref));
      break;
    case types::eutra_cgi:
      HANDLE_CODE(c.get<eutra_cgi_s>().pack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<ngran_cgi_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ngran_cgi_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ngran_cgi_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::nr_cgi:
      HANDLE_CODE(c.get<nr_cgi_s>().unpack(bref));
      break;
    case types::eutra_cgi:
      HANDLE_CODE(c.get<eutra_cgi_s>().unpack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<ngran_cgi_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ngran_cgi_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ngran_cgi_c::types_opts::to_string() const
{
  static const char* options[] = {"nR-CGI", "eUTRA-CGI", "choice-Extensions"};
  return convert_enum_idx(options, 3, value, "ngran_cgi_c::types");
}

// AreaOfInterestCellItem ::= SEQUENCE
SRSASN_CODE area_of_interest_cell_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ngran_cgi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE area_of_interest_cell_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ngran_cgi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void area_of_interest_cell_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("nGRAN-CGI");
  ngran_cgi.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AreaOfInterestRANNodeItem ::= SEQUENCE
SRSASN_CODE area_of_interest_ran_node_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(global_ran_node_id.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE area_of_interest_ran_node_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(global_ran_node_id.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void area_of_interest_ran_node_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("globalRANNodeID");
  global_ran_node_id.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AreaOfInterestTAIItem ::= SEQUENCE
SRSASN_CODE area_of_interest_tai_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(tai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE area_of_interest_tai_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(tai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void area_of_interest_tai_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("tAI");
  tai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AreaOfInterest ::= SEQUENCE
SRSASN_CODE area_of_interest_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(area_of_interest_tai_list.size() > 0, 1));
  HANDLE_CODE(bref.pack(area_of_interest_cell_list.size() > 0, 1));
  HANDLE_CODE(bref.pack(area_of_interest_ran_node_list.size() > 0, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (area_of_interest_tai_list.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, area_of_interest_tai_list, 1, 16, true));
  }
  if (area_of_interest_cell_list.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, area_of_interest_cell_list, 1, 256, true));
  }
  if (area_of_interest_ran_node_list.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, area_of_interest_ran_node_list, 1, 64, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE area_of_interest_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  bool area_of_interest_tai_list_present;
  HANDLE_CODE(bref.unpack(area_of_interest_tai_list_present, 1));
  bool area_of_interest_cell_list_present;
  HANDLE_CODE(bref.unpack(area_of_interest_cell_list_present, 1));
  bool area_of_interest_ran_node_list_present;
  HANDLE_CODE(bref.unpack(area_of_interest_ran_node_list_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (area_of_interest_tai_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(area_of_interest_tai_list, bref, 1, 16, true));
  }
  if (area_of_interest_cell_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(area_of_interest_cell_list, bref, 1, 256, true));
  }
  if (area_of_interest_ran_node_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(area_of_interest_ran_node_list, bref, 1, 64, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void area_of_interest_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (area_of_interest_tai_list.size() > 0) {
    j.start_array("areaOfInterestTAIList");
    for (const auto& e1 : area_of_interest_tai_list) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (area_of_interest_cell_list.size() > 0) {
    j.start_array("areaOfInterestCellList");
    for (const auto& e1 : area_of_interest_cell_list) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (area_of_interest_ran_node_list.size() > 0) {
    j.start_array("areaOfInterestRANNodeList");
    for (const auto& e1 : area_of_interest_ran_node_list) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AreaOfInterestItem ::= SEQUENCE
SRSASN_CODE area_of_interest_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(area_of_interest.pack(bref));
  HANDLE_CODE(pack_integer(bref, location_report_ref_id, (uint8_t)1u, (uint8_t)64u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE area_of_interest_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(area_of_interest.unpack(bref));
  HANDLE_CODE(unpack_integer(location_report_ref_id, bref, (uint8_t)1u, (uint8_t)64u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void area_of_interest_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("areaOfInterest");
  area_of_interest.to_json(j);
  j.write_int("locationReportingReferenceID", location_report_ref_id);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// RecommendedCellItem ::= SEQUENCE
SRSASN_CODE recommended_cell_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(time_stayed_in_cell_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ngran_cgi.pack(bref));
  if (time_stayed_in_cell_present) {
    HANDLE_CODE(pack_integer(bref, time_stayed_in_cell, (uint16_t)0u, (uint16_t)4095u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE recommended_cell_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(time_stayed_in_cell_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ngran_cgi.unpack(bref));
  if (time_stayed_in_cell_present) {
    HANDLE_CODE(unpack_integer(time_stayed_in_cell, bref, (uint16_t)0u, (uint16_t)4095u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void recommended_cell_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("nGRAN-CGI");
  ngran_cgi.to_json(j);
  if (time_stayed_in_cell_present) {
    j.write_int("timeStayedInCell", time_stayed_in_cell);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// NextPagingAreaScope ::= ENUMERATED
const char* next_paging_area_scope_opts::to_string() const
{
  static const char* options[] = {"same", "changed"};
  return convert_enum_idx(options, 2, value, "next_paging_area_scope_e");
}

// RecommendedCellsForPaging ::= SEQUENCE
SRSASN_CODE recommended_cells_for_paging_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, recommended_cell_list, 1, 16, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE recommended_cells_for_paging_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(recommended_cell_list, bref, 1, 16, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void recommended_cells_for_paging_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("recommendedCellList");
  for (const auto& e1 : recommended_cell_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AssistanceDataForRecommendedCells ::= SEQUENCE
SRSASN_CODE assist_data_for_recommended_cells_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(recommended_cells_for_paging.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE assist_data_for_recommended_cells_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(recommended_cells_for_paging.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void assist_data_for_recommended_cells_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("recommendedCellsForPaging");
  recommended_cells_for_paging.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PagingAttemptInformation ::= SEQUENCE
SRSASN_CODE paging_attempt_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(next_paging_area_scope_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, paging_attempt_count, (uint8_t)1u, (uint8_t)16u, true, true));
  HANDLE_CODE(pack_integer(bref, intended_nof_paging_attempts, (uint8_t)1u, (uint8_t)16u, true, true));
  if (next_paging_area_scope_present) {
    HANDLE_CODE(next_paging_area_scope.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE paging_attempt_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(next_paging_area_scope_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(paging_attempt_count, bref, (uint8_t)1u, (uint8_t)16u, true, true));
  HANDLE_CODE(unpack_integer(intended_nof_paging_attempts, bref, (uint8_t)1u, (uint8_t)16u, true, true));
  if (next_paging_area_scope_present) {
    HANDLE_CODE(next_paging_area_scope.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void paging_attempt_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pagingAttemptCount", paging_attempt_count);
  j.write_int("intendedNumberOfPagingAttempts", intended_nof_paging_attempts);
  if (next_paging_area_scope_present) {
    j.write_str("nextPagingAreaScope", next_paging_area_scope.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AssistanceDataForPaging ::= SEQUENCE
SRSASN_CODE assist_data_for_paging_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(assist_data_for_recommended_cells_present, 1));
  HANDLE_CODE(bref.pack(paging_attempt_info_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (assist_data_for_recommended_cells_present) {
    HANDLE_CODE(assist_data_for_recommended_cells.pack(bref));
  }
  if (paging_attempt_info_present) {
    HANDLE_CODE(paging_attempt_info.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE assist_data_for_paging_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(assist_data_for_recommended_cells_present, 1));
  HANDLE_CODE(bref.unpack(paging_attempt_info_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (assist_data_for_recommended_cells_present) {
    HANDLE_CODE(assist_data_for_recommended_cells.unpack(bref));
  }
  if (paging_attempt_info_present) {
    HANDLE_CODE(paging_attempt_info.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void assist_data_for_paging_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (assist_data_for_recommended_cells_present) {
    j.write_fieldname("assistanceDataForRecommendedCells");
    assist_data_for_recommended_cells.to_json(j);
  }
  if (paging_attempt_info_present) {
    j.write_fieldname("pagingAttemptInformation");
    paging_attempt_info.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// AssociatedQosFlowItem ::= SEQUENCE
SRSASN_CODE associated_qos_flow_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(qos_flow_map_ind_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, qos_flow_id, (uint8_t)0u, (uint8_t)63u, true, true));
  if (qos_flow_map_ind_present) {
    HANDLE_CODE(qos_flow_map_ind.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE associated_qos_flow_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(qos_flow_map_ind_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(qos_flow_id, bref, (uint8_t)0u, (uint8_t)63u, true, true));
  if (qos_flow_map_ind_present) {
    HANDLE_CODE(qos_flow_map_ind.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void associated_qos_flow_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("qosFlowIdentifier", qos_flow_id);
  if (qos_flow_map_ind_present) {
    j.write_str("qosFlowMappingIndication", qos_flow_map_ind.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

const char* associated_qos_flow_item_s::qos_flow_map_ind_opts::to_string() const
{
  static const char* options[] = {"ul", "dl"};
  return convert_enum_idx(options, 2, value, "associated_qos_flow_item_s::qos_flow_map_ind_e_");
}

// CancelledCellsInEAI-EUTRA-Item ::= SEQUENCE
SRSASN_CODE cancelled_cells_in_eai_eutra_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(eutra_cgi.pack(bref));
  HANDLE_CODE(pack_integer(bref, nof_broadcasts, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cancelled_cells_in_eai_eutra_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(eutra_cgi.unpack(bref));
  HANDLE_CODE(unpack_integer(nof_broadcasts, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cancelled_cells_in_eai_eutra_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eUTRA-CGI");
  eutra_cgi.to_json(j);
  j.write_int("numberOfBroadcasts", nof_broadcasts);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CancelledCellsInEAI-NR-Item ::= SEQUENCE
SRSASN_CODE cancelled_cells_in_eai_nr_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(nr_cgi.pack(bref));
  HANDLE_CODE(pack_integer(bref, nof_broadcasts, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cancelled_cells_in_eai_nr_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(nr_cgi.unpack(bref));
  HANDLE_CODE(unpack_integer(nof_broadcasts, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cancelled_cells_in_eai_nr_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("nR-CGI");
  nr_cgi.to_json(j);
  j.write_int("numberOfBroadcasts", nof_broadcasts);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CancelledCellsInTAI-EUTRA-Item ::= SEQUENCE
SRSASN_CODE cancelled_cells_in_tai_eutra_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(eutra_cgi.pack(bref));
  HANDLE_CODE(pack_integer(bref, nof_broadcasts, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cancelled_cells_in_tai_eutra_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(eutra_cgi.unpack(bref));
  HANDLE_CODE(unpack_integer(nof_broadcasts, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cancelled_cells_in_tai_eutra_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eUTRA-CGI");
  eutra_cgi.to_json(j);
  j.write_int("numberOfBroadcasts", nof_broadcasts);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CancelledCellsInTAI-NR-Item ::= SEQUENCE
SRSASN_CODE cancelled_cells_in_tai_nr_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(nr_cgi.pack(bref));
  HANDLE_CODE(pack_integer(bref, nof_broadcasts, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cancelled_cells_in_tai_nr_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(nr_cgi.unpack(bref));
  HANDLE_CODE(unpack_integer(nof_broadcasts, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cancelled_cells_in_tai_nr_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("nR-CGI");
  nr_cgi.to_json(j);
  j.write_int("numberOfBroadcasts", nof_broadcasts);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CellIDCancelledEUTRA-Item ::= SEQUENCE
SRSASN_CODE cell_id_cancelled_eutra_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(eutra_cgi.pack(bref));
  HANDLE_CODE(pack_integer(bref, nof_broadcasts, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_id_cancelled_eutra_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(eutra_cgi.unpack(bref));
  HANDLE_CODE(unpack_integer(nof_broadcasts, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_id_cancelled_eutra_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eUTRA-CGI");
  eutra_cgi.to_json(j);
  j.write_int("numberOfBroadcasts", nof_broadcasts);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CellIDCancelledNR-Item ::= SEQUENCE
SRSASN_CODE cell_id_cancelled_nr_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(nr_cgi.pack(bref));
  HANDLE_CODE(pack_integer(bref, nof_broadcasts, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_id_cancelled_nr_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(nr_cgi.unpack(bref));
  HANDLE_CODE(unpack_integer(nof_broadcasts, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_id_cancelled_nr_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("nR-CGI");
  nr_cgi.to_json(j);
  j.write_int("numberOfBroadcasts", nof_broadcasts);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EmergencyAreaIDCancelledEUTRA-Item ::= SEQUENCE
SRSASN_CODE emergency_area_id_cancelled_eutra_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(emergency_area_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, cancelled_cells_in_eai_eutra, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE emergency_area_id_cancelled_eutra_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(emergency_area_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(cancelled_cells_in_eai_eutra, bref, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void emergency_area_id_cancelled_eutra_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("emergencyAreaID", emergency_area_id.to_string());
  j.start_array("cancelledCellsInEAI-EUTRA");
  for (const auto& e1 : cancelled_cells_in_eai_eutra) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EmergencyAreaIDCancelledNR-Item ::= SEQUENCE
SRSASN_CODE emergency_area_id_cancelled_nr_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(emergency_area_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, cancelled_cells_in_eai_nr, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE emergency_area_id_cancelled_nr_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(emergency_area_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(cancelled_cells_in_eai_nr, bref, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void emergency_area_id_cancelled_nr_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("emergencyAreaID", emergency_area_id.to_string());
  j.start_array("cancelledCellsInEAI-NR");
  for (const auto& e1 : cancelled_cells_in_eai_nr) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TAICancelledEUTRA-Item ::= SEQUENCE
SRSASN_CODE tai_cancelled_eutra_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(tai.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, cancelled_cells_in_tai_eutra, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_cancelled_eutra_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(tai.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(cancelled_cells_in_tai_eutra, bref, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_cancelled_eutra_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("tAI");
  tai.to_json(j);
  j.start_array("cancelledCellsInTAI-EUTRA");
  for (const auto& e1 : cancelled_cells_in_tai_eutra) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TAICancelledNR-Item ::= SEQUENCE
SRSASN_CODE tai_cancelled_nr_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(tai.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, cancelled_cells_in_tai_nr, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_cancelled_nr_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(tai.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(cancelled_cells_in_tai_nr, bref, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_cancelled_nr_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("tAI");
  tai.to_json(j);
  j.start_array("cancelledCellsInTAI-NR");
  for (const auto& e1 : cancelled_cells_in_tai_nr) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// BroadcastCancelledAreaList ::= CHOICE
void broadcast_cancelled_area_list_c::destroy_()
{
  switch (type_) {
    case types::cell_id_cancelled_eutra:
      c.destroy<cell_id_cancelled_eutra_l>();
      break;
    case types::tai_cancelled_eutra:
      c.destroy<tai_cancelled_eutra_l>();
      break;
    case types::emergency_area_id_cancelled_eutra:
      c.destroy<emergency_area_id_cancelled_eutra_l>();
      break;
    case types::cell_id_cancelled_nr:
      c.destroy<cell_id_cancelled_nr_l>();
      break;
    case types::tai_cancelled_nr:
      c.destroy<tai_cancelled_nr_l>();
      break;
    case types::emergency_area_id_cancelled_nr:
      c.destroy<emergency_area_id_cancelled_nr_l>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<broadcast_cancelled_area_list_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void broadcast_cancelled_area_list_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cell_id_cancelled_eutra:
      c.init<cell_id_cancelled_eutra_l>();
      break;
    case types::tai_cancelled_eutra:
      c.init<tai_cancelled_eutra_l>();
      break;
    case types::emergency_area_id_cancelled_eutra:
      c.init<emergency_area_id_cancelled_eutra_l>();
      break;
    case types::cell_id_cancelled_nr:
      c.init<cell_id_cancelled_nr_l>();
      break;
    case types::tai_cancelled_nr:
      c.init<tai_cancelled_nr_l>();
      break;
    case types::emergency_area_id_cancelled_nr:
      c.init<emergency_area_id_cancelled_nr_l>();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<broadcast_cancelled_area_list_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
  }
}
broadcast_cancelled_area_list_c::broadcast_cancelled_area_list_c(const broadcast_cancelled_area_list_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cell_id_cancelled_eutra:
      c.init(other.c.get<cell_id_cancelled_eutra_l>());
      break;
    case types::tai_cancelled_eutra:
      c.init(other.c.get<tai_cancelled_eutra_l>());
      break;
    case types::emergency_area_id_cancelled_eutra:
      c.init(other.c.get<emergency_area_id_cancelled_eutra_l>());
      break;
    case types::cell_id_cancelled_nr:
      c.init(other.c.get<cell_id_cancelled_nr_l>());
      break;
    case types::tai_cancelled_nr:
      c.init(other.c.get<tai_cancelled_nr_l>());
      break;
    case types::emergency_area_id_cancelled_nr:
      c.init(other.c.get<emergency_area_id_cancelled_nr_l>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<broadcast_cancelled_area_list_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
  }
}
broadcast_cancelled_area_list_c&
broadcast_cancelled_area_list_c::operator=(const broadcast_cancelled_area_list_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cell_id_cancelled_eutra:
      c.set(other.c.get<cell_id_cancelled_eutra_l>());
      break;
    case types::tai_cancelled_eutra:
      c.set(other.c.get<tai_cancelled_eutra_l>());
      break;
    case types::emergency_area_id_cancelled_eutra:
      c.set(other.c.get<emergency_area_id_cancelled_eutra_l>());
      break;
    case types::cell_id_cancelled_nr:
      c.set(other.c.get<cell_id_cancelled_nr_l>());
      break;
    case types::tai_cancelled_nr:
      c.set(other.c.get<tai_cancelled_nr_l>());
      break;
    case types::emergency_area_id_cancelled_nr:
      c.set(other.c.get<emergency_area_id_cancelled_nr_l>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<broadcast_cancelled_area_list_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
  }

  return *this;
}
cell_id_cancelled_eutra_l& broadcast_cancelled_area_list_c::set_cell_id_cancelled_eutra()
{
  set(types::cell_id_cancelled_eutra);
  return c.get<cell_id_cancelled_eutra_l>();
}
tai_cancelled_eutra_l& broadcast_cancelled_area_list_c::set_tai_cancelled_eutra()
{
  set(types::tai_cancelled_eutra);
  return c.get<tai_cancelled_eutra_l>();
}
emergency_area_id_cancelled_eutra_l& broadcast_cancelled_area_list_c::set_emergency_area_id_cancelled_eutra()
{
  set(types::emergency_area_id_cancelled_eutra);
  return c.get<emergency_area_id_cancelled_eutra_l>();
}
cell_id_cancelled_nr_l& broadcast_cancelled_area_list_c::set_cell_id_cancelled_nr()
{
  set(types::cell_id_cancelled_nr);
  return c.get<cell_id_cancelled_nr_l>();
}
tai_cancelled_nr_l& broadcast_cancelled_area_list_c::set_tai_cancelled_nr()
{
  set(types::tai_cancelled_nr);
  return c.get<tai_cancelled_nr_l>();
}
emergency_area_id_cancelled_nr_l& broadcast_cancelled_area_list_c::set_emergency_area_id_cancelled_nr()
{
  set(types::emergency_area_id_cancelled_nr);
  return c.get<emergency_area_id_cancelled_nr_l>();
}
protocol_ie_single_container_s<broadcast_cancelled_area_list_ext_ies_o>&
broadcast_cancelled_area_list_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<broadcast_cancelled_area_list_ext_ies_o> >();
}
void broadcast_cancelled_area_list_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cell_id_cancelled_eutra:
      j.start_array("cellIDCancelledEUTRA");
      for (const auto& e1 : c.get<cell_id_cancelled_eutra_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::tai_cancelled_eutra:
      j.start_array("tAICancelledEUTRA");
      for (const auto& e1 : c.get<tai_cancelled_eutra_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::emergency_area_id_cancelled_eutra:
      j.start_array("emergencyAreaIDCancelledEUTRA");
      for (const auto& e1 : c.get<emergency_area_id_cancelled_eutra_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::cell_id_cancelled_nr:
      j.start_array("cellIDCancelledNR");
      for (const auto& e1 : c.get<cell_id_cancelled_nr_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::tai_cancelled_nr:
      j.start_array("tAICancelledNR");
      for (const auto& e1 : c.get<tai_cancelled_nr_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::emergency_area_id_cancelled_nr:
      j.start_array("emergencyAreaIDCancelledNR");
      for (const auto& e1 : c.get<emergency_area_id_cancelled_nr_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<broadcast_cancelled_area_list_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
  }
  j.end_obj();
}
SRSASN_CODE broadcast_cancelled_area_list_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::cell_id_cancelled_eutra:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<cell_id_cancelled_eutra_l>(), 1, 65535, true));
      break;
    case types::tai_cancelled_eutra:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<tai_cancelled_eutra_l>(), 1, 65535, true));
      break;
    case types::emergency_area_id_cancelled_eutra:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<emergency_area_id_cancelled_eutra_l>(), 1, 65535, true));
      break;
    case types::cell_id_cancelled_nr:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<cell_id_cancelled_nr_l>(), 1, 65535, true));
      break;
    case types::tai_cancelled_nr:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<tai_cancelled_nr_l>(), 1, 65535, true));
      break;
    case types::emergency_area_id_cancelled_nr:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<emergency_area_id_cancelled_nr_l>(), 1, 65535, true));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<broadcast_cancelled_area_list_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE broadcast_cancelled_area_list_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::cell_id_cancelled_eutra:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<cell_id_cancelled_eutra_l>(), bref, 1, 65535, true));
      break;
    case types::tai_cancelled_eutra:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<tai_cancelled_eutra_l>(), bref, 1, 65535, true));
      break;
    case types::emergency_area_id_cancelled_eutra:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<emergency_area_id_cancelled_eutra_l>(), bref, 1, 65535, true));
      break;
    case types::cell_id_cancelled_nr:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<cell_id_cancelled_nr_l>(), bref, 1, 65535, true));
      break;
    case types::tai_cancelled_nr:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<tai_cancelled_nr_l>(), bref, 1, 65535, true));
      break;
    case types::emergency_area_id_cancelled_nr:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<emergency_area_id_cancelled_nr_l>(), bref, 1, 65535, true));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<broadcast_cancelled_area_list_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_cancelled_area_list_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* broadcast_cancelled_area_list_c::types_opts::to_string() const
{
  static const char* options[] = {"cellIDCancelledEUTRA",
                                  "tAICancelledEUTRA",
                                  "emergencyAreaIDCancelledEUTRA",
                                  "cellIDCancelledNR",
                                  "tAICancelledNR",
                                  "emergencyAreaIDCancelledNR",
                                  "choice-Extensions"};
  return convert_enum_idx(options, 7, value, "broadcast_cancelled_area_list_c::types");
}

// CompletedCellsInEAI-EUTRA-Item ::= SEQUENCE
SRSASN_CODE completed_cells_in_eai_eutra_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(eutra_cgi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE completed_cells_in_eai_eutra_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(eutra_cgi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void completed_cells_in_eai_eutra_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eUTRA-CGI");
  eutra_cgi.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CompletedCellsInEAI-NR-Item ::= SEQUENCE
SRSASN_CODE completed_cells_in_eai_nr_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(nr_cgi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE completed_cells_in_eai_nr_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(nr_cgi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void completed_cells_in_eai_nr_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("nR-CGI");
  nr_cgi.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CompletedCellsInTAI-EUTRA-Item ::= SEQUENCE
SRSASN_CODE completed_cells_in_tai_eutra_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(eutra_cgi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE completed_cells_in_tai_eutra_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(eutra_cgi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void completed_cells_in_tai_eutra_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eUTRA-CGI");
  eutra_cgi.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CompletedCellsInTAI-NR-Item ::= SEQUENCE
SRSASN_CODE completed_cells_in_tai_nr_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(nr_cgi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE completed_cells_in_tai_nr_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(nr_cgi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void completed_cells_in_tai_nr_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("nR-CGI");
  nr_cgi.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CellIDBroadcastEUTRA-Item ::= SEQUENCE
SRSASN_CODE cell_id_broadcast_eutra_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(eutra_cgi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_id_broadcast_eutra_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(eutra_cgi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_id_broadcast_eutra_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eUTRA-CGI");
  eutra_cgi.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CellIDBroadcastNR-Item ::= SEQUENCE
SRSASN_CODE cell_id_broadcast_nr_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(nr_cgi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_id_broadcast_nr_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(nr_cgi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_id_broadcast_nr_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("nR-CGI");
  nr_cgi.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EmergencyAreaIDBroadcastEUTRA-Item ::= SEQUENCE
SRSASN_CODE emergency_area_id_broadcast_eutra_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(emergency_area_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, completed_cells_in_eai_eutra, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE emergency_area_id_broadcast_eutra_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(emergency_area_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(completed_cells_in_eai_eutra, bref, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void emergency_area_id_broadcast_eutra_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("emergencyAreaID", emergency_area_id.to_string());
  j.start_array("completedCellsInEAI-EUTRA");
  for (const auto& e1 : completed_cells_in_eai_eutra) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EmergencyAreaIDBroadcastNR-Item ::= SEQUENCE
SRSASN_CODE emergency_area_id_broadcast_nr_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(emergency_area_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, completed_cells_in_eai_nr, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE emergency_area_id_broadcast_nr_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(emergency_area_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(completed_cells_in_eai_nr, bref, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void emergency_area_id_broadcast_nr_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("emergencyAreaID", emergency_area_id.to_string());
  j.start_array("completedCellsInEAI-NR");
  for (const auto& e1 : completed_cells_in_eai_nr) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TAIBroadcastEUTRA-Item ::= SEQUENCE
SRSASN_CODE tai_broadcast_eutra_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(tai.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, completed_cells_in_tai_eutra, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_broadcast_eutra_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(tai.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(completed_cells_in_tai_eutra, bref, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_broadcast_eutra_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("tAI");
  tai.to_json(j);
  j.start_array("completedCellsInTAI-EUTRA");
  for (const auto& e1 : completed_cells_in_tai_eutra) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TAIBroadcastNR-Item ::= SEQUENCE
SRSASN_CODE tai_broadcast_nr_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(tai.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, completed_cells_in_tai_nr, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_broadcast_nr_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(tai.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(completed_cells_in_tai_nr, bref, 1, 65535, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_broadcast_nr_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("tAI");
  tai.to_json(j);
  j.start_array("completedCellsInTAI-NR");
  for (const auto& e1 : completed_cells_in_tai_nr) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// BroadcastCompletedAreaList ::= CHOICE
void broadcast_completed_area_list_c::destroy_()
{
  switch (type_) {
    case types::cell_id_broadcast_eutra:
      c.destroy<cell_id_broadcast_eutra_l>();
      break;
    case types::tai_broadcast_eutra:
      c.destroy<tai_broadcast_eutra_l>();
      break;
    case types::emergency_area_id_broadcast_eutra:
      c.destroy<emergency_area_id_broadcast_eutra_l>();
      break;
    case types::cell_id_broadcast_nr:
      c.destroy<cell_id_broadcast_nr_l>();
      break;
    case types::tai_broadcast_nr:
      c.destroy<tai_broadcast_nr_l>();
      break;
    case types::emergency_area_id_broadcast_nr:
      c.destroy<emergency_area_id_broadcast_nr_l>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<broadcast_completed_area_list_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void broadcast_completed_area_list_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cell_id_broadcast_eutra:
      c.init<cell_id_broadcast_eutra_l>();
      break;
    case types::tai_broadcast_eutra:
      c.init<tai_broadcast_eutra_l>();
      break;
    case types::emergency_area_id_broadcast_eutra:
      c.init<emergency_area_id_broadcast_eutra_l>();
      break;
    case types::cell_id_broadcast_nr:
      c.init<cell_id_broadcast_nr_l>();
      break;
    case types::tai_broadcast_nr:
      c.init<tai_broadcast_nr_l>();
      break;
    case types::emergency_area_id_broadcast_nr:
      c.init<emergency_area_id_broadcast_nr_l>();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<broadcast_completed_area_list_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
  }
}
broadcast_completed_area_list_c::broadcast_completed_area_list_c(const broadcast_completed_area_list_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cell_id_broadcast_eutra:
      c.init(other.c.get<cell_id_broadcast_eutra_l>());
      break;
    case types::tai_broadcast_eutra:
      c.init(other.c.get<tai_broadcast_eutra_l>());
      break;
    case types::emergency_area_id_broadcast_eutra:
      c.init(other.c.get<emergency_area_id_broadcast_eutra_l>());
      break;
    case types::cell_id_broadcast_nr:
      c.init(other.c.get<cell_id_broadcast_nr_l>());
      break;
    case types::tai_broadcast_nr:
      c.init(other.c.get<tai_broadcast_nr_l>());
      break;
    case types::emergency_area_id_broadcast_nr:
      c.init(other.c.get<emergency_area_id_broadcast_nr_l>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<broadcast_completed_area_list_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
  }
}
broadcast_completed_area_list_c&
broadcast_completed_area_list_c::operator=(const broadcast_completed_area_list_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cell_id_broadcast_eutra:
      c.set(other.c.get<cell_id_broadcast_eutra_l>());
      break;
    case types::tai_broadcast_eutra:
      c.set(other.c.get<tai_broadcast_eutra_l>());
      break;
    case types::emergency_area_id_broadcast_eutra:
      c.set(other.c.get<emergency_area_id_broadcast_eutra_l>());
      break;
    case types::cell_id_broadcast_nr:
      c.set(other.c.get<cell_id_broadcast_nr_l>());
      break;
    case types::tai_broadcast_nr:
      c.set(other.c.get<tai_broadcast_nr_l>());
      break;
    case types::emergency_area_id_broadcast_nr:
      c.set(other.c.get<emergency_area_id_broadcast_nr_l>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<broadcast_completed_area_list_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
  }

  return *this;
}
cell_id_broadcast_eutra_l& broadcast_completed_area_list_c::set_cell_id_broadcast_eutra()
{
  set(types::cell_id_broadcast_eutra);
  return c.get<cell_id_broadcast_eutra_l>();
}
tai_broadcast_eutra_l& broadcast_completed_area_list_c::set_tai_broadcast_eutra()
{
  set(types::tai_broadcast_eutra);
  return c.get<tai_broadcast_eutra_l>();
}
emergency_area_id_broadcast_eutra_l& broadcast_completed_area_list_c::set_emergency_area_id_broadcast_eutra()
{
  set(types::emergency_area_id_broadcast_eutra);
  return c.get<emergency_area_id_broadcast_eutra_l>();
}
cell_id_broadcast_nr_l& broadcast_completed_area_list_c::set_cell_id_broadcast_nr()
{
  set(types::cell_id_broadcast_nr);
  return c.get<cell_id_broadcast_nr_l>();
}
tai_broadcast_nr_l& broadcast_completed_area_list_c::set_tai_broadcast_nr()
{
  set(types::tai_broadcast_nr);
  return c.get<tai_broadcast_nr_l>();
}
emergency_area_id_broadcast_nr_l& broadcast_completed_area_list_c::set_emergency_area_id_broadcast_nr()
{
  set(types::emergency_area_id_broadcast_nr);
  return c.get<emergency_area_id_broadcast_nr_l>();
}
protocol_ie_single_container_s<broadcast_completed_area_list_ext_ies_o>&
broadcast_completed_area_list_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<broadcast_completed_area_list_ext_ies_o> >();
}
void broadcast_completed_area_list_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cell_id_broadcast_eutra:
      j.start_array("cellIDBroadcastEUTRA");
      for (const auto& e1 : c.get<cell_id_broadcast_eutra_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::tai_broadcast_eutra:
      j.start_array("tAIBroadcastEUTRA");
      for (const auto& e1 : c.get<tai_broadcast_eutra_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::emergency_area_id_broadcast_eutra:
      j.start_array("emergencyAreaIDBroadcastEUTRA");
      for (const auto& e1 : c.get<emergency_area_id_broadcast_eutra_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::cell_id_broadcast_nr:
      j.start_array("cellIDBroadcastNR");
      for (const auto& e1 : c.get<cell_id_broadcast_nr_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::tai_broadcast_nr:
      j.start_array("tAIBroadcastNR");
      for (const auto& e1 : c.get<tai_broadcast_nr_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::emergency_area_id_broadcast_nr:
      j.start_array("emergencyAreaIDBroadcastNR");
      for (const auto& e1 : c.get<emergency_area_id_broadcast_nr_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<broadcast_completed_area_list_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
  }
  j.end_obj();
}
SRSASN_CODE broadcast_completed_area_list_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::cell_id_broadcast_eutra:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<cell_id_broadcast_eutra_l>(), 1, 65535, true));
      break;
    case types::tai_broadcast_eutra:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<tai_broadcast_eutra_l>(), 1, 65535, true));
      break;
    case types::emergency_area_id_broadcast_eutra:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<emergency_area_id_broadcast_eutra_l>(), 1, 65535, true));
      break;
    case types::cell_id_broadcast_nr:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<cell_id_broadcast_nr_l>(), 1, 65535, true));
      break;
    case types::tai_broadcast_nr:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<tai_broadcast_nr_l>(), 1, 65535, true));
      break;
    case types::emergency_area_id_broadcast_nr:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<emergency_area_id_broadcast_nr_l>(), 1, 65535, true));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<broadcast_completed_area_list_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE broadcast_completed_area_list_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::cell_id_broadcast_eutra:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<cell_id_broadcast_eutra_l>(), bref, 1, 65535, true));
      break;
    case types::tai_broadcast_eutra:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<tai_broadcast_eutra_l>(), bref, 1, 65535, true));
      break;
    case types::emergency_area_id_broadcast_eutra:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<emergency_area_id_broadcast_eutra_l>(), bref, 1, 65535, true));
      break;
    case types::cell_id_broadcast_nr:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<cell_id_broadcast_nr_l>(), bref, 1, 65535, true));
      break;
    case types::tai_broadcast_nr:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<tai_broadcast_nr_l>(), bref, 1, 65535, true));
      break;
    case types::emergency_area_id_broadcast_nr:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<emergency_area_id_broadcast_nr_l>(), bref, 1, 65535, true));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<broadcast_completed_area_list_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "broadcast_completed_area_list_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* broadcast_completed_area_list_c::types_opts::to_string() const
{
  static const char* options[] = {"cellIDBroadcastEUTRA",
                                  "tAIBroadcastEUTRA",
                                  "emergencyAreaIDBroadcastEUTRA",
                                  "cellIDBroadcastNR",
                                  "tAIBroadcastNR",
                                  "emergencyAreaIDBroadcastNR",
                                  "choice-Extensions"};
  return convert_enum_idx(options, 7, value, "broadcast_completed_area_list_c::types");
}

// BroadcastPLMNItem ::= SEQUENCE
SRSASN_CODE broadcast_plmn_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, tai_slice_support_list, 1, 1024, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE broadcast_plmn_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(tai_slice_support_list, bref, 1, 1024, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void broadcast_plmn_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNIdentity", plmn_id.to_string());
  j.start_array("tAISliceSupportList");
  for (const auto& e1 : tai_slice_support_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// COUNTValueForPDCP-SN12 ::= SEQUENCE
SRSASN_CODE count_value_for_pdcp_sn12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdcp_sn12, (uint16_t)0u, (uint16_t)4095u, false, true));
  HANDLE_CODE(pack_integer(bref, hfn_pdcp_sn12, (uint32_t)0u, (uint32_t)1048575u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE count_value_for_pdcp_sn12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdcp_sn12, bref, (uint16_t)0u, (uint16_t)4095u, false, true));
  HANDLE_CODE(unpack_integer(hfn_pdcp_sn12, bref, (uint32_t)0u, (uint32_t)1048575u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void count_value_for_pdcp_sn12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDCP-SN12", pdcp_sn12);
  j.write_int("hFN-PDCP-SN12", hfn_pdcp_sn12);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// COUNTValueForPDCP-SN18 ::= SEQUENCE
SRSASN_CODE count_value_for_pdcp_sn18_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdcp_sn18, (uint32_t)0u, (uint32_t)262143u, false, true));
  HANDLE_CODE(pack_integer(bref, hfn_pdcp_sn18, (uint16_t)0u, (uint16_t)16383u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE count_value_for_pdcp_sn18_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdcp_sn18, bref, (uint32_t)0u, (uint32_t)262143u, false, true));
  HANDLE_CODE(unpack_integer(hfn_pdcp_sn18, bref, (uint16_t)0u, (uint16_t)16383u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void count_value_for_pdcp_sn18_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDCP-SN18", pdcp_sn18);
  j.write_int("hFN-PDCP-SN18", hfn_pdcp_sn18);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CellIDListForRestart ::= CHOICE
void cell_id_list_for_restart_c::destroy_()
{
  switch (type_) {
    case types::eutra_cgi_listfor_restart:
      c.destroy<eutra_cgi_list_l>();
      break;
    case types::nr_cgi_listfor_restart:
      c.destroy<nr_cgi_list_l>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<cell_id_list_for_restart_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void cell_id_list_for_restart_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::eutra_cgi_listfor_restart:
      c.init<eutra_cgi_list_l>();
      break;
    case types::nr_cgi_listfor_restart:
      c.init<nr_cgi_list_l>();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<cell_id_list_for_restart_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_id_list_for_restart_c");
  }
}
cell_id_list_for_restart_c::cell_id_list_for_restart_c(const cell_id_list_for_restart_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::eutra_cgi_listfor_restart:
      c.init(other.c.get<eutra_cgi_list_l>());
      break;
    case types::nr_cgi_listfor_restart:
      c.init(other.c.get<nr_cgi_list_l>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<cell_id_list_for_restart_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_id_list_for_restart_c");
  }
}
cell_id_list_for_restart_c& cell_id_list_for_restart_c::operator=(const cell_id_list_for_restart_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::eutra_cgi_listfor_restart:
      c.set(other.c.get<eutra_cgi_list_l>());
      break;
    case types::nr_cgi_listfor_restart:
      c.set(other.c.get<nr_cgi_list_l>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<cell_id_list_for_restart_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_id_list_for_restart_c");
  }

  return *this;
}
eutra_cgi_list_l& cell_id_list_for_restart_c::set_eutra_cgi_listfor_restart()
{
  set(types::eutra_cgi_listfor_restart);
  return c.get<eutra_cgi_list_l>();
}
nr_cgi_list_l& cell_id_list_for_restart_c::set_nr_cgi_listfor_restart()
{
  set(types::nr_cgi_listfor_restart);
  return c.get<nr_cgi_list_l>();
}
protocol_ie_single_container_s<cell_id_list_for_restart_ext_ies_o>& cell_id_list_for_restart_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<cell_id_list_for_restart_ext_ies_o> >();
}
void cell_id_list_for_restart_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::eutra_cgi_listfor_restart:
      j.start_array("eUTRA-CGIListforRestart");
      for (const auto& e1 : c.get<eutra_cgi_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::nr_cgi_listfor_restart:
      j.start_array("nR-CGIListforRestart");
      for (const auto& e1 : c.get<nr_cgi_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<cell_id_list_for_restart_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "cell_id_list_for_restart_c");
  }
  j.end_obj();
}
SRSASN_CODE cell_id_list_for_restart_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::eutra_cgi_listfor_restart:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<eutra_cgi_list_l>(), 1, 256, true));
      break;
    case types::nr_cgi_listfor_restart:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<nr_cgi_list_l>(), 1, 16384, true));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<cell_id_list_for_restart_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cell_id_list_for_restart_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_id_list_for_restart_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::eutra_cgi_listfor_restart:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<eutra_cgi_list_l>(), bref, 1, 256, true));
      break;
    case types::nr_cgi_listfor_restart:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<nr_cgi_list_l>(), bref, 1, 16384, true));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<cell_id_list_for_restart_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "cell_id_list_for_restart_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* cell_id_list_for_restart_c::types_opts::to_string() const
{
  static const char* options[] = {"eUTRA-CGIListforRestart", "nR-CGIListforRestart", "choice-Extensions"};
  return convert_enum_idx(options, 3, value, "cell_id_list_for_restart_c::types");
}

// CellTrafficTraceIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t cell_traffic_trace_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 44, 43, 109};
  return map_enum_number(options, 5, idx, "id");
}
bool cell_traffic_trace_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 44, 43, 109};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e cell_traffic_trace_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 44:
      return crit_e::ignore;
    case 43:
      return crit_e::ignore;
    case 109:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
cell_traffic_trace_ies_o::value_c cell_traffic_trace_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 44:
      ret.set(value_c::types::ngran_trace_id);
      break;
    case 43:
      ret.set(value_c::types::ngran_cgi);
      break;
    case 109:
      ret.set(value_c::types::trace_collection_entity_ip_address);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e cell_traffic_trace_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 44:
      return presence_e::mandatory;
    case 43:
      return presence_e::mandatory;
    case 109:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void cell_traffic_trace_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ngran_trace_id:
      c.destroy<fixed_octstring<8, true> >();
      break;
    case types::ngran_cgi:
      c.destroy<ngran_cgi_c>();
      break;
    case types::trace_collection_entity_ip_address:
      c.destroy<bounded_bitstring<1, 160, true, true> >();
      break;
    default:
      break;
  }
}
void cell_traffic_trace_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::ngran_trace_id:
      c.init<fixed_octstring<8, true> >();
      break;
    case types::ngran_cgi:
      c.init<ngran_cgi_c>();
      break;
    case types::trace_collection_entity_ip_address:
      c.init<bounded_bitstring<1, 160, true, true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
  }
}
cell_traffic_trace_ies_o::value_c::value_c(const cell_traffic_trace_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ngran_trace_id:
      c.init(other.c.get<fixed_octstring<8, true> >());
      break;
    case types::ngran_cgi:
      c.init(other.c.get<ngran_cgi_c>());
      break;
    case types::trace_collection_entity_ip_address:
      c.init(other.c.get<bounded_bitstring<1, 160, true, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
  }
}
cell_traffic_trace_ies_o::value_c&
cell_traffic_trace_ies_o::value_c::operator=(const cell_traffic_trace_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ngran_trace_id:
      c.set(other.c.get<fixed_octstring<8, true> >());
      break;
    case types::ngran_cgi:
      c.set(other.c.get<ngran_cgi_c>());
      break;
    case types::trace_collection_entity_ip_address:
      c.set(other.c.get<bounded_bitstring<1, 160, true, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
  }

  return *this;
}
uint64_t& cell_traffic_trace_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& cell_traffic_trace_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
fixed_octstring<8, true>& cell_traffic_trace_ies_o::value_c::ngran_trace_id()
{
  assert_choice_type(types::ngran_trace_id, type_, "Value");
  return c.get<fixed_octstring<8, true> >();
}
ngran_cgi_c& cell_traffic_trace_ies_o::value_c::ngran_cgi()
{
  assert_choice_type(types::ngran_cgi, type_, "Value");
  return c.get<ngran_cgi_c>();
}
bounded_bitstring<1, 160, true, true>& cell_traffic_trace_ies_o::value_c::trace_collection_entity_ip_address()
{
  assert_choice_type(types::trace_collection_entity_ip_address, type_, "Value");
  return c.get<bounded_bitstring<1, 160, true, true> >();
}
const uint64_t& cell_traffic_trace_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& cell_traffic_trace_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const fixed_octstring<8, true>& cell_traffic_trace_ies_o::value_c::ngran_trace_id() const
{
  assert_choice_type(types::ngran_trace_id, type_, "Value");
  return c.get<fixed_octstring<8, true> >();
}
const ngran_cgi_c& cell_traffic_trace_ies_o::value_c::ngran_cgi() const
{
  assert_choice_type(types::ngran_cgi, type_, "Value");
  return c.get<ngran_cgi_c>();
}
const bounded_bitstring<1, 160, true, true>&
cell_traffic_trace_ies_o::value_c::trace_collection_entity_ip_address() const
{
  assert_choice_type(types::trace_collection_entity_ip_address, type_, "Value");
  return c.get<bounded_bitstring<1, 160, true, true> >();
}
void cell_traffic_trace_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::ngran_trace_id:
      j.write_str("OCTET STRING", c.get<fixed_octstring<8, true> >().to_string());
      break;
    case types::ngran_cgi:
      j.write_fieldname("NGRAN-CGI");
      c.get<ngran_cgi_c>().to_json(j);
      break;
    case types::trace_collection_entity_ip_address:
      j.write_str("BIT STRING", c.get<bounded_bitstring<1, 160, true, true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE cell_traffic_trace_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::ngran_trace_id:
      HANDLE_CODE((c.get<fixed_octstring<8, true> >().pack(bref)));
      break;
    case types::ngran_cgi:
      HANDLE_CODE(c.get<ngran_cgi_c>().pack(bref));
      break;
    case types::trace_collection_entity_ip_address:
      HANDLE_CODE((c.get<bounded_bitstring<1, 160, true, true> >().pack(bref)));
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_traffic_trace_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::ngran_trace_id:
      HANDLE_CODE((c.get<fixed_octstring<8, true> >().unpack(bref)));
      break;
    case types::ngran_cgi:
      HANDLE_CODE(c.get<ngran_cgi_c>().unpack(bref));
      break;
    case types::trace_collection_entity_ip_address:
      HANDLE_CODE((c.get<bounded_bitstring<1, 160, true, true> >().unpack(bref)));
      break;
    default:
      log_invalid_choice_id(type_, "cell_traffic_trace_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* cell_traffic_trace_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..1099511627775)", "INTEGER (0..4294967295)", "OCTET STRING", "NGRAN-CGI", "BIT STRING"};
  return convert_enum_idx(options, 5, value, "cell_traffic_trace_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<cell_traffic_trace_ies_o>;

cell_traffic_trace_ies_container::cell_traffic_trace_ies_container() :
  amf_ue_ngap_id(10, crit_e::reject),
  ran_ue_ngap_id(85, crit_e::reject),
  ngran_trace_id(44, crit_e::ignore),
  ngran_cgi(43, crit_e::ignore),
  trace_collection_entity_ip_address(109, crit_e::ignore)
{}
SRSASN_CODE cell_traffic_trace_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 5;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  HANDLE_CODE(ngran_trace_id.pack(bref));
  HANDLE_CODE(ngran_cgi.pack(bref));
  HANDLE_CODE(trace_collection_entity_ip_address.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_traffic_trace_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 5;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 85: {
        nof_mandatory_ies--;
        ran_ue_ngap_id.id = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 44: {
        nof_mandatory_ies--;
        ngran_trace_id.id = id;
        HANDLE_CODE(ngran_trace_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ngran_trace_id.value.unpack(bref));
        break;
      }
      case 43: {
        nof_mandatory_ies--;
        ngran_cgi.id = id;
        HANDLE_CODE(ngran_cgi.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ngran_cgi.value.unpack(bref));
        break;
      }
      case 109: {
        nof_mandatory_ies--;
        trace_collection_entity_ip_address.id = id;
        HANDLE_CODE(trace_collection_entity_ip_address.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(trace_collection_entity_ip_address.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void cell_traffic_trace_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ran_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ngran_trace_id.to_json(j);
  j.write_fieldname("");
  ngran_cgi.to_json(j);
  j.write_fieldname("");
  trace_collection_entity_ip_address.to_json(j);
  j.end_obj();
}

// CellSize ::= ENUMERATED
const char* cell_size_opts::to_string() const
{
  static const char* options[] = {"verysmall", "small", "medium", "large"};
  return convert_enum_idx(options, 4, value, "cell_size_e");
}

// CellType ::= SEQUENCE
SRSASN_CODE cell_type_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(cell_size.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE cell_type_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(cell_size.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void cell_type_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("cellSize", cell_size.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ExpectedUEMovingTrajectoryItem ::= SEQUENCE
SRSASN_CODE expected_ue_moving_trajectory_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(time_stayed_in_cell_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ngran_cgi.pack(bref));
  if (time_stayed_in_cell_present) {
    HANDLE_CODE(pack_integer(bref, time_stayed_in_cell, (uint16_t)0u, (uint16_t)4095u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE expected_ue_moving_trajectory_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(time_stayed_in_cell_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ngran_cgi.unpack(bref));
  if (time_stayed_in_cell_present) {
    HANDLE_CODE(unpack_integer(time_stayed_in_cell, bref, (uint16_t)0u, (uint16_t)4095u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void expected_ue_moving_trajectory_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("nGRAN-CGI");
  ngran_cgi.to_json(j);
  if (time_stayed_in_cell_present) {
    j.write_int("timeStayedInCell", time_stayed_in_cell);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// SourceOfUEActivityBehaviourInformation ::= ENUMERATED
const char* source_of_ue_activity_behaviour_info_opts::to_string() const
{
  static const char* options[] = {"subscription-information", "statistics"};
  return convert_enum_idx(options, 2, value, "source_of_ue_activity_behaviour_info_e");
}

// ExpectedHOInterval ::= ENUMERATED
const char* expected_ho_interv_opts::to_string() const
{
  static const char* options[] = {"sec15", "sec30", "sec60", "sec90", "sec120", "sec180", "long-time"};
  return convert_enum_idx(options, 7, value, "expected_ho_interv_e");
}
uint8_t expected_ho_interv_opts::to_number() const
{
  static const uint8_t options[] = {15, 30, 60, 90, 120, 180};
  return map_enum_number(options, 6, value, "expected_ho_interv_e");
}

// ExpectedUEActivityBehaviour ::= SEQUENCE
SRSASN_CODE expected_ue_activity_behaviour_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(expected_activity_period_present, 1));
  HANDLE_CODE(bref.pack(expected_idle_period_present, 1));
  HANDLE_CODE(bref.pack(source_of_ue_activity_behaviour_info_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (expected_activity_period_present) {
    HANDLE_CODE(pack_integer(bref, expected_activity_period, (uint8_t)1u, (uint8_t)30u, true, true));
  }
  if (expected_idle_period_present) {
    HANDLE_CODE(pack_integer(bref, expected_idle_period, (uint8_t)1u, (uint8_t)30u, true, true));
  }
  if (source_of_ue_activity_behaviour_info_present) {
    HANDLE_CODE(source_of_ue_activity_behaviour_info.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE expected_ue_activity_behaviour_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(expected_activity_period_present, 1));
  HANDLE_CODE(bref.unpack(expected_idle_period_present, 1));
  HANDLE_CODE(bref.unpack(source_of_ue_activity_behaviour_info_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (expected_activity_period_present) {
    HANDLE_CODE(unpack_integer(expected_activity_period, bref, (uint8_t)1u, (uint8_t)30u, true, true));
  }
  if (expected_idle_period_present) {
    HANDLE_CODE(unpack_integer(expected_idle_period, bref, (uint8_t)1u, (uint8_t)30u, true, true));
  }
  if (source_of_ue_activity_behaviour_info_present) {
    HANDLE_CODE(source_of_ue_activity_behaviour_info.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void expected_ue_activity_behaviour_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (expected_activity_period_present) {
    j.write_int("expectedActivityPeriod", expected_activity_period);
  }
  if (expected_idle_period_present) {
    j.write_int("expectedIdlePeriod", expected_idle_period);
  }
  if (source_of_ue_activity_behaviour_info_present) {
    j.write_str("sourceOfUEActivityBehaviourInformation", source_of_ue_activity_behaviour_info.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ExpectedUEMobility ::= ENUMERATED
const char* expected_ue_mob_opts::to_string() const
{
  static const char* options[] = {"stationary", "mobile"};
  return convert_enum_idx(options, 2, value, "expected_ue_mob_e");
}

// TAIListForInactiveItem ::= SEQUENCE
SRSASN_CODE tai_list_for_inactive_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(tai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_list_for_inactive_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(tai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_list_for_inactive_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("tAI");
  tai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ExpectedUEBehaviour ::= SEQUENCE
SRSASN_CODE expected_ue_behaviour_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(expected_ue_activity_behaviour_present, 1));
  HANDLE_CODE(bref.pack(expected_ho_interv_present, 1));
  HANDLE_CODE(bref.pack(expected_ue_mob_present, 1));
  HANDLE_CODE(bref.pack(expected_ue_moving_trajectory.size() > 0, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (expected_ue_activity_behaviour_present) {
    HANDLE_CODE(expected_ue_activity_behaviour.pack(bref));
  }
  if (expected_ho_interv_present) {
    HANDLE_CODE(expected_ho_interv.pack(bref));
  }
  if (expected_ue_mob_present) {
    HANDLE_CODE(expected_ue_mob.pack(bref));
  }
  if (expected_ue_moving_trajectory.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, expected_ue_moving_trajectory, 1, 16, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE expected_ue_behaviour_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(expected_ue_activity_behaviour_present, 1));
  HANDLE_CODE(bref.unpack(expected_ho_interv_present, 1));
  HANDLE_CODE(bref.unpack(expected_ue_mob_present, 1));
  bool expected_ue_moving_trajectory_present;
  HANDLE_CODE(bref.unpack(expected_ue_moving_trajectory_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (expected_ue_activity_behaviour_present) {
    HANDLE_CODE(expected_ue_activity_behaviour.unpack(bref));
  }
  if (expected_ho_interv_present) {
    HANDLE_CODE(expected_ho_interv.unpack(bref));
  }
  if (expected_ue_mob_present) {
    HANDLE_CODE(expected_ue_mob.unpack(bref));
  }
  if (expected_ue_moving_trajectory_present) {
    HANDLE_CODE(unpack_dyn_seq_of(expected_ue_moving_trajectory, bref, 1, 16, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void expected_ue_behaviour_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (expected_ue_activity_behaviour_present) {
    j.write_fieldname("expectedUEActivityBehaviour");
    expected_ue_activity_behaviour.to_json(j);
  }
  if (expected_ho_interv_present) {
    j.write_str("expectedHOInterval", expected_ho_interv.to_string());
  }
  if (expected_ue_mob_present) {
    j.write_str("expectedUEMobility", expected_ue_mob.to_string());
  }
  if (expected_ue_moving_trajectory.size() > 0) {
    j.start_array("expectedUEMovingTrajectory");
    for (const auto& e1 : expected_ue_moving_trajectory) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// MICOModeIndication ::= ENUMERATED
const char* mico_mode_ind_opts::to_string() const
{
  static const char* options[] = {"true"};
  return convert_enum_idx(options, 1, value, "mico_mode_ind_e");
}

// PagingDRX ::= ENUMERATED
const char* paging_drx_opts::to_string() const
{
  static const char* options[] = {"v32", "v64", "v128", "v256"};
  return convert_enum_idx(options, 4, value, "paging_drx_e");
}
uint16_t paging_drx_opts::to_number() const
{
  static const uint16_t options[] = {32, 64, 128, 256};
  return map_enum_number(options, 4, value, "paging_drx_e");
}

// UEIdentityIndexValue ::= CHOICE
void ue_id_idx_value_c::destroy_()
{
  switch (type_) {
    case types::idx_len10:
      c.destroy<fixed_bitstring<10, false, true> >();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<ue_id_idx_value_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void ue_id_idx_value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::idx_len10:
      c.init<fixed_bitstring<10, false, true> >();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<ue_id_idx_value_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_id_idx_value_c");
  }
}
ue_id_idx_value_c::ue_id_idx_value_c(const ue_id_idx_value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::idx_len10:
      c.init(other.c.get<fixed_bitstring<10, false, true> >());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<ue_id_idx_value_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_id_idx_value_c");
  }
}
ue_id_idx_value_c& ue_id_idx_value_c::operator=(const ue_id_idx_value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::idx_len10:
      c.set(other.c.get<fixed_bitstring<10, false, true> >());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<ue_id_idx_value_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_id_idx_value_c");
  }

  return *this;
}
fixed_bitstring<10, false, true>& ue_id_idx_value_c::set_idx_len10()
{
  set(types::idx_len10);
  return c.get<fixed_bitstring<10, false, true> >();
}
protocol_ie_single_container_s<ue_id_idx_value_ext_ies_o>& ue_id_idx_value_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<ue_id_idx_value_ext_ies_o> >();
}
void ue_id_idx_value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::idx_len10:
      j.write_str("indexLength10", c.get<fixed_bitstring<10, false, true> >().to_string());
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<ue_id_idx_value_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ue_id_idx_value_c");
  }
  j.end_obj();
}
SRSASN_CODE ue_id_idx_value_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::idx_len10:
      HANDLE_CODE((c.get<fixed_bitstring<10, false, true> >().pack(bref)));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<ue_id_idx_value_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ue_id_idx_value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_id_idx_value_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::idx_len10:
      HANDLE_CODE((c.get<fixed_bitstring<10, false, true> >().unpack(bref)));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<ue_id_idx_value_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ue_id_idx_value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ue_id_idx_value_c::types_opts::to_string() const
{
  static const char* options[] = {"indexLength10", "choice-Extensions"};
  return convert_enum_idx(options, 2, value, "ue_id_idx_value_c::types");
}
uint8_t ue_id_idx_value_c::types_opts::to_number() const
{
  static const uint8_t options[] = {10};
  return map_enum_number(options, 1, value, "ue_id_idx_value_c::types");
}

// CoreNetworkAssistanceInformation ::= SEQUENCE
SRSASN_CODE core_network_assist_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(uespecific_drx_present, 1));
  HANDLE_CODE(bref.pack(mico_mode_ind_present, 1));
  HANDLE_CODE(bref.pack(expected_ue_behaviour_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ueid_idx_value.pack(bref));
  if (uespecific_drx_present) {
    HANDLE_CODE(uespecific_drx.pack(bref));
  }
  HANDLE_CODE(periodic_regist_upd_timer.pack(bref));
  if (mico_mode_ind_present) {
    HANDLE_CODE(mico_mode_ind.pack(bref));
  }
  HANDLE_CODE(pack_dyn_seq_of(bref, tai_list_for_inactive, 1, 16, true));
  if (expected_ue_behaviour_present) {
    HANDLE_CODE(expected_ue_behaviour.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE core_network_assist_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(uespecific_drx_present, 1));
  HANDLE_CODE(bref.unpack(mico_mode_ind_present, 1));
  HANDLE_CODE(bref.unpack(expected_ue_behaviour_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ueid_idx_value.unpack(bref));
  if (uespecific_drx_present) {
    HANDLE_CODE(uespecific_drx.unpack(bref));
  }
  HANDLE_CODE(periodic_regist_upd_timer.unpack(bref));
  if (mico_mode_ind_present) {
    HANDLE_CODE(mico_mode_ind.unpack(bref));
  }
  HANDLE_CODE(unpack_dyn_seq_of(tai_list_for_inactive, bref, 1, 16, true));
  if (expected_ue_behaviour_present) {
    HANDLE_CODE(expected_ue_behaviour.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void core_network_assist_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("uEIdentityIndexValue");
  ueid_idx_value.to_json(j);
  if (uespecific_drx_present) {
    j.write_str("uESpecificDRX", uespecific_drx.to_string());
  }
  j.write_str("periodicRegistrationUpdateTimer", periodic_regist_upd_timer.to_string());
  if (mico_mode_ind_present) {
    j.write_str("mICOModeIndication", "true");
  }
  j.start_array("tAIListForInactive");
  for (const auto& e1 : tai_list_for_inactive) {
    e1.to_json(j);
  }
  j.end_array();
  if (expected_ue_behaviour_present) {
    j.write_fieldname("expectedUEBehaviour");
    expected_ue_behaviour.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// DRBStatusDL12 ::= SEQUENCE
SRSASN_CODE drb_status_dl12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_ext_present, 1));

  HANDLE_CODE(dl_count_value.pack(bref));
  if (ie_ext_present) {
    HANDLE_CODE(ie_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE drb_status_dl12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_ext_present, 1));

  HANDLE_CODE(dl_count_value.unpack(bref));
  if (ie_ext_present) {
    HANDLE_CODE(ie_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void drb_status_dl12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("dL-COUNTValue");
  dl_count_value.to_json(j);
  if (ie_ext_present) {
    j.write_fieldname("iE-Extension");
    ie_ext.to_json(j);
  }
  j.end_obj();
}

// DRBStatusDL18 ::= SEQUENCE
SRSASN_CODE drb_status_dl18_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_ext_present, 1));

  HANDLE_CODE(dl_count_value.pack(bref));
  if (ie_ext_present) {
    HANDLE_CODE(ie_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE drb_status_dl18_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_ext_present, 1));

  HANDLE_CODE(dl_count_value.unpack(bref));
  if (ie_ext_present) {
    HANDLE_CODE(ie_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void drb_status_dl18_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("dL-COUNTValue");
  dl_count_value.to_json(j);
  if (ie_ext_present) {
    j.write_fieldname("iE-Extension");
    ie_ext.to_json(j);
  }
  j.end_obj();
}

// DRBStatusDL ::= CHOICE
void drb_status_dl_c::destroy_()
{
  switch (type_) {
    case types::drb_status_dl12:
      c.destroy<drb_status_dl12_s>();
      break;
    case types::drb_status_dl18:
      c.destroy<drb_status_dl18_s>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<drb_status_dl_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void drb_status_dl_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::drb_status_dl12:
      c.init<drb_status_dl12_s>();
      break;
    case types::drb_status_dl18:
      c.init<drb_status_dl18_s>();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<drb_status_dl_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "drb_status_dl_c");
  }
}
drb_status_dl_c::drb_status_dl_c(const drb_status_dl_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::drb_status_dl12:
      c.init(other.c.get<drb_status_dl12_s>());
      break;
    case types::drb_status_dl18:
      c.init(other.c.get<drb_status_dl18_s>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<drb_status_dl_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "drb_status_dl_c");
  }
}
drb_status_dl_c& drb_status_dl_c::operator=(const drb_status_dl_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::drb_status_dl12:
      c.set(other.c.get<drb_status_dl12_s>());
      break;
    case types::drb_status_dl18:
      c.set(other.c.get<drb_status_dl18_s>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<drb_status_dl_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "drb_status_dl_c");
  }

  return *this;
}
drb_status_dl12_s& drb_status_dl_c::set_drb_status_dl12()
{
  set(types::drb_status_dl12);
  return c.get<drb_status_dl12_s>();
}
drb_status_dl18_s& drb_status_dl_c::set_drb_status_dl18()
{
  set(types::drb_status_dl18);
  return c.get<drb_status_dl18_s>();
}
protocol_ie_single_container_s<drb_status_dl_ext_ies_o>& drb_status_dl_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<drb_status_dl_ext_ies_o> >();
}
void drb_status_dl_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::drb_status_dl12:
      j.write_fieldname("dRBStatusDL12");
      c.get<drb_status_dl12_s>().to_json(j);
      break;
    case types::drb_status_dl18:
      j.write_fieldname("dRBStatusDL18");
      c.get<drb_status_dl18_s>().to_json(j);
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<drb_status_dl_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "drb_status_dl_c");
  }
  j.end_obj();
}
SRSASN_CODE drb_status_dl_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::drb_status_dl12:
      HANDLE_CODE(c.get<drb_status_dl12_s>().pack(bref));
      break;
    case types::drb_status_dl18:
      HANDLE_CODE(c.get<drb_status_dl18_s>().pack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<drb_status_dl_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "drb_status_dl_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE drb_status_dl_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::drb_status_dl12:
      HANDLE_CODE(c.get<drb_status_dl12_s>().unpack(bref));
      break;
    case types::drb_status_dl18:
      HANDLE_CODE(c.get<drb_status_dl18_s>().unpack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<drb_status_dl_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "drb_status_dl_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* drb_status_dl_c::types_opts::to_string() const
{
  static const char* options[] = {"dRBStatusDL12", "dRBStatusDL18", "choice-Extensions"};
  return convert_enum_idx(options, 3, value, "drb_status_dl_c::types");
}
uint8_t drb_status_dl_c::types_opts::to_number() const
{
  static const uint8_t options[] = {12, 18};
  return map_enum_number(options, 2, value, "drb_status_dl_c::types");
}

// DRBStatusUL12 ::= SEQUENCE
SRSASN_CODE drb_status_ul12_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(receive_status_of_ul_pdcp_sdus_present, 1));
  HANDLE_CODE(bref.pack(ie_ext_present, 1));

  HANDLE_CODE(ul_count_value.pack(bref));
  if (receive_status_of_ul_pdcp_sdus_present) {
    HANDLE_CODE(receive_status_of_ul_pdcp_sdus.pack(bref));
  }
  if (ie_ext_present) {
    HANDLE_CODE(ie_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE drb_status_ul12_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(receive_status_of_ul_pdcp_sdus_present, 1));
  HANDLE_CODE(bref.unpack(ie_ext_present, 1));

  HANDLE_CODE(ul_count_value.unpack(bref));
  if (receive_status_of_ul_pdcp_sdus_present) {
    HANDLE_CODE(receive_status_of_ul_pdcp_sdus.unpack(bref));
  }
  if (ie_ext_present) {
    HANDLE_CODE(ie_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void drb_status_ul12_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("uL-COUNTValue");
  ul_count_value.to_json(j);
  if (receive_status_of_ul_pdcp_sdus_present) {
    j.write_str("receiveStatusOfUL-PDCP-SDUs", receive_status_of_ul_pdcp_sdus.to_string());
  }
  if (ie_ext_present) {
    j.write_fieldname("iE-Extension");
    ie_ext.to_json(j);
  }
  j.end_obj();
}

// DRBStatusUL18 ::= SEQUENCE
SRSASN_CODE drb_status_ul18_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(receive_status_of_ul_pdcp_sdus_present, 1));
  HANDLE_CODE(bref.pack(ie_ext_present, 1));

  HANDLE_CODE(ul_count_value.pack(bref));
  if (receive_status_of_ul_pdcp_sdus_present) {
    HANDLE_CODE(receive_status_of_ul_pdcp_sdus.pack(bref));
  }
  if (ie_ext_present) {
    HANDLE_CODE(ie_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE drb_status_ul18_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(receive_status_of_ul_pdcp_sdus_present, 1));
  HANDLE_CODE(bref.unpack(ie_ext_present, 1));

  HANDLE_CODE(ul_count_value.unpack(bref));
  if (receive_status_of_ul_pdcp_sdus_present) {
    HANDLE_CODE(receive_status_of_ul_pdcp_sdus.unpack(bref));
  }
  if (ie_ext_present) {
    HANDLE_CODE(ie_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void drb_status_ul18_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("uL-COUNTValue");
  ul_count_value.to_json(j);
  if (receive_status_of_ul_pdcp_sdus_present) {
    j.write_str("receiveStatusOfUL-PDCP-SDUs", receive_status_of_ul_pdcp_sdus.to_string());
  }
  if (ie_ext_present) {
    j.write_fieldname("iE-Extension");
    ie_ext.to_json(j);
  }
  j.end_obj();
}

// DRBStatusUL ::= CHOICE
void drb_status_ul_c::destroy_()
{
  switch (type_) {
    case types::drb_status_ul12:
      c.destroy<drb_status_ul12_s>();
      break;
    case types::drb_status_ul18:
      c.destroy<drb_status_ul18_s>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<drb_status_ul_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void drb_status_ul_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::drb_status_ul12:
      c.init<drb_status_ul12_s>();
      break;
    case types::drb_status_ul18:
      c.init<drb_status_ul18_s>();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<drb_status_ul_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "drb_status_ul_c");
  }
}
drb_status_ul_c::drb_status_ul_c(const drb_status_ul_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::drb_status_ul12:
      c.init(other.c.get<drb_status_ul12_s>());
      break;
    case types::drb_status_ul18:
      c.init(other.c.get<drb_status_ul18_s>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<drb_status_ul_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "drb_status_ul_c");
  }
}
drb_status_ul_c& drb_status_ul_c::operator=(const drb_status_ul_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::drb_status_ul12:
      c.set(other.c.get<drb_status_ul12_s>());
      break;
    case types::drb_status_ul18:
      c.set(other.c.get<drb_status_ul18_s>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<drb_status_ul_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "drb_status_ul_c");
  }

  return *this;
}
drb_status_ul12_s& drb_status_ul_c::set_drb_status_ul12()
{
  set(types::drb_status_ul12);
  return c.get<drb_status_ul12_s>();
}
drb_status_ul18_s& drb_status_ul_c::set_drb_status_ul18()
{
  set(types::drb_status_ul18);
  return c.get<drb_status_ul18_s>();
}
protocol_ie_single_container_s<drb_status_ul_ext_ies_o>& drb_status_ul_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<drb_status_ul_ext_ies_o> >();
}
void drb_status_ul_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::drb_status_ul12:
      j.write_fieldname("dRBStatusUL12");
      c.get<drb_status_ul12_s>().to_json(j);
      break;
    case types::drb_status_ul18:
      j.write_fieldname("dRBStatusUL18");
      c.get<drb_status_ul18_s>().to_json(j);
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<drb_status_ul_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "drb_status_ul_c");
  }
  j.end_obj();
}
SRSASN_CODE drb_status_ul_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::drb_status_ul12:
      HANDLE_CODE(c.get<drb_status_ul12_s>().pack(bref));
      break;
    case types::drb_status_ul18:
      HANDLE_CODE(c.get<drb_status_ul18_s>().pack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<drb_status_ul_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "drb_status_ul_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE drb_status_ul_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::drb_status_ul12:
      HANDLE_CODE(c.get<drb_status_ul12_s>().unpack(bref));
      break;
    case types::drb_status_ul18:
      HANDLE_CODE(c.get<drb_status_ul18_s>().unpack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<drb_status_ul_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "drb_status_ul_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* drb_status_ul_c::types_opts::to_string() const
{
  static const char* options[] = {"dRBStatusUL12", "dRBStatusUL18", "choice-Extensions"};
  return convert_enum_idx(options, 3, value, "drb_status_ul_c::types");
}
uint8_t drb_status_ul_c::types_opts::to_number() const
{
  static const uint8_t options[] = {12, 18};
  return map_enum_number(options, 2, value, "drb_status_ul_c::types");
}

// DRBsSubjectToStatusTransferItem ::= SEQUENCE
SRSASN_CODE drbs_subject_to_status_transfer_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_ext_present, 1));

  HANDLE_CODE(pack_integer(bref, drb_id, (uint8_t)1u, (uint8_t)32u, true, true));
  HANDLE_CODE(drb_status_ul.pack(bref));
  HANDLE_CODE(drb_status_dl.pack(bref));
  if (ie_ext_present) {
    HANDLE_CODE(ie_ext.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE drbs_subject_to_status_transfer_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_ext_present, 1));

  HANDLE_CODE(unpack_integer(drb_id, bref, (uint8_t)1u, (uint8_t)32u, true, true));
  HANDLE_CODE(drb_status_ul.unpack(bref));
  HANDLE_CODE(drb_status_dl.unpack(bref));
  if (ie_ext_present) {
    HANDLE_CODE(ie_ext.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void drbs_subject_to_status_transfer_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("dRB-ID", drb_id);
  j.write_fieldname("dRBStatusUL");
  drb_status_ul.to_json(j);
  j.write_fieldname("dRBStatusDL");
  drb_status_dl.to_json(j);
  if (ie_ext_present) {
    j.write_fieldname("iE-Extension");
    ie_ext.to_json(j);
  }
  j.end_obj();
}

// DRBsToQosFlowsMappingItem ::= SEQUENCE
SRSASN_CODE drbs_to_qos_flows_map_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, drb_id, (uint8_t)1u, (uint8_t)32u, true, true));
  HANDLE_CODE(pack_dyn_seq_of(bref, associated_qos_flow_list, 1, 64, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE drbs_to_qos_flows_map_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(drb_id, bref, (uint8_t)1u, (uint8_t)32u, true, true));
  HANDLE_CODE(unpack_dyn_seq_of(associated_qos_flow_list, bref, 1, 64, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void drbs_to_qos_flows_map_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("dRB-ID", drb_id);
  j.start_array("associatedQosFlowList");
  for (const auto& e1 : associated_qos_flow_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// DataForwardingResponseDRBItem ::= SEQUENCE
SRSASN_CODE data_forwarding_resp_drb_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(dlforwarding_up_tnl_info_present, 1));
  HANDLE_CODE(bref.pack(ulforwarding_up_tnl_info_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, drb_id, (uint8_t)1u, (uint8_t)32u, true, true));
  if (dlforwarding_up_tnl_info_present) {
    HANDLE_CODE(dlforwarding_up_tnl_info.pack(bref));
  }
  if (ulforwarding_up_tnl_info_present) {
    HANDLE_CODE(ulforwarding_up_tnl_info.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE data_forwarding_resp_drb_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(dlforwarding_up_tnl_info_present, 1));
  HANDLE_CODE(bref.unpack(ulforwarding_up_tnl_info_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(drb_id, bref, (uint8_t)1u, (uint8_t)32u, true, true));
  if (dlforwarding_up_tnl_info_present) {
    HANDLE_CODE(dlforwarding_up_tnl_info.unpack(bref));
  }
  if (ulforwarding_up_tnl_info_present) {
    HANDLE_CODE(ulforwarding_up_tnl_info.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void data_forwarding_resp_drb_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("dRB-ID", drb_id);
  if (dlforwarding_up_tnl_info_present) {
    j.write_fieldname("dLForwardingUP-TNLInformation");
    dlforwarding_up_tnl_info.to_json(j);
  }
  if (ulforwarding_up_tnl_info_present) {
    j.write_fieldname("uLForwardingUP-TNLInformation");
    ulforwarding_up_tnl_info.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// DeactivateTraceIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t deactiv_trace_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 44};
  return map_enum_number(options, 3, idx, "id");
}
bool deactiv_trace_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 44};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e deactiv_trace_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 44:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
deactiv_trace_ies_o::value_c deactiv_trace_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 44:
      ret.set(value_c::types::ngran_trace_id);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e deactiv_trace_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 44:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void deactiv_trace_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ngran_trace_id:
      c.destroy<fixed_octstring<8, true> >();
      break;
    default:
      break;
  }
}
void deactiv_trace_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::ngran_trace_id:
      c.init<fixed_octstring<8, true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
  }
}
deactiv_trace_ies_o::value_c::value_c(const deactiv_trace_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ngran_trace_id:
      c.init(other.c.get<fixed_octstring<8, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
  }
}
deactiv_trace_ies_o::value_c& deactiv_trace_ies_o::value_c::operator=(const deactiv_trace_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ngran_trace_id:
      c.set(other.c.get<fixed_octstring<8, true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
  }

  return *this;
}
uint64_t& deactiv_trace_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& deactiv_trace_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
fixed_octstring<8, true>& deactiv_trace_ies_o::value_c::ngran_trace_id()
{
  assert_choice_type(types::ngran_trace_id, type_, "Value");
  return c.get<fixed_octstring<8, true> >();
}
const uint64_t& deactiv_trace_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& deactiv_trace_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const fixed_octstring<8, true>& deactiv_trace_ies_o::value_c::ngran_trace_id() const
{
  assert_choice_type(types::ngran_trace_id, type_, "Value");
  return c.get<fixed_octstring<8, true> >();
}
void deactiv_trace_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::ngran_trace_id:
      j.write_str("OCTET STRING", c.get<fixed_octstring<8, true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE deactiv_trace_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::ngran_trace_id:
      HANDLE_CODE((c.get<fixed_octstring<8, true> >().pack(bref)));
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE deactiv_trace_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::ngran_trace_id:
      HANDLE_CODE((c.get<fixed_octstring<8, true> >().unpack(bref)));
      break;
    default:
      log_invalid_choice_id(type_, "deactiv_trace_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* deactiv_trace_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)", "INTEGER (0..4294967295)", "OCTET STRING"};
  return convert_enum_idx(options, 3, value, "deactiv_trace_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<deactiv_trace_ies_o>;

deactiv_trace_ies_container::deactiv_trace_ies_container() :
  amf_ue_ngap_id(10, crit_e::reject), ran_ue_ngap_id(85, crit_e::reject), ngran_trace_id(44, crit_e::ignore)
{}
SRSASN_CODE deactiv_trace_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  HANDLE_CODE(ngran_trace_id.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE deactiv_trace_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 85: {
        nof_mandatory_ies--;
        ran_ue_ngap_id.id = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 44: {
        nof_mandatory_ies--;
        ngran_trace_id.id = id;
        HANDLE_CODE(ngran_trace_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ngran_trace_id.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void deactiv_trace_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ran_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ngran_trace_id.to_json(j);
  j.end_obj();
}

// ForbiddenAreaInformation-Item ::= SEQUENCE
SRSASN_CODE forbidden_area_info_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, forbidden_tacs, 1, 4096, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE forbidden_area_info_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(forbidden_tacs, bref, 1, 4096, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void forbidden_area_info_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNIdentity", plmn_id.to_string());
  j.start_array("forbiddenTACs");
  for (const auto& e1 : forbidden_tacs) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// RATRestrictions-Item ::= SEQUENCE
SRSASN_CODE rat_restricts_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(rat_restrict_info.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE rat_restricts_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(rat_restrict_info.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void rat_restricts_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNIdentity", plmn_id.to_string());
  j.write_str("rATRestrictionInformation", rat_restrict_info.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ServiceAreaInformation-Item ::= SEQUENCE
SRSASN_CODE service_area_info_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(allowed_tacs.size() > 0, 1));
  HANDLE_CODE(bref.pack(not_allowed_tacs.size() > 0, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  if (allowed_tacs.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, allowed_tacs, 1, 16, true));
  }
  if (not_allowed_tacs.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, not_allowed_tacs, 1, 16, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE service_area_info_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  bool allowed_tacs_present;
  HANDLE_CODE(bref.unpack(allowed_tacs_present, 1));
  bool not_allowed_tacs_present;
  HANDLE_CODE(bref.unpack(not_allowed_tacs_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  if (allowed_tacs_present) {
    HANDLE_CODE(unpack_dyn_seq_of(allowed_tacs, bref, 1, 16, true));
  }
  if (not_allowed_tacs_present) {
    HANDLE_CODE(unpack_dyn_seq_of(not_allowed_tacs, bref, 1, 16, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void service_area_info_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNIdentity", plmn_id.to_string());
  if (allowed_tacs.size() > 0) {
    j.start_array("allowedTACs");
    for (const auto& e1 : allowed_tacs) {
      j.write_str(e1.to_string());
    }
    j.end_array();
  }
  if (not_allowed_tacs.size() > 0) {
    j.start_array("notAllowedTACs");
    for (const auto& e1 : not_allowed_tacs) {
      j.write_str(e1.to_string());
    }
    j.end_array();
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// MobilityRestrictionList-ExtIEs ::= OBJECT SET OF NGAP-PROTOCOL-EXTENSION
uint32_t mob_restrict_list_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {150};
  return map_enum_number(options, 1, idx, "id");
}
bool mob_restrict_list_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 150 == id;
}
crit_e mob_restrict_list_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 150) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
mob_restrict_list_ext_ies_o::ext_c mob_restrict_list_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 150) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e mob_restrict_list_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 150) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void mob_restrict_list_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("OCTET STRING", c.to_string());
  j.end_obj();
}
SRSASN_CODE mob_restrict_list_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE mob_restrict_list_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* mob_restrict_list_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"OCTET STRING"};
  return convert_enum_idx(options, 1, value, "mob_restrict_list_ext_ies_o::ext_c::types");
}

// MobilityRestrictionList ::= SEQUENCE
SRSASN_CODE mob_restrict_list_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(equivalent_plmns.size() > 0, 1));
  HANDLE_CODE(bref.pack(rat_restricts.size() > 0, 1));
  HANDLE_CODE(bref.pack(forbidden_area_info.size() > 0, 1));
  HANDLE_CODE(bref.pack(service_area_info.size() > 0, 1));
  HANDLE_CODE(bref.pack(ie_exts.size() > 0, 1));

  HANDLE_CODE(serving_plmn.pack(bref));
  if (equivalent_plmns.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, equivalent_plmns, 1, 15, true));
  }
  if (rat_restricts.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, rat_restricts, 1, 16, true));
  }
  if (forbidden_area_info.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, forbidden_area_info, 1, 16, true));
  }
  if (service_area_info.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, service_area_info, 1, 16, true));
  }
  if (ie_exts.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE mob_restrict_list_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  bool equivalent_plmns_present;
  HANDLE_CODE(bref.unpack(equivalent_plmns_present, 1));
  bool rat_restricts_present;
  HANDLE_CODE(bref.unpack(rat_restricts_present, 1));
  bool forbidden_area_info_present;
  HANDLE_CODE(bref.unpack(forbidden_area_info_present, 1));
  bool service_area_info_present;
  HANDLE_CODE(bref.unpack(service_area_info_present, 1));
  bool ie_exts_present;
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(serving_plmn.unpack(bref));
  if (equivalent_plmns_present) {
    HANDLE_CODE(unpack_dyn_seq_of(equivalent_plmns, bref, 1, 15, true));
  }
  if (rat_restricts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(rat_restricts, bref, 1, 16, true));
  }
  if (forbidden_area_info_present) {
    HANDLE_CODE(unpack_dyn_seq_of(forbidden_area_info, bref, 1, 16, true));
  }
  if (service_area_info_present) {
    HANDLE_CODE(unpack_dyn_seq_of(service_area_info, bref, 1, 16, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void mob_restrict_list_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("servingPLMN", serving_plmn.to_string());
  if (equivalent_plmns.size() > 0) {
    j.start_array("equivalentPLMNs");
    for (const auto& e1 : equivalent_plmns) {
      j.write_str(e1.to_string());
    }
    j.end_array();
  }
  if (rat_restricts.size() > 0) {
    j.start_array("rATRestrictions");
    for (const auto& e1 : rat_restricts) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (forbidden_area_info.size() > 0) {
    j.start_array("forbiddenAreaInformation");
    for (const auto& e1 : forbidden_area_info) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (service_area_info.size() > 0) {
    j.start_array("serviceAreaInformation");
    for (const auto& e1 : service_area_info) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ie_exts.size() > 0) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// UEAggregateMaximumBitRate ::= SEQUENCE
SRSASN_CODE ue_aggregate_maximum_bit_rate_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, ueaggregate_maximum_bit_rate_dl, (uint64_t)0u, (uint64_t)4000000000000u, true, true));
  HANDLE_CODE(pack_integer(bref, ueaggregate_maximum_bit_rate_ul, (uint64_t)0u, (uint64_t)4000000000000u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_aggregate_maximum_bit_rate_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(
      unpack_integer(ueaggregate_maximum_bit_rate_dl, bref, (uint64_t)0u, (uint64_t)4000000000000u, true, true));
  HANDLE_CODE(
      unpack_integer(ueaggregate_maximum_bit_rate_ul, bref, (uint64_t)0u, (uint64_t)4000000000000u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ue_aggregate_maximum_bit_rate_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("uEAggregateMaximumBitRateDL", ueaggregate_maximum_bit_rate_dl);
  j.write_int("uEAggregateMaximumBitRateUL", ueaggregate_maximum_bit_rate_ul);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// DownlinkNASTransport-IEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t dl_nas_transport_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 48, 83, 38, 36, 31, 110, 0};
  return map_enum_number(options, 9, idx, "id");
}
bool dl_nas_transport_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 48, 83, 38, 36, 31, 110, 0};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e dl_nas_transport_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 48:
      return crit_e::reject;
    case 83:
      return crit_e::ignore;
    case 38:
      return crit_e::reject;
    case 36:
      return crit_e::ignore;
    case 31:
      return crit_e::ignore;
    case 110:
      return crit_e::ignore;
    case 0:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
dl_nas_transport_ies_o::value_c dl_nas_transport_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 48:
      ret.set(value_c::types::old_amf);
      break;
    case 83:
      ret.set(value_c::types::ran_paging_prio);
      break;
    case 38:
      ret.set(value_c::types::nas_pdu);
      break;
    case 36:
      ret.set(value_c::types::mob_restrict_list);
      break;
    case 31:
      ret.set(value_c::types::idx_to_rfsp);
      break;
    case 110:
      ret.set(value_c::types::ue_aggregate_maximum_bit_rate);
      break;
    case 0:
      ret.set(value_c::types::allowed_nssai);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e dl_nas_transport_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 48:
      return presence_e::optional;
    case 83:
      return presence_e::optional;
    case 38:
      return presence_e::mandatory;
    case 36:
      return presence_e::optional;
    case 31:
      return presence_e::optional;
    case 110:
      return presence_e::optional;
    case 0:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void dl_nas_transport_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::old_amf:
      c.destroy<printable_string<1, 150, true, true> >();
      break;
    case types::nas_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::mob_restrict_list:
      c.destroy<mob_restrict_list_s>();
      break;
    case types::ue_aggregate_maximum_bit_rate:
      c.destroy<ue_aggregate_maximum_bit_rate_s>();
      break;
    case types::allowed_nssai:
      c.destroy<allowed_nssai_l>();
      break;
    default:
      break;
  }
}
void dl_nas_transport_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::old_amf:
      c.init<printable_string<1, 150, true, true> >();
      break;
    case types::ran_paging_prio:
      break;
    case types::nas_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::mob_restrict_list:
      c.init<mob_restrict_list_s>();
      break;
    case types::idx_to_rfsp:
      break;
    case types::ue_aggregate_maximum_bit_rate:
      c.init<ue_aggregate_maximum_bit_rate_s>();
      break;
    case types::allowed_nssai:
      c.init<allowed_nssai_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
  }
}
dl_nas_transport_ies_o::value_c::value_c(const dl_nas_transport_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::old_amf:
      c.init(other.c.get<printable_string<1, 150, true, true> >());
      break;
    case types::ran_paging_prio:
      c.init(other.c.get<uint16_t>());
      break;
    case types::nas_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::mob_restrict_list:
      c.init(other.c.get<mob_restrict_list_s>());
      break;
    case types::idx_to_rfsp:
      c.init(other.c.get<uint16_t>());
      break;
    case types::ue_aggregate_maximum_bit_rate:
      c.init(other.c.get<ue_aggregate_maximum_bit_rate_s>());
      break;
    case types::allowed_nssai:
      c.init(other.c.get<allowed_nssai_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
  }
}
dl_nas_transport_ies_o::value_c&
dl_nas_transport_ies_o::value_c::operator=(const dl_nas_transport_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::old_amf:
      c.set(other.c.get<printable_string<1, 150, true, true> >());
      break;
    case types::ran_paging_prio:
      c.set(other.c.get<uint16_t>());
      break;
    case types::nas_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::mob_restrict_list:
      c.set(other.c.get<mob_restrict_list_s>());
      break;
    case types::idx_to_rfsp:
      c.set(other.c.get<uint16_t>());
      break;
    case types::ue_aggregate_maximum_bit_rate:
      c.set(other.c.get<ue_aggregate_maximum_bit_rate_s>());
      break;
    case types::allowed_nssai:
      c.set(other.c.get<allowed_nssai_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
  }

  return *this;
}
uint64_t& dl_nas_transport_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& dl_nas_transport_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
printable_string<1, 150, true, true>& dl_nas_transport_ies_o::value_c::old_amf()
{
  assert_choice_type(types::old_amf, type_, "Value");
  return c.get<printable_string<1, 150, true, true> >();
}
uint16_t& dl_nas_transport_ies_o::value_c::ran_paging_prio()
{
  assert_choice_type(types::ran_paging_prio, type_, "Value");
  return c.get<uint16_t>();
}
unbounded_octstring<true>& dl_nas_transport_ies_o::value_c::nas_pdu()
{
  assert_choice_type(types::nas_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
mob_restrict_list_s& dl_nas_transport_ies_o::value_c::mob_restrict_list()
{
  assert_choice_type(types::mob_restrict_list, type_, "Value");
  return c.get<mob_restrict_list_s>();
}
uint16_t& dl_nas_transport_ies_o::value_c::idx_to_rfsp()
{
  assert_choice_type(types::idx_to_rfsp, type_, "Value");
  return c.get<uint16_t>();
}
ue_aggregate_maximum_bit_rate_s& dl_nas_transport_ies_o::value_c::ue_aggregate_maximum_bit_rate()
{
  assert_choice_type(types::ue_aggregate_maximum_bit_rate, type_, "Value");
  return c.get<ue_aggregate_maximum_bit_rate_s>();
}
allowed_nssai_l& dl_nas_transport_ies_o::value_c::allowed_nssai()
{
  assert_choice_type(types::allowed_nssai, type_, "Value");
  return c.get<allowed_nssai_l>();
}
const uint64_t& dl_nas_transport_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& dl_nas_transport_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const printable_string<1, 150, true, true>& dl_nas_transport_ies_o::value_c::old_amf() const
{
  assert_choice_type(types::old_amf, type_, "Value");
  return c.get<printable_string<1, 150, true, true> >();
}
const uint16_t& dl_nas_transport_ies_o::value_c::ran_paging_prio() const
{
  assert_choice_type(types::ran_paging_prio, type_, "Value");
  return c.get<uint16_t>();
}
const unbounded_octstring<true>& dl_nas_transport_ies_o::value_c::nas_pdu() const
{
  assert_choice_type(types::nas_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const mob_restrict_list_s& dl_nas_transport_ies_o::value_c::mob_restrict_list() const
{
  assert_choice_type(types::mob_restrict_list, type_, "Value");
  return c.get<mob_restrict_list_s>();
}
const uint16_t& dl_nas_transport_ies_o::value_c::idx_to_rfsp() const
{
  assert_choice_type(types::idx_to_rfsp, type_, "Value");
  return c.get<uint16_t>();
}
const ue_aggregate_maximum_bit_rate_s& dl_nas_transport_ies_o::value_c::ue_aggregate_maximum_bit_rate() const
{
  assert_choice_type(types::ue_aggregate_maximum_bit_rate, type_, "Value");
  return c.get<ue_aggregate_maximum_bit_rate_s>();
}
const allowed_nssai_l& dl_nas_transport_ies_o::value_c::allowed_nssai() const
{
  assert_choice_type(types::allowed_nssai, type_, "Value");
  return c.get<allowed_nssai_l>();
}
void dl_nas_transport_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::old_amf:
      j.write_str("PrintableString", c.get<printable_string<1, 150, true, true> >().to_string());
      break;
    case types::ran_paging_prio:
      j.write_int("INTEGER (1..256)", c.get<uint16_t>());
      break;
    case types::nas_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::mob_restrict_list:
      j.write_fieldname("MobilityRestrictionList");
      c.get<mob_restrict_list_s>().to_json(j);
      break;
    case types::idx_to_rfsp:
      j.write_int("INTEGER (1..256,...)", c.get<uint16_t>());
      break;
    case types::ue_aggregate_maximum_bit_rate:
      j.write_fieldname("UEAggregateMaximumBitRate");
      c.get<ue_aggregate_maximum_bit_rate_s>().to_json(j);
      break;
    case types::allowed_nssai:
      j.start_array("AllowedNSSAI");
      for (const auto& e1 : c.get<allowed_nssai_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE dl_nas_transport_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::old_amf:
      HANDLE_CODE((c.get<printable_string<1, 150, true, true> >().pack(bref)));
      break;
    case types::ran_paging_prio:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)1u, (uint16_t)256u, false, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::mob_restrict_list:
      HANDLE_CODE(c.get<mob_restrict_list_s>().pack(bref));
      break;
    case types::idx_to_rfsp:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)1u, (uint16_t)256u, true, true));
      break;
    case types::ue_aggregate_maximum_bit_rate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bit_rate_s>().pack(bref));
      break;
    case types::allowed_nssai:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<allowed_nssai_l>(), 1, 8, true));
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_nas_transport_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::old_amf:
      HANDLE_CODE((c.get<printable_string<1, 150, true, true> >().unpack(bref)));
      break;
    case types::ran_paging_prio:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)1u, (uint16_t)256u, false, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::mob_restrict_list:
      HANDLE_CODE(c.get<mob_restrict_list_s>().unpack(bref));
      break;
    case types::idx_to_rfsp:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)1u, (uint16_t)256u, true, true));
      break;
    case types::ue_aggregate_maximum_bit_rate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bit_rate_s>().unpack(bref));
      break;
    case types::allowed_nssai:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<allowed_nssai_l>(), bref, 1, 8, true));
      break;
    default:
      log_invalid_choice_id(type_, "dl_nas_transport_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* dl_nas_transport_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "PrintableString",
                                  "INTEGER (1..256)",
                                  "OCTET STRING",
                                  "MobilityRestrictionList",
                                  "INTEGER (1..256,...)",
                                  "UEAggregateMaximumBitRate",
                                  "AllowedNSSAI"};
  return convert_enum_idx(options, 9, value, "dl_nas_transport_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<dl_nas_transport_ies_o>;

dl_nas_transport_ies_container::dl_nas_transport_ies_container() :
  amf_ue_ngap_id(10, crit_e::reject),
  ran_ue_ngap_id(85, crit_e::reject),
  old_amf(48, crit_e::reject),
  ran_paging_prio(83, crit_e::ignore),
  nas_pdu(38, crit_e::reject),
  mob_restrict_list(36, crit_e::ignore),
  idx_to_rfsp(31, crit_e::ignore),
  ue_aggregate_maximum_bit_rate(110, crit_e::ignore),
  allowed_nssai(0, crit_e::reject)
{}
SRSASN_CODE dl_nas_transport_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += old_amf_present ? 1 : 0;
  nof_ies += ran_paging_prio_present ? 1 : 0;
  nof_ies += mob_restrict_list_present ? 1 : 0;
  nof_ies += idx_to_rfsp_present ? 1 : 0;
  nof_ies += ue_aggregate_maximum_bit_rate_present ? 1 : 0;
  nof_ies += allowed_nssai_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  if (old_amf_present) {
    HANDLE_CODE(old_amf.pack(bref));
  }
  if (ran_paging_prio_present) {
    HANDLE_CODE(ran_paging_prio.pack(bref));
  }
  HANDLE_CODE(nas_pdu.pack(bref));
  if (mob_restrict_list_present) {
    HANDLE_CODE(mob_restrict_list.pack(bref));
  }
  if (idx_to_rfsp_present) {
    HANDLE_CODE(idx_to_rfsp.pack(bref));
  }
  if (ue_aggregate_maximum_bit_rate_present) {
    HANDLE_CODE(ue_aggregate_maximum_bit_rate.pack(bref));
  }
  if (allowed_nssai_present) {
    HANDLE_CODE(allowed_nssai.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_nas_transport_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 85: {
        nof_mandatory_ies--;
        ran_ue_ngap_id.id = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 48: {
        old_amf_present = true;
        old_amf.id      = id;
        HANDLE_CODE(old_amf.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(old_amf.value.unpack(bref));
        break;
      }
      case 83: {
        ran_paging_prio_present = true;
        ran_paging_prio.id      = id;
        HANDLE_CODE(ran_paging_prio.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_paging_prio.value.unpack(bref));
        break;
      }
      case 38: {
        nof_mandatory_ies--;
        nas_pdu.id = id;
        HANDLE_CODE(nas_pdu.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(nas_pdu.value.unpack(bref));
        break;
      }
      case 36: {
        mob_restrict_list_present = true;
        mob_restrict_list.id      = id;
        HANDLE_CODE(mob_restrict_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mob_restrict_list.value.unpack(bref));
        break;
      }
      case 31: {
        idx_to_rfsp_present = true;
        idx_to_rfsp.id      = id;
        HANDLE_CODE(idx_to_rfsp.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(idx_to_rfsp.value.unpack(bref));
        break;
      }
      case 110: {
        ue_aggregate_maximum_bit_rate_present = true;
        ue_aggregate_maximum_bit_rate.id      = id;
        HANDLE_CODE(ue_aggregate_maximum_bit_rate.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ue_aggregate_maximum_bit_rate.value.unpack(bref));
        break;
      }
      case 0: {
        allowed_nssai_present = true;
        allowed_nssai.id      = id;
        HANDLE_CODE(allowed_nssai.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(allowed_nssai.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void dl_nas_transport_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ran_ue_ngap_id.to_json(j);
  if (old_amf_present) {
    j.write_fieldname("");
    old_amf.to_json(j);
  }
  if (ran_paging_prio_present) {
    j.write_fieldname("");
    ran_paging_prio.to_json(j);
  }
  j.write_fieldname("");
  nas_pdu.to_json(j);
  if (mob_restrict_list_present) {
    j.write_fieldname("");
    mob_restrict_list.to_json(j);
  }
  if (idx_to_rfsp_present) {
    j.write_fieldname("");
    idx_to_rfsp.to_json(j);
  }
  if (ue_aggregate_maximum_bit_rate_present) {
    j.write_fieldname("");
    ue_aggregate_maximum_bit_rate.to_json(j);
  }
  if (allowed_nssai_present) {
    j.write_fieldname("");
    allowed_nssai.to_json(j);
  }
  j.end_obj();
}

// DownlinkNonUEAssociatedNRPPaTransportIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t dl_non_ueassociated_nrp_pa_transport_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {89, 46};
  return map_enum_number(options, 2, idx, "id");
}
bool dl_non_ueassociated_nrp_pa_transport_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {89, 46};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e dl_non_ueassociated_nrp_pa_transport_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 89:
      return crit_e::reject;
    case 46:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
dl_non_ueassociated_nrp_pa_transport_ies_o::value_c
dl_non_ueassociated_nrp_pa_transport_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 89:
      ret.set(value_c::types::routing_id);
      break;
    case 46:
      ret.set(value_c::types::nrp_pa_pdu);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e dl_non_ueassociated_nrp_pa_transport_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 89:
      return presence_e::mandatory;
    case 46:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void dl_non_ueassociated_nrp_pa_transport_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::routing_id:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::nrp_pa_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    default:
      break;
  }
}
void dl_non_ueassociated_nrp_pa_transport_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::routing_id:
      c.init<unbounded_octstring<true> >();
      break;
    case types::nrp_pa_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_nrp_pa_transport_ies_o::value_c");
  }
}
dl_non_ueassociated_nrp_pa_transport_ies_o::value_c::value_c(
    const dl_non_ueassociated_nrp_pa_transport_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::routing_id:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nrp_pa_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_nrp_pa_transport_ies_o::value_c");
  }
}
dl_non_ueassociated_nrp_pa_transport_ies_o::value_c& dl_non_ueassociated_nrp_pa_transport_ies_o::value_c::operator=(
    const dl_non_ueassociated_nrp_pa_transport_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::routing_id:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nrp_pa_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_nrp_pa_transport_ies_o::value_c");
  }

  return *this;
}
unbounded_octstring<true>& dl_non_ueassociated_nrp_pa_transport_ies_o::value_c::routing_id()
{
  assert_choice_type(types::routing_id, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
unbounded_octstring<true>& dl_non_ueassociated_nrp_pa_transport_ies_o::value_c::nrp_pa_pdu()
{
  assert_choice_type(types::nrp_pa_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const unbounded_octstring<true>& dl_non_ueassociated_nrp_pa_transport_ies_o::value_c::routing_id() const
{
  assert_choice_type(types::routing_id, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const unbounded_octstring<true>& dl_non_ueassociated_nrp_pa_transport_ies_o::value_c::nrp_pa_pdu() const
{
  assert_choice_type(types::nrp_pa_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
void dl_non_ueassociated_nrp_pa_transport_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::routing_id:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::nrp_pa_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_nrp_pa_transport_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE dl_non_ueassociated_nrp_pa_transport_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::routing_id:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::nrp_pa_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_nrp_pa_transport_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_non_ueassociated_nrp_pa_transport_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::routing_id:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::nrp_pa_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_non_ueassociated_nrp_pa_transport_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* dl_non_ueassociated_nrp_pa_transport_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"OCTET STRING", "OCTET STRING"};
  return convert_enum_idx(options, 2, value, "dl_non_ueassociated_nrp_pa_transport_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<dl_non_ueassociated_nrp_pa_transport_ies_o>;

dl_non_ueassociated_nrp_pa_transport_ies_container::dl_non_ueassociated_nrp_pa_transport_ies_container() :
  routing_id(89, crit_e::reject), nrp_pa_pdu(46, crit_e::reject)
{}
SRSASN_CODE dl_non_ueassociated_nrp_pa_transport_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(routing_id.pack(bref));
  HANDLE_CODE(nrp_pa_pdu.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_non_ueassociated_nrp_pa_transport_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 89: {
        nof_mandatory_ies--;
        routing_id.id = id;
        HANDLE_CODE(routing_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(routing_id.value.unpack(bref));
        break;
      }
      case 46: {
        nof_mandatory_ies--;
        nrp_pa_pdu.id = id;
        HANDLE_CODE(nrp_pa_pdu.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(nrp_pa_pdu.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void dl_non_ueassociated_nrp_pa_transport_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  routing_id.to_json(j);
  j.write_fieldname("");
  nrp_pa_pdu.to_json(j);
  j.end_obj();
}

// XnExtTLA-Item ::= SEQUENCE
SRSASN_CODE xn_ext_tla_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ipsec_tla_present, 1));
  HANDLE_CODE(bref.pack(gtp_tlas.size() > 0, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (ipsec_tla_present) {
    HANDLE_CODE(ipsec_tla.pack(bref));
  }
  if (gtp_tlas.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, gtp_tlas, 1, 16, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE xn_ext_tla_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ipsec_tla_present, 1));
  bool gtp_tlas_present;
  HANDLE_CODE(bref.unpack(gtp_tlas_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (ipsec_tla_present) {
    HANDLE_CODE(ipsec_tla.unpack(bref));
  }
  if (gtp_tlas_present) {
    HANDLE_CODE(unpack_dyn_seq_of(gtp_tlas, bref, 1, 16, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void xn_ext_tla_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (ipsec_tla_present) {
    j.write_str("iPsecTLA", ipsec_tla.to_string());
  }
  if (gtp_tlas.size() > 0) {
    j.start_array("gTP-TLAs");
    for (const auto& e1 : gtp_tlas) {
      j.write_str(e1.to_string());
    }
    j.end_array();
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// XnTNLConfigurationInfo ::= SEQUENCE
SRSASN_CODE xn_tnl_cfg_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(xn_extended_transport_layer_addresses.size() > 0, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, xn_transport_layer_addresses, 1, 16, true));
  if (xn_extended_transport_layer_addresses.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, xn_extended_transport_layer_addresses, 1, 2, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE xn_tnl_cfg_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  bool xn_extended_transport_layer_addresses_present;
  HANDLE_CODE(bref.unpack(xn_extended_transport_layer_addresses_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(xn_transport_layer_addresses, bref, 1, 16, true));
  if (xn_extended_transport_layer_addresses_present) {
    HANDLE_CODE(unpack_dyn_seq_of(xn_extended_transport_layer_addresses, bref, 1, 2, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void xn_tnl_cfg_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("xnTransportLayerAddresses");
  for (const auto& e1 : xn_transport_layer_addresses) {
    j.write_str(e1.to_string());
  }
  j.end_array();
  if (xn_extended_transport_layer_addresses.size() > 0) {
    j.start_array("xnExtendedTransportLayerAddresses");
    for (const auto& e1 : xn_extended_transport_layer_addresses) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// SONInformationReply ::= SEQUENCE
SRSASN_CODE son_info_reply_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(xn_tnl_cfg_info_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (xn_tnl_cfg_info_present) {
    HANDLE_CODE(xn_tnl_cfg_info.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE son_info_reply_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(xn_tnl_cfg_info_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (xn_tnl_cfg_info_present) {
    HANDLE_CODE(xn_tnl_cfg_info.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void son_info_reply_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (xn_tnl_cfg_info_present) {
    j.write_fieldname("xnTNLConfigurationInfo");
    xn_tnl_cfg_info.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// SONInformationRequest ::= ENUMERATED
const char* son_info_request_opts::to_string() const
{
  static const char* options[] = {"xn-TNL-configuration-info"};
  return convert_enum_idx(options, 1, value, "son_info_request_e");
}

// SONInformation ::= CHOICE
void son_info_c::destroy_()
{
  switch (type_) {
    case types::son_info_reply:
      c.destroy<son_info_reply_s>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<son_info_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void son_info_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::son_info_request:
      break;
    case types::son_info_reply:
      c.init<son_info_reply_s>();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<son_info_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
  }
}
son_info_c::son_info_c(const son_info_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::son_info_request:
      c.init(other.c.get<son_info_request_e>());
      break;
    case types::son_info_reply:
      c.init(other.c.get<son_info_reply_s>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<son_info_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
  }
}
son_info_c& son_info_c::operator=(const son_info_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::son_info_request:
      c.set(other.c.get<son_info_request_e>());
      break;
    case types::son_info_reply:
      c.set(other.c.get<son_info_reply_s>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<son_info_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
  }

  return *this;
}
son_info_request_e& son_info_c::set_son_info_request()
{
  set(types::son_info_request);
  return c.get<son_info_request_e>();
}
son_info_reply_s& son_info_c::set_son_info_reply()
{
  set(types::son_info_reply);
  return c.get<son_info_reply_s>();
}
protocol_ie_single_container_s<son_info_ext_ies_o>& son_info_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<son_info_ext_ies_o> >();
}
void son_info_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::son_info_request:
      j.write_str("sONInformationRequest", "xn-TNL-configuration-info");
      break;
    case types::son_info_reply:
      j.write_fieldname("sONInformationReply");
      c.get<son_info_reply_s>().to_json(j);
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<son_info_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
  }
  j.end_obj();
}
SRSASN_CODE son_info_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::son_info_request:
      HANDLE_CODE(c.get<son_info_request_e>().pack(bref));
      break;
    case types::son_info_reply:
      HANDLE_CODE(c.get<son_info_reply_s>().pack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<son_info_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE son_info_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::son_info_request:
      HANDLE_CODE(c.get<son_info_request_e>().unpack(bref));
      break;
    case types::son_info_reply:
      HANDLE_CODE(c.get<son_info_reply_s>().unpack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<son_info_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "son_info_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* son_info_c::types_opts::to_string() const
{
  static const char* options[] = {"sONInformationRequest", "sONInformationReply", "choice-Extensions"};
  return convert_enum_idx(options, 3, value, "son_info_c::types");
}

// SourceRANNodeID ::= SEQUENCE
SRSASN_CODE source_ran_node_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(global_ran_node_id.pack(bref));
  HANDLE_CODE(sel_tai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE source_ran_node_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(global_ran_node_id.unpack(bref));
  HANDLE_CODE(sel_tai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void source_ran_node_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("globalRANNodeID");
  global_ran_node_id.to_json(j);
  j.write_fieldname("selectedTAI");
  sel_tai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TargetRANNodeID ::= SEQUENCE
SRSASN_CODE target_ran_node_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(global_ran_node_id.pack(bref));
  HANDLE_CODE(sel_tai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE target_ran_node_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(global_ran_node_id.unpack(bref));
  HANDLE_CODE(sel_tai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void target_ran_node_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("globalRANNodeID");
  global_ran_node_id.to_json(j);
  j.write_fieldname("selectedTAI");
  sel_tai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// SONConfigurationTransfer ::= SEQUENCE
SRSASN_CODE son_cfg_transfer_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(xn_tnl_cfg_info_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(target_ran_node_id.pack(bref));
  HANDLE_CODE(source_ran_node_id.pack(bref));
  HANDLE_CODE(son_info.pack(bref));
  if (xn_tnl_cfg_info_present) {
    HANDLE_CODE(xn_tnl_cfg_info.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE son_cfg_transfer_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(xn_tnl_cfg_info_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(target_ran_node_id.unpack(bref));
  HANDLE_CODE(source_ran_node_id.unpack(bref));
  HANDLE_CODE(son_info.unpack(bref));
  if (xn_tnl_cfg_info_present) {
    HANDLE_CODE(xn_tnl_cfg_info.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void son_cfg_transfer_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("targetRANNodeID");
  target_ran_node_id.to_json(j);
  j.write_fieldname("sourceRANNodeID");
  source_ran_node_id.to_json(j);
  j.write_fieldname("sONInformation");
  son_info.to_json(j);
  if (xn_tnl_cfg_info_present) {
    j.write_fieldname("xnTNLConfigurationInfo");
    xn_tnl_cfg_info.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// DownlinkRANConfigurationTransferIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t dl_ran_cfg_transfer_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {98, 157};
  return map_enum_number(options, 2, idx, "id");
}
bool dl_ran_cfg_transfer_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {98, 157};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e dl_ran_cfg_transfer_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 98:
      return crit_e::ignore;
    case 157:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
dl_ran_cfg_transfer_ies_o::value_c dl_ran_cfg_transfer_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 98:
      ret.set(value_c::types::son_cfg_transfer_dl);
      break;
    case 157:
      ret.set(value_c::types::endc_son_cfg_transfer_dl);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e dl_ran_cfg_transfer_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 98:
      return presence_e::optional;
    case 157:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void dl_ran_cfg_transfer_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::son_cfg_transfer_dl:
      c.destroy<son_cfg_transfer_s>();
      break;
    case types::endc_son_cfg_transfer_dl:
      c.destroy<unbounded_octstring<true> >();
      break;
    default:
      break;
  }
}
void dl_ran_cfg_transfer_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::son_cfg_transfer_dl:
      c.init<son_cfg_transfer_s>();
      break;
    case types::endc_son_cfg_transfer_dl:
      c.init<unbounded_octstring<true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_ran_cfg_transfer_ies_o::value_c");
  }
}
dl_ran_cfg_transfer_ies_o::value_c::value_c(const dl_ran_cfg_transfer_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::son_cfg_transfer_dl:
      c.init(other.c.get<son_cfg_transfer_s>());
      break;
    case types::endc_son_cfg_transfer_dl:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_ran_cfg_transfer_ies_o::value_c");
  }
}
dl_ran_cfg_transfer_ies_o::value_c&
dl_ran_cfg_transfer_ies_o::value_c::operator=(const dl_ran_cfg_transfer_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::son_cfg_transfer_dl:
      c.set(other.c.get<son_cfg_transfer_s>());
      break;
    case types::endc_son_cfg_transfer_dl:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_ran_cfg_transfer_ies_o::value_c");
  }

  return *this;
}
son_cfg_transfer_s& dl_ran_cfg_transfer_ies_o::value_c::son_cfg_transfer_dl()
{
  assert_choice_type(types::son_cfg_transfer_dl, type_, "Value");
  return c.get<son_cfg_transfer_s>();
}
unbounded_octstring<true>& dl_ran_cfg_transfer_ies_o::value_c::endc_son_cfg_transfer_dl()
{
  assert_choice_type(types::endc_son_cfg_transfer_dl, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const son_cfg_transfer_s& dl_ran_cfg_transfer_ies_o::value_c::son_cfg_transfer_dl() const
{
  assert_choice_type(types::son_cfg_transfer_dl, type_, "Value");
  return c.get<son_cfg_transfer_s>();
}
const unbounded_octstring<true>& dl_ran_cfg_transfer_ies_o::value_c::endc_son_cfg_transfer_dl() const
{
  assert_choice_type(types::endc_son_cfg_transfer_dl, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
void dl_ran_cfg_transfer_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::son_cfg_transfer_dl:
      j.write_fieldname("SONConfigurationTransfer");
      c.get<son_cfg_transfer_s>().to_json(j);
      break;
    case types::endc_son_cfg_transfer_dl:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "dl_ran_cfg_transfer_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE dl_ran_cfg_transfer_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::son_cfg_transfer_dl:
      HANDLE_CODE(c.get<son_cfg_transfer_s>().pack(bref));
      break;
    case types::endc_son_cfg_transfer_dl:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_ran_cfg_transfer_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_ran_cfg_transfer_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::son_cfg_transfer_dl:
      HANDLE_CODE(c.get<son_cfg_transfer_s>().unpack(bref));
      break;
    case types::endc_son_cfg_transfer_dl:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_ran_cfg_transfer_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* dl_ran_cfg_transfer_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"SONConfigurationTransfer", "OCTET STRING"};
  return convert_enum_idx(options, 2, value, "dl_ran_cfg_transfer_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<dl_ran_cfg_transfer_ies_o>;

dl_ran_cfg_transfer_ies_container::dl_ran_cfg_transfer_ies_container() :
  son_cfg_transfer_dl(98, crit_e::ignore), endc_son_cfg_transfer_dl(157, crit_e::ignore)
{}
SRSASN_CODE dl_ran_cfg_transfer_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += son_cfg_transfer_dl_present ? 1 : 0;
  nof_ies += endc_son_cfg_transfer_dl_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  if (son_cfg_transfer_dl_present) {
    HANDLE_CODE(son_cfg_transfer_dl.pack(bref));
  }
  if (endc_son_cfg_transfer_dl_present) {
    HANDLE_CODE(endc_son_cfg_transfer_dl.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_ran_cfg_transfer_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 98: {
        son_cfg_transfer_dl_present = true;
        son_cfg_transfer_dl.id      = id;
        HANDLE_CODE(son_cfg_transfer_dl.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(son_cfg_transfer_dl.value.unpack(bref));
        break;
      }
      case 157: {
        endc_son_cfg_transfer_dl_present = true;
        endc_son_cfg_transfer_dl.id      = id;
        HANDLE_CODE(endc_son_cfg_transfer_dl.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(endc_son_cfg_transfer_dl.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void dl_ran_cfg_transfer_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (son_cfg_transfer_dl_present) {
    j.write_fieldname("");
    son_cfg_transfer_dl.to_json(j);
  }
  if (endc_son_cfg_transfer_dl_present) {
    j.write_fieldname("");
    endc_son_cfg_transfer_dl.to_json(j);
  }
  j.end_obj();
}

// RANStatusTransfer-TransparentContainer ::= SEQUENCE
SRSASN_CODE ran_status_transfer_transparent_container_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, drbs_subject_to_status_transfer_list, 1, 32, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ran_status_transfer_transparent_container_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(drbs_subject_to_status_transfer_list, bref, 1, 32, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ran_status_transfer_transparent_container_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("dRBsSubjectToStatusTransferList");
  for (const auto& e1 : drbs_subject_to_status_transfer_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// DownlinkRANStatusTransferIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t dl_ran_status_transfer_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 84};
  return map_enum_number(options, 3, idx, "id");
}
bool dl_ran_status_transfer_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 84};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e dl_ran_status_transfer_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 84:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
dl_ran_status_transfer_ies_o::value_c dl_ran_status_transfer_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 84:
      ret.set(value_c::types::ran_status_transfer_transparent_container);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e dl_ran_status_transfer_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 84:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void dl_ran_status_transfer_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ran_status_transfer_transparent_container:
      c.destroy<ran_status_transfer_transparent_container_s>();
      break;
    default:
      break;
  }
}
void dl_ran_status_transfer_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::ran_status_transfer_transparent_container:
      c.init<ran_status_transfer_transparent_container_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_ran_status_transfer_ies_o::value_c");
  }
}
dl_ran_status_transfer_ies_o::value_c::value_c(const dl_ran_status_transfer_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_status_transfer_transparent_container:
      c.init(other.c.get<ran_status_transfer_transparent_container_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_ran_status_transfer_ies_o::value_c");
  }
}
dl_ran_status_transfer_ies_o::value_c&
dl_ran_status_transfer_ies_o::value_c::operator=(const dl_ran_status_transfer_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_status_transfer_transparent_container:
      c.set(other.c.get<ran_status_transfer_transparent_container_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_ran_status_transfer_ies_o::value_c");
  }

  return *this;
}
uint64_t& dl_ran_status_transfer_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& dl_ran_status_transfer_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
ran_status_transfer_transparent_container_s&
dl_ran_status_transfer_ies_o::value_c::ran_status_transfer_transparent_container()
{
  assert_choice_type(types::ran_status_transfer_transparent_container, type_, "Value");
  return c.get<ran_status_transfer_transparent_container_s>();
}
const uint64_t& dl_ran_status_transfer_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& dl_ran_status_transfer_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const ran_status_transfer_transparent_container_s&
dl_ran_status_transfer_ies_o::value_c::ran_status_transfer_transparent_container() const
{
  assert_choice_type(types::ran_status_transfer_transparent_container, type_, "Value");
  return c.get<ran_status_transfer_transparent_container_s>();
}
void dl_ran_status_transfer_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::ran_status_transfer_transparent_container:
      j.write_fieldname("RANStatusTransfer-TransparentContainer");
      c.get<ran_status_transfer_transparent_container_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "dl_ran_status_transfer_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE dl_ran_status_transfer_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::ran_status_transfer_transparent_container:
      HANDLE_CODE(c.get<ran_status_transfer_transparent_container_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_ran_status_transfer_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_ran_status_transfer_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::ran_status_transfer_transparent_container:
      HANDLE_CODE(c.get<ran_status_transfer_transparent_container_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_ran_status_transfer_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* dl_ran_status_transfer_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..1099511627775)", "INTEGER (0..4294967295)", "RANStatusTransfer-TransparentContainer"};
  return convert_enum_idx(options, 3, value, "dl_ran_status_transfer_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<dl_ran_status_transfer_ies_o>;

dl_ran_status_transfer_ies_container::dl_ran_status_transfer_ies_container() :
  amf_ue_ngap_id(10, crit_e::reject),
  ran_ue_ngap_id(85, crit_e::reject),
  ran_status_transfer_transparent_container(84, crit_e::reject)
{}
SRSASN_CODE dl_ran_status_transfer_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  HANDLE_CODE(ran_status_transfer_transparent_container.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_ran_status_transfer_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 85: {
        nof_mandatory_ies--;
        ran_ue_ngap_id.id = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 84: {
        nof_mandatory_ies--;
        ran_status_transfer_transparent_container.id = id;
        HANDLE_CODE(ran_status_transfer_transparent_container.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_status_transfer_transparent_container.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void dl_ran_status_transfer_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ran_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ran_status_transfer_transparent_container.to_json(j);
  j.end_obj();
}

// DownlinkUEAssociatedNRPPaTransportIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t dl_ueassociated_nrp_pa_transport_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 89, 46};
  return map_enum_number(options, 4, idx, "id");
}
bool dl_ueassociated_nrp_pa_transport_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 89, 46};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e dl_ueassociated_nrp_pa_transport_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 89:
      return crit_e::reject;
    case 46:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
dl_ueassociated_nrp_pa_transport_ies_o::value_c dl_ueassociated_nrp_pa_transport_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 89:
      ret.set(value_c::types::routing_id);
      break;
    case 46:
      ret.set(value_c::types::nrp_pa_pdu);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e dl_ueassociated_nrp_pa_transport_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 89:
      return presence_e::mandatory;
    case 46:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void dl_ueassociated_nrp_pa_transport_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::routing_id:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::nrp_pa_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    default:
      break;
  }
}
void dl_ueassociated_nrp_pa_transport_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::routing_id:
      c.init<unbounded_octstring<true> >();
      break;
    case types::nrp_pa_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_nrp_pa_transport_ies_o::value_c");
  }
}
dl_ueassociated_nrp_pa_transport_ies_o::value_c::value_c(const dl_ueassociated_nrp_pa_transport_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::routing_id:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nrp_pa_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_nrp_pa_transport_ies_o::value_c");
  }
}
dl_ueassociated_nrp_pa_transport_ies_o::value_c&
dl_ueassociated_nrp_pa_transport_ies_o::value_c::operator=(const dl_ueassociated_nrp_pa_transport_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::routing_id:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nrp_pa_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_nrp_pa_transport_ies_o::value_c");
  }

  return *this;
}
uint64_t& dl_ueassociated_nrp_pa_transport_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& dl_ueassociated_nrp_pa_transport_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
unbounded_octstring<true>& dl_ueassociated_nrp_pa_transport_ies_o::value_c::routing_id()
{
  assert_choice_type(types::routing_id, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
unbounded_octstring<true>& dl_ueassociated_nrp_pa_transport_ies_o::value_c::nrp_pa_pdu()
{
  assert_choice_type(types::nrp_pa_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const uint64_t& dl_ueassociated_nrp_pa_transport_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& dl_ueassociated_nrp_pa_transport_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const unbounded_octstring<true>& dl_ueassociated_nrp_pa_transport_ies_o::value_c::routing_id() const
{
  assert_choice_type(types::routing_id, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const unbounded_octstring<true>& dl_ueassociated_nrp_pa_transport_ies_o::value_c::nrp_pa_pdu() const
{
  assert_choice_type(types::nrp_pa_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
void dl_ueassociated_nrp_pa_transport_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::routing_id:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::nrp_pa_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_nrp_pa_transport_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE dl_ueassociated_nrp_pa_transport_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::routing_id:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::nrp_pa_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_nrp_pa_transport_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_ueassociated_nrp_pa_transport_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::routing_id:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::nrp_pa_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "dl_ueassociated_nrp_pa_transport_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* dl_ueassociated_nrp_pa_transport_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..1099511627775)", "INTEGER (0..4294967295)", "OCTET STRING", "OCTET STRING"};
  return convert_enum_idx(options, 4, value, "dl_ueassociated_nrp_pa_transport_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<dl_ueassociated_nrp_pa_transport_ies_o>;

dl_ueassociated_nrp_pa_transport_ies_container::dl_ueassociated_nrp_pa_transport_ies_container() :
  amf_ue_ngap_id(10, crit_e::reject),
  ran_ue_ngap_id(85, crit_e::reject),
  routing_id(89, crit_e::reject),
  nrp_pa_pdu(46, crit_e::reject)
{}
SRSASN_CODE dl_ueassociated_nrp_pa_transport_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 4;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  HANDLE_CODE(routing_id.pack(bref));
  HANDLE_CODE(nrp_pa_pdu.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE dl_ueassociated_nrp_pa_transport_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 4;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 85: {
        nof_mandatory_ies--;
        ran_ue_ngap_id.id = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 89: {
        nof_mandatory_ies--;
        routing_id.id = id;
        HANDLE_CODE(routing_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(routing_id.value.unpack(bref));
        break;
      }
      case 46: {
        nof_mandatory_ies--;
        nrp_pa_pdu.id = id;
        HANDLE_CODE(nrp_pa_pdu.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(nrp_pa_pdu.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void dl_ueassociated_nrp_pa_transport_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ran_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  routing_id.to_json(j);
  j.write_fieldname("");
  nrp_pa_pdu.to_json(j);
  j.end_obj();
}

// DelayCritical ::= ENUMERATED
const char* delay_crit_opts::to_string() const
{
  static const char* options[] = {"delay-critical", "non-delay-critical"};
  return convert_enum_idx(options, 2, value, "delay_crit_e");
}

// PacketErrorRate ::= SEQUENCE
SRSASN_CODE packet_error_rate_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, per_scalar, (uint8_t)0u, (uint8_t)9u, true, true));
  HANDLE_CODE(pack_integer(bref, per_exponent, (uint8_t)0u, (uint8_t)9u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE packet_error_rate_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(per_scalar, bref, (uint8_t)0u, (uint8_t)9u, true, true));
  HANDLE_CODE(unpack_integer(per_exponent, bref, (uint8_t)0u, (uint8_t)9u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void packet_error_rate_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pERScalar", per_scalar);
  j.write_int("pERExponent", per_exponent);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// Dynamic5QIDescriptor ::= SEQUENCE
SRSASN_CODE dynamic5_qi_descriptor_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(five_qi_present, 1));
  HANDLE_CODE(bref.pack(delay_crit_present, 1));
  HANDLE_CODE(bref.pack(averaging_win_present, 1));
  HANDLE_CODE(bref.pack(maximum_data_burst_volume_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, prio_level_qos, (uint8_t)1u, (uint8_t)127u, true, true));
  HANDLE_CODE(pack_integer(bref, packet_delay_budget, (uint16_t)0u, (uint16_t)1023u, true, true));
  HANDLE_CODE(packet_error_rate.pack(bref));
  if (five_qi_present) {
    HANDLE_CODE(pack_integer(bref, five_qi, (uint16_t)0u, (uint16_t)255u, true, true));
  }
  if (delay_crit_present) {
    HANDLE_CODE(delay_crit.pack(bref));
  }
  if (averaging_win_present) {
    HANDLE_CODE(pack_integer(bref, averaging_win, (uint16_t)0u, (uint16_t)4095u, true, true));
  }
  if (maximum_data_burst_volume_present) {
    HANDLE_CODE(pack_integer(bref, maximum_data_burst_volume, (uint16_t)0u, (uint16_t)4095u, true, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE dynamic5_qi_descriptor_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(five_qi_present, 1));
  HANDLE_CODE(bref.unpack(delay_crit_present, 1));
  HANDLE_CODE(bref.unpack(averaging_win_present, 1));
  HANDLE_CODE(bref.unpack(maximum_data_burst_volume_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(prio_level_qos, bref, (uint8_t)1u, (uint8_t)127u, true, true));
  HANDLE_CODE(unpack_integer(packet_delay_budget, bref, (uint16_t)0u, (uint16_t)1023u, true, true));
  HANDLE_CODE(packet_error_rate.unpack(bref));
  if (five_qi_present) {
    HANDLE_CODE(unpack_integer(five_qi, bref, (uint16_t)0u, (uint16_t)255u, true, true));
  }
  if (delay_crit_present) {
    HANDLE_CODE(delay_crit.unpack(bref));
  }
  if (averaging_win_present) {
    HANDLE_CODE(unpack_integer(averaging_win, bref, (uint16_t)0u, (uint16_t)4095u, true, true));
  }
  if (maximum_data_burst_volume_present) {
    HANDLE_CODE(unpack_integer(maximum_data_burst_volume, bref, (uint16_t)0u, (uint16_t)4095u, true, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void dynamic5_qi_descriptor_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("priorityLevelQos", prio_level_qos);
  j.write_int("packetDelayBudget", packet_delay_budget);
  j.write_fieldname("packetErrorRate");
  packet_error_rate.to_json(j);
  if (five_qi_present) {
    j.write_int("fiveQI", five_qi);
  }
  if (delay_crit_present) {
    j.write_str("delayCritical", delay_crit.to_string());
  }
  if (averaging_win_present) {
    j.write_int("averagingWindow", averaging_win);
  }
  if (maximum_data_burst_volume_present) {
    j.write_int("maximumDataBurstVolume", maximum_data_burst_volume);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// DLForwarding ::= ENUMERATED
const char* dl_forwarding_opts::to_string() const
{
  static const char* options[] = {"dl-forwarding-proposed"};
  return convert_enum_idx(options, 1, value, "dl_forwarding_e");
}

// E-RABInformationItem ::= SEQUENCE
SRSASN_CODE erab_info_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(dlforwarding_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, erab_id, (uint8_t)0u, (uint8_t)15u, true, true));
  if (dlforwarding_present) {
    HANDLE_CODE(dlforwarding.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE erab_info_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(dlforwarding_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(erab_id, bref, (uint8_t)0u, (uint8_t)15u, true, true));
  if (dlforwarding_present) {
    HANDLE_CODE(dlforwarding.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void erab_info_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("e-RAB-ID", erab_id);
  if (dlforwarding_present) {
    j.write_str("dLForwarding", "dl-forwarding-proposed");
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EPS-TAI ::= SEQUENCE
SRSASN_CODE eps_tai_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.pack(bref));
  HANDLE_CODE(eps_tac.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE eps_tai_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(plmn_id.unpack(bref));
  HANDLE_CODE(eps_tac.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void eps_tai_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("pLMNIdentity", plmn_id.to_string());
  j.write_str("ePS-TAC", eps_tac.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EmergencyFallbackRequestIndicator ::= ENUMERATED
const char* emergency_fallback_request_ind_opts::to_string() const
{
  static const char* options[] = {"emergency-fallback-requested"};
  return convert_enum_idx(options, 1, value, "emergency_fallback_request_ind_e");
}

// EmergencyServiceTargetCN ::= ENUMERATED
const char* emergency_service_target_cn_opts::to_string() const
{
  static const char* options[] = {"fiveGC", "epc"};
  return convert_enum_idx(options, 2, value, "emergency_service_target_cn_e");
}
uint8_t emergency_service_target_cn_opts::to_number() const
{
  static const uint8_t options[] = {5};
  return map_enum_number(options, 1, value, "emergency_service_target_cn_e");
}

// EmergencyFallbackIndicator ::= SEQUENCE
SRSASN_CODE emergency_fallback_ind_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(emergency_service_target_cn_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(emergency_fallback_request_ind.pack(bref));
  if (emergency_service_target_cn_present) {
    HANDLE_CODE(emergency_service_target_cn.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE emergency_fallback_ind_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(emergency_service_target_cn_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(emergency_fallback_request_ind.unpack(bref));
  if (emergency_service_target_cn_present) {
    HANDLE_CODE(emergency_service_target_cn.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void emergency_fallback_ind_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("emergencyFallbackRequestIndicator", "emergency-fallback-requested");
  if (emergency_service_target_cn_present) {
    j.write_str("emergencyServiceTargetCN", emergency_service_target_cn.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ErrorIndicationIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t error_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 15, 19};
  return map_enum_number(options, 4, idx, "id");
}
bool error_ind_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 15, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e error_ind_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::ignore;
    case 85:
      return crit_e::ignore;
    case 15:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
error_ind_ies_o::value_c error_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 15:
      ret.set(value_c::types::cause);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e error_ind_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::optional;
    case 85:
      return presence_e::optional;
    case 15:
      return presence_e::optional;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void error_ind_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void error_ind_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
  }
}
error_ind_ies_o::value_c::value_c(const error_ind_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
  }
}
error_ind_ies_o::value_c& error_ind_ies_o::value_c::operator=(const error_ind_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
  }

  return *this;
}
uint64_t& error_ind_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& error_ind_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
cause_c& error_ind_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
crit_diagnostics_s& error_ind_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& error_ind_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& error_ind_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const cause_c& error_ind_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const crit_diagnostics_s& error_ind_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void error_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE error_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE error_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "error_ind_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* error_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..1099511627775)", "INTEGER (0..4294967295)", "Cause", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 4, value, "error_ind_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<error_ind_ies_o>;

error_ind_ies_container::error_ind_ies_container() :
  amf_ue_ngap_id(10, crit_e::ignore),
  ran_ue_ngap_id(85, crit_e::ignore),
  cause(15, crit_e::ignore),
  crit_diagnostics(19, crit_e::ignore)
{}
SRSASN_CODE error_ind_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 0;
  nof_ies += amf_ue_ngap_id_present ? 1 : 0;
  nof_ies += ran_ue_ngap_id_present ? 1 : 0;
  nof_ies += cause_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  if (amf_ue_ngap_id_present) {
    HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  }
  if (ran_ue_ngap_id_present) {
    HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  }
  if (cause_present) {
    HANDLE_CODE(cause.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE error_ind_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        amf_ue_ngap_id_present = true;
        amf_ue_ngap_id.id      = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 85: {
        ran_ue_ngap_id_present = true;
        ran_ue_ngap_id.id      = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 15: {
        cause_present = true;
        cause.id      = id;
        HANDLE_CODE(cause.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cause.value.unpack(bref));
        break;
      }
      case 19: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }

  return SRSASN_SUCCESS;
}
void error_ind_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  if (amf_ue_ngap_id_present) {
    j.write_fieldname("");
    amf_ue_ngap_id.to_json(j);
  }
  if (ran_ue_ngap_id_present) {
    j.write_fieldname("");
    ran_ue_ngap_id.to_json(j);
  }
  if (cause_present) {
    j.write_fieldname("");
    cause.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// FiveG-S-TMSI ::= SEQUENCE
SRSASN_CODE five_g_s_tmsi_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(amf_set_id.pack(bref));
  HANDLE_CODE(amf_pointer.pack(bref));
  HANDLE_CODE(five_g_tmsi.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE five_g_s_tmsi_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(amf_set_id.unpack(bref));
  HANDLE_CODE(amf_pointer.unpack(bref));
  HANDLE_CODE(five_g_tmsi.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void five_g_s_tmsi_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("aMFSetID", amf_set_id.to_string());
  j.write_str("aMFPointer", amf_pointer.to_string());
  j.write_str("fiveG-TMSI", five_g_tmsi.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// NotificationControl ::= ENUMERATED
const char* notif_ctrl_opts::to_string() const
{
  static const char* options[] = {"notification-requested"};
  return convert_enum_idx(options, 1, value, "notif_ctrl_e");
}

// GBR-QosInformation ::= SEQUENCE
SRSASN_CODE gbr_qos_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(notif_ctrl_present, 1));
  HANDLE_CODE(bref.pack(maximum_packet_loss_rate_dl_present, 1));
  HANDLE_CODE(bref.pack(maximum_packet_loss_rate_ul_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, maximum_flow_bit_rate_dl, (uint64_t)0u, (uint64_t)4000000000000u, true, true));
  HANDLE_CODE(pack_integer(bref, maximum_flow_bit_rate_ul, (uint64_t)0u, (uint64_t)4000000000000u, true, true));
  HANDLE_CODE(pack_integer(bref, guaranteed_flow_bit_rate_dl, (uint64_t)0u, (uint64_t)4000000000000u, true, true));
  HANDLE_CODE(pack_integer(bref, guaranteed_flow_bit_rate_ul, (uint64_t)0u, (uint64_t)4000000000000u, true, true));
  if (notif_ctrl_present) {
    HANDLE_CODE(notif_ctrl.pack(bref));
  }
  if (maximum_packet_loss_rate_dl_present) {
    HANDLE_CODE(pack_integer(bref, maximum_packet_loss_rate_dl, (uint16_t)0u, (uint16_t)1000u, true, true));
  }
  if (maximum_packet_loss_rate_ul_present) {
    HANDLE_CODE(pack_integer(bref, maximum_packet_loss_rate_ul, (uint16_t)0u, (uint16_t)1000u, true, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE gbr_qos_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(notif_ctrl_present, 1));
  HANDLE_CODE(bref.unpack(maximum_packet_loss_rate_dl_present, 1));
  HANDLE_CODE(bref.unpack(maximum_packet_loss_rate_ul_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(maximum_flow_bit_rate_dl, bref, (uint64_t)0u, (uint64_t)4000000000000u, true, true));
  HANDLE_CODE(unpack_integer(maximum_flow_bit_rate_ul, bref, (uint64_t)0u, (uint64_t)4000000000000u, true, true));
  HANDLE_CODE(unpack_integer(guaranteed_flow_bit_rate_dl, bref, (uint64_t)0u, (uint64_t)4000000000000u, true, true));
  HANDLE_CODE(unpack_integer(guaranteed_flow_bit_rate_ul, bref, (uint64_t)0u, (uint64_t)4000000000000u, true, true));
  if (notif_ctrl_present) {
    HANDLE_CODE(notif_ctrl.unpack(bref));
  }
  if (maximum_packet_loss_rate_dl_present) {
    HANDLE_CODE(unpack_integer(maximum_packet_loss_rate_dl, bref, (uint16_t)0u, (uint16_t)1000u, true, true));
  }
  if (maximum_packet_loss_rate_ul_present) {
    HANDLE_CODE(unpack_integer(maximum_packet_loss_rate_ul, bref, (uint16_t)0u, (uint16_t)1000u, true, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void gbr_qos_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("maximumFlowBitRateDL", maximum_flow_bit_rate_dl);
  j.write_int("maximumFlowBitRateUL", maximum_flow_bit_rate_ul);
  j.write_int("guaranteedFlowBitRateDL", guaranteed_flow_bit_rate_dl);
  j.write_int("guaranteedFlowBitRateUL", guaranteed_flow_bit_rate_ul);
  if (notif_ctrl_present) {
    j.write_str("notificationControl", "notification-requested");
  }
  if (maximum_packet_loss_rate_dl_present) {
    j.write_int("maximumPacketLossRateDL", maximum_packet_loss_rate_dl);
  }
  if (maximum_packet_loss_rate_ul_present) {
    j.write_int("maximumPacketLossRateUL", maximum_packet_loss_rate_ul);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// HandoverCancelIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ho_cancel_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 15};
  return map_enum_number(options, 3, idx, "id");
}
bool ho_cancel_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 15};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_cancel_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 15:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_cancel_ies_o::value_c ho_cancel_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 15:
      ret.set(value_c::types::cause);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_cancel_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 15:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_cancel_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    default:
      break;
  }
}
void ho_cancel_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
  }
}
ho_cancel_ies_o::value_c::value_c(const ho_cancel_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
  }
}
ho_cancel_ies_o::value_c& ho_cancel_ies_o::value_c::operator=(const ho_cancel_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_cancel_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& ho_cancel_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
cause_c& ho_cancel_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const uint64_t& ho_cancel_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& ho_cancel_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const cause_c& ho_cancel_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
void ho_cancel_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_cancel_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cancel_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_cancel_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)", "INTEGER (0..4294967295)", "Cause"};
  return convert_enum_idx(options, 3, value, "ho_cancel_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_cancel_ies_o>;

ho_cancel_ies_container::ho_cancel_ies_container() :
  amf_ue_ngap_id(10, crit_e::reject), ran_ue_ngap_id(85, crit_e::reject), cause(15, crit_e::ignore)
{}
SRSASN_CODE ho_cancel_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  HANDLE_CODE(cause.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cancel_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 85: {
        nof_mandatory_ies--;
        ran_ue_ngap_id.id = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 15: {
        nof_mandatory_ies--;
        cause.id = id;
        HANDLE_CODE(cause.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cause.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_cancel_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ran_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  cause.to_json(j);
  j.end_obj();
}

// HandoverCancelAcknowledgeIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ho_cancel_ack_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 19};
  return map_enum_number(options, 3, idx, "id");
}
bool ho_cancel_ack_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_cancel_ack_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::ignore;
    case 85:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_cancel_ack_ies_o::value_c ho_cancel_ack_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_cancel_ack_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_cancel_ack_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ho_cancel_ack_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
  }
}
ho_cancel_ack_ies_o::value_c::value_c(const ho_cancel_ack_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
  }
}
ho_cancel_ack_ies_o::value_c& ho_cancel_ack_ies_o::value_c::operator=(const ho_cancel_ack_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_cancel_ack_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& ho_cancel_ack_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
crit_diagnostics_s& ho_cancel_ack_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& ho_cancel_ack_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& ho_cancel_ack_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const crit_diagnostics_s& ho_cancel_ack_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void ho_cancel_ack_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_cancel_ack_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cancel_ack_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cancel_ack_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_cancel_ack_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)", "INTEGER (0..4294967295)", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 3, value, "ho_cancel_ack_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_cancel_ack_ies_o>;

ho_cancel_ack_ies_container::ho_cancel_ack_ies_container() :
  amf_ue_ngap_id(10, crit_e::ignore), ran_ue_ngap_id(85, crit_e::ignore), crit_diagnostics(19, crit_e::ignore)
{}
SRSASN_CODE ho_cancel_ack_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cancel_ack_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 85: {
        nof_mandatory_ies--;
        ran_ue_ngap_id.id = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 19: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_cancel_ack_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ran_ue_ngap_id.to_json(j);
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceHandoverItem ::= SEQUENCE
SRSASN_CODE pdu_session_res_ho_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(ho_cmd_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_ho_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(ho_cmd_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_ho_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("handoverCommandTransfer", ho_cmd_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceToReleaseItemHOCmd ::= SEQUENCE
SRSASN_CODE pdu_session_res_to_release_item_ho_cmd_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(ho_prep_unsuccessful_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_to_release_item_ho_cmd_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(ho_prep_unsuccessful_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_to_release_item_ho_cmd_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("handoverPreparationUnsuccessfulTransfer", ho_prep_unsuccessful_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// HandoverType ::= ENUMERATED
const char* handov_type_opts::to_string() const
{
  static const char* options[] = {"intra5gs", "fivegs-to-eps", "eps-to-5gs"};
  return convert_enum_idx(options, 3, value, "handov_type_e");
}

// HandoverCommandIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ho_cmd_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 29, 39, 59, 78, 106, 19};
  return map_enum_number(options, 8, idx, "id");
}
bool ho_cmd_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 29, 39, 59, 78, 106, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_cmd_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 29:
      return crit_e::reject;
    case 39:
      return crit_e::reject;
    case 59:
      return crit_e::ignore;
    case 78:
      return crit_e::ignore;
    case 106:
      return crit_e::reject;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_cmd_ies_o::value_c ho_cmd_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 29:
      ret.set(value_c::types::handov_type);
      break;
    case 39:
      ret.set(value_c::types::nas_security_params_from_ngran);
      break;
    case 59:
      ret.set(value_c::types::pdu_session_res_ho_list);
      break;
    case 78:
      ret.set(value_c::types::pdu_session_res_to_release_list_ho_cmd);
      break;
    case 106:
      ret.set(value_c::types::target_to_source_transparent_container);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_cmd_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 29:
      return presence_e::mandatory;
    case 39:
      return presence_e::conditional;
    case 59:
      return presence_e::mandatory;
    case 78:
      return presence_e::optional;
    case 106:
      return presence_e::mandatory;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_cmd_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::nas_security_params_from_ngran:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::pdu_session_res_ho_list:
      c.destroy<pdu_session_res_ho_list_l>();
      break;
    case types::pdu_session_res_to_release_list_ho_cmd:
      c.destroy<pdu_session_res_to_release_list_ho_cmd_l>();
      break;
    case types::target_to_source_transparent_container:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ho_cmd_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::handov_type:
      break;
    case types::nas_security_params_from_ngran:
      c.init<unbounded_octstring<true> >();
      break;
    case types::pdu_session_res_ho_list:
      c.init<pdu_session_res_ho_list_l>();
      break;
    case types::pdu_session_res_to_release_list_ho_cmd:
      c.init<pdu_session_res_to_release_list_ho_cmd_l>();
      break;
    case types::target_to_source_transparent_container:
      c.init<unbounded_octstring<true> >();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
  }
}
ho_cmd_ies_o::value_c::value_c(const ho_cmd_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::handov_type:
      c.init(other.c.get<handov_type_e>());
      break;
    case types::nas_security_params_from_ngran:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::pdu_session_res_ho_list:
      c.init(other.c.get<pdu_session_res_ho_list_l>());
      break;
    case types::pdu_session_res_to_release_list_ho_cmd:
      c.init(other.c.get<pdu_session_res_to_release_list_ho_cmd_l>());
      break;
    case types::target_to_source_transparent_container:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
  }
}
ho_cmd_ies_o::value_c& ho_cmd_ies_o::value_c::operator=(const ho_cmd_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::handov_type:
      c.set(other.c.get<handov_type_e>());
      break;
    case types::nas_security_params_from_ngran:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::pdu_session_res_ho_list:
      c.set(other.c.get<pdu_session_res_ho_list_l>());
      break;
    case types::pdu_session_res_to_release_list_ho_cmd:
      c.set(other.c.get<pdu_session_res_to_release_list_ho_cmd_l>());
      break;
    case types::target_to_source_transparent_container:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_cmd_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& ho_cmd_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
handov_type_e& ho_cmd_ies_o::value_c::handov_type()
{
  assert_choice_type(types::handov_type, type_, "Value");
  return c.get<handov_type_e>();
}
unbounded_octstring<true>& ho_cmd_ies_o::value_c::nas_security_params_from_ngran()
{
  assert_choice_type(types::nas_security_params_from_ngran, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
pdu_session_res_ho_list_l& ho_cmd_ies_o::value_c::pdu_session_res_ho_list()
{
  assert_choice_type(types::pdu_session_res_ho_list, type_, "Value");
  return c.get<pdu_session_res_ho_list_l>();
}
pdu_session_res_to_release_list_ho_cmd_l& ho_cmd_ies_o::value_c::pdu_session_res_to_release_list_ho_cmd()
{
  assert_choice_type(types::pdu_session_res_to_release_list_ho_cmd, type_, "Value");
  return c.get<pdu_session_res_to_release_list_ho_cmd_l>();
}
unbounded_octstring<true>& ho_cmd_ies_o::value_c::target_to_source_transparent_container()
{
  assert_choice_type(types::target_to_source_transparent_container, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
crit_diagnostics_s& ho_cmd_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& ho_cmd_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& ho_cmd_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const handov_type_e& ho_cmd_ies_o::value_c::handov_type() const
{
  assert_choice_type(types::handov_type, type_, "Value");
  return c.get<handov_type_e>();
}
const unbounded_octstring<true>& ho_cmd_ies_o::value_c::nas_security_params_from_ngran() const
{
  assert_choice_type(types::nas_security_params_from_ngran, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const pdu_session_res_ho_list_l& ho_cmd_ies_o::value_c::pdu_session_res_ho_list() const
{
  assert_choice_type(types::pdu_session_res_ho_list, type_, "Value");
  return c.get<pdu_session_res_ho_list_l>();
}
const pdu_session_res_to_release_list_ho_cmd_l& ho_cmd_ies_o::value_c::pdu_session_res_to_release_list_ho_cmd() const
{
  assert_choice_type(types::pdu_session_res_to_release_list_ho_cmd, type_, "Value");
  return c.get<pdu_session_res_to_release_list_ho_cmd_l>();
}
const unbounded_octstring<true>& ho_cmd_ies_o::value_c::target_to_source_transparent_container() const
{
  assert_choice_type(types::target_to_source_transparent_container, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const crit_diagnostics_s& ho_cmd_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void ho_cmd_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::handov_type:
      j.write_str("HandoverType", c.get<handov_type_e>().to_string());
      break;
    case types::nas_security_params_from_ngran:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::pdu_session_res_ho_list:
      j.start_array("PDUSessionResourceHandoverList");
      for (const auto& e1 : c.get<pdu_session_res_ho_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::pdu_session_res_to_release_list_ho_cmd:
      j.start_array("PDUSessionResourceToReleaseListHOCmd");
      for (const auto& e1 : c.get<pdu_session_res_to_release_list_ho_cmd_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::target_to_source_transparent_container:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_cmd_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().pack(bref));
      break;
    case types::nas_security_params_from_ngran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::pdu_session_res_ho_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_ho_list_l>(), 1, 256, true));
      break;
    case types::pdu_session_res_to_release_list_ho_cmd:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_to_release_list_ho_cmd_l>(), 1, 256, true));
      break;
    case types::target_to_source_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cmd_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().unpack(bref));
      break;
    case types::nas_security_params_from_ngran:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::pdu_session_res_ho_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_ho_list_l>(), bref, 1, 256, true));
      break;
    case types::pdu_session_res_to_release_list_ho_cmd:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_to_release_list_ho_cmd_l>(), bref, 1, 256, true));
      break;
    case types::target_to_source_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_cmd_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_cmd_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "HandoverType",
                                  "OCTET STRING",
                                  "PDUSessionResourceHandoverList",
                                  "PDUSessionResourceToReleaseListHOCmd",
                                  "OCTET STRING",
                                  "CriticalityDiagnostics"};
  return convert_enum_idx(options, 8, value, "ho_cmd_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_cmd_ies_o>;

ho_cmd_ies_container::ho_cmd_ies_container() :
  amf_ue_ngap_id(10, crit_e::reject),
  ran_ue_ngap_id(85, crit_e::reject),
  handov_type(29, crit_e::reject),
  nas_security_params_from_ngran(39, crit_e::reject),
  pdu_session_res_ho_list(59, crit_e::ignore),
  pdu_session_res_to_release_list_ho_cmd(78, crit_e::ignore),
  target_to_source_transparent_container(106, crit_e::reject),
  crit_diagnostics(19, crit_e::ignore)
{}
SRSASN_CODE ho_cmd_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 5;
  nof_ies += nas_security_params_from_ngran_present ? 1 : 0;
  nof_ies += pdu_session_res_to_release_list_ho_cmd_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  HANDLE_CODE(handov_type.pack(bref));
  if (nas_security_params_from_ngran_present) {
    HANDLE_CODE(nas_security_params_from_ngran.pack(bref));
  }
  HANDLE_CODE(pdu_session_res_ho_list.pack(bref));
  if (pdu_session_res_to_release_list_ho_cmd_present) {
    HANDLE_CODE(pdu_session_res_to_release_list_ho_cmd.pack(bref));
  }
  HANDLE_CODE(target_to_source_transparent_container.pack(bref));
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cmd_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 5;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 85: {
        nof_mandatory_ies--;
        ran_ue_ngap_id.id = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 29: {
        nof_mandatory_ies--;
        handov_type.id = id;
        HANDLE_CODE(handov_type.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(handov_type.value.unpack(bref));
        break;
      }
      case 39: {
        nas_security_params_from_ngran_present = true;
        nas_security_params_from_ngran.id      = id;
        HANDLE_CODE(nas_security_params_from_ngran.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(nas_security_params_from_ngran.value.unpack(bref));
        break;
      }
      case 59: {
        nof_mandatory_ies--;
        pdu_session_res_ho_list.id = id;
        HANDLE_CODE(pdu_session_res_ho_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(pdu_session_res_ho_list.value.unpack(bref));
        break;
      }
      case 78: {
        pdu_session_res_to_release_list_ho_cmd_present = true;
        pdu_session_res_to_release_list_ho_cmd.id      = id;
        HANDLE_CODE(pdu_session_res_to_release_list_ho_cmd.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(pdu_session_res_to_release_list_ho_cmd.value.unpack(bref));
        break;
      }
      case 106: {
        nof_mandatory_ies--;
        target_to_source_transparent_container.id = id;
        HANDLE_CODE(target_to_source_transparent_container.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(target_to_source_transparent_container.value.unpack(bref));
        break;
      }
      case 19: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_cmd_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ran_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  handov_type.to_json(j);
  if (nas_security_params_from_ngran_present) {
    j.write_fieldname("");
    nas_security_params_from_ngran.to_json(j);
  }
  j.write_fieldname("");
  pdu_session_res_ho_list.to_json(j);
  if (pdu_session_res_to_release_list_ho_cmd_present) {
    j.write_fieldname("");
    pdu_session_res_to_release_list_ho_cmd.to_json(j);
  }
  j.write_fieldname("");
  target_to_source_transparent_container.to_json(j);
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// QosFlowPerTNLInformation ::= SEQUENCE
SRSASN_CODE qos_flow_per_tnl_info_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(uptransport_layer_info.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, associated_qos_flow_list, 1, 64, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE qos_flow_per_tnl_info_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(uptransport_layer_info.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(associated_qos_flow_list, bref, 1, 64, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void qos_flow_per_tnl_info_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("uPTransportLayerInformation");
  uptransport_layer_info.to_json(j);
  j.start_array("associatedQosFlowList");
  for (const auto& e1 : associated_qos_flow_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// QosFlowPerTNLInformationItem ::= SEQUENCE
SRSASN_CODE qos_flow_per_tnl_info_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(qos_flow_per_tnl_info.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE qos_flow_per_tnl_info_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(qos_flow_per_tnl_info.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void qos_flow_per_tnl_info_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("qosFlowPerTNLInformation");
  qos_flow_per_tnl_info.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// QosFlowToBeForwardedItem ::= SEQUENCE
SRSASN_CODE qos_flow_to_be_forwarded_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, qos_flow_id, (uint8_t)0u, (uint8_t)63u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE qos_flow_to_be_forwarded_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(qos_flow_id, bref, (uint8_t)0u, (uint8_t)63u, true, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void qos_flow_to_be_forwarded_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("qosFlowIdentifier", qos_flow_id);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// HandoverCommandTransfer-ExtIEs ::= OBJECT SET OF NGAP-PROTOCOL-EXTENSION
uint32_t ho_cmd_transfer_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {152};
  return map_enum_number(options, 1, idx, "id");
}
bool ho_cmd_transfer_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 152 == id;
}
crit_e ho_cmd_transfer_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 152) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
ho_cmd_transfer_ext_ies_o::ext_c ho_cmd_transfer_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 152) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_cmd_transfer_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 152) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void ho_cmd_transfer_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("QosFlowPerTNLInformationList");
  for (const auto& e1 : c) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}
SRSASN_CODE ho_cmd_transfer_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(pack_dyn_seq_of(bref, c, 1, 3, true));
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cmd_transfer_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(unpack_dyn_seq_of(c, bref, 1, 3, true));
  return SRSASN_SUCCESS;
}

const char* ho_cmd_transfer_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"QosFlowPerTNLInformationList"};
  return convert_enum_idx(options, 1, value, "ho_cmd_transfer_ext_ies_o::ext_c::types");
}

// HandoverCommandTransfer ::= SEQUENCE
SRSASN_CODE ho_cmd_transfer_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(dlforwarding_up_tnl_info_present, 1));
  HANDLE_CODE(bref.pack(qos_flow_to_be_forwarded_list.size() > 0, 1));
  HANDLE_CODE(bref.pack(data_forwarding_resp_drb_list.size() > 0, 1));
  HANDLE_CODE(bref.pack(ie_exts.size() > 0, 1));

  if (dlforwarding_up_tnl_info_present) {
    HANDLE_CODE(dlforwarding_up_tnl_info.pack(bref));
  }
  if (qos_flow_to_be_forwarded_list.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, qos_flow_to_be_forwarded_list, 1, 64, true));
  }
  if (data_forwarding_resp_drb_list.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, data_forwarding_resp_drb_list, 1, 32, true));
  }
  if (ie_exts.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_cmd_transfer_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(dlforwarding_up_tnl_info_present, 1));
  bool qos_flow_to_be_forwarded_list_present;
  HANDLE_CODE(bref.unpack(qos_flow_to_be_forwarded_list_present, 1));
  bool data_forwarding_resp_drb_list_present;
  HANDLE_CODE(bref.unpack(data_forwarding_resp_drb_list_present, 1));
  bool ie_exts_present;
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (dlforwarding_up_tnl_info_present) {
    HANDLE_CODE(dlforwarding_up_tnl_info.unpack(bref));
  }
  if (qos_flow_to_be_forwarded_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(qos_flow_to_be_forwarded_list, bref, 1, 64, true));
  }
  if (data_forwarding_resp_drb_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(data_forwarding_resp_drb_list, bref, 1, 32, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void ho_cmd_transfer_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (dlforwarding_up_tnl_info_present) {
    j.write_fieldname("dLForwardingUP-TNLInformation");
    dlforwarding_up_tnl_info.to_json(j);
  }
  if (qos_flow_to_be_forwarded_list.size() > 0) {
    j.start_array("qosFlowToBeForwardedList");
    for (const auto& e1 : qos_flow_to_be_forwarded_list) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (data_forwarding_resp_drb_list.size() > 0) {
    j.start_array("dataForwardingResponseDRBList");
    for (const auto& e1 : data_forwarding_resp_drb_list) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ie_exts.size() > 0) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// HandoverFailureIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ho_fail_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 15, 19};
  return map_enum_number(options, 3, idx, "id");
}
bool ho_fail_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 15, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_fail_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::ignore;
    case 15:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_fail_ies_o::value_c ho_fail_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 15:
      ret.set(value_c::types::cause);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_fail_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 15:
      return presence_e::mandatory;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_fail_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ho_fail_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
  }
}
ho_fail_ies_o::value_c::value_c(const ho_fail_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
  }
}
ho_fail_ies_o::value_c& ho_fail_ies_o::value_c::operator=(const ho_fail_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_fail_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
cause_c& ho_fail_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
crit_diagnostics_s& ho_fail_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& ho_fail_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const cause_c& ho_fail_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const crit_diagnostics_s& ho_fail_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void ho_fail_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_fail_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_fail_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_fail_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_fail_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)", "Cause", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 3, value, "ho_fail_ies_o::value_c::types");
}
uint8_t ho_fail_ies_o::value_c::types_opts::to_number() const
{
  static const uint8_t options[] = {0};
  return map_enum_number(options, 1, value, "ho_fail_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_fail_ies_o>;

ho_fail_ies_container::ho_fail_ies_container() :
  amf_ue_ngap_id(10, crit_e::ignore), cause(15, crit_e::ignore), crit_diagnostics(19, crit_e::ignore)
{}
SRSASN_CODE ho_fail_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(cause.pack(bref));
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_fail_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 15: {
        nof_mandatory_ies--;
        cause.id = id;
        HANDLE_CODE(cause.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cause.value.unpack(bref));
        break;
      }
      case 19: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_fail_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  cause.to_json(j);
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// UserLocationInformationEUTRA ::= SEQUENCE
SRSASN_CODE user_location_info_eutra_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(time_stamp_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(eutra_cgi.pack(bref));
  HANDLE_CODE(tai.pack(bref));
  if (time_stamp_present) {
    HANDLE_CODE(time_stamp.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE user_location_info_eutra_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(time_stamp_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(eutra_cgi.unpack(bref));
  HANDLE_CODE(tai.unpack(bref));
  if (time_stamp_present) {
    HANDLE_CODE(time_stamp.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void user_location_info_eutra_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("eUTRA-CGI");
  eutra_cgi.to_json(j);
  j.write_fieldname("tAI");
  tai.to_json(j);
  if (time_stamp_present) {
    j.write_str("timeStamp", time_stamp.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UserLocationInformationN3IWF ::= SEQUENCE
SRSASN_CODE user_location_info_n3_iwf_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ipaddress.pack(bref));
  HANDLE_CODE(port_num.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE user_location_info_n3_iwf_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ipaddress.unpack(bref));
  HANDLE_CODE(port_num.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void user_location_info_n3_iwf_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("iPAddress", ipaddress.to_string());
  j.write_str("portNumber", port_num.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UserLocationInformationNR ::= SEQUENCE
SRSASN_CODE user_location_info_nr_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(time_stamp_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(nr_cgi.pack(bref));
  HANDLE_CODE(tai.pack(bref));
  if (time_stamp_present) {
    HANDLE_CODE(time_stamp.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE user_location_info_nr_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(time_stamp_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(nr_cgi.unpack(bref));
  HANDLE_CODE(tai.unpack(bref));
  if (time_stamp_present) {
    HANDLE_CODE(time_stamp.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void user_location_info_nr_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("nR-CGI");
  nr_cgi.to_json(j);
  j.write_fieldname("tAI");
  tai.to_json(j);
  if (time_stamp_present) {
    j.write_str("timeStamp", time_stamp.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UserLocationInformation ::= CHOICE
void user_location_info_c::destroy_()
{
  switch (type_) {
    case types::user_location_info_eutra:
      c.destroy<user_location_info_eutra_s>();
      break;
    case types::user_location_info_nr:
      c.destroy<user_location_info_nr_s>();
      break;
    case types::user_location_info_n3_iwf:
      c.destroy<user_location_info_n3_iwf_s>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<user_location_info_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void user_location_info_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::user_location_info_eutra:
      c.init<user_location_info_eutra_s>();
      break;
    case types::user_location_info_nr:
      c.init<user_location_info_nr_s>();
      break;
    case types::user_location_info_n3_iwf:
      c.init<user_location_info_n3_iwf_s>();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<user_location_info_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "user_location_info_c");
  }
}
user_location_info_c::user_location_info_c(const user_location_info_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::user_location_info_eutra:
      c.init(other.c.get<user_location_info_eutra_s>());
      break;
    case types::user_location_info_nr:
      c.init(other.c.get<user_location_info_nr_s>());
      break;
    case types::user_location_info_n3_iwf:
      c.init(other.c.get<user_location_info_n3_iwf_s>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<user_location_info_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "user_location_info_c");
  }
}
user_location_info_c& user_location_info_c::operator=(const user_location_info_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::user_location_info_eutra:
      c.set(other.c.get<user_location_info_eutra_s>());
      break;
    case types::user_location_info_nr:
      c.set(other.c.get<user_location_info_nr_s>());
      break;
    case types::user_location_info_n3_iwf:
      c.set(other.c.get<user_location_info_n3_iwf_s>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<user_location_info_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "user_location_info_c");
  }

  return *this;
}
user_location_info_eutra_s& user_location_info_c::set_user_location_info_eutra()
{
  set(types::user_location_info_eutra);
  return c.get<user_location_info_eutra_s>();
}
user_location_info_nr_s& user_location_info_c::set_user_location_info_nr()
{
  set(types::user_location_info_nr);
  return c.get<user_location_info_nr_s>();
}
user_location_info_n3_iwf_s& user_location_info_c::set_user_location_info_n3_iwf()
{
  set(types::user_location_info_n3_iwf);
  return c.get<user_location_info_n3_iwf_s>();
}
protocol_ie_single_container_s<user_location_info_ext_ies_o>& user_location_info_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<user_location_info_ext_ies_o> >();
}
void user_location_info_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::user_location_info_eutra:
      j.write_fieldname("userLocationInformationEUTRA");
      c.get<user_location_info_eutra_s>().to_json(j);
      break;
    case types::user_location_info_nr:
      j.write_fieldname("userLocationInformationNR");
      c.get<user_location_info_nr_s>().to_json(j);
      break;
    case types::user_location_info_n3_iwf:
      j.write_fieldname("userLocationInformationN3IWF");
      c.get<user_location_info_n3_iwf_s>().to_json(j);
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<user_location_info_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "user_location_info_c");
  }
  j.end_obj();
}
SRSASN_CODE user_location_info_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::user_location_info_eutra:
      HANDLE_CODE(c.get<user_location_info_eutra_s>().pack(bref));
      break;
    case types::user_location_info_nr:
      HANDLE_CODE(c.get<user_location_info_nr_s>().pack(bref));
      break;
    case types::user_location_info_n3_iwf:
      HANDLE_CODE(c.get<user_location_info_n3_iwf_s>().pack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<user_location_info_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "user_location_info_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE user_location_info_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::user_location_info_eutra:
      HANDLE_CODE(c.get<user_location_info_eutra_s>().unpack(bref));
      break;
    case types::user_location_info_nr:
      HANDLE_CODE(c.get<user_location_info_nr_s>().unpack(bref));
      break;
    case types::user_location_info_n3_iwf:
      HANDLE_CODE(c.get<user_location_info_n3_iwf_s>().unpack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<user_location_info_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "user_location_info_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* user_location_info_c::types_opts::to_string() const
{
  static const char* options[] = {
      "userLocationInformationEUTRA", "userLocationInformationNR", "userLocationInformationN3IWF", "choice-Extensions"};
  return convert_enum_idx(options, 4, value, "user_location_info_c::types");
}
uint8_t user_location_info_c::types_opts::to_number() const
{
  if (value == user_location_info_n3_iwf) {
    return 3;
  }
  invalid_enum_number(value, "user_location_info_c::types");
  return 0;
}

// HandoverNotifyIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ho_notify_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 121};
  return map_enum_number(options, 3, idx, "id");
}
bool ho_notify_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 121};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_notify_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 121:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_notify_ies_o::value_c ho_notify_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 121:
      ret.set(value_c::types::user_location_info);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_notify_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 121:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_notify_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::user_location_info:
      c.destroy<user_location_info_c>();
      break;
    default:
      break;
  }
}
void ho_notify_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::user_location_info:
      c.init<user_location_info_c>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
  }
}
ho_notify_ies_o::value_c::value_c(const ho_notify_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::user_location_info:
      c.init(other.c.get<user_location_info_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
  }
}
ho_notify_ies_o::value_c& ho_notify_ies_o::value_c::operator=(const ho_notify_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::user_location_info:
      c.set(other.c.get<user_location_info_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_notify_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& ho_notify_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
user_location_info_c& ho_notify_ies_o::value_c::user_location_info()
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_c>();
}
const uint64_t& ho_notify_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& ho_notify_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const user_location_info_c& ho_notify_ies_o::value_c::user_location_info() const
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_c>();
}
void ho_notify_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::user_location_info:
      j.write_fieldname("UserLocationInformation");
      c.get<user_location_info_c>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_notify_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_c>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_notify_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_c>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_notify_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_notify_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)", "INTEGER (0..4294967295)", "UserLocationInformation"};
  return convert_enum_idx(options, 3, value, "ho_notify_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_notify_ies_o>;

ho_notify_ies_container::ho_notify_ies_container() :
  amf_ue_ngap_id(10, crit_e::reject), ran_ue_ngap_id(85, crit_e::reject), user_location_info(121, crit_e::ignore)
{}
SRSASN_CODE ho_notify_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  HANDLE_CODE(user_location_info.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_notify_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 85: {
        nof_mandatory_ies--;
        ran_ue_ngap_id.id = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 121: {
        nof_mandatory_ies--;
        user_location_info.id = id;
        HANDLE_CODE(user_location_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(user_location_info.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_notify_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ran_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  user_location_info.to_json(j);
  j.end_obj();
}

// HandoverPreparationFailureIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ho_prep_fail_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 15, 19};
  return map_enum_number(options, 4, idx, "id");
}
bool ho_prep_fail_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 15, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_prep_fail_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::ignore;
    case 85:
      return crit_e::ignore;
    case 15:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_prep_fail_ies_o::value_c ho_prep_fail_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 15:
      ret.set(value_c::types::cause);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_prep_fail_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 15:
      return presence_e::mandatory;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_prep_fail_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ho_prep_fail_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
  }
}
ho_prep_fail_ies_o::value_c::value_c(const ho_prep_fail_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
  }
}
ho_prep_fail_ies_o::value_c& ho_prep_fail_ies_o::value_c::operator=(const ho_prep_fail_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_prep_fail_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& ho_prep_fail_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
cause_c& ho_prep_fail_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
crit_diagnostics_s& ho_prep_fail_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& ho_prep_fail_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& ho_prep_fail_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const cause_c& ho_prep_fail_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const crit_diagnostics_s& ho_prep_fail_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void ho_prep_fail_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_prep_fail_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_prep_fail_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_prep_fail_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_prep_fail_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..1099511627775)", "INTEGER (0..4294967295)", "Cause", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 4, value, "ho_prep_fail_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_prep_fail_ies_o>;

ho_prep_fail_ies_container::ho_prep_fail_ies_container() :
  amf_ue_ngap_id(10, crit_e::ignore),
  ran_ue_ngap_id(85, crit_e::ignore),
  cause(15, crit_e::ignore),
  crit_diagnostics(19, crit_e::ignore)
{}
SRSASN_CODE ho_prep_fail_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  HANDLE_CODE(cause.pack(bref));
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_prep_fail_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 85: {
        nof_mandatory_ies--;
        ran_ue_ngap_id.id = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 15: {
        nof_mandatory_ies--;
        cause.id = id;
        HANDLE_CODE(cause.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cause.value.unpack(bref));
        break;
      }
      case 19: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_prep_fail_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ran_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  cause.to_json(j);
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// HandoverPreparationUnsuccessfulTransfer ::= SEQUENCE
SRSASN_CODE ho_prep_unsuccessful_transfer_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(cause.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_prep_unsuccessful_transfer_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(cause.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ho_prep_unsuccessful_transfer_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("cause");
  cause.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// EventType ::= ENUMERATED
const char* event_type_opts::to_string() const
{
  static const char* options[] = {"direct",
                                  "change-of-serve-cell",
                                  "ue-presence-in-area-of-interest",
                                  "stop-change-of-serve-cell",
                                  "stop-ue-presence-in-area-of-interest",
                                  "cancel-location-reporting-for-the-ue"};
  return convert_enum_idx(options, 6, value, "event_type_e");
}

// PDUSessionResourceSetupItemHOReq ::= SEQUENCE
SRSASN_CODE pdu_session_res_setup_item_ho_req_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(s_nssai.pack(bref));
  HANDLE_CODE(ho_request_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_setup_item_ho_req_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(s_nssai.unpack(bref));
  HANDLE_CODE(ho_request_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_setup_item_ho_req_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_fieldname("s-NSSAI");
  s_nssai.to_json(j);
  j.write_str("handoverRequestTransfer", ho_request_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ReportArea ::= ENUMERATED
const char* report_area_opts::to_string() const
{
  static const char* options[] = {"cell"};
  return convert_enum_idx(options, 1, value, "report_area_e");
}

// TraceDepth ::= ENUMERATED
const char* trace_depth_opts::to_string() const
{
  static const char* options[] = {"minimum",
                                  "medium",
                                  "maximum",
                                  "minimumWithoutVendorSpecificExtension",
                                  "mediumWithoutVendorSpecificExtension",
                                  "maximumWithoutVendorSpecificExtension"};
  return convert_enum_idx(options, 6, value, "trace_depth_e");
}

// LocationReportingRequestType ::= SEQUENCE
SRSASN_CODE location_report_request_type_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(area_of_interest_list.size() > 0, 1));
  HANDLE_CODE(bref.pack(location_report_ref_id_to_be_cancelled_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(event_type.pack(bref));
  HANDLE_CODE(report_area.pack(bref));
  if (area_of_interest_list.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, area_of_interest_list, 1, 64, true));
  }
  if (location_report_ref_id_to_be_cancelled_present) {
    HANDLE_CODE(pack_integer(bref, location_report_ref_id_to_be_cancelled, (uint8_t)1u, (uint8_t)64u, true, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE location_report_request_type_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  bool area_of_interest_list_present;
  HANDLE_CODE(bref.unpack(area_of_interest_list_present, 1));
  HANDLE_CODE(bref.unpack(location_report_ref_id_to_be_cancelled_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(event_type.unpack(bref));
  HANDLE_CODE(report_area.unpack(bref));
  if (area_of_interest_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(area_of_interest_list, bref, 1, 64, true));
  }
  if (location_report_ref_id_to_be_cancelled_present) {
    HANDLE_CODE(unpack_integer(location_report_ref_id_to_be_cancelled, bref, (uint8_t)1u, (uint8_t)64u, true, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void location_report_request_type_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("eventType", event_type.to_string());
  j.write_str("reportArea", "cell");
  if (area_of_interest_list.size() > 0) {
    j.start_array("areaOfInterestList");
    for (const auto& e1 : area_of_interest_list) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (location_report_ref_id_to_be_cancelled_present) {
    j.write_int("locationReportingReferenceIDToBeCancelled", location_report_ref_id_to_be_cancelled);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// NewSecurityContextInd ::= ENUMERATED
const char* new_security_context_ind_opts::to_string() const
{
  static const char* options[] = {"true"};
  return convert_enum_idx(options, 1, value, "new_security_context_ind_e");
}

// RRCInactiveTransitionReportRequest ::= ENUMERATED
const char* rrc_inactive_transition_report_request_opts::to_string() const
{
  static const char* options[] = {
      "subsequent-state-transition-report", "single-rrc-connected-state-report", "cancel-report"};
  return convert_enum_idx(options, 3, value, "rrc_inactive_transition_report_request_e");
}

// RedirectionVoiceFallback ::= ENUMERATED
const char* redirection_voice_fallback_opts::to_string() const
{
  static const char* options[] = {"possible", "not-possible"};
  return convert_enum_idx(options, 2, value, "redirection_voice_fallback_e");
}

// SecurityContext ::= SEQUENCE
SRSASN_CODE security_context_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, next_hop_chaining_count, (uint8_t)0u, (uint8_t)7u, false, true));
  HANDLE_CODE(next_hop_nh.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE security_context_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(next_hop_chaining_count, bref, (uint8_t)0u, (uint8_t)7u, false, true));
  HANDLE_CODE(next_hop_nh.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void security_context_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("nextHopChainingCount", next_hop_chaining_count);
  j.write_str("nextHopNH", next_hop_nh.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TraceActivation ::= SEQUENCE
SRSASN_CODE trace_activation_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(ngran_trace_id.pack(bref));
  HANDLE_CODE(interfaces_to_trace.pack(bref));
  HANDLE_CODE(trace_depth.pack(bref));
  HANDLE_CODE(trace_collection_entity_ip_address.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE trace_activation_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(ngran_trace_id.unpack(bref));
  HANDLE_CODE(interfaces_to_trace.unpack(bref));
  HANDLE_CODE(trace_depth.unpack(bref));
  HANDLE_CODE(trace_collection_entity_ip_address.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void trace_activation_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("nGRANTraceID", ngran_trace_id.to_string());
  j.write_str("interfacesToTrace", interfaces_to_trace.to_string());
  j.write_str("traceDepth", trace_depth.to_string());
  j.write_str("traceCollectionEntityIPAddress", trace_collection_entity_ip_address.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UESecurityCapabilities ::= SEQUENCE
SRSASN_CODE ue_security_cap_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(nrencryption_algorithms.pack(bref));
  HANDLE_CODE(nrintegrity_protection_algorithms.pack(bref));
  HANDLE_CODE(eutr_aencryption_algorithms.pack(bref));
  HANDLE_CODE(eutr_aintegrity_protection_algorithms.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_security_cap_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(nrencryption_algorithms.unpack(bref));
  HANDLE_CODE(nrintegrity_protection_algorithms.unpack(bref));
  HANDLE_CODE(eutr_aencryption_algorithms.unpack(bref));
  HANDLE_CODE(eutr_aintegrity_protection_algorithms.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ue_security_cap_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("nRencryptionAlgorithms", nrencryption_algorithms.to_string());
  j.write_str("nRintegrityProtectionAlgorithms", nrintegrity_protection_algorithms.to_string());
  j.write_str("eUTRAencryptionAlgorithms", eutr_aencryption_algorithms.to_string());
  j.write_str("eUTRAintegrityProtectionAlgorithms", eutr_aintegrity_protection_algorithms.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// HandoverRequestIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ho_request_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 29, 15, 110, 18, 119, 93, 41, 37, 73, 0, 108, 34, 101, 36, 33, 91, 28, 146};
  return map_enum_number(options, 19, idx, "id");
}
bool ho_request_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 29, 15, 110, 18, 119, 93, 41, 37, 73, 0, 108, 34, 101, 36, 33, 91, 28, 146};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_request_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 29:
      return crit_e::reject;
    case 15:
      return crit_e::ignore;
    case 110:
      return crit_e::reject;
    case 18:
      return crit_e::ignore;
    case 119:
      return crit_e::reject;
    case 93:
      return crit_e::reject;
    case 41:
      return crit_e::reject;
    case 37:
      return crit_e::reject;
    case 73:
      return crit_e::reject;
    case 0:
      return crit_e::reject;
    case 108:
      return crit_e::ignore;
    case 34:
      return crit_e::ignore;
    case 101:
      return crit_e::reject;
    case 36:
      return crit_e::ignore;
    case 33:
      return crit_e::ignore;
    case 91:
      return crit_e::ignore;
    case 28:
      return crit_e::reject;
    case 146:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_request_ies_o::value_c ho_request_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 29:
      ret.set(value_c::types::handov_type);
      break;
    case 15:
      ret.set(value_c::types::cause);
      break;
    case 110:
      ret.set(value_c::types::ue_aggregate_maximum_bit_rate);
      break;
    case 18:
      ret.set(value_c::types::core_network_assist_info);
      break;
    case 119:
      ret.set(value_c::types::ue_security_cap);
      break;
    case 93:
      ret.set(value_c::types::security_context);
      break;
    case 41:
      ret.set(value_c::types::new_security_context_ind);
      break;
    case 37:
      ret.set(value_c::types::nasc);
      break;
    case 73:
      ret.set(value_c::types::pdu_session_res_setup_list_ho_req);
      break;
    case 0:
      ret.set(value_c::types::allowed_nssai);
      break;
    case 108:
      ret.set(value_c::types::trace_activation);
      break;
    case 34:
      ret.set(value_c::types::masked_imeisv);
      break;
    case 101:
      ret.set(value_c::types::source_to_target_transparent_container);
      break;
    case 36:
      ret.set(value_c::types::mob_restrict_list);
      break;
    case 33:
      ret.set(value_c::types::location_report_request_type);
      break;
    case 91:
      ret.set(value_c::types::rrc_inactive_transition_report_request);
      break;
    case 28:
      ret.set(value_c::types::guami);
      break;
    case 146:
      ret.set(value_c::types::redirection_voice_fallback);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_request_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 29:
      return presence_e::mandatory;
    case 15:
      return presence_e::mandatory;
    case 110:
      return presence_e::mandatory;
    case 18:
      return presence_e::optional;
    case 119:
      return presence_e::mandatory;
    case 93:
      return presence_e::mandatory;
    case 41:
      return presence_e::optional;
    case 37:
      return presence_e::optional;
    case 73:
      return presence_e::mandatory;
    case 0:
      return presence_e::mandatory;
    case 108:
      return presence_e::optional;
    case 34:
      return presence_e::optional;
    case 101:
      return presence_e::mandatory;
    case 36:
      return presence_e::optional;
    case 33:
      return presence_e::optional;
    case 91:
      return presence_e::optional;
    case 28:
      return presence_e::mandatory;
    case 146:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_request_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::ue_aggregate_maximum_bit_rate:
      c.destroy<ue_aggregate_maximum_bit_rate_s>();
      break;
    case types::core_network_assist_info:
      c.destroy<core_network_assist_info_s>();
      break;
    case types::ue_security_cap:
      c.destroy<ue_security_cap_s>();
      break;
    case types::security_context:
      c.destroy<security_context_s>();
      break;
    case types::nasc:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::pdu_session_res_setup_list_ho_req:
      c.destroy<pdu_session_res_setup_list_ho_req_l>();
      break;
    case types::allowed_nssai:
      c.destroy<allowed_nssai_l>();
      break;
    case types::trace_activation:
      c.destroy<trace_activation_s>();
      break;
    case types::masked_imeisv:
      c.destroy<fixed_bitstring<64, false, true> >();
      break;
    case types::source_to_target_transparent_container:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::mob_restrict_list:
      c.destroy<mob_restrict_list_s>();
      break;
    case types::location_report_request_type:
      c.destroy<location_report_request_type_s>();
      break;
    case types::guami:
      c.destroy<guami_s>();
      break;
    default:
      break;
  }
}
void ho_request_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::handov_type:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::ue_aggregate_maximum_bit_rate:
      c.init<ue_aggregate_maximum_bit_rate_s>();
      break;
    case types::core_network_assist_info:
      c.init<core_network_assist_info_s>();
      break;
    case types::ue_security_cap:
      c.init<ue_security_cap_s>();
      break;
    case types::security_context:
      c.init<security_context_s>();
      break;
    case types::new_security_context_ind:
      break;
    case types::nasc:
      c.init<unbounded_octstring<true> >();
      break;
    case types::pdu_session_res_setup_list_ho_req:
      c.init<pdu_session_res_setup_list_ho_req_l>();
      break;
    case types::allowed_nssai:
      c.init<allowed_nssai_l>();
      break;
    case types::trace_activation:
      c.init<trace_activation_s>();
      break;
    case types::masked_imeisv:
      c.init<fixed_bitstring<64, false, true> >();
      break;
    case types::source_to_target_transparent_container:
      c.init<unbounded_octstring<true> >();
      break;
    case types::mob_restrict_list:
      c.init<mob_restrict_list_s>();
      break;
    case types::location_report_request_type:
      c.init<location_report_request_type_s>();
      break;
    case types::rrc_inactive_transition_report_request:
      break;
    case types::guami:
      c.init<guami_s>();
      break;
    case types::redirection_voice_fallback:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
  }
}
ho_request_ies_o::value_c::value_c(const ho_request_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::handov_type:
      c.init(other.c.get<handov_type_e>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::ue_aggregate_maximum_bit_rate:
      c.init(other.c.get<ue_aggregate_maximum_bit_rate_s>());
      break;
    case types::core_network_assist_info:
      c.init(other.c.get<core_network_assist_info_s>());
      break;
    case types::ue_security_cap:
      c.init(other.c.get<ue_security_cap_s>());
      break;
    case types::security_context:
      c.init(other.c.get<security_context_s>());
      break;
    case types::new_security_context_ind:
      c.init(other.c.get<new_security_context_ind_e>());
      break;
    case types::nasc:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::pdu_session_res_setup_list_ho_req:
      c.init(other.c.get<pdu_session_res_setup_list_ho_req_l>());
      break;
    case types::allowed_nssai:
      c.init(other.c.get<allowed_nssai_l>());
      break;
    case types::trace_activation:
      c.init(other.c.get<trace_activation_s>());
      break;
    case types::masked_imeisv:
      c.init(other.c.get<fixed_bitstring<64, false, true> >());
      break;
    case types::source_to_target_transparent_container:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::mob_restrict_list:
      c.init(other.c.get<mob_restrict_list_s>());
      break;
    case types::location_report_request_type:
      c.init(other.c.get<location_report_request_type_s>());
      break;
    case types::rrc_inactive_transition_report_request:
      c.init(other.c.get<rrc_inactive_transition_report_request_e>());
      break;
    case types::guami:
      c.init(other.c.get<guami_s>());
      break;
    case types::redirection_voice_fallback:
      c.init(other.c.get<redirection_voice_fallback_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
  }
}
ho_request_ies_o::value_c& ho_request_ies_o::value_c::operator=(const ho_request_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::handov_type:
      c.set(other.c.get<handov_type_e>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::ue_aggregate_maximum_bit_rate:
      c.set(other.c.get<ue_aggregate_maximum_bit_rate_s>());
      break;
    case types::core_network_assist_info:
      c.set(other.c.get<core_network_assist_info_s>());
      break;
    case types::ue_security_cap:
      c.set(other.c.get<ue_security_cap_s>());
      break;
    case types::security_context:
      c.set(other.c.get<security_context_s>());
      break;
    case types::new_security_context_ind:
      c.set(other.c.get<new_security_context_ind_e>());
      break;
    case types::nasc:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::pdu_session_res_setup_list_ho_req:
      c.set(other.c.get<pdu_session_res_setup_list_ho_req_l>());
      break;
    case types::allowed_nssai:
      c.set(other.c.get<allowed_nssai_l>());
      break;
    case types::trace_activation:
      c.set(other.c.get<trace_activation_s>());
      break;
    case types::masked_imeisv:
      c.set(other.c.get<fixed_bitstring<64, false, true> >());
      break;
    case types::source_to_target_transparent_container:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::mob_restrict_list:
      c.set(other.c.get<mob_restrict_list_s>());
      break;
    case types::location_report_request_type:
      c.set(other.c.get<location_report_request_type_s>());
      break;
    case types::rrc_inactive_transition_report_request:
      c.set(other.c.get<rrc_inactive_transition_report_request_e>());
      break;
    case types::guami:
      c.set(other.c.get<guami_s>());
      break;
    case types::redirection_voice_fallback:
      c.set(other.c.get<redirection_voice_fallback_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_request_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
handov_type_e& ho_request_ies_o::value_c::handov_type()
{
  assert_choice_type(types::handov_type, type_, "Value");
  return c.get<handov_type_e>();
}
cause_c& ho_request_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
ue_aggregate_maximum_bit_rate_s& ho_request_ies_o::value_c::ue_aggregate_maximum_bit_rate()
{
  assert_choice_type(types::ue_aggregate_maximum_bit_rate, type_, "Value");
  return c.get<ue_aggregate_maximum_bit_rate_s>();
}
core_network_assist_info_s& ho_request_ies_o::value_c::core_network_assist_info()
{
  assert_choice_type(types::core_network_assist_info, type_, "Value");
  return c.get<core_network_assist_info_s>();
}
ue_security_cap_s& ho_request_ies_o::value_c::ue_security_cap()
{
  assert_choice_type(types::ue_security_cap, type_, "Value");
  return c.get<ue_security_cap_s>();
}
security_context_s& ho_request_ies_o::value_c::security_context()
{
  assert_choice_type(types::security_context, type_, "Value");
  return c.get<security_context_s>();
}
new_security_context_ind_e& ho_request_ies_o::value_c::new_security_context_ind()
{
  assert_choice_type(types::new_security_context_ind, type_, "Value");
  return c.get<new_security_context_ind_e>();
}
unbounded_octstring<true>& ho_request_ies_o::value_c::nasc()
{
  assert_choice_type(types::nasc, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
pdu_session_res_setup_list_ho_req_l& ho_request_ies_o::value_c::pdu_session_res_setup_list_ho_req()
{
  assert_choice_type(types::pdu_session_res_setup_list_ho_req, type_, "Value");
  return c.get<pdu_session_res_setup_list_ho_req_l>();
}
allowed_nssai_l& ho_request_ies_o::value_c::allowed_nssai()
{
  assert_choice_type(types::allowed_nssai, type_, "Value");
  return c.get<allowed_nssai_l>();
}
trace_activation_s& ho_request_ies_o::value_c::trace_activation()
{
  assert_choice_type(types::trace_activation, type_, "Value");
  return c.get<trace_activation_s>();
}
fixed_bitstring<64, false, true>& ho_request_ies_o::value_c::masked_imeisv()
{
  assert_choice_type(types::masked_imeisv, type_, "Value");
  return c.get<fixed_bitstring<64, false, true> >();
}
unbounded_octstring<true>& ho_request_ies_o::value_c::source_to_target_transparent_container()
{
  assert_choice_type(types::source_to_target_transparent_container, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
mob_restrict_list_s& ho_request_ies_o::value_c::mob_restrict_list()
{
  assert_choice_type(types::mob_restrict_list, type_, "Value");
  return c.get<mob_restrict_list_s>();
}
location_report_request_type_s& ho_request_ies_o::value_c::location_report_request_type()
{
  assert_choice_type(types::location_report_request_type, type_, "Value");
  return c.get<location_report_request_type_s>();
}
rrc_inactive_transition_report_request_e& ho_request_ies_o::value_c::rrc_inactive_transition_report_request()
{
  assert_choice_type(types::rrc_inactive_transition_report_request, type_, "Value");
  return c.get<rrc_inactive_transition_report_request_e>();
}
guami_s& ho_request_ies_o::value_c::guami()
{
  assert_choice_type(types::guami, type_, "Value");
  return c.get<guami_s>();
}
redirection_voice_fallback_e& ho_request_ies_o::value_c::redirection_voice_fallback()
{
  assert_choice_type(types::redirection_voice_fallback, type_, "Value");
  return c.get<redirection_voice_fallback_e>();
}
const uint64_t& ho_request_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const handov_type_e& ho_request_ies_o::value_c::handov_type() const
{
  assert_choice_type(types::handov_type, type_, "Value");
  return c.get<handov_type_e>();
}
const cause_c& ho_request_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const ue_aggregate_maximum_bit_rate_s& ho_request_ies_o::value_c::ue_aggregate_maximum_bit_rate() const
{
  assert_choice_type(types::ue_aggregate_maximum_bit_rate, type_, "Value");
  return c.get<ue_aggregate_maximum_bit_rate_s>();
}
const core_network_assist_info_s& ho_request_ies_o::value_c::core_network_assist_info() const
{
  assert_choice_type(types::core_network_assist_info, type_, "Value");
  return c.get<core_network_assist_info_s>();
}
const ue_security_cap_s& ho_request_ies_o::value_c::ue_security_cap() const
{
  assert_choice_type(types::ue_security_cap, type_, "Value");
  return c.get<ue_security_cap_s>();
}
const security_context_s& ho_request_ies_o::value_c::security_context() const
{
  assert_choice_type(types::security_context, type_, "Value");
  return c.get<security_context_s>();
}
const new_security_context_ind_e& ho_request_ies_o::value_c::new_security_context_ind() const
{
  assert_choice_type(types::new_security_context_ind, type_, "Value");
  return c.get<new_security_context_ind_e>();
}
const unbounded_octstring<true>& ho_request_ies_o::value_c::nasc() const
{
  assert_choice_type(types::nasc, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const pdu_session_res_setup_list_ho_req_l& ho_request_ies_o::value_c::pdu_session_res_setup_list_ho_req() const
{
  assert_choice_type(types::pdu_session_res_setup_list_ho_req, type_, "Value");
  return c.get<pdu_session_res_setup_list_ho_req_l>();
}
const allowed_nssai_l& ho_request_ies_o::value_c::allowed_nssai() const
{
  assert_choice_type(types::allowed_nssai, type_, "Value");
  return c.get<allowed_nssai_l>();
}
const trace_activation_s& ho_request_ies_o::value_c::trace_activation() const
{
  assert_choice_type(types::trace_activation, type_, "Value");
  return c.get<trace_activation_s>();
}
const fixed_bitstring<64, false, true>& ho_request_ies_o::value_c::masked_imeisv() const
{
  assert_choice_type(types::masked_imeisv, type_, "Value");
  return c.get<fixed_bitstring<64, false, true> >();
}
const unbounded_octstring<true>& ho_request_ies_o::value_c::source_to_target_transparent_container() const
{
  assert_choice_type(types::source_to_target_transparent_container, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const mob_restrict_list_s& ho_request_ies_o::value_c::mob_restrict_list() const
{
  assert_choice_type(types::mob_restrict_list, type_, "Value");
  return c.get<mob_restrict_list_s>();
}
const location_report_request_type_s& ho_request_ies_o::value_c::location_report_request_type() const
{
  assert_choice_type(types::location_report_request_type, type_, "Value");
  return c.get<location_report_request_type_s>();
}
const rrc_inactive_transition_report_request_e&
ho_request_ies_o::value_c::rrc_inactive_transition_report_request() const
{
  assert_choice_type(types::rrc_inactive_transition_report_request, type_, "Value");
  return c.get<rrc_inactive_transition_report_request_e>();
}
const guami_s& ho_request_ies_o::value_c::guami() const
{
  assert_choice_type(types::guami, type_, "Value");
  return c.get<guami_s>();
}
const redirection_voice_fallback_e& ho_request_ies_o::value_c::redirection_voice_fallback() const
{
  assert_choice_type(types::redirection_voice_fallback, type_, "Value");
  return c.get<redirection_voice_fallback_e>();
}
void ho_request_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::handov_type:
      j.write_str("HandoverType", c.get<handov_type_e>().to_string());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::ue_aggregate_maximum_bit_rate:
      j.write_fieldname("UEAggregateMaximumBitRate");
      c.get<ue_aggregate_maximum_bit_rate_s>().to_json(j);
      break;
    case types::core_network_assist_info:
      j.write_fieldname("CoreNetworkAssistanceInformation");
      c.get<core_network_assist_info_s>().to_json(j);
      break;
    case types::ue_security_cap:
      j.write_fieldname("UESecurityCapabilities");
      c.get<ue_security_cap_s>().to_json(j);
      break;
    case types::security_context:
      j.write_fieldname("SecurityContext");
      c.get<security_context_s>().to_json(j);
      break;
    case types::new_security_context_ind:
      j.write_str("NewSecurityContextInd", "true");
      break;
    case types::nasc:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::pdu_session_res_setup_list_ho_req:
      j.start_array("PDUSessionResourceSetupListHOReq");
      for (const auto& e1 : c.get<pdu_session_res_setup_list_ho_req_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::allowed_nssai:
      j.start_array("AllowedNSSAI");
      for (const auto& e1 : c.get<allowed_nssai_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::trace_activation:
      j.write_fieldname("TraceActivation");
      c.get<trace_activation_s>().to_json(j);
      break;
    case types::masked_imeisv:
      j.write_str("BIT STRING", c.get<fixed_bitstring<64, false, true> >().to_string());
      break;
    case types::source_to_target_transparent_container:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::mob_restrict_list:
      j.write_fieldname("MobilityRestrictionList");
      c.get<mob_restrict_list_s>().to_json(j);
      break;
    case types::location_report_request_type:
      j.write_fieldname("LocationReportingRequestType");
      c.get<location_report_request_type_s>().to_json(j);
      break;
    case types::rrc_inactive_transition_report_request:
      j.write_str("RRCInactiveTransitionReportRequest", c.get<rrc_inactive_transition_report_request_e>().to_string());
      break;
    case types::guami:
      j.write_fieldname("GUAMI");
      c.get<guami_s>().to_json(j);
      break;
    case types::redirection_voice_fallback:
      j.write_str("RedirectionVoiceFallback", c.get<redirection_voice_fallback_e>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_request_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().pack(bref));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::ue_aggregate_maximum_bit_rate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bit_rate_s>().pack(bref));
      break;
    case types::core_network_assist_info:
      HANDLE_CODE(c.get<core_network_assist_info_s>().pack(bref));
      break;
    case types::ue_security_cap:
      HANDLE_CODE(c.get<ue_security_cap_s>().pack(bref));
      break;
    case types::security_context:
      HANDLE_CODE(c.get<security_context_s>().pack(bref));
      break;
    case types::new_security_context_ind:
      HANDLE_CODE(c.get<new_security_context_ind_e>().pack(bref));
      break;
    case types::nasc:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::pdu_session_res_setup_list_ho_req:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_setup_list_ho_req_l>(), 1, 256, true));
      break;
    case types::allowed_nssai:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<allowed_nssai_l>(), 1, 8, true));
      break;
    case types::trace_activation:
      HANDLE_CODE(c.get<trace_activation_s>().pack(bref));
      break;
    case types::masked_imeisv:
      HANDLE_CODE((c.get<fixed_bitstring<64, false, true> >().pack(bref)));
      break;
    case types::source_to_target_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::mob_restrict_list:
      HANDLE_CODE(c.get<mob_restrict_list_s>().pack(bref));
      break;
    case types::location_report_request_type:
      HANDLE_CODE(c.get<location_report_request_type_s>().pack(bref));
      break;
    case types::rrc_inactive_transition_report_request:
      HANDLE_CODE(c.get<rrc_inactive_transition_report_request_e>().pack(bref));
      break;
    case types::guami:
      HANDLE_CODE(c.get<guami_s>().pack(bref));
      break;
    case types::redirection_voice_fallback:
      HANDLE_CODE(c.get<redirection_voice_fallback_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_request_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().unpack(bref));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::ue_aggregate_maximum_bit_rate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bit_rate_s>().unpack(bref));
      break;
    case types::core_network_assist_info:
      HANDLE_CODE(c.get<core_network_assist_info_s>().unpack(bref));
      break;
    case types::ue_security_cap:
      HANDLE_CODE(c.get<ue_security_cap_s>().unpack(bref));
      break;
    case types::security_context:
      HANDLE_CODE(c.get<security_context_s>().unpack(bref));
      break;
    case types::new_security_context_ind:
      HANDLE_CODE(c.get<new_security_context_ind_e>().unpack(bref));
      break;
    case types::nasc:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::pdu_session_res_setup_list_ho_req:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_setup_list_ho_req_l>(), bref, 1, 256, true));
      break;
    case types::allowed_nssai:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<allowed_nssai_l>(), bref, 1, 8, true));
      break;
    case types::trace_activation:
      HANDLE_CODE(c.get<trace_activation_s>().unpack(bref));
      break;
    case types::masked_imeisv:
      HANDLE_CODE((c.get<fixed_bitstring<64, false, true> >().unpack(bref)));
      break;
    case types::source_to_target_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::mob_restrict_list:
      HANDLE_CODE(c.get<mob_restrict_list_s>().unpack(bref));
      break;
    case types::location_report_request_type:
      HANDLE_CODE(c.get<location_report_request_type_s>().unpack(bref));
      break;
    case types::rrc_inactive_transition_report_request:
      HANDLE_CODE(c.get<rrc_inactive_transition_report_request_e>().unpack(bref));
      break;
    case types::guami:
      HANDLE_CODE(c.get<guami_s>().unpack(bref));
      break;
    case types::redirection_voice_fallback:
      HANDLE_CODE(c.get<redirection_voice_fallback_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_request_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "HandoverType",
                                  "Cause",
                                  "UEAggregateMaximumBitRate",
                                  "CoreNetworkAssistanceInformation",
                                  "UESecurityCapabilities",
                                  "SecurityContext",
                                  "NewSecurityContextInd",
                                  "OCTET STRING",
                                  "PDUSessionResourceSetupListHOReq",
                                  "AllowedNSSAI",
                                  "TraceActivation",
                                  "BIT STRING",
                                  "OCTET STRING",
                                  "MobilityRestrictionList",
                                  "LocationReportingRequestType",
                                  "RRCInactiveTransitionReportRequest",
                                  "GUAMI",
                                  "RedirectionVoiceFallback"};
  return convert_enum_idx(options, 19, value, "ho_request_ies_o::value_c::types");
}
uint8_t ho_request_ies_o::value_c::types_opts::to_number() const
{
  switch (value) {
    case amf_ue_ngap_id:
      return 0;
    case core_network_assist_info:
      return 2;
    default:
      invalid_enum_number(value, "ho_request_ies_o::value_c::types");
  }
  return 0;
}

template struct asn1::protocol_ie_field_s<ho_request_ies_o>;

ho_request_ies_container::ho_request_ies_container() :
  amf_ue_ngap_id(10, crit_e::reject),
  handov_type(29, crit_e::reject),
  cause(15, crit_e::ignore),
  ue_aggregate_maximum_bit_rate(110, crit_e::reject),
  core_network_assist_info(18, crit_e::ignore),
  ue_security_cap(119, crit_e::reject),
  security_context(93, crit_e::reject),
  new_security_context_ind(41, crit_e::reject),
  nasc(37, crit_e::reject),
  pdu_session_res_setup_list_ho_req(73, crit_e::reject),
  allowed_nssai(0, crit_e::reject),
  trace_activation(108, crit_e::ignore),
  masked_imeisv(34, crit_e::ignore),
  source_to_target_transparent_container(101, crit_e::reject),
  mob_restrict_list(36, crit_e::ignore),
  location_report_request_type(33, crit_e::ignore),
  rrc_inactive_transition_report_request(91, crit_e::ignore),
  guami(28, crit_e::reject),
  redirection_voice_fallback(146, crit_e::ignore)
{}
SRSASN_CODE ho_request_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 10;
  nof_ies += core_network_assist_info_present ? 1 : 0;
  nof_ies += new_security_context_ind_present ? 1 : 0;
  nof_ies += nasc_present ? 1 : 0;
  nof_ies += trace_activation_present ? 1 : 0;
  nof_ies += masked_imeisv_present ? 1 : 0;
  nof_ies += mob_restrict_list_present ? 1 : 0;
  nof_ies += location_report_request_type_present ? 1 : 0;
  nof_ies += rrc_inactive_transition_report_request_present ? 1 : 0;
  nof_ies += redirection_voice_fallback_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(handov_type.pack(bref));
  HANDLE_CODE(cause.pack(bref));
  HANDLE_CODE(ue_aggregate_maximum_bit_rate.pack(bref));
  if (core_network_assist_info_present) {
    HANDLE_CODE(core_network_assist_info.pack(bref));
  }
  HANDLE_CODE(ue_security_cap.pack(bref));
  HANDLE_CODE(security_context.pack(bref));
  if (new_security_context_ind_present) {
    HANDLE_CODE(new_security_context_ind.pack(bref));
  }
  if (nasc_present) {
    HANDLE_CODE(nasc.pack(bref));
  }
  HANDLE_CODE(pdu_session_res_setup_list_ho_req.pack(bref));
  HANDLE_CODE(allowed_nssai.pack(bref));
  if (trace_activation_present) {
    HANDLE_CODE(trace_activation.pack(bref));
  }
  if (masked_imeisv_present) {
    HANDLE_CODE(masked_imeisv.pack(bref));
  }
  HANDLE_CODE(source_to_target_transparent_container.pack(bref));
  if (mob_restrict_list_present) {
    HANDLE_CODE(mob_restrict_list.pack(bref));
  }
  if (location_report_request_type_present) {
    HANDLE_CODE(location_report_request_type.pack(bref));
  }
  if (rrc_inactive_transition_report_request_present) {
    HANDLE_CODE(rrc_inactive_transition_report_request.pack(bref));
  }
  HANDLE_CODE(guami.pack(bref));
  if (redirection_voice_fallback_present) {
    HANDLE_CODE(redirection_voice_fallback.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_request_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 10;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 29: {
        nof_mandatory_ies--;
        handov_type.id = id;
        HANDLE_CODE(handov_type.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(handov_type.value.unpack(bref));
        break;
      }
      case 15: {
        nof_mandatory_ies--;
        cause.id = id;
        HANDLE_CODE(cause.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cause.value.unpack(bref));
        break;
      }
      case 110: {
        nof_mandatory_ies--;
        ue_aggregate_maximum_bit_rate.id = id;
        HANDLE_CODE(ue_aggregate_maximum_bit_rate.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ue_aggregate_maximum_bit_rate.value.unpack(bref));
        break;
      }
      case 18: {
        core_network_assist_info_present = true;
        core_network_assist_info.id      = id;
        HANDLE_CODE(core_network_assist_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(core_network_assist_info.value.unpack(bref));
        break;
      }
      case 119: {
        nof_mandatory_ies--;
        ue_security_cap.id = id;
        HANDLE_CODE(ue_security_cap.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ue_security_cap.value.unpack(bref));
        break;
      }
      case 93: {
        nof_mandatory_ies--;
        security_context.id = id;
        HANDLE_CODE(security_context.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(security_context.value.unpack(bref));
        break;
      }
      case 41: {
        new_security_context_ind_present = true;
        new_security_context_ind.id      = id;
        HANDLE_CODE(new_security_context_ind.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(new_security_context_ind.value.unpack(bref));
        break;
      }
      case 37: {
        nasc_present = true;
        nasc.id      = id;
        HANDLE_CODE(nasc.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(nasc.value.unpack(bref));
        break;
      }
      case 73: {
        nof_mandatory_ies--;
        pdu_session_res_setup_list_ho_req.id = id;
        HANDLE_CODE(pdu_session_res_setup_list_ho_req.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(pdu_session_res_setup_list_ho_req.value.unpack(bref));
        break;
      }
      case 0: {
        nof_mandatory_ies--;
        allowed_nssai.id = id;
        HANDLE_CODE(allowed_nssai.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(allowed_nssai.value.unpack(bref));
        break;
      }
      case 108: {
        trace_activation_present = true;
        trace_activation.id      = id;
        HANDLE_CODE(trace_activation.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(trace_activation.value.unpack(bref));
        break;
      }
      case 34: {
        masked_imeisv_present = true;
        masked_imeisv.id      = id;
        HANDLE_CODE(masked_imeisv.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(masked_imeisv.value.unpack(bref));
        break;
      }
      case 101: {
        nof_mandatory_ies--;
        source_to_target_transparent_container.id = id;
        HANDLE_CODE(source_to_target_transparent_container.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(source_to_target_transparent_container.value.unpack(bref));
        break;
      }
      case 36: {
        mob_restrict_list_present = true;
        mob_restrict_list.id      = id;
        HANDLE_CODE(mob_restrict_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mob_restrict_list.value.unpack(bref));
        break;
      }
      case 33: {
        location_report_request_type_present = true;
        location_report_request_type.id      = id;
        HANDLE_CODE(location_report_request_type.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(location_report_request_type.value.unpack(bref));
        break;
      }
      case 91: {
        rrc_inactive_transition_report_request_present = true;
        rrc_inactive_transition_report_request.id      = id;
        HANDLE_CODE(rrc_inactive_transition_report_request.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(rrc_inactive_transition_report_request.value.unpack(bref));
        break;
      }
      case 28: {
        nof_mandatory_ies--;
        guami.id = id;
        HANDLE_CODE(guami.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(guami.value.unpack(bref));
        break;
      }
      case 146: {
        redirection_voice_fallback_present = true;
        redirection_voice_fallback.id      = id;
        HANDLE_CODE(redirection_voice_fallback.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(redirection_voice_fallback.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_request_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  handov_type.to_json(j);
  j.write_fieldname("");
  cause.to_json(j);
  j.write_fieldname("");
  ue_aggregate_maximum_bit_rate.to_json(j);
  if (core_network_assist_info_present) {
    j.write_fieldname("");
    core_network_assist_info.to_json(j);
  }
  j.write_fieldname("");
  ue_security_cap.to_json(j);
  j.write_fieldname("");
  security_context.to_json(j);
  if (new_security_context_ind_present) {
    j.write_fieldname("");
    new_security_context_ind.to_json(j);
  }
  if (nasc_present) {
    j.write_fieldname("");
    nasc.to_json(j);
  }
  j.write_fieldname("");
  pdu_session_res_setup_list_ho_req.to_json(j);
  j.write_fieldname("");
  allowed_nssai.to_json(j);
  if (trace_activation_present) {
    j.write_fieldname("");
    trace_activation.to_json(j);
  }
  if (masked_imeisv_present) {
    j.write_fieldname("");
    masked_imeisv.to_json(j);
  }
  j.write_fieldname("");
  source_to_target_transparent_container.to_json(j);
  if (mob_restrict_list_present) {
    j.write_fieldname("");
    mob_restrict_list.to_json(j);
  }
  if (location_report_request_type_present) {
    j.write_fieldname("");
    location_report_request_type.to_json(j);
  }
  if (rrc_inactive_transition_report_request_present) {
    j.write_fieldname("");
    rrc_inactive_transition_report_request.to_json(j);
  }
  j.write_fieldname("");
  guami.to_json(j);
  if (redirection_voice_fallback_present) {
    j.write_fieldname("");
    redirection_voice_fallback.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceAdmittedItem ::= SEQUENCE
SRSASN_CODE pdu_session_res_admitted_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(ho_request_ack_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_admitted_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(ho_request_ack_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_admitted_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("handoverRequestAcknowledgeTransfer", ho_request_ack_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceFailedToSetupItemHOAck ::= SEQUENCE
SRSASN_CODE pdu_session_res_failed_to_setup_item_ho_ack_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(ho_res_alloc_unsuccessful_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_failed_to_setup_item_ho_ack_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(ho_res_alloc_unsuccessful_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_failed_to_setup_item_ho_ack_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("handoverResourceAllocationUnsuccessfulTransfer", ho_res_alloc_unsuccessful_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// HandoverRequestAcknowledgeIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ho_request_ack_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 53, 56, 106, 19};
  return map_enum_number(options, 6, idx, "id");
}
bool ho_request_ack_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 53, 56, 106, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_request_ack_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::ignore;
    case 85:
      return crit_e::ignore;
    case 53:
      return crit_e::ignore;
    case 56:
      return crit_e::ignore;
    case 106:
      return crit_e::reject;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_request_ack_ies_o::value_c ho_request_ack_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 53:
      ret.set(value_c::types::pdu_session_res_admitted_list);
      break;
    case 56:
      ret.set(value_c::types::pdu_session_res_failed_to_setup_list_ho_ack);
      break;
    case 106:
      ret.set(value_c::types::target_to_source_transparent_container);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_request_ack_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 53:
      return presence_e::mandatory;
    case 56:
      return presence_e::optional;
    case 106:
      return presence_e::mandatory;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_request_ack_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::pdu_session_res_admitted_list:
      c.destroy<pdu_session_res_admitted_list_l>();
      break;
    case types::pdu_session_res_failed_to_setup_list_ho_ack:
      c.destroy<pdu_session_res_failed_to_setup_list_ho_ack_l>();
      break;
    case types::target_to_source_transparent_container:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ho_request_ack_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::pdu_session_res_admitted_list:
      c.init<pdu_session_res_admitted_list_l>();
      break;
    case types::pdu_session_res_failed_to_setup_list_ho_ack:
      c.init<pdu_session_res_failed_to_setup_list_ho_ack_l>();
      break;
    case types::target_to_source_transparent_container:
      c.init<unbounded_octstring<true> >();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
  }
}
ho_request_ack_ies_o::value_c::value_c(const ho_request_ack_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_admitted_list:
      c.init(other.c.get<pdu_session_res_admitted_list_l>());
      break;
    case types::pdu_session_res_failed_to_setup_list_ho_ack:
      c.init(other.c.get<pdu_session_res_failed_to_setup_list_ho_ack_l>());
      break;
    case types::target_to_source_transparent_container:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
  }
}
ho_request_ack_ies_o::value_c& ho_request_ack_ies_o::value_c::operator=(const ho_request_ack_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_admitted_list:
      c.set(other.c.get<pdu_session_res_admitted_list_l>());
      break;
    case types::pdu_session_res_failed_to_setup_list_ho_ack:
      c.set(other.c.get<pdu_session_res_failed_to_setup_list_ho_ack_l>());
      break;
    case types::target_to_source_transparent_container:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_request_ack_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& ho_request_ack_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
pdu_session_res_admitted_list_l& ho_request_ack_ies_o::value_c::pdu_session_res_admitted_list()
{
  assert_choice_type(types::pdu_session_res_admitted_list, type_, "Value");
  return c.get<pdu_session_res_admitted_list_l>();
}
pdu_session_res_failed_to_setup_list_ho_ack_l&
ho_request_ack_ies_o::value_c::pdu_session_res_failed_to_setup_list_ho_ack()
{
  assert_choice_type(types::pdu_session_res_failed_to_setup_list_ho_ack, type_, "Value");
  return c.get<pdu_session_res_failed_to_setup_list_ho_ack_l>();
}
unbounded_octstring<true>& ho_request_ack_ies_o::value_c::target_to_source_transparent_container()
{
  assert_choice_type(types::target_to_source_transparent_container, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
crit_diagnostics_s& ho_request_ack_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& ho_request_ack_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& ho_request_ack_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const pdu_session_res_admitted_list_l& ho_request_ack_ies_o::value_c::pdu_session_res_admitted_list() const
{
  assert_choice_type(types::pdu_session_res_admitted_list, type_, "Value");
  return c.get<pdu_session_res_admitted_list_l>();
}
const pdu_session_res_failed_to_setup_list_ho_ack_l&
ho_request_ack_ies_o::value_c::pdu_session_res_failed_to_setup_list_ho_ack() const
{
  assert_choice_type(types::pdu_session_res_failed_to_setup_list_ho_ack, type_, "Value");
  return c.get<pdu_session_res_failed_to_setup_list_ho_ack_l>();
}
const unbounded_octstring<true>& ho_request_ack_ies_o::value_c::target_to_source_transparent_container() const
{
  assert_choice_type(types::target_to_source_transparent_container, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const crit_diagnostics_s& ho_request_ack_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void ho_request_ack_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::pdu_session_res_admitted_list:
      j.start_array("PDUSessionResourceAdmittedList");
      for (const auto& e1 : c.get<pdu_session_res_admitted_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::pdu_session_res_failed_to_setup_list_ho_ack:
      j.start_array("PDUSessionResourceFailedToSetupListHOAck");
      for (const auto& e1 : c.get<pdu_session_res_failed_to_setup_list_ho_ack_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::target_to_source_transparent_container:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_request_ack_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_admitted_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_admitted_list_l>(), 1, 256, true));
      break;
    case types::pdu_session_res_failed_to_setup_list_ho_ack:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_failed_to_setup_list_ho_ack_l>(), 1, 256, true));
      break;
    case types::target_to_source_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_request_ack_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_admitted_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_admitted_list_l>(), bref, 1, 256, true));
      break;
    case types::pdu_session_res_failed_to_setup_list_ho_ack:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_failed_to_setup_list_ho_ack_l>(), bref, 1, 256, true));
      break;
    case types::target_to_source_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_request_ack_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_request_ack_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "PDUSessionResourceAdmittedList",
                                  "PDUSessionResourceFailedToSetupListHOAck",
                                  "OCTET STRING",
                                  "CriticalityDiagnostics"};
  return convert_enum_idx(options, 6, value, "ho_request_ack_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_request_ack_ies_o>;

ho_request_ack_ies_container::ho_request_ack_ies_container() :
  amf_ue_ngap_id(10, crit_e::ignore),
  ran_ue_ngap_id(85, crit_e::ignore),
  pdu_session_res_admitted_list(53, crit_e::ignore),
  pdu_session_res_failed_to_setup_list_ho_ack(56, crit_e::ignore),
  target_to_source_transparent_container(106, crit_e::reject),
  crit_diagnostics(19, crit_e::ignore)
{}
SRSASN_CODE ho_request_ack_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 4;
  nof_ies += pdu_session_res_failed_to_setup_list_ho_ack_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  HANDLE_CODE(pdu_session_res_admitted_list.pack(bref));
  if (pdu_session_res_failed_to_setup_list_ho_ack_present) {
    HANDLE_CODE(pdu_session_res_failed_to_setup_list_ho_ack.pack(bref));
  }
  HANDLE_CODE(target_to_source_transparent_container.pack(bref));
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_request_ack_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 4;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 85: {
        nof_mandatory_ies--;
        ran_ue_ngap_id.id = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 53: {
        nof_mandatory_ies--;
        pdu_session_res_admitted_list.id = id;
        HANDLE_CODE(pdu_session_res_admitted_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(pdu_session_res_admitted_list.value.unpack(bref));
        break;
      }
      case 56: {
        pdu_session_res_failed_to_setup_list_ho_ack_present = true;
        pdu_session_res_failed_to_setup_list_ho_ack.id      = id;
        HANDLE_CODE(pdu_session_res_failed_to_setup_list_ho_ack.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(pdu_session_res_failed_to_setup_list_ho_ack.value.unpack(bref));
        break;
      }
      case 106: {
        nof_mandatory_ies--;
        target_to_source_transparent_container.id = id;
        HANDLE_CODE(target_to_source_transparent_container.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(target_to_source_transparent_container.value.unpack(bref));
        break;
      }
      case 19: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_request_ack_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ran_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  pdu_session_res_admitted_list.to_json(j);
  if (pdu_session_res_failed_to_setup_list_ho_ack_present) {
    j.write_fieldname("");
    pdu_session_res_failed_to_setup_list_ho_ack.to_json(j);
  }
  j.write_fieldname("");
  target_to_source_transparent_container.to_json(j);
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// ConfidentialityProtectionResult ::= ENUMERATED
const char* confidentiality_protection_result_opts::to_string() const
{
  static const char* options[] = {"performed", "not-performed"};
  return convert_enum_idx(options, 2, value, "confidentiality_protection_result_e");
}

// IntegrityProtectionResult ::= ENUMERATED
const char* integrity_protection_result_opts::to_string() const
{
  static const char* options[] = {"performed", "not-performed"};
  return convert_enum_idx(options, 2, value, "integrity_protection_result_e");
}

// QosFlowWithCauseItem ::= SEQUENCE
SRSASN_CODE qos_flow_with_cause_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, qos_flow_id, (uint8_t)0u, (uint8_t)63u, true, true));
  HANDLE_CODE(cause.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE qos_flow_with_cause_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(qos_flow_id, bref, (uint8_t)0u, (uint8_t)63u, true, true));
  HANDLE_CODE(cause.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void qos_flow_with_cause_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("qosFlowIdentifier", qos_flow_id);
  j.write_fieldname("cause");
  cause.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// HandoverRequestAcknowledgeTransfer-ExtIEs ::= OBJECT SET OF NGAP-PROTOCOL-EXTENSION
uint32_t ho_request_ack_transfer_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {153};
  return map_enum_number(options, 1, idx, "id");
}
bool ho_request_ack_transfer_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 153 == id;
}
crit_e ho_request_ack_transfer_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 153) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
ho_request_ack_transfer_ext_ies_o::ext_c ho_request_ack_transfer_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 153) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_request_ack_transfer_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 153) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void ho_request_ack_transfer_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("AdditionalDLUPTNLInformationForHOList");
  for (const auto& e1 : c) {
    e1.to_json(j);
  }
  j.end_array();
  j.end_obj();
}
SRSASN_CODE ho_request_ack_transfer_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(pack_dyn_seq_of(bref, c, 1, 3, true));
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_request_ack_transfer_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(unpack_dyn_seq_of(c, bref, 1, 3, true));
  return SRSASN_SUCCESS;
}

const char* ho_request_ack_transfer_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"AdditionalDLUPTNLInformationForHOList"};
  return convert_enum_idx(options, 1, value, "ho_request_ack_transfer_ext_ies_o::ext_c::types");
}

// SecurityResult ::= SEQUENCE
SRSASN_CODE security_result_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(integrity_protection_result.pack(bref));
  HANDLE_CODE(confidentiality_protection_result.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE security_result_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(integrity_protection_result.unpack(bref));
  HANDLE_CODE(confidentiality_protection_result.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void security_result_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("integrityProtectionResult", integrity_protection_result.to_string());
  j.write_str("confidentialityProtectionResult", confidentiality_protection_result.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// HandoverRequestAcknowledgeTransfer ::= SEQUENCE
SRSASN_CODE ho_request_ack_transfer_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(dlforwarding_up_tnl_info_present, 1));
  HANDLE_CODE(bref.pack(security_result_present, 1));
  HANDLE_CODE(bref.pack(qos_flow_failed_to_setup_list.size() > 0, 1));
  HANDLE_CODE(bref.pack(data_forwarding_resp_drb_list.size() > 0, 1));
  HANDLE_CODE(bref.pack(ie_exts.size() > 0, 1));

  HANDLE_CODE(dl_ngu_up_tnl_info.pack(bref));
  if (dlforwarding_up_tnl_info_present) {
    HANDLE_CODE(dlforwarding_up_tnl_info.pack(bref));
  }
  if (security_result_present) {
    HANDLE_CODE(security_result.pack(bref));
  }
  HANDLE_CODE(pack_dyn_seq_of(bref, qos_flow_setup_resp_list, 1, 64, true));
  if (qos_flow_failed_to_setup_list.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, qos_flow_failed_to_setup_list, 1, 64, true));
  }
  if (data_forwarding_resp_drb_list.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, data_forwarding_resp_drb_list, 1, 32, true));
  }
  if (ie_exts.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_request_ack_transfer_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(dlforwarding_up_tnl_info_present, 1));
  HANDLE_CODE(bref.unpack(security_result_present, 1));
  bool qos_flow_failed_to_setup_list_present;
  HANDLE_CODE(bref.unpack(qos_flow_failed_to_setup_list_present, 1));
  bool data_forwarding_resp_drb_list_present;
  HANDLE_CODE(bref.unpack(data_forwarding_resp_drb_list_present, 1));
  bool ie_exts_present;
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(dl_ngu_up_tnl_info.unpack(bref));
  if (dlforwarding_up_tnl_info_present) {
    HANDLE_CODE(dlforwarding_up_tnl_info.unpack(bref));
  }
  if (security_result_present) {
    HANDLE_CODE(security_result.unpack(bref));
  }
  HANDLE_CODE(unpack_dyn_seq_of(qos_flow_setup_resp_list, bref, 1, 64, true));
  if (qos_flow_failed_to_setup_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(qos_flow_failed_to_setup_list, bref, 1, 64, true));
  }
  if (data_forwarding_resp_drb_list_present) {
    HANDLE_CODE(unpack_dyn_seq_of(data_forwarding_resp_drb_list, bref, 1, 32, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void ho_request_ack_transfer_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("dL-NGU-UP-TNLInformation");
  dl_ngu_up_tnl_info.to_json(j);
  if (dlforwarding_up_tnl_info_present) {
    j.write_fieldname("dLForwardingUP-TNLInformation");
    dlforwarding_up_tnl_info.to_json(j);
  }
  if (security_result_present) {
    j.write_fieldname("securityResult");
    security_result.to_json(j);
  }
  j.start_array("qosFlowSetupResponseList");
  for (const auto& e1 : qos_flow_setup_resp_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (qos_flow_failed_to_setup_list.size() > 0) {
    j.start_array("qosFlowFailedToSetupList");
    for (const auto& e1 : qos_flow_failed_to_setup_list) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (data_forwarding_resp_drb_list.size() > 0) {
    j.start_array("dataForwardingResponseDRBList");
    for (const auto& e1 : data_forwarding_resp_drb_list) {
      e1.to_json(j);
    }
    j.end_array();
  }
  if (ie_exts.size() > 0) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// PDUSessionResourceItemHORqd ::= SEQUENCE
SRSASN_CODE pdu_session_res_item_ho_rqd_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(ho_required_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_item_ho_rqd_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(ho_required_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_item_ho_rqd_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("handoverRequiredTransfer", ho_required_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TargeteNB-ID ::= SEQUENCE
SRSASN_CODE targetenb_id_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(global_enb_id.pack(bref));
  HANDLE_CODE(sel_eps_tai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE targetenb_id_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(global_enb_id.unpack(bref));
  HANDLE_CODE(sel_eps_tai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void targetenb_id_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("globalENB-ID");
  global_enb_id.to_json(j);
  j.write_fieldname("selected-EPS-TAI");
  sel_eps_tai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// DirectForwardingPathAvailability ::= ENUMERATED
const char* direct_forwarding_path_availability_opts::to_string() const
{
  static const char* options[] = {"direct-path-available"};
  return convert_enum_idx(options, 1, value, "direct_forwarding_path_availability_e");
}

// TargetID ::= CHOICE
void target_id_c::destroy_()
{
  switch (type_) {
    case types::target_ran_node_id:
      c.destroy<target_ran_node_id_s>();
      break;
    case types::targetenb_id:
      c.destroy<targetenb_id_s>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<target_id_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void target_id_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::target_ran_node_id:
      c.init<target_ran_node_id_s>();
      break;
    case types::targetenb_id:
      c.init<targetenb_id_s>();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<target_id_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
  }
}
target_id_c::target_id_c(const target_id_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::target_ran_node_id:
      c.init(other.c.get<target_ran_node_id_s>());
      break;
    case types::targetenb_id:
      c.init(other.c.get<targetenb_id_s>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<target_id_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
  }
}
target_id_c& target_id_c::operator=(const target_id_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::target_ran_node_id:
      c.set(other.c.get<target_ran_node_id_s>());
      break;
    case types::targetenb_id:
      c.set(other.c.get<targetenb_id_s>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<target_id_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
  }

  return *this;
}
target_ran_node_id_s& target_id_c::set_target_ran_node_id()
{
  set(types::target_ran_node_id);
  return c.get<target_ran_node_id_s>();
}
targetenb_id_s& target_id_c::set_targetenb_id()
{
  set(types::targetenb_id);
  return c.get<targetenb_id_s>();
}
protocol_ie_single_container_s<target_id_ext_ies_o>& target_id_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<target_id_ext_ies_o> >();
}
void target_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::target_ran_node_id:
      j.write_fieldname("targetRANNodeID");
      c.get<target_ran_node_id_s>().to_json(j);
      break;
    case types::targetenb_id:
      j.write_fieldname("targeteNB-ID");
      c.get<targetenb_id_s>().to_json(j);
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<target_id_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
  }
  j.end_obj();
}
SRSASN_CODE target_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::target_ran_node_id:
      HANDLE_CODE(c.get<target_ran_node_id_s>().pack(bref));
      break;
    case types::targetenb_id:
      HANDLE_CODE(c.get<targetenb_id_s>().pack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<target_id_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE target_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::target_ran_node_id:
      HANDLE_CODE(c.get<target_ran_node_id_s>().unpack(bref));
      break;
    case types::targetenb_id:
      HANDLE_CODE(c.get<targetenb_id_s>().unpack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<target_id_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "target_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* target_id_c::types_opts::to_string() const
{
  static const char* options[] = {"targetRANNodeID", "targeteNB-ID", "choice-Extensions"};
  return convert_enum_idx(options, 3, value, "target_id_c::types");
}

// HandoverRequiredIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ho_required_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 29, 15, 105, 22, 61, 101};
  return map_enum_number(options, 8, idx, "id");
}
bool ho_required_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 29, 15, 105, 22, 61, 101};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ho_required_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 29:
      return crit_e::reject;
    case 15:
      return crit_e::ignore;
    case 105:
      return crit_e::reject;
    case 22:
      return crit_e::ignore;
    case 61:
      return crit_e::reject;
    case 101:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ho_required_ies_o::value_c ho_required_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 29:
      ret.set(value_c::types::handov_type);
      break;
    case 15:
      ret.set(value_c::types::cause);
      break;
    case 105:
      ret.set(value_c::types::target_id);
      break;
    case 22:
      ret.set(value_c::types::direct_forwarding_path_availability);
      break;
    case 61:
      ret.set(value_c::types::pdu_session_res_list_ho_rqd);
      break;
    case 101:
      ret.set(value_c::types::source_to_target_transparent_container);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ho_required_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 29:
      return presence_e::mandatory;
    case 15:
      return presence_e::mandatory;
    case 105:
      return presence_e::mandatory;
    case 22:
      return presence_e::optional;
    case 61:
      return presence_e::mandatory;
    case 101:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ho_required_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::target_id:
      c.destroy<target_id_c>();
      break;
    case types::pdu_session_res_list_ho_rqd:
      c.destroy<pdu_session_res_list_ho_rqd_l>();
      break;
    case types::source_to_target_transparent_container:
      c.destroy<unbounded_octstring<true> >();
      break;
    default:
      break;
  }
}
void ho_required_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::handov_type:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::target_id:
      c.init<target_id_c>();
      break;
    case types::direct_forwarding_path_availability:
      break;
    case types::pdu_session_res_list_ho_rqd:
      c.init<pdu_session_res_list_ho_rqd_l>();
      break;
    case types::source_to_target_transparent_container:
      c.init<unbounded_octstring<true> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
  }
}
ho_required_ies_o::value_c::value_c(const ho_required_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::handov_type:
      c.init(other.c.get<handov_type_e>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::target_id:
      c.init(other.c.get<target_id_c>());
      break;
    case types::direct_forwarding_path_availability:
      c.init(other.c.get<direct_forwarding_path_availability_e>());
      break;
    case types::pdu_session_res_list_ho_rqd:
      c.init(other.c.get<pdu_session_res_list_ho_rqd_l>());
      break;
    case types::source_to_target_transparent_container:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
  }
}
ho_required_ies_o::value_c& ho_required_ies_o::value_c::operator=(const ho_required_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::handov_type:
      c.set(other.c.get<handov_type_e>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::target_id:
      c.set(other.c.get<target_id_c>());
      break;
    case types::direct_forwarding_path_availability:
      c.set(other.c.get<direct_forwarding_path_availability_e>());
      break;
    case types::pdu_session_res_list_ho_rqd:
      c.set(other.c.get<pdu_session_res_list_ho_rqd_l>());
      break;
    case types::source_to_target_transparent_container:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
  }

  return *this;
}
uint64_t& ho_required_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& ho_required_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
handov_type_e& ho_required_ies_o::value_c::handov_type()
{
  assert_choice_type(types::handov_type, type_, "Value");
  return c.get<handov_type_e>();
}
cause_c& ho_required_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
target_id_c& ho_required_ies_o::value_c::target_id()
{
  assert_choice_type(types::target_id, type_, "Value");
  return c.get<target_id_c>();
}
direct_forwarding_path_availability_e& ho_required_ies_o::value_c::direct_forwarding_path_availability()
{
  assert_choice_type(types::direct_forwarding_path_availability, type_, "Value");
  return c.get<direct_forwarding_path_availability_e>();
}
pdu_session_res_list_ho_rqd_l& ho_required_ies_o::value_c::pdu_session_res_list_ho_rqd()
{
  assert_choice_type(types::pdu_session_res_list_ho_rqd, type_, "Value");
  return c.get<pdu_session_res_list_ho_rqd_l>();
}
unbounded_octstring<true>& ho_required_ies_o::value_c::source_to_target_transparent_container()
{
  assert_choice_type(types::source_to_target_transparent_container, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const uint64_t& ho_required_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& ho_required_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const handov_type_e& ho_required_ies_o::value_c::handov_type() const
{
  assert_choice_type(types::handov_type, type_, "Value");
  return c.get<handov_type_e>();
}
const cause_c& ho_required_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const target_id_c& ho_required_ies_o::value_c::target_id() const
{
  assert_choice_type(types::target_id, type_, "Value");
  return c.get<target_id_c>();
}
const direct_forwarding_path_availability_e& ho_required_ies_o::value_c::direct_forwarding_path_availability() const
{
  assert_choice_type(types::direct_forwarding_path_availability, type_, "Value");
  return c.get<direct_forwarding_path_availability_e>();
}
const pdu_session_res_list_ho_rqd_l& ho_required_ies_o::value_c::pdu_session_res_list_ho_rqd() const
{
  assert_choice_type(types::pdu_session_res_list_ho_rqd, type_, "Value");
  return c.get<pdu_session_res_list_ho_rqd_l>();
}
const unbounded_octstring<true>& ho_required_ies_o::value_c::source_to_target_transparent_container() const
{
  assert_choice_type(types::source_to_target_transparent_container, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
void ho_required_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::handov_type:
      j.write_str("HandoverType", c.get<handov_type_e>().to_string());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::target_id:
      j.write_fieldname("TargetID");
      c.get<target_id_c>().to_json(j);
      break;
    case types::direct_forwarding_path_availability:
      j.write_str("DirectForwardingPathAvailability", "direct-path-available");
      break;
    case types::pdu_session_res_list_ho_rqd:
      j.start_array("PDUSessionResourceListHORqd");
      for (const auto& e1 : c.get<pdu_session_res_list_ho_rqd_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::source_to_target_transparent_container:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ho_required_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().pack(bref));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::target_id:
      HANDLE_CODE(c.get<target_id_c>().pack(bref));
      break;
    case types::direct_forwarding_path_availability:
      HANDLE_CODE(c.get<direct_forwarding_path_availability_e>().pack(bref));
      break;
    case types::pdu_session_res_list_ho_rqd:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_list_ho_rqd_l>(), 1, 256, true));
      break;
    case types::source_to_target_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_required_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::handov_type:
      HANDLE_CODE(c.get<handov_type_e>().unpack(bref));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::target_id:
      HANDLE_CODE(c.get<target_id_c>().unpack(bref));
      break;
    case types::direct_forwarding_path_availability:
      HANDLE_CODE(c.get<direct_forwarding_path_availability_e>().unpack(bref));
      break;
    case types::pdu_session_res_list_ho_rqd:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_list_ho_rqd_l>(), bref, 1, 256, true));
      break;
    case types::source_to_target_transparent_container:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ho_required_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ho_required_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "HandoverType",
                                  "Cause",
                                  "TargetID",
                                  "DirectForwardingPathAvailability",
                                  "PDUSessionResourceListHORqd",
                                  "OCTET STRING"};
  return convert_enum_idx(options, 8, value, "ho_required_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<ho_required_ies_o>;

ho_required_ies_container::ho_required_ies_container() :
  amf_ue_ngap_id(10, crit_e::reject),
  ran_ue_ngap_id(85, crit_e::reject),
  handov_type(29, crit_e::reject),
  cause(15, crit_e::ignore),
  target_id(105, crit_e::reject),
  direct_forwarding_path_availability(22, crit_e::ignore),
  pdu_session_res_list_ho_rqd(61, crit_e::reject),
  source_to_target_transparent_container(101, crit_e::reject)
{}
SRSASN_CODE ho_required_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 7;
  nof_ies += direct_forwarding_path_availability_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  HANDLE_CODE(handov_type.pack(bref));
  HANDLE_CODE(cause.pack(bref));
  HANDLE_CODE(target_id.pack(bref));
  if (direct_forwarding_path_availability_present) {
    HANDLE_CODE(direct_forwarding_path_availability.pack(bref));
  }
  HANDLE_CODE(pdu_session_res_list_ho_rqd.pack(bref));
  HANDLE_CODE(source_to_target_transparent_container.pack(bref));

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_required_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 7;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 85: {
        nof_mandatory_ies--;
        ran_ue_ngap_id.id = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 29: {
        nof_mandatory_ies--;
        handov_type.id = id;
        HANDLE_CODE(handov_type.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(handov_type.value.unpack(bref));
        break;
      }
      case 15: {
        nof_mandatory_ies--;
        cause.id = id;
        HANDLE_CODE(cause.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cause.value.unpack(bref));
        break;
      }
      case 105: {
        nof_mandatory_ies--;
        target_id.id = id;
        HANDLE_CODE(target_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(target_id.value.unpack(bref));
        break;
      }
      case 22: {
        direct_forwarding_path_availability_present = true;
        direct_forwarding_path_availability.id      = id;
        HANDLE_CODE(direct_forwarding_path_availability.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(direct_forwarding_path_availability.value.unpack(bref));
        break;
      }
      case 61: {
        nof_mandatory_ies--;
        pdu_session_res_list_ho_rqd.id = id;
        HANDLE_CODE(pdu_session_res_list_ho_rqd.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(pdu_session_res_list_ho_rqd.value.unpack(bref));
        break;
      }
      case 101: {
        nof_mandatory_ies--;
        source_to_target_transparent_container.id = id;
        HANDLE_CODE(source_to_target_transparent_container.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(source_to_target_transparent_container.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void ho_required_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ran_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  handov_type.to_json(j);
  j.write_fieldname("");
  cause.to_json(j);
  j.write_fieldname("");
  target_id.to_json(j);
  if (direct_forwarding_path_availability_present) {
    j.write_fieldname("");
    direct_forwarding_path_availability.to_json(j);
  }
  j.write_fieldname("");
  pdu_session_res_list_ho_rqd.to_json(j);
  j.write_fieldname("");
  source_to_target_transparent_container.to_json(j);
  j.end_obj();
}

// HandoverRequiredTransfer ::= SEQUENCE
SRSASN_CODE ho_required_transfer_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(direct_forwarding_path_availability_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (direct_forwarding_path_availability_present) {
    HANDLE_CODE(direct_forwarding_path_availability.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_required_transfer_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(direct_forwarding_path_availability_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (direct_forwarding_path_availability_present) {
    HANDLE_CODE(direct_forwarding_path_availability.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ho_required_transfer_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (direct_forwarding_path_availability_present) {
    j.write_str("directForwardingPathAvailability", "direct-path-available");
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// HandoverResourceAllocationUnsuccessfulTransfer ::= SEQUENCE
SRSASN_CODE ho_res_alloc_unsuccessful_transfer_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(crit_diagnostics_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(cause.pack(bref));
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ho_res_alloc_unsuccessful_transfer_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(crit_diagnostics_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(cause.unpack(bref));
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ho_res_alloc_unsuccessful_transfer_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("cause");
  cause.to_json(j);
  if (crit_diagnostics_present) {
    j.write_fieldname("criticalityDiagnostics");
    crit_diagnostics.to_json(j);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// RecommendedRANNodeItem ::= SEQUENCE
SRSASN_CODE recommended_ran_node_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(amf_paging_target.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE recommended_ran_node_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(amf_paging_target.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void recommended_ran_node_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("aMFPagingTarget");
  amf_paging_target.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// RecommendedRANNodesForPaging ::= SEQUENCE
SRSASN_CODE recommended_ran_nodes_for_paging_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, recommended_ran_node_list, 1, 16, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE recommended_ran_nodes_for_paging_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(recommended_ran_node_list, bref, 1, 16, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void recommended_ran_nodes_for_paging_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("recommendedRANNodeList");
  for (const auto& e1 : recommended_ran_node_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// InfoOnRecommendedCellsAndRANNodesForPaging ::= SEQUENCE
SRSASN_CODE info_on_recommended_cells_and_ran_nodes_for_paging_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(recommended_cells_for_paging.pack(bref));
  HANDLE_CODE(recommend_ran_nodes_for_paging.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE info_on_recommended_cells_and_ran_nodes_for_paging_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(recommended_cells_for_paging.unpack(bref));
  HANDLE_CODE(recommend_ran_nodes_for_paging.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void info_on_recommended_cells_and_ran_nodes_for_paging_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("recommendedCellsForPaging");
  recommended_cells_for_paging.to_json(j);
  j.write_fieldname("recommendRANNodesForPaging");
  recommend_ran_nodes_for_paging.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceFailedToSetupItemCxtFail ::= SEQUENCE
SRSASN_CODE pdu_session_res_failed_to_setup_item_cxt_fail_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_setup_unsuccessful_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_failed_to_setup_item_cxt_fail_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_setup_unsuccessful_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_failed_to_setup_item_cxt_fail_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pDUSessionResourceSetupUnsuccessfulTransfer", pdu_session_res_setup_unsuccessful_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// InitialContextSetupFailureIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t init_context_setup_fail_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 132, 15, 19};
  return map_enum_number(options, 5, idx, "id");
}
bool init_context_setup_fail_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 132, 15, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e init_context_setup_fail_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::ignore;
    case 85:
      return crit_e::ignore;
    case 132:
      return crit_e::ignore;
    case 15:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
init_context_setup_fail_ies_o::value_c init_context_setup_fail_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 132:
      ret.set(value_c::types::pdu_session_res_failed_to_setup_list_cxt_fail);
      break;
    case 15:
      ret.set(value_c::types::cause);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e init_context_setup_fail_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 132:
      return presence_e::optional;
    case 15:
      return presence_e::mandatory;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void init_context_setup_fail_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::pdu_session_res_failed_to_setup_list_cxt_fail:
      c.destroy<pdu_session_res_failed_to_setup_list_cxt_fail_l>();
      break;
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void init_context_setup_fail_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::pdu_session_res_failed_to_setup_list_cxt_fail:
      c.init<pdu_session_res_failed_to_setup_list_cxt_fail_l>();
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_fail_ies_o::value_c");
  }
}
init_context_setup_fail_ies_o::value_c::value_c(const init_context_setup_fail_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_failed_to_setup_list_cxt_fail:
      c.init(other.c.get<pdu_session_res_failed_to_setup_list_cxt_fail_l>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_fail_ies_o::value_c");
  }
}
init_context_setup_fail_ies_o::value_c&
init_context_setup_fail_ies_o::value_c::operator=(const init_context_setup_fail_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_failed_to_setup_list_cxt_fail:
      c.set(other.c.get<pdu_session_res_failed_to_setup_list_cxt_fail_l>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_fail_ies_o::value_c");
  }

  return *this;
}
uint64_t& init_context_setup_fail_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& init_context_setup_fail_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
pdu_session_res_failed_to_setup_list_cxt_fail_l&
init_context_setup_fail_ies_o::value_c::pdu_session_res_failed_to_setup_list_cxt_fail()
{
  assert_choice_type(types::pdu_session_res_failed_to_setup_list_cxt_fail, type_, "Value");
  return c.get<pdu_session_res_failed_to_setup_list_cxt_fail_l>();
}
cause_c& init_context_setup_fail_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
crit_diagnostics_s& init_context_setup_fail_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& init_context_setup_fail_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& init_context_setup_fail_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const pdu_session_res_failed_to_setup_list_cxt_fail_l&
init_context_setup_fail_ies_o::value_c::pdu_session_res_failed_to_setup_list_cxt_fail() const
{
  assert_choice_type(types::pdu_session_res_failed_to_setup_list_cxt_fail, type_, "Value");
  return c.get<pdu_session_res_failed_to_setup_list_cxt_fail_l>();
}
const cause_c& init_context_setup_fail_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const crit_diagnostics_s& init_context_setup_fail_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void init_context_setup_fail_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::pdu_session_res_failed_to_setup_list_cxt_fail:
      j.start_array("PDUSessionResourceFailedToSetupListCxtFail");
      for (const auto& e1 : c.get<pdu_session_res_failed_to_setup_list_cxt_fail_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_fail_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE init_context_setup_fail_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_failed_to_setup_list_cxt_fail:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_failed_to_setup_list_cxt_fail_l>(), 1, 256, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_fail_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE init_context_setup_fail_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_failed_to_setup_list_cxt_fail:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_failed_to_setup_list_cxt_fail_l>(), bref, 1, 256, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_fail_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* init_context_setup_fail_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "PDUSessionResourceFailedToSetupListCxtFail",
                                  "Cause",
                                  "CriticalityDiagnostics"};
  return convert_enum_idx(options, 5, value, "init_context_setup_fail_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<init_context_setup_fail_ies_o>;

init_context_setup_fail_ies_container::init_context_setup_fail_ies_container() :
  amf_ue_ngap_id(10, crit_e::ignore),
  ran_ue_ngap_id(85, crit_e::ignore),
  pdu_session_res_failed_to_setup_list_cxt_fail(132, crit_e::ignore),
  cause(15, crit_e::ignore),
  crit_diagnostics(19, crit_e::ignore)
{}
SRSASN_CODE init_context_setup_fail_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 3;
  nof_ies += pdu_session_res_failed_to_setup_list_cxt_fail_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  if (pdu_session_res_failed_to_setup_list_cxt_fail_present) {
    HANDLE_CODE(pdu_session_res_failed_to_setup_list_cxt_fail.pack(bref));
  }
  HANDLE_CODE(cause.pack(bref));
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE init_context_setup_fail_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 3;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 85: {
        nof_mandatory_ies--;
        ran_ue_ngap_id.id = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 132: {
        pdu_session_res_failed_to_setup_list_cxt_fail_present = true;
        pdu_session_res_failed_to_setup_list_cxt_fail.id      = id;
        HANDLE_CODE(pdu_session_res_failed_to_setup_list_cxt_fail.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(pdu_session_res_failed_to_setup_list_cxt_fail.value.unpack(bref));
        break;
      }
      case 15: {
        nof_mandatory_ies--;
        cause.id = id;
        HANDLE_CODE(cause.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(cause.value.unpack(bref));
        break;
      }
      case 19: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void init_context_setup_fail_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ran_ue_ngap_id.to_json(j);
  if (pdu_session_res_failed_to_setup_list_cxt_fail_present) {
    j.write_fieldname("");
    pdu_session_res_failed_to_setup_list_cxt_fail.to_json(j);
  }
  j.write_fieldname("");
  cause.to_json(j);
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceSetupItemCxtReq ::= SEQUENCE
SRSASN_CODE pdu_session_res_setup_item_cxt_req_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(nas_pdu.size() > 0, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  if (nas_pdu.size() > 0) {
    HANDLE_CODE(nas_pdu.pack(bref));
  }
  HANDLE_CODE(s_nssai.pack(bref));
  HANDLE_CODE(pdu_session_res_setup_request_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_setup_item_cxt_req_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  bool nas_pdu_present;
  HANDLE_CODE(bref.unpack(nas_pdu_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  if (nas_pdu_present) {
    HANDLE_CODE(nas_pdu.unpack(bref));
  }
  HANDLE_CODE(s_nssai.unpack(bref));
  HANDLE_CODE(pdu_session_res_setup_request_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_setup_item_cxt_req_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  if (nas_pdu.size() > 0) {
    j.write_str("nAS-PDU", nas_pdu.to_string());
  }
  j.write_fieldname("s-NSSAI");
  s_nssai.to_json(j);
  j.write_str("pDUSessionResourceSetupRequestTransfer", pdu_session_res_setup_request_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UERadioCapabilityForPaging ::= SEQUENCE
SRSASN_CODE ue_radio_cap_for_paging_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ueradio_cap_for_paging_of_nr.size() > 0, 1));
  HANDLE_CODE(bref.pack(ueradio_cap_for_paging_of_eutra.size() > 0, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (ueradio_cap_for_paging_of_nr.size() > 0) {
    HANDLE_CODE(ueradio_cap_for_paging_of_nr.pack(bref));
  }
  if (ueradio_cap_for_paging_of_eutra.size() > 0) {
    HANDLE_CODE(ueradio_cap_for_paging_of_eutra.pack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_radio_cap_for_paging_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  bool ueradio_cap_for_paging_of_nr_present;
  HANDLE_CODE(bref.unpack(ueradio_cap_for_paging_of_nr_present, 1));
  bool ueradio_cap_for_paging_of_eutra_present;
  HANDLE_CODE(bref.unpack(ueradio_cap_for_paging_of_eutra_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (ueradio_cap_for_paging_of_nr_present) {
    HANDLE_CODE(ueradio_cap_for_paging_of_nr.unpack(bref));
  }
  if (ueradio_cap_for_paging_of_eutra_present) {
    HANDLE_CODE(ueradio_cap_for_paging_of_eutra.unpack(bref));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ue_radio_cap_for_paging_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (ueradio_cap_for_paging_of_nr.size() > 0) {
    j.write_str("uERadioCapabilityForPagingOfNR", ueradio_cap_for_paging_of_nr.to_string());
  }
  if (ueradio_cap_for_paging_of_eutra.size() > 0) {
    j.write_str("uERadioCapabilityForPagingOfEUTRA", ueradio_cap_for_paging_of_eutra.to_string());
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// InitialContextSetupRequestIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t init_context_setup_request_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10,  85, 48,  110, 18, 28, 71, 0,  119, 94,
                                     108, 36, 117, 31,  34, 38, 24, 91, 118, 146};
  return map_enum_number(options, 20, idx, "id");
}
bool init_context_setup_request_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10,  85, 48,  110, 18, 28, 71, 0,  119, 94,
                                     108, 36, 117, 31,  34, 38, 24, 91, 118, 146};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e init_context_setup_request_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 48:
      return crit_e::reject;
    case 110:
      return crit_e::reject;
    case 18:
      return crit_e::ignore;
    case 28:
      return crit_e::reject;
    case 71:
      return crit_e::reject;
    case 0:
      return crit_e::reject;
    case 119:
      return crit_e::reject;
    case 94:
      return crit_e::reject;
    case 108:
      return crit_e::ignore;
    case 36:
      return crit_e::ignore;
    case 117:
      return crit_e::ignore;
    case 31:
      return crit_e::ignore;
    case 34:
      return crit_e::ignore;
    case 38:
      return crit_e::ignore;
    case 24:
      return crit_e::reject;
    case 91:
      return crit_e::ignore;
    case 118:
      return crit_e::ignore;
    case 146:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
init_context_setup_request_ies_o::value_c init_context_setup_request_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 48:
      ret.set(value_c::types::old_amf);
      break;
    case 110:
      ret.set(value_c::types::ue_aggregate_maximum_bit_rate);
      break;
    case 18:
      ret.set(value_c::types::core_network_assist_info);
      break;
    case 28:
      ret.set(value_c::types::guami);
      break;
    case 71:
      ret.set(value_c::types::pdu_session_res_setup_list_cxt_req);
      break;
    case 0:
      ret.set(value_c::types::allowed_nssai);
      break;
    case 119:
      ret.set(value_c::types::ue_security_cap);
      break;
    case 94:
      ret.set(value_c::types::security_key);
      break;
    case 108:
      ret.set(value_c::types::trace_activation);
      break;
    case 36:
      ret.set(value_c::types::mob_restrict_list);
      break;
    case 117:
      ret.set(value_c::types::ue_radio_cap);
      break;
    case 31:
      ret.set(value_c::types::idx_to_rfsp);
      break;
    case 34:
      ret.set(value_c::types::masked_imeisv);
      break;
    case 38:
      ret.set(value_c::types::nas_pdu);
      break;
    case 24:
      ret.set(value_c::types::emergency_fallback_ind);
      break;
    case 91:
      ret.set(value_c::types::rrc_inactive_transition_report_request);
      break;
    case 118:
      ret.set(value_c::types::ue_radio_cap_for_paging);
      break;
    case 146:
      ret.set(value_c::types::redirection_voice_fallback);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e init_context_setup_request_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 48:
      return presence_e::optional;
    case 110:
      return presence_e::conditional;
    case 18:
      return presence_e::optional;
    case 28:
      return presence_e::mandatory;
    case 71:
      return presence_e::optional;
    case 0:
      return presence_e::mandatory;
    case 119:
      return presence_e::mandatory;
    case 94:
      return presence_e::mandatory;
    case 108:
      return presence_e::optional;
    case 36:
      return presence_e::optional;
    case 117:
      return presence_e::optional;
    case 31:
      return presence_e::optional;
    case 34:
      return presence_e::optional;
    case 38:
      return presence_e::optional;
    case 24:
      return presence_e::optional;
    case 91:
      return presence_e::optional;
    case 118:
      return presence_e::optional;
    case 146:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void init_context_setup_request_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::old_amf:
      c.destroy<printable_string<1, 150, true, true> >();
      break;
    case types::ue_aggregate_maximum_bit_rate:
      c.destroy<ue_aggregate_maximum_bit_rate_s>();
      break;
    case types::core_network_assist_info:
      c.destroy<core_network_assist_info_s>();
      break;
    case types::guami:
      c.destroy<guami_s>();
      break;
    case types::pdu_session_res_setup_list_cxt_req:
      c.destroy<pdu_session_res_setup_list_cxt_req_l>();
      break;
    case types::allowed_nssai:
      c.destroy<allowed_nssai_l>();
      break;
    case types::ue_security_cap:
      c.destroy<ue_security_cap_s>();
      break;
    case types::security_key:
      c.destroy<fixed_bitstring<256, false, true> >();
      break;
    case types::trace_activation:
      c.destroy<trace_activation_s>();
      break;
    case types::mob_restrict_list:
      c.destroy<mob_restrict_list_s>();
      break;
    case types::ue_radio_cap:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::masked_imeisv:
      c.destroy<fixed_bitstring<64, false, true> >();
      break;
    case types::nas_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::emergency_fallback_ind:
      c.destroy<emergency_fallback_ind_s>();
      break;
    case types::ue_radio_cap_for_paging:
      c.destroy<ue_radio_cap_for_paging_s>();
      break;
    default:
      break;
  }
}
void init_context_setup_request_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::old_amf:
      c.init<printable_string<1, 150, true, true> >();
      break;
    case types::ue_aggregate_maximum_bit_rate:
      c.init<ue_aggregate_maximum_bit_rate_s>();
      break;
    case types::core_network_assist_info:
      c.init<core_network_assist_info_s>();
      break;
    case types::guami:
      c.init<guami_s>();
      break;
    case types::pdu_session_res_setup_list_cxt_req:
      c.init<pdu_session_res_setup_list_cxt_req_l>();
      break;
    case types::allowed_nssai:
      c.init<allowed_nssai_l>();
      break;
    case types::ue_security_cap:
      c.init<ue_security_cap_s>();
      break;
    case types::security_key:
      c.init<fixed_bitstring<256, false, true> >();
      break;
    case types::trace_activation:
      c.init<trace_activation_s>();
      break;
    case types::mob_restrict_list:
      c.init<mob_restrict_list_s>();
      break;
    case types::ue_radio_cap:
      c.init<unbounded_octstring<true> >();
      break;
    case types::idx_to_rfsp:
      break;
    case types::masked_imeisv:
      c.init<fixed_bitstring<64, false, true> >();
      break;
    case types::nas_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::emergency_fallback_ind:
      c.init<emergency_fallback_ind_s>();
      break;
    case types::rrc_inactive_transition_report_request:
      break;
    case types::ue_radio_cap_for_paging:
      c.init<ue_radio_cap_for_paging_s>();
      break;
    case types::redirection_voice_fallback:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_request_ies_o::value_c");
  }
}
init_context_setup_request_ies_o::value_c::value_c(const init_context_setup_request_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::old_amf:
      c.init(other.c.get<printable_string<1, 150, true, true> >());
      break;
    case types::ue_aggregate_maximum_bit_rate:
      c.init(other.c.get<ue_aggregate_maximum_bit_rate_s>());
      break;
    case types::core_network_assist_info:
      c.init(other.c.get<core_network_assist_info_s>());
      break;
    case types::guami:
      c.init(other.c.get<guami_s>());
      break;
    case types::pdu_session_res_setup_list_cxt_req:
      c.init(other.c.get<pdu_session_res_setup_list_cxt_req_l>());
      break;
    case types::allowed_nssai:
      c.init(other.c.get<allowed_nssai_l>());
      break;
    case types::ue_security_cap:
      c.init(other.c.get<ue_security_cap_s>());
      break;
    case types::security_key:
      c.init(other.c.get<fixed_bitstring<256, false, true> >());
      break;
    case types::trace_activation:
      c.init(other.c.get<trace_activation_s>());
      break;
    case types::mob_restrict_list:
      c.init(other.c.get<mob_restrict_list_s>());
      break;
    case types::ue_radio_cap:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::idx_to_rfsp:
      c.init(other.c.get<uint16_t>());
      break;
    case types::masked_imeisv:
      c.init(other.c.get<fixed_bitstring<64, false, true> >());
      break;
    case types::nas_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::emergency_fallback_ind:
      c.init(other.c.get<emergency_fallback_ind_s>());
      break;
    case types::rrc_inactive_transition_report_request:
      c.init(other.c.get<rrc_inactive_transition_report_request_e>());
      break;
    case types::ue_radio_cap_for_paging:
      c.init(other.c.get<ue_radio_cap_for_paging_s>());
      break;
    case types::redirection_voice_fallback:
      c.init(other.c.get<redirection_voice_fallback_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_request_ies_o::value_c");
  }
}
init_context_setup_request_ies_o::value_c&
init_context_setup_request_ies_o::value_c::operator=(const init_context_setup_request_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::old_amf:
      c.set(other.c.get<printable_string<1, 150, true, true> >());
      break;
    case types::ue_aggregate_maximum_bit_rate:
      c.set(other.c.get<ue_aggregate_maximum_bit_rate_s>());
      break;
    case types::core_network_assist_info:
      c.set(other.c.get<core_network_assist_info_s>());
      break;
    case types::guami:
      c.set(other.c.get<guami_s>());
      break;
    case types::pdu_session_res_setup_list_cxt_req:
      c.set(other.c.get<pdu_session_res_setup_list_cxt_req_l>());
      break;
    case types::allowed_nssai:
      c.set(other.c.get<allowed_nssai_l>());
      break;
    case types::ue_security_cap:
      c.set(other.c.get<ue_security_cap_s>());
      break;
    case types::security_key:
      c.set(other.c.get<fixed_bitstring<256, false, true> >());
      break;
    case types::trace_activation:
      c.set(other.c.get<trace_activation_s>());
      break;
    case types::mob_restrict_list:
      c.set(other.c.get<mob_restrict_list_s>());
      break;
    case types::ue_radio_cap:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::idx_to_rfsp:
      c.set(other.c.get<uint16_t>());
      break;
    case types::masked_imeisv:
      c.set(other.c.get<fixed_bitstring<64, false, true> >());
      break;
    case types::nas_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::emergency_fallback_ind:
      c.set(other.c.get<emergency_fallback_ind_s>());
      break;
    case types::rrc_inactive_transition_report_request:
      c.set(other.c.get<rrc_inactive_transition_report_request_e>());
      break;
    case types::ue_radio_cap_for_paging:
      c.set(other.c.get<ue_radio_cap_for_paging_s>());
      break;
    case types::redirection_voice_fallback:
      c.set(other.c.get<redirection_voice_fallback_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_request_ies_o::value_c");
  }

  return *this;
}
uint64_t& init_context_setup_request_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& init_context_setup_request_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
printable_string<1, 150, true, true>& init_context_setup_request_ies_o::value_c::old_amf()
{
  assert_choice_type(types::old_amf, type_, "Value");
  return c.get<printable_string<1, 150, true, true> >();
}
ue_aggregate_maximum_bit_rate_s& init_context_setup_request_ies_o::value_c::ue_aggregate_maximum_bit_rate()
{
  assert_choice_type(types::ue_aggregate_maximum_bit_rate, type_, "Value");
  return c.get<ue_aggregate_maximum_bit_rate_s>();
}
core_network_assist_info_s& init_context_setup_request_ies_o::value_c::core_network_assist_info()
{
  assert_choice_type(types::core_network_assist_info, type_, "Value");
  return c.get<core_network_assist_info_s>();
}
guami_s& init_context_setup_request_ies_o::value_c::guami()
{
  assert_choice_type(types::guami, type_, "Value");
  return c.get<guami_s>();
}
pdu_session_res_setup_list_cxt_req_l& init_context_setup_request_ies_o::value_c::pdu_session_res_setup_list_cxt_req()
{
  assert_choice_type(types::pdu_session_res_setup_list_cxt_req, type_, "Value");
  return c.get<pdu_session_res_setup_list_cxt_req_l>();
}
allowed_nssai_l& init_context_setup_request_ies_o::value_c::allowed_nssai()
{
  assert_choice_type(types::allowed_nssai, type_, "Value");
  return c.get<allowed_nssai_l>();
}
ue_security_cap_s& init_context_setup_request_ies_o::value_c::ue_security_cap()
{
  assert_choice_type(types::ue_security_cap, type_, "Value");
  return c.get<ue_security_cap_s>();
}
fixed_bitstring<256, false, true>& init_context_setup_request_ies_o::value_c::security_key()
{
  assert_choice_type(types::security_key, type_, "Value");
  return c.get<fixed_bitstring<256, false, true> >();
}
trace_activation_s& init_context_setup_request_ies_o::value_c::trace_activation()
{
  assert_choice_type(types::trace_activation, type_, "Value");
  return c.get<trace_activation_s>();
}
mob_restrict_list_s& init_context_setup_request_ies_o::value_c::mob_restrict_list()
{
  assert_choice_type(types::mob_restrict_list, type_, "Value");
  return c.get<mob_restrict_list_s>();
}
unbounded_octstring<true>& init_context_setup_request_ies_o::value_c::ue_radio_cap()
{
  assert_choice_type(types::ue_radio_cap, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
uint16_t& init_context_setup_request_ies_o::value_c::idx_to_rfsp()
{
  assert_choice_type(types::idx_to_rfsp, type_, "Value");
  return c.get<uint16_t>();
}
fixed_bitstring<64, false, true>& init_context_setup_request_ies_o::value_c::masked_imeisv()
{
  assert_choice_type(types::masked_imeisv, type_, "Value");
  return c.get<fixed_bitstring<64, false, true> >();
}
unbounded_octstring<true>& init_context_setup_request_ies_o::value_c::nas_pdu()
{
  assert_choice_type(types::nas_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
emergency_fallback_ind_s& init_context_setup_request_ies_o::value_c::emergency_fallback_ind()
{
  assert_choice_type(types::emergency_fallback_ind, type_, "Value");
  return c.get<emergency_fallback_ind_s>();
}
rrc_inactive_transition_report_request_e&
init_context_setup_request_ies_o::value_c::rrc_inactive_transition_report_request()
{
  assert_choice_type(types::rrc_inactive_transition_report_request, type_, "Value");
  return c.get<rrc_inactive_transition_report_request_e>();
}
ue_radio_cap_for_paging_s& init_context_setup_request_ies_o::value_c::ue_radio_cap_for_paging()
{
  assert_choice_type(types::ue_radio_cap_for_paging, type_, "Value");
  return c.get<ue_radio_cap_for_paging_s>();
}
redirection_voice_fallback_e& init_context_setup_request_ies_o::value_c::redirection_voice_fallback()
{
  assert_choice_type(types::redirection_voice_fallback, type_, "Value");
  return c.get<redirection_voice_fallback_e>();
}
const uint64_t& init_context_setup_request_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& init_context_setup_request_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const printable_string<1, 150, true, true>& init_context_setup_request_ies_o::value_c::old_amf() const
{
  assert_choice_type(types::old_amf, type_, "Value");
  return c.get<printable_string<1, 150, true, true> >();
}
const ue_aggregate_maximum_bit_rate_s& init_context_setup_request_ies_o::value_c::ue_aggregate_maximum_bit_rate() const
{
  assert_choice_type(types::ue_aggregate_maximum_bit_rate, type_, "Value");
  return c.get<ue_aggregate_maximum_bit_rate_s>();
}
const core_network_assist_info_s& init_context_setup_request_ies_o::value_c::core_network_assist_info() const
{
  assert_choice_type(types::core_network_assist_info, type_, "Value");
  return c.get<core_network_assist_info_s>();
}
const guami_s& init_context_setup_request_ies_o::value_c::guami() const
{
  assert_choice_type(types::guami, type_, "Value");
  return c.get<guami_s>();
}
const pdu_session_res_setup_list_cxt_req_l&
init_context_setup_request_ies_o::value_c::pdu_session_res_setup_list_cxt_req() const
{
  assert_choice_type(types::pdu_session_res_setup_list_cxt_req, type_, "Value");
  return c.get<pdu_session_res_setup_list_cxt_req_l>();
}
const allowed_nssai_l& init_context_setup_request_ies_o::value_c::allowed_nssai() const
{
  assert_choice_type(types::allowed_nssai, type_, "Value");
  return c.get<allowed_nssai_l>();
}
const ue_security_cap_s& init_context_setup_request_ies_o::value_c::ue_security_cap() const
{
  assert_choice_type(types::ue_security_cap, type_, "Value");
  return c.get<ue_security_cap_s>();
}
const fixed_bitstring<256, false, true>& init_context_setup_request_ies_o::value_c::security_key() const
{
  assert_choice_type(types::security_key, type_, "Value");
  return c.get<fixed_bitstring<256, false, true> >();
}
const trace_activation_s& init_context_setup_request_ies_o::value_c::trace_activation() const
{
  assert_choice_type(types::trace_activation, type_, "Value");
  return c.get<trace_activation_s>();
}
const mob_restrict_list_s& init_context_setup_request_ies_o::value_c::mob_restrict_list() const
{
  assert_choice_type(types::mob_restrict_list, type_, "Value");
  return c.get<mob_restrict_list_s>();
}
const unbounded_octstring<true>& init_context_setup_request_ies_o::value_c::ue_radio_cap() const
{
  assert_choice_type(types::ue_radio_cap, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const uint16_t& init_context_setup_request_ies_o::value_c::idx_to_rfsp() const
{
  assert_choice_type(types::idx_to_rfsp, type_, "Value");
  return c.get<uint16_t>();
}
const fixed_bitstring<64, false, true>& init_context_setup_request_ies_o::value_c::masked_imeisv() const
{
  assert_choice_type(types::masked_imeisv, type_, "Value");
  return c.get<fixed_bitstring<64, false, true> >();
}
const unbounded_octstring<true>& init_context_setup_request_ies_o::value_c::nas_pdu() const
{
  assert_choice_type(types::nas_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const emergency_fallback_ind_s& init_context_setup_request_ies_o::value_c::emergency_fallback_ind() const
{
  assert_choice_type(types::emergency_fallback_ind, type_, "Value");
  return c.get<emergency_fallback_ind_s>();
}
const rrc_inactive_transition_report_request_e&
init_context_setup_request_ies_o::value_c::rrc_inactive_transition_report_request() const
{
  assert_choice_type(types::rrc_inactive_transition_report_request, type_, "Value");
  return c.get<rrc_inactive_transition_report_request_e>();
}
const ue_radio_cap_for_paging_s& init_context_setup_request_ies_o::value_c::ue_radio_cap_for_paging() const
{
  assert_choice_type(types::ue_radio_cap_for_paging, type_, "Value");
  return c.get<ue_radio_cap_for_paging_s>();
}
const redirection_voice_fallback_e& init_context_setup_request_ies_o::value_c::redirection_voice_fallback() const
{
  assert_choice_type(types::redirection_voice_fallback, type_, "Value");
  return c.get<redirection_voice_fallback_e>();
}
void init_context_setup_request_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::old_amf:
      j.write_str("PrintableString", c.get<printable_string<1, 150, true, true> >().to_string());
      break;
    case types::ue_aggregate_maximum_bit_rate:
      j.write_fieldname("UEAggregateMaximumBitRate");
      c.get<ue_aggregate_maximum_bit_rate_s>().to_json(j);
      break;
    case types::core_network_assist_info:
      j.write_fieldname("CoreNetworkAssistanceInformation");
      c.get<core_network_assist_info_s>().to_json(j);
      break;
    case types::guami:
      j.write_fieldname("GUAMI");
      c.get<guami_s>().to_json(j);
      break;
    case types::pdu_session_res_setup_list_cxt_req:
      j.start_array("PDUSessionResourceSetupListCxtReq");
      for (const auto& e1 : c.get<pdu_session_res_setup_list_cxt_req_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::allowed_nssai:
      j.start_array("AllowedNSSAI");
      for (const auto& e1 : c.get<allowed_nssai_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::ue_security_cap:
      j.write_fieldname("UESecurityCapabilities");
      c.get<ue_security_cap_s>().to_json(j);
      break;
    case types::security_key:
      j.write_str("BIT STRING", c.get<fixed_bitstring<256, false, true> >().to_string());
      break;
    case types::trace_activation:
      j.write_fieldname("TraceActivation");
      c.get<trace_activation_s>().to_json(j);
      break;
    case types::mob_restrict_list:
      j.write_fieldname("MobilityRestrictionList");
      c.get<mob_restrict_list_s>().to_json(j);
      break;
    case types::ue_radio_cap:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::idx_to_rfsp:
      j.write_int("INTEGER (1..256,...)", c.get<uint16_t>());
      break;
    case types::masked_imeisv:
      j.write_str("BIT STRING", c.get<fixed_bitstring<64, false, true> >().to_string());
      break;
    case types::nas_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::emergency_fallback_ind:
      j.write_fieldname("EmergencyFallbackIndicator");
      c.get<emergency_fallback_ind_s>().to_json(j);
      break;
    case types::rrc_inactive_transition_report_request:
      j.write_str("RRCInactiveTransitionReportRequest", c.get<rrc_inactive_transition_report_request_e>().to_string());
      break;
    case types::ue_radio_cap_for_paging:
      j.write_fieldname("UERadioCapabilityForPaging");
      c.get<ue_radio_cap_for_paging_s>().to_json(j);
      break;
    case types::redirection_voice_fallback:
      j.write_str("RedirectionVoiceFallback", c.get<redirection_voice_fallback_e>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_request_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE init_context_setup_request_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::old_amf:
      HANDLE_CODE((c.get<printable_string<1, 150, true, true> >().pack(bref)));
      break;
    case types::ue_aggregate_maximum_bit_rate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bit_rate_s>().pack(bref));
      break;
    case types::core_network_assist_info:
      HANDLE_CODE(c.get<core_network_assist_info_s>().pack(bref));
      break;
    case types::guami:
      HANDLE_CODE(c.get<guami_s>().pack(bref));
      break;
    case types::pdu_session_res_setup_list_cxt_req:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_setup_list_cxt_req_l>(), 1, 256, true));
      break;
    case types::allowed_nssai:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<allowed_nssai_l>(), 1, 8, true));
      break;
    case types::ue_security_cap:
      HANDLE_CODE(c.get<ue_security_cap_s>().pack(bref));
      break;
    case types::security_key:
      HANDLE_CODE((c.get<fixed_bitstring<256, false, true> >().pack(bref)));
      break;
    case types::trace_activation:
      HANDLE_CODE(c.get<trace_activation_s>().pack(bref));
      break;
    case types::mob_restrict_list:
      HANDLE_CODE(c.get<mob_restrict_list_s>().pack(bref));
      break;
    case types::ue_radio_cap:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::idx_to_rfsp:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)1u, (uint16_t)256u, true, true));
      break;
    case types::masked_imeisv:
      HANDLE_CODE((c.get<fixed_bitstring<64, false, true> >().pack(bref)));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::emergency_fallback_ind:
      HANDLE_CODE(c.get<emergency_fallback_ind_s>().pack(bref));
      break;
    case types::rrc_inactive_transition_report_request:
      HANDLE_CODE(c.get<rrc_inactive_transition_report_request_e>().pack(bref));
      break;
    case types::ue_radio_cap_for_paging:
      HANDLE_CODE(c.get<ue_radio_cap_for_paging_s>().pack(bref));
      break;
    case types::redirection_voice_fallback:
      HANDLE_CODE(c.get<redirection_voice_fallback_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_request_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE init_context_setup_request_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::old_amf:
      HANDLE_CODE((c.get<printable_string<1, 150, true, true> >().unpack(bref)));
      break;
    case types::ue_aggregate_maximum_bit_rate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bit_rate_s>().unpack(bref));
      break;
    case types::core_network_assist_info:
      HANDLE_CODE(c.get<core_network_assist_info_s>().unpack(bref));
      break;
    case types::guami:
      HANDLE_CODE(c.get<guami_s>().unpack(bref));
      break;
    case types::pdu_session_res_setup_list_cxt_req:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_setup_list_cxt_req_l>(), bref, 1, 256, true));
      break;
    case types::allowed_nssai:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<allowed_nssai_l>(), bref, 1, 8, true));
      break;
    case types::ue_security_cap:
      HANDLE_CODE(c.get<ue_security_cap_s>().unpack(bref));
      break;
    case types::security_key:
      HANDLE_CODE((c.get<fixed_bitstring<256, false, true> >().unpack(bref)));
      break;
    case types::trace_activation:
      HANDLE_CODE(c.get<trace_activation_s>().unpack(bref));
      break;
    case types::mob_restrict_list:
      HANDLE_CODE(c.get<mob_restrict_list_s>().unpack(bref));
      break;
    case types::ue_radio_cap:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::idx_to_rfsp:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)1u, (uint16_t)256u, true, true));
      break;
    case types::masked_imeisv:
      HANDLE_CODE((c.get<fixed_bitstring<64, false, true> >().unpack(bref)));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::emergency_fallback_ind:
      HANDLE_CODE(c.get<emergency_fallback_ind_s>().unpack(bref));
      break;
    case types::rrc_inactive_transition_report_request:
      HANDLE_CODE(c.get<rrc_inactive_transition_report_request_e>().unpack(bref));
      break;
    case types::ue_radio_cap_for_paging:
      HANDLE_CODE(c.get<ue_radio_cap_for_paging_s>().unpack(bref));
      break;
    case types::redirection_voice_fallback:
      HANDLE_CODE(c.get<redirection_voice_fallback_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_request_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* init_context_setup_request_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "PrintableString",
                                  "UEAggregateMaximumBitRate",
                                  "CoreNetworkAssistanceInformation",
                                  "GUAMI",
                                  "PDUSessionResourceSetupListCxtReq",
                                  "AllowedNSSAI",
                                  "UESecurityCapabilities",
                                  "BIT STRING",
                                  "TraceActivation",
                                  "MobilityRestrictionList",
                                  "OCTET STRING",
                                  "INTEGER (1..256,...)",
                                  "BIT STRING",
                                  "OCTET STRING",
                                  "EmergencyFallbackIndicator",
                                  "RRCInactiveTransitionReportRequest",
                                  "UERadioCapabilityForPaging",
                                  "RedirectionVoiceFallback"};
  return convert_enum_idx(options, 20, value, "init_context_setup_request_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<init_context_setup_request_ies_o>;

init_context_setup_request_ies_container::init_context_setup_request_ies_container() :
  amf_ue_ngap_id(10, crit_e::reject),
  ran_ue_ngap_id(85, crit_e::reject),
  old_amf(48, crit_e::reject),
  ue_aggregate_maximum_bit_rate(110, crit_e::reject),
  core_network_assist_info(18, crit_e::ignore),
  guami(28, crit_e::reject),
  pdu_session_res_setup_list_cxt_req(71, crit_e::reject),
  allowed_nssai(0, crit_e::reject),
  ue_security_cap(119, crit_e::reject),
  security_key(94, crit_e::reject),
  trace_activation(108, crit_e::ignore),
  mob_restrict_list(36, crit_e::ignore),
  ue_radio_cap(117, crit_e::ignore),
  idx_to_rfsp(31, crit_e::ignore),
  masked_imeisv(34, crit_e::ignore),
  nas_pdu(38, crit_e::ignore),
  emergency_fallback_ind(24, crit_e::reject),
  rrc_inactive_transition_report_request(91, crit_e::ignore),
  ue_radio_cap_for_paging(118, crit_e::ignore),
  redirection_voice_fallback(146, crit_e::ignore)
{}
SRSASN_CODE init_context_setup_request_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 6;
  nof_ies += old_amf_present ? 1 : 0;
  nof_ies += ue_aggregate_maximum_bit_rate_present ? 1 : 0;
  nof_ies += core_network_assist_info_present ? 1 : 0;
  nof_ies += pdu_session_res_setup_list_cxt_req_present ? 1 : 0;
  nof_ies += trace_activation_present ? 1 : 0;
  nof_ies += mob_restrict_list_present ? 1 : 0;
  nof_ies += ue_radio_cap_present ? 1 : 0;
  nof_ies += idx_to_rfsp_present ? 1 : 0;
  nof_ies += masked_imeisv_present ? 1 : 0;
  nof_ies += nas_pdu_present ? 1 : 0;
  nof_ies += emergency_fallback_ind_present ? 1 : 0;
  nof_ies += rrc_inactive_transition_report_request_present ? 1 : 0;
  nof_ies += ue_radio_cap_for_paging_present ? 1 : 0;
  nof_ies += redirection_voice_fallback_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  if (old_amf_present) {
    HANDLE_CODE(old_amf.pack(bref));
  }
  if (ue_aggregate_maximum_bit_rate_present) {
    HANDLE_CODE(ue_aggregate_maximum_bit_rate.pack(bref));
  }
  if (core_network_assist_info_present) {
    HANDLE_CODE(core_network_assist_info.pack(bref));
  }
  HANDLE_CODE(guami.pack(bref));
  if (pdu_session_res_setup_list_cxt_req_present) {
    HANDLE_CODE(pdu_session_res_setup_list_cxt_req.pack(bref));
  }
  HANDLE_CODE(allowed_nssai.pack(bref));
  HANDLE_CODE(ue_security_cap.pack(bref));
  HANDLE_CODE(security_key.pack(bref));
  if (trace_activation_present) {
    HANDLE_CODE(trace_activation.pack(bref));
  }
  if (mob_restrict_list_present) {
    HANDLE_CODE(mob_restrict_list.pack(bref));
  }
  if (ue_radio_cap_present) {
    HANDLE_CODE(ue_radio_cap.pack(bref));
  }
  if (idx_to_rfsp_present) {
    HANDLE_CODE(idx_to_rfsp.pack(bref));
  }
  if (masked_imeisv_present) {
    HANDLE_CODE(masked_imeisv.pack(bref));
  }
  if (nas_pdu_present) {
    HANDLE_CODE(nas_pdu.pack(bref));
  }
  if (emergency_fallback_ind_present) {
    HANDLE_CODE(emergency_fallback_ind.pack(bref));
  }
  if (rrc_inactive_transition_report_request_present) {
    HANDLE_CODE(rrc_inactive_transition_report_request.pack(bref));
  }
  if (ue_radio_cap_for_paging_present) {
    HANDLE_CODE(ue_radio_cap_for_paging.pack(bref));
  }
  if (redirection_voice_fallback_present) {
    HANDLE_CODE(redirection_voice_fallback.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE init_context_setup_request_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 6;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 85: {
        nof_mandatory_ies--;
        ran_ue_ngap_id.id = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 48: {
        old_amf_present = true;
        old_amf.id      = id;
        HANDLE_CODE(old_amf.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(old_amf.value.unpack(bref));
        break;
      }
      case 110: {
        ue_aggregate_maximum_bit_rate_present = true;
        ue_aggregate_maximum_bit_rate.id      = id;
        HANDLE_CODE(ue_aggregate_maximum_bit_rate.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ue_aggregate_maximum_bit_rate.value.unpack(bref));
        break;
      }
      case 18: {
        core_network_assist_info_present = true;
        core_network_assist_info.id      = id;
        HANDLE_CODE(core_network_assist_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(core_network_assist_info.value.unpack(bref));
        break;
      }
      case 28: {
        nof_mandatory_ies--;
        guami.id = id;
        HANDLE_CODE(guami.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(guami.value.unpack(bref));
        break;
      }
      case 71: {
        pdu_session_res_setup_list_cxt_req_present = true;
        pdu_session_res_setup_list_cxt_req.id      = id;
        HANDLE_CODE(pdu_session_res_setup_list_cxt_req.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(pdu_session_res_setup_list_cxt_req.value.unpack(bref));
        break;
      }
      case 0: {
        nof_mandatory_ies--;
        allowed_nssai.id = id;
        HANDLE_CODE(allowed_nssai.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(allowed_nssai.value.unpack(bref));
        break;
      }
      case 119: {
        nof_mandatory_ies--;
        ue_security_cap.id = id;
        HANDLE_CODE(ue_security_cap.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ue_security_cap.value.unpack(bref));
        break;
      }
      case 94: {
        nof_mandatory_ies--;
        security_key.id = id;
        HANDLE_CODE(security_key.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(security_key.value.unpack(bref));
        break;
      }
      case 108: {
        trace_activation_present = true;
        trace_activation.id      = id;
        HANDLE_CODE(trace_activation.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(trace_activation.value.unpack(bref));
        break;
      }
      case 36: {
        mob_restrict_list_present = true;
        mob_restrict_list.id      = id;
        HANDLE_CODE(mob_restrict_list.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(mob_restrict_list.value.unpack(bref));
        break;
      }
      case 117: {
        ue_radio_cap_present = true;
        ue_radio_cap.id      = id;
        HANDLE_CODE(ue_radio_cap.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ue_radio_cap.value.unpack(bref));
        break;
      }
      case 31: {
        idx_to_rfsp_present = true;
        idx_to_rfsp.id      = id;
        HANDLE_CODE(idx_to_rfsp.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(idx_to_rfsp.value.unpack(bref));
        break;
      }
      case 34: {
        masked_imeisv_present = true;
        masked_imeisv.id      = id;
        HANDLE_CODE(masked_imeisv.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(masked_imeisv.value.unpack(bref));
        break;
      }
      case 38: {
        nas_pdu_present = true;
        nas_pdu.id      = id;
        HANDLE_CODE(nas_pdu.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(nas_pdu.value.unpack(bref));
        break;
      }
      case 24: {
        emergency_fallback_ind_present = true;
        emergency_fallback_ind.id      = id;
        HANDLE_CODE(emergency_fallback_ind.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(emergency_fallback_ind.value.unpack(bref));
        break;
      }
      case 91: {
        rrc_inactive_transition_report_request_present = true;
        rrc_inactive_transition_report_request.id      = id;
        HANDLE_CODE(rrc_inactive_transition_report_request.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(rrc_inactive_transition_report_request.value.unpack(bref));
        break;
      }
      case 118: {
        ue_radio_cap_for_paging_present = true;
        ue_radio_cap_for_paging.id      = id;
        HANDLE_CODE(ue_radio_cap_for_paging.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ue_radio_cap_for_paging.value.unpack(bref));
        break;
      }
      case 146: {
        redirection_voice_fallback_present = true;
        redirection_voice_fallback.id      = id;
        HANDLE_CODE(redirection_voice_fallback.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(redirection_voice_fallback.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void init_context_setup_request_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ran_ue_ngap_id.to_json(j);
  if (old_amf_present) {
    j.write_fieldname("");
    old_amf.to_json(j);
  }
  if (ue_aggregate_maximum_bit_rate_present) {
    j.write_fieldname("");
    ue_aggregate_maximum_bit_rate.to_json(j);
  }
  if (core_network_assist_info_present) {
    j.write_fieldname("");
    core_network_assist_info.to_json(j);
  }
  j.write_fieldname("");
  guami.to_json(j);
  if (pdu_session_res_setup_list_cxt_req_present) {
    j.write_fieldname("");
    pdu_session_res_setup_list_cxt_req.to_json(j);
  }
  j.write_fieldname("");
  allowed_nssai.to_json(j);
  j.write_fieldname("");
  ue_security_cap.to_json(j);
  j.write_fieldname("");
  security_key.to_json(j);
  if (trace_activation_present) {
    j.write_fieldname("");
    trace_activation.to_json(j);
  }
  if (mob_restrict_list_present) {
    j.write_fieldname("");
    mob_restrict_list.to_json(j);
  }
  if (ue_radio_cap_present) {
    j.write_fieldname("");
    ue_radio_cap.to_json(j);
  }
  if (idx_to_rfsp_present) {
    j.write_fieldname("");
    idx_to_rfsp.to_json(j);
  }
  if (masked_imeisv_present) {
    j.write_fieldname("");
    masked_imeisv.to_json(j);
  }
  if (nas_pdu_present) {
    j.write_fieldname("");
    nas_pdu.to_json(j);
  }
  if (emergency_fallback_ind_present) {
    j.write_fieldname("");
    emergency_fallback_ind.to_json(j);
  }
  if (rrc_inactive_transition_report_request_present) {
    j.write_fieldname("");
    rrc_inactive_transition_report_request.to_json(j);
  }
  if (ue_radio_cap_for_paging_present) {
    j.write_fieldname("");
    ue_radio_cap_for_paging.to_json(j);
  }
  if (redirection_voice_fallback_present) {
    j.write_fieldname("");
    redirection_voice_fallback.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceFailedToSetupItemCxtRes ::= SEQUENCE
SRSASN_CODE pdu_session_res_failed_to_setup_item_cxt_res_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_setup_unsuccessful_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_failed_to_setup_item_cxt_res_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_setup_unsuccessful_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_failed_to_setup_item_cxt_res_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pDUSessionResourceSetupUnsuccessfulTransfer", pdu_session_res_setup_unsuccessful_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceSetupItemCxtRes ::= SEQUENCE
SRSASN_CODE pdu_session_res_setup_item_cxt_res_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_setup_resp_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_setup_item_cxt_res_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_setup_resp_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_setup_item_cxt_res_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pDUSessionResourceSetupResponseTransfer", pdu_session_res_setup_resp_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// InitialContextSetupResponseIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t init_context_setup_resp_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 72, 55, 19};
  return map_enum_number(options, 5, idx, "id");
}
bool init_context_setup_resp_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 72, 55, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e init_context_setup_resp_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::ignore;
    case 85:
      return crit_e::ignore;
    case 72:
      return crit_e::ignore;
    case 55:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
init_context_setup_resp_ies_o::value_c init_context_setup_resp_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 72:
      ret.set(value_c::types::pdu_session_res_setup_list_cxt_res);
      break;
    case 55:
      ret.set(value_c::types::pdu_session_res_failed_to_setup_list_cxt_res);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e init_context_setup_resp_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 72:
      return presence_e::optional;
    case 55:
      return presence_e::optional;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void init_context_setup_resp_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::pdu_session_res_setup_list_cxt_res:
      c.destroy<pdu_session_res_setup_list_cxt_res_l>();
      break;
    case types::pdu_session_res_failed_to_setup_list_cxt_res:
      c.destroy<pdu_session_res_failed_to_setup_list_cxt_res_l>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void init_context_setup_resp_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::pdu_session_res_setup_list_cxt_res:
      c.init<pdu_session_res_setup_list_cxt_res_l>();
      break;
    case types::pdu_session_res_failed_to_setup_list_cxt_res:
      c.init<pdu_session_res_failed_to_setup_list_cxt_res_l>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_resp_ies_o::value_c");
  }
}
init_context_setup_resp_ies_o::value_c::value_c(const init_context_setup_resp_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_setup_list_cxt_res:
      c.init(other.c.get<pdu_session_res_setup_list_cxt_res_l>());
      break;
    case types::pdu_session_res_failed_to_setup_list_cxt_res:
      c.init(other.c.get<pdu_session_res_failed_to_setup_list_cxt_res_l>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_resp_ies_o::value_c");
  }
}
init_context_setup_resp_ies_o::value_c&
init_context_setup_resp_ies_o::value_c::operator=(const init_context_setup_resp_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_setup_list_cxt_res:
      c.set(other.c.get<pdu_session_res_setup_list_cxt_res_l>());
      break;
    case types::pdu_session_res_failed_to_setup_list_cxt_res:
      c.set(other.c.get<pdu_session_res_failed_to_setup_list_cxt_res_l>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_resp_ies_o::value_c");
  }

  return *this;
}
uint64_t& init_context_setup_resp_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& init_context_setup_resp_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
pdu_session_res_setup_list_cxt_res_l& init_context_setup_resp_ies_o::value_c::pdu_session_res_setup_list_cxt_res()
{
  assert_choice_type(types::pdu_session_res_setup_list_cxt_res, type_, "Value");
  return c.get<pdu_session_res_setup_list_cxt_res_l>();
}
pdu_session_res_failed_to_setup_list_cxt_res_l&
init_context_setup_resp_ies_o::value_c::pdu_session_res_failed_to_setup_list_cxt_res()
{
  assert_choice_type(types::pdu_session_res_failed_to_setup_list_cxt_res, type_, "Value");
  return c.get<pdu_session_res_failed_to_setup_list_cxt_res_l>();
}
crit_diagnostics_s& init_context_setup_resp_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& init_context_setup_resp_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& init_context_setup_resp_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const pdu_session_res_setup_list_cxt_res_l&
init_context_setup_resp_ies_o::value_c::pdu_session_res_setup_list_cxt_res() const
{
  assert_choice_type(types::pdu_session_res_setup_list_cxt_res, type_, "Value");
  return c.get<pdu_session_res_setup_list_cxt_res_l>();
}
const pdu_session_res_failed_to_setup_list_cxt_res_l&
init_context_setup_resp_ies_o::value_c::pdu_session_res_failed_to_setup_list_cxt_res() const
{
  assert_choice_type(types::pdu_session_res_failed_to_setup_list_cxt_res, type_, "Value");
  return c.get<pdu_session_res_failed_to_setup_list_cxt_res_l>();
}
const crit_diagnostics_s& init_context_setup_resp_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void init_context_setup_resp_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::pdu_session_res_setup_list_cxt_res:
      j.start_array("PDUSessionResourceSetupListCxtRes");
      for (const auto& e1 : c.get<pdu_session_res_setup_list_cxt_res_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::pdu_session_res_failed_to_setup_list_cxt_res:
      j.start_array("PDUSessionResourceFailedToSetupListCxtRes");
      for (const auto& e1 : c.get<pdu_session_res_failed_to_setup_list_cxt_res_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_resp_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE init_context_setup_resp_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_setup_list_cxt_res:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_setup_list_cxt_res_l>(), 1, 256, true));
      break;
    case types::pdu_session_res_failed_to_setup_list_cxt_res:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_failed_to_setup_list_cxt_res_l>(), 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_resp_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE init_context_setup_resp_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_setup_list_cxt_res:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_setup_list_cxt_res_l>(), bref, 1, 256, true));
      break;
    case types::pdu_session_res_failed_to_setup_list_cxt_res:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_failed_to_setup_list_cxt_res_l>(), bref, 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "init_context_setup_resp_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* init_context_setup_resp_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "PDUSessionResourceSetupListCxtRes",
                                  "PDUSessionResourceFailedToSetupListCxtRes",
                                  "CriticalityDiagnostics"};
  return convert_enum_idx(options, 5, value, "init_context_setup_resp_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<init_context_setup_resp_ies_o>;

init_context_setup_resp_ies_container::init_context_setup_resp_ies_container() :
  amf_ue_ngap_id(10, crit_e::ignore),
  ran_ue_ngap_id(85, crit_e::ignore),
  pdu_session_res_setup_list_cxt_res(72, crit_e::ignore),
  pdu_session_res_failed_to_setup_list_cxt_res(55, crit_e::ignore),
  crit_diagnostics(19, crit_e::ignore)
{}
SRSASN_CODE init_context_setup_resp_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 2;
  nof_ies += pdu_session_res_setup_list_cxt_res_present ? 1 : 0;
  nof_ies += pdu_session_res_failed_to_setup_list_cxt_res_present ? 1 : 0;
  nof_ies += crit_diagnostics_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(amf_ue_ngap_id.pack(bref));
  HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  if (pdu_session_res_setup_list_cxt_res_present) {
    HANDLE_CODE(pdu_session_res_setup_list_cxt_res.pack(bref));
  }
  if (pdu_session_res_failed_to_setup_list_cxt_res_present) {
    HANDLE_CODE(pdu_session_res_failed_to_setup_list_cxt_res.pack(bref));
  }
  if (crit_diagnostics_present) {
    HANDLE_CODE(crit_diagnostics.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE init_context_setup_resp_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 2;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 10: {
        nof_mandatory_ies--;
        amf_ue_ngap_id.id = id;
        HANDLE_CODE(amf_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 85: {
        nof_mandatory_ies--;
        ran_ue_ngap_id.id = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 72: {
        pdu_session_res_setup_list_cxt_res_present = true;
        pdu_session_res_setup_list_cxt_res.id      = id;
        HANDLE_CODE(pdu_session_res_setup_list_cxt_res.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(pdu_session_res_setup_list_cxt_res.value.unpack(bref));
        break;
      }
      case 55: {
        pdu_session_res_failed_to_setup_list_cxt_res_present = true;
        pdu_session_res_failed_to_setup_list_cxt_res.id      = id;
        HANDLE_CODE(pdu_session_res_failed_to_setup_list_cxt_res.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(pdu_session_res_failed_to_setup_list_cxt_res.value.unpack(bref));
        break;
      }
      case 19: {
        crit_diagnostics_present = true;
        crit_diagnostics.id      = id;
        HANDLE_CODE(crit_diagnostics.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(crit_diagnostics.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void init_context_setup_resp_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  amf_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  ran_ue_ngap_id.to_json(j);
  if (pdu_session_res_setup_list_cxt_res_present) {
    j.write_fieldname("");
    pdu_session_res_setup_list_cxt_res.to_json(j);
  }
  if (pdu_session_res_failed_to_setup_list_cxt_res_present) {
    j.write_fieldname("");
    pdu_session_res_failed_to_setup_list_cxt_res.to_json(j);
  }
  if (crit_diagnostics_present) {
    j.write_fieldname("");
    crit_diagnostics.to_json(j);
  }
  j.end_obj();
}

// RRCEstablishmentCause ::= ENUMERATED
const char* rrcestablishment_cause_opts::to_string() const
{
  static const char* options[] = {"emergency",
                                  "highPriorityAccess",
                                  "mt-Access",
                                  "mo-Signalling",
                                  "mo-Data",
                                  "mo-VoiceCall",
                                  "mo-VideoCall",
                                  "mo-SMS",
                                  "mps-PriorityAccess",
                                  "mcs-PriorityAccess",
                                  "notAvailable"};
  return convert_enum_idx(options, 11, value, "rrcestablishment_cause_e");
}

// UEContextRequest ::= ENUMERATED
const char* ue_context_request_opts::to_string() const
{
  static const char* options[] = {"requested"};
  return convert_enum_idx(options, 1, value, "ue_context_request_e");
}

// InitialUEMessage-IEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t init_ue_msg_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {85, 38, 121, 90, 26, 3, 112, 0};
  return map_enum_number(options, 8, idx, "id");
}
bool init_ue_msg_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {85, 38, 121, 90, 26, 3, 112, 0};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e init_ue_msg_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 85:
      return crit_e::reject;
    case 38:
      return crit_e::reject;
    case 121:
      return crit_e::reject;
    case 90:
      return crit_e::ignore;
    case 26:
      return crit_e::reject;
    case 3:
      return crit_e::ignore;
    case 112:
      return crit_e::ignore;
    case 0:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
init_ue_msg_ies_o::value_c init_ue_msg_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 38:
      ret.set(value_c::types::nas_pdu);
      break;
    case 121:
      ret.set(value_c::types::user_location_info);
      break;
    case 90:
      ret.set(value_c::types::rrcestablishment_cause);
      break;
    case 26:
      ret.set(value_c::types::five_g_s_tmsi);
      break;
    case 3:
      ret.set(value_c::types::amf_set_id);
      break;
    case 112:
      ret.set(value_c::types::ue_context_request);
      break;
    case 0:
      ret.set(value_c::types::allowed_nssai);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e init_ue_msg_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 85:
      return presence_e::mandatory;
    case 38:
      return presence_e::mandatory;
    case 121:
      return presence_e::mandatory;
    case 90:
      return presence_e::mandatory;
    case 26:
      return presence_e::optional;
    case 3:
      return presence_e::optional;
    case 112:
      return presence_e::optional;
    case 0:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void init_ue_msg_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::nas_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::user_location_info:
      c.destroy<user_location_info_c>();
      break;
    case types::five_g_s_tmsi:
      c.destroy<five_g_s_tmsi_s>();
      break;
    case types::amf_set_id:
      c.destroy<fixed_bitstring<10, false, true> >();
      break;
    case types::allowed_nssai:
      c.destroy<allowed_nssai_l>();
      break;
    default:
      break;
  }
}
void init_ue_msg_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::ran_ue_ngap_id:
      break;
    case types::nas_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::user_location_info:
      c.init<user_location_info_c>();
      break;
    case types::rrcestablishment_cause:
      break;
    case types::five_g_s_tmsi:
      c.init<five_g_s_tmsi_s>();
      break;
    case types::amf_set_id:
      c.init<fixed_bitstring<10, false, true> >();
      break;
    case types::ue_context_request:
      break;
    case types::allowed_nssai:
      c.init<allowed_nssai_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_ue_msg_ies_o::value_c");
  }
}
init_ue_msg_ies_o::value_c::value_c(const init_ue_msg_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::nas_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::user_location_info:
      c.init(other.c.get<user_location_info_c>());
      break;
    case types::rrcestablishment_cause:
      c.init(other.c.get<rrcestablishment_cause_e>());
      break;
    case types::five_g_s_tmsi:
      c.init(other.c.get<five_g_s_tmsi_s>());
      break;
    case types::amf_set_id:
      c.init(other.c.get<fixed_bitstring<10, false, true> >());
      break;
    case types::ue_context_request:
      c.init(other.c.get<ue_context_request_e>());
      break;
    case types::allowed_nssai:
      c.init(other.c.get<allowed_nssai_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_ue_msg_ies_o::value_c");
  }
}
init_ue_msg_ies_o::value_c& init_ue_msg_ies_o::value_c::operator=(const init_ue_msg_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::nas_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::user_location_info:
      c.set(other.c.get<user_location_info_c>());
      break;
    case types::rrcestablishment_cause:
      c.set(other.c.get<rrcestablishment_cause_e>());
      break;
    case types::five_g_s_tmsi:
      c.set(other.c.get<five_g_s_tmsi_s>());
      break;
    case types::amf_set_id:
      c.set(other.c.get<fixed_bitstring<10, false, true> >());
      break;
    case types::ue_context_request:
      c.set(other.c.get<ue_context_request_e>());
      break;
    case types::allowed_nssai:
      c.set(other.c.get<allowed_nssai_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "init_ue_msg_ies_o::value_c");
  }

  return *this;
}
uint64_t& init_ue_msg_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
unbounded_octstring<true>& init_ue_msg_ies_o::value_c::nas_pdu()
{
  assert_choice_type(types::nas_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
user_location_info_c& init_ue_msg_ies_o::value_c::user_location_info()
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_c>();
}
rrcestablishment_cause_e& init_ue_msg_ies_o::value_c::rrcestablishment_cause()
{
  assert_choice_type(types::rrcestablishment_cause, type_, "Value");
  return c.get<rrcestablishment_cause_e>();
}
five_g_s_tmsi_s& init_ue_msg_ies_o::value_c::five_g_s_tmsi()
{
  assert_choice_type(types::five_g_s_tmsi, type_, "Value");
  return c.get<five_g_s_tmsi_s>();
}
fixed_bitstring<10, false, true>& init_ue_msg_ies_o::value_c::amf_set_id()
{
  assert_choice_type(types::amf_set_id, type_, "Value");
  return c.get<fixed_bitstring<10, false, true> >();
}
ue_context_request_e& init_ue_msg_ies_o::value_c::ue_context_request()
{
  assert_choice_type(types::ue_context_request, type_, "Value");
  return c.get<ue_context_request_e>();
}
allowed_nssai_l& init_ue_msg_ies_o::value_c::allowed_nssai()
{
  assert_choice_type(types::allowed_nssai, type_, "Value");
  return c.get<allowed_nssai_l>();
}
const uint64_t& init_ue_msg_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const unbounded_octstring<true>& init_ue_msg_ies_o::value_c::nas_pdu() const
{
  assert_choice_type(types::nas_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const user_location_info_c& init_ue_msg_ies_o::value_c::user_location_info() const
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_c>();
}
const rrcestablishment_cause_e& init_ue_msg_ies_o::value_c::rrcestablishment_cause() const
{
  assert_choice_type(types::rrcestablishment_cause, type_, "Value");
  return c.get<rrcestablishment_cause_e>();
}
const five_g_s_tmsi_s& init_ue_msg_ies_o::value_c::five_g_s_tmsi() const
{
  assert_choice_type(types::five_g_s_tmsi, type_, "Value");
  return c.get<five_g_s_tmsi_s>();
}
const fixed_bitstring<10, false, true>& init_ue_msg_ies_o::value_c::amf_set_id() const
{
  assert_choice_type(types::amf_set_id, type_, "Value");
  return c.get<fixed_bitstring<10, false, true> >();
}
const ue_context_request_e& init_ue_msg_ies_o::value_c::ue_context_request() const
{
  assert_choice_type(types::ue_context_request, type_, "Value");
  return c.get<ue_context_request_e>();
}
const allowed_nssai_l& init_ue_msg_ies_o::value_c::allowed_nssai() const
{
  assert_choice_type(types::allowed_nssai, type_, "Value");
  return c.get<allowed_nssai_l>();
}
void init_ue_msg_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::nas_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::user_location_info:
      j.write_fieldname("UserLocationInformation");
      c.get<user_location_info_c>().to_json(j);
      break;
    case types::rrcestablishment_cause:
      j.write_str("RRCEstablishmentCause", c.get<rrcestablishment_cause_e>().to_string());
      break;
    case types::five_g_s_tmsi:
      j.write_fieldname("FiveG-S-TMSI");
      c.get<five_g_s_tmsi_s>().to_json(j);
      break;
    case types::amf_set_id:
      j.write_str("BIT STRING", c.get<fixed_bitstring<10, false, true> >().to_string());
      break;
    case types::ue_context_request:
      j.write_str("UEContextRequest", "requested");
      break;
    case types::allowed_nssai:
      j.start_array("AllowedNSSAI");
      for (const auto& e1 : c.get<allowed_nssai_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "init_ue_msg_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE init_ue_msg_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_c>().pack(bref));
      break;
    case types::rrcestablishment_cause:
      HANDLE_CODE(c.get<rrcestablishment_cause_e>().pack(bref));
      break;
    case types::five_g_s_tmsi:
      HANDLE_CODE(c.get<five_g_s_tmsi_s>().pack(bref));
      break;
    case types::amf_set_id:
      HANDLE_CODE((c.get<fixed_bitstring<10, false, true> >().pack(bref)));
      break;
    case types::ue_context_request:
      HANDLE_CODE(c.get<ue_context_request_e>().pack(bref));
      break;
    case types::allowed_nssai:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<allowed_nssai_l>(), 1, 8, true));
      break;
    default:
      log_invalid_choice_id(type_, "init_ue_msg_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE init_ue_msg_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_c>().unpack(bref));
      break;
    case types::rrcestablishment_cause:
      HANDLE_CODE(c.get<rrcestablishment_cause_e>().unpack(bref));
      break;
    case types::five_g_s_tmsi:
      HANDLE_CODE(c.get<five_g_s_tmsi_s>().unpack(bref));
      break;
    case types::amf_set_id:
      HANDLE_CODE((c.get<fixed_bitstring<10, false, true> >().unpack(bref)));
      break;
    case types::ue_context_request:
      HANDLE_CODE(c.get<ue_context_request_e>().unpack(bref));
      break;
    case types::allowed_nssai:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<allowed_nssai_l>(), bref, 1, 8, true));
      break;
    default:
      log_invalid_choice_id(type_, "init_ue_msg_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* init_ue_msg_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "OCTET STRING",
                                  "UserLocationInformation",
                                  "RRCEstablishmentCause",
                                  "FiveG-S-TMSI",
                                  "BIT STRING",
                                  "UEContextRequest",
                                  "AllowedNSSAI"};
  return convert_enum_idx(options, 8, value, "init_ue_msg_ies_o::value_c::types");
}
uint8_t init_ue_msg_ies_o::value_c::types_opts::to_number() const
{
  static const uint8_t options[] = {0};
  return map_enum_number(options, 1, value, "init_ue_msg_ies_o::value_c::types");
}

template struct asn1::protocol_ie_field_s<init_ue_msg_ies_o>;

init_ue_msg_ies_container::init_ue_msg_ies_container() :
  ran_ue_ngap_id(85, crit_e::reject),
  nas_pdu(38, crit_e::reject),
  user_location_info(121, crit_e::reject),
  rrcestablishment_cause(90, crit_e::ignore),
  five_g_s_tmsi(26, crit_e::reject),
  amf_set_id(3, crit_e::ignore),
  ue_context_request(112, crit_e::ignore),
  allowed_nssai(0, crit_e::reject)
{}
SRSASN_CODE init_ue_msg_ies_container::pack(bit_ref& bref) const
{
  uint32_t nof_ies = 4;
  nof_ies += five_g_s_tmsi_present ? 1 : 0;
  nof_ies += amf_set_id_present ? 1 : 0;
  nof_ies += ue_context_request_present ? 1 : 0;
  nof_ies += allowed_nssai_present ? 1 : 0;
  pack_length(bref, nof_ies, 0u, 65535u, true);

  HANDLE_CODE(ran_ue_ngap_id.pack(bref));
  HANDLE_CODE(nas_pdu.pack(bref));
  HANDLE_CODE(user_location_info.pack(bref));
  HANDLE_CODE(rrcestablishment_cause.pack(bref));
  if (five_g_s_tmsi_present) {
    HANDLE_CODE(five_g_s_tmsi.pack(bref));
  }
  if (amf_set_id_present) {
    HANDLE_CODE(amf_set_id.pack(bref));
  }
  if (ue_context_request_present) {
    HANDLE_CODE(ue_context_request.pack(bref));
  }
  if (allowed_nssai_present) {
    HANDLE_CODE(allowed_nssai.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE init_ue_msg_ies_container::unpack(cbit_ref& bref)
{
  uint32_t nof_ies = 0;
  unpack_length(nof_ies, bref, 0u, 65535u, true);

  uint32_t nof_mandatory_ies = 4;

  for (; nof_ies > 0; --nof_ies) {
    uint32_t id;
    HANDLE_CODE(unpack_integer(id, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
    switch (id) {
      case 85: {
        nof_mandatory_ies--;
        ran_ue_ngap_id.id = id;
        HANDLE_CODE(ran_ue_ngap_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ran_ue_ngap_id.value.unpack(bref));
        break;
      }
      case 38: {
        nof_mandatory_ies--;
        nas_pdu.id = id;
        HANDLE_CODE(nas_pdu.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(nas_pdu.value.unpack(bref));
        break;
      }
      case 121: {
        nof_mandatory_ies--;
        user_location_info.id = id;
        HANDLE_CODE(user_location_info.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(user_location_info.value.unpack(bref));
        break;
      }
      case 90: {
        nof_mandatory_ies--;
        rrcestablishment_cause.id = id;
        HANDLE_CODE(rrcestablishment_cause.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(rrcestablishment_cause.value.unpack(bref));
        break;
      }
      case 26: {
        five_g_s_tmsi_present = true;
        five_g_s_tmsi.id      = id;
        HANDLE_CODE(five_g_s_tmsi.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(five_g_s_tmsi.value.unpack(bref));
        break;
      }
      case 3: {
        amf_set_id_present = true;
        amf_set_id.id      = id;
        HANDLE_CODE(amf_set_id.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(amf_set_id.value.unpack(bref));
        break;
      }
      case 112: {
        ue_context_request_present = true;
        ue_context_request.id      = id;
        HANDLE_CODE(ue_context_request.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(ue_context_request.value.unpack(bref));
        break;
      }
      case 0: {
        allowed_nssai_present = true;
        allowed_nssai.id      = id;
        HANDLE_CODE(allowed_nssai.crit.unpack(bref));
        varlength_field_unpack_guard varlen_scope(bref, true);
        HANDLE_CODE(allowed_nssai.value.unpack(bref));
        break;
      }
      default:
        asn1::log_error("Unpacked object ID=%d is not recognized\n", id);
        return SRSASN_ERROR_DECODE_FAIL;
    }
  }
  if (nof_mandatory_ies > 0) {
    asn1::log_error("Mandatory fields are missing\n");

    return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
void init_ue_msg_ies_container::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("");
  ran_ue_ngap_id.to_json(j);
  j.write_fieldname("");
  nas_pdu.to_json(j);
  j.write_fieldname("");
  user_location_info.to_json(j);
  j.write_fieldname("");
  rrcestablishment_cause.to_json(j);
  if (five_g_s_tmsi_present) {
    j.write_fieldname("");
    five_g_s_tmsi.to_json(j);
  }
  if (amf_set_id_present) {
    j.write_fieldname("");
    amf_set_id.to_json(j);
  }
  if (ue_context_request_present) {
    j.write_fieldname("");
    ue_context_request.to_json(j);
  }
  if (allowed_nssai_present) {
    j.write_fieldname("");
    allowed_nssai.to_json(j);
  }
  j.end_obj();
}

// OverloadAction ::= ENUMERATED
const char* overload_action_opts::to_string() const
{
  static const char* options[] = {"reject-non-emergency-mo-dt",
                                  "reject-rrc-cr-signalling",
                                  "permit-emergency-sessions-and-mobile-terminated-services-only",
                                  "permit-high-priority-sessions-and-mobile-terminated-services-only"};
  return convert_enum_idx(options, 4, value, "overload_action_e");
}

// SliceOverloadItem ::= SEQUENCE
SRSASN_CODE slice_overload_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(s_nssai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE slice_overload_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(s_nssai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void slice_overload_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("s-NSSAI");
  s_nssai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// OverloadResponse ::= CHOICE
void overload_resp_c::destroy_()
{
  switch (type_) {
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<overload_resp_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void overload_resp_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::overload_action:
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<overload_resp_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "overload_resp_c");
  }
}
overload_resp_c::overload_resp_c(const overload_resp_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::overload_action:
      c.init(other.c.get<overload_action_e>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<overload_resp_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "overload_resp_c");
  }
}
overload_resp_c& overload_resp_c::operator=(const overload_resp_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::overload_action:
      c.set(other.c.get<overload_action_e>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<overload_resp_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "overload_resp_c");
  }

  return *this;
}
overload_action_e& overload_resp_c::set_overload_action()
{
  set(types::overload_action);
  return c.get<overload_action_e>();
}
protocol_ie_single_container_s<overload_resp_ext_ies_o>& overload_resp_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<overload_resp_ext_ies_o> >();
}
void overload_resp_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::overload_action:
      j.write_str("overloadAction", c.get<overload_action_e>().to_string());
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<overload_resp_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "overload_resp_c");
  }
  j.end_obj();
}
SRSASN_CODE overload_resp_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::overload_action:
      HANDLE_CODE(c.get<overload_action_e>().pack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<overload_resp_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "overload_resp_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE overload_resp_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::overload_action:
      HANDLE_CODE(c.get<overload_action_e>().unpack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<overload_resp_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "overload_resp_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* overload_resp_c::types_opts::to_string() const
{
  static const char* options[] = {"overloadAction", "choice-Extensions"};
  return convert_enum_idx(options, 2, value, "overload_resp_c::types");
}

// PDUSessionResourceItemCxtRelCpl-ExtIEs ::= OBJECT SET OF NGAP-PROTOCOL-EXTENSION
uint32_t pdu_session_res_item_cxt_rel_cpl_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {145};
  return map_enum_number(options, 1, idx, "id");
}
bool pdu_session_res_item_cxt_rel_cpl_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 145 == id;
}
crit_e pdu_session_res_item_cxt_rel_cpl_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 145) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
pdu_session_res_item_cxt_rel_cpl_ext_ies_o::ext_c
pdu_session_res_item_cxt_rel_cpl_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 145) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pdu_session_res_item_cxt_rel_cpl_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 145) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void pdu_session_res_item_cxt_rel_cpl_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("OCTET STRING", c.to_string());
  j.end_obj();
}
SRSASN_CODE pdu_session_res_item_cxt_rel_cpl_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_item_cxt_rel_cpl_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* pdu_session_res_item_cxt_rel_cpl_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"OCTET STRING"};
  return convert_enum_idx(options, 1, value, "pdu_session_res_item_cxt_rel_cpl_ext_ies_o::ext_c::types");
}

// PDUSessionResourceModifyItemModReq-ExtIEs ::= OBJECT SET OF NGAP-PROTOCOL-EXTENSION
uint32_t pdu_session_res_modify_item_mod_req_ext_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {148};
  return map_enum_number(options, 1, idx, "id");
}
bool pdu_session_res_modify_item_mod_req_ext_ies_o::is_id_valid(const uint32_t& id)
{
  return 148 == id;
}
crit_e pdu_session_res_modify_item_mod_req_ext_ies_o::get_crit(const uint32_t& id)
{
  if (id == 148) {
    return crit_e::reject;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
pdu_session_res_modify_item_mod_req_ext_ies_o::ext_c
pdu_session_res_modify_item_mod_req_ext_ies_o::get_ext(const uint32_t& id)
{
  ext_c ret{};
  if (id != 148) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pdu_session_res_modify_item_mod_req_ext_ies_o::get_presence(const uint32_t& id)
{
  if (id == 148) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Extension ::= OPEN TYPE
void pdu_session_res_modify_item_mod_req_ext_ies_o::ext_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("S-NSSAI");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE pdu_session_res_modify_item_mod_req_ext_ies_o::ext_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_modify_item_mod_req_ext_ies_o::ext_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* pdu_session_res_modify_item_mod_req_ext_ies_o::ext_c::types_opts::to_string() const
{
  static const char* options[] = {"S-NSSAI"};
  return convert_enum_idx(options, 1, value, "pdu_session_res_modify_item_mod_req_ext_ies_o::ext_c::types");
}

// PrivateIE-ID ::= CHOICE
void private_ie_id_c::set(types::options e)
{
  type_ = e;
}
uint32_t& private_ie_id_c::set_local()
{
  set(types::local);
  return c;
}
void private_ie_id_c::set_global()
{
  set(types::global);
}
void private_ie_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::local:
      j.write_int("local", c);
      break;
    case types::global:
      break;
    default:
      log_invalid_choice_id(type_, "private_ie_id_c");
  }
  j.end_obj();
}
SRSASN_CODE private_ie_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::local:
      HANDLE_CODE(pack_integer(bref, c, (uint32_t)0u, (uint32_t)65535u, false, true));
      break;
    case types::global:
      break;
    default:
      log_invalid_choice_id(type_, "private_ie_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE private_ie_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::local:
      HANDLE_CODE(unpack_integer(c, bref, (uint32_t)0u, (uint32_t)65535u, false, true));
      break;
    case types::global:
      break;
    default:
      log_invalid_choice_id(type_, "private_ie_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* private_ie_id_c::types_opts::to_string() const
{
  static const char* options[] = {"local", "global"};
  return convert_enum_idx(options, 2, value, "private_ie_id_c::types");
}

// UE-associatedLogicalNG-connectionItem ::= SEQUENCE
SRSASN_CODE ue_associated_lc_ng_conn_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(amf_ue_ngap_id_present, 1));
  HANDLE_CODE(bref.pack(ran_ue_ngap_id_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  if (amf_ue_ngap_id_present) {
    HANDLE_CODE(pack_integer(bref, amf_ue_ngap_id, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
  }
  if (ran_ue_ngap_id_present) {
    HANDLE_CODE(pack_integer(bref, ran_ue_ngap_id, (uint64_t)0u, (uint64_t)4294967295u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_associated_lc_ng_conn_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(amf_ue_ngap_id_present, 1));
  HANDLE_CODE(bref.unpack(ran_ue_ngap_id_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  if (amf_ue_ngap_id_present) {
    HANDLE_CODE(unpack_integer(amf_ue_ngap_id, bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
  }
  if (ran_ue_ngap_id_present) {
    HANDLE_CODE(unpack_integer(ran_ue_ngap_id, bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ue_associated_lc_ng_conn_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  if (amf_ue_ngap_id_present) {
    j.write_int("aMF-UE-NGAP-ID", amf_ue_ngap_id);
  }
  if (ran_ue_ngap_id_present) {
    j.write_int("rAN-UE-NGAP-ID", ran_ue_ngap_id);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UEPresence ::= ENUMERATED
const char* ue_presence_opts::to_string() const
{
  static const char* options[] = {"in", "out", "unknown"};
  return convert_enum_idx(options, 3, value, "ue_presence_e");
}

// OverloadStartNSSAIItem ::= SEQUENCE
SRSASN_CODE overload_start_nssai_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(slice_overload_resp_present, 1));
  HANDLE_CODE(bref.pack(slice_traffic_load_reduction_ind_present, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_dyn_seq_of(bref, slice_overload_list, 1, 1024, true));
  if (slice_overload_resp_present) {
    HANDLE_CODE(slice_overload_resp.pack(bref));
  }
  if (slice_traffic_load_reduction_ind_present) {
    HANDLE_CODE(pack_integer(bref, slice_traffic_load_reduction_ind, (uint8_t)1u, (uint8_t)99u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE overload_start_nssai_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(slice_overload_resp_present, 1));
  HANDLE_CODE(bref.unpack(slice_traffic_load_reduction_ind_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_dyn_seq_of(slice_overload_list, bref, 1, 1024, true));
  if (slice_overload_resp_present) {
    HANDLE_CODE(slice_overload_resp.unpack(bref));
  }
  if (slice_traffic_load_reduction_ind_present) {
    HANDLE_CODE(unpack_integer(slice_traffic_load_reduction_ind, bref, (uint8_t)1u, (uint8_t)99u, false, true));
  }
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void overload_start_nssai_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.start_array("sliceOverloadList");
  for (const auto& e1 : slice_overload_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (slice_overload_resp_present) {
    j.write_fieldname("sliceOverloadResponse");
    slice_overload_resp.to_json(j);
  }
  if (slice_traffic_load_reduction_ind_present) {
    j.write_int("sliceTrafficLoadReductionIndication", slice_traffic_load_reduction_ind);
  }
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceFailedToModifyItemModCfm ::= SEQUENCE
SRSASN_CODE pdu_session_res_failed_to_modify_item_mod_cfm_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_modify_ind_unsuccessful_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_failed_to_modify_item_mod_cfm_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_modify_ind_unsuccessful_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_failed_to_modify_item_mod_cfm_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pDUSessionResourceModifyIndicationUnsuccessfulTransfer",
              pdu_session_res_modify_ind_unsuccessful_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceFailedToModifyItemModRes ::= SEQUENCE
SRSASN_CODE pdu_session_res_failed_to_modify_item_mod_res_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_modify_unsuccessful_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_failed_to_modify_item_mod_res_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_modify_unsuccessful_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_failed_to_modify_item_mod_res_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pDUSessionResourceModifyUnsuccessfulTransfer", pdu_session_res_modify_unsuccessful_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceFailedToSetupItemPSReq ::= SEQUENCE
SRSASN_CODE pdu_session_res_failed_to_setup_item_ps_req_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(path_switch_request_setup_failed_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_failed_to_setup_item_ps_req_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(path_switch_request_setup_failed_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_failed_to_setup_item_ps_req_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pathSwitchRequestSetupFailedTransfer", path_switch_request_setup_failed_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceFailedToSetupItemSURes ::= SEQUENCE
SRSASN_CODE pdu_session_res_failed_to_setup_item_su_res_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_setup_unsuccessful_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_failed_to_setup_item_su_res_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_setup_unsuccessful_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_failed_to_setup_item_su_res_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pDUSessionResourceSetupUnsuccessfulTransfer", pdu_session_res_setup_unsuccessful_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceItemCxtRelCpl ::= SEQUENCE
SRSASN_CODE pdu_session_res_item_cxt_rel_cpl_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts.size() > 0, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  if (ie_exts.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_item_cxt_rel_cpl_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  bool ie_exts_present;
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_item_cxt_rel_cpl_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  if (ie_exts.size() > 0) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// PDUSessionResourceItemCxtRelReq ::= SEQUENCE
SRSASN_CODE pdu_session_res_item_cxt_rel_req_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_item_cxt_rel_req_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_item_cxt_rel_req_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceModifyItemModCfm ::= SEQUENCE
SRSASN_CODE pdu_session_res_modify_item_mod_cfm_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_modify_confirm_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_modify_item_mod_cfm_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_modify_confirm_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_modify_item_mod_cfm_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pDUSessionResourceModifyConfirmTransfer", pdu_session_res_modify_confirm_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceModifyItemModInd ::= SEQUENCE
SRSASN_CODE pdu_session_res_modify_item_mod_ind_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_modify_ind_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_modify_item_mod_ind_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_modify_ind_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_modify_item_mod_ind_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pDUSessionResourceModifyIndicationTransfer", pdu_session_res_modify_ind_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceModifyItemModReq ::= SEQUENCE
SRSASN_CODE pdu_session_res_modify_item_mod_req_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(nas_pdu.size() > 0, 1));
  HANDLE_CODE(bref.pack(ie_exts.size() > 0, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  if (nas_pdu.size() > 0) {
    HANDLE_CODE(nas_pdu.pack(bref));
  }
  HANDLE_CODE(pdu_session_res_modify_request_transfer.pack(bref));
  if (ie_exts.size() > 0) {
    HANDLE_CODE(pack_dyn_seq_of(bref, ie_exts, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_modify_item_mod_req_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  bool nas_pdu_present;
  HANDLE_CODE(bref.unpack(nas_pdu_present, 1));
  bool ie_exts_present;
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  if (nas_pdu_present) {
    HANDLE_CODE(nas_pdu.unpack(bref));
  }
  HANDLE_CODE(pdu_session_res_modify_request_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(unpack_dyn_seq_of(ie_exts, bref, 1, 65535, true));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_modify_item_mod_req_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  if (nas_pdu.size() > 0) {
    j.write_str("nAS-PDU", nas_pdu.to_string());
  }
  j.write_str("pDUSessionResourceModifyRequestTransfer", pdu_session_res_modify_request_transfer.to_string());
  if (ie_exts.size() > 0) {
    j.write_fieldname("iE-Extensions");
  }
  j.end_obj();
}

// PDUSessionResourceModifyItemModRes ::= SEQUENCE
SRSASN_CODE pdu_session_res_modify_item_mod_res_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_modify_resp_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_modify_item_mod_res_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_modify_resp_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_modify_item_mod_res_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pDUSessionResourceModifyResponseTransfer", pdu_session_res_modify_resp_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceNotifyItem ::= SEQUENCE
SRSASN_CODE pdu_session_res_notify_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_notify_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_notify_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_notify_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_notify_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pDUSessionResourceNotifyTransfer", pdu_session_res_notify_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceReleasedItemNot ::= SEQUENCE
SRSASN_CODE pdu_session_res_released_item_not_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_notify_released_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_released_item_not_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_notify_released_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_released_item_not_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pDUSessionResourceNotifyReleasedTransfer", pdu_session_res_notify_released_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceReleasedItemPSAck ::= SEQUENCE
SRSASN_CODE pdu_session_res_released_item_ps_ack_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(path_switch_request_unsuccessful_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_released_item_ps_ack_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(path_switch_request_unsuccessful_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_released_item_ps_ack_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pathSwitchRequestUnsuccessfulTransfer", path_switch_request_unsuccessful_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceReleasedItemPSFail ::= SEQUENCE
SRSASN_CODE pdu_session_res_released_item_ps_fail_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(path_switch_request_unsuccessful_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_released_item_ps_fail_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(path_switch_request_unsuccessful_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_released_item_ps_fail_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pathSwitchRequestUnsuccessfulTransfer", path_switch_request_unsuccessful_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceReleasedItemRelRes ::= SEQUENCE
SRSASN_CODE pdu_session_res_released_item_rel_res_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_release_resp_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_released_item_rel_res_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_release_resp_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_released_item_rel_res_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pDUSessionResourceReleaseResponseTransfer", pdu_session_res_release_resp_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceSecondaryRATUsageItem ::= SEQUENCE
SRSASN_CODE pdu_session_res_secondary_ratusage_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(secondary_rat_data_usage_report_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_secondary_ratusage_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(secondary_rat_data_usage_report_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_secondary_ratusage_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("secondaryRATDataUsageReportTransfer", secondary_rat_data_usage_report_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceSetupItemSUReq ::= SEQUENCE
SRSASN_CODE pdu_session_res_setup_item_su_req_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(pdu_session_nas_pdu.size() > 0, 1));
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  if (pdu_session_nas_pdu.size() > 0) {
    HANDLE_CODE(pdu_session_nas_pdu.pack(bref));
  }
  HANDLE_CODE(s_nssai.pack(bref));
  HANDLE_CODE(pdu_session_res_setup_request_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_setup_item_su_req_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  bool pdu_session_nas_pdu_present;
  HANDLE_CODE(bref.unpack(pdu_session_nas_pdu_present, 1));
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  if (pdu_session_nas_pdu_present) {
    HANDLE_CODE(pdu_session_nas_pdu.unpack(bref));
  }
  HANDLE_CODE(s_nssai.unpack(bref));
  HANDLE_CODE(pdu_session_res_setup_request_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_setup_item_su_req_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  if (pdu_session_nas_pdu.size() > 0) {
    j.write_str("pDUSessionNAS-PDU", pdu_session_nas_pdu.to_string());
  }
  j.write_fieldname("s-NSSAI");
  s_nssai.to_json(j);
  j.write_str("pDUSessionResourceSetupRequestTransfer", pdu_session_res_setup_request_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceSetupItemSURes ::= SEQUENCE
SRSASN_CODE pdu_session_res_setup_item_su_res_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_setup_resp_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_setup_item_su_res_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_setup_resp_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_setup_item_su_res_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pDUSessionResourceSetupResponseTransfer", pdu_session_res_setup_resp_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceSwitchedItem ::= SEQUENCE
SRSASN_CODE pdu_session_res_switched_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(path_switch_request_ack_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_switched_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(path_switch_request_ack_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_switched_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pathSwitchRequestAcknowledgeTransfer", path_switch_request_ack_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceToBeSwitchedDLItem ::= SEQUENCE
SRSASN_CODE pdu_session_res_to_be_switched_dl_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(path_switch_request_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_to_be_switched_dl_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(path_switch_request_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_to_be_switched_dl_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pathSwitchRequestTransfer", path_switch_request_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// PDUSessionResourceToReleaseItemRelCmd ::= SEQUENCE
SRSASN_CODE pdu_session_res_to_release_item_rel_cmd_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, pdu_session_id, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_release_cmd_transfer.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_to_release_item_rel_cmd_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(pdu_session_id, bref, (uint16_t)0u, (uint16_t)255u, false, true));
  HANDLE_CODE(pdu_session_res_release_cmd_transfer.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void pdu_session_res_to_release_item_rel_cmd_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("pDUSessionID", pdu_session_id);
  j.write_str("pDUSessionResourceReleaseCommandTransfer", pdu_session_res_release_cmd_transfer.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// ResetAll ::= ENUMERATED
const char* reset_all_opts::to_string() const
{
  static const char* options[] = {"reset-all"};
  return convert_enum_idx(options, 1, value, "reset_all_e");
}

// SupportedTAItem ::= SEQUENCE
SRSASN_CODE supported_ta_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(tac.pack(bref));
  HANDLE_CODE(pack_dyn_seq_of(bref, broadcast_plmn_list, 1, 12, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE supported_ta_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(tac.unpack(bref));
  HANDLE_CODE(unpack_dyn_seq_of(broadcast_plmn_list, bref, 1, 12, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void supported_ta_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_str("tAC", tac.to_string());
  j.start_array("broadcastPLMNList");
  for (const auto& e1 : broadcast_plmn_list) {
    e1.to_json(j);
  }
  j.end_array();
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// TAIListForPagingItem ::= SEQUENCE
SRSASN_CODE tai_list_for_paging_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(tai.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE tai_list_for_paging_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(tai.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void tai_list_for_paging_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("tAI");
  tai.to_json(j);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UE-NGAP-ID-pair ::= SEQUENCE
SRSASN_CODE ue_ngap_id_pair_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, amf_ue_ngap_id, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
  HANDLE_CODE(pack_integer(bref, ran_ue_ngap_id, (uint64_t)0u, (uint64_t)4294967295u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_ngap_id_pair_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(amf_ue_ngap_id, bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
  HANDLE_CODE(unpack_integer(ran_ue_ngap_id, bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ue_ngap_id_pair_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("aMF-UE-NGAP-ID", amf_ue_ngap_id);
  j.write_int("rAN-UE-NGAP-ID", ran_ue_ngap_id);
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// UEPresenceInAreaOfInterestItem ::= SEQUENCE
SRSASN_CODE ue_presence_in_area_of_interest_item_s::pack(bit_ref& bref) const
{
  bref.pack(ext, 1);
  HANDLE_CODE(bref.pack(ie_exts_present, 1));

  HANDLE_CODE(pack_integer(bref, location_report_ref_id, (uint8_t)1u, (uint8_t)64u, true, true));
  HANDLE_CODE(uepresence.pack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.pack(bref));
  }

  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_presence_in_area_of_interest_item_s::unpack(cbit_ref& bref)
{
  bref.unpack(ext, 1);
  HANDLE_CODE(bref.unpack(ie_exts_present, 1));

  HANDLE_CODE(unpack_integer(location_report_ref_id, bref, (uint8_t)1u, (uint8_t)64u, true, true));
  HANDLE_CODE(uepresence.unpack(bref));
  if (ie_exts_present) {
    HANDLE_CODE(ie_exts.unpack(bref));
  }

  return SRSASN_SUCCESS;
}
void ue_presence_in_area_of_interest_item_s::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_int("locationReportingReferenceID", location_report_ref_id);
  j.write_str("uEPresence", uepresence.to_string());
  if (ie_exts_present) {
    j.write_fieldname("iE-Extensions");
    ie_exts.to_json(j);
  }
  j.end_obj();
}

// CancelAllWarningMessages ::= ENUMERATED
const char* cancel_all_warning_msgs_opts::to_string() const
{
  static const char* options[] = {"true"};
  return convert_enum_idx(options, 1, value, "cancel_all_warning_msgs_e");
}

// ConcurrentWarningMessageInd ::= ENUMERATED
const char* concurrent_warning_msg_ind_opts::to_string() const
{
  static const char* options[] = {"true"};
  return convert_enum_idx(options, 1, value, "concurrent_warning_msg_ind_e");
}

// HandoverFlag ::= ENUMERATED
const char* ho_flag_opts::to_string() const
{
  static const char* options[] = {"handover-preparation"};
  return convert_enum_idx(options, 1, value, "ho_flag_e");
}

// IMSVoiceSupportIndicator ::= ENUMERATED
const char* ims_voice_support_ind_opts::to_string() const
{
  static const char* options[] = {"supported", "not-supported"};
  return convert_enum_idx(options, 2, value, "ims_voice_support_ind_e");
}

// PWSFailedCellIDList ::= CHOICE
void pws_failed_cell_id_list_c::destroy_()
{
  switch (type_) {
    case types::eutra_cgi_pws_failed_list:
      c.destroy<eutra_cgi_list_l>();
      break;
    case types::nr_cgi_pws_failed_list:
      c.destroy<nr_cgi_list_l>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<pws_failed_cell_id_list_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void pws_failed_cell_id_list_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::eutra_cgi_pws_failed_list:
      c.init<eutra_cgi_list_l>();
      break;
    case types::nr_cgi_pws_failed_list:
      c.init<nr_cgi_list_l>();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<pws_failed_cell_id_list_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pws_failed_cell_id_list_c");
  }
}
pws_failed_cell_id_list_c::pws_failed_cell_id_list_c(const pws_failed_cell_id_list_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::eutra_cgi_pws_failed_list:
      c.init(other.c.get<eutra_cgi_list_l>());
      break;
    case types::nr_cgi_pws_failed_list:
      c.init(other.c.get<nr_cgi_list_l>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<pws_failed_cell_id_list_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pws_failed_cell_id_list_c");
  }
}
pws_failed_cell_id_list_c& pws_failed_cell_id_list_c::operator=(const pws_failed_cell_id_list_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::eutra_cgi_pws_failed_list:
      c.set(other.c.get<eutra_cgi_list_l>());
      break;
    case types::nr_cgi_pws_failed_list:
      c.set(other.c.get<nr_cgi_list_l>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<pws_failed_cell_id_list_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pws_failed_cell_id_list_c");
  }

  return *this;
}
eutra_cgi_list_l& pws_failed_cell_id_list_c::set_eutra_cgi_pws_failed_list()
{
  set(types::eutra_cgi_pws_failed_list);
  return c.get<eutra_cgi_list_l>();
}
nr_cgi_list_l& pws_failed_cell_id_list_c::set_nr_cgi_pws_failed_list()
{
  set(types::nr_cgi_pws_failed_list);
  return c.get<nr_cgi_list_l>();
}
protocol_ie_single_container_s<pws_failed_cell_id_list_ext_ies_o>& pws_failed_cell_id_list_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<pws_failed_cell_id_list_ext_ies_o> >();
}
void pws_failed_cell_id_list_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::eutra_cgi_pws_failed_list:
      j.start_array("eUTRA-CGI-PWSFailedList");
      for (const auto& e1 : c.get<eutra_cgi_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::nr_cgi_pws_failed_list:
      j.start_array("nR-CGI-PWSFailedList");
      for (const auto& e1 : c.get<nr_cgi_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<pws_failed_cell_id_list_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "pws_failed_cell_id_list_c");
  }
  j.end_obj();
}
SRSASN_CODE pws_failed_cell_id_list_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::eutra_cgi_pws_failed_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<eutra_cgi_list_l>(), 1, 256, true));
      break;
    case types::nr_cgi_pws_failed_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<nr_cgi_list_l>(), 1, 16384, true));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<pws_failed_cell_id_list_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pws_failed_cell_id_list_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pws_failed_cell_id_list_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::eutra_cgi_pws_failed_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<eutra_cgi_list_l>(), bref, 1, 256, true));
      break;
    case types::nr_cgi_pws_failed_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<nr_cgi_list_l>(), bref, 1, 16384, true));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<pws_failed_cell_id_list_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pws_failed_cell_id_list_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* pws_failed_cell_id_list_c::types_opts::to_string() const
{
  static const char* options[] = {"eUTRA-CGI-PWSFailedList", "nR-CGI-PWSFailedList", "choice-Extensions"};
  return convert_enum_idx(options, 3, value, "pws_failed_cell_id_list_c::types");
}

// PagingOrigin ::= ENUMERATED
const char* paging_origin_opts::to_string() const
{
  static const char* options[] = {"non-3gpp"};
  return convert_enum_idx(options, 1, value, "paging_origin_e");
}
int8_t paging_origin_opts::to_number() const
{
  static const int8_t options[] = {-3};
  return map_enum_number(options, 1, value, "paging_origin_e");
}

// PagingPriority ::= ENUMERATED
const char* paging_prio_opts::to_string() const
{
  static const char* options[] = {
      "priolevel1", "priolevel2", "priolevel3", "priolevel4", "priolevel5", "priolevel6", "priolevel7", "priolevel8"};
  return convert_enum_idx(options, 8, value, "paging_prio_e");
}
uint8_t paging_prio_opts::to_number() const
{
  static const uint8_t options[] = {1, 2, 3, 4, 5, 6, 7, 8};
  return map_enum_number(options, 8, value, "paging_prio_e");
}

// PrivateIE-Field{NGAP-PRIVATE-IES : IEsSetParam} ::= SEQUENCE{{NGAP-PRIVATE-IES}}
template <class ies_set_paramT_>
SRSASN_CODE private_ie_field_s<ies_set_paramT_>::pack(bit_ref& bref) const
{
  HANDLE_CODE(id.pack(bref));
  HANDLE_CODE(crit.pack(bref));
  HANDLE_CODE(value.pack(bref));

  return SRSASN_SUCCESS;
}
template <class ies_set_paramT_>
SRSASN_CODE private_ie_field_s<ies_set_paramT_>::unpack(cbit_ref& bref)
{
  HANDLE_CODE(id.unpack(bref));
  HANDLE_CODE(crit.unpack(bref));
  HANDLE_CODE(value.unpack(bref));

  return SRSASN_SUCCESS;
}
template <class ies_set_paramT_>
void private_ie_field_s<ies_set_paramT_>::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("id");
  id.to_json(j);
  j.write_str("criticality", crit.to_string());
  j.end_obj();
}

// RRCState ::= ENUMERATED
const char* rrc_state_opts::to_string() const
{
  static const char* options[] = {"inactive", "connected"};
  return convert_enum_idx(options, 2, value, "rrc_state_e");
}

// ResetType ::= CHOICE
void reset_type_c::destroy_()
{
  switch (type_) {
    case types::part_of_ng_interface:
      c.destroy<ue_associated_lc_ng_conn_list_l>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<reset_type_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void reset_type_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::ng_interface:
      break;
    case types::part_of_ng_interface:
      c.init<ue_associated_lc_ng_conn_list_l>();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<reset_type_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "reset_type_c");
  }
}
reset_type_c::reset_type_c(const reset_type_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::ng_interface:
      c.init(other.c.get<reset_all_e>());
      break;
    case types::part_of_ng_interface:
      c.init(other.c.get<ue_associated_lc_ng_conn_list_l>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<reset_type_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "reset_type_c");
  }
}
reset_type_c& reset_type_c::operator=(const reset_type_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::ng_interface:
      c.set(other.c.get<reset_all_e>());
      break;
    case types::part_of_ng_interface:
      c.set(other.c.get<ue_associated_lc_ng_conn_list_l>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<reset_type_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "reset_type_c");
  }

  return *this;
}
reset_all_e& reset_type_c::set_ng_interface()
{
  set(types::ng_interface);
  return c.get<reset_all_e>();
}
ue_associated_lc_ng_conn_list_l& reset_type_c::set_part_of_ng_interface()
{
  set(types::part_of_ng_interface);
  return c.get<ue_associated_lc_ng_conn_list_l>();
}
protocol_ie_single_container_s<reset_type_ext_ies_o>& reset_type_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<reset_type_ext_ies_o> >();
}
void reset_type_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::ng_interface:
      j.write_str("nG-Interface", "reset-all");
      break;
    case types::part_of_ng_interface:
      j.start_array("partOfNG-Interface");
      for (const auto& e1 : c.get<ue_associated_lc_ng_conn_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<reset_type_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "reset_type_c");
  }
  j.end_obj();
}
SRSASN_CODE reset_type_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::ng_interface:
      HANDLE_CODE(c.get<reset_all_e>().pack(bref));
      break;
    case types::part_of_ng_interface:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<ue_associated_lc_ng_conn_list_l>(), 1, 65536, true));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<reset_type_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "reset_type_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE reset_type_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::ng_interface:
      HANDLE_CODE(c.get<reset_all_e>().unpack(bref));
      break;
    case types::part_of_ng_interface:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<ue_associated_lc_ng_conn_list_l>(), bref, 1, 65536, true));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<reset_type_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "reset_type_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* reset_type_c::types_opts::to_string() const
{
  static const char* options[] = {"nG-Interface", "partOfNG-Interface", "choice-Extensions"};
  return convert_enum_idx(options, 3, value, "reset_type_c::types");
}

// UE-NGAP-IDs ::= CHOICE
void ue_ngap_ids_c::destroy_()
{
  switch (type_) {
    case types::ue_ngap_id_pair:
      c.destroy<ue_ngap_id_pair_s>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<ue_ngap_ids_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void ue_ngap_ids_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::ue_ngap_id_pair:
      c.init<ue_ngap_id_pair_s>();
      break;
    case types::amf_ue_ngap_id:
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<ue_ngap_ids_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_ngap_ids_c");
  }
}
ue_ngap_ids_c::ue_ngap_ids_c(const ue_ngap_ids_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::ue_ngap_id_pair:
      c.init(other.c.get<ue_ngap_id_pair_s>());
      break;
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<ue_ngap_ids_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_ngap_ids_c");
  }
}
ue_ngap_ids_c& ue_ngap_ids_c::operator=(const ue_ngap_ids_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::ue_ngap_id_pair:
      c.set(other.c.get<ue_ngap_id_pair_s>());
      break;
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<ue_ngap_ids_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_ngap_ids_c");
  }

  return *this;
}
ue_ngap_id_pair_s& ue_ngap_ids_c::set_ue_ngap_id_pair()
{
  set(types::ue_ngap_id_pair);
  return c.get<ue_ngap_id_pair_s>();
}
uint64_t& ue_ngap_ids_c::set_amf_ue_ngap_id()
{
  set(types::amf_ue_ngap_id);
  return c.get<uint64_t>();
}
protocol_ie_single_container_s<ue_ngap_ids_ext_ies_o>& ue_ngap_ids_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<ue_ngap_ids_ext_ies_o> >();
}
void ue_ngap_ids_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::ue_ngap_id_pair:
      j.write_fieldname("uE-NGAP-ID-pair");
      c.get<ue_ngap_id_pair_s>().to_json(j);
      break;
    case types::amf_ue_ngap_id:
      j.write_int("aMF-UE-NGAP-ID", c.get<uint64_t>());
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<ue_ngap_ids_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ue_ngap_ids_c");
  }
  j.end_obj();
}
SRSASN_CODE ue_ngap_ids_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::ue_ngap_id_pair:
      HANDLE_CODE(c.get<ue_ngap_id_pair_s>().pack(bref));
      break;
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<ue_ngap_ids_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ue_ngap_ids_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_ngap_ids_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::ue_ngap_id_pair:
      HANDLE_CODE(c.get<ue_ngap_id_pair_s>().unpack(bref));
      break;
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<ue_ngap_ids_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ue_ngap_ids_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ue_ngap_ids_c::types_opts::to_string() const
{
  static const char* options[] = {"uE-NGAP-ID-pair", "aMF-UE-NGAP-ID", "choice-Extensions"};
  return convert_enum_idx(options, 3, value, "ue_ngap_ids_c::types");
}

// UEPagingIdentity ::= CHOICE
void ue_paging_id_c::destroy_()
{
  switch (type_) {
    case types::five_g_s_tmsi:
      c.destroy<five_g_s_tmsi_s>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<ue_paging_id_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void ue_paging_id_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::five_g_s_tmsi:
      c.init<five_g_s_tmsi_s>();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<ue_paging_id_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_paging_id_c");
  }
}
ue_paging_id_c::ue_paging_id_c(const ue_paging_id_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::five_g_s_tmsi:
      c.init(other.c.get<five_g_s_tmsi_s>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<ue_paging_id_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_paging_id_c");
  }
}
ue_paging_id_c& ue_paging_id_c::operator=(const ue_paging_id_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::five_g_s_tmsi:
      c.set(other.c.get<five_g_s_tmsi_s>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<ue_paging_id_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ue_paging_id_c");
  }

  return *this;
}
five_g_s_tmsi_s& ue_paging_id_c::set_five_g_s_tmsi()
{
  set(types::five_g_s_tmsi);
  return c.get<five_g_s_tmsi_s>();
}
protocol_ie_single_container_s<ue_paging_id_ext_ies_o>& ue_paging_id_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<ue_paging_id_ext_ies_o> >();
}
void ue_paging_id_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::five_g_s_tmsi:
      j.write_fieldname("fiveG-S-TMSI");
      c.get<five_g_s_tmsi_s>().to_json(j);
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<ue_paging_id_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ue_paging_id_c");
  }
  j.end_obj();
}
SRSASN_CODE ue_paging_id_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::five_g_s_tmsi:
      HANDLE_CODE(c.get<five_g_s_tmsi_s>().pack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<ue_paging_id_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ue_paging_id_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ue_paging_id_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::five_g_s_tmsi:
      HANDLE_CODE(c.get<five_g_s_tmsi_s>().unpack(bref));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<ue_paging_id_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ue_paging_id_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ue_paging_id_c::types_opts::to_string() const
{
  static const char* options[] = {"fiveG-S-TMSI", "choice-Extensions"};
  return convert_enum_idx(options, 2, value, "ue_paging_id_c::types");
}
uint8_t ue_paging_id_c::types_opts::to_number() const
{
  static const uint8_t options[] = {5};
  return map_enum_number(options, 1, value, "ue_paging_id_c::types");
}

// UERetentionInformation ::= ENUMERATED
const char* ue_retention_info_opts::to_string() const
{
  static const char* options[] = {"ues-retained"};
  return convert_enum_idx(options, 1, value, "ue_retention_info_e");
}

// WarningAreaList ::= CHOICE
void warning_area_list_c::destroy_()
{
  switch (type_) {
    case types::eutra_cgi_list_for_warning:
      c.destroy<eutra_cgi_list_for_warning_l>();
      break;
    case types::nr_cgi_list_for_warning:
      c.destroy<nr_cgi_list_for_warning_l>();
      break;
    case types::tai_list_for_warning:
      c.destroy<tai_list_for_warning_l>();
      break;
    case types::emergency_area_id_list:
      c.destroy<emergency_area_id_list_l>();
      break;
    case types::choice_exts:
      c.destroy<protocol_ie_single_container_s<warning_area_list_ext_ies_o> >();
      break;
    default:
      break;
  }
}
void warning_area_list_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::eutra_cgi_list_for_warning:
      c.init<eutra_cgi_list_for_warning_l>();
      break;
    case types::nr_cgi_list_for_warning:
      c.init<nr_cgi_list_for_warning_l>();
      break;
    case types::tai_list_for_warning:
      c.init<tai_list_for_warning_l>();
      break;
    case types::emergency_area_id_list:
      c.init<emergency_area_id_list_l>();
      break;
    case types::choice_exts:
      c.init<protocol_ie_single_container_s<warning_area_list_ext_ies_o> >();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "warning_area_list_c");
  }
}
warning_area_list_c::warning_area_list_c(const warning_area_list_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::eutra_cgi_list_for_warning:
      c.init(other.c.get<eutra_cgi_list_for_warning_l>());
      break;
    case types::nr_cgi_list_for_warning:
      c.init(other.c.get<nr_cgi_list_for_warning_l>());
      break;
    case types::tai_list_for_warning:
      c.init(other.c.get<tai_list_for_warning_l>());
      break;
    case types::emergency_area_id_list:
      c.init(other.c.get<emergency_area_id_list_l>());
      break;
    case types::choice_exts:
      c.init(other.c.get<protocol_ie_single_container_s<warning_area_list_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "warning_area_list_c");
  }
}
warning_area_list_c& warning_area_list_c::operator=(const warning_area_list_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::eutra_cgi_list_for_warning:
      c.set(other.c.get<eutra_cgi_list_for_warning_l>());
      break;
    case types::nr_cgi_list_for_warning:
      c.set(other.c.get<nr_cgi_list_for_warning_l>());
      break;
    case types::tai_list_for_warning:
      c.set(other.c.get<tai_list_for_warning_l>());
      break;
    case types::emergency_area_id_list:
      c.set(other.c.get<emergency_area_id_list_l>());
      break;
    case types::choice_exts:
      c.set(other.c.get<protocol_ie_single_container_s<warning_area_list_ext_ies_o> >());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "warning_area_list_c");
  }

  return *this;
}
eutra_cgi_list_for_warning_l& warning_area_list_c::set_eutra_cgi_list_for_warning()
{
  set(types::eutra_cgi_list_for_warning);
  return c.get<eutra_cgi_list_for_warning_l>();
}
nr_cgi_list_for_warning_l& warning_area_list_c::set_nr_cgi_list_for_warning()
{
  set(types::nr_cgi_list_for_warning);
  return c.get<nr_cgi_list_for_warning_l>();
}
tai_list_for_warning_l& warning_area_list_c::set_tai_list_for_warning()
{
  set(types::tai_list_for_warning);
  return c.get<tai_list_for_warning_l>();
}
emergency_area_id_list_l& warning_area_list_c::set_emergency_area_id_list()
{
  set(types::emergency_area_id_list);
  return c.get<emergency_area_id_list_l>();
}
protocol_ie_single_container_s<warning_area_list_ext_ies_o>& warning_area_list_c::set_choice_exts()
{
  set(types::choice_exts);
  return c.get<protocol_ie_single_container_s<warning_area_list_ext_ies_o> >();
}
void warning_area_list_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::eutra_cgi_list_for_warning:
      j.start_array("eUTRA-CGIListForWarning");
      for (const auto& e1 : c.get<eutra_cgi_list_for_warning_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::nr_cgi_list_for_warning:
      j.start_array("nR-CGIListForWarning");
      for (const auto& e1 : c.get<nr_cgi_list_for_warning_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::tai_list_for_warning:
      j.start_array("tAIListForWarning");
      for (const auto& e1 : c.get<tai_list_for_warning_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::emergency_area_id_list:
      j.start_array("emergencyAreaIDList");
      for (const auto& e1 : c.get<emergency_area_id_list_l>()) {
        j.write_str(e1.to_string());
      }
      j.end_array();
      break;
    case types::choice_exts:
      j.write_fieldname("choice-Extensions");
      c.get<protocol_ie_single_container_s<warning_area_list_ext_ies_o> >().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "warning_area_list_c");
  }
  j.end_obj();
}
SRSASN_CODE warning_area_list_c::pack(bit_ref& bref) const
{
  type_.pack(bref);
  switch (type_) {
    case types::eutra_cgi_list_for_warning:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<eutra_cgi_list_for_warning_l>(), 1, 65535, true));
      break;
    case types::nr_cgi_list_for_warning:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<nr_cgi_list_for_warning_l>(), 1, 65535, true));
      break;
    case types::tai_list_for_warning:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<tai_list_for_warning_l>(), 1, 65535, true));
      break;
    case types::emergency_area_id_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<emergency_area_id_list_l>(), 1, 65535, true));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<warning_area_list_ext_ies_o> >().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "warning_area_list_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE warning_area_list_c::unpack(cbit_ref& bref)
{
  types e;
  e.unpack(bref);
  set(e);
  switch (type_) {
    case types::eutra_cgi_list_for_warning:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<eutra_cgi_list_for_warning_l>(), bref, 1, 65535, true));
      break;
    case types::nr_cgi_list_for_warning:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<nr_cgi_list_for_warning_l>(), bref, 1, 65535, true));
      break;
    case types::tai_list_for_warning:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<tai_list_for_warning_l>(), bref, 1, 65535, true));
      break;
    case types::emergency_area_id_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<emergency_area_id_list_l>(), bref, 1, 65535, true));
      break;
    case types::choice_exts:
      HANDLE_CODE(c.get<protocol_ie_single_container_s<warning_area_list_ext_ies_o> >().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "warning_area_list_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* warning_area_list_c::types_opts::to_string() const
{
  static const char* options[] = {"eUTRA-CGIListForWarning",
                                  "nR-CGIListForWarning",
                                  "tAIListForWarning",
                                  "emergencyAreaIDList",
                                  "choice-Extensions"};
  return convert_enum_idx(options, 5, value, "warning_area_list_c::types");
}

// LocationReportIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t location_report_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 121, 116, 33, 149};
  return map_enum_number(options, 6, idx, "id");
}
bool location_report_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 121, 116, 33, 149};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e location_report_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 121:
      return crit_e::ignore;
    case 116:
      return crit_e::ignore;
    case 33:
      return crit_e::ignore;
    case 149:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
location_report_ies_o::value_c location_report_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 121:
      ret.set(value_c::types::user_location_info);
      break;
    case 116:
      ret.set(value_c::types::ue_presence_in_area_of_interest_list);
      break;
    case 33:
      ret.set(value_c::types::location_report_request_type);
      break;
    case 149:
      ret.set(value_c::types::ps_cell_info);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e location_report_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 121:
      return presence_e::mandatory;
    case 116:
      return presence_e::optional;
    case 33:
      return presence_e::mandatory;
    case 149:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void location_report_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::user_location_info:
      c.destroy<user_location_info_c>();
      break;
    case types::ue_presence_in_area_of_interest_list:
      c.destroy<ue_presence_in_area_of_interest_list_l>();
      break;
    case types::location_report_request_type:
      c.destroy<location_report_request_type_s>();
      break;
    case types::ps_cell_info:
      c.destroy<ngran_cgi_c>();
      break;
    default:
      break;
  }
}
void location_report_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::user_location_info:
      c.init<user_location_info_c>();
      break;
    case types::ue_presence_in_area_of_interest_list:
      c.init<ue_presence_in_area_of_interest_list_l>();
      break;
    case types::location_report_request_type:
      c.init<location_report_request_type_s>();
      break;
    case types::ps_cell_info:
      c.init<ngran_cgi_c>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "location_report_ies_o::value_c");
  }
}
location_report_ies_o::value_c::value_c(const location_report_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::user_location_info:
      c.init(other.c.get<user_location_info_c>());
      break;
    case types::ue_presence_in_area_of_interest_list:
      c.init(other.c.get<ue_presence_in_area_of_interest_list_l>());
      break;
    case types::location_report_request_type:
      c.init(other.c.get<location_report_request_type_s>());
      break;
    case types::ps_cell_info:
      c.init(other.c.get<ngran_cgi_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "location_report_ies_o::value_c");
  }
}
location_report_ies_o::value_c& location_report_ies_o::value_c::operator=(const location_report_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::user_location_info:
      c.set(other.c.get<user_location_info_c>());
      break;
    case types::ue_presence_in_area_of_interest_list:
      c.set(other.c.get<ue_presence_in_area_of_interest_list_l>());
      break;
    case types::location_report_request_type:
      c.set(other.c.get<location_report_request_type_s>());
      break;
    case types::ps_cell_info:
      c.set(other.c.get<ngran_cgi_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "location_report_ies_o::value_c");
  }

  return *this;
}
uint64_t& location_report_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& location_report_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
user_location_info_c& location_report_ies_o::value_c::user_location_info()
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_c>();
}
ue_presence_in_area_of_interest_list_l& location_report_ies_o::value_c::ue_presence_in_area_of_interest_list()
{
  assert_choice_type(types::ue_presence_in_area_of_interest_list, type_, "Value");
  return c.get<ue_presence_in_area_of_interest_list_l>();
}
location_report_request_type_s& location_report_ies_o::value_c::location_report_request_type()
{
  assert_choice_type(types::location_report_request_type, type_, "Value");
  return c.get<location_report_request_type_s>();
}
ngran_cgi_c& location_report_ies_o::value_c::ps_cell_info()
{
  assert_choice_type(types::ps_cell_info, type_, "Value");
  return c.get<ngran_cgi_c>();
}
const uint64_t& location_report_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& location_report_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const user_location_info_c& location_report_ies_o::value_c::user_location_info() const
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_c>();
}
const ue_presence_in_area_of_interest_list_l&
location_report_ies_o::value_c::ue_presence_in_area_of_interest_list() const
{
  assert_choice_type(types::ue_presence_in_area_of_interest_list, type_, "Value");
  return c.get<ue_presence_in_area_of_interest_list_l>();
}
const location_report_request_type_s& location_report_ies_o::value_c::location_report_request_type() const
{
  assert_choice_type(types::location_report_request_type, type_, "Value");
  return c.get<location_report_request_type_s>();
}
const ngran_cgi_c& location_report_ies_o::value_c::ps_cell_info() const
{
  assert_choice_type(types::ps_cell_info, type_, "Value");
  return c.get<ngran_cgi_c>();
}
void location_report_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::user_location_info:
      j.write_fieldname("UserLocationInformation");
      c.get<user_location_info_c>().to_json(j);
      break;
    case types::ue_presence_in_area_of_interest_list:
      j.start_array("UEPresenceInAreaOfInterestList");
      for (const auto& e1 : c.get<ue_presence_in_area_of_interest_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::location_report_request_type:
      j.write_fieldname("LocationReportingRequestType");
      c.get<location_report_request_type_s>().to_json(j);
      break;
    case types::ps_cell_info:
      j.write_fieldname("NGRAN-CGI");
      c.get<ngran_cgi_c>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "location_report_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE location_report_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_c>().pack(bref));
      break;
    case types::ue_presence_in_area_of_interest_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<ue_presence_in_area_of_interest_list_l>(), 1, 64, true));
      break;
    case types::location_report_request_type:
      HANDLE_CODE(c.get<location_report_request_type_s>().pack(bref));
      break;
    case types::ps_cell_info:
      HANDLE_CODE(c.get<ngran_cgi_c>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "location_report_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE location_report_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_c>().unpack(bref));
      break;
    case types::ue_presence_in_area_of_interest_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<ue_presence_in_area_of_interest_list_l>(), bref, 1, 64, true));
      break;
    case types::location_report_request_type:
      HANDLE_CODE(c.get<location_report_request_type_s>().unpack(bref));
      break;
    case types::ps_cell_info:
      HANDLE_CODE(c.get<ngran_cgi_c>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "location_report_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* location_report_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "UserLocationInformation",
                                  "UEPresenceInAreaOfInterestList",
                                  "LocationReportingRequestType",
                                  "NGRAN-CGI"};
  return convert_enum_idx(options, 6, value, "location_report_ies_o::value_c::types");
}

// LocationReportingControlIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t location_report_ctrl_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 33};
  return map_enum_number(options, 3, idx, "id");
}
bool location_report_ctrl_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 33};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e location_report_ctrl_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 33:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
location_report_ctrl_ies_o::value_c location_report_ctrl_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 33:
      ret.set(value_c::types::location_report_request_type);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e location_report_ctrl_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 33:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void location_report_ctrl_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::location_report_request_type:
      c.destroy<location_report_request_type_s>();
      break;
    default:
      break;
  }
}
void location_report_ctrl_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::location_report_request_type:
      c.init<location_report_request_type_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "location_report_ctrl_ies_o::value_c");
  }
}
location_report_ctrl_ies_o::value_c::value_c(const location_report_ctrl_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::location_report_request_type:
      c.init(other.c.get<location_report_request_type_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "location_report_ctrl_ies_o::value_c");
  }
}
location_report_ctrl_ies_o::value_c&
location_report_ctrl_ies_o::value_c::operator=(const location_report_ctrl_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::location_report_request_type:
      c.set(other.c.get<location_report_request_type_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "location_report_ctrl_ies_o::value_c");
  }

  return *this;
}
uint64_t& location_report_ctrl_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& location_report_ctrl_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
location_report_request_type_s& location_report_ctrl_ies_o::value_c::location_report_request_type()
{
  assert_choice_type(types::location_report_request_type, type_, "Value");
  return c.get<location_report_request_type_s>();
}
const uint64_t& location_report_ctrl_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& location_report_ctrl_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const location_report_request_type_s& location_report_ctrl_ies_o::value_c::location_report_request_type() const
{
  assert_choice_type(types::location_report_request_type, type_, "Value");
  return c.get<location_report_request_type_s>();
}
void location_report_ctrl_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::location_report_request_type:
      j.write_fieldname("LocationReportingRequestType");
      c.get<location_report_request_type_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "location_report_ctrl_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE location_report_ctrl_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::location_report_request_type:
      HANDLE_CODE(c.get<location_report_request_type_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "location_report_ctrl_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE location_report_ctrl_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::location_report_request_type:
      HANDLE_CODE(c.get<location_report_request_type_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "location_report_ctrl_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* location_report_ctrl_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..1099511627775)", "INTEGER (0..4294967295)", "LocationReportingRequestType"};
  return convert_enum_idx(options, 3, value, "location_report_ctrl_ies_o::value_c::types");
}

// LocationReportingFailureIndicationIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t location_report_fail_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 15};
  return map_enum_number(options, 3, idx, "id");
}
bool location_report_fail_ind_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 15};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e location_report_fail_ind_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 15:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
location_report_fail_ind_ies_o::value_c location_report_fail_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 15:
      ret.set(value_c::types::cause);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e location_report_fail_ind_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 15:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void location_report_fail_ind_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    default:
      break;
  }
}
void location_report_fail_ind_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "location_report_fail_ind_ies_o::value_c");
  }
}
location_report_fail_ind_ies_o::value_c::value_c(const location_report_fail_ind_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "location_report_fail_ind_ies_o::value_c");
  }
}
location_report_fail_ind_ies_o::value_c&
location_report_fail_ind_ies_o::value_c::operator=(const location_report_fail_ind_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "location_report_fail_ind_ies_o::value_c");
  }

  return *this;
}
uint64_t& location_report_fail_ind_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& location_report_fail_ind_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
cause_c& location_report_fail_ind_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const uint64_t& location_report_fail_ind_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& location_report_fail_ind_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const cause_c& location_report_fail_ind_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
void location_report_fail_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "location_report_fail_ind_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE location_report_fail_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "location_report_fail_ind_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE location_report_fail_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "location_report_fail_ind_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* location_report_fail_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)", "INTEGER (0..4294967295)", "Cause"};
  return convert_enum_idx(options, 3, value, "location_report_fail_ind_ies_o::value_c::types");
}

// NASNonDeliveryIndication-IEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t nas_non_delivery_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 38, 15};
  return map_enum_number(options, 4, idx, "id");
}
bool nas_non_delivery_ind_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 38, 15};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e nas_non_delivery_ind_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 38:
      return crit_e::ignore;
    case 15:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
nas_non_delivery_ind_ies_o::value_c nas_non_delivery_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 38:
      ret.set(value_c::types::nas_pdu);
      break;
    case 15:
      ret.set(value_c::types::cause);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e nas_non_delivery_ind_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 38:
      return presence_e::mandatory;
    case 15:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void nas_non_delivery_ind_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::nas_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::cause:
      c.destroy<cause_c>();
      break;
    default:
      break;
  }
}
void nas_non_delivery_ind_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::nas_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::cause:
      c.init<cause_c>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "nas_non_delivery_ind_ies_o::value_c");
  }
}
nas_non_delivery_ind_ies_o::value_c::value_c(const nas_non_delivery_ind_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::nas_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "nas_non_delivery_ind_ies_o::value_c");
  }
}
nas_non_delivery_ind_ies_o::value_c&
nas_non_delivery_ind_ies_o::value_c::operator=(const nas_non_delivery_ind_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::nas_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "nas_non_delivery_ind_ies_o::value_c");
  }

  return *this;
}
uint64_t& nas_non_delivery_ind_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& nas_non_delivery_ind_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
unbounded_octstring<true>& nas_non_delivery_ind_ies_o::value_c::nas_pdu()
{
  assert_choice_type(types::nas_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
cause_c& nas_non_delivery_ind_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const uint64_t& nas_non_delivery_ind_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& nas_non_delivery_ind_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const unbounded_octstring<true>& nas_non_delivery_ind_ies_o::value_c::nas_pdu() const
{
  assert_choice_type(types::nas_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const cause_c& nas_non_delivery_ind_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
void nas_non_delivery_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::nas_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "nas_non_delivery_ind_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE nas_non_delivery_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "nas_non_delivery_ind_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE nas_non_delivery_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "nas_non_delivery_ind_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* nas_non_delivery_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)", "INTEGER (0..4294967295)", "OCTET STRING", "Cause"};
  return convert_enum_idx(options, 4, value, "nas_non_delivery_ind_ies_o::value_c::types");
}

// NGResetAcknowledgeIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ng_reset_ack_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {111, 19};
  return map_enum_number(options, 2, idx, "id");
}
bool ng_reset_ack_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {111, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ng_reset_ack_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 111:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ng_reset_ack_ies_o::value_c ng_reset_ack_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 111:
      ret.set(value_c::types::ue_associated_lc_ng_conn_list);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ng_reset_ack_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 111:
      return presence_e::optional;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ng_reset_ack_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ue_associated_lc_ng_conn_list:
      c.destroy<ue_associated_lc_ng_conn_list_l>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ng_reset_ack_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::ue_associated_lc_ng_conn_list:
      c.init<ue_associated_lc_ng_conn_list_l>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_reset_ack_ies_o::value_c");
  }
}
ng_reset_ack_ies_o::value_c::value_c(const ng_reset_ack_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::ue_associated_lc_ng_conn_list:
      c.init(other.c.get<ue_associated_lc_ng_conn_list_l>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_reset_ack_ies_o::value_c");
  }
}
ng_reset_ack_ies_o::value_c& ng_reset_ack_ies_o::value_c::operator=(const ng_reset_ack_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::ue_associated_lc_ng_conn_list:
      c.set(other.c.get<ue_associated_lc_ng_conn_list_l>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_reset_ack_ies_o::value_c");
  }

  return *this;
}
ue_associated_lc_ng_conn_list_l& ng_reset_ack_ies_o::value_c::ue_associated_lc_ng_conn_list()
{
  assert_choice_type(types::ue_associated_lc_ng_conn_list, type_, "Value");
  return c.get<ue_associated_lc_ng_conn_list_l>();
}
crit_diagnostics_s& ng_reset_ack_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const ue_associated_lc_ng_conn_list_l& ng_reset_ack_ies_o::value_c::ue_associated_lc_ng_conn_list() const
{
  assert_choice_type(types::ue_associated_lc_ng_conn_list, type_, "Value");
  return c.get<ue_associated_lc_ng_conn_list_l>();
}
const crit_diagnostics_s& ng_reset_ack_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void ng_reset_ack_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::ue_associated_lc_ng_conn_list:
      j.start_array("UE-associatedLogicalNG-connectionList");
      for (const auto& e1 : c.get<ue_associated_lc_ng_conn_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ng_reset_ack_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ng_reset_ack_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::ue_associated_lc_ng_conn_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<ue_associated_lc_ng_conn_list_l>(), 1, 65536, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ng_reset_ack_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ng_reset_ack_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::ue_associated_lc_ng_conn_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<ue_associated_lc_ng_conn_list_l>(), bref, 1, 65536, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ng_reset_ack_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ng_reset_ack_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"UE-associatedLogicalNG-connectionList", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 2, value, "ng_reset_ack_ies_o::value_c::types");
}

// NGResetIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ng_reset_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {15, 88};
  return map_enum_number(options, 2, idx, "id");
}
bool ng_reset_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {15, 88};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ng_reset_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 15:
      return crit_e::ignore;
    case 88:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ng_reset_ies_o::value_c ng_reset_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 15:
      ret.set(value_c::types::cause);
      break;
    case 88:
      ret.set(value_c::types::reset_type);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ng_reset_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 15:
      return presence_e::mandatory;
    case 88:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ng_reset_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::reset_type:
      c.destroy<reset_type_c>();
      break;
    default:
      break;
  }
}
void ng_reset_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cause:
      c.init<cause_c>();
      break;
    case types::reset_type:
      c.init<reset_type_c>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_reset_ies_o::value_c");
  }
}
ng_reset_ies_o::value_c::value_c(const ng_reset_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::reset_type:
      c.init(other.c.get<reset_type_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_reset_ies_o::value_c");
  }
}
ng_reset_ies_o::value_c& ng_reset_ies_o::value_c::operator=(const ng_reset_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::reset_type:
      c.set(other.c.get<reset_type_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_reset_ies_o::value_c");
  }

  return *this;
}
cause_c& ng_reset_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
reset_type_c& ng_reset_ies_o::value_c::reset_type()
{
  assert_choice_type(types::reset_type, type_, "Value");
  return c.get<reset_type_c>();
}
const cause_c& ng_reset_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const reset_type_c& ng_reset_ies_o::value_c::reset_type() const
{
  assert_choice_type(types::reset_type, type_, "Value");
  return c.get<reset_type_c>();
}
void ng_reset_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::reset_type:
      j.write_fieldname("ResetType");
      c.get<reset_type_c>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ng_reset_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ng_reset_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::reset_type:
      HANDLE_CODE(c.get<reset_type_c>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ng_reset_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ng_reset_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::reset_type:
      HANDLE_CODE(c.get<reset_type_c>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ng_reset_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ng_reset_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"Cause", "ResetType"};
  return convert_enum_idx(options, 2, value, "ng_reset_ies_o::value_c::types");
}

// NGSetupFailureIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ng_setup_fail_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {15, 107, 19};
  return map_enum_number(options, 3, idx, "id");
}
bool ng_setup_fail_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {15, 107, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ng_setup_fail_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 15:
      return crit_e::ignore;
    case 107:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ng_setup_fail_ies_o::value_c ng_setup_fail_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 15:
      ret.set(value_c::types::cause);
      break;
    case 107:
      ret.set(value_c::types::time_to_wait);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ng_setup_fail_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 15:
      return presence_e::mandatory;
    case 107:
      return presence_e::optional;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ng_setup_fail_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ng_setup_fail_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cause:
      c.init<cause_c>();
      break;
    case types::time_to_wait:
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_fail_ies_o::value_c");
  }
}
ng_setup_fail_ies_o::value_c::value_c(const ng_setup_fail_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::time_to_wait:
      c.init(other.c.get<time_to_wait_e>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_fail_ies_o::value_c");
  }
}
ng_setup_fail_ies_o::value_c& ng_setup_fail_ies_o::value_c::operator=(const ng_setup_fail_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::time_to_wait:
      c.set(other.c.get<time_to_wait_e>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_fail_ies_o::value_c");
  }

  return *this;
}
cause_c& ng_setup_fail_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
time_to_wait_e& ng_setup_fail_ies_o::value_c::time_to_wait()
{
  assert_choice_type(types::time_to_wait, type_, "Value");
  return c.get<time_to_wait_e>();
}
crit_diagnostics_s& ng_setup_fail_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const cause_c& ng_setup_fail_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const time_to_wait_e& ng_setup_fail_ies_o::value_c::time_to_wait() const
{
  assert_choice_type(types::time_to_wait, type_, "Value");
  return c.get<time_to_wait_e>();
}
const crit_diagnostics_s& ng_setup_fail_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void ng_setup_fail_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::time_to_wait:
      j.write_str("TimeToWait", c.get<time_to_wait_e>().to_string());
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_fail_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ng_setup_fail_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::time_to_wait:
      HANDLE_CODE(c.get<time_to_wait_e>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_fail_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ng_setup_fail_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::time_to_wait:
      HANDLE_CODE(c.get<time_to_wait_e>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_fail_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ng_setup_fail_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"Cause", "TimeToWait", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 3, value, "ng_setup_fail_ies_o::value_c::types");
}

// NGSetupRequestIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ng_setup_request_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {27, 82, 102, 21, 147};
  return map_enum_number(options, 5, idx, "id");
}
bool ng_setup_request_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {27, 82, 102, 21, 147};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ng_setup_request_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 27:
      return crit_e::reject;
    case 82:
      return crit_e::ignore;
    case 102:
      return crit_e::reject;
    case 21:
      return crit_e::ignore;
    case 147:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ng_setup_request_ies_o::value_c ng_setup_request_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 27:
      ret.set(value_c::types::global_ran_node_id);
      break;
    case 82:
      ret.set(value_c::types::ran_node_name);
      break;
    case 102:
      ret.set(value_c::types::supported_ta_list);
      break;
    case 21:
      ret.set(value_c::types::default_paging_drx);
      break;
    case 147:
      ret.set(value_c::types::ue_retention_info);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ng_setup_request_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 27:
      return presence_e::mandatory;
    case 82:
      return presence_e::optional;
    case 102:
      return presence_e::mandatory;
    case 21:
      return presence_e::mandatory;
    case 147:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ng_setup_request_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::global_ran_node_id:
      c.destroy<global_ran_node_id_c>();
      break;
    case types::ran_node_name:
      c.destroy<printable_string<1, 150, true, true> >();
      break;
    case types::supported_ta_list:
      c.destroy<supported_ta_list_l>();
      break;
    default:
      break;
  }
}
void ng_setup_request_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::global_ran_node_id:
      c.init<global_ran_node_id_c>();
      break;
    case types::ran_node_name:
      c.init<printable_string<1, 150, true, true> >();
      break;
    case types::supported_ta_list:
      c.init<supported_ta_list_l>();
      break;
    case types::default_paging_drx:
      break;
    case types::ue_retention_info:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_request_ies_o::value_c");
  }
}
ng_setup_request_ies_o::value_c::value_c(const ng_setup_request_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::global_ran_node_id:
      c.init(other.c.get<global_ran_node_id_c>());
      break;
    case types::ran_node_name:
      c.init(other.c.get<printable_string<1, 150, true, true> >());
      break;
    case types::supported_ta_list:
      c.init(other.c.get<supported_ta_list_l>());
      break;
    case types::default_paging_drx:
      c.init(other.c.get<paging_drx_e>());
      break;
    case types::ue_retention_info:
      c.init(other.c.get<ue_retention_info_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_request_ies_o::value_c");
  }
}
ng_setup_request_ies_o::value_c&
ng_setup_request_ies_o::value_c::operator=(const ng_setup_request_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::global_ran_node_id:
      c.set(other.c.get<global_ran_node_id_c>());
      break;
    case types::ran_node_name:
      c.set(other.c.get<printable_string<1, 150, true, true> >());
      break;
    case types::supported_ta_list:
      c.set(other.c.get<supported_ta_list_l>());
      break;
    case types::default_paging_drx:
      c.set(other.c.get<paging_drx_e>());
      break;
    case types::ue_retention_info:
      c.set(other.c.get<ue_retention_info_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_request_ies_o::value_c");
  }

  return *this;
}
global_ran_node_id_c& ng_setup_request_ies_o::value_c::global_ran_node_id()
{
  assert_choice_type(types::global_ran_node_id, type_, "Value");
  return c.get<global_ran_node_id_c>();
}
printable_string<1, 150, true, true>& ng_setup_request_ies_o::value_c::ran_node_name()
{
  assert_choice_type(types::ran_node_name, type_, "Value");
  return c.get<printable_string<1, 150, true, true> >();
}
supported_ta_list_l& ng_setup_request_ies_o::value_c::supported_ta_list()
{
  assert_choice_type(types::supported_ta_list, type_, "Value");
  return c.get<supported_ta_list_l>();
}
paging_drx_e& ng_setup_request_ies_o::value_c::default_paging_drx()
{
  assert_choice_type(types::default_paging_drx, type_, "Value");
  return c.get<paging_drx_e>();
}
ue_retention_info_e& ng_setup_request_ies_o::value_c::ue_retention_info()
{
  assert_choice_type(types::ue_retention_info, type_, "Value");
  return c.get<ue_retention_info_e>();
}
const global_ran_node_id_c& ng_setup_request_ies_o::value_c::global_ran_node_id() const
{
  assert_choice_type(types::global_ran_node_id, type_, "Value");
  return c.get<global_ran_node_id_c>();
}
const printable_string<1, 150, true, true>& ng_setup_request_ies_o::value_c::ran_node_name() const
{
  assert_choice_type(types::ran_node_name, type_, "Value");
  return c.get<printable_string<1, 150, true, true> >();
}
const supported_ta_list_l& ng_setup_request_ies_o::value_c::supported_ta_list() const
{
  assert_choice_type(types::supported_ta_list, type_, "Value");
  return c.get<supported_ta_list_l>();
}
const paging_drx_e& ng_setup_request_ies_o::value_c::default_paging_drx() const
{
  assert_choice_type(types::default_paging_drx, type_, "Value");
  return c.get<paging_drx_e>();
}
const ue_retention_info_e& ng_setup_request_ies_o::value_c::ue_retention_info() const
{
  assert_choice_type(types::ue_retention_info, type_, "Value");
  return c.get<ue_retention_info_e>();
}
void ng_setup_request_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::global_ran_node_id:
      j.write_fieldname("GlobalRANNodeID");
      c.get<global_ran_node_id_c>().to_json(j);
      break;
    case types::ran_node_name:
      j.write_str("PrintableString", c.get<printable_string<1, 150, true, true> >().to_string());
      break;
    case types::supported_ta_list:
      j.start_array("SupportedTAList");
      for (const auto& e1 : c.get<supported_ta_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::default_paging_drx:
      j.write_str("PagingDRX", c.get<paging_drx_e>().to_string());
      break;
    case types::ue_retention_info:
      j.write_str("UERetentionInformation", "ues-retained");
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_request_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ng_setup_request_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::global_ran_node_id:
      HANDLE_CODE(c.get<global_ran_node_id_c>().pack(bref));
      break;
    case types::ran_node_name:
      HANDLE_CODE((c.get<printable_string<1, 150, true, true> >().pack(bref)));
      break;
    case types::supported_ta_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<supported_ta_list_l>(), 1, 256, true));
      break;
    case types::default_paging_drx:
      HANDLE_CODE(c.get<paging_drx_e>().pack(bref));
      break;
    case types::ue_retention_info:
      HANDLE_CODE(c.get<ue_retention_info_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_request_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ng_setup_request_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::global_ran_node_id:
      HANDLE_CODE(c.get<global_ran_node_id_c>().unpack(bref));
      break;
    case types::ran_node_name:
      HANDLE_CODE((c.get<printable_string<1, 150, true, true> >().unpack(bref)));
      break;
    case types::supported_ta_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<supported_ta_list_l>(), bref, 1, 256, true));
      break;
    case types::default_paging_drx:
      HANDLE_CODE(c.get<paging_drx_e>().unpack(bref));
      break;
    case types::ue_retention_info:
      HANDLE_CODE(c.get<ue_retention_info_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_request_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ng_setup_request_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "GlobalRANNodeID", "PrintableString", "SupportedTAList", "PagingDRX", "UERetentionInformation"};
  return convert_enum_idx(options, 5, value, "ng_setup_request_ies_o::value_c::types");
}

// NGSetupResponseIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ng_setup_resp_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {1, 96, 86, 80, 19, 147};
  return map_enum_number(options, 6, idx, "id");
}
bool ng_setup_resp_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {1, 96, 86, 80, 19, 147};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ng_setup_resp_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 1:
      return crit_e::reject;
    case 96:
      return crit_e::reject;
    case 86:
      return crit_e::ignore;
    case 80:
      return crit_e::reject;
    case 19:
      return crit_e::ignore;
    case 147:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ng_setup_resp_ies_o::value_c ng_setup_resp_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 1:
      ret.set(value_c::types::amf_name);
      break;
    case 96:
      ret.set(value_c::types::served_guami_list);
      break;
    case 86:
      ret.set(value_c::types::relative_amf_capacity);
      break;
    case 80:
      ret.set(value_c::types::plmn_support_list);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    case 147:
      ret.set(value_c::types::ue_retention_info);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ng_setup_resp_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 1:
      return presence_e::mandatory;
    case 96:
      return presence_e::mandatory;
    case 86:
      return presence_e::mandatory;
    case 80:
      return presence_e::mandatory;
    case 19:
      return presence_e::optional;
    case 147:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ng_setup_resp_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::amf_name:
      c.destroy<printable_string<1, 150, true, true> >();
      break;
    case types::served_guami_list:
      c.destroy<served_guami_list_l>();
      break;
    case types::plmn_support_list:
      c.destroy<plmn_support_list_l>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ng_setup_resp_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_name:
      c.init<printable_string<1, 150, true, true> >();
      break;
    case types::served_guami_list:
      c.init<served_guami_list_l>();
      break;
    case types::relative_amf_capacity:
      break;
    case types::plmn_support_list:
      c.init<plmn_support_list_l>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::ue_retention_info:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_resp_ies_o::value_c");
  }
}
ng_setup_resp_ies_o::value_c::value_c(const ng_setup_resp_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_name:
      c.init(other.c.get<printable_string<1, 150, true, true> >());
      break;
    case types::served_guami_list:
      c.init(other.c.get<served_guami_list_l>());
      break;
    case types::relative_amf_capacity:
      c.init(other.c.get<uint16_t>());
      break;
    case types::plmn_support_list:
      c.init(other.c.get<plmn_support_list_l>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::ue_retention_info:
      c.init(other.c.get<ue_retention_info_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_resp_ies_o::value_c");
  }
}
ng_setup_resp_ies_o::value_c& ng_setup_resp_ies_o::value_c::operator=(const ng_setup_resp_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_name:
      c.set(other.c.get<printable_string<1, 150, true, true> >());
      break;
    case types::served_guami_list:
      c.set(other.c.get<served_guami_list_l>());
      break;
    case types::relative_amf_capacity:
      c.set(other.c.get<uint16_t>());
      break;
    case types::plmn_support_list:
      c.set(other.c.get<plmn_support_list_l>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::ue_retention_info:
      c.set(other.c.get<ue_retention_info_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_resp_ies_o::value_c");
  }

  return *this;
}
printable_string<1, 150, true, true>& ng_setup_resp_ies_o::value_c::amf_name()
{
  assert_choice_type(types::amf_name, type_, "Value");
  return c.get<printable_string<1, 150, true, true> >();
}
served_guami_list_l& ng_setup_resp_ies_o::value_c::served_guami_list()
{
  assert_choice_type(types::served_guami_list, type_, "Value");
  return c.get<served_guami_list_l>();
}
uint16_t& ng_setup_resp_ies_o::value_c::relative_amf_capacity()
{
  assert_choice_type(types::relative_amf_capacity, type_, "Value");
  return c.get<uint16_t>();
}
plmn_support_list_l& ng_setup_resp_ies_o::value_c::plmn_support_list()
{
  assert_choice_type(types::plmn_support_list, type_, "Value");
  return c.get<plmn_support_list_l>();
}
crit_diagnostics_s& ng_setup_resp_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
ue_retention_info_e& ng_setup_resp_ies_o::value_c::ue_retention_info()
{
  assert_choice_type(types::ue_retention_info, type_, "Value");
  return c.get<ue_retention_info_e>();
}
const printable_string<1, 150, true, true>& ng_setup_resp_ies_o::value_c::amf_name() const
{
  assert_choice_type(types::amf_name, type_, "Value");
  return c.get<printable_string<1, 150, true, true> >();
}
const served_guami_list_l& ng_setup_resp_ies_o::value_c::served_guami_list() const
{
  assert_choice_type(types::served_guami_list, type_, "Value");
  return c.get<served_guami_list_l>();
}
const uint16_t& ng_setup_resp_ies_o::value_c::relative_amf_capacity() const
{
  assert_choice_type(types::relative_amf_capacity, type_, "Value");
  return c.get<uint16_t>();
}
const plmn_support_list_l& ng_setup_resp_ies_o::value_c::plmn_support_list() const
{
  assert_choice_type(types::plmn_support_list, type_, "Value");
  return c.get<plmn_support_list_l>();
}
const crit_diagnostics_s& ng_setup_resp_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const ue_retention_info_e& ng_setup_resp_ies_o::value_c::ue_retention_info() const
{
  assert_choice_type(types::ue_retention_info, type_, "Value");
  return c.get<ue_retention_info_e>();
}
void ng_setup_resp_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_name:
      j.write_str("PrintableString", c.get<printable_string<1, 150, true, true> >().to_string());
      break;
    case types::served_guami_list:
      j.start_array("ServedGUAMIList");
      for (const auto& e1 : c.get<served_guami_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::relative_amf_capacity:
      j.write_int("INTEGER (0..255)", c.get<uint16_t>());
      break;
    case types::plmn_support_list:
      j.start_array("PLMNSupportList");
      for (const auto& e1 : c.get<plmn_support_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    case types::ue_retention_info:
      j.write_str("UERetentionInformation", "ues-retained");
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_resp_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ng_setup_resp_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_name:
      HANDLE_CODE((c.get<printable_string<1, 150, true, true> >().pack(bref)));
      break;
    case types::served_guami_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<served_guami_list_l>(), 1, 256, true));
      break;
    case types::relative_amf_capacity:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)0u, (uint16_t)255u, false, true));
      break;
    case types::plmn_support_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<plmn_support_list_l>(), 1, 12, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    case types::ue_retention_info:
      HANDLE_CODE(c.get<ue_retention_info_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_resp_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ng_setup_resp_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_name:
      HANDLE_CODE((c.get<printable_string<1, 150, true, true> >().unpack(bref)));
      break;
    case types::served_guami_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<served_guami_list_l>(), bref, 1, 256, true));
      break;
    case types::relative_amf_capacity:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)0u, (uint16_t)255u, false, true));
      break;
    case types::plmn_support_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<plmn_support_list_l>(), bref, 1, 12, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    case types::ue_retention_info:
      HANDLE_CODE(c.get<ue_retention_info_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ng_setup_resp_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ng_setup_resp_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"PrintableString",
                                  "ServedGUAMIList",
                                  "INTEGER (0..255)",
                                  "PLMNSupportList",
                                  "CriticalityDiagnostics",
                                  "UERetentionInformation"};
  return convert_enum_idx(options, 6, value, "ng_setup_resp_ies_o::value_c::types");
}
uint8_t ng_setup_resp_ies_o::value_c::types_opts::to_number() const
{
  if (value == relative_amf_capacity) {
    return 0;
  }
  invalid_enum_number(value, "ng_setup_resp_ies_o::value_c::types");
  return 0;
}

// OverloadStartIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t overload_start_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {2, 9, 49};
  return map_enum_number(options, 3, idx, "id");
}
bool overload_start_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {2, 9, 49};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e overload_start_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 2:
      return crit_e::reject;
    case 9:
      return crit_e::ignore;
    case 49:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
overload_start_ies_o::value_c overload_start_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 2:
      ret.set(value_c::types::amf_overload_resp);
      break;
    case 9:
      ret.set(value_c::types::amf_traffic_load_reduction_ind);
      break;
    case 49:
      ret.set(value_c::types::overload_start_nssai_list);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e overload_start_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 2:
      return presence_e::optional;
    case 9:
      return presence_e::optional;
    case 49:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void overload_start_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::amf_overload_resp:
      c.destroy<overload_resp_c>();
      break;
    case types::overload_start_nssai_list:
      c.destroy<overload_start_nssai_list_l>();
      break;
    default:
      break;
  }
}
void overload_start_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_overload_resp:
      c.init<overload_resp_c>();
      break;
    case types::amf_traffic_load_reduction_ind:
      break;
    case types::overload_start_nssai_list:
      c.init<overload_start_nssai_list_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "overload_start_ies_o::value_c");
  }
}
overload_start_ies_o::value_c::value_c(const overload_start_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_overload_resp:
      c.init(other.c.get<overload_resp_c>());
      break;
    case types::amf_traffic_load_reduction_ind:
      c.init(other.c.get<uint8_t>());
      break;
    case types::overload_start_nssai_list:
      c.init(other.c.get<overload_start_nssai_list_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "overload_start_ies_o::value_c");
  }
}
overload_start_ies_o::value_c& overload_start_ies_o::value_c::operator=(const overload_start_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_overload_resp:
      c.set(other.c.get<overload_resp_c>());
      break;
    case types::amf_traffic_load_reduction_ind:
      c.set(other.c.get<uint8_t>());
      break;
    case types::overload_start_nssai_list:
      c.set(other.c.get<overload_start_nssai_list_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "overload_start_ies_o::value_c");
  }

  return *this;
}
overload_resp_c& overload_start_ies_o::value_c::amf_overload_resp()
{
  assert_choice_type(types::amf_overload_resp, type_, "Value");
  return c.get<overload_resp_c>();
}
uint8_t& overload_start_ies_o::value_c::amf_traffic_load_reduction_ind()
{
  assert_choice_type(types::amf_traffic_load_reduction_ind, type_, "Value");
  return c.get<uint8_t>();
}
overload_start_nssai_list_l& overload_start_ies_o::value_c::overload_start_nssai_list()
{
  assert_choice_type(types::overload_start_nssai_list, type_, "Value");
  return c.get<overload_start_nssai_list_l>();
}
const overload_resp_c& overload_start_ies_o::value_c::amf_overload_resp() const
{
  assert_choice_type(types::amf_overload_resp, type_, "Value");
  return c.get<overload_resp_c>();
}
const uint8_t& overload_start_ies_o::value_c::amf_traffic_load_reduction_ind() const
{
  assert_choice_type(types::amf_traffic_load_reduction_ind, type_, "Value");
  return c.get<uint8_t>();
}
const overload_start_nssai_list_l& overload_start_ies_o::value_c::overload_start_nssai_list() const
{
  assert_choice_type(types::overload_start_nssai_list, type_, "Value");
  return c.get<overload_start_nssai_list_l>();
}
void overload_start_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_overload_resp:
      j.write_fieldname("OverloadResponse");
      c.get<overload_resp_c>().to_json(j);
      break;
    case types::amf_traffic_load_reduction_ind:
      j.write_int("INTEGER (1..99)", c.get<uint8_t>());
      break;
    case types::overload_start_nssai_list:
      j.start_array("OverloadStartNSSAIList");
      for (const auto& e1 : c.get<overload_start_nssai_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "overload_start_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE overload_start_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_overload_resp:
      HANDLE_CODE(c.get<overload_resp_c>().pack(bref));
      break;
    case types::amf_traffic_load_reduction_ind:
      HANDLE_CODE(pack_integer(bref, c.get<uint8_t>(), (uint8_t)1u, (uint8_t)99u, false, true));
      break;
    case types::overload_start_nssai_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<overload_start_nssai_list_l>(), 1, 1024, true));
      break;
    default:
      log_invalid_choice_id(type_, "overload_start_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE overload_start_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_overload_resp:
      HANDLE_CODE(c.get<overload_resp_c>().unpack(bref));
      break;
    case types::amf_traffic_load_reduction_ind:
      HANDLE_CODE(unpack_integer(c.get<uint8_t>(), bref, (uint8_t)1u, (uint8_t)99u, false, true));
      break;
    case types::overload_start_nssai_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<overload_start_nssai_list_l>(), bref, 1, 1024, true));
      break;
    default:
      log_invalid_choice_id(type_, "overload_start_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* overload_start_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"OverloadResponse", "INTEGER (1..99)", "OverloadStartNSSAIList"};
  return convert_enum_idx(options, 3, value, "overload_start_ies_o::value_c::types");
}
uint8_t overload_start_ies_o::value_c::types_opts::to_number() const
{
  if (value == amf_traffic_load_reduction_ind) {
    return 1;
  }
  invalid_enum_number(value, "overload_start_ies_o::value_c::types");
  return 0;
}

// PDUSessionResourceModifyConfirmIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t pdu_session_res_modify_confirm_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 62, 131, 19};
  return map_enum_number(options, 5, idx, "id");
}
bool pdu_session_res_modify_confirm_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 62, 131, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e pdu_session_res_modify_confirm_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::ignore;
    case 85:
      return crit_e::ignore;
    case 62:
      return crit_e::ignore;
    case 131:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
pdu_session_res_modify_confirm_ies_o::value_c pdu_session_res_modify_confirm_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 62:
      ret.set(value_c::types::pdu_session_res_modify_list_mod_cfm);
      break;
    case 131:
      ret.set(value_c::types::pdu_session_res_failed_to_modify_list_mod_cfm);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pdu_session_res_modify_confirm_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 62:
      return presence_e::mandatory;
    case 131:
      return presence_e::optional;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void pdu_session_res_modify_confirm_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::pdu_session_res_modify_list_mod_cfm:
      c.destroy<pdu_session_res_modify_list_mod_cfm_l>();
      break;
    case types::pdu_session_res_failed_to_modify_list_mod_cfm:
      c.destroy<pdu_session_res_failed_to_modify_list_mod_cfm_l>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void pdu_session_res_modify_confirm_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::pdu_session_res_modify_list_mod_cfm:
      c.init<pdu_session_res_modify_list_mod_cfm_l>();
      break;
    case types::pdu_session_res_failed_to_modify_list_mod_cfm:
      c.init<pdu_session_res_failed_to_modify_list_mod_cfm_l>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_confirm_ies_o::value_c");
  }
}
pdu_session_res_modify_confirm_ies_o::value_c::value_c(const pdu_session_res_modify_confirm_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_modify_list_mod_cfm:
      c.init(other.c.get<pdu_session_res_modify_list_mod_cfm_l>());
      break;
    case types::pdu_session_res_failed_to_modify_list_mod_cfm:
      c.init(other.c.get<pdu_session_res_failed_to_modify_list_mod_cfm_l>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_confirm_ies_o::value_c");
  }
}
pdu_session_res_modify_confirm_ies_o::value_c&
pdu_session_res_modify_confirm_ies_o::value_c::operator=(const pdu_session_res_modify_confirm_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_modify_list_mod_cfm:
      c.set(other.c.get<pdu_session_res_modify_list_mod_cfm_l>());
      break;
    case types::pdu_session_res_failed_to_modify_list_mod_cfm:
      c.set(other.c.get<pdu_session_res_failed_to_modify_list_mod_cfm_l>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_confirm_ies_o::value_c");
  }

  return *this;
}
uint64_t& pdu_session_res_modify_confirm_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& pdu_session_res_modify_confirm_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
pdu_session_res_modify_list_mod_cfm_l&
pdu_session_res_modify_confirm_ies_o::value_c::pdu_session_res_modify_list_mod_cfm()
{
  assert_choice_type(types::pdu_session_res_modify_list_mod_cfm, type_, "Value");
  return c.get<pdu_session_res_modify_list_mod_cfm_l>();
}
pdu_session_res_failed_to_modify_list_mod_cfm_l&
pdu_session_res_modify_confirm_ies_o::value_c::pdu_session_res_failed_to_modify_list_mod_cfm()
{
  assert_choice_type(types::pdu_session_res_failed_to_modify_list_mod_cfm, type_, "Value");
  return c.get<pdu_session_res_failed_to_modify_list_mod_cfm_l>();
}
crit_diagnostics_s& pdu_session_res_modify_confirm_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& pdu_session_res_modify_confirm_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& pdu_session_res_modify_confirm_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const pdu_session_res_modify_list_mod_cfm_l&
pdu_session_res_modify_confirm_ies_o::value_c::pdu_session_res_modify_list_mod_cfm() const
{
  assert_choice_type(types::pdu_session_res_modify_list_mod_cfm, type_, "Value");
  return c.get<pdu_session_res_modify_list_mod_cfm_l>();
}
const pdu_session_res_failed_to_modify_list_mod_cfm_l&
pdu_session_res_modify_confirm_ies_o::value_c::pdu_session_res_failed_to_modify_list_mod_cfm() const
{
  assert_choice_type(types::pdu_session_res_failed_to_modify_list_mod_cfm, type_, "Value");
  return c.get<pdu_session_res_failed_to_modify_list_mod_cfm_l>();
}
const crit_diagnostics_s& pdu_session_res_modify_confirm_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void pdu_session_res_modify_confirm_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::pdu_session_res_modify_list_mod_cfm:
      j.start_array("PDUSessionResourceModifyListModCfm");
      for (const auto& e1 : c.get<pdu_session_res_modify_list_mod_cfm_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::pdu_session_res_failed_to_modify_list_mod_cfm:
      j.start_array("PDUSessionResourceFailedToModifyListModCfm");
      for (const auto& e1 : c.get<pdu_session_res_failed_to_modify_list_mod_cfm_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_confirm_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE pdu_session_res_modify_confirm_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_modify_list_mod_cfm:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_modify_list_mod_cfm_l>(), 1, 256, true));
      break;
    case types::pdu_session_res_failed_to_modify_list_mod_cfm:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_failed_to_modify_list_mod_cfm_l>(), 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_confirm_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_modify_confirm_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_modify_list_mod_cfm:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_modify_list_mod_cfm_l>(), bref, 1, 256, true));
      break;
    case types::pdu_session_res_failed_to_modify_list_mod_cfm:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_failed_to_modify_list_mod_cfm_l>(), bref, 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_confirm_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* pdu_session_res_modify_confirm_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "PDUSessionResourceModifyListModCfm",
                                  "PDUSessionResourceFailedToModifyListModCfm",
                                  "CriticalityDiagnostics"};
  return convert_enum_idx(options, 5, value, "pdu_session_res_modify_confirm_ies_o::value_c::types");
}

// PDUSessionResourceModifyIndicationIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t pdu_session_res_modify_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 63};
  return map_enum_number(options, 3, idx, "id");
}
bool pdu_session_res_modify_ind_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 63};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e pdu_session_res_modify_ind_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 63:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
pdu_session_res_modify_ind_ies_o::value_c pdu_session_res_modify_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 63:
      ret.set(value_c::types::pdu_session_res_modify_list_mod_ind);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pdu_session_res_modify_ind_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 63:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void pdu_session_res_modify_ind_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::pdu_session_res_modify_list_mod_ind:
      c.destroy<pdu_session_res_modify_list_mod_ind_l>();
      break;
    default:
      break;
  }
}
void pdu_session_res_modify_ind_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::pdu_session_res_modify_list_mod_ind:
      c.init<pdu_session_res_modify_list_mod_ind_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_ind_ies_o::value_c");
  }
}
pdu_session_res_modify_ind_ies_o::value_c::value_c(const pdu_session_res_modify_ind_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_modify_list_mod_ind:
      c.init(other.c.get<pdu_session_res_modify_list_mod_ind_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_ind_ies_o::value_c");
  }
}
pdu_session_res_modify_ind_ies_o::value_c&
pdu_session_res_modify_ind_ies_o::value_c::operator=(const pdu_session_res_modify_ind_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_modify_list_mod_ind:
      c.set(other.c.get<pdu_session_res_modify_list_mod_ind_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_ind_ies_o::value_c");
  }

  return *this;
}
uint64_t& pdu_session_res_modify_ind_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& pdu_session_res_modify_ind_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
pdu_session_res_modify_list_mod_ind_l& pdu_session_res_modify_ind_ies_o::value_c::pdu_session_res_modify_list_mod_ind()
{
  assert_choice_type(types::pdu_session_res_modify_list_mod_ind, type_, "Value");
  return c.get<pdu_session_res_modify_list_mod_ind_l>();
}
const uint64_t& pdu_session_res_modify_ind_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& pdu_session_res_modify_ind_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const pdu_session_res_modify_list_mod_ind_l&
pdu_session_res_modify_ind_ies_o::value_c::pdu_session_res_modify_list_mod_ind() const
{
  assert_choice_type(types::pdu_session_res_modify_list_mod_ind, type_, "Value");
  return c.get<pdu_session_res_modify_list_mod_ind_l>();
}
void pdu_session_res_modify_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::pdu_session_res_modify_list_mod_ind:
      j.start_array("PDUSessionResourceModifyListModInd");
      for (const auto& e1 : c.get<pdu_session_res_modify_list_mod_ind_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_ind_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE pdu_session_res_modify_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_modify_list_mod_ind:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_modify_list_mod_ind_l>(), 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_ind_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_modify_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_modify_list_mod_ind:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_modify_list_mod_ind_l>(), bref, 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_ind_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* pdu_session_res_modify_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..1099511627775)", "INTEGER (0..4294967295)", "PDUSessionResourceModifyListModInd"};
  return convert_enum_idx(options, 3, value, "pdu_session_res_modify_ind_ies_o::value_c::types");
}

// PDUSessionResourceModifyRequestIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t pdu_session_res_modify_request_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 83, 64};
  return map_enum_number(options, 4, idx, "id");
}
bool pdu_session_res_modify_request_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 83, 64};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e pdu_session_res_modify_request_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 83:
      return crit_e::ignore;
    case 64:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
pdu_session_res_modify_request_ies_o::value_c pdu_session_res_modify_request_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 83:
      ret.set(value_c::types::ran_paging_prio);
      break;
    case 64:
      ret.set(value_c::types::pdu_session_res_modify_list_mod_req);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pdu_session_res_modify_request_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 83:
      return presence_e::optional;
    case 64:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void pdu_session_res_modify_request_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::pdu_session_res_modify_list_mod_req:
      c.destroy<pdu_session_res_modify_list_mod_req_l>();
      break;
    default:
      break;
  }
}
void pdu_session_res_modify_request_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::ran_paging_prio:
      break;
    case types::pdu_session_res_modify_list_mod_req:
      c.init<pdu_session_res_modify_list_mod_req_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_request_ies_o::value_c");
  }
}
pdu_session_res_modify_request_ies_o::value_c::value_c(const pdu_session_res_modify_request_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_paging_prio:
      c.init(other.c.get<uint16_t>());
      break;
    case types::pdu_session_res_modify_list_mod_req:
      c.init(other.c.get<pdu_session_res_modify_list_mod_req_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_request_ies_o::value_c");
  }
}
pdu_session_res_modify_request_ies_o::value_c&
pdu_session_res_modify_request_ies_o::value_c::operator=(const pdu_session_res_modify_request_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_paging_prio:
      c.set(other.c.get<uint16_t>());
      break;
    case types::pdu_session_res_modify_list_mod_req:
      c.set(other.c.get<pdu_session_res_modify_list_mod_req_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_request_ies_o::value_c");
  }

  return *this;
}
uint64_t& pdu_session_res_modify_request_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& pdu_session_res_modify_request_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint16_t& pdu_session_res_modify_request_ies_o::value_c::ran_paging_prio()
{
  assert_choice_type(types::ran_paging_prio, type_, "Value");
  return c.get<uint16_t>();
}
pdu_session_res_modify_list_mod_req_l&
pdu_session_res_modify_request_ies_o::value_c::pdu_session_res_modify_list_mod_req()
{
  assert_choice_type(types::pdu_session_res_modify_list_mod_req, type_, "Value");
  return c.get<pdu_session_res_modify_list_mod_req_l>();
}
const uint64_t& pdu_session_res_modify_request_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& pdu_session_res_modify_request_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint16_t& pdu_session_res_modify_request_ies_o::value_c::ran_paging_prio() const
{
  assert_choice_type(types::ran_paging_prio, type_, "Value");
  return c.get<uint16_t>();
}
const pdu_session_res_modify_list_mod_req_l&
pdu_session_res_modify_request_ies_o::value_c::pdu_session_res_modify_list_mod_req() const
{
  assert_choice_type(types::pdu_session_res_modify_list_mod_req, type_, "Value");
  return c.get<pdu_session_res_modify_list_mod_req_l>();
}
void pdu_session_res_modify_request_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::ran_paging_prio:
      j.write_int("INTEGER (1..256)", c.get<uint16_t>());
      break;
    case types::pdu_session_res_modify_list_mod_req:
      j.start_array("PDUSessionResourceModifyListModReq");
      for (const auto& e1 : c.get<pdu_session_res_modify_list_mod_req_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_request_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE pdu_session_res_modify_request_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::ran_paging_prio:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)1u, (uint16_t)256u, false, true));
      break;
    case types::pdu_session_res_modify_list_mod_req:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_modify_list_mod_req_l>(), 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_request_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_modify_request_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::ran_paging_prio:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)1u, (uint16_t)256u, false, true));
      break;
    case types::pdu_session_res_modify_list_mod_req:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_modify_list_mod_req_l>(), bref, 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_request_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* pdu_session_res_modify_request_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "INTEGER (1..256)",
                                  "PDUSessionResourceModifyListModReq"};
  return convert_enum_idx(options, 4, value, "pdu_session_res_modify_request_ies_o::value_c::types");
}

// PDUSessionResourceModifyResponseIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t pdu_session_res_modify_resp_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 65, 54, 121, 19};
  return map_enum_number(options, 6, idx, "id");
}
bool pdu_session_res_modify_resp_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 65, 54, 121, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e pdu_session_res_modify_resp_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::ignore;
    case 85:
      return crit_e::ignore;
    case 65:
      return crit_e::ignore;
    case 54:
      return crit_e::ignore;
    case 121:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
pdu_session_res_modify_resp_ies_o::value_c pdu_session_res_modify_resp_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 65:
      ret.set(value_c::types::pdu_session_res_modify_list_mod_res);
      break;
    case 54:
      ret.set(value_c::types::pdu_session_res_failed_to_modify_list_mod_res);
      break;
    case 121:
      ret.set(value_c::types::user_location_info);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pdu_session_res_modify_resp_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 65:
      return presence_e::optional;
    case 54:
      return presence_e::optional;
    case 121:
      return presence_e::optional;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void pdu_session_res_modify_resp_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::pdu_session_res_modify_list_mod_res:
      c.destroy<pdu_session_res_modify_list_mod_res_l>();
      break;
    case types::pdu_session_res_failed_to_modify_list_mod_res:
      c.destroy<pdu_session_res_failed_to_modify_list_mod_res_l>();
      break;
    case types::user_location_info:
      c.destroy<user_location_info_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void pdu_session_res_modify_resp_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::pdu_session_res_modify_list_mod_res:
      c.init<pdu_session_res_modify_list_mod_res_l>();
      break;
    case types::pdu_session_res_failed_to_modify_list_mod_res:
      c.init<pdu_session_res_failed_to_modify_list_mod_res_l>();
      break;
    case types::user_location_info:
      c.init<user_location_info_c>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_resp_ies_o::value_c");
  }
}
pdu_session_res_modify_resp_ies_o::value_c::value_c(const pdu_session_res_modify_resp_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_modify_list_mod_res:
      c.init(other.c.get<pdu_session_res_modify_list_mod_res_l>());
      break;
    case types::pdu_session_res_failed_to_modify_list_mod_res:
      c.init(other.c.get<pdu_session_res_failed_to_modify_list_mod_res_l>());
      break;
    case types::user_location_info:
      c.init(other.c.get<user_location_info_c>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_resp_ies_o::value_c");
  }
}
pdu_session_res_modify_resp_ies_o::value_c&
pdu_session_res_modify_resp_ies_o::value_c::operator=(const pdu_session_res_modify_resp_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_modify_list_mod_res:
      c.set(other.c.get<pdu_session_res_modify_list_mod_res_l>());
      break;
    case types::pdu_session_res_failed_to_modify_list_mod_res:
      c.set(other.c.get<pdu_session_res_failed_to_modify_list_mod_res_l>());
      break;
    case types::user_location_info:
      c.set(other.c.get<user_location_info_c>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_resp_ies_o::value_c");
  }

  return *this;
}
uint64_t& pdu_session_res_modify_resp_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& pdu_session_res_modify_resp_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
pdu_session_res_modify_list_mod_res_l& pdu_session_res_modify_resp_ies_o::value_c::pdu_session_res_modify_list_mod_res()
{
  assert_choice_type(types::pdu_session_res_modify_list_mod_res, type_, "Value");
  return c.get<pdu_session_res_modify_list_mod_res_l>();
}
pdu_session_res_failed_to_modify_list_mod_res_l&
pdu_session_res_modify_resp_ies_o::value_c::pdu_session_res_failed_to_modify_list_mod_res()
{
  assert_choice_type(types::pdu_session_res_failed_to_modify_list_mod_res, type_, "Value");
  return c.get<pdu_session_res_failed_to_modify_list_mod_res_l>();
}
user_location_info_c& pdu_session_res_modify_resp_ies_o::value_c::user_location_info()
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_c>();
}
crit_diagnostics_s& pdu_session_res_modify_resp_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& pdu_session_res_modify_resp_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& pdu_session_res_modify_resp_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const pdu_session_res_modify_list_mod_res_l&
pdu_session_res_modify_resp_ies_o::value_c::pdu_session_res_modify_list_mod_res() const
{
  assert_choice_type(types::pdu_session_res_modify_list_mod_res, type_, "Value");
  return c.get<pdu_session_res_modify_list_mod_res_l>();
}
const pdu_session_res_failed_to_modify_list_mod_res_l&
pdu_session_res_modify_resp_ies_o::value_c::pdu_session_res_failed_to_modify_list_mod_res() const
{
  assert_choice_type(types::pdu_session_res_failed_to_modify_list_mod_res, type_, "Value");
  return c.get<pdu_session_res_failed_to_modify_list_mod_res_l>();
}
const user_location_info_c& pdu_session_res_modify_resp_ies_o::value_c::user_location_info() const
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_c>();
}
const crit_diagnostics_s& pdu_session_res_modify_resp_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void pdu_session_res_modify_resp_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::pdu_session_res_modify_list_mod_res:
      j.start_array("PDUSessionResourceModifyListModRes");
      for (const auto& e1 : c.get<pdu_session_res_modify_list_mod_res_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::pdu_session_res_failed_to_modify_list_mod_res:
      j.start_array("PDUSessionResourceFailedToModifyListModRes");
      for (const auto& e1 : c.get<pdu_session_res_failed_to_modify_list_mod_res_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::user_location_info:
      j.write_fieldname("UserLocationInformation");
      c.get<user_location_info_c>().to_json(j);
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_resp_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE pdu_session_res_modify_resp_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_modify_list_mod_res:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_modify_list_mod_res_l>(), 1, 256, true));
      break;
    case types::pdu_session_res_failed_to_modify_list_mod_res:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_failed_to_modify_list_mod_res_l>(), 1, 256, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_c>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_resp_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_modify_resp_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_modify_list_mod_res:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_modify_list_mod_res_l>(), bref, 1, 256, true));
      break;
    case types::pdu_session_res_failed_to_modify_list_mod_res:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_failed_to_modify_list_mod_res_l>(), bref, 1, 256, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_c>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_modify_resp_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* pdu_session_res_modify_resp_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "PDUSessionResourceModifyListModRes",
                                  "PDUSessionResourceFailedToModifyListModRes",
                                  "UserLocationInformation",
                                  "CriticalityDiagnostics"};
  return convert_enum_idx(options, 6, value, "pdu_session_res_modify_resp_ies_o::value_c::types");
}

// PDUSessionResourceNotifyIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t pdu_session_res_notify_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 66, 67, 121};
  return map_enum_number(options, 5, idx, "id");
}
bool pdu_session_res_notify_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 66, 67, 121};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e pdu_session_res_notify_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 66:
      return crit_e::reject;
    case 67:
      return crit_e::ignore;
    case 121:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
pdu_session_res_notify_ies_o::value_c pdu_session_res_notify_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 66:
      ret.set(value_c::types::pdu_session_res_notify_list);
      break;
    case 67:
      ret.set(value_c::types::pdu_session_res_released_list_not);
      break;
    case 121:
      ret.set(value_c::types::user_location_info);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pdu_session_res_notify_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 66:
      return presence_e::optional;
    case 67:
      return presence_e::optional;
    case 121:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void pdu_session_res_notify_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::pdu_session_res_notify_list:
      c.destroy<pdu_session_res_notify_list_l>();
      break;
    case types::pdu_session_res_released_list_not:
      c.destroy<pdu_session_res_released_list_not_l>();
      break;
    case types::user_location_info:
      c.destroy<user_location_info_c>();
      break;
    default:
      break;
  }
}
void pdu_session_res_notify_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::pdu_session_res_notify_list:
      c.init<pdu_session_res_notify_list_l>();
      break;
    case types::pdu_session_res_released_list_not:
      c.init<pdu_session_res_released_list_not_l>();
      break;
    case types::user_location_info:
      c.init<user_location_info_c>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_notify_ies_o::value_c");
  }
}
pdu_session_res_notify_ies_o::value_c::value_c(const pdu_session_res_notify_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_notify_list:
      c.init(other.c.get<pdu_session_res_notify_list_l>());
      break;
    case types::pdu_session_res_released_list_not:
      c.init(other.c.get<pdu_session_res_released_list_not_l>());
      break;
    case types::user_location_info:
      c.init(other.c.get<user_location_info_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_notify_ies_o::value_c");
  }
}
pdu_session_res_notify_ies_o::value_c&
pdu_session_res_notify_ies_o::value_c::operator=(const pdu_session_res_notify_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_notify_list:
      c.set(other.c.get<pdu_session_res_notify_list_l>());
      break;
    case types::pdu_session_res_released_list_not:
      c.set(other.c.get<pdu_session_res_released_list_not_l>());
      break;
    case types::user_location_info:
      c.set(other.c.get<user_location_info_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_notify_ies_o::value_c");
  }

  return *this;
}
uint64_t& pdu_session_res_notify_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& pdu_session_res_notify_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
pdu_session_res_notify_list_l& pdu_session_res_notify_ies_o::value_c::pdu_session_res_notify_list()
{
  assert_choice_type(types::pdu_session_res_notify_list, type_, "Value");
  return c.get<pdu_session_res_notify_list_l>();
}
pdu_session_res_released_list_not_l& pdu_session_res_notify_ies_o::value_c::pdu_session_res_released_list_not()
{
  assert_choice_type(types::pdu_session_res_released_list_not, type_, "Value");
  return c.get<pdu_session_res_released_list_not_l>();
}
user_location_info_c& pdu_session_res_notify_ies_o::value_c::user_location_info()
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_c>();
}
const uint64_t& pdu_session_res_notify_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& pdu_session_res_notify_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const pdu_session_res_notify_list_l& pdu_session_res_notify_ies_o::value_c::pdu_session_res_notify_list() const
{
  assert_choice_type(types::pdu_session_res_notify_list, type_, "Value");
  return c.get<pdu_session_res_notify_list_l>();
}
const pdu_session_res_released_list_not_l&
pdu_session_res_notify_ies_o::value_c::pdu_session_res_released_list_not() const
{
  assert_choice_type(types::pdu_session_res_released_list_not, type_, "Value");
  return c.get<pdu_session_res_released_list_not_l>();
}
const user_location_info_c& pdu_session_res_notify_ies_o::value_c::user_location_info() const
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_c>();
}
void pdu_session_res_notify_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::pdu_session_res_notify_list:
      j.start_array("PDUSessionResourceNotifyList");
      for (const auto& e1 : c.get<pdu_session_res_notify_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::pdu_session_res_released_list_not:
      j.start_array("PDUSessionResourceReleasedListNot");
      for (const auto& e1 : c.get<pdu_session_res_released_list_not_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::user_location_info:
      j.write_fieldname("UserLocationInformation");
      c.get<user_location_info_c>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_notify_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE pdu_session_res_notify_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_notify_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_notify_list_l>(), 1, 256, true));
      break;
    case types::pdu_session_res_released_list_not:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_released_list_not_l>(), 1, 256, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_c>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_notify_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_notify_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_notify_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_notify_list_l>(), bref, 1, 256, true));
      break;
    case types::pdu_session_res_released_list_not:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_released_list_not_l>(), bref, 1, 256, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_c>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_notify_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* pdu_session_res_notify_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "PDUSessionResourceNotifyList",
                                  "PDUSessionResourceReleasedListNot",
                                  "UserLocationInformation"};
  return convert_enum_idx(options, 5, value, "pdu_session_res_notify_ies_o::value_c::types");
}

// PDUSessionResourceReleaseCommandIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t pdu_session_res_release_cmd_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 83, 38, 79};
  return map_enum_number(options, 5, idx, "id");
}
bool pdu_session_res_release_cmd_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 83, 38, 79};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e pdu_session_res_release_cmd_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 83:
      return crit_e::ignore;
    case 38:
      return crit_e::ignore;
    case 79:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
pdu_session_res_release_cmd_ies_o::value_c pdu_session_res_release_cmd_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 83:
      ret.set(value_c::types::ran_paging_prio);
      break;
    case 38:
      ret.set(value_c::types::nas_pdu);
      break;
    case 79:
      ret.set(value_c::types::pdu_session_res_to_release_list_rel_cmd);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pdu_session_res_release_cmd_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 83:
      return presence_e::optional;
    case 38:
      return presence_e::optional;
    case 79:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void pdu_session_res_release_cmd_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::nas_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::pdu_session_res_to_release_list_rel_cmd:
      c.destroy<pdu_session_res_to_release_list_rel_cmd_l>();
      break;
    default:
      break;
  }
}
void pdu_session_res_release_cmd_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::ran_paging_prio:
      break;
    case types::nas_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::pdu_session_res_to_release_list_rel_cmd:
      c.init<pdu_session_res_to_release_list_rel_cmd_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_release_cmd_ies_o::value_c");
  }
}
pdu_session_res_release_cmd_ies_o::value_c::value_c(const pdu_session_res_release_cmd_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_paging_prio:
      c.init(other.c.get<uint16_t>());
      break;
    case types::nas_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::pdu_session_res_to_release_list_rel_cmd:
      c.init(other.c.get<pdu_session_res_to_release_list_rel_cmd_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_release_cmd_ies_o::value_c");
  }
}
pdu_session_res_release_cmd_ies_o::value_c&
pdu_session_res_release_cmd_ies_o::value_c::operator=(const pdu_session_res_release_cmd_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_paging_prio:
      c.set(other.c.get<uint16_t>());
      break;
    case types::nas_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::pdu_session_res_to_release_list_rel_cmd:
      c.set(other.c.get<pdu_session_res_to_release_list_rel_cmd_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_release_cmd_ies_o::value_c");
  }

  return *this;
}
uint64_t& pdu_session_res_release_cmd_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& pdu_session_res_release_cmd_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint16_t& pdu_session_res_release_cmd_ies_o::value_c::ran_paging_prio()
{
  assert_choice_type(types::ran_paging_prio, type_, "Value");
  return c.get<uint16_t>();
}
unbounded_octstring<true>& pdu_session_res_release_cmd_ies_o::value_c::nas_pdu()
{
  assert_choice_type(types::nas_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
pdu_session_res_to_release_list_rel_cmd_l&
pdu_session_res_release_cmd_ies_o::value_c::pdu_session_res_to_release_list_rel_cmd()
{
  assert_choice_type(types::pdu_session_res_to_release_list_rel_cmd, type_, "Value");
  return c.get<pdu_session_res_to_release_list_rel_cmd_l>();
}
const uint64_t& pdu_session_res_release_cmd_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& pdu_session_res_release_cmd_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint16_t& pdu_session_res_release_cmd_ies_o::value_c::ran_paging_prio() const
{
  assert_choice_type(types::ran_paging_prio, type_, "Value");
  return c.get<uint16_t>();
}
const unbounded_octstring<true>& pdu_session_res_release_cmd_ies_o::value_c::nas_pdu() const
{
  assert_choice_type(types::nas_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const pdu_session_res_to_release_list_rel_cmd_l&
pdu_session_res_release_cmd_ies_o::value_c::pdu_session_res_to_release_list_rel_cmd() const
{
  assert_choice_type(types::pdu_session_res_to_release_list_rel_cmd, type_, "Value");
  return c.get<pdu_session_res_to_release_list_rel_cmd_l>();
}
void pdu_session_res_release_cmd_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::ran_paging_prio:
      j.write_int("INTEGER (1..256)", c.get<uint16_t>());
      break;
    case types::nas_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::pdu_session_res_to_release_list_rel_cmd:
      j.start_array("PDUSessionResourceToReleaseListRelCmd");
      for (const auto& e1 : c.get<pdu_session_res_to_release_list_rel_cmd_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_release_cmd_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE pdu_session_res_release_cmd_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::ran_paging_prio:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)1u, (uint16_t)256u, false, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::pdu_session_res_to_release_list_rel_cmd:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_to_release_list_rel_cmd_l>(), 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_release_cmd_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_release_cmd_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::ran_paging_prio:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)1u, (uint16_t)256u, false, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::pdu_session_res_to_release_list_rel_cmd:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_to_release_list_rel_cmd_l>(), bref, 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_release_cmd_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* pdu_session_res_release_cmd_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "INTEGER (1..256)",
                                  "OCTET STRING",
                                  "PDUSessionResourceToReleaseListRelCmd"};
  return convert_enum_idx(options, 5, value, "pdu_session_res_release_cmd_ies_o::value_c::types");
}

// PDUSessionResourceReleaseResponseIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t pdu_session_res_release_resp_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 70, 121, 19};
  return map_enum_number(options, 5, idx, "id");
}
bool pdu_session_res_release_resp_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 70, 121, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e pdu_session_res_release_resp_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::ignore;
    case 85:
      return crit_e::ignore;
    case 70:
      return crit_e::ignore;
    case 121:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
pdu_session_res_release_resp_ies_o::value_c pdu_session_res_release_resp_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 70:
      ret.set(value_c::types::pdu_session_res_released_list_rel_res);
      break;
    case 121:
      ret.set(value_c::types::user_location_info);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pdu_session_res_release_resp_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 70:
      return presence_e::mandatory;
    case 121:
      return presence_e::optional;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void pdu_session_res_release_resp_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::pdu_session_res_released_list_rel_res:
      c.destroy<pdu_session_res_released_list_rel_res_l>();
      break;
    case types::user_location_info:
      c.destroy<user_location_info_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void pdu_session_res_release_resp_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::pdu_session_res_released_list_rel_res:
      c.init<pdu_session_res_released_list_rel_res_l>();
      break;
    case types::user_location_info:
      c.init<user_location_info_c>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_release_resp_ies_o::value_c");
  }
}
pdu_session_res_release_resp_ies_o::value_c::value_c(const pdu_session_res_release_resp_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_released_list_rel_res:
      c.init(other.c.get<pdu_session_res_released_list_rel_res_l>());
      break;
    case types::user_location_info:
      c.init(other.c.get<user_location_info_c>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_release_resp_ies_o::value_c");
  }
}
pdu_session_res_release_resp_ies_o::value_c&
pdu_session_res_release_resp_ies_o::value_c::operator=(const pdu_session_res_release_resp_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_released_list_rel_res:
      c.set(other.c.get<pdu_session_res_released_list_rel_res_l>());
      break;
    case types::user_location_info:
      c.set(other.c.get<user_location_info_c>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_release_resp_ies_o::value_c");
  }

  return *this;
}
uint64_t& pdu_session_res_release_resp_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& pdu_session_res_release_resp_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
pdu_session_res_released_list_rel_res_l&
pdu_session_res_release_resp_ies_o::value_c::pdu_session_res_released_list_rel_res()
{
  assert_choice_type(types::pdu_session_res_released_list_rel_res, type_, "Value");
  return c.get<pdu_session_res_released_list_rel_res_l>();
}
user_location_info_c& pdu_session_res_release_resp_ies_o::value_c::user_location_info()
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_c>();
}
crit_diagnostics_s& pdu_session_res_release_resp_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& pdu_session_res_release_resp_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& pdu_session_res_release_resp_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const pdu_session_res_released_list_rel_res_l&
pdu_session_res_release_resp_ies_o::value_c::pdu_session_res_released_list_rel_res() const
{
  assert_choice_type(types::pdu_session_res_released_list_rel_res, type_, "Value");
  return c.get<pdu_session_res_released_list_rel_res_l>();
}
const user_location_info_c& pdu_session_res_release_resp_ies_o::value_c::user_location_info() const
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_c>();
}
const crit_diagnostics_s& pdu_session_res_release_resp_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void pdu_session_res_release_resp_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::pdu_session_res_released_list_rel_res:
      j.start_array("PDUSessionResourceReleasedListRelRes");
      for (const auto& e1 : c.get<pdu_session_res_released_list_rel_res_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::user_location_info:
      j.write_fieldname("UserLocationInformation");
      c.get<user_location_info_c>().to_json(j);
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_release_resp_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE pdu_session_res_release_resp_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_released_list_rel_res:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_released_list_rel_res_l>(), 1, 256, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_c>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_release_resp_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_release_resp_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_released_list_rel_res:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_released_list_rel_res_l>(), bref, 1, 256, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_c>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_release_resp_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* pdu_session_res_release_resp_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "PDUSessionResourceReleasedListRelRes",
                                  "UserLocationInformation",
                                  "CriticalityDiagnostics"};
  return convert_enum_idx(options, 5, value, "pdu_session_res_release_resp_ies_o::value_c::types");
}

// PDUSessionResourceSetupRequestIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t pdu_session_res_setup_request_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 83, 38, 74, 110};
  return map_enum_number(options, 6, idx, "id");
}
bool pdu_session_res_setup_request_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 83, 38, 74, 110};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e pdu_session_res_setup_request_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 83:
      return crit_e::ignore;
    case 38:
      return crit_e::reject;
    case 74:
      return crit_e::reject;
    case 110:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
pdu_session_res_setup_request_ies_o::value_c pdu_session_res_setup_request_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 83:
      ret.set(value_c::types::ran_paging_prio);
      break;
    case 38:
      ret.set(value_c::types::nas_pdu);
      break;
    case 74:
      ret.set(value_c::types::pdu_session_res_setup_list_su_req);
      break;
    case 110:
      ret.set(value_c::types::ue_aggregate_maximum_bit_rate);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pdu_session_res_setup_request_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 83:
      return presence_e::optional;
    case 38:
      return presence_e::optional;
    case 74:
      return presence_e::mandatory;
    case 110:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void pdu_session_res_setup_request_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::nas_pdu:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::pdu_session_res_setup_list_su_req:
      c.destroy<pdu_session_res_setup_list_su_req_l>();
      break;
    case types::ue_aggregate_maximum_bit_rate:
      c.destroy<ue_aggregate_maximum_bit_rate_s>();
      break;
    default:
      break;
  }
}
void pdu_session_res_setup_request_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::ran_paging_prio:
      break;
    case types::nas_pdu:
      c.init<unbounded_octstring<true> >();
      break;
    case types::pdu_session_res_setup_list_su_req:
      c.init<pdu_session_res_setup_list_su_req_l>();
      break;
    case types::ue_aggregate_maximum_bit_rate:
      c.init<ue_aggregate_maximum_bit_rate_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_setup_request_ies_o::value_c");
  }
}
pdu_session_res_setup_request_ies_o::value_c::value_c(const pdu_session_res_setup_request_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_paging_prio:
      c.init(other.c.get<uint16_t>());
      break;
    case types::nas_pdu:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::pdu_session_res_setup_list_su_req:
      c.init(other.c.get<pdu_session_res_setup_list_su_req_l>());
      break;
    case types::ue_aggregate_maximum_bit_rate:
      c.init(other.c.get<ue_aggregate_maximum_bit_rate_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_setup_request_ies_o::value_c");
  }
}
pdu_session_res_setup_request_ies_o::value_c&
pdu_session_res_setup_request_ies_o::value_c::operator=(const pdu_session_res_setup_request_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_paging_prio:
      c.set(other.c.get<uint16_t>());
      break;
    case types::nas_pdu:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::pdu_session_res_setup_list_su_req:
      c.set(other.c.get<pdu_session_res_setup_list_su_req_l>());
      break;
    case types::ue_aggregate_maximum_bit_rate:
      c.set(other.c.get<ue_aggregate_maximum_bit_rate_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_setup_request_ies_o::value_c");
  }

  return *this;
}
uint64_t& pdu_session_res_setup_request_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& pdu_session_res_setup_request_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint16_t& pdu_session_res_setup_request_ies_o::value_c::ran_paging_prio()
{
  assert_choice_type(types::ran_paging_prio, type_, "Value");
  return c.get<uint16_t>();
}
unbounded_octstring<true>& pdu_session_res_setup_request_ies_o::value_c::nas_pdu()
{
  assert_choice_type(types::nas_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
pdu_session_res_setup_list_su_req_l& pdu_session_res_setup_request_ies_o::value_c::pdu_session_res_setup_list_su_req()
{
  assert_choice_type(types::pdu_session_res_setup_list_su_req, type_, "Value");
  return c.get<pdu_session_res_setup_list_su_req_l>();
}
ue_aggregate_maximum_bit_rate_s& pdu_session_res_setup_request_ies_o::value_c::ue_aggregate_maximum_bit_rate()
{
  assert_choice_type(types::ue_aggregate_maximum_bit_rate, type_, "Value");
  return c.get<ue_aggregate_maximum_bit_rate_s>();
}
const uint64_t& pdu_session_res_setup_request_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& pdu_session_res_setup_request_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint16_t& pdu_session_res_setup_request_ies_o::value_c::ran_paging_prio() const
{
  assert_choice_type(types::ran_paging_prio, type_, "Value");
  return c.get<uint16_t>();
}
const unbounded_octstring<true>& pdu_session_res_setup_request_ies_o::value_c::nas_pdu() const
{
  assert_choice_type(types::nas_pdu, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const pdu_session_res_setup_list_su_req_l&
pdu_session_res_setup_request_ies_o::value_c::pdu_session_res_setup_list_su_req() const
{
  assert_choice_type(types::pdu_session_res_setup_list_su_req, type_, "Value");
  return c.get<pdu_session_res_setup_list_su_req_l>();
}
const ue_aggregate_maximum_bit_rate_s&
pdu_session_res_setup_request_ies_o::value_c::ue_aggregate_maximum_bit_rate() const
{
  assert_choice_type(types::ue_aggregate_maximum_bit_rate, type_, "Value");
  return c.get<ue_aggregate_maximum_bit_rate_s>();
}
void pdu_session_res_setup_request_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::ran_paging_prio:
      j.write_int("INTEGER (1..256)", c.get<uint16_t>());
      break;
    case types::nas_pdu:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::pdu_session_res_setup_list_su_req:
      j.start_array("PDUSessionResourceSetupListSUReq");
      for (const auto& e1 : c.get<pdu_session_res_setup_list_su_req_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::ue_aggregate_maximum_bit_rate:
      j.write_fieldname("UEAggregateMaximumBitRate");
      c.get<ue_aggregate_maximum_bit_rate_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_setup_request_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE pdu_session_res_setup_request_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::ran_paging_prio:
      HANDLE_CODE(pack_integer(bref, c.get<uint16_t>(), (uint16_t)1u, (uint16_t)256u, false, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::pdu_session_res_setup_list_su_req:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_setup_list_su_req_l>(), 1, 256, true));
      break;
    case types::ue_aggregate_maximum_bit_rate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bit_rate_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_setup_request_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_setup_request_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::ran_paging_prio:
      HANDLE_CODE(unpack_integer(c.get<uint16_t>(), bref, (uint16_t)1u, (uint16_t)256u, false, true));
      break;
    case types::nas_pdu:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::pdu_session_res_setup_list_su_req:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_setup_list_su_req_l>(), bref, 1, 256, true));
      break;
    case types::ue_aggregate_maximum_bit_rate:
      HANDLE_CODE(c.get<ue_aggregate_maximum_bit_rate_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_setup_request_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* pdu_session_res_setup_request_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "INTEGER (1..256)",
                                  "OCTET STRING",
                                  "PDUSessionResourceSetupListSUReq",
                                  "UEAggregateMaximumBitRate"};
  return convert_enum_idx(options, 6, value, "pdu_session_res_setup_request_ies_o::value_c::types");
}

// PDUSessionResourceSetupResponseIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t pdu_session_res_setup_resp_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 75, 58, 19};
  return map_enum_number(options, 5, idx, "id");
}
bool pdu_session_res_setup_resp_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 75, 58, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e pdu_session_res_setup_resp_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::ignore;
    case 85:
      return crit_e::ignore;
    case 75:
      return crit_e::ignore;
    case 58:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
pdu_session_res_setup_resp_ies_o::value_c pdu_session_res_setup_resp_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 75:
      ret.set(value_c::types::pdu_session_res_setup_list_su_res);
      break;
    case 58:
      ret.set(value_c::types::pdu_session_res_failed_to_setup_list_su_res);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pdu_session_res_setup_resp_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 75:
      return presence_e::optional;
    case 58:
      return presence_e::optional;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void pdu_session_res_setup_resp_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::pdu_session_res_setup_list_su_res:
      c.destroy<pdu_session_res_setup_list_su_res_l>();
      break;
    case types::pdu_session_res_failed_to_setup_list_su_res:
      c.destroy<pdu_session_res_failed_to_setup_list_su_res_l>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void pdu_session_res_setup_resp_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::pdu_session_res_setup_list_su_res:
      c.init<pdu_session_res_setup_list_su_res_l>();
      break;
    case types::pdu_session_res_failed_to_setup_list_su_res:
      c.init<pdu_session_res_failed_to_setup_list_su_res_l>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_setup_resp_ies_o::value_c");
  }
}
pdu_session_res_setup_resp_ies_o::value_c::value_c(const pdu_session_res_setup_resp_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_setup_list_su_res:
      c.init(other.c.get<pdu_session_res_setup_list_su_res_l>());
      break;
    case types::pdu_session_res_failed_to_setup_list_su_res:
      c.init(other.c.get<pdu_session_res_failed_to_setup_list_su_res_l>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_setup_resp_ies_o::value_c");
  }
}
pdu_session_res_setup_resp_ies_o::value_c&
pdu_session_res_setup_resp_ies_o::value_c::operator=(const pdu_session_res_setup_resp_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_setup_list_su_res:
      c.set(other.c.get<pdu_session_res_setup_list_su_res_l>());
      break;
    case types::pdu_session_res_failed_to_setup_list_su_res:
      c.set(other.c.get<pdu_session_res_failed_to_setup_list_su_res_l>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_setup_resp_ies_o::value_c");
  }

  return *this;
}
uint64_t& pdu_session_res_setup_resp_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& pdu_session_res_setup_resp_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
pdu_session_res_setup_list_su_res_l& pdu_session_res_setup_resp_ies_o::value_c::pdu_session_res_setup_list_su_res()
{
  assert_choice_type(types::pdu_session_res_setup_list_su_res, type_, "Value");
  return c.get<pdu_session_res_setup_list_su_res_l>();
}
pdu_session_res_failed_to_setup_list_su_res_l&
pdu_session_res_setup_resp_ies_o::value_c::pdu_session_res_failed_to_setup_list_su_res()
{
  assert_choice_type(types::pdu_session_res_failed_to_setup_list_su_res, type_, "Value");
  return c.get<pdu_session_res_failed_to_setup_list_su_res_l>();
}
crit_diagnostics_s& pdu_session_res_setup_resp_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& pdu_session_res_setup_resp_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& pdu_session_res_setup_resp_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const pdu_session_res_setup_list_su_res_l&
pdu_session_res_setup_resp_ies_o::value_c::pdu_session_res_setup_list_su_res() const
{
  assert_choice_type(types::pdu_session_res_setup_list_su_res, type_, "Value");
  return c.get<pdu_session_res_setup_list_su_res_l>();
}
const pdu_session_res_failed_to_setup_list_su_res_l&
pdu_session_res_setup_resp_ies_o::value_c::pdu_session_res_failed_to_setup_list_su_res() const
{
  assert_choice_type(types::pdu_session_res_failed_to_setup_list_su_res, type_, "Value");
  return c.get<pdu_session_res_failed_to_setup_list_su_res_l>();
}
const crit_diagnostics_s& pdu_session_res_setup_resp_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void pdu_session_res_setup_resp_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::pdu_session_res_setup_list_su_res:
      j.start_array("PDUSessionResourceSetupListSURes");
      for (const auto& e1 : c.get<pdu_session_res_setup_list_su_res_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::pdu_session_res_failed_to_setup_list_su_res:
      j.start_array("PDUSessionResourceFailedToSetupListSURes");
      for (const auto& e1 : c.get<pdu_session_res_failed_to_setup_list_su_res_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_setup_resp_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE pdu_session_res_setup_resp_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_setup_list_su_res:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_setup_list_su_res_l>(), 1, 256, true));
      break;
    case types::pdu_session_res_failed_to_setup_list_su_res:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_failed_to_setup_list_su_res_l>(), 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_setup_resp_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pdu_session_res_setup_resp_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_setup_list_su_res:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_setup_list_su_res_l>(), bref, 1, 256, true));
      break;
    case types::pdu_session_res_failed_to_setup_list_su_res:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_failed_to_setup_list_su_res_l>(), bref, 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pdu_session_res_setup_resp_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* pdu_session_res_setup_resp_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "PDUSessionResourceSetupListSURes",
                                  "PDUSessionResourceFailedToSetupListSURes",
                                  "CriticalityDiagnostics"};
  return convert_enum_idx(options, 5, value, "pdu_session_res_setup_resp_ies_o::value_c::types");
}

// PWSCancelRequestIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t pws_cancel_request_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {35, 95, 122, 14};
  return map_enum_number(options, 4, idx, "id");
}
bool pws_cancel_request_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {35, 95, 122, 14};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e pws_cancel_request_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 35:
      return crit_e::reject;
    case 95:
      return crit_e::reject;
    case 122:
      return crit_e::ignore;
    case 14:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
pws_cancel_request_ies_o::value_c pws_cancel_request_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 35:
      ret.set(value_c::types::msg_id);
      break;
    case 95:
      ret.set(value_c::types::serial_num);
      break;
    case 122:
      ret.set(value_c::types::warning_area_list);
      break;
    case 14:
      ret.set(value_c::types::cancel_all_warning_msgs);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pws_cancel_request_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 35:
      return presence_e::mandatory;
    case 95:
      return presence_e::mandatory;
    case 122:
      return presence_e::optional;
    case 14:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void pws_cancel_request_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::msg_id:
      c.destroy<fixed_bitstring<16, false, true> >();
      break;
    case types::serial_num:
      c.destroy<fixed_bitstring<16, false, true> >();
      break;
    case types::warning_area_list:
      c.destroy<warning_area_list_c>();
      break;
    default:
      break;
  }
}
void pws_cancel_request_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::msg_id:
      c.init<fixed_bitstring<16, false, true> >();
      break;
    case types::serial_num:
      c.init<fixed_bitstring<16, false, true> >();
      break;
    case types::warning_area_list:
      c.init<warning_area_list_c>();
      break;
    case types::cancel_all_warning_msgs:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pws_cancel_request_ies_o::value_c");
  }
}
pws_cancel_request_ies_o::value_c::value_c(const pws_cancel_request_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::msg_id:
      c.init(other.c.get<fixed_bitstring<16, false, true> >());
      break;
    case types::serial_num:
      c.init(other.c.get<fixed_bitstring<16, false, true> >());
      break;
    case types::warning_area_list:
      c.init(other.c.get<warning_area_list_c>());
      break;
    case types::cancel_all_warning_msgs:
      c.init(other.c.get<cancel_all_warning_msgs_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pws_cancel_request_ies_o::value_c");
  }
}
pws_cancel_request_ies_o::value_c&
pws_cancel_request_ies_o::value_c::operator=(const pws_cancel_request_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::msg_id:
      c.set(other.c.get<fixed_bitstring<16, false, true> >());
      break;
    case types::serial_num:
      c.set(other.c.get<fixed_bitstring<16, false, true> >());
      break;
    case types::warning_area_list:
      c.set(other.c.get<warning_area_list_c>());
      break;
    case types::cancel_all_warning_msgs:
      c.set(other.c.get<cancel_all_warning_msgs_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pws_cancel_request_ies_o::value_c");
  }

  return *this;
}
fixed_bitstring<16, false, true>& pws_cancel_request_ies_o::value_c::msg_id()
{
  assert_choice_type(types::msg_id, type_, "Value");
  return c.get<fixed_bitstring<16, false, true> >();
}
fixed_bitstring<16, false, true>& pws_cancel_request_ies_o::value_c::serial_num()
{
  assert_choice_type(types::serial_num, type_, "Value");
  return c.get<fixed_bitstring<16, false, true> >();
}
warning_area_list_c& pws_cancel_request_ies_o::value_c::warning_area_list()
{
  assert_choice_type(types::warning_area_list, type_, "Value");
  return c.get<warning_area_list_c>();
}
cancel_all_warning_msgs_e& pws_cancel_request_ies_o::value_c::cancel_all_warning_msgs()
{
  assert_choice_type(types::cancel_all_warning_msgs, type_, "Value");
  return c.get<cancel_all_warning_msgs_e>();
}
const fixed_bitstring<16, false, true>& pws_cancel_request_ies_o::value_c::msg_id() const
{
  assert_choice_type(types::msg_id, type_, "Value");
  return c.get<fixed_bitstring<16, false, true> >();
}
const fixed_bitstring<16, false, true>& pws_cancel_request_ies_o::value_c::serial_num() const
{
  assert_choice_type(types::serial_num, type_, "Value");
  return c.get<fixed_bitstring<16, false, true> >();
}
const warning_area_list_c& pws_cancel_request_ies_o::value_c::warning_area_list() const
{
  assert_choice_type(types::warning_area_list, type_, "Value");
  return c.get<warning_area_list_c>();
}
const cancel_all_warning_msgs_e& pws_cancel_request_ies_o::value_c::cancel_all_warning_msgs() const
{
  assert_choice_type(types::cancel_all_warning_msgs, type_, "Value");
  return c.get<cancel_all_warning_msgs_e>();
}
void pws_cancel_request_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::msg_id:
      j.write_str("BIT STRING", c.get<fixed_bitstring<16, false, true> >().to_string());
      break;
    case types::serial_num:
      j.write_str("BIT STRING", c.get<fixed_bitstring<16, false, true> >().to_string());
      break;
    case types::warning_area_list:
      j.write_fieldname("WarningAreaList");
      c.get<warning_area_list_c>().to_json(j);
      break;
    case types::cancel_all_warning_msgs:
      j.write_str("CancelAllWarningMessages", "true");
      break;
    default:
      log_invalid_choice_id(type_, "pws_cancel_request_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE pws_cancel_request_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::msg_id:
      HANDLE_CODE((c.get<fixed_bitstring<16, false, true> >().pack(bref)));
      break;
    case types::serial_num:
      HANDLE_CODE((c.get<fixed_bitstring<16, false, true> >().pack(bref)));
      break;
    case types::warning_area_list:
      HANDLE_CODE(c.get<warning_area_list_c>().pack(bref));
      break;
    case types::cancel_all_warning_msgs:
      HANDLE_CODE(c.get<cancel_all_warning_msgs_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pws_cancel_request_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pws_cancel_request_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::msg_id:
      HANDLE_CODE((c.get<fixed_bitstring<16, false, true> >().unpack(bref)));
      break;
    case types::serial_num:
      HANDLE_CODE((c.get<fixed_bitstring<16, false, true> >().unpack(bref)));
      break;
    case types::warning_area_list:
      HANDLE_CODE(c.get<warning_area_list_c>().unpack(bref));
      break;
    case types::cancel_all_warning_msgs:
      HANDLE_CODE(c.get<cancel_all_warning_msgs_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pws_cancel_request_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* pws_cancel_request_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"BIT STRING", "BIT STRING", "WarningAreaList", "CancelAllWarningMessages"};
  return convert_enum_idx(options, 4, value, "pws_cancel_request_ies_o::value_c::types");
}

// PWSCancelResponseIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t pws_cancel_resp_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {35, 95, 12, 19};
  return map_enum_number(options, 4, idx, "id");
}
bool pws_cancel_resp_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {35, 95, 12, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e pws_cancel_resp_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 35:
      return crit_e::reject;
    case 95:
      return crit_e::reject;
    case 12:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
pws_cancel_resp_ies_o::value_c pws_cancel_resp_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 35:
      ret.set(value_c::types::msg_id);
      break;
    case 95:
      ret.set(value_c::types::serial_num);
      break;
    case 12:
      ret.set(value_c::types::broadcast_cancelled_area_list);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pws_cancel_resp_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 35:
      return presence_e::mandatory;
    case 95:
      return presence_e::mandatory;
    case 12:
      return presence_e::optional;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void pws_cancel_resp_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::msg_id:
      c.destroy<fixed_bitstring<16, false, true> >();
      break;
    case types::serial_num:
      c.destroy<fixed_bitstring<16, false, true> >();
      break;
    case types::broadcast_cancelled_area_list:
      c.destroy<broadcast_cancelled_area_list_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void pws_cancel_resp_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::msg_id:
      c.init<fixed_bitstring<16, false, true> >();
      break;
    case types::serial_num:
      c.init<fixed_bitstring<16, false, true> >();
      break;
    case types::broadcast_cancelled_area_list:
      c.init<broadcast_cancelled_area_list_c>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pws_cancel_resp_ies_o::value_c");
  }
}
pws_cancel_resp_ies_o::value_c::value_c(const pws_cancel_resp_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::msg_id:
      c.init(other.c.get<fixed_bitstring<16, false, true> >());
      break;
    case types::serial_num:
      c.init(other.c.get<fixed_bitstring<16, false, true> >());
      break;
    case types::broadcast_cancelled_area_list:
      c.init(other.c.get<broadcast_cancelled_area_list_c>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pws_cancel_resp_ies_o::value_c");
  }
}
pws_cancel_resp_ies_o::value_c& pws_cancel_resp_ies_o::value_c::operator=(const pws_cancel_resp_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::msg_id:
      c.set(other.c.get<fixed_bitstring<16, false, true> >());
      break;
    case types::serial_num:
      c.set(other.c.get<fixed_bitstring<16, false, true> >());
      break;
    case types::broadcast_cancelled_area_list:
      c.set(other.c.get<broadcast_cancelled_area_list_c>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pws_cancel_resp_ies_o::value_c");
  }

  return *this;
}
fixed_bitstring<16, false, true>& pws_cancel_resp_ies_o::value_c::msg_id()
{
  assert_choice_type(types::msg_id, type_, "Value");
  return c.get<fixed_bitstring<16, false, true> >();
}
fixed_bitstring<16, false, true>& pws_cancel_resp_ies_o::value_c::serial_num()
{
  assert_choice_type(types::serial_num, type_, "Value");
  return c.get<fixed_bitstring<16, false, true> >();
}
broadcast_cancelled_area_list_c& pws_cancel_resp_ies_o::value_c::broadcast_cancelled_area_list()
{
  assert_choice_type(types::broadcast_cancelled_area_list, type_, "Value");
  return c.get<broadcast_cancelled_area_list_c>();
}
crit_diagnostics_s& pws_cancel_resp_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const fixed_bitstring<16, false, true>& pws_cancel_resp_ies_o::value_c::msg_id() const
{
  assert_choice_type(types::msg_id, type_, "Value");
  return c.get<fixed_bitstring<16, false, true> >();
}
const fixed_bitstring<16, false, true>& pws_cancel_resp_ies_o::value_c::serial_num() const
{
  assert_choice_type(types::serial_num, type_, "Value");
  return c.get<fixed_bitstring<16, false, true> >();
}
const broadcast_cancelled_area_list_c& pws_cancel_resp_ies_o::value_c::broadcast_cancelled_area_list() const
{
  assert_choice_type(types::broadcast_cancelled_area_list, type_, "Value");
  return c.get<broadcast_cancelled_area_list_c>();
}
const crit_diagnostics_s& pws_cancel_resp_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void pws_cancel_resp_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::msg_id:
      j.write_str("BIT STRING", c.get<fixed_bitstring<16, false, true> >().to_string());
      break;
    case types::serial_num:
      j.write_str("BIT STRING", c.get<fixed_bitstring<16, false, true> >().to_string());
      break;
    case types::broadcast_cancelled_area_list:
      j.write_fieldname("BroadcastCancelledAreaList");
      c.get<broadcast_cancelled_area_list_c>().to_json(j);
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "pws_cancel_resp_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE pws_cancel_resp_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::msg_id:
      HANDLE_CODE((c.get<fixed_bitstring<16, false, true> >().pack(bref)));
      break;
    case types::serial_num:
      HANDLE_CODE((c.get<fixed_bitstring<16, false, true> >().pack(bref)));
      break;
    case types::broadcast_cancelled_area_list:
      HANDLE_CODE(c.get<broadcast_cancelled_area_list_c>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pws_cancel_resp_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pws_cancel_resp_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::msg_id:
      HANDLE_CODE((c.get<fixed_bitstring<16, false, true> >().unpack(bref)));
      break;
    case types::serial_num:
      HANDLE_CODE((c.get<fixed_bitstring<16, false, true> >().unpack(bref)));
      break;
    case types::broadcast_cancelled_area_list:
      HANDLE_CODE(c.get<broadcast_cancelled_area_list_c>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pws_cancel_resp_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* pws_cancel_resp_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"BIT STRING", "BIT STRING", "BroadcastCancelledAreaList", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 4, value, "pws_cancel_resp_ies_o::value_c::types");
}

// PWSFailureIndicationIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t pws_fail_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {81, 27};
  return map_enum_number(options, 2, idx, "id");
}
bool pws_fail_ind_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {81, 27};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e pws_fail_ind_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 81:
      return crit_e::reject;
    case 27:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
pws_fail_ind_ies_o::value_c pws_fail_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 81:
      ret.set(value_c::types::pws_failed_cell_id_list);
      break;
    case 27:
      ret.set(value_c::types::global_ran_node_id);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pws_fail_ind_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 81:
      return presence_e::mandatory;
    case 27:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void pws_fail_ind_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::pws_failed_cell_id_list:
      c.destroy<pws_failed_cell_id_list_c>();
      break;
    case types::global_ran_node_id:
      c.destroy<global_ran_node_id_c>();
      break;
    default:
      break;
  }
}
void pws_fail_ind_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::pws_failed_cell_id_list:
      c.init<pws_failed_cell_id_list_c>();
      break;
    case types::global_ran_node_id:
      c.init<global_ran_node_id_c>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pws_fail_ind_ies_o::value_c");
  }
}
pws_fail_ind_ies_o::value_c::value_c(const pws_fail_ind_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::pws_failed_cell_id_list:
      c.init(other.c.get<pws_failed_cell_id_list_c>());
      break;
    case types::global_ran_node_id:
      c.init(other.c.get<global_ran_node_id_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pws_fail_ind_ies_o::value_c");
  }
}
pws_fail_ind_ies_o::value_c& pws_fail_ind_ies_o::value_c::operator=(const pws_fail_ind_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::pws_failed_cell_id_list:
      c.set(other.c.get<pws_failed_cell_id_list_c>());
      break;
    case types::global_ran_node_id:
      c.set(other.c.get<global_ran_node_id_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pws_fail_ind_ies_o::value_c");
  }

  return *this;
}
pws_failed_cell_id_list_c& pws_fail_ind_ies_o::value_c::pws_failed_cell_id_list()
{
  assert_choice_type(types::pws_failed_cell_id_list, type_, "Value");
  return c.get<pws_failed_cell_id_list_c>();
}
global_ran_node_id_c& pws_fail_ind_ies_o::value_c::global_ran_node_id()
{
  assert_choice_type(types::global_ran_node_id, type_, "Value");
  return c.get<global_ran_node_id_c>();
}
const pws_failed_cell_id_list_c& pws_fail_ind_ies_o::value_c::pws_failed_cell_id_list() const
{
  assert_choice_type(types::pws_failed_cell_id_list, type_, "Value");
  return c.get<pws_failed_cell_id_list_c>();
}
const global_ran_node_id_c& pws_fail_ind_ies_o::value_c::global_ran_node_id() const
{
  assert_choice_type(types::global_ran_node_id, type_, "Value");
  return c.get<global_ran_node_id_c>();
}
void pws_fail_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::pws_failed_cell_id_list:
      j.write_fieldname("PWSFailedCellIDList");
      c.get<pws_failed_cell_id_list_c>().to_json(j);
      break;
    case types::global_ran_node_id:
      j.write_fieldname("GlobalRANNodeID");
      c.get<global_ran_node_id_c>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "pws_fail_ind_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE pws_fail_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::pws_failed_cell_id_list:
      HANDLE_CODE(c.get<pws_failed_cell_id_list_c>().pack(bref));
      break;
    case types::global_ran_node_id:
      HANDLE_CODE(c.get<global_ran_node_id_c>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pws_fail_ind_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pws_fail_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::pws_failed_cell_id_list:
      HANDLE_CODE(c.get<pws_failed_cell_id_list_c>().unpack(bref));
      break;
    case types::global_ran_node_id:
      HANDLE_CODE(c.get<global_ran_node_id_c>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "pws_fail_ind_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* pws_fail_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"PWSFailedCellIDList", "GlobalRANNodeID"};
  return convert_enum_idx(options, 2, value, "pws_fail_ind_ies_o::value_c::types");
}

// PWSRestartIndicationIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t pws_restart_ind_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {16, 27, 104, 23};
  return map_enum_number(options, 4, idx, "id");
}
bool pws_restart_ind_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {16, 27, 104, 23};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e pws_restart_ind_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 16:
      return crit_e::reject;
    case 27:
      return crit_e::reject;
    case 104:
      return crit_e::reject;
    case 23:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
pws_restart_ind_ies_o::value_c pws_restart_ind_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 16:
      ret.set(value_c::types::cell_id_list_for_restart);
      break;
    case 27:
      ret.set(value_c::types::global_ran_node_id);
      break;
    case 104:
      ret.set(value_c::types::tai_list_for_restart);
      break;
    case 23:
      ret.set(value_c::types::emergency_area_id_list_for_restart);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e pws_restart_ind_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 16:
      return presence_e::mandatory;
    case 27:
      return presence_e::mandatory;
    case 104:
      return presence_e::mandatory;
    case 23:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void pws_restart_ind_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cell_id_list_for_restart:
      c.destroy<cell_id_list_for_restart_c>();
      break;
    case types::global_ran_node_id:
      c.destroy<global_ran_node_id_c>();
      break;
    case types::tai_list_for_restart:
      c.destroy<tai_list_for_restart_l>();
      break;
    case types::emergency_area_id_list_for_restart:
      c.destroy<emergency_area_id_list_for_restart_l>();
      break;
    default:
      break;
  }
}
void pws_restart_ind_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cell_id_list_for_restart:
      c.init<cell_id_list_for_restart_c>();
      break;
    case types::global_ran_node_id:
      c.init<global_ran_node_id_c>();
      break;
    case types::tai_list_for_restart:
      c.init<tai_list_for_restart_l>();
      break;
    case types::emergency_area_id_list_for_restart:
      c.init<emergency_area_id_list_for_restart_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pws_restart_ind_ies_o::value_c");
  }
}
pws_restart_ind_ies_o::value_c::value_c(const pws_restart_ind_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cell_id_list_for_restart:
      c.init(other.c.get<cell_id_list_for_restart_c>());
      break;
    case types::global_ran_node_id:
      c.init(other.c.get<global_ran_node_id_c>());
      break;
    case types::tai_list_for_restart:
      c.init(other.c.get<tai_list_for_restart_l>());
      break;
    case types::emergency_area_id_list_for_restart:
      c.init(other.c.get<emergency_area_id_list_for_restart_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pws_restart_ind_ies_o::value_c");
  }
}
pws_restart_ind_ies_o::value_c& pws_restart_ind_ies_o::value_c::operator=(const pws_restart_ind_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cell_id_list_for_restart:
      c.set(other.c.get<cell_id_list_for_restart_c>());
      break;
    case types::global_ran_node_id:
      c.set(other.c.get<global_ran_node_id_c>());
      break;
    case types::tai_list_for_restart:
      c.set(other.c.get<tai_list_for_restart_l>());
      break;
    case types::emergency_area_id_list_for_restart:
      c.set(other.c.get<emergency_area_id_list_for_restart_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "pws_restart_ind_ies_o::value_c");
  }

  return *this;
}
cell_id_list_for_restart_c& pws_restart_ind_ies_o::value_c::cell_id_list_for_restart()
{
  assert_choice_type(types::cell_id_list_for_restart, type_, "Value");
  return c.get<cell_id_list_for_restart_c>();
}
global_ran_node_id_c& pws_restart_ind_ies_o::value_c::global_ran_node_id()
{
  assert_choice_type(types::global_ran_node_id, type_, "Value");
  return c.get<global_ran_node_id_c>();
}
tai_list_for_restart_l& pws_restart_ind_ies_o::value_c::tai_list_for_restart()
{
  assert_choice_type(types::tai_list_for_restart, type_, "Value");
  return c.get<tai_list_for_restart_l>();
}
emergency_area_id_list_for_restart_l& pws_restart_ind_ies_o::value_c::emergency_area_id_list_for_restart()
{
  assert_choice_type(types::emergency_area_id_list_for_restart, type_, "Value");
  return c.get<emergency_area_id_list_for_restart_l>();
}
const cell_id_list_for_restart_c& pws_restart_ind_ies_o::value_c::cell_id_list_for_restart() const
{
  assert_choice_type(types::cell_id_list_for_restart, type_, "Value");
  return c.get<cell_id_list_for_restart_c>();
}
const global_ran_node_id_c& pws_restart_ind_ies_o::value_c::global_ran_node_id() const
{
  assert_choice_type(types::global_ran_node_id, type_, "Value");
  return c.get<global_ran_node_id_c>();
}
const tai_list_for_restart_l& pws_restart_ind_ies_o::value_c::tai_list_for_restart() const
{
  assert_choice_type(types::tai_list_for_restart, type_, "Value");
  return c.get<tai_list_for_restart_l>();
}
const emergency_area_id_list_for_restart_l& pws_restart_ind_ies_o::value_c::emergency_area_id_list_for_restart() const
{
  assert_choice_type(types::emergency_area_id_list_for_restart, type_, "Value");
  return c.get<emergency_area_id_list_for_restart_l>();
}
void pws_restart_ind_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cell_id_list_for_restart:
      j.write_fieldname("CellIDListForRestart");
      c.get<cell_id_list_for_restart_c>().to_json(j);
      break;
    case types::global_ran_node_id:
      j.write_fieldname("GlobalRANNodeID");
      c.get<global_ran_node_id_c>().to_json(j);
      break;
    case types::tai_list_for_restart:
      j.start_array("TAIListForRestart");
      for (const auto& e1 : c.get<tai_list_for_restart_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::emergency_area_id_list_for_restart:
      j.start_array("EmergencyAreaIDListForRestart");
      for (const auto& e1 : c.get<emergency_area_id_list_for_restart_l>()) {
        j.write_str(e1.to_string());
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "pws_restart_ind_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE pws_restart_ind_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::cell_id_list_for_restart:
      HANDLE_CODE(c.get<cell_id_list_for_restart_c>().pack(bref));
      break;
    case types::global_ran_node_id:
      HANDLE_CODE(c.get<global_ran_node_id_c>().pack(bref));
      break;
    case types::tai_list_for_restart:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<tai_list_for_restart_l>(), 1, 2048, true));
      break;
    case types::emergency_area_id_list_for_restart:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<emergency_area_id_list_for_restart_l>(), 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "pws_restart_ind_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE pws_restart_ind_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::cell_id_list_for_restart:
      HANDLE_CODE(c.get<cell_id_list_for_restart_c>().unpack(bref));
      break;
    case types::global_ran_node_id:
      HANDLE_CODE(c.get<global_ran_node_id_c>().unpack(bref));
      break;
    case types::tai_list_for_restart:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<tai_list_for_restart_l>(), bref, 1, 2048, true));
      break;
    case types::emergency_area_id_list_for_restart:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<emergency_area_id_list_for_restart_l>(), bref, 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "pws_restart_ind_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* pws_restart_ind_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "CellIDListForRestart", "GlobalRANNodeID", "TAIListForRestart", "EmergencyAreaIDListForRestart"};
  return convert_enum_idx(options, 4, value, "pws_restart_ind_ies_o::value_c::types");
}

// PagingIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t paging_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {115, 50, 103, 52, 118, 51, 11};
  return map_enum_number(options, 7, idx, "id");
}
bool paging_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {115, 50, 103, 52, 118, 51, 11};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e paging_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 115:
      return crit_e::ignore;
    case 50:
      return crit_e::ignore;
    case 103:
      return crit_e::ignore;
    case 52:
      return crit_e::ignore;
    case 118:
      return crit_e::ignore;
    case 51:
      return crit_e::ignore;
    case 11:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
paging_ies_o::value_c paging_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 115:
      ret.set(value_c::types::ue_paging_id);
      break;
    case 50:
      ret.set(value_c::types::paging_drx);
      break;
    case 103:
      ret.set(value_c::types::tai_list_for_paging);
      break;
    case 52:
      ret.set(value_c::types::paging_prio);
      break;
    case 118:
      ret.set(value_c::types::ue_radio_cap_for_paging);
      break;
    case 51:
      ret.set(value_c::types::paging_origin);
      break;
    case 11:
      ret.set(value_c::types::assist_data_for_paging);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e paging_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 115:
      return presence_e::mandatory;
    case 50:
      return presence_e::optional;
    case 103:
      return presence_e::mandatory;
    case 52:
      return presence_e::optional;
    case 118:
      return presence_e::optional;
    case 51:
      return presence_e::optional;
    case 11:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void paging_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ue_paging_id:
      c.destroy<ue_paging_id_c>();
      break;
    case types::tai_list_for_paging:
      c.destroy<tai_list_for_paging_l>();
      break;
    case types::ue_radio_cap_for_paging:
      c.destroy<ue_radio_cap_for_paging_s>();
      break;
    case types::assist_data_for_paging:
      c.destroy<assist_data_for_paging_s>();
      break;
    default:
      break;
  }
}
void paging_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::ue_paging_id:
      c.init<ue_paging_id_c>();
      break;
    case types::paging_drx:
      break;
    case types::tai_list_for_paging:
      c.init<tai_list_for_paging_l>();
      break;
    case types::paging_prio:
      break;
    case types::ue_radio_cap_for_paging:
      c.init<ue_radio_cap_for_paging_s>();
      break;
    case types::paging_origin:
      break;
    case types::assist_data_for_paging:
      c.init<assist_data_for_paging_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "paging_ies_o::value_c");
  }
}
paging_ies_o::value_c::value_c(const paging_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::ue_paging_id:
      c.init(other.c.get<ue_paging_id_c>());
      break;
    case types::paging_drx:
      c.init(other.c.get<paging_drx_e>());
      break;
    case types::tai_list_for_paging:
      c.init(other.c.get<tai_list_for_paging_l>());
      break;
    case types::paging_prio:
      c.init(other.c.get<paging_prio_e>());
      break;
    case types::ue_radio_cap_for_paging:
      c.init(other.c.get<ue_radio_cap_for_paging_s>());
      break;
    case types::paging_origin:
      c.init(other.c.get<paging_origin_e>());
      break;
    case types::assist_data_for_paging:
      c.init(other.c.get<assist_data_for_paging_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "paging_ies_o::value_c");
  }
}
paging_ies_o::value_c& paging_ies_o::value_c::operator=(const paging_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::ue_paging_id:
      c.set(other.c.get<ue_paging_id_c>());
      break;
    case types::paging_drx:
      c.set(other.c.get<paging_drx_e>());
      break;
    case types::tai_list_for_paging:
      c.set(other.c.get<tai_list_for_paging_l>());
      break;
    case types::paging_prio:
      c.set(other.c.get<paging_prio_e>());
      break;
    case types::ue_radio_cap_for_paging:
      c.set(other.c.get<ue_radio_cap_for_paging_s>());
      break;
    case types::paging_origin:
      c.set(other.c.get<paging_origin_e>());
      break;
    case types::assist_data_for_paging:
      c.set(other.c.get<assist_data_for_paging_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "paging_ies_o::value_c");
  }

  return *this;
}
ue_paging_id_c& paging_ies_o::value_c::ue_paging_id()
{
  assert_choice_type(types::ue_paging_id, type_, "Value");
  return c.get<ue_paging_id_c>();
}
paging_drx_e& paging_ies_o::value_c::paging_drx()
{
  assert_choice_type(types::paging_drx, type_, "Value");
  return c.get<paging_drx_e>();
}
tai_list_for_paging_l& paging_ies_o::value_c::tai_list_for_paging()
{
  assert_choice_type(types::tai_list_for_paging, type_, "Value");
  return c.get<tai_list_for_paging_l>();
}
paging_prio_e& paging_ies_o::value_c::paging_prio()
{
  assert_choice_type(types::paging_prio, type_, "Value");
  return c.get<paging_prio_e>();
}
ue_radio_cap_for_paging_s& paging_ies_o::value_c::ue_radio_cap_for_paging()
{
  assert_choice_type(types::ue_radio_cap_for_paging, type_, "Value");
  return c.get<ue_radio_cap_for_paging_s>();
}
paging_origin_e& paging_ies_o::value_c::paging_origin()
{
  assert_choice_type(types::paging_origin, type_, "Value");
  return c.get<paging_origin_e>();
}
assist_data_for_paging_s& paging_ies_o::value_c::assist_data_for_paging()
{
  assert_choice_type(types::assist_data_for_paging, type_, "Value");
  return c.get<assist_data_for_paging_s>();
}
const ue_paging_id_c& paging_ies_o::value_c::ue_paging_id() const
{
  assert_choice_type(types::ue_paging_id, type_, "Value");
  return c.get<ue_paging_id_c>();
}
const paging_drx_e& paging_ies_o::value_c::paging_drx() const
{
  assert_choice_type(types::paging_drx, type_, "Value");
  return c.get<paging_drx_e>();
}
const tai_list_for_paging_l& paging_ies_o::value_c::tai_list_for_paging() const
{
  assert_choice_type(types::tai_list_for_paging, type_, "Value");
  return c.get<tai_list_for_paging_l>();
}
const paging_prio_e& paging_ies_o::value_c::paging_prio() const
{
  assert_choice_type(types::paging_prio, type_, "Value");
  return c.get<paging_prio_e>();
}
const ue_radio_cap_for_paging_s& paging_ies_o::value_c::ue_radio_cap_for_paging() const
{
  assert_choice_type(types::ue_radio_cap_for_paging, type_, "Value");
  return c.get<ue_radio_cap_for_paging_s>();
}
const paging_origin_e& paging_ies_o::value_c::paging_origin() const
{
  assert_choice_type(types::paging_origin, type_, "Value");
  return c.get<paging_origin_e>();
}
const assist_data_for_paging_s& paging_ies_o::value_c::assist_data_for_paging() const
{
  assert_choice_type(types::assist_data_for_paging, type_, "Value");
  return c.get<assist_data_for_paging_s>();
}
void paging_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::ue_paging_id:
      j.write_fieldname("UEPagingIdentity");
      c.get<ue_paging_id_c>().to_json(j);
      break;
    case types::paging_drx:
      j.write_str("PagingDRX", c.get<paging_drx_e>().to_string());
      break;
    case types::tai_list_for_paging:
      j.start_array("TAIListForPaging");
      for (const auto& e1 : c.get<tai_list_for_paging_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::paging_prio:
      j.write_str("PagingPriority", c.get<paging_prio_e>().to_string());
      break;
    case types::ue_radio_cap_for_paging:
      j.write_fieldname("UERadioCapabilityForPaging");
      c.get<ue_radio_cap_for_paging_s>().to_json(j);
      break;
    case types::paging_origin:
      j.write_str("PagingOrigin", "non-3gpp");
      break;
    case types::assist_data_for_paging:
      j.write_fieldname("AssistanceDataForPaging");
      c.get<assist_data_for_paging_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "paging_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE paging_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::ue_paging_id:
      HANDLE_CODE(c.get<ue_paging_id_c>().pack(bref));
      break;
    case types::paging_drx:
      HANDLE_CODE(c.get<paging_drx_e>().pack(bref));
      break;
    case types::tai_list_for_paging:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<tai_list_for_paging_l>(), 1, 16, true));
      break;
    case types::paging_prio:
      HANDLE_CODE(c.get<paging_prio_e>().pack(bref));
      break;
    case types::ue_radio_cap_for_paging:
      HANDLE_CODE(c.get<ue_radio_cap_for_paging_s>().pack(bref));
      break;
    case types::paging_origin:
      HANDLE_CODE(c.get<paging_origin_e>().pack(bref));
      break;
    case types::assist_data_for_paging:
      HANDLE_CODE(c.get<assist_data_for_paging_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "paging_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE paging_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::ue_paging_id:
      HANDLE_CODE(c.get<ue_paging_id_c>().unpack(bref));
      break;
    case types::paging_drx:
      HANDLE_CODE(c.get<paging_drx_e>().unpack(bref));
      break;
    case types::tai_list_for_paging:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<tai_list_for_paging_l>(), bref, 1, 16, true));
      break;
    case types::paging_prio:
      HANDLE_CODE(c.get<paging_prio_e>().unpack(bref));
      break;
    case types::ue_radio_cap_for_paging:
      HANDLE_CODE(c.get<ue_radio_cap_for_paging_s>().unpack(bref));
      break;
    case types::paging_origin:
      HANDLE_CODE(c.get<paging_origin_e>().unpack(bref));
      break;
    case types::assist_data_for_paging:
      HANDLE_CODE(c.get<assist_data_for_paging_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "paging_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* paging_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"UEPagingIdentity",
                                  "PagingDRX",
                                  "TAIListForPaging",
                                  "PagingPriority",
                                  "UERadioCapabilityForPaging",
                                  "PagingOrigin",
                                  "AssistanceDataForPaging"};
  return convert_enum_idx(options, 7, value, "paging_ies_o::value_c::types");
}

// PathSwitchRequestAcknowledgeIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t path_switch_request_ack_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 119, 93, 41, 77, 68, 0, 18, 91, 19, 146};
  return map_enum_number(options, 12, idx, "id");
}
bool path_switch_request_ack_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 119, 93, 41, 77, 68, 0, 18, 91, 19, 146};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e path_switch_request_ack_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::ignore;
    case 85:
      return crit_e::ignore;
    case 119:
      return crit_e::reject;
    case 93:
      return crit_e::reject;
    case 41:
      return crit_e::reject;
    case 77:
      return crit_e::ignore;
    case 68:
      return crit_e::ignore;
    case 0:
      return crit_e::reject;
    case 18:
      return crit_e::ignore;
    case 91:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    case 146:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
path_switch_request_ack_ies_o::value_c path_switch_request_ack_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 119:
      ret.set(value_c::types::ue_security_cap);
      break;
    case 93:
      ret.set(value_c::types::security_context);
      break;
    case 41:
      ret.set(value_c::types::new_security_context_ind);
      break;
    case 77:
      ret.set(value_c::types::pdu_session_res_switched_list);
      break;
    case 68:
      ret.set(value_c::types::pdu_session_res_released_list_ps_ack);
      break;
    case 0:
      ret.set(value_c::types::allowed_nssai);
      break;
    case 18:
      ret.set(value_c::types::core_network_assist_info);
      break;
    case 91:
      ret.set(value_c::types::rrc_inactive_transition_report_request);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    case 146:
      ret.set(value_c::types::redirection_voice_fallback);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e path_switch_request_ack_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 119:
      return presence_e::optional;
    case 93:
      return presence_e::mandatory;
    case 41:
      return presence_e::optional;
    case 77:
      return presence_e::mandatory;
    case 68:
      return presence_e::optional;
    case 0:
      return presence_e::mandatory;
    case 18:
      return presence_e::optional;
    case 91:
      return presence_e::optional;
    case 19:
      return presence_e::optional;
    case 146:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void path_switch_request_ack_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ue_security_cap:
      c.destroy<ue_security_cap_s>();
      break;
    case types::security_context:
      c.destroy<security_context_s>();
      break;
    case types::pdu_session_res_switched_list:
      c.destroy<pdu_session_res_switched_list_l>();
      break;
    case types::pdu_session_res_released_list_ps_ack:
      c.destroy<pdu_session_res_released_list_ps_ack_l>();
      break;
    case types::allowed_nssai:
      c.destroy<allowed_nssai_l>();
      break;
    case types::core_network_assist_info:
      c.destroy<core_network_assist_info_s>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void path_switch_request_ack_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::ue_security_cap:
      c.init<ue_security_cap_s>();
      break;
    case types::security_context:
      c.init<security_context_s>();
      break;
    case types::new_security_context_ind:
      break;
    case types::pdu_session_res_switched_list:
      c.init<pdu_session_res_switched_list_l>();
      break;
    case types::pdu_session_res_released_list_ps_ack:
      c.init<pdu_session_res_released_list_ps_ack_l>();
      break;
    case types::allowed_nssai:
      c.init<allowed_nssai_l>();
      break;
    case types::core_network_assist_info:
      c.init<core_network_assist_info_s>();
      break;
    case types::rrc_inactive_transition_report_request:
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::redirection_voice_fallback:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_ack_ies_o::value_c");
  }
}
path_switch_request_ack_ies_o::value_c::value_c(const path_switch_request_ack_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ue_security_cap:
      c.init(other.c.get<ue_security_cap_s>());
      break;
    case types::security_context:
      c.init(other.c.get<security_context_s>());
      break;
    case types::new_security_context_ind:
      c.init(other.c.get<new_security_context_ind_e>());
      break;
    case types::pdu_session_res_switched_list:
      c.init(other.c.get<pdu_session_res_switched_list_l>());
      break;
    case types::pdu_session_res_released_list_ps_ack:
      c.init(other.c.get<pdu_session_res_released_list_ps_ack_l>());
      break;
    case types::allowed_nssai:
      c.init(other.c.get<allowed_nssai_l>());
      break;
    case types::core_network_assist_info:
      c.init(other.c.get<core_network_assist_info_s>());
      break;
    case types::rrc_inactive_transition_report_request:
      c.init(other.c.get<rrc_inactive_transition_report_request_e>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::redirection_voice_fallback:
      c.init(other.c.get<redirection_voice_fallback_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_ack_ies_o::value_c");
  }
}
path_switch_request_ack_ies_o::value_c&
path_switch_request_ack_ies_o::value_c::operator=(const path_switch_request_ack_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ue_security_cap:
      c.set(other.c.get<ue_security_cap_s>());
      break;
    case types::security_context:
      c.set(other.c.get<security_context_s>());
      break;
    case types::new_security_context_ind:
      c.set(other.c.get<new_security_context_ind_e>());
      break;
    case types::pdu_session_res_switched_list:
      c.set(other.c.get<pdu_session_res_switched_list_l>());
      break;
    case types::pdu_session_res_released_list_ps_ack:
      c.set(other.c.get<pdu_session_res_released_list_ps_ack_l>());
      break;
    case types::allowed_nssai:
      c.set(other.c.get<allowed_nssai_l>());
      break;
    case types::core_network_assist_info:
      c.set(other.c.get<core_network_assist_info_s>());
      break;
    case types::rrc_inactive_transition_report_request:
      c.set(other.c.get<rrc_inactive_transition_report_request_e>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::redirection_voice_fallback:
      c.set(other.c.get<redirection_voice_fallback_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_ack_ies_o::value_c");
  }

  return *this;
}
uint64_t& path_switch_request_ack_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& path_switch_request_ack_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
ue_security_cap_s& path_switch_request_ack_ies_o::value_c::ue_security_cap()
{
  assert_choice_type(types::ue_security_cap, type_, "Value");
  return c.get<ue_security_cap_s>();
}
security_context_s& path_switch_request_ack_ies_o::value_c::security_context()
{
  assert_choice_type(types::security_context, type_, "Value");
  return c.get<security_context_s>();
}
new_security_context_ind_e& path_switch_request_ack_ies_o::value_c::new_security_context_ind()
{
  assert_choice_type(types::new_security_context_ind, type_, "Value");
  return c.get<new_security_context_ind_e>();
}
pdu_session_res_switched_list_l& path_switch_request_ack_ies_o::value_c::pdu_session_res_switched_list()
{
  assert_choice_type(types::pdu_session_res_switched_list, type_, "Value");
  return c.get<pdu_session_res_switched_list_l>();
}
pdu_session_res_released_list_ps_ack_l& path_switch_request_ack_ies_o::value_c::pdu_session_res_released_list_ps_ack()
{
  assert_choice_type(types::pdu_session_res_released_list_ps_ack, type_, "Value");
  return c.get<pdu_session_res_released_list_ps_ack_l>();
}
allowed_nssai_l& path_switch_request_ack_ies_o::value_c::allowed_nssai()
{
  assert_choice_type(types::allowed_nssai, type_, "Value");
  return c.get<allowed_nssai_l>();
}
core_network_assist_info_s& path_switch_request_ack_ies_o::value_c::core_network_assist_info()
{
  assert_choice_type(types::core_network_assist_info, type_, "Value");
  return c.get<core_network_assist_info_s>();
}
rrc_inactive_transition_report_request_e&
path_switch_request_ack_ies_o::value_c::rrc_inactive_transition_report_request()
{
  assert_choice_type(types::rrc_inactive_transition_report_request, type_, "Value");
  return c.get<rrc_inactive_transition_report_request_e>();
}
crit_diagnostics_s& path_switch_request_ack_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
redirection_voice_fallback_e& path_switch_request_ack_ies_o::value_c::redirection_voice_fallback()
{
  assert_choice_type(types::redirection_voice_fallback, type_, "Value");
  return c.get<redirection_voice_fallback_e>();
}
const uint64_t& path_switch_request_ack_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& path_switch_request_ack_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const ue_security_cap_s& path_switch_request_ack_ies_o::value_c::ue_security_cap() const
{
  assert_choice_type(types::ue_security_cap, type_, "Value");
  return c.get<ue_security_cap_s>();
}
const security_context_s& path_switch_request_ack_ies_o::value_c::security_context() const
{
  assert_choice_type(types::security_context, type_, "Value");
  return c.get<security_context_s>();
}
const new_security_context_ind_e& path_switch_request_ack_ies_o::value_c::new_security_context_ind() const
{
  assert_choice_type(types::new_security_context_ind, type_, "Value");
  return c.get<new_security_context_ind_e>();
}
const pdu_session_res_switched_list_l& path_switch_request_ack_ies_o::value_c::pdu_session_res_switched_list() const
{
  assert_choice_type(types::pdu_session_res_switched_list, type_, "Value");
  return c.get<pdu_session_res_switched_list_l>();
}
const pdu_session_res_released_list_ps_ack_l&
path_switch_request_ack_ies_o::value_c::pdu_session_res_released_list_ps_ack() const
{
  assert_choice_type(types::pdu_session_res_released_list_ps_ack, type_, "Value");
  return c.get<pdu_session_res_released_list_ps_ack_l>();
}
const allowed_nssai_l& path_switch_request_ack_ies_o::value_c::allowed_nssai() const
{
  assert_choice_type(types::allowed_nssai, type_, "Value");
  return c.get<allowed_nssai_l>();
}
const core_network_assist_info_s& path_switch_request_ack_ies_o::value_c::core_network_assist_info() const
{
  assert_choice_type(types::core_network_assist_info, type_, "Value");
  return c.get<core_network_assist_info_s>();
}
const rrc_inactive_transition_report_request_e&
path_switch_request_ack_ies_o::value_c::rrc_inactive_transition_report_request() const
{
  assert_choice_type(types::rrc_inactive_transition_report_request, type_, "Value");
  return c.get<rrc_inactive_transition_report_request_e>();
}
const crit_diagnostics_s& path_switch_request_ack_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const redirection_voice_fallback_e& path_switch_request_ack_ies_o::value_c::redirection_voice_fallback() const
{
  assert_choice_type(types::redirection_voice_fallback, type_, "Value");
  return c.get<redirection_voice_fallback_e>();
}
void path_switch_request_ack_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::ue_security_cap:
      j.write_fieldname("UESecurityCapabilities");
      c.get<ue_security_cap_s>().to_json(j);
      break;
    case types::security_context:
      j.write_fieldname("SecurityContext");
      c.get<security_context_s>().to_json(j);
      break;
    case types::new_security_context_ind:
      j.write_str("NewSecurityContextInd", "true");
      break;
    case types::pdu_session_res_switched_list:
      j.start_array("PDUSessionResourceSwitchedList");
      for (const auto& e1 : c.get<pdu_session_res_switched_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::pdu_session_res_released_list_ps_ack:
      j.start_array("PDUSessionResourceReleasedListPSAck");
      for (const auto& e1 : c.get<pdu_session_res_released_list_ps_ack_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::allowed_nssai:
      j.start_array("AllowedNSSAI");
      for (const auto& e1 : c.get<allowed_nssai_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::core_network_assist_info:
      j.write_fieldname("CoreNetworkAssistanceInformation");
      c.get<core_network_assist_info_s>().to_json(j);
      break;
    case types::rrc_inactive_transition_report_request:
      j.write_str("RRCInactiveTransitionReportRequest", c.get<rrc_inactive_transition_report_request_e>().to_string());
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    case types::redirection_voice_fallback:
      j.write_str("RedirectionVoiceFallback", c.get<redirection_voice_fallback_e>().to_string());
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_ack_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE path_switch_request_ack_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::ue_security_cap:
      HANDLE_CODE(c.get<ue_security_cap_s>().pack(bref));
      break;
    case types::security_context:
      HANDLE_CODE(c.get<security_context_s>().pack(bref));
      break;
    case types::new_security_context_ind:
      HANDLE_CODE(c.get<new_security_context_ind_e>().pack(bref));
      break;
    case types::pdu_session_res_switched_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_switched_list_l>(), 1, 256, true));
      break;
    case types::pdu_session_res_released_list_ps_ack:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_released_list_ps_ack_l>(), 1, 256, true));
      break;
    case types::allowed_nssai:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<allowed_nssai_l>(), 1, 8, true));
      break;
    case types::core_network_assist_info:
      HANDLE_CODE(c.get<core_network_assist_info_s>().pack(bref));
      break;
    case types::rrc_inactive_transition_report_request:
      HANDLE_CODE(c.get<rrc_inactive_transition_report_request_e>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    case types::redirection_voice_fallback:
      HANDLE_CODE(c.get<redirection_voice_fallback_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_ack_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE path_switch_request_ack_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::ue_security_cap:
      HANDLE_CODE(c.get<ue_security_cap_s>().unpack(bref));
      break;
    case types::security_context:
      HANDLE_CODE(c.get<security_context_s>().unpack(bref));
      break;
    case types::new_security_context_ind:
      HANDLE_CODE(c.get<new_security_context_ind_e>().unpack(bref));
      break;
    case types::pdu_session_res_switched_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_switched_list_l>(), bref, 1, 256, true));
      break;
    case types::pdu_session_res_released_list_ps_ack:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_released_list_ps_ack_l>(), bref, 1, 256, true));
      break;
    case types::allowed_nssai:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<allowed_nssai_l>(), bref, 1, 8, true));
      break;
    case types::core_network_assist_info:
      HANDLE_CODE(c.get<core_network_assist_info_s>().unpack(bref));
      break;
    case types::rrc_inactive_transition_report_request:
      HANDLE_CODE(c.get<rrc_inactive_transition_report_request_e>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    case types::redirection_voice_fallback:
      HANDLE_CODE(c.get<redirection_voice_fallback_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_ack_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* path_switch_request_ack_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "UESecurityCapabilities",
                                  "SecurityContext",
                                  "NewSecurityContextInd",
                                  "PDUSessionResourceSwitchedList",
                                  "PDUSessionResourceReleasedListPSAck",
                                  "AllowedNSSAI",
                                  "CoreNetworkAssistanceInformation",
                                  "RRCInactiveTransitionReportRequest",
                                  "CriticalityDiagnostics",
                                  "RedirectionVoiceFallback"};
  return convert_enum_idx(options, 12, value, "path_switch_request_ack_ies_o::value_c::types");
}

// PathSwitchRequestFailureIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t path_switch_request_fail_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 69, 19};
  return map_enum_number(options, 4, idx, "id");
}
bool path_switch_request_fail_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 69, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e path_switch_request_fail_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::ignore;
    case 85:
      return crit_e::ignore;
    case 69:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
path_switch_request_fail_ies_o::value_c path_switch_request_fail_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 69:
      ret.set(value_c::types::pdu_session_res_released_list_ps_fail);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e path_switch_request_fail_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 69:
      return presence_e::mandatory;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void path_switch_request_fail_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::pdu_session_res_released_list_ps_fail:
      c.destroy<pdu_session_res_released_list_ps_fail_l>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void path_switch_request_fail_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::pdu_session_res_released_list_ps_fail:
      c.init<pdu_session_res_released_list_ps_fail_l>();
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_fail_ies_o::value_c");
  }
}
path_switch_request_fail_ies_o::value_c::value_c(const path_switch_request_fail_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_released_list_ps_fail:
      c.init(other.c.get<pdu_session_res_released_list_ps_fail_l>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_fail_ies_o::value_c");
  }
}
path_switch_request_fail_ies_o::value_c&
path_switch_request_fail_ies_o::value_c::operator=(const path_switch_request_fail_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_released_list_ps_fail:
      c.set(other.c.get<pdu_session_res_released_list_ps_fail_l>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_fail_ies_o::value_c");
  }

  return *this;
}
uint64_t& path_switch_request_fail_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& path_switch_request_fail_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
pdu_session_res_released_list_ps_fail_l&
path_switch_request_fail_ies_o::value_c::pdu_session_res_released_list_ps_fail()
{
  assert_choice_type(types::pdu_session_res_released_list_ps_fail, type_, "Value");
  return c.get<pdu_session_res_released_list_ps_fail_l>();
}
crit_diagnostics_s& path_switch_request_fail_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const uint64_t& path_switch_request_fail_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& path_switch_request_fail_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const pdu_session_res_released_list_ps_fail_l&
path_switch_request_fail_ies_o::value_c::pdu_session_res_released_list_ps_fail() const
{
  assert_choice_type(types::pdu_session_res_released_list_ps_fail, type_, "Value");
  return c.get<pdu_session_res_released_list_ps_fail_l>();
}
const crit_diagnostics_s& path_switch_request_fail_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void path_switch_request_fail_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::pdu_session_res_released_list_ps_fail:
      j.start_array("PDUSessionResourceReleasedListPSFail");
      for (const auto& e1 : c.get<pdu_session_res_released_list_ps_fail_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_fail_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE path_switch_request_fail_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_released_list_ps_fail:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_released_list_ps_fail_l>(), 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_fail_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE path_switch_request_fail_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_released_list_ps_fail:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_released_list_ps_fail_l>(), bref, 1, 256, true));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_fail_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* path_switch_request_fail_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "PDUSessionResourceReleasedListPSFail",
                                  "CriticalityDiagnostics"};
  return convert_enum_idx(options, 4, value, "path_switch_request_fail_ies_o::value_c::types");
}

// PathSwitchRequestIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t path_switch_request_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {85, 100, 121, 119, 76, 57};
  return map_enum_number(options, 6, idx, "id");
}
bool path_switch_request_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {85, 100, 121, 119, 76, 57};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e path_switch_request_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 85:
      return crit_e::reject;
    case 100:
      return crit_e::reject;
    case 121:
      return crit_e::ignore;
    case 119:
      return crit_e::ignore;
    case 76:
      return crit_e::reject;
    case 57:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
path_switch_request_ies_o::value_c path_switch_request_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 100:
      ret.set(value_c::types::source_amf_ue_ngap_id);
      break;
    case 121:
      ret.set(value_c::types::user_location_info);
      break;
    case 119:
      ret.set(value_c::types::ue_security_cap);
      break;
    case 76:
      ret.set(value_c::types::pdu_session_res_to_be_switched_dl_list);
      break;
    case 57:
      ret.set(value_c::types::pdu_session_res_failed_to_setup_list_ps_req);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e path_switch_request_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 85:
      return presence_e::mandatory;
    case 100:
      return presence_e::mandatory;
    case 121:
      return presence_e::mandatory;
    case 119:
      return presence_e::mandatory;
    case 76:
      return presence_e::mandatory;
    case 57:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void path_switch_request_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::user_location_info:
      c.destroy<user_location_info_c>();
      break;
    case types::ue_security_cap:
      c.destroy<ue_security_cap_s>();
      break;
    case types::pdu_session_res_to_be_switched_dl_list:
      c.destroy<pdu_session_res_to_be_switched_dl_list_l>();
      break;
    case types::pdu_session_res_failed_to_setup_list_ps_req:
      c.destroy<pdu_session_res_failed_to_setup_list_ps_req_l>();
      break;
    default:
      break;
  }
}
void path_switch_request_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::ran_ue_ngap_id:
      break;
    case types::source_amf_ue_ngap_id:
      break;
    case types::user_location_info:
      c.init<user_location_info_c>();
      break;
    case types::ue_security_cap:
      c.init<ue_security_cap_s>();
      break;
    case types::pdu_session_res_to_be_switched_dl_list:
      c.init<pdu_session_res_to_be_switched_dl_list_l>();
      break;
    case types::pdu_session_res_failed_to_setup_list_ps_req:
      c.init<pdu_session_res_failed_to_setup_list_ps_req_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_ies_o::value_c");
  }
}
path_switch_request_ies_o::value_c::value_c(const path_switch_request_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::source_amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::user_location_info:
      c.init(other.c.get<user_location_info_c>());
      break;
    case types::ue_security_cap:
      c.init(other.c.get<ue_security_cap_s>());
      break;
    case types::pdu_session_res_to_be_switched_dl_list:
      c.init(other.c.get<pdu_session_res_to_be_switched_dl_list_l>());
      break;
    case types::pdu_session_res_failed_to_setup_list_ps_req:
      c.init(other.c.get<pdu_session_res_failed_to_setup_list_ps_req_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_ies_o::value_c");
  }
}
path_switch_request_ies_o::value_c&
path_switch_request_ies_o::value_c::operator=(const path_switch_request_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::source_amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::user_location_info:
      c.set(other.c.get<user_location_info_c>());
      break;
    case types::ue_security_cap:
      c.set(other.c.get<ue_security_cap_s>());
      break;
    case types::pdu_session_res_to_be_switched_dl_list:
      c.set(other.c.get<pdu_session_res_to_be_switched_dl_list_l>());
      break;
    case types::pdu_session_res_failed_to_setup_list_ps_req:
      c.set(other.c.get<pdu_session_res_failed_to_setup_list_ps_req_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_ies_o::value_c");
  }

  return *this;
}
uint64_t& path_switch_request_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& path_switch_request_ies_o::value_c::source_amf_ue_ngap_id()
{
  assert_choice_type(types::source_amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
user_location_info_c& path_switch_request_ies_o::value_c::user_location_info()
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_c>();
}
ue_security_cap_s& path_switch_request_ies_o::value_c::ue_security_cap()
{
  assert_choice_type(types::ue_security_cap, type_, "Value");
  return c.get<ue_security_cap_s>();
}
pdu_session_res_to_be_switched_dl_list_l& path_switch_request_ies_o::value_c::pdu_session_res_to_be_switched_dl_list()
{
  assert_choice_type(types::pdu_session_res_to_be_switched_dl_list, type_, "Value");
  return c.get<pdu_session_res_to_be_switched_dl_list_l>();
}
pdu_session_res_failed_to_setup_list_ps_req_l&
path_switch_request_ies_o::value_c::pdu_session_res_failed_to_setup_list_ps_req()
{
  assert_choice_type(types::pdu_session_res_failed_to_setup_list_ps_req, type_, "Value");
  return c.get<pdu_session_res_failed_to_setup_list_ps_req_l>();
}
const uint64_t& path_switch_request_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& path_switch_request_ies_o::value_c::source_amf_ue_ngap_id() const
{
  assert_choice_type(types::source_amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const user_location_info_c& path_switch_request_ies_o::value_c::user_location_info() const
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_c>();
}
const ue_security_cap_s& path_switch_request_ies_o::value_c::ue_security_cap() const
{
  assert_choice_type(types::ue_security_cap, type_, "Value");
  return c.get<ue_security_cap_s>();
}
const pdu_session_res_to_be_switched_dl_list_l&
path_switch_request_ies_o::value_c::pdu_session_res_to_be_switched_dl_list() const
{
  assert_choice_type(types::pdu_session_res_to_be_switched_dl_list, type_, "Value");
  return c.get<pdu_session_res_to_be_switched_dl_list_l>();
}
const pdu_session_res_failed_to_setup_list_ps_req_l&
path_switch_request_ies_o::value_c::pdu_session_res_failed_to_setup_list_ps_req() const
{
  assert_choice_type(types::pdu_session_res_failed_to_setup_list_ps_req, type_, "Value");
  return c.get<pdu_session_res_failed_to_setup_list_ps_req_l>();
}
void path_switch_request_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::source_amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::user_location_info:
      j.write_fieldname("UserLocationInformation");
      c.get<user_location_info_c>().to_json(j);
      break;
    case types::ue_security_cap:
      j.write_fieldname("UESecurityCapabilities");
      c.get<ue_security_cap_s>().to_json(j);
      break;
    case types::pdu_session_res_to_be_switched_dl_list:
      j.start_array("PDUSessionResourceToBeSwitchedDLList");
      for (const auto& e1 : c.get<pdu_session_res_to_be_switched_dl_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::pdu_session_res_failed_to_setup_list_ps_req:
      j.start_array("PDUSessionResourceFailedToSetupListPSReq");
      for (const auto& e1 : c.get<pdu_session_res_failed_to_setup_list_ps_req_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE path_switch_request_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::source_amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_c>().pack(bref));
      break;
    case types::ue_security_cap:
      HANDLE_CODE(c.get<ue_security_cap_s>().pack(bref));
      break;
    case types::pdu_session_res_to_be_switched_dl_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_to_be_switched_dl_list_l>(), 1, 256, true));
      break;
    case types::pdu_session_res_failed_to_setup_list_ps_req:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_failed_to_setup_list_ps_req_l>(), 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE path_switch_request_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::source_amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_c>().unpack(bref));
      break;
    case types::ue_security_cap:
      HANDLE_CODE(c.get<ue_security_cap_s>().unpack(bref));
      break;
    case types::pdu_session_res_to_be_switched_dl_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_to_be_switched_dl_list_l>(), bref, 1, 256, true));
      break;
    case types::pdu_session_res_failed_to_setup_list_ps_req:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_failed_to_setup_list_ps_req_l>(), bref, 1, 256, true));
      break;
    default:
      log_invalid_choice_id(type_, "path_switch_request_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* path_switch_request_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..4294967295)",
                                  "INTEGER (0..1099511627775)",
                                  "UserLocationInformation",
                                  "UESecurityCapabilities",
                                  "PDUSessionResourceToBeSwitchedDLList",
                                  "PDUSessionResourceFailedToSetupListPSReq"};
  return convert_enum_idx(options, 6, value, "path_switch_request_ies_o::value_c::types");
}

// Value ::= OPEN TYPE
void ngap_private_ies_empty_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.end_obj();
}
SRSASN_CODE ngap_private_ies_empty_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  return SRSASN_SUCCESS;
}
SRSASN_CODE ngap_private_ies_empty_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  return SRSASN_SUCCESS;
}

const char* ngap_private_ies_empty_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {};
  return convert_enum_idx(options, 0, value, "ngap_private_ies_empty_o::value_c::types");
}

// RANConfigurationUpdateAcknowledgeIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ran_cfg_upd_ack_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {19};
  return map_enum_number(options, 1, idx, "id");
}
bool ran_cfg_upd_ack_ies_o::is_id_valid(const uint32_t& id)
{
  return 19 == id;
}
crit_e ran_cfg_upd_ack_ies_o::get_crit(const uint32_t& id)
{
  if (id == 19) {
    return crit_e::ignore;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}
ran_cfg_upd_ack_ies_o::value_c ran_cfg_upd_ack_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  if (id != 19) {
    asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ran_cfg_upd_ack_ies_o::get_presence(const uint32_t& id)
{
  if (id == 19) {
    return presence_e::optional;
  }
  asn1::log_error("The id=%d is not recognized", id);
  return {};
}

// Value ::= OPEN TYPE
void ran_cfg_upd_ack_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  j.write_fieldname("CriticalityDiagnostics");
  c.to_json(j);
  j.end_obj();
}
SRSASN_CODE ran_cfg_upd_ack_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.pack(bref));
  return SRSASN_SUCCESS;
}
SRSASN_CODE ran_cfg_upd_ack_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  HANDLE_CODE(c.unpack(bref));
  return SRSASN_SUCCESS;
}

const char* ran_cfg_upd_ack_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"CriticalityDiagnostics"};
  return convert_enum_idx(options, 1, value, "ran_cfg_upd_ack_ies_o::value_c::types");
}

// RANConfigurationUpdateFailureIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ran_cfg_upd_fail_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {15, 107, 19};
  return map_enum_number(options, 3, idx, "id");
}
bool ran_cfg_upd_fail_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {15, 107, 19};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ran_cfg_upd_fail_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 15:
      return crit_e::ignore;
    case 107:
      return crit_e::ignore;
    case 19:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ran_cfg_upd_fail_ies_o::value_c ran_cfg_upd_fail_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 15:
      ret.set(value_c::types::cause);
      break;
    case 107:
      ret.set(value_c::types::time_to_wait);
      break;
    case 19:
      ret.set(value_c::types::crit_diagnostics);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ran_cfg_upd_fail_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 15:
      return presence_e::mandatory;
    case 107:
      return presence_e::optional;
    case 19:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ran_cfg_upd_fail_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::cause:
      c.destroy<cause_c>();
      break;
    case types::crit_diagnostics:
      c.destroy<crit_diagnostics_s>();
      break;
    default:
      break;
  }
}
void ran_cfg_upd_fail_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::cause:
      c.init<cause_c>();
      break;
    case types::time_to_wait:
      break;
    case types::crit_diagnostics:
      c.init<crit_diagnostics_s>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ran_cfg_upd_fail_ies_o::value_c");
  }
}
ran_cfg_upd_fail_ies_o::value_c::value_c(const ran_cfg_upd_fail_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::cause:
      c.init(other.c.get<cause_c>());
      break;
    case types::time_to_wait:
      c.init(other.c.get<time_to_wait_e>());
      break;
    case types::crit_diagnostics:
      c.init(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ran_cfg_upd_fail_ies_o::value_c");
  }
}
ran_cfg_upd_fail_ies_o::value_c&
ran_cfg_upd_fail_ies_o::value_c::operator=(const ran_cfg_upd_fail_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::cause:
      c.set(other.c.get<cause_c>());
      break;
    case types::time_to_wait:
      c.set(other.c.get<time_to_wait_e>());
      break;
    case types::crit_diagnostics:
      c.set(other.c.get<crit_diagnostics_s>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ran_cfg_upd_fail_ies_o::value_c");
  }

  return *this;
}
cause_c& ran_cfg_upd_fail_ies_o::value_c::cause()
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
time_to_wait_e& ran_cfg_upd_fail_ies_o::value_c::time_to_wait()
{
  assert_choice_type(types::time_to_wait, type_, "Value");
  return c.get<time_to_wait_e>();
}
crit_diagnostics_s& ran_cfg_upd_fail_ies_o::value_c::crit_diagnostics()
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
const cause_c& ran_cfg_upd_fail_ies_o::value_c::cause() const
{
  assert_choice_type(types::cause, type_, "Value");
  return c.get<cause_c>();
}
const time_to_wait_e& ran_cfg_upd_fail_ies_o::value_c::time_to_wait() const
{
  assert_choice_type(types::time_to_wait, type_, "Value");
  return c.get<time_to_wait_e>();
}
const crit_diagnostics_s& ran_cfg_upd_fail_ies_o::value_c::crit_diagnostics() const
{
  assert_choice_type(types::crit_diagnostics, type_, "Value");
  return c.get<crit_diagnostics_s>();
}
void ran_cfg_upd_fail_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::cause:
      j.write_fieldname("Cause");
      c.get<cause_c>().to_json(j);
      break;
    case types::time_to_wait:
      j.write_str("TimeToWait", c.get<time_to_wait_e>().to_string());
      break;
    case types::crit_diagnostics:
      j.write_fieldname("CriticalityDiagnostics");
      c.get<crit_diagnostics_s>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ran_cfg_upd_fail_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ran_cfg_upd_fail_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().pack(bref));
      break;
    case types::time_to_wait:
      HANDLE_CODE(c.get<time_to_wait_e>().pack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ran_cfg_upd_fail_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ran_cfg_upd_fail_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::cause:
      HANDLE_CODE(c.get<cause_c>().unpack(bref));
      break;
    case types::time_to_wait:
      HANDLE_CODE(c.get<time_to_wait_e>().unpack(bref));
      break;
    case types::crit_diagnostics:
      HANDLE_CODE(c.get<crit_diagnostics_s>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ran_cfg_upd_fail_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ran_cfg_upd_fail_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"Cause", "TimeToWait", "CriticalityDiagnostics"};
  return convert_enum_idx(options, 3, value, "ran_cfg_upd_fail_ies_o::value_c::types");
}

// RANConfigurationUpdateIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t ran_cfg_upd_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {82, 102, 21, 27};
  return map_enum_number(options, 4, idx, "id");
}
bool ran_cfg_upd_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {82, 102, 21, 27};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e ran_cfg_upd_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 82:
      return crit_e::ignore;
    case 102:
      return crit_e::reject;
    case 21:
      return crit_e::ignore;
    case 27:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
ran_cfg_upd_ies_o::value_c ran_cfg_upd_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 82:
      ret.set(value_c::types::ran_node_name);
      break;
    case 102:
      ret.set(value_c::types::supported_ta_list);
      break;
    case 21:
      ret.set(value_c::types::default_paging_drx);
      break;
    case 27:
      ret.set(value_c::types::global_ran_node_id);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e ran_cfg_upd_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 82:
      return presence_e::optional;
    case 102:
      return presence_e::optional;
    case 21:
      return presence_e::optional;
    case 27:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void ran_cfg_upd_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ran_node_name:
      c.destroy<printable_string<1, 150, true, true> >();
      break;
    case types::supported_ta_list:
      c.destroy<supported_ta_list_l>();
      break;
    case types::global_ran_node_id:
      c.destroy<global_ran_node_id_c>();
      break;
    default:
      break;
  }
}
void ran_cfg_upd_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::ran_node_name:
      c.init<printable_string<1, 150, true, true> >();
      break;
    case types::supported_ta_list:
      c.init<supported_ta_list_l>();
      break;
    case types::default_paging_drx:
      break;
    case types::global_ran_node_id:
      c.init<global_ran_node_id_c>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ran_cfg_upd_ies_o::value_c");
  }
}
ran_cfg_upd_ies_o::value_c::value_c(const ran_cfg_upd_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::ran_node_name:
      c.init(other.c.get<printable_string<1, 150, true, true> >());
      break;
    case types::supported_ta_list:
      c.init(other.c.get<supported_ta_list_l>());
      break;
    case types::default_paging_drx:
      c.init(other.c.get<paging_drx_e>());
      break;
    case types::global_ran_node_id:
      c.init(other.c.get<global_ran_node_id_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ran_cfg_upd_ies_o::value_c");
  }
}
ran_cfg_upd_ies_o::value_c& ran_cfg_upd_ies_o::value_c::operator=(const ran_cfg_upd_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::ran_node_name:
      c.set(other.c.get<printable_string<1, 150, true, true> >());
      break;
    case types::supported_ta_list:
      c.set(other.c.get<supported_ta_list_l>());
      break;
    case types::default_paging_drx:
      c.set(other.c.get<paging_drx_e>());
      break;
    case types::global_ran_node_id:
      c.set(other.c.get<global_ran_node_id_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "ran_cfg_upd_ies_o::value_c");
  }

  return *this;
}
printable_string<1, 150, true, true>& ran_cfg_upd_ies_o::value_c::ran_node_name()
{
  assert_choice_type(types::ran_node_name, type_, "Value");
  return c.get<printable_string<1, 150, true, true> >();
}
supported_ta_list_l& ran_cfg_upd_ies_o::value_c::supported_ta_list()
{
  assert_choice_type(types::supported_ta_list, type_, "Value");
  return c.get<supported_ta_list_l>();
}
paging_drx_e& ran_cfg_upd_ies_o::value_c::default_paging_drx()
{
  assert_choice_type(types::default_paging_drx, type_, "Value");
  return c.get<paging_drx_e>();
}
global_ran_node_id_c& ran_cfg_upd_ies_o::value_c::global_ran_node_id()
{
  assert_choice_type(types::global_ran_node_id, type_, "Value");
  return c.get<global_ran_node_id_c>();
}
const printable_string<1, 150, true, true>& ran_cfg_upd_ies_o::value_c::ran_node_name() const
{
  assert_choice_type(types::ran_node_name, type_, "Value");
  return c.get<printable_string<1, 150, true, true> >();
}
const supported_ta_list_l& ran_cfg_upd_ies_o::value_c::supported_ta_list() const
{
  assert_choice_type(types::supported_ta_list, type_, "Value");
  return c.get<supported_ta_list_l>();
}
const paging_drx_e& ran_cfg_upd_ies_o::value_c::default_paging_drx() const
{
  assert_choice_type(types::default_paging_drx, type_, "Value");
  return c.get<paging_drx_e>();
}
const global_ran_node_id_c& ran_cfg_upd_ies_o::value_c::global_ran_node_id() const
{
  assert_choice_type(types::global_ran_node_id, type_, "Value");
  return c.get<global_ran_node_id_c>();
}
void ran_cfg_upd_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::ran_node_name:
      j.write_str("PrintableString", c.get<printable_string<1, 150, true, true> >().to_string());
      break;
    case types::supported_ta_list:
      j.start_array("SupportedTAList");
      for (const auto& e1 : c.get<supported_ta_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::default_paging_drx:
      j.write_str("PagingDRX", c.get<paging_drx_e>().to_string());
      break;
    case types::global_ran_node_id:
      j.write_fieldname("GlobalRANNodeID");
      c.get<global_ran_node_id_c>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "ran_cfg_upd_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE ran_cfg_upd_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::ran_node_name:
      HANDLE_CODE((c.get<printable_string<1, 150, true, true> >().pack(bref)));
      break;
    case types::supported_ta_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<supported_ta_list_l>(), 1, 256, true));
      break;
    case types::default_paging_drx:
      HANDLE_CODE(c.get<paging_drx_e>().pack(bref));
      break;
    case types::global_ran_node_id:
      HANDLE_CODE(c.get<global_ran_node_id_c>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ran_cfg_upd_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE ran_cfg_upd_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::ran_node_name:
      HANDLE_CODE((c.get<printable_string<1, 150, true, true> >().unpack(bref)));
      break;
    case types::supported_ta_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<supported_ta_list_l>(), bref, 1, 256, true));
      break;
    case types::default_paging_drx:
      HANDLE_CODE(c.get<paging_drx_e>().unpack(bref));
      break;
    case types::global_ran_node_id:
      HANDLE_CODE(c.get<global_ran_node_id_c>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "ran_cfg_upd_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* ran_cfg_upd_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"PrintableString", "SupportedTAList", "PagingDRX", "GlobalRANNodeID"};
  return convert_enum_idx(options, 4, value, "ran_cfg_upd_ies_o::value_c::types");
}

// RRCInactiveTransitionReportIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t rrc_inactive_transition_report_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 92, 121};
  return map_enum_number(options, 4, idx, "id");
}
bool rrc_inactive_transition_report_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 92, 121};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e rrc_inactive_transition_report_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::reject;
    case 85:
      return crit_e::reject;
    case 92:
      return crit_e::ignore;
    case 121:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
rrc_inactive_transition_report_ies_o::value_c rrc_inactive_transition_report_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 92:
      ret.set(value_c::types::rrc_state);
      break;
    case 121:
      ret.set(value_c::types::user_location_info);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e rrc_inactive_transition_report_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 92:
      return presence_e::mandatory;
    case 121:
      return presence_e::mandatory;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void rrc_inactive_transition_report_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::user_location_info:
      c.destroy<user_location_info_c>();
      break;
    default:
      break;
  }
}
void rrc_inactive_transition_report_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::rrc_state:
      break;
    case types::user_location_info:
      c.init<user_location_info_c>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "rrc_inactive_transition_report_ies_o::value_c");
  }
}
rrc_inactive_transition_report_ies_o::value_c::value_c(const rrc_inactive_transition_report_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::rrc_state:
      c.init(other.c.get<rrc_state_e>());
      break;
    case types::user_location_info:
      c.init(other.c.get<user_location_info_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "rrc_inactive_transition_report_ies_o::value_c");
  }
}
rrc_inactive_transition_report_ies_o::value_c&
rrc_inactive_transition_report_ies_o::value_c::operator=(const rrc_inactive_transition_report_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::rrc_state:
      c.set(other.c.get<rrc_state_e>());
      break;
    case types::user_location_info:
      c.set(other.c.get<user_location_info_c>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "rrc_inactive_transition_report_ies_o::value_c");
  }

  return *this;
}
uint64_t& rrc_inactive_transition_report_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& rrc_inactive_transition_report_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
rrc_state_e& rrc_inactive_transition_report_ies_o::value_c::rrc_state()
{
  assert_choice_type(types::rrc_state, type_, "Value");
  return c.get<rrc_state_e>();
}
user_location_info_c& rrc_inactive_transition_report_ies_o::value_c::user_location_info()
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_c>();
}
const uint64_t& rrc_inactive_transition_report_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& rrc_inactive_transition_report_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const rrc_state_e& rrc_inactive_transition_report_ies_o::value_c::rrc_state() const
{
  assert_choice_type(types::rrc_state, type_, "Value");
  return c.get<rrc_state_e>();
}
const user_location_info_c& rrc_inactive_transition_report_ies_o::value_c::user_location_info() const
{
  assert_choice_type(types::user_location_info, type_, "Value");
  return c.get<user_location_info_c>();
}
void rrc_inactive_transition_report_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::rrc_state:
      j.write_str("RRCState", c.get<rrc_state_e>().to_string());
      break;
    case types::user_location_info:
      j.write_fieldname("UserLocationInformation");
      c.get<user_location_info_c>().to_json(j);
      break;
    default:
      log_invalid_choice_id(type_, "rrc_inactive_transition_report_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE rrc_inactive_transition_report_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::rrc_state:
      HANDLE_CODE(c.get<rrc_state_e>().pack(bref));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_c>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "rrc_inactive_transition_report_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE rrc_inactive_transition_report_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::rrc_state:
      HANDLE_CODE(c.get<rrc_state_e>().unpack(bref));
      break;
    case types::user_location_info:
      HANDLE_CODE(c.get<user_location_info_c>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "rrc_inactive_transition_report_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* rrc_inactive_transition_report_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..1099511627775)", "INTEGER (0..4294967295)", "RRCState", "UserLocationInformation"};
  return convert_enum_idx(options, 4, value, "rrc_inactive_transition_report_ies_o::value_c::types");
}

// RerouteNASRequest-IEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t reroute_nas_request_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {85, 10, 42, 3, 0};
  return map_enum_number(options, 5, idx, "id");
}
bool reroute_nas_request_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {85, 10, 42, 3, 0};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e reroute_nas_request_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 85:
      return crit_e::reject;
    case 10:
      return crit_e::ignore;
    case 42:
      return crit_e::reject;
    case 3:
      return crit_e::reject;
    case 0:
      return crit_e::reject;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
reroute_nas_request_ies_o::value_c reroute_nas_request_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 42:
      ret.set(value_c::types::ngap_msg);
      break;
    case 3:
      ret.set(value_c::types::amf_set_id);
      break;
    case 0:
      ret.set(value_c::types::allowed_nssai);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e reroute_nas_request_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 85:
      return presence_e::mandatory;
    case 10:
      return presence_e::optional;
    case 42:
      return presence_e::mandatory;
    case 3:
      return presence_e::mandatory;
    case 0:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void reroute_nas_request_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::ngap_msg:
      c.destroy<unbounded_octstring<true> >();
      break;
    case types::amf_set_id:
      c.destroy<fixed_bitstring<10, false, true> >();
      break;
    case types::allowed_nssai:
      c.destroy<allowed_nssai_l>();
      break;
    default:
      break;
  }
}
void reroute_nas_request_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::ran_ue_ngap_id:
      break;
    case types::amf_ue_ngap_id:
      break;
    case types::ngap_msg:
      c.init<unbounded_octstring<true> >();
      break;
    case types::amf_set_id:
      c.init<fixed_bitstring<10, false, true> >();
      break;
    case types::allowed_nssai:
      c.init<allowed_nssai_l>();
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "reroute_nas_request_ies_o::value_c");
  }
}
reroute_nas_request_ies_o::value_c::value_c(const reroute_nas_request_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ngap_msg:
      c.init(other.c.get<unbounded_octstring<true> >());
      break;
    case types::amf_set_id:
      c.init(other.c.get<fixed_bitstring<10, false, true> >());
      break;
    case types::allowed_nssai:
      c.init(other.c.get<allowed_nssai_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "reroute_nas_request_ies_o::value_c");
  }
}
reroute_nas_request_ies_o::value_c&
reroute_nas_request_ies_o::value_c::operator=(const reroute_nas_request_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ngap_msg:
      c.set(other.c.get<unbounded_octstring<true> >());
      break;
    case types::amf_set_id:
      c.set(other.c.get<fixed_bitstring<10, false, true> >());
      break;
    case types::allowed_nssai:
      c.set(other.c.get<allowed_nssai_l>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "reroute_nas_request_ies_o::value_c");
  }

  return *this;
}
uint64_t& reroute_nas_request_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& reroute_nas_request_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
unbounded_octstring<true>& reroute_nas_request_ies_o::value_c::ngap_msg()
{
  assert_choice_type(types::ngap_msg, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
fixed_bitstring<10, false, true>& reroute_nas_request_ies_o::value_c::amf_set_id()
{
  assert_choice_type(types::amf_set_id, type_, "Value");
  return c.get<fixed_bitstring<10, false, true> >();
}
allowed_nssai_l& reroute_nas_request_ies_o::value_c::allowed_nssai()
{
  assert_choice_type(types::allowed_nssai, type_, "Value");
  return c.get<allowed_nssai_l>();
}
const uint64_t& reroute_nas_request_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& reroute_nas_request_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const unbounded_octstring<true>& reroute_nas_request_ies_o::value_c::ngap_msg() const
{
  assert_choice_type(types::ngap_msg, type_, "Value");
  return c.get<unbounded_octstring<true> >();
}
const fixed_bitstring<10, false, true>& reroute_nas_request_ies_o::value_c::amf_set_id() const
{
  assert_choice_type(types::amf_set_id, type_, "Value");
  return c.get<fixed_bitstring<10, false, true> >();
}
const allowed_nssai_l& reroute_nas_request_ies_o::value_c::allowed_nssai() const
{
  assert_choice_type(types::allowed_nssai, type_, "Value");
  return c.get<allowed_nssai_l>();
}
void reroute_nas_request_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ngap_msg:
      j.write_str("OCTET STRING", c.get<unbounded_octstring<true> >().to_string());
      break;
    case types::amf_set_id:
      j.write_str("BIT STRING", c.get<fixed_bitstring<10, false, true> >().to_string());
      break;
    case types::allowed_nssai:
      j.start_array("AllowedNSSAI");
      for (const auto& e1 : c.get<allowed_nssai_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    default:
      log_invalid_choice_id(type_, "reroute_nas_request_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE reroute_nas_request_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ngap_msg:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().pack(bref));
      break;
    case types::amf_set_id:
      HANDLE_CODE((c.get<fixed_bitstring<10, false, true> >().pack(bref)));
      break;
    case types::allowed_nssai:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<allowed_nssai_l>(), 1, 8, true));
      break;
    default:
      log_invalid_choice_id(type_, "reroute_nas_request_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE reroute_nas_request_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ngap_msg:
      HANDLE_CODE(c.get<unbounded_octstring<true> >().unpack(bref));
      break;
    case types::amf_set_id:
      HANDLE_CODE((c.get<fixed_bitstring<10, false, true> >().unpack(bref)));
      break;
    case types::allowed_nssai:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<allowed_nssai_l>(), bref, 1, 8, true));
      break;
    default:
      log_invalid_choice_id(type_, "reroute_nas_request_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* reroute_nas_request_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {
      "INTEGER (0..4294967295)", "INTEGER (0..1099511627775)", "OCTET STRING", "BIT STRING", "AllowedNSSAI"};
  return convert_enum_idx(options, 5, value, "reroute_nas_request_ies_o::value_c::types");
}

// SecondaryRATDataUsageReportIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t secondary_rat_data_usage_report_ies_o::idx_to_id(uint32_t idx)
{
  static const uint32_t options[] = {10, 85, 142, 143};
  return map_enum_number(options, 4, idx, "id");
}
bool secondary_rat_data_usage_report_ies_o::is_id_valid(const uint32_t& id)
{
  static const uint32_t options[] = {10, 85, 142, 143};
  for (const auto& o : options) {
    if (o == id) {
      return true;
    }
  }
  return false;
}
crit_e secondary_rat_data_usage_report_ies_o::get_crit(const uint32_t& id)
{
  switch (id) {
    case 10:
      return crit_e::ignore;
    case 85:
      return crit_e::ignore;
    case 142:
      return crit_e::ignore;
    case 143:
      return crit_e::ignore;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}
secondary_rat_data_usage_report_ies_o::value_c secondary_rat_data_usage_report_ies_o::get_value(const uint32_t& id)
{
  value_c ret{};
  switch (id) {
    case 10:
      ret.set(value_c::types::amf_ue_ngap_id);
      break;
    case 85:
      ret.set(value_c::types::ran_ue_ngap_id);
      break;
    case 142:
      ret.set(value_c::types::pdu_session_res_secondary_ratusage_list);
      break;
    case 143:
      ret.set(value_c::types::ho_flag);
      break;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return ret;
}
presence_e secondary_rat_data_usage_report_ies_o::get_presence(const uint32_t& id)
{
  switch (id) {
    case 10:
      return presence_e::mandatory;
    case 85:
      return presence_e::mandatory;
    case 142:
      return presence_e::mandatory;
    case 143:
      return presence_e::optional;
    default:
      asn1::log_error("The id=%d is not recognized", id);
  }
  return {};
}

// Value ::= OPEN TYPE
void secondary_rat_data_usage_report_ies_o::value_c::destroy_()
{
  switch (type_) {
    case types::pdu_session_res_secondary_ratusage_list:
      c.destroy<pdu_session_res_secondary_ratusage_list_l>();
      break;
    default:
      break;
  }
}
void secondary_rat_data_usage_report_ies_o::value_c::set(types::options e)
{
  destroy_();
  type_ = e;
  switch (type_) {
    case types::amf_ue_ngap_id:
      break;
    case types::ran_ue_ngap_id:
      break;
    case types::pdu_session_res_secondary_ratusage_list:
      c.init<pdu_session_res_secondary_ratusage_list_l>();
      break;
    case types::ho_flag:
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "secondary_rat_data_usage_report_ies_o::value_c");
  }
}
secondary_rat_data_usage_report_ies_o::value_c::value_c(const secondary_rat_data_usage_report_ies_o::value_c& other)
{
  type_ = other.type();
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.init(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_secondary_ratusage_list:
      c.init(other.c.get<pdu_session_res_secondary_ratusage_list_l>());
      break;
    case types::ho_flag:
      c.init(other.c.get<ho_flag_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "secondary_rat_data_usage_report_ies_o::value_c");
  }
}
secondary_rat_data_usage_report_ies_o::value_c&
secondary_rat_data_usage_report_ies_o::value_c::operator=(const secondary_rat_data_usage_report_ies_o::value_c& other)
{
  if (this == &other) {
    return *this;
  }
  set(other.type());
  switch (type_) {
    case types::amf_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      c.set(other.c.get<uint64_t>());
      break;
    case types::pdu_session_res_secondary_ratusage_list:
      c.set(other.c.get<pdu_session_res_secondary_ratusage_list_l>());
      break;
    case types::ho_flag:
      c.set(other.c.get<ho_flag_e>());
      break;
    case types::nulltype:
      break;
    default:
      log_invalid_choice_id(type_, "secondary_rat_data_usage_report_ies_o::value_c");
  }

  return *this;
}
uint64_t& secondary_rat_data_usage_report_ies_o::value_c::amf_ue_ngap_id()
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
uint64_t& secondary_rat_data_usage_report_ies_o::value_c::ran_ue_ngap_id()
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
pdu_session_res_secondary_ratusage_list_l&
secondary_rat_data_usage_report_ies_o::value_c::pdu_session_res_secondary_ratusage_list()
{
  assert_choice_type(types::pdu_session_res_secondary_ratusage_list, type_, "Value");
  return c.get<pdu_session_res_secondary_ratusage_list_l>();
}
ho_flag_e& secondary_rat_data_usage_report_ies_o::value_c::ho_flag()
{
  assert_choice_type(types::ho_flag, type_, "Value");
  return c.get<ho_flag_e>();
}
const uint64_t& secondary_rat_data_usage_report_ies_o::value_c::amf_ue_ngap_id() const
{
  assert_choice_type(types::amf_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const uint64_t& secondary_rat_data_usage_report_ies_o::value_c::ran_ue_ngap_id() const
{
  assert_choice_type(types::ran_ue_ngap_id, type_, "Value");
  return c.get<uint64_t>();
}
const pdu_session_res_secondary_ratusage_list_l&
secondary_rat_data_usage_report_ies_o::value_c::pdu_session_res_secondary_ratusage_list() const
{
  assert_choice_type(types::pdu_session_res_secondary_ratusage_list, type_, "Value");
  return c.get<pdu_session_res_secondary_ratusage_list_l>();
}
const ho_flag_e& secondary_rat_data_usage_report_ies_o::value_c::ho_flag() const
{
  assert_choice_type(types::ho_flag, type_, "Value");
  return c.get<ho_flag_e>();
}
void secondary_rat_data_usage_report_ies_o::value_c::to_json(json_writer& j) const
{
  j.start_obj();
  switch (type_) {
    case types::amf_ue_ngap_id:
      j.write_int("INTEGER (0..1099511627775)", c.get<uint64_t>());
      break;
    case types::ran_ue_ngap_id:
      j.write_int("INTEGER (0..4294967295)", c.get<uint64_t>());
      break;
    case types::pdu_session_res_secondary_ratusage_list:
      j.start_array("PDUSessionResourceSecondaryRATUsageList");
      for (const auto& e1 : c.get<pdu_session_res_secondary_ratusage_list_l>()) {
        e1.to_json(j);
      }
      j.end_array();
      break;
    case types::ho_flag:
      j.write_str("HandoverFlag", "handover-preparation");
      break;
    default:
      log_invalid_choice_id(type_, "secondary_rat_data_usage_report_ies_o::value_c");
  }
  j.end_obj();
}
SRSASN_CODE secondary_rat_data_usage_report_ies_o::value_c::pack(bit_ref& bref) const
{
  varlength_field_pack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(pack_integer(bref, c.get<uint64_t>(), (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_secondary_ratusage_list:
      HANDLE_CODE(pack_dyn_seq_of(bref, c.get<pdu_session_res_secondary_ratusage_list_l>(), 1, 256, true));
      break;
    case types::ho_flag:
      HANDLE_CODE(c.get<ho_flag_e>().pack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "secondary_rat_data_usage_report_ies_o::value_c");
      return SRSASN_ERROR_ENCODE_FAIL;
  }
  return SRSASN_SUCCESS;
}
SRSASN_CODE secondary_rat_data_usage_report_ies_o::value_c::unpack(cbit_ref& bref)
{
  varlength_field_unpack_guard varlen_scope(bref, true);
  switch (type_) {
    case types::amf_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)1099511627775u, false, true));
      break;
    case types::ran_ue_ngap_id:
      HANDLE_CODE(unpack_integer(c.get<uint64_t>(), bref, (uint64_t)0u, (uint64_t)4294967295u, false, true));
      break;
    case types::pdu_session_res_secondary_ratusage_list:
      HANDLE_CODE(unpack_dyn_seq_of(c.get<pdu_session_res_secondary_ratusage_list_l>(), bref, 1, 256, true));
      break;
    case types::ho_flag:
      HANDLE_CODE(c.get<ho_flag_e>().unpack(bref));
      break;
    default:
      log_invalid_choice_id(type_, "secondary_rat_data_usage_report_ies_o::value_c");
      return SRSASN_ERROR_DECODE_FAIL;
  }
  return SRSASN_SUCCESS;
}

const char* secondary_rat_data_usage_report_ies_o::value_c::types_opts::to_string() const
{
  static const char* options[] = {"INTEGER (0..1099511627775)",
                                  "INTEGER (0..4294967295)",
                                  "PDUSessionResourceSecondaryRATUsageList",
                                  "HandoverFlag"};
  return convert_enum_idx(options, 4, value, "secondary_rat_data_usage_report_ies_o::value_c::types");
}

// TraceFailureIndicationIEs ::= OBJECT SET OF NGAP-PROTOCOL-IES
uint32_t trace_fail_ind