#ifndef GEN_UI_GFX_X_XFIXES_H_
#define GEN_UI_GFX_X_XFIXES_H_

#include <array>
#include <cstddef>
#include <cstdint>
#include <cstring>
#include <vector>

#include "base/component_export.h"
#include "base/memory/ref_counted_memory.h"
#include "base/memory/scoped_refptr.h"
#include "base/optional.h"
#include "base/files/scoped_file.h"
#include "ui/gfx/x/xproto_types.h"
#include "render.h"
#include "shape.h"
#include "xproto.h"

namespace x11 {

class Connection;

class COMPONENT_EXPORT(X11) XFixes {
  public:
  static constexpr unsigned major_version = 5;
  static constexpr unsigned minor_version = 0;

  XFixes(Connection* connection,
      const x11::QueryExtensionReply& info);

  uint8_t present() const {
    return info_.present;
  }
  uint8_t major_opcode() const {
    return info_.major_opcode;
  }
  uint8_t first_event() const {
    return info_.first_event;
  }
  uint8_t first_error() const {
    return info_.first_error;
  }

  Connection* connection() const { return connection_; }

  enum class SaveSetMode : int {
    Insert = 0,
    Delete = 1,
  };

  enum class SaveSetTarget : int {
    Nearest = 0,
    Root = 1,
  };

  enum class SaveSetMapping : int {
    Map = 0,
    Unmap = 1,
  };

  enum class SelectionEvent : int {
    SetSelectionOwner = 0,
    SelectionWindowDestroy = 1,
    SelectionClientClose = 2,
  };

  enum class SelectionEventMask : int {
    SetSelectionOwner = 1 << 0,
    SelectionWindowDestroy = 1 << 1,
    SelectionClientClose = 1 << 2,
  };

  enum class CursorNotify : int {
    DisplayCursor = 0,
  };

  enum class CursorNotifyMask : int {
    DisplayCursor = 1 << 0,
  };

  enum class Region : uint32_t {
    None = 0,
  };

  enum class Barrier : uint32_t {};

  enum class BarrierDirections : int {
    PositiveX = 1 << 0,
    PositiveY = 1 << 1,
    NegativeX = 1 << 2,
    NegativeY = 1 << 3,
  };

  struct SelectionNotifyEvent {
    static constexpr int type_id = 18;
    static constexpr uint8_t opcode = 0;
    bool send_event{};
    SelectionEvent subtype{};
    uint16_t sequence{};
    Window window{};
    Window owner{};
    Atom selection{};
    Time timestamp{};
    Time selection_timestamp{};

    x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&window); }
  };

  struct CursorNotifyEvent {
    static constexpr int type_id = 19;
    static constexpr uint8_t opcode = 1;
    bool send_event{};
    CursorNotify subtype{};
    uint16_t sequence{};
    Window window{};
    uint32_t cursor_serial{};
    Time timestamp{};
    Atom name{};

    x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&window); }
  };

  struct BadRegionError {
    uint16_t sequence{};
  };

  struct QueryVersionRequest {
    uint32_t client_major_version{};
    uint32_t client_minor_version{};
  };

  struct QueryVersionReply {
    uint16_t sequence{};
    uint32_t major_version{};
    uint32_t minor_version{};
  };

  using QueryVersionResponse = Response<QueryVersionReply>;

  Future<QueryVersionReply> QueryVersion(
      const QueryVersionRequest& request);

  struct ChangeSaveSetRequest {
    SaveSetMode mode{};
    SaveSetTarget target{};
    SaveSetMapping map{};
    Window window{};
  };

  using ChangeSaveSetResponse = Response<void>;

  Future<void> ChangeSaveSet(
      const ChangeSaveSetRequest& request);

  struct SelectSelectionInputRequest {
    Window window{};
    Atom selection{};
    SelectionEventMask event_mask{};
  };

  using SelectSelectionInputResponse = Response<void>;

  Future<void> SelectSelectionInput(
      const SelectSelectionInputRequest& request);

  struct SelectCursorInputRequest {
    Window window{};
    CursorNotifyMask event_mask{};
  };

  using SelectCursorInputResponse = Response<void>;

  Future<void> SelectCursorInput(
      const SelectCursorInputRequest& request);

  struct GetCursorImageRequest {
  };

  struct GetCursorImageReply {
    uint16_t sequence{};
    int16_t x{};
    int16_t y{};
    uint16_t width{};
    uint16_t height{};
    uint16_t xhot{};
    uint16_t yhot{};
    uint32_t cursor_serial{};
    std::vector<uint32_t> cursor_image{};
  };

  using GetCursorImageResponse = Response<GetCursorImageReply>;

  Future<GetCursorImageReply> GetCursorImage(
      const GetCursorImageRequest& request);

  struct CreateRegionRequest {
    Region region{};
    std::vector<Rectangle> rectangles{};
  };

  using CreateRegionResponse = Response<void>;

  Future<void> CreateRegion(
      const CreateRegionRequest& request);

  struct CreateRegionFromBitmapRequest {
    Region region{};
    Pixmap bitmap{};
  };

  using CreateRegionFromBitmapResponse = Response<void>;

  Future<void> CreateRegionFromBitmap(
      const CreateRegionFromBitmapRequest& request);

  struct CreateRegionFromWindowRequest {
    Region region{};
    Window window{};
    Shape::Sk kind{};
  };

  using CreateRegionFromWindowResponse = Response<void>;

  Future<void> CreateRegionFromWindow(
      const CreateRegionFromWindowRequest& request);

  struct CreateRegionFromGCRequest {
    Region region{};
    GraphicsContext gc{};
  };

  using CreateRegionFromGCResponse = Response<void>;

  Future<void> CreateRegionFromGC(
      const CreateRegionFromGCRequest& request);

  struct CreateRegionFromPictureRequest {
    Region region{};
    Render::Picture picture{};
  };

  using CreateRegionFromPictureResponse = Response<void>;

  Future<void> CreateRegionFromPicture(
      const CreateRegionFromPictureRequest& request);

  struct DestroyRegionRequest {
    Region region{};
  };

  using DestroyRegionResponse = Response<void>;

  Future<void> DestroyRegion(
      const DestroyRegionRequest& request);

  struct SetRegionRequest {
    Region region{};
    std::vector<Rectangle> rectangles{};
  };

  using SetRegionResponse = Response<void>;

  Future<void> SetRegion(
      const SetRegionRequest& request);

  struct CopyRegionRequest {
    Region source{};
    Region destination{};
  };

  using CopyRegionResponse = Response<void>;

  Future<void> CopyRegion(
      const CopyRegionRequest& request);

  struct UnionRegionRequest {
    Region source1{};
    Region source2{};
    Region destination{};
  };

  using UnionRegionResponse = Response<void>;

  Future<void> UnionRegion(
      const UnionRegionRequest& request);

  struct IntersectRegionRequest {
    Region source1{};
    Region source2{};
    Region destination{};
  };

  using IntersectRegionResponse = Response<void>;

  Future<void> IntersectRegion(
      const IntersectRegionRequest& request);

  struct SubtractRegionRequest {
    Region source1{};
    Region source2{};
    Region destination{};
  };

  using SubtractRegionResponse = Response<void>;

  Future<void> SubtractRegion(
      const SubtractRegionRequest& request);

  struct InvertRegionRequest {
    Region source{};
    Rectangle bounds{};
    Region destination{};
  };

  using InvertRegionResponse = Response<void>;

  Future<void> InvertRegion(
      const InvertRegionRequest& request);

  struct TranslateRegionRequest {
    Region region{};
    int16_t dx{};
    int16_t dy{};
  };

  using TranslateRegionResponse = Response<void>;

  Future<void> TranslateRegion(
      const TranslateRegionRequest& request);

  struct RegionExtentsRequest {
    Region source{};
    Region destination{};
  };

  using RegionExtentsResponse = Response<void>;

  Future<void> RegionExtents(
      const RegionExtentsRequest& request);

  struct FetchRegionRequest {
    Region region{};
  };

  struct FetchRegionReply {
    uint16_t sequence{};
    Rectangle extents{};
    std::vector<Rectangle> rectangles{};
  };

  using FetchRegionResponse = Response<FetchRegionReply>;

  Future<FetchRegionReply> FetchRegion(
      const FetchRegionRequest& request);

  struct SetGCClipRegionRequest {
    GraphicsContext gc{};
    Region region{};
    int16_t x_origin{};
    int16_t y_origin{};
  };

  using SetGCClipRegionResponse = Response<void>;

  Future<void> SetGCClipRegion(
      const SetGCClipRegionRequest& request);

  struct SetWindowShapeRegionRequest {
    Window dest{};
    Shape::Sk dest_kind{};
    int16_t x_offset{};
    int16_t y_offset{};
    Region region{};
  };

  using SetWindowShapeRegionResponse = Response<void>;

  Future<void> SetWindowShapeRegion(
      const SetWindowShapeRegionRequest& request);

  struct SetPictureClipRegionRequest {
    Render::Picture picture{};
    Region region{};
    int16_t x_origin{};
    int16_t y_origin{};
  };

  using SetPictureClipRegionResponse = Response<void>;

  Future<void> SetPictureClipRegion(
      const SetPictureClipRegionRequest& request);

  struct SetCursorNameRequest {
    Cursor cursor{};
    std::string name{};
  };

  using SetCursorNameResponse = Response<void>;

  Future<void> SetCursorName(
      const SetCursorNameRequest& request);

  struct GetCursorNameRequest {
    Cursor cursor{};
  };

  struct GetCursorNameReply {
    uint16_t sequence{};
    Atom atom{};
    std::string name{};
  };

  using GetCursorNameResponse = Response<GetCursorNameReply>;

  Future<GetCursorNameReply> GetCursorName(
      const GetCursorNameRequest& request);

  struct GetCursorImageAndNameRequest {
  };

  struct GetCursorImageAndNameReply {
    uint16_t sequence{};
    int16_t x{};
    int16_t y{};
    uint16_t width{};
    uint16_t height{};
    uint16_t xhot{};
    uint16_t yhot{};
    uint32_t cursor_serial{};
    Atom cursor_atom{};
    std::vector<uint32_t> cursor_image{};
    std::string name{};
  };

  using GetCursorImageAndNameResponse = Response<GetCursorImageAndNameReply>;

  Future<GetCursorImageAndNameReply> GetCursorImageAndName(
      const GetCursorImageAndNameRequest& request);

  struct ChangeCursorRequest {
    Cursor source{};
    Cursor destination{};
  };

  using ChangeCursorResponse = Response<void>;

  Future<void> ChangeCursor(
      const ChangeCursorRequest& request);

  struct ChangeCursorByNameRequest {
    Cursor src{};
    std::string name{};
  };

  using ChangeCursorByNameResponse = Response<void>;

  Future<void> ChangeCursorByName(
      const ChangeCursorByNameRequest& request);

  struct ExpandRegionRequest {
    Region source{};
    Region destination{};
    uint16_t left{};
    uint16_t right{};
    uint16_t top{};
    uint16_t bottom{};
  };

  using ExpandRegionResponse = Response<void>;

  Future<void> ExpandRegion(
      const ExpandRegionRequest& request);

  struct HideCursorRequest {
    Window window{};
  };

  using HideCursorResponse = Response<void>;

  Future<void> HideCursor(
      const HideCursorRequest& request);

  struct ShowCursorRequest {
    Window window{};
  };

  using ShowCursorResponse = Response<void>;

  Future<void> ShowCursor(
      const ShowCursorRequest& request);

  struct CreatePointerBarrierRequest {
    Barrier barrier{};
    Window window{};
    uint16_t x1{};
    uint16_t y1{};
    uint16_t x2{};
    uint16_t y2{};
    BarrierDirections directions{};
    std::vector<uint16_t> devices{};
  };

  using CreatePointerBarrierResponse = Response<void>;

  Future<void> CreatePointerBarrier(
      const CreatePointerBarrierRequest& request);

  struct DeletePointerBarrierRequest {
    Barrier barrier{};
  };

  using DeletePointerBarrierResponse = Response<void>;

  Future<void> DeletePointerBarrier(
      const DeletePointerBarrierRequest& request);

  private:
  x11::Connection* const connection_;
  x11::QueryExtensionReply info_{};
};

}  // namespace x11

inline constexpr x11::XFixes::SaveSetMode operator|(
    x11::XFixes::SaveSetMode l, x11::XFixes::SaveSetMode r) {
  using T = std::underlying_type_t<x11::XFixes::SaveSetMode>;
  return static_cast<x11::XFixes::SaveSetMode>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::XFixes::SaveSetMode operator&(
    x11::XFixes::SaveSetMode l, x11::XFixes::SaveSetMode r) {
  using T = std::underlying_type_t<x11::XFixes::SaveSetMode>;
  return static_cast<x11::XFixes::SaveSetMode>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::XFixes::SaveSetTarget operator|(
    x11::XFixes::SaveSetTarget l, x11::XFixes::SaveSetTarget r) {
  using T = std::underlying_type_t<x11::XFixes::SaveSetTarget>;
  return static_cast<x11::XFixes::SaveSetTarget>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::XFixes::SaveSetTarget operator&(
    x11::XFixes::SaveSetTarget l, x11::XFixes::SaveSetTarget r) {
  using T = std::underlying_type_t<x11::XFixes::SaveSetTarget>;
  return static_cast<x11::XFixes::SaveSetTarget>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::XFixes::SaveSetMapping operator|(
    x11::XFixes::SaveSetMapping l, x11::XFixes::SaveSetMapping r) {
  using T = std::underlying_type_t<x11::XFixes::SaveSetMapping>;
  return static_cast<x11::XFixes::SaveSetMapping>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::XFixes::SaveSetMapping operator&(
    x11::XFixes::SaveSetMapping l, x11::XFixes::SaveSetMapping r) {
  using T = std::underlying_type_t<x11::XFixes::SaveSetMapping>;
  return static_cast<x11::XFixes::SaveSetMapping>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::XFixes::SelectionEvent operator|(
    x11::XFixes::SelectionEvent l, x11::XFixes::SelectionEvent r) {
  using T = std::underlying_type_t<x11::XFixes::SelectionEvent>;
  return static_cast<x11::XFixes::SelectionEvent>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::XFixes::SelectionEvent operator&(
    x11::XFixes::SelectionEvent l, x11::XFixes::SelectionEvent r) {
  using T = std::underlying_type_t<x11::XFixes::SelectionEvent>;
  return static_cast<x11::XFixes::SelectionEvent>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::XFixes::SelectionEventMask operator|(
    x11::XFixes::SelectionEventMask l, x11::XFixes::SelectionEventMask r) {
  using T = std::underlying_type_t<x11::XFixes::SelectionEventMask>;
  return static_cast<x11::XFixes::SelectionEventMask>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::XFixes::SelectionEventMask operator&(
    x11::XFixes::SelectionEventMask l, x11::XFixes::SelectionEventMask r) {
  using T = std::underlying_type_t<x11::XFixes::SelectionEventMask>;
  return static_cast<x11::XFixes::SelectionEventMask>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::XFixes::CursorNotify operator|(
    x11::XFixes::CursorNotify l, x11::XFixes::CursorNotify r) {
  using T = std::underlying_type_t<x11::XFixes::CursorNotify>;
  return static_cast<x11::XFixes::CursorNotify>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::XFixes::CursorNotify operator&(
    x11::XFixes::CursorNotify l, x11::XFixes::CursorNotify r) {
  using T = std::underlying_type_t<x11::XFixes::CursorNotify>;
  return static_cast<x11::XFixes::CursorNotify>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::XFixes::CursorNotifyMask operator|(
    x11::XFixes::CursorNotifyMask l, x11::XFixes::CursorNotifyMask r) {
  using T = std::underlying_type_t<x11::XFixes::CursorNotifyMask>;
  return static_cast<x11::XFixes::CursorNotifyMask>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::XFixes::CursorNotifyMask operator&(
    x11::XFixes::CursorNotifyMask l, x11::XFixes::CursorNotifyMask r) {
  using T = std::underlying_type_t<x11::XFixes::CursorNotifyMask>;
  return static_cast<x11::XFixes::CursorNotifyMask>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::XFixes::Region operator|(
    x11::XFixes::Region l, x11::XFixes::Region r) {
  using T = std::underlying_type_t<x11::XFixes::Region>;
  return static_cast<x11::XFixes::Region>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::XFixes::Region operator&(
    x11::XFixes::Region l, x11::XFixes::Region r) {
  using T = std::underlying_type_t<x11::XFixes::Region>;
  return static_cast<x11::XFixes::Region>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::XFixes::BarrierDirections operator|(
    x11::XFixes::BarrierDirections l, x11::XFixes::BarrierDirections r) {
  using T = std::underlying_type_t<x11::XFixes::BarrierDirections>;
  return static_cast<x11::XFixes::BarrierDirections>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::XFixes::BarrierDirections operator&(
    x11::XFixes::BarrierDirections l, x11::XFixes::BarrierDirections r) {
  using T = std::underlying_type_t<x11::XFixes::BarrierDirections>;
  return static_cast<x11::XFixes::BarrierDirections>(
      static_cast<T>(l) & static_cast<T>(r));
}


#endif  // GEN_UI_GFX_X_XFIXES_H_
