{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GBookmarkFile@ lets you parse, edit or create files containing bookmarks.
-- 
-- Bookmarks refer to a URI, along with some meta-data about the resource
-- pointed by the URI like its MIME type, the application that is registering
-- the bookmark and the icon that should be used to represent the bookmark.
-- The data is stored using the
-- <https://www.freedesktop.org/wiki/Specifications/desktop-bookmark-spec/ Desktop Bookmark Specification>.
-- 
-- The syntax of the bookmark files is described in detail inside the
-- Desktop Bookmark Specification, here is a quick summary: bookmark
-- files use a sub-class of the XML Bookmark Exchange Language
-- specification, consisting of valid UTF-8 encoded XML, under the
-- @\<xbel>@ root element; each bookmark is stored inside a
-- @\<bookmark>@ element, using its URI: no relative paths can
-- be used inside a bookmark file. The bookmark may have a user defined
-- title and description, to be used instead of the URI. Under the
-- @\<metadata>@ element, with its owner attribute set to
-- @http:\/\/freedesktop.org@, is stored the meta-data about a resource
-- pointed by its URI. The meta-data consists of the resource\'s MIME
-- type; the applications that have registered a bookmark; the groups
-- to which a bookmark belongs to; a visibility flag, used to set the
-- bookmark as \"private\" to the applications and groups that has it
-- registered; the URI and MIME type of an icon, to be used when
-- displaying the bookmark inside a GUI.
-- 
-- Here is an example of a bookmark file:
-- <https://gitlab.gnome.org/GNOME/glib/-/blob/HEAD/glib/tests/bookmarks.xbel bookmarks.xbel>
-- 
-- A bookmark file might contain more than one bookmark; each bookmark
-- is accessed through its URI.
-- 
-- The important caveat of bookmark files is that when you add a new
-- bookmark you must also add the application that is registering it, using
-- 'GI.GLib.Structs.BookmarkFile.bookmarkFileAddApplication' or 'GI.GLib.Structs.BookmarkFile.bookmarkFileSetApplicationInfo'.
-- If a bookmark has no applications then it won\'t be dumped when creating
-- the on disk representation, using 'GI.GLib.Structs.BookmarkFile.bookmarkFileToData' or
-- 'GI.GLib.Structs.BookmarkFile.bookmarkFileToFile'.
-- 
-- /Since: 2.12/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GLib.Structs.BookmarkFile
    ( 

-- * Exported types
    BookmarkFile(..)                        ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [addApplication]("GI.GLib.Structs.BookmarkFile#g:method:addApplication"), [addGroup]("GI.GLib.Structs.BookmarkFile#g:method:addGroup"), [copy]("GI.GLib.Structs.BookmarkFile#g:method:copy"), [free]("GI.GLib.Structs.BookmarkFile#g:method:free"), [hasApplication]("GI.GLib.Structs.BookmarkFile#g:method:hasApplication"), [hasGroup]("GI.GLib.Structs.BookmarkFile#g:method:hasGroup"), [hasItem]("GI.GLib.Structs.BookmarkFile#g:method:hasItem"), [loadFromData]("GI.GLib.Structs.BookmarkFile#g:method:loadFromData"), [loadFromDataDirs]("GI.GLib.Structs.BookmarkFile#g:method:loadFromDataDirs"), [loadFromFile]("GI.GLib.Structs.BookmarkFile#g:method:loadFromFile"), [moveItem]("GI.GLib.Structs.BookmarkFile#g:method:moveItem"), [removeApplication]("GI.GLib.Structs.BookmarkFile#g:method:removeApplication"), [removeGroup]("GI.GLib.Structs.BookmarkFile#g:method:removeGroup"), [removeItem]("GI.GLib.Structs.BookmarkFile#g:method:removeItem"), [toData]("GI.GLib.Structs.BookmarkFile#g:method:toData"), [toFile]("GI.GLib.Structs.BookmarkFile#g:method:toFile").
-- 
-- ==== Getters
-- [getAdded]("GI.GLib.Structs.BookmarkFile#g:method:getAdded"), [getAddedDateTime]("GI.GLib.Structs.BookmarkFile#g:method:getAddedDateTime"), [getAppInfo]("GI.GLib.Structs.BookmarkFile#g:method:getAppInfo"), [getApplicationInfo]("GI.GLib.Structs.BookmarkFile#g:method:getApplicationInfo"), [getApplications]("GI.GLib.Structs.BookmarkFile#g:method:getApplications"), [getDescription]("GI.GLib.Structs.BookmarkFile#g:method:getDescription"), [getGroups]("GI.GLib.Structs.BookmarkFile#g:method:getGroups"), [getIcon]("GI.GLib.Structs.BookmarkFile#g:method:getIcon"), [getIsPrivate]("GI.GLib.Structs.BookmarkFile#g:method:getIsPrivate"), [getMimeType]("GI.GLib.Structs.BookmarkFile#g:method:getMimeType"), [getModified]("GI.GLib.Structs.BookmarkFile#g:method:getModified"), [getModifiedDateTime]("GI.GLib.Structs.BookmarkFile#g:method:getModifiedDateTime"), [getSize]("GI.GLib.Structs.BookmarkFile#g:method:getSize"), [getTitle]("GI.GLib.Structs.BookmarkFile#g:method:getTitle"), [getUris]("GI.GLib.Structs.BookmarkFile#g:method:getUris"), [getVisited]("GI.GLib.Structs.BookmarkFile#g:method:getVisited"), [getVisitedDateTime]("GI.GLib.Structs.BookmarkFile#g:method:getVisitedDateTime").
-- 
-- ==== Setters
-- [setAdded]("GI.GLib.Structs.BookmarkFile#g:method:setAdded"), [setAddedDateTime]("GI.GLib.Structs.BookmarkFile#g:method:setAddedDateTime"), [setAppInfo]("GI.GLib.Structs.BookmarkFile#g:method:setAppInfo"), [setApplicationInfo]("GI.GLib.Structs.BookmarkFile#g:method:setApplicationInfo"), [setDescription]("GI.GLib.Structs.BookmarkFile#g:method:setDescription"), [setGroups]("GI.GLib.Structs.BookmarkFile#g:method:setGroups"), [setIcon]("GI.GLib.Structs.BookmarkFile#g:method:setIcon"), [setIsPrivate]("GI.GLib.Structs.BookmarkFile#g:method:setIsPrivate"), [setMimeType]("GI.GLib.Structs.BookmarkFile#g:method:setMimeType"), [setModified]("GI.GLib.Structs.BookmarkFile#g:method:setModified"), [setModifiedDateTime]("GI.GLib.Structs.BookmarkFile#g:method:setModifiedDateTime"), [setTitle]("GI.GLib.Structs.BookmarkFile#g:method:setTitle"), [setVisited]("GI.GLib.Structs.BookmarkFile#g:method:setVisited"), [setVisitedDateTime]("GI.GLib.Structs.BookmarkFile#g:method:setVisitedDateTime").

#if defined(ENABLE_OVERLOADING)
    ResolveBookmarkFileMethod               ,
#endif

-- ** addApplication #method:addApplication#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileAddApplicationMethodInfo    ,
#endif
    bookmarkFileAddApplication              ,


-- ** addGroup #method:addGroup#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileAddGroupMethodInfo          ,
#endif
    bookmarkFileAddGroup                    ,


-- ** copy #method:copy#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileCopyMethodInfo              ,
#endif
    bookmarkFileCopy                        ,


-- ** errorQuark #method:errorQuark#

    bookmarkFileErrorQuark                  ,


-- ** free #method:free#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileFreeMethodInfo              ,
#endif
    bookmarkFileFree                        ,


-- ** getAdded #method:getAdded#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetAddedMethodInfo          ,
#endif
    bookmarkFileGetAdded                    ,


-- ** getAddedDateTime #method:getAddedDateTime#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetAddedDateTimeMethodInfo  ,
#endif
    bookmarkFileGetAddedDateTime            ,


-- ** getAppInfo #method:getAppInfo#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetAppInfoMethodInfo        ,
#endif
    bookmarkFileGetAppInfo                  ,


-- ** getApplicationInfo #method:getApplicationInfo#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetApplicationInfoMethodInfo,
#endif
    bookmarkFileGetApplicationInfo          ,


-- ** getApplications #method:getApplications#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetApplicationsMethodInfo   ,
#endif
    bookmarkFileGetApplications             ,


-- ** getDescription #method:getDescription#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetDescriptionMethodInfo    ,
#endif
    bookmarkFileGetDescription              ,


-- ** getGroups #method:getGroups#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetGroupsMethodInfo         ,
#endif
    bookmarkFileGetGroups                   ,


-- ** getIcon #method:getIcon#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetIconMethodInfo           ,
#endif
    bookmarkFileGetIcon                     ,


-- ** getIsPrivate #method:getIsPrivate#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetIsPrivateMethodInfo      ,
#endif
    bookmarkFileGetIsPrivate                ,


-- ** getMimeType #method:getMimeType#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetMimeTypeMethodInfo       ,
#endif
    bookmarkFileGetMimeType                 ,


-- ** getModified #method:getModified#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetModifiedMethodInfo       ,
#endif
    bookmarkFileGetModified                 ,


-- ** getModifiedDateTime #method:getModifiedDateTime#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetModifiedDateTimeMethodInfo,
#endif
    bookmarkFileGetModifiedDateTime         ,


-- ** getSize #method:getSize#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetSizeMethodInfo           ,
#endif
    bookmarkFileGetSize                     ,


-- ** getTitle #method:getTitle#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetTitleMethodInfo          ,
#endif
    bookmarkFileGetTitle                    ,


-- ** getUris #method:getUris#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetUrisMethodInfo           ,
#endif
    bookmarkFileGetUris                     ,


-- ** getVisited #method:getVisited#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetVisitedMethodInfo        ,
#endif
    bookmarkFileGetVisited                  ,


-- ** getVisitedDateTime #method:getVisitedDateTime#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileGetVisitedDateTimeMethodInfo,
#endif
    bookmarkFileGetVisitedDateTime          ,


-- ** hasApplication #method:hasApplication#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileHasApplicationMethodInfo    ,
#endif
    bookmarkFileHasApplication              ,


-- ** hasGroup #method:hasGroup#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileHasGroupMethodInfo          ,
#endif
    bookmarkFileHasGroup                    ,


-- ** hasItem #method:hasItem#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileHasItemMethodInfo           ,
#endif
    bookmarkFileHasItem                     ,


-- ** loadFromData #method:loadFromData#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileLoadFromDataMethodInfo      ,
#endif
    bookmarkFileLoadFromData                ,


-- ** loadFromDataDirs #method:loadFromDataDirs#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileLoadFromDataDirsMethodInfo  ,
#endif
    bookmarkFileLoadFromDataDirs            ,


-- ** loadFromFile #method:loadFromFile#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileLoadFromFileMethodInfo      ,
#endif
    bookmarkFileLoadFromFile                ,


-- ** moveItem #method:moveItem#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileMoveItemMethodInfo          ,
#endif
    bookmarkFileMoveItem                    ,


-- ** new #method:new#

    bookmarkFileNew                         ,


-- ** removeApplication #method:removeApplication#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileRemoveApplicationMethodInfo ,
#endif
    bookmarkFileRemoveApplication           ,


-- ** removeGroup #method:removeGroup#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileRemoveGroupMethodInfo       ,
#endif
    bookmarkFileRemoveGroup                 ,


-- ** removeItem #method:removeItem#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileRemoveItemMethodInfo        ,
#endif
    bookmarkFileRemoveItem                  ,


-- ** setAdded #method:setAdded#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileSetAddedMethodInfo          ,
#endif
    bookmarkFileSetAdded                    ,


-- ** setAddedDateTime #method:setAddedDateTime#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileSetAddedDateTimeMethodInfo  ,
#endif
    bookmarkFileSetAddedDateTime            ,


-- ** setAppInfo #method:setAppInfo#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileSetAppInfoMethodInfo        ,
#endif
    bookmarkFileSetAppInfo                  ,


-- ** setApplicationInfo #method:setApplicationInfo#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileSetApplicationInfoMethodInfo,
#endif
    bookmarkFileSetApplicationInfo          ,


-- ** setDescription #method:setDescription#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileSetDescriptionMethodInfo    ,
#endif
    bookmarkFileSetDescription              ,


-- ** setGroups #method:setGroups#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileSetGroupsMethodInfo         ,
#endif
    bookmarkFileSetGroups                   ,


-- ** setIcon #method:setIcon#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileSetIconMethodInfo           ,
#endif
    bookmarkFileSetIcon                     ,


-- ** setIsPrivate #method:setIsPrivate#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileSetIsPrivateMethodInfo      ,
#endif
    bookmarkFileSetIsPrivate                ,


-- ** setMimeType #method:setMimeType#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileSetMimeTypeMethodInfo       ,
#endif
    bookmarkFileSetMimeType                 ,


-- ** setModified #method:setModified#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileSetModifiedMethodInfo       ,
#endif
    bookmarkFileSetModified                 ,


-- ** setModifiedDateTime #method:setModifiedDateTime#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileSetModifiedDateTimeMethodInfo,
#endif
    bookmarkFileSetModifiedDateTime         ,


-- ** setTitle #method:setTitle#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileSetTitleMethodInfo          ,
#endif
    bookmarkFileSetTitle                    ,


-- ** setVisited #method:setVisited#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileSetVisitedMethodInfo        ,
#endif
    bookmarkFileSetVisited                  ,


-- ** setVisitedDateTime #method:setVisitedDateTime#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileSetVisitedDateTimeMethodInfo,
#endif
    bookmarkFileSetVisitedDateTime          ,


-- ** toData #method:toData#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileToDataMethodInfo            ,
#endif
    bookmarkFileToData                      ,


-- ** toFile #method:toFile#

#if defined(ENABLE_OVERLOADING)
    BookmarkFileToFileMethodInfo            ,
#endif
    bookmarkFileToFile                      ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import {-# SOURCE #-} qualified GI.GLib.Enums as GLib.Enums
import {-# SOURCE #-} qualified GI.GLib.Structs.DateTime as GLib.DateTime
import {-# SOURCE #-} qualified GI.GLib.Structs.TimeVal as GLib.TimeVal
import {-# SOURCE #-} qualified GI.GLib.Structs.TimeZone as GLib.TimeZone

#else
import {-# SOURCE #-} qualified GI.GLib.Structs.DateTime as GLib.DateTime

#endif

-- | Memory-managed wrapper type.
newtype BookmarkFile = BookmarkFile (SP.ManagedPtr BookmarkFile)
    deriving (Eq)

instance SP.ManagedPtrNewtype BookmarkFile where
    toManagedPtr (BookmarkFile p) = p

foreign import ccall "g_bookmark_file_get_type" c_g_bookmark_file_get_type :: 
    IO GType

type instance O.ParentTypes BookmarkFile = '[]
instance O.HasParentTypes BookmarkFile

instance B.Types.TypedObject BookmarkFile where
    glibType = c_g_bookmark_file_get_type

instance B.Types.GBoxed BookmarkFile

-- | Convert t'BookmarkFile' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe BookmarkFile) where
    gvalueGType_ = c_g_bookmark_file_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr BookmarkFile)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr BookmarkFile)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed BookmarkFile ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList BookmarkFile
type instance O.AttributeList BookmarkFile = BookmarkFileAttributeList
type BookmarkFileAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method BookmarkFile::new
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "BookmarkFile" })
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_new" g_bookmark_file_new :: 
    IO (Ptr BookmarkFile)

-- | Creates a new empty t'GI.GLib.Structs.BookmarkFile.BookmarkFile' object.
-- 
-- Use 'GI.GLib.Structs.BookmarkFile.bookmarkFileLoadFromFile', 'GI.GLib.Structs.BookmarkFile.bookmarkFileLoadFromData'
-- or 'GI.GLib.Structs.BookmarkFile.bookmarkFileLoadFromDataDirs' to read an existing bookmark
-- file.
-- 
-- /Since: 2.12/
bookmarkFileNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m BookmarkFile
    -- ^ __Returns:__ an empty t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
bookmarkFileNew  = liftIO $ do
    result <- g_bookmark_file_new
    checkUnexpectedReturnNULL "bookmarkFileNew" result
    result' <- (wrapBoxed BookmarkFile) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method BookmarkFile::add_application
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the name of the application registering the bookmark\n  or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "exec"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "command line to be used to launch the bookmark or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_add_application" g_bookmark_file_add_application :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    CString ->                              -- name : TBasicType TUTF8
    CString ->                              -- exec : TBasicType TUTF8
    IO ()

-- | Adds the application with /@name@/ and /@exec@/ to the list of
-- applications that have registered a bookmark for /@uri@/ into
-- /@bookmark@/.
-- 
-- Every bookmark inside a t'GI.GLib.Structs.BookmarkFile.BookmarkFile' must have at least an
-- application registered.  Each application must provide a name, a
-- command line useful for launching the bookmark, the number of times
-- the bookmark has been registered by the application and the last
-- time the application registered this bookmark.
-- 
-- If /@name@/ is 'P.Nothing', the name of the application will be the
-- same returned by 'GI.GLib.Functions.getApplicationName'; if /@exec@/ is 'P.Nothing', the
-- command line will be a composition of the program name as
-- returned by 'GI.GLib.Functions.getPrgname' and the \"%u\" modifier, which will be
-- expanded to the bookmark\'s URI.
-- 
-- This function will automatically take care of updating the
-- registrations count and timestamping in case an application
-- with the same /@name@/ had already registered a bookmark for
-- /@uri@/ inside /@bookmark@/.
-- 
-- If no bookmark for /@uri@/ is found, one is created.
-- 
-- /Since: 2.12/
bookmarkFileAddApplication ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> Maybe (T.Text)
    -- ^ /@name@/: the name of the application registering the bookmark
    --   or 'P.Nothing'
    -> Maybe (T.Text)
    -- ^ /@exec@/: command line to be used to launch the bookmark or 'P.Nothing'
    -> m ()
bookmarkFileAddApplication bookmark uri name exec = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    maybeName <- case name of
        Nothing -> return FP.nullPtr
        Just jName -> do
            jName' <- textToCString jName
            return jName'
    maybeExec <- case exec of
        Nothing -> return FP.nullPtr
        Just jExec -> do
            jExec' <- textToCString jExec
            return jExec'
    g_bookmark_file_add_application bookmark' uri' maybeName maybeExec
    touchManagedPtr bookmark
    freeMem uri'
    freeMem maybeName
    freeMem maybeExec
    return ()

#if defined(ENABLE_OVERLOADING)
data BookmarkFileAddApplicationMethodInfo
instance (signature ~ (T.Text -> Maybe (T.Text) -> Maybe (T.Text) -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileAddApplicationMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileAddApplication

instance O.OverloadedMethodInfo BookmarkFileAddApplicationMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileAddApplication",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileAddApplication"
        })


#endif

-- method BookmarkFile::add_group
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the group name to be added"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_add_group" g_bookmark_file_add_group :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    CString ->                              -- group : TBasicType TUTF8
    IO ()

-- | Adds /@group@/ to the list of groups to which the bookmark for /@uri@/
-- belongs to.
-- 
-- If no bookmark for /@uri@/ is found then it is created.
-- 
-- /Since: 2.12/
bookmarkFileAddGroup ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> T.Text
    -- ^ /@group@/: the group name to be added
    -> m ()
bookmarkFileAddGroup bookmark uri group = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    group' <- textToCString group
    g_bookmark_file_add_group bookmark' uri' group'
    touchManagedPtr bookmark
    freeMem uri'
    freeMem group'
    return ()

#if defined(ENABLE_OVERLOADING)
data BookmarkFileAddGroupMethodInfo
instance (signature ~ (T.Text -> T.Text -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileAddGroupMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileAddGroup

instance O.OverloadedMethodInfo BookmarkFileAddGroupMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileAddGroup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileAddGroup"
        })


#endif

-- method BookmarkFile::copy
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "GLib" , name = "BookmarkFile" })
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_copy" g_bookmark_file_copy :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    IO (Ptr BookmarkFile)

-- | Deeply copies a /@bookmark@/ t'GI.GLib.Structs.BookmarkFile.BookmarkFile' object to a new one.
-- 
-- /Since: 2.76/
bookmarkFileCopy ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: A t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> m BookmarkFile
    -- ^ __Returns:__ the copy of /@bookmark@/. Use
    --   @/g_bookmark_free()/@ when finished using it.
bookmarkFileCopy bookmark = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    result <- g_bookmark_file_copy bookmark'
    checkUnexpectedReturnNULL "bookmarkFileCopy" result
    result' <- (wrapBoxed BookmarkFile) result
    touchManagedPtr bookmark
    return result'

#if defined(ENABLE_OVERLOADING)
data BookmarkFileCopyMethodInfo
instance (signature ~ (m BookmarkFile), MonadIO m) => O.OverloadedMethod BookmarkFileCopyMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileCopy

instance O.OverloadedMethodInfo BookmarkFileCopyMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileCopy",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileCopy"
        })


#endif

-- method BookmarkFile::free
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_free" g_bookmark_file_free :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    IO ()

-- | Frees a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'.
-- 
-- /Since: 2.12/
bookmarkFileFree ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> m ()
bookmarkFileFree bookmark = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    g_bookmark_file_free bookmark'
    touchManagedPtr bookmark
    return ()

#if defined(ENABLE_OVERLOADING)
data BookmarkFileFreeMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod BookmarkFileFreeMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileFree

instance O.OverloadedMethodInfo BookmarkFileFreeMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileFree",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileFree"
        })


#endif

-- method BookmarkFile::get_added
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType Ttime_t)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_get_added" g_bookmark_file_get_added :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO FCT.CTime

{-# DEPRECATED bookmarkFileGetAdded ["(Since version 2.66)","Use 'GI.GLib.Structs.BookmarkFile.bookmarkFileGetAddedDateTime' instead, as","   @time_t@ is deprecated due to the year 2038 problem."] #-}
-- | Gets the time the bookmark for /@uri@/ was added to /@bookmark@/
-- 
-- In the event the URI cannot be found, -1 is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.
-- 
-- /Since: 2.12/
bookmarkFileGetAdded ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> m FCT.CTime
    -- ^ __Returns:__ a timestamp /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileGetAdded bookmark uri = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    onException (do
        result <- propagateGError $ g_bookmark_file_get_added bookmark' uri'
        touchManagedPtr bookmark
        freeMem uri'
        return result
     ) (do
        freeMem uri'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetAddedMethodInfo
instance (signature ~ (T.Text -> m FCT.CTime), MonadIO m) => O.OverloadedMethod BookmarkFileGetAddedMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetAdded

instance O.OverloadedMethodInfo BookmarkFileGetAddedMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetAdded",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetAdded"
        })


#endif

-- method BookmarkFile::get_added_date_time
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_get_added_date_time" g_bookmark_file_get_added_date_time :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr GLib.DateTime.DateTime)

-- | Gets the time the bookmark for /@uri@/ was added to /@bookmark@/
-- 
-- In the event the URI cannot be found, 'P.Nothing' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.
-- 
-- /Since: 2.66/
bookmarkFileGetAddedDateTime ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> m GLib.DateTime.DateTime
    -- ^ __Returns:__ a t'GI.GLib.Structs.DateTime.DateTime' /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileGetAddedDateTime bookmark uri = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    onException (do
        result <- propagateGError $ g_bookmark_file_get_added_date_time bookmark' uri'
        checkUnexpectedReturnNULL "bookmarkFileGetAddedDateTime" result
        result' <- (newBoxed GLib.DateTime.DateTime) result
        touchManagedPtr bookmark
        freeMem uri'
        return result'
     ) (do
        freeMem uri'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetAddedDateTimeMethodInfo
instance (signature ~ (T.Text -> m GLib.DateTime.DateTime), MonadIO m) => O.OverloadedMethod BookmarkFileGetAddedDateTimeMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetAddedDateTime

instance O.OverloadedMethodInfo BookmarkFileGetAddedDateTimeMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetAddedDateTime",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetAddedDateTime"
        })


#endif

-- method BookmarkFile::get_app_info
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an application's name"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "exec"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "return location for the command line of the application, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "count"
--           , argType = TBasicType TUInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the registration count, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "stamp"
--           , argType = TBasicType Ttime_t
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the last registration time, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_get_app_info" g_bookmark_file_get_app_info :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    CString ->                              -- name : TBasicType TUTF8
    Ptr CString ->                          -- exec : TBasicType TUTF8
    Ptr Word32 ->                           -- count : TBasicType TUInt
    Ptr FCT.CTime ->                        -- stamp : TBasicType Ttime_t
    Ptr (Ptr GError) ->                     -- error
    IO CInt

{-# DEPRECATED bookmarkFileGetAppInfo ["(Since version 2.66)","Use 'GI.GLib.Structs.BookmarkFile.bookmarkFileGetApplicationInfo' instead, as","   @time_t@ is deprecated due to the year 2038 problem."] #-}
-- | Gets the registration information of /@appName@/ for the bookmark for
-- /@uri@/.  See 'GI.GLib.Structs.BookmarkFile.bookmarkFileSetApplicationInfo' for more information about
-- the returned data.
-- 
-- The string returned in /@appExec@/ must be freed.
-- 
-- In the event the URI cannot be found, 'P.False' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.  In the
-- event that no application with name /@appName@/ has registered a bookmark
-- for /@uri@/,  'P.False' is returned and error is set to
-- 'GI.GLib.Enums.BookmarkFileErrorAppNotRegistered'. In the event that unquoting
-- the command line fails, an error of the @/G_SHELL_ERROR/@ domain is
-- set and 'P.False' is returned.
-- 
-- /Since: 2.12/
bookmarkFileGetAppInfo ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> T.Text
    -- ^ /@name@/: an application\'s name
    -> m ((T.Text, Word32, FCT.CTime))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileGetAppInfo bookmark uri name = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    name' <- textToCString name
    exec <- callocMem :: IO (Ptr CString)
    count <- allocMem :: IO (Ptr Word32)
    stamp <- allocMem :: IO (Ptr FCT.CTime)
    onException (do
        _ <- propagateGError $ g_bookmark_file_get_app_info bookmark' uri' name' exec count stamp
        exec' <- peek exec
        exec'' <- cstringToText exec'
        freeMem exec'
        count' <- peek count
        stamp' <- peek stamp
        touchManagedPtr bookmark
        freeMem uri'
        freeMem name'
        freeMem exec
        freeMem count
        freeMem stamp
        return (exec'', count', stamp')
     ) (do
        freeMem uri'
        freeMem name'
        freeMem exec
        freeMem count
        freeMem stamp
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetAppInfoMethodInfo
instance (signature ~ (T.Text -> T.Text -> m ((T.Text, Word32, FCT.CTime))), MonadIO m) => O.OverloadedMethod BookmarkFileGetAppInfoMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetAppInfo

instance O.OverloadedMethodInfo BookmarkFileGetAppInfoMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetAppInfo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetAppInfo"
        })


#endif

-- method BookmarkFile::get_application_info
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an application's name"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "exec"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "return location for the command line of the application, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "count"
--           , argType = TBasicType TUInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the registration count, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "stamp"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the last registration time, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_get_application_info" g_bookmark_file_get_application_info :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    CString ->                              -- name : TBasicType TUTF8
    Ptr CString ->                          -- exec : TBasicType TUTF8
    Ptr Word32 ->                           -- count : TBasicType TUInt
    Ptr (Ptr GLib.DateTime.DateTime) ->     -- stamp : TInterface (Name {namespace = "GLib", name = "DateTime"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Gets the registration information of /@appName@/ for the bookmark for
-- /@uri@/.  See 'GI.GLib.Structs.BookmarkFile.bookmarkFileSetApplicationInfo' for more information about
-- the returned data.
-- 
-- The string returned in /@appExec@/ must be freed.
-- 
-- In the event the URI cannot be found, 'P.False' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.  In the
-- event that no application with name /@appName@/ has registered a bookmark
-- for /@uri@/,  'P.False' is returned and error is set to
-- 'GI.GLib.Enums.BookmarkFileErrorAppNotRegistered'. In the event that unquoting
-- the command line fails, an error of the @/G_SHELL_ERROR/@ domain is
-- set and 'P.False' is returned.
-- 
-- /Since: 2.66/
bookmarkFileGetApplicationInfo ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> T.Text
    -- ^ /@name@/: an application\'s name
    -> m ((T.Text, Word32, GLib.DateTime.DateTime))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileGetApplicationInfo bookmark uri name = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    name' <- textToCString name
    exec <- callocMem :: IO (Ptr CString)
    count <- allocMem :: IO (Ptr Word32)
    stamp <- callocMem :: IO (Ptr (Ptr GLib.DateTime.DateTime))
    onException (do
        _ <- propagateGError $ g_bookmark_file_get_application_info bookmark' uri' name' exec count stamp
        exec' <- peek exec
        exec'' <- cstringToText exec'
        freeMem exec'
        count' <- peek count
        stamp' <- peek stamp
        stamp'' <- (newBoxed GLib.DateTime.DateTime) stamp'
        touchManagedPtr bookmark
        freeMem uri'
        freeMem name'
        freeMem exec
        freeMem count
        freeMem stamp
        return (exec'', count', stamp'')
     ) (do
        freeMem uri'
        freeMem name'
        freeMem exec
        freeMem count
        freeMem stamp
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetApplicationInfoMethodInfo
instance (signature ~ (T.Text -> T.Text -> m ((T.Text, Word32, GLib.DateTime.DateTime))), MonadIO m) => O.OverloadedMethod BookmarkFileGetApplicationInfoMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetApplicationInfo

instance O.OverloadedMethodInfo BookmarkFileGetApplicationInfoMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetApplicationInfo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetApplicationInfo"
        })


#endif

-- method BookmarkFile::get_applications
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location of the length of the returned list, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionOut
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just "return location of the length of the returned list, or %NULL"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Just (TCArray False (-1) 2 (TBasicType TUTF8))
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_get_applications" g_bookmark_file_get_applications :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr FCT.CSize ->                        -- length : TBasicType TSize
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr CString)

-- | Retrieves the names of the applications that have registered the
-- bookmark for /@uri@/.
-- 
-- In the event the URI cannot be found, 'P.Nothing' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.
-- 
-- /Since: 2.12/
bookmarkFileGetApplications ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> m [T.Text]
    -- ^ __Returns:__ a newly allocated 'P.Nothing'-terminated array of strings.
    --   Use 'GI.GLib.Functions.strfreev' to free it. /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileGetApplications bookmark uri = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    length_ <- allocMem :: IO (Ptr FCT.CSize)
    onException (do
        result <- propagateGError $ g_bookmark_file_get_applications bookmark' uri' length_
        length_' <- peek length_
        checkUnexpectedReturnNULL "bookmarkFileGetApplications" result
        result' <- (unpackUTF8CArrayWithLength length_') result
        (mapCArrayWithLength length_') freeMem result
        freeMem result
        touchManagedPtr bookmark
        freeMem uri'
        freeMem length_
        return result'
     ) (do
        freeMem uri'
        freeMem length_
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetApplicationsMethodInfo
instance (signature ~ (T.Text -> m [T.Text]), MonadIO m) => O.OverloadedMethod BookmarkFileGetApplicationsMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetApplications

instance O.OverloadedMethodInfo BookmarkFileGetApplicationsMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetApplications",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetApplications"
        })


#endif

-- method BookmarkFile::get_description
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_get_description" g_bookmark_file_get_description :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CString

-- | Retrieves the description of the bookmark for /@uri@/.
-- 
-- In the event the URI cannot be found, 'P.Nothing' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.
-- 
-- /Since: 2.12/
bookmarkFileGetDescription ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> m T.Text
    -- ^ __Returns:__ a newly allocated string or 'P.Nothing' if the specified
    --   URI cannot be found. /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileGetDescription bookmark uri = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    onException (do
        result <- propagateGError $ g_bookmark_file_get_description bookmark' uri'
        checkUnexpectedReturnNULL "bookmarkFileGetDescription" result
        result' <- cstringToText result
        freeMem result
        touchManagedPtr bookmark
        freeMem uri'
        return result'
     ) (do
        freeMem uri'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetDescriptionMethodInfo
instance (signature ~ (T.Text -> m T.Text), MonadIO m) => O.OverloadedMethod BookmarkFileGetDescriptionMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetDescription

instance O.OverloadedMethodInfo BookmarkFileGetDescriptionMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetDescription",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetDescription"
        })


#endif

-- method BookmarkFile::get_groups
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "return location for the length of the returned string, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionOut
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just
--                          "return location for the length of the returned string, or %NULL"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Just (TCArray False (-1) 2 (TBasicType TUTF8))
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_get_groups" g_bookmark_file_get_groups :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr FCT.CSize ->                        -- length : TBasicType TSize
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr CString)

-- | Retrieves the list of group names of the bookmark for /@uri@/.
-- 
-- In the event the URI cannot be found, 'P.Nothing' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.
-- 
-- The returned array is 'P.Nothing' terminated, so /@length@/ may optionally
-- be 'P.Nothing'.
-- 
-- /Since: 2.12/
bookmarkFileGetGroups ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> m [T.Text]
    -- ^ __Returns:__ a newly allocated 'P.Nothing'-terminated array of group names.
    --   Use 'GI.GLib.Functions.strfreev' to free it. /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileGetGroups bookmark uri = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    length_ <- allocMem :: IO (Ptr FCT.CSize)
    onException (do
        result <- propagateGError $ g_bookmark_file_get_groups bookmark' uri' length_
        length_' <- peek length_
        checkUnexpectedReturnNULL "bookmarkFileGetGroups" result
        result' <- (unpackUTF8CArrayWithLength length_') result
        (mapCArrayWithLength length_') freeMem result
        freeMem result
        touchManagedPtr bookmark
        freeMem uri'
        freeMem length_
        return result'
     ) (do
        freeMem uri'
        freeMem length_
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetGroupsMethodInfo
instance (signature ~ (T.Text -> m [T.Text]), MonadIO m) => O.OverloadedMethod BookmarkFileGetGroupsMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetGroups

instance O.OverloadedMethodInfo BookmarkFileGetGroupsMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetGroups",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetGroups"
        })


#endif

-- method BookmarkFile::get_icon
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "href"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the icon's location or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "mime_type"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the icon's MIME type or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_get_icon" g_bookmark_file_get_icon :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr CString ->                          -- href : TBasicType TUTF8
    Ptr CString ->                          -- mime_type : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Gets the icon of the bookmark for /@uri@/.
-- 
-- In the event the URI cannot be found, 'P.False' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.
-- 
-- /Since: 2.12/
bookmarkFileGetIcon ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> m ((T.Text, T.Text))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileGetIcon bookmark uri = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    href <- callocMem :: IO (Ptr CString)
    mimeType <- callocMem :: IO (Ptr CString)
    onException (do
        _ <- propagateGError $ g_bookmark_file_get_icon bookmark' uri' href mimeType
        href' <- peek href
        href'' <- cstringToText href'
        freeMem href'
        mimeType' <- peek mimeType
        mimeType'' <- cstringToText mimeType'
        freeMem mimeType'
        touchManagedPtr bookmark
        freeMem uri'
        freeMem href
        freeMem mimeType
        return (href'', mimeType'')
     ) (do
        freeMem uri'
        freeMem href
        freeMem mimeType
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetIconMethodInfo
instance (signature ~ (T.Text -> m ((T.Text, T.Text))), MonadIO m) => O.OverloadedMethod BookmarkFileGetIconMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetIcon

instance O.OverloadedMethodInfo BookmarkFileGetIconMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetIcon",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetIcon"
        })


#endif

-- method BookmarkFile::get_is_private
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_get_is_private" g_bookmark_file_get_is_private :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Gets whether the private flag of the bookmark for /@uri@/ is set.
-- 
-- In the event the URI cannot be found, 'P.False' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.  In the
-- event that the private flag cannot be found, 'P.False' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorInvalidValue'.
-- 
-- /Since: 2.12/
bookmarkFileGetIsPrivate ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileGetIsPrivate bookmark uri = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    onException (do
        _ <- propagateGError $ g_bookmark_file_get_is_private bookmark' uri'
        touchManagedPtr bookmark
        freeMem uri'
        return ()
     ) (do
        freeMem uri'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetIsPrivateMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileGetIsPrivateMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetIsPrivate

instance O.OverloadedMethodInfo BookmarkFileGetIsPrivateMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetIsPrivate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetIsPrivate"
        })


#endif

-- method BookmarkFile::get_mime_type
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_get_mime_type" g_bookmark_file_get_mime_type :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CString

-- | Retrieves the MIME type of the resource pointed by /@uri@/.
-- 
-- In the event the URI cannot be found, 'P.Nothing' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.  In the
-- event that the MIME type cannot be found, 'P.Nothing' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorInvalidValue'.
-- 
-- /Since: 2.12/
bookmarkFileGetMimeType ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> m T.Text
    -- ^ __Returns:__ a newly allocated string or 'P.Nothing' if the specified
    --   URI cannot be found. /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileGetMimeType bookmark uri = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    onException (do
        result <- propagateGError $ g_bookmark_file_get_mime_type bookmark' uri'
        checkUnexpectedReturnNULL "bookmarkFileGetMimeType" result
        result' <- cstringToText result
        freeMem result
        touchManagedPtr bookmark
        freeMem uri'
        return result'
     ) (do
        freeMem uri'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetMimeTypeMethodInfo
instance (signature ~ (T.Text -> m T.Text), MonadIO m) => O.OverloadedMethod BookmarkFileGetMimeTypeMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetMimeType

instance O.OverloadedMethodInfo BookmarkFileGetMimeTypeMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetMimeType",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetMimeType"
        })


#endif

-- method BookmarkFile::get_modified
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType Ttime_t)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_get_modified" g_bookmark_file_get_modified :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO FCT.CTime

{-# DEPRECATED bookmarkFileGetModified ["(Since version 2.66)","Use 'GI.GLib.Structs.BookmarkFile.bookmarkFileGetModifiedDateTime' instead, as","   @time_t@ is deprecated due to the year 2038 problem."] #-}
-- | Gets the time when the bookmark for /@uri@/ was last modified.
-- 
-- In the event the URI cannot be found, -1 is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.
-- 
-- /Since: 2.12/
bookmarkFileGetModified ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> m FCT.CTime
    -- ^ __Returns:__ a timestamp /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileGetModified bookmark uri = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    onException (do
        result <- propagateGError $ g_bookmark_file_get_modified bookmark' uri'
        touchManagedPtr bookmark
        freeMem uri'
        return result
     ) (do
        freeMem uri'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetModifiedMethodInfo
instance (signature ~ (T.Text -> m FCT.CTime), MonadIO m) => O.OverloadedMethod BookmarkFileGetModifiedMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetModified

instance O.OverloadedMethodInfo BookmarkFileGetModifiedMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetModified",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetModified"
        })


#endif

-- method BookmarkFile::get_modified_date_time
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_get_modified_date_time" g_bookmark_file_get_modified_date_time :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr GLib.DateTime.DateTime)

-- | Gets the time when the bookmark for /@uri@/ was last modified.
-- 
-- In the event the URI cannot be found, 'P.Nothing' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.
-- 
-- /Since: 2.66/
bookmarkFileGetModifiedDateTime ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> m GLib.DateTime.DateTime
    -- ^ __Returns:__ a t'GI.GLib.Structs.DateTime.DateTime' /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileGetModifiedDateTime bookmark uri = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    onException (do
        result <- propagateGError $ g_bookmark_file_get_modified_date_time bookmark' uri'
        checkUnexpectedReturnNULL "bookmarkFileGetModifiedDateTime" result
        result' <- (newBoxed GLib.DateTime.DateTime) result
        touchManagedPtr bookmark
        freeMem uri'
        return result'
     ) (do
        freeMem uri'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetModifiedDateTimeMethodInfo
instance (signature ~ (T.Text -> m GLib.DateTime.DateTime), MonadIO m) => O.OverloadedMethod BookmarkFileGetModifiedDateTimeMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetModifiedDateTime

instance O.OverloadedMethodInfo BookmarkFileGetModifiedDateTimeMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetModifiedDateTime",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetModifiedDateTime"
        })


#endif

-- method BookmarkFile::get_size
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_get_size" g_bookmark_file_get_size :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    IO Int32

-- | Gets the number of bookmarks inside /@bookmark@/.
-- 
-- /Since: 2.12/
bookmarkFileGetSize ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> m Int32
    -- ^ __Returns:__ the number of bookmarks
bookmarkFileGetSize bookmark = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    result <- g_bookmark_file_get_size bookmark'
    touchManagedPtr bookmark
    return result

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetSizeMethodInfo
instance (signature ~ (m Int32), MonadIO m) => O.OverloadedMethod BookmarkFileGetSizeMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetSize

instance O.OverloadedMethodInfo BookmarkFileGetSizeMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetSize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetSize"
        })


#endif

-- method BookmarkFile::get_title
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_get_title" g_bookmark_file_get_title :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CString

-- | Returns the title of the bookmark for /@uri@/.
-- 
-- If /@uri@/ is 'P.Nothing', the title of /@bookmark@/ is returned.
-- 
-- In the event the URI cannot be found, 'P.Nothing' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.
-- 
-- /Since: 2.12/
bookmarkFileGetTitle ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> Maybe (T.Text)
    -- ^ /@uri@/: a valid URI or 'P.Nothing'
    -> m T.Text
    -- ^ __Returns:__ a newly allocated string or 'P.Nothing' if the specified
    --   URI cannot be found. /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileGetTitle bookmark uri = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    maybeUri <- case uri of
        Nothing -> return FP.nullPtr
        Just jUri -> do
            jUri' <- textToCString jUri
            return jUri'
    onException (do
        result <- propagateGError $ g_bookmark_file_get_title bookmark' maybeUri
        checkUnexpectedReturnNULL "bookmarkFileGetTitle" result
        result' <- cstringToText result
        freeMem result
        touchManagedPtr bookmark
        freeMem maybeUri
        return result'
     ) (do
        freeMem maybeUri
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetTitleMethodInfo
instance (signature ~ (Maybe (T.Text) -> m T.Text), MonadIO m) => O.OverloadedMethod BookmarkFileGetTitleMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetTitle

instance O.OverloadedMethodInfo BookmarkFileGetTitleMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetTitle",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetTitle"
        })


#endif

-- method BookmarkFile::get_uris
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the number of returned URIs, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionOut
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just "return location for the number of returned URIs, or %NULL"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Just (TCArray False (-1) 1 (TBasicType TUTF8))
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_get_uris" g_bookmark_file_get_uris :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    Ptr FCT.CSize ->                        -- length : TBasicType TSize
    IO (Ptr CString)

-- | Returns all URIs of the bookmarks in the bookmark file /@bookmark@/.
-- The array of returned URIs will be 'P.Nothing'-terminated, so /@length@/ may
-- optionally be 'P.Nothing'.
-- 
-- /Since: 2.12/
bookmarkFileGetUris ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> m [T.Text]
    -- ^ __Returns:__ a newly allocated 'P.Nothing'-terminated array of strings.
    --   Use 'GI.GLib.Functions.strfreev' to free it.
bookmarkFileGetUris bookmark = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    length_ <- allocMem :: IO (Ptr FCT.CSize)
    result <- g_bookmark_file_get_uris bookmark' length_
    length_' <- peek length_
    checkUnexpectedReturnNULL "bookmarkFileGetUris" result
    result' <- (unpackUTF8CArrayWithLength length_') result
    (mapCArrayWithLength length_') freeMem result
    freeMem result
    touchManagedPtr bookmark
    freeMem length_
    return result'

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetUrisMethodInfo
instance (signature ~ (m [T.Text]), MonadIO m) => O.OverloadedMethod BookmarkFileGetUrisMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetUris

instance O.OverloadedMethodInfo BookmarkFileGetUrisMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetUris",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetUris"
        })


#endif

-- method BookmarkFile::get_visited
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType Ttime_t)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_get_visited" g_bookmark_file_get_visited :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO FCT.CTime

{-# DEPRECATED bookmarkFileGetVisited ["(Since version 2.66)","Use 'GI.GLib.Structs.BookmarkFile.bookmarkFileGetVisitedDateTime' instead, as","   @time_t@ is deprecated due to the year 2038 problem."] #-}
-- | Gets the time the bookmark for /@uri@/ was last visited.
-- 
-- In the event the URI cannot be found, -1 is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.
-- 
-- /Since: 2.12/
bookmarkFileGetVisited ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> m FCT.CTime
    -- ^ __Returns:__ a timestamp. /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileGetVisited bookmark uri = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    onException (do
        result <- propagateGError $ g_bookmark_file_get_visited bookmark' uri'
        touchManagedPtr bookmark
        freeMem uri'
        return result
     ) (do
        freeMem uri'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetVisitedMethodInfo
instance (signature ~ (T.Text -> m FCT.CTime), MonadIO m) => O.OverloadedMethod BookmarkFileGetVisitedMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetVisited

instance O.OverloadedMethodInfo BookmarkFileGetVisitedMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetVisited",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetVisited"
        })


#endif

-- method BookmarkFile::get_visited_date_time
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "DateTime" })
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_get_visited_date_time" g_bookmark_file_get_visited_date_time :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr GLib.DateTime.DateTime)

-- | Gets the time the bookmark for /@uri@/ was last visited.
-- 
-- In the event the URI cannot be found, 'P.Nothing' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.
-- 
-- /Since: 2.66/
bookmarkFileGetVisitedDateTime ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> m GLib.DateTime.DateTime
    -- ^ __Returns:__ a t'GI.GLib.Structs.DateTime.DateTime' /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileGetVisitedDateTime bookmark uri = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    onException (do
        result <- propagateGError $ g_bookmark_file_get_visited_date_time bookmark' uri'
        checkUnexpectedReturnNULL "bookmarkFileGetVisitedDateTime" result
        result' <- (newBoxed GLib.DateTime.DateTime) result
        touchManagedPtr bookmark
        freeMem uri'
        return result'
     ) (do
        freeMem uri'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileGetVisitedDateTimeMethodInfo
instance (signature ~ (T.Text -> m GLib.DateTime.DateTime), MonadIO m) => O.OverloadedMethod BookmarkFileGetVisitedDateTimeMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileGetVisitedDateTime

instance O.OverloadedMethodInfo BookmarkFileGetVisitedDateTimeMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileGetVisitedDateTime",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileGetVisitedDateTime"
        })


#endif

-- method BookmarkFile::has_application
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the application"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_has_application" g_bookmark_file_has_application :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    CString ->                              -- name : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Checks whether the bookmark for /@uri@/ inside /@bookmark@/ has been
-- registered by application /@name@/.
-- 
-- In the event the URI cannot be found, 'P.False' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.
-- 
-- /Since: 2.12/
bookmarkFileHasApplication ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> T.Text
    -- ^ /@name@/: the name of the application
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileHasApplication bookmark uri name = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    name' <- textToCString name
    onException (do
        _ <- propagateGError $ g_bookmark_file_has_application bookmark' uri' name'
        touchManagedPtr bookmark
        freeMem uri'
        freeMem name'
        return ()
     ) (do
        freeMem uri'
        freeMem name'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileHasApplicationMethodInfo
instance (signature ~ (T.Text -> T.Text -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileHasApplicationMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileHasApplication

instance O.OverloadedMethodInfo BookmarkFileHasApplicationMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileHasApplication",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileHasApplication"
        })


#endif

-- method BookmarkFile::has_group
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the group name to be searched"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_has_group" g_bookmark_file_has_group :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    CString ->                              -- group : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Checks whether /@group@/ appears in the list of groups to which
-- the bookmark for /@uri@/ belongs to.
-- 
-- In the event the URI cannot be found, 'P.False' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.
-- 
-- /Since: 2.12/
bookmarkFileHasGroup ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> T.Text
    -- ^ /@group@/: the group name to be searched
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileHasGroup bookmark uri group = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    group' <- textToCString group
    onException (do
        _ <- propagateGError $ g_bookmark_file_has_group bookmark' uri' group'
        touchManagedPtr bookmark
        freeMem uri'
        freeMem group'
        return ()
     ) (do
        freeMem uri'
        freeMem group'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileHasGroupMethodInfo
instance (signature ~ (T.Text -> T.Text -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileHasGroupMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileHasGroup

instance O.OverloadedMethodInfo BookmarkFileHasGroupMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileHasGroup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileHasGroup"
        })


#endif

-- method BookmarkFile::has_item
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_has_item" g_bookmark_file_has_item :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    IO CInt

-- | Looks whether the desktop bookmark has an item with its URI set to /@uri@/.
-- 
-- /Since: 2.12/
bookmarkFileHasItem ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@uri@/ is inside /@bookmark@/, 'P.False' otherwise
bookmarkFileHasItem bookmark uri = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    result <- g_bookmark_file_has_item bookmark' uri'
    let result' = (/= 0) result
    touchManagedPtr bookmark
    freeMem uri'
    return result'

#if defined(ENABLE_OVERLOADING)
data BookmarkFileHasItemMethodInfo
instance (signature ~ (T.Text -> m Bool), MonadIO m) => O.OverloadedMethod BookmarkFileHasItemMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileHasItem

instance O.OverloadedMethodInfo BookmarkFileHasItemMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileHasItem",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileHasItem"
        })


#endif

-- method BookmarkFile::load_from_data
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an empty #GBookmarkFile struct"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "data"
--           , argType = TCArray False (-1) 2 (TBasicType TUInt8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "desktop bookmarks\n   loaded in memory"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the length of @data in bytes"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the length of @data in bytes"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_load_from_data" g_bookmark_file_load_from_data :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    Ptr Word8 ->                            -- data : TCArray False (-1) 2 (TBasicType TUInt8)
    FCT.CSize ->                            -- length : TBasicType TSize
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Loads a bookmark file from memory into an empty t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
-- structure.  If the object cannot be created then /@error@/ is set to a
-- t'GI.GLib.Enums.BookmarkFileError'.
-- 
-- /Since: 2.12/
bookmarkFileLoadFromData ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: an empty t'GI.GLib.Structs.BookmarkFile.BookmarkFile' struct
    -> ByteString
    -- ^ /@data@/: desktop bookmarks
    --    loaded in memory
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileLoadFromData bookmark data_ = liftIO $ do
    let length_ = fromIntegral $ B.length data_
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    data_' <- packByteString data_
    onException (do
        _ <- propagateGError $ g_bookmark_file_load_from_data bookmark' data_' length_
        touchManagedPtr bookmark
        freeMem data_'
        return ()
     ) (do
        freeMem data_'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileLoadFromDataMethodInfo
instance (signature ~ (ByteString -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileLoadFromDataMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileLoadFromData

instance O.OverloadedMethodInfo BookmarkFileLoadFromDataMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileLoadFromData",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileLoadFromData"
        })


#endif

-- method BookmarkFile::load_from_data_dirs
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "file"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a relative path to a filename to open and parse"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "full_path"
--           , argType = TBasicType TFileName
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "return location for a string\n   containing the full path of the file, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_load_from_data_dirs" g_bookmark_file_load_from_data_dirs :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- file : TBasicType TFileName
    Ptr CString ->                          -- full_path : TBasicType TFileName
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | This function looks for a desktop bookmark file named /@file@/ in the
-- paths returned from 'GI.GLib.Functions.getUserDataDir' and 'GI.GLib.Functions.getSystemDataDirs',
-- loads the file into /@bookmark@/ and returns the file\'s full path in
-- /@fullPath@/.  If the file could not be loaded then /@error@/ is
-- set to either a t'GI.GLib.Enums.FileError' or t'GI.GLib.Enums.BookmarkFileError'.
-- 
-- /Since: 2.12/
bookmarkFileLoadFromDataDirs ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> [Char]
    -- ^ /@file@/: a relative path to a filename to open and parse
    -> m ((Maybe [Char]))
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileLoadFromDataDirs bookmark file = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    file' <- stringToCString file
    fullPath <- callocMem :: IO (Ptr CString)
    onException (do
        _ <- propagateGError $ g_bookmark_file_load_from_data_dirs bookmark' file' fullPath
        fullPath' <- peek fullPath
        maybeFullPath' <- convertIfNonNull fullPath' $ \fullPath'' -> do
            fullPath''' <- cstringToString fullPath''
            return fullPath'''
        freeMem fullPath'
        touchManagedPtr bookmark
        freeMem file'
        freeMem fullPath
        return maybeFullPath'
     ) (do
        freeMem file'
        freeMem fullPath
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileLoadFromDataDirsMethodInfo
instance (signature ~ ([Char] -> m ((Maybe [Char]))), MonadIO m) => O.OverloadedMethod BookmarkFileLoadFromDataDirsMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileLoadFromDataDirs

instance O.OverloadedMethodInfo BookmarkFileLoadFromDataDirsMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileLoadFromDataDirs",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileLoadFromDataDirs"
        })


#endif

-- method BookmarkFile::load_from_file
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an empty #GBookmarkFile struct"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "filename"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the path of a filename to load, in the\n    GLib file name encoding"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_load_from_file" g_bookmark_file_load_from_file :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- filename : TBasicType TFileName
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Loads a desktop bookmark file into an empty t'GI.GLib.Structs.BookmarkFile.BookmarkFile' structure.
-- If the file could not be loaded then /@error@/ is set to either a t'GI.GLib.Enums.FileError'
-- or t'GI.GLib.Enums.BookmarkFileError'.
-- 
-- /Since: 2.12/
bookmarkFileLoadFromFile ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: an empty t'GI.GLib.Structs.BookmarkFile.BookmarkFile' struct
    -> [Char]
    -- ^ /@filename@/: the path of a filename to load, in the
    --     GLib file name encoding
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileLoadFromFile bookmark filename = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    filename' <- stringToCString filename
    onException (do
        _ <- propagateGError $ g_bookmark_file_load_from_file bookmark' filename'
        touchManagedPtr bookmark
        freeMem filename'
        return ()
     ) (do
        freeMem filename'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileLoadFromFileMethodInfo
instance (signature ~ ([Char] -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileLoadFromFileMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileLoadFromFile

instance O.OverloadedMethodInfo BookmarkFileLoadFromFileMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileLoadFromFile",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileLoadFromFile"
        })


#endif

-- method BookmarkFile::move_item
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "old_uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "new_uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_move_item" g_bookmark_file_move_item :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- old_uri : TBasicType TUTF8
    CString ->                              -- new_uri : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Changes the URI of a bookmark item from /@oldUri@/ to /@newUri@/.  Any
-- existing bookmark for /@newUri@/ will be overwritten.  If /@newUri@/ is
-- 'P.Nothing', then the bookmark is removed.
-- 
-- In the event the URI cannot be found, 'P.False' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.
-- 
-- /Since: 2.12/
bookmarkFileMoveItem ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@oldUri@/: a valid URI
    -> Maybe (T.Text)
    -- ^ /@newUri@/: a valid URI, or 'P.Nothing'
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileMoveItem bookmark oldUri newUri = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    oldUri' <- textToCString oldUri
    maybeNewUri <- case newUri of
        Nothing -> return FP.nullPtr
        Just jNewUri -> do
            jNewUri' <- textToCString jNewUri
            return jNewUri'
    onException (do
        _ <- propagateGError $ g_bookmark_file_move_item bookmark' oldUri' maybeNewUri
        touchManagedPtr bookmark
        freeMem oldUri'
        freeMem maybeNewUri
        return ()
     ) (do
        freeMem oldUri'
        freeMem maybeNewUri
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileMoveItemMethodInfo
instance (signature ~ (T.Text -> Maybe (T.Text) -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileMoveItemMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileMoveItem

instance O.OverloadedMethodInfo BookmarkFileMoveItemMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileMoveItem",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileMoveItem"
        })


#endif

-- method BookmarkFile::remove_application
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the application"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_remove_application" g_bookmark_file_remove_application :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    CString ->                              -- name : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Removes application registered with /@name@/ from the list of applications
-- that have registered a bookmark for /@uri@/ inside /@bookmark@/.
-- 
-- In the event the URI cannot be found, 'P.False' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.
-- In the event that no application with name /@appName@/ has registered
-- a bookmark for /@uri@/,  'P.False' is returned and error is set to
-- 'GI.GLib.Enums.BookmarkFileErrorAppNotRegistered'.
-- 
-- /Since: 2.12/
bookmarkFileRemoveApplication ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> T.Text
    -- ^ /@name@/: the name of the application
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileRemoveApplication bookmark uri name = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    name' <- textToCString name
    onException (do
        _ <- propagateGError $ g_bookmark_file_remove_application bookmark' uri' name'
        touchManagedPtr bookmark
        freeMem uri'
        freeMem name'
        return ()
     ) (do
        freeMem uri'
        freeMem name'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileRemoveApplicationMethodInfo
instance (signature ~ (T.Text -> T.Text -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileRemoveApplicationMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileRemoveApplication

instance O.OverloadedMethodInfo BookmarkFileRemoveApplicationMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileRemoveApplication",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileRemoveApplication"
        })


#endif

-- method BookmarkFile::remove_group
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the group name to be removed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_remove_group" g_bookmark_file_remove_group :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    CString ->                              -- group : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Removes /@group@/ from the list of groups to which the bookmark
-- for /@uri@/ belongs to.
-- 
-- In the event the URI cannot be found, 'P.False' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'.
-- In the event no group was defined, 'P.False' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorInvalidValue'.
-- 
-- /Since: 2.12/
bookmarkFileRemoveGroup ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> T.Text
    -- ^ /@group@/: the group name to be removed
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileRemoveGroup bookmark uri group = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    group' <- textToCString group
    onException (do
        _ <- propagateGError $ g_bookmark_file_remove_group bookmark' uri' group'
        touchManagedPtr bookmark
        freeMem uri'
        freeMem group'
        return ()
     ) (do
        freeMem uri'
        freeMem group'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileRemoveGroupMethodInfo
instance (signature ~ (T.Text -> T.Text -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileRemoveGroupMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileRemoveGroup

instance O.OverloadedMethodInfo BookmarkFileRemoveGroupMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileRemoveGroup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileRemoveGroup"
        })


#endif

-- method BookmarkFile::remove_item
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_remove_item" g_bookmark_file_remove_item :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Removes the bookmark for /@uri@/ from the bookmark file /@bookmark@/.
-- 
-- /Since: 2.12/
bookmarkFileRemoveItem ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileRemoveItem bookmark uri = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    onException (do
        _ <- propagateGError $ g_bookmark_file_remove_item bookmark' uri'
        touchManagedPtr bookmark
        freeMem uri'
        return ()
     ) (do
        freeMem uri'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileRemoveItemMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileRemoveItemMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileRemoveItem

instance O.OverloadedMethodInfo BookmarkFileRemoveItemMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileRemoveItem",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileRemoveItem"
        })


#endif

-- method BookmarkFile::set_added
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "added"
--           , argType = TBasicType Ttime_t
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a timestamp or -1 to use the current time"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_set_added" g_bookmark_file_set_added :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    FCT.CTime ->                            -- added : TBasicType Ttime_t
    IO ()

{-# DEPRECATED bookmarkFileSetAdded ["(Since version 2.66)","Use 'GI.GLib.Structs.BookmarkFile.bookmarkFileSetAddedDateTime' instead, as","   @time_t@ is deprecated due to the year 2038 problem."] #-}
-- | Sets the time the bookmark for /@uri@/ was added into /@bookmark@/.
-- 
-- If no bookmark for /@uri@/ is found then it is created.
-- 
-- /Since: 2.12/
bookmarkFileSetAdded ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> FCT.CTime
    -- ^ /@added@/: a timestamp or -1 to use the current time
    -> m ()
bookmarkFileSetAdded bookmark uri added = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    g_bookmark_file_set_added bookmark' uri' added
    touchManagedPtr bookmark
    freeMem uri'
    return ()

#if defined(ENABLE_OVERLOADING)
data BookmarkFileSetAddedMethodInfo
instance (signature ~ (T.Text -> FCT.CTime -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileSetAddedMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileSetAdded

instance O.OverloadedMethodInfo BookmarkFileSetAddedMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileSetAdded",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileSetAdded"
        })


#endif

-- method BookmarkFile::set_added_date_time
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "added"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_set_added_date_time" g_bookmark_file_set_added_date_time :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr GLib.DateTime.DateTime ->           -- added : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO ()

-- | Sets the time the bookmark for /@uri@/ was added into /@bookmark@/.
-- 
-- If no bookmark for /@uri@/ is found then it is created.
-- 
-- /Since: 2.66/
bookmarkFileSetAddedDateTime ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> GLib.DateTime.DateTime
    -- ^ /@added@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m ()
bookmarkFileSetAddedDateTime bookmark uri added = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    added' <- unsafeManagedPtrGetPtr added
    g_bookmark_file_set_added_date_time bookmark' uri' added'
    touchManagedPtr bookmark
    touchManagedPtr added
    freeMem uri'
    return ()

#if defined(ENABLE_OVERLOADING)
data BookmarkFileSetAddedDateTimeMethodInfo
instance (signature ~ (T.Text -> GLib.DateTime.DateTime -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileSetAddedDateTimeMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileSetAddedDateTime

instance O.OverloadedMethodInfo BookmarkFileSetAddedDateTimeMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileSetAddedDateTime",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileSetAddedDateTime"
        })


#endif

-- method BookmarkFile::set_app_info
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an application's name"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "exec"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an application's command line"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "count"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the number of registrations done for this application"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "stamp"
--           , argType = TBasicType Ttime_t
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the time of the last registration for this application"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_set_app_info" g_bookmark_file_set_app_info :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    CString ->                              -- name : TBasicType TUTF8
    CString ->                              -- exec : TBasicType TUTF8
    Int32 ->                                -- count : TBasicType TInt
    FCT.CTime ->                            -- stamp : TBasicType Ttime_t
    Ptr (Ptr GError) ->                     -- error
    IO CInt

{-# DEPRECATED bookmarkFileSetAppInfo ["(Since version 2.66)","Use 'GI.GLib.Structs.BookmarkFile.bookmarkFileSetApplicationInfo' instead, as","   @time_t@ is deprecated due to the year 2038 problem."] #-}
-- | Sets the meta-data of application /@name@/ inside the list of
-- applications that have registered a bookmark for /@uri@/ inside
-- /@bookmark@/.
-- 
-- You should rarely use this function; use 'GI.GLib.Structs.BookmarkFile.bookmarkFileAddApplication'
-- and 'GI.GLib.Structs.BookmarkFile.bookmarkFileRemoveApplication' instead.
-- 
-- /@name@/ can be any UTF-8 encoded string used to identify an
-- application.
-- /@exec@/ can have one of these two modifiers: \"%f\", which will
-- be expanded as the local file name retrieved from the bookmark\'s
-- URI; \"%u\", which will be expanded as the bookmark\'s URI.
-- The expansion is done automatically when retrieving the stored
-- command line using the 'GI.GLib.Structs.BookmarkFile.bookmarkFileGetApplicationInfo' function.
-- /@count@/ is the number of times the application has registered the
-- bookmark; if is \< 0, the current registration count will be increased
-- by one, if is 0, the application with /@name@/ will be removed from
-- the list of registered applications.
-- /@stamp@/ is the Unix time of the last registration; if it is -1, the
-- current time will be used.
-- 
-- If you try to remove an application by setting its registration count to
-- zero, and no bookmark for /@uri@/ is found, 'P.False' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'; similarly,
-- in the event that no application /@name@/ has registered a bookmark
-- for /@uri@/,  'P.False' is returned and error is set to
-- 'GI.GLib.Enums.BookmarkFileErrorAppNotRegistered'.  Otherwise, if no bookmark
-- for /@uri@/ is found, one is created.
-- 
-- /Since: 2.12/
bookmarkFileSetAppInfo ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> T.Text
    -- ^ /@name@/: an application\'s name
    -> T.Text
    -- ^ /@exec@/: an application\'s command line
    -> Int32
    -- ^ /@count@/: the number of registrations done for this application
    -> FCT.CTime
    -- ^ /@stamp@/: the time of the last registration for this application
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileSetAppInfo bookmark uri name exec count stamp = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    name' <- textToCString name
    exec' <- textToCString exec
    onException (do
        _ <- propagateGError $ g_bookmark_file_set_app_info bookmark' uri' name' exec' count stamp
        touchManagedPtr bookmark
        freeMem uri'
        freeMem name'
        freeMem exec'
        return ()
     ) (do
        freeMem uri'
        freeMem name'
        freeMem exec'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileSetAppInfoMethodInfo
instance (signature ~ (T.Text -> T.Text -> T.Text -> Int32 -> FCT.CTime -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileSetAppInfoMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileSetAppInfo

instance O.OverloadedMethodInfo BookmarkFileSetAppInfoMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileSetAppInfo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileSetAppInfo"
        })


#endif

-- method BookmarkFile::set_application_info
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an application's name"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "exec"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an application's command line"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "count"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the number of registrations done for this application"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "stamp"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the time of the last registration for this application,\n   which may be %NULL if @count is 0"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_set_application_info" g_bookmark_file_set_application_info :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    CString ->                              -- name : TBasicType TUTF8
    CString ->                              -- exec : TBasicType TUTF8
    Int32 ->                                -- count : TBasicType TInt
    Ptr GLib.DateTime.DateTime ->           -- stamp : TInterface (Name {namespace = "GLib", name = "DateTime"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Sets the meta-data of application /@name@/ inside the list of
-- applications that have registered a bookmark for /@uri@/ inside
-- /@bookmark@/.
-- 
-- You should rarely use this function; use 'GI.GLib.Structs.BookmarkFile.bookmarkFileAddApplication'
-- and 'GI.GLib.Structs.BookmarkFile.bookmarkFileRemoveApplication' instead.
-- 
-- /@name@/ can be any UTF-8 encoded string used to identify an
-- application.
-- /@exec@/ can have one of these two modifiers: \"%f\", which will
-- be expanded as the local file name retrieved from the bookmark\'s
-- URI; \"%u\", which will be expanded as the bookmark\'s URI.
-- The expansion is done automatically when retrieving the stored
-- command line using the 'GI.GLib.Structs.BookmarkFile.bookmarkFileGetApplicationInfo' function.
-- /@count@/ is the number of times the application has registered the
-- bookmark; if is \< 0, the current registration count will be increased
-- by one, if is 0, the application with /@name@/ will be removed from
-- the list of registered applications.
-- /@stamp@/ is the Unix time of the last registration.
-- 
-- If you try to remove an application by setting its registration count to
-- zero, and no bookmark for /@uri@/ is found, 'P.False' is returned and
-- /@error@/ is set to 'GI.GLib.Enums.BookmarkFileErrorUriNotFound'; similarly,
-- in the event that no application /@name@/ has registered a bookmark
-- for /@uri@/,  'P.False' is returned and error is set to
-- 'GI.GLib.Enums.BookmarkFileErrorAppNotRegistered'.  Otherwise, if no bookmark
-- for /@uri@/ is found, one is created.
-- 
-- /Since: 2.66/
bookmarkFileSetApplicationInfo ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> T.Text
    -- ^ /@name@/: an application\'s name
    -> T.Text
    -- ^ /@exec@/: an application\'s command line
    -> Int32
    -- ^ /@count@/: the number of registrations done for this application
    -> Maybe (GLib.DateTime.DateTime)
    -- ^ /@stamp@/: the time of the last registration for this application,
    --    which may be 'P.Nothing' if /@count@/ is 0
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileSetApplicationInfo bookmark uri name exec count stamp = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    name' <- textToCString name
    exec' <- textToCString exec
    maybeStamp <- case stamp of
        Nothing -> return FP.nullPtr
        Just jStamp -> do
            jStamp' <- unsafeManagedPtrGetPtr jStamp
            return jStamp'
    onException (do
        _ <- propagateGError $ g_bookmark_file_set_application_info bookmark' uri' name' exec' count maybeStamp
        touchManagedPtr bookmark
        whenJust stamp touchManagedPtr
        freeMem uri'
        freeMem name'
        freeMem exec'
        return ()
     ) (do
        freeMem uri'
        freeMem name'
        freeMem exec'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileSetApplicationInfoMethodInfo
instance (signature ~ (T.Text -> T.Text -> T.Text -> Int32 -> Maybe (GLib.DateTime.DateTime) -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileSetApplicationInfoMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileSetApplicationInfo

instance O.OverloadedMethodInfo BookmarkFileSetApplicationInfoMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileSetApplicationInfo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileSetApplicationInfo"
        })


#endif

-- method BookmarkFile::set_description
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "description"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a string" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_set_description" g_bookmark_file_set_description :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    CString ->                              -- description : TBasicType TUTF8
    IO ()

-- | Sets /@description@/ as the description of the bookmark for /@uri@/.
-- 
-- If /@uri@/ is 'P.Nothing', the description of /@bookmark@/ is set.
-- 
-- If a bookmark for /@uri@/ cannot be found then it is created.
-- 
-- /Since: 2.12/
bookmarkFileSetDescription ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> Maybe (T.Text)
    -- ^ /@uri@/: a valid URI or 'P.Nothing'
    -> T.Text
    -- ^ /@description@/: a string
    -> m ()
bookmarkFileSetDescription bookmark uri description = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    maybeUri <- case uri of
        Nothing -> return FP.nullPtr
        Just jUri -> do
            jUri' <- textToCString jUri
            return jUri'
    description' <- textToCString description
    g_bookmark_file_set_description bookmark' maybeUri description'
    touchManagedPtr bookmark
    freeMem maybeUri
    freeMem description'
    return ()

#if defined(ENABLE_OVERLOADING)
data BookmarkFileSetDescriptionMethodInfo
instance (signature ~ (Maybe (T.Text) -> T.Text -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileSetDescriptionMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileSetDescription

instance O.OverloadedMethodInfo BookmarkFileSetDescriptionMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileSetDescription",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileSetDescription"
        })


#endif

-- method BookmarkFile::set_groups
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an item's URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "groups"
--           , argType = TCArray False (-1) 3 (TBasicType TUTF8)
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "an array of\n   group names, or %NULL to remove all groups"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "number of group name values in @groups"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "number of group name values in @groups"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_set_groups" g_bookmark_file_set_groups :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr CString ->                          -- groups : TCArray False (-1) 3 (TBasicType TUTF8)
    FCT.CSize ->                            -- length : TBasicType TSize
    IO ()

-- | Sets a list of group names for the item with URI /@uri@/.  Each previously
-- set group name list is removed.
-- 
-- If /@uri@/ cannot be found then an item for it is created.
-- 
-- /Since: 2.12/
bookmarkFileSetGroups ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: an item\'s URI
    -> Maybe ([T.Text])
    -- ^ /@groups@/: an array of
    --    group names, or 'P.Nothing' to remove all groups
    -> m ()
bookmarkFileSetGroups bookmark uri groups = liftIO $ do
    let length_ = case groups of
            Nothing -> 0
            Just jGroups -> fromIntegral $ P.length jGroups
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    maybeGroups <- case groups of
        Nothing -> return FP.nullPtr
        Just jGroups -> do
            jGroups' <- packUTF8CArray jGroups
            return jGroups'
    g_bookmark_file_set_groups bookmark' uri' maybeGroups length_
    touchManagedPtr bookmark
    freeMem uri'
    (mapCArrayWithLength length_) freeMem maybeGroups
    freeMem maybeGroups
    return ()

#if defined(ENABLE_OVERLOADING)
data BookmarkFileSetGroupsMethodInfo
instance (signature ~ (T.Text -> Maybe ([T.Text]) -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileSetGroupsMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileSetGroups

instance O.OverloadedMethodInfo BookmarkFileSetGroupsMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileSetGroups",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileSetGroups"
        })


#endif

-- method BookmarkFile::set_icon
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "href"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the URI of the icon for the bookmark, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mime_type"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the MIME type of the icon for the bookmark"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_set_icon" g_bookmark_file_set_icon :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    CString ->                              -- href : TBasicType TUTF8
    CString ->                              -- mime_type : TBasicType TUTF8
    IO ()

-- | Sets the icon for the bookmark for /@uri@/. If /@href@/ is 'P.Nothing', unsets
-- the currently set icon. /@href@/ can either be a full URL for the icon
-- file or the icon name following the Icon Naming specification.
-- 
-- If no bookmark for /@uri@/ is found one is created.
-- 
-- /Since: 2.12/
bookmarkFileSetIcon ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> Maybe (T.Text)
    -- ^ /@href@/: the URI of the icon for the bookmark, or 'P.Nothing'
    -> T.Text
    -- ^ /@mimeType@/: the MIME type of the icon for the bookmark
    -> m ()
bookmarkFileSetIcon bookmark uri href mimeType = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    maybeHref <- case href of
        Nothing -> return FP.nullPtr
        Just jHref -> do
            jHref' <- textToCString jHref
            return jHref'
    mimeType' <- textToCString mimeType
    g_bookmark_file_set_icon bookmark' uri' maybeHref mimeType'
    touchManagedPtr bookmark
    freeMem uri'
    freeMem maybeHref
    freeMem mimeType'
    return ()

#if defined(ENABLE_OVERLOADING)
data BookmarkFileSetIconMethodInfo
instance (signature ~ (T.Text -> Maybe (T.Text) -> T.Text -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileSetIconMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileSetIcon

instance O.OverloadedMethodInfo BookmarkFileSetIconMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileSetIcon",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileSetIcon"
        })


#endif

-- method BookmarkFile::set_is_private
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "is_private"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "%TRUE if the bookmark should be marked as private"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_set_is_private" g_bookmark_file_set_is_private :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    CInt ->                                 -- is_private : TBasicType TBoolean
    IO ()

-- | Sets the private flag of the bookmark for /@uri@/.
-- 
-- If a bookmark for /@uri@/ cannot be found then it is created.
-- 
-- /Since: 2.12/
bookmarkFileSetIsPrivate ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> Bool
    -- ^ /@isPrivate@/: 'P.True' if the bookmark should be marked as private
    -> m ()
bookmarkFileSetIsPrivate bookmark uri isPrivate = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    let isPrivate' = (P.fromIntegral . P.fromEnum) isPrivate
    g_bookmark_file_set_is_private bookmark' uri' isPrivate'
    touchManagedPtr bookmark
    freeMem uri'
    return ()

#if defined(ENABLE_OVERLOADING)
data BookmarkFileSetIsPrivateMethodInfo
instance (signature ~ (T.Text -> Bool -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileSetIsPrivateMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileSetIsPrivate

instance O.OverloadedMethodInfo BookmarkFileSetIsPrivateMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileSetIsPrivate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileSetIsPrivate"
        })


#endif

-- method BookmarkFile::set_mime_type
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "mime_type"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a MIME type" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_set_mime_type" g_bookmark_file_set_mime_type :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    CString ->                              -- mime_type : TBasicType TUTF8
    IO ()

-- | Sets /@mimeType@/ as the MIME type of the bookmark for /@uri@/.
-- 
-- If a bookmark for /@uri@/ cannot be found then it is created.
-- 
-- /Since: 2.12/
bookmarkFileSetMimeType ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> T.Text
    -- ^ /@mimeType@/: a MIME type
    -> m ()
bookmarkFileSetMimeType bookmark uri mimeType = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    mimeType' <- textToCString mimeType
    g_bookmark_file_set_mime_type bookmark' uri' mimeType'
    touchManagedPtr bookmark
    freeMem uri'
    freeMem mimeType'
    return ()

#if defined(ENABLE_OVERLOADING)
data BookmarkFileSetMimeTypeMethodInfo
instance (signature ~ (T.Text -> T.Text -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileSetMimeTypeMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileSetMimeType

instance O.OverloadedMethodInfo BookmarkFileSetMimeTypeMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileSetMimeType",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileSetMimeType"
        })


#endif

-- method BookmarkFile::set_modified
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "modified"
--           , argType = TBasicType Ttime_t
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a timestamp or -1 to use the current time"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_set_modified" g_bookmark_file_set_modified :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    FCT.CTime ->                            -- modified : TBasicType Ttime_t
    IO ()

{-# DEPRECATED bookmarkFileSetModified ["(Since version 2.66)","Use 'GI.GLib.Structs.BookmarkFile.bookmarkFileSetModifiedDateTime' instead, as","   @time_t@ is deprecated due to the year 2038 problem."] #-}
-- | Sets the last time the bookmark for /@uri@/ was last modified.
-- 
-- If no bookmark for /@uri@/ is found then it is created.
-- 
-- The \"modified\" time should only be set when the bookmark\'s meta-data
-- was actually changed.  Every function of t'GI.GLib.Structs.BookmarkFile.BookmarkFile' that
-- modifies a bookmark also changes the modification time, except for
-- 'GI.GLib.Structs.BookmarkFile.bookmarkFileSetVisitedDateTime'.
-- 
-- /Since: 2.12/
bookmarkFileSetModified ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> FCT.CTime
    -- ^ /@modified@/: a timestamp or -1 to use the current time
    -> m ()
bookmarkFileSetModified bookmark uri modified = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    g_bookmark_file_set_modified bookmark' uri' modified
    touchManagedPtr bookmark
    freeMem uri'
    return ()

#if defined(ENABLE_OVERLOADING)
data BookmarkFileSetModifiedMethodInfo
instance (signature ~ (T.Text -> FCT.CTime -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileSetModifiedMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileSetModified

instance O.OverloadedMethodInfo BookmarkFileSetModifiedMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileSetModified",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileSetModified"
        })


#endif

-- method BookmarkFile::set_modified_date_time
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "modified"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_set_modified_date_time" g_bookmark_file_set_modified_date_time :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr GLib.DateTime.DateTime ->           -- modified : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO ()

-- | Sets the last time the bookmark for /@uri@/ was last modified.
-- 
-- If no bookmark for /@uri@/ is found then it is created.
-- 
-- The \"modified\" time should only be set when the bookmark\'s meta-data
-- was actually changed.  Every function of t'GI.GLib.Structs.BookmarkFile.BookmarkFile' that
-- modifies a bookmark also changes the modification time, except for
-- 'GI.GLib.Structs.BookmarkFile.bookmarkFileSetVisitedDateTime'.
-- 
-- /Since: 2.66/
bookmarkFileSetModifiedDateTime ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> GLib.DateTime.DateTime
    -- ^ /@modified@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m ()
bookmarkFileSetModifiedDateTime bookmark uri modified = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    modified' <- unsafeManagedPtrGetPtr modified
    g_bookmark_file_set_modified_date_time bookmark' uri' modified'
    touchManagedPtr bookmark
    touchManagedPtr modified
    freeMem uri'
    return ()

#if defined(ENABLE_OVERLOADING)
data BookmarkFileSetModifiedDateTimeMethodInfo
instance (signature ~ (T.Text -> GLib.DateTime.DateTime -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileSetModifiedDateTimeMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileSetModifiedDateTime

instance O.OverloadedMethodInfo BookmarkFileSetModifiedDateTimeMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileSetModifiedDateTime",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileSetModifiedDateTime"
        })


#endif

-- method BookmarkFile::set_title
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "title"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a UTF-8 encoded string"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_set_title" g_bookmark_file_set_title :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    CString ->                              -- title : TBasicType TUTF8
    IO ()

-- | Sets /@title@/ as the title of the bookmark for /@uri@/ inside the
-- bookmark file /@bookmark@/.
-- 
-- If /@uri@/ is 'P.Nothing', the title of /@bookmark@/ is set.
-- 
-- If a bookmark for /@uri@/ cannot be found then it is created.
-- 
-- /Since: 2.12/
bookmarkFileSetTitle ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> Maybe (T.Text)
    -- ^ /@uri@/: a valid URI or 'P.Nothing'
    -> T.Text
    -- ^ /@title@/: a UTF-8 encoded string
    -> m ()
bookmarkFileSetTitle bookmark uri title = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    maybeUri <- case uri of
        Nothing -> return FP.nullPtr
        Just jUri -> do
            jUri' <- textToCString jUri
            return jUri'
    title' <- textToCString title
    g_bookmark_file_set_title bookmark' maybeUri title'
    touchManagedPtr bookmark
    freeMem maybeUri
    freeMem title'
    return ()

#if defined(ENABLE_OVERLOADING)
data BookmarkFileSetTitleMethodInfo
instance (signature ~ (Maybe (T.Text) -> T.Text -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileSetTitleMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileSetTitle

instance O.OverloadedMethodInfo BookmarkFileSetTitleMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileSetTitle",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileSetTitle"
        })


#endif

-- method BookmarkFile::set_visited
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "visited"
--           , argType = TBasicType Ttime_t
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a timestamp or -1 to use the current time"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_set_visited" g_bookmark_file_set_visited :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    FCT.CTime ->                            -- visited : TBasicType Ttime_t
    IO ()

{-# DEPRECATED bookmarkFileSetVisited ["(Since version 2.66)","Use 'GI.GLib.Structs.BookmarkFile.bookmarkFileSetVisitedDateTime' instead, as","   @time_t@ is deprecated due to the year 2038 problem."] #-}
-- | Sets the time the bookmark for /@uri@/ was last visited.
-- 
-- If no bookmark for /@uri@/ is found then it is created.
-- 
-- The \"visited\" time should only be set if the bookmark was launched,
-- either using the command line retrieved by 'GI.GLib.Structs.BookmarkFile.bookmarkFileGetApplicationInfo'
-- or by the default application for the bookmark\'s MIME type, retrieved
-- using 'GI.GLib.Structs.BookmarkFile.bookmarkFileGetMimeType'.  Changing the \"visited\" time
-- does not affect the \"modified\" time.
-- 
-- /Since: 2.12/
bookmarkFileSetVisited ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> FCT.CTime
    -- ^ /@visited@/: a timestamp or -1 to use the current time
    -> m ()
bookmarkFileSetVisited bookmark uri visited = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    g_bookmark_file_set_visited bookmark' uri' visited
    touchManagedPtr bookmark
    freeMem uri'
    return ()

#if defined(ENABLE_OVERLOADING)
data BookmarkFileSetVisitedMethodInfo
instance (signature ~ (T.Text -> FCT.CTime -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileSetVisitedMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileSetVisited

instance O.OverloadedMethodInfo BookmarkFileSetVisitedMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileSetVisited",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileSetVisited"
        })


#endif

-- method BookmarkFile::set_visited_date_time
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "uri"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a valid URI" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "visited"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DateTime" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GDateTime" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_set_visited_date_time" g_bookmark_file_set_visited_date_time :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- uri : TBasicType TUTF8
    Ptr GLib.DateTime.DateTime ->           -- visited : TInterface (Name {namespace = "GLib", name = "DateTime"})
    IO ()

-- | Sets the time the bookmark for /@uri@/ was last visited.
-- 
-- If no bookmark for /@uri@/ is found then it is created.
-- 
-- The \"visited\" time should only be set if the bookmark was launched,
-- either using the command line retrieved by 'GI.GLib.Structs.BookmarkFile.bookmarkFileGetApplicationInfo'
-- or by the default application for the bookmark\'s MIME type, retrieved
-- using 'GI.GLib.Structs.BookmarkFile.bookmarkFileGetMimeType'.  Changing the \"visited\" time
-- does not affect the \"modified\" time.
-- 
-- /Since: 2.66/
bookmarkFileSetVisitedDateTime ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> T.Text
    -- ^ /@uri@/: a valid URI
    -> GLib.DateTime.DateTime
    -- ^ /@visited@/: a t'GI.GLib.Structs.DateTime.DateTime'
    -> m ()
bookmarkFileSetVisitedDateTime bookmark uri visited = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    uri' <- textToCString uri
    visited' <- unsafeManagedPtrGetPtr visited
    g_bookmark_file_set_visited_date_time bookmark' uri' visited'
    touchManagedPtr bookmark
    touchManagedPtr visited
    freeMem uri'
    return ()

#if defined(ENABLE_OVERLOADING)
data BookmarkFileSetVisitedDateTimeMethodInfo
instance (signature ~ (T.Text -> GLib.DateTime.DateTime -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileSetVisitedDateTimeMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileSetVisitedDateTime

instance O.OverloadedMethodInfo BookmarkFileSetVisitedDateTimeMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileSetVisitedDateTime",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileSetVisitedDateTime"
        })


#endif

-- method BookmarkFile::to_data
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TSize
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "return location for the length of the returned string, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "length"
--              , argType = TBasicType TSize
--              , direction = DirectionOut
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText =
--                        Just
--                          "return location for the length of the returned string, or %NULL"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferEverything
--              }
--          ]
-- returnType: Just (TCArray False (-1) 1 (TBasicType TUInt8))
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_to_data" g_bookmark_file_to_data :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    Ptr FCT.CSize ->                        -- length : TBasicType TSize
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr Word8)

-- | This function outputs /@bookmark@/ as a string.
-- 
-- /Since: 2.12/
bookmarkFileToData ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> m ByteString
    -- ^ __Returns:__ 
    --   a newly allocated string holding the contents of the t'GI.GLib.Structs.BookmarkFile.BookmarkFile' /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileToData bookmark = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    length_ <- allocMem :: IO (Ptr FCT.CSize)
    onException (do
        result <- propagateGError $ g_bookmark_file_to_data bookmark' length_
        length_' <- peek length_
        checkUnexpectedReturnNULL "bookmarkFileToData" result
        result' <- (unpackByteStringWithLength length_') result
        freeMem result
        touchManagedPtr bookmark
        freeMem length_
        return result'
     ) (do
        freeMem length_
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileToDataMethodInfo
instance (signature ~ (m ByteString), MonadIO m) => O.OverloadedMethod BookmarkFileToDataMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileToData

instance O.OverloadedMethodInfo BookmarkFileToDataMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileToData",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileToData"
        })


#endif

-- method BookmarkFile::to_file
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "bookmark"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "BookmarkFile" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GBookmarkFile" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "filename"
--           , argType = TBasicType TFileName
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "path of the output file"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_bookmark_file_to_file" g_bookmark_file_to_file :: 
    Ptr BookmarkFile ->                     -- bookmark : TInterface (Name {namespace = "GLib", name = "BookmarkFile"})
    CString ->                              -- filename : TBasicType TFileName
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | This function outputs /@bookmark@/ into a file.  The write process is
-- guaranteed to be atomic by using 'GI.GLib.Functions.fileSetContents' internally.
-- 
-- /Since: 2.12/
bookmarkFileToFile ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    BookmarkFile
    -- ^ /@bookmark@/: a t'GI.GLib.Structs.BookmarkFile.BookmarkFile'
    -> [Char]
    -- ^ /@filename@/: path of the output file
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
bookmarkFileToFile bookmark filename = liftIO $ do
    bookmark' <- unsafeManagedPtrGetPtr bookmark
    filename' <- stringToCString filename
    onException (do
        _ <- propagateGError $ g_bookmark_file_to_file bookmark' filename'
        touchManagedPtr bookmark
        freeMem filename'
        return ()
     ) (do
        freeMem filename'
     )

#if defined(ENABLE_OVERLOADING)
data BookmarkFileToFileMethodInfo
instance (signature ~ ([Char] -> m ()), MonadIO m) => O.OverloadedMethod BookmarkFileToFileMethodInfo BookmarkFile signature where
    overloadedMethod = bookmarkFileToFile

instance O.OverloadedMethodInfo BookmarkFileToFileMethodInfo BookmarkFile where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.BookmarkFile.bookmarkFileToFile",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-BookmarkFile.html#v:bookmarkFileToFile"
        })


#endif

-- method BookmarkFile::error_quark
-- method type : MemberFunction
-- Args: []
-- Lengths: []
-- returnType: Just (TBasicType TUInt32)
-- throws : False
-- Skip return : False

foreign import ccall "g_bookmark_file_error_quark" g_bookmark_file_error_quark :: 
    IO Word32

-- | /No description available in the introspection data./
bookmarkFileErrorQuark ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Word32
bookmarkFileErrorQuark  = liftIO $ do
    result <- g_bookmark_file_error_quark
    return result

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveBookmarkFileMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveBookmarkFileMethod "addApplication" o = BookmarkFileAddApplicationMethodInfo
    ResolveBookmarkFileMethod "addGroup" o = BookmarkFileAddGroupMethodInfo
    ResolveBookmarkFileMethod "copy" o = BookmarkFileCopyMethodInfo
    ResolveBookmarkFileMethod "free" o = BookmarkFileFreeMethodInfo
    ResolveBookmarkFileMethod "hasApplication" o = BookmarkFileHasApplicationMethodInfo
    ResolveBookmarkFileMethod "hasGroup" o = BookmarkFileHasGroupMethodInfo
    ResolveBookmarkFileMethod "hasItem" o = BookmarkFileHasItemMethodInfo
    ResolveBookmarkFileMethod "loadFromData" o = BookmarkFileLoadFromDataMethodInfo
    ResolveBookmarkFileMethod "loadFromDataDirs" o = BookmarkFileLoadFromDataDirsMethodInfo
    ResolveBookmarkFileMethod "loadFromFile" o = BookmarkFileLoadFromFileMethodInfo
    ResolveBookmarkFileMethod "moveItem" o = BookmarkFileMoveItemMethodInfo
    ResolveBookmarkFileMethod "removeApplication" o = BookmarkFileRemoveApplicationMethodInfo
    ResolveBookmarkFileMethod "removeGroup" o = BookmarkFileRemoveGroupMethodInfo
    ResolveBookmarkFileMethod "removeItem" o = BookmarkFileRemoveItemMethodInfo
    ResolveBookmarkFileMethod "toData" o = BookmarkFileToDataMethodInfo
    ResolveBookmarkFileMethod "toFile" o = BookmarkFileToFileMethodInfo
    ResolveBookmarkFileMethod "getAdded" o = BookmarkFileGetAddedMethodInfo
    ResolveBookmarkFileMethod "getAddedDateTime" o = BookmarkFileGetAddedDateTimeMethodInfo
    ResolveBookmarkFileMethod "getAppInfo" o = BookmarkFileGetAppInfoMethodInfo
    ResolveBookmarkFileMethod "getApplicationInfo" o = BookmarkFileGetApplicationInfoMethodInfo
    ResolveBookmarkFileMethod "getApplications" o = BookmarkFileGetApplicationsMethodInfo
    ResolveBookmarkFileMethod "getDescription" o = BookmarkFileGetDescriptionMethodInfo
    ResolveBookmarkFileMethod "getGroups" o = BookmarkFileGetGroupsMethodInfo
    ResolveBookmarkFileMethod "getIcon" o = BookmarkFileGetIconMethodInfo
    ResolveBookmarkFileMethod "getIsPrivate" o = BookmarkFileGetIsPrivateMethodInfo
    ResolveBookmarkFileMethod "getMimeType" o = BookmarkFileGetMimeTypeMethodInfo
    ResolveBookmarkFileMethod "getModified" o = BookmarkFileGetModifiedMethodInfo
    ResolveBookmarkFileMethod "getModifiedDateTime" o = BookmarkFileGetModifiedDateTimeMethodInfo
    ResolveBookmarkFileMethod "getSize" o = BookmarkFileGetSizeMethodInfo
    ResolveBookmarkFileMethod "getTitle" o = BookmarkFileGetTitleMethodInfo
    ResolveBookmarkFileMethod "getUris" o = BookmarkFileGetUrisMethodInfo
    ResolveBookmarkFileMethod "getVisited" o = BookmarkFileGetVisitedMethodInfo
    ResolveBookmarkFileMethod "getVisitedDateTime" o = BookmarkFileGetVisitedDateTimeMethodInfo
    ResolveBookmarkFileMethod "setAdded" o = BookmarkFileSetAddedMethodInfo
    ResolveBookmarkFileMethod "setAddedDateTime" o = BookmarkFileSetAddedDateTimeMethodInfo
    ResolveBookmarkFileMethod "setAppInfo" o = BookmarkFileSetAppInfoMethodInfo
    ResolveBookmarkFileMethod "setApplicationInfo" o = BookmarkFileSetApplicationInfoMethodInfo
    ResolveBookmarkFileMethod "setDescription" o = BookmarkFileSetDescriptionMethodInfo
    ResolveBookmarkFileMethod "setGroups" o = BookmarkFileSetGroupsMethodInfo
    ResolveBookmarkFileMethod "setIcon" o = BookmarkFileSetIconMethodInfo
    ResolveBookmarkFileMethod "setIsPrivate" o = BookmarkFileSetIsPrivateMethodInfo
    ResolveBookmarkFileMethod "setMimeType" o = BookmarkFileSetMimeTypeMethodInfo
    ResolveBookmarkFileMethod "setModified" o = BookmarkFileSetModifiedMethodInfo
    ResolveBookmarkFileMethod "setModifiedDateTime" o = BookmarkFileSetModifiedDateTimeMethodInfo
    ResolveBookmarkFileMethod "setTitle" o = BookmarkFileSetTitleMethodInfo
    ResolveBookmarkFileMethod "setVisited" o = BookmarkFileSetVisitedMethodInfo
    ResolveBookmarkFileMethod "setVisitedDateTime" o = BookmarkFileSetVisitedDateTimeMethodInfo
    ResolveBookmarkFileMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveBookmarkFileMethod t BookmarkFile, O.OverloadedMethod info BookmarkFile p) => OL.IsLabel t (BookmarkFile -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveBookmarkFileMethod t BookmarkFile, O.OverloadedMethod info BookmarkFile p, R.HasField t BookmarkFile p) => R.HasField t BookmarkFile p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveBookmarkFileMethod t BookmarkFile, O.OverloadedMethodInfo info BookmarkFile) => OL.IsLabel t (O.MethodProxy info BookmarkFile) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


