{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- @GAsyncInitable@ is an interface for asynchronously initializable objects.
-- 
-- This is the asynchronous version of t'GI.Gio.Interfaces.Initable.Initable'; it behaves the same
-- in all ways except that initialization is asynchronous. For more details
-- see the descriptions on @GInitable@.
-- 
-- A class may implement both the @GInitable@ and @GAsyncInitable@ interfaces.
-- 
-- Users of objects implementing this are not intended to use the interface
-- method directly; instead it will be used automatically in various ways.
-- For C applications you generally just call t'GI.Gio.Interfaces.AsyncInitable.AsyncInitable'.@/new_async/@()
-- directly, or indirectly via a @/foo_thing_new_async()/@ wrapper. This will call
-- 'GI.Gio.Interfaces.AsyncInitable.asyncInitableInitAsync' under the covers, calling back with @NULL@
-- and a set @GError@ on failure.
-- 
-- A typical implementation might look something like this:
-- 
-- 
-- === /c code/
-- >enum {
-- >   NOT_INITIALIZED,
-- >   INITIALIZING,
-- >   INITIALIZED
-- >};
-- >
-- >static void
-- >_foo_ready_cb (Foo *self)
-- >{
-- >  GList *l;
-- >
-- >  self->priv->state = INITIALIZED;
-- >
-- >  for (l = self->priv->init_results; l != NULL; l = l->next)
-- >    {
-- >      GTask *task = l->data;
-- >
-- >      if (self->priv->success)
-- >        g_task_return_boolean (task, TRUE);
-- >      else
-- >        g_task_return_new_error (task, ...);
-- >      g_object_unref (task);
-- >    }
-- >
-- >  g_list_free (self->priv->init_results);
-- >  self->priv->init_results = NULL;
-- >}
-- >
-- >static void
-- >foo_init_async (GAsyncInitable       *initable,
-- >                int                   io_priority,
-- >                GCancellable         *cancellable,
-- >                GAsyncReadyCallback   callback,
-- >                gpointer              user_data)
-- >{
-- >  Foo *self = FOO (initable);
-- >  GTask *task;
-- >
-- >  task = g_task_new (initable, cancellable, callback, user_data);
-- >  g_task_set_name (task, G_STRFUNC);
-- >
-- >  switch (self->priv->state)
-- >    {
-- >      case NOT_INITIALIZED:
-- >        _foo_get_ready (self);
-- >        self->priv->init_results = g_list_append (self->priv->init_results,
-- >                                                  task);
-- >        self->priv->state = INITIALIZING;
-- >        break;
-- >      case INITIALIZING:
-- >        self->priv->init_results = g_list_append (self->priv->init_results,
-- >                                                  task);
-- >        break;
-- >      case INITIALIZED:
-- >        if (!self->priv->success)
-- >          g_task_return_new_error (task, ...);
-- >        else
-- >          g_task_return_boolean (task, TRUE);
-- >        g_object_unref (task);
-- >        break;
-- >    }
-- >}
-- >
-- >static gboolean
-- >foo_init_finish (GAsyncInitable       *initable,
-- >                 GAsyncResult         *result,
-- >                 GError              **error)
-- >{
-- >  g_return_val_if_fail (g_task_is_valid (result, initable), FALSE);
-- >
-- >  return g_task_propagate_boolean (G_TASK (result), error);
-- >}
-- >
-- >static void
-- >foo_async_initable_iface_init (gpointer g_iface,
-- >                               gpointer data)
-- >{
-- >  GAsyncInitableIface *iface = g_iface;
-- >
-- >  iface->init_async = foo_init_async;
-- >  iface->init_finish = foo_init_finish;
-- >}
-- 
-- 
-- /Since: 2.22/

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gio.Interfaces.AsyncInitable
    ( 

-- * Exported types
    AsyncInitable(..)                       ,
    IsAsyncInitable                         ,
    toAsyncInitable                         ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [initAsync]("GI.Gio.Interfaces.AsyncInitable#g:method:initAsync"), [initFinish]("GI.Gio.Interfaces.AsyncInitable#g:method:initFinish"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [newFinish]("GI.Gio.Interfaces.AsyncInitable#g:method:newFinish"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getData]("GI.GObject.Objects.Object#g:method:getData"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveAsyncInitableMethod              ,
#endif

-- ** initAsync #method:initAsync#

#if defined(ENABLE_OVERLOADING)
    AsyncInitableInitAsyncMethodInfo        ,
#endif
    asyncInitableInitAsync                  ,


-- ** initFinish #method:initFinish#

#if defined(ENABLE_OVERLOADING)
    AsyncInitableInitFinishMethodInfo       ,
#endif
    asyncInitableInitFinish                 ,


-- ** newFinish #method:newFinish#

#if defined(ENABLE_OVERLOADING)
    AsyncInitableNewFinishMethodInfo        ,
#endif
    asyncInitableNewFinish                  ,


-- ** newvAsync #method:newvAsync#

    asyncInitableNewvAsync                  ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.PollFD as GLib.PollFD
import qualified GI.GLib.Structs.Source as GLib.Source
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.GObject.Structs.Parameter as GObject.Parameter
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.GObject.Structs.Parameter as GObject.Parameter
import qualified GI.Gio.Callbacks as Gio.Callbacks
import {-# SOURCE #-} qualified GI.Gio.Interfaces.AsyncResult as Gio.AsyncResult
import {-# SOURCE #-} qualified GI.Gio.Objects.Cancellable as Gio.Cancellable

#endif

-- interface AsyncInitable 
-- | Memory-managed wrapper type.
newtype AsyncInitable = AsyncInitable (SP.ManagedPtr AsyncInitable)
    deriving (Eq)

instance SP.ManagedPtrNewtype AsyncInitable where
    toManagedPtr (AsyncInitable p) = p

foreign import ccall "g_async_initable_get_type"
    c_g_async_initable_get_type :: IO B.Types.GType

instance B.Types.TypedObject AsyncInitable where
    glibType = c_g_async_initable_get_type

instance B.Types.GObject AsyncInitable

-- | Type class for types which can be safely cast to t'AsyncInitable', for instance with `toAsyncInitable`.
class (SP.GObject o, O.IsDescendantOf AsyncInitable o) => IsAsyncInitable o
instance (SP.GObject o, O.IsDescendantOf AsyncInitable o) => IsAsyncInitable o

instance O.HasParentTypes AsyncInitable
type instance O.ParentTypes AsyncInitable = '[GObject.Object.Object]

-- | Cast to t'AsyncInitable', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toAsyncInitable :: (MIO.MonadIO m, IsAsyncInitable o) => o -> m AsyncInitable
toAsyncInitable = MIO.liftIO . B.ManagedPtr.unsafeCastTo AsyncInitable

-- | Convert t'AsyncInitable' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe AsyncInitable) where
    gvalueGType_ = c_g_async_initable_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr AsyncInitable)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr AsyncInitable)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject AsyncInitable ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList AsyncInitable
type instance O.AttributeList AsyncInitable = AsyncInitableAttributeList
type AsyncInitableAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveAsyncInitableMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveAsyncInitableMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveAsyncInitableMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveAsyncInitableMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveAsyncInitableMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveAsyncInitableMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveAsyncInitableMethod "initAsync" o = AsyncInitableInitAsyncMethodInfo
    ResolveAsyncInitableMethod "initFinish" o = AsyncInitableInitFinishMethodInfo
    ResolveAsyncInitableMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveAsyncInitableMethod "newFinish" o = AsyncInitableNewFinishMethodInfo
    ResolveAsyncInitableMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveAsyncInitableMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveAsyncInitableMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveAsyncInitableMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveAsyncInitableMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveAsyncInitableMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveAsyncInitableMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveAsyncInitableMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveAsyncInitableMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveAsyncInitableMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveAsyncInitableMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveAsyncInitableMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveAsyncInitableMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveAsyncInitableMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveAsyncInitableMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveAsyncInitableMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveAsyncInitableMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveAsyncInitableMethod t AsyncInitable, O.OverloadedMethod info AsyncInitable p) => OL.IsLabel t (AsyncInitable -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveAsyncInitableMethod t AsyncInitable, O.OverloadedMethod info AsyncInitable p, R.HasField t AsyncInitable p) => R.HasField t AsyncInitable p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveAsyncInitableMethod t AsyncInitable, O.OverloadedMethodInfo info AsyncInitable) => OL.IsLabel t (O.MethodProxy info AsyncInitable) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- method AsyncInitable::init_async
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "initable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncInitable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncInitable." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the operation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GAsyncReadyCallback to call when the request is satisfied"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 4
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_async_initable_init_async" g_async_initable_init_async :: 
    Ptr AsyncInitable ->                    -- initable : TInterface (Name {namespace = "Gio", name = "AsyncInitable"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

-- | Starts asynchronous initialization of the object implementing the
-- interface. This must be done before any real use of the object after
-- initial construction. If the object also implements t'GI.Gio.Interfaces.Initable.Initable' you can
-- optionally call 'GI.Gio.Interfaces.Initable.initableInit' instead.
-- 
-- This method is intended for language bindings. If writing in C,
-- @/g_async_initable_new_async()/@ should typically be used instead.
-- 
-- When the initialization is finished, /@callback@/ will be called. You can
-- then call 'GI.Gio.Interfaces.AsyncInitable.asyncInitableInitFinish' to get the result of the
-- initialization.
-- 
-- Implementations may also support cancellation. If /@cancellable@/ is not
-- 'P.Nothing', then initialization can be cancelled by triggering the cancellable
-- object from another thread. If the operation was cancelled, the error
-- 'GI.Gio.Enums.IOErrorEnumCancelled' will be returned. If /@cancellable@/ is not 'P.Nothing', and
-- the object doesn\'t support cancellable initialization, the error
-- 'GI.Gio.Enums.IOErrorEnumNotSupported' will be returned.
-- 
-- As with t'GI.Gio.Interfaces.Initable.Initable', if the object is not initialized, or initialization
-- returns with an error, then all operations on the object except
-- 'GI.GObject.Objects.Object.objectRef' and 'GI.GObject.Objects.Object.objectUnref' are considered to be invalid, and
-- have undefined behaviour. They will often fail with @/g_critical()/@ or
-- @/g_warning()/@, but this must not be relied on.
-- 
-- Callers should not assume that a class which implements t'GI.Gio.Interfaces.AsyncInitable.AsyncInitable' can
-- be initialized multiple times; for more information, see 'GI.Gio.Interfaces.Initable.initableInit'.
-- If a class explicitly supports being initialized multiple times,
-- implementation requires yielding all subsequent calls to @/init_async()/@ on the
-- results of the first call.
-- 
-- For classes that also support the t'GI.Gio.Interfaces.Initable.Initable' interface, the default
-- implementation of this method will run the 'GI.Gio.Interfaces.Initable.initableInit' function
-- in a thread, so if you want to support asynchronous initialization via
-- threads, just implement the t'GI.Gio.Interfaces.AsyncInitable.AsyncInitable' interface without overriding
-- any interface methods.
-- 
-- /Since: 2.22/
asyncInitableInitAsync ::
    (B.CallStack.HasCallStack, MonadIO m, IsAsyncInitable a, Gio.Cancellable.IsCancellable b) =>
    a
    -- ^ /@initable@/: a t'GI.Gio.Interfaces.AsyncInitable.AsyncInitable'.
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the operation
    -> Maybe (b)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore.
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call when the request is satisfied
    -> m ()
asyncInitableInitAsync initable ioPriority cancellable callback = liftIO $ do
    initable' <- unsafeManagedPtrCastPtr initable
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_async_initable_init_async initable' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr initable
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data AsyncInitableInitAsyncMethodInfo
instance (signature ~ (Int32 -> Maybe (b) -> Maybe (Gio.Callbacks.AsyncReadyCallback) -> m ()), MonadIO m, IsAsyncInitable a, Gio.Cancellable.IsCancellable b) => O.OverloadedMethod AsyncInitableInitAsyncMethodInfo a signature where
    overloadedMethod = asyncInitableInitAsync

instance O.OverloadedMethodInfo AsyncInitableInitAsyncMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.AsyncInitable.asyncInitableInitAsync",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-AsyncInitable.html#v:asyncInitableInitAsync"
        })


#endif

-- method AsyncInitable::init_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "initable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncInitable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncInitable." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GAsyncResult." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "g_async_initable_init_finish" g_async_initable_init_finish :: 
    Ptr AsyncInitable ->                    -- initable : TInterface (Name {namespace = "Gio", name = "AsyncInitable"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

-- | Finishes asynchronous initialization and returns the result.
-- See 'GI.Gio.Interfaces.AsyncInitable.asyncInitableInitAsync'.
-- 
-- /Since: 2.22/
asyncInitableInitFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsAsyncInitable a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@initable@/: a t'GI.Gio.Interfaces.AsyncInitable.AsyncInitable'.
    -> b
    -- ^ /@res@/: a t'GI.Gio.Interfaces.AsyncResult.AsyncResult'.
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
asyncInitableInitFinish initable res = liftIO $ do
    initable' <- unsafeManagedPtrCastPtr initable
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        _ <- propagateGError $ g_async_initable_init_finish initable' res'
        touchManagedPtr initable
        touchManagedPtr res
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data AsyncInitableInitFinishMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsAsyncInitable a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod AsyncInitableInitFinishMethodInfo a signature where
    overloadedMethod = asyncInitableInitFinish

instance O.OverloadedMethodInfo AsyncInitableInitFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.AsyncInitable.asyncInitableInitFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-AsyncInitable.html#v:asyncInitableInitFinish"
        })


#endif

-- method AsyncInitable::new_finish
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "initable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncInitable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GAsyncInitable from the callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "res"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncResult" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GAsyncResult from the callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GObject" , name = "Object" })
-- throws : True
-- Skip return : False

foreign import ccall "g_async_initable_new_finish" g_async_initable_new_finish :: 
    Ptr AsyncInitable ->                    -- initable : TInterface (Name {namespace = "Gio", name = "AsyncInitable"})
    Ptr Gio.AsyncResult.AsyncResult ->      -- res : TInterface (Name {namespace = "Gio", name = "AsyncResult"})
    Ptr (Ptr GError) ->                     -- error
    IO (Ptr GObject.Object.Object)

-- | Finishes the async construction for the various g_async_initable_new
-- calls, returning the created object or 'P.Nothing' on error.
-- 
-- /Since: 2.22/
asyncInitableNewFinish ::
    (B.CallStack.HasCallStack, MonadIO m, IsAsyncInitable a, Gio.AsyncResult.IsAsyncResult b) =>
    a
    -- ^ /@initable@/: the t'GI.Gio.Interfaces.AsyncInitable.AsyncInitable' from the callback
    -> b
    -- ^ /@res@/: the t'GI.Gio.Interfaces.AsyncResult.AsyncResult' from the callback
    -> m GObject.Object.Object
    -- ^ __Returns:__ a newly created t'GI.GObject.Objects.Object.Object',
    --      or 'P.Nothing' on error. Free with 'GI.GObject.Objects.Object.objectUnref'. /(Can throw 'Data.GI.Base.GError.GError')/
asyncInitableNewFinish initable res = liftIO $ do
    initable' <- unsafeManagedPtrCastPtr initable
    res' <- unsafeManagedPtrCastPtr res
    onException (do
        result <- propagateGError $ g_async_initable_new_finish initable' res'
        checkUnexpectedReturnNULL "asyncInitableNewFinish" result
        result' <- (wrapObject GObject.Object.Object) result
        touchManagedPtr initable
        touchManagedPtr res
        return result'
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data AsyncInitableNewFinishMethodInfo
instance (signature ~ (b -> m GObject.Object.Object), MonadIO m, IsAsyncInitable a, Gio.AsyncResult.IsAsyncResult b) => O.OverloadedMethod AsyncInitableNewFinishMethodInfo a signature where
    overloadedMethod = asyncInitableNewFinish

instance O.OverloadedMethodInfo AsyncInitableNewFinishMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gio.Interfaces.AsyncInitable.asyncInitableNewFinish",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gio-2.0.38/docs/GI-Gio-Interfaces-AsyncInitable.html#v:asyncInitableNewFinish"
        })


#endif

-- method AsyncInitable::newv_async
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "object_type"
--           , argType = TBasicType TGType
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GType supporting #GAsyncInitable."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_parameters"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of parameters in @parameters"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "parameters"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Parameter" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the parameters to use to construct the object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "io_priority"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the [I/O priority](iface.AsyncResult.html#io-priority) of the operation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cancellable"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "Cancellable" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "optional #GCancellable object, %NULL to ignore."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "AsyncReadyCallback" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GAsyncReadyCallback to call when the initialization is\n    finished"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = 6
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the data to pass to callback function"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_async_initable_newv_async" g_async_initable_newv_async :: 
    CGType ->                               -- object_type : TBasicType TGType
    Word32 ->                               -- n_parameters : TBasicType TUInt
    Ptr GObject.Parameter.Parameter ->      -- parameters : TInterface (Name {namespace = "GObject", name = "Parameter"})
    Int32 ->                                -- io_priority : TBasicType TInt
    Ptr Gio.Cancellable.Cancellable ->      -- cancellable : TInterface (Name {namespace = "Gio", name = "Cancellable"})
    FunPtr Gio.Callbacks.C_AsyncReadyCallback -> -- callback : TInterface (Name {namespace = "Gio", name = "AsyncReadyCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    IO ()

{-# DEPRECATED asyncInitableNewvAsync ["(Since version 2.54)","Use @/g_object_new_with_properties()/@ and","'GI.Gio.Interfaces.AsyncInitable.asyncInitableInitAsync' instead. See t'GI.GObject.Structs.Parameter.Parameter' for more information."] #-}
-- | Helper function for constructing t'GI.Gio.Interfaces.AsyncInitable.AsyncInitable' object. This is
-- similar to 'GI.GObject.Objects.Object.objectNewv' but also initializes the object asynchronously.
-- 
-- When the initialization is finished, /@callback@/ will be called. You can
-- then call 'GI.Gio.Interfaces.AsyncInitable.asyncInitableNewFinish' to get the new object and check
-- for any errors.
-- 
-- /Since: 2.22/
asyncInitableNewvAsync ::
    (B.CallStack.HasCallStack, MonadIO m, Gio.Cancellable.IsCancellable a) =>
    GType
    -- ^ /@objectType@/: a t'GType' supporting t'GI.Gio.Interfaces.AsyncInitable.AsyncInitable'.
    -> Word32
    -- ^ /@nParameters@/: the number of parameters in /@parameters@/
    -> GObject.Parameter.Parameter
    -- ^ /@parameters@/: the parameters to use to construct the object
    -> Int32
    -- ^ /@ioPriority@/: the <http://developer.gnome.org/gio/stable/iface.AsyncResult.html#io-priority I/O priority> of the operation
    -> Maybe (a)
    -- ^ /@cancellable@/: optional t'GI.Gio.Objects.Cancellable.Cancellable' object, 'P.Nothing' to ignore.
    -> Maybe (Gio.Callbacks.AsyncReadyCallback)
    -- ^ /@callback@/: a t'GI.Gio.Callbacks.AsyncReadyCallback' to call when the initialization is
    --     finished
    -> m ()
asyncInitableNewvAsync objectType nParameters parameters ioPriority cancellable callback = liftIO $ do
    let objectType' = gtypeToCGType objectType
    parameters' <- unsafeManagedPtrGetPtr parameters
    maybeCancellable <- case cancellable of
        Nothing -> return FP.nullPtr
        Just jCancellable -> do
            jCancellable' <- unsafeManagedPtrCastPtr jCancellable
            return jCancellable'
    maybeCallback <- case callback of
        Nothing -> return FP.nullFunPtr
        Just jCallback -> do
            ptrcallback <- callocMem :: IO (Ptr (FunPtr Gio.Callbacks.C_AsyncReadyCallback))
            jCallback' <- Gio.Callbacks.mk_AsyncReadyCallback (Gio.Callbacks.wrap_AsyncReadyCallback (Just ptrcallback) (Gio.Callbacks.drop_closures_AsyncReadyCallback jCallback))
            poke ptrcallback jCallback'
            return jCallback'
    let userData = nullPtr
    g_async_initable_newv_async objectType' nParameters parameters' ioPriority maybeCancellable maybeCallback userData
    touchManagedPtr parameters
    whenJust cancellable touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList AsyncInitable = AsyncInitableSignalList
type AsyncInitableSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif


