// Copyright The OpenTelemetry Authors
// SPDX-License-Identifier: Apache-2.0

//go:generate mdatagen metadata.yaml

package googlecloudpubsubexporter // import "github.com/open-telemetry/opentelemetry-collector-contrib/exporter/googlecloudpubsubexporter"

import (
	"context"
	"fmt"
	"strings"
	"time"

	"github.com/google/uuid"
	"go.opentelemetry.io/collector/component"
	"go.opentelemetry.io/collector/consumer"
	"go.opentelemetry.io/collector/exporter"
	"go.opentelemetry.io/collector/exporter/exporterhelper"
	"go.opentelemetry.io/collector/pdata/plog"
	"go.opentelemetry.io/collector/pdata/pmetric"
	"go.opentelemetry.io/collector/pdata/ptrace"

	"github.com/open-telemetry/opentelemetry-collector-contrib/exporter/googlecloudpubsubexporter/internal/metadata"
)

const (
	defaultTimeout = 12 * time.Second
)

// NewFactory creates a factory for Google Cloud Pub/Sub exporter.
func NewFactory() exporter.Factory {
	return exporter.NewFactory(
		metadata.Type,
		createDefaultConfig,
		exporter.WithTraces(createTracesExporter, metadata.TracesStability),
		exporter.WithMetrics(createMetricsExporter, metadata.MetricsStability),
		exporter.WithLogs(createLogsExporter, metadata.LogsStability))
}

var exporters = map[*Config]*pubsubExporter{}

func ensureExporter(params exporter.Settings, pCfg *Config) *pubsubExporter {
	exp := exporters[pCfg]
	if exp != nil {
		return exp
	}
	exp = &pubsubExporter{
		logger:            params.Logger,
		userAgent:         strings.ReplaceAll(pCfg.UserAgent, "{{version}}", params.BuildInfo.Version),
		ceSource:          fmt.Sprintf("/opentelemetry/collector/%s/%s", metadata.Type.String(), params.BuildInfo.Version),
		config:            pCfg,
		tracesMarshaler:   &ptrace.ProtoMarshaler{},
		tracesAttributes:  map[string]string{},
		metricsMarshaler:  &pmetric.ProtoMarshaler{},
		metricsAttributes: map[string]string{},
		logsMarshaler:     &plog.ProtoMarshaler{},
		logsAttributes:    map[string]string{},
		makeUUID:          uuid.NewRandom,
		makeClient:        newPublisherClient,
	}
	// we ignore the error here as the config is already validated with the same method
	exp.ceCompression, _ = pCfg.parseCompression()
	watermarkBehavior, _ := pCfg.Watermark.parseWatermarkBehavior()
	switch watermarkBehavior {
	case earliest:
		exp.tracesWatermarkFunc = earliestTracesWatermark
		exp.metricsWatermarkFunc = earliestMetricsWatermark
		exp.logsWatermarkFunc = earliestLogsWatermark
	case current:
		exp.tracesWatermarkFunc = currentTracesWatermark
		exp.metricsWatermarkFunc = currentMetricsWatermark
		exp.logsWatermarkFunc = currentLogsWatermark
	}
	exporters[pCfg] = exp
	return exp
}

// createDefaultConfig creates the default configuration for exporter.
func createDefaultConfig() component.Config {
	return &Config{
		UserAgent:       "opentelemetry-collector-contrib {{version}}",
		TimeoutSettings: exporterhelper.TimeoutConfig{Timeout: defaultTimeout},
		Watermark: WatermarkConfig{
			Behavior:     "current",
			AllowedDrift: 0,
		},
		Ordering: OrderingConfig{
			Enabled:                 false,
			FromResourceAttribute:   "",
			RemoveResourceAttribute: false,
		},
	}
}

func createTracesExporter(ctx context.Context, set exporter.Settings, cfg component.Config) (exporter.Traces, error) {
	pCfg := cfg.(*Config)
	exp := ensureExporter(set, pCfg)

	return exporterhelper.NewTraces(
		ctx,
		set,
		cfg,
		exp.consumeTraces,
		exporterhelper.WithCapabilities(consumer.Capabilities{MutatesData: true}),
		exporterhelper.WithTimeout(pCfg.TimeoutSettings),
		exporterhelper.WithRetry(pCfg.BackOffConfig),
		exporterhelper.WithQueue(pCfg.QueueSettings),
		exporterhelper.WithStart(exp.start),
		exporterhelper.WithShutdown(exp.shutdown),
	)
}

func createMetricsExporter(ctx context.Context, set exporter.Settings, cfg component.Config) (exporter.Metrics, error) {
	pCfg := cfg.(*Config)
	exp := ensureExporter(set, pCfg)
	return exporterhelper.NewMetrics(
		ctx,
		set,
		cfg,
		exp.consumeMetrics,
		exporterhelper.WithCapabilities(consumer.Capabilities{MutatesData: true}),
		exporterhelper.WithTimeout(pCfg.TimeoutSettings),
		exporterhelper.WithRetry(pCfg.BackOffConfig),
		exporterhelper.WithQueue(pCfg.QueueSettings),
		exporterhelper.WithStart(exp.start),
		exporterhelper.WithShutdown(exp.shutdown),
	)
}

func createLogsExporter(ctx context.Context, set exporter.Settings, cfg component.Config) (exporter.Logs, error) {
	pCfg := cfg.(*Config)
	exp := ensureExporter(set, pCfg)

	return exporterhelper.NewLogs(
		ctx,
		set,
		cfg,
		exp.consumeLogs,
		exporterhelper.WithCapabilities(consumer.Capabilities{MutatesData: true}),
		exporterhelper.WithTimeout(pCfg.TimeoutSettings),
		exporterhelper.WithRetry(pCfg.BackOffConfig),
		exporterhelper.WithQueue(pCfg.QueueSettings),
		exporterhelper.WithStart(exp.start),
		exporterhelper.WithShutdown(exp.shutdown),
	)
}
