/**************************************************************************\
 * Copyright (c) Kongsberg Oil & Gas Technologies AS
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met:
 * 
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * 
 * Redistributions in binary form must reproduce the above copyright
 * notice, this list of conditions and the following disclaimer in the
 * documentation and/or other materials provided with the distribution.
 * 
 * Neither the name of the copyright holder nor the names of its
 * contributors may be used to endorse or promote products derived from
 * this software without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 * A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 * HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
\**************************************************************************/

/*!
  \class SoMotion3Event SoMotion3Event.h Inventor/events/SoMotion3Event.h
  \brief The SoMotion3Event class contains information about 3D movement
  events.

  \ingroup coin_events

  Motion3 events are generated by devices capable of "3D motion", for
  instance a spaceball device. Instances of this class contain information
  about these devices' translations and rotations in all 3 dimensions.

  \sa SoEvent, SoLocation2Event
  \sa SoEventCallback, SoHandleEventAction */

#include <Inventor/events/SoMotion3Event.h>
#include <Inventor/SbName.h>
#include <cassert>


SO_EVENT_SOURCE(SoMotion3Event);

/*!
  Initialize the type information data.
 */
void
SoMotion3Event::initClass(void)
{
  SO_EVENT_INIT_CLASS(SoMotion3Event, SoEvent);
}

/*!
  Constructor. Resets values for translation and rotation information.
 */
SoMotion3Event::SoMotion3Event(void)
{
  this->translation.setValue(0.0f, 0.0f, 0.0f);
  this->rotation = SbRotation::identity();
}

/*!
  Destructor.
 */
SoMotion3Event::~SoMotion3Event()
{
}

/*!
  Sets the translation value of this event. Method is used by the
  window specific device class(es) generating the event.

  \sa getTranslation()
 */
void
SoMotion3Event::setTranslation(const SbVec3f & t)
{
  this->translation = t;
}

/*!
  Returns translation value. Note that translation are specified relative
  to former events, i.e. the values given here are \e not accumulated
  global coordinates.

  \sa getTime()
 */
const SbVec3f &
SoMotion3Event::getTranslation(void) const
{
  return this->translation;
}

/*!
  Sets the rotation value of this event. Method is used by the
  window specific device class(es) generating the event.
 */
void
SoMotion3Event::setRotation(const SbRotation & r)
{
  this->rotation = r;
}

/*!
  Returns rotational value. Note that rotations are specified relative
  to former events, i.e. the values given here are \e not accumulated
  rotations.

  \sa getTime()
 */
const SbRotation &
SoMotion3Event::getRotation(void) const
{
  return this->rotation;
}
