﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/UUID.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/finspace/FinspaceRequest.h>
#include <aws/finspace/Finspace_EXPORTS.h>
#include <aws/finspace/model/KxDatabaseConfiguration.h>
#include <aws/finspace/model/KxDeploymentConfiguration.h>

#include <utility>

namespace Aws {
namespace finspace {
namespace Model {

/**
 */
class UpdateKxClusterDatabasesRequest : public FinspaceRequest {
 public:
  AWS_FINSPACE_API UpdateKxClusterDatabasesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateKxClusterDatabases"; }

  AWS_FINSPACE_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The unique identifier of a kdb environment.</p>
   */
  inline const Aws::String& GetEnvironmentId() const { return m_environmentId; }
  inline bool EnvironmentIdHasBeenSet() const { return m_environmentIdHasBeenSet; }
  template <typename EnvironmentIdT = Aws::String>
  void SetEnvironmentId(EnvironmentIdT&& value) {
    m_environmentIdHasBeenSet = true;
    m_environmentId = std::forward<EnvironmentIdT>(value);
  }
  template <typename EnvironmentIdT = Aws::String>
  UpdateKxClusterDatabasesRequest& WithEnvironmentId(EnvironmentIdT&& value) {
    SetEnvironmentId(std::forward<EnvironmentIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A unique name for the cluster that you want to modify.</p>
   */
  inline const Aws::String& GetClusterName() const { return m_clusterName; }
  inline bool ClusterNameHasBeenSet() const { return m_clusterNameHasBeenSet; }
  template <typename ClusterNameT = Aws::String>
  void SetClusterName(ClusterNameT&& value) {
    m_clusterNameHasBeenSet = true;
    m_clusterName = std::forward<ClusterNameT>(value);
  }
  template <typename ClusterNameT = Aws::String>
  UpdateKxClusterDatabasesRequest& WithClusterName(ClusterNameT&& value) {
    SetClusterName(std::forward<ClusterNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A token that ensures idempotency. This token expires in 10 minutes.</p>
   */
  inline const Aws::String& GetClientToken() const { return m_clientToken; }
  inline bool ClientTokenHasBeenSet() const { return m_clientTokenHasBeenSet; }
  template <typename ClientTokenT = Aws::String>
  void SetClientToken(ClientTokenT&& value) {
    m_clientTokenHasBeenSet = true;
    m_clientToken = std::forward<ClientTokenT>(value);
  }
  template <typename ClientTokenT = Aws::String>
  UpdateKxClusterDatabasesRequest& WithClientToken(ClientTokenT&& value) {
    SetClientToken(std::forward<ClientTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The structure of databases mounted on the cluster.</p>
   */
  inline const Aws::Vector<KxDatabaseConfiguration>& GetDatabases() const { return m_databases; }
  inline bool DatabasesHasBeenSet() const { return m_databasesHasBeenSet; }
  template <typename DatabasesT = Aws::Vector<KxDatabaseConfiguration>>
  void SetDatabases(DatabasesT&& value) {
    m_databasesHasBeenSet = true;
    m_databases = std::forward<DatabasesT>(value);
  }
  template <typename DatabasesT = Aws::Vector<KxDatabaseConfiguration>>
  UpdateKxClusterDatabasesRequest& WithDatabases(DatabasesT&& value) {
    SetDatabases(std::forward<DatabasesT>(value));
    return *this;
  }
  template <typename DatabasesT = KxDatabaseConfiguration>
  UpdateKxClusterDatabasesRequest& AddDatabases(DatabasesT&& value) {
    m_databasesHasBeenSet = true;
    m_databases.emplace_back(std::forward<DatabasesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p> The configuration that allows you to choose how you want to update the
   * databases on a cluster. </p>
   */
  inline const KxDeploymentConfiguration& GetDeploymentConfiguration() const { return m_deploymentConfiguration; }
  inline bool DeploymentConfigurationHasBeenSet() const { return m_deploymentConfigurationHasBeenSet; }
  template <typename DeploymentConfigurationT = KxDeploymentConfiguration>
  void SetDeploymentConfiguration(DeploymentConfigurationT&& value) {
    m_deploymentConfigurationHasBeenSet = true;
    m_deploymentConfiguration = std::forward<DeploymentConfigurationT>(value);
  }
  template <typename DeploymentConfigurationT = KxDeploymentConfiguration>
  UpdateKxClusterDatabasesRequest& WithDeploymentConfiguration(DeploymentConfigurationT&& value) {
    SetDeploymentConfiguration(std::forward<DeploymentConfigurationT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_environmentId;

  Aws::String m_clusterName;

  Aws::String m_clientToken{Aws::Utils::UUID::PseudoRandomUUID()};

  Aws::Vector<KxDatabaseConfiguration> m_databases;

  KxDeploymentConfiguration m_deploymentConfiguration;
  bool m_environmentIdHasBeenSet = false;
  bool m_clusterNameHasBeenSet = false;
  bool m_clientTokenHasBeenSet = true;
  bool m_databasesHasBeenSet = false;
  bool m_deploymentConfigurationHasBeenSet = false;
};

}  // namespace Model
}  // namespace finspace
}  // namespace Aws
