#ifndef GEN_UI_GFX_X_XV_H_
#define GEN_UI_GFX_X_XV_H_

#include <array>
#include <cstddef>
#include <cstdint>
#include <cstring>
#include <vector>

#include "base/component_export.h"
#include "base/memory/ref_counted_memory.h"
#include "base/memory/scoped_refptr.h"
#include "base/optional.h"
#include "base/files/scoped_file.h"
#include "ui/gfx/x/xproto_types.h"
#include "shm.h"
#include "xproto.h"

namespace x11 {

class Connection;

class COMPONENT_EXPORT(X11) Xv {
  public:
  static constexpr unsigned major_version = 2;
  static constexpr unsigned minor_version = 2;

  Xv(Connection* connection,
      const x11::QueryExtensionReply& info);

  uint8_t present() const {
    return info_.present;
  }
  uint8_t major_opcode() const {
    return info_.major_opcode;
  }
  uint8_t first_event() const {
    return info_.first_event;
  }
  uint8_t first_error() const {
    return info_.first_error;
  }

  Connection* connection() const { return connection_; }

  enum class Port : uint32_t {};

  enum class Encoding : uint32_t {};

  enum class Type : int {
    InputMask = 1 << 0,
    OutputMask = 1 << 1,
    VideoMask = 1 << 2,
    StillMask = 1 << 3,
    ImageMask = 1 << 4,
  };

  enum class ImageFormatInfoType : int {
    RGB = 0,
    YUV = 1,
  };

  enum class ImageFormatInfoFormat : int {
    Packed = 0,
    Planar = 1,
  };

  enum class AttributeFlag : int {
    Gettable = 1 << 0,
    Settable = 1 << 1,
  };

  enum class VideoNotifyReason : int {
    Started = 0,
    Stopped = 1,
    Busy = 2,
    Preempted = 3,
    HardError = 4,
  };

  enum class ScanlineOrder : int {
    TopToBottom = 0,
    BottomToTop = 1,
  };

  enum class GrabPortStatus : int {
    Success = 0,
    BadExtension = 1,
    AlreadyGrabbed = 2,
    InvalidTime = 3,
    BadReply = 4,
    BadAlloc = 5,
  };

  struct Rational {
    int32_t numerator{};
    int32_t denominator{};
  };

  struct Format {
    VisualId visual{};
    uint8_t depth{};
  };

  struct AdaptorInfo {
    Port base_id{};
    uint16_t num_ports{};
    Type type{};
    std::string name{};
    std::vector<Format> formats{};
  };

  struct EncodingInfo {
    Encoding encoding{};
    uint16_t width{};
    uint16_t height{};
    Rational rate{};
    std::string name{};
  };

  struct Image {
    uint32_t id{};
    uint16_t width{};
    uint16_t height{};
    std::vector<uint32_t> pitches{};
    std::vector<uint32_t> offsets{};
    std::vector<uint8_t> data{};
  };

  struct AttributeInfo {
    AttributeFlag flags{};
    int32_t min{};
    int32_t max{};
    std::string name{};
  };

  struct ImageFormatInfo {
    uint32_t id{};
    ImageFormatInfoType type{};
    ImageOrder byte_order{};
    std::array<uint8_t, 16> guid{};
    uint8_t bpp{};
    uint8_t num_planes{};
    uint8_t depth{};
    uint32_t red_mask{};
    uint32_t green_mask{};
    uint32_t blue_mask{};
    ImageFormatInfoFormat format{};
    uint32_t y_sample_bits{};
    uint32_t u_sample_bits{};
    uint32_t v_sample_bits{};
    uint32_t vhorz_y_period{};
    uint32_t vhorz_u_period{};
    uint32_t vhorz_v_period{};
    uint32_t vvert_y_period{};
    uint32_t vvert_u_period{};
    uint32_t vvert_v_period{};
    std::array<uint8_t, 32> vcomp_order{};
    ScanlineOrder vscanline_order{};
  };

  struct BadPortError {
    uint16_t sequence{};
  };

  struct BadEncodingError {
    uint16_t sequence{};
  };

  struct BadControlError {
    uint16_t sequence{};
  };

  struct VideoNotifyEvent {
    static constexpr int type_id = 81;
    static constexpr uint8_t opcode = 0;
    bool send_event{};
    VideoNotifyReason reason{};
    uint16_t sequence{};
    Time time{};
    Drawable drawable{};
    Port port{};

    x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&drawable); }
  };

  struct PortNotifyEvent {
    static constexpr int type_id = 82;
    static constexpr uint8_t opcode = 1;
    bool send_event{};
    uint16_t sequence{};
    Time time{};
    Port port{};
    Atom attribute{};
    int32_t value{};

    x11::Window* GetWindow() { return nullptr; }
  };

  struct QueryExtensionRequest {
  };

  struct QueryExtensionReply {
    uint16_t sequence{};
    uint16_t major{};
    uint16_t minor{};
  };

  using QueryExtensionResponse = Response<QueryExtensionReply>;

  Future<QueryExtensionReply> QueryExtension(
      const QueryExtensionRequest& request);

  struct QueryAdaptorsRequest {
    Window window{};
  };

  struct QueryAdaptorsReply {
    uint16_t sequence{};
    std::vector<AdaptorInfo> info{};
  };

  using QueryAdaptorsResponse = Response<QueryAdaptorsReply>;

  Future<QueryAdaptorsReply> QueryAdaptors(
      const QueryAdaptorsRequest& request);

  struct QueryEncodingsRequest {
    Port port{};
  };

  struct QueryEncodingsReply {
    uint16_t sequence{};
    std::vector<EncodingInfo> info{};
  };

  using QueryEncodingsResponse = Response<QueryEncodingsReply>;

  Future<QueryEncodingsReply> QueryEncodings(
      const QueryEncodingsRequest& request);

  struct GrabPortRequest {
    Port port{};
    Time time{};
  };

  struct GrabPortReply {
    GrabPortStatus result{};
    uint16_t sequence{};
  };

  using GrabPortResponse = Response<GrabPortReply>;

  Future<GrabPortReply> GrabPort(
      const GrabPortRequest& request);

  struct UngrabPortRequest {
    Port port{};
    Time time{};
  };

  using UngrabPortResponse = Response<void>;

  Future<void> UngrabPort(
      const UngrabPortRequest& request);

  struct PutVideoRequest {
    Port port{};
    Drawable drawable{};
    GraphicsContext gc{};
    int16_t vid_x{};
    int16_t vid_y{};
    uint16_t vid_w{};
    uint16_t vid_h{};
    int16_t drw_x{};
    int16_t drw_y{};
    uint16_t drw_w{};
    uint16_t drw_h{};
  };

  using PutVideoResponse = Response<void>;

  Future<void> PutVideo(
      const PutVideoRequest& request);

  struct PutStillRequest {
    Port port{};
    Drawable drawable{};
    GraphicsContext gc{};
    int16_t vid_x{};
    int16_t vid_y{};
    uint16_t vid_w{};
    uint16_t vid_h{};
    int16_t drw_x{};
    int16_t drw_y{};
    uint16_t drw_w{};
    uint16_t drw_h{};
  };

  using PutStillResponse = Response<void>;

  Future<void> PutStill(
      const PutStillRequest& request);

  struct GetVideoRequest {
    Port port{};
    Drawable drawable{};
    GraphicsContext gc{};
    int16_t vid_x{};
    int16_t vid_y{};
    uint16_t vid_w{};
    uint16_t vid_h{};
    int16_t drw_x{};
    int16_t drw_y{};
    uint16_t drw_w{};
    uint16_t drw_h{};
  };

  using GetVideoResponse = Response<void>;

  Future<void> GetVideo(
      const GetVideoRequest& request);

  struct GetStillRequest {
    Port port{};
    Drawable drawable{};
    GraphicsContext gc{};
    int16_t vid_x{};
    int16_t vid_y{};
    uint16_t vid_w{};
    uint16_t vid_h{};
    int16_t drw_x{};
    int16_t drw_y{};
    uint16_t drw_w{};
    uint16_t drw_h{};
  };

  using GetStillResponse = Response<void>;

  Future<void> GetStill(
      const GetStillRequest& request);

  struct StopVideoRequest {
    Port port{};
    Drawable drawable{};
  };

  using StopVideoResponse = Response<void>;

  Future<void> StopVideo(
      const StopVideoRequest& request);

  struct SelectVideoNotifyRequest {
    Drawable drawable{};
    uint8_t onoff{};
  };

  using SelectVideoNotifyResponse = Response<void>;

  Future<void> SelectVideoNotify(
      const SelectVideoNotifyRequest& request);

  struct SelectPortNotifyRequest {
    Port port{};
    uint8_t onoff{};
  };

  using SelectPortNotifyResponse = Response<void>;

  Future<void> SelectPortNotify(
      const SelectPortNotifyRequest& request);

  struct QueryBestSizeRequest {
    Port port{};
    uint16_t vid_w{};
    uint16_t vid_h{};
    uint16_t drw_w{};
    uint16_t drw_h{};
    uint8_t motion{};
  };

  struct QueryBestSizeReply {
    uint16_t sequence{};
    uint16_t actual_width{};
    uint16_t actual_height{};
  };

  using QueryBestSizeResponse = Response<QueryBestSizeReply>;

  Future<QueryBestSizeReply> QueryBestSize(
      const QueryBestSizeRequest& request);

  struct SetPortAttributeRequest {
    Port port{};
    Atom attribute{};
    int32_t value{};
  };

  using SetPortAttributeResponse = Response<void>;

  Future<void> SetPortAttribute(
      const SetPortAttributeRequest& request);

  struct GetPortAttributeRequest {
    Port port{};
    Atom attribute{};
  };

  struct GetPortAttributeReply {
    uint16_t sequence{};
    int32_t value{};
  };

  using GetPortAttributeResponse = Response<GetPortAttributeReply>;

  Future<GetPortAttributeReply> GetPortAttribute(
      const GetPortAttributeRequest& request);

  struct QueryPortAttributesRequest {
    Port port{};
  };

  struct QueryPortAttributesReply {
    uint16_t sequence{};
    uint32_t text_size{};
    std::vector<AttributeInfo> attributes{};
  };

  using QueryPortAttributesResponse = Response<QueryPortAttributesReply>;

  Future<QueryPortAttributesReply> QueryPortAttributes(
      const QueryPortAttributesRequest& request);

  struct ListImageFormatsRequest {
    Port port{};
  };

  struct ListImageFormatsReply {
    uint16_t sequence{};
    std::vector<ImageFormatInfo> format{};
  };

  using ListImageFormatsResponse = Response<ListImageFormatsReply>;

  Future<ListImageFormatsReply> ListImageFormats(
      const ListImageFormatsRequest& request);

  struct QueryImageAttributesRequest {
    Port port{};
    uint32_t id{};
    uint16_t width{};
    uint16_t height{};
  };

  struct QueryImageAttributesReply {
    uint16_t sequence{};
    uint32_t data_size{};
    uint16_t width{};
    uint16_t height{};
    std::vector<uint32_t> pitches{};
    std::vector<uint32_t> offsets{};
  };

  using QueryImageAttributesResponse = Response<QueryImageAttributesReply>;

  Future<QueryImageAttributesReply> QueryImageAttributes(
      const QueryImageAttributesRequest& request);

  struct PutImageRequest {
    Port port{};
    Drawable drawable{};
    GraphicsContext gc{};
    uint32_t id{};
    int16_t src_x{};
    int16_t src_y{};
    uint16_t src_w{};
    uint16_t src_h{};
    int16_t drw_x{};
    int16_t drw_y{};
    uint16_t drw_w{};
    uint16_t drw_h{};
    uint16_t width{};
    uint16_t height{};
    std::vector<uint8_t> data{};
  };

  using PutImageResponse = Response<void>;

  Future<void> PutImage(
      const PutImageRequest& request);

  struct ShmPutImageRequest {
    Port port{};
    Drawable drawable{};
    GraphicsContext gc{};
    Shm::Seg shmseg{};
    uint32_t id{};
    uint32_t offset{};
    int16_t src_x{};
    int16_t src_y{};
    uint16_t src_w{};
    uint16_t src_h{};
    int16_t drw_x{};
    int16_t drw_y{};
    uint16_t drw_w{};
    uint16_t drw_h{};
    uint16_t width{};
    uint16_t height{};
    uint8_t send_event{};
  };

  using ShmPutImageResponse = Response<void>;

  Future<void> ShmPutImage(
      const ShmPutImageRequest& request);

  private:
  x11::Connection* const connection_;
  x11::QueryExtensionReply info_{};
};

}  // namespace x11

inline constexpr x11::Xv::Type operator|(
    x11::Xv::Type l, x11::Xv::Type r) {
  using T = std::underlying_type_t<x11::Xv::Type>;
  return static_cast<x11::Xv::Type>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Xv::Type operator&(
    x11::Xv::Type l, x11::Xv::Type r) {
  using T = std::underlying_type_t<x11::Xv::Type>;
  return static_cast<x11::Xv::Type>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Xv::ImageFormatInfoType operator|(
    x11::Xv::ImageFormatInfoType l, x11::Xv::ImageFormatInfoType r) {
  using T = std::underlying_type_t<x11::Xv::ImageFormatInfoType>;
  return static_cast<x11::Xv::ImageFormatInfoType>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Xv::ImageFormatInfoType operator&(
    x11::Xv::ImageFormatInfoType l, x11::Xv::ImageFormatInfoType r) {
  using T = std::underlying_type_t<x11::Xv::ImageFormatInfoType>;
  return static_cast<x11::Xv::ImageFormatInfoType>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Xv::ImageFormatInfoFormat operator|(
    x11::Xv::ImageFormatInfoFormat l, x11::Xv::ImageFormatInfoFormat r) {
  using T = std::underlying_type_t<x11::Xv::ImageFormatInfoFormat>;
  return static_cast<x11::Xv::ImageFormatInfoFormat>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Xv::ImageFormatInfoFormat operator&(
    x11::Xv::ImageFormatInfoFormat l, x11::Xv::ImageFormatInfoFormat r) {
  using T = std::underlying_type_t<x11::Xv::ImageFormatInfoFormat>;
  return static_cast<x11::Xv::ImageFormatInfoFormat>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Xv::AttributeFlag operator|(
    x11::Xv::AttributeFlag l, x11::Xv::AttributeFlag r) {
  using T = std::underlying_type_t<x11::Xv::AttributeFlag>;
  return static_cast<x11::Xv::AttributeFlag>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Xv::AttributeFlag operator&(
    x11::Xv::AttributeFlag l, x11::Xv::AttributeFlag r) {
  using T = std::underlying_type_t<x11::Xv::AttributeFlag>;
  return static_cast<x11::Xv::AttributeFlag>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Xv::VideoNotifyReason operator|(
    x11::Xv::VideoNotifyReason l, x11::Xv::VideoNotifyReason r) {
  using T = std::underlying_type_t<x11::Xv::VideoNotifyReason>;
  return static_cast<x11::Xv::VideoNotifyReason>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Xv::VideoNotifyReason operator&(
    x11::Xv::VideoNotifyReason l, x11::Xv::VideoNotifyReason r) {
  using T = std::underlying_type_t<x11::Xv::VideoNotifyReason>;
  return static_cast<x11::Xv::VideoNotifyReason>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Xv::ScanlineOrder operator|(
    x11::Xv::ScanlineOrder l, x11::Xv::ScanlineOrder r) {
  using T = std::underlying_type_t<x11::Xv::ScanlineOrder>;
  return static_cast<x11::Xv::ScanlineOrder>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Xv::ScanlineOrder operator&(
    x11::Xv::ScanlineOrder l, x11::Xv::ScanlineOrder r) {
  using T = std::underlying_type_t<x11::Xv::ScanlineOrder>;
  return static_cast<x11::Xv::ScanlineOrder>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Xv::GrabPortStatus operator|(
    x11::Xv::GrabPortStatus l, x11::Xv::GrabPortStatus r) {
  using T = std::underlying_type_t<x11::Xv::GrabPortStatus>;
  return static_cast<x11::Xv::GrabPortStatus>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Xv::GrabPortStatus operator&(
    x11::Xv::GrabPortStatus l, x11::Xv::GrabPortStatus r) {
  using T = std::underlying_type_t<x11::Xv::GrabPortStatus>;
  return static_cast<x11::Xv::GrabPortStatus>(
      static_cast<T>(l) & static_cast<T>(r));
}


#endif  // GEN_UI_GFX_X_XV_H_
