#ifndef GEN_UI_GFX_X_XPROTO_H_
#define GEN_UI_GFX_X_XPROTO_H_

#include <array>
#include <cstddef>
#include <cstdint>
#include <cstring>
#include <vector>

#include "base/component_export.h"
#include "base/memory/ref_counted_memory.h"
#include "base/memory/scoped_refptr.h"
#include "base/optional.h"
#include "base/files/scoped_file.h"
#include "ui/gfx/x/xproto_types.h"

namespace x11 {

class Connection;

enum class GraphicsContext : uint32_t {};

enum class ColorMap : uint32_t {};

enum class Bool32 : uint32_t {};

enum class VisualId : uint32_t {};

enum class KeySym : uint32_t {};

enum class KeyCode : uint8_t {};

enum class KeyCode32 : uint32_t {};

enum class Button : uint8_t {};

enum class VisualClass : int {
  StaticGray = 0,
  GrayScale = 1,
  StaticColor = 2,
  PseudoColor = 3,
  TrueColor = 4,
  DirectColor = 5,
};

enum class EventMask : int {
  NoEvent = 0,
  KeyPress = 1 << 0,
  KeyRelease = 1 << 1,
  ButtonPress = 1 << 2,
  ButtonRelease = 1 << 3,
  EnterWindow = 1 << 4,
  LeaveWindow = 1 << 5,
  PointerMotion = 1 << 6,
  PointerMotionHint = 1 << 7,
  Button1Motion = 1 << 8,
  Button2Motion = 1 << 9,
  Button3Motion = 1 << 10,
  Button4Motion = 1 << 11,
  Button5Motion = 1 << 12,
  ButtonMotion = 1 << 13,
  KeymapState = 1 << 14,
  Exposure = 1 << 15,
  VisibilityChange = 1 << 16,
  StructureNotify = 1 << 17,
  ResizeRedirect = 1 << 18,
  SubstructureNotify = 1 << 19,
  SubstructureRedirect = 1 << 20,
  FocusChange = 1 << 21,
  PropertyChange = 1 << 22,
  ColorMapChange = 1 << 23,
  OwnerGrabButton = 1 << 24,
};

enum class BackingStore : int {
  NotUseful = 0,
  WhenMapped = 1,
  Always = 2,
};

enum class ImageOrder : int {
  LSBFirst = 0,
  MSBFirst = 1,
};

enum class ModMask : int {
  Shift = 1 << 0,
  Lock = 1 << 1,
  Control = 1 << 2,
  c_1 = 1 << 3,
  c_2 = 1 << 4,
  c_3 = 1 << 5,
  c_4 = 1 << 6,
  c_5 = 1 << 7,
  Any = 1 << 15,
};

enum class KeyButMask : int {
  Shift = 1 << 0,
  Lock = 1 << 1,
  Control = 1 << 2,
  Mod1 = 1 << 3,
  Mod2 = 1 << 4,
  Mod3 = 1 << 5,
  Mod4 = 1 << 6,
  Mod5 = 1 << 7,
  Button1 = 1 << 8,
  Button2 = 1 << 9,
  Button3 = 1 << 10,
  Button4 = 1 << 11,
  Button5 = 1 << 12,
};

enum class Window : uint32_t {
  None = 0,
};

enum class ButtonMask : int {
  c_1 = 1 << 8,
  c_2 = 1 << 9,
  c_3 = 1 << 10,
  c_4 = 1 << 11,
  c_5 = 1 << 12,
  Any = 1 << 15,
};

enum class Motion : int {
  Normal = 0,
  Hint = 1,
};

enum class NotifyDetail : int {
  Ancestor = 0,
  Virtual = 1,
  Inferior = 2,
  Nonlinear = 3,
  NonlinearVirtual = 4,
  Pointer = 5,
  PointerRoot = 6,
  None = 7,
};

enum class NotifyMode : int {
  Normal = 0,
  Grab = 1,
  Ungrab = 2,
  WhileGrabbed = 3,
};

enum class Visibility : int {
  Unobscured = 0,
  PartiallyObscured = 1,
  FullyObscured = 2,
};

enum class Place : int {
  OnTop = 0,
  OnBottom = 1,
};

enum class Property : int {
  NewValue = 0,
  Delete = 1,
};

enum class Time : uint32_t {
  CurrentTime = 0,
};

enum class Atom : uint32_t {
  None = 0,
  Any = 0,
  PRIMARY = 1,
  SECONDARY = 2,
  ARC = 3,
  ATOM = 4,
  BITMAP = 5,
  CARDINAL = 6,
  COLORMAP = 7,
  CURSOR = 8,
  CUT_BUFFER0 = 9,
  CUT_BUFFER1 = 10,
  CUT_BUFFER2 = 11,
  CUT_BUFFER3 = 12,
  CUT_BUFFER4 = 13,
  CUT_BUFFER5 = 14,
  CUT_BUFFER6 = 15,
  CUT_BUFFER7 = 16,
  DRAWABLE = 17,
  FONT = 18,
  INTEGER = 19,
  PIXMAP = 20,
  POINT = 21,
  RECTANGLE = 22,
  RESOURCE_MANAGER = 23,
  RGB_COLOR_MAP = 24,
  RGB_BEST_MAP = 25,
  RGB_BLUE_MAP = 26,
  RGB_DEFAULT_MAP = 27,
  RGB_GRAY_MAP = 28,
  RGB_GREEN_MAP = 29,
  RGB_RED_MAP = 30,
  STRING = 31,
  VISUALID = 32,
  WINDOW = 33,
  WM_COMMAND = 34,
  WM_HINTS = 35,
  WM_CLIENT_MACHINE = 36,
  WM_ICON_NAME = 37,
  WM_ICON_SIZE = 38,
  WM_NAME = 39,
  WM_NORMAL_HINTS = 40,
  WM_SIZE_HINTS = 41,
  WM_ZOOM_HINTS = 42,
  MIN_SPACE = 43,
  NORM_SPACE = 44,
  MAX_SPACE = 45,
  END_SPACE = 46,
  SUPERSCRIPT_X = 47,
  SUPERSCRIPT_Y = 48,
  SUBSCRIPT_X = 49,
  SUBSCRIPT_Y = 50,
  UNDERLINE_POSITION = 51,
  UNDERLINE_THICKNESS = 52,
  STRIKEOUT_ASCENT = 53,
  STRIKEOUT_DESCENT = 54,
  ITALIC_ANGLE = 55,
  X_HEIGHT = 56,
  QUAD_WIDTH = 57,
  WEIGHT = 58,
  POINT_SIZE = 59,
  RESOLUTION = 60,
  COPYRIGHT = 61,
  NOTICE = 62,
  FONT_NAME = 63,
  FAMILY_NAME = 64,
  FULL_NAME = 65,
  CAP_HEIGHT = 66,
  WM_CLASS = 67,
  WM_TRANSIENT_FOR = 68,
};

enum class ColormapState : int {
  Uninstalled = 0,
  Installed = 1,
};

enum class Colormap : int {
  None = 0,
};

enum class Mapping : int {
  Modifier = 0,
  Keyboard = 1,
  Pointer = 2,
};

enum class WindowClass : int {
  CopyFromParent = 0,
  InputOutput = 1,
  InputOnly = 2,
};

enum class CreateWindowAttribute : int {
  BackPixmap = 1 << 0,
  BackPixel = 1 << 1,
  BorderPixmap = 1 << 2,
  BorderPixel = 1 << 3,
  BitGravity = 1 << 4,
  WinGravity = 1 << 5,
  BackingStore = 1 << 6,
  BackingPlanes = 1 << 7,
  BackingPixel = 1 << 8,
  OverrideRedirect = 1 << 9,
  SaveUnder = 1 << 10,
  EventMask = 1 << 11,
  DontPropagate = 1 << 12,
  Colormap = 1 << 13,
  Cursor = 1 << 14,
};

enum class BackPixmap : int {
  None = 0,
  ParentRelative = 1,
};

enum class Gravity : int {
  BitForget = 0,
  WinUnmap = 0,
  NorthWest = 1,
  North = 2,
  NorthEast = 3,
  West = 4,
  Center = 5,
  East = 6,
  SouthWest = 7,
  South = 8,
  SouthEast = 9,
  Static = 10,
};

enum class MapState : int {
  Unmapped = 0,
  Unviewable = 1,
  Viewable = 2,
};

enum class SetMode : int {
  Insert = 0,
  Delete = 1,
};

enum class ConfigWindow : int {
  X = 1 << 0,
  Y = 1 << 1,
  Width = 1 << 2,
  Height = 1 << 3,
  BorderWidth = 1 << 4,
  Sibling = 1 << 5,
  StackMode = 1 << 6,
};

enum class StackMode : int {
  Above = 0,
  Below = 1,
  TopIf = 2,
  BottomIf = 3,
  Opposite = 4,
};

enum class Circulate : int {
  RaiseLowest = 0,
  LowerHighest = 1,
};

enum class PropMode : int {
  Replace = 0,
  Prepend = 1,
  Append = 2,
};

enum class GetPropertyType : int {
  Any = 0,
};

enum class SendEventDest : int {
  PointerWindow = 0,
  ItemFocus = 1,
};

enum class GrabMode : int {
  Sync = 0,
  Async = 1,
};

enum class GrabStatus : int {
  Success = 0,
  AlreadyGrabbed = 1,
  InvalidTime = 2,
  NotViewable = 3,
  Frozen = 4,
};

enum class Cursor : uint32_t {
  None = 0,
};

enum class ButtonIndex : int {
  Any = 0,
  c_1 = 1,
  c_2 = 2,
  c_3 = 3,
  c_4 = 4,
  c_5 = 5,
};

enum class Grab : int {
  Any = 0,
};

enum class Allow : int {
  AsyncPointer = 0,
  SyncPointer = 1,
  ReplayPointer = 2,
  AsyncKeyboard = 3,
  SyncKeyboard = 4,
  ReplayKeyboard = 5,
  AsyncBoth = 6,
  SyncBoth = 7,
};

enum class InputFocus : int {
  None = 0,
  PointerRoot = 1,
  Parent = 2,
  FollowKeyboard = 3,
};

enum class FontDraw : int {
  LeftToRight = 0,
  RightToLeft = 1,
};

enum class GraphicsContextAttribute : int {
  Function = 1 << 0,
  PlaneMask = 1 << 1,
  Foreground = 1 << 2,
  Background = 1 << 3,
  LineWidth = 1 << 4,
  LineStyle = 1 << 5,
  CapStyle = 1 << 6,
  JoinStyle = 1 << 7,
  FillStyle = 1 << 8,
  FillRule = 1 << 9,
  Tile = 1 << 10,
  Stipple = 1 << 11,
  TileStippleOriginX = 1 << 12,
  TileStippleOriginY = 1 << 13,
  Font = 1 << 14,
  SubwindowMode = 1 << 15,
  GraphicsExposures = 1 << 16,
  ClipOriginX = 1 << 17,
  ClipOriginY = 1 << 18,
  ClipMask = 1 << 19,
  DashOffset = 1 << 20,
  DashList = 1 << 21,
  ArcMode = 1 << 22,
};

enum class Gx : int {
  clear = 0,
  c_and = 1,
  andReverse = 2,
  copy = 3,
  andInverted = 4,
  noop = 5,
  c_xor = 6,
  c_or = 7,
  nor = 8,
  equiv = 9,
  invert = 10,
  orReverse = 11,
  copyInverted = 12,
  orInverted = 13,
  nand = 14,
  set = 15,
};

enum class LineStyle : int {
  Solid = 0,
  OnOffDash = 1,
  DoubleDash = 2,
};

enum class CapStyle : int {
  NotLast = 0,
  Butt = 1,
  Round = 2,
  Projecting = 3,
};

enum class JoinStyle : int {
  Miter = 0,
  Round = 1,
  Bevel = 2,
};

enum class FillStyle : int {
  Solid = 0,
  Tiled = 1,
  Stippled = 2,
  OpaqueStippled = 3,
};

enum class FillRule : int {
  EvenOdd = 0,
  Winding = 1,
};

enum class SubwindowMode : int {
  ClipByChildren = 0,
  IncludeInferiors = 1,
};

enum class ArcMode : int {
  Chord = 0,
  PieSlice = 1,
};

enum class ClipOrdering : int {
  Unsorted = 0,
  YSorted = 1,
  YXSorted = 2,
  YXBanded = 3,
};

enum class CoordMode : int {
  Origin = 0,
  Previous = 1,
};

enum class PolyShape : int {
  Complex = 0,
  Nonconvex = 1,
  Convex = 2,
};

enum class ImageFormat : int {
  XYBitmap = 0,
  XYPixmap = 1,
  ZPixmap = 2,
};

enum class ColormapAlloc : int {
  None = 0,
  All = 1,
};

enum class ColorFlag : int {
  Red = 1 << 0,
  Green = 1 << 1,
  Blue = 1 << 2,
};

enum class Pixmap : uint32_t {
  None = 0,
};

enum class Font : uint32_t {
  None = 0,
};

enum class QueryShapeOf : int {
  LargestCursor = 0,
  FastestTile = 1,
  FastestStipple = 2,
};

enum class Keyboard : int {
  KeyClickPercent = 1 << 0,
  BellPercent = 1 << 1,
  BellPitch = 1 << 2,
  BellDuration = 1 << 3,
  Led = 1 << 4,
  LedMode = 1 << 5,
  Key = 1 << 6,
  AutoRepeatMode = 1 << 7,
};

enum class LedMode : int {
  Off = 0,
  On = 1,
};

enum class AutoRepeatMode : int {
  Off = 0,
  On = 1,
  Default = 2,
};

enum class Blanking : int {
  NotPreferred = 0,
  Preferred = 1,
  Default = 2,
};

enum class Exposures : int {
  NotAllowed = 0,
  Allowed = 1,
  Default = 2,
};

enum class HostMode : int {
  Insert = 0,
  Delete = 1,
};

enum class Family : int {
  Internet = 0,
  DECnet = 1,
  Chaos = 2,
  ServerInterpreted = 5,
  Internet6 = 6,
};

enum class AccessControl : int {
  Disable = 0,
  Enable = 1,
};

enum class CloseDown : int {
  DestroyAll = 0,
  RetainPermanent = 1,
  RetainTemporary = 2,
};

enum class Kill : int {
  AllTemporary = 0,
};

enum class ScreenSaverMode : int {
  Reset = 0,
  Active = 1,
};

enum class MappingStatus : int {
  Success = 0,
  Busy = 1,
  Failure = 2,
};

enum class MapIndex : int {
  Shift = 0,
  Lock = 1,
  Control = 2,
  c_1 = 3,
  c_2 = 4,
  c_3 = 5,
  c_4 = 6,
  c_5 = 7,
};

struct Drawable {
  Drawable() : value{} {}

  Drawable(Window value) : value{static_cast<uint32_t>(value)} {}
  operator Window() const { return static_cast<Window>(value); }

  Drawable(Pixmap value) : value{static_cast<uint32_t>(value)} {}
  operator Pixmap() const { return static_cast<Pixmap>(value); }

  uint32_t value{};
};

struct Fontable {
  Fontable() : value{} {}

  Fontable(Font value) : value{static_cast<uint32_t>(value)} {}
  operator Font() const { return static_cast<Font>(value); }

  Fontable(GraphicsContext value) : value{static_cast<uint32_t>(value)} {}
  operator GraphicsContext() const { return static_cast<GraphicsContext>(value); }

  uint32_t value{};
};

struct Char16 {
  uint8_t byte1{};
  uint8_t byte2{};
};

struct Point {
  int16_t x{};
  int16_t y{};
};

struct Rectangle {
  int16_t x{};
  int16_t y{};
  uint16_t width{};
  uint16_t height{};
};

struct Arc {
  int16_t x{};
  int16_t y{};
  uint16_t width{};
  uint16_t height{};
  int16_t angle1{};
  int16_t angle2{};
};

struct Format {
  uint8_t depth{};
  uint8_t bits_per_pixel{};
  uint8_t scanline_pad{};
};

struct VisualType {
  VisualId visual_id{};
  VisualClass c_class{};
  uint8_t bits_per_rgb_value{};
  uint16_t colormap_entries{};
  uint32_t red_mask{};
  uint32_t green_mask{};
  uint32_t blue_mask{};
};

struct Depth {
  uint8_t depth{};
  std::vector<VisualType> visuals{};
};

struct Screen {
  Window root{};
  ColorMap default_colormap{};
  uint32_t white_pixel{};
  uint32_t black_pixel{};
  EventMask current_input_masks{};
  uint16_t width_in_pixels{};
  uint16_t height_in_pixels{};
  uint16_t width_in_millimeters{};
  uint16_t height_in_millimeters{};
  uint16_t min_installed_maps{};
  uint16_t max_installed_maps{};
  VisualId root_visual{};
  BackingStore backing_stores{};
  uint8_t save_unders{};
  uint8_t root_depth{};
  std::vector<Depth> allowed_depths{};
};

struct SetupRequest {
  uint8_t byte_order{};
  uint16_t protocol_major_version{};
  uint16_t protocol_minor_version{};
  std::string authorization_protocol_name{};
  std::string authorization_protocol_data{};
};

struct SetupFailed {
  uint8_t status{};
  uint16_t protocol_major_version{};
  uint16_t protocol_minor_version{};
  uint16_t length{};
  std::string reason{};
};

struct SetupAuthenticate {
  uint8_t status{};
  uint16_t length{};
  std::string reason{};
};

struct Setup {
  uint8_t status{};
  uint16_t protocol_major_version{};
  uint16_t protocol_minor_version{};
  uint16_t length{};
  uint32_t release_number{};
  uint32_t resource_id_base{};
  uint32_t resource_id_mask{};
  uint32_t motion_buffer_size{};
  uint16_t maximum_request_length{};
  ImageOrder image_byte_order{};
  ImageOrder bitmap_format_bit_order{};
  uint8_t bitmap_format_scanline_unit{};
  uint8_t bitmap_format_scanline_pad{};
  KeyCode min_keycode{};
  KeyCode max_keycode{};
  std::string vendor{};
  std::vector<Format> pixmap_formats{};
  std::vector<Screen> roots{};
};

struct KeyEvent {
  static constexpr int type_id = 52;
  enum Opcode {
    Press = 2,
    Release = 3,
  } opcode{};
  bool send_event{};
  KeyCode detail{};
  uint16_t sequence{};
  Time time{};
  Window root{};
  Window event{};
  Window child{};
  int16_t root_x{};
  int16_t root_y{};
  int16_t event_x{};
  int16_t event_y{};
  KeyButMask state{};
  uint8_t same_screen{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&event); }
};

struct ButtonEvent {
  static constexpr int type_id = 53;
  enum Opcode {
    Press = 4,
    Release = 5,
  } opcode{};
  bool send_event{};
  Button detail{};
  uint16_t sequence{};
  Time time{};
  Window root{};
  Window event{};
  Window child{};
  int16_t root_x{};
  int16_t root_y{};
  int16_t event_x{};
  int16_t event_y{};
  KeyButMask state{};
  uint8_t same_screen{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&event); }
};

struct MotionNotifyEvent {
  static constexpr int type_id = 54;
  static constexpr uint8_t opcode = 6;
  bool send_event{};
  Motion detail{};
  uint16_t sequence{};
  Time time{};
  Window root{};
  Window event{};
  Window child{};
  int16_t root_x{};
  int16_t root_y{};
  int16_t event_x{};
  int16_t event_y{};
  KeyButMask state{};
  uint8_t same_screen{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&event); }
};

struct CrossingEvent {
  static constexpr int type_id = 55;
  enum Opcode {
    EnterNotify = 7,
    LeaveNotify = 8,
  } opcode{};
  bool send_event{};
  NotifyDetail detail{};
  uint16_t sequence{};
  Time time{};
  Window root{};
  Window event{};
  Window child{};
  int16_t root_x{};
  int16_t root_y{};
  int16_t event_x{};
  int16_t event_y{};
  KeyButMask state{};
  NotifyMode mode{};
  uint8_t same_screen_focus{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&event); }
};

struct FocusEvent {
  static constexpr int type_id = 56;
  enum Opcode {
    In = 9,
    Out = 10,
  } opcode{};
  bool send_event{};
  NotifyDetail detail{};
  uint16_t sequence{};
  Window event{};
  NotifyMode mode{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&event); }
};

struct KeymapNotifyEvent {
  static constexpr int type_id = 57;
  static constexpr uint8_t opcode = 11;
  bool send_event{};
  std::array<uint8_t, 31> keys{};

  x11::Window* GetWindow() { return nullptr; }
};

struct ExposeEvent {
  static constexpr int type_id = 58;
  static constexpr uint8_t opcode = 12;
  bool send_event{};
  uint16_t sequence{};
  Window window{};
  uint16_t x{};
  uint16_t y{};
  uint16_t width{};
  uint16_t height{};
  uint16_t count{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&window); }
};

struct GraphicsExposureEvent {
  static constexpr int type_id = 59;
  static constexpr uint8_t opcode = 13;
  bool send_event{};
  uint16_t sequence{};
  Drawable drawable{};
  uint16_t x{};
  uint16_t y{};
  uint16_t width{};
  uint16_t height{};
  uint16_t minor_opcode{};
  uint16_t count{};
  uint8_t major_opcode{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&drawable); }
};

struct NoExposureEvent {
  static constexpr int type_id = 60;
  static constexpr uint8_t opcode = 14;
  bool send_event{};
  uint16_t sequence{};
  Drawable drawable{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&drawable); }
};

struct VisibilityNotifyEvent {
  static constexpr int type_id = 61;
  static constexpr uint8_t opcode = 15;
  bool send_event{};
  uint16_t sequence{};
  Window window{};
  Visibility state{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&window); }
};

struct CreateNotifyEvent {
  static constexpr int type_id = 62;
  static constexpr uint8_t opcode = 16;
  bool send_event{};
  uint16_t sequence{};
  Window parent{};
  Window window{};
  int16_t x{};
  int16_t y{};
  uint16_t width{};
  uint16_t height{};
  uint16_t border_width{};
  uint8_t override_redirect{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&window); }
};

struct DestroyNotifyEvent {
  static constexpr int type_id = 63;
  static constexpr uint8_t opcode = 17;
  bool send_event{};
  uint16_t sequence{};
  Window event{};
  Window window{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&event); }
};

struct UnmapNotifyEvent {
  static constexpr int type_id = 64;
  static constexpr uint8_t opcode = 18;
  bool send_event{};
  uint16_t sequence{};
  Window event{};
  Window window{};
  uint8_t from_configure{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&event); }
};

struct MapNotifyEvent {
  static constexpr int type_id = 65;
  static constexpr uint8_t opcode = 19;
  bool send_event{};
  uint16_t sequence{};
  Window event{};
  Window window{};
  uint8_t override_redirect{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&event); }
};

struct MapRequestEvent {
  static constexpr int type_id = 66;
  static constexpr uint8_t opcode = 20;
  bool send_event{};
  uint16_t sequence{};
  Window parent{};
  Window window{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&window); }
};

struct ReparentNotifyEvent {
  static constexpr int type_id = 67;
  static constexpr uint8_t opcode = 21;
  bool send_event{};
  uint16_t sequence{};
  Window event{};
  Window window{};
  Window parent{};
  int16_t x{};
  int16_t y{};
  uint8_t override_redirect{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&event); }
};

struct ConfigureNotifyEvent {
  static constexpr int type_id = 68;
  static constexpr uint8_t opcode = 22;
  bool send_event{};
  uint16_t sequence{};
  Window event{};
  Window window{};
  Window above_sibling{};
  int16_t x{};
  int16_t y{};
  uint16_t width{};
  uint16_t height{};
  uint16_t border_width{};
  uint8_t override_redirect{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&event); }
};

struct ConfigureRequestEvent {
  static constexpr int type_id = 69;
  static constexpr uint8_t opcode = 23;
  bool send_event{};
  StackMode stack_mode{};
  uint16_t sequence{};
  Window parent{};
  Window window{};
  Window sibling{};
  int16_t x{};
  int16_t y{};
  uint16_t width{};
  uint16_t height{};
  uint16_t border_width{};
  ConfigWindow value_mask{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&window); }
};

struct GravityNotifyEvent {
  static constexpr int type_id = 70;
  static constexpr uint8_t opcode = 24;
  bool send_event{};
  uint16_t sequence{};
  Window event{};
  Window window{};
  int16_t x{};
  int16_t y{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&event); }
};

struct ResizeRequestEvent {
  static constexpr int type_id = 71;
  static constexpr uint8_t opcode = 25;
  bool send_event{};
  uint16_t sequence{};
  Window window{};
  uint16_t width{};
  uint16_t height{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&window); }
};

struct CirculateEvent {
  static constexpr int type_id = 72;
  enum Opcode {
    Notify = 26,
    Request = 27,
  } opcode{};
  bool send_event{};
  uint16_t sequence{};
  Window event{};
  Window window{};
  Place place{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&event); }
};

struct PropertyNotifyEvent {
  static constexpr int type_id = 73;
  static constexpr uint8_t opcode = 28;
  bool send_event{};
  uint16_t sequence{};
  Window window{};
  Atom atom{};
  Time time{};
  Property state{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&window); }
};

struct SelectionClearEvent {
  static constexpr int type_id = 74;
  static constexpr uint8_t opcode = 29;
  bool send_event{};
  uint16_t sequence{};
  Time time{};
  Window owner{};
  Atom selection{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&owner); }
};

struct SelectionRequestEvent {
  static constexpr int type_id = 75;
  static constexpr uint8_t opcode = 30;
  bool send_event{};
  uint16_t sequence{};
  Time time{};
  Window owner{};
  Window requestor{};
  Atom selection{};
  Atom target{};
  Atom property{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&owner); }
};

struct SelectionNotifyEvent {
  static constexpr int type_id = 76;
  static constexpr uint8_t opcode = 31;
  bool send_event{};
  uint16_t sequence{};
  Time time{};
  Window requestor{};
  Atom selection{};
  Atom target{};
  Atom property{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&requestor); }
};

struct ColormapNotifyEvent {
  static constexpr int type_id = 77;
  static constexpr uint8_t opcode = 32;
  bool send_event{};
  uint16_t sequence{};
  Window window{};
  ColorMap colormap{};
  uint8_t c_new{};
  ColormapState state{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&window); }
};

union ClientMessageData {
  ClientMessageData() { memset(this, 0, sizeof(*this)); }

  std::array<uint8_t, 20> data8;
  std::array<uint16_t, 10> data16;
  std::array<uint32_t, 5> data32;
};
static_assert(std::is_trivially_copyable<ClientMessageData>::value, "");

struct ClientMessageEvent {
  static constexpr int type_id = 78;
  static constexpr uint8_t opcode = 33;
  bool send_event{};
  uint8_t format{};
  uint16_t sequence{};
  Window window{};
  Atom type{};
  ClientMessageData data{};

  x11::Window* GetWindow() { return reinterpret_cast<x11::Window*>(&window); }
};

struct MappingNotifyEvent {
  static constexpr int type_id = 79;
  static constexpr uint8_t opcode = 34;
  bool send_event{};
  uint16_t sequence{};
  Mapping request{};
  KeyCode first_keycode{};
  uint8_t count{};

  x11::Window* GetWindow() { return nullptr; }
};

struct GeGenericEvent {
  static constexpr int type_id = 80;
  static constexpr uint8_t opcode = 35;
  bool send_event{};
  uint16_t sequence{};

  x11::Window* GetWindow() { return nullptr; }
};

struct RequestError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct ValueError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct WindowError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct PixmapError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct AtomError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct CursorError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct FontError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct MatchError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct DrawableError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct AccessError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct AllocError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct ColormapError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct GContextError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct IDChoiceError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct NameError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct LengthError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct ImplementationError {
  uint16_t sequence{};
  uint32_t bad_value{};
  uint16_t minor_opcode{};
  uint8_t major_opcode{};
};

struct TimeCoord {
  Time time{};
  int16_t x{};
  int16_t y{};
};

struct FontProperty {
  Atom name{};
  uint32_t value{};
};

struct CharInfo {
  int16_t left_side_bearing{};
  int16_t right_side_bearing{};
  int16_t character_width{};
  int16_t ascent{};
  int16_t descent{};
  uint16_t attributes{};
};

struct Str {
  std::string name{};
};

struct Segment {
  int16_t x1{};
  int16_t y1{};
  int16_t x2{};
  int16_t y2{};
};

struct ColorItem {
  uint32_t pixel{};
  uint16_t red{};
  uint16_t green{};
  uint16_t blue{};
  ColorFlag flags{};
};

struct Rgb {
  uint16_t red{};
  uint16_t green{};
  uint16_t blue{};
};

struct Host {
  Family family{};
  std::vector<uint8_t> address{};
};

struct CreateWindowRequest {
  uint8_t depth{};
  Window wid{};
  Window parent{};
  int16_t x{};
  int16_t y{};
  uint16_t width{};
  uint16_t height{};
  uint16_t border_width{};
  WindowClass c_class{};
  VisualId visual{};
  base::Optional<Pixmap> background_pixmap{};
  base::Optional<uint32_t> background_pixel{};
  base::Optional<Pixmap> border_pixmap{};
  base::Optional<uint32_t> border_pixel{};
  base::Optional<Gravity> bit_gravity{};
  base::Optional<Gravity> win_gravity{};
  base::Optional<BackingStore> backing_store{};
  base::Optional<uint32_t> backing_planes{};
  base::Optional<uint32_t> backing_pixel{};
  base::Optional<Bool32> override_redirect{};
  base::Optional<Bool32> save_under{};
  base::Optional<EventMask> event_mask{};
  base::Optional<EventMask> do_not_propogate_mask{};
  base::Optional<ColorMap> colormap{};
  base::Optional<Cursor> cursor{};
};

using CreateWindowResponse = Response<void>;

struct ChangeWindowAttributesRequest {
  Window window{};
  base::Optional<Pixmap> background_pixmap{};
  base::Optional<uint32_t> background_pixel{};
  base::Optional<Pixmap> border_pixmap{};
  base::Optional<uint32_t> border_pixel{};
  base::Optional<Gravity> bit_gravity{};
  base::Optional<Gravity> win_gravity{};
  base::Optional<BackingStore> backing_store{};
  base::Optional<uint32_t> backing_planes{};
  base::Optional<uint32_t> backing_pixel{};
  base::Optional<Bool32> override_redirect{};
  base::Optional<Bool32> save_under{};
  base::Optional<EventMask> event_mask{};
  base::Optional<EventMask> do_not_propogate_mask{};
  base::Optional<ColorMap> colormap{};
  base::Optional<Cursor> cursor{};
};

using ChangeWindowAttributesResponse = Response<void>;

struct GetWindowAttributesRequest {
  Window window{};
};

struct GetWindowAttributesReply {
  BackingStore backing_store{};
  uint16_t sequence{};
  VisualId visual{};
  WindowClass c_class{};
  Gravity bit_gravity{};
  Gravity win_gravity{};
  uint32_t backing_planes{};
  uint32_t backing_pixel{};
  uint8_t save_under{};
  uint8_t map_is_installed{};
  MapState map_state{};
  uint8_t override_redirect{};
  ColorMap colormap{};
  EventMask all_event_masks{};
  EventMask your_event_mask{};
  EventMask do_not_propagate_mask{};
};

using GetWindowAttributesResponse = Response<GetWindowAttributesReply>;

struct DestroyWindowRequest {
  Window window{};
};

using DestroyWindowResponse = Response<void>;

struct DestroySubwindowsRequest {
  Window window{};
};

using DestroySubwindowsResponse = Response<void>;

struct ChangeSaveSetRequest {
  SetMode mode{};
  Window window{};
};

using ChangeSaveSetResponse = Response<void>;

struct ReparentWindowRequest {
  Window window{};
  Window parent{};
  int16_t x{};
  int16_t y{};
};

using ReparentWindowResponse = Response<void>;

struct MapWindowRequest {
  Window window{};
};

using MapWindowResponse = Response<void>;

struct MapSubwindowsRequest {
  Window window{};
};

using MapSubwindowsResponse = Response<void>;

struct UnmapWindowRequest {
  Window window{};
};

using UnmapWindowResponse = Response<void>;

struct UnmapSubwindowsRequest {
  Window window{};
};

using UnmapSubwindowsResponse = Response<void>;

struct ConfigureWindowRequest {
  Window window{};
  base::Optional<int32_t> x{};
  base::Optional<int32_t> y{};
  base::Optional<uint32_t> width{};
  base::Optional<uint32_t> height{};
  base::Optional<uint32_t> border_width{};
  base::Optional<Window> sibling{};
  base::Optional<StackMode> stack_mode{};
};

using ConfigureWindowResponse = Response<void>;

struct CirculateWindowRequest {
  Circulate direction{};
  Window window{};
};

using CirculateWindowResponse = Response<void>;

struct GetGeometryRequest {
  Drawable drawable{};
};

struct GetGeometryReply {
  uint8_t depth{};
  uint16_t sequence{};
  Window root{};
  int16_t x{};
  int16_t y{};
  uint16_t width{};
  uint16_t height{};
  uint16_t border_width{};
};

using GetGeometryResponse = Response<GetGeometryReply>;

struct QueryTreeRequest {
  Window window{};
};

struct QueryTreeReply {
  uint16_t sequence{};
  Window root{};
  Window parent{};
  std::vector<Window> children{};
};

using QueryTreeResponse = Response<QueryTreeReply>;

struct InternAtomRequest {
  uint8_t only_if_exists{};
  std::string name{};
};

struct InternAtomReply {
  uint16_t sequence{};
  Atom atom{};
};

using InternAtomResponse = Response<InternAtomReply>;

struct GetAtomNameRequest {
  Atom atom{};
};

struct GetAtomNameReply {
  uint16_t sequence{};
  std::string name{};
};

using GetAtomNameResponse = Response<GetAtomNameReply>;

struct ChangePropertyRequest {
  PropMode mode{};
  Window window{};
  Atom property{};
  Atom type{};
  uint8_t format{};
  uint32_t data_len{};
  scoped_refptr<base::RefCountedMemory> data{};
};

using ChangePropertyResponse = Response<void>;

struct DeletePropertyRequest {
  Window window{};
  Atom property{};
};

using DeletePropertyResponse = Response<void>;

struct GetPropertyRequest {
  uint8_t c_delete{};
  Window window{};
  Atom property{};
  Atom type{};
  uint32_t long_offset{};
  uint32_t long_length{};
};

struct GetPropertyReply {
  uint8_t format{};
  uint16_t sequence{};
  Atom type{};
  uint32_t bytes_after{};
  uint32_t value_len{};
  scoped_refptr<base::RefCountedMemory> value{};
};

using GetPropertyResponse = Response<GetPropertyReply>;

struct ListPropertiesRequest {
  Window window{};
};

struct ListPropertiesReply {
  uint16_t sequence{};
  std::vector<Atom> atoms{};
};

using ListPropertiesResponse = Response<ListPropertiesReply>;

struct SetSelectionOwnerRequest {
  Window owner{};
  Atom selection{};
  Time time{};
};

using SetSelectionOwnerResponse = Response<void>;

struct GetSelectionOwnerRequest {
  Atom selection{};
};

struct GetSelectionOwnerReply {
  uint16_t sequence{};
  Window owner{};
};

using GetSelectionOwnerResponse = Response<GetSelectionOwnerReply>;

struct ConvertSelectionRequest {
  Window requestor{};
  Atom selection{};
  Atom target{};
  Atom property{};
  Time time{};
};

using ConvertSelectionResponse = Response<void>;

struct SendEventRequest {
  uint8_t propagate{};
  Window destination{};
  EventMask event_mask{};
  std::array<char, 32> event{};
};

using SendEventResponse = Response<void>;

struct GrabPointerRequest {
  uint8_t owner_events{};
  Window grab_window{};
  EventMask event_mask{};
  GrabMode pointer_mode{};
  GrabMode keyboard_mode{};
  Window confine_to{};
  Cursor cursor{};
  Time time{};
};

struct GrabPointerReply {
  GrabStatus status{};
  uint16_t sequence{};
};

using GrabPointerResponse = Response<GrabPointerReply>;

struct UngrabPointerRequest {
  Time time{};
};

using UngrabPointerResponse = Response<void>;

struct GrabButtonRequest {
  uint8_t owner_events{};
  Window grab_window{};
  EventMask event_mask{};
  GrabMode pointer_mode{};
  GrabMode keyboard_mode{};
  Window confine_to{};
  Cursor cursor{};
  ButtonIndex button{};
  ModMask modifiers{};
};

using GrabButtonResponse = Response<void>;

struct UngrabButtonRequest {
  ButtonIndex button{};
  Window grab_window{};
  ModMask modifiers{};
};

using UngrabButtonResponse = Response<void>;

struct ChangeActivePointerGrabRequest {
  Cursor cursor{};
  Time time{};
  EventMask event_mask{};
};

using ChangeActivePointerGrabResponse = Response<void>;

struct GrabKeyboardRequest {
  uint8_t owner_events{};
  Window grab_window{};
  Time time{};
  GrabMode pointer_mode{};
  GrabMode keyboard_mode{};
};

struct GrabKeyboardReply {
  GrabStatus status{};
  uint16_t sequence{};
};

using GrabKeyboardResponse = Response<GrabKeyboardReply>;

struct UngrabKeyboardRequest {
  Time time{};
};

using UngrabKeyboardResponse = Response<void>;

struct GrabKeyRequest {
  uint8_t owner_events{};
  Window grab_window{};
  ModMask modifiers{};
  KeyCode key{};
  GrabMode pointer_mode{};
  GrabMode keyboard_mode{};
};

using GrabKeyResponse = Response<void>;

struct UngrabKeyRequest {
  KeyCode key{};
  Window grab_window{};
  ModMask modifiers{};
};

using UngrabKeyResponse = Response<void>;

struct AllowEventsRequest {
  Allow mode{};
  Time time{};
};

using AllowEventsResponse = Response<void>;

struct GrabServerRequest {
};

using GrabServerResponse = Response<void>;

struct UngrabServerRequest {
};

using UngrabServerResponse = Response<void>;

struct QueryPointerRequest {
  Window window{};
};

struct QueryPointerReply {
  uint8_t same_screen{};
  uint16_t sequence{};
  Window root{};
  Window child{};
  int16_t root_x{};
  int16_t root_y{};
  int16_t win_x{};
  int16_t win_y{};
  KeyButMask mask{};
};

using QueryPointerResponse = Response<QueryPointerReply>;

struct GetMotionEventsRequest {
  Window window{};
  Time start{};
  Time stop{};
};

struct GetMotionEventsReply {
  uint16_t sequence{};
  std::vector<TimeCoord> events{};
};

using GetMotionEventsResponse = Response<GetMotionEventsReply>;

struct TranslateCoordinatesRequest {
  Window src_window{};
  Window dst_window{};
  int16_t src_x{};
  int16_t src_y{};
};

struct TranslateCoordinatesReply {
  uint8_t same_screen{};
  uint16_t sequence{};
  Window child{};
  int16_t dst_x{};
  int16_t dst_y{};
};

using TranslateCoordinatesResponse = Response<TranslateCoordinatesReply>;

struct WarpPointerRequest {
  Window src_window{};
  Window dst_window{};
  int16_t src_x{};
  int16_t src_y{};
  uint16_t src_width{};
  uint16_t src_height{};
  int16_t dst_x{};
  int16_t dst_y{};
};

using WarpPointerResponse = Response<void>;

struct SetInputFocusRequest {
  InputFocus revert_to{};
  Window focus{};
  Time time{};
};

using SetInputFocusResponse = Response<void>;

struct GetInputFocusRequest {
};

struct GetInputFocusReply {
  InputFocus revert_to{};
  uint16_t sequence{};
  Window focus{};
};

using GetInputFocusResponse = Response<GetInputFocusReply>;

struct QueryKeymapRequest {
};

struct QueryKeymapReply {
  uint16_t sequence{};
  std::array<uint8_t, 32> keys{};
};

using QueryKeymapResponse = Response<QueryKeymapReply>;

struct OpenFontRequest {
  Font fid{};
  std::string name{};
};

using OpenFontResponse = Response<void>;

struct CloseFontRequest {
  Font font{};
};

using CloseFontResponse = Response<void>;

struct QueryFontRequest {
  Fontable font{};
};

struct QueryFontReply {
  uint16_t sequence{};
  CharInfo min_bounds{};
  CharInfo max_bounds{};
  uint16_t min_char_or_byte2{};
  uint16_t max_char_or_byte2{};
  uint16_t default_char{};
  FontDraw draw_direction{};
  uint8_t min_byte1{};
  uint8_t max_byte1{};
  uint8_t all_chars_exist{};
  int16_t font_ascent{};
  int16_t font_descent{};
  std::vector<FontProperty> properties{};
  std::vector<CharInfo> char_infos{};
};

using QueryFontResponse = Response<QueryFontReply>;

struct QueryTextExtentsRequest {
  Fontable font{};
  std::vector<Char16> string{};
};

struct QueryTextExtentsReply {
  FontDraw draw_direction{};
  uint16_t sequence{};
  int16_t font_ascent{};
  int16_t font_descent{};
  int16_t overall_ascent{};
  int16_t overall_descent{};
  int32_t overall_width{};
  int32_t overall_left{};
  int32_t overall_right{};
};

using QueryTextExtentsResponse = Response<QueryTextExtentsReply>;

struct ListFontsRequest {
  uint16_t max_names{};
  std::string pattern{};
};

struct ListFontsReply {
  uint16_t sequence{};
  std::vector<Str> names{};
};

using ListFontsResponse = Response<ListFontsReply>;

struct ListFontsWithInfoRequest {
  uint16_t max_names{};
  std::string pattern{};
};

struct ListFontsWithInfoReply {
  uint16_t sequence{};
  CharInfo min_bounds{};
  CharInfo max_bounds{};
  uint16_t min_char_or_byte2{};
  uint16_t max_char_or_byte2{};
  uint16_t default_char{};
  FontDraw draw_direction{};
  uint8_t min_byte1{};
  uint8_t max_byte1{};
  uint8_t all_chars_exist{};
  int16_t font_ascent{};
  int16_t font_descent{};
  uint32_t replies_hint{};
  std::vector<FontProperty> properties{};
  std::string name{};
};

using ListFontsWithInfoResponse = Response<ListFontsWithInfoReply>;

struct SetFontPathRequest {
  std::vector<Str> font{};
};

using SetFontPathResponse = Response<void>;

struct GetFontPathRequest {
};

struct GetFontPathReply {
  uint16_t sequence{};
  std::vector<Str> path{};
};

using GetFontPathResponse = Response<GetFontPathReply>;

struct CreatePixmapRequest {
  uint8_t depth{};
  Pixmap pid{};
  Drawable drawable{};
  uint16_t width{};
  uint16_t height{};
};

using CreatePixmapResponse = Response<void>;

struct FreePixmapRequest {
  Pixmap pixmap{};
};

using FreePixmapResponse = Response<void>;

struct CreateGCRequest {
  GraphicsContext cid{};
  Drawable drawable{};
  base::Optional<Gx> function{};
  base::Optional<uint32_t> plane_mask{};
  base::Optional<uint32_t> foreground{};
  base::Optional<uint32_t> background{};
  base::Optional<uint32_t> line_width{};
  base::Optional<LineStyle> line_style{};
  base::Optional<CapStyle> cap_style{};
  base::Optional<JoinStyle> join_style{};
  base::Optional<FillStyle> fill_style{};
  base::Optional<FillRule> fill_rule{};
  base::Optional<Pixmap> tile{};
  base::Optional<Pixmap> stipple{};
  base::Optional<int32_t> tile_stipple_x_origin{};
  base::Optional<int32_t> tile_stipple_y_origin{};
  base::Optional<Font> font{};
  base::Optional<SubwindowMode> subwindow_mode{};
  base::Optional<Bool32> graphics_exposures{};
  base::Optional<int32_t> clip_x_origin{};
  base::Optional<int32_t> clip_y_origin{};
  base::Optional<Pixmap> clip_mask{};
  base::Optional<uint32_t> dash_offset{};
  base::Optional<uint32_t> dashes{};
  base::Optional<ArcMode> arc_mode{};
};

using CreateGCResponse = Response<void>;

struct ChangeGCRequest {
  GraphicsContext gc{};
  base::Optional<Gx> function{};
  base::Optional<uint32_t> plane_mask{};
  base::Optional<uint32_t> foreground{};
  base::Optional<uint32_t> background{};
  base::Optional<uint32_t> line_width{};
  base::Optional<LineStyle> line_style{};
  base::Optional<CapStyle> cap_style{};
  base::Optional<JoinStyle> join_style{};
  base::Optional<FillStyle> fill_style{};
  base::Optional<FillRule> fill_rule{};
  base::Optional<Pixmap> tile{};
  base::Optional<Pixmap> stipple{};
  base::Optional<int32_t> tile_stipple_x_origin{};
  base::Optional<int32_t> tile_stipple_y_origin{};
  base::Optional<Font> font{};
  base::Optional<SubwindowMode> subwindow_mode{};
  base::Optional<Bool32> graphics_exposures{};
  base::Optional<int32_t> clip_x_origin{};
  base::Optional<int32_t> clip_y_origin{};
  base::Optional<Pixmap> clip_mask{};
  base::Optional<uint32_t> dash_offset{};
  base::Optional<uint32_t> dashes{};
  base::Optional<ArcMode> arc_mode{};
};

using ChangeGCResponse = Response<void>;

struct CopyGCRequest {
  GraphicsContext src_gc{};
  GraphicsContext dst_gc{};
  GraphicsContextAttribute value_mask{};
};

using CopyGCResponse = Response<void>;

struct SetDashesRequest {
  GraphicsContext gc{};
  uint16_t dash_offset{};
  std::vector<uint8_t> dashes{};
};

using SetDashesResponse = Response<void>;

struct SetClipRectanglesRequest {
  ClipOrdering ordering{};
  GraphicsContext gc{};
  int16_t clip_x_origin{};
  int16_t clip_y_origin{};
  std::vector<Rectangle> rectangles{};
};

using SetClipRectanglesResponse = Response<void>;

struct FreeGCRequest {
  GraphicsContext gc{};
};

using FreeGCResponse = Response<void>;

struct ClearAreaRequest {
  uint8_t exposures{};
  Window window{};
  int16_t x{};
  int16_t y{};
  uint16_t width{};
  uint16_t height{};
};

using ClearAreaResponse = Response<void>;

struct CopyAreaRequest {
  Drawable src_drawable{};
  Drawable dst_drawable{};
  GraphicsContext gc{};
  int16_t src_x{};
  int16_t src_y{};
  int16_t dst_x{};
  int16_t dst_y{};
  uint16_t width{};
  uint16_t height{};
};

using CopyAreaResponse = Response<void>;

struct CopyPlaneRequest {
  Drawable src_drawable{};
  Drawable dst_drawable{};
  GraphicsContext gc{};
  int16_t src_x{};
  int16_t src_y{};
  int16_t dst_x{};
  int16_t dst_y{};
  uint16_t width{};
  uint16_t height{};
  uint32_t bit_plane{};
};

using CopyPlaneResponse = Response<void>;

struct PolyPointRequest {
  CoordMode coordinate_mode{};
  Drawable drawable{};
  GraphicsContext gc{};
  std::vector<Point> points{};
};

using PolyPointResponse = Response<void>;

struct PolyLineRequest {
  CoordMode coordinate_mode{};
  Drawable drawable{};
  GraphicsContext gc{};
  std::vector<Point> points{};
};

using PolyLineResponse = Response<void>;

struct PolySegmentRequest {
  Drawable drawable{};
  GraphicsContext gc{};
  std::vector<Segment> segments{};
};

using PolySegmentResponse = Response<void>;

struct PolyRectangleRequest {
  Drawable drawable{};
  GraphicsContext gc{};
  std::vector<Rectangle> rectangles{};
};

using PolyRectangleResponse = Response<void>;

struct PolyArcRequest {
  Drawable drawable{};
  GraphicsContext gc{};
  std::vector<Arc> arcs{};
};

using PolyArcResponse = Response<void>;

struct FillPolyRequest {
  Drawable drawable{};
  GraphicsContext gc{};
  PolyShape shape{};
  CoordMode coordinate_mode{};
  std::vector<Point> points{};
};

using FillPolyResponse = Response<void>;

struct PolyFillRectangleRequest {
  Drawable drawable{};
  GraphicsContext gc{};
  std::vector<Rectangle> rectangles{};
};

using PolyFillRectangleResponse = Response<void>;

struct PolyFillArcRequest {
  Drawable drawable{};
  GraphicsContext gc{};
  std::vector<Arc> arcs{};
};

using PolyFillArcResponse = Response<void>;

struct PutImageRequest {
  ImageFormat format{};
  Drawable drawable{};
  GraphicsContext gc{};
  uint16_t width{};
  uint16_t height{};
  int16_t dst_x{};
  int16_t dst_y{};
  uint8_t left_pad{};
  uint8_t depth{};
  scoped_refptr<base::RefCountedMemory> data{};
};

using PutImageResponse = Response<void>;

struct GetImageRequest {
  ImageFormat format{};
  Drawable drawable{};
  int16_t x{};
  int16_t y{};
  uint16_t width{};
  uint16_t height{};
  uint32_t plane_mask{};
};

struct GetImageReply {
  uint8_t depth{};
  uint16_t sequence{};
  VisualId visual{};
  scoped_refptr<base::RefCountedMemory> data{};
};

using GetImageResponse = Response<GetImageReply>;

struct PolyText8Request {
  Drawable drawable{};
  GraphicsContext gc{};
  int16_t x{};
  int16_t y{};
  std::vector<uint8_t> items{};
};

using PolyText8Response = Response<void>;

struct PolyText16Request {
  Drawable drawable{};
  GraphicsContext gc{};
  int16_t x{};
  int16_t y{};
  std::vector<uint8_t> items{};
};

using PolyText16Response = Response<void>;

struct ImageText8Request {
  Drawable drawable{};
  GraphicsContext gc{};
  int16_t x{};
  int16_t y{};
  std::string string{};
};

using ImageText8Response = Response<void>;

struct ImageText16Request {
  Drawable drawable{};
  GraphicsContext gc{};
  int16_t x{};
  int16_t y{};
  std::vector<Char16> string{};
};

using ImageText16Response = Response<void>;

struct CreateColormapRequest {
  ColormapAlloc alloc{};
  ColorMap mid{};
  Window window{};
  VisualId visual{};
};

using CreateColormapResponse = Response<void>;

struct FreeColormapRequest {
  ColorMap cmap{};
};

using FreeColormapResponse = Response<void>;

struct CopyColormapAndFreeRequest {
  ColorMap mid{};
  ColorMap src_cmap{};
};

using CopyColormapAndFreeResponse = Response<void>;

struct InstallColormapRequest {
  ColorMap cmap{};
};

using InstallColormapResponse = Response<void>;

struct UninstallColormapRequest {
  ColorMap cmap{};
};

using UninstallColormapResponse = Response<void>;

struct ListInstalledColormapsRequest {
  Window window{};
};

struct ListInstalledColormapsReply {
  uint16_t sequence{};
  std::vector<ColorMap> cmaps{};
};

using ListInstalledColormapsResponse = Response<ListInstalledColormapsReply>;

struct AllocColorRequest {
  ColorMap cmap{};
  uint16_t red{};
  uint16_t green{};
  uint16_t blue{};
};

struct AllocColorReply {
  uint16_t sequence{};
  uint16_t red{};
  uint16_t green{};
  uint16_t blue{};
  uint32_t pixel{};
};

using AllocColorResponse = Response<AllocColorReply>;

struct AllocNamedColorRequest {
  ColorMap cmap{};
  std::string name{};
};

struct AllocNamedColorReply {
  uint16_t sequence{};
  uint32_t pixel{};
  uint16_t exact_red{};
  uint16_t exact_green{};
  uint16_t exact_blue{};
  uint16_t visual_red{};
  uint16_t visual_green{};
  uint16_t visual_blue{};
};

using AllocNamedColorResponse = Response<AllocNamedColorReply>;

struct AllocColorCellsRequest {
  uint8_t contiguous{};
  ColorMap cmap{};
  uint16_t colors{};
  uint16_t planes{};
};

struct AllocColorCellsReply {
  uint16_t sequence{};
  std::vector<uint32_t> pixels{};
  std::vector<uint32_t> masks{};
};

using AllocColorCellsResponse = Response<AllocColorCellsReply>;

struct AllocColorPlanesRequest {
  uint8_t contiguous{};
  ColorMap cmap{};
  uint16_t colors{};
  uint16_t reds{};
  uint16_t greens{};
  uint16_t blues{};
};

struct AllocColorPlanesReply {
  uint16_t sequence{};
  uint32_t red_mask{};
  uint32_t green_mask{};
  uint32_t blue_mask{};
  std::vector<uint32_t> pixels{};
};

using AllocColorPlanesResponse = Response<AllocColorPlanesReply>;

struct FreeColorsRequest {
  ColorMap cmap{};
  uint32_t plane_mask{};
  std::vector<uint32_t> pixels{};
};

using FreeColorsResponse = Response<void>;

struct StoreColorsRequest {
  ColorMap cmap{};
  std::vector<ColorItem> items{};
};

using StoreColorsResponse = Response<void>;

struct StoreNamedColorRequest {
  ColorFlag flags{};
  ColorMap cmap{};
  uint32_t pixel{};
  std::string name{};
};

using StoreNamedColorResponse = Response<void>;

struct QueryColorsRequest {
  ColorMap cmap{};
  std::vector<uint32_t> pixels{};
};

struct QueryColorsReply {
  uint16_t sequence{};
  std::vector<Rgb> colors{};
};

using QueryColorsResponse = Response<QueryColorsReply>;

struct LookupColorRequest {
  ColorMap cmap{};
  std::string name{};
};

struct LookupColorReply {
  uint16_t sequence{};
  uint16_t exact_red{};
  uint16_t exact_green{};
  uint16_t exact_blue{};
  uint16_t visual_red{};
  uint16_t visual_green{};
  uint16_t visual_blue{};
};

using LookupColorResponse = Response<LookupColorReply>;

struct CreateCursorRequest {
  Cursor cid{};
  Pixmap source{};
  Pixmap mask{};
  uint16_t fore_red{};
  uint16_t fore_green{};
  uint16_t fore_blue{};
  uint16_t back_red{};
  uint16_t back_green{};
  uint16_t back_blue{};
  uint16_t x{};
  uint16_t y{};
};

using CreateCursorResponse = Response<void>;

struct CreateGlyphCursorRequest {
  Cursor cid{};
  Font source_font{};
  Font mask_font{};
  uint16_t source_char{};
  uint16_t mask_char{};
  uint16_t fore_red{};
  uint16_t fore_green{};
  uint16_t fore_blue{};
  uint16_t back_red{};
  uint16_t back_green{};
  uint16_t back_blue{};
};

using CreateGlyphCursorResponse = Response<void>;

struct FreeCursorRequest {
  Cursor cursor{};
};

using FreeCursorResponse = Response<void>;

struct RecolorCursorRequest {
  Cursor cursor{};
  uint16_t fore_red{};
  uint16_t fore_green{};
  uint16_t fore_blue{};
  uint16_t back_red{};
  uint16_t back_green{};
  uint16_t back_blue{};
};

using RecolorCursorResponse = Response<void>;

struct QueryBestSizeRequest {
  QueryShapeOf c_class{};
  Drawable drawable{};
  uint16_t width{};
  uint16_t height{};
};

struct QueryBestSizeReply {
  uint16_t sequence{};
  uint16_t width{};
  uint16_t height{};
};

using QueryBestSizeResponse = Response<QueryBestSizeReply>;

struct QueryExtensionRequest {
  std::string name{};
};

struct QueryExtensionReply {
  uint16_t sequence{};
  uint8_t present{};
  uint8_t major_opcode{};
  uint8_t first_event{};
  uint8_t first_error{};
};

using QueryExtensionResponse = Response<QueryExtensionReply>;

struct ListExtensionsRequest {
};

struct ListExtensionsReply {
  uint16_t sequence{};
  std::vector<Str> names{};
};

using ListExtensionsResponse = Response<ListExtensionsReply>;

struct ChangeKeyboardMappingRequest {
  uint8_t keycode_count{};
  KeyCode first_keycode{};
  uint8_t keysyms_per_keycode{};
  std::vector<KeySym> keysyms{};
};

using ChangeKeyboardMappingResponse = Response<void>;

struct GetKeyboardMappingRequest {
  KeyCode first_keycode{};
  uint8_t count{};
};

struct GetKeyboardMappingReply {
  uint8_t keysyms_per_keycode{};
  uint16_t sequence{};
  std::vector<KeySym> keysyms{};
};

using GetKeyboardMappingResponse = Response<GetKeyboardMappingReply>;

struct ChangeKeyboardControlRequest {
  base::Optional<int32_t> key_click_percent{};
  base::Optional<int32_t> bell_percent{};
  base::Optional<int32_t> bell_pitch{};
  base::Optional<int32_t> bell_duration{};
  base::Optional<uint32_t> led{};
  base::Optional<LedMode> led_mode{};
  base::Optional<KeyCode32> key{};
  base::Optional<AutoRepeatMode> auto_repeat_mode{};
};

using ChangeKeyboardControlResponse = Response<void>;

struct GetKeyboardControlRequest {
};

struct GetKeyboardControlReply {
  AutoRepeatMode global_auto_repeat{};
  uint16_t sequence{};
  uint32_t led_mask{};
  uint8_t key_click_percent{};
  uint8_t bell_percent{};
  uint16_t bell_pitch{};
  uint16_t bell_duration{};
  std::array<uint8_t, 32> auto_repeats{};
};

using GetKeyboardControlResponse = Response<GetKeyboardControlReply>;

struct BellRequest {
  int8_t percent{};
};

using BellResponse = Response<void>;

struct ChangePointerControlRequest {
  int16_t acceleration_numerator{};
  int16_t acceleration_denominator{};
  int16_t threshold{};
  uint8_t do_acceleration{};
  uint8_t do_threshold{};
};

using ChangePointerControlResponse = Response<void>;

struct GetPointerControlRequest {
};

struct GetPointerControlReply {
  uint16_t sequence{};
  uint16_t acceleration_numerator{};
  uint16_t acceleration_denominator{};
  uint16_t threshold{};
};

using GetPointerControlResponse = Response<GetPointerControlReply>;

struct SetScreenSaverRequest {
  int16_t timeout{};
  int16_t interval{};
  Blanking prefer_blanking{};
  Exposures allow_exposures{};
};

using SetScreenSaverResponse = Response<void>;

struct GetScreenSaverRequest {
};

struct GetScreenSaverReply {
  uint16_t sequence{};
  uint16_t timeout{};
  uint16_t interval{};
  Blanking prefer_blanking{};
  Exposures allow_exposures{};
};

using GetScreenSaverResponse = Response<GetScreenSaverReply>;

struct ChangeHostsRequest {
  HostMode mode{};
  Family family{};
  std::vector<uint8_t> address{};
};

using ChangeHostsResponse = Response<void>;

struct ListHostsRequest {
};

struct ListHostsReply {
  AccessControl mode{};
  uint16_t sequence{};
  std::vector<Host> hosts{};
};

using ListHostsResponse = Response<ListHostsReply>;

struct SetAccessControlRequest {
  AccessControl mode{};
};

using SetAccessControlResponse = Response<void>;

struct SetCloseDownModeRequest {
  CloseDown mode{};
};

using SetCloseDownModeResponse = Response<void>;

struct KillClientRequest {
  uint32_t resource{};
};

using KillClientResponse = Response<void>;

struct RotatePropertiesRequest {
  Window window{};
  int16_t delta{};
  std::vector<Atom> atoms{};
};

using RotatePropertiesResponse = Response<void>;

struct ForceScreenSaverRequest {
  ScreenSaverMode mode{};
};

using ForceScreenSaverResponse = Response<void>;

struct SetPointerMappingRequest {
  std::vector<uint8_t> map{};
};

struct SetPointerMappingReply {
  MappingStatus status{};
  uint16_t sequence{};
};

using SetPointerMappingResponse = Response<SetPointerMappingReply>;

struct GetPointerMappingRequest {
};

struct GetPointerMappingReply {
  uint16_t sequence{};
  std::vector<uint8_t> map{};
};

using GetPointerMappingResponse = Response<GetPointerMappingReply>;

struct SetModifierMappingRequest {
  uint8_t keycodes_per_modifier{};
  std::vector<KeyCode> keycodes{};
};

struct SetModifierMappingReply {
  MappingStatus status{};
  uint16_t sequence{};
};

using SetModifierMappingResponse = Response<SetModifierMappingReply>;

struct GetModifierMappingRequest {
};

struct GetModifierMappingReply {
  uint8_t keycodes_per_modifier{};
  uint16_t sequence{};
  std::vector<KeyCode> keycodes{};
};

using GetModifierMappingResponse = Response<GetModifierMappingReply>;

struct NoOperationRequest {
};

using NoOperationResponse = Response<void>;

class COMPONENT_EXPORT(X11) XProto {
  public:
  explicit XProto(Connection* connection);

  Connection* connection() const { return connection_; }

  Future<void> CreateWindow(
      const CreateWindowRequest& request);

  Future<void> ChangeWindowAttributes(
      const ChangeWindowAttributesRequest& request);

  Future<GetWindowAttributesReply> GetWindowAttributes(
      const GetWindowAttributesRequest& request);

  Future<void> DestroyWindow(
      const DestroyWindowRequest& request);

  Future<void> DestroySubwindows(
      const DestroySubwindowsRequest& request);

  Future<void> ChangeSaveSet(
      const ChangeSaveSetRequest& request);

  Future<void> ReparentWindow(
      const ReparentWindowRequest& request);

  Future<void> MapWindow(
      const MapWindowRequest& request);

  Future<void> MapSubwindows(
      const MapSubwindowsRequest& request);

  Future<void> UnmapWindow(
      const UnmapWindowRequest& request);

  Future<void> UnmapSubwindows(
      const UnmapSubwindowsRequest& request);

  Future<void> ConfigureWindow(
      const ConfigureWindowRequest& request);

  Future<void> CirculateWindow(
      const CirculateWindowRequest& request);

  Future<GetGeometryReply> GetGeometry(
      const GetGeometryRequest& request);

  Future<QueryTreeReply> QueryTree(
      const QueryTreeRequest& request);

  Future<InternAtomReply> InternAtom(
      const InternAtomRequest& request);

  Future<GetAtomNameReply> GetAtomName(
      const GetAtomNameRequest& request);

  Future<void> ChangeProperty(
      const ChangePropertyRequest& request);

  Future<void> DeleteProperty(
      const DeletePropertyRequest& request);

  Future<GetPropertyReply> GetProperty(
      const GetPropertyRequest& request);

  Future<ListPropertiesReply> ListProperties(
      const ListPropertiesRequest& request);

  Future<void> SetSelectionOwner(
      const SetSelectionOwnerRequest& request);

  Future<GetSelectionOwnerReply> GetSelectionOwner(
      const GetSelectionOwnerRequest& request);

  Future<void> ConvertSelection(
      const ConvertSelectionRequest& request);

  Future<void> SendEvent(
      const SendEventRequest& request);

  Future<GrabPointerReply> GrabPointer(
      const GrabPointerRequest& request);

  Future<void> UngrabPointer(
      const UngrabPointerRequest& request);

  Future<void> GrabButton(
      const GrabButtonRequest& request);

  Future<void> UngrabButton(
      const UngrabButtonRequest& request);

  Future<void> ChangeActivePointerGrab(
      const ChangeActivePointerGrabRequest& request);

  Future<GrabKeyboardReply> GrabKeyboard(
      const GrabKeyboardRequest& request);

  Future<void> UngrabKeyboard(
      const UngrabKeyboardRequest& request);

  Future<void> GrabKey(
      const GrabKeyRequest& request);

  Future<void> UngrabKey(
      const UngrabKeyRequest& request);

  Future<void> AllowEvents(
      const AllowEventsRequest& request);

  Future<void> GrabServer(
      const GrabServerRequest& request);

  Future<void> UngrabServer(
      const UngrabServerRequest& request);

  Future<QueryPointerReply> QueryPointer(
      const QueryPointerRequest& request);

  Future<GetMotionEventsReply> GetMotionEvents(
      const GetMotionEventsRequest& request);

  Future<TranslateCoordinatesReply> TranslateCoordinates(
      const TranslateCoordinatesRequest& request);

  Future<void> WarpPointer(
      const WarpPointerRequest& request);

  Future<void> SetInputFocus(
      const SetInputFocusRequest& request);

  Future<GetInputFocusReply> GetInputFocus(
      const GetInputFocusRequest& request);

  Future<QueryKeymapReply> QueryKeymap(
      const QueryKeymapRequest& request);

  Future<void> OpenFont(
      const OpenFontRequest& request);

  Future<void> CloseFont(
      const CloseFontRequest& request);

  Future<QueryFontReply> QueryFont(
      const QueryFontRequest& request);

  Future<QueryTextExtentsReply> QueryTextExtents(
      const QueryTextExtentsRequest& request);

  Future<ListFontsReply> ListFonts(
      const ListFontsRequest& request);

  Future<ListFontsWithInfoReply> ListFontsWithInfo(
      const ListFontsWithInfoRequest& request);

  Future<void> SetFontPath(
      const SetFontPathRequest& request);

  Future<GetFontPathReply> GetFontPath(
      const GetFontPathRequest& request);

  Future<void> CreatePixmap(
      const CreatePixmapRequest& request);

  Future<void> FreePixmap(
      const FreePixmapRequest& request);

  Future<void> CreateGC(
      const CreateGCRequest& request);

  Future<void> ChangeGC(
      const ChangeGCRequest& request);

  Future<void> CopyGC(
      const CopyGCRequest& request);

  Future<void> SetDashes(
      const SetDashesRequest& request);

  Future<void> SetClipRectangles(
      const SetClipRectanglesRequest& request);

  Future<void> FreeGC(
      const FreeGCRequest& request);

  Future<void> ClearArea(
      const ClearAreaRequest& request);

  Future<void> CopyArea(
      const CopyAreaRequest& request);

  Future<void> CopyPlane(
      const CopyPlaneRequest& request);

  Future<void> PolyPoint(
      const PolyPointRequest& request);

  Future<void> PolyLine(
      const PolyLineRequest& request);

  Future<void> PolySegment(
      const PolySegmentRequest& request);

  Future<void> PolyRectangle(
      const PolyRectangleRequest& request);

  Future<void> PolyArc(
      const PolyArcRequest& request);

  Future<void> FillPoly(
      const FillPolyRequest& request);

  Future<void> PolyFillRectangle(
      const PolyFillRectangleRequest& request);

  Future<void> PolyFillArc(
      const PolyFillArcRequest& request);

  Future<void> PutImage(
      const PutImageRequest& request);

  Future<GetImageReply> GetImage(
      const GetImageRequest& request);

  Future<void> PolyText8(
      const PolyText8Request& request);

  Future<void> PolyText16(
      const PolyText16Request& request);

  Future<void> ImageText8(
      const ImageText8Request& request);

  Future<void> ImageText16(
      const ImageText16Request& request);

  Future<void> CreateColormap(
      const CreateColormapRequest& request);

  Future<void> FreeColormap(
      const FreeColormapRequest& request);

  Future<void> CopyColormapAndFree(
      const CopyColormapAndFreeRequest& request);

  Future<void> InstallColormap(
      const InstallColormapRequest& request);

  Future<void> UninstallColormap(
      const UninstallColormapRequest& request);

  Future<ListInstalledColormapsReply> ListInstalledColormaps(
      const ListInstalledColormapsRequest& request);

  Future<AllocColorReply> AllocColor(
      const AllocColorRequest& request);

  Future<AllocNamedColorReply> AllocNamedColor(
      const AllocNamedColorRequest& request);

  Future<AllocColorCellsReply> AllocColorCells(
      const AllocColorCellsRequest& request);

  Future<AllocColorPlanesReply> AllocColorPlanes(
      const AllocColorPlanesRequest& request);

  Future<void> FreeColors(
      const FreeColorsRequest& request);

  Future<void> StoreColors(
      const StoreColorsRequest& request);

  Future<void> StoreNamedColor(
      const StoreNamedColorRequest& request);

  Future<QueryColorsReply> QueryColors(
      const QueryColorsRequest& request);

  Future<LookupColorReply> LookupColor(
      const LookupColorRequest& request);

  Future<void> CreateCursor(
      const CreateCursorRequest& request);

  Future<void> CreateGlyphCursor(
      const CreateGlyphCursorRequest& request);

  Future<void> FreeCursor(
      const FreeCursorRequest& request);

  Future<void> RecolorCursor(
      const RecolorCursorRequest& request);

  Future<QueryBestSizeReply> QueryBestSize(
      const QueryBestSizeRequest& request);

  Future<QueryExtensionReply> QueryExtension(
      const QueryExtensionRequest& request);

  Future<ListExtensionsReply> ListExtensions(
      const ListExtensionsRequest& request);

  Future<void> ChangeKeyboardMapping(
      const ChangeKeyboardMappingRequest& request);

  Future<GetKeyboardMappingReply> GetKeyboardMapping(
      const GetKeyboardMappingRequest& request);

  Future<void> ChangeKeyboardControl(
      const ChangeKeyboardControlRequest& request);

  Future<GetKeyboardControlReply> GetKeyboardControl(
      const GetKeyboardControlRequest& request);

  Future<void> Bell(
      const BellRequest& request);

  Future<void> ChangePointerControl(
      const ChangePointerControlRequest& request);

  Future<GetPointerControlReply> GetPointerControl(
      const GetPointerControlRequest& request);

  Future<void> SetScreenSaver(
      const SetScreenSaverRequest& request);

  Future<GetScreenSaverReply> GetScreenSaver(
      const GetScreenSaverRequest& request);

  Future<void> ChangeHosts(
      const ChangeHostsRequest& request);

  Future<ListHostsReply> ListHosts(
      const ListHostsRequest& request);

  Future<void> SetAccessControl(
      const SetAccessControlRequest& request);

  Future<void> SetCloseDownMode(
      const SetCloseDownModeRequest& request);

  Future<void> KillClient(
      const KillClientRequest& request);

  Future<void> RotateProperties(
      const RotatePropertiesRequest& request);

  Future<void> ForceScreenSaver(
      const ForceScreenSaverRequest& request);

  Future<SetPointerMappingReply> SetPointerMapping(
      const SetPointerMappingRequest& request);

  Future<GetPointerMappingReply> GetPointerMapping(
      const GetPointerMappingRequest& request);

  Future<SetModifierMappingReply> SetModifierMapping(
      const SetModifierMappingRequest& request);

  Future<GetModifierMappingReply> GetModifierMapping(
      const GetModifierMappingRequest& request);

  Future<void> NoOperation(
      const NoOperationRequest& request);

  private:
  x11::Connection* const connection_;
};

}  // namespace x11

inline constexpr x11::VisualClass operator|(
    x11::VisualClass l, x11::VisualClass r) {
  using T = std::underlying_type_t<x11::VisualClass>;
  return static_cast<x11::VisualClass>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::VisualClass operator&(
    x11::VisualClass l, x11::VisualClass r) {
  using T = std::underlying_type_t<x11::VisualClass>;
  return static_cast<x11::VisualClass>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::EventMask operator|(
    x11::EventMask l, x11::EventMask r) {
  using T = std::underlying_type_t<x11::EventMask>;
  return static_cast<x11::EventMask>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::EventMask operator&(
    x11::EventMask l, x11::EventMask r) {
  using T = std::underlying_type_t<x11::EventMask>;
  return static_cast<x11::EventMask>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::BackingStore operator|(
    x11::BackingStore l, x11::BackingStore r) {
  using T = std::underlying_type_t<x11::BackingStore>;
  return static_cast<x11::BackingStore>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::BackingStore operator&(
    x11::BackingStore l, x11::BackingStore r) {
  using T = std::underlying_type_t<x11::BackingStore>;
  return static_cast<x11::BackingStore>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::ImageOrder operator|(
    x11::ImageOrder l, x11::ImageOrder r) {
  using T = std::underlying_type_t<x11::ImageOrder>;
  return static_cast<x11::ImageOrder>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::ImageOrder operator&(
    x11::ImageOrder l, x11::ImageOrder r) {
  using T = std::underlying_type_t<x11::ImageOrder>;
  return static_cast<x11::ImageOrder>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::ModMask operator|(
    x11::ModMask l, x11::ModMask r) {
  using T = std::underlying_type_t<x11::ModMask>;
  return static_cast<x11::ModMask>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::ModMask operator&(
    x11::ModMask l, x11::ModMask r) {
  using T = std::underlying_type_t<x11::ModMask>;
  return static_cast<x11::ModMask>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::KeyButMask operator|(
    x11::KeyButMask l, x11::KeyButMask r) {
  using T = std::underlying_type_t<x11::KeyButMask>;
  return static_cast<x11::KeyButMask>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::KeyButMask operator&(
    x11::KeyButMask l, x11::KeyButMask r) {
  using T = std::underlying_type_t<x11::KeyButMask>;
  return static_cast<x11::KeyButMask>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Window operator|(
    x11::Window l, x11::Window r) {
  using T = std::underlying_type_t<x11::Window>;
  return static_cast<x11::Window>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Window operator&(
    x11::Window l, x11::Window r) {
  using T = std::underlying_type_t<x11::Window>;
  return static_cast<x11::Window>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::ButtonMask operator|(
    x11::ButtonMask l, x11::ButtonMask r) {
  using T = std::underlying_type_t<x11::ButtonMask>;
  return static_cast<x11::ButtonMask>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::ButtonMask operator&(
    x11::ButtonMask l, x11::ButtonMask r) {
  using T = std::underlying_type_t<x11::ButtonMask>;
  return static_cast<x11::ButtonMask>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Motion operator|(
    x11::Motion l, x11::Motion r) {
  using T = std::underlying_type_t<x11::Motion>;
  return static_cast<x11::Motion>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Motion operator&(
    x11::Motion l, x11::Motion r) {
  using T = std::underlying_type_t<x11::Motion>;
  return static_cast<x11::Motion>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::NotifyDetail operator|(
    x11::NotifyDetail l, x11::NotifyDetail r) {
  using T = std::underlying_type_t<x11::NotifyDetail>;
  return static_cast<x11::NotifyDetail>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::NotifyDetail operator&(
    x11::NotifyDetail l, x11::NotifyDetail r) {
  using T = std::underlying_type_t<x11::NotifyDetail>;
  return static_cast<x11::NotifyDetail>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::NotifyMode operator|(
    x11::NotifyMode l, x11::NotifyMode r) {
  using T = std::underlying_type_t<x11::NotifyMode>;
  return static_cast<x11::NotifyMode>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::NotifyMode operator&(
    x11::NotifyMode l, x11::NotifyMode r) {
  using T = std::underlying_type_t<x11::NotifyMode>;
  return static_cast<x11::NotifyMode>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Visibility operator|(
    x11::Visibility l, x11::Visibility r) {
  using T = std::underlying_type_t<x11::Visibility>;
  return static_cast<x11::Visibility>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Visibility operator&(
    x11::Visibility l, x11::Visibility r) {
  using T = std::underlying_type_t<x11::Visibility>;
  return static_cast<x11::Visibility>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Place operator|(
    x11::Place l, x11::Place r) {
  using T = std::underlying_type_t<x11::Place>;
  return static_cast<x11::Place>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Place operator&(
    x11::Place l, x11::Place r) {
  using T = std::underlying_type_t<x11::Place>;
  return static_cast<x11::Place>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Property operator|(
    x11::Property l, x11::Property r) {
  using T = std::underlying_type_t<x11::Property>;
  return static_cast<x11::Property>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Property operator&(
    x11::Property l, x11::Property r) {
  using T = std::underlying_type_t<x11::Property>;
  return static_cast<x11::Property>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Time operator|(
    x11::Time l, x11::Time r) {
  using T = std::underlying_type_t<x11::Time>;
  return static_cast<x11::Time>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Time operator&(
    x11::Time l, x11::Time r) {
  using T = std::underlying_type_t<x11::Time>;
  return static_cast<x11::Time>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Atom operator|(
    x11::Atom l, x11::Atom r) {
  using T = std::underlying_type_t<x11::Atom>;
  return static_cast<x11::Atom>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Atom operator&(
    x11::Atom l, x11::Atom r) {
  using T = std::underlying_type_t<x11::Atom>;
  return static_cast<x11::Atom>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::ColormapState operator|(
    x11::ColormapState l, x11::ColormapState r) {
  using T = std::underlying_type_t<x11::ColormapState>;
  return static_cast<x11::ColormapState>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::ColormapState operator&(
    x11::ColormapState l, x11::ColormapState r) {
  using T = std::underlying_type_t<x11::ColormapState>;
  return static_cast<x11::ColormapState>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Colormap operator|(
    x11::Colormap l, x11::Colormap r) {
  using T = std::underlying_type_t<x11::Colormap>;
  return static_cast<x11::Colormap>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Colormap operator&(
    x11::Colormap l, x11::Colormap r) {
  using T = std::underlying_type_t<x11::Colormap>;
  return static_cast<x11::Colormap>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Mapping operator|(
    x11::Mapping l, x11::Mapping r) {
  using T = std::underlying_type_t<x11::Mapping>;
  return static_cast<x11::Mapping>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Mapping operator&(
    x11::Mapping l, x11::Mapping r) {
  using T = std::underlying_type_t<x11::Mapping>;
  return static_cast<x11::Mapping>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::WindowClass operator|(
    x11::WindowClass l, x11::WindowClass r) {
  using T = std::underlying_type_t<x11::WindowClass>;
  return static_cast<x11::WindowClass>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::WindowClass operator&(
    x11::WindowClass l, x11::WindowClass r) {
  using T = std::underlying_type_t<x11::WindowClass>;
  return static_cast<x11::WindowClass>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::CreateWindowAttribute operator|(
    x11::CreateWindowAttribute l, x11::CreateWindowAttribute r) {
  using T = std::underlying_type_t<x11::CreateWindowAttribute>;
  return static_cast<x11::CreateWindowAttribute>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::CreateWindowAttribute operator&(
    x11::CreateWindowAttribute l, x11::CreateWindowAttribute r) {
  using T = std::underlying_type_t<x11::CreateWindowAttribute>;
  return static_cast<x11::CreateWindowAttribute>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::BackPixmap operator|(
    x11::BackPixmap l, x11::BackPixmap r) {
  using T = std::underlying_type_t<x11::BackPixmap>;
  return static_cast<x11::BackPixmap>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::BackPixmap operator&(
    x11::BackPixmap l, x11::BackPixmap r) {
  using T = std::underlying_type_t<x11::BackPixmap>;
  return static_cast<x11::BackPixmap>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Gravity operator|(
    x11::Gravity l, x11::Gravity r) {
  using T = std::underlying_type_t<x11::Gravity>;
  return static_cast<x11::Gravity>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Gravity operator&(
    x11::Gravity l, x11::Gravity r) {
  using T = std::underlying_type_t<x11::Gravity>;
  return static_cast<x11::Gravity>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::MapState operator|(
    x11::MapState l, x11::MapState r) {
  using T = std::underlying_type_t<x11::MapState>;
  return static_cast<x11::MapState>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::MapState operator&(
    x11::MapState l, x11::MapState r) {
  using T = std::underlying_type_t<x11::MapState>;
  return static_cast<x11::MapState>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::SetMode operator|(
    x11::SetMode l, x11::SetMode r) {
  using T = std::underlying_type_t<x11::SetMode>;
  return static_cast<x11::SetMode>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::SetMode operator&(
    x11::SetMode l, x11::SetMode r) {
  using T = std::underlying_type_t<x11::SetMode>;
  return static_cast<x11::SetMode>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::ConfigWindow operator|(
    x11::ConfigWindow l, x11::ConfigWindow r) {
  using T = std::underlying_type_t<x11::ConfigWindow>;
  return static_cast<x11::ConfigWindow>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::ConfigWindow operator&(
    x11::ConfigWindow l, x11::ConfigWindow r) {
  using T = std::underlying_type_t<x11::ConfigWindow>;
  return static_cast<x11::ConfigWindow>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::StackMode operator|(
    x11::StackMode l, x11::StackMode r) {
  using T = std::underlying_type_t<x11::StackMode>;
  return static_cast<x11::StackMode>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::StackMode operator&(
    x11::StackMode l, x11::StackMode r) {
  using T = std::underlying_type_t<x11::StackMode>;
  return static_cast<x11::StackMode>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Circulate operator|(
    x11::Circulate l, x11::Circulate r) {
  using T = std::underlying_type_t<x11::Circulate>;
  return static_cast<x11::Circulate>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Circulate operator&(
    x11::Circulate l, x11::Circulate r) {
  using T = std::underlying_type_t<x11::Circulate>;
  return static_cast<x11::Circulate>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::PropMode operator|(
    x11::PropMode l, x11::PropMode r) {
  using T = std::underlying_type_t<x11::PropMode>;
  return static_cast<x11::PropMode>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::PropMode operator&(
    x11::PropMode l, x11::PropMode r) {
  using T = std::underlying_type_t<x11::PropMode>;
  return static_cast<x11::PropMode>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::GetPropertyType operator|(
    x11::GetPropertyType l, x11::GetPropertyType r) {
  using T = std::underlying_type_t<x11::GetPropertyType>;
  return static_cast<x11::GetPropertyType>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::GetPropertyType operator&(
    x11::GetPropertyType l, x11::GetPropertyType r) {
  using T = std::underlying_type_t<x11::GetPropertyType>;
  return static_cast<x11::GetPropertyType>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::SendEventDest operator|(
    x11::SendEventDest l, x11::SendEventDest r) {
  using T = std::underlying_type_t<x11::SendEventDest>;
  return static_cast<x11::SendEventDest>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::SendEventDest operator&(
    x11::SendEventDest l, x11::SendEventDest r) {
  using T = std::underlying_type_t<x11::SendEventDest>;
  return static_cast<x11::SendEventDest>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::GrabMode operator|(
    x11::GrabMode l, x11::GrabMode r) {
  using T = std::underlying_type_t<x11::GrabMode>;
  return static_cast<x11::GrabMode>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::GrabMode operator&(
    x11::GrabMode l, x11::GrabMode r) {
  using T = std::underlying_type_t<x11::GrabMode>;
  return static_cast<x11::GrabMode>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::GrabStatus operator|(
    x11::GrabStatus l, x11::GrabStatus r) {
  using T = std::underlying_type_t<x11::GrabStatus>;
  return static_cast<x11::GrabStatus>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::GrabStatus operator&(
    x11::GrabStatus l, x11::GrabStatus r) {
  using T = std::underlying_type_t<x11::GrabStatus>;
  return static_cast<x11::GrabStatus>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Cursor operator|(
    x11::Cursor l, x11::Cursor r) {
  using T = std::underlying_type_t<x11::Cursor>;
  return static_cast<x11::Cursor>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Cursor operator&(
    x11::Cursor l, x11::Cursor r) {
  using T = std::underlying_type_t<x11::Cursor>;
  return static_cast<x11::Cursor>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::ButtonIndex operator|(
    x11::ButtonIndex l, x11::ButtonIndex r) {
  using T = std::underlying_type_t<x11::ButtonIndex>;
  return static_cast<x11::ButtonIndex>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::ButtonIndex operator&(
    x11::ButtonIndex l, x11::ButtonIndex r) {
  using T = std::underlying_type_t<x11::ButtonIndex>;
  return static_cast<x11::ButtonIndex>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Grab operator|(
    x11::Grab l, x11::Grab r) {
  using T = std::underlying_type_t<x11::Grab>;
  return static_cast<x11::Grab>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Grab operator&(
    x11::Grab l, x11::Grab r) {
  using T = std::underlying_type_t<x11::Grab>;
  return static_cast<x11::Grab>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Allow operator|(
    x11::Allow l, x11::Allow r) {
  using T = std::underlying_type_t<x11::Allow>;
  return static_cast<x11::Allow>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Allow operator&(
    x11::Allow l, x11::Allow r) {
  using T = std::underlying_type_t<x11::Allow>;
  return static_cast<x11::Allow>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::InputFocus operator|(
    x11::InputFocus l, x11::InputFocus r) {
  using T = std::underlying_type_t<x11::InputFocus>;
  return static_cast<x11::InputFocus>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::InputFocus operator&(
    x11::InputFocus l, x11::InputFocus r) {
  using T = std::underlying_type_t<x11::InputFocus>;
  return static_cast<x11::InputFocus>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::FontDraw operator|(
    x11::FontDraw l, x11::FontDraw r) {
  using T = std::underlying_type_t<x11::FontDraw>;
  return static_cast<x11::FontDraw>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::FontDraw operator&(
    x11::FontDraw l, x11::FontDraw r) {
  using T = std::underlying_type_t<x11::FontDraw>;
  return static_cast<x11::FontDraw>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::GraphicsContextAttribute operator|(
    x11::GraphicsContextAttribute l, x11::GraphicsContextAttribute r) {
  using T = std::underlying_type_t<x11::GraphicsContextAttribute>;
  return static_cast<x11::GraphicsContextAttribute>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::GraphicsContextAttribute operator&(
    x11::GraphicsContextAttribute l, x11::GraphicsContextAttribute r) {
  using T = std::underlying_type_t<x11::GraphicsContextAttribute>;
  return static_cast<x11::GraphicsContextAttribute>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Gx operator|(
    x11::Gx l, x11::Gx r) {
  using T = std::underlying_type_t<x11::Gx>;
  return static_cast<x11::Gx>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Gx operator&(
    x11::Gx l, x11::Gx r) {
  using T = std::underlying_type_t<x11::Gx>;
  return static_cast<x11::Gx>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::LineStyle operator|(
    x11::LineStyle l, x11::LineStyle r) {
  using T = std::underlying_type_t<x11::LineStyle>;
  return static_cast<x11::LineStyle>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::LineStyle operator&(
    x11::LineStyle l, x11::LineStyle r) {
  using T = std::underlying_type_t<x11::LineStyle>;
  return static_cast<x11::LineStyle>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::CapStyle operator|(
    x11::CapStyle l, x11::CapStyle r) {
  using T = std::underlying_type_t<x11::CapStyle>;
  return static_cast<x11::CapStyle>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::CapStyle operator&(
    x11::CapStyle l, x11::CapStyle r) {
  using T = std::underlying_type_t<x11::CapStyle>;
  return static_cast<x11::CapStyle>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::JoinStyle operator|(
    x11::JoinStyle l, x11::JoinStyle r) {
  using T = std::underlying_type_t<x11::JoinStyle>;
  return static_cast<x11::JoinStyle>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::JoinStyle operator&(
    x11::JoinStyle l, x11::JoinStyle r) {
  using T = std::underlying_type_t<x11::JoinStyle>;
  return static_cast<x11::JoinStyle>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::FillStyle operator|(
    x11::FillStyle l, x11::FillStyle r) {
  using T = std::underlying_type_t<x11::FillStyle>;
  return static_cast<x11::FillStyle>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::FillStyle operator&(
    x11::FillStyle l, x11::FillStyle r) {
  using T = std::underlying_type_t<x11::FillStyle>;
  return static_cast<x11::FillStyle>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::FillRule operator|(
    x11::FillRule l, x11::FillRule r) {
  using T = std::underlying_type_t<x11::FillRule>;
  return static_cast<x11::FillRule>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::FillRule operator&(
    x11::FillRule l, x11::FillRule r) {
  using T = std::underlying_type_t<x11::FillRule>;
  return static_cast<x11::FillRule>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::SubwindowMode operator|(
    x11::SubwindowMode l, x11::SubwindowMode r) {
  using T = std::underlying_type_t<x11::SubwindowMode>;
  return static_cast<x11::SubwindowMode>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::SubwindowMode operator&(
    x11::SubwindowMode l, x11::SubwindowMode r) {
  using T = std::underlying_type_t<x11::SubwindowMode>;
  return static_cast<x11::SubwindowMode>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::ArcMode operator|(
    x11::ArcMode l, x11::ArcMode r) {
  using T = std::underlying_type_t<x11::ArcMode>;
  return static_cast<x11::ArcMode>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::ArcMode operator&(
    x11::ArcMode l, x11::ArcMode r) {
  using T = std::underlying_type_t<x11::ArcMode>;
  return static_cast<x11::ArcMode>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::ClipOrdering operator|(
    x11::ClipOrdering l, x11::ClipOrdering r) {
  using T = std::underlying_type_t<x11::ClipOrdering>;
  return static_cast<x11::ClipOrdering>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::ClipOrdering operator&(
    x11::ClipOrdering l, x11::ClipOrdering r) {
  using T = std::underlying_type_t<x11::ClipOrdering>;
  return static_cast<x11::ClipOrdering>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::CoordMode operator|(
    x11::CoordMode l, x11::CoordMode r) {
  using T = std::underlying_type_t<x11::CoordMode>;
  return static_cast<x11::CoordMode>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::CoordMode operator&(
    x11::CoordMode l, x11::CoordMode r) {
  using T = std::underlying_type_t<x11::CoordMode>;
  return static_cast<x11::CoordMode>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::PolyShape operator|(
    x11::PolyShape l, x11::PolyShape r) {
  using T = std::underlying_type_t<x11::PolyShape>;
  return static_cast<x11::PolyShape>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::PolyShape operator&(
    x11::PolyShape l, x11::PolyShape r) {
  using T = std::underlying_type_t<x11::PolyShape>;
  return static_cast<x11::PolyShape>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::ImageFormat operator|(
    x11::ImageFormat l, x11::ImageFormat r) {
  using T = std::underlying_type_t<x11::ImageFormat>;
  return static_cast<x11::ImageFormat>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::ImageFormat operator&(
    x11::ImageFormat l, x11::ImageFormat r) {
  using T = std::underlying_type_t<x11::ImageFormat>;
  return static_cast<x11::ImageFormat>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::ColormapAlloc operator|(
    x11::ColormapAlloc l, x11::ColormapAlloc r) {
  using T = std::underlying_type_t<x11::ColormapAlloc>;
  return static_cast<x11::ColormapAlloc>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::ColormapAlloc operator&(
    x11::ColormapAlloc l, x11::ColormapAlloc r) {
  using T = std::underlying_type_t<x11::ColormapAlloc>;
  return static_cast<x11::ColormapAlloc>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::ColorFlag operator|(
    x11::ColorFlag l, x11::ColorFlag r) {
  using T = std::underlying_type_t<x11::ColorFlag>;
  return static_cast<x11::ColorFlag>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::ColorFlag operator&(
    x11::ColorFlag l, x11::ColorFlag r) {
  using T = std::underlying_type_t<x11::ColorFlag>;
  return static_cast<x11::ColorFlag>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Pixmap operator|(
    x11::Pixmap l, x11::Pixmap r) {
  using T = std::underlying_type_t<x11::Pixmap>;
  return static_cast<x11::Pixmap>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Pixmap operator&(
    x11::Pixmap l, x11::Pixmap r) {
  using T = std::underlying_type_t<x11::Pixmap>;
  return static_cast<x11::Pixmap>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Font operator|(
    x11::Font l, x11::Font r) {
  using T = std::underlying_type_t<x11::Font>;
  return static_cast<x11::Font>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Font operator&(
    x11::Font l, x11::Font r) {
  using T = std::underlying_type_t<x11::Font>;
  return static_cast<x11::Font>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::QueryShapeOf operator|(
    x11::QueryShapeOf l, x11::QueryShapeOf r) {
  using T = std::underlying_type_t<x11::QueryShapeOf>;
  return static_cast<x11::QueryShapeOf>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::QueryShapeOf operator&(
    x11::QueryShapeOf l, x11::QueryShapeOf r) {
  using T = std::underlying_type_t<x11::QueryShapeOf>;
  return static_cast<x11::QueryShapeOf>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Keyboard operator|(
    x11::Keyboard l, x11::Keyboard r) {
  using T = std::underlying_type_t<x11::Keyboard>;
  return static_cast<x11::Keyboard>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Keyboard operator&(
    x11::Keyboard l, x11::Keyboard r) {
  using T = std::underlying_type_t<x11::Keyboard>;
  return static_cast<x11::Keyboard>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::LedMode operator|(
    x11::LedMode l, x11::LedMode r) {
  using T = std::underlying_type_t<x11::LedMode>;
  return static_cast<x11::LedMode>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::LedMode operator&(
    x11::LedMode l, x11::LedMode r) {
  using T = std::underlying_type_t<x11::LedMode>;
  return static_cast<x11::LedMode>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::AutoRepeatMode operator|(
    x11::AutoRepeatMode l, x11::AutoRepeatMode r) {
  using T = std::underlying_type_t<x11::AutoRepeatMode>;
  return static_cast<x11::AutoRepeatMode>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::AutoRepeatMode operator&(
    x11::AutoRepeatMode l, x11::AutoRepeatMode r) {
  using T = std::underlying_type_t<x11::AutoRepeatMode>;
  return static_cast<x11::AutoRepeatMode>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Blanking operator|(
    x11::Blanking l, x11::Blanking r) {
  using T = std::underlying_type_t<x11::Blanking>;
  return static_cast<x11::Blanking>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Blanking operator&(
    x11::Blanking l, x11::Blanking r) {
  using T = std::underlying_type_t<x11::Blanking>;
  return static_cast<x11::Blanking>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Exposures operator|(
    x11::Exposures l, x11::Exposures r) {
  using T = std::underlying_type_t<x11::Exposures>;
  return static_cast<x11::Exposures>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Exposures operator&(
    x11::Exposures l, x11::Exposures r) {
  using T = std::underlying_type_t<x11::Exposures>;
  return static_cast<x11::Exposures>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::HostMode operator|(
    x11::HostMode l, x11::HostMode r) {
  using T = std::underlying_type_t<x11::HostMode>;
  return static_cast<x11::HostMode>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::HostMode operator&(
    x11::HostMode l, x11::HostMode r) {
  using T = std::underlying_type_t<x11::HostMode>;
  return static_cast<x11::HostMode>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Family operator|(
    x11::Family l, x11::Family r) {
  using T = std::underlying_type_t<x11::Family>;
  return static_cast<x11::Family>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Family operator&(
    x11::Family l, x11::Family r) {
  using T = std::underlying_type_t<x11::Family>;
  return static_cast<x11::Family>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::AccessControl operator|(
    x11::AccessControl l, x11::AccessControl r) {
  using T = std::underlying_type_t<x11::AccessControl>;
  return static_cast<x11::AccessControl>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::AccessControl operator&(
    x11::AccessControl l, x11::AccessControl r) {
  using T = std::underlying_type_t<x11::AccessControl>;
  return static_cast<x11::AccessControl>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::CloseDown operator|(
    x11::CloseDown l, x11::CloseDown r) {
  using T = std::underlying_type_t<x11::CloseDown>;
  return static_cast<x11::CloseDown>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::CloseDown operator&(
    x11::CloseDown l, x11::CloseDown r) {
  using T = std::underlying_type_t<x11::CloseDown>;
  return static_cast<x11::CloseDown>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::Kill operator|(
    x11::Kill l, x11::Kill r) {
  using T = std::underlying_type_t<x11::Kill>;
  return static_cast<x11::Kill>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::Kill operator&(
    x11::Kill l, x11::Kill r) {
  using T = std::underlying_type_t<x11::Kill>;
  return static_cast<x11::Kill>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::ScreenSaverMode operator|(
    x11::ScreenSaverMode l, x11::ScreenSaverMode r) {
  using T = std::underlying_type_t<x11::ScreenSaverMode>;
  return static_cast<x11::ScreenSaverMode>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::ScreenSaverMode operator&(
    x11::ScreenSaverMode l, x11::ScreenSaverMode r) {
  using T = std::underlying_type_t<x11::ScreenSaverMode>;
  return static_cast<x11::ScreenSaverMode>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::MappingStatus operator|(
    x11::MappingStatus l, x11::MappingStatus r) {
  using T = std::underlying_type_t<x11::MappingStatus>;
  return static_cast<x11::MappingStatus>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::MappingStatus operator&(
    x11::MappingStatus l, x11::MappingStatus r) {
  using T = std::underlying_type_t<x11::MappingStatus>;
  return static_cast<x11::MappingStatus>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::MapIndex operator|(
    x11::MapIndex l, x11::MapIndex r) {
  using T = std::underlying_type_t<x11::MapIndex>;
  return static_cast<x11::MapIndex>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::MapIndex operator&(
    x11::MapIndex l, x11::MapIndex r) {
  using T = std::underlying_type_t<x11::MapIndex>;
  return static_cast<x11::MapIndex>(
      static_cast<T>(l) & static_cast<T>(r));
}


#endif  // GEN_UI_GFX_X_XPROTO_H_
