#ifndef GEN_UI_GFX_X_XF86VIDMODE_H_
#define GEN_UI_GFX_X_XF86VIDMODE_H_

#include <array>
#include <cstddef>
#include <cstdint>
#include <cstring>
#include <vector>

#include "base/component_export.h"
#include "base/memory/ref_counted_memory.h"
#include "base/memory/scoped_refptr.h"
#include "base/optional.h"
#include "base/files/scoped_file.h"
#include "ui/gfx/x/xproto_types.h"
#include "xproto.h"

namespace x11 {

class Connection;

class COMPONENT_EXPORT(X11) XF86VidMode {
  public:
  static constexpr unsigned major_version = 2;
  static constexpr unsigned minor_version = 2;

  XF86VidMode(Connection* connection,
      const x11::QueryExtensionReply& info);

  uint8_t present() const {
    return info_.present;
  }
  uint8_t major_opcode() const {
    return info_.major_opcode;
  }
  uint8_t first_event() const {
    return info_.first_event;
  }
  uint8_t first_error() const {
    return info_.first_error;
  }

  Connection* connection() const { return connection_; }

  enum class Syncrange : uint32_t {};

  enum class DotClock : uint32_t {};

  enum class ModeFlag : int {
    Positive_HSync = 1 << 0,
    Negative_HSync = 1 << 1,
    Positive_VSync = 1 << 2,
    Negative_VSync = 1 << 3,
    Interlace = 1 << 4,
    Composite_Sync = 1 << 5,
    Positive_CSync = 1 << 6,
    Negative_CSync = 1 << 7,
    HSkew = 1 << 8,
    Broadcast = 1 << 9,
    Pixmux = 1 << 10,
    Double_Clock = 1 << 11,
    Half_Clock = 1 << 12,
  };

  enum class ClockFlag : int {
    Programable = 1 << 0,
  };

  enum class Permission : int {
    Read = 1 << 0,
    Write = 1 << 1,
  };

  struct ModeInfo {
    DotClock dotclock{};
    uint16_t hdisplay{};
    uint16_t hsyncstart{};
    uint16_t hsyncend{};
    uint16_t htotal{};
    uint32_t hskew{};
    uint16_t vdisplay{};
    uint16_t vsyncstart{};
    uint16_t vsyncend{};
    uint16_t vtotal{};
    ModeFlag flags{};
    uint32_t privsize{};
  };

  struct BadClockError {
    uint16_t sequence{};
  };

  struct BadHTimingsError {
    uint16_t sequence{};
  };

  struct BadVTimingsError {
    uint16_t sequence{};
  };

  struct ModeUnsuitableError {
    uint16_t sequence{};
  };

  struct ExtensionDisabledError {
    uint16_t sequence{};
  };

  struct ClientNotLocalError {
    uint16_t sequence{};
  };

  struct ZoomLockedError {
    uint16_t sequence{};
  };

  struct QueryVersionRequest {
  };

  struct QueryVersionReply {
    uint16_t sequence{};
    uint16_t major_version{};
    uint16_t minor_version{};
  };

  using QueryVersionResponse = Response<QueryVersionReply>;

  Future<QueryVersionReply> QueryVersion(
      const QueryVersionRequest& request);

  struct GetModeLineRequest {
    uint16_t screen{};
  };

  struct GetModeLineReply {
    uint16_t sequence{};
    DotClock dotclock{};
    uint16_t hdisplay{};
    uint16_t hsyncstart{};
    uint16_t hsyncend{};
    uint16_t htotal{};
    uint16_t hskew{};
    uint16_t vdisplay{};
    uint16_t vsyncstart{};
    uint16_t vsyncend{};
    uint16_t vtotal{};
    ModeFlag flags{};
    std::vector<uint8_t> c_private{};
  };

  using GetModeLineResponse = Response<GetModeLineReply>;

  Future<GetModeLineReply> GetModeLine(
      const GetModeLineRequest& request);

  struct ModModeLineRequest {
    uint32_t screen{};
    uint16_t hdisplay{};
    uint16_t hsyncstart{};
    uint16_t hsyncend{};
    uint16_t htotal{};
    uint16_t hskew{};
    uint16_t vdisplay{};
    uint16_t vsyncstart{};
    uint16_t vsyncend{};
    uint16_t vtotal{};
    ModeFlag flags{};
    std::vector<uint8_t> c_private{};
  };

  using ModModeLineResponse = Response<void>;

  Future<void> ModModeLine(
      const ModModeLineRequest& request);

  struct SwitchModeRequest {
    uint16_t screen{};
    uint16_t zoom{};
  };

  using SwitchModeResponse = Response<void>;

  Future<void> SwitchMode(
      const SwitchModeRequest& request);

  struct GetMonitorRequest {
    uint16_t screen{};
  };

  struct GetMonitorReply {
    uint16_t sequence{};
    std::vector<Syncrange> hsync{};
    std::vector<Syncrange> vsync{};
    std::string vendor{};
    scoped_refptr<base::RefCountedMemory> alignment_pad{};
    std::string model{};
  };

  using GetMonitorResponse = Response<GetMonitorReply>;

  Future<GetMonitorReply> GetMonitor(
      const GetMonitorRequest& request);

  struct LockModeSwitchRequest {
    uint16_t screen{};
    uint16_t lock{};
  };

  using LockModeSwitchResponse = Response<void>;

  Future<void> LockModeSwitch(
      const LockModeSwitchRequest& request);

  struct GetAllModeLinesRequest {
    uint16_t screen{};
  };

  struct GetAllModeLinesReply {
    uint16_t sequence{};
    std::vector<ModeInfo> modeinfo{};
  };

  using GetAllModeLinesResponse = Response<GetAllModeLinesReply>;

  Future<GetAllModeLinesReply> GetAllModeLines(
      const GetAllModeLinesRequest& request);

  struct AddModeLineRequest {
    uint32_t screen{};
    DotClock dotclock{};
    uint16_t hdisplay{};
    uint16_t hsyncstart{};
    uint16_t hsyncend{};
    uint16_t htotal{};
    uint16_t hskew{};
    uint16_t vdisplay{};
    uint16_t vsyncstart{};
    uint16_t vsyncend{};
    uint16_t vtotal{};
    ModeFlag flags{};
    DotClock after_dotclock{};
    uint16_t after_hdisplay{};
    uint16_t after_hsyncstart{};
    uint16_t after_hsyncend{};
    uint16_t after_htotal{};
    uint16_t after_hskew{};
    uint16_t after_vdisplay{};
    uint16_t after_vsyncstart{};
    uint16_t after_vsyncend{};
    uint16_t after_vtotal{};
    ModeFlag after_flags{};
    std::vector<uint8_t> c_private{};
  };

  using AddModeLineResponse = Response<void>;

  Future<void> AddModeLine(
      const AddModeLineRequest& request);

  struct DeleteModeLineRequest {
    uint32_t screen{};
    DotClock dotclock{};
    uint16_t hdisplay{};
    uint16_t hsyncstart{};
    uint16_t hsyncend{};
    uint16_t htotal{};
    uint16_t hskew{};
    uint16_t vdisplay{};
    uint16_t vsyncstart{};
    uint16_t vsyncend{};
    uint16_t vtotal{};
    ModeFlag flags{};
    std::vector<uint8_t> c_private{};
  };

  using DeleteModeLineResponse = Response<void>;

  Future<void> DeleteModeLine(
      const DeleteModeLineRequest& request);

  struct ValidateModeLineRequest {
    uint32_t screen{};
    DotClock dotclock{};
    uint16_t hdisplay{};
    uint16_t hsyncstart{};
    uint16_t hsyncend{};
    uint16_t htotal{};
    uint16_t hskew{};
    uint16_t vdisplay{};
    uint16_t vsyncstart{};
    uint16_t vsyncend{};
    uint16_t vtotal{};
    ModeFlag flags{};
    std::vector<uint8_t> c_private{};
  };

  struct ValidateModeLineReply {
    uint16_t sequence{};
    uint32_t status{};
  };

  using ValidateModeLineResponse = Response<ValidateModeLineReply>;

  Future<ValidateModeLineReply> ValidateModeLine(
      const ValidateModeLineRequest& request);

  struct SwitchToModeRequest {
    uint32_t screen{};
    DotClock dotclock{};
    uint16_t hdisplay{};
    uint16_t hsyncstart{};
    uint16_t hsyncend{};
    uint16_t htotal{};
    uint16_t hskew{};
    uint16_t vdisplay{};
    uint16_t vsyncstart{};
    uint16_t vsyncend{};
    uint16_t vtotal{};
    ModeFlag flags{};
    std::vector<uint8_t> c_private{};
  };

  using SwitchToModeResponse = Response<void>;

  Future<void> SwitchToMode(
      const SwitchToModeRequest& request);

  struct GetViewPortRequest {
    uint16_t screen{};
  };

  struct GetViewPortReply {
    uint16_t sequence{};
    uint32_t x{};
    uint32_t y{};
  };

  using GetViewPortResponse = Response<GetViewPortReply>;

  Future<GetViewPortReply> GetViewPort(
      const GetViewPortRequest& request);

  struct SetViewPortRequest {
    uint16_t screen{};
    uint32_t x{};
    uint32_t y{};
  };

  using SetViewPortResponse = Response<void>;

  Future<void> SetViewPort(
      const SetViewPortRequest& request);

  struct GetDotClocksRequest {
    uint16_t screen{};
  };

  struct GetDotClocksReply {
    uint16_t sequence{};
    ClockFlag flags{};
    uint32_t clocks{};
    uint32_t maxclocks{};
    std::vector<uint32_t> clock{};
  };

  using GetDotClocksResponse = Response<GetDotClocksReply>;

  Future<GetDotClocksReply> GetDotClocks(
      const GetDotClocksRequest& request);

  struct SetClientVersionRequest {
    uint16_t major{};
    uint16_t minor{};
  };

  using SetClientVersionResponse = Response<void>;

  Future<void> SetClientVersion(
      const SetClientVersionRequest& request);

  struct SetGammaRequest {
    uint16_t screen{};
    uint32_t red{};
    uint32_t green{};
    uint32_t blue{};
  };

  using SetGammaResponse = Response<void>;

  Future<void> SetGamma(
      const SetGammaRequest& request);

  struct GetGammaRequest {
    uint16_t screen{};
  };

  struct GetGammaReply {
    uint16_t sequence{};
    uint32_t red{};
    uint32_t green{};
    uint32_t blue{};
  };

  using GetGammaResponse = Response<GetGammaReply>;

  Future<GetGammaReply> GetGamma(
      const GetGammaRequest& request);

  struct GetGammaRampRequest {
    uint16_t screen{};
    uint16_t size{};
  };

  struct GetGammaRampReply {
    uint16_t sequence{};
    uint16_t size{};
    std::vector<uint16_t> red{};
    std::vector<uint16_t> green{};
    std::vector<uint16_t> blue{};
  };

  using GetGammaRampResponse = Response<GetGammaRampReply>;

  Future<GetGammaRampReply> GetGammaRamp(
      const GetGammaRampRequest& request);

  struct SetGammaRampRequest {
    uint16_t screen{};
    uint16_t size{};
    std::vector<uint16_t> red{};
    std::vector<uint16_t> green{};
    std::vector<uint16_t> blue{};
  };

  using SetGammaRampResponse = Response<void>;

  Future<void> SetGammaRamp(
      const SetGammaRampRequest& request);

  struct GetGammaRampSizeRequest {
    uint16_t screen{};
  };

  struct GetGammaRampSizeReply {
    uint16_t sequence{};
    uint16_t size{};
  };

  using GetGammaRampSizeResponse = Response<GetGammaRampSizeReply>;

  Future<GetGammaRampSizeReply> GetGammaRampSize(
      const GetGammaRampSizeRequest& request);

  struct GetPermissionsRequest {
    uint16_t screen{};
  };

  struct GetPermissionsReply {
    uint16_t sequence{};
    Permission permissions{};
  };

  using GetPermissionsResponse = Response<GetPermissionsReply>;

  Future<GetPermissionsReply> GetPermissions(
      const GetPermissionsRequest& request);

  private:
  x11::Connection* const connection_;
  x11::QueryExtensionReply info_{};
};

}  // namespace x11

inline constexpr x11::XF86VidMode::ModeFlag operator|(
    x11::XF86VidMode::ModeFlag l, x11::XF86VidMode::ModeFlag r) {
  using T = std::underlying_type_t<x11::XF86VidMode::ModeFlag>;
  return static_cast<x11::XF86VidMode::ModeFlag>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::XF86VidMode::ModeFlag operator&(
    x11::XF86VidMode::ModeFlag l, x11::XF86VidMode::ModeFlag r) {
  using T = std::underlying_type_t<x11::XF86VidMode::ModeFlag>;
  return static_cast<x11::XF86VidMode::ModeFlag>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::XF86VidMode::ClockFlag operator|(
    x11::XF86VidMode::ClockFlag l, x11::XF86VidMode::ClockFlag r) {
  using T = std::underlying_type_t<x11::XF86VidMode::ClockFlag>;
  return static_cast<x11::XF86VidMode::ClockFlag>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::XF86VidMode::ClockFlag operator&(
    x11::XF86VidMode::ClockFlag l, x11::XF86VidMode::ClockFlag r) {
  using T = std::underlying_type_t<x11::XF86VidMode::ClockFlag>;
  return static_cast<x11::XF86VidMode::ClockFlag>(
      static_cast<T>(l) & static_cast<T>(r));
}

inline constexpr x11::XF86VidMode::Permission operator|(
    x11::XF86VidMode::Permission l, x11::XF86VidMode::Permission r) {
  using T = std::underlying_type_t<x11::XF86VidMode::Permission>;
  return static_cast<x11::XF86VidMode::Permission>(
      static_cast<T>(l) | static_cast<T>(r));
}

inline constexpr x11::XF86VidMode::Permission operator&(
    x11::XF86VidMode::Permission l, x11::XF86VidMode::Permission r) {
  using T = std::underlying_type_t<x11::XF86VidMode::Permission>;
  return static_cast<x11::XF86VidMode::Permission>(
      static_cast<T>(l) & static_cast<T>(r));
}


#endif  // GEN_UI_GFX_X_XF86VIDMODE_H_
