// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_fuse.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_fuse__A0__F259

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 32})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 64})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 32 },
    { .BLOCK_M = 64, .BLOCK_N = 64 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_T_1___gfx950)xyzw" };
#else
{ R"xyzw(amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_T_1___gfx950)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊bf16@16_96_0_F_T_1)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx950)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xf7c8ae24u, 0x18a939ecu, 0, 6 }, // f7c8ae2418a939ec = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_T_1___gfx950__P__16_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xe95dd87bu, 0x79f77d61u, 0, 143 }, // e95dd87b79f77d61 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_T_1___gfx950__P__16_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x0449c370u, 0x972fdfcau, 23, 109 }, // 0449c370972fdfca = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_T_1___gfx950__P__32_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x201b3f9du, 0xd3aa4db3u, 23, 6 }, // 201b3f9dd3aa4db3 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_T_1___gfx950__P__32_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x9c3e10f7u, 0x52124c99u, 23, 143 }, // 9c3e10f752124c99 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_T_1___gfx950__P__32_16__CO__wave3_warp2_stg1--Arch_gfx950
    { 0x75b11e3cu, 0xbbbf71c6u, 63, 6 }, // 75b11e3cbbbf71c6 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_T_1___gfx950__P__32_32__CO__wave2_warp2_stg1--Arch_gfx950
    { 0x00d0ba11u, 0x0c4b6529u, 103, 109 }, // 00d0ba110c4b6529 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_T_1___gfx950__P__64_16__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x56605d48u, 0x81611889u, 103, 6 }, // 56605d4881611889 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_T_1___gfx950__P__64_16__CO__wave2_warp2_stg1--Arch_gfx950
    { 0xc371ac71u, 0xb4e8bfc1u, 103, 69 }, // c371ac71b4e8bfc1 = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_T_1___gfx950__P__64_16__CO__wave2_warp4_stg1--Arch_gfx950
    { 0x9e40bca3u, 0x03b9eafeu, 160, 109 }, // 9e40bca303b9eafe = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_T_1___gfx950__P__64_32__CO__wave1_warp2_stg1--Arch_gfx950
    { 0x5babfe9cu, 0x94c88fdbu, 166, 109 }, // 5babfe9c94c88fdb = b2sum -l 64 <<< amd-gfx950/flash/bwd_kernel_fuse/FONLY__＊bf16@16_96_0_F_T_1___gfx950__P__64_64__CO__wave1_warp2_stg1--Arch_gfx950
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_fuse_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_fuse_packed_string);

static int8_t lut[1][7][7] =
{
// GPU gfx950_mod0
{{ 4, 8, 6, 9, 2, 7, 7},
 { 3, 8, 2, 2, 7, 7, 7},
 { 3, 2, 4, 4, 5, 4, 4},
 { 2, 2, 4, 4, 4, 4, 4},
 { 4, 4, 5, 4, 4, 4, 4},
 { 4, 4, 4, 4, 4, 4, 5},
 { 4, 7, 4, 4, 4,10,10}}
// End of GPU gfx950_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelFuseContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelFuseContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_fuse__A0__F259 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 1;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_fuse__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 1;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*bf16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = 0
// philox_seed_ptr = 0
// philox_offset1 = 0
// philox_offset2 = 0
// Window_left = 0
// BLOCK_DMODEL = 96
// CAUSAL_TYPE = 0
// ENABLE_DROPOUT = False
// PADDED_HEAD = True
// BIAS_TYPE = 1

// SELECT * FROM FLASH$bwd_kernel_fuse WHERE gpu IN ('gfx950_mod0') AND inputs$Q_dtype = 'torch.bfloat16' AND inputs$BLOCK_DMODEL = 96 AND inputs$CAUSAL_TYPE = 0 AND inputs$ENABLE_DROPOUT = False AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 1

// vim: set fileencoding=utf-8

