use crate::util::Util;
use glib::{Enum, Properties, prelude::*, subclass::prelude::*};
use serde::{Deserialize, Serialize};
use std::cell::{Cell, RefCell};

#[derive(Default, Debug, Clone, Copy, Serialize, Deserialize, Eq, PartialEq, Enum)]
#[repr(u32)]
#[enum_type(name = "ProxyProtocoll")]
pub enum ProxyProtocoll {
    #[default]
    All,
    Http,
    Https,
}

mod imp {
    use super::*;

    #[derive(Default, Debug, Serialize, Deserialize, Properties)]
    #[properties(wrapper_type = super::ProxyModel)]
    pub struct ProxyModel {
        #[property(get, set, builder(ProxyProtocoll::All))]
        pub protocoll: Cell<ProxyProtocoll>,

        #[property(get, set)]
        pub url: RefCell<String>,

        #[property(get, set, nullable)]
        pub user: RefCell<Option<String>>,

        #[property(get, set, nullable)]
        pub password: RefCell<Option<String>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ProxyModel {
        const NAME: &'static str = "ProxyModel";
        type Type = super::ProxyModel;
    }

    #[glib::derived_properties]
    impl ObjectImpl for ProxyModel {}
}

glib::wrapper! {
    pub struct ProxyModel(ObjectSubclass<imp::ProxyModel>);
}

impl Serialize for ProxyModel {
    fn serialize<S>(&self, serializer: S) -> Result<S::Ok, S::Error>
    where
        S: serde::Serializer,
    {
        self.imp().serialize(serializer)
    }
}

impl<'de> Deserialize<'de> for ProxyModel {
    fn deserialize<D>(deserializer: D) -> Result<Self, D::Error>
    where
        D: serde::Deserializer<'de>,
    {
        let inner = imp::ProxyModel::deserialize(deserializer)?;
        Ok(inner.into())
    }
}

impl From<imp::ProxyModel> for ProxyModel {
    fn from(inner: imp::ProxyModel) -> Self {
        glib::Object::builder()
            .property("protocoll", inner.protocoll.get())
            .property("url", inner.url.borrow().clone())
            .property("user", inner.user.borrow().clone())
            .property("password", inner.password.borrow().clone())
            .build()
    }
}

impl Default for ProxyModel {
    fn default() -> Self {
        imp::ProxyModel::default().into()
    }
}

const fn _default_true() -> bool {
    true
}

fn _default_ping_urls() -> Vec<String> {
    vec!["http://example.com/".into(), "https://google.com/".into()]
}

#[derive(Clone, Debug, Serialize, Deserialize)]
pub struct AdvancedSettings {
    pub proxy: Vec<ProxyModel>,
    pub accept_invalid_certs: bool,
    pub accept_invalid_hostnames: bool,
    #[serde(default)]
    pub inspect_article_view: bool,
    #[serde(default = "_default_true")]
    pub article_view_load_images: bool,
    #[serde(default = "_default_ping_urls")]
    #[serde(deserialize_with = "Util::deserialize_string_or_vec")]
    pub ping_url: Vec<String>,
}

impl Default for AdvancedSettings {
    fn default() -> Self {
        AdvancedSettings {
            proxy: Vec::new(),
            accept_invalid_certs: false,
            accept_invalid_hostnames: false,
            inspect_article_view: false,
            article_view_load_images: true,
            ping_url: _default_ping_urls(),
        }
    }
}
