mod builder;
mod drag_buffer;
mod drag_scroll_tracker;
mod header;
mod loader;
mod progress_overlay;
mod scrollable;
mod url_overlay;
mod webview;

use self::progress_overlay::ProgressOverlay;
use self::url_overlay::UrlOverlay;
pub use self::webview::Webview;
use crate::App;
use crate::content_page::ArticleViewColumn;
use crate::gobject_models::GArticle;
use crate::image_dialog::ImageDialog;
use crate::main_window::MainWindow;
use crate::util::constants;
pub use builder::ArticleBuilder;
use drag_scroll_tracker::DragScrollTracker;
use futures::channel::oneshot::Sender as OneShotSender;
use gdk4::{ModifierType, RGBA, ScrollDirection, ScrollEvent};
use glib::{Propagation, Properties, subclass::*};
use gtk4::{
    Accessible, Box, Buildable, CompositeTemplate, ConstraintTarget, EventControllerScroll, EventSequenceState,
    GestureDrag, Widget, prelude::*, subclass::prelude::*,
};
use libadwaita::prelude::*;
pub use loader::ArticleLoader;
use news_flash::models::Url;
use std::cell::{Cell, RefCell};
use std::rc::Rc;
use std::str;
use url::{Host, Origin};

mod imp {
    use super::*;

    #[derive(Debug, Default, CompositeTemplate, Properties)]
    #[properties(wrapper_type = super::ArticleView)]
    #[template(file = "data/resources/ui_templates/article_view/view.blp")]
    pub struct ArticleView {
        #[template_child]
        pub view: TemplateChild<Webview>,

        #[template_child]
        pub scroll_event: TemplateChild<EventControllerScroll>,
        #[template_child]
        pub drag_gesture: TemplateChild<GestureDrag>,
        #[template_child]
        pub drag_scroll_tracker: TemplateChild<DragScrollTracker>,
        #[template_child]
        pub progress_overlay: TemplateChild<ProgressOverlay>,

        #[property(get, set, default = 1.0)]
        pub zoom: Cell<f64>,

        #[property(get = Self::get_font_size, set = Self::set_font_size, type = u32, name = "font-size")]
        #[property(get, set, name = "status-margin")]
        pub status_margin: Cell<i32>,

        #[property(get, set, name = "prefer-scraped-content")]
        pub prefer_scraped_content: Cell<bool>,

        #[property(get, set)]
        pub is_fullscreen: Cell<bool>,

        #[property(get, set = Self::set_article, nullable)]
        pub article: RefCell<Option<GArticle>>,

        pub remember_toolbar_state: Rc<RefCell<(bool, bool)>>,
    }

    #[glib::object_subclass]
    impl ObjectSubclass for ArticleView {
        const NAME: &'static str = "ArticleView";
        type ParentType = Box;
        type Type = super::ArticleView;

        fn class_init(klass: &mut Self::Class) {
            UrlOverlay::ensure_type();

            klass.bind_template();
            klass.bind_template_callbacks();
        }

        fn instance_init(obj: &InitializingObject<Self>) {
            obj.init_template();
        }
    }

    #[glib::derived_properties]
    impl ObjectImpl for ArticleView {}

    impl WidgetImpl for ArticleView {}

    impl BoxImpl for ArticleView {}

    #[gtk4::template_callbacks]
    impl ArticleView {
        //----------------------------------
        // zoom with ctrl+scroll
        //----------------------------------
        #[template_callback]
        fn on_scroll_zoom(&self, _x_delta: f64, y_delta: f64) -> Propagation {
            let obj = self.obj();

            let Some(event) = self.scroll_event.current_event() else {
                return Propagation::Stop;
            };

            // stop drag if scrolling during drag
            // https://gitlab.com/news-flash/news_flash_gtk/-/issues/549
            self.drag_scroll_tracker.stop();

            if !self
                .scroll_event
                .current_event_state()
                .contains(ModifierType::CONTROL_MASK)
            {
                return Propagation::Proceed;
            }

            let Ok(scroll_event) = event.downcast::<ScrollEvent>() else {
                return Propagation::Proceed;
            };

            let zoom = obj.zoom();

            match scroll_event.direction() {
                ScrollDirection::Up => obj.activate_action("articleview.zoom-in", None).unwrap(),
                ScrollDirection::Down => obj.activate_action("articleview.zoom-out", None).unwrap(),
                ScrollDirection::Smooth => {
                    if (y_delta > 0.0 && zoom >= constants::ARTICLE_ZOOM_UPPER)
                        || (y_delta < 0.0 && zoom <= constants::ARTICLE_ZOOM_LOWER)
                    {
                        return Propagation::Stop;
                    } else {
                        let diff = y_delta * constants::ARTICLE_PIXEL_SCROLL_FACTOR;
                        obj.set_zoom(zoom + diff);
                    }
                }
                _ => {}
            }

            Propagation::Stop
        }

        #[template_callback]
        fn on_drag_scroll(&self, diff: f64) {
            self.view.scroll_diff(diff);
        }

        #[template_callback]
        fn on_drag_begin(&self, _x: f64, _y: f64) {
            self.drag_gesture.set_state(EventSequenceState::Claimed);
            self.drag_scroll_tracker.begin_drag();
        }

        #[template_callback]
        fn on_drag_update(&self, _x: f64, y: f64) {
            self.drag_gesture.set_state(EventSequenceState::Claimed);
            self.drag_scroll_tracker.update_drag(y);
        }

        #[template_callback]
        fn on_drag_end(&self, _x: f64, _y: f64) {
            self.drag_gesture.set_state(EventSequenceState::Claimed);
            let scroll_pos = self.view.get_scroll_pos();
            let scroll_upper = self.view.get_scroll_upper();
            let page_size = self.view.height() as f64;
            self.drag_scroll_tracker.end_drag(scroll_pos, scroll_upper, page_size);
        }

        #[template_callback]
        fn on_image_dialog(&self, data: String) {
            let image_dialog = if data.starts_with("data:") {
                ImageDialog::new_base64_data(data.as_str())
            } else {
                ArticleViewColumn::instance()
                    .article()
                    .map(|article| ImageDialog::new_url(&article.article_id().into(), data.as_str()))
            };

            if let Some(dialog) = image_dialog {
                self.view.set_image_dialog_visible(true);

                dialog.connect_closed(|_dialog| {
                    Webview::instance().set_image_dialog_visible(false);
                });
                dialog.present(Some(&MainWindow::instance()))
            }
        }

        fn set_article(&self, new_article: Option<GArticle>) {
            let Some(new_article) = new_article else {
                self.close_article();
                return;
            };

            let obj = self.obj();

            let prefer_scraped_content_differs = self.prefer_scraped_content.get() != new_article.has_scraped_content();
            obj.set_prefer_scraped_content(new_article.has_scraped_content());

            let is_new_article = if let Some(current_article) = self.article.borrow().as_ref() {
                current_article.article_id() != new_article.article_id()
                    || current_article.feed_title() != new_article.feed_title()
                    || current_article.has_scraped_content() != new_article.has_scraped_content()
                    || prefer_scraped_content_differs
            } else {
                true
            };

            self.view.set_hovered_url(None::<String>);
            self.progress_overlay.set_progress(0.0);

            self.drag_scroll_tracker.stop();

            self.article.replace(Some(new_article));

            if is_new_article {
                let html = self.build_article();
                let base_url = self.get_base_url();

                self.view.set_scroll_pos(0.0);
                self.view.load(html, base_url.as_deref());
            }

            obj.grab_focus();
        }

        fn get_font_size(&self) -> u32 {
            App::default().settings().article_view().font_size()
        }

        fn set_font_size(&self, font_size: u32) {
            tracing::debug!(%font_size, "inrease fontsize");
            App::default().settings().article_view().set_font_size(font_size);
            self.obj().redraw_article();
        }

        pub fn close_article(&self) {
            self.view.clear();
            self.view.set_hovered_url(None::<String>);
            self.progress_overlay.set_progress(0.0);
            self.article.replace(None);

            self.obj().set_prefer_scraped_content(false);
        }

        pub(super) fn build_article(&self) -> String {
            let Some(article) = &*self.article.borrow() else {
                return String::new();
            };

            ArticleBuilder::from_garticle(article, self.prefer_scraped_content.get(), false)
        }

        pub(super) fn get_base_url(&self) -> Option<String> {
            let Some(article) = &*self.article.borrow() else {
                return None;
            };

            let url = article.url().and_then(|url| Url::parse(&url).ok())?;

            match url.origin() {
                Origin::Opaque(_op) => None,
                Origin::Tuple(scheme, host, port) => {
                    let host = match host {
                        Host::Domain(domain) => domain,
                        Host::Ipv4(ipv4) => ipv4.to_string(),
                        Host::Ipv6(ipv6) => ipv6.to_string(),
                    };
                    if port == 80 || port == 443 {
                        Some(format!("{scheme}://{host}/"))
                    } else {
                        Some(format!("{scheme}://{host}:{port}/"))
                    }
                }
            }
        }
    }
}

glib::wrapper! {
    pub struct ArticleView(ObjectSubclass<imp::ArticleView>)
        @extends Widget, Box,
        @implements Accessible, Buildable, ConstraintTarget;
}

impl Default for ArticleView {
    fn default() -> Self {
        glib::Object::new::<Self>()
    }
}

impl ArticleView {
    pub fn instance() -> Self {
        ArticleViewColumn::instance().imp().article_view.get()
    }

    pub fn redraw_article(&self) {
        let imp = self.imp();

        if imp.view.is_empty() {
            tracing::warn!("Can't redraw article view. No article is on display.");
            return;
        }

        imp.drag_scroll_tracker.stop();

        let html = imp.build_article();
        imp.view.load(html, imp.get_base_url().as_deref());
    }

    pub fn update_background_color(&self, color: &RGBA) {
        self.imp().view.update_background_color(color);
        self.redraw_article();
    }

    pub fn scroll_diff(&self, diff: f64) {
        self.imp().view.scroll_diff(diff);
    }

    pub fn clear_cache(&self, oneshot_sender: OneShotSender<()>) {
        self.imp().view.clear_cache(oneshot_sender);
    }

    pub fn reload_user_data(&self) {
        self.imp().view.imp().load_user_data();
    }
}
