{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- GtkWidget is the base class all widgets in GTK+ derive from. It manages the
-- widget lifecycle, states and style.
-- 
-- # Height-for-width Geometry Management # {@/geometry/@-management}
-- 
-- GTK+ uses a height-for-width (and width-for-height) geometry management
-- system. Height-for-width means that a widget can change how much
-- vertical space it needs, depending on the amount of horizontal space
-- that it is given (and similar for width-for-height). The most common
-- example is a label that reflows to fill up the available width, wraps
-- to fewer lines, and therefore needs less height.
-- 
-- Height-for-width geometry management is implemented in GTK+ by way
-- of five virtual methods:
-- 
-- * t'GI.Gtk.Structs.WidgetClass.WidgetClass'.@/get_request_mode/@()
-- * t'GI.Gtk.Structs.WidgetClass.WidgetClass'.@/get_preferred_width/@()
-- * t'GI.Gtk.Structs.WidgetClass.WidgetClass'.@/get_preferred_height/@()
-- * t'GI.Gtk.Structs.WidgetClass.WidgetClass'.@/get_preferred_height_for_width/@()
-- * t'GI.Gtk.Structs.WidgetClass.WidgetClass'.@/get_preferred_width_for_height/@()
-- * t'GI.Gtk.Structs.WidgetClass.WidgetClass'.@/get_preferred_height_and_baseline_for_width/@()
-- 
-- 
-- There are some important things to keep in mind when implementing
-- height-for-width and when using it in container implementations.
-- 
-- The geometry management system will query a widget hierarchy in
-- only one orientation at a time. When widgets are initially queried
-- for their minimum sizes it is generally done in two initial passes
-- in the t'GI.Gtk.Enums.SizeRequestMode' chosen by the toplevel.
-- 
-- For example, when queried in the normal
-- 'GI.Gtk.Enums.SizeRequestModeHeightForWidth' mode:
-- First, the default minimum and natural width for each widget
-- in the interface will be computed using 'GI.Gtk.Objects.Widget.widgetGetPreferredWidth'.
-- Because the preferred widths for each container depend on the preferred
-- widths of their children, this information propagates up the hierarchy,
-- and finally a minimum and natural width is determined for the entire
-- toplevel. Next, the toplevel will use the minimum width to query for the
-- minimum height contextual to that width using
-- 'GI.Gtk.Objects.Widget.widgetGetPreferredHeightForWidth', which will also be a highly
-- recursive operation. The minimum height for the minimum width is normally
-- used to set the minimum size constraint on the toplevel
-- (unless 'GI.Gtk.Objects.Window.windowSetGeometryHints' is explicitly used instead).
-- 
-- After the toplevel window has initially requested its size in both
-- dimensions it can go on to allocate itself a reasonable size (or a size
-- previously specified with 'GI.Gtk.Objects.Window.windowSetDefaultSize'). During the
-- recursive allocation process it’s important to note that request cycles
-- will be recursively executed while container widgets allocate their children.
-- Each container widget, once allocated a size, will go on to first share the
-- space in one orientation among its children and then request each child\'s
-- height for its target allocated width or its width for allocated height,
-- depending. In this way a t'GI.Gtk.Objects.Widget.Widget' will typically be requested its size
-- a number of times before actually being allocated a size. The size a
-- widget is finally allocated can of course differ from the size it has
-- requested. For this reason, t'GI.Gtk.Objects.Widget.Widget' caches a  small number of results
-- to avoid re-querying for the same sizes in one allocation cycle.
-- 
-- See
-- [GtkContainer’s geometry management section][container-geometry-management]
-- to learn more about how height-for-width allocations are performed
-- by container widgets.
-- 
-- If a widget does move content around to intelligently use up the
-- allocated size then it must support the request in both
-- @/GtkSizeRequestModes/@ even if the widget in question only
-- trades sizes in a single orientation.
-- 
-- For instance, a t'GI.Gtk.Objects.Label.Label' that does height-for-width word wrapping
-- will not expect to have t'GI.Gtk.Structs.WidgetClass.WidgetClass'.@/get_preferred_height/@() called
-- because that call is specific to a width-for-height request. In this
-- case the label must return the height required for its own minimum
-- possible width. By following this rule any widget that handles
-- height-for-width or width-for-height requests will always be allocated
-- at least enough space to fit its own content.
-- 
-- Here are some examples of how a 'GI.Gtk.Enums.SizeRequestModeHeightForWidth' widget
-- generally deals with width-for-height requests, for t'GI.Gtk.Structs.WidgetClass.WidgetClass'.@/get_preferred_height/@()
-- it will do:
-- 
-- 
-- === /C code/
-- >
-- >static void
-- >foo_widget_get_preferred_height (GtkWidget *widget,
-- >                                 gint *min_height,
-- >                                 gint *nat_height)
-- >{
-- >   if (i_am_in_height_for_width_mode)
-- >     {
-- >       gint min_width, nat_width;
-- >
-- >       GTK_WIDGET_GET_CLASS (widget)->get_preferred_width (widget,
-- >                                                           &min_width,
-- >                                                           &nat_width);
-- >       GTK_WIDGET_GET_CLASS (widget)->get_preferred_height_for_width
-- >                                                          (widget,
-- >                                                           min_width,
-- >                                                           min_height,
-- >                                                           nat_height);
-- >     }
-- >   else
-- >     {
-- >        ... some widgets do both. For instance, if a GtkLabel is
-- >        rotated to 90 degrees it will return the minimum and
-- >        natural height for the rotated label here.
-- >     }
-- >}
-- 
-- 
-- And in t'GI.Gtk.Structs.WidgetClass.WidgetClass'.@/get_preferred_width_for_height/@() it will simply return
-- the minimum and natural width:
-- 
-- === /C code/
-- >
-- >static void
-- >foo_widget_get_preferred_width_for_height (GtkWidget *widget,
-- >                                           gint for_height,
-- >                                           gint *min_width,
-- >                                           gint *nat_width)
-- >{
-- >   if (i_am_in_height_for_width_mode)
-- >     {
-- >       GTK_WIDGET_GET_CLASS (widget)->get_preferred_width (widget,
-- >                                                           min_width,
-- >                                                           nat_width);
-- >     }
-- >   else
-- >     {
-- >        ... again if a widget is sometimes operating in
-- >        width-for-height mode (like a rotated GtkLabel) it can go
-- >        ahead and do its real width for height calculation here.
-- >     }
-- >}
-- 
-- 
-- Often a widget needs to get its own request during size request or
-- allocation. For example, when computing height it may need to also
-- compute width. Or when deciding how to use an allocation, the widget
-- may need to know its natural size. In these cases, the widget should
-- be careful to call its virtual methods directly, like this:
-- 
-- 
-- === /C code/
-- >
-- >GTK_WIDGET_GET_CLASS(widget)->get_preferred_width (widget,
-- >                                                   &min,
-- >                                                   &natural);
-- 
-- 
-- It will not work to use the wrapper functions, such as
-- 'GI.Gtk.Objects.Widget.widgetGetPreferredWidth' inside your own size request
-- implementation. These return a request adjusted by t'GI.Gtk.Objects.SizeGroup.SizeGroup'
-- and by the t'GI.Gtk.Structs.WidgetClass.WidgetClass'.@/adjust_size_request/@() virtual method. If a
-- widget used the wrappers inside its virtual method implementations,
-- then the adjustments (such as widget margins) would be applied
-- twice. GTK+ therefore does not allow this and will warn if you try
-- to do it.
-- 
-- Of course if you are getting the size request for
-- another widget, such as a child of a
-- container, you must use the wrapper APIs.
-- Otherwise, you would not properly consider widget margins,
-- t'GI.Gtk.Objects.SizeGroup.SizeGroup', and so forth.
-- 
-- Since 3.10 GTK+ also supports baseline vertical alignment of widgets. This
-- means that widgets are positioned such that the typographical baseline of
-- widgets in the same row are aligned. This happens if a widget supports baselines,
-- has a vertical alignment of 'GI.Gtk.Enums.AlignBaseline', and is inside a container
-- that supports baselines and has a natural “row” that it aligns to the baseline,
-- or a baseline assigned to it by the grandparent.
-- 
-- Baseline alignment support for a widget is done by the t'GI.Gtk.Structs.WidgetClass.WidgetClass'.@/get_preferred_height_and_baseline_for_width/@()
-- virtual function. It allows you to report a baseline in combination with the
-- minimum and natural height. If there is no baseline you can return -1 to indicate
-- this. The default implementation of this virtual function calls into the
-- t'GI.Gtk.Structs.WidgetClass.WidgetClass'.@/get_preferred_height/@() and t'GI.Gtk.Structs.WidgetClass.WidgetClass'.@/get_preferred_height_for_width/@(),
-- so if baselines are not supported it doesn’t need to be implemented.
-- 
-- If a widget ends up baseline aligned it will be allocated all the space in the parent
-- as if it was 'GI.Gtk.Enums.AlignFill', but the selected baseline can be found via 'GI.Gtk.Objects.Widget.widgetGetAllocatedBaseline'.
-- If this has a value other than -1 you need to align the widget such that the baseline
-- appears at the position.
-- 
-- = Style Properties
-- 
-- t'GI.Gtk.Objects.Widget.Widget' introduces “style
-- properties” - these are basically object properties that are stored
-- not on the object, but in the style object associated to the widget. Style
-- properties are set in [resource files][gtk3-Resource-Files].
-- This mechanism is used for configuring such things as the location of the
-- scrollbar arrows through the theme, giving theme authors more control over the
-- look of applications without the need to write a theme engine in C.
-- 
-- Use 'GI.Gtk.Structs.WidgetClass.widgetClassInstallStyleProperty' to install style properties for
-- a widget class, 'GI.Gtk.Structs.WidgetClass.widgetClassFindStyleProperty' or
-- 'GI.Gtk.Structs.WidgetClass.widgetClassListStyleProperties' to get information about existing
-- style properties and 'GI.Gtk.Objects.Widget.widgetStyleGetProperty', @/gtk_widget_style_get()/@ or
-- @/gtk_widget_style_get_valist()/@ to obtain the value of a style property.
-- 
-- = GtkWidget as GtkBuildable
-- 
-- The GtkWidget implementation of the GtkBuildable interface supports a
-- custom @\<accelerator>@ element, which has attributes named ”key”, ”modifiers”
-- and ”signal” and allows to specify accelerators.
-- 
-- An example of a UI definition fragment specifying an accelerator:
-- 
-- 
-- === /xml code/
-- >
-- ><object class="GtkButton">
-- >  <accelerator key="q" modifiers="GDK_CONTROL_MASK" signal="clicked"/>
-- ></object>
-- 
-- 
-- In addition to accelerators, GtkWidget also support a custom @\<accessible>@
-- element, which supports actions and relations. Properties on the accessible
-- implementation of an object can be set by accessing the internal child
-- “accessible” of a t'GI.Gtk.Objects.Widget.Widget'.
-- 
-- An example of a UI definition fragment specifying an accessible:
-- 
-- 
-- === /xml code/
-- >
-- ><object class="GtkLabel" id="label1"/>
-- >  <property name="label">I am a Label for a Button</property>
-- ></object>
-- ><object class="GtkButton" id="button1">
-- >  <accessibility>
-- >    <action action_name="click" translatable="yes">Click the button.</action>
-- >    <relation target="label1" type="labelled-by"/>
-- >  </accessibility>
-- >  <child internal-child="accessible">
-- >    <object class="AtkObject" id="a11y-button1">
-- >      <property name="accessible-name">Clickable Button</property>
-- >    </object>
-- >  </child>
-- ></object>
-- 
-- 
-- Finally, GtkWidget allows style information such as style classes to
-- be associated with widgets, using the custom @\<style>@ element:
-- 
-- 
-- === /xml code/
-- >
-- ><object class="GtkButton" id="button1">
-- >  <style>
-- >    <class name="my-special-button-class"/>
-- >    <class name="dark-button"/>
-- >  </style>
-- ></object>
-- 
-- 
-- # Building composite widgets from template XML ## {@/composite/@-templates}
-- 
-- GtkWidget exposes some facilities to automate the procedure
-- of creating composite widgets using t'GI.Gtk.Objects.Builder.Builder' interface description
-- language.
-- 
-- To create composite widgets with t'GI.Gtk.Objects.Builder.Builder' XML, one must associate
-- the interface description with the widget class at class initialization
-- time using 'GI.Gtk.Structs.WidgetClass.widgetClassSetTemplate'.
-- 
-- The interface description semantics expected in composite template descriptions
-- is slightly different from regular t'GI.Gtk.Objects.Builder.Builder' XML.
-- 
-- Unlike regular interface descriptions, 'GI.Gtk.Structs.WidgetClass.widgetClassSetTemplate' will
-- expect a @\<template>@ tag as a direct child of the toplevel @\<interface>@
-- tag. The @\<template>@ tag must specify the “class” attribute which must be
-- the type name of the widget. Optionally, the “parent” attribute may be
-- specified to specify the direct parent type of the widget type, this is
-- ignored by the GtkBuilder but required for Glade to introspect what kind
-- of properties and internal children exist for a given type when the actual
-- type does not exist.
-- 
-- The XML which is contained inside the @\<template>@ tag behaves as if it were
-- added to the @\<object>@ tag defining \"widget\" itself. You may set properties
-- on /@widget@/ by inserting @\<property>@ tags into the @\<template>@ tag, and also
-- add @\<child>@ tags to add children and extend \"widget\" in the normal way you
-- would with @\<object>@ tags.
-- 
-- Additionally, @\<object>@ tags can also be added before and after the initial
-- @\<template>@ tag in the normal way, allowing one to define auxiliary objects
-- which might be referenced by other widgets declared as children of the
-- @\<template>@ tag.
-- 
-- An example of a GtkBuilder Template Definition:
-- 
-- 
-- === /xml code/
-- >
-- ><interface>
-- >  <template class="FooWidget" parent="GtkBox">
-- >    <property name="orientation">GTK_ORIENTATION_HORIZONTAL</property>
-- >    <property name="spacing">4</property>
-- >    <child>
-- >      <object class="GtkButton" id="hello_button">
-- >        <property name="label">Hello World</property>
-- >        <signal name="clicked" handler="hello_button_clicked" object="FooWidget" swapped="yes"/>
-- >      </object>
-- >    </child>
-- >    <child>
-- >      <object class="GtkButton" id="goodbye_button">
-- >        <property name="label">Goodbye World</property>
-- >      </object>
-- >    </child>
-- >  </template>
-- ></interface>
-- 
-- 
-- Typically, you\'ll place the template fragment into a file that is
-- bundled with your project, using t'GI.Gio.Structs.Resource.Resource'. In order to load the
-- template, you need to call 'GI.Gtk.Structs.WidgetClass.widgetClassSetTemplateFromResource'
-- from the class initialization of your t'GI.Gtk.Objects.Widget.Widget' type:
-- 
-- 
-- === /C code/
-- >
-- >static void
-- >foo_widget_class_init (FooWidgetClass *klass)
-- >{
-- >  // ...
-- >
-- >  gtk_widget_class_set_template_from_resource (GTK_WIDGET_CLASS (klass),
-- >                                               "/com/example/ui/foowidget.ui");
-- >}
-- 
-- 
-- You will also need to call 'GI.Gtk.Objects.Widget.widgetInitTemplate' from the instance
-- initialization function:
-- 
-- 
-- === /C code/
-- >
-- >static void
-- >foo_widget_init (FooWidget *self)
-- >{
-- >  // ...
-- >  gtk_widget_init_template (GTK_WIDGET (self));
-- >}
-- 
-- 
-- You can access widgets defined in the template using the
-- 'GI.Gtk.Objects.Widget.widgetGetTemplateChild' function, but you will typically declare
-- a pointer in the instance private data structure of your type using the same
-- name as the widget in the template definition, and call
-- @/gtk_widget_class_bind_template_child_private()/@ with that name, e.g.
-- 
-- 
-- === /C code/
-- >
-- >typedef struct {
-- >  GtkWidget *hello_button;
-- >  GtkWidget *goodbye_button;
-- >} FooWidgetPrivate;
-- >
-- >G_DEFINE_TYPE_WITH_PRIVATE (FooWidget, foo_widget, GTK_TYPE_BOX)
-- >
-- >static void
-- >foo_widget_class_init (FooWidgetClass *klass)
-- >{
-- >  // ...
-- >  gtk_widget_class_set_template_from_resource (GTK_WIDGET_CLASS (klass),
-- >                                               "/com/example/ui/foowidget.ui");
-- >  gtk_widget_class_bind_template_child_private (GTK_WIDGET_CLASS (klass),
-- >                                                FooWidget, hello_button);
-- >  gtk_widget_class_bind_template_child_private (GTK_WIDGET_CLASS (klass),
-- >                                                FooWidget, goodbye_button);
-- >}
-- >
-- >static void
-- >foo_widget_init (FooWidget *widget)
-- >{
-- >
-- >}
-- 
-- 
-- You can also use @/gtk_widget_class_bind_template_callback()/@ to connect a signal
-- callback defined in the template with a function visible in the scope of the
-- class, e.g.
-- 
-- 
-- === /C code/
-- >
-- >// the signal handler has the instance and user data swapped
-- >// because of the swapped="yes" attribute in the template XML
-- >static void
-- >hello_button_clicked (FooWidget *self,
-- >                      GtkButton *button)
-- >{
-- >  g_print ("Hello, world!\n");
-- >}
-- >
-- >static void
-- >foo_widget_class_init (FooWidgetClass *klass)
-- >{
-- >  // ...
-- >  gtk_widget_class_set_template_from_resource (GTK_WIDGET_CLASS (klass),
-- >                                               "/com/example/ui/foowidget.ui");
-- >  gtk_widget_class_bind_template_callback (GTK_WIDGET_CLASS (klass), hello_button_clicked);
-- >}
-- 

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gtk.Objects.Widget
    ( 

-- * Exported types
    Widget(..)                              ,
    IsWidget                                ,
    toWidget                                ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [activate]("GI.Gtk.Objects.Widget#g:method:activate"), [addAccelerator]("GI.Gtk.Objects.Widget#g:method:addAccelerator"), [addChild]("GI.Gtk.Interfaces.Buildable#g:method:addChild"), [addDeviceEvents]("GI.Gtk.Objects.Widget#g:method:addDeviceEvents"), [addEvents]("GI.Gtk.Objects.Widget#g:method:addEvents"), [addMnemonicLabel]("GI.Gtk.Objects.Widget#g:method:addMnemonicLabel"), [addTickCallback]("GI.Gtk.Objects.Widget#g:method:addTickCallback"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [canActivateAccel]("GI.Gtk.Objects.Widget#g:method:canActivateAccel"), [childFocus]("GI.Gtk.Objects.Widget#g:method:childFocus"), [childNotify]("GI.Gtk.Objects.Widget#g:method:childNotify"), [classPath]("GI.Gtk.Objects.Widget#g:method:classPath"), [computeExpand]("GI.Gtk.Objects.Widget#g:method:computeExpand"), [constructChild]("GI.Gtk.Interfaces.Buildable#g:method:constructChild"), [createPangoContext]("GI.Gtk.Objects.Widget#g:method:createPangoContext"), [createPangoLayout]("GI.Gtk.Objects.Widget#g:method:createPangoLayout"), [customFinished]("GI.Gtk.Interfaces.Buildable#g:method:customFinished"), [customTagEnd]("GI.Gtk.Interfaces.Buildable#g:method:customTagEnd"), [customTagStart]("GI.Gtk.Interfaces.Buildable#g:method:customTagStart"), [destroy]("GI.Gtk.Objects.Widget#g:method:destroy"), [destroyed]("GI.Gtk.Objects.Widget#g:method:destroyed"), [deviceIsShadowed]("GI.Gtk.Objects.Widget#g:method:deviceIsShadowed"), [dragBegin]("GI.Gtk.Objects.Widget#g:method:dragBegin"), [dragBeginWithCoordinates]("GI.Gtk.Objects.Widget#g:method:dragBeginWithCoordinates"), [dragCheckThreshold]("GI.Gtk.Objects.Widget#g:method:dragCheckThreshold"), [dragDestAddImageTargets]("GI.Gtk.Objects.Widget#g:method:dragDestAddImageTargets"), [dragDestAddTextTargets]("GI.Gtk.Objects.Widget#g:method:dragDestAddTextTargets"), [dragDestAddUriTargets]("GI.Gtk.Objects.Widget#g:method:dragDestAddUriTargets"), [dragDestFindTarget]("GI.Gtk.Objects.Widget#g:method:dragDestFindTarget"), [dragDestGetTargetList]("GI.Gtk.Objects.Widget#g:method:dragDestGetTargetList"), [dragDestGetTrackMotion]("GI.Gtk.Objects.Widget#g:method:dragDestGetTrackMotion"), [dragDestSet]("GI.Gtk.Objects.Widget#g:method:dragDestSet"), [dragDestSetProxy]("GI.Gtk.Objects.Widget#g:method:dragDestSetProxy"), [dragDestSetTargetList]("GI.Gtk.Objects.Widget#g:method:dragDestSetTargetList"), [dragDestSetTrackMotion]("GI.Gtk.Objects.Widget#g:method:dragDestSetTrackMotion"), [dragDestUnset]("GI.Gtk.Objects.Widget#g:method:dragDestUnset"), [dragGetData]("GI.Gtk.Objects.Widget#g:method:dragGetData"), [dragHighlight]("GI.Gtk.Objects.Widget#g:method:dragHighlight"), [dragSourceAddImageTargets]("GI.Gtk.Objects.Widget#g:method:dragSourceAddImageTargets"), [dragSourceAddTextTargets]("GI.Gtk.Objects.Widget#g:method:dragSourceAddTextTargets"), [dragSourceAddUriTargets]("GI.Gtk.Objects.Widget#g:method:dragSourceAddUriTargets"), [dragSourceGetTargetList]("GI.Gtk.Objects.Widget#g:method:dragSourceGetTargetList"), [dragSourceSet]("GI.Gtk.Objects.Widget#g:method:dragSourceSet"), [dragSourceSetIconGicon]("GI.Gtk.Objects.Widget#g:method:dragSourceSetIconGicon"), [dragSourceSetIconName]("GI.Gtk.Objects.Widget#g:method:dragSourceSetIconName"), [dragSourceSetIconPixbuf]("GI.Gtk.Objects.Widget#g:method:dragSourceSetIconPixbuf"), [dragSourceSetIconStock]("GI.Gtk.Objects.Widget#g:method:dragSourceSetIconStock"), [dragSourceSetTargetList]("GI.Gtk.Objects.Widget#g:method:dragSourceSetTargetList"), [dragSourceUnset]("GI.Gtk.Objects.Widget#g:method:dragSourceUnset"), [dragUnhighlight]("GI.Gtk.Objects.Widget#g:method:dragUnhighlight"), [draw]("GI.Gtk.Objects.Widget#g:method:draw"), [ensureStyle]("GI.Gtk.Objects.Widget#g:method:ensureStyle"), [errorBell]("GI.Gtk.Objects.Widget#g:method:errorBell"), [event]("GI.Gtk.Objects.Widget#g:method:event"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeChildNotify]("GI.Gtk.Objects.Widget#g:method:freezeChildNotify"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [grabAdd]("GI.Gtk.Objects.Widget#g:method:grabAdd"), [grabDefault]("GI.Gtk.Objects.Widget#g:method:grabDefault"), [grabFocus]("GI.Gtk.Objects.Widget#g:method:grabFocus"), [grabRemove]("GI.Gtk.Objects.Widget#g:method:grabRemove"), [hasDefault]("GI.Gtk.Objects.Widget#g:method:hasDefault"), [hasFocus]("GI.Gtk.Objects.Widget#g:method:hasFocus"), [hasGrab]("GI.Gtk.Objects.Widget#g:method:hasGrab"), [hasRcStyle]("GI.Gtk.Objects.Widget#g:method:hasRcStyle"), [hasScreen]("GI.Gtk.Objects.Widget#g:method:hasScreen"), [hasVisibleFocus]("GI.Gtk.Objects.Widget#g:method:hasVisibleFocus"), [hide]("GI.Gtk.Objects.Widget#g:method:hide"), [hideOnDelete]("GI.Gtk.Objects.Widget#g:method:hideOnDelete"), [inDestruction]("GI.Gtk.Objects.Widget#g:method:inDestruction"), [initTemplate]("GI.Gtk.Objects.Widget#g:method:initTemplate"), [inputShapeCombineRegion]("GI.Gtk.Objects.Widget#g:method:inputShapeCombineRegion"), [insertActionGroup]("GI.Gtk.Objects.Widget#g:method:insertActionGroup"), [intersect]("GI.Gtk.Objects.Widget#g:method:intersect"), [isAncestor]("GI.Gtk.Objects.Widget#g:method:isAncestor"), [isComposited]("GI.Gtk.Objects.Widget#g:method:isComposited"), [isDrawable]("GI.Gtk.Objects.Widget#g:method:isDrawable"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [isFocus]("GI.Gtk.Objects.Widget#g:method:isFocus"), [isSensitive]("GI.Gtk.Objects.Widget#g:method:isSensitive"), [isToplevel]("GI.Gtk.Objects.Widget#g:method:isToplevel"), [isVisible]("GI.Gtk.Objects.Widget#g:method:isVisible"), [keynavFailed]("GI.Gtk.Objects.Widget#g:method:keynavFailed"), [listAccelClosures]("GI.Gtk.Objects.Widget#g:method:listAccelClosures"), [listActionPrefixes]("GI.Gtk.Objects.Widget#g:method:listActionPrefixes"), [listMnemonicLabels]("GI.Gtk.Objects.Widget#g:method:listMnemonicLabels"), [map]("GI.Gtk.Objects.Widget#g:method:map"), [mnemonicActivate]("GI.Gtk.Objects.Widget#g:method:mnemonicActivate"), [modifyBase]("GI.Gtk.Objects.Widget#g:method:modifyBase"), [modifyBg]("GI.Gtk.Objects.Widget#g:method:modifyBg"), [modifyCursor]("GI.Gtk.Objects.Widget#g:method:modifyCursor"), [modifyFg]("GI.Gtk.Objects.Widget#g:method:modifyFg"), [modifyFont]("GI.Gtk.Objects.Widget#g:method:modifyFont"), [modifyStyle]("GI.Gtk.Objects.Widget#g:method:modifyStyle"), [modifyText]("GI.Gtk.Objects.Widget#g:method:modifyText"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [overrideBackgroundColor]("GI.Gtk.Objects.Widget#g:method:overrideBackgroundColor"), [overrideColor]("GI.Gtk.Objects.Widget#g:method:overrideColor"), [overrideCursor]("GI.Gtk.Objects.Widget#g:method:overrideCursor"), [overrideFont]("GI.Gtk.Objects.Widget#g:method:overrideFont"), [overrideSymbolicColor]("GI.Gtk.Objects.Widget#g:method:overrideSymbolicColor"), [parserFinished]("GI.Gtk.Interfaces.Buildable#g:method:parserFinished"), [path]("GI.Gtk.Objects.Widget#g:method:path"), [queueAllocate]("GI.Gtk.Objects.Widget#g:method:queueAllocate"), [queueComputeExpand]("GI.Gtk.Objects.Widget#g:method:queueComputeExpand"), [queueDraw]("GI.Gtk.Objects.Widget#g:method:queueDraw"), [queueDrawArea]("GI.Gtk.Objects.Widget#g:method:queueDrawArea"), [queueDrawRegion]("GI.Gtk.Objects.Widget#g:method:queueDrawRegion"), [queueResize]("GI.Gtk.Objects.Widget#g:method:queueResize"), [queueResizeNoRedraw]("GI.Gtk.Objects.Widget#g:method:queueResizeNoRedraw"), [realize]("GI.Gtk.Objects.Widget#g:method:realize"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [regionIntersect]("GI.Gtk.Objects.Widget#g:method:regionIntersect"), [registerWindow]("GI.Gtk.Objects.Widget#g:method:registerWindow"), [removeAccelerator]("GI.Gtk.Objects.Widget#g:method:removeAccelerator"), [removeMnemonicLabel]("GI.Gtk.Objects.Widget#g:method:removeMnemonicLabel"), [removeTickCallback]("GI.Gtk.Objects.Widget#g:method:removeTickCallback"), [renderIcon]("GI.Gtk.Objects.Widget#g:method:renderIcon"), [renderIconPixbuf]("GI.Gtk.Objects.Widget#g:method:renderIconPixbuf"), [reparent]("GI.Gtk.Objects.Widget#g:method:reparent"), [resetRcStyles]("GI.Gtk.Objects.Widget#g:method:resetRcStyles"), [resetStyle]("GI.Gtk.Objects.Widget#g:method:resetStyle"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [sendExpose]("GI.Gtk.Objects.Widget#g:method:sendExpose"), [sendFocusChange]("GI.Gtk.Objects.Widget#g:method:sendFocusChange"), [shapeCombineRegion]("GI.Gtk.Objects.Widget#g:method:shapeCombineRegion"), [show]("GI.Gtk.Objects.Widget#g:method:show"), [showAll]("GI.Gtk.Objects.Widget#g:method:showAll"), [showNow]("GI.Gtk.Objects.Widget#g:method:showNow"), [sizeAllocate]("GI.Gtk.Objects.Widget#g:method:sizeAllocate"), [sizeAllocateWithBaseline]("GI.Gtk.Objects.Widget#g:method:sizeAllocateWithBaseline"), [sizeRequest]("GI.Gtk.Objects.Widget#g:method:sizeRequest"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [styleAttach]("GI.Gtk.Objects.Widget#g:method:styleAttach"), [styleGetProperty]("GI.Gtk.Objects.Widget#g:method:styleGetProperty"), [thawChildNotify]("GI.Gtk.Objects.Widget#g:method:thawChildNotify"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [translateCoordinates]("GI.Gtk.Objects.Widget#g:method:translateCoordinates"), [triggerTooltipQuery]("GI.Gtk.Objects.Widget#g:method:triggerTooltipQuery"), [unmap]("GI.Gtk.Objects.Widget#g:method:unmap"), [unparent]("GI.Gtk.Objects.Widget#g:method:unparent"), [unrealize]("GI.Gtk.Objects.Widget#g:method:unrealize"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [unregisterWindow]("GI.Gtk.Objects.Widget#g:method:unregisterWindow"), [unsetStateFlags]("GI.Gtk.Objects.Widget#g:method:unsetStateFlags"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getAccessible]("GI.Gtk.Objects.Widget#g:method:getAccessible"), [getActionGroup]("GI.Gtk.Objects.Widget#g:method:getActionGroup"), [getAllocatedBaseline]("GI.Gtk.Objects.Widget#g:method:getAllocatedBaseline"), [getAllocatedHeight]("GI.Gtk.Objects.Widget#g:method:getAllocatedHeight"), [getAllocatedSize]("GI.Gtk.Objects.Widget#g:method:getAllocatedSize"), [getAllocatedWidth]("GI.Gtk.Objects.Widget#g:method:getAllocatedWidth"), [getAllocation]("GI.Gtk.Objects.Widget#g:method:getAllocation"), [getAncestor]("GI.Gtk.Objects.Widget#g:method:getAncestor"), [getAppPaintable]("GI.Gtk.Objects.Widget#g:method:getAppPaintable"), [getCanDefault]("GI.Gtk.Objects.Widget#g:method:getCanDefault"), [getCanFocus]("GI.Gtk.Objects.Widget#g:method:getCanFocus"), [getChildRequisition]("GI.Gtk.Objects.Widget#g:method:getChildRequisition"), [getChildVisible]("GI.Gtk.Objects.Widget#g:method:getChildVisible"), [getClip]("GI.Gtk.Objects.Widget#g:method:getClip"), [getClipboard]("GI.Gtk.Objects.Widget#g:method:getClipboard"), [getCompositeName]("GI.Gtk.Objects.Widget#g:method:getCompositeName"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getDeviceEnabled]("GI.Gtk.Objects.Widget#g:method:getDeviceEnabled"), [getDeviceEvents]("GI.Gtk.Objects.Widget#g:method:getDeviceEvents"), [getDirection]("GI.Gtk.Objects.Widget#g:method:getDirection"), [getDisplay]("GI.Gtk.Objects.Widget#g:method:getDisplay"), [getDoubleBuffered]("GI.Gtk.Objects.Widget#g:method:getDoubleBuffered"), [getEvents]("GI.Gtk.Objects.Widget#g:method:getEvents"), [getFocusOnClick]("GI.Gtk.Objects.Widget#g:method:getFocusOnClick"), [getFontMap]("GI.Gtk.Objects.Widget#g:method:getFontMap"), [getFontOptions]("GI.Gtk.Objects.Widget#g:method:getFontOptions"), [getFrameClock]("GI.Gtk.Objects.Widget#g:method:getFrameClock"), [getHalign]("GI.Gtk.Objects.Widget#g:method:getHalign"), [getHasTooltip]("GI.Gtk.Objects.Widget#g:method:getHasTooltip"), [getHasWindow]("GI.Gtk.Objects.Widget#g:method:getHasWindow"), [getHexpand]("GI.Gtk.Objects.Widget#g:method:getHexpand"), [getHexpandSet]("GI.Gtk.Objects.Widget#g:method:getHexpandSet"), [getInternalChild]("GI.Gtk.Interfaces.Buildable#g:method:getInternalChild"), [getMapped]("GI.Gtk.Objects.Widget#g:method:getMapped"), [getMarginBottom]("GI.Gtk.Objects.Widget#g:method:getMarginBottom"), [getMarginEnd]("GI.Gtk.Objects.Widget#g:method:getMarginEnd"), [getMarginLeft]("GI.Gtk.Objects.Widget#g:method:getMarginLeft"), [getMarginRight]("GI.Gtk.Objects.Widget#g:method:getMarginRight"), [getMarginStart]("GI.Gtk.Objects.Widget#g:method:getMarginStart"), [getMarginTop]("GI.Gtk.Objects.Widget#g:method:getMarginTop"), [getModifierMask]("GI.Gtk.Objects.Widget#g:method:getModifierMask"), [getModifierStyle]("GI.Gtk.Objects.Widget#g:method:getModifierStyle"), [getName]("GI.Gtk.Objects.Widget#g:method:getName"), [getNoShowAll]("GI.Gtk.Objects.Widget#g:method:getNoShowAll"), [getOpacity]("GI.Gtk.Objects.Widget#g:method:getOpacity"), [getPangoContext]("GI.Gtk.Objects.Widget#g:method:getPangoContext"), [getParent]("GI.Gtk.Objects.Widget#g:method:getParent"), [getParentWindow]("GI.Gtk.Objects.Widget#g:method:getParentWindow"), [getPath]("GI.Gtk.Objects.Widget#g:method:getPath"), [getPointer]("GI.Gtk.Objects.Widget#g:method:getPointer"), [getPreferredHeight]("GI.Gtk.Objects.Widget#g:method:getPreferredHeight"), [getPreferredHeightAndBaselineForWidth]("GI.Gtk.Objects.Widget#g:method:getPreferredHeightAndBaselineForWidth"), [getPreferredHeightForWidth]("GI.Gtk.Objects.Widget#g:method:getPreferredHeightForWidth"), [getPreferredSize]("GI.Gtk.Objects.Widget#g:method:getPreferredSize"), [getPreferredWidth]("GI.Gtk.Objects.Widget#g:method:getPreferredWidth"), [getPreferredWidthForHeight]("GI.Gtk.Objects.Widget#g:method:getPreferredWidthForHeight"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getRealized]("GI.Gtk.Objects.Widget#g:method:getRealized"), [getReceivesDefault]("GI.Gtk.Objects.Widget#g:method:getReceivesDefault"), [getRequestMode]("GI.Gtk.Objects.Widget#g:method:getRequestMode"), [getRequisition]("GI.Gtk.Objects.Widget#g:method:getRequisition"), [getRootWindow]("GI.Gtk.Objects.Widget#g:method:getRootWindow"), [getScaleFactor]("GI.Gtk.Objects.Widget#g:method:getScaleFactor"), [getScreen]("GI.Gtk.Objects.Widget#g:method:getScreen"), [getSensitive]("GI.Gtk.Objects.Widget#g:method:getSensitive"), [getSettings]("GI.Gtk.Objects.Widget#g:method:getSettings"), [getSizeRequest]("GI.Gtk.Objects.Widget#g:method:getSizeRequest"), [getState]("GI.Gtk.Objects.Widget#g:method:getState"), [getStateFlags]("GI.Gtk.Objects.Widget#g:method:getStateFlags"), [getStyle]("GI.Gtk.Objects.Widget#g:method:getStyle"), [getStyleContext]("GI.Gtk.Objects.Widget#g:method:getStyleContext"), [getSupportMultidevice]("GI.Gtk.Objects.Widget#g:method:getSupportMultidevice"), [getTemplateChild]("GI.Gtk.Objects.Widget#g:method:getTemplateChild"), [getTooltipMarkup]("GI.Gtk.Objects.Widget#g:method:getTooltipMarkup"), [getTooltipText]("GI.Gtk.Objects.Widget#g:method:getTooltipText"), [getTooltipWindow]("GI.Gtk.Objects.Widget#g:method:getTooltipWindow"), [getToplevel]("GI.Gtk.Objects.Widget#g:method:getToplevel"), [getValign]("GI.Gtk.Objects.Widget#g:method:getValign"), [getValignWithBaseline]("GI.Gtk.Objects.Widget#g:method:getValignWithBaseline"), [getVexpand]("GI.Gtk.Objects.Widget#g:method:getVexpand"), [getVexpandSet]("GI.Gtk.Objects.Widget#g:method:getVexpandSet"), [getVisible]("GI.Gtk.Objects.Widget#g:method:getVisible"), [getVisual]("GI.Gtk.Objects.Widget#g:method:getVisual"), [getWindow]("GI.Gtk.Objects.Widget#g:method:getWindow").
-- 
-- ==== Setters
-- [setAccelPath]("GI.Gtk.Objects.Widget#g:method:setAccelPath"), [setAllocation]("GI.Gtk.Objects.Widget#g:method:setAllocation"), [setAppPaintable]("GI.Gtk.Objects.Widget#g:method:setAppPaintable"), [setBuildableProperty]("GI.Gtk.Interfaces.Buildable#g:method:setBuildableProperty"), [setCanDefault]("GI.Gtk.Objects.Widget#g:method:setCanDefault"), [setCanFocus]("GI.Gtk.Objects.Widget#g:method:setCanFocus"), [setChildVisible]("GI.Gtk.Objects.Widget#g:method:setChildVisible"), [setClip]("GI.Gtk.Objects.Widget#g:method:setClip"), [setCompositeName]("GI.Gtk.Objects.Widget#g:method:setCompositeName"), [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setDeviceEnabled]("GI.Gtk.Objects.Widget#g:method:setDeviceEnabled"), [setDeviceEvents]("GI.Gtk.Objects.Widget#g:method:setDeviceEvents"), [setDirection]("GI.Gtk.Objects.Widget#g:method:setDirection"), [setDoubleBuffered]("GI.Gtk.Objects.Widget#g:method:setDoubleBuffered"), [setEvents]("GI.Gtk.Objects.Widget#g:method:setEvents"), [setFocusOnClick]("GI.Gtk.Objects.Widget#g:method:setFocusOnClick"), [setFontMap]("GI.Gtk.Objects.Widget#g:method:setFontMap"), [setFontOptions]("GI.Gtk.Objects.Widget#g:method:setFontOptions"), [setHalign]("GI.Gtk.Objects.Widget#g:method:setHalign"), [setHasTooltip]("GI.Gtk.Objects.Widget#g:method:setHasTooltip"), [setHasWindow]("GI.Gtk.Objects.Widget#g:method:setHasWindow"), [setHexpand]("GI.Gtk.Objects.Widget#g:method:setHexpand"), [setHexpandSet]("GI.Gtk.Objects.Widget#g:method:setHexpandSet"), [setMapped]("GI.Gtk.Objects.Widget#g:method:setMapped"), [setMarginBottom]("GI.Gtk.Objects.Widget#g:method:setMarginBottom"), [setMarginEnd]("GI.Gtk.Objects.Widget#g:method:setMarginEnd"), [setMarginLeft]("GI.Gtk.Objects.Widget#g:method:setMarginLeft"), [setMarginRight]("GI.Gtk.Objects.Widget#g:method:setMarginRight"), [setMarginStart]("GI.Gtk.Objects.Widget#g:method:setMarginStart"), [setMarginTop]("GI.Gtk.Objects.Widget#g:method:setMarginTop"), [setName]("GI.Gtk.Objects.Widget#g:method:setName"), [setNoShowAll]("GI.Gtk.Objects.Widget#g:method:setNoShowAll"), [setOpacity]("GI.Gtk.Objects.Widget#g:method:setOpacity"), [setParent]("GI.Gtk.Objects.Widget#g:method:setParent"), [setParentWindow]("GI.Gtk.Objects.Widget#g:method:setParentWindow"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setRealized]("GI.Gtk.Objects.Widget#g:method:setRealized"), [setReceivesDefault]("GI.Gtk.Objects.Widget#g:method:setReceivesDefault"), [setRedrawOnAllocate]("GI.Gtk.Objects.Widget#g:method:setRedrawOnAllocate"), [setSensitive]("GI.Gtk.Objects.Widget#g:method:setSensitive"), [setSizeRequest]("GI.Gtk.Objects.Widget#g:method:setSizeRequest"), [setState]("GI.Gtk.Objects.Widget#g:method:setState"), [setStateFlags]("GI.Gtk.Objects.Widget#g:method:setStateFlags"), [setStyle]("GI.Gtk.Objects.Widget#g:method:setStyle"), [setSupportMultidevice]("GI.Gtk.Objects.Widget#g:method:setSupportMultidevice"), [setTooltipMarkup]("GI.Gtk.Objects.Widget#g:method:setTooltipMarkup"), [setTooltipText]("GI.Gtk.Objects.Widget#g:method:setTooltipText"), [setTooltipWindow]("GI.Gtk.Objects.Widget#g:method:setTooltipWindow"), [setValign]("GI.Gtk.Objects.Widget#g:method:setValign"), [setVexpand]("GI.Gtk.Objects.Widget#g:method:setVexpand"), [setVexpandSet]("GI.Gtk.Objects.Widget#g:method:setVexpandSet"), [setVisible]("GI.Gtk.Objects.Widget#g:method:setVisible"), [setVisual]("GI.Gtk.Objects.Widget#g:method:setVisual"), [setWindow]("GI.Gtk.Objects.Widget#g:method:setWindow").

#if defined(ENABLE_OVERLOADING)
    ResolveWidgetMethod                     ,
#endif

-- ** activate #method:activate#

#if defined(ENABLE_OVERLOADING)
    WidgetActivateMethodInfo                ,
#endif
    widgetActivate                          ,


-- ** addAccelerator #method:addAccelerator#

#if defined(ENABLE_OVERLOADING)
    WidgetAddAcceleratorMethodInfo          ,
#endif
    widgetAddAccelerator                    ,


-- ** addDeviceEvents #method:addDeviceEvents#

#if defined(ENABLE_OVERLOADING)
    WidgetAddDeviceEventsMethodInfo         ,
#endif
    widgetAddDeviceEvents                   ,


-- ** addEvents #method:addEvents#

#if defined(ENABLE_OVERLOADING)
    WidgetAddEventsMethodInfo               ,
#endif
    widgetAddEvents                         ,


-- ** addMnemonicLabel #method:addMnemonicLabel#

#if defined(ENABLE_OVERLOADING)
    WidgetAddMnemonicLabelMethodInfo        ,
#endif
    widgetAddMnemonicLabel                  ,


-- ** addTickCallback #method:addTickCallback#

#if defined(ENABLE_OVERLOADING)
    WidgetAddTickCallbackMethodInfo         ,
#endif
    widgetAddTickCallback                   ,


-- ** canActivateAccel #method:canActivateAccel#

#if defined(ENABLE_OVERLOADING)
    WidgetCanActivateAccelMethodInfo        ,
#endif
    widgetCanActivateAccel                  ,


-- ** childFocus #method:childFocus#

#if defined(ENABLE_OVERLOADING)
    WidgetChildFocusMethodInfo              ,
#endif
    widgetChildFocus                        ,


-- ** childNotify #method:childNotify#

#if defined(ENABLE_OVERLOADING)
    WidgetChildNotifyMethodInfo             ,
#endif
    widgetChildNotify                       ,


-- ** classPath #method:classPath#

#if defined(ENABLE_OVERLOADING)
    WidgetClassPathMethodInfo               ,
#endif
    widgetClassPath                         ,


-- ** computeExpand #method:computeExpand#

#if defined(ENABLE_OVERLOADING)
    WidgetComputeExpandMethodInfo           ,
#endif
    widgetComputeExpand                     ,


-- ** createPangoContext #method:createPangoContext#

#if defined(ENABLE_OVERLOADING)
    WidgetCreatePangoContextMethodInfo      ,
#endif
    widgetCreatePangoContext                ,


-- ** createPangoLayout #method:createPangoLayout#

#if defined(ENABLE_OVERLOADING)
    WidgetCreatePangoLayoutMethodInfo       ,
#endif
    widgetCreatePangoLayout                 ,


-- ** destroy #method:destroy#

#if defined(ENABLE_OVERLOADING)
    WidgetDestroyMethodInfo                 ,
#endif
    widgetDestroy                           ,


-- ** destroyed #method:destroyed#

#if defined(ENABLE_OVERLOADING)
    WidgetDestroyedMethodInfo               ,
#endif
    widgetDestroyed                         ,


-- ** deviceIsShadowed #method:deviceIsShadowed#

#if defined(ENABLE_OVERLOADING)
    WidgetDeviceIsShadowedMethodInfo        ,
#endif
    widgetDeviceIsShadowed                  ,


-- ** dragBegin #method:dragBegin#

#if defined(ENABLE_OVERLOADING)
    WidgetDragBeginMethodInfo               ,
#endif
    widgetDragBegin                         ,


-- ** dragBeginWithCoordinates #method:dragBeginWithCoordinates#

#if defined(ENABLE_OVERLOADING)
    WidgetDragBeginWithCoordinatesMethodInfo,
#endif
    widgetDragBeginWithCoordinates          ,


-- ** dragCheckThreshold #method:dragCheckThreshold#

#if defined(ENABLE_OVERLOADING)
    WidgetDragCheckThresholdMethodInfo      ,
#endif
    widgetDragCheckThreshold                ,


-- ** dragDestAddImageTargets #method:dragDestAddImageTargets#

#if defined(ENABLE_OVERLOADING)
    WidgetDragDestAddImageTargetsMethodInfo ,
#endif
    widgetDragDestAddImageTargets           ,


-- ** dragDestAddTextTargets #method:dragDestAddTextTargets#

#if defined(ENABLE_OVERLOADING)
    WidgetDragDestAddTextTargetsMethodInfo  ,
#endif
    widgetDragDestAddTextTargets            ,


-- ** dragDestAddUriTargets #method:dragDestAddUriTargets#

#if defined(ENABLE_OVERLOADING)
    WidgetDragDestAddUriTargetsMethodInfo   ,
#endif
    widgetDragDestAddUriTargets             ,


-- ** dragDestFindTarget #method:dragDestFindTarget#

#if defined(ENABLE_OVERLOADING)
    WidgetDragDestFindTargetMethodInfo      ,
#endif
    widgetDragDestFindTarget                ,


-- ** dragDestGetTargetList #method:dragDestGetTargetList#

#if defined(ENABLE_OVERLOADING)
    WidgetDragDestGetTargetListMethodInfo   ,
#endif
    widgetDragDestGetTargetList             ,


-- ** dragDestGetTrackMotion #method:dragDestGetTrackMotion#

#if defined(ENABLE_OVERLOADING)
    WidgetDragDestGetTrackMotionMethodInfo  ,
#endif
    widgetDragDestGetTrackMotion            ,


-- ** dragDestSet #method:dragDestSet#

#if defined(ENABLE_OVERLOADING)
    WidgetDragDestSetMethodInfo             ,
#endif
    widgetDragDestSet                       ,


-- ** dragDestSetProxy #method:dragDestSetProxy#

#if defined(ENABLE_OVERLOADING)
    WidgetDragDestSetProxyMethodInfo        ,
#endif
    widgetDragDestSetProxy                  ,


-- ** dragDestSetTargetList #method:dragDestSetTargetList#

#if defined(ENABLE_OVERLOADING)
    WidgetDragDestSetTargetListMethodInfo   ,
#endif
    widgetDragDestSetTargetList             ,


-- ** dragDestSetTrackMotion #method:dragDestSetTrackMotion#

#if defined(ENABLE_OVERLOADING)
    WidgetDragDestSetTrackMotionMethodInfo  ,
#endif
    widgetDragDestSetTrackMotion            ,


-- ** dragDestUnset #method:dragDestUnset#

#if defined(ENABLE_OVERLOADING)
    WidgetDragDestUnsetMethodInfo           ,
#endif
    widgetDragDestUnset                     ,


-- ** dragGetData #method:dragGetData#

#if defined(ENABLE_OVERLOADING)
    WidgetDragGetDataMethodInfo             ,
#endif
    widgetDragGetData                       ,


-- ** dragHighlight #method:dragHighlight#

#if defined(ENABLE_OVERLOADING)
    WidgetDragHighlightMethodInfo           ,
#endif
    widgetDragHighlight                     ,


-- ** dragSourceAddImageTargets #method:dragSourceAddImageTargets#

#if defined(ENABLE_OVERLOADING)
    WidgetDragSourceAddImageTargetsMethodInfo,
#endif
    widgetDragSourceAddImageTargets         ,


-- ** dragSourceAddTextTargets #method:dragSourceAddTextTargets#

#if defined(ENABLE_OVERLOADING)
    WidgetDragSourceAddTextTargetsMethodInfo,
#endif
    widgetDragSourceAddTextTargets          ,


-- ** dragSourceAddUriTargets #method:dragSourceAddUriTargets#

#if defined(ENABLE_OVERLOADING)
    WidgetDragSourceAddUriTargetsMethodInfo ,
#endif
    widgetDragSourceAddUriTargets           ,


-- ** dragSourceGetTargetList #method:dragSourceGetTargetList#

#if defined(ENABLE_OVERLOADING)
    WidgetDragSourceGetTargetListMethodInfo ,
#endif
    widgetDragSourceGetTargetList           ,


-- ** dragSourceSet #method:dragSourceSet#

#if defined(ENABLE_OVERLOADING)
    WidgetDragSourceSetMethodInfo           ,
#endif
    widgetDragSourceSet                     ,


-- ** dragSourceSetIconGicon #method:dragSourceSetIconGicon#

#if defined(ENABLE_OVERLOADING)
    WidgetDragSourceSetIconGiconMethodInfo  ,
#endif
    widgetDragSourceSetIconGicon            ,


-- ** dragSourceSetIconName #method:dragSourceSetIconName#

#if defined(ENABLE_OVERLOADING)
    WidgetDragSourceSetIconNameMethodInfo   ,
#endif
    widgetDragSourceSetIconName             ,


-- ** dragSourceSetIconPixbuf #method:dragSourceSetIconPixbuf#

#if defined(ENABLE_OVERLOADING)
    WidgetDragSourceSetIconPixbufMethodInfo ,
#endif
    widgetDragSourceSetIconPixbuf           ,


-- ** dragSourceSetIconStock #method:dragSourceSetIconStock#

#if defined(ENABLE_OVERLOADING)
    WidgetDragSourceSetIconStockMethodInfo  ,
#endif
    widgetDragSourceSetIconStock            ,


-- ** dragSourceSetTargetList #method:dragSourceSetTargetList#

#if defined(ENABLE_OVERLOADING)
    WidgetDragSourceSetTargetListMethodInfo ,
#endif
    widgetDragSourceSetTargetList           ,


-- ** dragSourceUnset #method:dragSourceUnset#

#if defined(ENABLE_OVERLOADING)
    WidgetDragSourceUnsetMethodInfo         ,
#endif
    widgetDragSourceUnset                   ,


-- ** dragUnhighlight #method:dragUnhighlight#

#if defined(ENABLE_OVERLOADING)
    WidgetDragUnhighlightMethodInfo         ,
#endif
    widgetDragUnhighlight                   ,


-- ** draw #method:draw#

#if defined(ENABLE_OVERLOADING)
    WidgetDrawMethodInfo                    ,
#endif
    widgetDraw                              ,


-- ** ensureStyle #method:ensureStyle#

#if defined(ENABLE_OVERLOADING)
    WidgetEnsureStyleMethodInfo             ,
#endif
    widgetEnsureStyle                       ,


-- ** errorBell #method:errorBell#

#if defined(ENABLE_OVERLOADING)
    WidgetErrorBellMethodInfo               ,
#endif
    widgetErrorBell                         ,


-- ** event #method:event#

#if defined(ENABLE_OVERLOADING)
    WidgetEventMethodInfo                   ,
#endif
    widgetEvent                             ,


-- ** freezeChildNotify #method:freezeChildNotify#

#if defined(ENABLE_OVERLOADING)
    WidgetFreezeChildNotifyMethodInfo       ,
#endif
    widgetFreezeChildNotify                 ,


-- ** getAccessible #method:getAccessible#

#if defined(ENABLE_OVERLOADING)
    WidgetGetAccessibleMethodInfo           ,
#endif
    widgetGetAccessible                     ,


-- ** getActionGroup #method:getActionGroup#

#if defined(ENABLE_OVERLOADING)
    WidgetGetActionGroupMethodInfo          ,
#endif
    widgetGetActionGroup                    ,


-- ** getAllocatedBaseline #method:getAllocatedBaseline#

#if defined(ENABLE_OVERLOADING)
    WidgetGetAllocatedBaselineMethodInfo    ,
#endif
    widgetGetAllocatedBaseline              ,


-- ** getAllocatedHeight #method:getAllocatedHeight#

#if defined(ENABLE_OVERLOADING)
    WidgetGetAllocatedHeightMethodInfo      ,
#endif
    widgetGetAllocatedHeight                ,


-- ** getAllocatedSize #method:getAllocatedSize#

#if defined(ENABLE_OVERLOADING)
    WidgetGetAllocatedSizeMethodInfo        ,
#endif
    widgetGetAllocatedSize                  ,


-- ** getAllocatedWidth #method:getAllocatedWidth#

#if defined(ENABLE_OVERLOADING)
    WidgetGetAllocatedWidthMethodInfo       ,
#endif
    widgetGetAllocatedWidth                 ,


-- ** getAllocation #method:getAllocation#

#if defined(ENABLE_OVERLOADING)
    WidgetGetAllocationMethodInfo           ,
#endif
    widgetGetAllocation                     ,


-- ** getAncestor #method:getAncestor#

#if defined(ENABLE_OVERLOADING)
    WidgetGetAncestorMethodInfo             ,
#endif
    widgetGetAncestor                       ,


-- ** getAppPaintable #method:getAppPaintable#

#if defined(ENABLE_OVERLOADING)
    WidgetGetAppPaintableMethodInfo         ,
#endif
    widgetGetAppPaintable                   ,


-- ** getCanDefault #method:getCanDefault#

#if defined(ENABLE_OVERLOADING)
    WidgetGetCanDefaultMethodInfo           ,
#endif
    widgetGetCanDefault                     ,


-- ** getCanFocus #method:getCanFocus#

#if defined(ENABLE_OVERLOADING)
    WidgetGetCanFocusMethodInfo             ,
#endif
    widgetGetCanFocus                       ,


-- ** getChildRequisition #method:getChildRequisition#

#if defined(ENABLE_OVERLOADING)
    WidgetGetChildRequisitionMethodInfo     ,
#endif
    widgetGetChildRequisition               ,


-- ** getChildVisible #method:getChildVisible#

#if defined(ENABLE_OVERLOADING)
    WidgetGetChildVisibleMethodInfo         ,
#endif
    widgetGetChildVisible                   ,


-- ** getClip #method:getClip#

#if defined(ENABLE_OVERLOADING)
    WidgetGetClipMethodInfo                 ,
#endif
    widgetGetClip                           ,


-- ** getClipboard #method:getClipboard#

#if defined(ENABLE_OVERLOADING)
    WidgetGetClipboardMethodInfo            ,
#endif
    widgetGetClipboard                      ,


-- ** getCompositeName #method:getCompositeName#

#if defined(ENABLE_OVERLOADING)
    WidgetGetCompositeNameMethodInfo        ,
#endif
    widgetGetCompositeName                  ,


-- ** getDefaultDirection #method:getDefaultDirection#

    widgetGetDefaultDirection               ,


-- ** getDefaultStyle #method:getDefaultStyle#

    widgetGetDefaultStyle                   ,


-- ** getDeviceEnabled #method:getDeviceEnabled#

#if defined(ENABLE_OVERLOADING)
    WidgetGetDeviceEnabledMethodInfo        ,
#endif
    widgetGetDeviceEnabled                  ,


-- ** getDeviceEvents #method:getDeviceEvents#

#if defined(ENABLE_OVERLOADING)
    WidgetGetDeviceEventsMethodInfo         ,
#endif
    widgetGetDeviceEvents                   ,


-- ** getDirection #method:getDirection#

#if defined(ENABLE_OVERLOADING)
    WidgetGetDirectionMethodInfo            ,
#endif
    widgetGetDirection                      ,


-- ** getDisplay #method:getDisplay#

#if defined(ENABLE_OVERLOADING)
    WidgetGetDisplayMethodInfo              ,
#endif
    widgetGetDisplay                        ,


-- ** getDoubleBuffered #method:getDoubleBuffered#

#if defined(ENABLE_OVERLOADING)
    WidgetGetDoubleBufferedMethodInfo       ,
#endif
    widgetGetDoubleBuffered                 ,


-- ** getEvents #method:getEvents#

#if defined(ENABLE_OVERLOADING)
    WidgetGetEventsMethodInfo               ,
#endif
    widgetGetEvents                         ,


-- ** getFocusOnClick #method:getFocusOnClick#

#if defined(ENABLE_OVERLOADING)
    WidgetGetFocusOnClickMethodInfo         ,
#endif
    widgetGetFocusOnClick                   ,


-- ** getFontMap #method:getFontMap#

#if defined(ENABLE_OVERLOADING)
    WidgetGetFontMapMethodInfo              ,
#endif
    widgetGetFontMap                        ,


-- ** getFontOptions #method:getFontOptions#

#if defined(ENABLE_OVERLOADING)
    WidgetGetFontOptionsMethodInfo          ,
#endif
    widgetGetFontOptions                    ,


-- ** getFrameClock #method:getFrameClock#

#if defined(ENABLE_OVERLOADING)
    WidgetGetFrameClockMethodInfo           ,
#endif
    widgetGetFrameClock                     ,


-- ** getHalign #method:getHalign#

#if defined(ENABLE_OVERLOADING)
    WidgetGetHalignMethodInfo               ,
#endif
    widgetGetHalign                         ,


-- ** getHasTooltip #method:getHasTooltip#

#if defined(ENABLE_OVERLOADING)
    WidgetGetHasTooltipMethodInfo           ,
#endif
    widgetGetHasTooltip                     ,


-- ** getHasWindow #method:getHasWindow#

#if defined(ENABLE_OVERLOADING)
    WidgetGetHasWindowMethodInfo            ,
#endif
    widgetGetHasWindow                      ,


-- ** getHexpand #method:getHexpand#

#if defined(ENABLE_OVERLOADING)
    WidgetGetHexpandMethodInfo              ,
#endif
    widgetGetHexpand                        ,


-- ** getHexpandSet #method:getHexpandSet#

#if defined(ENABLE_OVERLOADING)
    WidgetGetHexpandSetMethodInfo           ,
#endif
    widgetGetHexpandSet                     ,


-- ** getMapped #method:getMapped#

#if defined(ENABLE_OVERLOADING)
    WidgetGetMappedMethodInfo               ,
#endif
    widgetGetMapped                         ,


-- ** getMarginBottom #method:getMarginBottom#

#if defined(ENABLE_OVERLOADING)
    WidgetGetMarginBottomMethodInfo         ,
#endif
    widgetGetMarginBottom                   ,


-- ** getMarginEnd #method:getMarginEnd#

#if defined(ENABLE_OVERLOADING)
    WidgetGetMarginEndMethodInfo            ,
#endif
    widgetGetMarginEnd                      ,


-- ** getMarginLeft #method:getMarginLeft#

#if defined(ENABLE_OVERLOADING)
    WidgetGetMarginLeftMethodInfo           ,
#endif
    widgetGetMarginLeft                     ,


-- ** getMarginRight #method:getMarginRight#

#if defined(ENABLE_OVERLOADING)
    WidgetGetMarginRightMethodInfo          ,
#endif
    widgetGetMarginRight                    ,


-- ** getMarginStart #method:getMarginStart#

#if defined(ENABLE_OVERLOADING)
    WidgetGetMarginStartMethodInfo          ,
#endif
    widgetGetMarginStart                    ,


-- ** getMarginTop #method:getMarginTop#

#if defined(ENABLE_OVERLOADING)
    WidgetGetMarginTopMethodInfo            ,
#endif
    widgetGetMarginTop                      ,


-- ** getModifierMask #method:getModifierMask#

#if defined(ENABLE_OVERLOADING)
    WidgetGetModifierMaskMethodInfo         ,
#endif
    widgetGetModifierMask                   ,


-- ** getModifierStyle #method:getModifierStyle#

#if defined(ENABLE_OVERLOADING)
    WidgetGetModifierStyleMethodInfo        ,
#endif
    widgetGetModifierStyle                  ,


-- ** getName #method:getName#

#if defined(ENABLE_OVERLOADING)
    WidgetGetNameMethodInfo                 ,
#endif
    widgetGetName                           ,


-- ** getNoShowAll #method:getNoShowAll#

#if defined(ENABLE_OVERLOADING)
    WidgetGetNoShowAllMethodInfo            ,
#endif
    widgetGetNoShowAll                      ,


-- ** getOpacity #method:getOpacity#

#if defined(ENABLE_OVERLOADING)
    WidgetGetOpacityMethodInfo              ,
#endif
    widgetGetOpacity                        ,


-- ** getPangoContext #method:getPangoContext#

#if defined(ENABLE_OVERLOADING)
    WidgetGetPangoContextMethodInfo         ,
#endif
    widgetGetPangoContext                   ,


-- ** getParent #method:getParent#

#if defined(ENABLE_OVERLOADING)
    WidgetGetParentMethodInfo               ,
#endif
    widgetGetParent                         ,


-- ** getParentWindow #method:getParentWindow#

#if defined(ENABLE_OVERLOADING)
    WidgetGetParentWindowMethodInfo         ,
#endif
    widgetGetParentWindow                   ,


-- ** getPath #method:getPath#

#if defined(ENABLE_OVERLOADING)
    WidgetGetPathMethodInfo                 ,
#endif
    widgetGetPath                           ,


-- ** getPointer #method:getPointer#

#if defined(ENABLE_OVERLOADING)
    WidgetGetPointerMethodInfo              ,
#endif
    widgetGetPointer                        ,


-- ** getPreferredHeight #method:getPreferredHeight#

#if defined(ENABLE_OVERLOADING)
    WidgetGetPreferredHeightMethodInfo      ,
#endif
    widgetGetPreferredHeight                ,


-- ** getPreferredHeightAndBaselineForWidth #method:getPreferredHeightAndBaselineForWidth#

#if defined(ENABLE_OVERLOADING)
    WidgetGetPreferredHeightAndBaselineForWidthMethodInfo,
#endif
    widgetGetPreferredHeightAndBaselineForWidth,


-- ** getPreferredHeightForWidth #method:getPreferredHeightForWidth#

#if defined(ENABLE_OVERLOADING)
    WidgetGetPreferredHeightForWidthMethodInfo,
#endif
    widgetGetPreferredHeightForWidth        ,


-- ** getPreferredSize #method:getPreferredSize#

#if defined(ENABLE_OVERLOADING)
    WidgetGetPreferredSizeMethodInfo        ,
#endif
    widgetGetPreferredSize                  ,


-- ** getPreferredWidth #method:getPreferredWidth#

#if defined(ENABLE_OVERLOADING)
    WidgetGetPreferredWidthMethodInfo       ,
#endif
    widgetGetPreferredWidth                 ,


-- ** getPreferredWidthForHeight #method:getPreferredWidthForHeight#

#if defined(ENABLE_OVERLOADING)
    WidgetGetPreferredWidthForHeightMethodInfo,
#endif
    widgetGetPreferredWidthForHeight        ,


-- ** getRealized #method:getRealized#

#if defined(ENABLE_OVERLOADING)
    WidgetGetRealizedMethodInfo             ,
#endif
    widgetGetRealized                       ,


-- ** getReceivesDefault #method:getReceivesDefault#

#if defined(ENABLE_OVERLOADING)
    WidgetGetReceivesDefaultMethodInfo      ,
#endif
    widgetGetReceivesDefault                ,


-- ** getRequestMode #method:getRequestMode#

#if defined(ENABLE_OVERLOADING)
    WidgetGetRequestModeMethodInfo          ,
#endif
    widgetGetRequestMode                    ,


-- ** getRequisition #method:getRequisition#

#if defined(ENABLE_OVERLOADING)
    WidgetGetRequisitionMethodInfo          ,
#endif
    widgetGetRequisition                    ,


-- ** getRootWindow #method:getRootWindow#

#if defined(ENABLE_OVERLOADING)
    WidgetGetRootWindowMethodInfo           ,
#endif
    widgetGetRootWindow                     ,


-- ** getScaleFactor #method:getScaleFactor#

#if defined(ENABLE_OVERLOADING)
    WidgetGetScaleFactorMethodInfo          ,
#endif
    widgetGetScaleFactor                    ,


-- ** getScreen #method:getScreen#

#if defined(ENABLE_OVERLOADING)
    WidgetGetScreenMethodInfo               ,
#endif
    widgetGetScreen                         ,


-- ** getSensitive #method:getSensitive#

#if defined(ENABLE_OVERLOADING)
    WidgetGetSensitiveMethodInfo            ,
#endif
    widgetGetSensitive                      ,


-- ** getSettings #method:getSettings#

#if defined(ENABLE_OVERLOADING)
    WidgetGetSettingsMethodInfo             ,
#endif
    widgetGetSettings                       ,


-- ** getSizeRequest #method:getSizeRequest#

#if defined(ENABLE_OVERLOADING)
    WidgetGetSizeRequestMethodInfo          ,
#endif
    widgetGetSizeRequest                    ,


-- ** getState #method:getState#

#if defined(ENABLE_OVERLOADING)
    WidgetGetStateMethodInfo                ,
#endif
    widgetGetState                          ,


-- ** getStateFlags #method:getStateFlags#

#if defined(ENABLE_OVERLOADING)
    WidgetGetStateFlagsMethodInfo           ,
#endif
    widgetGetStateFlags                     ,


-- ** getStyle #method:getStyle#

#if defined(ENABLE_OVERLOADING)
    WidgetGetStyleMethodInfo                ,
#endif
    widgetGetStyle                          ,


-- ** getStyleContext #method:getStyleContext#

#if defined(ENABLE_OVERLOADING)
    WidgetGetStyleContextMethodInfo         ,
#endif
    widgetGetStyleContext                   ,


-- ** getSupportMultidevice #method:getSupportMultidevice#

#if defined(ENABLE_OVERLOADING)
    WidgetGetSupportMultideviceMethodInfo   ,
#endif
    widgetGetSupportMultidevice             ,


-- ** getTemplateChild #method:getTemplateChild#

#if defined(ENABLE_OVERLOADING)
    WidgetGetTemplateChildMethodInfo        ,
#endif
    widgetGetTemplateChild                  ,


-- ** getTooltipMarkup #method:getTooltipMarkup#

#if defined(ENABLE_OVERLOADING)
    WidgetGetTooltipMarkupMethodInfo        ,
#endif
    widgetGetTooltipMarkup                  ,


-- ** getTooltipText #method:getTooltipText#

#if defined(ENABLE_OVERLOADING)
    WidgetGetTooltipTextMethodInfo          ,
#endif
    widgetGetTooltipText                    ,


-- ** getTooltipWindow #method:getTooltipWindow#

#if defined(ENABLE_OVERLOADING)
    WidgetGetTooltipWindowMethodInfo        ,
#endif
    widgetGetTooltipWindow                  ,


-- ** getToplevel #method:getToplevel#

#if defined(ENABLE_OVERLOADING)
    WidgetGetToplevelMethodInfo             ,
#endif
    widgetGetToplevel                       ,


-- ** getValign #method:getValign#

#if defined(ENABLE_OVERLOADING)
    WidgetGetValignMethodInfo               ,
#endif
    widgetGetValign                         ,


-- ** getValignWithBaseline #method:getValignWithBaseline#

#if defined(ENABLE_OVERLOADING)
    WidgetGetValignWithBaselineMethodInfo   ,
#endif
    widgetGetValignWithBaseline             ,


-- ** getVexpand #method:getVexpand#

#if defined(ENABLE_OVERLOADING)
    WidgetGetVexpandMethodInfo              ,
#endif
    widgetGetVexpand                        ,


-- ** getVexpandSet #method:getVexpandSet#

#if defined(ENABLE_OVERLOADING)
    WidgetGetVexpandSetMethodInfo           ,
#endif
    widgetGetVexpandSet                     ,


-- ** getVisible #method:getVisible#

#if defined(ENABLE_OVERLOADING)
    WidgetGetVisibleMethodInfo              ,
#endif
    widgetGetVisible                        ,


-- ** getVisual #method:getVisual#

#if defined(ENABLE_OVERLOADING)
    WidgetGetVisualMethodInfo               ,
#endif
    widgetGetVisual                         ,


-- ** getWindow #method:getWindow#

#if defined(ENABLE_OVERLOADING)
    WidgetGetWindowMethodInfo               ,
#endif
    widgetGetWindow                         ,


-- ** grabAdd #method:grabAdd#

#if defined(ENABLE_OVERLOADING)
    WidgetGrabAddMethodInfo                 ,
#endif
    widgetGrabAdd                           ,


-- ** grabDefault #method:grabDefault#

#if defined(ENABLE_OVERLOADING)
    WidgetGrabDefaultMethodInfo             ,
#endif
    widgetGrabDefault                       ,


-- ** grabFocus #method:grabFocus#

#if defined(ENABLE_OVERLOADING)
    WidgetGrabFocusMethodInfo               ,
#endif
    widgetGrabFocus                         ,


-- ** grabRemove #method:grabRemove#

#if defined(ENABLE_OVERLOADING)
    WidgetGrabRemoveMethodInfo              ,
#endif
    widgetGrabRemove                        ,


-- ** hasDefault #method:hasDefault#

#if defined(ENABLE_OVERLOADING)
    WidgetHasDefaultMethodInfo              ,
#endif
    widgetHasDefault                        ,


-- ** hasFocus #method:hasFocus#

#if defined(ENABLE_OVERLOADING)
    WidgetHasFocusMethodInfo                ,
#endif
    widgetHasFocus                          ,


-- ** hasGrab #method:hasGrab#

#if defined(ENABLE_OVERLOADING)
    WidgetHasGrabMethodInfo                 ,
#endif
    widgetHasGrab                           ,


-- ** hasRcStyle #method:hasRcStyle#

#if defined(ENABLE_OVERLOADING)
    WidgetHasRcStyleMethodInfo              ,
#endif
    widgetHasRcStyle                        ,


-- ** hasScreen #method:hasScreen#

#if defined(ENABLE_OVERLOADING)
    WidgetHasScreenMethodInfo               ,
#endif
    widgetHasScreen                         ,


-- ** hasVisibleFocus #method:hasVisibleFocus#

#if defined(ENABLE_OVERLOADING)
    WidgetHasVisibleFocusMethodInfo         ,
#endif
    widgetHasVisibleFocus                   ,


-- ** hide #method:hide#

#if defined(ENABLE_OVERLOADING)
    WidgetHideMethodInfo                    ,
#endif
    widgetHide                              ,


-- ** hideOnDelete #method:hideOnDelete#

#if defined(ENABLE_OVERLOADING)
    WidgetHideOnDeleteMethodInfo            ,
#endif
    widgetHideOnDelete                      ,


-- ** inDestruction #method:inDestruction#

#if defined(ENABLE_OVERLOADING)
    WidgetInDestructionMethodInfo           ,
#endif
    widgetInDestruction                     ,


-- ** initTemplate #method:initTemplate#

#if defined(ENABLE_OVERLOADING)
    WidgetInitTemplateMethodInfo            ,
#endif
    widgetInitTemplate                      ,


-- ** inputShapeCombineRegion #method:inputShapeCombineRegion#

#if defined(ENABLE_OVERLOADING)
    WidgetInputShapeCombineRegionMethodInfo ,
#endif
    widgetInputShapeCombineRegion           ,


-- ** insertActionGroup #method:insertActionGroup#

#if defined(ENABLE_OVERLOADING)
    WidgetInsertActionGroupMethodInfo       ,
#endif
    widgetInsertActionGroup                 ,


-- ** intersect #method:intersect#

#if defined(ENABLE_OVERLOADING)
    WidgetIntersectMethodInfo               ,
#endif
    widgetIntersect                         ,


-- ** isAncestor #method:isAncestor#

#if defined(ENABLE_OVERLOADING)
    WidgetIsAncestorMethodInfo              ,
#endif
    widgetIsAncestor                        ,


-- ** isComposited #method:isComposited#

#if defined(ENABLE_OVERLOADING)
    WidgetIsCompositedMethodInfo            ,
#endif
    widgetIsComposited                      ,


-- ** isDrawable #method:isDrawable#

#if defined(ENABLE_OVERLOADING)
    WidgetIsDrawableMethodInfo              ,
#endif
    widgetIsDrawable                        ,


-- ** isFocus #method:isFocus#

#if defined(ENABLE_OVERLOADING)
    WidgetIsFocusMethodInfo                 ,
#endif
    widgetIsFocus                           ,


-- ** isSensitive #method:isSensitive#

#if defined(ENABLE_OVERLOADING)
    WidgetIsSensitiveMethodInfo             ,
#endif
    widgetIsSensitive                       ,


-- ** isToplevel #method:isToplevel#

#if defined(ENABLE_OVERLOADING)
    WidgetIsToplevelMethodInfo              ,
#endif
    widgetIsToplevel                        ,


-- ** isVisible #method:isVisible#

#if defined(ENABLE_OVERLOADING)
    WidgetIsVisibleMethodInfo               ,
#endif
    widgetIsVisible                         ,


-- ** keynavFailed #method:keynavFailed#

#if defined(ENABLE_OVERLOADING)
    WidgetKeynavFailedMethodInfo            ,
#endif
    widgetKeynavFailed                      ,


-- ** listAccelClosures #method:listAccelClosures#

#if defined(ENABLE_OVERLOADING)
    WidgetListAccelClosuresMethodInfo       ,
#endif
    widgetListAccelClosures                 ,


-- ** listActionPrefixes #method:listActionPrefixes#

#if defined(ENABLE_OVERLOADING)
    WidgetListActionPrefixesMethodInfo      ,
#endif
    widgetListActionPrefixes                ,


-- ** listMnemonicLabels #method:listMnemonicLabels#

#if defined(ENABLE_OVERLOADING)
    WidgetListMnemonicLabelsMethodInfo      ,
#endif
    widgetListMnemonicLabels                ,


-- ** map #method:map#

#if defined(ENABLE_OVERLOADING)
    WidgetMapMethodInfo                     ,
#endif
    widgetMap                               ,


-- ** mnemonicActivate #method:mnemonicActivate#

#if defined(ENABLE_OVERLOADING)
    WidgetMnemonicActivateMethodInfo        ,
#endif
    widgetMnemonicActivate                  ,


-- ** modifyBase #method:modifyBase#

#if defined(ENABLE_OVERLOADING)
    WidgetModifyBaseMethodInfo              ,
#endif
    widgetModifyBase                        ,


-- ** modifyBg #method:modifyBg#

#if defined(ENABLE_OVERLOADING)
    WidgetModifyBgMethodInfo                ,
#endif
    widgetModifyBg                          ,


-- ** modifyCursor #method:modifyCursor#

#if defined(ENABLE_OVERLOADING)
    WidgetModifyCursorMethodInfo            ,
#endif
    widgetModifyCursor                      ,


-- ** modifyFg #method:modifyFg#

#if defined(ENABLE_OVERLOADING)
    WidgetModifyFgMethodInfo                ,
#endif
    widgetModifyFg                          ,


-- ** modifyFont #method:modifyFont#

#if defined(ENABLE_OVERLOADING)
    WidgetModifyFontMethodInfo              ,
#endif
    widgetModifyFont                        ,


-- ** modifyStyle #method:modifyStyle#

#if defined(ENABLE_OVERLOADING)
    WidgetModifyStyleMethodInfo             ,
#endif
    widgetModifyStyle                       ,


-- ** modifyText #method:modifyText#

#if defined(ENABLE_OVERLOADING)
    WidgetModifyTextMethodInfo              ,
#endif
    widgetModifyText                        ,


-- ** overrideBackgroundColor #method:overrideBackgroundColor#

#if defined(ENABLE_OVERLOADING)
    WidgetOverrideBackgroundColorMethodInfo ,
#endif
    widgetOverrideBackgroundColor           ,


-- ** overrideColor #method:overrideColor#

#if defined(ENABLE_OVERLOADING)
    WidgetOverrideColorMethodInfo           ,
#endif
    widgetOverrideColor                     ,


-- ** overrideCursor #method:overrideCursor#

#if defined(ENABLE_OVERLOADING)
    WidgetOverrideCursorMethodInfo          ,
#endif
    widgetOverrideCursor                    ,


-- ** overrideFont #method:overrideFont#

#if defined(ENABLE_OVERLOADING)
    WidgetOverrideFontMethodInfo            ,
#endif
    widgetOverrideFont                      ,


-- ** overrideSymbolicColor #method:overrideSymbolicColor#

#if defined(ENABLE_OVERLOADING)
    WidgetOverrideSymbolicColorMethodInfo   ,
#endif
    widgetOverrideSymbolicColor             ,


-- ** path #method:path#

#if defined(ENABLE_OVERLOADING)
    WidgetPathMethodInfo                    ,
#endif
    widgetPath                              ,


-- ** popCompositeChild #method:popCompositeChild#

    widgetPopCompositeChild                 ,


-- ** pushCompositeChild #method:pushCompositeChild#

    widgetPushCompositeChild                ,


-- ** queueAllocate #method:queueAllocate#

#if defined(ENABLE_OVERLOADING)
    WidgetQueueAllocateMethodInfo           ,
#endif
    widgetQueueAllocate                     ,


-- ** queueComputeExpand #method:queueComputeExpand#

#if defined(ENABLE_OVERLOADING)
    WidgetQueueComputeExpandMethodInfo      ,
#endif
    widgetQueueComputeExpand                ,


-- ** queueDraw #method:queueDraw#

#if defined(ENABLE_OVERLOADING)
    WidgetQueueDrawMethodInfo               ,
#endif
    widgetQueueDraw                         ,


-- ** queueDrawArea #method:queueDrawArea#

#if defined(ENABLE_OVERLOADING)
    WidgetQueueDrawAreaMethodInfo           ,
#endif
    widgetQueueDrawArea                     ,


-- ** queueDrawRegion #method:queueDrawRegion#

#if defined(ENABLE_OVERLOADING)
    WidgetQueueDrawRegionMethodInfo         ,
#endif
    widgetQueueDrawRegion                   ,


-- ** queueResize #method:queueResize#

#if defined(ENABLE_OVERLOADING)
    WidgetQueueResizeMethodInfo             ,
#endif
    widgetQueueResize                       ,


-- ** queueResizeNoRedraw #method:queueResizeNoRedraw#

#if defined(ENABLE_OVERLOADING)
    WidgetQueueResizeNoRedrawMethodInfo     ,
#endif
    widgetQueueResizeNoRedraw               ,


-- ** realize #method:realize#

#if defined(ENABLE_OVERLOADING)
    WidgetRealizeMethodInfo                 ,
#endif
    widgetRealize                           ,


-- ** regionIntersect #method:regionIntersect#

#if defined(ENABLE_OVERLOADING)
    WidgetRegionIntersectMethodInfo         ,
#endif
    widgetRegionIntersect                   ,


-- ** registerWindow #method:registerWindow#

#if defined(ENABLE_OVERLOADING)
    WidgetRegisterWindowMethodInfo          ,
#endif
    widgetRegisterWindow                    ,


-- ** removeAccelerator #method:removeAccelerator#

#if defined(ENABLE_OVERLOADING)
    WidgetRemoveAcceleratorMethodInfo       ,
#endif
    widgetRemoveAccelerator                 ,


-- ** removeMnemonicLabel #method:removeMnemonicLabel#

#if defined(ENABLE_OVERLOADING)
    WidgetRemoveMnemonicLabelMethodInfo     ,
#endif
    widgetRemoveMnemonicLabel               ,


-- ** removeTickCallback #method:removeTickCallback#

#if defined(ENABLE_OVERLOADING)
    WidgetRemoveTickCallbackMethodInfo      ,
#endif
    widgetRemoveTickCallback                ,


-- ** renderIcon #method:renderIcon#

#if defined(ENABLE_OVERLOADING)
    WidgetRenderIconMethodInfo              ,
#endif
    widgetRenderIcon                        ,


-- ** renderIconPixbuf #method:renderIconPixbuf#

#if defined(ENABLE_OVERLOADING)
    WidgetRenderIconPixbufMethodInfo        ,
#endif
    widgetRenderIconPixbuf                  ,


-- ** reparent #method:reparent#

#if defined(ENABLE_OVERLOADING)
    WidgetReparentMethodInfo                ,
#endif
    widgetReparent                          ,


-- ** resetRcStyles #method:resetRcStyles#

#if defined(ENABLE_OVERLOADING)
    WidgetResetRcStylesMethodInfo           ,
#endif
    widgetResetRcStyles                     ,


-- ** resetStyle #method:resetStyle#

#if defined(ENABLE_OVERLOADING)
    WidgetResetStyleMethodInfo              ,
#endif
    widgetResetStyle                        ,


-- ** sendExpose #method:sendExpose#

#if defined(ENABLE_OVERLOADING)
    WidgetSendExposeMethodInfo              ,
#endif
    widgetSendExpose                        ,


-- ** sendFocusChange #method:sendFocusChange#

#if defined(ENABLE_OVERLOADING)
    WidgetSendFocusChangeMethodInfo         ,
#endif
    widgetSendFocusChange                   ,


-- ** setAccelPath #method:setAccelPath#

#if defined(ENABLE_OVERLOADING)
    WidgetSetAccelPathMethodInfo            ,
#endif
    widgetSetAccelPath                      ,


-- ** setAllocation #method:setAllocation#

#if defined(ENABLE_OVERLOADING)
    WidgetSetAllocationMethodInfo           ,
#endif
    widgetSetAllocation                     ,


-- ** setAppPaintable #method:setAppPaintable#

#if defined(ENABLE_OVERLOADING)
    WidgetSetAppPaintableMethodInfo         ,
#endif
    widgetSetAppPaintable                   ,


-- ** setCanDefault #method:setCanDefault#

#if defined(ENABLE_OVERLOADING)
    WidgetSetCanDefaultMethodInfo           ,
#endif
    widgetSetCanDefault                     ,


-- ** setCanFocus #method:setCanFocus#

#if defined(ENABLE_OVERLOADING)
    WidgetSetCanFocusMethodInfo             ,
#endif
    widgetSetCanFocus                       ,


-- ** setChildVisible #method:setChildVisible#

#if defined(ENABLE_OVERLOADING)
    WidgetSetChildVisibleMethodInfo         ,
#endif
    widgetSetChildVisible                   ,


-- ** setClip #method:setClip#

#if defined(ENABLE_OVERLOADING)
    WidgetSetClipMethodInfo                 ,
#endif
    widgetSetClip                           ,


-- ** setCompositeName #method:setCompositeName#

#if defined(ENABLE_OVERLOADING)
    WidgetSetCompositeNameMethodInfo        ,
#endif
    widgetSetCompositeName                  ,


-- ** setDefaultDirection #method:setDefaultDirection#

    widgetSetDefaultDirection               ,


-- ** setDeviceEnabled #method:setDeviceEnabled#

#if defined(ENABLE_OVERLOADING)
    WidgetSetDeviceEnabledMethodInfo        ,
#endif
    widgetSetDeviceEnabled                  ,


-- ** setDeviceEvents #method:setDeviceEvents#

#if defined(ENABLE_OVERLOADING)
    WidgetSetDeviceEventsMethodInfo         ,
#endif
    widgetSetDeviceEvents                   ,


-- ** setDirection #method:setDirection#

#if defined(ENABLE_OVERLOADING)
    WidgetSetDirectionMethodInfo            ,
#endif
    widgetSetDirection                      ,


-- ** setDoubleBuffered #method:setDoubleBuffered#

#if defined(ENABLE_OVERLOADING)
    WidgetSetDoubleBufferedMethodInfo       ,
#endif
    widgetSetDoubleBuffered                 ,


-- ** setEvents #method:setEvents#

#if defined(ENABLE_OVERLOADING)
    WidgetSetEventsMethodInfo               ,
#endif
    widgetSetEvents                         ,


-- ** setFocusOnClick #method:setFocusOnClick#

#if defined(ENABLE_OVERLOADING)
    WidgetSetFocusOnClickMethodInfo         ,
#endif
    widgetSetFocusOnClick                   ,


-- ** setFontMap #method:setFontMap#

#if defined(ENABLE_OVERLOADING)
    WidgetSetFontMapMethodInfo              ,
#endif
    widgetSetFontMap                        ,


-- ** setFontOptions #method:setFontOptions#

#if defined(ENABLE_OVERLOADING)
    WidgetSetFontOptionsMethodInfo          ,
#endif
    widgetSetFontOptions                    ,


-- ** setHalign #method:setHalign#

#if defined(ENABLE_OVERLOADING)
    WidgetSetHalignMethodInfo               ,
#endif
    widgetSetHalign                         ,


-- ** setHasTooltip #method:setHasTooltip#

#if defined(ENABLE_OVERLOADING)
    WidgetSetHasTooltipMethodInfo           ,
#endif
    widgetSetHasTooltip                     ,


-- ** setHasWindow #method:setHasWindow#

#if defined(ENABLE_OVERLOADING)
    WidgetSetHasWindowMethodInfo            ,
#endif
    widgetSetHasWindow                      ,


-- ** setHexpand #method:setHexpand#

#if defined(ENABLE_OVERLOADING)
    WidgetSetHexpandMethodInfo              ,
#endif
    widgetSetHexpand                        ,


-- ** setHexpandSet #method:setHexpandSet#

#if defined(ENABLE_OVERLOADING)
    WidgetSetHexpandSetMethodInfo           ,
#endif
    widgetSetHexpandSet                     ,


-- ** setMapped #method:setMapped#

#if defined(ENABLE_OVERLOADING)
    WidgetSetMappedMethodInfo               ,
#endif
    widgetSetMapped                         ,


-- ** setMarginBottom #method:setMarginBottom#

#if defined(ENABLE_OVERLOADING)
    WidgetSetMarginBottomMethodInfo         ,
#endif
    widgetSetMarginBottom                   ,


-- ** setMarginEnd #method:setMarginEnd#

#if defined(ENABLE_OVERLOADING)
    WidgetSetMarginEndMethodInfo            ,
#endif
    widgetSetMarginEnd                      ,


-- ** setMarginLeft #method:setMarginLeft#

#if defined(ENABLE_OVERLOADING)
    WidgetSetMarginLeftMethodInfo           ,
#endif
    widgetSetMarginLeft                     ,


-- ** setMarginRight #method:setMarginRight#

#if defined(ENABLE_OVERLOADING)
    WidgetSetMarginRightMethodInfo          ,
#endif
    widgetSetMarginRight                    ,


-- ** setMarginStart #method:setMarginStart#

#if defined(ENABLE_OVERLOADING)
    WidgetSetMarginStartMethodInfo          ,
#endif
    widgetSetMarginStart                    ,


-- ** setMarginTop #method:setMarginTop#

#if defined(ENABLE_OVERLOADING)
    WidgetSetMarginTopMethodInfo            ,
#endif
    widgetSetMarginTop                      ,


-- ** setName #method:setName#

#if defined(ENABLE_OVERLOADING)
    WidgetSetNameMethodInfo                 ,
#endif
    widgetSetName                           ,


-- ** setNoShowAll #method:setNoShowAll#

#if defined(ENABLE_OVERLOADING)
    WidgetSetNoShowAllMethodInfo            ,
#endif
    widgetSetNoShowAll                      ,


-- ** setOpacity #method:setOpacity#

#if defined(ENABLE_OVERLOADING)
    WidgetSetOpacityMethodInfo              ,
#endif
    widgetSetOpacity                        ,


-- ** setParent #method:setParent#

#if defined(ENABLE_OVERLOADING)
    WidgetSetParentMethodInfo               ,
#endif
    widgetSetParent                         ,


-- ** setParentWindow #method:setParentWindow#

#if defined(ENABLE_OVERLOADING)
    WidgetSetParentWindowMethodInfo         ,
#endif
    widgetSetParentWindow                   ,


-- ** setRealized #method:setRealized#

#if defined(ENABLE_OVERLOADING)
    WidgetSetRealizedMethodInfo             ,
#endif
    widgetSetRealized                       ,


-- ** setReceivesDefault #method:setReceivesDefault#

#if defined(ENABLE_OVERLOADING)
    WidgetSetReceivesDefaultMethodInfo      ,
#endif
    widgetSetReceivesDefault                ,


-- ** setRedrawOnAllocate #method:setRedrawOnAllocate#

#if defined(ENABLE_OVERLOADING)
    WidgetSetRedrawOnAllocateMethodInfo     ,
#endif
    widgetSetRedrawOnAllocate               ,


-- ** setSensitive #method:setSensitive#

#if defined(ENABLE_OVERLOADING)
    WidgetSetSensitiveMethodInfo            ,
#endif
    widgetSetSensitive                      ,


-- ** setSizeRequest #method:setSizeRequest#

#if defined(ENABLE_OVERLOADING)
    WidgetSetSizeRequestMethodInfo          ,
#endif
    widgetSetSizeRequest                    ,


-- ** setState #method:setState#

#if defined(ENABLE_OVERLOADING)
    WidgetSetStateMethodInfo                ,
#endif
    widgetSetState                          ,


-- ** setStateFlags #method:setStateFlags#

#if defined(ENABLE_OVERLOADING)
    WidgetSetStateFlagsMethodInfo           ,
#endif
    widgetSetStateFlags                     ,


-- ** setStyle #method:setStyle#

#if defined(ENABLE_OVERLOADING)
    WidgetSetStyleMethodInfo                ,
#endif
    widgetSetStyle                          ,


-- ** setSupportMultidevice #method:setSupportMultidevice#

#if defined(ENABLE_OVERLOADING)
    WidgetSetSupportMultideviceMethodInfo   ,
#endif
    widgetSetSupportMultidevice             ,


-- ** setTooltipMarkup #method:setTooltipMarkup#

#if defined(ENABLE_OVERLOADING)
    WidgetSetTooltipMarkupMethodInfo        ,
#endif
    widgetSetTooltipMarkup                  ,


-- ** setTooltipText #method:setTooltipText#

#if defined(ENABLE_OVERLOADING)
    WidgetSetTooltipTextMethodInfo          ,
#endif
    widgetSetTooltipText                    ,


-- ** setTooltipWindow #method:setTooltipWindow#

#if defined(ENABLE_OVERLOADING)
    WidgetSetTooltipWindowMethodInfo        ,
#endif
    widgetSetTooltipWindow                  ,


-- ** setValign #method:setValign#

#if defined(ENABLE_OVERLOADING)
    WidgetSetValignMethodInfo               ,
#endif
    widgetSetValign                         ,


-- ** setVexpand #method:setVexpand#

#if defined(ENABLE_OVERLOADING)
    WidgetSetVexpandMethodInfo              ,
#endif
    widgetSetVexpand                        ,


-- ** setVexpandSet #method:setVexpandSet#

#if defined(ENABLE_OVERLOADING)
    WidgetSetVexpandSetMethodInfo           ,
#endif
    widgetSetVexpandSet                     ,


-- ** setVisible #method:setVisible#

#if defined(ENABLE_OVERLOADING)
    WidgetSetVisibleMethodInfo              ,
#endif
    widgetSetVisible                        ,


-- ** setVisual #method:setVisual#

#if defined(ENABLE_OVERLOADING)
    WidgetSetVisualMethodInfo               ,
#endif
    widgetSetVisual                         ,


-- ** setWindow #method:setWindow#

#if defined(ENABLE_OVERLOADING)
    WidgetSetWindowMethodInfo               ,
#endif
    widgetSetWindow                         ,


-- ** shapeCombineRegion #method:shapeCombineRegion#

#if defined(ENABLE_OVERLOADING)
    WidgetShapeCombineRegionMethodInfo      ,
#endif
    widgetShapeCombineRegion                ,


-- ** show #method:show#

#if defined(ENABLE_OVERLOADING)
    WidgetShowMethodInfo                    ,
#endif
    widgetShow                              ,


-- ** showAll #method:showAll#

#if defined(ENABLE_OVERLOADING)
    WidgetShowAllMethodInfo                 ,
#endif
    widgetShowAll                           ,


-- ** showNow #method:showNow#

#if defined(ENABLE_OVERLOADING)
    WidgetShowNowMethodInfo                 ,
#endif
    widgetShowNow                           ,


-- ** sizeAllocate #method:sizeAllocate#

#if defined(ENABLE_OVERLOADING)
    WidgetSizeAllocateMethodInfo            ,
#endif
    widgetSizeAllocate                      ,


-- ** sizeAllocateWithBaseline #method:sizeAllocateWithBaseline#

#if defined(ENABLE_OVERLOADING)
    WidgetSizeAllocateWithBaselineMethodInfo,
#endif
    widgetSizeAllocateWithBaseline          ,


-- ** sizeRequest #method:sizeRequest#

#if defined(ENABLE_OVERLOADING)
    WidgetSizeRequestMethodInfo             ,
#endif
    widgetSizeRequest                       ,


-- ** styleAttach #method:styleAttach#

#if defined(ENABLE_OVERLOADING)
    WidgetStyleAttachMethodInfo             ,
#endif
    widgetStyleAttach                       ,


-- ** styleGetProperty #method:styleGetProperty#

#if defined(ENABLE_OVERLOADING)
    WidgetStyleGetPropertyMethodInfo        ,
#endif
    widgetStyleGetProperty                  ,


-- ** thawChildNotify #method:thawChildNotify#

#if defined(ENABLE_OVERLOADING)
    WidgetThawChildNotifyMethodInfo         ,
#endif
    widgetThawChildNotify                   ,


-- ** translateCoordinates #method:translateCoordinates#

#if defined(ENABLE_OVERLOADING)
    WidgetTranslateCoordinatesMethodInfo    ,
#endif
    widgetTranslateCoordinates              ,


-- ** triggerTooltipQuery #method:triggerTooltipQuery#

#if defined(ENABLE_OVERLOADING)
    WidgetTriggerTooltipQueryMethodInfo     ,
#endif
    widgetTriggerTooltipQuery               ,


-- ** unmap #method:unmap#

#if defined(ENABLE_OVERLOADING)
    WidgetUnmapMethodInfo                   ,
#endif
    widgetUnmap                             ,


-- ** unparent #method:unparent#

#if defined(ENABLE_OVERLOADING)
    WidgetUnparentMethodInfo                ,
#endif
    widgetUnparent                          ,


-- ** unrealize #method:unrealize#

#if defined(ENABLE_OVERLOADING)
    WidgetUnrealizeMethodInfo               ,
#endif
    widgetUnrealize                         ,


-- ** unregisterWindow #method:unregisterWindow#

#if defined(ENABLE_OVERLOADING)
    WidgetUnregisterWindowMethodInfo        ,
#endif
    widgetUnregisterWindow                  ,


-- ** unsetStateFlags #method:unsetStateFlags#

#if defined(ENABLE_OVERLOADING)
    WidgetUnsetStateFlagsMethodInfo         ,
#endif
    widgetUnsetStateFlags                   ,




 -- * Properties


-- ** appPaintable #attr:appPaintable#
-- | /No description available in the introspection data./

#if defined(ENABLE_OVERLOADING)
    WidgetAppPaintablePropertyInfo          ,
#endif
    constructWidgetAppPaintable             ,
    getWidgetAppPaintable                   ,
    setWidgetAppPaintable                   ,
#if defined(ENABLE_OVERLOADING)
    widgetAppPaintable                      ,
#endif


-- ** canDefault #attr:canDefault#
-- | /No description available in the introspection data./

#if defined(ENABLE_OVERLOADING)
    WidgetCanDefaultPropertyInfo            ,
#endif
    constructWidgetCanDefault               ,
    getWidgetCanDefault                     ,
    setWidgetCanDefault                     ,
#if defined(ENABLE_OVERLOADING)
    widgetCanDefault                        ,
#endif


-- ** canFocus #attr:canFocus#
-- | /No description available in the introspection data./

#if defined(ENABLE_OVERLOADING)
    WidgetCanFocusPropertyInfo              ,
#endif
    constructWidgetCanFocus                 ,
    getWidgetCanFocus                       ,
    setWidgetCanFocus                       ,
#if defined(ENABLE_OVERLOADING)
    widgetCanFocus                          ,
#endif


-- ** compositeChild #attr:compositeChild#
-- | /No description available in the introspection data./

#if defined(ENABLE_OVERLOADING)
    WidgetCompositeChildPropertyInfo        ,
#endif
    getWidgetCompositeChild                 ,
#if defined(ENABLE_OVERLOADING)
    widgetCompositeChild                    ,
#endif


-- ** doubleBuffered #attr:doubleBuffered#
-- | Whether the widget is double buffered.
-- 
-- /Since: 2.18/

#if defined(ENABLE_OVERLOADING)
    WidgetDoubleBufferedPropertyInfo        ,
#endif
    constructWidgetDoubleBuffered           ,
    getWidgetDoubleBuffered                 ,
    setWidgetDoubleBuffered                 ,
#if defined(ENABLE_OVERLOADING)
    widgetDoubleBuffered                    ,
#endif


-- ** events #attr:events#
-- | /No description available in the introspection data./

#if defined(ENABLE_OVERLOADING)
    WidgetEventsPropertyInfo                ,
#endif
    constructWidgetEvents                   ,
    getWidgetEvents                         ,
    setWidgetEvents                         ,
#if defined(ENABLE_OVERLOADING)
    widgetEvents                            ,
#endif


-- ** expand #attr:expand#
-- | Whether to expand in both directions. Setting this sets both [Widget:hexpand]("GI.Gtk.Objects.Widget#g:attr:hexpand") and [Widget:vexpand]("GI.Gtk.Objects.Widget#g:attr:vexpand")
-- 
-- /Since: 3.0/

#if defined(ENABLE_OVERLOADING)
    WidgetExpandPropertyInfo                ,
#endif
    constructWidgetExpand                   ,
    getWidgetExpand                         ,
    setWidgetExpand                         ,
#if defined(ENABLE_OVERLOADING)
    widgetExpand                            ,
#endif


-- ** focusOnClick #attr:focusOnClick#
-- | Whether the widget should grab focus when it is clicked with the mouse.
-- 
-- This property is only relevant for widgets that can take focus.
-- 
-- Before 3.20, several widgets (GtkButton, GtkFileChooserButton,
-- GtkComboBox) implemented this property individually.
-- 
-- /Since: 3.20/

#if defined(ENABLE_OVERLOADING)
    WidgetFocusOnClickPropertyInfo          ,
#endif
    constructWidgetFocusOnClick             ,
    getWidgetFocusOnClick                   ,
    setWidgetFocusOnClick                   ,
#if defined(ENABLE_OVERLOADING)
    widgetFocusOnClick                      ,
#endif


-- ** halign #attr:halign#
-- | How to distribute horizontal space if widget gets extra space, see t'GI.Gtk.Enums.Align'
-- 
-- /Since: 3.0/

#if defined(ENABLE_OVERLOADING)
    WidgetHalignPropertyInfo                ,
#endif
    constructWidgetHalign                   ,
    getWidgetHalign                         ,
    setWidgetHalign                         ,
#if defined(ENABLE_OVERLOADING)
    widgetHalign                            ,
#endif


-- ** hasDefault #attr:hasDefault#
-- | /No description available in the introspection data./

#if defined(ENABLE_OVERLOADING)
    WidgetHasDefaultPropertyInfo            ,
#endif
    constructWidgetHasDefault               ,
    getWidgetHasDefault                     ,
    setWidgetHasDefault                     ,


-- ** hasFocus #attr:hasFocus#
-- | /No description available in the introspection data./

#if defined(ENABLE_OVERLOADING)
    WidgetHasFocusPropertyInfo              ,
#endif
    constructWidgetHasFocus                 ,
    getWidgetHasFocus                       ,
    setWidgetHasFocus                       ,


-- ** hasTooltip #attr:hasTooltip#
-- | Enables or disables the emission of [Widget::queryTooltip]("GI.Gtk.Objects.Widget#g:signal:queryTooltip") on /@widget@/.
-- A value of 'P.True' indicates that /@widget@/ can have a tooltip, in this case
-- the widget will be queried using [Widget::queryTooltip]("GI.Gtk.Objects.Widget#g:signal:queryTooltip") to determine
-- whether it will provide a tooltip or not.
-- 
-- Note that setting this property to 'P.True' for the first time will change
-- the event masks of the GdkWindows of this widget to include leave-notify
-- and motion-notify events.  This cannot and will not be undone when the
-- property is set to 'P.False' again.
-- 
-- /Since: 2.12/

#if defined(ENABLE_OVERLOADING)
    WidgetHasTooltipPropertyInfo            ,
#endif
    constructWidgetHasTooltip               ,
    getWidgetHasTooltip                     ,
    setWidgetHasTooltip                     ,
#if defined(ENABLE_OVERLOADING)
    widgetHasTooltip                        ,
#endif


-- ** heightRequest #attr:heightRequest#
-- | /No description available in the introspection data./

#if defined(ENABLE_OVERLOADING)
    WidgetHeightRequestPropertyInfo         ,
#endif
    constructWidgetHeightRequest            ,
    getWidgetHeightRequest                  ,
    setWidgetHeightRequest                  ,
#if defined(ENABLE_OVERLOADING)
    widgetHeightRequest                     ,
#endif


-- ** hexpand #attr:hexpand#
-- | Whether to expand horizontally. See 'GI.Gtk.Objects.Widget.widgetSetHexpand'.
-- 
-- /Since: 3.0/

#if defined(ENABLE_OVERLOADING)
    WidgetHexpandPropertyInfo               ,
#endif
    constructWidgetHexpand                  ,
    getWidgetHexpand                        ,
    setWidgetHexpand                        ,
#if defined(ENABLE_OVERLOADING)
    widgetHexpand                           ,
#endif


-- ** hexpandSet #attr:hexpandSet#
-- | Whether to use the [Widget:hexpand]("GI.Gtk.Objects.Widget#g:attr:hexpand") property. See 'GI.Gtk.Objects.Widget.widgetGetHexpandSet'.
-- 
-- /Since: 3.0/

#if defined(ENABLE_OVERLOADING)
    WidgetHexpandSetPropertyInfo            ,
#endif
    constructWidgetHexpandSet               ,
    getWidgetHexpandSet                     ,
    setWidgetHexpandSet                     ,
#if defined(ENABLE_OVERLOADING)
    widgetHexpandSet                        ,
#endif


-- ** isFocus #attr:isFocus#
-- | /No description available in the introspection data./

#if defined(ENABLE_OVERLOADING)
    WidgetIsFocusPropertyInfo               ,
#endif
    constructWidgetIsFocus                  ,
    getWidgetIsFocus                        ,
    setWidgetIsFocus                        ,


-- ** margin #attr:margin#
-- | Sets all four sides\' margin at once. If read, returns max
-- margin on any side.
-- 
-- /Since: 3.0/

#if defined(ENABLE_OVERLOADING)
    WidgetMarginPropertyInfo                ,
#endif
    constructWidgetMargin                   ,
    getWidgetMargin                         ,
    setWidgetMargin                         ,
#if defined(ENABLE_OVERLOADING)
    widgetMargin                            ,
#endif


-- ** marginBottom #attr:marginBottom#
-- | Margin on bottom side of widget.
-- 
-- This property adds margin outside of the widget\'s normal size
-- request, the margin will be added in addition to the size from
-- 'GI.Gtk.Objects.Widget.widgetSetSizeRequest' for example.
-- 
-- /Since: 3.0/

#if defined(ENABLE_OVERLOADING)
    WidgetMarginBottomPropertyInfo          ,
#endif
    constructWidgetMarginBottom             ,
    getWidgetMarginBottom                   ,
    setWidgetMarginBottom                   ,
#if defined(ENABLE_OVERLOADING)
    widgetMarginBottom                      ,
#endif


-- ** marginEnd #attr:marginEnd#
-- | Margin on end of widget, horizontally. This property supports
-- left-to-right and right-to-left text directions.
-- 
-- This property adds margin outside of the widget\'s normal size
-- request, the margin will be added in addition to the size from
-- 'GI.Gtk.Objects.Widget.widgetSetSizeRequest' for example.
-- 
-- /Since: 3.12/

#if defined(ENABLE_OVERLOADING)
    WidgetMarginEndPropertyInfo             ,
#endif
    constructWidgetMarginEnd                ,
    getWidgetMarginEnd                      ,
    setWidgetMarginEnd                      ,
#if defined(ENABLE_OVERLOADING)
    widgetMarginEnd                         ,
#endif


-- ** marginLeft #attr:marginLeft#
-- | Margin on left side of widget.
-- 
-- This property adds margin outside of the widget\'s normal size
-- request, the margin will be added in addition to the size from
-- 'GI.Gtk.Objects.Widget.widgetSetSizeRequest' for example.
-- 
-- /Since: 3.0/

#if defined(ENABLE_OVERLOADING)
    WidgetMarginLeftPropertyInfo            ,
#endif
    constructWidgetMarginLeft               ,
    getWidgetMarginLeft                     ,
    setWidgetMarginLeft                     ,
#if defined(ENABLE_OVERLOADING)
    widgetMarginLeft                        ,
#endif


-- ** marginRight #attr:marginRight#
-- | Margin on right side of widget.
-- 
-- This property adds margin outside of the widget\'s normal size
-- request, the margin will be added in addition to the size from
-- 'GI.Gtk.Objects.Widget.widgetSetSizeRequest' for example.
-- 
-- /Since: 3.0/

#if defined(ENABLE_OVERLOADING)
    WidgetMarginRightPropertyInfo           ,
#endif
    constructWidgetMarginRight              ,
    getWidgetMarginRight                    ,
    setWidgetMarginRight                    ,
#if defined(ENABLE_OVERLOADING)
    widgetMarginRight                       ,
#endif


-- ** marginStart #attr:marginStart#
-- | Margin on start of widget, horizontally. This property supports
-- left-to-right and right-to-left text directions.
-- 
-- This property adds margin outside of the widget\'s normal size
-- request, the margin will be added in addition to the size from
-- 'GI.Gtk.Objects.Widget.widgetSetSizeRequest' for example.
-- 
-- /Since: 3.12/

#if defined(ENABLE_OVERLOADING)
    WidgetMarginStartPropertyInfo           ,
#endif
    constructWidgetMarginStart              ,
    getWidgetMarginStart                    ,
    setWidgetMarginStart                    ,
#if defined(ENABLE_OVERLOADING)
    widgetMarginStart                       ,
#endif


-- ** marginTop #attr:marginTop#
-- | Margin on top side of widget.
-- 
-- This property adds margin outside of the widget\'s normal size
-- request, the margin will be added in addition to the size from
-- 'GI.Gtk.Objects.Widget.widgetSetSizeRequest' for example.
-- 
-- /Since: 3.0/

#if defined(ENABLE_OVERLOADING)
    WidgetMarginTopPropertyInfo             ,
#endif
    constructWidgetMarginTop                ,
    getWidgetMarginTop                      ,
    setWidgetMarginTop                      ,
#if defined(ENABLE_OVERLOADING)
    widgetMarginTop                         ,
#endif


-- ** name #attr:name#
-- | /No description available in the introspection data./

#if defined(ENABLE_OVERLOADING)
    WidgetNamePropertyInfo                  ,
#endif
    constructWidgetName                     ,
    getWidgetName                           ,
    setWidgetName                           ,
#if defined(ENABLE_OVERLOADING)
    widgetName                              ,
#endif


-- ** noShowAll #attr:noShowAll#
-- | /No description available in the introspection data./

#if defined(ENABLE_OVERLOADING)
    WidgetNoShowAllPropertyInfo             ,
#endif
    constructWidgetNoShowAll                ,
    getWidgetNoShowAll                      ,
    setWidgetNoShowAll                      ,
#if defined(ENABLE_OVERLOADING)
    widgetNoShowAll                         ,
#endif


-- ** opacity #attr:opacity#
-- | The requested opacity of the widget. See 'GI.Gtk.Objects.Widget.widgetSetOpacity' for
-- more details about window opacity.
-- 
-- Before 3.8 this was only available in GtkWindow
-- 
-- /Since: 3.8/

#if defined(ENABLE_OVERLOADING)
    WidgetOpacityPropertyInfo               ,
#endif
    constructWidgetOpacity                  ,
    getWidgetOpacity                        ,
    setWidgetOpacity                        ,
#if defined(ENABLE_OVERLOADING)
    widgetOpacity                           ,
#endif


-- ** parent #attr:parent#
-- | /No description available in the introspection data./

#if defined(ENABLE_OVERLOADING)
    WidgetParentPropertyInfo                ,
#endif
    clearWidgetParent                       ,
    constructWidgetParent                   ,
    getWidgetParent                         ,
    setWidgetParent                         ,
#if defined(ENABLE_OVERLOADING)
    widgetParent                            ,
#endif


-- ** receivesDefault #attr:receivesDefault#
-- | /No description available in the introspection data./

#if defined(ENABLE_OVERLOADING)
    WidgetReceivesDefaultPropertyInfo       ,
#endif
    constructWidgetReceivesDefault          ,
    getWidgetReceivesDefault                ,
    setWidgetReceivesDefault                ,
#if defined(ENABLE_OVERLOADING)
    widgetReceivesDefault                   ,
#endif


-- ** scaleFactor #attr:scaleFactor#
-- | The scale factor of the widget. See 'GI.Gtk.Objects.Widget.widgetGetScaleFactor' for
-- more details about widget scaling.
-- 
-- /Since: 3.10/

#if defined(ENABLE_OVERLOADING)
    WidgetScaleFactorPropertyInfo           ,
#endif
    getWidgetScaleFactor                    ,
#if defined(ENABLE_OVERLOADING)
    widgetScaleFactor                       ,
#endif


-- ** sensitive #attr:sensitive#
-- | /No description available in the introspection data./

#if defined(ENABLE_OVERLOADING)
    WidgetSensitivePropertyInfo             ,
#endif
    constructWidgetSensitive                ,
    getWidgetSensitive                      ,
    setWidgetSensitive                      ,
#if defined(ENABLE_OVERLOADING)
    widgetSensitive                         ,
#endif


-- ** style #attr:style#
-- | The style of the widget, which contains information about how it will look (colors, etc).

#if defined(ENABLE_OVERLOADING)
    WidgetStylePropertyInfo                 ,
#endif
    clearWidgetStyle                        ,
    constructWidgetStyle                    ,
    getWidgetStyle                          ,
    setWidgetStyle                          ,
#if defined(ENABLE_OVERLOADING)
    widgetStyle                             ,
#endif


-- ** tooltipMarkup #attr:tooltipMarkup#
-- | Sets the text of tooltip to be the given string, which is marked up
-- with the [Pango text markup language][PangoMarkupFormat].
-- Also see 'GI.Gtk.Objects.Tooltip.tooltipSetMarkup'.
-- 
-- This is a convenience property which will take care of getting the
-- tooltip shown if the given string is not 'P.Nothing': [Widget:hasTooltip]("GI.Gtk.Objects.Widget#g:attr:hasTooltip")
-- will automatically be set to 'P.True' and there will be taken care of
-- [Widget::queryTooltip]("GI.Gtk.Objects.Widget#g:signal:queryTooltip") in the default signal handler.
-- 
-- Note that if both [Widget:tooltipText]("GI.Gtk.Objects.Widget#g:attr:tooltipText") and [Widget:tooltipMarkup]("GI.Gtk.Objects.Widget#g:attr:tooltipMarkup")
-- are set, the last one wins.
-- 
-- /Since: 2.12/

#if defined(ENABLE_OVERLOADING)
    WidgetTooltipMarkupPropertyInfo         ,
#endif
    clearWidgetTooltipMarkup                ,
    constructWidgetTooltipMarkup            ,
    getWidgetTooltipMarkup                  ,
    setWidgetTooltipMarkup                  ,
#if defined(ENABLE_OVERLOADING)
    widgetTooltipMarkup                     ,
#endif


-- ** tooltipText #attr:tooltipText#
-- | Sets the text of tooltip to be the given string.
-- 
-- Also see 'GI.Gtk.Objects.Tooltip.tooltipSetText'.
-- 
-- This is a convenience property which will take care of getting the
-- tooltip shown if the given string is not 'P.Nothing': [Widget:hasTooltip]("GI.Gtk.Objects.Widget#g:attr:hasTooltip")
-- will automatically be set to 'P.True' and there will be taken care of
-- [Widget::queryTooltip]("GI.Gtk.Objects.Widget#g:signal:queryTooltip") in the default signal handler.
-- 
-- Note that if both [Widget:tooltipText]("GI.Gtk.Objects.Widget#g:attr:tooltipText") and [Widget:tooltipMarkup]("GI.Gtk.Objects.Widget#g:attr:tooltipMarkup")
-- are set, the last one wins.
-- 
-- /Since: 2.12/

#if defined(ENABLE_OVERLOADING)
    WidgetTooltipTextPropertyInfo           ,
#endif
    clearWidgetTooltipText                  ,
    constructWidgetTooltipText              ,
    getWidgetTooltipText                    ,
    setWidgetTooltipText                    ,
#if defined(ENABLE_OVERLOADING)
    widgetTooltipText                       ,
#endif


-- ** valign #attr:valign#
-- | How to distribute vertical space if widget gets extra space, see t'GI.Gtk.Enums.Align'
-- 
-- /Since: 3.0/

#if defined(ENABLE_OVERLOADING)
    WidgetValignPropertyInfo                ,
#endif
    constructWidgetValign                   ,
    getWidgetValign                         ,
    setWidgetValign                         ,
#if defined(ENABLE_OVERLOADING)
    widgetValign                            ,
#endif


-- ** vexpand #attr:vexpand#
-- | Whether to expand vertically. See 'GI.Gtk.Objects.Widget.widgetSetVexpand'.
-- 
-- /Since: 3.0/

#if defined(ENABLE_OVERLOADING)
    WidgetVexpandPropertyInfo               ,
#endif
    constructWidgetVexpand                  ,
    getWidgetVexpand                        ,
    setWidgetVexpand                        ,
#if defined(ENABLE_OVERLOADING)
    widgetVexpand                           ,
#endif


-- ** vexpandSet #attr:vexpandSet#
-- | Whether to use the [Widget:vexpand]("GI.Gtk.Objects.Widget#g:attr:vexpand") property. See 'GI.Gtk.Objects.Widget.widgetGetVexpandSet'.
-- 
-- /Since: 3.0/

#if defined(ENABLE_OVERLOADING)
    WidgetVexpandSetPropertyInfo            ,
#endif
    constructWidgetVexpandSet               ,
    getWidgetVexpandSet                     ,
    setWidgetVexpandSet                     ,
#if defined(ENABLE_OVERLOADING)
    widgetVexpandSet                        ,
#endif


-- ** visible #attr:visible#
-- | /No description available in the introspection data./

#if defined(ENABLE_OVERLOADING)
    WidgetVisiblePropertyInfo               ,
#endif
    constructWidgetVisible                  ,
    getWidgetVisible                        ,
    setWidgetVisible                        ,
#if defined(ENABLE_OVERLOADING)
    widgetVisible                           ,
#endif


-- ** widthRequest #attr:widthRequest#
-- | /No description available in the introspection data./

#if defined(ENABLE_OVERLOADING)
    WidgetWidthRequestPropertyInfo          ,
#endif
    constructWidgetWidthRequest             ,
    getWidgetWidthRequest                   ,
    setWidgetWidthRequest                   ,
#if defined(ENABLE_OVERLOADING)
    widgetWidthRequest                      ,
#endif


-- ** window #attr:window#
-- | The widget\'s window if it is realized, 'P.Nothing' otherwise.
-- 
-- /Since: 2.14/

#if defined(ENABLE_OVERLOADING)
    WidgetWindowPropertyInfo                ,
#endif
    getWidgetWindow                         ,
#if defined(ENABLE_OVERLOADING)
    widgetWindow                            ,
#endif




 -- * Signals


-- ** accelClosuresChanged #signal:accelClosuresChanged#

    WidgetAccelClosuresChangedCallback      ,
#if defined(ENABLE_OVERLOADING)
    WidgetAccelClosuresChangedSignalInfo    ,
#endif
    afterWidgetAccelClosuresChanged         ,
    onWidgetAccelClosuresChanged            ,


-- ** buttonPressEvent #signal:buttonPressEvent#

    WidgetButtonPressEventCallback          ,
#if defined(ENABLE_OVERLOADING)
    WidgetButtonPressEventSignalInfo        ,
#endif
    afterWidgetButtonPressEvent             ,
    onWidgetButtonPressEvent                ,


-- ** buttonReleaseEvent #signal:buttonReleaseEvent#

    WidgetButtonReleaseEventCallback        ,
#if defined(ENABLE_OVERLOADING)
    WidgetButtonReleaseEventSignalInfo      ,
#endif
    afterWidgetButtonReleaseEvent           ,
    onWidgetButtonReleaseEvent              ,


-- ** canActivateAccel #signal:canActivateAccel#

    WidgetCanActivateAccelCallback          ,
#if defined(ENABLE_OVERLOADING)
    WidgetCanActivateAccelSignalInfo        ,
#endif
    afterWidgetCanActivateAccel             ,
    onWidgetCanActivateAccel                ,


-- ** childNotify #signal:childNotify#

    WidgetChildNotifyCallback               ,
#if defined(ENABLE_OVERLOADING)
    WidgetChildNotifySignalInfo             ,
#endif
    afterWidgetChildNotify                  ,
    onWidgetChildNotify                     ,


-- ** compositedChanged #signal:compositedChanged#

    WidgetCompositedChangedCallback         ,
#if defined(ENABLE_OVERLOADING)
    WidgetCompositedChangedSignalInfo       ,
#endif
    afterWidgetCompositedChanged            ,
    onWidgetCompositedChanged               ,


-- ** configureEvent #signal:configureEvent#

    WidgetConfigureEventCallback            ,
#if defined(ENABLE_OVERLOADING)
    WidgetConfigureEventSignalInfo          ,
#endif
    afterWidgetConfigureEvent               ,
    onWidgetConfigureEvent                  ,


-- ** damageEvent #signal:damageEvent#

    WidgetDamageEventCallback               ,
#if defined(ENABLE_OVERLOADING)
    WidgetDamageEventSignalInfo             ,
#endif
    afterWidgetDamageEvent                  ,
    onWidgetDamageEvent                     ,


-- ** deleteEvent #signal:deleteEvent#

    WidgetDeleteEventCallback               ,
#if defined(ENABLE_OVERLOADING)
    WidgetDeleteEventSignalInfo             ,
#endif
    afterWidgetDeleteEvent                  ,
    onWidgetDeleteEvent                     ,


-- ** destroy #signal:destroy#

    WidgetDestroyCallback                   ,
#if defined(ENABLE_OVERLOADING)
    WidgetDestroySignalInfo                 ,
#endif
    afterWidgetDestroy                      ,
    onWidgetDestroy                         ,


-- ** destroyEvent #signal:destroyEvent#

    WidgetDestroyEventCallback              ,
#if defined(ENABLE_OVERLOADING)
    WidgetDestroyEventSignalInfo            ,
#endif
    afterWidgetDestroyEvent                 ,
    onWidgetDestroyEvent                    ,


-- ** directionChanged #signal:directionChanged#

    WidgetDirectionChangedCallback          ,
#if defined(ENABLE_OVERLOADING)
    WidgetDirectionChangedSignalInfo        ,
#endif
    afterWidgetDirectionChanged             ,
    onWidgetDirectionChanged                ,


-- ** dragBegin #signal:dragBegin#

    WidgetDragBeginCallback                 ,
#if defined(ENABLE_OVERLOADING)
    WidgetDragBeginSignalInfo               ,
#endif
    afterWidgetDragBegin                    ,
    onWidgetDragBegin                       ,


-- ** dragDataDelete #signal:dragDataDelete#

    WidgetDragDataDeleteCallback            ,
#if defined(ENABLE_OVERLOADING)
    WidgetDragDataDeleteSignalInfo          ,
#endif
    afterWidgetDragDataDelete               ,
    onWidgetDragDataDelete                  ,


-- ** dragDataGet #signal:dragDataGet#

    WidgetDragDataGetCallback               ,
#if defined(ENABLE_OVERLOADING)
    WidgetDragDataGetSignalInfo             ,
#endif
    afterWidgetDragDataGet                  ,
    onWidgetDragDataGet                     ,


-- ** dragDataReceived #signal:dragDataReceived#

    WidgetDragDataReceivedCallback          ,
#if defined(ENABLE_OVERLOADING)
    WidgetDragDataReceivedSignalInfo        ,
#endif
    afterWidgetDragDataReceived             ,
    onWidgetDragDataReceived                ,


-- ** dragDrop #signal:dragDrop#

    WidgetDragDropCallback                  ,
#if defined(ENABLE_OVERLOADING)
    WidgetDragDropSignalInfo                ,
#endif
    afterWidgetDragDrop                     ,
    onWidgetDragDrop                        ,


-- ** dragEnd #signal:dragEnd#

    WidgetDragEndCallback                   ,
#if defined(ENABLE_OVERLOADING)
    WidgetDragEndSignalInfo                 ,
#endif
    afterWidgetDragEnd                      ,
    onWidgetDragEnd                         ,


-- ** dragFailed #signal:dragFailed#

    WidgetDragFailedCallback                ,
#if defined(ENABLE_OVERLOADING)
    WidgetDragFailedSignalInfo              ,
#endif
    afterWidgetDragFailed                   ,
    onWidgetDragFailed                      ,


-- ** dragLeave #signal:dragLeave#

    WidgetDragLeaveCallback                 ,
#if defined(ENABLE_OVERLOADING)
    WidgetDragLeaveSignalInfo               ,
#endif
    afterWidgetDragLeave                    ,
    onWidgetDragLeave                       ,


-- ** dragMotion #signal:dragMotion#

    WidgetDragMotionCallback                ,
#if defined(ENABLE_OVERLOADING)
    WidgetDragMotionSignalInfo              ,
#endif
    afterWidgetDragMotion                   ,
    onWidgetDragMotion                      ,


-- ** draw #signal:draw#

    WidgetDrawCallback                      ,
#if defined(ENABLE_OVERLOADING)
    WidgetDrawSignalInfo                    ,
#endif
    afterWidgetDraw                         ,
    onWidgetDraw                            ,


-- ** enterNotifyEvent #signal:enterNotifyEvent#

    WidgetEnterNotifyEventCallback          ,
#if defined(ENABLE_OVERLOADING)
    WidgetEnterNotifyEventSignalInfo        ,
#endif
    afterWidgetEnterNotifyEvent             ,
    onWidgetEnterNotifyEvent                ,


-- ** event #signal:event#

    WidgetEventCallback                     ,
#if defined(ENABLE_OVERLOADING)
    WidgetEventSignalInfo                   ,
#endif
    afterWidgetEvent                        ,
    onWidgetEvent                           ,


-- ** eventAfter #signal:eventAfter#

    WidgetEventAfterCallback                ,
#if defined(ENABLE_OVERLOADING)
    WidgetEventAfterSignalInfo              ,
#endif
    afterWidgetEventAfter                   ,
    onWidgetEventAfter                      ,


-- ** focus #signal:focus#

    WidgetFocusCallback                     ,
#if defined(ENABLE_OVERLOADING)
    WidgetFocusSignalInfo                   ,
#endif
    afterWidgetFocus                        ,
    onWidgetFocus                           ,


-- ** focusInEvent #signal:focusInEvent#

    WidgetFocusInEventCallback              ,
#if defined(ENABLE_OVERLOADING)
    WidgetFocusInEventSignalInfo            ,
#endif
    afterWidgetFocusInEvent                 ,
    onWidgetFocusInEvent                    ,


-- ** focusOutEvent #signal:focusOutEvent#

    WidgetFocusOutEventCallback             ,
#if defined(ENABLE_OVERLOADING)
    WidgetFocusOutEventSignalInfo           ,
#endif
    afterWidgetFocusOutEvent                ,
    onWidgetFocusOutEvent                   ,


-- ** grabBrokenEvent #signal:grabBrokenEvent#

    WidgetGrabBrokenEventCallback           ,
#if defined(ENABLE_OVERLOADING)
    WidgetGrabBrokenEventSignalInfo         ,
#endif
    afterWidgetGrabBrokenEvent              ,
    onWidgetGrabBrokenEvent                 ,


-- ** grabFocus #signal:grabFocus#

    WidgetGrabFocusCallback                 ,
#if defined(ENABLE_OVERLOADING)
    WidgetGrabFocusSignalInfo               ,
#endif
    afterWidgetGrabFocus                    ,
    onWidgetGrabFocus                       ,


-- ** grabNotify #signal:grabNotify#

    WidgetGrabNotifyCallback                ,
#if defined(ENABLE_OVERLOADING)
    WidgetGrabNotifySignalInfo              ,
#endif
    afterWidgetGrabNotify                   ,
    onWidgetGrabNotify                      ,


-- ** hide #signal:hide#

    WidgetHideCallback                      ,
#if defined(ENABLE_OVERLOADING)
    WidgetHideSignalInfo                    ,
#endif
    afterWidgetHide                         ,
    onWidgetHide                            ,


-- ** hierarchyChanged #signal:hierarchyChanged#

    WidgetHierarchyChangedCallback          ,
#if defined(ENABLE_OVERLOADING)
    WidgetHierarchyChangedSignalInfo        ,
#endif
    afterWidgetHierarchyChanged             ,
    onWidgetHierarchyChanged                ,


-- ** keyPressEvent #signal:keyPressEvent#

    WidgetKeyPressEventCallback             ,
#if defined(ENABLE_OVERLOADING)
    WidgetKeyPressEventSignalInfo           ,
#endif
    afterWidgetKeyPressEvent                ,
    onWidgetKeyPressEvent                   ,


-- ** keyReleaseEvent #signal:keyReleaseEvent#

    WidgetKeyReleaseEventCallback           ,
#if defined(ENABLE_OVERLOADING)
    WidgetKeyReleaseEventSignalInfo         ,
#endif
    afterWidgetKeyReleaseEvent              ,
    onWidgetKeyReleaseEvent                 ,


-- ** keynavFailed #signal:keynavFailed#

    WidgetKeynavFailedCallback              ,
#if defined(ENABLE_OVERLOADING)
    WidgetKeynavFailedSignalInfo            ,
#endif
    afterWidgetKeynavFailed                 ,
    onWidgetKeynavFailed                    ,


-- ** leaveNotifyEvent #signal:leaveNotifyEvent#

    WidgetLeaveNotifyEventCallback          ,
#if defined(ENABLE_OVERLOADING)
    WidgetLeaveNotifyEventSignalInfo        ,
#endif
    afterWidgetLeaveNotifyEvent             ,
    onWidgetLeaveNotifyEvent                ,


-- ** map #signal:map#

    WidgetMapCallback                       ,
#if defined(ENABLE_OVERLOADING)
    WidgetMapSignalInfo                     ,
#endif
    afterWidgetMap                          ,
    onWidgetMap                             ,


-- ** mapEvent #signal:mapEvent#

    WidgetMapEventCallback                  ,
#if defined(ENABLE_OVERLOADING)
    WidgetMapEventSignalInfo                ,
#endif
    afterWidgetMapEvent                     ,
    onWidgetMapEvent                        ,


-- ** mnemonicActivate #signal:mnemonicActivate#

    WidgetMnemonicActivateCallback          ,
#if defined(ENABLE_OVERLOADING)
    WidgetMnemonicActivateSignalInfo        ,
#endif
    afterWidgetMnemonicActivate             ,
    onWidgetMnemonicActivate                ,


-- ** motionNotifyEvent #signal:motionNotifyEvent#

    WidgetMotionNotifyEventCallback         ,
#if defined(ENABLE_OVERLOADING)
    WidgetMotionNotifyEventSignalInfo       ,
#endif
    afterWidgetMotionNotifyEvent            ,
    onWidgetMotionNotifyEvent               ,


-- ** moveFocus #signal:moveFocus#

    WidgetMoveFocusCallback                 ,
#if defined(ENABLE_OVERLOADING)
    WidgetMoveFocusSignalInfo               ,
#endif
    afterWidgetMoveFocus                    ,
    onWidgetMoveFocus                       ,


-- ** parentSet #signal:parentSet#

    WidgetParentSetCallback                 ,
#if defined(ENABLE_OVERLOADING)
    WidgetParentSetSignalInfo               ,
#endif
    afterWidgetParentSet                    ,
    onWidgetParentSet                       ,


-- ** popupMenu #signal:popupMenu#

    WidgetPopupMenuCallback                 ,
#if defined(ENABLE_OVERLOADING)
    WidgetPopupMenuSignalInfo               ,
#endif
    afterWidgetPopupMenu                    ,
    onWidgetPopupMenu                       ,


-- ** propertyNotifyEvent #signal:propertyNotifyEvent#

    WidgetPropertyNotifyEventCallback       ,
#if defined(ENABLE_OVERLOADING)
    WidgetPropertyNotifyEventSignalInfo     ,
#endif
    afterWidgetPropertyNotifyEvent          ,
    onWidgetPropertyNotifyEvent             ,


-- ** proximityInEvent #signal:proximityInEvent#

    WidgetProximityInEventCallback          ,
#if defined(ENABLE_OVERLOADING)
    WidgetProximityInEventSignalInfo        ,
#endif
    afterWidgetProximityInEvent             ,
    onWidgetProximityInEvent                ,


-- ** proximityOutEvent #signal:proximityOutEvent#

    WidgetProximityOutEventCallback         ,
#if defined(ENABLE_OVERLOADING)
    WidgetProximityOutEventSignalInfo       ,
#endif
    afterWidgetProximityOutEvent            ,
    onWidgetProximityOutEvent               ,


-- ** queryTooltip #signal:queryTooltip#

    WidgetQueryTooltipCallback              ,
#if defined(ENABLE_OVERLOADING)
    WidgetQueryTooltipSignalInfo            ,
#endif
    afterWidgetQueryTooltip                 ,
    onWidgetQueryTooltip                    ,


-- ** realize #signal:realize#

    WidgetRealizeCallback                   ,
#if defined(ENABLE_OVERLOADING)
    WidgetRealizeSignalInfo                 ,
#endif
    afterWidgetRealize                      ,
    onWidgetRealize                         ,


-- ** screenChanged #signal:screenChanged#

    WidgetScreenChangedCallback             ,
#if defined(ENABLE_OVERLOADING)
    WidgetScreenChangedSignalInfo           ,
#endif
    afterWidgetScreenChanged                ,
    onWidgetScreenChanged                   ,


-- ** scrollEvent #signal:scrollEvent#

    WidgetScrollEventCallback               ,
#if defined(ENABLE_OVERLOADING)
    WidgetScrollEventSignalInfo             ,
#endif
    afterWidgetScrollEvent                  ,
    onWidgetScrollEvent                     ,


-- ** selectionClearEvent #signal:selectionClearEvent#

    WidgetSelectionClearEventCallback       ,
#if defined(ENABLE_OVERLOADING)
    WidgetSelectionClearEventSignalInfo     ,
#endif
    afterWidgetSelectionClearEvent          ,
    onWidgetSelectionClearEvent             ,


-- ** selectionGet #signal:selectionGet#

    WidgetSelectionGetCallback              ,
#if defined(ENABLE_OVERLOADING)
    WidgetSelectionGetSignalInfo            ,
#endif
    afterWidgetSelectionGet                 ,
    onWidgetSelectionGet                    ,


-- ** selectionNotifyEvent #signal:selectionNotifyEvent#

    WidgetSelectionNotifyEventCallback      ,
#if defined(ENABLE_OVERLOADING)
    WidgetSelectionNotifyEventSignalInfo    ,
#endif
    afterWidgetSelectionNotifyEvent         ,
    onWidgetSelectionNotifyEvent            ,


-- ** selectionReceived #signal:selectionReceived#

    WidgetSelectionReceivedCallback         ,
#if defined(ENABLE_OVERLOADING)
    WidgetSelectionReceivedSignalInfo       ,
#endif
    afterWidgetSelectionReceived            ,
    onWidgetSelectionReceived               ,


-- ** selectionRequestEvent #signal:selectionRequestEvent#

    WidgetSelectionRequestEventCallback     ,
#if defined(ENABLE_OVERLOADING)
    WidgetSelectionRequestEventSignalInfo   ,
#endif
    afterWidgetSelectionRequestEvent        ,
    onWidgetSelectionRequestEvent           ,


-- ** show #signal:show#

    WidgetShowCallback                      ,
#if defined(ENABLE_OVERLOADING)
    WidgetShowSignalInfo                    ,
#endif
    afterWidgetShow                         ,
    onWidgetShow                            ,


-- ** showHelp #signal:showHelp#

    WidgetShowHelpCallback                  ,
#if defined(ENABLE_OVERLOADING)
    WidgetShowHelpSignalInfo                ,
#endif
    afterWidgetShowHelp                     ,
    onWidgetShowHelp                        ,


-- ** sizeAllocate #signal:sizeAllocate#

    WidgetSizeAllocateCallback              ,
#if defined(ENABLE_OVERLOADING)
    WidgetSizeAllocateSignalInfo            ,
#endif
    afterWidgetSizeAllocate                 ,
    onWidgetSizeAllocate                    ,


-- ** stateChanged #signal:stateChanged#

    WidgetStateChangedCallback              ,
#if defined(ENABLE_OVERLOADING)
    WidgetStateChangedSignalInfo            ,
#endif
    afterWidgetStateChanged                 ,
    onWidgetStateChanged                    ,


-- ** stateFlagsChanged #signal:stateFlagsChanged#

    WidgetStateFlagsChangedCallback         ,
#if defined(ENABLE_OVERLOADING)
    WidgetStateFlagsChangedSignalInfo       ,
#endif
    afterWidgetStateFlagsChanged            ,
    onWidgetStateFlagsChanged               ,


-- ** styleSet #signal:styleSet#

    WidgetStyleSetCallback                  ,
#if defined(ENABLE_OVERLOADING)
    WidgetStyleSetSignalInfo                ,
#endif
    afterWidgetStyleSet                     ,
    onWidgetStyleSet                        ,


-- ** styleUpdated #signal:styleUpdated#

    WidgetStyleUpdatedCallback              ,
#if defined(ENABLE_OVERLOADING)
    WidgetStyleUpdatedSignalInfo            ,
#endif
    afterWidgetStyleUpdated                 ,
    onWidgetStyleUpdated                    ,


-- ** touchEvent #signal:touchEvent#

    WidgetTouchEventCallback                ,
#if defined(ENABLE_OVERLOADING)
    WidgetTouchEventSignalInfo              ,
#endif
    afterWidgetTouchEvent                   ,
    onWidgetTouchEvent                      ,


-- ** unmap #signal:unmap#

    WidgetUnmapCallback                     ,
#if defined(ENABLE_OVERLOADING)
    WidgetUnmapSignalInfo                   ,
#endif
    afterWidgetUnmap                        ,
    onWidgetUnmap                           ,


-- ** unmapEvent #signal:unmapEvent#

    WidgetUnmapEventCallback                ,
#if defined(ENABLE_OVERLOADING)
    WidgetUnmapEventSignalInfo              ,
#endif
    afterWidgetUnmapEvent                   ,
    onWidgetUnmapEvent                      ,


-- ** unrealize #signal:unrealize#

    WidgetUnrealizeCallback                 ,
#if defined(ENABLE_OVERLOADING)
    WidgetUnrealizeSignalInfo               ,
#endif
    afterWidgetUnrealize                    ,
    onWidgetUnrealize                       ,


-- ** visibilityNotifyEvent #signal:visibilityNotifyEvent#

    WidgetVisibilityNotifyEventCallback     ,
#if defined(ENABLE_OVERLOADING)
    WidgetVisibilityNotifyEventSignalInfo   ,
#endif
    afterWidgetVisibilityNotifyEvent        ,
    onWidgetVisibilityNotifyEvent           ,


-- ** windowStateEvent #signal:windowStateEvent#

    WidgetWindowStateEventCallback          ,
#if defined(ENABLE_OVERLOADING)
    WidgetWindowStateEventSignalInfo        ,
#endif
    afterWidgetWindowStateEvent             ,
    onWidgetWindowStateEvent                ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Atk.Interfaces.ImplementorIface as Atk.ImplementorIface
import qualified GI.Atk.Objects.Object as Atk.Object
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.Cairo.Structs.FontOptions as Cairo.FontOptions
import qualified GI.Cairo.Structs.Region as Cairo.Region
import qualified GI.Cairo.Structs.Surface as Cairo.Surface
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.MarkupParser as GLib.MarkupParser
import qualified GI.GObject.Callbacks as GObject.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Enums as Gdk.Enums
import qualified GI.Gdk.Flags as Gdk.Flags
import qualified GI.Gdk.Objects.Device as Gdk.Device
import qualified GI.Gdk.Objects.Display as Gdk.Display
import qualified GI.Gdk.Objects.DragContext as Gdk.DragContext
import qualified GI.Gdk.Objects.FrameClock as Gdk.FrameClock
import qualified GI.Gdk.Objects.Screen as Gdk.Screen
import qualified GI.Gdk.Objects.Visual as Gdk.Visual
import qualified GI.Gdk.Objects.Window as Gdk.Window
import qualified GI.Gdk.Structs.Atom as Gdk.Atom
import qualified GI.Gdk.Structs.Color as Gdk.Color
import qualified GI.Gdk.Structs.EventAny as Gdk.EventAny
import qualified GI.Gdk.Structs.EventButton as Gdk.EventButton
import qualified GI.Gdk.Structs.EventConfigure as Gdk.EventConfigure
import qualified GI.Gdk.Structs.EventCrossing as Gdk.EventCrossing
import qualified GI.Gdk.Structs.EventExpose as Gdk.EventExpose
import qualified GI.Gdk.Structs.EventFocus as Gdk.EventFocus
import qualified GI.Gdk.Structs.EventGrabBroken as Gdk.EventGrabBroken
import qualified GI.Gdk.Structs.EventKey as Gdk.EventKey
import qualified GI.Gdk.Structs.EventMotion as Gdk.EventMotion
import qualified GI.Gdk.Structs.EventOwnerChange as Gdk.EventOwnerChange
import qualified GI.Gdk.Structs.EventProperty as Gdk.EventProperty
import qualified GI.Gdk.Structs.EventProximity as Gdk.EventProximity
import qualified GI.Gdk.Structs.EventScroll as Gdk.EventScroll
import qualified GI.Gdk.Structs.EventSelection as Gdk.EventSelection
import qualified GI.Gdk.Structs.EventVisibility as Gdk.EventVisibility
import qualified GI.Gdk.Structs.EventWindowState as Gdk.EventWindowState
import qualified GI.Gdk.Structs.Geometry as Gdk.Geometry
import qualified GI.Gdk.Structs.RGBA as Gdk.RGBA
import qualified GI.Gdk.Structs.Rectangle as Gdk.Rectangle
import qualified GI.Gdk.Unions.Event as Gdk.Event
import qualified GI.GdkPixbuf.Objects.Pixbuf as GdkPixbuf.Pixbuf
import qualified GI.Gio.Flags as Gio.Flags
import qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import qualified GI.Gio.Interfaces.ActionMap as Gio.ActionMap
import qualified GI.Gio.Interfaces.File as Gio.File
import qualified GI.Gio.Interfaces.Icon as Gio.Icon
import qualified GI.Gio.Objects.Application as Gio.Application
import qualified GI.Gio.Objects.Menu as Gio.Menu
import qualified GI.Gio.Objects.MenuModel as Gio.MenuModel
import qualified GI.Gtk.Callbacks as Gtk.Callbacks
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Flags as Gtk.Flags
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.StyleProvider as Gtk.StyleProvider
import {-# SOURCE #-} qualified GI.Gtk.Objects.AccelGroup as Gtk.AccelGroup
import {-# SOURCE #-} qualified GI.Gtk.Objects.Adjustment as Gtk.Adjustment
import {-# SOURCE #-} qualified GI.Gtk.Objects.Application as Gtk.Application
import {-# SOURCE #-} qualified GI.Gtk.Objects.Bin as Gtk.Bin
import {-# SOURCE #-} qualified GI.Gtk.Objects.Builder as Gtk.Builder
import {-# SOURCE #-} qualified GI.Gtk.Objects.Clipboard as Gtk.Clipboard
import {-# SOURCE #-} qualified GI.Gtk.Objects.Container as Gtk.Container
import {-# SOURCE #-} qualified GI.Gtk.Objects.IconFactory as Gtk.IconFactory
import {-# SOURCE #-} qualified GI.Gtk.Objects.RcStyle as Gtk.RcStyle
import {-# SOURCE #-} qualified GI.Gtk.Objects.Settings as Gtk.Settings
import {-# SOURCE #-} qualified GI.Gtk.Objects.Style as Gtk.Style
import {-# SOURCE #-} qualified GI.Gtk.Objects.StyleContext as Gtk.StyleContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.StyleProperties as Gtk.StyleProperties
import {-# SOURCE #-} qualified GI.Gtk.Objects.TextBuffer as Gtk.TextBuffer
import {-# SOURCE #-} qualified GI.Gtk.Objects.TextChildAnchor as Gtk.TextChildAnchor
import {-# SOURCE #-} qualified GI.Gtk.Objects.TextMark as Gtk.TextMark
import {-# SOURCE #-} qualified GI.Gtk.Objects.TextTag as Gtk.TextTag
import {-# SOURCE #-} qualified GI.Gtk.Objects.TextTagTable as Gtk.TextTagTable
import {-# SOURCE #-} qualified GI.Gtk.Objects.Tooltip as Gtk.Tooltip
import {-# SOURCE #-} qualified GI.Gtk.Objects.Window as Gtk.Window
import {-# SOURCE #-} qualified GI.Gtk.Objects.WindowGroup as Gtk.WindowGroup
import {-# SOURCE #-} qualified GI.Gtk.Structs.AccelGroupEntry as Gtk.AccelGroupEntry
import {-# SOURCE #-} qualified GI.Gtk.Structs.AccelKey as Gtk.AccelKey
import {-# SOURCE #-} qualified GI.Gtk.Structs.Border as Gtk.Border
import {-# SOURCE #-} qualified GI.Gtk.Structs.CssSection as Gtk.CssSection
import {-# SOURCE #-} qualified GI.Gtk.Structs.IconSet as Gtk.IconSet
import {-# SOURCE #-} qualified GI.Gtk.Structs.IconSource as Gtk.IconSource
import {-# SOURCE #-} qualified GI.Gtk.Structs.Requisition as Gtk.Requisition
import {-# SOURCE #-} qualified GI.Gtk.Structs.SelectionData as Gtk.SelectionData
import {-# SOURCE #-} qualified GI.Gtk.Structs.SettingsValue as Gtk.SettingsValue
import {-# SOURCE #-} qualified GI.Gtk.Structs.SymbolicColor as Gtk.SymbolicColor
import {-# SOURCE #-} qualified GI.Gtk.Structs.TargetEntry as Gtk.TargetEntry
import {-# SOURCE #-} qualified GI.Gtk.Structs.TargetList as Gtk.TargetList
import {-# SOURCE #-} qualified GI.Gtk.Structs.TextAppearance as Gtk.TextAppearance
import {-# SOURCE #-} qualified GI.Gtk.Structs.TextAttributes as Gtk.TextAttributes
import {-# SOURCE #-} qualified GI.Gtk.Structs.TextIter as Gtk.TextIter
import {-# SOURCE #-} qualified GI.Gtk.Structs.WidgetPath as Gtk.WidgetPath
import qualified GI.Pango.Enums as Pango.Enums
import qualified GI.Pango.Objects.Context as Pango.Context
import qualified GI.Pango.Objects.FontMap as Pango.FontMap
import qualified GI.Pango.Objects.Layout as Pango.Layout
import qualified GI.Pango.Structs.FontDescription as Pango.FontDescription
import qualified GI.Pango.Structs.Language as Pango.Language
import qualified GI.Pango.Structs.TabArray as Pango.TabArray

#else
import qualified GI.Atk.Interfaces.ImplementorIface as Atk.ImplementorIface
import qualified GI.Atk.Objects.Object as Atk.Object
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.Cairo.Structs.FontOptions as Cairo.FontOptions
import qualified GI.Cairo.Structs.Region as Cairo.Region
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Enums as Gdk.Enums
import qualified GI.Gdk.Flags as Gdk.Flags
import qualified GI.Gdk.Objects.Device as Gdk.Device
import qualified GI.Gdk.Objects.Display as Gdk.Display
import qualified GI.Gdk.Objects.DragContext as Gdk.DragContext
import qualified GI.Gdk.Objects.FrameClock as Gdk.FrameClock
import qualified GI.Gdk.Objects.Screen as Gdk.Screen
import qualified GI.Gdk.Objects.Visual as Gdk.Visual
import qualified GI.Gdk.Objects.Window as Gdk.Window
import qualified GI.Gdk.Structs.Atom as Gdk.Atom
import qualified GI.Gdk.Structs.Color as Gdk.Color
import qualified GI.Gdk.Structs.EventAny as Gdk.EventAny
import qualified GI.Gdk.Structs.EventButton as Gdk.EventButton
import qualified GI.Gdk.Structs.EventConfigure as Gdk.EventConfigure
import qualified GI.Gdk.Structs.EventCrossing as Gdk.EventCrossing
import qualified GI.Gdk.Structs.EventExpose as Gdk.EventExpose
import qualified GI.Gdk.Structs.EventFocus as Gdk.EventFocus
import qualified GI.Gdk.Structs.EventGrabBroken as Gdk.EventGrabBroken
import qualified GI.Gdk.Structs.EventKey as Gdk.EventKey
import qualified GI.Gdk.Structs.EventMotion as Gdk.EventMotion
import qualified GI.Gdk.Structs.EventProperty as Gdk.EventProperty
import qualified GI.Gdk.Structs.EventProximity as Gdk.EventProximity
import qualified GI.Gdk.Structs.EventScroll as Gdk.EventScroll
import qualified GI.Gdk.Structs.EventSelection as Gdk.EventSelection
import qualified GI.Gdk.Structs.EventVisibility as Gdk.EventVisibility
import qualified GI.Gdk.Structs.EventWindowState as Gdk.EventWindowState
import qualified GI.Gdk.Structs.RGBA as Gdk.RGBA
import qualified GI.Gdk.Structs.Rectangle as Gdk.Rectangle
import qualified GI.Gdk.Unions.Event as Gdk.Event
import qualified GI.GdkPixbuf.Objects.Pixbuf as GdkPixbuf.Pixbuf
import qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import qualified GI.Gio.Interfaces.Icon as Gio.Icon
import qualified GI.Gtk.Callbacks as Gtk.Callbacks
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Flags as Gtk.Flags
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Objects.AccelGroup as Gtk.AccelGroup
import {-# SOURCE #-} qualified GI.Gtk.Objects.Clipboard as Gtk.Clipboard
import {-# SOURCE #-} qualified GI.Gtk.Objects.Container as Gtk.Container
import {-# SOURCE #-} qualified GI.Gtk.Objects.RcStyle as Gtk.RcStyle
import {-# SOURCE #-} qualified GI.Gtk.Objects.Settings as Gtk.Settings
import {-# SOURCE #-} qualified GI.Gtk.Objects.Style as Gtk.Style
import {-# SOURCE #-} qualified GI.Gtk.Objects.StyleContext as Gtk.StyleContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.Tooltip as Gtk.Tooltip
import {-# SOURCE #-} qualified GI.Gtk.Objects.Window as Gtk.Window
import {-# SOURCE #-} qualified GI.Gtk.Structs.Requisition as Gtk.Requisition
import {-# SOURCE #-} qualified GI.Gtk.Structs.SelectionData as Gtk.SelectionData
import {-# SOURCE #-} qualified GI.Gtk.Structs.TargetEntry as Gtk.TargetEntry
import {-# SOURCE #-} qualified GI.Gtk.Structs.TargetList as Gtk.TargetList
import {-# SOURCE #-} qualified GI.Gtk.Structs.WidgetPath as Gtk.WidgetPath
import qualified GI.Pango.Objects.Context as Pango.Context
import qualified GI.Pango.Objects.FontMap as Pango.FontMap
import qualified GI.Pango.Objects.Layout as Pango.Layout
import qualified GI.Pango.Structs.FontDescription as Pango.FontDescription

#endif

-- | Memory-managed wrapper type.
newtype Widget = Widget (SP.ManagedPtr Widget)
    deriving (Eq)

instance SP.ManagedPtrNewtype Widget where
    toManagedPtr (Widget p) = p

foreign import ccall "gtk_widget_get_type"
    c_gtk_widget_get_type :: IO B.Types.GType

instance B.Types.TypedObject Widget where
    glibType = c_gtk_widget_get_type

instance B.Types.GObject Widget

-- | Type class for types which can be safely cast to t'Widget', for instance with `toWidget`.
class (SP.GObject o, O.IsDescendantOf Widget o) => IsWidget o
instance (SP.GObject o, O.IsDescendantOf Widget o) => IsWidget o

instance O.HasParentTypes Widget
type instance O.ParentTypes Widget = '[GObject.Object.Object, Atk.ImplementorIface.ImplementorIface, Gtk.Buildable.Buildable]

-- | Cast to t'Widget', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toWidget :: (MIO.MonadIO m, IsWidget o) => o -> m Widget
toWidget = MIO.liftIO . B.ManagedPtr.unsafeCastTo Widget

-- | Convert t'Widget' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Widget) where
    gvalueGType_ = c_gtk_widget_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Widget)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Widget)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Widget ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveWidgetMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveWidgetMethod "activate" o = WidgetActivateMethodInfo
    ResolveWidgetMethod "addAccelerator" o = WidgetAddAcceleratorMethodInfo
    ResolveWidgetMethod "addChild" o = Gtk.Buildable.BuildableAddChildMethodInfo
    ResolveWidgetMethod "addDeviceEvents" o = WidgetAddDeviceEventsMethodInfo
    ResolveWidgetMethod "addEvents" o = WidgetAddEventsMethodInfo
    ResolveWidgetMethod "addMnemonicLabel" o = WidgetAddMnemonicLabelMethodInfo
    ResolveWidgetMethod "addTickCallback" o = WidgetAddTickCallbackMethodInfo
    ResolveWidgetMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveWidgetMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveWidgetMethod "canActivateAccel" o = WidgetCanActivateAccelMethodInfo
    ResolveWidgetMethod "childFocus" o = WidgetChildFocusMethodInfo
    ResolveWidgetMethod "childNotify" o = WidgetChildNotifyMethodInfo
    ResolveWidgetMethod "classPath" o = WidgetClassPathMethodInfo
    ResolveWidgetMethod "computeExpand" o = WidgetComputeExpandMethodInfo
    ResolveWidgetMethod "constructChild" o = Gtk.Buildable.BuildableConstructChildMethodInfo
    ResolveWidgetMethod "createPangoContext" o = WidgetCreatePangoContextMethodInfo
    ResolveWidgetMethod "createPangoLayout" o = WidgetCreatePangoLayoutMethodInfo
    ResolveWidgetMethod "customFinished" o = Gtk.Buildable.BuildableCustomFinishedMethodInfo
    ResolveWidgetMethod "customTagEnd" o = Gtk.Buildable.BuildableCustomTagEndMethodInfo
    ResolveWidgetMethod "customTagStart" o = Gtk.Buildable.BuildableCustomTagStartMethodInfo
    ResolveWidgetMethod "destroy" o = WidgetDestroyMethodInfo
    ResolveWidgetMethod "destroyed" o = WidgetDestroyedMethodInfo
    ResolveWidgetMethod "deviceIsShadowed" o = WidgetDeviceIsShadowedMethodInfo
    ResolveWidgetMethod "dragBegin" o = WidgetDragBeginMethodInfo
    ResolveWidgetMethod "dragBeginWithCoordinates" o = WidgetDragBeginWithCoordinatesMethodInfo
    ResolveWidgetMethod "dragCheckThreshold" o = WidgetDragCheckThresholdMethodInfo
    ResolveWidgetMethod "dragDestAddImageTargets" o = WidgetDragDestAddImageTargetsMethodInfo
    ResolveWidgetMethod "dragDestAddTextTargets" o = WidgetDragDestAddTextTargetsMethodInfo
    ResolveWidgetMethod "dragDestAddUriTargets" o = WidgetDragDestAddUriTargetsMethodInfo
    ResolveWidgetMethod "dragDestFindTarget" o = WidgetDragDestFindTargetMethodInfo
    ResolveWidgetMethod "dragDestGetTargetList" o = WidgetDragDestGetTargetListMethodInfo
    ResolveWidgetMethod "dragDestGetTrackMotion" o = WidgetDragDestGetTrackMotionMethodInfo
    ResolveWidgetMethod "dragDestSet" o = WidgetDragDestSetMethodInfo
    ResolveWidgetMethod "dragDestSetProxy" o = WidgetDragDestSetProxyMethodInfo
    ResolveWidgetMethod "dragDestSetTargetList" o = WidgetDragDestSetTargetListMethodInfo
    ResolveWidgetMethod "dragDestSetTrackMotion" o = WidgetDragDestSetTrackMotionMethodInfo
    ResolveWidgetMethod "dragDestUnset" o = WidgetDragDestUnsetMethodInfo
    ResolveWidgetMethod "dragGetData" o = WidgetDragGetDataMethodInfo
    ResolveWidgetMethod "dragHighlight" o = WidgetDragHighlightMethodInfo
    ResolveWidgetMethod "dragSourceAddImageTargets" o = WidgetDragSourceAddImageTargetsMethodInfo
    ResolveWidgetMethod "dragSourceAddTextTargets" o = WidgetDragSourceAddTextTargetsMethodInfo
    ResolveWidgetMethod "dragSourceAddUriTargets" o = WidgetDragSourceAddUriTargetsMethodInfo
    ResolveWidgetMethod "dragSourceGetTargetList" o = WidgetDragSourceGetTargetListMethodInfo
    ResolveWidgetMethod "dragSourceSet" o = WidgetDragSourceSetMethodInfo
    ResolveWidgetMethod "dragSourceSetIconGicon" o = WidgetDragSourceSetIconGiconMethodInfo
    ResolveWidgetMethod "dragSourceSetIconName" o = WidgetDragSourceSetIconNameMethodInfo
    ResolveWidgetMethod "dragSourceSetIconPixbuf" o = WidgetDragSourceSetIconPixbufMethodInfo
    ResolveWidgetMethod "dragSourceSetIconStock" o = WidgetDragSourceSetIconStockMethodInfo
    ResolveWidgetMethod "dragSourceSetTargetList" o = WidgetDragSourceSetTargetListMethodInfo
    ResolveWidgetMethod "dragSourceUnset" o = WidgetDragSourceUnsetMethodInfo
    ResolveWidgetMethod "dragUnhighlight" o = WidgetDragUnhighlightMethodInfo
    ResolveWidgetMethod "draw" o = WidgetDrawMethodInfo
    ResolveWidgetMethod "ensureStyle" o = WidgetEnsureStyleMethodInfo
    ResolveWidgetMethod "errorBell" o = WidgetErrorBellMethodInfo
    ResolveWidgetMethod "event" o = WidgetEventMethodInfo
    ResolveWidgetMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveWidgetMethod "freezeChildNotify" o = WidgetFreezeChildNotifyMethodInfo
    ResolveWidgetMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveWidgetMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveWidgetMethod "grabAdd" o = WidgetGrabAddMethodInfo
    ResolveWidgetMethod "grabDefault" o = WidgetGrabDefaultMethodInfo
    ResolveWidgetMethod "grabFocus" o = WidgetGrabFocusMethodInfo
    ResolveWidgetMethod "grabRemove" o = WidgetGrabRemoveMethodInfo
    ResolveWidgetMethod "hasDefault" o = WidgetHasDefaultMethodInfo
    ResolveWidgetMethod "hasFocus" o = WidgetHasFocusMethodInfo
    ResolveWidgetMethod "hasGrab" o = WidgetHasGrabMethodInfo
    ResolveWidgetMethod "hasRcStyle" o = WidgetHasRcStyleMethodInfo
    ResolveWidgetMethod "hasScreen" o = WidgetHasScreenMethodInfo
    ResolveWidgetMethod "hasVisibleFocus" o = WidgetHasVisibleFocusMethodInfo
    ResolveWidgetMethod "hide" o = WidgetHideMethodInfo
    ResolveWidgetMethod "hideOnDelete" o = WidgetHideOnDeleteMethodInfo
    ResolveWidgetMethod "inDestruction" o = WidgetInDestructionMethodInfo
    ResolveWidgetMethod "initTemplate" o = WidgetInitTemplateMethodInfo
    ResolveWidgetMethod "inputShapeCombineRegion" o = WidgetInputShapeCombineRegionMethodInfo
    ResolveWidgetMethod "insertActionGroup" o = WidgetInsertActionGroupMethodInfo
    ResolveWidgetMethod "intersect" o = WidgetIntersectMethodInfo
    ResolveWidgetMethod "isAncestor" o = WidgetIsAncestorMethodInfo
    ResolveWidgetMethod "isComposited" o = WidgetIsCompositedMethodInfo
    ResolveWidgetMethod "isDrawable" o = WidgetIsDrawableMethodInfo
    ResolveWidgetMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveWidgetMethod "isFocus" o = WidgetIsFocusMethodInfo
    ResolveWidgetMethod "isSensitive" o = WidgetIsSensitiveMethodInfo
    ResolveWidgetMethod "isToplevel" o = WidgetIsToplevelMethodInfo
    ResolveWidgetMethod "isVisible" o = WidgetIsVisibleMethodInfo
    ResolveWidgetMethod "keynavFailed" o = WidgetKeynavFailedMethodInfo
    ResolveWidgetMethod "listAccelClosures" o = WidgetListAccelClosuresMethodInfo
    ResolveWidgetMethod "listActionPrefixes" o = WidgetListActionPrefixesMethodInfo
    ResolveWidgetMethod "listMnemonicLabels" o = WidgetListMnemonicLabelsMethodInfo
    ResolveWidgetMethod "map" o = WidgetMapMethodInfo
    ResolveWidgetMethod "mnemonicActivate" o = WidgetMnemonicActivateMethodInfo
    ResolveWidgetMethod "modifyBase" o = WidgetModifyBaseMethodInfo
    ResolveWidgetMethod "modifyBg" o = WidgetModifyBgMethodInfo
    ResolveWidgetMethod "modifyCursor" o = WidgetModifyCursorMethodInfo
    ResolveWidgetMethod "modifyFg" o = WidgetModifyFgMethodInfo
    ResolveWidgetMethod "modifyFont" o = WidgetModifyFontMethodInfo
    ResolveWidgetMethod "modifyStyle" o = WidgetModifyStyleMethodInfo
    ResolveWidgetMethod "modifyText" o = WidgetModifyTextMethodInfo
    ResolveWidgetMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveWidgetMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveWidgetMethod "overrideBackgroundColor" o = WidgetOverrideBackgroundColorMethodInfo
    ResolveWidgetMethod "overrideColor" o = WidgetOverrideColorMethodInfo
    ResolveWidgetMethod "overrideCursor" o = WidgetOverrideCursorMethodInfo
    ResolveWidgetMethod "overrideFont" o = WidgetOverrideFontMethodInfo
    ResolveWidgetMethod "overrideSymbolicColor" o = WidgetOverrideSymbolicColorMethodInfo
    ResolveWidgetMethod "parserFinished" o = Gtk.Buildable.BuildableParserFinishedMethodInfo
    ResolveWidgetMethod "path" o = WidgetPathMethodInfo
    ResolveWidgetMethod "queueAllocate" o = WidgetQueueAllocateMethodInfo
    ResolveWidgetMethod "queueComputeExpand" o = WidgetQueueComputeExpandMethodInfo
    ResolveWidgetMethod "queueDraw" o = WidgetQueueDrawMethodInfo
    ResolveWidgetMethod "queueDrawArea" o = WidgetQueueDrawAreaMethodInfo
    ResolveWidgetMethod "queueDrawRegion" o = WidgetQueueDrawRegionMethodInfo
    ResolveWidgetMethod "queueResize" o = WidgetQueueResizeMethodInfo
    ResolveWidgetMethod "queueResizeNoRedraw" o = WidgetQueueResizeNoRedrawMethodInfo
    ResolveWidgetMethod "realize" o = WidgetRealizeMethodInfo
    ResolveWidgetMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveWidgetMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveWidgetMethod "regionIntersect" o = WidgetRegionIntersectMethodInfo
    ResolveWidgetMethod "registerWindow" o = WidgetRegisterWindowMethodInfo
    ResolveWidgetMethod "removeAccelerator" o = WidgetRemoveAcceleratorMethodInfo
    ResolveWidgetMethod "removeMnemonicLabel" o = WidgetRemoveMnemonicLabelMethodInfo
    ResolveWidgetMethod "removeTickCallback" o = WidgetRemoveTickCallbackMethodInfo
    ResolveWidgetMethod "renderIcon" o = WidgetRenderIconMethodInfo
    ResolveWidgetMethod "renderIconPixbuf" o = WidgetRenderIconPixbufMethodInfo
    ResolveWidgetMethod "reparent" o = WidgetReparentMethodInfo
    ResolveWidgetMethod "resetRcStyles" o = WidgetResetRcStylesMethodInfo
    ResolveWidgetMethod "resetStyle" o = WidgetResetStyleMethodInfo
    ResolveWidgetMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveWidgetMethod "sendExpose" o = WidgetSendExposeMethodInfo
    ResolveWidgetMethod "sendFocusChange" o = WidgetSendFocusChangeMethodInfo
    ResolveWidgetMethod "shapeCombineRegion" o = WidgetShapeCombineRegionMethodInfo
    ResolveWidgetMethod "show" o = WidgetShowMethodInfo
    ResolveWidgetMethod "showAll" o = WidgetShowAllMethodInfo
    ResolveWidgetMethod "showNow" o = WidgetShowNowMethodInfo
    ResolveWidgetMethod "sizeAllocate" o = WidgetSizeAllocateMethodInfo
    ResolveWidgetMethod "sizeAllocateWithBaseline" o = WidgetSizeAllocateWithBaselineMethodInfo
    ResolveWidgetMethod "sizeRequest" o = WidgetSizeRequestMethodInfo
    ResolveWidgetMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveWidgetMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveWidgetMethod "styleAttach" o = WidgetStyleAttachMethodInfo
    ResolveWidgetMethod "styleGetProperty" o = WidgetStyleGetPropertyMethodInfo
    ResolveWidgetMethod "thawChildNotify" o = WidgetThawChildNotifyMethodInfo
    ResolveWidgetMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveWidgetMethod "translateCoordinates" o = WidgetTranslateCoordinatesMethodInfo
    ResolveWidgetMethod "triggerTooltipQuery" o = WidgetTriggerTooltipQueryMethodInfo
    ResolveWidgetMethod "unmap" o = WidgetUnmapMethodInfo
    ResolveWidgetMethod "unparent" o = WidgetUnparentMethodInfo
    ResolveWidgetMethod "unrealize" o = WidgetUnrealizeMethodInfo
    ResolveWidgetMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveWidgetMethod "unregisterWindow" o = WidgetUnregisterWindowMethodInfo
    ResolveWidgetMethod "unsetStateFlags" o = WidgetUnsetStateFlagsMethodInfo
    ResolveWidgetMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveWidgetMethod "getAccessible" o = WidgetGetAccessibleMethodInfo
    ResolveWidgetMethod "getActionGroup" o = WidgetGetActionGroupMethodInfo
    ResolveWidgetMethod "getAllocatedBaseline" o = WidgetGetAllocatedBaselineMethodInfo
    ResolveWidgetMethod "getAllocatedHeight" o = WidgetGetAllocatedHeightMethodInfo
    ResolveWidgetMethod "getAllocatedSize" o = WidgetGetAllocatedSizeMethodInfo
    ResolveWidgetMethod "getAllocatedWidth" o = WidgetGetAllocatedWidthMethodInfo
    ResolveWidgetMethod "getAllocation" o = WidgetGetAllocationMethodInfo
    ResolveWidgetMethod "getAncestor" o = WidgetGetAncestorMethodInfo
    ResolveWidgetMethod "getAppPaintable" o = WidgetGetAppPaintableMethodInfo
    ResolveWidgetMethod "getCanDefault" o = WidgetGetCanDefaultMethodInfo
    ResolveWidgetMethod "getCanFocus" o = WidgetGetCanFocusMethodInfo
    ResolveWidgetMethod "getChildRequisition" o = WidgetGetChildRequisitionMethodInfo
    ResolveWidgetMethod "getChildVisible" o = WidgetGetChildVisibleMethodInfo
    ResolveWidgetMethod "getClip" o = WidgetGetClipMethodInfo
    ResolveWidgetMethod "getClipboard" o = WidgetGetClipboardMethodInfo
    ResolveWidgetMethod "getCompositeName" o = WidgetGetCompositeNameMethodInfo
    ResolveWidgetMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveWidgetMethod "getDeviceEnabled" o = WidgetGetDeviceEnabledMethodInfo
    ResolveWidgetMethod "getDeviceEvents" o = WidgetGetDeviceEventsMethodInfo
    ResolveWidgetMethod "getDirection" o = WidgetGetDirectionMethodInfo
    ResolveWidgetMethod "getDisplay" o = WidgetGetDisplayMethodInfo
    ResolveWidgetMethod "getDoubleBuffered" o = WidgetGetDoubleBufferedMethodInfo
    ResolveWidgetMethod "getEvents" o = WidgetGetEventsMethodInfo
    ResolveWidgetMethod "getFocusOnClick" o = WidgetGetFocusOnClickMethodInfo
    ResolveWidgetMethod "getFontMap" o = WidgetGetFontMapMethodInfo
    ResolveWidgetMethod "getFontOptions" o = WidgetGetFontOptionsMethodInfo
    ResolveWidgetMethod "getFrameClock" o = WidgetGetFrameClockMethodInfo
    ResolveWidgetMethod "getHalign" o = WidgetGetHalignMethodInfo
    ResolveWidgetMethod "getHasTooltip" o = WidgetGetHasTooltipMethodInfo
    ResolveWidgetMethod "getHasWindow" o = WidgetGetHasWindowMethodInfo
    ResolveWidgetMethod "getHexpand" o = WidgetGetHexpandMethodInfo
    ResolveWidgetMethod "getHexpandSet" o = WidgetGetHexpandSetMethodInfo
    ResolveWidgetMethod "getInternalChild" o = Gtk.Buildable.BuildableGetInternalChildMethodInfo
    ResolveWidgetMethod "getMapped" o = WidgetGetMappedMethodInfo
    ResolveWidgetMethod "getMarginBottom" o = WidgetGetMarginBottomMethodInfo
    ResolveWidgetMethod "getMarginEnd" o = WidgetGetMarginEndMethodInfo
    ResolveWidgetMethod "getMarginLeft" o = WidgetGetMarginLeftMethodInfo
    ResolveWidgetMethod "getMarginRight" o = WidgetGetMarginRightMethodInfo
    ResolveWidgetMethod "getMarginStart" o = WidgetGetMarginStartMethodInfo
    ResolveWidgetMethod "getMarginTop" o = WidgetGetMarginTopMethodInfo
    ResolveWidgetMethod "getModifierMask" o = WidgetGetModifierMaskMethodInfo
    ResolveWidgetMethod "getModifierStyle" o = WidgetGetModifierStyleMethodInfo
    ResolveWidgetMethod "getName" o = WidgetGetNameMethodInfo
    ResolveWidgetMethod "getNoShowAll" o = WidgetGetNoShowAllMethodInfo
    ResolveWidgetMethod "getOpacity" o = WidgetGetOpacityMethodInfo
    ResolveWidgetMethod "getPangoContext" o = WidgetGetPangoContextMethodInfo
    ResolveWidgetMethod "getParent" o = WidgetGetParentMethodInfo
    ResolveWidgetMethod "getParentWindow" o = WidgetGetParentWindowMethodInfo
    ResolveWidgetMethod "getPath" o = WidgetGetPathMethodInfo
    ResolveWidgetMethod "getPointer" o = WidgetGetPointerMethodInfo
    ResolveWidgetMethod "getPreferredHeight" o = WidgetGetPreferredHeightMethodInfo
    ResolveWidgetMethod "getPreferredHeightAndBaselineForWidth" o = WidgetGetPreferredHeightAndBaselineForWidthMethodInfo
    ResolveWidgetMethod "getPreferredHeightForWidth" o = WidgetGetPreferredHeightForWidthMethodInfo
    ResolveWidgetMethod "getPreferredSize" o = WidgetGetPreferredSizeMethodInfo
    ResolveWidgetMethod "getPreferredWidth" o = WidgetGetPreferredWidthMethodInfo
    ResolveWidgetMethod "getPreferredWidthForHeight" o = WidgetGetPreferredWidthForHeightMethodInfo
    ResolveWidgetMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveWidgetMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveWidgetMethod "getRealized" o = WidgetGetRealizedMethodInfo
    ResolveWidgetMethod "getReceivesDefault" o = WidgetGetReceivesDefaultMethodInfo
    ResolveWidgetMethod "getRequestMode" o = WidgetGetRequestModeMethodInfo
    ResolveWidgetMethod "getRequisition" o = WidgetGetRequisitionMethodInfo
    ResolveWidgetMethod "getRootWindow" o = WidgetGetRootWindowMethodInfo
    ResolveWidgetMethod "getScaleFactor" o = WidgetGetScaleFactorMethodInfo
    ResolveWidgetMethod "getScreen" o = WidgetGetScreenMethodInfo
    ResolveWidgetMethod "getSensitive" o = WidgetGetSensitiveMethodInfo
    ResolveWidgetMethod "getSettings" o = WidgetGetSettingsMethodInfo
    ResolveWidgetMethod "getSizeRequest" o = WidgetGetSizeRequestMethodInfo
    ResolveWidgetMethod "getState" o = WidgetGetStateMethodInfo
    ResolveWidgetMethod "getStateFlags" o = WidgetGetStateFlagsMethodInfo
    ResolveWidgetMethod "getStyle" o = WidgetGetStyleMethodInfo
    ResolveWidgetMethod "getStyleContext" o = WidgetGetStyleContextMethodInfo
    ResolveWidgetMethod "getSupportMultidevice" o = WidgetGetSupportMultideviceMethodInfo
    ResolveWidgetMethod "getTemplateChild" o = WidgetGetTemplateChildMethodInfo
    ResolveWidgetMethod "getTooltipMarkup" o = WidgetGetTooltipMarkupMethodInfo
    ResolveWidgetMethod "getTooltipText" o = WidgetGetTooltipTextMethodInfo
    ResolveWidgetMethod "getTooltipWindow" o = WidgetGetTooltipWindowMethodInfo
    ResolveWidgetMethod "getToplevel" o = WidgetGetToplevelMethodInfo
    ResolveWidgetMethod "getValign" o = WidgetGetValignMethodInfo
    ResolveWidgetMethod "getValignWithBaseline" o = WidgetGetValignWithBaselineMethodInfo
    ResolveWidgetMethod "getVexpand" o = WidgetGetVexpandMethodInfo
    ResolveWidgetMethod "getVexpandSet" o = WidgetGetVexpandSetMethodInfo
    ResolveWidgetMethod "getVisible" o = WidgetGetVisibleMethodInfo
    ResolveWidgetMethod "getVisual" o = WidgetGetVisualMethodInfo
    ResolveWidgetMethod "getWindow" o = WidgetGetWindowMethodInfo
    ResolveWidgetMethod "setAccelPath" o = WidgetSetAccelPathMethodInfo
    ResolveWidgetMethod "setAllocation" o = WidgetSetAllocationMethodInfo
    ResolveWidgetMethod "setAppPaintable" o = WidgetSetAppPaintableMethodInfo
    ResolveWidgetMethod "setBuildableProperty" o = Gtk.Buildable.BuildableSetBuildablePropertyMethodInfo
    ResolveWidgetMethod "setCanDefault" o = WidgetSetCanDefaultMethodInfo
    ResolveWidgetMethod "setCanFocus" o = WidgetSetCanFocusMethodInfo
    ResolveWidgetMethod "setChildVisible" o = WidgetSetChildVisibleMethodInfo
    ResolveWidgetMethod "setClip" o = WidgetSetClipMethodInfo
    ResolveWidgetMethod "setCompositeName" o = WidgetSetCompositeNameMethodInfo
    ResolveWidgetMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveWidgetMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveWidgetMethod "setDeviceEnabled" o = WidgetSetDeviceEnabledMethodInfo
    ResolveWidgetMethod "setDeviceEvents" o = WidgetSetDeviceEventsMethodInfo
    ResolveWidgetMethod "setDirection" o = WidgetSetDirectionMethodInfo
    ResolveWidgetMethod "setDoubleBuffered" o = WidgetSetDoubleBufferedMethodInfo
    ResolveWidgetMethod "setEvents" o = WidgetSetEventsMethodInfo
    ResolveWidgetMethod "setFocusOnClick" o = WidgetSetFocusOnClickMethodInfo
    ResolveWidgetMethod "setFontMap" o = WidgetSetFontMapMethodInfo
    ResolveWidgetMethod "setFontOptions" o = WidgetSetFontOptionsMethodInfo
    ResolveWidgetMethod "setHalign" o = WidgetSetHalignMethodInfo
    ResolveWidgetMethod "setHasTooltip" o = WidgetSetHasTooltipMethodInfo
    ResolveWidgetMethod "setHasWindow" o = WidgetSetHasWindowMethodInfo
    ResolveWidgetMethod "setHexpand" o = WidgetSetHexpandMethodInfo
    ResolveWidgetMethod "setHexpandSet" o = WidgetSetHexpandSetMethodInfo
    ResolveWidgetMethod "setMapped" o = WidgetSetMappedMethodInfo
    ResolveWidgetMethod "setMarginBottom" o = WidgetSetMarginBottomMethodInfo
    ResolveWidgetMethod "setMarginEnd" o = WidgetSetMarginEndMethodInfo
    ResolveWidgetMethod "setMarginLeft" o = WidgetSetMarginLeftMethodInfo
    ResolveWidgetMethod "setMarginRight" o = WidgetSetMarginRightMethodInfo
    ResolveWidgetMethod "setMarginStart" o = WidgetSetMarginStartMethodInfo
    ResolveWidgetMethod "setMarginTop" o = WidgetSetMarginTopMethodInfo
    ResolveWidgetMethod "setName" o = WidgetSetNameMethodInfo
    ResolveWidgetMethod "setNoShowAll" o = WidgetSetNoShowAllMethodInfo
    ResolveWidgetMethod "setOpacity" o = WidgetSetOpacityMethodInfo
    ResolveWidgetMethod "setParent" o = WidgetSetParentMethodInfo
    ResolveWidgetMethod "setParentWindow" o = WidgetSetParentWindowMethodInfo
    ResolveWidgetMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveWidgetMethod "setRealized" o = WidgetSetRealizedMethodInfo
    ResolveWidgetMethod "setReceivesDefault" o = WidgetSetReceivesDefaultMethodInfo
    ResolveWidgetMethod "setRedrawOnAllocate" o = WidgetSetRedrawOnAllocateMethodInfo
    ResolveWidgetMethod "setSensitive" o = WidgetSetSensitiveMethodInfo
    ResolveWidgetMethod "setSizeRequest" o = WidgetSetSizeRequestMethodInfo
    ResolveWidgetMethod "setState" o = WidgetSetStateMethodInfo
    ResolveWidgetMethod "setStateFlags" o = WidgetSetStateFlagsMethodInfo
    ResolveWidgetMethod "setStyle" o = WidgetSetStyleMethodInfo
    ResolveWidgetMethod "setSupportMultidevice" o = WidgetSetSupportMultideviceMethodInfo
    ResolveWidgetMethod "setTooltipMarkup" o = WidgetSetTooltipMarkupMethodInfo
    ResolveWidgetMethod "setTooltipText" o = WidgetSetTooltipTextMethodInfo
    ResolveWidgetMethod "setTooltipWindow" o = WidgetSetTooltipWindowMethodInfo
    ResolveWidgetMethod "setValign" o = WidgetSetValignMethodInfo
    ResolveWidgetMethod "setVexpand" o = WidgetSetVexpandMethodInfo
    ResolveWidgetMethod "setVexpandSet" o = WidgetSetVexpandSetMethodInfo
    ResolveWidgetMethod "setVisible" o = WidgetSetVisibleMethodInfo
    ResolveWidgetMethod "setVisual" o = WidgetSetVisualMethodInfo
    ResolveWidgetMethod "setWindow" o = WidgetSetWindowMethodInfo
    ResolveWidgetMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveWidgetMethod t Widget, O.OverloadedMethod info Widget p) => OL.IsLabel t (Widget -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveWidgetMethod t Widget, O.OverloadedMethod info Widget p, R.HasField t Widget p) => R.HasField t Widget p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveWidgetMethod t Widget, O.OverloadedMethodInfo info Widget) => OL.IsLabel t (O.MethodProxy info Widget) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- signal Widget::accel-closures-changed
-- | /No description available in the introspection data./
type WidgetAccelClosuresChangedCallback =
    IO ()

type C_WidgetAccelClosuresChangedCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetAccelClosuresChangedCallback`.
foreign import ccall "wrapper"
    mk_WidgetAccelClosuresChangedCallback :: C_WidgetAccelClosuresChangedCallback -> IO (FunPtr C_WidgetAccelClosuresChangedCallback)

wrap_WidgetAccelClosuresChangedCallback :: 
    GObject a => (a -> WidgetAccelClosuresChangedCallback) ->
    C_WidgetAccelClosuresChangedCallback
wrap_WidgetAccelClosuresChangedCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [accelClosuresChanged](#signal:accelClosuresChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #accelClosuresChanged callback
-- @
-- 
-- 
onWidgetAccelClosuresChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetAccelClosuresChangedCallback) -> m SignalHandlerId
onWidgetAccelClosuresChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetAccelClosuresChangedCallback wrapped
    wrapped'' <- mk_WidgetAccelClosuresChangedCallback wrapped'
    connectSignalFunPtr obj "accel-closures-changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [accelClosuresChanged](#signal:accelClosuresChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #accelClosuresChanged callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetAccelClosuresChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetAccelClosuresChangedCallback) -> m SignalHandlerId
afterWidgetAccelClosuresChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetAccelClosuresChangedCallback wrapped
    wrapped'' <- mk_WidgetAccelClosuresChangedCallback wrapped'
    connectSignalFunPtr obj "accel-closures-changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetAccelClosuresChangedSignalInfo
instance SignalInfo WidgetAccelClosuresChangedSignalInfo where
    type HaskellCallbackType WidgetAccelClosuresChangedSignalInfo = WidgetAccelClosuresChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetAccelClosuresChangedCallback cb
        cb'' <- mk_WidgetAccelClosuresChangedCallback cb'
        connectSignalFunPtr obj "accel-closures-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::accel-closures-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:accelClosuresChanged"})

#endif

-- signal Widget::button-press-event
-- | The [buttonPressEvent](#g:signal:buttonPressEvent) signal will be emitted when a button
-- (typically from a mouse) is pressed.
-- 
-- To receive this signal, the t'GI.Gdk.Objects.Window.Window' associated to the
-- widget needs to enable the @/GDK_BUTTON_PRESS_MASK/@ mask.
-- 
-- This signal will be sent to the grab widget if there is one.
type WidgetButtonPressEventCallback =
    Gdk.EventButton.EventButton
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventButton.EventButton' which triggered
    --   this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetButtonPressEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventButton.EventButton ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetButtonPressEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetButtonPressEventCallback :: C_WidgetButtonPressEventCallback -> IO (FunPtr C_WidgetButtonPressEventCallback)

wrap_WidgetButtonPressEventCallback :: 
    GObject a => (a -> WidgetButtonPressEventCallback) ->
    C_WidgetButtonPressEventCallback
wrap_WidgetButtonPressEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventButton.EventButton) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [buttonPressEvent](#signal:buttonPressEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #buttonPressEvent callback
-- @
-- 
-- 
onWidgetButtonPressEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetButtonPressEventCallback) -> m SignalHandlerId
onWidgetButtonPressEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetButtonPressEventCallback wrapped
    wrapped'' <- mk_WidgetButtonPressEventCallback wrapped'
    connectSignalFunPtr obj "button-press-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [buttonPressEvent](#signal:buttonPressEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #buttonPressEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetButtonPressEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetButtonPressEventCallback) -> m SignalHandlerId
afterWidgetButtonPressEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetButtonPressEventCallback wrapped
    wrapped'' <- mk_WidgetButtonPressEventCallback wrapped'
    connectSignalFunPtr obj "button-press-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetButtonPressEventSignalInfo
instance SignalInfo WidgetButtonPressEventSignalInfo where
    type HaskellCallbackType WidgetButtonPressEventSignalInfo = WidgetButtonPressEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetButtonPressEventCallback cb
        cb'' <- mk_WidgetButtonPressEventCallback cb'
        connectSignalFunPtr obj "button-press-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::button-press-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:buttonPressEvent"})

#endif

-- signal Widget::button-release-event
-- | The [buttonReleaseEvent](#g:signal:buttonReleaseEvent) signal will be emitted when a button
-- (typically from a mouse) is released.
-- 
-- To receive this signal, the t'GI.Gdk.Objects.Window.Window' associated to the
-- widget needs to enable the @/GDK_BUTTON_RELEASE_MASK/@ mask.
-- 
-- This signal will be sent to the grab widget if there is one.
type WidgetButtonReleaseEventCallback =
    Gdk.EventButton.EventButton
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventButton.EventButton' which triggered
    --   this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetButtonReleaseEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventButton.EventButton ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetButtonReleaseEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetButtonReleaseEventCallback :: C_WidgetButtonReleaseEventCallback -> IO (FunPtr C_WidgetButtonReleaseEventCallback)

wrap_WidgetButtonReleaseEventCallback :: 
    GObject a => (a -> WidgetButtonReleaseEventCallback) ->
    C_WidgetButtonReleaseEventCallback
wrap_WidgetButtonReleaseEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventButton.EventButton) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [buttonReleaseEvent](#signal:buttonReleaseEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #buttonReleaseEvent callback
-- @
-- 
-- 
onWidgetButtonReleaseEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetButtonReleaseEventCallback) -> m SignalHandlerId
onWidgetButtonReleaseEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetButtonReleaseEventCallback wrapped
    wrapped'' <- mk_WidgetButtonReleaseEventCallback wrapped'
    connectSignalFunPtr obj "button-release-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [buttonReleaseEvent](#signal:buttonReleaseEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #buttonReleaseEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetButtonReleaseEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetButtonReleaseEventCallback) -> m SignalHandlerId
afterWidgetButtonReleaseEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetButtonReleaseEventCallback wrapped
    wrapped'' <- mk_WidgetButtonReleaseEventCallback wrapped'
    connectSignalFunPtr obj "button-release-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetButtonReleaseEventSignalInfo
instance SignalInfo WidgetButtonReleaseEventSignalInfo where
    type HaskellCallbackType WidgetButtonReleaseEventSignalInfo = WidgetButtonReleaseEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetButtonReleaseEventCallback cb
        cb'' <- mk_WidgetButtonReleaseEventCallback cb'
        connectSignalFunPtr obj "button-release-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::button-release-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:buttonReleaseEvent"})

#endif

-- signal Widget::can-activate-accel
-- | Determines whether an accelerator that activates the signal
-- identified by /@signalId@/ can currently be activated.
-- This signal is present to allow applications and derived
-- widgets to override the default t'GI.Gtk.Objects.Widget.Widget' handling
-- for determining whether an accelerator can be activated.
type WidgetCanActivateAccelCallback =
    Word32
    -- ^ /@signalId@/: the ID of a signal installed on /@widget@/
    -> IO Bool
    -- ^ __Returns:__ 'P.True' if the signal can be activated.

type C_WidgetCanActivateAccelCallback =
    Ptr Widget ->                           -- object
    Word32 ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetCanActivateAccelCallback`.
foreign import ccall "wrapper"
    mk_WidgetCanActivateAccelCallback :: C_WidgetCanActivateAccelCallback -> IO (FunPtr C_WidgetCanActivateAccelCallback)

wrap_WidgetCanActivateAccelCallback :: 
    GObject a => (a -> WidgetCanActivateAccelCallback) ->
    C_WidgetCanActivateAccelCallback
wrap_WidgetCanActivateAccelCallback gi'cb gi'selfPtr signalId _ = do
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  signalId
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [canActivateAccel](#signal:canActivateAccel) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #canActivateAccel callback
-- @
-- 
-- 
onWidgetCanActivateAccel :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetCanActivateAccelCallback) -> m SignalHandlerId
onWidgetCanActivateAccel obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetCanActivateAccelCallback wrapped
    wrapped'' <- mk_WidgetCanActivateAccelCallback wrapped'
    connectSignalFunPtr obj "can-activate-accel" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [canActivateAccel](#signal:canActivateAccel) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #canActivateAccel callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetCanActivateAccel :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetCanActivateAccelCallback) -> m SignalHandlerId
afterWidgetCanActivateAccel obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetCanActivateAccelCallback wrapped
    wrapped'' <- mk_WidgetCanActivateAccelCallback wrapped'
    connectSignalFunPtr obj "can-activate-accel" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetCanActivateAccelSignalInfo
instance SignalInfo WidgetCanActivateAccelSignalInfo where
    type HaskellCallbackType WidgetCanActivateAccelSignalInfo = WidgetCanActivateAccelCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetCanActivateAccelCallback cb
        cb'' <- mk_WidgetCanActivateAccelCallback cb'
        connectSignalFunPtr obj "can-activate-accel" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::can-activate-accel"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:canActivateAccel"})

#endif

-- signal Widget::child-notify
-- | The [childNotify](#g:signal:childNotify) signal is emitted for each
-- [child property][child-properties]  that has
-- changed on an object. The signal\'s detail holds the property name.
type WidgetChildNotifyCallback =
    GParamSpec
    -- ^ /@childProperty@/: the t'GI.GObject.Objects.ParamSpec.ParamSpec' of the changed child property
    -> IO ()

type C_WidgetChildNotifyCallback =
    Ptr Widget ->                           -- object
    Ptr GParamSpec ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetChildNotifyCallback`.
foreign import ccall "wrapper"
    mk_WidgetChildNotifyCallback :: C_WidgetChildNotifyCallback -> IO (FunPtr C_WidgetChildNotifyCallback)

wrap_WidgetChildNotifyCallback :: 
    GObject a => (a -> WidgetChildNotifyCallback) ->
    C_WidgetChildNotifyCallback
wrap_WidgetChildNotifyCallback gi'cb gi'selfPtr childProperty _ = do
    childProperty' <- B.GParamSpec.newGParamSpecFromPtr childProperty
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  childProperty'


-- | Connect a signal handler for the [childNotify](#signal:childNotify) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #childNotify callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@child-notify::detail@” instead.
-- 
onWidgetChildNotify :: (IsWidget a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => WidgetChildNotifyCallback) -> m SignalHandlerId
onWidgetChildNotify obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetChildNotifyCallback wrapped
    wrapped'' <- mk_WidgetChildNotifyCallback wrapped'
    connectSignalFunPtr obj "child-notify" wrapped'' SignalConnectBefore detail

-- | Connect a signal handler for the [childNotify](#signal:childNotify) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #childNotify callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@child-notify::detail@” instead.
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetChildNotify :: (IsWidget a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => WidgetChildNotifyCallback) -> m SignalHandlerId
afterWidgetChildNotify obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetChildNotifyCallback wrapped
    wrapped'' <- mk_WidgetChildNotifyCallback wrapped'
    connectSignalFunPtr obj "child-notify" wrapped'' SignalConnectAfter detail


#if defined(ENABLE_OVERLOADING)
data WidgetChildNotifySignalInfo
instance SignalInfo WidgetChildNotifySignalInfo where
    type HaskellCallbackType WidgetChildNotifySignalInfo = WidgetChildNotifyCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetChildNotifyCallback cb
        cb'' <- mk_WidgetChildNotifyCallback cb'
        connectSignalFunPtr obj "child-notify" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::child-notify"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:childNotify"})

#endif

-- signal Widget::composited-changed
{-# DEPRECATED WidgetCompositedChangedCallback ["(Since version 3.22)","Use GdkScreen[compositedChanged](#g:signal:compositedChanged) instead."] #-}
-- | The [compositedChanged](#g:signal:compositedChanged) signal is emitted when the composited
-- status of /@widgets@/ screen changes.
-- See 'GI.Gdk.Objects.Screen.screenIsComposited'.
type WidgetCompositedChangedCallback =
    IO ()

type C_WidgetCompositedChangedCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetCompositedChangedCallback`.
foreign import ccall "wrapper"
    mk_WidgetCompositedChangedCallback :: C_WidgetCompositedChangedCallback -> IO (FunPtr C_WidgetCompositedChangedCallback)

wrap_WidgetCompositedChangedCallback :: 
    GObject a => (a -> WidgetCompositedChangedCallback) ->
    C_WidgetCompositedChangedCallback
wrap_WidgetCompositedChangedCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [compositedChanged](#signal:compositedChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #compositedChanged callback
-- @
-- 
-- 
onWidgetCompositedChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetCompositedChangedCallback) -> m SignalHandlerId
onWidgetCompositedChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetCompositedChangedCallback wrapped
    wrapped'' <- mk_WidgetCompositedChangedCallback wrapped'
    connectSignalFunPtr obj "composited-changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [compositedChanged](#signal:compositedChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #compositedChanged callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetCompositedChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetCompositedChangedCallback) -> m SignalHandlerId
afterWidgetCompositedChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetCompositedChangedCallback wrapped
    wrapped'' <- mk_WidgetCompositedChangedCallback wrapped'
    connectSignalFunPtr obj "composited-changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetCompositedChangedSignalInfo
instance SignalInfo WidgetCompositedChangedSignalInfo where
    type HaskellCallbackType WidgetCompositedChangedSignalInfo = WidgetCompositedChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetCompositedChangedCallback cb
        cb'' <- mk_WidgetCompositedChangedCallback cb'
        connectSignalFunPtr obj "composited-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::composited-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:compositedChanged"})

#endif

-- signal Widget::configure-event
-- | The [configureEvent](#g:signal:configureEvent) signal will be emitted when the size, position or
-- stacking of the /@widget@/\'s window has changed.
-- 
-- To receive this signal, the t'GI.Gdk.Objects.Window.Window' associated to the widget needs
-- to enable the @/GDK_STRUCTURE_MASK/@ mask. GDK will enable this mask
-- automatically for all new windows.
type WidgetConfigureEventCallback =
    Gdk.EventConfigure.EventConfigure
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventConfigure.EventConfigure' which triggered
    --   this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetConfigureEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventConfigure.EventConfigure ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetConfigureEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetConfigureEventCallback :: C_WidgetConfigureEventCallback -> IO (FunPtr C_WidgetConfigureEventCallback)

wrap_WidgetConfigureEventCallback :: 
    GObject a => (a -> WidgetConfigureEventCallback) ->
    C_WidgetConfigureEventCallback
wrap_WidgetConfigureEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventConfigure.EventConfigure) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [configureEvent](#signal:configureEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #configureEvent callback
-- @
-- 
-- 
onWidgetConfigureEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetConfigureEventCallback) -> m SignalHandlerId
onWidgetConfigureEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetConfigureEventCallback wrapped
    wrapped'' <- mk_WidgetConfigureEventCallback wrapped'
    connectSignalFunPtr obj "configure-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [configureEvent](#signal:configureEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #configureEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetConfigureEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetConfigureEventCallback) -> m SignalHandlerId
afterWidgetConfigureEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetConfigureEventCallback wrapped
    wrapped'' <- mk_WidgetConfigureEventCallback wrapped'
    connectSignalFunPtr obj "configure-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetConfigureEventSignalInfo
instance SignalInfo WidgetConfigureEventSignalInfo where
    type HaskellCallbackType WidgetConfigureEventSignalInfo = WidgetConfigureEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetConfigureEventCallback cb
        cb'' <- mk_WidgetConfigureEventCallback cb'
        connectSignalFunPtr obj "configure-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::configure-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:configureEvent"})

#endif

-- signal Widget::damage-event
-- | Emitted when a redirected window belonging to /@widget@/ gets drawn into.
-- The region\/area members of the event shows what area of the redirected
-- drawable was drawn into.
-- 
-- /Since: 2.14/
type WidgetDamageEventCallback =
    Gdk.EventExpose.EventExpose
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventExpose.EventExpose' event
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetDamageEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventExpose.EventExpose ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetDamageEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetDamageEventCallback :: C_WidgetDamageEventCallback -> IO (FunPtr C_WidgetDamageEventCallback)

wrap_WidgetDamageEventCallback :: 
    GObject a => (a -> WidgetDamageEventCallback) ->
    C_WidgetDamageEventCallback
wrap_WidgetDamageEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventExpose.EventExpose) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [damageEvent](#signal:damageEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #damageEvent callback
-- @
-- 
-- 
onWidgetDamageEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDamageEventCallback) -> m SignalHandlerId
onWidgetDamageEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDamageEventCallback wrapped
    wrapped'' <- mk_WidgetDamageEventCallback wrapped'
    connectSignalFunPtr obj "damage-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [damageEvent](#signal:damageEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #damageEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDamageEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDamageEventCallback) -> m SignalHandlerId
afterWidgetDamageEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDamageEventCallback wrapped
    wrapped'' <- mk_WidgetDamageEventCallback wrapped'
    connectSignalFunPtr obj "damage-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDamageEventSignalInfo
instance SignalInfo WidgetDamageEventSignalInfo where
    type HaskellCallbackType WidgetDamageEventSignalInfo = WidgetDamageEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDamageEventCallback cb
        cb'' <- mk_WidgetDamageEventCallback cb'
        connectSignalFunPtr obj "damage-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::damage-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:damageEvent"})

#endif

-- signal Widget::delete-event
-- | The [deleteEvent](#g:signal:deleteEvent) signal is emitted if a user requests that
-- a toplevel window is closed. The default handler for this signal
-- destroys the window. Connecting 'GI.Gtk.Objects.Widget.widgetHideOnDelete' to
-- this signal will cause the window to be hidden instead, so that
-- it can later be shown again without reconstructing it.
type WidgetDeleteEventCallback =
    Gdk.Event.Event
    -- ^ /@event@/: the event which triggered this signal
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetDeleteEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.Event.Event ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetDeleteEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetDeleteEventCallback :: C_WidgetDeleteEventCallback -> IO (FunPtr C_WidgetDeleteEventCallback)

wrap_WidgetDeleteEventCallback :: 
    GObject a => (a -> WidgetDeleteEventCallback) ->
    C_WidgetDeleteEventCallback
wrap_WidgetDeleteEventCallback gi'cb gi'selfPtr event _ = do
    B.ManagedPtr.withTransient  event $ \event' -> do
        result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
        let result' = (P.fromIntegral . P.fromEnum) result
        return result'


-- | Connect a signal handler for the [deleteEvent](#signal:deleteEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #deleteEvent callback
-- @
-- 
-- 
onWidgetDeleteEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDeleteEventCallback) -> m SignalHandlerId
onWidgetDeleteEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDeleteEventCallback wrapped
    wrapped'' <- mk_WidgetDeleteEventCallback wrapped'
    connectSignalFunPtr obj "delete-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [deleteEvent](#signal:deleteEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #deleteEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDeleteEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDeleteEventCallback) -> m SignalHandlerId
afterWidgetDeleteEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDeleteEventCallback wrapped
    wrapped'' <- mk_WidgetDeleteEventCallback wrapped'
    connectSignalFunPtr obj "delete-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDeleteEventSignalInfo
instance SignalInfo WidgetDeleteEventSignalInfo where
    type HaskellCallbackType WidgetDeleteEventSignalInfo = WidgetDeleteEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDeleteEventCallback cb
        cb'' <- mk_WidgetDeleteEventCallback cb'
        connectSignalFunPtr obj "delete-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::delete-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:deleteEvent"})

#endif

-- signal Widget::destroy
-- | Signals that all holders of a reference to the widget should release
-- the reference that they hold. May result in finalization of the widget
-- if all references are released.
-- 
-- This signal is not suitable for saving widget state.
type WidgetDestroyCallback =
    IO ()

type C_WidgetDestroyCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetDestroyCallback`.
foreign import ccall "wrapper"
    mk_WidgetDestroyCallback :: C_WidgetDestroyCallback -> IO (FunPtr C_WidgetDestroyCallback)

wrap_WidgetDestroyCallback :: 
    GObject a => (a -> WidgetDestroyCallback) ->
    C_WidgetDestroyCallback
wrap_WidgetDestroyCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [destroy](#signal:destroy) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #destroy callback
-- @
-- 
-- 
onWidgetDestroy :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDestroyCallback) -> m SignalHandlerId
onWidgetDestroy obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDestroyCallback wrapped
    wrapped'' <- mk_WidgetDestroyCallback wrapped'
    connectSignalFunPtr obj "destroy" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [destroy](#signal:destroy) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #destroy callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDestroy :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDestroyCallback) -> m SignalHandlerId
afterWidgetDestroy obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDestroyCallback wrapped
    wrapped'' <- mk_WidgetDestroyCallback wrapped'
    connectSignalFunPtr obj "destroy" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDestroySignalInfo
instance SignalInfo WidgetDestroySignalInfo where
    type HaskellCallbackType WidgetDestroySignalInfo = WidgetDestroyCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDestroyCallback cb
        cb'' <- mk_WidgetDestroyCallback cb'
        connectSignalFunPtr obj "destroy" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::destroy"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:destroy"})

#endif

-- signal Widget::destroy-event
-- | The [destroyEvent](#g:signal:destroyEvent) signal is emitted when a t'GI.Gdk.Objects.Window.Window' is destroyed.
-- You rarely get this signal, because most widgets disconnect themselves
-- from their window before they destroy it, so no widget owns the
-- window at destroy time.
-- 
-- To receive this signal, the t'GI.Gdk.Objects.Window.Window' associated to the widget needs
-- to enable the @/GDK_STRUCTURE_MASK/@ mask. GDK will enable this mask
-- automatically for all new windows.
type WidgetDestroyEventCallback =
    Gdk.Event.Event
    -- ^ /@event@/: the event which triggered this signal
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetDestroyEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.Event.Event ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetDestroyEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetDestroyEventCallback :: C_WidgetDestroyEventCallback -> IO (FunPtr C_WidgetDestroyEventCallback)

wrap_WidgetDestroyEventCallback :: 
    GObject a => (a -> WidgetDestroyEventCallback) ->
    C_WidgetDestroyEventCallback
wrap_WidgetDestroyEventCallback gi'cb gi'selfPtr event _ = do
    B.ManagedPtr.withTransient  event $ \event' -> do
        result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
        let result' = (P.fromIntegral . P.fromEnum) result
        return result'


-- | Connect a signal handler for the [destroyEvent](#signal:destroyEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #destroyEvent callback
-- @
-- 
-- 
onWidgetDestroyEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDestroyEventCallback) -> m SignalHandlerId
onWidgetDestroyEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDestroyEventCallback wrapped
    wrapped'' <- mk_WidgetDestroyEventCallback wrapped'
    connectSignalFunPtr obj "destroy-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [destroyEvent](#signal:destroyEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #destroyEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDestroyEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDestroyEventCallback) -> m SignalHandlerId
afterWidgetDestroyEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDestroyEventCallback wrapped
    wrapped'' <- mk_WidgetDestroyEventCallback wrapped'
    connectSignalFunPtr obj "destroy-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDestroyEventSignalInfo
instance SignalInfo WidgetDestroyEventSignalInfo where
    type HaskellCallbackType WidgetDestroyEventSignalInfo = WidgetDestroyEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDestroyEventCallback cb
        cb'' <- mk_WidgetDestroyEventCallback cb'
        connectSignalFunPtr obj "destroy-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::destroy-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:destroyEvent"})

#endif

-- signal Widget::direction-changed
-- | The [directionChanged](#g:signal:directionChanged) signal is emitted when the text direction
-- of a widget changes.
type WidgetDirectionChangedCallback =
    Gtk.Enums.TextDirection
    -- ^ /@previousDirection@/: the previous text direction of /@widget@/
    -> IO ()

type C_WidgetDirectionChangedCallback =
    Ptr Widget ->                           -- object
    CUInt ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetDirectionChangedCallback`.
foreign import ccall "wrapper"
    mk_WidgetDirectionChangedCallback :: C_WidgetDirectionChangedCallback -> IO (FunPtr C_WidgetDirectionChangedCallback)

wrap_WidgetDirectionChangedCallback :: 
    GObject a => (a -> WidgetDirectionChangedCallback) ->
    C_WidgetDirectionChangedCallback
wrap_WidgetDirectionChangedCallback gi'cb gi'selfPtr previousDirection _ = do
    let previousDirection' = (toEnum . fromIntegral) previousDirection
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  previousDirection'


-- | Connect a signal handler for the [directionChanged](#signal:directionChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #directionChanged callback
-- @
-- 
-- 
onWidgetDirectionChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDirectionChangedCallback) -> m SignalHandlerId
onWidgetDirectionChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDirectionChangedCallback wrapped
    wrapped'' <- mk_WidgetDirectionChangedCallback wrapped'
    connectSignalFunPtr obj "direction-changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [directionChanged](#signal:directionChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #directionChanged callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDirectionChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDirectionChangedCallback) -> m SignalHandlerId
afterWidgetDirectionChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDirectionChangedCallback wrapped
    wrapped'' <- mk_WidgetDirectionChangedCallback wrapped'
    connectSignalFunPtr obj "direction-changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDirectionChangedSignalInfo
instance SignalInfo WidgetDirectionChangedSignalInfo where
    type HaskellCallbackType WidgetDirectionChangedSignalInfo = WidgetDirectionChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDirectionChangedCallback cb
        cb'' <- mk_WidgetDirectionChangedCallback cb'
        connectSignalFunPtr obj "direction-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::direction-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:directionChanged"})

#endif

-- signal Widget::drag-begin
-- | The [dragBegin](#g:signal:dragBegin) signal is emitted on the drag source when a drag is
-- started. A typical reason to connect to this signal is to set up a
-- custom drag icon with e.g. 'GI.Gtk.Objects.Widget.widgetDragSourceSetIconPixbuf'.
-- 
-- Note that some widgets set up a drag icon in the default handler of
-- this signal, so you may have to use @/g_signal_connect_after()/@ to
-- override what the default handler did.
type WidgetDragBeginCallback =
    Gdk.DragContext.DragContext
    -- ^ /@context@/: the drag context
    -> IO ()

type C_WidgetDragBeginCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.DragContext.DragContext ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetDragBeginCallback`.
foreign import ccall "wrapper"
    mk_WidgetDragBeginCallback :: C_WidgetDragBeginCallback -> IO (FunPtr C_WidgetDragBeginCallback)

wrap_WidgetDragBeginCallback :: 
    GObject a => (a -> WidgetDragBeginCallback) ->
    C_WidgetDragBeginCallback
wrap_WidgetDragBeginCallback gi'cb gi'selfPtr context _ = do
    context' <- (newObject Gdk.DragContext.DragContext) context
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  context'


-- | Connect a signal handler for the [dragBegin](#signal:dragBegin) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #dragBegin callback
-- @
-- 
-- 
onWidgetDragBegin :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragBeginCallback) -> m SignalHandlerId
onWidgetDragBegin obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragBeginCallback wrapped
    wrapped'' <- mk_WidgetDragBeginCallback wrapped'
    connectSignalFunPtr obj "drag-begin" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [dragBegin](#signal:dragBegin) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #dragBegin callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDragBegin :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragBeginCallback) -> m SignalHandlerId
afterWidgetDragBegin obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragBeginCallback wrapped
    wrapped'' <- mk_WidgetDragBeginCallback wrapped'
    connectSignalFunPtr obj "drag-begin" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDragBeginSignalInfo
instance SignalInfo WidgetDragBeginSignalInfo where
    type HaskellCallbackType WidgetDragBeginSignalInfo = WidgetDragBeginCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDragBeginCallback cb
        cb'' <- mk_WidgetDragBeginCallback cb'
        connectSignalFunPtr obj "drag-begin" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::drag-begin"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:dragBegin"})

#endif

-- signal Widget::drag-data-delete
-- | The [dragDataDelete](#g:signal:dragDataDelete) signal is emitted on the drag source when a drag
-- with the action 'GI.Gdk.Flags.DragActionMove' is successfully completed. The signal
-- handler is responsible for deleting the data that has been dropped. What
-- \"delete\" means depends on the context of the drag operation.
type WidgetDragDataDeleteCallback =
    Gdk.DragContext.DragContext
    -- ^ /@context@/: the drag context
    -> IO ()

type C_WidgetDragDataDeleteCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.DragContext.DragContext ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetDragDataDeleteCallback`.
foreign import ccall "wrapper"
    mk_WidgetDragDataDeleteCallback :: C_WidgetDragDataDeleteCallback -> IO (FunPtr C_WidgetDragDataDeleteCallback)

wrap_WidgetDragDataDeleteCallback :: 
    GObject a => (a -> WidgetDragDataDeleteCallback) ->
    C_WidgetDragDataDeleteCallback
wrap_WidgetDragDataDeleteCallback gi'cb gi'selfPtr context _ = do
    context' <- (newObject Gdk.DragContext.DragContext) context
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  context'


-- | Connect a signal handler for the [dragDataDelete](#signal:dragDataDelete) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #dragDataDelete callback
-- @
-- 
-- 
onWidgetDragDataDelete :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragDataDeleteCallback) -> m SignalHandlerId
onWidgetDragDataDelete obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragDataDeleteCallback wrapped
    wrapped'' <- mk_WidgetDragDataDeleteCallback wrapped'
    connectSignalFunPtr obj "drag-data-delete" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [dragDataDelete](#signal:dragDataDelete) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #dragDataDelete callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDragDataDelete :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragDataDeleteCallback) -> m SignalHandlerId
afterWidgetDragDataDelete obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragDataDeleteCallback wrapped
    wrapped'' <- mk_WidgetDragDataDeleteCallback wrapped'
    connectSignalFunPtr obj "drag-data-delete" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDragDataDeleteSignalInfo
instance SignalInfo WidgetDragDataDeleteSignalInfo where
    type HaskellCallbackType WidgetDragDataDeleteSignalInfo = WidgetDragDataDeleteCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDragDataDeleteCallback cb
        cb'' <- mk_WidgetDragDataDeleteCallback cb'
        connectSignalFunPtr obj "drag-data-delete" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::drag-data-delete"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:dragDataDelete"})

#endif

-- signal Widget::drag-data-get
-- | The [dragDataGet](#g:signal:dragDataGet) signal is emitted on the drag source when the drop
-- site requests the data which is dragged. It is the responsibility of
-- the signal handler to fill /@data@/ with the data in the format which
-- is indicated by /@info@/. See 'GI.Gtk.Structs.SelectionData.selectionDataSet' and
-- 'GI.Gtk.Structs.SelectionData.selectionDataSetText'.
type WidgetDragDataGetCallback =
    Gdk.DragContext.DragContext
    -- ^ /@context@/: the drag context
    -> Gtk.SelectionData.SelectionData
    -- ^ /@data@/: the t'GI.Gtk.Structs.SelectionData.SelectionData' to be filled with the dragged data
    -> Word32
    -- ^ /@info@/: the info that has been registered with the target in the
    --        t'GI.Gtk.Structs.TargetList.TargetList'
    -> Word32
    -- ^ /@time@/: the timestamp at which the data was requested
    -> IO ()

type C_WidgetDragDataGetCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.DragContext.DragContext ->
    Ptr Gtk.SelectionData.SelectionData ->
    Word32 ->
    Word32 ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetDragDataGetCallback`.
foreign import ccall "wrapper"
    mk_WidgetDragDataGetCallback :: C_WidgetDragDataGetCallback -> IO (FunPtr C_WidgetDragDataGetCallback)

wrap_WidgetDragDataGetCallback :: 
    GObject a => (a -> WidgetDragDataGetCallback) ->
    C_WidgetDragDataGetCallback
wrap_WidgetDragDataGetCallback gi'cb gi'selfPtr context data_ info time _ = do
    context' <- (newObject Gdk.DragContext.DragContext) context
    B.ManagedPtr.withTransient  data_ $ \data_' -> do
        B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  context' data_' info time


-- | Connect a signal handler for the [dragDataGet](#signal:dragDataGet) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #dragDataGet callback
-- @
-- 
-- 
onWidgetDragDataGet :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragDataGetCallback) -> m SignalHandlerId
onWidgetDragDataGet obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragDataGetCallback wrapped
    wrapped'' <- mk_WidgetDragDataGetCallback wrapped'
    connectSignalFunPtr obj "drag-data-get" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [dragDataGet](#signal:dragDataGet) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #dragDataGet callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDragDataGet :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragDataGetCallback) -> m SignalHandlerId
afterWidgetDragDataGet obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragDataGetCallback wrapped
    wrapped'' <- mk_WidgetDragDataGetCallback wrapped'
    connectSignalFunPtr obj "drag-data-get" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDragDataGetSignalInfo
instance SignalInfo WidgetDragDataGetSignalInfo where
    type HaskellCallbackType WidgetDragDataGetSignalInfo = WidgetDragDataGetCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDragDataGetCallback cb
        cb'' <- mk_WidgetDragDataGetCallback cb'
        connectSignalFunPtr obj "drag-data-get" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::drag-data-get"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:dragDataGet"})

#endif

-- signal Widget::drag-data-received
-- | The [dragDataReceived](#g:signal:dragDataReceived) signal is emitted on the drop site when the
-- dragged data has been received. If the data was received in order to
-- determine whether the drop will be accepted, the handler is expected
-- to call 'GI.Gdk.Functions.dragStatus' and not finish the drag.
-- If the data was received in response to a [Widget::dragDrop]("GI.Gtk.Objects.Widget#g:signal:dragDrop") signal
-- (and this is the last target to be received), the handler for this
-- signal is expected to process the received data and then call
-- 'GI.Gtk.Functions.dragFinish', setting the /@success@/ parameter depending on
-- whether the data was processed successfully.
-- 
-- Applications must create some means to determine why the signal was emitted
-- and therefore whether to call 'GI.Gdk.Functions.dragStatus' or 'GI.Gtk.Functions.dragFinish'.
-- 
-- The handler may inspect the selected action with
-- 'GI.Gdk.Objects.DragContext.dragContextGetSelectedAction' before calling
-- 'GI.Gtk.Functions.dragFinish', e.g. to implement 'GI.Gdk.Flags.DragActionAsk' as
-- shown in the following example:
-- 
-- === /C code/
-- >
-- >void
-- >drag_data_received (GtkWidget          *widget,
-- >                    GdkDragContext     *context,
-- >                    gint                x,
-- >                    gint                y,
-- >                    GtkSelectionData   *data,
-- >                    guint               info,
-- >                    guint               time)
-- >{
-- >  if ((data->length >= 0) && (data->format == 8))
-- >    {
-- >      GdkDragAction action;
-- >
-- >      // handle data here
-- >
-- >      action = gdk_drag_context_get_selected_action (context);
-- >      if (action == GDK_ACTION_ASK)
-- >        {
-- >          GtkWidget *dialog;
-- >          gint response;
-- >
-- >          dialog = gtk_message_dialog_new (NULL,
-- >                                           GTK_DIALOG_MODAL |
-- >                                           GTK_DIALOG_DESTROY_WITH_PARENT,
-- >                                           GTK_MESSAGE_INFO,
-- >                                           GTK_BUTTONS_YES_NO,
-- >                                           "Move the data ?\n");
-- >          response = gtk_dialog_run (GTK_DIALOG (dialog));
-- >          gtk_widget_destroy (dialog);
-- >
-- >          if (response == GTK_RESPONSE_YES)
-- >            action = GDK_ACTION_MOVE;
-- >          else
-- >            action = GDK_ACTION_COPY;
-- >         }
-- >
-- >      gtk_drag_finish (context, TRUE, action == GDK_ACTION_MOVE, time);
-- >    }
-- >  else
-- >    gtk_drag_finish (context, FALSE, FALSE, time);
-- > }
type WidgetDragDataReceivedCallback =
    Gdk.DragContext.DragContext
    -- ^ /@context@/: the drag context
    -> Int32
    -- ^ /@x@/: where the drop happened
    -> Int32
    -- ^ /@y@/: where the drop happened
    -> Gtk.SelectionData.SelectionData
    -- ^ /@data@/: the received data
    -> Word32
    -- ^ /@info@/: the info that has been registered with the target in the
    --        t'GI.Gtk.Structs.TargetList.TargetList'
    -> Word32
    -- ^ /@time@/: the timestamp at which the data was received
    -> IO ()

type C_WidgetDragDataReceivedCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.DragContext.DragContext ->
    Int32 ->
    Int32 ->
    Ptr Gtk.SelectionData.SelectionData ->
    Word32 ->
    Word32 ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetDragDataReceivedCallback`.
foreign import ccall "wrapper"
    mk_WidgetDragDataReceivedCallback :: C_WidgetDragDataReceivedCallback -> IO (FunPtr C_WidgetDragDataReceivedCallback)

wrap_WidgetDragDataReceivedCallback :: 
    GObject a => (a -> WidgetDragDataReceivedCallback) ->
    C_WidgetDragDataReceivedCallback
wrap_WidgetDragDataReceivedCallback gi'cb gi'selfPtr context x y data_ info time _ = do
    context' <- (newObject Gdk.DragContext.DragContext) context
    B.ManagedPtr.withTransient  data_ $ \data_' -> do
        B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  context' x y data_' info time


-- | Connect a signal handler for the [dragDataReceived](#signal:dragDataReceived) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #dragDataReceived callback
-- @
-- 
-- 
onWidgetDragDataReceived :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragDataReceivedCallback) -> m SignalHandlerId
onWidgetDragDataReceived obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragDataReceivedCallback wrapped
    wrapped'' <- mk_WidgetDragDataReceivedCallback wrapped'
    connectSignalFunPtr obj "drag-data-received" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [dragDataReceived](#signal:dragDataReceived) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #dragDataReceived callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDragDataReceived :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragDataReceivedCallback) -> m SignalHandlerId
afterWidgetDragDataReceived obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragDataReceivedCallback wrapped
    wrapped'' <- mk_WidgetDragDataReceivedCallback wrapped'
    connectSignalFunPtr obj "drag-data-received" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDragDataReceivedSignalInfo
instance SignalInfo WidgetDragDataReceivedSignalInfo where
    type HaskellCallbackType WidgetDragDataReceivedSignalInfo = WidgetDragDataReceivedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDragDataReceivedCallback cb
        cb'' <- mk_WidgetDragDataReceivedCallback cb'
        connectSignalFunPtr obj "drag-data-received" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::drag-data-received"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:dragDataReceived"})

#endif

-- signal Widget::drag-drop
-- | The [dragDrop](#g:signal:dragDrop) signal is emitted on the drop site when the user drops
-- the data onto the widget. The signal handler must determine whether
-- the cursor position is in a drop zone or not. If it is not in a drop
-- zone, it returns 'P.False' and no further processing is necessary.
-- Otherwise, the handler returns 'P.True'. In this case, the handler must
-- ensure that 'GI.Gtk.Functions.dragFinish' is called to let the source know that
-- the drop is done. The call to 'GI.Gtk.Functions.dragFinish' can be done either
-- directly or in a [Widget::dragDataReceived]("GI.Gtk.Objects.Widget#g:signal:dragDataReceived") handler which gets
-- triggered by calling 'GI.Gtk.Objects.Widget.widgetDragGetData' to receive the data for one
-- or more of the supported targets.
type WidgetDragDropCallback =
    Gdk.DragContext.DragContext
    -- ^ /@context@/: the drag context
    -> Int32
    -- ^ /@x@/: the x coordinate of the current cursor position
    -> Int32
    -- ^ /@y@/: the y coordinate of the current cursor position
    -> Word32
    -- ^ /@time@/: the timestamp of the motion event
    -> IO Bool
    -- ^ __Returns:__ whether the cursor position is in a drop zone

type C_WidgetDragDropCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.DragContext.DragContext ->
    Int32 ->
    Int32 ->
    Word32 ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetDragDropCallback`.
foreign import ccall "wrapper"
    mk_WidgetDragDropCallback :: C_WidgetDragDropCallback -> IO (FunPtr C_WidgetDragDropCallback)

wrap_WidgetDragDropCallback :: 
    GObject a => (a -> WidgetDragDropCallback) ->
    C_WidgetDragDropCallback
wrap_WidgetDragDropCallback gi'cb gi'selfPtr context x y time _ = do
    context' <- (newObject Gdk.DragContext.DragContext) context
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  context' x y time
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [dragDrop](#signal:dragDrop) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #dragDrop callback
-- @
-- 
-- 
onWidgetDragDrop :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragDropCallback) -> m SignalHandlerId
onWidgetDragDrop obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragDropCallback wrapped
    wrapped'' <- mk_WidgetDragDropCallback wrapped'
    connectSignalFunPtr obj "drag-drop" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [dragDrop](#signal:dragDrop) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #dragDrop callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDragDrop :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragDropCallback) -> m SignalHandlerId
afterWidgetDragDrop obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragDropCallback wrapped
    wrapped'' <- mk_WidgetDragDropCallback wrapped'
    connectSignalFunPtr obj "drag-drop" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDragDropSignalInfo
instance SignalInfo WidgetDragDropSignalInfo where
    type HaskellCallbackType WidgetDragDropSignalInfo = WidgetDragDropCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDragDropCallback cb
        cb'' <- mk_WidgetDragDropCallback cb'
        connectSignalFunPtr obj "drag-drop" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::drag-drop"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:dragDrop"})

#endif

-- signal Widget::drag-end
-- | The [dragEnd](#g:signal:dragEnd) signal is emitted on the drag source when a drag is
-- finished.  A typical reason to connect to this signal is to undo
-- things done in [Widget::dragBegin]("GI.Gtk.Objects.Widget#g:signal:dragBegin").
type WidgetDragEndCallback =
    Gdk.DragContext.DragContext
    -- ^ /@context@/: the drag context
    -> IO ()

type C_WidgetDragEndCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.DragContext.DragContext ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetDragEndCallback`.
foreign import ccall "wrapper"
    mk_WidgetDragEndCallback :: C_WidgetDragEndCallback -> IO (FunPtr C_WidgetDragEndCallback)

wrap_WidgetDragEndCallback :: 
    GObject a => (a -> WidgetDragEndCallback) ->
    C_WidgetDragEndCallback
wrap_WidgetDragEndCallback gi'cb gi'selfPtr context _ = do
    context' <- (newObject Gdk.DragContext.DragContext) context
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  context'


-- | Connect a signal handler for the [dragEnd](#signal:dragEnd) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #dragEnd callback
-- @
-- 
-- 
onWidgetDragEnd :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragEndCallback) -> m SignalHandlerId
onWidgetDragEnd obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragEndCallback wrapped
    wrapped'' <- mk_WidgetDragEndCallback wrapped'
    connectSignalFunPtr obj "drag-end" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [dragEnd](#signal:dragEnd) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #dragEnd callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDragEnd :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragEndCallback) -> m SignalHandlerId
afterWidgetDragEnd obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragEndCallback wrapped
    wrapped'' <- mk_WidgetDragEndCallback wrapped'
    connectSignalFunPtr obj "drag-end" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDragEndSignalInfo
instance SignalInfo WidgetDragEndSignalInfo where
    type HaskellCallbackType WidgetDragEndSignalInfo = WidgetDragEndCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDragEndCallback cb
        cb'' <- mk_WidgetDragEndCallback cb'
        connectSignalFunPtr obj "drag-end" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::drag-end"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:dragEnd"})

#endif

-- signal Widget::drag-failed
-- | The [dragFailed](#g:signal:dragFailed) signal is emitted on the drag source when a drag has
-- failed. The signal handler may hook custom code to handle a failed DnD
-- operation based on the type of error, it returns 'P.True' is the failure has
-- been already handled (not showing the default \"drag operation failed\"
-- animation), otherwise it returns 'P.False'.
-- 
-- /Since: 2.12/
type WidgetDragFailedCallback =
    Gdk.DragContext.DragContext
    -- ^ /@context@/: the drag context
    -> Gtk.Enums.DragResult
    -- ^ /@result@/: the result of the drag operation
    -> IO Bool
    -- ^ __Returns:__ 'P.True' if the failed drag operation has been already handled.

type C_WidgetDragFailedCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.DragContext.DragContext ->
    CUInt ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetDragFailedCallback`.
foreign import ccall "wrapper"
    mk_WidgetDragFailedCallback :: C_WidgetDragFailedCallback -> IO (FunPtr C_WidgetDragFailedCallback)

wrap_WidgetDragFailedCallback :: 
    GObject a => (a -> WidgetDragFailedCallback) ->
    C_WidgetDragFailedCallback
wrap_WidgetDragFailedCallback gi'cb gi'selfPtr context result_ _ = do
    context' <- (newObject Gdk.DragContext.DragContext) context
    let result_' = (toEnum . fromIntegral) result_
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  context' result_'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [dragFailed](#signal:dragFailed) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #dragFailed callback
-- @
-- 
-- 
onWidgetDragFailed :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragFailedCallback) -> m SignalHandlerId
onWidgetDragFailed obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragFailedCallback wrapped
    wrapped'' <- mk_WidgetDragFailedCallback wrapped'
    connectSignalFunPtr obj "drag-failed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [dragFailed](#signal:dragFailed) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #dragFailed callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDragFailed :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragFailedCallback) -> m SignalHandlerId
afterWidgetDragFailed obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragFailedCallback wrapped
    wrapped'' <- mk_WidgetDragFailedCallback wrapped'
    connectSignalFunPtr obj "drag-failed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDragFailedSignalInfo
instance SignalInfo WidgetDragFailedSignalInfo where
    type HaskellCallbackType WidgetDragFailedSignalInfo = WidgetDragFailedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDragFailedCallback cb
        cb'' <- mk_WidgetDragFailedCallback cb'
        connectSignalFunPtr obj "drag-failed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::drag-failed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:dragFailed"})

#endif

-- signal Widget::drag-leave
-- | The [dragLeave](#g:signal:dragLeave) signal is emitted on the drop site when the cursor
-- leaves the widget. A typical reason to connect to this signal is to
-- undo things done in [Widget::dragMotion]("GI.Gtk.Objects.Widget#g:signal:dragMotion"), e.g. undo highlighting
-- with 'GI.Gtk.Objects.Widget.widgetDragUnhighlight'.
-- 
-- 
-- Likewise, the [Widget::dragLeave]("GI.Gtk.Objects.Widget#g:signal:dragLeave") signal is also emitted before the
-- [dragDrop](#g:signal:dragDrop) signal, for instance to allow cleaning up of a preview item
-- created in the [Widget::dragMotion]("GI.Gtk.Objects.Widget#g:signal:dragMotion") signal handler.
type WidgetDragLeaveCallback =
    Gdk.DragContext.DragContext
    -- ^ /@context@/: the drag context
    -> Word32
    -- ^ /@time@/: the timestamp of the motion event
    -> IO ()

type C_WidgetDragLeaveCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.DragContext.DragContext ->
    Word32 ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetDragLeaveCallback`.
foreign import ccall "wrapper"
    mk_WidgetDragLeaveCallback :: C_WidgetDragLeaveCallback -> IO (FunPtr C_WidgetDragLeaveCallback)

wrap_WidgetDragLeaveCallback :: 
    GObject a => (a -> WidgetDragLeaveCallback) ->
    C_WidgetDragLeaveCallback
wrap_WidgetDragLeaveCallback gi'cb gi'selfPtr context time _ = do
    context' <- (newObject Gdk.DragContext.DragContext) context
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  context' time


-- | Connect a signal handler for the [dragLeave](#signal:dragLeave) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #dragLeave callback
-- @
-- 
-- 
onWidgetDragLeave :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragLeaveCallback) -> m SignalHandlerId
onWidgetDragLeave obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragLeaveCallback wrapped
    wrapped'' <- mk_WidgetDragLeaveCallback wrapped'
    connectSignalFunPtr obj "drag-leave" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [dragLeave](#signal:dragLeave) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #dragLeave callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDragLeave :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragLeaveCallback) -> m SignalHandlerId
afterWidgetDragLeave obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragLeaveCallback wrapped
    wrapped'' <- mk_WidgetDragLeaveCallback wrapped'
    connectSignalFunPtr obj "drag-leave" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDragLeaveSignalInfo
instance SignalInfo WidgetDragLeaveSignalInfo where
    type HaskellCallbackType WidgetDragLeaveSignalInfo = WidgetDragLeaveCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDragLeaveCallback cb
        cb'' <- mk_WidgetDragLeaveCallback cb'
        connectSignalFunPtr obj "drag-leave" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::drag-leave"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:dragLeave"})

#endif

-- signal Widget::drag-motion
-- | The [dragMotion](#g:signal:dragMotion) signal is emitted on the drop site when the user
-- moves the cursor over the widget during a drag. The signal handler
-- must determine whether the cursor position is in a drop zone or not.
-- If it is not in a drop zone, it returns 'P.False' and no further processing
-- is necessary. Otherwise, the handler returns 'P.True'. In this case, the
-- handler is responsible for providing the necessary information for
-- displaying feedback to the user, by calling 'GI.Gdk.Functions.dragStatus'.
-- 
-- If the decision whether the drop will be accepted or rejected can\'t be
-- made based solely on the cursor position and the type of the data, the
-- handler may inspect the dragged data by calling 'GI.Gtk.Objects.Widget.widgetDragGetData' and
-- defer the 'GI.Gdk.Functions.dragStatus' call to the [Widget::dragDataReceived]("GI.Gtk.Objects.Widget#g:signal:dragDataReceived")
-- handler. Note that you must pass @/GTK_DEST_DEFAULT_DROP/@,
-- @/GTK_DEST_DEFAULT_MOTION/@ or @/GTK_DEST_DEFAULT_ALL/@ to 'GI.Gtk.Objects.Widget.widgetDragDestSet'
-- when using the drag-motion signal that way.
-- 
-- Also note that there is no drag-enter signal. The drag receiver has to
-- keep track of whether he has received any drag-motion signals since the
-- last [Widget::dragLeave]("GI.Gtk.Objects.Widget#g:signal:dragLeave") and if not, treat the drag-motion signal as
-- an \"enter\" signal. Upon an \"enter\", the handler will typically highlight
-- the drop site with 'GI.Gtk.Objects.Widget.widgetDragHighlight'.
-- 
-- === /C code/
-- >
-- >static void
-- >drag_motion (GtkWidget      *widget,
-- >             GdkDragContext *context,
-- >             gint            x,
-- >             gint            y,
-- >             guint           time)
-- >{
-- >  GdkAtom target;
-- >
-- >  PrivateData *private_data = GET_PRIVATE_DATA (widget);
-- >
-- >  if (!private_data->drag_highlight)
-- >   {
-- >     private_data->drag_highlight = 1;
-- >     gtk_drag_highlight (widget);
-- >   }
-- >
-- >  target = gtk_drag_dest_find_target (widget, context, NULL);
-- >  if (target == GDK_NONE)
-- >    gdk_drag_status (context, 0, time);
-- >  else
-- >   {
-- >     private_data->pending_status
-- >        = gdk_drag_context_get_suggested_action (context);
-- >     gtk_drag_get_data (widget, context, target, time);
-- >   }
-- >
-- >  return TRUE;
-- >}
-- >
-- >static void
-- >drag_data_received (GtkWidget        *widget,
-- >                    GdkDragContext   *context,
-- >                    gint              x,
-- >                    gint              y,
-- >                    GtkSelectionData *selection_data,
-- >                    guint             info,
-- >                    guint             time)
-- >{
-- >  PrivateData *private_data = GET_PRIVATE_DATA (widget);
-- >
-- >  if (private_data->suggested_action)
-- >   {
-- >     private_data->suggested_action = 0;
-- >
-- >     // We are getting this data due to a request in drag_motion,
-- >     // rather than due to a request in drag_drop, so we are just
-- >     // supposed to call gdk_drag_status(), not actually paste in
-- >     // the data.
-- >
-- >     str = gtk_selection_data_get_text (selection_data);
-- >     if (!data_is_acceptable (str))
-- >       gdk_drag_status (context, 0, time);
-- >     else
-- >       gdk_drag_status (context,
-- >                        private_data->suggested_action,
-- >                        time);
-- >   }
-- >  else
-- >   {
-- >     // accept the drop
-- >   }
-- >}
type WidgetDragMotionCallback =
    Gdk.DragContext.DragContext
    -- ^ /@context@/: the drag context
    -> Int32
    -- ^ /@x@/: the x coordinate of the current cursor position
    -> Int32
    -- ^ /@y@/: the y coordinate of the current cursor position
    -> Word32
    -- ^ /@time@/: the timestamp of the motion event
    -> IO Bool
    -- ^ __Returns:__ whether the cursor position is in a drop zone

type C_WidgetDragMotionCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.DragContext.DragContext ->
    Int32 ->
    Int32 ->
    Word32 ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetDragMotionCallback`.
foreign import ccall "wrapper"
    mk_WidgetDragMotionCallback :: C_WidgetDragMotionCallback -> IO (FunPtr C_WidgetDragMotionCallback)

wrap_WidgetDragMotionCallback :: 
    GObject a => (a -> WidgetDragMotionCallback) ->
    C_WidgetDragMotionCallback
wrap_WidgetDragMotionCallback gi'cb gi'selfPtr context x y time _ = do
    context' <- (newObject Gdk.DragContext.DragContext) context
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  context' x y time
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [dragMotion](#signal:dragMotion) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #dragMotion callback
-- @
-- 
-- 
onWidgetDragMotion :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragMotionCallback) -> m SignalHandlerId
onWidgetDragMotion obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragMotionCallback wrapped
    wrapped'' <- mk_WidgetDragMotionCallback wrapped'
    connectSignalFunPtr obj "drag-motion" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [dragMotion](#signal:dragMotion) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #dragMotion callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDragMotion :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDragMotionCallback) -> m SignalHandlerId
afterWidgetDragMotion obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDragMotionCallback wrapped
    wrapped'' <- mk_WidgetDragMotionCallback wrapped'
    connectSignalFunPtr obj "drag-motion" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDragMotionSignalInfo
instance SignalInfo WidgetDragMotionSignalInfo where
    type HaskellCallbackType WidgetDragMotionSignalInfo = WidgetDragMotionCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDragMotionCallback cb
        cb'' <- mk_WidgetDragMotionCallback cb'
        connectSignalFunPtr obj "drag-motion" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::drag-motion"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:dragMotion"})

#endif

-- signal Widget::draw
-- | This signal is emitted when a widget is supposed to render itself.
-- The /@widget@/\'s top left corner must be painted at the origin of
-- the passed in context and be sized to the values returned by
-- 'GI.Gtk.Objects.Widget.widgetGetAllocatedWidth' and
-- 'GI.Gtk.Objects.Widget.widgetGetAllocatedHeight'.
-- 
-- Signal handlers connected to this signal can modify the cairo
-- context passed as /@cr@/ in any way they like and don\'t need to
-- restore it. The signal emission takes care of calling @/cairo_save()/@
-- before and @/cairo_restore()/@ after invoking the handler.
-- 
-- The signal handler will get a /@cr@/ with a clip region already set to the
-- widget\'s dirty region, i.e. to the area that needs repainting.  Complicated
-- widgets that want to avoid redrawing themselves completely can get the full
-- extents of the clip region with 'GI.Gdk.Functions.cairoGetClipRectangle', or they can
-- get a finer-grained representation of the dirty region with
-- @/cairo_copy_clip_rectangle_list()/@.
-- 
-- /Since: 3.0/
type WidgetDrawCallback =
    Cairo.Context.Context
    -- ^ /@cr@/: the cairo context to draw to
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    -- 'P.False' to propagate the event further.

type C_WidgetDrawCallback =
    Ptr Widget ->                           -- object
    Ptr Cairo.Context.Context ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetDrawCallback`.
foreign import ccall "wrapper"
    mk_WidgetDrawCallback :: C_WidgetDrawCallback -> IO (FunPtr C_WidgetDrawCallback)

wrap_WidgetDrawCallback :: 
    GObject a => (a -> WidgetDrawCallback) ->
    C_WidgetDrawCallback
wrap_WidgetDrawCallback gi'cb gi'selfPtr cr _ = do
    B.ManagedPtr.withTransient  cr $ \cr' -> do
        result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  cr'
        let result' = (P.fromIntegral . P.fromEnum) result
        return result'


-- | Connect a signal handler for the [draw](#signal:draw) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #draw callback
-- @
-- 
-- 
onWidgetDraw :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDrawCallback) -> m SignalHandlerId
onWidgetDraw obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDrawCallback wrapped
    wrapped'' <- mk_WidgetDrawCallback wrapped'
    connectSignalFunPtr obj "draw" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [draw](#signal:draw) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #draw callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetDraw :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetDrawCallback) -> m SignalHandlerId
afterWidgetDraw obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetDrawCallback wrapped
    wrapped'' <- mk_WidgetDrawCallback wrapped'
    connectSignalFunPtr obj "draw" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetDrawSignalInfo
instance SignalInfo WidgetDrawSignalInfo where
    type HaskellCallbackType WidgetDrawSignalInfo = WidgetDrawCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetDrawCallback cb
        cb'' <- mk_WidgetDrawCallback cb'
        connectSignalFunPtr obj "draw" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::draw"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:draw"})

#endif

-- signal Widget::enter-notify-event
-- | The [enterNotifyEvent](#g:signal:enterNotifyEvent) will be emitted when the pointer enters
-- the /@widget@/\'s window.
-- 
-- To receive this signal, the t'GI.Gdk.Objects.Window.Window' associated to the widget needs
-- to enable the @/GDK_ENTER_NOTIFY_MASK/@ mask.
-- 
-- This signal will be sent to the grab widget if there is one.
type WidgetEnterNotifyEventCallback =
    Gdk.EventCrossing.EventCrossing
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventCrossing.EventCrossing' which triggered
    --   this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetEnterNotifyEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventCrossing.EventCrossing ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetEnterNotifyEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetEnterNotifyEventCallback :: C_WidgetEnterNotifyEventCallback -> IO (FunPtr C_WidgetEnterNotifyEventCallback)

wrap_WidgetEnterNotifyEventCallback :: 
    GObject a => (a -> WidgetEnterNotifyEventCallback) ->
    C_WidgetEnterNotifyEventCallback
wrap_WidgetEnterNotifyEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventCrossing.EventCrossing) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [enterNotifyEvent](#signal:enterNotifyEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #enterNotifyEvent callback
-- @
-- 
-- 
onWidgetEnterNotifyEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetEnterNotifyEventCallback) -> m SignalHandlerId
onWidgetEnterNotifyEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetEnterNotifyEventCallback wrapped
    wrapped'' <- mk_WidgetEnterNotifyEventCallback wrapped'
    connectSignalFunPtr obj "enter-notify-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [enterNotifyEvent](#signal:enterNotifyEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #enterNotifyEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetEnterNotifyEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetEnterNotifyEventCallback) -> m SignalHandlerId
afterWidgetEnterNotifyEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetEnterNotifyEventCallback wrapped
    wrapped'' <- mk_WidgetEnterNotifyEventCallback wrapped'
    connectSignalFunPtr obj "enter-notify-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetEnterNotifyEventSignalInfo
instance SignalInfo WidgetEnterNotifyEventSignalInfo where
    type HaskellCallbackType WidgetEnterNotifyEventSignalInfo = WidgetEnterNotifyEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetEnterNotifyEventCallback cb
        cb'' <- mk_WidgetEnterNotifyEventCallback cb'
        connectSignalFunPtr obj "enter-notify-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::enter-notify-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:enterNotifyEvent"})

#endif

-- signal Widget::event
-- | The GTK+ main loop will emit three signals for each GDK event delivered
-- to a widget: one generic [event](#g:signal:event) signal, another, more specific,
-- signal that matches the type of event delivered (e.g.
-- [Widget::keyPressEvent]("GI.Gtk.Objects.Widget#g:signal:keyPressEvent")) and finally a generic
-- [Widget::eventAfter]("GI.Gtk.Objects.Widget#g:signal:eventAfter") signal.
type WidgetEventCallback =
    Gdk.Event.Event
    -- ^ /@event@/: the t'GI.Gdk.Unions.Event.Event' which triggered this signal
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event
    -- and to cancel the emission of the second specific [event](#g:signal:event) signal.
    --   'P.False' to propagate the event further and to allow the emission of
    --   the second signal. The [eventAfter](#g:signal:eventAfter) signal is emitted regardless of
    --   the return value.

type C_WidgetEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.Event.Event ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetEventCallback :: C_WidgetEventCallback -> IO (FunPtr C_WidgetEventCallback)

wrap_WidgetEventCallback :: 
    GObject a => (a -> WidgetEventCallback) ->
    C_WidgetEventCallback
wrap_WidgetEventCallback gi'cb gi'selfPtr event _ = do
    B.ManagedPtr.withTransient  event $ \event' -> do
        result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
        let result' = (P.fromIntegral . P.fromEnum) result
        return result'


-- | Connect a signal handler for the [event](#signal:event) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #event callback
-- @
-- 
-- 
onWidgetEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetEventCallback) -> m SignalHandlerId
onWidgetEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetEventCallback wrapped
    wrapped'' <- mk_WidgetEventCallback wrapped'
    connectSignalFunPtr obj "event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [event](#signal:event) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #event callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetEventCallback) -> m SignalHandlerId
afterWidgetEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetEventCallback wrapped
    wrapped'' <- mk_WidgetEventCallback wrapped'
    connectSignalFunPtr obj "event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetEventSignalInfo
instance SignalInfo WidgetEventSignalInfo where
    type HaskellCallbackType WidgetEventSignalInfo = WidgetEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetEventCallback cb
        cb'' <- mk_WidgetEventCallback cb'
        connectSignalFunPtr obj "event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:event"})

#endif

-- signal Widget::event-after
-- | After the emission of the [Widget::event]("GI.Gtk.Objects.Widget#g:signal:event") signal and (optionally)
-- the second more specific signal, [eventAfter](#g:signal:eventAfter) will be emitted
-- regardless of the previous two signals handlers return values.
type WidgetEventAfterCallback =
    Gdk.Event.Event
    -- ^ /@event@/: the t'GI.Gdk.Unions.Event.Event' which triggered this signal
    -> IO ()

type C_WidgetEventAfterCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.Event.Event ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetEventAfterCallback`.
foreign import ccall "wrapper"
    mk_WidgetEventAfterCallback :: C_WidgetEventAfterCallback -> IO (FunPtr C_WidgetEventAfterCallback)

wrap_WidgetEventAfterCallback :: 
    GObject a => (a -> WidgetEventAfterCallback) ->
    C_WidgetEventAfterCallback
wrap_WidgetEventAfterCallback gi'cb gi'selfPtr event _ = do
    B.ManagedPtr.withTransient  event $ \event' -> do
        B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'


-- | Connect a signal handler for the [eventAfter](#signal:eventAfter) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #eventAfter callback
-- @
-- 
-- 
onWidgetEventAfter :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetEventAfterCallback) -> m SignalHandlerId
onWidgetEventAfter obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetEventAfterCallback wrapped
    wrapped'' <- mk_WidgetEventAfterCallback wrapped'
    connectSignalFunPtr obj "event-after" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [eventAfter](#signal:eventAfter) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #eventAfter callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetEventAfter :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetEventAfterCallback) -> m SignalHandlerId
afterWidgetEventAfter obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetEventAfterCallback wrapped
    wrapped'' <- mk_WidgetEventAfterCallback wrapped'
    connectSignalFunPtr obj "event-after" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetEventAfterSignalInfo
instance SignalInfo WidgetEventAfterSignalInfo where
    type HaskellCallbackType WidgetEventAfterSignalInfo = WidgetEventAfterCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetEventAfterCallback cb
        cb'' <- mk_WidgetEventAfterCallback cb'
        connectSignalFunPtr obj "event-after" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::event-after"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:eventAfter"})

#endif

-- signal Widget::focus
-- | /No description available in the introspection data./
type WidgetFocusCallback =
    Gtk.Enums.DirectionType
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event. 'P.False' to propagate the event further.

type C_WidgetFocusCallback =
    Ptr Widget ->                           -- object
    CUInt ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetFocusCallback`.
foreign import ccall "wrapper"
    mk_WidgetFocusCallback :: C_WidgetFocusCallback -> IO (FunPtr C_WidgetFocusCallback)

wrap_WidgetFocusCallback :: 
    GObject a => (a -> WidgetFocusCallback) ->
    C_WidgetFocusCallback
wrap_WidgetFocusCallback gi'cb gi'selfPtr direction _ = do
    let direction' = (toEnum . fromIntegral) direction
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  direction'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [focus](#signal:focus) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #focus callback
-- @
-- 
-- 
onWidgetFocus :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetFocusCallback) -> m SignalHandlerId
onWidgetFocus obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetFocusCallback wrapped
    wrapped'' <- mk_WidgetFocusCallback wrapped'
    connectSignalFunPtr obj "focus" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [focus](#signal:focus) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #focus callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetFocus :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetFocusCallback) -> m SignalHandlerId
afterWidgetFocus obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetFocusCallback wrapped
    wrapped'' <- mk_WidgetFocusCallback wrapped'
    connectSignalFunPtr obj "focus" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetFocusSignalInfo
instance SignalInfo WidgetFocusSignalInfo where
    type HaskellCallbackType WidgetFocusSignalInfo = WidgetFocusCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetFocusCallback cb
        cb'' <- mk_WidgetFocusCallback cb'
        connectSignalFunPtr obj "focus" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::focus"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:focus"})

#endif

-- signal Widget::focus-in-event
-- | The [focusInEvent](#g:signal:focusInEvent) signal will be emitted when the keyboard focus
-- enters the /@widget@/\'s window.
-- 
-- To receive this signal, the t'GI.Gdk.Objects.Window.Window' associated to the widget needs
-- to enable the @/GDK_FOCUS_CHANGE_MASK/@ mask.
type WidgetFocusInEventCallback =
    Gdk.EventFocus.EventFocus
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventFocus.EventFocus' which triggered
    --   this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetFocusInEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventFocus.EventFocus ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetFocusInEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetFocusInEventCallback :: C_WidgetFocusInEventCallback -> IO (FunPtr C_WidgetFocusInEventCallback)

wrap_WidgetFocusInEventCallback :: 
    GObject a => (a -> WidgetFocusInEventCallback) ->
    C_WidgetFocusInEventCallback
wrap_WidgetFocusInEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventFocus.EventFocus) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [focusInEvent](#signal:focusInEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #focusInEvent callback
-- @
-- 
-- 
onWidgetFocusInEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetFocusInEventCallback) -> m SignalHandlerId
onWidgetFocusInEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetFocusInEventCallback wrapped
    wrapped'' <- mk_WidgetFocusInEventCallback wrapped'
    connectSignalFunPtr obj "focus-in-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [focusInEvent](#signal:focusInEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #focusInEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetFocusInEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetFocusInEventCallback) -> m SignalHandlerId
afterWidgetFocusInEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetFocusInEventCallback wrapped
    wrapped'' <- mk_WidgetFocusInEventCallback wrapped'
    connectSignalFunPtr obj "focus-in-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetFocusInEventSignalInfo
instance SignalInfo WidgetFocusInEventSignalInfo where
    type HaskellCallbackType WidgetFocusInEventSignalInfo = WidgetFocusInEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetFocusInEventCallback cb
        cb'' <- mk_WidgetFocusInEventCallback cb'
        connectSignalFunPtr obj "focus-in-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::focus-in-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:focusInEvent"})

#endif

-- signal Widget::focus-out-event
-- | The [focusOutEvent](#g:signal:focusOutEvent) signal will be emitted when the keyboard focus
-- leaves the /@widget@/\'s window.
-- 
-- To receive this signal, the t'GI.Gdk.Objects.Window.Window' associated to the widget needs
-- to enable the @/GDK_FOCUS_CHANGE_MASK/@ mask.
type WidgetFocusOutEventCallback =
    Gdk.EventFocus.EventFocus
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventFocus.EventFocus' which triggered this
    --   signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetFocusOutEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventFocus.EventFocus ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetFocusOutEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetFocusOutEventCallback :: C_WidgetFocusOutEventCallback -> IO (FunPtr C_WidgetFocusOutEventCallback)

wrap_WidgetFocusOutEventCallback :: 
    GObject a => (a -> WidgetFocusOutEventCallback) ->
    C_WidgetFocusOutEventCallback
wrap_WidgetFocusOutEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventFocus.EventFocus) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [focusOutEvent](#signal:focusOutEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #focusOutEvent callback
-- @
-- 
-- 
onWidgetFocusOutEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetFocusOutEventCallback) -> m SignalHandlerId
onWidgetFocusOutEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetFocusOutEventCallback wrapped
    wrapped'' <- mk_WidgetFocusOutEventCallback wrapped'
    connectSignalFunPtr obj "focus-out-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [focusOutEvent](#signal:focusOutEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #focusOutEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetFocusOutEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetFocusOutEventCallback) -> m SignalHandlerId
afterWidgetFocusOutEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetFocusOutEventCallback wrapped
    wrapped'' <- mk_WidgetFocusOutEventCallback wrapped'
    connectSignalFunPtr obj "focus-out-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetFocusOutEventSignalInfo
instance SignalInfo WidgetFocusOutEventSignalInfo where
    type HaskellCallbackType WidgetFocusOutEventSignalInfo = WidgetFocusOutEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetFocusOutEventCallback cb
        cb'' <- mk_WidgetFocusOutEventCallback cb'
        connectSignalFunPtr obj "focus-out-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::focus-out-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:focusOutEvent"})

#endif

-- signal Widget::grab-broken-event
-- | Emitted when a pointer or keyboard grab on a window belonging
-- to /@widget@/ gets broken.
-- 
-- On X11, this happens when the grab window becomes unviewable
-- (i.e. it or one of its ancestors is unmapped), or if the same
-- application grabs the pointer or keyboard again.
-- 
-- /Since: 2.8/
type WidgetGrabBrokenEventCallback =
    Gdk.EventGrabBroken.EventGrabBroken
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventGrabBroken.EventGrabBroken' event
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for
    --   the event. 'P.False' to propagate the event further.

type C_WidgetGrabBrokenEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventGrabBroken.EventGrabBroken ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetGrabBrokenEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetGrabBrokenEventCallback :: C_WidgetGrabBrokenEventCallback -> IO (FunPtr C_WidgetGrabBrokenEventCallback)

wrap_WidgetGrabBrokenEventCallback :: 
    GObject a => (a -> WidgetGrabBrokenEventCallback) ->
    C_WidgetGrabBrokenEventCallback
wrap_WidgetGrabBrokenEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventGrabBroken.EventGrabBroken) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [grabBrokenEvent](#signal:grabBrokenEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #grabBrokenEvent callback
-- @
-- 
-- 
onWidgetGrabBrokenEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetGrabBrokenEventCallback) -> m SignalHandlerId
onWidgetGrabBrokenEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetGrabBrokenEventCallback wrapped
    wrapped'' <- mk_WidgetGrabBrokenEventCallback wrapped'
    connectSignalFunPtr obj "grab-broken-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [grabBrokenEvent](#signal:grabBrokenEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #grabBrokenEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetGrabBrokenEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetGrabBrokenEventCallback) -> m SignalHandlerId
afterWidgetGrabBrokenEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetGrabBrokenEventCallback wrapped
    wrapped'' <- mk_WidgetGrabBrokenEventCallback wrapped'
    connectSignalFunPtr obj "grab-broken-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetGrabBrokenEventSignalInfo
instance SignalInfo WidgetGrabBrokenEventSignalInfo where
    type HaskellCallbackType WidgetGrabBrokenEventSignalInfo = WidgetGrabBrokenEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetGrabBrokenEventCallback cb
        cb'' <- mk_WidgetGrabBrokenEventCallback cb'
        connectSignalFunPtr obj "grab-broken-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::grab-broken-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:grabBrokenEvent"})

#endif

-- signal Widget::grab-focus
-- | /No description available in the introspection data./
type WidgetGrabFocusCallback =
    IO ()

type C_WidgetGrabFocusCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetGrabFocusCallback`.
foreign import ccall "wrapper"
    mk_WidgetGrabFocusCallback :: C_WidgetGrabFocusCallback -> IO (FunPtr C_WidgetGrabFocusCallback)

wrap_WidgetGrabFocusCallback :: 
    GObject a => (a -> WidgetGrabFocusCallback) ->
    C_WidgetGrabFocusCallback
wrap_WidgetGrabFocusCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [grabFocus](#signal:grabFocus) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #grabFocus callback
-- @
-- 
-- 
onWidgetGrabFocus :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetGrabFocusCallback) -> m SignalHandlerId
onWidgetGrabFocus obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetGrabFocusCallback wrapped
    wrapped'' <- mk_WidgetGrabFocusCallback wrapped'
    connectSignalFunPtr obj "grab-focus" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [grabFocus](#signal:grabFocus) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #grabFocus callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetGrabFocus :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetGrabFocusCallback) -> m SignalHandlerId
afterWidgetGrabFocus obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetGrabFocusCallback wrapped
    wrapped'' <- mk_WidgetGrabFocusCallback wrapped'
    connectSignalFunPtr obj "grab-focus" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetGrabFocusSignalInfo
instance SignalInfo WidgetGrabFocusSignalInfo where
    type HaskellCallbackType WidgetGrabFocusSignalInfo = WidgetGrabFocusCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetGrabFocusCallback cb
        cb'' <- mk_WidgetGrabFocusCallback cb'
        connectSignalFunPtr obj "grab-focus" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::grab-focus"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:grabFocus"})

#endif

-- signal Widget::grab-notify
-- | The [grabNotify](#g:signal:grabNotify) signal is emitted when a widget becomes
-- shadowed by a GTK+ grab (not a pointer or keyboard grab) on
-- another widget, or when it becomes unshadowed due to a grab
-- being removed.
-- 
-- A widget is shadowed by a 'GI.Gtk.Objects.Widget.widgetGrabAdd' when the topmost
-- grab widget in the grab stack of its window group is not
-- its ancestor.
type WidgetGrabNotifyCallback =
    Bool
    -- ^ /@wasGrabbed@/: 'P.False' if the widget becomes shadowed, 'P.True'
    --               if it becomes unshadowed
    -> IO ()

type C_WidgetGrabNotifyCallback =
    Ptr Widget ->                           -- object
    CInt ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetGrabNotifyCallback`.
foreign import ccall "wrapper"
    mk_WidgetGrabNotifyCallback :: C_WidgetGrabNotifyCallback -> IO (FunPtr C_WidgetGrabNotifyCallback)

wrap_WidgetGrabNotifyCallback :: 
    GObject a => (a -> WidgetGrabNotifyCallback) ->
    C_WidgetGrabNotifyCallback
wrap_WidgetGrabNotifyCallback gi'cb gi'selfPtr wasGrabbed _ = do
    let wasGrabbed' = (/= 0) wasGrabbed
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  wasGrabbed'


-- | Connect a signal handler for the [grabNotify](#signal:grabNotify) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #grabNotify callback
-- @
-- 
-- 
onWidgetGrabNotify :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetGrabNotifyCallback) -> m SignalHandlerId
onWidgetGrabNotify obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetGrabNotifyCallback wrapped
    wrapped'' <- mk_WidgetGrabNotifyCallback wrapped'
    connectSignalFunPtr obj "grab-notify" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [grabNotify](#signal:grabNotify) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #grabNotify callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetGrabNotify :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetGrabNotifyCallback) -> m SignalHandlerId
afterWidgetGrabNotify obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetGrabNotifyCallback wrapped
    wrapped'' <- mk_WidgetGrabNotifyCallback wrapped'
    connectSignalFunPtr obj "grab-notify" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetGrabNotifySignalInfo
instance SignalInfo WidgetGrabNotifySignalInfo where
    type HaskellCallbackType WidgetGrabNotifySignalInfo = WidgetGrabNotifyCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetGrabNotifyCallback cb
        cb'' <- mk_WidgetGrabNotifyCallback cb'
        connectSignalFunPtr obj "grab-notify" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::grab-notify"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:grabNotify"})

#endif

-- signal Widget::hide
-- | The [hide](#g:signal:hide) signal is emitted when /@widget@/ is hidden, for example with
-- 'GI.Gtk.Objects.Widget.widgetHide'.
type WidgetHideCallback =
    IO ()

type C_WidgetHideCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetHideCallback`.
foreign import ccall "wrapper"
    mk_WidgetHideCallback :: C_WidgetHideCallback -> IO (FunPtr C_WidgetHideCallback)

wrap_WidgetHideCallback :: 
    GObject a => (a -> WidgetHideCallback) ->
    C_WidgetHideCallback
wrap_WidgetHideCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [hide](#signal:hide) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #hide callback
-- @
-- 
-- 
onWidgetHide :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetHideCallback) -> m SignalHandlerId
onWidgetHide obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetHideCallback wrapped
    wrapped'' <- mk_WidgetHideCallback wrapped'
    connectSignalFunPtr obj "hide" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [hide](#signal:hide) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #hide callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetHide :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetHideCallback) -> m SignalHandlerId
afterWidgetHide obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetHideCallback wrapped
    wrapped'' <- mk_WidgetHideCallback wrapped'
    connectSignalFunPtr obj "hide" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetHideSignalInfo
instance SignalInfo WidgetHideSignalInfo where
    type HaskellCallbackType WidgetHideSignalInfo = WidgetHideCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetHideCallback cb
        cb'' <- mk_WidgetHideCallback cb'
        connectSignalFunPtr obj "hide" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::hide"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:hide"})

#endif

-- signal Widget::hierarchy-changed
-- | The [hierarchyChanged](#g:signal:hierarchyChanged) signal is emitted when the
-- anchored state of a widget changes. A widget is
-- “anchored” when its toplevel
-- ancestor is a t'GI.Gtk.Objects.Window.Window'. This signal is emitted when
-- a widget changes from un-anchored to anchored or vice-versa.
type WidgetHierarchyChangedCallback =
    Maybe Widget
    -- ^ /@previousToplevel@/: the previous toplevel ancestor, or 'P.Nothing'
    --   if the widget was previously unanchored
    -> IO ()

type C_WidgetHierarchyChangedCallback =
    Ptr Widget ->                           -- object
    Ptr Widget ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetHierarchyChangedCallback`.
foreign import ccall "wrapper"
    mk_WidgetHierarchyChangedCallback :: C_WidgetHierarchyChangedCallback -> IO (FunPtr C_WidgetHierarchyChangedCallback)

wrap_WidgetHierarchyChangedCallback :: 
    GObject a => (a -> WidgetHierarchyChangedCallback) ->
    C_WidgetHierarchyChangedCallback
wrap_WidgetHierarchyChangedCallback gi'cb gi'selfPtr previousToplevel _ = do
    maybePreviousToplevel <-
        if previousToplevel == FP.nullPtr
        then return Nothing
        else do
            previousToplevel' <- (newObject Widget) previousToplevel
            return $ Just previousToplevel'
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  maybePreviousToplevel


-- | Connect a signal handler for the [hierarchyChanged](#signal:hierarchyChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #hierarchyChanged callback
-- @
-- 
-- 
onWidgetHierarchyChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetHierarchyChangedCallback) -> m SignalHandlerId
onWidgetHierarchyChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetHierarchyChangedCallback wrapped
    wrapped'' <- mk_WidgetHierarchyChangedCallback wrapped'
    connectSignalFunPtr obj "hierarchy-changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [hierarchyChanged](#signal:hierarchyChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #hierarchyChanged callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetHierarchyChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetHierarchyChangedCallback) -> m SignalHandlerId
afterWidgetHierarchyChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetHierarchyChangedCallback wrapped
    wrapped'' <- mk_WidgetHierarchyChangedCallback wrapped'
    connectSignalFunPtr obj "hierarchy-changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetHierarchyChangedSignalInfo
instance SignalInfo WidgetHierarchyChangedSignalInfo where
    type HaskellCallbackType WidgetHierarchyChangedSignalInfo = WidgetHierarchyChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetHierarchyChangedCallback cb
        cb'' <- mk_WidgetHierarchyChangedCallback cb'
        connectSignalFunPtr obj "hierarchy-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::hierarchy-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:hierarchyChanged"})

#endif

-- signal Widget::key-press-event
-- | The [keyPressEvent](#g:signal:keyPressEvent) signal is emitted when a key is pressed. The signal
-- emission will reoccur at the key-repeat rate when the key is kept pressed.
-- 
-- To receive this signal, the t'GI.Gdk.Objects.Window.Window' associated to the widget needs
-- to enable the @/GDK_KEY_PRESS_MASK/@ mask.
-- 
-- This signal will be sent to the grab widget if there is one.
type WidgetKeyPressEventCallback =
    Gdk.EventKey.EventKey
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventKey.EventKey' which triggered this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetKeyPressEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventKey.EventKey ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetKeyPressEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetKeyPressEventCallback :: C_WidgetKeyPressEventCallback -> IO (FunPtr C_WidgetKeyPressEventCallback)

wrap_WidgetKeyPressEventCallback :: 
    GObject a => (a -> WidgetKeyPressEventCallback) ->
    C_WidgetKeyPressEventCallback
wrap_WidgetKeyPressEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventKey.EventKey) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [keyPressEvent](#signal:keyPressEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #keyPressEvent callback
-- @
-- 
-- 
onWidgetKeyPressEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetKeyPressEventCallback) -> m SignalHandlerId
onWidgetKeyPressEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetKeyPressEventCallback wrapped
    wrapped'' <- mk_WidgetKeyPressEventCallback wrapped'
    connectSignalFunPtr obj "key-press-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [keyPressEvent](#signal:keyPressEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #keyPressEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetKeyPressEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetKeyPressEventCallback) -> m SignalHandlerId
afterWidgetKeyPressEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetKeyPressEventCallback wrapped
    wrapped'' <- mk_WidgetKeyPressEventCallback wrapped'
    connectSignalFunPtr obj "key-press-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetKeyPressEventSignalInfo
instance SignalInfo WidgetKeyPressEventSignalInfo where
    type HaskellCallbackType WidgetKeyPressEventSignalInfo = WidgetKeyPressEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetKeyPressEventCallback cb
        cb'' <- mk_WidgetKeyPressEventCallback cb'
        connectSignalFunPtr obj "key-press-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::key-press-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:keyPressEvent"})

#endif

-- signal Widget::key-release-event
-- | The [keyReleaseEvent](#g:signal:keyReleaseEvent) signal is emitted when a key is released.
-- 
-- To receive this signal, the t'GI.Gdk.Objects.Window.Window' associated to the widget needs
-- to enable the @/GDK_KEY_RELEASE_MASK/@ mask.
-- 
-- This signal will be sent to the grab widget if there is one.
type WidgetKeyReleaseEventCallback =
    Gdk.EventKey.EventKey
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventKey.EventKey' which triggered this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetKeyReleaseEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventKey.EventKey ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetKeyReleaseEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetKeyReleaseEventCallback :: C_WidgetKeyReleaseEventCallback -> IO (FunPtr C_WidgetKeyReleaseEventCallback)

wrap_WidgetKeyReleaseEventCallback :: 
    GObject a => (a -> WidgetKeyReleaseEventCallback) ->
    C_WidgetKeyReleaseEventCallback
wrap_WidgetKeyReleaseEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventKey.EventKey) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [keyReleaseEvent](#signal:keyReleaseEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #keyReleaseEvent callback
-- @
-- 
-- 
onWidgetKeyReleaseEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetKeyReleaseEventCallback) -> m SignalHandlerId
onWidgetKeyReleaseEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetKeyReleaseEventCallback wrapped
    wrapped'' <- mk_WidgetKeyReleaseEventCallback wrapped'
    connectSignalFunPtr obj "key-release-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [keyReleaseEvent](#signal:keyReleaseEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #keyReleaseEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetKeyReleaseEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetKeyReleaseEventCallback) -> m SignalHandlerId
afterWidgetKeyReleaseEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetKeyReleaseEventCallback wrapped
    wrapped'' <- mk_WidgetKeyReleaseEventCallback wrapped'
    connectSignalFunPtr obj "key-release-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetKeyReleaseEventSignalInfo
instance SignalInfo WidgetKeyReleaseEventSignalInfo where
    type HaskellCallbackType WidgetKeyReleaseEventSignalInfo = WidgetKeyReleaseEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetKeyReleaseEventCallback cb
        cb'' <- mk_WidgetKeyReleaseEventCallback cb'
        connectSignalFunPtr obj "key-release-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::key-release-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:keyReleaseEvent"})

#endif

-- signal Widget::keynav-failed
-- | Gets emitted if keyboard navigation fails.
-- See 'GI.Gtk.Objects.Widget.widgetKeynavFailed' for details.
-- 
-- /Since: 2.12/
type WidgetKeynavFailedCallback =
    Gtk.Enums.DirectionType
    -- ^ /@direction@/: the direction of movement
    -> IO Bool
    -- ^ __Returns:__ 'P.True' if stopping keyboard navigation is fine, 'P.False'
    --          if the emitting widget should try to handle the keyboard
    --          navigation attempt in its parent container(s).

type C_WidgetKeynavFailedCallback =
    Ptr Widget ->                           -- object
    CUInt ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetKeynavFailedCallback`.
foreign import ccall "wrapper"
    mk_WidgetKeynavFailedCallback :: C_WidgetKeynavFailedCallback -> IO (FunPtr C_WidgetKeynavFailedCallback)

wrap_WidgetKeynavFailedCallback :: 
    GObject a => (a -> WidgetKeynavFailedCallback) ->
    C_WidgetKeynavFailedCallback
wrap_WidgetKeynavFailedCallback gi'cb gi'selfPtr direction _ = do
    let direction' = (toEnum . fromIntegral) direction
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  direction'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [keynavFailed](#signal:keynavFailed) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #keynavFailed callback
-- @
-- 
-- 
onWidgetKeynavFailed :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetKeynavFailedCallback) -> m SignalHandlerId
onWidgetKeynavFailed obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetKeynavFailedCallback wrapped
    wrapped'' <- mk_WidgetKeynavFailedCallback wrapped'
    connectSignalFunPtr obj "keynav-failed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [keynavFailed](#signal:keynavFailed) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #keynavFailed callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetKeynavFailed :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetKeynavFailedCallback) -> m SignalHandlerId
afterWidgetKeynavFailed obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetKeynavFailedCallback wrapped
    wrapped'' <- mk_WidgetKeynavFailedCallback wrapped'
    connectSignalFunPtr obj "keynav-failed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetKeynavFailedSignalInfo
instance SignalInfo WidgetKeynavFailedSignalInfo where
    type HaskellCallbackType WidgetKeynavFailedSignalInfo = WidgetKeynavFailedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetKeynavFailedCallback cb
        cb'' <- mk_WidgetKeynavFailedCallback cb'
        connectSignalFunPtr obj "keynav-failed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::keynav-failed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:keynavFailed"})

#endif

-- signal Widget::leave-notify-event
-- | The [leaveNotifyEvent](#g:signal:leaveNotifyEvent) will be emitted when the pointer leaves
-- the /@widget@/\'s window.
-- 
-- To receive this signal, the t'GI.Gdk.Objects.Window.Window' associated to the widget needs
-- to enable the @/GDK_LEAVE_NOTIFY_MASK/@ mask.
-- 
-- This signal will be sent to the grab widget if there is one.
type WidgetLeaveNotifyEventCallback =
    Gdk.EventCrossing.EventCrossing
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventCrossing.EventCrossing' which triggered
    --   this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetLeaveNotifyEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventCrossing.EventCrossing ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetLeaveNotifyEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetLeaveNotifyEventCallback :: C_WidgetLeaveNotifyEventCallback -> IO (FunPtr C_WidgetLeaveNotifyEventCallback)

wrap_WidgetLeaveNotifyEventCallback :: 
    GObject a => (a -> WidgetLeaveNotifyEventCallback) ->
    C_WidgetLeaveNotifyEventCallback
wrap_WidgetLeaveNotifyEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventCrossing.EventCrossing) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [leaveNotifyEvent](#signal:leaveNotifyEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #leaveNotifyEvent callback
-- @
-- 
-- 
onWidgetLeaveNotifyEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetLeaveNotifyEventCallback) -> m SignalHandlerId
onWidgetLeaveNotifyEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetLeaveNotifyEventCallback wrapped
    wrapped'' <- mk_WidgetLeaveNotifyEventCallback wrapped'
    connectSignalFunPtr obj "leave-notify-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [leaveNotifyEvent](#signal:leaveNotifyEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #leaveNotifyEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetLeaveNotifyEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetLeaveNotifyEventCallback) -> m SignalHandlerId
afterWidgetLeaveNotifyEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetLeaveNotifyEventCallback wrapped
    wrapped'' <- mk_WidgetLeaveNotifyEventCallback wrapped'
    connectSignalFunPtr obj "leave-notify-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetLeaveNotifyEventSignalInfo
instance SignalInfo WidgetLeaveNotifyEventSignalInfo where
    type HaskellCallbackType WidgetLeaveNotifyEventSignalInfo = WidgetLeaveNotifyEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetLeaveNotifyEventCallback cb
        cb'' <- mk_WidgetLeaveNotifyEventCallback cb'
        connectSignalFunPtr obj "leave-notify-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::leave-notify-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:leaveNotifyEvent"})

#endif

-- signal Widget::map
-- | The [map](#g:signal:map) signal is emitted when /@widget@/ is going to be mapped, that is
-- when the widget is visible (which is controlled with
-- 'GI.Gtk.Objects.Widget.widgetSetVisible') and all its parents up to the toplevel widget
-- are also visible. Once the map has occurred, [Widget::mapEvent]("GI.Gtk.Objects.Widget#g:signal:mapEvent") will
-- be emitted.
-- 
-- The [map](#g:signal:map) signal can be used to determine whether a widget will be drawn,
-- for instance it can resume an animation that was stopped during the
-- emission of [Widget::unmap]("GI.Gtk.Objects.Widget#g:signal:unmap").
type WidgetMapCallback =
    IO ()

type C_WidgetMapCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetMapCallback`.
foreign import ccall "wrapper"
    mk_WidgetMapCallback :: C_WidgetMapCallback -> IO (FunPtr C_WidgetMapCallback)

wrap_WidgetMapCallback :: 
    GObject a => (a -> WidgetMapCallback) ->
    C_WidgetMapCallback
wrap_WidgetMapCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [map](#signal:map) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #map callback
-- @
-- 
-- 
onWidgetMap :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetMapCallback) -> m SignalHandlerId
onWidgetMap obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetMapCallback wrapped
    wrapped'' <- mk_WidgetMapCallback wrapped'
    connectSignalFunPtr obj "map" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [map](#signal:map) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #map callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetMap :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetMapCallback) -> m SignalHandlerId
afterWidgetMap obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetMapCallback wrapped
    wrapped'' <- mk_WidgetMapCallback wrapped'
    connectSignalFunPtr obj "map" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetMapSignalInfo
instance SignalInfo WidgetMapSignalInfo where
    type HaskellCallbackType WidgetMapSignalInfo = WidgetMapCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetMapCallback cb
        cb'' <- mk_WidgetMapCallback cb'
        connectSignalFunPtr obj "map" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::map"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:map"})

#endif

-- signal Widget::map-event
-- | The [mapEvent](#g:signal:mapEvent) signal will be emitted when the /@widget@/\'s window is
-- mapped. A window is mapped when it becomes visible on the screen.
-- 
-- To receive this signal, the t'GI.Gdk.Objects.Window.Window' associated to the widget needs
-- to enable the @/GDK_STRUCTURE_MASK/@ mask. GDK will enable this mask
-- automatically for all new windows.
type WidgetMapEventCallback =
    Gdk.EventAny.EventAny
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventAny.EventAny' which triggered this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetMapEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventAny.EventAny ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetMapEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetMapEventCallback :: C_WidgetMapEventCallback -> IO (FunPtr C_WidgetMapEventCallback)

wrap_WidgetMapEventCallback :: 
    GObject a => (a -> WidgetMapEventCallback) ->
    C_WidgetMapEventCallback
wrap_WidgetMapEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventAny.EventAny) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [mapEvent](#signal:mapEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #mapEvent callback
-- @
-- 
-- 
onWidgetMapEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetMapEventCallback) -> m SignalHandlerId
onWidgetMapEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetMapEventCallback wrapped
    wrapped'' <- mk_WidgetMapEventCallback wrapped'
    connectSignalFunPtr obj "map-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [mapEvent](#signal:mapEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #mapEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetMapEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetMapEventCallback) -> m SignalHandlerId
afterWidgetMapEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetMapEventCallback wrapped
    wrapped'' <- mk_WidgetMapEventCallback wrapped'
    connectSignalFunPtr obj "map-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetMapEventSignalInfo
instance SignalInfo WidgetMapEventSignalInfo where
    type HaskellCallbackType WidgetMapEventSignalInfo = WidgetMapEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetMapEventCallback cb
        cb'' <- mk_WidgetMapEventCallback cb'
        connectSignalFunPtr obj "map-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::map-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:mapEvent"})

#endif

-- signal Widget::mnemonic-activate
-- | The default handler for this signal activates /@widget@/ if /@groupCycling@/
-- is 'P.False', or just makes /@widget@/ grab focus if /@groupCycling@/ is 'P.True'.
type WidgetMnemonicActivateCallback =
    Bool
    -- ^ /@groupCycling@/: 'P.True' if there are other widgets with the same mnemonic
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    -- 'P.False' to propagate the event further.

type C_WidgetMnemonicActivateCallback =
    Ptr Widget ->                           -- object
    CInt ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetMnemonicActivateCallback`.
foreign import ccall "wrapper"
    mk_WidgetMnemonicActivateCallback :: C_WidgetMnemonicActivateCallback -> IO (FunPtr C_WidgetMnemonicActivateCallback)

wrap_WidgetMnemonicActivateCallback :: 
    GObject a => (a -> WidgetMnemonicActivateCallback) ->
    C_WidgetMnemonicActivateCallback
wrap_WidgetMnemonicActivateCallback gi'cb gi'selfPtr groupCycling _ = do
    let groupCycling' = (/= 0) groupCycling
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  groupCycling'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [mnemonicActivate](#signal:mnemonicActivate) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #mnemonicActivate callback
-- @
-- 
-- 
onWidgetMnemonicActivate :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetMnemonicActivateCallback) -> m SignalHandlerId
onWidgetMnemonicActivate obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetMnemonicActivateCallback wrapped
    wrapped'' <- mk_WidgetMnemonicActivateCallback wrapped'
    connectSignalFunPtr obj "mnemonic-activate" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [mnemonicActivate](#signal:mnemonicActivate) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #mnemonicActivate callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetMnemonicActivate :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetMnemonicActivateCallback) -> m SignalHandlerId
afterWidgetMnemonicActivate obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetMnemonicActivateCallback wrapped
    wrapped'' <- mk_WidgetMnemonicActivateCallback wrapped'
    connectSignalFunPtr obj "mnemonic-activate" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetMnemonicActivateSignalInfo
instance SignalInfo WidgetMnemonicActivateSignalInfo where
    type HaskellCallbackType WidgetMnemonicActivateSignalInfo = WidgetMnemonicActivateCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetMnemonicActivateCallback cb
        cb'' <- mk_WidgetMnemonicActivateCallback cb'
        connectSignalFunPtr obj "mnemonic-activate" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::mnemonic-activate"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:mnemonicActivate"})

#endif

-- signal Widget::motion-notify-event
-- | The [motionNotifyEvent](#g:signal:motionNotifyEvent) signal is emitted when the pointer moves
-- over the widget\'s t'GI.Gdk.Objects.Window.Window'.
-- 
-- To receive this signal, the t'GI.Gdk.Objects.Window.Window' associated to the widget
-- needs to enable the @/GDK_POINTER_MOTION_MASK/@ mask.
-- 
-- This signal will be sent to the grab widget if there is one.
type WidgetMotionNotifyEventCallback =
    Gdk.EventMotion.EventMotion
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventMotion.EventMotion' which triggered
    --   this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetMotionNotifyEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventMotion.EventMotion ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetMotionNotifyEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetMotionNotifyEventCallback :: C_WidgetMotionNotifyEventCallback -> IO (FunPtr C_WidgetMotionNotifyEventCallback)

wrap_WidgetMotionNotifyEventCallback :: 
    GObject a => (a -> WidgetMotionNotifyEventCallback) ->
    C_WidgetMotionNotifyEventCallback
wrap_WidgetMotionNotifyEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventMotion.EventMotion) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [motionNotifyEvent](#signal:motionNotifyEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #motionNotifyEvent callback
-- @
-- 
-- 
onWidgetMotionNotifyEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetMotionNotifyEventCallback) -> m SignalHandlerId
onWidgetMotionNotifyEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetMotionNotifyEventCallback wrapped
    wrapped'' <- mk_WidgetMotionNotifyEventCallback wrapped'
    connectSignalFunPtr obj "motion-notify-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [motionNotifyEvent](#signal:motionNotifyEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #motionNotifyEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetMotionNotifyEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetMotionNotifyEventCallback) -> m SignalHandlerId
afterWidgetMotionNotifyEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetMotionNotifyEventCallback wrapped
    wrapped'' <- mk_WidgetMotionNotifyEventCallback wrapped'
    connectSignalFunPtr obj "motion-notify-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetMotionNotifyEventSignalInfo
instance SignalInfo WidgetMotionNotifyEventSignalInfo where
    type HaskellCallbackType WidgetMotionNotifyEventSignalInfo = WidgetMotionNotifyEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetMotionNotifyEventCallback cb
        cb'' <- mk_WidgetMotionNotifyEventCallback cb'
        connectSignalFunPtr obj "motion-notify-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::motion-notify-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:motionNotifyEvent"})

#endif

-- signal Widget::move-focus
-- | /No description available in the introspection data./
type WidgetMoveFocusCallback =
    Gtk.Enums.DirectionType
    -> IO ()

type C_WidgetMoveFocusCallback =
    Ptr Widget ->                           -- object
    CUInt ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetMoveFocusCallback`.
foreign import ccall "wrapper"
    mk_WidgetMoveFocusCallback :: C_WidgetMoveFocusCallback -> IO (FunPtr C_WidgetMoveFocusCallback)

wrap_WidgetMoveFocusCallback :: 
    GObject a => (a -> WidgetMoveFocusCallback) ->
    C_WidgetMoveFocusCallback
wrap_WidgetMoveFocusCallback gi'cb gi'selfPtr direction _ = do
    let direction' = (toEnum . fromIntegral) direction
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  direction'


-- | Connect a signal handler for the [moveFocus](#signal:moveFocus) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #moveFocus callback
-- @
-- 
-- 
onWidgetMoveFocus :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetMoveFocusCallback) -> m SignalHandlerId
onWidgetMoveFocus obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetMoveFocusCallback wrapped
    wrapped'' <- mk_WidgetMoveFocusCallback wrapped'
    connectSignalFunPtr obj "move-focus" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [moveFocus](#signal:moveFocus) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #moveFocus callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetMoveFocus :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetMoveFocusCallback) -> m SignalHandlerId
afterWidgetMoveFocus obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetMoveFocusCallback wrapped
    wrapped'' <- mk_WidgetMoveFocusCallback wrapped'
    connectSignalFunPtr obj "move-focus" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetMoveFocusSignalInfo
instance SignalInfo WidgetMoveFocusSignalInfo where
    type HaskellCallbackType WidgetMoveFocusSignalInfo = WidgetMoveFocusCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetMoveFocusCallback cb
        cb'' <- mk_WidgetMoveFocusCallback cb'
        connectSignalFunPtr obj "move-focus" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::move-focus"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:moveFocus"})

#endif

-- signal Widget::parent-set
-- | The [parentSet](#g:signal:parentSet) signal is emitted when a new parent
-- has been set on a widget.
type WidgetParentSetCallback =
    Maybe Widget
    -- ^ /@oldParent@/: the previous parent, or 'P.Nothing' if the widget
    --   just got its initial parent.
    -> IO ()

type C_WidgetParentSetCallback =
    Ptr Widget ->                           -- object
    Ptr Widget ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetParentSetCallback`.
foreign import ccall "wrapper"
    mk_WidgetParentSetCallback :: C_WidgetParentSetCallback -> IO (FunPtr C_WidgetParentSetCallback)

wrap_WidgetParentSetCallback :: 
    GObject a => (a -> WidgetParentSetCallback) ->
    C_WidgetParentSetCallback
wrap_WidgetParentSetCallback gi'cb gi'selfPtr oldParent _ = do
    maybeOldParent <-
        if oldParent == FP.nullPtr
        then return Nothing
        else do
            oldParent' <- (newObject Widget) oldParent
            return $ Just oldParent'
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  maybeOldParent


-- | Connect a signal handler for the [parentSet](#signal:parentSet) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #parentSet callback
-- @
-- 
-- 
onWidgetParentSet :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetParentSetCallback) -> m SignalHandlerId
onWidgetParentSet obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetParentSetCallback wrapped
    wrapped'' <- mk_WidgetParentSetCallback wrapped'
    connectSignalFunPtr obj "parent-set" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [parentSet](#signal:parentSet) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #parentSet callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetParentSet :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetParentSetCallback) -> m SignalHandlerId
afterWidgetParentSet obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetParentSetCallback wrapped
    wrapped'' <- mk_WidgetParentSetCallback wrapped'
    connectSignalFunPtr obj "parent-set" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetParentSetSignalInfo
instance SignalInfo WidgetParentSetSignalInfo where
    type HaskellCallbackType WidgetParentSetSignalInfo = WidgetParentSetCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetParentSetCallback cb
        cb'' <- mk_WidgetParentSetCallback cb'
        connectSignalFunPtr obj "parent-set" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::parent-set"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:parentSet"})

#endif

-- signal Widget::popup-menu
-- | This signal gets emitted whenever a widget should pop up a context
-- menu. This usually happens through the standard key binding mechanism;
-- by pressing a certain key while a widget is focused, the user can cause
-- the widget to pop up a menu.  For example, the t'GI.Gtk.Objects.Entry.Entry' widget creates
-- a menu with clipboard commands. See the
-- [Popup Menu Migration Checklist][checklist-popup-menu]
-- for an example of how to use this signal.
type WidgetPopupMenuCallback =
    IO Bool
    -- ^ __Returns:__ 'P.True' if a menu was activated

type C_WidgetPopupMenuCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetPopupMenuCallback`.
foreign import ccall "wrapper"
    mk_WidgetPopupMenuCallback :: C_WidgetPopupMenuCallback -> IO (FunPtr C_WidgetPopupMenuCallback)

wrap_WidgetPopupMenuCallback :: 
    GObject a => (a -> WidgetPopupMenuCallback) ->
    C_WidgetPopupMenuCallback
wrap_WidgetPopupMenuCallback gi'cb gi'selfPtr _ = do
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [popupMenu](#signal:popupMenu) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #popupMenu callback
-- @
-- 
-- 
onWidgetPopupMenu :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetPopupMenuCallback) -> m SignalHandlerId
onWidgetPopupMenu obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetPopupMenuCallback wrapped
    wrapped'' <- mk_WidgetPopupMenuCallback wrapped'
    connectSignalFunPtr obj "popup-menu" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [popupMenu](#signal:popupMenu) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #popupMenu callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetPopupMenu :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetPopupMenuCallback) -> m SignalHandlerId
afterWidgetPopupMenu obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetPopupMenuCallback wrapped
    wrapped'' <- mk_WidgetPopupMenuCallback wrapped'
    connectSignalFunPtr obj "popup-menu" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetPopupMenuSignalInfo
instance SignalInfo WidgetPopupMenuSignalInfo where
    type HaskellCallbackType WidgetPopupMenuSignalInfo = WidgetPopupMenuCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetPopupMenuCallback cb
        cb'' <- mk_WidgetPopupMenuCallback cb'
        connectSignalFunPtr obj "popup-menu" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::popup-menu"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:popupMenu"})

#endif

-- signal Widget::property-notify-event
-- | The [propertyNotifyEvent](#g:signal:propertyNotifyEvent) signal will be emitted when a property on
-- the /@widget@/\'s window has been changed or deleted.
-- 
-- To receive this signal, the t'GI.Gdk.Objects.Window.Window' associated to the widget needs
-- to enable the @/GDK_PROPERTY_CHANGE_MASK/@ mask.
type WidgetPropertyNotifyEventCallback =
    Gdk.EventProperty.EventProperty
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventProperty.EventProperty' which triggered
    --   this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetPropertyNotifyEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventProperty.EventProperty ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetPropertyNotifyEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetPropertyNotifyEventCallback :: C_WidgetPropertyNotifyEventCallback -> IO (FunPtr C_WidgetPropertyNotifyEventCallback)

wrap_WidgetPropertyNotifyEventCallback :: 
    GObject a => (a -> WidgetPropertyNotifyEventCallback) ->
    C_WidgetPropertyNotifyEventCallback
wrap_WidgetPropertyNotifyEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventProperty.EventProperty) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [propertyNotifyEvent](#signal:propertyNotifyEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #propertyNotifyEvent callback
-- @
-- 
-- 
onWidgetPropertyNotifyEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetPropertyNotifyEventCallback) -> m SignalHandlerId
onWidgetPropertyNotifyEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetPropertyNotifyEventCallback wrapped
    wrapped'' <- mk_WidgetPropertyNotifyEventCallback wrapped'
    connectSignalFunPtr obj "property-notify-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [propertyNotifyEvent](#signal:propertyNotifyEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #propertyNotifyEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetPropertyNotifyEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetPropertyNotifyEventCallback) -> m SignalHandlerId
afterWidgetPropertyNotifyEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetPropertyNotifyEventCallback wrapped
    wrapped'' <- mk_WidgetPropertyNotifyEventCallback wrapped'
    connectSignalFunPtr obj "property-notify-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetPropertyNotifyEventSignalInfo
instance SignalInfo WidgetPropertyNotifyEventSignalInfo where
    type HaskellCallbackType WidgetPropertyNotifyEventSignalInfo = WidgetPropertyNotifyEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetPropertyNotifyEventCallback cb
        cb'' <- mk_WidgetPropertyNotifyEventCallback cb'
        connectSignalFunPtr obj "property-notify-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::property-notify-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:propertyNotifyEvent"})

#endif

-- signal Widget::proximity-in-event
-- | To receive this signal the t'GI.Gdk.Objects.Window.Window' associated to the widget needs
-- to enable the @/GDK_PROXIMITY_IN_MASK/@ mask.
-- 
-- This signal will be sent to the grab widget if there is one.
type WidgetProximityInEventCallback =
    Gdk.EventProximity.EventProximity
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventProximity.EventProximity' which triggered
    --   this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetProximityInEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventProximity.EventProximity ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetProximityInEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetProximityInEventCallback :: C_WidgetProximityInEventCallback -> IO (FunPtr C_WidgetProximityInEventCallback)

wrap_WidgetProximityInEventCallback :: 
    GObject a => (a -> WidgetProximityInEventCallback) ->
    C_WidgetProximityInEventCallback
wrap_WidgetProximityInEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventProximity.EventProximity) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [proximityInEvent](#signal:proximityInEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #proximityInEvent callback
-- @
-- 
-- 
onWidgetProximityInEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetProximityInEventCallback) -> m SignalHandlerId
onWidgetProximityInEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetProximityInEventCallback wrapped
    wrapped'' <- mk_WidgetProximityInEventCallback wrapped'
    connectSignalFunPtr obj "proximity-in-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [proximityInEvent](#signal:proximityInEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #proximityInEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetProximityInEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetProximityInEventCallback) -> m SignalHandlerId
afterWidgetProximityInEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetProximityInEventCallback wrapped
    wrapped'' <- mk_WidgetProximityInEventCallback wrapped'
    connectSignalFunPtr obj "proximity-in-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetProximityInEventSignalInfo
instance SignalInfo WidgetProximityInEventSignalInfo where
    type HaskellCallbackType WidgetProximityInEventSignalInfo = WidgetProximityInEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetProximityInEventCallback cb
        cb'' <- mk_WidgetProximityInEventCallback cb'
        connectSignalFunPtr obj "proximity-in-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::proximity-in-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:proximityInEvent"})

#endif

-- signal Widget::proximity-out-event
-- | To receive this signal the t'GI.Gdk.Objects.Window.Window' associated to the widget needs
-- to enable the @/GDK_PROXIMITY_OUT_MASK/@ mask.
-- 
-- This signal will be sent to the grab widget if there is one.
type WidgetProximityOutEventCallback =
    Gdk.EventProximity.EventProximity
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventProximity.EventProximity' which triggered
    --   this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetProximityOutEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventProximity.EventProximity ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetProximityOutEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetProximityOutEventCallback :: C_WidgetProximityOutEventCallback -> IO (FunPtr C_WidgetProximityOutEventCallback)

wrap_WidgetProximityOutEventCallback :: 
    GObject a => (a -> WidgetProximityOutEventCallback) ->
    C_WidgetProximityOutEventCallback
wrap_WidgetProximityOutEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventProximity.EventProximity) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [proximityOutEvent](#signal:proximityOutEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #proximityOutEvent callback
-- @
-- 
-- 
onWidgetProximityOutEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetProximityOutEventCallback) -> m SignalHandlerId
onWidgetProximityOutEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetProximityOutEventCallback wrapped
    wrapped'' <- mk_WidgetProximityOutEventCallback wrapped'
    connectSignalFunPtr obj "proximity-out-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [proximityOutEvent](#signal:proximityOutEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #proximityOutEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetProximityOutEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetProximityOutEventCallback) -> m SignalHandlerId
afterWidgetProximityOutEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetProximityOutEventCallback wrapped
    wrapped'' <- mk_WidgetProximityOutEventCallback wrapped'
    connectSignalFunPtr obj "proximity-out-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetProximityOutEventSignalInfo
instance SignalInfo WidgetProximityOutEventSignalInfo where
    type HaskellCallbackType WidgetProximityOutEventSignalInfo = WidgetProximityOutEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetProximityOutEventCallback cb
        cb'' <- mk_WidgetProximityOutEventCallback cb'
        connectSignalFunPtr obj "proximity-out-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::proximity-out-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:proximityOutEvent"})

#endif

-- signal Widget::query-tooltip
-- | Emitted when [Widget:hasTooltip]("GI.Gtk.Objects.Widget#g:attr:hasTooltip") is 'P.True' and the hover timeout
-- has expired with the cursor hovering \"above\" /@widget@/; or emitted when /@widget@/ got
-- focus in keyboard mode.
-- 
-- Using the given coordinates, the signal handler should determine
-- whether a tooltip should be shown for /@widget@/. If this is the case
-- 'P.True' should be returned, 'P.False' otherwise.  Note that if
-- /@keyboardMode@/ is 'P.True', the values of /@x@/ and /@y@/ are undefined and
-- should not be used.
-- 
-- The signal handler is free to manipulate /@tooltip@/ with the therefore
-- destined function calls.
-- 
-- /Since: 2.12/
type WidgetQueryTooltipCallback =
    Int32
    -- ^ /@x@/: the x coordinate of the cursor position where the request has
    --     been emitted, relative to /@widget@/\'s left side
    -> Int32
    -- ^ /@y@/: the y coordinate of the cursor position where the request has
    --     been emitted, relative to /@widget@/\'s top
    -> Bool
    -- ^ /@keyboardMode@/: 'P.True' if the tooltip was triggered using the keyboard
    -> Gtk.Tooltip.Tooltip
    -- ^ /@tooltip@/: a t'GI.Gtk.Objects.Tooltip.Tooltip'
    -> IO Bool
    -- ^ __Returns:__ 'P.True' if /@tooltip@/ should be shown right now, 'P.False' otherwise.

type C_WidgetQueryTooltipCallback =
    Ptr Widget ->                           -- object
    Int32 ->
    Int32 ->
    CInt ->
    Ptr Gtk.Tooltip.Tooltip ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetQueryTooltipCallback`.
foreign import ccall "wrapper"
    mk_WidgetQueryTooltipCallback :: C_WidgetQueryTooltipCallback -> IO (FunPtr C_WidgetQueryTooltipCallback)

wrap_WidgetQueryTooltipCallback :: 
    GObject a => (a -> WidgetQueryTooltipCallback) ->
    C_WidgetQueryTooltipCallback
wrap_WidgetQueryTooltipCallback gi'cb gi'selfPtr x y keyboardMode tooltip _ = do
    let keyboardMode' = (/= 0) keyboardMode
    tooltip' <- (newObject Gtk.Tooltip.Tooltip) tooltip
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  x y keyboardMode' tooltip'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [queryTooltip](#signal:queryTooltip) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #queryTooltip callback
-- @
-- 
-- 
onWidgetQueryTooltip :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetQueryTooltipCallback) -> m SignalHandlerId
onWidgetQueryTooltip obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetQueryTooltipCallback wrapped
    wrapped'' <- mk_WidgetQueryTooltipCallback wrapped'
    connectSignalFunPtr obj "query-tooltip" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [queryTooltip](#signal:queryTooltip) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #queryTooltip callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetQueryTooltip :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetQueryTooltipCallback) -> m SignalHandlerId
afterWidgetQueryTooltip obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetQueryTooltipCallback wrapped
    wrapped'' <- mk_WidgetQueryTooltipCallback wrapped'
    connectSignalFunPtr obj "query-tooltip" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetQueryTooltipSignalInfo
instance SignalInfo WidgetQueryTooltipSignalInfo where
    type HaskellCallbackType WidgetQueryTooltipSignalInfo = WidgetQueryTooltipCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetQueryTooltipCallback cb
        cb'' <- mk_WidgetQueryTooltipCallback cb'
        connectSignalFunPtr obj "query-tooltip" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::query-tooltip"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:queryTooltip"})

#endif

-- signal Widget::realize
-- | The [realize](#g:signal:realize) signal is emitted when /@widget@/ is associated with a
-- t'GI.Gdk.Objects.Window.Window', which means that 'GI.Gtk.Objects.Widget.widgetRealize' has been called or the
-- widget has been mapped (that is, it is going to be drawn).
type WidgetRealizeCallback =
    IO ()

type C_WidgetRealizeCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetRealizeCallback`.
foreign import ccall "wrapper"
    mk_WidgetRealizeCallback :: C_WidgetRealizeCallback -> IO (FunPtr C_WidgetRealizeCallback)

wrap_WidgetRealizeCallback :: 
    GObject a => (a -> WidgetRealizeCallback) ->
    C_WidgetRealizeCallback
wrap_WidgetRealizeCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [realize](#signal:realize) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #realize callback
-- @
-- 
-- 
onWidgetRealize :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetRealizeCallback) -> m SignalHandlerId
onWidgetRealize obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetRealizeCallback wrapped
    wrapped'' <- mk_WidgetRealizeCallback wrapped'
    connectSignalFunPtr obj "realize" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [realize](#signal:realize) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #realize callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetRealize :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetRealizeCallback) -> m SignalHandlerId
afterWidgetRealize obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetRealizeCallback wrapped
    wrapped'' <- mk_WidgetRealizeCallback wrapped'
    connectSignalFunPtr obj "realize" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetRealizeSignalInfo
instance SignalInfo WidgetRealizeSignalInfo where
    type HaskellCallbackType WidgetRealizeSignalInfo = WidgetRealizeCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetRealizeCallback cb
        cb'' <- mk_WidgetRealizeCallback cb'
        connectSignalFunPtr obj "realize" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::realize"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:realize"})

#endif

-- signal Widget::screen-changed
-- | The [screenChanged](#g:signal:screenChanged) signal gets emitted when the
-- screen of a widget has changed.
type WidgetScreenChangedCallback =
    Maybe Gdk.Screen.Screen
    -- ^ /@previousScreen@/: the previous screen, or 'P.Nothing' if the
    --   widget was not associated with a screen before
    -> IO ()

type C_WidgetScreenChangedCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.Screen.Screen ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetScreenChangedCallback`.
foreign import ccall "wrapper"
    mk_WidgetScreenChangedCallback :: C_WidgetScreenChangedCallback -> IO (FunPtr C_WidgetScreenChangedCallback)

wrap_WidgetScreenChangedCallback :: 
    GObject a => (a -> WidgetScreenChangedCallback) ->
    C_WidgetScreenChangedCallback
wrap_WidgetScreenChangedCallback gi'cb gi'selfPtr previousScreen _ = do
    maybePreviousScreen <-
        if previousScreen == FP.nullPtr
        then return Nothing
        else do
            previousScreen' <- (newObject Gdk.Screen.Screen) previousScreen
            return $ Just previousScreen'
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  maybePreviousScreen


-- | Connect a signal handler for the [screenChanged](#signal:screenChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #screenChanged callback
-- @
-- 
-- 
onWidgetScreenChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetScreenChangedCallback) -> m SignalHandlerId
onWidgetScreenChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetScreenChangedCallback wrapped
    wrapped'' <- mk_WidgetScreenChangedCallback wrapped'
    connectSignalFunPtr obj "screen-changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [screenChanged](#signal:screenChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #screenChanged callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetScreenChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetScreenChangedCallback) -> m SignalHandlerId
afterWidgetScreenChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetScreenChangedCallback wrapped
    wrapped'' <- mk_WidgetScreenChangedCallback wrapped'
    connectSignalFunPtr obj "screen-changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetScreenChangedSignalInfo
instance SignalInfo WidgetScreenChangedSignalInfo where
    type HaskellCallbackType WidgetScreenChangedSignalInfo = WidgetScreenChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetScreenChangedCallback cb
        cb'' <- mk_WidgetScreenChangedCallback cb'
        connectSignalFunPtr obj "screen-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::screen-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:screenChanged"})

#endif

-- signal Widget::scroll-event
-- | The [scrollEvent](#g:signal:scrollEvent) signal is emitted when a button in the 4 to 7
-- range is pressed. Wheel mice are usually configured to generate
-- button press events for buttons 4 and 5 when the wheel is turned.
-- 
-- To receive this signal, the t'GI.Gdk.Objects.Window.Window' associated to the widget needs
-- to enable the @/GDK_SCROLL_MASK/@ mask.
-- 
-- This signal will be sent to the grab widget if there is one.
type WidgetScrollEventCallback =
    Gdk.EventScroll.EventScroll
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventScroll.EventScroll' which triggered
    --   this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetScrollEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventScroll.EventScroll ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetScrollEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetScrollEventCallback :: C_WidgetScrollEventCallback -> IO (FunPtr C_WidgetScrollEventCallback)

wrap_WidgetScrollEventCallback :: 
    GObject a => (a -> WidgetScrollEventCallback) ->
    C_WidgetScrollEventCallback
wrap_WidgetScrollEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventScroll.EventScroll) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [scrollEvent](#signal:scrollEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #scrollEvent callback
-- @
-- 
-- 
onWidgetScrollEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetScrollEventCallback) -> m SignalHandlerId
onWidgetScrollEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetScrollEventCallback wrapped
    wrapped'' <- mk_WidgetScrollEventCallback wrapped'
    connectSignalFunPtr obj "scroll-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [scrollEvent](#signal:scrollEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #scrollEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetScrollEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetScrollEventCallback) -> m SignalHandlerId
afterWidgetScrollEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetScrollEventCallback wrapped
    wrapped'' <- mk_WidgetScrollEventCallback wrapped'
    connectSignalFunPtr obj "scroll-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetScrollEventSignalInfo
instance SignalInfo WidgetScrollEventSignalInfo where
    type HaskellCallbackType WidgetScrollEventSignalInfo = WidgetScrollEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetScrollEventCallback cb
        cb'' <- mk_WidgetScrollEventCallback cb'
        connectSignalFunPtr obj "scroll-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::scroll-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:scrollEvent"})

#endif

-- signal Widget::selection-clear-event
-- | The [selectionClearEvent](#g:signal:selectionClearEvent) signal will be emitted when the
-- the /@widget@/\'s window has lost ownership of a selection.
type WidgetSelectionClearEventCallback =
    Gdk.EventSelection.EventSelection
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventSelection.EventSelection' which triggered
    --   this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetSelectionClearEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventSelection.EventSelection ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetSelectionClearEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetSelectionClearEventCallback :: C_WidgetSelectionClearEventCallback -> IO (FunPtr C_WidgetSelectionClearEventCallback)

wrap_WidgetSelectionClearEventCallback :: 
    GObject a => (a -> WidgetSelectionClearEventCallback) ->
    C_WidgetSelectionClearEventCallback
wrap_WidgetSelectionClearEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventSelection.EventSelection) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [selectionClearEvent](#signal:selectionClearEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #selectionClearEvent callback
-- @
-- 
-- 
onWidgetSelectionClearEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetSelectionClearEventCallback) -> m SignalHandlerId
onWidgetSelectionClearEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetSelectionClearEventCallback wrapped
    wrapped'' <- mk_WidgetSelectionClearEventCallback wrapped'
    connectSignalFunPtr obj "selection-clear-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [selectionClearEvent](#signal:selectionClearEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #selectionClearEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetSelectionClearEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetSelectionClearEventCallback) -> m SignalHandlerId
afterWidgetSelectionClearEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetSelectionClearEventCallback wrapped
    wrapped'' <- mk_WidgetSelectionClearEventCallback wrapped'
    connectSignalFunPtr obj "selection-clear-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetSelectionClearEventSignalInfo
instance SignalInfo WidgetSelectionClearEventSignalInfo where
    type HaskellCallbackType WidgetSelectionClearEventSignalInfo = WidgetSelectionClearEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetSelectionClearEventCallback cb
        cb'' <- mk_WidgetSelectionClearEventCallback cb'
        connectSignalFunPtr obj "selection-clear-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::selection-clear-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:selectionClearEvent"})

#endif

-- signal Widget::selection-get
-- | /No description available in the introspection data./
type WidgetSelectionGetCallback =
    Gtk.SelectionData.SelectionData
    -> Word32
    -> Word32
    -> IO ()

type C_WidgetSelectionGetCallback =
    Ptr Widget ->                           -- object
    Ptr Gtk.SelectionData.SelectionData ->
    Word32 ->
    Word32 ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetSelectionGetCallback`.
foreign import ccall "wrapper"
    mk_WidgetSelectionGetCallback :: C_WidgetSelectionGetCallback -> IO (FunPtr C_WidgetSelectionGetCallback)

wrap_WidgetSelectionGetCallback :: 
    GObject a => (a -> WidgetSelectionGetCallback) ->
    C_WidgetSelectionGetCallback
wrap_WidgetSelectionGetCallback gi'cb gi'selfPtr data_ info time _ = do
    B.ManagedPtr.withTransient  data_ $ \data_' -> do
        B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  data_' info time


-- | Connect a signal handler for the [selectionGet](#signal:selectionGet) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #selectionGet callback
-- @
-- 
-- 
onWidgetSelectionGet :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetSelectionGetCallback) -> m SignalHandlerId
onWidgetSelectionGet obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetSelectionGetCallback wrapped
    wrapped'' <- mk_WidgetSelectionGetCallback wrapped'
    connectSignalFunPtr obj "selection-get" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [selectionGet](#signal:selectionGet) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #selectionGet callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetSelectionGet :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetSelectionGetCallback) -> m SignalHandlerId
afterWidgetSelectionGet obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetSelectionGetCallback wrapped
    wrapped'' <- mk_WidgetSelectionGetCallback wrapped'
    connectSignalFunPtr obj "selection-get" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetSelectionGetSignalInfo
instance SignalInfo WidgetSelectionGetSignalInfo where
    type HaskellCallbackType WidgetSelectionGetSignalInfo = WidgetSelectionGetCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetSelectionGetCallback cb
        cb'' <- mk_WidgetSelectionGetCallback cb'
        connectSignalFunPtr obj "selection-get" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::selection-get"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:selectionGet"})

#endif

-- signal Widget::selection-notify-event
-- | /No description available in the introspection data./
type WidgetSelectionNotifyEventCallback =
    Gdk.EventSelection.EventSelection
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event. 'P.False' to propagate the event further.

type C_WidgetSelectionNotifyEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventSelection.EventSelection ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetSelectionNotifyEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetSelectionNotifyEventCallback :: C_WidgetSelectionNotifyEventCallback -> IO (FunPtr C_WidgetSelectionNotifyEventCallback)

wrap_WidgetSelectionNotifyEventCallback :: 
    GObject a => (a -> WidgetSelectionNotifyEventCallback) ->
    C_WidgetSelectionNotifyEventCallback
wrap_WidgetSelectionNotifyEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventSelection.EventSelection) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [selectionNotifyEvent](#signal:selectionNotifyEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #selectionNotifyEvent callback
-- @
-- 
-- 
onWidgetSelectionNotifyEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetSelectionNotifyEventCallback) -> m SignalHandlerId
onWidgetSelectionNotifyEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetSelectionNotifyEventCallback wrapped
    wrapped'' <- mk_WidgetSelectionNotifyEventCallback wrapped'
    connectSignalFunPtr obj "selection-notify-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [selectionNotifyEvent](#signal:selectionNotifyEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #selectionNotifyEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetSelectionNotifyEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetSelectionNotifyEventCallback) -> m SignalHandlerId
afterWidgetSelectionNotifyEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetSelectionNotifyEventCallback wrapped
    wrapped'' <- mk_WidgetSelectionNotifyEventCallback wrapped'
    connectSignalFunPtr obj "selection-notify-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetSelectionNotifyEventSignalInfo
instance SignalInfo WidgetSelectionNotifyEventSignalInfo where
    type HaskellCallbackType WidgetSelectionNotifyEventSignalInfo = WidgetSelectionNotifyEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetSelectionNotifyEventCallback cb
        cb'' <- mk_WidgetSelectionNotifyEventCallback cb'
        connectSignalFunPtr obj "selection-notify-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::selection-notify-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:selectionNotifyEvent"})

#endif

-- signal Widget::selection-received
-- | /No description available in the introspection data./
type WidgetSelectionReceivedCallback =
    Gtk.SelectionData.SelectionData
    -> Word32
    -> IO ()

type C_WidgetSelectionReceivedCallback =
    Ptr Widget ->                           -- object
    Ptr Gtk.SelectionData.SelectionData ->
    Word32 ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetSelectionReceivedCallback`.
foreign import ccall "wrapper"
    mk_WidgetSelectionReceivedCallback :: C_WidgetSelectionReceivedCallback -> IO (FunPtr C_WidgetSelectionReceivedCallback)

wrap_WidgetSelectionReceivedCallback :: 
    GObject a => (a -> WidgetSelectionReceivedCallback) ->
    C_WidgetSelectionReceivedCallback
wrap_WidgetSelectionReceivedCallback gi'cb gi'selfPtr data_ time _ = do
    B.ManagedPtr.withTransient  data_ $ \data_' -> do
        B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  data_' time


-- | Connect a signal handler for the [selectionReceived](#signal:selectionReceived) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #selectionReceived callback
-- @
-- 
-- 
onWidgetSelectionReceived :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetSelectionReceivedCallback) -> m SignalHandlerId
onWidgetSelectionReceived obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetSelectionReceivedCallback wrapped
    wrapped'' <- mk_WidgetSelectionReceivedCallback wrapped'
    connectSignalFunPtr obj "selection-received" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [selectionReceived](#signal:selectionReceived) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #selectionReceived callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetSelectionReceived :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetSelectionReceivedCallback) -> m SignalHandlerId
afterWidgetSelectionReceived obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetSelectionReceivedCallback wrapped
    wrapped'' <- mk_WidgetSelectionReceivedCallback wrapped'
    connectSignalFunPtr obj "selection-received" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetSelectionReceivedSignalInfo
instance SignalInfo WidgetSelectionReceivedSignalInfo where
    type HaskellCallbackType WidgetSelectionReceivedSignalInfo = WidgetSelectionReceivedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetSelectionReceivedCallback cb
        cb'' <- mk_WidgetSelectionReceivedCallback cb'
        connectSignalFunPtr obj "selection-received" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::selection-received"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:selectionReceived"})

#endif

-- signal Widget::selection-request-event
-- | The [selectionRequestEvent](#g:signal:selectionRequestEvent) signal will be emitted when
-- another client requests ownership of the selection owned by
-- the /@widget@/\'s window.
type WidgetSelectionRequestEventCallback =
    Gdk.EventSelection.EventSelection
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventSelection.EventSelection' which triggered
    --   this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetSelectionRequestEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventSelection.EventSelection ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetSelectionRequestEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetSelectionRequestEventCallback :: C_WidgetSelectionRequestEventCallback -> IO (FunPtr C_WidgetSelectionRequestEventCallback)

wrap_WidgetSelectionRequestEventCallback :: 
    GObject a => (a -> WidgetSelectionRequestEventCallback) ->
    C_WidgetSelectionRequestEventCallback
wrap_WidgetSelectionRequestEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventSelection.EventSelection) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [selectionRequestEvent](#signal:selectionRequestEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #selectionRequestEvent callback
-- @
-- 
-- 
onWidgetSelectionRequestEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetSelectionRequestEventCallback) -> m SignalHandlerId
onWidgetSelectionRequestEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetSelectionRequestEventCallback wrapped
    wrapped'' <- mk_WidgetSelectionRequestEventCallback wrapped'
    connectSignalFunPtr obj "selection-request-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [selectionRequestEvent](#signal:selectionRequestEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #selectionRequestEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetSelectionRequestEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetSelectionRequestEventCallback) -> m SignalHandlerId
afterWidgetSelectionRequestEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetSelectionRequestEventCallback wrapped
    wrapped'' <- mk_WidgetSelectionRequestEventCallback wrapped'
    connectSignalFunPtr obj "selection-request-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetSelectionRequestEventSignalInfo
instance SignalInfo WidgetSelectionRequestEventSignalInfo where
    type HaskellCallbackType WidgetSelectionRequestEventSignalInfo = WidgetSelectionRequestEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetSelectionRequestEventCallback cb
        cb'' <- mk_WidgetSelectionRequestEventCallback cb'
        connectSignalFunPtr obj "selection-request-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::selection-request-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:selectionRequestEvent"})

#endif

-- signal Widget::show
-- | The [show](#g:signal:show) signal is emitted when /@widget@/ is shown, for example with
-- 'GI.Gtk.Objects.Widget.widgetShow'.
type WidgetShowCallback =
    IO ()

type C_WidgetShowCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetShowCallback`.
foreign import ccall "wrapper"
    mk_WidgetShowCallback :: C_WidgetShowCallback -> IO (FunPtr C_WidgetShowCallback)

wrap_WidgetShowCallback :: 
    GObject a => (a -> WidgetShowCallback) ->
    C_WidgetShowCallback
wrap_WidgetShowCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [show](#signal:show) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #show callback
-- @
-- 
-- 
onWidgetShow :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetShowCallback) -> m SignalHandlerId
onWidgetShow obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetShowCallback wrapped
    wrapped'' <- mk_WidgetShowCallback wrapped'
    connectSignalFunPtr obj "show" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [show](#signal:show) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #show callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetShow :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetShowCallback) -> m SignalHandlerId
afterWidgetShow obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetShowCallback wrapped
    wrapped'' <- mk_WidgetShowCallback wrapped'
    connectSignalFunPtr obj "show" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetShowSignalInfo
instance SignalInfo WidgetShowSignalInfo where
    type HaskellCallbackType WidgetShowSignalInfo = WidgetShowCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetShowCallback cb
        cb'' <- mk_WidgetShowCallback cb'
        connectSignalFunPtr obj "show" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::show"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:show"})

#endif

-- signal Widget::show-help
-- | /No description available in the introspection data./
type WidgetShowHelpCallback =
    Gtk.Enums.WidgetHelpType
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    -- 'P.False' to propagate the event further.

type C_WidgetShowHelpCallback =
    Ptr Widget ->                           -- object
    CUInt ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetShowHelpCallback`.
foreign import ccall "wrapper"
    mk_WidgetShowHelpCallback :: C_WidgetShowHelpCallback -> IO (FunPtr C_WidgetShowHelpCallback)

wrap_WidgetShowHelpCallback :: 
    GObject a => (a -> WidgetShowHelpCallback) ->
    C_WidgetShowHelpCallback
wrap_WidgetShowHelpCallback gi'cb gi'selfPtr helpType _ = do
    let helpType' = (toEnum . fromIntegral) helpType
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  helpType'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [showHelp](#signal:showHelp) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #showHelp callback
-- @
-- 
-- 
onWidgetShowHelp :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetShowHelpCallback) -> m SignalHandlerId
onWidgetShowHelp obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetShowHelpCallback wrapped
    wrapped'' <- mk_WidgetShowHelpCallback wrapped'
    connectSignalFunPtr obj "show-help" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [showHelp](#signal:showHelp) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #showHelp callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetShowHelp :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetShowHelpCallback) -> m SignalHandlerId
afterWidgetShowHelp obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetShowHelpCallback wrapped
    wrapped'' <- mk_WidgetShowHelpCallback wrapped'
    connectSignalFunPtr obj "show-help" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetShowHelpSignalInfo
instance SignalInfo WidgetShowHelpSignalInfo where
    type HaskellCallbackType WidgetShowHelpSignalInfo = WidgetShowHelpCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetShowHelpCallback cb
        cb'' <- mk_WidgetShowHelpCallback cb'
        connectSignalFunPtr obj "show-help" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::show-help"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:showHelp"})

#endif

-- signal Widget::size-allocate
-- | /No description available in the introspection data./
type WidgetSizeAllocateCallback =
    Gdk.Rectangle.Rectangle
    -- ^ /@allocation@/: the region which has been
    --   allocated to the widget.
    -> IO ()

type C_WidgetSizeAllocateCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.Rectangle.Rectangle ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetSizeAllocateCallback`.
foreign import ccall "wrapper"
    mk_WidgetSizeAllocateCallback :: C_WidgetSizeAllocateCallback -> IO (FunPtr C_WidgetSizeAllocateCallback)

wrap_WidgetSizeAllocateCallback :: 
    GObject a => (a -> WidgetSizeAllocateCallback) ->
    C_WidgetSizeAllocateCallback
wrap_WidgetSizeAllocateCallback gi'cb gi'selfPtr allocation _ = do
    B.ManagedPtr.withTransient  allocation $ \allocation' -> do
        B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  allocation'


-- | Connect a signal handler for the [sizeAllocate](#signal:sizeAllocate) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #sizeAllocate callback
-- @
-- 
-- 
onWidgetSizeAllocate :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetSizeAllocateCallback) -> m SignalHandlerId
onWidgetSizeAllocate obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetSizeAllocateCallback wrapped
    wrapped'' <- mk_WidgetSizeAllocateCallback wrapped'
    connectSignalFunPtr obj "size-allocate" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [sizeAllocate](#signal:sizeAllocate) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #sizeAllocate callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetSizeAllocate :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetSizeAllocateCallback) -> m SignalHandlerId
afterWidgetSizeAllocate obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetSizeAllocateCallback wrapped
    wrapped'' <- mk_WidgetSizeAllocateCallback wrapped'
    connectSignalFunPtr obj "size-allocate" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetSizeAllocateSignalInfo
instance SignalInfo WidgetSizeAllocateSignalInfo where
    type HaskellCallbackType WidgetSizeAllocateSignalInfo = WidgetSizeAllocateCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetSizeAllocateCallback cb
        cb'' <- mk_WidgetSizeAllocateCallback cb'
        connectSignalFunPtr obj "size-allocate" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::size-allocate"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:sizeAllocate"})

#endif

-- signal Widget::state-changed
{-# DEPRECATED WidgetStateChangedCallback ["(Since version 3.0)","Use [Widget::stateFlagsChanged](\"GI.Gtk.Objects.Widget#g:signal:stateFlagsChanged\") instead."] #-}
-- | The [stateChanged](#g:signal:stateChanged) signal is emitted when the widget state changes.
-- See 'GI.Gtk.Objects.Widget.widgetGetState'.
type WidgetStateChangedCallback =
    Gtk.Enums.StateType
    -- ^ /@state@/: the previous state
    -> IO ()

type C_WidgetStateChangedCallback =
    Ptr Widget ->                           -- object
    CUInt ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetStateChangedCallback`.
foreign import ccall "wrapper"
    mk_WidgetStateChangedCallback :: C_WidgetStateChangedCallback -> IO (FunPtr C_WidgetStateChangedCallback)

wrap_WidgetStateChangedCallback :: 
    GObject a => (a -> WidgetStateChangedCallback) ->
    C_WidgetStateChangedCallback
wrap_WidgetStateChangedCallback gi'cb gi'selfPtr state _ = do
    let state' = (toEnum . fromIntegral) state
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  state'


-- | Connect a signal handler for the [stateChanged](#signal:stateChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #stateChanged callback
-- @
-- 
-- 
onWidgetStateChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetStateChangedCallback) -> m SignalHandlerId
onWidgetStateChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetStateChangedCallback wrapped
    wrapped'' <- mk_WidgetStateChangedCallback wrapped'
    connectSignalFunPtr obj "state-changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [stateChanged](#signal:stateChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #stateChanged callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetStateChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetStateChangedCallback) -> m SignalHandlerId
afterWidgetStateChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetStateChangedCallback wrapped
    wrapped'' <- mk_WidgetStateChangedCallback wrapped'
    connectSignalFunPtr obj "state-changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetStateChangedSignalInfo
instance SignalInfo WidgetStateChangedSignalInfo where
    type HaskellCallbackType WidgetStateChangedSignalInfo = WidgetStateChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetStateChangedCallback cb
        cb'' <- mk_WidgetStateChangedCallback cb'
        connectSignalFunPtr obj "state-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::state-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:stateChanged"})

#endif

-- signal Widget::state-flags-changed
-- | The [stateFlagsChanged](#g:signal:stateFlagsChanged) signal is emitted when the widget state
-- changes, see 'GI.Gtk.Objects.Widget.widgetGetStateFlags'.
-- 
-- /Since: 3.0/
type WidgetStateFlagsChangedCallback =
    [Gtk.Flags.StateFlags]
    -- ^ /@flags@/: The previous state flags.
    -> IO ()

type C_WidgetStateFlagsChangedCallback =
    Ptr Widget ->                           -- object
    CUInt ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetStateFlagsChangedCallback`.
foreign import ccall "wrapper"
    mk_WidgetStateFlagsChangedCallback :: C_WidgetStateFlagsChangedCallback -> IO (FunPtr C_WidgetStateFlagsChangedCallback)

wrap_WidgetStateFlagsChangedCallback :: 
    GObject a => (a -> WidgetStateFlagsChangedCallback) ->
    C_WidgetStateFlagsChangedCallback
wrap_WidgetStateFlagsChangedCallback gi'cb gi'selfPtr flags _ = do
    let flags' = wordToGFlags flags
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  flags'


-- | Connect a signal handler for the [stateFlagsChanged](#signal:stateFlagsChanged) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #stateFlagsChanged callback
-- @
-- 
-- 
onWidgetStateFlagsChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetStateFlagsChangedCallback) -> m SignalHandlerId
onWidgetStateFlagsChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetStateFlagsChangedCallback wrapped
    wrapped'' <- mk_WidgetStateFlagsChangedCallback wrapped'
    connectSignalFunPtr obj "state-flags-changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [stateFlagsChanged](#signal:stateFlagsChanged) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #stateFlagsChanged callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetStateFlagsChanged :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetStateFlagsChangedCallback) -> m SignalHandlerId
afterWidgetStateFlagsChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetStateFlagsChangedCallback wrapped
    wrapped'' <- mk_WidgetStateFlagsChangedCallback wrapped'
    connectSignalFunPtr obj "state-flags-changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetStateFlagsChangedSignalInfo
instance SignalInfo WidgetStateFlagsChangedSignalInfo where
    type HaskellCallbackType WidgetStateFlagsChangedSignalInfo = WidgetStateFlagsChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetStateFlagsChangedCallback cb
        cb'' <- mk_WidgetStateFlagsChangedCallback cb'
        connectSignalFunPtr obj "state-flags-changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::state-flags-changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:stateFlagsChanged"})

#endif

-- signal Widget::style-set
{-# DEPRECATED WidgetStyleSetCallback ["(Since version 3.0)","Use the [Widget::styleUpdated](\"GI.Gtk.Objects.Widget#g:signal:styleUpdated\") signal"] #-}
-- | The [styleSet](#g:signal:styleSet) signal is emitted when a new style has been set
-- on a widget. Note that style-modifying functions like
-- 'GI.Gtk.Objects.Widget.widgetModifyBase' also cause this signal to be emitted.
-- 
-- Note that this signal is emitted for changes to the deprecated
-- t'GI.Gtk.Objects.Style.Style'. To track changes to the t'GI.Gtk.Objects.StyleContext.StyleContext' associated
-- with a widget, use the [Widget::styleUpdated]("GI.Gtk.Objects.Widget#g:signal:styleUpdated") signal.
type WidgetStyleSetCallback =
    Maybe Gtk.Style.Style
    -- ^ /@previousStyle@/: the previous style, or 'P.Nothing' if the widget
    --   just got its initial style
    -> IO ()

type C_WidgetStyleSetCallback =
    Ptr Widget ->                           -- object
    Ptr Gtk.Style.Style ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetStyleSetCallback`.
foreign import ccall "wrapper"
    mk_WidgetStyleSetCallback :: C_WidgetStyleSetCallback -> IO (FunPtr C_WidgetStyleSetCallback)

wrap_WidgetStyleSetCallback :: 
    GObject a => (a -> WidgetStyleSetCallback) ->
    C_WidgetStyleSetCallback
wrap_WidgetStyleSetCallback gi'cb gi'selfPtr previousStyle _ = do
    maybePreviousStyle <-
        if previousStyle == FP.nullPtr
        then return Nothing
        else do
            previousStyle' <- (newObject Gtk.Style.Style) previousStyle
            return $ Just previousStyle'
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  maybePreviousStyle


-- | Connect a signal handler for the [styleSet](#signal:styleSet) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #styleSet callback
-- @
-- 
-- 
onWidgetStyleSet :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetStyleSetCallback) -> m SignalHandlerId
onWidgetStyleSet obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetStyleSetCallback wrapped
    wrapped'' <- mk_WidgetStyleSetCallback wrapped'
    connectSignalFunPtr obj "style-set" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [styleSet](#signal:styleSet) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #styleSet callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetStyleSet :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetStyleSetCallback) -> m SignalHandlerId
afterWidgetStyleSet obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetStyleSetCallback wrapped
    wrapped'' <- mk_WidgetStyleSetCallback wrapped'
    connectSignalFunPtr obj "style-set" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetStyleSetSignalInfo
instance SignalInfo WidgetStyleSetSignalInfo where
    type HaskellCallbackType WidgetStyleSetSignalInfo = WidgetStyleSetCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetStyleSetCallback cb
        cb'' <- mk_WidgetStyleSetCallback cb'
        connectSignalFunPtr obj "style-set" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::style-set"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:styleSet"})

#endif

-- signal Widget::style-updated
-- | The [styleUpdated](#g:signal:styleUpdated) signal is a convenience signal that is emitted when the
-- [StyleContext::changed]("GI.Gtk.Objects.StyleContext#g:signal:changed") signal is emitted on the /@widget@/\'s associated
-- t'GI.Gtk.Objects.StyleContext.StyleContext' as returned by 'GI.Gtk.Objects.Widget.widgetGetStyleContext'.
-- 
-- Note that style-modifying functions like 'GI.Gtk.Objects.Widget.widgetOverrideColor' also
-- cause this signal to be emitted.
-- 
-- /Since: 3.0/
type WidgetStyleUpdatedCallback =
    IO ()

type C_WidgetStyleUpdatedCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetStyleUpdatedCallback`.
foreign import ccall "wrapper"
    mk_WidgetStyleUpdatedCallback :: C_WidgetStyleUpdatedCallback -> IO (FunPtr C_WidgetStyleUpdatedCallback)

wrap_WidgetStyleUpdatedCallback :: 
    GObject a => (a -> WidgetStyleUpdatedCallback) ->
    C_WidgetStyleUpdatedCallback
wrap_WidgetStyleUpdatedCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [styleUpdated](#signal:styleUpdated) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #styleUpdated callback
-- @
-- 
-- 
onWidgetStyleUpdated :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetStyleUpdatedCallback) -> m SignalHandlerId
onWidgetStyleUpdated obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetStyleUpdatedCallback wrapped
    wrapped'' <- mk_WidgetStyleUpdatedCallback wrapped'
    connectSignalFunPtr obj "style-updated" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [styleUpdated](#signal:styleUpdated) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #styleUpdated callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetStyleUpdated :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetStyleUpdatedCallback) -> m SignalHandlerId
afterWidgetStyleUpdated obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetStyleUpdatedCallback wrapped
    wrapped'' <- mk_WidgetStyleUpdatedCallback wrapped'
    connectSignalFunPtr obj "style-updated" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetStyleUpdatedSignalInfo
instance SignalInfo WidgetStyleUpdatedSignalInfo where
    type HaskellCallbackType WidgetStyleUpdatedSignalInfo = WidgetStyleUpdatedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetStyleUpdatedCallback cb
        cb'' <- mk_WidgetStyleUpdatedCallback cb'
        connectSignalFunPtr obj "style-updated" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::style-updated"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:styleUpdated"})

#endif

-- signal Widget::touch-event
-- | /No description available in the introspection data./
type WidgetTouchEventCallback =
    Gdk.Event.Event
    -> IO Bool

type C_WidgetTouchEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.Event.Event ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetTouchEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetTouchEventCallback :: C_WidgetTouchEventCallback -> IO (FunPtr C_WidgetTouchEventCallback)

wrap_WidgetTouchEventCallback :: 
    GObject a => (a -> WidgetTouchEventCallback) ->
    C_WidgetTouchEventCallback
wrap_WidgetTouchEventCallback gi'cb gi'selfPtr object _ = do
    B.ManagedPtr.withTransient  object $ \object' -> do
        result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  object'
        let result' = (P.fromIntegral . P.fromEnum) result
        return result'


-- | Connect a signal handler for the [touchEvent](#signal:touchEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #touchEvent callback
-- @
-- 
-- 
onWidgetTouchEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetTouchEventCallback) -> m SignalHandlerId
onWidgetTouchEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetTouchEventCallback wrapped
    wrapped'' <- mk_WidgetTouchEventCallback wrapped'
    connectSignalFunPtr obj "touch-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [touchEvent](#signal:touchEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #touchEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetTouchEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetTouchEventCallback) -> m SignalHandlerId
afterWidgetTouchEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetTouchEventCallback wrapped
    wrapped'' <- mk_WidgetTouchEventCallback wrapped'
    connectSignalFunPtr obj "touch-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetTouchEventSignalInfo
instance SignalInfo WidgetTouchEventSignalInfo where
    type HaskellCallbackType WidgetTouchEventSignalInfo = WidgetTouchEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetTouchEventCallback cb
        cb'' <- mk_WidgetTouchEventCallback cb'
        connectSignalFunPtr obj "touch-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::touch-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:touchEvent"})

#endif

-- signal Widget::unmap
-- | The [unmap](#g:signal:unmap) signal is emitted when /@widget@/ is going to be unmapped, which
-- means that either it or any of its parents up to the toplevel widget have
-- been set as hidden.
-- 
-- As [unmap](#g:signal:unmap) indicates that a widget will not be shown any longer, it can be
-- used to, for example, stop an animation on the widget.
type WidgetUnmapCallback =
    IO ()

type C_WidgetUnmapCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetUnmapCallback`.
foreign import ccall "wrapper"
    mk_WidgetUnmapCallback :: C_WidgetUnmapCallback -> IO (FunPtr C_WidgetUnmapCallback)

wrap_WidgetUnmapCallback :: 
    GObject a => (a -> WidgetUnmapCallback) ->
    C_WidgetUnmapCallback
wrap_WidgetUnmapCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [unmap](#signal:unmap) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #unmap callback
-- @
-- 
-- 
onWidgetUnmap :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetUnmapCallback) -> m SignalHandlerId
onWidgetUnmap obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetUnmapCallback wrapped
    wrapped'' <- mk_WidgetUnmapCallback wrapped'
    connectSignalFunPtr obj "unmap" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [unmap](#signal:unmap) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #unmap callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetUnmap :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetUnmapCallback) -> m SignalHandlerId
afterWidgetUnmap obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetUnmapCallback wrapped
    wrapped'' <- mk_WidgetUnmapCallback wrapped'
    connectSignalFunPtr obj "unmap" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetUnmapSignalInfo
instance SignalInfo WidgetUnmapSignalInfo where
    type HaskellCallbackType WidgetUnmapSignalInfo = WidgetUnmapCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetUnmapCallback cb
        cb'' <- mk_WidgetUnmapCallback cb'
        connectSignalFunPtr obj "unmap" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::unmap"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:unmap"})

#endif

-- signal Widget::unmap-event
-- | The [unmapEvent](#g:signal:unmapEvent) signal will be emitted when the /@widget@/\'s window is
-- unmapped. A window is unmapped when it becomes invisible on the screen.
-- 
-- To receive this signal, the t'GI.Gdk.Objects.Window.Window' associated to the widget needs
-- to enable the @/GDK_STRUCTURE_MASK/@ mask. GDK will enable this mask
-- automatically for all new windows.
type WidgetUnmapEventCallback =
    Gdk.EventAny.EventAny
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventAny.EventAny' which triggered this signal
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetUnmapEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventAny.EventAny ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetUnmapEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetUnmapEventCallback :: C_WidgetUnmapEventCallback -> IO (FunPtr C_WidgetUnmapEventCallback)

wrap_WidgetUnmapEventCallback :: 
    GObject a => (a -> WidgetUnmapEventCallback) ->
    C_WidgetUnmapEventCallback
wrap_WidgetUnmapEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventAny.EventAny) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [unmapEvent](#signal:unmapEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #unmapEvent callback
-- @
-- 
-- 
onWidgetUnmapEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetUnmapEventCallback) -> m SignalHandlerId
onWidgetUnmapEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetUnmapEventCallback wrapped
    wrapped'' <- mk_WidgetUnmapEventCallback wrapped'
    connectSignalFunPtr obj "unmap-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [unmapEvent](#signal:unmapEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #unmapEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetUnmapEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetUnmapEventCallback) -> m SignalHandlerId
afterWidgetUnmapEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetUnmapEventCallback wrapped
    wrapped'' <- mk_WidgetUnmapEventCallback wrapped'
    connectSignalFunPtr obj "unmap-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetUnmapEventSignalInfo
instance SignalInfo WidgetUnmapEventSignalInfo where
    type HaskellCallbackType WidgetUnmapEventSignalInfo = WidgetUnmapEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetUnmapEventCallback cb
        cb'' <- mk_WidgetUnmapEventCallback cb'
        connectSignalFunPtr obj "unmap-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::unmap-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:unmapEvent"})

#endif

-- signal Widget::unrealize
-- | The [unrealize](#g:signal:unrealize) signal is emitted when the t'GI.Gdk.Objects.Window.Window' associated with
-- /@widget@/ is destroyed, which means that 'GI.Gtk.Objects.Widget.widgetUnrealize' has been
-- called or the widget has been unmapped (that is, it is going to be
-- hidden).
type WidgetUnrealizeCallback =
    IO ()

type C_WidgetUnrealizeCallback =
    Ptr Widget ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_WidgetUnrealizeCallback`.
foreign import ccall "wrapper"
    mk_WidgetUnrealizeCallback :: C_WidgetUnrealizeCallback -> IO (FunPtr C_WidgetUnrealizeCallback)

wrap_WidgetUnrealizeCallback :: 
    GObject a => (a -> WidgetUnrealizeCallback) ->
    C_WidgetUnrealizeCallback
wrap_WidgetUnrealizeCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [unrealize](#signal:unrealize) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #unrealize callback
-- @
-- 
-- 
onWidgetUnrealize :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetUnrealizeCallback) -> m SignalHandlerId
onWidgetUnrealize obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetUnrealizeCallback wrapped
    wrapped'' <- mk_WidgetUnrealizeCallback wrapped'
    connectSignalFunPtr obj "unrealize" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [unrealize](#signal:unrealize) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #unrealize callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetUnrealize :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetUnrealizeCallback) -> m SignalHandlerId
afterWidgetUnrealize obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetUnrealizeCallback wrapped
    wrapped'' <- mk_WidgetUnrealizeCallback wrapped'
    connectSignalFunPtr obj "unrealize" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetUnrealizeSignalInfo
instance SignalInfo WidgetUnrealizeSignalInfo where
    type HaskellCallbackType WidgetUnrealizeSignalInfo = WidgetUnrealizeCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetUnrealizeCallback cb
        cb'' <- mk_WidgetUnrealizeCallback cb'
        connectSignalFunPtr obj "unrealize" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::unrealize"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:unrealize"})

#endif

-- signal Widget::visibility-notify-event
{-# DEPRECATED WidgetVisibilityNotifyEventCallback ["(Since version 3.12)","Modern composited windowing systems with pervasive","    transparency make it impossible to track the visibility of a window","    reliably, so this signal can not be guaranteed to provide useful","    information."] #-}
-- | The [visibilityNotifyEvent](#g:signal:visibilityNotifyEvent) will be emitted when the /@widget@/\'s
-- window is obscured or unobscured.
-- 
-- To receive this signal the t'GI.Gdk.Objects.Window.Window' associated to the widget needs
-- to enable the @/GDK_VISIBILITY_NOTIFY_MASK/@ mask.
type WidgetVisibilityNotifyEventCallback =
    Gdk.EventVisibility.EventVisibility
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventVisibility.EventVisibility' which
    --   triggered this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the event.
    --   'P.False' to propagate the event further.

type C_WidgetVisibilityNotifyEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventVisibility.EventVisibility ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetVisibilityNotifyEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetVisibilityNotifyEventCallback :: C_WidgetVisibilityNotifyEventCallback -> IO (FunPtr C_WidgetVisibilityNotifyEventCallback)

wrap_WidgetVisibilityNotifyEventCallback :: 
    GObject a => (a -> WidgetVisibilityNotifyEventCallback) ->
    C_WidgetVisibilityNotifyEventCallback
wrap_WidgetVisibilityNotifyEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventVisibility.EventVisibility) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [visibilityNotifyEvent](#signal:visibilityNotifyEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #visibilityNotifyEvent callback
-- @
-- 
-- 
onWidgetVisibilityNotifyEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetVisibilityNotifyEventCallback) -> m SignalHandlerId
onWidgetVisibilityNotifyEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetVisibilityNotifyEventCallback wrapped
    wrapped'' <- mk_WidgetVisibilityNotifyEventCallback wrapped'
    connectSignalFunPtr obj "visibility-notify-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [visibilityNotifyEvent](#signal:visibilityNotifyEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #visibilityNotifyEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetVisibilityNotifyEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetVisibilityNotifyEventCallback) -> m SignalHandlerId
afterWidgetVisibilityNotifyEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetVisibilityNotifyEventCallback wrapped
    wrapped'' <- mk_WidgetVisibilityNotifyEventCallback wrapped'
    connectSignalFunPtr obj "visibility-notify-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetVisibilityNotifyEventSignalInfo
instance SignalInfo WidgetVisibilityNotifyEventSignalInfo where
    type HaskellCallbackType WidgetVisibilityNotifyEventSignalInfo = WidgetVisibilityNotifyEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetVisibilityNotifyEventCallback cb
        cb'' <- mk_WidgetVisibilityNotifyEventCallback cb'
        connectSignalFunPtr obj "visibility-notify-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::visibility-notify-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:visibilityNotifyEvent"})

#endif

-- signal Widget::window-state-event
-- | The [windowStateEvent](#g:signal:windowStateEvent) will be emitted when the state of the
-- toplevel window associated to the /@widget@/ changes.
-- 
-- To receive this signal the t'GI.Gdk.Objects.Window.Window' associated to the widget
-- needs to enable the @/GDK_STRUCTURE_MASK/@ mask. GDK will enable
-- this mask automatically for all new windows.
type WidgetWindowStateEventCallback =
    Gdk.EventWindowState.EventWindowState
    -- ^ /@event@/: the t'GI.Gdk.Structs.EventWindowState.EventWindowState' which
    --   triggered this signal.
    -> IO Bool
    -- ^ __Returns:__ 'P.True' to stop other handlers from being invoked for the
    --   event. 'P.False' to propagate the event further.

type C_WidgetWindowStateEventCallback =
    Ptr Widget ->                           -- object
    Ptr Gdk.EventWindowState.EventWindowState ->
    Ptr () ->                               -- user_data
    IO CInt

-- | Generate a function pointer callable from C code, from a `C_WidgetWindowStateEventCallback`.
foreign import ccall "wrapper"
    mk_WidgetWindowStateEventCallback :: C_WidgetWindowStateEventCallback -> IO (FunPtr C_WidgetWindowStateEventCallback)

wrap_WidgetWindowStateEventCallback :: 
    GObject a => (a -> WidgetWindowStateEventCallback) ->
    C_WidgetWindowStateEventCallback
wrap_WidgetWindowStateEventCallback gi'cb gi'selfPtr event _ = do
    event' <- (newPtr Gdk.EventWindowState.EventWindowState) event
    result <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  event'
    let result' = (P.fromIntegral . P.fromEnum) result
    return result'


-- | Connect a signal handler for the [windowStateEvent](#signal:windowStateEvent) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' widget #windowStateEvent callback
-- @
-- 
-- 
onWidgetWindowStateEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetWindowStateEventCallback) -> m SignalHandlerId
onWidgetWindowStateEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetWindowStateEventCallback wrapped
    wrapped'' <- mk_WidgetWindowStateEventCallback wrapped'
    connectSignalFunPtr obj "window-state-event" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [windowStateEvent](#signal:windowStateEvent) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' widget #windowStateEvent callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterWidgetWindowStateEvent :: (IsWidget a, MonadIO m) => a -> ((?self :: a) => WidgetWindowStateEventCallback) -> m SignalHandlerId
afterWidgetWindowStateEvent obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_WidgetWindowStateEventCallback wrapped
    wrapped'' <- mk_WidgetWindowStateEventCallback wrapped'
    connectSignalFunPtr obj "window-state-event" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data WidgetWindowStateEventSignalInfo
instance SignalInfo WidgetWindowStateEventSignalInfo where
    type HaskellCallbackType WidgetWindowStateEventSignalInfo = WidgetWindowStateEventCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_WidgetWindowStateEventCallback cb
        cb'' <- mk_WidgetWindowStateEventCallback cb'
        connectSignalFunPtr obj "window-state-event" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget::window-state-event"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:signal:windowStateEvent"})

#endif

-- VVV Prop "app-paintable"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@app-paintable@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #appPaintable
-- @
getWidgetAppPaintable :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetAppPaintable obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "app-paintable"

-- | Set the value of the “@app-paintable@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #appPaintable 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetAppPaintable :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetAppPaintable obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "app-paintable" val

-- | Construct a t'GValueConstruct' with valid value for the “@app-paintable@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetAppPaintable :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetAppPaintable val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "app-paintable" val

#if defined(ENABLE_OVERLOADING)
data WidgetAppPaintablePropertyInfo
instance AttrInfo WidgetAppPaintablePropertyInfo where
    type AttrAllowedOps WidgetAppPaintablePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetAppPaintablePropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetAppPaintablePropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetAppPaintablePropertyInfo = (~) Bool
    type AttrTransferType WidgetAppPaintablePropertyInfo = Bool
    type AttrGetType WidgetAppPaintablePropertyInfo = Bool
    type AttrLabel WidgetAppPaintablePropertyInfo = "app-paintable"
    type AttrOrigin WidgetAppPaintablePropertyInfo = Widget
    attrGet = getWidgetAppPaintable
    attrSet = setWidgetAppPaintable
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetAppPaintable
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.appPaintable"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:appPaintable"
        })
#endif

-- VVV Prop "can-default"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@can-default@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #canDefault
-- @
getWidgetCanDefault :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetCanDefault obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "can-default"

-- | Set the value of the “@can-default@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #canDefault 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetCanDefault :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetCanDefault obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "can-default" val

-- | Construct a t'GValueConstruct' with valid value for the “@can-default@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetCanDefault :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetCanDefault val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "can-default" val

#if defined(ENABLE_OVERLOADING)
data WidgetCanDefaultPropertyInfo
instance AttrInfo WidgetCanDefaultPropertyInfo where
    type AttrAllowedOps WidgetCanDefaultPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetCanDefaultPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetCanDefaultPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetCanDefaultPropertyInfo = (~) Bool
    type AttrTransferType WidgetCanDefaultPropertyInfo = Bool
    type AttrGetType WidgetCanDefaultPropertyInfo = Bool
    type AttrLabel WidgetCanDefaultPropertyInfo = "can-default"
    type AttrOrigin WidgetCanDefaultPropertyInfo = Widget
    attrGet = getWidgetCanDefault
    attrSet = setWidgetCanDefault
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetCanDefault
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.canDefault"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:canDefault"
        })
#endif

-- VVV Prop "can-focus"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@can-focus@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #canFocus
-- @
getWidgetCanFocus :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetCanFocus obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "can-focus"

-- | Set the value of the “@can-focus@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #canFocus 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetCanFocus :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetCanFocus obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "can-focus" val

-- | Construct a t'GValueConstruct' with valid value for the “@can-focus@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetCanFocus :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetCanFocus val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "can-focus" val

#if defined(ENABLE_OVERLOADING)
data WidgetCanFocusPropertyInfo
instance AttrInfo WidgetCanFocusPropertyInfo where
    type AttrAllowedOps WidgetCanFocusPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetCanFocusPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetCanFocusPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetCanFocusPropertyInfo = (~) Bool
    type AttrTransferType WidgetCanFocusPropertyInfo = Bool
    type AttrGetType WidgetCanFocusPropertyInfo = Bool
    type AttrLabel WidgetCanFocusPropertyInfo = "can-focus"
    type AttrOrigin WidgetCanFocusPropertyInfo = Widget
    attrGet = getWidgetCanFocus
    attrSet = setWidgetCanFocus
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetCanFocus
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.canFocus"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:canFocus"
        })
#endif

-- VVV Prop "composite-child"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@composite-child@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #compositeChild
-- @
getWidgetCompositeChild :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetCompositeChild obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "composite-child"

#if defined(ENABLE_OVERLOADING)
data WidgetCompositeChildPropertyInfo
instance AttrInfo WidgetCompositeChildPropertyInfo where
    type AttrAllowedOps WidgetCompositeChildPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint WidgetCompositeChildPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetCompositeChildPropertyInfo = (~) ()
    type AttrTransferTypeConstraint WidgetCompositeChildPropertyInfo = (~) ()
    type AttrTransferType WidgetCompositeChildPropertyInfo = ()
    type AttrGetType WidgetCompositeChildPropertyInfo = Bool
    type AttrLabel WidgetCompositeChildPropertyInfo = "composite-child"
    type AttrOrigin WidgetCompositeChildPropertyInfo = Widget
    attrGet = getWidgetCompositeChild
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.compositeChild"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:compositeChild"
        })
#endif

-- VVV Prop "double-buffered"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@double-buffered@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #doubleBuffered
-- @
getWidgetDoubleBuffered :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetDoubleBuffered obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "double-buffered"

-- | Set the value of the “@double-buffered@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #doubleBuffered 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetDoubleBuffered :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetDoubleBuffered obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "double-buffered" val

-- | Construct a t'GValueConstruct' with valid value for the “@double-buffered@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetDoubleBuffered :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetDoubleBuffered val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "double-buffered" val

#if defined(ENABLE_OVERLOADING)
data WidgetDoubleBufferedPropertyInfo
instance AttrInfo WidgetDoubleBufferedPropertyInfo where
    type AttrAllowedOps WidgetDoubleBufferedPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetDoubleBufferedPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetDoubleBufferedPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetDoubleBufferedPropertyInfo = (~) Bool
    type AttrTransferType WidgetDoubleBufferedPropertyInfo = Bool
    type AttrGetType WidgetDoubleBufferedPropertyInfo = Bool
    type AttrLabel WidgetDoubleBufferedPropertyInfo = "double-buffered"
    type AttrOrigin WidgetDoubleBufferedPropertyInfo = Widget
    attrGet = getWidgetDoubleBuffered
    attrSet = setWidgetDoubleBuffered
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetDoubleBuffered
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.doubleBuffered"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:doubleBuffered"
        })
#endif

-- VVV Prop "events"
   -- Type: TInterface (Name {namespace = "Gdk", name = "EventMask"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Just False)

-- | Get the value of the “@events@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #events
-- @
getWidgetEvents :: (MonadIO m, IsWidget o) => o -> m [Gdk.Flags.EventMask]
getWidgetEvents obj = MIO.liftIO $ B.Properties.getObjectPropertyFlags obj "events"

-- | Set the value of the “@events@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #events 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetEvents :: (MonadIO m, IsWidget o) => o -> [Gdk.Flags.EventMask] -> m ()
setWidgetEvents obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyFlags obj "events" val

-- | Construct a t'GValueConstruct' with valid value for the “@events@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetEvents :: (IsWidget o, MIO.MonadIO m) => [Gdk.Flags.EventMask] -> m (GValueConstruct o)
constructWidgetEvents val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyFlags "events" val

#if defined(ENABLE_OVERLOADING)
data WidgetEventsPropertyInfo
instance AttrInfo WidgetEventsPropertyInfo where
    type AttrAllowedOps WidgetEventsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetEventsPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetEventsPropertyInfo = (~) [Gdk.Flags.EventMask]
    type AttrTransferTypeConstraint WidgetEventsPropertyInfo = (~) [Gdk.Flags.EventMask]
    type AttrTransferType WidgetEventsPropertyInfo = [Gdk.Flags.EventMask]
    type AttrGetType WidgetEventsPropertyInfo = [Gdk.Flags.EventMask]
    type AttrLabel WidgetEventsPropertyInfo = "events"
    type AttrOrigin WidgetEventsPropertyInfo = Widget
    attrGet = getWidgetEvents
    attrSet = setWidgetEvents
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetEvents
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.events"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:events"
        })
#endif

-- VVV Prop "expand"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@expand@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #expand
-- @
getWidgetExpand :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetExpand obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "expand"

-- | Set the value of the “@expand@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #expand 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetExpand :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetExpand obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "expand" val

-- | Construct a t'GValueConstruct' with valid value for the “@expand@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetExpand :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetExpand val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "expand" val

#if defined(ENABLE_OVERLOADING)
data WidgetExpandPropertyInfo
instance AttrInfo WidgetExpandPropertyInfo where
    type AttrAllowedOps WidgetExpandPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetExpandPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetExpandPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetExpandPropertyInfo = (~) Bool
    type AttrTransferType WidgetExpandPropertyInfo = Bool
    type AttrGetType WidgetExpandPropertyInfo = Bool
    type AttrLabel WidgetExpandPropertyInfo = "expand"
    type AttrOrigin WidgetExpandPropertyInfo = Widget
    attrGet = getWidgetExpand
    attrSet = setWidgetExpand
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetExpand
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.expand"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:expand"
        })
#endif

-- VVV Prop "focus-on-click"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@focus-on-click@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #focusOnClick
-- @
getWidgetFocusOnClick :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetFocusOnClick obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "focus-on-click"

-- | Set the value of the “@focus-on-click@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #focusOnClick 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetFocusOnClick :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetFocusOnClick obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "focus-on-click" val

-- | Construct a t'GValueConstruct' with valid value for the “@focus-on-click@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetFocusOnClick :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetFocusOnClick val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "focus-on-click" val

#if defined(ENABLE_OVERLOADING)
data WidgetFocusOnClickPropertyInfo
instance AttrInfo WidgetFocusOnClickPropertyInfo where
    type AttrAllowedOps WidgetFocusOnClickPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetFocusOnClickPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetFocusOnClickPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetFocusOnClickPropertyInfo = (~) Bool
    type AttrTransferType WidgetFocusOnClickPropertyInfo = Bool
    type AttrGetType WidgetFocusOnClickPropertyInfo = Bool
    type AttrLabel WidgetFocusOnClickPropertyInfo = "focus-on-click"
    type AttrOrigin WidgetFocusOnClickPropertyInfo = Widget
    attrGet = getWidgetFocusOnClick
    attrSet = setWidgetFocusOnClick
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetFocusOnClick
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.focusOnClick"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:focusOnClick"
        })
#endif

-- VVV Prop "halign"
   -- Type: TInterface (Name {namespace = "Gtk", name = "Align"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@halign@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #halign
-- @
getWidgetHalign :: (MonadIO m, IsWidget o) => o -> m Gtk.Enums.Align
getWidgetHalign obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "halign"

-- | Set the value of the “@halign@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #halign 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetHalign :: (MonadIO m, IsWidget o) => o -> Gtk.Enums.Align -> m ()
setWidgetHalign obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "halign" val

-- | Construct a t'GValueConstruct' with valid value for the “@halign@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetHalign :: (IsWidget o, MIO.MonadIO m) => Gtk.Enums.Align -> m (GValueConstruct o)
constructWidgetHalign val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "halign" val

#if defined(ENABLE_OVERLOADING)
data WidgetHalignPropertyInfo
instance AttrInfo WidgetHalignPropertyInfo where
    type AttrAllowedOps WidgetHalignPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetHalignPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetHalignPropertyInfo = (~) Gtk.Enums.Align
    type AttrTransferTypeConstraint WidgetHalignPropertyInfo = (~) Gtk.Enums.Align
    type AttrTransferType WidgetHalignPropertyInfo = Gtk.Enums.Align
    type AttrGetType WidgetHalignPropertyInfo = Gtk.Enums.Align
    type AttrLabel WidgetHalignPropertyInfo = "halign"
    type AttrOrigin WidgetHalignPropertyInfo = Widget
    attrGet = getWidgetHalign
    attrSet = setWidgetHalign
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetHalign
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.halign"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:halign"
        })
#endif

-- VVV Prop "has-default"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@has-default@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #hasDefault
-- @
getWidgetHasDefault :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetHasDefault obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "has-default"

-- | Set the value of the “@has-default@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #hasDefault 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetHasDefault :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetHasDefault obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "has-default" val

-- | Construct a t'GValueConstruct' with valid value for the “@has-default@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetHasDefault :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetHasDefault val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "has-default" val

#if defined(ENABLE_OVERLOADING)
data WidgetHasDefaultPropertyInfo
instance AttrInfo WidgetHasDefaultPropertyInfo where
    type AttrAllowedOps WidgetHasDefaultPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetHasDefaultPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetHasDefaultPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetHasDefaultPropertyInfo = (~) Bool
    type AttrTransferType WidgetHasDefaultPropertyInfo = Bool
    type AttrGetType WidgetHasDefaultPropertyInfo = Bool
    type AttrLabel WidgetHasDefaultPropertyInfo = "has-default"
    type AttrOrigin WidgetHasDefaultPropertyInfo = Widget
    attrGet = getWidgetHasDefault
    attrSet = setWidgetHasDefault
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetHasDefault
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.hasDefault"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:hasDefault"
        })
#endif

-- VVV Prop "has-focus"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@has-focus@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #hasFocus
-- @
getWidgetHasFocus :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetHasFocus obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "has-focus"

-- | Set the value of the “@has-focus@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #hasFocus 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetHasFocus :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetHasFocus obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "has-focus" val

-- | Construct a t'GValueConstruct' with valid value for the “@has-focus@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetHasFocus :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetHasFocus val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "has-focus" val

#if defined(ENABLE_OVERLOADING)
data WidgetHasFocusPropertyInfo
instance AttrInfo WidgetHasFocusPropertyInfo where
    type AttrAllowedOps WidgetHasFocusPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetHasFocusPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetHasFocusPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetHasFocusPropertyInfo = (~) Bool
    type AttrTransferType WidgetHasFocusPropertyInfo = Bool
    type AttrGetType WidgetHasFocusPropertyInfo = Bool
    type AttrLabel WidgetHasFocusPropertyInfo = "has-focus"
    type AttrOrigin WidgetHasFocusPropertyInfo = Widget
    attrGet = getWidgetHasFocus
    attrSet = setWidgetHasFocus
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetHasFocus
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.hasFocus"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:hasFocus"
        })
#endif

-- VVV Prop "has-tooltip"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@has-tooltip@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #hasTooltip
-- @
getWidgetHasTooltip :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetHasTooltip obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "has-tooltip"

-- | Set the value of the “@has-tooltip@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #hasTooltip 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetHasTooltip :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetHasTooltip obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "has-tooltip" val

-- | Construct a t'GValueConstruct' with valid value for the “@has-tooltip@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetHasTooltip :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetHasTooltip val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "has-tooltip" val

#if defined(ENABLE_OVERLOADING)
data WidgetHasTooltipPropertyInfo
instance AttrInfo WidgetHasTooltipPropertyInfo where
    type AttrAllowedOps WidgetHasTooltipPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetHasTooltipPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetHasTooltipPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetHasTooltipPropertyInfo = (~) Bool
    type AttrTransferType WidgetHasTooltipPropertyInfo = Bool
    type AttrGetType WidgetHasTooltipPropertyInfo = Bool
    type AttrLabel WidgetHasTooltipPropertyInfo = "has-tooltip"
    type AttrOrigin WidgetHasTooltipPropertyInfo = Widget
    attrGet = getWidgetHasTooltip
    attrSet = setWidgetHasTooltip
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetHasTooltip
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.hasTooltip"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:hasTooltip"
        })
#endif

-- VVV Prop "height-request"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@height-request@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #heightRequest
-- @
getWidgetHeightRequest :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetHeightRequest obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "height-request"

-- | Set the value of the “@height-request@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #heightRequest 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetHeightRequest :: (MonadIO m, IsWidget o) => o -> Int32 -> m ()
setWidgetHeightRequest obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "height-request" val

-- | Construct a t'GValueConstruct' with valid value for the “@height-request@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetHeightRequest :: (IsWidget o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructWidgetHeightRequest val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "height-request" val

#if defined(ENABLE_OVERLOADING)
data WidgetHeightRequestPropertyInfo
instance AttrInfo WidgetHeightRequestPropertyInfo where
    type AttrAllowedOps WidgetHeightRequestPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetHeightRequestPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetHeightRequestPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint WidgetHeightRequestPropertyInfo = (~) Int32
    type AttrTransferType WidgetHeightRequestPropertyInfo = Int32
    type AttrGetType WidgetHeightRequestPropertyInfo = Int32
    type AttrLabel WidgetHeightRequestPropertyInfo = "height-request"
    type AttrOrigin WidgetHeightRequestPropertyInfo = Widget
    attrGet = getWidgetHeightRequest
    attrSet = setWidgetHeightRequest
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetHeightRequest
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.heightRequest"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:heightRequest"
        })
#endif

-- VVV Prop "hexpand"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@hexpand@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #hexpand
-- @
getWidgetHexpand :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetHexpand obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "hexpand"

-- | Set the value of the “@hexpand@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #hexpand 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetHexpand :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetHexpand obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "hexpand" val

-- | Construct a t'GValueConstruct' with valid value for the “@hexpand@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetHexpand :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetHexpand val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "hexpand" val

#if defined(ENABLE_OVERLOADING)
data WidgetHexpandPropertyInfo
instance AttrInfo WidgetHexpandPropertyInfo where
    type AttrAllowedOps WidgetHexpandPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetHexpandPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetHexpandPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetHexpandPropertyInfo = (~) Bool
    type AttrTransferType WidgetHexpandPropertyInfo = Bool
    type AttrGetType WidgetHexpandPropertyInfo = Bool
    type AttrLabel WidgetHexpandPropertyInfo = "hexpand"
    type AttrOrigin WidgetHexpandPropertyInfo = Widget
    attrGet = getWidgetHexpand
    attrSet = setWidgetHexpand
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetHexpand
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.hexpand"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:hexpand"
        })
#endif

-- VVV Prop "hexpand-set"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@hexpand-set@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #hexpandSet
-- @
getWidgetHexpandSet :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetHexpandSet obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "hexpand-set"

-- | Set the value of the “@hexpand-set@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #hexpandSet 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetHexpandSet :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetHexpandSet obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "hexpand-set" val

-- | Construct a t'GValueConstruct' with valid value for the “@hexpand-set@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetHexpandSet :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetHexpandSet val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "hexpand-set" val

#if defined(ENABLE_OVERLOADING)
data WidgetHexpandSetPropertyInfo
instance AttrInfo WidgetHexpandSetPropertyInfo where
    type AttrAllowedOps WidgetHexpandSetPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetHexpandSetPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetHexpandSetPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetHexpandSetPropertyInfo = (~) Bool
    type AttrTransferType WidgetHexpandSetPropertyInfo = Bool
    type AttrGetType WidgetHexpandSetPropertyInfo = Bool
    type AttrLabel WidgetHexpandSetPropertyInfo = "hexpand-set"
    type AttrOrigin WidgetHexpandSetPropertyInfo = Widget
    attrGet = getWidgetHexpandSet
    attrSet = setWidgetHexpandSet
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetHexpandSet
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.hexpandSet"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:hexpandSet"
        })
#endif

-- VVV Prop "is-focus"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@is-focus@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #isFocus
-- @
getWidgetIsFocus :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetIsFocus obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "is-focus"

-- | Set the value of the “@is-focus@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #isFocus 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetIsFocus :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetIsFocus obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "is-focus" val

-- | Construct a t'GValueConstruct' with valid value for the “@is-focus@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetIsFocus :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetIsFocus val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "is-focus" val

#if defined(ENABLE_OVERLOADING)
data WidgetIsFocusPropertyInfo
instance AttrInfo WidgetIsFocusPropertyInfo where
    type AttrAllowedOps WidgetIsFocusPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetIsFocusPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetIsFocusPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetIsFocusPropertyInfo = (~) Bool
    type AttrTransferType WidgetIsFocusPropertyInfo = Bool
    type AttrGetType WidgetIsFocusPropertyInfo = Bool
    type AttrLabel WidgetIsFocusPropertyInfo = "is-focus"
    type AttrOrigin WidgetIsFocusPropertyInfo = Widget
    attrGet = getWidgetIsFocus
    attrSet = setWidgetIsFocus
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetIsFocus
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.isFocus"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:isFocus"
        })
#endif

-- VVV Prop "margin"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@margin@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #margin
-- @
getWidgetMargin :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetMargin obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "margin"

-- | Set the value of the “@margin@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #margin 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetMargin :: (MonadIO m, IsWidget o) => o -> Int32 -> m ()
setWidgetMargin obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "margin" val

-- | Construct a t'GValueConstruct' with valid value for the “@margin@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetMargin :: (IsWidget o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructWidgetMargin val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "margin" val

#if defined(ENABLE_OVERLOADING)
data WidgetMarginPropertyInfo
instance AttrInfo WidgetMarginPropertyInfo where
    type AttrAllowedOps WidgetMarginPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetMarginPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetMarginPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint WidgetMarginPropertyInfo = (~) Int32
    type AttrTransferType WidgetMarginPropertyInfo = Int32
    type AttrGetType WidgetMarginPropertyInfo = Int32
    type AttrLabel WidgetMarginPropertyInfo = "margin"
    type AttrOrigin WidgetMarginPropertyInfo = Widget
    attrGet = getWidgetMargin
    attrSet = setWidgetMargin
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetMargin
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.margin"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:margin"
        })
#endif

-- VVV Prop "margin-bottom"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@margin-bottom@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #marginBottom
-- @
getWidgetMarginBottom :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetMarginBottom obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "margin-bottom"

-- | Set the value of the “@margin-bottom@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #marginBottom 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetMarginBottom :: (MonadIO m, IsWidget o) => o -> Int32 -> m ()
setWidgetMarginBottom obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "margin-bottom" val

-- | Construct a t'GValueConstruct' with valid value for the “@margin-bottom@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetMarginBottom :: (IsWidget o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructWidgetMarginBottom val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "margin-bottom" val

#if defined(ENABLE_OVERLOADING)
data WidgetMarginBottomPropertyInfo
instance AttrInfo WidgetMarginBottomPropertyInfo where
    type AttrAllowedOps WidgetMarginBottomPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetMarginBottomPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetMarginBottomPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint WidgetMarginBottomPropertyInfo = (~) Int32
    type AttrTransferType WidgetMarginBottomPropertyInfo = Int32
    type AttrGetType WidgetMarginBottomPropertyInfo = Int32
    type AttrLabel WidgetMarginBottomPropertyInfo = "margin-bottom"
    type AttrOrigin WidgetMarginBottomPropertyInfo = Widget
    attrGet = getWidgetMarginBottom
    attrSet = setWidgetMarginBottom
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetMarginBottom
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.marginBottom"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:marginBottom"
        })
#endif

-- VVV Prop "margin-end"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@margin-end@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #marginEnd
-- @
getWidgetMarginEnd :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetMarginEnd obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "margin-end"

-- | Set the value of the “@margin-end@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #marginEnd 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetMarginEnd :: (MonadIO m, IsWidget o) => o -> Int32 -> m ()
setWidgetMarginEnd obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "margin-end" val

-- | Construct a t'GValueConstruct' with valid value for the “@margin-end@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetMarginEnd :: (IsWidget o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructWidgetMarginEnd val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "margin-end" val

#if defined(ENABLE_OVERLOADING)
data WidgetMarginEndPropertyInfo
instance AttrInfo WidgetMarginEndPropertyInfo where
    type AttrAllowedOps WidgetMarginEndPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetMarginEndPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetMarginEndPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint WidgetMarginEndPropertyInfo = (~) Int32
    type AttrTransferType WidgetMarginEndPropertyInfo = Int32
    type AttrGetType WidgetMarginEndPropertyInfo = Int32
    type AttrLabel WidgetMarginEndPropertyInfo = "margin-end"
    type AttrOrigin WidgetMarginEndPropertyInfo = Widget
    attrGet = getWidgetMarginEnd
    attrSet = setWidgetMarginEnd
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetMarginEnd
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.marginEnd"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:marginEnd"
        })
#endif

-- VVV Prop "margin-left"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@margin-left@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #marginLeft
-- @
getWidgetMarginLeft :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetMarginLeft obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "margin-left"

-- | Set the value of the “@margin-left@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #marginLeft 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetMarginLeft :: (MonadIO m, IsWidget o) => o -> Int32 -> m ()
setWidgetMarginLeft obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "margin-left" val

-- | Construct a t'GValueConstruct' with valid value for the “@margin-left@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetMarginLeft :: (IsWidget o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructWidgetMarginLeft val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "margin-left" val

#if defined(ENABLE_OVERLOADING)
data WidgetMarginLeftPropertyInfo
instance AttrInfo WidgetMarginLeftPropertyInfo where
    type AttrAllowedOps WidgetMarginLeftPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetMarginLeftPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetMarginLeftPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint WidgetMarginLeftPropertyInfo = (~) Int32
    type AttrTransferType WidgetMarginLeftPropertyInfo = Int32
    type AttrGetType WidgetMarginLeftPropertyInfo = Int32
    type AttrLabel WidgetMarginLeftPropertyInfo = "margin-left"
    type AttrOrigin WidgetMarginLeftPropertyInfo = Widget
    attrGet = getWidgetMarginLeft
    attrSet = setWidgetMarginLeft
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetMarginLeft
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.marginLeft"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:marginLeft"
        })
#endif

-- VVV Prop "margin-right"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@margin-right@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #marginRight
-- @
getWidgetMarginRight :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetMarginRight obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "margin-right"

-- | Set the value of the “@margin-right@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #marginRight 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetMarginRight :: (MonadIO m, IsWidget o) => o -> Int32 -> m ()
setWidgetMarginRight obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "margin-right" val

-- | Construct a t'GValueConstruct' with valid value for the “@margin-right@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetMarginRight :: (IsWidget o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructWidgetMarginRight val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "margin-right" val

#if defined(ENABLE_OVERLOADING)
data WidgetMarginRightPropertyInfo
instance AttrInfo WidgetMarginRightPropertyInfo where
    type AttrAllowedOps WidgetMarginRightPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetMarginRightPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetMarginRightPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint WidgetMarginRightPropertyInfo = (~) Int32
    type AttrTransferType WidgetMarginRightPropertyInfo = Int32
    type AttrGetType WidgetMarginRightPropertyInfo = Int32
    type AttrLabel WidgetMarginRightPropertyInfo = "margin-right"
    type AttrOrigin WidgetMarginRightPropertyInfo = Widget
    attrGet = getWidgetMarginRight
    attrSet = setWidgetMarginRight
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetMarginRight
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.marginRight"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:marginRight"
        })
#endif

-- VVV Prop "margin-start"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@margin-start@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #marginStart
-- @
getWidgetMarginStart :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetMarginStart obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "margin-start"

-- | Set the value of the “@margin-start@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #marginStart 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetMarginStart :: (MonadIO m, IsWidget o) => o -> Int32 -> m ()
setWidgetMarginStart obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "margin-start" val

-- | Construct a t'GValueConstruct' with valid value for the “@margin-start@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetMarginStart :: (IsWidget o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructWidgetMarginStart val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "margin-start" val

#if defined(ENABLE_OVERLOADING)
data WidgetMarginStartPropertyInfo
instance AttrInfo WidgetMarginStartPropertyInfo where
    type AttrAllowedOps WidgetMarginStartPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetMarginStartPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetMarginStartPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint WidgetMarginStartPropertyInfo = (~) Int32
    type AttrTransferType WidgetMarginStartPropertyInfo = Int32
    type AttrGetType WidgetMarginStartPropertyInfo = Int32
    type AttrLabel WidgetMarginStartPropertyInfo = "margin-start"
    type AttrOrigin WidgetMarginStartPropertyInfo = Widget
    attrGet = getWidgetMarginStart
    attrSet = setWidgetMarginStart
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetMarginStart
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.marginStart"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:marginStart"
        })
#endif

-- VVV Prop "margin-top"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@margin-top@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #marginTop
-- @
getWidgetMarginTop :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetMarginTop obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "margin-top"

-- | Set the value of the “@margin-top@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #marginTop 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetMarginTop :: (MonadIO m, IsWidget o) => o -> Int32 -> m ()
setWidgetMarginTop obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "margin-top" val

-- | Construct a t'GValueConstruct' with valid value for the “@margin-top@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetMarginTop :: (IsWidget o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructWidgetMarginTop val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "margin-top" val

#if defined(ENABLE_OVERLOADING)
data WidgetMarginTopPropertyInfo
instance AttrInfo WidgetMarginTopPropertyInfo where
    type AttrAllowedOps WidgetMarginTopPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetMarginTopPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetMarginTopPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint WidgetMarginTopPropertyInfo = (~) Int32
    type AttrTransferType WidgetMarginTopPropertyInfo = Int32
    type AttrGetType WidgetMarginTopPropertyInfo = Int32
    type AttrLabel WidgetMarginTopPropertyInfo = "margin-top"
    type AttrOrigin WidgetMarginTopPropertyInfo = Widget
    attrGet = getWidgetMarginTop
    attrSet = setWidgetMarginTop
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetMarginTop
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.marginTop"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:marginTop"
        })
#endif

-- VVV Prop "name"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #name
-- @
getWidgetName :: (MonadIO m, IsWidget o) => o -> m T.Text
getWidgetName obj = MIO.liftIO $ checkUnexpectedNothing "getWidgetName" $ B.Properties.getObjectPropertyString obj "name"

-- | Set the value of the “@name@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #name 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetName :: (MonadIO m, IsWidget o) => o -> T.Text -> m ()
setWidgetName obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "name" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@name@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetName :: (IsWidget o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructWidgetName val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "name" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data WidgetNamePropertyInfo
instance AttrInfo WidgetNamePropertyInfo where
    type AttrAllowedOps WidgetNamePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetNamePropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetNamePropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint WidgetNamePropertyInfo = (~) T.Text
    type AttrTransferType WidgetNamePropertyInfo = T.Text
    type AttrGetType WidgetNamePropertyInfo = T.Text
    type AttrLabel WidgetNamePropertyInfo = "name"
    type AttrOrigin WidgetNamePropertyInfo = Widget
    attrGet = getWidgetName
    attrSet = setWidgetName
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetName
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.name"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:name"
        })
#endif

-- VVV Prop "no-show-all"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@no-show-all@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #noShowAll
-- @
getWidgetNoShowAll :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetNoShowAll obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "no-show-all"

-- | Set the value of the “@no-show-all@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #noShowAll 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetNoShowAll :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetNoShowAll obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "no-show-all" val

-- | Construct a t'GValueConstruct' with valid value for the “@no-show-all@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetNoShowAll :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetNoShowAll val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "no-show-all" val

#if defined(ENABLE_OVERLOADING)
data WidgetNoShowAllPropertyInfo
instance AttrInfo WidgetNoShowAllPropertyInfo where
    type AttrAllowedOps WidgetNoShowAllPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetNoShowAllPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetNoShowAllPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetNoShowAllPropertyInfo = (~) Bool
    type AttrTransferType WidgetNoShowAllPropertyInfo = Bool
    type AttrGetType WidgetNoShowAllPropertyInfo = Bool
    type AttrLabel WidgetNoShowAllPropertyInfo = "no-show-all"
    type AttrOrigin WidgetNoShowAllPropertyInfo = Widget
    attrGet = getWidgetNoShowAll
    attrSet = setWidgetNoShowAll
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetNoShowAll
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.noShowAll"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:noShowAll"
        })
#endif

-- VVV Prop "opacity"
   -- Type: TBasicType TDouble
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@opacity@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #opacity
-- @
getWidgetOpacity :: (MonadIO m, IsWidget o) => o -> m Double
getWidgetOpacity obj = MIO.liftIO $ B.Properties.getObjectPropertyDouble obj "opacity"

-- | Set the value of the “@opacity@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #opacity 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetOpacity :: (MonadIO m, IsWidget o) => o -> Double -> m ()
setWidgetOpacity obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyDouble obj "opacity" val

-- | Construct a t'GValueConstruct' with valid value for the “@opacity@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetOpacity :: (IsWidget o, MIO.MonadIO m) => Double -> m (GValueConstruct o)
constructWidgetOpacity val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyDouble "opacity" val

#if defined(ENABLE_OVERLOADING)
data WidgetOpacityPropertyInfo
instance AttrInfo WidgetOpacityPropertyInfo where
    type AttrAllowedOps WidgetOpacityPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetOpacityPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetOpacityPropertyInfo = (~) Double
    type AttrTransferTypeConstraint WidgetOpacityPropertyInfo = (~) Double
    type AttrTransferType WidgetOpacityPropertyInfo = Double
    type AttrGetType WidgetOpacityPropertyInfo = Double
    type AttrLabel WidgetOpacityPropertyInfo = "opacity"
    type AttrOrigin WidgetOpacityPropertyInfo = Widget
    attrGet = getWidgetOpacity
    attrSet = setWidgetOpacity
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetOpacity
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.opacity"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:opacity"
        })
#endif

-- VVV Prop "parent"
   -- Type: TInterface (Name {namespace = "Gtk", name = "Container"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@parent@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #parent
-- @
getWidgetParent :: (MonadIO m, IsWidget o) => o -> m (Maybe Gtk.Container.Container)
getWidgetParent obj = MIO.liftIO $ B.Properties.getObjectPropertyObject obj "parent" Gtk.Container.Container

-- | Set the value of the “@parent@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #parent 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetParent :: (MonadIO m, IsWidget o, Gtk.Container.IsContainer a) => o -> a -> m ()
setWidgetParent obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyObject obj "parent" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@parent@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetParent :: (IsWidget o, MIO.MonadIO m, Gtk.Container.IsContainer a) => a -> m (GValueConstruct o)
constructWidgetParent val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyObject "parent" (P.Just val)

-- | Set the value of the “@parent@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #parent
-- @
clearWidgetParent :: (MonadIO m, IsWidget o) => o -> m ()
clearWidgetParent obj = liftIO $ B.Properties.setObjectPropertyObject obj "parent" (Nothing :: Maybe Gtk.Container.Container)

#if defined(ENABLE_OVERLOADING)
data WidgetParentPropertyInfo
instance AttrInfo WidgetParentPropertyInfo where
    type AttrAllowedOps WidgetParentPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint WidgetParentPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetParentPropertyInfo = Gtk.Container.IsContainer
    type AttrTransferTypeConstraint WidgetParentPropertyInfo = Gtk.Container.IsContainer
    type AttrTransferType WidgetParentPropertyInfo = Gtk.Container.Container
    type AttrGetType WidgetParentPropertyInfo = (Maybe Gtk.Container.Container)
    type AttrLabel WidgetParentPropertyInfo = "parent"
    type AttrOrigin WidgetParentPropertyInfo = Widget
    attrGet = getWidgetParent
    attrSet = setWidgetParent
    attrTransfer _ v = do
        unsafeCastTo Gtk.Container.Container v
    attrConstruct = constructWidgetParent
    attrClear = clearWidgetParent
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.parent"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:parent"
        })
#endif

-- VVV Prop "receives-default"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@receives-default@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #receivesDefault
-- @
getWidgetReceivesDefault :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetReceivesDefault obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "receives-default"

-- | Set the value of the “@receives-default@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #receivesDefault 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetReceivesDefault :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetReceivesDefault obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "receives-default" val

-- | Construct a t'GValueConstruct' with valid value for the “@receives-default@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetReceivesDefault :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetReceivesDefault val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "receives-default" val

#if defined(ENABLE_OVERLOADING)
data WidgetReceivesDefaultPropertyInfo
instance AttrInfo WidgetReceivesDefaultPropertyInfo where
    type AttrAllowedOps WidgetReceivesDefaultPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetReceivesDefaultPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetReceivesDefaultPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetReceivesDefaultPropertyInfo = (~) Bool
    type AttrTransferType WidgetReceivesDefaultPropertyInfo = Bool
    type AttrGetType WidgetReceivesDefaultPropertyInfo = Bool
    type AttrLabel WidgetReceivesDefaultPropertyInfo = "receives-default"
    type AttrOrigin WidgetReceivesDefaultPropertyInfo = Widget
    attrGet = getWidgetReceivesDefault
    attrSet = setWidgetReceivesDefault
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetReceivesDefault
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.receivesDefault"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:receivesDefault"
        })
#endif

-- VVV Prop "scale-factor"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@scale-factor@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #scaleFactor
-- @
getWidgetScaleFactor :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetScaleFactor obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "scale-factor"

#if defined(ENABLE_OVERLOADING)
data WidgetScaleFactorPropertyInfo
instance AttrInfo WidgetScaleFactorPropertyInfo where
    type AttrAllowedOps WidgetScaleFactorPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint WidgetScaleFactorPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetScaleFactorPropertyInfo = (~) ()
    type AttrTransferTypeConstraint WidgetScaleFactorPropertyInfo = (~) ()
    type AttrTransferType WidgetScaleFactorPropertyInfo = ()
    type AttrGetType WidgetScaleFactorPropertyInfo = Int32
    type AttrLabel WidgetScaleFactorPropertyInfo = "scale-factor"
    type AttrOrigin WidgetScaleFactorPropertyInfo = Widget
    attrGet = getWidgetScaleFactor
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.scaleFactor"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:scaleFactor"
        })
#endif

-- VVV Prop "sensitive"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@sensitive@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #sensitive
-- @
getWidgetSensitive :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetSensitive obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "sensitive"

-- | Set the value of the “@sensitive@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #sensitive 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetSensitive :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetSensitive obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "sensitive" val

-- | Construct a t'GValueConstruct' with valid value for the “@sensitive@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetSensitive :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetSensitive val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "sensitive" val

#if defined(ENABLE_OVERLOADING)
data WidgetSensitivePropertyInfo
instance AttrInfo WidgetSensitivePropertyInfo where
    type AttrAllowedOps WidgetSensitivePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetSensitivePropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetSensitivePropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetSensitivePropertyInfo = (~) Bool
    type AttrTransferType WidgetSensitivePropertyInfo = Bool
    type AttrGetType WidgetSensitivePropertyInfo = Bool
    type AttrLabel WidgetSensitivePropertyInfo = "sensitive"
    type AttrOrigin WidgetSensitivePropertyInfo = Widget
    attrGet = getWidgetSensitive
    attrSet = setWidgetSensitive
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetSensitive
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.sensitive"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:sensitive"
        })
#endif

-- VVV Prop "style"
   -- Type: TInterface (Name {namespace = "Gtk", name = "Style"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just True)

-- | Get the value of the “@style@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #style
-- @
getWidgetStyle :: (MonadIO m, IsWidget o) => o -> m Gtk.Style.Style
getWidgetStyle obj = MIO.liftIO $ checkUnexpectedNothing "getWidgetStyle" $ B.Properties.getObjectPropertyObject obj "style" Gtk.Style.Style

-- | Set the value of the “@style@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #style 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetStyle :: (MonadIO m, IsWidget o, Gtk.Style.IsStyle a) => o -> a -> m ()
setWidgetStyle obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyObject obj "style" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@style@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetStyle :: (IsWidget o, MIO.MonadIO m, Gtk.Style.IsStyle a) => a -> m (GValueConstruct o)
constructWidgetStyle val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyObject "style" (P.Just val)

-- | Set the value of the “@style@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #style
-- @
clearWidgetStyle :: (MonadIO m, IsWidget o) => o -> m ()
clearWidgetStyle obj = liftIO $ B.Properties.setObjectPropertyObject obj "style" (Nothing :: Maybe Gtk.Style.Style)

#if defined(ENABLE_OVERLOADING)
data WidgetStylePropertyInfo
instance AttrInfo WidgetStylePropertyInfo where
    type AttrAllowedOps WidgetStylePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint WidgetStylePropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetStylePropertyInfo = Gtk.Style.IsStyle
    type AttrTransferTypeConstraint WidgetStylePropertyInfo = Gtk.Style.IsStyle
    type AttrTransferType WidgetStylePropertyInfo = Gtk.Style.Style
    type AttrGetType WidgetStylePropertyInfo = Gtk.Style.Style
    type AttrLabel WidgetStylePropertyInfo = "style"
    type AttrOrigin WidgetStylePropertyInfo = Widget
    attrGet = getWidgetStyle
    attrSet = setWidgetStyle
    attrTransfer _ v = do
        unsafeCastTo Gtk.Style.Style v
    attrConstruct = constructWidgetStyle
    attrClear = clearWidgetStyle
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.style"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:style"
        })
#endif

-- VVV Prop "tooltip-markup"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Just True)

-- | Get the value of the “@tooltip-markup@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #tooltipMarkup
-- @
getWidgetTooltipMarkup :: (MonadIO m, IsWidget o) => o -> m (Maybe T.Text)
getWidgetTooltipMarkup obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "tooltip-markup"

-- | Set the value of the “@tooltip-markup@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #tooltipMarkup 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetTooltipMarkup :: (MonadIO m, IsWidget o) => o -> T.Text -> m ()
setWidgetTooltipMarkup obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "tooltip-markup" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@tooltip-markup@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetTooltipMarkup :: (IsWidget o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructWidgetTooltipMarkup val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "tooltip-markup" (P.Just val)

-- | Set the value of the “@tooltip-markup@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #tooltipMarkup
-- @
clearWidgetTooltipMarkup :: (MonadIO m, IsWidget o) => o -> m ()
clearWidgetTooltipMarkup obj = liftIO $ B.Properties.setObjectPropertyString obj "tooltip-markup" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data WidgetTooltipMarkupPropertyInfo
instance AttrInfo WidgetTooltipMarkupPropertyInfo where
    type AttrAllowedOps WidgetTooltipMarkupPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint WidgetTooltipMarkupPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetTooltipMarkupPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint WidgetTooltipMarkupPropertyInfo = (~) T.Text
    type AttrTransferType WidgetTooltipMarkupPropertyInfo = T.Text
    type AttrGetType WidgetTooltipMarkupPropertyInfo = (Maybe T.Text)
    type AttrLabel WidgetTooltipMarkupPropertyInfo = "tooltip-markup"
    type AttrOrigin WidgetTooltipMarkupPropertyInfo = Widget
    attrGet = getWidgetTooltipMarkup
    attrSet = setWidgetTooltipMarkup
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetTooltipMarkup
    attrClear = clearWidgetTooltipMarkup
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.tooltipMarkup"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:tooltipMarkup"
        })
#endif

-- VVV Prop "tooltip-text"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Just True)

-- | Get the value of the “@tooltip-text@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #tooltipText
-- @
getWidgetTooltipText :: (MonadIO m, IsWidget o) => o -> m (Maybe T.Text)
getWidgetTooltipText obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "tooltip-text"

-- | Set the value of the “@tooltip-text@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #tooltipText 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetTooltipText :: (MonadIO m, IsWidget o) => o -> T.Text -> m ()
setWidgetTooltipText obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "tooltip-text" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@tooltip-text@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetTooltipText :: (IsWidget o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructWidgetTooltipText val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "tooltip-text" (P.Just val)

-- | Set the value of the “@tooltip-text@” property to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #tooltipText
-- @
clearWidgetTooltipText :: (MonadIO m, IsWidget o) => o -> m ()
clearWidgetTooltipText obj = liftIO $ B.Properties.setObjectPropertyString obj "tooltip-text" (Nothing :: Maybe T.Text)

#if defined(ENABLE_OVERLOADING)
data WidgetTooltipTextPropertyInfo
instance AttrInfo WidgetTooltipTextPropertyInfo where
    type AttrAllowedOps WidgetTooltipTextPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint WidgetTooltipTextPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetTooltipTextPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint WidgetTooltipTextPropertyInfo = (~) T.Text
    type AttrTransferType WidgetTooltipTextPropertyInfo = T.Text
    type AttrGetType WidgetTooltipTextPropertyInfo = (Maybe T.Text)
    type AttrLabel WidgetTooltipTextPropertyInfo = "tooltip-text"
    type AttrOrigin WidgetTooltipTextPropertyInfo = Widget
    attrGet = getWidgetTooltipText
    attrSet = setWidgetTooltipText
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetTooltipText
    attrClear = clearWidgetTooltipText
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.tooltipText"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:tooltipText"
        })
#endif

-- VVV Prop "valign"
   -- Type: TInterface (Name {namespace = "Gtk", name = "Align"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@valign@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #valign
-- @
getWidgetValign :: (MonadIO m, IsWidget o) => o -> m Gtk.Enums.Align
getWidgetValign obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "valign"

-- | Set the value of the “@valign@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #valign 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetValign :: (MonadIO m, IsWidget o) => o -> Gtk.Enums.Align -> m ()
setWidgetValign obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "valign" val

-- | Construct a t'GValueConstruct' with valid value for the “@valign@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetValign :: (IsWidget o, MIO.MonadIO m) => Gtk.Enums.Align -> m (GValueConstruct o)
constructWidgetValign val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "valign" val

#if defined(ENABLE_OVERLOADING)
data WidgetValignPropertyInfo
instance AttrInfo WidgetValignPropertyInfo where
    type AttrAllowedOps WidgetValignPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetValignPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetValignPropertyInfo = (~) Gtk.Enums.Align
    type AttrTransferTypeConstraint WidgetValignPropertyInfo = (~) Gtk.Enums.Align
    type AttrTransferType WidgetValignPropertyInfo = Gtk.Enums.Align
    type AttrGetType WidgetValignPropertyInfo = Gtk.Enums.Align
    type AttrLabel WidgetValignPropertyInfo = "valign"
    type AttrOrigin WidgetValignPropertyInfo = Widget
    attrGet = getWidgetValign
    attrSet = setWidgetValign
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetValign
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.valign"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:valign"
        })
#endif

-- VVV Prop "vexpand"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@vexpand@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #vexpand
-- @
getWidgetVexpand :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetVexpand obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "vexpand"

-- | Set the value of the “@vexpand@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #vexpand 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetVexpand :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetVexpand obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "vexpand" val

-- | Construct a t'GValueConstruct' with valid value for the “@vexpand@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetVexpand :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetVexpand val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "vexpand" val

#if defined(ENABLE_OVERLOADING)
data WidgetVexpandPropertyInfo
instance AttrInfo WidgetVexpandPropertyInfo where
    type AttrAllowedOps WidgetVexpandPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetVexpandPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetVexpandPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetVexpandPropertyInfo = (~) Bool
    type AttrTransferType WidgetVexpandPropertyInfo = Bool
    type AttrGetType WidgetVexpandPropertyInfo = Bool
    type AttrLabel WidgetVexpandPropertyInfo = "vexpand"
    type AttrOrigin WidgetVexpandPropertyInfo = Widget
    attrGet = getWidgetVexpand
    attrSet = setWidgetVexpand
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetVexpand
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.vexpand"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:vexpand"
        })
#endif

-- VVV Prop "vexpand-set"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@vexpand-set@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #vexpandSet
-- @
getWidgetVexpandSet :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetVexpandSet obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "vexpand-set"

-- | Set the value of the “@vexpand-set@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #vexpandSet 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetVexpandSet :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetVexpandSet obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "vexpand-set" val

-- | Construct a t'GValueConstruct' with valid value for the “@vexpand-set@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetVexpandSet :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetVexpandSet val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "vexpand-set" val

#if defined(ENABLE_OVERLOADING)
data WidgetVexpandSetPropertyInfo
instance AttrInfo WidgetVexpandSetPropertyInfo where
    type AttrAllowedOps WidgetVexpandSetPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetVexpandSetPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetVexpandSetPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetVexpandSetPropertyInfo = (~) Bool
    type AttrTransferType WidgetVexpandSetPropertyInfo = Bool
    type AttrGetType WidgetVexpandSetPropertyInfo = Bool
    type AttrLabel WidgetVexpandSetPropertyInfo = "vexpand-set"
    type AttrOrigin WidgetVexpandSetPropertyInfo = Widget
    attrGet = getWidgetVexpandSet
    attrSet = setWidgetVexpandSet
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetVexpandSet
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.vexpandSet"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:vexpandSet"
        })
#endif

-- VVV Prop "visible"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@visible@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #visible
-- @
getWidgetVisible :: (MonadIO m, IsWidget o) => o -> m Bool
getWidgetVisible obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "visible"

-- | Set the value of the “@visible@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #visible 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetVisible :: (MonadIO m, IsWidget o) => o -> Bool -> m ()
setWidgetVisible obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "visible" val

-- | Construct a t'GValueConstruct' with valid value for the “@visible@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetVisible :: (IsWidget o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructWidgetVisible val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "visible" val

#if defined(ENABLE_OVERLOADING)
data WidgetVisiblePropertyInfo
instance AttrInfo WidgetVisiblePropertyInfo where
    type AttrAllowedOps WidgetVisiblePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetVisiblePropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetVisiblePropertyInfo = (~) Bool
    type AttrTransferTypeConstraint WidgetVisiblePropertyInfo = (~) Bool
    type AttrTransferType WidgetVisiblePropertyInfo = Bool
    type AttrGetType WidgetVisiblePropertyInfo = Bool
    type AttrLabel WidgetVisiblePropertyInfo = "visible"
    type AttrOrigin WidgetVisiblePropertyInfo = Widget
    attrGet = getWidgetVisible
    attrSet = setWidgetVisible
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetVisible
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.visible"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:visible"
        })
#endif

-- VVV Prop "width-request"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@width-request@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #widthRequest
-- @
getWidgetWidthRequest :: (MonadIO m, IsWidget o) => o -> m Int32
getWidgetWidthRequest obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "width-request"

-- | Set the value of the “@width-request@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' widget [ #widthRequest 'Data.GI.Base.Attributes.:=' value ]
-- @
setWidgetWidthRequest :: (MonadIO m, IsWidget o) => o -> Int32 -> m ()
setWidgetWidthRequest obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "width-request" val

-- | Construct a t'GValueConstruct' with valid value for the “@width-request@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructWidgetWidthRequest :: (IsWidget o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructWidgetWidthRequest val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "width-request" val

#if defined(ENABLE_OVERLOADING)
data WidgetWidthRequestPropertyInfo
instance AttrInfo WidgetWidthRequestPropertyInfo where
    type AttrAllowedOps WidgetWidthRequestPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint WidgetWidthRequestPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetWidthRequestPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint WidgetWidthRequestPropertyInfo = (~) Int32
    type AttrTransferType WidgetWidthRequestPropertyInfo = Int32
    type AttrGetType WidgetWidthRequestPropertyInfo = Int32
    type AttrLabel WidgetWidthRequestPropertyInfo = "width-request"
    type AttrOrigin WidgetWidthRequestPropertyInfo = Widget
    attrGet = getWidgetWidthRequest
    attrSet = setWidgetWidthRequest
    attrTransfer _ v = do
        return v
    attrConstruct = constructWidgetWidthRequest
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widthRequest"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:widthRequest"
        })
#endif

-- VVV Prop "window"
   -- Type: TInterface (Name {namespace = "Gdk", name = "Window"})
   -- Flags: [PropertyReadable]
   -- Nullable: (Just True,Nothing)

-- | Get the value of the “@window@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' widget #window
-- @
getWidgetWindow :: (MonadIO m, IsWidget o) => o -> m (Maybe Gdk.Window.Window)
getWidgetWindow obj = MIO.liftIO $ B.Properties.getObjectPropertyObject obj "window" Gdk.Window.Window

#if defined(ENABLE_OVERLOADING)
data WidgetWindowPropertyInfo
instance AttrInfo WidgetWindowPropertyInfo where
    type AttrAllowedOps WidgetWindowPropertyInfo = '[ 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint WidgetWindowPropertyInfo = IsWidget
    type AttrSetTypeConstraint WidgetWindowPropertyInfo = (~) ()
    type AttrTransferTypeConstraint WidgetWindowPropertyInfo = (~) ()
    type AttrTransferType WidgetWindowPropertyInfo = ()
    type AttrGetType WidgetWindowPropertyInfo = (Maybe Gdk.Window.Window)
    type AttrLabel WidgetWindowPropertyInfo = "window"
    type AttrOrigin WidgetWindowPropertyInfo = Widget
    attrGet = getWidgetWindow
    attrSet = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.window"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#g:attr:window"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Widget
type instance O.AttributeList Widget = WidgetAttributeList
type WidgetAttributeList = ('[ '("appPaintable", WidgetAppPaintablePropertyInfo), '("canDefault", WidgetCanDefaultPropertyInfo), '("canFocus", WidgetCanFocusPropertyInfo), '("compositeChild", WidgetCompositeChildPropertyInfo), '("doubleBuffered", WidgetDoubleBufferedPropertyInfo), '("events", WidgetEventsPropertyInfo), '("expand", WidgetExpandPropertyInfo), '("focusOnClick", WidgetFocusOnClickPropertyInfo), '("halign", WidgetHalignPropertyInfo), '("hasDefault", WidgetHasDefaultPropertyInfo), '("hasFocus", WidgetHasFocusPropertyInfo), '("hasTooltip", WidgetHasTooltipPropertyInfo), '("heightRequest", WidgetHeightRequestPropertyInfo), '("hexpand", WidgetHexpandPropertyInfo), '("hexpandSet", WidgetHexpandSetPropertyInfo), '("isFocus", WidgetIsFocusPropertyInfo), '("margin", WidgetMarginPropertyInfo), '("marginBottom", WidgetMarginBottomPropertyInfo), '("marginEnd", WidgetMarginEndPropertyInfo), '("marginLeft", WidgetMarginLeftPropertyInfo), '("marginRight", WidgetMarginRightPropertyInfo), '("marginStart", WidgetMarginStartPropertyInfo), '("marginTop", WidgetMarginTopPropertyInfo), '("name", WidgetNamePropertyInfo), '("noShowAll", WidgetNoShowAllPropertyInfo), '("opacity", WidgetOpacityPropertyInfo), '("parent", WidgetParentPropertyInfo), '("receivesDefault", WidgetReceivesDefaultPropertyInfo), '("scaleFactor", WidgetScaleFactorPropertyInfo), '("sensitive", WidgetSensitivePropertyInfo), '("style", WidgetStylePropertyInfo), '("tooltipMarkup", WidgetTooltipMarkupPropertyInfo), '("tooltipText", WidgetTooltipTextPropertyInfo), '("valign", WidgetValignPropertyInfo), '("vexpand", WidgetVexpandPropertyInfo), '("vexpandSet", WidgetVexpandSetPropertyInfo), '("visible", WidgetVisiblePropertyInfo), '("widthRequest", WidgetWidthRequestPropertyInfo), '("window", WidgetWindowPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
widgetAppPaintable :: AttrLabelProxy "appPaintable"
widgetAppPaintable = AttrLabelProxy

widgetCanDefault :: AttrLabelProxy "canDefault"
widgetCanDefault = AttrLabelProxy

widgetCanFocus :: AttrLabelProxy "canFocus"
widgetCanFocus = AttrLabelProxy

widgetCompositeChild :: AttrLabelProxy "compositeChild"
widgetCompositeChild = AttrLabelProxy

widgetDoubleBuffered :: AttrLabelProxy "doubleBuffered"
widgetDoubleBuffered = AttrLabelProxy

widgetEvents :: AttrLabelProxy "events"
widgetEvents = AttrLabelProxy

widgetExpand :: AttrLabelProxy "expand"
widgetExpand = AttrLabelProxy

widgetFocusOnClick :: AttrLabelProxy "focusOnClick"
widgetFocusOnClick = AttrLabelProxy

widgetHalign :: AttrLabelProxy "halign"
widgetHalign = AttrLabelProxy

widgetHasTooltip :: AttrLabelProxy "hasTooltip"
widgetHasTooltip = AttrLabelProxy

widgetHeightRequest :: AttrLabelProxy "heightRequest"
widgetHeightRequest = AttrLabelProxy

widgetHexpand :: AttrLabelProxy "hexpand"
widgetHexpand = AttrLabelProxy

widgetHexpandSet :: AttrLabelProxy "hexpandSet"
widgetHexpandSet = AttrLabelProxy

widgetMargin :: AttrLabelProxy "margin"
widgetMargin = AttrLabelProxy

widgetMarginBottom :: AttrLabelProxy "marginBottom"
widgetMarginBottom = AttrLabelProxy

widgetMarginEnd :: AttrLabelProxy "marginEnd"
widgetMarginEnd = AttrLabelProxy

widgetMarginLeft :: AttrLabelProxy "marginLeft"
widgetMarginLeft = AttrLabelProxy

widgetMarginRight :: AttrLabelProxy "marginRight"
widgetMarginRight = AttrLabelProxy

widgetMarginStart :: AttrLabelProxy "marginStart"
widgetMarginStart = AttrLabelProxy

widgetMarginTop :: AttrLabelProxy "marginTop"
widgetMarginTop = AttrLabelProxy

widgetName :: AttrLabelProxy "name"
widgetName = AttrLabelProxy

widgetNoShowAll :: AttrLabelProxy "noShowAll"
widgetNoShowAll = AttrLabelProxy

widgetOpacity :: AttrLabelProxy "opacity"
widgetOpacity = AttrLabelProxy

widgetParent :: AttrLabelProxy "parent"
widgetParent = AttrLabelProxy

widgetReceivesDefault :: AttrLabelProxy "receivesDefault"
widgetReceivesDefault = AttrLabelProxy

widgetScaleFactor :: AttrLabelProxy "scaleFactor"
widgetScaleFactor = AttrLabelProxy

widgetSensitive :: AttrLabelProxy "sensitive"
widgetSensitive = AttrLabelProxy

widgetStyle :: AttrLabelProxy "style"
widgetStyle = AttrLabelProxy

widgetTooltipMarkup :: AttrLabelProxy "tooltipMarkup"
widgetTooltipMarkup = AttrLabelProxy

widgetTooltipText :: AttrLabelProxy "tooltipText"
widgetTooltipText = AttrLabelProxy

widgetValign :: AttrLabelProxy "valign"
widgetValign = AttrLabelProxy

widgetVexpand :: AttrLabelProxy "vexpand"
widgetVexpand = AttrLabelProxy

widgetVexpandSet :: AttrLabelProxy "vexpandSet"
widgetVexpandSet = AttrLabelProxy

widgetVisible :: AttrLabelProxy "visible"
widgetVisible = AttrLabelProxy

widgetWidthRequest :: AttrLabelProxy "widthRequest"
widgetWidthRequest = AttrLabelProxy

widgetWindow :: AttrLabelProxy "window"
widgetWindow = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Widget = WidgetSignalList
type WidgetSignalList = ('[ '("accelClosuresChanged", WidgetAccelClosuresChangedSignalInfo), '("buttonPressEvent", WidgetButtonPressEventSignalInfo), '("buttonReleaseEvent", WidgetButtonReleaseEventSignalInfo), '("canActivateAccel", WidgetCanActivateAccelSignalInfo), '("childNotify", WidgetChildNotifySignalInfo), '("compositedChanged", WidgetCompositedChangedSignalInfo), '("configureEvent", WidgetConfigureEventSignalInfo), '("damageEvent", WidgetDamageEventSignalInfo), '("deleteEvent", WidgetDeleteEventSignalInfo), '("destroy", WidgetDestroySignalInfo), '("destroyEvent", WidgetDestroyEventSignalInfo), '("directionChanged", WidgetDirectionChangedSignalInfo), '("dragBegin", WidgetDragBeginSignalInfo), '("dragDataDelete", WidgetDragDataDeleteSignalInfo), '("dragDataGet", WidgetDragDataGetSignalInfo), '("dragDataReceived", WidgetDragDataReceivedSignalInfo), '("dragDrop", WidgetDragDropSignalInfo), '("dragEnd", WidgetDragEndSignalInfo), '("dragFailed", WidgetDragFailedSignalInfo), '("dragLeave", WidgetDragLeaveSignalInfo), '("dragMotion", WidgetDragMotionSignalInfo), '("draw", WidgetDrawSignalInfo), '("enterNotifyEvent", WidgetEnterNotifyEventSignalInfo), '("event", WidgetEventSignalInfo), '("eventAfter", WidgetEventAfterSignalInfo), '("focus", WidgetFocusSignalInfo), '("focusInEvent", WidgetFocusInEventSignalInfo), '("focusOutEvent", WidgetFocusOutEventSignalInfo), '("grabBrokenEvent", WidgetGrabBrokenEventSignalInfo), '("grabFocus", WidgetGrabFocusSignalInfo), '("grabNotify", WidgetGrabNotifySignalInfo), '("hide", WidgetHideSignalInfo), '("hierarchyChanged", WidgetHierarchyChangedSignalInfo), '("keyPressEvent", WidgetKeyPressEventSignalInfo), '("keyReleaseEvent", WidgetKeyReleaseEventSignalInfo), '("keynavFailed", WidgetKeynavFailedSignalInfo), '("leaveNotifyEvent", WidgetLeaveNotifyEventSignalInfo), '("map", WidgetMapSignalInfo), '("mapEvent", WidgetMapEventSignalInfo), '("mnemonicActivate", WidgetMnemonicActivateSignalInfo), '("motionNotifyEvent", WidgetMotionNotifyEventSignalInfo), '("moveFocus", WidgetMoveFocusSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo), '("parentSet", WidgetParentSetSignalInfo), '("popupMenu", WidgetPopupMenuSignalInfo), '("propertyNotifyEvent", WidgetPropertyNotifyEventSignalInfo), '("proximityInEvent", WidgetProximityInEventSignalInfo), '("proximityOutEvent", WidgetProximityOutEventSignalInfo), '("queryTooltip", WidgetQueryTooltipSignalInfo), '("realize", WidgetRealizeSignalInfo), '("screenChanged", WidgetScreenChangedSignalInfo), '("scrollEvent", WidgetScrollEventSignalInfo), '("selectionClearEvent", WidgetSelectionClearEventSignalInfo), '("selectionGet", WidgetSelectionGetSignalInfo), '("selectionNotifyEvent", WidgetSelectionNotifyEventSignalInfo), '("selectionReceived", WidgetSelectionReceivedSignalInfo), '("selectionRequestEvent", WidgetSelectionRequestEventSignalInfo), '("show", WidgetShowSignalInfo), '("showHelp", WidgetShowHelpSignalInfo), '("sizeAllocate", WidgetSizeAllocateSignalInfo), '("stateChanged", WidgetStateChangedSignalInfo), '("stateFlagsChanged", WidgetStateFlagsChangedSignalInfo), '("styleSet", WidgetStyleSetSignalInfo), '("styleUpdated", WidgetStyleUpdatedSignalInfo), '("touchEvent", WidgetTouchEventSignalInfo), '("unmap", WidgetUnmapSignalInfo), '("unmapEvent", WidgetUnmapEventSignalInfo), '("unrealize", WidgetUnrealizeSignalInfo), '("visibilityNotifyEvent", WidgetVisibilityNotifyEventSignalInfo), '("windowStateEvent", WidgetWindowStateEventSignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method Widget::activate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget that\8217s activatable"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_activate" gtk_widget_activate :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | For widgets that can be “activated” (buttons, menu items, etc.)
-- this function activates them. Activation is what happens when you
-- press Enter on a widget during key navigation. If /@widget@/ isn\'t
-- activatable, the function returns 'P.False'.
widgetActivate ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' that’s activatable
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the widget was activatable
widgetActivate widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_activate widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetActivateMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetActivateMethodInfo a signature where
    overloadedMethod = widgetActivate

instance O.OverloadedMethodInfo WidgetActivateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetActivate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetActivate"
        })


#endif

-- method Widget::add_accelerator
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "widget to install an accelerator on"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "accel_signal"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "widget signal to emit on accelerator activation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "accel_group"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "AccelGroup" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "accel group for this widget, added to its toplevel"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "accel_key"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "GDK keyval of the accelerator"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "accel_mods"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "ModifierType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "modifier key combination of the accelerator"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "accel_flags"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "AccelFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flag accelerators, e.g. %GTK_ACCEL_VISIBLE"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_add_accelerator" gtk_widget_add_accelerator :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- accel_signal : TBasicType TUTF8
    Ptr Gtk.AccelGroup.AccelGroup ->        -- accel_group : TInterface (Name {namespace = "Gtk", name = "AccelGroup"})
    Word32 ->                               -- accel_key : TBasicType TUInt
    CUInt ->                                -- accel_mods : TInterface (Name {namespace = "Gdk", name = "ModifierType"})
    CUInt ->                                -- accel_flags : TInterface (Name {namespace = "Gtk", name = "AccelFlags"})
    IO ()

-- | Installs an accelerator for this /@widget@/ in /@accelGroup@/ that causes
-- /@accelSignal@/ to be emitted if the accelerator is activated.
-- The /@accelGroup@/ needs to be added to the widget’s toplevel via
-- 'GI.Gtk.Objects.Window.windowAddAccelGroup', and the signal must be of type 'GI.GObject.Flags.SignalFlagsAction'.
-- Accelerators added through this function are not user changeable during
-- runtime. If you want to support accelerators that can be changed by the
-- user, use 'GI.Gtk.Objects.AccelMap.accelMapAddEntry' and 'GI.Gtk.Objects.Widget.widgetSetAccelPath' or
-- 'GI.Gtk.Objects.MenuItem.menuItemSetAccelPath' instead.
widgetAddAccelerator ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, Gtk.AccelGroup.IsAccelGroup b) =>
    a
    -- ^ /@widget@/: widget to install an accelerator on
    -> T.Text
    -- ^ /@accelSignal@/: widget signal to emit on accelerator activation
    -> b
    -- ^ /@accelGroup@/: accel group for this widget, added to its toplevel
    -> Word32
    -- ^ /@accelKey@/: GDK keyval of the accelerator
    -> [Gdk.Flags.ModifierType]
    -- ^ /@accelMods@/: modifier key combination of the accelerator
    -> [Gtk.Flags.AccelFlags]
    -- ^ /@accelFlags@/: flag accelerators, e.g. 'GI.Gtk.Flags.AccelFlagsVisible'
    -> m ()
widgetAddAccelerator widget accelSignal accelGroup accelKey accelMods accelFlags = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    accelSignal' <- textToCString accelSignal
    accelGroup' <- unsafeManagedPtrCastPtr accelGroup
    let accelMods' = gflagsToWord accelMods
    let accelFlags' = gflagsToWord accelFlags
    gtk_widget_add_accelerator widget' accelSignal' accelGroup' accelKey accelMods' accelFlags'
    touchManagedPtr widget
    touchManagedPtr accelGroup
    freeMem accelSignal'
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetAddAcceleratorMethodInfo
instance (signature ~ (T.Text -> b -> Word32 -> [Gdk.Flags.ModifierType] -> [Gtk.Flags.AccelFlags] -> m ()), MonadIO m, IsWidget a, Gtk.AccelGroup.IsAccelGroup b) => O.OverloadedMethod WidgetAddAcceleratorMethodInfo a signature where
    overloadedMethod = widgetAddAccelerator

instance O.OverloadedMethodInfo WidgetAddAcceleratorMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetAddAccelerator",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetAddAccelerator"
        })


#endif

-- method Widget::add_device_events
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "device"
--           , argType = TInterface Name { namespace = "Gdk" , name = "Device" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GdkDevice" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "events"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "EventMask" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an event mask, see #GdkEventMask"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_add_device_events" gtk_widget_add_device_events :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.Device.Device ->                -- device : TInterface (Name {namespace = "Gdk", name = "Device"})
    CUInt ->                                -- events : TInterface (Name {namespace = "Gdk", name = "EventMask"})
    IO ()

-- | Adds the device events in the bitfield /@events@/ to the event mask for
-- /@widget@/. See 'GI.Gtk.Objects.Widget.widgetSetDeviceEvents' for details.
-- 
-- /Since: 3.0/
widgetAddDeviceEvents ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, Gdk.Device.IsDevice b) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> b
    -- ^ /@device@/: a t'GI.Gdk.Objects.Device.Device'
    -> [Gdk.Flags.EventMask]
    -- ^ /@events@/: an event mask, see t'GI.Gdk.Flags.EventMask'
    -> m ()
widgetAddDeviceEvents widget device events = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    device' <- unsafeManagedPtrCastPtr device
    let events' = gflagsToWord events
    gtk_widget_add_device_events widget' device' events'
    touchManagedPtr widget
    touchManagedPtr device
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetAddDeviceEventsMethodInfo
instance (signature ~ (b -> [Gdk.Flags.EventMask] -> m ()), MonadIO m, IsWidget a, Gdk.Device.IsDevice b) => O.OverloadedMethod WidgetAddDeviceEventsMethodInfo a signature where
    overloadedMethod = widgetAddDeviceEvents

instance O.OverloadedMethodInfo WidgetAddDeviceEventsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetAddDeviceEvents",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetAddDeviceEvents"
        })


#endif

-- method Widget::add_events
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "events"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "EventMask" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an event mask, see #GdkEventMask"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_add_events" gtk_widget_add_events :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- events : TInterface (Name {namespace = "Gdk", name = "EventMask"})
    IO ()

-- | Adds the events in the bitfield /@events@/ to the event mask for
-- /@widget@/. See 'GI.Gtk.Objects.Widget.widgetSetEvents' and the
-- [input handling overview][event-masks] for details.
widgetAddEvents ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> [Gdk.Flags.EventMask]
    -- ^ /@events@/: an event mask, see t'GI.Gdk.Flags.EventMask'
    -> m ()
widgetAddEvents widget events = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let events' = gflagsToWord events
    gtk_widget_add_events widget' events'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetAddEventsMethodInfo
instance (signature ~ ([Gdk.Flags.EventMask] -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetAddEventsMethodInfo a signature where
    overloadedMethod = widgetAddEvents

instance O.OverloadedMethodInfo WidgetAddEventsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetAddEvents",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetAddEvents"
        })


#endif

-- method Widget::add_mnemonic_label
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "label"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a #GtkWidget that acts as a mnemonic label for @widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_add_mnemonic_label" gtk_widget_add_mnemonic_label :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Widget ->                           -- label : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Adds a widget to the list of mnemonic labels for
-- this widget. (See 'GI.Gtk.Objects.Widget.widgetListMnemonicLabels'). Note the
-- list of mnemonic labels for the widget is cleared when the
-- widget is destroyed, so the caller must make sure to update
-- its internal state at this point as well, by using a connection
-- to the [Widget::destroy]("GI.Gtk.Objects.Widget#g:signal:destroy") signal or a weak notifier.
-- 
-- /Since: 2.4/
widgetAddMnemonicLabel ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, IsWidget b) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> b
    -- ^ /@label@/: a t'GI.Gtk.Objects.Widget.Widget' that acts as a mnemonic label for /@widget@/
    -> m ()
widgetAddMnemonicLabel widget label = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    label' <- unsafeManagedPtrCastPtr label
    gtk_widget_add_mnemonic_label widget' label'
    touchManagedPtr widget
    touchManagedPtr label
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetAddMnemonicLabelMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsWidget a, IsWidget b) => O.OverloadedMethod WidgetAddMnemonicLabelMethodInfo a signature where
    overloadedMethod = widgetAddMnemonicLabel

instance O.OverloadedMethodInfo WidgetAddMnemonicLabelMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetAddMnemonicLabel",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetAddMnemonicLabel"
        })


#endif

-- method Widget::add_tick_callback
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "callback"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "TickCallback" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "function to call for updating animations"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeNotified
--           , argClosure = 2
--           , argDestroy = 3
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "user_data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data to pass to @callback"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "notify"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DestroyNotify" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "function to call to free @user_data when the callback is removed."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_add_tick_callback" gtk_widget_add_tick_callback :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    FunPtr Gtk.Callbacks.C_TickCallback ->  -- callback : TInterface (Name {namespace = "Gtk", name = "TickCallback"})
    Ptr () ->                               -- user_data : TBasicType TPtr
    FunPtr GLib.Callbacks.C_DestroyNotify -> -- notify : TInterface (Name {namespace = "GLib", name = "DestroyNotify"})
    IO Word32

-- | Queues an animation frame update and adds a callback to be called
-- before each frame. Until the tick callback is removed, it will be
-- called frequently (usually at the frame rate of the output device
-- or as quickly as the application can be repainted, whichever is
-- slower). For this reason, is most suitable for handling graphics
-- that change every frame or every few frames. The tick callback does
-- not automatically imply a relayout or repaint. If you want a
-- repaint or relayout, and aren’t changing widget properties that
-- would trigger that (for example, changing the text of a t'GI.Gtk.Objects.Label.Label'),
-- then you will have to call 'GI.Gtk.Objects.Widget.widgetQueueResize' or
-- 'GI.Gtk.Objects.Widget.widgetQueueDrawArea' yourself.
-- 
-- 'GI.Gdk.Objects.FrameClock.frameClockGetFrameTime' should generally be used for timing
-- continuous animations and
-- 'GI.Gdk.Structs.FrameTimings.frameTimingsGetPredictedPresentationTime' if you are
-- trying to display isolated frames at particular times.
-- 
-- This is a more convenient alternative to connecting directly to the
-- [FrameClock::update]("GI.Gdk.Objects.FrameClock#g:signal:update") signal of t'GI.Gdk.Objects.FrameClock.FrameClock', since you don\'t
-- have to worry about when a t'GI.Gdk.Objects.FrameClock.FrameClock' is assigned to a widget.
-- 
-- /Since: 3.8/
widgetAddTickCallback ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> Gtk.Callbacks.TickCallback
    -- ^ /@callback@/: function to call for updating animations
    -> m Word32
    -- ^ __Returns:__ an id for the connection of this callback. Remove the callback
    --     by passing it to 'GI.Gtk.Objects.Widget.widgetRemoveTickCallback'
widgetAddTickCallback widget callback = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    callback' <- Gtk.Callbacks.mk_TickCallback (Gtk.Callbacks.wrap_TickCallback Nothing (Gtk.Callbacks.drop_closures_TickCallback callback))
    let userData = castFunPtrToPtr callback'
    let notify = SP.safeFreeFunPtrPtr
    result <- gtk_widget_add_tick_callback widget' callback' userData notify
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetAddTickCallbackMethodInfo
instance (signature ~ (Gtk.Callbacks.TickCallback -> m Word32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetAddTickCallbackMethodInfo a signature where
    overloadedMethod = widgetAddTickCallback

instance O.OverloadedMethodInfo WidgetAddTickCallbackMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetAddTickCallback",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetAddTickCallback"
        })


#endif

-- method Widget::can_activate_accel
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "signal_id"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the ID of a signal installed on @widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_can_activate_accel" gtk_widget_can_activate_accel :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Word32 ->                               -- signal_id : TBasicType TUInt
    IO CInt

-- | Determines whether an accelerator that activates the signal
-- identified by /@signalId@/ can currently be activated.
-- This is done by emitting the [Widget::canActivateAccel]("GI.Gtk.Objects.Widget#g:signal:canActivateAccel")
-- signal on /@widget@/; if the signal isn’t overridden by a
-- handler or in a derived widget, then the default check is
-- that the widget must be sensitive, and the widget and all
-- its ancestors mapped.
-- 
-- /Since: 2.4/
widgetCanActivateAccel ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> Word32
    -- ^ /@signalId@/: the ID of a signal installed on /@widget@/
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the accelerator can be activated.
widgetCanActivateAccel widget signalId = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_can_activate_accel widget' signalId
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetCanActivateAccelMethodInfo
instance (signature ~ (Word32 -> m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetCanActivateAccelMethodInfo a signature where
    overloadedMethod = widgetCanActivateAccel

instance O.OverloadedMethodInfo WidgetCanActivateAccelMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetCanActivateAccel",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetCanActivateAccel"
        })


#endif

-- method Widget::child_focus
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "direction"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "DirectionType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "direction of focus movement"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_child_focus" gtk_widget_child_focus :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- direction : TInterface (Name {namespace = "Gtk", name = "DirectionType"})
    IO CInt

-- | This function is used by custom widget implementations; if you\'re
-- writing an app, you’d use 'GI.Gtk.Objects.Widget.widgetGrabFocus' to move the focus
-- to a particular widget, and 'GI.Gtk.Objects.Container.containerSetFocusChain' to
-- change the focus tab order. So you may want to investigate those
-- functions instead.
-- 
-- 'GI.Gtk.Objects.Widget.widgetChildFocus' is called by containers as the user moves
-- around the window using keyboard shortcuts. /@direction@/ indicates
-- what kind of motion is taking place (up, down, left, right, tab
-- forward, tab backward). 'GI.Gtk.Objects.Widget.widgetChildFocus' emits the
-- [Widget::focus]("GI.Gtk.Objects.Widget#g:signal:focus") signal; widgets override the default handler
-- for this signal in order to implement appropriate focus behavior.
-- 
-- The default [focus](#g:signal:focus) handler for a widget should return 'P.True' if
-- moving in /@direction@/ left the focus on a focusable location inside
-- that widget, and 'P.False' if moving in /@direction@/ moved the focus
-- outside the widget. If returning 'P.True', widgets normally
-- call 'GI.Gtk.Objects.Widget.widgetGrabFocus' to place the focus accordingly;
-- if returning 'P.False', they don’t modify the current focus location.
widgetChildFocus ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> Gtk.Enums.DirectionType
    -- ^ /@direction@/: direction of focus movement
    -> m Bool
    -- ^ __Returns:__ 'P.True' if focus ended up inside /@widget@/
widgetChildFocus widget direction = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let direction' = (fromIntegral . fromEnum) direction
    result <- gtk_widget_child_focus widget' direction'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetChildFocusMethodInfo
instance (signature ~ (Gtk.Enums.DirectionType -> m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetChildFocusMethodInfo a signature where
    overloadedMethod = widgetChildFocus

instance O.OverloadedMethodInfo WidgetChildFocusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetChildFocus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetChildFocus"
        })


#endif

-- method Widget::child_notify
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "child_property"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the name of a child property installed on the\n                 class of @widget\8217s parent"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_child_notify" gtk_widget_child_notify :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- child_property : TBasicType TUTF8
    IO ()

-- | Emits a [Widget::childNotify]("GI.Gtk.Objects.Widget#g:signal:childNotify") signal for the
-- [child property][child-properties] /@childProperty@/
-- on /@widget@/.
-- 
-- This is the analogue of 'GI.GObject.Objects.Object.objectNotify' for child properties.
-- 
-- Also see 'GI.Gtk.Objects.Container.containerChildNotify'.
widgetChildNotify ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> T.Text
    -- ^ /@childProperty@/: the name of a child property installed on the
    --                  class of /@widget@/’s parent
    -> m ()
widgetChildNotify widget childProperty = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    childProperty' <- textToCString childProperty
    gtk_widget_child_notify widget' childProperty'
    touchManagedPtr widget
    freeMem childProperty'
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetChildNotifyMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetChildNotifyMethodInfo a signature where
    overloadedMethod = widgetChildNotify

instance O.OverloadedMethodInfo WidgetChildNotifyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetChildNotify",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetChildNotify"
        })


#endif

-- method Widget::class_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "path_length"
--           , argType = TBasicType TUInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "location to store the length of the\n    class path, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "location to store the class path as an\n    allocated string, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "path_reversed"
--           , argType = TBasicType TUTF8
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "location to store the reverse\n    class path as an allocated string, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_class_path" gtk_widget_class_path :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Word32 ->                           -- path_length : TBasicType TUInt
    Ptr CString ->                          -- path : TBasicType TUTF8
    Ptr CString ->                          -- path_reversed : TBasicType TUTF8
    IO ()

{-# DEPRECATED widgetClassPath ["(Since version 3.0)","Use 'GI.Gtk.Objects.Widget.widgetGetPath' instead"] #-}
-- | Same as 'GI.Gtk.Objects.Widget.widgetPath', but always uses the name of a widget’s type,
-- never uses a custom name set with 'GI.Gtk.Objects.Widget.widgetSetName'.
widgetClassPath ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m ((Word32, T.Text, T.Text))
widgetClassPath widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    pathLength <- allocMem :: IO (Ptr Word32)
    path <- callocMem :: IO (Ptr CString)
    pathReversed <- callocMem :: IO (Ptr CString)
    gtk_widget_class_path widget' pathLength path pathReversed
    pathLength' <- peek pathLength
    path' <- peek path
    path'' <- cstringToText path'
    freeMem path'
    pathReversed' <- peek pathReversed
    pathReversed'' <- cstringToText pathReversed'
    freeMem pathReversed'
    touchManagedPtr widget
    freeMem pathLength
    freeMem path
    freeMem pathReversed
    return (pathLength', path'', pathReversed'')

#if defined(ENABLE_OVERLOADING)
data WidgetClassPathMethodInfo
instance (signature ~ (m ((Word32, T.Text, T.Text))), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetClassPathMethodInfo a signature where
    overloadedMethod = widgetClassPath

instance O.OverloadedMethodInfo WidgetClassPathMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetClassPath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetClassPath"
        })


#endif

-- method Widget::compute_expand
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "orientation"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Orientation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "expand direction" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_compute_expand" gtk_widget_compute_expand :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- orientation : TInterface (Name {namespace = "Gtk", name = "Orientation"})
    IO CInt

-- | Computes whether a container should give this widget extra space
-- when possible. Containers should check this, rather than
-- looking at 'GI.Gtk.Objects.Widget.widgetGetHexpand' or 'GI.Gtk.Objects.Widget.widgetGetVexpand'.
-- 
-- This function already checks whether the widget is visible, so
-- visibility does not need to be checked separately. Non-visible
-- widgets are not expanded.
-- 
-- The computed expand value uses either the expand setting explicitly
-- set on the widget itself, or, if none has been explicitly set,
-- the widget may expand if some of its children do.
widgetComputeExpand ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget
    -> Gtk.Enums.Orientation
    -- ^ /@orientation@/: expand direction
    -> m Bool
    -- ^ __Returns:__ whether widget tree rooted here should be expanded
widgetComputeExpand widget orientation = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let orientation' = (fromIntegral . fromEnum) orientation
    result <- gtk_widget_compute_expand widget' orientation'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetComputeExpandMethodInfo
instance (signature ~ (Gtk.Enums.Orientation -> m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetComputeExpandMethodInfo a signature where
    overloadedMethod = widgetComputeExpand

instance O.OverloadedMethodInfo WidgetComputeExpandMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetComputeExpand",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetComputeExpand"
        })


#endif

-- method Widget::create_pango_context
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Context" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_create_pango_context" gtk_widget_create_pango_context :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Pango.Context.Context)

-- | Creates a new t'GI.Pango.Objects.Context.Context' with the appropriate font map,
-- font options, font description, and base direction for drawing
-- text for this widget. See also 'GI.Gtk.Objects.Widget.widgetGetPangoContext'.
widgetCreatePangoContext ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Pango.Context.Context
    -- ^ __Returns:__ the new t'GI.Pango.Objects.Context.Context'
widgetCreatePangoContext widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_create_pango_context widget'
    checkUnexpectedReturnNULL "widgetCreatePangoContext" result
    result' <- (wrapObject Pango.Context.Context) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetCreatePangoContextMethodInfo
instance (signature ~ (m Pango.Context.Context), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetCreatePangoContextMethodInfo a signature where
    overloadedMethod = widgetCreatePangoContext

instance O.OverloadedMethodInfo WidgetCreatePangoContextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetCreatePangoContext",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetCreatePangoContext"
        })


#endif

-- method Widget::create_pango_layout
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "text to set on the layout (can be %NULL)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Layout" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_create_pango_layout" gtk_widget_create_pango_layout :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- text : TBasicType TUTF8
    IO (Ptr Pango.Layout.Layout)

-- | Creates a new t'GI.Pango.Objects.Layout.Layout' with the appropriate font map,
-- font description, and base direction for drawing text for
-- this widget.
-- 
-- If you keep a t'GI.Pango.Objects.Layout.Layout' created in this way around, you need
-- to re-create it when the widget t'GI.Pango.Objects.Context.Context' is replaced.
-- This can be tracked by using the [Widget::screenChanged]("GI.Gtk.Objects.Widget#g:signal:screenChanged") signal
-- on the widget.
widgetCreatePangoLayout ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> Maybe (T.Text)
    -- ^ /@text@/: text to set on the layout (can be 'P.Nothing')
    -> m Pango.Layout.Layout
    -- ^ __Returns:__ the new t'GI.Pango.Objects.Layout.Layout'
widgetCreatePangoLayout widget text = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    maybeText <- case text of
        Nothing -> return FP.nullPtr
        Just jText -> do
            jText' <- textToCString jText
            return jText'
    result <- gtk_widget_create_pango_layout widget' maybeText
    checkUnexpectedReturnNULL "widgetCreatePangoLayout" result
    result' <- (wrapObject Pango.Layout.Layout) result
    touchManagedPtr widget
    freeMem maybeText
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetCreatePangoLayoutMethodInfo
instance (signature ~ (Maybe (T.Text) -> m Pango.Layout.Layout), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetCreatePangoLayoutMethodInfo a signature where
    overloadedMethod = widgetCreatePangoLayout

instance O.OverloadedMethodInfo WidgetCreatePangoLayoutMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetCreatePangoLayout",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetCreatePangoLayout"
        })


#endif

-- method Widget::destroy
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_destroy" gtk_widget_destroy :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Destroys a widget.
-- 
-- When a widget is destroyed all references it holds on other objects
-- will be released:
-- 
-- * if the widget is inside a container, it will be removed from its
--   parent
-- * if the widget is a container, all its children will be destroyed,
--   recursively
-- * if the widget is a top level, it will be removed from the list
--   of top level widgets that GTK+ maintains internally
-- 
-- 
-- It\'s expected that all references held on the widget will also
-- be released; you should connect to the [Widget::destroy]("GI.Gtk.Objects.Widget#g:signal:destroy") signal
-- if you hold a reference to /@widget@/ and you wish to remove it when
-- this function is called. It is not necessary to do so if you are
-- implementing a t'GI.Gtk.Objects.Container.Container', as you\'ll be able to use the
-- t'GI.Gtk.Structs.ContainerClass.ContainerClass'.@/remove/@() virtual function for that.
-- 
-- It\'s important to notice that 'GI.Gtk.Objects.Widget.widgetDestroy' will only cause
-- the /@widget@/ to be finalized if no additional references, acquired
-- using 'GI.GObject.Objects.Object.objectRef', are held on it. In case additional references
-- are in place, the /@widget@/ will be in an \"inert\" state after calling
-- this function; /@widget@/ will still point to valid memory, allowing you
-- to release the references you hold, but you may not query the widget\'s
-- own state.
-- 
-- You should typically call this function on top level widgets, and
-- rarely on child widgets.
-- 
-- See also: 'GI.Gtk.Objects.Container.containerRemove'
widgetDestroy ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m ()
widgetDestroy widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_destroy widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDestroyMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDestroyMethodInfo a signature where
    overloadedMethod = widgetDestroy

instance O.OverloadedMethodInfo WidgetDestroyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDestroy",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDestroy"
        })


#endif

-- method Widget::destroyed
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "widget_pointer"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionInout
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "address of a variable that contains @widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_destroyed" gtk_widget_destroyed :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr (Ptr Widget) ->                     -- widget_pointer : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | This function sets */@widgetPointer@/ to 'P.Nothing' if /@widgetPointer@/ !=
-- 'P.Nothing'.  It’s intended to be used as a callback connected to the
-- “destroy” signal of a widget. You connect 'GI.Gtk.Objects.Widget.widgetDestroyed'
-- as a signal handler, and pass the address of your widget variable
-- as user data. Then when the widget is destroyed, the variable will
-- be set to 'P.Nothing'. Useful for example to avoid multiple copies
-- of the same dialog.
widgetDestroyed ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, IsWidget b) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> b
    -- ^ /@widgetPointer@/: address of a variable that contains /@widget@/
    -> m (Widget)
widgetDestroyed widget widgetPointer = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    widgetPointer' <- unsafeManagedPtrCastPtr widgetPointer
    widgetPointer'' <- allocMem :: IO (Ptr (Ptr Widget))
    poke widgetPointer'' widgetPointer'
    gtk_widget_destroyed widget' widgetPointer''
    widgetPointer''' <- peek widgetPointer''
    widgetPointer'''' <- (newObject Widget) widgetPointer'''
    touchManagedPtr widget
    touchManagedPtr widgetPointer
    freeMem widgetPointer''
    return widgetPointer''''

#if defined(ENABLE_OVERLOADING)
data WidgetDestroyedMethodInfo
instance (signature ~ (b -> m (Widget)), MonadIO m, IsWidget a, IsWidget b) => O.OverloadedMethod WidgetDestroyedMethodInfo a signature where
    overloadedMethod = widgetDestroyed

instance O.OverloadedMethodInfo WidgetDestroyedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDestroyed",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDestroyed"
        })


#endif

-- method Widget::device_is_shadowed
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "device"
--           , argType = TInterface Name { namespace = "Gdk" , name = "Device" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GdkDevice" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_device_is_shadowed" gtk_widget_device_is_shadowed :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.Device.Device ->                -- device : TInterface (Name {namespace = "Gdk", name = "Device"})
    IO CInt

-- | Returns 'P.True' if /@device@/ has been shadowed by a GTK+
-- device grab on another widget, so it would stop sending
-- events to /@widget@/. This may be used in the
-- [Widget::grabNotify]("GI.Gtk.Objects.Widget#g:signal:grabNotify") signal to check for specific
-- devices. See 'GI.Gtk.Functions.deviceGrabAdd'.
-- 
-- /Since: 3.0/
widgetDeviceIsShadowed ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, Gdk.Device.IsDevice b) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> b
    -- ^ /@device@/: a t'GI.Gdk.Objects.Device.Device'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if there is an ongoing grab on /@device@/
    --          by another t'GI.Gtk.Objects.Widget.Widget' than /@widget@/.
widgetDeviceIsShadowed widget device = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    device' <- unsafeManagedPtrCastPtr device
    result <- gtk_widget_device_is_shadowed widget' device'
    let result' = (/= 0) result
    touchManagedPtr widget
    touchManagedPtr device
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetDeviceIsShadowedMethodInfo
instance (signature ~ (b -> m Bool), MonadIO m, IsWidget a, Gdk.Device.IsDevice b) => O.OverloadedMethod WidgetDeviceIsShadowedMethodInfo a signature where
    overloadedMethod = widgetDeviceIsShadowed

instance O.OverloadedMethodInfo WidgetDeviceIsShadowedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDeviceIsShadowed",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDeviceIsShadowed"
        })


#endif

-- method Widget::drag_begin
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the source widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "targets"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "TargetList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The targets (data formats) in which the\n   source can provide the data"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "actions"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "DragAction" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "A bitmask of the allowed drag actions for this drag"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "button"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "The button the user clicked to start the drag"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "event"
--           , argType = TInterface Name { namespace = "Gdk" , name = "Event" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The event that triggered the start of the drag,\n   or %NULL if none can be obtained."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "DragContext" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_begin" gtk_drag_begin :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gtk.TargetList.TargetList ->        -- targets : TInterface (Name {namespace = "Gtk", name = "TargetList"})
    CUInt ->                                -- actions : TInterface (Name {namespace = "Gdk", name = "DragAction"})
    Int32 ->                                -- button : TBasicType TInt
    Ptr Gdk.Event.Event ->                  -- event : TInterface (Name {namespace = "Gdk", name = "Event"})
    IO (Ptr Gdk.DragContext.DragContext)

{-# DEPRECATED widgetDragBegin ["(Since version 3.10)","Use 'GI.Gtk.Objects.Widget.widgetDragBeginWithCoordinates' instead"] #-}
-- | This function is equivalent to 'GI.Gtk.Objects.Widget.widgetDragBeginWithCoordinates',
-- passing -1, -1 as coordinates.
widgetDragBegin ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the source widget
    -> Gtk.TargetList.TargetList
    -- ^ /@targets@/: The targets (data formats) in which the
    --    source can provide the data
    -> [Gdk.Flags.DragAction]
    -- ^ /@actions@/: A bitmask of the allowed drag actions for this drag
    -> Int32
    -- ^ /@button@/: The button the user clicked to start the drag
    -> Maybe (Gdk.Event.Event)
    -- ^ /@event@/: The event that triggered the start of the drag,
    --    or 'P.Nothing' if none can be obtained.
    -> m Gdk.DragContext.DragContext
    -- ^ __Returns:__ the context for this drag
widgetDragBegin widget targets actions button event = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    targets' <- unsafeManagedPtrGetPtr targets
    let actions' = gflagsToWord actions
    maybeEvent <- case event of
        Nothing -> return FP.nullPtr
        Just jEvent -> do
            jEvent' <- unsafeManagedPtrGetPtr jEvent
            return jEvent'
    result <- gtk_drag_begin widget' targets' actions' button maybeEvent
    checkUnexpectedReturnNULL "widgetDragBegin" result
    result' <- (newObject Gdk.DragContext.DragContext) result
    touchManagedPtr widget
    touchManagedPtr targets
    whenJust event touchManagedPtr
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetDragBeginMethodInfo
instance (signature ~ (Gtk.TargetList.TargetList -> [Gdk.Flags.DragAction] -> Int32 -> Maybe (Gdk.Event.Event) -> m Gdk.DragContext.DragContext), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragBeginMethodInfo a signature where
    overloadedMethod = widgetDragBegin

instance O.OverloadedMethodInfo WidgetDragBeginMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragBegin",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragBegin"
        })


#endif

-- method Widget::drag_begin_with_coordinates
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the source widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "targets"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "TargetList" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The targets (data formats) in which the\n   source can provide the data"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "actions"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "DragAction" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "A bitmask of the allowed drag actions for this drag"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "button"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "The button the user clicked to start the drag"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "event"
--           , argType = TInterface Name { namespace = "Gdk" , name = "Event" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The event that triggered the start of the drag,\n   or %NULL if none can be obtained."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The initial x coordinate to start dragging from, in the coordinate space\n   of @widget. If -1 is passed, the coordinates are retrieved from @event or\n   the current pointer position"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The initial y coordinate to start dragging from, in the coordinate space\n   of @widget. If -1 is passed, the coordinates are retrieved from @event or\n   the current pointer position"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "DragContext" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_begin_with_coordinates" gtk_drag_begin_with_coordinates :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gtk.TargetList.TargetList ->        -- targets : TInterface (Name {namespace = "Gtk", name = "TargetList"})
    CUInt ->                                -- actions : TInterface (Name {namespace = "Gdk", name = "DragAction"})
    Int32 ->                                -- button : TBasicType TInt
    Ptr Gdk.Event.Event ->                  -- event : TInterface (Name {namespace = "Gdk", name = "Event"})
    Int32 ->                                -- x : TBasicType TInt
    Int32 ->                                -- y : TBasicType TInt
    IO (Ptr Gdk.DragContext.DragContext)

-- | Initiates a drag on the source side. The function only needs to be used
-- when the application is starting drags itself, and is not needed when
-- 'GI.Gtk.Objects.Widget.widgetDragSourceSet' is used.
-- 
-- The /@event@/ is used to retrieve the timestamp that will be used internally to
-- grab the pointer.  If /@event@/ is 'P.Nothing', then 'GI.Gdk.Constants.CURRENT_TIME' will be used.
-- However, you should try to pass a real event in all cases, since that can be
-- used to get information about the drag.
-- 
-- Generally there are three cases when you want to start a drag by hand by
-- calling this function:
-- 
-- 1. During a [Widget::buttonPressEvent]("GI.Gtk.Objects.Widget#g:signal:buttonPressEvent") handler, if you want to start a drag
--    immediately when the user presses the mouse button.  Pass the /@event@/
--    that you have in your [Widget::buttonPressEvent]("GI.Gtk.Objects.Widget#g:signal:buttonPressEvent") handler.
-- 
-- 2. During a [Widget::motionNotifyEvent]("GI.Gtk.Objects.Widget#g:signal:motionNotifyEvent") handler, if you want to start a drag
--    when the mouse moves past a certain threshold distance after a button-press.
--    Pass the /@event@/ that you have in your [Widget::motionNotifyEvent]("GI.Gtk.Objects.Widget#g:signal:motionNotifyEvent") handler.
-- 
-- 3. During a timeout handler, if you want to start a drag after the mouse
--    button is held down for some time.  Try to save the last event that you got
--    from the mouse, using 'GI.Gdk.Unions.Event.eventCopy', and pass it to this function
--    (remember to free the event with 'GI.Gdk.Unions.Event.eventFree' when you are done).
--    If you really cannot pass a real event, pass 'P.Nothing' instead.
-- 
-- 
-- 
-- /Since: 3.10/
widgetDragBeginWithCoordinates ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the source widget
    -> Gtk.TargetList.TargetList
    -- ^ /@targets@/: The targets (data formats) in which the
    --    source can provide the data
    -> [Gdk.Flags.DragAction]
    -- ^ /@actions@/: A bitmask of the allowed drag actions for this drag
    -> Int32
    -- ^ /@button@/: The button the user clicked to start the drag
    -> Maybe (Gdk.Event.Event)
    -- ^ /@event@/: The event that triggered the start of the drag,
    --    or 'P.Nothing' if none can be obtained.
    -> Int32
    -- ^ /@x@/: The initial x coordinate to start dragging from, in the coordinate space
    --    of /@widget@/. If -1 is passed, the coordinates are retrieved from /@event@/ or
    --    the current pointer position
    -> Int32
    -- ^ /@y@/: The initial y coordinate to start dragging from, in the coordinate space
    --    of /@widget@/. If -1 is passed, the coordinates are retrieved from /@event@/ or
    --    the current pointer position
    -> m Gdk.DragContext.DragContext
    -- ^ __Returns:__ the context for this drag
widgetDragBeginWithCoordinates widget targets actions button event x y = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    targets' <- unsafeManagedPtrGetPtr targets
    let actions' = gflagsToWord actions
    maybeEvent <- case event of
        Nothing -> return FP.nullPtr
        Just jEvent -> do
            jEvent' <- unsafeManagedPtrGetPtr jEvent
            return jEvent'
    result <- gtk_drag_begin_with_coordinates widget' targets' actions' button maybeEvent x y
    checkUnexpectedReturnNULL "widgetDragBeginWithCoordinates" result
    result' <- (newObject Gdk.DragContext.DragContext) result
    touchManagedPtr widget
    touchManagedPtr targets
    whenJust event touchManagedPtr
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetDragBeginWithCoordinatesMethodInfo
instance (signature ~ (Gtk.TargetList.TargetList -> [Gdk.Flags.DragAction] -> Int32 -> Maybe (Gdk.Event.Event) -> Int32 -> Int32 -> m Gdk.DragContext.DragContext), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragBeginWithCoordinatesMethodInfo a signature where
    overloadedMethod = widgetDragBeginWithCoordinates

instance O.OverloadedMethodInfo WidgetDragBeginWithCoordinatesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragBeginWithCoordinates",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragBeginWithCoordinates"
        })


#endif

-- method Widget::drag_check_threshold
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_x"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "X coordinate of start of drag"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_y"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "Y coordinate of start of drag"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "current_x"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "current X coordinate"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "current_y"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "current Y coordinate"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_check_threshold" gtk_drag_check_threshold :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Int32 ->                                -- start_x : TBasicType TInt
    Int32 ->                                -- start_y : TBasicType TInt
    Int32 ->                                -- current_x : TBasicType TInt
    Int32 ->                                -- current_y : TBasicType TInt
    IO CInt

-- | Checks to see if a mouse drag starting at (/@startX@/, /@startY@/) and ending
-- at (/@currentX@/, /@currentY@/) has passed the GTK+ drag threshold, and thus
-- should trigger the beginning of a drag-and-drop operation.
widgetDragCheckThreshold ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> Int32
    -- ^ /@startX@/: X coordinate of start of drag
    -> Int32
    -- ^ /@startY@/: Y coordinate of start of drag
    -> Int32
    -- ^ /@currentX@/: current X coordinate
    -> Int32
    -- ^ /@currentY@/: current Y coordinate
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the drag threshold has been passed.
widgetDragCheckThreshold widget startX startY currentX currentY = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_drag_check_threshold widget' startX startY currentX currentY
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetDragCheckThresholdMethodInfo
instance (signature ~ (Int32 -> Int32 -> Int32 -> Int32 -> m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragCheckThresholdMethodInfo a signature where
    overloadedMethod = widgetDragCheckThreshold

instance O.OverloadedMethodInfo WidgetDragCheckThresholdMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragCheckThreshold",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragCheckThreshold"
        })


#endif

-- method Widget::drag_dest_add_image_targets
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget that\8217s a drag destination"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_dest_add_image_targets" gtk_drag_dest_add_image_targets :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Add the image targets supported by t'GI.Gtk.Structs.SelectionData.SelectionData' to
-- the target list of the drag destination. The targets
-- are added with /@info@/ = 0. If you need another value,
-- use 'GI.Gtk.Structs.TargetList.targetListAddImageTargets' and
-- 'GI.Gtk.Objects.Widget.widgetDragDestSetTargetList'.
-- 
-- /Since: 2.6/
widgetDragDestAddImageTargets ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' that’s a drag destination
    -> m ()
widgetDragDestAddImageTargets widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_drag_dest_add_image_targets widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragDestAddImageTargetsMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragDestAddImageTargetsMethodInfo a signature where
    overloadedMethod = widgetDragDestAddImageTargets

instance O.OverloadedMethodInfo WidgetDragDestAddImageTargetsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragDestAddImageTargets",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragDestAddImageTargets"
        })


#endif

-- method Widget::drag_dest_add_text_targets
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget that\8217s a drag destination"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_dest_add_text_targets" gtk_drag_dest_add_text_targets :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Add the text targets supported by t'GI.Gtk.Structs.SelectionData.SelectionData' to
-- the target list of the drag destination. The targets
-- are added with /@info@/ = 0. If you need another value,
-- use 'GI.Gtk.Structs.TargetList.targetListAddTextTargets' and
-- 'GI.Gtk.Objects.Widget.widgetDragDestSetTargetList'.
-- 
-- /Since: 2.6/
widgetDragDestAddTextTargets ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' that’s a drag destination
    -> m ()
widgetDragDestAddTextTargets widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_drag_dest_add_text_targets widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragDestAddTextTargetsMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragDestAddTextTargetsMethodInfo a signature where
    overloadedMethod = widgetDragDestAddTextTargets

instance O.OverloadedMethodInfo WidgetDragDestAddTextTargetsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragDestAddTextTargets",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragDestAddTextTargets"
        })


#endif

-- method Widget::drag_dest_add_uri_targets
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget that\8217s a drag destination"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_dest_add_uri_targets" gtk_drag_dest_add_uri_targets :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Add the URI targets supported by t'GI.Gtk.Structs.SelectionData.SelectionData' to
-- the target list of the drag destination. The targets
-- are added with /@info@/ = 0. If you need another value,
-- use 'GI.Gtk.Structs.TargetList.targetListAddUriTargets' and
-- 'GI.Gtk.Objects.Widget.widgetDragDestSetTargetList'.
-- 
-- /Since: 2.6/
widgetDragDestAddUriTargets ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' that’s a drag destination
    -> m ()
widgetDragDestAddUriTargets widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_drag_dest_add_uri_targets widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragDestAddUriTargetsMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragDestAddUriTargetsMethodInfo a signature where
    overloadedMethod = widgetDragDestAddUriTargets

instance O.OverloadedMethodInfo WidgetDragDestAddUriTargetsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragDestAddUriTargets",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragDestAddUriTargets"
        })


#endif

-- method Widget::drag_dest_find_target
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "drag destination widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "context"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "DragContext" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "drag context" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "target_list"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "TargetList" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "list of droppable targets, or %NULL to use\n   gtk_drag_dest_get_target_list (@widget)."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Atom" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_dest_find_target" gtk_drag_dest_find_target :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.DragContext.DragContext ->      -- context : TInterface (Name {namespace = "Gdk", name = "DragContext"})
    Ptr Gtk.TargetList.TargetList ->        -- target_list : TInterface (Name {namespace = "Gtk", name = "TargetList"})
    IO (Ptr Gdk.Atom.Atom)

-- | Looks for a match between the supported targets of /@context@/ and the
-- /@destTargetList@/, returning the first matching target, otherwise
-- returning @/GDK_NONE/@. /@destTargetList@/ should usually be the return
-- value from 'GI.Gtk.Objects.Widget.widgetDragDestGetTargetList', but some widgets may
-- have different valid targets for different parts of the widget; in
-- that case, they will have to implement a drag_motion handler that
-- passes the correct target list to this function.
widgetDragDestFindTarget ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, Gdk.DragContext.IsDragContext b) =>
    a
    -- ^ /@widget@/: drag destination widget
    -> b
    -- ^ /@context@/: drag context
    -> Maybe (Gtk.TargetList.TargetList)
    -- ^ /@targetList@/: list of droppable targets, or 'P.Nothing' to use
    --    gtk_drag_dest_get_target_list (/@widget@/).
    -> m (Maybe Gdk.Atom.Atom)
    -- ^ __Returns:__ first target that the source offers
    --     and the dest can accept, or @/GDK_NONE/@
widgetDragDestFindTarget widget context targetList = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    context' <- unsafeManagedPtrCastPtr context
    maybeTargetList <- case targetList of
        Nothing -> return FP.nullPtr
        Just jTargetList -> do
            jTargetList' <- unsafeManagedPtrGetPtr jTargetList
            return jTargetList'
    result <- gtk_drag_dest_find_target widget' context' maybeTargetList
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newPtr Gdk.Atom.Atom) result'
        return result''
    touchManagedPtr widget
    touchManagedPtr context
    whenJust targetList touchManagedPtr
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetDragDestFindTargetMethodInfo
instance (signature ~ (b -> Maybe (Gtk.TargetList.TargetList) -> m (Maybe Gdk.Atom.Atom)), MonadIO m, IsWidget a, Gdk.DragContext.IsDragContext b) => O.OverloadedMethod WidgetDragDestFindTargetMethodInfo a signature where
    overloadedMethod = widgetDragDestFindTarget

instance O.OverloadedMethodInfo WidgetDragDestFindTargetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragDestFindTarget",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragDestFindTarget"
        })


#endif

-- method Widget::drag_dest_get_target_list
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "TargetList" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_dest_get_target_list" gtk_drag_dest_get_target_list :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gtk.TargetList.TargetList)

-- | Returns the list of targets this widget can accept from
-- drag-and-drop.
widgetDragDestGetTargetList ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m (Maybe Gtk.TargetList.TargetList)
    -- ^ __Returns:__ the t'GI.Gtk.Structs.TargetList.TargetList', or 'P.Nothing' if none
widgetDragDestGetTargetList widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_drag_dest_get_target_list widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed Gtk.TargetList.TargetList) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetDragDestGetTargetListMethodInfo
instance (signature ~ (m (Maybe Gtk.TargetList.TargetList)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragDestGetTargetListMethodInfo a signature where
    overloadedMethod = widgetDragDestGetTargetList

instance O.OverloadedMethodInfo WidgetDragDestGetTargetListMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragDestGetTargetList",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragDestGetTargetList"
        })


#endif

-- method Widget::drag_dest_get_track_motion
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget that\8217s a drag destination"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_dest_get_track_motion" gtk_drag_dest_get_track_motion :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Returns whether the widget has been configured to always
-- emit [Widget::dragMotion]("GI.Gtk.Objects.Widget#g:signal:dragMotion") signals.
-- 
-- /Since: 2.10/
widgetDragDestGetTrackMotion ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' that’s a drag destination
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the widget always emits
    --   [Widget::dragMotion]("GI.Gtk.Objects.Widget#g:signal:dragMotion") events
widgetDragDestGetTrackMotion widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_drag_dest_get_track_motion widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetDragDestGetTrackMotionMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragDestGetTrackMotionMethodInfo a signature where
    overloadedMethod = widgetDragDestGetTrackMotion

instance O.OverloadedMethodInfo WidgetDragDestGetTrackMotionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragDestGetTrackMotion",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragDestGetTrackMotion"
        })


#endif

-- method Widget::drag_dest_set
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "DestDefaults" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "which types of default drag behavior to use"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "targets"
--           , argType =
--               TCArray
--                 False
--                 (-1)
--                 3
--                 (TInterface Name { namespace = "Gtk" , name = "TargetEntry" })
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a pointer to an array of\n    #GtkTargetEntrys indicating the drop types that this @widget will\n    accept, or %NULL. Later you can access the list with\n    gtk_drag_dest_get_target_list() and gtk_drag_dest_find_target()."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_targets"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of entries in @targets"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "actions"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "DragAction" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "a bitmask of possible actions for a drop onto this @widget."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "n_targets"
--              , argType = TBasicType TInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the number of entries in @targets"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_dest_set" gtk_drag_dest_set :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- flags : TInterface (Name {namespace = "Gtk", name = "DestDefaults"})
    Ptr Gtk.TargetEntry.TargetEntry ->      -- targets : TCArray False (-1) 3 (TInterface (Name {namespace = "Gtk", name = "TargetEntry"}))
    Int32 ->                                -- n_targets : TBasicType TInt
    CUInt ->                                -- actions : TInterface (Name {namespace = "Gdk", name = "DragAction"})
    IO ()

-- | Sets a widget as a potential drop destination, and adds default behaviors.
-- 
-- The default behaviors listed in /@flags@/ have an effect similar
-- to installing default handlers for the widget’s drag-and-drop signals
-- ([Widget::dragMotion]("GI.Gtk.Objects.Widget#g:signal:dragMotion"), [Widget::dragDrop]("GI.Gtk.Objects.Widget#g:signal:dragDrop"), ...). They all exist
-- for convenience. When passing @/GTK_DEST_DEFAULT_ALL/@ for instance it is
-- sufficient to connect to the widget’s [Widget::dragDataReceived]("GI.Gtk.Objects.Widget#g:signal:dragDataReceived")
-- signal to get primitive, but consistent drag-and-drop support.
-- 
-- Things become more complicated when you try to preview the dragged data,
-- as described in the documentation for [Widget::dragMotion]("GI.Gtk.Objects.Widget#g:signal:dragMotion"). The default
-- behaviors described by /@flags@/ make some assumptions, that can conflict
-- with your own signal handlers. For instance @/GTK_DEST_DEFAULT_DROP/@ causes
-- invokations of 'GI.Gdk.Functions.dragStatus' in the context of [Widget::dragMotion]("GI.Gtk.Objects.Widget#g:signal:dragMotion"),
-- and invokations of 'GI.Gtk.Functions.dragFinish' in [Widget::dragDataReceived]("GI.Gtk.Objects.Widget#g:signal:dragDataReceived").
-- Especially the later is dramatic, when your own [Widget::dragMotion]("GI.Gtk.Objects.Widget#g:signal:dragMotion")
-- handler calls 'GI.Gtk.Objects.Widget.widgetDragGetData' to inspect the dragged data.
-- 
-- There’s no way to set a default action here, you can use the
-- [Widget::dragMotion]("GI.Gtk.Objects.Widget#g:signal:dragMotion") callback for that. Here’s an example which selects
-- the action to use depending on whether the control key is pressed or not:
-- 
-- === /C code/
-- >
-- >static void
-- >drag_motion (GtkWidget *widget,
-- >             GdkDragContext *context,
-- >             gint x,
-- >             gint y,
-- >             guint time)
-- >{
-- >  GdkModifierType mask;
-- >
-- >  gdk_window_get_pointer (gtk_widget_get_window (widget),
-- >                          NULL, NULL, &mask);
-- >  if (mask & GDK_CONTROL_MASK)
-- >    gdk_drag_status (context, GDK_ACTION_COPY, time);
-- >  else
-- >    gdk_drag_status (context, GDK_ACTION_MOVE, time);
-- >}
widgetDragDestSet ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> [Gtk.Flags.DestDefaults]
    -- ^ /@flags@/: which types of default drag behavior to use
    -> Maybe ([Gtk.TargetEntry.TargetEntry])
    -- ^ /@targets@/: a pointer to an array of
    --     @/GtkTargetEntrys/@ indicating the drop types that this /@widget@/ will
    --     accept, or 'P.Nothing'. Later you can access the list with
    --     'GI.Gtk.Objects.Widget.widgetDragDestGetTargetList' and 'GI.Gtk.Objects.Widget.widgetDragDestFindTarget'.
    -> [Gdk.Flags.DragAction]
    -- ^ /@actions@/: a bitmask of possible actions for a drop onto this /@widget@/.
    -> m ()
widgetDragDestSet widget flags targets actions = liftIO $ do
    let nTargets = case targets of
            Nothing -> 0
            Just jTargets -> fromIntegral $ P.length jTargets
    widget' <- unsafeManagedPtrCastPtr widget
    let flags' = gflagsToWord flags
    maybeTargets <- case targets of
        Nothing -> return FP.nullPtr
        Just jTargets -> do
            jTargets' <- mapM unsafeManagedPtrGetPtr jTargets
            jTargets'' <- packBlockArray 16 jTargets'
            return jTargets''
    let actions' = gflagsToWord actions
    gtk_drag_dest_set widget' flags' maybeTargets nTargets actions'
    touchManagedPtr widget
    whenJust targets (mapM_ touchManagedPtr)
    freeMem maybeTargets
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragDestSetMethodInfo
instance (signature ~ ([Gtk.Flags.DestDefaults] -> Maybe ([Gtk.TargetEntry.TargetEntry]) -> [Gdk.Flags.DragAction] -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragDestSetMethodInfo a signature where
    overloadedMethod = widgetDragDestSet

instance O.OverloadedMethodInfo WidgetDragDestSetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragDestSet",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragDestSet"
        })


#endif

-- method Widget::drag_dest_set_proxy
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "proxy_window"
--           , argType = TInterface Name { namespace = "Gdk" , name = "Window" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the window to which to forward drag events"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "protocol"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "DragProtocol" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the drag protocol which the @proxy_window accepts\n  (You can use gdk_drag_get_protocol() to determine this)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "use_coordinates"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "If %TRUE, send the same coordinates to the\n  destination, because it is an embedded\n  subwindow."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_dest_set_proxy" gtk_drag_dest_set_proxy :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.Window.Window ->                -- proxy_window : TInterface (Name {namespace = "Gdk", name = "Window"})
    CUInt ->                                -- protocol : TInterface (Name {namespace = "Gdk", name = "DragProtocol"})
    CInt ->                                 -- use_coordinates : TBasicType TBoolean
    IO ()

{-# DEPRECATED widgetDragDestSetProxy ["(Since version 3.22)"] #-}
-- | Sets this widget as a proxy for drops to another window.
widgetDragDestSetProxy ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, Gdk.Window.IsWindow b) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> b
    -- ^ /@proxyWindow@/: the window to which to forward drag events
    -> Gdk.Enums.DragProtocol
    -- ^ /@protocol@/: the drag protocol which the /@proxyWindow@/ accepts
    --   (You can use @/gdk_drag_get_protocol()/@ to determine this)
    -> Bool
    -- ^ /@useCoordinates@/: If 'P.True', send the same coordinates to the
    --   destination, because it is an embedded
    --   subwindow.
    -> m ()
widgetDragDestSetProxy widget proxyWindow protocol useCoordinates = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    proxyWindow' <- unsafeManagedPtrCastPtr proxyWindow
    let protocol' = (fromIntegral . fromEnum) protocol
    let useCoordinates' = (P.fromIntegral . P.fromEnum) useCoordinates
    gtk_drag_dest_set_proxy widget' proxyWindow' protocol' useCoordinates'
    touchManagedPtr widget
    touchManagedPtr proxyWindow
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragDestSetProxyMethodInfo
instance (signature ~ (b -> Gdk.Enums.DragProtocol -> Bool -> m ()), MonadIO m, IsWidget a, Gdk.Window.IsWindow b) => O.OverloadedMethod WidgetDragDestSetProxyMethodInfo a signature where
    overloadedMethod = widgetDragDestSetProxy

instance O.OverloadedMethodInfo WidgetDragDestSetProxyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragDestSetProxy",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragDestSetProxy"
        })


#endif

-- method Widget::drag_dest_set_target_list
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget that\8217s a drag destination"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "target_list"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "TargetList" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "list of droppable targets, or %NULL for none"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_dest_set_target_list" gtk_drag_dest_set_target_list :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gtk.TargetList.TargetList ->        -- target_list : TInterface (Name {namespace = "Gtk", name = "TargetList"})
    IO ()

-- | Sets the target types that this widget can accept from drag-and-drop.
-- The widget must first be made into a drag destination with
-- 'GI.Gtk.Objects.Widget.widgetDragDestSet'.
widgetDragDestSetTargetList ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' that’s a drag destination
    -> Maybe (Gtk.TargetList.TargetList)
    -- ^ /@targetList@/: list of droppable targets, or 'P.Nothing' for none
    -> m ()
widgetDragDestSetTargetList widget targetList = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    maybeTargetList <- case targetList of
        Nothing -> return FP.nullPtr
        Just jTargetList -> do
            jTargetList' <- unsafeManagedPtrGetPtr jTargetList
            return jTargetList'
    gtk_drag_dest_set_target_list widget' maybeTargetList
    touchManagedPtr widget
    whenJust targetList touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragDestSetTargetListMethodInfo
instance (signature ~ (Maybe (Gtk.TargetList.TargetList) -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragDestSetTargetListMethodInfo a signature where
    overloadedMethod = widgetDragDestSetTargetList

instance O.OverloadedMethodInfo WidgetDragDestSetTargetListMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragDestSetTargetList",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragDestSetTargetList"
        })


#endif

-- method Widget::drag_dest_set_track_motion
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget that\8217s a drag destination"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "track_motion"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "whether to accept all targets"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_dest_set_track_motion" gtk_drag_dest_set_track_motion :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CInt ->                                 -- track_motion : TBasicType TBoolean
    IO ()

-- | Tells the widget to emit [Widget::dragMotion]("GI.Gtk.Objects.Widget#g:signal:dragMotion") and
-- [Widget::dragLeave]("GI.Gtk.Objects.Widget#g:signal:dragLeave") events regardless of the targets and the
-- 'GI.Gtk.Flags.DestDefaultsMotion' flag.
-- 
-- This may be used when a widget wants to do generic
-- actions regardless of the targets that the source offers.
-- 
-- /Since: 2.10/
widgetDragDestSetTrackMotion ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' that’s a drag destination
    -> Bool
    -- ^ /@trackMotion@/: whether to accept all targets
    -> m ()
widgetDragDestSetTrackMotion widget trackMotion = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let trackMotion' = (P.fromIntegral . P.fromEnum) trackMotion
    gtk_drag_dest_set_track_motion widget' trackMotion'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragDestSetTrackMotionMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragDestSetTrackMotionMethodInfo a signature where
    overloadedMethod = widgetDragDestSetTrackMotion

instance O.OverloadedMethodInfo WidgetDragDestSetTrackMotionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragDestSetTrackMotion",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragDestSetTrackMotion"
        })


#endif

-- method Widget::drag_dest_unset
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_dest_unset" gtk_drag_dest_unset :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Clears information about a drop destination set with
-- 'GI.Gtk.Objects.Widget.widgetDragDestSet'. The widget will no longer receive
-- notification of drags.
widgetDragDestUnset ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m ()
widgetDragDestUnset widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_drag_dest_unset widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragDestUnsetMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragDestUnsetMethodInfo a signature where
    overloadedMethod = widgetDragDestUnset

instance O.OverloadedMethodInfo WidgetDragDestUnsetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragDestUnset",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragDestUnset"
        })


#endif

-- method Widget::drag_get_data
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the widget that will receive the\n  #GtkWidget::drag-data-received signal"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "context"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "DragContext" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the drag context" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "target"
--           , argType = TInterface Name { namespace = "Gdk" , name = "Atom" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the target (form of the data) to retrieve"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "time_"
--           , argType = TBasicType TUInt32
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a timestamp for retrieving the data. This will\n  generally be the time received in a #GtkWidget::drag-motion\n  or #GtkWidget::drag-drop signal"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_get_data" gtk_drag_get_data :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.DragContext.DragContext ->      -- context : TInterface (Name {namespace = "Gdk", name = "DragContext"})
    Ptr Gdk.Atom.Atom ->                    -- target : TInterface (Name {namespace = "Gdk", name = "Atom"})
    Word32 ->                               -- time_ : TBasicType TUInt32
    IO ()

-- | Gets the data associated with a drag. When the data
-- is received or the retrieval fails, GTK+ will emit a
-- [Widget::dragDataReceived]("GI.Gtk.Objects.Widget#g:signal:dragDataReceived") signal. Failure of the retrieval
-- is indicated by the length field of the /@selectionData@/
-- signal parameter being negative. However, when 'GI.Gtk.Objects.Widget.widgetDragGetData'
-- is called implicitely because the 'GI.Gtk.Flags.DestDefaultsDrop' was set,
-- then the widget will not receive notification of failed
-- drops.
widgetDragGetData ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, Gdk.DragContext.IsDragContext b) =>
    a
    -- ^ /@widget@/: the widget that will receive the
    --   [Widget::dragDataReceived]("GI.Gtk.Objects.Widget#g:signal:dragDataReceived") signal
    -> b
    -- ^ /@context@/: the drag context
    -> Gdk.Atom.Atom
    -- ^ /@target@/: the target (form of the data) to retrieve
    -> Word32
    -- ^ /@time_@/: a timestamp for retrieving the data. This will
    --   generally be the time received in a [Widget::dragMotion]("GI.Gtk.Objects.Widget#g:signal:dragMotion")
    --   or [Widget::dragDrop]("GI.Gtk.Objects.Widget#g:signal:dragDrop") signal
    -> m ()
widgetDragGetData widget context target time_ = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    context' <- unsafeManagedPtrCastPtr context
    target' <- unsafeManagedPtrGetPtr target
    gtk_drag_get_data widget' context' target' time_
    touchManagedPtr widget
    touchManagedPtr context
    touchManagedPtr target
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragGetDataMethodInfo
instance (signature ~ (b -> Gdk.Atom.Atom -> Word32 -> m ()), MonadIO m, IsWidget a, Gdk.DragContext.IsDragContext b) => O.OverloadedMethod WidgetDragGetDataMethodInfo a signature where
    overloadedMethod = widgetDragGetData

instance O.OverloadedMethodInfo WidgetDragGetDataMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragGetData",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragGetData"
        })


#endif

-- method Widget::drag_highlight
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget to highlight"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_highlight" gtk_drag_highlight :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Highlights a widget as a currently hovered drop target.
-- To end the highlight, call 'GI.Gtk.Objects.Widget.widgetDragUnhighlight'.
-- GTK+ calls this automatically if 'GI.Gtk.Flags.DestDefaultsHighlight' is set.
widgetDragHighlight ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget to highlight
    -> m ()
widgetDragHighlight widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_drag_highlight widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragHighlightMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragHighlightMethodInfo a signature where
    overloadedMethod = widgetDragHighlight

instance O.OverloadedMethodInfo WidgetDragHighlightMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragHighlight",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragHighlight"
        })


#endif

-- method Widget::drag_source_add_image_targets
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget that\8217s is a drag source"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_source_add_image_targets" gtk_drag_source_add_image_targets :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Add the writable image targets supported by t'GI.Gtk.Structs.SelectionData.SelectionData' to
-- the target list of the drag source. The targets
-- are added with /@info@/ = 0. If you need another value,
-- use 'GI.Gtk.Structs.TargetList.targetListAddImageTargets' and
-- 'GI.Gtk.Objects.Widget.widgetDragSourceSetTargetList'.
-- 
-- /Since: 2.6/
widgetDragSourceAddImageTargets ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' that’s is a drag source
    -> m ()
widgetDragSourceAddImageTargets widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_drag_source_add_image_targets widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragSourceAddImageTargetsMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragSourceAddImageTargetsMethodInfo a signature where
    overloadedMethod = widgetDragSourceAddImageTargets

instance O.OverloadedMethodInfo WidgetDragSourceAddImageTargetsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragSourceAddImageTargets",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragSourceAddImageTargets"
        })


#endif

-- method Widget::drag_source_add_text_targets
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget that\8217s is a drag source"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_source_add_text_targets" gtk_drag_source_add_text_targets :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Add the text targets supported by t'GI.Gtk.Structs.SelectionData.SelectionData' to
-- the target list of the drag source.  The targets
-- are added with /@info@/ = 0. If you need another value,
-- use 'GI.Gtk.Structs.TargetList.targetListAddTextTargets' and
-- 'GI.Gtk.Objects.Widget.widgetDragSourceSetTargetList'.
-- 
-- /Since: 2.6/
widgetDragSourceAddTextTargets ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' that’s is a drag source
    -> m ()
widgetDragSourceAddTextTargets widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_drag_source_add_text_targets widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragSourceAddTextTargetsMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragSourceAddTextTargetsMethodInfo a signature where
    overloadedMethod = widgetDragSourceAddTextTargets

instance O.OverloadedMethodInfo WidgetDragSourceAddTextTargetsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragSourceAddTextTargets",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragSourceAddTextTargets"
        })


#endif

-- method Widget::drag_source_add_uri_targets
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget that\8217s is a drag source"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_source_add_uri_targets" gtk_drag_source_add_uri_targets :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Add the URI targets supported by t'GI.Gtk.Structs.SelectionData.SelectionData' to
-- the target list of the drag source.  The targets
-- are added with /@info@/ = 0. If you need another value,
-- use 'GI.Gtk.Structs.TargetList.targetListAddUriTargets' and
-- 'GI.Gtk.Objects.Widget.widgetDragSourceSetTargetList'.
-- 
-- /Since: 2.6/
widgetDragSourceAddUriTargets ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' that’s is a drag source
    -> m ()
widgetDragSourceAddUriTargets widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_drag_source_add_uri_targets widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragSourceAddUriTargetsMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragSourceAddUriTargetsMethodInfo a signature where
    overloadedMethod = widgetDragSourceAddUriTargets

instance O.OverloadedMethodInfo WidgetDragSourceAddUriTargetsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragSourceAddUriTargets",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragSourceAddUriTargets"
        })


#endif

-- method Widget::drag_source_get_target_list
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "TargetList" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_source_get_target_list" gtk_drag_source_get_target_list :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gtk.TargetList.TargetList)

-- | Gets the list of targets this widget can provide for
-- drag-and-drop.
-- 
-- /Since: 2.4/
widgetDragSourceGetTargetList ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m (Maybe Gtk.TargetList.TargetList)
    -- ^ __Returns:__ the t'GI.Gtk.Structs.TargetList.TargetList', or 'P.Nothing' if none
widgetDragSourceGetTargetList widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_drag_source_get_target_list widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed Gtk.TargetList.TargetList) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetDragSourceGetTargetListMethodInfo
instance (signature ~ (m (Maybe Gtk.TargetList.TargetList)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragSourceGetTargetListMethodInfo a signature where
    overloadedMethod = widgetDragSourceGetTargetList

instance O.OverloadedMethodInfo WidgetDragSourceGetTargetListMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragSourceGetTargetList",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragSourceGetTargetList"
        })


#endif

-- method Widget::drag_source_set
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_button_mask"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "ModifierType" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the bitmask of buttons that can start the drag"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "targets"
--           , argType =
--               TCArray
--                 False
--                 (-1)
--                 3
--                 (TInterface Name { namespace = "Gtk" , name = "TargetEntry" })
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the table of targets\n    that the drag will support, may be %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_targets"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of items in @targets"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "actions"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "DragAction" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the bitmask of possible actions for a drag from this widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "n_targets"
--              , argType = TBasicType TInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the number of items in @targets"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_source_set" gtk_drag_source_set :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- start_button_mask : TInterface (Name {namespace = "Gdk", name = "ModifierType"})
    Ptr Gtk.TargetEntry.TargetEntry ->      -- targets : TCArray False (-1) 3 (TInterface (Name {namespace = "Gtk", name = "TargetEntry"}))
    Int32 ->                                -- n_targets : TBasicType TInt
    CUInt ->                                -- actions : TInterface (Name {namespace = "Gdk", name = "DragAction"})
    IO ()

-- | Sets up a widget so that GTK+ will start a drag operation when the user
-- clicks and drags on the widget. The widget must have a window.
widgetDragSourceSet ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> [Gdk.Flags.ModifierType]
    -- ^ /@startButtonMask@/: the bitmask of buttons that can start the drag
    -> Maybe ([Gtk.TargetEntry.TargetEntry])
    -- ^ /@targets@/: the table of targets
    --     that the drag will support, may be 'P.Nothing'
    -> [Gdk.Flags.DragAction]
    -- ^ /@actions@/: the bitmask of possible actions for a drag from this widget
    -> m ()
widgetDragSourceSet widget startButtonMask targets actions = liftIO $ do
    let nTargets = case targets of
            Nothing -> 0
            Just jTargets -> fromIntegral $ P.length jTargets
    widget' <- unsafeManagedPtrCastPtr widget
    let startButtonMask' = gflagsToWord startButtonMask
    maybeTargets <- case targets of
        Nothing -> return FP.nullPtr
        Just jTargets -> do
            jTargets' <- mapM unsafeManagedPtrGetPtr jTargets
            jTargets'' <- packBlockArray 16 jTargets'
            return jTargets''
    let actions' = gflagsToWord actions
    gtk_drag_source_set widget' startButtonMask' maybeTargets nTargets actions'
    touchManagedPtr widget
    whenJust targets (mapM_ touchManagedPtr)
    freeMem maybeTargets
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragSourceSetMethodInfo
instance (signature ~ ([Gdk.Flags.ModifierType] -> Maybe ([Gtk.TargetEntry.TargetEntry]) -> [Gdk.Flags.DragAction] -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragSourceSetMethodInfo a signature where
    overloadedMethod = widgetDragSourceSet

instance O.OverloadedMethodInfo WidgetDragSourceSetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragSourceSet",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragSourceSet"
        })


#endif

-- method Widget::drag_source_set_icon_gicon
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "icon"
--           , argType = TInterface Name { namespace = "Gio" , name = "Icon" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GIcon" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_source_set_icon_gicon" gtk_drag_source_set_icon_gicon :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gio.Icon.Icon ->                    -- icon : TInterface (Name {namespace = "Gio", name = "Icon"})
    IO ()

-- | Sets the icon that will be used for drags from a particular source
-- to /@icon@/. See the docs for t'GI.Gtk.Objects.IconTheme.IconTheme' for more details.
-- 
-- /Since: 3.2/
widgetDragSourceSetIconGicon ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, Gio.Icon.IsIcon b) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> b
    -- ^ /@icon@/: A t'GI.Gio.Interfaces.Icon.Icon'
    -> m ()
widgetDragSourceSetIconGicon widget icon = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    icon' <- unsafeManagedPtrCastPtr icon
    gtk_drag_source_set_icon_gicon widget' icon'
    touchManagedPtr widget
    touchManagedPtr icon
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragSourceSetIconGiconMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsWidget a, Gio.Icon.IsIcon b) => O.OverloadedMethod WidgetDragSourceSetIconGiconMethodInfo a signature where
    overloadedMethod = widgetDragSourceSetIconGicon

instance O.OverloadedMethodInfo WidgetDragSourceSetIconGiconMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragSourceSetIconGicon",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragSourceSetIconGicon"
        })


#endif

-- method Widget::drag_source_set_icon_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "icon_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "name of icon to use"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_source_set_icon_name" gtk_drag_source_set_icon_name :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- icon_name : TBasicType TUTF8
    IO ()

-- | Sets the icon that will be used for drags from a particular source
-- to a themed icon. See the docs for t'GI.Gtk.Objects.IconTheme.IconTheme' for more details.
-- 
-- /Since: 2.8/
widgetDragSourceSetIconName ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> T.Text
    -- ^ /@iconName@/: name of icon to use
    -> m ()
widgetDragSourceSetIconName widget iconName = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    iconName' <- textToCString iconName
    gtk_drag_source_set_icon_name widget' iconName'
    touchManagedPtr widget
    freeMem iconName'
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragSourceSetIconNameMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragSourceSetIconNameMethodInfo a signature where
    overloadedMethod = widgetDragSourceSetIconName

instance O.OverloadedMethodInfo WidgetDragSourceSetIconNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragSourceSetIconName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragSourceSetIconName"
        })


#endif

-- method Widget::drag_source_set_icon_pixbuf
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "pixbuf"
--           , argType =
--               TInterface Name { namespace = "GdkPixbuf" , name = "Pixbuf" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GdkPixbuf for the drag icon"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_source_set_icon_pixbuf" gtk_drag_source_set_icon_pixbuf :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr GdkPixbuf.Pixbuf.Pixbuf ->          -- pixbuf : TInterface (Name {namespace = "GdkPixbuf", name = "Pixbuf"})
    IO ()

-- | Sets the icon that will be used for drags from a particular widget
-- from a t'GI.GdkPixbuf.Objects.Pixbuf.Pixbuf'. GTK+ retains a reference for /@pixbuf@/ and will
-- release it when it is no longer needed.
widgetDragSourceSetIconPixbuf ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, GdkPixbuf.Pixbuf.IsPixbuf b) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> b
    -- ^ /@pixbuf@/: the t'GI.GdkPixbuf.Objects.Pixbuf.Pixbuf' for the drag icon
    -> m ()
widgetDragSourceSetIconPixbuf widget pixbuf = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    pixbuf' <- unsafeManagedPtrCastPtr pixbuf
    gtk_drag_source_set_icon_pixbuf widget' pixbuf'
    touchManagedPtr widget
    touchManagedPtr pixbuf
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragSourceSetIconPixbufMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsWidget a, GdkPixbuf.Pixbuf.IsPixbuf b) => O.OverloadedMethod WidgetDragSourceSetIconPixbufMethodInfo a signature where
    overloadedMethod = widgetDragSourceSetIconPixbuf

instance O.OverloadedMethodInfo WidgetDragSourceSetIconPixbufMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragSourceSetIconPixbuf",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragSourceSetIconPixbuf"
        })


#endif

-- method Widget::drag_source_set_icon_stock
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "stock_id"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the ID of the stock icon to use"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_source_set_icon_stock" gtk_drag_source_set_icon_stock :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- stock_id : TBasicType TUTF8
    IO ()

{-# DEPRECATED widgetDragSourceSetIconStock ["(Since version 3.10)","Use 'GI.Gtk.Objects.Widget.widgetDragSourceSetIconName' instead."] #-}
-- | Sets the icon that will be used for drags from a particular source
-- to a stock icon.
widgetDragSourceSetIconStock ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> T.Text
    -- ^ /@stockId@/: the ID of the stock icon to use
    -> m ()
widgetDragSourceSetIconStock widget stockId = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    stockId' <- textToCString stockId
    gtk_drag_source_set_icon_stock widget' stockId'
    touchManagedPtr widget
    freeMem stockId'
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragSourceSetIconStockMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragSourceSetIconStockMethodInfo a signature where
    overloadedMethod = widgetDragSourceSetIconStock

instance O.OverloadedMethodInfo WidgetDragSourceSetIconStockMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragSourceSetIconStock",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragSourceSetIconStock"
        })


#endif

-- method Widget::drag_source_set_target_list
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget that\8217s a drag source"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "target_list"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "TargetList" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "list of draggable targets, or %NULL for none"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_source_set_target_list" gtk_drag_source_set_target_list :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gtk.TargetList.TargetList ->        -- target_list : TInterface (Name {namespace = "Gtk", name = "TargetList"})
    IO ()

-- | Changes the target types that this widget offers for drag-and-drop.
-- The widget must first be made into a drag source with
-- 'GI.Gtk.Objects.Widget.widgetDragSourceSet'.
-- 
-- /Since: 2.4/
widgetDragSourceSetTargetList ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' that’s a drag source
    -> Maybe (Gtk.TargetList.TargetList)
    -- ^ /@targetList@/: list of draggable targets, or 'P.Nothing' for none
    -> m ()
widgetDragSourceSetTargetList widget targetList = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    maybeTargetList <- case targetList of
        Nothing -> return FP.nullPtr
        Just jTargetList -> do
            jTargetList' <- unsafeManagedPtrGetPtr jTargetList
            return jTargetList'
    gtk_drag_source_set_target_list widget' maybeTargetList
    touchManagedPtr widget
    whenJust targetList touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragSourceSetTargetListMethodInfo
instance (signature ~ (Maybe (Gtk.TargetList.TargetList) -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragSourceSetTargetListMethodInfo a signature where
    overloadedMethod = widgetDragSourceSetTargetList

instance O.OverloadedMethodInfo WidgetDragSourceSetTargetListMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragSourceSetTargetList",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragSourceSetTargetList"
        })


#endif

-- method Widget::drag_source_unset
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_source_unset" gtk_drag_source_unset :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Undoes the effects of 'GI.Gtk.Objects.Widget.widgetDragSourceSet'.
widgetDragSourceUnset ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m ()
widgetDragSourceUnset widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_drag_source_unset widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragSourceUnsetMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragSourceUnsetMethodInfo a signature where
    overloadedMethod = widgetDragSourceUnset

instance O.OverloadedMethodInfo WidgetDragSourceUnsetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragSourceUnset",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragSourceUnset"
        })


#endif

-- method Widget::drag_unhighlight
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget to remove the highlight from"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_drag_unhighlight" gtk_drag_unhighlight :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Removes a highlight set by 'GI.Gtk.Objects.Widget.widgetDragHighlight' from
-- a widget.
widgetDragUnhighlight ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a widget to remove the highlight from
    -> m ()
widgetDragUnhighlight widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_drag_unhighlight widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDragUnhighlightMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDragUnhighlightMethodInfo a signature where
    overloadedMethod = widgetDragUnhighlight

instance O.OverloadedMethodInfo WidgetDragUnhighlightMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDragUnhighlight",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDragUnhighlight"
        })


#endif

-- method Widget::draw
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the widget to draw. It must be drawable (see\n  gtk_widget_is_drawable()) and a size must have been allocated."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "cr"
--           , argType =
--               TInterface Name { namespace = "cairo" , name = "Context" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a cairo context to draw to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_draw" gtk_widget_draw :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Cairo.Context.Context ->            -- cr : TInterface (Name {namespace = "cairo", name = "Context"})
    IO ()

-- | Draws /@widget@/ to /@cr@/. The top left corner of the widget will be
-- drawn to the currently set origin point of /@cr@/.
-- 
-- You should pass a cairo context as /@cr@/ argument that is in an
-- original state. Otherwise the resulting drawing is undefined. For
-- example changing the operator using @/cairo_set_operator()/@ or the
-- line width using @/cairo_set_line_width()/@ might have unwanted side
-- effects.
-- You may however change the context’s transform matrix - like with
-- @/cairo_scale()/@, @/cairo_translate()/@ or @/cairo_set_matrix()/@ and clip
-- region with @/cairo_clip()/@ prior to calling this function. Also, it
-- is fine to modify the context with @/cairo_save()/@ and
-- @/cairo_push_group()/@ prior to calling this function.
-- 
-- Note that special-purpose widgets may contain special code for
-- rendering to the screen and might appear differently on screen
-- and when rendered using 'GI.Gtk.Objects.Widget.widgetDraw'.
-- 
-- /Since: 3.0/
widgetDraw ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget to draw. It must be drawable (see
    --   'GI.Gtk.Objects.Widget.widgetIsDrawable') and a size must have been allocated.
    -> Cairo.Context.Context
    -- ^ /@cr@/: a cairo context to draw to
    -> m ()
widgetDraw widget cr = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    cr' <- unsafeManagedPtrGetPtr cr
    gtk_widget_draw widget' cr'
    touchManagedPtr widget
    touchManagedPtr cr
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetDrawMethodInfo
instance (signature ~ (Cairo.Context.Context -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetDrawMethodInfo a signature where
    overloadedMethod = widgetDraw

instance O.OverloadedMethodInfo WidgetDrawMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetDraw",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetDraw"
        })


#endif

-- method Widget::ensure_style
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_ensure_style" gtk_widget_ensure_style :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

{-# DEPRECATED widgetEnsureStyle ["(Since version 3.0)","Use t'GI.Gtk.Objects.StyleContext.StyleContext' instead"] #-}
-- | Ensures that /@widget@/ has a style (/@widget@/->style).
-- 
-- Not a very useful function; most of the time, if you
-- want the style, the widget is realized, and realized
-- widgets are guaranteed to have a style already.
widgetEnsureStyle ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m ()
widgetEnsureStyle widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_ensure_style widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetEnsureStyleMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetEnsureStyleMethodInfo a signature where
    overloadedMethod = widgetEnsureStyle

instance O.OverloadedMethodInfo WidgetEnsureStyleMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetEnsureStyle",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetEnsureStyle"
        })


#endif

-- method Widget::error_bell
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_error_bell" gtk_widget_error_bell :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Notifies the user about an input-related error on this widget.
-- If the [Settings:gtkErrorBell]("GI.Gtk.Objects.Settings#g:attr:gtkErrorBell") setting is 'P.True', it calls
-- 'GI.Gdk.Objects.Window.windowBeep', otherwise it does nothing.
-- 
-- Note that the effect of 'GI.Gdk.Objects.Window.windowBeep' can be configured in many
-- ways, depending on the windowing backend and the desktop environment
-- or window manager that is used.
-- 
-- /Since: 2.12/
widgetErrorBell ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m ()
widgetErrorBell widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_error_bell widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetErrorBellMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetErrorBellMethodInfo a signature where
    overloadedMethod = widgetErrorBell

instance O.OverloadedMethodInfo WidgetErrorBellMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetErrorBell",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetErrorBell"
        })


#endif

-- method Widget::event
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "event"
--           , argType = TInterface Name { namespace = "Gdk" , name = "Event" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GdkEvent" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_event" gtk_widget_event :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.Event.Event ->                  -- event : TInterface (Name {namespace = "Gdk", name = "Event"})
    IO CInt

-- | Rarely-used function. This function is used to emit
-- the event signals on a widget (those signals should never
-- be emitted without using this function to do so).
-- If you want to synthesize an event though, don’t use this function;
-- instead, use 'GI.Gtk.Functions.mainDoEvent' so the event will behave as if
-- it were in the event queue. Don’t synthesize expose events; instead,
-- use 'GI.Gdk.Objects.Window.windowInvalidateRect' to invalidate a region of the
-- window.
widgetEvent ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> Gdk.Event.Event
    -- ^ /@event@/: a t'GI.Gdk.Unions.Event.Event'
    -> m Bool
    -- ^ __Returns:__ return from the event signal emission ('P.True' if
    --               the event was handled)
widgetEvent widget event = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    event' <- unsafeManagedPtrGetPtr event
    result <- gtk_widget_event widget' event'
    let result' = (/= 0) result
    touchManagedPtr widget
    touchManagedPtr event
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetEventMethodInfo
instance (signature ~ (Gdk.Event.Event -> m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetEventMethodInfo a signature where
    overloadedMethod = widgetEvent

instance O.OverloadedMethodInfo WidgetEventMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetEvent",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetEvent"
        })


#endif

-- method Widget::freeze_child_notify
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_freeze_child_notify" gtk_widget_freeze_child_notify :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Stops emission of [Widget::childNotify]("GI.Gtk.Objects.Widget#g:signal:childNotify") signals on /@widget@/. The
-- signals are queued until 'GI.Gtk.Objects.Widget.widgetThawChildNotify' is called
-- on /@widget@/.
-- 
-- This is the analogue of 'GI.GObject.Objects.Object.objectFreezeNotify' for child properties.
widgetFreezeChildNotify ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m ()
widgetFreezeChildNotify widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_freeze_child_notify widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetFreezeChildNotifyMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetFreezeChildNotifyMethodInfo a signature where
    overloadedMethod = widgetFreezeChildNotify

instance O.OverloadedMethodInfo WidgetFreezeChildNotifyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetFreezeChildNotify",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetFreezeChildNotify"
        })


#endif

-- method Widget::get_accessible
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Atk" , name = "Object" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_accessible" gtk_widget_get_accessible :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Atk.Object.Object)

-- | Returns the accessible object that describes the widget to an
-- assistive technology.
-- 
-- If accessibility support is not available, this t'GI.Atk.Objects.Object.Object'
-- instance may be a no-op. Likewise, if no class-specific t'GI.Atk.Objects.Object.Object'
-- implementation is available for the widget instance in question,
-- it will inherit an t'GI.Atk.Objects.Object.Object' implementation from the first ancestor
-- class for which such an implementation is defined.
-- 
-- The documentation of the
-- <http://developer.gnome.org/atk/stable/ ATK>
-- library contains more information about accessible objects and their uses.
widgetGetAccessible ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Atk.Object.Object
    -- ^ __Returns:__ the t'GI.Atk.Objects.Object.Object' associated with /@widget@/
widgetGetAccessible widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_accessible widget'
    checkUnexpectedReturnNULL "widgetGetAccessible" result
    result' <- (newObject Atk.Object.Object) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetAccessibleMethodInfo
instance (signature ~ (m Atk.Object.Object), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetAccessibleMethodInfo a signature where
    overloadedMethod = widgetGetAccessible

instance O.OverloadedMethodInfo WidgetGetAccessibleMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetAccessible",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetAccessible"
        })


#endif

-- method Widget::get_action_group
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "prefix"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "The \8220prefix\8221 of the action group."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gio" , name = "ActionGroup" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_action_group" gtk_widget_get_action_group :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- prefix : TBasicType TUTF8
    IO (Ptr Gio.ActionGroup.ActionGroup)

-- | Retrieves the t'GI.Gio.Interfaces.ActionGroup.ActionGroup' that was registered using /@prefix@/. The resulting
-- t'GI.Gio.Interfaces.ActionGroup.ActionGroup' may have been registered to /@widget@/ or any t'GI.Gtk.Objects.Widget.Widget' in its
-- ancestry.
-- 
-- If no action group was found matching /@prefix@/, then 'P.Nothing' is returned.
-- 
-- /Since: 3.16/
widgetGetActionGroup ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: A t'GI.Gtk.Objects.Widget.Widget'
    -> T.Text
    -- ^ /@prefix@/: The “prefix” of the action group.
    -> m (Maybe Gio.ActionGroup.ActionGroup)
    -- ^ __Returns:__ A t'GI.Gio.Interfaces.ActionGroup.ActionGroup' or 'P.Nothing'.
widgetGetActionGroup widget prefix = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    prefix' <- textToCString prefix
    result <- gtk_widget_get_action_group widget' prefix'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gio.ActionGroup.ActionGroup) result'
        return result''
    touchManagedPtr widget
    freeMem prefix'
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetActionGroupMethodInfo
instance (signature ~ (T.Text -> m (Maybe Gio.ActionGroup.ActionGroup)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetActionGroupMethodInfo a signature where
    overloadedMethod = widgetGetActionGroup

instance O.OverloadedMethodInfo WidgetGetActionGroupMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetActionGroup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetActionGroup"
        })


#endif

-- method Widget::get_allocated_baseline
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_allocated_baseline" gtk_widget_get_allocated_baseline :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Returns the baseline that has currently been allocated to /@widget@/.
-- This function is intended to be used when implementing handlers
-- for the [Widget::draw]("GI.Gtk.Objects.Widget#g:signal:draw") function, and when allocating child
-- widgets in t'GI.Gtk.Objects.Widget.Widget'::@/size_allocate/@.
-- 
-- /Since: 3.10/
widgetGetAllocatedBaseline ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget to query
    -> m Int32
    -- ^ __Returns:__ the baseline of the /@widget@/, or -1 if none
widgetGetAllocatedBaseline widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_allocated_baseline widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetAllocatedBaselineMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetAllocatedBaselineMethodInfo a signature where
    overloadedMethod = widgetGetAllocatedBaseline

instance O.OverloadedMethodInfo WidgetGetAllocatedBaselineMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetAllocatedBaseline",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetAllocatedBaseline"
        })


#endif

-- method Widget::get_allocated_height
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_allocated_height" gtk_widget_get_allocated_height :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Returns the height that has currently been allocated to /@widget@/.
-- This function is intended to be used when implementing handlers
-- for the [Widget::draw]("GI.Gtk.Objects.Widget#g:signal:draw") function.
widgetGetAllocatedHeight ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget to query
    -> m Int32
    -- ^ __Returns:__ the height of the /@widget@/
widgetGetAllocatedHeight widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_allocated_height widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetAllocatedHeightMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetAllocatedHeightMethodInfo a signature where
    overloadedMethod = widgetGetAllocatedHeight

instance O.OverloadedMethodInfo WidgetGetAllocatedHeightMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetAllocatedHeight",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetAllocatedHeight"
        })


#endif

-- method Widget::get_allocated_size
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "allocation"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Rectangle" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a pointer to a #GtkAllocation to copy to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "baseline"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a pointer to an integer to copy to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_allocated_size" gtk_widget_get_allocated_size :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.Rectangle.Rectangle ->          -- allocation : TInterface (Name {namespace = "Gdk", name = "Rectangle"})
    Ptr Int32 ->                            -- baseline : TBasicType TInt
    IO ()

-- | Retrieves the widget’s allocated size.
-- 
-- This function returns the last values passed to
-- 'GI.Gtk.Objects.Widget.widgetSizeAllocateWithBaseline'. The value differs from
-- the size returned in 'GI.Gtk.Objects.Widget.widgetGetAllocation' in that functions
-- like 'GI.Gtk.Objects.Widget.widgetSetHalign' can adjust the allocation, but not
-- the value returned by this function.
-- 
-- If a widget is not visible, its allocated size is 0.
-- 
-- /Since: 3.20/
widgetGetAllocatedSize ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m ((Gdk.Rectangle.Rectangle, Int32))
widgetGetAllocatedSize widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    allocation <- SP.callocBoxedBytes 16 :: IO (Ptr Gdk.Rectangle.Rectangle)
    baseline <- allocMem :: IO (Ptr Int32)
    gtk_widget_get_allocated_size widget' allocation baseline
    allocation' <- (wrapBoxed Gdk.Rectangle.Rectangle) allocation
    baseline' <- peek baseline
    touchManagedPtr widget
    freeMem baseline
    return (allocation', baseline')

#if defined(ENABLE_OVERLOADING)
data WidgetGetAllocatedSizeMethodInfo
instance (signature ~ (m ((Gdk.Rectangle.Rectangle, Int32))), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetAllocatedSizeMethodInfo a signature where
    overloadedMethod = widgetGetAllocatedSize

instance O.OverloadedMethodInfo WidgetGetAllocatedSizeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetAllocatedSize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetAllocatedSize"
        })


#endif

-- method Widget::get_allocated_width
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_allocated_width" gtk_widget_get_allocated_width :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Returns the width that has currently been allocated to /@widget@/.
-- This function is intended to be used when implementing handlers
-- for the [Widget::draw]("GI.Gtk.Objects.Widget#g:signal:draw") function.
widgetGetAllocatedWidth ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget to query
    -> m Int32
    -- ^ __Returns:__ the width of the /@widget@/
widgetGetAllocatedWidth widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_allocated_width widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetAllocatedWidthMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetAllocatedWidthMethodInfo a signature where
    overloadedMethod = widgetGetAllocatedWidth

instance O.OverloadedMethodInfo WidgetGetAllocatedWidthMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetAllocatedWidth",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetAllocatedWidth"
        })


#endif

-- method Widget::get_allocation
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "allocation"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Rectangle" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a pointer to a #GtkAllocation to copy to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_allocation" gtk_widget_get_allocation :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.Rectangle.Rectangle ->          -- allocation : TInterface (Name {namespace = "Gdk", name = "Rectangle"})
    IO ()

-- | Retrieves the widget’s allocation.
-- 
-- Note, when implementing a t'GI.Gtk.Objects.Container.Container': a widget’s allocation will
-- be its “adjusted” allocation, that is, the widget’s parent
-- container typically calls 'GI.Gtk.Objects.Widget.widgetSizeAllocate' with an
-- allocation, and that allocation is then adjusted (to handle margin
-- and alignment for example) before assignment to the widget.
-- 'GI.Gtk.Objects.Widget.widgetGetAllocation' returns the adjusted allocation that
-- was actually assigned to the widget. The adjusted allocation is
-- guaranteed to be completely contained within the
-- 'GI.Gtk.Objects.Widget.widgetSizeAllocate' allocation, however. So a t'GI.Gtk.Objects.Container.Container'
-- is guaranteed that its children stay inside the assigned bounds,
-- but not that they have exactly the bounds the container assigned.
-- There is no way to get the original allocation assigned by
-- 'GI.Gtk.Objects.Widget.widgetSizeAllocate', since it isn’t stored; if a container
-- implementation needs that information it will have to track it itself.
-- 
-- /Since: 2.18/
widgetGetAllocation ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m (Gdk.Rectangle.Rectangle)
widgetGetAllocation widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    allocation <- SP.callocBoxedBytes 16 :: IO (Ptr Gdk.Rectangle.Rectangle)
    gtk_widget_get_allocation widget' allocation
    allocation' <- (wrapBoxed Gdk.Rectangle.Rectangle) allocation
    touchManagedPtr widget
    return allocation'

#if defined(ENABLE_OVERLOADING)
data WidgetGetAllocationMethodInfo
instance (signature ~ (m (Gdk.Rectangle.Rectangle)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetAllocationMethodInfo a signature where
    overloadedMethod = widgetGetAllocation

instance O.OverloadedMethodInfo WidgetGetAllocationMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetAllocation",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetAllocation"
        })


#endif

-- method Widget::get_ancestor
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "widget_type"
--           , argType = TBasicType TGType
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "ancestor type" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Widget" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_ancestor" gtk_widget_get_ancestor :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CGType ->                               -- widget_type : TBasicType TGType
    IO (Ptr Widget)

-- | Gets the first ancestor of /@widget@/ with type /@widgetType@/. For example,
-- @gtk_widget_get_ancestor (widget, GTK_TYPE_BOX)@ gets
-- the first t'GI.Gtk.Objects.Box.Box' that’s an ancestor of /@widget@/. No reference will be
-- added to the returned widget; it should not be unreferenced. See note
-- about checking for a toplevel t'GI.Gtk.Objects.Window.Window' in the docs for
-- 'GI.Gtk.Objects.Widget.widgetGetToplevel'.
-- 
-- Note that unlike 'GI.Gtk.Objects.Widget.widgetIsAncestor', 'GI.Gtk.Objects.Widget.widgetGetAncestor'
-- considers /@widget@/ to be an ancestor of itself.
widgetGetAncestor ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> GType
    -- ^ /@widgetType@/: ancestor type
    -> m (Maybe Widget)
    -- ^ __Returns:__ the ancestor widget, or 'P.Nothing' if not found
widgetGetAncestor widget widgetType = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let widgetType' = gtypeToCGType widgetType
    result <- gtk_widget_get_ancestor widget' widgetType'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Widget) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetAncestorMethodInfo
instance (signature ~ (GType -> m (Maybe Widget)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetAncestorMethodInfo a signature where
    overloadedMethod = widgetGetAncestor

instance O.OverloadedMethodInfo WidgetGetAncestorMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetAncestor",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetAncestor"
        })


#endif

-- method Widget::get_app_paintable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_app_paintable" gtk_widget_get_app_paintable :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether the application intends to draw on the widget in
-- an [Widget::draw]("GI.Gtk.Objects.Widget#g:signal:draw") handler.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetAppPaintable'
-- 
-- /Since: 2.18/
widgetGetAppPaintable ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the widget is app paintable
widgetGetAppPaintable widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_app_paintable widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetAppPaintableMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetAppPaintableMethodInfo a signature where
    overloadedMethod = widgetGetAppPaintable

instance O.OverloadedMethodInfo WidgetGetAppPaintableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetAppPaintable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetAppPaintable"
        })


#endif

-- method Widget::get_can_default
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_can_default" gtk_widget_get_can_default :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether /@widget@/ can be a default widget. See
-- 'GI.Gtk.Objects.Widget.widgetSetCanDefault'.
-- 
-- /Since: 2.18/
widgetGetCanDefault ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@widget@/ can be a default widget, 'P.False' otherwise
widgetGetCanDefault widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_can_default widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetCanDefaultMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetCanDefaultMethodInfo a signature where
    overloadedMethod = widgetGetCanDefault

instance O.OverloadedMethodInfo WidgetGetCanDefaultMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetCanDefault",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetCanDefault"
        })


#endif

-- method Widget::get_can_focus
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_can_focus" gtk_widget_get_can_focus :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether /@widget@/ can own the input focus. See
-- 'GI.Gtk.Objects.Widget.widgetSetCanFocus'.
-- 
-- /Since: 2.18/
widgetGetCanFocus ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@widget@/ can own the input focus, 'P.False' otherwise
widgetGetCanFocus widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_can_focus widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetCanFocusMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetCanFocusMethodInfo a signature where
    overloadedMethod = widgetGetCanFocus

instance O.OverloadedMethodInfo WidgetGetCanFocusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetCanFocus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetCanFocus"
        })


#endif

-- method Widget::get_child_requisition
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "requisition"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Requisition" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkRequisition to be filled in"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_child_requisition" gtk_widget_get_child_requisition :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gtk.Requisition.Requisition ->      -- requisition : TInterface (Name {namespace = "Gtk", name = "Requisition"})
    IO ()

{-# DEPRECATED widgetGetChildRequisition ["(Since version 3.0)","Use 'GI.Gtk.Objects.Widget.widgetGetPreferredSize' instead."] #-}
-- | This function is only for use in widget implementations. Obtains
-- /@widget@/->requisition, unless someone has forced a particular
-- geometry on the widget (e.g. with 'GI.Gtk.Objects.Widget.widgetSetSizeRequest'),
-- in which case it returns that geometry instead of the widget\'s
-- requisition.
-- 
-- This function differs from 'GI.Gtk.Objects.Widget.widgetSizeRequest' in that
-- it retrieves the last size request value from /@widget@/->requisition,
-- while 'GI.Gtk.Objects.Widget.widgetSizeRequest' actually calls the \"size_request\" method
-- on /@widget@/ to compute the size request and fill in /@widget@/->requisition,
-- and only then returns /@widget@/->requisition.
-- 
-- Because this function does not call the “size_request” method, it
-- can only be used when you know that /@widget@/->requisition is
-- up-to-date, that is, 'GI.Gtk.Objects.Widget.widgetSizeRequest' has been called
-- since the last time a resize was queued. In general, only container
-- implementations have this information; applications should use
-- 'GI.Gtk.Objects.Widget.widgetSizeRequest'.
widgetGetChildRequisition ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m (Gtk.Requisition.Requisition)
widgetGetChildRequisition widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    requisition <- SP.callocBoxedBytes 8 :: IO (Ptr Gtk.Requisition.Requisition)
    gtk_widget_get_child_requisition widget' requisition
    requisition' <- (wrapBoxed Gtk.Requisition.Requisition) requisition
    touchManagedPtr widget
    return requisition'

#if defined(ENABLE_OVERLOADING)
data WidgetGetChildRequisitionMethodInfo
instance (signature ~ (m (Gtk.Requisition.Requisition)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetChildRequisitionMethodInfo a signature where
    overloadedMethod = widgetGetChildRequisition

instance O.OverloadedMethodInfo WidgetGetChildRequisitionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetChildRequisition",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetChildRequisition"
        })


#endif

-- method Widget::get_child_visible
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_child_visible" gtk_widget_get_child_visible :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Gets the value set with 'GI.Gtk.Objects.Widget.widgetSetChildVisible'.
-- If you feel a need to use this function, your code probably
-- needs reorganization.
-- 
-- This function is only useful for container implementations and
-- never should be called by an application.
widgetGetChildVisible ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the widget is mapped with the parent.
widgetGetChildVisible widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_child_visible widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetChildVisibleMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetChildVisibleMethodInfo a signature where
    overloadedMethod = widgetGetChildVisible

instance O.OverloadedMethodInfo WidgetGetChildVisibleMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetChildVisible",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetChildVisible"
        })


#endif

-- method Widget::get_clip
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "clip"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Rectangle" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a pointer to a #GtkAllocation to copy to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_clip" gtk_widget_get_clip :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.Rectangle.Rectangle ->          -- clip : TInterface (Name {namespace = "Gdk", name = "Rectangle"})
    IO ()

-- | Retrieves the widget’s clip area.
-- 
-- The clip area is the area in which all of /@widget@/\'s drawing will
-- happen. Other toolkits call it the bounding box.
-- 
-- Historically, in GTK+ the clip area has been equal to the allocation
-- retrieved via 'GI.Gtk.Objects.Widget.widgetGetAllocation'.
-- 
-- /Since: 3.14/
widgetGetClip ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m (Gdk.Rectangle.Rectangle)
widgetGetClip widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    clip <- SP.callocBoxedBytes 16 :: IO (Ptr Gdk.Rectangle.Rectangle)
    gtk_widget_get_clip widget' clip
    clip' <- (wrapBoxed Gdk.Rectangle.Rectangle) clip
    touchManagedPtr widget
    return clip'

#if defined(ENABLE_OVERLOADING)
data WidgetGetClipMethodInfo
instance (signature ~ (m (Gdk.Rectangle.Rectangle)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetClipMethodInfo a signature where
    overloadedMethod = widgetGetClip

instance O.OverloadedMethodInfo WidgetGetClipMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetClip",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetClip"
        })


#endif

-- method Widget::get_clipboard
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "selection"
--           , argType = TInterface Name { namespace = "Gdk" , name = "Atom" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GdkAtom which identifies the clipboard\n            to use. %GDK_SELECTION_CLIPBOARD gives the\n            default clipboard. Another common value\n            is %GDK_SELECTION_PRIMARY, which gives\n            the primary X selection."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Clipboard" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_clipboard" gtk_widget_get_clipboard :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.Atom.Atom ->                    -- selection : TInterface (Name {namespace = "Gdk", name = "Atom"})
    IO (Ptr Gtk.Clipboard.Clipboard)

-- | Returns the clipboard object for the given selection to
-- be used with /@widget@/. /@widget@/ must have a t'GI.Gdk.Objects.Display.Display'
-- associated with it, so must be attached to a toplevel
-- window.
-- 
-- /Since: 2.2/
widgetGetClipboard ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> Gdk.Atom.Atom
    -- ^ /@selection@/: a t'GI.Gdk.Structs.Atom.Atom' which identifies the clipboard
    --             to use. @/GDK_SELECTION_CLIPBOARD/@ gives the
    --             default clipboard. Another common value
    --             is @/GDK_SELECTION_PRIMARY/@, which gives
    --             the primary X selection.
    -> m Gtk.Clipboard.Clipboard
    -- ^ __Returns:__ the appropriate clipboard object. If no
    --             clipboard already exists, a new one will
    --             be created. Once a clipboard object has
    --             been created, it is persistent for all time.
widgetGetClipboard widget selection = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    selection' <- unsafeManagedPtrGetPtr selection
    result <- gtk_widget_get_clipboard widget' selection'
    checkUnexpectedReturnNULL "widgetGetClipboard" result
    result' <- (newObject Gtk.Clipboard.Clipboard) result
    touchManagedPtr widget
    touchManagedPtr selection
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetClipboardMethodInfo
instance (signature ~ (Gdk.Atom.Atom -> m Gtk.Clipboard.Clipboard), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetClipboardMethodInfo a signature where
    overloadedMethod = widgetGetClipboard

instance O.OverloadedMethodInfo WidgetGetClipboardMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetClipboard",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetClipboard"
        })


#endif

-- method Widget::get_composite_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_composite_name" gtk_widget_get_composite_name :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CString

{-# DEPRECATED widgetGetCompositeName ["(Since version 3.10)","Use 'GI.Gtk.Structs.WidgetClass.widgetClassSetTemplate', or don\8217t use this API at all."] #-}
-- | Obtains the composite name of a widget.
widgetGetCompositeName ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m T.Text
    -- ^ __Returns:__ the composite name of /@widget@/, or 'P.Nothing' if /@widget@/ is not
    --   a composite child. The string should be freed when it is no
    --   longer needed.
widgetGetCompositeName widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_composite_name widget'
    checkUnexpectedReturnNULL "widgetGetCompositeName" result
    result' <- cstringToText result
    freeMem result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetCompositeNameMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetCompositeNameMethodInfo a signature where
    overloadedMethod = widgetGetCompositeName

instance O.OverloadedMethodInfo WidgetGetCompositeNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetCompositeName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetCompositeName"
        })


#endif

-- method Widget::get_device_enabled
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "device"
--           , argType = TInterface Name { namespace = "Gdk" , name = "Device" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GdkDevice" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_device_enabled" gtk_widget_get_device_enabled :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.Device.Device ->                -- device : TInterface (Name {namespace = "Gdk", name = "Device"})
    IO CInt

-- | Returns whether /@device@/ can interact with /@widget@/ and its
-- children. See 'GI.Gtk.Objects.Widget.widgetSetDeviceEnabled'.
-- 
-- /Since: 3.0/
widgetGetDeviceEnabled ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, Gdk.Device.IsDevice b) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> b
    -- ^ /@device@/: a t'GI.Gdk.Objects.Device.Device'
    -> m Bool
    -- ^ __Returns:__ 'P.True' is /@device@/ is enabled for /@widget@/
widgetGetDeviceEnabled widget device = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    device' <- unsafeManagedPtrCastPtr device
    result <- gtk_widget_get_device_enabled widget' device'
    let result' = (/= 0) result
    touchManagedPtr widget
    touchManagedPtr device
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetDeviceEnabledMethodInfo
instance (signature ~ (b -> m Bool), MonadIO m, IsWidget a, Gdk.Device.IsDevice b) => O.OverloadedMethod WidgetGetDeviceEnabledMethodInfo a signature where
    overloadedMethod = widgetGetDeviceEnabled

instance O.OverloadedMethodInfo WidgetGetDeviceEnabledMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetDeviceEnabled",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetDeviceEnabled"
        })


#endif

-- method Widget::get_device_events
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "device"
--           , argType = TInterface Name { namespace = "Gdk" , name = "Device" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GdkDevice" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "EventMask" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_device_events" gtk_widget_get_device_events :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.Device.Device ->                -- device : TInterface (Name {namespace = "Gdk", name = "Device"})
    IO CUInt

-- | Returns the events mask for the widget corresponding to an specific device. These
-- are the events that the widget will receive when /@device@/ operates on it.
-- 
-- /Since: 3.0/
widgetGetDeviceEvents ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, Gdk.Device.IsDevice b) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> b
    -- ^ /@device@/: a t'GI.Gdk.Objects.Device.Device'
    -> m [Gdk.Flags.EventMask]
    -- ^ __Returns:__ device event mask for /@widget@/
widgetGetDeviceEvents widget device = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    device' <- unsafeManagedPtrCastPtr device
    result <- gtk_widget_get_device_events widget' device'
    let result' = wordToGFlags result
    touchManagedPtr widget
    touchManagedPtr device
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetDeviceEventsMethodInfo
instance (signature ~ (b -> m [Gdk.Flags.EventMask]), MonadIO m, IsWidget a, Gdk.Device.IsDevice b) => O.OverloadedMethod WidgetGetDeviceEventsMethodInfo a signature where
    overloadedMethod = widgetGetDeviceEvents

instance O.OverloadedMethodInfo WidgetGetDeviceEventsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetDeviceEvents",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetDeviceEvents"
        })


#endif

-- method Widget::get_direction
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gtk" , name = "TextDirection" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_direction" gtk_widget_get_direction :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CUInt

-- | Gets the reading direction for a particular widget. See
-- 'GI.Gtk.Objects.Widget.widgetSetDirection'.
widgetGetDirection ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Gtk.Enums.TextDirection
    -- ^ __Returns:__ the reading direction for the widget.
widgetGetDirection widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_direction widget'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetDirectionMethodInfo
instance (signature ~ (m Gtk.Enums.TextDirection), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetDirectionMethodInfo a signature where
    overloadedMethod = widgetGetDirection

instance O.OverloadedMethodInfo WidgetGetDirectionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetDirection",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetDirection"
        })


#endif

-- method Widget::get_display
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Display" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_display" gtk_widget_get_display :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gdk.Display.Display)

-- | Get the t'GI.Gdk.Objects.Display.Display' for the toplevel window associated with
-- this widget. This function can only be called after the widget
-- has been added to a widget hierarchy with a t'GI.Gtk.Objects.Window.Window' at the top.
-- 
-- In general, you should only create display specific
-- resources when a widget has been realized, and you should
-- free those resources when the widget is unrealized.
-- 
-- /Since: 2.2/
widgetGetDisplay ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Gdk.Display.Display
    -- ^ __Returns:__ the t'GI.Gdk.Objects.Display.Display' for the toplevel for this widget.
widgetGetDisplay widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_display widget'
    checkUnexpectedReturnNULL "widgetGetDisplay" result
    result' <- (newObject Gdk.Display.Display) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetDisplayMethodInfo
instance (signature ~ (m Gdk.Display.Display), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetDisplayMethodInfo a signature where
    overloadedMethod = widgetGetDisplay

instance O.OverloadedMethodInfo WidgetGetDisplayMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetDisplay",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetDisplay"
        })


#endif

-- method Widget::get_double_buffered
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_double_buffered" gtk_widget_get_double_buffered :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether the widget is double buffered.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetDoubleBuffered'
-- 
-- /Since: 2.18/
widgetGetDoubleBuffered ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the widget is double buffered
widgetGetDoubleBuffered widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_double_buffered widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetDoubleBufferedMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetDoubleBufferedMethodInfo a signature where
    overloadedMethod = widgetGetDoubleBuffered

instance O.OverloadedMethodInfo WidgetGetDoubleBufferedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetDoubleBuffered",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetDoubleBuffered"
        })


#endif

-- method Widget::get_events
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_events" gtk_widget_get_events :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Returns the event mask (see t'GI.Gdk.Flags.EventMask') for the widget. These are the
-- events that the widget will receive.
-- 
-- Note: Internally, the widget event mask will be the logical OR of the event
-- mask set through 'GI.Gtk.Objects.Widget.widgetSetEvents' or 'GI.Gtk.Objects.Widget.widgetAddEvents', and the
-- event mask necessary to cater for every t'GI.Gtk.Objects.EventController.EventController' created for the
-- widget.
widgetGetEvents ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Int32
    -- ^ __Returns:__ event mask for /@widget@/
widgetGetEvents widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_events widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetEventsMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetEventsMethodInfo a signature where
    overloadedMethod = widgetGetEvents

instance O.OverloadedMethodInfo WidgetGetEventsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetEvents",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetEvents"
        })


#endif

-- method Widget::get_focus_on_click
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_focus_on_click" gtk_widget_get_focus_on_click :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Returns whether the widget should grab focus when it is clicked with the mouse.
-- See 'GI.Gtk.Objects.Widget.widgetSetFocusOnClick'.
-- 
-- /Since: 3.20/
widgetGetFocusOnClick ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the widget should grab focus when it is clicked with
    --               the mouse.
widgetGetFocusOnClick widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_focus_on_click widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetFocusOnClickMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetFocusOnClickMethodInfo a signature where
    overloadedMethod = widgetGetFocusOnClick

instance O.OverloadedMethodInfo WidgetGetFocusOnClickMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetFocusOnClick",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetFocusOnClick"
        })


#endif

-- method Widget::get_font_map
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "FontMap" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_font_map" gtk_widget_get_font_map :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Pango.FontMap.FontMap)

-- | Gets the font map that has been set with 'GI.Gtk.Objects.Widget.widgetSetFontMap'.
-- 
-- /Since: 3.18/
widgetGetFontMap ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m (Maybe Pango.FontMap.FontMap)
    -- ^ __Returns:__ A t'GI.Pango.Objects.FontMap.FontMap', or 'P.Nothing'
widgetGetFontMap widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_font_map widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Pango.FontMap.FontMap) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetFontMapMethodInfo
instance (signature ~ (m (Maybe Pango.FontMap.FontMap)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetFontMapMethodInfo a signature where
    overloadedMethod = widgetGetFontMap

instance O.OverloadedMethodInfo WidgetGetFontMapMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetFontMap",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetFontMap"
        })


#endif

-- method Widget::get_font_options
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "cairo" , name = "FontOptions" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_font_options" gtk_widget_get_font_options :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Cairo.FontOptions.FontOptions)

-- | Returns the t'GI.Cairo.Structs.FontOptions.FontOptions' used for Pango rendering. When not set,
-- the defaults font options for the t'GI.Gdk.Objects.Screen.Screen' will be used.
-- 
-- /Since: 3.18/
widgetGetFontOptions ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m (Maybe Cairo.FontOptions.FontOptions)
    -- ^ __Returns:__ the t'GI.Cairo.Structs.FontOptions.FontOptions' or 'P.Nothing' if not set
widgetGetFontOptions widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_font_options widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed Cairo.FontOptions.FontOptions) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetFontOptionsMethodInfo
instance (signature ~ (m (Maybe Cairo.FontOptions.FontOptions)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetFontOptionsMethodInfo a signature where
    overloadedMethod = widgetGetFontOptions

instance O.OverloadedMethodInfo WidgetGetFontOptionsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetFontOptions",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetFontOptions"
        })


#endif

-- method Widget::get_frame_clock
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "FrameClock" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_frame_clock" gtk_widget_get_frame_clock :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gdk.FrameClock.FrameClock)

-- | Obtains the frame clock for a widget. The frame clock is a global
-- “ticker” that can be used to drive animations and repaints.  The
-- most common reason to get the frame clock is to call
-- 'GI.Gdk.Objects.FrameClock.frameClockGetFrameTime', in order to get a time to use for
-- animating. For example you might record the start of the animation
-- with an initial value from 'GI.Gdk.Objects.FrameClock.frameClockGetFrameTime', and
-- then update the animation by calling
-- 'GI.Gdk.Objects.FrameClock.frameClockGetFrameTime' again during each repaint.
-- 
-- 'GI.Gdk.Objects.FrameClock.frameClockRequestPhase' will result in a new frame on the
-- clock, but won’t necessarily repaint any widgets. To repaint a
-- widget, you have to use 'GI.Gtk.Objects.Widget.widgetQueueDraw' which invalidates
-- the widget (thus scheduling it to receive a draw on the next
-- frame). 'GI.Gtk.Objects.Widget.widgetQueueDraw' will also end up requesting a frame
-- on the appropriate frame clock.
-- 
-- A widget’s frame clock will not change while the widget is
-- mapped. Reparenting a widget (which implies a temporary unmap) can
-- change the widget’s frame clock.
-- 
-- Unrealized widgets do not have a frame clock.
-- 
-- /Since: 3.8/
widgetGetFrameClock ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m (Maybe Gdk.FrameClock.FrameClock)
    -- ^ __Returns:__ a t'GI.Gdk.Objects.FrameClock.FrameClock',
    -- or 'P.Nothing' if widget is unrealized
widgetGetFrameClock widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_frame_clock widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gdk.FrameClock.FrameClock) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetFrameClockMethodInfo
instance (signature ~ (m (Maybe Gdk.FrameClock.FrameClock)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetFrameClockMethodInfo a signature where
    overloadedMethod = widgetGetFrameClock

instance O.OverloadedMethodInfo WidgetGetFrameClockMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetFrameClock",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetFrameClock"
        })


#endif

-- method Widget::get_halign
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Align" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_halign" gtk_widget_get_halign :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CUInt

-- | Gets the value of the [Widget:halign]("GI.Gtk.Objects.Widget#g:attr:halign") property.
-- 
-- For backwards compatibility reasons this method will never return
-- 'GI.Gtk.Enums.AlignBaseline', but instead it will convert it to
-- 'GI.Gtk.Enums.AlignFill'. Baselines are not supported for horizontal
-- alignment.
widgetGetHalign ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Gtk.Enums.Align
    -- ^ __Returns:__ the horizontal alignment of /@widget@/
widgetGetHalign widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_halign widget'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetHalignMethodInfo
instance (signature ~ (m Gtk.Enums.Align), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetHalignMethodInfo a signature where
    overloadedMethod = widgetGetHalign

instance O.OverloadedMethodInfo WidgetGetHalignMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetHalign",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetHalign"
        })


#endif

-- method Widget::get_has_tooltip
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_has_tooltip" gtk_widget_get_has_tooltip :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Returns the current value of the has-tooltip property.  See
-- [Widget:hasTooltip]("GI.Gtk.Objects.Widget#g:attr:hasTooltip") for more information.
-- 
-- /Since: 2.12/
widgetGetHasTooltip ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ current value of has-tooltip on /@widget@/.
widgetGetHasTooltip widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_has_tooltip widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetHasTooltipMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetHasTooltipMethodInfo a signature where
    overloadedMethod = widgetGetHasTooltip

instance O.OverloadedMethodInfo WidgetGetHasTooltipMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetHasTooltip",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetHasTooltip"
        })


#endif

-- method Widget::get_has_window
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_has_window" gtk_widget_get_has_window :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether /@widget@/ has a t'GI.Gdk.Objects.Window.Window' of its own. See
-- 'GI.Gtk.Objects.Widget.widgetSetHasWindow'.
-- 
-- /Since: 2.18/
widgetGetHasWindow ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@widget@/ has a window, 'P.False' otherwise
widgetGetHasWindow widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_has_window widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetHasWindowMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetHasWindowMethodInfo a signature where
    overloadedMethod = widgetGetHasWindow

instance O.OverloadedMethodInfo WidgetGetHasWindowMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetHasWindow",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetHasWindow"
        })


#endif

-- method Widget::get_hexpand
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_hexpand" gtk_widget_get_hexpand :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Gets whether the widget would like any available extra horizontal
-- space. When a user resizes a t'GI.Gtk.Objects.Window.Window', widgets with expand=TRUE
-- generally receive the extra space. For example, a list or
-- scrollable area or document in your window would often be set to
-- expand.
-- 
-- Containers should use 'GI.Gtk.Objects.Widget.widgetComputeExpand' rather than
-- this function, to see whether a widget, or any of its children,
-- has the expand flag set. If any child of a widget wants to
-- expand, the parent may ask to expand also.
-- 
-- This function only looks at the widget’s own hexpand flag, rather
-- than computing whether the entire widget tree rooted at this widget
-- wants to expand.
widgetGetHexpand ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget
    -> m Bool
    -- ^ __Returns:__ whether hexpand flag is set
widgetGetHexpand widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_hexpand widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetHexpandMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetHexpandMethodInfo a signature where
    overloadedMethod = widgetGetHexpand

instance O.OverloadedMethodInfo WidgetGetHexpandMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetHexpand",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetHexpand"
        })


#endif

-- method Widget::get_hexpand_set
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_hexpand_set" gtk_widget_get_hexpand_set :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Gets whether 'GI.Gtk.Objects.Widget.widgetSetHexpand' has been used to
-- explicitly set the expand flag on this widget.
-- 
-- If hexpand is set, then it overrides any computed
-- expand value based on child widgets. If hexpand is not
-- set, then the expand value depends on whether any
-- children of the widget would like to expand.
-- 
-- There are few reasons to use this function, but it’s here
-- for completeness and consistency.
widgetGetHexpandSet ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget
    -> m Bool
    -- ^ __Returns:__ whether hexpand has been explicitly set
widgetGetHexpandSet widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_hexpand_set widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetHexpandSetMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetHexpandSetMethodInfo a signature where
    overloadedMethod = widgetGetHexpandSet

instance O.OverloadedMethodInfo WidgetGetHexpandSetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetHexpandSet",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetHexpandSet"
        })


#endif

-- method Widget::get_mapped
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_mapped" gtk_widget_get_mapped :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Whether the widget is mapped.
-- 
-- /Since: 2.20/
widgetGetMapped ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the widget is mapped, 'P.False' otherwise.
widgetGetMapped widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_mapped widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetMappedMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetMappedMethodInfo a signature where
    overloadedMethod = widgetGetMapped

instance O.OverloadedMethodInfo WidgetGetMappedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetMapped",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetMapped"
        })


#endif

-- method Widget::get_margin_bottom
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_margin_bottom" gtk_widget_get_margin_bottom :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Gets the value of the [Widget:marginBottom]("GI.Gtk.Objects.Widget#g:attr:marginBottom") property.
-- 
-- /Since: 3.0/
widgetGetMarginBottom ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Int32
    -- ^ __Returns:__ The bottom margin of /@widget@/
widgetGetMarginBottom widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_margin_bottom widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetMarginBottomMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetMarginBottomMethodInfo a signature where
    overloadedMethod = widgetGetMarginBottom

instance O.OverloadedMethodInfo WidgetGetMarginBottomMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetMarginBottom",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetMarginBottom"
        })


#endif

-- method Widget::get_margin_end
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_margin_end" gtk_widget_get_margin_end :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Gets the value of the [Widget:marginEnd]("GI.Gtk.Objects.Widget#g:attr:marginEnd") property.
-- 
-- /Since: 3.12/
widgetGetMarginEnd ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Int32
    -- ^ __Returns:__ The end margin of /@widget@/
widgetGetMarginEnd widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_margin_end widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetMarginEndMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetMarginEndMethodInfo a signature where
    overloadedMethod = widgetGetMarginEnd

instance O.OverloadedMethodInfo WidgetGetMarginEndMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetMarginEnd",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetMarginEnd"
        })


#endif

-- method Widget::get_margin_left
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_margin_left" gtk_widget_get_margin_left :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

{-# DEPRECATED widgetGetMarginLeft ["(Since version 3.12)","Use 'GI.Gtk.Objects.Widget.widgetGetMarginStart' instead."] #-}
-- | Gets the value of the [Widget:marginLeft]("GI.Gtk.Objects.Widget#g:attr:marginLeft") property.
-- 
-- /Since: 3.0/
widgetGetMarginLeft ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Int32
    -- ^ __Returns:__ The left margin of /@widget@/
widgetGetMarginLeft widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_margin_left widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetMarginLeftMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetMarginLeftMethodInfo a signature where
    overloadedMethod = widgetGetMarginLeft

instance O.OverloadedMethodInfo WidgetGetMarginLeftMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetMarginLeft",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetMarginLeft"
        })


#endif

-- method Widget::get_margin_right
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_margin_right" gtk_widget_get_margin_right :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

{-# DEPRECATED widgetGetMarginRight ["(Since version 3.12)","Use 'GI.Gtk.Objects.Widget.widgetGetMarginEnd' instead."] #-}
-- | Gets the value of the [Widget:marginRight]("GI.Gtk.Objects.Widget#g:attr:marginRight") property.
-- 
-- /Since: 3.0/
widgetGetMarginRight ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Int32
    -- ^ __Returns:__ The right margin of /@widget@/
widgetGetMarginRight widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_margin_right widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetMarginRightMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetMarginRightMethodInfo a signature where
    overloadedMethod = widgetGetMarginRight

instance O.OverloadedMethodInfo WidgetGetMarginRightMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetMarginRight",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetMarginRight"
        })


#endif

-- method Widget::get_margin_start
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_margin_start" gtk_widget_get_margin_start :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Gets the value of the [Widget:marginStart]("GI.Gtk.Objects.Widget#g:attr:marginStart") property.
-- 
-- /Since: 3.12/
widgetGetMarginStart ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Int32
    -- ^ __Returns:__ The start margin of /@widget@/
widgetGetMarginStart widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_margin_start widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetMarginStartMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetMarginStartMethodInfo a signature where
    overloadedMethod = widgetGetMarginStart

instance O.OverloadedMethodInfo WidgetGetMarginStartMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetMarginStart",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetMarginStart"
        })


#endif

-- method Widget::get_margin_top
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_margin_top" gtk_widget_get_margin_top :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Gets the value of the [Widget:marginTop]("GI.Gtk.Objects.Widget#g:attr:marginTop") property.
-- 
-- /Since: 3.0/
widgetGetMarginTop ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Int32
    -- ^ __Returns:__ The top margin of /@widget@/
widgetGetMarginTop widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_margin_top widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetMarginTopMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetMarginTopMethodInfo a signature where
    overloadedMethod = widgetGetMarginTop

instance O.OverloadedMethodInfo WidgetGetMarginTopMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetMarginTop",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetMarginTop"
        })


#endif

-- method Widget::get_modifier_mask
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "intent"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "ModifierIntent" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the use case for the modifier mask"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gdk" , name = "ModifierType" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_modifier_mask" gtk_widget_get_modifier_mask :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CUInt ->                                -- intent : TInterface (Name {namespace = "Gdk", name = "ModifierIntent"})
    IO CUInt

-- | Returns the modifier mask the /@widget@/’s windowing system backend
-- uses for a particular purpose.
-- 
-- See 'GI.Gdk.Objects.Keymap.keymapGetModifierMask'.
-- 
-- /Since: 3.4/
widgetGetModifierMask ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> Gdk.Enums.ModifierIntent
    -- ^ /@intent@/: the use case for the modifier mask
    -> m [Gdk.Flags.ModifierType]
    -- ^ __Returns:__ the modifier mask used for /@intent@/.
widgetGetModifierMask widget intent = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let intent' = (fromIntegral . fromEnum) intent
    result <- gtk_widget_get_modifier_mask widget' intent'
    let result' = wordToGFlags result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetModifierMaskMethodInfo
instance (signature ~ (Gdk.Enums.ModifierIntent -> m [Gdk.Flags.ModifierType]), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetModifierMaskMethodInfo a signature where
    overloadedMethod = widgetGetModifierMask

instance O.OverloadedMethodInfo WidgetGetModifierMaskMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetModifierMask",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetModifierMask"
        })


#endif

-- method Widget::get_modifier_style
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "RcStyle" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_modifier_style" gtk_widget_get_modifier_style :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gtk.RcStyle.RcStyle)

{-# DEPRECATED widgetGetModifierStyle ["(Since version 3.0)","Use t'GI.Gtk.Objects.StyleContext.StyleContext' with a custom t'GI.Gtk.Interfaces.StyleProvider.StyleProvider' instead"] #-}
-- | Returns the current modifier style for the widget. (As set by
-- 'GI.Gtk.Objects.Widget.widgetModifyStyle'.) If no style has previously set, a new
-- t'GI.Gtk.Objects.RcStyle.RcStyle' will be created with all values unset, and set as the
-- modifier style for the widget. If you make changes to this rc
-- style, you must call 'GI.Gtk.Objects.Widget.widgetModifyStyle', passing in the
-- returned rc style, to make sure that your changes take effect.
-- 
-- Caution: passing the style back to 'GI.Gtk.Objects.Widget.widgetModifyStyle' will
-- normally end up destroying it, because 'GI.Gtk.Objects.Widget.widgetModifyStyle' copies
-- the passed-in style and sets the copy as the new modifier style,
-- thus dropping any reference to the old modifier style. Add a reference
-- to the modifier style if you want to keep it alive.
widgetGetModifierStyle ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Gtk.RcStyle.RcStyle
    -- ^ __Returns:__ the modifier style for the widget.
    --     This rc style is owned by the widget. If you want to keep a
    --     pointer to value this around, you must add a refcount using
    --     'GI.GObject.Objects.Object.objectRef'.
widgetGetModifierStyle widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_modifier_style widget'
    checkUnexpectedReturnNULL "widgetGetModifierStyle" result
    result' <- (newObject Gtk.RcStyle.RcStyle) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetModifierStyleMethodInfo
instance (signature ~ (m Gtk.RcStyle.RcStyle), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetModifierStyleMethodInfo a signature where
    overloadedMethod = widgetGetModifierStyle

instance O.OverloadedMethodInfo WidgetGetModifierStyleMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetModifierStyle",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetModifierStyle"
        })


#endif

-- method Widget::get_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_name" gtk_widget_get_name :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CString

-- | Retrieves the name of a widget. See 'GI.Gtk.Objects.Widget.widgetSetName' for the
-- significance of widget names.
widgetGetName ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m T.Text
    -- ^ __Returns:__ name of the widget. This string is owned by GTK+ and
    -- should not be modified or freed
widgetGetName widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_name widget'
    checkUnexpectedReturnNULL "widgetGetName" result
    result' <- cstringToText result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetNameMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetNameMethodInfo a signature where
    overloadedMethod = widgetGetName

instance O.OverloadedMethodInfo WidgetGetNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetName"
        })


#endif

-- method Widget::get_no_show_all
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_no_show_all" gtk_widget_get_no_show_all :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Returns the current value of the [Widget:noShowAll]("GI.Gtk.Objects.Widget#g:attr:noShowAll") property,
-- which determines whether calls to 'GI.Gtk.Objects.Widget.widgetShowAll'
-- will affect this widget.
-- 
-- /Since: 2.4/
widgetGetNoShowAll ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ the current value of the “no-show-all” property.
widgetGetNoShowAll widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_no_show_all widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetNoShowAllMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetNoShowAllMethodInfo a signature where
    overloadedMethod = widgetGetNoShowAll

instance O.OverloadedMethodInfo WidgetGetNoShowAllMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetNoShowAll",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetNoShowAll"
        })


#endif

-- method Widget::get_opacity
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TDouble)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_opacity" gtk_widget_get_opacity :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CDouble

-- | Fetches the requested opacity for this widget.
-- See 'GI.Gtk.Objects.Widget.widgetSetOpacity'.
-- 
-- /Since: 3.8/
widgetGetOpacity ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Double
    -- ^ __Returns:__ the requested opacity for this widget.
widgetGetOpacity widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_opacity widget'
    let result' = realToFrac result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetOpacityMethodInfo
instance (signature ~ (m Double), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetOpacityMethodInfo a signature where
    overloadedMethod = widgetGetOpacity

instance O.OverloadedMethodInfo WidgetGetOpacityMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetOpacity",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetOpacity"
        })


#endif

-- method Widget::get_pango_context
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Pango" , name = "Context" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_pango_context" gtk_widget_get_pango_context :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Pango.Context.Context)

-- | Gets a t'GI.Pango.Objects.Context.Context' with the appropriate font map, font description,
-- and base direction for this widget. Unlike the context returned
-- by 'GI.Gtk.Objects.Widget.widgetCreatePangoContext', this context is owned by
-- the widget (it can be used until the screen for the widget changes
-- or the widget is removed from its toplevel), and will be updated to
-- match any changes to the widget’s attributes. This can be tracked
-- by using the [Widget::screenChanged]("GI.Gtk.Objects.Widget#g:signal:screenChanged") signal on the widget.
widgetGetPangoContext ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Pango.Context.Context
    -- ^ __Returns:__ the t'GI.Pango.Objects.Context.Context' for the widget.
widgetGetPangoContext widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_pango_context widget'
    checkUnexpectedReturnNULL "widgetGetPangoContext" result
    result' <- (newObject Pango.Context.Context) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetPangoContextMethodInfo
instance (signature ~ (m Pango.Context.Context), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetPangoContextMethodInfo a signature where
    overloadedMethod = widgetGetPangoContext

instance O.OverloadedMethodInfo WidgetGetPangoContextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetPangoContext",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetPangoContext"
        })


#endif

-- method Widget::get_parent
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Widget" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_parent" gtk_widget_get_parent :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Widget)

-- | Returns the parent container of /@widget@/.
widgetGetParent ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m (Maybe Widget)
    -- ^ __Returns:__ the parent container of /@widget@/, or 'P.Nothing'
widgetGetParent widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_parent widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Widget) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetParentMethodInfo
instance (signature ~ (m (Maybe Widget)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetParentMethodInfo a signature where
    overloadedMethod = widgetGetParent

instance O.OverloadedMethodInfo WidgetGetParentMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetParent",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetParent"
        })


#endif

-- method Widget::get_parent_window
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget." , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Window" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_parent_window" gtk_widget_get_parent_window :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gdk.Window.Window)

-- | Gets /@widget@/’s parent window, or 'P.Nothing' if it does not have one.
widgetGetParentWindow ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'.
    -> m (Maybe Gdk.Window.Window)
    -- ^ __Returns:__ the parent window of /@widget@/, or 'P.Nothing'
    -- if it does not have a parent window.
widgetGetParentWindow widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_parent_window widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gdk.Window.Window) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetParentWindowMethodInfo
instance (signature ~ (m (Maybe Gdk.Window.Window)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetParentWindowMethodInfo a signature where
    overloadedMethod = widgetGetParentWindow

instance O.OverloadedMethodInfo WidgetGetParentWindowMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetParentWindow",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetParentWindow"
        })


#endif

-- method Widget::get_path
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "WidgetPath" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_path" gtk_widget_get_path :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gtk.WidgetPath.WidgetPath)

-- | Returns the t'GI.Gtk.Structs.WidgetPath.WidgetPath' representing /@widget@/, if the widget
-- is not connected to a toplevel widget, a partial path will be
-- created.
widgetGetPath ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Gtk.WidgetPath.WidgetPath
    -- ^ __Returns:__ The t'GI.Gtk.Structs.WidgetPath.WidgetPath' representing /@widget@/
widgetGetPath widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_path widget'
    checkUnexpectedReturnNULL "widgetGetPath" result
    result' <- (newBoxed Gtk.WidgetPath.WidgetPath) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetPathMethodInfo
instance (signature ~ (m Gtk.WidgetPath.WidgetPath), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetPathMethodInfo a signature where
    overloadedMethod = widgetGetPath

instance O.OverloadedMethodInfo WidgetGetPathMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetPath",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetPath"
        })


#endif

-- method Widget::get_pointer
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the X coordinate, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the Y coordinate, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_pointer" gtk_widget_get_pointer :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Int32 ->                            -- x : TBasicType TInt
    Ptr Int32 ->                            -- y : TBasicType TInt
    IO ()

{-# DEPRECATED widgetGetPointer ["(Since version 3.4)","Use 'GI.Gdk.Objects.Window.windowGetDevicePosition' instead."] #-}
-- | Obtains the location of the mouse pointer in widget coordinates.
-- Widget coordinates are a bit odd; for historical reasons, they are
-- defined as /@widget@/->window coordinates for widgets that return 'P.True' for
-- 'GI.Gtk.Objects.Widget.widgetGetHasWindow'; and are relative to /@widget@/->allocation.x,
-- /@widget@/->allocation.y otherwise.
widgetGetPointer ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m ((Int32, Int32))
widgetGetPointer widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    x <- allocMem :: IO (Ptr Int32)
    y <- allocMem :: IO (Ptr Int32)
    gtk_widget_get_pointer widget' x y
    x' <- peek x
    y' <- peek y
    touchManagedPtr widget
    freeMem x
    freeMem y
    return (x', y')

#if defined(ENABLE_OVERLOADING)
data WidgetGetPointerMethodInfo
instance (signature ~ (m ((Int32, Int32))), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetPointerMethodInfo a signature where
    overloadedMethod = widgetGetPointer

instance O.OverloadedMethodInfo WidgetGetPointerMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetPointer",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetPointer"
        })


#endif

-- method Widget::get_preferred_height
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget instance"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "minimum_height"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "location to store the minimum height, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "natural_height"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "location to store the natural height, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_preferred_height" gtk_widget_get_preferred_height :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Int32 ->                            -- minimum_height : TBasicType TInt
    Ptr Int32 ->                            -- natural_height : TBasicType TInt
    IO ()

-- | Retrieves a widget’s initial minimum and natural height.
-- 
-- This call is specific to width-for-height requests.
-- 
-- The returned request will be modified by the
-- GtkWidgetClass[adjust_size_request](#g:signal:adjust_size_request) virtual method and by any
-- @/GtkSizeGroups/@ that have been applied. That is, the returned request
-- is the one that should be used for layout, not necessarily the one
-- returned by the widget itself.
-- 
-- /Since: 3.0/
widgetGetPreferredHeight ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' instance
    -> m ((Int32, Int32))
widgetGetPreferredHeight widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    minimumHeight <- allocMem :: IO (Ptr Int32)
    naturalHeight <- allocMem :: IO (Ptr Int32)
    gtk_widget_get_preferred_height widget' minimumHeight naturalHeight
    minimumHeight' <- peek minimumHeight
    naturalHeight' <- peek naturalHeight
    touchManagedPtr widget
    freeMem minimumHeight
    freeMem naturalHeight
    return (minimumHeight', naturalHeight')

#if defined(ENABLE_OVERLOADING)
data WidgetGetPreferredHeightMethodInfo
instance (signature ~ (m ((Int32, Int32))), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetPreferredHeightMethodInfo a signature where
    overloadedMethod = widgetGetPreferredHeight

instance O.OverloadedMethodInfo WidgetGetPreferredHeightMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetPreferredHeight",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetPreferredHeight"
        })


#endif

-- method Widget::get_preferred_height_and_baseline_for_width
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget instance"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "width"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the width which is available for allocation, or -1 if none"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "minimum_height"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "location for storing the minimum height, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "natural_height"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "location for storing the natural height, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "minimum_baseline"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "location for storing the baseline for the minimum height, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "natural_baseline"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "location for storing the baseline for the natural height, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_preferred_height_and_baseline_for_width" gtk_widget_get_preferred_height_and_baseline_for_width :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Int32 ->                                -- width : TBasicType TInt
    Ptr Int32 ->                            -- minimum_height : TBasicType TInt
    Ptr Int32 ->                            -- natural_height : TBasicType TInt
    Ptr Int32 ->                            -- minimum_baseline : TBasicType TInt
    Ptr Int32 ->                            -- natural_baseline : TBasicType TInt
    IO ()

-- | Retrieves a widget’s minimum and natural height and the corresponding baselines if it would be given
-- the specified /@width@/, or the default height if /@width@/ is -1. The baselines may be -1 which means
-- that no baseline is requested for this widget.
-- 
-- The returned request will be modified by the
-- GtkWidgetClass[adjust_size_request](#g:signal:adjust_size_request) and GtkWidgetClass[adjust_baseline_request](#g:signal:adjust_baseline_request) virtual methods
-- and by any @/GtkSizeGroups/@ that have been applied. That is, the returned request
-- is the one that should be used for layout, not necessarily the one
-- returned by the widget itself.
-- 
-- /Since: 3.10/
widgetGetPreferredHeightAndBaselineForWidth ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' instance
    -> Int32
    -- ^ /@width@/: the width which is available for allocation, or -1 if none
    -> m ((Int32, Int32, Int32, Int32))
widgetGetPreferredHeightAndBaselineForWidth widget width = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    minimumHeight <- allocMem :: IO (Ptr Int32)
    naturalHeight <- allocMem :: IO (Ptr Int32)
    minimumBaseline <- allocMem :: IO (Ptr Int32)
    naturalBaseline <- allocMem :: IO (Ptr Int32)
    gtk_widget_get_preferred_height_and_baseline_for_width widget' width minimumHeight naturalHeight minimumBaseline naturalBaseline
    minimumHeight' <- peek minimumHeight
    naturalHeight' <- peek naturalHeight
    minimumBaseline' <- peek minimumBaseline
    naturalBaseline' <- peek naturalBaseline
    touchManagedPtr widget
    freeMem minimumHeight
    freeMem naturalHeight
    freeMem minimumBaseline
    freeMem naturalBaseline
    return (minimumHeight', naturalHeight', minimumBaseline', naturalBaseline')

#if defined(ENABLE_OVERLOADING)
data WidgetGetPreferredHeightAndBaselineForWidthMethodInfo
instance (signature ~ (Int32 -> m ((Int32, Int32, Int32, Int32))), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetPreferredHeightAndBaselineForWidthMethodInfo a signature where
    overloadedMethod = widgetGetPreferredHeightAndBaselineForWidth

instance O.OverloadedMethodInfo WidgetGetPreferredHeightAndBaselineForWidthMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetPreferredHeightAndBaselineForWidth",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetPreferredHeightAndBaselineForWidth"
        })


#endif

-- method Widget::get_preferred_height_for_width
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget instance"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "width"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the width which is available for allocation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "minimum_height"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "location for storing the minimum height, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "natural_height"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "location for storing the natural height, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_preferred_height_for_width" gtk_widget_get_preferred_height_for_width :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Int32 ->                                -- width : TBasicType TInt
    Ptr Int32 ->                            -- minimum_height : TBasicType TInt
    Ptr Int32 ->                            -- natural_height : TBasicType TInt
    IO ()

-- | Retrieves a widget’s minimum and natural height if it would be given
-- the specified /@width@/.
-- 
-- The returned request will be modified by the
-- GtkWidgetClass[adjust_size_request](#g:signal:adjust_size_request) virtual method and by any
-- @/GtkSizeGroups/@ that have been applied. That is, the returned request
-- is the one that should be used for layout, not necessarily the one
-- returned by the widget itself.
-- 
-- /Since: 3.0/
widgetGetPreferredHeightForWidth ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' instance
    -> Int32
    -- ^ /@width@/: the width which is available for allocation
    -> m ((Int32, Int32))
widgetGetPreferredHeightForWidth widget width = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    minimumHeight <- allocMem :: IO (Ptr Int32)
    naturalHeight <- allocMem :: IO (Ptr Int32)
    gtk_widget_get_preferred_height_for_width widget' width minimumHeight naturalHeight
    minimumHeight' <- peek minimumHeight
    naturalHeight' <- peek naturalHeight
    touchManagedPtr widget
    freeMem minimumHeight
    freeMem naturalHeight
    return (minimumHeight', naturalHeight')

#if defined(ENABLE_OVERLOADING)
data WidgetGetPreferredHeightForWidthMethodInfo
instance (signature ~ (Int32 -> m ((Int32, Int32))), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetPreferredHeightForWidthMethodInfo a signature where
    overloadedMethod = widgetGetPreferredHeightForWidth

instance O.OverloadedMethodInfo WidgetGetPreferredHeightForWidthMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetPreferredHeightForWidth",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetPreferredHeightForWidth"
        })


#endif

-- method Widget::get_preferred_size
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget instance"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "minimum_size"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Requisition" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "location for storing the minimum size, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "natural_size"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Requisition" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "location for storing the natural size, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_preferred_size" gtk_widget_get_preferred_size :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gtk.Requisition.Requisition ->      -- minimum_size : TInterface (Name {namespace = "Gtk", name = "Requisition"})
    Ptr Gtk.Requisition.Requisition ->      -- natural_size : TInterface (Name {namespace = "Gtk", name = "Requisition"})
    IO ()

-- | Retrieves the minimum and natural size of a widget, taking
-- into account the widget’s preference for height-for-width management.
-- 
-- This is used to retrieve a suitable size by container widgets which do
-- not impose any restrictions on the child placement. It can be used
-- to deduce toplevel window and menu sizes as well as child widgets in
-- free-form containers such as GtkLayout.
-- 
-- Handle with care. Note that the natural height of a height-for-width
-- widget will generally be a smaller size than the minimum height, since the required
-- height for the natural width is generally smaller than the required height for
-- the minimum width.
-- 
-- Use 'GI.Gtk.Objects.Widget.widgetGetPreferredHeightAndBaselineForWidth' if you want to support
-- baseline alignment.
-- 
-- /Since: 3.0/
widgetGetPreferredSize ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' instance
    -> m ((Gtk.Requisition.Requisition, Gtk.Requisition.Requisition))
widgetGetPreferredSize widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    minimumSize <- SP.callocBoxedBytes 8 :: IO (Ptr Gtk.Requisition.Requisition)
    naturalSize <- SP.callocBoxedBytes 8 :: IO (Ptr Gtk.Requisition.Requisition)
    gtk_widget_get_preferred_size widget' minimumSize naturalSize
    minimumSize' <- (wrapBoxed Gtk.Requisition.Requisition) minimumSize
    naturalSize' <- (wrapBoxed Gtk.Requisition.Requisition) naturalSize
    touchManagedPtr widget
    return (minimumSize', naturalSize')

#if defined(ENABLE_OVERLOADING)
data WidgetGetPreferredSizeMethodInfo
instance (signature ~ (m ((Gtk.Requisition.Requisition, Gtk.Requisition.Requisition))), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetPreferredSizeMethodInfo a signature where
    overloadedMethod = widgetGetPreferredSize

instance O.OverloadedMethodInfo WidgetGetPreferredSizeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetPreferredSize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetPreferredSize"
        })


#endif

-- method Widget::get_preferred_width
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget instance"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "minimum_width"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to store the minimum width, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "natural_width"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to store the natural width, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_preferred_width" gtk_widget_get_preferred_width :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Int32 ->                            -- minimum_width : TBasicType TInt
    Ptr Int32 ->                            -- natural_width : TBasicType TInt
    IO ()

-- | Retrieves a widget’s initial minimum and natural width.
-- 
-- This call is specific to height-for-width requests.
-- 
-- The returned request will be modified by the
-- GtkWidgetClass[adjust_size_request](#g:signal:adjust_size_request) virtual method and by any
-- @/GtkSizeGroups/@ that have been applied. That is, the returned request
-- is the one that should be used for layout, not necessarily the one
-- returned by the widget itself.
-- 
-- /Since: 3.0/
widgetGetPreferredWidth ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' instance
    -> m ((Int32, Int32))
widgetGetPreferredWidth widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    minimumWidth <- allocMem :: IO (Ptr Int32)
    naturalWidth <- allocMem :: IO (Ptr Int32)
    gtk_widget_get_preferred_width widget' minimumWidth naturalWidth
    minimumWidth' <- peek minimumWidth
    naturalWidth' <- peek naturalWidth
    touchManagedPtr widget
    freeMem minimumWidth
    freeMem naturalWidth
    return (minimumWidth', naturalWidth')

#if defined(ENABLE_OVERLOADING)
data WidgetGetPreferredWidthMethodInfo
instance (signature ~ (m ((Int32, Int32))), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetPreferredWidthMethodInfo a signature where
    overloadedMethod = widgetGetPreferredWidth

instance O.OverloadedMethodInfo WidgetGetPreferredWidthMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetPreferredWidth",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetPreferredWidth"
        })


#endif

-- method Widget::get_preferred_width_for_height
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget instance"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "height"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the height which is available for allocation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "minimum_width"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "location for storing the minimum width, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "natural_width"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "location for storing the natural width, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_preferred_width_for_height" gtk_widget_get_preferred_width_for_height :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Int32 ->                                -- height : TBasicType TInt
    Ptr Int32 ->                            -- minimum_width : TBasicType TInt
    Ptr Int32 ->                            -- natural_width : TBasicType TInt
    IO ()

-- | Retrieves a widget’s minimum and natural width if it would be given
-- the specified /@height@/.
-- 
-- The returned request will be modified by the
-- GtkWidgetClass[adjust_size_request](#g:signal:adjust_size_request) virtual method and by any
-- @/GtkSizeGroups/@ that have been applied. That is, the returned request
-- is the one that should be used for layout, not necessarily the one
-- returned by the widget itself.
-- 
-- /Since: 3.0/
widgetGetPreferredWidthForHeight ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' instance
    -> Int32
    -- ^ /@height@/: the height which is available for allocation
    -> m ((Int32, Int32))
widgetGetPreferredWidthForHeight widget height = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    minimumWidth <- allocMem :: IO (Ptr Int32)
    naturalWidth <- allocMem :: IO (Ptr Int32)
    gtk_widget_get_preferred_width_for_height widget' height minimumWidth naturalWidth
    minimumWidth' <- peek minimumWidth
    naturalWidth' <- peek naturalWidth
    touchManagedPtr widget
    freeMem minimumWidth
    freeMem naturalWidth
    return (minimumWidth', naturalWidth')

#if defined(ENABLE_OVERLOADING)
data WidgetGetPreferredWidthForHeightMethodInfo
instance (signature ~ (Int32 -> m ((Int32, Int32))), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetPreferredWidthForHeightMethodInfo a signature where
    overloadedMethod = widgetGetPreferredWidthForHeight

instance O.OverloadedMethodInfo WidgetGetPreferredWidthForHeightMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetPreferredWidthForHeight",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetPreferredWidthForHeight"
        })


#endif

-- method Widget::get_realized
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_realized" gtk_widget_get_realized :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether /@widget@/ is realized.
-- 
-- /Since: 2.20/
widgetGetRealized ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@widget@/ is realized, 'P.False' otherwise
widgetGetRealized widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_realized widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetRealizedMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetRealizedMethodInfo a signature where
    overloadedMethod = widgetGetRealized

instance O.OverloadedMethodInfo WidgetGetRealizedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetRealized",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetRealized"
        })


#endif

-- method Widget::get_receives_default
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_receives_default" gtk_widget_get_receives_default :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether /@widget@/ is always treated as the default widget
-- within its toplevel when it has the focus, even if another widget
-- is the default.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetReceivesDefault'.
-- 
-- /Since: 2.18/
widgetGetReceivesDefault ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@widget@/ acts as the default widget when focused,
    --               'P.False' otherwise
widgetGetReceivesDefault widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_receives_default widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetReceivesDefaultMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetReceivesDefaultMethodInfo a signature where
    overloadedMethod = widgetGetReceivesDefault

instance O.OverloadedMethodInfo WidgetGetReceivesDefaultMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetReceivesDefault",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetReceivesDefault"
        })


#endif

-- method Widget::get_request_mode
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget instance"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gtk" , name = "SizeRequestMode" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_request_mode" gtk_widget_get_request_mode :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CUInt

-- | Gets whether the widget prefers a height-for-width layout
-- or a width-for-height layout.
-- 
-- t'GI.Gtk.Objects.Bin.Bin' widgets generally propagate the preference of
-- their child, container widgets need to request something either in
-- context of their children or in context of their allocation
-- capabilities.
-- 
-- /Since: 3.0/
widgetGetRequestMode ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget' instance
    -> m Gtk.Enums.SizeRequestMode
    -- ^ __Returns:__ The t'GI.Gtk.Enums.SizeRequestMode' preferred by /@widget@/.
widgetGetRequestMode widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_request_mode widget'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetRequestModeMethodInfo
instance (signature ~ (m Gtk.Enums.SizeRequestMode), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetRequestModeMethodInfo a signature where
    overloadedMethod = widgetGetRequestMode

instance O.OverloadedMethodInfo WidgetGetRequestModeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetRequestMode",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetRequestMode"
        })


#endif

-- method Widget::get_requisition
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "requisition"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Requisition" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a pointer to a #GtkRequisition to copy to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_requisition" gtk_widget_get_requisition :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gtk.Requisition.Requisition ->      -- requisition : TInterface (Name {namespace = "Gtk", name = "Requisition"})
    IO ()

{-# DEPRECATED widgetGetRequisition ["(Since version 3.0)","The t'GI.Gtk.Structs.Requisition.Requisition' cache on the widget was","removed, If you need to cache sizes across requests and allocations,","add an explicit cache to the widget in question instead."] #-}
-- | Retrieves the widget’s requisition.
-- 
-- This function should only be used by widget implementations in
-- order to figure whether the widget’s requisition has actually
-- changed after some internal state change (so that they can call
-- 'GI.Gtk.Objects.Widget.widgetQueueResize' instead of 'GI.Gtk.Objects.Widget.widgetQueueDraw').
-- 
-- Normally, 'GI.Gtk.Objects.Widget.widgetSizeRequest' should be used.
-- 
-- /Since: 2.20/
widgetGetRequisition ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m (Gtk.Requisition.Requisition)
widgetGetRequisition widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    requisition <- SP.callocBoxedBytes 8 :: IO (Ptr Gtk.Requisition.Requisition)
    gtk_widget_get_requisition widget' requisition
    requisition' <- (wrapBoxed Gtk.Requisition.Requisition) requisition
    touchManagedPtr widget
    return requisition'

#if defined(ENABLE_OVERLOADING)
data WidgetGetRequisitionMethodInfo
instance (signature ~ (m (Gtk.Requisition.Requisition)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetRequisitionMethodInfo a signature where
    overloadedMethod = widgetGetRequisition

instance O.OverloadedMethodInfo WidgetGetRequisitionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetRequisition",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetRequisition"
        })


#endif

-- method Widget::get_root_window
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Window" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_root_window" gtk_widget_get_root_window :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gdk.Window.Window)

{-# DEPRECATED widgetGetRootWindow ["(Since version 3.12)","Use 'GI.Gdk.Objects.Screen.screenGetRootWindow' instead"] #-}
-- | Get the root window where this widget is located. This function can
-- only be called after the widget has been added to a widget
-- hierarchy with t'GI.Gtk.Objects.Window.Window' at the top.
-- 
-- The root window is useful for such purposes as creating a popup
-- t'GI.Gdk.Objects.Window.Window' associated with the window. In general, you should only
-- create display specific resources when a widget has been realized,
-- and you should free those resources when the widget is unrealized.
-- 
-- /Since: 2.2/
widgetGetRootWindow ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Gdk.Window.Window
    -- ^ __Returns:__ the t'GI.Gdk.Objects.Window.Window' root window for the toplevel for this widget.
widgetGetRootWindow widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_root_window widget'
    checkUnexpectedReturnNULL "widgetGetRootWindow" result
    result' <- (newObject Gdk.Window.Window) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetRootWindowMethodInfo
instance (signature ~ (m Gdk.Window.Window), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetRootWindowMethodInfo a signature where
    overloadedMethod = widgetGetRootWindow

instance O.OverloadedMethodInfo WidgetGetRootWindowMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetRootWindow",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetRootWindow"
        })


#endif

-- method Widget::get_scale_factor
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_scale_factor" gtk_widget_get_scale_factor :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

-- | Retrieves the internal scale factor that maps from window coordinates
-- to the actual device pixels. On traditional systems this is 1, on
-- high density outputs, it can be a higher value (typically 2).
-- 
-- See 'GI.Gdk.Objects.Window.windowGetScaleFactor'.
-- 
-- /Since: 3.10/
widgetGetScaleFactor ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Int32
    -- ^ __Returns:__ the scale factor for /@widget@/
widgetGetScaleFactor widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_scale_factor widget'
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data WidgetGetScaleFactorMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetScaleFactorMethodInfo a signature where
    overloadedMethod = widgetGetScaleFactor

instance O.OverloadedMethodInfo WidgetGetScaleFactorMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetScaleFactor",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetScaleFactor"
        })


#endif

-- method Widget::get_screen
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Screen" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_screen" gtk_widget_get_screen :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gdk.Screen.Screen)

-- | Get the t'GI.Gdk.Objects.Screen.Screen' from the toplevel window associated with
-- this widget. This function can only be called after the widget
-- has been added to a widget hierarchy with a t'GI.Gtk.Objects.Window.Window'
-- at the top.
-- 
-- In general, you should only create screen specific
-- resources when a widget has been realized, and you should
-- free those resources when the widget is unrealized.
-- 
-- /Since: 2.2/
widgetGetScreen ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Gdk.Screen.Screen
    -- ^ __Returns:__ the t'GI.Gdk.Objects.Screen.Screen' for the toplevel for this widget.
widgetGetScreen widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_screen widget'
    checkUnexpectedReturnNULL "widgetGetScreen" result
    result' <- (newObject Gdk.Screen.Screen) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetScreenMethodInfo
instance (signature ~ (m Gdk.Screen.Screen), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetScreenMethodInfo a signature where
    overloadedMethod = widgetGetScreen

instance O.OverloadedMethodInfo WidgetGetScreenMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetScreen",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetScreen"
        })


#endif

-- method Widget::get_sensitive
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_sensitive" gtk_widget_get_sensitive :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Returns the widget’s sensitivity (in the sense of returning
-- the value that has been set using 'GI.Gtk.Objects.Widget.widgetSetSensitive').
-- 
-- The effective sensitivity of a widget is however determined by both its
-- own and its parent widget’s sensitivity. See 'GI.Gtk.Objects.Widget.widgetIsSensitive'.
-- 
-- /Since: 2.18/
widgetGetSensitive ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the widget is sensitive
widgetGetSensitive widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_sensitive widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetSensitiveMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetSensitiveMethodInfo a signature where
    overloadedMethod = widgetGetSensitive

instance O.OverloadedMethodInfo WidgetGetSensitiveMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetSensitive",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetSensitive"
        })


#endif

-- method Widget::get_settings
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Settings" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_settings" gtk_widget_get_settings :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gtk.Settings.Settings)

-- | Gets the settings object holding the settings used for this widget.
-- 
-- Note that this function can only be called when the t'GI.Gtk.Objects.Widget.Widget'
-- is attached to a toplevel, since the settings object is specific
-- to a particular t'GI.Gdk.Objects.Screen.Screen'.
widgetGetSettings ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Gtk.Settings.Settings
    -- ^ __Returns:__ the relevant t'GI.Gtk.Objects.Settings.Settings' object
widgetGetSettings widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_settings widget'
    checkUnexpectedReturnNULL "widgetGetSettings" result
    result' <- (newObject Gtk.Settings.Settings) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetSettingsMethodInfo
instance (signature ~ (m Gtk.Settings.Settings), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetSettingsMethodInfo a signature where
    overloadedMethod = widgetGetSettings

instance O.OverloadedMethodInfo WidgetGetSettingsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetSettings",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetSettings"
        })


#endif

-- method Widget::get_size_request
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "width"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for width, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "height"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for height, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_size_request" gtk_widget_get_size_request :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Int32 ->                            -- width : TBasicType TInt
    Ptr Int32 ->                            -- height : TBasicType TInt
    IO ()

-- | Gets the size request that was explicitly set for the widget using
-- 'GI.Gtk.Objects.Widget.widgetSetSizeRequest'. A value of -1 stored in /@width@/ or
-- /@height@/ indicates that that dimension has not been set explicitly
-- and the natural requisition of the widget will be used instead. See
-- 'GI.Gtk.Objects.Widget.widgetSetSizeRequest'. To get the size a widget will
-- actually request, call 'GI.Gtk.Objects.Widget.widgetGetPreferredSize' instead of
-- this function.
widgetGetSizeRequest ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m ((Int32, Int32))
widgetGetSizeRequest widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    width <- allocMem :: IO (Ptr Int32)
    height <- allocMem :: IO (Ptr Int32)
    gtk_widget_get_size_request widget' width height
    width' <- peek width
    height' <- peek height
    touchManagedPtr widget
    freeMem width
    freeMem height
    return (width', height')

#if defined(ENABLE_OVERLOADING)
data WidgetGetSizeRequestMethodInfo
instance (signature ~ (m ((Int32, Int32))), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetSizeRequestMethodInfo a signature where
    overloadedMethod = widgetGetSizeRequest

instance O.OverloadedMethodInfo WidgetGetSizeRequestMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetSizeRequest",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetSizeRequest"
        })


#endif

-- method Widget::get_state
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "StateType" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_state" gtk_widget_get_state :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CUInt

{-# DEPRECATED widgetGetState ["(Since version 3.0)","Use 'GI.Gtk.Objects.Widget.widgetGetStateFlags' instead."] #-}
-- | Returns the widget’s state. See 'GI.Gtk.Objects.Widget.widgetSetState'.
-- 
-- /Since: 2.18/
widgetGetState ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Gtk.Enums.StateType
    -- ^ __Returns:__ the state of /@widget@/.
widgetGetState widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_state widget'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetStateMethodInfo
instance (signature ~ (m Gtk.Enums.StateType), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetStateMethodInfo a signature where
    overloadedMethod = widgetGetState

instance O.OverloadedMethodInfo WidgetGetStateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetState",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetState"
        })


#endif

-- method Widget::get_state_flags
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "StateFlags" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_state_flags" gtk_widget_get_state_flags :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CUInt

-- | Returns the widget state as a flag set. It is worth mentioning
-- that the effective 'GI.Gtk.Flags.StateFlagsInsensitive' state will be
-- returned, that is, also based on parent insensitivity, even if
-- /@widget@/ itself is sensitive.
-- 
-- Also note that if you are looking for a way to obtain the
-- t'GI.Gtk.Flags.StateFlags' to pass to a t'GI.Gtk.Objects.StyleContext.StyleContext' method, you
-- should look at 'GI.Gtk.Objects.StyleContext.styleContextGetState'.
-- 
-- /Since: 3.0/
widgetGetStateFlags ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m [Gtk.Flags.StateFlags]
    -- ^ __Returns:__ The state flags for widget
widgetGetStateFlags widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_state_flags widget'
    let result' = wordToGFlags result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetStateFlagsMethodInfo
instance (signature ~ (m [Gtk.Flags.StateFlags]), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetStateFlagsMethodInfo a signature where
    overloadedMethod = widgetGetStateFlags

instance O.OverloadedMethodInfo WidgetGetStateFlagsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetStateFlags",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetStateFlags"
        })


#endif

-- method Widget::get_style
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Style" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_style" gtk_widget_get_style :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gtk.Style.Style)

{-# DEPRECATED widgetGetStyle ["(Since version 3.0)","Use t'GI.Gtk.Objects.StyleContext.StyleContext' instead"] #-}
-- | Simply an accessor function that returns /@widget@/->style.
widgetGetStyle ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Gtk.Style.Style
    -- ^ __Returns:__ the widget’s t'GI.Gtk.Objects.Style.Style'
widgetGetStyle widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_style widget'
    checkUnexpectedReturnNULL "widgetGetStyle" result
    result' <- (newObject Gtk.Style.Style) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetStyleMethodInfo
instance (signature ~ (m Gtk.Style.Style), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetStyleMethodInfo a signature where
    overloadedMethod = widgetGetStyle

instance O.OverloadedMethodInfo WidgetGetStyleMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetStyle",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetStyle"
        })


#endif

-- method Widget::get_style_context
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gtk" , name = "StyleContext" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_style_context" gtk_widget_get_style_context :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gtk.StyleContext.StyleContext)

-- | Returns the style context associated to /@widget@/. The returned object is
-- guaranteed to be the same for the lifetime of /@widget@/.
widgetGetStyleContext ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Gtk.StyleContext.StyleContext
    -- ^ __Returns:__ a t'GI.Gtk.Objects.StyleContext.StyleContext'. This memory is owned by /@widget@/ and
    --          must not be freed.
widgetGetStyleContext widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_style_context widget'
    checkUnexpectedReturnNULL "widgetGetStyleContext" result
    result' <- (newObject Gtk.StyleContext.StyleContext) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetStyleContextMethodInfo
instance (signature ~ (m Gtk.StyleContext.StyleContext), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetStyleContextMethodInfo a signature where
    overloadedMethod = widgetGetStyleContext

instance O.OverloadedMethodInfo WidgetGetStyleContextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetStyleContext",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetStyleContext"
        })


#endif

-- method Widget::get_support_multidevice
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_support_multidevice" gtk_widget_get_support_multidevice :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Returns 'P.True' if /@widget@/ is multiple pointer aware. See
-- 'GI.Gtk.Objects.Widget.widgetSetSupportMultidevice' for more information.
widgetGetSupportMultidevice ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@widget@/ is multidevice aware.
widgetGetSupportMultidevice widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_support_multidevice widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetSupportMultideviceMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetSupportMultideviceMethodInfo a signature where
    overloadedMethod = widgetGetSupportMultidevice

instance O.OverloadedMethodInfo WidgetGetSupportMultideviceMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetSupportMultidevice",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetSupportMultidevice"
        })


#endif

-- method Widget::get_template_child
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "widget_type"
--           , argType = TBasicType TGType
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "The #GType to get a template child for"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "The \8220id\8221 of the child defined in the template XML"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GObject" , name = "Object" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_template_child" gtk_widget_get_template_child :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CGType ->                               -- widget_type : TBasicType TGType
    CString ->                              -- name : TBasicType TUTF8
    IO (Ptr GObject.Object.Object)

-- | Fetch an object build from the template XML for /@widgetType@/ in this /@widget@/ instance.
-- 
-- This will only report children which were previously declared with
-- 'GI.Gtk.Structs.WidgetClass.widgetClassBindTemplateChildFull' or one of its
-- variants.
-- 
-- This function is only meant to be called for code which is private to the /@widgetType@/ which
-- declared the child and is meant for language bindings which cannot easily make use
-- of the GObject structure offsets.
widgetGetTemplateChild ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: A t'GI.Gtk.Objects.Widget.Widget'
    -> GType
    -- ^ /@widgetType@/: The t'GType' to get a template child for
    -> T.Text
    -- ^ /@name@/: The “id” of the child defined in the template XML
    -> m GObject.Object.Object
    -- ^ __Returns:__ The object built in the template XML with the id /@name@/
widgetGetTemplateChild widget widgetType name = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    let widgetType' = gtypeToCGType widgetType
    name' <- textToCString name
    result <- gtk_widget_get_template_child widget' widgetType' name'
    checkUnexpectedReturnNULL "widgetGetTemplateChild" result
    result' <- (newObject GObject.Object.Object) result
    touchManagedPtr widget
    freeMem name'
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetTemplateChildMethodInfo
instance (signature ~ (GType -> T.Text -> m GObject.Object.Object), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetTemplateChildMethodInfo a signature where
    overloadedMethod = widgetGetTemplateChild

instance O.OverloadedMethodInfo WidgetGetTemplateChildMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetTemplateChild",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetTemplateChild"
        })


#endif

-- method Widget::get_tooltip_markup
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_tooltip_markup" gtk_widget_get_tooltip_markup :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CString

-- | Gets the contents of the tooltip for /@widget@/.
-- 
-- /Since: 2.12/
widgetGetTooltipMarkup ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ the tooltip text, or 'P.Nothing'. You should free the
    --   returned string with 'GI.GLib.Functions.free' when done.
widgetGetTooltipMarkup widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_tooltip_markup widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        freeMem result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetTooltipMarkupMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetTooltipMarkupMethodInfo a signature where
    overloadedMethod = widgetGetTooltipMarkup

instance O.OverloadedMethodInfo WidgetGetTooltipMarkupMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetTooltipMarkup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetTooltipMarkup"
        })


#endif

-- method Widget::get_tooltip_text
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_tooltip_text" gtk_widget_get_tooltip_text :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CString

-- | Gets the contents of the tooltip for /@widget@/.
-- 
-- /Since: 2.12/
widgetGetTooltipText ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m (Maybe T.Text)
    -- ^ __Returns:__ the tooltip text, or 'P.Nothing'. You should free the
    --   returned string with 'GI.GLib.Functions.free' when done.
widgetGetTooltipText widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_tooltip_text widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        freeMem result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetTooltipTextMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetTooltipTextMethodInfo a signature where
    overloadedMethod = widgetGetTooltipText

instance O.OverloadedMethodInfo WidgetGetTooltipTextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetTooltipText",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetTooltipText"
        })


#endif

-- method Widget::get_tooltip_window
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Window" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_tooltip_window" gtk_widget_get_tooltip_window :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gtk.Window.Window)

-- | Returns the t'GI.Gtk.Objects.Window.Window' of the current tooltip. This can be the
-- GtkWindow created by default, or the custom tooltip window set
-- using 'GI.Gtk.Objects.Widget.widgetSetTooltipWindow'.
-- 
-- /Since: 2.12/
widgetGetTooltipWindow ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Gtk.Window.Window
    -- ^ __Returns:__ The t'GI.Gtk.Objects.Window.Window' of the current tooltip.
widgetGetTooltipWindow widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_tooltip_window widget'
    checkUnexpectedReturnNULL "widgetGetTooltipWindow" result
    result' <- (newObject Gtk.Window.Window) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetTooltipWindowMethodInfo
instance (signature ~ (m Gtk.Window.Window), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetTooltipWindowMethodInfo a signature where
    overloadedMethod = widgetGetTooltipWindow

instance O.OverloadedMethodInfo WidgetGetTooltipWindowMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetTooltipWindow",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetTooltipWindow"
        })


#endif

-- method Widget::get_toplevel
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Widget" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_toplevel" gtk_widget_get_toplevel :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Widget)

-- | This function returns the topmost widget in the container hierarchy
-- /@widget@/ is a part of. If /@widget@/ has no parent widgets, it will be
-- returned as the topmost widget. No reference will be added to the
-- returned widget; it should not be unreferenced.
-- 
-- Note the difference in behavior vs. 'GI.Gtk.Objects.Widget.widgetGetAncestor';
-- @gtk_widget_get_ancestor (widget, GTK_TYPE_WINDOW)@
-- would return
-- 'P.Nothing' if /@widget@/ wasn’t inside a toplevel window, and if the
-- window was inside a t'GI.Gtk.Objects.Window.Window'-derived widget which was in turn
-- inside the toplevel t'GI.Gtk.Objects.Window.Window'. While the second case may
-- seem unlikely, it actually happens when a t'GI.Gtk.Objects.Plug.Plug' is embedded
-- inside a t'GI.Gtk.Objects.Socket.Socket' within the same application.
-- 
-- To reliably find the toplevel t'GI.Gtk.Objects.Window.Window', use
-- 'GI.Gtk.Objects.Widget.widgetGetToplevel' and call @/GTK_IS_WINDOW()/@
-- on the result. For instance, to get the title of a widget\'s toplevel
-- window, one might use:
-- 
-- === /C code/
-- >
-- >static const char *
-- >get_widget_toplevel_title (GtkWidget *widget)
-- >{
-- >  GtkWidget *toplevel = gtk_widget_get_toplevel (widget);
-- >  if (GTK_IS_WINDOW (toplevel))
-- >    {
-- >      return gtk_window_get_title (GTK_WINDOW (toplevel));
-- >    }
-- >
-- >  return NULL;
-- >}
widgetGetToplevel ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Widget
    -- ^ __Returns:__ the topmost ancestor of /@widget@/, or /@widget@/ itself
    --    if there’s no ancestor.
widgetGetToplevel widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_toplevel widget'
    checkUnexpectedReturnNULL "widgetGetToplevel" result
    result' <- (newObject Widget) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetToplevelMethodInfo
instance (signature ~ (m Widget), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetToplevelMethodInfo a signature where
    overloadedMethod = widgetGetToplevel

instance O.OverloadedMethodInfo WidgetGetToplevelMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetToplevel",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetToplevel"
        })


#endif

-- method Widget::get_valign
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Align" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_valign" gtk_widget_get_valign :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CUInt

-- | Gets the value of the [Widget:valign]("GI.Gtk.Objects.Widget#g:attr:valign") property.
-- 
-- For backwards compatibility reasons this method will never return
-- 'GI.Gtk.Enums.AlignBaseline', but instead it will convert it to
-- 'GI.Gtk.Enums.AlignFill'. If your widget want to support baseline aligned
-- children it must use 'GI.Gtk.Objects.Widget.widgetGetValignWithBaseline', or
-- @g_object_get (widget, \"valign\", &value, NULL)@, which will
-- also report the true value.
widgetGetValign ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Gtk.Enums.Align
    -- ^ __Returns:__ the vertical alignment of /@widget@/, ignoring baseline alignment
widgetGetValign widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_valign widget'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetValignMethodInfo
instance (signature ~ (m Gtk.Enums.Align), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetValignMethodInfo a signature where
    overloadedMethod = widgetGetValign

instance O.OverloadedMethodInfo WidgetGetValignMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetValign",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetValign"
        })


#endif

-- method Widget::get_valign_with_baseline
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Align" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_valign_with_baseline" gtk_widget_get_valign_with_baseline :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CUInt

-- | Gets the value of the [Widget:valign]("GI.Gtk.Objects.Widget#g:attr:valign") property, including
-- 'GI.Gtk.Enums.AlignBaseline'.
-- 
-- /Since: 3.10/
widgetGetValignWithBaseline ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Gtk.Enums.Align
    -- ^ __Returns:__ the vertical alignment of /@widget@/
widgetGetValignWithBaseline widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_valign_with_baseline widget'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetValignWithBaselineMethodInfo
instance (signature ~ (m Gtk.Enums.Align), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetValignWithBaselineMethodInfo a signature where
    overloadedMethod = widgetGetValignWithBaseline

instance O.OverloadedMethodInfo WidgetGetValignWithBaselineMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetValignWithBaseline",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetValignWithBaseline"
        })


#endif

-- method Widget::get_vexpand
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_vexpand" gtk_widget_get_vexpand :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Gets whether the widget would like any available extra vertical
-- space.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetGetHexpand' for more detail.
widgetGetVexpand ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget
    -> m Bool
    -- ^ __Returns:__ whether vexpand flag is set
widgetGetVexpand widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_vexpand widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetVexpandMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetVexpandMethodInfo a signature where
    overloadedMethod = widgetGetVexpand

instance O.OverloadedMethodInfo WidgetGetVexpandMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetVexpand",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetVexpand"
        })


#endif

-- method Widget::get_vexpand_set
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the widget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_vexpand_set" gtk_widget_get_vexpand_set :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Gets whether 'GI.Gtk.Objects.Widget.widgetSetVexpand' has been used to
-- explicitly set the expand flag on this widget.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetGetHexpandSet' for more detail.
widgetGetVexpandSet ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: the widget
    -> m Bool
    -- ^ __Returns:__ whether vexpand has been explicitly set
widgetGetVexpandSet widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_vexpand_set widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetVexpandSetMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetVexpandSetMethodInfo a signature where
    overloadedMethod = widgetGetVexpandSet

instance O.OverloadedMethodInfo WidgetGetVexpandSetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetVexpandSet",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetVexpandSet"
        })


#endif

-- method Widget::get_visible
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_visible" gtk_widget_get_visible :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether the widget is visible. If you want to
-- take into account whether the widget’s parent is also marked as
-- visible, use 'GI.Gtk.Objects.Widget.widgetIsVisible' instead.
-- 
-- This function does not check if the widget is obscured in any way.
-- 
-- See 'GI.Gtk.Objects.Widget.widgetSetVisible'.
-- 
-- /Since: 2.18/
widgetGetVisible ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the widget is visible
widgetGetVisible widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_visible widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetVisibleMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetVisibleMethodInfo a signature where
    overloadedMethod = widgetGetVisible

instance O.OverloadedMethodInfo WidgetGetVisibleMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetVisible",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetVisible"
        })


#endif

-- method Widget::get_visual
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Visual" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_visual" gtk_widget_get_visual :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gdk.Visual.Visual)

-- | Gets the visual that will be used to render /@widget@/.
widgetGetVisual ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Gdk.Visual.Visual
    -- ^ __Returns:__ the visual for /@widget@/
widgetGetVisual widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_visual widget'
    checkUnexpectedReturnNULL "widgetGetVisual" result
    result' <- (newObject Gdk.Visual.Visual) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetGetVisualMethodInfo
instance (signature ~ (m Gdk.Visual.Visual), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetVisualMethodInfo a signature where
    overloadedMethod = widgetGetVisual

instance O.OverloadedMethodInfo WidgetGetVisualMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetVisual",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetVisual"
        })


#endif

-- method Widget::get_window
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gdk" , name = "Window" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_get_window" gtk_widget_get_window :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO (Ptr Gdk.Window.Window)

-- | Returns the widget’s window if it is realized, 'P.Nothing' otherwise
-- 
-- /Since: 2.14/
widgetGetWindow ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m (Maybe Gdk.Window.Window)
    -- ^ __Returns:__ /@widget@/’s window.
widgetGetWindow widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_get_window widget'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gdk.Window.Window) result'
        return result''
    touchManagedPtr widget
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data WidgetGetWindowMethodInfo
instance (signature ~ (m (Maybe Gdk.Window.Window)), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGetWindowMethodInfo a signature where
    overloadedMethod = widgetGetWindow

instance O.OverloadedMethodInfo WidgetGetWindowMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGetWindow",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGetWindow"
        })


#endif

-- method Widget::grab_add
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "The widget that grabs keyboard and pointer events"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_grab_add" gtk_grab_add :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Makes /@widget@/ the current grabbed widget.
-- 
-- This means that interaction with other widgets in the same
-- application is blocked and mouse as well as keyboard events
-- are delivered to this widget.
-- 
-- If /@widget@/ is not sensitive, it is not set as the current
-- grabbed widget and this function does nothing.
widgetGrabAdd ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: The widget that grabs keyboard and pointer events
    -> m ()
widgetGrabAdd widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_grab_add widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetGrabAddMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGrabAddMethodInfo a signature where
    overloadedMethod = widgetGrabAdd

instance O.OverloadedMethodInfo WidgetGrabAddMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGrabAdd",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGrabAdd"
        })


#endif

-- method Widget::grab_default
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_grab_default" gtk_widget_grab_default :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Causes /@widget@/ to become the default widget. /@widget@/ must be able to be
-- a default widget; typically you would ensure this yourself
-- by calling 'GI.Gtk.Objects.Widget.widgetSetCanDefault' with a 'P.True' value.
-- The default widget is activated when
-- the user presses Enter in a window. Default widgets must be
-- activatable, that is, 'GI.Gtk.Objects.Widget.widgetActivate' should affect them. Note
-- that t'GI.Gtk.Objects.Entry.Entry' widgets require the “activates-default” property
-- set to 'P.True' before they activate the default widget when Enter
-- is pressed and the t'GI.Gtk.Objects.Entry.Entry' is focused.
widgetGrabDefault ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m ()
widgetGrabDefault widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_grab_default widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetGrabDefaultMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGrabDefaultMethodInfo a signature where
    overloadedMethod = widgetGrabDefault

instance O.OverloadedMethodInfo WidgetGrabDefaultMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGrabDefault",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGrabDefault"
        })


#endif

-- method Widget::grab_focus
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_grab_focus" gtk_widget_grab_focus :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Causes /@widget@/ to have the keyboard focus for the t'GI.Gtk.Objects.Window.Window' it\'s
-- inside. /@widget@/ must be a focusable widget, such as a t'GI.Gtk.Objects.Entry.Entry';
-- something like t'GI.Gtk.Objects.Frame.Frame' won’t work.
-- 
-- More precisely, it must have the @/GTK_CAN_FOCUS/@ flag set. Use
-- 'GI.Gtk.Objects.Widget.widgetSetCanFocus' to modify that flag.
-- 
-- The widget also needs to be realized and mapped. This is indicated by the
-- related signals. Grabbing the focus immediately after creating the widget
-- will likely fail and cause critical warnings.
widgetGrabFocus ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m ()
widgetGrabFocus widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_grab_focus widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetGrabFocusMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGrabFocusMethodInfo a signature where
    overloadedMethod = widgetGrabFocus

instance O.OverloadedMethodInfo WidgetGrabFocusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGrabFocus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGrabFocus"
        })


#endif

-- method Widget::grab_remove
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "The widget which gives up the grab"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_grab_remove" gtk_grab_remove :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Removes the grab from the given widget.
-- 
-- You have to pair calls to 'GI.Gtk.Objects.Widget.widgetGrabAdd' and 'GI.Gtk.Objects.Widget.widgetGrabRemove'.
-- 
-- If /@widget@/ does not have the grab, this function does nothing.
widgetGrabRemove ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: The widget which gives up the grab
    -> m ()
widgetGrabRemove widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_grab_remove widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetGrabRemoveMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetGrabRemoveMethodInfo a signature where
    overloadedMethod = widgetGrabRemove

instance O.OverloadedMethodInfo WidgetGrabRemoveMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetGrabRemove",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetGrabRemove"
        })


#endif

-- method Widget::has_default
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_has_default" gtk_widget_has_default :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether /@widget@/ is the current default widget within its
-- toplevel. See 'GI.Gtk.Objects.Widget.widgetSetCanDefault'.
-- 
-- /Since: 2.18/
widgetHasDefault ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@widget@/ is the current default widget within
    --     its toplevel, 'P.False' otherwise
widgetHasDefault widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_has_default widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetHasDefaultMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetHasDefaultMethodInfo a signature where
    overloadedMethod = widgetHasDefault

instance O.OverloadedMethodInfo WidgetHasDefaultMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetHasDefault",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetHasDefault"
        })


#endif

-- method Widget::has_focus
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_has_focus" gtk_widget_has_focus :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines if the widget has the global input focus. See
-- 'GI.Gtk.Objects.Widget.widgetIsFocus' for the difference between having the global
-- input focus, and only having the focus within a toplevel.
-- 
-- /Since: 2.18/
widgetHasFocus ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the widget has the global input focus.
widgetHasFocus widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_has_focus widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetHasFocusMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetHasFocusMethodInfo a signature where
    overloadedMethod = widgetHasFocus

instance O.OverloadedMethodInfo WidgetHasFocusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetHasFocus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetHasFocus"
        })


#endif

-- method Widget::has_grab
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_has_grab" gtk_widget_has_grab :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether the widget is currently grabbing events, so it
-- is the only widget receiving input events (keyboard and mouse).
-- 
-- See also 'GI.Gtk.Objects.Widget.widgetGrabAdd'.
-- 
-- /Since: 2.18/
widgetHasGrab ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the widget is in the grab_widgets stack
widgetHasGrab widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_has_grab widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetHasGrabMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetHasGrabMethodInfo a signature where
    overloadedMethod = widgetHasGrab

instance O.OverloadedMethodInfo WidgetHasGrabMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetHasGrab",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetHasGrab"
        })


#endif

-- method Widget::has_rc_style
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_has_rc_style" gtk_widget_has_rc_style :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

{-# DEPRECATED widgetHasRcStyle ["(Since version 3.0)","Use t'GI.Gtk.Objects.StyleContext.StyleContext' instead"] #-}
-- | Determines if the widget style has been looked up through the rc mechanism.
-- 
-- /Since: 2.20/
widgetHasRcStyle ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the widget has been looked up through the rc
    --   mechanism, 'P.False' otherwise.
widgetHasRcStyle widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_has_rc_style widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetHasRcStyleMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetHasRcStyleMethodInfo a signature where
    overloadedMethod = widgetHasRcStyle

instance O.OverloadedMethodInfo WidgetHasRcStyleMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetHasRcStyle",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetHasRcStyle"
        })


#endif

-- method Widget::has_screen
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_has_screen" gtk_widget_has_screen :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Checks whether there is a t'GI.Gdk.Objects.Screen.Screen' is associated with
-- this widget. All toplevel widgets have an associated
-- screen, and all widgets added into a hierarchy with a toplevel
-- window at the top.
-- 
-- /Since: 2.2/
widgetHasScreen ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if there is a t'GI.Gdk.Objects.Screen.Screen' associated
    --   with the widget.
widgetHasScreen widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_has_screen widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetHasScreenMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetHasScreenMethodInfo a signature where
    overloadedMethod = widgetHasScreen

instance O.OverloadedMethodInfo WidgetHasScreenMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetHasScreen",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetHasScreen"
        })


#endif

-- method Widget::has_visible_focus
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_has_visible_focus" gtk_widget_has_visible_focus :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines if the widget should show a visible indication that
-- it has the global input focus. This is a convenience function for
-- use in [draw](#g:signal:draw) handlers that takes into account whether focus
-- indication should currently be shown in the toplevel window of
-- /@widget@/. See 'GI.Gtk.Objects.Window.windowGetFocusVisible' for more information
-- about focus indication.
-- 
-- To find out if the widget has the global input focus, use
-- 'GI.Gtk.Objects.Widget.widgetHasFocus'.
-- 
-- /Since: 3.2/
widgetHasVisibleFocus ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the widget should display a “focus rectangle”
widgetHasVisibleFocus widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_has_visible_focus widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetHasVisibleFocusMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetHasVisibleFocusMethodInfo a signature where
    overloadedMethod = widgetHasVisibleFocus

instance O.OverloadedMethodInfo WidgetHasVisibleFocusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetHasVisibleFocus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetHasVisibleFocus"
        })


#endif

-- method Widget::hide
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_hide" gtk_widget_hide :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Reverses the effects of 'GI.Gtk.Objects.Widget.widgetShow', causing the widget to be
-- hidden (invisible to the user).
widgetHide ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m ()
widgetHide widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_hide widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetHideMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetHideMethodInfo a signature where
    overloadedMethod = widgetHide

instance O.OverloadedMethodInfo WidgetHideMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetHide",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetHide"
        })


#endif

-- method Widget::hide_on_delete
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_hide_on_delete" gtk_widget_hide_on_delete :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Utility function; intended to be connected to the [Widget::deleteEvent]("GI.Gtk.Objects.Widget#g:signal:deleteEvent")
-- signal on a t'GI.Gtk.Objects.Window.Window'. The function calls 'GI.Gtk.Objects.Widget.widgetHide' on its
-- argument, then returns 'P.True'. If connected to [deleteEvent](#g:signal:deleteEvent), the
-- result is that clicking the close button for a window (on the
-- window frame, top right corner usually) will hide but not destroy
-- the window. By default, GTK+ destroys windows when [deleteEvent](#g:signal:deleteEvent)
-- is received.
widgetHideOnDelete ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True'
widgetHideOnDelete widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_hide_on_delete widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetHideOnDeleteMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetHideOnDeleteMethodInfo a signature where
    overloadedMethod = widgetHideOnDelete

instance O.OverloadedMethodInfo WidgetHideOnDeleteMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetHideOnDelete",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetHideOnDelete"
        })


#endif

-- method Widget::in_destruction
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_in_destruction" gtk_widget_in_destruction :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Returns whether the widget is currently being destroyed.
-- This information can sometimes be used to avoid doing
-- unnecessary work.
widgetInDestruction ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@widget@/ is being destroyed
widgetInDestruction widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_in_destruction widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetInDestructionMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetInDestructionMethodInfo a signature where
    overloadedMethod = widgetInDestruction

instance O.OverloadedMethodInfo WidgetInDestructionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetInDestruction",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetInDestruction"
        })


#endif

-- method Widget::init_template
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_init_template" gtk_widget_init_template :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO ()

-- | Creates and initializes child widgets defined in templates. This
-- function must be called in the instance initializer for any
-- class which assigned itself a template using 'GI.Gtk.Structs.WidgetClass.widgetClassSetTemplate'
-- 
-- It is important to call this function in the instance initializer
-- of a t'GI.Gtk.Objects.Widget.Widget' subclass and not in t'GI.GObject.Objects.Object.Object'.@/constructed/@() or
-- t'GI.GObject.Objects.Object.Object'.@/constructor/@() for two reasons.
-- 
-- One reason is that generally derived widgets will assume that parent
-- class composite widgets have been created in their instance
-- initializers.
-- 
-- Another reason is that when calling @/g_object_new()/@ on a widget with
-- composite templates, it’s important to build the composite widgets
-- before the construct properties are set. Properties passed to @/g_object_new()/@
-- should take precedence over properties set in the private template XML.
-- 
-- /Since: 3.10/
widgetInitTemplate ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m ()
widgetInitTemplate widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    gtk_widget_init_template widget'
    touchManagedPtr widget
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetInitTemplateMethodInfo
instance (signature ~ (m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetInitTemplateMethodInfo a signature where
    overloadedMethod = widgetInitTemplate

instance O.OverloadedMethodInfo WidgetInitTemplateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetInitTemplate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetInitTemplate"
        })


#endif

-- method Widget::input_shape_combine_region
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "region"
--           , argType =
--               TInterface Name { namespace = "cairo" , name = "Region" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "shape to be added, or %NULL to remove an existing shape"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_input_shape_combine_region" gtk_widget_input_shape_combine_region :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Cairo.Region.Region ->              -- region : TInterface (Name {namespace = "cairo", name = "Region"})
    IO ()

-- | Sets an input shape for this widget’s GDK window. This allows for
-- windows which react to mouse click in a nonrectangular region, see
-- 'GI.Gdk.Objects.Window.windowInputShapeCombineRegion' for more information.
-- 
-- /Since: 3.0/
widgetInputShapeCombineRegion ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> Maybe (Cairo.Region.Region)
    -- ^ /@region@/: shape to be added, or 'P.Nothing' to remove an existing shape
    -> m ()
widgetInputShapeCombineRegion widget region = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    maybeRegion <- case region of
        Nothing -> return FP.nullPtr
        Just jRegion -> do
            jRegion' <- unsafeManagedPtrGetPtr jRegion
            return jRegion'
    gtk_widget_input_shape_combine_region widget' maybeRegion
    touchManagedPtr widget
    whenJust region touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetInputShapeCombineRegionMethodInfo
instance (signature ~ (Maybe (Cairo.Region.Region) -> m ()), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetInputShapeCombineRegionMethodInfo a signature where
    overloadedMethod = widgetInputShapeCombineRegion

instance O.OverloadedMethodInfo WidgetInputShapeCombineRegionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetInputShapeCombineRegion",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetInputShapeCombineRegion"
        })


#endif

-- method Widget::insert_action_group
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the prefix for actions in @group"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "group"
--           , argType =
--               TInterface Name { namespace = "Gio" , name = "ActionGroup" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GActionGroup, or %NULL"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_insert_action_group" gtk_widget_insert_action_group :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    CString ->                              -- name : TBasicType TUTF8
    Ptr Gio.ActionGroup.ActionGroup ->      -- group : TInterface (Name {namespace = "Gio", name = "ActionGroup"})
    IO ()

-- | Inserts /@group@/ into /@widget@/. Children of /@widget@/ that implement
-- t'GI.Gtk.Interfaces.Actionable.Actionable' can then be associated with actions in /@group@/ by
-- setting their “action-name” to
-- /@prefix@/.@action-name@.
-- 
-- If /@group@/ is 'P.Nothing', a previously inserted group for /@name@/ is removed
-- from /@widget@/.
-- 
-- /Since: 3.6/
widgetInsertActionGroup ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, Gio.ActionGroup.IsActionGroup b) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> T.Text
    -- ^ /@name@/: the prefix for actions in /@group@/
    -> Maybe (b)
    -- ^ /@group@/: a t'GI.Gio.Interfaces.ActionGroup.ActionGroup', or 'P.Nothing'
    -> m ()
widgetInsertActionGroup widget name group = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    name' <- textToCString name
    maybeGroup <- case group of
        Nothing -> return FP.nullPtr
        Just jGroup -> do
            jGroup' <- unsafeManagedPtrCastPtr jGroup
            return jGroup'
    gtk_widget_insert_action_group widget' name' maybeGroup
    touchManagedPtr widget
    whenJust group touchManagedPtr
    freeMem name'
    return ()

#if defined(ENABLE_OVERLOADING)
data WidgetInsertActionGroupMethodInfo
instance (signature ~ (T.Text -> Maybe (b) -> m ()), MonadIO m, IsWidget a, Gio.ActionGroup.IsActionGroup b) => O.OverloadedMethod WidgetInsertActionGroupMethodInfo a signature where
    overloadedMethod = widgetInsertActionGroup

instance O.OverloadedMethodInfo WidgetInsertActionGroupMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetInsertActionGroup",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetInsertActionGroup"
        })


#endif

-- method Widget::intersect
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "area"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Rectangle" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a rectangle" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "intersection"
--           , argType =
--               TInterface Name { namespace = "Gdk" , name = "Rectangle" }
--           , direction = DirectionOut
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "rectangle to store\n  intersection of @widget and @area"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_intersect" gtk_widget_intersect :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Gdk.Rectangle.Rectangle ->          -- area : TInterface (Name {namespace = "Gdk", name = "Rectangle"})
    Ptr Gdk.Rectangle.Rectangle ->          -- intersection : TInterface (Name {namespace = "Gdk", name = "Rectangle"})
    IO CInt

-- | Computes the intersection of a /@widget@/’s area and /@area@/, storing
-- the intersection in /@intersection@/, and returns 'P.True' if there was
-- an intersection.  /@intersection@/ may be 'P.Nothing' if you’re only
-- interested in whether there was an intersection.
widgetIntersect ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> Gdk.Rectangle.Rectangle
    -- ^ /@area@/: a rectangle
    -> m ((Bool, Maybe Gdk.Rectangle.Rectangle))
    -- ^ __Returns:__ 'P.True' if there was an intersection
widgetIntersect widget area = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    area' <- unsafeManagedPtrGetPtr area
    intersection <- SP.callocBoxedBytes 16 :: IO (Ptr Gdk.Rectangle.Rectangle)
    result <- gtk_widget_intersect widget' area' intersection
    let result' = (/= 0) result
    maybeIntersection <- convertIfNonNull intersection $ \intersection' -> do
        intersection'' <- (wrapBoxed Gdk.Rectangle.Rectangle) intersection'
        return intersection''
    touchManagedPtr widget
    touchManagedPtr area
    return (result', maybeIntersection)

#if defined(ENABLE_OVERLOADING)
data WidgetIntersectMethodInfo
instance (signature ~ (Gdk.Rectangle.Rectangle -> m ((Bool, Maybe Gdk.Rectangle.Rectangle))), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetIntersectMethodInfo a signature where
    overloadedMethod = widgetIntersect

instance O.OverloadedMethodInfo WidgetIntersectMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetIntersect",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetIntersect"
        })


#endif

-- method Widget::is_ancestor
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "ancestor"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "another #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_is_ancestor" gtk_widget_is_ancestor :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Ptr Widget ->                           -- ancestor : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether /@widget@/ is somewhere inside /@ancestor@/, possibly with
-- intermediate containers.
widgetIsAncestor ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a, IsWidget b) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> b
    -- ^ /@ancestor@/: another t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@ancestor@/ contains /@widget@/ as a child,
    --    grandchild, great grandchild, etc.
widgetIsAncestor widget ancestor = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    ancestor' <- unsafeManagedPtrCastPtr ancestor
    result <- gtk_widget_is_ancestor widget' ancestor'
    let result' = (/= 0) result
    touchManagedPtr widget
    touchManagedPtr ancestor
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetIsAncestorMethodInfo
instance (signature ~ (b -> m Bool), MonadIO m, IsWidget a, IsWidget b) => O.OverloadedMethod WidgetIsAncestorMethodInfo a signature where
    overloadedMethod = widgetIsAncestor

instance O.OverloadedMethodInfo WidgetIsAncestorMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetIsAncestor",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetIsAncestor"
        })


#endif

-- method Widget::is_composited
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_is_composited" gtk_widget_is_composited :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

{-# DEPRECATED widgetIsComposited ["(Since version 3.22)","Use 'GI.Gdk.Objects.Screen.screenIsComposited' instead."] #-}
-- | Whether /@widget@/ can rely on having its alpha channel
-- drawn correctly. On X11 this function returns whether a
-- compositing manager is running for /@widget@/’s screen.
-- 
-- Please note that the semantics of this call will change
-- in the future if used on a widget that has a composited
-- window in its hierarchy (as set by 'GI.Gdk.Objects.Window.windowSetComposited').
-- 
-- /Since: 2.10/
widgetIsComposited ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the widget can rely on its alpha
    -- channel being drawn correctly.
widgetIsComposited widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_is_composited widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetIsCompositedMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetIsCompositedMethodInfo a signature where
    overloadedMethod = widgetIsComposited

instance O.OverloadedMethodInfo WidgetIsCompositedMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetIsComposited",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetIsComposited"
        })


#endif

-- method Widget::is_drawable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_is_drawable" gtk_widget_is_drawable :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether /@widget@/ can be drawn to. A widget can be drawn
-- to if it is mapped and visible.
-- 
-- /Since: 2.18/
widgetIsDrawable ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@widget@/ is drawable, 'P.False' otherwise
widgetIsDrawable widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_is_drawable widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetIsDrawableMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetIsDrawableMethodInfo a signature where
    overloadedMethod = widgetIsDrawable

instance O.OverloadedMethodInfo WidgetIsDrawableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetIsDrawable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetIsDrawable"
        })


#endif

-- method Widget::is_focus
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_is_focus" gtk_widget_is_focus :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines if the widget is the focus widget within its
-- toplevel. (This does not mean that the [Widget:hasFocus]("GI.Gtk.Objects.Widget#g:attr:hasFocus") property is
-- necessarily set; [Widget:hasFocus]("GI.Gtk.Objects.Widget#g:attr:hasFocus") will only be set if the
-- toplevel widget additionally has the global input focus.)
widgetIsFocus ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the widget is the focus widget.
widgetIsFocus widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_is_focus widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetIsFocusMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetIsFocusMethodInfo a signature where
    overloadedMethod = widgetIsFocus

instance O.OverloadedMethodInfo WidgetIsFocusMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetIsFocus",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetIsFocus"
        })


#endif

-- method Widget::is_sensitive
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_is_sensitive" gtk_widget_is_sensitive :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Returns the widget’s effective sensitivity, which means
-- it is sensitive itself and also its parent widget is sensitive
-- 
-- /Since: 2.18/
widgetIsSensitive ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the widget is effectively sensitive
widgetIsSensitive widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_is_sensitive widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetIsSensitiveMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetIsSensitiveMethodInfo a signature where
    overloadedMethod = widgetIsSensitive

instance O.OverloadedMethodInfo WidgetIsSensitiveMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetIsSensitive",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetIsSensitive"
        })


#endif

-- method Widget::is_toplevel
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_widget_is_toplevel" gtk_widget_is_toplevel :: 
    Ptr Widget ->                           -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO CInt

-- | Determines whether /@widget@/ is a toplevel widget.
-- 
-- Currently only t'GI.Gtk.Objects.Window.Window' and t'GI.Gtk.Objects.Invisible.Invisible' (and out-of-process
-- @/GtkPlugs/@) are toplevel widgets. Toplevel widgets have no parent
-- widget.
-- 
-- /Since: 2.18/
widgetIsToplevel ::
    (B.CallStack.HasCallStack, MonadIO m, IsWidget a) =>
    a
    -- ^ /@widget@/: a t'GI.Gtk.Objects.Widget.Widget'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@widget@/ is a toplevel, 'P.False' otherwise
widgetIsToplevel widget = liftIO $ do
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_widget_is_toplevel widget'
    let result' = (/= 0) result
    touchManagedPtr widget
    return result'

#if defined(ENABLE_OVERLOADING)
data WidgetIsToplevelMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsWidget a) => O.OverloadedMethod WidgetIsToplevelMethodInfo a signature where
    overloadedMethod = widgetIsToplevel

instance O.OverloadedMethodInfo WidgetIsToplevelMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Widget.widgetIsToplevel",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-3.0.42/docs/GI-Gtk-Objects-Widget.html#v:widgetIsToplevel"
        })


#endif

-- method Widget::is_visible
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GtkWidget" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--      