{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Describes a 3D transform.
-- 
-- Unlike @graphene_matrix_t@, @GskTransform@ retains the steps in how
-- a transform was constructed, and allows inspecting them. It is modeled
-- after the way CSS describes transforms.
-- 
-- @GskTransform@ objects are immutable and cannot be changed after creation.
-- This means code can safely expose them as properties of objects without
-- having to worry about others changing them.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gsk.Structs.Transform
    ( 

-- * Exported types
    Transform(..)                           ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [equal]("GI.Gsk.Structs.Transform#g:method:equal"), [invert]("GI.Gsk.Structs.Transform#g:method:invert"), [matrix]("GI.Gsk.Structs.Transform#g:method:matrix"), [matrix2d]("GI.Gsk.Structs.Transform#g:method:matrix2d"), [perspective]("GI.Gsk.Structs.Transform#g:method:perspective"), [print]("GI.Gsk.Structs.Transform#g:method:print"), [ref]("GI.Gsk.Structs.Transform#g:method:ref"), [rotate]("GI.Gsk.Structs.Transform#g:method:rotate"), [rotate3d]("GI.Gsk.Structs.Transform#g:method:rotate3d"), [scale]("GI.Gsk.Structs.Transform#g:method:scale"), [scale3d]("GI.Gsk.Structs.Transform#g:method:scale3d"), [skew]("GI.Gsk.Structs.Transform#g:method:skew"), [to2d]("GI.Gsk.Structs.Transform#g:method:to2d"), [to2dComponents]("GI.Gsk.Structs.Transform#g:method:to2dComponents"), [toAffine]("GI.Gsk.Structs.Transform#g:method:toAffine"), [toMatrix]("GI.Gsk.Structs.Transform#g:method:toMatrix"), [toString]("GI.Gsk.Structs.Transform#g:method:toString"), [toTranslate]("GI.Gsk.Structs.Transform#g:method:toTranslate"), [transform]("GI.Gsk.Structs.Transform#g:method:transform"), [transformBounds]("GI.Gsk.Structs.Transform#g:method:transformBounds"), [transformPoint]("GI.Gsk.Structs.Transform#g:method:transformPoint"), [translate]("GI.Gsk.Structs.Transform#g:method:translate"), [translate3d]("GI.Gsk.Structs.Transform#g:method:translate3d"), [unref]("GI.Gsk.Structs.Transform#g:method:unref").
-- 
-- ==== Getters
-- [getCategory]("GI.Gsk.Structs.Transform#g:method:getCategory").
-- 
-- ==== Setters
-- /None/.

#if defined(ENABLE_OVERLOADING)
    ResolveTransformMethod                  ,
#endif

-- ** equal #method:equal#

#if defined(ENABLE_OVERLOADING)
    TransformEqualMethodInfo                ,
#endif
    transformEqual                          ,


-- ** getCategory #method:getCategory#

#if defined(ENABLE_OVERLOADING)
    TransformGetCategoryMethodInfo          ,
#endif
    transformGetCategory                    ,


-- ** invert #method:invert#

#if defined(ENABLE_OVERLOADING)
    TransformInvertMethodInfo               ,
#endif
    transformInvert                         ,


-- ** matrix #method:matrix#

#if defined(ENABLE_OVERLOADING)
    TransformMatrixMethodInfo               ,
#endif
    transformMatrix                         ,


-- ** matrix2d #method:matrix2d#

#if defined(ENABLE_OVERLOADING)
    TransformMatrix2dMethodInfo             ,
#endif
    transformMatrix2d                       ,


-- ** new #method:new#

    transformNew                            ,


-- ** parse #method:parse#

    transformParse                          ,


-- ** perspective #method:perspective#

#if defined(ENABLE_OVERLOADING)
    TransformPerspectiveMethodInfo          ,
#endif
    transformPerspective                    ,


-- ** print #method:print#

#if defined(ENABLE_OVERLOADING)
    TransformPrintMethodInfo                ,
#endif
    transformPrint                          ,


-- ** ref #method:ref#

#if defined(ENABLE_OVERLOADING)
    TransformRefMethodInfo                  ,
#endif
    transformRef                            ,


-- ** rotate #method:rotate#

#if defined(ENABLE_OVERLOADING)
    TransformRotateMethodInfo               ,
#endif
    transformRotate                         ,


-- ** rotate3d #method:rotate3d#

#if defined(ENABLE_OVERLOADING)
    TransformRotate3dMethodInfo             ,
#endif
    transformRotate3d                       ,


-- ** scale #method:scale#

#if defined(ENABLE_OVERLOADING)
    TransformScaleMethodInfo                ,
#endif
    transformScale                          ,


-- ** scale3d #method:scale3d#

#if defined(ENABLE_OVERLOADING)
    TransformScale3dMethodInfo              ,
#endif
    transformScale3d                        ,


-- ** skew #method:skew#

#if defined(ENABLE_OVERLOADING)
    TransformSkewMethodInfo                 ,
#endif
    transformSkew                           ,


-- ** to2d #method:to2d#

#if defined(ENABLE_OVERLOADING)
    TransformTo2dMethodInfo                 ,
#endif
    transformTo2d                           ,


-- ** to2dComponents #method:to2dComponents#

#if defined(ENABLE_OVERLOADING)
    TransformTo2dComponentsMethodInfo       ,
#endif
    transformTo2dComponents                 ,


-- ** toAffine #method:toAffine#

#if defined(ENABLE_OVERLOADING)
    TransformToAffineMethodInfo             ,
#endif
    transformToAffine                       ,


-- ** toMatrix #method:toMatrix#

#if defined(ENABLE_OVERLOADING)
    TransformToMatrixMethodInfo             ,
#endif
    transformToMatrix                       ,


-- ** toString #method:toString#

#if defined(ENABLE_OVERLOADING)
    TransformToStringMethodInfo             ,
#endif
    transformToString                       ,


-- ** toTranslate #method:toTranslate#

#if defined(ENABLE_OVERLOADING)
    TransformToTranslateMethodInfo          ,
#endif
    transformToTranslate                    ,


-- ** transform #method:transform#

#if defined(ENABLE_OVERLOADING)
    TransformTransformMethodInfo            ,
#endif
    transformTransform                      ,


-- ** transformBounds #method:transformBounds#

#if defined(ENABLE_OVERLOADING)
    TransformTransformBoundsMethodInfo      ,
#endif
    transformTransformBounds                ,


-- ** transformPoint #method:transformPoint#

#if defined(ENABLE_OVERLOADING)
    TransformTransformPointMethodInfo       ,
#endif
    transformTransformPoint                 ,


-- ** translate #method:translate#

#if defined(ENABLE_OVERLOADING)
    TransformTranslateMethodInfo            ,
#endif
    transformTranslate                      ,


-- ** translate3d #method:translate3d#

#if defined(ENABLE_OVERLOADING)
    TransformTranslate3dMethodInfo          ,
#endif
    transformTranslate3d                    ,


-- ** unref #method:unref#

#if defined(ENABLE_OVERLOADING)
    TransformUnrefMethodInfo                ,
#endif
    transformUnref                          ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Structs.String as GLib.String
import qualified GI.Graphene.Structs.Matrix as Graphene.Matrix
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Point3D as Graphene.Point3D
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Graphene.Structs.Vec3 as Graphene.Vec3
import {-# SOURCE #-} qualified GI.Gsk.Enums as Gsk.Enums

#else
import qualified GI.GLib.Structs.String as GLib.String
import qualified GI.Graphene.Structs.Matrix as Graphene.Matrix
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Point3D as Graphene.Point3D
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Graphene.Structs.Vec3 as Graphene.Vec3
import {-# SOURCE #-} qualified GI.Gsk.Enums as Gsk.Enums

#endif

-- | Memory-managed wrapper type.
newtype Transform = Transform (SP.ManagedPtr Transform)
    deriving (Eq)

instance SP.ManagedPtrNewtype Transform where
    toManagedPtr (Transform p) = p

foreign import ccall "gsk_transform_get_type" c_gsk_transform_get_type :: 
    IO GType

type instance O.ParentTypes Transform = '[]
instance O.HasParentTypes Transform

instance B.Types.TypedObject Transform where
    glibType = c_gsk_transform_get_type

instance B.Types.GBoxed Transform

-- | Convert t'Transform' to and from 'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Transform) where
    gvalueGType_ = c_gsk_transform_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_boxed gv (FP.nullPtr :: FP.Ptr Transform)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_boxed gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_boxed gv :: IO (Ptr Transform)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newBoxed Transform ptr
        else return P.Nothing
        
    


#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Transform
type instance O.AttributeList Transform = TransformAttributeList
type TransformAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

-- method Transform::new
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "Transform" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_new" gsk_transform_new :: 
    IO (Ptr Transform)

-- | Creates a new identity transform.
-- 
-- This function is meant to be used by language
-- bindings. For C code, this is equivalent to using @NULL@.
transformNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Transform
    -- ^ __Returns:__ A new identity transform
transformNew  = liftIO $ do
    result <- gsk_transform_new
    checkUnexpectedReturnNULL "transformNew" result
    result' <- (wrapBoxed Transform) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Transform::equal
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "first"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the first transform"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "second"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the second transform"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_equal" gsk_transform_equal :: 
    Ptr Transform ->                        -- first : TInterface (Name {namespace = "Gsk", name = "Transform"})
    Ptr Transform ->                        -- second : TInterface (Name {namespace = "Gsk", name = "Transform"})
    IO CInt

-- | Checks two transforms for equality.
transformEqual ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@first@/: the first transform
    -> Maybe (Transform)
    -- ^ /@second@/: the second transform
    -> m Bool
    -- ^ __Returns:__ true if the two transforms perform the same operation
transformEqual first second = liftIO $ do
    maybeFirst <- case first of
        Nothing -> return FP.nullPtr
        Just jFirst -> do
            jFirst' <- unsafeManagedPtrGetPtr jFirst
            return jFirst'
    maybeSecond <- case second of
        Nothing -> return FP.nullPtr
        Just jSecond -> do
            jSecond' <- unsafeManagedPtrGetPtr jSecond
            return jSecond'
    result <- gsk_transform_equal maybeFirst maybeSecond
    let result' = (/= 0) result
    whenJust first touchManagedPtr
    whenJust second touchManagedPtr
    return result'

#if defined(ENABLE_OVERLOADING)
data TransformEqualMethodInfo
instance (signature ~ (Maybe (Transform) -> m Bool), MonadIO m) => O.OverloadedMethod TransformEqualMethodInfo Transform signature where
    overloadedMethod i = transformEqual (Just i)

instance O.OverloadedMethodInfo TransformEqualMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformEqual",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformEqual"
        })


#endif

-- method Transform::get_category
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface
--                  Name { namespace = "Gsk" , name = "TransformCategory" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_get_category" gsk_transform_get_category :: 
    Ptr Transform ->                        -- self : TInterface (Name {namespace = "Gsk", name = "Transform"})
    IO CUInt

-- | Returns the category this transform belongs to.
transformGetCategory ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@self@/: a transform
    -> m Gsk.Enums.TransformCategory
    -- ^ __Returns:__ The category of the transform
transformGetCategory self = liftIO $ do
    maybeSelf <- case self of
        Nothing -> return FP.nullPtr
        Just jSelf -> do
            jSelf' <- unsafeManagedPtrGetPtr jSelf
            return jSelf'
    result <- gsk_transform_get_category maybeSelf
    let result' = (toEnum . fromIntegral) result
    whenJust self touchManagedPtr
    return result'

#if defined(ENABLE_OVERLOADING)
data TransformGetCategoryMethodInfo
instance (signature ~ (m Gsk.Enums.TransformCategory), MonadIO m) => O.OverloadedMethod TransformGetCategoryMethodInfo Transform signature where
    overloadedMethod i = transformGetCategory (Just i)

instance O.OverloadedMethodInfo TransformGetCategoryMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformGetCategory",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformGetCategory"
        })


#endif

-- method Transform::invert
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "transform to invert"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "Transform" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_invert" gsk_transform_invert :: 
    Ptr Transform ->                        -- self : TInterface (Name {namespace = "Gsk", name = "Transform"})
    IO (Ptr Transform)

-- | Inverts the given transform.
-- 
-- If /@self@/ is not invertible, @NULL@ is returned.
-- Note that inverting @NULL@ also returns @NULL@, which is
-- the correct inverse of @NULL@. If you need to differentiate
-- between those cases, you should check /@self@/ is not @NULL@
-- before calling this function.
-- 
-- This function consumes /@self@/. Use 'GI.Gsk.Structs.Transform.transformRef' first
-- if you want to keep it around.
transformInvert ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@self@/: transform to invert
    -> m (Maybe Transform)
    -- ^ __Returns:__ The inverted transform
transformInvert self = liftIO $ do
    maybeSelf <- case self of
        Nothing -> return FP.nullPtr
        Just jSelf -> do
            jSelf' <- B.ManagedPtr.disownBoxed jSelf
            return jSelf'
    result <- gsk_transform_invert maybeSelf
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed Transform) result'
        return result''
    whenJust self touchManagedPtr
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data TransformInvertMethodInfo
instance (signature ~ (m (Maybe Transform)), MonadIO m) => O.OverloadedMethod TransformInvertMethodInfo Transform signature where
    overloadedMethod i = transformInvert (Just i)

instance O.OverloadedMethodInfo TransformInvertMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformInvert",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformInvert"
        })


#endif

-- method Transform::matrix
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "next"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the next transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "matrix"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Matrix" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the matrix to multiply @next with"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "Transform" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_matrix" gsk_transform_matrix :: 
    Ptr Transform ->                        -- next : TInterface (Name {namespace = "Gsk", name = "Transform"})
    Ptr Graphene.Matrix.Matrix ->           -- matrix : TInterface (Name {namespace = "Graphene", name = "Matrix"})
    IO (Ptr Transform)

-- | Multiplies /@next@/ with the given /@matrix@/.
-- 
-- This function consumes /@next@/. Use 'GI.Gsk.Structs.Transform.transformRef' first
-- if you want to keep it around.
transformMatrix ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@next@/: the next transform
    -> Graphene.Matrix.Matrix
    -- ^ /@matrix@/: the matrix to multiply /@next@/ with
    -> m Transform
    -- ^ __Returns:__ The new transform
transformMatrix next matrix = liftIO $ do
    maybeNext <- case next of
        Nothing -> return FP.nullPtr
        Just jNext -> do
            jNext' <- B.ManagedPtr.disownBoxed jNext
            return jNext'
    matrix' <- unsafeManagedPtrGetPtr matrix
    result <- gsk_transform_matrix maybeNext matrix'
    checkUnexpectedReturnNULL "transformMatrix" result
    result' <- (wrapBoxed Transform) result
    whenJust next touchManagedPtr
    touchManagedPtr matrix
    return result'

#if defined(ENABLE_OVERLOADING)
data TransformMatrixMethodInfo
instance (signature ~ (Graphene.Matrix.Matrix -> m Transform), MonadIO m) => O.OverloadedMethod TransformMatrixMethodInfo Transform signature where
    overloadedMethod i = transformMatrix (Just i)

instance O.OverloadedMethodInfo TransformMatrixMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformMatrix",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformMatrix"
        })


#endif

-- method Transform::matrix_2d
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "next"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the next transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "xx"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the xx member" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "yx"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the yx member" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "xy"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the xy member" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "yy"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the yy member" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "dx"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the x0 member" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "dy"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the y0 member" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "Transform" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_matrix_2d" gsk_transform_matrix_2d :: 
    Ptr Transform ->                        -- next : TInterface (Name {namespace = "Gsk", name = "Transform"})
    CFloat ->                               -- xx : TBasicType TFloat
    CFloat ->                               -- yx : TBasicType TFloat
    CFloat ->                               -- xy : TBasicType TFloat
    CFloat ->                               -- yy : TBasicType TFloat
    CFloat ->                               -- dx : TBasicType TFloat
    CFloat ->                               -- dy : TBasicType TFloat
    IO (Ptr Transform)

-- | Multiplies /@next@/ with the matrix [ xx yx x0; xy yy y0; 0 0 1 ].
-- 
-- The result of calling 'GI.Gsk.Structs.Transform.transformTo2d' on the returned
-- t'GI.Gsk.Structs.Transform.Transform' should match the input passed to this
-- function.
-- 
-- This function consumes /@next@/. Use 'GI.Gsk.Structs.Transform.transformRef' first
-- if you want to keep it around.
-- 
-- /Since: 4.20/
transformMatrix2d ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@next@/: the next transform
    -> Float
    -- ^ /@xx@/: the xx member
    -> Float
    -- ^ /@yx@/: the yx member
    -> Float
    -- ^ /@xy@/: the xy member
    -> Float
    -- ^ /@yy@/: the yy member
    -> Float
    -- ^ /@dx@/: the x0 member
    -> Float
    -- ^ /@dy@/: the y0 member
    -> m (Maybe Transform)
    -- ^ __Returns:__ The new transform
transformMatrix2d next xx yx xy yy dx dy = liftIO $ do
    maybeNext <- case next of
        Nothing -> return FP.nullPtr
        Just jNext -> do
            jNext' <- B.ManagedPtr.disownBoxed jNext
            return jNext'
    let xx' = realToFrac xx
    let yx' = realToFrac yx
    let xy' = realToFrac xy
    let yy' = realToFrac yy
    let dx' = realToFrac dx
    let dy' = realToFrac dy
    result <- gsk_transform_matrix_2d maybeNext xx' yx' xy' yy' dx' dy'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed Transform) result'
        return result''
    whenJust next touchManagedPtr
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data TransformMatrix2dMethodInfo
instance (signature ~ (Float -> Float -> Float -> Float -> Float -> Float -> m (Maybe Transform)), MonadIO m) => O.OverloadedMethod TransformMatrix2dMethodInfo Transform signature where
    overloadedMethod i = transformMatrix2d (Just i)

instance O.OverloadedMethodInfo TransformMatrix2dMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformMatrix2d",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformMatrix2d"
        })


#endif

-- method Transform::perspective
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "next"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the next transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "depth"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "distance of the z=0 plane. Lower values give a more\n  flattened pyramid and therefore a more pronounced\n  perspective effect."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "Transform" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_perspective" gsk_transform_perspective :: 
    Ptr Transform ->                        -- next : TInterface (Name {namespace = "Gsk", name = "Transform"})
    CFloat ->                               -- depth : TBasicType TFloat
    IO (Ptr Transform)

-- | Applies a perspective projection transform.
-- 
-- This transform scales points in X and Y based on their Z value,
-- scaling points with positive Z values away from the origin, and
-- those with negative Z values towards the origin. Points
-- on the z=0 plane are unchanged.
-- 
-- This function consumes /@next@/. Use 'GI.Gsk.Structs.Transform.transformRef' first
-- if you want to keep it around.
transformPerspective ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@next@/: the next transform
    -> Float
    -- ^ /@depth@/: distance of the z=0 plane. Lower values give a more
    --   flattened pyramid and therefore a more pronounced
    --   perspective effect.
    -> m Transform
    -- ^ __Returns:__ The new transform
transformPerspective next depth = liftIO $ do
    maybeNext <- case next of
        Nothing -> return FP.nullPtr
        Just jNext -> do
            jNext' <- B.ManagedPtr.disownBoxed jNext
            return jNext'
    let depth' = realToFrac depth
    result <- gsk_transform_perspective maybeNext depth'
    checkUnexpectedReturnNULL "transformPerspective" result
    result' <- (wrapBoxed Transform) result
    whenJust next touchManagedPtr
    return result'

#if defined(ENABLE_OVERLOADING)
data TransformPerspectiveMethodInfo
instance (signature ~ (Float -> m Transform), MonadIO m) => O.OverloadedMethod TransformPerspectiveMethodInfo Transform signature where
    overloadedMethod i = transformPerspective (Just i)

instance O.OverloadedMethodInfo TransformPerspectiveMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformPerspective",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformPerspective"
        })


#endif

-- method Transform::print
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "string"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "String" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "The string to print into"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_print" gsk_transform_print :: 
    Ptr Transform ->                        -- self : TInterface (Name {namespace = "Gsk", name = "Transform"})
    Ptr GLib.String.String ->               -- string : TInterface (Name {namespace = "GLib", name = "String"})
    IO ()

-- | Converts the transform into a human-readable representation.
-- 
-- The result of this function can later be parsed with
-- 'GI.Gsk.Structs.Transform.transformParse'.
transformPrint ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@self@/: a transform
    -> GLib.String.String
    -- ^ /@string@/: The string to print into
    -> m ()
transformPrint self string = liftIO $ do
    maybeSelf <- case self of
        Nothing -> return FP.nullPtr
        Just jSelf -> do
            jSelf' <- unsafeManagedPtrGetPtr jSelf
            return jSelf'
    string' <- unsafeManagedPtrGetPtr string
    gsk_transform_print maybeSelf string'
    whenJust self touchManagedPtr
    touchManagedPtr string
    return ()

#if defined(ENABLE_OVERLOADING)
data TransformPrintMethodInfo
instance (signature ~ (GLib.String.String -> m ()), MonadIO m) => O.OverloadedMethod TransformPrintMethodInfo Transform signature where
    overloadedMethod i = transformPrint (Just i)

instance O.OverloadedMethodInfo TransformPrintMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformPrint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformPrint"
        })


#endif

-- method Transform::ref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "Transform" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_ref" gsk_transform_ref :: 
    Ptr Transform ->                        -- self : TInterface (Name {namespace = "Gsk", name = "Transform"})
    IO (Ptr Transform)

-- | Acquires a reference on the given transform.
transformRef ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@self@/: a transform
    -> m (Maybe Transform)
    -- ^ __Returns:__ the transform with an additional reference
transformRef self = liftIO $ do
    maybeSelf <- case self of
        Nothing -> return FP.nullPtr
        Just jSelf -> do
            jSelf' <- unsafeManagedPtrGetPtr jSelf
            return jSelf'
    result <- gsk_transform_ref maybeSelf
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newBoxed Transform) result'
        return result''
    whenJust self touchManagedPtr
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data TransformRefMethodInfo
instance (signature ~ (m (Maybe Transform)), MonadIO m) => O.OverloadedMethod TransformRefMethodInfo Transform signature where
    overloadedMethod i = transformRef (Just i)

instance O.OverloadedMethodInfo TransformRefMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformRef",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformRef"
        })


#endif

-- method Transform::rotate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "next"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the next transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "angle"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the rotation angle, in degrees (clockwise)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "Transform" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_rotate" gsk_transform_rotate :: 
    Ptr Transform ->                        -- next : TInterface (Name {namespace = "Gsk", name = "Transform"})
    CFloat ->                               -- angle : TBasicType TFloat
    IO (Ptr Transform)

-- | Rotates /@next@/ by an angle around the Z axis.
-- 
-- The rotation happens around the origin point of (0, 0).
-- 
-- This function consumes /@next@/. Use 'GI.Gsk.Structs.Transform.transformRef' first
-- if you want to keep it around.
transformRotate ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@next@/: the next transform
    -> Float
    -- ^ /@angle@/: the rotation angle, in degrees (clockwise)
    -> m (Maybe Transform)
    -- ^ __Returns:__ The new transform
transformRotate next angle = liftIO $ do
    maybeNext <- case next of
        Nothing -> return FP.nullPtr
        Just jNext -> do
            jNext' <- B.ManagedPtr.disownBoxed jNext
            return jNext'
    let angle' = realToFrac angle
    result <- gsk_transform_rotate maybeNext angle'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed Transform) result'
        return result''
    whenJust next touchManagedPtr
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data TransformRotateMethodInfo
instance (signature ~ (Float -> m (Maybe Transform)), MonadIO m) => O.OverloadedMethod TransformRotateMethodInfo Transform signature where
    overloadedMethod i = transformRotate (Just i)

instance O.OverloadedMethodInfo TransformRotateMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformRotate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformRotate"
        })


#endif

-- method Transform::rotate_3d
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "next"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the next transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "angle"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the rotation angle, in degrees (clockwise)"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "axis"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Vec3" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "The rotation axis" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "Transform" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_rotate_3d" gsk_transform_rotate_3d :: 
    Ptr Transform ->                        -- next : TInterface (Name {namespace = "Gsk", name = "Transform"})
    CFloat ->                               -- angle : TBasicType TFloat
    Ptr Graphene.Vec3.Vec3 ->               -- axis : TInterface (Name {namespace = "Graphene", name = "Vec3"})
    IO (Ptr Transform)

-- | Rotates /@next@/ /@angle@/ degrees around /@axis@/.
-- 
-- For a rotation in 2D space, use 'GI.Gsk.Structs.Transform.transformRotate'
-- 
-- This function consumes /@next@/. Use 'GI.Gsk.Structs.Transform.transformRef' first
-- if you want to keep it around.
transformRotate3d ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@next@/: the next transform
    -> Float
    -- ^ /@angle@/: the rotation angle, in degrees (clockwise)
    -> Graphene.Vec3.Vec3
    -- ^ /@axis@/: The rotation axis
    -> m (Maybe Transform)
    -- ^ __Returns:__ The new transform
transformRotate3d next angle axis = liftIO $ do
    maybeNext <- case next of
        Nothing -> return FP.nullPtr
        Just jNext -> do
            jNext' <- B.ManagedPtr.disownBoxed jNext
            return jNext'
    let angle' = realToFrac angle
    axis' <- unsafeManagedPtrGetPtr axis
    result <- gsk_transform_rotate_3d maybeNext angle' axis'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed Transform) result'
        return result''
    whenJust next touchManagedPtr
    touchManagedPtr axis
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data TransformRotate3dMethodInfo
instance (signature ~ (Float -> Graphene.Vec3.Vec3 -> m (Maybe Transform)), MonadIO m) => O.OverloadedMethod TransformRotate3dMethodInfo Transform signature where
    overloadedMethod i = transformRotate3d (Just i)

instance O.OverloadedMethodInfo TransformRotate3dMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformRotate3d",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformRotate3d"
        })


#endif

-- method Transform::scale
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "next"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the next transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "factor_x"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "scaling factor on the X axis"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "factor_y"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "scaling factor on the Y axis"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "Transform" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_scale" gsk_transform_scale :: 
    Ptr Transform ->                        -- next : TInterface (Name {namespace = "Gsk", name = "Transform"})
    CFloat ->                               -- factor_x : TBasicType TFloat
    CFloat ->                               -- factor_y : TBasicType TFloat
    IO (Ptr Transform)

-- | Scales /@next@/ in 2-dimensional space by the given factors.
-- 
-- Use 'GI.Gsk.Structs.Transform.transformScale3d' to scale in all 3 dimensions.
-- 
-- This function consumes /@next@/. Use 'GI.Gsk.Structs.Transform.transformRef' first
-- if you want to keep it around.
transformScale ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@next@/: the next transform
    -> Float
    -- ^ /@factorX@/: scaling factor on the X axis
    -> Float
    -- ^ /@factorY@/: scaling factor on the Y axis
    -> m (Maybe Transform)
    -- ^ __Returns:__ The new transform
transformScale next factorX factorY = liftIO $ do
    maybeNext <- case next of
        Nothing -> return FP.nullPtr
        Just jNext -> do
            jNext' <- B.ManagedPtr.disownBoxed jNext
            return jNext'
    let factorX' = realToFrac factorX
    let factorY' = realToFrac factorY
    result <- gsk_transform_scale maybeNext factorX' factorY'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed Transform) result'
        return result''
    whenJust next touchManagedPtr
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data TransformScaleMethodInfo
instance (signature ~ (Float -> Float -> m (Maybe Transform)), MonadIO m) => O.OverloadedMethod TransformScaleMethodInfo Transform signature where
    overloadedMethod i = transformScale (Just i)

instance O.OverloadedMethodInfo TransformScaleMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformScale",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformScale"
        })


#endif

-- method Transform::scale_3d
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "next"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the next transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "factor_x"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "scaling factor on the X axis"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "factor_y"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "scaling factor on the Y axis"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "factor_z"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "scaling factor on the Z axis"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "Transform" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_scale_3d" gsk_transform_scale_3d :: 
    Ptr Transform ->                        -- next : TInterface (Name {namespace = "Gsk", name = "Transform"})
    CFloat ->                               -- factor_x : TBasicType TFloat
    CFloat ->                               -- factor_y : TBasicType TFloat
    CFloat ->                               -- factor_z : TBasicType TFloat
    IO (Ptr Transform)

-- | Scales /@next@/ by the given factors.
-- 
-- This function consumes /@next@/. Use 'GI.Gsk.Structs.Transform.transformRef' first
-- if you want to keep it around.
transformScale3d ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@next@/: the next transform
    -> Float
    -- ^ /@factorX@/: scaling factor on the X axis
    -> Float
    -- ^ /@factorY@/: scaling factor on the Y axis
    -> Float
    -- ^ /@factorZ@/: scaling factor on the Z axis
    -> m (Maybe Transform)
    -- ^ __Returns:__ The new transform
transformScale3d next factorX factorY factorZ = liftIO $ do
    maybeNext <- case next of
        Nothing -> return FP.nullPtr
        Just jNext -> do
            jNext' <- B.ManagedPtr.disownBoxed jNext
            return jNext'
    let factorX' = realToFrac factorX
    let factorY' = realToFrac factorY
    let factorZ' = realToFrac factorZ
    result <- gsk_transform_scale_3d maybeNext factorX' factorY' factorZ'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed Transform) result'
        return result''
    whenJust next touchManagedPtr
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data TransformScale3dMethodInfo
instance (signature ~ (Float -> Float -> Float -> m (Maybe Transform)), MonadIO m) => O.OverloadedMethod TransformScale3dMethodInfo Transform signature where
    overloadedMethod i = transformScale3d (Just i)

instance O.OverloadedMethodInfo TransformScale3dMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformScale3d",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformScale3d"
        })


#endif

-- method Transform::skew
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "next"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the next transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "skew_x"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "skew factor, in degrees, on the X axis"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "skew_y"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "skew factor, in degrees, on the Y axis"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "Transform" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_skew" gsk_transform_skew :: 
    Ptr Transform ->                        -- next : TInterface (Name {namespace = "Gsk", name = "Transform"})
    CFloat ->                               -- skew_x : TBasicType TFloat
    CFloat ->                               -- skew_y : TBasicType TFloat
    IO (Ptr Transform)

-- | Applies a skew transform.
-- 
-- This function consumes /@next@/. Use 'GI.Gsk.Structs.Transform.transformRef' first
-- if you want to keep it around.
-- 
-- /Since: 4.6/
transformSkew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@next@/: the next transform
    -> Float
    -- ^ /@skewX@/: skew factor, in degrees, on the X axis
    -> Float
    -- ^ /@skewY@/: skew factor, in degrees, on the Y axis
    -> m (Maybe Transform)
    -- ^ __Returns:__ The new transform
transformSkew next skewX skewY = liftIO $ do
    maybeNext <- case next of
        Nothing -> return FP.nullPtr
        Just jNext -> do
            jNext' <- B.ManagedPtr.disownBoxed jNext
            return jNext'
    let skewX' = realToFrac skewX
    let skewY' = realToFrac skewY
    result <- gsk_transform_skew maybeNext skewX' skewY'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed Transform) result'
        return result''
    whenJust next touchManagedPtr
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data TransformSkewMethodInfo
instance (signature ~ (Float -> Float -> m (Maybe Transform)), MonadIO m) => O.OverloadedMethod TransformSkewMethodInfo Transform signature where
    overloadedMethod i = transformSkew (Just i)

instance O.OverloadedMethodInfo TransformSkewMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformSkew",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformSkew"
        })


#endif

-- method Transform::to_2d
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a 2D transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_xx"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for the xx member"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "out_yx"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for the yx member"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "out_xy"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for the xy member"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "out_yy"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for the yy member"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "out_dx"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for the x0 member"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "out_dy"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for the y0 member"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_to_2d" gsk_transform_to_2d :: 
    Ptr Transform ->                        -- self : TInterface (Name {namespace = "Gsk", name = "Transform"})
    Ptr CFloat ->                           -- out_xx : TBasicType TFloat
    Ptr CFloat ->                           -- out_yx : TBasicType TFloat
    Ptr CFloat ->                           -- out_xy : TBasicType TFloat
    Ptr CFloat ->                           -- out_yy : TBasicType TFloat
    Ptr CFloat ->                           -- out_dx : TBasicType TFloat
    Ptr CFloat ->                           -- out_dy : TBasicType TFloat
    IO ()

-- | Converts a transform to a 2D transformation matrix.
-- 
-- /@self@/ must be a 2D transformation. If you are not
-- sure, use
-- 
--     'GI.Gsk.Structs.Transform.transformGetCategory' >= GSK_TRANSFORM_CATEGORY_2D
-- 
-- to check.
-- 
-- The returned values are a subset of the full 4x4 matrix that
-- is computed by 'GI.Gsk.Structs.Transform.transformToMatrix' and have the
-- following layout:
-- 
-- 
-- 	
-- >  | xx yx |   |  a  b  0 |
-- >  | xy yy | = |  c  d  0 |
-- >  | dx dy |   | tx ty  1 |
-- 
-- 
-- This function can be used to convert between a @GskTransform@
-- and a matrix type from other 2D drawing libraries, in particular
-- Cairo.
transformTo2d ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Transform
    -- ^ /@self@/: a 2D transform
    -> m ((Float, Float, Float, Float, Float, Float))
transformTo2d self = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    outXx <- allocMem :: IO (Ptr CFloat)
    outYx <- allocMem :: IO (Ptr CFloat)
    outXy <- allocMem :: IO (Ptr CFloat)
    outYy <- allocMem :: IO (Ptr CFloat)
    outDx <- allocMem :: IO (Ptr CFloat)
    outDy <- allocMem :: IO (Ptr CFloat)
    gsk_transform_to_2d self' outXx outYx outXy outYy outDx outDy
    outXx' <- peek outXx
    let outXx'' = realToFrac outXx'
    outYx' <- peek outYx
    let outYx'' = realToFrac outYx'
    outXy' <- peek outXy
    let outXy'' = realToFrac outXy'
    outYy' <- peek outYy
    let outYy'' = realToFrac outYy'
    outDx' <- peek outDx
    let outDx'' = realToFrac outDx'
    outDy' <- peek outDy
    let outDy'' = realToFrac outDy'
    touchManagedPtr self
    freeMem outXx
    freeMem outYx
    freeMem outXy
    freeMem outYy
    freeMem outDx
    freeMem outDy
    return (outXx'', outYx'', outXy'', outYy'', outDx'', outDy'')

#if defined(ENABLE_OVERLOADING)
data TransformTo2dMethodInfo
instance (signature ~ (m ((Float, Float, Float, Float, Float, Float))), MonadIO m) => O.OverloadedMethod TransformTo2dMethodInfo Transform signature where
    overloadedMethod = transformTo2d

instance O.OverloadedMethodInfo TransformTo2dMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformTo2d",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformTo2d"
        })


#endif

-- method Transform::to_2d_components
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_skew_x"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the skew factor\n  in the  x direction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "out_skew_y"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the skew factor\n  in the  y direction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "out_scale_x"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the scale\n  factor in the x direction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "out_scale_y"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the scale\n  factor in the y direction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "out_angle"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for the rotation angle"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "out_dx"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the translation\n  in the x direction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "out_dy"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the translation\n  in the y direction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_to_2d_components" gsk_transform_to_2d_components :: 
    Ptr Transform ->                        -- self : TInterface (Name {namespace = "Gsk", name = "Transform"})
    Ptr CFloat ->                           -- out_skew_x : TBasicType TFloat
    Ptr CFloat ->                           -- out_skew_y : TBasicType TFloat
    Ptr CFloat ->                           -- out_scale_x : TBasicType TFloat
    Ptr CFloat ->                           -- out_scale_y : TBasicType TFloat
    Ptr CFloat ->                           -- out_angle : TBasicType TFloat
    Ptr CFloat ->                           -- out_dx : TBasicType TFloat
    Ptr CFloat ->                           -- out_dy : TBasicType TFloat
    IO ()

-- | Converts a transform to 2D transformation factors.
-- 
-- To recreate an equivalent transform from the factors returned
-- by this function, use
-- 
--     gsk_transform_skew (
--         gsk_transform_scale (
--             gsk_transform_rotate (
--                 gsk_transform_translate (NULL, &GRAPHENE_POINT_INIT (dx, dy)),
--                 angle),
--             scale_x, scale_y),
--         skew_x, skew_y)
-- 
-- /@self@/ must be a 2D transformation. If you are not sure, use
-- 
--     'GI.Gsk.Structs.Transform.transformGetCategory' >= GSK_TRANSFORM_CATEGORY_2D
-- 
-- to check.
-- 
-- /Since: 4.6/
transformTo2dComponents ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Transform
    -- ^ /@self@/: a transform
    -> m ((Float, Float, Float, Float, Float, Float, Float))
transformTo2dComponents self = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    outSkewX <- allocMem :: IO (Ptr CFloat)
    outSkewY <- allocMem :: IO (Ptr CFloat)
    outScaleX <- allocMem :: IO (Ptr CFloat)
    outScaleY <- allocMem :: IO (Ptr CFloat)
    outAngle <- allocMem :: IO (Ptr CFloat)
    outDx <- allocMem :: IO (Ptr CFloat)
    outDy <- allocMem :: IO (Ptr CFloat)
    gsk_transform_to_2d_components self' outSkewX outSkewY outScaleX outScaleY outAngle outDx outDy
    outSkewX' <- peek outSkewX
    let outSkewX'' = realToFrac outSkewX'
    outSkewY' <- peek outSkewY
    let outSkewY'' = realToFrac outSkewY'
    outScaleX' <- peek outScaleX
    let outScaleX'' = realToFrac outScaleX'
    outScaleY' <- peek outScaleY
    let outScaleY'' = realToFrac outScaleY'
    outAngle' <- peek outAngle
    let outAngle'' = realToFrac outAngle'
    outDx' <- peek outDx
    let outDx'' = realToFrac outDx'
    outDy' <- peek outDy
    let outDy'' = realToFrac outDy'
    touchManagedPtr self
    freeMem outSkewX
    freeMem outSkewY
    freeMem outScaleX
    freeMem outScaleY
    freeMem outAngle
    freeMem outDx
    freeMem outDy
    return (outSkewX'', outSkewY'', outScaleX'', outScaleY'', outAngle'', outDx'', outDy'')

#if defined(ENABLE_OVERLOADING)
data TransformTo2dComponentsMethodInfo
instance (signature ~ (m ((Float, Float, Float, Float, Float, Float, Float))), MonadIO m) => O.OverloadedMethod TransformTo2dComponentsMethodInfo Transform signature where
    overloadedMethod = transformTo2dComponents

instance O.OverloadedMethodInfo TransformTo2dComponentsMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformTo2dComponents",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformTo2dComponents"
        })


#endif

-- method Transform::to_affine
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_scale_x"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the scale\n  factor in the x direction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "out_scale_y"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the scale\n  factor in the y direction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "out_dx"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the translation\n  in the x direction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "out_dy"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the translation\n  in the y direction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_to_affine" gsk_transform_to_affine :: 
    Ptr Transform ->                        -- self : TInterface (Name {namespace = "Gsk", name = "Transform"})
    Ptr CFloat ->                           -- out_scale_x : TBasicType TFloat
    Ptr CFloat ->                           -- out_scale_y : TBasicType TFloat
    Ptr CFloat ->                           -- out_dx : TBasicType TFloat
    Ptr CFloat ->                           -- out_dy : TBasicType TFloat
    IO ()

-- | Converts a transform to 2D affine transformation factors.
-- 
-- To recreate an equivalent transform from the factors returned
-- by this function, use
-- 
--     gsk_transform_scale (
--         gsk_transform_translate (
--             NULL,
--             &GRAPHENE_POINT_T (dx, dy)),
--         sx, sy)
-- 
-- /@self@/ must be a 2D affine transformation. If you are not
-- sure, use
-- 
--     'GI.Gsk.Structs.Transform.transformGetCategory' >= GSK_TRANSFORM_CATEGORY_2D_AFFINE
-- 
-- to check.
transformToAffine ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Transform
    -- ^ /@self@/: a transform
    -> m ((Float, Float, Float, Float))
transformToAffine self = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    outScaleX <- allocMem :: IO (Ptr CFloat)
    outScaleY <- allocMem :: IO (Ptr CFloat)
    outDx <- allocMem :: IO (Ptr CFloat)
    outDy <- allocMem :: IO (Ptr CFloat)
    gsk_transform_to_affine self' outScaleX outScaleY outDx outDy
    outScaleX' <- peek outScaleX
    let outScaleX'' = realToFrac outScaleX'
    outScaleY' <- peek outScaleY
    let outScaleY'' = realToFrac outScaleY'
    outDx' <- peek outDx
    let outDx'' = realToFrac outDx'
    outDy' <- peek outDy
    let outDy'' = realToFrac outDy'
    touchManagedPtr self
    freeMem outScaleX
    freeMem outScaleY
    freeMem outDx
    freeMem outDy
    return (outScaleX'', outScaleY'', outDx'', outDy'')

#if defined(ENABLE_OVERLOADING)
data TransformToAffineMethodInfo
instance (signature ~ (m ((Float, Float, Float, Float))), MonadIO m) => O.OverloadedMethod TransformToAffineMethodInfo Transform signature where
    overloadedMethod = transformToAffine

instance O.OverloadedMethodInfo TransformToAffineMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformToAffine",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformToAffine"
        })


#endif

-- method Transform::to_matrix
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_matrix"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Matrix" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for the matrix"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_to_matrix" gsk_transform_to_matrix :: 
    Ptr Transform ->                        -- self : TInterface (Name {namespace = "Gsk", name = "Transform"})
    Ptr Graphene.Matrix.Matrix ->           -- out_matrix : TInterface (Name {namespace = "Graphene", name = "Matrix"})
    IO ()

-- | Computes the 4x4 matrix for the transform.
-- 
-- The previous value of /@outMatrix@/ will be ignored.
transformToMatrix ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@self@/: a transform
    -> m (Graphene.Matrix.Matrix)
transformToMatrix self = liftIO $ do
    maybeSelf <- case self of
        Nothing -> return FP.nullPtr
        Just jSelf -> do
            jSelf' <- unsafeManagedPtrGetPtr jSelf
            return jSelf'
    outMatrix <- SP.callocBoxedBytes 64 :: IO (Ptr Graphene.Matrix.Matrix)
    gsk_transform_to_matrix maybeSelf outMatrix
    outMatrix' <- (wrapBoxed Graphene.Matrix.Matrix) outMatrix
    whenJust self touchManagedPtr
    return outMatrix'

#if defined(ENABLE_OVERLOADING)
data TransformToMatrixMethodInfo
instance (signature ~ (m (Graphene.Matrix.Matrix)), MonadIO m) => O.OverloadedMethod TransformToMatrixMethodInfo Transform signature where
    overloadedMethod i = transformToMatrix (Just i)

instance O.OverloadedMethodInfo TransformToMatrixMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformToMatrix",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformToMatrix"
        })


#endif

-- method Transform::to_string
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_to_string" gsk_transform_to_string :: 
    Ptr Transform ->                        -- self : TInterface (Name {namespace = "Gsk", name = "Transform"})
    IO CString

-- | Converts the transform into a human-readable string.
-- 
-- The resulting string can be parsed with 'GI.Gsk.Structs.Transform.transformParse'.
-- 
-- This is a wrapper around 'GI.Gsk.Structs.Transform.transformPrint'.
transformToString ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@self@/: a transform
    -> m T.Text
    -- ^ __Returns:__ A new string for /@self@/
transformToString self = liftIO $ do
    maybeSelf <- case self of
        Nothing -> return FP.nullPtr
        Just jSelf -> do
            jSelf' <- unsafeManagedPtrGetPtr jSelf
            return jSelf'
    result <- gsk_transform_to_string maybeSelf
    checkUnexpectedReturnNULL "transformToString" result
    result' <- cstringToText result
    freeMem result
    whenJust self touchManagedPtr
    return result'

#if defined(ENABLE_OVERLOADING)
data TransformToStringMethodInfo
instance (signature ~ (m T.Text), MonadIO m) => O.OverloadedMethod TransformToStringMethodInfo Transform signature where
    overloadedMethod i = transformToString (Just i)

instance O.OverloadedMethodInfo TransformToStringMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformToString",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformToString"
        })


#endif

-- method Transform::to_translate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_dx"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the translation\n  in the x direction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "out_dy"
--           , argType = TBasicType TFloat
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "return location for the translation\n  in the y direction"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_to_translate" gsk_transform_to_translate :: 
    Ptr Transform ->                        -- self : TInterface (Name {namespace = "Gsk", name = "Transform"})
    Ptr CFloat ->                           -- out_dx : TBasicType TFloat
    Ptr CFloat ->                           -- out_dy : TBasicType TFloat
    IO ()

-- | Converts a transform to a translation operation.
-- 
-- /@self@/ must be a 2D transformation. If you are not
-- sure, use
-- 
--     'GI.Gsk.Structs.Transform.transformGetCategory' >= GSK_TRANSFORM_CATEGORY_2D_TRANSLATE
-- 
-- to check.
transformToTranslate ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Transform
    -- ^ /@self@/: a transform
    -> m ((Float, Float))
transformToTranslate self = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    outDx <- allocMem :: IO (Ptr CFloat)
    outDy <- allocMem :: IO (Ptr CFloat)
    gsk_transform_to_translate self' outDx outDy
    outDx' <- peek outDx
    let outDx'' = realToFrac outDx'
    outDy' <- peek outDy
    let outDy'' = realToFrac outDy'
    touchManagedPtr self
    freeMem outDx
    freeMem outDy
    return (outDx'', outDy'')

#if defined(ENABLE_OVERLOADING)
data TransformToTranslateMethodInfo
instance (signature ~ (m ((Float, Float))), MonadIO m) => O.OverloadedMethod TransformToTranslateMethodInfo Transform signature where
    overloadedMethod = transformToTranslate

instance O.OverloadedMethodInfo TransformToTranslateMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformToTranslate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformToTranslate"
        })


#endif

-- method Transform::transform
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "next"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "transform to apply @other to"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "other"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "transform to apply" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "Transform" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_transform" gsk_transform_transform :: 
    Ptr Transform ->                        -- next : TInterface (Name {namespace = "Gsk", name = "Transform"})
    Ptr Transform ->                        -- other : TInterface (Name {namespace = "Gsk", name = "Transform"})
    IO (Ptr Transform)

-- | Applies all the operations from /@other@/ to /@next@/.
-- 
-- This function consumes /@next@/. Use 'GI.Gsk.Structs.Transform.transformRef' first
-- if you want to keep it around.
transformTransform ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@next@/: transform to apply /@other@/ to
    -> Maybe (Transform)
    -- ^ /@other@/: transform to apply
    -> m (Maybe Transform)
    -- ^ __Returns:__ The new transform
transformTransform next other = liftIO $ do
    maybeNext <- case next of
        Nothing -> return FP.nullPtr
        Just jNext -> do
            jNext' <- B.ManagedPtr.disownBoxed jNext
            return jNext'
    maybeOther <- case other of
        Nothing -> return FP.nullPtr
        Just jOther -> do
            jOther' <- unsafeManagedPtrGetPtr jOther
            return jOther'
    result <- gsk_transform_transform maybeNext maybeOther
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed Transform) result'
        return result''
    whenJust next touchManagedPtr
    whenJust other touchManagedPtr
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data TransformTransformMethodInfo
instance (signature ~ (Maybe (Transform) -> m (Maybe Transform)), MonadIO m) => O.OverloadedMethod TransformTransformMethodInfo Transform signature where
    overloadedMethod i = transformTransform (Just i)

instance O.OverloadedMethodInfo TransformTransformMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformTransform",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformTransform"
        })


#endif

-- method Transform::transform_bounds
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "rect"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Rect" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the rectangle to transform"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_rect"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Rect" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "return location for the bounds\n  of the transformed rectangle"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_transform_bounds" gsk_transform_transform_bounds :: 
    Ptr Transform ->                        -- self : TInterface (Name {namespace = "Gsk", name = "Transform"})
    Ptr Graphene.Rect.Rect ->               -- rect : TInterface (Name {namespace = "Graphene", name = "Rect"})
    Ptr Graphene.Rect.Rect ->               -- out_rect : TInterface (Name {namespace = "Graphene", name = "Rect"})
    IO ()

-- | Transforms a rectangle using the given transform.
-- 
-- The result is the bounding box containing the coplanar quad.
transformTransformBounds ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Transform
    -- ^ /@self@/: a transform
    -> Graphene.Rect.Rect
    -- ^ /@rect@/: the rectangle to transform
    -> m (Graphene.Rect.Rect)
transformTransformBounds self rect = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    rect' <- unsafeManagedPtrGetPtr rect
    outRect <- SP.callocBoxedBytes 16 :: IO (Ptr Graphene.Rect.Rect)
    gsk_transform_transform_bounds self' rect' outRect
    outRect' <- (wrapBoxed Graphene.Rect.Rect) outRect
    touchManagedPtr self
    touchManagedPtr rect
    return outRect'

#if defined(ENABLE_OVERLOADING)
data TransformTransformBoundsMethodInfo
instance (signature ~ (Graphene.Rect.Rect -> m (Graphene.Rect.Rect)), MonadIO m) => O.OverloadedMethod TransformTransformBoundsMethodInfo Transform signature where
    overloadedMethod = transformTransformBounds

instance O.OverloadedMethodInfo TransformTransformBoundsMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformTransformBounds",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformTransformBounds"
        })


#endif

-- method Transform::transform_point
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "point"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Point" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the point to transform"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_point"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Point" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for\n  the transformed point"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = True
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_transform_point" gsk_transform_transform_point :: 
    Ptr Transform ->                        -- self : TInterface (Name {namespace = "Gsk", name = "Transform"})
    Ptr Graphene.Point.Point ->             -- point : TInterface (Name {namespace = "Graphene", name = "Point"})
    Ptr Graphene.Point.Point ->             -- out_point : TInterface (Name {namespace = "Graphene", name = "Point"})
    IO ()

-- | Transforms a point using the given transform.
transformTransformPoint ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Transform
    -- ^ /@self@/: a transform
    -> Graphene.Point.Point
    -- ^ /@point@/: the point to transform
    -> m (Graphene.Point.Point)
transformTransformPoint self point = liftIO $ do
    self' <- unsafeManagedPtrGetPtr self
    point' <- unsafeManagedPtrGetPtr point
    outPoint <- SP.callocBoxedBytes 8 :: IO (Ptr Graphene.Point.Point)
    gsk_transform_transform_point self' point' outPoint
    outPoint' <- (wrapBoxed Graphene.Point.Point) outPoint
    touchManagedPtr self
    touchManagedPtr point
    return outPoint'

#if defined(ENABLE_OVERLOADING)
data TransformTransformPointMethodInfo
instance (signature ~ (Graphene.Point.Point -> m (Graphene.Point.Point)), MonadIO m) => O.OverloadedMethod TransformTransformPointMethodInfo Transform signature where
    overloadedMethod = transformTransformPoint

instance O.OverloadedMethodInfo TransformTransformPointMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformTransformPoint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformTransformPoint"
        })


#endif

-- method Transform::translate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "next"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the next transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "point"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Point" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the point to translate the transform by"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "Transform" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_translate" gsk_transform_translate :: 
    Ptr Transform ->                        -- next : TInterface (Name {namespace = "Gsk", name = "Transform"})
    Ptr Graphene.Point.Point ->             -- point : TInterface (Name {namespace = "Graphene", name = "Point"})
    IO (Ptr Transform)

-- | Translates /@next@/ in 2-dimensional space by /@point@/.
-- 
-- This function consumes /@next@/. Use 'GI.Gsk.Structs.Transform.transformRef' first
-- if you want to keep it around.
transformTranslate ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@next@/: the next transform
    -> Graphene.Point.Point
    -- ^ /@point@/: the point to translate the transform by
    -> m (Maybe Transform)
    -- ^ __Returns:__ The new transform
transformTranslate next point = liftIO $ do
    maybeNext <- case next of
        Nothing -> return FP.nullPtr
        Just jNext -> do
            jNext' <- B.ManagedPtr.disownBoxed jNext
            return jNext'
    point' <- unsafeManagedPtrGetPtr point
    result <- gsk_transform_translate maybeNext point'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed Transform) result'
        return result''
    whenJust next touchManagedPtr
    touchManagedPtr point
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data TransformTranslateMethodInfo
instance (signature ~ (Graphene.Point.Point -> m (Maybe Transform)), MonadIO m) => O.OverloadedMethod TransformTranslateMethodInfo Transform signature where
    overloadedMethod i = transformTranslate (Just i)

instance O.OverloadedMethodInfo TransformTranslateMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformTranslate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformTranslate"
        })


#endif

-- method Transform::translate_3d
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "next"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the next transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "point"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Point3D" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the point to translate the transform by"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "Transform" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_translate_3d" gsk_transform_translate_3d :: 
    Ptr Transform ->                        -- next : TInterface (Name {namespace = "Gsk", name = "Transform"})
    Ptr Graphene.Point3D.Point3D ->         -- point : TInterface (Name {namespace = "Graphene", name = "Point3D"})
    IO (Ptr Transform)

-- | Translates /@next@/ by /@point@/.
-- 
-- This function consumes /@next@/. Use 'GI.Gsk.Structs.Transform.transformRef' first
-- if you want to keep it around.
transformTranslate3d ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@next@/: the next transform
    -> Graphene.Point3D.Point3D
    -- ^ /@point@/: the point to translate the transform by
    -> m (Maybe Transform)
    -- ^ __Returns:__ The new transform
transformTranslate3d next point = liftIO $ do
    maybeNext <- case next of
        Nothing -> return FP.nullPtr
        Just jNext -> do
            jNext' <- B.ManagedPtr.disownBoxed jNext
            return jNext'
    point' <- unsafeManagedPtrGetPtr point
    result <- gsk_transform_translate_3d maybeNext point'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapBoxed Transform) result'
        return result''
    whenJust next touchManagedPtr
    touchManagedPtr point
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data TransformTranslate3dMethodInfo
instance (signature ~ (Graphene.Point3D.Point3D -> m (Maybe Transform)), MonadIO m) => O.OverloadedMethod TransformTranslate3dMethodInfo Transform signature where
    overloadedMethod i = transformTranslate3d (Just i)

instance O.OverloadedMethodInfo TransformTranslate3dMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformTranslate3d",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformTranslate3d"
        })


#endif

-- method Transform::unref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a transform" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_unref" gsk_transform_unref :: 
    Ptr Transform ->                        -- self : TInterface (Name {namespace = "Gsk", name = "Transform"})
    IO ()

-- | Releases a reference on the given transform.
-- 
-- If the reference was the last, the resources associated to the /@self@/ are
-- freed.
transformUnref ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    Maybe (Transform)
    -- ^ /@self@/: a transform
    -> m ()
transformUnref self = liftIO $ do
    maybeSelf <- case self of
        Nothing -> return FP.nullPtr
        Just jSelf -> do
            jSelf' <- unsafeManagedPtrGetPtr jSelf
            return jSelf'
    gsk_transform_unref maybeSelf
    whenJust self touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data TransformUnrefMethodInfo
instance (signature ~ (m ()), MonadIO m) => O.OverloadedMethod TransformUnrefMethodInfo Transform signature where
    overloadedMethod i = transformUnref (Just i)

instance O.OverloadedMethodInfo TransformUnrefMethodInfo Transform where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Structs.Transform.transformUnref",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Structs-Transform.html#v:transformUnref"
        })


#endif

-- method Transform::parse
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "string"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to parse"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_transform"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Transform" }
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for the transform"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_transform_parse" gsk_transform_parse :: 
    CString ->                              -- string : TBasicType TUTF8
    Ptr (Ptr Transform) ->                  -- out_transform : TInterface (Name {namespace = "Gsk", name = "Transform"})
    IO CInt

-- | Parses a given into a transform.
-- 
-- Strings printed via 'GI.Gsk.Structs.Transform.transformToString'
-- can be read in again successfully using this function.
-- 
-- If /@string@/ does not describe a valid transform, false
-- is returned and @NULL@ is put in /@outTransform@/.
transformParse ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@string@/: the string to parse
    -> m ((Bool, Transform))
    -- ^ __Returns:__ true if /@string@/ described a valid transform
transformParse string = liftIO $ do
    string' <- textToCString string
    outTransform <- callocMem :: IO (Ptr (Ptr Transform))
    result <- gsk_transform_parse string' outTransform
    let result' = (/= 0) result
    outTransform' <- peek outTransform
    outTransform'' <- (wrapBoxed Transform) outTransform'
    freeMem string'
    freeMem outTransform
    return (result', outTransform'')

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveTransformMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveTransformMethod "equal" o = TransformEqualMethodInfo
    ResolveTransformMethod "invert" o = TransformInvertMethodInfo
    ResolveTransformMethod "matrix" o = TransformMatrixMethodInfo
    ResolveTransformMethod "matrix2d" o = TransformMatrix2dMethodInfo
    ResolveTransformMethod "perspective" o = TransformPerspectiveMethodInfo
    ResolveTransformMethod "print" o = TransformPrintMethodInfo
    ResolveTransformMethod "ref" o = TransformRefMethodInfo
    ResolveTransformMethod "rotate" o = TransformRotateMethodInfo
    ResolveTransformMethod "rotate3d" o = TransformRotate3dMethodInfo
    ResolveTransformMethod "scale" o = TransformScaleMethodInfo
    ResolveTransformMethod "scale3d" o = TransformScale3dMethodInfo
    ResolveTransformMethod "skew" o = TransformSkewMethodInfo
    ResolveTransformMethod "to2d" o = TransformTo2dMethodInfo
    ResolveTransformMethod "to2dComponents" o = TransformTo2dComponentsMethodInfo
    ResolveTransformMethod "toAffine" o = TransformToAffineMethodInfo
    ResolveTransformMethod "toMatrix" o = TransformToMatrixMethodInfo
    ResolveTransformMethod "toString" o = TransformToStringMethodInfo
    ResolveTransformMethod "toTranslate" o = TransformToTranslateMethodInfo
    ResolveTransformMethod "transform" o = TransformTransformMethodInfo
    ResolveTransformMethod "transformBounds" o = TransformTransformBoundsMethodInfo
    ResolveTransformMethod "transformPoint" o = TransformTransformPointMethodInfo
    ResolveTransformMethod "translate" o = TransformTranslateMethodInfo
    ResolveTransformMethod "translate3d" o = TransformTranslate3dMethodInfo
    ResolveTransformMethod "unref" o = TransformUnrefMethodInfo
    ResolveTransformMethod "getCategory" o = TransformGetCategoryMethodInfo
    ResolveTransformMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveTransformMethod t Transform, O.OverloadedMethod info Transform p) => OL.IsLabel t (Transform -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveTransformMethod t Transform, O.OverloadedMethod info Transform p, R.HasField t Transform p) => R.HasField t Transform p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveTransformMethod t Transform, O.OverloadedMethodInfo info Transform) => OL.IsLabel t (O.MethodProxy info Transform) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


