{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- A @GTrashStack@ is an efficient way to keep a stack of unused allocated
-- memory chunks. Each memory chunk is required to be large enough to hold
-- a @gpointer@. This allows the stack to be maintained without any space
-- overhead, since the stack pointers can be stored inside the memory chunks.
-- 
-- There is no function to create a @GTrashStack@. A @NULL@ @GTrashStack*@
-- is a perfectly valid empty stack.
-- 
-- Each piece of memory that is pushed onto the stack is cast to a
-- @GTrashStack*@.
-- 
-- There is no longer any good reason to use @GTrashStack@.  If you have
-- extra pieces of memory, @free()@ them and allocate them again later.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GLib.Structs.TrashStack
    ( 

-- * Exported types
    TrashStack(..)                          ,
    newZeroTrashStack                       ,


 -- * Methods

#if defined(ENABLE_OVERLOADING)
    ResolveTrashStackMethod                 ,
#endif

-- ** height #method:height#

    trashStackHeight                        ,


-- ** peek #method:peek#

    trashStackPeek                          ,


-- ** pop #method:pop#

    trashStackPop                           ,


-- ** push #method:push#

    trashStackPush                          ,




 -- * Properties


-- ** next #attr:next#
-- | pointer to the previous element of the stack,
--     gets stored in the first @sizeof (gpointer)@
--     bytes of the element

    clearTrashStackNext                     ,
    getTrashStackNext                       ,
    setTrashStackNext                       ,
#if defined(ENABLE_OVERLOADING)
    trashStack_next                         ,
#endif




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)

#else

#endif

-- | Memory-managed wrapper type.
newtype TrashStack = TrashStack (SP.ManagedPtr TrashStack)
    deriving (Eq)

instance SP.ManagedPtrNewtype TrashStack where
    toManagedPtr (TrashStack p) = p

instance BoxedPtr TrashStack where
    boxedPtrCopy = \p -> B.ManagedPtr.withManagedPtr p (copyBytes 8 >=> B.ManagedPtr.wrapPtr TrashStack)
    boxedPtrFree = \x -> SP.withManagedPtr x SP.freeMem
instance CallocPtr TrashStack where
    boxedPtrCalloc = callocBytes 8


-- | Construct a t'TrashStack' struct initialized to zero.
newZeroTrashStack :: MonadIO m => m TrashStack
newZeroTrashStack = liftIO $ boxedPtrCalloc >>= wrapPtr TrashStack

instance tag ~ 'AttrSet => Constructible TrashStack tag where
    new _ attrs = do
        o <- newZeroTrashStack
        GI.Attributes.set o attrs
        return o


-- | Get the value of the “@next@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' trashStack #next
-- @
getTrashStackNext :: MonadIO m => TrashStack -> m (Maybe TrashStack)
getTrashStackNext s = liftIO $ withManagedPtr s $ \ptr -> do
    val <- peek (ptr `plusPtr` 0) :: IO (Ptr TrashStack)
    result <- SP.convertIfNonNull val $ \val' -> do
        val'' <- (newPtr TrashStack) val'
        return val''
    return result

-- | Set the value of the “@next@” field.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' trashStack [ #next 'Data.GI.Base.Attributes.:=' value ]
-- @
setTrashStackNext :: MonadIO m => TrashStack -> Ptr TrashStack -> m ()
setTrashStackNext s val = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 0) (val :: Ptr TrashStack)

-- | Set the value of the “@next@” field to `Nothing`.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.clear' #next
-- @
clearTrashStackNext :: MonadIO m => TrashStack -> m ()
clearTrashStackNext s = liftIO $ withManagedPtr s $ \ptr -> do
    poke (ptr `plusPtr` 0) (FP.nullPtr :: Ptr TrashStack)

#if defined(ENABLE_OVERLOADING)
data TrashStackNextFieldInfo
instance AttrInfo TrashStackNextFieldInfo where
    type AttrBaseTypeConstraint TrashStackNextFieldInfo = (~) TrashStack
    type AttrAllowedOps TrashStackNextFieldInfo = '[ 'AttrSet, 'AttrGet, 'AttrClear]
    type AttrSetTypeConstraint TrashStackNextFieldInfo = (~) (Ptr TrashStack)
    type AttrTransferTypeConstraint TrashStackNextFieldInfo = (~)(Ptr TrashStack)
    type AttrTransferType TrashStackNextFieldInfo = (Ptr TrashStack)
    type AttrGetType TrashStackNextFieldInfo = Maybe TrashStack
    type AttrLabel TrashStackNextFieldInfo = "next"
    type AttrOrigin TrashStackNextFieldInfo = TrashStack
    attrGet = getTrashStackNext
    attrSet = setTrashStackNext
    attrConstruct = undefined
    attrClear = clearTrashStackNext
    attrTransfer _ v = do
        return v
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GLib.Structs.TrashStack.next"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-glib-2.0.30/docs/GI-GLib-Structs-TrashStack.html#g:attr:next"
        })

trashStack_next :: AttrLabelProxy "next"
trashStack_next = AttrLabelProxy

#endif



#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList TrashStack
type instance O.AttributeList TrashStack = TrashStackAttributeList
type TrashStackAttributeList = ('[ '("next", TrashStackNextFieldInfo)] :: [(Symbol, DK.Type)])
#endif

-- method TrashStack::height
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "stack_p"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TrashStack" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTrashStack" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "g_trash_stack_height" g_trash_stack_height :: 
    Ptr TrashStack ->                       -- stack_p : TInterface (Name {namespace = "GLib", name = "TrashStack"})
    IO Word32

{-# DEPRECATED trashStackHeight ["(Since version 2.48)","t'GI.GLib.Structs.TrashStack.TrashStack' is deprecated without replacement"] #-}
-- | Returns the height of a t'GI.GLib.Structs.TrashStack.TrashStack'.
-- 
-- Note that execution of this function is of O(N) complexity
-- where N denotes the number of items on the stack.
trashStackHeight ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TrashStack
    -- ^ /@stackP@/: a t'GI.GLib.Structs.TrashStack.TrashStack'
    -> m Word32
    -- ^ __Returns:__ the height of the stack
trashStackHeight stackP = liftIO $ do
    stackP' <- unsafeManagedPtrGetPtr stackP
    result <- g_trash_stack_height stackP'
    touchManagedPtr stackP
    return result

#if defined(ENABLE_OVERLOADING)
#endif

-- method TrashStack::peek
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "stack_p"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TrashStack" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTrashStack" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TPtr)
-- throws : False
-- Skip return : False

foreign import ccall "g_trash_stack_peek" g_trash_stack_peek :: 
    Ptr TrashStack ->                       -- stack_p : TInterface (Name {namespace = "GLib", name = "TrashStack"})
    IO (Ptr ())

{-# DEPRECATED trashStackPeek ["(Since version 2.48)","t'GI.GLib.Structs.TrashStack.TrashStack' is deprecated without replacement"] #-}
-- | Returns the element at the top of a t'GI.GLib.Structs.TrashStack.TrashStack'
-- which may be 'P.Nothing'.
trashStackPeek ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TrashStack
    -- ^ /@stackP@/: a t'GI.GLib.Structs.TrashStack.TrashStack'
    -> m (Ptr ())
    -- ^ __Returns:__ the element at the top of the stack
trashStackPeek stackP = liftIO $ do
    stackP' <- unsafeManagedPtrGetPtr stackP
    result <- g_trash_stack_peek stackP'
    touchManagedPtr stackP
    return result

#if defined(ENABLE_OVERLOADING)
#endif

-- method TrashStack::pop
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "stack_p"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TrashStack" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTrashStack" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TPtr)
-- throws : False
-- Skip return : False

foreign import ccall "g_trash_stack_pop" g_trash_stack_pop :: 
    Ptr TrashStack ->                       -- stack_p : TInterface (Name {namespace = "GLib", name = "TrashStack"})
    IO (Ptr ())

{-# DEPRECATED trashStackPop ["(Since version 2.48)","t'GI.GLib.Structs.TrashStack.TrashStack' is deprecated without replacement"] #-}
-- | Pops a piece of memory off a t'GI.GLib.Structs.TrashStack.TrashStack'.
trashStackPop ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TrashStack
    -- ^ /@stackP@/: a t'GI.GLib.Structs.TrashStack.TrashStack'
    -> m (Ptr ())
    -- ^ __Returns:__ the element at the top of the stack
trashStackPop stackP = liftIO $ do
    stackP' <- unsafeManagedPtrGetPtr stackP
    result <- g_trash_stack_pop stackP'
    touchManagedPtr stackP
    return result

#if defined(ENABLE_OVERLOADING)
#endif

-- method TrashStack::push
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "stack_p"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "TrashStack" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GTrashStack" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "data_p"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the piece of memory to push on the stack"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_trash_stack_push" g_trash_stack_push :: 
    Ptr TrashStack ->                       -- stack_p : TInterface (Name {namespace = "GLib", name = "TrashStack"})
    Ptr () ->                               -- data_p : TBasicType TPtr
    IO ()

{-# DEPRECATED trashStackPush ["(Since version 2.48)","t'GI.GLib.Structs.TrashStack.TrashStack' is deprecated without replacement"] #-}
-- | Pushes a piece of memory onto a t'GI.GLib.Structs.TrashStack.TrashStack'.
trashStackPush ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    TrashStack
    -- ^ /@stackP@/: a t'GI.GLib.Structs.TrashStack.TrashStack'
    -> Ptr ()
    -- ^ /@dataP@/: the piece of memory to push on the stack
    -> m ()
trashStackPush stackP dataP = liftIO $ do
    stackP' <- unsafeManagedPtrGetPtr stackP
    g_trash_stack_push stackP' dataP
    touchManagedPtr stackP
    return ()

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveTrashStackMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveTrashStackMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveTrashStackMethod t TrashStack, O.OverloadedMethod info TrashStack p) => OL.IsLabel t (TrashStack -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveTrashStackMethod t TrashStack, O.OverloadedMethod info TrashStack p, R.HasField t TrashStack p) => R.HasField t TrashStack p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveTrashStackMethod t TrashStack, O.OverloadedMethodInfo info TrashStack) => OL.IsLabel t (O.MethodProxy info TrashStack) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif


