#pragma once
#include "dnsdist-rust-bridge.hh"
#include <algorithm>
#include <array>
#include <cassert>
#include <cstddef>
#include <cstdint>
#include <initializer_list>
#include <iterator>
#include <memory>
#include <new>
#include <stdexcept>
#include <string>
#include <type_traits>
#include <utility>
#if __cplusplus >= 201703L
#include <string_view>
#endif
#if __cplusplus >= 202002L
#include <ranges>
#endif

namespace rust {
inline namespace cxxbridge1 {
// #include "rust/cxx.h"

#ifndef CXXBRIDGE1_PANIC
#define CXXBRIDGE1_PANIC
template <typename Exception>
void panic [[noreturn]] (const char *msg);
#endif // CXXBRIDGE1_PANIC

struct unsafe_bitcopy_t;

namespace {
template <typename T>
class impl;
} // namespace

class Opaque;

template <typename T>
::std::size_t size_of();
template <typename T>
::std::size_t align_of();

#ifndef CXXBRIDGE1_RUST_STRING
#define CXXBRIDGE1_RUST_STRING
class String final {
public:
  String() noexcept;
  String(const String &) noexcept;
  String(String &&) noexcept;
  ~String() noexcept;

  String(const std::string &);
  String(const char *);
  String(const char *, std::size_t);
  String(const char16_t *);
  String(const char16_t *, std::size_t);
#ifdef __cpp_char8_t
  String(const char8_t *s);
  String(const char8_t *s, std::size_t len);
#endif

  static String lossy(const std::string &) noexcept;
  static String lossy(const char *) noexcept;
  static String lossy(const char *, std::size_t) noexcept;
  static String lossy(const char16_t *) noexcept;
  static String lossy(const char16_t *, std::size_t) noexcept;

  String &operator=(const String &) & noexcept;
  String &operator=(String &&) & noexcept;

  explicit operator std::string() const;

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  const char *c_str() noexcept;

  std::size_t capacity() const noexcept;
  void reserve(size_t new_cap) noexcept;

  using iterator = char *;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const String &) const noexcept;
  bool operator!=(const String &) const noexcept;
  bool operator<(const String &) const noexcept;
  bool operator<=(const String &) const noexcept;
  bool operator>(const String &) const noexcept;
  bool operator>=(const String &) const noexcept;

  void swap(String &) noexcept;

  String(unsafe_bitcopy_t, const String &) noexcept;

private:
  struct lossy_t;
  String(lossy_t, const char *, std::size_t) noexcept;
  String(lossy_t, const char16_t *, std::size_t) noexcept;
  friend void swap(String &lhs, String &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};
#endif // CXXBRIDGE1_RUST_STRING

#ifndef CXXBRIDGE1_RUST_STR
#define CXXBRIDGE1_RUST_STR
class Str final {
public:
  Str() noexcept;
  Str(const String &) noexcept;
  Str(const std::string &);
  Str(const char *);
  Str(const char *, std::size_t);

  Str &operator=(const Str &) & noexcept = default;

  explicit operator std::string() const;
#if __cplusplus >= 201703L
  explicit operator std::string_view() const;
#endif

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  Str(const Str &) noexcept = default;
  ~Str() noexcept = default;

  using iterator = const char *;
  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const Str &) const noexcept;
  bool operator!=(const Str &) const noexcept;
  bool operator<(const Str &) const noexcept;
  bool operator<=(const Str &) const noexcept;
  bool operator>(const Str &) const noexcept;
  bool operator>=(const Str &) const noexcept;

  void swap(Str &) noexcept;

private:
  class uninit;
  Str(uninit) noexcept;
  friend impl<Str>;

  std::array<std::uintptr_t, 2> repr;
};
#endif // CXXBRIDGE1_RUST_STR

#ifndef CXXBRIDGE1_RUST_SLICE
#define CXXBRIDGE1_RUST_SLICE
namespace detail {
template <bool>
struct copy_assignable_if {};

template <>
struct copy_assignable_if<false> {
  copy_assignable_if() noexcept = default;
  copy_assignable_if(const copy_assignable_if &) noexcept = default;
  copy_assignable_if &operator=(const copy_assignable_if &) & noexcept = delete;
  copy_assignable_if &operator=(copy_assignable_if &&) & noexcept = default;
};
} // namespace detail

template <typename T>
class Slice final
    : private detail::copy_assignable_if<std::is_const<T>::value> {
public:
  using value_type = T;

  Slice() noexcept;
  Slice(T *, std::size_t count) noexcept;

  template <typename C>
  explicit Slice(C &c) : Slice(c.data(), c.size()) {}

  Slice &operator=(const Slice<T> &) & noexcept = default;
  Slice &operator=(Slice<T> &&) & noexcept = default;

  T *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  T &operator[](std::size_t n) const noexcept;
  T &at(std::size_t n) const;
  T &front() const noexcept;
  T &back() const noexcept;

  Slice(const Slice<T> &) noexcept = default;
  ~Slice() noexcept = default;

  class iterator;
  iterator begin() const noexcept;
  iterator end() const noexcept;

  void swap(Slice &) noexcept;

private:
  class uninit;
  Slice(uninit) noexcept;
  friend impl<Slice>;
  friend void sliceInit(void *, const void *, std::size_t) noexcept;
  friend void *slicePtr(const void *) noexcept;
  friend std::size_t sliceLen(const void *) noexcept;

  std::array<std::uintptr_t, 2> repr;
};

#ifdef __cpp_deduction_guides
template <typename C>
explicit Slice(C &c)
    -> Slice<std::remove_reference_t<decltype(*std::declval<C>().data())>>;
#endif // __cpp_deduction_guides

template <typename T>
class Slice<T>::iterator final {
public:
#if __cplusplus >= 202002L
  using iterator_category = std::contiguous_iterator_tag;
#else
  using iterator_category = std::random_access_iterator_tag;
#endif
  using value_type = T;
  using difference_type = std::ptrdiff_t;
  using pointer = typename std::add_pointer<T>::type;
  using reference = typename std::add_lvalue_reference<T>::type;

  reference operator*() const noexcept;
  pointer operator->() const noexcept;
  reference operator[](difference_type) const noexcept;

  iterator &operator++() noexcept;
  iterator operator++(int) noexcept;
  iterator &operator--() noexcept;
  iterator operator--(int) noexcept;

  iterator &operator+=(difference_type) noexcept;
  iterator &operator-=(difference_type) noexcept;
  iterator operator+(difference_type) const noexcept;
  friend inline iterator operator+(difference_type lhs, iterator rhs) noexcept {
    return rhs + lhs;
  }
  iterator operator-(difference_type) const noexcept;
  difference_type operator-(const iterator &) const noexcept;

  bool operator==(const iterator &) const noexcept;
  bool operator!=(const iterator &) const noexcept;
  bool operator<(const iterator &) const noexcept;
  bool operator<=(const iterator &) const noexcept;
  bool operator>(const iterator &) const noexcept;
  bool operator>=(const iterator &) const noexcept;

private:
  friend class Slice;
  void *pos;
  std::size_t stride;
};

#if __cplusplus >= 202002L
static_assert(std::ranges::contiguous_range<rust::Slice<const uint8_t>>);
static_assert(std::contiguous_iterator<rust::Slice<const uint8_t>::iterator>);
#endif

template <typename T>
Slice<T>::Slice() noexcept {
  sliceInit(this, reinterpret_cast<void *>(align_of<T>()), 0);
}

template <typename T>
Slice<T>::Slice(T *s, std::size_t count) noexcept {
  assert(s != nullptr || count == 0);
  sliceInit(this,
            s == nullptr && count == 0
                ? reinterpret_cast<void *>(align_of<T>())
                : const_cast<typename std::remove_const<T>::type *>(s),
            count);
}

template <typename T>
T *Slice<T>::data() const noexcept {
  return reinterpret_cast<T *>(slicePtr(this));
}

template <typename T>
std::size_t Slice<T>::size() const noexcept {
  return sliceLen(this);
}

template <typename T>
std::size_t Slice<T>::length() const noexcept {
  return this->size();
}

template <typename T>
bool Slice<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T &Slice<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto ptr = static_cast<char *>(slicePtr(this)) + size_of<T>() * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
T &Slice<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Slice index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Slice<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Slice<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
typename Slice<T>::iterator::reference
Slice<T>::iterator::operator*() const noexcept {
  return *static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::pointer
Slice<T>::iterator::operator->() const noexcept {
  return static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::reference Slice<T>::iterator::operator[](
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ptr = static_cast<char *>(this->pos) + this->stride * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator++() noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator++(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator--() noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator--(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator+=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator-=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator+(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) + this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator-(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) - this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator::difference_type
Slice<T>::iterator::operator-(const iterator &other) const noexcept {
  auto diff = std::distance(static_cast<char *>(other.pos),
                            static_cast<char *>(this->pos));
  return diff / static_cast<typename Slice<T>::iterator::difference_type>(
                    this->stride);
}

template <typename T>
bool Slice<T>::iterator::operator==(const iterator &other) const noexcept {
  return this->pos == other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator!=(const iterator &other) const noexcept {
  return this->pos != other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<(const iterator &other) const noexcept {
  return this->pos < other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<=(const iterator &other) const noexcept {
  return this->pos <= other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>(const iterator &other) const noexcept {
  return this->pos > other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>=(const iterator &other) const noexcept {
  return this->pos >= other.pos;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::begin() const noexcept {
  iterator it;
  it.pos = slicePtr(this);
  it.stride = size_of<T>();
  return it;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::end() const noexcept {
  iterator it = this->begin();
  it.pos = static_cast<char *>(it.pos) + it.stride * this->size();
  return it;
}

template <typename T>
void Slice<T>::swap(Slice &rhs) noexcept {
  std::swap(*this, rhs);
}
#endif // CXXBRIDGE1_RUST_SLICE

#ifndef CXXBRIDGE1_RUST_BITCOPY_T
#define CXXBRIDGE1_RUST_BITCOPY_T
struct unsafe_bitcopy_t final {
  explicit unsafe_bitcopy_t() = default;
};
#endif // CXXBRIDGE1_RUST_BITCOPY_T

#ifndef CXXBRIDGE1_RUST_VEC
#define CXXBRIDGE1_RUST_VEC
template <typename T>
class Vec final {
public:
  using value_type = T;

  Vec() noexcept;
  Vec(std::initializer_list<T>);
  Vec(const Vec &);
  Vec(Vec &&) noexcept;
  ~Vec() noexcept;

  Vec &operator=(Vec &&) & noexcept;
  Vec &operator=(const Vec &) &;

  std::size_t size() const noexcept;
  bool empty() const noexcept;
  const T *data() const noexcept;
  T *data() noexcept;
  std::size_t capacity() const noexcept;

  const T &operator[](std::size_t n) const noexcept;
  const T &at(std::size_t n) const;
  const T &front() const noexcept;
  const T &back() const noexcept;

  T &operator[](std::size_t n) noexcept;
  T &at(std::size_t n);
  T &front() noexcept;
  T &back() noexcept;

  void reserve(std::size_t new_cap);
  void push_back(const T &value);
  void push_back(T &&value);
  template <typename... Args>
  void emplace_back(Args &&...args);
  void truncate(std::size_t len);
  void clear();

  using iterator = typename Slice<T>::iterator;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = typename Slice<const T>::iterator;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  void swap(Vec &) noexcept;

  Vec(unsafe_bitcopy_t, const Vec &) noexcept;

private:
  void reserve_total(std::size_t new_cap) noexcept;
  void set_len(std::size_t len) noexcept;
  void drop() noexcept;

  friend void swap(Vec &lhs, Vec &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};

template <typename T>
Vec<T>::Vec(std::initializer_list<T> init) : Vec{} {
  this->reserve_total(init.size());
  std::move(init.begin(), init.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(const Vec &other) : Vec() {
  this->reserve_total(other.size());
  std::copy(other.begin(), other.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(Vec &&other) noexcept : repr(other.repr) {
  new (&other) Vec();
}

template <typename T>
Vec<T>::~Vec() noexcept {
  this->drop();
}

template <typename T>
Vec<T> &Vec<T>::operator=(Vec &&other) & noexcept {
  this->drop();
  this->repr = other.repr;
  new (&other) Vec();
  return *this;
}

template <typename T>
Vec<T> &Vec<T>::operator=(const Vec &other) & {
  if (this != &other) {
    this->drop();
    new (this) Vec(other);
  }
  return *this;
}

template <typename T>
bool Vec<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T *Vec<T>::data() noexcept {
  return const_cast<T *>(const_cast<const Vec<T> *>(this)->data());
}

template <typename T>
const T &Vec<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<const char *>(this->data());
  return *reinterpret_cast<const T *>(data + n * size_of<T>());
}

template <typename T>
const T &Vec<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
const T &Vec<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
const T &Vec<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
T &Vec<T>::operator[](std::size_t n) noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<char *>(this->data());
  return *reinterpret_cast<T *>(data + n * size_of<T>());
}

template <typename T>
T &Vec<T>::at(std::size_t n) {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Vec<T>::front() noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Vec<T>::back() noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
void Vec<T>::reserve(std::size_t new_cap) {
  this->reserve_total(new_cap);
}

template <typename T>
void Vec<T>::push_back(const T &value) {
  this->emplace_back(value);
}

template <typename T>
void Vec<T>::push_back(T &&value) {
  this->emplace_back(std::move(value));
}

template <typename T>
template <typename... Args>
void Vec<T>::emplace_back(Args &&...args) {
  auto size = this->size();
  this->reserve_total(size + 1);
  ::new (reinterpret_cast<T *>(reinterpret_cast<char *>(this->data()) +
                               size * size_of<T>()))
      T(std::forward<Args>(args)...);
  this->set_len(size + 1);
}

template <typename T>
void Vec<T>::clear() {
  this->truncate(0);
}

template <typename T>
typename Vec<T>::iterator Vec<T>::begin() noexcept {
  return Slice<T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::iterator Vec<T>::end() noexcept {
  return Slice<T>(this->data(), this->size()).end();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::begin() const noexcept {
  return this->cbegin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::end() const noexcept {
  return this->cend();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cbegin() const noexcept {
  return Slice<const T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cend() const noexcept {
  return Slice<const T>(this->data(), this->size()).end();
}

template <typename T>
void Vec<T>::swap(Vec &rhs) noexcept {
  using std::swap;
  swap(this->repr, rhs.repr);
}

template <typename T>
Vec<T>::Vec(unsafe_bitcopy_t, const Vec &bits) noexcept : repr(bits.repr) {}
#endif // CXXBRIDGE1_RUST_VEC

#ifndef CXXBRIDGE1_IS_COMPLETE
#define CXXBRIDGE1_IS_COMPLETE
namespace detail {
namespace {
template <typename T, typename = std::size_t>
struct is_complete : std::false_type {};
template <typename T>
struct is_complete<T, decltype(sizeof(T))> : std::true_type {};
} // namespace
} // namespace detail
#endif // CXXBRIDGE1_IS_COMPLETE

#ifndef CXXBRIDGE1_LAYOUT
#define CXXBRIDGE1_LAYOUT
class layout {
  template <typename T>
  friend std::size_t size_of();
  template <typename T>
  friend std::size_t align_of();
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return T::layout::size();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return sizeof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      size_of() {
    return do_size_of<T>();
  }
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return T::layout::align();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return alignof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      align_of() {
    return do_align_of<T>();
  }
};

template <typename T>
std::size_t size_of() {
  return layout::size_of<T>();
}

template <typename T>
std::size_t align_of() {
  return layout::align_of<T>();
}
#endif // CXXBRIDGE1_LAYOUT
} // namespace cxxbridge1
} // namespace rust

#if __cplusplus >= 201402L
#define CXX_DEFAULT_VALUE(value) = value
#else
#define CXX_DEFAULT_VALUE(value)
#endif

namespace dnsdist {
  namespace rust {
    namespace settings {
      struct ResponseConfig;
      struct SOAParams;
      struct SVCRecordAdditionalParams;
      struct SVCRecordParameters;
      struct SharedDNSSelector;
      struct SharedDNSAction;
      struct SharedDNSResponseAction;
      struct AllowActionConfiguration;
      struct ContinueActionConfiguration;
      struct DelayActionConfiguration;
      struct DnstapLogActionConfiguration;
      struct DropActionConfiguration;
      struct SetEDNSOptionActionConfiguration;
      struct ERCodeActionConfiguration;
      struct HTTPStatusActionConfiguration;
      struct KeyValueStoreLookupActionConfiguration;
      struct KeyValueStoreRangeLookupActionConfiguration;
      struct LogActionConfiguration;
      struct LuaActionConfiguration;
      struct LuaFFIActionConfiguration;
      struct LuaFFIPerThreadActionConfiguration;
      struct NegativeAndSOAActionConfiguration;
      struct NoneActionConfiguration;
      struct PoolActionConfiguration;
      struct QPSActionConfiguration;
      struct QPSPoolActionConfiguration;
      struct RCodeActionConfiguration;
      struct RemoteLogActionConfiguration;
      struct SetAdditionalProxyProtocolValueActionConfiguration;
      struct SetDisableECSActionConfiguration;
      struct SetDisableValidationActionConfiguration;
      struct SetECSActionConfiguration;
      struct SetECSOverrideActionConfiguration;
      struct SetECSPrefixLengthActionConfiguration;
      struct SetExtendedDNSErrorActionConfiguration;
      struct SetMacAddrActionConfiguration;
      struct SetMaxReturnedTTLActionConfiguration;
      struct SetNoRecurseActionConfiguration;
      struct SetProxyProtocolValuesActionConfiguration;
      struct SetSkipCacheActionConfiguration;
      struct SetTagActionConfiguration;
      struct SetTempFailureCacheTTLActionConfiguration;
      struct SNMPTrapActionConfiguration;
      struct SpoofActionConfiguration;
      struct SpoofCNAMEActionConfiguration;
      struct SpoofPacketActionConfiguration;
      struct SpoofRawActionConfiguration;
      struct SpoofSVCActionConfiguration;
      struct TCActionConfiguration;
      struct TeeActionConfiguration;
      struct AllowResponseActionConfiguration;
      struct ClearRecordTypesResponseActionConfiguration;
      struct DelayResponseActionConfiguration;
      struct DnstapLogResponseActionConfiguration;
      struct DropResponseActionConfiguration;
      struct LimitTTLResponseActionConfiguration;
      struct LogResponseActionConfiguration;
      struct LuaResponseActionConfiguration;
      struct LuaFFIResponseActionConfiguration;
      struct LuaFFIPerThreadResponseActionConfiguration;
      struct RemoteLogResponseActionConfiguration;
      struct SetEDNSOptionResponseActionConfiguration;
      struct SetExtendedDNSErrorResponseActionConfiguration;
      struct SetMaxReturnedTTLResponseActionConfiguration;
      struct SetMaxTTLResponseActionConfiguration;
      struct SetMinTTLResponseActionConfiguration;
      struct SetReducedTTLResponseActionConfiguration;
      struct SetSkipCacheResponseActionConfiguration;
      struct SetTagResponseActionConfiguration;
      struct SNMPTrapResponseActionConfiguration;
      struct TCResponseActionConfiguration;
      struct AllSelectorConfiguration;
      struct AndSelectorConfiguration;
      struct ByNameSelectorConfiguration;
      struct DNSSECSelectorConfiguration;
      struct DSTPortSelectorConfiguration;
      struct EDNSOptionSelectorConfiguration;
      struct EDNSVersionSelectorConfiguration;
      struct ERCodeSelectorConfiguration;
      struct HTTPHeaderSelectorConfiguration;
      struct HTTPPathSelectorConfiguration;
      struct HTTPPathRegexSelectorConfiguration;
      struct KeyValueStoreLookupSelectorConfiguration;
      struct KeyValueStoreRangeLookupSelectorConfiguration;
      struct LuaSelectorConfiguration;
      struct LuaFFISelectorConfiguration;
      struct LuaFFIPerThreadSelectorConfiguration;
      struct MaxQPSSelectorConfiguration;
      struct MaxQPSIPSelectorConfiguration;
      struct NetmaskGroupSelectorConfiguration;
      struct NotSelectorConfiguration;
      struct OpcodeSelectorConfiguration;
      struct OrSelectorConfiguration;
      struct PayloadSizeSelectorConfiguration;
      struct PoolAvailableSelectorConfiguration;
      struct PoolOutstandingSelectorConfiguration;
      struct ProbaSelectorConfiguration;
      struct ProxyProtocolValueSelectorConfiguration;
      struct QClassSelectorConfiguration;
      struct QNameSelectorConfiguration;
      struct QNameLabelsCountSelectorConfiguration;
      struct QNameSetSelectorConfiguration;
      struct QNameSuffixSelectorConfiguration;
      struct QNameWireLengthSelectorConfiguration;
      struct QTypeSelectorConfiguration;
      struct RCodeSelectorConfiguration;
      struct RDSelectorConfiguration;
      struct RE2SelectorConfiguration;
      struct RecordsCountSelectorConfiguration;
      struct RecordsTypeCountSelectorConfiguration;
      struct RegexSelectorConfiguration;
      struct SNISelectorConfiguration;
      struct TagSelectorConfiguration;
      struct TCPSelectorConfiguration;
      struct TrailingDataSelectorConfiguration;
      struct GlobalConfiguration;
      struct MetricsConfiguration;
      struct CarbonConfiguration;
      struct RemoteLoggingConfiguration;
      struct ProtobufLoggerConfiguration;
      struct DnstapLoggerConfiguration;
      struct ProtoBufMetaConfiguration;
      struct LmdbKvStoreConfiguration;
      struct CdbKvStoreConfiguration;
      struct KvsLookupKeySourceIpConfiguration;
      struct KvsLookupKeyQnameConfiguration;
      struct KvsLookupKeySuffixConfiguration;
      struct KvsLookupKeyTagConfiguration;
      struct KvsLookupKeysConfiguration;
      struct KeyValueStoresConfiguration;
      struct WebserverConfiguration;
      struct ConsoleConfiguration;
      struct EbpfMapConfiguration;
      struct EbpfConfiguration;
      struct EdnsClientSubnetConfiguration;
      struct DynamicRulesSettingsConfiguration;
      struct DynamicRuleConfiguration;
      struct DynamicRulesConfiguration;
      struct RingBuffersConfiguration;
      struct IncomingTlsCertificateKeyPairConfiguration;
      struct IncomingTlsConfiguration;
      struct OutgoingTlsConfiguration;
      struct HttpCustomResponseHeaderConfiguration;
      struct HttpResponsesMapConfiguration;
      struct IncomingDohConfiguration;
      struct IncomingDoqConfiguration;
      struct IncomingQuicConfiguration;
      struct IncomingDnscryptCertificateKeyPairConfiguration;
      struct IncomingDnscryptConfiguration;
      struct OutgoingDohConfiguration;
      struct IncomingTcpConfiguration;
      struct BindConfiguration;
      struct OutgoingTcpConfiguration;
      struct ProxyProtocolValueConfiguration;
      struct LazyHealthCheckConfiguration;
      struct HealthCheckConfiguration;
      struct OutgoingAutoUpgradeConfiguration;
      struct BackendConfiguration;
      struct TuningConfiguration;
      struct TcpTuningConfiguration;
      struct UdpTuningConfiguration;
      struct TlsEngineConfiguration;
      struct TlsTuningConfiguration;
      struct DohTuningConfiguration;
      struct CacheSettingsConfiguration;
      struct SecurityPollingConfiguration;
      struct StructuredLoggingConfiguration;
      struct LoggingConfiguration;
      struct GeneralConfiguration;
      struct NetmaskGroupConfiguration;
      struct PacketCacheConfiguration;
      struct ProxyProtocolConfiguration;
      struct SnmpConfiguration;
      struct QueryCountConfiguration;
      struct PoolConfiguration;
      struct CustomLoadBalancingPolicyConfiguration;
      struct LoadBalancingPoliciesConfiguration;
      struct QueryRuleConfiguration;
      struct ResponseRuleConfiguration;
      struct XskConfiguration;
      using DNSSelector = ::dnsdist::rust::settings::DNSSelector;
      using DNSActionWrapper = ::dnsdist::rust::settings::DNSActionWrapper;
      using DNSResponseActionWrapper = ::dnsdist::rust::settings::DNSResponseActionWrapper;
    }
  }
}

namespace dnsdist {
namespace rust {
namespace settings {
#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ResponseConfig
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ResponseConfig
struct ResponseConfig final {
  bool set_aa CXX_DEFAULT_VALUE(false);
  bool set_ad CXX_DEFAULT_VALUE(false);
  bool set_ra CXX_DEFAULT_VALUE(false);
  ::std::uint32_t ttl CXX_DEFAULT_VALUE(0);

  bool operator==(ResponseConfig const &) const noexcept;
  bool operator!=(ResponseConfig const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ResponseConfig

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SOAParams
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SOAParams
struct SOAParams final {
  ::std::uint32_t serial CXX_DEFAULT_VALUE(0);
  ::std::uint32_t refresh CXX_DEFAULT_VALUE(0);
  ::std::uint32_t retry CXX_DEFAULT_VALUE(0);
  ::std::uint32_t expire CXX_DEFAULT_VALUE(0);
  ::std::uint32_t minimum CXX_DEFAULT_VALUE(0);

  bool operator==(SOAParams const &) const noexcept;
  bool operator!=(SOAParams const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SOAParams

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SVCRecordAdditionalParams
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SVCRecordAdditionalParams
struct SVCRecordAdditionalParams final {
  ::std::uint16_t key CXX_DEFAULT_VALUE(0);
  ::rust::String value;

  bool operator==(SVCRecordAdditionalParams const &) const noexcept;
  bool operator!=(SVCRecordAdditionalParams const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SVCRecordAdditionalParams

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SVCRecordParameters
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SVCRecordParameters
struct SVCRecordParameters final {
  ::rust::Vec<::std::uint16_t> mandatory_params;
  ::rust::Vec<::rust::String> alpns;
  ::rust::Vec<::rust::String> ipv4_hints;
  ::rust::Vec<::rust::String> ipv6_hints;
  ::rust::Vec<::dnsdist::rust::settings::SVCRecordAdditionalParams> additional_params;
  ::rust::String target;
  ::std::uint16_t port CXX_DEFAULT_VALUE(0);
  ::std::uint16_t priority CXX_DEFAULT_VALUE(0);
  bool no_default_alpn CXX_DEFAULT_VALUE(false);

  bool operator==(SVCRecordParameters const &) const noexcept;
  bool operator!=(SVCRecordParameters const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SVCRecordParameters

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSSelector
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSSelector
struct SharedDNSSelector final {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSSelector> selector;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSSelector

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSAction
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSAction
struct SharedDNSAction final {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSActionWrapper> action;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSAction

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSResponseAction
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSResponseAction
struct SharedDNSResponseAction final {
  ::std::shared_ptr<::dnsdist::rust::settings::DNSResponseActionWrapper> action;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SharedDNSResponseAction

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllowActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllowActionConfiguration
struct AllowActionConfiguration final {
  ::rust::String name;

  bool operator==(AllowActionConfiguration const &) const noexcept;
  bool operator!=(AllowActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllowActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ContinueActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ContinueActionConfiguration
struct ContinueActionConfiguration final {
  ::rust::String name;
  ::dnsdist::rust::settings::SharedDNSAction action;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ContinueActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DelayActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DelayActionConfiguration
struct DelayActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t msec CXX_DEFAULT_VALUE(0);

  bool operator==(DelayActionConfiguration const &) const noexcept;
  bool operator!=(DelayActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DelayActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLogActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLogActionConfiguration
struct DnstapLogActionConfiguration final {
  ::rust::String name;
  ::rust::String identity;
  ::rust::String logger_name;
  ::rust::String alter_function_name;
  ::rust::String alter_function_code;
  ::rust::String alter_function_file;

  bool operator==(DnstapLogActionConfiguration const &) const noexcept;
  bool operator!=(DnstapLogActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLogActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DropActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DropActionConfiguration
struct DropActionConfiguration final {
  ::rust::String name;

  bool operator==(DropActionConfiguration const &) const noexcept;
  bool operator!=(DropActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DropActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetEDNSOptionActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetEDNSOptionActionConfiguration
struct SetEDNSOptionActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t code CXX_DEFAULT_VALUE(0);
  ::rust::String data;

  bool operator==(SetEDNSOptionActionConfiguration const &) const noexcept;
  bool operator!=(SetEDNSOptionActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetEDNSOptionActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ERCodeActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ERCodeActionConfiguration
struct ERCodeActionConfiguration final {
  ::rust::String name;
  ::rust::String rcode;
  ::dnsdist::rust::settings::ResponseConfig vars;

  bool operator==(ERCodeActionConfiguration const &) const noexcept;
  bool operator!=(ERCodeActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ERCodeActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPStatusActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPStatusActionConfiguration
struct HTTPStatusActionConfiguration final {
  ::rust::String name;
  ::std::uint16_t status CXX_DEFAULT_VALUE(0);
  ::rust::String body;
  ::rust::String content_type;
  ::dnsdist::rust::settings::ResponseConfig vars;

  bool operator==(HTTPStatusActionConfiguration const &) const noexcept;
  bool operator!=(HTTPStatusActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPStatusActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreLookupActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreLookupActionConfiguration
struct KeyValueStoreLookupActionConfiguration final {
  ::rust::String name;
  ::rust::String kvs_name;
  ::rust::String lookup_key_name;
  ::rust::String destination_tag;

  bool operator==(KeyValueStoreLookupActionConfiguration const &) const noexcept;
  bool operator!=(KeyValueStoreLookupActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreLookupActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreRangeLookupActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreRangeLookupActionConfiguration
struct KeyValueStoreRangeLookupActionConfiguration final {
  ::rust::String name;
  ::rust::String kvs_name;
  ::rust::String lookup_key_name;
  ::rust::String destination_tag;

  bool operator==(KeyValueStoreRangeLookupActionConfiguration const &) const noexcept;
  bool operator!=(KeyValueStoreRangeLookupActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreRangeLookupActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LogActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LogActionConfiguration
struct LogActionConfiguration final {
  ::rust::String name;
  ::rust::String file_name;
  bool binary CXX_DEFAULT_VALUE(false);
  bool append CXX_DEFAULT_VALUE(false);
  bool buffered CXX_DEFAULT_VALUE(false);
  bool verbose_only CXX_DEFAULT_VALUE(false);
  bool include_timestamp CXX_DEFAULT_VALUE(false);

  bool operator==(LogActionConfiguration const &) const noexcept;
  bool operator!=(LogActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LogActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaActionConfiguration
struct LuaActionConfiguration final {
  ::rust::String name;
  ::rust::String function_name;
  ::rust::String function_code;
  ::rust::String function_file;

  bool operator==(LuaActionConfiguration const &) const noexcept;
  bool operator!=(LuaActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIActionConfiguration
struct LuaFFIActionConfiguration final {
  ::rust::String name;
  ::rust::String function_name;
  ::rust::String function_code;
  ::rust::String function_file;

  bool operator==(LuaFFIActionConfiguration const &) const noexcept;
  bool operator!=(LuaFFIActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadActionConfiguration
struct LuaFFIPerThreadActionConfiguration final {
  ::rust::String name;
  ::rust::String code;

  bool operator==(LuaFFIPerThreadActionConfiguration const &) const noexcept;
  bool operator!=(LuaFFIPerThreadActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NegativeAndSOAActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NegativeAndSOAActionConfiguration
struct NegativeAndSOAActionConfiguration final {
  ::rust::String name;
  bool nxd CXX_DEFAULT_VALUE(false);
  ::rust::String zone;
  ::std::uint32_t ttl CXX_DEFAULT_VALUE(0);
  ::rust::String mname;
  ::rust::String rname;
  ::dnsdist::rust::settings::SOAParams soa_parameters;
  bool soa_in_authority CXX_DEFAULT_VALUE(false);
  ::dnsdist::rust::settings::ResponseConfig vars;

  bool operator==(NegativeAndSOAActionConfiguration const &) const noexcept;
  bool operator!=(NegativeAndSOAActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NegativeAndSOAActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NoneActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NoneActionConfiguration
struct NoneActionConfiguration final {
  ::rust::String name;

  bool operator==(NoneActionConfiguration const &) const noexcept;
  bool operator!=(NoneActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NoneActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolActionConfiguration
struct PoolActionConfiguration final {
  ::rust::String name;
  ::rust::String pool_name;
  bool stop_processing CXX_DEFAULT_VALUE(false);

  bool operator==(PoolActionConfiguration const &) const noexcept;
  bool operator!=(PoolActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QPSActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QPSActionConfiguration
struct QPSActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t limit CXX_DEFAULT_VALUE(0);

  bool operator==(QPSActionConfiguration const &) const noexcept;
  bool operator!=(QPSActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QPSActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QPSPoolActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QPSPoolActionConfiguration
struct QPSPoolActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t limit CXX_DEFAULT_VALUE(0);
  ::rust::String pool_name;
  bool stop_processing CXX_DEFAULT_VALUE(false);

  bool operator==(QPSPoolActionConfiguration const &) const noexcept;
  bool operator!=(QPSPoolActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QPSPoolActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RCodeActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RCodeActionConfiguration
struct RCodeActionConfiguration final {
  ::rust::String name;
  ::rust::String rcode;
  ::dnsdist::rust::settings::ResponseConfig vars;

  bool operator==(RCodeActionConfiguration const &) const noexcept;
  bool operator!=(RCodeActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RCodeActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLogActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLogActionConfiguration
struct RemoteLogActionConfiguration final {
  ::rust::String name;
  ::rust::String logger_name;
  ::rust::String alter_function_name;
  ::rust::String alter_function_code;
  ::rust::String alter_function_file;
  ::rust::String server_id;
  ::rust::String ip_encrypt_key;
  ::rust::Vec<::rust::String> export_tags;
  ::rust::Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration> metas;

  bool operator==(RemoteLogActionConfiguration const &) const noexcept;
  bool operator!=(RemoteLogActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLogActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetAdditionalProxyProtocolValueActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetAdditionalProxyProtocolValueActionConfiguration
struct SetAdditionalProxyProtocolValueActionConfiguration final {
  ::rust::String name;
  ::std::uint8_t proxy_type CXX_DEFAULT_VALUE(0);
  ::rust::String value;

  bool operator==(SetAdditionalProxyProtocolValueActionConfiguration const &) const noexcept;
  bool operator!=(SetAdditionalProxyProtocolValueActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetAdditionalProxyProtocolValueActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetDisableECSActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetDisableECSActionConfiguration
struct SetDisableECSActionConfiguration final {
  ::rust::String name;

  bool operator==(SetDisableECSActionConfiguration const &) const noexcept;
  bool operator!=(SetDisableECSActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetDisableECSActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetDisableValidationActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetDisableValidationActionConfiguration
struct SetDisableValidationActionConfiguration final {
  ::rust::String name;

  bool operator==(SetDisableValidationActionConfiguration const &) const noexcept;
  bool operator!=(SetDisableValidationActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetDisableValidationActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSActionConfiguration
struct SetECSActionConfiguration final {
  ::rust::String name;
  ::rust::String ipv4;
  ::rust::String ipv6;

  bool operator==(SetECSActionConfiguration const &) const noexcept;
  bool operator!=(SetECSActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSOverrideActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSOverrideActionConfiguration
struct SetECSOverrideActionConfiguration final {
  ::rust::String name;
  bool override_existing CXX_DEFAULT_VALUE(false);

  bool operator==(SetECSOverrideActionConfiguration const &) const noexcept;
  bool operator!=(SetECSOverrideActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSOverrideActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSPrefixLengthActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSPrefixLengthActionConfiguration
struct SetECSPrefixLengthActionConfiguration final {
  ::rust::String name;
  ::std::uint16_t ipv4 CXX_DEFAULT_VALUE(0);
  ::std::uint16_t ipv6 CXX_DEFAULT_VALUE(0);

  bool operator==(SetECSPrefixLengthActionConfiguration const &) const noexcept;
  bool operator!=(SetECSPrefixLengthActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetECSPrefixLengthActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetExtendedDNSErrorActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetExtendedDNSErrorActionConfiguration
struct SetExtendedDNSErrorActionConfiguration final {
  ::rust::String name;
  ::std::uint16_t info_code CXX_DEFAULT_VALUE(0);
  ::rust::String extra_text;

  bool operator==(SetExtendedDNSErrorActionConfiguration const &) const noexcept;
  bool operator!=(SetExtendedDNSErrorActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetExtendedDNSErrorActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMacAddrActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMacAddrActionConfiguration
struct SetMacAddrActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t code CXX_DEFAULT_VALUE(0);

  bool operator==(SetMacAddrActionConfiguration const &) const noexcept;
  bool operator!=(SetMacAddrActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMacAddrActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxReturnedTTLActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxReturnedTTLActionConfiguration
struct SetMaxReturnedTTLActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t max CXX_DEFAULT_VALUE(0);

  bool operator==(SetMaxReturnedTTLActionConfiguration const &) const noexcept;
  bool operator!=(SetMaxReturnedTTLActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxReturnedTTLActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetNoRecurseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetNoRecurseActionConfiguration
struct SetNoRecurseActionConfiguration final {
  ::rust::String name;

  bool operator==(SetNoRecurseActionConfiguration const &) const noexcept;
  bool operator!=(SetNoRecurseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetNoRecurseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetProxyProtocolValuesActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetProxyProtocolValuesActionConfiguration
struct SetProxyProtocolValuesActionConfiguration final {
  ::rust::String name;
  ::rust::Vec<::dnsdist::rust::settings::ProxyProtocolValueConfiguration> values;

  bool operator==(SetProxyProtocolValuesActionConfiguration const &) const noexcept;
  bool operator!=(SetProxyProtocolValuesActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetProxyProtocolValuesActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetSkipCacheActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetSkipCacheActionConfiguration
struct SetSkipCacheActionConfiguration final {
  ::rust::String name;

  bool operator==(SetSkipCacheActionConfiguration const &) const noexcept;
  bool operator!=(SetSkipCacheActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetSkipCacheActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTagActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTagActionConfiguration
struct SetTagActionConfiguration final {
  ::rust::String name;
  ::rust::String tag;
  ::rust::String value;

  bool operator==(SetTagActionConfiguration const &) const noexcept;
  bool operator!=(SetTagActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTagActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTempFailureCacheTTLActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTempFailureCacheTTLActionConfiguration
struct SetTempFailureCacheTTLActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t ttl CXX_DEFAULT_VALUE(0);

  bool operator==(SetTempFailureCacheTTLActionConfiguration const &) const noexcept;
  bool operator!=(SetTempFailureCacheTTLActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTempFailureCacheTTLActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNMPTrapActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNMPTrapActionConfiguration
struct SNMPTrapActionConfiguration final {
  ::rust::String name;
  ::rust::String reason;

  bool operator==(SNMPTrapActionConfiguration const &) const noexcept;
  bool operator!=(SNMPTrapActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNMPTrapActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofActionConfiguration
struct SpoofActionConfiguration final {
  ::rust::String name;
  ::rust::Vec<::rust::String> ips;
  ::dnsdist::rust::settings::ResponseConfig vars;

  bool operator==(SpoofActionConfiguration const &) const noexcept;
  bool operator!=(SpoofActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofCNAMEActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofCNAMEActionConfiguration
struct SpoofCNAMEActionConfiguration final {
  ::rust::String name;
  ::rust::String cname;
  ::dnsdist::rust::settings::ResponseConfig vars;

  bool operator==(SpoofCNAMEActionConfiguration const &) const noexcept;
  bool operator!=(SpoofCNAMEActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofCNAMEActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofPacketActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofPacketActionConfiguration
struct SpoofPacketActionConfiguration final {
  ::rust::String name;
  ::rust::String response;
  ::std::uint32_t len CXX_DEFAULT_VALUE(0);

  bool operator==(SpoofPacketActionConfiguration const &) const noexcept;
  bool operator!=(SpoofPacketActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofPacketActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofRawActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofRawActionConfiguration
struct SpoofRawActionConfiguration final {
  ::rust::String name;
  ::rust::Vec<::rust::String> answers;
  ::rust::String qtype_for_any;
  ::dnsdist::rust::settings::ResponseConfig vars;

  bool operator==(SpoofRawActionConfiguration const &) const noexcept;
  bool operator!=(SpoofRawActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofRawActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofSVCActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofSVCActionConfiguration
struct SpoofSVCActionConfiguration final {
  ::rust::String name;
  ::rust::Vec<::dnsdist::rust::settings::SVCRecordParameters> parameters;
  ::dnsdist::rust::settings::ResponseConfig vars;

  bool operator==(SpoofSVCActionConfiguration const &) const noexcept;
  bool operator!=(SpoofSVCActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SpoofSVCActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCActionConfiguration
struct TCActionConfiguration final {
  ::rust::String name;

  bool operator==(TCActionConfiguration const &) const noexcept;
  bool operator!=(TCActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TeeActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TeeActionConfiguration
struct TeeActionConfiguration final {
  ::rust::String name;
  ::rust::String rca;
  ::rust::String lca;
  bool add_ecs CXX_DEFAULT_VALUE(false);
  bool add_proxy_protocol CXX_DEFAULT_VALUE(false);

  bool operator==(TeeActionConfiguration const &) const noexcept;
  bool operator!=(TeeActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TeeActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllowResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllowResponseActionConfiguration
struct AllowResponseActionConfiguration final {
  ::rust::String name;

  bool operator==(AllowResponseActionConfiguration const &) const noexcept;
  bool operator!=(AllowResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllowResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ClearRecordTypesResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ClearRecordTypesResponseActionConfiguration
struct ClearRecordTypesResponseActionConfiguration final {
  ::rust::String name;
  ::rust::Vec<::std::uint16_t> types;

  bool operator==(ClearRecordTypesResponseActionConfiguration const &) const noexcept;
  bool operator!=(ClearRecordTypesResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ClearRecordTypesResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DelayResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DelayResponseActionConfiguration
struct DelayResponseActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t msec CXX_DEFAULT_VALUE(0);

  bool operator==(DelayResponseActionConfiguration const &) const noexcept;
  bool operator!=(DelayResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DelayResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLogResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLogResponseActionConfiguration
struct DnstapLogResponseActionConfiguration final {
  ::rust::String name;
  ::rust::String identity;
  ::rust::String logger_name;
  ::rust::String alter_function_name;
  ::rust::String alter_function_code;
  ::rust::String alter_function_file;

  bool operator==(DnstapLogResponseActionConfiguration const &) const noexcept;
  bool operator!=(DnstapLogResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLogResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DropResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DropResponseActionConfiguration
struct DropResponseActionConfiguration final {
  ::rust::String name;

  bool operator==(DropResponseActionConfiguration const &) const noexcept;
  bool operator!=(DropResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DropResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LimitTTLResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LimitTTLResponseActionConfiguration
struct LimitTTLResponseActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t min CXX_DEFAULT_VALUE(0);
  ::std::uint32_t max CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::std::uint16_t> types;

  bool operator==(LimitTTLResponseActionConfiguration const &) const noexcept;
  bool operator!=(LimitTTLResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LimitTTLResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LogResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LogResponseActionConfiguration
struct LogResponseActionConfiguration final {
  ::rust::String name;
  ::rust::String file_name;
  bool append CXX_DEFAULT_VALUE(false);
  bool buffered CXX_DEFAULT_VALUE(false);
  bool verbose_only CXX_DEFAULT_VALUE(false);
  bool include_timestamp CXX_DEFAULT_VALUE(false);

  bool operator==(LogResponseActionConfiguration const &) const noexcept;
  bool operator!=(LogResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LogResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaResponseActionConfiguration
struct LuaResponseActionConfiguration final {
  ::rust::String name;
  ::rust::String function_name;
  ::rust::String function_code;
  ::rust::String function_file;

  bool operator==(LuaResponseActionConfiguration const &) const noexcept;
  bool operator!=(LuaResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIResponseActionConfiguration
struct LuaFFIResponseActionConfiguration final {
  ::rust::String name;
  ::rust::String function_name;
  ::rust::String function_code;
  ::rust::String function_file;

  bool operator==(LuaFFIResponseActionConfiguration const &) const noexcept;
  bool operator!=(LuaFFIResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadResponseActionConfiguration
struct LuaFFIPerThreadResponseActionConfiguration final {
  ::rust::String name;
  ::rust::String code;

  bool operator==(LuaFFIPerThreadResponseActionConfiguration const &) const noexcept;
  bool operator!=(LuaFFIPerThreadResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLogResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLogResponseActionConfiguration
struct RemoteLogResponseActionConfiguration final {
  ::rust::String name;
  ::rust::String logger_name;
  ::rust::String alter_function_name;
  ::rust::String alter_function_code;
  ::rust::String alter_function_file;
  ::rust::String server_id;
  ::rust::String ip_encrypt_key;
  bool include_cname CXX_DEFAULT_VALUE(false);
  ::rust::Vec<::rust::String> export_tags;
  ::rust::String export_extended_errors_to_meta;
  ::rust::Vec<::dnsdist::rust::settings::ProtoBufMetaConfiguration> metas;

  bool operator==(RemoteLogResponseActionConfiguration const &) const noexcept;
  bool operator!=(RemoteLogResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLogResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetEDNSOptionResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetEDNSOptionResponseActionConfiguration
struct SetEDNSOptionResponseActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t code CXX_DEFAULT_VALUE(0);
  ::rust::String data;

  bool operator==(SetEDNSOptionResponseActionConfiguration const &) const noexcept;
  bool operator!=(SetEDNSOptionResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetEDNSOptionResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetExtendedDNSErrorResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetExtendedDNSErrorResponseActionConfiguration
struct SetExtendedDNSErrorResponseActionConfiguration final {
  ::rust::String name;
  ::std::uint16_t info_code CXX_DEFAULT_VALUE(0);
  ::rust::String extra_text;

  bool operator==(SetExtendedDNSErrorResponseActionConfiguration const &) const noexcept;
  bool operator!=(SetExtendedDNSErrorResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetExtendedDNSErrorResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxReturnedTTLResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxReturnedTTLResponseActionConfiguration
struct SetMaxReturnedTTLResponseActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t max CXX_DEFAULT_VALUE(0);

  bool operator==(SetMaxReturnedTTLResponseActionConfiguration const &) const noexcept;
  bool operator!=(SetMaxReturnedTTLResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxReturnedTTLResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxTTLResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxTTLResponseActionConfiguration
struct SetMaxTTLResponseActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t max CXX_DEFAULT_VALUE(0);

  bool operator==(SetMaxTTLResponseActionConfiguration const &) const noexcept;
  bool operator!=(SetMaxTTLResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMaxTTLResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMinTTLResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMinTTLResponseActionConfiguration
struct SetMinTTLResponseActionConfiguration final {
  ::rust::String name;
  ::std::uint32_t min CXX_DEFAULT_VALUE(0);

  bool operator==(SetMinTTLResponseActionConfiguration const &) const noexcept;
  bool operator!=(SetMinTTLResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetMinTTLResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetReducedTTLResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetReducedTTLResponseActionConfiguration
struct SetReducedTTLResponseActionConfiguration final {
  ::rust::String name;
  ::std::uint8_t percentage CXX_DEFAULT_VALUE(0);

  bool operator==(SetReducedTTLResponseActionConfiguration const &) const noexcept;
  bool operator!=(SetReducedTTLResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetReducedTTLResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetSkipCacheResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetSkipCacheResponseActionConfiguration
struct SetSkipCacheResponseActionConfiguration final {
  ::rust::String name;

  bool operator==(SetSkipCacheResponseActionConfiguration const &) const noexcept;
  bool operator!=(SetSkipCacheResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetSkipCacheResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTagResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTagResponseActionConfiguration
struct SetTagResponseActionConfiguration final {
  ::rust::String name;
  ::rust::String tag;
  ::rust::String value;

  bool operator==(SetTagResponseActionConfiguration const &) const noexcept;
  bool operator!=(SetTagResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SetTagResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNMPTrapResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNMPTrapResponseActionConfiguration
struct SNMPTrapResponseActionConfiguration final {
  ::rust::String name;
  ::rust::String reason;

  bool operator==(SNMPTrapResponseActionConfiguration const &) const noexcept;
  bool operator!=(SNMPTrapResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNMPTrapResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCResponseActionConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCResponseActionConfiguration
struct TCResponseActionConfiguration final {
  ::rust::String name;

  bool operator==(TCResponseActionConfiguration const &) const noexcept;
  bool operator!=(TCResponseActionConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCResponseActionConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllSelectorConfiguration
struct AllSelectorConfiguration final {
  ::rust::String name;

  bool operator==(AllSelectorConfiguration const &) const noexcept;
  bool operator!=(AllSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AllSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AndSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AndSelectorConfiguration
struct AndSelectorConfiguration final {
  ::rust::String name;
  ::rust::Vec<::dnsdist::rust::settings::SharedDNSSelector> selectors;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$AndSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ByNameSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ByNameSelectorConfiguration
struct ByNameSelectorConfiguration final {
  ::rust::String name;
  ::rust::String selector_name;

  bool operator==(ByNameSelectorConfiguration const &) const noexcept;
  bool operator!=(ByNameSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ByNameSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DNSSECSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DNSSECSelectorConfiguration
struct DNSSECSelectorConfiguration final {
  ::rust::String name;

  bool operator==(DNSSECSelectorConfiguration const &) const noexcept;
  bool operator!=(DNSSECSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DNSSECSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DSTPortSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DSTPortSelectorConfiguration
struct DSTPortSelectorConfiguration final {
  ::rust::String name;
  ::std::uint16_t port CXX_DEFAULT_VALUE(0);

  bool operator==(DSTPortSelectorConfiguration const &) const noexcept;
  bool operator!=(DSTPortSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DSTPortSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EDNSOptionSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EDNSOptionSelectorConfiguration
struct EDNSOptionSelectorConfiguration final {
  ::rust::String name;
  ::std::uint16_t option_code CXX_DEFAULT_VALUE(0);

  bool operator==(EDNSOptionSelectorConfiguration const &) const noexcept;
  bool operator!=(EDNSOptionSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EDNSOptionSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EDNSVersionSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EDNSVersionSelectorConfiguration
struct EDNSVersionSelectorConfiguration final {
  ::rust::String name;
  ::std::uint8_t version CXX_DEFAULT_VALUE(0);

  bool operator==(EDNSVersionSelectorConfiguration const &) const noexcept;
  bool operator!=(EDNSVersionSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EDNSVersionSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ERCodeSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ERCodeSelectorConfiguration
struct ERCodeSelectorConfiguration final {
  ::rust::String name;
  ::rust::String rcode;

  bool operator==(ERCodeSelectorConfiguration const &) const noexcept;
  bool operator!=(ERCodeSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ERCodeSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPHeaderSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPHeaderSelectorConfiguration
struct HTTPHeaderSelectorConfiguration final {
  ::rust::String name;
  ::rust::String header;
  ::rust::String expression;

  bool operator==(HTTPHeaderSelectorConfiguration const &) const noexcept;
  bool operator!=(HTTPHeaderSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPHeaderSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPPathSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPPathSelectorConfiguration
struct HTTPPathSelectorConfiguration final {
  ::rust::String name;
  ::rust::String path;

  bool operator==(HTTPPathSelectorConfiguration const &) const noexcept;
  bool operator!=(HTTPPathSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPPathSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPPathRegexSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPPathRegexSelectorConfiguration
struct HTTPPathRegexSelectorConfiguration final {
  ::rust::String name;
  ::rust::String expression;

  bool operator==(HTTPPathRegexSelectorConfiguration const &) const noexcept;
  bool operator!=(HTTPPathRegexSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HTTPPathRegexSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreLookupSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreLookupSelectorConfiguration
struct KeyValueStoreLookupSelectorConfiguration final {
  ::rust::String name;
  ::rust::String kvs_name;
  ::rust::String lookup_key_name;

  bool operator==(KeyValueStoreLookupSelectorConfiguration const &) const noexcept;
  bool operator!=(KeyValueStoreLookupSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreLookupSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreRangeLookupSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreRangeLookupSelectorConfiguration
struct KeyValueStoreRangeLookupSelectorConfiguration final {
  ::rust::String name;
  ::rust::String kvs_name;
  ::rust::String lookup_key_name;

  bool operator==(KeyValueStoreRangeLookupSelectorConfiguration const &) const noexcept;
  bool operator!=(KeyValueStoreRangeLookupSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoreRangeLookupSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaSelectorConfiguration
struct LuaSelectorConfiguration final {
  ::rust::String name;
  ::rust::String function_name;
  ::rust::String function_code;
  ::rust::String function_file;

  bool operator==(LuaSelectorConfiguration const &) const noexcept;
  bool operator!=(LuaSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFISelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFISelectorConfiguration
struct LuaFFISelectorConfiguration final {
  ::rust::String name;
  ::rust::String function_name;
  ::rust::String function_code;
  ::rust::String function_file;

  bool operator==(LuaFFISelectorConfiguration const &) const noexcept;
  bool operator!=(LuaFFISelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFISelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadSelectorConfiguration
struct LuaFFIPerThreadSelectorConfiguration final {
  ::rust::String name;
  ::rust::String code;

  bool operator==(LuaFFIPerThreadSelectorConfiguration const &) const noexcept;
  bool operator!=(LuaFFIPerThreadSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LuaFFIPerThreadSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MaxQPSSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MaxQPSSelectorConfiguration
struct MaxQPSSelectorConfiguration final {
  ::rust::String name;
  ::std::uint32_t qps CXX_DEFAULT_VALUE(0);
  ::std::uint32_t burst CXX_DEFAULT_VALUE(0);

  bool operator==(MaxQPSSelectorConfiguration const &) const noexcept;
  bool operator!=(MaxQPSSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MaxQPSSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MaxQPSIPSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MaxQPSIPSelectorConfiguration
struct MaxQPSIPSelectorConfiguration final {
  ::rust::String name;
  ::std::uint32_t qps CXX_DEFAULT_VALUE(0);
  ::std::uint8_t ipv4_mask CXX_DEFAULT_VALUE(0);
  ::std::uint8_t ipv6_mask CXX_DEFAULT_VALUE(0);
  ::std::uint32_t burst CXX_DEFAULT_VALUE(0);
  ::std::uint32_t expiration CXX_DEFAULT_VALUE(0);
  ::std::uint32_t cleanup_delay CXX_DEFAULT_VALUE(0);
  ::std::uint32_t scan_fraction CXX_DEFAULT_VALUE(0);
  ::std::uint32_t shards CXX_DEFAULT_VALUE(0);

  bool operator==(MaxQPSIPSelectorConfiguration const &) const noexcept;
  bool operator!=(MaxQPSIPSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MaxQPSIPSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NetmaskGroupSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NetmaskGroupSelectorConfiguration
struct NetmaskGroupSelectorConfiguration final {
  ::rust::String name;
  ::rust::String netmask_group_name;
  ::rust::Vec<::rust::String> netmasks;
  bool source CXX_DEFAULT_VALUE(false);
  bool quiet CXX_DEFAULT_VALUE(false);

  bool operator==(NetmaskGroupSelectorConfiguration const &) const noexcept;
  bool operator!=(NetmaskGroupSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NetmaskGroupSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NotSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NotSelectorConfiguration
struct NotSelectorConfiguration final {
  ::rust::String name;
  ::dnsdist::rust::settings::SharedDNSSelector selector;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NotSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OpcodeSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OpcodeSelectorConfiguration
struct OpcodeSelectorConfiguration final {
  ::rust::String name;
  ::rust::String code;

  bool operator==(OpcodeSelectorConfiguration const &) const noexcept;
  bool operator!=(OpcodeSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OpcodeSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OrSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OrSelectorConfiguration
struct OrSelectorConfiguration final {
  ::rust::String name;
  ::rust::Vec<::dnsdist::rust::settings::SharedDNSSelector> selectors;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OrSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PayloadSizeSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PayloadSizeSelectorConfiguration
struct PayloadSizeSelectorConfiguration final {
  ::rust::String name;
  ::rust::String comparison;
  ::std::uint16_t size CXX_DEFAULT_VALUE(0);

  bool operator==(PayloadSizeSelectorConfiguration const &) const noexcept;
  bool operator!=(PayloadSizeSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PayloadSizeSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolAvailableSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolAvailableSelectorConfiguration
struct PoolAvailableSelectorConfiguration final {
  ::rust::String name;
  ::rust::String pool;

  bool operator==(PoolAvailableSelectorConfiguration const &) const noexcept;
  bool operator!=(PoolAvailableSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolAvailableSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolOutstandingSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolOutstandingSelectorConfiguration
struct PoolOutstandingSelectorConfiguration final {
  ::rust::String name;
  ::rust::String pool;
  ::std::uint64_t max_outstanding CXX_DEFAULT_VALUE(0);

  bool operator==(PoolOutstandingSelectorConfiguration const &) const noexcept;
  bool operator!=(PoolOutstandingSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolOutstandingSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProbaSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProbaSelectorConfiguration
struct ProbaSelectorConfiguration final {
  ::rust::String name;
  double probability CXX_DEFAULT_VALUE(0);

  bool operator==(ProbaSelectorConfiguration const &) const noexcept;
  bool operator!=(ProbaSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProbaSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolValueSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolValueSelectorConfiguration
struct ProxyProtocolValueSelectorConfiguration final {
  ::rust::String name;
  ::std::uint8_t option_type CXX_DEFAULT_VALUE(0);
  ::rust::String option_value;

  bool operator==(ProxyProtocolValueSelectorConfiguration const &) const noexcept;
  bool operator!=(ProxyProtocolValueSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolValueSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QClassSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QClassSelectorConfiguration
struct QClassSelectorConfiguration final {
  ::rust::String name;
  ::rust::String qclass;
  ::std::uint16_t numeric_value CXX_DEFAULT_VALUE(0);

  bool operator==(QClassSelectorConfiguration const &) const noexcept;
  bool operator!=(QClassSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QClassSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSelectorConfiguration
struct QNameSelectorConfiguration final {
  ::rust::String name;
  ::rust::String qname;

  bool operator==(QNameSelectorConfiguration const &) const noexcept;
  bool operator!=(QNameSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameLabelsCountSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameLabelsCountSelectorConfiguration
struct QNameLabelsCountSelectorConfiguration final {
  ::rust::String name;
  ::std::uint16_t min_labels_count CXX_DEFAULT_VALUE(0);
  ::std::uint16_t max_labels_count CXX_DEFAULT_VALUE(0);

  bool operator==(QNameLabelsCountSelectorConfiguration const &) const noexcept;
  bool operator!=(QNameLabelsCountSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameLabelsCountSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSetSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSetSelectorConfiguration
struct QNameSetSelectorConfiguration final {
  ::rust::String name;
  ::rust::Vec<::rust::String> qnames;

  bool operator==(QNameSetSelectorConfiguration const &) const noexcept;
  bool operator!=(QNameSetSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSetSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSuffixSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSuffixSelectorConfiguration
struct QNameSuffixSelectorConfiguration final {
  ::rust::String name;
  ::rust::Vec<::rust::String> suffixes;
  bool quiet CXX_DEFAULT_VALUE(false);

  bool operator==(QNameSuffixSelectorConfiguration const &) const noexcept;
  bool operator!=(QNameSuffixSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameSuffixSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameWireLengthSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameWireLengthSelectorConfiguration
struct QNameWireLengthSelectorConfiguration final {
  ::rust::String name;
  ::std::uint16_t min CXX_DEFAULT_VALUE(0);
  ::std::uint16_t max CXX_DEFAULT_VALUE(0);

  bool operator==(QNameWireLengthSelectorConfiguration const &) const noexcept;
  bool operator!=(QNameWireLengthSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QNameWireLengthSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QTypeSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QTypeSelectorConfiguration
struct QTypeSelectorConfiguration final {
  ::rust::String name;
  ::rust::String qtype;
  ::std::uint16_t numeric_value CXX_DEFAULT_VALUE(0);

  bool operator==(QTypeSelectorConfiguration const &) const noexcept;
  bool operator!=(QTypeSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QTypeSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RCodeSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RCodeSelectorConfiguration
struct RCodeSelectorConfiguration final {
  ::rust::String name;
  ::rust::String rcode;

  bool operator==(RCodeSelectorConfiguration const &) const noexcept;
  bool operator!=(RCodeSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RCodeSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RDSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RDSelectorConfiguration
struct RDSelectorConfiguration final {
  ::rust::String name;

  bool operator==(RDSelectorConfiguration const &) const noexcept;
  bool operator!=(RDSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RDSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RE2SelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RE2SelectorConfiguration
struct RE2SelectorConfiguration final {
  ::rust::String name;
  ::rust::String expression;

  bool operator==(RE2SelectorConfiguration const &) const noexcept;
  bool operator!=(RE2SelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RE2SelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RecordsCountSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RecordsCountSelectorConfiguration
struct RecordsCountSelectorConfiguration final {
  ::rust::String name;
  ::std::uint8_t section CXX_DEFAULT_VALUE(0);
  ::std::uint16_t minimum CXX_DEFAULT_VALUE(0);
  ::std::uint16_t maximum CXX_DEFAULT_VALUE(0);

  bool operator==(RecordsCountSelectorConfiguration const &) const noexcept;
  bool operator!=(RecordsCountSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RecordsCountSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RecordsTypeCountSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RecordsTypeCountSelectorConfiguration
struct RecordsTypeCountSelectorConfiguration final {
  ::rust::String name;
  ::std::uint8_t section CXX_DEFAULT_VALUE(0);
  ::std::uint16_t record_type CXX_DEFAULT_VALUE(0);
  ::std::uint16_t minimum CXX_DEFAULT_VALUE(0);
  ::std::uint16_t maximum CXX_DEFAULT_VALUE(0);

  bool operator==(RecordsTypeCountSelectorConfiguration const &) const noexcept;
  bool operator!=(RecordsTypeCountSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RecordsTypeCountSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RegexSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RegexSelectorConfiguration
struct RegexSelectorConfiguration final {
  ::rust::String name;
  ::rust::String expression;

  bool operator==(RegexSelectorConfiguration const &) const noexcept;
  bool operator!=(RegexSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RegexSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNISelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNISelectorConfiguration
struct SNISelectorConfiguration final {
  ::rust::String name;
  ::rust::String server_name;

  bool operator==(SNISelectorConfiguration const &) const noexcept;
  bool operator!=(SNISelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SNISelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TagSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TagSelectorConfiguration
struct TagSelectorConfiguration final {
  ::rust::String name;
  ::rust::String tag;
  ::rust::String value;
  bool empty_as_wildcard CXX_DEFAULT_VALUE(false);

  bool operator==(TagSelectorConfiguration const &) const noexcept;
  bool operator!=(TagSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TagSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCPSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCPSelectorConfiguration
struct TCPSelectorConfiguration final {
  ::rust::String name;
  bool tcp CXX_DEFAULT_VALUE(false);

  bool operator==(TCPSelectorConfiguration const &) const noexcept;
  bool operator!=(TCPSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TCPSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TrailingDataSelectorConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TrailingDataSelectorConfiguration
struct TrailingDataSelectorConfiguration final {
  ::rust::String name;

  bool operator==(TrailingDataSelectorConfiguration const &) const noexcept;
  bool operator!=(TrailingDataSelectorConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TrailingDataSelectorConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CacheSettingsConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CacheSettingsConfiguration
struct CacheSettingsConfiguration final {
  ::std::uint32_t stale_entries_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint16_t cleaning_delay CXX_DEFAULT_VALUE(0);
  ::std::uint16_t cleaning_percentage CXX_DEFAULT_VALUE(0);

  bool operator==(CacheSettingsConfiguration const &) const noexcept;
  bool operator!=(CacheSettingsConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CacheSettingsConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ConsoleConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ConsoleConfiguration
struct ConsoleConfiguration final {
  ::rust::String listen_address;
  ::rust::String key;
  ::rust::Vec<::rust::String> acl;
  ::std::uint32_t maximum_output_size CXX_DEFAULT_VALUE(0);
  bool log_connections CXX_DEFAULT_VALUE(false);
  ::std::uint64_t max_concurrent_connections CXX_DEFAULT_VALUE(0);

  bool operator==(ConsoleConfiguration const &) const noexcept;
  bool operator!=(ConsoleConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ConsoleConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRulesSettingsConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRulesSettingsConfiguration
struct DynamicRulesSettingsConfiguration final {
  ::std::uint64_t purge_interval CXX_DEFAULT_VALUE(0);
  ::rust::String default_action;

  bool operator==(DynamicRulesSettingsConfiguration const &) const noexcept;
  bool operator!=(DynamicRulesSettingsConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRulesSettingsConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EbpfMapConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EbpfMapConfiguration
struct EbpfMapConfiguration final {
  ::std::uint32_t max_entries CXX_DEFAULT_VALUE(0);
  ::rust::String pinned_path;

  bool operator==(EbpfMapConfiguration const &) const noexcept;
  bool operator!=(EbpfMapConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EbpfMapConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EbpfConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EbpfConfiguration
struct EbpfConfiguration final {
  ::dnsdist::rust::settings::EbpfMapConfiguration ipv4;
  ::dnsdist::rust::settings::EbpfMapConfiguration ipv6;
  ::dnsdist::rust::settings::EbpfMapConfiguration cidr_ipv4;
  ::dnsdist::rust::settings::EbpfMapConfiguration cidr_ipv6;
  ::dnsdist::rust::settings::EbpfMapConfiguration qnames;
  bool external CXX_DEFAULT_VALUE(false);

  bool operator==(EbpfConfiguration const &) const noexcept;
  bool operator!=(EbpfConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EbpfConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EdnsClientSubnetConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EdnsClientSubnetConfiguration
struct EdnsClientSubnetConfiguration final {
  bool override_existing CXX_DEFAULT_VALUE(false);
  ::std::uint8_t source_prefix_v4 CXX_DEFAULT_VALUE(0);
  ::std::uint8_t source_prefix_v6 CXX_DEFAULT_VALUE(0);

  bool operator==(EdnsClientSubnetConfiguration const &) const noexcept;
  bool operator!=(EdnsClientSubnetConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$EdnsClientSubnetConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$GeneralConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$GeneralConfiguration
struct GeneralConfiguration final {
  ::std::uint16_t edns_udp_payload_size_self_generated_answers CXX_DEFAULT_VALUE(0);
  bool add_edns_to_self_generated_answers CXX_DEFAULT_VALUE(false);
  bool truncate_tc_answers CXX_DEFAULT_VALUE(false);
  bool fixup_case CXX_DEFAULT_VALUE(false);
  bool allow_empty_responses CXX_DEFAULT_VALUE(false);
  bool drop_empty_queries CXX_DEFAULT_VALUE(false);
  ::rust::Vec<::rust::String> capabilities_to_retain;

  bool operator==(GeneralConfiguration const &) const noexcept;
  bool operator!=(GeneralConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$GeneralConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeysConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeysConfiguration
struct KvsLookupKeysConfiguration final {
  ::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySourceIpConfiguration> source_ip_keys;
  ::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyQnameConfiguration> qname_keys;
  ::rust::Vec<::dnsdist::rust::settings::KvsLookupKeySuffixConfiguration> suffix_keys;
  ::rust::Vec<::dnsdist::rust::settings::KvsLookupKeyTagConfiguration> tag_keys;

  bool operator==(KvsLookupKeysConfiguration const &) const noexcept;
  bool operator!=(KvsLookupKeysConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeysConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoresConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoresConfiguration
struct KeyValueStoresConfiguration final {
  ::rust::Vec<::dnsdist::rust::settings::LmdbKvStoreConfiguration> lmdb;
  ::rust::Vec<::dnsdist::rust::settings::CdbKvStoreConfiguration> cdb;
  ::dnsdist::rust::settings::KvsLookupKeysConfiguration lookup_keys;

  bool operator==(KeyValueStoresConfiguration const &) const noexcept;
  bool operator!=(KeyValueStoresConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KeyValueStoresConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LoadBalancingPoliciesConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LoadBalancingPoliciesConfiguration
struct LoadBalancingPoliciesConfiguration final {
  ::rust::String default_policy;
  bool servfail_on_no_server CXX_DEFAULT_VALUE(false);
  bool round_robin_servfail_on_no_server CXX_DEFAULT_VALUE(false);
  double weighted_balancing_factor CXX_DEFAULT_VALUE(0);
  double consistent_hashing_balancing_factor CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::dnsdist::rust::settings::CustomLoadBalancingPolicyConfiguration> custom_policies;
  ::std::uint32_t hash_perturbation CXX_DEFAULT_VALUE(0);

  bool operator==(LoadBalancingPoliciesConfiguration const &) const noexcept;
  bool operator!=(LoadBalancingPoliciesConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LoadBalancingPoliciesConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$StructuredLoggingConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$StructuredLoggingConfiguration
struct StructuredLoggingConfiguration final {
  bool enabled CXX_DEFAULT_VALUE(false);
  ::rust::String level_prefix;
  ::rust::String time_format;

  bool operator==(StructuredLoggingConfiguration const &) const noexcept;
  bool operator!=(StructuredLoggingConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$StructuredLoggingConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LoggingConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LoggingConfiguration
struct LoggingConfiguration final {
  bool verbose CXX_DEFAULT_VALUE(false);
  bool verbose_health_checks CXX_DEFAULT_VALUE(false);
  ::rust::String verbose_log_destination;
  ::rust::String syslog_facility;
  ::dnsdist::rust::settings::StructuredLoggingConfiguration structured;

  bool operator==(LoggingConfiguration const &) const noexcept;
  bool operator!=(LoggingConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LoggingConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MetricsConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MetricsConfiguration
struct MetricsConfiguration final {
  ::rust::Vec<::dnsdist::rust::settings::CarbonConfiguration> carbon;

  bool operator==(MetricsConfiguration const &) const noexcept;
  bool operator!=(MetricsConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$MetricsConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolConfiguration
struct ProxyProtocolConfiguration final {
  ::rust::Vec<::rust::String> acl;
  ::std::uint32_t maximum_payload_size CXX_DEFAULT_VALUE(0);
  bool apply_acl_to_proxied_clients CXX_DEFAULT_VALUE(false);

  bool operator==(ProxyProtocolConfiguration const &) const noexcept;
  bool operator!=(ProxyProtocolConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QueryCountConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QueryCountConfiguration
struct QueryCountConfiguration final {
  bool enabled CXX_DEFAULT_VALUE(false);
  ::rust::String filter_function_name;
  ::rust::String filter_function_code;
  ::rust::String filter_function_file;

  bool operator==(QueryCountConfiguration const &) const noexcept;
  bool operator!=(QueryCountConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QueryCountConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLoggingConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLoggingConfiguration
struct RemoteLoggingConfiguration final {
  ::rust::Vec<::dnsdist::rust::settings::ProtobufLoggerConfiguration> protobuf_loggers;
  ::rust::Vec<::dnsdist::rust::settings::DnstapLoggerConfiguration> dnstap_loggers;

  bool operator==(RemoteLoggingConfiguration const &) const noexcept;
  bool operator!=(RemoteLoggingConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RemoteLoggingConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RingBuffersConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RingBuffersConfiguration
struct RingBuffersConfiguration final {
  ::std::uint64_t size CXX_DEFAULT_VALUE(0);
  ::std::uint64_t shards CXX_DEFAULT_VALUE(0);
  ::std::uint64_t lock_retries CXX_DEFAULT_VALUE(0);
  bool record_queries CXX_DEFAULT_VALUE(false);
  bool record_responses CXX_DEFAULT_VALUE(false);

  bool operator==(RingBuffersConfiguration const &) const noexcept;
  bool operator!=(RingBuffersConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$RingBuffersConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SecurityPollingConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SecurityPollingConfiguration
struct SecurityPollingConfiguration final {
  ::std::uint32_t polling_interval CXX_DEFAULT_VALUE(0);
  ::rust::String suffix;

  bool operator==(SecurityPollingConfiguration const &) const noexcept;
  bool operator!=(SecurityPollingConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SecurityPollingConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SnmpConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SnmpConfiguration
struct SnmpConfiguration final {
  bool enabled CXX_DEFAULT_VALUE(false);
  bool traps_enabled CXX_DEFAULT_VALUE(false);
  ::rust::String daemon_socket;

  bool operator==(SnmpConfiguration const &) const noexcept;
  bool operator!=(SnmpConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$SnmpConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DohTuningConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DohTuningConfiguration
struct DohTuningConfiguration final {
  ::std::uint32_t outgoing_worker_threads CXX_DEFAULT_VALUE(0);
  ::std::uint64_t outgoing_max_idle_time CXX_DEFAULT_VALUE(0);
  ::std::uint64_t outgoing_cleanup_interval CXX_DEFAULT_VALUE(0);
  ::std::uint64_t outgoing_max_idle_connection_per_backend CXX_DEFAULT_VALUE(0);

  bool operator==(DohTuningConfiguration const &) const noexcept;
  bool operator!=(DohTuningConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DohTuningConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TcpTuningConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TcpTuningConfiguration
struct TcpTuningConfiguration final {
  ::std::uint32_t worker_threads CXX_DEFAULT_VALUE(0);
  ::std::uint32_t receive_timeout CXX_DEFAULT_VALUE(0);
  ::std::uint32_t send_timeout CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_queries_per_connection CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_connection_duration CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_queued_connections CXX_DEFAULT_VALUE(0);
  ::std::uint32_t internal_pipe_buffer_size CXX_DEFAULT_VALUE(0);
  ::std::uint64_t outgoing_max_idle_time CXX_DEFAULT_VALUE(0);
  ::std::uint64_t outgoing_cleanup_interval CXX_DEFAULT_VALUE(0);
  ::std::uint64_t outgoing_max_idle_connection_per_backend CXX_DEFAULT_VALUE(0);
  ::std::uint32_t max_connections_per_client CXX_DEFAULT_VALUE(0);
  ::rust::String fast_open_key;
  ::std::uint8_t connections_overload_threshold CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_connection_rate_per_client CXX_DEFAULT_VALUE(0);
  ::std::uint64_t connection_rate_interval CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_tls_new_session_rate_per_client CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_tls_resumed_session_rate_per_client CXX_DEFAULT_VALUE(0);
  ::std::uint32_t max_read_ios_per_query CXX_DEFAULT_VALUE(0);
  ::std::uint32_t ban_duration_for_exceeding_max_read_ios_per_query CXX_DEFAULT_VALUE(0);
  ::std::uint32_t ban_duration_for_exceeding_tcp_tls_rate CXX_DEFAULT_VALUE(0);
  ::std::uint8_t connections_mask_v4 CXX_DEFAULT_VALUE(0);
  ::std::uint8_t connections_mask_v6 CXX_DEFAULT_VALUE(0);
  ::std::uint8_t connections_mask_v4_port CXX_DEFAULT_VALUE(0);

  bool operator==(TcpTuningConfiguration const &) const noexcept;
  bool operator!=(TcpTuningConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TcpTuningConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TlsTuningConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TlsTuningConfiguration
struct TlsTuningConfiguration final {
  ::std::uint16_t outgoing_tickets_cache_cleanup_delay CXX_DEFAULT_VALUE(0);
  ::std::uint16_t outgoing_tickets_cache_validity CXX_DEFAULT_VALUE(0);
  ::std::uint16_t max_outgoing_tickets_per_backend CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> providers;
  ::rust::Vec<::dnsdist::rust::settings::TlsEngineConfiguration> engines;

  bool operator==(TlsTuningConfiguration const &) const noexcept;
  bool operator!=(TlsTuningConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TlsTuningConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$UdpTuningConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$UdpTuningConfiguration
struct UdpTuningConfiguration final {
  ::std::uint32_t messages_per_round CXX_DEFAULT_VALUE(0);
  ::std::uint32_t send_buffer_size CXX_DEFAULT_VALUE(0);
  ::std::uint32_t receive_buffer_size CXX_DEFAULT_VALUE(0);
  ::std::uint32_t max_outstanding_per_backend CXX_DEFAULT_VALUE(0);
  ::std::uint8_t timeout CXX_DEFAULT_VALUE(0);
  bool randomize_outgoing_sockets_to_backend CXX_DEFAULT_VALUE(false);
  bool randomize_ids_to_backend CXX_DEFAULT_VALUE(false);

  bool operator==(UdpTuningConfiguration const &) const noexcept;
  bool operator!=(UdpTuningConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$UdpTuningConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TuningConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TuningConfiguration
struct TuningConfiguration final {
  ::dnsdist::rust::settings::DohTuningConfiguration doh;
  ::dnsdist::rust::settings::TcpTuningConfiguration tcp;
  ::dnsdist::rust::settings::TlsTuningConfiguration tls;
  ::dnsdist::rust::settings::UdpTuningConfiguration udp;

  bool operator==(TuningConfiguration const &) const noexcept;
  bool operator!=(TuningConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TuningConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$WebserverConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$WebserverConfiguration
struct WebserverConfiguration final {
  ::rust::Vec<::rust::String> listen_addresses;
  ::rust::String password;
  ::rust::String api_key;
  ::rust::Vec<::rust::String> acl;
  bool api_requires_authentication CXX_DEFAULT_VALUE(false);
  bool stats_require_authentication CXX_DEFAULT_VALUE(false);
  bool dashboard_requires_authentication CXX_DEFAULT_VALUE(false);
  ::std::uint32_t max_concurrent_connections CXX_DEFAULT_VALUE(0);
  bool hash_plaintext_credentials CXX_DEFAULT_VALUE(false);
  ::rust::Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration> custom_headers;
  ::rust::String api_configuration_directory;
  bool api_read_write CXX_DEFAULT_VALUE(false);

  bool operator==(WebserverConfiguration const &) const noexcept;
  bool operator!=(WebserverConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$WebserverConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$GlobalConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$GlobalConfiguration
struct GlobalConfiguration final {
  ::rust::Vec<::rust::String> acl;
  ::rust::Vec<::dnsdist::rust::settings::BackendConfiguration> backends;
  ::rust::Vec<::dnsdist::rust::settings::BindConfiguration> binds;
  ::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> cache_hit_response_rules;
  ::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> cache_inserted_response_rules;
  ::rust::Vec<::dnsdist::rust::settings::QueryRuleConfiguration> cache_miss_rules;
  ::dnsdist::rust::settings::CacheSettingsConfiguration cache_settings;
  ::dnsdist::rust::settings::ConsoleConfiguration console;
  ::rust::Vec<::dnsdist::rust::settings::DynamicRulesConfiguration> dynamic_rules;
  ::dnsdist::rust::settings::DynamicRulesSettingsConfiguration dynamic_rules_settings;
  ::dnsdist::rust::settings::EbpfConfiguration ebpf;
  ::dnsdist::rust::settings::EdnsClientSubnetConfiguration edns_client_subnet;
  ::dnsdist::rust::settings::GeneralConfiguration general;
  ::dnsdist::rust::settings::KeyValueStoresConfiguration key_value_stores;
  ::dnsdist::rust::settings::LoadBalancingPoliciesConfiguration load_balancing_policies;
  ::dnsdist::rust::settings::LoggingConfiguration logging;
  ::dnsdist::rust::settings::MetricsConfiguration metrics;
  ::rust::Vec<::dnsdist::rust::settings::NetmaskGroupConfiguration> netmask_groups;
  ::rust::Vec<::dnsdist::rust::settings::PacketCacheConfiguration> packet_caches;
  ::rust::Vec<::dnsdist::rust::settings::PoolConfiguration> pools;
  ::dnsdist::rust::settings::ProxyProtocolConfiguration proxy_protocol;
  ::dnsdist::rust::settings::QueryCountConfiguration query_count;
  ::rust::Vec<::dnsdist::rust::settings::QueryRuleConfiguration> query_rules;
  ::dnsdist::rust::settings::RemoteLoggingConfiguration remote_logging;
  ::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> response_rules;
  ::dnsdist::rust::settings::RingBuffersConfiguration ring_buffers;
  ::dnsdist::rust::settings::SecurityPollingConfiguration security_polling;
  ::rust::Vec<::dnsdist::rust::settings::SharedDNSSelector> selectors;
  ::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> self_answered_response_rules;
  ::dnsdist::rust::settings::SnmpConfiguration snmp;
  ::dnsdist::rust::settings::TuningConfiguration tuning;
  ::dnsdist::rust::settings::WebserverConfiguration webserver;
  ::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> xfr_response_rules;
  ::rust::Vec<::dnsdist::rust::settings::XskConfiguration> xsk;
  ::rust::Vec<::dnsdist::rust::settings::ResponseRuleConfiguration> timeout_response_rules;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$GlobalConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CarbonConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CarbonConfiguration
struct CarbonConfiguration final {
  ::rust::String address;
  ::rust::String name;
  ::std::uint32_t interval CXX_DEFAULT_VALUE(0);
  ::rust::String name_space;
  ::rust::String instance;

  bool operator==(CarbonConfiguration const &) const noexcept;
  bool operator!=(CarbonConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CarbonConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProtobufLoggerConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProtobufLoggerConfiguration
struct ProtobufLoggerConfiguration final {
  ::rust::String name;
  ::rust::String address;
  ::std::uint16_t timeout CXX_DEFAULT_VALUE(0);
  ::std::uint64_t max_queued_entries CXX_DEFAULT_VALUE(0);
  ::std::uint8_t reconnect_wait_time CXX_DEFAULT_VALUE(0);
  ::std::uint64_t connection_count CXX_DEFAULT_VALUE(0);

  bool operator==(ProtobufLoggerConfiguration const &) const noexcept;
  bool operator!=(ProtobufLoggerConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProtobufLoggerConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLoggerConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLoggerConfiguration
struct DnstapLoggerConfiguration final {
  ::rust::String name;
  ::rust::String transport;
  ::rust::String address;
  ::std::uint64_t buffer_hint CXX_DEFAULT_VALUE(0);
  ::std::uint64_t flush_timeout CXX_DEFAULT_VALUE(0);
  ::std::uint64_t input_queue_size CXX_DEFAULT_VALUE(0);
  ::std::uint64_t output_queue_size CXX_DEFAULT_VALUE(0);
  ::std::uint64_t queue_notify_threshold CXX_DEFAULT_VALUE(0);
  ::std::uint64_t reopen_interval CXX_DEFAULT_VALUE(0);
  ::std::uint64_t connection_count CXX_DEFAULT_VALUE(0);

  bool operator==(DnstapLoggerConfiguration const &) const noexcept;
  bool operator!=(DnstapLoggerConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DnstapLoggerConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProtoBufMetaConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProtoBufMetaConfiguration
struct ProtoBufMetaConfiguration final {
  ::rust::String key;
  ::rust::String value;

  bool operator==(ProtoBufMetaConfiguration const &) const noexcept;
  bool operator!=(ProtoBufMetaConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProtoBufMetaConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LmdbKvStoreConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LmdbKvStoreConfiguration
struct LmdbKvStoreConfiguration final {
  ::rust::String name;
  ::rust::String file_name;
  ::rust::String database_name;
  bool no_lock CXX_DEFAULT_VALUE(false);

  bool operator==(LmdbKvStoreConfiguration const &) const noexcept;
  bool operator!=(LmdbKvStoreConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LmdbKvStoreConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CdbKvStoreConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CdbKvStoreConfiguration
struct CdbKvStoreConfiguration final {
  ::rust::String name;
  ::rust::String file_name;
  ::std::uint32_t refresh_delay CXX_DEFAULT_VALUE(0);

  bool operator==(CdbKvStoreConfiguration const &) const noexcept;
  bool operator!=(CdbKvStoreConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CdbKvStoreConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration
struct KvsLookupKeySourceIpConfiguration final {
  ::rust::String name;
  ::std::uint8_t v4_mask CXX_DEFAULT_VALUE(0);
  ::std::uint8_t v6_mask CXX_DEFAULT_VALUE(0);
  bool include_port CXX_DEFAULT_VALUE(false);

  bool operator==(KvsLookupKeySourceIpConfiguration const &) const noexcept;
  bool operator!=(KvsLookupKeySourceIpConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeySourceIpConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeyQnameConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeyQnameConfiguration
struct KvsLookupKeyQnameConfiguration final {
  ::rust::String name;
  bool wire_format CXX_DEFAULT_VALUE(false);

  bool operator==(KvsLookupKeyQnameConfiguration const &) const noexcept;
  bool operator!=(KvsLookupKeyQnameConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeyQnameConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeySuffixConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeySuffixConfiguration
struct KvsLookupKeySuffixConfiguration final {
  ::rust::String name;
  ::std::uint16_t minimum_labels CXX_DEFAULT_VALUE(0);
  bool wire_format CXX_DEFAULT_VALUE(false);

  bool operator==(KvsLookupKeySuffixConfiguration const &) const noexcept;
  bool operator!=(KvsLookupKeySuffixConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeySuffixConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeyTagConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeyTagConfiguration
struct KvsLookupKeyTagConfiguration final {
  ::rust::String name;
  ::rust::String tag;

  bool operator==(KvsLookupKeyTagConfiguration const &) const noexcept;
  bool operator!=(KvsLookupKeyTagConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$KvsLookupKeyTagConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRuleConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRuleConfiguration
struct DynamicRuleConfiguration final {
  ::rust::String rule_type;
  ::std::uint32_t seconds CXX_DEFAULT_VALUE(0);
  ::std::uint32_t action_duration CXX_DEFAULT_VALUE(0);
  ::rust::String comment;
  ::std::uint32_t rate CXX_DEFAULT_VALUE(0);
  double ratio CXX_DEFAULT_VALUE(0);
  ::rust::String action;
  ::std::uint32_t warning_rate CXX_DEFAULT_VALUE(0);
  double warning_ratio CXX_DEFAULT_VALUE(0);
  ::rust::String tag_name;
  ::rust::String tag_value;
  ::rust::String visitor_function_name;
  ::rust::String visitor_function_code;
  ::rust::String visitor_function_file;
  ::rust::String rcode;
  ::rust::String qtype;
  ::std::uint32_t minimum_number_of_responses CXX_DEFAULT_VALUE(0);
  double minimum_global_cache_hit_ratio CXX_DEFAULT_VALUE(0);

  bool operator==(DynamicRuleConfiguration const &) const noexcept;
  bool operator!=(DynamicRuleConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRuleConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRulesConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRulesConfiguration
struct DynamicRulesConfiguration final {
  ::rust::String name;
  ::std::uint8_t mask_ipv4 CXX_DEFAULT_VALUE(0);
  ::std::uint8_t mask_ipv6 CXX_DEFAULT_VALUE(0);
  ::std::uint8_t mask_port CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> exclude_ranges;
  ::rust::Vec<::rust::String> include_ranges;
  ::rust::Vec<::rust::String> exclude_domains;
  ::rust::Vec<::dnsdist::rust::settings::DynamicRuleConfiguration> rules;

  bool operator==(DynamicRulesConfiguration const &) const noexcept;
  bool operator!=(DynamicRulesConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$DynamicRulesConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration
struct IncomingTlsCertificateKeyPairConfiguration final {
  ::rust::String certificate;
  ::rust::String key;
  ::rust::String password;

  bool operator==(IncomingTlsCertificateKeyPairConfiguration const &) const noexcept;
  bool operator!=(IncomingTlsCertificateKeyPairConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTlsCertificateKeyPairConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTlsConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTlsConfiguration
struct IncomingTlsConfiguration final {
  ::rust::String provider;
  ::rust::Vec<::dnsdist::rust::settings::IncomingTlsCertificateKeyPairConfiguration> certificates;
  ::rust::String ciphers;
  ::rust::String ciphers_tls_13;
  ::rust::String minimum_version;
  ::rust::String ticket_key_file;
  ::std::uint32_t tickets_keys_rotation_delay CXX_DEFAULT_VALUE(0);
  ::std::uint32_t number_of_tickets_keys CXX_DEFAULT_VALUE(0);
  bool prefer_server_ciphers CXX_DEFAULT_VALUE(false);
  ::std::uint32_t session_timeout CXX_DEFAULT_VALUE(0);
  bool session_tickets CXX_DEFAULT_VALUE(false);
  ::std::uint32_t number_of_stored_sessions CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> ocsp_response_files;
  ::rust::String key_log_file;
  bool release_buffers CXX_DEFAULT_VALUE(false);
  bool enable_renegotiation CXX_DEFAULT_VALUE(false);
  bool async_mode CXX_DEFAULT_VALUE(false);
  bool ktls CXX_DEFAULT_VALUE(false);
  bool read_ahead CXX_DEFAULT_VALUE(false);
  bool proxy_protocol_outside_tls CXX_DEFAULT_VALUE(false);
  bool ignore_configuration_errors CXX_DEFAULT_VALUE(false);

  bool operator==(IncomingTlsConfiguration const &) const noexcept;
  bool operator!=(IncomingTlsConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTlsConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingTlsConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingTlsConfiguration
struct OutgoingTlsConfiguration final {
  ::rust::String provider;
  ::rust::String subject_name;
  ::rust::String subject_address;
  bool validate_certificate CXX_DEFAULT_VALUE(false);
  ::rust::String ca_store;
  ::rust::String ciphers;
  ::rust::String ciphers_tls_13;
  ::rust::String key_log_file;
  bool release_buffers CXX_DEFAULT_VALUE(false);
  bool enable_renegotiation CXX_DEFAULT_VALUE(false);
  bool ktls CXX_DEFAULT_VALUE(false);

  bool operator==(OutgoingTlsConfiguration const &) const noexcept;
  bool operator!=(OutgoingTlsConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingTlsConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration
struct HttpCustomResponseHeaderConfiguration final {
  ::rust::String key;
  ::rust::String value;

  bool operator==(HttpCustomResponseHeaderConfiguration const &) const noexcept;
  bool operator!=(HttpCustomResponseHeaderConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HttpCustomResponseHeaderConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HttpResponsesMapConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HttpResponsesMapConfiguration
struct HttpResponsesMapConfiguration final {
  ::rust::String expression;
  ::std::uint16_t status CXX_DEFAULT_VALUE(0);
  ::rust::String content;
  ::rust::Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration> headers;

  bool operator==(HttpResponsesMapConfiguration const &) const noexcept;
  bool operator!=(HttpResponsesMapConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HttpResponsesMapConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDohConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDohConfiguration
struct IncomingDohConfiguration final {
  ::rust::String provider;
  ::rust::Vec<::rust::String> paths;
  ::std::uint64_t idle_timeout CXX_DEFAULT_VALUE(0);
  ::rust::String server_tokens;
  bool send_cache_control_headers CXX_DEFAULT_VALUE(false);
  bool keep_incoming_headers CXX_DEFAULT_VALUE(false);
  bool trust_forwarded_for_header CXX_DEFAULT_VALUE(false);
  bool early_acl_drop CXX_DEFAULT_VALUE(false);
  bool exact_path_matching CXX_DEFAULT_VALUE(false);
  ::std::uint32_t internal_pipe_buffer_size CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::dnsdist::rust::settings::HttpCustomResponseHeaderConfiguration> custom_response_headers;
  ::rust::Vec<::dnsdist::rust::settings::HttpResponsesMapConfiguration> responses_map;

  bool operator==(IncomingDohConfiguration const &) const noexcept;
  bool operator!=(IncomingDohConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDohConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDoqConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDoqConfiguration
struct IncomingDoqConfiguration final {
  ::std::uint64_t max_concurrent_queries_per_connection CXX_DEFAULT_VALUE(0);

  bool operator==(IncomingDoqConfiguration const &) const noexcept;
  bool operator!=(IncomingDoqConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDoqConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingQuicConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingQuicConfiguration
struct IncomingQuicConfiguration final {
  ::std::uint64_t idle_timeout CXX_DEFAULT_VALUE(0);
  ::rust::String congestion_control_algorithm;
  ::std::uint32_t internal_pipe_buffer_size CXX_DEFAULT_VALUE(0);

  bool operator==(IncomingQuicConfiguration const &) const noexcept;
  bool operator!=(IncomingQuicConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingQuicConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration
struct IncomingDnscryptCertificateKeyPairConfiguration final {
  ::rust::String certificate;
  ::rust::String key;

  bool operator==(IncomingDnscryptCertificateKeyPairConfiguration const &) const noexcept;
  bool operator!=(IncomingDnscryptCertificateKeyPairConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDnscryptCertificateKeyPairConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDnscryptConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDnscryptConfiguration
struct IncomingDnscryptConfiguration final {
  ::rust::String provider_name;
  ::rust::Vec<::dnsdist::rust::settings::IncomingDnscryptCertificateKeyPairConfiguration> certificates;

  bool operator==(IncomingDnscryptConfiguration const &) const noexcept;
  bool operator!=(IncomingDnscryptConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingDnscryptConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingDohConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingDohConfiguration
struct OutgoingDohConfiguration final {
  ::rust::String path;
  bool add_x_forwarded_headers CXX_DEFAULT_VALUE(false);

  bool operator==(OutgoingDohConfiguration const &) const noexcept;
  bool operator!=(OutgoingDohConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingDohConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTcpConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTcpConfiguration
struct IncomingTcpConfiguration final {
  ::std::uint32_t max_in_flight_queries CXX_DEFAULT_VALUE(0);
  ::std::uint32_t listen_queue_size CXX_DEFAULT_VALUE(0);
  ::std::uint32_t fast_open_queue_size CXX_DEFAULT_VALUE(0);
  ::std::uint32_t max_concurrent_connections CXX_DEFAULT_VALUE(0);

  bool operator==(IncomingTcpConfiguration const &) const noexcept;
  bool operator!=(IncomingTcpConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$IncomingTcpConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$BindConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$BindConfiguration
struct BindConfiguration final {
  ::rust::String listen_address;
  bool reuseport CXX_DEFAULT_VALUE(false);
  ::rust::String protocol;
  ::std::uint32_t threads CXX_DEFAULT_VALUE(0);
  ::rust::String interface;
  ::rust::String cpus;
  bool enable_proxy_protocol CXX_DEFAULT_VALUE(false);
  ::dnsdist::rust::settings::IncomingTcpConfiguration tcp;
  ::dnsdist::rust::settings::IncomingTlsConfiguration tls;
  ::dnsdist::rust::settings::IncomingDohConfiguration doh;
  ::dnsdist::rust::settings::IncomingDoqConfiguration doq;
  ::dnsdist::rust::settings::IncomingQuicConfiguration quic;
  ::dnsdist::rust::settings::IncomingDnscryptConfiguration dnscrypt;
  ::rust::Vec<::rust::String> additional_addresses;
  ::rust::String xsk;

  bool operator==(BindConfiguration const &) const noexcept;
  bool operator!=(BindConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$BindConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingTcpConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingTcpConfiguration
struct OutgoingTcpConfiguration final {
  ::std::uint16_t retries CXX_DEFAULT_VALUE(0);
  ::std::uint16_t connect_timeout CXX_DEFAULT_VALUE(0);
  ::std::uint16_t send_timeout CXX_DEFAULT_VALUE(0);
  ::std::uint16_t receive_timeout CXX_DEFAULT_VALUE(0);
  bool fast_open CXX_DEFAULT_VALUE(false);

  bool operator==(OutgoingTcpConfiguration const &) const noexcept;
  bool operator!=(OutgoingTcpConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingTcpConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolValueConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolValueConfiguration
struct ProxyProtocolValueConfiguration final {
  ::std::uint8_t key CXX_DEFAULT_VALUE(0);
  ::rust::String value;

  bool operator==(ProxyProtocolValueConfiguration const &) const noexcept;
  bool operator!=(ProxyProtocolValueConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ProxyProtocolValueConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LazyHealthCheckConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LazyHealthCheckConfiguration
struct LazyHealthCheckConfiguration final {
  ::std::uint16_t interval CXX_DEFAULT_VALUE(0);
  ::std::uint16_t min_sample_count CXX_DEFAULT_VALUE(0);
  ::rust::String mode;
  ::std::uint16_t sample_size CXX_DEFAULT_VALUE(0);
  ::std::uint16_t threshold CXX_DEFAULT_VALUE(0);
  bool use_exponential_back_off CXX_DEFAULT_VALUE(false);
  ::std::uint16_t max_back_off CXX_DEFAULT_VALUE(0);

  bool operator==(LazyHealthCheckConfiguration const &) const noexcept;
  bool operator!=(LazyHealthCheckConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$LazyHealthCheckConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HealthCheckConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HealthCheckConfiguration
struct HealthCheckConfiguration final {
  ::rust::String mode;
  ::rust::String qname;
  ::rust::String qclass;
  ::rust::String qtype;
  ::rust::String function;
  ::rust::String lua;
  ::rust::String lua_file;
  ::std::uint16_t timeout CXX_DEFAULT_VALUE(0);
  bool set_cd CXX_DEFAULT_VALUE(false);
  ::std::uint8_t max_failures CXX_DEFAULT_VALUE(0);
  ::std::uint8_t rise CXX_DEFAULT_VALUE(0);
  ::std::uint32_t interval CXX_DEFAULT_VALUE(0);
  bool must_resolve CXX_DEFAULT_VALUE(false);
  bool use_tcp CXX_DEFAULT_VALUE(false);
  ::dnsdist::rust::settings::LazyHealthCheckConfiguration lazy;

  bool operator==(HealthCheckConfiguration const &) const noexcept;
  bool operator!=(HealthCheckConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$HealthCheckConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingAutoUpgradeConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingAutoUpgradeConfiguration
struct OutgoingAutoUpgradeConfiguration final {
  bool enabled CXX_DEFAULT_VALUE(false);
  ::std::uint32_t interval CXX_DEFAULT_VALUE(0);
  bool keep CXX_DEFAULT_VALUE(false);
  ::rust::String pool;
  ::std::uint8_t doh_key CXX_DEFAULT_VALUE(0);
  bool use_lazy_health_check CXX_DEFAULT_VALUE(false);

  bool operator==(OutgoingAutoUpgradeConfiguration const &) const noexcept;
  bool operator!=(OutgoingAutoUpgradeConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$OutgoingAutoUpgradeConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$BackendConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$BackendConfiguration
struct BackendConfiguration final {
  ::rust::String address;
  ::rust::String id;
  ::rust::String name;
  ::rust::String protocol;
  ::dnsdist::rust::settings::OutgoingTlsConfiguration tls;
  ::dnsdist::rust::settings::OutgoingDohConfiguration doh;
  bool use_client_subnet CXX_DEFAULT_VALUE(false);
  bool use_proxy_protocol CXX_DEFAULT_VALUE(false);
  ::std::uint32_t queries_per_second CXX_DEFAULT_VALUE(0);
  ::std::uint32_t order CXX_DEFAULT_VALUE(0);
  ::std::uint32_t weight CXX_DEFAULT_VALUE(0);
  ::std::uint8_t udp_timeout CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> pools;
  ::dnsdist::rust::settings::OutgoingTcpConfiguration tcp;
  bool ip_bind_addr_no_port CXX_DEFAULT_VALUE(false);
  ::dnsdist::rust::settings::HealthCheckConfiguration health_checks;
  ::rust::String source;
  ::std::uint32_t sockets CXX_DEFAULT_VALUE(0);
  bool disable_zero_scope CXX_DEFAULT_VALUE(false);
  bool reconnect_on_up CXX_DEFAULT_VALUE(false);
  ::std::uint32_t max_in_flight CXX_DEFAULT_VALUE(0);
  bool tcp_only CXX_DEFAULT_VALUE(false);
  ::dnsdist::rust::settings::OutgoingAutoUpgradeConfiguration auto_upgrade;
  ::std::uint32_t max_concurrent_tcp_connections CXX_DEFAULT_VALUE(0);
  bool proxy_protocol_advertise_tls CXX_DEFAULT_VALUE(false);
  ::rust::String mac_address;
  ::rust::String cpus;
  ::rust::String xsk;
  ::std::uint8_t dscp CXX_DEFAULT_VALUE(0);

  bool operator==(BackendConfiguration const &) const noexcept;
  bool operator!=(BackendConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$BackendConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TlsEngineConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TlsEngineConfiguration
struct TlsEngineConfiguration final {
  ::rust::String name;
  ::rust::String default_string;

  bool operator==(TlsEngineConfiguration const &) const noexcept;
  bool operator!=(TlsEngineConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$TlsEngineConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NetmaskGroupConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NetmaskGroupConfiguration
struct NetmaskGroupConfiguration final {
  ::rust::String name;
  ::rust::Vec<::rust::String> netmasks;

  bool operator==(NetmaskGroupConfiguration const &) const noexcept;
  bool operator!=(NetmaskGroupConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$NetmaskGroupConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PacketCacheConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PacketCacheConfiguration
struct PacketCacheConfiguration final {
  ::rust::String name;
  ::std::uint64_t size CXX_DEFAULT_VALUE(0);
  bool deferrable_insert_lock CXX_DEFAULT_VALUE(false);
  bool dont_age CXX_DEFAULT_VALUE(false);
  bool keep_stale_data CXX_DEFAULT_VALUE(false);
  ::std::uint32_t max_negative_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint32_t max_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint32_t min_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint32_t shards CXX_DEFAULT_VALUE(0);
  bool parse_ecs CXX_DEFAULT_VALUE(false);
  ::std::uint32_t stale_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint32_t temporary_failure_ttl CXX_DEFAULT_VALUE(0);
  ::std::uint32_t truncated_ttl CXX_DEFAULT_VALUE(0);
  bool cookie_hashing CXX_DEFAULT_VALUE(false);
  ::std::uint32_t maximum_entry_size CXX_DEFAULT_VALUE(0);
  ::rust::Vec<::rust::String> options_to_skip;
  ::rust::Vec<::std::uint16_t> payload_ranks;

  bool operator==(PacketCacheConfiguration const &) const noexcept;
  bool operator!=(PacketCacheConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PacketCacheConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolConfiguration
struct PoolConfiguration final {
  ::rust::String name;
  ::rust::String packet_cache;
  ::rust::String policy;

  bool operator==(PoolConfiguration const &) const noexcept;
  bool operator!=(PoolConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$PoolConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration
struct CustomLoadBalancingPolicyConfiguration final {
  ::rust::String name;
  ::rust::String function_name;
  ::rust::String function_code;
  ::rust::String function_file;
  bool ffi CXX_DEFAULT_VALUE(false);
  bool per_thread CXX_DEFAULT_VALUE(false);

  bool operator==(CustomLoadBalancingPolicyConfiguration const &) const noexcept;
  bool operator!=(CustomLoadBalancingPolicyConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$CustomLoadBalancingPolicyConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QueryRuleConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QueryRuleConfiguration
struct QueryRuleConfiguration final {
  ::rust::String name;
  ::rust::String uuid;
  ::dnsdist::rust::settings::SharedDNSSelector selector;
  ::dnsdist::rust::settings::SharedDNSAction action;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$QueryRuleConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ResponseRuleConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ResponseRuleConfiguration
struct ResponseRuleConfiguration final {
  ::rust::String name;
  ::rust::String uuid;
  ::dnsdist::rust::settings::SharedDNSSelector selector;
  ::dnsdist::rust::settings::SharedDNSResponseAction action;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$ResponseRuleConfiguration

#ifndef CXXBRIDGE1_STRUCT_dnsdist$rust$settings$XskConfiguration
#define CXXBRIDGE1_STRUCT_dnsdist$rust$settings$XskConfiguration
struct XskConfiguration final {
  ::rust::String name;
  ::rust::String interface;
  ::std::uint16_t queues CXX_DEFAULT_VALUE(0);
  ::std::uint32_t frames CXX_DEFAULT_VALUE(0);
  ::rust::String map_path;

  bool operator==(XskConfiguration const &) const noexcept;
  bool operator!=(XskConfiguration const &) const noexcept;
  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_dnsdist$rust$settings$XskConfiguration

::dnsdist::rust::settings::GlobalConfiguration from_yaml_string(::rust::Str str);
} // namespace settings
} // namespace rust
} // namespace dnsdist
