/*
 * Copyright (C) 2014-2026 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#include <algorithm> /* ::std::sort */
#include <QDir>
#include <QMenu>
#include <QMessageBox>
#include <QFileDialog>
#include <QPropertyAnimation>
#include <QSet>

#include "src/datovka_shared/io/filesystem.h"
#include "src/datovka_shared/log/log.h"
#include "src/datovka_shared/settings/prefs.h"
#include "src/datovka_shared/utility/strings.h" /* Utility::generateRandomString */
#include "src/datovka_shared/worker/pool.h"
#include "src/dimensions/dimensions.h"
#include "src/gui/datovka.h"
#include "src/gui/dlg_msg_box_detail.h"
#include "src/gui/dlg_timestamp_renewal.h"
#include "src/gui/dlg_view_zfo.h"
#include "src/gui/message_operations.h" /* MsgOrigin */
#include "src/global.h"
#include "src/io/isds_sessions.h"
#include "src/io/message_db_set_container.h"
#include "src/io/message_db_set.h"
#include "src/io/message_db.h"
#include "src/io/timestamp_db.h"
#include "src/isds/message_functions.h" /* Isds::guessFileRawType */
#include "src/model_interaction/account_interaction.h"
#include "src/settings/accounts.h"
#include "src/settings/prefs_specific.h"
#include "src/views/on_widget_hidden_filter.h"
#include "src/views/table_home_end_filter.h"
#include "src/views/table_key_press_filter.h"
#include "src/views/table_tab_ignore_filter.h"
#include "src/worker/message_emitter.h"
#include "src/worker/task_archive_document.h"
#include "ui_dlg_timestamp_renewal.h"

static const QString dlgName("timestamp_renewal");
static const QString tstTableName("timestamp_list");

#define PERIODIC_TIME_UPDATE_SECS 7200 /* 2 hours */

#define MSG_PROGRESS_COL MsgTstListingModel::COL_ERROR

DlgTimestampRenewal::DlgTimestampRenewal(const QList<AcntIdDb> &acntIdDbList,
    TimestampDb *timestampDb,
    DbContainer *msgDbs, IsdsSessions *sessions, MainWindow *mw,
    QWidget *parent, Qt::WindowFlags flags)
    : QDialog(parent, flags),
    m_ui(new (::std::nothrow) Ui::DlgTimestampRenewal),
    m_modelRowSelection(),
    m_timestampDb(timestampDb),
    m_msgDbs(msgDbs),
    m_sessions(sessions),
    m_mw(mw),
    m_updateTimer(this),
    m_sortProxyModel(this),
    m_progressProxyModel(this),
    m_progressDelegate(this),
    m_timestampTableModel(this),
    m_dfltFilerLineStyleSheet(SortFilterProxyModel::invalidFilterEditStyle),
    m_trackedTransactionMap(),
    m_externalTransaction(),
    m_tasks(),
    m_externalTask(Q_NULLPTR),
    m_speeds(),
    m_acntIdDbList(acntIdDbList),
    m_dfltSize()
{
	m_ui->setupUi(this);
	/* Tab order is defined in UI file. */

	m_dfltSize = this->size();
	{
		const QSize newSize = Dimensions::dialogueSize(this,
		    PrefsSpecific::dlgSize(*GlobInstcs::prefsPtr, dlgName),
		    m_dfltSize);
		if (newSize.isValid()) {
			this->resize(newSize);
		}
	}

	initTimestampWindow();

	/* Tracked messages view. */
	{
		TableKeyPressFilter *filter =
		    new (::std::nothrow) TableKeyPressFilter(m_ui->trackedMessagesView);
		if (filter != Q_NULLPTR) {
			filter->registerAction(Qt::CTRL | Qt::Key_F,
			    &showFilterLineViaFilter, this, true);
			m_ui->trackedMessagesView->installEventFilter(filter);
		}
	}

	/* Filter line. */
	/* Clear message filter line when line edit is hidden. */
	m_ui->messagesFilterHorizontalLayout->installEventFilter(
	    new (::std::nothrow) OnWidgetHiddenFilter(
	        &clearFilterLineViaFilter, this, m_ui->messageFilterLine));
	/* Hide message filter line when escape pressed in line edit. */
	{
		TableKeyPressFilter *filter =
		    new (::std::nothrow) TableKeyPressFilter(m_ui->messageFilterLine);
		if (Q_NULLPTR != filter) {
			filter->registerAction(Qt::Key_Escape,
			    &hideFilterLineViaFilter, this, true);
			m_ui->messageFilterLine->installEventFilter(filter);
		}
	}

	/* Load timestamps. */
	if ((Q_NULLPTR != m_timestampDb) && (Q_NULLPTR != m_msgDbs)) {
		QList<TstEntry> tstEntries;
		m_timestampDb->getMsgTimestampListing(tstEntries,
		    QDateTime(), QDateTime());

		/* Sort message identifiers according to user names. */
		QHash<AcntId, QList<MsgId> > acntToMsgIds;
		for (const TstEntry &tstEntry : tstEntries) {
			const AcntId acntId(tstEntry.username, (tstEntry.msgId.testEnv() == Isds::Type::BOOL_TRUE));
			if (Q_UNLIKELY(!acntId.isValid())) {
				continue;
			}
			acntToMsgIds[acntId].append(
			    MsgId(tstEntry.msgId.dmId(),
			        tstEntry.msgId.deliveryTime()));
		}

		/* Get message entries according to user name and message id. */
		QHash<AcntId, QHash<MsgId, MessageDb::SoughtMsg> > msgEntries;
		for (const AcntId &acntId : acntToMsgIds.keys()) {
			enum AccountInteraction::AccessStatus status = AccountInteraction::AS_OK;
			QString dbDir;
			QString namesStr;
			MessageDbSet *dbSet = AccountInteraction::accessDbSet(
			    acntId, status, dbDir, namesStr);

			QHash<MsgId, MessageDb::SoughtMsg> foundEntries;

			if (Q_UNLIKELY(Q_NULLPTR == dbSet)) {
				continue;
			}
			if (dbSet->msgsGetMsgDataFromIds(
			        acntToMsgIds.value(acntId),
			        foundEntries)) {
				msgEntries[acntId] = foundEntries;
			}

		}

		m_timestampTableModel.appendData(tstEntries, msgEntries);
	}

	connect(&m_updateTimer, SIGNAL(timeout()),
	    &m_timestampTableModel, SLOT(updateCurrentTime()));
	m_updateTimer.start(PERIODIC_TIME_UPDATE_SECS * 1000);
}

DlgTimestampRenewal::~DlgTimestampRenewal(void)
{
	PrefsSpecific::setDlgSize(*GlobInstcs::prefsPtr,
	    dlgName, this->size(), m_dfltSize);

	PrefsSpecific::setDlgTblRelColWidths(*GlobInstcs::prefsPtr,
	    dlgName, tstTableName,
	    Dimensions::relativeTableColumnWidths(m_ui->trackedMessagesView));
	PrefsSpecific::setDlgTblColSortOrder(*GlobInstcs::prefsPtr,
	    dlgName, tstTableName,
	    Dimensions::tableColumnSortOrder(m_ui->trackedMessagesView));

	delete m_ui;
}

void DlgTimestampRenewal::showEvent(QShowEvent *event)
{
	QDialog::showEvent(event);

	Dimensions::setRelativeTableColumnWidths(m_ui->trackedMessagesView,
	    PrefsSpecific::dlgTblRelColWidths(*GlobInstcs::prefsPtr,
	    dlgName, tstTableName));
	Dimensions::setTableColumnSortOrder(m_ui->trackedMessagesView,
	    PrefsSpecific::dlgTblColSortOrder(*GlobInstcs::prefsPtr,
	    dlgName, tstTableName));
}

/*!
 * @brief Convert enum MessageDb::MessageType to enum MessageDirection.
 *
 * @param[in]  type Message type.
 * @param[out] ok Set to true on success.
 * @return Message direction.
 */
static
enum MessageDirection messageType2messageDirection(
    enum MessageDb::MessageType type, bool *ok = Q_NULLPTR)
{
	enum MessageDirection direct = MSG_ALL;

	switch (type) {
	case MessageDb::TYPE_RECEIVED:
		direct = MSG_RECEIVED;
		break;
	case MessageDb::TYPE_SENT:
		direct = MSG_SENT;
		break;
	default:
		Q_ASSERT(0);
		if (ok != Q_NULLPTR) {
			*ok = false;
		}
		return direct;
		break;
	}

	if (ok != Q_NULLPTR) {
		*ok = true;
	}
	return direct;
}

/*!
 * @brief Construct exhaustive message selection list.
 *
 * @brief Attempts to connect to ISDS with respective accounts.
 *
 * @param[in] modelRowSelection Message selection status.
 * @param[in] model Table model.
 * @param[in,out] sessions Session container.
 * @param[in] mw Main application window.
 * @return Origin list of selected messages.
 */
static
QList<MsgOrigin> messageOriginList(const QList<int> &modelRowSelection,
    const MsgTstListingModel &model, IsdsSessions *sessions, MainWindow *mw)
{
	QList<MsgOrigin> originList;

	QList<int> orderedRowSelection;

	/* Sort selected rows according to expiration time, oldest fist. */
	{
		/*
		 * Maps expiration time to wows.
		 * There may be multiple rows with same expiration times.
		 */
		QMap<QDateTime, QSet<int> > timeToRow;

		/* Map expiration times to rows. */
		for (int row : modelRowSelection) {
			const TstEntry &tstEntry = model.tstEntry(row);
			timeToRow[tstEntry.tstExpiration].insert(row);
		}

		QList<QDateTime> keys = timeToRow.keys();
		/* Sort expiration times. */
		::std::sort(keys.begin(), keys.end());

		/* Gather rows, older (lower) expiration times first. */
		for (const QDateTime &key : keys) {
			/* Row order with equal expiration time is unspecified. */
			for (int row : timeToRow[key]) {
				orderedRowSelection.append(row);
			}
		}
	}

	for (int row : orderedRowSelection) {
		if (Q_UNLIKELY(row < 0)) {
			Q_ASSERT(0);
			continue;
		}
		const TstEntry &tstEntry = model.tstEntry(row);
		const MessageDb::SoughtMsg &msgEntry = model.msgEntry(row);

		const AcntId acntId(tstEntry.username, (tstEntry.msgId.testEnv() == Isds::Type::BOOL_TRUE));

		enum AccountInteraction::AccessStatus status = AccountInteraction::AS_OK;
		QString dbDir;
		QString namesStr;
		MessageDbSet *dbSet = AccountInteraction::accessDbSet(
		    acntId, status, dbDir, namesStr);

		AcntIdDb acntIdDb(acntId, dbSet);
		if ((Q_NULLPTR != sessions) && (Q_NULLPTR != mw)) {
			if (!sessions->isConnectedToIsds(acntIdDb.username())) {
				if (Q_UNLIKELY(!mw->connectToIsds(acntIdDb))) {
					/* Ignore inactive accounts. */
					continue;
				}
			}
		}

		originList.append(MsgOrigin(acntIdDb, msgEntry.mId,
		    msgEntry.isVodz, messageType2messageDirection(msgEntry.type),
		    msgEntry.isDownloaded));
	}

	return originList;
}

void DlgTimestampRenewal::done(int resultCode)
{
	/* Don't ask if global halt requested. */
	if ((!GlobInstcs::req_halt)
	    && ((!m_tasks.empty()) || (Q_NULLPTR != m_externalTask))) {

		int ret = DlgMsgBoxDetail::message(this, QMessageBox::Question,
		    tr("Close Time Stamp Renewal Dialogue?"),
		    tr("Do you want to close this dialogue?"),
		    tr("Closing this dialogue will abort all unfinished actions."), QString(),
		    QMessageBox::Yes | QMessageBox::No, QMessageBox::No);
		if (ret == QMessageBox::No) {
			/* Don't close the dialogue. */
			return;
		}
	}

	/* Ask all pending tasks to quit. */
	for (TaskArchiveIsdsDocument *task : m_tasks) {
		task->requestQuit();
	}
	if (Q_NULLPTR != m_externalTask) {
		m_externalTask->requestQuit();
	}

	/*
	 * Wait for tasks to finish and collect them if to global shutdown
	 * required.
	 *
	 * TODO -- A better way to dispose discarded tasks must be implemented.
	 *
	 * The use of QCoreApplication::processEvents() is discouraged by Qt
	 * documentation.
	 *
	 * Extend worker/pool interface to accommodate more generic approach.
	 * All tasks should have a transaction/stak identifier.
	 * All tasks should have a requestQuit() method.
	 * The pool could have a storage for finished tasks.
	 * Finished tasks should be acquired via a poll interface rather than
	 * remembering their pointers in the caller code.
	 * Uncollected tasks should be deleted after a period of time?
	 */
	if (!GlobInstcs::req_halt) {
		while ((!m_tasks.empty()) || (Q_NULLPTR != m_externalTask)) {
			QCoreApplication::processEvents(QEventLoop::AllEvents, 100);
		}
	}

	/* Close the dialogue. */
	QDialog::done(resultCode);
}

void DlgTimestampRenewal::renewTimestamps(void)
{
	const QString timeStr = QDateTime::currentDateTimeUtc().toString();

	m_ui->restampPushButton->setEnabled(false);
	m_ui->actionRestamp->setEnabled(false);
	m_ui->restampExternalPushButton->setEnabled(false);
	this->setCursor(Qt::BusyCursor);

	QList<MsgOrigin> originList = messageOriginList(m_modelRowSelection,
	    m_timestampTableModel, m_sessions, m_mw);
	if (Q_UNLIKELY(originList.isEmpty())) {
		updateActionActivation();
		this->setCursor(Qt::ArrowCursor);
	}

	for (const MsgOrigin &msgOrigin : originList) {
		const QString transactionId = QString("%1_%2_%3_%4")
		    .arg(msgOrigin.acntIdDb.username())
		    .arg(timeStr)
		    .arg(msgOrigin.msgId.dmId())
		    .arg(Utility::generateRandomString(6));

		m_timestampTableModel.updateError(msgOrigin.acntIdDb, msgOrigin.msgId, QString());

		TaskArchiveIsdsDocument *task = new (::std::nothrow) TaskArchiveIsdsDocument(
		    msgOrigin.acntIdDb, transactionId,
		    msgOrigin.msgId, msgOrigin.direction);
		if (Q_UNLIKELY(task == Q_NULLPTR)) {
			Q_ASSERT(0);
			break;
		}

		m_trackedTransactionMap[transactionId] = QPair<AcntId, MsgId>(msgOrigin.acntIdDb, msgOrigin.msgId);
		m_tasks[transactionId] = task;

		task->setAutoDelete(false);
		GlobInstcs::workPoolPtr->assignLo(task);

		/* Tasks are deleted inside watchDownloadProgressFinished(). */
	}

	/* Actions are activated inside watchDownloadProgressFinished(). */
}

static
QList<int> selectedSrcRowNums(const QAbstractItemView &view, int column,
   QAbstractProxyModel &proxyModel1, QAbstractProxyModel &proxyModel2)
{
	QList<int> rowList;

	QModelIndexList selIdxs = view.selectionModel()->selectedRows(column);

	/* Translate to underlying model. */
	for (const QModelIndex &idx : selIdxs) {
		QModelIndex srcIdx = proxyModel1.mapToSource(idx);
		if (Q_UNLIKELY(!srcIdx.isValid())) {
			Q_ASSERT(0);
			continue;
		}
		srcIdx = proxyModel2.mapToSource(srcIdx);
		if (Q_UNLIKELY(!srcIdx.isValid())) {
			Q_ASSERT(0);
			continue;
		}
		rowList.append(srcIdx.row());
	}

	return rowList;
}

void DlgTimestampRenewal::updateMessageSelection(const QItemSelection &selected,
    const QItemSelection &deselected)
{
	Q_UNUSED(selected);
	Q_UNUSED(deselected);

	m_modelRowSelection = selectedSrcRowNums(
	    *(m_ui->trackedMessagesView), MsgTstListingModel::COL_ACCOUNT_ID,
	    m_sortProxyModel, m_progressProxyModel);

	updateActionActivation();
}

void DlgTimestampRenewal::toggleTimestampTracking(bool enable)
{
	if (enable) {
		/* All are enabled by default. */
	} else {
		Json::MsgId2List msgId2List;

		for (int row : m_modelRowSelection) {
			const TstEntry &tstEntry =
			    m_timestampTableModel.tstEntry(row);

			msgId2List.append(tstEntry.msgId);
		}

		m_timestampDb->deleteIdentifications(msgId2List);
	}
}

void DlgTimestampRenewal::viewTrackedMessagesListContextMenu(const QPoint &point)
{
	QModelIndex srcIdx = m_sortProxyModel.mapToSource(
	    m_ui->trackedMessagesView->indexAt(point));
	srcIdx = m_progressProxyModel.mapToSource(srcIdx);

	QMenu *menu = new (::std::nothrow) QMenu(m_ui->trackedMessagesView);
	if (Q_UNLIKELY(menu == Q_NULLPTR)) {
		Q_ASSERT(0);
		return;
	}

	if (srcIdx.isValid()) {
		menu->addAction(m_ui->actionRestamp);
		menu->addSeparator();

		menu->addAction(m_ui->actionTrackTimestampExpiration);
	}

	menu->exec(QCursor::pos());
	menu->deleteLater();
}

void DlgTimestampRenewal::messageItemDoubleClicked(const QModelIndex &index)
{
	if (Q_UNLIKELY(!index.isValid())) {
		return;
	}

	QModelIndex srcIdx = m_sortProxyModel.mapToSource(index);
	if (Q_UNLIKELY(!srcIdx.isValid())) {
		Q_ASSERT(0);
		return;
	}
	srcIdx = m_progressProxyModel.mapToSource(srcIdx);
	if (Q_UNLIKELY(!srcIdx.isValid())) {
		Q_ASSERT(0);
		return;
	}

	const TstEntry &tstEntry = m_timestampTableModel.tstEntry(srcIdx.row());
	const MessageDb::SoughtMsg &msgEntry = m_timestampTableModel.msgEntry(srcIdx.row());

	const AcntId acntId(tstEntry.username, (tstEntry.msgId.testEnv() == Isds::Type::BOOL_TRUE));

	if (Q_UNLIKELY((!acntId.isValid()) || (!msgEntry.isValid()))) {
		return;
	}

	Q_EMIT focusSelectedMsg(acntId, tstEntry.msgId.dmId(),
	    MessageDbSet::yearFromDateTime(tstEntry.msgId.deliveryTime()),
	    (int)msgEntry.type);
	/* Don't close the dialogue. */
}

void DlgTimestampRenewal::filterMessages(const QString &text)
{
	debugSlotCall();

	/* Store style at first invocation. */
	if (m_dfltFilerLineStyleSheet == SortFilterProxyModel::invalidFilterEditStyle) {
		m_dfltFilerLineStyleSheet = m_ui->messageFilterLine->styleSheet();
	}

	/* The model is always associated to the proxy model. */

	m_sortProxyModel.setSortRole(MsgTstListingModel::ROLE_PROXYSORT);

#if (QT_VERSION >= QT_VERSION_CHECK(5, 15, 0))
	m_sortProxyModel.setFilterRegularExpression(
	    QRegularExpression(QRegularExpression::escape(text),
	        QRegularExpression::CaseInsensitiveOption));
#else /* < Qt-5.15 */
	m_sortProxyModel.setFilterRegExp(QRegExp(text,
	    Qt::CaseInsensitive, QRegExp::FixedString));
#endif /* >= Qt-5.15 */
	{
		QList<int> columnList;
		columnList.append(MsgTstListingModel::COL_ACCOUNT_ID);
		columnList.append(MsgTstListingModel::COL_MESSAGE_ID);
		columnList.append(MsgTstListingModel::COL_ANNOTATION);
		columnList.append(MsgTstListingModel::COL_EXPIRATION_TIME);
		columnList.append(MsgTstListingModel::COL_ERROR);
		m_sortProxyModel.setFilterKeyColumns(columnList);
	}

	/* Set filter field background colour. */
	if (text.isEmpty()) {
		m_ui->messageFilterLine->setStyleSheet(m_dfltFilerLineStyleSheet);
	} else if (m_sortProxyModel.rowCount() != 0) {
		m_ui->messageFilterLine->setStyleSheet(
		    SortFilterProxyModel::foundFilterEditStyle);
	} else {
		m_ui->messageFilterLine->setStyleSheet(
		    SortFilterProxyModel::notFoundFilterEditStyle);
	}
}

#define ANIMATION_DURATION 250

void DlgTimestampRenewal::showMessageFilter(void)
{
	if (!m_ui->messagesFilterHorizontalLayout->isVisible()) {
		m_ui->messagesFilterHorizontalLayout->setMaximumHeight(0);
		m_ui->messagesFilterHorizontalLayout->setVisible(true);

		QPropertyAnimation *animation =
		    new (::std::nothrow) QPropertyAnimation(
		        m_ui->messagesFilterHorizontalLayout, "maximumHeight");
		if (Q_NULLPTR != animation) {
			animation->setDuration(ANIMATION_DURATION);
			animation->setStartValue(0);
			animation->setEndValue(
			    m_ui->messagesFilterHorizontalLayout->sizeHint().height());

			animation->start(QAbstractAnimation::DeleteWhenStopped);
		}
	}

	m_ui->messageFilterLine->setFocus(Qt::OtherFocusReason);
}

void DlgTimestampRenewal::hideMessageFilter(void)
{
	{
		QPropertyAnimation *animation =
		    new (::std::nothrow) QPropertyAnimation(
		        m_ui->messagesFilterHorizontalLayout, "maximumHeight");
		if (Q_NULLPTR != animation) {
			animation->setDuration(ANIMATION_DURATION);
			animation->setStartValue(
			    m_ui->messagesFilterHorizontalLayout->height());
			animation->setEndValue(0);

			/* Hide line after animation finishes. */
			connect(animation, SIGNAL(finished()),
			    m_ui->messagesFilterHorizontalLayout, SLOT(hide()));

			animation->start(QAbstractAnimation::DeleteWhenStopped);
		}
	}
	m_ui->messagesFilterHorizontalLayout->setMaximumHeight(QWIDGETSIZE_MAX);

	m_ui->trackedMessagesView->setFocus(Qt::OtherFocusReason);
}

#undef ANIMATION_DURATION

void DlgTimestampRenewal::watchIdentificationsInserted(
    const Json::MsgId2List &msgIds, const AcntId &acntId)
{
	QList<TstEntry> tstEntries;
	QHash<AcntId, QList<MsgId> > acntToMsgIds;

	for (const Json::MsgId2 &msgId : msgIds) {
		tstEntries.append(TstEntry(msgId, acntId.username(), QDateTime()));
		acntToMsgIds[acntId].append(
		    MsgId(msgId.dmId(), msgId.deliveryTime()));
	}

	/* Get message entries according to user name and message id. */
	QHash<AcntId, QHash<MsgId, MessageDb::SoughtMsg> > msgEntries;
	for (const AcntId &acntId : acntToMsgIds.keys()) {
		enum AccountInteraction::AccessStatus status = AccountInteraction::AS_OK;
		QString dbDir;
		QString namesStr;
		MessageDbSet *dbSet = AccountInteraction::accessDbSet(
		    acntId, status, dbDir, namesStr);

		QHash<MsgId, MessageDb::SoughtMsg> foundEntries;

		if (Q_UNLIKELY(Q_NULLPTR == dbSet)) {
			continue;
		}
		if (dbSet->msgsGetMsgDataFromIds(
		        acntToMsgIds.value(acntId),
		        foundEntries)) {
			msgEntries[acntId] = foundEntries;
		}

	}

	m_timestampTableModel.appendData(tstEntries, msgEntries);
}

void DlgTimestampRenewal::watchIdentificationsDeleted(
    const Json::MsgId2List &msgIds)
{
	m_timestampTableModel.remove(msgIds);
}

void DlgTimestampRenewal::watchMsgTimestampDataUpdated(
    const TstValidityHash &values)
{
	m_timestampTableModel.updateExpiration(values);
}

/*!
 * @brief Get last directory where the external file was loaded from.
 */
static
QString loadArchivedZfoDir(const Prefs &prefs)
{
	QString val;
	prefs.strVal("global.path.load.archive_zfo", val);
	/* Has default value set. */
	return val;
}

/*!
 * @brief Store directory where the external file was loaded from.
 */
static
void setLoadArchivedZfoDir(Prefs &prefs, const QString &val)
{
	prefs.setStrVal("global.path.load.archive_zfo", val);
}

/*!
 * @brief Generate a target file name suggestion.
 *
 * @param[in] srcPath Source file path.
 * @return Altered file name in same location.
 */
static
QString generateTgtFilePath(const QString &srcPath)
{
	if (Q_UNLIKELY(srcPath.isEmpty())) {
		return QString();
	}

	const QFileInfo fi(srcPath);

	QString tgtPath = fi.absolutePath() + "/"
	    + fi.baseName() + "_" + DlgTimestampRenewal::tr("restamped")
	    + "." + fi.completeSuffix();

	return nonconflictingFileName(tgtPath);
}

void DlgTimestampRenewal::selectExternalSrcZfo(void)
{
	QString lastArchivePath = loadArchivedZfoDir(*GlobInstcs::prefsPtr);

	QFileDialog dialog(this, tr("Select ZFO File"));
	dialog.setDirectory(lastArchivePath);
	dialog.setNameFilter(tr("ZFO file (*.zfo)"));
	dialog.setFileMode(QFileDialog::ExistingFile);
	QStringList fileNames;

	if (dialog.exec()) {
		fileNames = dialog.selectedFiles();
		lastArchivePath = dialog.directory().absolutePath();
		setLoadArchivedZfoDir(*GlobInstcs::prefsPtr, lastArchivePath);
	}

	if (fileNames.size() > 0) {
		const QString &fileName = fileNames.at(0);
		{
			const QFileInfo fi(fileName);

			if ((!fi.isFile()) || (!fi.isReadable())) {
				QMessageBox::warning(this, tr("Not a Readable File"),
				    tr("The path '%1' doesn't point onto a readable file.")
				        .arg(QDir::toNativeSeparators(fileName)));
				return;
			}
		}

		if (Q_UNLIKELY(Isds::Type::RT_UNKNOWN == Isds::guessFileRawType(fileName))) {
			QMessageBox::warning(this, tr("No ZFO Content"),
			    tr("The file '%1' doesn't seem to contain a valid message or delivery info.")
			        .arg(QDir::toNativeSeparators(fileName)));
			return;
		}

		m_ui->srcZfoLineEdit->setText(
		    QDir::toNativeSeparators(fileName));

		m_ui->tgtZfoLineEdit->setText(
		    QDir::toNativeSeparators(generateTgtFilePath(fileName)));
	}

	updateActionActivation();
}

void DlgTimestampRenewal::viewExternalSrcZfo(void)
{
	if (Q_UNLIKELY(m_ui->srcZfoLineEdit->text().isEmpty())) {
		return;
	}

	DlgViewZfo::view(
	    QDir::fromNativeSeparators(m_ui->srcZfoLineEdit->text()), this);
}

void DlgTimestampRenewal::selectExternalTgtZfo(void)
{
	QString fileName = QFileDialog::getSaveFileName(this,
	    tr("Select ZFO File"),
	    QDir::fromNativeSeparators(m_ui->tgtZfoLineEdit->text()),
	    tr("ZFO file (*.zfo)"));

	m_ui->tgtZfoLineEdit->setText(QDir::toNativeSeparators(fileName));

	updateActionActivation();
}

void DlgTimestampRenewal::renewExternalTimestamps(void)
{
	const QString timeStr = QDateTime::currentDateTimeUtc().toString();

	const QString srcPath = QDir::fromNativeSeparators(m_ui->srcZfoLineEdit->text());
	const QString tgtPath = QDir::fromNativeSeparators(m_ui->tgtZfoLineEdit->text());

	const QFileInfo tgtFi(tgtPath);
	if (tgtFi.exists()) {
		if (!tgtFi.isFile()) {
			QMessageBox::warning(this, tr("Target not a File"),
			    tr("The target '%1' exists and is not a file.\n"
			        "Please choose a different target or delete the current one.")
			        .arg(m_ui->tgtZfoLineEdit->text()));
				return;
		} else {
			if (!tgtFi.isWritable()) {
				QMessageBox::warning(this, tr("Target not Writeable"),
				    tr("The target '%1' exists but it is not writeable.\n"
				        "Please choose a different target or ensure that it can be written to.")
				        .arg(m_ui->tgtZfoLineEdit->text()));
				return;
			}
			if (srcPath == tgtPath) {
				const int ret = QMessageBox::question(this,
				    tr("Source and Target are Equal"),
				    tr("The target '%1' corresponds to the source.\n"
				        "Do you want to overwrite the source file?")
				        .arg(m_ui->tgtZfoLineEdit->text()));
				if (QMessageBox::No == ret) {
					return;
				}
			}
		}
	}
	{
		const QFileInfo pathFi(tgtFi.absolutePath());
		if (!pathFi.exists()) {
			QMessageBox::warning(this, tr("Location Non-Existent"),
			    tr("The target location '%1' doesn't exist.\n"
			        "Please choose a different target or create the chosen one.")
			        .arg(QDir::toNativeSeparators(tgtFi.absolutePath())));
			return;
		} else {
			if (!pathFi.isWritable()) {
				QMessageBox::warning(this, tr("Location not Writeable"),
				    tr("The target location '%1' exists but it is not writeable.\n"
				        "Please choose a different location or ensure that it can be written to.")
				        .arg(QDir::toNativeSeparators(tgtFi.absolutePath())));
				return;
			}
		}
	}

	const AcntIdDb &acntIdDb =
	    m_acntIdDbList.at(m_ui->accountComboBox->currentIndex());

	if (!m_sessions->isConnectedToIsds(acntIdDb.username())) {
		if (Q_UNLIKELY(!m_mw->connectToIsds(acntIdDb))) {
			/* Ignore inactive accounts. */
			return;
		}
	}

	m_ui->restampPushButton->setEnabled(false);
	m_ui->actionRestamp->setEnabled(false);
	m_ui->restampExternalPushButton->setEnabled(false);
	this->setCursor(Qt::BusyCursor);

	const QString transactionId = QString("%1_%2_%3_%4")
	    .arg(acntIdDb.username())
	    .arg(timeStr)
	    .arg("external")
	    .arg(Utility::generateRandomString(6));

	m_ui->singleProgressBar->setMinimum(0);
	m_ui->singleProgressBar->setMaximum(100);
	m_ui->singleProgressBar->setValue(0);
	m_ui->singleProgressBar->setEnabled(true);
	m_ui->singleProgressBar->setVisible(true);
	m_ui->singleResultLabel->setText(QString());

	TaskArchiveIsdsDocumentFile *task = new (::std::nothrow) TaskArchiveIsdsDocumentFile(
	    acntIdDb, transactionId,
	    srcPath, tgtPath);
	if (Q_UNLIKELY(task == Q_NULLPTR)) {
		Q_ASSERT(0);
		return;
	}

	m_externalTransaction = transactionId;
	m_externalTask = task;

	task->setAutoDelete(false);
	GlobInstcs::workPoolPtr->assignLo(task);
}

void DlgTimestampRenewal::watchUploadProgress(const AcntId &acntId,
    const QString &transactId, qint64 uploadTotal, qint64 uploadCurrent)
{
	Q_UNUSED(acntId);

	if (m_trackedTransactionMap.contains(transactId)) {

		if (Q_UNLIKELY(!m_speeds.contains(transactId))) {
			m_speeds[transactId] = NetworkSpeed::createCounter(uploadCurrent);
		} else {
			NetworkSpeed &netSpeed = m_speeds[transactId];
			netSpeed.updateCounter(uploadCurrent);
		}

		const NetworkSpeed &netSpeed = m_speeds[transactId];

		const QPair<AcntId, MsgId> pair = m_trackedTransactionMap.value(transactId);
		int row = m_timestampTableModel.rowOf(pair.first, pair.second);
		if (row >= 0) {
			uploadTotal = 2 * uploadTotal;

			m_progressProxyModel.setProgress(row, MSG_PROGRESS_COL,
			    0, uploadTotal, uploadCurrent, netSpeed.uploadOverview());
		}

	} else if (m_externalTransaction == transactId) {

		if (Q_UNLIKELY(!m_speeds.contains(transactId))) {
			m_speeds[transactId] = NetworkSpeed::createCounter(uploadCurrent);
		} else {
			NetworkSpeed &netSpeed = m_speeds[transactId];
			netSpeed.updateCounter(uploadCurrent);
		}

		const NetworkSpeed &netSpeed = m_speeds[transactId];

		uploadTotal = 2 * uploadTotal;

		m_ui->singleProgressBar->setMinimum(0);
		m_ui->singleProgressBar->setMaximum(uploadTotal);
		m_ui->singleProgressBar->setValue(uploadCurrent);
		/*
		 * Don't set speed in m_ui->singleProgressBar->setFormat()
		 * because when maximum us unknown the no text is displayed.
		 */
		m_ui->singleResultLabel->setText(netSpeed.uploadOverview());

	}
}

void DlgTimestampRenewal::watchUploadProgressFinished(const AcntId &acntId,
    const QString &transactId, int result, const QString &resultDesc,
    const QVariant &resultVal)
{
	Q_UNUSED(acntId);
	Q_UNUSED(result);
	Q_UNUSED(resultDesc);
	Q_UNUSED(resultVal);

	if (m_trackedTransactionMap.contains(transactId)) {

		m_speeds.remove(transactId);

		const QPair<AcntId, MsgId> pair = m_trackedTransactionMap.value(transactId);
		int row = m_timestampTableModel.rowOf(pair.first, pair.second);
		if (row >= 0) {
			m_progressProxyModel.clearProgress(
			    row, MSG_PROGRESS_COL);
		}

	} else if (m_externalTransaction == transactId) {

		m_speeds.remove(transactId);

		m_ui->singleProgressBar->setMinimum(0);
		m_ui->singleProgressBar->setMaximum(0);
		m_ui->singleProgressBar->setValue(0);
		m_ui->singleResultLabel->setText(QString());

	}
}

void DlgTimestampRenewal::watchDownloadProgress(const AcntId &acntId,
    const QString &transactId, const MsgId &msgId,
    qint64 downloadTotal, qint64 downloadCurrent)
{
	if (m_trackedTransactionMap.contains(transactId)) {

		if (Q_UNLIKELY(!m_speeds.contains(transactId))) {
			m_speeds[transactId] = NetworkSpeed::createCounter(downloadCurrent);
		} else {
			NetworkSpeed &netSpeed = m_speeds[transactId];
			netSpeed.updateCounter(downloadCurrent);
		}

		const NetworkSpeed &netSpeed = m_speeds[transactId];

		int row = m_timestampTableModel.rowOf(acntId, msgId);
		if (row >= 0) {
			if (downloadTotal > 0) {
				downloadCurrent = downloadCurrent + downloadTotal;
				downloadTotal = 2 * downloadTotal;
			}

			m_progressProxyModel.setProgress(row, MSG_PROGRESS_COL,
			    0, downloadTotal, downloadCurrent, netSpeed.downloadOverview());
		}

	} else if (m_externalTransaction == transactId) {

		if (Q_UNLIKELY(!m_speeds.contains(transactId))) {
			m_speeds[transactId] = NetworkSpeed::createCounter(downloadCurrent);
		} else {
			NetworkSpeed &netSpeed = m_speeds[transactId];
			netSpeed.updateCounter(downloadCurrent);
		}

		const NetworkSpeed &netSpeed = m_speeds[transactId];

		if (downloadTotal > 0) {
			downloadCurrent = downloadCurrent + downloadTotal;
			downloadTotal = 2 * downloadTotal;
		}

		m_ui->singleProgressBar->setMinimum(0);
		m_ui->singleProgressBar->setMaximum(downloadTotal);
		m_ui->singleProgressBar->setValue(downloadCurrent);
		/*
		 * Don't set speed in m_ui->singleProgressBar->setFormat()
		 * because when maximum us unknown the no text is displayed.
		 */
		m_ui->singleResultLabel->setText(netSpeed.downloadOverview());
	}
}

void DlgTimestampRenewal::watchDownloadProgressFinished(const AcntId &acntId,
    const QString &transactId, const MsgId &msgId,
    int result, const QString &resultDesc)
{
	Q_UNUSED(result);

	if (m_trackedTransactionMap.contains(transactId)) {

		m_speeds.remove(transactId);

		int row = m_timestampTableModel.rowOf(acntId, msgId);
		if (row >= 0) {
			m_progressProxyModel.clearProgress(
			    row, MSG_PROGRESS_COL);
		}

		m_timestampTableModel.updateError(acntId, msgId, resultDesc);

		m_trackedTransactionMap.remove(transactId);

		TaskArchiveIsdsDocument *task = m_tasks.take(transactId);

		/* Update expiration time in database and model. */
		if ((Q_NULLPTR != task)
		        && (TaskArchiveIsdsDocument::AID_SUCCESS == task->m_result)
		        && (task->m_newExpirationTime.isValid())) {
			TstValidityHash expirationValues;
			expirationValues[Json::MsgId1(acntId.testing() ? Isds::Type::BOOL_TRUE : Isds::Type::BOOL_FALSE, msgId.dmId())] =
			    TstValidity(true, task->m_newExpirationTime);
			m_timestampDb->updateMsgTimestampData(expirationValues);
		}

		delete task;

		if (m_tasks.isEmpty() && (Q_NULLPTR == m_externalTask)) {
			updateActionActivation();
			this->setCursor(Qt::ArrowCursor);
		}

	} else if (m_externalTransaction == transactId) {

		m_speeds.remove(transactId);

		m_ui->singleProgressBar->setMinimum(0);
		m_ui->singleProgressBar->setMaximum(0);
		m_ui->singleProgressBar->setValue(0);
		m_ui->singleProgressBar->resetFormat();
		m_ui->singleProgressBar->setEnabled(false);
		m_ui->singleProgressBar->setVisible(false);
		m_ui->singleResultLabel->setText(resultDesc);

		m_externalTransaction.clear();
		delete m_externalTask; m_externalTask = Q_NULLPTR;

		if (m_tasks.isEmpty() && (Q_NULLPTR == m_externalTask)) {
			updateActionActivation();
			this->setCursor(Qt::ArrowCursor);
		}

	}
}

void DlgTimestampRenewal::initTimestampWindow(void)
{
	/* tracked */
	{
		/* Set default line height for table views/widgets. */
		m_ui->trackedMessagesView->setNarrowedLineHeight();
		m_ui->trackedMessagesView->horizontalHeader()->setDefaultAlignment(
		    Qt::AlignVCenter | Qt::AlignLeft);

		m_ui->trackedMessagesView->setContextMenuPolicy(Qt::CustomContextMenu);
		connect(m_ui->trackedMessagesView, SIGNAL(customContextMenuRequested(QPoint)),
		    this, SLOT(viewTrackedMessagesListContextMenu(QPoint)));
		m_ui->trackedMessagesView->setSelectionMode(
		    QAbstractItemView::ExtendedSelection);
		m_ui->trackedMessagesView->setSelectionBehavior(
		    QAbstractItemView::SelectRows);

		m_progressProxyModel.setSourceModel(&m_timestampTableModel);

		m_sortProxyModel.setSortRole(MsgTstListingModel::ROLE_PROXYSORT);
		m_sortProxyModel.setSourceModel(&m_progressProxyModel);
		{
			QList<int> columnList;
			columnList.append(MsgTstListingModel::COL_ACCOUNT_ID);
			columnList.append(MsgTstListingModel::COL_MESSAGE_ID);
			columnList.append(MsgTstListingModel::COL_ANNOTATION);
			columnList.append(MsgTstListingModel::COL_EXPIRATION_TIME);
			columnList.append(MsgTstListingModel::COL_ERROR);
			m_sortProxyModel.setFilterKeyColumns(columnList);
		}
		m_ui->trackedMessagesView->setModel(&m_sortProxyModel);
		m_ui->trackedMessagesView->setItemDelegateForColumn(MSG_PROGRESS_COL, &m_progressDelegate);

		m_ui->trackedMessagesView->setSortingEnabled(true);
		m_ui->trackedMessagesView->sortByColumn(MsgTstListingModel::COL_ACCOUNT_ID,
		    Qt::AscendingOrder);

		connect(m_ui->trackedMessagesView, SIGNAL(doubleClicked(QModelIndex)),
		    this, SLOT(messageItemDoubleClicked(QModelIndex)));

		m_ui->trackedMessagesView->installEventFilter(
		    new (::std::nothrow) TableHomeEndFilter(m_ui->trackedMessagesView));
		m_ui->trackedMessagesView->installEventFilter(
		    new (::std::nothrow) TableTabIgnoreFilter(m_ui->trackedMessagesView));

		connect(m_ui->trackedMessagesView->selectionModel(),
		    SIGNAL(selectionChanged(QItemSelection, QItemSelection)),
		    this,
		    SLOT(updateMessageSelection(QItemSelection, QItemSelection)));

		m_ui->messagesFilterHorizontalLayout->setVisible(false);
		connect(m_ui->messageFilterLine, SIGNAL(textChanged(QString)),
		    this, SLOT(filterMessages(QString)));
		m_ui->closeFilterButton->setIcon(style()->standardIcon(QStyle::SP_DockWidgetCloseButton));
		connect(m_ui->closeFilterButton, SIGNAL(clicked()),
		    this, SLOT(hideMessageFilter()));

		connect(m_ui->restampPushButton, SIGNAL(clicked()),
		    this, SLOT(renewTimestamps()));
	}

	/* external */
	{
		/* Set accounts into combobox - account name and user name. */
		for (const AcntIdDb &acntIdDb : m_acntIdDbList) {
			const AcntId aId(acntIdDb);
			const QString accountName =
			    GlobInstcs::acntMapPtr->acntData(aId).accountName() +
			    " (" + acntIdDb.username() + ")";
			m_ui->accountComboBox->addItem(accountName, QVariant::fromValue(aId));
		}

		connect(m_ui->chooseSrcPushButton, SIGNAL(clicked()),
		    this, SLOT(selectExternalSrcZfo()));

		connect(m_ui->viewSrcPushButton, SIGNAL(clicked()),
		    this, SLOT(viewExternalSrcZfo()));

		connect(m_ui->chooseTgtPushButton, SIGNAL(clicked()),
		    this, SLOT(selectExternalTgtZfo()));

		connect(m_ui->restampExternalPushButton, SIGNAL(clicked()),
		    this, SLOT(renewExternalTimestamps()));

		m_ui->srcZfoLineEdit->setReadOnly(true);
		m_ui->tgtZfoLineEdit->setReadOnly(true);

		/* Enable only if having available accounts. */
		m_ui->chooseSrcPushButton->setEnabled(m_ui->accountComboBox->count() > 0);

		m_ui->chooseSrcPushButton->setEnabled(true);
		m_ui->viewSrcPushButton->setEnabled(false);

		m_ui->tgtZfoLineEdit->setEnabled(false);
		m_ui->chooseTgtPushButton->setEnabled(false);

		m_ui->restampExternalPushButton->setEnabled(false);

		m_ui->singleProgressBar->setEnabled(false);
		m_ui->singleProgressBar->setVisible(false);

		m_ui->singleResultLabel->setText(QString());
	}

	updateActionActivation();

	menuConnectActions();
	databaseConnectActions();
	workerConnectActions();
}

void DlgTimestampRenewal::menuConnectActions(void)
{
	connect(m_ui->actionRestamp, SIGNAL(triggered()),
	    this, SLOT(renewTimestamps()));
	connect(m_ui->actionTrackTimestampExpiration, SIGNAL(triggered(bool)),
	    this, SLOT(toggleTimestampTracking(bool)));
}

void DlgTimestampRenewal::databaseConnectActions(void)
{
	if (Q_NULLPTR != m_timestampDb) {
		connect(m_timestampDb, SIGNAL(identificationsInserted(Json::MsgId2List, AcntId)),
		    this, SLOT(watchIdentificationsInserted(Json::MsgId2List, AcntId)));
		connect(m_timestampDb, SIGNAL(identificationsDeleted(Json::MsgId2List)),
		    this, SLOT(watchIdentificationsDeleted(Json::MsgId2List)));
		connect(m_timestampDb, SIGNAL(msgTimestampDataUpdated(TstValidityHash)),
		    this, SLOT(watchMsgTimestampDataUpdated(TstValidityHash)),
		    Qt::QueuedConnection); /* Qt::QueuedConnection because it must be executed in receiver's thread. */
	}
}

void DlgTimestampRenewal::workerConnectActions(void)
{
	connect(GlobInstcs::msgProcEmitterPtr, SIGNAL(uploadProgress(AcntId, QString, qint64, qint64)),
	    this, SLOT(watchUploadProgress(AcntId, QString, qint64, qint64)),
	    Qt::QueuedConnection); /* Qt::QueuedConnection because it must be executed in receiver's thread. */

	connect(GlobInstcs::msgProcEmitterPtr, SIGNAL(uploadProgressFinished(AcntId, QString, int, QString, QVariant)),
	    this, SLOT(watchUploadProgressFinished(AcntId, QString, int, QString, QVariant)),
	    Qt::QueuedConnection); /* Qt::QueuedConnection because it must be executed in receiver's thread. */

	connect(GlobInstcs::msgProcEmitterPtr, SIGNAL(downloadProgress(AcntId, QString, MsgId, qint64, qint64)),
	    this, SLOT(watchDownloadProgress(AcntId, QString, MsgId, qint64, qint64)),
	    Qt::QueuedConnection); /* Qt::QueuedConnection because it must be executed in receiver's thread. */

	connect(GlobInstcs::msgProcEmitterPtr, SIGNAL(downloadProgressFinished(AcntId, QString, MsgId, int, QString)),
	    this, SLOT(watchDownloadProgressFinished(AcntId, QString, MsgId, int, QString)),
	    Qt::QueuedConnection); /* Qt::QueuedConnection because it must be executed in receiver's thread. */
}

void DlgTimestampRenewal::updateActionActivation(void)
{
	const bool noTasks = m_tasks.isEmpty() && (Q_NULLPTR == m_externalTask);

	/* tracked */
	{
		const bool enabled = m_modelRowSelection.size() > 0;

		m_ui->restampPushButton->setEnabled(enabled && noTasks);

		m_ui->actionRestamp->setEnabled(enabled && noTasks);

		m_ui->actionTrackTimestampExpiration->setChecked(enabled);
		m_ui->actionTrackTimestampExpiration->setEnabled(enabled);
	}

	/* external */
	{
		const bool enabled = (m_ui->accountComboBox->count() > 0)
		    && (Q_NULLPTR == m_externalTask);

		m_ui->chooseSrcPushButton->setEnabled(enabled);

		m_ui->viewSrcPushButton->setEnabled(enabled && (!m_ui->srcZfoLineEdit->text().isEmpty()));

		m_ui->tgtZfoLineEdit->setEnabled(enabled && (!m_ui->srcZfoLineEdit->text().isEmpty()));
		m_ui->chooseTgtPushButton->setEnabled(enabled && (!m_ui->srcZfoLineEdit->text().isEmpty()));

		m_ui->restampExternalPushButton->setEnabled(enabled
		    && (!m_ui->srcZfoLineEdit->text().isEmpty())
		    && (!m_ui->tgtZfoLineEdit->text().isEmpty())
		    && noTasks);
	}
}

void DlgTimestampRenewal::clearFilterLineViaFilter(QObject *dlgPtr)
{
	if (Q_NULLPTR == dlgPtr) {
		return;
	}

	DlgTimestampRenewal *dlg = qobject_cast<DlgTimestampRenewal *>(dlgPtr);
	if (Q_UNLIKELY(Q_NULLPTR == dlg)) {
		Q_ASSERT(0);
		return;
	}

	if (dlg->m_ui->messageFilterLine != Q_NULLPTR) {
		dlg->m_ui->messageFilterLine->clear();
	}
}

void DlgTimestampRenewal::showFilterLineViaFilter(QObject *dlgPtr)
{
	if (Q_NULLPTR == dlgPtr) {
		return;
	}

	DlgTimestampRenewal *dlg = qobject_cast<DlgTimestampRenewal *>(dlgPtr);
	if (Q_UNLIKELY(Q_NULLPTR == dlg)) {
		Q_ASSERT(0);
		return;
	}
	if (dlg->m_ui->messageFilterLine != Q_NULLPTR) {
		dlg->showMessageFilter();
	}
}

void DlgTimestampRenewal::hideFilterLineViaFilter(QObject *dlgPtr)
{
	if (Q_NULLPTR == dlgPtr) {
		return;
	}

	DlgTimestampRenewal *dlg = qobject_cast<DlgTimestampRenewal *>(dlgPtr);
	if (Q_UNLIKELY(Q_NULLPTR == dlg)) {
		Q_ASSERT(0);
		return;
	}
	if (dlg->m_ui->messageFilterLine != Q_NULLPTR) {
		dlg->hideMessageFilter();
	}
}
