/*
    SPDX-FileCopyrightText: 2014-2015 Eike Hein <hein@kde.org>

    SPDX-License-Identifier: GPL-2.0-or-later
*/
pragma ComponentBehavior: Bound

import QtQuick
import QtQuick.Layouts

import org.kde.kirigami as Kirigami
import org.kde.plasma.components as PlasmaComponents3
import org.kde.plasma.core as PlasmaCore
import org.kde.plasma.plasmoid
import org.kde.plasma.private.kicker as Kicker

PlasmoidItem {
    id: kicker

    anchors.fill: parent

    signal reset
    signal modelRefreshed

    readonly property bool isDash: Plasmoid.pluginName === "org.kde.plasma.kickerdash"

    switchWidth: isDash || !fullRepresentationItem ? 0 : fullRepresentationItem.Layout.minimumWidth
    switchHeight: isDash || !fullRepresentationItem ? 0 : fullRepresentationItem.Layout.minimumHeight

    // this is a bit of a hack to prevent Plasma from spawning a dialog on its own when we're Dash
    preferredRepresentation: isDash ? fullRepresentation : null

    compactRepresentation: isDash ? null : compactRepresentationComponent
    fullRepresentation: isDash ? compactRepresentationComponent : menuRepresentationComponent

    readonly property Component itemListDialogComponent: Component {
        ItemListDialog {}
    }

    property Item dragSource: null

    property Kicker.KAStatsFavoritesModel globalFavorites: rootModel.favoritesModel as Kicker.KAStatsFavoritesModel
    property Kicker.SimpleFavoritesModel systemFavorites: rootModel.systemFavoritesModel as Kicker.SimpleFavoritesModel

    Plasmoid.icon: Plasmoid.configuration.useCustomButtonImage ? Plasmoid.configuration.customButtonImage : Plasmoid.configuration.icon

    onSystemFavoritesChanged: {
        if (systemFavorites) {
            systemFavorites.favorites = Plasmoid.configuration.favoriteSystemActions;
        }
    }

    function action_menuedit() {
        processRunner.runMenuEditor();
    }

    Component {
        id: compactRepresentationComponent
        CompactRepresentation {
            dashWindowComponent: dashboardRepresentationComponent
            toolTipSubText: kicker.toolTipSubText
        }
    }

    Component {
        id: menuRepresentationComponent
        MenuRepresentation {
            globalFavorites: kicker.globalFavorites
            systemFavorites: kicker.systemFavorites
            rootModel: rootModel
            runnerModel: runnerModel
            onInteractionConcluded: kicker.expanded = false
        }
    }

    Component {
        id: dashboardRepresentationComponent
        DashboardRepresentation {
            globalFavorites: kicker.globalFavorites
            systemFavorites: kicker.systemFavorites
            rootModel: rootModel
            runnerModel: runnerModel
            onInteractionConcluded: toggle()
        }
    }

    Kicker.RootModel {
        id: rootModel

        autoPopulate: false

        appNameFormat: kicker.isDash ? 0 : Plasmoid.configuration.appNameFormat // appNameFormat = 0 -> AppName Only
        flat: kicker.isDash || Plasmoid.configuration.limitDepth
        sorted: Plasmoid.configuration.alphaSort
        showSeparators: !kicker.isDash
        // TODO: appletInterface property now can be ported to "applet" and have the real Applet* assigned directly
        appletInterface: kicker

        showAllApps: kicker.isDash
        showAllAppsCategorized: true
        showTopLevelItems: !kicker.isDash
        showRecentApps: Plasmoid.configuration.showRecentApps
        showRecentDocs: Plasmoid.configuration.showRecentDocs
        recentOrdering: Plasmoid.configuration.recentOrdering

        onShowRecentAppsChanged: {
            Plasmoid.configuration.showRecentApps = showRecentApps;
        }

        onShowRecentDocsChanged: {
            Plasmoid.configuration.showRecentDocs = showRecentDocs;
        }

        onRecentOrderingChanged: {
            Plasmoid.configuration.recentOrdering = recentOrdering;
        }

        Component.onCompleted: {
            (favoritesModel as Kicker.KAStatsFavoritesModel).initForClient("org.kde.plasma.kicker.favorites.instance-" + Plasmoid.id)

            if (!Plasmoid.configuration.favoritesPortedToKAstats) {
                if (favoritesModel.count < 1) {
                    (favoritesModel as Kicker.KAStatsFavoritesModel).portOldFavorites(Plasmoid.configuration.favoriteApps);
                }
                Plasmoid.configuration.favoritesPortedToKAstats = true;
            }
        }
    }

    Connections {
        target: kicker.globalFavorites

        function onFavoritesChanged() {
            Plasmoid.configuration.favoriteApps = kicker.globalFavorites.favorites;
        }
    }

    Connections {
        target: kicker.systemFavorites

        function onFavoritesChanged() {
            Plasmoid.configuration.favoriteSystemActions = kicker.systemFavorites.favorites;
        }
    }

    Connections {
        target: Plasmoid.configuration

        function onFavoriteAppsChanged() {
            kicker.globalFavorites.favorites = Plasmoid.configuration.favoriteApps;
        }

        function onFavoriteSystemActionsChanged() {
            kicker.systemFavorites.favorites = Plasmoid.configuration.favoriteSystemActions;
        }
    }

    Kicker.RunnerModel {
        id: runnerModel

        appletInterface: kicker

        favoritesModel: kicker.globalFavorites as Kicker.KAStatsFavoritesModel

        runners: {
            const results = ["krunner_services",
                             "krunner_systemsettings",
                             "krunner_sessions",
                             "krunner_powerdevil",
                             "calculator",
                             "unitconverter"];

            if (Plasmoid.configuration.useExtraRunners) {
                results.push(...Plasmoid.configuration.extraRunners);
            }

            return results;
        }
    }

    Kicker.DragHelper {
        id: dragHelper

        dragIconSize: Kirigami.Units.iconSizes.medium
    }

    Kicker.ProcessRunner {
        id: processRunner
    }

    Kicker.WindowSystem { // only for X11; TODO Plasma 6.8: remove (also from plasma-workspace)
        id: windowSystem
    }

    Connections {
        target: kicker

        function onExpandedChanged(expanded) {
            if (!expanded) {
                kicker.reset();
            }
        }
    }

    function resetDragSource() {
        dragSource = null;
    }

    Plasmoid.contextualActions: [
        PlasmaCore.Action {
            text: i18nc("@action:inmenu launches kmenuedit", "Edit Applications…")
            icon.name: "kmenuedit"
            visible: Plasmoid.immutability !== PlasmaCore.Types.SystemImmutable
            onTriggered: processRunner.runMenuEditor()
        }
    ]

    Component.onCompleted: {
        if (Plasmoid.hasOwnProperty("activationTogglesExpanded")) {
            Plasmoid.activationTogglesExpanded = !kicker.isDash
        }

        rootModel.refreshed.connect(modelRefreshed);

        dragHelper.dropped.connect(resetDragSource);
    }
}
