#!/usr/bin/env perl
use strict;
use warnings;

use Test::More;
use Math::Prime::Util qw/binomial/;

my @binomials = (
 [ 20,15, 15504],
 [ 35,16, 4059928950 ],             # We can do this natively even in 32-bit
 [ 40,19, "131282408400" ],         # We can do this in 64-bit
 [ 67,31, "11923179284862717872" ], # ...and this
 [ 228,12, "30689926618143230620" ],# But the result of this is too big.
 [ 34,16, "2203961430" ],
 [ 62,28, "349615716557887465" ],
 [ 67,30, "9989690752182277136" ],
 [ 70,25, "6455761770304780752" ],
 [ 125,61, "2923171367321931373425996933337783875" ],
 [ 131,64, "183062151498210163887302260440097215750" ],
 [ 177,78, "3314450882216440395106465322941753788648564665022000" ],
 [ 61,17, "536830054536825" ],
 [ -11,22, 64512240 ],
 [ -12,23, -286097760 ],
 [ -23,-26, -2300 ],     # Kronenburg extension
 [ -12,-23, -705432 ],   # same
 [  12,-23, 0 ],
 [  12,-12, 0 ],
 [ -12,0, 1 ],
 [ "36893488147419103233",1, "36893488147419103233" ],
 [ "36893488147419103233",2, "680564733841876926945195958937245974528" ],
 [ "36893488147419103233",3, "8369468980515574351781052564276888554796991677927476166656" ],
);

plan tests => 6 + 3+ scalar(@binomials);

#
# https://pdf.sciencedirectassets.com/271536/1-s2.0-S0012365X08X00172/1-s2.0-S0012365X07007443/main.pdf
# Sprugnoli 2008, Table 1, Extended Pascal Array
# mpu 'for my $n (-5..5) { say sprintf("%2d",$n),": ",join(" ",map{sprintf("%4d",binomial($n,$_))}-5..5); }'
#

{
  my @bin; for my $n (0..10) { for my $k (0..10) { push @bin, [$n,$k]; } }
  is_deeply( [map { binomial($_->[0],$_->[1]) } @bin],
             [qw/1 0 0 0 0 0 0 0 0 0 0 1 1 0 0 0 0 0 0 0 0 0 1 2 1 0 0 0 0 0 0 0 0 1 3 3 1 0 0 0 0 0 0 0 1 4 6 4 1 0 0 0 0 0 0 1 5 10 10 5 1 0 0 0 0 0 1 6 15 20 15 6 1 0 0 0 0 1 7 21 35 35 21 7 1 0 0 0 1 8 28 56 70 56 28 8 1 0 0 1 9 36 84 126 126 84 36 9 1 0 1 10 45 120 210 252 210 120 45 10 1/],
             "binomial(0..10,0..10)" );
}
{
  my @bin; for my $n (-10..-1) { for my $k (0..10) { push @bin, [$n,$k]; } }
  is_deeply( [map { binomial($_->[0],$_->[1]) } @bin],
             [qw/1 -10 55 -220 715 -2002 5005 -11440 24310 -48620 92378 1 -9 45 -165 495 -1287 3003 -6435 12870 -24310 43758 1 -8 36 -120 330 -792 1716 -3432 6435 -11440 19448 1 -7 28 -84 210 -462 924 -1716 3003 -5005 8008 1 -6 21 -56 126 -252 462 -792 1287 -2002 3003 1 -5 15 -35 70 -126 210 -330 495 -715 1001 1 -4 10 -20 35 -56 84 -120 165 -220 286 1 -3 6 -10 15 -21 28 -36 45 -55 66 1 -2 3 -4 5 -6 7 -8 9 -10 11 1 -1 1 -1 1 -1 1 -1 1 -1 1/],
             "binomial(-10..-1,0..10)" );
}
{
  my @bin; for my $n (0..10) { for my $k (-10..-1) { push @bin, [$n,$k]; } }
  is_deeply( [map { binomial($_->[0],$_->[1]) } @bin],
             [qw/0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0/],
             "binomial(0..10,-10..-1)" );
}
{
  my @bin; for my $n (-10..-1) { for my $k (-10..-1) { push @bin, [$n,$k]; } }
  is_deeply( [map { binomial($_->[0],$_->[1]) } @bin],
             [qw/1 0 0 0 0 0 0 0 0 0 -9 1 0 0 0 0 0 0 0 0 36 -8 1 0 0 0 0 0 0 0 -84 28 -7 1 0 0 0 0 0 0 126 -56 21 -6 1 0 0 0 0 0 -126 70 -35 15 -5 1 0 0 0 0 84 -56 35 -20 10 -4 1 0 0 0 -36 28 -21 15 -10 6 -3 1 0 0 9 -8 7 -6 5 -4 3 -2 1 0 -1 1 -1 1 -1 1 -1 1 -1 1/],
             "binomial(-10..-1,-10..-1)" );
}

is_deeply( [map { binomial(13, $_) } -15 .. 15],
           [qw/0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 1 13 78 286 715 1287 1716 1716 1287 715 286 78 13 1 0 0/],
           "binomial(13,n) for n in -15 .. 15" );
is_deeply( [map { binomial(-13, $_) } -15 .. 15],
           [qw/91 -13 1 0 0 0 0 0 0 0 0 0 0 0 0 1 -13 91 -455 1820 -6188 18564 -50388 125970 -293930 646646 -1352078 2704156 -5200300 9657700 -17383860/],
           "binomial(-13,n) for n in -15 .. 15" );

# Binomials from Loeb 1995
{ my $n;
  is_deeply( [map { $n=$_; map { binomial($n,$_) } 0..$n } 0..7],
             [qw/1 1 1 1 2 1 1 3 3 1 1 4 6 4 1 1 5 10 10 5 1 1 6 15 20 15 6 1 1 7 21 35 35 21 7 1/],
             "binomial: Loeb 1995 Region 1 (positive n, positive k)" );
  is_deeply( [map { $n=$_; map { binomial($n,$_) } 0..6 } -5 .. -1],
             [qw/1 -5 15 -35 70 -126 210 1 -4 10 -20 35 -56 84 1 -3 6 -10 15 -21 28 1 -2 3 -4 5 -6 7 1 -1 1 -1 1 -1 1/],
             "binomial: Loeb 1995 Region 2 (negative n, positive k)" );
  is_deeply( [map { $n=$_; map { binomial($n,$_) } -6..$n } -6 .. -1],
             [qw/1 -5 1 10 -4 1 -10 6 -3 1 5 -4 3 -2 1 -1 1 -1 1 -1 1/],
             "binomial: Loeb 1995 Region 3 (negative n, negative k)" );
  # Region 4 is positive n and k > n.  We are always 0.
  # Region 5 is positive n and negative k.  We are always 0.
  # Region 6 is negative n and n < k < 0.  We are always 0.
}

# Selected binomials
foreach my $r (@binomials) {
  my($n, $k, $exp) = @$r;
  is( "".binomial($n,$k), $exp, "binomial($n,$k)) = $exp" );
}
