# created by util/generate at Sun Mar  1 05:11:41 2026
# https://github.com/briandfoy/cpan-security-advisory.git 2ade993f5e92834ebb64ccfd73ca92b788f2d777

=encoding utf8

=head1 NAME

CPANSA::DB - the CPAN Security Advisory data as a Perl data structure, mostly for CPAN::Audit

=head1 SYNOPSIS

This module is primarily used by L<CPAN::Audit>.

	use CPANSA::DB;

	my $db = CPANSA::DB->db;

=head1 DESCRIPTION

The C<db> subroutine returns the CPAN Security Advisory (CPANSA) reports
as a Perl data structure. However, anything can use this.

Each release also comes with a F<.gpg> file that has the signature
for the file. If you cannot confirm that the module file has the
right signature, it might have been corrupted or modified.

This module is available outside of CPAN as a release on GitHub:
L<https://github.com/briandfoy/cpan-security-advisory/releases>.
Each release on GitHub includes an attestation.

There is also a JSON file that provides the same datastructure.

=head2 Subroutines

There is exactly one subroutine:

=over 4

=item * db

Returns the hashref of all the CPANSA reports.

=back

=head1 VERIFYING

This distribution now uses
L<GitHub Attestations|https://github.blog/2024-05-02-introducing-artifact-attestations-now-in-public-beta/>,
which allow you to verify that the archive file you have was made from
the official repo.

You need a GitHub account and the L<gh tool|https://github.com/larsks/ghcli>.

	# download the distro file from GitHub, MetaCPAN, or a CPAN mirror
	$ gh auth login
	...follow instructions...
	$ gh attestation verify CPANSA-DB-20241111.tar.gz --owner briandfoy

Additionally, each release codes with GPG signature that allows you to
verify that this. The key is the same one used when the database was
distributed with L<CPAN::Audit>:

	$ gpg --verify lib/CPANSA/DB.pm.gpg lib/CPANSA/DB.pm
	gpg: Signature made Mon Nov 18 11:00:10 2024 EST
	gpg:                using RSA key 75AAB42CBA0D7F37F0D6886DF83F8D5E878B6041
	gpg: Good signature from "CPAN::Audit (brian d foy) (https://github.com/briandfoy/cpan-audit) <bdfoy@cpan.org>" [ultimate]

=head1 SEE ALSO

Everything is managed in GitHub:

=over 4

=item * L<https://github.com/briandfoy/cpan-security-advisory/releases>

=back

=cut

package CPANSA::DB;

use strict;
use warnings;

our $VERSION = '20260301.001';

sub db {
	{"dists" => {"ActivePerl" => {"advisories" => [{"affected_versions" => ["==5.16.1.1601"],"cves" => ["CVE-2012-5377"],"darkpan" => "true","description" => "Untrusted search path vulnerability in the installation functionality in ActivePerl 5.16.1.1601, when installed in the top-level C:\\ directory, allows local users to gain privileges via a Trojan horse DLL in the C:\\Perl\\Site\\bin directory, which is added to the PATH system environment variable, as demonstrated by a Trojan horse wlbsctrl.dll file used by the \"IKE and AuthIP IPsec Keying Modules\" system service in Windows Vista SP1, Windows Server 2008 SP2, Windows 7 SP1, and Windows 8 Release Preview.\n","distribution" => "ActivePerl","fixed_versions" => [],"id" => "CPANSA-ActivePerl-2012-5377","references" => ["https://www.htbridge.com/advisory/HTB23108","http://osvdb.org/86177"],"reported" => "2012-10-11","severity" => undef},{"affected_versions" => ["==5.8.8.817"],"cves" => ["CVE-2006-2856"],"darkpan" => "true","description" => "ActiveState ActivePerl 5.8.8.817 for Windows configures the site/lib directory with \"Users\" group permissions for changing files, which allows local users to gain privileges by creating a malicious sitecustomize.pl file in that directory.  NOTE: The provenance of this information is unknown; the details are obtained solely from third party information.\n","distribution" => "ActivePerl","fixed_versions" => [],"id" => "CPANSA-ActivePerl-2006-2856","references" => ["http://secunia.com/advisories/20328","http://www.securityfocus.com/bid/18269","http://www.osvdb.org/25974","http://www.vupen.com/english/advisories/2006/2140","https://exchange.xforce.ibmcloud.com/vulnerabilities/26915"],"reported" => "2006-06-06","severity" => undef},{"affected_versions" => ["<=5.8.1"],"cves" => ["CVE-2004-2286"],"darkpan" => "true","description" => "Integer overflow in the duplication operator in ActivePerl allows remote attackers to cause a denial of service (crash) and possibly execute arbitrary code via a large multiplier, which may trigger a buffer overflow.\n","distribution" => "ActivePerl","fixed_versions" => [],"id" => "CPANSA-ActivePerl-2004-2286","references" => ["http://archives.neohapsis.com/archives/fulldisclosure/2004-05/0878.html","http://www.securityfocus.com/bid/10380","https://exchange.xforce.ibmcloud.com/vulnerabilities/16224"],"reported" => "2004-12-31","severity" => undef},{"affected_versions" => ["<5.10"],"cves" => ["CVE-2004-2022"],"darkpan" => "true","description" => "ActivePerl 5.8.x and others, and Larry Wall's Perl 5.6.1 and others, when running on Windows systems, allows attackers to cause a denial of service (crash) and possibly execute arbitrary code via a long argument to the system command, which leads to a stack-based buffer overflow.  NOTE: it is unclear whether this bug is in Perl or the OS API that is used by Perl.\n","distribution" => "ActivePerl","fixed_versions" => [],"id" => "CPANSA-ActivePerl-2004-2022","references" => ["http://www.oliverkarow.de/research/ActivePerlSystemBOF.txt","http://www.perlmonks.org/index.pl?node_id=354145","http://www.securityfocus.com/bid/10375","http://archives.neohapsis.com/archives/fulldisclosure/2004-05/0905.html","http://marc.info/?l=full-disclosure&m=108489112131099&w=2","http://marc.info/?l=full-disclosure&m=108482796105922&w=2","http://marc.info/?l=full-disclosure&m=108483058514596&w=2","http://marc.info/?l=bugtraq&m=108489894009025&w=2","https://exchange.xforce.ibmcloud.com/vulnerabilities/16169"],"reported" => "2004-12-31","severity" => undef},{"affected_versions" => [],"cves" => ["CVE-2004-0377"],"darkpan" => "true","description" => "Buffer overflow in the win32_stat function for (1) ActiveState's ActivePerl and (2) Larry Wall's Perl before 5.8.3 allows local or remote attackers to execute arbitrary commands via filenames that end in a backslash character.\n","distribution" => "ActivePerl","fixed_versions" => [],"id" => "CPANSA-ActivePerl-2004-0377","references" => ["http://www.kb.cert.org/vuls/id/722414","http://lists.grok.org.uk/pipermail/full-disclosure/2004-April/019794.html","http://public.activestate.com/cgi-bin/perlbrowse?patch=22552","http://www.idefense.com/application/poi/display?id=93&type=vulnerabilities","http://marc.info/?l=bugtraq&m=108118694327979&w=2","https://exchange.xforce.ibmcloud.com/vulnerabilities/15732"],"reported" => "2004-05-04","severity" => undef},{"affected_versions" => ["<=5.6.1.629"],"cves" => ["CVE-2001-0815"],"darkpan" => "true","description" => "Buffer overflow in PerlIS.dll in Activestate ActivePerl 5.6.1.629 and earlier allows remote attackers to execute arbitrary code via an HTTP request for a long filename that ends in a .pl extension.\n","distribution" => "ActivePerl","fixed_versions" => [],"id" => "CPANSA-ActivePerl-2001-0815","references" => ["http://bugs.activestate.com/show_bug.cgi?id=18062","http://www.securityfocus.com/bid/3526","http://www.osvdb.org/678","http://marc.info/?l=bugtraq&m=100583978302585&w=2","https://exchange.xforce.ibmcloud.com/vulnerabilities/7539"],"reported" => "2001-12-06","severity" => undef}],"main_module" => "","versions" => []},"Alien-FreeImage" => {"advisories" => [{"affected_versions" => [">=0.001,<=0.011"],"cves" => ["CVE-2015-0852"],"description" => "Multiple integer underflows in PluginPCX.cpp in FreeImage 3.17.0 and earlier allow remote attackers to cause a denial of service (heap memory corruption) via vectors related to the height and width of a window.\n","distribution" => "Alien-FreeImage","fixed_versions" => [],"id" => "CPANSA-Alien-FreeImage-2015-0852-freeimage","references" => ["https://github.com/kmx/alien-freeimage/issues/5","http://lists.fedoraproject.org/pipermail/package-announce/2015-September/167766.html","http://www.openwall.com/lists/oss-security/2015/08/28/1","https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=797165","http://www.debian.org/security/2015/dsa-3392","http://lists.fedoraproject.org/pipermail/package-announce/2015-November/172491.html","http://lists.fedoraproject.org/pipermail/package-announce/2015-November/172583.html","http://www.securitytracker.com/id/1034077","http://lists.fedoraproject.org/pipermail/package-announce/2015-October/168000.html","http://lists.fedoraproject.org/pipermail/package-announce/2015-October/168023.html","https://security.gentoo.org/glsa/201701-68","https://www.oracle.com/technetwork/security-advisory/cpujan2019-5072801.html"],"reported" => "2015-09-29","severity" => undef},{"affected_versions" => [">=1.000_1,<=1.001"],"cves" => ["CVE-2015-0852"],"description" => "Multiple integer underflows in PluginPCX.cpp in FreeImage 3.17.0 and earlier allow remote attackers to cause a denial of service (heap memory corruption) via vectors related to the height and width of a window.\n","distribution" => "Alien-FreeImage","fixed_versions" => [],"id" => "CPANSA-Alien-FreeImage-2015-0852-freeimage","references" => ["https://github.com/kmx/alien-freeimage/issues/5","http://lists.fedoraproject.org/pipermail/package-announce/2015-September/167766.html","http://www.openwall.com/lists/oss-security/2015/08/28/1","https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=797165","http://www.debian.org/security/2015/dsa-3392","http://lists.fedoraproject.org/pipermail/package-announce/2015-November/172491.html","http://lists.fedoraproject.org/pipermail/package-announce/2015-November/172583.html","http://www.securitytracker.com/id/1034077","http://lists.fedoraproject.org/pipermail/package-announce/2015-October/168000.html","http://lists.fedoraproject.org/pipermail/package-announce/2015-October/168023.html","https://security.gentoo.org/glsa/201701-68","https://www.oracle.com/technetwork/security-advisory/cpujan2019-5072801.html"],"reported" => "2015-09-29","severity" => undef},{"affected_versions" => [">=0.001,<=1.001"],"cves" => ["CVE-2018-25032"],"description" => "zlib before 1.2.12 allows memory corruption when deflating (i.e., when compressing) if the input has many distant matches.\n","distribution" => "Alien-FreeImage","fixed_versions" => [],"id" => "CPANSA-Alien-FreeImage-2018-25032-zlib","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143579","https://www.openwall.com/lists/oss-security/2022/03/24/1","https://github.com/madler/zlib/commit/5c44459c3b28a9bd3283aaceab7c615f8020c531","http://www.openwall.com/lists/oss-security/2022/03/25/2","http://www.openwall.com/lists/oss-security/2022/03/26/1","https://www.openwall.com/lists/oss-security/2022/03/28/1","https://github.com/madler/zlib/compare/v1.2.11...v1.2.12","https://www.openwall.com/lists/oss-security/2022/03/28/3","https://github.com/madler/zlib/issues/605","https://www.debian.org/security/2022/dsa-5111","https://lists.debian.org/debian-lts-announce/2022/04/msg00000.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/NS2D2GFPFGOJUL4WQ3DUAY7HF4VWQ77F/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/XOKFMSNQ5D5WGMALBNBXU3GE442V74WU/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/VOKNP2L734AEL47NRYGVZIKEFOUBQY5Y/","https://lists.debian.org/debian-lts-announce/2022/05/msg00008.html","https://support.apple.com/kb/HT213255","https://support.apple.com/kb/HT213256","https://support.apple.com/kb/HT213257","http://seclists.org/fulldisclosure/2022/May/33","http://seclists.org/fulldisclosure/2022/May/35","http://seclists.org/fulldisclosure/2022/May/38","https://security.netapp.com/advisory/ntap-20220526-0009/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DF62MVMH3QUGMBDCB3DY2ERQ6EBHTADB/"],"reported" => "2022-03-25","severity" => "high"}],"main_module" => "Alien::FreeImage","versions" => [{"date" => "2014-11-27T21:33:19","version" => "0.001"},{"date" => "2014-11-27T23:23:17","version" => "0.002"},{"date" => "2014-11-28T06:50:21","version" => "0.003"},{"date" => "2014-11-28T08:16:43","version" => "0.004"},{"date" => "2014-11-28T09:42:55","version" => "0.005"},{"date" => "2014-11-29T17:54:12","version" => "0.006"},{"date" => "2014-11-29T22:00:16","version" => "0.007"},{"date" => "2014-11-29T22:04:22","version" => "0.008"},{"date" => "2014-11-30T21:50:53","version" => "0.009"},{"date" => "2014-12-08T22:22:02","version" => "0.010"},{"date" => "2014-12-09T21:26:56","version" => "0.011"},{"date" => "2017-06-25T21:05:55","version" => "1.000_1"},{"date" => "2017-06-26T17:54:11","version" => "1.000_2"},{"date" => "2017-06-27T08:30:16","version" => "1.000_3"},{"date" => "2017-07-11T11:46:10","version" => "1.001"}]},"Alien-GCrypt" => {"advisories" => [{"affected_versions" => [">=1.6.2.0,<=1.6.2.1"],"cves" => ["CVE-2021-40528"],"description" => "The ElGamal implementation in Libgcrypt before 1.9.4 allows plaintext recovery because, during interaction between two cryptographic libraries, a certain dangerous combination of the prime defined by the receiver's public key, the generator defined by the receiver's public key, and the sender's ephemeral exponents can lead to a cross-configuration attack against OpenPGP.\n","distribution" => "Alien-GCrypt","fixed_versions" => [],"id" => "CPANSA-Alien-GCrypt-2021-40528-libgcrypt","references" => ["https://ibm.github.io/system-security-research-updates/2021/07/20/insecurity-elgamal-pt1","https://ibm.github.io/system-security-research-updates/2021/09/06/insecurity-elgamal-pt2","https://eprint.iacr.org/2021/923","https://git.gnupg.org/cgi-bin/gitweb.cgi?p=libgcrypt.git;a=commit;h=3462280f2e23e16adf3ed5176e0f2413d8861320"],"reported" => "2021-09-06","severity" => "medium"},{"affected_versions" => ["==1.6.5.0"],"cves" => ["CVE-2021-40528"],"description" => "The ElGamal implementation in Libgcrypt before 1.9.4 allows plaintext recovery because, during interaction between two cryptographic libraries, a certain dangerous combination of the prime defined by the receiver's public key, the generator defined by the receiver's public key, and the sender's ephemeral exponents can lead to a cross-configuration attack against OpenPGP.\n","distribution" => "Alien-GCrypt","fixed_versions" => [],"id" => "CPANSA-Alien-GCrypt-2021-40528-libgcrypt","references" => ["https://ibm.github.io/system-security-research-updates/2021/07/20/insecurity-elgamal-pt1","https://ibm.github.io/system-security-research-updates/2021/09/06/insecurity-elgamal-pt2","https://eprint.iacr.org/2021/923","https://git.gnupg.org/cgi-bin/gitweb.cgi?p=libgcrypt.git;a=commit;h=3462280f2e23e16adf3ed5176e0f2413d8861320"],"reported" => "2021-09-06","severity" => "medium"}],"main_module" => "Alien::GCrypt","versions" => [{"date" => "2014-11-19T00:20:20","version" => "1.6.2.0"},{"date" => "2014-11-21T22:25:49","version" => "1.6.2.1"},{"date" => "2016-03-11T00:00:36","version" => "1.6.5.0"}]},"Alien-OTR" => {"advisories" => [{"affected_versions" => [">=4.0.0.0,<=4.0.0.1"],"cves" => ["CVE-2016-2851"],"description" => "Integer overflow in proto.c in libotr before 4.1.1 on 64-bit platforms allows remote attackers to cause a denial of service (memory corruption and application crash) or execute arbitrary code via a series of large OTR messages, which triggers a heap-based buffer overflow.\n","distribution" => "Alien-OTR","fixed_versions" => [],"id" => "CPANSA-Alien-OTR-2016-2851-libotr","references" => ["https://www.x41-dsec.de/lab/advisories/x41-2016-001-libotr/","http://www.debian.org/security/2016/dsa-3512","http://lists.opensuse.org/opensuse-security-announce/2016-03/msg00021.html","https://lists.cypherpunks.ca/pipermail/otr-users/2016-March/002581.html","http://seclists.org/fulldisclosure/2016/Mar/21","http://www.securityfocus.com/bid/84285","http://www.ubuntu.com/usn/USN-2926-1","http://lists.opensuse.org/opensuse-security-announce/2016-03/msg00030.html","https://security.gentoo.org/glsa/201701-10","https://www.exploit-db.com/exploits/39550/","http://www.securityfocus.com/archive/1/537745/100/0/threaded"],"reported" => "2016-04-07","severity" => "critical"},{"affected_versions" => ["==4.1.0.0"],"cves" => ["CVE-2016-2851"],"description" => "Integer overflow in proto.c in libotr before 4.1.1 on 64-bit platforms allows remote attackers to cause a denial of service (memory corruption and application crash) or execute arbitrary code via a series of large OTR messages, which triggers a heap-based buffer overflow.\n","distribution" => "Alien-OTR","fixed_versions" => [],"id" => "CPANSA-Alien-OTR-2016-2851-libotr","references" => ["https://www.x41-dsec.de/lab/advisories/x41-2016-001-libotr/","http://www.debian.org/security/2016/dsa-3512","http://lists.opensuse.org/opensuse-security-announce/2016-03/msg00021.html","https://lists.cypherpunks.ca/pipermail/otr-users/2016-March/002581.html","http://seclists.org/fulldisclosure/2016/Mar/21","http://www.securityfocus.com/bid/84285","http://www.ubuntu.com/usn/USN-2926-1","http://lists.opensuse.org/opensuse-security-announce/2016-03/msg00030.html","https://security.gentoo.org/glsa/201701-10","https://www.exploit-db.com/exploits/39550/","http://www.securityfocus.com/archive/1/537745/100/0/threaded"],"reported" => "2016-04-07","severity" => "critical"}],"main_module" => "Alien::OTR","versions" => [{"date" => "2014-02-04T00:25:37","version" => "4.0.0.0"},{"date" => "2014-06-16T00:29:25","version" => "4.0.0.1"},{"date" => "2014-11-19T00:30:34","version" => "4.1.0.0"},{"date" => "2016-03-10T23:38:55","version" => "4.1.1.0"}]},"Alien-PCRE2" => {"advisories" => [{"affected_versions" => ["<0.016000"],"comment" => "This Alien module fetches libpcre2 sources from the network. It tries to get the latest unless you set environment variables to get a different version.\n","cves" => ["CVE-2019-20454"],"description" => "An out-of-bounds read was discovered in PCRE before 10.34 when the pattern \\X is JIT compiled and used to match specially crafted subjects in non-UTF mode. Applications that use PCRE to parse untrusted input may be vulnerable to this flaw, which would allow an attacker to crash the application. The flaw occurs in do_extuni_no_utf in pcre2_jit_compile.c.\n","distribution" => "Alien-PCRE2","fixed_versions" => [">=0.016000"],"id" => "CPANSA-Alien-PCRE2-2019-20454","references" => ["https://bugs.php.net/bug.php?id=78338","https://bugs.exim.org/show_bug.cgi?id=2421","https://bugzilla.redhat.com/show_bug.cgi?id=1735494","https://vcs.pcre.org/pcre2?view=revision&revision=1092","https://security.gentoo.org/glsa/202006-16","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/OQRAHYHLRNMBTPR3KXVM27NSZP3KTOPI/"],"reported" => "2020-02-14","severity" => "high"}],"main_module" => "Alien::PCRE2","versions" => [{"date" => "2017-06-30T23:18:21","version" => "0.001000"},{"date" => "2017-07-01T02:48:02","version" => "0.002000"},{"date" => "2017-07-02T04:51:35","version" => "0.003000"},{"date" => "2017-07-02T06:53:29","version" => "0.004000"},{"date" => "2017-07-02T09:21:41","version" => "0.005000"},{"date" => "2017-07-03T01:03:23","version" => "0.006000"},{"date" => "2017-07-12T17:40:07","version" => "0.007000"},{"date" => "2017-07-13T07:43:28","version" => "0.008000"},{"date" => "2017-07-15T10:31:20","version" => "0.009000"},{"date" => "2017-07-17T04:44:54","version" => "0.010000"},{"date" => "2017-07-18T18:30:06","version" => "0.011000"},{"date" => "2017-07-19T05:07:21","version" => "0.012000"},{"date" => "2017-07-23T04:43:01","version" => "0.013000"},{"date" => "2017-11-01T02:50:14","version" => "0.014000"},{"date" => "2017-11-08T00:42:33","version" => "0.015000"},{"date" => "2022-05-08T20:22:53","version" => "0.016000"},{"date" => "2023-02-04T00:21:59","version" => "0.017000"}]},"Alien-SVN" => {"advisories" => [{"affected_versions" => [">=1.4.5.0,<=1.4.5.3"],"cves" => ["CVE-2016-8734"],"description" => "Apache Subversion's mod_dontdothat module and HTTP clients 1.4.0 through 1.8.16, and 1.9.0 through 1.9.4 are vulnerable to a denial-of-service attack caused by exponential XML entity expansion. The attack can cause the targeted process to consume an excessive amount of CPU resources or memory.\n","distribution" => "Alien-SVN","fixed_versions" => [],"id" => "CPANSA-Alien-SVN-2016-8734-subversion","references" => ["https://subversion.apache.org/security/CVE-2016-8734-advisory.txt","https://lists.apache.org/thread.html/7798f5cda1b2a3c70db4be77694b12dec8fcc1a441b00009d44f0e09\@%3Cannounce.apache.org%3E","http://www.securitytracker.com/id/1037361","http://www.securityfocus.com/bid/94588","http://www.debian.org/security/2017/dsa-3932","https://www.oracle.com/security-alerts/cpuoct2020.html"],"reported" => "2017-10-16","severity" => "medium"},{"affected_versions" => [">=1.4.6.0,<=1.4.6.0"],"cves" => ["CVE-2016-8734"],"description" => "Apache Subversion's mod_dontdothat module and HTTP clients 1.4.0 through 1.8.16, and 1.9.0 through 1.9.4 are vulnerable to a denial-of-service attack caused by exponential XML entity expansion. The attack can cause the targeted process to consume an excessive amount of CPU resources or memory.\n","distribution" => "Alien-SVN","fixed_versions" => [],"id" => "CPANSA-Alien-SVN-2016-8734-subversion","references" => ["https://subversion.apache.org/security/CVE-2016-8734-advisory.txt","https://lists.apache.org/thread.html/7798f5cda1b2a3c70db4be77694b12dec8fcc1a441b00009d44f0e09\@%3Cannounce.apache.org%3E","http://www.securitytracker.com/id/1037361","http://www.securityfocus.com/bid/94588","http://www.debian.org/security/2017/dsa-3932","https://www.oracle.com/security-alerts/cpuoct2020.html"],"reported" => "2017-10-16","severity" => "medium"},{"affected_versions" => [">=1.6.12.0,<=1.6.12.1"],"cves" => ["CVE-2016-8734"],"description" => "Apache Subversion's mod_dontdothat module and HTTP clients 1.4.0 through 1.8.16, and 1.9.0 through 1.9.4 are vulnerable to a denial-of-service attack caused by exponential XML entity expansion. The attack can cause the targeted process to consume an excessive amount of CPU resources or memory.\n","distribution" => "Alien-SVN","fixed_versions" => [],"id" => "CPANSA-Alien-SVN-2016-8734-subversion","references" => ["https://subversion.apache.org/security/CVE-2016-8734-advisory.txt","https://lists.apache.org/thread.html/7798f5cda1b2a3c70db4be77694b12dec8fcc1a441b00009d44f0e09\@%3Cannounce.apache.org%3E","http://www.securitytracker.com/id/1037361","http://www.securityfocus.com/bid/94588","http://www.debian.org/security/2017/dsa-3932","https://www.oracle.com/security-alerts/cpuoct2020.html"],"reported" => "2017-10-16","severity" => "medium"},{"affected_versions" => [">=1.7.3.0,<=1.17.3.0"],"cves" => ["CVE-2016-8734"],"description" => "Apache Subversion's mod_dontdothat module and HTTP clients 1.4.0 through 1.8.16, and 1.9.0 through 1.9.4 are vulnerable to a denial-of-service attack caused by exponential XML entity expansion. The attack can cause the targeted process to consume an excessive amount of CPU resources or memory.\n","distribution" => "Alien-SVN","fixed_versions" => [],"id" => "CPANSA-Alien-SVN-2016-8734-subversion","references" => ["https://subversion.apache.org/security/CVE-2016-8734-advisory.txt","https://lists.apache.org/thread.html/7798f5cda1b2a3c70db4be77694b12dec8fcc1a441b00009d44f0e09\@%3Cannounce.apache.org%3E","http://www.securitytracker.com/id/1037361","http://www.securityfocus.com/bid/94588","http://www.debian.org/security/2017/dsa-3932","https://www.oracle.com/security-alerts/cpuoct2020.html"],"reported" => "2017-10-16","severity" => "medium"},{"affected_versions" => [">=1.7.17.0,<=1.17.1.0"],"cves" => ["CVE-2016-8734"],"description" => "Apache Subversion's mod_dontdothat module and HTTP clients 1.4.0 through 1.8.16, and 1.9.0 through 1.9.4 are vulnerable to a denial-of-service attack caused by exponential XML entity expansion. The attack can cause the targeted process to consume an excessive amount of CPU resources or memory.\n","distribution" => "Alien-SVN","fixed_versions" => [],"id" => "CPANSA-Alien-SVN-2016-8734-subversion","references" => ["https://subversion.apache.org/security/CVE-2016-8734-advisory.txt","https://lists.apache.org/thread.html/7798f5cda1b2a3c70db4be77694b12dec8fcc1a441b00009d44f0e09\@%3Cannounce.apache.org%3E","http://www.securitytracker.com/id/1037361","http://www.securityfocus.com/bid/94588","http://www.debian.org/security/2017/dsa-3932","https://www.oracle.com/security-alerts/cpuoct2020.html"],"reported" => "2017-10-16","severity" => "medium"},{"affected_versions" => ["==1.7.19.0"],"cves" => ["CVE-2016-8734"],"description" => "Apache Subversion's mod_dontdothat module and HTTP clients 1.4.0 through 1.8.16, and 1.9.0 through 1.9.4 are vulnerable to a denial-of-service attack caused by exponential XML entity expansion. The attack can cause the targeted process to consume an excessive amount of CPU resources or memory.\n","distribution" => "Alien-SVN","fixed_versions" => [],"id" => "CPANSA-Alien-SVN-2016-8734-subversion","references" => ["https://subversion.apache.org/security/CVE-2016-8734-advisory.txt","https://lists.apache.org/thread.html/7798f5cda1b2a3c70db4be77694b12dec8fcc1a441b00009d44f0e09\@%3Cannounce.apache.org%3E","http://www.securitytracker.com/id/1037361","http://www.securityfocus.com/bid/94588","http://www.debian.org/security/2017/dsa-3932","https://www.oracle.com/security-alerts/cpuoct2020.html"],"reported" => "2017-10-16","severity" => "medium"},{"affected_versions" => ["==1.8.11.0"],"cves" => ["CVE-2015-0248"],"description" => "The (1) mod_dav_svn and (2) svnserve servers in Subversion 1.6.0 through 1.7.19 and 1.8.0 through 1.8.11 allow remote attackers to cause a denial of service (assertion failure and abort) via crafted parameter combinations related to dynamically evaluated revision numbers.\n","distribution" => "Alien-SVN","fixed_versions" => [],"id" => "CPANSA-Alien-SVN-2015-0248-subversion","references" => ["http://www.mandriva.com/security/advisories?name=MDVSA-2015:192","http://subversion.apache.org/security/CVE-2015-0248-advisory.txt","http://lists.opensuse.org/opensuse-updates/2015-04/msg00008.html","http://rhn.redhat.com/errata/RHSA-2015-1633.html","https://support.apple.com/HT205217","http://lists.apple.com/archives/security-announce/2015/Sep/msg00002.html","http://www.oracle.com/technetwork/topics/security/bulletinoct2015-2511968.html","http://www.securityfocus.com/bid/74260","http://rhn.redhat.com/errata/RHSA-2015-1742.html","http://www.ubuntu.com/usn/USN-2721-1","http://www.debian.org/security/2015/dsa-3231","https://security.gentoo.org/glsa/201610-05","http://www.securitytracker.com/id/1033214"],"reported" => "2015-04-08","severity" => undef},{"affected_versions" => ["==1.8.11.0"],"cves" => ["CVE-2015-0251"],"description" => "The mod_dav_svn server in Subversion 1.5.0 through 1.7.19 and 1.8.0 through 1.8.11 allows remote authenticated users to spoof the svn:author property via a crafted v1 HTTP protocol request sequences.\n","distribution" => "Alien-SVN","fixed_versions" => [],"id" => "CPANSA-Alien-SVN-2015-0251-subversion","references" => ["http://subversion.apache.org/security/CVE-2015-0251-advisory.txt","http://www.mandriva.com/security/advisories?name=MDVSA-2015:192","http://lists.opensuse.org/opensuse-updates/2015-04/msg00008.html","http://rhn.redhat.com/errata/RHSA-2015-1633.html","https://support.apple.com/HT205217","http://lists.apple.com/archives/security-announce/2015/Sep/msg00002.html","http://www.oracle.com/technetwork/topics/security/bulletinoct2015-2511968.html","http://www.securityfocus.com/bid/74259","http://rhn.redhat.com/errata/RHSA-2015-1742.html","http://www.ubuntu.com/usn/USN-2721-1","http://seclists.org/fulldisclosure/2015/Jun/32","http://www.debian.org/security/2015/dsa-3231","https://security.gentoo.org/glsa/201610-05","http://www.securitytracker.com/id/1033214"],"reported" => "2015-04-08","severity" => undef},{"affected_versions" => [">=1.4.5.0,<=1.4.5.3"],"cves" => ["CVE-2017-9800"],"description" => "A maliciously constructed svn+ssh:// URL would cause Subversion clients before 1.8.19, 1.9.x before 1.9.7, and 1.10.0.x through 1.10.0-alpha3 to run an arbitrary shell command. Such a URL could be generated by a malicious server, by a malicious user committing to a honest server (to attack another user of that server's repositories), or by a proxy server. The vulnerability affects all clients, including those that use file://, http://, and plain (untunneled) svn://.\n","distribution" => "Alien-SVN","fixed_versions" => [],"id" => "CPANSA-Alien-SVN-2017-9800-svn","references" => ["https://subversion.apache.org/security/CVE-2017-9800-advisory.txt","https://lists.apache.org/thread.html/cb607dc2f13bab9769147759ddccb14a4f9d8e5cdcad5e99c0d03b63\@%3Cannounce.apache.org%3E","http://www.securitytracker.com/id/1039127","http://www.securityfocus.com/bid/100259","https://confluence.atlassian.com/sourcetreekb/sourcetree-security-advisory-2017-08-11-933099891.html","http://packetstormsecurity.com/files/143722/Apache-Subversion-Arbitrary-Code-Execution.html","https://security.gentoo.org/glsa/201709-09","https://support.apple.com/HT208103","http://www.debian.org/security/2017/dsa-3932","https://access.redhat.com/errata/RHSA-2017:2480","http://www.securityfocus.com/archive/1/540999/100/0/threaded","https://lists.apache.org/thread.html/d8cf53affd700dfce90bad4968fb8b1dfb69cf7c443052c70398ff76\@%3Ccommits.subversion.apache.org%3E","https://www.oracle.com/security-alerts/cpuoct2020.html"],"reported" => "2017-08-11","severity" => "critical"},{"affected_versions" => ["==1.4.6.0"],"cves" => ["CVE-2017-9800"],"description" => "A maliciously constructed svn+ssh:// URL would cause Subversion clients before 1.8.19, 1.9.x before 1.9.7, and 1.10.0.x through 1.10.0-alpha3 to run an arbitrary shell command. Such a URL could be generated by a malicious server, by a malicious user committing to a honest server (to attack another user of that server's repositories), or by a proxy server. The vulnerability affects all clients, including those that use file://, http://, and plain (untunneled) svn://.\n","distribution" => "Alien-SVN","fixed_versions" => [],"id" => "CPANSA-Alien-SVN-2017-9800-svn","references" => ["https://subversion.apache.org/security/CVE-2017-9800-advisory.txt","https://lists.apache.org/thread.html/cb607dc2f13bab9769147759ddccb14a4f9d8e5cdcad5e99c0d03b63\@%3Cannounce.apache.org%3E","http://www.securitytracker.com/id/1039127","http://www.securityfocus.com/bid/100259","https://confluence.atlassian.com/sourcetreekb/sourcetree-security-advisory-2017-08-11-933099891.html","http://packetstormsecurity.com/files/143722/Apache-Subversion-Arbitrary-Code-Execution.html","https://security.gentoo.org/glsa/201709-09","https://support.apple.com/HT208103","http://www.debian.org/security/2017/dsa-3932","https://access.redhat.com/errata/RHSA-2017:2480","http://www.securityfocus.com/archive/1/540999/100/0/threaded","https://lists.apache.org/thread.html/d8cf53affd700dfce90bad4968fb8b1dfb69cf7c443052c70398ff76\@%3Ccommits.subversion.apache.org%3E","https://www.oracle.com/security-alerts/cpuoct2020.html"],"reported" => "2017-08-11","severity" => "critical"},{"affected_versions" => [">=1.6.12.0,<=1.6.12.1"],"cves" => ["CVE-2017-9800"],"description" => "A maliciously constructed svn+ssh:// URL would cause Subversion clients before 1.8.19, 1.9.x before 1.9.7, and 1.10.0.x through 1.10.0-alpha3 to run an arbitrary shell command. Such a URL could be generated by a malicious server, by a malicious user committing to a honest server (to attack another user of that server's repositories), or by a proxy server. The vulnerability affects all clients, including those that use file://, http://, and plain (untunneled) svn://.\n","distribution" => "Alien-SVN","fixed_versions" => [],"id" => "CPANSA-Alien-SVN-2017-9800-svn","references" => ["https://subversion.apache.org/security/CVE-2017-9800-advisory.txt","https://lists.apache.org/thread.html/cb607dc2f13bab9769147759ddccb14a4f9d8e5cdcad5e99c0d03b63\@%3Cannounce.apache.org%3E","http://www.securitytracker.com/id/1039127","http://www.securityfocus.com/bid/100259","https://confluence.atlassian.com/sourcetreekb/sourcetree-security-advisory-2017-08-11-933099891.html","http://packetstormsecurity.com/files/143722/Apache-Subversion-Arbitrary-Code-Execution.html","https://security.gentoo.org/glsa/201709-09","https://support.apple.com/HT208103","http://www.debian.org/security/2017/dsa-3932","https://access.redhat.com/errata/RHSA-2017:2480","http://www.securityfocus.com/archive/1/540999/100/0/threaded","https://lists.apache.org/thread.html/d8cf53affd700dfce90bad4968fb8b1dfb69cf7c443052c70398ff76\@%3Ccommits.subversion.apache.org%3E","https://www.oracle.com/security-alerts/cpuoct2020.html"],"reported" => "2017-08-11","severity" => "critical"},{"affected_versions" => [">=1.7.17.0,<=1.7.17.1"],"cves" => ["CVE-2017-9800"],"description" => "A maliciously constructed svn+ssh:// URL would cause Subversion clients before 1.8.19, 1.9.x before 1.9.7, and 1.10.0.x through 1.10.0-alpha3 to run an arbitrary shell command. Such a URL could be generated by a malicious server, by a malicious user committing to a honest server (to attack another user of that server's repositories), or by a proxy server. The vulnerability affects all clients, including those that use file://, http://, and plain (untunneled) svn://.\n","distribution" => "Alien-SVN","fixed_versions" => [],"id" => "CPANSA-Alien-SVN-2017-9800-svn","references" => ["https://subversion.apache.org/security/CVE-2017-9800-advisory.txt","https://lists.apache.org/thread.html/cb607dc2f13bab9769147759ddccb14a4f9d8e5cdcad5e99c0d03b63\@%3Cannounce.apache.org%3E","http://www.securitytracker.com/id/1039127","http://www.securityfocus.com/bid/100259","https://confluence.atlassian.com/sourcetreekb/sourcetree-security-advisory-2017-08-11-933099891.html","http://packetstormsecurity.com/files/143722/Apache-Subversion-Arbitrary-Code-Execution.html","https://security.gentoo.org/glsa/201709-09","https://support.apple.com/HT208103","http://www.debian.org/security/2017/dsa-3932","https://access.redhat.com/errata/RHSA-2017:2480","http://www.securityfocus.com/archive/1/540999/100/0/threaded","https://lists.apache.org/thread.html/d8cf53affd700dfce90bad4968fb8b1dfb69cf7c443052c70398ff76\@%3Ccommits.subversion.apache.org%3E","https://www.oracle.com/security-alerts/cpuoct2020.html"],"reported" => "2017-08-11","severity" => "critical"},{"affected_versions" => ["==1.7.19.0"],"cves" => ["CVE-2017-9800"],"description" => "A maliciously constructed svn+ssh:// URL would cause Subversion clients before 1.8.19, 1.9.x before 1.9.7, and 1.10.0.x through 1.10.0-alpha3 to run an arbitrary shell command. Such a URL could be generated by a malicious server, by a malicious user committing to a honest server (to attack another user of that server's repositories), or by a proxy server. The vulnerability affects all clients, including those that use file://, http://, and plain (untunneled) svn://.\n","distribution" => "Alien-SVN","fixed_versions" => [],"id" => "CPANSA-Alien-SVN-2017-9800-svn","references" => ["https://subversion.apache.org/security/CVE-2017-9800-advisory.txt","https://lists.apache.org/thread.html/cb607dc2f13bab9769147759ddccb14a4f9d8e5cdcad5e99c0d03b63\@%3Cannounce.apache.org%3E","http://www.securitytracker.com/id/1039127","http://www.securityfocus.com/bid/100259","https://confluence.atlassian.com/sourcetreekb/sourcetree-security-advisory-2017-08-11-933099891.html","http://packetstormsecurity.com/files/143722/Apache-Subversion-Arbitrary-Code-Execution.html","https://security.gentoo.org/glsa/201709-09","https://support.apple.com/HT208103","http://www.debian.org/security/2017/dsa-3932","https://access.redhat.com/errata/RHSA-2017:2480","http://www.securityfocus.com/archive/1/540999/100/0/threaded","https://lists.apache.org/thread.html/d8cf53affd700dfce90bad4968fb8b1dfb69cf7c443052c70398ff76\@%3Ccommits.subversion.apache.org%3E","https://www.oracle.com/security-alerts/cpuoct2020.html"],"reported" => "2017-08-11","severity" => "critical"},{"affected_versions" => [">=1.7.3.0,<=1.7.3.1"],"cves" => ["CVE-2017-9800"],"description" => "A maliciously constructed svn+ssh:// URL would cause Subversion clients before 1.8.19, 1.9.x before 1.9.7, and 1.10.0.x through 1.10.0-alpha3 to run an arbitrary shell command. Such a URL could be generated by a malicious server, by a malicious user committing to a honest server (to attack another user of that server's repositories), or by a proxy server. The vulnerability affects all clients, including those that use file://, http://, and plain (untunneled) svn://.\n","distribution" => "Alien-SVN","fixed_versions" => [],"id" => "CPANSA-Alien-SVN-2017-9800-svn","references" => ["https://subversion.apache.org/security/CVE-2017-9800-advisory.txt","https://lists.apache.org/thread.html/cb607dc2f13bab9769147759ddccb14a4f9d8e5cdcad5e99c0d03b63\@%3Cannounce.apache.org%3E","http://www.securitytracker.com/id/1039127","http://www.securityfocus.com/bid/100259","https://confluence.atlassian.com/sourcetreekb/sourcetree-security-advisory-2017-08-11-933099891.html","http://packetstormsecurity.com/files/143722/Apache-Subversion-Arbitrary-Code-Execution.html","https://security.gentoo.org/glsa/201709-09","https://support.apple.com/HT208103","http://www.debian.org/security/2017/dsa-3932","https://access.redhat.com/errata/RHSA-2017:2480","http://www.securityfocus.com/archive/1/540999/100/0/threaded","https://lists.apache.org/thread.html/d8cf53affd700dfce90bad4968fb8b1dfb69cf7c443052c70398ff76\@%3Ccommits.subversion.apache.org%3E","https://www.oracle.com/security-alerts/cpuoct2020.html"],"reported" => "2017-08-11","severity" => "critical"},{"affected_versions" => ["==1.8.11.0"],"cves" => ["CVE-2017-9800"],"description" => "A maliciously constructed svn+ssh:// URL would cause Subversion clients before 1.8.19, 1.9.x before 1.9.7, and 1.10.0.x through 1.10.0-alpha3 to run an arbitrary shell command. Such a URL could be generated by a malicious server, by a malicious user committing to a honest server (to attack another user of that server's repositories), or by a proxy server. The vulnerability affects all clients, including those that use file://, http://, and plain (untunneled) svn://.\n","distribution" => "Alien-SVN","fixed_versions" => [],"id" => "CPANSA-Alien-SVN-2017-9800-svn","references" => ["https://subversion.apache.org/security/CVE-2017-9800-advisory.txt","https://lists.apache.org/thread.html/cb607dc2f13bab9769147759ddccb14a4f9d8e5cdcad5e99c0d03b63\@%3Cannounce.apache.org%3E","http://www.securitytracker.com/id/1039127","http://www.securityfocus.com/bid/100259","https://confluence.atlassian.com/sourcetreekb/sourcetree-security-advisory-2017-08-11-933099891.html","http://packetstormsecurity.com/files/143722/Apache-Subversion-Arbitrary-Code-Execution.html","https://security.gentoo.org/glsa/201709-09","https://support.apple.com/HT208103","http://www.debian.org/security/2017/dsa-3932","https://access.redhat.com/errata/RHSA-2017:2480","http://www.securityfocus.com/archive/1/540999/100/0/threaded","https://lists.apache.org/thread.html/d8cf53affd700dfce90bad4968fb8b1dfb69cf7c443052c70398ff76\@%3Ccommits.subversion.apache.org%3E","https://www.oracle.com/security-alerts/cpuoct2020.html"],"reported" => "2017-08-11","severity" => "critical"}],"main_module" => "Alien::SVN","versions" => [{"date" => "2007-09-12T10:21:02","version" => "1.4.5.0"},{"date" => "2007-09-21T01:13:48","version" => "1.4.5.1"},{"date" => "2007-09-21T11:45:13","version" => "1.4.5.2"},{"date" => "2007-12-26T09:04:20","version" => "1.4.5.3"},{"date" => "2007-12-27T05:34:26","version" => "1.4.6.0"},{"date" => "2010-08-18T07:45:18","version" => "v1.6.12.0"},{"date" => "2011-02-23T00:51:22","version" => "v1.6.12.1"},{"date" => "2012-03-02T00:57:20","version" => "v1.7.3.0"},{"date" => "2012-03-18T22:14:33","version" => "v1.7.3.1"},{"date" => "2014-06-12T04:08:38","version" => "v1.7.17.0"},{"date" => "2014-06-12T17:19:44","version" => "v1.7.17.1"},{"date" => "2015-01-12T23:26:41","version" => "v1.7.19.0"},{"date" => "2015-01-13T00:12:19","version" => "v1.8.11.0"}]},"Amon2-Auth-Site-LINE" => {"advisories" => [{"affected_versions" => ["<0.05"],"cves" => ["CVE-2024-57835"],"description" => "Amon2::Auth::Site::LINE uses the String::Random module\x{a0}to generate nonce values.\x{a0}  String::Random\x{a0}defaults to Perl's built-in predictable\x{a0}random number generator,\x{a0}the rand() function, which is not cryptographically secure","distribution" => "Amon2-Auth-Site-LINE","fixed_versions" => [">=0.05"],"id" => "CPANSA-Amon2-Auth-Site-LINE-2024-57835","references" => ["https://metacpan.org/release/SHLOMIF/String-Random-0.32/source/lib/String/Random.pm#L377","https://metacpan.org/release/TANIGUCHI/Amon2-Auth-Site-LINE-0.04/source/lib/Amon2/Auth/Site/LINE.pm#L235","https://metacpan.org/release/TANIGUCHI/Amon2-Auth-Site-LINE-0.04/source/lib/Amon2/Auth/Site/LINE.pm#L255","https://security.metacpan.org/docs/guides/random-data-for-security.html","https://jvndb.jvn.jp/ja/contents/2025/JVNDB-2025-003449.html"],"reported" => "2025-04-05","severity" => "moderate"}],"main_module" => "Amon2::Auth::Site::LINE","versions" => [{"date" => "2020-11-21T06:34:32","version" => "0.01"},{"date" => "2020-11-23T00:05:03","version" => "0.02"},{"date" => "2020-11-25T01:33:35","version" => "0.03"},{"date" => "2020-11-26T07:04:40","version" => "0.04"},{"date" => "2025-05-20T12:14:56","version" => "0.05"}]},"Apache-ASP" => {"advisories" => [{"affected_versions" => ["<1.95"],"cves" => [],"description" => "A bug would allow a malicious user possible writing of files in the same directory as the source.asp script.\n","distribution" => "Apache-ASP","fixed_versions" => [">=1.95"],"id" => "CPANSA-Apache-ASP-2000-01","references" => ["https://metacpan.org/release/CHAMAS/Apache-ASP-2.63/source/README"],"reported" => "2000-07-10","severity" => undef}],"main_module" => "Apache::ASP","versions" => [{"date" => "1998-06-24T02:10:51","version" => "0.01"},{"date" => "1998-07-11T01:48:14","version" => "0.02"},{"date" => "1998-09-14T11:13:32","version" => "0.03"},{"date" => "1998-10-12T07:50:56","version" => "0.04"},{"date" => "1998-10-18T21:29:19","version" => "0.05"},{"date" => "1999-02-06T06:04:50","version" => "0.08"},{"date" => "1999-04-22T08:30:57","version" => "0.09"},{"date" => "1999-06-24T20:04:52","version" => "0.11"},{"date" => "1999-07-02T07:05:05","version" => "0.12"},{"date" => "1999-07-29T10:58:20","version" => "0.14"},{"date" => "1999-08-25T02:02:31","version" => "0.15"},{"date" => "1999-09-22T20:54:01","version" => "0.16"},{"date" => "1999-11-16T04:44:48","version" => "0.17"},{"date" => "2000-02-04T02:14:14","version" => "0.18"},{"date" => "2000-07-03T13:08:54","version" => "1.91"},{"date" => "2000-07-03T22:43:45","version" => "1.93"},{"date" => "2000-07-11T01:44:02","version" => "1.95"},{"date" => "2000-07-16T07:17:39","version" => "2.00"},{"date" => "2000-07-22T23:31:36","version" => "2.01"},{"date" => "2000-08-02T00:11:15","version" => "2.03"},{"date" => "2000-11-26T19:15:48","version" => "2.07"},{"date" => "2001-01-31T04:03:17","version" => "2.09"},{"date" => "2001-05-30T01:37:39","version" => "2.11"},{"date" => "2001-06-12T00:41:33","version" => "2.15"},{"date" => "2001-06-18T02:35:48","version" => "2.17"},{"date" => "2001-07-11T05:27:22","version" => "2.19"},{"date" => "2001-08-05T23:01:50","version" => "2.21"},{"date" => "2001-10-11T07:54:39","version" => "2.23"},{"date" => "2001-10-11T23:34:01","version" => "2.25"},{"date" => "2001-11-01T01:11:12","version" => "2.27"},{"date" => "2001-11-19T21:41:12","version" => "2.29"},{"date" => "2002-01-22T09:52:49","version" => "2.31"},{"date" => "2002-04-30T09:12:20","version" => "2.33"},{"date" => "2002-05-30T19:47:22","version" => "2.35"},{"date" => "2002-07-03T21:11:15","version" => "2.37"},{"date" => "2002-09-12T08:16:20","version" => "2.39"},{"date" => "2002-09-30T06:35:47","version" => "2.41"},{"date" => "2002-10-14T04:01:36","version" => "2.45"},{"date" => "2002-11-07T02:03:41","version" => "2.47"},{"date" => "2002-11-11T07:15:21","version" => "2.49"},{"date" => "2003-02-10T21:11:34","version" => "2.51"},{"date" => "2003-04-10T16:27:14","version" => "2.53"},{"date" => "2003-08-10T07:39:57","version" => "2.55"},{"date" => "2004-01-29T08:30:48","version" => "2.57"},{"date" => "2005-05-24T05:52:39","version" => "2.59"},{"date" => "2008-05-25T23:07:57","version" => "2.61"},{"date" => "2011-10-02T19:18:10","version" => "2.62"},{"date" => "2012-02-13T23:15:04","version" => "2.62"},{"date" => "2018-03-15T05:28:37","version" => "2.63"}]},"Apache-AuthCAS" => {"advisories" => [{"affected_versions" => ["<0.5"],"cves" => ["CVE-2007-6342"],"description" => "A tainted cookie could be sent by a malicious user and it would be used in an SQL query without protection against SQL injection.\n","distribution" => "Apache-AuthCAS","fixed_versions" => [">=0.5"],"id" => "CPANSA-Apache-AuthCAS-2007-01","references" => ["https://metacpan.org/changes/distribution/Apache-AuthCAS","https://cxsecurity.com/issue/WLB-2007120031"],"reported" => "2007-12-13","severity" => "high"}],"main_module" => "Apache::AuthCAS","versions" => [{"date" => "2004-09-15T19:17:43","version" => "0.1"},{"date" => "2004-09-15T20:11:40","version" => "0.2"},{"date" => "2004-10-05T22:51:50","version" => "0.3"},{"date" => "2004-10-13T00:45:52","version" => "0.4"},{"date" => "2008-03-23T23:03:16","version" => "0.5"}]},"Apache-AuthenHook" => {"advisories" => [{"affected_versions" => [">=2.00_04"],"cves" => ["CVE-2010-3845"],"description" => "libapache-authenhook-perl 2.00-04 stores usernames and passwords in plaintext in the vhost error log.\n","distribution" => "Apache-AuthenHook","fixed_versions" => [],"id" => "CPANSA-Apache-AuthenHook-2010-3845","references" => ["https://rt.cpan.org/Public/Bug/Display.html?id=62040","https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=599712","http://seclists.org/oss-sec/2010/q4/63"],"reported" => "2017-08-08","severity" => "critical"}],"main_module" => "Apache::AuthenHook","versions" => [{"date" => "2003-06-20T19:05:21","version" => "2.00_01"},{"date" => "2004-04-06T01:20:10","version" => "2.00_03"},{"date" => "2005-04-14T12:57:55","version" => "2.00_04"}]},"Apache-MP3" => {"advisories" => [{"affected_versions" => ["<2.15"],"cves" => [],"description" => "A security bug allowed people to bypass the AllowDownload setting.\n","distribution" => "Apache-MP3","fixed_versions" => [">=2.15"],"id" => "CPANSA-Apache-MP3-2001-01","references" => ["https://metacpan.org/dist/Apache-MP3/changes"],"reported" => "2001-01-01","severity" => undef}],"main_module" => "Apache::MP3","versions" => [{"date" => "2000-03-20T13:00:07","version" => "1.00"},{"date" => "2000-05-27T04:19:21","version" => "2.00"},{"date" => "2000-05-27T04:34:42","version" => "2.01"},{"date" => "2000-05-28T16:17:59","version" => "2.02"},{"date" => "2000-08-23T13:46:23","version" => "2.04"},{"date" => "2000-08-25T14:45:54","version" => "2.05"},{"date" => "2000-08-26T03:41:07","version" => "2.06"},{"date" => "2000-08-31T20:28:28","version" => "2.08"},{"date" => "2000-09-03T18:31:17","version" => "2.10"},{"date" => "2000-09-09T22:12:04","version" => "2.11"},{"date" => "2000-11-21T22:15:07","version" => "2.12"},{"date" => "2000-12-31T04:29:03","version" => "2.14"},{"date" => "2001-01-02T03:37:33","version" => "2.15"},{"date" => "2001-05-01T02:43:47","version" => "2.16"},{"date" => "2001-06-10T22:02:46","version" => "2.18"},{"date" => "2001-07-17T01:39:59","version" => "2.19"},{"date" => "2001-09-26T01:14:42","version" => "2.20"},{"date" => "2002-01-06T20:38:33","version" => "2.22"},{"date" => "2002-05-31T01:12:04","version" => "2.26"},{"date" => "2002-08-16T04:18:25","version" => "3.00"},{"date" => "2002-08-18T17:41:46","version" => "3.01"},{"date" => "2002-10-14T03:26:03","version" => "3.03"},{"date" => "2003-02-15T00:51:19","version" => "3.04"},{"date" => "2003-10-06T14:12:34","version" => "3.05"},{"date" => "2006-04-15T01:26:38","version" => "4.00"}]},"Apache-Session-Browseable" => {"advisories" => [{"affected_versions" => ["<1.3.6"],"cves" => ["CVE-2020-36659"],"description" => "In Apache::Session::Browseable before 1.3.6, validity of the X.509 certificate is not checked by default when connecting to remote LDAP backends, because the default configuration of the Net::LDAPS module for Perl is used. NOTE: this can, for example, be fixed in conjunction with the CVE-2020-16093 fix.\n","distribution" => "Apache-Session-Browseable","fixed_versions" => [">=1.3.6"],"id" => "CPANSA-Apache-Session-Browseable-2020-36659","references" => ["https://github.com/LemonLDAPNG/Apache-Session-Browseable/commit/fdf393235140b293cae5578ef136055a78f3574f","https://lists.debian.org/debian-lts-announce/2023/01/msg00025.html"],"reported" => "2023-01-27","severity" => undef}],"main_module" => "Apache::Session::Browseable","versions" => [{"date" => "2009-10-31T08:09:42","version" => "0.1"},{"date" => "2009-11-01T09:10:13","version" => "0.2"},{"date" => "2009-11-01T16:21:16","version" => "0.3"},{"date" => "2010-08-16T15:26:19","version" => "0.4"},{"date" => "2010-12-06T21:08:25","version" => "0.5"},{"date" => "2010-12-08T15:45:21","version" => "0.6"},{"date" => "2012-06-24T07:14:37","version" => "0.7"},{"date" => "2012-10-13T16:15:41","version" => "0.8"},{"date" => "2013-02-28T06:05:09","version" => "0.9"},{"date" => "2013-08-28T04:42:23","version" => "1.0"},{"date" => "2013-08-30T04:47:02","version" => "1.0"},{"date" => "2013-10-20T05:39:14","version" => "v1.0.2"},{"date" => "2015-06-12T15:56:45","version" => "1.1"},{"date" => "2016-03-09T05:31:13","version" => "1.2"},{"date" => "2016-03-10T06:30:41","version" => "v1.2.1"},{"date" => "2016-04-01T11:34:51","version" => "v1.2.2"},{"date" => "2016-06-07T13:59:19","version" => "v1.2.3"},{"date" => "2017-02-19T07:34:18","version" => "v1.2.4"},{"date" => "2017-04-04T05:18:26","version" => "v1.2.5"},{"date" => "2017-09-12T09:35:30","version" => "v1.2.5"},{"date" => "2017-10-03T05:00:07","version" => "v1.2.7"},{"date" => "2017-10-03T10:42:35","version" => "v1.2.8"},{"date" => "2019-02-08T06:29:20","version" => "v1.2.9"},{"date" => "2019-02-08T09:31:22","version" => "v1.3.0"},{"date" => "2019-05-04T10:55:48","version" => "v1.3.1"},{"date" => "2019-07-04T18:30:30","version" => "v1.3.2"},{"date" => "2019-09-19T20:44:43","version" => "v1.3.3"},{"date" => "2019-11-20T19:43:04","version" => "v1.3.4"},{"date" => "2020-01-21T10:20:26","version" => "v1.3.5"},{"date" => "2020-09-04T13:23:31","version" => "v1.3.6"},{"date" => "2020-09-04T13:39:40","version" => "v1.3.7"},{"date" => "2020-09-06T21:03:06","version" => "v1.3.8"},{"date" => "2021-08-10T04:44:06","version" => "v1.3.9"},{"date" => "2022-03-08T13:51:31","version" => "v1.3.10"},{"date" => "2022-09-26T16:41:24","version" => "v1.3.11"},{"date" => "2023-07-06T10:43:25","version" => "v1.3.12"},{"date" => "2023-07-06T11:38:32","version" => "v1.3.13"},{"date" => "2024-12-19T07:59:19","version" => "v1.3.13"},{"date" => "2025-04-10T19:24:48","version" => "v1.3.15"},{"date" => "2025-04-12T10:31:56","version" => "v1.3.16"},{"date" => "2025-06-18T12:49:41","version" => "v1.3.17"},{"date" => "2025-09-23T10:46:46","version" => "v1.3.18"}]},"Apache-Session-LDAP" => {"advisories" => [{"affected_versions" => ["<0.5"],"cves" => ["CVE-2020-36658"],"description" => "In Apache::Session::LDAP before 0.5, validity of the X.509 certificate is not checked by default when connecting to remote LDAP backends, because the default configuration of the Net::LDAPS module for Perl is used. NOTE: this can, for example, be fixed in conjunction with the CVE-2020-16093 fix.\n","distribution" => "Apache-Session-LDAP","fixed_versions" => [">=0.5"],"id" => "CPANSA-Apache-Session-LDAP-2020-36658","references" => ["https://github.com/LemonLDAPNG/Apache-Session-LDAP/commit/490722b71eed1ed1ab33d58c78578f23e043561f","https://lists.debian.org/debian-lts-announce/2023/01/msg00024.html"],"reported" => "2023-01-27","severity" => undef}],"main_module" => "Apache::Session::LDAP","versions" => [{"date" => "2009-04-18T17:09:10","version" => "0.01"},{"date" => "2009-04-18T19:43:50","version" => "0.02"},{"date" => "2010-12-08T15:30:51","version" => "0.1"},{"date" => "2012-06-26T04:22:47","version" => "0.2"},{"date" => "2014-10-24T12:21:07","version" => "0.2"},{"date" => "2015-06-12T15:47:40","version" => "0.4"},{"date" => "2020-09-06T13:13:20","version" => "0.2"}]},"Apache-SessionX" => {"advisories" => [{"affected_versions" => [">0"],"cves" => ["CVE-2025-40932"],"description" => "Apache::SessionX versions through 2.01 for Perl create insecure session id.  Apache::SessionX generates session ids insecurely. The default session id generator in Apache::SessionX::Generate::MD5 returns a MD5 hash seeded with the built-in rand() function, the epoch time, and the PID. The PID will come from a small set of numbers, and the epoch time may be guessed, if it is not leaked from the HTTP Date header. The built-in rand function is unsuitable for cryptographic usage. Predicable session ids could allow an attacker to gain access to systems.","distribution" => "Apache-SessionX","fixed_versions" => [],"id" => "CPANSA-Apache-SessionX-2005-01","references" => ["https://metacpan.org/release/GRICHTER/Apache-SessionX-2.01/source/SessionX/Generate/MD5.pm#L29","https://metacpan.org/changes/distribution/Apache-SessionX"],"reported" => "2005-11-15","severity" => undef}],"main_module" => "Apache::SessionX","versions" => [{"date" => "2001-11-20T15:36:53","version" => "2.00"},{"date" => "2003-03-02T14:18:57","version" => "2.00"},{"date" => "2005-11-15T05:21:49","version" => "2.01"}]},"Apache-Wyrd" => {"advisories" => [{"affected_versions" => ["<0.97"],"cves" => [],"description" => "User-submitted data cab be executed if it is displayed on a page, if the data contains a string that can be interpreted as a Wyrd.\n","distribution" => "Apache-Wyrd","fixed_versions" => [">=0.97"],"id" => "CPANSA-Apache-Wyrd-2008-01","references" => ["https://metacpan.org/dist/Apache-Wyrd/changes"],"reported" => "2008-04-14","severity" => undef}],"main_module" => "Apache::Wyrd","versions" => [{"date" => "2004-03-17T21:36:52","version" => "0.8"},{"date" => "2004-03-18T22:52:04","version" => "0.81"},{"date" => "2004-03-25T23:52:49","version" => "0.82"},{"date" => "2004-08-19T15:42:55","version" => "0.83"},{"date" => "2004-09-03T19:44:01","version" => "0.84"},{"date" => "2004-09-22T16:08:23","version" => "0.85"},{"date" => "2004-09-23T02:04:43","version" => "0.86"},{"date" => "2004-10-31T20:59:42","version" => "0.87"},{"date" => "2004-12-16T20:56:33","version" => "0.90"},{"date" => "2005-01-09T21:52:49","version" => "0.91"},{"date" => "2005-01-13T17:42:18","version" => "0.92"},{"date" => "2005-03-25T21:22:56","version" => "0.93"},{"date" => "2006-10-22T22:57:04","version" => "0.94"},{"date" => "2007-04-30T23:02:05","version" => "0.95"},{"date" => "2007-05-01T15:20:02","version" => "0.96"},{"date" => "2008-04-14T18:49:14","version" => "0.97"},{"date" => "2008-04-15T21:32:47","version" => "0.98"}]},"Apache2-AuthAny" => {"advisories" => [{"affected_versions" => [">0"],"cves" => ["CVE-2025-40933"],"description" => "Apache::AuthAny::Cookie v0.201 or earlier for Perl generates session ids insecurely.  Session ids are generated using an MD5 hash of the epoch time and a call to the built-in rand function. The epoch time may be guessed, if it is not leaked from the HTTP Date header. The built-in rand function is unsuitable for cryptographic usage.  Predicable session ids could allow an attacker to gain access to systems.","distribution" => "Apache2-AuthAny","fixed_versions" => [],"id" => "CPANSA-Apache2-AuthAny-2025-40933","references" => ["https://metacpan.org/release/KGOLDOV/Apache2-AuthAny-0.201/source/lib/Apache2/AuthAny/Cookie.pm"],"reported" => "2025-09-17","severity" => undef}],"main_module" => "Apache2::AuthAny","versions" => [{"date" => "2011-05-09T22:32:29","version" => "0.20"},{"date" => "2011-05-16T18:32:03","version" => "0.201"}]},"App-Context" => {"advisories" => [{"affected_versions" => [">=0.01,<=0.968"],"cves" => ["CVE-2012-6141"],"description" => "The App::Context module 0.01 through 0.968 for Perl does not properly use the Storable::thaw function, which allows remote attackers to execute arbitrary code via a crafted request to (1) App::Session::Cookie or (2) App::Session::HTMLHidden, which is not properly handled when it is deserialized.\n","distribution" => "App-Context","fixed_versions" => [">0.968"],"id" => "CPANSA-App-Context-2012-6141","references" => ["http://seclists.org/oss-sec/2013/q2/318","https://exchange.xforce.ibmcloud.com/vulnerabilities/84198"],"reported" => "2014-06-04","severity" => undef}],"main_module" => "App::Context","versions" => [{"date" => "2002-10-10T21:31:39","version" => "0.01"},{"date" => "2004-09-02T21:17:44","version" => "0.90"},{"date" => "2005-01-07T14:02:06","version" => "0.93"},{"date" => "2005-08-09T20:05:02","version" => "0.95"},{"date" => "2006-03-10T04:24:13","version" => "0.96"},{"date" => "2006-03-12T01:30:11","version" => "0.962"},{"date" => "2006-07-25T02:30:21","version" => "0.963"},{"date" => "2006-09-04T19:41:12","version" => "0.964"},{"date" => "2007-04-17T13:33:24","version" => "0.965"},{"date" => "2008-02-27T03:13:41","version" => "0.966"},{"date" => "2008-02-27T14:19:23","version" => "0.9661"},{"date" => "2009-09-11T14:31:52","version" => "0.967"},{"date" => "2010-06-09T21:33:19","version" => "0.968"}]},"App-Genpass" => {"advisories" => [{"affected_versions" => ["<0.2400"],"cves" => [],"description" => "App-genpass before v0.2400 generated passwords using build in rand()\n","distribution" => "App-Genpass","fixed_versions" => [">=0.2400"],"id" => "CPANSA-App-Genpass-2024-001","references" => ["https://metacpan.org/dist/App-Genpass/changes","https://github.com/xsawyerx/app-genpass/pull/5","https://github.com/briandfoy/cpan-security-advisory/issues/178"],"reported" => undef,"severity" => undef}],"main_module" => "App::Genpass","versions" => [{"date" => "2009-12-14T22:15:31","version" => "0.03"},{"date" => "2010-01-01T18:06:50","version" => "0.04"},{"date" => "2010-01-02T07:45:49","version" => "0.05"},{"date" => "2010-05-28T21:46:01","version" => "0.06"},{"date" => "2010-05-29T21:37:11","version" => "0.07"},{"date" => "2010-05-30T08:35:54","version" => "0.08"},{"date" => "2010-05-31T18:39:55","version" => "0.09"},{"date" => "2010-06-07T10:16:54","version" => "0.10"},{"date" => "2010-07-16T21:15:53","version" => "0.11"},{"date" => "2010-07-16T22:36:16","version" => "1.00"},{"date" => "2010-07-18T15:20:18","version" => "1.01"},{"date" => "2011-02-17T10:52:08","version" => "2.00"},{"date" => "2011-03-10T12:26:49","version" => "2.01"},{"date" => "2011-08-03T11:58:46","version" => "2.02"},{"date" => "2011-08-03T16:05:37","version" => "2.03"},{"date" => "2011-08-06T07:36:59","version" => "2.04"},{"date" => "2011-08-08T12:51:57","version" => "2.10"},{"date" => "2011-11-27T17:45:15","version" => "2.20"},{"date" => "2012-03-26T19:55:19","version" => "2.30"},{"date" => "2012-06-26T08:16:36","version" => "2.31"},{"date" => "2012-06-30T23:12:23","version" => "2.32"},{"date" => "2012-11-20T08:48:46","version" => "2.33"},{"date" => "2014-08-04T20:00:26","version" => "2.34"},{"date" => "2016-10-12T08:56:56","version" => "2.400"},{"date" => "2016-10-14T21:27:13","version" => "2.401"}]},"App-Github-Email" => {"advisories" => [{"affected_versions" => ["<0.3.3"],"cves" => ["CVE-2015-7686"],"description" => "Insecure dependency on Email::Address.\n","distribution" => "App-Github-Email","fixed_versions" => [">=0.3.3"],"id" => "CPANSA-App-Github-Email-2018-01","references" => ["https://metacpan.org/changes/distribution/App-Github-Email","https://github.com/faraco/App-Github-Email/commit/b7f052280d1c8ae97bdefc106ca3cbba4aea7213"],"reported" => "2018-01-20"}],"main_module" => "App::Github::Email","versions" => [{"date" => "2017-01-16T08:03:02","version" => "0.0.1"},{"date" => "2017-01-16T12:56:51","version" => "0.0.2"},{"date" => "2017-01-16T17:38:16","version" => "0.0.3"},{"date" => "2017-03-11T10:45:23","version" => "0.0.4"},{"date" => "2017-04-05T11:19:02","version" => "0.0.5"},{"date" => "2017-04-15T17:35:18","version" => "0.0.6"},{"date" => "2017-05-19T05:05:24","version" => "0.0.7"},{"date" => "2017-12-18T14:11:19","version" => "0.1.0"},{"date" => "2017-12-21T08:24:12","version" => "0.1.1"},{"date" => "2018-01-15T03:18:05","version" => "0.2.0"},{"date" => "2018-01-20T12:55:34","version" => "0.2.1"},{"date" => "2018-08-30T16:07:18","version" => "0.3.1"},{"date" => "2018-08-30T16:13:54","version" => "0.3.2"},{"date" => "2018-08-31T03:49:31","version" => "0.3.3"}]},"App-Netdisco" => {"advisories" => [{"affected_versions" => [">=2.001000_001,<=2.007000_001"],"cves" => ["CVE-2020-11022"],"description" => "In jQuery versions greater than or equal to 1.2 and before 3.5.0, passing HTML from untrusted sources - even after sanitizing it - to one of jQuery's DOM manipulation methods (i.e. .html(), .append(), and others) may execute untrusted code. This problem is patched in jQuery 3.5.0.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2020-11022-jquery","references" => ["https://github.com/jquery/jquery/security/advisories/GHSA-gxr4-xjj5-5px2","https://blog.jquery.com/2020/04/10/jquery-3-5-0-released/","https://jquery.com/upgrade-guide/3.5/","https://github.com/jquery/jquery/commit/1d61fd9407e6fbe82fe55cb0b938307aa0791f77","https://security.netapp.com/advisory/ntap-20200511-0006/","https://www.drupal.org/sa-core-2020-002","https://www.debian.org/security/2020/dsa-4693","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/VOE7P7APPRQKD4FGNHBKJPDY6FFCOH3W/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/QPN2L2XVQGUA2V5HNQJWHK3APSK3VN7K/","https://www.oracle.com/security-alerts/cpujul2020.html","http://lists.opensuse.org/opensuse-security-announce/2020-07/msg00067.html","https://security.gentoo.org/glsa/202007-03","http://lists.opensuse.org/opensuse-security-announce/2020-07/msg00085.html","https://lists.apache.org/thread.html/rdf44341677cf7eec7e9aa96dcf3f37ed709544863d619cca8c36f133\@%3Ccommits.airflow.apache.org%3E","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/AVKYXLWCLZBV2N7M46KYK4LVA5OXWPBY/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/SFP4UK4EGP4AFH2MWYJ5A5Z4I7XVFQ6B/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/SAPQVX3XDNPGFT26QAQ6AJIXZZBZ4CD4/","https://www.oracle.com/security-alerts/cpuoct2020.html","https://lists.apache.org/thread.html/rbb448222ba62c430e21e13f940be4cb5cfc373cd3bce56b48c0ffa67\@%3Cdev.flink.apache.org%3E","https://lists.apache.org/thread.html/r706cfbc098420f7113968cc377247ec3d1439bce42e679c11c609e2d\@%3Cissues.flink.apache.org%3E","http://lists.opensuse.org/opensuse-security-announce/2020-11/msg00039.html","https://lists.apache.org/thread.html/r49ce4243b4738dd763caeb27fa8ad6afb426ae3e8c011ff00b8b1f48\@%3Cissues.flink.apache.org%3E","https://www.tenable.com/security/tns-2020-10","https://www.tenable.com/security/tns-2020-11","https://www.oracle.com/security-alerts/cpujan2021.html","https://lists.apache.org/thread.html/r8f70b0f65d6bedf316ecd899371fd89e65333bc988f6326d2956735c\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r564585d97bc069137e64f521e68ba490c7c9c5b342df5d73c49a0760\@%3Cissues.flink.apache.org%3E","https://www.tenable.com/security/tns-2021-02","https://lists.debian.org/debian-lts-announce/2021/03/msg00033.html","http://packetstormsecurity.com/files/162159/jQuery-1.2-Cross-Site-Scripting.html","https://lists.apache.org/thread.html/ree3bd8ddb23df5fa4e372d11c226830ea3650056b1059f3965b3fce2\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/rede9cfaa756e050a3d83045008f84a62802fc68c17f2b4eabeaae5e4\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/re4ae96fa5c1a2fe71ccbb7b7ac1538bd0cb677be270a2bf6e2f8d108\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r54565a8f025c7c4f305355fdfd75b68eca442eebdb5f31c2e7d977ae\@%3Cissues.flink.apache.org%3E","https://www.tenable.com/security/tns-2021-10","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com//security-alerts/cpujul2021.html","https://www.oracle.com/security-alerts/cpuoct2021.html","https://lists.apache.org/thread.html/r0483ba0072783c2e1bfea613984bfb3c86e73ba8879d780dc1cc7d36\@%3Cissues.flink.apache.org%3E","https://www.oracle.com/security-alerts/cpujan2022.html","https://www.oracle.com/security-alerts/cpuapr2022.html"],"reported" => "2020-04-29","severity" => "medium"},{"affected_versions" => [">=2.001000_001,<=2.007000_001"],"cves" => ["CVE-2020-11023"],"description" => "In jQuery versions greater than or equal to 1.0.3 and before 3.5.0, passing HTML containing <option> elements from untrusted sources - even after sanitizing it - to one of jQuery's DOM manipulation methods (i.e. .html(), .append(), and others) may execute untrusted code. This problem is patched in jQuery 3.5.0.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2020-11023-jquery","references" => ["https://jquery.com/upgrade-guide/3.5/","https://github.com/jquery/jquery/security/advisories/GHSA-jpcq-cgw6-v4j6","https://blog.jquery.com/2020/04/10/jquery-3-5-0-released","https://security.netapp.com/advisory/ntap-20200511-0006/","https://www.drupal.org/sa-core-2020-002","https://www.debian.org/security/2020/dsa-4693","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/QPN2L2XVQGUA2V5HNQJWHK3APSK3VN7K/","https://www.oracle.com/security-alerts/cpujul2020.html","http://lists.opensuse.org/opensuse-security-announce/2020-07/msg00067.html","https://security.gentoo.org/glsa/202007-03","http://lists.opensuse.org/opensuse-security-announce/2020-07/msg00085.html","https://lists.apache.org/thread.html/ra3c9219fcb0b289e18e9ec5a5ebeaa5c17d6b79a201667675af6721c\@%3Cgitbox.hive.apache.org%3E","https://lists.apache.org/thread.html/r094f435595582f6b5b24b66fedf80543aa8b1d57a3688fbcc21f06ec\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/r9c5fda81e4bca8daee305b4c03283dddb383ab8428a151d4cb0b3b15\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/rf661a90a15da8da5922ba6127b3f5f8194d4ebec8855d60a0dd13248\@%3Cdev.hive.apache.org%3E","https://lists.apache.org/thread.html/radcb2aa874a79647789f3563fcbbceaf1045a029ee8806b59812a8ea\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/rd38b4185a797b324c8dd940d9213cf99fcdc2dbf1fc5a63ba7dee8c9\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/r6e97b37963926f6059ecc1e417721608723a807a76af41d4e9dbed49\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/rb69b7d8217c1a6a2100247a5d06ce610836b31e3f5d73fc113ded8e7\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/r4aadb98086ca72ed75391f54167522d91489a0d0ae25b12baa8fc7c5\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/ra374bb0299b4aa3e04edde01ebc03ed6f90cf614dad40dd428ce8f72\@%3Cgitbox.hive.apache.org%3E","https://lists.apache.org/thread.html/rb25c3bc7418ae75cba07988dafe1b6912f76a9dd7d94757878320d61\@%3Cgitbox.hive.apache.org%3E","https://lists.apache.org/thread.html/ra32c7103ded9041c7c1cb8c12c8d125a6b2f3f3270e2937ef8417fac\@%3Cgitbox.hive.apache.org%3E","https://lists.apache.org/thread.html/rf1ba79e564fe7efc56aef7c986106f1cf67a3427d08e997e088e7a93\@%3Cgitbox.hive.apache.org%3E","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/AVKYXLWCLZBV2N7M46KYK4LVA5OXWPBY/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/SFP4UK4EGP4AFH2MWYJ5A5Z4I7XVFQ6B/","https://lists.apache.org/thread.html/rab82dd040f302018c85bd07d33f5604113573514895ada523c3401d9\@%3Ccommits.hive.apache.org%3E","https://lists.apache.org/thread.html/ra406b3adfcffcb5ce8707013bdb7c35e3ffc2776a8a99022f15274c6\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/r6c4df3b33e625a44471009a172dabe6865faec8d8f21cac2303463b1\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/r1fed19c860a0d470f2a3eded12795772c8651ff583ef951ddac4918c\@%3Cgitbox.hive.apache.org%3E","https://lists.apache.org/thread.html/r0593393ca1e97b1e7e098fe69d414d6bd0a467148e9138d07e86ebbb\@%3Cissues.hive.apache.org%3E","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/SAPQVX3XDNPGFT26QAQ6AJIXZZBZ4CD4/","https://lists.apache.org/thread.html/rda99599896c3667f2cc9e9d34c7b6ef5d2bbed1f4801e1d75a2b0679\@%3Ccommits.nifi.apache.org%3E","https://www.oracle.com/security-alerts/cpuoct2020.html","https://lists.apache.org/thread.html/rbb448222ba62c430e21e13f940be4cb5cfc373cd3bce56b48c0ffa67\@%3Cdev.flink.apache.org%3E","https://lists.apache.org/thread.html/r706cfbc098420f7113968cc377247ec3d1439bce42e679c11c609e2d\@%3Cissues.flink.apache.org%3E","http://lists.opensuse.org/opensuse-security-announce/2020-11/msg00039.html","https://lists.apache.org/thread.html/r49ce4243b4738dd763caeb27fa8ad6afb426ae3e8c011ff00b8b1f48\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r2c85121a47442036c7f8353a3724aa04f8ecdfda1819d311ba4f5330\@%3Cdev.felix.apache.org%3E","https://lists.apache.org/thread.html/r3702ede0ff83a29ba3eb418f6f11c473d6e3736baba981a8dbd9c9ef\@%3Cdev.felix.apache.org%3E","https://lists.apache.org/thread.html/r9e0bd31b7da9e7403478d22652b8760c946861f8ebd7bd750844898e\@%3Cdev.felix.apache.org%3E","https://lists.apache.org/thread.html/rf0f8939596081d84be1ae6a91d6248b96a02d8388898c372ac807817\@%3Cdev.felix.apache.org%3E","https://lists.apache.org/thread.html/r07ab379471fb15644bf7a92e4a98cbc7df3cf4e736abae0cc7625fe6\@%3Cdev.felix.apache.org%3E","https://lists.apache.org/thread.html/r4dba67be3239b34861f1b9cfdf9dfb3a90272585dcce374112ed6e16\@%3Cdev.felix.apache.org%3E","https://lists.apache.org/thread.html/r9006ad2abf81d02a0ef2126bab5177987e59095b7194a487c4ea247c\@%3Ccommits.felix.apache.org%3E","https://lists.apache.org/thread.html/r55f5e066cc7301e3630ce90bbbf8d28c82212ae1f2d4871012141494\@%3Cdev.felix.apache.org%3E","https://www.oracle.com/security-alerts/cpujan2021.html","https://lists.apache.org/thread.html/r8f70b0f65d6bedf316ecd899371fd89e65333bc988f6326d2956735c\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r564585d97bc069137e64f521e68ba490c7c9c5b342df5d73c49a0760\@%3Cissues.flink.apache.org%3E","https://www.tenable.com/security/tns-2021-02","https://lists.debian.org/debian-lts-announce/2021/03/msg00033.html","http://packetstormsecurity.com/files/162160/jQuery-1.0.3-Cross-Site-Scripting.html","https://lists.apache.org/thread.html/ree3bd8ddb23df5fa4e372d11c226830ea3650056b1059f3965b3fce2\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/rede9cfaa756e050a3d83045008f84a62802fc68c17f2b4eabeaae5e4\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/re4ae96fa5c1a2fe71ccbb7b7ac1538bd0cb677be270a2bf6e2f8d108\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r54565a8f025c7c4f305355fdfd75b68eca442eebdb5f31c2e7d977ae\@%3Cissues.flink.apache.org%3E","https://www.tenable.com/security/tns-2021-10","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com//security-alerts/cpujul2021.html","https://www.oracle.com/security-alerts/cpuoct2021.html","https://lists.apache.org/thread.html/r0483ba0072783c2e1bfea613984bfb3c86e73ba8879d780dc1cc7d36\@%3Cissues.flink.apache.org%3E","https://www.oracle.com/security-alerts/cpujan2022.html","https://www.oracle.com/security-alerts/cpuapr2022.html"],"reported" => "2020-04-29","severity" => "medium"},{"affected_versions" => [">=2.001000_001,<=2.007000_001"],"cves" => ["CVE-2019-11358"],"description" => "jQuery before 3.4.0, as used in Drupal, Backdrop CMS, and other products, mishandles jQuery.extend(true, {}, ...) because of Object.prototype pollution. If an unsanitized source object contained an enumerable __proto__ property, it could extend the native Object.prototype.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2019-11358-jquery","references" => ["https://www.drupal.org/sa-core-2019-006","https://snyk.io/vuln/SNYK-JS-JQUERY-174006","https://github.com/jquery/jquery/pull/4333","https://github.com/jquery/jquery/commit/753d591aea698e57d6db58c9f722cd0808619b1b","https://blog.jquery.com/2019/04/10/jquery-3-4-0-released/","https://backdropcms.org/security/backdrop-sa-core-2019-009","https://www.debian.org/security/2019/dsa-4434","https://seclists.org/bugtraq/2019/Apr/32","http://www.securityfocus.com/bid/108023","https://lists.apache.org/thread.html/b736d0784cf02f5a30fbb4c5902762a15ad6d47e17e2c5a17b7d6205\@%3Ccommits.airflow.apache.org%3E","https://lists.apache.org/thread.html/08720ef215ee7ab3386c05a1a90a7d1c852bf0706f176a7816bf65fc\@%3Ccommits.airflow.apache.org%3E","https://lists.apache.org/thread.html/88fb0362fd40e5b605ea8149f63241537b8b6fb5bfa315391fc5cbb7\@%3Ccommits.airflow.apache.org%3E","https://lists.apache.org/thread.html/6097cdbd6f0a337bedd9bb5cc441b2d525ff002a96531de367e4259f\@%3Ccommits.airflow.apache.org%3E","https://lists.apache.org/thread.html/5928aa293e39d248266472210c50f176cac1535220f2486e6a7fa844\@%3Ccommits.airflow.apache.org%3E","https://lists.debian.org/debian-lts-announce/2019/05/msg00006.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/5IABSKTYZ5JUGL735UKGXL5YPRYOPUYI/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/4UOAZIFCSZ3ENEFOR5IXX6NFAD3HV7FA/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/KYH3OAGR2RTCHRA5NOKX2TES7SNQMWGO/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/RLXRX23725JL366CNZGJZ7AQQB7LHQ6F/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/WZW27UCJ5CYFL4KFFFMYMIBNMIU2ALG5/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/QV3PKZC3PQCO3273HAT76PAQZFBEO4KP/","https://seclists.org/bugtraq/2019/May/18","http://packetstormsecurity.com/files/152787/dotCMS-5.1.1-Vulnerable-Dependencies.html","http://seclists.org/fulldisclosure/2019/May/13","http://seclists.org/fulldisclosure/2019/May/11","http://seclists.org/fulldisclosure/2019/May/10","https://lists.debian.org/debian-lts-announce/2019/05/msg00029.html","http://www.openwall.com/lists/oss-security/2019/06/03/2","http://packetstormsecurity.com/files/153237/RetireJS-CORS-Issue-Script-Execution.html","https://access.redhat.com/errata/RHSA-2019:1456","https://www.debian.org/security/2019/dsa-4460","https://seclists.org/bugtraq/2019/Jun/12","https://www.oracle.com/technetwork/security-advisory/cpujul2019-5072835.html","https://www.privacy-wise.com/mitigating-cve-2019-11358-in-old-versions-of-jquery/","http://lists.opensuse.org/opensuse-security-announce/2019-08/msg00006.html","https://access.redhat.com/errata/RHBA-2019:1570","http://lists.opensuse.org/opensuse-security-announce/2019-08/msg00025.html","https://lists.apache.org/thread.html/ba79cf1658741e9f146e4c59b50aee56656ea95d841d358d006c18b6\@%3Ccommits.roller.apache.org%3E","https://access.redhat.com/errata/RHSA-2019:2587","https://security.netapp.com/advisory/ntap-20190919-0001/","https://access.redhat.com/errata/RHSA-2019:3023","https://access.redhat.com/errata/RHSA-2019:3024","https://www.oracle.com/technetwork/security-advisory/cpuoct2019-5072832.html","https://lists.apache.org/thread.html/b0656d359c7d40ec9f39c8cc61bca66802ef9a2a12ee199f5b0c1442\@%3Cdev.drill.apache.org%3E","https://lists.apache.org/thread.html/519eb0fd45642dcecd9ff74cb3e71c20a4753f7d82e2f07864b5108f\@%3Cdev.drill.apache.org%3E","https://lists.apache.org/thread.html/f9bc3e55f4e28d1dcd1a69aae6d53e609a758e34d2869b4d798e13cc\@%3Cissues.drill.apache.org%3E","https://www.synology.com/security/advisory/Synology_SA_19_19","https://lists.apache.org/thread.html/bcce5a9c532b386c68dab2f6b3ce8b0cc9b950ec551766e76391caa3\@%3Ccommits.nifi.apache.org%3E","https://www.tenable.com/security/tns-2019-08","https://www.oracle.com/security-alerts/cpujan2020.html","https://lists.apache.org/thread.html/rca37935d661f4689cb4119f1b3b224413b22be161b678e6e6ce0c69b\@%3Ccommits.nifi.apache.org%3E","https://lists.debian.org/debian-lts-announce/2020/02/msg00024.html","http://packetstormsecurity.com/files/156743/OctoberCMS-Insecure-Dependencies.html","https://www.tenable.com/security/tns-2020-02","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.apache.org/thread.html/r38f0d1aa3c923c22977fe7376508f030f22e22c1379fbb155bf29766\@%3Cdev.syncope.apache.org%3E","https://lists.apache.org/thread.html/r7aac081cbddb6baa24b75e74abf0929bf309b176755a53e3ed810355\@%3Cdev.flink.apache.org%3E","https://lists.apache.org/thread.html/rac25da84ecdcd36f6de5ad0d255f4e967209bbbebddb285e231da37d\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r2041a75d3fc09dec55adfd95d598b38d22715303f65c997c054844c9\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r7e8ebccb7c022e41295f6fdb7b971209b83702339f872ddd8cf8bf73\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r41b5bfe009c845f67d4f68948cc9419ac2d62e287804aafd72892b08\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r2baacab6e0acb5a2092eb46ae04fd6c3e8277b4fd79b1ffb7f3254fa\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r7d64895cc4dff84d0becfc572b20c0e4bf9bfa7b10c6f5f73e783734\@%3Cdev.storm.apache.org%3E","https://www.oracle.com/security-alerts/cpujul2020.html","https://www.oracle.com/security-alerts/cpuoct2020.html","https://kb.pulsesecure.net/articles/Pulse_Security_Advisories/SA44601","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com//security-alerts/cpujul2021.html","https://www.oracle.com/security-alerts/cpuoct2021.html","https://www.oracle.com/security-alerts/cpujan2022.html"],"reported" => "2019-04-20","severity" => "medium"},{"affected_versions" => [">=2.001000_001,<=2.007000_001"],"cves" => ["CVE-2015-9251"],"description" => "jQuery before 3.0.0 is vulnerable to Cross-site Scripting (XSS) attacks when a cross-domain Ajax request is performed without the dataType option, causing text/javascript responses to be executed.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2015-9251-jquery","references" => ["https://snyk.io/vuln/npm:jquery:20150627","https://github.com/jquery/jquery/pull/2588/commits/c254d308a7d3f1eac4d0b42837804cfffcba4bb2","https://github.com/jquery/jquery/pull/2588","https://github.com/jquery/jquery/issues/2432","https://github.com/jquery/jquery/commit/f60729f3903d17917dc351f3ac87794de379b0cc","https://sw.aveva.com/hubfs/assets-2018/pdf/security-bulletin/SecurityBulletin_LFSec126.pdf","https://ics-cert.us-cert.gov/advisories/ICSA-18-212-04","http://www.oracle.com/technetwork/security-advisory/cpuoct2018-4428296.html","http://www.securityfocus.com/bid/105658","https://www.oracle.com/technetwork/security-advisory/cpujan2019-5072801.html","https://www.oracle.com/technetwork/security-advisory/cpuapr2019-5072813.html","https://seclists.org/bugtraq/2019/May/18","http://packetstormsecurity.com/files/152787/dotCMS-5.1.1-Vulnerable-Dependencies.html","http://seclists.org/fulldisclosure/2019/May/13","http://seclists.org/fulldisclosure/2019/May/11","http://seclists.org/fulldisclosure/2019/May/10","http://packetstormsecurity.com/files/153237/RetireJS-CORS-Issue-Script-Execution.html","https://www.oracle.com/technetwork/security-advisory/cpujul2019-5072835.html","https://lists.apache.org/thread.html/54df3aeb4239b64b50b356f0ca6f986e3c4ca5b84c515dce077c7854\@%3Cuser.flink.apache.org%3E","https://lists.apache.org/thread.html/10f0f3aefd51444d1198c65f44ffdf2d78ca3359423dbc1c168c9731\@%3Cdev.flink.apache.org%3E","https://lists.apache.org/thread.html/17ff53f7999e74fbe3cc0ceb4e1c3b00b180b7c5afec8e978837bc49\@%3Cuser.flink.apache.org%3E","https://lists.apache.org/thread.html/52bafac05ad174000ea465fe275fd3cc7bd5c25535a7631c0bc9bfb2\@%3Cuser.flink.apache.org%3E","https://lists.apache.org/thread.html/ba79cf1658741e9f146e4c59b50aee56656ea95d841d358d006c18b6\@%3Ccommits.roller.apache.org%3E","https://www.oracle.com/technetwork/security-advisory/cpuoct2019-5072832.html","https://lists.apache.org/thread.html/b0656d359c7d40ec9f39c8cc61bca66802ef9a2a12ee199f5b0c1442\@%3Cdev.drill.apache.org%3E","https://lists.apache.org/thread.html/519eb0fd45642dcecd9ff74cb3e71c20a4753f7d82e2f07864b5108f\@%3Cdev.drill.apache.org%3E","https://www.tenable.com/security/tns-2019-08","https://lists.apache.org/thread.html/f9bc3e55f4e28d1dcd1a69aae6d53e609a758e34d2869b4d798e13cc\@%3Cissues.drill.apache.org%3E","https://www.oracle.com/security-alerts/cpujan2020.html","https://access.redhat.com/errata/RHSA-2020:0481","https://access.redhat.com/errata/RHSA-2020:0729","http://packetstormsecurity.com/files/156743/OctoberCMS-Insecure-Dependencies.html","http://lists.opensuse.org/opensuse-security-announce/2020-03/msg00041.html","https://www.oracle.com/security-alerts/cpuapr2020.html","https://www.oracle.com/security-alerts/cpujul2020.html","https://www.oracle.com/security-alerts/cpuoct2020.html","https://kb.pulsesecure.net/articles/Pulse_Security_Advisories/SA44601","https://security.netapp.com/advisory/ntap-20210108-0004/"],"reported" => "2018-01-18","severity" => "medium"},{"affected_versions" => [">=2.001000_001,<=2.007000_001"],"cves" => ["CVE-2011-4969"],"description" => "Cross-site scripting (XSS) vulnerability in jQuery before 1.6.3, when using location.hash to select elements, allows remote attackers to inject arbitrary web script or HTML via a crafted tag.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2011-4969-jquery","references" => ["https://github.com/jquery/jquery/commit/db9e023e62c1ff5d8f21ed9868ab6878da2005e9","http://bugs.jquery.com/ticket/9521","http://www.ubuntu.com/usn/USN-1722-1","http://blog.jquery.com/2011/09/01/jquery-1-6-3-released/","http://www.osvdb.org/80056","http://www.openwall.com/lists/oss-security/2013/01/31/3","http://blog.mindedsecurity.com/2011/07/jquery-is-sink.html","https://h20566.www2.hpe.com/portal/site/hpsc/public/kb/docDisplay?docId=emr_na-c05111017","https://h20566.www2.hpe.com/portal/site/hpsc/public/kb/docDisplay?docId=emr_na-c05158380","https://h20566.www2.hpe.com/portal/site/hpsc/public/kb/docDisplay?docId=emr_na-c05232730","http://www.securityfocus.com/bid/58458","http://www.securitytracker.com/id/1036620","https://security.netapp.com/advisory/ntap-20190416-0007/","https://lists.apache.org/thread.html/ff8dcfe29377088ab655fda9d585dccd5b1f07fabd94ae84fd60a7f8\@%3Ccommits.pulsar.apache.org%3E"],"reported" => "2013-03-08","severity" => undef},{"affected_versions" => [">=2.001000_001,<=2.007000_001"],"cves" => ["CVE-2012-6708"],"description" => "jQuery before 1.9.0 is vulnerable to Cross-site Scripting (XSS) attacks. The jQuery(strInput) function does not differentiate selectors from HTML in a reliable fashion. In vulnerable versions, jQuery determined whether the input was HTML by looking for the '<' character anywhere in the string, giving attackers more flexibility when attempting to construct a malicious payload. In fixed versions, jQuery only deems the input to be HTML if it explicitly starts with the '<' character, limiting exploitability only to attackers who can control the beginning of a string, which is far less common.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2012-6708-jquery","references" => ["https://snyk.io/vuln/npm:jquery:20120206","https://github.com/jquery/jquery/commit/05531fc4080ae24070930d15ae0cea7ae056457d","https://bugs.jquery.com/ticket/11290","http://www.securityfocus.com/bid/102792","https://help.ecostruxureit.com/display/public/UADCE725/Security+fixes+in+StruxureWare+Data+Center+Expert+v7.6.0","http://packetstormsecurity.com/files/153237/RetireJS-CORS-Issue-Script-Execution.html","https://lists.apache.org/thread.html/b0656d359c7d40ec9f39c8cc61bca66802ef9a2a12ee199f5b0c1442\@%3Cdev.drill.apache.org%3E","https://lists.apache.org/thread.html/519eb0fd45642dcecd9ff74cb3e71c20a4753f7d82e2f07864b5108f\@%3Cdev.drill.apache.org%3E","http://lists.opensuse.org/opensuse-security-announce/2020-03/msg00041.html","https://lists.apache.org/thread.html/f9bc3e55f4e28d1dcd1a69aae6d53e609a758e34d2869b4d798e13cc\@%3Cissues.drill.apache.org%3E","http://packetstormsecurity.com/files/161972/Linksys-EA7500-2.0.8.194281-Cross-Site-Scripting.html"],"reported" => "2018-01-18","severity" => "medium"},{"affected_versions" => [">=2.001000_001,<=2.007000_001"],"cves" => ["CVE-2020-7656"],"description" => "jquery prior to 1.9.0 allows Cross-site Scripting attacks via the load method. The load method fails to recognize and remove \"<script>\" HTML tags that contain a whitespace character, i.e: \"</script >\", which results in the enclosed script logic to be executed.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2020-7656-jquery","references" => ["https://snyk.io/vuln/SNYK-JS-JQUERY-569619","https://security.netapp.com/advisory/ntap-20200528-0001/"],"reported" => "2020-05-19","severity" => "medium"},{"affected_versions" => [">=2.001000_001,<=2.007000_001"],"cves" => ["CVE-2019-5428"],"description" => "Prototype Pollution is a vulnerability affecting JavaScript. Prototype Pollution refers to the ability to inject properties into existing JavaScript language construct prototypes, such as objects. JavaScript allows all Object attributes to be altered, including their magical attributes such as _proto_, constructor and prototype. An attacker manipulates these attributes to overwrite, or pollute, a JavaScript application object prototype of the base object by injecting other values. Properties on the Object.prototype are then inherited by all the JavaScript objects through the prototype chain. When that happens, this leads to either denial of service by triggering JavaScript exceptions, or it tampers with the application source code to force the code path that the attacker injects, thereby leading to remote code execution.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2019-5428-jquery","references" => ["https://security.snyk.io/vuln/SNYK-JS-JQUERY-174006"],"reported" => undef,"severity" => undef},{"affected_versions" => [">=2.001000_001,<=2.007000_001"],"cves" => ["CVE-2014-6071"],"description" => "jQuery 1.4.2 allows remote attackers to conduct cross-site scripting (XSS) attacks via vectors related to use of the text method inside after.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2014-6071-jquery","references" => ["https://bugzilla.redhat.com/show_bug.cgi?id=1136683","http://seclists.org/fulldisclosure/2014/Sep/10","https://help.ecostruxureit.com/display/public/UADCE725/Security+fixes+in+StruxureWare+Data+Center+Expert+v7.6.0"],"reported" => "2018-01-16","severity" => "medium"},{"affected_versions" => [">=2.007000_002,<=2.055001"],"cves" => ["CVE-2020-11022"],"description" => "In jQuery versions greater than or equal to 1.2 and before 3.5.0, passing HTML from untrusted sources - even after sanitizing it - to one of jQuery's DOM manipulation methods (i.e. .html(), .append(), and others) may execute untrusted code. This problem is patched in jQuery 3.5.0.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2020-11022-jquery","references" => ["https://github.com/jquery/jquery/security/advisories/GHSA-gxr4-xjj5-5px2","https://blog.jquery.com/2020/04/10/jquery-3-5-0-released/","https://jquery.com/upgrade-guide/3.5/","https://github.com/jquery/jquery/commit/1d61fd9407e6fbe82fe55cb0b938307aa0791f77","https://security.netapp.com/advisory/ntap-20200511-0006/","https://www.drupal.org/sa-core-2020-002","https://www.debian.org/security/2020/dsa-4693","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/VOE7P7APPRQKD4FGNHBKJPDY6FFCOH3W/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/QPN2L2XVQGUA2V5HNQJWHK3APSK3VN7K/","https://www.oracle.com/security-alerts/cpujul2020.html","http://lists.opensuse.org/opensuse-security-announce/2020-07/msg00067.html","https://security.gentoo.org/glsa/202007-03","http://lists.opensuse.org/opensuse-security-announce/2020-07/msg00085.html","https://lists.apache.org/thread.html/rdf44341677cf7eec7e9aa96dcf3f37ed709544863d619cca8c36f133\@%3Ccommits.airflow.apache.org%3E","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/AVKYXLWCLZBV2N7M46KYK4LVA5OXWPBY/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/SFP4UK4EGP4AFH2MWYJ5A5Z4I7XVFQ6B/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/SAPQVX3XDNPGFT26QAQ6AJIXZZBZ4CD4/","https://www.oracle.com/security-alerts/cpuoct2020.html","https://lists.apache.org/thread.html/rbb448222ba62c430e21e13f940be4cb5cfc373cd3bce56b48c0ffa67\@%3Cdev.flink.apache.org%3E","https://lists.apache.org/thread.html/r706cfbc098420f7113968cc377247ec3d1439bce42e679c11c609e2d\@%3Cissues.flink.apache.org%3E","http://lists.opensuse.org/opensuse-security-announce/2020-11/msg00039.html","https://lists.apache.org/thread.html/r49ce4243b4738dd763caeb27fa8ad6afb426ae3e8c011ff00b8b1f48\@%3Cissues.flink.apache.org%3E","https://www.tenable.com/security/tns-2020-10","https://www.tenable.com/security/tns-2020-11","https://www.oracle.com/security-alerts/cpujan2021.html","https://lists.apache.org/thread.html/r8f70b0f65d6bedf316ecd899371fd89e65333bc988f6326d2956735c\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r564585d97bc069137e64f521e68ba490c7c9c5b342df5d73c49a0760\@%3Cissues.flink.apache.org%3E","https://www.tenable.com/security/tns-2021-02","https://lists.debian.org/debian-lts-announce/2021/03/msg00033.html","http://packetstormsecurity.com/files/162159/jQuery-1.2-Cross-Site-Scripting.html","https://lists.apache.org/thread.html/ree3bd8ddb23df5fa4e372d11c226830ea3650056b1059f3965b3fce2\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/rede9cfaa756e050a3d83045008f84a62802fc68c17f2b4eabeaae5e4\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/re4ae96fa5c1a2fe71ccbb7b7ac1538bd0cb677be270a2bf6e2f8d108\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r54565a8f025c7c4f305355fdfd75b68eca442eebdb5f31c2e7d977ae\@%3Cissues.flink.apache.org%3E","https://www.tenable.com/security/tns-2021-10","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com//security-alerts/cpujul2021.html","https://www.oracle.com/security-alerts/cpuoct2021.html","https://lists.apache.org/thread.html/r0483ba0072783c2e1bfea613984bfb3c86e73ba8879d780dc1cc7d36\@%3Cissues.flink.apache.org%3E","https://www.oracle.com/security-alerts/cpujan2022.html","https://www.oracle.com/security-alerts/cpuapr2022.html"],"reported" => "2020-04-29","severity" => "medium"},{"affected_versions" => [">=2.007000_002,<=2.055001"],"cves" => ["CVE-2020-11023"],"description" => "In jQuery versions greater than or equal to 1.0.3 and before 3.5.0, passing HTML containing <option> elements from untrusted sources - even after sanitizing it - to one of jQuery's DOM manipulation methods (i.e. .html(), .append(), and others) may execute untrusted code. This problem is patched in jQuery 3.5.0.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2020-11023-jquery","references" => ["https://jquery.com/upgrade-guide/3.5/","https://github.com/jquery/jquery/security/advisories/GHSA-jpcq-cgw6-v4j6","https://blog.jquery.com/2020/04/10/jquery-3-5-0-released","https://security.netapp.com/advisory/ntap-20200511-0006/","https://www.drupal.org/sa-core-2020-002","https://www.debian.org/security/2020/dsa-4693","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/QPN2L2XVQGUA2V5HNQJWHK3APSK3VN7K/","https://www.oracle.com/security-alerts/cpujul2020.html","http://lists.opensuse.org/opensuse-security-announce/2020-07/msg00067.html","https://security.gentoo.org/glsa/202007-03","http://lists.opensuse.org/opensuse-security-announce/2020-07/msg00085.html","https://lists.apache.org/thread.html/ra3c9219fcb0b289e18e9ec5a5ebeaa5c17d6b79a201667675af6721c\@%3Cgitbox.hive.apache.org%3E","https://lists.apache.org/thread.html/r094f435595582f6b5b24b66fedf80543aa8b1d57a3688fbcc21f06ec\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/r9c5fda81e4bca8daee305b4c03283dddb383ab8428a151d4cb0b3b15\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/rf661a90a15da8da5922ba6127b3f5f8194d4ebec8855d60a0dd13248\@%3Cdev.hive.apache.org%3E","https://lists.apache.org/thread.html/radcb2aa874a79647789f3563fcbbceaf1045a029ee8806b59812a8ea\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/rd38b4185a797b324c8dd940d9213cf99fcdc2dbf1fc5a63ba7dee8c9\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/r6e97b37963926f6059ecc1e417721608723a807a76af41d4e9dbed49\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/rb69b7d8217c1a6a2100247a5d06ce610836b31e3f5d73fc113ded8e7\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/r4aadb98086ca72ed75391f54167522d91489a0d0ae25b12baa8fc7c5\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/ra374bb0299b4aa3e04edde01ebc03ed6f90cf614dad40dd428ce8f72\@%3Cgitbox.hive.apache.org%3E","https://lists.apache.org/thread.html/rb25c3bc7418ae75cba07988dafe1b6912f76a9dd7d94757878320d61\@%3Cgitbox.hive.apache.org%3E","https://lists.apache.org/thread.html/ra32c7103ded9041c7c1cb8c12c8d125a6b2f3f3270e2937ef8417fac\@%3Cgitbox.hive.apache.org%3E","https://lists.apache.org/thread.html/rf1ba79e564fe7efc56aef7c986106f1cf67a3427d08e997e088e7a93\@%3Cgitbox.hive.apache.org%3E","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/AVKYXLWCLZBV2N7M46KYK4LVA5OXWPBY/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/SFP4UK4EGP4AFH2MWYJ5A5Z4I7XVFQ6B/","https://lists.apache.org/thread.html/rab82dd040f302018c85bd07d33f5604113573514895ada523c3401d9\@%3Ccommits.hive.apache.org%3E","https://lists.apache.org/thread.html/ra406b3adfcffcb5ce8707013bdb7c35e3ffc2776a8a99022f15274c6\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/r6c4df3b33e625a44471009a172dabe6865faec8d8f21cac2303463b1\@%3Cissues.hive.apache.org%3E","https://lists.apache.org/thread.html/r1fed19c860a0d470f2a3eded12795772c8651ff583ef951ddac4918c\@%3Cgitbox.hive.apache.org%3E","https://lists.apache.org/thread.html/r0593393ca1e97b1e7e098fe69d414d6bd0a467148e9138d07e86ebbb\@%3Cissues.hive.apache.org%3E","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/SAPQVX3XDNPGFT26QAQ6AJIXZZBZ4CD4/","https://lists.apache.org/thread.html/rda99599896c3667f2cc9e9d34c7b6ef5d2bbed1f4801e1d75a2b0679\@%3Ccommits.nifi.apache.org%3E","https://www.oracle.com/security-alerts/cpuoct2020.html","https://lists.apache.org/thread.html/rbb448222ba62c430e21e13f940be4cb5cfc373cd3bce56b48c0ffa67\@%3Cdev.flink.apache.org%3E","https://lists.apache.org/thread.html/r706cfbc098420f7113968cc377247ec3d1439bce42e679c11c609e2d\@%3Cissues.flink.apache.org%3E","http://lists.opensuse.org/opensuse-security-announce/2020-11/msg00039.html","https://lists.apache.org/thread.html/r49ce4243b4738dd763caeb27fa8ad6afb426ae3e8c011ff00b8b1f48\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r2c85121a47442036c7f8353a3724aa04f8ecdfda1819d311ba4f5330\@%3Cdev.felix.apache.org%3E","https://lists.apache.org/thread.html/r3702ede0ff83a29ba3eb418f6f11c473d6e3736baba981a8dbd9c9ef\@%3Cdev.felix.apache.org%3E","https://lists.apache.org/thread.html/r9e0bd31b7da9e7403478d22652b8760c946861f8ebd7bd750844898e\@%3Cdev.felix.apache.org%3E","https://lists.apache.org/thread.html/rf0f8939596081d84be1ae6a91d6248b96a02d8388898c372ac807817\@%3Cdev.felix.apache.org%3E","https://lists.apache.org/thread.html/r07ab379471fb15644bf7a92e4a98cbc7df3cf4e736abae0cc7625fe6\@%3Cdev.felix.apache.org%3E","https://lists.apache.org/thread.html/r4dba67be3239b34861f1b9cfdf9dfb3a90272585dcce374112ed6e16\@%3Cdev.felix.apache.org%3E","https://lists.apache.org/thread.html/r9006ad2abf81d02a0ef2126bab5177987e59095b7194a487c4ea247c\@%3Ccommits.felix.apache.org%3E","https://lists.apache.org/thread.html/r55f5e066cc7301e3630ce90bbbf8d28c82212ae1f2d4871012141494\@%3Cdev.felix.apache.org%3E","https://www.oracle.com/security-alerts/cpujan2021.html","https://lists.apache.org/thread.html/r8f70b0f65d6bedf316ecd899371fd89e65333bc988f6326d2956735c\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r564585d97bc069137e64f521e68ba490c7c9c5b342df5d73c49a0760\@%3Cissues.flink.apache.org%3E","https://www.tenable.com/security/tns-2021-02","https://lists.debian.org/debian-lts-announce/2021/03/msg00033.html","http://packetstormsecurity.com/files/162160/jQuery-1.0.3-Cross-Site-Scripting.html","https://lists.apache.org/thread.html/ree3bd8ddb23df5fa4e372d11c226830ea3650056b1059f3965b3fce2\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/rede9cfaa756e050a3d83045008f84a62802fc68c17f2b4eabeaae5e4\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/re4ae96fa5c1a2fe71ccbb7b7ac1538bd0cb677be270a2bf6e2f8d108\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r54565a8f025c7c4f305355fdfd75b68eca442eebdb5f31c2e7d977ae\@%3Cissues.flink.apache.org%3E","https://www.tenable.com/security/tns-2021-10","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com//security-alerts/cpujul2021.html","https://www.oracle.com/security-alerts/cpuoct2021.html","https://lists.apache.org/thread.html/r0483ba0072783c2e1bfea613984bfb3c86e73ba8879d780dc1cc7d36\@%3Cissues.flink.apache.org%3E","https://www.oracle.com/security-alerts/cpujan2022.html","https://www.oracle.com/security-alerts/cpuapr2022.html"],"reported" => "2020-04-29","severity" => "medium"},{"affected_versions" => [">=2.007000_002,<=2.055001"],"cves" => ["CVE-2019-11358"],"description" => "jQuery before 3.4.0, as used in Drupal, Backdrop CMS, and other products, mishandles jQuery.extend(true, {}, ...) because of Object.prototype pollution. If an unsanitized source object contained an enumerable __proto__ property, it could extend the native Object.prototype.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2019-11358-jquery","references" => ["https://www.drupal.org/sa-core-2019-006","https://snyk.io/vuln/SNYK-JS-JQUERY-174006","https://github.com/jquery/jquery/pull/4333","https://github.com/jquery/jquery/commit/753d591aea698e57d6db58c9f722cd0808619b1b","https://blog.jquery.com/2019/04/10/jquery-3-4-0-released/","https://backdropcms.org/security/backdrop-sa-core-2019-009","https://www.debian.org/security/2019/dsa-4434","https://seclists.org/bugtraq/2019/Apr/32","http://www.securityfocus.com/bid/108023","https://lists.apache.org/thread.html/b736d0784cf02f5a30fbb4c5902762a15ad6d47e17e2c5a17b7d6205\@%3Ccommits.airflow.apache.org%3E","https://lists.apache.org/thread.html/08720ef215ee7ab3386c05a1a90a7d1c852bf0706f176a7816bf65fc\@%3Ccommits.airflow.apache.org%3E","https://lists.apache.org/thread.html/88fb0362fd40e5b605ea8149f63241537b8b6fb5bfa315391fc5cbb7\@%3Ccommits.airflow.apache.org%3E","https://lists.apache.org/thread.html/6097cdbd6f0a337bedd9bb5cc441b2d525ff002a96531de367e4259f\@%3Ccommits.airflow.apache.org%3E","https://lists.apache.org/thread.html/5928aa293e39d248266472210c50f176cac1535220f2486e6a7fa844\@%3Ccommits.airflow.apache.org%3E","https://lists.debian.org/debian-lts-announce/2019/05/msg00006.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/5IABSKTYZ5JUGL735UKGXL5YPRYOPUYI/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/4UOAZIFCSZ3ENEFOR5IXX6NFAD3HV7FA/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/KYH3OAGR2RTCHRA5NOKX2TES7SNQMWGO/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/RLXRX23725JL366CNZGJZ7AQQB7LHQ6F/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/WZW27UCJ5CYFL4KFFFMYMIBNMIU2ALG5/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/QV3PKZC3PQCO3273HAT76PAQZFBEO4KP/","https://seclists.org/bugtraq/2019/May/18","http://packetstormsecurity.com/files/152787/dotCMS-5.1.1-Vulnerable-Dependencies.html","http://seclists.org/fulldisclosure/2019/May/13","http://seclists.org/fulldisclosure/2019/May/11","http://seclists.org/fulldisclosure/2019/May/10","https://lists.debian.org/debian-lts-announce/2019/05/msg00029.html","http://www.openwall.com/lists/oss-security/2019/06/03/2","http://packetstormsecurity.com/files/153237/RetireJS-CORS-Issue-Script-Execution.html","https://access.redhat.com/errata/RHSA-2019:1456","https://www.debian.org/security/2019/dsa-4460","https://seclists.org/bugtraq/2019/Jun/12","https://www.oracle.com/technetwork/security-advisory/cpujul2019-5072835.html","https://www.privacy-wise.com/mitigating-cve-2019-11358-in-old-versions-of-jquery/","http://lists.opensuse.org/opensuse-security-announce/2019-08/msg00006.html","https://access.redhat.com/errata/RHBA-2019:1570","http://lists.opensuse.org/opensuse-security-announce/2019-08/msg00025.html","https://lists.apache.org/thread.html/ba79cf1658741e9f146e4c59b50aee56656ea95d841d358d006c18b6\@%3Ccommits.roller.apache.org%3E","https://access.redhat.com/errata/RHSA-2019:2587","https://security.netapp.com/advisory/ntap-20190919-0001/","https://access.redhat.com/errata/RHSA-2019:3023","https://access.redhat.com/errata/RHSA-2019:3024","https://www.oracle.com/technetwork/security-advisory/cpuoct2019-5072832.html","https://lists.apache.org/thread.html/b0656d359c7d40ec9f39c8cc61bca66802ef9a2a12ee199f5b0c1442\@%3Cdev.drill.apache.org%3E","https://lists.apache.org/thread.html/519eb0fd45642dcecd9ff74cb3e71c20a4753f7d82e2f07864b5108f\@%3Cdev.drill.apache.org%3E","https://lists.apache.org/thread.html/f9bc3e55f4e28d1dcd1a69aae6d53e609a758e34d2869b4d798e13cc\@%3Cissues.drill.apache.org%3E","https://www.synology.com/security/advisory/Synology_SA_19_19","https://lists.apache.org/thread.html/bcce5a9c532b386c68dab2f6b3ce8b0cc9b950ec551766e76391caa3\@%3Ccommits.nifi.apache.org%3E","https://www.tenable.com/security/tns-2019-08","https://www.oracle.com/security-alerts/cpujan2020.html","https://lists.apache.org/thread.html/rca37935d661f4689cb4119f1b3b224413b22be161b678e6e6ce0c69b\@%3Ccommits.nifi.apache.org%3E","https://lists.debian.org/debian-lts-announce/2020/02/msg00024.html","http://packetstormsecurity.com/files/156743/OctoberCMS-Insecure-Dependencies.html","https://www.tenable.com/security/tns-2020-02","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.apache.org/thread.html/r38f0d1aa3c923c22977fe7376508f030f22e22c1379fbb155bf29766\@%3Cdev.syncope.apache.org%3E","https://lists.apache.org/thread.html/r7aac081cbddb6baa24b75e74abf0929bf309b176755a53e3ed810355\@%3Cdev.flink.apache.org%3E","https://lists.apache.org/thread.html/rac25da84ecdcd36f6de5ad0d255f4e967209bbbebddb285e231da37d\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r2041a75d3fc09dec55adfd95d598b38d22715303f65c997c054844c9\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r7e8ebccb7c022e41295f6fdb7b971209b83702339f872ddd8cf8bf73\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r41b5bfe009c845f67d4f68948cc9419ac2d62e287804aafd72892b08\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r2baacab6e0acb5a2092eb46ae04fd6c3e8277b4fd79b1ffb7f3254fa\@%3Cissues.flink.apache.org%3E","https://lists.apache.org/thread.html/r7d64895cc4dff84d0becfc572b20c0e4bf9bfa7b10c6f5f73e783734\@%3Cdev.storm.apache.org%3E","https://www.oracle.com/security-alerts/cpujul2020.html","https://www.oracle.com/security-alerts/cpuoct2020.html","https://kb.pulsesecure.net/articles/Pulse_Security_Advisories/SA44601","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com//security-alerts/cpujul2021.html","https://www.oracle.com/security-alerts/cpuoct2021.html","https://www.oracle.com/security-alerts/cpujan2022.html"],"reported" => "2019-04-20","severity" => "medium"},{"affected_versions" => [">=2.007000_002,<=2.055001"],"cves" => ["CVE-2015-9251"],"description" => "jQuery before 3.0.0 is vulnerable to Cross-site Scripting (XSS) attacks when a cross-domain Ajax request is performed without the dataType option, causing text/javascript responses to be executed.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2015-9251-jquery","references" => ["https://snyk.io/vuln/npm:jquery:20150627","https://github.com/jquery/jquery/pull/2588/commits/c254d308a7d3f1eac4d0b42837804cfffcba4bb2","https://github.com/jquery/jquery/pull/2588","https://github.com/jquery/jquery/issues/2432","https://github.com/jquery/jquery/commit/f60729f3903d17917dc351f3ac87794de379b0cc","https://sw.aveva.com/hubfs/assets-2018/pdf/security-bulletin/SecurityBulletin_LFSec126.pdf","https://ics-cert.us-cert.gov/advisories/ICSA-18-212-04","http://www.oracle.com/technetwork/security-advisory/cpuoct2018-4428296.html","http://www.securityfocus.com/bid/105658","https://www.oracle.com/technetwork/security-advisory/cpujan2019-5072801.html","https://www.oracle.com/technetwork/security-advisory/cpuapr2019-5072813.html","https://seclists.org/bugtraq/2019/May/18","http://packetstormsecurity.com/files/152787/dotCMS-5.1.1-Vulnerable-Dependencies.html","http://seclists.org/fulldisclosure/2019/May/13","http://seclists.org/fulldisclosure/2019/May/11","http://seclists.org/fulldisclosure/2019/May/10","http://packetstormsecurity.com/files/153237/RetireJS-CORS-Issue-Script-Execution.html","https://www.oracle.com/technetwork/security-advisory/cpujul2019-5072835.html","https://lists.apache.org/thread.html/54df3aeb4239b64b50b356f0ca6f986e3c4ca5b84c515dce077c7854\@%3Cuser.flink.apache.org%3E","https://lists.apache.org/thread.html/10f0f3aefd51444d1198c65f44ffdf2d78ca3359423dbc1c168c9731\@%3Cdev.flink.apache.org%3E","https://lists.apache.org/thread.html/17ff53f7999e74fbe3cc0ceb4e1c3b00b180b7c5afec8e978837bc49\@%3Cuser.flink.apache.org%3E","https://lists.apache.org/thread.html/52bafac05ad174000ea465fe275fd3cc7bd5c25535a7631c0bc9bfb2\@%3Cuser.flink.apache.org%3E","https://lists.apache.org/thread.html/ba79cf1658741e9f146e4c59b50aee56656ea95d841d358d006c18b6\@%3Ccommits.roller.apache.org%3E","https://www.oracle.com/technetwork/security-advisory/cpuoct2019-5072832.html","https://lists.apache.org/thread.html/b0656d359c7d40ec9f39c8cc61bca66802ef9a2a12ee199f5b0c1442\@%3Cdev.drill.apache.org%3E","https://lists.apache.org/thread.html/519eb0fd45642dcecd9ff74cb3e71c20a4753f7d82e2f07864b5108f\@%3Cdev.drill.apache.org%3E","https://www.tenable.com/security/tns-2019-08","https://lists.apache.org/thread.html/f9bc3e55f4e28d1dcd1a69aae6d53e609a758e34d2869b4d798e13cc\@%3Cissues.drill.apache.org%3E","https://www.oracle.com/security-alerts/cpujan2020.html","https://access.redhat.com/errata/RHSA-2020:0481","https://access.redhat.com/errata/RHSA-2020:0729","http://packetstormsecurity.com/files/156743/OctoberCMS-Insecure-Dependencies.html","http://lists.opensuse.org/opensuse-security-announce/2020-03/msg00041.html","https://www.oracle.com/security-alerts/cpuapr2020.html","https://www.oracle.com/security-alerts/cpujul2020.html","https://www.oracle.com/security-alerts/cpuoct2020.html","https://kb.pulsesecure.net/articles/Pulse_Security_Advisories/SA44601","https://security.netapp.com/advisory/ntap-20210108-0004/"],"reported" => "2018-01-18","severity" => "medium"},{"affected_versions" => [">=2.007000_002,<=2.055001"],"cves" => ["CVE-2011-4969"],"description" => "Cross-site scripting (XSS) vulnerability in jQuery before 1.6.3, when using location.hash to select elements, allows remote attackers to inject arbitrary web script or HTML via a crafted tag.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2011-4969-jquery","references" => ["https://github.com/jquery/jquery/commit/db9e023e62c1ff5d8f21ed9868ab6878da2005e9","http://bugs.jquery.com/ticket/9521","http://www.ubuntu.com/usn/USN-1722-1","http://blog.jquery.com/2011/09/01/jquery-1-6-3-released/","http://www.osvdb.org/80056","http://www.openwall.com/lists/oss-security/2013/01/31/3","http://blog.mindedsecurity.com/2011/07/jquery-is-sink.html","https://h20566.www2.hpe.com/portal/site/hpsc/public/kb/docDisplay?docId=emr_na-c05111017","https://h20566.www2.hpe.com/portal/site/hpsc/public/kb/docDisplay?docId=emr_na-c05158380","https://h20566.www2.hpe.com/portal/site/hpsc/public/kb/docDisplay?docId=emr_na-c05232730","http://www.securityfocus.com/bid/58458","http://www.securitytracker.com/id/1036620","https://security.netapp.com/advisory/ntap-20190416-0007/","https://lists.apache.org/thread.html/ff8dcfe29377088ab655fda9d585dccd5b1f07fabd94ae84fd60a7f8\@%3Ccommits.pulsar.apache.org%3E"],"reported" => "2013-03-08","severity" => undef},{"affected_versions" => [">=2.007000_002,<=2.055001"],"cves" => ["CVE-2012-6708"],"description" => "jQuery before 1.9.0 is vulnerable to Cross-site Scripting (XSS) attacks. The jQuery(strInput) function does not differentiate selectors from HTML in a reliable fashion. In vulnerable versions, jQuery determined whether the input was HTML by looking for the '<' character anywhere in the string, giving attackers more flexibility when attempting to construct a malicious payload. In fixed versions, jQuery only deems the input to be HTML if it explicitly starts with the '<' character, limiting exploitability only to attackers who can control the beginning of a string, which is far less common.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2012-6708-jquery","references" => ["https://snyk.io/vuln/npm:jquery:20120206","https://github.com/jquery/jquery/commit/05531fc4080ae24070930d15ae0cea7ae056457d","https://bugs.jquery.com/ticket/11290","http://www.securityfocus.com/bid/102792","https://help.ecostruxureit.com/display/public/UADCE725/Security+fixes+in+StruxureWare+Data+Center+Expert+v7.6.0","http://packetstormsecurity.com/files/153237/RetireJS-CORS-Issue-Script-Execution.html","https://lists.apache.org/thread.html/b0656d359c7d40ec9f39c8cc61bca66802ef9a2a12ee199f5b0c1442\@%3Cdev.drill.apache.org%3E","https://lists.apache.org/thread.html/519eb0fd45642dcecd9ff74cb3e71c20a4753f7d82e2f07864b5108f\@%3Cdev.drill.apache.org%3E","http://lists.opensuse.org/opensuse-security-announce/2020-03/msg00041.html","https://lists.apache.org/thread.html/f9bc3e55f4e28d1dcd1a69aae6d53e609a758e34d2869b4d798e13cc\@%3Cissues.drill.apache.org%3E","http://packetstormsecurity.com/files/161972/Linksys-EA7500-2.0.8.194281-Cross-Site-Scripting.html"],"reported" => "2018-01-18","severity" => "medium"},{"affected_versions" => [">=2.007000_002,<=2.055001"],"cves" => ["CVE-2020-7656"],"description" => "jquery prior to 1.9.0 allows Cross-site Scripting attacks via the load method. The load method fails to recognize and remove \"<script>\" HTML tags that contain a whitespace character, i.e: \"</script >\", which results in the enclosed script logic to be executed.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2020-7656-jquery","references" => ["https://snyk.io/vuln/SNYK-JS-JQUERY-569619","https://security.netapp.com/advisory/ntap-20200528-0001/"],"reported" => "2020-05-19","severity" => "medium"},{"affected_versions" => [">=2.007000_002,<=2.055001"],"cves" => ["CVE-2019-5428"],"description" => "Prototype Pollution is a vulnerability affecting JavaScript. Prototype Pollution refers to the ability to inject properties into existing JavaScript language construct prototypes, such as objects. JavaScript allows all Object attributes to be altered, including their magical attributes such as _proto_, constructor and prototype. An attacker manipulates these attributes to overwrite, or pollute, a JavaScript application object prototype of the base object by injecting other values. Properties on the Object.prototype are then inherited by all the JavaScript objects through the prototype chain. When that happens, this leads to either denial of service by triggering JavaScript exceptions, or it tampers with the application source code to force the code path that the attacker injects, thereby leading to remote code execution.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2019-5428-jquery","references" => ["https://security.snyk.io/vuln/SNYK-JS-JQUERY-174006"],"reported" => undef,"severity" => undef},{"affected_versions" => [">=2.007000_002,<=2.055001"],"cves" => ["CVE-2014-6071"],"description" => "jQuery 1.4.2 allows remote attackers to conduct cross-site scripting (XSS) attacks via vectors related to use of the text method inside after.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2014-6071-jquery","references" => ["https://bugzilla.redhat.com/show_bug.cgi?id=1136683","http://seclists.org/fulldisclosure/2014/Sep/10","https://help.ecostruxureit.com/display/public/UADCE725/Security+fixes+in+StruxureWare+Data+Center+Expert+v7.6.0"],"reported" => "2018-01-16","severity" => "medium"},{"affected_versions" => [">=2.028008,<=2.052002"],"cves" => ["CVE-2022-24785"],"description" => "Moment.js is a JavaScript date library for parsing, validating, manipulating, and formatting dates. A path traversal vulnerability impacts npm (server) users of Moment.js between versions 1.0.1 and 2.29.1, especially if a user-provided locale string is directly used to switch moment locale. This problem is patched in 2.29.2, and the patch can be applied to all affected versions. As a workaround, sanitize the user-provided locale name before passing it to Moment.js.\n","distribution" => "App-Netdisco","fixed_versions" => [],"id" => "CPANSA-App-Netdisco-2022-24785-momentjs","references" => ["https://github.com/moment/moment/commit/4211bfc8f15746be4019bba557e29a7ba83d54c5","https://github.com/moment/moment/security/advisories/GHSA-8hfj-j24r-96c4","https://www.tenable.com/security/tns-2022-09","https://security.netapp.com/advisory/ntap-20220513-0006/"],"reported" => "2022-04-04","severity" => "high"}],"main_module" => "App::Netdisco","versions" => [{"date" => "2012-12-20T21:16:29","version" => "2.00_011"},{"date" => "2012-12-21T08:21:35","version" => "2.00_012"},{"date" => "2013-01-05T16:14:21","version" => "2.00_012"},{"date" => "2013-01-06T01:16:03","version" => "2.00_012"},{"date" => "2013-01-06T02:03:22","version" => "2.00_012"},{"date" => "2013-01-14T22:16:29","version" => "2.00_012"},{"date" => "2013-01-30T13:23:14","version" => "2.004002"},{"date" => "2013-02-09T22:37:41","version" => "2.005000_001"},{"date" => "2013-02-10T21:39:04","version" => "2.005000_001"},{"date" => "2013-03-05T23:03:41","version" => "2.005000_003"},{"date" => "2013-03-05T23:21:44","version" => "2.005000_004"},{"date" => "2013-03-07T21:52:05","version" => "2.006000"},{"date" => "2013-03-17T14:50:06","version" => "2.007000_001"},{"date" => "2013-06-03T19:54:38","version" => "2.007000_002"},{"date" => "2013-06-08T20:22:28","version" => "2.007000_003"},{"date" => "2013-06-09T10:31:46","version" => "2.007000_004"},{"date" => "2013-06-09T13:10:45","version" => "2.007000_005"},{"date" => "2013-06-09T14:45:42","version" => "2.008000"},{"date" => "2013-06-11T12:39:12","version" => "2.008001"},{"date" => "2013-06-11T21:55:59","version" => "2.008002"},{"date" => "2013-06-16T17:29:20","version" => "2.009000_001"},{"date" => "2013-06-17T07:18:07","version" => "2.010000"},{"date" => "2013-06-17T22:10:21","version" => "2.010001_001"},{"date" => "2013-06-20T12:55:28","version" => "2.010001_002"},{"date" => "2013-06-20T12:58:16","version" => "2.010001_003"},{"date" => "2013-07-23T23:02:00","version" => "2.010002"},{"date" => "2013-07-24T22:50:05","version" => "2.010004"},{"date" => "2013-07-29T07:04:27","version" => "2.011000"},{"date" => "2013-08-06T17:37:28","version" => "2.012000"},{"date" => "2013-08-06T17:42:25","version" => "2.012001"},{"date" => "2013-08-07T09:06:31","version" => "2.012002"},{"date" => "2013-08-16T16:28:13","version" => "2.012003_001"},{"date" => "2013-08-16T16:48:37","version" => "2.012004"},{"date" => "2013-08-16T16:51:08","version" => "2.012005"},{"date" => "2013-08-23T05:52:12","version" => "2.012006"},{"date" => "2013-08-23T10:29:04","version" => "2.013000"},{"date" => "2013-08-23T11:34:38","version" => "2.013001"},{"date" => "2013-08-26T21:44:14","version" => "2.014000"},{"date" => "2013-09-05T23:57:20","version" => "2.015000"},{"date" => "2013-09-10T22:33:43","version" => "2.016000"},{"date" => "2013-09-11T21:38:31","version" => "2.016001"},{"date" => "2013-09-11T22:15:54","version" => "2.016002"},{"date" => "2013-09-12T07:28:46","version" => "2.016003"},{"date" => "2013-09-23T19:49:48","version" => "2.017000"},{"date" => "2013-10-06T22:38:36","version" => "2.017001_001"},{"date" => "2013-10-07T20:55:41","version" => "2.017001_002"},{"date" => "2013-10-07T22:36:36","version" => "2.017001_003"},{"date" => "2013-10-08T10:28:21","version" => "2.018000"},{"date" => "2013-10-16T22:57:00","version" => "2.018000_001"},{"date" => "2013-10-22T12:44:58","version" => "2.018000_002"},{"date" => "2013-10-22T13:19:30","version" => "2.019000"},{"date" => "2013-10-22T14:41:32","version" => "2.019001"},{"date" => "2013-10-24T04:57:13","version" => "2.019002"},{"date" => "2013-10-27T03:07:39","version" => "2.019003"},{"date" => "2013-12-08T19:46:22","version" => "2.020000"},{"date" => "2013-12-08T21:49:04","version" => "2.020001"},{"date" => "2013-12-11T15:59:18","version" => "2.020002"},{"date" => "2013-12-29T21:34:57","version" => "2.020003_001"},{"date" => "2014-01-01T23:33:18","version" => "2.020003_002"},{"date" => "2014-01-06T20:49:38","version" => "2.020003_003"},{"date" => "2014-01-12T17:36:59","version" => "2.021000"},{"date" => "2014-01-13T00:42:23","version" => "2.021000_001"},{"date" => "2014-01-13T14:02:33","version" => "2.021000_002"},{"date" => "2014-01-13T18:50:36","version" => "2.021000_004"},{"date" => "2014-01-26T13:49:10","version" => "2.022000"},{"date" => "2014-02-10T21:24:32","version" => "2.023000"},{"date" => "2014-02-14T19:41:51","version" => "2.023001"},{"date" => "2014-02-17T13:23:06","version" => "2.023002"},{"date" => "2014-02-22T19:18:19","version" => "2.024000"},{"date" => "2014-02-25T22:36:15","version" => "2.024001"},{"date" => "2014-02-27T17:39:32","version" => "2.024002"},{"date" => "2014-02-27T17:52:08","version" => "2.024003"},{"date" => "2014-03-02T23:30:02","version" => "2.024003_001"},{"date" => "2014-03-04T22:23:50","version" => "2.024004"},{"date" => "2014-03-28T07:32:33","version" => "2.025000_001"},{"date" => "2014-04-08T18:51:46","version" => "2.025001"},{"date" => "2014-04-10T20:17:35","version" => "2.026000"},{"date" => "2014-04-17T06:01:16","version" => "2.026001_001"},{"date" => "2014-04-18T22:35:47","version" => "2.026001_002"},{"date" => "2014-04-20T22:48:43","version" => "2.026001_003"},{"date" => "2014-04-28T21:01:11","version" => "2.026001_004"},{"date" => "2014-05-03T07:27:54","version" => "2.027001"},{"date" => "2014-05-04T09:01:14","version" => "2.027002"},{"date" => "2014-05-15T07:12:35","version" => "2.027003"},{"date" => "2014-05-15T15:55:07","version" => "2.027004"},{"date" => "2014-05-21T20:21:35","version" => "2.027005_001"},{"date" => "2014-05-27T06:05:59","version" => "2.027006"},{"date" => "2014-05-27T09:45:15","version" => "2.027007"},{"date" => "2014-06-23T12:59:01","version" => "2.027008_001"},{"date" => "2014-07-02T08:20:20","version" => "2.028000"},{"date" => "2014-07-13T17:55:04","version" => "2.028001"},{"date" => "2014-07-13T20:59:54","version" => "2.028002_001"},{"date" => "2014-07-15T16:10:41","version" => "2.028003"},{"date" => "2014-07-16T07:05:29","version" => "2.028004"},{"date" => "2014-07-17T13:25:34","version" => "2.028005"},{"date" => "2014-07-21T08:09:06","version" => "2.028006"},{"date" => "2014-07-22T07:01:44","version" => "2.028008"},{"date" => "2014-07-22T21:40:24","version" => "2.028010"},{"date" => "2014-07-22T21:49:10","version" => "2.028011"},{"date" => "2014-07-22T22:21:11","version" => "2.028012"},{"date" => "2014-07-30T23:57:34","version" => "2.028013"},{"date" => "2014-08-08T06:35:55","version" => "2.029000_001"},{"date" => "2014-08-08T21:43:46","version" => "2.029000_002"},{"date" => "2014-08-10T20:21:10","version" => "2.029001"},{"date" => "2014-08-10T20:37:39","version" => "2.029002"},{"date" => "2014-08-11T15:14:59","version" => "2.029003"},{"date" => "2014-08-11T21:04:08","version" => "2.029004"},{"date" => "2014-08-13T10:48:53","version" => "2.029005"},{"date" => "2014-08-25T16:24:00","version" => "2.029006"},{"date" => "2014-09-12T13:09:36","version" => "2.029007"},{"date" => "2014-09-23T19:32:12","version" => "2.029008"},{"date" => "2014-09-27T10:37:24","version" => "2.029009"},{"date" => "2014-10-07T07:39:18","version" => "2.029010"},{"date" => "2014-10-07T17:50:07","version" => "2.029011"},{"date" => "2014-10-09T16:01:27","version" => "2.029012"},{"date" => "2014-11-14T00:16:10","version" => "2.029013_001"},{"date" => "2014-11-14T23:58:24","version" => "2.029013_002"},{"date" => "2014-11-20T08:04:38","version" => "2.029014"},{"date" => "2015-01-08T11:10:55","version" => "2.030000"},{"date" => "2015-02-04T15:28:08","version" => "2.031000"},{"date" => "2015-02-04T18:45:47","version" => "2.031001"},{"date" => "2015-02-04T19:01:00","version" => "2.031002"},{"date" => "2015-02-04T22:47:46","version" => "2.031003"},{"date" => "2015-02-05T14:19:47","version" => "2.031004"},{"date" => "2015-02-06T10:20:08","version" => "2.031005"},{"date" => "2015-02-15T15:40:46","version" => "2.031006"},{"date" => "2015-02-19T08:51:44","version" => "2.031007"},{"date" => "2015-02-22T09:43:23","version" => "2.031008"},{"date" => "2015-02-25T21:21:31","version" => "2.031009"},{"date" => "2015-02-25T22:12:31","version" => "2.031010"},{"date" => "2015-02-27T08:35:31","version" => "2.031011"},{"date" => "2015-02-28T11:59:22","version" => "2.031012"},{"date" => "2015-03-07T17:12:38","version" => "2.032000_001"},{"date" => "2015-03-24T22:46:31","version" => "2.032001"},{"date" => "2015-04-03T19:21:56","version" => "2.032002"},{"date" => "2015-05-05T19:42:05","version" => "2.032003"},{"date" => "2015-05-17T21:09:24","version" => "2.032004"},{"date" => "2015-05-18T09:25:35","version" => "2.032005"},{"date" => "2015-07-19T11:40:08","version" => "2.032006"},{"date" => "2015-07-30T16:33:06","version" => "2.032007"},{"date" => "2015-08-26T11:27:02","version" => "2.033000"},{"date" => "2015-08-27T14:50:17","version" => "2.033001"},{"date" => "2015-09-29T08:56:31","version" => "2.033002"},{"date" => "2015-10-13T21:37:21","version" => "2.033003"},{"date" => "2015-11-16T21:41:13","version" => "2.033004"},{"date" => "2016-02-02T09:11:15","version" => "2.033005"},{"date" => "2016-03-20T13:17:57","version" => "2.033005"},{"date" => "2016-10-03T15:58:17","version" => "2.034000"},{"date" => "2016-11-20T17:51:25","version" => "2.034001"},{"date" => "2017-01-06T14:35:56","version" => "2.034002"},{"date" => "2017-04-19T20:59:13","version" => "2.035000"},{"date" => "2017-04-19T21:18:39","version" => "2.035001"},{"date" => "2017-04-24T11:50:12","version" => "2.035002"},{"date" => "2017-04-24T13:44:38","version" => "2.035003"},{"date" => "2017-04-25T09:54:37","version" => "2.035004"},{"date" => "2017-04-29T08:13:48","version" => "2.035005"},{"date" => "2017-04-29T08:31:09","version" => "2.035006"},{"date" => "2017-05-17T06:44:07","version" => "2.035999_001"},{"date" => "2017-05-27T14:50:21","version" => "2.035999_002"},{"date" => "2017-05-29T16:22:27","version" => "2.035999_003"},{"date" => "2017-05-30T10:40:20","version" => "2.035999_004"},{"date" => "2017-05-30T11:05:45","version" => "2.035999_005"},{"date" => "2017-05-30T15:03:49","version" => "2.035999_006"},{"date" => "2017-05-30T20:27:22","version" => "2.035999_007"},{"date" => "2017-06-13T06:23:11","version" => "2.035999_008"},{"date" => "2017-06-18T22:37:11","version" => "2.035999_009"},{"date" => "2017-06-19T17:50:27","version" => "2.035999_010"},{"date" => "2017-06-22T07:36:42","version" => "2.036000"},{"date" => "2017-06-22T11:25:23","version" => "2.036001"},{"date" => "2017-06-26T18:58:33","version" => "2.036002"},{"date" => "2017-06-28T15:44:41","version" => "2.036003"},{"date" => "2017-07-02T08:56:33","version" => "2.036004"},{"date" => "2017-07-05T05:07:47","version" => "2.036005"},{"date" => "2017-07-09T13:28:10","version" => "2.036006"},{"date" => "2017-07-12T06:01:03","version" => "2.036007"},{"date" => "2017-07-14T12:52:34","version" => "2.036008"},{"date" => "2017-08-01T09:30:17","version" => "2.036009"},{"date" => "2017-10-08T13:22:48","version" => "2.036010"},{"date" => "2017-10-09T07:01:31","version" => "2.036011"},{"date" => "2017-10-11T17:33:31","version" => "2.036012_001"},{"date" => "2017-11-19T13:49:04","version" => "2.036012_002"},{"date" => "2017-11-28T21:49:40","version" => "2.036012_003"},{"date" => "2017-12-14T21:49:14","version" => "2.037000"},{"date" => "2017-12-14T21:57:42","version" => "2.037001"},{"date" => "2017-12-17T20:22:25","version" => "2.037002"},{"date" => "2017-12-18T17:35:24","version" => "2.037003"},{"date" => "2017-12-21T20:06:32","version" => "2.037004"},{"date" => "2017-12-22T23:46:44","version" => "2.037005"},{"date" => "2017-12-31T09:54:24","version" => "2.038000"},{"date" => "2018-01-02T13:10:42","version" => "2.038001"},{"date" => "2018-01-02T22:07:51","version" => "2.038002_001"},{"date" => "2018-01-04T20:21:13","version" => "2.038002_002"},{"date" => "2018-01-04T22:38:07","version" => "2.038002_003"},{"date" => "2018-01-04T22:53:29","version" => "2.038003"},{"date" => "2018-01-05T17:43:24","version" => "2.038004"},{"date" => "2018-01-05T20:22:23","version" => "2.038005"},{"date" => "2018-01-08T14:14:33","version" => "2.038006"},{"date" => "2018-01-09T09:57:13","version" => "2.038007"},{"date" => "2018-01-09T15:38:57","version" => "2.038008"},{"date" => "2018-01-10T01:16:32","version" => "2.038009"},{"date" => "2018-01-15T11:34:50","version" => "2.038028"},{"date" => "2018-01-23T22:56:08","version" => "2.038031"},{"date" => "2018-01-28T20:04:09","version" => "2.038032"},{"date" => "2018-01-31T15:06:37","version" => "2.038033"},{"date" => "2018-01-31T20:00:58","version" => "2.038034"},{"date" => "2018-02-02T14:54:43","version" => "2.039000"},{"date" => "2018-02-02T18:35:11","version" => "2.039001"},{"date" => "2018-02-07T23:03:50","version" => "2.039002"},{"date" => "2018-02-12T21:11:07","version" => "2.039003"},{"date" => "2018-02-15T08:29:55","version" => "2.039004"},{"date" => "2018-02-15T19:55:25","version" => "2.039005"},{"date" => "2018-02-15T20:17:31","version" => "2.039006"},{"date" => "2018-02-16T08:23:49","version" => "2.039007"},{"date" => "2018-02-22T22:06:19","version" => "2.039008"},{"date" => "2018-02-22T22:23:38","version" => "2.039009"},{"date" => "2018-02-22T22:52:04","version" => "2.039010"},{"date" => "2018-02-25T09:28:46","version" => "2.039011"},{"date" => "2018-03-02T13:12:05","version" => "2.039012"},{"date" => "2018-03-02T14:18:44","version" => "2.039013"},{"date" => "2018-03-04T09:58:06","version" => "2.039014"},{"date" => "2018-03-05T23:01:48","version" => "2.039015"},{"date" => "2018-03-19T23:12:52","version" => "2.039016"},{"date" => "2018-03-20T10:12:42","version" => "2.039017"},{"date" => "2018-03-22T21:46:51","version" => "2.039018"},{"date" => "2018-03-23T09:55:03","version" => "2.039019"},{"date" => "2018-03-26T21:59:24","version" => "2.039020"},{"date" => "2018-04-10T20:47:57","version" => "2.039021"},{"date" => "2018-04-18T21:24:35","version" => "2.039022"},{"date" => "2018-04-19T07:27:07","version" => "2.039023"},{"date" => "2018-04-22T17:54:24","version" => "2.039024"},{"date" => "2018-04-27T12:27:18","version" => "2.039025"},{"date" => "2018-04-28T12:11:41","version" => "2.039026"},{"date" => "2018-04-28T21:16:54","version" => "2.039027"},{"date" => "2018-05-05T15:29:52","version" => "2.039028"},{"date" => "2018-05-09T05:55:14","version" => "2.039029"},{"date" => "2018-05-09T06:00:13","version" => "2.039030"},{"date" => "2018-06-17T20:58:47","version" => "2.039031"},{"date" => "2018-10-19T14:38:26","version" => "2.039032"},{"date" => "2018-10-19T20:36:53","version" => "2.039033"},{"date" => "2018-12-28T17:07:03","version" => "2.040000"},{"date" => "2018-12-30T10:53:04","version" => "2.040001"},{"date" => "2018-12-30T10:59:07","version" => "2.040002"},{"date" => "2019-01-18T07:10:03","version" => "2.040003"},{"date" => "2019-03-03T14:56:07","version" => "2.040004"},{"date" => "2019-03-04T10:02:25","version" => "2.040005"},{"date" => "2019-03-04T12:04:34","version" => "2.040006"},{"date" => "2019-03-06T18:44:33","version" => "2.040007"},{"date" => "2019-03-12T19:59:49","version" => "2.041000"},{"date" => "2019-03-15T05:34:08","version" => "2.041001"},{"date" => "2019-03-17T09:37:27","version" => "2.041002"},{"date" => "2019-03-17T20:32:01","version" => "2.042000"},{"date" => "2019-03-18T21:28:43","version" => "2.042001"},{"date" => "2019-03-20T12:26:14","version" => "2.042002"},{"date" => "2019-03-21T16:19:51","version" => "2.042003"},{"date" => "2019-03-28T23:00:19","version" => "2.042004"},{"date" => "2019-04-03T13:56:55","version" => "2.042005"},{"date" => "2019-04-16T16:48:15","version" => "2.042006"},{"date" => "2019-04-28T19:57:19","version" => "2.042007"},{"date" => "2019-04-30T10:51:06","version" => "2.042008"},{"date" => "2019-05-30T06:13:10","version" => "2.042009"},{"date" => "2019-06-02T06:55:13","version" => "2.042010"},{"date" => "2019-09-03T19:27:26","version" => "2.043000"},{"date" => "2019-09-04T12:36:05","version" => "2.043001"},{"date" => "2019-09-23T13:58:04","version" => "2.044000"},{"date" => "2019-09-26T14:01:50","version" => "2.044001"},{"date" => "2019-10-01T09:43:52","version" => "2.044002"},{"date" => "2019-10-15T17:57:05","version" => "2.044003"},{"date" => "2019-10-30T19:52:42","version" => "2.044004"},{"date" => "2020-01-19T15:31:55","version" => "2.044005"},{"date" => "2020-01-22T21:20:09","version" => "2.044006"},{"date" => "2020-01-22T21:25:34","version" => "2.044007"},{"date" => "2020-01-23T18:44:49","version" => "2.044008"},{"date" => "2020-01-23T18:48:48","version" => "2.044009"},{"date" => "2020-01-25T18:09:41","version" => "2.044010"},{"date" => "2020-01-26T21:46:22","version" => "2.044011"},{"date" => "2020-02-01T13:27:10","version" => "2.044012"},{"date" => "2020-02-04T21:35:18","version" => "2.044013"},{"date" => "2020-02-09T10:03:07","version" => "2.044014"},{"date" => "2020-02-12T16:56:14","version" => "2.044015"},{"date" => "2020-04-15T20:25:36","version" => "2.045000"},{"date" => "2020-04-18T08:50:13","version" => "2.045001"},{"date" => "2020-04-19T17:03:54","version" => "2.045002"},{"date" => "2020-05-15T11:02:33","version" => "2.045003"},{"date" => "2020-05-18T11:34:20","version" => "2.045005"},{"date" => "2020-05-24T18:43:31","version" => "2.045006"},{"date" => "2020-06-05T08:11:31","version" => "2.045007"},{"date" => "2020-07-08T21:29:53","version" => "2.046000"},{"date" => "2020-07-10T21:30:48","version" => "2.046001"},{"date" => "2020-08-07T10:02:15","version" => "2.046002"},{"date" => "2020-10-17T12:15:43","version" => "2.046003"},{"date" => "2020-10-17T13:29:56","version" => "2.046004"},{"date" => "2020-10-17T13:40:12","version" => "2.046005"},{"date" => "2020-10-31T11:15:17","version" => "2.046006"},{"date" => "2020-12-23T11:58:41","version" => "2.047000"},{"date" => "2020-12-29T13:08:42","version" => "2.047001"},{"date" => "2020-12-30T20:42:02","version" => "2.047002"},{"date" => "2021-02-14T14:05:50","version" => "2.047003"},{"date" => "2021-02-15T22:13:51","version" => "2.047004"},{"date" => "2021-02-24T10:48:16","version" => "2.047005"},{"date" => "2021-07-14T11:34:50","version" => "2.047006"},{"date" => "2021-07-14T12:15:22","version" => "2.047007"},{"date" => "2021-07-21T08:54:04","version" => "2.047008"},{"date" => "2021-08-14T12:38:48","version" => "2.048000"},{"date" => "2021-08-22T13:29:25","version" => "2.049000"},{"date" => "2021-08-22T19:32:21","version" => "2.049001"},{"date" => "2021-09-03T05:26:59","version" => "2.049002"},{"date" => "2021-09-03T07:11:01","version" => "2.049003"},{"date" => "2021-09-03T20:36:58","version" => "2.049004"},{"date" => "2021-09-09T07:52:58","version" => "2.049005"},{"date" => "2021-10-03T07:55:21","version" => "2.049006"},{"date" => "2021-10-05T16:38:38","version" => "2.049007"},{"date" => "2021-10-06T15:53:00","version" => "2.049008"},{"date" => "2021-10-06T21:33:32","version" => "2.049009"},{"date" => "2021-10-11T20:34:00","version" => "2.049010"},{"date" => "2021-10-12T07:43:57","version" => "2.049011"},{"date" => "2021-10-12T12:28:03","version" => "2.050000"},{"date" => "2021-10-12T14:28:01","version" => "2.050001"},{"date" => "2021-10-19T08:13:11","version" => "2.050003"},{"date" => "2021-11-14T19:39:02","version" => "2.051001"},{"date" => "2021-11-23T16:10:12","version" => "2.051002"},{"date" => "2021-11-24T13:15:54","version" => "2.051003"},{"date" => "2021-11-25T11:53:35","version" => "2.051004"},{"date" => "2021-11-25T20:20:22","version" => "2.051005"},{"date" => "2021-11-30T05:19:10","version" => "2.052000"},{"date" => "2022-02-01T20:51:26","version" => "2.052001"},{"date" => "2022-02-28T18:14:51","version" => "2.052002"},{"date" => "2022-04-13T19:12:04","version" => "2.052003"},{"date" => "2022-04-22T08:58:41","version" => "2.052005"},{"date" => "2022-05-17T21:06:21","version" => "2.052006"},{"date" => "2022-06-03T21:24:35","version" => "2.052007"},{"date" => "2022-07-12T08:18:54","version" => "2.052008"},{"date" => "2022-07-26T21:00:56","version" => "2.052009"},{"date" => "2022-07-27T21:54:42","version" => "2.052010"},{"date" => "2022-08-01T16:54:16","version" => "2.052011"},{"date" => "2022-08-02T16:05:09","version" => "2.052012"},{"date" => "2022-08-02T20:51:15","version" => "2.053000"},{"date" => "2022-08-02T21:21:25","version" => "2.053001"},{"date" => "2022-08-03T17:05:56","version" => "2.053002"},{"date" => "2022-08-03T21:05:28","version" => "2.053003"},{"date" => "2022-08-04T22:02:30","version" => "2.053004"},{"date" => "2022-08-04T22:11:32","version" => "2.053005"},{"date" => "2022-08-07T22:32:50","version" => "2.053006"},{"date" => "2022-08-09T09:32:35","version" => "2.053007"},{"date" => "2022-08-15T12:46:43","version" => "2.054000"},{"date" => "2022-08-17T10:15:23","version" => "2.055000"},{"date" => "2022-09-02T08:05:05","version" => "2.055001"},{"date" => "2022-09-24T19:09:03","version" => "2.056000"},{"date" => "2022-09-24T19:09:14","version" => "2.057000"},{"date" => "2022-09-24T19:09:26","version" => "2.057001"},{"date" => "2022-09-24T21:42:34","version" => "2.057002"},{"date" => "2022-09-27T15:34:42","version" => "2.057004"},{"date" => "2022-09-28T14:20:19","version" => "2.057005"},{"date" => "2022-09-30T21:07:39","version" => "2.057006"},{"date" => "2022-10-04T12:22:31","version" => "2.057007"},{"date" => "2022-10-18T12:00:41","version" => "2.057008"},{"date" => "2022-11-04T10:29:49","version" => "2.058000"},{"date" => "2022-11-04T15:42:53","version" => "2.058001"},{"date" => "2022-11-04T15:54:41","version" => "2.058003"},{"date" => "2022-11-25T15:29:29","version" => "2.059000"},{"date" => "2022-11-26T20:37:56","version" => "2.059001"},{"date" => "2022-12-09T10:32:14","version" => "2.060000"},{"date" => "2022-12-11T16:58:49","version" => "2.060001"},{"date" => "2022-12-13T15:34:56","version" => "2.060002"},{"date" => "2022-12-14T16:55:04","version" => "2.060003"},{"date" => "2023-01-11T15:14:43","version" => "2.060004"},{"date" => "2023-02-21T14:22:36","version" => "2.060005"},{"date" => "2023-03-03T15:43:58","version" => "2.060007"},{"date" => "2023-03-08T17:21:35","version" => "2.060008"},{"date" => "2023-03-10T18:09:47","version" => "2.060009"},{"date" => "2023-03-10T18:12:29","version" => "2.060010"},{"date" => "2023-03-29T10:43:01","version" => "2.061000"},{"date" => "2023-04-27T15:33:52","version" => "2.061001"},{"date" => "2023-05-30T08:58:07","version" => "2.062000"},{"date" => "2023-06-05T17:02:14","version" => "2.062001"},{"date" => "2023-06-06T06:07:49","version" => "2.062002"},{"date" => "2023-06-20T09:11:03","version" => "2.062003"},{"date" => "2023-06-26T17:00:40","version" => "2.062004"},{"date" => "2023-06-26T18:35:55","version" => "2.062005"},{"date" => "2023-06-28T09:03:56","version" => "2.063000"},{"date" => "2023-06-28T16:06:44","version" => "2.063001"},{"date" => "2023-07-14T21:25:14","version" => "2.063002"},{"date" => "2023-07-15T10:11:43","version" => "2.063004"},{"date" => "2023-07-22T09:17:38","version" => "2.064000"},{"date" => "2023-07-25T12:03:07","version" => "2.064001"},{"date" => "2023-08-13T15:06:31","version" => "2.065000"},{"date" => "2023-08-13T18:47:39","version" => "2.065001"},{"date" => "2023-09-03T08:12:02","version" => "2.065002"},{"date" => "2023-09-19T18:11:32","version" => "2.066000"},{"date" => "2023-09-27T13:20:00","version" => "2.067001"},{"date" => "2023-09-27T13:27:43","version" => "2.067002"},{"date" => "2023-10-27T14:38:37","version" => "2.068000"},{"date" => "2023-11-01T21:58:28","version" => "2.068001"},{"date" => "2023-11-12T07:36:25","version" => "2.069000"},{"date" => "2023-11-14T19:10:46","version" => "2.070000"},{"date" => "2023-11-15T11:29:20","version" => "2.070001"},{"date" => "2023-11-21T16:01:49","version" => "2.070002"},{"date" => "2023-11-24T20:50:38","version" => "2.070003"},{"date" => "2023-12-07T08:00:38","version" => "2.071000"},{"date" => "2023-12-07T15:51:30","version" => "2.071001"},{"date" => "2024-01-06T14:13:03","version" => "2.071002"},{"date" => "2024-01-10T20:49:02","version" => "2.071003"},{"date" => "2024-01-14T16:49:02","version" => "2.072000"},{"date" => "2024-01-15T20:04:01","version" => "2.072001"},{"date" => "2024-01-21T11:04:41","version" => "2.072002"},{"date" => "2024-02-14T21:31:03","version" => "2.072003"},{"date" => "2024-03-13T15:45:46","version" => "2.073000"},{"date" => "2024-03-13T16:54:38","version" => "2.073001"},{"date" => "2024-03-19T09:53:20","version" => "2.074000"},{"date" => "2024-03-19T17:08:31","version" => "2.074001"},{"date" => "2024-04-08T18:12:37","version" => "2.075000"},{"date" => "2024-04-09T10:16:31","version" => "2.075001"},{"date" => "2024-04-10T10:43:31","version" => "2.075002"},{"date" => "2024-04-12T10:31:45","version" => "2.075003"},{"date" => "2024-04-22T16:27:19","version" => "2.076000"},{"date" => "2024-04-24T20:20:10","version" => "2.076001"},{"date" => "2024-04-30T16:36:08","version" => "2.076002"},{"date" => "2024-05-03T14:28:39","version" => "2.076004"},{"date" => "2024-05-20T18:19:33","version" => "2.076005"},{"date" => "2024-08-10T18:36:30","version" => "2.076006"},{"date" => "2024-08-15T09:14:32","version" => "2.077000"},{"date" => "2024-08-15T10:17:44","version" => "2.077001"},{"date" => "2024-08-15T10:52:40","version" => "2.077002"},{"date" => "2024-08-15T19:54:33","version" => "2.077003"},{"date" => "2024-08-15T21:20:21","version" => "2.077004"},{"date" => "2024-08-16T00:14:40","version" => "2.077005"},{"date" => "2024-08-18T06:35:14","version" => "2.077006"},{"date" => "2024-08-18T12:19:30","version" => "2.077007"},{"date" => "2024-08-19T06:08:24","version" => "2.077008"},{"date" => "2024-08-19T11:03:29","version" => "2.077009"},{"date" => "2024-08-23T09:20:50","version" => "2.077010"},{"date" => "2024-08-23T10:06:31","version" => "2.077011"},{"date" => "2024-08-27T08:13:19","version" => "2.078000"},{"date" => "2024-09-12T20:31:33","version" => "2.079000"},{"date" => "2024-09-13T12:33:44","version" => "2.079001"},{"date" => "2024-10-29T18:29:18","version" => "2.080000"},{"date" => "2024-10-29T22:40:05","version" => "2.080001"},{"date" => "2024-10-30T10:32:44","version" => "2.080002"},{"date" => "2024-10-30T14:56:55","version" => "2.080003"},{"date" => "2024-12-30T11:04:42","version" => "2.081000"},{"date" => "2024-12-30T22:06:11","version" => "2.081001"},{"date" => "2024-12-31T14:05:40","version" => "2.081002"},{"date" => "2024-12-31T18:13:01","version" => "2.081003"},{"date" => "2025-01-19T11:32:49","version" => "2.081004"},{"date" => "2025-01-29T09:05:34","version" => "2.082000"},{"date" => "2025-01-29T09:05:46","version" => "2.082001"},{"date" => "2025-02-02T21:01:02","version" => "2.082002"},{"date" => "2025-02-04T20:24:13","version" => "2.082003"},{"date" => "2025-02-06T12:37:52","version" => "2.083000"},{"date" => "2025-02-06T13:18:05","version" => "2.083001"},{"date" => "2025-03-05T17:35:07","version" => "2.084000"},{"date" => "2025-03-09T18:50:08","version" => "2.084001"},{"date" => "2025-04-19T14:09:15","version" => "2.084002"},{"date" => "2025-04-26T18:03:12","version" => "2.085000"},{"date" => "2025-05-02T11:38:20","version" => "2.085001"},{"date" => "2025-05-22T04:57:55","version" => "2.085002"},{"date" => "2025-05-24T17:59:36","version" => "2.085003"},{"date" => "2025-06-03T17:29:52","version" => "2.086000"},{"date" => "2025-06-04T16:09:11","version" => "2.086001"},{"date" => "2025-06-18T16:02:11","version" => "2.086002"},{"date" => "2025-06-21T21:19:20","version" => "2.086003"},{"date" => "2025-07-14T06:58:58","version" => "2.087000"},{"date" => "2025-08-20T08:48:08","version" => "2.087001"},{"date" => "2025-08-26T19:48:48","version" => "2.088000"},{"date" => "2025-08-26T20:55:03","version" => "2.088001"},{"date" => "2025-08-31T18:25:51","version" => "2.088002"},{"date" => "2025-09-02T16:27:31","version" => "2.088003"},{"date" => "2025-09-05T14:23:42","version" => "2.088004"},{"date" => "2025-09-07T21:39:05","version" => "2.089000"},{"date" => "2025-09-07T22:24:00","version" => "2.089001"},{"date" => "2025-09-12T08:53:38","version" => "2.089002"},{"date" => "2025-09-21T12:24:20","version" => "2.089003"},{"date" => "2025-09-21T17:24:33","version" => "2.089004"},{"date" => "2025-09-21T19:32:52","version" => "2.090000"},{"date" => "2025-09-25T11:53:09","version" => "2.090001"},{"date" => "2025-09-25T14:07:38","version" => "2.090002"},{"date" => "2025-09-30T18:57:58","version" => "2.091000"},{"date" => "2025-10-09T13:32:39","version" => "2.091001"},{"date" => "2025-10-19T21:09:39","version" => "2.092000"},{"date" => "2025-10-21T13:26:48","version" => "2.092001"},{"date" => "2025-10-22T15:55:40","version" => "2.092002"},{"date" => "2025-10-24T15:11:24","version" => "2.093000"},{"date" => "2025-10-24T15:34:57","version" => "2.093001"},{"date" => "2025-10-25T18:33:51","version" => "2.094000"},{"date" => "2025-10-29T13:02:15","version" => "2.094001"},{"date" => "2025-10-29T21:19:58","version" => "2.094002"},{"date" => "2025-11-03T21:28:10","version" => "2.094003"},{"date" => "2025-11-15T12:44:12","version" => "2.095000"},{"date" => "2025-11-15T17:24:25","version" => "2.095001"},{"date" => "2025-11-16T18:08:46","version" => "2.095002"},{"date" => "2025-11-18T20:13:49","version" => "2.095003"},{"date" => "2025-11-23T19:49:42","version" => "2.095004"},{"date" => "2025-11-30T16:31:48","version" => "2.095005"},{"date" => "2025-11-30T18:18:11","version" => "2.095006"},{"date" => "2025-12-07T20:39:51","version" => "2.096000"},{"date" => "2025-12-13T16:24:36","version" => "2.096001"},{"date" => "2025-12-16T12:26:36","version" => "2.097000"},{"date" => "2026-01-08T22:37:02","version" => "2.097001"},{"date" => "2026-01-09T10:32:45","version" => "2.097002"},{"date" => "2026-02-21T11:16:58","version" => "2.097003"}]},"App-cpanminus" => {"advisories" => [{"affected_versions" => ["<=1.7044"],"cves" => ["CVE-2020-16154"],"description" => "The App::cpanminus package 1.7044 for Perl allows Signature Verification Bypass.\n","distribution" => "App-cpanminus","fixed_versions" => [">=1.7045"],"id" => "CPANSA-App-cpanminus-2020-01","references" => ["https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DENFY4CRTIZL5WYYUYUM4VKCJNXO4QIW/","https://metacpan.org/pod/App::cpanminus","https://blog.hackeriet.no/cpan-signature-verification-vulnerabilities/","https://access.redhat.com/security/cve/cve-2020-16154","https://security-tracker.debian.org/tracker/CVE-2020-16154","https://blogs.perl.org/users/neilb/2021/11/addressing-cpan-vulnerabilities-related-to-checksums.html","https://github.com/miyagawa/cpanminus/pull/638"],"reported" => "2020-07-30"},{"affected_versions" => [">0"],"cves" => ["CVE-2024-45321"],"description" => "The App::cpanminus package through 1.7047 for Perl downloads code via insecure HTTP, enabling code execution for network attackers.\n","distribution" => "App-cpanminus","fixed_versions" => [],"id" => "CPANSA-App-cpanminus-2024-45321","references" => ["https://github.com/miyagawa/cpanminus/issues/611","https://github.com/miyagawa/cpanminus/pull/674","https://security.metacpan.org/2024/08/26/cpanminus-downloads-code-using-insecure-http.html"],"reported" => "2024-08-27"}],"main_module" => "App::cpanminus","versions" => [{"date" => "2010-02-20T02:27:00","version" => "0.01"},{"date" => "2010-02-20T03:14:10","version" => "0.02"},{"date" => "2010-02-20T03:39:56","version" => "0.03"},{"date" => "2010-02-20T05:04:01","version" => "0.04"},{"date" => "2010-02-20T05:16:03","version" => "0.05"},{"date" => "2010-02-20T05:48:44","version" => "0.06"},{"date" => "2010-02-20T20:20:08","version" => "0.07"},{"date" => "2010-02-20T23:41:01","version" => "0.08"},{"date" => "2010-02-21T07:48:49","version" => "0.09"},{"date" => "2010-02-23T20:52:46","version" => "0.99_01"},{"date" => "2010-02-24T06:20:33","version" => "0.99_02"},{"date" => "2010-02-24T07:05:47","version" => "0.99_03"},{"date" => "2010-02-25T17:20:05","version" => "0.99_04"},{"date" => "2010-02-25T22:42:22","version" => "0.99_05"},{"date" => "2010-02-25T22:51:43","version" => "0.99_06"},{"date" => "2010-02-28T05:14:30","version" => "0.99_07"},{"date" => "2010-03-01T05:59:15","version" => "0.9910"},{"date" => "2010-03-02T00:29:51","version" => "0.9911"},{"date" => "2010-03-03T02:55:22","version" => "0.9912"},{"date" => "2010-03-03T03:21:59","version" => "0.9913"},{"date" => "2010-03-04T08:42:03","version" => "0.9914"},{"date" => "2010-03-04T09:58:11","version" => "0.9915"},{"date" => "2010-03-04T19:35:22","version" => "0.9916"},{"date" => "2010-03-09T13:58:32","version" => "0.9917"},{"date" => "2010-03-10T02:26:03","version" => "0.9918"},{"date" => "2010-03-10T02:41:31","version" => "0.9919"},{"date" => "2010-03-10T09:49:22","version" => "0.99_20"},{"date" => "2010-03-10T15:03:38","version" => "0.9921"},{"date" => "2010-03-11T02:01:28","version" => "0.9922"},{"date" => "2010-03-16T00:07:01","version" => "0.9923"},{"date" => "2010-03-22T05:05:33","version" => "0.99_24"},{"date" => "2010-03-23T02:54:44","version" => "0.99_25"},{"date" => "2010-03-23T03:24:34","version" => "0.99_26"},{"date" => "2010-03-23T18:24:55","version" => "0.9927"},{"date" => "2010-03-26T05:09:12","version" => "0.9928"},{"date" => "2010-03-27T04:42:41","version" => "0.9929"},{"date" => "2010-03-30T21:29:41","version" => "0.9930"},{"date" => "2010-04-05T01:18:12","version" => "0.9931"},{"date" => "2010-04-05T23:28:11","version" => "0.9932"},{"date" => "2010-04-11T07:51:27","version" => "0.99_33"},{"date" => "2010-04-11T11:55:44","version" => "0.9934"},{"date" => "2010-04-12T11:09:47","version" => "0.999_01"},{"date" => "2010-04-13T07:11:08","version" => "0.999_02"},{"date" => "2010-04-14T09:29:25","version" => "0.999_03"},{"date" => "2010-04-14T09:30:52","version" => "0.9935"},{"date" => "2010-04-19T06:23:01","version" => "0.999_04"},{"date" => "2010-04-21T11:40:46","version" => "0.999_05"},{"date" => "2010-04-21T11:53:47","version" => "0.9936"},{"date" => "2010-04-24T08:23:24","version" => "1.0000"},{"date" => "2010-04-24T08:26:40","version" => "1.0001"},{"date" => "2010-05-02T03:51:09","version" => "1.0002"},{"date" => "2010-05-04T23:16:18","version" => "1.0003"},{"date" => "2010-05-14T23:10:54","version" => "1.0004"},{"date" => "2010-07-02T23:39:32","version" => "1.0005"},{"date" => "2010-07-02T23:54:14","version" => "1.0006"},{"date" => "2010-07-30T19:55:47","version" => "1.0010"},{"date" => "2010-08-18T23:42:36","version" => "1.0011"},{"date" => "2010-08-20T19:58:19","version" => "1.0012"},{"date" => "2010-09-12T19:54:17","version" => "1.0013"},{"date" => "2010-09-21T19:43:20","version" => "1.0014"},{"date" => "2010-09-24T23:52:00","version" => "1.0015"},{"date" => "2010-11-12T07:57:33","version" => "1.1000"},{"date" => "2010-11-12T20:37:49","version" => "1.1001"},{"date" => "2010-11-17T02:28:44","version" => "1.1002"},{"date" => "2010-11-25T09:18:34","version" => "1.1003"},{"date" => "2010-11-30T18:07:12","version" => "1.1004"},{"date" => "2010-12-14T23:08:40","version" => "1.1005"},{"date" => "2010-12-16T19:32:01","version" => "1.1006"},{"date" => "2011-01-18T17:40:16","version" => "1.1007"},{"date" => "2011-01-18T20:54:34","version" => "1.19_01"},{"date" => "2011-01-18T22:11:52","version" => "1.19_02"},{"date" => "2011-01-26T22:08:11","version" => "1.1008"},{"date" => "2011-02-16T18:11:52","version" => "1.2000"},{"date" => "2011-02-16T18:55:46","version" => "1.2001"},{"date" => "2011-03-01T02:59:25","version" => "1.29_01"},{"date" => "2011-03-02T00:09:00","version" => "1.29_02"},{"date" => "2011-03-02T22:41:40","version" => "1.3000"},{"date" => "2011-03-04T02:35:03","version" => "1.3001"},{"date" => "2011-03-04T03:54:53","version" => "1.30_02"},{"date" => "2011-03-04T08:32:56","version" => "1.30_03"},{"date" => "2011-03-04T08:53:22","version" => "1.30_04"},{"date" => "2011-03-04T09:41:34","version" => "1.30_05"},{"date" => "2011-03-04T22:57:43","version" => "1.30_06"},{"date" => "2011-03-04T23:20:45","version" => "1.30_07"},{"date" => "2011-03-05T00:07:44","version" => "1.30_08"},{"date" => "2011-03-05T02:16:54","version" => "1.30_09"},{"date" => "2011-03-05T22:57:38","version" => "1.30_10"},{"date" => "2011-03-06T09:37:36","version" => "1.30_11"},{"date" => "2011-03-07T03:00:09","version" => "1.30_12"},{"date" => "2011-03-07T18:54:03","version" => "1.30_13"},{"date" => "2011-03-08T09:11:33","version" => "1.4000"},{"date" => "2011-03-08T18:11:57","version" => "1.4001"},{"date" => "2011-03-09T01:57:18","version" => "1.4002"},{"date" => "2011-03-10T02:15:19","version" => "1.4003"},{"date" => "2011-03-10T18:09:34","version" => "1.4004"},{"date" => "2011-05-11T19:49:38","version" => "1.4005"},{"date" => "2011-05-16T17:17:29","version" => "1.4006"},{"date" => "2011-05-17T17:54:45","version" => "1.4007"},{"date" => "2011-06-16T01:00:00","version" => "1.4008"},{"date" => "2011-06-26T17:59:17","version" => "1.49_01"},{"date" => "2011-10-12T09:57:03","version" => "1.49_02"},{"date" => "2011-10-13T06:40:49","version" => "1.5000"},{"date" => "2011-10-13T15:21:16","version" => "1.5001"},{"date" => "2011-10-18T00:13:36","version" => "1.5002"},{"date" => "2011-10-19T07:31:10","version" => "1.5003"},{"date" => "2011-11-08T22:29:31","version" => "1.5004"},{"date" => "2011-11-22T21:31:21","version" => "1.5005"},{"date" => "2011-11-29T19:49:42","version" => "1.5006"},{"date" => "2011-12-20T18:18:50","version" => "1.5007"},{"date" => "2012-03-18T01:23:40","version" => "1.5008"},{"date" => "2012-03-30T16:45:43","version" => "1.5009"},{"date" => "2012-03-31T11:01:47","version" => "1.5010"},{"date" => "2012-04-12T09:59:39","version" => "1.5011"},{"date" => "2012-05-11T03:50:22","version" => "1.5012"},{"date" => "2012-05-12T03:18:19","version" => "1.5013"},{"date" => "2012-06-13T01:34:12","version" => "1.5014"},{"date" => "2012-06-24T22:37:49","version" => "1.5015"},{"date" => "2012-07-17T19:02:48","version" => "1.5016"},{"date" => "2012-07-18T15:41:26","version" => "1.5017"},{"date" => "2012-09-19T05:42:19","version" => "1.5018"},{"date" => "2012-12-22T17:22:02","version" => "1.5019"},{"date" => "2013-01-29T18:32:26","version" => "1.5020"},{"date" => "2013-01-31T08:45:31","version" => "1.5021"},{"date" => "2013-01-31T18:07:46","version" => "1.59_01"},{"date" => "2013-02-01T03:12:10","version" => "1.59_02"},{"date" => "2013-02-01T18:54:58","version" => "1.59_03"},{"date" => "2013-02-03T17:07:16","version" => "1.59_04"},{"date" => "2013-02-04T19:52:48","version" => "1.59_05"},{"date" => "2013-02-05T20:40:30","version" => "1.59_06"},{"date" => "2013-02-06T19:17:51","version" => "1.59_07"},{"date" => "2013-02-06T19:32:27","version" => "1.59_08"},{"date" => "2013-02-07T09:59:04","version" => "1.59_09"},{"date" => "2013-02-08T00:29:16","version" => "1.59_10"},{"date" => "2013-02-11T22:12:12","version" => "1.59_11"},{"date" => "2013-02-14T02:15:12","version" => "1.59_12"},{"date" => "2013-02-25T20:16:34","version" => "1.59_13"},{"date" => "2013-02-26T17:57:00","version" => "1.6000"},{"date" => "2013-02-27T01:04:54","version" => "1.6001"},{"date" => "2013-02-27T20:13:45","version" => "1.6002"},{"date" => "2013-03-08T19:03:47","version" => "1.6003"},{"date" => "2013-03-08T19:32:25","version" => "1.6004"},{"date" => "2013-03-08T19:48:06","version" => "1.6005"},{"date" => "2013-03-14T06:00:27","version" => "1.6006"},{"date" => "2013-03-17T21:34:17","version" => "1.6007"},{"date" => "2013-03-19T17:03:36","version" => "1.6008"},{"date" => "2013-03-25T04:10:51","version" => "1.6100"},{"date" => "2013-03-25T20:41:37","version" => "1.6101"},{"date" => "2013-03-28T00:16:09","version" => "1.6102"},{"date" => "2013-03-30T21:36:49","version" => "1.6103"},{"date" => "2013-04-03T01:04:42","version" => "1.6104"},{"date" => "2013-04-05T05:17:38","version" => "1.6105"},{"date" => "2013-04-06T21:19:18","version" => "1.6106"},{"date" => "2013-04-07T04:19:16","version" => "1.6107"},{"date" => "2013-04-13T06:32:52","version" => "1.6108"},{"date" => "2013-04-13T11:48:43","version" => "1.6190"},{"date" => "2013-04-14T03:09:40","version" => "1.6191"},{"date" => "2013-04-14T08:17:32","version" => "1.6192"},{"date" => "2013-04-15T07:37:08","version" => "1.6193"},{"date" => "2013-04-15T07:42:51","version" => "1.6900"},{"date" => "2013-04-21T00:50:44","version" => "1.6901"},{"date" => "2013-04-21T01:06:02","version" => "1.6109"},{"date" => "2013-04-21T01:18:10","version" => "1.6902"},{"date" => "2013-04-22T01:07:09","version" => "1.6903"},{"date" => "2013-04-24T02:24:37","version" => "1.6904"},{"date" => "2013-04-24T03:05:21","version" => "1.6905"},{"date" => "2013-04-25T06:20:23","version" => "1.6906"},{"date" => "2013-04-26T18:40:08","version" => "1.6907"},{"date" => "2013-04-27T01:12:17","version" => "1.6908"},{"date" => "2013-04-29T08:49:53","version" => "1.6909"},{"date" => "2013-05-03T07:29:32","version" => "1.6910"},{"date" => "2013-05-04T20:28:02","version" => "1.6911"},{"date" => "2013-05-06T20:59:52","version" => "1.6912"},{"date" => "2013-05-10T00:05:10","version" => "1.6913"},{"date" => "2013-05-12T23:03:52","version" => "1.6914"},{"date" => "2013-05-16T02:01:33","version" => "1.6915"},{"date" => "2013-06-04T10:55:37","version" => "1.6916"},{"date" => "2013-06-05T01:07:33","version" => "1.6917"},{"date" => "2013-06-10T20:03:21","version" => "1.6918"},{"date" => "2013-06-12T15:33:22","version" => "1.6919"},{"date" => "2013-06-14T21:09:54","version" => "1.6920"},{"date" => "2013-06-18T10:19:43","version" => "1.6921"},{"date" => "2013-06-19T20:57:09","version" => "1.6922"},{"date" => "2013-07-04T05:17:11","version" => "1.6923"},{"date" => "2013-07-16T18:38:21","version" => "1.6924"},{"date" => "2013-07-20T05:08:06","version" => "1.6925"},{"date" => "2013-07-20T16:03:14","version" => "1.6926"},{"date" => "2013-07-23T07:45:33","version" => "1.6927"},{"date" => "2013-07-23T21:07:02","version" => "1.6928"},{"date" => "2013-07-24T18:46:29","version" => "1.6929"},{"date" => "2013-07-24T20:48:14","version" => "1.6930"},{"date" => "2013-07-24T21:51:33","version" => "1.6931"},{"date" => "2013-07-24T22:29:04","version" => "1.6932"},{"date" => "2013-07-25T16:58:24","version" => "1.6933"},{"date" => "2013-07-26T23:17:21","version" => "1.6934"},{"date" => "2013-07-31T18:36:57","version" => "1.6935"},{"date" => "2013-08-05T04:37:54","version" => "1.6936"},{"date" => "2013-08-06T01:55:29","version" => "1.6937"},{"date" => "2013-08-06T06:12:45","version" => "1.6938"},{"date" => "2013-08-06T09:55:55","version" => "1.6939"},{"date" => "2013-08-08T19:36:34","version" => "1.6940"},{"date" => "2013-08-20T18:32:44","version" => "1.6941"},{"date" => "2013-08-27T18:11:47","version" => "1.6942"},{"date" => "2013-09-03T23:40:37","version" => "1.6943"},{"date" => "2013-09-04T22:02:21","version" => "1.7000"},{"date" => "2013-09-08T20:12:16","version" => "1.7001"},{"date" => "2013-09-19T05:31:34","version" => "1.7100"},{"date" => "2013-09-19T11:15:59","version" => "1.7101"},{"date" => "2013-09-20T04:33:50","version" => "1.7102"},{"date" => "2014-04-27T05:46:31","version" => "1.7002"},{"date" => "2014-04-27T15:11:46","version" => "1.7003"},{"date" => "2014-04-27T16:23:35","version" => "1.7004"},{"date" => "2014-09-02T04:00:49","version" => "1.7005"},{"date" => "2014-09-02T06:27:35","version" => "1.7006"},{"date" => "2014-09-05T12:04:41","version" => "1.7005"},{"date" => "2014-09-05T22:45:37","version" => "1.7006"},{"date" => "2014-09-09T16:26:54","version" => "1.7007"},{"date" => "2014-09-10T08:19:24","version" => "1.7008"},{"date" => "2014-09-10T08:44:00","version" => "1.7009"},{"date" => "2014-09-17T09:28:23","version" => "1.7010"},{"date" => "2014-09-22T06:08:51","version" => "1.7011"},{"date" => "2014-09-27T02:29:33","version" => "1.7012"},{"date" => "2014-10-07T06:52:45","version" => "1.7013"},{"date" => "2014-10-08T03:54:02","version" => "1.7014"},{"date" => "2014-11-14T21:14:40","version" => "1.7015"},{"date" => "2014-11-16T19:47:26","version" => "1.7016"},{"date" => "2014-11-25T22:01:56","version" => "1.7017"},{"date" => "2014-11-25T22:08:49","version" => "1.7018"},{"date" => "2014-12-04T20:52:24","version" => "1.7019"},{"date" => "2014-12-09T01:54:37","version" => "1.7020"},{"date" => "2014-12-12T05:43:01","version" => "1.7021"},{"date" => "2014-12-13T00:42:15","version" => "1.7022"},{"date" => "2015-01-04T23:00:30","version" => "1.7023"},{"date" => "2015-01-12T21:32:45","version" => "1.7024"},{"date" => "2015-02-07T06:59:17","version" => "1.7025"},{"date" => "2015-02-14T01:12:18","version" => "1.7026"},{"date" => "2015-02-14T20:15:20","version" => "1.7027"},{"date" => "2015-04-17T17:24:16","version" => "1.7028"},{"date" => "2015-04-18T22:16:17","version" => "1.7029"},{"date" => "2015-04-19T12:15:59","version" => "1.7030"},{"date" => "2015-04-22T21:14:17","version" => "1.7031"},{"date" => "2015-04-30T01:52:49","version" => "1.7032"},{"date" => "2015-05-02T00:18:54","version" => "1.7033"},{"date" => "2015-05-07T21:21:07","version" => "1.7034"},{"date" => "2015-06-05T17:51:53","version" => "1.7035"},{"date" => "2015-06-06T05:08:20","version" => "1.7036"},{"date" => "2015-06-18T21:38:47","version" => "1.7037"},{"date" => "2015-06-23T01:05:25","version" => "1.7038"},{"date" => "2015-06-29T01:06:18","version" => "1.7039"},{"date" => "2016-01-07T19:29:19","version" => "1.7040"},{"date" => "2016-05-08T18:29:30","version" => "1.7041"},{"date" => "2016-05-24T07:49:34","version" => "1.7042"},{"date" => "2017-04-03T03:57:15","version" => "1.7043"},{"date" => "2018-04-19T11:54:56","version" => "1.7044"},{"date" => "2018-04-20T12:17:48","version" => "1.7900"},{"date" => "2018-04-20T12:20:35","version" => "1.7901"},{"date" => "2018-04-20T12:43:24","version" => "1.7902"},{"date" => "2018-04-20T14:54:30","version" => "1.7903"},{"date" => "2018-04-20T21:22:56","version" => "1.7904"},{"date" => "2018-04-21T09:40:47","version" => "1.7905"},{"date" => "2018-04-21T10:57:20","version" => "1.9015"},{"date" => "2018-04-21T11:17:58","version" => "1.9016"},{"date" => "2018-04-21T17:31:13","version" => "1.9017"},{"date" => "2018-04-22T13:54:32","version" => "1.9018"},{"date" => "2018-04-25T09:27:31","version" => "1.7906"},{"date" => "2018-04-26T11:36:59","version" => "1.7907"},{"date" => "2022-01-27T03:05:02","version" => "1.7045"},{"date" => "2022-04-27T06:01:26","version" => "1.7046"},{"date" => "2023-07-30T06:01:02","version" => "1.7047"},{"date" => "2024-10-29T18:49:03","version" => "1.7048"}]},"App-japerl" => {"advisories" => [{"affected_versions" => ["<0.09"],"cves" => ["CVE-2016-1238"],"description" => "(1) cpan/Archive-Tar/bin/ptar, (2) cpan/Archive-Tar/bin/ptardiff, (3) cpan/Archive-Tar/bin/ptargrep, (4) cpan/CPAN/scripts/cpan, (5) cpan/Digest-SHA/shasum, (6) cpan/Encode/bin/enc2xs, (7) cpan/Encode/bin/encguess, (8) cpan/Encode/bin/piconv, (9) cpan/Encode/bin/ucmlint, (10) cpan/Encode/bin/unidump, (11) cpan/ExtUtils-MakeMaker/bin/instmodsh, (12) cpan/IO-Compress/bin/zipdetails, (13) cpan/JSON-PP/bin/json_pp, (14) cpan/Test-Harness/bin/prove, (15) dist/ExtUtils-ParseXS/lib/ExtUtils/xsubpp, (16) dist/Module-CoreList/corelist, (17) ext/Pod-Html/bin/pod2html, (18) utils/c2ph.PL, (19) utils/h2ph.PL, (20) utils/h2xs.PL, (21) utils/libnetcfg.PL, (22) utils/perlbug.PL, (23) utils/perldoc.PL, (24) utils/perlivp.PL, and (25) utils/splain.PL in Perl 5.x before 5.22.3-RC2 and 5.24 before 5.24.1-RC2 do not properly remove . (period) characters from the end of the includes directory array, which might allow local users to gain privileges via a Trojan horse module under the current working directory.\n","distribution" => "App-japerl","fixed_versions" => [">=0.09"],"id" => "CPANSA-App-japerl-2016-1238","references" => ["http://www.nntp.perl.org/group/perl.perl5.porters/2016/07/msg238271.html","http://www.securitytracker.com/id/1036440","http://perl5.git.perl.org/perl.git/commit/cee96d52c39b1e7b36e1c62d38bcd8d86e9a41ab","https://rt.perl.org/Public/Bug/Display.html?id=127834","http://www.securityfocus.com/bid/92136","http://www.debian.org/security/2016/dsa-3628","https://h20566.www2.hpe.com/portal/site/hpsc/public/kb/docDisplay?docId=emr_na-c05240731","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DOFRQWJRP2NQJEYEWOMECVW3HAMD5SYN/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/TZBNQH3DMI7HDELJAZ4TFJJANHXOEDWH/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/2FBQOCV3GBAN2EYZUM3CFDJ4ECA3GZOK/","https://security.gentoo.org/glsa/201701-75","https://lists.apache.org/thread.html/7f6a16bc0fd0fd5e67c7fd95bd655069a2ac7d1f88e42d3c853e601c\@%3Cannounce.apache.org%3E","https://lists.debian.org/debian-lts-announce/2018/11/msg00016.html","https://security.gentoo.org/glsa/201812-07","http://lists.opensuse.org/opensuse-security-announce/2019-08/msg00002.html"],"reported" => "2016-08-02","severity" => "high"}],"main_module" => "App::japerl","versions" => [{"date" => "2018-09-07T15:19:24","version" => "0.10"},{"date" => "2018-09-08T15:07:20","version" => "0.11"},{"date" => "2019-07-14T03:35:18","version" => "0.12"},{"date" => "2021-02-18T14:03:58","version" => "0.13"},{"date" => "2021-09-18T18:20:37","version" => "0.14"},{"date" => "2023-03-25T01:04:11","version" => "0.15"}]},"App-perlall" => {"advisories" => [{"affected_versions" => ["<0.33"],"cves" => ["CVE-2013-1667"],"description" => "The rehash mechanism in Perl 5.8.2 through 5.16.x allows context-dependent attackers to cause a denial of service (memory consumption and crash) via a crafted hash key.\n","distribution" => "App-perlall","fixed_versions" => [">=0.33"],"id" => "CPANSA-App-perlall-2013-1667","references" => ["http://www.securityfocus.com/bid/58311","http://perl5.git.perl.org/perl.git/commitdiff/d59e31f","http://perl5.git.perl.org/perl.git/commitdiff/9d83adc","http://www.nntp.perl.org/group/perl.perl5.porters/2013/03/msg199755.html","http://www.debian.org/security/2013/dsa-2641","http://secunia.com/advisories/52499","http://secunia.com/advisories/52472","https://bugzilla.redhat.com/show_bug.cgi?id=912276","http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=702296","http://perl5.git.perl.org/perl.git/commitdiff/6e79fe5","http://osvdb.org/90892","http://www.ubuntu.com/usn/USN-1770-1","http://rhn.redhat.com/errata/RHSA-2013-0685.html","http://lists.apple.com/archives/security-announce/2013/Oct/msg00004.html","http://marc.info/?l=bugtraq&m=137891988921058&w=2","http://www.mandriva.com/security/advisories?name=MDVSA-2013:113","https://wiki.mageia.org/en/Support/Advisories/MGASA-2013-0094","http://www.oracle.com/technetwork/topics/security/ovmbulletinjul2016-3090546.html","http://kb.juniper.net/InfoCenter/index?page=content&id=JSA10735","http://kb.juniper.net/InfoCenter/index?page=content&id=JSA10705","https://exchange.xforce.ibmcloud.com/vulnerabilities/82598","https://oval.cisecurity.org/repository/search/definition/oval%3Aorg.mitre.oval%3Adef%3A18771"],"reported" => "2013-03-14","severity" => undef}],"main_module" => "App::perlall","versions" => [{"date" => "2011-12-23T21:52:22","version" => "0.01"},{"date" => "2011-12-24T00:56:03","version" => "0.02"},{"date" => "2012-01-06T17:07:08","version" => "0.03"},{"date" => "2012-01-09T22:05:35","version" => "0.04"},{"date" => "2012-01-31T21:18:20","version" => "0.05"},{"date" => "2012-02-06T23:12:27","version" => "0.06"},{"date" => "2012-02-07T20:52:55","version" => "0.07"},{"date" => "2012-02-23T10:35:50","version" => "0.08"},{"date" => "2012-03-22T18:24:53","version" => "0.09"},{"date" => "2012-05-03T13:44:26","version" => "0.10"},{"date" => "2012-05-05T02:22:56","version" => "0.11"},{"date" => "2012-05-05T14:18:09","version" => "0.12"},{"date" => "2012-05-29T15:34:02","version" => "0.13"},{"date" => "2012-06-07T16:07:09","version" => "0.14"},{"date" => "2012-07-18T17:55:03","version" => "0.15"},{"date" => "2012-07-18T18:05:33","version" => "0.15_01"},{"date" => "2012-07-19T19:07:14","version" => "0.16"},{"date" => "2012-08-06T15:11:54","version" => "0.17"},{"date" => "2012-11-06T22:12:59","version" => "0.18"},{"date" => "2012-11-08T15:37:31","version" => "0.19"},{"date" => "2012-11-08T15:50:30","version" => "0.20"},{"date" => "2012-11-08T18:53:37","version" => "0.21"},{"date" => "2012-11-09T22:04:21","version" => "0.22"},{"date" => "2012-11-11T19:50:41","version" => "0.23"},{"date" => "2012-11-13T20:46:09","version" => "0.25"},{"date" => "2012-11-13T22:45:49","version" => "0.26"},{"date" => "2012-11-15T16:26:40","version" => "0.27"},{"date" => "2012-12-13T20:09:18","version" => "0.28"},{"date" => "2012-12-20T22:29:59","version" => "0.29"},{"date" => "2013-01-09T20:22:21","version" => "0.30"},{"date" => "2013-02-04T19:58:18","version" => "0.31"},{"date" => "2013-02-23T21:35:31","version" => "0.32"},{"date" => "2013-03-05T01:04:28","version" => "0.33"},{"date" => "2013-03-05T15:34:37","version" => "0.34"},{"date" => "2013-03-22T22:34:57","version" => "0.35"},{"date" => "2013-07-11T19:58:07","version" => "0.36"},{"date" => "2013-07-13T19:53:25","version" => "0.37"},{"date" => "2013-10-23T15:58:48","version" => "0.39"},{"date" => "2013-11-18T16:12:38","version" => "0.40"},{"date" => "2013-12-03T17:08:11","version" => "0.41"},{"date" => "2013-12-06T18:40:51","version" => "0.42"},{"date" => "2013-12-09T18:31:19","version" => "0.43"},{"date" => "2014-01-11T23:39:19","version" => "0.44"},{"date" => "2014-07-25T13:36:23","version" => "0.45"},{"date" => "2014-08-10T01:42:03","version" => "0.46"},{"date" => "2015-07-08T07:43:56","version" => "0.47"},{"date" => "2015-10-06T09:33:35","version" => "0.48"},{"date" => "2015-11-27T15:53:11","version" => "0.49"},{"date" => "2016-06-12T12:48:37","version" => "0.50"},{"date" => "2019-12-10T20:02:45","version" => "0.51"}]},"App-revealup" => {"advisories" => [{"affected_versions" => [">=0.01,<=0.19"],"cves" => ["CVE-2020-8127"],"description" => "Insufficient validation in cross-origin communication (postMessage) in reveal.js version 3.9.1 and earlier allow attackers to perform cross-site scripting attacks.\n","distribution" => "App-revealup","fixed_versions" => [],"id" => "CPANSA-App-revealup-2020-8127-revealjs","references" => ["https://hackerone.com/reports/691977"],"reported" => "2020-02-28","severity" => "medium"},{"affected_versions" => [">=0.20,<=0.21"],"cves" => ["CVE-2020-8127"],"description" => "Insufficient validation in cross-origin communication (postMessage) in reveal.js version 3.9.1 and earlier allow attackers to perform cross-site scripting attacks.\n","distribution" => "App-revealup","fixed_versions" => [],"id" => "CPANSA-App-revealup-2020-8127-revealjs","references" => ["https://hackerone.com/reports/691977"],"reported" => "2020-02-28","severity" => "medium"},{"affected_versions" => ["==0.22"],"cves" => ["CVE-2022-0776"],"description" => "Cross-site Scripting (XSS) - DOM in GitHub repository hakimel/reveal.js prior to 4.3.0.\n","distribution" => "App-revealup","fixed_versions" => [],"id" => "CPANSA-App-revealup-2022-0776-revealjs","references" => ["https://github.com/hakimel/reveal.js/commit/32cdd3b1872ba8e2267c9e87ae216cb55f40f4d2","https://huntr.dev/bounties/be2b7ee4-f487-42e1-874a-6bcc410e4001","https://github.com/yusukebe/App-revealup/issues/12#issuecomment-1169417411","https://github.com/yusukebe/App-revealup/commit/c8fea67994b1aa6d734066bff9ada4e834b09cb7"],"reported" => "2022-03-01","severity" => "medium"}],"main_module" => "App::revealup","versions" => [{"date" => "2014-05-25T10:34:08","version" => "0.01"},{"date" => "2014-05-25T11:39:22","version" => "0.02"},{"date" => "2014-06-06T08:03:43","version" => "0.03"},{"date" => "2014-06-06T22:08:16","version" => "0.04"},{"date" => "2014-06-11T05:44:23","version" => "0.05"},{"date" => "2014-06-11T06:22:41","version" => "0.06"},{"date" => "2014-06-11T11:27:29","version" => "0.07"},{"date" => "2014-06-16T01:22:48","version" => "0.08"},{"date" => "2014-06-17T02:53:12","version" => "0.09"},{"date" => "2014-07-05T21:47:41","version" => "0.10"},{"date" => "2014-07-05T21:54:30","version" => "0.11"},{"date" => "2014-09-03T20:57:24","version" => "0.12"},{"date" => "2014-09-16T03:40:03","version" => "0.13"},{"date" => "2014-11-07T16:32:52","version" => "0.14"},{"date" => "2014-11-24T06:40:45","version" => "0.15"},{"date" => "2014-12-19T06:25:32","version" => "0.16"},{"date" => "2014-12-19T20:12:33","version" => "0.17"},{"date" => "2014-12-21T22:32:08","version" => "0.18"},{"date" => "2014-12-21T22:43:49","version" => "0.19"},{"date" => "2015-07-07T15:34:28","version" => "0.20"},{"date" => "2015-08-28T12:57:12","version" => "0.21"},{"date" => "2020-02-06T12:53:05","version" => "0.22"},{"date" => "2022-06-29T00:31:20","version" => "0.23"}]},"Archive-Tar" => {"advisories" => [{"affected_versions" => ["<2.28"],"cves" => ["CVE-2018-12015"],"description" => "In Perl through 5.26.2, the Archive::Tar module allows remote attackers to bypass a directory-traversal protection mechanism, and overwrite arbitrary files, via an archive file containing a symlink and a regular file with the same name.\n","distribution" => "Archive-Tar","fixed_versions" => [">=2.28"],"id" => "CPANSA-Archive-Tar-2018-01","references" => ["https://security-tracker.debian.org/tracker/CVE-2018-12015","https://github.com/jib/archive-tar-new/commit/ae65651eab053fc6dc4590dbb863a268215c1fc5"],"reported" => "2018-06-12","severity" => "medium"},{"affected_versions" => ["<=1.36"],"cves" => ["CVE-2007-4829"],"description" => "Directory traversal vulnerability in the Archive::Tar Perl module 1.36 and earlier allows user-assisted remote attackers to overwrite arbitrary files via a TAR archive that contains a file whose name is an absolute path or has \"..\" sequences.\n","distribution" => "Archive-Tar","fixed_versions" => [">1.36"],"id" => "CPANSA-Archive-Tar-2007-4829","references" => ["http://rt.cpan.org/Public/Bug/Display.html?id=29517","https://bugzilla.redhat.com/show_bug.cgi?id=295021","http://rt.cpan.org/Public/Bug/Display.html?id=30380","https://issues.rpath.com/browse/RPL-1716","http://www.securityfocus.com/bid/26355","http://secunia.com/advisories/27539","http://osvdb.org/40410","http://www.ubuntu.com/usn/usn-700-1","http://secunia.com/advisories/33314","http://www.ubuntu.com/usn/usn-700-2","http://www.gentoo.org/security/en/glsa/glsa-200812-10.xml","http://secunia.com/advisories/33116","http://www.vupen.com/english/advisories/2007/3755","https://exchange.xforce.ibmcloud.com/vulnerabilities/38285","https://oval.cisecurity.org/repository/search/definition/oval%3Aorg.mitre.oval%3Adef%3A11658"],"reported" => "2007-11-02","severity" => undef},{"affected_versions" => ["<2.10"],"cves" => ["CVE-2016-1238"],"description" => "'(1) cpan/Archive-Tar/bin/ptar, (2) cpan/Archive-Tar/bin/ptardiff, (3) cpan/Archive-Tar/bin/ptargrep, (4) cpan/CPAN/scripts/cpan, (5) cpan/Digest-SHA/shasum, (6) cpan/Encode/bin/enc2xs, (7) cpan/Encode/bin/encguess, (8) cpan/Encode/bin/piconv, (9) cpan/Encode/bin/ucmlint, (10) cpan/Encode/bin/unidump, (11) cpan/ExtUtils-MakeMaker/bin/instmodsh, (12) cpan/IO-Compress/bin/zipdetails, (13) cpan/JSON-PP/bin/json_pp, (14) cpan/Test-Harness/bin/prove, (15) dist/ExtUtils-ParseXS/lib/ExtUtils/xsubpp, (16) dist/Module-CoreList/corelist, (17) ext/Pod-Html/bin/pod2html, (18) utils/c2ph.PL, (19) utils/h2ph.PL, (20) utils/h2xs.PL, (21) utils/libnetcfg.PL, (22) utils/perlbug.PL, (23) utils/perldoc.PL, (24) utils/perlivp.PL, and (25) utils/splain.PL in Perl 5.x before 5.22.3-RC2 and 5.24 before 5.24.1-RC2 do not properly remove . (period) characters from the end of the includes directory array, which might allow local users to gain privileges via a Trojan horse module under the current working directory.'\n","distribution" => "Archive-Tar","fixed_versions" => [">=2.10"],"id" => "CPANSA-Archive-Tar-2016-1238","references" => ["http://www.nntp.perl.org/group/perl.perl5.porters/2016/07/msg238271.html","http://www.securitytracker.com/id/1036440","http://perl5.git.perl.org/perl.git/commit/cee96d52c39b1e7b36e1c62d38bcd8d86e9a41ab","https://rt.perl.org/Public/Bug/Display.html?id=127834","http://www.securityfocus.com/bid/92136","http://www.debian.org/security/2016/dsa-3628","https://h20566.www2.hpe.com/portal/site/hpsc/public/kb/docDisplay?docId=emr_na-c05240731","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DOFRQWJRP2NQJEYEWOMECVW3HAMD5SYN/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/TZBNQH3DMI7HDELJAZ4TFJJANHXOEDWH/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/2FBQOCV3GBAN2EYZUM3CFDJ4ECA3GZOK/","https://security.gentoo.org/glsa/201701-75","https://lists.apache.org/thread.html/7f6a16bc0fd0fd5e67c7fd95bd655069a2ac7d1f88e42d3c853e601c\@%3Cannounce.apache.org%3E","https://lists.debian.org/debian-lts-announce/2018/11/msg00016.html","https://security.gentoo.org/glsa/201812-07","http://lists.opensuse.org/opensuse-security-announce/2019-08/msg00002.html"],"reported" => "2016-08-02","severity" => "high"}],"main_module" => "Archive::Tar","versions" => [{"date" => "1998-02-02T06:13:59","version" => "0.071"},{"date" => "1998-04-10T17:07:35","version" => "0.072"},{"date" => "1998-07-30T00:56:03","version" => "0.08"},{"date" => "1999-01-10T02:22:23","version" => "0.20"},{"date" => "1999-02-02T19:01:41","version" => "0.21"},{"date" => "2000-04-28T00:37:46","version" => "0.22"},{"date" => "2003-01-21T23:07:30","version" => "0.23"},{"date" => "2003-03-18T17:08:50","version" => "0.99_01"},{"date" => "2003-03-26T14:57:35","version" => "0.99_02"},{"date" => "2003-04-28T16:01:24","version" => "0.99_03"},{"date" => "2003-04-28T16:57:58","version" => "0.99_04"},{"date" => "2003-04-30T12:52:19","version" => "0.99_05"},{"date" => "2003-05-05T12:06:35","version" => "0.99_06"},{"date" => "2003-05-31T09:27:33","version" => "1.00"},{"date" => "2003-06-08T10:46:56","version" => "1.01"},{"date" => "2003-06-12T09:47:58","version" => "1.02"},{"date" => "2003-06-26T12:52:19","version" => "1.03"},{"date" => "2003-07-27T17:07:50","version" => "1.04"},{"date" => "2003-08-25T13:38:44","version" => "1.05"},{"date" => "2003-10-15T14:35:12","version" => "1.06"},{"date" => "2003-10-17T11:42:14","version" => "1.07"},{"date" => "2004-01-05T12:59:23","version" => "1.08"},{"date" => "2004-05-22T12:32:02","version" => "1.09"},{"date" => "2004-06-11T19:24:06","version" => "1.10"},{"date" => "2004-11-09T16:12:40","version" => "1.20"},{"date" => "2004-11-10T16:04:13","version" => "1.21"},{"date" => "2004-11-21T10:09:52","version" => "1.22"},{"date" => "2004-12-03T15:53:06","version" => "1.23"},{"date" => "2005-05-03T13:11:19","version" => "1.24"},{"date" => "2005-08-20T10:14:40","version" => "1.25"},{"date" => "2005-08-22T09:29:53","version" => "1.26"},{"date" => "2006-01-19T13:31:53","version" => "1.28"},{"date" => "2006-03-03T13:56:20","version" => "1.29"},{"date" => "2006-08-02T15:00:41","version" => "1.30"},{"date" => "2007-05-18T12:18:49","version" => "1.31"},{"date" => "2007-05-25T09:32:48","version" => "1.32"},{"date" => "2007-08-15T14:20:33","version" => "1.34"},{"date" => "2007-09-16T09:13:21","version" => "1.36"},{"date" => "2007-11-11T11:59:00","version" => "1.37_01"},{"date" => "2007-12-24T11:02:07","version" => "1.38"},{"date" => "2008-08-22T16:33:49","version" => "1.39_01"},{"date" => "2008-08-25T03:56:58","version" => "1.39_02"},{"date" => "2008-08-25T22:07:56","version" => "1.39_03"},{"date" => "2008-09-08T12:14:37","version" => "1.39_04"},{"date" => "2008-10-13T13:42:10","version" => "1.40"},{"date" => "2008-12-13T17:10:15","version" => "1.42"},{"date" => "2009-01-19T17:08:08","version" => "1.44"},{"date" => "2009-03-05T16:10:06","version" => "1.46"},{"date" => "2009-04-20T17:07:30","version" => "1.48"},{"date" => "2009-06-12T12:01:54","version" => "1.50"},{"date" => "2009-06-13T11:29:50","version" => "1.52"},{"date" => "2009-09-10T12:13:03","version" => "1.54"},{"date" => "2010-02-03T14:40:15","version" => "1.56"},{"date" => "2010-02-17T21:47:16","version" => "1.58"},{"date" => "2010-04-23T14:12:31","version" => "1.60"},{"date" => "2010-06-28T21:02:59","version" => "1.62"},{"date" => "2010-07-09T11:04:45","version" => "1.64"},{"date" => "2010-07-26T08:44:00","version" => "1.66"},{"date" => "2010-08-17T16:06:19","version" => "1.68"},{"date" => "2010-11-15T22:02:53","version" => "1.70"},{"date" => "2010-11-18T19:22:01","version" => "1.72"},{"date" => "2010-12-18T21:19:51","version" => "1.74"},{"date" => "2011-01-07T22:27:40","version" => "1.76"},{"date" => "2011-09-08T22:13:33","version" => "1.78"},{"date" => "2011-10-13T10:25:39","version" => "1.80"},{"date" => "2011-11-21T12:14:43","version" => "1.82"},{"date" => "2012-03-03T00:00:05","version" => "1.84"},{"date" => "2012-05-24T11:38:09","version" => "1.86"},{"date" => "2012-06-01T11:06:25","version" => "1.88"},{"date" => "2012-09-05T18:19:00","version" => "1.90"},{"date" => "2013-06-18T15:13:27","version" => "1.92"},{"date" => "2013-10-22T14:28:22","version" => "0.93_01"},{"date" => "2013-10-22T14:36:08","version" => "1.93_02"},{"date" => "2013-10-24T18:02:48","version" => "1.94"},{"date" => "2013-10-24T19:10:34","version" => "1.96"},{"date" => "2014-06-14T17:12:02","version" => "1.98"},{"date" => "2014-06-15T14:59:24","version" => "2.00"},{"date" => "2014-09-14T18:03:23","version" => "2.02"},{"date" => "2014-12-14T20:13:33","version" => "2.04"},{"date" => "2016-04-24T14:05:11","version" => "2.06"},{"date" => "2016-05-12T08:57:35","version" => "2.08"},{"date" => "2016-07-27T12:40:29","version" => "2.10"},{"date" => "2016-10-16T11:27:58","version" => "2.12"},{"date" => "2016-10-20T12:38:57","version" => "2.14"},{"date" => "2016-11-01T19:19:36","version" => "2.16"},{"date" => "2016-11-07T13:36:15","version" => "2.18"},{"date" => "2016-12-15T10:54:40","version" => "2.20"},{"date" => "2016-12-16T09:46:28","version" => "2.22"},{"date" => "2016-12-16T15:27:38","version" => "2.24"},{"date" => "2017-05-12T12:46:05","version" => "2.26"},{"date" => "2018-06-08T10:57:04","version" => "2.28"},{"date" => "2018-06-19T11:55:28","version" => "2.30"},{"date" => "2018-09-13T07:17:10","version" => "2.32"},{"date" => "2020-02-01T16:41:47","version" => "2.34"},{"date" => "2020-02-02T13:34:34","version" => "2.36"},{"date" => "2020-06-25T07:51:56","version" => "2.38"},{"date" => "2021-07-27T09:51:54","version" => "2.40"},{"date" => "2023-03-25T12:10:20","version" => "3.00"},{"date" => "2023-04-12T23:09:11","version" => "3.02"},{"date" => "2025-02-25T20:25:09","version" => "3.04"},{"date" => "2006-01-28T00:00:00","dual_lived" => 1,"perl_release" => "5.009003","version" => "1.26_01"},{"date" => "2006-08-15T00:00:00","dual_lived" => 1,"perl_release" => "5.009004","version" => "1.30_01"},{"date" => "2017-01-14T00:00:00","dual_lived" => 1,"perl_release" => "5.022003","version" => "2.04_01"},{"date" => "2018-11-29T00:00:00","dual_lived" => 1,"perl_release" => "5.026003","version" => "2.24_01"},{"date" => "2023-12-30T00:00:00","dual_lived" => 1,"perl_release" => "5.039006","version" => "3.02_001"}]},"Archive-Unzip-Burst" => {"advisories" => [{"affected_versions" => [">=0.01"],"comment" => "0.09 is the latest version, so all versions are affected","cves" => ["CVE-2022-4976"],"description" => "Archive::Unzip::Burst from 0.01 through 0.09 for Perl contains a bundled InfoZip library that is affected by several vulnerabilities.  The bundled library is affected by CVE-2014-8139, CVE-2014-8140 and CVE-2014-8141.","distribution" => "Archive-Unzip-Burst","fixed_versions" => [],"id" => "CPANSA-Archive-Unzip-Burst-2022-4976","references" => ["https://rt.cpan.org/Public/Bug/Display.html?id=143547"],"reported" => "2025-06-12","severity" => undef},{"affected_versions" => [">=0.01,<=0.02_02"],"cves" => ["CVE-2014-8141"],"description" => "Heap-based buffer overflow in the getZip64Data function in Info-ZIP UnZip 6.0 and earlier allows remote attackers to execute arbitrary code via a crafted zip file in the -t command argument to the unzip command.\n","distribution" => "Archive-Unzip-Burst","fixed_versions" => [],"id" => "CPANSA-Archive-Unzip-Burst-2014-8141-unzip","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143547","https://bugzilla.redhat.com/show_bug.cgi?id=1174856","http://www.securitytracker.com/id/1031433","http://www.ocert.org/advisories/ocert-2014-011.html","https://access.redhat.com/errata/RHSA-2015:0700"],"reported" => "2020-01-31","severity" => "high"},{"affected_versions" => [">=0.01,<=0.02_02"],"cves" => ["CVE-2014-8140"],"description" => "Heap-based buffer overflow in the test_compr_eb function in Info-ZIP UnZip 6.0 and earlier allows remote attackers to execute arbitrary code via a crafted zip file in the -t command argument to the unzip command.\n","distribution" => "Archive-Unzip-Burst","fixed_versions" => [],"id" => "CPANSA-Archive-Unzip-Burst-2014-8140-unzip","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143547","https://bugzilla.redhat.com/show_bug.cgi?id=1174851","http://www.securitytracker.com/id/1031433","http://www.ocert.org/advisories/ocert-2014-011.html","https://access.redhat.com/errata/RHSA-2015:0700"],"reported" => "2020-01-31","severity" => "high"},{"affected_versions" => [">=0.01,<=0.02_02"],"cves" => ["CVE-2014-8139"],"description" => "Heap-based buffer overflow in the CRC32 verification in Info-ZIP UnZip 6.0 and earlier allows remote attackers to execute arbitrary code via a crafted zip file in the -t command argument to the unzip command.\n","distribution" => "Archive-Unzip-Burst","fixed_versions" => [],"id" => "CPANSA-Archive-Unzip-Burst-2014-8139-unzip","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143547","http://www.securitytracker.com/id/1031433","http://www.ocert.org/advisories/ocert-2014-011.html","https://access.redhat.com/errata/RHSA-2015:0700","https://bugzilla.redhat.com/show_bug.cgi?id=1174844"],"reported" => "2020-01-31","severity" => "high"},{"affected_versions" => [">=0.03,<=0.09"],"cves" => ["CVE-2014-8141"],"description" => "Heap-based buffer overflow in the getZip64Data function in Info-ZIP UnZip 6.0 and earlier allows remote attackers to execute arbitrary code via a crafted zip file in the -t command argument to the unzip command.\n","distribution" => "Archive-Unzip-Burst","fixed_versions" => [],"id" => "CPANSA-Archive-Unzip-Burst-2014-8141-unzip","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143547","https://bugzilla.redhat.com/show_bug.cgi?id=1174856","http://www.securitytracker.com/id/1031433","http://www.ocert.org/advisories/ocert-2014-011.html","https://access.redhat.com/errata/RHSA-2015:0700"],"reported" => "2020-01-31","severity" => "high"},{"affected_versions" => [">=0.03,<=0.09"],"cves" => ["CVE-2014-8140"],"description" => "Heap-based buffer overflow in the test_compr_eb function in Info-ZIP UnZip 6.0 and earlier allows remote attackers to execute arbitrary code via a crafted zip file in the -t command argument to the unzip command.\n","distribution" => "Archive-Unzip-Burst","fixed_versions" => [],"id" => "CPANSA-Archive-Unzip-Burst-2014-8140-unzip","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143547","https://bugzilla.redhat.com/show_bug.cgi?id=1174851","http://www.securitytracker.com/id/1031433","http://www.ocert.org/advisories/ocert-2014-011.html","https://access.redhat.com/errata/RHSA-2015:0700"],"reported" => "2020-01-31","severity" => "high"},{"affected_versions" => [">=0.03,<=0.09"],"cves" => ["CVE-2014-8139"],"description" => "Heap-based buffer overflow in the CRC32 verification in Info-ZIP UnZip 6.0 and earlier allows remote attackers to execute arbitrary code via a crafted zip file in the -t command argument to the unzip command.\n","distribution" => "Archive-Unzip-Burst","fixed_versions" => [],"id" => "CPANSA-Archive-Unzip-Burst-2014-8139-unzip","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143547","http://www.securitytracker.com/id/1031433","http://www.ocert.org/advisories/ocert-2014-011.html","https://access.redhat.com/errata/RHSA-2015:0700","https://bugzilla.redhat.com/show_bug.cgi?id=1174844"],"reported" => "2020-01-31","severity" => "high"}],"main_module" => "Archive::Unzip::Burst","versions" => [{"date" => "2007-07-29T11:01:34","version" => "0.01"},{"date" => "2007-08-30T13:06:24","version" => "0.02"},{"date" => "2007-09-05T15:27:53","version" => "0.02_01"},{"date" => "2008-05-16T12:03:35","version" => "0.02_02"},{"date" => "2015-01-15T02:22:44","version" => "0.03"},{"date" => "2016-05-01T14:28:01","version" => "0.04"},{"date" => "2016-05-02T04:28:22","version" => "0.05"},{"date" => "2016-05-07T01:31:26","version" => "0.06"},{"date" => "2016-05-08T17:38:43","version" => "0.07"},{"date" => "2016-05-08T17:42:49","version" => "0.08"},{"date" => "2018-03-16T20:38:14","version" => "0.09"},{"date" => "2025-05-19T13:29:32","version" => "0.03"}]},"Archive-Zip" => {"advisories" => [{"affected_versions" => ["<1.61"],"cves" => ["CVE-2018-10860"],"description" => "perl-archive-zip is vulnerable to a directory traversal in Archive::Zip. It was found that the Archive::Zip module did not properly sanitize paths while extracting zip files. An attacker able to provide a specially crafted archive for processing could use this flaw to write or overwrite arbitrary files in the context of the perl interpreter.\n","distribution" => "Archive-Zip","fixed_versions" => [],"id" => "CPANSA-Archive-Zip-2018-01","references" => ["https://security-tracker.debian.org/tracker/CVE-2018-10860","https://github.com/redhotpenguin/perl-Archive-Zip/pull/33"],"reported" => "2018-06-28","severity" => "medium"},{"affected_versions" => ["<1.14"],"cves" => ["CVE-2004-1096"],"description" => "Archive::Zip Perl module before 1.14, when used by antivirus programs such as amavisd-new, allows remote attackers to bypass antivirus protection via a compressed file with both local and global headers set to zero, which does not prevent the compressed file from being opened on a target system.\n","distribution" => "Archive-Zip","fixed_versions" => [],"id" => "CPANSA-Archive-Zip-2004-1096","references" => ["http://www.securityfocus.com/bid/11448","http://www.gentoo.org/security/en/glsa/glsa-200410-31.xml","http://www.idefense.com/application/poi/display?id=153&type=vulnerabilities&flashstatus=true","http://www.kb.cert.org/vuls/id/492545","http://secunia.com/advisories/13038/","http://www.mandriva.com/security/advisories?name=MDKSA-2004:118","https://exchange.xforce.ibmcloud.com/vulnerabilities/17761"],"reported" => "2005-01-10","severity" => undef}],"main_module" => "Archive::Zip","versions" => [{"date" => "2000-03-22T00:10:21","version" => "0.06"},{"date" => "2000-03-29T17:03:46","version" => "0.07"},{"date" => "2000-06-16T16:48:41","version" => "0.09"},{"date" => "2000-08-08T20:56:31","version" => "0.10"},{"date" => "2001-01-17T08:06:58","version" => "0.11"},{"date" => "2002-04-22T15:32:49","version" => "1.00"},{"date" => "2002-05-11T02:45:20","version" => "1.01"},{"date" => "2002-08-24T00:19:19","version" => "1.02"},{"date" => "2002-09-03T04:40:33","version" => "1.03"},{"date" => "2002-09-11T15:17:37","version" => "1.04"},{"date" => "2002-09-11T19:35:26","version" => "1.05"},{"date" => "2003-07-17T18:18:14","version" => "1.06"},{"date" => "2003-10-20T13:59:00","version" => "1.07"},{"date" => "2003-10-21T17:04:03","version" => "1.08"},{"date" => "2003-11-27T18:02:03","version" => "1.09"},{"date" => "2004-03-25T14:39:05","version" => "1.10"},{"date" => "2004-07-05T23:25:19","version" => "1_11"},{"date" => "2004-07-08T17:31:27","version" => "1.11"},{"date" => "2004-07-08T19:14:46","version" => "1.12"},{"date" => "2004-07-27T22:50:39","version" => "1.12_02"},{"date" => "2004-07-29T15:15:49","version" => "1.12_03"},{"date" => "2004-08-23T15:39:23","version" => "1.13"},{"date" => "2004-10-21T15:28:12","version" => "1.14"},{"date" => "2005-03-10T04:34:04","version" => "1.15_01"},{"date" => "2005-03-12T15:29:48","version" => "1.15_02"},{"date" => "2005-06-22T18:29:34","version" => "1.15"},{"date" => "2005-07-04T17:55:17","version" => "1.16"},{"date" => "2006-04-30T03:53:15","version" => "1.17_01"},{"date" => "2006-05-07T02:49:30","version" => "1.17_02"},{"date" => "2006-09-15T15:56:10","version" => "1.17_03"},{"date" => "2006-10-24T15:06:32","version" => "1.17_05"},{"date" => "2006-10-25T12:24:52","version" => "1.18"},{"date" => "2007-06-05T01:50:42","version" => "1.20"},{"date" => "2007-11-01T02:59:20","version" => "1.21"},{"date" => "2007-11-02T01:52:47","version" => "1.22"},{"date" => "2007-11-07T13:04:41","version" => "1.23"},{"date" => "2008-08-23T23:35:50","version" => "1.24"},{"date" => "2008-10-10T05:28:17","version" => "1.25"},{"date" => "2008-10-12T14:13:05","version" => "1.26"},{"date" => "2008-12-16T13:23:21","version" => "1.27_01"},{"date" => "2009-06-16T10:09:03","version" => "1.28"},{"date" => "2009-06-29T13:27:17","version" => "1.29"},{"date" => "2009-06-30T14:13:29","version" => "1.30"},{"date" => "2010-03-05T05:11:20","version" => "1.31_01"},{"date" => "2011-03-08T15:52:02","version" => "1.31_02"},{"date" => "2011-08-23T03:42:14","version" => "1.31_03"},{"date" => "2012-01-23T06:28:16","version" => "1.31_04"},{"date" => "2013-11-09T00:05:06","version" => "1.32"},{"date" => "2013-11-10T03:50:45","version" => "1.33"},{"date" => "2013-12-02T22:16:54","version" => "1.34"},{"date" => "2013-12-30T19:16:52","version" => "1.35"},{"date" => "2013-12-30T22:12:14","version" => "1.36"},{"date" => "2014-01-13T18:32:19","version" => "1.37"},{"date" => "2014-09-02T23:23:11","version" => "1.38"},{"date" => "2014-10-22T04:17:15","version" => "1.39"},{"date" => "2015-01-05T05:58:46","version" => "1.40"},{"date" => "2015-01-10T02:47:42","version" => "1.41"},{"date" => "2015-01-12T00:46:36","version" => "1.42"},{"date" => "2015-01-15T06:37:32","version" => "1.43"},{"date" => "2015-01-24T06:12:21","version" => "1.44"},{"date" => "2015-01-27T07:51:17","version" => "1.45"},{"date" => "2015-03-25T05:19:23","version" => "1.46"},{"date" => "2015-06-17T18:26:02","version" => "1.47"},{"date" => "2015-06-18T21:13:37","version" => "1.48"},{"date" => "2015-07-31T19:01:40","version" => "1.49"},{"date" => "2015-08-26T00:11:35","version" => "1.50"},{"date" => "2015-09-22T06:03:54","version" => "1.51"},{"date" => "2015-09-23T17:43:44","version" => "1.53"},{"date" => "2015-12-04T19:36:41","version" => "1.55"},{"date" => "2015-12-17T18:29:06","version" => "1.56"},{"date" => "2016-04-01T18:06:36","version" => "1.57"},{"date" => "2016-08-02T17:50:20","version" => "1.58"},{"date" => "2016-08-11T20:09:16","version" => "1.59"},{"date" => "2017-12-19T18:44:16","version" => "1.60"},{"date" => "2018-08-19T03:35:10","version" => "1.61"},{"date" => "2018-08-20T03:29:01","version" => "1.62"},{"date" => "2018-08-22T15:42:15","version" => "1.63"},{"date" => "2018-09-12T15:50:29","version" => "1.64"},{"date" => "2019-09-08T05:20:03","version" => "1.65"},{"date" => "2019-09-17T04:37:32","version" => "1.66"},{"date" => "2019-10-07T04:30:05","version" => "1.67"},{"date" => "2020-03-12T17:59:46","version" => "1.68"}]},"Authen-DigestMD5" => {"advisories" => [{"affected_versions" => [">=0.01"],"comment" => "The report incorrectly reports 0.02, although this problem is present in 0.04, which is the latest version","cves" => ["CVE-2025-40919"],"description" => "Authen::DigestMD5 versions 0.01 through 0.02 for Perl generate the cnonce insecurely.  The cnonce (client nonce) is generated from an MD5 hash of the PID, the epoch time and the built-in rand function. The PID will come from a small set of numbers, and the epoch time may be guessed, if it is not leaked from the HTTP Date header. The built-in rand function is unsuitable for cryptographic usage.  According to RFC 2831, \"The cnonce-value is an opaque quoted string value provided by the client and used by both client and server to avoid chosen plaintext attacks, and to provide mutual authentication. The security of the implementation depends on a good choice. It is RECOMMENDED that it contain at least 64 bits of entropy.\"","distribution" => "Authen-DigestMD5","fixed_versions" => [],"id" => "CPANSA-Authen-DigestMD5-2025-40919","references" => ["https://datatracker.ietf.org/doc/html/rfc2831","https://metacpan.org/release/SALVA/Authen-DigestMD5-0.01/source/DigestMD5.pm#L126"],"reported" => "2025-07-16","severity" => undef}],"main_module" => "Authen::DigestMD5","versions" => [{"date" => "2003-10-29T00:00:32","version" => "0.01"},{"date" => "2003-10-29T17:18:03","version" => "0.02"},{"date" => "2003-11-08T21:21:26","version" => "0.03"},{"date" => "2003-11-08T22:58:09","version" => "0.04"}]},"Authen-SASL" => {"advisories" => [{"affected_versions" => [">=2.04,<=2.1900"],"cves" => ["CVE-2025-40918"],"description" => "Authen::SASL::Perl::DIGEST_MD5 versions 2.04 through 2.1800 for Perl generates the cnonce insecurely.  The cnonce (client nonce) is generated from an MD5 hash of the PID, the epoch time and the built-in rand function. The PID will come from a small set of numbers, and the epoch time may be guessed, if it is not leaked from the HTTP Date header. The built-in rand function is unsuitable for cryptographic usage.  According to RFC 2831, The cnonce-value is an opaque quoted string value provided by the client and used by both client and server to avoid chosen plaintext attacks, and to provide mutual authentication. The security of the implementation  depends on a good choice. It is RECOMMENDED that it contain at least 64 bits of entropy.","distribution" => "Authen-SASL","fixed_versions" => [">=2.1900"],"id" => "CPANSA-Authen-SASL-2025-40918","references" => ["https://datatracker.ietf.org/doc/html/rfc2831","https://github.com/gbarr/perl-authen-sasl/pull/22","https://metacpan.org/dist/Authen-SASL/source/lib/Authen/SASL/Perl/DIGEST_MD5.pm#L263","https://security.metacpan.org/patches/A/Authen-SASL/2.1800/CVE-2025-40918-r1.patch"],"reported" => "2025-07-16","severity" => undef}],"main_module" => "Authen::SASL","versions" => [{"date" => "2002-01-31T17:03:51","version" => "2.00"},{"date" => "2002-03-31T14:44:21","version" => "2.01"},{"date" => "2002-05-28T14:24:59","version" => "2.02"},{"date" => "2003-01-21T19:16:46","version" => "2.03"},{"date" => "2003-05-19T21:44:39","version" => "2.04"},{"date" => "2003-10-17T21:16:45","version" => "2.05"},{"date" => "2003-11-01T21:26:08","version" => "2.06"},{"date" => "2004-04-10T08:18:07","version" => "2.07"},{"date" => "2004-05-25T10:31:46","version" => "2.08"},{"date" => "2005-04-26T13:37:18","version" => "2.09"},{"date" => "2006-03-25T23:40:21","version" => "2.10"},{"date" => "2008-04-21T15:43:42","version" => "2.11"},{"date" => "2008-07-01T02:59:22","version" => "2.12"},{"date" => "2009-09-24T22:36:34","version" => "2.13"},{"date" => "2010-03-11T15:12:30","version" => "2.14"},{"date" => "2010-03-29T19:28:04","version" => "2.1401"},{"date" => "2010-06-02T18:58:54","version" => "2.15"},{"date" => "2012-09-04T16:12:29","version" => "2.16"},{"date" => "2023-08-09T22:53:31","version" => "2.1700"},{"date" => "2023-08-10T10:19:40","version" => "2.1700"},{"date" => "2025-04-25T16:10:56","version" => "2.1800"},{"date" => "2025-08-05T13:23:40","version" => "2.1900"},{"date" => "2026-01-28T22:01:16","version" => "2.2000"}]},"BSON-XS" => {"advisories" => [{"affected_versions" => ["<=0.8.4"],"cves" => ["CVE-2025-40906","CVE-2017-14227","CVE-2018-16790","CVE-2023-0437","CVE-2024-6381","CVE-2024-6383","CVE-2025-0755"],"description" => "BSON::XS versions 0.8.4 and earlier for Perl includes a bundled libbson 1.1.7, which has several vulnerabilities. Those include CVE-2017-14227, CVE-2018-16790, CVE-2023-0437, CVE-2024-6381, CVE-2024-6383, and CVE-2025-0755. BSON-XS was the official Perl XS implementation of MongoDB's BSON serialization, but this distribution has reached its end of life as of August 13, 2020 and is no longer supported.\n","distribution" => "BSON-XS","fixed_versions" => [],"id" => "CPANSA-BSON-XS-2025-40906","references" => ["https://lists.debian.org/debian-lts-announce/2025/05/msg00012.html","https://www.mongodb.com/community/forums/t/mongodb-perl-driver-end-of-life/7890"],"reported" => "2025-05-16","severity" => "critical"},{"affected_versions" => [">=0.2.0,<=0.8.4"],"cves" => ["CVE-2024-6383"],"description" => "The bson_string_append function in MongoDB C Driver may be vulnerable to a buffer overflow where the function might attempt to allocate too small of buffer and may lead to memory corruption of neighbouring heap memory. This issue affects libbson versions prior to 1.27.1\n","distribution" => "BSON-XS","fixed_versions" => [],"id" => "CPANSA-BSON-XS-2024-6383-libbson","references" => ["https://nvd.nist.gov/vuln/detail/CVE-2024-6383","https://jira.mongodb.org/browse/CDRIVER-5628"],"reported" => "2024-07-03","severity" => "moderate"}],"main_module" => "BSON::XS","versions" => [{"date" => "2016-10-25T01:44:04","version" => "v0.2.0"},{"date" => "2016-10-27T14:29:08","version" => "v0.2.1"},{"date" => "2016-10-27T21:57:22","version" => "v0.2.2"},{"date" => "2018-05-17T20:38:16","version" => "v0.4.0"},{"date" => "2018-05-25T17:23:21","version" => "v0.4.1"},{"date" => "2018-06-13T14:02:45","version" => "v0.4.2"},{"date" => "2018-07-10T13:54:25","version" => "v0.4.3"},{"date" => "2018-09-13T03:31:32","version" => "v0.4.4"},{"date" => "2018-10-12T01:39:57","version" => "v0.4.5"},{"date" => "2018-10-12T15:51:36","version" => "v0.4.6"},{"date" => "2018-11-29T22:12:13","version" => "v0.6.0"},{"date" => "2019-07-12T18:08:23","version" => "v0.8.0"},{"date" => "2019-08-13T12:22:17","version" => "v0.8.1"},{"date" => "2019-12-05T18:59:17","version" => "v0.8.2"},{"date" => "2020-04-13T14:58:34","version" => "v0.8.3"},{"date" => "2020-08-13T14:54:04","version" => "v0.8.4"}]},"BackupPC-XS" => {"advisories" => [{"affected_versions" => [">=0.50,<=0.62"],"cves" => ["CVE-2018-25032"],"description" => "zlib before 1.2.12 allows memory corruption when deflating (i.e., when compressing) if the input has many distant matches.\n","distribution" => "BackupPC-XS","fixed_versions" => [],"id" => "CPANSA-BackupPC-XS-2018-25032-zlib","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143579","https://www.openwall.com/lists/oss-security/2022/03/24/1","https://github.com/madler/zlib/commit/5c44459c3b28a9bd3283aaceab7c615f8020c531","http://www.openwall.com/lists/oss-security/2022/03/25/2","http://www.openwall.com/lists/oss-security/2022/03/26/1","https://www.openwall.com/lists/oss-security/2022/03/28/1","https://github.com/madler/zlib/compare/v1.2.11...v1.2.12","https://www.openwall.com/lists/oss-security/2022/03/28/3","https://github.com/madler/zlib/issues/605","https://www.debian.org/security/2022/dsa-5111","https://lists.debian.org/debian-lts-announce/2022/04/msg00000.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/NS2D2GFPFGOJUL4WQ3DUAY7HF4VWQ77F/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/XOKFMSNQ5D5WGMALBNBXU3GE442V74WU/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/VOKNP2L734AEL47NRYGVZIKEFOUBQY5Y/","https://lists.debian.org/debian-lts-announce/2022/05/msg00008.html","https://support.apple.com/kb/HT213255","https://support.apple.com/kb/HT213256","https://support.apple.com/kb/HT213257","http://seclists.org/fulldisclosure/2022/May/33","http://seclists.org/fulldisclosure/2022/May/35","http://seclists.org/fulldisclosure/2022/May/38","https://security.netapp.com/advisory/ntap-20220526-0009/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DF62MVMH3QUGMBDCB3DY2ERQ6EBHTADB/"],"reported" => "2022-03-25","severity" => "high"}],"main_module" => "BackupPC::XS","versions" => [{"date" => "2017-03-04T04:41:25","version" => "0.50"},{"date" => "2017-03-05T21:24:18","version" => "0.51"},{"date" => "2017-03-12T00:59:11","version" => "0.52"},{"date" => "2017-03-24T02:35:47","version" => "0.53"},{"date" => "2017-05-30T03:39:23","version" => "0.54"},{"date" => "2017-06-04T17:33:27","version" => "0.55"},{"date" => "2017-06-11T17:54:50","version" => "0.56"},{"date" => "2017-12-03T20:14:07","version" => "0.57"},{"date" => "2018-11-25T20:28:38","version" => "0.58"},{"date" => "2019-04-08T00:18:42","version" => "0.59"},{"date" => "2020-05-30T18:21:55","version" => "0.60"},{"date" => "2020-06-09T02:06:28","version" => "0.61"},{"date" => "2020-06-20T18:25:12","version" => "0.62"}]},"Batch-Batchrun" => {"advisories" => [{"affected_versions" => [">=1.03"],"cves" => ["CVE-2011-4117"],"description" => "The Batch::Batchrun module 1.03 for Perl does not properly handle temporary files.\n","distribution" => "Batch-Batchrun","fixed_versions" => [],"id" => "CPANSA-Batch-Batchrun-2011-4117","references" => ["http://www.openwall.com/lists/oss-security/2011/11/04/2","http://www.openwall.com/lists/oss-security/2011/11/04/4","https://rt.cpan.org/Public/Bug/Display.html?id=69594"],"reported" => "2020-01-31","severity" => "high"}],"main_module" => "Batch::Batchrun","versions" => [{"date" => "1999-08-21T20:25:47","version" => "1.03"}]},"Boost-Graph" => {"advisories" => [{"affected_versions" => [">=1,1,<=1.4"],"cves" => ["CVE-2008-0171"],"description" => "regex/v4/perl_matcher_non_recursive.hpp in the Boost regex library (aka Boost.Regex) in Boost 1.33 and 1.34 allows context-dependent attackers to cause a denial of service (failed assertion and crash) via an invalid regular expression.\n","distribution" => "Boost-Graph","fixed_versions" => [],"id" => "CPANSA-Boost-Graph-2008-0171-boost","references" => ["http://bugs.gentoo.org/show_bug.cgi?id=205955","http://svn.boost.org/trac/boost/changeset/42674","http://svn.boost.org/trac/boost/changeset/42745","https://issues.rpath.com/browse/RPL-2143","http://www.ubuntu.com/usn/usn-570-1","http://www.securityfocus.com/bid/27325","https://www.redhat.com/archives/fedora-package-announce/2008-January/msg00760.html","http://secunia.com/advisories/28545","http://www.mandriva.com/en/security/advisories?name=MDVSA-2008:032","http://secunia.com/advisories/28705","http://secunia.com/advisories/28511","http://secunia.com/advisories/28527","http://wiki.rpath.com/Advisories:rPSA-2008-0063","http://www.gentoo.org/security/en/glsa/glsa-200802-08.xml","http://secunia.com/advisories/28943","http://secunia.com/advisories/28860","http://lists.opensuse.org/opensuse-security-announce/2008-03/msg00004.html","http://secunia.com/advisories/29323","http://www.vupen.com/english/advisories/2008/0249","http://secunia.com/advisories/48099","http://www.securityfocus.com/archive/1/488102/100/0/threaded"],"reported" => "2008-01-17","severity" => undef},{"affected_versions" => [">=1,1,<=1.4"],"cves" => ["CVE-2008-0172"],"description" => "The get_repeat_type function in basic_regex_creator.hpp in the Boost regex library (aka Boost.Regex) in Boost 1.33 and 1.34 allows context-dependent attackers to cause a denial of service (NULL dereference and crash) via an invalid regular expression.\n","distribution" => "Boost-Graph","fixed_versions" => [],"id" => "CPANSA-Boost-Graph-2008-0172-boost","references" => ["http://bugs.gentoo.org/show_bug.cgi?id=205955","http://svn.boost.org/trac/boost/changeset/42674","http://svn.boost.org/trac/boost/changeset/42745","https://issues.rpath.com/browse/RPL-2143","http://www.ubuntu.com/usn/usn-570-1","http://www.securityfocus.com/bid/27325","https://www.redhat.com/archives/fedora-package-announce/2008-January/msg00760.html","http://secunia.com/advisories/28545","http://www.mandriva.com/en/security/advisories?name=MDVSA-2008:032","http://secunia.com/advisories/28705","http://secunia.com/advisories/28511","http://secunia.com/advisories/28527","http://wiki.rpath.com/Advisories:rPSA-2008-0063","http://www.gentoo.org/security/en/glsa/glsa-200802-08.xml","http://secunia.com/advisories/28943","http://secunia.com/advisories/28860","http://lists.opensuse.org/opensuse-security-announce/2008-03/msg00004.html","http://secunia.com/advisories/29323","http://www.vupen.com/english/advisories/2008/0249","http://secunia.com/advisories/48099","http://www.securityfocus.com/archive/1/488102/100/0/threaded"],"reported" => "2008-01-17","severity" => undef}],"main_module" => "Boost::Graph","versions" => [{"date" => "2006-05-15T23:21:14","version" => "1.1"},{"date" => "2006-06-19T20:32:45","version" => "1.2"},{"date" => "2006-11-13T18:58:01","version" => "1.2"},{"date" => "2007-07-11T18:07:27","version" => "1.4"},{"date" => "2013-11-05T09:24:41","version" => "1.4_001"}]},"CBOR-XS" => {"advisories" => [{"affected_versions" => ["<1.7"],"cves" => [],"description" => "An out-of bound sharedref or stringref index could cause an out of bounds access - might be exploitable. A decoding error during indefinite array or hash decoding could cause an endless loop.\n","distribution" => "CBOR-XS","fixed_versions" => [">=1.7"],"id" => "CPANSA-CBOR-XS-2017-01","references" => ["https://metacpan.org/dist/CBOR-XS/changes"],"reported" => "2017-07-27","severity" => undef}],"main_module" => "CBOR::XS","versions" => [{"date" => "2013-10-25T23:10:42","version" => "0.01"},{"date" => "2013-10-26T11:09:56","version" => "0.02"},{"date" => "2013-10-26T23:04:01","version" => "0.03"},{"date" => "2013-10-27T22:48:22","version" => "0.04"},{"date" => "2013-10-28T21:28:30","version" => "0.05"},{"date" => "2013-10-29T15:57:13","version" => "0.06"},{"date" => "2013-10-29T22:05:30","version" => "0.07"},{"date" => "2013-10-30T10:11:46","version" => "0.08"},{"date" => "2013-11-22T16:19:26","version" => "0.09"},{"date" => "2013-11-28T16:09:19","version" => "1.0"},{"date" => "2013-11-30T18:42:59","version" => "1.1"},{"date" => "2013-12-01T17:11:47","version" => "1.11"},{"date" => "2013-12-03T10:25:03","version" => "1.12"},{"date" => "2013-12-10T21:07:58","version" => "1.2"},{"date" => "2014-01-05T14:25:36","version" => "1.25"},{"date" => "2014-10-25T06:37:38","version" => "1.26"},{"date" => "2015-04-27T20:22:15","version" => "1.3"},{"date" => "2016-02-08T04:38:25","version" => "1.4"},{"date" => "2016-02-25T14:23:47","version" => "1.41"},{"date" => "2016-04-27T09:40:38","version" => "1.5"},{"date" => "2016-12-07T14:14:49","version" => "1.6"},{"date" => "2017-06-27T02:03:48","version" => "1.7"},{"date" => "2018-11-15T19:53:50","version" => "1.71"},{"date" => "2020-11-29T21:36:13","version" => "1.8"},{"date" => "2020-11-30T18:31:32","version" => "1.81"},{"date" => "2020-12-01T01:50:49","version" => "1.82"},{"date" => "2020-12-08T08:30:59","version" => "1.83"},{"date" => "2021-10-21T01:16:11","version" => "1.84"},{"date" => "2021-10-23T03:00:48","version" => "1.85"},{"date" => "2021-11-04T16:50:24","version" => "1.86"},{"date" => "2023-09-10T20:45:43","version" => "1.87"}]},"CGI" => {"advisories" => [{"affected_versions" => ["<3.63"],"cves" => ["CVE-2012-5526"],"description" => "CGI.pm module before 3.63 for Perl does not properly escape newlines in (1) Set-Cookie or (2) P3P headers, which might allow remote attackers to inject arbitrary headers into responses from applications that use CGI.pm.\n","distribution" => "CGI","fixed_versions" => [">=3.63"],"id" => "CPANSA-CGI-2012-5526","references" => ["http://www.securityfocus.com/bid/56562","http://www.openwall.com/lists/oss-security/2012/11/15/6","https://github.com/markstos/CGI.pm/pull/23","http://www.securitytracker.com/id?1027780","http://cpansearch.perl.org/src/MARKSTOS/CGI.pm-3.63/Changes","http://secunia.com/advisories/51457","http://www.ubuntu.com/usn/USN-1643-1","http://www.debian.org/security/2012/dsa-2586","http://rhn.redhat.com/errata/RHSA-2013-0685.html","http://secunia.com/advisories/55314","http://www.oracle.com/technetwork/topics/security/ovmbulletinjul2016-3090546.html","http://kb.juniper.net/InfoCenter/index?page=content&id=JSA10735","http://kb.juniper.net/InfoCenter/index?page=content&id=JSA10705","https://exchange.xforce.ibmcloud.com/vulnerabilities/80098"],"reported" => "2012-11-21","severity" => undef},{"affected_versions" => ["<3.56"],"cves" => ["CVE-2011-2766"],"description" => "Usage of deprecated FCGI.pm API.\n","distribution" => "CGI","fixed_versions" => [">=3.56"],"id" => "CPANSA-CGI-2011-2766","references" => ["https://rt.cpan.org/Public/Bug/Display.html?id=68380","http://web.nvd.nist.gov/view/vuln/detail?vulnId=CVE-2011-2766"],"reported" => "2011-11-08"},{"affected_versions" => ["<3.50"],"cves" => [],"description" => "Non-random MIME boundary.\n","distribution" => "CGI","fixed_versions" => [">=3.50"],"id" => "CPANSA-CGI-2010-02","reported" => "2010-11-08"},{"affected_versions" => ["<3.49"],"cves" => [],"description" => "Newlines in headers.\n","distribution" => "CGI","fixed_versions" => [">=3.49"],"id" => "CPANSA-CGI-2010-01","reported" => "2010-02-05"},{"affected_versions" => ["<3.50"],"cves" => ["CVE-2010-4411"],"description" => "Unspecified vulnerability in CGI.pm 3.50 and earlier allows remote attackers to inject arbitrary HTTP headers and conduct HTTP response splitting attacks via unknown vectors.  NOTE: this issue exists because of an incomplete fix for CVE-2010-2761.\n","distribution" => "CGI","fixed_versions" => [">=3.50"],"id" => "CPANSA-CGI-2010-4411","references" => ["http://openwall.com/lists/oss-security/2010/12/01/3","http://www.mandriva.com/security/advisories?name=MDVSA-2011:008","http://www.vupen.com/english/advisories/2011/0106","http://www.bugzilla.org/security/3.2.9/","http://secunia.com/advisories/43033","https://bugzilla.mozilla.org/show_bug.cgi?id=591165","http://www.vupen.com/english/advisories/2011/0207","http://www.vupen.com/english/advisories/2011/0271","http://www.vupen.com/english/advisories/2011/0212","http://lists.fedoraproject.org/pipermail/package-announce/2011-February/053665.html","http://secunia.com/advisories/43068","http://lists.opensuse.org/opensuse-security-announce/2011-01/msg00006.html","http://secunia.com/advisories/43165","http://lists.fedoraproject.org/pipermail/package-announce/2011-February/053678.html","http://lists.opensuse.org/opensuse-security-announce/2011-04/msg00000.html"],"reported" => "2010-12-06","severity" => undef},{"affected_versions" => ["<3.50"],"cves" => ["CVE-2010-2761"],"description" => "The multipart_init function in (1) CGI.pm before 3.50 and (2) Simple.pm in CGI::Simple 1.112 and earlier uses a hardcoded value of the MIME boundary string in multipart/x-mixed-replace content, which allows remote attackers to inject arbitrary HTTP headers and conduct HTTP response splitting attacks via crafted input that contains this value, a different vulnerability than CVE-2010-3172.\n","distribution" => "CGI","fixed_versions" => [">=3.50"],"id" => "CPANSA-CGI-2010-2761","references" => ["https://github.com/AndyA/CGI--Simple/commit/e4942b871a26c1317a175a91ebb7262eea59b380","http://cpansearch.perl.org/src/LDS/CGI.pm-3.50/Changes","http://openwall.com/lists/oss-security/2010/12/01/1","http://www.nntp.perl.org/group/perl.perl5.changes/2010/11/msg28043.html","http://perl5.git.perl.org/perl.git/blobdiff/a0b94c2432b1d8c20653453a0f6970cb10f59aec..84601d63a7e34958da47dad1e61e27cb3bd467d1:/cpan/CGI/lib/CGI.pm","http://perl5.git.perl.org/perl.git/commit/84601d63a7e34958da47dad1e61e27cb3bd467d1","http://openwall.com/lists/oss-security/2010/12/01/2","http://openwall.com/lists/oss-security/2010/12/01/3","https://bugzilla.mozilla.org/show_bug.cgi?id=600464","http://osvdb.org/69588","http://osvdb.org/69589","http://lists.opensuse.org/opensuse-security-announce/2011-01/msg00003.html","http://www.mandriva.com/security/advisories?name=MDVSA-2010:237","http://www.vupen.com/english/advisories/2011/0076","http://www.mandriva.com/security/advisories?name=MDVSA-2010:250","http://secunia.com/advisories/42877","https://bugzilla.mozilla.org/show_bug.cgi?id=591165","http://www.vupen.com/english/advisories/2011/0207","http://www.bugzilla.org/security/3.2.9/","http://secunia.com/advisories/43033","http://lists.fedoraproject.org/pipermail/package-announce/2011-January/053591.html","http://secunia.com/advisories/43147","http://lists.fedoraproject.org/pipermail/package-announce/2011-January/053576.html","http://www.vupen.com/english/advisories/2011/0249","http://www.vupen.com/english/advisories/2011/0271","http://lists.fedoraproject.org/pipermail/package-announce/2011-February/053678.html","http://www.vupen.com/english/advisories/2011/0212","http://lists.opensuse.org/opensuse-security-announce/2011-01/msg00006.html","http://secunia.com/advisories/43165","http://secunia.com/advisories/43068","http://lists.fedoraproject.org/pipermail/package-announce/2011-February/053665.html","http://www.redhat.com/support/errata/RHSA-2011-1797.html","http://lists.opensuse.org/opensuse-security-announce/2011-04/msg00000.html","http://kb.juniper.net/InfoCenter/index?page=content&id=JSA10735","http://kb.juniper.net/InfoCenter/index?page=content&id=JSA10705"],"reported" => "2010-12-06","severity" => undef}],"main_module" => "CGI","versions" => [{"date" => "1995-11-25T09:21:00","version" => "2.10"},{"date" => "1995-12-28T09:08:00","version" => "2.13"},{"date" => "1996-05-22T22:30:00","version" => "2.20"},{"date" => "1996-05-31T05:31:00","version" => "2.21"},{"date" => "1996-08-07T09:24:00","version" => "2.22"},{"date" => "1996-08-14T08:17:00","version" => "2.23"},{"date" => "1996-08-21T09:09:00","version" => "2.24"},{"date" => "1996-09-10T14:23:00","version" => "2.25"},{"date" => "1996-10-22T11:17:00","version" => "2.26"},{"date" => "1996-10-24T18:21:00","version" => "2.27"},{"date" => "1996-12-02T11:48:00","version" => "2.28"},{"date" => "1996-12-09T13:39:00","version" => "2.29"},{"date" => "1997-01-02T16:40:00","version" => "2.30"},{"date" => "1997-02-15T15:36:00","version" => "2.31"},{"date" => "1997-03-25T08:58:00","version" => "2.32"},{"date" => "1997-04-04T20:45:00","version" => "2.33"},{"date" => "1997-04-10T15:41:00","version" => "2.34"},{"date" => "1997-04-20T18:29:00","version" => "2.35"},{"date" => "1997-05-09T09:33:00","version" => "2.36"},{"date" => "1997-08-29T04:42:00","version" => "2.37"},{"date" => "1997-09-15T17:51:00","version" => "2.37"},{"date" => "1997-10-01T04:15:00","version" => "2.37"},{"date" => "1997-10-12T07:10:00","version" => "2.37"},{"date" => "1997-11-23T11:37:00","version" => "2.37"},{"date" => "1997-12-20T09:57:00","version" => "2.37"},{"date" => "1998-01-16T12:22:00","version" => "2.37"},{"date" => "1998-01-19T11:34:00","version" => "2.37"},{"date" => "1998-01-26T11:00:00","version" => "2.37"},{"date" => "1998-01-29T19:48:00","version" => "2.37"},{"date" => "1998-02-02T13:37:00","version" => "2.37"},{"date" => "1998-02-05T08:25:00","version" => "2.37"},{"date" => "1998-02-09T13:58:00","version" => "2.37"},{"date" => "1998-02-16T16:21:00","version" => "2.37"},{"date" => "1998-02-23T08:33:00","version" => "2.37"},{"date" => "1998-02-24T16:52:00","version" => "2.37"},{"date" => "1998-03-13T15:33:00","version" => "2.37"},{"date" => "1998-03-22T21:12:00","version" => "2.38"},{"date" => "1998-03-24T22:37:08","version" => "2.39"},{"date" => "1998-05-20T09:17:00","version" => "2.40"},{"date" => "1998-05-28T15:03:00","version" => "2.41"},{"date" => "1998-06-09T09:00:00","version" => "2.42"},{"date" => "1998-10-14T19:06:31","version" => "2.43"},{"date" => "1998-11-24T19:41:41","version" => "2.44"},{"date" => "1998-11-26T11:07:27","version" => "2.45"},{"date" => "1998-12-06T10:22:46","version" => "2.46"},{"date" => "1999-02-18T03:50:16","version" => "2.47"},{"date" => "1999-02-19T14:10:25","version" => "2.48"},{"date" => "1999-02-23T22:00:33","version" => "2.49"},{"date" => "1999-06-08T15:13:15","version" => "2.52"},{"date" => "1999-06-09T14:56:06","version" => "2.53"},{"date" => "1999-08-09T14:18:33","version" => "2.54"},{"date" => "1999-08-31T17:11:00","version" => "2.55"},{"date" => "1999-09-13T21:11:51","version" => "2.56"},{"date" => "2000-03-23T23:00:12","version" => "2.58"},{"date" => "2000-03-24T12:31:52","version" => "2.59"},{"date" => "2000-03-27T22:11:34","version" => "2.60"},{"date" => "2000-03-28T02:50:18","version" => "2.61"},{"date" => "2000-03-28T21:38:03","version" => "2.62"},{"date" => "2000-04-10T15:19:54","version" => "2.63"},{"date" => "2000-04-11T15:25:13","version" => "2.64"},{"date" => "2000-04-11T15:55:40","version" => "2.65"},{"date" => "2000-04-12T20:16:46","version" => "2.66"},{"date" => "2000-05-16T01:38:08","version" => "2.67"},{"date" => "2000-05-18T17:55:55","version" => "2.68"},{"date" => "2000-07-28T03:06:11","version" => "2.69"},{"date" => "2000-08-04T19:37:27","version" => "2.70"},{"date" => "2000-08-13T16:09:25","version" => "2.71"},{"date" => "2000-08-20T17:35:50","version" => "2.72"},{"date" => "2000-08-24T13:33:37","version" => "3."},{"date" => "2000-09-13T02:55:51","version" => "2.73"},{"date" => "2000-09-13T16:35:14","version" => "2.74"},{"date" => "2001-02-02T15:43:07","version" => "2.75"},{"date" => "2001-02-02T15:50:53","version" => "2.751"},{"date" => "2001-02-04T23:49:27","version" => "2.752"},{"date" => "2001-03-12T17:00:13","version" => "2.753"},{"date" => "2001-06-15T15:33:28","version" => "3.02"},{"date" => "2001-06-29T14:47:39","version" => "3.02_"},{"date" => "2001-07-05T16:13:55","version" => "3.03_01"},{"date" => "2001-07-26T21:29:22","version" => "2.76"},{"date" => "2001-08-07T12:33:22","version" => "2.77"},{"date" => "2001-09-26T02:26:36","version" => "2.78"},{"date" => "2001-12-09T21:39:11","version" => "2.79"},{"date" => "2002-01-12T02:47:17","version" => "2.80"},{"date" => "2002-04-10T19:39:49","version" => "2.81"},{"date" => "2002-09-11T12:27:48","version" => "2.84"},{"date" => "2002-09-11T14:01:02","version" => "2.85"},{"date" => "2002-09-12T03:58:40","version" => "2.86"},{"date" => "2002-10-07T02:00:58","version" => "2.87"},{"date" => "2002-10-14T13:58:09","version" => "2.88"},{"date" => "2002-10-16T17:50:26","version" => "2.89"},{"date" => "2002-11-22T23:03:39","version" => 0},{"date" => "2003-02-10T20:11:57","version" => "2.90"},{"date" => "2003-02-11T14:15:15","version" => "2.91"},{"date" => "2003-04-28T00:44:10","version" => "2.92"},{"date" => "2003-04-28T13:37:43","version" => "2.93"},{"date" => "2003-06-09T12:15:29","version" => "2.94"},{"date" => "2003-06-13T02:35:42","version" => "2.95"},{"date" => "2003-06-16T18:42:38","version" => "2.96"},{"date" => "2003-06-17T23:32:52","version" => "2.97"},{"date" => "2003-07-16T17:06:29","version" => "2.98"},{"date" => "2003-08-01T14:43:54","version" => "2.99"},{"date" => "2003-08-18T17:51:48","version" => "3.00"},{"date" => "2003-12-10T17:05:47","version" => "3.01"},{"date" => "2004-01-13T16:34:47","version" => "3.03"},{"date" => "2004-01-19T12:44:30","version" => "3.04"},{"date" => "2004-04-12T20:39:57","version" => "3.05"},{"date" => "2005-03-09T21:06:46","version" => "3.06"},{"date" => "2005-03-14T16:34:03","version" => "3.07"},{"date" => "2005-04-20T15:31:11","version" => "3.08"},{"date" => "2005-05-05T20:16:55","version" => "3.09"},{"date" => "2005-05-13T21:48:46","version" => "3.10"},{"date" => "2005-08-03T21:17:14","version" => "3.11"},{"date" => "2005-12-04T16:46:53","version" => "3.12"},{"date" => "2005-12-05T13:54:26","version" => "3.13"},{"date" => "2005-12-06T22:14:19","version" => "3.14"},{"date" => "2005-12-07T20:16:49","version" => "3.15"},{"date" => "2006-02-08T18:50:56","version" => "3.16"},{"date" => "2006-02-24T19:04:58","version" => "3.17"},{"date" => "2006-04-17T13:56:06","version" => "3.19"},{"date" => "2006-04-23T14:27:55","version" => "3.20"},{"date" => "2006-08-21T19:12:36","version" => "3.21"},{"date" => "2006-08-23T15:24:41","version" => "3.22"},{"date" => "2006-08-24T11:53:26","version" => "3.23"},{"date" => "2006-09-28T17:09:45","version" => "3.25"},{"date" => "2007-02-27T15:42:54","version" => "3.27"},{"date" => "2007-03-29T15:38:01","version" => "3.28"},{"date" => "2007-04-16T17:00:18","version" => "3.29"},{"date" => "2007-11-30T19:06:19","version" => "3.31"},{"date" => "2007-12-27T18:41:32","version" => "3.32"},{"date" => "2008-01-03T15:03:17","version" => "3.33"},{"date" => "2008-03-18T16:04:41","version" => "3.34"},{"date" => "2008-03-27T14:26:48","version" => "3.35"},{"date" => "2008-04-23T13:09:44","version" => "3.37"},{"date" => "2008-06-25T14:58:32","version" => "3.38"},{"date" => "2008-07-29T15:01:52","version" => "3.39"},{"date" => "2008-08-06T18:21:51","version" => "3.40"},{"date" => "2008-08-26T13:56:27","version" => "3.41"},{"date" => "2008-09-08T14:15:41","version" => "3.42"},{"date" => "2009-04-06T18:35:19","version" => "3.43"},{"date" => "2009-07-30T16:34:17","version" => "3.44"},{"date" => "2009-08-14T13:37:12","version" => "3.45"},{"date" => "2009-09-09T15:39:42","version" => "3.46"},{"date" => "2009-09-09T20:03:01","version" => "3.47"},{"date" => "2009-09-25T15:07:03","version" => "3.48"},{"date" => "2010-02-05T16:24:53","version" => "3.49"},{"date" => "2010-11-08T21:53:26","version" => "3.50"},{"date" => "2011-01-05T18:28:41","version" => "3.51"},{"date" => "2011-01-25T04:30:05","version" => "3.52"},{"date" => "2011-04-25T23:01:21","version" => "3.53"},{"date" => "2011-04-28T14:36:41","version" => "3.54"},{"date" => "2011-06-03T15:39:16","version" => "3.55"},{"date" => "2011-11-09T02:00:20","version" => "3.56"},{"date" => "2011-11-09T15:59:18","version" => "3.57"},{"date" => "2011-11-12T03:36:07","version" => "3.58"},{"date" => "2011-12-30T13:35:35","version" => "3.59"},{"date" => "2012-08-16T03:21:13","version" => "3.60"},{"date" => "2012-11-03T02:10:42","version" => "3.61"},{"date" => "2012-11-10T01:40:50","version" => "3.62"},{"date" => "2012-11-14T23:45:29","version" => "3.63"},{"date" => "2013-11-24T01:22:00","version" => "3.64"},{"date" => "2014-02-12T03:13:58","version" => "3.65"},{"date" => "2014-05-15T12:59:58","version" => "3.65_01"},{"date" => "2014-05-16T11:43:33","version" => "3.65_02"},{"date" => "2014-05-20T12:31:46","version" => "3.65_03"},{"date" => "2014-05-22T19:58:14","version" => "4.00"},{"date" => "2014-05-27T13:13:51","version" => "4.01"},{"date" => "2014-06-09T13:55:49","version" => "4.02"},{"date" => "2014-07-02T14:53:06","version" => "4.03"},{"date" => "2014-07-28T18:30:34","version" => "4.03_01"},{"date" => "2014-07-30T14:26:40","version" => "4.03_02"},{"date" => "2014-08-13T11:40:14","version" => "4.03_03"},{"date" => "2014-09-04T14:42:14","version" => "4.04"},{"date" => "2014-09-20T16:08:55","version" => "4.04_01"},{"date" => "2014-09-28T19:57:05","version" => "4.04_02"},{"date" => "2014-09-29T09:50:07","version" => "4.04_03"},{"date" => "2014-10-06T12:01:14","version" => "4.04_04"},{"date" => "2014-10-06T12:24:10","version" => "4.04_05"},{"date" => "2014-10-08T07:42:49","version" => "4.05"},{"date" => "2014-10-10T11:35:49","version" => "4.06"},{"date" => "2014-10-12T16:29:35","version" => "4.07"},{"date" => "2014-10-18T11:00:38","version" => "4.08"},{"date" => "2014-10-21T07:33:36","version" => "4.09"},{"date" => "2014-11-25T21:06:50","version" => "4.09_01"},{"date" => "2014-11-27T12:53:51","version" => "4.10"},{"date" => "2014-11-30T12:12:26","version" => "4.10_01"},{"date" => "2014-12-03T07:25:15","version" => "4.11"},{"date" => "2014-12-18T08:35:52","version" => "4.12"},{"date" => "2014-12-18T09:21:52","version" => "4.13"},{"date" => "2015-02-12T14:19:13","version" => "4.13_01"},{"date" => "2015-02-13T08:01:29","version" => "4.13_02"},{"date" => "2015-03-01T13:28:25","version" => "4.13_03"},{"date" => "2015-03-08T16:09:21","version" => "4.13_04"},{"date" => "2015-03-25T17:55:15","version" => "4.13_05"},{"date" => "2015-04-01T06:51:57","version" => "4.14"},{"date" => "2015-04-17T14:27:39","version" => "4.14_01"},{"date" => "2015-04-20T07:15:45","version" => "4.15"},{"date" => "2015-05-29T14:48:42","version" => "4.20"},{"date" => "2015-06-22T07:50:02","version" => "4.21"},{"date" => "2015-10-16T09:46:31","version" => "4.22"},{"date" => "2015-12-20T18:33:35","version" => "4.24"},{"date" => "2015-12-21T09:29:19","version" => "4.25"},{"date" => "2016-02-04T16:37:12","version" => "4.26"},{"date" => "2016-03-02T08:03:46","version" => "4.27"},{"date" => "2016-03-14T07:21:48","version" => "4.28"},{"date" => "2016-05-22T12:23:19","version" => "4.28_01"},{"date" => "2016-05-22T12:54:23","version" => "4.28_02"},{"date" => "2016-05-23T08:25:25","version" => "4.28_03"},{"date" => "2016-06-09T12:01:20","version" => "4.29"},{"date" => "2016-06-09T12:11:54","version" => "4.30"},{"date" => "2016-06-14T07:14:00","version" => "4.31"},{"date" => "2016-07-19T07:05:46","version" => "4.32"},{"date" => "2016-09-16T09:47:49","version" => "4.33"},{"date" => "2016-10-13T11:58:55","version" => "4.34"},{"date" => "2016-10-13T13:56:21","version" => "4.35"},{"date" => "2017-03-29T08:56:26","version" => "4.35_01"},{"date" => "2017-04-06T14:42:12","version" => "4.36"},{"date" => "2017-11-01T10:17:40","version" => "4.37"},{"date" => "2017-12-01T08:41:02","version" => "4.38"},{"date" => "2018-08-13T15:57:52","version" => "4.39"},{"date" => "2018-08-15T08:39:39","version" => "4.40"},{"date" => "2019-03-26T15:58:49","version" => "4.41"},{"date" => "2019-03-26T16:33:27","version" => "4.42"},{"date" => "2019-05-01T14:28:45","version" => "4.43"},{"date" => "2019-06-03T09:00:55","version" => "4.44"},{"date" => "2020-01-13T07:03:55","version" => "4.45"},{"date" => "2020-02-03T14:49:22","version" => "4.46"},{"date" => "2020-05-01T13:01:44","version" => "4.47"},{"date" => "2020-06-02T08:22:41","version" => "4.48"},{"date" => "2020-06-08T09:46:25","version" => "4.49"},{"date" => "2020-06-22T07:35:25","version" => "4.50"},{"date" => "2020-10-05T06:14:39","version" => "4.51"},{"date" => "2021-05-04T08:02:27","version" => "4.52"},{"date" => "2021-06-03T06:45:55","version" => "4.53"},{"date" => "2022-02-03T07:52:34","version" => "4.54"},{"date" => "2023-01-03T07:45:53","version" => "4.55"},{"date" => "2023-03-03T08:51:51","version" => "4.56"},{"date" => "2023-05-02T13:16:01","version" => "4.57"},{"date" => "2023-10-02T07:08:45","version" => "4.58"},{"date" => "2023-10-02T07:14:30","version" => "4.59"},{"date" => "2023-11-01T07:57:12","version" => "4.60"},{"date" => "2024-01-08T15:17:04","version" => "4.61"},{"date" => "2024-03-01T13:46:49","version" => "4.62"},{"date" => "2024-03-06T15:20:30","version" => "4.63"},{"date" => "2024-03-18T12:10:48","version" => "4.64"},{"date" => "2024-06-04T15:15:17","version" => "4.65"},{"date" => "2024-06-19T08:59:52","version" => "4.66"},{"date" => "2025-01-08T15:27:45","version" => "4.67"},{"date" => "2025-04-01T09:38:18","version" => "4.68"},{"date" => "2025-06-11T06:21:57","version" => "4.69"},{"date" => "2025-07-07T11:59:39","version" => "4.70"},{"date" => "2025-10-01T08:09:27","version" => "4.71"},{"date" => "2006-01-28T00:00:00","dual_lived" => 1,"perl_release" => "5.009003","version" => "3.15_01"}]},"CGI-Application" => {"advisories" => [{"affected_versions" => ["<4.50_51"],"cves" => ["CVE-2013-7329"],"description" => "The CGI::Application module before 4.50_50 and 4.50_51 for Perl, when run modes are not specified, allows remote attackers to obtain sensitive information (web queries and environment details) via vectors related to the dump_html function.\n","distribution" => "CGI-Application","fixed_versions" => [],"id" => "CPANSA-CGI-Application-2013-7329","references" => ["https://github.com/markstos/CGI--Application/pull/15","http://openwall.com/lists/oss-security/2014/02/19/11","http://lists.fedoraproject.org/pipermail/package-announce/2014-March/129436.html","http://lists.fedoraproject.org/pipermail/package-announce/2014-March/129444.html","https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=739505","http://www.securityfocus.com/bid/65687","https://rt.cpan.org/Public/Bug/Display.html?id=84403","https://bugzilla.redhat.com/show_bug.cgi?id=1067180","https://exchange.xforce.ibmcloud.com/vulnerabilities/91735"],"reported" => "2014-10-06","severity" => undef}],"main_module" => "CGI::Application","versions" => [{"date" => "2000-07-11T04:23:51","version" => "1.0"},{"date" => "2000-07-12T15:21:41","version" => "1.1"},{"date" => "2000-07-18T21:11:44","version" => "1.2"},{"date" => "2001-05-21T12:03:59","version" => "1.3"},{"date" => "2001-05-28T18:29:06","version" => "1.31"},{"date" => "2001-06-25T03:17:50","version" => "2.0"},{"date" => "2001-08-11T22:18:28","version" => "2.1"},{"date" => "2002-05-06T03:21:57","version" => "2.2"},{"date" => "2002-05-06T11:57:30","version" => "2.3"},{"date" => "2002-05-27T01:01:18","version" => "2.4"},{"date" => "2002-07-18T11:59:16","version" => "2.5"},{"date" => "2002-10-07T13:03:27","version" => "2.6"},{"date" => "2003-02-01T13:52:45","version" => "3.0"},{"date" => "2003-06-02T13:01:50","version" => "3.1"},{"date" => "2004-02-04T03:23:56","version" => "3.2"},{"date" => "2004-02-04T15:53:56","version" => "3.2"},{"date" => "2004-02-14T01:47:53","version" => "3.22"},{"date" => "2004-09-26T19:22:20","version" => "3.30"},{"date" => "2004-09-26T19:35:26","version" => "3.31"},{"date" => "2005-03-19T14:42:14","version" => "4.0_2"},{"date" => "2005-06-07T03:25:55","version" => "4.0_4"},{"date" => "2005-06-11T04:00:57","version" => "4.0"},{"date" => "2005-06-13T19:15:12","version" => "4.01_01"},{"date" => "2005-06-14T14:37:30","version" => "4.01"},{"date" => "2005-07-24T19:08:18","version" => "4.02_1"},{"date" => "2005-07-31T03:11:25","version" => "4.02"},{"date" => "2005-08-04T23:45:52","version" => "4.03"},{"date" => "2005-09-01T02:54:00","version" => "4.04_01"},{"date" => "2005-09-09T01:12:21","version" => "4.04_02"},{"date" => "2005-10-12T02:12:18","version" => "4.04"},{"date" => "2006-03-02T01:58:41","version" => "4.05"},{"date" => "2006-04-13T02:34:40","version" => "4.06"},{"date" => "2006-07-02T05:05:34","version" => "4.07_01"},{"date" => "2007-10-31T23:34:31","version" => "4.07_02"},{"date" => "2008-06-16T20:09:18","version" => "4.07_03"},{"date" => "2008-06-18T03:30:33","version" => "4.10"},{"date" => "2008-08-10T15:36:00","version" => "4.11"},{"date" => "2008-09-27T04:18:05","version" => "4.11"},{"date" => "2008-11-02T00:43:18","version" => "4.11"},{"date" => "2009-01-03T16:13:59","version" => "4.21"},{"date" => "2009-07-30T01:35:48","version" => "4.30"},{"date" => "2009-07-30T01:42:34","version" => "4.31"},{"date" => "2010-02-14T00:33:00","version" => "4.32_1"},{"date" => "2011-06-16T17:07:16","version" => "4.50"},{"date" => "2015-01-23T12:19:21","version" => "4.50_50"},{"date" => "2015-07-18T12:57:01","version" => "4.50_51"},{"date" => "2018-02-28T13:57:05","version" => "4.60"},{"date" => "2018-03-01T13:29:17","version" => "4.60_1"},{"date" => "2018-03-02T09:20:24","version" => "4.61"}]},"CGI-Application-Dispatch" => {"advisories" => [{"affected_versions" => ["<1.02"],"cves" => [],"description" => "Untainted module names.\n","distribution" => "CGI-Application-Dispatch","fixed_versions" => [">=1.02"],"id" => "CPANSA-CGI-Application-Dispatch-2005-001","references" => ["https://metacpan.org/changes/distribution/CGI-Application-Dispatch"],"reported" => "2005-01-20"}],"main_module" => "CGI::Application::Dispatch","versions" => [{"date" => "2004-09-13T01:35:58","version" => "0.01"},{"date" => "2004-10-19T18:26:01","version" => "0.02"},{"date" => "2004-10-29T16:53:40","version" => "0.03"},{"date" => "2005-01-06T15:34:49","version" => "1.00"},{"date" => "2005-01-08T12:42:00","version" => "1.01"},{"date" => "2005-01-20T14:43:28","version" => "1.02"},{"date" => "2005-03-04T16:28:16","version" => "1.03"},{"date" => "2005-07-12T21:44:54","version" => "1.04"},{"date" => "2006-01-12T15:56:53","version" => "2.00_02"},{"date" => "2006-02-06T15:50:52","version" => "2.00_03"},{"date" => "2006-02-14T15:41:25","version" => "2.00_04"},{"date" => "2006-04-12T14:18:22","version" => "2.00_05"},{"date" => "2006-06-27T04:29:04","version" => "2.00_06"},{"date" => "2006-07-03T15:52:12","version" => "2.00"},{"date" => "2006-08-14T14:14:10","version" => "2.01"},{"date" => "2006-08-17T14:57:55","version" => "2.02"},{"date" => "2006-09-30T02:13:40","version" => "2.03"},{"date" => "2007-01-03T18:12:57","version" => "2.10_01"},{"date" => "2007-01-11T18:55:41","version" => "2.10_02"},{"date" => "2007-01-15T14:08:30","version" => "2.10"},{"date" => "2007-12-28T20:23:49","version" => "2.11"},{"date" => "2007-12-31T20:43:51","version" => "2.12_01"},{"date" => "2008-01-03T14:39:57","version" => "2.12"},{"date" => "2008-03-08T18:33:34","version" => "2.13_01"},{"date" => "2008-03-11T16:41:27","version" => "2.13_02"},{"date" => "2008-09-17T00:44:02","version" => "2.13"},{"date" => "2008-11-03T01:33:21","version" => "2.14"},{"date" => "2008-12-04T16:00:05","version" => "2.15"},{"date" => "2009-03-24T02:03:51","version" => "2.16"},{"date" => "2009-12-30T19:06:27","version" => "2.17"},{"date" => "2011-01-05T03:42:59","version" => "2.18"},{"date" => "2011-06-16T17:42:14","version" => "3.00"},{"date" => "2011-06-24T02:53:20","version" => "3.01"},{"date" => "2011-06-24T22:33:01","version" => "3.02"},{"date" => "2011-06-26T03:52:14","version" => "3.03"},{"date" => "2011-06-29T13:45:53","version" => "3.04"},{"date" => "2011-09-07T22:21:15","version" => "3.05"},{"date" => "2011-09-09T15:29:58","version" => "3.06"},{"date" => "2011-09-09T17:32:11","version" => "3.07"},{"date" => "2012-09-03T04:04:19","version" => "3.10"},{"date" => "2012-09-14T01:02:58","version" => "3.11"},{"date" => "2012-09-14T01:19:52","version" => "3.12"}]},"CGI-Application-Plugin-AutoRunmode" => {"advisories" => [{"affected_versions" => ["<0.04"],"cves" => [],"description" => "Non-word characters are allowed in runmode name.\n","distribution" => "CGI-Application-Plugin-AutoRunmode","fixed_versions" => [">=0.04"],"id" => "CPANSA-CGI-Application-Plugin-AutoRunmode-2005-01","references" => ["https://metacpan.org/changes/distribution/CGI-Application-Plugin-AutoRunmode"],"reported" => "2005-03-04"}],"main_module" => "CGI::Application::Plugin::AutoRunmode","versions" => [{"date" => "2005-03-04T06:59:51","version" => "0.04"},{"date" => "2005-03-10T07:22:55","version" => "0.05"},{"date" => "2005-06-15T10:20:17","version" => "0.06"},{"date" => "2005-06-18T02:09:08","version" => "0.07"},{"date" => "2005-07-17T00:49:10","version" => "0.08"},{"date" => "2005-09-22T12:31:22","version" => "0.09"},{"date" => "2005-10-16T00:17:47","version" => "0.10"},{"date" => "2005-10-18T13:23:50","version" => "0.11"},{"date" => "2005-11-03T01:10:37","version" => "0.12"},{"date" => "2006-04-08T07:18:44","version" => "0.13"},{"date" => "2006-05-21T05:04:48","version" => "0.14"},{"date" => "2006-12-17T07:46:24","version" => "0.15"},{"date" => "2009-02-14T09:16:39","version" => "0.16"},{"date" => "2010-05-21T04:24:45","version" => "0.17"},{"date" => "2011-02-18T09:23:15","version" => "0.18"}]},"CGI-Application-Plugin-CAPTCHA" => {"advisories" => [{"affected_versions" => ["<0.02"],"cves" => [],"description" => "A malicious programmer creating an application to use the service can just have his application send along a cookie that he has created himself, and with that supply an appropriate verification string for his cookie. To avoid that you need to include som kind of hidden server-side password in the string being encrypted, and also include it when you verify.\n","distribution" => "CGI-Application-Plugin-CAPTCHA","fixed_versions" => [">=0.02"],"id" => "CPANSA-CGI-Application-Plugin-CAPTCHA-2024-001","references" => ["https://rt.cpan.org/Public/Bug/Display.html?id=30759","https://metacpan.org/release/CROMEDOME/CGI-Application-Plugin-CAPTCHA-0.02/changes","https://github.com/cromedome/cgi-application-plugin-captcha/commit/9acb5b6561a9983787ad85f55b01c69a895014e6"],"reported" => undef,"severity" => undef}],"main_module" => "CGI::Application::Plugin::CAPTCHA","versions" => [{"date" => "2005-08-28T18:31:21","version" => "0.01"},{"date" => "2011-01-05T05:07:55","version" => "0.02"},{"date" => "2011-01-05T05:10:54","version" => "0.03"},{"date" => "2011-01-06T14:08:56","version" => "0.04"}]},"CGI-Application-Plugin-RunmodeDeclare" => {"advisories" => [{"affected_versions" => ["<0.03"],"cves" => [],"description" => "Wrong order of arguments.\n","distribution" => "CGI-Application-Plugin-RunmodeDeclare","fixed_versions" => [">=0.03"],"id" => "CPANSA-CGI-Application-Plugin-RunmodeDeclare-2008-01","references" => ["https://metacpan.org/changes/distribution/CGI-Application-Plugin-RunmodeDeclare"],"reported" => "2008-10-20"}],"main_module" => "CGI::Application::Plugin::RunmodeDeclare","versions" => [{"date" => "2008-09-26T19:59:14","version" => "0.01"},{"date" => "2008-09-26T21:37:11","version" => "0.02"},{"date" => "2008-10-19T23:22:06","version" => "0.03"},{"date" => "2008-10-23T14:18:23","version" => "0.03_01"},{"date" => "2008-10-24T13:32:43","version" => "0.03_02"},{"date" => "2008-10-24T16:20:27","version" => "0.03_03"},{"date" => "2008-10-25T10:54:25","version" => "0.04"},{"date" => "2008-10-25T11:46:28","version" => "0.05"},{"date" => "2008-10-25T16:39:34","version" => "0.06"},{"date" => "2009-01-10T02:32:39","version" => "0.07"},{"date" => "2009-05-17T22:29:18","version" => "0.08"},{"date" => "2010-01-07T13:24:09","version" => "0.09"},{"date" => "2012-02-10T00:53:54","version" => "0.10"}]},"CGI-Auth-Basic" => {"advisories" => [{"affected_versions" => ["<1.11"],"cves" => [],"description" => "TBD\n","distribution" => "CGI-Auth-Basic","fixed_versions" => [">=1.11"],"id" => "CPANSA-CGI-Auth-Basic-2007-01","references" => ["https://metacpan.org/changes/distribution/CGI-Auth-Basic"],"reported" => "2007-12-30"}],"main_module" => "CGI::Auth::Basic","versions" => [{"date" => "2004-02-21T14:58:09","version" => "1.0"},{"date" => "2004-08-31T13:29:28","version" => "1.01"},{"date" => "2004-11-07T03:34:32","version" => "1.02"},{"date" => "2006-06-18T01:12:15","version" => "1.10"},{"date" => "2007-12-30T20:53:33","version" => "1.11"},{"date" => "2009-04-18T04:22:51","version" => "1.20"},{"date" => "2009-04-23T17:00:50","version" => "1.21"},{"date" => "2009-04-24T15:07:48","version" => "1.21"},{"date" => "2012-08-27T01:50:53","version" => "1.22"},{"date" => "2015-01-21T00:26:01","version" => "1.23"},{"date" => "2018-12-23T21:03:03","version" => "1.24"}]},"CGI-Session" => {"advisories" => [{"affected_versions" => ["<4.10"],"cves" => ["CVE-2006-1279"],"description" => "CGI::Session 4.03-1 allows local users to overwrite arbitrary files via a symlink attack on temporary files used by (1) Driver::File, (2) Driver::db_file, and possibly (3) Driver::sqlite.\n","distribution" => "CGI-Session","fixed_versions" => [],"id" => "CPANSA-CGI-Session-2006-1279","references" => ["http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=356555","http://secunia.com/advisories/19211","http://www.securityfocus.com/bid/17177","http://www.osvdb.org/23865","http://www.vupen.com/english/advisories/2006/0946","https://exchange.xforce.ibmcloud.com/vulnerabilities/25285"],"reported" => "2006-03-19","severity" => undef},{"affected_versions" => ["<4.12"],"cves" => [],"description" => "possible SQL injection attack\n","distribution" => "CGI-Session","fixed_versions" => [">=4.12"],"id" => "CPANSA-CGI-Session-2006-01","references" => ["https://rt.cpan.org/Public/Bug/Display.html?id=18578"],"reported" => "2006-04-06","severity" => undef}],"main_module" => "CGI::Session","versions" => [{"date" => "2001-10-30T08:59:10","version" => "0.01"},{"date" => "2002-05-10T12:04:15","version" => "2.0"},{"date" => "2002-05-10T17:38:46","version" => "2.1"},{"date" => "2002-05-14T18:21:39","version" => "2.2"},{"date" => "2002-05-17T18:02:23","version" => "2.4"},{"date" => "2002-05-27T09:52:46","version" => "2.7"},{"date" => "2002-06-06T08:08:21","version" => "2.9"},{"date" => "2002-06-06T08:36:26","version" => "2.91"},{"date" => "2002-06-18T18:15:57","version" => "2.92"},{"date" => "2002-08-26T08:23:54","version" => "2.94"},{"date" => "2002-11-27T07:20:47","version" => "3.1"},{"date" => "2002-11-27T12:27:59","version" => "3.2"},{"date" => "2002-11-28T03:19:31","version" => "v3.2.2.1"},{"date" => "2002-11-28T03:26:41","version" => "3.3"},{"date" => "2002-11-28T03:44:39","version" => "3.4"},{"date" => "2002-11-28T06:55:29","version" => "3.5"},{"date" => "2002-11-28T17:12:32","version" => "3.6"},{"date" => "2002-11-29T21:29:53","version" => "3.7"},{"date" => "2002-12-03T16:26:55","version" => "3.8"},{"date" => "2002-12-04T07:37:02","version" => "3.9"},{"date" => "2002-12-09T09:02:18","version" => "3.10"},{"date" => "2002-12-09T20:09:24","version" => "3.11"},{"date" => "2003-03-09T11:26:21","version" => "3.91"},{"date" => "2003-03-10T02:42:16","version" => "3.92"},{"date" => "2003-03-14T13:21:20","version" => "3.93"},{"date" => "2003-05-02T20:12:40","version" => "3.94"},{"date" => "2003-07-26T13:51:31","version" => "3.95"},{"date" => "2005-02-09T08:35:23","version" => "4.00_01"},{"date" => "2005-02-09T09:54:17","version" => "4.00_02"},{"date" => "2005-02-11T08:23:00","version" => "4.00_03"},{"date" => "2005-02-17T03:24:21","version" => "4.00_04"},{"date" => "2005-02-22T17:56:43","version" => "4.00_05"},{"date" => "2005-02-24T18:46:45","version" => "4.00_06"},{"date" => "2005-03-13T19:18:37","version" => "4.00_07"},{"date" => "2005-03-15T16:48:17","version" => "4.00_08"},{"date" => "2005-07-22T02:00:21","version" => "4.00_09"},{"date" => "2005-09-01T05:57:49","version" => "4.00"},{"date" => "2005-09-01T16:25:46","version" => "4.01"},{"date" => "2005-09-02T15:51:20","version" => "4.02"},{"date" => "2005-09-24T02:12:22","version" => "4.02_01"},{"date" => "2005-10-05T23:22:54","version" => "4.03"},{"date" => "2006-03-02T03:00:28","version" => "4.04"},{"date" => "2006-03-04T00:08:26","version" => "4.05"},{"date" => "2006-03-09T03:13:06","version" => "4.06"},{"date" => "2006-03-09T12:09:28","version" => "4.07"},{"date" => "2006-03-16T02:38:25","version" => "4.08"},{"date" => "2006-03-17T04:08:57","version" => "4.09"},{"date" => "2006-03-28T07:00:52","version" => "4.10"},{"date" => "2006-04-03T19:33:30","version" => "4.11"},{"date" => "2006-04-07T14:34:06","version" => "4.12"},{"date" => "2006-04-12T17:05:18","version" => "4.13"},{"date" => "2006-06-11T11:36:57","version" => "4.14"},{"date" => "2006-11-24T14:10:38","version" => "4.20_1"},{"date" => "2006-12-05T02:08:37","version" => "4.20"},{"date" => "2008-03-22T02:42:57","version" => "4.29_1"},{"date" => "2008-03-28T01:45:56","version" => "4.29_2"},{"date" => "2008-04-26T01:31:34","version" => "4.30"},{"date" => "2008-06-16T14:44:06","version" => "4.31"},{"date" => "2008-06-17T21:35:03","version" => "4.32"},{"date" => "2008-07-08T01:27:59","version" => "4.33"},{"date" => "2008-07-13T02:39:59","version" => "4.34"},{"date" => "2008-07-16T00:36:46","version" => "4.35"},{"date" => "2008-09-13T15:45:06","version" => "4.36"},{"date" => "2008-10-23T02:57:30","version" => "4.37"},{"date" => "2008-11-01T03:47:46","version" => "4.38"},{"date" => "2008-12-16T01:22:42","version" => "4.39"},{"date" => "2009-01-03T01:18:15","version" => "4.40"},{"date" => "2009-03-21T02:01:09","version" => "4.41"},{"date" => "2009-08-26T13:38:39","version" => "4.42"},{"date" => "2010-12-12T00:32:27","version" => "4.43"},{"date" => "2011-06-06T20:48:07","version" => "4.44"},{"date" => "2011-07-02T01:33:33","version" => "4.45"},{"date" => "2011-07-08T14:34:42","version" => "4.46"},{"date" => "2011-07-08T19:31:44","version" => "4.47"},{"date" => "2011-07-11T13:02:37","version" => "4.48"},{"date" => "2021-02-08T07:00:20","version" => "4.49"}]},"CGI-Simple" => {"advisories" => [{"affected_versions" => ["<1.113"],"cves" => ["CVE-2010-4410"],"description" => "Newlines in headers, which could lead to header injection attacks.\n","distribution" => "CGI-Simple","fixed_versions" => [">=1.113"],"id" => "CPANSA-CGI-Simple-2010-02","references" => ["https://metacpan.org/changes/distribution/CGI-Simple"],"reported" => "2010-12-27"},{"affected_versions" => ["<1.113"],"cves" => [],"description" => "Non-random multipart boundary.\n","distribution" => "CGI-Simple","fixed_versions" => [">=1.113"],"id" => "CPANSA-CGI-Simple-2010-01","references" => ["https://metacpan.org/changes/distribution/CGI-Simple"],"reported" => "2010-12-27"},{"affected_versions" => ["<=1.112"],"cves" => ["CVE-2010-2761"],"description" => "The multipart_init function in (1) CGI.pm before 3.50 and (2) Simple.pm in CGI::Simple 1.112 and earlier uses a hardcoded value of the MIME boundary string in multipart/x-mixed-replace content, which allows remote attackers to inject arbitrary HTTP headers and conduct HTTP response splitting attacks via crafted input that contains this value, a different vulnerability than CVE-2010-3172.\n","distribution" => "CGI-Simple","fixed_versions" => [],"id" => "CPANSA-CGI-Simple-2010-2761","references" => ["https://github.com/AndyA/CGI--Simple/commit/e4942b871a26c1317a175a91ebb7262eea59b380","http://cpansearch.perl.org/src/LDS/CGI.pm-3.50/Changes","http://openwall.com/lists/oss-security/2010/12/01/1","http://www.nntp.perl.org/group/perl.perl5.changes/2010/11/msg28043.html","http://perl5.git.perl.org/perl.git/blobdiff/a0b94c2432b1d8c20653453a0f6970cb10f59aec..84601d63a7e34958da47dad1e61e27cb3bd467d1:/cpan/CGI/lib/CGI.pm","http://perl5.git.perl.org/perl.git/commit/84601d63a7e34958da47dad1e61e27cb3bd467d1","http://openwall.com/lists/oss-security/2010/12/01/2","http://openwall.com/lists/oss-security/2010/12/01/3","https://bugzilla.mozilla.org/show_bug.cgi?id=600464","http://osvdb.org/69588","http://osvdb.org/69589","http://lists.opensuse.org/opensuse-security-announce/2011-01/msg00003.html","http://www.mandriva.com/security/advisories?name=MDVSA-2010:237","http://www.vupen.com/english/advisories/2011/0076","http://www.mandriva.com/security/advisories?name=MDVSA-2010:250","http://secunia.com/advisories/42877","https://bugzilla.mozilla.org/show_bug.cgi?id=591165","http://www.vupen.com/english/advisories/2011/0207","http://www.bugzilla.org/security/3.2.9/","http://secunia.com/advisories/43033","http://lists.fedoraproject.org/pipermail/package-announce/2011-January/053591.html","http://secunia.com/advisories/43147","http://lists.fedoraproject.org/pipermail/package-announce/2011-January/053576.html","http://www.vupen.com/english/advisories/2011/0249","http://www.vupen.com/english/advisories/2011/0271","http://lists.fedoraproject.org/pipermail/package-announce/2011-February/053678.html","http://www.vupen.com/english/advisories/2011/0212","http://lists.opensuse.org/opensuse-security-announce/2011-01/msg00006.html","http://secunia.com/advisories/43165","http://secunia.com/advisories/43068","http://lists.fedoraproject.org/pipermail/package-announce/2011-February/053665.html","http://www.redhat.com/support/errata/RHSA-2011-1797.html","http://lists.opensuse.org/opensuse-security-announce/2011-04/msg00000.html","http://kb.juniper.net/InfoCenter/index?page=content&id=JSA10735","http://kb.juniper.net/InfoCenter/index?page=content&id=JSA10705"],"reported" => "2010-12-06","severity" => undef},{"affected_versions" => ["<=1.282"],"cves" => ["CVE-2025-40927"],"description" => "CGI::Simple versions before 1.282 for Perl has a HTTP response splitting flaw This vulnerability is a confirmed HTTP response splitting\x{a0}flaw in CGI::Simple\x{a0}that allows HTTP response header injection, which can be used for reflected XSS or open redirect under certain conditions.  Although some validation exists, it can be bypassed using URL-encoded values, allowing an attacker to inject untrusted content into the response via query parameters.    As a result, an attacker can inject a line break (e.g. %0A) into the parameter value, causing the server to split the HTTP response and inject arbitrary headers or even an HTML/JavaScript body, leading to reflected cross-site scripting (XSS), open redirect or other attacks.  The issue documented in CVE-2010-4410 https://www.cve.org/CVERecord?id=CVE-2010-4410 is related but the fix was incomplete.  Impact  By injecting %0A\x{a0}(newline) into a query string parameter, an attacker can:    *  Break the current HTTP header   *  Inject a new header or entire body   *  Deliver a script payload that is reflected in the server\x{2019}s response That can lead to the following attacks:    *  reflected XSS   *  open redirect   *  cache poisoning   *  header manipulation","distribution" => "CGI-Simple","fixed_versions" => [">=1.282"],"id" => "CPANSA-CGI-Simple-2025-40927","references" => ["https://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2004-2320","https://datatracker.ietf.org/doc/html/rfc7230#section-3","https://metacpan.org/release/MANWAR/CGI-Simple-1.281/diff/MANWAR/CGI-Simple-1.282/lib/CGI/Simple.pm","https://metacpan.org/release/MANWAR/CGI-Simple-1.281/source/lib/CGI/Simple.pm#L1031-1035","https://owasp.org/www-community/attacks/HTTP_Response_Splitting","https://rt.perl.org/Public/Bug/Display.html?id=21951"],"reported" => "2025-08-29","severity" => undef}],"main_module" => "CGI::Simple","versions" => [{"date" => "2007-01-09T22:31:27","version" => "0.078"},{"date" => "2007-02-23T16:22:19","version" => "0.079"},{"date" => "2007-03-30T20:15:35","version" => "0.080"},{"date" => "2007-05-20T19:19:40","version" => "0.081"},{"date" => "2007-05-22T18:43:01","version" => "0.082"},{"date" => "2007-05-22T18:54:06","version" => "0.83"},{"date" => "2007-05-24T03:15:01","version" => "1.0"},{"date" => "2007-07-13T18:58:16","version" => "1.1"},{"date" => "2007-07-31T01:57:01","version" => "1.1.1"},{"date" => "2007-07-31T02:04:25","version" => "1.1.2"},{"date" => "2007-07-31T02:10:47","version" => "1.103"},{"date" => "2008-05-13T15:46:18","version" => "1.104"},{"date" => "2008-05-16T14:37:31","version" => "1.105"},{"date" => "2008-09-14T13:29:51","version" => "1.106"},{"date" => "2009-03-07T21:24:59","version" => "1.107"},{"date" => "2009-03-13T14:06:24","version" => "1.108"},{"date" => "2009-04-16T17:54:13","version" => "1.109"},{"date" => "2009-05-24T21:25:22","version" => "1.110"},{"date" => "2009-05-28T18:02:08","version" => "1.111"},{"date" => "2009-05-31T10:43:56","version" => "1.112"},{"date" => "2010-12-27T13:11:56","version" => "1.113"},{"date" => "2014-10-19T12:53:24","version" => "1.115"},{"date" => "2018-03-01T15:09:42","version" => "1.13"},{"date" => "2018-03-03T10:42:06","version" => "1.14"},{"date" => "2018-03-04T03:42:20","version" => "1.15"},{"date" => "2018-07-25T15:17:39","version" => "1.16"},{"date" => "2018-10-02T09:48:08","version" => "1.17"},{"date" => "2018-10-03T14:21:12","version" => "1.18"},{"date" => "2018-10-04T12:05:58","version" => "1.19"},{"date" => "2018-10-05T11:30:05","version" => "1.20"},{"date" => "2018-10-06T07:21:31","version" => "1.21"},{"date" => "2019-09-07T04:28:17","version" => "1.22"},{"date" => "2020-02-06T06:12:09","version" => "1.23"},{"date" => "2020-02-07T11:11:56","version" => "1.24"},{"date" => "2020-02-10T13:00:54","version" => "1.25"},{"date" => "2022-01-02T18:00:56","version" => "1.26"},{"date" => "2022-01-06T16:00:18","version" => "1.27"},{"date" => "2022-01-11T15:16:20","version" => "1.280"},{"date" => "2024-01-31T14:19:02","version" => "1.281"},{"date" => "2025-08-28T19:12:51","version" => "1.282"}]},"CGI-apacheSSI" => {"advisories" => [{"affected_versions" => ["<0.95"],"cves" => [],"description" => "Security and parsing problems with \"include\" calls.\n","distribution" => "CGI-apacheSSI","fixed_versions" => [">=0.95"],"id" => "CPANSA-CGI-apacheSSI-2016-01","references" => ["https://metacpan.org/changes/distribution/CGI-apacheSSI"],"reported" => "2016-01-31"}],"main_module" => "CGI::apacheSSI","versions" => [{"date" => "2014-08-20T22:55:20","version" => "0.93"},{"date" => "2016-01-30T12:57:47","version" => "0.94"},{"date" => "2016-01-31T22:48:55","version" => "0.95"},{"date" => "2016-02-01T00:36:49","version" => "0.96"}]},"CPAN" => {"advisories" => [{"affected_versions" => ["<2.35"],"cves" => ["CVE-2023-31484"],"description" => "The verify_SSL flag is missing from HTTP::Tiny, and allows a network attacker to MITM the connection if it is used by the CPAN client\n","distribution" => "CPAN","fixed_versions" => [">=2.35"],"id" => "CPANSA-CPAN-2023-31484","previous_id" => ["CPANSA-CPAN-2023-01"],"references" => ["https://github.com/andk/cpanpm/commit/9c98370287f4e709924aee7c58ef21c85289a7f0","https://blog.hackeriet.no/perl-http-tiny-insecure-tls-default-affects-cpan-modules/","https://github.com/andk/cpanpm/pull/175","https://www.openwall.com/lists/oss-security/2023/04/18/14"],"reported" => "2023-02-28"},{"affected_versions" => ["<1.93"],"cves" => [],"description" => "Archive::Tar preserves permissions in the tarball; extracted file permissions will be set from users umask instead.\n","distribution" => "CPAN","fixed_versions" => [">=1.93"],"id" => "CPANSA-CPAN-2009-01","references" => ["https://github.com/andk/cpanpm/commit/079fa2e7ee77d626eab8bb06d0465c6a05f6c8b6","https://rt.cpan.org/Ticket/Display.html?id=46384"],"reported" => "2009-09-23"},{"affected_versions" => ["<2.28"],"cves" => ["CVE-2020-16156"],"description" => "CPAN 2.28 allows Signature Verification Bypass.","distribution" => "CPAN","fixed_versions" => [">=2.29"],"id" => "CPANSA-CPAN-2020-16156","references" => ["https://metacpan.org/pod/distribution/CPAN/scripts/cpan","https://blog.hackeriet.no/cpan-signature-verification-vulnerabilities/","http://blogs.perl.org/users/neilb/2021/11/addressing-cpan-vulnerabilities-related-to-checksums.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/SZ32AJIV4RHJMLWLU5QULGKMMIHYOMDC/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/SD6RYOJII7HRJ6WVORFNVTYNOFY5JDXN/"],"reported" => "2021-12-13","severity" => "high"}],"main_module" => "CPAN","versions" => [{"date" => "1996-09-10T17:13:59","version" => "0.17"},{"date" => "1996-09-10T20:51:00","version" => "0.20"},{"date" => "1996-09-12T05:53:35","version" => "0.26"},{"date" => "1996-09-12T14:01:39","version" => "0.27"},{"date" => "1996-09-16T20:18:59","version" => "0.28"},{"date" => "1996-09-17T17:14:51","version" => "0.29"},{"date" => "1996-09-19T05:24:17","version" => "0.30"},{"date" => "1996-09-20T10:40:01","version" => "0.31"},{"date" => "1996-09-22T19:30:33","version" => "0.35"},{"date" => "1996-09-23T12:55:23","version" => "0.36"},{"date" => "1996-09-23T14:05:44","version" => "0.37"},{"date" => "1996-09-27T12:52:07","version" => "0.39"},{"date" => "1996-09-28T20:51:31","version" => "0.40"},{"date" => "1996-10-01T21:14:27","version" => "0.41"},{"date" => "1996-11-17T07:56:02","version" => "0.42"},{"date" => "1996-11-17T14:51:59","version" => "0.43"},{"date" => "1996-11-30T17:04:28","version" => "0.44"},{"date" => "1996-12-01T12:19:19","version" => "0.45"},{"date" => "1996-12-01T18:24:17","version" => "0.46"},{"date" => "1996-12-10T00:58:25","version" => "1.00"},{"date" => "1996-12-10T10:17:15","version" => "1.01"},{"date" => "1996-12-11T01:31:55","version" => "1.02"},{"date" => "1996-12-21T03:10:23","version" => "1.03"},{"date" => "1996-12-21T20:08:49","version" => "1.04"},{"date" => "1996-12-22T13:04:58","version" => "1.05"},{"date" => "1996-12-22T14:16:08","version" => "1.06"},{"date" => "1996-12-23T04:05:01","version" => "1.07"},{"date" => "1996-12-23T13:18:01","version" => "1.08"},{"date" => "1996-12-24T00:46:19","version" => "1.09"},{"date" => "1997-01-17T02:29:49","version" => "1.09_01"},{"date" => "1997-01-21T01:06:40","version" => "1.10"},{"date" => "1997-01-22T18:50:00","version" => "1.11"},{"date" => "1997-01-23T00:07:58","version" => "1.12"},{"date" => "1997-01-24T01:07:44","version" => "1.14"},{"date" => "1997-01-24T12:32:12","version" => "1.15"},{"date" => "1997-02-02T13:51:48","version" => "1.16_01"},{"date" => "1997-02-02T21:05:12","version" => "1.17"},{"date" => "1997-02-03T00:38:36","version" => "1.18"},{"date" => "1997-02-03T09:13:48","version" => "1.19"},{"date" => "1997-02-05T09:38:00","version" => "1.20"},{"date" => "1997-02-11T06:32:42","version" => "1.21"},{"date" => "1997-03-13T23:14:59","version" => "1.22_01"},{"date" => "1997-03-31T12:03:55","version" => "1.23"},{"date" => "1997-03-31T22:47:11","version" => "1.24"},{"date" => "1997-06-30T18:13:23","version" => "1.25"},{"date" => "1997-07-28T13:58:09","version" => "1.27"},{"date" => "1997-08-04T06:09:33","version" => "1.28"},{"date" => "1997-08-11T23:33:58","version" => "1.29"},{"date" => "1997-08-29T14:34:37","version" => "1.30"},{"date" => "1997-09-21T08:53:03","version" => "1.31"},{"date" => "1997-09-23T18:45:50","version" => "1.3101"},{"date" => "1998-01-02T18:22:35","version" => "1.32"},{"date" => "1998-01-10T18:24:23","version" => "1.33"},{"date" => "1998-02-03T18:06:41","version" => "1.35"},{"date" => "1998-02-08T08:55:55","version" => "1.36"},{"date" => "1998-06-12T06:51:25","version" => "1.37"},{"date" => "1998-06-14T20:18:08","version" => "1.38"},{"date" => "1998-07-24T20:13:41","version" => "1.40"},{"date" => "1998-12-01T02:20:32","version" => "1.41"},{"date" => "1998-12-01T07:58:35","version" => "1.42"},{"date" => "1998-12-01T22:16:27","version" => "1.43"},{"date" => "1998-12-03T17:07:54","version" => "1.43"},{"date" => "1999-01-09T18:38:33","version" => "1.44"},{"date" => "1999-01-10T19:38:27","version" => "1.44_51"},{"date" => "1999-01-13T12:15:42","version" => "1.44_52"},{"date" => "1999-01-15T09:26:40","version" => "1.44_53"},{"date" => "1999-01-15T09:27:45","version" => "1.44_54"},{"date" => "1999-01-23T14:56:16","version" => "1.45"},{"date" => "1999-01-25T01:43:42","version" => "1.46"},{"date" => "1999-01-25T13:11:23","version" => "1.47"},{"date" => "1999-03-06T19:34:54","version" => "1.48"},{"date" => "1999-05-22T16:45:00","version" => "1.49"},{"date" => "1999-05-23T14:32:20","version" => "1.50"},{"date" => "1999-10-23T03:06:39","version" => "1.50_01"},{"date" => "1999-12-29T22:30:22","version" => "1.51"},{"date" => "2000-01-08T15:32:55","version" => "1.52"},{"date" => "2000-03-23T23:39:41","version" => "1.53"},{"date" => "2000-03-25T22:51:15","version" => "1.54"},{"date" => "2000-07-30T11:15:04","version" => "1.55"},{"date" => "2000-08-01T20:47:09","version" => "1.56"},{"date" => "2000-08-16T12:54:07","version" => "1.57"},{"date" => "2000-08-21T19:44:18","version" => "1.57_51"},{"date" => "2000-08-27T22:09:36","version" => "1.57_53"},{"date" => "2000-08-30T16:54:50","version" => "1.57_54"},{"date" => "2000-08-31T08:11:01","version" => "1.57_55"},{"date" => "2000-08-31T22:16:21","version" => "1.57_56"},{"date" => "2000-09-01T12:18:43","version" => "1.57_57"},{"date" => "2000-09-03T22:19:20","version" => "1.57_58"},{"date" => "2000-09-05T09:44:05","version" => "1.57_59"},{"date" => "2000-09-05T19:55:34","version" => "1.57_60"},{"date" => "2000-09-06T10:54:07","version" => "1.57_61"},{"date" => "2000-09-08T02:19:06","version" => "1.57_62"},{"date" => "2000-09-10T08:54:37","version" => "1.57_65"},{"date" => "2000-09-12T08:46:40","version" => "1.57_66"},{"date" => "2000-09-17T10:24:31","version" => "1.57_67"},{"date" => "2000-10-08T14:25:04","version" => "1.57_68"},{"date" => "2000-10-18T14:53:45","version" => "1.58"},{"date" => "2000-10-21T14:21:06","version" => "1.58_51"},{"date" => "2000-10-25T07:05:38","version" => "1.58_52"},{"date" => "2000-10-26T11:03:29","version" => "1.58_53"},{"date" => "2000-10-26T15:34:21","version" => "1.58_54"},{"date" => "2000-10-27T07:59:03","version" => "1.58_55"},{"date" => "2000-11-04T09:36:53","version" => "1.58_56"},{"date" => "2000-11-06T19:30:27","version" => "1.58_57"},{"date" => "2000-11-08T08:10:51","version" => "1.58_90"},{"date" => "2000-11-13T10:26:38","version" => "1.58_91"},{"date" => "2000-11-14T18:24:18","version" => "1.58_92"},{"date" => "2000-11-15T07:19:56","version" => "1.58_93"},{"date" => "2000-12-01T06:05:58","version" => "1.59"},{"date" => "2000-12-01T08:19:58","version" => "1.59_51"},{"date" => "2000-12-26T13:54:06","version" => "1.59_52"},{"date" => "2001-01-02T16:37:24","version" => "1.59_53"},{"date" => "2001-02-09T21:44:55","version" => "1.59_54"},{"date" => "2002-04-19T13:29:54","version" => "1.60"},{"date" => "2002-04-20T02:18:41","version" => "1.60"},{"date" => "2002-04-21T11:31:25","version" => "1.60"},{"date" => "2002-05-07T10:38:54","version" => "1.61"},{"date" => "2002-07-28T10:51:47","version" => "1.62"},{"date" => "2002-08-30T08:58:10","version" => "1.63"},{"date" => "2003-02-06T10:04:06","version" => "1.64"},{"date" => "2003-02-08T17:10:13","version" => "1.65"},{"date" => "2003-03-04T19:38:21","version" => "1.70"},{"date" => "2003-04-11T04:33:18","version" => "1.70_52"},{"date" => "2003-04-13T12:43:40","version" => "1.70_53"},{"date" => "2003-05-15T21:04:52","version" => "1.70_54"},{"date" => "2003-07-04T09:48:08","version" => "1.71"},{"date" => "2003-07-27T20:35:05","version" => "1.72"},{"date" => "2003-07-28T08:21:47","version" => "1.73"},{"date" => "2003-07-28T22:58:08","version" => "1.74"},{"date" => "2003-07-29T15:14:13","version" => "1.75"},{"date" => "2003-07-31T15:14:02","version" => "1.76"},{"date" => "2003-09-21T21:25:41","version" => "1.76_01"},{"date" => "2005-09-19T06:37:38","version" => "1.76_51"},{"date" => "2005-09-22T07:02:02","version" => "1.76_52"},{"date" => "2005-09-22T07:09:48","version" => "1.76_53"},{"date" => "2005-10-01T08:23:38","version" => "1.76_54"},{"date" => "2005-10-19T06:10:58","version" => "1.76_55"},{"date" => "2005-10-21T04:59:36","version" => "1.76_56"},{"date" => "2005-10-27T07:08:29","version" => "1.76_57"},{"date" => "2005-11-02T04:03:28","version" => "1.76_58"},{"date" => "2005-11-03T06:37:52","version" => "1.76_59"},{"date" => "2005-11-03T07:38:40","version" => "1.76_60"},{"date" => "2005-11-06T10:36:53","version" => "1.76_61"},{"date" => "2005-11-07T04:22:19","version" => "1.76_62"},{"date" => "2005-11-07T04:47:05","version" => "1.76_63"},{"date" => "2005-11-07T21:58:06","version" => "1.76_64"},{"date" => "2005-11-07T22:18:44","version" => "1.76_65"},{"date" => "2005-12-03T10:12:08","version" => "1.80"},{"date" => "2005-12-18T11:29:26","version" => "1.80_51"},{"date" => "2005-12-21T12:13:15","version" => "1.80_53"},{"date" => "2005-12-22T08:42:59","version" => "1.80_54"},{"date" => "2005-12-24T07:25:34","version" => "1.80_55"},{"date" => "2005-12-24T09:59:47","version" => "1.80_56"},{"date" => "2005-12-31T11:58:10","version" => "1.80_57"},{"date" => "2006-01-01T09:01:43","version" => "1.80_58"},{"date" => "2006-01-02T23:15:15","version" => "1.81"},{"date" => "2006-01-04T07:47:25","version" => "1.82"},{"date" => "2006-01-05T08:03:36","version" => "1.83"},{"date" => "2006-01-08T13:35:16","version" => "1.83_51"},{"date" => "2006-01-10T05:00:26","version" => "1.83_52"},{"date" => "2006-01-12T07:54:36","version" => "1.83_53"},{"date" => "2006-01-13T08:20:42","version" => "1.83_54"},{"date" => "2006-01-14T11:34:47","version" => "1.83_55"},{"date" => "2006-01-18T06:03:44","version" => "1.83_56"},{"date" => "2006-01-19T08:00:02","version" => "1.83_57"},{"date" => "2006-01-22T12:05:01","version" => "1.83_58"},{"date" => "2006-01-25T13:10:20","version" => "1.83_59"},{"date" => "2006-01-30T10:35:47","version" => "1.83_60"},{"date" => "2006-01-30T23:18:09","version" => "1.83_61"},{"date" => "2006-01-31T10:28:57","version" => "1.83_62"},{"date" => "2006-02-01T07:49:36","version" => "1.83_63"},{"date" => "2006-02-02T09:17:39","version" => "1.83_64"},{"date" => "2006-02-04T11:20:05","version" => "1.83_65"},{"date" => "2006-02-04T17:05:00","version" => "1.83_66"},{"date" => "2006-02-06T00:46:27","version" => "1.83_67"},{"date" => "2006-02-08T07:43:36","version" => "1.83_68"},{"date" => "2006-02-14T08:17:55","version" => "1.83_69"},{"date" => "2006-02-15T07:01:02","version" => "1.84"},{"date" => "2006-02-19T17:05:36","version" => "1.85"},{"date" => "2006-02-20T08:36:51","version" => "1.86"},{"date" => "2006-02-21T06:05:05","version" => "1.86_51"},{"date" => "2006-02-22T22:29:54","version" => "1.86_52"},{"date" => "2006-02-24T08:24:09","version" => "1.86_53"},{"date" => "2006-02-27T07:01:10","version" => "1.87"},{"date" => "2006-03-06T08:02:28","version" => "1.87_51"},{"date" => "2006-07-21T22:33:11","version" => "1.87_52"},{"date" => "2006-07-22T18:55:13","version" => "1.87_53"},{"date" => "2006-07-23T21:37:11","version" => "1.87_54"},{"date" => "2006-07-29T19:36:50","version" => "1.87_55"},{"date" => "2006-08-24T05:57:41","version" => "1.87_56"},{"date" => "2006-08-26T17:05:56","version" => "1.87_57"},{"date" => "2006-08-31T06:50:49","version" => "1.87_58"},{"date" => "2006-09-03T21:05:29","version" => "1.87_59"},{"date" => "2006-09-10T11:57:33","version" => "1.87_61"},{"date" => "2006-09-11T21:24:18","version" => "1.87_62"},{"date" => "2006-09-13T05:44:15","version" => "1.87_63"},{"date" => "2006-09-16T11:02:25","version" => "1.87_64"},{"date" => "2006-09-19T03:44:51","version" => "1.87_65"},{"date" => "2006-09-21T20:30:41","version" => "1.88"},{"date" => "2006-09-22T20:40:40","version" => "1.8801"},{"date" => "2006-09-30T10:41:20","version" => "1.88_51"},{"date" => "2006-10-03T09:51:49","version" => "1.88_52"},{"date" => "2006-10-09T19:31:56","version" => "1.88_53"},{"date" => "2006-10-14T09:37:15","version" => "1.88_54"},{"date" => "2006-10-16T06:59:27","version" => "1.88_55"},{"date" => "2006-10-22T10:34:16","version" => "1.88_56"},{"date" => "2006-10-23T07:17:30","version" => "1.8802"},{"date" => "2006-10-24T07:18:16","version" => "1.88_57"},{"date" => "2006-10-28T15:00:07","version" => "1.88_58"},{"date" => "2006-11-05T21:24:52","version" => "1.88_59"},{"date" => "2006-11-10T08:39:55","version" => "1.88_61"},{"date" => "2006-11-13T07:44:27","version" => "1.88_62"},{"date" => "2006-11-29T08:11:50","version" => "1.88_63"},{"date" => "2006-12-04T07:53:37","version" => "1.88_64"},{"date" => "2006-12-11T21:36:04","version" => "1.88_65"},{"date" => "2006-12-19T08:21:17","version" => "1.88_66"},{"date" => "2006-12-31T17:18:53","version" => "1.88_67"},{"date" => "2007-01-07T21:22:12","version" => "1.88_68"},{"date" => "2007-01-08T03:42:56","version" => "1.88_69"},{"date" => "2007-01-27T16:57:49","version" => "1.88_71"},{"date" => "2007-01-31T07:11:33","version" => "1.88_72"},{"date" => "2007-02-13T05:24:13","version" => "1.88_73"},{"date" => "2007-02-15T07:12:17","version" => "1.88_74"},{"date" => "2007-02-18T16:52:49","version" => "1.88_75"},{"date" => "2007-02-19T06:20:20","version" => "1.88_76"},{"date" => "2007-02-19T21:26:47","version" => "1.88_77"},{"date" => "2007-03-05T23:26:57","version" => "1.88_78"},{"date" => "2007-03-16T01:54:55","version" => "1.88_79"},{"date" => "2007-04-07T07:41:18","version" => "1.90"},{"date" => "2007-04-19T07:03:03","version" => "1.91"},{"date" => "2007-04-23T00:09:11","version" => "1.9101"},{"date" => "2007-05-08T20:35:04","version" => "1.9102"},{"date" => "2007-07-07T16:15:40","version" => "1.91_51"},{"date" => "2007-07-14T18:45:58","version" => "1.91_52"},{"date" => "2007-08-09T06:49:38","version" => "1.91_53"},{"date" => "2007-09-14T21:18:33","version" => "1.91_54"},{"date" => "2007-09-15T07:14:26","version" => "1.91_55"},{"date" => "2007-09-23T11:15:08","version" => "1.92"},{"date" => "2007-09-27T07:11:10","version" => "1.9201"},{"date" => "2007-09-28T06:58:04","version" => "1.9202"},{"date" => "2007-09-28T07:13:26","version" => "1.9203"},{"date" => "2007-11-04T23:04:18","version" => "1.92_51"},{"date" => "2007-11-05T23:30:06","version" => "1.9204"},{"date" => "2007-11-11T11:27:20","version" => "1.92_52"},{"date" => "2007-11-11T18:49:37","version" => "1.9205"},{"date" => "2007-12-09T23:27:18","version" => "1.92_53"},{"date" => "2007-12-27T04:57:34","version" => "1.92_54"},{"date" => "2007-12-30T15:24:13","version" => "1.92_55"},{"date" => "2008-02-04T21:56:28","version" => "1.92_56"},{"date" => "2008-02-27T05:13:49","version" => "1.92_57"},{"date" => "2008-03-12T07:56:18","version" => "1.92_58"},{"date" => "2008-03-16T18:57:04","version" => "1.92_59"},{"date" => "2008-03-26T07:53:08","version" => "1.92_60"},{"date" => "2008-04-25T04:47:52","version" => "1.92_61"},{"date" => "2008-05-23T04:07:04","version" => "1.92_62"},{"date" => "2008-06-19T06:42:18","version" => "1.92_63"},{"date" => "2008-09-03T05:27:35","version" => "1.92_64"},{"date" => "2008-09-14T09:54:03","version" => "1.92_65"},{"date" => "2008-09-29T23:15:10","version" => "1.92_66"},{"date" => "2008-10-12T16:07:51","version" => "1.93"},{"date" => "2008-10-13T19:37:43","version" => "1.9301"},{"date" => "2009-01-11T22:07:01","version" => "1.93_02"},{"date" => "2009-02-01T12:38:23","version" => "1.93_03"},{"date" => "2009-02-01T21:06:21","version" => "1.93_51"},{"date" => "2009-02-28T15:58:39","version" => "1.9304"},{"date" => "2009-04-13T19:24:43","version" => "1.93_52"},{"date" => "2009-05-04T06:11:28","version" => "1.93_53"},{"date" => "2009-05-07T20:13:16","version" => "1.93_54"},{"date" => "2009-05-24T05:37:28","version" => "1.94"},{"date" => "2009-06-14T19:53:52","version" => "1.94_01"},{"date" => "2009-06-27T02:55:22","version" => "1.9402"},{"date" => "2009-09-14T02:47:24","version" => "1.94_51"},{"date" => "2009-10-15T19:33:19","version" => "1.94_52"},{"date" => "2009-12-18T07:00:09","version" => "1.94_53"},{"date" => "2010-01-14T08:01:42","version" => "1.94_54"},{"date" => "2010-02-03T03:43:49","version" => "1.94_55"},{"date" => "2010-02-17T13:39:33","version" => "1.94_56"},{"date" => "2010-05-24T19:33:41","version" => "1.94_57"},{"date" => "2010-06-24T06:34:13","version" => "1.94_58"},{"date" => "2010-09-26T20:23:30","version" => "1.94_59"},{"date" => "2010-09-28T20:44:58","version" => "1.94_60"},{"date" => "2010-10-03T17:29:37","version" => "1.94_61"},{"date" => "2010-10-26T06:43:51","version" => "1.94_62"},{"date" => "2011-01-16T17:58:10","version" => "1.94_63"},{"date" => "2011-01-21T04:58:35","version" => "1.94_64"},{"date" => "2011-02-14T12:10:12","version" => "1.94_65"},{"date" => "2011-03-12T11:30:03","version" => "1.9600"},{"date" => "2011-06-27T06:56:01","version" => "1.97_51"},{"date" => "2011-08-07T09:40:33","version" => "1.9800"},{"date" => "2012-10-16T21:42:49","version" => "1.99_51"},{"date" => "2013-02-06T07:41:54","version" => "2.00-TRIAL"},{"date" => "2013-04-12T16:57:44","version" => "2.00"},{"date" => "2013-06-22T20:27:32","version" => "2.01-TRIAL"},{"date" => "2013-06-23T07:33:40","version" => "2.02-TRIAL"},{"date" => "2013-09-15T09:42:33","version" => "2.03-TRIAL"},{"date" => "2014-03-18T22:33:22","version" => "2.04-TRIAL"},{"date" => "2014-03-31T20:55:24","version" => "2.05-TRIAL"},{"date" => "2014-04-04T02:07:20","version" => "2.05-TRIAL2"},{"date" => "2014-04-18T13:35:51","version" => "2.05"},{"date" => "2014-08-06T19:32:53","version" => "2.06-TRIAL"},{"date" => "2015-01-04T18:54:54","version" => "2.06-TRIAL"},{"date" => "2015-01-05T06:31:55","version" => "2.08-TRIAL"},{"date" => "2015-02-02T04:41:02","version" => "2.09-TRIAL"},{"date" => "2015-02-22T15:57:42","version" => "2.10-TRIAL"},{"date" => "2015-03-13T07:45:04","version" => "2.10"},{"date" => "2015-12-31T11:00:08","version" => "2.12-TRIAL"},{"date" => "2016-05-16T09:56:01","version" => "2.13-TRIAL"},{"date" => "2016-06-04T14:41:28","version" => "2.14-TRIAL"},{"date" => "2016-06-25T04:32:45","version" => "2.14"},{"date" => "2016-07-17T12:10:30","version" => "2.15-TRIAL"},{"date" => "2017-01-16T16:20:27","version" => "2.16-TRIAL"},{"date" => "2017-01-16T21:27:06","version" => "2.16-TRIAL2"},{"date" => "2017-02-14T16:22:20","version" => "2.16"},{"date" => "2017-02-15T09:37:10","version" => "2.17-TRIAL"},{"date" => "2017-02-16T09:48:46","version" => "2.17-TRIAL2"},{"date" => "2017-03-30T21:38:23","version" => "2.18-TRIAL"},{"date" => "2017-11-04T23:27:47","version" => "2.19-TRIAL"},{"date" => "2017-11-26T22:10:39","version" => "2.20-TRIAL"},{"date" => "2018-09-22T20:46:35","version" => "2.21-TRIAL"},{"date" => "2018-12-16T10:35:04","version" => "2.22-TRIAL"},{"date" => "2018-12-23T09:11:29","version" => "2.22"},{"date" => "2019-02-10T20:28:53","version" => "2.23-TRIAL"},{"date" => "2019-02-14T21:21:03","version" => "2.24-TRIAL"},{"date" => "2019-02-16T05:56:23","version" => "2.25-TRIAL"},{"date" => "2019-03-03T06:27:10","version" => "2.25"},{"date" => "2019-03-19T00:04:34","version" => "2.26"},{"date" => "2019-05-31T21:11:50","version" => "2.27-TRIAL"},{"date" => "2019-06-09T05:48:20","version" => "2.27-TRIAL2"},{"date" => "2019-07-03T20:15:40","version" => "2.27"},{"date" => "2020-04-03T02:52:43","version" => "2.28-TRIAL"},{"date" => "2020-06-13T04:57:39","version" => "2.28"},{"date" => "2021-11-23T16:58:45","version" => "2.29"},{"date" => "2021-12-12T09:16:03","version" => "2.30-TRIAL"},{"date" => "2021-12-14T20:52:30","version" => "2.31-TRIAL"},{"date" => "2021-12-26T21:35:55","version" => "2.32-TRIAL"},{"date" => "2022-01-21T04:09:07","version" => "2.33-TRIAL"},{"date" => "2022-03-27T17:53:47","version" => "2.33"},{"date" => "2022-04-03T19:19:13","version" => "2.34-TRIAL"},{"date" => "2022-04-17T17:40:25","version" => "2.34"},{"date" => "2023-04-15T14:44:27","version" => "2.35-TRIAL"},{"date" => "2023-04-27T13:05:07","version" => "2.35"},{"date" => "2023-05-10T07:08:30","version" => "2.36-TRIAL"},{"date" => "2023-05-14T19:36:11","version" => "2.36"},{"date" => "2024-08-18T17:03:50","version" => "2.37-TRIAL"},{"date" => "2024-08-30T17:18:31","version" => "2.37"},{"date" => "2024-10-03T10:21:39","version" => "2.38-TRIAL"},{"date" => "2024-10-13T11:32:53","version" => "2.38-TRIAL2"},{"date" => "2024-11-17T19:52:34","version" => "2.38"},{"date" => "1997-05-15T00:00:00","dual_lived" => 1,"perl_release" => "5.004","version" => "1.2401"},{"date" => "1998-07-22T00:00:00","dual_lived" => 1,"perl_release" => "5.005","version" => "1.3901"},{"date" => "2002-03-05T00:00:00","dual_lived" => 1,"perl_release" => "5.007003","version" => "1.59_56"},{"date" => "2006-01-31T00:00:00","dual_lived" => 1,"perl_release" => "5.008008","version" => "1.76_02"},{"date" => "2009-12-20T00:00:00","dual_lived" => 1,"perl_release" => "5.011003","version" => "1.94_5301"},{"date" => "2011-09-26T00:00:00","dual_lived" => 1,"perl_release" => "5.014002","version" => "1.9600_01"},{"date" => "2013-09-20T00:00:00","dual_lived" => 1,"perl_release" => "5.019004","version" => "2.03"},{"date" => "2014-03-20T00:00:00","dual_lived" => 1,"perl_release" => "5.019010","version" => "2.04"},{"date" => "2015-04-20T00:00:00","dual_lived" => 1,"perl_release" => "5.021011","version" => "2.11"},{"date" => "2017-01-14T00:00:00","dual_lived" => 1,"perl_release" => "5.022003","version" => "2.11_01"},{"date" => "2016-08-20T00:00:00","dual_lived" => 1,"perl_release" => "5.025004","version" => "2.14_01"},{"date" => "2017-02-20T00:00:00","dual_lived" => 1,"perl_release" => "5.02501","version" => "2.17"},{"date" => "2017-04-20T00:00:00","dual_lived" => 1,"perl_release" => "5.025012","version" => "2.18"},{"date" => "2017-12-20T00:00:00","dual_lived" => 1,"perl_release" => "5.027007","version" => "2.20"},{"date" => "2018-10-20T00:00:00","dual_lived" => 1,"perl_release" => "5.029004","version" => "2.21"}]},"CPAN-Checksums" => {"advisories" => [{"affected_versions" => ["<=2.12"],"cves" => ["CVE-2020-16155"],"description" => "The CPAN::Checksums package 2.12 for Perl does not uniquely define signed data.\n","distribution" => "CPAN-Checksums","fixed_versions" => [">=2.13"],"id" => "CPANSA-CPAN-Checksums-2020-16155","references" => ["https://metacpan.org/pod/CPAN::Checksums","https://blog.hackeriet.no/cpan-signature-verification-vulnerabilities/"],"reported" => "2021-12-13","severity" => "medium"}],"main_module" => "CPAN::Checksums","versions" => [{"date" => "2000-12-10T09:31:19","version" => "1.003"},{"date" => "2000-12-11T08:49:05","version" => "1.004"},{"date" => "2000-12-11T10:20:16","version" => "1.006"},{"date" => "2000-12-11T13:38:18","version" => "1.007"},{"date" => "2000-12-13T11:24:09","version" => "1.008"},{"date" => "2002-03-31T20:27:49","version" => "1.009"},{"date" => "2002-10-06T17:22:35","version" => "1.0"},{"date" => "2002-10-07T08:56:12","version" => "1.011"},{"date" => "2003-02-02T10:26:00","version" => "1.014"},{"date" => "2003-02-03T15:44:57","version" => "1.015"},{"date" => "2003-02-03T19:21:46","version" => "1.016"},{"date" => "2005-01-24T07:59:41","version" => "1.018"},{"date" => "2005-10-31T07:27:02","version" => "1.032"},{"date" => "2005-11-11T07:16:04","version" => "1.039"},{"date" => "2006-05-01T13:34:41","version" => "1.048"},{"date" => "2006-05-09T03:30:39","version" => "1.050"},{"date" => "2007-08-05T12:10:58","version" => "1.061"},{"date" => "2007-10-09T03:09:45","version" => "1.064"},{"date" => "2008-05-17T05:26:24","version" => "2.00"},{"date" => "2008-09-03T19:33:28","version" => "2.01"},{"date" => "2008-10-31T06:54:59","version" => "2.02"},{"date" => "2009-09-20T01:50:36","version" => "2.03"},{"date" => "2009-09-28T04:10:09","version" => "2.04"},{"date" => "2010-01-23T05:39:17","version" => "2.05"},{"date" => "2010-10-24T12:13:44","version" => "2.06"},{"date" => "2010-11-20T22:18:39","version" => "2.07"},{"date" => "2011-08-30T06:32:02","version" => "2.08"},{"date" => "2014-04-04T04:06:11","version" => "2.09"},{"date" => "2015-04-11T05:48:38","version" => "2.10"},{"date" => "2016-04-09T05:42:27","version" => "2.11"},{"date" => "2016-06-14T02:42:03","version" => "2.12"},{"date" => "2021-11-23T16:57:18","version" => "2.13"},{"date" => "2021-12-04T10:00:42","version" => "2.14"}]},"Capture-Tiny" => {"advisories" => [{"affected_versions" => ["<0.24"],"cves" => ["CVE-2014-1875"],"description" => "The Capture::Tiny module before 0.24 for Perl allows local users to write to arbitrary files via a symlink attack on a temporary file.\n","distribution" => "Capture-Tiny","fixed_versions" => [">=0.24"],"id" => "CPANSA-Capture-Tiny-2014-1875","references" => ["http://osvdb.org/102963","https://bugzilla.redhat.com/show_bug.cgi?id=1062424","http://www.securityfocus.com/bid/65475","https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=737835","https://github.com/dagolden/Capture-Tiny/issues/16","http://lists.fedoraproject.org/pipermail/package-announce/2014-February/128882.html","https://github.com/dagolden/Capture-Tiny/commit/635c9eabd52ab8042b0c841823bd6e692de87924","http://cpansearch.perl.org/src/DAGOLDEN/Capture-Tiny-0.24/Changes","http://lists.fedoraproject.org/pipermail/package-announce/2014-February/128823.html","http://seclists.org/oss-sec/2014/q1/272","http://seclists.org/oss-sec/2014/q1/267","http://secunia.com/advisories/56823","https://exchange.xforce.ibmcloud.com/vulnerabilities/91464"],"reported" => "2014-10-06","severity" => undef}],"main_module" => "Capture::Tiny","versions" => [{"date" => "2009-02-14T04:25:26","version" => "0.01"},{"date" => "2009-02-17T22:26:18","version" => "0.02"},{"date" => "2009-02-20T18:09:46","version" => "0.03"},{"date" => "2009-02-25T14:29:32","version" => "0.04"},{"date" => "2009-03-03T11:58:12","version" => "0.05"},{"date" => "2009-04-21T11:07:47","version" => "0.05_51"},{"date" => "2009-05-07T10:57:33","version" => "0.06"},{"date" => "2010-01-24T05:21:56","version" => "0.07"},{"date" => "2010-06-20T23:17:16","version" => "0.08"},{"date" => "2011-01-28T04:53:00","version" => "0.09"},{"date" => "2011-02-07T12:02:15","version" => "0.10"},{"date" => "2011-05-20T03:35:28","version" => "0.11"},{"date" => "2011-12-01T22:00:04","version" => "0.12"},{"date" => "2011-12-02T18:40:05","version" => "0.13"},{"date" => "2011-12-22T15:16:31","version" => "0.14"},{"date" => "2011-12-23T16:12:30","version" => "0.15"},{"date" => "2012-02-13T02:06:15","version" => "0.16"},{"date" => "2012-02-22T13:09:42","version" => "0.17"},{"date" => "2012-03-07T23:25:31","version" => "0.17_51"},{"date" => "2012-03-09T16:46:53","version" => "0.17_52"},{"date" => "2012-05-04T20:33:43","version" => "0.18"},{"date" => "2012-08-07T00:28:08","version" => "0.19"},{"date" => "2012-09-19T17:22:24","version" => "0.20"},{"date" => "2012-11-15T00:13:08","version" => "0.21"},{"date" => "2013-03-27T19:52:10","version" => "0.22"},{"date" => "2013-10-20T15:28:15","version" => "0.23"},{"date" => "2014-02-06T22:18:06","version" => "0.24"},{"date" => "2014-08-16T14:09:48","version" => "0.25"},{"date" => "2014-11-04T11:57:19","version" => "0.26"},{"date" => "2014-11-05T04:12:33","version" => "0.27"},{"date" => "2015-02-11T11:41:44","version" => "0.28"},{"date" => "2015-04-19T16:44:50","version" => "0.29"},{"date" => "2015-05-16T00:45:01","version" => "0.30"},{"date" => "2016-02-14T14:39:55","version" => "0.31"},{"date" => "2016-02-18T15:14:06","version" => "0.32"},{"date" => "2016-02-19T04:29:41","version" => "0.34"},{"date" => "2016-02-29T02:38:12","version" => "0.36"},{"date" => "2016-05-02T11:09:27","version" => "0.37"},{"date" => "2016-05-02T14:24:23","version" => "0.39"},{"date" => "2016-05-23T15:45:16","version" => "0.40"},{"date" => "2016-05-23T16:01:05","version" => "0.41"},{"date" => "2016-05-31T16:41:30","version" => "0.42"},{"date" => "2016-08-05T18:02:43","version" => "0.44"},{"date" => "2017-02-23T18:32:44","version" => "0.45"},{"date" => "2017-02-25T19:26:54","version" => "0.46"},{"date" => "2017-07-26T14:36:03","version" => "0.47"},{"date" => "2018-04-22T07:09:08","version" => "0.48"},{"date" => "2024-12-16T13:11:27","version" => "0.49"},{"date" => "2024-12-19T13:16:05","version" => "0.50"}]},"Catalyst-Action-REST" => {"advisories" => [{"affected_versions" => ["<1.12"],"cves" => [],"description" => "YAML and YAML::HTML parsers are a potential security hole, as they may allow arbitrary Perl objects to be instantiated.\n","distribution" => "Catalyst-Action-REST","fixed_versions" => [">=1.12"],"id" => "CPANSA-Catalyst-Action-REST-2013-01","references" => ["https://metacpan.org/dist/Catalyst-Action-REST/changes"],"reported" => "2013-09-03","severity" => undef}],"main_module" => "Catalyst::Action::REST","versions" => [{"date" => "2006-11-20T03:15:08","version" => "0.1"},{"date" => "2006-12-01T01:42:22","version" => "0.2"},{"date" => "2006-12-04T00:22:45","version" => "0.30"},{"date" => "2006-12-06T08:48:49","version" => "0.31"},{"date" => "2007-03-10T00:44:35","version" => "0.40"},{"date" => "2007-05-24T21:09:40","version" => "0.41"},{"date" => "2007-07-07T19:33:22","version" => "0.50"},{"date" => "2008-01-04T01:33:04","version" => "0.60"},{"date" => "2008-06-30T19:30:56","version" => "0.61"},{"date" => "2008-07-02T15:25:10","version" => "0.62"},{"date" => "2008-08-07T17:14:34","version" => "0.63"},{"date" => "2008-08-14T16:09:53","version" => "0.64"},{"date" => "2008-08-20T17:45:46","version" => "0.65"},{"date" => "2008-08-22T18:24:57","version" => "0.66"},{"date" => "2009-03-25T16:38:07","version" => "0.67_01"},{"date" => "2009-03-26T05:04:33","version" => "0.67"},{"date" => "2009-03-26T05:37:53","version" => "0.68"},{"date" => "2009-03-26T21:19:43","version" => "0.69"},{"date" => "2009-03-28T06:23:19","version" => "0.70"},{"date" => "2009-03-28T16:19:10","version" => "0.71"},{"date" => "2009-06-25T18:56:47","version" => "0.72"},{"date" => "2009-06-28T00:22:51","version" => "0.73"},{"date" => "2009-07-22T23:12:44","version" => "0.74"},{"date" => "2009-08-17T13:11:15","version" => "0.75"},{"date" => "2009-08-21T20:42:44","version" => "0.76"},{"date" => "2009-08-27T01:26:49","version" => "0.77"},{"date" => "2009-09-28T14:05:11","version" => "0.78"},{"date" => "2009-12-11T01:11:49","version" => "0.79"},{"date" => "2009-12-19T14:59:13","version" => "0.80"},{"date" => "2010-01-14T20:56:56","version" => "0.81"},{"date" => "2010-02-04T22:35:05","version" => "0.82"},{"date" => "2010-02-08T22:24:29","version" => "0.83"},{"date" => "2010-05-06T08:34:09","version" => "0.84"},{"date" => "2010-05-13T08:15:30","version" => "0.85"},{"date" => "2010-09-01T22:17:14","version" => "0.86"},{"date" => "2010-11-03T19:48:23","version" => "0.87"},{"date" => "2011-01-11T23:12:42","version" => "0.88"},{"date" => "2011-01-24T21:59:02","version" => "0.89"},{"date" => "2011-02-25T13:58:06","version" => "0.90"},{"date" => "2011-08-04T12:46:05","version" => "0.91"},{"date" => "2011-10-01T10:11:59","version" => "0.91"},{"date" => "2011-10-12T18:35:31","version" => "0.93"},{"date" => "2011-12-09T08:51:25","version" => "0.94"},{"date" => "2012-01-04T19:54:14","version" => "0.95"},{"date" => "2012-01-30T11:32:44","version" => "0.96"},{"date" => "2012-02-21T10:06:13","version" => "0.97"},{"date" => "2012-02-21T11:44:32","version" => "0.98"},{"date" => "2012-02-28T09:14:17","version" => "0.99"},{"date" => "2012-04-13T08:37:31","version" => "1.00"},{"date" => "2012-05-29T20:02:44","version" => "1.01"},{"date" => "2012-06-05T21:45:05","version" => "1.02"},{"date" => "2012-06-27T23:52:31","version" => "1.03"},{"date" => "2012-06-30T09:32:24","version" => "1.04"},{"date" => "2012-07-02T19:16:30","version" => "1.05"},{"date" => "2012-12-11T22:13:48","version" => "1.06"},{"date" => "2013-04-11T19:25:51","version" => "1.07"},{"date" => "2013-04-16T07:36:02","version" => "1.08"},{"date" => "2013-04-19T12:43:57","version" => "1.09"},{"date" => "2013-04-22T13:43:09","version" => "1.10"},{"date" => "2013-06-16T14:28:48","version" => "1.11"},{"date" => "2013-09-11T17:50:54","version" => "1.12"},{"date" => "2013-11-08T15:49:54","version" => "1.13"},{"date" => "2013-12-27T21:33:09","version" => "1.14"},{"date" => "2014-05-07T14:04:03","version" => "1.15"},{"date" => "2014-09-12T18:22:35","version" => "1.16"},{"date" => "2014-10-24T00:59:39","version" => "1.17"},{"date" => "2015-01-20T18:22:02","version" => "1.18"},{"date" => "2015-02-06T15:40:56","version" => "1.19"},{"date" => "2015-10-29T20:34:02","version" => "1.20"},{"date" => "2017-12-05T15:16:47","version" => "1.21"}]},"Catalyst-Authentication-Credential-HTTP" => {"advisories" => [{"affected_versions" => ["<=1.018"],"cves" => ["CVE-2025-40920"],"description" => "Catalyst::Authentication::Credential::HTTP versions 1.018 and earlier for Perl generate nonces using the Perl Data::UUID library.   *  Data::UUID does not use a strong cryptographic source for generating UUIDs.   *  Data::UUID returns v3 UUIDs, which are generated from known information and are unsuitable for security, as per RFC 9562.   *  The nonces should be generated from a strong cryptographic source, as per RFC 7616.","distribution" => "Catalyst-Authentication-Credential-HTTP","fixed_versions" => [">=1.019"],"id" => "CPANSA-Catalyst-Authentication-Credential-HTTP-2025-40920","references" => ["https://datatracker.ietf.org/doc/html/rfc7616#section-5.12","https://datatracker.ietf.org/doc/html/rfc9562#name-security-considerations","https://github.com/perl-catalyst/Catalyst-Authentication-Credential-HTTP/pull/1","https://metacpan.org/release/ETHER/Catalyst-Authentication-Credential-HTTP-1.018/source/lib/Catalyst/Authentication/Credential/HTTP.pm#L391","https://security.metacpan.org/patches/C/Catalyst-Authentication-Credential-HTTP/1.018/CVE-2025-40920-r1.patch"],"reported" => "2025-08-11","severity" => undef}],"main_module" => "Catalyst::Authentication::Credential::HTTP","versions" => [{"date" => "2008-09-01T13:41:15","version" => "1.000"},{"date" => "2008-09-02T18:15:58","version" => "1.001"},{"date" => "2008-09-03T00:16:26","version" => "1.002"},{"date" => "2008-09-11T14:35:17","version" => "1.003"},{"date" => "2008-09-11T18:06:53","version" => "0.12"},{"date" => "2008-09-12T18:21:26","version" => "1.004"},{"date" => "2008-09-25T22:13:58","version" => "1.005"},{"date" => "2008-10-06T18:56:06","version" => "1.006"},{"date" => "2008-11-19T09:41:15","version" => "1.007"},{"date" => "2008-12-10T23:58:04","version" => "1.008"},{"date" => "2009-01-04T21:37:39","version" => "1.009"},{"date" => "2009-05-14T08:34:09","version" => "1.010"},{"date" => "2009-06-27T04:00:10","version" => "1.011"},{"date" => "2010-03-07T21:07:20","version" => "1.012"},{"date" => "2010-12-14T22:03:35","version" => "1.013"},{"date" => "2012-02-05T18:51:03","version" => "1.014"},{"date" => "2012-06-27T18:43:56","version" => "1.015"},{"date" => "2013-07-27T20:38:37","version" => "1.016"},{"date" => "2017-06-27T23:22:26","version" => "1.017"},{"date" => "2017-06-28T00:29:58","version" => "1.018"},{"date" => "2025-08-20T17:38:38","version" => "1.019"}]},"Catalyst-Authentication-Store-LDAP" => {"advisories" => [{"affected_versions" => ["<1.013"],"cves" => [],"description" => "Incorrect password check binds to the unauthenticated user.\n","distribution" => "Catalyst-Authentication-Store-LDAP","fixed_versions" => [">=1.013"],"id" => "CPANSA-Catalyst-Authentication-Store-LDAP-2012-01","references" => ["https://rt.cpan.org/Public/Bug/Display.html?id=81908"],"reported" => "2012-12-11","severity" => "high"}],"main_module" => "Catalyst::Authentication::Store::LDAP","versions" => [{"date" => "2008-02-05T18:18:24","version" => "0.1000"},{"date" => "2008-04-10T02:06:58","version" => "0.1001"},{"date" => "2008-07-09T20:45:43","version" => "0.1002"},{"date" => "2008-09-10T13:21:33","version" => "0.1003"},{"date" => "2008-10-22T01:57:27","version" => "0.1003"},{"date" => "2009-05-01T02:34:18","version" => "0.1005"},{"date" => "2009-12-11T18:54:26","version" => "1.006"},{"date" => "2010-03-19T10:07:13","version" => "1.007"},{"date" => "2010-04-03T03:04:13","version" => "1.008"},{"date" => "2010-05-15T07:14:41","version" => "1.009"},{"date" => "2010-07-06T21:39:55","version" => "1.010"},{"date" => "2010-07-07T20:41:22","version" => "1.011"},{"date" => "2010-10-05T08:11:56","version" => "1.012"},{"date" => "2013-01-09T14:58:46","version" => "1.013"},{"date" => "2013-04-26T19:51:28","version" => "1.014"},{"date" => "2015-02-20T18:07:31","version" => "1.015"},{"date" => "2016-02-11T17:50:52","version" => "1.016"},{"date" => "2021-05-26T09:59:28","version" => "1.017"}]},"Catalyst-Controller-Combine" => {"advisories" => [{"affected_versions" => ["<0.12"],"cves" => [],"description" => "Allows to use url-encoded path-parts to crawl along the file system and read files outside the intended directory.\n","distribution" => "Catalyst-Controller-Combine","fixed_versions" => [">=0.12"],"id" => "CPANSA-Catalyst-Controller-Combine-2010-01","references" => ["https://metacpan.org/changes/distribution/Catalyst-Controller-Combine"],"reported" => "2010-05-21"}],"main_module" => "Catalyst::Controller::Combine","versions" => [{"date" => "2009-07-11T17:58:25","version" => "0.06"},{"date" => "2009-07-13T06:49:00","version" => "0.07"},{"date" => "2009-10-24T12:48:21","version" => "0.08"},{"date" => "2010-03-13T19:31:13","version" => "0.09"},{"date" => "2010-03-27T18:44:05","version" => "0.10"},{"date" => "2010-06-21T20:47:02","version" => "0.12"},{"date" => "2011-07-28T19:53:12","version" => "0.13"},{"date" => "2012-02-20T20:59:00","version" => "0.14"},{"date" => "2012-05-04T10:43:12","version" => "0.15"}]},"Catalyst-Plugin-Session" => {"advisories" => [{"affected_versions" => ["<0.44"],"cves" => ["CVE-2025-40924"],"description" => "Catalyst::Plugin::Session before version 0.44 for Perl generates session ids insecurely.  The session id is generated from a (usually SHA-1) hash of a simple counter, the epoch time, the built-in rand function, the PID and the current Catalyst context. This information is of low entropy. The PID will come from a small set of numbers, and the epoch time may be guessed, if it is not leaked from the HTTP Date header. The built-in rand function is unsuitable for cryptographic usage.  Predicable session ids could allow an attacker to gain access to systems.","distribution" => "Catalyst-Plugin-Session","fixed_versions" => [">=0.44"],"id" => "CPANSA-Catalyst-Plugin-Session-2025-40924","references" => ["https://github.com/perl-catalyst/Catalyst-Plugin-Session/commit/c0e2b4ab1e42ebce1008286db8c571b6ee98c22c.patch","https://github.com/perl-catalyst/Catalyst-Plugin-Session/pull/5","https://metacpan.org/release/HAARG/Catalyst-Plugin-Session-0.43/source/lib/Catalyst/Plugin/Session.pm#L632"],"reported" => "2025-07-17","severity" => undef}],"main_module" => "Catalyst::Plugin::Session","versions" => [{"date" => "2005-11-14T20:45:06","version" => "0.01"},{"date" => "2005-11-23T12:29:16","version" => "0.02"},{"date" => "2005-12-26T08:24:04","version" => "0.03"},{"date" => "2005-12-28T11:51:50","version" => "0.04"},{"date" => "2006-01-01T10:45:07","version" => "0.05"},{"date" => "2006-07-29T16:35:24","version" => "0.06"},{"date" => "2006-07-31T11:24:16","version" => "0.07"},{"date" => "2006-07-31T12:11:58","version" => "0.08"},{"date" => "2006-07-31T18:24:07","version" => "0.09"},{"date" => "2006-08-01T08:08:13","version" => "0.10"},{"date" => "2006-08-10T15:03:04","version" => "0.11"},{"date" => "2006-08-26T17:55:09","version" => "0.12"},{"date" => "2006-10-12T19:54:32","version" => "0.13"},{"date" => "2007-01-31T11:24:20","version" => "0.14"},{"date" => "2007-06-24T15:17:44","version" => "0.15"},{"date" => "2007-07-03T14:40:50","version" => "0.16"},{"date" => "2007-07-16T10:20:50","version" => "0.17"},{"date" => "2007-08-15T18:06:22","version" => "0.18"},{"date" => "2007-10-08T18:18:10","version" => "0.19"},{"date" => "2009-01-09T02:13:40","version" => "0.19_01"},{"date" => "2009-02-05T14:50:15","version" => "0.20"},{"date" => "2009-04-30T20:54:07","version" => "0.21"},{"date" => "2009-05-13T21:00:18","version" => "0.22"},{"date" => "2009-06-16T19:43:53","version" => "0.23"},{"date" => "2009-06-23T08:20:00","version" => "0.24"},{"date" => "2009-07-08T21:54:31","version" => "0.25"},{"date" => "2009-08-19T21:23:25","version" => "0.26"},{"date" => "2009-10-06T08:45:28","version" => "0.26_01"},{"date" => "2009-10-08T21:38:42","version" => "0.27"},{"date" => "2009-10-29T09:59:18","version" => "0.28"},{"date" => "2009-11-04T23:43:22","version" => "0.29"},{"date" => "2010-06-24T12:54:05","version" => "0.30"},{"date" => "2010-10-08T14:39:33","version" => "0.31"},{"date" => "2011-06-08T12:05:42","version" => "0.32"},{"date" => "2012-03-26T10:03:59","version" => "0.33"},{"date" => "2012-04-02T14:51:39","version" => "0.34"},{"date" => "2012-04-24T08:24:54","version" => "0.35"},{"date" => "2012-10-19T22:40:25","version" => "0.36"},{"date" => "2013-02-25T14:04:31","version" => "0.37"},{"date" => "2013-09-18T14:03:08","version" => "0.38"},{"date" => "2013-10-16T15:09:02","version" => "0.39"},{"date" => "2015-01-27T01:20:24","version" => "0.40"},{"date" => "2018-12-06T02:31:20","version" => "0.41"},{"date" => "2022-05-31T00:20:53","version" => "0.42"},{"date" => "2022-06-03T14:15:38","version" => "0.43"},{"date" => "2025-07-16T14:18:57","version" => "0.44"}]},"Catalyst-Plugin-Static" => {"advisories" => [{"affected_versions" => ["<0.10"],"cves" => [],"description" => "Serving files outside of \$config->{root} directory.\n","distribution" => "Catalyst-Plugin-Static","fixed_versions" => [">=0.10"],"id" => "CPANSA-Catalyst-Plugin-Static-2005-01","reported" => "2005-11-14"}],"main_module" => "Catalyst::Plugin::Static","versions" => [{"date" => "2005-01-29T00:00:20","version" => "0.01"},{"date" => "2005-02-19T20:28:50","version" => "0.02"},{"date" => "2005-03-17T01:01:03","version" => "0.03"},{"date" => "2005-03-17T19:10:36","version" => "0.04"},{"date" => "2005-03-21T13:34:27","version" => "0.05"},{"date" => "2005-03-23T06:48:05","version" => "0.05"},{"date" => "2005-04-15T16:58:18","version" => "0.06"},{"date" => "2005-04-17T14:50:45","version" => "0.07"},{"date" => "2005-09-06T13:42:42","version" => "0.08"},{"date" => "2005-11-14T08:38:35","version" => "0.09"},{"date" => "2005-11-14T10:26:31","version" => "0.10"},{"date" => "2009-10-18T18:13:00","version" => "0.11"}]},"Catalyst-Plugin-Static-Simple" => {"advisories" => [{"affected_versions" => ["<0.34"],"cves" => ["CVE-2017-16248"],"description" => "The Catalyst-Plugin-Static-Simple module before 0.34 for Perl allows remote attackers to read arbitrary files if there is a '.' character anywhere in the pathname, which differs from the intended policy of allowing access only when the filename itself has a '.' character.\n","distribution" => "Catalyst-Plugin-Static-Simple","fixed_versions" => [">=0.34"],"id" => "CPANSA-Catalyst-Plugin-Static-Simple-2017-01","references" => ["https://blog.hackeriet.no/cpan-signature-verification-vulnerabilities/","https://metacpan.org/pod/CPAN::Checksums"],"reported" => "2017-10-31"}],"main_module" => "Catalyst::Plugin::Static::Simple","versions" => [{"date" => "2005-08-12T01:37:04","version" => "0.01"},{"date" => "2005-08-16T22:09:54","version" => "0.02"},{"date" => "2005-08-22T03:44:24","version" => "0.03"},{"date" => "2005-08-22T15:59:08","version" => "0.04"},{"date" => "2005-08-26T15:58:06","version" => "0.05"},{"date" => "2005-09-05T19:36:58","version" => "0.06"},{"date" => "2005-09-06T01:07:28","version" => "0.07"},{"date" => "2005-09-07T22:52:21","version" => "0.08"},{"date" => "2005-10-07T17:40:16","version" => "0.09"},{"date" => "2005-10-19T21:19:04","version" => "0.10"},{"date" => "2005-11-14T00:28:01","version" => "0.11"},{"date" => "2005-12-15T14:56:40","version" => "0.13"},{"date" => "2006-03-24T16:18:59","version" => "0.14"},{"date" => "2006-07-05T16:35:54","version" => "0.14"},{"date" => "2006-12-09T03:25:57","version" => "0.15"},{"date" => "2007-04-30T18:48:25","version" => "0.16"},{"date" => "2007-05-11T14:52:16","version" => "0.17"},{"date" => "2007-07-01T04:12:31","version" => "0.18"},{"date" => "2007-07-02T20:54:05","version" => "0.19"},{"date" => "2007-09-24T13:50:15","version" => "0.20"},{"date" => "2009-03-29T18:47:56","version" => "0.21"},{"date" => "2009-08-21T16:21:17","version" => "0.22"},{"date" => "2009-10-06T16:51:19","version" => "0.23"},{"date" => "2009-10-18T18:12:48","version" => "0.24"},{"date" => "2009-10-22T20:49:26","version" => "0.25"},{"date" => "2009-12-06T12:32:46","version" => "0.26"},{"date" => "2010-01-03T14:56:26","version" => "0.27"},{"date" => "2010-01-04T13:18:25","version" => "0.28"},{"date" => "2010-02-01T18:48:45","version" => "0.29"},{"date" => "2012-05-04T17:17:29","version" => "0.30"},{"date" => "2013-09-09T14:32:43","version" => "0.31"},{"date" => "2014-06-05T12:44:48","version" => "0.32"},{"date" => "2014-10-29T16:02:17","version" => "0.33"},{"date" => "2017-08-02T17:00:14","version" => "0.34"},{"date" => "2018-03-14T12:13:30","version" => "0.35"},{"date" => "2018-03-15T11:41:17","version" => "0.36"},{"date" => "2021-05-05T14:30:07","version" => "0.37"}]},"Catalyst-Runtime" => {"advisories" => [{"affected_versions" => ["<5.90020"],"cves" => [],"description" => "Passing a special host to the redirect page link makes it vulnerable to XSS attack.\n","distribution" => "Catalyst-Runtime","fixed_versions" => [">=5.90020"],"id" => "CPANSA-Catalyst-Runtime-2013-01","references" => ["http://git.shadowcat.co.uk/gitweb/gitweb.cgi?p=catagits/Catalyst-Runtime.git;a=commitdiff;h=7af54927870a7c6f89323ac1876d49f92e7841f5"],"reported" => "2013-01-23"},{"affected_versions" => ["<5.58"],"cves" => [],"description" => "Path traversal in Static::Simple plugin.\n","distribution" => "Catalyst-Runtime","fixed_versions" => [">=5.58"],"id" => "CPANSA-Catalyst-Runtime-2005-01","reported" => "2005-11-24"}],"main_module" => "Catalyst::Runtime","versions" => [{"date" => "2006-06-25T19:20:53","version" => "5.70_01"},{"date" => "2006-06-27T18:10:20","version" => "5.70_02"},{"date" => "2006-06-28T21:50:30","version" => "5.70_03"},{"date" => "2006-07-07T22:47:30","version" => "5.7000"},{"date" => "2006-07-20T06:00:58","version" => "5.7001"},{"date" => "2006-09-19T07:36:29","version" => "5.7002"},{"date" => "2006-09-23T17:43:12","version" => "5.7003"},{"date" => "2006-11-06T23:28:40","version" => "5.7004"},{"date" => "2006-11-07T19:43:56","version" => "5.7005"},{"date" => "2006-11-15T08:27:59","version" => "5.7006"},{"date" => "2007-03-14T11:13:37","version" => "5.7007"},{"date" => "2007-08-13T06:36:11","version" => "5.7008"},{"date" => "2007-08-21T22:23:53","version" => "5.7009"},{"date" => "2007-08-22T05:51:41","version" => "5.7010"},{"date" => "2007-10-18T18:11:24","version" => "5.7011"},{"date" => "2007-12-17T08:19:28","version" => "5.7012"},{"date" => "2008-05-17T12:41:13","version" => "5.7013"},{"date" => "2008-05-25T21:16:45","version" => "5.7013"},{"date" => "2008-06-25T20:43:41","version" => "5.7099_01"},{"date" => "2008-07-18T11:41:25","version" => "5.7099_02"},{"date" => "2008-07-20T08:15:02","version" => "5.7099_02"},{"date" => "2008-10-13T20:55:41","version" => "5.7099_02"},{"date" => "2008-10-14T06:06:06","version" => "5.7099_02"},{"date" => "2008-10-15T21:44:15","version" => "5.7015"},{"date" => "2008-10-17T12:42:53","version" => "5.8000_03"},{"date" => "2008-12-05T15:11:14","version" => "5.8000_04"},{"date" => "2009-01-12T15:46:59","version" => "5.7099_04"},{"date" => "2009-01-19T17:36:04","version" => "5.71000"},{"date" => "2009-01-29T08:56:09","version" => "5.8000_05"},{"date" => "2009-02-04T20:08:22","version" => "5.8000_06"},{"date" => "2009-03-27T09:21:12","version" => "5.71001"},{"date" => "2009-04-13T19:03:36","version" => "5.8000_07"},{"date" => "2009-04-18T20:26:00","version" => "5.80001"},{"date" => "2009-04-21T23:45:45","version" => "5.80002"},{"date" => "2009-04-29T14:39:21","version" => "5.80003"},{"date" => "2009-05-18T15:16:38","version" => "5.80004"},{"date" => "2009-06-06T12:49:15","version" => "5.80005"},{"date" => "2009-06-29T22:11:48","version" => "5.80006"},{"date" => "2009-06-30T22:11:36","version" => "5.80007"},{"date" => "2009-08-21T16:14:33","version" => "5.80008"},{"date" => "2009-08-21T20:29:33","version" => "5.80009"},{"date" => "2009-08-21T21:42:08","version" => "5.80010"},{"date" => "2009-08-23T11:57:26","version" => "5.80011"},{"date" => "2009-09-09T18:01:32","version" => "5.80012"},{"date" => "2009-09-17T09:35:20","version" => "5.80013"},{"date" => "2009-11-21T02:32:20","version" => "5.80014"},{"date" => "2009-11-22T20:24:47","version" => "5.80014_01"},{"date" => "2009-12-01T01:14:00","version" => "5.80014_02"},{"date" => "2009-12-02T15:42:50","version" => "5.80015"},{"date" => "2009-12-11T23:37:44","version" => "5.80016"},{"date" => "2010-01-10T02:01:50","version" => "5.80017"},{"date" => "2010-01-12T21:39:47","version" => "5.80018"},{"date" => "2010-01-29T00:18:07","version" => "5.80019"},{"date" => "2010-02-04T06:19:31","version" => "5.80020"},{"date" => "2010-03-03T23:16:29","version" => "5.80021"},{"date" => "2010-03-28T19:25:48","version" => "5.80022"},{"date" => "2010-05-07T22:07:08","version" => "5.80023"},{"date" => "2010-05-15T09:57:52","version" => "5.80024"},{"date" => "2010-07-29T00:59:16","version" => "5.80025"},{"date" => "2010-09-01T15:10:42","version" => "5.80026"},{"date" => "2010-09-02T11:33:03","version" => "5.80027"},{"date" => "2010-09-28T20:14:11","version" => "5.80028"},{"date" => "2010-10-03T16:24:08","version" => "5.80029"},{"date" => "2011-01-04T12:56:30","version" => "5.80030"},{"date" => "2011-01-24T10:50:27","version" => "5.89000"},{"date" => "2011-01-31T08:25:21","version" => "5.80031"},{"date" => "2011-02-23T08:28:58","version" => "5.80032"},{"date" => "2011-03-01T14:56:37","version" => "5.89001"},{"date" => "2011-03-02T10:37:42","version" => "5.89002"},{"date" => "2011-07-24T15:58:37","version" => "5.80033"},{"date" => "2011-07-28T20:05:01","version" => "5.89003"},{"date" => "2011-08-15T21:35:34","version" => "5.9000"},{"date" => "2011-08-15T21:59:58","version" => "5.90001"},{"date" => "2011-08-22T20:55:10","version" => "5.90002"},{"date" => "2011-10-05T07:48:57","version" => "5.90003"},{"date" => "2011-10-11T15:19:05","version" => "5.90004"},{"date" => "2011-10-22T21:01:24","version" => "5.90005"},{"date" => "2011-10-25T17:54:34","version" => "5.90006"},{"date" => "2011-11-22T20:40:44","version" => "5.90007"},{"date" => "2012-02-06T21:08:28","version" => "5.90008"},{"date" => "2012-02-16T09:29:44","version" => "5.90009"},{"date" => "2012-02-18T00:49:30","version" => "5.90010"},{"date" => "2012-03-08T21:53:00","version" => "5.90011"},{"date" => "2012-05-19T07:13:21","version" => "5.90012"},{"date" => "2012-06-08T00:37:40","version" => "5.90013"},{"date" => "2012-06-21T20:41:41","version" => "5.90013"},{"date" => "2012-06-26T14:34:56","version" => "5.90014"},{"date" => "2012-06-30T18:00:53","version" => "5.90015"},{"date" => "2012-08-17T01:39:42","version" => "5.90016"},{"date" => "2012-10-19T21:51:54","version" => "5.90017"},{"date" => "2012-10-24T01:01:44","version" => "5.90018"},{"date" => "2012-12-04T22:04:19","version" => "5.90019"},{"date" => "2013-02-22T14:05:39","version" => "5.90020"},{"date" => "2013-04-12T17:09:27","version" => "5.90030"},{"date" => "2013-06-12T21:26:14","version" => "5.90040"},{"date" => "2013-06-15T02:10:17","version" => "5.90041"},{"date" => "2013-06-16T01:57:47","version" => "5.90042"},{"date" => "2013-07-26T19:13:01","version" => "5.90049_001"},{"date" => "2013-08-21T02:39:45","version" => "5.90049_002"},{"date" => "2013-09-20T19:03:54","version" => "5.90049_003"},{"date" => "2013-10-18T22:19:33","version" => "5.90049_004"},{"date" => "2013-10-31T20:48:42","version" => "5.90049_005"},{"date" => "2013-11-05T03:25:31","version" => "5.90049_006"},{"date" => "2013-11-05T22:35:22","version" => "5.90050"},{"date" => "2013-11-07T17:14:35","version" => "5.90051"},{"date" => "2013-12-18T20:03:22","version" => "5.90052"},{"date" => "2013-12-19T14:33:08","version" => "5.90059_001"},{"date" => "2013-12-22T16:18:16","version" => "5.90053"},{"date" => "2013-12-22T16:34:11","version" => "5.90059_002"},{"date" => "2013-12-27T02:27:08","version" => "5.90059_003"},{"date" => "2014-01-27T17:20:51","version" => "5.90059_004"},{"date" => "2014-01-28T19:36:58","version" => "5.90059_005"},{"date" => "2014-02-06T20:41:25","version" => "5.90059_006"},{"date" => "2014-02-08T03:11:11","version" => "5.90060"},{"date" => "2014-03-10T14:46:10","version" => "5.90061"},{"date" => "2014-04-14T18:53:26","version" => "5.90062"},{"date" => "2014-05-02T00:15:16","version" => "5.90063"},{"date" => "2014-05-05T14:55:25","version" => "5.90064"},{"date" => "2014-05-27T18:08:08","version" => "5.90069_001"},{"date" => "2014-06-05T12:44:59","version" => "5.90065"},{"date" => "2014-06-10T00:22:42","version" => "5.90069_002"},{"date" => "2014-08-06T15:09:29","version" => "5.90069_003"},{"date" => "2014-08-07T15:59:15","version" => "5.90069_004"},{"date" => "2014-08-07T21:49:59","version" => "5.90070"},{"date" => "2014-08-10T13:15:52","version" => "5.90071"},{"date" => "2014-09-15T16:30:58","version" => "5.90072"},{"date" => "2014-09-23T17:24:54","version" => "5.90073"},{"date" => "2014-10-01T21:45:12","version" => "5.90074"},{"date" => "2014-10-07T00:07:51","version" => "5.90075"},{"date" => "2014-11-14T00:20:16","version" => "5.90076"},{"date" => "2014-11-19T00:28:27","version" => "5.90077"},{"date" => "2014-12-02T21:50:30","version" => "5.90079_001"},{"date" => "2014-12-02T23:22:07","version" => "5.90079_002"},{"date" => "2014-12-03T19:45:16","version" => "5.90079_003"},{"date" => "2014-12-26T23:05:46","version" => "5.90079_004"},{"date" => "2014-12-31T16:26:20","version" => "5.90078"},{"date" => "2014-12-31T21:04:56","version" => "5.90079_005"},{"date" => "2015-01-02T15:11:55","version" => "5.90079_006"},{"date" => "2015-01-02T18:11:38","version" => "5.90079"},{"date" => "2015-01-07T20:01:40","version" => "5.90079_007"},{"date" => "2015-01-07T23:26:17","version" => "5.90079_008"},{"date" => "2015-01-09T17:04:47","version" => "5.90080"},{"date" => "2015-01-10T22:39:56","version" => "5.90081"},{"date" => "2015-01-10T23:33:56","version" => "5.90082"},{"date" => "2015-02-17T02:29:50","version" => "5.90083"},{"date" => "2015-02-23T22:24:50","version" => "5.90084"},{"date" => "2015-03-25T18:58:11","version" => "5.90085"},{"date" => "2015-03-26T21:30:15","version" => "5.90089_001"},{"date" => "2015-04-17T21:32:30","version" => "5.90089_002"},{"date" => "2015-04-27T20:20:40","version" => "5.90089_003"},{"date" => "2015-04-28T18:24:12","version" => "5.90089_004"},{"date" => "2015-04-29T14:04:24","version" => "5.90090"},{"date" => "2015-05-08T20:36:59","version" => "5.90091"},{"date" => "2015-05-19T16:48:30","version" => "5.90092"},{"date" => "2015-05-29T17:06:23","version" => "5.90093"},{"date" => "2015-07-24T20:17:46","version" => "5.90094"},{"date" => "2015-07-27T14:32:30","version" => "5.90095"},{"date" => "2015-07-27T15:44:59","version" => "5.90096"},{"date" => "2015-07-28T20:33:41","version" => "5.90097"},{"date" => "2015-08-24T16:30:12","version" => "5.90100"},{"date" => "2015-09-04T22:57:40","version" => "5.90101"},{"date" => "2015-10-29T19:39:24","version" => "5.90102"},{"date" => "2015-11-12T10:19:42","version" => "5.90103"},{"date" => "2016-04-04T17:18:38","version" => "5.90104"},{"date" => "2016-06-08T20:06:53","version" => "5.90105"},{"date" => "2016-07-06T01:21:42","version" => "5.90106"},{"date" => "2016-07-20T19:12:32","version" => "5.90110"},{"date" => "2016-07-20T20:07:16","version" => "5.90111"},{"date" => "2016-07-25T21:03:05","version" => "5.90112"},{"date" => "2016-12-15T21:35:30","version" => "5.90113"},{"date" => "2016-12-19T16:54:08","version" => "5.90114"},{"date" => "2017-05-01T16:42:46","version" => "5.90115"},{"date" => "2018-01-19T20:55:15","version" => "5.90116"},{"date" => "2018-01-21T23:47:21","version" => "5.90117"},{"date" => "2018-05-01T09:59:20","version" => "5.90118"},{"date" => "2018-09-24T00:25:48","version" => "5.90119"},{"date" => "2018-10-19T06:13:58","version" => "5.90120"},{"date" => "2018-10-22T20:39:48","version" => "5.90_121"},{"date" => "2018-11-03T14:52:06","version" => "5.90122"},{"date" => "2018-11-27T15:39:35","version" => "5.90123"},{"date" => "2019-01-18T22:36:07","version" => "5.90124"},{"date" => "2020-01-19T01:11:05","version" => "5.90125"},{"date" => "2020-01-20T01:40:16","version" => "5.90126"},{"date" => "2020-07-27T01:25:21","version" => "5.90_127"},{"date" => "2020-09-11T12:38:26","version" => "5.90128"},{"date" => "2022-07-23T13:13:34","version" => "5.90129"},{"date" => "2022-11-09T15:37:01","version" => "5.90130"},{"date" => "2023-07-20T23:09:29","version" => "5.90131"},{"date" => "2024-11-08T19:56:41","version" => "5.90132"}]},"Clipboard" => {"advisories" => [{"affected_versions" => ["<0.16"],"cves" => ["CVE-2014-5509"],"description" => "clipedit in the Clipboard module for Perl allows local users to delete arbitrary files via a symlink attack on /tmp/clipedit\$\$.\n","distribution" => "Clipboard","fixed_versions" => [">=0.16"],"id" => "CPANSA-Clipboard-2014-5509","references" => ["https://rt.cpan.org/Public/Bug/Display.html?id=98435","https://bugzilla.redhat.com/show_bug.cgi?id=1135624","http://www.securityfocus.com/bid/69473","http://www.openwall.com/lists/oss-security/2014/08/30/2"],"reported" => "2018-01-08","severity" => "medium"}],"main_module" => "Clipboard","versions" => [{"date" => "2005-05-01T17:19:57","version" => "0.01"},{"date" => "2005-05-02T06:17:33","version" => "0.02"},{"date" => "2005-05-04T06:17:44","version" => "0.03"},{"date" => "2005-05-08T05:54:44","version" => "0.04"},{"date" => "2005-06-01T17:00:34","version" => "0.06"},{"date" => "2005-06-02T05:06:37","version" => "0.07"},{"date" => "2005-06-22T17:05:15","version" => "0.08"},{"date" => "2005-11-19T06:12:48","version" => "0.09"},{"date" => "2010-10-07T01:39:10","version" => "0.10"},{"date" => "2010-10-07T04:49:39","version" => "0.11"},{"date" => "2010-10-11T06:13:22","version" => "0.12"},{"date" => "2010-10-13T04:46:50","version" => "0.13"},{"date" => "2019-01-30T10:47:45","version" => "0.14"},{"date" => "2019-01-30T11:00:22","version" => "0.15"},{"date" => "2019-01-30T11:22:23","version" => "0.16"},{"date" => "2019-01-30T14:00:52","version" => "0.17"},{"date" => "2019-01-30T20:12:11","version" => "0.18"},{"date" => "2019-01-31T11:00:20","version" => "0.19"},{"date" => "2019-04-17T20:55:35","version" => "0.20"},{"date" => "2019-12-02T06:04:27","version" => "0.21"},{"date" => "2020-01-28T18:10:34","version" => "0.22"},{"date" => "2020-03-06T15:43:11","version" => "0.23"},{"date" => "2020-03-07T08:25:07","version" => "0.24"},{"date" => "2020-05-14T06:33:28","version" => "0.25"},{"date" => "2020-05-16T07:56:58","version" => "0.26"},{"date" => "2021-02-13T18:13:34","version" => "0.27"},{"date" => "2021-02-23T07:47:20","version" => "0.28"},{"date" => "2024-04-07T03:11:57","version" => "0.29"},{"date" => "2024-06-16T12:03:21","version" => "0.30"},{"date" => "2025-01-21T17:04:47","version" => "0.31"},{"date" => "2025-02-10T08:24:16","version" => "0.32"}]},"Cmd-Dwarf" => {"advisories" => [{"affected_versions" => ["<1.20"],"cves" => [],"description" => "JSON highjacking possibility.\n","distribution" => "Cmd-Dwarf","fixed_versions" => [">=1.20"],"id" => "CPANSA-Cmd-Dwarf-2014-01","references" => ["https://github.com/seagirl/dwarf/commit/14cf7a1d55db635a07f4838e16f3d9a28e63f529","https://metacpan.org/changes/distribution/Cmd-Dwarf"],"reported" => "2014-12-03"}],"main_module" => "Cmd::Dwarf","versions" => [{"date" => "2015-07-30T06:48:35","version" => "1.27"},{"date" => "2015-08-26T13:27:51","version" => "1.28"},{"date" => "2015-08-28T08:58:33","version" => "1.29"},{"date" => "2015-09-17T08:17:32","version" => "1.30"},{"date" => "2016-01-20T06:39:15","version" => "1.31"},{"date" => "2016-10-25T05:56:33","version" => "1.41"},{"date" => "2017-03-29T04:42:05","version" => "1.42"},{"date" => "2017-06-21T07:06:05","version" => "1.50"},{"date" => "2017-10-05T08:08:01","version" => "1.60"},{"date" => "2018-03-17T07:35:19","version" => "1.70"},{"date" => "2018-08-18T11:43:10","version" => "1.80"},{"date" => "2019-04-05T05:22:33","version" => "1.81"},{"date" => "2019-04-05T05:36:46","version" => "1.82"},{"date" => "2019-04-05T09:06:55","version" => "1.83"}]},"Compress-LZ4" => {"advisories" => [{"affected_versions" => ["<0.20"],"cves" => [],"description" => "Outdated LZ4 source code with security issue on 32bit systems.\n","distribution" => "Compress-LZ4","fixed_versions" => [">=0.20"],"id" => "CPANSA-Compress-LZ4-2014-01","references" => ["https://metacpan.org/changes/distribution/Compress-LZ4","https://github.com/gray/compress-lz4/commit/fc503812b4cbba16429658e1dfe20ad8bbfd77a0"],"reported" => "2014-07-07"}],"main_module" => "Compress::LZ4","versions" => [{"date" => "2012-02-11T16:33:26","version" => "0.01"},{"date" => "2012-02-20T21:26:48","version" => "0.02"},{"date" => "2012-03-02T04:47:50","version" => "0.03"},{"date" => "2012-03-18T07:09:30","version" => "0.04"},{"date" => "2012-03-18T19:45:25","version" => "0.05"},{"date" => "2012-03-22T09:23:45","version" => "0.06"},{"date" => "2012-03-22T16:12:43","version" => "0.07"},{"date" => "2012-03-23T16:29:14","version" => "0.08"},{"date" => "2012-03-23T17:27:12","version" => "0.09"},{"date" => "2012-03-26T11:28:24","version" => "0.10"},{"date" => "2012-04-03T21:36:24","version" => "0.11"},{"date" => "2012-04-04T12:55:22","version" => "0.12"},{"date" => "2012-06-01T18:55:41","version" => "0.13"},{"date" => "2012-08-10T00:21:56","version" => "0.14"},{"date" => "2012-08-11T16:37:53","version" => "0.15"},{"date" => "2012-09-08T18:18:41","version" => "0.16"},{"date" => "2013-03-19T00:39:07","version" => "0.17"},{"date" => "2013-11-19T00:56:57","version" => "0.18"},{"date" => "2014-02-08T00:35:09","version" => "0.19"},{"date" => "2014-07-07T21:08:49","version" => "0.20"},{"date" => "2015-05-12T19:01:36","version" => "0.21"},{"date" => "2015-05-20T06:16:53","version" => "0.22"},{"date" => "2016-07-25T20:45:05","version" => "0.23"},{"date" => "2017-03-23T04:34:45","version" => "0.24"},{"date" => "2017-04-06T16:38:31","version" => "0.25"}]},"Compress-Raw-Bzip2" => {"advisories" => [{"affected_versions" => ["<2.031"],"cves" => ["CVE-2010-0405"],"description" => "Integer overflow in the BZ2_decompress function in decompress.c in bzip2 and libbzip2 before 1.0.6 allows context-dependent attackers to cause a denial of service (application crash) or possibly execute arbitrary code via a crafted compressed file.\n","distribution" => "Compress-Raw-Bzip2","fixed_versions" => [">=2.031"],"id" => "CPANSA-Compress-Raw-Bzip2-2010-0405","references" => ["https://metacpan.org/changes/distribution/Compress-Raw-Bzip2"],"reported" => "2010-01-27"},{"affected_versions" => ["<2.018"],"cves" => ["CVE-2009-1884"],"description" => "Off-by-one error in the bzinflate function in Bzip2.xs in the Compress-Raw-Bzip2 module before 2.018 for Perl allows context-dependent attackers to cause a denial of service (application hang or crash) via a crafted bzip2 compressed stream that triggers a buffer overflow, a related issue to CVE-2009-1391.\n","distribution" => "Compress-Raw-Bzip2","fixed_versions" => [">=2.018"],"id" => "CPANSA-Compress-Raw-Bzip2-2009-1884","references" => ["http://security.gentoo.org/glsa/glsa-200908-07.xml","https://bugs.gentoo.org/show_bug.cgi?id=281955","https://www.redhat.com/archives/fedora-package-announce/2009-August/msg00999.html","https://www.redhat.com/archives/fedora-package-announce/2009-August/msg00982.html","http://www.securityfocus.com/bid/36082","http://secunia.com/advisories/36386","https://bugzilla.redhat.com/show_bug.cgi?id=518278","http://secunia.com/advisories/36415","https://exchange.xforce.ibmcloud.com/vulnerabilities/52628"],"reported" => "2009-08-19","severity" => undef}],"main_module" => "Compress::Raw::Bzip2","versions" => [{"date" => "2006-03-13T16:14:00","version" => "2.000_10"},{"date" => "2006-04-15T21:23:09","version" => "2.000_11"},{"date" => "2006-05-17T12:43:30","version" => "2.000_12"},{"date" => "2006-06-20T12:43:47","version" => "2.000_13"},{"date" => "2006-10-26T14:15:45","version" => "2.000_14"},{"date" => "2006-11-01T10:35:27","version" => "2.001"},{"date" => "2006-12-29T20:40:23","version" => "2.002"},{"date" => "2007-01-02T13:03:45","version" => "2.003"},{"date" => "2007-03-03T15:50:04","version" => "2.004"},{"date" => "2007-07-01T00:06:51","version" => "2.005"},{"date" => "2007-09-01T19:44:48","version" => "2.006"},{"date" => "2007-11-10T11:59:25","version" => "2.008"},{"date" => "2008-04-20T14:41:25","version" => "2.009"},{"date" => "2008-05-05T17:18:15","version" => "2.010"},{"date" => "2008-05-17T11:16:17","version" => "2.011"},{"date" => "2008-07-15T22:23:56","version" => "2.012"},{"date" => "2008-09-02T20:20:05","version" => "2.014"},{"date" => "2008-09-03T20:47:15","version" => "2.015"},{"date" => "2009-04-04T09:47:36","version" => "2.017"},{"date" => "2009-05-03T16:26:57","version" => "2.018"},{"date" => "2009-05-04T09:42:06","version" => "2.019"},{"date" => "2009-06-03T17:48:18","version" => "2.020"},{"date" => "2009-08-30T20:25:24","version" => "2.021"},{"date" => "2009-11-09T23:25:19","version" => "2.023"},{"date" => "2010-01-09T17:56:12","version" => "2.024"},{"date" => "2010-03-28T12:56:33","version" => "2.025"},{"date" => "2010-04-07T19:49:29","version" => "2.026"},{"date" => "2010-04-24T19:15:32","version" => "2.027"},{"date" => "2010-07-24T14:29:17","version" => "2.030"},{"date" => "2010-09-21T19:44:52","version" => "2.031"},{"date" => "2011-01-06T11:26:00","version" => "2.032"},{"date" => "2011-01-11T14:02:05","version" => "2.033"},{"date" => "2011-05-02T21:50:15","version" => "2.034"},{"date" => "2011-05-07T08:30:09","version" => "2.035"},{"date" => "2011-06-18T21:45:13","version" => "2.036"},{"date" => "2011-06-22T07:17:56","version" => "2.037"},{"date" => "2011-10-28T14:27:59","version" => "2.039"},{"date" => "2011-10-28T22:18:59","version" => "2.040"},{"date" => "2011-11-17T23:44:58","version" => "2.042"},{"date" => "2011-11-20T21:31:34","version" => "2.043"},{"date" => "2011-12-03T22:48:47","version" => "2.044"},{"date" => "2011-12-04T19:19:58","version" => "2.045"},{"date" => "2012-01-28T23:26:44","version" => "2.047"},{"date" => "2012-01-29T16:58:55","version" => "2.048"},{"date" => "2012-02-18T15:56:34","version" => "2.049"},{"date" => "2012-04-29T12:40:06","version" => "2.052"},{"date" => "2012-08-05T20:35:37","version" => "2.055"},{"date" => "2012-11-10T19:08:29","version" => "2.057"},{"date" => "2012-11-12T22:14:16","version" => "2.058"},{"date" => "2012-11-25T13:38:19","version" => "2.059"},{"date" => "2013-01-07T20:02:08","version" => "2.060"},{"date" => "2013-05-27T09:54:30","version" => "2.061"},{"date" => "2013-08-12T19:06:20","version" => "2.062"},{"date" => "2013-11-02T17:14:54","version" => "2.063"},{"date" => "2014-02-01T23:19:50","version" => "2.064"},{"date" => "2014-09-21T12:40:58","version" => "2.066"},{"date" => "2014-12-08T15:12:21","version" => "2.067"},{"date" => "2014-12-23T17:44:34","version" => "2.068"},{"date" => "2015-09-27T14:33:57","version" => "2.069"},{"date" => "2016-12-28T23:07:42","version" => "2.070"},{"date" => "2017-02-12T20:39:20","version" => "2.072"},{"date" => "2017-02-19T20:35:17","version" => "2.073"},{"date" => "2017-02-19T22:11:17","version" => "2.074"},{"date" => "2018-04-03T18:20:04","version" => "2.080"},{"date" => "2018-04-08T15:01:21","version" => "2.081"},{"date" => "2018-12-30T22:38:05","version" => "2.083"},{"date" => "2019-01-06T08:56:52","version" => "2.084"},{"date" => "2019-03-31T19:13:22","version" => "2.086"},{"date" => "2019-08-10T18:11:44","version" => "2.087"},{"date" => "2019-11-03T08:56:50","version" => "2.088"},{"date" => "2019-11-03T19:53:42","version" => "2.089"},{"date" => "2019-11-09T18:35:48","version" => "2.090"},{"date" => "2019-11-23T19:34:12","version" => "2.091"},{"date" => "2019-12-04T22:08:25","version" => "2.092"},{"date" => "2019-12-07T16:05:12","version" => "2.093"},{"date" => "2020-07-13T10:53:44","version" => "2.094"},{"date" => "2020-07-20T19:13:40","version" => "2.095"},{"date" => "2020-07-31T20:50:12","version" => "2.096"},{"date" => "2021-01-07T13:00:00","version" => "2.100"},{"date" => "2021-02-20T14:08:53","version" => "2.101"},{"date" => "2022-04-03T19:48:28","version" => "2.103"},{"date" => "2022-06-25T09:02:32","version" => "2.201"},{"date" => "2023-02-08T19:23:39","version" => "2.204"},{"date" => "2023-07-16T15:36:44","version" => "2.205"},{"date" => "2023-07-25T15:36:59","version" => "2.206"},{"date" => "2024-02-18T22:19:11","version" => "2.207"},{"date" => "2024-02-19T09:28:45","version" => "2.208"},{"date" => "2024-02-20T13:23:07","version" => "2.209"},{"date" => "2024-02-26T09:33:37","version" => "2.210"},{"date" => "2024-04-06T13:40:27","version" => "2.211"},{"date" => "2024-04-27T12:52:31","version" => "2.212"},{"date" => "2024-08-28T15:29:28","version" => "2.213"},{"date" => "2025-10-24T16:23:16","version" => "2.214"},{"date" => "2026-01-31T23:47:12","version" => "2.217"},{"date" => "2012-07-20T00:00:00","dual_lived" => 1,"perl_release" => "5.017002","version" => "2.05201"},{"date" => "2023-07-02T00:00:00","dual_lived" => 1,"perl_release" => "5.038","version" => "2.204_001"}]},"Compress-Raw-Zlib" => {"advisories" => [{"affected_versions" => ["<2.017"],"cves" => ["CVE-2009-1391"],"description" => "Off-by-one error in the inflate function in Zlib.xs in Compress::Raw::Zlib Perl module before 2.017, as used in AMaViS, SpamAssassin, and possibly other products, allows context-dependent attackers to cause a denial of service (hang or crash) via a crafted zlib compressed stream that triggers a heap-based buffer overflow, as exploited in the wild by Trojan.Downloader-71014 in June 2009.\n","distribution" => "Compress-Raw-Zlib","fixed_versions" => [">=2.017"],"id" => "CPANSA-Compress-Raw-Zlib-2009-1391","references" => ["http://article.gmane.org/gmane.mail.virus.amavis.user/33635","http://article.gmane.org/gmane.mail.virus.amavis.user/33638","http://www.securityfocus.com/bid/35307","http://secunia.com/advisories/35422","https://bugzilla.redhat.com/show_bug.cgi?id=504386","http://www.vupen.com/english/advisories/2009/1571","http://thread.gmane.org/gmane.mail.virus.amavis.user/33635","http://osvdb.org/55041","http://lists.opensuse.org/opensuse-security-announce/2009-07/msg00002.html","http://secunia.com/advisories/35876","http://secunia.com/advisories/35685","https://www.redhat.com/archives/fedora-package-announce/2009-July/msg00607.html","http://secunia.com/advisories/35689","http://www.mandriva.com/security/advisories?name=MDVSA-2009:157","http://security.gentoo.org/glsa/glsa-200908-07.xml","https://bugs.gentoo.org/show_bug.cgi?id=273141","https://exchange.xforce.ibmcloud.com/vulnerabilities/51062","https://usn.ubuntu.com/794-1/"],"reported" => "2009-06-16","severity" => undef},{"affected_versions" => [">=2.000_09,<=2.024"],"cves" => ["CVE-2018-25032"],"description" => "zlib before 1.2.12 allows memory corruption when deflating (i.e., when compressing) if the input has many distant matches.\n","distribution" => "Compress-Raw-Zlib","fixed_versions" => [],"id" => "CPANSA-Compress-Raw-Zlib-2018-25032-zlib","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143579","https://www.openwall.com/lists/oss-security/2022/03/24/1","https://github.com/madler/zlib/commit/5c44459c3b28a9bd3283aaceab7c615f8020c531","http://www.openwall.com/lists/oss-security/2022/03/25/2","http://www.openwall.com/lists/oss-security/2022/03/26/1","https://www.openwall.com/lists/oss-security/2022/03/28/1","https://github.com/madler/zlib/compare/v1.2.11...v1.2.12","https://www.openwall.com/lists/oss-security/2022/03/28/3","https://github.com/madler/zlib/issues/605","https://www.debian.org/security/2022/dsa-5111","https://lists.debian.org/debian-lts-announce/2022/04/msg00000.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/NS2D2GFPFGOJUL4WQ3DUAY7HF4VWQ77F/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/XOKFMSNQ5D5WGMALBNBXU3GE442V74WU/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/VOKNP2L734AEL47NRYGVZIKEFOUBQY5Y/","https://lists.debian.org/debian-lts-announce/2022/05/msg00008.html","https://support.apple.com/kb/HT213255","https://support.apple.com/kb/HT213256","https://support.apple.com/kb/HT213257","http://seclists.org/fulldisclosure/2022/May/33","http://seclists.org/fulldisclosure/2022/May/35","http://seclists.org/fulldisclosure/2022/May/38","https://security.netapp.com/advisory/ntap-20220526-0009/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DF62MVMH3QUGMBDCB3DY2ERQ6EBHTADB/"],"reported" => "2022-03-25","severity" => "high"},{"affected_versions" => [">=2.025,<=2.048"],"cves" => ["CVE-2018-25032"],"description" => "zlib before 1.2.12 allows memory corruption when deflating (i.e., when compressing) if the input has many distant matches.\n","distribution" => "Compress-Raw-Zlib","fixed_versions" => [],"id" => "CPANSA-Compress-Raw-Zlib-2018-25032-zlib","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143579","https://www.openwall.com/lists/oss-security/2022/03/24/1","https://github.com/madler/zlib/commit/5c44459c3b28a9bd3283aaceab7c615f8020c531","http://www.openwall.com/lists/oss-security/2022/03/25/2","http://www.openwall.com/lists/oss-security/2022/03/26/1","https://www.openwall.com/lists/oss-security/2022/03/28/1","https://github.com/madler/zlib/compare/v1.2.11...v1.2.12","https://www.openwall.com/lists/oss-security/2022/03/28/3","https://github.com/madler/zlib/issues/605","https://www.debian.org/security/2022/dsa-5111","https://lists.debian.org/debian-lts-announce/2022/04/msg00000.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/NS2D2GFPFGOJUL4WQ3DUAY7HF4VWQ77F/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/XOKFMSNQ5D5WGMALBNBXU3GE442V74WU/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/VOKNP2L734AEL47NRYGVZIKEFOUBQY5Y/","https://lists.debian.org/debian-lts-announce/2022/05/msg00008.html","https://support.apple.com/kb/HT213255","https://support.apple.com/kb/HT213256","https://support.apple.com/kb/HT213257","http://seclists.org/fulldisclosure/2022/May/33","http://seclists.org/fulldisclosure/2022/May/35","http://seclists.org/fulldisclosure/2022/May/38","https://security.netapp.com/advisory/ntap-20220526-0009/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DF62MVMH3QUGMBDCB3DY2ERQ6EBHTADB/"],"reported" => "2022-03-25","severity" => "high"},{"affected_versions" => [">=2.049,<=2.052"],"cves" => ["CVE-2018-25032"],"description" => "zlib before 1.2.12 allows memory corruption when deflating (i.e., when compressing) if the input has many distant matches.\n","distribution" => "Compress-Raw-Zlib","fixed_versions" => [],"id" => "CPANSA-Compress-Raw-Zlib-2018-25032-zlib","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143579","https://www.openwall.com/lists/oss-security/2022/03/24/1","https://github.com/madler/zlib/commit/5c44459c3b28a9bd3283aaceab7c615f8020c531","http://www.openwall.com/lists/oss-security/2022/03/25/2","http://www.openwall.com/lists/oss-security/2022/03/26/1","https://www.openwall.com/lists/oss-security/2022/03/28/1","https://github.com/madler/zlib/compare/v1.2.11...v1.2.12","https://www.openwall.com/lists/oss-security/2022/03/28/3","https://github.com/madler/zlib/issues/605","https://www.debian.org/security/2022/dsa-5111","https://lists.debian.org/debian-lts-announce/2022/04/msg00000.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/NS2D2GFPFGOJUL4WQ3DUAY7HF4VWQ77F/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/XOKFMSNQ5D5WGMALBNBXU3GE442V74WU/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/VOKNP2L734AEL47NRYGVZIKEFOUBQY5Y/","https://lists.debian.org/debian-lts-announce/2022/05/msg00008.html","https://support.apple.com/kb/HT213255","https://support.apple.com/kb/HT213256","https://support.apple.com/kb/HT213257","http://seclists.org/fulldisclosure/2022/May/33","http://seclists.org/fulldisclosure/2022/May/35","http://seclists.org/fulldisclosure/2022/May/38","https://security.netapp.com/advisory/ntap-20220526-0009/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DF62MVMH3QUGMBDCB3DY2ERQ6EBHTADB/"],"reported" => "2022-03-25","severity" => "high"},{"affected_versions" => [">=2.053,<=2.060"],"cves" => ["CVE-2018-25032"],"description" => "zlib before 1.2.12 allows memory corruption when deflating (i.e., when compressing) if the input has many distant matches.\n","distribution" => "Compress-Raw-Zlib","fixed_versions" => [],"id" => "CPANSA-Compress-Raw-Zlib-2018-25032-zlib","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143579","https://www.openwall.com/lists/oss-security/2022/03/24/1","https://github.com/madler/zlib/commit/5c44459c3b28a9bd3283aaceab7c615f8020c531","http://www.openwall.com/lists/oss-security/2022/03/25/2","http://www.openwall.com/lists/oss-security/2022/03/26/1","https://www.openwall.com/lists/oss-security/2022/03/28/1","https://github.com/madler/zlib/compare/v1.2.11...v1.2.12","https://www.openwall.com/lists/oss-security/2022/03/28/3","https://github.com/madler/zlib/issues/605","https://www.debian.org/security/2022/dsa-5111","https://lists.debian.org/debian-lts-announce/2022/04/msg00000.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/NS2D2GFPFGOJUL4WQ3DUAY7HF4VWQ77F/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/XOKFMSNQ5D5WGMALBNBXU3GE442V74WU/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/VOKNP2L734AEL47NRYGVZIKEFOUBQY5Y/","https://lists.debian.org/debian-lts-announce/2022/05/msg00008.html","https://support.apple.com/kb/HT213255","https://support.apple.com/kb/HT213256","https://support.apple.com/kb/HT213257","http://seclists.org/fulldisclosure/2022/May/33","http://seclists.org/fulldisclosure/2022/May/35","http://seclists.org/fulldisclosure/2022/May/38","https://security.netapp.com/advisory/ntap-20220526-0009/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DF62MVMH3QUGMBDCB3DY2ERQ6EBHTADB/"],"reported" => "2022-03-25","severity" => "high"},{"affected_versions" => [">=2.061,<=2.074"],"cves" => ["CVE-2018-25032"],"description" => "zlib before 1.2.12 allows memory corruption when deflating (i.e., when compressing) if the input has many distant matches.\n","distribution" => "Compress-Raw-Zlib","fixed_versions" => [],"id" => "CPANSA-Compress-Raw-Zlib-2018-25032-zlib","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143579","https://www.openwall.com/lists/oss-security/2022/03/24/1","https://github.com/madler/zlib/commit/5c44459c3b28a9bd3283aaceab7c615f8020c531","http://www.openwall.com/lists/oss-security/2022/03/25/2","http://www.openwall.com/lists/oss-security/2022/03/26/1","https://www.openwall.com/lists/oss-security/2022/03/28/1","https://github.com/madler/zlib/compare/v1.2.11...v1.2.12","https://www.openwall.com/lists/oss-security/2022/03/28/3","https://github.com/madler/zlib/issues/605","https://www.debian.org/security/2022/dsa-5111","https://lists.debian.org/debian-lts-announce/2022/04/msg00000.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/NS2D2GFPFGOJUL4WQ3DUAY7HF4VWQ77F/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/XOKFMSNQ5D5WGMALBNBXU3GE442V74WU/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/VOKNP2L734AEL47NRYGVZIKEFOUBQY5Y/","https://lists.debian.org/debian-lts-announce/2022/05/msg00008.html","https://support.apple.com/kb/HT213255","https://support.apple.com/kb/HT213256","https://support.apple.com/kb/HT213257","http://seclists.org/fulldisclosure/2022/May/33","http://seclists.org/fulldisclosure/2022/May/35","http://seclists.org/fulldisclosure/2022/May/38","https://security.netapp.com/advisory/ntap-20220526-0009/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DF62MVMH3QUGMBDCB3DY2ERQ6EBHTADB/"],"reported" => "2022-03-25","severity" => "high"},{"affected_versions" => [">=2.075,<=2.101"],"cves" => ["CVE-2018-25032"],"description" => "zlib before 1.2.12 allows memory corruption when deflating (i.e., when compressing) if the input has many distant matches.\n","distribution" => "Compress-Raw-Zlib","fixed_versions" => [],"id" => "CPANSA-Compress-Raw-Zlib-2018-25032-zlib","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143579","https://www.openwall.com/lists/oss-security/2022/03/24/1","https://github.com/madler/zlib/commit/5c44459c3b28a9bd3283aaceab7c615f8020c531","http://www.openwall.com/lists/oss-security/2022/03/25/2","http://www.openwall.com/lists/oss-security/2022/03/26/1","https://www.openwall.com/lists/oss-security/2022/03/28/1","https://github.com/madler/zlib/compare/v1.2.11...v1.2.12","https://www.openwall.com/lists/oss-security/2022/03/28/3","https://github.com/madler/zlib/issues/605","https://www.debian.org/security/2022/dsa-5111","https://lists.debian.org/debian-lts-announce/2022/04/msg00000.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/NS2D2GFPFGOJUL4WQ3DUAY7HF4VWQ77F/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/XOKFMSNQ5D5WGMALBNBXU3GE442V74WU/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/VOKNP2L734AEL47NRYGVZIKEFOUBQY5Y/","https://lists.debian.org/debian-lts-announce/2022/05/msg00008.html","https://support.apple.com/kb/HT213255","https://support.apple.com/kb/HT213256","https://support.apple.com/kb/HT213257","http://seclists.org/fulldisclosure/2022/May/33","http://seclists.org/fulldisclosure/2022/May/35","http://seclists.org/fulldisclosure/2022/May/38","https://security.netapp.com/advisory/ntap-20220526-0009/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DF62MVMH3QUGMBDCB3DY2ERQ6EBHTADB/"],"reported" => "2022-03-25","severity" => "high"}],"main_module" => "Compress::Raw::Zlib","versions" => [{"date" => "2006-03-03T23:06:38","version" => "2.000_09"},{"date" => "2006-03-13T16:14:20","version" => "2.000_10"},{"date" => "2006-04-15T21:23:24","version" => "2.000_11"},{"date" => "2006-05-17T12:43:41","version" => "2.000_12"},{"date" => "2006-06-20T12:45:30","version" => "2.000_13"},{"date" => "2006-10-26T14:15:34","version" => "2.000_09"},{"date" => "2006-11-01T10:35:38","version" => "2.001"},{"date" => "2006-12-29T20:40:35","version" => "2.002"},{"date" => "2007-01-02T13:03:57","version" => "2.003"},{"date" => "2007-03-03T15:50:15","version" => "2.004"},{"date" => "2007-07-01T00:07:02","version" => "2.005"},{"date" => "2007-09-01T19:44:59","version" => "2.006"},{"date" => "2007-11-10T11:59:36","version" => "2.008"},{"date" => "2008-04-20T14:42:52","version" => "2.009"},{"date" => "2008-05-05T17:18:27","version" => "2.010"},{"date" => "2008-05-17T11:16:28","version" => "2.011"},{"date" => "2008-07-15T22:24:07","version" => "2.012"},{"date" => "2008-09-02T20:20:16","version" => "2.014"},{"date" => "2008-09-03T20:47:27","version" => "2.015"},{"date" => "2009-04-04T09:49:03","version" => "2.017"},{"date" => "2009-05-03T16:27:08","version" => "2.018"},{"date" => "2009-05-04T09:42:17","version" => "2.019"},{"date" => "2009-06-03T17:48:30","version" => "2.020"},{"date" => "2009-08-30T20:25:35","version" => "2.021"},{"date" => "2009-11-09T23:26:59","version" => "2.023"},{"date" => "2010-01-09T17:56:35","version" => "2.024"},{"date" => "2010-03-28T12:57:08","version" => "2.025"},{"date" => "2010-04-07T19:51:09","version" => "2.026"},{"date" => "2010-04-24T19:15:54","version" => "2.027"},{"date" => "2010-07-24T14:31:01","version" => "2.030"},{"date" => "2011-01-06T11:23:45","version" => "2.032"},{"date" => "2011-01-11T14:03:45","version" => "2.033"},{"date" => "2011-05-02T22:05:37","version" => "2.034"},{"date" => "2011-05-07T08:31:57","version" => "2.035"},{"date" => "2011-06-18T21:45:36","version" => "2.036"},{"date" => "2011-06-22T07:18:22","version" => "2.037"},{"date" => "2011-10-28T14:28:35","version" => "2.039"},{"date" => "2011-10-28T22:20:38","version" => "2.040"},{"date" => "2011-11-17T23:45:21","version" => "2.042"},{"date" => "2011-11-20T21:33:33","version" => "2.043"},{"date" => "2011-12-03T22:49:10","version" => "2.044"},{"date" => "2011-12-04T19:21:36","version" => "2.045"},{"date" => "2012-01-28T23:28:28","version" => "2.047"},{"date" => "2012-01-29T17:00:33","version" => "2.048"},{"date" => "2012-02-18T15:58:12","version" => "2.049"},{"date" => "2012-02-21T19:35:18","version" => "2.050"},{"date" => "2012-02-22T20:43:23","version" => "2.051"},{"date" => "2012-04-29T12:41:57","version" => "2.052"},{"date" => "2012-05-06T08:40:06","version" => "2.053"},{"date" => "2012-05-08T19:22:47","version" => "2.054"},{"date" => "2012-08-05T20:36:06","version" => "2.055"},{"date" => "2012-08-10T22:20:09","version" => "2.056"},{"date" => "2012-11-10T19:08:56","version" => "2.057"},{"date" => "2012-11-12T22:14:42","version" => "2.058"},{"date" => "2012-11-25T13:38:42","version" => "2.059"},{"date" => "2013-01-07T20:02:22","version" => "2.060"},{"date" => "2013-05-27T09:54:54","version" => "2.061"},{"date" => "2013-08-12T19:08:05","version" => "2.062"},{"date" => "2013-11-02T17:15:17","version" => "2.063"},{"date" => "2014-02-01T23:21:28","version" => "2.064"},{"date" => "2014-02-03T20:23:00","version" => "2.065"},{"date" => "2014-09-21T12:42:35","version" => "2.066"},{"date" => "2014-12-08T15:14:00","version" => "2.067"},{"date" => "2014-12-23T17:44:57","version" => "2.068"},{"date" => "2015-09-26T18:41:58","version" => "2.069"},{"date" => "2016-12-28T23:09:21","version" => "2.070"},{"date" => "2016-12-30T22:58:08","version" => "2.071"},{"date" => "2017-02-12T20:41:25","version" => "2.072"},{"date" => "2017-02-19T20:37:20","version" => "2.073"},{"date" => "2017-02-19T22:11:41","version" => "2.074"},{"date" => "2017-11-14T15:43:26","version" => "2.075"},{"date" => "2017-11-21T22:29:23","version" => "2.076"},{"date" => "2018-04-03T18:22:06","version" => "2.080"},{"date" => "2018-04-08T15:02:55","version" => "2.081"},{"date" => "2018-12-30T22:40:08","version" => "2.083"},{"date" => "2019-01-06T08:57:15","version" => "2.084"},{"date" => "2019-03-31T19:11:54","version" => "2.086"},{"date" => "2019-08-10T18:12:03","version" => "2.087"},{"date" => "2019-11-03T08:55:23","version" => "2.088"},{"date" => "2019-11-03T19:54:04","version" => "2.089"},{"date" => "2019-11-09T15:58:48","version" => "2.090"},{"date" => "2019-11-23T19:34:34","version" => "2.091"},{"date" => "2019-12-04T22:08:37","version" => "2.092"},{"date" => "2019-12-07T16:05:34","version" => "2.093"},{"date" => "2020-07-13T10:54:06","version" => "2.094"},{"date" => "2020-07-21T06:57:01","version" => "2.095"},{"date" => "2020-07-31T20:48:45","version" => "2.096"},{"date" => "2021-01-07T13:00:23","version" => "2.100"},{"date" => "2021-02-20T14:10:43","version" => "2.101"},{"date" => "2022-04-03T19:48:50","version" => "2.103"},{"date" => "2022-05-13T06:30:30","version" => "2.104"},{"date" => "2022-05-14T14:24:32","version" => "2.105"},{"date" => "2022-06-21T21:19:21","version" => "2.200"},{"date" => "2022-06-25T09:04:10","version" => "2.201"},{"date" => "2022-06-27T08:18:10","version" => "2.202"},{"date" => "2023-02-08T19:26:25","version" => "2.204"},{"date" => "2023-07-16T15:32:41","version" => "2.205"},{"date" => "2023-07-25T15:35:40","version" => "2.206"},{"date" => "2024-02-18T22:16:24","version" => "2.207"},{"date" => "2024-02-19T09:27:19","version" => "2.208"},{"date" => "2024-02-26T16:11:33","version" => "2.209"},{"date" => "2024-04-06T13:41:58","version" => "2.211"},{"date" => "2024-04-27T12:55:28","version" => "2.212"},{"date" => "2024-08-28T15:27:59","version" => "2.213"},{"date" => "2025-10-24T16:23:27","version" => "2.214"},{"date" => "2026-01-31T22:31:04","version" => "2.217"},{"date" => "2026-02-03T10:45:59","version" => "2.218"},{"date" => "2026-02-23T15:24:28","version" => "2.219"},{"date" => "2026-02-27T10:04:09","version" => "2.220"},{"date" => "2026-02-27T13:17:42","version" => "2.221"},{"date" => "2010-07-20T00:00:00","dual_lived" => 1,"perl_release" => "5.013003","version" => "2.027_01"},{"date" => "2012-07-20T00:00:00","dual_lived" => 1,"perl_release" => "5.017002","version" => "2.05401"},{"date" => "2015-09-20T00:00:00","dual_lived" => 1,"perl_release" => "5.023003","version" => "2.068_01"},{"date" => "2023-07-02T00:00:00","dual_lived" => 1,"perl_release" => "5.038","version" => "2.204_001"}]},"Concierge-Sessions" => {"advisories" => [{"affected_versions" => [">=0.8.1,<0.8.5"],"cves" => ["CVE-2026-2439"],"description" => "Concierge::Sessions versions from 0.8.1 before 0.8.5 for Perl generate insecure session ids. The generate_session_id function in Concierge::Sessions::Base defaults to using the uuidgen command to generate a UUID, with a fallback to using Perl's built-in rand function. Neither of these methods are secure, and attackers are able to guess session_ids that can grant them access to systems. Specifically,    *  There is no warning when uuidgen fails. The software can be quietly using the fallback rand() function with no warnings if the command fails for any reason.   *  The uuidgen command will generate a time-based UUID if the system does not have a high-quality random number source, because the call does not explicitly specify the --random option. Note that the system time is shared in HTTP responses.   *  UUIDs are identifiers whose mere possession grants access, as per RFC 9562.   *  The output of the built-in rand() function is predictable and unsuitable for security applications.","distribution" => "Concierge-Sessions","fixed_versions" => [">=0.8.5"],"id" => "CPANSA-Concierge-Sessions-2026-2439","references" => ["https://github.com/bwva/Concierge-Sessions/commit/20bb28e92e8fba307c4ff8264701c215be65e73b","https://metacpan.org/release/BVA/Concierge-Sessions-v0.8.4/diff/BVA/Concierge-Sessions-v0.8.5#lib/Concierge/Sessions/Base.pm","https://perldoc.perl.org/5.42.0/functions/rand","https://security.metacpan.org/docs/guides/random-data-for-security.html","https://www.rfc-editor.org/rfc/rfc9562.html#name-security-considerations"],"reported" => "2026-02-16","severity" => undef}],"main_module" => "Concierge::Sessions","versions" => [{"date" => "2026-02-11T23:31:48","version" => "v0.8.1"},{"date" => "2026-02-12T04:02:53","version" => "v0.8.2"},{"date" => "2026-02-12T08:43:53","version" => "v0.8.3"},{"date" => "2026-02-12T09:47:28","version" => "v0.8.4"},{"date" => "2026-02-12T16:49:19","version" => "v0.8.5"},{"date" => "2026-02-13T04:18:00","version" => "v0.8.6"},{"date" => "2026-02-13T15:38:42","version" => "v0.8.7"},{"date" => "2026-02-13T17:47:48","version" => "v0.8.8"},{"date" => "2026-02-13T22:10:44","version" => "v0.8.9"},{"date" => "2026-02-13T22:32:58","version" => "v0.9.0"},{"date" => "2026-02-15T04:14:21","version" => "v0.10.0"},{"date" => "2026-02-15T18:14:32","version" => "v0.11.0"}]},"Config-IniFiles" => {"advisories" => [{"affected_versions" => ["<2.71"],"cves" => ["CVE-2012-2451"],"description" => "The Config::IniFiles module before 2.71 for Perl creates temporary files with predictable names, which allows local users to overwrite arbitrary files via a symlink attack.  NOTE: some of these details are obtained from third party information.  NOTE: it has been reported that this might only be exploitable by writing in the same directory as the .ini file. If this is the case, then this issue might not cross privilege boundaries.\n","distribution" => "Config-IniFiles","fixed_versions" => [],"id" => "CPANSA-Config-IniFiles-2012-2451","references" => ["http://www.openwall.com/lists/oss-security/2012/05/02/6","http://www.osvdb.org/81671","http://secunia.com/advisories/48990","https://bitbucket.org/shlomif/perl-config-inifiles/changeset/a08fa26f4f59","https://bugzilla.redhat.com/show_bug.cgi?id=818386","http://lists.fedoraproject.org/pipermail/package-announce/2012-May/080713.html","http://lists.fedoraproject.org/pipermail/package-announce/2012-May/080716.html","http://lists.fedoraproject.org/pipermail/package-announce/2012-May/081207.html","http://www.securityfocus.com/bid/53361","http://www.ubuntu.com/usn/USN-1543-1","https://exchange.xforce.ibmcloud.com/vulnerabilities/75328"],"reported" => "2012-06-27","severity" => undef}],"main_module" => "Config::IniFiles","versions" => [{"date" => "2000-06-20T02:07:11","version" => "0.05"},{"date" => "2000-06-26T02:38:13","version" => "0.06"},{"date" => "2000-07-31T01:32:53","version" => "0.08"},{"date" => "2000-07-31T01:51:15","version" => "0.09"},{"date" => "2000-09-17T03:29:50","version" => "0.11"},{"date" => "2000-09-20T01:06:10","version" => "1.6"},{"date" => "2000-10-17T02:01:37","version" => "1.8"},{"date" => "2000-11-25T03:36:14","version" => "1.11"},{"date" => "2000-11-28T14:09:00","version" => "1.12"},{"date" => "2000-11-28T14:09:15","version" => "1.13"},{"date" => "2000-12-13T17:45:25","version" => "2.10"},{"date" => "2000-12-18T17:56:28","version" => "2.13"},{"date" => "2001-01-08T18:27:24","version" => "2.14"},{"date" => "2001-03-24T21:54:18","version" => "2.17"},{"date" => "2001-03-30T04:48:27","version" => "2.18"},{"date" => "2001-04-05T01:06:51","version" => "2.19"},{"date" => "2001-08-28T11:05:35","version" => "2.21"},{"date" => "2001-12-06T17:15:03","version" => "2.22"},{"date" => "2001-12-07T16:24:14","version" => "2.23"},{"date" => "2001-12-07T17:27:08","version" => "2.24"},{"date" => "2001-12-12T20:50:06","version" => "2.25"},{"date" => "2001-12-19T22:26:32","version" => "2.26"},{"date" => "2001-12-20T16:11:24","version" => "2.27"},{"date" => "2002-08-15T21:41:35","version" => "2.29"},{"date" => "2002-10-15T18:59:21","version" => "2.30"},{"date" => "2002-12-18T01:58:55","version" => "2.36"},{"date" => "2003-01-31T23:06:08","version" => "2.37"},{"date" => "2003-05-14T01:38:13","version" => "2.38"},{"date" => "2005-04-29T20:33:23","version" => "2.39"},{"date" => "2008-12-04T17:02:19","version" => "2.43"},{"date" => "2008-12-25T09:47:08","version" => "2.44"},{"date" => "2008-12-27T15:25:59","version" => "2.45"},{"date" => "2009-01-17T14:40:26","version" => "2.46"},{"date" => "2009-01-21T09:41:11","version" => "2.47"},{"date" => "2009-04-07T12:26:44","version" => "2.48"},{"date" => "2009-05-02T14:27:53","version" => "2.49"},{"date" => "2009-05-31T11:58:04","version" => "2.50"},{"date" => "2009-06-08T09:41:11","version" => "2.51"},{"date" => "2009-06-28T13:21:57","version" => "2.52"},{"date" => "2009-11-13T09:58:28","version" => "2.53"},{"date" => "2009-11-18T11:15:13","version" => "2.54"},{"date" => "2009-12-22T15:48:07","version" => "2.55"},{"date" => "2009-12-31T04:57:40","version" => "2.56"},{"date" => "2010-03-01T13:51:57","version" => "2.57"},{"date" => "2010-05-17T07:45:33","version" => "2.58"},{"date" => "2010-11-12T11:33:52","version" => "2.59"},{"date" => "2010-11-13T07:22:50","version" => "2.60"},{"date" => "2010-11-14T08:57:26","version" => "2.61"},{"date" => "2010-11-19T13:37:37","version" => "2.62"},{"date" => "2010-11-19T14:54:12","version" => "2.63"},{"date" => "2010-11-20T09:55:05","version" => "2.64"},{"date" => "2010-11-25T18:48:52","version" => "2.65"},{"date" => "2011-01-29T16:40:18","version" => "2.66"},{"date" => "2011-06-21T11:59:37","version" => "2.67"},{"date" => "2011-06-21T19:18:33","version" => "2.68"},{"date" => "2012-04-05T09:10:11","version" => "2.69"},{"date" => "2012-04-06T09:52:14","version" => "2.70"},{"date" => "2012-05-02T08:05:15","version" => "2.71"},{"date" => "2012-05-05T16:56:55","version" => "2.72"},{"date" => "2012-05-14T07:49:33","version" => "2.73"},{"date" => "2012-05-23T21:47:46","version" => "2.74"},{"date" => "2012-05-25T12:29:48","version" => "2.75"},{"date" => "2012-06-15T14:47:10","version" => "2.76"},{"date" => "2012-06-21T16:39:23","version" => "2.77"},{"date" => "2012-10-21T11:18:39","version" => "2.78"},{"date" => "2013-05-06T07:10:33","version" => "2.79"},{"date" => "2013-05-14T19:25:07","version" => "2.80"},{"date" => "2013-05-16T10:36:17","version" => "2.81"},{"date" => "2013-05-21T15:35:10","version" => "2.82"},{"date" => "2014-01-27T09:01:28","version" => "2.83"},{"date" => "2015-04-13T18:40:30","version" => "2.84"},{"date" => "2015-04-13T19:08:57","version" => "2.85"},{"date" => "2015-04-14T07:55:59","version" => "2.86"},{"date" => "2015-06-16T09:06:37","version" => "2.87"},{"date" => "2015-07-10T08:38:11","version" => "2.88"},{"date" => "2016-05-03T09:14:13","version" => "2.89"},{"date" => "2016-06-02T13:09:19","version" => "2.90"},{"date" => "2016-06-03T03:11:38","version" => "2.91"},{"date" => "2016-06-17T09:34:08","version" => "2.92"},{"date" => "2016-07-24T08:34:00","version" => "2.93"},{"date" => "2016-11-29T17:31:38","version" => "2.94"},{"date" => "2018-03-16T11:14:39","version" => "2.95"},{"date" => "2018-04-07T08:45:56","version" => "2.96"},{"date" => "2018-04-21T09:13:56","version" => "2.97"},{"date" => "2018-04-21T11:50:34","version" => "2.98"},{"date" => "2018-09-13T07:11:41","version" => "3.000000"},{"date" => "2019-01-16T09:54:40","version" => "3.000001"},{"date" => "2019-03-14T13:34:40","version" => "3.000002"},{"date" => "2020-03-24T15:45:08","version" => "3.000003"}]},"Config-Model" => {"advisories" => [{"affected_versions" => ["<2.102"],"cves" => ["CVE-2017-0373"],"description" => "The gen_class_pod implementation in lib/Config/Model/Utils/GenClassPod.pm in Config-Model (aka libconfig-model-perl) before 2.102 has a dangerous \"use lib\" line, which allows remote attackers to have an unspecified impact via a crafted Debian package file.\n","distribution" => "Config-Model","fixed_versions" => [">=2.102"],"id" => "CPANSA-Config-Model-2017-03","references" => ["https://metacpan.org/changes/distribution/Config-Model","https://github.com/dod38fr/config-model/commit/9bd64d9ec6c0939166a2216a37d58dd19a725951"],"reported" => "2017-05-10"},{"affected_versions" => ["<2.102"],"cves" => ["CVE-2017-0374"],"description" => "Loads models from a local directory, making it possible to substitute the model.\n","distribution" => "Config-Model","fixed_versions" => [">=2.102"],"id" => "CPANSA-Config-Model-2017-02","references" => ["https://metacpan.org/changes/distribution/Config-Model","https://github.com/dod38fr/config-model/commit/4d37c75b0c4f9633b67999f8260b08027a6bc524"],"reported" => "2017-05-10"},{"affected_versions" => ["<2.102"],"cves" => [],"description" => "YAML or YAML::XS can be loaded automatically making it possible to run arbitrary code loading a specially crafted YAML file.\n","distribution" => "Config-Model","fixed_versions" => [">=2.102"],"id" => "CPANSA-Config-Model-2017-01","references" => ["https://metacpan.org/changes/distribution/Config-Model","https://github.com/dod38fr/config-model/commit/01d212348bfbadf31bd74aadd26b1e391ff2fd92"],"reported" => "2017-05-10"}],"main_module" => "Config::Model","versions" => [{"date" => "2006-04-21T12:27:44","version" => "0.505"},{"date" => "2006-05-19T13:32:14","version" => "0.506"},{"date" => "2006-06-15T12:10:38","version" => "0.507"},{"date" => "2006-07-20T12:28:36","version" => "0.601"},{"date" => "2006-09-07T12:06:17","version" => "0.602"},{"date" => "2006-10-19T11:24:40","version" => "0.603"},{"date" => "2006-12-06T12:58:35","version" => "0.604"},{"date" => "2007-01-08T13:16:42","version" => "0.605"},{"date" => "2007-01-11T12:42:09","version" => "0.606"},{"date" => "2007-01-12T13:06:38","version" => "0.607"},{"date" => "2007-02-23T13:00:34","version" => "0.608"},{"date" => "2007-05-14T11:41:18","version" => "0.609"},{"date" => "2007-06-06T12:28:06","version" => "0.610"},{"date" => "2007-07-03T15:35:21","version" => "0.611"},{"date" => "2007-07-27T11:38:57","version" => "0.612"},{"date" => "2007-10-01T15:52:56","version" => "0.613"},{"date" => "2007-10-23T16:10:29","version" => "0.614"},{"date" => "2007-11-15T12:36:18","version" => "0.615"},{"date" => "2007-12-04T12:41:22","version" => "0.616"},{"date" => "2008-01-28T11:55:50","version" => "0.617"},{"date" => "2008-02-14T12:56:25","version" => "0.618"},{"date" => "2008-02-29T12:08:41","version" => "0.619"},{"date" => "2008-03-18T17:40:57","version" => "0.620"},{"date" => "2008-03-20T07:49:00","version" => "0.6201"},{"date" => "2008-04-04T11:38:49","version" => "0.621"},{"date" => "2008-04-25T16:23:31","version" => "0.622"},{"date" => "2008-05-19T11:47:46","version" => "0.623"},{"date" => "2008-07-25T11:35:07","version" => "0.624"},{"date" => "2008-07-30T12:02:43","version" => "0.625"},{"date" => "2008-09-22T12:20:00","version" => "0.626"},{"date" => "2008-09-23T11:05:58","version" => "0.627"},{"date" => "2008-09-29T12:35:05","version" => "0.628"},{"date" => "2008-10-13T15:09:27","version" => "0.629"},{"date" => "2008-10-21T11:59:27","version" => "0.630"},{"date" => "2008-11-10T14:37:44","version" => "0.631"},{"date" => "2008-12-16T13:32:26","version" => "0.632"},{"date" => "2008-12-23T15:36:48","version" => "0.633"},{"date" => "2009-03-05T13:06:32","version" => "0.634"},{"date" => "2009-04-20T12:21:46","version" => "0.635"},{"date" => "2009-05-30T16:19:54","version" => "0.636"},{"date" => "2009-06-23T12:07:41","version" => "0.637"},{"date" => "2009-06-30T11:31:35","version" => "0.638"},{"date" => "2009-09-08T11:35:25","version" => "0.639"},{"date" => "2009-09-09T16:10:41","version" => "0.640"},{"date" => "2010-01-20T17:30:14","version" => "0.641"},{"date" => "2010-01-21T17:17:34","version" => "0.642"},{"date" => "2010-02-25T13:04:52","version" => "0.643"},{"date" => "2010-03-12T15:24:45","version" => "0.644"},{"date" => "2010-03-28T14:53:46","version" => "1.001"},{"date" => "2010-04-22T12:22:00","version" => "1.202"},{"date" => "2010-06-03T11:09:45","version" => "1.203"},{"date" => "2010-06-03T11:20:09","version" => "1.204"},{"date" => "2010-06-07T16:04:03","version" => "1.205"},{"date" => "2010-08-13T10:53:09","version" => "1.206"},{"date" => "2010-09-14T16:14:40","version" => "1.207"},{"date" => "2010-09-16T11:46:11","version" => "1.208"},{"date" => "2010-09-20T12:29:12","version" => "1.209"},{"date" => "2010-09-30T16:34:27","version" => "1.210"},{"date" => "2010-10-08T10:46:45","version" => "1.211"},{"date" => "2010-10-15T11:08:52","version" => "1.212"},{"date" => "2010-10-19T12:29:03","version" => "1.213"},{"date" => "2010-10-19T15:17:01","version" => "1.214"},{"date" => "2010-10-19T15:28:56","version" => "1.215"},{"date" => "2010-10-26T12:16:51","version" => "1.216"},{"date" => "2010-10-30T12:44:11","version" => "1.217"},{"date" => "2010-11-05T11:53:14","version" => "1.218"},{"date" => "2010-11-09T13:20:51","version" => "1.219"},{"date" => "2010-11-10T08:41:22","version" => "1.220"},{"date" => "2010-11-21T17:40:10","version" => "1.221"},{"date" => "2010-11-22T14:01:55","version" => "1.222"},{"date" => "2010-11-28T17:34:03","version" => "1.223"},{"date" => "2010-12-06T13:18:53","version" => "1.224"},{"date" => "2010-12-07T08:01:43","version" => "1.225"},{"date" => "2010-12-08T18:48:08","version" => "1.226"},{"date" => "2011-01-07T18:12:45","version" => "1.227"},{"date" => "2011-01-09T12:27:15","version" => "1.228"},{"date" => "2011-01-10T19:57:53","version" => "1.229"},{"date" => "2011-01-20T16:47:27","version" => "1.230"},{"date" => "2011-01-30T11:30:23","version" => "1.231"},{"date" => "2011-01-30T13:51:34","version" => "1.232"},{"date" => "2011-02-11T12:25:32","version" => "1.233"},{"date" => "2011-02-21T17:11:22","version" => "1.234"},{"date" => "2011-03-01T13:06:28","version" => "1.235"},{"date" => "2011-04-01T14:09:03","version" => "1.236"},{"date" => "2011-04-04T12:57:04","version" => "1.237"},{"date" => "2011-04-05T14:45:45","version" => "1.238"},{"date" => "2011-04-05T17:40:17","version" => "1.240"},{"date" => "2011-04-07T18:09:49","version" => "1.241"},{"date" => "2011-04-25T15:28:14","version" => "1.242"},{"date" => "2011-05-02T12:33:33","version" => "1.243"},{"date" => "2011-05-16T15:52:46","version" => "1.244"},{"date" => "2011-06-17T12:10:22","version" => "1.245"},{"date" => "2011-06-20T12:32:24","version" => "1.246"},{"date" => "2011-06-27T14:14:52","version" => "1.247"},{"date" => "2011-07-05T15:48:52","version" => "1.248"},{"date" => "2011-07-12T09:54:39","version" => "1.249"},{"date" => "2011-07-22T12:40:47","version" => "1.250"},{"date" => "2011-08-30T12:16:32","version" => "1.251"},{"date" => "2011-09-01T16:06:19","version" => "1.252"},{"date" => "2011-09-02T16:03:35","version" => "1.253"},{"date" => "2011-09-04T15:21:52","version" => "1.254"},{"date" => "2011-09-15T15:23:39","version" => "1.255"},{"date" => "2011-09-16T12:28:51","version" => "1.256"},{"date" => "2011-09-23T10:52:00","version" => "1.257"},{"date" => "2011-10-14T14:45:06","version" => "1.258"},{"date" => "2011-10-16T10:17:53","version" => "1.259"},{"date" => "2011-10-28T13:28:02","version" => "1.260"},{"date" => "2011-11-18T17:02:26","version" => "1.261"},{"date" => "2011-11-19T11:55:30","version" => "1.262"},{"date" => "2011-11-29T15:43:38","version" => "1.263"},{"date" => "2011-11-30T07:50:25","version" => "1.264"},{"date" => "2011-12-06T18:26:54","version" => "1.265"},{"date" => "2012-02-06T11:55:29","version" => "2.001"},{"date" => "2012-02-08T09:49:49","version" => "2.002"},{"date" => "2012-02-08T13:14:22","version" => "2.003"},{"date" => "2012-02-09T11:28:18","version" => "2.004"},{"date" => "2012-02-23T18:25:32","version" => "2.005"},{"date" => "2012-02-25T11:30:41","version" => "2.006"},{"date" => "2012-02-26T16:34:50","version" => "2.007"},{"date" => "2012-03-01T12:40:23","version" => "2.008"},{"date" => "2012-03-13T13:11:49","version" => "2.009"},{"date" => "2012-03-13T13:15:03","version" => "2.010"},{"date" => "2012-03-19T21:41:44","version" => "2.011"},{"date" => "2012-04-05T11:41:54","version" => "2.012"},{"date" => "2012-04-06T12:10:46","version" => "2.013"},{"date" => "2012-05-04T13:57:13","version" => "2.014"},{"date" => "2012-05-14T10:06:13","version" => "2.015"},{"date" => "2012-05-20T08:38:36","version" => "2.016"},{"date" => "2012-05-21T10:56:35","version" => "2.017"},{"date" => "2012-05-29T13:53:06","version" => "2.018"},{"date" => "2012-06-05T12:34:15","version" => "2.019"},{"date" => "2012-06-18T08:34:26","version" => "2.020"},{"date" => "2012-06-27T14:44:55","version" => "2.021_01"},{"date" => "2012-06-28T15:30:52","version" => "2.021"},{"date" => "2012-07-03T14:47:31","version" => "2.022"},{"date" => "2012-07-04T13:50:37","version" => "2.023"},{"date" => "2012-09-04T11:30:02","version" => "2.024"},{"date" => "2012-09-10T10:52:02","version" => "2.025"},{"date" => "2012-09-20T17:12:09","version" => "2.026_1"},{"date" => "2012-09-21T10:38:47","version" => "2.026_2"},{"date" => "2012-09-27T11:53:42","version" => "2.026"},{"date" => "2012-10-30T12:48:16","version" => "2.027"},{"date" => "2012-11-27T12:44:55","version" => "2.028"},{"date" => "2012-11-28T13:31:04","version" => "2.029"},{"date" => "2013-02-27T18:37:05","version" => "2.030_01"},{"date" => "2013-03-23T09:47:53","version" => "2.030"},{"date" => "2013-04-03T17:22:28","version" => "2.031"},{"date" => "2013-04-15T11:28:33","version" => "2.032"},{"date" => "2013-04-15T19:27:14","version" => "2.033"},{"date" => "2013-04-17T19:29:52","version" => "2.034"},{"date" => "2013-04-27T15:05:09","version" => "2.035"},{"date" => "2013-05-25T17:53:04","version" => "2.036"},{"date" => "2013-06-15T17:46:45","version" => "2.037"},{"date" => "2013-07-03T19:30:32","version" => "2.038"},{"date" => "2013-07-18T18:12:07","version" => "2.039"},{"date" => "2013-07-20T09:46:11","version" => "2.040"},{"date" => "2013-08-14T17:58:40","version" => "2.041"},{"date" => "2013-09-15T17:41:45","version" => "2.042"},{"date" => "2013-09-20T17:35:06","version" => "2.043"},{"date" => "2013-10-13T16:02:40","version" => "2.044"},{"date" => "2013-10-18T17:48:15","version" => "2.045"},{"date" => "2013-12-15T13:07:37","version" => "2.046"},{"date" => "2014-01-25T15:54:37","version" => "2.047"},{"date" => "2014-02-23T18:02:19","version" => "2.048"},{"date" => "2014-02-26T19:45:44","version" => "2.049"},{"date" => "2014-02-27T18:12:32","version" => "2.050"},{"date" => "2014-03-06T18:23:11","version" => "2.051"},{"date" => "2014-03-23T16:20:43","version" => "2.052"},{"date" => "2014-03-25T19:11:57","version" => "2.053"},{"date" => "2014-04-01T17:51:50","version" => "2.054"},{"date" => "2014-05-02T11:33:28","version" => "2.055"},{"date" => "2014-05-18T19:34:53","version" => "2.056"},{"date" => "2014-06-12T19:32:47","version" => "2.057"},{"date" => "2014-06-19T19:43:18","version" => "2.058"},{"date" => "2014-06-29T15:08:02","version" => "2.059"},{"date" => "2014-08-19T12:43:59","version" => "2.060"},{"date" => "2014-09-23T19:21:04","version" => "2.061"},{"date" => "2014-11-23T19:45:05","version" => "2.062"},{"date" => "2014-11-28T17:55:21","version" => "2.063"},{"date" => "2014-12-04T18:47:05","version" => "2.064"},{"date" => "2015-01-06T20:16:15","version" => "2.065"},{"date" => "2015-02-15T16:13:00","version" => "2.066"},{"date" => "2015-03-01T18:38:28","version" => "2.067"},{"date" => "2015-03-29T13:39:56","version" => "2.068"},{"date" => "2015-04-25T19:29:15","version" => "2.069"},{"date" => "2015-05-03T14:00:52","version" => "2.070"},{"date" => "2015-05-23T11:15:16","version" => "2.071"},{"date" => "2015-07-18T19:31:43","version" => "2.072"},{"date" => "2015-07-19T07:35:51","version" => "2.073"},{"date" => "2015-09-30T18:56:39","version" => "2.074"},{"date" => "2015-11-22T20:11:19","version" => "2.075"},{"date" => "2016-01-14T18:13:20","version" => "2.076"},{"date" => "2016-01-20T19:55:36","version" => "2.077"},{"date" => "2016-01-24T18:48:46","version" => "2.078"},{"date" => "2016-02-12T20:44:28","version" => "2.079"},{"date" => "2016-02-27T17:59:55","version" => "2.080"},{"date" => "2016-02-29T19:01:45","version" => "2.081"},{"date" => "2016-03-29T18:22:30","version" => "2.082"},{"date" => "2016-04-20T18:32:29","version" => "2.083"},{"date" => "2016-05-26T17:35:53","version" => "2.084"},{"date" => "2016-05-29T17:13:14","version" => "2.085"},{"date" => "2016-06-04T19:28:08","version" => "2.086"},{"date" => "2016-06-29T17:35:35","version" => "2.087"},{"date" => "2016-07-09T18:06:03","version" => "2.088"},{"date" => "2016-09-04T13:17:52","version" => "2.089"},{"date" => "2016-09-10T16:07:07","version" => "2.090"},{"date" => "2016-09-13T17:05:56","version" => "2.091"},{"date" => "2016-09-23T17:46:04","version" => "2.092"},{"date" => "2016-11-08T18:33:39","version" => "2.093"},{"date" => "2016-11-09T18:23:05","version" => "2.094"},{"date" => "2016-12-06T18:01:00","version" => "2.095"},{"date" => "2016-12-11T20:28:14","version" => "2.096"},{"date" => "2016-12-22T17:35:34","version" => "2.097"},{"date" => "2017-02-26T18:58:23","version" => "2.098"},{"date" => "2017-03-05T17:09:37","version" => "2.099"},{"date" => "2017-03-18T12:06:34","version" => "2.100"},{"date" => "2017-04-28T17:40:56","version" => "2.101"},{"date" => "2017-05-14T19:10:40","version" => "2.102"},{"date" => "2017-05-25T08:15:17","version" => "2.103"},{"date" => "2017-06-03T13:23:33","version" => "2.104"},{"date" => "2017-06-09T17:26:55","version" => "2.105"},{"date" => "2017-07-16T14:07:23","version" => "2.106"},{"date" => "2017-08-30T19:12:10","version" => "2.107"},{"date" => "2017-08-31T17:23:43","version" => "2.108"},{"date" => "2017-09-18T17:52:57","version" => "2.109"},{"date" => "2017-09-21T19:12:32","version" => "2.110"},{"date" => "2017-09-22T18:41:04","version" => "2.111"},{"date" => "2017-10-01T09:12:45","version" => "2.112"},{"date" => "2017-10-12T19:07:46","version" => "2.113"},{"date" => "2017-11-11T16:35:03","version" => "2.114"},{"date" => "2017-12-14T18:03:18","version" => "2.115"},{"date" => "2017-12-16T09:52:09","version" => "2.116"},{"date" => "2018-02-03T18:09:35","version" => "2.117"},{"date" => "2018-03-26T18:33:19","version" => "2.118"},{"date" => "2018-04-02T16:55:50","version" => "2.119"},{"date" => "2018-04-08T07:56:03","version" => "2.120"},{"date" => "2018-04-15T17:08:18","version" => "2.121"},{"date" => "2018-04-17T17:20:14","version" => "2.122"},{"date" => "2018-05-01T17:18:09","version" => "2.123"},{"date" => "2018-06-09T17:16:59","version" => "2.124"},{"date" => "2018-06-24T12:47:24","version" => "2.125"},{"date" => "2018-08-20T13:10:09","version" => "2.126"},{"date" => "2018-09-30T16:44:13","version" => "2.127"},{"date" => "2018-11-21T19:33:41","version" => "2.128"},{"date" => "2018-12-05T18:44:58","version" => "2.129"},{"date" => "2018-12-07T19:02:10","version" => "2.130"},{"date" => "2018-12-16T18:32:58","version" => "2.131"},{"date" => "2018-12-22T17:50:27","version" => "2.132"},{"date" => "2019-01-13T20:17:07","version" => "2.133"},{"date" => "2019-05-05T10:51:38","version" => "2.134"},{"date" => "2019-06-05T17:21:24","version" => "2.135"},{"date" => "2019-07-29T15:44:09","version" => "2.136"},{"date" => "2019-12-01T17:32:00","version" => "2.137"},{"date" => "2019-12-27T14:43:21","version" => "2.138"},{"date" => "2020-07-18T14:38:14","version" => "2.139"},{"date" => "2020-07-31T08:24:37","version" => "2.140"},{"date" => "2021-01-17T18:04:01","version" => "2.141"},{"date" => "2021-04-07T17:08:47","version" => "2.142"},{"date" => "2021-10-31T17:28:44","version" => "2.143"},{"date" => "2021-11-04T17:26:40","version" => "2.144"},{"date" => "2021-11-06T18:23:25","version" => "2.145"},{"date" => "2021-11-28T18:13:47","version" => "2.146"},{"date" => "2021-11-29T18:42:25","version" => "2.147"},{"date" => "2022-01-09T15:02:17","version" => "2.148"},{"date" => "2022-01-13T16:42:50","version" => "2.149"},{"date" => "2022-05-08T15:10:12","version" => "2.150"},{"date" => "2022-07-26T14:32:41","version" => "2.151"},{"date" => "2022-07-28T08:07:07","version" => "2.152"},{"date" => "2023-07-14T14:05:14","version" => "2.153"},{"date" => "2023-07-14T17:35:53","version" => "2.153"},{"date" => "2024-06-15T14:47:56","version" => "2.154"},{"date" => "2024-11-24T15:11:43","version" => "2.155"},{"date" => "2026-02-02T15:05:59","version" => "2.156"}]},"Convert-ASN1" => {"advisories" => [{"affected_versions" => ["<0.27"],"cves" => ["CVE-2013-7488"],"description" => "perl-Convert-ASN1 (aka the Convert::ASN1 module for Perl) through 0.27 allows remote attackers to cause an infinite loop via unexpected input.\n","distribution" => "Convert-ASN1","fixed_versions" => [],"id" => "CPANSA-Convert-ASN1-2013-7488","references" => ["https://github.com/gbarr/perl-Convert-ASN1/issues/14","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/ONNQSW4SSKMG5RUEFZJZA5T5R2WXEGQF/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/6V3PJEQOT47ZO77263XPGS3Y3AJROI4X/"],"reported" => "2020-04-07","severity" => "high"}],"main_module" => "Convert::ASN1","versions" => [{"date" => "2000-04-03T21:52:45","version" => "0.04"},{"date" => "2000-05-12T10:14:51","version" => "0.05"},{"date" => "2000-05-22T11:08:54","version" => "0.06"},{"date" => "2000-05-30T14:31:54","version" => "0.07"},{"date" => "2001-02-05T22:44:53","version" => "0.08"},{"date" => "2001-04-19T23:07:25","version" => "0.09"},{"date" => "2001-04-20T06:40:05","version" => "0.10"},{"date" => "2001-06-11T13:37:41","version" => "0.11"},{"date" => "2001-07-31T17:11:59","version" => "0.12"},{"date" => "2001-08-26T07:23:52","version" => "0.13"},{"date" => "2001-09-10T18:07:31","version" => "0.14"},{"date" => "2002-01-22T11:33:25","version" => "0.15"},{"date" => "2002-08-20T00:05:24","version" => "0.16"},{"date" => "2003-05-12T17:52:00","version" => "0.17"},{"date" => "2003-10-08T14:31:56","version" => "0.18"},{"date" => "2005-04-19T00:51:07","version" => "0.19"},{"date" => "2006-02-22T01:29:15","version" => "0.20"},{"date" => "2007-02-03T02:50:32","version" => "0.21"},{"date" => "2008-09-15T19:39:08","version" => "0.22"},{"date" => "2012-05-03T21:33:29","version" => "0.23"},{"date" => "2012-06-04T22:12:03","version" => "0.24"},{"date" => "2012-06-09T00:32:31","version" => "0.25"},{"date" => "2012-06-09T18:31:05","version" => "0.26"},{"date" => "2014-06-25T18:49:11","version" => "0.27"},{"date" => "2021-05-23T21:05:04","version" => "0.28"},{"date" => "2021-05-24T21:29:37","version" => "0.29"},{"date" => "2021-05-30T00:58:54","version" => "0.30"},{"date" => "2021-06-03T01:30:40","version" => "0.31"},{"date" => "2021-09-21T21:46:25","version" => "0.32"},{"date" => "2021-09-22T22:51:23","version" => "0.33"},{"date" => "2023-08-07T22:47:22","version" => "0.34"}]},"Convert-UUlib" => {"advisories" => [{"affected_versions" => ["<1.051"],"cves" => ["CVE-2005-1349"],"description" => "Buffer overflow in Convert-UUlib (Convert::UUlib) before 1.051 allows remote attackers to execute arbitrary code via a malformed parameter to a read operation.\n","distribution" => "Convert-UUlib","fixed_versions" => [],"id" => "CPANSA-Convert-UUlib-2005-1349","references" => ["http://www.gentoo.org/security/en/glsa/glsa-200504-26.xml","http://secunia.com/advisories/15130","http://www.securityfocus.com/bid/13401","http://www.mandriva.com/security/advisories?name=MDKSA-2006:022","https://exchange.xforce.ibmcloud.com/vulnerabilities/20275"],"reported" => "2005-05-02","severity" => undef}],"main_module" => "Convert::UUlib","versions" => [{"date" => "1999-05-08T20:44:28","version" => "0.03"},{"date" => "1999-05-25T19:26:16","version" => "0.05"},{"date" => "1999-07-29T21:35:05","version" => "0.06"},{"date" => "2000-07-16T20:52:56","version" => "0.11"},{"date" => "2001-05-04T21:14:40","version" => "0.111"},{"date" => "2001-06-14T16:49:29","version" => "0.2"},{"date" => "2001-09-16T01:45:11","version" => "0.201"},{"date" => "2002-03-31T22:10:15","version" => "0.21"},{"date" => "2002-03-31T22:52:00","version" => "0.21"},{"date" => "2002-04-05T22:18:55","version" => "0.211"},{"date" => "2002-04-06T02:39:32","version" => "0.212"},{"date" => "2002-07-27T19:20:26","version" => "0.213"},{"date" => "2002-10-13T18:14:28","version" => "0.3"},{"date" => "2002-10-15T23:26:09","version" => "0.31"},{"date" => "2003-11-24T16:10:49","version" => "1.0"},{"date" => "2004-03-16T20:05:14","version" => "1.01"},{"date" => "2004-04-18T14:51:27","version" => "1.02"},{"date" => "2004-04-18T20:16:15","version" => "1.03"},{"date" => "2004-12-28T14:12:40","version" => "1.04"},{"date" => "2005-03-03T17:52:16","version" => "1.051"},{"date" => "2005-12-05T23:58:50","version" => "1.06"},{"date" => "2006-12-10T16:45:11","version" => "1.07"},{"date" => "2006-12-16T22:31:30","version" => "1.08"},{"date" => "2007-05-25T17:40:35","version" => "1.09"},{"date" => "2008-06-13T13:27:38","version" => "1.10"},{"date" => "2008-06-13T13:34:18","version" => "1.11"},{"date" => "2008-10-13T12:13:26","version" => "1.12"},{"date" => "2009-08-28T23:26:34","version" => "1.3"},{"date" => "2009-09-16T07:05:05","version" => "1.31"},{"date" => "2009-09-16T18:10:46","version" => "1.32"},{"date" => "2009-10-28T08:05:40","version" => "1.33"},{"date" => "2010-12-14T21:21:33","version" => "1.34"},{"date" => "2011-05-29T15:23:57","version" => "1.4"},{"date" => "2015-07-11T01:57:19","version" => "1.5"},{"date" => "2019-10-24T15:19:15","version" => "1.6"},{"date" => "2020-02-17T22:21:21","version" => "1.62"},{"date" => "2020-02-29T21:09:26","version" => "1.7"},{"date" => "2020-03-16T23:54:43","version" => "1.71"},{"date" => "2020-12-17T01:25:02","version" => "1.8"}]},"Cpanel-JSON-XS" => {"advisories" => [{"affected_versions" => ["<3.0225"],"cves" => [],"description" => "Overflow during processing of ill-formed UTF-8 strings.\n","distribution" => "Cpanel-JSON-XS","fixed_versions" => [">=3.0225"],"id" => "CPANSA-Cpanel-JSON-XS-2016-02","references" => ["https://metacpan.org/changes/distribution/Cpanel-JSON-XS","https://github.com/rurban/Cpanel-JSON-XS/commit/f71768984ba7f50b0476c17a4f3b3f2ca88a6951","https://github.com/dankogai/p5-encode/issues/64"],"reported" => "2016-11-23"},{"affected_versions" => ["<3.0218"],"cves" => [],"description" => "Possible overflows in av and hv length types.\n","distribution" => "Cpanel-JSON-XS","fixed_versions" => [">=3.0218"],"id" => "CPANSA-Cpanel-JSON-XS-2016-01","references" => ["https://metacpan.org/changes/distribution/Cpanel-JSON-XS","https://github.com/rurban/Cpanel-JSON-XS/commit/6554531b39fac236321d8601d35eaaa75ae45e20"],"reported" => undef},{"affected_versions" => ["<4.033"],"cves" => ["CVE-2022-48623"],"description" => "Wrong error messages/sometimes crashes or endless loops with invalid JSON in relaxed mode\n","distribution" => "Cpanel-JSON-XS","fixed_versions" => [">=4.033"],"id" => "CPANSA-Cpanel-JSON-XS-2023-01","references" => ["https://metacpan.org/changes/distribution/Cpanel-JSON-XS","https://github.com/rurban/Cpanel-JSON-XS/issues/208","https://metacpan.org/release/RURBAN/Cpanel-JSON-XS-4.33/changes","https://nvd.nist.gov/vuln/detail/CVE-2022-48623","https://github.com/rurban/Cpanel-JSON-XS/commit/41f32396eee9395a40f9ed80145c37622560de9b","https://github.com/advisories/GHSA-44qr-8pf6-6q33"],"reported" => "2023-02-21"},{"affected_versions" => ["<4.40"],"cves" => ["CVE-2025-40929"],"description" => "Cpanel::JSON::XS before version 4.40 for Perl has an integer buffer overflow causing a segfault when parsing crafted JSON, enabling denial-of-service attacks or other unspecified impact","distribution" => "Cpanel-JSON-XS","fixed_versions" => [">=4.40"],"id" => "CPANSA-Cpanel-JSON-XS-2025-40929","references" => ["https://github.com/rurban/Cpanel-JSON-XS/commit/378236219eaa35742c3962ecbdee364903b0a1f2.patch","https://metacpan.org/release/RURBAN/Cpanel-JSON-XS-4.39/source/XS.xs#L713","https://metacpan.org/release/RURBAN/Cpanel-JSON-XS-4.40/changes"],"reported" => "2025-09-08","severity" => undef}],"main_module" => "Cpanel::JSON::XS","versions" => [{"date" => "2013-03-01T00:52:41","version" => "2.33_03"},{"date" => "2013-03-01T22:07:06","version" => "2.33_04"},{"date" => "2013-03-27T16:53:34","version" => "2.3305"},{"date" => "2013-03-27T17:17:51","version" => "2.3306"},{"date" => "2013-03-27T22:58:47","version" => "2.3307"},{"date" => "2013-03-28T14:28:56","version" => "2.3308"},{"date" => "2013-03-28T15:12:42","version" => "2.3309"},{"date" => "2013-03-28T17:33:21","version" => "2.3310"},{"date" => "2013-06-26T16:24:40","version" => "2.3313"},{"date" => "2013-09-09T05:54:40","version" => "2.3314"},{"date" => "2013-10-02T20:06:47","version" => "2.3401"},{"date" => "2013-11-02T14:42:20","version" => "2.3402"},{"date" => "2013-11-02T15:17:41","version" => "2.3403"},{"date" => "2014-01-30T15:58:58","version" => "2.3404"},{"date" => "2014-04-15T21:17:11","version" => "3.0101"},{"date" => "2014-04-17T18:37:34","version" => "3.0102"},{"date" => "2014-04-21T17:49:09","version" => "3.0103"},{"date" => "2014-04-26T16:04:39","version" => "3.0104"},{"date" => "2014-11-06T10:38:31","version" => "3.0105"},{"date" => "2014-11-11T21:57:49","version" => "3.0106"},{"date" => "2014-11-28T12:16:29","version" => "3.0107"},{"date" => "2014-12-11T17:02:07","version" => "3.0108"},{"date" => "2014-12-12T10:24:33","version" => "3.0109"},{"date" => "2014-12-12T22:35:37","version" => "3.0110"},{"date" => "2014-12-13T18:40:06","version" => "3.0111"},{"date" => "2014-12-14T16:34:01","version" => "3.0112"},{"date" => "2014-12-15T12:23:32","version" => "3.0113"},{"date" => "2015-01-04T14:06:03","version" => "3.0114"},{"date" => "2015-01-31T21:42:51","version" => "3.0115"},{"date" => "2015-11-26T08:58:33","version" => "3.0201"},{"date" => "2015-11-26T13:16:40","version" => "3.0202"},{"date" => "2015-11-26T13:42:02","version" => "3.0203"},{"date" => "2015-11-26T22:30:26","version" => "3.0204"},{"date" => "2015-11-29T14:09:00","version" => "3.0205"},{"date" => "2015-11-30T16:16:48","version" => "3.0206"},{"date" => "2015-12-02T16:34:35","version" => "3.0207"},{"date" => "2015-12-02T22:46:58","version" => "3.0208"},{"date" => "2015-12-03T09:45:04","version" => "3.0209"},{"date" => "2015-12-03T11:59:24","version" => "3.0210"},{"date" => "2016-01-10T17:38:25","version" => "3.0211"},{"date" => "2016-02-27T13:30:04","version" => "3.0212"},{"date" => "2016-03-02T10:28:37","version" => "3.0213"},{"date" => "2016-04-12T08:40:05","version" => "3.0213_01"},{"date" => "2016-04-13T10:40:03","version" => "3.0213_02"},{"date" => "2016-06-02T16:18:51","version" => "3.0214"},{"date" => "2016-06-06T13:28:49","version" => "3.0215"},{"date" => "2016-06-12T12:14:20","version" => "3.0216"},{"date" => "2016-06-18T09:59:27","version" => "3.0217"},{"date" => "2016-10-04T10:11:33","version" => "3.0217_01"},{"date" => "2016-10-04T14:47:29","version" => "3.0217_02"},{"date" => "2016-10-06T08:46:17","version" => "3.0217_03"},{"date" => "2016-10-07T12:11:03","version" => "3.0217_04"},{"date" => "2016-10-07T17:22:48","version" => "3.0217_05"},{"date" => "2016-10-08T08:01:50","version" => "3.0217_06"},{"date" => "2016-10-13T12:47:31","version" => "3.0218"},{"date" => "2016-10-26T11:45:35","version" => "3.0219"},{"date" => "2016-10-28T08:34:28","version" => "3.0220"},{"date" => "2016-10-30T12:27:36","version" => "3.0221"},{"date" => "2016-10-30T15:04:32","version" => "3.0222"},{"date" => "2016-11-16T11:47:38","version" => "3.0223"},{"date" => "2016-11-20T11:31:34","version" => "3.0224"},{"date" => "2016-11-23T18:43:00","version" => "3.0225"},{"date" => "2017-02-11T13:24:48","version" => "3.0226"},{"date" => "2017-02-13T10:57:06","version" => "3.0227"},{"date" => "2017-03-07T23:57:39","version" => "3.0228"},{"date" => "2017-03-10T14:08:07","version" => "3.0229"},{"date" => "2017-03-12T09:52:13","version" => "3.0230"},{"date" => "2017-03-29T09:51:51","version" => "3.0231"},{"date" => "2017-05-01T05:35:12","version" => "3.0232"},{"date" => "2017-05-01T14:54:56","version" => "3.0233"},{"date" => "2017-07-27T15:43:41","version" => "3.0234"},{"date" => "2017-07-27T16:21:47","version" => "3.0235"},{"date" => "2017-07-27T20:15:25","version" => "3.0236"},{"date" => "2017-07-28T11:15:05","version" => "3.0237"},{"date" => "2017-08-25T20:53:56","version" => "3.0238"},{"date" => "2017-08-28T20:48:37","version" => "3.0239"},{"date" => "2018-01-30T11:52:27","version" => "3.99_01"},{"date" => "2018-01-31T12:58:24","version" => "3.99_02"},{"date" => "2018-01-31T17:18:58","version" => "3.99_03"},{"date" => "2018-02-02T01:57:54","version" => "4.00"},{"date" => "2018-02-03T11:50:36","version" => "4.01"},{"date" => "2018-02-27T16:08:55","version" => "4.02"},{"date" => "2018-06-21T11:16:14","version" => "4.03"},{"date" => "2018-06-22T17:37:07","version" => "4.04"},{"date" => "2018-08-19T16:55:22","version" => "4.05"},{"date" => "2018-08-23T07:50:22","version" => "4.06"},{"date" => "2018-11-02T09:51:34","version" => "4.07"},{"date" => "2018-11-28T14:26:40","version" => "4.08"},{"date" => "2019-02-15T10:09:53","version" => "4.09"},{"date" => "2019-03-18T07:50:15","version" => "4.10"},{"date" => "2019-03-26T16:46:53","version" => "4.11"},{"date" => "2019-06-11T08:04:04","version" => "4.12"},{"date" => "2019-10-14T14:14:37","version" => "4.13"},{"date" => "2019-10-15T15:16:21","version" => "4.14"},{"date" => "2019-10-22T07:01:03","version" => "4.15"},{"date" => "2019-11-04T15:51:01","version" => "4.16"},{"date" => "2019-11-05T13:48:29","version" => "4.17"},{"date" => "2019-12-13T15:54:58","version" => "4.18"},{"date" => "2020-02-06T15:07:47","version" => "4.19"},{"date" => "2020-08-12T12:18:46","version" => "4.20"},{"date" => "2020-08-13T06:56:18","version" => "4.21"},{"date" => "2020-09-04T19:26:28","version" => "4.22"},{"date" => "2020-09-05T10:21:25","version" => "4.23"},{"date" => "2020-10-02T09:05:37","version" => "4.24"},{"date" => "2020-10-28T07:04:49","version" => "4.25"},{"date" => "2021-04-12T06:34:32","version" => "4.26"},{"date" => "2021-10-14T19:19:01","version" => "4.27"},{"date" => "2022-05-05T14:46:07","version" => "4.28"},{"date" => "2022-05-27T15:32:51","version" => "4.29"},{"date" => "2022-06-16T19:19:38","version" => "4.30"},{"date" => "2022-08-10T14:25:08","version" => "4.31"},{"date" => "2022-08-13T07:13:40","version" => "4.32"},{"date" => "2023-02-21T16:34:10","version" => "4.33"},{"date" => "2023-02-21T18:39:09","version" => "4.34"},{"date" => "2023-02-22T15:40:53","version" => "4.35"},{"date" => "2023-03-02T15:11:52","version" => "4.36"},{"date" => "2023-07-04T10:35:53","version" => "4.37"},{"date" => "2024-05-28T07:42:37","version" => "4.38"},{"date" => "2024-12-12T21:17:16","version" => "4.39"},{"date" => "2025-09-08T14:02:35","version" => "4.40"}]},"Crypt-CBC" => {"advisories" => [{"affected_versions" => ["<3.04"],"cves" => [],"description" => "Fixed bug involving manually-specified IV not being used in some circumstances.\n","distribution" => "Crypt-CBC","fixed_versions" => [">=3.04"],"id" => "CPANSA-Crypt-CBC-2021-0001","references" => ["https://metacpan.org/changes/distribution/Crypt-CBC","https://github.com/briandfoy/cpan-security-advisory/issues/165"],"reported" => "2021-05-17","severity" => undef},{"affected_versions" => ["<2.17"],"cves" => ["CVE-2006-0898"],"description" => "Crypt::CBC Perl module 2.16 and earlier, when running in RandomIV mode, uses an initialization vector (IV) of 8 bytes, which results in weaker encryption when used with a cipher that requires a larger block size than 8 bytes, such as Rijndael.\n","distribution" => "Crypt-CBC","fixed_versions" => [">=2.17"],"id" => "CPANSA-Crypt-CBC-2006-0898","references" => ["https://metacpan.org/changes/distribution/Crypt-CBC","http://www.securityfocus.com/bid/16802","http://secunia.com/advisories/18755","http://www.debian.org/security/2006/dsa-996","http://secunia.com/advisories/19187","http://www.gentoo.org/security/en/glsa/glsa-200603-15.xml","http://secunia.com/advisories/19303","http://www.novell.com/linux/security/advisories/2006_38_security.html","http://secunia.com/advisories/20899","http://securityreason.com/securityalert/488","http://www.redhat.com/support/errata/RHSA-2008-0261.html","http://secunia.com/advisories/31493","http://rhn.redhat.com/errata/RHSA-2008-0630.html","https://exchange.xforce.ibmcloud.com/vulnerabilities/24954","http://www.securityfocus.com/archive/1/425966/100/0/threaded"],"reported" => "2006-02-25","severity" => undef},{"affected_versions" => [">=1.21"],"cves" => ["CVE-2025-2814"],"description" => "Crypt::CBC versions between 1.21 and 3.04 for Perl may use the rand() function as the default source of entropy, which is not cryptographically secure, for cryptographic functions.  This issue affects operating systems where \"/dev/urandom'\" is unavailable.\x{a0} In that case, Crypt::CBC will fallback to use the insecure rand() function.","distribution" => "Crypt-CBC","fixed_versions" => [">=3.07"],"id" => "CPANSA-Crypt-CBC-2025-2814","references" => ["https://metacpan.org/dist/Crypt-CBC/source/lib/Crypt/CBC.pm#L777","https://perldoc.perl.org/functions/rand","https://security.metacpan.org/docs/guides/random-data-for-security.html","https://github.com/lstein/Lib-Crypt-CBC/issues/9"],"reported" => "2025-04-13","severity" => undef}],"main_module" => "Crypt::CBC","versions" => [{"date" => "1998-06-19T19:48:52","version" => "1.00"},{"date" => "1998-09-22T18:30:35","version" => "1.10"},{"date" => "1998-12-20T23:36:49","version" => "1.20"},{"date" => "2000-01-27T00:27:56","version" => "1.22"},{"date" => "2000-02-22T15:20:56","version" => "1.23"},{"date" => "2000-06-07T18:55:59","version" => "1.24"},{"date" => "2000-06-08T15:59:07","version" => "1.25"},{"date" => "2001-12-10T17:16:25","version" => "2.01"},{"date" => "2002-01-24T05:30:16","version" => "2.02"},{"date" => "2002-06-02T18:40:15","version" => "2.03"},{"date" => "2002-06-12T02:20:51","version" => "2.04"},{"date" => "2002-06-22T13:02:09","version" => "2.05"},{"date" => "2002-08-08T18:47:49","version" => "2.07"},{"date" => "2002-09-11T12:17:23","version" => "2.08"},{"date" => "2004-05-27T15:20:52","version" => "2.09"},{"date" => "2004-05-29T17:29:19","version" => "2.10"},{"date" => "2004-06-03T16:22:32","version" => "2.11"},{"date" => "2004-06-17T15:55:19","version" => "2.11"},{"date" => "2005-05-05T20:11:50","version" => "2.14"},{"date" => "2005-08-01T14:02:45","version" => "2.15"},{"date" => "2006-02-16T14:08:57","version" => "2.17"},{"date" => "2006-06-06T23:22:02","version" => "2.18"},{"date" => "2006-08-12T19:52:11","version" => "2.19"},{"date" => "2006-10-16T23:40:13","version" => "2.21"},{"date" => "2006-10-29T21:55:34","version" => "2.22"},{"date" => "2007-09-28T15:25:53","version" => "2.24"},{"date" => "2008-03-28T14:17:29","version" => "2.27"},{"date" => "2008-03-31T14:56:52","version" => "2.28"},{"date" => "2008-04-22T14:27:07","version" => "2.29"},{"date" => "2008-09-30T15:17:58","version" => "2.30"},{"date" => "2012-10-30T11:08:06","version" => "2.31"},{"date" => "2012-12-14T19:30:14","version" => "2.32"},{"date" => "2013-07-30T20:03:53","version" => "2.33"},{"date" => "2021-02-07T15:30:51","version" => "3.00"},{"date" => "2021-02-08T21:38:16","version" => "3.01"},{"date" => "2021-04-11T22:16:48","version" => "3.02"},{"date" => "2021-04-19T02:59:12","version" => "3.03"},{"date" => "2021-05-17T15:03:53","version" => "3.04"},{"date" => "2025-07-21T00:57:11","version" => "3.05"},{"date" => "2025-07-26T16:23:53","version" => "3.06"},{"date" => "2025-07-27T14:50:49","version" => "3.07"}]},"Crypt-DSA" => {"advisories" => [{"affected_versions" => ["<1.18"],"cves" => ["CVE-2011-3599"],"description" => "The Crypt::DSA (aka Crypt-DSA) module 1.17 and earlier for Perl, when /dev/random is absent, uses the Data::Random module, which makes it easier for remote attackers to spoof a signature, or determine the signing key of a signed message, via a brute-force attack.\n","distribution" => "Crypt-DSA","fixed_versions" => [],"id" => "CPANSA-Crypt-DSA-2011-3599","references" => ["https://rt.cpan.org/Public/Bug/Display.html?id=71421","https://bugzilla.redhat.com/show_bug.cgi?id=743567","http://www.openwall.com/lists/oss-security/2011/10/05/9","http://www.openwall.com/lists/oss-security/2011/10/05/5","http://secunia.com/advisories/46275","http://www.securityfocus.com/bid/49928","http://osvdb.org/76025"],"reported" => "2011-10-10","severity" => undef}],"main_module" => "Crypt::DSA","versions" => [{"date" => "2001-03-24T01:21:08","version" => "0.02"},{"date" => "2001-04-07T07:44:41","version" => "0.03"},{"date" => "2001-04-23T00:09:38","version" => "0.10"},{"date" => "2001-05-02T23:26:09","version" => "0.11"},{"date" => "2001-05-04T06:12:08","version" => "0.12"},{"date" => "2005-05-26T16:19:59","version" => "0.13"},{"date" => "2006-05-08T18:43:01","version" => "0.14"},{"date" => "2009-08-19T11:11:31","version" => "0.15_01"},{"date" => "2009-09-11T12:47:36","version" => "1.16"},{"date" => "2011-06-17T01:49:57","version" => "1.17"},{"date" => "2024-12-04T04:25:53","version" => "1.18"},{"date" => "2024-12-04T13:54:34","version" => "1.18"},{"date" => "2024-12-04T14:50:02","version" => "1.19"}]},"Crypt-JWT" => {"advisories" => [{"affected_versions" => ["<0.023"],"cves" => ["CVE-2019-1010263"],"description" => "Perl Crypt::JWT prior to 0.023 is affected by: Incorrect Access Control. The impact is: allow attackers to bypass authentication by providing a token by crafting with hmac(). The component is: JWT.pm, line 614. The attack vector is: network connectivity. The fixed version is: after commit b98a59b42ded9f9e51b2560410106207c2152d6c.\n","distribution" => "Crypt-JWT","fixed_versions" => [">=0.023"],"id" => "CPANSA-Crypt-JWT-2019-01","references" => ["https://github.com/DCIT/perl-Crypt-JWT/commit/b98a59b42ded9f9e51b2560410106207c2152d6c","https://www.openwall.com/lists/oss-security/2018/09/07/1"],"reported" => "2019-03-20","severity" => "high"},{"affected_versions" => ["<0.022"],"cves" => ["CVE-2019-1010161"],"description" => "perl-CRYPT-JWT 0.022 and earlier is affected by: Incorrect Access Control. The impact is: bypass authentication. The component is: JWT.pm for JWT security token, line 614 in _decode_jws(). The attack vector is: network connectivity(crafting user-controlled input to bypass authentication). The fixed version is: 0.023.\n","distribution" => "Crypt-JWT","fixed_versions" => [">=0.022"],"id" => "CPANSA-Crypt-JWT-2019-01","references" => ["https://github.com/DCIT/perl-Crypt-JWT/issues/3#issuecomment-417947483"],"reported" => "2019-03-20","severity" => "high"}],"main_module" => "Crypt::JWT","versions" => [{"date" => "2015-07-02T22:24:01","version" => "0.004"},{"date" => "2015-07-02T22:54:29","version" => "0.005"},{"date" => "2015-07-07T19:43:45","version" => "0.010"},{"date" => "2015-10-22T07:08:48","version" => "0.011"},{"date" => "2016-05-02T17:33:01","version" => "0.012"},{"date" => "2016-05-03T07:10:15","version" => "0.013"},{"date" => "2016-05-04T05:48:13","version" => "0.014"},{"date" => "2016-05-12T05:38:49","version" => "0.015"},{"date" => "2016-05-12T08:03:49","version" => "0.016"},{"date" => "2016-06-03T10:17:48","version" => "0.017"},{"date" => "2016-08-31T19:49:41","version" => "0.018"},{"date" => "2018-01-26T16:07:54","version" => "0.019"},{"date" => "2018-02-02T15:02:28","version" => "0.020"},{"date" => "2018-03-15T11:59:25","version" => "0.021"},{"date" => "2018-06-24T20:29:50","version" => "0.022"},{"date" => "2018-09-01T16:09:10","version" => "0.023"},{"date" => "2019-03-26T11:11:25","version" => "0.024"},{"date" => "2019-09-29T15:23:58","version" => "0.025"},{"date" => "2020-02-02T08:44:56","version" => "0.026"},{"date" => "2020-06-04T22:35:39","version" => "0.027"},{"date" => "2020-06-14T18:17:45","version" => "0.028"},{"date" => "2020-06-22T13:13:53","version" => "0.029"},{"date" => "2021-01-08T14:22:56","version" => "0.030"},{"date" => "2021-01-10T14:18:25","version" => "0.031"},{"date" => "2021-03-18T21:02:33","version" => "0.032"},{"date" => "2021-05-01T17:18:31","version" => "0.033"},{"date" => "2021-11-28T22:08:38","version" => "0.034"},{"date" => "2023-10-03T10:20:23","version" => "0.035"},{"date" => "2025-01-26T10:17:48","version" => "0.036"},{"date" => "2025-04-27T15:02:48","version" => "0.037"}]},"Crypt-NaCl-Sodium" => {"advisories" => [{"affected_versions" => ["<2.002"],"cves" => ["CVE-2026-2588"],"description" => "Crypt::NaCl::Sodium versions through 2.001 for Perl has an integer overflow flaw on 32-bit systems.  Sodium.xs casts a STRLEN (size_t) to unsigned long long when passing a length pointer to libsodium functions.  On 32-bit systems size_t is typically 32-bits while an unsigned long long is at least 64-bits.","distribution" => "Crypt-NaCl-Sodium","fixed_versions" => [">=2.002"],"id" => "CPANSA-Crypt-NaCl-Sodium-2026-2588","references" => ["https://github.com/cpan-authors/crypt-nacl-sodium/commit/557388bdb4da416a56663cda0154b80cd524395c.patch","https://github.com/cpan-authors/crypt-nacl-sodium/commit/8cf7f66ba922443e131c9deae1ee00fafe4f62e4.patch","https://metacpan.org/release/TIMLEGGE/Crypt-NaCl-Sodium-2.001/source/Sodium.xs#L2119"],"reported" => "2026-02-23","severity" => undef}],"main_module" => "Crypt::NaCl::Sodium","versions" => [{"date" => "2015-05-11T23:46:38","version" => "0.01"},{"date" => "2015-05-12T00:36:27","version" => "0.02"},{"date" => "2015-05-12T21:28:08","version" => "0.03"},{"date" => "2015-05-17T23:32:58","version" => "0.04"},{"date" => "2015-05-19T21:42:19","version" => "0.05"},{"date" => "2015-05-20T21:42:03","version" => "0.06"},{"date" => "2015-07-13T21:38:48","version" => "0.07"},{"date" => "2015-07-16T23:17:55","version" => "0.08"},{"date" => "2015-11-22T23:01:21","version" => "1.0.6.0"},{"date" => "2015-11-25T23:52:50","version" => "1.0.6.1"},{"date" => "2015-12-24T02:46:57","version" => "1.0.7.0"},{"date" => "2015-12-27T21:47:41","version" => "1.0.8.0"},{"date" => "2026-02-11T00:21:19","version" => "2.000"},{"date" => "2026-02-12T23:20:54","version" => "2.001"},{"date" => "2026-02-22T23:28:45","version" => "2.002"}]},"Crypt-OpenSSL-DSA" => {"advisories" => [{"affected_versions" => ["<0.14"],"cves" => ["CVE-2009-0129"],"description" => "Missing error check in do_verify, which might allow remote attackers to bypass validation of the certificate chain via a malformed SSL/TLS signature.\n","distribution" => "Crypt-OpenSSL-DSA","fixed_versions" => [">=0.14"],"id" => "CPANSA-Crypt-OpenSSL-DSA-2009-01","references" => ["https://metacpan.org/changes/distribution/Crypt-OpenSSL-DSA","https://www.openwall.com/lists/oss-security/2009/01/12/4","https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=511519"],"reported" => "2009-01-15"}],"main_module" => "Crypt::OpenSSL::DSA","versions" => [{"date" => "2001-09-19T04:45:14","version" => "0.01"},{"date" => "2001-09-24T17:32:49","version" => "0.02"},{"date" => "2002-02-07T05:57:36","version" => "0.03"},{"date" => "2002-09-24T04:52:06","version" => "0.04"},{"date" => "2002-09-26T00:21:17","version" => "0.10"},{"date" => "2003-01-06T19:08:08","version" => "0.11"},{"date" => "2005-05-23T01:44:36","version" => "0.12"},{"date" => "2005-10-15T21:37:10","version" => "0.13"},{"date" => "2012-10-16T22:55:16","version" => "0.14"},{"date" => "2015-02-03T21:57:37","version" => "0.15"},{"date" => "2016-10-27T11:25:18","version" => "0.16"},{"date" => "2016-10-27T18:54:42","version" => "0.17"},{"date" => "2016-11-17T10:33:35","version" => "0.18"},{"date" => "2017-01-13T08:24:56","version" => "0.19"},{"date" => "2021-03-20T12:31:50","version" => "0.20"}]},"Crypt-OpenSSL-RSA" => {"advisories" => [{"affected_versions" => ["<0.35"],"cves" => ["CVE-2024-2467"],"description" => "A timing-based side-channel flaw exists in the perl-Crypt-OpenSSL-RSA package, which could be sufficient to recover plaintext across a network in a Bleichenbacher-style attack. To achieve successful decryption, an attacker would have to be able to send a large number of trial messages. The vulnerability affects the legacy PKCS#1v1.5 RSA encryption padding mode.\n","distribution" => "Crypt-OpenSSL-RSA","fixed_versions" => [">=0.35"],"id" => "CPANSA-Crypt-OpenSSL-RSA-2024-2467","references" => ["https://access.redhat.com/security/cve/CVE-2024-2467","https://bugzilla.redhat.com/show_bug.cgi?id=2269567","https://github.com/toddr/Crypt-OpenSSL-RSA/issues/42","https://people.redhat.com/~hkario/marvin/"],"reported" => "2024-04-25","severity" => undef}],"main_module" => "Crypt::OpenSSL::RSA","versions" => [{"date" => "2001-04-02T04:24:26","version" => "0.08"},{"date" => "2001-04-02T16:43:12","version" => "0.09"},{"date" => "2001-04-10T20:56:43","version" => "0.10"},{"date" => "2001-04-11T02:58:01","version" => "0.11"},{"date" => "2001-09-07T03:14:26","version" => "0.12"},{"date" => "2002-03-22T04:46:17","version" => "0.13"},{"date" => "2002-05-19T16:54:53","version" => "0.14"},{"date" => "2002-06-07T13:25:40","version" => "0.15"},{"date" => "2002-06-12T02:15:47","version" => "0.16"},{"date" => "2003-01-07T04:08:58","version" => "0.17"},{"date" => "2003-02-24T02:27:12","version" => "0.18"},{"date" => "2003-04-27T22:48:40","version" => "0.19"},{"date" => "2004-02-15T20:43:41","version" => "0.20"},{"date" => "2004-02-16T02:19:45","version" => "0.21"},{"date" => "2005-06-06T10:51:24","version" => "0.22"},{"date" => "2005-11-15T04:34:15","version" => "0.22"},{"date" => "2006-04-13T04:38:04","version" => "0.23"},{"date" => "2006-11-13T15:34:59","version" => "0.24"},{"date" => "2007-05-20T19:06:56","version" => "0.25"},{"date" => "2009-11-22T20:40:31","version" => "0.26"},{"date" => "2011-06-29T18:49:35","version" => "0.26_01"},{"date" => "2011-07-03T20:14:52","version" => "0.27"},{"date" => "2011-08-24T23:04:56","version" => "0.28"},{"date" => "2017-11-27T03:36:04","version" => "0.28"},{"date" => "2018-04-14T05:01:11","version" => "0.29_01"},{"date" => "2018-04-15T18:55:41","version" => "0.29_02"},{"date" => "2018-04-16T20:47:56","version" => "0.29_03"},{"date" => "2018-05-01T16:37:12","version" => "0.30"},{"date" => "2018-09-24T17:36:24","version" => "0.31"},{"date" => "2021-09-08T15:50:47","version" => "0.32"},{"date" => "2022-07-08T11:25:11","version" => "0.33"},{"date" => "2025-05-03T12:48:15","version" => "0.34_01"},{"date" => "2025-05-04T13:50:42","version" => "0.34_02"},{"date" => "2025-05-04T14:18:26","version" => "0.34_03"},{"date" => "2025-05-05T13:44:07","version" => "0.34"},{"date" => "2025-05-07T16:52:11","version" => "0.35"},{"date" => "2025-10-29T21:22:55","version" => "0.36"},{"date" => "2025-10-29T21:41:15","version" => "0.37"}]},"Crypt-Passwd-XS" => {"advisories" => [{"affected_versions" => ["<0.601"],"cves" => ["CVE-2012-2143"],"description" => "The crypt_des (aka DES-based crypt) function does not process the complete cleartext password if this password contains a 0x80 character, which makes it easier for context-dependent attackers to obtain access via an authentication attempt with an initial substring of the intended password, as demonstrated by a Unicode password.\n","distribution" => "Crypt-Passwd-XS","fixed_versions" => [">=0.601"],"id" => "CPANSA-Crypt-Passwd-XS-2012-01","references" => ["https://metacpan.org/changes/distribution/Crypt-Passwd-XS"],"reported" => "2012-05-07"}],"main_module" => "Crypt::Passwd::XS","versions" => [{"date" => "2010-11-14T21:18:18","version" => "0.4"},{"date" => "2010-11-17T02:03:54","version" => "0.501"},{"date" => "2010-11-17T23:25:17","version" => "0.503"},{"date" => "2010-11-20T00:37:33","version" => "0.504"},{"date" => "2010-11-24T00:59:34","version" => "0.505"},{"date" => "2011-03-09T16:18:01","version" => "0.506"},{"date" => "2011-03-09T21:40:38","version" => "0.507"},{"date" => "2011-07-26T16:37:20","version" => "0.600"},{"date" => "2012-12-06T19:57:57","version" => "0.601"}]},"Crypt-Perl" => {"advisories" => [{"affected_versions" => ["<0.33"],"cves" => ["CVE-2020-17478"],"description" => "ECDSA/EC/Point.pm in Crypt::Perl before 0.33 does not properly consider timing attacks against the EC point multiplication algorithm.\n","distribution" => "Crypt-Perl","fixed_versions" => [">=0.33"],"id" => "CPANSA-Crypt-Perl-2020-01","references" => ["https://github.com/FGasper/p5-Crypt-Perl/compare/0.32...0.33"],"reported" => "2020-08-10","severity" => "high"},{"affected_versions" => ["<0.32"],"cves" => ["CVE-2020-13895"],"description" => "Crypt::Perl::ECDSA in the Crypt::Perl (aka p5-Crypt-Perl) module before 0.32 for Perl fails to verify correct ECDSA signatures when r and s are small and when s = 1. This happens when using the curve secp256r1 (prime256v1). This could conceivably have a security-relevant impact if an attacker wishes to use public r and s values when guessing whether signature verification will fail.\n","distribution" => "Crypt-Perl","fixed_versions" => [">=0.32"],"id" => "CPANSA-Crypt-Perl-2020-02","references" => ["https://github.com/FGasper/p5-Crypt-Perl/commit/f960ce75502acf7404187231a706672f8369acb2","https://github.com/FGasper/p5-Crypt-Perl/issues/14"],"reported" => "2020-06-07","severity" => "high"}],"main_module" => "Crypt::Perl","versions" => [{"date" => "2016-12-14T06:20:54","version" => "0.01"},{"date" => "2016-12-17T05:33:45","version" => "0.02"},{"date" => "2016-12-19T14:55:23","version" => "0.021"},{"date" => "2016-12-19T22:19:39","version" => "0.022"},{"date" => "2016-12-20T08:45:47","version" => "0.03"},{"date" => "2016-12-20T18:44:49","version" => "0.031"},{"date" => "2016-12-21T05:18:09","version" => "0.032"},{"date" => "2016-12-23T05:59:32","version" => "0.033"},{"date" => "2016-12-29T16:03:59","version" => "0.1"},{"date" => "2016-12-31T06:23:01","version" => "0.11"},{"date" => "2017-01-02T21:24:31","version" => "0.12"},{"date" => "2017-01-03T14:55:13","version" => "0.13"},{"date" => "2017-01-03T16:30:45","version" => "0.14"},{"date" => "2017-01-04T00:12:29","version" => "0.15"},{"date" => "2017-02-02T09:09:40","version" => "0.15_1"},{"date" => "2017-02-03T03:31:34","version" => "0.16_rc1"},{"date" => "2017-02-03T03:38:07","version" => "0.15_2"},{"date" => "2017-02-03T07:21:32","version" => "0.15_3"},{"date" => "2017-02-07T04:16:06","version" => "0.16"},{"date" => "2017-02-08T07:56:45","version" => "0.16_1"},{"date" => "2017-02-08T09:25:41","version" => "0.17"},{"date" => "2017-02-09T04:07:34","version" => "0.17_1"},{"date" => "2018-06-18T01:48:20","version" => "0.18-TRIAL1"},{"date" => "2018-06-18T02:52:52","version" => "0.18-TRIAL2"},{"date" => "2018-06-18T12:03:17","version" => "0.18-TRIAL3"},{"date" => "2018-06-18T15:07:20","version" => "0.18-TRIAL4"},{"date" => "2018-06-18T20:34:04","version" => "0.18-TRIAL5"},{"date" => "2018-06-18T21:06:07","version" => "0.18-TRIAL6"},{"date" => "2018-06-18T21:47:43","version" => "0.18-TRIAL7"},{"date" => "2018-06-18T22:42:19","version" => "0.18"},{"date" => "2018-06-19T04:25:06","version" => "0.19"},{"date" => "2018-06-19T06:14:32","version" => "0.20-TRIAL1"},{"date" => "2018-06-19T14:23:57","version" => "0.20-TRIAL2"},{"date" => "2018-06-19T15:50:08","version" => "0.20"},{"date" => "2018-06-19T15:56:15","version" => "0.21"},{"date" => "2018-06-21T03:33:59","version" => "0.22-TRIAL1"},{"date" => "2018-06-21T13:31:18","version" => "0.22-TRIAL2"},{"date" => "2018-06-22T14:43:21","version" => "0.22"},{"date" => "2018-06-23T00:40:40","version" => "0.23-TRIAL1"},{"date" => "2018-06-25T14:35:15","version" => "0.23"},{"date" => "2018-07-23T03:11:37","version" => "0.24_TRIAL1"},{"date" => "2018-07-23T03:13:05","version" => "0.24_TRIAL2"},{"date" => "2018-07-23T03:16:58","version" => "0.24-TRIAL3"},{"date" => "2018-07-23T12:12:48","version" => "0.24-TRIAL4"},{"date" => "2018-07-24T22:03:18","version" => "0.24"},{"date" => "2018-07-25T01:54:45","version" => "0.25"},{"date" => "2018-07-28T19:52:07","version" => "0.26-TRIAL1"},{"date" => "2018-07-28T22:07:05","version" => "0.26"},{"date" => "2018-07-28T22:26:02","version" => "0.27-TRIAL1"},{"date" => "2018-07-29T02:05:05","version" => "0.27"},{"date" => "2018-08-25T00:24:23","version" => "0.28"},{"date" => "2018-08-25T02:38:45","version" => "0.29"},{"date" => "2018-12-22T02:29:37","version" => "0.30-TRIAL1"},{"date" => "2018-12-22T15:18:25","version" => "0.30-TRIAL2"},{"date" => "2018-12-24T03:14:46","version" => "0.30-TRIAL3"},{"date" => "2019-09-12T03:13:59","version" => "0.30"},{"date" => "2020-02-11T00:54:58","version" => "0.31_01"},{"date" => "2020-02-11T02:50:09","version" => "0.31_02"},{"date" => "2020-02-12T01:19:36","version" => "0.31"},{"date" => "2020-06-04T12:31:25","version" => "0.32_01"},{"date" => "2020-06-04T12:56:11","version" => "0.32_02"},{"date" => "2020-06-05T02:53:59","version" => "0.32"},{"date" => "2020-08-10T15:39:12","version" => "0.33"},{"date" => "2020-09-24T07:31:56","version" => "0.34_02"},{"date" => "2020-09-24T07:37:16","version" => "0.34_03"},{"date" => "2020-09-25T01:38:34","version" => "0.34_04"},{"date" => "2020-09-26T03:44:57","version" => "0.34_05"},{"date" => "2020-09-26T12:38:56","version" => "0.34_06"},{"date" => "2020-09-26T18:03:25","version" => "0.34_07"},{"date" => "2020-09-27T13:00:26","version" => "0.34_08"},{"date" => "2020-09-27T23:51:08","version" => "0.34_09"},{"date" => "2020-09-28T07:22:06","version" => "0.34"},{"date" => "2021-11-17T15:13:58","version" => "0.35_02"},{"date" => "2021-11-18T03:39:19","version" => "0.35"},{"date" => "2021-11-18T03:44:32","version" => "0.36"},{"date" => "2021-11-20T13:20:35","version" => "0.37_01"},{"date" => "2021-11-21T03:44:48","version" => "0.37_02"},{"date" => "2021-11-22T04:28:59","version" => "0.37_03"},{"date" => "2021-11-29T02:09:35","version" => "0.37_04"},{"date" => "2021-11-30T02:16:10","version" => "0.37_05"},{"date" => "2021-11-30T16:09:14","version" => "0.37_06"},{"date" => "2021-12-01T01:39:08","version" => "0.37_07"},{"date" => "2021-12-02T21:42:02","version" => "0.37_08"},{"date" => "2021-12-06T15:38:41","version" => "0.37_09"},{"date" => "2021-12-08T01:12:53","version" => "0.37_10"},{"date" => "2021-12-09T13:42:15","version" => "0.37_11"},{"date" => "2021-12-14T02:08:33","version" => "0.37_12"},{"date" => "2021-12-15T05:19:53","version" => "0.37_13"},{"date" => "2021-12-17T19:36:38","version" => "0.37"},{"date" => "2022-10-17T15:04:13","version" => "0.38"}]},"Crypt-Primes" => {"advisories" => [{"affected_versions" => ["<0.52"],"cves" => [],"description" => "bin/largeprimes uses a custom shebang, which allows it to load modules from several locations: '..', '../lib', 'lib'. This could lead to load modules from an unpredictable location depending from where the script is run and what user is running it.\n","distribution" => "Crypt-Primes","fixed_versions" => [">=0.52"],"id" => "CPANSA-Crypt-Primes-2024-001","references" => ["https://rt.cpan.org/Public/Bug/Display.html?id=128058","https://github.com/atoomic/Crypt-Primes/pull/2","https://github.com/perl-Crypt-OpenPGP/Crypt-Primes/blob/main/Changes"],"reported" => undef,"severity" => undef}],"main_module" => "Crypt::Primes","versions" => [{"date" => "2000-11-09T23:33:04","version" => "0.38"},{"date" => "2001-03-05T09:29:12","version" => "0.46"},{"date" => "2001-06-11T09:15:28","version" => "0.49"},{"date" => "2003-01-16T20:11:04","version" => "0.50"},{"date" => "2025-01-25T02:41:34","version" => "0.51"},{"date" => "2025-01-25T13:14:32","version" => "0.52"}]},"Crypt-Random" => {"advisories" => [{"affected_versions" => [">0"],"cves" => [],"description" => "The makerandom program that comes with Crypt::Random adds module search paths in its shebang line, potentially leading to issues with unexpected modules being loaded\n","distribution" => "Crypt-Random","fixed_versions" => [],"id" => "CPANSA-Crypt-Random-2024-001","references" => ["https://metacpan.org/dist/Crypt-Random/changes","https://rt.cpan.org/Ticket/Display.html?id=128062","https://github.com/atoomic/Crypt-Random/pull/1"],"reported" => undef,"severity" => undef},{"affected_versions" => [">=1.05,<=1.55"],"cves" => ["CVE-2025-1828"],"description" => "Crypt::Random Perl package 1.05 through 1.55 may use rand() function,\x{a0}which is not\x{a0}cryptographically strong,\x{a0}for cryptographic functions.  If the Provider is not specified and /dev/urandom or an Entropy Gathering Daemon (egd) service is not available\x{a0}Crypt::Random will default to use the insecure\x{a0}Crypt::Random::rand provider.  In particular, Windows versions of perl will encounter this issue by default.","distribution" => "Crypt-Random","fixed_versions" => [">1.55"],"id" => "CPANSA-Crypt-Random-2025-1828","references" => ["https://github.com/perl-Crypt-OpenPGP/Crypt-Random/commit/1f8b29e9e89d8d083fd025152e76ec918136cc05","https://github.com/perl-Crypt-OpenPGP/Crypt-Random/pull/1","https://perldoc.perl.org/functions/rand"],"reported" => "2025-03-11","severity" => undef}],"main_module" => "Crypt::Random","versions" => [{"date" => "1999-04-09T19:08:40","version" => "0.17"},{"date" => "1999-10-13T23:27:57","version" => "0.18"},{"date" => "2000-09-18T22:56:56","version" => "0.23"},{"date" => "2001-02-14T08:55:34","version" => "0.32"},{"date" => "2001-03-05T09:27:27","version" => "0.33"},{"date" => "2001-04-17T22:01:25","version" => "0.34"},{"date" => "2001-06-22T22:11:42","version" => "1.07"},{"date" => "2001-06-23T02:22:55","version" => "1.08"},{"date" => "2001-07-09T19:07:34","version" => "1.10"},{"date" => "2001-07-12T16:02:21","version" => "1.11"},{"date" => "2003-01-08T13:41:34","version" => "1.12"},{"date" => "2003-03-11T18:44:11","version" => "1.13"},{"date" => "2004-05-21T21:18:13","version" => "1.20"},{"date" => "2004-05-24T23:04:52","version" => "1.21"},{"date" => "2004-06-01T22:58:47","version" => "1.22"},{"date" => "2004-06-02T18:52:24","version" => "1.23"},{"date" => "2005-03-07T23:05:09","version" => "1.24"},{"date" => "2005-03-07T23:18:08","version" => "1.25"},{"date" => "2018-12-22T16:21:07","version" => "1.51"},{"date" => "2018-12-22T19:30:28","version" => "1.52"},{"date" => "2021-06-03T18:19:46","version" => "1.53"},{"date" => "2021-06-03T18:31:44","version" => "1.54"},{"date" => "2025-01-30T05:20:08","version" => "1.55"},{"date" => "2025-02-05T01:49:00","version" => "1.56"},{"date" => "2025-02-10T23:28:24","version" => "1.57"}]},"Crypt-Random-Source" => {"advisories" => [{"affected_versions" => ["<=0.12"],"cves" => ["CVE-2018-25107"],"description" => "In versions prior to 0.13, rand could be used as a result of calling get_weak, or get, if no random device was available. This implies that not explicitly asking for get_strong on a non POSIX operating system (e.g. Win32 without the Win32 backend) could have resulted in non cryptographically random data.\n","distribution" => "Crypt-Random-Source","fixed_versions" => [">=0.13"],"id" => "CPANSA-Crypt-Random-Source-2024-001","references" => ["https://metacpan.org/dist/Crypt-Random-Source/changes","https://nvd.nist.gov/vuln/detail/CVE-2018-25107","https://github.com/karenetheridge/Crypt-Random-Source/pull/3","https://metacpan.org/release/ETHER/Crypt-Random-Source-0.13/changes"],"reported" => undef,"severity" => undef}],"main_module" => "Crypt::Random::Source","versions" => [{"date" => "2008-06-17T00:15:09","version" => "0.01_01"},{"date" => "2008-06-17T01:51:37","version" => "0.01"},{"date" => "2008-06-17T01:53:15","version" => "0.02"},{"date" => "2008-06-17T06:01:16","version" => "0.03"},{"date" => "2009-11-25T17:09:48","version" => "0.04"},{"date" => "2009-11-25T17:11:14","version" => "0.05"},{"date" => "2010-12-23T03:04:46","version" => "0.06"},{"date" => "2011-01-05T08:42:20","version" => "0.07"},{"date" => "2014-08-05T00:05:07","version" => "0.08"},{"date" => "2014-08-30T17:12:48","version" => "0.09"},{"date" => "2014-08-31T18:06:40","version" => "0.10"},{"date" => "2015-10-24T04:00:11","version" => "0.11"},{"date" => "2016-03-11T03:43:41","version" => "0.12"},{"date" => "2018-04-08T01:09:20","version" => "0.13"},{"date" => "2018-04-10T02:58:59","version" => "0.14"}]},"Crypt-RandomEncryption" => {"advisories" => [{"affected_versions" => [">=0.01"],"cves" => ["CVE-2024-58040"],"description" => "Crypt::RandomEncryption for Perl version 0.01 uses insecure rand() function during encryption.","distribution" => "Crypt-RandomEncryption","fixed_versions" => [],"id" => "CPANSA-Crypt-RandomEncryption-2024-58040","references" => ["https://metacpan.org/release/QWER/Crypt-RandomEncryption-0.01/source/lib/Crypt/RandomEncryption.pm#L33","https://perldoc.perl.org/functions/rand","https://security.metacpan.org/docs/guides/random-data-for-security.html"],"reported" => "2025-09-30","severity" => undef}],"main_module" => "Crypt::RandomEncryption","versions" => [{"date" => "2013-05-16T18:07:28","version" => "0.01"}]},"Crypt-Salt" => {"advisories" => [{"affected_versions" => ["<=0.01"],"cves" => ["CVE-2025-1805"],"description" => "Crypt::Salt for Perl version 0.01 uses insecure rand() function when generating salts for cryptographic purposes.","distribution" => "Crypt-Salt","fixed_versions" => [],"id" => "CPANSA-Crypt-Salt-2025-1805","references" => ["https://metacpan.org/release/HACHI/Crypt-Salt-0.01/source/lib/Crypt/Salt.pm#L76","https://perldoc.perl.org/functions/rand","https://security.metacpan.org/docs/guides/random-data-for-security.html"],"reported" => "2025-04-02","severity" => undef}],"main_module" => "Crypt::Salt","versions" => [{"date" => "2003-11-12T06:13:44","version" => "0.01"}]},"Crypt-Sodium-XS" => {"advisories" => [{"affected_versions" => ["<0.000042"],"cves" => ["CVE-2025-15444"],"description" => "Crypt::Sodium::XS module versions prior to\x{a0}0.000042,\x{a0}for Perl, include a vulnerable version of libsodium  libsodium <= 1.0.20 or a version of libsodium released before December 30, 2025 contains a vulnerability documented as CVE-2025-69277\x{a0} https://www.cve.org/CVERecord?id=CVE-2025-69277 .  The libsodium vulnerability states:  In atypical use cases involving certain custom cryptography or untrusted data to crypto_core_ed25519_is_valid_point, mishandles checks for whether an elliptic curve point is valid because it sometimes allows points that aren't in the main cryptographic group.  0.000042 includes a version of\x{a0}libsodium updated to 1.0.20-stable, released January 3, 2026, which includes a fix for the vulnerability.","distribution" => "Crypt-Sodium-XS","embedded_vulnerability" => {"distributed_version" => "<-1.0.20","name" => "libsodium"},"fixed_versions" => [">=0.000042"],"id" => "CPANSA-Crypt-Sodium-XS-2025-15444","references" => ["https://00f.net/2025/12/30/libsodium-vulnerability/","https://github.com/jedisct1/libsodium/commit/ad3004ec8731730e93fcfbbc824e67eadc1c1bae","https://metacpan.org/dist/Crypt-Sodium-XS/changes"],"reported" => "2026-01-06","severity" => undef}],"main_module" => "Crypt::Sodium::XS","versions" => [{"date" => "2025-07-06T21:01:50","version" => "0.000018"},{"date" => "2025-07-07T16:16:33","version" => "0.000019"},{"date" => "2025-07-08T01:32:47","version" => "0.000020"},{"date" => "2025-07-08T02:22:03","version" => "0.000021"},{"date" => "2025-07-08T06:45:10","version" => "0.000022"},{"date" => "2025-07-08T22:14:02","version" => "0.000024"},{"date" => "2025-07-09T20:24:46","version" => "0.000025"},{"date" => "2025-07-09T21:42:18","version" => "0.000026"},{"date" => "2025-07-12T16:33:41","version" => "0.000027"},{"date" => "2025-07-15T19:55:23","version" => "0.000028"},{"date" => "2025-07-17T00:10:50","version" => "0.000029"},{"date" => "2025-07-19T21:47:23","version" => "0.000030"},{"date" => "2025-07-21T15:02:40","version" => "0.000031"},{"date" => "2025-07-23T23:48:57","version" => "0.000032"},{"date" => "2025-07-24T20:46:46","version" => "0.000033"},{"date" => "2025-08-01T19:01:56","version" => "0.000034"},{"date" => "2025-08-02T00:09:58","version" => "0.000035"},{"date" => "2025-08-03T14:55:38","version" => "0.000036"},{"date" => "2025-08-05T21:41:13","version" => "0.000037"},{"date" => "2025-08-09T18:31:17","version" => "0.000038"},{"date" => "2025-08-21T06:03:08","version" => "0.000039"},{"date" => "2025-12-04T06:38:40","version" => "0.000040"},{"date" => "2025-12-05T05:32:25","version" => "0.000041"},{"date" => "2026-01-04T09:58:53","version" => "0.000042"},{"date" => "2026-01-21T04:10:41","version" => "0.001000"}]},"Crypt-SysRandom-XS" => {"advisories" => [{"affected_versions" => ["<0.010"],"cves" => ["CVE-2026-2597"],"description" => "Crypt::SysRandom::XS versions before 0.010 for Perl is vulnerable to a heap buffer overflow in the XS function random_bytes().  The function does not validate that the length parameter is non-negative. If a negative value (e.g. -1) is supplied, the expression length + 1u causes an integer wraparound, resulting in a zero-byte allocation. The subsequent call to chosen random function (e.g. getrandom) passes the original negative value, which is implicitly converted to a large unsigned value (typically SIZE_MAX). This can result in writes beyond the allocated buffer, leading to heap memory corruption and application crash (denial of service).  In common usage, the length argument is typically hardcoded by the caller, which reduces the likelihood of attacker-controlled exploitation. Applications that pass untrusted input to this parameter may be affected.","distribution" => "Crypt-SysRandom-XS","fixed_versions" => [">=0.010"],"id" => "CPANSA-Crypt-SysRandom-XS-2026-2597","references" => ["https://metacpan.org/dist/Crypt-SysRandom-XS/changes","https://metacpan.org/release/LEONT/Crypt-SysRandom-XS-0.011/source/lib/Crypt/SysRandom/XS.xs#L51-52"],"reported" => "2026-02-27","severity" => undef}],"main_module" => "Crypt::SysRandom::XS","versions" => [{"date" => "2025-02-04T01:59:42","version" => "0.006"},{"date" => "2025-02-05T19:46:04","version" => "0.007"},{"date" => "2025-02-20T12:52:45","version" => "0.008"},{"date" => "2025-04-11T16:46:48","version" => "0.009"},{"date" => "2026-02-16T20:43:40","version" => "0.010"},{"date" => "2026-02-16T23:58:52","version" => "0.011"}]},"Crypt-URandom" => {"advisories" => [{"affected_versions" => [">=0.41,<0.55"],"cves" => ["CVE-2026-2474"],"description" => "Crypt::URandom versions from 0.41 before 0.55 for Perl is vulnerable to a heap buffer overflow in the XS function crypt_urandom_getrandom().  The function does not validate that the length parameter is non-negative. If a negative value (e.g. -1) is supplied, the expression length + 1u causes an integer wraparound, resulting in a zero-byte allocation. The subsequent call to getrandom(data, length, GRND_NONBLOCK) passes the original negative value, which is implicitly converted to a large unsigned value (typically SIZE_MAX). This can result in writes beyond the allocated buffer, leading to heap memory corruption and application crash (denial of service).  In common usage, the length argument is typically hardcoded by the caller, which reduces the likelihood of attacker-controlled exploitation. Applications that pass untrusted input to this parameter may be affected.","distribution" => "Crypt-URandom","fixed_versions" => [">=0.55"],"id" => "CPANSA-Crypt-URandom-2026-2474","references" => ["https://metacpan.org/release/DDICK/Crypt-URandom-0.54/source/URandom.xs#L35-79","https://metacpan.org/release/DDICK/Crypt-URandom-0.55/source/Changes"],"reported" => "2026-02-16","severity" => undef}],"main_module" => "Crypt::URandom","versions" => [{"date" => "2011-04-19T21:19:07","version" => "0.0.27"},{"date" => "2011-09-03T08:32:28","version" => "0.28"},{"date" => "2013-02-14T08:52:19","version" => "0.29"},{"date" => "2013-02-26T10:34:54","version" => "0.30"},{"date" => "2013-03-01T07:59:28","version" => "0.31"},{"date" => "2013-03-09T04:25:55","version" => "0.32"},{"date" => "2013-03-09T04:32:35","version" => "0.33"},{"date" => "2013-03-10T09:20:53","version" => "0.34"},{"date" => "2015-05-13T20:12:12","version" => "0.35"},{"date" => "2015-06-01T11:45:38","version" => "0.36"},{"date" => "2023-05-10T11:55:21","version" => "0.37"},{"date" => "2023-05-10T19:57:07","version" => "0.38"},{"date" => "2023-05-21T09:05:01","version" => "0.39"},{"date" => "2024-02-24T09:03:35","version" => "0.40"},{"date" => "2024-12-29T23:34:13","version" => "0.41_01"},{"date" => "2024-12-30T00:00:05","version" => "0.41_02"},{"date" => "2025-01-01T22:58:00","version" => "0.41"},{"date" => "2025-01-02T23:56:10","version" => "0.42"},{"date" => "2025-01-03T09:04:23","version" => "0.43"},{"date" => "2025-01-03T20:47:27","version" => "0.44"},{"date" => "2025-01-03T22:28:26","version" => "0.45"},{"date" => "2025-01-04T08:47:50","version" => "0.46"},{"date" => "2025-01-06T10:45:06","version" => "0.47"},{"date" => "2025-01-06T21:08:58","version" => "0.48"},{"date" => "2025-01-07T21:28:34","version" => "0.49"},{"date" => "2025-01-08T21:56:14","version" => "0.50"},{"date" => "2025-01-19T07:28:53","version" => "0.51_01"},{"date" => "2025-01-19T07:57:30","version" => "0.51_02"},{"date" => "2025-01-22T11:25:07","version" => "0.51"},{"date" => "2025-01-22T19:39:34","version" => "0.52"},{"date" => "2025-02-08T09:07:55","version" => "0.53"},{"date" => "2025-03-15T09:46:36","version" => "0.54"},{"date" => "2026-02-16T20:08:04","version" => "0.55"}]},"CryptX" => {"advisories" => [{"affected_versions" => ["<0.062"],"cves" => ["CVE-2018-25099"],"description" => "A user can pass anything as the tag into gcm_decrypt_verify() and it will return decrypted plaintext.\n","distribution" => "CryptX","fixed_versions" => [">=0.062"],"id" => "CPANSA-CryptX-2018-01","references" => ["https://github.com/DCIT/perl-CryptX/issues/47","https://github.com/libtom/libtomcrypt/pull/451"],"reported" => "2018-10-26","severity" => undef},{"affected_versions" => ["<0.065"],"cves" => ["CVE-2025-40912","CVE-2019-17362"],"description" => "CryptX for Perl before version 0.065 contains a dependency that may be susceptible to malformed unicode.  CryptX embeds the tomcrypt library. The versions of that library in CryptX before 0.065 may be susceptible to CVE-2019-17362.","distribution" => "CryptX","embedded_vulnerability" => {"distributed_version" => undef,"name" => "libtomcrypt"},"fixed_versions" => [">=0.065"],"id" => "CPANSA-CryptX-2025-40912","references" => ["https://github.com/libtom/libtomcrypt/issues/507"],"reported" => "2025-06-11","severity" => undef},{"affected_versions" => ["<0.087"],"cves" => ["CVE-2025-40914","CVE-2023-36328"],"description" => "Perl CryptX before version 0.087 contains a dependency that may be susceptible to an integer overflow.  CryptX embeds a version of the libtommath library that is susceptible to an integer overflow associated with CVE-2023-36328.","distribution" => "CryptX","embedded_vulnerability" => {"distributed_version" => undef,"name" => "libtommath"},"fixed_versions" => [">=0.087"],"id" => "CPANSA-CryptX-2025-40914","references" => ["https://github.com/advisories/GHSA-j3xv-6967-cv88","https://github.com/libtom/libtommath/pull/546","https://metacpan.org/release/MIK/CryptX-0.086/source/src/ltm/bn_mp_grow.c","https://www.cve.org/CVERecord?id=CVE-2023-36328"],"reported" => "2025-06-11","severity" => undef}],"main_module" => "CryptX","versions" => [{"date" => "2013-04-11T15:10:25","version" => "0.002"},{"date" => "2013-04-16T07:29:07","version" => "0.003"},{"date" => "2013-04-16T07:37:29","version" => "0.004"},{"date" => "2013-04-18T12:09:09","version" => "0.005"},{"date" => "2013-04-19T09:05:57","version" => "0.006"},{"date" => "2013-04-23T10:03:54","version" => "0.007"},{"date" => "2013-05-02T08:52:42","version" => "0.008"},{"date" => "2013-05-19T20:25:17","version" => "0.009"},{"date" => "2013-06-14T22:37:18","version" => "0.011"},{"date" => "2013-06-17T20:00:17","version" => "0.012"},{"date" => "2013-08-28T07:12:15","version" => "0.013"},{"date" => "2013-09-11T14:31:01","version" => "0.013_1"},{"date" => "2013-09-11T21:11:18","version" => "0.014"},{"date" => "2013-09-12T08:21:21","version" => "0.014_1"},{"date" => "2013-09-12T09:12:21","version" => "0.014_2"},{"date" => "2013-09-12T09:32:06","version" => "0.015"},{"date" => "2013-09-15T19:36:52","version" => "0.016"},{"date" => "2013-09-22T19:20:20","version" => "0.016_1"},{"date" => "2013-09-23T14:24:27","version" => "0.016_2"},{"date" => "2013-09-24T05:00:07","version" => "0.016_3"},{"date" => "2013-09-24T08:54:42","version" => "0.017"},{"date" => "2013-10-18T13:21:35","version" => "0.018"},{"date" => "2013-10-20T21:13:35","version" => "0.019"},{"date" => "2014-01-14T21:30:37","version" => "0.019_1"},{"date" => "2014-01-15T21:59:30","version" => "0.019_2"},{"date" => "2014-01-16T21:26:19","version" => "0.019_3"},{"date" => "2014-01-17T09:42:29","version" => "0.019_4"},{"date" => "2014-01-18T08:29:06","version" => "0.020"},{"date" => "2014-01-23T15:47:50","version" => "0.021"},{"date" => "2015-04-17T01:18:47","version" => "0.021_1"},{"date" => "2015-04-17T10:12:01","version" => "0.021_3"},{"date" => "2015-05-06T07:55:21","version" => "0.021_4"},{"date" => "2015-05-11T21:56:39","version" => "0.021_5"},{"date" => "2015-05-12T05:52:15","version" => "0.021_6"},{"date" => "2015-05-12T07:10:42","version" => "0.021_7"},{"date" => "2015-05-14T09:27:41","version" => "0.021_8"},{"date" => "2015-05-15T11:08:26","version" => "0.021_9"},{"date" => "2015-05-22T16:57:26","version" => "0.022"},{"date" => "2015-06-10T20:37:54","version" => "0.023"},{"date" => "2015-06-26T06:13:25","version" => "0.023_1"},{"date" => "2015-06-26T09:18:06","version" => "0.023_2"},{"date" => "2015-06-29T20:09:16","version" => "0.024"},{"date" => "2015-07-08T07:56:50","version" => "0.025"},{"date" => "2015-10-22T15:14:53","version" => "0.025_01"},{"date" => "2015-11-28T18:58:17","version" => "0.026"},{"date" => "2015-12-29T21:20:15","version" => "0.026_01"},{"date" => "2015-12-29T21:20:27","version" => "0.026_02"},{"date" => "2016-01-02T14:38:13","version" => "0.026_05"},{"date" => "2016-01-02T15:57:58","version" => "0.026_06"},{"date" => "2016-01-03T14:36:53","version" => "0.026_08"},{"date" => "2016-01-10T09:47:31","version" => "0.026_15"},{"date" => "2016-01-10T09:47:43","version" => "0.026_16"},{"date" => "2016-01-10T11:52:21","version" => "0.026_18"},{"date" => "2016-01-10T11:53:48","version" => "0.026_19"},{"date" => "2016-01-10T17:03:45","version" => "0.026_23"},{"date" => "2016-01-10T17:03:56","version" => "0.026_24"},{"date" => "2016-01-10T19:02:14","version" => "0.026_28"},{"date" => "2016-01-10T20:49:06","version" => "0.026_29"},{"date" => "2016-01-12T22:25:58","version" => "0.026_30"},{"date" => "2016-01-12T22:37:33","version" => "0.026_31"},{"date" => "2016-01-13T23:53:06","version" => "0.026_32"},{"date" => "2016-01-14T19:29:18","version" => "0.026_33"},{"date" => "2016-01-14T19:30:45","version" => "0.026_34"},{"date" => "2016-01-14T21:16:15","version" => "0.026_35"},{"date" => "2016-01-14T21:19:01","version" => "0.026_36"},{"date" => "2016-01-22T23:01:16","version" => "0.026_39"},{"date" => "2016-01-24T22:12:32","version" => "0.026_40"},{"date" => "2016-01-24T22:36:42","version" => "0.026_41"},{"date" => "2016-01-25T20:44:46","version" => "0.027"},{"date" => "2016-01-25T21:42:58","version" => "0.027_01"},{"date" => "2016-01-26T10:06:26","version" => "0.027_05"},{"date" => "2016-03-15T09:51:01","version" => "0.027_06"},{"date" => "2016-03-23T19:44:54","version" => "0.028"},{"date" => "2016-03-28T14:31:13","version" => "0.028_01"},{"date" => "2016-03-28T19:32:58","version" => "0.028_02"},{"date" => "2016-03-31T12:07:46","version" => "0.028_03"},{"date" => "2016-04-13T09:30:27","version" => "0.029"},{"date" => "2016-04-13T09:46:59","version" => "0.030"},{"date" => "2016-05-01T16:53:05","version" => "0.031"},{"date" => "2016-05-04T17:45:30","version" => "0.032"},{"date" => "2016-05-09T20:20:49","version" => "0.033"},{"date" => "2016-05-10T22:31:32","version" => "0.034"},{"date" => "2016-06-03T10:17:59","version" => "0.035"},{"date" => "2016-06-07T19:22:05","version" => "0.036"},{"date" => "2016-06-16T17:04:27","version" => "0.037"},{"date" => "2016-07-06T18:27:46","version" => "0.038"},{"date" => "2016-08-03T05:53:42","version" => "0.039"},{"date" => "2016-09-12T08:42:39","version" => "0.040"},{"date" => "2016-10-12T09:32:48","version" => "0.041"},{"date" => "2016-10-19T10:25:05","version" => "0.041_001"},{"date" => "2016-11-02T09:00:59","version" => "0.041_002"},{"date" => "2016-11-02T09:19:09","version" => "0.041_003"},{"date" => "2016-11-12T15:21:01","version" => "0.042"},{"date" => "2016-11-27T21:19:27","version" => "0.043"},{"date" => "2016-11-28T07:45:32","version" => "0.044"},{"date" => "2017-02-21T21:54:33","version" => "0.044_001"},{"date" => "2017-02-23T15:58:42","version" => "0.044_003"},{"date" => "2017-02-23T20:35:46","version" => "0.044_004"},{"date" => "2017-02-23T20:44:50","version" => "0.044_005"},{"date" => "2017-02-28T12:22:27","version" => "0.044_006"},{"date" => "2017-02-28T13:58:51","version" => "0.044_007"},{"date" => "2017-03-01T09:26:34","version" => "0.044_008"},{"date" => "2017-03-01T09:49:29","version" => "0.044_009"},{"date" => "2017-03-01T10:02:35","version" => "0.044_010"},{"date" => "2017-03-31T09:28:10","version" => "0.045"},{"date" => "2017-04-04T09:08:33","version" => "0.046"},{"date" => "2017-04-05T20:09:35","version" => "0.047"},{"date" => "2017-04-07T18:22:15","version" => "0.047_001"},{"date" => "2017-04-07T21:40:24","version" => "0.047_002"},{"date" => "2017-04-10T08:16:03","version" => "0.047_003"},{"date" => "2017-04-24T15:23:29","version" => "0.047_004"},{"date" => "2017-04-26T15:36:02","version" => "0.047_005"},{"date" => "2017-05-01T19:11:50","version" => "0.047_006"},{"date" => "2017-05-31T20:22:56","version" => "0.048"},{"date" => "2017-07-09T19:38:38","version" => "0.048_001"},{"date" => "2017-07-14T17:43:25","version" => "0.048_002"},{"date" => "2017-07-18T05:56:42","version" => "0.049"},{"date" => "2017-07-18T20:37:45","version" => "0.050"},{"date" => "2017-08-08T08:14:05","version" => "0.051"},{"date" => "2017-09-15T12:32:56","version" => "0.053"},{"date" => "2017-09-19T07:51:19","version" => "0.053_001"},{"date" => "2017-09-19T18:46:56","version" => "0.053_002"},{"date" => "2017-09-20T09:56:04","version" => "0.053_003"},{"date" => "2017-10-10T21:04:53","version" => "0.053_004"},{"date" => "2017-10-12T07:27:42","version" => "0.054"},{"date" => "2017-10-23T13:18:12","version" => "0.054_001"},{"date" => "2017-10-23T17:44:49","version" => "0.054_002"},{"date" => "2017-10-25T07:43:53","version" => "0.054_003"},{"date" => "2017-10-30T17:53:14","version" => "0.054_004"},{"date" => "2017-10-31T18:27:22","version" => "0.054_005"},{"date" => "2017-11-20T18:51:03","version" => "0.054_006"},{"date" => "2017-11-24T08:15:31","version" => "0.054_007"},{"date" => "2017-11-24T14:21:46","version" => "0.054_008"},{"date" => "2017-11-24T16:33:40","version" => "0.054_009"},{"date" => "2017-11-28T10:19:52","version" => "0.055"},{"date" => "2017-12-18T19:05:35","version" => "0.055_001"},{"date" => "2017-12-22T13:22:16","version" => "0.056"},{"date" => "2018-01-26T16:05:07","version" => "0.056_001"},{"date" => "2018-01-29T06:18:08","version" => "0.056_002"},{"date" => "2018-01-29T10:02:58","version" => "0.056_003"},{"date" => "2018-01-29T23:05:27","version" => "0.056_004"},{"date" => "2018-01-30T10:23:40","version" => "0.056_005"},{"date" => "2018-01-30T14:11:33","version" => "0.056_006"},{"date" => "2018-01-30T16:08:38","version" => "0.056_007"},{"date" => "2018-01-30T16:29:41","version" => "0.056_008"},{"date" => "2018-01-30T16:43:48","version" => "0.056_009"},{"date" => "2018-01-31T08:56:12","version" => "0.057"},{"date" => "2018-02-27T17:13:52","version" => "0.058"},{"date" => "2018-03-08T09:30:22","version" => "0.058_001"},{"date" => "2018-03-18T16:27:43","version" => "0.058_002"},{"date" => "2018-03-25T15:45:36","version" => "0.059"},{"date" => "2018-04-27T17:14:03","version" => "0.059_001"},{"date" => "2018-04-28T20:59:58","version" => "0.059_002"},{"date" => "2018-04-29T18:12:50","version" => "0.059_003"},{"date" => "2018-05-01T09:32:27","version" => "0.060"},{"date" => "2018-05-27T19:05:34","version" => "0.060_001"},{"date" => "2018-05-28T07:18:37","version" => "0.060_002"},{"date" => "2018-06-06T15:49:28","version" => "0.060_003"},{"date" => "2018-06-07T05:25:50","version" => "0.061"},{"date" => "2018-10-24T20:35:24","version" => "0.061_001"},{"date" => "2018-10-26T17:10:16","version" => "0.061_002"},{"date" => "2018-10-29T10:46:25","version" => "0.061_003"},{"date" => "2018-10-30T06:27:48","version" => "0.062"},{"date" => "2018-11-22T10:43:01","version" => "0.062_001"},{"date" => "2018-11-28T10:48:28","version" => "0.063"},{"date" => "2019-06-06T09:36:14","version" => "0.063_001"},{"date" => "2019-06-06T17:35:59","version" => "0.063_002"},{"date" => "2019-06-10T17:24:53","version" => "0.063_003"},{"date" => "2019-06-12T13:33:28","version" => "0.063_004"},{"date" => "2019-06-12T23:12:09","version" => "0.063_005"},{"date" => "2019-06-14T07:01:03","version" => "0.064"},{"date" => "2019-10-19T18:49:19","version" => "0.065"},{"date" => "2019-10-20T16:30:22","version" => "0.066"},{"date" => "2020-01-26T20:23:46","version" => "0.066_001"},{"date" => "2020-01-30T10:21:29","version" => "0.066_002"},{"date" => "2020-02-01T13:24:27","version" => "0.067"},{"date" => "2020-03-08T19:21:55","version" => "0.067_001"},{"date" => "2020-03-10T13:04:08","version" => "0.068"},{"date" => "2020-08-02T08:51:06","version" => "0.068_001"},{"date" => "2020-08-25T07:12:43","version" => "0.069"},{"date" => "2021-02-12T14:44:41","version" => "0.070"},{"date" => "2021-03-30T09:39:33","version" => "0.071"},{"date" => "2021-04-29T08:23:01","version" => "0.072"},{"date" => "2021-07-12T16:40:01","version" => "0.072_001"},{"date" => "2021-07-13T07:03:12","version" => "0.072_002"},{"date" => "2021-07-13T20:54:22","version" => "0.072_003"},{"date" => "2021-07-18T12:16:09","version" => "0.073"},{"date" => "2021-10-04T18:34:39","version" => "0.073_001"},{"date" => "2021-10-10T18:41:04","version" => "0.073_002"},{"date" => "2021-10-13T18:32:43","version" => "0.073_003"},{"date" => "2021-11-06T09:26:22","version" => "0.074"},{"date" => "2021-12-25T09:39:17","version" => "0.075"},{"date" => "2022-01-01T00:36:25","version" => "0.075_001"},{"date" => "2022-01-01T13:19:24","version" => "0.075_002"},{"date" => "2022-01-01T19:48:49","version" => "0.075_003"},{"date" => "2022-01-07T20:55:06","version" => "0.076"},{"date" => "2022-06-09T18:18:34","version" => "0.076_001"},{"date" => "2022-08-20T15:42:12","version" => "0.076_002"},{"date" => "2022-08-20T18:14:10","version" => "0.076_003"},{"date" => "2022-08-21T07:46:06","version" => "0.077"},{"date" => "2023-04-28T12:31:25","version" => "0.078"},{"date" => "2023-07-25T18:36:58","version" => "0.078_001"},{"date" => "2023-10-01T12:20:32","version" => "0.079"},{"date" => "2023-10-01T17:35:55","version" => "0.079_002"},{"date" => "2023-10-01T17:36:06","version" => "0.079_003"},{"date" => "2023-10-02T07:47:50","version" => "0.079_004"},{"date" => "2023-10-02T11:22:48","version" => "0.079_005"},{"date" => "2023-10-02T15:06:17","version" => "0.079_006"},{"date" => "2023-10-03T10:16:25","version" => "0.079_007"},{"date" => "2023-10-04T11:07:16","version" => "0.080"},{"date" => "2023-10-07T11:45:30","version" => "0.080_001"},{"date" => "2024-08-17T10:06:21","version" => "0.080_003"},{"date" => "2024-08-17T17:16:06","version" => "0.080_004"},{"date" => "2024-08-17T20:28:14","version" => "0.080_005"},{"date" => "2024-08-30T18:43:56","version" => "0.080_006"},{"date" => "2024-09-01T08:32:21","version" => "0.080_007"},{"date" => "2024-09-01T09:26:40","version" => "0.080_008"},{"date" => "2024-09-01T11:23:19","version" => "0.080_009"},{"date" => "2024-09-02T14:51:29","version" => "0.080_010"},{"date" => "2024-09-03T11:32:03","version" => "0.080_011"},{"date" => "2024-09-03T18:01:58","version" => "0.080_012"},{"date" => "2024-09-08T16:12:50","version" => "0.081"},{"date" => "2024-10-03T11:12:24","version" => "0.081_001"},{"date" => "2024-10-07T13:31:29","version" => "0.082"},{"date" => "2024-10-14T11:36:41","version" => "0.082_001"},{"date" => "2024-10-15T09:31:49","version" => "0.083"},{"date" => "2024-10-15T15:09:00","version" => "0.083_001"},{"date" => "2024-10-16T11:23:26","version" => "0.084"},{"date" => "2025-01-25T22:45:03","version" => "0.084_001"},{"date" => "2025-02-08T10:02:22","version" => "0.085"},{"date" => "2025-02-20T21:06:09","version" => "0.085_001"},{"date" => "2025-04-27T15:46:56","version" => "0.085_002"},{"date" => "2025-04-27T17:37:48","version" => "0.085_003"},{"date" => "2025-05-02T21:40:16","version" => "0.086"},{"date" => "2025-06-08T22:06:49","version" => "0.086_001"},{"date" => "2025-06-09T18:09:54","version" => "0.086_002"},{"date" => "2025-06-09T21:44:43","version" => "0.086_003"},{"date" => "2025-06-10T05:57:40","version" => "0.086_004"},{"date" => "2025-06-11T10:52:53","version" => "0.086_005"},{"date" => "2025-06-11T13:52:26","version" => "0.087"},{"date" => "2025-10-05T16:50:53","version" => "0.087_001"}]},"DBD-MariaDB" => {"advisories" => [{"affected_versions" => ["<1.00"],"cves" => ["CVE-2018-2767"],"description" => "SSL problems of MySQL and MariaDB clients.\n","distribution" => "DBD-MariaDB","fixed_versions" => [">=1.00"],"id" => "CPANSA-DBD-MariaDB-2018-01","references" => ["https://metacpan.org/changes/distribution/DBD-MariaDB"],"reported" => "2017-07-01"},{"affected_versions" => ["<1.00"],"cves" => ["CVE-2017-10788"],"description" => "Use-after-free after calling mysql_stmt_close().\n","distribution" => "DBD-MariaDB","fixed_versions" => [">=1.00"],"id" => "CPANSA-DBD-MariaDB-2017-02","references" => ["https://metacpan.org/changes/distribution/DBD-MariaDB"],"reported" => "2017-07-01"},{"affected_versions" => ["<1.00"],"cves" => ["CVE-2017-3302"],"description" => "Leaking dangling pointers.\n","distribution" => "DBD-MariaDB","fixed_versions" => [">=1.00"],"id" => "CPANSA-DBD-MariaDB-2017-01","references" => ["https://metacpan.org/changes/distribution/DBD-MariaDB"],"reported" => "2017-07-01"}],"main_module" => "DBD::MariaDB","versions" => [{"date" => "2018-06-26T14:23:29","version" => "0.90_01"},{"date" => "2018-07-12T13:36:05","version" => "1.00"},{"date" => "2018-12-05T12:21:26","version" => "1.10"},{"date" => "2019-01-02T15:38:57","version" => "1.11"},{"date" => "2019-02-22T16:31:33","version" => "1.20"},{"date" => "2019-02-27T11:08:40","version" => "1.21"},{"date" => "2022-04-21T23:16:33","version" => "1.22"},{"date" => "2023-09-10T14:27:09","version" => "1.23"},{"date" => "2025-05-04T19:33:22","version" => "1.24"}]},"DBD-Pg" => {"advisories" => [{"affected_versions" => ["<2.19.0"],"cves" => ["CVE-2012-1151"],"description" => "Multiple format string vulnerabilities in dbdimp.c in DBD::Pg (aka DBD-Pg or libdbd-pg-perl) module before 2.19.0 for Perl allow remote PostgreSQL database servers to cause a denial of service (process crash) via format string specifiers in (1) a crafted database warning to the pg_warn function or (2) a crafted DBD statement to the dbd_st_prepare function.\n","distribution" => "DBD-Pg","fixed_versions" => [],"id" => "CPANSA-DBD-Pg-2012-1151","references" => ["http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=661536","https://rt.cpan.org/Public/Bug/Display.html?id=75642","http://secunia.com/advisories/48319","https://bugzilla.redhat.com/show_bug.cgi?id=801733","http://www.openwall.com/lists/oss-security/2012/03/10/4","http://secunia.com/advisories/48307","http://www.debian.org/security/2012/dsa-2431","http://cpansearch.perl.org/src/TURNSTEP/DBD-Pg-2.19.1/Changes","http://www.openwall.com/lists/oss-security/2012/03/09/6","http://rhn.redhat.com/errata/RHSA-2012-1116.html","http://secunia.com/advisories/48824","http://security.gentoo.org/glsa/glsa-201204-08.xml","http://www.mandriva.com/security/advisories?name=MDVSA-2012:112","https://exchange.xforce.ibmcloud.com/vulnerabilities/73855","https://exchange.xforce.ibmcloud.com/vulnerabilities/73854"],"reported" => "2012-09-09","severity" => undef},{"affected_versions" => ["==1.49"],"cves" => ["CVE-2009-0663"],"description" => "Heap-based buffer overflow in the DBD::Pg (aka DBD-Pg or libdbd-pg-perl) module 1.49 for Perl might allow context-dependent attackers to execute arbitrary code via unspecified input to an application that uses the getline and pg_getline functions to read database rows.\n","distribution" => "DBD-Pg","fixed_versions" => [],"id" => "CPANSA-DBD-Pg-2009-0663","references" => ["http://security.debian.org/pool/updates/main/libd/libdbd-pg-perl/libdbd-pg-perl_1.49-2+etch1.diff.gz","https://launchpad.net/bugs/cve/2009-0663","http://www.debian.org/security/2009/dsa-1780","http://secunia.com/advisories/34909","http://www.securityfocus.com/bid/34755","http://www.redhat.com/support/errata/RHSA-2009-0479.html","http://secunia.com/advisories/35058","http://www.redhat.com/support/errata/RHSA-2009-1067.html","http://lists.opensuse.org/opensuse-security-announce/2009-07/msg00002.html","http://secunia.com/advisories/35685","https://exchange.xforce.ibmcloud.com/vulnerabilities/50467","https://oval.cisecurity.org/repository/search/definition/oval%3Aorg.mitre.oval%3Adef%3A9499"],"reported" => "2009-04-30","severity" => undef},{"affected_versions" => ["<2.0.0"],"cves" => ["CVE-2009-1341"],"description" => "Memory leak in the dequote_bytea function in quote.c in the DBD::Pg (aka DBD-Pg or libdbd-pg-perl) module before 2.0.0 for Perl allows context-dependent attackers to cause a denial of service (memory consumption) by fetching data with BYTEA columns.\n","distribution" => "DBD-Pg","fixed_versions" => [">=2.0.0"],"id" => "CPANSA-DBD-Pg-2009-1341","references" => ["http://rt.cpan.org/Public/Bug/Display.html?id=21392","https://launchpad.net/bugs/cve/2009-1341","http://security.debian.org/pool/updates/main/libd/libdbd-pg-perl/libdbd-pg-perl_1.49-2+etch1.diff.gz","http://www.debian.org/security/2009/dsa-1780","http://cpansearch.perl.org/src/TURNSTEP/DBD-Pg-2.13.1/Changes","http://secunia.com/advisories/34909","http://www.securityfocus.com/bid/34757","http://www.redhat.com/support/errata/RHSA-2009-0479.html","http://secunia.com/advisories/35058","http://www.redhat.com/support/errata/RHSA-2009-1067.html","http://lists.opensuse.org/opensuse-security-announce/2009-07/msg00002.html","http://secunia.com/advisories/35685","https://exchange.xforce.ibmcloud.com/vulnerabilities/50387","https://oval.cisecurity.org/repository/search/definition/oval%3Aorg.mitre.oval%3Adef%3A9680"],"reported" => "2009-04-30","severity" => undef}],"main_module" => "DBD::Pg","versions" => [{"date" => "1997-03-13T21:11:59","version" => "0.1"},{"date" => "1997-04-28T20:48:18","version" => "0.3"},{"date" => "1997-08-10T20:45:01","version" => "0.5"},{"date" => "1997-08-16T06:48:54","version" => "0.51"},{"date" => "1997-08-17T09:14:26","version" => "0.52"},{"date" => "1997-08-23T20:41:48","version" => "0.61"},{"date" => "1997-08-26T21:39:36","version" => "0.62"},{"date" => "1997-10-05T18:41:32","version" => "0.63"},{"date" => "1998-02-02T21:05:57","version" => "0.64"},{"date" => "1998-02-15T09:59:55","version" => "0.65"},{"date" => "1998-02-19T21:13:44","version" => "0.66"},{"date" => "1998-02-21T16:49:34","version" => "0.67"},{"date" => "1998-03-03T21:22:18","version" => "0.68"},{"date" => "1998-03-06T22:15:44","version" => "0.69"},{"date" => "1998-04-20T20:14:28","version" => "0.72"},{"date" => "1998-06-03T15:15:54","version" => "0.73"},{"date" => "1998-11-05T21:41:28","version" => "0.89"},{"date" => "1999-01-16T06:10:32","version" => "0.90"},{"date" => "1999-02-14T20:41:05","version" => "0.91"},{"date" => "1999-06-16T19:06:21","version" => "0.92"},{"date" => "1999-09-29T21:25:02","version" => "0.93"},{"date" => "2000-07-07T10:45:34","version" => "0.94"},{"date" => "2000-07-10T18:07:30","version" => "0.95"},{"date" => "2001-04-09T17:58:20","version" => "0.96"},{"date" => "2001-04-20T21:11:46","version" => "0.97"},{"date" => "2001-04-25T14:13:22","version" => "0.98"},{"date" => "2001-05-24T17:36:05","version" => "0.99"},{"date" => "2001-05-27T14:14:13","version" => "1.00"},{"date" => "2001-06-27T18:03:08","version" => "1.01"},{"date" => "2002-03-06T23:05:52","version" => "1.10"},{"date" => "2002-03-07T01:34:18","version" => "1.11"},{"date" => "2002-04-10T02:03:57","version" => "1.12"},{"date" => "2002-04-27T20:50:11","version" => "1.13"},{"date" => "2002-11-27T17:57:11","version" => "1.20"},{"date" => "2003-01-13T06:46:43","version" => "1.21"},{"date" => "2003-03-27T04:46:08","version" => "1.22"},{"date" => "2003-09-10T02:12:07","version" => "1.31_5"},{"date" => "2003-10-29T21:33:29","version" => "1.31_7"},{"date" => "2003-11-10T03:52:37","version" => "1.31_8"},{"date" => "2003-11-14T22:17:30","version" => "1.31_9"},{"date" => "2003-11-18T18:34:28","version" => "1.31"},{"date" => "2004-02-13T18:57:25","version" => "1.32_1"},{"date" => "2004-02-19T02:40:51","version" => "1.32_2"},{"date" => "2004-02-25T19:23:08","version" => "1.32"},{"date" => "2005-02-06T21:18:12","version" => "1.39_02"},{"date" => "2005-02-22T06:07:17","version" => "1.40"},{"date" => "2005-03-31T12:35:04","version" => "1.40_1"},{"date" => "2005-03-31T23:34:15","version" => "1.40_2"},{"date" => "2005-04-01T23:56:01","version" => "1.40_03"},{"date" => "2005-04-05T02:47:47","version" => "1.40_04"},{"date" => "2005-04-06T22:53:50","version" => "1.41"},{"date" => "2005-05-07T18:48:36","version" => "1.41_1"},{"date" => "2005-05-19T03:23:24","version" => "1.41_2"},{"date" => "2005-05-21T14:56:23","version" => "1.42"},{"date" => "2005-06-22T00:42:23","version" => "1.42_1"},{"date" => "2005-06-23T12:09:13","version" => "1.43"},{"date" => "2005-09-13T01:39:06","version" => "1.43_1"},{"date" => "2006-02-13T03:50:51","version" => "1.43_2"},{"date" => "2006-02-22T03:00:40","version" => "1.44"},{"date" => "2006-02-26T19:15:10","version" => "1.45"},{"date" => "2006-03-17T17:17:03","version" => "1.46"},{"date" => "2006-03-19T21:19:12","version" => "1.47"},{"date" => "2006-04-05T15:39:30","version" => "1.48"},{"date" => "2006-05-05T16:40:59","version" => "1.49"},{"date" => "2008-01-17T14:34:38","version" => "2.0.0"},{"date" => "2008-01-17T22:47:38","version" => "2.0.0_2"},{"date" => "2008-01-18T04:21:25","version" => "2.0.0_3"},{"date" => "2008-01-18T16:25:19","version" => "2.0.0_4"},{"date" => "2008-01-19T19:05:27","version" => "2.0.0_5"},{"date" => "2008-01-23T19:19:45","version" => "2.0.0_6"},{"date" => "2008-01-26T17:48:03","version" => "2.0.0_7"},{"date" => "2008-01-28T17:08:31","version" => "2.0.0_8"},{"date" => "2008-01-28T21:21:19","version" => "2.0.0_9"},{"date" => "2008-02-10T19:34:31","version" => "2.0.0"},{"date" => "2008-02-17T04:58:29","version" => "2.0.0_1"},{"date" => "2008-02-17T13:10:04","version" => "2.0.0_2"},{"date" => "2008-02-17T18:17:25","version" => "2.0.1_1"},{"date" => "2008-02-18T02:28:30","version" => "2.0.1_2"},{"date" => "2008-02-18T23:52:09","version" => "2.0.1_3"},{"date" => "2008-02-19T02:09:09","version" => "2.1.0"},{"date" => "2008-02-19T04:45:29","version" => "2.1.1"},{"date" => "2008-02-19T15:36:06","version" => "2.1.1_1"},{"date" => "2008-02-20T02:56:37","version" => "2.1.2"},{"date" => "2008-02-21T00:31:43","version" => "2.1.3"},{"date" => "2008-02-26T01:59:20","version" => "2.1.3_1"},{"date" => "2008-02-26T13:50:47","version" => "2.1.3_2"},{"date" => "2008-02-28T04:08:34","version" => "2.2.0"},{"date" => "2008-03-02T03:01:12","version" => "2.2.1"},{"date" => "2008-03-03T17:14:01","version" => "2.2.2"},{"date" => "2008-03-19T14:50:06","version" => "2.3.0"},{"date" => "2008-03-21T16:41:42","version" => "2.4.0"},{"date" => "2008-03-23T16:55:36","version" => "2.5.0"},{"date" => "2008-04-07T19:16:57","version" => "2.5.1"},{"date" => "2008-04-14T15:57:23","version" => "2.5.2_1"},{"date" => "2008-04-16T18:16:11","version" => "2.6.0"},{"date" => "2008-04-22T18:06:55","version" => "2.6.1"},{"date" => "2008-04-28T21:15:56","version" => "2.6.1_1"},{"date" => "2008-04-30T23:18:14","version" => "2.6.2"},{"date" => "2008-05-01T16:03:12","version" => "2.6.3"},{"date" => "2008-05-02T17:09:20","version" => "2.6.4"},{"date" => "2008-05-07T14:10:57","version" => "2.6.5"},{"date" => "2008-05-07T20:41:03","version" => "2.6.6"},{"date" => "2008-05-10T22:37:02","version" => "2.7.0"},{"date" => "2008-05-11T20:58:52","version" => "2.7.1"},{"date" => "2008-05-14T13:19:24","version" => "2.7.2"},{"date" => "2008-06-02T01:33:26","version" => "2.8.0"},{"date" => "2008-06-12T04:06:28","version" => "2.8.1"},{"date" => "2008-06-30T02:21:03","version" => "2.8.2"},{"date" => "2008-07-07T02:09:01","version" => "2.8.3"},{"date" => "2008-07-10T18:18:54","version" => "2.8.4"},{"date" => "2008-07-13T14:41:49","version" => "2.8.5"},{"date" => "2008-07-21T15:23:39","version" => "2.8.6"},{"date" => "2008-07-24T05:27:41","version" => "2.8.7"},{"date" => "2008-08-03T19:48:22","version" => "2.9.0"},{"date" => "2008-08-18T03:49:13","version" => "2.9.1"},{"date" => "2008-08-18T14:00:03","version" => "2.9.2"},{"date" => "2008-08-27T02:46:34","version" => "2.10.0"},{"date" => "2008-08-31T16:29:33","version" => "2.10.1"},{"date" => "2008-09-01T01:40:52","version" => "2.10.2"},{"date" => "2008-09-01T01:48:43","version" => "2.10.3"},{"date" => "2008-09-16T15:35:48","version" => "2.10.4"},{"date" => "2008-09-16T15:43:37","version" => "2.10.5"},{"date" => "2008-09-19T13:38:32","version" => "2.10.6"},{"date" => "2008-09-23T03:13:47","version" => "2.10.7"},{"date" => "2008-10-13T11:50:18","version" => "2.11.0"},{"date" => "2008-10-14T04:21:14","version" => "2.11.1"},{"date" => "2008-10-16T00:55:38","version" => "2.11.2"},{"date" => "2008-11-03T13:51:55","version" => "2.11.3"},{"date" => "2008-11-12T22:11:37","version" => "2.11.4"},{"date" => "2008-11-24T23:56:25","version" => "2.11.5"},{"date" => "2008-11-30T23:05:28","version" => "2.11.6"},{"date" => "2008-12-13T17:02:24","version" => "2.11.7"},{"date" => "2008-12-28T19:24:29","version" => "2.11.8"},{"date" => "2009-03-24T02:23:04","version" => "2.11.8_1"},{"date" => "2009-03-26T18:53:00","version" => "2.11.8_2"},{"date" => "2009-03-28T14:56:16","version" => "2.12.0"},{"date" => "2009-04-14T02:14:44","version" => "2.13.0"},{"date" => "2009-04-23T16:30:10","version" => "2.13.1"},{"date" => "2009-07-13T19:43:16","version" => "2.13.1_1"},{"date" => "2009-07-14T14:34:31","version" => "2.13.1_2"},{"date" => "2009-07-15T21:12:47","version" => "2.13.1_3"},{"date" => "2009-07-17T01:15:13","version" => "2.13.1_4"},{"date" => "2009-07-20T23:54:06","version" => "2.13.1_5"},{"date" => "2009-07-21T16:03:25","version" => "2.13.1_6"},{"date" => "2009-07-21T21:43:56","version" => "2.13.1_7"},{"date" => "2009-07-27T22:45:52","version" => "2.14.0"},{"date" => "2009-07-28T17:05:35","version" => "2.14.1"},{"date" => "2009-08-04T04:08:56","version" => "2.14.1_1"},{"date" => "2009-08-04T18:18:51","version" => "2.15.0"},{"date" => "2009-08-07T15:05:27","version" => "2.15.1"},{"date" => "2009-12-17T15:41:55","version" => "2.8.8"},{"date" => "2009-12-17T17:14:41","version" => "2.16.0"},{"date" => "2010-01-20T21:13:23","version" => "2.16.1"},{"date" => "2010-04-06T18:56:34","version" => "2.17.0"},{"date" => "2010-04-08T15:32:24","version" => "2.17.1"},{"date" => "2010-11-21T05:14:52","version" => "2.17.2"},{"date" => "2011-03-27T03:53:00","version" => "2.17.2_1"},{"date" => "2011-03-29T00:36:37","version" => "2.18.0"},{"date" => "2011-05-09T16:40:13","version" => "2.18.1"},{"date" => "2011-06-19T18:46:40","version" => "2.99.9_1"},{"date" => "2011-06-20T20:47:06","version" => "2.99.9_2"},{"date" => "2012-03-09T22:51:54","version" => "2.19.0"},{"date" => "2012-03-11T03:28:47","version" => "2.19.1"},{"date" => "2012-03-12T20:58:56","version" => "2.19.2"},{"date" => "2012-08-21T17:18:39","version" => "2.19.3"},{"date" => "2013-11-16T03:47:03","version" => "2.20.1_1"},{"date" => "2013-11-21T03:22:26","version" => "2.20.1_2"},{"date" => "2013-11-26T19:03:57","version" => "2.20.1_3"},{"date" => "2013-11-27T19:35:07","version" => "2.20.1_4"},{"date" => "2014-01-11T20:31:09","version" => "2.20.1_6"},{"date" => "2014-02-04T01:38:37","version" => "3.0.0"},{"date" => "2014-04-05T11:08:15","version" => "3.1.0"},{"date" => "2014-04-06T13:17:49","version" => "3.1.1"},{"date" => "2014-05-15T17:20:49","version" => "3.2.0"},{"date" => "2014-05-20T16:38:44","version" => "3.2.1"},{"date" => "2014-05-31T18:50:07","version" => "3.3.0"},{"date" => "2014-08-16T19:09:15","version" => "3.4.0"},{"date" => "2014-08-20T20:38:19","version" => "3.4.1"},{"date" => "2014-09-25T21:16:23","version" => "3.4.2"},{"date" => "2015-01-06T20:41:04","version" => "3.5.0"},{"date" => "2015-02-07T13:09:54","version" => "3.5.0_1"},{"date" => "2015-02-16T19:17:14","version" => "3.5.0_2"},{"date" => "2015-02-17T21:20:22","version" => "3.5.1"},{"date" => "2015-09-29T15:46:33","version" => "3.5.2"},{"date" => "2015-10-01T14:06:04","version" => "3.5.3"},{"date" => "2017-04-05T10:23:22","version" => "3.5.9_1"},{"date" => "2017-04-17T13:34:12","version" => "3.6.0"},{"date" => "2017-05-22T16:49:32","version" => "3.6.1"},{"date" => "2017-05-23T14:25:49","version" => "3.6.2"},{"date" => "2017-09-22T16:30:49","version" => "3.6.9_1"},{"date" => "2017-09-23T02:10:34","version" => "3.6.9_2"},{"date" => "2017-09-24T19:30:09","version" => "3.7.0"},{"date" => "2018-02-11T19:23:39","version" => "3.7.1"},{"date" => "2018-02-12T13:39:58","version" => "v3.7.3"},{"date" => "2018-02-13T04:10:10","version" => "3.7.4"},{"date" => "2019-04-26T02:20:41","version" => "3.8.0"},{"date" => "2019-07-06T19:44:25","version" => "3.8.1"},{"date" => "2019-07-25T15:48:44","version" => "3.8.9_1"},{"date" => "2019-08-13T21:10:51","version" => "3.9.0"},{"date" => "2019-08-15T19:46:43","version" => "3.9.1"},{"date" => "2019-09-03T15:18:09","version" => "3.10.0"},{"date" => "2020-01-14T03:27:38","version" => "3.10.1"},{"date" => "2020-01-17T22:34:46","version" => "3.10.2"},{"date" => "2020-01-20T21:01:45","version" => "3.10.3"},{"date" => "2020-02-03T17:19:38","version" => "3.10.4"},{"date" => "2020-03-23T17:47:23","version" => "3.10.5"},{"date" => "2020-04-23T16:46:52","version" => "3.11.0"},{"date" => "2020-04-28T15:12:38","version" => "3.11.1"},{"date" => "2020-05-07T18:35:28","version" => "3.12.0"},{"date" => "2020-06-03T13:39:22","version" => "3.12.1"},{"date" => "2020-06-04T15:30:54","version" => "3.12.2"},{"date" => "2020-06-05T17:59:13","version" => "3.12.3"},{"date" => "2020-06-08T20:38:00","version" => "3.12.3_1"},{"date" => "2020-06-15T21:25:55","version" => "3.12.3_2"},{"date" => "2020-06-17T15:53:25","version" => "3.13.0"},{"date" => "2020-07-20T00:24:23","version" => "3.14.0"},{"date" => "2020-08-12T16:17:33","version" => "3.14.1"},{"date" => "2020-08-13T13:36:09","version" => "3.14.2"},{"date" => "2021-05-21T21:20:28","version" => "3.15.0"},{"date" => "2022-02-14T15:39:15","version" => "3.15.1"},{"date" => "2022-08-08T18:03:02","version" => "3.16.0"},{"date" => "2023-03-06T00:06:35","version" => "3.16.1"},{"date" => "2023-04-04T19:49:11","version" => "3.16.2"},{"date" => "2023-04-04T20:43:26","version" => "3.16.3"},{"date" => "2023-08-24T00:42:24","version" => "3.17.0"},{"date" => "2023-12-06T23:47:13","version" => "3.18.0"}]},"DBD-SQLite" => {"advisories" => [{"affected_versions" => ["<1.65_03"],"cves" => ["CVE-2020-15358"],"description" => "In SQLite before 3.32.3, select.c mishandles query-flattener optimization, leading to a multiSelectOrderBy heap overflow because of misuse of transitive properties for constant propagation.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2020-15358","references" => ["https://www.sqlite.org/src/info/10fa79d00f8091e5","https://www.sqlite.org/src/timeline?p=version-3.32.3&bt=version-3.32.2","https://www.sqlite.org/src/tktview?name=8f157e8010","https://security.netapp.com/advisory/ntap-20200709-0001/","https://security.gentoo.org/glsa/202007-26","https://usn.ubuntu.com/4438-1/","https://www.oracle.com/security-alerts/cpuoct2020.html","https://support.apple.com/kb/HT211931","https://support.apple.com/kb/HT211844","https://support.apple.com/kb/HT211850","https://support.apple.com/kb/HT211843","https://support.apple.com/kb/HT211847","http://seclists.org/fulldisclosure/2020/Nov/19","http://seclists.org/fulldisclosure/2020/Nov/22","http://seclists.org/fulldisclosure/2020/Nov/20","http://seclists.org/fulldisclosure/2020/Dec/32","https://www.oracle.com/security-alerts/cpujan2021.html","https://support.apple.com/kb/HT212147","http://seclists.org/fulldisclosure/2021/Feb/14","https://www.oracle.com/security-alerts/cpuApr2021.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf","https://www.oracle.com/security-alerts/cpuapr2022.html"],"reported" => "2020-06-27","severity" => "medium"},{"affected_versions" => ["<1.65_03"],"cves" => ["CVE-2020-13632"],"description" => "ext/fts3/fts3_snippet.c in SQLite before 3.32.0 has a NULL pointer dereference via a crafted matchinfo() query.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2020-13632","references" => ["https://bugs.chromium.org/p/chromium/issues/detail?id=1080459","https://sqlite.org/src/info/a4dd148928ea65bd","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/L7KXQWHIY2MQP4LNM6ODWJENMXYYQYBN/","https://security.netapp.com/advisory/ntap-20200608-0002/","https://usn.ubuntu.com/4394-1/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://www.oracle.com/security-alerts/cpuoct2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-05-27","severity" => "medium"},{"affected_versions" => ["<1.65_03"],"cves" => ["CVE-2020-13631"],"description" => "SQLite before 3.32.0 allows a virtual table to be renamed to the name of one of its shadow tables, related to alter.c and build.c.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2020-13631","references" => ["https://bugs.chromium.org/p/chromium/issues/detail?id=1080459","https://sqlite.org/src/info/eca0ba2cf4c0fdf7","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/L7KXQWHIY2MQP4LNM6ODWJENMXYYQYBN/","https://security.netapp.com/advisory/ntap-20200608-0002/","https://usn.ubuntu.com/4394-1/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://www.oracle.com/security-alerts/cpuoct2020.html","https://support.apple.com/kb/HT211931","https://support.apple.com/kb/HT211844","https://support.apple.com/kb/HT211850","https://support.apple.com/kb/HT211843","https://support.apple.com/kb/HT211952","http://seclists.org/fulldisclosure/2020/Nov/19","http://seclists.org/fulldisclosure/2020/Nov/22","http://seclists.org/fulldisclosure/2020/Nov/20","https://support.apple.com/kb/HT211935","http://seclists.org/fulldisclosure/2020/Dec/32","https://lists.apache.org/thread.html/rc713534b10f9daeee2e0990239fa407e2118e4aa9e88a7041177497c\@%3Cissues.guacamole.apache.org%3E","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-05-27","severity" => "medium"},{"affected_versions" => ["<1.65_03"],"cves" => ["CVE-2020-13630"],"description" => "ext/fts3/fts3.c in SQLite before 3.32.0 has a use-after-free in fts3EvalNextRow, related to the snippet feature.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2020-13630","references" => ["https://bugs.chromium.org/p/chromium/issues/detail?id=1080459","https://sqlite.org/src/info/0d69f76f0865f962","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/L7KXQWHIY2MQP4LNM6ODWJENMXYYQYBN/","https://security.netapp.com/advisory/ntap-20200608-0002/","https://usn.ubuntu.com/4394-1/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://www.oracle.com/security-alerts/cpuoct2020.html","https://support.apple.com/kb/HT211931","https://support.apple.com/kb/HT211844","https://support.apple.com/kb/HT211850","https://support.apple.com/kb/HT211843","https://support.apple.com/kb/HT211952","http://seclists.org/fulldisclosure/2020/Nov/19","http://seclists.org/fulldisclosure/2020/Nov/22","http://seclists.org/fulldisclosure/2020/Nov/20","https://support.apple.com/kb/HT211935","http://seclists.org/fulldisclosure/2020/Dec/32","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-05-27","severity" => "high"},{"affected_versions" => ["<1.65_03"],"cves" => ["CVE-2020-13435"],"description" => "SQLite through 3.32.0 has a segmentation fault in sqlite3ExprCodeTarget in expr.c.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2020-13435","references" => ["https://www.sqlite.org/src/info/7a5279a25c57adf1","https://security.netapp.com/advisory/ntap-20200528-0004/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/L7KXQWHIY2MQP4LNM6ODWJENMXYYQYBN/","https://usn.ubuntu.com/4394-1/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://support.apple.com/kb/HT211931","https://support.apple.com/kb/HT211844","https://support.apple.com/kb/HT211850","https://support.apple.com/kb/HT211843","https://support.apple.com/kb/HT211952","http://seclists.org/fulldisclosure/2020/Nov/19","http://seclists.org/fulldisclosure/2020/Nov/22","http://seclists.org/fulldisclosure/2020/Nov/20","https://support.apple.com/kb/HT211935","http://seclists.org/fulldisclosure/2020/Dec/32","https://www.oracle.com/security-alerts/cpuApr2021.html"],"reported" => "2020-05-24","severity" => "medium"},{"affected_versions" => ["<1.65_03"],"cves" => ["CVE-2020-13434"],"description" => "SQLite through 3.32.0 has an integer overflow in sqlite3_str_vappendf in printf.c.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2020-13434","references" => ["https://www.sqlite.org/src/info/23439ea582241138","https://www.sqlite.org/src/info/d08d3405878d394e","https://lists.debian.org/debian-lts-announce/2020/05/msg00024.html","https://security.netapp.com/advisory/ntap-20200528-0004/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/L7KXQWHIY2MQP4LNM6ODWJENMXYYQYBN/","https://usn.ubuntu.com/4394-1/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://support.apple.com/kb/HT211931","https://support.apple.com/kb/HT211844","https://support.apple.com/kb/HT211850","https://support.apple.com/kb/HT211843","https://support.apple.com/kb/HT211952","http://seclists.org/fulldisclosure/2020/Nov/19","http://seclists.org/fulldisclosure/2020/Nov/22","http://seclists.org/fulldisclosure/2020/Nov/20","https://support.apple.com/kb/HT211935","http://seclists.org/fulldisclosure/2020/Dec/32","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com/security-alerts/cpuapr2022.html"],"reported" => "2020-05-24","severity" => "medium"},{"affected_versions" => ["<1.65_03"],"cves" => ["CVE-2020-11656"],"description" => "In SQLite through 3.31.1, the ALTER TABLE implementation has a use-after-free, as demonstrated by an ORDER BY clause that belongs to a compound SELECT statement.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2020-11656","references" => ["https://www3.sqlite.org/cgi/src/info/b64674919f673602","https://www.sqlite.org/src/info/d09f8c3621d5f7f8","https://security.netapp.com/advisory/ntap-20200416-0001/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-04-09","severity" => "critical"},{"affected_versions" => ["<1.65_03"],"cves" => ["CVE-2020-11655"],"description" => "SQLite through 3.31.1 allows attackers to cause a denial of service (segmentation fault) via a malformed window-function query because the AggInfo object's initialization is mishandled.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2020-11655","references" => ["https://www3.sqlite.org/cgi/src/info/4a302b42c7bf5e11","https://www3.sqlite.org/cgi/src/tktview?name=af4556bb5c","https://security.netapp.com/advisory/ntap-20200416-0001/","https://lists.debian.org/debian-lts-announce/2020/05/msg00006.html","https://usn.ubuntu.com/4394-1/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-04-09","severity" => "high"},{"affected_versions" => ["<1.65_03"],"cves" => ["CVE-2020-9327"],"description" => "In SQLite 3.31.1, isAuxiliaryVtabOperator allows attackers to trigger a NULL pointer dereference and segmentation fault because of generated column optimizations.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2020-9327","references" => ["https://www.sqlite.org/cgi/src/info/4374860b29383380","https://www.sqlite.org/cgi/src/info/abc473fb8fb99900","https://www.sqlite.org/cgi/src/info/9d0d4ab95dc0c56e","https://security.netapp.com/advisory/ntap-20200313-0002/","https://security.gentoo.org/glsa/202003-16","https://usn.ubuntu.com/4298-1/","https://www.oracle.com/security-alerts/cpujul2020.html","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-02-21","severity" => "high"},{"affected_versions" => ["==1.65_02"],"cves" => ["CVE-2019-20218"],"description" => "selectExpander in select.c in SQLite 3.30.1 proceeds with WITH stack unwinding even after a parsing error.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2019-20218","references" => ["https://github.com/sqlite/sqlite/commit/a6c1a71cde082e09750465d5675699062922e387","https://usn.ubuntu.com/4298-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://security.gentoo.org/glsa/202007-26","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://lists.debian.org/debian-lts-announce/2020/12/msg00016.html"],"reported" => "2020-01-02","severity" => "high"},{"affected_versions" => ["==1.65_02"],"cves" => ["CVE-2019-19959"],"description" => "ext/misc/zipfile.c in SQLite 3.30.1 mishandles certain uses of INSERT INTO in situations involving embedded '\\\\0' characters in filenames, leading to a memory-management error that can be detected by (for example) valgrind.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2019-19959","references" => ["https://github.com/sqlite/sqlite/commit/1e490c4ca6b43a9cf8637d695907888349f69bec","https://github.com/sqlite/sqlite/commit/d8f2d46cbc9925e034a68aaaf60aad788d9373c1","https://security.netapp.com/advisory/ntap-20200204-0001/","https://usn.ubuntu.com/4298-1/","https://www.oracle.com/security-alerts/cpuapr2020.html"],"reported" => "2020-01-03","severity" => "high"},{"affected_versions" => ["==1.65_02"],"cves" => ["CVE-2019-19926","CVE-2019-19880"],"description" => "multiSelect in select.c in SQLite 3.30.1 mishandles certain errors during parsing, as demonstrated by errors from sqlite3WindowRewrite() calls. NOTE: this vulnerability exists because of an incomplete fix for CVE-2019-19880.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2019-19926","references" => ["https://github.com/sqlite/sqlite/commit/8428b3b437569338a9d1e10c4cd8154acbe33089","https://security.netapp.com/advisory/ntap-20200114-0003/","http://lists.opensuse.org/opensuse-security-announce/2020-02/msg00010.html","http://lists.opensuse.org/opensuse-security-announce/2020-02/msg00015.html","https://access.redhat.com/errata/RHSA-2020:0514","http://lists.opensuse.org/opensuse-security-announce/2020-02/msg00025.html","https://www.debian.org/security/2020/dsa-4638","https://usn.ubuntu.com/4298-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://usn.ubuntu.com/4298-2/","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-12-23","severity" => "high"},{"affected_versions" => [">=1.19_01,<1.63_03"],"cves" => ["CVE-2019-8457"],"description" => "SQLite3 from 3.6.0 to and including 3.27.2 is vulnerable to heap out-of-bound read in the rtreenode() function when handling invalid rtree tables.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.63_04"],"id" => "CPANSA-DBD-SQLite-2019-8457","references" => ["https://www.sqlite.org/src/info/90acdbfce9c08858","https://www.sqlite.org/releaselog/3_28_0.html","https://usn.ubuntu.com/4004-1/","https://usn.ubuntu.com/4004-2/","https://security.netapp.com/advisory/ntap-20190606-0002/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","http://lists.opensuse.org/opensuse-security-announce/2019-06/msg00074.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/SJPFGA45DI4F5MCF2OAACGH3HQOF4G3M/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/OPKYSWCOM3CL66RI76TYVIG6TJ263RXH/","https://www.oracle.com/technetwork/security-advisory/cpuoct2019-5072832.html","https://www.oracle.com/security-alerts/cpujan2020.html","https://www.oracle.com/security-alerts/cpuapr2020.html","https://www.oracle.com/security-alerts/cpujul2020.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-05-30","severity" => "critical"},{"affected_versions" => [">=1.61_03,<1.63_03"],"cves" => ["CVE-2019-5018"],"description" => "An exploitable use after free vulnerability exists in the window function functionality of Sqlite3 3.26.0. A specially crafted SQL command can cause a use after free vulnerability, potentially resulting in remote code execution. An attacker can send a malicious SQL command to trigger this vulnerability.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.63_04"],"id" => "CPANSA-DBD-SQLite-2019-5018","references" => ["https://talosintelligence.com/vulnerability_reports/TALOS-2019-0777","http://www.securityfocus.com/bid/108294","http://packetstormsecurity.com/files/152809/Sqlite3-Window-Function-Remote-Code-Execution.html","https://security.netapp.com/advisory/ntap-20190521-0001/","https://security.gentoo.org/glsa/201908-09","https://usn.ubuntu.com/4205-1/"],"reported" => "2019-05-10","severity" => "high"},{"affected_versions" => ["==1.65_02"],"cves" => ["CVE-2019-19925"],"description" => "zipfileUpdate in ext/misc/zipfile.c in SQLite 3.30.1 mishandles a NULL pathname during an update of a ZIP archive.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2019-19925","references" => ["https://github.com/sqlite/sqlite/commit/54d501092d88c0cf89bec4279951f548fb0b8618","https://security.netapp.com/advisory/ntap-20200114-0003/","http://lists.opensuse.org/opensuse-security-announce/2020-02/msg00010.html","http://lists.opensuse.org/opensuse-security-announce/2020-02/msg00015.html","https://access.redhat.com/errata/RHSA-2020:0514","http://lists.opensuse.org/opensuse-security-announce/2020-02/msg00025.html","https://www.debian.org/security/2020/dsa-4638","https://usn.ubuntu.com/4298-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-12-24","severity" => "high"},{"affected_versions" => ["==1.65_02"],"cves" => ["CVE-2019-19924"],"description" => "SQLite 3.30.1 mishandles certain parser-tree rewriting, related to expr.c, vdbeaux.c, and window.c. This is caused by incorrect sqlite3WindowRewrite() error handling.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2019-19924","references" => ["https://github.com/sqlite/sqlite/commit/8654186b0236d556aa85528c2573ee0b6ab71be3","https://security.netapp.com/advisory/ntap-20200114-0003/","https://usn.ubuntu.com/4298-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.apache.org/thread.html/rf4c02775860db415b4955778a131c2795223f61cb8c6a450893651e4\@%3Cissues.bookkeeper.apache.org%3E","https://lists.apache.org/thread.html/r58af02e294bd07f487e2c64ffc0a29b837db5600e33b6e698b9d696b\@%3Cissues.bookkeeper.apache.org%3E","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-12-24","severity" => "medium"},{"affected_versions" => ["==1.65_02"],"cves" => ["CVE-2019-19923"],"description" => "flattenSubquery in select.c in SQLite 3.30.1 mishandles certain uses of SELECT DISTINCT involving a LEFT JOIN in which the right-hand side is a view. This can cause a NULL pointer dereference (or incorrect results).\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2019-19923","references" => ["https://github.com/sqlite/sqlite/commit/396afe6f6aa90a31303c183e11b2b2d4b7956b35","https://security.netapp.com/advisory/ntap-20200114-0003/","http://lists.opensuse.org/opensuse-security-announce/2020-02/msg00010.html","http://lists.opensuse.org/opensuse-security-announce/2020-02/msg00015.html","https://access.redhat.com/errata/RHSA-2020:0514","http://lists.opensuse.org/opensuse-security-announce/2020-02/msg00025.html","https://www.debian.org/security/2020/dsa-4638","https://usn.ubuntu.com/4298-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-12-24","severity" => "high"},{"affected_versions" => ["==1.65_02"],"cves" => ["CVE-2019-19880"],"description" => "exprListAppendList in window.c in SQLite 3.30.1 allows attackers to trigger an invalid pointer dereference because constant integer values in ORDER BY clauses of window definitions are mishandled.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2019-19880","references" => ["https://github.com/sqlite/sqlite/commit/75e95e1fcd52d3ec8282edb75ac8cd0814095d54","https://security.netapp.com/advisory/ntap-20200114-0001/","http://lists.opensuse.org/opensuse-security-announce/2020-02/msg00010.html","http://lists.opensuse.org/opensuse-security-announce/2020-02/msg00015.html","https://access.redhat.com/errata/RHSA-2020:0514","http://lists.opensuse.org/opensuse-security-announce/2020-02/msg00025.html","https://www.debian.org/security/2020/dsa-4638","https://usn.ubuntu.com/4298-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-12-18","severity" => "high"},{"affected_versions" => ["<=1.65_02"],"cves" => ["CVE-2019-19646"],"description" => "pragma.c in SQLite through 3.30.1 mishandles NOT NULL in an integrity_check PRAGMA command in certain cases of generated columns.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2019-19646","references" => ["https://github.com/sqlite/sqlite/commit/ebd70eedd5d6e6a890a670b5ee874a5eae86b4dd","https://github.com/sqlite/sqlite/commit/926f796e8feec15f3836aa0a060ed906f8ae04d3","https://www.sqlite.org/","https://security.netapp.com/advisory/ntap-20191223-0001/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-12-09","severity" => "critical"},{"affected_versions" => ["<=1.65_02"],"cves" => ["CVE-2019-19645"],"description" => "alter.c in SQLite through 3.30.1 allows attackers to trigger infinite recursion via certain types of self-referential views in conjunction with ALTER TABLE statements.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2019-19645","references" => ["https://github.com/sqlite/sqlite/commit/38096961c7cd109110ac21d3ed7dad7e0cb0ae06","https://security.netapp.com/advisory/ntap-20191223-0001/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://usn.ubuntu.com/4394-1/","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-12-09","severity" => "medium"},{"affected_versions" => ["==1.65_02"],"cves" => ["CVE-2019-19603"],"description" => "SQLite 3.30.1 mishandles certain SELECT statements with a nonexistent VIEW, leading to an application crash.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2019-19603","references" => ["https://github.com/sqlite/sqlite/commit/527cbd4a104cb93bf3994b3dd3619a6299a78b13","https://www.sqlite.org/","https://security.netapp.com/advisory/ntap-20191223-0001/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://usn.ubuntu.com/4394-1/","https://lists.apache.org/thread.html/rc713534b10f9daeee2e0990239fa407e2118e4aa9e88a7041177497c\@%3Cissues.guacamole.apache.org%3E","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-12-09","severity" => "high"},{"affected_versions" => ["==1.65_02"],"cves" => ["CVE-2019-19317"],"description" => "lookupName in resolve.c in SQLite 3.30.1 omits bits from the colUsed bitmask in the case of a generated column, which allows attackers to cause a denial of service or possibly have unspecified other impact.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2019-19317","references" => ["https://github.com/sqlite/sqlite/commit/522ebfa7cee96fb325a22ea3a2464a63485886a8","https://github.com/sqlite/sqlite/commit/73bacb7f93eab9f4bd5a65cbc4ae242acf63c9e3","https://security.netapp.com/advisory/ntap-20191223-0001/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-12-05","severity" => "critical"},{"affected_versions" => ["==1.65_02"],"cves" => ["CVE-2019-19244"],"description" => "sqlite3Select in select.c in SQLite 3.30.1 allows a crash if a sub-select uses both DISTINCT and window functions, and also has certain ORDER BY usage.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2019-19244","references" => ["https://github.com/sqlite/sqlite/commit/e59c562b3f6894f84c715772c4b116d7b5c01348","https://usn.ubuntu.com/4205-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-11-25","severity" => "high"},{"affected_versions" => ["==1.65_02"],"cves" => ["CVE-2019-19242"],"description" => "SQLite 3.30.1 mishandles pExpr->y.pTab, as demonstrated by the TK_COLUMN case in sqlite3ExprCodeTarget in expr.c.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.65_03"],"id" => "CPANSA-DBD-SQLite-2019-19242","references" => ["https://github.com/sqlite/sqlite/commit/57f7ece78410a8aae86aa4625fb7556897db384c","https://usn.ubuntu.com/4205-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-11-27","severity" => "medium"},{"affected_versions" => ["<1.61_01"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.61_01"],"id" => "CPANSA-DBD-SQLite-2018-20506","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["==1.59_02"],"cves" => ["CVE-2018-20505"],"description" => "SQLite 3.25.2, when queries are run on a table with a malformed PRIMARY KEY, allows remote attackers to cause a denial of service (application crash) by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases).\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.59_03"],"id" => "CPANSA-DBD-SQLite-2018-20505","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/1a84668dcfdebaf12415d","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["<1.61_01"],"cves" => ["CVE-2018-20346"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries that occur after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases), aka Magellan.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.61_01"],"id" => "CPANSA-DBD-SQLite-2018-20346","references" => ["https://www.sqlite.org/releaselog/3_25_3.html","https://www.mail-archive.com/sqlite-users\@mailinglists.sqlite.org/msg113218.html","https://crbug.com/900910","https://chromium.googlesource.com/chromium/src/+/c368e30ae55600a1c3c9cb1710a54f9c55de786e","https://chromereleases.googleblog.com/2018/12/stable-channel-update-for-desktop.html","https://bugzilla.redhat.com/show_bug.cgi?id=1659677","https://bugzilla.redhat.com/show_bug.cgi?id=1659379","https://blade.tencent.com/magellan/index_en.html","https://access.redhat.com/articles/3758321","https://worthdoingbadly.com/sqlitebug/","https://sqlite.org/src/info/d44318f59044162e","https://sqlite.org/src/info/940f2adc8541a838","https://news.ycombinator.com/item?id=18685296","https://github.com/zhuowei/worthdoingbadly.com/blob/master/_posts/2018-12-14-sqlitebug.html","https://lists.debian.org/debian-lts-announce/2018/12/msg00012.html","https://www.synology.com/security/advisory/Synology_SA_18_61","http://www.securityfocus.com/bid/106323","https://www.freebsd.org/security/advisories/FreeBSD-EN-19:03.sqlite.asc","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00040.html","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.gentoo.org/glsa/201904-21","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/PU4NZ6DDU4BEM3ACM3FM6GLEPX56ZQXK/","https://support.apple.com/HT209448","https://support.apple.com/HT209447","https://support.apple.com/HT209446","https://support.apple.com/HT209451","https://support.apple.com/HT209443","https://support.apple.com/HT209450","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2018-12-21","severity" => "high"},{"affected_versions" => ["<1.59_01"],"cves" => ["CVE-2018-8740"],"description" => "In SQLite through 3.22.0, databases whose schema is corrupted using a CREATE TABLE AS statement could cause a NULL pointer dereference, related to build.c and prepare.c.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.59_01"],"id" => "CPANSA-DBD-SQLite-2018-8740","references" => ["https://www.sqlite.org/cgi/src/timeline?r=corrupt-schema","https://bugs.launchpad.net/ubuntu/+source/sqlite3/+bug/1756349","https://bugs.chromium.org/p/oss-fuzz/issues/detail?id=6964","https://www.sqlite.org/cgi/src/vdiff?from=1774f1c3baf0bc3d&to=d75e67654aa9620b","http://www.securityfocus.com/bid/103466","https://lists.debian.org/debian-lts-announce/2019/01/msg00009.html","http://lists.opensuse.org/opensuse-security-announce/2019-05/msg00050.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/PU4NZ6DDU4BEM3ACM3FM6GLEPX56ZQXK/","https://usn.ubuntu.com/4205-1/","https://usn.ubuntu.com/4394-1/","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://lists.apache.org/thread.html/rf4c02775860db415b4955778a131c2795223f61cb8c6a450893651e4\@%3Cissues.bookkeeper.apache.org%3E","https://lists.apache.org/thread.html/r58af02e294bd07f487e2c64ffc0a29b837db5600e33b6e698b9d696b\@%3Cissues.bookkeeper.apache.org%3E"],"reported" => "2018-03-17","severity" => "high"},{"affected_versions" => ["==1.55_06","<=1.55_03"],"cves" => ["CVE-2017-10989"],"description" => "The getNodeSize function in ext/rtree/rtree.c in SQLite through 3.19.3, as used in GDAL and other products, mishandles undersized RTree blobs in a crafted database, leading to a heap-based buffer over-read or possibly unspecified other impact.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.55_07"],"id" => "CPANSA-DBD-SQLite-2017-10989","references" => ["https://sqlite.org/src/info/66de6f4a","https://bugs.launchpad.net/ubuntu/+source/sqlite3/+bug/1700937","https://bugs.chromium.org/p/oss-fuzz/issues/detail?id=2405","https://sqlite.org/src/vpatch?from=0db20efe201736b3&to=66de6f4a9504ec26","http://marc.info/?l=sqlite-users&m=149933696214713&w=2","http://www.securityfocus.com/bid/99502","http://www.securitytracker.com/id/1039427","https://support.apple.com/HT208144","https://support.apple.com/HT208115","https://support.apple.com/HT208113","https://support.apple.com/HT208112","http://www.oracle.com/technetwork/security-advisory/cpujul2018-4258247.html","https://lists.debian.org/debian-lts-announce/2019/01/msg00009.html","http://lists.opensuse.org/opensuse-security-announce/2019-05/msg00050.html","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/"],"reported" => "2017-07-07","severity" => "critical"},{"affected_versions" => ["<=1.55_01"],"cves" => ["CVE-2016-6153"],"description" => "os_unix.c in SQLite before 3.13.0 improperly implements the temporary directory search algorithm, which might allow local users to obtain sensitive information, cause a denial of service (application crash), or have unspecified other impact by leveraging use of the current working directory for temporary files.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.55_01"],"id" => "CPANSA-DBD-SQLite-2016-6153","references" => ["http://www.openwall.com/lists/oss-security/2016/07/01/1","http://www.securityfocus.com/bid/91546","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/IGQTH7V45QVHFDXJAEECHEO3HHD644WZ/","https://www.korelogic.com/Resources/Advisories/KL-001-2016-003.txt","https://www.sqlite.org/releaselog/3_13_0.html","http://www.sqlite.org/cgi/src/info/67985761aa93fb61","http://www.openwall.com/lists/oss-security/2016/07/01/2","http://lists.opensuse.org/opensuse-updates/2016-08/msg00053.html","https://www.tenable.com/security/tns-2016-20","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/PU4NZ6DDU4BEM3ACM3FM6GLEPX56ZQXK/","https://github.com/briandfoy/cpan-security-advisory/issues/187","https://rt.cpan.org/Public/Bug/Display.html?id=118395"],"reported" => "2016-09-26","severity" => "medium"},{"affected_versions" => ["<=1.47_01"],"cves" => ["CVE-2015-3416"],"description" => "The sqlite3VXPrintf function in printf.c in SQLite before 3.8.9 does not properly handle precision and width values during floating-point conversions, which allows context-dependent attackers to cause a denial of service (integer overflow and stack-based buffer overflow) or possibly have unspecified other impact via large integers in a crafted printf function call in a SELECT statement.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.47_02"],"id" => "CPANSA-DBD-SQLite-2015-3416","references" => ["http://www.sqlite.org/src/info/c494171f77dc2e5e04cb6d865e688448f04e5920","http://seclists.org/fulldisclosure/2015/Apr/31","http://www.debian.org/security/2015/dsa-3252","http://www.mandriva.com/security/advisories?name=MDVSA-2015:217","http://www.ubuntu.com/usn/USN-2698-1","https://support.apple.com/HT205267","http://lists.apple.com/archives/security-announce/2015/Sep/msg00008.html","http://lists.apple.com/archives/security-announce/2015/Sep/msg00005.html","https://support.apple.com/HT205213","http://www.oracle.com/technetwork/topics/security/bulletinapr2016-2952098.html","http://www.securitytracker.com/id/1033703","https://security.gentoo.org/glsa/201507-05","http://rhn.redhat.com/errata/RHSA-2015-1635.html","http://rhn.redhat.com/errata/RHSA-2015-1634.html","http://www.securityfocus.com/bid/74228","http://www.oracle.com/technetwork/security-advisory/cpujul2018-4258247.html"],"reported" => "2015-04-24","severity" => undef},{"affected_versions" => ["<=1.47_01"],"cves" => ["CVE-2015-3415"],"description" => "The sqlite3VdbeExec function in vdbe.c in SQLite before 3.8.9 does not properly implement comparison operators, which allows context-dependent attackers to cause a denial of service (invalid free operation) or possibly have unspecified other impact via a crafted CHECK clause, as demonstrated by CHECK(0&O>O) in a CREATE TABLE statement.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.47_02"],"id" => "CPANSA-DBD-SQLite-2015-3415","references" => ["https://www.sqlite.org/src/info/02e3c88fbf6abdcf3975fb0fb71972b0ab30da30","http://seclists.org/fulldisclosure/2015/Apr/31","http://www.debian.org/security/2015/dsa-3252","http://www.mandriva.com/security/advisories?name=MDVSA-2015:217","http://www.ubuntu.com/usn/USN-2698-1","https://support.apple.com/HT205267","http://lists.apple.com/archives/security-announce/2015/Sep/msg00008.html","http://lists.apple.com/archives/security-announce/2015/Sep/msg00005.html","https://support.apple.com/HT205213","http://www.oracle.com/technetwork/topics/security/bulletinapr2016-2952098.html","http://www.securitytracker.com/id/1033703","https://security.gentoo.org/glsa/201507-05","http://rhn.redhat.com/errata/RHSA-2015-1635.html","http://www.securityfocus.com/bid/74228","http://www.oracle.com/technetwork/security-advisory/cpujul2018-4258247.html"],"reported" => "2015-04-24","severity" => undef},{"affected_versions" => ["<=1.47_01"],"cves" => ["CVE-2015-3414"],"description" => "SQLite before 3.8.9 does not properly implement the dequoting of collation-sequence names, which allows context-dependent attackers to cause a denial of service (uninitialized memory access and application crash) or possibly have unspecified other impact via a crafted COLLATE clause, as demonstrated by COLLATE\"\"\"\"\"\"\"\" at the end of a SELECT statement.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.47_02"],"id" => "CPANSA-DBD-SQLite-2015-3414","references" => ["https://www.sqlite.org/src/info/eddc05e7bb31fae74daa86e0504a3478b99fa0f2","http://seclists.org/fulldisclosure/2015/Apr/31","http://www.debian.org/security/2015/dsa-3252","http://www.mandriva.com/security/advisories?name=MDVSA-2015:217","http://www.ubuntu.com/usn/USN-2698-1","https://support.apple.com/HT205267","http://lists.apple.com/archives/security-announce/2015/Sep/msg00008.html","http://lists.apple.com/archives/security-announce/2015/Sep/msg00005.html","https://support.apple.com/HT205213","http://www.oracle.com/technetwork/topics/security/bulletinapr2016-2952098.html","http://www.securitytracker.com/id/1033703","https://security.gentoo.org/glsa/201507-05","http://rhn.redhat.com/errata/RHSA-2015-1635.html","http://www.securityfocus.com/bid/74228","http://www.oracle.com/technetwork/security-advisory/cpujul2018-4258247.html"],"reported" => "2015-04-24","severity" => undef},{"affected_versions" => ["==1.47_01"],"cves" => ["CVE-2013-7443"],"description" => "Buffer overflow in the skip-scan optimization in SQLite 3.8.2 allows remote attackers to cause a denial of service (crash) via crafted SQL statements.\n","distribution" => "DBD-SQLite","fixed_versions" => [">=1.47_02"],"id" => "CPANSA-DBD-SQLite-2013-7443","references" => ["https://www.sqlite.org/src/info/520070ec7fbaac73eda0e0123596b7bb3e9a6897","https://bugs.launchpad.net/ubuntu/+source/sqlite3/+bug/1448758","https://www.sqlite.org/src/info/ac5852d6403c9c9628ca0aa7be135c702f000698","http://ubuntu.com/usn/usn-2698-1","http://www.openwall.com/lists/oss-security/2015/07/14/5","http://www.openwall.com/lists/oss-security/2015/07/15/4","http://www.securityfocus.com/bid/76089"],"reported" => "2015-08-12","severity" => undef},{"affected_versions" => [">=1.00,<=1.02"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.03,<=1.04"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.05"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.06"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.07"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.08"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.09"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.10,<=1.11"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.12"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.13"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.14"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.19_01,<=1.22_04"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.22_05,<=1.26_01"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.26_02"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.26_03"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.26_04"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.26_05,<=1.26_06"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.26_07,<=1.27"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.28_01,<=1.28_02"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.29,<=1.30_02"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.30_03"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.30_04,<=1.31"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.32_01"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.32_02"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.32_03"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.32_04"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.33"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.34_01,<=1.34_03"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.36_01"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.36_02"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.36_03"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.36_04"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.37"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.38_01"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.38_02"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.38_03,<=1.40"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.41_01"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.41_02"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.41_03"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.41_04,<=1.41_05"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.41_06"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.41_07,<=1.43_02"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.43_03,<=1.43_07"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.43_08,<=1.44"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.45_01,<=1.45_05"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.45_06,<=1.46"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.47_01"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.47_02,<1.47_04"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.47_05,<=1.48"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.49_01,<=1.49_02"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.49_03,<=1.49_04"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.49_05"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.49_06"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.49_07,<=1.50"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.51_01,<=1.51_03"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.51_04"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.51_05"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.51_06"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.51_07,<=1.54"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.55_01"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.55_02"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.55_03"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.55_04,<=1.55_05"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.55_06"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.55_07,<=1.58"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => ["=1.59_01"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.59_02,<=1.60"],"cves" => ["CVE-2018-20506"],"description" => "SQLite before 3.25.3, when the FTS3 extension is enabled, encounters an integer overflow (and resultant buffer overflow) for FTS3 queries in a \"merge\" operation that occurs after crafted changes to FTS3 shadow tables, allowing remote attackers to execute arbitrary code by leveraging the ability to run arbitrary SQL statements (such as in certain WebSQL use cases). This is a different vulnerability than CVE-2018-20346.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-20506-sqlite","references" => ["https://support.apple.com/kb/HT209451","https://support.apple.com/kb/HT209450","https://support.apple.com/kb/HT209448","https://support.apple.com/kb/HT209447","https://support.apple.com/kb/HT209446","https://support.apple.com/kb/HT209443","https://sqlite.org/src/info/940f2adc8541a838","https://seclists.org/bugtraq/2019/Jan/39","https://seclists.org/bugtraq/2019/Jan/33","https://seclists.org/bugtraq/2019/Jan/32","https://seclists.org/bugtraq/2019/Jan/31","https://seclists.org/bugtraq/2019/Jan/29","https://seclists.org/bugtraq/2019/Jan/28","http://www.securityfocus.com/bid/106698","http://seclists.org/fulldisclosure/2019/Jan/69","http://seclists.org/fulldisclosure/2019/Jan/68","http://seclists.org/fulldisclosure/2019/Jan/67","http://seclists.org/fulldisclosure/2019/Jan/66","http://seclists.org/fulldisclosure/2019/Jan/64","http://seclists.org/fulldisclosure/2019/Jan/62","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00070.html","https://security.netapp.com/advisory/ntap-20190502-0004/","https://usn.ubuntu.com/4019-1/","https://usn.ubuntu.com/4019-2/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://kc.mcafee.com/corporate/index?page=content&id=SB10365"],"reported" => "2019-04-03","severity" => "high"},{"affected_versions" => [">=1.61_01,<=1.61_02"],"cves" => ["CVE-2020-11656"],"description" => "In SQLite through 3.31.1, the ALTER TABLE implementation has a use-after-free, as demonstrated by an ORDER BY clause that belongs to a compound SELECT statement.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2020-11656-sqlite","references" => ["https://www3.sqlite.org/cgi/src/info/b64674919f673602","https://www.sqlite.org/src/info/d09f8c3621d5f7f8","https://security.netapp.com/advisory/ntap-20200416-0001/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-04-09","severity" => "critical"},{"affected_versions" => [">=1.61_01,<=1.61_02"],"cves" => ["CVE-2020-11655"],"description" => "SQLite through 3.31.1 allows attackers to cause a denial of service (segmentation fault) via a malformed window-function query because the AggInfo object's initialization is mishandled.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2020-11655-sqlite","references" => ["https://www3.sqlite.org/cgi/src/info/4a302b42c7bf5e11","https://www3.sqlite.org/cgi/src/tktview?name=af4556bb5c","https://security.netapp.com/advisory/ntap-20200416-0001/","https://lists.debian.org/debian-lts-announce/2020/05/msg00006.html","https://usn.ubuntu.com/4394-1/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-04-09","severity" => "high"},{"affected_versions" => [">=1.61_03,<=1.63_01"],"cves" => ["CVE-2020-11656"],"description" => "In SQLite through 3.31.1, the ALTER TABLE implementation has a use-after-free, as demonstrated by an ORDER BY clause that belongs to a compound SELECT statement.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2020-11656-sqlite","references" => ["https://www3.sqlite.org/cgi/src/info/b64674919f673602","https://www.sqlite.org/src/info/d09f8c3621d5f7f8","https://security.netapp.com/advisory/ntap-20200416-0001/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-04-09","severity" => "critical"},{"affected_versions" => [">=1.61_03,<=1.63_01"],"cves" => ["CVE-2020-11655"],"description" => "SQLite through 3.31.1 allows attackers to cause a denial of service (segmentation fault) via a malformed window-function query because the AggInfo object's initialization is mishandled.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2020-11655-sqlite","references" => ["https://www3.sqlite.org/cgi/src/info/4a302b42c7bf5e11","https://www3.sqlite.org/cgi/src/tktview?name=af4556bb5c","https://security.netapp.com/advisory/ntap-20200416-0001/","https://lists.debian.org/debian-lts-announce/2020/05/msg00006.html","https://usn.ubuntu.com/4394-1/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-04-09","severity" => "high"},{"affected_versions" => [">=1.63_02,<=1.63_03"],"cves" => ["CVE-2020-11656"],"description" => "In SQLite through 3.31.1, the ALTER TABLE implementation has a use-after-free, as demonstrated by an ORDER BY clause that belongs to a compound SELECT statement.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2020-11656-sqlite","references" => ["https://www3.sqlite.org/cgi/src/info/b64674919f673602","https://www.sqlite.org/src/info/d09f8c3621d5f7f8","https://security.netapp.com/advisory/ntap-20200416-0001/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-04-09","severity" => "critical"},{"affected_versions" => [">=1.63_02,<=1.63_03"],"cves" => ["CVE-2020-11655"],"description" => "SQLite through 3.31.1 allows attackers to cause a denial of service (segmentation fault) via a malformed window-function query because the AggInfo object's initialization is mishandled.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2020-11655-sqlite","references" => ["https://www3.sqlite.org/cgi/src/info/4a302b42c7bf5e11","https://www3.sqlite.org/cgi/src/tktview?name=af4556bb5c","https://security.netapp.com/advisory/ntap-20200416-0001/","https://lists.debian.org/debian-lts-announce/2020/05/msg00006.html","https://usn.ubuntu.com/4394-1/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-04-09","severity" => "high"},{"affected_versions" => ["=1.63_04"],"cves" => ["CVE-2020-11656"],"description" => "In SQLite through 3.31.1, the ALTER TABLE implementation has a use-after-free, as demonstrated by an ORDER BY clause that belongs to a compound SELECT statement.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2020-11656-sqlite","references" => ["https://www3.sqlite.org/cgi/src/info/b64674919f673602","https://www.sqlite.org/src/info/d09f8c3621d5f7f8","https://security.netapp.com/advisory/ntap-20200416-0001/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-04-09","severity" => "critical"},{"affected_versions" => ["=1.63_04"],"cves" => ["CVE-2020-11655"],"description" => "SQLite through 3.31.1 allows attackers to cause a denial of service (segmentation fault) via a malformed window-function query because the AggInfo object's initialization is mishandled.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2020-11655-sqlite","references" => ["https://www3.sqlite.org/cgi/src/info/4a302b42c7bf5e11","https://www3.sqlite.org/cgi/src/tktview?name=af4556bb5c","https://security.netapp.com/advisory/ntap-20200416-0001/","https://lists.debian.org/debian-lts-announce/2020/05/msg00006.html","https://usn.ubuntu.com/4394-1/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-04-09","severity" => "high"},{"affected_versions" => [">=1.63_05,<=1.64"],"cves" => ["CVE-2020-11656"],"description" => "In SQLite through 3.31.1, the ALTER TABLE implementation has a use-after-free, as demonstrated by an ORDER BY clause that belongs to a compound SELECT statement.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2020-11656-sqlite","references" => ["https://www3.sqlite.org/cgi/src/info/b64674919f673602","https://www.sqlite.org/src/info/d09f8c3621d5f7f8","https://security.netapp.com/advisory/ntap-20200416-0001/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-04-09","severity" => "critical"},{"affected_versions" => [">=1.63_05,<=1.64"],"cves" => ["CVE-2020-11655"],"description" => "SQLite through 3.31.1 allows attackers to cause a denial of service (segmentation fault) via a malformed window-function query because the AggInfo object's initialization is mishandled.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2020-11655-sqlite","references" => ["https://www3.sqlite.org/cgi/src/info/4a302b42c7bf5e11","https://www3.sqlite.org/cgi/src/tktview?name=af4556bb5c","https://security.netapp.com/advisory/ntap-20200416-0001/","https://lists.debian.org/debian-lts-announce/2020/05/msg00006.html","https://usn.ubuntu.com/4394-1/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-04-09","severity" => "high"},{"affected_versions" => ["=1.65_01"],"cves" => ["CVE-2020-11656"],"description" => "In SQLite through 3.31.1, the ALTER TABLE implementation has a use-after-free, as demonstrated by an ORDER BY clause that belongs to a compound SELECT statement.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2020-11656-sqlite","references" => ["https://www3.sqlite.org/cgi/src/info/b64674919f673602","https://www.sqlite.org/src/info/d09f8c3621d5f7f8","https://security.netapp.com/advisory/ntap-20200416-0001/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-04-09","severity" => "critical"},{"affected_versions" => ["=1.65_01"],"cves" => ["CVE-2020-11655"],"description" => "SQLite through 3.31.1 allows attackers to cause a denial of service (segmentation fault) via a malformed window-function query because the AggInfo object's initialization is mishandled.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2020-11655-sqlite","references" => ["https://www3.sqlite.org/cgi/src/info/4a302b42c7bf5e11","https://www3.sqlite.org/cgi/src/tktview?name=af4556bb5c","https://security.netapp.com/advisory/ntap-20200416-0001/","https://lists.debian.org/debian-lts-announce/2020/05/msg00006.html","https://usn.ubuntu.com/4394-1/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-04-09","severity" => "high"},{"affected_versions" => ["=1.65_02"],"cves" => ["CVE-2020-11656"],"description" => "In SQLite through 3.31.1, the ALTER TABLE implementation has a use-after-free, as demonstrated by an ORDER BY clause that belongs to a compound SELECT statement.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2020-11656-sqlite","references" => ["https://www3.sqlite.org/cgi/src/info/b64674919f673602","https://www.sqlite.org/src/info/d09f8c3621d5f7f8","https://security.netapp.com/advisory/ntap-20200416-0001/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-04-09","severity" => "critical"},{"affected_versions" => ["=1.65_02"],"cves" => ["CVE-2020-11655"],"description" => "SQLite through 3.31.1 allows attackers to cause a denial of service (segmentation fault) via a malformed window-function query because the AggInfo object's initialization is mishandled.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2020-11655-sqlite","references" => ["https://www3.sqlite.org/cgi/src/info/4a302b42c7bf5e11","https://www3.sqlite.org/cgi/src/tktview?name=af4556bb5c","https://security.netapp.com/advisory/ntap-20200416-0001/","https://lists.debian.org/debian-lts-announce/2020/05/msg00006.html","https://usn.ubuntu.com/4394-1/","https://www.oracle.com/security-alerts/cpujul2020.html","https://security.gentoo.org/glsa/202007-26","https://security.FreeBSD.org/advisories/FreeBSD-SA-20:22.sqlite.asc","https://lists.debian.org/debian-lts-announce/2020/08/msg00037.html","https://www.oracle.com/security-alerts/cpuoct2020.html","https://www.oracle.com/security-alerts/cpujan2021.html","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.tenable.com/security/tns-2021-14","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2020-04-09","severity" => "high"},{"affected_versions" => [">=1.65_03,<=1.66"],"cves" => ["CVE-2019-19242"],"description" => "SQLite 3.30.1 mishandles pExpr->y.pTab, as demonstrated by the TK_COLUMN case in sqlite3ExprCodeTarget in expr.c.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2019-19242-sqlite","references" => ["https://github.com/sqlite/sqlite/commit/57f7ece78410a8aae86aa4625fb7556897db384c","https://usn.ubuntu.com/4205-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-11-27","severity" => "medium"},{"affected_versions" => [">=1.65_03,<=1.66"],"cves" => ["CVE-2018-3906"],"description" => "An exploitable stack-based buffer overflow vulnerability exists in the retrieval of a database field in video-core's HTTP server of Samsung SmartThings Hub. The video-core process insecurely extracts the shard.videoHostURL field from its SQLite database, leading to a buffer overflow on the stack. An attacker can send an HTTP request to trigger this vulnerability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-3906-sqlite","references" => ["https://talosintelligence.com/vulnerability_reports/TALOS-2018-0576"],"reported" => "2018-09-21","severity" => "high"},{"affected_versions" => [">=1.65_03,<=1.66"],"cves" => ["CVE-2021-20227"],"description" => "A flaw was found in SQLite's SELECT query functionality (src/select.c). This flaw allows an attacker who is capable of running SQL queries locally on the SQLite database to cause a denial of service or possible code execution by triggering a use-after-free. The highest threat from this vulnerability is to system availability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2021-20227-sqlite","references" => ["https://bugzilla.redhat.com/show_bug.cgi?id=1924886","https://www.sqlite.org/releaselog/3_34_1.html","https://security.gentoo.org/glsa/202103-04","https://security.netapp.com/advisory/ntap-20210423-0010/","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com//security-alerts/cpujul2021.html","https://www.oracle.com/security-alerts/cpuoct2021.html","https://security.gentoo.org/glsa/202210-40"],"reported" => "2021-03-23","severity" => "medium"},{"affected_versions" => ["=1.67_01"],"cves" => ["CVE-2019-19242"],"description" => "SQLite 3.30.1 mishandles pExpr->y.pTab, as demonstrated by the TK_COLUMN case in sqlite3ExprCodeTarget in expr.c.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2019-19242-sqlite","references" => ["https://github.com/sqlite/sqlite/commit/57f7ece78410a8aae86aa4625fb7556897db384c","https://usn.ubuntu.com/4205-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-11-27","severity" => "medium"},{"affected_versions" => ["=1.67_01"],"cves" => ["CVE-2018-3906"],"description" => "An exploitable stack-based buffer overflow vulnerability exists in the retrieval of a database field in video-core's HTTP server of Samsung SmartThings Hub. The video-core process insecurely extracts the shard.videoHostURL field from its SQLite database, leading to a buffer overflow on the stack. An attacker can send an HTTP request to trigger this vulnerability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-3906-sqlite","references" => ["https://talosintelligence.com/vulnerability_reports/TALOS-2018-0576"],"reported" => "2018-09-21","severity" => "high"},{"affected_versions" => ["=1.67_01"],"cves" => ["CVE-2021-20227"],"description" => "A flaw was found in SQLite's SELECT query functionality (src/select.c). This flaw allows an attacker who is capable of running SQL queries locally on the SQLite database to cause a denial of service or possible code execution by triggering a use-after-free. The highest threat from this vulnerability is to system availability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2021-20227-sqlite","references" => ["https://bugzilla.redhat.com/show_bug.cgi?id=1924886","https://www.sqlite.org/releaselog/3_34_1.html","https://security.gentoo.org/glsa/202103-04","https://security.netapp.com/advisory/ntap-20210423-0010/","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com//security-alerts/cpujul2021.html","https://www.oracle.com/security-alerts/cpuoct2021.html","https://security.gentoo.org/glsa/202210-40"],"reported" => "2021-03-23","severity" => "medium"},{"affected_versions" => ["=1.67_02"],"cves" => ["CVE-2019-19242"],"description" => "SQLite 3.30.1 mishandles pExpr->y.pTab, as demonstrated by the TK_COLUMN case in sqlite3ExprCodeTarget in expr.c.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2019-19242-sqlite","references" => ["https://github.com/sqlite/sqlite/commit/57f7ece78410a8aae86aa4625fb7556897db384c","https://usn.ubuntu.com/4205-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-11-27","severity" => "medium"},{"affected_versions" => ["=1.67_02"],"cves" => ["CVE-2018-3906"],"description" => "An exploitable stack-based buffer overflow vulnerability exists in the retrieval of a database field in video-core's HTTP server of Samsung SmartThings Hub. The video-core process insecurely extracts the shard.videoHostURL field from its SQLite database, leading to a buffer overflow on the stack. An attacker can send an HTTP request to trigger this vulnerability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-3906-sqlite","references" => ["https://talosintelligence.com/vulnerability_reports/TALOS-2018-0576"],"reported" => "2018-09-21","severity" => "high"},{"affected_versions" => ["=1.67_02"],"cves" => ["CVE-2021-20227"],"description" => "A flaw was found in SQLite's SELECT query functionality (src/select.c). This flaw allows an attacker who is capable of running SQL queries locally on the SQLite database to cause a denial of service or possible code execution by triggering a use-after-free. The highest threat from this vulnerability is to system availability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2021-20227-sqlite","references" => ["https://bugzilla.redhat.com/show_bug.cgi?id=1924886","https://www.sqlite.org/releaselog/3_34_1.html","https://security.gentoo.org/glsa/202103-04","https://security.netapp.com/advisory/ntap-20210423-0010/","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com//security-alerts/cpujul2021.html","https://www.oracle.com/security-alerts/cpuoct2021.html","https://security.gentoo.org/glsa/202210-40"],"reported" => "2021-03-23","severity" => "medium"},{"affected_versions" => ["=1.67_03"],"cves" => ["CVE-2019-19242"],"description" => "SQLite 3.30.1 mishandles pExpr->y.pTab, as demonstrated by the TK_COLUMN case in sqlite3ExprCodeTarget in expr.c.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2019-19242-sqlite","references" => ["https://github.com/sqlite/sqlite/commit/57f7ece78410a8aae86aa4625fb7556897db384c","https://usn.ubuntu.com/4205-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-11-27","severity" => "medium"},{"affected_versions" => ["=1.67_03"],"cves" => ["CVE-2018-3906"],"description" => "An exploitable stack-based buffer overflow vulnerability exists in the retrieval of a database field in video-core's HTTP server of Samsung SmartThings Hub. The video-core process insecurely extracts the shard.videoHostURL field from its SQLite database, leading to a buffer overflow on the stack. An attacker can send an HTTP request to trigger this vulnerability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-3906-sqlite","references" => ["https://talosintelligence.com/vulnerability_reports/TALOS-2018-0576"],"reported" => "2018-09-21","severity" => "high"},{"affected_versions" => ["=1.67_03"],"cves" => ["CVE-2021-20227"],"description" => "A flaw was found in SQLite's SELECT query functionality (src/select.c). This flaw allows an attacker who is capable of running SQL queries locally on the SQLite database to cause a denial of service or possible code execution by triggering a use-after-free. The highest threat from this vulnerability is to system availability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2021-20227-sqlite","references" => ["https://bugzilla.redhat.com/show_bug.cgi?id=1924886","https://www.sqlite.org/releaselog/3_34_1.html","https://security.gentoo.org/glsa/202103-04","https://security.netapp.com/advisory/ntap-20210423-0010/","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com//security-alerts/cpujul2021.html","https://www.oracle.com/security-alerts/cpuoct2021.html","https://security.gentoo.org/glsa/202210-40"],"reported" => "2021-03-23","severity" => "medium"},{"affected_versions" => [">=1.67_04,<=1.67_06"],"cves" => ["CVE-2019-19242"],"description" => "SQLite 3.30.1 mishandles pExpr->y.pTab, as demonstrated by the TK_COLUMN case in sqlite3ExprCodeTarget in expr.c.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2019-19242-sqlite","references" => ["https://github.com/sqlite/sqlite/commit/57f7ece78410a8aae86aa4625fb7556897db384c","https://usn.ubuntu.com/4205-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-11-27","severity" => "medium"},{"affected_versions" => [">=1.67_04,<=1.67_06"],"cves" => ["CVE-2018-3906"],"description" => "An exploitable stack-based buffer overflow vulnerability exists in the retrieval of a database field in video-core's HTTP server of Samsung SmartThings Hub. The video-core process insecurely extracts the shard.videoHostURL field from its SQLite database, leading to a buffer overflow on the stack. An attacker can send an HTTP request to trigger this vulnerability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-3906-sqlite","references" => ["https://talosintelligence.com/vulnerability_reports/TALOS-2018-0576"],"reported" => "2018-09-21","severity" => "high"},{"affected_versions" => [">=1.67_04,<=1.67_06"],"cves" => ["CVE-2021-20227"],"description" => "A flaw was found in SQLite's SELECT query functionality (src/select.c). This flaw allows an attacker who is capable of running SQL queries locally on the SQLite database to cause a denial of service or possible code execution by triggering a use-after-free. The highest threat from this vulnerability is to system availability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2021-20227-sqlite","references" => ["https://bugzilla.redhat.com/show_bug.cgi?id=1924886","https://www.sqlite.org/releaselog/3_34_1.html","https://security.gentoo.org/glsa/202103-04","https://security.netapp.com/advisory/ntap-20210423-0010/","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com//security-alerts/cpujul2021.html","https://www.oracle.com/security-alerts/cpuoct2021.html","https://security.gentoo.org/glsa/202210-40"],"reported" => "2021-03-23","severity" => "medium"},{"affected_versions" => [">=1.67_07,<=1.70"],"cves" => ["CVE-2019-19242"],"description" => "SQLite 3.30.1 mishandles pExpr->y.pTab, as demonstrated by the TK_COLUMN case in sqlite3ExprCodeTarget in expr.c.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2019-19242-sqlite","references" => ["https://github.com/sqlite/sqlite/commit/57f7ece78410a8aae86aa4625fb7556897db384c","https://usn.ubuntu.com/4205-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-11-27","severity" => "medium"},{"affected_versions" => [">=1.67_07,<=1.70"],"cves" => ["CVE-2018-3906"],"description" => "An exploitable stack-based buffer overflow vulnerability exists in the retrieval of a database field in video-core's HTTP server of Samsung SmartThings Hub. The video-core process insecurely extracts the shard.videoHostURL field from its SQLite database, leading to a buffer overflow on the stack. An attacker can send an HTTP request to trigger this vulnerability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-3906-sqlite","references" => ["https://talosintelligence.com/vulnerability_reports/TALOS-2018-0576"],"reported" => "2018-09-21","severity" => "high"},{"affected_versions" => [">=1.67_07,<=1.70"],"cves" => ["CVE-2021-20227"],"description" => "A flaw was found in SQLite's SELECT query functionality (src/select.c). This flaw allows an attacker who is capable of running SQL queries locally on the SQLite database to cause a denial of service or possible code execution by triggering a use-after-free. The highest threat from this vulnerability is to system availability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2021-20227-sqlite","references" => ["https://bugzilla.redhat.com/show_bug.cgi?id=1924886","https://www.sqlite.org/releaselog/3_34_1.html","https://security.gentoo.org/glsa/202103-04","https://security.netapp.com/advisory/ntap-20210423-0010/","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com//security-alerts/cpujul2021.html","https://www.oracle.com/security-alerts/cpuoct2021.html","https://security.gentoo.org/glsa/202210-40"],"reported" => "2021-03-23","severity" => "medium"},{"affected_versions" => ["=1.71_01"],"cves" => ["CVE-2019-19242"],"description" => "SQLite 3.30.1 mishandles pExpr->y.pTab, as demonstrated by the TK_COLUMN case in sqlite3ExprCodeTarget in expr.c.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2019-19242-sqlite","references" => ["https://github.com/sqlite/sqlite/commit/57f7ece78410a8aae86aa4625fb7556897db384c","https://usn.ubuntu.com/4205-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-11-27","severity" => "medium"},{"affected_versions" => ["=1.71_01"],"cves" => ["CVE-2018-3906"],"description" => "An exploitable stack-based buffer overflow vulnerability exists in the retrieval of a database field in video-core's HTTP server of Samsung SmartThings Hub. The video-core process insecurely extracts the shard.videoHostURL field from its SQLite database, leading to a buffer overflow on the stack. An attacker can send an HTTP request to trigger this vulnerability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-3906-sqlite","references" => ["https://talosintelligence.com/vulnerability_reports/TALOS-2018-0576"],"reported" => "2018-09-21","severity" => "high"},{"affected_versions" => ["=1.71_01"],"cves" => ["CVE-2021-20227"],"description" => "A flaw was found in SQLite's SELECT query functionality (src/select.c). This flaw allows an attacker who is capable of running SQL queries locally on the SQLite database to cause a denial of service or possible code execution by triggering a use-after-free. The highest threat from this vulnerability is to system availability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2021-20227-sqlite","references" => ["https://bugzilla.redhat.com/show_bug.cgi?id=1924886","https://www.sqlite.org/releaselog/3_34_1.html","https://security.gentoo.org/glsa/202103-04","https://security.netapp.com/advisory/ntap-20210423-0010/","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com//security-alerts/cpujul2021.html","https://www.oracle.com/security-alerts/cpuoct2021.html","https://security.gentoo.org/glsa/202210-40"],"reported" => "2021-03-23","severity" => "medium"},{"affected_versions" => ["=1.71_02"],"cves" => ["CVE-2019-19242"],"description" => "SQLite 3.30.1 mishandles pExpr->y.pTab, as demonstrated by the TK_COLUMN case in sqlite3ExprCodeTarget in expr.c.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2019-19242-sqlite","references" => ["https://github.com/sqlite/sqlite/commit/57f7ece78410a8aae86aa4625fb7556897db384c","https://usn.ubuntu.com/4205-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-11-27","severity" => "medium"},{"affected_versions" => ["=1.71_02"],"cves" => ["CVE-2018-3906"],"description" => "An exploitable stack-based buffer overflow vulnerability exists in the retrieval of a database field in video-core's HTTP server of Samsung SmartThings Hub. The video-core process insecurely extracts the shard.videoHostURL field from its SQLite database, leading to a buffer overflow on the stack. An attacker can send an HTTP request to trigger this vulnerability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-3906-sqlite","references" => ["https://talosintelligence.com/vulnerability_reports/TALOS-2018-0576"],"reported" => "2018-09-21","severity" => "high"},{"affected_versions" => ["=1.71_02"],"cves" => ["CVE-2021-20227"],"description" => "A flaw was found in SQLite's SELECT query functionality (src/select.c). This flaw allows an attacker who is capable of running SQL queries locally on the SQLite database to cause a denial of service or possible code execution by triggering a use-after-free. The highest threat from this vulnerability is to system availability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2021-20227-sqlite","references" => ["https://bugzilla.redhat.com/show_bug.cgi?id=1924886","https://www.sqlite.org/releaselog/3_34_1.html","https://security.gentoo.org/glsa/202103-04","https://security.netapp.com/advisory/ntap-20210423-0010/","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com//security-alerts/cpujul2021.html","https://www.oracle.com/security-alerts/cpuoct2021.html","https://security.gentoo.org/glsa/202210-40"],"reported" => "2021-03-23","severity" => "medium"},{"affected_versions" => [">=1.71_03,<=1.71_06"],"cves" => ["CVE-2019-19242"],"description" => "SQLite 3.30.1 mishandles pExpr->y.pTab, as demonstrated by the TK_COLUMN case in sqlite3ExprCodeTarget in expr.c.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2019-19242-sqlite","references" => ["https://github.com/sqlite/sqlite/commit/57f7ece78410a8aae86aa4625fb7556897db384c","https://usn.ubuntu.com/4205-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-11-27","severity" => "medium"},{"affected_versions" => [">=1.71_03,<=1.71_06"],"cves" => ["CVE-2018-3906"],"description" => "An exploitable stack-based buffer overflow vulnerability exists in the retrieval of a database field in video-core's HTTP server of Samsung SmartThings Hub. The video-core process insecurely extracts the shard.videoHostURL field from its SQLite database, leading to a buffer overflow on the stack. An attacker can send an HTTP request to trigger this vulnerability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-3906-sqlite","references" => ["https://talosintelligence.com/vulnerability_reports/TALOS-2018-0576"],"reported" => "2018-09-21","severity" => "high"},{"affected_versions" => [">=1.71_03,<=1.71_06"],"cves" => ["CVE-2021-20227"],"description" => "A flaw was found in SQLite's SELECT query functionality (src/select.c). This flaw allows an attacker who is capable of running SQL queries locally on the SQLite database to cause a denial of service or possible code execution by triggering a use-after-free. The highest threat from this vulnerability is to system availability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2021-20227-sqlite","references" => ["https://bugzilla.redhat.com/show_bug.cgi?id=1924886","https://www.sqlite.org/releaselog/3_34_1.html","https://security.gentoo.org/glsa/202103-04","https://security.netapp.com/advisory/ntap-20210423-0010/","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com//security-alerts/cpujul2021.html","https://www.oracle.com/security-alerts/cpuoct2021.html","https://security.gentoo.org/glsa/202210-40"],"reported" => "2021-03-23","severity" => "medium"},{"affected_versions" => [">=1.71_07,<=1.72"],"cves" => ["CVE-2019-19242"],"description" => "SQLite 3.30.1 mishandles pExpr->y.pTab, as demonstrated by the TK_COLUMN case in sqlite3ExprCodeTarget in expr.c.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2019-19242-sqlite","references" => ["https://github.com/sqlite/sqlite/commit/57f7ece78410a8aae86aa4625fb7556897db384c","https://usn.ubuntu.com/4205-1/","https://www.oracle.com/security-alerts/cpuapr2020.html","https://cert-portal.siemens.com/productcert/pdf/ssa-389290.pdf"],"reported" => "2019-11-27","severity" => "medium"},{"affected_versions" => [">=1.71_07,<=1.72"],"cves" => ["CVE-2018-3906"],"description" => "An exploitable stack-based buffer overflow vulnerability exists in the retrieval of a database field in video-core's HTTP server of Samsung SmartThings Hub. The video-core process insecurely extracts the shard.videoHostURL field from its SQLite database, leading to a buffer overflow on the stack. An attacker can send an HTTP request to trigger this vulnerability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2018-3906-sqlite","references" => ["https://talosintelligence.com/vulnerability_reports/TALOS-2018-0576"],"reported" => "2018-09-21","severity" => "high"},{"affected_versions" => [">=1.71_07,<=1.72"],"cves" => ["CVE-2021-20227"],"description" => "A flaw was found in SQLite's SELECT query functionality (src/select.c). This flaw allows an attacker who is capable of running SQL queries locally on the SQLite database to cause a denial of service or possible code execution by triggering a use-after-free. The highest threat from this vulnerability is to system availability.\n","distribution" => "DBD-SQLite","fixed_versions" => [],"id" => "CPANSA-DBD-SQLite-2021-20227-sqlite","references" => ["https://bugzilla.redhat.com/show_bug.cgi?id=1924886","https://www.sqlite.org/releaselog/3_34_1.html","https://security.gentoo.org/glsa/202103-04","https://security.netapp.com/advisory/ntap-20210423-0010/","https://www.oracle.com/security-alerts/cpuApr2021.html","https://www.oracle.com//security-alerts/cpujul2021.html","https://www.oracle.com/security-alerts/cpuoct2021.html","https://security.gentoo.org/glsa/202210-40"],"reported" => "2021-03-23","severity" => "medium"}],"main_module" => "DBD::SQLite","versions" => [{"date" => "2002-02-19T18:56:55","version" => "0.05"},{"date" => "2002-02-22T07:29:26","version" => "0.06"},{"date" => "2002-02-23T11:19:51","version" => "0.07"},{"date" => "2002-02-25T12:59:46","version" => "0.08"},{"date" => "2002-02-27T19:46:01","version" => "0.09"},{"date" => "2002-02-28T11:11:10","version" => "0.10"},{"date" => "2002-03-13T12:16:23","version" => "0.11"},{"date" => "2002-03-21T16:49:25","version" => "0.12"},{"date" => "2002-03-26T22:39:39","version" => "0.13"},{"date" => "2002-03-28T15:59:02","version" => "0.14"},{"date" => "2002-04-02T10:48:10","version" => "0.15"},{"date" => "2002-06-17T23:50:53","version" => "0.16"},{"date" => "2002-06-26T13:59:05","version" => "0.17"},{"date" => "2002-07-12T13:46:37","version" => "0.18"},{"date" => "2002-08-13T22:19:45","version" => "0.19"},{"date" => "2002-10-17T16:25:57","version" => "0.20"},{"date" => "2002-10-18T07:08:42","version" => "0.21"},{"date" => "2002-12-18T18:02:11","version" => "0.22"},{"date" => "2002-12-29T16:25:29","version" => "0.23"},{"date" => "2003-01-29T16:47:23","version" => "0.24"},{"date" => "2003-03-06T22:24:48","version" => "0.25"},{"date" => "2003-07-31T15:16:06","version" => "0.26"},{"date" => "2003-08-18T20:17:31","version" => "0.27"},{"date" => "2003-08-25T13:42:07","version" => "0.28"},{"date" => "2003-12-05T15:42:05","version" => "0.29"},{"date" => "2004-02-08T18:45:02","version" => "0.30"},{"date" => "2004-02-14T19:18:16","version" => "0.31"},{"date" => "2004-07-21T21:19:33","version" => "1.00"},{"date" => "2004-08-01T04:49:09","version" => "1.01"},{"date" => "2004-08-02T18:35:42","version" => "1.02"},{"date" => "2004-08-09T15:43:59","version" => "1.03"},{"date" => "2004-08-23T15:14:21","version" => "1.04"},{"date" => "2004-09-10T15:46:34","version" => "1.05"},{"date" => "2004-09-21T17:26:28","version" => "1.06"},{"date" => "2004-10-12T09:07:33","version" => "1.07"},{"date" => "2005-02-26T13:47:33","version" => "1.08"},{"date" => "2005-06-20T15:42:32","version" => "1.09"},{"date" => "2005-12-01T20:56:30","version" => "1.10"},{"date" => "2005-12-02T19:13:29","version" => "1.11"},{"date" => "2006-04-10T02:24:08","version" => "1.12"},{"date" => "2006-09-08T05:02:06","version" => "1.13"},{"date" => "2007-09-19T19:25:09","version" => "1.14"},{"date" => "2009-03-27T11:11:41","version" => "1.19_01"},{"date" => "2009-03-28T16:46:41","version" => "1.19_02"},{"date" => "2009-03-30T21:58:59","version" => "1.19_03"},{"date" => "2009-03-31T20:31:37","version" => "1.19_04"},{"date" => "2009-04-02T04:24:12","version" => "1.19_05"},{"date" => "2009-04-03T19:21:54","version" => "1.19_06"},{"date" => "2009-04-04T00:49:42","version" => "1.19_07"},{"date" => "2009-04-04T04:29:03","version" => "1.19_08"},{"date" => "2009-04-05T03:16:37","version" => "1.19_09"},{"date" => "2009-04-05T19:43:04","version" => "1.19_10"},{"date" => "2009-04-07T14:00:36","version" => "1.20"},{"date" => "2009-04-08T01:24:11","version" => "1.21"},{"date" => "2009-04-08T02:05:13","version" => "1.22_01"},{"date" => "2009-04-08T11:49:36","version" => "1.22_02"},{"date" => "2009-04-09T09:40:39","version" => "1.22_03"},{"date" => "2009-04-11T01:58:53","version" => "1.22_04"},{"date" => "2009-04-14T15:52:05","version" => "1.22_05"},{"date" => "2009-04-15T14:59:20","version" => "1.22_06"},{"date" => "2009-04-16T05:40:28","version" => "1.22_07"},{"date" => "2009-04-17T09:08:15","version" => "1.22_08"},{"date" => "2009-04-19T09:53:00","version" => "1.23"},{"date" => "2009-04-22T02:14:33","version" => "1.24_01"},{"date" => "2009-04-23T00:50:02","version" => "1.24_02"},{"date" => "2009-04-23T10:20:49","version" => "1.25"},{"date" => "2009-05-05T06:04:00","version" => "1.26_01"},{"date" => "2009-06-19T06:56:29","version" => "1.26_02"},{"date" => "2009-08-12T06:01:13","version" => "1.26_03"},{"date" => "2009-10-06T06:23:40","version" => "1.26_04"},{"date" => "2009-10-15T04:05:19","version" => "1.26_05"},{"date" => "2009-10-28T11:16:12","version" => "1.26_06"},{"date" => "2009-11-16T01:47:37","version" => "1.26_07"},{"date" => "2009-11-23T11:15:09","version" => "1.27"},{"date" => "2009-12-23T11:44:07","version" => "1.28_01"},{"date" => "2010-01-03T05:56:21","version" => "1.28_02"},{"date" => "2010-01-08T09:14:18","version" => "1.29"},{"date" => "2010-03-10T15:55:37","version" => "1.30_01"},{"date" => "2010-03-30T11:45:57","version" => "1.30_02"},{"date" => "2010-05-31T03:13:24","version" => "1.30_03"},{"date" => "2010-08-25T09:25:41","version" => "1.30_04"},{"date" => "2010-08-27T15:31:59","version" => "1.30_05"},{"date" => "2010-09-09T01:49:17","version" => "1.30_06"},{"date" => "2010-09-15T07:30:11","version" => "1.31"},{"date" => "2010-12-10T05:14:51","version" => "1.32_01"},{"date" => "2011-03-07T06:57:51","version" => "1.32_02"},{"date" => "2011-05-12T05:05:38","version" => "1.32_03"},{"date" => "2011-05-20T02:39:29","version" => "1.32_04"},{"date" => "2011-05-30T07:39:31","version" => "1.33"},{"date" => "2011-09-21T16:26:23","version" => "1.34_01"},{"date" => "2011-10-21T06:13:45","version" => "1.34_02"},{"date" => "2011-11-01T03:51:19","version" => "1.34_03"},{"date" => "2011-11-29T00:16:47","version" => "1.35"},{"date" => "2012-01-19T06:15:08","version" => "1.36_01"},{"date" => "2012-02-23T04:11:05","version" => "1.36_02"},{"date" => "2012-05-07T22:56:21","version" => "1.36_03"},{"date" => "2012-05-19T09:46:14","version" => "1.36_04"},{"date" => "2012-06-09T14:43:03","version" => "1.37"},{"date" => "2012-09-24T10:18:25","version" => "1.38_01"},{"date" => "2013-04-09T05:03:21","version" => "1.38_02"},{"date" => "2013-05-21T05:14:23","version" => "1.38_03"},{"date" => "2013-05-29T07:11:57","version" => "1.38_04"},{"date" => "2013-05-31T04:39:53","version" => "1.38_05"},{"date" => "2013-06-09T15:10:40","version" => "1.39"},{"date" => "2013-07-28T05:31:53","version" => "1.40"},{"date" => "2013-08-27T06:41:37","version" => "1.41_01"},{"date" => "2013-08-29T18:53:29","version" => "1.41_02"},{"date" => "2013-09-04T17:57:50","version" => "1.41_03"},{"date" => "2014-01-12T01:19:09","version" => "1.41_04"},{"date" => "2014-01-22T03:53:26","version" => "1.41_05"},{"date" => "2014-02-12T02:53:38","version" => "1.41_06"},{"date" => "2014-03-13T13:44:52","version" => "1.41_07"},{"date" => "2014-03-19T15:29:13","version" => "1.42"},{"date" => "2014-03-25T18:50:08","version" => "1.43_01"},{"date" => "2014-03-25T19:58:13","version" => "1.43_02"},{"date" => "2014-06-12T05:01:15","version" => "1.43_03"},{"date" => "2014-07-21T01:13:47","version" => "1.43_04"},{"date" => "2014-07-21T05:45:41","version" => "1.43_05"},{"date" => "2014-07-22T00:31:31","version" => "1.43_06"},{"date" => "2014-07-29T17:03:09","version" => "1.43_07"},{"date" => "2014-08-21T09:01:11","version" => "1.43_08"},{"date" => "2014-10-20T07:50:46","version" => "1.43_09"},{"date" => "2014-10-22T14:15:00","version" => "1.44"},{"date" => "2014-10-22T15:33:37","version" => "1.45_01"},{"date" => "2014-10-23T08:21:27","version" => "1.45_02"},{"date" => "2014-10-24T17:57:53","version" => "1.45_03"},{"date" => "2014-10-28T08:28:00","version" => "1.45_04"},{"date" => "2014-11-25T04:07:43","version" => "1.45_05"},{"date" => "2014-11-26T08:52:49","version" => "1.45_06"},{"date" => "2014-12-10T06:23:03","version" => "1.46"},{"date" => "2015-02-17T07:00:46","version" => "1.47_01"},{"date" => "2015-04-16T13:30:38","version" => "1.47_02"},{"date" => "2015-04-16T14:45:00","version" => "1.47_03"},{"date" => "2015-05-01T17:37:17","version" => "1.47_04"},{"date" => "2015-05-08T13:49:32","version" => "1.47_05"},{"date" => "2015-06-11T16:10:44","version" => "1.48"},{"date" => "2015-08-04T11:18:05","version" => "1.49_01"},{"date" => "2015-10-10T03:43:45","version" => "1.49_02"},{"date" => "2015-11-05T05:52:27","version" => "1.49_03"},{"date" => "2015-11-24T12:59:11","version" => "1.49_04"},{"date" => "2016-01-11T13:32:43","version" => "1.49_05"},{"date" => "2016-01-15T03:40:44","version" => "1.49_06"},{"date" => "2016-01-21T01:11:59","version" => "1.49_07"},{"date" => "2016-01-30T00:55:58","version" => "1.49_08"},{"date" => "2016-02-10T15:04:42","version" => "1.50"},{"date" => "2016-02-20T01:03:50","version" => "1.51_01"},{"date" => "2016-02-20T01:49:29","version" => "1.51_02"},{"date" => "2016-02-20T11:06:51","version" => "1.51_03"},{"date" => "2016-03-07T04:33:35","version" => "1.51_04"},{"date" => "2016-06-23T01:22:57","version" => "1.51_05"},{"date" => "2016-10-15T00:21:14","version" => "1.51_06"},{"date" => "2016-10-16T05:16:29","version" => "1.51_07"},{"date" => "2016-11-15T13:02:35","version" => "1.52"},{"date" => "2016-11-26T01:34:30","version" => "1.53_01"},{"date" => "2016-12-24T02:36:45","version" => "1.54"},{"date" => "2017-01-03T15:42:47","version" => "1.55_01"},{"date" => "2017-01-07T16:49:21","version" => "1.55_02"},{"date" => "2017-02-14T01:31:43","version" => "1.55_03"},{"date" => "2017-11-21T17:07:32","version" => "1.55_04"},{"date" => "2017-12-15T18:52:29","version" => "1.55_05"},{"date" => "2018-01-27T07:33:51","version" => "1.55_06"},{"date" => "2018-01-27T07:42:58","version" => "1.55_07"},{"date" => "2018-02-28T09:01:25","version" => "1.56"},{"date" => "2018-03-21T06:45:29","version" => "1.57_01"},{"date" => "2018-03-28T11:56:19","version" => "1.58"},{"date" => "2018-09-16T19:25:50","version" => "1.59_01"},{"date" => "2018-09-30T06:09:34","version" => "1.59_02"},{"date" => "2018-11-03T12:14:20","version" => "1.59_03"},{"date" => "2018-12-01T02:42:29","version" => "1.60"},{"date" => "2018-12-01T08:01:30","version" => "1.61_01"},{"date" => "2018-12-01T09:10:18","version" => "1.61_02"},{"date" => "2018-12-19T13:03:22","version" => "1.61_03"},{"date" => "2018-12-22T06:37:21","version" => "1.61_04"},{"date" => "2018-12-28T17:59:27","version" => "1.62"},{"date" => "2019-01-25T22:31:45","version" => "1.63_01"},{"date" => "2019-02-13T19:09:44","version" => "1.63_02"},{"date" => "2019-02-14T16:56:40","version" => "1.63_03"},{"date" => "2019-05-24T16:39:18","version" => "1.63_04"},{"date" => "2019-07-11T17:50:51","version" => "1.63_05"},{"date" => "2019-08-12T09:02:59","version" => "1.64"},{"date" => "2020-01-18T01:56:18","version" => "1.65_01"},{"date" => "2020-02-08T13:02:59","version" => "1.65_02"},{"date" => "2020-07-26T16:42:08","version" => "1.65_03"},{"date" => "2020-08-30T02:14:15","version" => "1.66"},{"date" => "2020-11-24T12:57:56","version" => "1.67_01"},{"date" => "2020-12-05T17:06:24","version" => "1.67_02"},{"date" => "2021-03-30T21:37:13","version" => "1.67_03"},{"date" => "2021-05-30T22:56:01","version" => "1.67_04"},{"date" => "2021-06-12T23:39:11","version" => "1.67_05"},{"date" => "2021-06-14T03:49:54","version" => "1.67_06"},{"date" => "2021-06-19T00:57:41","version" => "1.67_07"},{"date" => "2021-07-22T05:30:17","version" => "1.68"},{"date" => "2021-07-29T21:09:19","version" => "1.69_01"},{"date" => "2021-07-30T14:21:39","version" => "1.69_02"},{"date" => "2021-08-01T10:20:33","version" => "1.70"},{"date" => "2021-12-01T17:03:29","version" => "1.71_01"},{"date" => "2022-01-06T20:51:05","version" => "1.71_02"},{"date" => "2022-02-23T10:49:28","version" => "1.71_03"},{"date" => "2022-02-26T00:59:40","version" => "1.71_04"},{"date" => "2022-02-26T02:49:09","version" => "1.71_05"},{"date" => "2022-03-12T02:54:15","version" => "1.71_06"},{"date" => "2022-10-25T18:36:30","version" => "1.71_07"},{"date" => "2022-11-03T16:28:17","version" => "1.72"},{"date" => "2023-07-09T01:04:52","version" => "1.73_01"},{"date" => "2023-09-19T17:26:03","version" => "1.74"},{"date" => "2024-09-17T14:05:40","version" => "1.75_01"},{"date" => "2024-10-19T04:47:07","version" => "1.76"},{"date" => "2025-11-24T04:21:58","version" => "1.77_01"},{"date" => "2025-11-24T08:08:46","version" => "1.77_02"},{"date" => "2025-12-27T02:02:17","version" => "1.77_03"},{"date" => "2026-01-02T01:23:08","version" => "1.78"}]},"DBD-mysql" => {"advisories" => [{"affected_versions" => ["<4.044"],"cves" => ["CVE-2017-10788"],"description" => "The DBD::mysql module through 4.043 for Perl allows remote attackers to cause a denial of service (use-after-free and application crash) or possibly have unspecified other impact by triggering (1) certain error responses from a MySQL server or (2) a loss of a network connection to a MySQL server. The use-after-free defect was introduced by relying on incorrect Oracle mysql_stmt_close documentation and code examples.\n","distribution" => "DBD-mysql","fixed_versions" => [">=4.044"],"id" => "CPANSA-DBD-mysql-2017-02","references" => ["https://github.com/perl5-dbi/DBD-mysql/issues/120","http://www.securityfocus.com/bid/99374","http://seclists.org/oss-sec/2017/q2/443"],"reported" => "2017-04-13"},{"affected_versions" => ["<4.044"],"cves" => ["CVE-2017-10789"],"description" => "The DBD::mysql module through 4.043 for Perl uses the mysql_ssl=1 setting to mean that SSL is optional (even though this setting's documentation has a \"your communication with the server will be encrypted\" statement), which allows man-in-the-middle attackers to spoof servers via a cleartext-downgrade attack, a related issue to CVE-2015-3152.\n","distribution" => "DBD-mysql","fixed_versions" => [">=4.044"],"id" => "CPANSA-DBD-mysql-2017-01","references" => ["https://github.com/perl5-dbi/DBD-mysql/pull/114"],"reported" => "2017-03-23"},{"affected_versions" => [">=2.9003,<4.039"],"cves" => ["CVE-2016-1249"],"description" => "Out-of-bounds read.\n","distribution" => "DBD-mysql","fixed_versions" => ["<2.9003,>=4.039"],"id" => "CPANSA-DBD-mysql-2016-03","references" => ["https://github.com/perl5-dbi/DBD-mysql/commit/793b72b1a0baa5070adacaac0e12fd995a6fbabe"],"reported" => "2016-11-16"},{"affected_versions" => ["<4.037"],"cves" => ["CVE-2016-1246"],"description" => "Buffer overflow in the DBD::mysql module before 4.037 for Perl allows context-dependent attackers to cause a denial of service (crash) via vectors related to an error message.\n","distribution" => "DBD-mysql","fixed_versions" => [">=4.037"],"id" => "CPANSA-DBD-mysql-2016-02","references" => ["https://github.com/perl5-dbi/DBD-mysql/commit/7c164a0c86cec6ee95df1d141e67b0e85dfdefd2","http://blogs.perl.org/users/mike_b/2016/10/security-release---buffer-overflow-in-dbdmysql-perl-library.html"],"reported" => "2016-10-02"},{"affected_versions" => ["<4.034"],"cves" => ["CVE-2015-8949"],"description" => "Use-after-free vulnerability in the my_login function in DBD::mysql before 4.033_01 allows attackers to have unspecified impact by leveraging a call to mysql_errno after a failure of my_login.\n","distribution" => "DBD-mysql","fixed_versions" => [">=4.034"],"id" => "CPANSA-DBD-mysql-2016-01","references" => ["https://github.com/perl5-dbi/DBD-mysql/commit/cf0aa7751f6ef8445e9310a64b14dc81460ca156"],"reported" => "2016-08-19"},{"affected_versions" => ["<4.041"],"cves" => ["CVE-2016-1251"],"description" => "There is a vulnerability of type use-after-free affecting DBD::mysql (aka DBD-mysql or the Database Interface (DBI) MySQL driver for Perl) 3.x and 4.x before 4.041 when used with mysql_server_prepare=1.\n","distribution" => "DBD-mysql","fixed_versions" => [">=4.041"],"id" => "CPANSA-DBD-mysql-2015-01","references" => ["https://github.com/perl5-dbi/DBD-mysql/commit/3619c170461a3107a258d1fd2d00ed4832adb1b1"],"reported" => "2015-12-27"},{"affected_versions" => ["<4.028"],"cves" => ["CVE-2014-9906"],"description" => "Use-after-free vulnerability in DBD::mysql before 4.029 allows attackers to cause a denial of service (program crash) or possibly execute arbitrary code via vectors related to a lost server connection.\n","distribution" => "DBD-mysql","fixed_versions" => [">=4.028"],"id" => "CPANSA-DBD-mysql-2014-01","references" => ["https://github.com/perl5-dbi/DBD-mysql/commit/a56ae87a4c1c1fead7d09c3653905841ccccf1cc","https://rt.cpan.org/Public/Bug/Display.html?id=97625"],"reported" => "2014-07-30"}],"main_module" => "DBD::mysql","versions" => [{"date" => "2000-04-15T20:17:36","version" => "v1.2212."},{"date" => "2001-05-06T21:47:46","version" => "2.0900"},{"date" => "2001-05-25T21:24:45","version" => "2.0901"},{"date" => "2001-07-09T21:10:17","version" => "2.0902"},{"date" => "2001-10-28T22:53:19","version" => "2.0903"},{"date" => "2001-10-31T04:01:07","version" => "2.1000"},{"date" => "2001-11-04T17:55:04","version" => "2.1001"},{"date" => "2001-11-04T18:22:30","version" => "2.1002"},{"date" => "2001-11-05T20:14:34","version" => "2.1003"},{"date" => "2001-11-13T01:24:26","version" => "2.1004"},{"date" => "2001-12-13T09:07:53","version" => "2.1005"},{"date" => "2001-12-27T18:10:04","version" => "2.1007"},{"date" => "2001-12-27T18:10:21","version" => "2.1006"},{"date" => "2001-12-28T17:06:05","version" => "2.1008"},{"date" => "2002-01-01T20:02:26","version" => "2.1009"},{"date" => "2002-01-07T21:33:21","version" => "2.1010"},{"date" => "2002-02-12T11:09:53","version" => "2.1011"},{"date" => "2002-04-12T07:21:06","version" => "2.1012"},{"date" => "2002-04-15T07:49:36","version" => "2.1013"},{"date" => "2002-04-17T21:24:26","version" => "2.1014"},{"date" => "2002-04-29T20:53:41","version" => "2.1015"},{"date" => "2002-05-01T20:07:05","version" => "2.1016"},{"date" => "2002-05-02T20:59:04","version" => "2.1017"},{"date" => "2002-08-13T17:52:25","version" => "2.1018"},{"date" => "2002-09-16T18:42:20","version" => "2.1019"},{"date" => "2002-09-23T20:42:50","version" => "2.1020"},{"date" => "2002-12-17T20:46:14","version" => "2.1021"},{"date" => "2003-01-03T02:46:24","version" => "2.1022"},{"date" => "2003-01-19T21:19:03","version" => "2.1023"},{"date" => "2003-01-20T12:08:27","version" => "2.1024"},{"date" => "2003-02-07T21:09:44","version" => "2.1025"},{"date" => "2003-03-03T20:46:27","version" => "2.1026"},{"date" => "2003-05-31T18:08:15","version" => "2.1027"},{"date" => "2003-06-25T16:12:36","version" => "2.1028"},{"date" => "2003-06-27T04:32:05","version" => "2.9002"},{"date" => "2003-09-12T17:04:42","version" => "2.9003_1"},{"date" => "2003-10-27T03:39:04","version" => "2.9003"},{"date" => "2004-07-01T03:24:14","version" => "2.9004_2"},{"date" => "2004-07-14T03:07:34","version" => "2.9004"},{"date" => "2004-10-20T17:27:25","version" => "2.9005_1"},{"date" => "2004-10-28T00:39:25","version" => "2.9005_3"},{"date" => "2005-03-29T02:43:14","version" => "2.9005"},{"date" => "2005-04-04T04:27:00","version" => "2.9006"},{"date" => "2005-04-27T00:13:49","version" => "2.9015_3"},{"date" => "2005-04-27T00:14:06","version" => "2.9007"},{"date" => "2005-06-06T01:39:20","version" => "2.9008"},{"date" => "2005-07-01T01:48:20","version" => "3.0000"},{"date" => "2005-07-03T21:56:11","version" => "3.0000_0"},{"date" => "2005-07-04T15:53:40","version" => "3.0001_0"},{"date" => "2005-07-04T16:16:00","version" => "3.0001_1"},{"date" => "2005-07-07T01:14:17","version" => "3.0001"},{"date" => "2005-07-07T01:22:39","version" => "3.0001_2"},{"date" => "2005-07-08T05:37:13","version" => "3.0001_3"},{"date" => "2005-07-11T16:49:47","version" => "3.0002"},{"date" => "2005-08-04T02:50:35","version" => "3.0002_1"},{"date" => "2005-09-26T23:22:57","version" => "3.0002_2"},{"date" => "2005-09-28T18:58:55","version" => "3.0002_3"},{"date" => "2005-11-06T21:47:29","version" => "3.0002_4"},{"date" => "2006-02-01T23:20:01","version" => "3.0002_5"},{"date" => "2006-05-04T17:49:06","version" => "3.0003"},{"date" => "2006-05-04T17:49:23","version" => "3.0003_1"},{"date" => "2006-05-21T17:28:22","version" => "3.0004"},{"date" => "2006-05-21T17:28:33","version" => "3.0004_1"},{"date" => "2006-06-10T01:21:49","version" => "3.0005_1"},{"date" => "2006-06-10T01:22:01","version" => "3.0005"},{"date" => "2006-06-11T17:05:25","version" => "3.0006"},{"date" => "2006-06-11T17:05:36","version" => "3.0006_1"},{"date" => "2006-09-08T23:12:02","version" => "3.0007"},{"date" => "2006-09-08T23:13:45","version" => "3.0007_1"},{"date" => "2006-10-07T12:59:23","version" => "3.0007_2"},{"date" => "2006-10-16T13:42:13","version" => "3.0008"},{"date" => "2006-10-16T13:42:24","version" => "3.0008_1"},{"date" => "2006-12-24T14:11:04","version" => "4.00"},{"date" => "2007-01-08T01:11:12","version" => "4.001"},{"date" => "2007-03-02T03:32:59","version" => "4.002"},{"date" => "2007-03-02T14:13:37","version" => "4.003"},{"date" => "2007-03-22T22:31:22","version" => "4.004"},{"date" => "2007-06-08T15:33:34","version" => "4.005"},{"date" => "2007-12-26T22:50:48","version" => "4.006"},{"date" => "2008-05-11T15:56:07","version" => "4.007"},{"date" => "2008-08-15T14:06:50","version" => "4.008"},{"date" => "2008-10-22T01:05:54","version" => "4.009"},{"date" => "2008-10-24T14:00:41","version" => "4.010"},{"date" => "2009-04-14T02:40:31","version" => "4.011"},{"date" => "2009-06-19T02:08:06","version" => "4.012"},{"date" => "2009-09-16T18:37:29","version" => "4.013"},{"date" => "2010-04-15T03:17:58","version" => "4.014"},{"date" => "2010-07-09T19:48:58","version" => "4.015"},{"date" => "2010-07-10T16:50:49","version" => "4.016"},{"date" => "2010-08-12T05:50:17","version" => "4.017"},{"date" => "2010-10-26T16:59:27","version" => "4.018"},{"date" => "2011-05-09T01:28:25","version" => "4.019"},{"date" => "2011-08-20T18:45:49","version" => "4.020"},{"date" => "2012-04-28T14:18:16","version" => "4.021"},{"date" => "2012-08-30T02:00:19","version" => "4.022"},{"date" => "2013-04-12T21:48:10","version" => "4.023"},{"date" => "2013-09-17T16:04:11","version" => "4.024"},{"date" => "2013-11-04T18:29:18","version" => "4.025"},{"date" => "2014-01-16T01:33:03","version" => "4.026"},{"date" => "2014-03-19T14:25:36","version" => "4.027"},{"date" => "2014-08-01T19:59:28","version" => "4.028"},{"date" => "2014-12-09T02:39:44","version" => "4.029"},{"date" => "2015-01-28T03:53:42","version" => "4.030_01"},{"date" => "2015-03-02T20:44:31","version" => "4.030_02"},{"date" => "2015-03-06T20:12:05","version" => "4.031"},{"date" => "2015-04-16T22:28:43","version" => "4.032_01"},{"date" => "2015-07-21T12:15:24","version" => "4.032"},{"date" => "2015-10-25T19:59:17","version" => "4.032_03"},{"date" => "2015-10-27T03:37:29","version" => "4.033"},{"date" => "2015-12-15T07:16:36","version" => "4.033_01"},{"date" => "2015-12-18T07:00:41","version" => "4.033_02"},{"date" => "2016-07-04T19:32:50","version" => "4.033_03"},{"date" => "2016-07-06T06:32:05","version" => "4.034"},{"date" => "2016-07-09T05:50:13","version" => "4.035"},{"date" => "2016-08-01T06:29:25","version" => "4.035_01"},{"date" => "2016-08-11T08:11:18","version" => "4.035_02"},{"date" => "2016-08-19T15:52:10","version" => "4.035_03"},{"date" => "2016-08-23T05:59:26","version" => "4.036"},{"date" => "2016-10-03T07:00:29","version" => "4.037"},{"date" => "2016-10-14T20:56:49","version" => "4.037_01"},{"date" => "2016-10-19T19:37:55","version" => "4.037_02"},{"date" => "2016-10-20T02:33:04","version" => "4.038"},{"date" => "2016-10-30T08:45:31","version" => "4.038_01"},{"date" => "2016-11-16T03:57:57","version" => "4.039"},{"date" => "2016-11-19T19:56:51","version" => "4.040"},{"date" => "2016-11-28T20:40:41","version" => "4.041"},{"date" => "2016-12-13T06:59:09","version" => "4.041_01"},{"date" => "2017-02-28T20:57:20","version" => "4.041_02"},{"date" => "2017-03-08T20:32:52","version" => "4.042"},{"date" => "2017-06-29T21:12:09","version" => "4.043"},{"date" => "2018-01-23T01:53:30","version" => "4.044"},{"date" => "2018-02-07T21:43:00","version" => "4.044"},{"date" => "2018-02-08T20:30:55","version" => "4.045"},{"date" => "2018-02-08T20:48:11","version" => "4.046"},{"date" => "2018-03-09T20:27:44","version" => "4.046_01"},{"date" => "2018-09-09T03:02:20","version" => "4.047"},{"date" => "2018-09-15T12:46:51","version" => "4.048"},{"date" => "2018-11-17T18:58:09","version" => "4.049"},{"date" => "2019-01-09T09:07:15","version" => "4.050"},{"date" => "2019-10-07T10:06:13","version" => "4.018_01"},{"date" => "2023-10-04T07:10:45","version" => "4.051"},{"date" => "2023-10-04T07:20:03","version" => "5.001"},{"date" => "2023-10-24T09:02:42","version" => "5.002"},{"date" => "2023-12-01T07:13:15","version" => "4.052"},{"date" => "2023-12-01T07:14:42","version" => "5.003"},{"date" => "2024-03-19T08:16:14","version" => "5.004"},{"date" => "2024-05-01T09:04:33","version" => "5.005"},{"date" => "2024-06-04T19:59:44","version" => "5.006"},{"date" => "2024-07-01T06:02:18","version" => "5.007"},{"date" => "2024-07-30T07:47:38","version" => "5.008"},{"date" => "2024-09-19T08:35:24","version" => "5.009"},{"date" => "2024-11-11T06:43:01","version" => "5.010"},{"date" => "2025-01-06T06:52:30","version" => "4.053"},{"date" => "2025-01-06T06:55:27","version" => "5.011"},{"date" => "2025-04-11T16:18:15","version" => "5.012"},{"date" => "2025-08-03T08:51:10","version" => "4.054"},{"date" => "2025-08-03T08:51:21","version" => "5.013"},{"date" => "2026-02-23T07:39:11","version" => "4.055"}]},"DBD-mysqlPP" => {"advisories" => [{"affected_versions" => ["<0.93"],"cves" => [],"description" => "SQL injection.\n","distribution" => "DBD-mysqlPP","fixed_versions" => [">=0.03"],"id" => "CPANSA-DBD-mysqlPP-2011-01","references" => ["https://metacpan.org/changes/distribution/DBD-mysqlPP","https://jvn.jp/en/jp/JVN51216285/index.html"],"reported" => "2011-10-14","severity" => "high"}],"main_module" => "DBD::mysqlPP","versions" => [{"date" => "2002-04-04T07:20:36","version" => "0.02"},{"date" => "2002-04-15T10:26:39","version" => "0.03"},{"date" => "2003-01-24T11:14:14","version" => "0.04"},{"date" => "2011-10-21T23:07:07","version" => "0.05"},{"date" => "2011-10-26T22:17:22","version" => "0.06"},{"date" => "2011-11-17T22:24:50","version" => "0.07"}]},"DBI" => {"advisories" => [{"affected_versions" => ["<1.643"],"cves" => ["CVE-2020-14393"],"description" => "A buffer overflow was found in perl-DBI < 1.643 in DBI.xs. A local attacker who is able to supply a string longer than 300 characters could cause an out-of-bounds write, affecting the availability of the service or integrity of data.\n","distribution" => "DBI","fixed_versions" => [">=1.643"],"id" => "CPANSA-DBI-2020-01","references" => ["https://metacpan.org/changes/distribution/DBI","https://bugzilla.redhat.com/show_bug.cgi?id=1877409"],"reported" => "2020-09-16","severity" => "high"},{"affected_versions" => ["<1.643"],"cves" => ["CVE-2020-14392"],"description" => "An untrusted pointer dereference flaw was found in Perl-DBI < 1.643. A local attacker who is able to manipulate calls to dbd_db_login6_sv() could cause memory corruption, affecting the service's availability.\n","distribution" => "DBI","fixed_versions" => [">=1.643"],"id" => "CPANSA-DBI-2020-03","references" => ["https://metacpan.org/changes/distribution/DBI","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/JXLKODJ7B57GITDEZZXNSHPK4VBYXYHR/","https://bugzilla.redhat.com/show_bug.cgi?id=1877402","https://bugzilla.redhat.com/show_bug.cgi?id=1877402","https://lists.debian.org/debian-lts-announce/2020/09/msg00026.html","http://lists.opensuse.org/opensuse-security-announce/2020-09/msg00067.html","http://lists.opensuse.org/opensuse-security-announce/2020-09/msg00074.html","https://usn.ubuntu.com/4503-1/"],"reported" => "2020-06-17","severity" => "high"},{"affected_versions" => ["<1.643"],"cves" => ["CVE-2019-20919"],"description" => "An issue was discovered in the DBI module before 1.643 for Perl. The hv_fetch() documentation requires checking for NULL and the code does that. But, shortly thereafter, it calls SvOK(profile), causing a NULL pointer dereference.\n","distribution" => "DBI","fixed_versions" => [">=1.643"],"id" => "CPANSA-DBI-2020-02","references" => ["https://metacpan.org/changes/distribution/DBI","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/JXLKODJ7B57GITDEZZXNSHPK4VBYXYHR/","https://cve.mitre.org/cgi-bin/cvename.cgi?name=CVE-2019-20919","https://github.com/perl5-dbi/dbi/commit/eca7d7c8f43d96f6277e86d1000e842eb4cc67ff","https://bugzilla.redhat.com/show_bug.cgi?id=1877405","https://lists.debian.org/debian-lts-announce/2020/09/msg00026.html","https://lists.opensuse.org/archives/list/security-announce\@lists.opensuse.org/message/US6VXPKVAYHOKNFSAFLM3FWNYZSJKQHS/","https://lists.opensuse.org/archives/list/security-announce\@lists.opensuse.org/message/KJN7E27GD6QQ2CRGEJ3TNW2DJFXA2AKN/","https://ubuntu.com/security/notices/USN-4534-1"],"reported" => "2020-09-17","severity" => "high"},{"affected_versions" => ["<1.632"],"cves" => [],"description" => "DBD::File drivers open files from folders other than specifically passed using the f_dir attribute.\n","distribution" => "DBI","fixed_versions" => [">=1.632"],"id" => "CPANSA-DBI-2014-01","references" => ["https://metacpan.org/changes/distribution/DBI","https://rt.cpan.org/Public/Bug/Display.html?id=99508"],"reported" => "2014-10-15","severity" => "high"},{"affected_versions" => ["<1.47"],"cves" => ["CVE-2005-0077"],"description" => "Allows local users to overwrite arbitrary files via a symlink attack on a temporary PID file.\n","distribution" => "DBI","fixed_versions" => [">=1.47"],"id" => "CPANSA-DBI-2005-01","references" => ["https://metacpan.org/changes/distribution/DBI"],"reported" => "2005-05-02"},{"affected_versions" => ["<1.643"],"cves" => ["CVE-2014-10402"],"description" => "An issue was discovered in the DBI module through 1.643 for Perl. DBD::File drivers can open files from folders other than those specifically passed via the f_dir attribute in the data source name (DSN). NOTE: this issue exists because of an incomplete fix for CVE-2014-10401.\n","distribution" => "DBI","fixed_versions" => [">=1.644"],"id" => "CPANSA-DBI-2014-10402","references" => ["https://rt.cpan.org/Public/Bug/Display.html?id=99508#txn-1911590","https://metacpan.org/release/HMBRAND/DBI-1.643_01/view/Changes"],"reported" => "2020-09-16","severity" => "medium"},{"affected_versions" => ["<1.632"],"cves" => ["CVE-2014-10401"],"description" => "An issue was discovered in the DBI module before 1.632 for Perl. DBD::File drivers can open files from folders other than those specifically passed via the f_dir attribute.\n","distribution" => "DBI","fixed_versions" => [">=1.644"],"id" => "CPANSA-DBI-2014-10401","references" => ["https://rt.cpan.org/Public/Bug/Display.html?id=99508","https://metacpan.org/pod/distribution/DBI/Changes#Changes-in-DBI-1.632-9th-Nov-2014","https://github.com/perl5-dbi/dbi/commit/caedc0d7d602f5b2ae5efc1b00f39efeafb7b05a","https://usn.ubuntu.com/4509-1/","https://metacpan.org/release/HMBRAND/DBI-1.643_01/view/Changes"],"reported" => "2020-09-11","severity" => "medium"},{"affected_versions" => ["<1.628"],"cves" => ["CVE-2013-7491"],"description" => "An issue was discovered in the DBI module before 1.628 for Perl. Stack corruption occurs when a user-defined function requires a non-trivial amount of memory and the Perl stack gets reallocated.\n","distribution" => "DBI","fixed_versions" => [">=1.628"],"id" => "CPANSA-DBI-2013-7491","references" => ["https://github.com/perl5-dbi/dbi/commit/401f1221311c71f760e21c98772f0f7e3cbead1d","https://metacpan.org/pod/distribution/DBI/Changes#Changes-in-DBI-1.628-22nd-July-2013","https://rt.cpan.org/Public/Bug/Display.html?id=85562"],"reported" => "2020-09-11","severity" => "medium"},{"affected_versions" => ["<1.632"],"cves" => ["CVE-2013-7490"],"description" => "An issue was discovered in the DBI module before 1.632 for Perl. Using many arguments to methods for Callbacks may lead to memory corruption.\n","distribution" => "DBI","fixed_versions" => [">=1.632"],"id" => "CPANSA-DBI-2013-7490","references" => ["https://github.com/perl5-dbi/dbi/commit/a8b98e988d6ea2946f5f56691d6d5ead53f65766","https://metacpan.org/pod/distribution/DBI/Changes#Changes-in-DBI-1.632-9th-Nov-2014","https://rt.cpan.org/Public/Bug/Display.html?id=86744#txn-1880941","https://usn.ubuntu.com/4509-1/"],"reported" => "2020-09-11","severity" => "medium"}],"main_module" => "DBI","versions" => [{"date" => "1995-10-27T08:14:00","version" => "0.64"},{"date" => "1996-02-15T22:07:00","version" => "0.67"},{"date" => "1996-04-22T10:22:00","version" => "0.68"},{"date" => "1996-05-07T19:46:00","version" => "0.69"},{"date" => "1996-06-16T21:08:00","version" => "0.70"},{"date" => "1996-07-10T00:49:00","version" => "0.71"},{"date" => "1996-09-23T16:33:00","version" => "0.72"},{"date" => "1996-10-15T00:58:00","version" => "0.73"},{"date" => "1997-01-14T16:59:00","version" => "0.74"},{"date" => "1997-01-27T21:59:00","version" => "0.75"},{"date" => "1997-02-03T18:54:00","version" => "0.76"},{"date" => "1997-02-21T14:27:00","version" => "0.77"},{"date" => "1997-03-28T14:36:00","version" => "0.78"},{"date" => "1997-04-07T18:28:00","version" => "0.79"},{"date" => "1997-05-07T11:45:00","version" => "0.80"},{"date" => "1997-05-07T14:05:00","version" => "0.81"},{"date" => "1997-05-23T15:56:00","version" => "0.82"},{"date" => "1997-06-11T21:40:00","version" => "0.83"},{"date" => "1997-06-20T15:36:00","version" => "0.84"},{"date" => "1997-06-25T10:25:00","version" => "0.85"},{"date" => "1997-07-16T16:38:00","version" => "0.001"},{"date" => "1997-07-18T11:27:00","version" => "0.87"},{"date" => "1997-07-22T21:27:00","version" => "0.88"},{"date" => "1997-07-25T13:46:55","version" => "0.89"},{"date" => "1997-09-05T19:38:52","version" => "0.90"},{"date" => "1997-12-10T17:15:14","version" => "0.91"},{"date" => "1998-02-05T20:45:45","version" => "0.92"},{"date" => "1998-02-13T15:21:52","version" => "0.93"},{"date" => "1998-08-10T03:23:46","version" => "0.94"},{"date" => "1998-08-11T13:21:19","version" => "0.95"},{"date" => "1998-08-14T20:38:42","version" => "1.00"},{"date" => "1998-09-02T14:59:47","version" => "1.01"},{"date" => "1998-09-04T12:29:52","version" => "1.02"},{"date" => "1999-01-18T21:52:15","version" => "1.06"},{"date" => "1999-05-13T01:49:11","version" => "1.08"},{"date" => "1999-06-02T13:44:40","version" => "1.08"},{"date" => "1999-06-09T20:57:59","version" => "1.09"},{"date" => "1999-06-13T23:52:03","version" => "1.10"},{"date" => "1999-06-17T13:22:36","version" => "1.11"},{"date" => "1999-06-29T23:07:41","version" => "1.12"},{"date" => "1999-07-12T03:28:41","version" => "1.13"},{"date" => "2000-06-11T02:39:59","version" => "1.03_80"},{"date" => "2000-06-14T20:30:57","version" => "1.14"},{"date" => "2001-03-30T15:03:31","version" => "1.15"},{"date" => "2001-05-29T23:25:57","version" => "1.16"},{"date" => "2001-06-04T17:12:30","version" => "1.17"},{"date" => "2001-06-04T19:00:37","version" => "1.18"},{"date" => "2001-07-20T22:29:24","version" => "1.19"},{"date" => "2001-08-24T23:32:10","version" => "1.20"},{"date" => "2002-01-10T15:25:45","version" => "1.201"},{"date" => "2002-02-07T03:30:16","version" => "1.21"},{"date" => "2002-03-13T14:18:00","version" => "1.21"},{"date" => "2002-05-22T13:42:15","version" => "1.22"},{"date" => "2002-05-25T17:38:03","version" => "1.23"},{"date" => "2002-06-05T03:32:38","version" => "1.24"},{"date" => "2002-06-05T22:42:04","version" => "1.25"},{"date" => "2002-06-13T12:30:47","version" => "1.26"},{"date" => "2002-06-13T15:19:06","version" => "1.27"},{"date" => "2002-06-14T13:13:53","version" => "1.28"},{"date" => "2002-06-26T09:34:24","version" => "1.28"},{"date" => "2002-07-15T11:24:40","version" => "1.29"},{"date" => "2002-07-18T14:27:25","version" => "1.30"},{"date" => "2002-11-30T00:49:54","version" => "1.31"},{"date" => "2002-12-01T23:01:26","version" => "1.32"},{"date" => "2002-12-20T16:23:29","version" => "1.32"},{"date" => "2003-02-26T18:01:24","version" => "1.32_90"},{"date" => "2003-02-27T00:25:32","version" => "1.33"},{"date" => "2003-02-28T17:53:35","version" => "1.34"},{"date" => "2003-03-07T22:02:20","version" => "1.35"},{"date" => "2003-05-14T11:13:39","version" => "1.36"},{"date" => "2003-05-15T18:02:26","version" => "1.37"},{"date" => "2003-08-25T20:36:26","version" => "1.38"},{"date" => "2003-11-27T23:46:40","version" => "1.39"},{"date" => "2004-01-08T14:04:59","version" => "1.39"},{"date" => "2004-02-23T14:54:21","version" => "1.41"},{"date" => "2004-03-12T16:40:08","version" => "1.41"},{"date" => "2004-07-05T10:02:05","version" => "1.43"},{"date" => "2004-10-05T21:27:23","version" => "1.44"},{"date" => "2004-10-06T13:49:20","version" => "1.45"},{"date" => "2004-11-16T12:38:32","version" => "1.46"},{"date" => "2005-02-02T11:28:46","version" => "1.47"},{"date" => "2005-03-14T17:03:33","version" => "1.48"},{"date" => "2005-11-29T19:59:40","version" => "1.49"},{"date" => "2005-12-14T16:55:16","version" => "1.50"},{"date" => "2006-04-19T15:56:38","version" => "1.45"},{"date" => "2006-06-06T12:08:36","version" => "1.51"},{"date" => "2006-08-08T21:13:32","version" => "1.52"},{"date" => "2006-11-02T00:38:01","version" => "1.53"},{"date" => "2007-02-23T17:15:23","version" => "1.54"},{"date" => "2007-05-04T14:56:38","version" => "1.55"},{"date" => "2007-05-10T14:04:04","version" => "1.56"},{"date" => "2007-05-13T22:00:58","version" => "1.56"},{"date" => "2007-06-13T16:45:34","version" => "1.57"},{"date" => "2007-06-15T17:06:42","version" => "1.57"},{"date" => "2007-06-18T15:15:31","version" => "1.57"},{"date" => "2007-06-25T22:11:47","version" => "1.58"},{"date" => "2007-08-22T17:02:10","version" => "1.59"},{"date" => "2007-08-23T12:22:26","version" => "1.59"},{"date" => "2007-08-23T13:59:53","version" => "1.59"},{"date" => "2007-08-24T09:19:29","version" => "1.59"},{"date" => "2007-10-16T13:12:55","version" => "1.601"},{"date" => "2007-10-21T22:12:52","version" => "1.601"},{"date" => "2008-02-09T22:06:13","version" => "1.602"},{"date" => "2008-03-22T00:11:03","version" => "1.603"},{"date" => "2008-03-24T14:11:41","version" => "1.604"},{"date" => "2008-06-16T19:19:43","version" => "1.605"},{"date" => "2008-07-22T21:01:09","version" => "1.606"},{"date" => "2008-07-22T21:50:54","version" => "1.607"},{"date" => "2009-05-02T22:58:48","version" => "1.608"},{"date" => "2009-05-05T12:05:19","version" => "1.608"},{"date" => "2009-06-05T22:57:34","version" => "1.609"},{"date" => "2009-06-08T10:29:18","version" => "1.609"},{"date" => "2010-03-02T21:26:39","version" => "1.611"},{"date" => "2010-04-22T11:06:31","version" => "1.611"},{"date" => "2010-04-27T15:13:32","version" => "1.611"},{"date" => "2010-04-29T19:54:44","version" => "1.611"},{"date" => "2010-05-28T10:29:17","version" => "1.612"},{"date" => "2010-06-15T22:47:23","version" => "1.612"},{"date" => "2010-06-16T19:18:05","version" => "1.612"},{"date" => "2010-07-02T14:26:03","version" => "1.612"},{"date" => "2010-07-15T15:00:53","version" => "1.612"},{"date" => "2010-07-16T19:36:42","version" => "1.612"},{"date" => "2010-07-22T17:34:16","version" => "1.613"},{"date" => "2010-07-25T15:50:15","version" => "1.613"},{"date" => "2010-07-30T14:17:33","version" => "1.614"},{"date" => "2010-08-16T16:34:58","version" => "1.614"},{"date" => "2010-08-30T20:11:00","version" => "1.614"},{"date" => "2010-08-30T20:26:37","version" => "1.614"},{"date" => "2010-08-30T20:56:09","version" => "1.614"},{"date" => "2010-09-02T15:44:21","version" => "1.614"},{"date" => "2010-09-09T10:24:11","version" => "1.614"},{"date" => "2010-09-16T16:23:50","version" => "1.614"},{"date" => "2010-09-17T09:48:02","version" => "1.614"},{"date" => "2010-09-21T10:14:29","version" => "1.615"},{"date" => "2010-09-22T12:28:20","version" => "1.615"},{"date" => "2010-12-18T21:51:52","version" => "1.616"},{"date" => "2010-12-21T23:26:46","version" => "1.616"},{"date" => "2010-12-29T14:39:48","version" => "1.616"},{"date" => "2010-12-30T10:26:51","version" => "1.616"},{"date" => "2012-01-02T17:12:53","version" => "1.617"},{"date" => "2012-01-28T09:34:18","version" => "1.617"},{"date" => "2012-01-30T10:06:49","version" => "1.617"},{"date" => "2012-02-07T22:54:02","version" => "1.618"},{"date" => "2012-02-13T18:24:33","version" => "1.618"},{"date" => "2012-02-23T11:05:45","version" => "1.618"},{"date" => "2012-02-25T14:24:39","version" => "1.618"},{"date" => "2012-04-18T11:57:55","version" => "1.619"},{"date" => "2012-04-20T20:21:54","version" => "1.619"},{"date" => "2012-04-23T22:09:14","version" => "1.619"},{"date" => "2012-04-25T12:46:54","version" => "1.620"},{"date" => "2012-05-21T13:06:09","version" => "1.621"},{"date" => "2012-05-22T22:17:06","version" => "1.621"},{"date" => "2012-06-06T16:51:00","version" => "1.622"},{"date" => "2012-07-13T15:24:35","version" => "1.623"},{"date" => "2012-10-30T13:01:14","version" => "1.623"},{"date" => "2012-11-19T23:27:04","version" => "1.623"},{"date" => "2012-12-13T16:26:23","version" => "1.623"},{"date" => "2012-12-21T17:22:01","version" => "1.623"},{"date" => "2013-01-02T10:09:42","version" => "1.623"},{"date" => "2013-03-22T20:41:50","version" => "1.624"},{"date" => "2013-03-28T21:59:38","version" => "1.625"},{"date" => "2013-05-15T11:28:03","version" => "1.626"},{"date" => "2013-05-16T20:30:50","version" => "1.627"},{"date" => "2013-06-24T21:56:27","version" => "1.628"},{"date" => "2013-06-24T22:12:23","version" => "1.628"},{"date" => "2013-06-30T19:08:08","version" => "1.628"},{"date" => "2013-07-02T11:27:23","version" => "1.628"},{"date" => "2013-07-22T13:22:40","version" => "1.628"},{"date" => "2013-10-11T12:28:12","version" => "1.629"},{"date" => "2013-10-13T16:02:52","version" => "1.629"},{"date" => "2013-10-15T12:24:53","version" => "1.629"},{"date" => "2013-10-22T11:58:53","version" => "1.629_50"},{"date" => "2013-10-28T12:51:39","version" => "1.630"},{"date" => "2014-01-13T13:51:01","version" => "1.631"},{"date" => "2014-01-16T11:34:34","version" => "1.631"},{"date" => "2014-01-20T11:12:44","version" => "1.631"},{"date" => "2014-10-23T14:08:22","version" => "1.631"},{"date" => "2014-11-05T11:15:07","version" => "1.632"},{"date" => "2015-01-08T14:31:52","version" => "1.632"},{"date" => "2015-01-11T13:26:05","version" => "1.633"},{"date" => "2015-07-18T13:16:07","version" => "1.633"},{"date" => "2015-07-19T14:34:22","version" => "1.633_91"},{"date" => "2015-07-22T15:27:59","version" => "1.633_92"},{"date" => "2015-08-02T16:52:48","version" => "1.633_93"},{"date" => "2015-08-03T14:52:56","version" => "1.634"},{"date" => "2016-04-23T15:28:02","version" => "1.634"},{"date" => "2016-04-24T11:57:03","version" => "1.635"},{"date" => "2016-04-24T22:20:56","version" => "1.636"},{"date" => "2017-08-14T10:10:55","version" => "1.637"},{"date" => "2017-08-16T09:02:40","version" => "1.637"},{"date" => "2017-12-28T14:40:44","version" => "1.639"},{"date" => "2018-01-28T20:50:53","version" => "1.640"},{"date" => "2018-03-19T18:06:08","version" => "1.641"},{"date" => "2018-10-28T15:08:54","version" => "1.641_90"},{"date" => "2018-10-29T10:43:41","version" => "1.642"},{"date" => "2020-01-26T20:48:52","version" => "1.642_90"},{"date" => "2020-01-31T19:02:41","version" => "1.643"},{"date" => "2024-08-20T11:29:56","version" => "1.643_01"},{"date" => "2024-08-22T07:09:52","version" => "1.643_02"},{"date" => "2024-08-23T17:54:09","version" => "1.644"},{"date" => "2024-09-03T09:25:33","version" => "1.645"},{"date" => "2025-01-11T12:59:58","version" => "1.646"},{"date" => "2025-01-20T08:14:47","version" => "1.647"}]},"DBIx-Class-EncodedColumn" => {"advisories" => [{"affected_versions" => ["<0.11"],"cves" => ["CVE-2025-27551"],"description" => "DBIx::Class::EncodedColumn use the rand() function, which is not cryptographically secure to salt password hashes.  This vulnerability is associated with program files lib/DBIx/Class/EncodedColumn/Digest.pm.  This issue affects DBIx::Class::EncodedColumn until 0.00032.","distribution" => "DBIx-Class-EncodedColumn","fixed_versions" => [">=0.11"],"id" => "CPANSA-DBIx-Class-EncodedColumn-2025-27551","references" => ["https://metacpan.org/release/WREIS/DBIx-Class-EncodedColumn-0.00032/changes","https://security.metacpan.org/docs/guides/random-data-for-security.html"],"reported" => "2025-03-26","severity" => undef},{"affected_versions" => ["<0.11"],"cves" => ["CVE-2025-27552"],"description" => "DBIx::Class::EncodedColumn use the rand() function, which is not cryptographically secure to salt password hashes.  This vulnerability is associated with program files Crypt/Eksblowfish/Bcrypt.pm.  This issue affects DBIx::Class::EncodedColumn until 0.00032.","distribution" => "DBIx-Class-EncodedColumn","fixed_versions" => [">=0.11"],"id" => "CPANSA-DBIx-Class-EncodedColumn-2025-27552","references" => ["https://metacpan.org/release/WREIS/DBIx-Class-EncodedColumn-0.00032/changes","https://security.metacpan.org/docs/guides/random-data-for-security.html"],"reported" => "2025-03-26","severity" => undef}],"main_module" => "DBIx::Class::EncodedColumn","versions" => [{"date" => "2008-01-29T23:47:22","version" => "0.00001_01"},{"date" => "2008-01-31T23:21:56","version" => "0.00001_02"},{"date" => "2008-02-01T00:17:42","version" => "0.00001_03"},{"date" => "2008-02-01T18:52:21","version" => "0.00001"},{"date" => "2008-07-28T22:45:39","version" => "0.00002"},{"date" => "2009-09-01T15:45:04","version" => "0.00003"},{"date" => "2009-09-03T18:11:37","version" => "0.00004"},{"date" => "2009-10-11T15:06:19","version" => "0.00005"},{"date" => "2010-01-16T00:39:59","version" => "0.00006"},{"date" => "2010-04-30T00:12:56","version" => "0.00007"},{"date" => "2010-04-30T17:51:03","version" => "0.00008"},{"date" => "2010-05-17T20:20:11","version" => "0.00009_1"},{"date" => "2010-05-18T14:56:30","version" => "0.00009"},{"date" => "2010-08-27T18:36:04","version" => "0.00010"},{"date" => "2011-04-11T20:21:16","version" => "0.00011"},{"date" => "2013-04-29T14:32:51","version" => "0.00012"},{"date" => "2014-02-27T13:50:24","version" => "0.00013"},{"date" => "2016-05-31T13:31:04","version" => "0.00014"},{"date" => "2016-06-01T14:04:39","version" => "0.00015"},{"date" => "2019-06-12T12:59:07","version" => "0.00016"},{"date" => "2019-09-03T21:54:20","version" => "0.00017"},{"date" => "2019-09-16T18:10:46","version" => "0.00018"},{"date" => "2019-09-19T18:13:13","version" => "0.00019"},{"date" => "2019-09-25T12:34:33","version" => "0.00020"},{"date" => "2025-03-25T14:30:45","version" => "0.00030"},{"date" => "2025-03-25T17:55:22","version" => "0.00031"},{"date" => "2025-03-25T18:05:54","version" => "0.00032"},{"date" => "2025-03-26T11:34:06","version" => "0.1.0"},{"date" => "2025-03-26T11:40:46","version" => "0.11"}]},"DBIx-Custom" => {"advisories" => [{"affected_versions" => ["<0.1641"],"cves" => [],"description" => "SQL injection when passing special column names.\n","distribution" => "DBIx-Custom","fixed_versions" => [">=0.1641"],"id" => "CPANSA-DBIx-Custom-2011-01","references" => ["https://metacpan.org/changes/distribution/DBIx-Custom","https://github.com/yuki-kimoto/DBIx-Custom/commit/5b00b9f9a966e7abecabd91710c8fa893784d919"],"reported" => "2011-01-27","severity" => "high"}],"main_module" => "DBIx::Custom","versions" => [{"date" => "2009-11-08T04:18:19","version" => "0.0101"},{"date" => "2009-11-09T10:46:44","version" => "0.0201"},{"date" => "2009-11-12T14:12:47","version" => "0.0301"},{"date" => "2009-11-15T11:43:40","version" => "0.0401"},{"date" => "2009-11-16T11:10:52","version" => "0.0501"},{"date" => "2009-11-17T12:37:33","version" => "0.0502"},{"date" => "2009-11-19T12:05:50","version" => "0.0601"},{"date" => "2009-11-19T13:37:39","version" => "0.0602"},{"date" => "2009-11-20T12:08:31","version" => "0.0603"},{"date" => "2009-11-23T13:39:53","version" => "0.0604"},{"date" => "2009-11-23T14:45:46","version" => "0.0605"},{"date" => "2009-11-25T13:57:52","version" => "0.0701"},{"date" => "2009-12-01T07:30:25","version" => "0.0702"},{"date" => "2009-12-02T13:59:36","version" => "0.0801"},{"date" => "2009-12-09T14:27:53","version" => "0.0901"},{"date" => "2009-12-22T13:40:07","version" => "0.0902"},{"date" => "2010-01-18T12:42:57","version" => "0.0903"},{"date" => "2010-01-21T14:29:12","version" => "0.0904"},{"date" => "2010-01-22T12:51:23","version" => "0.0905"},{"date" => "2010-01-24T09:49:30","version" => "0.0906"},{"date" => "2010-01-30T00:15:17","version" => "0.1001"},{"date" => "2010-01-30T03:51:04","version" => "0.1101"},{"date" => "2010-05-01T13:02:19","version" => "0.1301"},{"date" => "2010-05-01T23:29:22","version" => "0.1401"},{"date" => "2010-05-02T06:04:57","version" => "0.1402"},{"date" => "2010-05-26T15:13:04","version" => "0.1501"},{"date" => "2010-05-27T14:00:04","version" => "0.1502"},{"date" => "2010-05-28T13:28:16","version" => "0.1503"},{"date" => "2010-06-25T12:11:33","version" => "0.1602"},{"date" => "2010-07-14T13:55:33","version" => "0.1603"},{"date" => "2010-08-03T14:43:14","version" => "0.1604"},{"date" => "2010-08-05T15:17:49","version" => "0.1605"},{"date" => "2010-08-05T15:24:36","version" => "0.1606"},{"date" => "2010-08-06T14:57:35","version" => "0.1607"},{"date" => "2010-08-07T05:49:19","version" => "0.1608"},{"date" => "2010-08-08T04:45:12","version" => "0.1609"},{"date" => "2010-08-08T12:44:43","version" => "0.1610"},{"date" => "2010-08-09T12:08:31","version" => "0.1611"},{"date" => "2010-08-10T11:19:41","version" => "0.1612"},{"date" => "2010-08-10T12:35:17","version" => "0.1613"},{"date" => "2010-08-12T15:01:01","version" => "0.1614"},{"date" => "2010-08-15T04:00:44","version" => "0.1615"},{"date" => "2010-08-24T10:18:06","version" => "0.1616"},{"date" => "2010-09-07T12:12:04","version" => "0.1617"},{"date" => "2010-10-17T05:44:56","version" => "0.1618"},{"date" => "2010-10-20T15:01:35","version" => "0.1619"},{"date" => "2010-10-21T14:38:05","version" => "0.1620"},{"date" => "2010-11-10T06:54:46","version" => "0.1621"},{"date" => "2010-12-20T14:58:38","version" => "0.1622"},{"date" => "2010-12-21T16:10:25","version" => "0.1623"},{"date" => "2010-12-22T08:41:09","version" => "0.1624"},{"date" => "2011-01-01T16:08:48","version" => "0.1625"},{"date" => "2011-01-02T04:21:11","version" => "0.1626"},{"date" => "2011-01-04T15:18:21","version" => "0.1627"},{"date" => "2011-01-12T07:29:29","version" => "0.1628"},{"date" => "2011-01-12T15:35:11","version" => "0.1629"},{"date" => "2011-01-13T15:41:25","version" => "0.1630"},{"date" => "2011-01-17T15:53:44","version" => "0.1631"},{"date" => "2011-01-18T14:43:16","version" => "0.1632"},{"date" => "2011-01-18T15:22:37","version" => "0.1633"},{"date" => "2011-01-19T14:52:48","version" => "0.1634"},{"date" => "2011-01-21T14:04:02","version" => "0.1635"},{"date" => "2011-01-22T13:02:55","version" => "0.1636"},{"date" => "2011-01-24T12:58:40","version" => "0.1637"},{"date" => "2011-01-25T12:32:26","version" => "0.1638"},{"date" => "2011-01-26T09:23:22","version" => "0.1639"},{"date" => "2011-01-26T13:59:10","version" => "0.1640"},{"date" => "2011-01-27T05:19:14","version" => "0.1641"},{"date" => "2011-01-28T12:18:42","version" => "0.1642"},{"date" => "2011-02-09T08:54:11","version" => "0.1643"},{"date" => "2011-02-11T14:07:25","version" => "0.1644"},{"date" => "2011-02-14T15:24:30","version" => "0.1645"},{"date" => "2011-02-18T17:48:52","version" => "0.1646"},{"date" => "2011-02-19T00:30:41","version" => "0.1647"},{"date" => "2011-02-21T16:13:29","version" => "0.1648"},{"date" => "2011-02-22T14:53:08","version" => "0.1649"},{"date" => "2011-02-24T05:45:44","version" => "0.1650"},{"date" => "2011-02-24T14:35:20","version" => "0.1651"},{"date" => "2011-02-25T14:39:56","version" => "0.1652"},{"date" => "2011-02-28T13:18:03","version" => "0.1653"},{"date" => "2011-03-06T14:32:11","version" => "0.1654"},{"date" => "2011-03-08T14:59:08","version" => "0.1655"},{"date" => "2011-03-09T13:44:35","version" => "0.1656"},{"date" => "2011-03-10T15:44:50","version" => "0.1657"},{"date" => "2011-03-11T16:23:11","version" => "0.1658"},{"date" => "2011-03-12T08:20:07","version" => "0.1659"},{"date" => "2011-03-14T11:16:27","version" => "0.1660"},{"date" => "2011-03-15T16:32:52","version" => "0.1661"},{"date" => "2011-03-19T14:40:50","version" => "0.1662"},{"date" => "2011-03-21T03:53:25","version" => "0.1663"},{"date" => "2011-03-24T14:45:52","version" => "0.1664"},{"date" => "2011-03-25T14:25:43","version" => "0.1665"},{"date" => "2011-03-29T17:26:27","version" => "0.1666"},{"date" => "2011-03-30T08:03:39","version" => "0.1667"},{"date" => "2011-03-30T15:04:03","version" => "0.1668"},{"date" => "2011-03-30T15:25:45","version" => "0.1669"},{"date" => "2011-04-01T15:29:33","version" => "0.1670"},{"date" => "2011-04-02T16:31:44","version" => "0.1671"},{"date" => "2011-04-04T13:37:34","version" => "0.1672"},{"date" => "2011-04-05T11:45:54","version" => "0.1673"},{"date" => "2011-04-05T11:59:11","version" => "0.1674"},{"date" => "2011-04-11T13:47:34","version" => "0.1675"},{"date" => "2011-04-11T14:55:38","version" => "0.1676"},{"date" => "2011-04-12T15:17:24","version" => "0.1677"},{"date" => "2011-04-18T13:36:31","version" => "0.1678"},{"date" => "2011-04-19T11:07:27","version" => "0.1679"},{"date" => "2011-04-25T14:05:23","version" => "0.1680"},{"date" => "2011-04-26T14:07:02","version" => "0.1681"},{"date" => "2011-05-23T14:40:41","version" => "0.1682"},{"date" => "2011-06-06T11:52:44","version" => "0.1683"},{"date" => "2011-06-07T13:07:20","version" => "0.1684"},{"date" => "2011-06-08T10:32:35","version" => "0.1685"},{"date" => "2011-06-08T12:24:07","version" => "0.1686"},{"date" => "2011-06-09T13:59:44","version" => "0.1687"},{"date" => "2011-06-10T13:26:20","version" => "0.1688"},{"date" => "2011-06-12T03:22:26","version" => "0.1689"},{"date" => "2011-06-12T12:01:43","version" => "0.1690"},{"date" => "2011-06-13T13:31:21","version" => "0.1691"},{"date" => "2011-06-14T13:27:31","version" => "0.1692"},{"date" => "2011-06-15T08:51:43","version" => "0.1693"},{"date" => "2011-06-17T14:38:23","version" => "0.1694"},{"date" => "2011-06-20T13:08:47","version" => "0.1695"},{"date" => "2011-06-21T13:12:38","version" => "0.1696"},{"date" => "2011-06-24T13:42:00","version" => "0.1697"},{"date" => "2011-06-27T13:23:13","version" => "0.1698"},{"date" => "2011-06-28T14:39:21","version" => "0.1699"},{"date" => "2011-07-01T11:04:37","version" => "0.1700"},{"date" => "2011-07-11T13:19:20","version" => "0.1701"},{"date" => "2011-07-26T14:09:43","version" => "0.1702"},{"date" => "2011-07-28T04:59:20","version" => "0.1703"},{"date" => "2011-07-29T13:45:24","version" => "0.1704"},{"date" => "2011-07-29T14:35:38","version" => "0.1705"},{"date" => "2011-07-30T04:25:21","version" => "0.1706"},{"date" => "2011-07-30T05:16:05","version" => "0.1707"},{"date" => "2011-07-30T14:32:34","version" => "0.1708"},{"date" => "2011-08-01T12:48:52","version" => "0.1709"},{"date" => "2011-08-02T13:30:15","version" => "0.1710"},{"date" => "2011-08-09T14:11:24","version" => "0.1711"},{"date" => "2011-08-10T16:16:52","version" => "0.1712"},{"date" => "2011-08-12T13:45:58","version" => "0.1713"},{"date" => "2011-08-13T13:38:02","version" => "0.1714"},{"date" => "2011-08-14T03:47:28","version" => "0.1715"},{"date" => "2011-08-15T14:00:28","version" => "0.1716"},{"date" => "2011-08-16T04:03:16","version" => "0.1717"},{"date" => "2011-08-20T09:40:46","version" => "0.1718"},{"date" => "2011-08-22T13:43:21","version" => "0.1720"},{"date" => "2011-08-26T14:11:53","version" => "0.1721"},{"date" => "2011-09-02T15:12:10","version" => "0.1722"},{"date" => "2011-09-12T12:24:14","version" => "0.1723"},{"date" => "2011-09-16T15:15:54","version" => "0.1724"},{"date" => "2011-09-27T11:48:33","version" => "0.1725"},{"date" => "2011-09-30T11:21:45","version" => "0.1726"},{"date" => "2011-10-03T10:43:32","version" => "0.1727"},{"date" => "2011-10-05T04:10:35","version" => "0.1728"},{"date" => "2011-10-05T08:12:55","version" => "0.1729"},{"date" => "2011-10-10T11:35:23","version" => "0.1730"},{"date" => "2011-10-11T14:30:46","version" => "0.1731"},{"date" => "2011-10-20T11:56:08","version" => "0.1732"},{"date" => "2011-10-21T22:47:50","version" => "0.1733"},{"date" => "2011-10-22T22:02:37","version" => "0.1734"},{"date" => "2011-10-23T00:11:48","version" => "0.1735"},{"date" => "2011-10-23T13:08:15","version" => "0.1736"},{"date" => "2011-10-24T14:07:44","version" => "0.1737"},{"date" => "2011-10-25T14:31:15","version" => "0.1738"},{"date" => "2011-10-26T01:14:58","version" => "0.1739"},{"date" => "2011-10-27T12:59:00","version" => "0.1740"},{"date" => "2011-10-28T11:49:57","version" => "0.1741"},{"date" => "2011-10-31T15:37:07","version" => "0.1742"},{"date" => "2011-11-01T12:02:38","version" => "0.1743"},{"date" => "2011-11-03T13:38:04","version" => "0.1744"},{"date" => "2011-11-04T14:16:11","version" => "0.1745"},{"date" => "2011-11-07T12:19:53","version" => "0.1746"},{"date" => "2011-11-11T11:59:27","version" => "0.1747"},{"date" => "2011-11-16T00:36:45","version" => "0.20_01"},{"date" => "2011-11-16T08:50:11","version" => "0.2100"},{"date" => "2011-11-21T11:05:36","version" => "0.2101"},{"date" => "2011-11-25T14:34:26","version" => "0.2102"},{"date" => "2011-11-28T10:38:56","version" => "0.2103"},{"date" => "2011-11-29T13:48:49","version" => "0.2104"},{"date" => "2012-01-14T13:39:10","version" => "0.2105"},{"date" => "2012-01-20T15:16:34","version" => "0.2106"},{"date" => "2012-01-25T08:56:44","version" => "0.2107"},{"date" => "2012-01-29T14:30:53","version" => "0.2108"},{"date" => "2012-02-07T13:31:49","version" => "0.2109"},{"date" => "2012-02-10T14:51:17","version" => "0.2110"},{"date" => "2012-02-11T14:45:41","version" => "0.2111"},{"date" => "2012-02-28T14:33:03","version" => "0.22"},{"date" => "2012-03-01T00:07:11","version" => "0.23"},{"date" => "2012-03-02T14:57:03","version" => "0.24"},{"date" => "2012-03-19T11:58:43","version" => "0.25"},{"date" => "2012-07-11T08:20:53","version" => "0.26"},{"date" => "2012-09-17T13:15:26","version" => "0.27"},{"date" => "2013-03-04T11:25:17","version" => "0.28"},{"date" => "2014-02-03T09:21:29","version" => "0.29"},{"date" => "2014-02-04T00:17:32","version" => "0.30"},{"date" => "2015-01-13T01:36:24","version" => "0.31"},{"date" => "2015-01-13T05:24:10","version" => "0.32"},{"date" => "2015-01-13T07:52:20","version" => "0.33"},{"date" => "2015-01-15T02:04:26","version" => "0.34"},{"date" => "2015-05-23T05:44:25","version" => "0.35"},{"date" => "2015-05-25T02:52:16","version" => "0.36"},{"date" => "2016-05-21T07:00:46","version" => "0.37"},{"date" => "2017-03-16T07:48:58","version" => "0.38"},{"date" => "2017-03-29T02:29:03","version" => "0.39"},{"date" => "2017-03-30T01:41:11","version" => "0.40"},{"date" => "2017-11-06T15:17:26","version" => "0.41"},{"date" => "2019-10-15T04:14:26","version" => "0.41_99"},{"date" => "2019-10-19T08:52:17","version" => "0.42"},{"date" => "2020-04-01T05:39:43","version" => "0.43"},{"date" => "2020-08-03T00:46:29","version" => "0.44"},{"date" => "2021-12-16T00:31:02","version" => "0.45"}]},"Dancer" => {"advisories" => [{"affected_versions" => ["<1.3114"],"cves" => ["CVE-2012-5572"],"description" => "CRLF injection vulnerability in the cookie method allows remote attackers to inject arbitrary HTTP headers and conduct HTTP response splitting attacks via a cookie name.\n","distribution" => "Dancer","fixed_versions" => [">=1.3114"],"id" => "CPANSA-Dancer-2014-01","references" => ["https://metacpan.org/changes/distribution/Dancer","https://github.com/PerlDancer/Dancer/commit/46ef9124f3149f697455061499ac7cee40930349"],"reported" => "2014-05-30"},{"affected_versions" => ["<1.3051"],"cves" => ["CVE-2011-1589"],"description" => "Directory traversal vulnerability (Mojolicious report, but Dancer was vulnerable as well).\n","distribution" => "Dancer","fixed_versions" => [">=1.3051"],"id" => "CPANSA-Dancer-2011-01","references" => ["https://metacpan.org/changes/distribution/Dancer","https://github.com/PerlDancer/Dancer/commit/91d0bf6a36705b0971b18f7d38fa2f3df8c7b994"],"reported" => "2011-04-05"}],"main_module" => "Dancer","versions" => [{"date" => "2009-07-27T13:18:07","version" => "20090727.1315"},{"date" => "2009-07-27T14:14:13","version" => "0_0.99"},{"date" => "2009-08-01T13:48:20","version" => "0.9901"},{"date" => "2009-08-04T10:01:54","version" => "0.9902"},{"date" => "2009-08-07T12:29:03","version" => "0.9003"},{"date" => "2009-09-19T15:30:19","version" => "0.9904"},{"date" => "2009-09-23T21:33:51","version" => "0.9905"},{"date" => "2009-11-20T11:14:20","version" => "1.000"},{"date" => "2010-01-06T13:53:28","version" => "1.100"},{"date" => "2010-01-11T09:46:45","version" => "1.110"},{"date" => "2010-01-15T16:03:35","version" => "1.120"},{"date" => "2010-01-15T17:53:08","version" => "1.121"},{"date" => "2010-01-20T07:48:38","version" => "1.122"},{"date" => "2010-01-29T17:29:24","version" => "1.130"},{"date" => "2010-02-09T07:55:18","version" => "1.140"},{"date" => "2010-02-17T15:09:48","version" => "1.150"},{"date" => "2010-03-07T17:50:01","version" => "1.160"},{"date" => "2010-03-24T11:19:00","version" => "1.170"},{"date" => "2010-03-24T13:44:04","version" => "1.171"},{"date" => "2010-03-28T15:09:59","version" => "1.172"},{"date" => "2010-04-01T14:13:30","version" => "1.173"},{"date" => "2010-04-04T11:03:53","version" => "1.173_01"},{"date" => "2010-04-08T13:49:39","version" => "1.174"},{"date" => "2010-04-11T10:49:39","version" => "1.175"},{"date" => "2010-04-19T08:43:22","version" => "1.175_01"},{"date" => "2010-04-22T20:29:56","version" => "1.176"},{"date" => "2010-05-05T12:21:26","version" => "1.178_01"},{"date" => "2010-05-16T10:28:47","version" => "1.1800"},{"date" => "2010-05-19T14:17:57","version" => "1.1801"},{"date" => "2010-05-19T17:32:52","version" => "1.1802"},{"date" => "2010-05-23T20:45:17","version" => "1.1803"},{"date" => "2010-06-18T11:59:20","version" => "1.1804"},{"date" => "2010-06-22T06:41:58","version" => "1.1805"},{"date" => "2010-07-07T06:15:55","version" => "1.1806_01"},{"date" => "2010-08-14T16:37:45","version" => "1.1806_02"},{"date" => "2010-08-23T17:47:12","version" => "1.1807"},{"date" => "2010-08-24T06:23:38","version" => "1.1808"},{"date" => "2010-08-25T05:41:15","version" => "1.1809"},{"date" => "2010-09-01T06:19:20","version" => "1.1810"},{"date" => "2010-09-03T09:23:14","version" => "1.1811"},{"date" => "2010-09-21T12:19:35","version" => "1.1812"},{"date" => "2010-09-24T14:25:44","version" => "1.1901"},{"date" => "2010-10-14T09:25:03","version" => "1.1999_01"},{"date" => "2010-10-28T15:41:17","version" => "1.1999_02"},{"date" => "2010-11-02T14:14:32","version" => "1.1902"},{"date" => "2010-11-02T14:25:04","version" => "1.1902"},{"date" => "2010-11-03T17:07:29","version" => "1.1903"},{"date" => "2010-11-04T11:16:17","version" => "1.1904"},{"date" => "2010-11-11T07:43:21","version" => "1.1999_03"},{"date" => "2010-11-14T08:08:56","version" => "1.1999_04"},{"date" => "2010-11-18T15:54:33","version" => "1.200"},{"date" => "2010-11-18T16:52:47","version" => "1.2000"},{"date" => "2010-11-29T22:05:38","version" => "1.2000_01"},{"date" => "2010-11-30T10:00:23","version" => "1.2000_02"},{"date" => "2010-11-30T19:59:09","version" => "1.2001"},{"date" => "2010-12-02T12:18:12","version" => "1.2001_01"},{"date" => "2010-12-03T20:28:56","version" => "1.2002"},{"date" => "2010-12-07T18:05:50","version" => "1.2002_01"},{"date" => "2010-12-08T21:38:17","version" => "1.2002_02"},{"date" => "2010-12-10T18:28:16","version" => "1.2003"},{"date" => "2010-12-22T17:57:55","version" => "1.3000_01"},{"date" => "2011-01-03T15:17:14","version" => "1.3000_02"},{"date" => "2011-01-27T10:00:22","version" => "1.2004"},{"date" => "2011-01-27T10:09:31","version" => "1.3001"},{"date" => "2011-02-02T15:42:28","version" => "1.3002"},{"date" => "2011-02-05T17:07:15","version" => "1.2005"},{"date" => "2011-02-06T13:12:28","version" => "1.3003"},{"date" => "2011-02-10T20:48:48","version" => "1.3010"},{"date" => "2011-02-12T12:50:18","version" => "1.3010_01"},{"date" => "2011-02-14T15:58:10","version" => "1.3011"},{"date" => "2011-03-01T19:00:52","version" => "1.3012"},{"date" => "2011-03-03T08:41:00","version" => "1.3013"},{"date" => "2011-03-04T12:56:36","version" => "1.3014"},{"date" => "2011-03-10T14:16:24","version" => "1.3014_01"},{"date" => "2011-03-13T13:17:43","version" => "1.3019_01"},{"date" => "2011-03-14T07:44:57","version" => "1.3019_02"},{"date" => "2011-03-21T13:44:17","version" => "1.3020"},{"date" => "2011-04-01T15:22:58","version" => "1.3029_01"},{"date" => "2011-04-08T20:07:26","version" => "1.3029_02"},{"date" => "2011-04-10T08:18:44","version" => "1.3029_03"},{"date" => "2011-04-13T08:26:50","version" => "1.3030"},{"date" => "2011-04-27T14:58:57","version" => "1.3039_01"},{"date" => "2011-05-01T14:55:49","version" => "1.3040"},{"date" => "2011-05-14T15:03:00","version" => "1.3049_01"},{"date" => "2011-05-20T10:57:10","version" => "1.3050"},{"date" => "2011-05-27T12:57:27","version" => "1.3051"},{"date" => "2011-05-27T13:07:51","version" => "1.3059_01"},{"date" => "2011-05-29T14:06:24","version" => "1.3059_02"},{"date" => "2011-06-11T14:02:50","version" => "1.3059_03"},{"date" => "2011-06-12T17:31:55","version" => "1.3059_04"},{"date" => "2011-06-15T10:35:07","version" => "1.3060"},{"date" => "2011-07-07T13:19:45","version" => "1.3069_01"},{"date" => "2011-07-10T16:14:53","version" => "1.3069_02"},{"date" => "2011-07-14T13:47:19","version" => "1.3070"},{"date" => "2011-07-26T16:21:51","version" => "1.3071"},{"date" => "2011-08-17T15:27:53","version" => "1.3079_01"},{"date" => "2011-08-23T09:55:46","version" => "1.3072"},{"date" => "2011-08-28T14:13:40","version" => "1.3079_02"},{"date" => "2011-09-10T15:10:29","version" => "1.3079_03"},{"date" => "2011-10-02T16:07:02","version" => "1.3079_04"},{"date" => "2011-10-18T14:43:22","version" => "1.3079_05"},{"date" => "2011-10-25T21:16:42","version" => "1.3080"},{"date" => "2011-11-27T06:51:43","version" => "1.3089_01"},{"date" => "2011-12-13T14:41:24","version" => "1.3090"},{"date" => "2011-12-17T11:09:48","version" => "1.3091"},{"date" => "2012-01-27T14:38:05","version" => "1.3092"},{"date" => "2012-02-29T14:34:55","version" => "1.3093"},{"date" => "2012-03-31T09:57:40","version" => "1.3094"},{"date" => "2012-04-01T19:22:56","version" => "1.3095"},{"date" => "2012-06-22T20:18:54","version" => "1.3095_01"},{"date" => "2012-07-03T07:27:28","version" => "1.3095_02"},{"date" => "2012-07-05T23:09:20","version" => "1.3096"},{"date" => "2012-07-08T18:36:14","version" => "1.3097"},{"date" => "2012-07-28T14:40:15","version" => "1.3098"},{"date" => "2012-08-11T13:54:49","version" => "1.3099"},{"date" => "2012-08-25T19:42:47","version" => "1.3100"},{"date" => "2012-10-06T13:24:53","version" => "1.3110"},{"date" => "2012-12-24T13:17:58","version" => "1.9999_01"},{"date" => "2012-12-24T13:48:35","version" => "1.9999_02"},{"date" => "2013-01-22T21:38:11","version" => "2.0000_01"},{"date" => "2013-02-22T15:33:14","version" => "2.000001"},{"date" => "2013-02-24T22:51:59","version" => "1.3111"},{"date" => "2013-03-30T16:33:05","version" => "1.3111_01"},{"date" => "2013-04-01T22:31:08","version" => "1.3111_02"},{"date" => "2013-04-11T01:04:37","version" => "1.3112"},{"date" => "2013-05-09T00:36:16","version" => "1.3113"},{"date" => "2013-06-02T16:49:58","version" => "1.3114"},{"date" => "2013-06-09T23:54:16","version" => "1.3115"},{"date" => "2013-07-04T01:35:27","version" => "1.3116"},{"date" => "2013-07-31T22:40:52","version" => "1.3117"},{"date" => "2013-09-01T16:45:13","version" => "1.3118"},{"date" => "2013-10-26T19:42:59","version" => "1.3119"},{"date" => "2013-12-24T16:23:20","version" => "1.3120"},{"date" => "2014-02-02T22:26:53","version" => "1.3121"},{"date" => "2014-04-10T23:16:40","version" => "1.3122"},{"date" => "2014-04-12T15:47:53","version" => "1.3123"},{"date" => "2014-05-10T16:15:17","version" => "1.3124"},{"date" => "2014-07-12T17:19:08","version" => "1.3125"},{"date" => "2014-07-15T02:01:21","version" => "1.3126"},{"date" => "2014-09-09T00:49:19","version" => "1.3127"},{"date" => "2014-09-09T11:47:21","version" => "1.3128"},{"date" => "2014-09-10T00:50:37","version" => "1.3129"},{"date" => "2014-09-16T01:21:25","version" => "1.3130"},{"date" => "2014-10-11T18:59:22","version" => "1.3131_0"},{"date" => "2014-10-13T23:25:36","version" => "1.3131_1"},{"date" => "2014-10-20T23:14:23","version" => "1.3132"},{"date" => "2014-11-26T22:20:35","version" => "1.3133"},{"date" => "2015-02-23T01:33:08","version" => "1.3134"},{"date" => "2015-04-23T01:54:25","version" => "1.3135"},{"date" => "2015-05-24T15:48:19","version" => "1.3136"},{"date" => "2015-06-05T20:05:21","version" => "1.3137"},{"date" => "2015-06-12T20:55:50","version" => "1.3138"},{"date" => "2015-06-25T20:13:45","version" => "1.3139"},{"date" => "2015-07-03T13:56:32","version" => "1.3140"},{"date" => "2015-09-07T15:15:26","version" => "1.3141"},{"date" => "2015-09-15T00:52:23","version" => "1.3142"},{"date" => "2015-10-26T21:15:31","version" => "1.3143"},{"date" => "2015-11-04T12:36:07","version" => "1.3144"},{"date" => "2015-11-06T22:12:42","version" => "1.3200"},{"date" => "2015-11-07T19:27:25","version" => "1.3201"},{"date" => "2015-11-07T21:52:17","version" => "1.3202"},{"date" => "2016-02-15T21:33:45","version" => "1.3300"},{"date" => "2016-02-16T22:42:44","version" => "1.3301"},{"date" => "2018-05-20T19:52:07","version" => "1.3203"},{"date" => "2018-05-23T13:43:34","version" => "1.3204"},{"date" => "2018-06-13T22:02:36","version" => "1.3205"},{"date" => "2018-06-15T22:11:45","version" => "1.3400"},{"date" => "2018-10-01T11:53:31","version" => "1.3401"},{"date" => "2018-10-10T10:44:29","version" => "1.3402"},{"date" => "2018-10-11T22:45:37","version" => "1.3403"},{"date" => "2018-10-12T20:33:54","version" => "1.3500"},{"date" => "2019-03-14T19:27:25","version" => "1.3501"},{"date" => "2019-03-19T14:49:14","version" => "1.3510"},{"date" => "2019-03-29T11:18:31","version" => "1.3511"},{"date" => "2019-03-31T19:16:29","version" => "1.3512"},{"date" => "2020-01-29T21:03:12","version" => "1.3513"},{"date" => "2020-06-29T16:44:22","version" => "1.3514"},{"date" => "2020-10-02T20:51:17","version" => "1.3514_02"},{"date" => "2020-10-06T21:24:49","version" => "1.3514_03"},{"date" => "2022-06-29T22:00:04","version" => "1.3514_04"},{"date" => "2023-01-02T10:57:26","version" => "1.3520"},{"date" => "2023-02-05T23:40:49","version" => "1.3521"},{"date" => "2023-02-08T20:58:09","version" => "1.3521"},{"date" => "2026-01-26T22:30:00","version" => "1.3522"}]},"Dancer2" => {"advisories" => [{"affected_versions" => ["<0.206000"],"cves" => [],"description" => "There is a potential RCE with regards to Storable. We have added session ID validation to the session engine so that session backends based on Storable can reject malformed session IDs that may lead to exploitation of the RCE.\n","distribution" => "Dancer2","fixed_versions" => [">=0.206000"],"id" => "CPANSA-Dancer2-2018-01","references" => ["https://metacpan.org/changes/distribution/Dancer2","http://lists.preshweb.co.uk/pipermail/dancer-users/2018-April/005952.html","https://github.com/PerlDancer/Dancer2/commit/3580f5d0874a9abf5483528f73bda9a7fd9ec7f1"],"reported" => "2018-01-30","severity" => "critical"}],"main_module" => "Dancer2","versions" => [{"date" => "2013-02-22T15:39:46","version" => "0.01"},{"date" => "2013-02-24T11:04:25","version" => "0.02"},{"date" => "2013-03-07T17:30:37","version" => "0.03"},{"date" => "2013-04-22T19:58:02","version" => "0.04"},{"date" => "2013-07-20T16:53:37","version" => "0.05"},{"date" => "2013-07-30T14:29:42","version" => "0.06"},{"date" => "2013-08-03T22:17:54","version" => "0.07"},{"date" => "2013-08-18T12:24:31","version" => "0.08"},{"date" => "2013-09-01T21:19:26","version" => "0.09"},{"date" => "2013-09-28T13:29:35","version" => "0.10"},{"date" => "2013-12-15T13:21:28","version" => "0.11"},{"date" => "2014-04-07T21:05:16","version" => "0.12"},{"date" => "2014-04-13T17:20:22","version" => "0.13"},{"date" => "2014-04-28T21:16:57","version" => "0.140000"},{"date" => "2014-05-01T08:50:43","version" => "0.140001"},{"date" => "2014-06-07T20:35:57","version" => "0.140900_01"},{"date" => "2014-06-08T20:29:28","version" => "0.141000"},{"date" => "2014-06-24T19:18:07","version" => "0.142000"},{"date" => "2014-07-05T19:43:17","version" => "0.143000"},{"date" => "2014-07-23T19:34:51","version" => "0.149000_01"},{"date" => "2014-08-10T11:53:37","version" => "0.149000_02"},{"date" => "2014-08-16T23:38:39","version" => "0.150000"},{"date" => "2014-10-08T19:51:49","version" => "0.151000"},{"date" => "2014-10-14T02:33:06","version" => "0.152000"},{"date" => "2014-10-23T21:48:36","version" => "0.153000"},{"date" => "2014-10-29T21:41:13","version" => "0.153001"},{"date" => "2014-10-30T08:29:15","version" => "0.153002"},{"date" => "2014-11-17T14:41:14","version" => "0.154000"},{"date" => "2014-11-28T00:21:55","version" => "0.155000"},{"date" => "2014-11-28T16:44:27","version" => "0.155001"},{"date" => "2014-12-02T22:02:03","version" => "0.155002"},{"date" => "2014-12-03T21:35:35","version" => "0.155003"},{"date" => "2014-12-04T10:57:08","version" => "0.155004"},{"date" => "2014-12-07T17:07:21","version" => "0.156000"},{"date" => "2014-12-08T22:08:30","version" => "0.156001"},{"date" => "2014-12-14T17:25:53","version" => "0.157000"},{"date" => "2014-12-21T19:42:24","version" => "0.157001"},{"date" => "2015-01-01T17:11:48","version" => "0.158000"},{"date" => "2015-02-24T03:54:24","version" => "0.159000"},{"date" => "2015-02-25T14:33:59","version" => "0.159001"},{"date" => "2015-03-03T18:25:28","version" => "0.159002"},{"date" => "2015-03-23T14:00:19","version" => "0.159003"},{"date" => "2015-04-26T22:15:22","version" => "0.160000"},{"date" => "2015-05-14T18:46:02","version" => "0.160001"},{"date" => "2015-06-04T11:07:02","version" => "0.160002"},{"date" => "2015-06-06T09:11:43","version" => "0.160003"},{"date" => "2015-07-08T13:04:02","version" => "0.161000"},{"date" => "2015-08-28T13:32:02","version" => "0.161000_01"},{"date" => "2015-09-06T11:13:10","version" => "0.162000"},{"date" => "2015-10-13T15:08:16","version" => "0.162000_01"},{"date" => "2015-10-15T11:00:10","version" => "0.163000"},{"date" => "2015-12-16T22:44:32","version" => "0.164000"},{"date" => "2015-12-17T08:23:24","version" => "0.165000"},{"date" => "2016-01-12T18:04:57","version" => "0.166000"},{"date" => "2016-01-22T06:57:11","version" => "0.166001"},{"date" => "2016-04-19T19:52:27","version" => "0.166001_01"},{"date" => "2016-04-29T14:45:41","version" => "0.166001_02"},{"date" => "2016-05-27T11:25:55","version" => "0.166001_03"},{"date" => "2016-05-27T12:57:04","version" => "0.166001_04"},{"date" => "2016-05-31T13:29:37","version" => "0.200000"},{"date" => "2016-06-16T14:00:23","version" => "0.200001"},{"date" => "2016-06-22T14:41:29","version" => "0.200002"},{"date" => "2016-07-05T19:36:46","version" => "0.200003"},{"date" => "2016-07-11T15:21:33","version" => "0.200003"},{"date" => "2016-07-22T04:41:26","version" => "0.200004"},{"date" => "2016-07-22T13:28:45","version" => "0.201000"},{"date" => "2016-08-13T18:53:07","version" => "0.202000"},{"date" => "2016-08-25T03:12:19","version" => "0.203000"},{"date" => "2016-09-04T02:01:29","version" => "0.203001"},{"date" => "2016-10-11T01:59:49","version" => "0.204000"},{"date" => "2016-10-17T13:32:25","version" => "0.204001"},{"date" => "2016-12-21T21:47:24","version" => "0.204002"},{"date" => "2017-01-25T21:23:22","version" => "0.204003"},{"date" => "2017-01-26T17:31:30","version" => "0.204004"},{"date" => "2017-03-10T21:40:43","version" => "0.205000"},{"date" => "2017-07-11T13:04:56","version" => "0.205001"},{"date" => "2017-10-17T21:10:03","version" => "0.205002"},{"date" => "2018-04-09T00:54:25","version" => "0.206000_01"},{"date" => "2018-04-10T01:50:18","version" => "0.206000_02"},{"date" => "2018-04-20T02:12:22","version" => "0.206000"},{"date" => "2018-11-14T22:26:15","version" => "0.207000"},{"date" => "2019-06-19T14:23:06","version" => "0.208000"},{"date" => "2019-08-05T01:12:14","version" => "0.208001"},{"date" => "2019-12-14T21:13:32","version" => "0.208002"},{"date" => "2019-12-24T05:57:09","version" => "0.300000"},{"date" => "2020-04-06T16:18:33","version" => "0.300001"},{"date" => "2020-04-07T15:49:22","version" => "0.300002"},{"date" => "2020-04-09T14:42:55","version" => "0.300003"},{"date" => "2020-05-27T00:54:55","version" => "0.300004"},{"date" => "2021-01-26T20:59:33","version" => "0.300005"},{"date" => "2021-03-15T23:12:49","version" => "0.301000"},{"date" => "2021-03-17T12:56:09","version" => "0.301001"},{"date" => "2021-04-18T19:33:05","version" => "0.301002"},{"date" => "2021-06-03T13:29:26","version" => "0.301003"},{"date" => "2021-06-06T17:32:08","version" => "0.301004"},{"date" => "2022-03-14T02:18:12","version" => "0.400000"},{"date" => "2023-02-05T23:42:54","version" => "0.400001"},{"date" => "2023-10-09T14:11:25","version" => "1.0.0"},{"date" => "2023-12-12T01:29:05","version" => "1.1.0"},{"date" => "2024-07-18T23:49:14","version" => "1.1.1"},{"date" => "2024-11-25T13:36:09","version" => "1.1.2"},{"date" => "2025-09-15T21:50:07","version" => "2.0.0"},{"date" => "2025-10-22T22:14:58","version" => "2.0.1"}]},"Data-Dumper" => {"advisories" => [{"affected_versions" => ["<2.154"],"cves" => ["CVE-2014-4330"],"description" => "Infinite recursion.\n","distribution" => "Data-Dumper","fixed_versions" => [">=2.154"],"id" => "CPANSA-Data-Dumper-2014-01","references" => ["https://metacpan.org/changes/distribution/Data-Dumper"],"reported" => "2014-09-30"}],"main_module" => "Data::Dumper","versions" => [{"date" => "1995-11-19T22:29:08","version" => "1.21"},{"date" => "1995-11-23T05:45:27","version" => "1.22"},{"date" => "1995-12-04T03:12:16","version" => "1.23"},{"date" => "1996-04-09T15:54:26","version" => "2.00"},{"date" => "1996-04-10T04:25:17","version" => "2.01"},{"date" => "1996-04-13T07:14:35","version" => "2.02"},{"date" => "1996-08-26T14:36:59","version" => "2.03"},{"date" => "1996-08-28T20:11:49","version" => "2.04"},{"date" => "1996-12-02T13:42:49","version" => "2.05"},{"date" => "1996-12-02T23:07:56","version" => "2.06"},{"date" => "1996-12-07T17:28:27","version" => "2.07"},{"date" => "1997-12-07T21:27:09","version" => "2.08"},{"date" => "1998-01-15T20:36:46","version" => "2.081"},{"date" => "1998-03-06T21:08:49","version" => "2.081"},{"date" => "1998-07-17T05:23:08","version" => "2.09"},{"date" => "1998-07-21T12:08:19","version" => "2.09"},{"date" => "1998-10-31T12:10:30","version" => "2.10"},{"date" => "1999-05-01T02:01:03","version" => "2.101"},{"date" => "1999-06-02T01:30:55","version" => "2.101"},{"date" => "2003-07-20T16:59:48","version" => "2.12_01"},{"date" => "2003-07-31T19:12:44","version" => "2.12_02"},{"date" => "2003-08-25T11:49:41","version" => "2.121"},{"date" => "2009-06-06T14:45:36","version" => "2.121_20"},{"date" => "2009-06-09T15:49:12","version" => "2.122"},{"date" => "2009-06-11T08:07:01","version" => "2.123"},{"date" => "2009-06-13T15:22:32","version" => "2.124"},{"date" => "2009-08-08T10:33:01","version" => "2.125"},{"date" => "2010-04-15T19:55:01","version" => "2.126"},{"date" => "2010-09-06T14:28:10","version" => "2.126_01"},{"date" => "2010-09-10T07:08:41","version" => "2.127"},{"date" => "2010-09-10T07:11:52","version" => "2.128"},{"date" => "2011-05-20T15:53:12","version" => "2.130_03"},{"date" => "2011-05-27T14:19:03","version" => "2.131"},{"date" => "2011-12-19T08:23:05","version" => "2.135_01"},{"date" => "2011-12-29T17:09:49","version" => "2.135_02"},{"date" => "2012-08-07T06:59:51","version" => "2.135_07"},{"date" => "2012-10-04T07:35:07","version" => "2.136"},{"date" => "2012-12-12T06:30:48","version" => "2.139"},{"date" => "2013-02-26T06:57:29","version" => "2.143"},{"date" => "2013-03-15T09:46:49","version" => "2.145"},{"date" => "2014-03-07T09:28:44","version" => "2.151"},{"date" => "2014-09-18T15:47:37","version" => "2.154"},{"date" => "2016-07-03T19:17:57","version" => "2.160"},{"date" => "2016-07-11T20:13:06","version" => "2.161"},{"date" => "2017-07-31T15:31:28","version" => "2.167_01"},{"date" => "2017-08-04T08:05:22","version" => "2.167_02"},{"date" => "2018-09-19T14:41:58","version" => "2.172"},{"date" => "2018-11-10T10:10:30","version" => "2.173"},{"date" => "2021-05-14T12:47:34","version" => "2.179_50"},{"date" => "2021-05-17T05:53:02","version" => "2.180"},{"date" => "2021-05-22T09:51:29","version" => "2.180_50"},{"date" => "2021-05-23T14:14:12","version" => "2.180_51"},{"date" => "2021-05-24T08:03:55","version" => "2.180_52"},{"date" => "2021-05-25T05:20:34","version" => "2.180_53"},{"date" => "2021-05-26T06:46:41","version" => "2.181"},{"date" => "2021-06-29T10:42:11","version" => "2.181_50"},{"date" => "2021-06-30T09:36:34","version" => "2.182"},{"date" => "2021-07-01T07:05:45","version" => "2.182_50"},{"date" => "2021-07-03T13:07:49","version" => "2.182_51"},{"date" => "2021-07-05T07:07:44","version" => "2.183"},{"date" => "2001-04-08T00:00:00","dual_lived" => 1,"perl_release" => "5.006001","version" => "2.102"},{"date" => "2002-03-05T00:00:00","dual_lived" => 1,"perl_release" => "5.007003","version" => "2.12"},{"date" => "2004-11-27T00:00:00","dual_lived" => 1,"perl_release" => "5.008006","version" => "2.121_02"},{"date" => "2005-05-30T00:00:00","dual_lived" => 1,"perl_release" => "5.008007","version" => "2.121_04"},{"date" => "2006-01-31T00:00:00","dual_lived" => 1,"perl_release" => "5.008008","version" => "2.121_08"},{"date" => "2008-12-14T00:00:00","dual_lived" => 1,"perl_release" => "5.008009","version" => "2.121_17"},{"date" => "2007-07-07T00:00:00","dual_lived" => 1,"perl_release" => "5.009005","version" => "2.121_13"},{"date" => "2007-12-18T00:00:00","dual_lived" => 1,"perl_release" => "5.010000","version" => "2.121_14"},{"date" => "2010-10-20T00:00:00","dual_lived" => 1,"perl_release" => "5.013006","version" => "2.129"},{"date" => "2010-11-20T00:00:00","dual_lived" => 1,"perl_release" => "5.013007","version" => "2.130_01"},{"date" => "2010-12-20T00:00:00","dual_lived" => 1,"perl_release" => "5.013008","version" => "2.130_02"},{"date" => "2011-08-20T00:00:00","dual_lived" => 1,"perl_release" => "5.015002","version" => "2.132"},{"date" => "2011-09-20T00:00:00","dual_lived" => 1,"perl_release" => "5.015003","version" => "2.134"},{"date" => "2012-01-20T00:00:00","dual_lived" => 1,"perl_release" => "5.015007","version" => "2.135_03"},{"date" => "2012-02-20T00:00:00","dual_lived" => 1,"perl_release" => "5.015008","version" => "2.135_05"},{"date" => "2012-03-20T00:00:00","dual_lived" => 1,"perl_release" => "5.015009","version" => "2.135_06"},{"date" => "2012-11-20T00:00:00","dual_lived" => 1,"perl_release" => "5.017006","version" => "2.137"},{"date" => "2013-01-20T00:00:00","dual_lived" => 1,"perl_release" => "5.017008","version" => "2.141"},{"date" => "2013-02-20T00:00:00","dual_lived" => 1,"perl_release" => "5.017009","version" => "2.142"},{"date" => "2013-06-21T00:00:00","dual_lived" => 1,"perl_release" => "5.019001","version" => "2.146"},{"date" => "2013-07-22T00:00:00","dual_lived" => 1,"perl_release" => "5.019002","version" => "2.147"},{"date" => "2013-08-20T00:00:00","dual_lived" => 1,"perl_release" => "5.019003","version" => "2.148"},{"date" => "2013-09-20T00:00:00","dual_lived" => 1,"perl_release" => "5.019004","version" => "2.149"},{"date" => "2013-12-20T00:00:00","dual_lived" => 1,"perl_release" => "5.019007","version" => "2.150"},{"date" => "2015-02-14T00:00:00","dual_lived" => 1,"perl_release" => "5.020002","version" => "2.151_01"},{"date" => "2014-06-20T00:00:00","dual_lived" => 1,"perl_release" => "5.021001","version" => "2.152"},{"date" => "2014-12-20T00:00:00","dual_lived" => 1,"perl_release" => "5.021007","version" => "2.155"},{"date" => "2015-01-20T00:00:00","dual_lived" => 1,"perl_release" => "5.021008","version" => "2.156"},{"date" => "2015-02-21T00:00:00","dual_lived" => 1,"perl_release" => "5.021009","version" => "2.157"},{"date" => "2015-03-20T00:00:00","dual_lived" => 1,"perl_release" => "5.02101","version" => "2.158"},{"date" => "2015-12-21T00:00:00","dual_lived" => 1,"perl_release" => "5.023006","version" => "2.159"},{"date" => "2016-10-20T00:00:00","dual_lived" => 1,"perl_release" => "5.025006","version" => "2.162"},{"date" => "2016-11-20T00:00:00","dual_lived" => 1,"perl_release" => "5.025007","version" => "2.165"},{"date" => "2016-12-20T00:00:00","dual_lived" => 1,"perl_release" => "5.025008","version" => "2.166"},{"date" => "2017-01-20T00:00:00","dual_lived" => 1,"perl_release" => "5.025009","version" => "2.167"},{"date" => "2017-12-20T00:00:00","dual_lived" => 1,"perl_release" => "5.027007","version" => "2.169"},{"date" => "2018-01-20T00:00:00","dual_lived" => 1,"perl_release" => "5.027008","version" => "2.170"},{"date" => "2018-08-20T00:00:00","dual_lived" => 1,"perl_release" => "5.029002","version" => "2.171"},{"date" => "2019-04-20T00:00:00","dual_lived" => 1,"perl_release" => "5.02901","version" => "2.174"},{"date" => "2021-01-23T00:00:00","dual_lived" => 1,"perl_release" => "5.032001","version" => "2.174_01"},{"date" => "2020-08-20T00:00:00","dual_lived" => 1,"perl_release" => "5.033001","version" => "2.175"},{"date" => "2020-10-20T00:00:00","dual_lived" => 1,"perl_release" => "5.033003","version" => "2.176"},{"date" => "2021-03-20T00:00:00","dual_lived" => 1,"perl_release" => "5.033008","version" => "2.177"},{"date" => "2021-04-20T00:00:00","dual_lived" => 1,"perl_release" => "5.033009","version" => "2.178"},{"date" => "2021-05-20T00:00:00","dual_lived" => 1,"perl_release" => "5.034","version" => "2.179"},{"date" => "2022-01-20T00:00:00","dual_lived" => 1,"perl_release" => "5.035008","version" => "2.184"},{"date" => "2022-06-20T00:00:00","dual_lived" => 1,"perl_release" => "5.037001","version" => "2.185"},{"date" => "2022-07-20T00:00:00","dual_lived" => 1,"perl_release" => "5.037002","version" => "2.186"},{"date" => "2022-11-20T00:00:00","dual_lived" => 1,"perl_release" => "5.037006","version" => "2.187"},{"date" => "2023-03-20T00:00:00","dual_lived" => 1,"perl_release" => "5.03701","version" => "2.188"},{"date" => "2023-09-20T00:00:00","dual_lived" => 1,"perl_release" => "5.039003","version" => "2.189"},{"date" => "2024-07-02T00:00:00","dual_lived" => 1,"perl_release" => "5.041001","version" => "2.190"},{"date" => "2025-01-20T00:00:00","dual_lived" => 1,"perl_release" => "5.041008","version" => "2.191"},{"date" => "2025-04-20T00:00:00","dual_lived" => 1,"perl_release" => "5.041011","version" => "2.192"}]},"Data-Entropy" => {"advisories" => [{"affected_versions" => ["<=0.007"],"cves" => ["CVE-2025-1860"],"description" => "Data::Entropy for Perl 0.007 and earlier use the rand() function as the default source of entropy, which is not\x{a0}cryptographically secure,\x{a0}for cryptographic functions.","distribution" => "Data-Entropy","fixed_versions" => [">0.007"],"id" => "CPANSA-Data-Entropy-2025-1860","references" => ["https://metacpan.org/release/ZEFRAM/Data-Entropy-0.007/source/lib/Data/Entropy.pm#L80","https://perldoc.perl.org/functions/rand","https://lists.debian.org/debian-lts-announce/2025/03/msg00026.html"],"reported" => "2025-03-28","severity" => undef}],"main_module" => "Data::Entropy","versions" => [{"date" => "2006-07-19T01:09:30","version" => "0.000"},{"date" => "2006-08-03T20:27:12","version" => "0.001"},{"date" => "2006-08-05T09:15:08","version" => "0.002"},{"date" => "2007-01-21T00:51:31","version" => "0.003"},{"date" => "2007-09-03T21:25:09","version" => "0.004"},{"date" => "2009-03-03T20:31:03","version" => "0.005"},{"date" => "2009-11-21T14:01:52","version" => "0.006"},{"date" => "2011-04-27T20:03:17","version" => "0.007"},{"date" => "2025-03-27T19:11:37","version" => "0.008"}]},"Data-FormValidator" => {"advisories" => [{"affected_versions" => ["<=4.66"],"cves" => ["CVE-2011-2201"],"description" => "The Data::FormValidator module 4.66 and earlier for Perl, when untaint_all_constraints is enabled, does not properly preserve the taint attribute of data, which might allow remote attackers to bypass the taint protection mechanism via form input.\n","distribution" => "Data-FormValidator","fixed_versions" => [">4.66"],"id" => "CPANSA-Data-FormValidator-2011-2201","references" => ["http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=629511","http://www.openwall.com/lists/oss-security/2011/06/13/13","https://bugzilla.redhat.com/show_bug.cgi?id=712694","http://www.openwall.com/lists/oss-security/2011/06/12/3","http://www.securityfocus.com/bid/48167","https://rt.cpan.org/Public/Bug/Display.html?id=61792","http://lists.fedoraproject.org/pipermail/package-announce/2011-September/065416.html","http://www.openwall.com/lists/oss-security/2011/06/13/5"],"reported" => "2011-09-14","severity" => undef}],"main_module" => "Data::FormValidator","versions" => [{"date" => "2001-06-19T21:43:01","version" => "1.3"},{"date" => "2001-06-22T16:36:08","version" => "1.4"},{"date" => "2001-06-25T17:02:43","version" => "1.4"},{"date" => "2001-06-28T15:13:01","version" => "1.5"},{"date" => "2001-07-18T14:23:17","version" => "v1.5.1"},{"date" => "2001-09-23T22:42:22","version" => "1.6"},{"date" => "2001-11-03T18:16:00","version" => "1.7"},{"date" => "2002-02-14T22:45:46","version" => "1.8"},{"date" => "2002-02-18T02:20:12","version" => "1.9"},{"date" => "2002-04-21T13:42:36","version" => "1.10"},{"date" => "2002-06-29T21:04:14","version" => "1.11"},{"date" => "2002-10-07T02:06:39","version" => "1.91"},{"date" => "2002-12-23T23:36:37","version" => "1.92"},{"date" => "2003-03-08T13:10:33","version" => "1.93"},{"date" => "2003-03-23T03:01:57","version" => "2.00"},{"date" => "2003-04-02T15:18:15","version" => "2.01"},{"date" => "2003-04-09T15:54:50","version" => "2.02"},{"date" => "2003-04-10T16:12:40","version" => "2.03"},{"date" => "2003-04-12T02:58:35","version" => "2.04"},{"date" => "2003-04-20T22:23:44","version" => "2.10"},{"date" => "2003-04-24T02:51:03","version" => "2.10"},{"date" => "2003-05-11T21:29:55","version" => "3.00"},{"date" => "2003-05-16T04:06:05","version" => "3.01"},{"date" => "2003-05-26T23:18:18","version" => "3.1"},{"date" => "2003-05-27T19:41:01","version" => "3.11"},{"date" => "2003-06-23T01:27:03","version" => "3.12"},{"date" => "2003-11-02T21:19:10","version" => "3.13"},{"date" => "2003-11-03T17:59:41","version" => "3.14"},{"date" => "2003-11-30T20:36:41","version" => "3.15"},{"date" => "2004-01-04T01:37:01","version" => "3.49_1"},{"date" => "2004-01-12T22:04:27","version" => "3.50"},{"date" => "2004-02-27T04:19:47","version" => "3.51"},{"date" => "2004-03-21T17:42:11","version" => "3.52"},{"date" => "2004-03-23T02:33:53","version" => "3.53"},{"date" => "2004-03-24T14:55:49","version" => "3.54"},{"date" => "2004-04-17T02:30:02","version" => "3.56"},{"date" => "2004-04-22T02:26:41","version" => "3.57"},{"date" => "2004-05-05T21:55:00","version" => "3.58"},{"date" => "2004-07-02T17:48:51","version" => "3.59"},{"date" => "2004-09-28T02:25:35","version" => "3.61"},{"date" => "2004-10-09T04:00:51","version" => "3.62"},{"date" => "2004-11-17T22:27:13","version" => "3.63"},{"date" => "2005-05-20T01:25:45","version" => "4.00_01"},{"date" => "2005-07-03T19:37:11","version" => "4.00_02"},{"date" => "2005-07-20T02:07:36","version" => "3.70"},{"date" => "2005-07-31T17:36:02","version" => "3.71"},{"date" => "2005-08-14T16:09:26","version" => "4.00"},{"date" => "2005-08-20T18:20:14","version" => "4.01"},{"date" => "2005-09-01T02:31:29","version" => "4.02"},{"date" => "2005-12-23T01:00:49","version" => "4.10"},{"date" => "2006-01-03T23:49:53","version" => "4.11"},{"date" => "2006-01-06T02:14:25","version" => "4.12"},{"date" => "2006-02-10T02:48:33","version" => "4.13"},{"date" => "2006-02-17T18:48:14","version" => "4.14"},{"date" => "2006-06-13T01:12:23","version" => "4.20"},{"date" => "2006-07-01T15:42:37","version" => "4.21_01"},{"date" => "2006-07-11T01:45:51","version" => "4.30"},{"date" => "2006-08-21T23:43:58","version" => "4.40"},{"date" => "2006-10-03T18:16:57","version" => "4.49_1"},{"date" => "2006-12-05T02:41:19","version" => "4.50"},{"date" => "2007-07-14T03:36:00","version" => "4.51"},{"date" => "2007-10-19T19:41:46","version" => "4.52"},{"date" => "2007-10-20T20:02:19","version" => "4.50"},{"date" => "2007-10-21T13:30:20","version" => "4.54"},{"date" => "2007-10-21T15:50:42","version" => "4.55"},{"date" => "2007-10-31T16:49:55","version" => "4.56"},{"date" => "2007-11-02T02:55:19","version" => "4.57"},{"date" => "2008-06-16T18:28:54","version" => "4.60"},{"date" => "2008-06-16T18:46:47","version" => "4.61"},{"date" => "2009-01-03T17:14:18","version" => "4.62"},{"date" => "2009-01-03T17:49:48","version" => "4.63"},{"date" => "2009-12-31T03:22:00","version" => "4.65"},{"date" => "2010-02-24T14:33:48","version" => "4.66"},{"date" => "2011-11-12T02:18:45","version" => "4.67"},{"date" => "2011-11-12T03:11:55","version" => "4.70"},{"date" => "2012-10-02T20:40:09","version" => "4.71"},{"date" => "2012-11-01T15:14:05","version" => "4.80"},{"date" => "2013-07-19T12:31:06","version" => "4.81"},{"date" => "2017-02-23T15:15:37","version" => "4.82"},{"date" => "2017-02-23T16:39:53","version" => "4.83"},{"date" => "2017-02-25T15:12:25","version" => "4.84"},{"date" => "2017-02-25T20:34:08","version" => "4.85"},{"date" => "2017-03-26T19:39:49","version" => "4.86"},{"date" => "2017-08-28T12:41:42","version" => "4.88"}]},"Data-UUID" => {"advisories" => [{"affected_versions" => [">1.219,<1.227"],"cves" => ["CVE-2013-4184"],"description" => "Perl module Data::UUID from CPAN version 1.219 vulnerable to symlink attacks\n","distribution" => "Data-UUID","fixed_versions" => [">=1.227"],"id" => "CPANSA-Data-UUID-2013-4184","references" => ["https://bugzilla.suse.com/show_bug.cgi?id=CVE-2013-4184","http://www.openwall.com/lists/oss-security/2013/07/31/4","http://www.securityfocus.com/bid/61534","https://exchange.xforce.ibmcloud.com/vulnerabilities/86103","https://security-tracker.debian.org/tracker/CVE-2013-4184","https://access.redhat.com/security/cve/cve-2013-4184","https://bugs.gentoo.org/show_bug.cgi?id=CVE-2013-4184"],"reported" => "2019-12-10","severity" => "medium"}],"main_module" => "Data::UUID","versions" => [{"date" => "2001-10-30T17:35:43","version" => "0.01"},{"date" => "2001-10-31T17:16:07","version" => "0.02"},{"date" => "2001-11-05T17:56:17","version" => "0.03"},{"date" => "2001-12-12T21:23:28","version" => "0.04"},{"date" => "2002-02-13T19:57:54","version" => "0.05"},{"date" => "2002-03-11T15:42:56","version" => "0.06"},{"date" => "2002-06-12T21:38:06","version" => "0.07"},{"date" => "2002-11-29T17:19:21","version" => "0.08"},{"date" => "2003-02-26T21:56:46","version" => "0.09"},{"date" => "2003-07-17T21:55:38","version" => "0.10"},{"date" => "2003-08-27T20:17:10","version" => "0.11"},{"date" => "2006-02-24T00:44:57","version" => "0.12_01"},{"date" => "2006-02-25T20:45:33","version" => "0.13"},{"date" => "2006-03-18T13:42:09","version" => "0.14"},{"date" => "2006-09-06T02:19:07","version" => "0.141"},{"date" => "2006-09-06T02:46:59","version" => "0.142"},{"date" => "2006-09-18T02:16:47","version" => "0.143"},{"date" => "2006-09-19T22:29:33","version" => "0.145"},{"date" => "2006-11-15T01:22:33","version" => "0.146"},{"date" => "2006-11-16T14:16:50","version" => "0.147_01"},{"date" => "2006-11-16T15:25:08","version" => "0.148"},{"date" => "2007-03-08T16:05:15","version" => "1.148"},{"date" => "2008-11-01T16:36:57","version" => "1.149"},{"date" => "2008-11-02T03:21:27","version" => "1.200_01"},{"date" => "2008-11-11T21:40:52","version" => "1.200_02"},{"date" => "2009-04-18T18:12:28","version" => "1.201"},{"date" => "2009-06-15T22:47:18","version" => "1.202"},{"date" => "2009-11-03T21:49:20","version" => "1.203"},{"date" => "2010-05-07T01:57:28","version" => "1.210"},{"date" => "2010-05-07T12:00:52","version" => "1.211"},{"date" => "2010-05-07T22:59:24","version" => "1.212"},{"date" => "2010-05-09T19:29:59","version" => "1.213"},{"date" => "2010-05-15T01:06:55","version" => "1.214"},{"date" => "2010-05-25T02:47:15","version" => "1.215"},{"date" => "2010-09-04T18:14:56","version" => "1.216"},{"date" => "2010-09-14T01:48:04","version" => "1.217"},{"date" => "2012-08-01T03:25:46","version" => "1.218"},{"date" => "2013-07-07T03:00:13","version" => "1.219"},{"date" => "2014-12-16T00:07:05","version" => "1.220"},{"date" => "2015-08-10T12:37:32","version" => "1.221"},{"date" => "2018-04-29T22:11:17","version" => "1.222"},{"date" => "2019-02-25T22:28:34","version" => "1.223"},{"date" => "2019-03-02T14:20:33","version" => "1.224"},{"date" => "2020-04-12T18:42:29","version" => "1.225"},{"date" => "2020-04-12T18:43:57","version" => "1.226"},{"date" => "2024-03-18T19:39:28","version" => "1.227"}]},"Data-Validate-IP" => {"advisories" => [{"affected_versions" => ["<=0.29"],"cves" => ["CVE-2021-29662"],"description" => "The Data::Validate::IP module through 0.29 for Perl does not properly consider extraneous zero characters at the beginning of an IP address string, which (in some situations) allows attackers to bypass access control that is based on IP addresses.\n","distribution" => "Data-Validate-IP","fixed_versions" => [">0.29"],"id" => "CPANSA-Data-Validate-IP-2021-01","references" => ["https://security.netapp.com/advisory/ntap-20210604-0002/","https://blog.urth.org/2021/03/29/security-issues-in-perl-ip-address-distros/","https://github.com/houseabsolute/Data-Validate-IP","https://github.com/houseabsolute/Data-Validate-IP/commit/3bba13c819d616514a75e089badd75002fd4f14e","https://github.com/sickcodes/security/blob/master/advisories/SICK-2021-018.md","https://sick.codes/sick-2021-018/"],"reported" => "2021-03-31"}],"main_module" => "Data::Validate::IP","versions" => [{"date" => "2005-03-04T16:46:50","version" => "0.02"},{"date" => "2005-03-04T20:06:14","version" => "0.03"},{"date" => "2005-04-28T15:11:20","version" => "0.04"},{"date" => "2007-03-06T19:45:16","version" => "0.05"},{"date" => "2007-05-16T16:08:59","version" => "0.06"},{"date" => "2007-05-18T02:42:07","version" => "0.07"},{"date" => "2007-12-06T18:48:53","version" => "0.08"},{"date" => "2009-06-04T17:52:28","version" => "0.10"},{"date" => "2010-03-01T19:40:48","version" => "0.11"},{"date" => "2010-12-29T21:23:08","version" => "0.12"},{"date" => "2011-01-06T14:25:53","version" => "0.13"},{"date" => "2011-01-06T14:45:14","version" => "0.14"},{"date" => "2013-02-05T00:19:11","version" => "0.15"},{"date" => "2013-02-06T15:18:38","version" => "0.16"},{"date" => "2013-02-19T15:58:21","version" => "0.17"},{"date" => "2013-02-20T00:31:32","version" => "0.18"},{"date" => "2013-03-13T15:48:07","version" => "0.19"},{"date" => "2013-07-13T19:21:15","version" => "0.20"},{"date" => "2013-12-05T21:16:41","version" => "0.21"},{"date" => "2013-12-05T22:47:38","version" => "0.22"},{"date" => "2014-03-09T16:00:20","version" => "0.23"},{"date" => "2014-08-28T16:00:00","version" => "0.24"},{"date" => "2016-02-02T16:17:46","version" => "0.25"},{"date" => "2016-05-31T17:31:50","version" => "0.26"},{"date" => "2016-11-17T18:05:57","version" => "0.27"},{"date" => "2021-03-29T17:01:17","version" => "0.28"},{"date" => "2021-03-29T17:07:58","version" => "0.29"},{"date" => "2021-03-29T21:50:39","version" => "0.30"},{"date" => "2022-11-28T18:19:55","version" => "0.31"}]},"Devel-PPPort" => {"advisories" => [{"affected_versions" => ["<3.41"],"cves" => [],"description" => "Function croak() takes first parameter printf-like format. Arbitrary string from the variable \$\@ can cause perl crash when contains one or more '%'.\n","distribution" => "Devel-PPPort","fixed_versions" => [">=3.41"],"id" => "CPANSA-Devel-PPPort-2017-01","references" => ["https://metacpan.org/dist/Devel-PPPort/changes","https://github.com/Dual-Life/Devel-PPPort/pull/47"],"reported" => "2017-02-14","severity" => undef}],"main_module" => "Devel::PPPort","versions" => [{"date" => "1999-03-01T05:05:50","version" => "1.0004"},{"date" => "1999-03-08T02:57:01","version" => "1.0005"},{"date" => "1999-03-24T16:17:40","version" => "1.0006"},{"date" => "1999-03-29T16:29:09","version" => "1.0007"},{"date" => "2004-08-07T14:09:53","version" => "2.99_01"},{"date" => "2004-08-08T17:24:46","version" => "2.99_02"},{"date" => "2004-08-09T20:40:45","version" => "2.99_03"},{"date" => "2004-08-10T21:37:23","version" => "2.99_04"},{"date" => "2004-08-10T21:52:34","version" => "2.99_05"},{"date" => "2004-08-11T21:14:33","version" => "2.99_06"},{"date" => "2004-08-13T11:05:16","version" => "2.99_07"},{"date" => "2004-08-16T09:37:21","version" => "3.00"},{"date" => "2004-08-17T21:45:21","version" => "3.00_01"},{"date" => "2004-08-19T11:23:25","version" => "3.00_02"},{"date" => "2004-08-20T13:31:59","version" => "3.00_03"},{"date" => "2004-08-23T05:52:31","version" => "3.01"},{"date" => "2004-09-08T19:25:27","version" => "3.02"},{"date" => "2004-09-08T20:39:17","version" => "3.03"},{"date" => "2004-12-29T14:03:53","version" => "3.04"},{"date" => "2005-01-31T18:29:11","version" => "3.05"},{"date" => "2005-02-02T21:53:39","version" => "3.06"},{"date" => "2005-06-25T16:59:34","version" => "3.06_01"},{"date" => "2005-10-18T19:59:34","version" => "3.06_02"},{"date" => "2005-10-18T21:43:58","version" => "3.06_03"},{"date" => "2005-10-30T11:10:01","version" => "3.06_04"},{"date" => "2006-01-16T18:10:31","version" => "3.07"},{"date" => "2006-01-19T18:40:04","version" => "3.08"},{"date" => "2006-05-20T11:11:00","version" => "3.08_01"},{"date" => "2006-05-22T11:17:01","version" => "3.08_02"},{"date" => "2006-05-25T15:33:51","version" => "3.08_03"},{"date" => "2006-05-29T17:44:18","version" => "3.08_04"},{"date" => "2006-06-23T19:00:30","version" => "3.08_05"},{"date" => "2006-06-25T08:09:51","version" => "3.08_06"},{"date" => "2006-07-03T21:36:39","version" => "3.08_07"},{"date" => "2006-07-08T16:22:49","version" => "3.09"},{"date" => "2006-07-21T17:25:17","version" => "3.09_01"},{"date" => "2006-07-25T18:45:27","version" => "3.09_02"},{"date" => "2006-08-14T19:31:33","version" => "3.10"},{"date" => "2006-12-02T16:26:47","version" => "3.10_01"},{"date" => "2006-12-02T17:23:57","version" => "3.10_02"},{"date" => "2007-02-14T13:10:03","version" => "3.11"},{"date" => "2007-03-23T17:09:16","version" => "3.11_01"},{"date" => "2007-08-12T23:08:25","version" => "3.11_02"},{"date" => "2007-08-14T19:14:20","version" => "3.11_03"},{"date" => "2007-08-20T16:31:23","version" => "3.11_04"},{"date" => "2007-08-20T17:29:16","version" => "3.11_05"},{"date" => "2007-09-11T21:41:31","version" => "3.11_06"},{"date" => "2007-09-22T08:00:55","version" => "3.12"},{"date" => "2007-10-04T10:33:11","version" => "3.13"},{"date" => "2008-01-04T14:09:17","version" => "3.13_01"},{"date" => "2008-04-13T13:11:47","version" => "3.13_02"},{"date" => "2008-05-13T19:07:49","version" => "3.13_03"},{"date" => "2008-06-01T12:08:17","version" => "3.14"},{"date" => "2008-07-11T20:42:44","version" => "3.14_01"},{"date" => "2008-10-12T19:49:45","version" => "3.14_02"},{"date" => "2008-10-21T21:20:59","version" => "3.14_03"},{"date" => "2008-10-30T18:55:01","version" => "3.14_04"},{"date" => "2008-10-31T07:20:25","version" => "3.14_05"},{"date" => "2009-01-18T13:49:22","version" => "3.15"},{"date" => "2009-01-23T17:33:31","version" => "3.16"},{"date" => "2009-03-15T15:45:38","version" => "3.17"},{"date" => "2009-06-12T11:05:52","version" => "3.18"},{"date" => "2009-06-12T11:25:31","version" => "3.18_01"},{"date" => "2009-06-14T09:59:59","version" => "3.19"},{"date" => "2010-02-20T18:48:07","version" => "3.19_01"},{"date" => "2010-03-07T12:51:00","version" => "3.19_02"},{"date" => "2011-04-13T07:49:49","version" => "3.19_03"},{"date" => "2011-09-10T19:32:27","version" => "3.20"},{"date" => "2013-08-17T13:27:59","version" => "3.21"},{"date" => "2014-03-20T02:17:15","version" => "3.22"},{"date" => "2014-04-13T00:04:17","version" => "3.23"},{"date" => "2014-05-09T00:15:50","version" => "3.24"},{"date" => "2014-12-02T13:08:04","version" => "3.25"},{"date" => "2015-01-08T02:42:03","version" => "3.26"},{"date" => "2015-01-13T15:02:40","version" => "3.27"},{"date" => "2015-01-16T12:33:46","version" => "3.28"},{"date" => "2015-03-05T13:22:18","version" => "3.29"},{"date" => "2015-03-05T20:29:10","version" => "3.30"},{"date" => "2015-03-12T14:27:15","version" => "3.31"},{"date" => "2015-09-30T16:31:45","version" => "3.32"},{"date" => "2016-05-06T11:01:12","version" => "3.32_01"},{"date" => "2016-05-24T13:59:25","version" => "3.32_02"},{"date" => "2016-06-03T13:47:32","version" => "3.33"},{"date" => "2016-06-12T23:09:20","version" => "3.34"},{"date" => "2016-06-17T18:22:04","version" => "3.35"},{"date" => "2017-05-14T08:53:44","version" => "3.36"},{"date" => "2018-04-21T12:18:05","version" => "3.41"},{"date" => "2018-04-21T14:45:37","version" => "3.42"},{"date" => "2018-09-19T14:47:44","version" => "3.43"},{"date" => "2018-10-12T17:16:24","version" => "3.43_04"},{"date" => "2019-02-20T23:05:14","version" => "3.44"},{"date" => "2019-03-19T20:55:38","version" => "3.45"},{"date" => "2019-04-26T19:50:59","version" => "3.46"},{"date" => "2019-04-28T05:30:49","version" => "3.47"},{"date" => "2019-04-28T21:43:34","version" => "3.48"},{"date" => "2019-04-28T21:57:51","version" => "3.48"},{"date" => "2019-04-29T17:48:49","version" => "3.49"},{"date" => "2019-04-30T19:05:34","version" => "3.50"},{"date" => "2019-04-30T23:09:43","version" => "3.51"},{"date" => "2019-05-14T17:18:17","version" => "3.52"},{"date" => "2019-06-09T16:13:03","version" => "3.52_04"},{"date" => "2019-06-11T07:57:30","version" => "3.53_04"},{"date" => "2019-09-28T00:25:55","version" => "3.53"},{"date" => "2019-09-28T00:35:17","version" => "3.54"},{"date" => "2019-11-07T21:15:22","version" => "3.55"},{"date" => "2019-11-25T17:04:32","version" => "3.56"},{"date" => "2020-01-31T20:46:51","version" => "3.57"},{"date" => "2020-02-10T22:15:17","version" => "3.57_01"},{"date" => "2020-03-04T21:32:50","version" => "3.57_02"},{"date" => "2020-03-09T20:42:29","version" => "3.58"},{"date" => "2020-08-06T22:31:34","version" => "3.58_01"},{"date" => "2020-08-10T16:51:52","version" => "3.59"},{"date" => "2020-08-11T19:44:32","version" => "3.60"},{"date" => "2020-09-30T23:22:16","version" => "3.60_01"},{"date" => "2020-10-07T14:59:47","version" => "3.60_02"},{"date" => "2020-10-12T23:25:45","version" => "3.61"},{"date" => "2020-10-16T20:01:45","version" => "3.62"},{"date" => "2021-07-07T00:08:28","version" => "3.63"},{"date" => "2022-02-01T18:16:40","version" => "3.64"},{"date" => "2022-03-02T22:12:02","version" => "3.65"},{"date" => "2022-03-02T22:41:01","version" => "3.66"},{"date" => "2022-03-08T19:25:43","version" => "3.67"},{"date" => "2022-03-18T22:08:30","version" => "3.68"},{"date" => "2002-03-05T00:00:00","dual_lived" => 1,"perl_release" => "5.007003","version" => "2.0002"},{"date" => "2003-09-25T00:00:00","dual_lived" => 1,"perl_release" => "5.008001","version" => "2.007"},{"date" => "2003-11-05T00:00:00","dual_lived" => 1,"perl_release" => "5.008002","version" => "2.009"},{"date" => "2004-01-14T00:00:00","dual_lived" => 1,"perl_release" => "5.008003","version" => "2.011"},{"date" => "2003-10-27T00:00:00","dual_lived" => 1,"perl_release" => "5.009","version" => "2.008"},{"date" => "2004-03-16T00:00:00","dual_lived" => 1,"perl_release" => "5.009001","version" => "2.011_01"},{"date" => "2017-12-20T00:00:00","dual_lived" => 1,"perl_release" => "5.027007","version" => "3.37"},{"date" => "2018-01-20T00:00:00","dual_lived" => 1,"perl_release" => "5.027008","version" => "3.38"},{"date" => "2018-02-20T00:00:00","dual_lived" => 1,"perl_release" => "5.027009","version" => "3.39"},{"date" => "2018-04-20T00:00:00","dual_lived" => 1,"perl_release" => "5.027011","version" => "3.40"},{"date" => "2022-11-20T00:00:00","dual_lived" => 1,"perl_release" => "5.037006","version" => "3.69"},{"date" => "2023-02-20T00:00:00","dual_lived" => 1,"perl_release" => "5.037009","version" => "3.70"},{"date" => "2023-04-20T00:00:00","dual_lived" => 1,"perl_release" => "5.037011","version" => "3.71"},{"date" => "2023-09-20T00:00:00","dual_lived" => 1,"perl_release" => "5.039003","version" => "3.72"},{"date" => "2024-07-20T00:00:00","dual_lived" => 1,"perl_release" => "5.041002","version" => "3.73"}]},"Devel-StackTrace" => {"advisories" => [{"affected_versions" => ["<1.19"],"cves" => ["CVE-2008-3502"],"description" => "Unspecified vulnerability in Best Practical Solutions RT 3.0.0 through 3.6.6 allows remote authenticated users to cause a denial of service (CPU or memory consumption) via unspecified vectors related to the Devel::StackTrace module for Perl.\n","distribution" => "Devel-StackTrace","fixed_versions" => [">=1.19"],"id" => "CPANSA-Devel-StackTrace-2008-3502","references" => ["http://lists.bestpractical.com/pipermail/rt-announce/2008-June/000158.html","http://www.securityfocus.com/bid/29925","http://secunia.com/advisories/30830","https://exchange.xforce.ibmcloud.com/vulnerabilities/43337"],"reported" => "2008-08-06","severity" => undef}],"main_module" => "Devel::StackTrace","versions" => [{"date" => "2000-06-27T19:21:12","version" => "0.7"},{"date" => "2000-07-04T16:34:23","version" => "0.75"},{"date" => "2000-09-03T02:55:27","version" => "0.8"},{"date" => "2000-09-03T04:10:13","version" => "0.85"},{"date" => "2001-11-24T06:37:34","version" => "0.9"},{"date" => "2002-08-23T09:12:26","version" => "1.00"},{"date" => "2002-09-18T16:19:28","version" => "1.01"},{"date" => "2002-09-19T22:12:09","version" => "1.02"},{"date" => "2003-01-22T20:33:08","version" => "1.03"},{"date" => "2003-09-25T19:15:23","version" => "1.04"},{"date" => "2004-02-17T20:35:35","version" => "1.05"},{"date" => "2004-02-22T00:14:28","version" => "1.06"},{"date" => "2004-02-22T00:30:48","version" => "1.07"},{"date" => "2004-02-23T15:25:26","version" => "1.08"},{"date" => "2004-02-26T22:30:00","version" => "1.09"},{"date" => "2004-03-10T21:25:04","version" => "1.10"},{"date" => "2004-04-12T05:11:33","version" => "1.11"},{"date" => "2005-09-30T05:47:47","version" => "1.12"},{"date" => "2006-04-01T04:51:47","version" => "1.13"},{"date" => "2007-03-16T15:29:38","version" => "1.14"},{"date" => "2007-04-28T20:07:57","version" => "1.15"},{"date" => "2008-02-02T06:09:06","version" => "1.16"},{"date" => "2008-03-30T17:20:19","version" => "1.17"},{"date" => "2008-03-31T14:16:23","version" => "1.18"},{"date" => "2008-06-13T18:07:37","version" => "1.19"},{"date" => "2008-06-13T23:46:42","version" => "1.1901"},{"date" => "2008-07-16T13:20:57","version" => "1.1902"},{"date" => "2008-10-26T01:44:25","version" => "1.20"},{"date" => "2009-07-02T04:50:03","version" => "1.21"},{"date" => "2009-07-15T19:51:37","version" => "1.22"},{"date" => "2010-08-28T01:47:36","version" => "1.23"},{"date" => "2010-09-03T14:18:22","version" => "1.24"},{"date" => "2010-09-06T14:54:15","version" => "1.25"},{"date" => "2010-10-15T15:25:58","version" => "1.26"},{"date" => "2011-01-16T18:57:01","version" => "1.27"},{"date" => "2012-11-16T16:59:05","version" => "1.28"},{"date" => "2012-11-16T17:47:00","version" => "1.29"},{"date" => "2012-11-20T05:07:49","version" => "1.30"},{"date" => "2014-01-16T22:37:16","version" => "1.31"},{"date" => "2014-05-05T08:01:10","version" => "1.32"},{"date" => "2014-06-26T20:43:33","version" => "1.33"},{"date" => "2014-06-26T21:50:12","version" => "1.34"},{"date" => "2014-11-01T18:06:29","version" => "2.00"},{"date" => "2016-03-02T17:23:15","version" => "2.01"},{"date" => "2016-12-07T19:51:47","version" => "2.02"},{"date" => "2017-11-18T17:10:57","version" => "2.03"},{"date" => "2019-05-24T18:54:07","version" => "2.04"},{"date" => "2024-01-08T04:48:56","version" => "2.05"}]},"Dezi" => {"advisories" => [{"affected_versions" => ["<0.002002"],"cves" => [],"description" => "Bypassing authentication on the /index URL app with non-idempotent requests to /search URL.\n","distribution" => "Dezi","fixed_versions" => [">=0.002002"],"id" => "CPANSA-Dezi-2012-01","references" => ["https://metacpan.org/changes/distribution/Dezi","https://github.com/karpet/Dezi/commit/f1ad292b4dd988d1a38202c804bb7a2a3bcca3c8"],"reported" => "2012-09-13"}],"main_module" => "Dezi","versions" => [{"date" => "2011-06-22T04:53:57","version" => "0.001000"},{"date" => "2011-08-03T02:42:22","version" => "0.001001"},{"date" => "2011-09-30T03:35:08","version" => "0.001002"},{"date" => "2011-10-23T02:12:02","version" => "0.001003"},{"date" => "2012-03-17T02:40:15","version" => "0.001004"},{"date" => "2012-07-11T03:20:40","version" => "0.001005"},{"date" => "2012-08-18T02:43:23","version" => "0.001006"},{"date" => "2012-08-22T03:58:33","version" => "0.001007"},{"date" => "2012-09-04T02:05:34","version" => "0.001008"},{"date" => "2012-09-12T03:51:13","version" => "0.002000"},{"date" => "2012-09-13T01:50:59","version" => "0.002001"},{"date" => "2012-09-13T14:10:02","version" => "0.002002"},{"date" => "2012-10-16T00:57:46","version" => "0.002003"},{"date" => "2012-10-18T03:15:21","version" => "0.002004"},{"date" => "2012-12-19T05:25:13","version" => "0.002005"},{"date" => "2013-02-03T02:49:07","version" => "0.002006"},{"date" => "2013-02-05T15:02:54","version" => "0.002007"},{"date" => "2013-02-09T05:37:41","version" => "0.002008"},{"date" => "2013-02-13T02:30:33","version" => "0.002009"},{"date" => "2013-02-13T04:31:49","version" => "0.002010"},{"date" => "2013-11-13T17:08:03","version" => "0.002011"},{"date" => "2014-02-27T18:08:30","version" => "0.002012"},{"date" => "2014-06-05T06:59:12","version" => "0.002998_01"},{"date" => "2014-06-08T04:59:17","version" => "0.003000"},{"date" => "2014-07-30T20:40:24","version" => "0.004000"},{"date" => "2014-09-02T02:45:00","version" => "0.004001"},{"date" => "2015-04-30T22:01:11","version" => "0.004002"},{"date" => "2018-05-16T02:24:24","version" => "0.004003"}]},"Digest" => {"advisories" => [{"affected_versions" => ["<1.17"],"cves" => ["CVE-2011-3597"],"description" => "Eval injection vulnerability in the Digest module before 1.17 for Perl allows context-dependent attackers to execute arbitrary commands via the new constructor.\n","distribution" => "Digest","fixed_versions" => [">=0.17"],"id" => "CPANSA-Digest-2011-3597","references" => ["http://aix.software.ibm.com/aix/efixes/security/perl_advisory2.asc","http://www.redhat.com/support/errata/RHSA-2011-1797.html","http://www.redhat.com/support/errata/RHSA-2011-1424.html","https://bugzilla.redhat.com/show_bug.cgi?id=743010","http://www.securityfocus.com/bid/49911","http://secunia.com/advisories/46279","http://cpansearch.perl.org/src/GAAS/Digest-1.17/Changes","http://www.mandriva.com/security/advisories?name=MDVSA-2012:009","http://www.mandriva.com/security/advisories?name=MDVSA-2012:008","http://secunia.com/advisories/51457","http://www.ubuntu.com/usn/USN-1643-1","http://kb.juniper.net/InfoCenter/index?page=content&id=JSA10735","http://kb.juniper.net/InfoCenter/index?page=content&id=JSA10705","https://oval.cisecurity.org/repository/search/definition/oval%3Aorg.mitre.oval%3Adef%3A19446"],"reported" => "2012-01-13","reviewed_by" => [{"date" => "2022-07-05","email" => "rrwo\@cpan.org","name" => "Robert Rothenberg"}],"severity" => undef},{"affected_versions" => ["<1.19"],"cves" => ["CVE-2016-1238"],"description" => "Includes . in \@INC which might allow local users to gain privileges via a Trojan horse module under the current working directory.\n","distribution" => "Digest","fixed_versions" => [">=1.19"],"id" => "CPANSA-Digest-2016-1238","references" => ["http://www.nntp.perl.org/group/perl.perl5.porters/2016/07/msg238271.html","http://www.securitytracker.com/id/1036440","http://perl5.git.perl.org/perl.git/commit/cee96d52c39b1e7b36e1c62d38bcd8d86e9a41ab","https://rt.perl.org/Public/Bug/Display.html?id=127834","http://www.securityfocus.com/bid/92136","http://www.debian.org/security/2016/dsa-3628","https://h20566.www2.hpe.com/portal/site/hpsc/public/kb/docDisplay?docId=emr_na-c05240731","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DOFRQWJRP2NQJEYEWOMECVW3HAMD5SYN/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/TZBNQH3DMI7HDELJAZ4TFJJANHXOEDWH/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/2FBQOCV3GBAN2EYZUM3CFDJ4ECA3GZOK/","https://security.gentoo.org/glsa/201701-75","https://lists.apache.org/thread.html/7f6a16bc0fd0fd5e67c7fd95bd655069a2ac7d1f88e42d3c853e601c\@%3Cannounce.apache.org%3E","https://lists.debian.org/debian-lts-announce/2018/11/msg00016.html","https://security.gentoo.org/glsa/201812-07","http://lists.opensuse.org/opensuse-security-announce/2019-08/msg00002.html"],"reported" => "2016-08-02","severity" => "high"}],"main_module" => "Digest","versions" => [{"date" => "2001-03-14T06:33:08","version" => "1.00"},{"date" => "2003-01-05T01:23:53","version" => "1.01"},{"date" => "2003-01-19T04:35:36","version" => "1.02"},{"date" => "2003-11-28T12:29:42","version" => "1.03"},{"date" => "2003-11-29T12:08:20","version" => "1.04"},{"date" => "2003-12-01T07:58:06","version" => "1.05"},{"date" => "2004-04-01T10:55:24","version" => "1.06"},{"date" => "2004-04-25T14:39:53","version" => "1.07"},{"date" => "2004-04-29T07:56:42","version" => "1.08"},{"date" => "2004-11-05T12:20:28","version" => "1.09"},{"date" => "2004-11-08T09:41:14","version" => "1.10"},{"date" => "2005-09-11T11:14:33","version" => "1.11"},{"date" => "2005-09-29T10:20:20","version" => "1.12"},{"date" => "2005-10-18T11:59:24","version" => "1.13"},{"date" => "2005-11-26T10:10:21","version" => "1.14"},{"date" => "2006-03-20T15:18:01","version" => "1.15"},{"date" => "2009-06-09T18:58:26","version" => "1.16"},{"date" => "2011-10-02T10:14:32","version" => "1.17"},{"date" => "2020-10-13T19:16:47","version" => "1.18"},{"date" => "2020-10-13T20:02:35","version" => "1.19"},{"date" => "2021-08-24T13:51:51","version" => "1.20"},{"date" => "2012-10-12T00:00:00","dual_lived" => 1,"perl_release" => "5.014003","version" => "1.16_01"},{"date" => "2017-01-14T00:00:00","dual_lived" => 1,"perl_release" => "5.022003","version" => "1.17_01"}]},"Digest-MD5" => {"advisories" => [{"affected_versions" => ["<2.25"],"cves" => ["CVE-2002-0703"],"description" => "An interaction between the Perl MD5 module (perl-Digest-MD5) and Perl could produce incorrect MD5 checksums for UTF-8 data, which could prevent a system from properly verifying the integrity of the data.\n","distribution" => "Digest-MD5","fixed_versions" => [],"id" => "CPANSA-Digest-MD5-2002-0703","references" => ["http://www.linux-mandrake.com/en/security/2002/MDKSA-2002-035.php","http://www.iss.net/security_center/static/9051.php","http://www.securityfocus.com/bid/4716","http://rhn.redhat.com/errata/RHSA-2002-081.html"],"reported" => "2002-07-26","severity" => undef}],"main_module" => "Digest::MD5","versions" => [{"date" => "1998-10-23T12:30:56","version" => "1.99_53"},{"date" => "1998-10-24T13:58:24","version" => "1.99_54"},{"date" => "1998-10-24T22:44:03","version" => "1.99_55"},{"date" => "1998-10-24T23:07:15","version" => "1.99_56"},{"date" => "1998-10-27T21:09:37","version" => "1.99_57"},{"date" => "1998-10-28T14:11:30","version" => "1.99_58"},{"date" => "1998-10-28T20:57:10","version" => "1.99_59"},{"date" => "1998-10-30T17:23:27","version" => "1.99_60"},{"date" => "1998-11-04T22:27:42","version" => "2.00"},{"date" => "1998-12-30T04:01:06","version" => "2.01"},{"date" => "1999-01-31T16:44:38","version" => "2.02"},{"date" => "1999-02-01T20:25:06","version" => "2.02"},{"date" => "1999-02-27T21:39:24","version" => "2.03"},{"date" => "1999-03-05T21:17:35","version" => "2.04"},{"date" => "1999-03-15T10:58:32","version" => "2.05"},{"date" => "1999-03-19T05:05:36","version" => "2.05"},{"date" => "1999-03-26T13:51:38","version" => "2.06"},{"date" => "1999-04-26T09:45:43","version" => "2.07"},{"date" => "1999-06-02T13:44:41","version" => "2.07"},{"date" => "1999-07-28T10:55:54","version" => "2.08"},{"date" => "1999-08-05T23:29:15","version" => "2.09"},{"date" => "1999-09-02T12:45:17","version" => "2.09"},{"date" => "2000-08-18T08:49:59","version" => "2.10"},{"date" => "2000-08-19T17:39:04","version" => "2.11"},{"date" => "2000-09-18T15:10:45","version" => "2.12"},{"date" => "2001-01-19T06:08:47","version" => "2.12"},{"date" => "2001-03-14T05:56:41","version" => "2.13"},{"date" => "2001-03-17T04:35:32","version" => "2.13"},{"date" => "2001-06-24T07:37:20","version" => "2.13"},{"date" => "2001-07-18T13:40:13","version" => "2.14"},{"date" => "2001-08-27T17:53:29","version" => "2.15"},{"date" => "2001-08-29T06:32:30","version" => "2.15"},{"date" => "2001-09-07T05:52:46","version" => "2.16"},{"date" => "2002-04-25T17:24:14","version" => "2.17"},{"date" => "2002-05-01T23:34:50","version" => "2.18"},{"date" => "2002-05-02T03:21:40","version" => "2.19"},{"date" => "2002-05-06T05:20:38","version" => "2.20"},{"date" => "2002-12-28T05:33:19","version" => "2.21"},{"date" => "2003-01-05T01:04:07","version" => "2.22"},{"date" => "2003-01-19T04:55:24","version" => "2.23"},{"date" => "2003-03-09T15:26:49","version" => "2.24"},{"date" => "2003-07-05T05:33:54","version" => "2.25"},{"date" => "2003-07-22T06:15:03","version" => "2.26"},{"date" => "2003-08-05T06:12:31","version" => "2.27"},{"date" => "2003-10-06T13:16:20","version" => "2.28"},{"date" => "2003-10-06T17:37:30","version" => "2.29"},{"date" => "2003-10-09T09:40:47","version" => "2.30"},{"date" => "2003-11-28T13:10:59","version" => "2.31"},{"date" => "2003-12-05T10:15:43","version" => "2.32"},{"date" => "2003-12-07T10:31:15","version" => "2.33"},{"date" => "2005-11-26T10:05:19","version" => "2.34"},{"date" => "2005-11-26T11:15:35","version" => "2.35"},{"date" => "2005-11-30T13:55:38","version" => "2.36"},{"date" => "2008-11-12T09:36:42","version" => "2.37"},{"date" => "2008-11-14T13:50:45","version" => "2.38"},{"date" => "2009-06-09T20:21:55","version" => "2.39"},{"date" => "2010-07-03T14:01:25","version" => "2.40"},{"date" => "2010-09-25T22:12:42","version" => "2.50"},{"date" => "2010-09-30T19:46:29","version" => "2.51"},{"date" => "2012-06-07T22:37:00","version" => "2.52"},{"date" => "2013-07-02T17:56:06","version" => "2.53"},{"date" => "2015-01-12T21:19:42","version" => "2.54"},{"date" => "2016-03-09T21:17:10","version" => "2.55"},{"date" => "2020-10-05T17:19:37","version" => "2.56"},{"date" => "2020-10-05T17:42:48","version" => "2.57"},{"date" => "2020-10-05T21:53:32","version" => "2.58"},{"date" => "2023-12-30T21:01:56","version" => "2.59"},{"date" => "2007-07-07T00:00:00","dual_lived" => 1,"perl_release" => "5.009005","version" => "2.36_01"},{"date" => "2019-10-20T00:00:00","dual_lived" => 1,"perl_release" => "5.031005","version" => "2.55_01"},{"date" => "2023-07-02T00:00:00","dual_lived" => 1,"perl_release" => "5.038","version" => "2.58_01"}]},"Digest-SHA" => {"advisories" => [{"affected_versions" => ["<5.96"],"cves" => ["CVE-2016-1238"],"description" => "Digest::SHA before 5.96 with perls earlier than v5.26 included the current working directory in the module search path, which could lead to the inadvernant loading of unexpected versions of a module. The current directory was removed from the default module search path in perls from v5.26 and later.\n","distribution" => "Digest-SHA","fixed_versions" => [">=5.96"],"id" => "CPANSA-Digest-SHA-2016-1238","references" => ["https://rt.cpan.org/Public/Bug/Display.html?id=116513","https://github.com/advisories/GHSA-hm5v-6984-hfqp","https://metacpan.org/release/MSHELOR/Digest-SHA-5.96/diff/MSHELOR/Digest-SHA-5.95","https://www.nntp.perl.org/group/perl.perl5.porters/2016/07/msg238271.html","https://lists.debian.org/debian-security-announce/2016/msg00206.html","https://security.gentoo.org/glsa/201701-75"],"reported" => undef,"severity" => "high"}],"main_module" => "Digest::SHA","versions" => [{"date" => "2003-10-11T09:45:19","version" => "0.9"},{"date" => "2003-10-20T09:50:18","version" => "1.0"},{"date" => "2003-10-25T11:31:51","version" => "1.01"},{"date" => "2003-11-01T21:22:08","version" => "2.0"},{"date" => "2003-11-09T11:06:19","version" => "2.1"},{"date" => "2003-11-16T11:08:06","version" => "2.2"},{"date" => "2003-11-19T11:54:09","version" => "2.3"},{"date" => "2003-11-23T00:52:20","version" => "2.4"},{"date" => "2003-11-26T12:32:09","version" => "3.0"},{"date" => "2003-11-30T07:47:28","version" => "v4.0.0"},{"date" => "2003-11-30T16:19:28","version" => "3.1-alpha1"},{"date" => "2003-12-01T13:27:50","version" => "4.0.2"},{"date" => "2003-12-03T11:20:55","version" => "v4.0.3"},{"date" => "2003-12-04T07:54:40","version" => "v4.0.4"},{"date" => "2003-12-06T09:23:46","version" => "v4.0.5"},{"date" => "2003-12-11T11:30:14","version" => "v4.0.6"},{"date" => "2003-12-13T09:30:39","version" => "v4.0.7"},{"date" => "2003-12-19T07:30:18","version" => "v4.0.8"},{"date" => "2003-12-24T10:37:28","version" => "v4.0.9"},{"date" => "2003-12-25T08:13:32","version" => "v4.1.0"},{"date" => "2003-12-28T00:09:30","version" => "v4.2.0"},{"date" => "2004-01-24T08:43:05","version" => "v4.2.1"},{"date" => "2004-02-01T08:52:29","version" => "v4.2.2"},{"date" => "2004-02-07T10:45:32","version" => "v4.3.0"},{"date" => "2004-03-04T10:41:51","version" => "v4.3.1"},{"date" => "2004-04-28T11:30:19","version" => "4.3.2"},{"date" => "2004-05-05T07:56:21","version" => "4.3.3"},{"date" => "2004-05-14T12:08:55","version" => "5.00"},{"date" => "2004-05-21T20:20:18","version" => "5.01"},{"date" => "2004-07-29T10:13:58","version" => "5.02"},{"date" => "2004-07-31T07:34:46","version" => "5.03"},{"date" => "2004-08-06T09:44:08","version" => "5.10"},{"date" => "2004-08-15T12:25:38","version" => "5.20"},{"date" => "2004-08-23T12:35:36","version" => "5.21"},{"date" => "2004-09-08T08:01:56","version" => "5.22"},{"date" => "2004-09-10T06:51:39","version" => "5.23"},{"date" => "2004-09-12T11:33:41","version" => "5.24"},{"date" => "2004-09-13T02:27:16","version" => "5.25"},{"date" => "2004-10-10T09:13:00","version" => "5.26"},{"date" => "2004-10-24T11:25:48","version" => "5.27"},{"date" => "2004-11-17T09:23:50","version" => "5.28"},{"date" => "2005-08-15T09:20:59","version" => "5.29"},{"date" => "2005-08-21T00:35:30","version" => "5.30"},{"date" => "2005-09-05T08:36:39","version" => "5.31"},{"date" => "2005-12-02T10:13:07","version" => "5.32"},{"date" => "2006-02-03T02:22:56","version" => "5.34"},{"date" => "2006-05-08T01:10:50","version" => "5.35"},{"date" => "2006-05-08T11:19:00","version" => "5.36"},{"date" => "2006-05-15T11:31:17","version" => "5.37"},{"date" => "2006-05-25T10:10:52","version" => "5.38"},{"date" => "2006-05-28T10:49:11","version" => "5.39"},{"date" => "2006-06-02T21:45:07","version" => "5.40"},{"date" => "2006-06-03T09:31:44","version" => "5.41"},{"date" => "2006-07-24T11:22:26","version" => "5.42"},{"date" => "2006-08-05T10:13:57","version" => "5.43"},{"date" => "2006-10-14T07:59:30","version" => "5.44"},{"date" => "2007-06-26T10:20:05","version" => "5.45"},{"date" => "2008-04-09T12:40:29","version" => "5.46"},{"date" => "2008-04-30T11:17:26","version" => "5.47"},{"date" => "2010-01-05T02:07:18","version" => "5.48"},{"date" => "2010-12-12T14:44:43","version" => "5.49"},{"date" => "2010-12-14T13:46:10","version" => "5.50"},{"date" => "2011-03-03T13:19:38","version" => "5.60"},{"date" => "2011-03-09T12:56:01","version" => "5.61"},{"date" => "2011-05-14T11:11:34","version" => "5.62"},{"date" => "2011-11-08T13:27:54","version" => "5.63"},{"date" => "2011-12-14T10:18:37","version" => "5.70"},{"date" => "2012-02-29T11:11:59","version" => "5.71"},{"date" => "2012-09-25T01:14:59","version" => "5.72"},{"date" => "2012-10-31T11:42:32","version" => "5.73"},{"date" => "2012-11-24T11:40:47","version" => "5.74"},{"date" => "2012-12-10T21:21:06","version" => "5.80"},{"date" => "2013-01-14T14:32:22","version" => "5.81"},{"date" => "2013-01-24T12:06:14","version" => "5.82"},{"date" => "2013-03-04T16:22:03","version" => "5.83"},{"date" => "2013-03-10T00:42:51","version" => "5.84"},{"date" => "2013-06-26T11:11:56","version" => "5.85"},{"date" => "2014-01-30T15:40:50","version" => "5.86"},{"date" => "2014-02-18T01:26:20","version" => "5.87"},{"date" => "2014-03-17T16:05:33","version" => "5.88"},{"date" => "2014-04-19T13:09:10","version" => "5.89"},{"date" => "2014-05-07T15:54:15","version" => "5.90"},{"date" => "2014-05-16T17:36:12","version" => "5.91"},{"date" => "2014-06-01T07:25:04","version" => "5.92"},{"date" => "2014-10-26T13:15:37","version" => "5.93"},{"date" => "2015-01-10T09:49:55","version" => "5.94"},{"date" => "2015-01-10T20:24:40","version" => "5.95"},{"date" => "2016-07-28T11:11:53","version" => "5.96"},{"date" => "2017-09-06T09:38:45","version" => "5.97"},{"date" => "2017-10-04T08:38:13","version" => "5.98"},{"date" => "2017-12-09T06:04:13","version" => "6.00"},{"date" => "2017-12-25T07:41:55","version" => "6.01"},{"date" => "2018-04-20T23:47:19","version" => "6.02"},{"date" => "2022-08-08T18:56:41","version" => "6.03"},{"date" => "2023-02-25T19:06:34","version" => "6.04"},{"date" => "2013-08-12T00:00:00","dual_lived" => 1,"perl_release" => "5.018001","version" => "5.84_01"},{"date" => "2014-10-01T00:00:00","dual_lived" => 1,"perl_release" => "5.018003","version" => "5.84_02"},{"date" => "2017-01-14T00:00:00","dual_lived" => 1,"perl_release" => "5.022003","version" => "5.95_01"}]},"Dpkg" => {"advisories" => [{"affected_versions" => ["<1.21.8"],"cves" => ["CVE-2022-1664"],"description" => "Dpkg::Source::Archive in dpkg, the Debian package management system, before version 1.21.8, 1.20.10, 1.19.8, 1.18.26 is prone to a directory traversal vulnerability. When extracting untrusted source packages in v2 and v3 source package formats that include a debian.tar, the in-place extraction can lead to directory traversal situations on specially crafted orig.tar and debian.tar tarballs.\n","distribution" => "Dpkg","fixed_versions" => [">=1.21.8"],"id" => "CPANSA-Dpkg-2022-1664","references" => ["https://lists.debian.org/debian-security-announce/2022/msg00115.html","https://git.dpkg.org/cgit/dpkg/dpkg.git/commit/?id=faa4c92debe45412bfcf8a44f26e827800bb24be","https://git.dpkg.org/cgit/dpkg/dpkg.git/commit/?id=7a6c03cb34d4a09f35df2f10779cbf1b70a5200b","https://lists.debian.org/debian-lts-announce/2022/05/msg00033.html","https://git.dpkg.org/cgit/dpkg/dpkg.git/commit/?id=58814cacee39c4ce9e2cd0e3a3b9b57ad437eff5","https://git.dpkg.org/cgit/dpkg/dpkg.git/commit/?id=1f23dddc17f69c9598477098c7fb9936e15fa495"],"reported" => "2022-05-26","severity" => "critical"},{"affected_versions" => ["<1.18.24"],"cves" => ["CVE-2017-8283"],"description" => "dpkg-source in dpkg 1.3.0 through 1.18.23 is able to use a non-GNU patch program and does not offer a protection mechanism for blank-indented diff hunks, which allows remote attackers to conduct directory traversal attacks via a crafted Debian source package, as demonstrated by use of dpkg-source on NetBSD.\n","distribution" => "Dpkg","fixed_versions" => [">=1.18.24"],"id" => "CPANSA-Dpkg-2017-8283","references" => ["http://www.openwall.com/lists/oss-security/2017/04/20/2","http://www.securityfocus.com/bid/98064"],"reported" => "2017-04-26","severity" => "critical"},{"affected_versions" => ["<1.18.11"],"cves" => ["CVE-2016-1238"],"description" => "(1) cpan/Archive-Tar/bin/ptar, (2) cpan/Archive-Tar/bin/ptardiff, (3) cpan/Archive-Tar/bin/ptargrep, (4) cpan/CPAN/scripts/cpan, (5) cpan/Digest-SHA/shasum, (6) cpan/Encode/bin/enc2xs, (7) cpan/Encode/bin/encguess, (8) cpan/Encode/bin/piconv, (9) cpan/Encode/bin/ucmlint, (10) cpan/Encode/bin/unidump, (11) cpan/ExtUtils-MakeMaker/bin/instmodsh, (12) cpan/IO-Compress/bin/zipdetails, (13) cpan/JSON-PP/bin/json_pp, (14) cpan/Test-Harness/bin/prove, (15) dist/ExtUtils-ParseXS/lib/ExtUtils/xsubpp, (16) dist/Module-CoreList/corelist, (17) ext/Pod-Html/bin/pod2html, (18) utils/c2ph.PL, (19) utils/h2ph.PL, (20) utils/h2xs.PL, (21) utils/libnetcfg.PL, (22) utils/perlbug.PL, (23) utils/perldoc.PL, (24) utils/perlivp.PL, and (25) utils/splain.PL in Perl 5.x before 5.22.3-RC2 and 5.24 before 5.24.1-RC2 do not properly remove . (period) characters from the end of the includes directory array, which might allow local users to gain privileges via a Trojan horse module under the current working directory.\n","distribution" => "Dpkg","fixed_versions" => [">=1.18.11"],"id" => "CPANSA-Dpkg-2016-1238","references" => ["http://www.nntp.perl.org/group/perl.perl5.porters/2016/07/msg238271.html","http://www.securitytracker.com/id/1036440","http://perl5.git.perl.org/perl.git/commit/cee96d52c39b1e7b36e1c62d38bcd8d86e9a41ab","https://rt.perl.org/Public/Bug/Display.html?id=127834","http://www.securityfocus.com/bid/92136","http://www.debian.org/security/2016/dsa-3628","https://h20566.www2.hpe.com/portal/site/hpsc/public/kb/docDisplay?docId=emr_na-c05240731","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DOFRQWJRP2NQJEYEWOMECVW3HAMD5SYN/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/TZBNQH3DMI7HDELJAZ4TFJJANHXOEDWH/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/2FBQOCV3GBAN2EYZUM3CFDJ4ECA3GZOK/","https://security.gentoo.org/glsa/201701-75","https://lists.apache.org/thread.html/7f6a16bc0fd0fd5e67c7fd95bd655069a2ac7d1f88e42d3c853e601c\@%3Cannounce.apache.org%3E","https://lists.debian.org/debian-lts-announce/2018/11/msg00016.html","https://security.gentoo.org/glsa/201812-07","http://lists.opensuse.org/opensuse-security-announce/2019-08/msg00002.html"],"reported" => "2016-08-02","severity" => "high"},{"affected_versions" => ["<1.18.4"],"cves" => ["CVE-2015-0860"],"description" => "Off-by-one error in the extracthalf function in dpkg-deb/extract.c in the dpkg-deb component in Debian dpkg 1.16.x before 1.16.17 and 1.17.x before 1.17.26 allows remote attackers to execute arbitrary code via the archive magic version number in an \"old-style\" Debian binary package, which triggers a stack-based buffer overflow.\n","distribution" => "Dpkg","fixed_versions" => [">=1.18.4"],"id" => "CPANSA-Dpkg-2015-0860","references" => ["http://www.ubuntu.com/usn/USN-2820-1","http://www.debian.org/security/2015/dsa-3407","https://blog.fuzzing-project.org/30-Stack-overflows-and-out-of-bounds-read-in-dpkg-Debian.html","https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=798324","https://anonscm.debian.org/cgit/dpkg/dpkg.git/commit/dpkg-deb/extract.c?id=e65aa3db04eb908c9507d5d356a95cedb890814d","https://security.gentoo.org/glsa/201612-07"],"reported" => "2015-12-03","severity" => undef},{"affected_versions" => ["<1.18.0"],"cves" => ["CVE-2015-0840"],"description" => "The dpkg-source command in Debian dpkg before 1.16.16 and 1.17.x before 1.17.25 allows remote attackers to bypass signature verification via a crafted Debian source control file (.dsc).\n","distribution" => "Dpkg","fixed_versions" => [">=1.18.0"],"id" => "CPANSA-Dpkg-2015-0840","references" => ["http://www.ubuntu.com/usn/USN-2566-1","http://www.debian.org/security/2015/dsa-3217","http://lists.opensuse.org/opensuse-updates/2015-06/msg00029.html","http://lists.fedoraproject.org/pipermail/package-announce/2015-May/157387.html"],"reported" => "2015-04-13","severity" => undef},{"affected_versions" => ["<1.17.22"],"cves" => ["CVE-2014-8625"],"description" => "Multiple format string vulnerabilities in the parse_error_msg function in parsehelp.c in dpkg before 1.17.22 allow remote attackers to cause a denial of service (crash) and possibly execute arbitrary code via format string specifiers in the (1) package or (2) architecture name.\n","distribution" => "Dpkg","fixed_versions" => [">=1.17.22"],"id" => "CPANSA-Dpkg-2014-8625","references" => ["http://seclists.org/oss-sec/2014/q4/539","https://bugs.launchpad.net/ubuntu/+source/dpkg/+bug/1389135","http://seclists.org/oss-sec/2014/q4/622","https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=768485","http://seclists.org/oss-sec/2014/q4/551","http://lists.fedoraproject.org/pipermail/package-announce/2015-May/157387.html","https://exchange.xforce.ibmcloud.com/vulnerabilities/98551"],"reported" => "2015-01-20","severity" => undef},{"affected_versions" => ["<1.17.10"],"cves" => ["CVE-2014-0471"],"description" => "Directory traversal vulnerability in the unpacking functionality in dpkg before 1.15.9, 1.16.x before 1.16.13, and 1.17.x before 1.17.8 allows remote attackers to write arbitrary files via a crafted source package, related to \"C-style filename quoting.\"\n","distribution" => "Dpkg","fixed_versions" => [">=1.17.10"],"id" => "CPANSA-Dpkg-2014-0471","references" => ["http://www.debian.org/security/2014/dsa-2915","http://www.ubuntu.com/usn/USN-2183-1","http://www.securityfocus.com/bid/67106"],"reported" => "2014-04-30","severity" => undef},{"affected_versions" => ["<1.17.10"],"cves" => ["CVE-2014-3127"],"description" => "dpkg 1.15.9 on Debian squeeze introduces support for the \"C-style encoded filenames\" feature without recognizing that the squeeze patch program lacks this feature, which triggers an interaction error that allows remote attackers to conduct directory traversal attacks and modify files outside of the intended directories via a crafted source package. NOTE: this can be considered a release engineering problem in the effort to fix CVE-2014-0471.\n","distribution" => "Dpkg","fixed_versions" => [">=1.17.10"],"id" => "CPANSA-Dpkg-2014-3127","references" => ["https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=746306","http://www.securityfocus.com/bid/67181","http://seclists.org/oss-sec/2014/q2/227","http://seclists.org/oss-sec/2014/q2/191","http://metadata.ftp-master.debian.org/changelogs//main/d/dpkg/dpkg_1.15.10_changelog"],"reported" => "2014-05-14","severity" => undef},{"affected_versions" => ["<1.17.10"],"cves" => ["CVE-2014-3864"],"description" => "Directory traversal vulnerability in dpkg-source in dpkg-dev 1.3.0 allows remote attackers to modify files outside of the intended directories via a crafted source package that lacks a --- header line.\n","distribution" => "Dpkg","fixed_versions" => [">=1.17.10"],"id" => "CPANSA-Dpkg-2014-3864","references" => ["http://openwall.com/lists/oss-security/2014/05/25/2","https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=746498","http://www.securityfocus.com/bid/67725","http://www.ubuntu.com/usn/USN-2242-1","http://www.debian.org/security/2014/dsa-2953"],"reported" => "2014-05-30","severity" => undef},{"affected_versions" => ["<1.17.10"],"cves" => ["CVE-2014-3865"],"description" => "Multiple directory traversal vulnerabilities in dpkg-source in dpkg-dev 1.3.0 allow remote attackers to modify files outside of the intended directories via a source package with a crafted Index: pseudo-header in conjunction with (1) missing --- and +++ header lines or (2) a +++ header line with a blank pathname.\n","distribution" => "Dpkg","fixed_versions" => [">=1.17.10"],"id" => "CPANSA-Dpkg-2014-3865","references" => ["https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=749183","http://openwall.com/lists/oss-security/2014/05/25/2","http://www.securityfocus.com/bid/67727","http://www.ubuntu.com/usn/USN-2242-1","http://www.debian.org/security/2014/dsa-2953"],"reported" => "2014-05-30","severity" => undef},{"affected_versions" => ["<1.17.9"],"cves" => ["CVE-2014-0471"],"description" => "Directory traversal vulnerability in the unpacking functionality in dpkg before 1.15.9, 1.16.x before 1.16.13, and 1.17.x before 1.17.8 allows remote attackers to write arbitrary files via a crafted source package, related to \"C-style filename quoting.\"\n","distribution" => "Dpkg","fixed_versions" => [">=1.17.9"],"id" => "CPANSA-Dpkg-2014-0471","references" => ["http://www.debian.org/security/2014/dsa-2915","http://www.ubuntu.com/usn/USN-2183-1","http://www.securityfocus.com/bid/67106"],"reported" => "2014-04-30","severity" => undef},{"affected_versions" => ["<1.17.9"],"cves" => ["CVE-2014-3127"],"description" => "dpkg 1.15.9 on Debian squeeze introduces support for the \"C-style encoded filenames\" feature without recognizing that the squeeze patch program lacks this feature, which triggers an interaction error that allows remote attackers to conduct directory traversal attacks and modify files outside of the intended directories via a crafted source package. NOTE: this can be considered a release engineering problem in the effort to fix CVE-2014-0471.\n","distribution" => "Dpkg","fixed_versions" => [">=1.17.9"],"id" => "CPANSA-Dpkg-2014-3127","references" => ["https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=746306","http://www.securityfocus.com/bid/67181","http://seclists.org/oss-sec/2014/q2/227","http://seclists.org/oss-sec/2014/q2/191","http://metadata.ftp-master.debian.org/changelogs//main/d/dpkg/dpkg_1.15.10_changelog"],"reported" => "2014-05-14","severity" => undef},{"affected_versions" => ["<1.17.8"],"cves" => ["CVE-2014-0471"],"description" => "Directory traversal vulnerability in the unpacking functionality in dpkg before 1.15.9, 1.16.x before 1.16.13, and 1.17.x before 1.17.8 allows remote attackers to write arbitrary files via a crafted source package, related to \"C-style filename quoting.\"\n","distribution" => "Dpkg","fixed_versions" => [">=1.17.8"],"id" => "CPANSA-Dpkg-2014-0471","references" => ["http://www.debian.org/security/2014/dsa-2915","http://www.ubuntu.com/usn/USN-2183-1","http://www.securityfocus.com/bid/67106"],"reported" => "2014-04-30","severity" => undef},{"affected_versions" => ["<1.15.8.8"],"cves" => ["CVE-2010-1679"],"description" => "Directory traversal vulnerability in dpkg-source in dpkg before 1.14.31 and 1.15.x allows user-assisted remote attackers to modify arbitrary files via directory traversal sequences in a patch for a source-format 3.0 package.\n","distribution" => "Dpkg","fixed_versions" => [">=1.15.8.8"],"id" => "CPANSA-Dpkg-2010-1679","references" => ["http://www.vupen.com/english/advisories/2011/0044","http://secunia.com/advisories/42831","http://secunia.com/advisories/42826","http://www.ubuntu.com/usn/USN-1038-1","http://www.debian.org/security/2011/dsa-2142","http://www.vupen.com/english/advisories/2011/0040","http://osvdb.org/70368","http://lists.fedoraproject.org/pipermail/package-announce/2011-January/053311.html","http://lists.fedoraproject.org/pipermail/package-announce/2011-January/053306.html","http://secunia.com/advisories/43054","http://www.securityfocus.com/bid/45703","http://www.vupen.com/english/advisories/2011/0196","https://exchange.xforce.ibmcloud.com/vulnerabilities/64615"],"reported" => "2011-01-11","severity" => undef},{"affected_versions" => ["<1.15.6"],"cves" => ["CVE-2010-0396"],"description" => "Directory traversal vulnerability in the dpkg-source component in dpkg before 1.14.29 allows remote attackers to modify arbitrary files via a crafted Debian source archive.\n","distribution" => "Dpkg","fixed_versions" => [">=1.15.6"],"id" => "CPANSA-Dpkg-2010-0396","references" => ["http://www.debian.org/security/2010/dsa-2011","http://www.vupen.com/english/advisories/2010/0582","http://security.debian.org/pool/updates/main/d/dpkg/dpkg_1.14.29.tar.gz","https://exchange.xforce.ibmcloud.com/vulnerabilities/56887"],"reported" => "2010-03-15","severity" => undef},{"affected_versions" => ["==1.9.21"],"cves" => ["CVE-2004-2768"],"description" => "dpkg 1.9.21 does not properly reset the metadata of a file during replacement of the file in a package upgrade, which might allow local users to gain privileges by creating a hard link to a vulnerable (1) setuid file, (2) setgid file, or (3) device, a related issue to CVE-2010-2059.\n","distribution" => "Dpkg","fixed_versions" => [">=1.9.22"],"id" => "CPANSA-Dpkg-2004-2768","references" => ["http://www.hackinglinuxexposed.com/articles/20031214.html","http://lists.jammed.com/ISN/2003/12/0056.html","https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=225692","https://bugzilla.redhat.com/show_bug.cgi?id=598775","https://exchange.xforce.ibmcloud.com/vulnerabilities/59428"],"reported" => "2010-06-08","severity" => undef},{"affected_versions" => ["<1.15.10"],"cves" => ["CVE-2011-0402"],"description" => "dpkg-source in dpkg before 1.14.31 and 1.15.x allows user-assisted remote attackers to modify arbitrary files via a symlink attack on unspecified files in the .pc directory.\n","distribution" => "Dpkg","fixed_versions" => [">=1.15.10"],"id" => "CPANSA-Dpkg-2011-0402","references" => ["http://www.ubuntu.com/usn/USN-1038-1","http://secunia.com/advisories/42831","http://www.debian.org/security/2011/dsa-2142","http://secunia.com/advisories/42826","http://www.vupen.com/english/advisories/2011/0040","http://www.vupen.com/english/advisories/2011/0044","http://osvdb.org/70367","http://lists.fedoraproject.org/pipermail/package-announce/2011-January/053311.html","http://lists.fedoraproject.org/pipermail/package-announce/2011-January/053306.html","http://www.vupen.com/english/advisories/2011/0196","http://www.securityfocus.com/bid/45703","http://secunia.com/advisories/43054","https://exchange.xforce.ibmcloud.com/vulnerabilities/64614"],"reported" => "2011-01-11","severity" => undef}],"main_module" => "Dpkg","versions" => [{"date" => "2018-09-26T18:53:52","version" => "v1.19.1"},{"date" => "2018-10-08T10:54:58","version" => "v1.19.2"},{"date" => "2019-01-22T18:41:25","version" => "v1.19.3"},{"date" => "2019-02-23T17:40:31","version" => "v1.19.5"},{"date" => "2019-03-25T14:54:21","version" => "v1.19.6"},{"date" => "2019-06-03T21:51:58","version" => "v1.19.7"},{"date" => "2020-03-08T03:05:24","version" => "v1.20.0"},{"date" => "2020-06-27T01:26:33","version" => "v1.20.1"},{"date" => "2020-06-27T23:35:03","version" => "v1.20.2"},{"date" => "2020-06-29T11:02:10","version" => "v1.20.3"},{"date" => "2020-07-07T06:22:23","version" => "v1.20.4"},{"date" => "2020-07-08T03:55:55","version" => "v1.20.5"},{"date" => "2021-01-08T04:23:50","version" => "v1.20.6"},{"date" => "2021-01-09T00:19:44","version" => "v1.20.7"},{"date" => "2021-04-13T21:44:34","version" => "v1.20.8"},{"date" => "2021-04-13T23:33:15","version" => "v1.20.9"},{"date" => "2021-12-05T18:08:48","version" => "v1.21.0"},{"date" => "2021-12-06T20:23:10","version" => "v1.21.1"},{"date" => "2022-03-13T20:07:04","version" => "v1.21.2"},{"date" => "2022-03-24T20:19:38","version" => "v1.21.3"},{"date" => "2022-03-26T12:56:21","version" => "v1.21.4"},{"date" => "2022-03-29T01:07:10","version" => "v1.21.5"},{"date" => "2022-05-25T15:21:07","version" => "v1.21.8"},{"date" => "2022-07-01T09:48:45","version" => "v1.21.9"},{"date" => "2022-12-01T12:08:26","version" => "v1.21.10"},{"date" => "2022-12-02T23:34:17","version" => "v1.21.11"},{"date" => "2022-12-19T01:27:49","version" => "v1.21.13"},{"date" => "2023-01-01T23:04:24","version" => "v1.21.14"},{"date" => "2023-01-25T22:18:51","version" => "v1.21.19"},{"date" => "2023-05-16T22:34:01","version" => "v1.21.22"},{"date" => "2023-08-30T10:44:22","version" => "v1.22.0"},{"date" => "2023-10-30T03:47:45","version" => "v1.22.1"},{"date" => "2023-12-18T03:09:08","version" => "v1.22.2"},{"date" => "2024-01-24T12:39:35","version" => "v1.22.4"},{"date" => "2024-02-27T03:56:46","version" => "v1.22.5"},{"date" => "2024-03-10T21:52:57","version" => "v1.22.6"},{"date" => "2024-07-16T23:58:08","version" => "v1.22.7"},{"date" => "2024-07-21T18:44:31","version" => "v1.22.8"},{"date" => "2024-08-01T11:07:53","version" => "v1.22.11"},{"date" => "2025-01-02T03:22:30","version" => "v1.22.12"},{"date" => "2025-01-03T11:09:37","version" => "v1.22.13"},{"date" => "2025-03-07T02:57:57","version" => "v1.22.16"},{"date" => "2025-03-09T18:23:59","version" => "v1.22.18"},{"date" => "2025-05-18T22:53:57","version" => "v1.22.19"},{"date" => "2025-06-04T23:18:14","version" => "v1.22.20"},{"date" => "2025-07-02T00:09:01","version" => "v1.22.21"},{"date" => "2025-12-16T22:55:42","version" => "v1.23.0"},{"date" => "2025-12-17T12:41:12","version" => "v1.23.1"},{"date" => "2026-01-18T17:58:28","version" => "v1.23.4"}]},"EV-Hiredis" => {"advisories" => [{"affected_versions" => [">=0.01,<=0.04"],"cves" => ["CVE-2020-7105"],"description" => "async.c and dict.c in libhiredis.a in hiredis through 0.14.0 allow a NULL pointer dereference because malloc return values are unchecked.\n","distribution" => "EV-Hiredis","fixed_versions" => [],"id" => "CPANSA-EV-Hiredis-2020-7105-hiredis","references" => ["https://github.com/redis/hiredis/issues/747","https://lists.debian.org/debian-lts-announce/2020/01/msg00028.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/76ZDGWBV3YEEQ2YC65ZJEQLKQFVFBZHX/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/ZKOTCIYFEWJJILUGL4JQ3CJAM3TWYZ2A/"],"reported" => "2020-01-16","severity" => "high"}],"main_module" => "EV::Hiredis","versions" => [{"date" => "2013-01-09T10:22:05","version" => "0.01"},{"date" => "2013-03-13T06:16:24","version" => "0.02"},{"date" => "2014-09-18T09:39:46","version" => "0.03"},{"date" => "2017-04-23T10:09:14","version" => "0.04"},{"date" => "2022-09-11T04:29:22","version" => "0.05"},{"date" => "2023-04-25T22:39:52","version" => "0.06"},{"date" => "2023-05-03T14:14:01","version" => "0.07"}]},"EasyTCP" => {"advisories" => [{"affected_versions" => ["<0.15"],"cves" => ["CVE-2002-20002"],"description" => "The Net::EasyTCP package before 0.15 for Perl always uses Perl's builtin rand(), which is not a strong random number generator, for cryptographic keys.\n","distribution" => "EasyTCP","fixed_versions" => [">=0.15"],"id" => "CPANSA-EasyTCP-2002-20002","references" => ["https://github.com/briandfoy/cpan-security-advisory/issues/184","https://metacpan.org/release/MNAGUIB/EasyTCP-0.15/view/EasyTCP.pm","https://metacpan.org/release/MNAGUIB/EasyTCP-0.26/changes"],"reported" => "2025-01-02","severity" => "moderate"},{"affected_versions" => [">=0.15"],"cves" => ["CVE-2024-56830"],"description" => "The Net::EasyTCP package 0.15 through 0.26 for Perl uses Perl's builtin rand() if no strong randomization module is present.\n","distribution" => "EasyTCP","fixed_versions" => [],"id" => "CPANSA-EasyTCP-2024-56830","references" => ["https://github.com/briandfoy/cpan-security-advisory/issues/184","https://metacpan.org/release/MNAGUIB/EasyTCP-0.26/changes"],"reported" => "2025-01-02","severity" => "moderate"}],"main_module" => "Net::EasyTCP","versions" => [{"date" => "2002-01-03T20:21:16","version" => "0.01"},{"date" => "2002-01-10T23:58:29","version" => "0.02"},{"date" => "2002-01-12T22:23:25","version" => "0.03"},{"date" => "2002-01-14T20:13:58","version" => "0.04"},{"date" => "2002-01-16T16:30:59","version" => "0.05"},{"date" => "2002-01-18T04:48:03","version" => "0.06"},{"date" => "2002-01-21T20:26:09","version" => "0.07"},{"date" => "2002-01-22T21:08:52","version" => "0.08"},{"date" => "2002-01-31T16:33:35","version" => "0.09"},{"date" => "2002-02-01T02:09:00","version" => "0.10"},{"date" => "2002-02-05T20:35:47","version" => "0.11"},{"date" => "2002-02-22T19:51:44","version" => "0.12"},{"date" => "2002-03-22T20:47:32","version" => "0.13"},{"date" => "2002-09-07T05:21:49","version" => "0.14"},{"date" => "2002-09-25T03:02:37","version" => "0.15"},{"date" => "2002-09-30T20:20:38","version" => "0.16"},{"date" => "2002-11-11T19:38:16","version" => "0.17"},{"date" => "2003-02-26T22:15:03","version" => "0.18"},{"date" => "2003-03-02T05:47:04","version" => "0.19"},{"date" => "2003-05-14T19:40:01","version" => "0.20"},{"date" => "2003-05-14T20:22:30","version" => "0.21"},{"date" => "2003-05-15T13:10:31","version" => "0.22"},{"date" => "2003-05-24T13:19:56","version" => "0.23"},{"date" => "2003-07-15T01:11:20","version" => "0.24"},{"date" => "2003-08-07T12:19:25","version" => "0.25"},{"date" => "2004-04-06T02:50:22","version" => "0.26"}]},"Elive" => {"advisories" => [{"affected_versions" => ["<1.20"],"cves" => [],"description" => "Elive::DAO->set() did not die on tainted data.\n","distribution" => "Elive","fixed_versions" => [">=1.20"],"id" => "CPANSA-Elive-2011-01","references" => ["https://metacpan.org/dist/Elive/changes"],"reported" => "2011-10-15","severity" => undef}],"main_module" => "Elive","versions" => [{"date" => "2009-03-17T06:37:43","version" => "0.01"},{"date" => "2009-04-13T23:51:59","version" => "0.02"},{"date" => "2009-04-14T20:26:27","version" => "0.03"},{"date" => "2009-04-15T22:30:08","version" => "0.04"},{"date" => "2009-04-17T07:27:23","version" => "0.05"},{"date" => "2009-04-17T22:04:55","version" => "0.06"},{"date" => "2009-04-22T00:14:13","version" => "0.07"},{"date" => "2009-04-22T03:10:13","version" => "0.08"},{"date" => "2009-04-24T22:26:35","version" => "0.09"},{"date" => "2009-04-28T07:30:45","version" => "0.10"},{"date" => "2009-04-29T21:49:12","version" => "0.11"},{"date" => "2009-05-01T23:15:47","version" => "0.12"},{"date" => "2009-05-04T22:19:09","version" => "0.13"},{"date" => "2009-05-05T20:09:18","version" => "0.14"},{"date" => "2009-05-08T22:04:14","version" => "0.15"},{"date" => "2009-05-11T20:38:56","version" => "0.16"},{"date" => "2009-05-13T21:31:52","version" => "0.17"},{"date" => "2009-05-15T03:47:36","version" => "0.18"},{"date" => "2009-05-18T21:43:03","version" => "0.19"},{"date" => "2009-05-24T00:13:36","version" => "0.20"},{"date" => "2009-05-24T20:48:19","version" => "0.21"},{"date" => "2009-05-27T22:05:37","version" => "0.22"},{"date" => "2009-05-29T05:09:57","version" => "0.23"},{"date" => "2009-06-03T04:48:43","version" => "0.24"},{"date" => "2009-06-03T22:18:02","version" => "0.25"},{"date" => "2009-06-12T22:36:31","version" => "0.26"},{"date" => "2009-06-19T21:34:40","version" => "0.27"},{"date" => "2009-06-22T03:47:43","version" => "0.28"},{"date" => "2009-06-24T04:14:37","version" => "0.29"},{"date" => "2009-06-26T23:24:47","version" => "0.30"},{"date" => "2009-07-03T06:18:23","version" => "0.31"},{"date" => "2009-07-17T22:56:55","version" => "0.32"},{"date" => "2009-07-22T03:22:18","version" => "0.33"},{"date" => "2009-07-28T06:46:45","version" => "0.34"},{"date" => "2009-08-02T22:36:31","version" => "0.35"},{"date" => "2009-08-03T22:44:25","version" => "0.36"},{"date" => "2009-08-05T21:02:32","version" => "0.37"},{"date" => "2009-08-21T08:29:37","version" => "0.38"},{"date" => "2009-08-31T02:24:45","version" => "0.39"},{"date" => "2009-09-10T01:20:54","version" => "0.40"},{"date" => "2009-09-11T21:34:13","version" => "0.41"},{"date" => "2009-10-08T00:53:22","version" => "0.42"},{"date" => "2009-10-20T23:09:46","version" => "0.43"},{"date" => "2009-10-22T00:05:22","version" => "0.44"},{"date" => "2009-10-26T04:15:36","version" => "0.45"},{"date" => "2009-10-28T08:27:27","version" => "0.46"},{"date" => "2009-10-28T21:26:06","version" => "0.47"},{"date" => "2009-10-29T00:00:43","version" => "0.48"},{"date" => "2009-11-02T21:37:24","version" => "0.48_01"},{"date" => "2009-11-06T20:36:30","version" => "0.49"},{"date" => "2009-11-09T21:34:02","version" => "0.50"},{"date" => "2009-11-16T00:26:26","version" => "0.51"},{"date" => "2009-11-30T20:38:39","version" => "0.52"},{"date" => "2009-12-14T23:14:43","version" => "0.53"},{"date" => "2009-12-18T00:24:06","version" => "0.53_1"},{"date" => "2009-12-18T22:36:34","version" => "0.54"},{"date" => "2009-12-20T20:02:22","version" => "0.55"},{"date" => "2010-01-04T06:35:00","version" => "0.56"},{"date" => "2010-01-04T21:18:52","version" => "0.57"},{"date" => "2010-01-14T00:08:40","version" => "0.58"},{"date" => "2010-01-21T22:46:27","version" => "0.59"},{"date" => "2010-01-24T21:24:09","version" => "0.60"},{"date" => "2010-01-26T22:38:54","version" => "0.61"},{"date" => "2010-02-15T23:06:41","version" => "0.62"},{"date" => "2010-03-06T22:34:53","version" => "0.63"},{"date" => "2010-03-11T22:45:28","version" => "0.64"},{"date" => "2010-05-17T00:40:50","version" => "0.65"},{"date" => "2010-05-21T23:54:39","version" => "0.66"},{"date" => "2010-05-27T22:12:29","version" => "0.67"},{"date" => "2010-06-02T07:33:50","version" => "0.68"},{"date" => "2010-06-11T00:12:21","version" => "0.69"},{"date" => "2010-06-22T05:13:22","version" => "0.70"},{"date" => "2010-06-22T22:20:27","version" => "0.71"},{"date" => "2010-08-13T01:10:30","version" => "0.72"},{"date" => "2010-09-03T03:48:51","version" => "0.73"},{"date" => "2010-10-14T20:54:08","version" => "0.74_2"},{"date" => "2010-10-18T01:49:41","version" => "0.74"},{"date" => "2010-10-27T23:52:59","version" => "0.75"},{"date" => "2010-11-09T23:46:08","version" => "0.76"},{"date" => "2010-12-08T21:27:13","version" => "0.77"},{"date" => "2010-12-08T23:17:00","version" => "0.78"},{"date" => "2011-01-20T02:01:43","version" => "0.79"},{"date" => "2011-01-27T19:56:34","version" => "0.80"},{"date" => "2011-02-03T03:17:09","version" => "0.81"},{"date" => "2011-02-10T00:02:08","version" => "0.82"},{"date" => "2011-03-10T05:19:08","version" => "0.83"},{"date" => "2011-03-11T01:11:39","version" => "0.84"},{"date" => "2011-03-14T00:55:18","version" => "0.85"},{"date" => "2011-03-14T21:15:08","version" => "0.86"},{"date" => "2011-04-11T00:59:22","version" => "0.87"},{"date" => "2011-04-11T19:19:42","version" => "0.87.1"},{"date" => "2011-04-15T02:12:50","version" => "0.87.2"},{"date" => "2011-04-27T02:43:51","version" => "0.88"},{"date" => "2011-05-20T00:15:55","version" => "0.89"},{"date" => "2011-06-08T23:34:06","version" => "0.90"},{"date" => "2011-06-14T23:35:27","version" => "0.91"},{"date" => "2011-06-28T07:09:46","version" => "0.95"},{"date" => "2011-06-29T21:42:38","version" => "0.96"},{"date" => "2011-07-05T06:35:18","version" => "0.97"},{"date" => "2011-07-08T00:35:18","version" => "0.98"},{"date" => "2011-07-14T03:25:12","version" => "0.99"},{"date" => "2011-07-19T00:14:00","version" => "1.00"},{"date" => "2011-07-20T01:14:39","version" => "1.01"},{"date" => "2011-07-21T05:49:47","version" => "1.02"},{"date" => "2011-07-23T23:23:35","version" => "1.03"},{"date" => "2011-07-29T00:14:06","version" => "1.04"},{"date" => "2011-08-01T02:20:53","version" => "1.05"},{"date" => "2011-08-05T21:36:24","version" => "1.06"},{"date" => "2011-08-07T01:43:31","version" => "1.07"},{"date" => "2011-08-09T00:51:44","version" => "1.08"},{"date" => "2011-08-10T05:13:13","version" => "1.09"},{"date" => "2011-08-10T21:06:42","version" => "1.10"},{"date" => "2011-08-11T22:27:24","version" => "1.11"},{"date" => "2011-08-15T00:58:40","version" => "1.12"},{"date" => "2011-08-19T00:21:11","version" => "1.13"},{"date" => "2011-08-20T22:44:01","version" => "1.14"},{"date" => "2011-08-23T21:43:48","version" => "1.15"},{"date" => "2011-08-26T22:25:28","version" => "1.16"},{"date" => "2011-09-08T22:32:49","version" => "1.17"},{"date" => "2011-09-16T00:00:34","version" => "1.18"},{"date" => "2011-09-28T07:09:24","version" => "1.19"},{"date" => "2011-11-15T01:28:33","version" => "1.20"},{"date" => "2011-12-03T01:49:03","version" => "1.21"},{"date" => "2012-01-05T04:04:10","version" => "1.22"},{"date" => "2012-01-25T20:01:01","version" => "1.23"},{"date" => "2012-02-28T01:03:16","version" => "1.24"},{"date" => "2012-04-18T04:53:06","version" => "1.25"},{"date" => "2012-05-04T04:11:34","version" => "1.26"},{"date" => "2012-07-13T21:59:27","version" => "1.27"},{"date" => "2012-10-12T02:45:37","version" => "1.28"},{"date" => "2012-10-26T21:16:49","version" => "1.29"},{"date" => "2013-01-04T01:33:50","version" => "1.30"},{"date" => "2013-03-28T02:39:54","version" => "1.31"},{"date" => "2014-02-28T16:40:50","version" => "1.32"},{"date" => "2015-01-21T21:14:50","version" => "1.33"},{"date" => "2015-04-03T22:38:32","version" => "1.34"},{"date" => "2015-06-29T02:59:33","version" => "1.35"},{"date" => "2015-12-03T20:48:05","version" => "1.36"},{"date" => "2015-12-04T02:58:35","version" => "1.37"}]},"Email-Address" => {"advisories" => [{"affected_versions" => ["<1.905"],"cves" => ["CVE-2014-0477"],"description" => "Inefficient regular expression, which allows remote attackers to cause a denial of service (CPU consumption) via an empty quoted string in an RFC 2822 address.\n","distribution" => "Email-Address","fixed_versions" => [">=1.905"],"id" => "CPANSA-Email-Address-2014-01","references" => ["https://metacpan.org/changes/distribution/Email-Address"],"reported" => "2014-07-03"},{"affected_versions" => ["<1.909"],"cves" => ["CVE-2018-12558"],"description" => "The parse() method in the Email::Address module through 1.909 for Perl is vulnerable to Algorithmic complexity on specially prepared input, leading to Denial of Service. Prepared special input that caused this problem contained 30 form-field characters (\"\\f\").\n","distribution" => "Email-Address","fixed_versions" => [">=1.909"],"id" => "CPANSA-Email-Address-2014-01","references" => ["https://bugs.debian.org/cgi-bin/bugreport.cgi?bug=901873","http://www.openwall.com/lists/oss-security/2018/06/19/3","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00012.html"],"reported" => "2018-06-19"},{"affected_versions" => ["<1.904"],"cves" => ["CVE-2014-4720"],"description" => "Email::Address module before 1.904 for Perl uses an inefficient regular expression, which allows remote attackers to cause a denial of service (CPU consumption) via vectors related to \"backtracking into the phrase,\" a different vulnerability than CVE-2014-0477.\n","distribution" => "Email-Address","fixed_versions" => [">=1.904"],"id" => "CPANSA-Email-Address-2014-4720","references" => ["https://github.com/rjbs/Email-Address/blob/master/Changes","http://seclists.org/oss-sec/2014/q2/563"],"reported" => "2014-07-06","severity" => undef}],"main_module" => "Email::Address","versions" => [{"date" => "2004-05-27T03:19:56","version" => "1.1"},{"date" => "2004-06-02T16:35:30","version" => "1.2"},{"date" => "2004-08-16T21:39:58","version" => "1.3"},{"date" => "2004-10-05T18:10:42","version" => "1.5"},{"date" => "2004-10-05T18:20:42","version" => "1.6"},{"date" => "2004-10-13T10:21:17","version" => "1.7"},{"date" => "2004-10-22T16:37:27","version" => "1.80"},{"date" => "2006-07-11T15:04:28","version" => "1.85"},{"date" => "2006-07-22T00:42:17","version" => "1.86"},{"date" => "2006-08-10T16:48:44","version" => "1.870"},{"date" => "2006-10-12T19:35:04","version" => "1.861"},{"date" => "2006-10-12T22:16:28","version" => "1.871"},{"date" => "2006-11-11T16:01:38","version" => "1.880"},{"date" => "2006-11-19T21:19:02","version" => "1.881"},{"date" => "2006-11-22T01:26:44","version" => "1.882"},{"date" => "2006-11-25T13:53:46","version" => "1.883"},{"date" => "2006-12-05T03:41:39","version" => "1.884"},{"date" => "2007-03-01T01:08:16","version" => "1.885"},{"date" => "2007-03-01T20:18:53","version" => "1.886"},{"date" => "2007-04-01T19:15:49","version" => "1.887"},{"date" => "2007-06-23T01:27:24","version" => "1.888"},{"date" => "2007-12-19T22:14:37","version" => "1.889"},{"date" => "2010-08-22T19:03:33","version" => "1.890"},{"date" => "2010-08-31T00:56:53","version" => "1.891"},{"date" => "2010-09-03T23:45:13","version" => "1.892"},{"date" => "2012-01-03T03:55:12","version" => "1.893"},{"date" => "2012-01-14T16:17:56","version" => "1.894"},{"date" => "2012-01-15T18:41:33","version" => "1.895"},{"date" => "2012-08-01T03:07:33","version" => "1.896"},{"date" => "2012-12-17T15:16:33","version" => "1.897"},{"date" => "2013-02-07T21:41:48","version" => "1.898"},{"date" => "2013-08-02T14:54:13","version" => "1.899"},{"date" => "2013-08-08T18:46:07","version" => "1.900"},{"date" => "2014-01-29T03:43:28","version" => "1.901"},{"date" => "2014-04-17T15:19:31","version" => "1.902"},{"date" => "2014-04-18T01:07:10","version" => "1.903"},{"date" => "2014-06-14T04:22:22","version" => "1.904"},{"date" => "2014-06-18T02:55:59","version" => "1.905"},{"date" => "2015-02-03T21:49:39","version" => "1.906"},{"date" => "2015-02-03T22:48:46","version" => "1.907"},{"date" => "2015-09-20T02:55:12","version" => "1.908"},{"date" => "2018-03-05T03:26:56","version" => "1.909"},{"date" => "2018-12-18T02:29:23","version" => "1.910"},{"date" => "2018-12-22T16:31:37","version" => "1.911"},{"date" => "2018-12-31T19:51:36","version" => "1.912"},{"date" => "2023-01-10T00:42:33","version" => "1.913"}]},"Email-MIME" => {"advisories" => [{"affected_versions" => ["<1.954"],"cves" => ["CVE-2024-4140"],"description" => "An excessive memory use issue (CWE-770) exists in Email-MIME, before version 1.954, which can cause denial of service when parsing multipart MIME messages. The patch set (from 2020 and 2024) limits excessive depth and the total number of parts.\n","distribution" => "Email-MIME","fixed_versions" => [">=1.954"],"id" => "CPANSA-Email-MIME-2024-4140","references" => ["https://bugs.debian.org/960062","https://github.com/rjbs/Email-MIME/commit/02bf3e26812c8f38a86a33c168571f9783365df2","https://github.com/rjbs/Email-MIME/commit/3a12edd119e493156a5a05e45dd50f4e36b702e8","https://github.com/rjbs/Email-MIME/commit/3dcf096eeccb8e4dd42738de676c8f4a5aa7a531","https://github.com/rjbs/Email-MIME/commit/7e96ecfa1da44914a407f82ae98ba817bba08f2d","https://github.com/rjbs/Email-MIME/commit/b2cb62f19e12580dd235f79e2546d44a6bec54d1","https://github.com/rjbs/Email-MIME/commit/fc0fededd24a71ccc51bcd8b1e486385d09aae63","https://github.com/rjbs/Email-MIME/issues/66","https://github.com/rjbs/Email-MIME/pull/80","https://www.cve.org/CVERecord?id=CVE-2024-4140"],"reported" => "2024-05-02","severity" => undef}],"main_module" => "Email::MIME","versions" => [{"date" => "2004-01-23T12:04:54","version" => "1.0_01"},{"date" => "2004-03-24T16:01:33","version" => "1.1"},{"date" => "2004-04-02T08:52:35","version" => "1.2"},{"date" => "2004-04-05T16:27:42","version" => "1.3"},{"date" => "2004-07-01T17:38:00","version" => "1.4"},{"date" => "2004-07-01T17:40:06","version" => "1.5"},{"date" => "2004-07-04T20:17:06","version" => "1.6"},{"date" => "2004-07-05T21:56:19","version" => "1.7"},{"date" => "2004-08-25T09:58:28","version" => "1.8"},{"date" => "2004-10-30T00:19:03","version" => "1.81"},{"date" => "2004-11-18T01:02:46","version" => "1.82"},{"date" => "2006-07-13T12:09:00","version" => "1.85"},{"date" => "2006-08-22T12:15:01","version" => "1.851"},{"date" => "2006-09-06T03:22:24","version" => "1.852"},{"date" => "2006-10-12T19:23:59","version" => "1.853"},{"date" => "2006-10-15T12:57:06","version" => "1.854"},{"date" => "2006-10-19T19:26:56","version" => "1.855"},{"date" => "2006-11-28T01:54:55","version" => "1.856"},{"date" => "2006-11-28T02:45:42","version" => "1.857"},{"date" => "2007-02-10T03:53:13","version" => "1.858"},{"date" => "2007-03-21T02:13:51","version" => "1.859"},{"date" => "2007-07-14T02:17:11","version" => "1.860"},{"date" => "2007-11-06T02:03:23","version" => "1.861"},{"date" => "2008-09-08T22:24:06","version" => "1.861_01"},{"date" => "2009-01-24T03:09:24","version" => "1.862"},{"date" => "2009-01-30T13:35:20","version" => "1.863"},{"date" => "2009-11-03T20:49:02","version" => "1.900"},{"date" => "2009-11-05T19:29:16","version" => "1.901"},{"date" => "2009-11-11T20:53:15","version" => "1.902"},{"date" => "2009-12-23T14:14:45","version" => "1.903"},{"date" => "2010-09-04T22:05:49","version" => "1.904"},{"date" => "2010-09-06T13:25:51","version" => "1.905"},{"date" => "2010-10-08T01:06:33","version" => "1.906"},{"date" => "2011-02-02T22:52:42","version" => "1.907"},{"date" => "2011-06-02T03:16:50","version" => "1.908"},{"date" => "2011-09-08T19:27:38","version" => "1.909"},{"date" => "2011-09-12T16:45:52","version" => "1.910"},{"date" => "2012-07-22T23:12:42","version" => "1.911"},{"date" => "2013-04-08T19:42:09","version" => "1.912_01"},{"date" => "2013-06-17T15:24:37","version" => "1.920"},{"date" => "2013-07-02T02:51:36","version" => "1.921"},{"date" => "2013-07-10T12:45:29","version" => "1.922"},{"date" => "2013-08-09T02:00:30","version" => "1.923"},{"date" => "2013-08-11T03:25:40","version" => "1.924"},{"date" => "2013-11-08T12:02:21","version" => "1.925"},{"date" => "2014-01-29T04:29:29","version" => "1.926"},{"date" => "2014-12-04T15:22:49","version" => "1.927"},{"date" => "2014-12-16T02:49:06","version" => "1.928"},{"date" => "2015-02-17T14:32:33","version" => "1.929"},{"date" => "2015-03-26T03:00:09","version" => "1.930"},{"date" => "2015-07-12T21:32:56","version" => "1.931"},{"date" => "2015-07-25T02:25:32","version" => "1.932"},{"date" => "2015-07-25T13:33:07","version" => "1.933"},{"date" => "2015-08-02T00:35:40","version" => "1.934"},{"date" => "2015-08-31T20:49:57","version" => "1.935"},{"date" => "2015-09-11T02:48:33","version" => "1.936"},{"date" => "2016-01-28T18:33:58","version" => "1.937"},{"date" => "2017-01-02T01:04:29","version" => "1.938"},{"date" => "2017-01-14T19:59:46","version" => "1.939"},{"date" => "2017-01-29T15:34:49","version" => "1.940"},{"date" => "2017-03-05T00:18:30","version" => "1.941"},{"date" => "2017-03-05T13:16:39","version" => "1.942"},{"date" => "2017-06-09T23:01:41","version" => "1.943"},{"date" => "2017-07-25T16:40:42","version" => "1.944"},{"date" => "2017-07-25T18:18:48","version" => "1.945"},{"date" => "2017-08-31T13:31:14","version" => "1.946"},{"date" => "2020-05-09T18:30:39","version" => "1.947"},{"date" => "2020-05-09T19:06:22","version" => "1.948"},{"date" => "2020-05-24T14:27:02","version" => "1.949"},{"date" => "2020-11-03T00:22:52","version" => "1.950"},{"date" => "2021-12-14T14:43:29","version" => "1.951"},{"date" => "2021-12-14T14:58:13","version" => "1.952"},{"date" => "2023-01-09T00:03:49","version" => "1.953"},{"date" => "2024-05-02T21:13:55","version" => "1.954"}]},"Encode" => {"advisories" => [{"affected_versions" => ["<2.85"],"cves" => ["CVE-2016-1238"],"description" => "Loading optional modules from . (current directory).\n","distribution" => "Encode","fixed_versions" => [">=2.85"],"id" => "CPANSA-Encode-2016-01","references" => ["https://metacpan.org/changes/distribution/Encode","https://github.com/dankogai/p5-encode/pull/58/commits/12be15d64ce089154c4367dc1842cd0dc0993ec6"],"reported" => "2016-07-27","severity" => "high"},{"affected_versions" => [">=3.05","<=3.11"],"cves" => ["CVE-2021-36770"],"description" => "Encode.pm, as distributed in Perl through 5.34.0, allows local users to gain privileges via a Trojan horse Encode::ConfigLocal library (in the current working directory) that preempts dynamic module loading. Exploitation requires an unusual configuration, and certain 2021 versions of Encode.pm (3.05 through 3.11). This issue occurs because the || operator evaluates \@INC in a scalar context, and thus \@INC has only an integer value.\n","distribution" => "Encode","fixed_versions" => [">3.11"],"id" => "CPANSA-Encode-2021-01","references" => ["https://github.com/Perl/perl5/commit/c1a937fef07c061600a0078f4cb53fe9c2136bb9","https://github.com/dankogai/p5-encode/commit/527e482dc70b035d0df4f8c77a00d81f8d775c74","https://metacpan.org/dist/Encode/changes","https://news.cpanel.com/unscheduled-tsr-10-august-2021/","https://security.netapp.com/advisory/ntap-20210909-0003/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/5NDGQSGMEZ75FJGBKNYC75OTO7TF7XHB/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/6KOZYD7BH2DNIAEZ2ZL4PJ4QUVQI6Y33/","https://security-tracker.debian.org/tracker/CVE-2021-36770"],"reported" => "2021-07-17"},{"affected_versions" => ["<2.44"],"cves" => ["CVE-2011-2939"],"description" => "Off-by-one error in the decode_xs function in Unicode/Unicode.xs in the Encode module before 2.44, as used in Perl before 5.15.6, might allow context-dependent attackers to cause a denial of service (memory corruption) via a crafted Unicode string, which triggers a heap-based buffer overflow.\n","distribution" => "Encode","fixed_versions" => [">=2.44"],"id" => "CPANSA-Encode-2011-2939","references" => ["http://perl5.git.perl.org/perl.git/commitdiff/e46d973584785af1f445c4dedbee4243419cb860#patch5","https://bugzilla.redhat.com/show_bug.cgi?id=731246","http://www.openwall.com/lists/oss-security/2011/08/19/17","http://www.redhat.com/support/errata/RHSA-2011-1424.html","http://www.openwall.com/lists/oss-security/2011/08/18/8","http://secunia.com/advisories/46989","http://www.mandriva.com/security/advisories?name=MDVSA-2012:008","http://secunia.com/advisories/51457","http://www.ubuntu.com/usn/USN-1643-1","http://www.securityfocus.com/bid/49858","http://secunia.com/advisories/46172","http://cpansearch.perl.org/src/FLORA/perl-5.14.2/pod/perldelta.pod","http://secunia.com/advisories/55314","http://search.cpan.org/~flora/perl-5.14.2/pod/perldelta.pod#Encode_decode_xs_n-byte_heap-overflow_(CVE-2011-2939)"],"reported" => "2012-01-13","severity" => undef,"x-commit" => "Encode CVE-2011-2939 GitHub #13"}],"main_module" => "Encode","versions" => [{"date" => "2002-03-20T08:30:40","version" => "0.93"},{"date" => "2002-03-20T20:15:52","version" => "0.94"},{"date" => "2002-03-21T16:07:21","version" => "0.95"},{"date" => "2002-03-22T22:33:15","version" => "0.96"},{"date" => "2002-03-23T20:36:05","version" => "0.97"},{"date" => "2002-03-24T16:07:09","version" => "0.98"},{"date" => "2002-03-25T19:45:16","version" => "0.99"},{"date" => "2002-03-28T23:39:49","version" => "1.00"},{"date" => "2002-03-29T21:43:17","version" => "1.01"},{"date" => "2002-03-31T21:40:25","version" => "1.10"},{"date" => "2002-03-31T22:27:07","version" => "1.11"},{"date" => "2002-04-04T20:02:40","version" => "1.20"},{"date" => "2002-04-07T15:36:48","version" => "1.26"},{"date" => "2002-04-07T18:49:41","version" => "1.27"},{"date" => "2002-04-07T19:05:34","version" => "1.28"},{"date" => "2002-04-08T02:49:31","version" => "1.30"},{"date" => "2002-04-08T18:51:14","version" => "1.31"},{"date" => "2002-04-09T20:26:37","version" => "1.32"},{"date" => "2002-04-10T22:44:19","version" => "1.33"},{"date" => "2002-04-14T22:49:10","version" => "1.40"},{"date" => "2002-04-16T23:47:16","version" => "1.41"},{"date" => "2002-04-19T06:18:26","version" => "1.50"},{"date" => "2002-04-20T10:08:39","version" => "1.51"},{"date" => "2002-04-20T23:55:45","version" => "1.52"},{"date" => "2002-04-22T09:56:04","version" => "1.56"},{"date" => "2002-04-22T20:37:12","version" => "1.57"},{"date" => "2002-04-23T00:22:06","version" => "1.58"},{"date" => "2002-04-24T20:23:42","version" => "1.60"},{"date" => "2002-04-26T03:19:40","version" => "1.61"},{"date" => "2002-04-27T11:43:39","version" => "1.62"},{"date" => "2002-04-27T19:52:51","version" => "1.63"},{"date" => "2002-04-29T07:20:38","version" => "1.64"},{"date" => "2002-04-30T16:40:07","version" => "1.65"},{"date" => "2002-05-01T05:51:35","version" => "1.66"},{"date" => "2002-05-02T07:43:35","version" => "1.67"},{"date" => "2002-05-03T12:29:47","version" => "1.68"},{"date" => "2002-05-04T16:50:40","version" => "1.69"},{"date" => "2002-05-06T10:36:39","version" => "1.70"},{"date" => "2002-05-07T16:30:42","version" => "1.71"},{"date" => "2002-05-20T16:04:48","version" => "1.72"},{"date" => "2002-05-28T18:41:36","version" => "1.74"},{"date" => "2002-06-01T18:17:49","version" => "1.75"},{"date" => "2002-08-25T15:18:49","version" => "1.76"},{"date" => "2002-10-06T03:59:19","version" => "1.77"},{"date" => "2002-10-20T15:55:16","version" => "1.78"},{"date" => "2002-10-21T06:11:36","version" => "1.79"},{"date" => "2002-10-21T20:42:56","version" => "1.80"},{"date" => "2002-11-08T18:42:11","version" => "1.81"},{"date" => "2002-11-14T23:17:11","version" => "1.82"},{"date" => "2002-11-18T18:06:47","version" => "1.83"},{"date" => "2003-01-10T12:09:05","version" => "1.84"},{"date" => "2003-01-21T22:23:28","version" => "1.85"},{"date" => "2003-01-22T03:36:42","version" => "1.86"},{"date" => "2003-02-06T02:01:00","version" => "1.87"},{"date" => "2003-02-20T14:46:12","version" => "1.88"},{"date" => "2003-02-28T01:45:53","version" => "1.89"},{"date" => "2003-03-09T17:54:26","version" => "1.90"},{"date" => "2003-03-09T20:12:08","version" => "1.91"},{"date" => "2003-03-31T03:51:31","version" => "1.92"},{"date" => "2003-04-24T17:50:54","version" => "1.93"},{"date" => "2003-05-10T18:31:48","version" => "1.94"},{"date" => "2003-05-21T09:22:43","version" => "1.95"},{"date" => "2003-06-18T09:41:21","version" => "1.96"},{"date" => "2003-07-08T22:01:28","version" => "1.97"},{"date" => "2003-08-25T11:47:32","version" => "1.98"},{"date" => "2003-12-29T02:52:28","version" => "1.99"},{"date" => "2004-05-16T21:05:06","version" => "2.00"},{"date" => "2004-05-25T16:31:35","version" => "2.01"},{"date" => "2004-08-31T11:01:51","version" => "2.02"},{"date" => "2004-10-06T06:50:47","version" => "2.03"},{"date" => "2004-10-16T21:26:58","version" => "2.04"},{"date" => "2004-10-19T05:03:32","version" => "2.05"},{"date" => "2004-10-22T06:29:14","version" => "2.06"},{"date" => "2004-10-22T19:43:19","version" => "2.07"},{"date" => "2004-10-24T13:04:29","version" => "2.08"},{"date" => "2004-12-03T19:21:42","version" => "2.09"},{"date" => "2005-05-16T18:54:53","version" => "2.10"},{"date" => "2005-08-05T11:26:06","version" => "2.11"},{"date" => "2005-09-08T14:23:38","version" => "2.12"},{"date" => "2006-01-15T15:12:01","version" => "2.13"},{"date" => "2006-01-15T15:57:41","version" => "2.14"},{"date" => "2006-04-06T16:01:30","version" => "2.15"},{"date" => "2006-05-03T18:38:44","version" => "2.16"},{"date" => "2006-05-09T17:14:04","version" => "2.17"},{"date" => "2006-06-03T20:34:08","version" => "2.18"},{"date" => "2007-04-06T13:05:52","version" => "2.19"},{"date" => "2007-04-22T15:17:34","version" => "2.20"},{"date" => "2007-05-12T06:50:09","version" => "2.21"},{"date" => "2007-05-29T07:43:07","version" => "2.22"},{"date" => "2007-05-29T18:21:25","version" => "2.23"},{"date" => "2008-03-12T10:12:18","version" => "2.24"},{"date" => "2008-05-07T21:06:08","version" => "2.25"},{"date" => "2008-07-01T21:03:33","version" => "2.26"},{"date" => "2009-01-21T23:01:50","version" => "2.27"},{"date" => "2009-02-01T13:16:44","version" => "2.29"},{"date" => "2009-02-15T17:48:01","version" => "2.30"},{"date" => "2009-02-16T06:25:32","version" => "2.31"},{"date" => "2009-03-07T07:45:00","version" => "2.32"},{"date" => "2009-03-25T08:01:10","version" => "2.33"},{"date" => "2009-07-08T13:53:25","version" => "2.34"},{"date" => "2009-07-13T02:32:45","version" => "2.35"},{"date" => "2009-09-06T09:20:21","version" => "2.36"},{"date" => "2009-09-06T14:37:23","version" => "2.37"},{"date" => "2009-11-16T14:34:43","version" => "2.38"},{"date" => "2009-11-26T09:31:02","version" => "2.39"},{"date" => "2010-09-18T18:47:17","version" => "2.40"},{"date" => "2010-12-23T11:12:33","version" => "2.41"},{"date" => "2010-12-31T22:52:35","version" => "2.42"},{"date" => "2011-05-21T23:21:24","version" => "2.43"},{"date" => "2011-08-09T08:01:30","version" => "2.44"},{"date" => "2012-08-05T23:15:11","version" => "2.45"},{"date" => "2012-08-12T05:52:45","version" => "2.46"},{"date" => "2012-08-15T05:40:21","version" => "2.47"},{"date" => "2013-02-18T02:43:35","version" => "2.48"},{"date" => "2013-03-05T03:19:15","version" => "2.49"},{"date" => "2013-04-26T18:36:59","version" => "2.50"},{"date" => "2013-04-29T22:21:31","version" => "2.51"},{"date" => "2013-08-14T02:33:46","version" => "2.52"},{"date" => "2013-08-29T15:27:02","version" => "2.53"},{"date" => "2013-08-29T16:50:08","version" => "2.54"},{"date" => "2013-09-14T07:58:54","version" => "2.55"},{"date" => "2013-12-22T04:12:07","version" => "2.56"},{"date" => "2014-01-03T04:55:36","version" => "2.57"},{"date" => "2014-03-28T02:41:54","version" => "2.58"},{"date" => "2014-04-06T17:41:19","version" => "2.59"},{"date" => "2014-04-29T16:34:10","version" => "2.60"},{"date" => "2014-05-31T09:55:56","version" => "2.61"},{"date" => "2014-05-31T12:20:28","version" => "2.62"},{"date" => "2014-10-19T07:13:44","version" => "2.63"},{"date" => "2014-10-29T15:42:04","version" => "2.64"},{"date" => "2014-11-27T14:12:57","version" => "2.65"},{"date" => "2014-12-02T23:37:28","version" => "2.66"},{"date" => "2014-12-04T20:28:33","version" => "2.67"},{"date" => "2015-01-22T10:29:46","version" => "2.68"},{"date" => "2015-02-05T10:43:34","version" => "2.69"},{"date" => "2015-02-05T10:56:52","version" => "2.70"},{"date" => "2015-03-12T00:14:19","version" => "2.71"},{"date" => "2015-03-14T02:51:25","version" => "2.72"},{"date" => "2015-04-15T23:27:13","version" => "2.73"},{"date" => "2015-06-25T00:59:20","version" => "2.74"},{"date" => "2015-06-30T10:10:03","version" => "2.75"},{"date" => "2015-07-31T02:26:51","version" => "2.76"},{"date" => "2015-09-15T14:03:35","version" => "2.77"},{"date" => "2015-09-24T02:29:52","version" => "2.78"},{"date" => "2016-01-22T07:08:25","version" => "2.79"},{"date" => "2016-01-25T15:04:42","version" => "2.80"},{"date" => "2016-02-06T19:34:58","version" => "2.81"},{"date" => "2016-02-06T20:21:37","version" => "2.82"},{"date" => "2016-03-24T08:00:30","version" => "2.83"},{"date" => "2016-04-11T07:24:26","version" => "2.84"},{"date" => "2016-08-04T03:37:23","version" => "2.85"},{"date" => "2016-08-10T18:25:39","version" => "2.86"},{"date" => "2016-10-28T05:15:33","version" => "2.87"},{"date" => "2016-11-29T23:38:19","version" => "2.88"},{"date" => "2017-04-21T05:24:59","version" => "2.89"},{"date" => "2017-06-10T17:46:11","version" => "2.90"},{"date" => "2017-06-22T08:18:22","version" => "2.91"},{"date" => "2017-07-18T07:23:39","version" => "2.92"},{"date" => "2017-10-06T22:33:35","version" => "2.93"},{"date" => "2018-01-09T06:04:38","version" => "2.94"},{"date" => "2018-02-08T00:41:02","version" => "2.95"},{"date" => "2018-02-11T05:41:37","version" => "2.96"},{"date" => "2018-02-21T12:30:05","version" => "2.97"},{"date" => "2018-04-22T09:14:59","version" => "2.98"},{"date" => "2019-01-21T03:28:35","version" => "2.99"},{"date" => "2019-01-31T04:42:29","version" => "2.100"},{"date" => "2019-01-31T05:05:06","version" => "3.00"},{"date" => "2019-03-13T00:45:28","version" => "3.01"},{"date" => "2019-12-25T09:47:36","version" => "3.02"},{"date" => "2020-03-02T04:45:26","version" => "3.03"},{"date" => "2020-03-10T22:40:35","version" => "3.04"},{"date" => "2020-03-18T05:03:23","version" => "3.05"},{"date" => "2020-05-02T02:40:38","version" => "3.06"},{"date" => "2020-07-25T13:08:13","version" => "3.07"},{"date" => "2020-12-02T09:20:23","version" => "3.08"},{"date" => "2021-05-14T11:03:11","version" => "3.09"},{"date" => "2021-05-18T07:51:48","version" => "3.10"},{"date" => "2021-07-23T02:41:38","version" => "3.11"},{"date" => "2021-08-09T14:30:33","version" => "3.12"},{"date" => "2021-10-06T00:57:50","version" => "3.13"},{"date" => "2021-10-08T00:35:29","version" => "3.14"},{"date" => "2021-10-08T15:45:44","version" => "3.15"},{"date" => "2021-10-13T08:39:09","version" => "3.16"},{"date" => "2022-04-07T03:18:23","version" => "3.17"},{"date" => "2022-06-25T02:14:35","version" => "3.18"},{"date" => "2022-08-04T04:51:01","version" => "3.19"},{"date" => "2023-11-10T01:26:15","version" => "3.20"},{"date" => "2024-02-25T23:19:43","version" => "3.21"},{"date" => "2002-03-05T00:00:00","dual_lived" => 1,"perl_release" => "5.007003","version" => "0.40"},{"date" => "2003-09-25T00:00:00","dual_lived" => 1,"perl_release" => "5.008001","version" => "1.9801"},{"date" => "2004-04-21T00:00:00","dual_lived" => 1,"perl_release" => "5.008004","version" => "1.99_01"},{"date" => "2006-08-15T00:00:00","dual_lived" => 1,"perl_release" => "5.009004","version" => "2.18_01"},{"date" => "2012-11-10T00:00:00","dual_lived" => 1,"perl_release" => "5.012005","version" => "2.39_01"},{"date" => "2011-09-26T00:00:00","dual_lived" => 1,"perl_release" => "5.014002","version" => "2.42_01"},{"date" => "2013-03-10T00:00:00","dual_lived" => 1,"perl_release" => "5.014004","version" => "2.42_02"},{"date" => "2013-03-11T00:00:00","dual_lived" => 1,"perl_release" => "5.016003","version" => "2.44_01"},{"date" => "2017-01-14T00:00:00","dual_lived" => 1,"perl_release" => "5.022003","version" => "2.72_01"},{"date" => "2017-01-14T00:00:00","dual_lived" => 1,"perl_release" => "5.024001","version" => "2.80_01"},{"date" => "2022-03-13T00:00:00","dual_lived" => 1,"perl_release" => "5.034001","version" => "3.08_01"}]},"ExtUtils-MakeMaker" => {"advisories" => [{"affected_versions" => ["<7.22"],"cves" => ["CVE-2016-1238"],"description" => "Loading modules from . (current directory).\n","distribution" => "ExtUtils-MakeMaker","fixed_versions" => [">=7.22"],"id" => "CPANSA-ExtUtils-MakeMaker-2016-01","references" => ["https://metacpan.org/changes/distribution/ExtUtils-MakeMaker","https://github.com/Perl-Toolchain-Gang/ExtUtils-MakeMaker/commit/3e9df17d11c40f2561c23ec79693c8c390e0ae88"],"reported" => "2016-08-07","severity" => "high"}],"main_module" => "ExtUtils::MakeMaker","versions" => [{"date" => "2001-07-06T08:23:56","version" => "5.47_01"},{"date" => "2002-01-16T20:19:18","version" => "5.48_01"},{"date" => "2002-01-18T04:56:33","version" => "5.48_03"},{"date" => "2002-01-22T00:33:31","version" => "5.48_04"},{"date" => "2002-02-04T08:46:04","version" => "5.49_01"},{"date" => "2002-03-05T04:53:40","version" => "5.50_01"},{"date" => "2002-03-25T07:53:14","version" => "5.51_01"},{"date" => "2002-03-26T05:56:07","version" => "5.52_01"},{"date" => "2002-03-31T03:55:52","version" => "5.54_01"},{"date" => "2002-04-05T05:01:52","version" => "5.55_01"},{"date" => "2002-04-06T08:29:20","version" => "5.55_02"},{"date" => "2002-04-07T03:04:18","version" => "5.55_03"},{"date" => "2002-04-11T05:32:04","version" => "5.90_01"},{"date" => "2002-04-24T04:21:44","version" => "5.91_01"},{"date" => "2002-04-30T03:43:53","version" => "5.92_01"},{"date" => "2002-05-06T06:02:08","version" => "5.93_01"},{"date" => "2002-05-17T19:04:41","version" => "5.94_01"},{"date" => "2002-05-17T21:24:13","version" => "5.94_02"},{"date" => "2002-05-18T18:43:02","version" => "5.95_01"},{"date" => "2002-05-23T21:01:02","version" => "5.96_01"},{"date" => "2002-05-26T01:25:25","version" => "6.00"},{"date" => "2002-05-30T19:02:20","version" => "6.01"},{"date" => "2002-06-16T05:41:28","version" => "6.02"},{"date" => "2002-06-19T21:24:32","version" => "6.03"},{"date" => "2002-08-27T01:42:36","version" => "6.04"},{"date" => "2002-08-27T23:24:30","version" => "6.05"},{"date" => "2002-12-19T08:42:01","version" => "6.06_01"},{"date" => "2002-12-24T04:54:53","version" => "6.06_02"},{"date" => "2003-03-30T03:49:59","version" => "6.06_03"},{"date" => "2003-03-31T04:37:55","version" => "6.06_04"},{"date" => "2003-03-31T10:50:00","version" => "6.06_05"},{"date" => "2003-04-07T02:46:10","version" => "6.10_01"},{"date" => "2003-04-07T08:33:23","version" => "6.10_02"},{"date" => "2003-04-11T07:27:36","version" => "6.10_03"},{"date" => "2003-05-23T09:05:27","version" => "6.10_04"},{"date" => "2003-06-07T01:32:29","version" => "6.10_05"},{"date" => "2003-06-07T08:00:14","version" => "6.10_06"},{"date" => "2003-07-05T23:40:34","version" => "6.10_07"},{"date" => "2003-07-22T01:23:46","version" => "6.10_08"},{"date" => "2003-07-28T04:00:19","version" => "6.11"},{"date" => "2003-07-30T05:28:47","version" => "6.12"},{"date" => "2003-07-31T23:51:40","version" => "6.13"},{"date" => "2003-08-03T23:27:51","version" => "6.14"},{"date" => "2003-08-03T23:46:11","version" => "6.15"},{"date" => "2003-08-18T08:43:08","version" => "6.16"},{"date" => "2003-09-15T22:23:01","version" => "6.17"},{"date" => "2003-11-04T04:12:53","version" => "6.18"},{"date" => "2003-11-04T07:03:30","version" => "6.19"},{"date" => "2003-11-06T10:37:47","version" => "6.20"},{"date" => "2003-11-11T08:26:17","version" => "6.21"},{"date" => "2004-04-03T21:33:45","version" => "6.21_03"},{"date" => "2004-11-24T04:06:20","version" => "6.22"},{"date" => "2004-11-26T21:15:45","version" => "6.23"},{"date" => "2004-11-30T20:42:14","version" => "6.24"},{"date" => "2004-12-09T06:00:53","version" => "6.24_01"},{"date" => "2004-12-15T12:05:50","version" => "6.25"},{"date" => "2004-12-18T02:34:56","version" => "6.25_01"},{"date" => "2004-12-20T08:36:56","version" => "6.25_02"},{"date" => "2004-12-21T04:17:27","version" => "6.25_03"},{"date" => "2004-12-21T05:58:10","version" => "6.25_04"},{"date" => "2004-12-22T13:05:53","version" => "6.25_05"},{"date" => "2004-12-26T22:26:26","version" => "6.25_06"},{"date" => "2004-12-31T08:53:31","version" => "6.25_07"},{"date" => "2005-02-08T14:21:17","version" => "6.25_08"},{"date" => "2005-03-12T18:29:26","version" => "6.25_09"},{"date" => "2005-03-14T00:17:26","version" => "6.25_10"},{"date" => "2005-03-15T10:05:07","version" => "6.25_11"},{"date" => "2005-03-19T00:19:47","version" => "6.25_12"},{"date" => "2005-03-22T22:50:34","version" => "6.26"},{"date" => "2005-03-29T05:48:40","version" => "6.26_01"},{"date" => "2005-04-04T23:55:46","version" => "6.27"},{"date" => "2005-04-12T23:23:53","version" => "6.28"},{"date" => "2005-05-19T21:22:00","version" => "6.29"},{"date" => "2005-05-20T23:14:45","version" => "6.30"},{"date" => "2005-08-17T06:59:11","version" => "6.30_01"},{"date" => "2006-09-01T19:07:28","version" => "6.30_02"},{"date" => "2006-09-01T21:06:57","version" => "6.30_03"},{"date" => "2006-09-11T20:20:27","version" => "6.30_04"},{"date" => "2006-10-10T01:04:44","version" => "6.31"},{"date" => "2007-02-21T16:02:09","version" => "6.32"},{"date" => "2007-06-29T22:18:15","version" => "6.33"},{"date" => "2007-06-30T16:10:15","version" => "6.34"},{"date" => "2007-07-02T03:56:25","version" => "6.35"},{"date" => "2007-07-03T08:10:57","version" => "6.36"},{"date" => "2007-11-26T01:10:14","version" => "6.37_01"},{"date" => "2007-11-26T07:35:50","version" => "6.37_02"},{"date" => "2007-11-26T22:18:55","version" => "6.37_03"},{"date" => "2007-11-29T00:04:35","version" => "6.38"},{"date" => "2007-12-06T11:08:15","version" => "6.40"},{"date" => "2007-12-08T01:02:26","version" => "6.42"},{"date" => "2008-01-02T00:09:23","version" => "6.43_01"},{"date" => "2008-02-29T00:08:42","version" => "6.44"},{"date" => "2008-09-06T10:22:44","version" => "6.45_01"},{"date" => "2008-09-07T21:18:05","version" => "6.45_02"},{"date" => "2008-09-27T21:37:54","version" => "6.46"},{"date" => "2008-10-14T16:41:49","version" => "6.47_01"},{"date" => "2008-10-16T23:18:52","version" => "6.47_02"},{"date" => "2008-10-20T18:20:40","version" => "6.48"},{"date" => "2009-02-20T01:11:08","version" => "6.49_01"},{"date" => "2009-03-22T19:30:00","version" => "6.50"},{"date" => "2009-04-10T21:33:29","version" => "6.51_01"},{"date" => "2009-04-14T04:22:58","version" => "6.51_02"},{"date" => "2009-05-24T05:41:35","version" => "6.51_03"},{"date" => "2009-05-24T21:07:28","version" => "6.51_04"},{"date" => "2009-05-30T18:41:35","version" => "6.52"},{"date" => "2009-06-08T02:05:24","version" => "6.53_01"},{"date" => "2009-06-08T02:28:24","version" => "6.53_02"},{"date" => "2009-07-02T21:55:25","version" => "6.53_03"},{"date" => "2009-07-07T23:53:09","version" => "6.54"},{"date" => "2009-07-14T23:02:39","version" => "6.55_01"},{"date" => "2009-08-05T07:40:59","version" => "6.55_02"},{"date" => "2009-12-05T07:09:23","version" => "6.55_03"},{"date" => "2009-12-17T22:06:47","version" => "6.56"},{"date" => "2010-08-24T08:38:36","version" => "6.57_01"},{"date" => "2010-09-07T23:43:49","version" => "6.57_02"},{"date" => "2010-09-08T22:33:36","version" => "6.57_03"},{"date" => "2010-09-09T23:52:37","version" => "6.57_04"},{"date" => "2010-09-11T20:25:23","version" => "6.57_05"},{"date" => "2010-10-06T10:53:43","version" => "6.57_06"},{"date" => "2011-03-25T03:41:39","version" => "6.57_07"},{"date" => "2011-03-27T11:00:41","version" => "6.57_08"},{"date" => "2011-03-28T00:15:59","version" => "6.57_09"},{"date" => "2011-04-04T05:33:46","version" => "6.57_10"},{"date" => "2011-05-20T00:34:23","version" => "6.57_11"},{"date" => "2011-07-06T21:22:27","version" => "6.58"},{"date" => "2011-08-03T20:25:34","version" => "6.58_01"},{"date" => "2011-08-05T13:07:58","version" => "6.59"},{"date" => "2011-09-25T05:23:43","version" => "6.61_01"},{"date" => "2011-10-23T23:48:06","version" => "6.62"},{"date" => "2011-10-24T00:40:49","version" => "6.63_01"},{"date" => "2011-11-02T00:07:43","version" => "6.63_02"},{"date" => "2012-11-02T03:58:40","version" => "6.63_03"},{"date" => "2012-11-22T21:25:35","version" => "6.63_04"},{"date" => "2012-12-17T02:35:20","version" => "6.64"},{"date" => "2013-03-18T23:21:28","version" => "6.65_01"},{"date" => "2013-04-14T09:59:15","version" => "6.65_02"},{"date" => "2013-04-15T12:50:31","version" => "6.65_03"},{"date" => "2013-04-19T17:52:08","version" => "6.66"},{"date" => "2013-04-25T20:08:31","version" => "6.67_01"},{"date" => "2013-06-02T17:31:16","version" => "6.67_02"},{"date" => "2013-06-05T21:09:00","version" => "6.67_03"},{"date" => "2013-06-10T19:25:22","version" => "6.67_04"},{"date" => "2013-06-13T20:55:25","version" => "6.67_05"},{"date" => "2013-06-14T22:35:24","version" => "6.68"},{"date" => "2013-06-20T12:00:00","version" => "6.69_01"},{"date" => "2013-07-02T12:16:23","version" => "6.69_02"},{"date" => "2013-07-09T21:47:07","version" => "6.69_03"},{"date" => "2013-07-10T10:50:08","version" => "6.69_04"},{"date" => "2013-07-11T21:20:53","version" => "6.69_05"},{"date" => "2013-07-12T13:51:50","version" => "6.69_06"},{"date" => "2013-07-16T14:34:32","version" => "6.69_07"},{"date" => "2013-07-16T23:40:44","version" => "6.69_08"},{"date" => "2013-07-21T08:26:44","version" => "6.69_09"},{"date" => "2013-07-23T21:42:47","version" => "6.70"},{"date" => "2013-07-24T08:33:58","version" => "6.71_01"},{"date" => "2013-07-24T17:42:20","version" => "6.72"},{"date" => "2013-07-24T22:53:41","version" => "6.73_01"},{"date" => "2013-07-26T12:34:19","version" => "6.73_02"},{"date" => "2013-07-30T21:12:02","version" => "6.73_03"},{"date" => "2013-08-01T21:41:12","version" => "6.73_04"},{"date" => "2013-08-05T16:45:38","version" => "6.73_05"},{"date" => "2013-08-05T23:52:18","version" => "6.73_06"},{"date" => "2013-08-07T15:09:12","version" => "6.73_07"},{"date" => "2013-08-09T18:52:24","version" => "6.73_08"},{"date" => "2013-08-09T19:00:18","version" => "6.73_09"},{"date" => "2013-08-16T15:43:35","version" => "6.73_10"},{"date" => "2013-08-17T21:57:55","version" => "6.73_11"},{"date" => "2013-08-23T09:52:43","version" => "6.73_12"},{"date" => "2013-08-27T11:45:55","version" => "6.74"},{"date" => "2013-08-29T14:09:22","version" => "6.75_01"},{"date" => "2013-09-01T20:52:29","version" => "6.75_02"},{"date" => "2013-09-02T23:26:56","version" => "6.75_03"},{"date" => "2013-09-05T11:10:20","version" => "6.75_04"},{"date" => "2013-09-06T12:40:59","version" => "6.76"},{"date" => "2013-09-10T14:22:45","version" => "6.77_01"},{"date" => "2013-09-12T20:23:49","version" => "6.77_02"},{"date" => "2013-09-16T11:23:59","version" => "6.77_03"},{"date" => "2013-09-18T18:25:33","version" => "6.77_04"},{"date" => "2013-09-19T13:12:32","version" => "6.77_05"},{"date" => "2013-09-19T14:43:24","version" => "6.77_06"},{"date" => "2013-09-21T08:48:44","version" => "6.77_07"},{"date" => "2013-09-22T17:46:50","version" => "6.77_08"},{"date" => "2013-09-23T12:47:39","version" => "6.78"},{"date" => "2013-10-01T14:01:33","version" => "6.79_01"},{"date" => "2013-10-11T12:01:23","version" => "6.79_02"},{"date" => "2013-10-11T13:00:29","version" => "6.79_03"},{"date" => "2013-10-11T17:59:30","version" => "6.79_04"},{"date" => "2013-10-15T15:08:06","version" => "6.80"},{"date" => "2013-10-16T08:04:29","version" => "6.81_01"},{"date" => "2013-10-17T11:24:19","version" => "6.81_02"},{"date" => "2013-10-24T19:54:34","version" => "6.81_03"},{"date" => "2013-11-01T19:56:13","version" => "6.81_04"},{"date" => "2013-11-02T21:44:06","version" => "6.81_05"},{"date" => "2013-11-04T19:24:38","version" => "6.82"},{"date" => "2013-11-05T11:45:54","version" => "6.83_01"},{"date" => "2013-11-12T11:15:21","version" => "6.83_02"},{"date" => "2013-11-15T09:49:39","version" => "6.83_03"},{"date" => "2013-11-17T11:44:01","version" => "6.83_04"},{"date" => "2013-11-25T22:52:46","version" => "6.83_05"},{"date" => "2013-11-29T21:55:40","version" => "6.83_06"},{"date" => "2013-11-30T15:27:01","version" => "6.84"},{"date" => "2013-12-16T13:18:35","version" => "6.85_01"},{"date" => "2013-12-17T10:17:50","version" => "6.85_02"},{"date" => "2013-12-23T14:59:36","version" => "6.85_03"},{"date" => "2013-12-23T15:02:38","version" => "6.85_04"},{"date" => "2013-12-29T11:28:14","version" => "6.85_05"},{"date" => "2013-12-30T23:18:09","version" => "6.85_06"},{"date" => "2014-01-01T19:00:36","version" => "6.85_07"},{"date" => "2014-01-04T12:21:05","version" => "6.86"},{"date" => "2014-01-12T10:34:38","version" => "6.87_01"},{"date" => "2014-01-18T13:30:15","version" => "6.87_02"},{"date" => "2014-01-19T17:53:19","version" => "6.87_03"},{"date" => "2014-01-26T19:33:34","version" => "6.87_04"},{"date" => "2014-01-28T14:00:44","version" => "6.87_05"},{"date" => "2014-01-31T20:59:13","version" => "6.88"},{"date" => "2014-02-17T16:23:55","version" => "6.89_01"},{"date" => "2014-02-20T20:49:24","version" => "6.90"},{"date" => "2014-03-06T13:52:24","version" => "6.91_01"},{"date" => "2014-03-13T16:34:37","version" => "6.92"},{"date" => "2014-03-24T16:57:01","version" => "6.93_01"},{"date" => "2014-03-25T20:38:21","version" => "6.94"},{"date" => "2014-04-02T20:52:53","version" => "6.95_01"},{"date" => "2014-04-07T14:29:26","version" => "6.95_02"},{"date" => "2014-04-11T21:09:21","version" => "6.96"},{"date" => "2014-04-24T13:29:12","version" => "6.97_01"},{"date" => "2014-04-28T10:55:44","version" => "6.97_02"},{"date" => "2014-04-29T20:41:00","version" => "6.98"},{"date" => "2014-06-03T21:19:42","version" => "6.99_01"},{"date" => "2014-06-05T11:18:25","version" => "6.99_02"},{"date" => "2014-07-04T10:15:23","version" => "6.99_03"},{"date" => "2014-07-12T11:54:35","version" => "6.99_04"},{"date" => "2014-07-22T11:42:12","version" => "6.99_05"},{"date" => "2014-07-28T14:07:14","version" => "6.99_06"},{"date" => "2014-07-30T16:44:02","version" => "6.99_07"},{"date" => "2014-08-18T13:19:18","version" => "6.99_08"},{"date" => "2014-08-28T10:13:30","version" => "6.99_09"},{"date" => "2014-09-04T14:04:55","version" => "6.99_10"},{"date" => "2014-09-08T13:39:46","version" => "6.99_11"},{"date" => "2014-09-11T14:32:19","version" => "6.99_12"},{"date" => "2014-09-15T19:11:34","version" => "6.99_13"},{"date" => "2014-09-19T14:06:14","version" => "6.99_14"},{"date" => "2014-09-21T12:23:58","version" => "6.99_15"},{"date" => "2014-10-02T18:50:08","version" => "6.99_16"},{"date" => "2014-10-12T18:41:24","version" => "6.99_17"},{"date" => "2014-10-20T09:14:39","version" => "6.99_18"},{"date" => "2014-10-22T19:48:56","version" => "7.00"},{"date" => "2014-10-25T12:49:55","version" => "7.01_01"},{"date" => "2014-10-25T16:49:40","version" => "7.01_02"},{"date" => "2014-10-30T19:48:04","version" => "7.01_03"},{"date" => "2014-10-31T10:13:56","version" => "7.01_04"},{"date" => "2014-11-03T12:53:43","version" => "7.01_05"},{"date" => "2014-11-03T20:55:23","version" => "7.01_06"},{"date" => "2014-11-04T19:40:07","version" => "7.01_07"},{"date" => "2014-11-04T20:29:00","version" => "7.01_08"},{"date" => "2014-11-06T21:59:55","version" => "7.01_09"},{"date" => "2014-11-08T10:39:16","version" => "7.02"},{"date" => "2014-11-18T21:47:11","version" => "7.03_01"},{"date" => "2014-11-24T13:26:46","version" => "7.03_02"},{"date" => "2014-11-25T16:43:06","version" => "7.03_03"},{"date" => "2014-11-27T14:42:51","version" => "7.03_04"},{"date" => "2014-11-28T18:32:48","version" => "7.03_05"},{"date" => "2014-12-01T15:37:46","version" => "7.03_06"},{"date" => "2014-12-02T12:56:02","version" => "7.04"},{"date" => "2014-12-06T16:58:07","version" => "7.05_01"},{"date" => "2014-12-15T20:13:08","version" => "7.05_02"},{"date" => "2014-12-24T12:12:00","version" => "7.05_03"},{"date" => "2014-12-24T14:49:46","version" => "7.05_04"},{"date" => "2014-12-31T23:21:05","version" => "7.05_05"},{"date" => "2015-01-08T19:09:29","version" => "7.05_06"},{"date" => "2015-01-09T16:23:43","version" => "7.05_07"},{"date" => "2015-01-20T10:13:21","version" => "7.05_08"},{"date" => "2015-01-23T10:51:30","version" => "7.05_09"},{"date" => "2015-01-26T15:19:01","version" => "7.05_10"},{"date" => "2015-01-31T16:40:19","version" => "7.05_11"},{"date" => "2015-02-07T15:19:11","version" => "7.05_12"},{"date" => "2015-02-18T22:49:29","version" => "7.05_13"},{"date" => "2015-02-20T17:32:55","version" => "7.05_14"},{"date" => "2015-03-05T19:44:02","version" => "7.05_15"},{"date" => "2015-03-09T11:35:12","version" => "7.05_16"},{"date" => "2015-03-24T12:27:52","version" => "7.05_17"},{"date" => "2015-03-27T12:20:03","version" => "7.05_18"},{"date" => "2015-03-27T16:59:34","version" => "7.05_19"},{"date" => "2015-04-04T15:53:36","version" => "7.05_20"},{"date" => "2015-06-13T14:19:26","version" => "7.05_21"},{"date" => "2015-06-14T13:44:56","version" => "7.05_22"},{"date" => "2015-06-24T19:51:24","version" => "7.05_23"},{"date" => "2015-07-01T18:30:38","version" => "7.05_24"},{"date" => "2015-07-07T17:18:36","version" => "7.05_25"},{"date" => "2015-08-04T19:41:25","version" => "7.05_26"},{"date" => "2015-08-05T09:35:40","version" => "7.05_27"},{"date" => "2015-08-19T18:10:20","version" => "7.05_28"},{"date" => "2015-08-24T15:26:22","version" => "7.05_29"},{"date" => "2015-08-31T18:06:48","version" => "7.06"},{"date" => "2015-09-02T11:55:33","version" => "7.07_01"},{"date" => "2015-09-08T19:59:05","version" => "7.08"},{"date" => "2015-09-10T18:55:41","version" => "7.10"},{"date" => "2015-11-12T12:35:03","version" => "7.11_01"},{"date" => "2015-11-21T20:23:22","version" => "7.11_02"},{"date" => "2015-11-25T15:40:06","version" => "7.11_03"},{"date" => "2016-02-15T11:40:55","version" => "7.11_04"},{"date" => "2016-03-19T10:07:11","version" => "7.11_05"},{"date" => "2016-03-29T18:44:47","version" => "7.11_06"},{"date" => "2016-04-19T11:41:10","version" => "7.12"},{"date" => "2016-04-23T16:35:56","version" => "7.13_01"},{"date" => "2016-04-24T13:20:40","version" => "7.14"},{"date" => "2016-04-27T18:27:25","version" => "7.15_01"},{"date" => "2016-04-28T12:15:28","version" => "7.15_02"},{"date" => "2016-05-01T13:29:10","version" => "7.15_03"},{"date" => "2016-05-07T10:28:49","version" => "7.16"},{"date" => "2016-05-09T19:14:54","version" => "7.17_01"},{"date" => "2016-05-09T23:07:33","version" => "7.17_02"},{"date" => "2016-05-11T18:22:21","version" => "7.17_03"},{"date" => "2016-05-23T15:39:08","version" => "7.18"},{"date" => "2016-06-02T14:01:28","version" => "7.19_01"},{"date" => "2016-06-13T09:11:52","version" => "7.19_02"},{"date" => "2016-06-13T13:44:33","version" => "7.19_03"},{"date" => "2016-06-14T11:35:43","version" => "7.19_04"},{"date" => "2016-06-20T14:40:57","version" => "7.19_05"},{"date" => "2016-06-27T12:04:29","version" => "7.19_06"},{"date" => "2016-07-03T14:30:23","version" => "7.19_07"},{"date" => "2016-07-28T12:26:56","version" => "7.19_08"},{"date" => "2016-08-05T08:57:09","version" => "7.20"},{"date" => "2016-08-07T09:54:04","version" => "7.21_01"},{"date" => "2016-08-08T08:42:10","version" => "7.22"},{"date" => "2016-08-19T09:24:06","version" => "7.23_01"},{"date" => "2016-08-20T12:35:27","version" => "7.24"},{"date" => "2017-02-03T15:21:22","version" => "7.25_01"},{"date" => "2017-05-11T11:19:49","version" => "7.25_02"},{"date" => "2017-05-11T17:09:16","version" => "7.25_03"},{"date" => "2017-05-12T12:25:54","version" => "7.25_04"},{"date" => "2017-05-15T09:41:49","version" => "7.25_05"},{"date" => "2017-05-23T19:31:28","version" => "7.25_06"},{"date" => "2017-05-27T20:21:06","version" => "7.26"},{"date" => "2017-05-28T10:50:55","version" => "7.27_01"},{"date" => "2017-05-30T08:56:32","version" => "7.27_02"},{"date" => "2017-05-30T21:26:23","version" => "7.28"},{"date" => "2017-05-31T08:32:44","version" => "7.29_01"},{"date" => "2017-06-11T11:17:55","version" => "7.29_02"},{"date" => "2017-06-12T12:31:08","version" => "7.30"},{"date" => "2017-06-14T15:10:23","version" => "7.31_01"},{"date" => "2017-06-26T13:14:10","version" => "7.31_02"},{"date" => "2017-07-10T09:02:35","version" => "7.31_03"},{"date" => "2017-10-05T12:19:00","version" => "7.31_04"},{"date" => "2017-11-25T09:37:04","version" => "7.31_05"},{"date" => "2018-01-16T13:28:46","version" => "7.31_06"},{"date" => "2018-01-16T16:24:23","version" => "7.31_07"},{"date" => "2018-02-12T12:32:45","version" => "7.31_08"},{"date" => "2018-02-16T20:25:44","version" => "7.32"},{"date" => "2018-02-20T10:44:19","version" => "7.33_01"},{"date" => "2018-02-24T14:05:00","version" => "7.33_02"},{"date" => "2018-02-24T20:21:42","version" => "7.33_03"},{"date" => "2018-03-19T10:51:54","version" => "7.34"},{"date" => "2018-04-19T12:46:01","version" => "7.35_01"},{"date" => "2018-04-24T11:01:35","version" => "7.35_02"},{"date" => "2018-04-27T13:59:23","version" => "7.35_03"},{"date" => "2018-07-09T09:50:43","version" => "7.35_04"},{"date" => "2018-07-10T09:18:31","version" => "7.35_05"},{"date" => "2018-07-19T19:49:08","version" => "7.35_06"},{"date" => "2018-11-23T11:59:44","version" => "7.35_07"},{"date" => "2018-12-06T10:56:33","version" => "7.35_08"},{"date" => "2019-02-18T10:27:00","version" => "7.35_09"},{"date" => "2019-02-20T10:06:48","version" => "7.35_10"},{"date" => "2019-04-25T11:10:29","version" => "7.35_11"},{"date" => "2019-04-27T22:17:58","version" => "7.35_12"},{"date" => "2019-04-28T11:23:25","version" => "7.35_13"},{"date" => "2019-04-28T13:15:57","version" => "7.35_14"},{"date" => "2019-04-28T15:48:41","version" => "7.36"},{"date" => "2019-06-07T10:55:49","version" => "7.37_01"},{"date" => "2019-06-27T10:35:57","version" => "7.37_02"},{"date" => "2019-08-03T12:27:47","version" => "7.37_03"},{"date" => "2019-08-22T14:34:47","version" => "7.37_04"},{"date" => "2019-09-11T09:16:48","version" => "7.38"},{"date" => "2019-09-16T06:54:51","version" => "7.39_01"},{"date" => "2019-11-07T10:03:13","version" => "7.39_02"},{"date" => "2019-11-17T20:12:14","version" => "7.39_03"},{"date" => "2019-11-18T15:20:20","version" => "7.39_04"},{"date" => "2019-11-21T12:10:17","version" => "7.39_05"},{"date" => "2019-12-16T20:02:27","version" => "7.40"},{"date" => "2019-12-16T21:53:56","version" => "7.41_01"},{"date" => "2019-12-17T22:30:33","version" => "7.42"},{"date" => "2020-01-05T13:00:40","version" => "7.43_01"},{"date" => "2020-01-14T16:54:08","version" => "7.44"},{"date" => "2020-05-28T16:58:08","version" => "7.45_01"},{"date" => "2020-06-23T10:14:10","version" => "7.46"},{"date" => "2020-06-26T10:13:17","version" => "7.47_01"},{"date" => "2020-07-07T07:38:50","version" => "7.47_02"},{"date" => "2020-07-08T21:54:35","version" => "7.47_03"},{"date" => "2020-07-28T19:00:26","version" => "7.47_04"},{"date" => "2020-07-31T09:57:33","version" => "7.47_05"},{"date" => "2020-08-01T13:53:05","version" => "7.47_06"},{"date" => "2020-08-03T21:39:02","version" => "7.47_07"},{"date" => "2020-08-31T09:02:22","version" => "7.47_08"},{"date" => "2020-09-14T13:50:45","version" => "7.47_09"},{"date" => "2020-09-15T18:45:02","version" => "7.47_10"},{"date" => "2020-09-20T09:20:24","version" => "7.47_11"},{"date" => "2020-09-30T15:40:12","version" => "7.47_12"},{"date" => "2020-10-04T10:56:39","version" => "7.48"},{"date" => "2020-10-06T17:29:16","version" => "7.49_01"},{"date" => "2020-10-08T12:03:50","version" => "7.49_02"},{"date" => "2020-10-09T20:46:22","version" => "7.49_03"},{"date" => "2020-10-13T18:34:34","version" => "7.49_04"},{"date" => "2020-10-21T18:14:52","version" => "7.50"},{"date" => "2020-11-04T00:05:13","version" => "7.51_01"},{"date" => "2020-11-04T19:51:52","version" => "7.52"},{"date" => "2020-11-10T03:50:49","version" => "7.53_01"},{"date" => "2020-11-12T19:50:41","version" => "7.54"},{"date" => "2020-11-18T18:25:16","version" => "7.55_01"},{"date" => "2020-11-19T20:00:09","version" => "7.56"},{"date" => "2020-12-18T13:45:54","version" => "7.57_01"},{"date" => "2020-12-18T23:07:45","version" => "7.57_02"},{"date" => "2020-12-21T18:31:44","version" => "7.58"},{"date" => "2021-02-02T10:13:35","version" => "7.59_01"},{"date" => "2021-02-17T11:05:23","version" => "7.60"},{"date" => "2021-03-21T15:00:35","version" => "7.61_01"},{"date" => "2021-04-13T18:13:28","version" => "7.62"},{"date" => "2021-05-25T18:00:03","version" => "7.63_01"},{"date" => "2021-06-03T19:05:10","version" => "7.63_02"},{"date" => "2021-06-22T13:53:51","version" => "7.63_03"},{"date" => "2021-06-30T14:30:46","version" => "7.63_04"},{"date" => "2021-08-14T08:19:32","version" => "7.63_05"},{"date" => "2021-11-03T01:44:47","version" => "7.63_06"},{"date" => "2021-11-27T11:51:29","version" => "7.63_07"},{"date" => "2021-11-27T17:31:21","version" => "7.63_08"},{"date" => "2021-12-08T22:35:25","version" => "7.63_09"},{"date" => "2021-12-13T16:54:00","version" => "7.63_10"},{"date" => "2021-12-14T17:00:18","version" => "7.63_11"},{"date" => "2021-12-17T19:24:34","version" => "7.64"},{"date" => "2022-05-30T10:07:14","version" => "7.65_01"},{"date" => "2022-07-22T13:01:08","version" => "7.65_02"},{"date" => "2022-12-24T00:32:29","version" => "7.65_03"},{"date" => "2022-12-25T09:06:33","version" => "7.66"},{"date" => "2023-03-01T13:47:08","version" => "7.67_01"},{"date" => "2023-03-06T11:17:11","version" => "7.67_02"},{"date" => "2023-03-14T21:41:23","version" => "7.68"},{"date" => "2023-03-25T11:45:00","version" => "7.69_01"},{"date" => "2023-03-26T13:29:08","version" => "7.70"},{"date" => "2024-06-24T19:34:30","version" => "7.71_01"},{"date" => "2024-11-22T19:08:50","version" => "7.71_02"},{"date" => "2025-02-19T01:40:18","version" => "7.71_03"},{"date" => "2025-02-24T15:29:06","version" => "7.71_04"},{"date" => "2025-02-28T18:43:37","version" => "7.71_05"},{"date" => "2025-03-03T16:59:13","version" => "7.71_06"},{"date" => "2025-03-05T21:46:33","version" => "7.71_07"},{"date" => "2025-03-08T23:59:14","version" => "7.71_08"},{"date" => "2025-03-14T11:11:41","version" => "7.72"},{"date" => "2025-03-30T10:57:25","version" => "7.73_01"},{"date" => "2025-04-09T12:39:45","version" => "7.74"},{"date" => "2025-05-23T14:13:25","version" => "7.75_01"},{"date" => "2025-05-23T19:17:36","version" => "7.76"},{"date" => "2025-07-28T18:05:55","version" => "7.77_01"},{"date" => "2025-08-20T11:28:18","version" => "7.77_02"},{"date" => "1994-10-17T00:00:00","dual_lived" => 1,"perl_release" => 5,"version" => undef},{"date" => "1996-02-29T00:00:00","dual_lived" => 1,"perl_release" => "5.002","version" => "5.21"},{"date" => "1996-10-10T00:00:00","dual_lived" => 1,"perl_release" => "5.00307","version" => "5.38"},{"date" => "1997-05-15T00:00:00","dual_lived" => 1,"perl_release" => "5.004","version" => "5.4002"},{"date" => "1999-04-29T00:00:00","dual_lived" => 1,"perl_release" => "5.00405","version" => "5.42"},{"date" => "1998-07-22T00:00:00","dual_lived" => 1,"perl_release" => "5.005","version" => "5.4301"},{"date" => "1999-03-28T00:00:00","dual_lived" => 1,"perl_release" => "5.00503","version" => "5.4302"},{"date" => "2000-03-22T00:00:00","dual_lived" => 1,"perl_release" => "5.006000","version" => "5.45"},{"date" => "2004-03-16T00:00:00","dual_lived" => 1,"perl_release" => "5.009001","version" => "6.21_02"},{"date" => "2010-04-20T00:00:00","dual_lived" => 1,"perl_release" => "5.013","version" => "6.5601"},{"date" => "2015-06-01T00:00:00","dual_lived" => 1,"perl_release" => "5.022","version" => "7.04_01"},{"date" => "2017-01-14T00:00:00","dual_lived" => 1,"perl_release" => "5.022003","version" => "7.04_02"},{"date" => "2016-02-20T00:00:00","dual_lived" => 1,"perl_release" => "5.023008","version" => "7.10_01"},{"date" => "2017-01-14T00:00:00","dual_lived" => 1,"perl_release" => "5.024001","version" => "7.10_02"}]},"ExtUtils-ParseXS" => {"advisories" => [{"affected_versions" => ["<3.35"],"cves" => ["CVE-2016-1238"],"description" => "(1) cpan/Archive-Tar/bin/ptar, (2) cpan/Archive-Tar/bin/ptardiff, (3) cpan/Archive-Tar/bin/ptargrep, (4) cpan/CPAN/scripts/cpan, (5) cpan/Digest-SHA/shasum, (6) cpan/Encode/bin/enc2xs, (7) cpan/Encode/bin/encguess, (8) cpan/Encode/bin/piconv, (9) cpan/Encode/bin/ucmlint, (10) cpan/Encode/bin/unidump, (11) cpan/ExtUtils-MakeMaker/bin/instmodsh, (12) cpan/IO-Compress/bin/zipdetails, (13) cpan/JSON-PP/bin/json_pp, (14) cpan/Test-Harness/bin/prove, (15) dist/ExtUtils-ParseXS/lib/ExtUtils/xsubpp, (16) dist/Module-CoreList/corelist, (17) ext/Pod-Html/bin/pod2html, (18) utils/c2ph.PL, (19) utils/h2ph.PL, (20) utils/h2xs.PL, (21) utils/libnetcfg.PL, (22) utils/perlbug.PL, (23) utils/perldoc.PL, (24) utils/perlivp.PL, and (25) utils/splain.PL in Perl 5.x before 5.22.3-RC2 and 5.24 before 5.24.1-RC2 do not properly remove . (period) characters from the end of the includes directory array, which might allow local users to gain privileges via a Trojan horse module under the current working directory.","distribution" => "ExtUtils-ParseXS","fixed_versions" => [">=3.35"],"id" => "CPANSA-ExtUtils-ParseXS-2016-1238","references" => ["http://lists.opensuse.org/opensuse-security-announce/2019-08/msg00002.html","http://perl5.git.perl.org/perl.git/commit/cee96d52c39b1e7b36e1c62d38bcd8d86e9a41ab","http://www.debian.org/security/2016/dsa-3628","http://www.nntp.perl.org/group/perl.perl5.porters/2016/07/msg238271.html","http://www.securityfocus.com/bid/92136","http://www.securitytracker.com/id/1036440","https://h20566.www2.hpe.com/portal/site/hpsc/public/kb/docDisplay?docId=emr_na-c05240731","https://lists.apache.org/thread.html/7f6a16bc0fd0fd5e67c7fd95bd655069a2ac7d1f88e42d3c853e601c%40%3Cannounce.apache.org%3E","https://lists.debian.org/debian-lts-announce/2018/11/msg00016.html","https://lists.fedoraproject.org/archives/list/package-announce%40lists.fedoraproject.org/message/2FBQOCV3GBAN2EYZUM3CFDJ4ECA3GZOK/","https://lists.fedoraproject.org/archives/list/package-announce%40lists.fedoraproject.org/message/DOFRQWJRP2NQJEYEWOMECVW3HAMD5SYN/","https://lists.fedoraproject.org/archives/list/package-announce%40lists.fedoraproject.org/message/TZBNQH3DMI7HDELJAZ4TFJJANHXOEDWH/","https://rt.perl.org/Public/Bug/Display.html?id=127834","https://security.gentoo.org/glsa/201701-75","https://security.gentoo.org/glsa/201812-07","http://lists.opensuse.org/opensuse-security-announce/2019-08/msg00002.html","http://perl5.git.perl.org/perl.git/commit/cee96d52c39b1e7b36e1c62d38bcd8d86e9a41ab","http://www.debian.org/security/2016/dsa-3628","http://www.nntp.perl.org/group/perl.perl5.porters/2016/07/msg238271.html","http://www.securityfocus.com/bid/92136","http://www.securitytracker.com/id/1036440","https://h20566.www2.hpe.com/portal/site/hpsc/public/kb/docDisplay?docId=emr_na-c05240731","https://lists.apache.org/thread.html/7f6a16bc0fd0fd5e67c7fd95bd655069a2ac7d1f88e42d3c853e601c%40%3Cannounce.apache.org%3E","https://lists.debian.org/debian-lts-announce/2018/11/msg00016.html","https://lists.fedoraproject.org/archives/list/package-announce%40lists.fedoraproject.org/message/2FBQOCV3GBAN2EYZUM3CFDJ4ECA3GZOK/","https://lists.fedoraproject.org/archives/list/package-announce%40lists.fedoraproject.org/message/DOFRQWJRP2NQJEYEWOMECVW3HAMD5SYN/","https://lists.fedoraproject.org/archives/list/package-announce%40lists.fedoraproject.org/message/TZBNQH3DMI7HDELJAZ4TFJJANHXOEDWH/","https://rt.perl.org/Public/Bug/Display.html?id=127834","https://security.gentoo.org/glsa/201701-75","https://security.gentoo.org/glsa/201812-07"],"reported" => "2016-08-02","severity" => "high"}],"main_module" => "ExtUtils::ParseXS","versions" => [{"date" => "2002-12-09T00:53:36","version" => "1.98_01"},{"date" => "2003-02-05T18:22:19","version" => "1.99"},{"date" => "2003-02-23T22:45:04","version" => "2.00"},{"date" => "2003-03-20T15:25:07","version" => "2.01"},{"date" => "2003-03-31T00:25:32","version" => "2.02"},{"date" => "2003-08-16T22:57:00","version" => "2.03"},{"date" => "2003-09-04T18:14:59","version" => "2.04"},{"date" => "2003-09-29T15:35:39","version" => "2.05"},{"date" => "2003-12-26T15:05:42","version" => "2.06"},{"date" => "2004-01-25T23:04:13","version" => "2.07"},{"date" => "2004-02-21T03:46:57","version" => "2.08"},{"date" => "2005-03-27T17:18:20","version" => "2.09"},{"date" => "2005-05-31T02:37:25","version" => "2.10"},{"date" => "2005-06-14T04:04:10","version" => "2.11"},{"date" => "2005-08-25T01:07:16","version" => "2.12"},{"date" => "2005-10-04T03:02:19","version" => "2.13"},{"date" => "2005-10-09T01:52:46","version" => "2.14"},{"date" => "2005-10-10T15:09:54","version" => "2.15"},{"date" => "2006-09-16T03:35:22","version" => "2.16"},{"date" => "2006-11-20T23:08:18","version" => "2.17"},{"date" => "2007-01-30T02:58:43","version" => "2.18"},{"date" => "2008-02-17T20:29:31","version" => "2.19"},{"date" => "2008-08-07T03:20:09","version" => "2.19_02"},{"date" => "2009-06-28T03:01:41","version" => "2.19_03"},{"date" => "2009-06-29T15:51:33","version" => "2.19_04"},{"date" => "2009-07-01T17:49:20","version" => "2.20"},{"date" => "2009-07-08T16:47:56","version" => "2.20_01"},{"date" => "2009-07-18T21:23:28","version" => "2.2002"},{"date" => "2009-07-24T03:16:46","version" => "2.20_03"},{"date" => "2009-08-10T15:44:42","version" => "2.20_04"},{"date" => "2009-08-23T01:50:17","version" => "2.20_05"},{"date" => "2009-09-15T02:36:48","version" => "2.200401"},{"date" => "2009-10-02T05:28:52","version" => "2.200402"},{"date" => "2009-10-02T06:06:00","version" => "2.200403"},{"date" => "2009-10-03T03:49:34","version" => "2.20_06"},{"date" => "2009-10-03T15:28:29","version" => "2.20_07"},{"date" => "2009-10-05T15:25:07","version" => "2.21"},{"date" => "2009-12-19T12:43:55","version" => "2.21_01"},{"date" => "2009-12-19T15:58:28","version" => "2.21_02"},{"date" => "2010-01-11T20:03:31","version" => "2.22"},{"date" => "2010-01-25T21:14:41","version" => "2.2201"},{"date" => "2010-01-27T20:07:51","version" => "2.2202"},{"date" => "2010-02-11T19:04:49","version" => "2.2203"},{"date" => "2010-03-10T19:27:43","version" => "2.2204"},{"date" => "2010-03-10T23:17:47","version" => "2.2205"},{"date" => "2010-07-04T19:53:47","version" => "2.2206"},{"date" => "2011-07-12T20:42:45","version" => "3.00_01"},{"date" => "2011-07-14T13:21:26","version" => "3.00_02"},{"date" => "2011-07-23T15:09:23","version" => "3.00_03"},{"date" => "2011-07-27T20:24:26","version" => "3.00_04"},{"date" => "2011-07-27T20:57:56","version" => "3.00_05"},{"date" => "2011-08-04T16:06:39","version" => "3.01"},{"date" => "2011-08-04T18:09:18","version" => "3.02"},{"date" => "2011-08-11T06:25:52","version" => "3.03"},{"date" => "2011-08-21T11:40:28","version" => "3.03_02"},{"date" => "2011-08-24T17:51:31","version" => "3.03_03"},{"date" => "2011-08-25T06:33:30","version" => "3.04"},{"date" => "2011-08-28T15:57:42","version" => "3.04_01"},{"date" => "2011-09-03T13:31:37","version" => "3.04_02"},{"date" => "2011-09-04T16:55:05","version" => "3.04_03"},{"date" => "2011-09-12T06:28:10","version" => "3.04_04"},{"date" => "2011-10-05T06:18:44","version" => "3.05"},{"date" => "2011-12-07T07:35:08","version" => "3.06"},{"date" => "2011-12-07T13:15:12","version" => "3.07"},{"date" => "2011-12-19T17:10:40","version" => "3.08"},{"date" => "2011-12-28T18:05:57","version" => "3.09"},{"date" => "2011-12-29T17:00:16","version" => "3.11"},{"date" => "2012-01-28T12:07:45","version" => "3.13_01"},{"date" => "2012-02-01T17:51:52","version" => "3.14"},{"date" => "2012-02-02T07:15:27","version" => "3.15"},{"date" => "2012-11-19T06:42:48","version" => "3.18"},{"date" => "2013-04-11T18:19:45","version" => "3.18_01"},{"date" => "2013-04-15T05:41:18","version" => "3.18_02"},{"date" => "2013-04-19T16:47:41","version" => "3.18_03"},{"date" => "2013-06-20T15:51:15","version" => "3.18_04"},{"date" => "2013-08-09T17:14:04","version" => "3.21"},{"date" => "2013-08-29T17:31:29","version" => "3.22"},{"date" => "2014-03-07T09:35:16","version" => "3.24"},{"date" => "2015-08-10T08:49:21","version" => "3.29_01"},{"date" => "2015-08-31T08:44:00","version" => "3.30"},{"date" => "2017-07-31T15:52:17","version" => "3.35"},{"date" => "2017-12-18T12:31:00","version" => "3.36_03"},{"date" => "2021-04-17T17:48:59","version" => "3.43_02"},{"date" => "2022-01-06T23:02:34","version" => "3.44"},{"date" => "2023-09-02T13:28:52","version" => "3.51"},{"date" => "2025-05-02T15:03:49","version" => "3.52"},{"date" => "2025-05-02T15:06:38","version" => "3.53"},{"date" => "2025-05-02T15:17:11","version" => "3.54"},{"date" => "2025-05-02T15:38:05","version" => "3.55"},{"date" => "2025-05-02T15:40:54","version" => "3.56"},{"date" => "2025-05-02T15:45:00","version" => "3.57"},{"date" => "2025-07-20T19:24:38","version" => "3.58"},{"date" => "2025-09-05T13:37:50","version" => "3.59"},{"date" => "2025-09-26T22:20:43","version" => "3.60"},{"date" => "2026-01-09T17:11:34","version" => "3.61"},{"date" => "2006-01-28T00:00:00","dual_lived" => 1,"perl_release" => "5.009003","version" => "2.15_02"},{"date" => "2007-12-18T00:00:00","dual_lived" => 1,"perl_release" => "5.010000","version" => "2.18_02"},{"date" => "2010-08-20T00:00:00","dual_lived" => 1,"perl_release" => "5.013004","version" => "2.2207"},{"date" => "2010-12-20T00:00:00","dual_lived" => 1,"perl_release" => "5.013008","version" => "2.2208"},{"date" => "2011-01-20T00:00:00","dual_lived" => 1,"perl_release" => "5.013009","version" => "2.2209"},{"date" => "2011-05-14T00:00:00","dual_lived" => 1,"perl_release" => "5.014000","version" => "2.2210"},{"date" => "2011-08-20T00:00:00","dual_lived" => 1,"perl_release" => "5.015002","version" => "3.03_01"},{"date" => "2012-01-20T00:00:00","dual_lived" => 1,"perl_release" => "5.015007","version" => "3.12"},{"date" => "2012-02-20T00:00:00","dual_lived" => 1,"perl_release" => "5.015008","version" => "3.16"},{"date" => "2012-05-26T00:00:00","dual_lived" => 1,"perl_release" => "5.017","version" => "3.17"},{"date" => "2013-06-21T00:00:00","dual_lived" => 1,"perl_release" => "5.019001","version" => "3.19"},{"date" => "2013-09-20T00:00:00","dual_lived" => 1,"perl_release" => "5.019004","version" => "3.23"},{"date" => "2014-06-20T00:00:00","dual_lived" => 1,"perl_release" => "5.021001","version" => "3.25"},{"date" => "2014-11-20T00:00:00","dual_lived" => 1,"perl_release" => "5.021006","version" => "3.26"},{"date" => "2014-12-20T00:00:00","dual_lived" => 1,"perl_release" => "5.021007","version" => "3.27"},{"date" => "2015-04-20T00:00:00","dual_lived" => 1,"perl_release" => "5.021011","version" => "3.28"},{"date" => "2015-08-20T00:00:00","dual_lived" => 1,"perl_release" => "5.023002","version" => "3.29"},{"date" => "2016-01-20T00:00:00","dual_lived" => 1,"perl_release" => "5.023007","version" => "3.31"},{"date" => "2016-06-20T00:00:00","dual_lived" => 1,"perl_release" => "5.025002","version" => "3.32"},{"date" => "2016-07-20T00:00:00","dual_lived" => 1,"perl_release" => "5.025003","version" => "3.33"},{"date" => "2017-03-20T00:00:00","dual_lived" => 1,"perl_release" => "5.025011","version" => "3.34"},{"date" => "2017-11-20T00:00:00","dual_lived" => 1,"perl_release" => "5.027006","version" => "3.36"},{"date" => "2018-02-20T00:00:00","dual_lived" => 1,"perl_release" => "5.027009","version" => "3.38"},{"date" => "2018-03-20T00:00:00","dual_lived" => 1,"perl_release" => "5.02701","version" => "3.39"},{"date" => "2018-12-18T00:00:00","dual_lived" => 1,"perl_release" => "5.029006","version" => "3.40"},{"date" => "2020-08-20T00:00:00","dual_lived" => 1,"perl_release" => "5.033001","version" => "3.41"},{"date" => "2020-12-20T00:00:00","dual_lived" => 1,"perl_release" => "5.033005","version" => "3.42"},{"date" => "2021-04-20T00:00:00","dual_lived" => 1,"perl_release" => "5.033009","version" => "3.43"},{"date" => "2022-03-20T00:00:00","dual_lived" => 1,"perl_release" => "5.03501","version" => "3.45"},{"date" => "2022-09-20T00:00:00","dual_lived" => 1,"perl_release" => "5.037004","version" => "3.46"},{"date" => "2022-11-20T00:00:00","dual_lived" => 1,"perl_release" => "5.037006","version" => "3.48"},{"date" => "2022-12-20T00:00:00","dual_lived" => 1,"perl_release" => "5.037007","version" => "3.49"},{"date" => "2023-04-20T00:00:00","dual_lived" => 1,"perl_release" => "5.037011","version" => "3.50"}]},"FCGI" => {"advisories" => [{"affected_versions" => ["<0.74"],"cves" => ["CVE-2011-2766"],"description" => "Leaking information across requests when using the deprecated and undocumented old FCGI interface.\n","distribution" => "FCGI","fixed_versions" => [">=0.74"],"id" => "CPANSA-FCGI-2011-01","references" => ["https://metacpan.org/changes/distribution/FCGI","https://github.com/perl-catalyst/FCGI/commit/297693dc8362d25bb25e473899c72508a0f71d2e"],"reported" => "2011-09-24"},{"affected_versions" => [">=0.44"],"cves" => ["CVE-2025-40907"],"description" => "FCGI versions 0.44 through 0.82, for Perl, include a vulnerable version of the FastCGI fcgi2 (aka fcgi) library.  The included FastCGI library is affected by  CVE-2025-23016, causing an integer overflow (and resultant heap-based buffer overflow) via crafted nameLen or valueLen values in data to the IPC socket. This occurs in ReadParams in fcgiapp.c.","distribution" => "FCGI","fixed_versions" => [],"id" => "CPANSA-FCGI-2025-40907","references" => ["http://www.openwall.com/lists/oss-security/2025/04/23/4","https://github.com/FastCGI-Archives/fcgi2/issues/67","https://github.com/FastCGI-Archives/fcgi2/releases/tag/2.4.5","https://github.com/perl-catalyst/FCGI/issues/14","https://patch-diff.githubusercontent.com/raw/FastCGI-Archives/fcgi2/pull/74.patch","https://www.synacktiv.com/en/publications/cve-2025-23016-exploiting-the-fastcgi-library"],"reported" => "2025-05-16","severity" => undef}],"main_module" => "FCGI","versions" => [{"date" => "1996-09-25T17:48:57","version" => "0.25"},{"date" => "1996-10-15T21:51:06","version" => "0.26"},{"date" => "1997-02-20T08:55:44","version" => "0.27"},{"date" => "1997-02-25T07:14:13","version" => "0.28"},{"date" => "1997-06-10T18:16:17","version" => "0.29"},{"date" => "1997-06-24T17:17:05","version" => "0.30"},{"date" => "1997-07-24T11:05:43","version" => "0.31"},{"date" => "1998-06-17T10:24:17","version" => "0.34"},{"date" => "1998-06-22T15:38:51","version" => "0.35"},{"date" => "1998-06-24T19:42:57","version" => "0.36"},{"date" => "1998-06-27T16:08:39","version" => "0.37"},{"date" => "1998-07-15T15:24:00","version" => "0.40"},{"date" => "1998-07-29T16:05:51","version" => "0.41"},{"date" => "1998-08-28T15:30:49","version" => "0.42"},{"date" => "1998-12-22T22:34:14","version" => "0.43"},{"date" => "1998-12-23T11:28:39","version" => "0.44"},{"date" => "1999-03-08T17:04:02","version" => "0.45"},{"date" => "1999-07-30T08:26:31","version" => "0.46"},{"date" => "1999-07-31T21:58:01","version" => "0.47"},{"date" => "1999-08-27T13:41:54","version" => "0.48"},{"date" => "2000-04-09T18:58:32","version" => "0.49"},{"date" => "2000-04-10T07:04:43","version" => "0.50"},{"date" => "2000-04-12T12:27:09","version" => "0.51"},{"date" => "2000-04-12T14:10:02","version" => "0.52"},{"date" => "2000-07-10T10:01:51","version" => "0.53"},{"date" => "2000-10-08T19:52:29","version" => "0.54"},{"date" => "2000-10-18T21:22:46","version" => "0.55"},{"date" => "2000-11-03T15:44:28","version" => "0.56"},{"date" => "2000-11-12T15:15:01","version" => "0.57"},{"date" => "2000-11-14T23:20:24","version" => "0.58"},{"date" => "2000-12-31T22:05:44","version" => "0.59"},{"date" => "2001-06-08T15:19:08","version" => "0.60"},{"date" => "2001-09-20T12:34:13","version" => "0.61"},{"date" => "2001-09-21T16:19:42","version" => "0.62"},{"date" => "2001-09-24T20:43:48","version" => "0.63"},{"date" => "2001-09-25T08:26:24","version" => "0.64"},{"date" => "2002-02-19T14:16:27","version" => "0.65"},{"date" => "2002-09-05T16:23:07","version" => "0.66"},{"date" => "2002-12-23T10:21:36","version" => "0.67"},{"date" => "2009-12-20T21:05:48","version" => "0.67_01"},{"date" => "2010-01-06T10:07:05","version" => "0.68"},{"date" => "2010-01-10T01:35:11","version" => "0.68_01"},{"date" => "2010-01-13T19:25:40","version" => "0.68_02"},{"date" => "2010-02-15T23:08:12","version" => "0.69"},{"date" => "2010-03-22T14:35:03","version" => "0.70"},{"date" => "2010-04-01T00:55:33","version" => "0.71"},{"date" => "2010-08-24T21:32:56","version" => "0.71_01"},{"date" => "2011-04-28T08:50:09","version" => "0.71_02"},{"date" => "2011-04-28T09:05:42","version" => "0.71_03"},{"date" => "2011-05-19T09:06:02","version" => "0.72"},{"date" => "2011-05-28T01:35:17","version" => "0.73"},{"date" => "2011-09-24T08:31:47","version" => "0.74"},{"date" => "2014-07-17T00:19:02","version" => "0.75"},{"date" => "2014-08-05T01:29:06","version" => "0.76"},{"date" => "2014-08-05T15:53:28","version" => "0.77"},{"date" => "2016-03-07T00:08:23","version" => "0.78"},{"date" => "2019-12-14T18:29:19","version" => "0.79"},{"date" => "2021-07-25T04:54:49","version" => "0.80"},{"date" => "2021-07-30T23:19:01","version" => "0.81"},{"date" => "2021-07-31T03:26:34","version" => "0.82"}]},"Fake-Encode" => {"advisories" => [{"affected_versions" => ["<0.08"],"cves" => ["CVE-2016-1238"],"description" => "Loading modules from . (current directory).\n","distribution" => "Fake-Encode","fixed_versions" => [">=0.08"],"id" => "CPANSA-Fake-Encode-2017-01","references" => ["https://metacpan.org/changes/distribution/Fake-Encode"],"reported" => "2017-01-23"}],"main_module" => "Fake::Encode","versions" => [{"date" => "2016-05-31T14:11:49","version" => "0.01"},{"date" => "2017-01-23T12:34:23","version" => "0.02"},{"date" => "2017-01-25T15:52:13","version" => "0.03"},{"date" => "2017-01-26T15:17:01","version" => "0.04"},{"date" => "2017-03-06T16:01:40","version" => "0.05"},{"date" => "2017-09-08T17:54:14","version" => "0.06"},{"date" => "2017-09-09T15:27:50","version" => "0.07"},{"date" => "2018-02-03T14:50:49","version" => "0.08"},{"date" => "2018-02-19T12:21:04","version" => "0.09"},{"date" => "2019-07-11T16:26:06","version" => "0.10"},{"date" => "2023-03-25T02:26:13","version" => "0.11"}]},"Fake-Our" => {"advisories" => [{"affected_versions" => ["<0.06"],"cves" => ["CVE-2016-1238"],"description" => "Loading modules from . (current directory).\n","distribution" => "Fake-Our","fixed_versions" => [">=0.06"],"id" => "CPANSA-Fake-Our-2017-01","references" => ["https://metacpan.org/changes/distribution/Fake-Our"],"reported" => "2017-01-23"}],"main_module" => "Fake::Our","versions" => [{"date" => "2014-02-09T05:36:09","version" => "0.01"},{"date" => "2014-08-06T17:33:15","version" => "0.02"},{"date" => "2014-08-09T02:35:25","version" => "0.03"},{"date" => "2014-08-10T15:33:58","version" => "0.04"},{"date" => "2015-06-21T04:09:47","version" => "0.05"},{"date" => "2017-01-23T12:34:34","version" => "0.06"},{"date" => "2017-01-26T15:21:45","version" => "0.07"},{"date" => "2017-01-27T15:18:56","version" => "0.08"},{"date" => "2017-01-28T15:07:50","version" => "0.09"},{"date" => "2017-03-06T16:01:51","version" => "0.10"},{"date" => "2018-02-03T11:05:49","version" => "0.11"},{"date" => "2018-02-16T17:54:00","version" => "0.12"},{"date" => "2018-02-17T01:35:58","version" => "0.13"},{"date" => "2018-02-18T15:32:17","version" => "0.14"},{"date" => "2019-07-11T16:27:42","version" => "0.15"},{"date" => "2019-07-14T00:51:24","version" => "0.16"},{"date" => "2023-03-25T02:32:44","version" => "0.17"}]},"File-DataClass" => {"advisories" => [{"affected_versions" => ["<0.72.1"],"cves" => ["CVE-2016-1238"],"description" => "Loading modules from . (current directory).\n","distribution" => "File-DataClass","fixed_versions" => [">=0.72.1"],"id" => "CPANSA-File-DataClass-2017-01","references" => ["https://metacpan.org/changes/distribution/File-DataClass"],"reported" => "2017-04-01"}],"main_module" => "File::DataClass","versions" => [{"date" => "2010-09-29T16:37:04","version" => "0.1.228"},{"date" => "2010-10-06T14:20:31","version" => "0.2.234"},{"date" => "2011-01-26T18:14:50","version" => "0.3.238"},{"date" => "2011-02-27T23:09:38","version" => "0.3.239"},{"date" => "2011-04-12T19:44:59","version" => "0.3.259"},{"date" => "2011-05-15T17:45:09","version" => "0.4.268"},{"date" => "2011-05-30T01:47:40","version" => "0.5.271"},{"date" => "2011-07-11T13:39:10","version" => "0.6.286"},{"date" => "2011-11-30T00:05:18","version" => "0.7.321"},{"date" => "2011-12-02T04:40:20","version" => "0.7.325"},{"date" => "2011-12-02T22:39:25","version" => "0.7.326"},{"date" => "2011-12-03T18:43:58","version" => "0.7.328"},{"date" => "2012-02-22T18:28:29","version" => "0.7.330"},{"date" => "2012-02-23T11:00:24","version" => "0.7.331"},{"date" => "2012-02-24T10:52:18","version" => "0.7.332"},{"date" => "2012-03-12T17:34:58","version" => "0.7.335"},{"date" => "2012-03-20T18:24:26","version" => "0.7.336"},{"date" => "2012-03-21T22:43:50","version" => "0.7.338"},{"date" => "2012-03-22T13:48:59","version" => "0.7.339"},{"date" => "2012-03-24T00:37:31","version" => "0.7.343"},{"date" => "2012-03-28T23:58:41","version" => "0.8.351"},{"date" => "2012-03-29T22:05:21","version" => "0.8.355"},{"date" => "2012-04-03T00:26:12","version" => "0.8.357"},{"date" => "2012-04-04T15:19:03","version" => "0.8.360"},{"date" => "2012-04-17T18:57:01","version" => "0.9.368"},{"date" => "2012-05-19T21:05:56","version" => "0.10.380"},{"date" => "2012-07-10T00:34:23","version" => "0.11.401"},{"date" => "2012-09-02T13:43:37","version" => "0.12.406"},{"date" => "2012-09-06T14:02:06","version" => "0.12.409"},{"date" => "2012-11-07T07:49:39","version" => "0.13.416"},{"date" => "2012-11-13T20:16:27","version" => "0.13.418"},{"date" => "2012-12-12T23:25:16","version" => "0.13.420"},{"date" => "2012-12-14T17:58:08","version" => "0.13.421"},{"date" => "2012-12-19T22:23:08","version" => "0.13.422"},{"date" => "2012-12-21T20:48:41","version" => "0.13.424"},{"date" => "2012-12-30T03:05:28","version" => "0.13.427"},{"date" => "2013-01-07T00:52:48","version" => "0.14.429"},{"date" => "2013-04-01T01:14:44","version" => "0.15.431"},{"date" => "2013-04-02T14:21:13","version" => "0.15.434"},{"date" => "2013-04-14T16:15:55","version" => "v0.16.438"},{"date" => "2013-04-15T20:42:56","version" => "v0.16.442"},{"date" => "2013-04-24T03:47:54","version" => "v0.16.445"},{"date" => "2013-04-29T17:12:37","version" => "v0.17.450"},{"date" => "2013-04-30T22:15:36","version" => "v0.18.6"},{"date" => "2013-05-02T14:14:57","version" => "v0.19.1"},{"date" => "2013-05-07T23:33:06","version" => "v0.20.6"},{"date" => "2013-05-10T14:58:03","version" => "v0.20.7"},{"date" => "2013-05-14T13:32:28","version" => "v0.20.8"},{"date" => "2013-05-15T20:03:34","version" => "v0.20.9"},{"date" => "2013-05-16T00:11:50","version" => "v0.20.10"},{"date" => "2013-05-17T16:07:41","version" => "v0.20.12"},{"date" => "2013-06-08T13:26:40","version" => "v0.20.13"},{"date" => "2013-07-28T17:41:14","version" => "v0.22.1"},{"date" => "2013-07-29T11:39:49","version" => "v0.22.2"},{"date" => "2013-07-29T11:46:28","version" => "v0.22.3"},{"date" => "2013-07-29T18:37:14","version" => "v0.22.4"},{"date" => "2013-07-30T10:19:23","version" => "v0.22.5"},{"date" => "2013-07-30T16:25:59","version" => "v0.22.7"},{"date" => "2013-07-31T09:54:30","version" => "v0.22.8"},{"date" => "2013-08-02T19:06:49","version" => "v0.22.9"},{"date" => "2013-08-06T17:19:31","version" => "v0.23.1"},{"date" => "2013-08-07T13:14:13","version" => "v0.23.2"},{"date" => "2013-08-13T18:01:24","version" => "0.24.1"},{"date" => "2013-08-16T22:49:23","version" => "0.24.3"},{"date" => "2013-09-03T13:11:17","version" => "0.25.1"},{"date" => "2013-09-26T16:04:18","version" => "0.26.1"},{"date" => "2013-11-22T09:42:00","version" => "0.27.1"},{"date" => "2014-01-01T15:02:23","version" => "0.28.1"},{"date" => "2014-01-01T17:03:18","version" => "0.29.1"},{"date" => "2014-01-02T02:33:28","version" => "0.30.1"},{"date" => "2014-01-13T18:41:29","version" => "0.31.1"},{"date" => "2014-01-24T20:56:21","version" => "0.33.1"},{"date" => "2014-04-04T10:52:59","version" => "0.34.1"},{"date" => "2014-05-01T14:40:32","version" => "0.35.1"},{"date" => "2014-05-13T10:03:54","version" => "0.36.1"},{"date" => "2014-05-13T21:08:07","version" => "0.37.1"},{"date" => "2014-05-15T00:11:43","version" => "0.38.1"},{"date" => "2014-05-16T08:19:01","version" => "0.39.1"},{"date" => "2014-05-22T09:37:34","version" => "0.40.1"},{"date" => "2014-05-22T14:10:49","version" => "0.40.2"},{"date" => "2014-05-28T10:28:42","version" => "0.41.1"},{"date" => "2014-07-03T23:27:53","version" => "0.42.1"},{"date" => "2014-07-04T09:25:10","version" => "0.42.2"},{"date" => "2014-07-04T12:19:02","version" => "0.43.1"},{"date" => "2014-07-16T12:39:03","version" => "0.44.1"},{"date" => "2014-08-18T23:00:05","version" => "0.45.1"},{"date" => "2014-08-26T12:43:14","version" => "0.45.5"},{"date" => "2014-08-26T16:41:35","version" => "0.46.1"},{"date" => "2014-08-27T16:17:50","version" => "0.47.1"},{"date" => "2014-09-03T22:25:51","version" => "0.48.1"},{"date" => "2014-10-02T17:39:13","version" => "0.48.3"},{"date" => "2014-10-02T19:59:28","version" => "0.49.1"},{"date" => "2014-11-07T18:51:52","version" => "0.50.1"},{"date" => "2014-11-08T21:45:45","version" => "0.50.2"},{"date" => "2014-11-09T13:19:50","version" => "0.50.3"},{"date" => "2014-11-09T15:52:41","version" => "0.51.1"},{"date" => "2014-11-10T12:44:49","version" => "0.52.1"},{"date" => "2014-12-19T11:49:49","version" => "0.53.1"},{"date" => "2014-12-19T22:54:41","version" => "0.54.1"},{"date" => "2015-02-05T00:04:33","version" => "0.55.1"},{"date" => "2015-03-19T14:59:03","version" => "0.56.1"},{"date" => "2015-04-04T20:00:58","version" => "0.57.1"},{"date" => "2015-04-04T20:16:31","version" => "0.58.1"},{"date" => "2015-04-05T17:58:31","version" => "0.59.1"},{"date" => "2015-04-08T23:10:28","version" => "0.60.1"},{"date" => "2015-05-11T12:15:05","version" => "0.61.1"},{"date" => "2015-05-24T11:52:28","version" => "0.62.1"},{"date" => "2015-06-21T21:42:17","version" => "0.63.1"},{"date" => "2015-08-29T08:58:54","version" => "0.66.1"},{"date" => "2016-02-01T00:18:43","version" => "0.67.1"},{"date" => "2016-02-01T14:33:11","version" => "0.68.1"},{"date" => "2016-07-05T00:36:52","version" => "0.69.1"},{"date" => "2016-07-29T15:24:24","version" => "0.70.1"},{"date" => "2016-07-29T18:59:13","version" => "0.71.1"},{"date" => "2017-04-02T08:23:47","version" => "0.72.1"},{"date" => "2017-06-02T00:03:17","version" => "0.73.1"}]},"File-Find-Rule" => {"advisories" => [{"affected_versions" => ["<=0.34"],"cves" => ["CVE-2011-10007"],"description" => "File::Find::Rule through 0.34 for Perl is vulnerable to Arbitrary Code Execution when `grep()` encounters a crafted filename.  A file handle is opened with the 2 argument form of `open()` allowing an attacker controlled filename to provide the MODE parameter to `open()`, turning the filename into a command to be executed.  Example:  \$ mkdir /tmp/poc; echo > \"/tmp/poc/|id\" \$ perl -MFile::Find::Rule \\ \x{a0} \x{a0} -E 'File::Find::Rule->grep(\"foo\")->in(\"/tmp/poc\")' uid=1000(user) gid=1000(user) groups=1000(user),100(users)","distribution" => "File-Find-Rule","fixed_versions" => [">=0.35"],"id" => "CPANSA-File-Find-Rule-2011-10007","references" => ["https://github.com/richardc/perl-file-find-rule/commit/df58128bcee4c1da78c34d7f3fe1357e575ad56f.patch","https://github.com/richardc/perl-file-find-rule/pull/4","https://metacpan.org/release/RCLAMP/File-Find-Rule-0.34/source/lib/File/Find/Rule.pm#L423","https://rt.cpan.org/Public/Bug/Display.html?id=64504","http://www.openwall.com/lists/oss-security/2025/06/05/4","http://www.openwall.com/lists/oss-security/2025/06/06/1","http://www.openwall.com/lists/oss-security/2025/06/06/3","https://lists.debian.org/debian-lts-announce/2025/06/msg00006.html","https://github.com/richardc/perl-file-find-rule/pull/4"],"reported" => "2025-06-05","severity" => undef}],"main_module" => "File::Find::Rule","versions" => [{"date" => "2002-07-26T13:03:10","version" => "0.01"},{"date" => "2002-08-14T22:28:12","version" => "0.02"},{"date" => "2002-08-24T17:34:12","version" => "0.03"},{"date" => "2002-09-10T08:54:04","version" => "0.04"},{"date" => "2002-10-21T16:37:18","version" => "0.05"},{"date" => "2002-10-22T07:30:31","version" => "0.06"},{"date" => "2002-10-25T15:54:13","version" => "0.07"},{"date" => "2002-12-04T13:55:56","version" => "0.08"},{"date" => "2003-01-21T10:56:48","version" => "0.09"},{"date" => "2003-03-10T02:07:24","version" => "0.10"},{"date" => "2003-06-22T21:04:15","version" => "0.20_01"},{"date" => "2003-06-25T11:36:22","version" => "0.20_02"},{"date" => "2003-07-29T19:24:32","version" => "0.11"},{"date" => "2003-08-04T09:27:12","version" => "0.20_03"},{"date" => "2003-09-08T17:44:26","version" => "0.20"},{"date" => "2003-09-15T12:16:58","version" => "0.21"},{"date" => "2003-10-03T19:33:19","version" => "0.22"},{"date" => "2003-10-03T22:57:25","version" => "0.23"},{"date" => "2003-10-04T11:20:43","version" => "0.24_01"},{"date" => "2003-10-06T14:22:20","version" => "0.24"},{"date" => "2003-10-22T17:11:46","version" => "0.25"},{"date" => "2003-11-10T22:10:06","version" => "0.26"},{"date" => "2004-02-25T10:55:36","version" => "0.27"},{"date" => "2004-05-18T20:37:58","version" => "0.28"},{"date" => "2006-05-16T14:28:43","version" => "0.29"},{"date" => "2006-06-01T15:39:35","version" => "0.30"},{"date" => "2009-11-27T22:58:10","version" => "0.31"},{"date" => "2009-11-28T00:47:34","version" => "0.32"},{"date" => "2011-09-19T11:56:02","version" => "0.33"},{"date" => "2015-12-03T14:31:54","version" => "0.34"},{"date" => "2025-06-05T15:35:41","version" => "0.35"}]},"File-KeePass" => {"advisories" => [{"affected_versions" => [">0"],"cves" => [],"description" => "The module is making use of the perl rand function for key and iv generation (for Crypt::Rijndael).\n","distribution" => "File-KeePass","fixed_versions" => [],"id" => "CPANSA-File-KeePass-2016-01","references" => ["https://rt.cpan.org/Ticket/Display.html?id=117836"],"reported" => "2016-09-14","severity" => undef}],"main_module" => "File::KeePass","versions" => [{"date" => "2010-06-29T14:52:50","version" => "0.01"},{"date" => "2010-12-04T04:33:41","version" => "0.02"},{"date" => "2010-12-07T06:06:57","version" => "0.03"},{"date" => "2012-09-13T04:48:56","version" => "2.00"},{"date" => "2012-09-13T14:17:11","version" => "2.01"},{"date" => "2012-09-13T15:22:40","version" => "2.02"},{"date" => "2012-09-15T22:25:43","version" => "2.03"}]},"File-Path" => {"advisories" => [{"affected_versions" => ["<2.13"],"cves" => ["CVE-2017-6512"],"description" => "Race condition in the rmtree and remove_tree functions allows attackers to set the mode on arbitrary files via vectors involving directory-permission loosening logic.\n","distribution" => "File-Path","fixed_versions" => [">=2.13"],"id" => "CPANSA-File-Path-2017-01","references" => ["https://metacpan.org/changes/distribution/File-Path","https://github.com/jkeenan/File-Path/commit/e5ef95276ee8ad471c66ee574a5d42552b3a6af2"],"reported" => "2017-05-02"},{"affected_versions" => ["<=1.08"],"cves" => ["CVE-2008-5303"],"description" => "Race condition in the rmtree function in File::Path 1.08 (lib/File/Path.pm) in Perl 5.8.8 allows local users to to delete arbitrary files via a symlink attack, a different vulnerability than CVE-2005-0448, CVE-2004-0452, and CVE-2008-2827. NOTE: this is a regression error related to CVE-2005-0448. It is different from CVE-2008-5302 due to affected versions.\n","distribution" => "File-Path","fixed_versions" => [],"id" => "CPANSA-File-Path-2008-5303","references" => ["http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=286905","http://www.openwall.com/lists/oss-security/2008/11/28/2","http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=286922#36","http://www.gossamer-threads.com/lists/perl/porters/233695#233695","http://www.debian.org/security/2008/dsa-1678","http://www.ubuntu.com/usn/usn-700-1","http://secunia.com/advisories/33314","http://wiki.rpath.com/Advisories:rPSA-2009-0011","http://www.ubuntu.com/usn/usn-700-2","http://secunia.com/advisories/32980","http://support.apple.com/kb/HT4077","http://lists.apple.com/archives/security-announce/2010//Mar/msg00001.html","http://www.redhat.com/support/errata/RHSA-2010-0458.html","http://secunia.com/advisories/40052","http://www.mandriva.com/security/advisories?name=MDVSA-2010:116","http://kb.juniper.net/InfoCenter/index?page=content&id=JSA10735","http://kb.juniper.net/InfoCenter/index?page=content&id=JSA10705","https://exchange.xforce.ibmcloud.com/vulnerabilities/47044","https://oval.cisecurity.org/repository/search/definition/oval%3Aorg.mitre.oval%3Adef%3A9699","https://oval.cisecurity.org/repository/search/definition/oval%3Aorg.mitre.oval%3Adef%3A6680","http://www.securityfocus.com/archive/1/500210/100/0/threaded"],"reported" => "2008-12-01","severity" => undef},{"affected_versions" => ["==1.08","==2.07"],"cves" => ["CVE-2008-5302"],"description" => "Race condition in the rmtree function in File::Path 1.08 and 2.07 (lib/File/Path.pm) in Perl 5.8.8 and 5.10.0 allows local users to create arbitrary setuid binaries via a symlink attack, a different vulnerability than CVE-2005-0448, CVE-2004-0452, and CVE-2008-2827. NOTE: this is a regression error related to CVE-2005-0448.  It is different from CVE-2008-5303 due to affected versions.\n","distribution" => "File-Path","fixed_versions" => [],"id" => "CPANSA-File-Path-2008-5302","references" => ["http://www.gossamer-threads.com/lists/perl/porters/233695#233695","http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=286922#36","http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=286905","http://www.openwall.com/lists/oss-security/2008/11/28/2","http://www.debian.org/security/2008/dsa-1678","http://secunia.com/advisories/33314","http://www.ubuntu.com/usn/usn-700-1","http://wiki.rpath.com/Advisories:rPSA-2009-0011","http://www.ubuntu.com/usn/usn-700-2","http://lists.opensuse.org/opensuse-security-announce/2009-02/msg00002.html","http://secunia.com/advisories/32980","http://lists.apple.com/archives/security-announce/2010//Mar/msg00001.html","http://support.apple.com/kb/HT4077","http://secunia.com/advisories/40052","http://www.mandriva.com/security/advisories?name=MDVSA-2010:116","http://www.redhat.com/support/errata/RHSA-2010-0458.html","http://kb.juniper.net/InfoCenter/index?page=content&id=JSA10735","http://kb.juniper.net/InfoCenter/index?page=content&id=JSA10705","https://exchange.xforce.ibmcloud.com/vulnerabilities/47043","https://oval.cisecurity.org/repository/search/definition/oval%3Aorg.mitre.oval%3Adef%3A6890","https://oval.cisecurity.org/repository/search/definition/oval%3Aorg.mitre.oval%3Adef%3A11076","http://www.securityfocus.com/archive/1/500210/100/0/threaded"],"reported" => "2008-12-01","severity" => undef},{"affected_versions" => [">=2.04,<2.07"],"cves" => ["CVE-2008-2827"],"description" => "The rmtree function in lib/File/Path.pm in Perl 5.10 does not properly check permissions before performing a chmod, which allows local users to modify the permissions of arbitrary files via a symlink attack, a different vulnerability than CVE-2005-0448 and CVE-2004-0452.\n","distribution" => "File-Path","fixed_versions" => [">=2.07"],"id" => "CPANSA-File-Path-2008-2827","references" => ["http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=487319","http://rt.cpan.org/Public/Bug/Display.html?id=36982","http://www.securityfocus.com/bid/29902","http://secunia.com/advisories/30790","http://lists.opensuse.org/opensuse-security-announce/2008-08/msg00006.html","http://www.mandriva.com/security/advisories?name=MDVSA-2008:165","https://www.redhat.com/archives/fedora-package-announce/2008-June/msg01025.html","http://secunia.com/advisories/30837","http://secunia.com/advisories/31687","http://www.securitytracker.com/id?1020373","https://exchange.xforce.ibmcloud.com/vulnerabilities/43308"],"reported" => "2008-06-23","severity" => undef},{"affected_versions" => ["<1.07"],"cves" => ["CVE-2005-0448"],"description" => "Race condition in the rmtree function in File::Path.pm in Perl before 5.8.4 allows local users to create arbitrary setuid binaries in the tree being deleted, a different vulnerability than CVE-2004-0452.\n","distribution" => "File-Path","fixed_versions" => [],"id" => "CPANSA-File-Path-2005-0448","references" => ["http://www.debian.org/security/2005/dsa-696","http://www.gentoo.org/security/en/glsa/glsa-200501-38.xml","http://www.redhat.com/support/errata/RHSA-2005-881.html","http://secunia.com/advisories/18075","http://www.securityfocus.com/bid/12767","ftp://patches.sgi.com/support/free/security/advisories/20060101-01-U","http://secunia.com/advisories/14531","http://secunia.com/advisories/18517","http://fedoranews.org/updates/FEDORA--.shtml","http://www.redhat.com/support/errata/RHSA-2005-674.html","http://secunia.com/advisories/17079","http://distro.conectiva.com.br/atualizacoes/?id=a&anuncio=001056","http://www.mandriva.com/security/advisories?name=MDKSA-2005:079","http://www.securityfocus.com/advisories/8704","http://secunia.com/advisories/55314","https://oval.cisecurity.org/repository/search/definition/oval%3Aorg.mitre.oval%3Adef%3A728","https://oval.cisecurity.org/repository/search/definition/oval%3Aorg.mitre.oval%3Adef%3A10475","https://usn.ubuntu.com/94-1/"],"reported" => "2005-05-02","severity" => undef},{"affected_versions" => [">=1.06,<=1.404"],"cves" => ["CVE-2004-0452"],"description" => "Race condition in the rmtree function in the File::Path module in Perl 5.6.1 and 5.8.4 sets read/write permissions for the world, which allows local users to delete arbitrary files and directories, and possibly read files and directories, via a symlink attack.\n","distribution" => "File-Path","fixed_versions" => [],"id" => "CPANSA-File-Path-2004-0452","references" => ["http://www.debian.org/security/2004/dsa-620","http://www.gentoo.org/security/en/glsa/glsa-200501-38.xml","http://www.redhat.com/support/errata/RHSA-2005-103.html","ftp://patches.sgi.com/support/free/security/advisories/20060101-01-U","http://www.securityfocus.com/bid/12072","http://secunia.com/advisories/12991","http://secunia.com/advisories/18517","http://fedoranews.org/updates/FEDORA--.shtml","http://www.redhat.com/support/errata/RHSA-2005-105.html","http://secunia.com/advisories/55314","http://marc.info/?l=bugtraq&m=110547693019788&w=2","https://www.ubuntu.com/usn/usn-44-1/","https://exchange.xforce.ibmcloud.com/vulnerabilities/18650","https://oval.cisecurity.org/repository/search/definition/oval%3Aorg.mitre.oval%3Adef%3A9938"],"reported" => "2004-12-21","severity" => undef}],"main_module" => "File::Path","versions" => [{"date" => "2007-05-17T13:46:15","version" => "1.99_01"},{"date" => "2007-05-27T09:29:48","version" => "1.99_02"},{"date" => "2007-06-27T19:23:09","version" => "2.00_05"},{"date" => "2007-07-04T21:37:13","version" => "2.00_06"},{"date" => "2007-07-09T19:37:02","version" => "2.00_07"},{"date" => "2007-08-01T00:10:43","version" => "2.00_08"},{"date" => "2007-08-20T18:15:55","version" => "2.00_09"},{"date" => "2007-09-04T17:20:45","version" => "2.00_10"},{"date" => "2007-09-08T12:53:07","version" => "2.00_11"},{"date" => "2007-09-29T10:29:32","version" => "2.01"},{"date" => "2007-10-24T10:36:09","version" => "2.02"},{"date" => "2007-11-04T18:36:19","version" => "2.03"},{"date" => "2007-11-24T09:53:23","version" => "2.04"},{"date" => "2008-05-07T08:25:05","version" => "2.05"},{"date" => "2008-05-08T09:36:50","version" => "2.06"},{"date" => "2008-05-10T21:02:47","version" => "2.06_01"},{"date" => "2008-05-12T10:07:46","version" => "2.06_02"},{"date" => "2008-05-12T21:43:43","version" => "2.06_03"},{"date" => "2008-05-13T14:40:30","version" => "2.06_04"},{"date" => "2008-10-01T20:41:37","version" => "2.06_05"},{"date" => "2008-10-05T21:59:58","version" => "2.06_06"},{"date" => "2008-10-29T17:55:36","version" => "2.06_07"},{"date" => "2008-11-05T00:12:29","version" => "2.06_08"},{"date" => "2008-11-09T13:11:17","version" => "2.07"},{"date" => "2009-06-21T13:23:32","version" => "2.07_03"},{"date" => "2009-10-04T10:31:05","version" => "2.08"},{"date" => "2013-01-16T21:36:05","version" => "2.09"},{"date" => "2015-06-24T17:03:22","version" => "2.10_001"},{"date" => "2015-06-26T17:28:20","version" => "2.10_002"},{"date" => "2015-07-08T16:59:11","version" => "2.10_003"},{"date" => "2015-07-10T11:34:44","version" => "2.10_004"},{"date" => "2015-07-17T15:03:07","version" => "2.10_005"},{"date" => "2015-07-18T02:28:14","version" => "2.11"},{"date" => "2015-07-24T23:01:36","version" => "2.11_001"},{"date" => "2015-07-25T09:56:18","version" => "2.11_002"},{"date" => "2015-08-03T18:07:05","version" => "2.11_003"},{"date" => "2015-10-01T19:34:07","version" => "2.11_004"},{"date" => "2015-10-09T12:11:52","version" => "2.12"},{"date" => "2016-09-18T13:35:39","version" => "2.12_001"},{"date" => "2017-03-12T22:09:35","version" => "2.12_002"},{"date" => "2017-04-07T13:59:30","version" => "2.12_003"},{"date" => "2017-04-18T18:37:56","version" => "2.12_004"},{"date" => "2017-04-21T12:03:20","version" => "2.12_005"},{"date" => "2017-04-21T21:58:56","version" => "2.12_006"},{"date" => "2017-04-22T20:09:24","version" => "2.12_007"},{"date" => "2017-05-07T17:48:35","version" => "2.12_008"},{"date" => "2017-05-31T23:44:51","version" => "2.13"},{"date" => "2017-06-07T21:34:52","version" => "2.14"},{"date" => "2017-07-30T02:40:36","version" => "2.15"},{"date" => "2018-08-31T13:04:13","version" => "2.16"},{"date" => "2020-07-18T18:29:28","version" => "2.17"},{"date" => "2020-11-04T12:38:02","version" => "2.18_001"},{"date" => "2020-11-05T01:30:15","version" => "2.18"},{"date" => "1995-03-14T00:00:00","dual_lived" => 1,"perl_release" => "5.001","version" => undef},{"date" => "1996-02-29T00:00:00","dual_lived" => 1,"perl_release" => "5.002","version" => "1.01"},{"date" => "1997-05-15T00:00:00","dual_lived" => 1,"perl_release" => "5.004","version" => "1.04"},{"date" => "1999-04-29T00:00:00","dual_lived" => 1,"perl_release" => "5.00405","version" => "1.0402"},{"date" => "1998-07-22T00:00:00","dual_lived" => 1,"perl_release" => "5.005","version" => "1.0401"},{"date" => "2000-03-22T00:00:00","dual_lived" => 1,"perl_release" => "5.006000","version" => "1.0403"},{"date" => "2001-04-08T00:00:00","dual_lived" => 1,"perl_release" => "5.006001","version" => "1.0404"},{"date" => "2002-03-05T00:00:00","dual_lived" => 1,"perl_release" => "5.007003","version" => "1.05"},{"date" => "2003-09-25T00:00:00","dual_lived" => 1,"perl_release" => "5.008001","version" => "1.06"},{"date" => "2005-05-30T00:00:00","dual_lived" => 1,"perl_release" => "5.008007","version" => "1.07"},{"date" => "2006-01-31T00:00:00","dual_lived" => 1,"perl_release" => "5.008008","version" => "1.08"},{"date" => "2008-12-14T00:00:00","dual_lived" => 1,"perl_release" => "5.008009","version" => "2.07_02"},{"date" => "2009-12-20T00:00:00","dual_lived" => 1,"perl_release" => "5.011003","version" => "2.08_01"},{"date" => "2016-05-09T00:00:00","dual_lived" => 1,"perl_release" => "5.024","version" => "2.12_01"}]},"File-Slurp" => {"advisories" => [{"affected_versions" => ["<9999.26"],"cves" => [],"description" => "Use of sysread treats any :encoding(...) as effectively :utf8.\n","distribution" => "File-Slurp","fixed_versions" => [">=9999.26"],"id" => "CPANSA-File-Slurp-2013-01","references" => ["https://rt.cpan.org/Public/Bug/Display.html?id=83126","https://rt.perl.org/Ticket/Display.html?id=121870"],"reported" => "2013-02-04"}],"main_module" => "File::Slurp","versions" => [{"date" => "1996-04-22T21:18:00","version" => "96.042202"},{"date" => "1998-07-19T16:25:00","version" => "98.071901"},{"date" => "2001-11-04T03:17:00","version" => "2001.1103"},{"date" => "2002-03-08T05:22:00","version" => "2002.0305"},{"date" => "2002-11-01T03:14:00","version" => "2002.1031"},{"date" => "2003-09-04T16:28:00","version" => "2004.0904"},{"date" => "2003-11-24T07:45:57","version" => "0.01"},{"date" => "2003-11-24T08:02:47","version" => "9999.01"},{"date" => "2003-12-17T09:20:57","version" => "9999.02"},{"date" => "2003-12-22T06:54:57","version" => "9999.03"},{"date" => "2004-02-23T19:27:53","version" => "9999.04"},{"date" => "2004-09-21T05:23:58","version" => "9999.06"},{"date" => "2005-01-30T10:01:07","version" => "9999.07"},{"date" => "2005-04-16T05:06:09","version" => "9999.08"},{"date" => "2005-04-29T06:09:11","version" => "9999.09"},{"date" => "2006-01-19T18:29:42","version" => "9999.10"},{"date" => "2006-01-20T06:45:13","version" => "9999.11"},{"date" => "2006-03-07T07:13:42","version" => "9999.12"},{"date" => "2008-01-24T04:57:12","version" => "9999.13"},{"date" => "2011-03-22T22:41:08","version" => "9999.14"},{"date" => "2011-03-24T22:52:42","version" => "9999.15"},{"date" => "2011-04-24T04:26:18","version" => "9999.16"},{"date" => "2011-05-13T06:23:08","version" => "9999.17"},{"date" => "2011-05-13T07:03:44","version" => "9999.18"},{"date" => "2011-06-07T08:08:06","version" => "9999.19"},{"date" => "2018-09-28T01:57:50","version" => "9999.20_01"},{"date" => "2018-10-05T01:23:09","version" => "9999.20_02"},{"date" => "2018-10-08T21:16:27","version" => "9999.21"},{"date" => "2018-10-16T03:15:39","version" => "9999.22"},{"date" => "2018-10-20T20:06:53","version" => "9999.23"},{"date" => "2018-10-30T02:45:09","version" => "9999.24"},{"date" => "2018-11-16T16:11:34","version" => "9999.25"},{"date" => "2019-02-13T16:35:40","version" => "9999.26"},{"date" => "2019-04-05T13:28:05","version" => "9999.27"},{"date" => "2019-09-13T00:36:22","version" => "9999.28"},{"date" => "2019-11-27T20:40:47","version" => "9999.29"},{"date" => "2020-03-09T14:31:40","version" => "9999.30"},{"date" => "2020-06-28T22:33:21","version" => "9999.31"},{"date" => "2020-07-01T00:34:08","version" => "9999.32"}]},"File-Temp" => {"advisories" => [{"affected_versions" => [">0"],"cves" => ["CVE-2011-4116"],"description" => "_is_safe in the File::Temp module for Perl does not properly handle symlinks.\n","distribution" => "File-Temp","fixed_versions" => [],"id" => "CPANSA-File-Temp-2011-4116","references" => ["http://www.openwall.com/lists/oss-security/2011/11/04/2","https://rt.cpan.org/Public/Bug/Display.html?id=69106","https://github.com/Perl-Toolchain-Gang/File-Temp/issues/14","http://www.openwall.com/lists/oss-security/2011/11/04/4","https://seclists.org/oss-sec/2011/q4/238"],"reported" => "2020-01-31","severity" => "high"}],"main_module" => "File::Temp","versions" => [{"date" => "2000-03-14T20:15:55","version" => "0.05"},{"date" => "2000-04-28T04:48:55","version" => "0.07"},{"date" => "2000-05-16T01:10:28","version" => "0.08"},{"date" => "2000-07-26T20:30:30","version" => "0.09"},{"date" => "2000-12-12T21:04:53","version" => "0.11"},{"date" => "2001-02-23T00:37:44","version" => "0.12"},{"date" => "2003-08-16T04:06:11","version" => "0.13"},{"date" => "2003-08-17T04:42:50","version" => "0.14"},{"date" => "2005-02-22T05:40:33","version" => "0.15"},{"date" => "2005-02-22T21:42:47","version" => "0.16"},{"date" => "2006-08-18T22:40:10","version" => "0.17"},{"date" => "2007-01-22T00:18:40","version" => "0.18"},{"date" => "2007-11-20T08:28:08","version" => "0.19"},{"date" => "2007-12-21T00:46:29","version" => "0.20"},{"date" => "2008-11-14T01:30:09","version" => "0.21"},{"date" => "2009-06-29T07:41:24","version" => "0.22"},{"date" => "2013-02-07T17:03:45","version" => "0.22_90"},{"date" => "2013-03-14T21:57:42","version" => "0.23"},{"date" => "2013-04-11T15:31:13","version" => "0.2301"},{"date" => "2013-09-26T13:48:13","version" => "0.2302"},{"date" => "2013-10-09T13:59:01","version" => "0.2303"},{"date" => "2013-10-10T13:17:32","version" => "0.2304"},{"date" => "2018-04-19T12:01:34","version" => "0.2305"},{"date" => "2018-06-24T19:34:31","version" => "0.2306"},{"date" => "2018-06-24T19:41:28","version" => "0.2307"},{"date" => "2018-07-11T21:07:49","version" => "0.2308"},{"date" => "2019-01-06T20:32:53","version" => "0.2309"},{"date" => "2020-09-26T17:39:38","version" => "0.2310"},{"date" => "2020-10-03T04:04:55","version" => "0.2311"},{"date" => "2025-09-01T18:57:33","version" => "0.2312"},{"date" => "2006-01-28T00:00:00","dual_lived" => 1,"perl_release" => "5.009003","version" => "0.16_01"}]},"Filesys-SmbClientParser" => {"advisories" => [{"affected_versions" => ["<=2.7"],"cves" => ["CVE-2008-3285"],"description" => "The Filesys::SmbClientParser module 2.7 and earlier for Perl allows remote SMB servers to execute arbitrary code via a folder name containing shell metacharacters.\n","distribution" => "Filesys-SmbClientParser","fixed_versions" => [],"id" => "CPANSA-Filesys-SmbClientParser-2008-3285","references" => ["http://www.securityfocus.com/bid/30290","http://secunia.com/advisories/31175","http://securityreason.com/securityalert/4027","https://exchange.xforce.ibmcloud.com/vulnerabilities/43910","http://www.securityfocus.com/archive/1/494536/100/0/threaded"],"reported" => "2008-07-24","severity" => undef}],"main_module" => "Filesys::SmbClientParser","versions" => [{"date" => "2000-11-19T21:10:38","version" => "0.01"},{"date" => "2000-11-20T19:41:09","version" => "0.2"},{"date" => "2001-01-12T00:31:50","version" => "0.3"},{"date" => "2001-04-15T22:37:14","version" => "1.2"},{"date" => "2001-04-19T17:38:19","version" => "1.3"},{"date" => "2001-05-30T08:04:44","version" => "1.4"},{"date" => "2002-01-25T12:18:47","version" => "2.0"},{"date" => "2002-04-19T21:56:09","version" => "2.1"},{"date" => "2002-08-09T11:24:20","version" => "2.2"},{"date" => "2002-08-13T14:55:48","version" => "2.3"},{"date" => "2002-11-08T23:57:07","version" => "2.4"},{"date" => "2002-11-12T18:59:33","version" => "2.5"},{"date" => "2004-01-28T23:06:58","version" => "2.6"},{"date" => "2004-04-14T21:56:02","version" => "2.7"}]},"GBrowse" => {"advisories" => [{"affected_versions" => ["<2.56"],"cves" => [],"description" => "An attacker is able to delete other users' accounts.  No httponly cookie flag.  Cross-site scripting vulnerability in generation of citation text.\n","distribution" => "GBrowse","fixed_versions" => [">=2.56"],"id" => "CPANSA-GBrowse-2017-01","references" => ["https://metacpan.org/changes/distribution/GBrowse"],"reported" => "2017-01-15"},{"affected_versions" => ["<1.62"],"cves" => [],"description" => "Cross-site scripting.\n","distribution" => "GBrowse","fixed_versions" => [">=1.62"],"id" => "CPANSA-GBrowse-2004-01","references" => ["https://metacpan.org/changes/distribution/GBrowse"],"reported" => "2004-04-05"},{"affected_versions" => ["<1.54"],"cves" => [],"description" => "Path traversal.\n","distribution" => "GBrowse","fixed_versions" => [">=1.54"],"id" => "CPANSA-GBrowse-2003-01","references" => ["https://metacpan.org/changes/distribution/GBrowse"],"reported" => "2003-08-23"}],"main_module" => "CGI::Toggle","versions" => [{"date" => "2008-12-29T15:38:27","version" => "1.981"},{"date" => "2008-12-29T17:21:12","version" => "1.982"},{"date" => "2009-01-06T07:42:56","version" => "1.983"},{"date" => "2009-01-06T09:17:54","version" => "1.984"},{"date" => "2009-01-10T12:37:42","version" => "1.985"},{"date" => "2009-01-12T16:58:20","version" => "1.986"},{"date" => "2009-01-22T19:49:50","version" => "1.987"},{"date" => "2009-01-30T00:12:57","version" => "1.988"},{"date" => "2009-03-10T19:24:17","version" => "1.989"},{"date" => "2009-04-03T19:29:22","version" => "1.99"},{"date" => "2009-05-04T05:30:31","version" => "1.991"},{"date" => "2009-05-05T23:40:00","version" => "1.992"},{"date" => "2009-05-07T14:11:11","version" => "1.993"},{"date" => "2009-05-30T22:07:17","version" => "1.994"},{"date" => "2009-06-08T21:27:08","version" => "1.995"},{"date" => "2009-07-06T14:12:57","version" => "1.996"},{"date" => "2009-07-30T16:40:54","version" => "1.997"},{"date" => "2009-08-19T19:19:44","version" => "1.9971"},{"date" => "2009-12-09T21:39:37","version" => "1.998"},{"date" => "2009-12-15T15:59:37","version" => "1.9982"},{"date" => "2009-12-18T19:25:25","version" => "1.9983"},{"date" => "2009-12-22T21:20:40","version" => "1.9984"},{"date" => "2009-12-23T21:56:31","version" => "1.999"},{"date" => "2010-01-28T02:58:41","version" => "2.00"},{"date" => "2010-02-09T18:13:33","version" => "2.01"},{"date" => "2010-03-10T05:56:50","version" => "2.02"},{"date" => "2010-03-25T16:06:21","version" => "2.03"},{"date" => "2010-04-18T21:44:27","version" => "2.04"},{"date" => "2010-05-13T03:30:32","version" => "2.05"},{"date" => "2010-05-13T21:17:05","version" => "2.06"},{"date" => "2010-05-17T14:49:41","version" => "2.07"},{"date" => "2010-05-21T02:52:47","version" => "2.08"},{"date" => "2010-06-10T20:17:32","version" => "2.09"},{"date" => "2010-06-15T14:20:30","version" => "2.10"},{"date" => "2010-06-30T19:15:37","version" => "2.11"},{"date" => "2010-06-30T19:30:03","version" => "2.12"},{"date" => "2010-07-05T20:17:39","version" => "2.13"},{"date" => "2010-08-27T15:06:04","version" => "2.14"},{"date" => "2010-09-13T22:17:44","version" => "2.15"},{"date" => "2010-11-01T16:24:01","version" => "2.16"},{"date" => "2010-11-18T17:08:57","version" => "2.17"},{"date" => "2011-01-18T22:35:59","version" => "2.20"},{"date" => "2011-01-22T17:17:34","version" => "2.21"},{"date" => "2011-01-26T14:31:35","version" => "2.22"},{"date" => "2011-01-30T20:03:25","version" => "2.23"},{"date" => "2011-01-31T17:19:08","version" => "2.24"},{"date" => "2011-02-02T18:53:40","version" => "2.25"},{"date" => "2011-02-04T18:51:54","version" => "2.26"},{"date" => "2011-04-10T21:07:42","version" => "2.27"},{"date" => "2011-04-10T21:32:05","version" => "2.28"},{"date" => "2011-05-02T16:12:11","version" => "2.29"},{"date" => "2011-05-03T12:17:18","version" => "2.30"},{"date" => "2011-05-03T15:50:21","version" => "2.31"},{"date" => "2011-05-04T18:47:51","version" => "2.32"},{"date" => "2011-05-07T03:27:32","version" => "2.33"},{"date" => "2011-06-01T15:19:47","version" => "2.34"},{"date" => "2011-06-03T13:41:28","version" => "2.35"},{"date" => "2011-06-04T14:58:14","version" => "2.36"},{"date" => "2011-06-06T21:24:59","version" => "2.37"},{"date" => "2011-06-09T16:00:48","version" => "2.38"},{"date" => "2011-06-29T17:45:00","version" => "2.39"},{"date" => "2011-09-30T16:56:29","version" => "2.40"},{"date" => "2011-10-07T13:31:48","version" => "2.41"},{"date" => "2011-10-12T19:33:22","version" => "2.42"},{"date" => "2011-10-24T16:43:23","version" => "2.43"},{"date" => "2011-12-08T23:09:26","version" => "2.44"},{"date" => "2012-01-03T21:35:41","version" => "2.45"},{"date" => "2012-02-10T17:28:20","version" => "2.46"},{"date" => "2012-02-16T12:40:04","version" => "2.47"},{"date" => "2012-02-24T21:06:10","version" => "2.48"},{"date" => "2012-04-17T23:48:26","version" => "2.49"},{"date" => "2012-09-04T16:22:21","version" => "2.50"},{"date" => "2012-09-18T03:01:31","version" => "2.51"},{"date" => "2012-09-26T02:54:36","version" => "2.52"},{"date" => "2012-12-10T11:23:34","version" => "2.53"},{"date" => "2012-12-11T15:49:03","version" => "2.54"},{"date" => "2013-07-10T14:51:25","version" => "2.55"},{"date" => "2017-01-15T21:29:11","version" => "2.56"}]},"GD" => {"advisories" => [{"affected_versions" => ["<2.72"],"cves" => ["CVE-2019-6977"],"description" => "gdImageColorMatch in gd_color_match.c in the GD Graphics Library (aka LibGD) 2.2.5, as used in the imagecolormatch function in PHP before 5.6.40, 7.x before 7.1.26, 7.2.x before 7.2.14, and 7.3.x before 7.3.1, has a heap-based buffer overflow. This can be exploited by an attacker who is able to trigger imagecolormatch calls with crafted image data.\n","distribution" => "GD","fixed_versions" => [">=2.72"],"id" => "CPANSA-GD-2019-6977","references" => ["https://nvd.nist.gov/vuln/detail/CVE-2019-6977","https://bugs.php.net/bug.php?id=77270","http://php.net/ChangeLog-7.php","http://php.net/ChangeLog-5.php","http://www.securityfocus.com/bid/106731","https://lists.debian.org/debian-lts-announce/2019/01/msg00028.html","https://www.debian.org/security/2019/dsa-4384","https://usn.ubuntu.com/3900-1/","https://security.netapp.com/advisory/ntap-20190315-0003/","https://security.gentoo.org/glsa/201903-18","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00025.html","http://lists.opensuse.org/opensuse-security-announce/2019-04/msg00031.html","http://packetstormsecurity.com/files/152459/PHP-7.2-imagecolormatch-Out-Of-Band-Heap-Write.html","https://www.exploit-db.com/exploits/46677/","https://access.redhat.com/errata/RHSA-2019:2519","https://access.redhat.com/errata/RHSA-2019:3299","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/TEYUUOW75YD3DENIPYMO263E6NL2NFHI/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/3WRUPZVT2MWFUEMVGTRAGDOBHLNMGK5R/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/TTXSLRZI5BCQT3H5KALG3DHUWUMNPDX2/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/3CZ2QADQTKRHTGB2AHD7J4QQNDLBEMM6/"],"reported" => "2019-01-27","severity" => "high"}],"main_module" => "GD","versions" => [{"date" => "1996-05-17T08:12:00","version" => "1.00"},{"date" => "1996-07-17T10:16:00","version" => "1.01"},{"date" => "1996-09-07T16:53:00","version" => "1.10"},{"date" => "1996-09-09T10:37:00","version" => "1.11"},{"date" => "1996-09-10T12:04:00","version" => "1.12"},{"date" => "1996-09-11T07:27:00","version" => "1.13"},{"date" => "1996-09-12T16:11:00","version" => "1.14"},{"date" => "1997-11-19T21:13:00","version" => "1.15"},{"date" => "1997-12-19T14:26:00","version" => "1.16"},{"date" => "1998-01-16T13:34:00","version" => "1.17"},{"date" => "1998-01-26T08:44:00","version" => "1.18"},{"date" => "1998-03-08T16:43:26","version" => "1.18"},{"date" => "1999-01-31T17:52:34","version" => "1.18"},{"date" => "1999-04-26T20:35:55","version" => "1.19"},{"date" => "1999-06-02T13:44:43","version" => "1.19"},{"date" => "1999-08-31T03:38:46","version" => "1.20"},{"date" => "1999-08-31T14:55:24","version" => "1.21"},{"date" => "1999-09-30T21:46:47","version" => "1.22"},{"date" => "1999-11-11T14:26:14","version" => "1.23"},{"date" => "2000-02-15T19:54:37","version" => "1.24"},{"date" => "2000-02-22T15:20:41","version" => "1.25"},{"date" => "2000-03-18T23:21:50","version" => "1.26"},{"date" => "2000-03-22T19:41:56","version" => "1.27"},{"date" => "2000-06-23T12:15:51","version" => "1.28"},{"date" => "2000-06-23T18:26:31","version" => "1.29"},{"date" => "2000-07-07T02:42:47","version" => "1.30"},{"date" => "2000-11-10T16:00:09","version" => "1.32"},{"date" => "2001-04-05T04:42:53","version" => "1.33"},{"date" => "2001-09-26T05:19:41","version" => "1.31"},{"date" => "2001-12-06T22:57:11","version" => "1.35"},{"date" => "2001-12-17T19:13:23","version" => "1.36"},{"date" => "2001-12-19T21:34:33","version" => "1.37"},{"date" => "2002-01-04T15:33:18","version" => "1.38"},{"date" => "2002-06-12T02:09:05","version" => "1.39"},{"date" => "2002-06-19T12:11:07","version" => "1.40"},{"date" => "2002-07-22T07:33:14","version" => "1.41"},{"date" => "2002-08-09T16:31:00","version" => "2.00"},{"date" => "2002-08-09T16:39:49","version" => "2.01"},{"date" => "2002-10-14T13:07:59","version" => "2.02"},{"date" => "2002-11-01T15:46:28","version" => "2.04"},{"date" => "2002-11-05T00:55:52","version" => "2.041"},{"date" => "2002-11-25T01:35:10","version" => "2.05"},{"date" => "2003-01-08T16:49:15","version" => "2.06"},{"date" => "2003-04-24T05:06:33","version" => "2.07"},{"date" => "2003-10-06T23:04:15","version" => "2.10"},{"date" => "2003-10-07T22:33:21","version" => "2.11"},{"date" => "2004-02-06T14:33:56","version" => "2.12"},{"date" => "2004-07-22T20:32:01","version" => "2.15"},{"date" => "2004-07-27T00:47:05","version" => "2.16"},{"date" => "2004-11-10T19:15:39","version" => "2.17"},{"date" => "2004-11-12T15:19:40","version" => "2.18"},{"date" => "2004-11-16T13:36:22","version" => "2.19"},{"date" => "2005-02-09T18:50:44","version" => "2.21"},{"date" => "2005-03-07T18:09:39","version" => "2.22"},{"date" => "2005-03-09T21:04:40","version" => "2.23"},{"date" => "2005-07-15T18:47:39","version" => "2.25"},{"date" => "2005-08-04T13:34:01","version" => "2.26"},{"date" => "2005-08-06T14:52:27","version" => "2.27"},{"date" => "2005-08-08T17:28:37","version" => "2.28"},{"date" => "2005-10-19T05:44:52","version" => "2.29"},{"date" => "2005-10-19T07:51:48","version" => "2.30"},{"date" => "2006-02-20T19:48:20","version" => "2.31"},{"date" => "2006-03-08T20:19:06","version" => "2.32"},{"date" => "2006-06-01T20:02:57","version" => "2.34"},{"date" => "2006-08-23T15:31:17","version" => "2.35"},{"date" => "2008-04-21T14:15:26","version" => "2.39"},{"date" => "2008-08-07T18:48:46","version" => "2.40"},{"date" => "2008-08-07T19:17:19","version" => "2.41"},{"date" => "2009-06-10T14:44:33","version" => "2.43"},{"date" => "2009-07-10T18:12:58","version" => "2.44"},{"date" => "2010-04-30T18:52:21","version" => "2.45"},{"date" => "2011-05-01T17:47:22","version" => "2.46"},{"date" => "2013-02-26T10:54:32","version" => "2.48"},{"date" => "2013-02-26T11:04:16","version" => "2.49"},{"date" => "2013-07-02T20:48:59","version" => "2.50"},{"date" => "2014-02-04T16:53:54","version" => "2.51"},{"date" => "2014-02-19T04:29:23","version" => "2.52"},{"date" => "2014-04-01T14:26:31","version" => "2.53"},{"date" => "2014-10-27T02:29:14","version" => "2.55"},{"date" => "2014-10-28T01:35:39","version" => "2.56"},{"date" => "2017-04-19T14:45:56","version" => "2.56_01"},{"date" => "2017-04-19T14:56:35","version" => "2.56_02"},{"date" => "2017-04-19T15:19:33","version" => "2.56_03"},{"date" => "2017-04-21T06:22:54","version" => "2.57"},{"date" => "2017-04-21T08:38:24","version" => "2.58"},{"date" => "2017-04-21T10:35:21","version" => "2.59"},{"date" => "2017-04-21T22:11:18","version" => "2.60"},{"date" => "2017-04-22T15:52:15","version" => "2.61"},{"date" => "2017-04-22T22:27:02","version" => "2.62"},{"date" => "2017-04-23T08:23:06","version" => "2.63"},{"date" => "2017-04-23T09:03:18","version" => "2.64"},{"date" => "2017-04-23T10:36:23","version" => "2.65"},{"date" => "2017-04-23T13:08:08","version" => "2.66"},{"date" => "2017-11-15T08:33:16","version" => "2.67"},{"date" => "2018-02-18T19:56:41","version" => "2.68"},{"date" => "2018-08-26T15:40:02","version" => "2.69"},{"date" => "2019-01-10T12:57:08","version" => "2.70"},{"date" => "2019-02-12T11:53:42","version" => "2.71"},{"date" => "2020-07-18T06:02:50","version" => "2.72"},{"date" => "2020-09-24T13:01:57","version" => "2.73"},{"date" => "2022-01-23T15:34:48","version" => "2.74"},{"date" => "2022-01-25T16:54:22","version" => "2.75"},{"date" => "2022-02-01T14:50:45","version" => "2.76"},{"date" => "2023-05-29T07:10:47","version" => "2.77"},{"date" => "2023-07-04T09:13:52","version" => "2.78"},{"date" => "2024-04-29T19:57:07","version" => "2.79"},{"date" => "2024-05-03T11:16:48","version" => "2.80"},{"date" => "2024-05-03T17:02:57","version" => "2.81"},{"date" => "2024-05-27T10:31:38","version" => "2.82"},{"date" => "2024-06-23T15:46:01","version" => "2.83"},{"date" => "2026-01-04T19:40:59","version" => "2.84"}]},"GPIB" => {"advisories" => [{"affected_versions" => [">0"],"cves" => ["CVE-2006-1565"],"description" => "Untrusted search path vulnerability in libgpib-perl 3.2.06-2 in Debian GNU/Linux includes an RPATH value under the /tmp/buildd directory for the LinuxGpib.so module, which might allow local users to gain privileges by installing malicious libraries in that directory.\n","distribution" => "GPIB","fixed_versions" => [],"id" => "CPANSA-GPIB-2006-1565","references" => ["http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=359239","http://www.securityfocus.com/bid/17288","https://exchange.xforce.ibmcloud.com/vulnerabilities/25681"],"reported" => "2006-03-31","severity" => undef}],"main_module" => "GPIB","versions" => [{"date" => "2002-01-02T03:13:38","version" => "0_30"}]},"Galileo" => {"advisories" => [{"affected_versions" => ["<0.043"],"cves" => ["CVE-2019-7410"],"description" => "There is stored cross site scripting (XSS) in Galileo CMS v0.042. Remote authenticated users could inject arbitrary web script or HTML via \$page_title in /lib/Galileo/files/templates/page/show.html.ep (aka the PAGE TITLE Field).\n","distribution" => "Galileo","fixed_versions" => [">=0.043"],"id" => "CPANSA-Galileo-2019-7410","references" => ["https://metamorfosec.com/Files/Commits/METC-2020-002-Escape_banner_in_Galileo_CMS_v0.042.txt","https://metacpan.org/changes/distribution/Galileo","https://github.com/jberger/Galileo/pull/55/files","https://metamorfosec.com/Files/Advisories/METS-2020-002-A_Stored_XSS_Vulnerability_in_Galileo_CMS_v0.042.txt"],"reported" => "2020-08-14","severity" => "medium"}],"main_module" => "Galileo","versions" => [{"date" => "2012-07-30T17:03:42","version" => "0.001"},{"date" => "2012-07-30T17:38:19","version" => "0.002"},{"date" => "2012-07-30T20:36:06","version" => "0.003"},{"date" => "2012-07-31T15:41:59","version" => "0.004"},{"date" => "2012-08-06T21:26:17","version" => "0.005"},{"date" => "2012-08-09T20:39:14","version" => "0.006"},{"date" => "2012-09-10T17:57:20","version" => "0.007"},{"date" => "2012-10-27T17:40:15","version" => "0.008"},{"date" => "2012-10-28T18:03:00","version" => "0.009"},{"date" => "2012-10-28T18:10:15","version" => "0.01"},{"date" => "2012-10-31T22:10:19","version" => "0.011"},{"date" => "2013-01-15T15:10:42","version" => "0.012"},{"date" => "2013-01-15T21:07:00","version" => "0.013"},{"date" => "2013-01-16T21:37:32","version" => "0.014"},{"date" => "2013-01-19T03:59:45","version" => "0.015"},{"date" => "2013-01-26T21:28:59","version" => "0.016"},{"date" => "2013-02-05T03:32:32","version" => "0.017"},{"date" => "2013-02-06T02:47:47","version" => "0.018"},{"date" => "2013-02-06T03:09:27","version" => "0.019"},{"date" => "2013-02-11T23:33:00","version" => "0.020"},{"date" => "2013-02-17T01:24:51","version" => "0.021"},{"date" => "2013-02-17T02:44:14","version" => "0.022"},{"date" => "2013-03-04T18:25:01","version" => "0.023"},{"date" => "2013-03-12T15:24:22","version" => "0.024"},{"date" => "2013-03-12T18:48:22","version" => "0.025"},{"date" => "2013-03-15T15:18:18","version" => "0.026"},{"date" => "2013-04-03T20:04:15","version" => "0.027"},{"date" => "2013-05-14T15:59:46","version" => "0.028"},{"date" => "2013-06-29T03:30:18","version" => "0.029"},{"date" => "2013-08-27T03:43:39","version" => "0.030"},{"date" => "2013-11-28T18:36:10","version" => "0.031"},{"date" => "2014-04-06T16:17:22","version" => "0.032"},{"date" => "2014-05-10T19:38:50","version" => "0.033"},{"date" => "2014-05-31T13:06:42","version" => "0.034"},{"date" => "2014-08-16T22:10:46","version" => "0.035"},{"date" => "2014-08-31T15:31:15","version" => "0.036"},{"date" => "2014-10-14T04:03:53","version" => "0.037"},{"date" => "2015-01-25T18:08:54","version" => "0.038"},{"date" => "2015-09-28T18:25:31","version" => "0.039"},{"date" => "2016-01-07T16:33:46","version" => "0.040"},{"date" => "2016-08-13T18:15:17","version" => "0.041"},{"date" => "2017-03-16T03:14:04","version" => "0.042"},{"date" => "2020-08-06T16:26:58","version" => "0.043"}]},"Git-Raw" => {"advisories" => [{"affected_versions" => [">=0.08,<=0.24"],"cves" => ["X-libgit2-2015-0001"],"description" => "On the heels of CVE 2014-9390, we are announcing another round of security updates to libgit2. Similar to the prior vulnerability, an attacker can construct a git commit that, when checked out, may cause files to be written to your .git directory which may lead to arbitrary code execution.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-X-libgit2-2015-0001-libgit2","references" => ["https://www.edwardthomson.com/blog/another-libgit2-security-update.html"],"reported" => "2015-01-20","severity" => "critical"},{"affected_versions" => [">=0.25,<=0.28"],"cves" => ["X-libgit2-2015-0001"],"description" => "On the heels of CVE 2014-9390, we are announcing another round of security updates to libgit2. Similar to the prior vulnerability, an attacker can construct a git commit that, when checked out, may cause files to be written to your .git directory which may lead to arbitrary code execution.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-X-libgit2-2015-0001-libgit2","references" => ["https://www.edwardthomson.com/blog/another-libgit2-security-update.html"],"reported" => "2015-01-20","severity" => "critical"},{"affected_versions" => [">=0.29,<=0.40"],"cves" => ["X-libgit2-2015-0001"],"description" => "On the heels of CVE 2014-9390, we are announcing another round of security updates to libgit2. Similar to the prior vulnerability, an attacker can construct a git commit that, when checked out, may cause files to be written to your .git directory which may lead to arbitrary code execution.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-X-libgit2-2015-0001-libgit2","references" => ["https://www.edwardthomson.com/blog/another-libgit2-security-update.html"],"reported" => "2015-01-20","severity" => "critical"},{"affected_versions" => [">=0.41,<=0.50"],"cves" => ["X-libgit2-2015-0001"],"description" => "On the heels of CVE 2014-9390, we are announcing another round of security updates to libgit2. Similar to the prior vulnerability, an attacker can construct a git commit that, when checked out, may cause files to be written to your .git directory which may lead to arbitrary code execution.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-X-libgit2-2015-0001-libgit2","references" => ["https://www.edwardthomson.com/blog/another-libgit2-security-update.html"],"reported" => "2015-01-20","severity" => "critical"},{"affected_versions" => [">=0.51,<=0.53"],"cves" => ["X-libgit2-2015-0001"],"description" => "On the heels of CVE 2014-9390, we are announcing another round of security updates to libgit2. Similar to the prior vulnerability, an attacker can construct a git commit that, when checked out, may cause files to be written to your .git directory which may lead to arbitrary code execution.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-X-libgit2-2015-0001-libgit2","references" => ["https://www.edwardthomson.com/blog/another-libgit2-security-update.html"],"reported" => "2015-01-20","severity" => "critical"},{"affected_versions" => [">=0.54,<=0.58"],"cves" => ["X-libgit2-2015-0001"],"description" => "On the heels of CVE 2014-9390, we are announcing another round of security updates to libgit2. Similar to the prior vulnerability, an attacker can construct a git commit that, when checked out, may cause files to be written to your .git directory which may lead to arbitrary code execution.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-X-libgit2-2015-0001-libgit2","references" => ["https://www.edwardthomson.com/blog/another-libgit2-security-update.html"],"reported" => "2015-01-20","severity" => "critical"},{"affected_versions" => [">=0.59,<=0.60"],"cves" => ["X-libgit2-2015-0001"],"description" => "On the heels of CVE 2014-9390, we are announcing another round of security updates to libgit2. Similar to the prior vulnerability, an attacker can construct a git commit that, when checked out, may cause files to be written to your .git directory which may lead to arbitrary code execution.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-X-libgit2-2015-0001-libgit2","references" => ["https://www.edwardthomson.com/blog/another-libgit2-security-update.html"],"reported" => "2015-01-20","severity" => "critical"},{"affected_versions" => [">=0.61,<=0.74"],"cves" => ["X-libgit2-2015-0001"],"description" => "On the heels of CVE 2014-9390, we are announcing another round of security updates to libgit2. Similar to the prior vulnerability, an attacker can construct a git commit that, when checked out, may cause files to be written to your .git directory which may lead to arbitrary code execution.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-X-libgit2-2015-0001-libgit2","references" => ["https://www.edwardthomson.com/blog/another-libgit2-security-update.html"],"reported" => "2015-01-20","severity" => "critical"},{"affected_versions" => ["==0.75"],"cves" => ["X-libgit2-2015-0001"],"description" => "On the heels of CVE 2014-9390, we are announcing another round of security updates to libgit2. Similar to the prior vulnerability, an attacker can construct a git commit that, when checked out, may cause files to be written to your .git directory which may lead to arbitrary code execution.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-X-libgit2-2015-0001-libgit2","references" => ["https://www.edwardthomson.com/blog/another-libgit2-security-update.html"],"reported" => "2015-01-20","severity" => "critical"},{"affected_versions" => [">=0.76,<=0.82"],"cves" => ["X-libgit2-2015-0001"],"description" => "On the heels of CVE 2014-9390, we are announcing another round of security updates to libgit2. Similar to the prior vulnerability, an attacker can construct a git commit that, when checked out, may cause files to be written to your .git directory which may lead to arbitrary code execution.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-X-libgit2-2015-0001-libgit2","references" => ["https://www.edwardthomson.com/blog/another-libgit2-security-update.html"],"reported" => "2015-01-20","severity" => "critical"},{"affected_versions" => [">=0.83,<=0.84"],"cves" => ["X-libgit2-2015-0001"],"description" => "On the heels of CVE 2014-9390, we are announcing another round of security updates to libgit2. Similar to the prior vulnerability, an attacker can construct a git commit that, when checked out, may cause files to be written to your .git directory which may lead to arbitrary code execution.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-X-libgit2-2015-0001-libgit2","references" => ["https://www.edwardthomson.com/blog/another-libgit2-security-update.html"],"reported" => "2015-01-20","severity" => "critical"},{"affected_versions" => [">=0.85,<=0.87"],"cves" => ["X-libgit2-2015-0001"],"description" => "On the heels of CVE 2014-9390, we are announcing another round of security updates to libgit2. Similar to the prior vulnerability, an attacker can construct a git commit that, when checked out, may cause files to be written to your .git directory which may lead to arbitrary code execution.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-X-libgit2-2015-0001-libgit2","references" => ["https://www.edwardthomson.com/blog/another-libgit2-security-update.html"],"reported" => "2015-01-20","severity" => "critical"},{"affected_versions" => ["==0.88"],"cves" => ["X-libgit2-2015-0001"],"description" => "On the heels of CVE 2014-9390, we are announcing another round of security updates to libgit2. Similar to the prior vulnerability, an attacker can construct a git commit that, when checked out, may cause files to be written to your .git directory which may lead to arbitrary code execution.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-X-libgit2-2015-0001-libgit2","references" => ["https://www.edwardthomson.com/blog/another-libgit2-security-update.html"],"reported" => "2015-01-20","severity" => "critical"},{"affected_versions" => ["==0.89"],"cves" => ["X-libgit2-2015-0001"],"description" => "On the heels of CVE 2014-9390, we are announcing another round of security updates to libgit2. Similar to the prior vulnerability, an attacker can construct a git commit that, when checked out, may cause files to be written to your .git directory which may lead to arbitrary code execution.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-X-libgit2-2015-0001-libgit2","references" => ["https://www.edwardthomson.com/blog/another-libgit2-security-update.html"],"reported" => "2015-01-20","severity" => "critical"},{"affected_versions" => [">=0.08,<=0.40"],"cves" => ["CVE-2018-25032"],"description" => "zlib before 1.2.12 allows memory corruption when deflating (i.e., when compressing) if the input has many distant matches.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-2018-25032-zlib","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143579","https://www.openwall.com/lists/oss-security/2022/03/24/1","https://github.com/madler/zlib/commit/5c44459c3b28a9bd3283aaceab7c615f8020c531","http://www.openwall.com/lists/oss-security/2022/03/25/2","http://www.openwall.com/lists/oss-security/2022/03/26/1","https://www.openwall.com/lists/oss-security/2022/03/28/1","https://github.com/madler/zlib/compare/v1.2.11...v1.2.12","https://www.openwall.com/lists/oss-security/2022/03/28/3","https://github.com/madler/zlib/issues/605","https://www.debian.org/security/2022/dsa-5111","https://lists.debian.org/debian-lts-announce/2022/04/msg00000.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/NS2D2GFPFGOJUL4WQ3DUAY7HF4VWQ77F/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/XOKFMSNQ5D5WGMALBNBXU3GE442V74WU/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/VOKNP2L734AEL47NRYGVZIKEFOUBQY5Y/","https://lists.debian.org/debian-lts-announce/2022/05/msg00008.html","https://support.apple.com/kb/HT213255","https://support.apple.com/kb/HT213256","https://support.apple.com/kb/HT213257","http://seclists.org/fulldisclosure/2022/May/33","http://seclists.org/fulldisclosure/2022/May/35","http://seclists.org/fulldisclosure/2022/May/38","https://security.netapp.com/advisory/ntap-20220526-0009/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DF62MVMH3QUGMBDCB3DY2ERQ6EBHTADB/"],"reported" => "2022-03-25","severity" => "high"},{"affected_versions" => [">=0.41,<=0.75"],"cves" => ["CVE-2018-25032"],"description" => "zlib before 1.2.12 allows memory corruption when deflating (i.e., when compressing) if the input has many distant matches.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-2018-25032-zlib","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143579","https://www.openwall.com/lists/oss-security/2022/03/24/1","https://github.com/madler/zlib/commit/5c44459c3b28a9bd3283aaceab7c615f8020c531","http://www.openwall.com/lists/oss-security/2022/03/25/2","http://www.openwall.com/lists/oss-security/2022/03/26/1","https://www.openwall.com/lists/oss-security/2022/03/28/1","https://github.com/madler/zlib/compare/v1.2.11...v1.2.12","https://www.openwall.com/lists/oss-security/2022/03/28/3","https://github.com/madler/zlib/issues/605","https://www.debian.org/security/2022/dsa-5111","https://lists.debian.org/debian-lts-announce/2022/04/msg00000.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/NS2D2GFPFGOJUL4WQ3DUAY7HF4VWQ77F/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/XOKFMSNQ5D5WGMALBNBXU3GE442V74WU/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/VOKNP2L734AEL47NRYGVZIKEFOUBQY5Y/","https://lists.debian.org/debian-lts-announce/2022/05/msg00008.html","https://support.apple.com/kb/HT213255","https://support.apple.com/kb/HT213256","https://support.apple.com/kb/HT213257","http://seclists.org/fulldisclosure/2022/May/33","http://seclists.org/fulldisclosure/2022/May/35","http://seclists.org/fulldisclosure/2022/May/38","https://security.netapp.com/advisory/ntap-20220526-0009/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DF62MVMH3QUGMBDCB3DY2ERQ6EBHTADB/"],"reported" => "2022-03-25","severity" => "high"},{"affected_versions" => [">=0.76,<=0.88"],"cves" => ["CVE-2018-25032"],"description" => "zlib before 1.2.12 allows memory corruption when deflating (i.e., when compressing) if the input has many distant matches.\n","distribution" => "Git-Raw","fixed_versions" => [],"id" => "CPANSA-Git-Raw-2018-25032-zlib","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143579","https://www.openwall.com/lists/oss-security/2022/03/24/1","https://github.com/madler/zlib/commit/5c44459c3b28a9bd3283aaceab7c615f8020c531","http://www.openwall.com/lists/oss-security/2022/03/25/2","http://www.openwall.com/lists/oss-security/2022/03/26/1","https://www.openwall.com/lists/oss-security/2022/03/28/1","https://github.com/madler/zlib/compare/v1.2.11...v1.2.12","https://www.openwall.com/lists/oss-security/2022/03/28/3","https://github.com/madler/zlib/issues/605","https://www.debian.org/security/2022/dsa-5111","https://lists.debian.org/debian-lts-announce/2022/04/msg00000.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/NS2D2GFPFGOJUL4WQ3DUAY7HF4VWQ77F/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/XOKFMSNQ5D5WGMALBNBXU3GE442V74WU/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/VOKNP2L734AEL47NRYGVZIKEFOUBQY5Y/","https://lists.debian.org/debian-lts-announce/2022/05/msg00008.html","https://support.apple.com/kb/HT213255","https://support.apple.com/kb/HT213256","https://support.apple.com/kb/HT213257","http://seclists.org/fulldisclosure/2022/May/33","http://seclists.org/fulldisclosure/2022/May/35","http://seclists.org/fulldisclosure/2022/May/38","https://security.netapp.com/advisory/ntap-20220526-0009/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DF62MVMH3QUGMBDCB3DY2ERQ6EBHTADB/"],"reported" => "2022-03-25","severity" => "high"}],"main_module" => "Git::Raw","versions" => [{"date" => "2012-08-05T11:44:00","version" => "0.01"},{"date" => "2012-08-05T20:19:35","version" => "0.02"},{"date" => "2012-08-06T16:46:15","version" => "0.03"},{"date" => "2012-08-08T21:43:23","version" => "0.04"},{"date" => "2012-08-09T15:57:26","version" => "0.05"},{"date" => "2012-08-17T14:13:37","version" => "0.06"},{"date" => "2012-08-20T19:03:21","version" => "0.07"},{"date" => "2012-08-21T15:15:44","version" => "0.08"},{"date" => "2012-08-22T17:57:45","version" => "0.09"},{"date" => "2012-08-22T18:54:02","version" => "0.10"},{"date" => "2012-08-23T09:42:11","version" => "0.11"},{"date" => "2012-09-22T17:26:43","version" => "0.12"},{"date" => "2012-10-02T16:28:54","version" => "0.13"},{"date" => "2012-10-19T10:29:16","version" => "0.14"},{"date" => "2012-10-30T21:41:51","version" => "0.15"},{"date" => "2012-11-30T11:44:22","version" => "0.16"},{"date" => "2012-12-04T12:50:53","version" => "0.17"},{"date" => "2012-12-14T17:41:07","version" => "0.18"},{"date" => "2013-01-03T18:26:04","version" => "0.19"},{"date" => "2013-01-26T12:47:33","version" => "0.20"},{"date" => "2013-02-20T16:43:28","version" => "0.21"},{"date" => "2013-02-23T14:21:03","version" => "0.22"},{"date" => "2013-03-25T13:12:14","version" => "0.23"},{"date" => "2013-03-31T11:09:03","version" => "0.24"},{"date" => "2013-10-09T13:20:24","version" => "0.25"},{"date" => "2013-10-09T14:13:40","version" => "0.26"},{"date" => "2013-10-23T10:37:15","version" => "0.27"},{"date" => "2013-11-18T14:02:54","version" => "0.28"},{"date" => "2013-11-24T19:20:25","version" => "0.29"},{"date" => "2014-01-26T13:41:14","version" => "0.30"},{"date" => "2014-02-08T18:10:43","version" => "0.31"},{"date" => "2014-03-16T14:06:42","version" => "0.32"},{"date" => "2014-04-14T10:32:15","version" => "0.33"},{"date" => "2014-04-27T10:41:00","version" => "0.34"},{"date" => "2014-04-28T16:45:42","version" => "0.35"},{"date" => "2014-05-02T14:01:36","version" => "0.36"},{"date" => "2014-06-08T15:27:13","version" => "0.37"},{"date" => "2014-06-11T18:05:33","version" => "0.38"},{"date" => "2014-06-12T16:08:55","version" => "0.39"},{"date" => "2014-06-25T19:08:19","version" => "0.40"},{"date" => "2014-08-03T16:09:58","version" => "0.41"},{"date" => "2014-08-09T13:22:08","version" => "0.42"},{"date" => "2014-08-14T13:21:12","version" => "0.43"},{"date" => "2014-08-18T08:35:38","version" => "0.44"},{"date" => "2014-09-18T21:31:44","version" => "0.45"},{"date" => "2014-09-19T10:44:04","version" => "0.46"},{"date" => "2014-09-30T11:08:31","version" => "0.47"},{"date" => "2014-10-13T21:10:25","version" => "0.48"},{"date" => "2014-10-24T10:06:17","version" => "0.49"},{"date" => "2014-11-15T18:12:53","version" => "0.50"},{"date" => "2015-02-09T07:56:23","version" => "0.51"},{"date" => "2015-03-19T11:47:40","version" => "0.52"},{"date" => "2015-04-14T18:26:22","version" => "0.53"},{"date" => "2015-11-12T19:30:27","version" => "0.54"},{"date" => "2015-11-14T09:21:11","version" => "0.55"},{"date" => "2015-11-17T11:54:04","version" => "0.56"},{"date" => "2015-11-21T13:30:25","version" => "0.57"},{"date" => "2015-11-23T05:52:12","version" => "0.58"},{"date" => "2016-05-23T04:45:30","version" => "0.59"},{"date" => "2016-06-09T17:50:00","version" => "0.60"},{"date" => "2016-12-05T17:51:20","version" => "0.61"},{"date" => "2016-12-06T16:59:22","version" => "0.62"},{"date" => "2016-12-08T18:31:51","version" => "0.63"},{"date" => "2016-12-16T12:56:55","version" => "0.64"},{"date" => "2016-12-21T16:02:45","version" => "0.65"},{"date" => "2016-12-28T16:06:29","version" => "0.66"},{"date" => "2016-12-28T17:03:40","version" => "0.67"},{"date" => "2016-12-30T08:07:24","version" => "0.68"},{"date" => "2016-12-30T08:11:44","version" => "0.69"},{"date" => "2016-12-30T19:19:00","version" => "0.70"},{"date" => "2017-01-09T06:53:53","version" => "0.71"},{"date" => "2017-01-10T05:12:24","version" => "0.72"},{"date" => "2017-03-22T16:43:32","version" => "0.73"},{"date" => "2017-03-24T09:07:21","version" => "0.74"},{"date" => "2018-01-25T18:54:11","version" => "0.75"},{"date" => "2018-03-08T16:00:17","version" => "0.76"},{"date" => "2018-03-09T04:57:30","version" => "0.77"},{"date" => "2018-03-09T13:30:01","version" => "0.78"},{"date" => "2018-03-23T18:40:02","version" => "0.79"},{"date" => "2018-06-17T08:47:43","version" => "0.80"},{"date" => "2018-06-27T17:23:13","version" => "0.81"},{"date" => "2018-12-12T15:18:03","version" => "0.82"},{"date" => "2019-05-20T13:42:02","version" => "0.83"},{"date" => "2019-08-19T20:36:03","version" => "0.84"},{"date" => "2020-04-19T11:32:47","version" => "0.85"},{"date" => "2020-04-25T11:27:33","version" => "0.86"},{"date" => "2020-08-30T12:19:25","version" => "0.87"},{"date" => "2021-08-08T12:37:22","version" => "0.88"},{"date" => "2022-10-23T16:31:07","version" => "0.89"},{"date" => "2022-10-27T08:52:11","version" => "0.90"}]},"Git-XS" => {"advisories" => [{"affected_versions" => [">=0.01,<=0.02"],"cves" => ["CVE-2018-7159"],"description" => "The HTTP parser in all current versions of Node.js ignores spaces in the `Content-Length` header, allowing input such as `Content-Length: 1 2` to be interpreted as having a value of `12`. The HTTP specification does not allow for spaces in the `Content-Length` value and the Node.js HTTP parser has been brought into line on this particular difference. The security risk of this flaw to Node.js users is considered to be VERY LOW as it is difficult, and may be impossible, to craft an attack that makes use of this flaw in a way that could not already be achieved by supplying an incorrect value for `Content-Length`. Vulnerabilities may exist in user-code that make incorrect assumptions about the potential accuracy of this value compared to the actual length of the data supplied. Node.js users crafting lower-level HTTP utilities are advised to re-check the length of any input supplied after parsing is complete.'\n","distribution" => "Git-XS","fixed_versions" => [],"id" => "CPANSA-Git-XS-2018-7159-http-parser","references" => ["https://nodejs.org/en/blog/vulnerability/march-2018-security-releases/","https://access.redhat.com/errata/RHSA-2019:2258","https://support.f5.com/csp/article/K27228191?utm_source=f5support&amp;utm_medium=RSS"],"reported" => "2018-05-17","severity" => "medium"},{"affected_versions" => [">=0.01,<=0.02"],"cves" => ["CVE-2018-12121"],"description" => "Denial of Service with large HTTP headers by using a combination of many requests with maximum sized headers (almost 80 KB per connection), and carefully timed completion of the headers, it is possible to cause the HTTP server to abort from heap allocation failure. Attack potential is mitigated by the use of a load balancer or other proxy layer.\n","distribution" => "Git-XS","fixed_versions" => [],"id" => "CPANSA-Git-XS-2018-12121-http-parser","references" => ["https://nodejs.org/en/blog/vulnerability/november-2018-security-releases/","http://www.securityfocus.com/bid/106043","https://access.redhat.com/errata/RHSA-2019:1821","https://access.redhat.com/errata/RHSA-2019:2258","https://access.redhat.com/errata/RHSA-2019:3497","https://security.gentoo.org/glsa/202003-48"],"reported" => "2018-11-28","severity" => "high"},{"affected_versions" => [">=0.08,<=0.17"],"cves" => ["CVE-2018-7159"],"description" => "The HTTP parser in all current versions of Node.js ignores spaces in the `Content-Length` header, allowing input such as `Content-Length: 1 2` to be interpreted as having a value of `12`. The HTTP specification does not allow for spaces in the `Content-Length` value and the Node.js HTTP parser has been brought into line on this particular difference. The security risk of this flaw to Node.js users is considered to be VERY LOW as it is difficult, and may be impossible, to craft an attack that makes use of this flaw in a way that could not already be achieved by supplying an incorrect value for `Content-Length`. Vulnerabilities may exist in user-code that make incorrect assumptions about the potential accuracy of this value compared to the actual length of the data supplied. Node.js users crafting lower-level HTTP utilities are advised to re-check the length of any input supplied after parsing is complete.'\n","distribution" => "Git-XS","fixed_versions" => [],"id" => "CPANSA-Git-XS-2018-7159-http-parser","references" => ["https://nodejs.org/en/blog/vulnerability/march-2018-security-releases/","https://access.redhat.com/errata/RHSA-2019:2258","https://support.f5.com/csp/article/K27228191?utm_source=f5support&amp;utm_medium=RSS"],"reported" => "2018-05-17","severity" => "medium"},{"affected_versions" => [">=0.08,<=0.17"],"cves" => ["CVE-2018-12121"],"description" => "Denial of Service with large HTTP headers by using a combination of many requests with maximum sized headers (almost 80 KB per connection), and carefully timed completion of the headers, it is possible to cause the HTTP server to abort from heap allocation failure. Attack potential is mitigated by the use of a load balancer or other proxy layer.\n","distribution" => "Git-XS","fixed_versions" => [],"id" => "CPANSA-Git-XS-2018-12121-http-parser","references" => ["https://nodejs.org/en/blog/vulnerability/november-2018-security-releases/","http://www.securityfocus.com/bid/106043","https://access.redhat.com/errata/RHSA-2019:1821","https://access.redhat.com/errata/RHSA-2019:2258","https://access.redhat.com/errata/RHSA-2019:3497","https://security.gentoo.org/glsa/202003-48"],"reported" => "2018-11-28","severity" => "high"},{"affected_versions" => [">=0.17,<=0.88"],"cves" => ["CVE-2018-7159"],"description" => "The HTTP parser in all current versions of Node.js ignores spaces in the `Content-Length` header, allowing input such as `Content-Length: 1 2` to be interpreted as having a value of `12`. The HTTP specification does not allow for spaces in the `Content-Length` value and the Node.js HTTP parser has been brought into line on this particular difference. The security risk of this flaw to Node.js users is considered to be VERY LOW as it is difficult, and may be impossible, to craft an attack that makes use of this flaw in a way that could not already be achieved by supplying an incorrect value for `Content-Length`. Vulnerabilities may exist in user-code that make incorrect assumptions about the potential accuracy of this value compared to the actual length of the data supplied. Node.js users crafting lower-level HTTP utilities are advised to re-check the length of any input supplied after parsing is complete.'\n","distribution" => "Git-XS","fixed_versions" => [],"id" => "CPANSA-Git-XS-2018-7159-http-parser","references" => ["https://nodejs.org/en/blog/vulnerability/march-2018-security-releases/","https://access.redhat.com/errata/RHSA-2019:2258","https://support.f5.com/csp/article/K27228191?utm_source=f5support&amp;utm_medium=RSS"],"reported" => "2018-05-17","severity" => "medium"},{"affected_versions" => [">=0.17,<=0.88"],"cves" => ["CVE-2018-12121"],"description" => "Denial of Service with large HTTP headers by using a combination of many requests with maximum sized headers (almost 80 KB per connection), and carefully timed completion of the headers, it is possible to cause the HTTP server to abort from heap allocation failure. Attack potential is mitigated by the use of a load balancer or other proxy layer.\n","distribution" => "Git-XS","fixed_versions" => [],"id" => "CPANSA-Git-XS-2018-12121-http-parser","references" => ["https://nodejs.org/en/blog/vulnerability/november-2018-security-releases/","http://www.securityfocus.com/bid/106043","https://access.redhat.com/errata/RHSA-2019:1821","https://access.redhat.com/errata/RHSA-2019:2258","https://access.redhat.com/errata/RHSA-2019:3497","https://security.gentoo.org/glsa/202003-48"],"reported" => "2018-11-28","severity" => "high"},{"affected_versions" => [">=0.01,<=0.02"],"cves" => ["X-libgit2-2015-0001"],"description" => "On the heels of CVE 2014-9390, we are announcing another round of security updates to libgit2. Similar to the prior vulnerability, an attacker can construct a git commit that, when checked out, may cause files to be written to your .git directory which may lead to arbitrary code execution.\n","distribution" => "Git-XS","fixed_versions" => [],"id" => "CPANSA-Git-XS-X-libgit2-2015-0001-libgit2","references" => ["https://www.edwardthomson.com/blog/another-libgit2-security-update.html"],"reported" => "2015-01-20","severity" => "critical"},{"affected_versions" => [">=0.01,<=0.02"],"cves" => ["CVE-2018-25032"],"description" => "zlib before 1.2.12 allows memory corruption when deflating (i.e., when compressing) if the input has many distant matches.\n","distribution" => "Git-XS","fixed_versions" => [],"id" => "CPANSA-Git-XS-2018-25032-zlib","references" => ["https://rt.cpan.org/Ticket/Display.html?id=143579","https://www.openwall.com/lists/oss-security/2022/03/24/1","https://github.com/madler/zlib/commit/5c44459c3b28a9bd3283aaceab7c615f8020c531","http://www.openwall.com/lists/oss-security/2022/03/25/2","http://www.openwall.com/lists/oss-security/2022/03/26/1","https://www.openwall.com/lists/oss-security/2022/03/28/1","https://github.com/madler/zlib/compare/v1.2.11...v1.2.12","https://www.openwall.com/lists/oss-security/2022/03/28/3","https://github.com/madler/zlib/issues/605","https://www.debian.org/security/2022/dsa-5111","https://lists.debian.org/debian-lts-announce/2022/04/msg00000.html","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/NS2D2GFPFGOJUL4WQ3DUAY7HF4VWQ77F/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/XOKFMSNQ5D5WGMALBNBXU3GE442V74WU/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/VOKNP2L734AEL47NRYGVZIKEFOUBQY5Y/","https://lists.debian.org/debian-lts-announce/2022/05/msg00008.html","https://support.apple.com/kb/HT213255","https://support.apple.com/kb/HT213256","https://support.apple.com/kb/HT213257","http://seclists.org/fulldisclosure/2022/May/33","http://seclists.org/fulldisclosure/2022/May/35","http://seclists.org/fulldisclosure/2022/May/38","https://security.netapp.com/advisory/ntap-20220526-0009/","https://lists.fedoraproject.org/archives/list/package-announce\@lists.fedoraproject.org/message/DF62MVMH3QUGMBDCB3DY2ERQ6EBHTADB/"],"reported" => "2022-03-25","severity" => "high"}],"main_module" => "Git::XS","versions" => [{"date" => "2011-12-27T05:42:38","version" => "0.01"},{"date" => "2011-12-27T23:09:56","version" => "0.02"}]},"GitLab-API-v4" => {"advisories" => [{"affected_versions" => [">=0.26"],"cves" => ["CVE-2023-31485"],"description" => "GitLab::API::v4 is missing the verify_SSL=>1 flag in HTTP::Tiny, allowing a network attacker to MITM connections to the GitLab server.\n","distribution" => "GitLab-API-v4","fixed_versions" => [],"id" => "CPANSA-GitLab-API-v4-2023-31485","references" => ["https://github.com/bluefeet/GitLab-API-v4/pull/57","https://blog.hackeriet.no/perl-http-tiny-insecure-tls-default-affects-cpan-modules/","https://www.openwall.com/lists/oss-security/2023/04/18/14","https://github.com/chansen/p5-http-tiny/pull/151"],"reported" => "2023-02-28"}],"main_module" => "GitLab::API::v4","versions" => [{"date" => "2017-12-11T18:40:52","version" => "0.01"},{"date" => "2017-12-16T00:11:18","version" => "0.02"},{"date" => "2018-01-12T21:26:23","version" => "0.03"},{"date" => "2018-02-03T23:15:40","version" => "0.04"},{"date" => "2018-03-06T18:26:52","version" => "0.05"},{"date" => "2018-04-09T20:06:01","version" => "0.06"},{"date" => "2018-05-10T21:33:05","version" => "0.07"},{"date" => "2018-05-14T17:33:39","version" => "0.08"},{"date" => "2018-05-29T04:48:13","version" => "0.09"},{"date" => "2018-06-06T06:47:41","version" => "0.10"},{"date" => "2018-09-02T05:24:51","version" => "0.11"},{"date" => "2018-09-12T01:04:05","version" => "0.12"},{"date" => "2018-11-08T13:44:56","version" => "0.13"},{"date" => "2018-12-04T20:39:42","version" => "0.14"},{"date" => "2019-01-09T18:13:46","version" => "0.15"},{"date" => "2019-02-18T06:21:38","version" => "0.16"},{"date" => "2019-02-20T22:49:13","version" => "0.17"},{"date" => "2019-04-01T04:36:27","version" => "0.18"},{"date" => "2019-05-17T20:41:31","version" => "0.19"},{"date" => "2019-07-23T21:42:57","version" => "0.20"},{"date" => "2019-08-24T18:56:25","version" => "0.21"},{"date" => "2019-09-13T15:03:00","version" => "0.22"},{"date" => "2019-10-18T19:55:54","version" => "0.23"},{"date" => "2020-02-12T22:10:58","version" => "0.24"},{"date" => "2020-02-12T22:21:40","version" => "0.25"},{"date" => "2021-01-30T07:11:26","version" => "0.26"},{"date" => "2023-06-07T20:51:14","version" => "0.27"}]},"Graphics-ColorNames" => {"advisories" => [{"affected_versions" => [">=2.0_01,<=3.1.2"],"cves" => ["CVE-2024-55918"],"description" => "A specially-named file may lead to HTML injection attacks.\n","distribution" => "Graphics-ColorNames","fixed_versions" => [">3.1.2"],"id" => "CPANSA-Graphics-ColorNames-2010-02","references" => ["https://metacpan.org/changes/distribution/Graphics-ColorNames","https://rt.cpan.org/Public/Bug/Display.html?id=54500"],"reported" => "2010-02-11"}],"main_module" => "Graphics::ColorNames","versions" => [{"date" => "2001-02-20T03:47:48","version" => "0.10"},{"date" => "2001-04-12T02:32:22","version" => "0.20"},{"date" => "2001-04-13T04:37:27","version" => "0.21"},{"date" => "2001-04-15T14:26:41","version" => "0.22"},{"date" => "2001-04-18T03:13:51","version" => "0.23"},{"date" => "2001-04-28T16:09:48","version" => "0.24"},{"date" => "2001-10-05T02:42:20","version" => "0.30"},{"date" => "2002-10-24T01:17:51","version" => "0.31"},{"date" => "2002-12-05T03:07:24","version" => "0.32"},{"date" => "2004-07-22T00:41:35","version" => "0.3901"},{"date" => "2004-07-22T20:01:47","version" => "0.39_02"},{"date" => "2004-07-23T01:52:58","version" => "0.39_03"},{"date" => "2004-07-26T06:36:47","version" => "0.39_04"},{"date" => "2004-08-01T01:21:33","version" => "1.00"},{"date" => "2004-08-18T20:32:07","version" => "1.01"},{"date" => "2004-08-24T15:53:20","version" => "1.02"},{"date" => "2004-08-26T21:51:46","version" => "1.03"},{"date" => "2004-09-03T06:56:23","version" => "1.04"},{"date" => "2004-09-03T07:00:16","version" => "1.05"},{"date" => "2005-03-29T23:06:41","version" => "1.06"},{"date" => "2005-04-04T15:17:24","version" => "2.0_01"},{"date" => "2005-04-07T16:08:52","version" => "2.0_02"},{"date" => "2005-04-08T16:48:24","version" => "2.0_03"},{"date" => "2006-10-24T13:58:29","version" => "2.0_04"},{"date" => "2007-12-16T15:33:27","version" => "2.01"},{"date" => "2007-12-16T16:04:00","version" => "2.02"},{"date" => "2007-12-17T12:49:37","version" => "2.03"},{"date" => "2007-12-17T20:01:53","version" => "2.04"},{"date" => "2007-12-20T16:01:35","version" => "2.10_01"},{"date" => "2008-01-04T15:55:53","version" => "2.10_02"},{"date" => "2008-01-05T13:14:32","version" => "2.10_03"},{"date" => "2008-01-06T21:52:18","version" => "2.10_04"},{"date" => "2008-01-08T16:20:38","version" => "2.10_05"},{"date" => "2008-01-10T21:43:53","version" => "2.11"},{"date" => "2018-09-27T23:02:17","version" => "v3.0.0"},{"date" => "2018-09-28T12:40:06","version" => "v3.0.1"},{"date" => "2018-09-28T16:56:39","version" => "v3.0.2"},{"date" => "2018-09-30T12:37:45","version" => "v3.1.0"},{"date" => "2018-10-01T16:51:16","version" => "v3.1.1"},{"date" => "2018-10-01T22:15:39","version" => "v3.1.2"},{"date" => "2018-10-03T23:36:26","version" => "v3.2.0"},{"date" => "2018-10-06T10:00:38","version" => "v3.2.1"},{"date" => "2018-10-23T20:30:22","version" => "v3.3.0"},{"date" => "2018-10-24T15:03:58","version" => "v3.3.1"},{"date" => "2018-10-27T16:33:30","version" => "v3.3.2"},{"date" => "2018-10-27T18:31:44","version" => "v3.3.3"},{"date" => "2018-11-11T15:13:51","version" => "v3.3.4"},{"date" => "2018-11-18T19:13:42","version" => "v3.4.0"},{"date" => "2019-06-06T20:30:43","version" => "v3.5.0"}]},"HTML-EP" => {"advisories" => [{"affected_versions" => [">=0.2011"],"cves" => ["CVE-2012-6142"],"description" => "HTML::EP::Session::Cookie in the HTML::EP module 0.2011 for Perl does not properly use the Storable::thaw function, which allows remote attackers to execute arbitrary code via a crafted request, which is not properly handled when it is deserialized.\n","distribution" => "HTML-EP","fixed_versions" => [],"id" => "CPANSA-HTML-EP-2012-6142","references" => ["http://www.securityfocus.com/bid/59833","http://seclists.org/oss-sec/2013/q2/318","https://exchange.xforce.ibmcloud.com/vulnerabilities/84199"],"reported" => "2014-06-04","severity" => undef}],"main_module" => "HTML::EP","versions" => [{"date" => "1998-06-24T20:39:44","version" => "0.1000"},{"date" => "1998-06-26T12:21:47","version" => "0.1002"},{"date" => "1998-07-17T21:28:11","version" => "0.1005"},{"date" => "1998-07-18T16:05:32","version" => "0.1006"},{"date" => "1998-07-24T20:40:11","version" => "0.1100"},{"date" => "1998-09-12T20:26:57","version" => "0.1106"},{"date" => "1998-09-14T00:09:23","version" => "0.1107"},{"date" => "1998-09-18T01:41:54","version" => "0.1108"},{"date" => "1998-10-06T09:42:57","version" => "0.1109"},{"date" => "1998-10-13T16:57:33","version" => "0.1111"},{"date" => "1998-10-15T19:02:15","version" => "0.1112"},{"date" => "1998-10-21T21:58:15","version" => "0.1113"},{"date" => "1998-11-06T20:01:59","version" => "0.1116"},{"date" => "1998-11-29T18:25:07","version" => "0.1117"},{"date" => "1998-12-03T17:11:04","version" => "0.1118"},{"date" => "1999-01-26T02:07:08","version" => "0.1123"},{"date" => "1999-02-01T00:08:19","version" => "0.1124"},{"date" => "1999-02-07T20:07:50","version" => "0.1125"},{"date" => "1999-02-13T12:36:36","version" => "0.1126"},{"date" => "1999-02-23T18:47:31","version" => "0.1127"},{"date" => "1999-02-26T18:27:47","version" => "0.1128"},{"date" => "1999-05-04T22:59:11","version" => "0.1130"},{"date" => "1999-08-26T15:05:04","version" => "0.11321"},{"date" => "1999-08-27T11:29:51","version" => "0.1133"},{"date" => "1999-08-31T11:04:44","version" => "0.1134"},{"date" => "1999-09-21T10:22:21","version" => "0.1135"},{"date" => "1999-09-26T13:27:28","version" => "0.20_00"},{"date" => "1999-09-27T10:28:51","version" => "0.20_01"},{"date" => "1999-11-05T11:38:40","version" => "0.2003"},{"date" => "1999-11-08T15:38:12","version" => "0.2004"},{"date" => "1999-11-08T18:18:11","version" => "0.2005"},{"date" => "1999-11-17T12:28:55","version" => "0.2006"},{"date" => "1999-11-17T17:23:52","version" => "0.2007"},{"date" => "1999-11-25T10:15:38","version" => "0.2008"},{"date" => "1999-12-07T20:43:46","version" => "0.2009"},{"date" => "1999-12-15T22:41:39","version" => "0.2010"},{"date" => "2001-01-05T13:26:37","version" => "0.2011"},{"date" => "2001-01-05T13:27:07","version" => 0}]},"HTML-Parser" => {"advisories" => [{"affected_versions" => ["<3.63"],"cves" => ["CVE-2009-3627"],"description" => "The decode_entities function in util.c in HTML-Parser before 3.63 allows context-dependent attackers to cause a denial of service (infinite loop) via an incomplete SGML numeric character reference, which triggers generation of an invalid UTF-8 character.\n","distribution" => "HTML-Parser","fixed_versions" => [">=3.63"],"id" => "CPANSA-HTML-Parser-2009-3627","references" => ["https://issues.apache.org/SpamAssassin/show_bug.cgi?id=6225","http://www.openwall.com/lists/oss-security/2009/10/23/9","http://secunia.com/advisories/37155","http://www.securityfocus.com/bid/36807","https://bugzilla.redhat.com/show_bug.cgi?id=530604","http://www.vupen.com/english/advisories/2009/3022","http://github.com/gisle/html-parser/commit/b9aae1e43eb2c8e989510187cff0ba3e996f9a4c","https://exchange.xforce.ibmcloud.com/vulnerabilities/53941"],"reported" => "2009-10-29","severity" => undef}],"main_module" => "HTML::Parser","versions" => [{"date" => "1998-03-31T22:25:05","version" => "2.14"},{"date" => "1998-04-02T11:40:38","version" => "2.16"},{"date" => "1998-04-28T08:35:12","version" => "2.17"},{"date" => "1998-06-22T19:45:24","version" => "2.18"},{"date" => "1998-07-06T23:12:33","version" => "2.19"},{"date" => "1998-07-08T13:05:04","version" => "2.20"},{"date" => "1998-11-13T21:48:08","version" => "2.21"},{"date" => "1999-06-09T10:34:35","version" => "2.23"},{"date" => "1999-11-03T07:10:32","version" => "2.24"},{"date" => "1999-11-05T09:46:11","version" => "2.25"},{"date" => "1999-12-14T23:01:49","version" => "3.00"},{"date" => "1999-12-19T07:07:38","version" => "3.01"},{"date" => "1999-12-21T09:55:56","version" => "3.02"},{"date" => "2000-01-15T16:25:39","version" => "3.04"},{"date" => "2000-01-22T16:14:25","version" => "3.05"},{"date" => "2000-03-06T14:00:15","version" => "3.06"},{"date" => "2000-03-20T12:47:48","version" => "3.07"},{"date" => "2000-05-23T10:59:19","version" => "3.08"},{"date" => "2000-06-28T08:49:09","version" => "3.09"},{"date" => "2000-06-29T07:52:47","version" => "3.10"},{"date" => "2000-08-22T10:04:28","version" => "3.11"},{"date" => "2000-09-14T18:22:47","version" => "3.12"},{"date" => "2000-09-17T01:55:45","version" => "3.13"},{"date" => "2000-12-04T06:05:39","version" => "3.14"},{"date" => "2000-12-26T09:04:53","version" => "3.15"},{"date" => "2001-02-23T07:21:20","version" => "3.16"},{"date" => "2001-02-24T06:32:38","version" => "3.17"},{"date" => "2001-02-25T04:51:50","version" => "3.18"},{"date" => "2001-03-10T04:32:27","version" => "3.19"},{"date" => "2001-03-13T19:44:52","version" => "3.19_90"},{"date" => "2001-03-16T02:29:32","version" => "3.19"},{"date" => "2001-03-19T19:26:50","version" => "3.19_91"},{"date" => "2001-03-26T15:37:39","version" => "3.19_92"},{"date" => "2001-03-27T19:44:20","version" => "3.19_93"},{"date" => "2001-03-30T08:24:11","version" => "3.19_94"},{"date" => "2001-04-03T00:18:14","version" => "3.20"},{"date" => "2001-04-10T22:34:03","version" => "3.21"},{"date" => "2001-04-18T05:31:23","version" => "3.22"},{"date" => "2001-05-02T04:18:51","version" => "3.23"},{"date" => "2001-05-05T06:18:21","version" => "3.23"},{"date" => "2001-05-09T07:23:34","version" => "3.24"},{"date" => "2001-05-11T17:26:39","version" => "3.25"},{"date" => "2001-06-12T08:35:06","version" => "3.25"},{"date" => "2002-03-17T20:11:55","version" => "3.26"},{"date" => "2003-01-18T13:08:01","version" => "3.27"},{"date" => "2003-04-17T03:56:32","version" => "3.28"},{"date" => "2003-08-15T06:11:17","version" => "3.29"},{"date" => "2003-08-18T05:48:21","version" => "3.30"},{"date" => "2003-08-19T14:56:07","version" => "3.31"},{"date" => "2003-10-10T14:31:48","version" => "3.32"},{"date" => "2003-10-14T10:53:29","version" => "3.33"},{"date" => "2003-10-27T21:23:09","version" => "3.34"},{"date" => "2003-12-12T14:27:23","version" => "3.35"},{"date" => "2004-04-01T12:21:44","version" => "3.36"},{"date" => "2004-11-10T18:56:54","version" => "3.37"},{"date" => "2004-11-11T10:19:56","version" => "3.38"},{"date" => "2004-11-17T14:33:49","version" => "3.39_90"},{"date" => "2004-11-23T11:46:30","version" => "3.39_91"},{"date" => "2004-11-23T22:25:21","version" => "3.39_92"},{"date" => "2004-11-29T11:14:34","version" => "3.40"},{"date" => "2004-11-30T09:30:56","version" => "3.41"},{"date" => "2004-12-04T11:54:54","version" => "3.42"},{"date" => "2004-12-06T09:19:28","version" => "3.43"},{"date" => "2004-12-28T14:07:28","version" => "3.44"},{"date" => "2005-01-06T09:09:45","version" => "3.45"},{"date" => "2005-10-24T12:34:04","version" => "3.46"},{"date" => "2005-11-22T21:50:09","version" => "3.47"},{"date" => "2005-12-02T17:41:00","version" => "3.48"},{"date" => "2006-02-08T10:58:39","version" => "3.49"},{"date" => "2006-02-14T18:32:51","version" => "3.50"},{"date" => "2006-03-22T09:26:15","version" => "3.51"},{"date" => "2006-04-26T08:43:13","version" => "3.52"},{"date" => "2006-04-27T11:55:34","version" => "3.53"},{"date" => "2006-04-28T08:21:04","version" => "3.54"},{"date" => "2006-07-10T09:16:22","version" => "3.55"},{"date" => "2007-01-12T11:00:07","version" => "3.56"},{"date" => "2008-11-16T21:45:07","version" => "3.57"},{"date" => "2008-11-17T11:35:37","version" => "3.58"},{"date" => "2008-11-24T09:15:09","version" => "3.59"},{"date" => "2009-02-09T11:26:08","version" => "3.60"},{"date" => "2009-06-20T09:34:17","version" => "3.61"},{"date" => "2009-08-13T21:01:27","version" => "3.62"},{"date" => "2009-10-22T20:11:52","version" => "3.63"},{"date" => "2009-10-25T12:24:11","version" => "3.64"},{"date" => "2010-04-04T20:44:00","version" => "3.65"},{"date" => "2010-07-09T13:27:13","version" => "3.66"},{"date" => "2010-08-17T17:15:19","version" => "3.67"},{"date" => "2010-09-01T21:28:52","version" => "3.68"},{"date" => "2011-10-15T15:35:01","version" => "3.69"},{"date" => "2013-03-28T22:21:30","version" => "3.70"},{"date" => "2013-05-08T22:23:29","version" => "3.71"},{"date" => "2016-01-19T17:44:02","version" => "3.72"},{"date" => "2020-08-25T17:40:17","version" => "3.73"},{"date" => "2020-08-30T18:40:48","version" => "3.74"},{"date" => "2020-08-30T19:58:22","version" => "3.75"},{"date" => "2021-03-04T18:06:59","version" => "3.76"},{"date" => "2022-03-14T22:12:49","version" => "3.77"},{"date" => "2022-03-28T15:23:23","version" => "3.78"},{"date" => "2022-10-12T15:41:58","version" => "3.79"},{"date" => "2022-11-01T14:19:26","version" => "3.80"},{"date" => "2023-01-31T03:13:18","version" => "3.81"},{"date" => "2024-03-13T20:11:51","version" => "3.82"},{"date" => "2024-07-30T16:42:50","version" => "3.83"}]},"HTML-Perlinfo" => {"advisories" => [{"affected_versions" => ["<1.52"],"cves" => [],"description" => "Possibility of denial-of-service attack.\n","distribution" => "HTML-Perlinfo","fixed_versions" => [">=1.52"],"id" => "CPANSA-HTML-Perlinfo-2008-01","references" => ["https://metacpan.org/changes/release/ACCARDO/HTML-Perlinfo-1.52"],"reported" => "2008-07-04"}],"main_module" => "HTML::Perlinfo","versions" => [{"date" => "2005-08-18T21:39:08","version" => "1.00"},{"date" => "2005-09-19T20:41:07","version" => "1.05"},{"date" => "2006-01-14T05:25:20","version" => "1.25"},{"date" => "2006-08-13T03:42:36","version" => "1.40"},{"date" => "2006-08-13T09:29:45","version" => "1.41"},{"date" => "2006-08-15T01:04:34","version" => "1.42"},{"date" => "2006-08-25T07:51:28","version" => "1.43"},{"date" => "2006-09-10T23:20:13","version" => "1.44"},{"date" => "2006-09-24T02:22:48","version" => "1.45"},{"date" => "2006-09-27T20:08:12","version" => "1.46"},{"date" => "2006-10-02T19:30:30","version" => "1.47"},{"date" => "2008-03-16T03:15:04","version" => "1.48"},{"date" => "2008-04-26T04:17:07","version" => "1.49"},{"date" => "2008-04-30T20:44:40","version" => "1.50"},{"date" => "2008-06-08T21:07:29","version" => "1.51"},{"date" => "2008-07-03T23:57:26","version" => "1.52"},{"date" => "2008-07-21T22:24:22","version" => "1.53"},{"date" => "2008-07-27T23:52:36","version" => "1.54"},{"date" => "2009-04-08T01:09:54","version" => "1.55"},{"date" => "2009-04-08T19:06:59","version" => "1.56"},{"date" => "2009-04-16T15:57:34","version" => "1.57"},{"date" => "2009-04-17T02:41:48","version" => "1.58"},{"date" => "2009-04-22T03:29:45","version" => "1.59"},{"date" => "2009-05-02T20:48:38","version" => "1.60"},{"date" => "2009-05-02T22:21:42","version" => "1.60"},{"date" => "2009-05-03T23:02:35","version" => "1.61"},{"date" => "2011-06-13T19:28:39","version" => "1.62"},{"date" => "2014-08-19T21:37:30","version" => "1.63"},{"date" => "2014-08-19T22:29:15","version" => "1.64"},{"date" => "2015-06-06T23:25:41","version" => "1.65"},{"date" => "2015-06-08T18:20:03","version" => "1.66"},{"date" => "2015-06-08T20:22:33","version" => "1.67"},{"date" => "2015-06-12T02:03:18","version" => "1.68"},{"date" => "2016-11-29T19:21:00","version" => "1.69"},{"date" => "2019-06-24T15:33:44","version" => "1.70"},{"date" => "2019-06-25T02:15:30","version" => "1.71"},{"date" => "2019-07-02T19:22:14","version" => "1.72"},{"date" => "2019-07-02T20:41:23","version" => "1.73"}]},"HTML-Scrubber" => {"advisories" => [{"affected_versions" => ["<0.15"],"cves" => ["CVE-2015-5667"],"description" => "Cross-site scripting (XSS) vulnerability in the HTML-Scrubber module before 0.15 for Perl, when the comment feature is enabled, allows remote attackers to inject arbitrary web script or HTML via a crafted comment.\n","distribution" => "HTML-Scrubber","fixed_versions" => [">=0.15"],"id" => "CPANSA-HTML-Scrubber-2015-5667","references" => ["http://jvn.jp/en/jp/JVN53973084/index.html","http://jvndb.jvn.jp/jvndb/JVNDB-2015-000171","https://metacpan.org/release/HTML-Scrubber","http://lists.fedoraproject.org/pipermail/package-announce/2015-November/172997.html","http://lists.fedoraproject.org/pipermail/package-announce/2015-November/172983.html","http://lists.fedoraproject.org/pipermail/package-announce/2015-November/172934.html"],"reported" => "2015-10-31","severity" => undef}],"main_module" => "HTML::Scrubber","versions" => [{"date" => "2003-04-18T14:10:19","version" => "0.02"},{"date" => "2003-07-21T14:57:02","version" => "0.03"},{"date" => "2003-10-30T02:31:36","version" => "0.04"},{"date" => "2003-10-31T07:27:00","version" => "0.05"},{"date" => "2003-11-02T11:10:49","version" => "0.06"},{"date" => "2004-03-18T14:35:12","version" => "0.07"},{"date" => "2004-04-01T22:12:20","version" => "0.08"},{"date" => "2011-04-01T15:36:18","version" => "0.09"},{"date" => "2013-09-27T14:06:41","version" => "0.10"},{"date" => "2013-10-11T14:13:11","version" => "0.11"},{"date" => "2015-03-14T18:25:35","version" => "0.12"},{"date" => "2015-03-19T16:31:12","version" => "0.13"},{"date" => "2015-04-02T16:20:48","version" => "0.14"},{"date" => "2015-10-10T14:02:08","version" => "0.15"},{"date" => "2017-06-25T19:31:24","version" => "0.16"},{"date" => "2017-06-27T13:04:46","version" => "0.17"},{"date" => "2019-09-22T11:11:50","version" => "0.18"},{"date" => "2019-09-24T12:28:19","version" => "0.19"}]},"HTML-StripScripts" => {"advisories" => [{"affected_versions" => ["<=1.06"],"cves" => ["CVE-2023-24038"],"description" => "The HTML-StripScripts module through 1.06 for Perl allows _hss_attval_style ReDoS because of catastrophic backtracking for HTML content with certain style attributes.\n","distribution" => "HTML-StripScripts","fixed_versions" => [],"id" => "CPANSA-HTML-StripScripts-2023-24038","references" => ["https://github.com/clintongormley/perl-html-stripscripts/issues/3","https://lists.debian.org/debian-lts-announce/2023/01/msg00036.html","https://www.debian.org/security/2023/dsa-5339"],"reported" => "2023-01-21","severity" => "high"}],"main_module" => "HTML::StripScripts","versions" => [{"date" => "2003-03-30T10:08:51","version" => "0.01"},{"date" => "2003-07-25T18:13:42","version" => "0.02"},{"date" => "2004-04-27T16:45:34","version" => "0.03"},{"date" => "2007-05-28T11:57:07","version" => "0.90"},{"date" => "2007-05-28T12:31:03","version" => "0.99"},{"date" => "2007-05-29T13:15:46","version" => "0.991"},{"date" => "2007-06-05T12:44:56","version" => "1.00"},{"date" => "2007-10-22T14:30:52","version" => "1.01"},{"date" => "2007-10-22T15:47:44","version" => "1.02"},{"date" => "2007-10-22T17:21:36","version" => "1.03"},{"date" => "2007-11-16T17:53:46","version" => "1.04"},{"date" => "2009-11-05T10:25:59","version" => "1.05"},{"date" => "2016-05-12T09:44:35","version" => "1.06"}]},"HTML-Template-Pro" => {"advisories" => [{"affected_versions" => ["<0.9507"],"cves" => ["CVE-2011-4616"],"description" => "Cross-site scripting (XSS) vulnerability in the HTML-Template-Pro module before 0.9507 for Perl allows remote attackers to inject arbitrary web script or HTML via template parameters, related to improper handling of > (greater than) and < (less than) characters.\n","distribution" => "HTML-Template-Pro","fixed_versions" => [">=0.9507"],"id" => "CPANSA-HTML-Template-Pro-2011-4616","references" => ["http://bugs.debian.org/cgi-bin/bugreport.cgi?bug=652587","http://openwall.com/lists/oss-security/2011/12/19/1","http://metacpan.org/diff/release/VIY/HTML-Template-Pro-0.9505/VIY/HTML-Template-Pro-0.9507","http://secunia.com/advisories/47184","http://cpansearch.perl.org/src/VIY/HTML-Template-Pro-0.9507/Changes","http://www.securityfocus.com/bid/51117","http://lists.fedoraproject.org/pipermail/package-announce/2012-October/089603.html","http://lists.fedoraproject.org/pipermail/package-announce/2012-October/089889.html","http://lists.fedoraproject.org/pipermail/package-announce/2012-October/089888.html"],"reported" => "2012-01-06","severity" => undef}],"main_module" => "HTML::Template::Pro","versions" => [{"date" => "2005-06-09T11:07:59","version" => "0.38"},{"date" => "2005-06-22T09:55:37","version" => "0.39"},{"date" => "2005-07-07T09:11:59","version" => "0.40"},{"date" => "2005-07-26T16:58:29","version" => "0.41"},{"date" => "2005-08-04T15:58:27","version" => "0.42"},{"date" => "2005-08-04T17:36:21","version" => "0.43"},{"date" => "2005-08-12T16:32:44","version" => "0.44"},{"date" => "2005-08-19T19:10:08","version" => "0.45"},{"date" => "2005-08-26T18:24:23","version" => "0.46"},{"date" => "2005-08-31T17:43:09","version" => "0.48"},{"date" => "2005-09-08T17:43:14","version" => "0.50"},{"date" => "2005-09-30T15:59:34","version" => "0.52"},{"date" => "2005-10-06T17:14:51","version" => "0.53"},{"date" => "2005-10-17T13:37:05","version" => "0.54"},{"date" => "2005-10-26T12:18:18","version" => "0.55"},{"date" => "2005-11-03T16:46:06","version" => "0.56"},{"date" => "2005-11-13T16:12:39","version" => "0.57"},{"date" => "2005-12-02T08:10:18","version" => "0.58"},{"date" => "2006-01-22T20:07:54","version" => "0.59"},{"date" => "2006-02-02T16:32:55","version" => "0.60"},{"date" => "2006-02-06T20:45:02","version" => "0.61"},{"date" => "2006-02-22T20:05:55","version" => "0.62"},{"date" => "2006-04-18T20:24:51","version" => "0.64"},{"date" => "2007-06-01T14:46:48","version" => "0.65"},{"date" => "2007-10-04T11:08:55","version" => "0.66"},{"date" => "2007-12-02T23:20:56","version" => "0.67"},{"date" => "2008-01-08T18:01:32","version" => "0.68"},{"date" => "2008-01-08T20:03:26","version" => "0.68"},{"date" => "2008-03-01T19:46:47","version" => "0.69"},{"date" => "2008-06-09T09:06:12","version" => "0.70"},{"date" => "2008-09-05T19:36:06","version" => "0.71"},{"date" => "2008-12-19T08:16:12","version" => "0.72"},{"date" => "2009-04-02T20:36:25","version" => "0.73"},{"date" => "2009-04-10T20:41:07","version" => "0.74"},{"date" => "2009-07-05T16:40:09","version" => "0.75"},{"date" => "2009-07-13T08:33:36","version" => "0.76"},{"date" => "2009-07-23T17:37:10","version" => "0.80"},{"date" => "2009-07-28T15:58:37","version" => "0.81"},{"date" => "2009-08-04T15:46:34","version" => "0.82"},{"date" => "2009-08-05T20:27:52","version" => "0.83"},{"date" => "2009-08-08T18:13:20","version" => "0.84"},{"date" => "2009-08-09T16:45:02","version" => "0.85"},{"date" => "2009-08-24T08:00:34","version" => "0.86"},{"date" => "2009-08-29T19:22:41","version" => "0.87"},{"date" => "2009-09-11T16:53:57","version" => "0.90"},{"date" => "2009-09-24T15:48:49","version" => "0.91"},{"date" => "2009-09-29T20:14:35","version" => "0.92"},{"date" => "2009-11-23T20:25:34","version" => "0.93"},{"date" => "2010-03-26T19:12:55","version" => "0.94"},{"date" => "2010-05-21T19:34:29","version" => "0.95"},{"date" => "2010-06-16T19:00:45","version" => "0.9501"},{"date" => "2010-06-24T18:50:34","version" => "0.9502"},{"date" => "2010-08-29T12:45:12","version" => "0.9503"},{"date" => "2010-09-29T07:16:03","version" => "0.9504"},{"date" => "2011-07-01T10:40:21","version" => "0.9505"},{"date" => "2011-10-04T20:31:16","version" => "0.9506"},{"date" => "2011-12-09T07:59:17","version" => "0.9507"},{"date" => "2011-12-26T21:57:41","version" => "0.9508"},{"date" => "2012-02-28T19:59:05","version" => "0.9509"},{"date" => "2013-05-13T08:40:09","version" => "0.9510"},{"date" => "2021-11-30T23:21:23","version" => "0.9520"},{"date" => "2021-12-02T07:27:12","version" => "0.9521"},{"date" => "2021-12-06T17:53:48","version" => "0.9522"},{"date" => "2021-12-15T09:50:03","version" => "0.9523"},{"date" => "2022-01-16T20:42:34","version" => "0.9524"}]},"HTTP-Body" => {"advisories" => [{"affected_versions" => [">=1.08,<1.23"],"cves" => ["CVE-2013-4407"],"description" => "HTTP::Body::Multipart in the HTTP-Body 1.08, 1.22, and earlier module for Perl uses the part of the uploaded file's name after the first \".\" character as the suffix of a temporary