package Hypersonic::JIT::Util;
use strict;
use warnings;

use Config;
use Carp qw(croak);

our $VERSION = '0.09';

# =============================================================================
# Cache Directory Management
# =============================================================================

# Base cache directory
our $CACHE_BASE = '_hypersonic_cache';

# Get cache directory for a module
sub cache_dir {
    my ($class, $subdir) = @_;
    return $subdir ? "$CACHE_BASE/$subdir" : $CACHE_BASE;
}

# Subdirectory constants
sub cache_dir_core     { shift->cache_dir('core') }
sub cache_dir_socket   { shift->cache_dir('socket') }
sub cache_dir_request  { shift->cache_dir('request') }
sub cache_dir_response { shift->cache_dir('response') }
sub cache_dir_session  { shift->cache_dir('session') }
sub cache_dir_tls      { shift->cache_dir('tls') }
sub cache_dir_future   { shift->cache_dir('future') }

# =============================================================================
# Fork Detection
# =============================================================================

sub can_fork {
    return $Config{d_fork} && eval { 
        my $pid = fork();
        if (defined $pid && $pid == 0) {
            exit(0);  # Child exits immediately
        }
        waitpid($pid, 0) if defined $pid;
        1;
    };
}

# =============================================================================
# Parallel Compilation
# =============================================================================

# Standalone modules that can compile independently
our @STANDALONE_MODULES = qw(
    Hypersonic::Socket
    Hypersonic::Response
    Hypersonic::Session
    Hypersonic::TLS
);

sub compile_standalone_modules {
    my ($class, %opts) = @_;
    
    my @modules = $opts{modules} // @STANDALONE_MODULES;
    my $parallel = $opts{parallel} // 1;
    
    # Load modules
    for my $mod (@modules) {
        eval "require $mod" or do {
            warn "Failed to load $mod: $@";
            return 0;
        };
    }
    
    if ($parallel && $class->can_fork) {
        return $class->_compile_parallel(\@modules, \%opts);
    } else {
        return $class->_compile_sequential(\@modules, \%opts);
    }
}

sub _compile_parallel {
    my ($class, $modules, $opts) = @_;
    
    my @pids;
    my %child_to_mod;
    
    for my $mod (@$modules) {
        my $pid = fork();
        
        if (!defined $pid) {
            warn "Fork failed for $mod: $!";
            # Fallback to sequential for this module
            eval { $mod->compile(%$opts) };
            warn "Compile failed for $mod: $@" if $@;
        } elsif ($pid == 0) {
            # Child process
            eval { $mod->compile(%$opts) };
            exit($@ ? 1 : 0);
        } else {
            # Parent
            push @pids, $pid;
            $child_to_mod{$pid} = $mod;
        }
    }
    
    # Wait for all children
    my $all_ok = 1;
    for my $pid (@pids) {
        waitpid($pid, 0);
        if ($? != 0) {
            my $mod = $child_to_mod{$pid};
            warn "Compilation failed for $mod (exit code: " . ($? >> 8) . ")";
            $all_ok = 0;
        }
    }
    
    # After fork compilation, parent must load the compiled modules
    # (children wrote to cache, but parent needs to load)
    for my $mod (@$modules) {
        eval { $mod->compile(%$opts) };  # Will load from cache
        if ($@) {
            warn "Failed to load compiled $mod: $@";
            $all_ok = 0;
        }
    }
    
    return $all_ok;
}

sub _compile_sequential {
    my ($class, $modules, $opts) = @_;
    
    my $all_ok = 1;
    for my $mod (@$modules) {
        eval { $mod->compile(%$opts) };
        if ($@) {
            warn "Compile failed for $mod: $@";
            $all_ok = 0;
        }
    }
    
    return $all_ok;
}

# =============================================================================
# Common Include Patterns
# =============================================================================

sub add_standard_includes {
    my ($class, $builder, @features) = @_;
    
    my %features = map { $_ => 1 } @features;
    
    # Always needed
    $builder->line('#include <stdlib.h>')
            ->line('#include <string.h>')
            ->line('#include <errno.h>');
    
    if ($features{stdio}) {
        $builder->line('#include <stdio.h>');
    }
    
    if ($features{unistd}) {
        $builder->line('#include <unistd.h>');
    }
    
    if ($features{fcntl}) {
        $builder->line('#include <fcntl.h>');
    }
    
    if ($features{socket}) {
        $builder->line('#include <sys/socket.h>')
                ->line('#include <sys/types.h>')
                ->line('#include <netinet/in.h>')
                ->line('#include <netinet/tcp.h>')
                ->line('#include <arpa/inet.h>')
                ->line('#include <sys/uio.h>');
    }
    
    if ($features{threading}) {
        $builder->line('#ifndef _WIN32')
                ->line('#include <pthread.h>')
                ->line('#endif');
    }
    
    if ($features{eventfd}) {
        $builder->line('#ifdef __linux__')
                ->line('#include <sys/eventfd.h>')
                ->line('#endif');
    }
    
    if ($features{time}) {
        $builder->line('#include <time.h>');
    }
    
    if ($features{signal}) {
        $builder->line('#include <signal.h>');
    }
    
    if ($features{openssl}) {
        $builder->line('#include <openssl/hmac.h>')
                ->line('#include <openssl/evp.h>')
                ->line('#include <openssl/rand.h>');
    }
    
    $builder->blank;
    
    return $builder;
}

# =============================================================================
# Platform Detection Helpers
# =============================================================================

sub add_platform_eventfd {
    my ($class, $builder) = @_;
    
    $builder->line('#ifndef _WIN32')
            ->line('#ifdef __linux__')
            ->line('#include <sys/eventfd.h>')
            ->line('#define USE_EVENTFD 1')
            ->line('#else')
            ->line('#define USE_EVENTFD 0')
            ->line('#endif')
            ->line('#endif /* !_WIN32 */')
            ->blank;
    
    return $builder;
}

sub add_platform_detection {
    my ($class, $builder) = @_;
    
    $builder->line('/* Platform detection */')
            ->line('#if defined(__APPLE__)')
            ->line('#define HYPERSONIC_MACOS 1')
            ->line('#elif defined(__linux__)')
            ->line('#define HYPERSONIC_LINUX 1')
            ->line('#elif defined(_WIN32)')
            ->line('#define HYPERSONIC_WINDOWS 1')
            ->line('#elif defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)')
            ->line('#define HYPERSONIC_BSD 1')
            ->line('#endif')
            ->blank;
    
    return $builder;
}

# =============================================================================
# Library Detection
# =============================================================================

# Check if a library can actually be linked (compile+link test)
# This is the definitive test - if this fails, the JIT code won't load
sub can_link {
    my ($class, $cflags, $ldflags, $test_symbol, $extra_includes) = @_;
    $cflags //= '';
    $ldflags //= '';
    $extra_includes //= '';

    require File::Temp;

    # Create minimal C code that references the symbol
    # Use a void* cast to avoid prototype conflicts
    my $test_code = <<"C";
$extra_includes
int main() {
    void *p = (void*)$test_symbol;
    return p ? 0 : 1;
}
C

    # Write to temp file
    my $src = File::Temp->new(SUFFIX => '.c', UNLINK => 1);
    my $src_path = $src->filename;
    print $src $test_code;
    close $src;

    # Output file
    my $out = File::Temp->new(SUFFIX => '', UNLINK => 1);
    my $out_path = $out->filename;
    close $out;

    # Try to compile and link
    my $cc = $ENV{CC} || 'cc';
    my $result = system("$cc $cflags -o $out_path $src_path $ldflags 2>/dev/null");

    # Clean up output file
    unlink $out_path if -f $out_path;

    return $result == 0;
}

sub detect_library {
    my ($class, $lib_name, %opts) = @_;

    my $result = {
        available => 0,
        cflags    => '',
        ldflags   => '',
    };

    my $test_symbol = $opts{test_symbol};
    my $test_include = $opts{test_include} // '';

    # Try Alien module first
    my $alien_module = $opts{alien} // "Alien::\u$lib_name";
    if (eval "require $alien_module; 1") {
        my $cflags = $alien_module->cflags // '';
        my $ldflags = $alien_module->libs // '';

        # Verify it actually links if test_symbol provided
        if (!$test_symbol || $class->can_link($cflags, $ldflags, $test_symbol, $test_include)) {
            $result->{available} = 1;
            $result->{cflags}  = $cflags;
            $result->{ldflags} = $ldflags;
            return $result;
        }
    }

    # Try pkg-config
    my $pkg_name = $opts{pkg_config} // $lib_name;
    my $cflags = `pkg-config --cflags $pkg_name 2>/dev/null`;
    my $ldflags = `pkg-config --libs $pkg_name 2>/dev/null`;

    if ($? == 0 && $ldflags) {
        chomp($cflags);
        chomp($ldflags);

        # Verify it actually links if test_symbol provided
        if (!$test_symbol || $class->can_link($cflags, $ldflags, $test_symbol, $test_include)) {
            $result->{available} = 1;
            $result->{cflags}  = $cflags;
            $result->{ldflags} = $ldflags;
            return $result;
        }
    }

    # Try common paths (with additional Homebrew versioned paths for OpenSSL)
    my @search_paths = @{$opts{paths} // [
        # macOS Homebrew (Apple Silicon) - versioned first
        '/opt/homebrew/opt/openssl@3',
        '/opt/homebrew/opt/openssl',
        '/opt/homebrew',
        # macOS Homebrew (Intel) - versioned first
        '/usr/local/opt/openssl@3',
        '/usr/local/opt/openssl',
        '/usr/local',
        # Linux standard locations
        '/usr',
        '/opt/local',
    ]};

    my $header = $opts{header};
    my $lib = $opts{lib} // "lib$lib_name";

    for my $prefix (@search_paths) {
        my $inc_dir = "$prefix/include";
        my $lib_dir = "$prefix/lib";

        # Check for header if specified
        if ($header && !-f "$inc_dir/$header") {
            next;
        }

        # Check for library file
        my $found_lib = 0;
        for my $ext (qw(.dylib .so .a)) {
            if (-f "$lib_dir/$lib$ext") {
                $found_lib = 1;
                last;
            }
        }

        if ($found_lib) {
            my $try_cflags = "-I$inc_dir";
            my $try_ldflags = "-L$lib_dir -l$lib_name";

            # Verify it actually links if test_symbol provided
            if (!$test_symbol || $class->can_link($try_cflags, $try_ldflags, $test_symbol, $test_include)) {
                $result->{available} = 1;
                $result->{cflags}  = $try_cflags;
                $result->{ldflags} = $try_ldflags;
                return $result;
            }
        }
    }

    return $result;
}

# Convenience methods for common libraries
sub detect_openssl {
    my ($class) = @_;
    return $class->detect_library('ssl',
        alien        => 'Alien::OpenSSL',
        pkg_config   => 'openssl',
        header       => 'openssl/ssl.h',
        lib          => 'libssl',
        test_symbol  => 'SSL_new',
        test_include => '#include <openssl/ssl.h>',
    );
}

sub detect_zlib {
    my ($class) = @_;
    return $class->detect_library('z',
        alien        => 'Alien::zlib',
        pkg_config   => 'zlib',
        header       => 'zlib.h',
        lib          => 'libz',
        test_symbol  => 'deflate',
        test_include => '#include <zlib.h>',
    );
}

sub detect_nghttp2 {
    my ($class) = @_;
    return $class->detect_library('nghttp2',
        pkg_config   => 'libnghttp2',
        header       => 'nghttp2/nghttp2.h',
        test_symbol  => 'nghttp2_session_client_new',
        test_include => '#include <nghttp2/nghttp2.h>',
    );
}

1;

__END__

=head1 NAME

Hypersonic::JIT::Util - Utilities for JIT compilation in Hypersonic

=head1 SYNOPSIS

    use Hypersonic::JIT::Util;
    
    # Get cache directory
    my $cache = Hypersonic::JIT::Util->cache_dir('socket');
    # Returns: _hypersonic_cache/socket
    
    # Compile standalone modules in parallel
    Hypersonic::JIT::Util->compile_standalone_modules(parallel => 1);
    
    # Add common includes to builder
    Hypersonic::JIT::Util->add_standard_includes($builder, qw(socket threading));
    
    # Detect library
    my $ssl = Hypersonic::JIT::Util->detect_openssl();
    if ($ssl->{available}) {
        # Use $ssl->{cflags} and $ssl->{ldflags}
    }

=head1 DESCRIPTION

This module provides common utilities for JIT compilation across Hypersonic modules:

=over 4

=item * Unified cache directory structure

=item * Fork-based parallel compilation

=item * Standard include patterns

=item * Library detection (Alien → pkg-config → path search)

=back

=head1 METHODS

=head2 cache_dir($subdir)

Returns the cache directory path. With C<$subdir>, returns C<_hypersonic_cache/$subdir>.

=head2 compile_standalone_modules(%opts)

Compiles all standalone modules (Socket, Response, Session, TLS).

Options:

=over 4

=item parallel => 1

Use fork for parallel compilation (default: 1)

=item modules => \@list

Override list of modules to compile

=back

=head2 can_fork()

Returns true if fork() is available on this platform.

=head2 add_standard_includes($builder, @features)

Adds standard C includes to the builder based on features:
C<socket>, C<threading>, C<time>, C<signal>, C<unistd>, C<fcntl>

=head2 detect_library($name, %opts)

Detects a C library. Returns hashref with C<available>, C<cflags>, C<ldflags>.

=head2 detect_openssl(), detect_zlib(), detect_nghttp2()

Convenience methods for common libraries.

=head1 AUTHOR

Hypersonic Contributors

=cut
